/*
 *	Netlink route / adddress utilities
 *
 *	Copyright (C) 2009-2012 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2012 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *	MA  02110-1301  USA
 *
 *	Authors: Olaf Kirch <okir@suse.de>
 *	         Marius Tomaschewski <mt@suse.de>
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	You can find the wicked project at http://gitorious.org/wicked/.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <netlink/msg.h>
#include <netlink/attr.h>
#include <netlink/netlink.h>
#include <netlink/route/rtnl.h>
#include <netlink/route/route.h>
#include <net/if.h>

#include "ntypes.h"
#include "nutils.h"
#include "nlutils.h"
#include "logging.h"


struct nc_netlink {
	struct nl_cb *		cb;
#if defined(__NETCONTROL_LIBNL1)
	struct nl_handle *	handle;
#else
	struct nl_sock *	handle;
#endif
};

static inline void *
__nc_netlink_alloc_sock_handle(struct nl_cb *cb)
{
#if defined(__NETCONTROL_LIBNL1)
	return nl_handle_alloc_cb(cb);
#else
	return nl_socket_alloc_cb(cb);
#endif
}

static inline void
__nc_netlink_free_sock_handle(nc_netlink_t *nl)
{
	if (nl && nl->handle) {
#if defined(__NETCONTROL_LIBNL1)
		nl_handle_destroy(nl->handle);
#else
		nl_socket_free(nl->handle);
#endif
	}
}

void
nc_netlink_close(nc_netlink_t **nl)
{
	if(nl && *nl) {
		__nc_netlink_free_sock_handle(*nl);
		if((*nl)->cb)
			nl_cb_put((*nl)->cb);
		free(*nl);
		*nl = NULL;
	}
}

nc_netlink_t *
nc_netlink_open()
{
	nc_netlink_t *nl;

	nl = calloc(1, sizeof(*nl));
	if(!nl) {
		nc_error("Cannot allocate netlink structure: %m");
		return NULL;
	}

	nl->cb = nl_cb_alloc(NL_CB_DEFAULT);
	if(!nl->cb) {
		nc_error("Cannot allocate netlink callback: %m");
		nc_netlink_close(&nl);
	}

	nl->handle = __nc_netlink_alloc_sock_handle(nl->cb);
	if(!nl->handle) {
		nc_error("Cannot allocate netlink socket handle: %m");
		nc_netlink_close(&nl);
		return NULL;
	}

	if(nl_connect(nl->handle, NETLINK_ROUTE) < 0) {
		nc_error("Cannot connect to rtnetlink: %m");
		nc_netlink_close(&nl);
		return NULL;
	}

	return nl;
}

static struct nl_cb *
__nc_nl_cb_clone(nc_netlink_t *nl)
{
	struct nl_cb *ccb, *ocb;

	if( (ccb = nl->cb) != NULL) {
		ccb = nl_cb_clone(ccb);
	} else {
		ocb = nl_socket_get_cb(nl->handle);
		ccb = nl_cb_clone(ocb);
		nl_cb_put(ocb);
	}
	return ccb;
}

static int
__nc_nl_dump(nc_netlink_t *nl, int af, int type, nl_recvmsg_msg_cb_t func, void *data)
{
	struct nl_cb *cb;

	if(!nl || !nl->handle || !func) {
		nc_error("Invalid netlink handle");
		return -1;
	}

	if(!(cb = __nc_nl_cb_clone(nl))) {
		nc_error("Cannot clone netlink callback: %m");
		return -1;
	}

	if(nl_rtgen_request(nl->handle, type, af, NLM_F_DUMP) < 0) {
		nc_error("Failed to send netlink request: %m");
		nl_cb_put(cb);
		return -1;
	}

	if(nl_cb_set(cb, NL_CB_VALID, NL_CB_CUSTOM, func, data) < 0) {
		nc_error("Cannot set custom netlink callback: %m");
		nl_cb_put(cb);
		return -1;
	}

	if(nl_recvmsgs(nl->handle, cb) < 0) {
		nc_error("Failed to receive netlink response: %m");
		nl_cb_put(cb);

		return -1;
	}

	nl_cb_put(cb);

	return 0;
}

static int
__nc_nla_get_addr(int af, nc_sockaddr_t *ss, struct nlattr *nla)
{
	unsigned int alen, maxlen;
	void *dst;

	memset(ss, 0, sizeof(*ss));
	ss->ss_family = AF_UNSPEC;

	if (!nla)
		return -1;

	alen = nla_len(nla);
	if (alen > sizeof(*ss))
		alen = sizeof(*ss);

	switch (af) {
		case AF_INET:
			dst = &ss->sin.sin_addr;
			maxlen = sizeof(ss->sin.sin_addr);
		break;
		case AF_INET6:
			dst = &ss->six.sin6_addr;
			maxlen = sizeof(ss->six.sin6_addr);
		break;
		default:
			return -1;
		break;
	}
	if (alen != maxlen)
		return -1;

	memcpy(dst, nla_data(nla), alen);
	ss->ss_family = af;
	return 0;
}

int
nc_netlink_parse_route_msg(struct nl_msg *msg, nc_route_t **rp, int *ifindex)
{
	const struct sockaddr_nl *sender = nlmsg_get_src(msg);
	struct nlmsghdr *nlh;
	struct nlattr *rta;
	struct rtmsg *rtm;
	int oif_index = -1;
	struct nlattr *tb[RTN_MAX+1];
	char ifname[IF_NAMESIZE + 1] = {'\0'};
	nc_sockaddr_t dst_addr;
	nc_sockaddr_t gw_addr;

	if (!msg || !rp || !ifindex) {
		nc_error("Usage error -- missed msg or data");
		return NL_SKIP;
	}

	if (sender->nl_pid != 0) {
		nc_error("Received netlink message from PID %u", sender->nl_pid);
		return NL_SKIP;
	}

	nlh = nlmsg_hdr(msg);
	if (nlh->nlmsg_type != RTM_NEWROUTE) {
		nc_error("Netlink has unexpected message type %d; expected %d",
			 nlh->nlmsg_type, RTM_NEWROUTE);
		return NL_SKIP;
	}

	memset(tb, 0, sizeof(tb));
	if (nlmsg_parse(nlh, sizeof(*rtm), tb, RTN_MAX, NULL) < 0) {
		nc_error("Unable to parse netlink route message: %m");
		return NL_SKIP;
	}

	/* nlmsg_parse verifies the length using nlmsg_valid_hdr */

	rta = nlmsg_find_attr(nlh, sizeof(*rtm), RTA_OIF);
	if (rta != NULL)
		oif_index = nla_get_u32(rta);

	if(oif_index == -1) {
		nc_error("Unable to find interface index in netlink route message");
		return NL_SKIP;
	}

	rtm = nlmsg_data(nlh);
	if (rtm->rtm_table != RT_TABLE_MAIN)
		return NL_SKIP;

	if (rtm->rtm_protocol == RTPROT_REDIRECT)
		return NL_SKIP;

	/* source route ... skip */
	if (rtm->rtm_src_len != 0)
		return NL_SKIP;

#if 0
	{
		char rttype_buf[256]  = {'\0'};
		char rtproto_buf[256] = {'\0'};
		char rttable_buf[256] = {'\0'};
		char rtflags_buf[256] = {'\0'};
		char rtscope_buf[256] = {'\0'};

		nc_error("RTM_NEWROUTE family=%d dstlen=%u srclen=%u type=%s proto=%s flags=%s table=%s scope=%s",
			rtm->rtm_family,
			rtm->rtm_dst_len,
	       		rtm->rtm_src_len,
		       	nl_rtntype2str(rtm->rtm_type, rttype_buf, sizeof(rttype_buf)),
			rtnl_route_proto2str(rtm->rtm_protocol, rtproto_buf, sizeof(rtproto_buf)),
			rtnl_route_nh_flags2str(rtm->rtm_flags, rtflags_buf, sizeof(rtflags_buf)),
			rtnl_route_table2str(rtm->rtm_table, rttable_buf, sizeof(rttable_buf)),
			rtnl_scope2str(rtm->rtm_scope, rtscope_buf, sizeof(rtscope_buf)));
	}
#endif

	memset(&dst_addr, 0, sizeof(dst_addr));
	memset(&gw_addr, 0, sizeof(gw_addr));

	if (rtm->rtm_dst_len != 0) {
		if (tb[RTA_DST] == NULL)
			return NL_SKIP;
		__nc_nla_get_addr(rtm->rtm_family, &dst_addr, tb[RTA_DST]);
	}

	if (tb[RTA_GATEWAY] != NULL)
		__nc_nla_get_addr(rtm->rtm_family, &gw_addr, tb[RTA_GATEWAY]);

	if(if_indextoname(oif_index, ifname) == NULL) {
		nc_error("Cannot map a network interface index to its name: %m");
		return NL_SKIP;
	}

	*rp = nc_route_create(rtm->rtm_dst_len, &dst_addr, &gw_addr, ifname, NULL);
	if(*rp == NULL) {
		nc_error("Cannot create route from netlink message: %m");
		return NL_SKIP;
	}
	*ifindex = oif_index;

	return NL_OK;
}

int
nc_netlink_parse_addr_msg(struct nl_msg *msg, nc_address_t **ap, int *ifindex)
{
	const struct sockaddr_nl *sender = nlmsg_get_src(msg);
	struct nlmsghdr *nlh;
	struct ifaddrmsg *ifa;
	struct nlattr *tb[IFA_MAX+1];
	nc_address_t tmp;

	if (!msg || !ap || !ifindex) {
		nc_error("Usage error -- missed msg or data");
		return NL_SKIP;
	}

	if (sender->nl_pid != 0) {
		nc_error("Received netlink message from PID %u", sender->nl_pid);
		return NL_SKIP;
	}

	nlh = nlmsg_hdr(msg);
	if (nlh->nlmsg_type != RTM_NEWADDR) {
		nc_error("Netlink has unexpected message type %d; expected %d",
			 nlh->nlmsg_type, RTM_NEWADDR);
		return NL_SKIP;
	}

	memset(tb, 0, sizeof(tb));
	if (nlmsg_parse(nlh, sizeof(*ifa), tb, IFA_MAX, NULL) < 0) {
		nc_error("Unable to parse netlink address message");
		return NL_SKIP;
	}

	/* nlmsg_parse verifies the length using nlmsg_valid_hdr */

	ifa = nlmsg_data(nlh);
	memset(&tmp, 0, sizeof(tmp));
	/*
	 * Hmm... does this catches the ptp or not fish?
	 */
	if(!tb[IFA_BROADCAST] && tb[IFA_LOCAL] && tb[IFA_ADDRESS]) {
		__nc_nla_get_addr(ifa->ifa_family, &tmp.local_addr, tb[IFA_LOCAL]);
		__nc_nla_get_addr(ifa->ifa_family, &tmp.peer_addr, tb[IFA_ADDRESS]);
	} else {
		__nc_nla_get_addr(ifa->ifa_family, &tmp.local_addr, tb[IFA_ADDRESS]);
		__nc_nla_get_addr(ifa->ifa_family, &tmp.bcast_addr, tb[IFA_BROADCAST]);
	}
	__nc_nla_get_addr(ifa->ifa_family, &tmp.anycast_addr, tb[IFA_ANYCAST]);

	*ap = nc_address_new(ifa->ifa_family, ifa->ifa_prefixlen, &tmp.local_addr);
	if(!*ap) {
		nc_error("Cannot create address from netlink message: %m");
		return NL_SKIP;
	}
	(*ap)->scope = ifa->ifa_scope;
	(*ap)->flags = ifa->ifa_flags;
	(*ap)->peer_addr = tmp.peer_addr;
	(*ap)->bcast_addr = tmp.bcast_addr;
	(*ap)->anycast_addr = tmp.anycast_addr;
	*ifindex = ifa->ifa_index;
#if 0
	{
		nc_stringbuf_t addr = NC_STRINGBUF_INIT;
		char ifname[IF_NAMESIZE + 1] = {'\0'};
		if_indextoname(*ifindex, ifname);
		nc_trace("interface %s local: %s", ifname, nc_address_format(&tmp.local_addr, &addr));
		nc_trace("interface %s peer : %s", ifname, nc_address_format(&tmp.peer_addr, &addr));
		nc_trace("interface %s brd  : %s", ifname, nc_address_format(&tmp.bcast_addr, &addr));
		nc_trace("interface %s any  : %s", ifname, nc_address_format(&tmp.anycast_addr, &addr));
		nc_stringbuf_destroy(&addr);
	}
#endif

	return NL_OK;
}

int
nc_netlink_get_addrs(nc_netlink_t *nl, nl_recvmsg_msg_cb_t func, void *data)
{
	return __nc_nl_dump(nl, AF_UNSPEC, RTM_GETADDR, func, data);
}

int
nc_netlink_get_routes(nc_netlink_t *nl, nl_recvmsg_msg_cb_t func, void *data)
{
	return __nc_nl_dump(nl, AF_UNSPEC, RTM_GETROUTE, func, data);
}

