/*
 *	network related utilities
 *
 *	Copyright (C) 2009-2010 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2021 SUSE LCC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Olaf Kirch
 *		Marius Tomaschewski
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <assert.h>

#include <sutils.h>
#include <nutils.h>
#include <logging.h>


#define NC_ADDRESS_ARRAY_CHUNK		16
#define NC_ROUTE_ARRAY_CHUNK		16

static const unsigned char *	__nc_address_data(const nc_sockaddr_t *ss, unsigned int *len);
static int			__nc_address_info(int af, unsigned int *off, unsigned int *len);
static int			__nc_address_array_realloc(nc_address_array_t *array,
								unsigned int newsize);

static void			__nc_route_free(nc_route_t *rp);
static int			__nc_route_array_realloc(nc_route_array_t *array,
								unsigned int newsize);


/*** hw address ***/
int
nc_hw_address_parse(nc_hwaddr_t *hwa, unsigned int type, const char *macstr)
{
	int len;

	memset(hwa, 0, sizeof(*hwa));
	switch (type) {
		/* black list */
		case NC_IFTYPE_TUNNEL:
		case NC_IFTYPE_SIT:
		case NC_IFTYPE_GRE:
		case NC_IFTYPE_TUNNEL6:
			nc_error("setting addrs not yet implemented for %d", type);
			return -1;
		break;

		/* white list */
		case NC_IFTYPE_INFINIBAND:
		case NC_IFTYPE_ETHERNET:
		case NC_IFTYPE_BRIDGE:
		case NC_IFTYPE_BOND:
		case NC_IFTYPE_VLAN:
		case NC_IFTYPE_WIRELESS:
		case NC_IFTYPE_TOKENRING:
		case NC_IFTYPE_DUMMY:
		case NC_IFTYPE_TAP:
		break;

		case NC_IFTYPE_LOOPBACK:
		default:
			return 0;
		break;
	}

	len = nc_parse_hex(macstr, hwa->data, sizeof(hwa->data));
	if(len < 0)
		return len;

	hwa->type = type;
	hwa->len = len;
	return 0;
}

const char *
nc_hw_address_format(const nc_hwaddr_t *hwa, nc_stringbuf_t *nsb)
{
	char buf[128] = {'\0'};

	if(!hwa || !nsb)
		return NULL;

	nc_stringbuf_clear(nsb);
	switch (hwa->type) {
		case NC_IFTYPE_TUNNEL:
		case NC_IFTYPE_SIT:
		case NC_IFTYPE_GRE:
		/*
			if (inet_ntop(AF_INET, hwa->data, buf, siz) == 0)
				return -1;
		break;
		*/
		case NC_IFTYPE_TUNNEL6:
		/*
			if (inet_ntop(AF_INET6, hwa->data, buf, siz) == 0)
				return -1;
		*/
			nc_error("%s: formating addrs not yet implemented for %d",
			    __FUNCTION__, hwa->type);
		break;

		case NC_IFTYPE_ETHERNET:
		case NC_IFTYPE_BRIDGE:
		case NC_IFTYPE_BOND:
		case NC_IFTYPE_VLAN:
		case NC_IFTYPE_WIRELESS:
		case NC_IFTYPE_TOKENRING:
		case NC_IFTYPE_DUMMY:
		case NC_IFTYPE_TAP:
		case NC_IFTYPE_INFINIBAND:
			nc_format_hex(hwa->data, hwa->len, buf, sizeof(buf));
			nc_stringbuf_puts(nsb, buf);
			if(nsb->len > 0)
				return nsb->string;
		break;

		case NC_IFTYPE_LOOPBACK:
		default:
		break;
	}

	return NULL;
}

int /* bool */
nc_hw_address_equal(const nc_hwaddr_t *hwa1, const nc_hwaddr_t *hwa2)
{
	if (hwa1->type != hwa2->type || hwa1->len != hwa2->len)
		return 0;
	return !memcmp(hwa1->data, hwa2->data, hwa1->len);
}

/*** ip address ***/
nc_address_t *
nc_address_new(int af, unsigned int prefix_len, const nc_sockaddr_t *local_addr)
{
	nc_address_t *ap;

	assert(!local_addr || local_addr->ss_family == af);

	ap = calloc(1, sizeof(*ap));
	assert(ap);
	if(!ap)
		return NULL;

	ap->config_method = NC_ADDRCONF_STATIC;
	ap->family = af;
	ap->prefixlen = prefix_len;
	ap->scope = -1;
	if (local_addr) {
		ap->local_addr = *local_addr;
	}

	/* FIXME: is this the right place to do this? */
	if (af == AF_INET && local_addr) {
		ap->bcast_addr = *local_addr;
		ap->bcast_addr.sin.sin_addr.s_addr |= htonl(0xFFFFFFFFUL >> prefix_len);
	}

	/* FIXME: we need to do this as long as we don't track the IPv6
	 * prefixes received via RAs. */
	if (af == AF_INET6) {
		const unsigned char *data;
		unsigned int len;

		data = __nc_address_data(&ap->local_addr, &len);
		if (data && data[0] == 0xFE && data[1] == 0x80) {
			/* Link-local; always autoconf */
			ap->config_method = NC_ADDRCONF_AUTOCONF;
		}
	}

	return ap;
}

nc_address_t *
nc_address_clone(const nc_address_t *src)
{
	nc_address_t *dst;
	if (!src)
		return NULL;
	dst = malloc(sizeof(*dst));
	assert(dst);
	if(dst) {
		memcpy(dst, src, sizeof(*dst));
	}
	return dst;
}

void
nc_address_free(nc_address_t *ap)
{
	free(ap);
}

unsigned int
nc_address_bytes(int af)
{
	switch (af) {
	case AF_INET:
		return 4;
	break;
	case AF_INET6:
		return 16;
	break;
	default:
	break;
	}
	return 0;
}

unsigned int
nc_address_bits(int af)
{
	return nc_address_bytes(af) * 8;
}

int
nc_address_parse(nc_sockaddr_t *ss, const char *string, int af)
{
	memset(ss, 0, sizeof(*ss));

	if (!string || !*string)
		return -1;
	
	if ((af == AF_UNSPEC || af == AF_INET6) &&
	    inet_pton(AF_INET6, string, &ss->six.sin6_addr) == 1) {
		ss->ss_family = AF_INET6;
		return 0;
	}

	if ((af == AF_UNSPEC || af == AF_INET) &&
	    inet_pton(AF_INET, string, &ss->sin.sin_addr) == 1) {
		ss->ss_family = AF_INET;
		return 0;
	}

	/*
	 * This may be something like "127" or "10.2", which are sometimes
	 * used as shorthand in ifcfg and ifroute files; see inet_aton(3).
	 */
	 if ((af == AF_UNSPEC || af == AF_INET) &&
	     inet_aton(string, &ss->sin.sin_addr) != 0) {
		 ss->ss_family = AF_INET;
		 return 0;
	}

	return -1;
}

const char *
nc_address_format(const nc_sockaddr_t *ss, nc_stringbuf_t *nsb)
{
	char buf[INET6_ADDRSTRLEN + 1] = {'\0'};

	if(!ss || !nsb)
		return NULL;

	nc_stringbuf_clear(nsb);
	switch (ss->ss_family) {
	case AF_INET6:
		if(inet_ntop(AF_INET6, &ss->six.sin6_addr, buf, sizeof(buf))) {
			nc_stringbuf_puts(nsb, buf);
		}
	break;
	case AF_INET:
		if(inet_ntop(AF_INET, &ss->sin.sin_addr, buf, sizeof(buf))) {
			nc_stringbuf_puts(nsb, buf);
		}
	default:
	break;
	}
	return nsb->string;
}

int
nc_address_prefix_parse(nc_sockaddr_t *ss, unsigned int *plen, const char *string, int af)
{
	char *tmp = NULL;
	char *ptr;

	if (nc_string_dup(&tmp, string) || !tmp)
		return  -1;

	ptr = strchr(tmp, '/');
	if (ptr) {
		*ptr = '\0';
		if (nc_parse_uint(ptr + 1, plen, 10))
			goto failure;
	}

	if (nc_address_parse(ss, tmp, af))
		goto failure;

	if (!ptr)
		*plen = nc_address_bits(ss->ss_family);
	else
	if (*plen > nc_address_bits(ss->ss_family))
		goto failure;

	free(tmp);
	return 0;

failure:
	free(tmp);
	return -1;
}

int /* bool */
nc_address_equal(const nc_sockaddr_t *ss1, const nc_sockaddr_t *ss2)
{
	const unsigned char *ap1, *ap2;
	unsigned int len;

	if (ss1->ss_family != ss2->ss_family)
		return 0;
	if (ss1->ss_family == AF_UNSPEC)
		return 1;

	ap1 = __nc_address_data(ss1, &len);
	ap2 = __nc_address_data(ss2, &len);
	if (!ap1 || !ap2 || ss1->ss_family != ss2->ss_family)
		return 0;

	return !memcmp(ap1, ap2, len);
}

int /* bool */
nc_address_prefix_match(unsigned int prefix_bits, const nc_sockaddr_t *laddr,
                                                  const nc_sockaddr_t *gw)
{
	const unsigned char *laddr_ptr, *gw_ptr;
	unsigned int offset = 0, len;
	unsigned int cc;

	laddr_ptr = __nc_address_data(laddr, &len);
	gw_ptr = __nc_address_data(gw, &len);

	if (!laddr_ptr || !gw_ptr || laddr->ss_family != gw->ss_family)
		return 0;

	if (prefix_bits > (len * 8))
		prefix_bits = len * 8;

	if (prefix_bits > 8) {
		if (memcmp(laddr_ptr, gw_ptr, prefix_bits / 8))
			return 0;
		offset = prefix_bits / 8;
		prefix_bits = prefix_bits % 8;
	}

	/* If the prefix length is not a multiple of 8, we need to check the
	 * top N bits of the next octet. */
	if (prefix_bits != 0) {
		cc = laddr_ptr[offset] ^ gw_ptr[offset];
		if ((0xFF00 & (cc << prefix_bits)) != 0)
			return 0;
	}

	return 1;
}

int /* bool */
nc_address_can_reach(const nc_address_t *laddr, const nc_sockaddr_t *gw)
{
	if (!laddr || !gw || laddr->family != gw->ss_family)
		return 0;
	if (laddr->family == AF_UNSPEC || gw->ss_family == AF_UNSPEC)
		return 0;
	return nc_address_prefix_match(laddr->prefixlen, &laddr->local_addr, gw);
}

int /* bool */
nc_address_can_reach_one(const nc_address_array_t *laddrs, const nc_sockaddr_t *gw)
{
	unsigned int i;

	if(!laddrs || !gw)
		return 0;

	for(i = 0; i < laddrs->count; ++i) {
		const nc_address_t *ap =  laddrs->data[i];
		if(!ap)
			continue;
		if(nc_address_can_reach(ap, gw))
			return 1;
	}
	return 0;
}

static const unsigned char *
__nc_address_data(const nc_sockaddr_t *ss, unsigned int *len)
{
	unsigned int offset;

	*len = 0;
	if (ss == NULL)
		return NULL;
	
	if (!__nc_address_info(ss->ss_family, &offset, len))
		return NULL;
	
	return ((const unsigned char *) ss) + offset;
}

static int
__nc_address_info(int af, unsigned int *offset, unsigned int *len)
{
	switch (af) {
	case AF_INET:
		*offset = offsetof(struct sockaddr_in, sin_addr);
		*len = nc_address_bytes(af);
		return 1;
	break;
	case AF_INET6:
		*offset = offsetof(struct sockaddr_in6, sin6_addr);
		*len = nc_address_bytes(af);
		return 1;
	break;
	default:
	break;
	}
	return 0;
}

/* ip address netmask */
int
nc_netmask_build(int af, unsigned int prefix_len, nc_sockaddr_t *mask)
{
	unsigned int    offset, len, i, bits;
	unsigned char   *raw;

	memset(mask, 0, sizeof(*mask));
	mask->ss_family = af;

	if (!__nc_address_info(af, &offset, &len))
		return -1;
	
	raw = &((unsigned char *) mask)[offset];
	for (i = 0; i < len && prefix_len != 0; ++i) {
		bits = (prefix_len < 8) ? prefix_len : 8;
		*raw++ = 0xFF00 >> bits;
		prefix_len -= bits;
	}
	return prefix_len ? -1 : 0;
}

unsigned int
nc_netmask_bits(const nc_sockaddr_t *mask)
{
	unsigned int    offset, len, i, bits = 0;
	unsigned char   *raw;

	if (!__nc_address_info(mask->ss_family, &offset, &len))
		return 0;
	
	raw = &((unsigned char *) mask)[offset];
	for (i = 0; i < len; ++i) {
		unsigned char cc = *raw++;
		if (cc == 0xFF) {
			bits += 8;
		} else {
			while (cc & 0x80) {
				bits++;
				cc <<= 1;
			}
			break;
		}
	}

	return bits;
}

/*** ip address array ***/
void
nc_address_array_init(nc_address_array_t *array)
{
	if(array) {
		memset(array, 0, sizeof(*array));
	}
}

int
nc_address_array_copy(nc_address_array_t *dst, const nc_address_array_t *src)
{
	unsigned int i;

	assert(src && dst);
	nc_address_array_destroy(dst);
	for (i = 0; i < src->count; ++i) {
		nc_address_t *ap;
		ap = nc_address_clone(src->data[i]);
		if(!ap || (nc_address_array_append(dst, ap) < 0)) {
			nc_address_free(ap);
			nc_address_array_destroy(dst);
			return -1;
		}
	}
	return 0;
}

void
nc_address_array_destroy(nc_address_array_t *array)
{
	if(array) {
		while(array->count--) {
			nc_address_free(array->data[array->count]);
		}
		free(array->data);
		nc_address_array_init(array);
	}
}

static int
__nc_address_array_realloc(nc_address_array_t *array, unsigned int newsize)
{
	nc_address_t **newdata;
	unsigned int i;

	newsize = (newsize + NC_ADDRESS_ARRAY_CHUNK);
	newdata = realloc(array->data, newsize * sizeof(nc_address_t *));
	assert(newdata != NULL);
	if (!newdata)
		return -1;

	array->data = newdata;
	for (i = array->count; i < newsize; ++i) {
		array->data[i] = NULL;
	}
	return 0;
}

int
nc_address_array_append (nc_address_array_t *array, nc_address_t *ap)
{
	if(!array || !ap)
		return -1;
	if((array->count % NC_ADDRESS_ARRAY_CHUNK) == 0) {
		if(__nc_address_array_realloc(array, array->count) != 0)
			return -1;
	}
	array->data[array->count++] = ap;
	return 0;
}

/*** ip route ***/
nc_route_t *
nc_route_new(void)
{
	nc_route_t *rp;

	rp = calloc(1, sizeof(*rp));
	assert(rp);
	if(rp) {
		rp->users = 1;
	}
	return rp;
}

nc_route_t *
nc_route_create(unsigned int prefixlen, const nc_sockaddr_t *dest,
		const nc_sockaddr_t *gw, const char *ifname, const char *opts)
{
	static const nc_sockaddr_t null_addr;
	nc_route_t *rp;
	int af;

	if (!dest)
		dest = &null_addr;
	if (!gw)
		gw = &null_addr;

	af = dest->ss_family;
	if (gw->ss_family == AF_UNSPEC) {
		/*
		 * No gateway - this is a direct subnet route.
		 * Make sure sure the destination is not the default
		 * and we have an interface
		 */
		if (af == AF_UNSPEC) {
			nc_error("Cannot create route - destination and gw are both 0/0");
			return NULL;
		}
		if(!nc_string_len(ifname)) {
			nc_error("Cannot create route - interface route requires an interface");
			return NULL;
		}
	} else {
		if(af == AF_UNSPEC) {
			af = gw->ss_family;
		} else
		if (dest->ss_family != gw->ss_family) {
			nc_error("Cannot create route - destination and gateway address "
					"family mismatch");
			return NULL;
		}
	}

	rp = nc_route_new();
	if(rp) {
		rp->config_method = NC_ADDRCONF_STATIC;
		rp->family = af;
		rp->prefixlen = prefixlen;
		rp->destination = *dest;
		if (rp->destination.ss_family == AF_UNSPEC) {
			memset(&rp->destination, 0, sizeof(rp->destination));
			rp->destination.ss_family = af;
		}
		rp->nh.gateway = *gw;
		if(nc_string_len(ifname) && nc_string_dup(&rp->nh.device, ifname) < 0) {
			nc_route_free(rp);
			rp = NULL;
		}
		if(opts && *opts && nc_string_dup(&rp->opts, opts) < 0) {
			nc_route_free(rp);
			rp = NULL;
		}
	}
	return rp;
}

nc_route_t *
nc_route_clone(const nc_route_t *rp)
{
	nc_route_t         *clone;
	const nc_route_nexthop_t *hp;
	nc_route_nexthop_t *hop, **pos;

	if(!rp)
		return NULL;

	clone = nc_route_new();
	if(!clone)
		return NULL;

	memcpy(clone, rp, sizeof(*clone));
	clone->users = 1;
	clone->opts = NULL;
	if(nc_string_dup(&clone->opts, rp->opts) != 0) {
		goto failure;
	}

	clone->nh.next = NULL;
	clone->nh.device = NULL;
	if(nc_string_dup(&clone->nh.device, rp->nh.device) != 0) {
		goto failure;
	}
	pos = &clone->nh.next;
	for(hp = rp->nh.next; hp; hp=hp->next) {
		hop = malloc(sizeof(*hop));
		if(!hop)
			goto failure;

		memcpy(hop, hp, sizeof(*hop));
		hop->next = NULL;
		hop->device = NULL;
		if(nc_string_dup(&hop->device, hp->device) != 0) {
			free(hop);
			goto failure;
		}

		*pos = hop;
		pos = &hop->next;
	}

	return clone;

failure:
	__nc_route_free(clone);
	return NULL;
}

nc_route_t *
nc_route_ref(nc_route_t *rp)
{
	assert(rp && rp->users != 0);
	if(rp) {
		if(!rp->users)
			return NULL;
		rp->users++;
	}
	return rp;
}

unsigned int
nc_route_free(nc_route_t *rp)
{
	if(!rp)
		return 0;

	assert(rp->users);
	if(!rp->users) {
		nc_error("nc_route_free: bad reference");
		return 0;
	}
	rp->users--;
	if (rp->users == 0) {
		__nc_route_free(rp);
		return 0;
	}
	return rp->users;
}

static void
__nc_route_free(nc_route_t *rp)
{
	if(rp) {
		nc_route_nexthop_t* hp;
		while( (hp = rp->nh.next) != NULL) {
			rp->nh.next = hp->next;
			nc_string_free(&hp->device);
			free(hp);
		}
		nc_string_free(&rp->nh.device);
		nc_string_free(&rp->opts);
		free(rp);
	}
}

/*** ip route array ***/
void
nc_route_array_init(nc_route_array_t *array)
{
	if(array) {
		memset(array, 0, sizeof(*array));
	}
}

void
nc_route_array_destroy(nc_route_array_t *array)
{
	if(array) {
		while(array->count--) {
			nc_route_free(array->data[array->count]);
		}
		free(array->data);
		nc_route_array_init(array);
	}
}

static int
__nc_route_array_realloc(nc_route_array_t *array, unsigned int newsize)
{
	nc_route_t **newdata;
	unsigned int i;

	newsize = (newsize + NC_ROUTE_ARRAY_CHUNK);
	newdata = realloc(array->data, newsize * sizeof(nc_route_t *));
	assert(newdata != NULL);
	if (!newdata)
		return -1;

	array->data = newdata;
	for (i = array->count; i < newsize; ++i) {
		array->data[i] = NULL;
	}
	return 0;
}

int
nc_route_array_copy(nc_route_array_t *dst, const nc_route_array_t *src)
{
	unsigned int i;

	assert(src && dst);
	if(!src || !dst)
		return -1;

	nc_route_array_destroy(dst);
	for (i = 0; i < src->count; ++i) {
		nc_route_t *rp = nc_route_clone(src->data[i]);
		if(!rp || (nc_route_array_append(dst, rp) < 0)) {
			if(rp)
				nc_route_free(rp);
			nc_route_array_destroy(dst);
			return -1;
		}
	}
	return 0;
}

int
nc_route_array_append (nc_route_array_t *array, nc_route_t *rp)
{
	if(!array || !rp)
		return -1;

	if((array->count % NC_ROUTE_ARRAY_CHUNK) == 0) {
		if(__nc_route_array_realloc(array, array->count) != 0)
			return -1;
	}

	array->data[array->count++] = rp;
	return 0;
}

int
nc_route_array_remove_index(nc_route_array_t *array, unsigned int pos)
{
	if(!array || pos >= array->count)
		return -1;
	
	nc_route_free(array->data[pos]);
	/* Note: this also copies the NULL pointer following the last element */
	memmove(&array->data[pos], &array->data[pos + 1],
		(array->count - pos) * sizeof(nc_route_t *));
	array->count--;

	/* Don't bother with shrinking the array. It's not worth the trouble */
	return 0;
}

