/*
 *	internal handle
 *
 *	Copyright (C) 2009-2010 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2021 SUSE LCC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Olaf Kirch
 *		Marius Tomaschewski
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 *
 */
#ifndef __NETCONTROL_HANDLE_H
#define __NETCONTROL_HANDLE_H
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif

#include "ntypes.h"
#include "service.h"

typedef struct nc_handle
{
	nc_interface_array_t	ifaces;
	nc_route_array_t	routes;

	struct nc_ops *		op;

	char *			err_last;
	nc_string_array_t	err_once;
} nc_handle_t;

struct nc_ops {
	int		(*interfaces_refresh)(nc_handle_t *);
	int		(*interfaces_configure)(nc_handle_t *, nc_interface_t *);
	int		(*interfaces_delete)(nc_handle_t *, nc_interface_t *);
	int		(*interface_status)(nc_handle_t *, nc_interface_t *);
	int		(*interface_start)(nc_handle_t *, nc_interface_t *);
	int		(*interface_stop)(nc_handle_t *, nc_interface_t *);
	void		(*close)(nc_handle_t *);
};

nc_handle_t *		nc_indirect_open(const char *ns_path, const char *root_dir, nc_service_t *service);
void			nc_close(nc_handle_t *nh);

#define			nc_error_detail_fmt(nh, fmt, args...) \
			__nc_error_detail_fmt(__FUNCTION__,__FILE__,__LINE__,nh,fmt,##args)

void			__nc_error_detail_fmt(const char *func, const char *file, long long line,
						nc_handle_t *nh, const char *fmt, ...);
void			nc_error_detail_move(nc_handle_t *nh, char **errstr);
void			nc_error_detail_clear(nc_handle_t *nh);

int /* bool */		nc_error_once_check(nc_handle_t *nh, const char *once_ctx, ...);
void			nc_error_once_clear(nc_handle_t *nh, const char *once_ctx, ...);

void			nc_bad_reference(nc_handle_t *nh, const nc_interface_t *referrer,
				const char *ifname);

nc_interface_t *	nc_interface_new(const char *name, unsigned int index);
nc_interface_t *	nc_interface_ref(nc_interface_t *ifp);
int			nc_interface_free(nc_interface_t *ifp);
nc_interface_t *	nc_interface_clone(const nc_interface_t *ofp);
void			nc_interface_reset(nc_interface_t *ifp);

nc_interface_t *	nc_interface_first(nc_handle_t *nh, unsigned int *pos);
nc_interface_t *	nc_interface_next(nc_handle_t *nh, unsigned int *pos);
nc_interface_t *	nc_interface_find(nc_handle_t *nh, unsigned int *pos,
			       			int ifindex, const char *ifname,
					       	int modified, int deleted);

nc_interface_t *	nc_interface_by_name(nc_handle_t *nh, const char *ifname);
nc_interface_t *	nc_interface_by_ifname(nc_handle_t *nh, const char *ifname);
nc_interface_t *	nc_interface_by_ifindex(nc_handle_t *nh, unsigned int ifindex);

int			nc_interface_device_is_up(const nc_interface_t *ifp);
int			nc_interface_link_is_up(const nc_interface_t *ifp);
int			nc_interface_network_is_up(const nc_interface_t *ifp);

int			nc_interfaces_refresh(nc_handle_t *nh);
int			nc_interfaces_create_topology(nc_handle_t *nh, nc_var_array_t *);
int			nc_interfaces_configure(nc_handle_t *nh, const nc_interface_t *cfg);
int			nc_interfaces_delete(nc_handle_t *nh, const nc_interface_t *cfg);

int			nc_interface_status(nc_handle_t *nh, nc_interface_t *ifp);
int			nc_interface_start(nc_handle_t *nh, nc_interface_t *ifp);
int			nc_interface_stop(nc_handle_t *nh, nc_interface_t *ifp);

int			nc_interfaces_add(nc_handle_t *nh, nc_interface_t *ifp);
void			nc_interfaces_clear(nc_handle_t *nh);

nc_interface_array_t *	nc_interface_array_new    (void);
void			nc_interface_array_free   (nc_interface_array_t *array);

void			nc_interface_array_init   (nc_interface_array_t *array);
void			nc_interface_array_destroy(nc_interface_array_t *array);
int			nc_interface_array_append (nc_interface_array_t *array,
			                           nc_interface_t *ifp);
int			nc_interface_array_index_by_name(const nc_interface_array_t *array,
			                           	 const char *name);
int			nc_interface_array_remove_index(nc_interface_array_t *array,
			                                unsigned int pos);

nc_interface_t *	nc_interface_array_first(nc_interface_array_t *array,
	       					unsigned int *pos);
nc_interface_t *	nc_interface_array_next (nc_interface_array_t *array,
	       					unsigned int *pos);

nc_interface_t *	nc_interface_array_find (nc_interface_array_t *array,
	       					unsigned int *pos,
					       	int ifindex, const char *ifname,
					       	int modified, int deleted);

#endif /* __NETCONTROL_HANDLE_H */
