/*
 * This file is part of CycloneDX Rust Cargo.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 */

use once_cell::sync::Lazy;
use regex::Regex;

use crate::external_models::normalized_string::NormalizedString;
use crate::validation::{FailureReason, Validate, ValidationContext, ValidationResult};

/// Defines how a component or service is affected by a vulnerability as described in the [CycloneDX use cases](https://cyclonedx.org/use-cases/#vulnerability-exploitability)
///
/// Defined via the [XML schema](https://cyclonedx.org/docs/1.4/xml/#type_vulnerabilityType)
#[derive(Clone, Debug, PartialEq, Eq)]
pub struct VulnerabilityTarget {
    pub bom_ref: String,
    pub versions: Option<Versions>,
}

impl VulnerabilityTarget {
    /// Construct a `VulnerabilityTarget` be referring to a component or service via a BOM reference
    /// ```
    /// use cyclonedx_bom::models::vulnerability_target::VulnerabilityTarget;
    ///
    /// let target = VulnerabilityTarget::new("12a34a5b-6780-1bae-2345-67890cfe12a3".to_string());
    /// ```
    pub fn new(bom_ref: String) -> Self {
        Self {
            bom_ref,
            versions: None,
        }
    }
}

impl Validate for VulnerabilityTarget {
    fn validate_with_context(&self, context: ValidationContext) -> ValidationResult {
        let mut results: Vec<ValidationResult> = vec![];

        if let Some(versions) = &self.versions {
            let context = context.with_struct("VulnerabilityTarget", "versions");

            results.push(versions.validate_with_context(context));
        }

        results
            .into_iter()
            .fold(ValidationResult::default(), |acc, result| acc.merge(result))
    }
}

#[derive(Clone, Debug, PartialEq, Eq)]
pub struct VulnerabilityTargets(pub Vec<VulnerabilityTarget>);

impl Validate for VulnerabilityTargets {
    fn validate_with_context(&self, context: ValidationContext) -> ValidationResult {
        let mut results: Vec<ValidationResult> = vec![];

        for (index, vulnerability_target) in self.0.iter().enumerate() {
            let context = context.with_index(index);
            results.push(vulnerability_target.validate_with_context(context));
        }

        results
            .into_iter()
            .fold(ValidationResult::default(), |acc, result| acc.merge(result))
    }
}

#[derive(Clone, Debug, PartialEq, Eq)]
pub struct Versions(pub Vec<Version>);

impl Validate for Versions {
    fn validate_with_context(&self, context: ValidationContext) -> ValidationResult {
        let mut results: Vec<ValidationResult> = vec![];

        for (index, version) in self.0.iter().enumerate() {
            let context = context.with_index(index);
            results.push(version.validate_with_context(context));
        }

        results
            .into_iter()
            .fold(ValidationResult::default(), |acc, result| acc.merge(result))
    }
}

#[derive(Clone, Debug, PartialEq, Eq)]
pub struct Version {
    pub version_range: VersionRange,
    pub status: Status,
}

impl Version {
    /// Construct a `Version by providing a version and a status`
    /// ```
    /// use cyclonedx_bom::models::vulnerability_target::Version;
    ///
    /// let version = Version::new("1.0", "unaffected");
    /// ```
    pub fn new(version: &str, status: &str) -> Self {
        Version {
            version_range: VersionRange::new(version),
            status: Status::new_unchecked(status),
        }
    }
}

impl Validate for Version {
    fn validate_with_context(&self, context: ValidationContext) -> ValidationResult {
        let mut results: Vec<ValidationResult> = vec![];

        let version_range_context = context.with_struct("Version", "version_range");

        results.push(
            self.version_range
                .validate_with_context(version_range_context),
        );

        let status_context = context.with_struct("Version", "status");

        results.push(self.status.validate_with_context(status_context));

        results
            .into_iter()
            .fold(ValidationResult::default(), |acc, result| acc.merge(result))
    }
}

/// Specifies a single version or a version range.
///
/// Defined via the [PURL specification](https://github.com/package-url/purl-spec/blob/master/PURL-SPECIFICATION.rst)
/// Spec for version ranges still work in progress [PURL version-range-spec](https://github.com/package-url/purl-spec/blob/version-range-spec/VERSION-RANGE-SPEC.rst)
#[derive(Clone, Debug, PartialEq, Eq)]
pub enum VersionRange {
    Version(NormalizedString),
    Range(NormalizedString),
    #[doc(hidden)]
    UndefinedVersionRange(String),
}

impl VersionRange {
    pub fn new(value: &str) -> Self {
        match matches_purl_version_range_regex(value) {
            true => VersionRange::Range(NormalizedString::new(value)),
            false => VersionRange::Version(NormalizedString::new(value)),
        }
    }
}

impl Validate for VersionRange {
    fn validate_with_context(&self, context: ValidationContext) -> ValidationResult {
        match self {
            VersionRange::UndefinedVersionRange(_) => ValidationResult::Failed {
                reasons: vec![FailureReason {
                    message: "Undefined version range".to_string(),
                    context,
                }],
            },
            _ => ValidationResult::Passed,
        }
    }
}

impl ToString for VersionRange {
    fn to_string(&self) -> String {
        match self {
            VersionRange::Version(version) => version.to_string(),
            VersionRange::Range(range) => range.to_string(),
            VersionRange::UndefinedVersionRange(undefined) => undefined.to_string(),
        }
    }
}

fn matches_purl_version_range_regex(value: &str) -> bool {
    static PURL_VERSION_RANGE_REGEX: Lazy<Regex> =
        Lazy::new(|| Regex::new(r"^vers:.*$").expect("Failed to compile regex."));

    PURL_VERSION_RANGE_REGEX.is_match(value)
}

/// Specifies if a vulnerability affects a component or service.
///
/// Defined via the [XML schema](https://cyclonedx.org/docs/1.4/xml/#type_impactAnalysisAffectedStatusType)
#[derive(Clone, Debug, PartialEq, Eq)]
pub enum Status {
    Affected,
    Unaffected,
    Unknown,
    #[doc(hidden)]
    UndefinedStatus(String),
}

impl Status {
    pub(crate) fn new_unchecked<A: AsRef<str>>(value: A) -> Self {
        match value.as_ref() {
            "affected" => Self::Affected,
            "unaffected" => Self::Unaffected,
            "unknown" => Self::Unknown,
            undefined => Self::UndefinedStatus(undefined.to_string()),
        }
    }
}

impl Validate for Status {
    fn validate_with_context(&self, context: ValidationContext) -> ValidationResult {
        match self {
            Status::UndefinedStatus(_) => ValidationResult::Failed {
                reasons: vec![FailureReason {
                    message: "Undefined status".to_string(),
                    context,
                }],
            },
            _ => ValidationResult::Passed,
        }
    }
}

impl ToString for Status {
    fn to_string(&self) -> String {
        match self {
            Status::Affected => "affected",
            Status::Unaffected => "unaffected",
            Status::Unknown => "unknown",
            Status::UndefinedStatus(undefined) => undefined,
        }
        .to_string()
    }
}

#[cfg(test)]
mod test {
    use super::*;
    use pretty_assertions::assert_eq;

    #[test]
    fn valid_version_range() {
        assert_eq!(
            VersionRange::Version(NormalizedString("1.0".to_string())),
            Version::new("1.0", "unaffected").version_range,
        );
    }

    #[test]
    fn valid_vulnerability_targets_should_pass_validation() {
        let validation_result = VulnerabilityTargets(vec![VulnerabilityTarget {
            bom_ref: "bom ref".to_string(),
            versions: Some(Versions(vec![Version {
                version_range: VersionRange::Version(NormalizedString::new("1.0")),
                status: Status::Affected,
            }])),
        }])
        .validate();

        assert_eq!(validation_result, ValidationResult::Passed);
    }

    #[test]
    fn invalid_vulnerability_targets_should_fail_validation() {
        let validation_result = VulnerabilityTargets(vec![VulnerabilityTarget {
            bom_ref: "bom ref".to_string(),
            versions: Some(Versions(vec![Version {
                version_range: VersionRange::UndefinedVersionRange("invalid\tversion".to_string()),
                status: Status::UndefinedStatus("invalid\tstatus".to_string()),
            }])),
        }])
        .validate();

        assert_eq!(
            validation_result,
            ValidationResult::Failed {
                reasons: vec![
                    FailureReason::new(
                        "Undefined version range",
                        ValidationContext::new()
                            .with_index(0)
                            .with_struct("VulnerabilityTarget", "versions")
                            .with_index(0)
                            .with_struct("Version", "version_range")
                    ),
                    FailureReason::new(
                        "Undefined status",
                        ValidationContext::new()
                            .with_index(0)
                            .with_struct("VulnerabilityTarget", "versions")
                            .with_index(0)
                            .with_struct("Version", "status")
                    )
                ]
            }
        );
    }
}
