#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <usb.h>
#include "alvausb.h"
#include "../brl.h"
#include "../brl_driver.h"
#include "../sbllog.h"

static int alva_usb_scan ();

static usb_dev_handle *alva_usb = NULL;

static int interface_claimed = 0;

static int devnr = -1;

static unsigned char *prev_disp = NULL;

void brl_debug (unsigned int dbg)
{
  if (dbg == 5)
    return;
}

struct brlinfo identbrl (const char *name, const char *dev)
{
  struct brlinfo ret;

  memset (&ret, 0, sizeof (ret));
 /* only to avoid compiler warnings */
  dev=dev;

  if (!strcmp (name, "alvausb"))
   {
     int id = 0;

     if ((id = alva_usb_scan ()) >= 0)
      {
        int i = 0;

        printf ("got ID\n");

        while (alva_devs[i].name)
         {
           if (alva_devs[i].baud == id)
            {
              printf ("alva=%s %02x %02x\n", alva_devs[i].name,
                      alva_devs[i].baud, id);
              devnr = i;

              return alva_devs[i];
            }

           i++;
         }

      }

     return alva_devs[0];

   }

  return ret;

}

void initbrl (brldim * brl, const char *dev)
{
  int id = 0;

  if (!alva_usb)
   {
     if ((id = alva_usb_scan ()) < 0)
       return;
     if (alva_devs[devnr].baud != id)
      {
        printf ("alva: wrong ID 0x%02x\n", id);
        return;
      }

   }

  brl->usb = 1;
  brl->brl_fd = -1;
  brl->x = alva_devs[devnr].cols;
  brl->y = 1;
  brl->disp = (unsigned char *) malloc (brl->x * brl->y);
  prev_disp = (unsigned char *) malloc (brl->x * brl->y);

  /* just to avoid compiler warnings */
  if (dev)
    return;
}

void closebrl (brldim * brl)    /* close braille display */
{
  if (alva_usb)
   {
     printf ("close it\n");
     if (interface_claimed)
      {
        usb_release_interface (alva_usb, 0);
        printf ("alva: released\n");
        interface_claimed = 0;
      }

     usb_close (alva_usb);
     printf ("alva: usb_close\n");
   }

  brl->usb = 0;
  brl->brl_fd = -1;
  alva_usb = NULL;

  if (prev_disp)
   {
     free (prev_disp);
     prev_disp = NULL;
   }
  if (brl->disp)
   {
     free (brl->disp);
     brl->disp = NULL;
   }

  return;
}

void writebrl (brldim * dev)    /* write to braille display */
{
  int size;

  static unsigned char str[100] = {
    0x1B, 'B', 3, 3
  };
  int i = 0;

  if (!dev || !dev->disp)
   {
     return;
   }

  /* number of cells to write */
  str[2] = alva_devs[devnr].st_cells;
  str[3] = alva_devs[devnr].cols;
  memcpy (str + 4, dev->disp, dev->x);
  for (i = 4; i < (dev->x + 4); i++)
    str[i] = transtbl[dev->disp[i - 4]];
  str[dev->x + 4] = 0x0d;
  /* round size to a multiple of 8 */
  size = dev->x + 4;
/*  printf("write size=%d\n",size);*/
  if (size % 8 > 0)
   {
     size = size / 8;
     size += 1;
     size = size * 8;
   }

  if (!alva_usb)
   {
     printf ("alva: write null\n");
     return;
   }
  if (usb_bulk_write (alva_usb, 0x2, (char *) str, size, 20) < 8)
   {
     printf ("error writing data %d\n", errno);
     return;
   }

  /* dirty hack - each usb operation triggers errno 11 */
  if (errno == 11)
    errno = 0;

  return;
}

int readbrl (int *type)         /* get key press from braille display */
{
  int size;

  unsigned char data[255];

  memset (data, 0, sizeof (data));

  if (!alva_usb)
   {
     printf ("alva: not connected");
     return EOF;
   }
  size = usb_bulk_read (alva_usb, 0x81, (char *) data, 8, 50);
  if (errno == 11)
   {
     errno = 0;
   }
  if (size < 8)
   {
     *type = 0;
     return EOF;
   }
/*  printf ("read: read %d byte %s\n", size,data);*/

  if (data[1] & 0x80)           /* a key is released */
   {
     *type = 0;
   }
  else
    *type = 1;

  switch (data[0])
   {
   case 0x71:                  /* front keys */
     /* bottom line routing */
     if (data[1] >= 0x20 && data[1] <= 0x24)
       return (data[1] - 0x20) + 100;
     else
       /* top line routing */
     if (data[1] >= 0x30 && data[1] <= 0x34)
       return (data[1] - 0x30) + 200;

     switch (data[1])
      {
      case 0:                  /* PROG1 */
      case 0x80:
        return EOF;
      case 1:                  /* HOME1 */
      case 0x81:
        return 1;
      case 2:                  /* CUR1 */
      case 0x82:
        return 6;
      case 3:                  /* UP */
      case 0x83:
        return 3;
      case 4:                  /* LEFT */
      case 0x84:
        return 2;
      case 5:                  /* RIGHT */
      case 0x85:
        return 5;
      case 6:                  /* DOWN */
      case 0x86:
        return 4;
      default:
        return EOF;
      }
     return EOF;
   case 0x72:                  /* Cursor - bottom line */
     if (data[1] < 0x80)
       return data[1] + 100 + alva_devs[devnr].st_cells;
     break;
   case 0x75:                  /* Cursor - top line */
     if (data[1] < 0x80)
       return data[1] + 200 + alva_devs[devnr].st_cells;
     break;
   case 0x77:                  /* Navigation keys */
     /* left navig : 1 0 2 4 5 3

        rigth navig use the same codes + 0x20 */
     switch (data[1])
      {
      case 0:
      case 0x80:
        return 20;
      case 0x20:
      case 0xa0:
        return 30;
      case 1:
      case 0x81:
        return 21;
      case 0x21:
      case 0xa1:
        return 31;
      case 2:
      case 0x82:
        return 22;
      case 0x22:
      case 0xa2:
        return 32;
      case 3:
      case 0x83:
        return 23;
      case 0x23:
      case 0xa3:
        return 33;
      case 4:
      case 0x84:
        return 24;
      case 0x24:
      case 0xa4:
        return 34;
      case 5:
      case 0x85:
        return 25;
      case 0x25:
      case 0xa5:
        return 35;
      default:
        return EOF;
      }
     return EOF;
   }

  return EOF;
}

void setbrlstat (const unsigned char *txt)      /* set status cells */
{
  if (txt)
    return;
  return;
}

static int alva_usb_scan ()
{
  struct usb_bus *busses;

  struct usb_bus *bus;

  struct usb_device *usb_dev = NULL;


  int i;

  usb_init ();
  usb_find_busses ();
  usb_find_devices ();
  busses = usb_get_busses ();
  for (bus = busses; bus; bus = bus->next)
   {
     if (!bus->devices)
      {
        printf ("no bus\n");
        return -1;
      }

     for (usb_dev = bus->devices; usb_dev; usb_dev = usb_dev->next)
      {
        /* Check if this device is an Alva */
        /* printf ("vendor = %0x\n", usb_dev->descriptor.idVendor); */
        if (usb_dev->descriptor.idVendor != ALVA_VID)
          continue;
        else
         {
           switch (usb_dev->descriptor.idProduct)
            {
            case ALVA_PID:
              break;

            default:
              printf ("Unknown Alva USB display - %d",
                      usb_dev->descriptor.idProduct);
            }
           break;
         }
      }
     if (usb_dev)
       break;
   }
  if (!usb_dev)
   {
     /* No Alva USB display detected\n */
     return -1;
   }

  /* open device */
  alva_usb = usb_open (usb_dev);
  if (!alva_usb)
    return -2;

  /* device opened successfully */

  usleep (50000);
  if (usb_reset (alva_usb) < 0)
   {
     /* printf ("usb_reset error\n"); */
     return -3;
   }

  /* Set configuration */
  if (usb_set_configuration (alva_usb, 1) < 0)
   {
     /* printf ("Error setting configuration: %s\n", strerror (errno)); */
     return -4;
   }

  /* Claiming the device */
  usleep (50000);
  if (usb_claim_interface (alva_usb, 0) < 0)
   {
     /* Error claiming interface */
     return 6;
   }
  interface_claimed = 1;
  {
    unsigned char initmsg[] = INITMSG;

    if (usb_bulk_write
        (alva_usb, 0x2, (char *) initmsg, sizeof (initmsg),
         255) < (int) sizeof (initmsg))
     {
/*       printf ("couldn't write on usb: %s\n", strerror (errno));*/
       return 7;
     }
  }

  /* detect model */
  for (i = 0; i < 10; i++)
   {
     unsigned char initmsg[] = INITMSG;

     unsigned char answer[] = ID_REPLY;

     unsigned char data[8];

     int size;

     if (usb_bulk_write
         (alva_usb, 0x2, (char *) initmsg, sizeof (initmsg),
          255) < (int) sizeof (initmsg))
      {
        /* couldn't write on usb-dev */
        return -8;
      }
     size = usb_interrupt_read (alva_usb, 0x81, (char *) data, 8, 0);
     if (size < 6 || memcmp (answer, data, sizeof (answer)) != 0)
      {
        printf ("read %d bytes - invalid ID packet\n", size);
        continue;
      }
     else
      {
        printf ("read valid ID packet 0x%02x\n", data[sizeof (answer)]);
        return 0x0e;
        /* (int)data[sizeof (answer)]; */
      }
   }
  return -1;
}
