/*
 *	sysfs utilities
 *
 *	Copyright (C) 2009-2010 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *	MA  02110-1301  USA
 *
 *	Authors: Olaf Kirch <okir@suse.de>
 *	         Marius Tomaschewski <mt@suse.de>
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	You can find the wicked project at http://gitorious.org/wicked/.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <stdlib.h>

#include <sysfs.h>
#include <sutils.h>
#include <futils.h>
#include <logging.h>
#include <bridge.h>
#include <errno.h>

static int
__nc_sysfs_read_attr(const char *path, const char *name, const char *attr, char **result)
{
	char filename[PATH_MAX + 1];
	char buffer[NAME_MAX + 1] = {'\0'};
	FILE *fp;
	int ret = 0;

	nc_string_free(result);
	snprintf(filename, sizeof(filename), "%s/%s/%s", path, name, attr);

	if (!(fp = fopen(filename, "re"))) {
		if(errno != ENOENT)
			nc_warn("unable to open %s: %m", filename);
		return -1;
	}

	if (fgets(buffer, sizeof(buffer), fp) != NULL) {
		buffer[strcspn(buffer, "\n")] = '\0';
		ret = nc_string_dup(result, buffer);
	}
	fclose(fp);
	return ret;
}

static int
__nc_sysfs_read_list(const char *path, const char *name, const char *attr, nc_string_array_t *list)
{
	char filename[PATH_MAX + 1];
	char buffer[NAME_MAX + 1] = {'\0'};
	FILE *fp;
	int ret = 0;

	snprintf(filename, sizeof(filename), "%s/%s/%s", path, name, attr);

	if (!(fp = fopen(filename, "re"))) {
		nc_error("unable to open %s: %m", filename);
		return -1;
	}

	ret = list->count;
	while (fgets(buffer, sizeof(buffer), fp) != NULL) {
		char *s, *p = NULL;

		for (s = strtok_r(buffer, " \t\n", &p); s;
		     s = strtok_r(NULL, " \t\n", &p)) {
			if(nc_string_array_append(list, s) != 0)
				return -1;
		}
	}
	fclose(fp);
	return list->count - ret;
}

int /* bool */
nc_sysfs_netif_exists(const char *ifname, const char *attr)
{
	return nc_file_exists_fmt("%s/%s/%s", NC_SYSFS_CLASS_NET_PATH,
						ifname, attr);
}

int
nc_sysfs_netif_get_string(const char *ifname, const char *attr, char **value)
{

	return __nc_sysfs_read_attr(NC_SYSFS_CLASS_NET_PATH, ifname, attr, value);
}

int
nc_sysfs_netif_get_int(const char *ifname, const char *attr, int *value)
{
	char *tmp = NULL;
	int ret;

	if(__nc_sysfs_read_attr(NC_SYSFS_CLASS_NET_PATH, ifname, attr, &tmp) < 0)
		return -1;

	ret = nc_parse_int(tmp, value, 0);
	nc_string_free(&tmp);
	return ret;
}

int
nc_sysfs_netif_get_uint(const char *ifname, const char *attr, unsigned int *value)
{
	char *tmp = NULL;
	int ret;

	if(__nc_sysfs_read_attr(NC_SYSFS_CLASS_NET_PATH, ifname, attr, &tmp) < 0)
		return -1;
	
	ret = nc_parse_uint(tmp, value, 0);
	nc_string_free(&tmp);
	return ret;
}

int
nc_sysfs_netif_get_long(const char *ifname, const char *attr, long *value)
{
	char *tmp = NULL;
	int ret;

	if(__nc_sysfs_read_attr(NC_SYSFS_CLASS_NET_PATH, ifname, attr, &tmp) < 0)
		return -1;

	ret = nc_parse_long(tmp, value, 0);
	nc_string_free(&tmp);
	return ret;
}

int
nc_sysfs_netif_get_ulong(const char *ifname, const char *attr, unsigned long *value)
{
	char *tmp = NULL;
	int ret;

	if(__nc_sysfs_read_attr(NC_SYSFS_CLASS_NET_PATH, ifname, attr, &tmp) < 0)
		return -1;
	
	ret = nc_parse_ulong(tmp, value, 0);
	nc_string_free(&tmp);
	return ret;
}

int
nc_sysfs_netif_get_bus_id(const char *ifname, char **bus_id)
{
	char path[PATH_MAX + 1] = {'\0'};
	char *res = NULL, *ptr;

	snprintf(path, sizeof(path), "%s/%s/%s",
		NC_SYSFS_CLASS_NET_PATH, ifname, "device");

	if(nc_readlink(path, &res) == 0) {
		ptr = strrchr(res, '/');
		ptr = ptr ? ptr + 1 : res;

		if(nc_string_dup(bus_id, ptr) == 0) {
			free(res);
			return 0;
		}
		free(res);
	}
	return -1;
}

int
nc_sysfs_netif_get_linkpath(const char *name, const char *attr, char **res)
{
	char path[PATH_MAX + 1] = {'\0'};

	snprintf(path, sizeof(path), "%s/%s/%s",
		NC_SYSFS_CLASS_NET_PATH, name, attr);

	return nc_readlink(path, res);
}

int
nc_sysfs_netif_get_linkbase(const char *name, const char *attr, char **res)
{
	char *path = NULL, *ptr;

	if (nc_sysfs_netif_get_linkpath(name, attr, &path) == 0) {
		ptr = strrchr(path, '/');
		ptr = ptr ? ptr + 1 : path;

		if(nc_string_dup(res, ptr) == 0) {
			free(path);
			return 0;
		}
		free(path);
	}
	return -1;
}

int
nc_sysfs_get_bonding_parent(const char *name, char **parent)
{
	if (!nc_sysfs_netif_exists(name, "bonding_slave"))
		return -1;
	return nc_sysfs_netif_get_linkbase(name, "master", parent);
}

int
nc_sysfs_get_bridge_parent(const char *name, char **parent)
{
	return nc_sysfs_netif_get_linkbase(name, "brport/bridge", parent);
}

int
nc_sysfs_bonding_get_slaves(const char *master, nc_string_array_t *list)
{
	return __nc_sysfs_read_list(NC_SYSFS_CLASS_NET_PATH, master,
					"bonding/slaves", list);
}

int
nc_sysfs_bonding_get_attr(const char *master, const char *attr, char **result)
{
	nc_stringbuf_t tmp = NC_STRINGBUF_INIT;
	int            ret = -1;

	if(nc_stringbuf_printf(&tmp, "bonding/%s", attr) > 0) {
		ret = __nc_sysfs_read_attr(NC_SYSFS_CLASS_NET_PATH, master,
						tmp.string, result);
	}
	nc_stringbuf_destroy(&tmp);
	return ret;
}

int
nc_sysfs_bonding_get_arp_targets(const char *master, nc_string_array_t *list)
{
	return __nc_sysfs_read_list(NC_SYSFS_CLASS_NET_PATH, master,
					"bonding/arp_ip_target", list);
}

int
nc_sysfs_bridge_get_config(const char *ifname, nc_bridge_config_t *config)
{
	if(!ifname || !config)
		return -1;

	if(nc_sysfs_netif_get_int (ifname, SYSFS_BRIDGE_ATTR"/stp_state",      &config->stp_enabled))
		return -1;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_ATTR"/priority",       &config->priority))
		return -1;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/forward_delay", &config->forward_delay))
		return -1;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/ageing_time",   &config->ageing_time))
		return -1;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/hello_time",    &config->hello_time))
		return -1;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/max_age",       &config->max_age))
		return -1;

	return 0;
}

int
nc_sysfs_bridge_get_status(const char *ifname, nc_bridge_status_t **status)
{
	nc_bridge_status_t *bs;

	if(!ifname || !status)
		return -1;

	if(*status)
		nc_bridge_status_free(*status);
	*status = NULL;

	bs = calloc(1, sizeof(nc_bridge_status_t));
	if (!bs)
		return -1;

	if(nc_sysfs_netif_get_int(ifname, SYSFS_BRIDGE_ATTR"/stp_state", &bs->stp_state))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_ATTR"/priority", &bs->priority))
		goto failure;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/forward_delay", &bs->forward_delay))
		goto failure;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/ageing_time", &bs->ageing_time))
		goto failure;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/hello_time", &bs->hello_time))
		goto failure;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/max_age", &bs->max_age))
		goto failure;
	if(nc_sysfs_netif_get_string(ifname, SYSFS_BRIDGE_ATTR"/root_id", &bs->root_id))
		goto failure;
	if(nc_sysfs_netif_get_string(ifname, SYSFS_BRIDGE_ATTR"/bridge_id", &bs->bridge_id))
		goto failure;
	if(nc_sysfs_netif_get_string(ifname, SYSFS_BRIDGE_ATTR"/group_addr", &bs->group_addr))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_ATTR"/root_port", &bs->root_port))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_ATTR"/root_path_cost", &bs->root_path_cost))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_ATTR"/topology_change", &bs->topology_change))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_ATTR"/topology_change_detected",
									&bs->topology_change_detected))
		goto failure;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/gc_timer", &bs->gc_timer))
		goto failure;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/tcn_timer", &bs->tcn_timer))
		goto failure;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/hello_timer", &bs->hello_timer))
		goto failure;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_ATTR"/topology_change_timer",
									&bs->topology_change_timer))
		goto failure;

	*status = bs;
	return 0;

failure:
	nc_bridge_status_free(bs);
	return -1;
}

int
nc_sysfs_bridge_get_port_names(const char *ifname, nc_string_array_t *names)
{
	char dirname[PATH_MAX + 1];

	if(!ifname || !names)
		return -1;

	snprintf(dirname, sizeof(dirname), "%s/%s/%s",
		NC_SYSFS_CLASS_NET_PATH, ifname, SYSFS_BRIDGE_PORT_SUBDIR);

	return nc_scandir(dirname, NULL, names);
}

int
nc_sysfs_bridge_port_get_config(const char *ifname, nc_bridge_port_config_t *config)
{
	if(!ifname || !config)
		return -1;

	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_PORT_ATTR"/priority", &config->priority))
		return -1;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_PORT_ATTR"/path_cost", &config->path_cost))
		return -1;
	return 0;
}

int
nc_sysfs_bridge_port_get_status(const char *ifname, nc_bridge_port_status_t **status)
{
	nc_bridge_port_status_t *ps;

	if(!ifname || !status)
		return -1;

	if(*status)
		nc_bridge_port_status_free(*status);
	*status = NULL;

	ps = calloc(1, sizeof(nc_bridge_port_status_t));
	if (!ps)
		return -1;

	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_PORT_ATTR"/priority", &ps->priority))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_PORT_ATTR"/path_cost", &ps->path_cost))
		goto failure;
	if(nc_sysfs_netif_get_int(ifname, SYSFS_BRIDGE_PORT_ATTR"/state", &ps->state))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_PORT_ATTR"/port_no", &ps->port_no))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_PORT_ATTR"/port_id", &ps->port_no))
		goto failure;
	if(nc_sysfs_netif_get_string(ifname, SYSFS_BRIDGE_PORT_ATTR"/designated_root", &ps->designated_root))
		goto failure;
	if(nc_sysfs_netif_get_string(ifname, SYSFS_BRIDGE_PORT_ATTR"/designated_bridge", &ps->designated_bridge))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_PORT_ATTR"/designated_port", &ps->designated_port))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_PORT_ATTR"/designated_cost", &ps->designated_cost))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_PORT_ATTR"/change_ack", &ps->change_ack))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_PORT_ATTR"/hairpin_mode", &ps->hairpin_mode))
		goto failure;
	if(nc_sysfs_netif_get_uint(ifname, SYSFS_BRIDGE_PORT_ATTR"/config_pending", &ps->config_pending))
		goto failure;

	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_PORT_ATTR"/hold_timer", &ps->hold_timer))
		goto failure;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_PORT_ATTR"/message_age_timer", &ps->message_age_timer))
		goto failure;
	if(nc_sysfs_netif_get_ulong(ifname, SYSFS_BRIDGE_PORT_ATTR"/forward_delay_timer", &ps->forward_delay_timer))
		goto failure;

	*status = ps;
	return 0;

failure:
	nc_bridge_port_status_free(ps);
	return -1;
}

