// Copyright(c) 2018, Intel Corporation
//
// Redistribution  and  use  in source  and  binary  forms,  with  or  without
// modification, are permitted provided that the following conditions are met:
//
// * Redistributions of  source code  must retain the  above copyright notice,
//   this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright notice,
//   this list of conditions and the following disclaimer in the documentation
//   and/or other materials provided with the distribution.
// * Neither the name  of Intel Corporation  nor the names of its contributors
//   may be used to  endorse or promote  products derived  from this  software
//   without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,  BUT NOT LIMITED TO,  THE
// IMPLIED WARRANTIES OF  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT  SHALL THE COPYRIGHT OWNER  OR CONTRIBUTORS BE
// LIABLE  FOR  ANY  DIRECT,  INDIRECT,  INCIDENTAL,  SPECIAL,  EXEMPLARY,  OR
// CONSEQUENTIAL  DAMAGES  (INCLUDING,  BUT  NOT LIMITED  TO,  PROCUREMENT  OF
// SUBSTITUTE GOODS OR SERVICES;  LOSS OF USE,  DATA, OR PROFITS;  OR BUSINESS
// INTERRUPTION)  HOWEVER CAUSED  AND ON ANY THEORY  OF LIABILITY,  WHETHER IN
// CONTRACT,  STRICT LIABILITY,  OR TORT  (INCLUDING NEGLIGENCE  OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,  EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//****************************************************************************
// Russell.H.Weight@Intel.com   Created: 09-04-18
// Revision: 09-07-18  10:00

// The pll_freq_config[] array of iopll_config data structures contains
// the information required to program the IOPLL with the desired
// frequencies for the 1x and 2x output clocks. These data structures
// are indexed by the desired 1x frequency in MHz. For example, to request
// 100MHz for the 1x output (and 200MHz for the 2x output), one would use
// an index of 100.
//
// This table assumes a reference frequency of 100MHz and supports
// 1x output frequencies of 10 MHz to 600MHz. For the 2x output, the
// output frequency is clamped at 600MHz (i.e. the 2x output is muted
// when the requested 1x output exceeds 300MHz).
//
// For a requested frequency, The opae library will pass the entire
// data structure as a binary blob to the kernel driver through the
// sysfs interface.

#define  IOPLL_MAX_FREQ         600
#define  IOPLL_MIN_FREQ         10

struct iopll_config {
	unsigned int pll_freq_khz;
	unsigned int pll_m;
	unsigned int pll_n;
	unsigned int pll_c1;
	unsigned int pll_c0;
	unsigned int pll_lf;
	unsigned int pll_cp;
	unsigned int pll_rc;
};

// Reference frequency: 100MHz
const struct iopll_config iopll_freq_config[] = {
	{ 0 }, // Freq 0 not configured
	{ 0 }, // Freq 1 not configured
	{ 0 }, // Freq 2 not configured
	{ 0 }, // Freq 3 not configured
	{ 0 }, // Freq 4 not configured
	{ 0 }, // Freq 5 not configured
	{ 0 }, // Freq 6 not configured
	{ 0 }, // Freq 7 not configured
	{ 0 }, // Freq 8 not configured
	{ 0 }, // Freq 9 not configured
	{ 10000, 0x808, 0x10000, 0x5050, 0x2828, 0xc0, 0x4, 0x2 },
	{ 11000, 0x22726, 0x20302, 0x4646, 0x2323, 0x180, 0x4, 0x2 },
	{ 12000, 0x2727, 0x20302, 0x4141, 0x22120, 0x180, 0x4, 0x2 },
	{ 13000, 0x2727, 0x20302, 0x3c3c, 0x1e1e, 0x180, 0x4, 0x2 },
	{ 14000, 0x22726, 0x20302, 0x3737, 0x21c1b, 0x180, 0x4, 0x2 },
	{ 15000, 0x2504f, 0x505, 0x3535, 0x21b1a, 0x180, 0x6, 0x2 },
	{ 16000, 0x808, 0x10000, 0x3232, 0x1919, 0xc0, 0x4, 0x2 },
	{ 17000, 0x24d4c, 0x505, 0x2d2d, 0x21716, 0x180, 0x6, 0x2 },
	{ 18000, 0x2424, 0x20302, 0x2828, 0x1414, 0x180, 0x4, 0x2 },
	{ 19000, 0x2626, 0x20302, 0x2828, 0x1414, 0x180, 0x4, 0x2 },
	{ 20000, 0x808, 0x10000, 0x2828, 0x1414, 0xc0, 0x4, 0x2 },
	{ 21000, 0x24a49, 0x505, 0x2323, 0x21211, 0x180, 0x6, 0x2 },
	{ 22000, 0x22726, 0x20302, 0x2323, 0x21211, 0x180, 0x4, 0x2 },
	{ 23000, 0x22322, 0x20302, 0x1e1e, 0xf0f, 0x180, 0x4, 0x2 },
	{ 24000, 0x2424, 0x20302, 0x1e1e, 0xf0f, 0x180, 0x4, 0x2 },
	{ 25000, 0x808, 0x10000, 0x2020, 0x1010, 0xc0, 0x4, 0x2 },
	{ 26000, 0x2727, 0x20302, 0x1e1e, 0xf0f, 0x180, 0x4, 0x2 },
	{ 27000, 0x20e0d, 0x101, 0x1919, 0x20d0c, 0x100, 0x4, 0x0 },
	{ 28000, 0x707, 0x10000, 0x1919, 0x20d0c, 0xc0, 0x4, 0x0 },
	{ 29000, 0x20f0e, 0x101, 0x1919, 0x20d0c, 0x100, 0x4, 0x0 },
	{ 30000, 0x2727, 0x20302, 0x1a1a, 0xd0d, 0x180, 0x4, 0x2 },
	{ 31000, 0x2100f, 0x101, 0x1919, 0x20d0c, 0x100, 0x4, 0x0 },
	{ 32000, 0x808, 0x10000, 0x1919, 0x20d0c, 0xc0, 0x4, 0x2 },
	{ 33000, 0x2121, 0x20302, 0x1414, 0xa0a, 0x180, 0x4, 0x2 },
	{ 34000, 0x2222, 0x20302, 0x1414, 0xa0a, 0x180, 0x4, 0x2 },
	{ 35000, 0x22726, 0x20302, 0x1616, 0xb0b, 0x180, 0x4, 0x2 },
	{ 36000, 0x2424, 0x20302, 0x1414, 0xa0a, 0x180, 0x4, 0x2 },
	{ 37000, 0x2525, 0x20302, 0x1414, 0xa0a, 0x180, 0x4, 0x2 },
	{ 38000, 0x2626, 0x20302, 0x1414, 0xa0a, 0x180, 0x4, 0x2 },
	{ 39000, 0x2727, 0x20302, 0x1414, 0xa0a, 0x180, 0x4, 0x2 },
	{ 40000, 0x808, 0x10000, 0x1414, 0xa0a, 0xc0, 0x4, 0x2 },
	{ 41000, 0x23e3d, 0x505, 0xf0f, 0x20807, 0x180, 0x4, 0x2 },
	{ 42000, 0x2201f, 0x20302, 0xf0f, 0x20807, 0x140, 0x4, 0x2 },
	{ 43000, 0x24140, 0x505, 0xf0f, 0x20807, 0x180, 0x6, 0x2 },
	{ 44000, 0x2121, 0x20302, 0xf0f, 0x20807, 0x180, 0x4, 0x2 },
	{ 45000, 0x24d4c, 0x505, 0x1111, 0x20908, 0x180, 0x6, 0x2 },
	{ 46000, 0x22322, 0x20302, 0xf0f, 0x20807, 0x180, 0x4, 0x2 },
	{ 47000, 0x24746, 0x505, 0xf0f, 0x20807, 0x180, 0x6, 0x2 },
	{ 48000, 0x2424, 0x20302, 0xf0f, 0x20807, 0x180, 0x4, 0x2 },
	{ 49000, 0x24a49, 0x505, 0xf0f, 0x20807, 0x180, 0x6, 0x2 },
	{ 50000, 0x808, 0x10000, 0x1010, 0x808, 0xc0, 0x4, 0x2 },
	{ 51000, 0x24d4c, 0x505, 0xf0f, 0x20807, 0x180, 0x6, 0x2 },
	{ 52000, 0x2727, 0x20302, 0xf0f, 0x20807, 0x180, 0x4, 0x2 },
	{ 53000, 0x2504f, 0x505, 0xf0f, 0x20807, 0x180, 0x6, 0x2 },
	{ 54000, 0x1b1b, 0x20302, 0xa0a, 0x505, 0x140, 0x4, 0x2 },
	{ 55000, 0x22726, 0x20302, 0xe0e, 0x707, 0x180, 0x4, 0x2 },
	{ 56000, 0x1c1c, 0x20302, 0xa0a, 0x505, 0x140, 0x4, 0x2 },
	{ 57000, 0x21d1c, 0x20302, 0xa0a, 0x505, 0x140, 0x4, 0x2 },
	{ 58000, 0x1d1d, 0x20302, 0xa0a, 0x505, 0x140, 0x4, 0x2 },
	{ 59000, 0x21e1d, 0x20302, 0xa0a, 0x505, 0x140, 0x4, 0x2 },
	{ 60000, 0x2727, 0x20302, 0xd0d, 0x20706, 0x180, 0x4, 0x2 },
	{ 61000, 0x21f1e, 0x20302, 0xa0a, 0x505, 0x140, 0x4, 0x2 },
	{ 62000, 0x1f1f, 0x20302, 0xa0a, 0x505, 0x140, 0x4, 0x2 },
	{ 63000, 0x2201f, 0x20302, 0xa0a, 0x505, 0x140, 0x4, 0x2 },
	{ 64000, 0x2020, 0x20302, 0xa0a, 0x505, 0x140, 0x4, 0x2 },
	{ 65000, 0x2727, 0x20302, 0xc0c, 0x606, 0x180, 0x4, 0x2 },
	{ 66000, 0x2121, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 67000, 0x22221, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 68000, 0x2222, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 69000, 0x22322, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 70000, 0x22726, 0x20302, 0xb0b, 0x20605, 0x180, 0x4, 0x2 },
	{ 71000, 0x22423, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 72000, 0x2424, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 73000, 0x22524, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 74000, 0x2525, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 75000, 0x20807, 0x10000, 0xa0a, 0x505, 0xc0, 0x4, 0x0 },
	{ 76000, 0x2626, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 77000, 0x22726, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 78000, 0x2727, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 79000, 0x22827, 0x20302, 0xa0a, 0x505, 0x180, 0x4, 0x2 },
	{ 80000, 0x808, 0x10000, 0xa0a, 0x505, 0xc0, 0x4, 0x2 },
	{ 81000, 0x22928, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 82000, 0x21514, 0x20302, 0x505, 0x20302, 0x100, 0x4, 0x0 },
	{ 83000, 0x22a29, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 84000, 0x1515, 0x20302, 0x505, 0x20302, 0x100, 0x4, 0x0 },
	{ 85000, 0x24d4c, 0x505, 0x909, 0x20504, 0x180, 0x6, 0x2 },
	{ 86000, 0x21615, 0x20302, 0x505, 0x20302, 0x100, 0x4, 0x0 },
	{ 87000, 0x22c2b, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 88000, 0x1616, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 89000, 0x22d2c, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 90000, 0x2424, 0x20302, 0x808, 0x404, 0x180, 0x4, 0x2 },
	{ 91000, 0x22e2d, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 92000, 0x1717, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 93000, 0x22f2e, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 94000, 0x21817, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 95000, 0x2626, 0x20302, 0x808, 0x404, 0x180, 0x4, 0x2 },
	{ 96000, 0x1818, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 97000, 0x23130, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 98000, 0x21918, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 99000, 0x23231, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 100000, 0x808, 0x10000, 0x808, 0x404, 0xc0, 0x4, 0x2 },
	{ 101000, 0x23332, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 102000, 0x21a19, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 103000, 0x23433, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 104000, 0x1a1a, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 105000, 0x24a49, 0x505, 0x707, 0x20403, 0x180, 0x6, 0x2 },
	{ 106000, 0x21b1a, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 107000, 0x23635, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 108000, 0x1b1b, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 109000, 0x23736, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 110000, 0x22726, 0x20302, 0x707, 0x20403, 0x180, 0x4, 0x2 },
	{ 111000, 0x23837, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 112000, 0x1c1c, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 113000, 0x23938, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 114000, 0x21d1c, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 115000, 0x22322, 0x20302, 0x606, 0x303, 0x180, 0x4, 0x2 },
	{ 116000, 0x1d1d, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 117000, 0x23b3a, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 118000, 0x21e1d, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 119000, 0x23c3b, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 120000, 0x2424, 0x20302, 0x606, 0x303, 0x180, 0x4, 0x2 },
	{ 121000, 0x23d3c, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 122000, 0x21f1e, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 123000, 0x23e3d, 0x505, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 124000, 0x1f1f, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 125000, 0x20807, 0x10000, 0x606, 0x303, 0xc0, 0x4, 0x0 },
	{ 126000, 0x2201f, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 127000, 0x2403f, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 128000, 0x2020, 0x20302, 0x505, 0x20302, 0x140, 0x4, 0x2 },
	{ 129000, 0x24140, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 130000, 0x2727, 0x20302, 0x606, 0x303, 0x180, 0x4, 0x2 },
	{ 131000, 0x24241, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 132000, 0x2121, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 133000, 0x24342, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 134000, 0x22221, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 135000, 0x20e0d, 0x101, 0x505, 0x20302, 0x100, 0x4, 0x0 },
	{ 136000, 0x2222, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 137000, 0x24544, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 138000, 0x22322, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 139000, 0x24645, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 140000, 0x707, 0x10000, 0x505, 0x20302, 0xc0, 0x4, 0x0 },
	{ 141000, 0x24746, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 142000, 0x22423, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 143000, 0x24847, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 144000, 0x2424, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 145000, 0x20f0e, 0x101, 0x505, 0x20302, 0x100, 0x4, 0x0 },
	{ 146000, 0x22524, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 147000, 0x24a49, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 148000, 0x2525, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 149000, 0x24b4a, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 150000, 0x20807, 0x10000, 0x505, 0x20302, 0xc0, 0x4, 0x0 },
	{ 151000, 0x24c4b, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 152000, 0x2626, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 153000, 0x24d4c, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 154000, 0x22726, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 155000, 0x2100f, 0x101, 0x505, 0x20302, 0x100, 0x4, 0x0 },
	{ 156000, 0x2727, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 157000, 0x24f4e, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 158000, 0x22827, 0x20302, 0x505, 0x20302, 0x180, 0x4, 0x2 },
	{ 159000, 0x2504f, 0x505, 0x505, 0x20302, 0x180, 0x6, 0x2 },
	{ 160000, 0x808, 0x10000, 0x505, 0x20302, 0xc0, 0x4, 0x2 },
	{ 160937, 0x23433, 0x404, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 161904, 0x2222, 0x20403, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 162962, 0x2c2c, 0x20504, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 163888, 0x3b3b, 0x20504, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 165000, 0x2121, 0x20302, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 165625, 0x21b1a, 0x202, 0x404, 0x202, 0x140, 0x4, 0x2 },
	{ 166666, 0x1414, 0x20201, 0x404, 0x202, 0x100, 0x4, 0x0 },
	{ 167857, 0x2f2f, 0x20403, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 168750, 0x20e0d, 0x101, 0x404, 0x202, 0x100, 0x4, 0x0 },
	{ 170000, 0x2222, 0x20302, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 170833, 0x21514, 0x20201, 0x404, 0x202, 0x100, 0x4, 0x0 },
	{ 171875, 0x21c1b, 0x202, 0x404, 0x202, 0x140, 0x4, 0x2 },
	{ 172916, 0x22a29, 0x303, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 173809, 0x22524, 0x20403, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 175000, 0x707, 0x10000, 0x404, 0x202, 0xc0, 0x4, 0x0 },
	{ 175925, 0x2302f, 0x20504, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 176785, 0x23231, 0x20403, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 177777, 0x4040, 0x20504, 0x404, 0x202, 0x180, 0x6, 0x2 },
	{ 178750, 0x24847, 0x505, 0x404, 0x202, 0x180, 0x6, 0x2 },
	{ 180000, 0x2424, 0x20302, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 180952, 0x2626, 0x20403, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 181944, 0x24241, 0x20504, 0x404, 0x202, 0x180, 0x6, 0x2 },
	{ 182812, 0x23b3a, 0x404, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 183928, 0x23433, 0x20403, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 185000, 0x2525, 0x20302, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 185937, 0x23c3b, 0x404, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 186666, 0x1c1c, 0x20302, 0x303, 0x20201, 0x140, 0x4, 0x2 },
	{ 187500, 0x20807, 0x10000, 0x404, 0x202, 0xc0, 0x4, 0x0 },
	{ 188888, 0x4444, 0x20504, 0x404, 0x202, 0x180, 0x6, 0x2 },
	{ 190000, 0x2626, 0x20302, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 190740, 0x23433, 0x20504, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 191666, 0x1717, 0x20201, 0x404, 0x202, 0x140, 0x4, 0x2 },
	{ 192857, 0x3636, 0x20403, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 193750, 0x2100f, 0x101, 0x404, 0x202, 0x100, 0x4, 0x0 },
	{ 195000, 0x2727, 0x20302, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 195833, 0x21817, 0x20201, 0x404, 0x202, 0x140, 0x4, 0x2 },
	{ 196875, 0x2201f, 0x202, 0x404, 0x202, 0x140, 0x4, 0x2 },
	{ 197916, 0x2302f, 0x303, 0x404, 0x202, 0x180, 0x4, 0x2 },
	{ 198750, 0x2504f, 0x505, 0x404, 0x202, 0x180, 0x6, 0x2 },
	{ 200000, 0x808, 0x10000, 0x404, 0x202, 0xc0, 0x4, 0x2 },
	{ 200000, 0x808, 0x10000, 0x404, 0x202, 0xc0, 0x4, 0x2 },
	{ 201851, 0x23736, 0x20504, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 202777, 0x22524, 0x303, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 203703, 0x3737, 0x20504, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 205000, 0x23e3d, 0x505, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 205555, 0x21312, 0x20201, 0x303, 0x20201, 0x100, 0x4, 0x0 },
	{ 206666, 0x1f1f, 0x20302, 0x303, 0x20201, 0x140, 0x4, 0x2 },
	{ 207500, 0x22a29, 0x505, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 208333, 0x20d0c, 0x101, 0x303, 0x20201, 0x100, 0x4, 0x0 },
	{ 210000, 0x2201f, 0x20302, 0x303, 0x20201, 0x140, 0x4, 0x2 },
	{ 210714, 0x21e1d, 0x20403, 0x202, 0x101, 0x140, 0x4, 0x2 },
	{ 211904, 0x22d2c, 0x20403, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 212962, 0x23a39, 0x20504, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 213888, 0x22726, 0x303, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 215000, 0x24140, 0x505, 0x303, 0x20201, 0x180, 0x6, 0x2 },
	{ 215625, 0x22322, 0x404, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 216666, 0x20706, 0x10000, 0x303, 0x20201, 0xc0, 0x4, 0x0 },
	{ 217857, 0x21f1e, 0x20403, 0x202, 0x101, 0x140, 0x4, 0x2 },
	{ 218750, 0x23534, 0x404, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 220000, 0x2121, 0x20302, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 220833, 0x21b1a, 0x202, 0x303, 0x20201, 0x140, 0x4, 0x2 },
	{ 221875, 0x22423, 0x404, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 222916, 0x23635, 0x404, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 223809, 0x2f2f, 0x20403, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 225000, 0x20e0d, 0x101, 0x303, 0x20201, 0x100, 0x4, 0x0 },
	{ 225925, 0x3d3d, 0x20504, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 226666, 0x2222, 0x20302, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 227777, 0x21514, 0x20201, 0x303, 0x20201, 0x100, 0x4, 0x0 },
	{ 228571, 0x3030, 0x20403, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 230000, 0x22322, 0x20302, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 230952, 0x23130, 0x20403, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 231666, 0x24645, 0x505, 0x303, 0x20201, 0x180, 0x6, 0x2 },
	{ 232500, 0x22f2e, 0x505, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 233333, 0x707, 0x10000, 0x303, 0x20201, 0xc0, 0x4, 0x0 },
	{ 235000, 0x24746, 0x505, 0x303, 0x20201, 0x180, 0x6, 0x2 },
	{ 235714, 0x23231, 0x20403, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 236666, 0x22423, 0x20302, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 237500, 0x21d1c, 0x202, 0x303, 0x20201, 0x140, 0x4, 0x2 },
	{ 238888, 0x21615, 0x20201, 0x303, 0x20201, 0x100, 0x4, 0x0 },
	{ 240000, 0x2424, 0x20302, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 240740, 0x4141, 0x20504, 0x303, 0x20201, 0x180, 0x6, 0x2 },
	{ 241666, 0x20f0e, 0x101, 0x303, 0x20201, 0x100, 0x4, 0x0 },
	{ 242857, 0x3333, 0x20403, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 243750, 0x23b3a, 0x404, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 245000, 0x24a49, 0x505, 0x303, 0x20201, 0x180, 0x6, 0x2 },
	{ 245833, 0x21e1d, 0x202, 0x303, 0x20201, 0x140, 0x4, 0x2 },
	{ 246875, 0x22827, 0x404, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 247916, 0x23c3b, 0x404, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 248333, 0x24b4a, 0x505, 0x303, 0x20201, 0x180, 0x6, 0x2 },
	{ 250000, 0x20807, 0x10000, 0x303, 0x20201, 0xc0, 0x4, 0x0 },
	{ 250000, 0x20807, 0x10000, 0x303, 0x20201, 0xc0, 0x4, 0x0 },
	{ 251851, 0x4444, 0x20504, 0x303, 0x20201, 0x180, 0x6, 0x2 },
	{ 252777, 0x22e2d, 0x303, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 253703, 0x24544, 0x20504, 0x303, 0x20201, 0x180, 0x6, 0x2 },
	{ 255000, 0x24d4c, 0x505, 0x303, 0x20201, 0x180, 0x6, 0x2 },
	{ 255555, 0x1717, 0x20201, 0x303, 0x20201, 0x140, 0x4, 0x2 },
	{ 256666, 0x22726, 0x20302, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 257500, 0x23433, 0x505, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 258333, 0x2100f, 0x101, 0x303, 0x20201, 0x100, 0x4, 0x0 },
	{ 260000, 0x2727, 0x20302, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 260714, 0x22524, 0x20403, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 261904, 0x3737, 0x20403, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 262962, 0x4747, 0x20504, 0x303, 0x20201, 0x180, 0x6, 0x2 },
	{ 263888, 0x2302f, 0x303, 0x303, 0x20201, 0x180, 0x4, 0x2 },
	{ 265000, 0x2504f, 0x505, 0x303, 0x20201, 0x180, 0x6, 0x2 },
	{ 265625, 0x22b2a, 0x404, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 266666, 0x808, 0x10000, 0x303, 0x20201, 0xc0, 0x4, 0x2 },
	{ 267857, 0x22625, 0x20403, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 268750, 0x21615, 0x202, 0x202, 0x101, 0x100, 0x4, 0x0 },
	{ 270000, 0x1b1b, 0x20302, 0x202, 0x101, 0x140, 0x4, 0x2 },
	{ 270833, 0x22120, 0x303, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 271875, 0x22c2b, 0x404, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 272500, 0x23736, 0x505, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 272500, 0x23736, 0x505, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 275000, 0x20605, 0x10000, 0x202, 0x101, 0xc0, 0x4, 0x0 },
	{ 275000, 0x20605, 0x10000, 0x202, 0x101, 0xc0, 0x4, 0x0 },
	{ 275000, 0x20605, 0x10000, 0x202, 0x101, 0xc0, 0x4, 0x0 },
	{ 277777, 0x3232, 0x20504, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 278571, 0x2727, 0x20403, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 280000, 0x1c1c, 0x20302, 0x202, 0x101, 0x140, 0x4, 0x2 },
	{ 280555, 0x23332, 0x20504, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 281250, 0x21716, 0x202, 0x202, 0x101, 0x140, 0x4, 0x2 },
	{ 282500, 0x23938, 0x505, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 283333, 0x1111, 0x20201, 0x202, 0x101, 0x100, 0x4, 0x0 },
	{ 285000, 0x21d1c, 0x20302, 0x202, 0x101, 0x140, 0x4, 0x2 },
	{ 285714, 0x2828, 0x20403, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 286111, 0x23433, 0x20504, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 287500, 0x20c0b, 0x101, 0x202, 0x101, 0xc0, 0x4, 0x2 },
	{ 288888, 0x3434, 0x20504, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 290000, 0x1d1d, 0x20302, 0x202, 0x101, 0x140, 0x4, 0x2 },
	{ 290625, 0x22f2e, 0x404, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 291666, 0x21211, 0x20201, 0x202, 0x101, 0x100, 0x4, 0x0 },
	{ 292857, 0x2929, 0x20403, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 293750, 0x21817, 0x202, 0x202, 0x101, 0x140, 0x4, 0x2 },
	{ 295000, 0x21e1d, 0x20302, 0x202, 0x101, 0x140, 0x4, 0x2 },
	{ 295833, 0x22423, 0x303, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 296875, 0x2302f, 0x404, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 297500, 0x23c3b, 0x505, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 297500, 0x23c3b, 0x505, 0x202, 0x101, 0x180, 0x4, 0x2 },
	{ 300000, 0x606, 0x10000, 0x202, 0x101, 0xc0, 0x4, 0x0 },
	{ 300000, 0x606, 0x10000, 0x202, 0x101, 0xc0, 0x4, 0x0 },
	{ 302000, 0x24c4b, 0x505, 0x20302, 0x20302, 0x180, 0x6, 0x2 },
	{ 302857, 0x3535, 0x20403, 0x20302, 0x20302, 0x180, 0x4, 0x2 },
	{ 304000, 0x2626, 0x20302, 0x20302, 0x20302, 0x180, 0x4, 0x2 },
	{ 305000, 0x21f1e, 0x202, 0x20302, 0x20302, 0x140, 0x4, 0x2 },
	{ 306000, 0x24d4c, 0x505, 0x20302, 0x20302, 0x180, 0x6, 0x2 },
	{ 306666, 0x1717, 0x20201, 0x20302, 0x20302, 0x140, 0x4, 0x2 },
	{ 308000, 0x22726, 0x20302, 0x20302, 0x20302, 0x180, 0x4, 0x2 },
	{ 308888, 0x24645, 0x20504, 0x20302, 0x20302, 0x180, 0x6, 0x2 },
	{ 310000, 0x2100f, 0x101, 0x20302, 0x20302, 0x100, 0x4, 0x0 },
	{ 310714, 0x22c2b, 0x20403, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 312000, 0x2727, 0x20302, 0x20302, 0x20302, 0x180, 0x4, 0x2 },
	{ 312500, 0x23f3e, 0x404, 0x20302, 0x20302, 0x180, 0x6, 0x2 },
	{ 314000, 0x24f4e, 0x505, 0x20302, 0x20302, 0x180, 0x6, 0x2 },
	{ 315000, 0x2201f, 0x202, 0x20302, 0x20302, 0x140, 0x4, 0x2 },
	{ 316000, 0x22827, 0x20302, 0x20302, 0x20302, 0x180, 0x4, 0x2 },
	{ 316666, 0x2302f, 0x303, 0x20302, 0x20302, 0x180, 0x4, 0x2 },
	{ 318000, 0x2504f, 0x505, 0x20302, 0x20302, 0x180, 0x6, 0x2 },
	{ 318750, 0x21a19, 0x202, 0x202, 0x202, 0x140, 0x4, 0x2 },
	{ 320000, 0x808, 0x10000, 0x20302, 0x20302, 0xc0, 0x4, 0x2 },
	{ 320833, 0x22726, 0x303, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 321875, 0x23433, 0x404, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 322500, 0x24140, 0x505, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 323809, 0x2222, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 325000, 0x20706, 0x10000, 0x202, 0x202, 0xc0, 0x4, 0x0 },
	{ 325925, 0x2c2c, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 326666, 0x21918, 0x20302, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 327777, 0x3b3b, 0x20504, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 328571, 0x2e2e, 0x20403, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 330000, 0x2121, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 330555, 0x23c3b, 0x20504, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 331250, 0x21b1a, 0x202, 0x202, 0x202, 0x140, 0x4, 0x2 },
	{ 332500, 0x24342, 0x505, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 333333, 0x1414, 0x20201, 0x202, 0x202, 0x100, 0x4, 0x0 },
	{ 335000, 0x22221, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 335714, 0x2f2f, 0x20403, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 336666, 0x23332, 0x505, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 337500, 0x20e0d, 0x101, 0x202, 0x202, 0x100, 0x4, 0x0 },
	{ 338888, 0x3d3d, 0x20504, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 340000, 0x2222, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 340740, 0x2e2e, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 341666, 0x21514, 0x20201, 0x202, 0x202, 0x100, 0x4, 0x0 },
	{ 342857, 0x3030, 0x20403, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 343750, 0x21c1b, 0x202, 0x202, 0x202, 0x140, 0x4, 0x2 },
	{ 345000, 0x22322, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 345833, 0x22a29, 0x303, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 346875, 0x23837, 0x404, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 347619, 0x22524, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 348148, 0x2f2f, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 350000, 0x707, 0x10000, 0x202, 0x202, 0xc0, 0x4, 0x0 },
	{ 350000, 0x707, 0x10000, 0x202, 0x202, 0xc0, 0x4, 0x0 },
	{ 351851, 0x2302f, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 352777, 0x2403f, 0x20504, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 353571, 0x23231, 0x20403, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 355000, 0x22423, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 355555, 0x4040, 0x20504, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 356666, 0x23635, 0x505, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 357500, 0x24847, 0x505, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 358333, 0x21615, 0x20201, 0x202, 0x202, 0x100, 0x4, 0x0 },
	{ 360000, 0x2424, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 360714, 0x23332, 0x20403, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 361904, 0x2626, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 362962, 0x3131, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 363888, 0x24241, 0x20504, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 365000, 0x22524, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 365625, 0x23b3a, 0x404, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 366666, 0x1616, 0x20201, 0x202, 0x202, 0x140, 0x4, 0x2 },
	{ 367857, 0x23433, 0x20403, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 368750, 0x21e1d, 0x202, 0x202, 0x202, 0x140, 0x4, 0x2 },
	{ 370000, 0x2525, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 370833, 0x22d2c, 0x303, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 371875, 0x23c3b, 0x404, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 372500, 0x24b4a, 0x505, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 373333, 0x1c1c, 0x20302, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 375000, 0x20807, 0x10000, 0x202, 0x202, 0xc0, 0x4, 0x0 },
	{ 375000, 0x20807, 0x10000, 0x202, 0x202, 0xc0, 0x4, 0x0 },
	{ 376666, 0x23938, 0x505, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 377777, 0x4444, 0x20504, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 378571, 0x3535, 0x20403, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 380000, 0x2626, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 380952, 0x2828, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 381481, 0x23433, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 382500, 0x24d4c, 0x505, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 383333, 0x1717, 0x20201, 0x202, 0x202, 0x140, 0x4, 0x2 },
	{ 385000, 0x22726, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 385714, 0x3636, 0x20403, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 386666, 0x1d1d, 0x20302, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 387500, 0x2100f, 0x101, 0x202, 0x202, 0x100, 0x4, 0x0 },
	{ 388888, 0x4646, 0x20504, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 390000, 0x2727, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 390625, 0x23f3e, 0x404, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 391666, 0x21817, 0x20201, 0x202, 0x202, 0x140, 0x4, 0x2 },
	{ 392857, 0x3737, 0x20403, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 393750, 0x2201f, 0x202, 0x202, 0x202, 0x140, 0x4, 0x2 },
	{ 395000, 0x22827, 0x20302, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 395833, 0x2302f, 0x303, 0x202, 0x202, 0x180, 0x4, 0x2 },
	{ 396875, 0x2403f, 0x404, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 397500, 0x2504f, 0x505, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 397500, 0x2504f, 0x505, 0x202, 0x202, 0x180, 0x6, 0x2 },
	{ 400000, 0x808, 0x10000, 0x202, 0x202, 0xc0, 0x4, 0x2 },
	{ 400000, 0x808, 0x10000, 0x202, 0x202, 0xc0, 0x4, 0x2 },
	{ 400000, 0x808, 0x10000, 0x202, 0x202, 0xc0, 0x4, 0x2 },
	{ 400000, 0x808, 0x10000, 0x202, 0x202, 0xc0, 0x4, 0x2 },
	{ 403703, 0x23736, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 405000, 0x22928, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 405555, 0x22524, 0x303, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 406666, 0x21f1e, 0x20302, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 407407, 0x3737, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 408333, 0x21918, 0x202, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 410000, 0x23e3d, 0x505, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 410000, 0x23e3d, 0x505, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 411111, 0x21312, 0x20201, 0x20201, 0x20201, 0x100, 0x4, 0x0 },
	{ 412500, 0x23231, 0x404, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 413333, 0x1f1f, 0x20302, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 415000, 0x22a29, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 415000, 0x22a29, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 416666, 0x20d0c, 0x101, 0x20201, 0x20201, 0x100, 0x4, 0x0 },
	{ 416666, 0x20d0c, 0x101, 0x20201, 0x20201, 0x100, 0x4, 0x0 },
	{ 418750, 0x22221, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 420000, 0x2201f, 0x20302, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 420833, 0x23332, 0x404, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 421428, 0x21e1d, 0x20403, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 422222, 0x1313, 0x20201, 0x20201, 0x20201, 0x100, 0x4, 0x0 },
	{ 423809, 0x22d2c, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 425000, 0x21a19, 0x202, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 425925, 0x23a39, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 426666, 0x2020, 0x20302, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 427777, 0x22726, 0x303, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 428571, 0x2d2d, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 430000, 0x24140, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 430000, 0x24140, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 431250, 0x22322, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 431250, 0x22322, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 433333, 0x20706, 0x10000, 0x20201, 0x20201, 0xc0, 0x4, 0x0 },
	{ 435000, 0x22c2b, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 435714, 0x21f1e, 0x20403, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 436666, 0x24241, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 437500, 0x23534, 0x404, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 438888, 0x22827, 0x303, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 440000, 0x2121, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 440740, 0x23c3b, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 441666, 0x21b1a, 0x202, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 442857, 0x22f2e, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 443750, 0x22423, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 445000, 0x22d2c, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 445833, 0x23635, 0x404, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 446666, 0x22221, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 447619, 0x2f2f, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 448148, 0x23d3c, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 450000, 0x20e0d, 0x101, 0x20201, 0x20201, 0x100, 0x4, 0x0 },
	{ 450000, 0x20e0d, 0x101, 0x20201, 0x20201, 0x100, 0x4, 0x0 },
	{ 451851, 0x3d3d, 0x20504, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 452380, 0x2302f, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 453333, 0x2222, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 455000, 0x22e2d, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 455555, 0x21514, 0x20201, 0x20201, 0x20201, 0x100, 0x4, 0x0 },
	{ 456666, 0x24544, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 457142, 0x3030, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 458333, 0x21c1b, 0x202, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 460000, 0x22322, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 460000, 0x22322, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 461904, 0x23130, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 462962, 0x23f3e, 0x20504, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 463333, 0x24645, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 465000, 0x22f2e, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 465000, 0x22f2e, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 466666, 0x707, 0x10000, 0x20201, 0x20201, 0xc0, 0x4, 0x0 },
	{ 466666, 0x707, 0x10000, 0x20201, 0x20201, 0xc0, 0x4, 0x0 },
	{ 468750, 0x22625, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 470000, 0x24746, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 470833, 0x23938, 0x404, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 471428, 0x23231, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 472222, 0x22b2a, 0x303, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 473333, 0x22423, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 475000, 0x21d1c, 0x202, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 475000, 0x21d1c, 0x202, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 476666, 0x24847, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 477777, 0x21615, 0x20201, 0x20201, 0x20201, 0x100, 0x4, 0x0 },
	{ 478571, 0x22221, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 480000, 0x2424, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 480952, 0x23332, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 481481, 0x4141, 0x20504, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 481481, 0x4141, 0x20504, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 483333, 0x20f0e, 0x101, 0x20201, 0x20201, 0x100, 0x4, 0x0 },
	{ 485000, 0x23130, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 485714, 0x3333, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 486666, 0x22524, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 487500, 0x23b3a, 0x404, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 488888, 0x1616, 0x20201, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 490000, 0x24a49, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 490476, 0x23433, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 491666, 0x21e1d, 0x202, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 492857, 0x22322, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 493750, 0x22827, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 495000, 0x23231, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 495833, 0x23c3b, 0x404, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 496666, 0x24b4a, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 496666, 0x24b4a, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 496666, 0x24b4a, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 500000, 0x20807, 0x10000, 0x20201, 0x20201, 0xc0, 0x4, 0x0 },
	{ 500000, 0x20807, 0x10000, 0x20201, 0x20201, 0xc0, 0x4, 0x0 },
	{ 500000, 0x20807, 0x10000, 0x20201, 0x20201, 0xc0, 0x4, 0x0 },
	{ 500000, 0x20807, 0x10000, 0x20201, 0x20201, 0xc0, 0x4, 0x0 },
	{ 503703, 0x4444, 0x20504, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 505000, 0x23332, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 505555, 0x22e2d, 0x303, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 506666, 0x2626, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 507407, 0x24544, 0x20504, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 508333, 0x21f1e, 0x202, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 510000, 0x24d4c, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 510000, 0x24d4c, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 511111, 0x1717, 0x20201, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 512500, 0x23e3d, 0x404, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 513333, 0x22726, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 515000, 0x23433, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 515000, 0x23433, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 516666, 0x2100f, 0x101, 0x20201, 0x20201, 0x100, 0x4, 0x0 },
	{ 516666, 0x2100f, 0x101, 0x20201, 0x20201, 0x100, 0x4, 0x0 },
	{ 518750, 0x22a29, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 520000, 0x2727, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 520833, 0x23f3e, 0x404, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 521428, 0x22524, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 522222, 0x21817, 0x20201, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 523809, 0x3737, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 525000, 0x2201f, 0x202, 0x20201, 0x20201, 0x140, 0x4, 0x2 },
	{ 525925, 0x4747, 0x20504, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 526666, 0x22827, 0x20302, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 527777, 0x2302f, 0x303, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 528571, 0x23837, 0x20403, 0x20201, 0x20201, 0x180, 0x4, 0x2 },
	{ 530000, 0x2504f, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 530000, 0x2504f, 0x505, 0x20201, 0x20201, 0x180, 0x6, 0x2 },
	{ 531250, 0x22b2a, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 531250, 0x22b2a, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 533333, 0x808, 0x10000, 0x20201, 0x20201, 0xc0, 0x4, 0x2 },
	{ 535000, 0x23635, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 535714, 0x22625, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 535714, 0x22625, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 537500, 0x21615, 0x202, 0x101, 0x101, 0x100, 0x4, 0x0 },
	{ 538888, 0x23130, 0x20504, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 540000, 0x1b1b, 0x20302, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 540000, 0x1b1b, 0x20302, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 541666, 0x22120, 0x303, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 542857, 0x2626, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 543750, 0x22c2b, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 545000, 0x23736, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 545000, 0x23736, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 545000, 0x23736, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 545000, 0x23736, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 545000, 0x23736, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 550000, 0x20605, 0x10000, 0x101, 0x101, 0xc0, 0x4, 0x0 },
	{ 550000, 0x20605, 0x10000, 0x101, 0x101, 0xc0, 0x4, 0x0 },
	{ 550000, 0x20605, 0x10000, 0x101, 0x101, 0xc0, 0x4, 0x0 },
	{ 550000, 0x20605, 0x10000, 0x101, 0x101, 0xc0, 0x4, 0x0 },
	{ 550000, 0x20605, 0x10000, 0x101, 0x101, 0xc0, 0x4, 0x0 },
	{ 555000, 0x23837, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 555555, 0x3232, 0x20504, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 556250, 0x22d2c, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 557142, 0x2727, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 558333, 0x22221, 0x303, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 560000, 0x1c1c, 0x20302, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 560000, 0x1c1c, 0x20302, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 561111, 0x23332, 0x20504, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 562500, 0x21716, 0x202, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 562500, 0x21716, 0x202, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 565000, 0x23938, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 565000, 0x23938, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 566666, 0x1111, 0x20201, 0x101, 0x101, 0x100, 0x4, 0x0 },
	{ 566666, 0x1111, 0x20201, 0x101, 0x101, 0x100, 0x4, 0x0 },
	{ 568750, 0x22e2d, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 570000, 0x21d1c, 0x20302, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 570000, 0x21d1c, 0x20302, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 571428, 0x2828, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 572222, 0x23433, 0x20504, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 572222, 0x23433, 0x20504, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 575000, 0x20c0b, 0x101, 0x101, 0x101, 0xc0, 0x4, 0x2 },
	{ 575000, 0x20c0b, 0x101, 0x101, 0x101, 0xc0, 0x4, 0x2 },
	{ 575000, 0x20c0b, 0x101, 0x101, 0x101, 0xc0, 0x4, 0x2 },
	{ 577777, 0x3434, 0x20504, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 578571, 0x22928, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 580000, 0x1d1d, 0x20302, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 580000, 0x1d1d, 0x20302, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 581250, 0x22f2e, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 581250, 0x22f2e, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 583333, 0x21211, 0x20201, 0x101, 0x101, 0x100, 0x4, 0x0 },
	{ 585000, 0x23b3a, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 585714, 0x2929, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 585714, 0x2929, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 587500, 0x21817, 0x202, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 588888, 0x3535, 0x20504, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 590000, 0x21e1d, 0x20302, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 590000, 0x21e1d, 0x20302, 0x101, 0x101, 0x140, 0x4, 0x2 },
	{ 591666, 0x22423, 0x303, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 592857, 0x22a29, 0x20403, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 593750, 0x2302f, 0x404, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 595000, 0x23c3b, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 595000, 0x23c3b, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 595000, 0x23c3b, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 595000, 0x23c3b, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 595000, 0x23c3b, 0x505, 0x101, 0x101, 0x180, 0x4, 0x2 },
	{ 600000, 0x606, 0x10000, 0x101, 0x101, 0xc0, 0x4, 0x0 }
};
