import unittest
import shutil
import os
from test_common import javautils_script, get_config_file_list, \
        get_actual_config, get_expected_config, DIRPATH

from xml_compare import compare_xml_files

class TestMvnPackage(unittest.TestCase):

    maxDiff = 2048

    def setUp(self):
        self.olddir = os.getcwd()
        self.workdir = os.path.join(DIRPATH, 'workdir')
        shutil.rmtree(self.workdir, ignore_errors=True)
        os.mkdir(self.workdir)
        os.chdir(self.workdir)

    def tearDown(self):
        try:
            shutil.rmtree(self.workdir)
        except OSError:
            pass
        os.chdir(self.olddir)

    @javautils_script('mvn_package', [])
    def test_run_no_args(self, stdout, stderr, return_value):
        self.assertNotEqual(return_value, 0)
        self.assertEqual("Usage:", stderr[:6])

    @javautils_script('mvn_package', ['-h'])
    def test_help(self, stdout, stderr, return_value):
        self.assertTrue(stdout)

    @javautils_script('mvn_package',['aaa:bbb', 'pack', ])
    def test_simple(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'simple'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',['aaa:bbb:1.2', 'pack', ])
    def test_version(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'version'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',['aaa:bbb:ccc:', 'pack', ])
    def test_extension(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'extension'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',['aaa:bbb:ccc:ddd:', 'pack', ])
    def test_classifier(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'classifier'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',['aaa:bbb:ccc:ddd:21', 'pack', ])
    def test_all(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'all'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',[':bbb', 'pack', ])
    def test_wildcard1(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'wildcard1'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',[':', 'pack', ])
    def test_wildcard2(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'wildcard2'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',['*:*', 'pack', ])
    def test_wildcard3(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'wildcard3'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',[':bbb-{a,b,c}', 'pack', ])
    def test_braces1(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'braces1'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',['a-{b,c}:{x,y}-z', 'pack', ])
    def test_braces2(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'braces2'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',['aaa:bbb', ])
    def test_single(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'single'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',['aaa:bbb', 'pack', 'evil', ])
    def test_more(self, stdout, stderr, return_value):
        self.assertNotEqual(return_value, 0)
        self.assertTrue(stderr)

    @javautils_script('mvn_package',[':bbb-{a,b', 'pack', ])
    def test_odd_braces(self, stdout, stderr, return_value):
        self.assertNotEqual(return_value, 0)
        self.assertTrue(stderr)

    @javautils_script('mvn_package',[':bb-a,b}', 'pack', ])
    def test_odd_braces1(self, stdout, stderr, return_value):
        self.assertNotEqual(return_value, 0)
        self.assertTrue(stderr)

    @javautils_script('mvn_package',['a-{b,c}:{x,y}-z', '@2', ])
    def test_backref1(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'backref1'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',['a-{b,c}:{x,y}-z', '@1', ])
    def test_backref2(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'backref2'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

    @javautils_script('mvn_package',['a:b', '@1', ])
    def test_backref3(self, stdout, stderr, return_value):
        self.assertNotEqual(return_value, 0)
        self.assertTrue(stderr)

    @javautils_script('mvn_package',['a-{b,c}:x', '@3', ])
    def test_backref4(self, stdout, stderr, return_value):
        self.assertNotEqual(return_value, 0)
        self.assertTrue(stderr)

    @javautils_script('mvn_package',['aaa:bbb', '__noinstall', ])
    def test_noinstall(self, stdout, stderr, return_value):
        self.assertEqual(return_value, 0, stderr)
        filelist = get_config_file_list()
        self.assertEqual(len(filelist), 1)
        for filename in filelist:
            report = compare_xml_files(get_actual_config(filename),
                 get_expected_config(filename, 'mvn_package', 'noinstall'),
                 ['artifactGlob'])
            self.assertFalse(report, '\n' + report)

if __name__ == '__main__':
    unittest.main()
