/**********************************************************************
 * $read_stimulus_short example -- C source code using TF PLI routines
 *
 * For the book, "The Verilog PLI Handbook", Chapter 11
 * Copyright 1998, Sutherland HDL Inc, Portland, Oregon.
 *
 * C source to read test vectors from a file and apply them to
 * simulation.  Once called, this application will call itself back
 * until it reaches End-of-File.  NOTE: A more versatile version of
 * this application is presented in the $read_stimulus_ba example.
 *
 * Each line in the test vector file contains a delay time and a vector
 * value, separated by white space.  Delay values define when the
 * vector is to be applied to simulation.  Delays must be relative to
 * the previous time.  The test vector value must be in binary.  
 * An example file is:
 *
 *     10  00000000
 *     20  10111001
 *     ...
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *
 * Usage:
 *
 *  initial
 *    $read_stimulus_short("file_name", verilog_reg);
 *
 *  where:
 *    "file_name"  is the name of the file to be read, in quotes.
 *    verilog_reg  is a verilog register data type of the same bit
 *                 width as the patterns to be read.
 *
 * NOTES:
 *   1. There is no error checking on the data read from the file.
 *   2. Delay times are limited to 32-bit integers.
 *
 * Routine definitions for a veriusertfs array:
 *  /* routine prototypes -/
 *   extern int PLIbook_ReadStimulusShort_calltf(),
 *              PLIbook_ReadStimulusShort_misctf();
 *  /* table entries -/
 *   {usertask,                          /* type of PLI routine -/
 *     0,                                /* user_data value -/
 *     0,                                /* checktf routine -/
 *     0,                                /* sizetf routine -/
 *     PLIbook_ReadStimulusShort_calltf, /* calltf routine -/
 *     PLIbook_ReadStimulusShort_misctf, /* misctf routine -/
 *     "$read_stimulus_short",           /* system task/function name -/
 *     1                                 /* forward reference = true -/
 *   },
 *********************************************************************/

#include <stdio.h>            /* ANSI C standard I/O library */
#include "veriuser.h"         /* IEEE 1364 PLI TF  routine library */

/*********************************************************************/
/* calltf routine                                                    */
/*********************************************************************/
int PLIbook_ReadStimulusShort_calltf()
{                 /* call the misctf routine at the end of current   */
  tf_setdelay(0); /* time step, with REASON_REACTIVATE; the misctf */
                  /* routine reads the stimulus file.                */
  return(0);
}

/*********************************************************************/
/* misctf routine                                                    */
/*********************************************************************/
int PLIbook_ReadStimulusShort_misctf(int user_data, int reason)
{
  int   delay, foo;
  FILE *file_p;        /* pointer to the test vector file        */
  char  vector[1024];  /* stimulus vector -- hard coded limit    */

  switch(reason) {
    case REASON_ENDOFCOMPILE: /* misctf called at start of simulation */
      file_p = fopen(tf_getcstringp(1), "r");
      /* store a pointer to the file in the work area */
      tf_setworkarea((char *)file_p);
      break;

    case REASON_REACTIVATE:  /* misctf called by tf_setdelay */
      /* get the file pointer from the work area */
      file_p = (FILE *)tf_getworkarea();
      /* read next line from the file */
      if ( (fscanf(file_p,"%d %s\n", &delay, vector)) == EOF) {
        io_printf("$read_stimuulus reached end-of-file\n");
        return(0);  /* exit without scheduling another callback */
        break;
      }
      /* schedule the vector to be applied after the delay period */
      tf_strdelputp(2, tf_sizep(2), 'b', vector, delay, 2);
      /* call this routine back after delay time */
      tf_setdelay(delay);  
      break;
  }
  return(0);
}
/*********************************************************************/
