# -*- coding: utf-8 -*-
# Copyright 2023 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import os

# try/except added for compatibility with python < 3.8
try:
    from unittest import mock
    from unittest.mock import AsyncMock  # pragma: NO COVER
except ImportError:  # pragma: NO COVER
    import mock

from collections.abc import Iterable
import json
import math

from google.api_core import gapic_v1, grpc_helpers, grpc_helpers_async, path_template
from google.api_core import api_core_version, client_options
from google.api_core import exceptions as core_exceptions
import google.auth
from google.auth import credentials as ga_credentials
from google.auth.exceptions import MutualTLSChannelError
from google.cloud.location import locations_pb2
from google.iam.v1 import iam_policy_pb2  # type: ignore
from google.iam.v1 import options_pb2  # type: ignore
from google.iam.v1 import policy_pb2  # type: ignore
from google.longrunning import operations_pb2  # type: ignore
from google.oauth2 import service_account
from google.protobuf import field_mask_pb2  # type: ignore
from google.protobuf import json_format
from google.protobuf import timestamp_pb2  # type: ignore
import grpc
from grpc.experimental import aio
from proto.marshal.rules import wrappers
from proto.marshal.rules.dates import DurationRule, TimestampRule
import pytest
from requests import PreparedRequest, Request, Response
from requests.sessions import Session

from google.cloud.kms_v1.services.ekm_service import (
    EkmServiceAsyncClient,
    EkmServiceClient,
    pagers,
    transports,
)
from google.cloud.kms_v1.types import ekm_service


def client_cert_source_callback():
    return b"cert bytes", b"key bytes"


# If default endpoint is localhost, then default mtls endpoint will be the same.
# This method modifies the default endpoint so the client can produce a different
# mtls endpoint for endpoint testing purposes.
def modify_default_endpoint(client):
    return (
        "foo.googleapis.com"
        if ("localhost" in client.DEFAULT_ENDPOINT)
        else client.DEFAULT_ENDPOINT
    )


# If default endpoint template is localhost, then default mtls endpoint will be the same.
# This method modifies the default endpoint template so the client can produce a different
# mtls endpoint for endpoint testing purposes.
def modify_default_endpoint_template(client):
    return (
        "test.{UNIVERSE_DOMAIN}"
        if ("localhost" in client._DEFAULT_ENDPOINT_TEMPLATE)
        else client._DEFAULT_ENDPOINT_TEMPLATE
    )


# Anonymous Credentials with universe domain property. If no universe domain is provided, then
# the default universe domain is "googleapis.com".
class _AnonymousCredentialsWithUniverseDomain(ga_credentials.AnonymousCredentials):
    def __init__(self, universe_domain="googleapis.com"):
        super(_AnonymousCredentialsWithUniverseDomain, self).__init__()
        self._universe_domain = universe_domain

    @property
    def universe_domain(self):
        return self._universe_domain


def test__get_default_mtls_endpoint():
    api_endpoint = "example.googleapis.com"
    api_mtls_endpoint = "example.mtls.googleapis.com"
    sandbox_endpoint = "example.sandbox.googleapis.com"
    sandbox_mtls_endpoint = "example.mtls.sandbox.googleapis.com"
    non_googleapi = "api.example.com"

    assert EkmServiceClient._get_default_mtls_endpoint(None) is None
    assert (
        EkmServiceClient._get_default_mtls_endpoint(api_endpoint) == api_mtls_endpoint
    )
    assert (
        EkmServiceClient._get_default_mtls_endpoint(api_mtls_endpoint)
        == api_mtls_endpoint
    )
    assert (
        EkmServiceClient._get_default_mtls_endpoint(sandbox_endpoint)
        == sandbox_mtls_endpoint
    )
    assert (
        EkmServiceClient._get_default_mtls_endpoint(sandbox_mtls_endpoint)
        == sandbox_mtls_endpoint
    )
    assert EkmServiceClient._get_default_mtls_endpoint(non_googleapi) == non_googleapi


def test__read_environment_variables():
    assert EkmServiceClient._read_environment_variables() == (False, "auto", None)

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "true"}):
        assert EkmServiceClient._read_environment_variables() == (True, "auto", None)

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "false"}):
        assert EkmServiceClient._read_environment_variables() == (False, "auto", None)

    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "Unsupported"}
    ):
        with pytest.raises(ValueError) as excinfo:
            EkmServiceClient._read_environment_variables()
    assert (
        str(excinfo.value)
        == "Environment variable `GOOGLE_API_USE_CLIENT_CERTIFICATE` must be either `true` or `false`"
    )

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "never"}):
        assert EkmServiceClient._read_environment_variables() == (False, "never", None)

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "always"}):
        assert EkmServiceClient._read_environment_variables() == (False, "always", None)

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "auto"}):
        assert EkmServiceClient._read_environment_variables() == (False, "auto", None)

    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "Unsupported"}):
        with pytest.raises(MutualTLSChannelError) as excinfo:
            EkmServiceClient._read_environment_variables()
    assert (
        str(excinfo.value)
        == "Environment variable `GOOGLE_API_USE_MTLS_ENDPOINT` must be `never`, `auto` or `always`"
    )

    with mock.patch.dict(os.environ, {"GOOGLE_CLOUD_UNIVERSE_DOMAIN": "foo.com"}):
        assert EkmServiceClient._read_environment_variables() == (
            False,
            "auto",
            "foo.com",
        )


def test__get_client_cert_source():
    mock_provided_cert_source = mock.Mock()
    mock_default_cert_source = mock.Mock()

    assert EkmServiceClient._get_client_cert_source(None, False) is None
    assert (
        EkmServiceClient._get_client_cert_source(mock_provided_cert_source, False)
        is None
    )
    assert (
        EkmServiceClient._get_client_cert_source(mock_provided_cert_source, True)
        == mock_provided_cert_source
    )

    with mock.patch(
        "google.auth.transport.mtls.has_default_client_cert_source", return_value=True
    ):
        with mock.patch(
            "google.auth.transport.mtls.default_client_cert_source",
            return_value=mock_default_cert_source,
        ):
            assert (
                EkmServiceClient._get_client_cert_source(None, True)
                is mock_default_cert_source
            )
            assert (
                EkmServiceClient._get_client_cert_source(
                    mock_provided_cert_source, "true"
                )
                is mock_provided_cert_source
            )


@mock.patch.object(
    EkmServiceClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(EkmServiceClient),
)
@mock.patch.object(
    EkmServiceAsyncClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(EkmServiceAsyncClient),
)
def test__get_api_endpoint():
    api_override = "foo.com"
    mock_client_cert_source = mock.Mock()
    default_universe = EkmServiceClient._DEFAULT_UNIVERSE
    default_endpoint = EkmServiceClient._DEFAULT_ENDPOINT_TEMPLATE.format(
        UNIVERSE_DOMAIN=default_universe
    )
    mock_universe = "bar.com"
    mock_endpoint = EkmServiceClient._DEFAULT_ENDPOINT_TEMPLATE.format(
        UNIVERSE_DOMAIN=mock_universe
    )

    assert (
        EkmServiceClient._get_api_endpoint(
            api_override, mock_client_cert_source, default_universe, "always"
        )
        == api_override
    )
    assert (
        EkmServiceClient._get_api_endpoint(
            None, mock_client_cert_source, default_universe, "auto"
        )
        == EkmServiceClient.DEFAULT_MTLS_ENDPOINT
    )
    assert (
        EkmServiceClient._get_api_endpoint(None, None, default_universe, "auto")
        == default_endpoint
    )
    assert (
        EkmServiceClient._get_api_endpoint(None, None, default_universe, "always")
        == EkmServiceClient.DEFAULT_MTLS_ENDPOINT
    )
    assert (
        EkmServiceClient._get_api_endpoint(
            None, mock_client_cert_source, default_universe, "always"
        )
        == EkmServiceClient.DEFAULT_MTLS_ENDPOINT
    )
    assert (
        EkmServiceClient._get_api_endpoint(None, None, mock_universe, "never")
        == mock_endpoint
    )
    assert (
        EkmServiceClient._get_api_endpoint(None, None, default_universe, "never")
        == default_endpoint
    )

    with pytest.raises(MutualTLSChannelError) as excinfo:
        EkmServiceClient._get_api_endpoint(
            None, mock_client_cert_source, mock_universe, "auto"
        )
    assert (
        str(excinfo.value)
        == "mTLS is not supported in any universe other than googleapis.com."
    )


def test__get_universe_domain():
    client_universe_domain = "foo.com"
    universe_domain_env = "bar.com"

    assert (
        EkmServiceClient._get_universe_domain(
            client_universe_domain, universe_domain_env
        )
        == client_universe_domain
    )
    assert (
        EkmServiceClient._get_universe_domain(None, universe_domain_env)
        == universe_domain_env
    )
    assert (
        EkmServiceClient._get_universe_domain(None, None)
        == EkmServiceClient._DEFAULT_UNIVERSE
    )

    with pytest.raises(ValueError) as excinfo:
        EkmServiceClient._get_universe_domain("", None)
    assert str(excinfo.value) == "Universe Domain cannot be an empty string."


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name",
    [
        (EkmServiceClient, transports.EkmServiceGrpcTransport, "grpc"),
        (EkmServiceClient, transports.EkmServiceRestTransport, "rest"),
    ],
)
def test__validate_universe_domain(client_class, transport_class, transport_name):
    client = client_class(
        transport=transport_class(credentials=_AnonymousCredentialsWithUniverseDomain())
    )
    assert client._validate_universe_domain() == True

    # Test the case when universe is already validated.
    assert client._validate_universe_domain() == True

    if transport_name == "grpc":
        # Test the case where credentials are provided by the
        # `local_channel_credentials`. The default universes in both match.
        channel = grpc.secure_channel(
            "http://localhost/", grpc.local_channel_credentials()
        )
        client = client_class(transport=transport_class(channel=channel))
        assert client._validate_universe_domain() == True

        # Test the case where credentials do not exist: e.g. a transport is provided
        # with no credentials. Validation should still succeed because there is no
        # mismatch with non-existent credentials.
        channel = grpc.secure_channel(
            "http://localhost/", grpc.local_channel_credentials()
        )
        transport = transport_class(channel=channel)
        transport._credentials = None
        client = client_class(transport=transport)
        assert client._validate_universe_domain() == True

    # Test the case when there is a universe mismatch from the credentials.
    client = client_class(
        transport=transport_class(
            credentials=_AnonymousCredentialsWithUniverseDomain(
                universe_domain="foo.com"
            )
        )
    )
    with pytest.raises(ValueError) as excinfo:
        client._validate_universe_domain()
    assert (
        str(excinfo.value)
        == "The configured universe domain (googleapis.com) does not match the universe domain found in the credentials (foo.com). If you haven't configured the universe domain explicitly, `googleapis.com` is the default."
    )

    # Test the case when there is a universe mismatch from the client.
    #
    # TODO: Make this test unconditional once the minimum supported version of
    # google-api-core becomes 2.15.0 or higher.
    api_core_major, api_core_minor, _ = [
        int(part) for part in api_core_version.__version__.split(".")
    ]
    if api_core_major > 2 or (api_core_major == 2 and api_core_minor >= 15):
        client = client_class(
            client_options={"universe_domain": "bar.com"},
            transport=transport_class(
                credentials=_AnonymousCredentialsWithUniverseDomain(),
            ),
        )
        with pytest.raises(ValueError) as excinfo:
            client._validate_universe_domain()
        assert (
            str(excinfo.value)
            == "The configured universe domain (bar.com) does not match the universe domain found in the credentials (googleapis.com). If you haven't configured the universe domain explicitly, `googleapis.com` is the default."
        )


@pytest.mark.parametrize(
    "client_class,transport_name",
    [
        (EkmServiceClient, "grpc"),
        (EkmServiceAsyncClient, "grpc_asyncio"),
        (EkmServiceClient, "rest"),
    ],
)
def test_ekm_service_client_from_service_account_info(client_class, transport_name):
    creds = _AnonymousCredentialsWithUniverseDomain()
    with mock.patch.object(
        service_account.Credentials, "from_service_account_info"
    ) as factory:
        factory.return_value = creds
        info = {"valid": True}
        client = client_class.from_service_account_info(info, transport=transport_name)
        assert client.transport._credentials == creds
        assert isinstance(client, client_class)

        assert client.transport._host == (
            "cloudkms.googleapis.com:443"
            if transport_name in ["grpc", "grpc_asyncio"]
            else "https://cloudkms.googleapis.com"
        )


@pytest.mark.parametrize(
    "transport_class,transport_name",
    [
        (transports.EkmServiceGrpcTransport, "grpc"),
        (transports.EkmServiceGrpcAsyncIOTransport, "grpc_asyncio"),
        (transports.EkmServiceRestTransport, "rest"),
    ],
)
def test_ekm_service_client_service_account_always_use_jwt(
    transport_class, transport_name
):
    with mock.patch.object(
        service_account.Credentials, "with_always_use_jwt_access", create=True
    ) as use_jwt:
        creds = service_account.Credentials(None, None, None)
        transport = transport_class(credentials=creds, always_use_jwt_access=True)
        use_jwt.assert_called_once_with(True)

    with mock.patch.object(
        service_account.Credentials, "with_always_use_jwt_access", create=True
    ) as use_jwt:
        creds = service_account.Credentials(None, None, None)
        transport = transport_class(credentials=creds, always_use_jwt_access=False)
        use_jwt.assert_not_called()


@pytest.mark.parametrize(
    "client_class,transport_name",
    [
        (EkmServiceClient, "grpc"),
        (EkmServiceAsyncClient, "grpc_asyncio"),
        (EkmServiceClient, "rest"),
    ],
)
def test_ekm_service_client_from_service_account_file(client_class, transport_name):
    creds = _AnonymousCredentialsWithUniverseDomain()
    with mock.patch.object(
        service_account.Credentials, "from_service_account_file"
    ) as factory:
        factory.return_value = creds
        client = client_class.from_service_account_file(
            "dummy/file/path.json", transport=transport_name
        )
        assert client.transport._credentials == creds
        assert isinstance(client, client_class)

        client = client_class.from_service_account_json(
            "dummy/file/path.json", transport=transport_name
        )
        assert client.transport._credentials == creds
        assert isinstance(client, client_class)

        assert client.transport._host == (
            "cloudkms.googleapis.com:443"
            if transport_name in ["grpc", "grpc_asyncio"]
            else "https://cloudkms.googleapis.com"
        )


def test_ekm_service_client_get_transport_class():
    transport = EkmServiceClient.get_transport_class()
    available_transports = [
        transports.EkmServiceGrpcTransport,
        transports.EkmServiceRestTransport,
    ]
    assert transport in available_transports

    transport = EkmServiceClient.get_transport_class("grpc")
    assert transport == transports.EkmServiceGrpcTransport


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name",
    [
        (EkmServiceClient, transports.EkmServiceGrpcTransport, "grpc"),
        (
            EkmServiceAsyncClient,
            transports.EkmServiceGrpcAsyncIOTransport,
            "grpc_asyncio",
        ),
        (EkmServiceClient, transports.EkmServiceRestTransport, "rest"),
    ],
)
@mock.patch.object(
    EkmServiceClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(EkmServiceClient),
)
@mock.patch.object(
    EkmServiceAsyncClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(EkmServiceAsyncClient),
)
def test_ekm_service_client_client_options(
    client_class, transport_class, transport_name
):
    # Check that if channel is provided we won't create a new one.
    with mock.patch.object(EkmServiceClient, "get_transport_class") as gtc:
        transport = transport_class(
            credentials=_AnonymousCredentialsWithUniverseDomain()
        )
        client = client_class(transport=transport)
        gtc.assert_not_called()

    # Check that if channel is provided via str we will create a new one.
    with mock.patch.object(EkmServiceClient, "get_transport_class") as gtc:
        client = client_class(transport=transport_name)
        gtc.assert_called()

    # Check the case api_endpoint is provided.
    options = client_options.ClientOptions(api_endpoint="squid.clam.whelk")
    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(transport=transport_name, client_options=options)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file=None,
            host="squid.clam.whelk",
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )

    # Check the case api_endpoint is not provided and GOOGLE_API_USE_MTLS_ENDPOINT is
    # "never".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "never"}):
        with mock.patch.object(transport_class, "__init__") as patched:
            patched.return_value = None
            client = client_class(transport=transport_name)
            patched.assert_called_once_with(
                credentials=None,
                credentials_file=None,
                host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                    UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
                ),
                scopes=None,
                client_cert_source_for_mtls=None,
                quota_project_id=None,
                client_info=transports.base.DEFAULT_CLIENT_INFO,
                always_use_jwt_access=True,
                api_audience=None,
            )

    # Check the case api_endpoint is not provided and GOOGLE_API_USE_MTLS_ENDPOINT is
    # "always".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "always"}):
        with mock.patch.object(transport_class, "__init__") as patched:
            patched.return_value = None
            client = client_class(transport=transport_name)
            patched.assert_called_once_with(
                credentials=None,
                credentials_file=None,
                host=client.DEFAULT_MTLS_ENDPOINT,
                scopes=None,
                client_cert_source_for_mtls=None,
                quota_project_id=None,
                client_info=transports.base.DEFAULT_CLIENT_INFO,
                always_use_jwt_access=True,
                api_audience=None,
            )

    # Check the case api_endpoint is not provided and GOOGLE_API_USE_MTLS_ENDPOINT has
    # unsupported value.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "Unsupported"}):
        with pytest.raises(MutualTLSChannelError) as excinfo:
            client = client_class(transport=transport_name)
    assert (
        str(excinfo.value)
        == "Environment variable `GOOGLE_API_USE_MTLS_ENDPOINT` must be `never`, `auto` or `always`"
    )

    # Check the case GOOGLE_API_USE_CLIENT_CERTIFICATE has unsupported value.
    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "Unsupported"}
    ):
        with pytest.raises(ValueError) as excinfo:
            client = client_class(transport=transport_name)
    assert (
        str(excinfo.value)
        == "Environment variable `GOOGLE_API_USE_CLIENT_CERTIFICATE` must be either `true` or `false`"
    )

    # Check the case quota_project_id is provided
    options = client_options.ClientOptions(quota_project_id="octopus")
    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(client_options=options, transport=transport_name)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file=None,
            host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
            ),
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id="octopus",
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )
    # Check the case api_endpoint is provided
    options = client_options.ClientOptions(
        api_audience="https://language.googleapis.com"
    )
    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(client_options=options, transport=transport_name)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file=None,
            host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
            ),
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience="https://language.googleapis.com",
        )


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name,use_client_cert_env",
    [
        (EkmServiceClient, transports.EkmServiceGrpcTransport, "grpc", "true"),
        (
            EkmServiceAsyncClient,
            transports.EkmServiceGrpcAsyncIOTransport,
            "grpc_asyncio",
            "true",
        ),
        (EkmServiceClient, transports.EkmServiceGrpcTransport, "grpc", "false"),
        (
            EkmServiceAsyncClient,
            transports.EkmServiceGrpcAsyncIOTransport,
            "grpc_asyncio",
            "false",
        ),
        (EkmServiceClient, transports.EkmServiceRestTransport, "rest", "true"),
        (EkmServiceClient, transports.EkmServiceRestTransport, "rest", "false"),
    ],
)
@mock.patch.object(
    EkmServiceClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(EkmServiceClient),
)
@mock.patch.object(
    EkmServiceAsyncClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(EkmServiceAsyncClient),
)
@mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "auto"})
def test_ekm_service_client_mtls_env_auto(
    client_class, transport_class, transport_name, use_client_cert_env
):
    # This tests the endpoint autoswitch behavior. Endpoint is autoswitched to the default
    # mtls endpoint, if GOOGLE_API_USE_CLIENT_CERTIFICATE is "true" and client cert exists.

    # Check the case client_cert_source is provided. Whether client cert is used depends on
    # GOOGLE_API_USE_CLIENT_CERTIFICATE value.
    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": use_client_cert_env}
    ):
        options = client_options.ClientOptions(
            client_cert_source=client_cert_source_callback
        )
        with mock.patch.object(transport_class, "__init__") as patched:
            patched.return_value = None
            client = client_class(client_options=options, transport=transport_name)

            if use_client_cert_env == "false":
                expected_client_cert_source = None
                expected_host = client._DEFAULT_ENDPOINT_TEMPLATE.format(
                    UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
                )
            else:
                expected_client_cert_source = client_cert_source_callback
                expected_host = client.DEFAULT_MTLS_ENDPOINT

            patched.assert_called_once_with(
                credentials=None,
                credentials_file=None,
                host=expected_host,
                scopes=None,
                client_cert_source_for_mtls=expected_client_cert_source,
                quota_project_id=None,
                client_info=transports.base.DEFAULT_CLIENT_INFO,
                always_use_jwt_access=True,
                api_audience=None,
            )

    # Check the case ADC client cert is provided. Whether client cert is used depends on
    # GOOGLE_API_USE_CLIENT_CERTIFICATE value.
    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": use_client_cert_env}
    ):
        with mock.patch.object(transport_class, "__init__") as patched:
            with mock.patch(
                "google.auth.transport.mtls.has_default_client_cert_source",
                return_value=True,
            ):
                with mock.patch(
                    "google.auth.transport.mtls.default_client_cert_source",
                    return_value=client_cert_source_callback,
                ):
                    if use_client_cert_env == "false":
                        expected_host = client._DEFAULT_ENDPOINT_TEMPLATE.format(
                            UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
                        )
                        expected_client_cert_source = None
                    else:
                        expected_host = client.DEFAULT_MTLS_ENDPOINT
                        expected_client_cert_source = client_cert_source_callback

                    patched.return_value = None
                    client = client_class(transport=transport_name)
                    patched.assert_called_once_with(
                        credentials=None,
                        credentials_file=None,
                        host=expected_host,
                        scopes=None,
                        client_cert_source_for_mtls=expected_client_cert_source,
                        quota_project_id=None,
                        client_info=transports.base.DEFAULT_CLIENT_INFO,
                        always_use_jwt_access=True,
                        api_audience=None,
                    )

    # Check the case client_cert_source and ADC client cert are not provided.
    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": use_client_cert_env}
    ):
        with mock.patch.object(transport_class, "__init__") as patched:
            with mock.patch(
                "google.auth.transport.mtls.has_default_client_cert_source",
                return_value=False,
            ):
                patched.return_value = None
                client = client_class(transport=transport_name)
                patched.assert_called_once_with(
                    credentials=None,
                    credentials_file=None,
                    host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                        UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
                    ),
                    scopes=None,
                    client_cert_source_for_mtls=None,
                    quota_project_id=None,
                    client_info=transports.base.DEFAULT_CLIENT_INFO,
                    always_use_jwt_access=True,
                    api_audience=None,
                )


@pytest.mark.parametrize("client_class", [EkmServiceClient, EkmServiceAsyncClient])
@mock.patch.object(
    EkmServiceClient, "DEFAULT_ENDPOINT", modify_default_endpoint(EkmServiceClient)
)
@mock.patch.object(
    EkmServiceAsyncClient,
    "DEFAULT_ENDPOINT",
    modify_default_endpoint(EkmServiceAsyncClient),
)
def test_ekm_service_client_get_mtls_endpoint_and_cert_source(client_class):
    mock_client_cert_source = mock.Mock()

    # Test the case GOOGLE_API_USE_CLIENT_CERTIFICATE is "true".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "true"}):
        mock_api_endpoint = "foo"
        options = client_options.ClientOptions(
            client_cert_source=mock_client_cert_source, api_endpoint=mock_api_endpoint
        )
        api_endpoint, cert_source = client_class.get_mtls_endpoint_and_cert_source(
            options
        )
        assert api_endpoint == mock_api_endpoint
        assert cert_source == mock_client_cert_source

    # Test the case GOOGLE_API_USE_CLIENT_CERTIFICATE is "false".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "false"}):
        mock_client_cert_source = mock.Mock()
        mock_api_endpoint = "foo"
        options = client_options.ClientOptions(
            client_cert_source=mock_client_cert_source, api_endpoint=mock_api_endpoint
        )
        api_endpoint, cert_source = client_class.get_mtls_endpoint_and_cert_source(
            options
        )
        assert api_endpoint == mock_api_endpoint
        assert cert_source is None

    # Test the case GOOGLE_API_USE_MTLS_ENDPOINT is "never".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "never"}):
        api_endpoint, cert_source = client_class.get_mtls_endpoint_and_cert_source()
        assert api_endpoint == client_class.DEFAULT_ENDPOINT
        assert cert_source is None

    # Test the case GOOGLE_API_USE_MTLS_ENDPOINT is "always".
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "always"}):
        api_endpoint, cert_source = client_class.get_mtls_endpoint_and_cert_source()
        assert api_endpoint == client_class.DEFAULT_MTLS_ENDPOINT
        assert cert_source is None

    # Test the case GOOGLE_API_USE_MTLS_ENDPOINT is "auto" and default cert doesn't exist.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "true"}):
        with mock.patch(
            "google.auth.transport.mtls.has_default_client_cert_source",
            return_value=False,
        ):
            api_endpoint, cert_source = client_class.get_mtls_endpoint_and_cert_source()
            assert api_endpoint == client_class.DEFAULT_ENDPOINT
            assert cert_source is None

    # Test the case GOOGLE_API_USE_MTLS_ENDPOINT is "auto" and default cert exists.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "true"}):
        with mock.patch(
            "google.auth.transport.mtls.has_default_client_cert_source",
            return_value=True,
        ):
            with mock.patch(
                "google.auth.transport.mtls.default_client_cert_source",
                return_value=mock_client_cert_source,
            ):
                (
                    api_endpoint,
                    cert_source,
                ) = client_class.get_mtls_endpoint_and_cert_source()
                assert api_endpoint == client_class.DEFAULT_MTLS_ENDPOINT
                assert cert_source == mock_client_cert_source

    # Check the case api_endpoint is not provided and GOOGLE_API_USE_MTLS_ENDPOINT has
    # unsupported value.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "Unsupported"}):
        with pytest.raises(MutualTLSChannelError) as excinfo:
            client_class.get_mtls_endpoint_and_cert_source()

        assert (
            str(excinfo.value)
            == "Environment variable `GOOGLE_API_USE_MTLS_ENDPOINT` must be `never`, `auto` or `always`"
        )

    # Check the case GOOGLE_API_USE_CLIENT_CERTIFICATE has unsupported value.
    with mock.patch.dict(
        os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "Unsupported"}
    ):
        with pytest.raises(ValueError) as excinfo:
            client_class.get_mtls_endpoint_and_cert_source()

        assert (
            str(excinfo.value)
            == "Environment variable `GOOGLE_API_USE_CLIENT_CERTIFICATE` must be either `true` or `false`"
        )


@pytest.mark.parametrize("client_class", [EkmServiceClient, EkmServiceAsyncClient])
@mock.patch.object(
    EkmServiceClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(EkmServiceClient),
)
@mock.patch.object(
    EkmServiceAsyncClient,
    "_DEFAULT_ENDPOINT_TEMPLATE",
    modify_default_endpoint_template(EkmServiceAsyncClient),
)
def test_ekm_service_client_client_api_endpoint(client_class):
    mock_client_cert_source = client_cert_source_callback
    api_override = "foo.com"
    default_universe = EkmServiceClient._DEFAULT_UNIVERSE
    default_endpoint = EkmServiceClient._DEFAULT_ENDPOINT_TEMPLATE.format(
        UNIVERSE_DOMAIN=default_universe
    )
    mock_universe = "bar.com"
    mock_endpoint = EkmServiceClient._DEFAULT_ENDPOINT_TEMPLATE.format(
        UNIVERSE_DOMAIN=mock_universe
    )

    # If ClientOptions.api_endpoint is set and GOOGLE_API_USE_CLIENT_CERTIFICATE="true",
    # use ClientOptions.api_endpoint as the api endpoint regardless.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_CLIENT_CERTIFICATE": "true"}):
        with mock.patch(
            "google.auth.transport.requests.AuthorizedSession.configure_mtls_channel"
        ):
            options = client_options.ClientOptions(
                client_cert_source=mock_client_cert_source, api_endpoint=api_override
            )
            client = client_class(
                client_options=options,
                credentials=_AnonymousCredentialsWithUniverseDomain(),
            )
            assert client.api_endpoint == api_override

    # If ClientOptions.api_endpoint is not set and GOOGLE_API_USE_MTLS_ENDPOINT="never",
    # use the _DEFAULT_ENDPOINT_TEMPLATE populated with GDU as the api endpoint.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "never"}):
        client = client_class(credentials=_AnonymousCredentialsWithUniverseDomain())
        assert client.api_endpoint == default_endpoint

    # If ClientOptions.api_endpoint is not set and GOOGLE_API_USE_MTLS_ENDPOINT="always",
    # use the DEFAULT_MTLS_ENDPOINT as the api endpoint.
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "always"}):
        client = client_class(credentials=_AnonymousCredentialsWithUniverseDomain())
        assert client.api_endpoint == client_class.DEFAULT_MTLS_ENDPOINT

    # If ClientOptions.api_endpoint is not set, GOOGLE_API_USE_MTLS_ENDPOINT="auto" (default),
    # GOOGLE_API_USE_CLIENT_CERTIFICATE="false" (default), default cert source doesn't exist,
    # and ClientOptions.universe_domain="bar.com",
    # use the _DEFAULT_ENDPOINT_TEMPLATE populated with universe domain as the api endpoint.
    options = client_options.ClientOptions()
    universe_exists = hasattr(options, "universe_domain")
    if universe_exists:
        options = client_options.ClientOptions(universe_domain=mock_universe)
        client = client_class(
            client_options=options,
            credentials=_AnonymousCredentialsWithUniverseDomain(),
        )
    else:
        client = client_class(
            client_options=options,
            credentials=_AnonymousCredentialsWithUniverseDomain(),
        )
    assert client.api_endpoint == (
        mock_endpoint if universe_exists else default_endpoint
    )
    assert client.universe_domain == (
        mock_universe if universe_exists else default_universe
    )

    # If ClientOptions does not have a universe domain attribute and GOOGLE_API_USE_MTLS_ENDPOINT="never",
    # use the _DEFAULT_ENDPOINT_TEMPLATE populated with GDU as the api endpoint.
    options = client_options.ClientOptions()
    if hasattr(options, "universe_domain"):
        delattr(options, "universe_domain")
    with mock.patch.dict(os.environ, {"GOOGLE_API_USE_MTLS_ENDPOINT": "never"}):
        client = client_class(
            client_options=options,
            credentials=_AnonymousCredentialsWithUniverseDomain(),
        )
        assert client.api_endpoint == default_endpoint


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name",
    [
        (EkmServiceClient, transports.EkmServiceGrpcTransport, "grpc"),
        (
            EkmServiceAsyncClient,
            transports.EkmServiceGrpcAsyncIOTransport,
            "grpc_asyncio",
        ),
        (EkmServiceClient, transports.EkmServiceRestTransport, "rest"),
    ],
)
def test_ekm_service_client_client_options_scopes(
    client_class, transport_class, transport_name
):
    # Check the case scopes are provided.
    options = client_options.ClientOptions(
        scopes=["1", "2"],
    )
    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(client_options=options, transport=transport_name)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file=None,
            host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
            ),
            scopes=["1", "2"],
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name,grpc_helpers",
    [
        (EkmServiceClient, transports.EkmServiceGrpcTransport, "grpc", grpc_helpers),
        (
            EkmServiceAsyncClient,
            transports.EkmServiceGrpcAsyncIOTransport,
            "grpc_asyncio",
            grpc_helpers_async,
        ),
        (EkmServiceClient, transports.EkmServiceRestTransport, "rest", None),
    ],
)
def test_ekm_service_client_client_options_credentials_file(
    client_class, transport_class, transport_name, grpc_helpers
):
    # Check the case credentials file is provided.
    options = client_options.ClientOptions(credentials_file="credentials.json")

    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(client_options=options, transport=transport_name)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file="credentials.json",
            host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
            ),
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )


def test_ekm_service_client_client_options_from_dict():
    with mock.patch(
        "google.cloud.kms_v1.services.ekm_service.transports.EkmServiceGrpcTransport.__init__"
    ) as grpc_transport:
        grpc_transport.return_value = None
        client = EkmServiceClient(client_options={"api_endpoint": "squid.clam.whelk"})
        grpc_transport.assert_called_once_with(
            credentials=None,
            credentials_file=None,
            host="squid.clam.whelk",
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )


@pytest.mark.parametrize(
    "client_class,transport_class,transport_name,grpc_helpers",
    [
        (EkmServiceClient, transports.EkmServiceGrpcTransport, "grpc", grpc_helpers),
        (
            EkmServiceAsyncClient,
            transports.EkmServiceGrpcAsyncIOTransport,
            "grpc_asyncio",
            grpc_helpers_async,
        ),
    ],
)
def test_ekm_service_client_create_channel_credentials_file(
    client_class, transport_class, transport_name, grpc_helpers
):
    # Check the case credentials file is provided.
    options = client_options.ClientOptions(credentials_file="credentials.json")

    with mock.patch.object(transport_class, "__init__") as patched:
        patched.return_value = None
        client = client_class(client_options=options, transport=transport_name)
        patched.assert_called_once_with(
            credentials=None,
            credentials_file="credentials.json",
            host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
            ),
            scopes=None,
            client_cert_source_for_mtls=None,
            quota_project_id=None,
            client_info=transports.base.DEFAULT_CLIENT_INFO,
            always_use_jwt_access=True,
            api_audience=None,
        )

    # test that the credentials from file are saved and used as the credentials.
    with mock.patch.object(
        google.auth, "load_credentials_from_file", autospec=True
    ) as load_creds, mock.patch.object(
        google.auth, "default", autospec=True
    ) as adc, mock.patch.object(
        grpc_helpers, "create_channel"
    ) as create_channel:
        creds = _AnonymousCredentialsWithUniverseDomain()
        file_creds = _AnonymousCredentialsWithUniverseDomain()
        load_creds.return_value = (file_creds, None)
        adc.return_value = (creds, None)
        client = client_class(client_options=options, transport=transport_name)
        create_channel.assert_called_with(
            "cloudkms.googleapis.com:443",
            credentials=file_creds,
            credentials_file=None,
            quota_project_id=None,
            default_scopes=(
                "https://www.googleapis.com/auth/cloud-platform",
                "https://www.googleapis.com/auth/cloudkms",
            ),
            scopes=None,
            default_host="cloudkms.googleapis.com",
            ssl_credentials=None,
            options=[
                ("grpc.max_send_message_length", -1),
                ("grpc.max_receive_message_length", -1),
            ],
        )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.ListEkmConnectionsRequest,
        dict,
    ],
)
def test_list_ekm_connections(request_type, transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_ekm_connections), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.ListEkmConnectionsResponse(
            next_page_token="next_page_token_value",
            total_size=1086,
        )
        response = client.list_ekm_connections(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.ListEkmConnectionsRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, pagers.ListEkmConnectionsPager)
    assert response.next_page_token == "next_page_token_value"
    assert response.total_size == 1086


def test_list_ekm_connections_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_ekm_connections), "__call__"
    ) as call:
        client.list_ekm_connections()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.ListEkmConnectionsRequest()


@pytest.mark.asyncio
async def test_list_ekm_connections_async(
    transport: str = "grpc_asyncio", request_type=ekm_service.ListEkmConnectionsRequest
):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_ekm_connections), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.ListEkmConnectionsResponse(
                next_page_token="next_page_token_value",
                total_size=1086,
            )
        )
        response = await client.list_ekm_connections(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.ListEkmConnectionsRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, pagers.ListEkmConnectionsAsyncPager)
    assert response.next_page_token == "next_page_token_value"
    assert response.total_size == 1086


@pytest.mark.asyncio
async def test_list_ekm_connections_async_from_dict():
    await test_list_ekm_connections_async(request_type=dict)


def test_list_ekm_connections_field_headers():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.ListEkmConnectionsRequest()

    request.parent = "parent_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_ekm_connections), "__call__"
    ) as call:
        call.return_value = ekm_service.ListEkmConnectionsResponse()
        client.list_ekm_connections(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "parent=parent_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_list_ekm_connections_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.ListEkmConnectionsRequest()

    request.parent = "parent_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_ekm_connections), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.ListEkmConnectionsResponse()
        )
        await client.list_ekm_connections(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "parent=parent_value",
    ) in kw["metadata"]


def test_list_ekm_connections_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_ekm_connections), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.ListEkmConnectionsResponse()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.list_ekm_connections(
            parent="parent_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].parent
        mock_val = "parent_value"
        assert arg == mock_val


def test_list_ekm_connections_flattened_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.list_ekm_connections(
            ekm_service.ListEkmConnectionsRequest(),
            parent="parent_value",
        )


@pytest.mark.asyncio
async def test_list_ekm_connections_flattened_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_ekm_connections), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.ListEkmConnectionsResponse()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.ListEkmConnectionsResponse()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.list_ekm_connections(
            parent="parent_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].parent
        mock_val = "parent_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_list_ekm_connections_flattened_error_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.list_ekm_connections(
            ekm_service.ListEkmConnectionsRequest(),
            parent="parent_value",
        )


def test_list_ekm_connections_pager(transport_name: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport_name,
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_ekm_connections), "__call__"
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                ],
                next_page_token="abc",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[],
                next_page_token="def",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                ],
                next_page_token="ghi",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                ],
            ),
            RuntimeError,
        )

        metadata = ()
        metadata = tuple(metadata) + (
            gapic_v1.routing_header.to_grpc_metadata((("parent", ""),)),
        )
        pager = client.list_ekm_connections(request={})

        assert pager._metadata == metadata

        results = list(pager)
        assert len(results) == 6
        assert all(isinstance(i, ekm_service.EkmConnection) for i in results)


def test_list_ekm_connections_pages(transport_name: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport_name,
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_ekm_connections), "__call__"
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                ],
                next_page_token="abc",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[],
                next_page_token="def",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                ],
                next_page_token="ghi",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                ],
            ),
            RuntimeError,
        )
        pages = list(client.list_ekm_connections(request={}).pages)
        for page_, token in zip(pages, ["abc", "def", "ghi", ""]):
            assert page_.raw_page.next_page_token == token


@pytest.mark.asyncio
async def test_list_ekm_connections_async_pager():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_ekm_connections),
        "__call__",
        new_callable=mock.AsyncMock,
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                ],
                next_page_token="abc",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[],
                next_page_token="def",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                ],
                next_page_token="ghi",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                ],
            ),
            RuntimeError,
        )
        async_pager = await client.list_ekm_connections(
            request={},
        )
        assert async_pager.next_page_token == "abc"
        responses = []
        async for response in async_pager:  # pragma: no branch
            responses.append(response)

        assert len(responses) == 6
        assert all(isinstance(i, ekm_service.EkmConnection) for i in responses)


@pytest.mark.asyncio
async def test_list_ekm_connections_async_pages():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.list_ekm_connections),
        "__call__",
        new_callable=mock.AsyncMock,
    ) as call:
        # Set the response to a series of pages.
        call.side_effect = (
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                ],
                next_page_token="abc",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[],
                next_page_token="def",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                ],
                next_page_token="ghi",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                ],
            ),
            RuntimeError,
        )
        pages = []
        # Workaround issue in python 3.9 related to code coverage by adding `# pragma: no branch`
        # See https://github.com/googleapis/gapic-generator-python/pull/1174#issuecomment-1025132372
        async for page_ in (  # pragma: no branch
            await client.list_ekm_connections(request={})
        ).pages:
            pages.append(page_)
        for page_, token in zip(pages, ["abc", "def", "ghi", ""]):
            assert page_.raw_page.next_page_token == token


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.GetEkmConnectionRequest,
        dict,
    ],
)
def test_get_ekm_connection(request_type, transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConnection(
            name="name_value",
            etag="etag_value",
            key_management_mode=ekm_service.EkmConnection.KeyManagementMode.MANUAL,
            crypto_space_path="crypto_space_path_value",
        )
        response = client.get_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.GetEkmConnectionRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConnection)
    assert response.name == "name_value"
    assert response.etag == "etag_value"
    assert (
        response.key_management_mode
        == ekm_service.EkmConnection.KeyManagementMode.MANUAL
    )
    assert response.crypto_space_path == "crypto_space_path_value"


def test_get_ekm_connection_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_ekm_connection), "__call__"
    ) as call:
        client.get_ekm_connection()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.GetEkmConnectionRequest()


@pytest.mark.asyncio
async def test_get_ekm_connection_async(
    transport: str = "grpc_asyncio", request_type=ekm_service.GetEkmConnectionRequest
):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConnection(
                name="name_value",
                etag="etag_value",
                key_management_mode=ekm_service.EkmConnection.KeyManagementMode.MANUAL,
                crypto_space_path="crypto_space_path_value",
            )
        )
        response = await client.get_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.GetEkmConnectionRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConnection)
    assert response.name == "name_value"
    assert response.etag == "etag_value"
    assert (
        response.key_management_mode
        == ekm_service.EkmConnection.KeyManagementMode.MANUAL
    )
    assert response.crypto_space_path == "crypto_space_path_value"


@pytest.mark.asyncio
async def test_get_ekm_connection_async_from_dict():
    await test_get_ekm_connection_async(request_type=dict)


def test_get_ekm_connection_field_headers():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.GetEkmConnectionRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_ekm_connection), "__call__"
    ) as call:
        call.return_value = ekm_service.EkmConnection()
        client.get_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_get_ekm_connection_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.GetEkmConnectionRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_ekm_connection), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConnection()
        )
        await client.get_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_get_ekm_connection_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConnection()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.get_ekm_connection(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


def test_get_ekm_connection_flattened_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.get_ekm_connection(
            ekm_service.GetEkmConnectionRequest(),
            name="name_value",
        )


@pytest.mark.asyncio
async def test_get_ekm_connection_flattened_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.get_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConnection()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConnection()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.get_ekm_connection(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_get_ekm_connection_flattened_error_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.get_ekm_connection(
            ekm_service.GetEkmConnectionRequest(),
            name="name_value",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.CreateEkmConnectionRequest,
        dict,
    ],
)
def test_create_ekm_connection(request_type, transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConnection(
            name="name_value",
            etag="etag_value",
            key_management_mode=ekm_service.EkmConnection.KeyManagementMode.MANUAL,
            crypto_space_path="crypto_space_path_value",
        )
        response = client.create_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.CreateEkmConnectionRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConnection)
    assert response.name == "name_value"
    assert response.etag == "etag_value"
    assert (
        response.key_management_mode
        == ekm_service.EkmConnection.KeyManagementMode.MANUAL
    )
    assert response.crypto_space_path == "crypto_space_path_value"


def test_create_ekm_connection_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_ekm_connection), "__call__"
    ) as call:
        client.create_ekm_connection()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.CreateEkmConnectionRequest()


@pytest.mark.asyncio
async def test_create_ekm_connection_async(
    transport: str = "grpc_asyncio", request_type=ekm_service.CreateEkmConnectionRequest
):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConnection(
                name="name_value",
                etag="etag_value",
                key_management_mode=ekm_service.EkmConnection.KeyManagementMode.MANUAL,
                crypto_space_path="crypto_space_path_value",
            )
        )
        response = await client.create_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.CreateEkmConnectionRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConnection)
    assert response.name == "name_value"
    assert response.etag == "etag_value"
    assert (
        response.key_management_mode
        == ekm_service.EkmConnection.KeyManagementMode.MANUAL
    )
    assert response.crypto_space_path == "crypto_space_path_value"


@pytest.mark.asyncio
async def test_create_ekm_connection_async_from_dict():
    await test_create_ekm_connection_async(request_type=dict)


def test_create_ekm_connection_field_headers():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.CreateEkmConnectionRequest()

    request.parent = "parent_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_ekm_connection), "__call__"
    ) as call:
        call.return_value = ekm_service.EkmConnection()
        client.create_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "parent=parent_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_create_ekm_connection_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.CreateEkmConnectionRequest()

    request.parent = "parent_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_ekm_connection), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConnection()
        )
        await client.create_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "parent=parent_value",
    ) in kw["metadata"]


def test_create_ekm_connection_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConnection()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.create_ekm_connection(
            parent="parent_value",
            ekm_connection_id="ekm_connection_id_value",
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].parent
        mock_val = "parent_value"
        assert arg == mock_val
        arg = args[0].ekm_connection_id
        mock_val = "ekm_connection_id_value"
        assert arg == mock_val
        arg = args[0].ekm_connection
        mock_val = ekm_service.EkmConnection(name="name_value")
        assert arg == mock_val


def test_create_ekm_connection_flattened_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.create_ekm_connection(
            ekm_service.CreateEkmConnectionRequest(),
            parent="parent_value",
            ekm_connection_id="ekm_connection_id_value",
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
        )


@pytest.mark.asyncio
async def test_create_ekm_connection_flattened_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.create_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConnection()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConnection()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.create_ekm_connection(
            parent="parent_value",
            ekm_connection_id="ekm_connection_id_value",
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].parent
        mock_val = "parent_value"
        assert arg == mock_val
        arg = args[0].ekm_connection_id
        mock_val = "ekm_connection_id_value"
        assert arg == mock_val
        arg = args[0].ekm_connection
        mock_val = ekm_service.EkmConnection(name="name_value")
        assert arg == mock_val


@pytest.mark.asyncio
async def test_create_ekm_connection_flattened_error_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.create_ekm_connection(
            ekm_service.CreateEkmConnectionRequest(),
            parent="parent_value",
            ekm_connection_id="ekm_connection_id_value",
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
        )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.UpdateEkmConnectionRequest,
        dict,
    ],
)
def test_update_ekm_connection(request_type, transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConnection(
            name="name_value",
            etag="etag_value",
            key_management_mode=ekm_service.EkmConnection.KeyManagementMode.MANUAL,
            crypto_space_path="crypto_space_path_value",
        )
        response = client.update_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.UpdateEkmConnectionRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConnection)
    assert response.name == "name_value"
    assert response.etag == "etag_value"
    assert (
        response.key_management_mode
        == ekm_service.EkmConnection.KeyManagementMode.MANUAL
    )
    assert response.crypto_space_path == "crypto_space_path_value"


def test_update_ekm_connection_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_connection), "__call__"
    ) as call:
        client.update_ekm_connection()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.UpdateEkmConnectionRequest()


@pytest.mark.asyncio
async def test_update_ekm_connection_async(
    transport: str = "grpc_asyncio", request_type=ekm_service.UpdateEkmConnectionRequest
):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConnection(
                name="name_value",
                etag="etag_value",
                key_management_mode=ekm_service.EkmConnection.KeyManagementMode.MANUAL,
                crypto_space_path="crypto_space_path_value",
            )
        )
        response = await client.update_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.UpdateEkmConnectionRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConnection)
    assert response.name == "name_value"
    assert response.etag == "etag_value"
    assert (
        response.key_management_mode
        == ekm_service.EkmConnection.KeyManagementMode.MANUAL
    )
    assert response.crypto_space_path == "crypto_space_path_value"


@pytest.mark.asyncio
async def test_update_ekm_connection_async_from_dict():
    await test_update_ekm_connection_async(request_type=dict)


def test_update_ekm_connection_field_headers():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.UpdateEkmConnectionRequest()

    request.ekm_connection.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_connection), "__call__"
    ) as call:
        call.return_value = ekm_service.EkmConnection()
        client.update_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "ekm_connection.name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_update_ekm_connection_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.UpdateEkmConnectionRequest()

    request.ekm_connection.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_connection), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConnection()
        )
        await client.update_ekm_connection(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "ekm_connection.name=name_value",
    ) in kw["metadata"]


def test_update_ekm_connection_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConnection()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.update_ekm_connection(
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].ekm_connection
        mock_val = ekm_service.EkmConnection(name="name_value")
        assert arg == mock_val
        arg = args[0].update_mask
        mock_val = field_mask_pb2.FieldMask(paths=["paths_value"])
        assert arg == mock_val


def test_update_ekm_connection_flattened_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.update_ekm_connection(
            ekm_service.UpdateEkmConnectionRequest(),
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )


@pytest.mark.asyncio
async def test_update_ekm_connection_flattened_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_connection), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConnection()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConnection()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.update_ekm_connection(
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].ekm_connection
        mock_val = ekm_service.EkmConnection(name="name_value")
        assert arg == mock_val
        arg = args[0].update_mask
        mock_val = field_mask_pb2.FieldMask(paths=["paths_value"])
        assert arg == mock_val


@pytest.mark.asyncio
async def test_update_ekm_connection_flattened_error_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.update_ekm_connection(
            ekm_service.UpdateEkmConnectionRequest(),
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.GetEkmConfigRequest,
        dict,
    ],
)
def test_get_ekm_config(request_type, transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_ekm_config), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConfig(
            name="name_value",
            default_ekm_connection="default_ekm_connection_value",
        )
        response = client.get_ekm_config(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.GetEkmConfigRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConfig)
    assert response.name == "name_value"
    assert response.default_ekm_connection == "default_ekm_connection_value"


def test_get_ekm_config_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_ekm_config), "__call__") as call:
        client.get_ekm_config()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.GetEkmConfigRequest()


@pytest.mark.asyncio
async def test_get_ekm_config_async(
    transport: str = "grpc_asyncio", request_type=ekm_service.GetEkmConfigRequest
):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_ekm_config), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConfig(
                name="name_value",
                default_ekm_connection="default_ekm_connection_value",
            )
        )
        response = await client.get_ekm_config(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.GetEkmConfigRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConfig)
    assert response.name == "name_value"
    assert response.default_ekm_connection == "default_ekm_connection_value"


@pytest.mark.asyncio
async def test_get_ekm_config_async_from_dict():
    await test_get_ekm_config_async(request_type=dict)


def test_get_ekm_config_field_headers():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.GetEkmConfigRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_ekm_config), "__call__") as call:
        call.return_value = ekm_service.EkmConfig()
        client.get_ekm_config(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_get_ekm_config_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.GetEkmConfigRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_ekm_config), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConfig()
        )
        await client.get_ekm_config(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_get_ekm_config_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_ekm_config), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConfig()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.get_ekm_config(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


def test_get_ekm_config_flattened_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.get_ekm_config(
            ekm_service.GetEkmConfigRequest(),
            name="name_value",
        )


@pytest.mark.asyncio
async def test_get_ekm_config_flattened_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_ekm_config), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConfig()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConfig()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.get_ekm_config(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_get_ekm_config_flattened_error_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.get_ekm_config(
            ekm_service.GetEkmConfigRequest(),
            name="name_value",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.UpdateEkmConfigRequest,
        dict,
    ],
)
def test_update_ekm_config(request_type, transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_config), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConfig(
            name="name_value",
            default_ekm_connection="default_ekm_connection_value",
        )
        response = client.update_ekm_config(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.UpdateEkmConfigRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConfig)
    assert response.name == "name_value"
    assert response.default_ekm_connection == "default_ekm_connection_value"


def test_update_ekm_config_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_config), "__call__"
    ) as call:
        client.update_ekm_config()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.UpdateEkmConfigRequest()


@pytest.mark.asyncio
async def test_update_ekm_config_async(
    transport: str = "grpc_asyncio", request_type=ekm_service.UpdateEkmConfigRequest
):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_config), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConfig(
                name="name_value",
                default_ekm_connection="default_ekm_connection_value",
            )
        )
        response = await client.update_ekm_config(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.UpdateEkmConfigRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConfig)
    assert response.name == "name_value"
    assert response.default_ekm_connection == "default_ekm_connection_value"


@pytest.mark.asyncio
async def test_update_ekm_config_async_from_dict():
    await test_update_ekm_config_async(request_type=dict)


def test_update_ekm_config_field_headers():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.UpdateEkmConfigRequest()

    request.ekm_config.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_config), "__call__"
    ) as call:
        call.return_value = ekm_service.EkmConfig()
        client.update_ekm_config(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "ekm_config.name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_update_ekm_config_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.UpdateEkmConfigRequest()

    request.ekm_config.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_config), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConfig()
        )
        await client.update_ekm_config(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "ekm_config.name=name_value",
    ) in kw["metadata"]


def test_update_ekm_config_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_config), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConfig()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.update_ekm_config(
            ekm_config=ekm_service.EkmConfig(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].ekm_config
        mock_val = ekm_service.EkmConfig(name="name_value")
        assert arg == mock_val
        arg = args[0].update_mask
        mock_val = field_mask_pb2.FieldMask(paths=["paths_value"])
        assert arg == mock_val


def test_update_ekm_config_flattened_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.update_ekm_config(
            ekm_service.UpdateEkmConfigRequest(),
            ekm_config=ekm_service.EkmConfig(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )


@pytest.mark.asyncio
async def test_update_ekm_config_flattened_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.update_ekm_config), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.EkmConfig()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.EkmConfig()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.update_ekm_config(
            ekm_config=ekm_service.EkmConfig(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].ekm_config
        mock_val = ekm_service.EkmConfig(name="name_value")
        assert arg == mock_val
        arg = args[0].update_mask
        mock_val = field_mask_pb2.FieldMask(paths=["paths_value"])
        assert arg == mock_val


@pytest.mark.asyncio
async def test_update_ekm_config_flattened_error_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.update_ekm_config(
            ekm_service.UpdateEkmConfigRequest(),
            ekm_config=ekm_service.EkmConfig(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.VerifyConnectivityRequest,
        dict,
    ],
)
def test_verify_connectivity(request_type, transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.verify_connectivity), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.VerifyConnectivityResponse()
        response = client.verify_connectivity(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.VerifyConnectivityRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.VerifyConnectivityResponse)


def test_verify_connectivity_empty_call():
    # This test is a coverage failsafe to make sure that totally empty calls,
    # i.e. request == None and no flattened fields passed, work.
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="grpc",
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.verify_connectivity), "__call__"
    ) as call:
        client.verify_connectivity()
        call.assert_called()
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.VerifyConnectivityRequest()


@pytest.mark.asyncio
async def test_verify_connectivity_async(
    transport: str = "grpc_asyncio", request_type=ekm_service.VerifyConnectivityRequest
):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = request_type()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.verify_connectivity), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.VerifyConnectivityResponse()
        )
        response = await client.verify_connectivity(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == ekm_service.VerifyConnectivityRequest()

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.VerifyConnectivityResponse)


@pytest.mark.asyncio
async def test_verify_connectivity_async_from_dict():
    await test_verify_connectivity_async(request_type=dict)


def test_verify_connectivity_field_headers():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.VerifyConnectivityRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.verify_connectivity), "__call__"
    ) as call:
        call.return_value = ekm_service.VerifyConnectivityResponse()
        client.verify_connectivity(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_verify_connectivity_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = ekm_service.VerifyConnectivityRequest()

    request.name = "name_value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.verify_connectivity), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.VerifyConnectivityResponse()
        )
        await client.verify_connectivity(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=name_value",
    ) in kw["metadata"]


def test_verify_connectivity_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.verify_connectivity), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.VerifyConnectivityResponse()
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        client.verify_connectivity(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


def test_verify_connectivity_flattened_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.verify_connectivity(
            ekm_service.VerifyConnectivityRequest(),
            name="name_value",
        )


@pytest.mark.asyncio
async def test_verify_connectivity_flattened_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.verify_connectivity), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = ekm_service.VerifyConnectivityResponse()

        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            ekm_service.VerifyConnectivityResponse()
        )
        # Call the method with a truthy value for each flattened field,
        # using the keyword arguments to the method.
        response = await client.verify_connectivity(
            name="name_value",
        )

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        arg = args[0].name
        mock_val = "name_value"
        assert arg == mock_val


@pytest.mark.asyncio
async def test_verify_connectivity_flattened_error_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        await client.verify_connectivity(
            ekm_service.VerifyConnectivityRequest(),
            name="name_value",
        )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.ListEkmConnectionsRequest,
        dict,
    ],
)
def test_list_ekm_connections_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # send a request that will satisfy transcoding
    request_init = {"parent": "projects/sample1/locations/sample2"}
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.ListEkmConnectionsResponse(
            next_page_token="next_page_token_value",
            total_size=1086,
        )

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.ListEkmConnectionsResponse.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value
        response = client.list_ekm_connections(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, pagers.ListEkmConnectionsPager)
    assert response.next_page_token == "next_page_token_value"
    assert response.total_size == 1086


def test_list_ekm_connections_rest_required_fields(
    request_type=ekm_service.ListEkmConnectionsRequest,
):
    transport_class = transports.EkmServiceRestTransport

    request_init = {}
    request_init["parent"] = ""
    request = request_type(**request_init)
    pb_request = request_type.pb(request)
    jsonified_request = json.loads(
        json_format.MessageToJson(
            pb_request,
            including_default_value_fields=False,
            use_integers_for_enums=False,
        )
    )

    # verify fields with default values are dropped

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).list_ekm_connections._get_unset_required_fields(jsonified_request)
    jsonified_request.update(unset_fields)

    # verify required fields with default values are now present

    jsonified_request["parent"] = "parent_value"

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).list_ekm_connections._get_unset_required_fields(jsonified_request)
    # Check that path parameters and body parameters are not mixing in.
    assert not set(unset_fields) - set(
        (
            "filter",
            "order_by",
            "page_size",
            "page_token",
        )
    )
    jsonified_request.update(unset_fields)

    # verify required fields with non-default values are left alone
    assert "parent" in jsonified_request
    assert jsonified_request["parent"] == "parent_value"

    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request = request_type(**request_init)

    # Designate an appropriate value for the returned response.
    return_value = ekm_service.ListEkmConnectionsResponse()
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(Session, "request") as req:
        # We need to mock transcode() because providing default values
        # for required fields will fail the real version if the http_options
        # expect actual values for those fields.
        with mock.patch.object(path_template, "transcode") as transcode:
            # A uri without fields and an empty body will force all the
            # request fields to show up in the query_params.
            pb_request = request_type.pb(request)
            transcode_result = {
                "uri": "v1/sample_method",
                "method": "get",
                "query_params": pb_request,
            }
            transcode.return_value = transcode_result

            response_value = Response()
            response_value.status_code = 200

            # Convert return value to protobuf type
            return_value = ekm_service.ListEkmConnectionsResponse.pb(return_value)
            json_return_value = json_format.MessageToJson(return_value)

            response_value._content = json_return_value.encode("UTF-8")
            req.return_value = response_value

            response = client.list_ekm_connections(request)

            expected_params = [("$alt", "json;enum-encoding=int")]
            actual_params = req.call_args.kwargs["params"]
            assert expected_params == actual_params


def test_list_ekm_connections_rest_unset_required_fields():
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain
    )

    unset_fields = transport.list_ekm_connections._get_unset_required_fields({})
    assert set(unset_fields) == (
        set(
            (
                "filter",
                "orderBy",
                "pageSize",
                "pageToken",
            )
        )
        & set(("parent",))
    )


@pytest.mark.parametrize("null_interceptor", [True, False])
def test_list_ekm_connections_rest_interceptors(null_interceptor):
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        interceptor=None
        if null_interceptor
        else transports.EkmServiceRestInterceptor(),
    )
    client = EkmServiceClient(transport=transport)
    with mock.patch.object(
        type(client.transport._session), "request"
    ) as req, mock.patch.object(
        path_template, "transcode"
    ) as transcode, mock.patch.object(
        transports.EkmServiceRestInterceptor, "post_list_ekm_connections"
    ) as post, mock.patch.object(
        transports.EkmServiceRestInterceptor, "pre_list_ekm_connections"
    ) as pre:
        pre.assert_not_called()
        post.assert_not_called()
        pb_message = ekm_service.ListEkmConnectionsRequest.pb(
            ekm_service.ListEkmConnectionsRequest()
        )
        transcode.return_value = {
            "method": "post",
            "uri": "my_uri",
            "body": pb_message,
            "query_params": pb_message,
        }

        req.return_value = Response()
        req.return_value.status_code = 200
        req.return_value.request = PreparedRequest()
        req.return_value._content = ekm_service.ListEkmConnectionsResponse.to_json(
            ekm_service.ListEkmConnectionsResponse()
        )

        request = ekm_service.ListEkmConnectionsRequest()
        metadata = [
            ("key", "val"),
            ("cephalopod", "squid"),
        ]
        pre.return_value = request, metadata
        post.return_value = ekm_service.ListEkmConnectionsResponse()

        client.list_ekm_connections(
            request,
            metadata=[
                ("key", "val"),
                ("cephalopod", "squid"),
            ],
        )

        pre.assert_called_once()
        post.assert_called_once()


def test_list_ekm_connections_rest_bad_request(
    transport: str = "rest", request_type=ekm_service.ListEkmConnectionsRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # send a request that will satisfy transcoding
    request_init = {"parent": "projects/sample1/locations/sample2"}
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.list_ekm_connections(request)


def test_list_ekm_connections_rest_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.ListEkmConnectionsResponse()

        # get arguments that satisfy an http rule for this method
        sample_request = {"parent": "projects/sample1/locations/sample2"}

        # get truthy value for each flattened field
        mock_args = dict(
            parent="parent_value",
        )
        mock_args.update(sample_request)

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.ListEkmConnectionsResponse.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)
        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        client.list_ekm_connections(**mock_args)

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(req.mock_calls) == 1
        _, args, _ = req.mock_calls[0]
        assert path_template.validate(
            "%s/v1/{parent=projects/*/locations/*}/ekmConnections"
            % client.transport._host,
            args[1],
        )


def test_list_ekm_connections_rest_flattened_error(transport: str = "rest"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.list_ekm_connections(
            ekm_service.ListEkmConnectionsRequest(),
            parent="parent_value",
        )


def test_list_ekm_connections_rest_pager(transport: str = "rest"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(Session, "request") as req:
        # TODO(kbandes): remove this mock unless there's a good reason for it.
        # with mock.patch.object(path_template, 'transcode') as transcode:
        # Set the response as a series of pages
        response = (
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                ],
                next_page_token="abc",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[],
                next_page_token="def",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                ],
                next_page_token="ghi",
            ),
            ekm_service.ListEkmConnectionsResponse(
                ekm_connections=[
                    ekm_service.EkmConnection(),
                    ekm_service.EkmConnection(),
                ],
            ),
        )
        # Two responses for two calls
        response = response + response

        # Wrap the values into proper Response objs
        response = tuple(
            ekm_service.ListEkmConnectionsResponse.to_json(x) for x in response
        )
        return_values = tuple(Response() for i in response)
        for return_val, response_val in zip(return_values, response):
            return_val._content = response_val.encode("UTF-8")
            return_val.status_code = 200
        req.side_effect = return_values

        sample_request = {"parent": "projects/sample1/locations/sample2"}

        pager = client.list_ekm_connections(request=sample_request)

        results = list(pager)
        assert len(results) == 6
        assert all(isinstance(i, ekm_service.EkmConnection) for i in results)

        pages = list(client.list_ekm_connections(request=sample_request).pages)
        for page_, token in zip(pages, ["abc", "def", "ghi", ""]):
            assert page_.raw_page.next_page_token == token


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.GetEkmConnectionRequest,
        dict,
    ],
)
def test_get_ekm_connection_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # send a request that will satisfy transcoding
    request_init = {"name": "projects/sample1/locations/sample2/ekmConnections/sample3"}
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.EkmConnection(
            name="name_value",
            etag="etag_value",
            key_management_mode=ekm_service.EkmConnection.KeyManagementMode.MANUAL,
            crypto_space_path="crypto_space_path_value",
        )

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.EkmConnection.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value
        response = client.get_ekm_connection(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConnection)
    assert response.name == "name_value"
    assert response.etag == "etag_value"
    assert (
        response.key_management_mode
        == ekm_service.EkmConnection.KeyManagementMode.MANUAL
    )
    assert response.crypto_space_path == "crypto_space_path_value"


def test_get_ekm_connection_rest_required_fields(
    request_type=ekm_service.GetEkmConnectionRequest,
):
    transport_class = transports.EkmServiceRestTransport

    request_init = {}
    request_init["name"] = ""
    request = request_type(**request_init)
    pb_request = request_type.pb(request)
    jsonified_request = json.loads(
        json_format.MessageToJson(
            pb_request,
            including_default_value_fields=False,
            use_integers_for_enums=False,
        )
    )

    # verify fields with default values are dropped

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).get_ekm_connection._get_unset_required_fields(jsonified_request)
    jsonified_request.update(unset_fields)

    # verify required fields with default values are now present

    jsonified_request["name"] = "name_value"

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).get_ekm_connection._get_unset_required_fields(jsonified_request)
    jsonified_request.update(unset_fields)

    # verify required fields with non-default values are left alone
    assert "name" in jsonified_request
    assert jsonified_request["name"] == "name_value"

    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request = request_type(**request_init)

    # Designate an appropriate value for the returned response.
    return_value = ekm_service.EkmConnection()
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(Session, "request") as req:
        # We need to mock transcode() because providing default values
        # for required fields will fail the real version if the http_options
        # expect actual values for those fields.
        with mock.patch.object(path_template, "transcode") as transcode:
            # A uri without fields and an empty body will force all the
            # request fields to show up in the query_params.
            pb_request = request_type.pb(request)
            transcode_result = {
                "uri": "v1/sample_method",
                "method": "get",
                "query_params": pb_request,
            }
            transcode.return_value = transcode_result

            response_value = Response()
            response_value.status_code = 200

            # Convert return value to protobuf type
            return_value = ekm_service.EkmConnection.pb(return_value)
            json_return_value = json_format.MessageToJson(return_value)

            response_value._content = json_return_value.encode("UTF-8")
            req.return_value = response_value

            response = client.get_ekm_connection(request)

            expected_params = [("$alt", "json;enum-encoding=int")]
            actual_params = req.call_args.kwargs["params"]
            assert expected_params == actual_params


def test_get_ekm_connection_rest_unset_required_fields():
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain
    )

    unset_fields = transport.get_ekm_connection._get_unset_required_fields({})
    assert set(unset_fields) == (set(()) & set(("name",)))


@pytest.mark.parametrize("null_interceptor", [True, False])
def test_get_ekm_connection_rest_interceptors(null_interceptor):
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        interceptor=None
        if null_interceptor
        else transports.EkmServiceRestInterceptor(),
    )
    client = EkmServiceClient(transport=transport)
    with mock.patch.object(
        type(client.transport._session), "request"
    ) as req, mock.patch.object(
        path_template, "transcode"
    ) as transcode, mock.patch.object(
        transports.EkmServiceRestInterceptor, "post_get_ekm_connection"
    ) as post, mock.patch.object(
        transports.EkmServiceRestInterceptor, "pre_get_ekm_connection"
    ) as pre:
        pre.assert_not_called()
        post.assert_not_called()
        pb_message = ekm_service.GetEkmConnectionRequest.pb(
            ekm_service.GetEkmConnectionRequest()
        )
        transcode.return_value = {
            "method": "post",
            "uri": "my_uri",
            "body": pb_message,
            "query_params": pb_message,
        }

        req.return_value = Response()
        req.return_value.status_code = 200
        req.return_value.request = PreparedRequest()
        req.return_value._content = ekm_service.EkmConnection.to_json(
            ekm_service.EkmConnection()
        )

        request = ekm_service.GetEkmConnectionRequest()
        metadata = [
            ("key", "val"),
            ("cephalopod", "squid"),
        ]
        pre.return_value = request, metadata
        post.return_value = ekm_service.EkmConnection()

        client.get_ekm_connection(
            request,
            metadata=[
                ("key", "val"),
                ("cephalopod", "squid"),
            ],
        )

        pre.assert_called_once()
        post.assert_called_once()


def test_get_ekm_connection_rest_bad_request(
    transport: str = "rest", request_type=ekm_service.GetEkmConnectionRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # send a request that will satisfy transcoding
    request_init = {"name": "projects/sample1/locations/sample2/ekmConnections/sample3"}
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.get_ekm_connection(request)


def test_get_ekm_connection_rest_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.EkmConnection()

        # get arguments that satisfy an http rule for this method
        sample_request = {
            "name": "projects/sample1/locations/sample2/ekmConnections/sample3"
        }

        # get truthy value for each flattened field
        mock_args = dict(
            name="name_value",
        )
        mock_args.update(sample_request)

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.EkmConnection.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)
        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        client.get_ekm_connection(**mock_args)

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(req.mock_calls) == 1
        _, args, _ = req.mock_calls[0]
        assert path_template.validate(
            "%s/v1/{name=projects/*/locations/*/ekmConnections/*}"
            % client.transport._host,
            args[1],
        )


def test_get_ekm_connection_rest_flattened_error(transport: str = "rest"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.get_ekm_connection(
            ekm_service.GetEkmConnectionRequest(),
            name="name_value",
        )


def test_get_ekm_connection_rest_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(), transport="rest"
    )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.CreateEkmConnectionRequest,
        dict,
    ],
)
def test_create_ekm_connection_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # send a request that will satisfy transcoding
    request_init = {"parent": "projects/sample1/locations/sample2"}
    request_init["ekm_connection"] = {
        "name": "name_value",
        "create_time": {"seconds": 751, "nanos": 543},
        "service_resolvers": [
            {
                "service_directory_service": "service_directory_service_value",
                "endpoint_filter": "endpoint_filter_value",
                "hostname": "hostname_value",
                "server_certificates": [
                    {
                        "raw_der": b"raw_der_blob",
                        "parsed": True,
                        "issuer": "issuer_value",
                        "subject": "subject_value",
                        "subject_alternative_dns_names": [
                            "subject_alternative_dns_names_value1",
                            "subject_alternative_dns_names_value2",
                        ],
                        "not_before_time": {},
                        "not_after_time": {},
                        "serial_number": "serial_number_value",
                        "sha256_fingerprint": "sha256_fingerprint_value",
                    }
                ],
            }
        ],
        "etag": "etag_value",
        "key_management_mode": 1,
        "crypto_space_path": "crypto_space_path_value",
    }
    # The version of a generated dependency at test runtime may differ from the version used during generation.
    # Delete any fields which are not present in the current runtime dependency
    # See https://github.com/googleapis/gapic-generator-python/issues/1748

    # Determine if the message type is proto-plus or protobuf
    test_field = ekm_service.CreateEkmConnectionRequest.meta.fields["ekm_connection"]

    def get_message_fields(field):
        # Given a field which is a message (composite type), return a list with
        # all the fields of the message.
        # If the field is not a composite type, return an empty list.
        message_fields = []

        if hasattr(field, "message") and field.message:
            is_field_type_proto_plus_type = not hasattr(field.message, "DESCRIPTOR")

            if is_field_type_proto_plus_type:
                message_fields = field.message.meta.fields.values()
            # Add `# pragma: NO COVER` because there may not be any `*_pb2` field types
            else:  # pragma: NO COVER
                message_fields = field.message.DESCRIPTOR.fields
        return message_fields

    runtime_nested_fields = [
        (field.name, nested_field.name)
        for field in get_message_fields(test_field)
        for nested_field in get_message_fields(field)
    ]

    subfields_not_in_runtime = []

    # For each item in the sample request, create a list of sub fields which are not present at runtime
    # Add `# pragma: NO COVER` because this test code will not run if all subfields are present at runtime
    for field, value in request_init["ekm_connection"].items():  # pragma: NO COVER
        result = None
        is_repeated = False
        # For repeated fields
        if isinstance(value, list) and len(value):
            is_repeated = True
            result = value[0]
        # For fields where the type is another message
        if isinstance(value, dict):
            result = value

        if result and hasattr(result, "keys"):
            for subfield in result.keys():
                if (field, subfield) not in runtime_nested_fields:
                    subfields_not_in_runtime.append(
                        {
                            "field": field,
                            "subfield": subfield,
                            "is_repeated": is_repeated,
                        }
                    )

    # Remove fields from the sample request which are not present in the runtime version of the dependency
    # Add `# pragma: NO COVER` because this test code will not run if all subfields are present at runtime
    for subfield_to_delete in subfields_not_in_runtime:  # pragma: NO COVER
        field = subfield_to_delete.get("field")
        field_repeated = subfield_to_delete.get("is_repeated")
        subfield = subfield_to_delete.get("subfield")
        if subfield:
            if field_repeated:
                for i in range(0, len(request_init["ekm_connection"][field])):
                    del request_init["ekm_connection"][field][i][subfield]
            else:
                del request_init["ekm_connection"][field][subfield]
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.EkmConnection(
            name="name_value",
            etag="etag_value",
            key_management_mode=ekm_service.EkmConnection.KeyManagementMode.MANUAL,
            crypto_space_path="crypto_space_path_value",
        )

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.EkmConnection.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value
        response = client.create_ekm_connection(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConnection)
    assert response.name == "name_value"
    assert response.etag == "etag_value"
    assert (
        response.key_management_mode
        == ekm_service.EkmConnection.KeyManagementMode.MANUAL
    )
    assert response.crypto_space_path == "crypto_space_path_value"


def test_create_ekm_connection_rest_required_fields(
    request_type=ekm_service.CreateEkmConnectionRequest,
):
    transport_class = transports.EkmServiceRestTransport

    request_init = {}
    request_init["parent"] = ""
    request_init["ekm_connection_id"] = ""
    request = request_type(**request_init)
    pb_request = request_type.pb(request)
    jsonified_request = json.loads(
        json_format.MessageToJson(
            pb_request,
            including_default_value_fields=False,
            use_integers_for_enums=False,
        )
    )

    # verify fields with default values are dropped
    assert "ekmConnectionId" not in jsonified_request

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).create_ekm_connection._get_unset_required_fields(jsonified_request)
    jsonified_request.update(unset_fields)

    # verify required fields with default values are now present
    assert "ekmConnectionId" in jsonified_request
    assert jsonified_request["ekmConnectionId"] == request_init["ekm_connection_id"]

    jsonified_request["parent"] = "parent_value"
    jsonified_request["ekmConnectionId"] = "ekm_connection_id_value"

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).create_ekm_connection._get_unset_required_fields(jsonified_request)
    # Check that path parameters and body parameters are not mixing in.
    assert not set(unset_fields) - set(("ekm_connection_id",))
    jsonified_request.update(unset_fields)

    # verify required fields with non-default values are left alone
    assert "parent" in jsonified_request
    assert jsonified_request["parent"] == "parent_value"
    assert "ekmConnectionId" in jsonified_request
    assert jsonified_request["ekmConnectionId"] == "ekm_connection_id_value"

    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request = request_type(**request_init)

    # Designate an appropriate value for the returned response.
    return_value = ekm_service.EkmConnection()
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(Session, "request") as req:
        # We need to mock transcode() because providing default values
        # for required fields will fail the real version if the http_options
        # expect actual values for those fields.
        with mock.patch.object(path_template, "transcode") as transcode:
            # A uri without fields and an empty body will force all the
            # request fields to show up in the query_params.
            pb_request = request_type.pb(request)
            transcode_result = {
                "uri": "v1/sample_method",
                "method": "post",
                "query_params": pb_request,
            }
            transcode_result["body"] = pb_request
            transcode.return_value = transcode_result

            response_value = Response()
            response_value.status_code = 200

            # Convert return value to protobuf type
            return_value = ekm_service.EkmConnection.pb(return_value)
            json_return_value = json_format.MessageToJson(return_value)

            response_value._content = json_return_value.encode("UTF-8")
            req.return_value = response_value

            response = client.create_ekm_connection(request)

            expected_params = [
                (
                    "ekmConnectionId",
                    "",
                ),
                ("$alt", "json;enum-encoding=int"),
            ]
            actual_params = req.call_args.kwargs["params"]
            assert expected_params == actual_params


def test_create_ekm_connection_rest_unset_required_fields():
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain
    )

    unset_fields = transport.create_ekm_connection._get_unset_required_fields({})
    assert set(unset_fields) == (
        set(("ekmConnectionId",))
        & set(
            (
                "parent",
                "ekmConnectionId",
                "ekmConnection",
            )
        )
    )


@pytest.mark.parametrize("null_interceptor", [True, False])
def test_create_ekm_connection_rest_interceptors(null_interceptor):
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        interceptor=None
        if null_interceptor
        else transports.EkmServiceRestInterceptor(),
    )
    client = EkmServiceClient(transport=transport)
    with mock.patch.object(
        type(client.transport._session), "request"
    ) as req, mock.patch.object(
        path_template, "transcode"
    ) as transcode, mock.patch.object(
        transports.EkmServiceRestInterceptor, "post_create_ekm_connection"
    ) as post, mock.patch.object(
        transports.EkmServiceRestInterceptor, "pre_create_ekm_connection"
    ) as pre:
        pre.assert_not_called()
        post.assert_not_called()
        pb_message = ekm_service.CreateEkmConnectionRequest.pb(
            ekm_service.CreateEkmConnectionRequest()
        )
        transcode.return_value = {
            "method": "post",
            "uri": "my_uri",
            "body": pb_message,
            "query_params": pb_message,
        }

        req.return_value = Response()
        req.return_value.status_code = 200
        req.return_value.request = PreparedRequest()
        req.return_value._content = ekm_service.EkmConnection.to_json(
            ekm_service.EkmConnection()
        )

        request = ekm_service.CreateEkmConnectionRequest()
        metadata = [
            ("key", "val"),
            ("cephalopod", "squid"),
        ]
        pre.return_value = request, metadata
        post.return_value = ekm_service.EkmConnection()

        client.create_ekm_connection(
            request,
            metadata=[
                ("key", "val"),
                ("cephalopod", "squid"),
            ],
        )

        pre.assert_called_once()
        post.assert_called_once()


def test_create_ekm_connection_rest_bad_request(
    transport: str = "rest", request_type=ekm_service.CreateEkmConnectionRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # send a request that will satisfy transcoding
    request_init = {"parent": "projects/sample1/locations/sample2"}
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.create_ekm_connection(request)


def test_create_ekm_connection_rest_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.EkmConnection()

        # get arguments that satisfy an http rule for this method
        sample_request = {"parent": "projects/sample1/locations/sample2"}

        # get truthy value for each flattened field
        mock_args = dict(
            parent="parent_value",
            ekm_connection_id="ekm_connection_id_value",
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
        )
        mock_args.update(sample_request)

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.EkmConnection.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)
        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        client.create_ekm_connection(**mock_args)

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(req.mock_calls) == 1
        _, args, _ = req.mock_calls[0]
        assert path_template.validate(
            "%s/v1/{parent=projects/*/locations/*}/ekmConnections"
            % client.transport._host,
            args[1],
        )


def test_create_ekm_connection_rest_flattened_error(transport: str = "rest"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.create_ekm_connection(
            ekm_service.CreateEkmConnectionRequest(),
            parent="parent_value",
            ekm_connection_id="ekm_connection_id_value",
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
        )


def test_create_ekm_connection_rest_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(), transport="rest"
    )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.UpdateEkmConnectionRequest,
        dict,
    ],
)
def test_update_ekm_connection_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # send a request that will satisfy transcoding
    request_init = {
        "ekm_connection": {
            "name": "projects/sample1/locations/sample2/ekmConnections/sample3"
        }
    }
    request_init["ekm_connection"] = {
        "name": "projects/sample1/locations/sample2/ekmConnections/sample3",
        "create_time": {"seconds": 751, "nanos": 543},
        "service_resolvers": [
            {
                "service_directory_service": "service_directory_service_value",
                "endpoint_filter": "endpoint_filter_value",
                "hostname": "hostname_value",
                "server_certificates": [
                    {
                        "raw_der": b"raw_der_blob",
                        "parsed": True,
                        "issuer": "issuer_value",
                        "subject": "subject_value",
                        "subject_alternative_dns_names": [
                            "subject_alternative_dns_names_value1",
                            "subject_alternative_dns_names_value2",
                        ],
                        "not_before_time": {},
                        "not_after_time": {},
                        "serial_number": "serial_number_value",
                        "sha256_fingerprint": "sha256_fingerprint_value",
                    }
                ],
            }
        ],
        "etag": "etag_value",
        "key_management_mode": 1,
        "crypto_space_path": "crypto_space_path_value",
    }
    # The version of a generated dependency at test runtime may differ from the version used during generation.
    # Delete any fields which are not present in the current runtime dependency
    # See https://github.com/googleapis/gapic-generator-python/issues/1748

    # Determine if the message type is proto-plus or protobuf
    test_field = ekm_service.UpdateEkmConnectionRequest.meta.fields["ekm_connection"]

    def get_message_fields(field):
        # Given a field which is a message (composite type), return a list with
        # all the fields of the message.
        # If the field is not a composite type, return an empty list.
        message_fields = []

        if hasattr(field, "message") and field.message:
            is_field_type_proto_plus_type = not hasattr(field.message, "DESCRIPTOR")

            if is_field_type_proto_plus_type:
                message_fields = field.message.meta.fields.values()
            # Add `# pragma: NO COVER` because there may not be any `*_pb2` field types
            else:  # pragma: NO COVER
                message_fields = field.message.DESCRIPTOR.fields
        return message_fields

    runtime_nested_fields = [
        (field.name, nested_field.name)
        for field in get_message_fields(test_field)
        for nested_field in get_message_fields(field)
    ]

    subfields_not_in_runtime = []

    # For each item in the sample request, create a list of sub fields which are not present at runtime
    # Add `# pragma: NO COVER` because this test code will not run if all subfields are present at runtime
    for field, value in request_init["ekm_connection"].items():  # pragma: NO COVER
        result = None
        is_repeated = False
        # For repeated fields
        if isinstance(value, list) and len(value):
            is_repeated = True
            result = value[0]
        # For fields where the type is another message
        if isinstance(value, dict):
            result = value

        if result and hasattr(result, "keys"):
            for subfield in result.keys():
                if (field, subfield) not in runtime_nested_fields:
                    subfields_not_in_runtime.append(
                        {
                            "field": field,
                            "subfield": subfield,
                            "is_repeated": is_repeated,
                        }
                    )

    # Remove fields from the sample request which are not present in the runtime version of the dependency
    # Add `# pragma: NO COVER` because this test code will not run if all subfields are present at runtime
    for subfield_to_delete in subfields_not_in_runtime:  # pragma: NO COVER
        field = subfield_to_delete.get("field")
        field_repeated = subfield_to_delete.get("is_repeated")
        subfield = subfield_to_delete.get("subfield")
        if subfield:
            if field_repeated:
                for i in range(0, len(request_init["ekm_connection"][field])):
                    del request_init["ekm_connection"][field][i][subfield]
            else:
                del request_init["ekm_connection"][field][subfield]
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.EkmConnection(
            name="name_value",
            etag="etag_value",
            key_management_mode=ekm_service.EkmConnection.KeyManagementMode.MANUAL,
            crypto_space_path="crypto_space_path_value",
        )

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.EkmConnection.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value
        response = client.update_ekm_connection(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConnection)
    assert response.name == "name_value"
    assert response.etag == "etag_value"
    assert (
        response.key_management_mode
        == ekm_service.EkmConnection.KeyManagementMode.MANUAL
    )
    assert response.crypto_space_path == "crypto_space_path_value"


def test_update_ekm_connection_rest_required_fields(
    request_type=ekm_service.UpdateEkmConnectionRequest,
):
    transport_class = transports.EkmServiceRestTransport

    request_init = {}
    request = request_type(**request_init)
    pb_request = request_type.pb(request)
    jsonified_request = json.loads(
        json_format.MessageToJson(
            pb_request,
            including_default_value_fields=False,
            use_integers_for_enums=False,
        )
    )

    # verify fields with default values are dropped

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).update_ekm_connection._get_unset_required_fields(jsonified_request)
    jsonified_request.update(unset_fields)

    # verify required fields with default values are now present

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).update_ekm_connection._get_unset_required_fields(jsonified_request)
    # Check that path parameters and body parameters are not mixing in.
    assert not set(unset_fields) - set(("update_mask",))
    jsonified_request.update(unset_fields)

    # verify required fields with non-default values are left alone

    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request = request_type(**request_init)

    # Designate an appropriate value for the returned response.
    return_value = ekm_service.EkmConnection()
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(Session, "request") as req:
        # We need to mock transcode() because providing default values
        # for required fields will fail the real version if the http_options
        # expect actual values for those fields.
        with mock.patch.object(path_template, "transcode") as transcode:
            # A uri without fields and an empty body will force all the
            # request fields to show up in the query_params.
            pb_request = request_type.pb(request)
            transcode_result = {
                "uri": "v1/sample_method",
                "method": "patch",
                "query_params": pb_request,
            }
            transcode_result["body"] = pb_request
            transcode.return_value = transcode_result

            response_value = Response()
            response_value.status_code = 200

            # Convert return value to protobuf type
            return_value = ekm_service.EkmConnection.pb(return_value)
            json_return_value = json_format.MessageToJson(return_value)

            response_value._content = json_return_value.encode("UTF-8")
            req.return_value = response_value

            response = client.update_ekm_connection(request)

            expected_params = [("$alt", "json;enum-encoding=int")]
            actual_params = req.call_args.kwargs["params"]
            assert expected_params == actual_params


def test_update_ekm_connection_rest_unset_required_fields():
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain
    )

    unset_fields = transport.update_ekm_connection._get_unset_required_fields({})
    assert set(unset_fields) == (
        set(("updateMask",))
        & set(
            (
                "ekmConnection",
                "updateMask",
            )
        )
    )


@pytest.mark.parametrize("null_interceptor", [True, False])
def test_update_ekm_connection_rest_interceptors(null_interceptor):
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        interceptor=None
        if null_interceptor
        else transports.EkmServiceRestInterceptor(),
    )
    client = EkmServiceClient(transport=transport)
    with mock.patch.object(
        type(client.transport._session), "request"
    ) as req, mock.patch.object(
        path_template, "transcode"
    ) as transcode, mock.patch.object(
        transports.EkmServiceRestInterceptor, "post_update_ekm_connection"
    ) as post, mock.patch.object(
        transports.EkmServiceRestInterceptor, "pre_update_ekm_connection"
    ) as pre:
        pre.assert_not_called()
        post.assert_not_called()
        pb_message = ekm_service.UpdateEkmConnectionRequest.pb(
            ekm_service.UpdateEkmConnectionRequest()
        )
        transcode.return_value = {
            "method": "post",
            "uri": "my_uri",
            "body": pb_message,
            "query_params": pb_message,
        }

        req.return_value = Response()
        req.return_value.status_code = 200
        req.return_value.request = PreparedRequest()
        req.return_value._content = ekm_service.EkmConnection.to_json(
            ekm_service.EkmConnection()
        )

        request = ekm_service.UpdateEkmConnectionRequest()
        metadata = [
            ("key", "val"),
            ("cephalopod", "squid"),
        ]
        pre.return_value = request, metadata
        post.return_value = ekm_service.EkmConnection()

        client.update_ekm_connection(
            request,
            metadata=[
                ("key", "val"),
                ("cephalopod", "squid"),
            ],
        )

        pre.assert_called_once()
        post.assert_called_once()


def test_update_ekm_connection_rest_bad_request(
    transport: str = "rest", request_type=ekm_service.UpdateEkmConnectionRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # send a request that will satisfy transcoding
    request_init = {
        "ekm_connection": {
            "name": "projects/sample1/locations/sample2/ekmConnections/sample3"
        }
    }
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.update_ekm_connection(request)


def test_update_ekm_connection_rest_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.EkmConnection()

        # get arguments that satisfy an http rule for this method
        sample_request = {
            "ekm_connection": {
                "name": "projects/sample1/locations/sample2/ekmConnections/sample3"
            }
        }

        # get truthy value for each flattened field
        mock_args = dict(
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )
        mock_args.update(sample_request)

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.EkmConnection.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)
        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        client.update_ekm_connection(**mock_args)

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(req.mock_calls) == 1
        _, args, _ = req.mock_calls[0]
        assert path_template.validate(
            "%s/v1/{ekm_connection.name=projects/*/locations/*/ekmConnections/*}"
            % client.transport._host,
            args[1],
        )


def test_update_ekm_connection_rest_flattened_error(transport: str = "rest"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.update_ekm_connection(
            ekm_service.UpdateEkmConnectionRequest(),
            ekm_connection=ekm_service.EkmConnection(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )


def test_update_ekm_connection_rest_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(), transport="rest"
    )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.GetEkmConfigRequest,
        dict,
    ],
)
def test_get_ekm_config_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # send a request that will satisfy transcoding
    request_init = {"name": "projects/sample1/locations/sample2/ekmConfig"}
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.EkmConfig(
            name="name_value",
            default_ekm_connection="default_ekm_connection_value",
        )

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.EkmConfig.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value
        response = client.get_ekm_config(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConfig)
    assert response.name == "name_value"
    assert response.default_ekm_connection == "default_ekm_connection_value"


def test_get_ekm_config_rest_required_fields(
    request_type=ekm_service.GetEkmConfigRequest,
):
    transport_class = transports.EkmServiceRestTransport

    request_init = {}
    request_init["name"] = ""
    request = request_type(**request_init)
    pb_request = request_type.pb(request)
    jsonified_request = json.loads(
        json_format.MessageToJson(
            pb_request,
            including_default_value_fields=False,
            use_integers_for_enums=False,
        )
    )

    # verify fields with default values are dropped

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).get_ekm_config._get_unset_required_fields(jsonified_request)
    jsonified_request.update(unset_fields)

    # verify required fields with default values are now present

    jsonified_request["name"] = "name_value"

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).get_ekm_config._get_unset_required_fields(jsonified_request)
    jsonified_request.update(unset_fields)

    # verify required fields with non-default values are left alone
    assert "name" in jsonified_request
    assert jsonified_request["name"] == "name_value"

    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request = request_type(**request_init)

    # Designate an appropriate value for the returned response.
    return_value = ekm_service.EkmConfig()
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(Session, "request") as req:
        # We need to mock transcode() because providing default values
        # for required fields will fail the real version if the http_options
        # expect actual values for those fields.
        with mock.patch.object(path_template, "transcode") as transcode:
            # A uri without fields and an empty body will force all the
            # request fields to show up in the query_params.
            pb_request = request_type.pb(request)
            transcode_result = {
                "uri": "v1/sample_method",
                "method": "get",
                "query_params": pb_request,
            }
            transcode.return_value = transcode_result

            response_value = Response()
            response_value.status_code = 200

            # Convert return value to protobuf type
            return_value = ekm_service.EkmConfig.pb(return_value)
            json_return_value = json_format.MessageToJson(return_value)

            response_value._content = json_return_value.encode("UTF-8")
            req.return_value = response_value

            response = client.get_ekm_config(request)

            expected_params = [("$alt", "json;enum-encoding=int")]
            actual_params = req.call_args.kwargs["params"]
            assert expected_params == actual_params


def test_get_ekm_config_rest_unset_required_fields():
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain
    )

    unset_fields = transport.get_ekm_config._get_unset_required_fields({})
    assert set(unset_fields) == (set(()) & set(("name",)))


@pytest.mark.parametrize("null_interceptor", [True, False])
def test_get_ekm_config_rest_interceptors(null_interceptor):
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        interceptor=None
        if null_interceptor
        else transports.EkmServiceRestInterceptor(),
    )
    client = EkmServiceClient(transport=transport)
    with mock.patch.object(
        type(client.transport._session), "request"
    ) as req, mock.patch.object(
        path_template, "transcode"
    ) as transcode, mock.patch.object(
        transports.EkmServiceRestInterceptor, "post_get_ekm_config"
    ) as post, mock.patch.object(
        transports.EkmServiceRestInterceptor, "pre_get_ekm_config"
    ) as pre:
        pre.assert_not_called()
        post.assert_not_called()
        pb_message = ekm_service.GetEkmConfigRequest.pb(
            ekm_service.GetEkmConfigRequest()
        )
        transcode.return_value = {
            "method": "post",
            "uri": "my_uri",
            "body": pb_message,
            "query_params": pb_message,
        }

        req.return_value = Response()
        req.return_value.status_code = 200
        req.return_value.request = PreparedRequest()
        req.return_value._content = ekm_service.EkmConfig.to_json(
            ekm_service.EkmConfig()
        )

        request = ekm_service.GetEkmConfigRequest()
        metadata = [
            ("key", "val"),
            ("cephalopod", "squid"),
        ]
        pre.return_value = request, metadata
        post.return_value = ekm_service.EkmConfig()

        client.get_ekm_config(
            request,
            metadata=[
                ("key", "val"),
                ("cephalopod", "squid"),
            ],
        )

        pre.assert_called_once()
        post.assert_called_once()


def test_get_ekm_config_rest_bad_request(
    transport: str = "rest", request_type=ekm_service.GetEkmConfigRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # send a request that will satisfy transcoding
    request_init = {"name": "projects/sample1/locations/sample2/ekmConfig"}
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.get_ekm_config(request)


def test_get_ekm_config_rest_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.EkmConfig()

        # get arguments that satisfy an http rule for this method
        sample_request = {"name": "projects/sample1/locations/sample2/ekmConfig"}

        # get truthy value for each flattened field
        mock_args = dict(
            name="name_value",
        )
        mock_args.update(sample_request)

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.EkmConfig.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)
        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        client.get_ekm_config(**mock_args)

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(req.mock_calls) == 1
        _, args, _ = req.mock_calls[0]
        assert path_template.validate(
            "%s/v1/{name=projects/*/locations/*/ekmConfig}" % client.transport._host,
            args[1],
        )


def test_get_ekm_config_rest_flattened_error(transport: str = "rest"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.get_ekm_config(
            ekm_service.GetEkmConfigRequest(),
            name="name_value",
        )


def test_get_ekm_config_rest_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(), transport="rest"
    )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.UpdateEkmConfigRequest,
        dict,
    ],
)
def test_update_ekm_config_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # send a request that will satisfy transcoding
    request_init = {
        "ekm_config": {"name": "projects/sample1/locations/sample2/ekmConfig"}
    }
    request_init["ekm_config"] = {
        "name": "projects/sample1/locations/sample2/ekmConfig",
        "default_ekm_connection": "default_ekm_connection_value",
    }
    # The version of a generated dependency at test runtime may differ from the version used during generation.
    # Delete any fields which are not present in the current runtime dependency
    # See https://github.com/googleapis/gapic-generator-python/issues/1748

    # Determine if the message type is proto-plus or protobuf
    test_field = ekm_service.UpdateEkmConfigRequest.meta.fields["ekm_config"]

    def get_message_fields(field):
        # Given a field which is a message (composite type), return a list with
        # all the fields of the message.
        # If the field is not a composite type, return an empty list.
        message_fields = []

        if hasattr(field, "message") and field.message:
            is_field_type_proto_plus_type = not hasattr(field.message, "DESCRIPTOR")

            if is_field_type_proto_plus_type:
                message_fields = field.message.meta.fields.values()
            # Add `# pragma: NO COVER` because there may not be any `*_pb2` field types
            else:  # pragma: NO COVER
                message_fields = field.message.DESCRIPTOR.fields
        return message_fields

    runtime_nested_fields = [
        (field.name, nested_field.name)
        for field in get_message_fields(test_field)
        for nested_field in get_message_fields(field)
    ]

    subfields_not_in_runtime = []

    # For each item in the sample request, create a list of sub fields which are not present at runtime
    # Add `# pragma: NO COVER` because this test code will not run if all subfields are present at runtime
    for field, value in request_init["ekm_config"].items():  # pragma: NO COVER
        result = None
        is_repeated = False
        # For repeated fields
        if isinstance(value, list) and len(value):
            is_repeated = True
            result = value[0]
        # For fields where the type is another message
        if isinstance(value, dict):
            result = value

        if result and hasattr(result, "keys"):
            for subfield in result.keys():
                if (field, subfield) not in runtime_nested_fields:
                    subfields_not_in_runtime.append(
                        {
                            "field": field,
                            "subfield": subfield,
                            "is_repeated": is_repeated,
                        }
                    )

    # Remove fields from the sample request which are not present in the runtime version of the dependency
    # Add `# pragma: NO COVER` because this test code will not run if all subfields are present at runtime
    for subfield_to_delete in subfields_not_in_runtime:  # pragma: NO COVER
        field = subfield_to_delete.get("field")
        field_repeated = subfield_to_delete.get("is_repeated")
        subfield = subfield_to_delete.get("subfield")
        if subfield:
            if field_repeated:
                for i in range(0, len(request_init["ekm_config"][field])):
                    del request_init["ekm_config"][field][i][subfield]
            else:
                del request_init["ekm_config"][field][subfield]
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.EkmConfig(
            name="name_value",
            default_ekm_connection="default_ekm_connection_value",
        )

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.EkmConfig.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value
        response = client.update_ekm_config(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.EkmConfig)
    assert response.name == "name_value"
    assert response.default_ekm_connection == "default_ekm_connection_value"


def test_update_ekm_config_rest_required_fields(
    request_type=ekm_service.UpdateEkmConfigRequest,
):
    transport_class = transports.EkmServiceRestTransport

    request_init = {}
    request = request_type(**request_init)
    pb_request = request_type.pb(request)
    jsonified_request = json.loads(
        json_format.MessageToJson(
            pb_request,
            including_default_value_fields=False,
            use_integers_for_enums=False,
        )
    )

    # verify fields with default values are dropped

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).update_ekm_config._get_unset_required_fields(jsonified_request)
    jsonified_request.update(unset_fields)

    # verify required fields with default values are now present

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).update_ekm_config._get_unset_required_fields(jsonified_request)
    # Check that path parameters and body parameters are not mixing in.
    assert not set(unset_fields) - set(("update_mask",))
    jsonified_request.update(unset_fields)

    # verify required fields with non-default values are left alone

    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request = request_type(**request_init)

    # Designate an appropriate value for the returned response.
    return_value = ekm_service.EkmConfig()
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(Session, "request") as req:
        # We need to mock transcode() because providing default values
        # for required fields will fail the real version if the http_options
        # expect actual values for those fields.
        with mock.patch.object(path_template, "transcode") as transcode:
            # A uri without fields and an empty body will force all the
            # request fields to show up in the query_params.
            pb_request = request_type.pb(request)
            transcode_result = {
                "uri": "v1/sample_method",
                "method": "patch",
                "query_params": pb_request,
            }
            transcode_result["body"] = pb_request
            transcode.return_value = transcode_result

            response_value = Response()
            response_value.status_code = 200

            # Convert return value to protobuf type
            return_value = ekm_service.EkmConfig.pb(return_value)
            json_return_value = json_format.MessageToJson(return_value)

            response_value._content = json_return_value.encode("UTF-8")
            req.return_value = response_value

            response = client.update_ekm_config(request)

            expected_params = [("$alt", "json;enum-encoding=int")]
            actual_params = req.call_args.kwargs["params"]
            assert expected_params == actual_params


def test_update_ekm_config_rest_unset_required_fields():
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain
    )

    unset_fields = transport.update_ekm_config._get_unset_required_fields({})
    assert set(unset_fields) == (
        set(("updateMask",))
        & set(
            (
                "ekmConfig",
                "updateMask",
            )
        )
    )


@pytest.mark.parametrize("null_interceptor", [True, False])
def test_update_ekm_config_rest_interceptors(null_interceptor):
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        interceptor=None
        if null_interceptor
        else transports.EkmServiceRestInterceptor(),
    )
    client = EkmServiceClient(transport=transport)
    with mock.patch.object(
        type(client.transport._session), "request"
    ) as req, mock.patch.object(
        path_template, "transcode"
    ) as transcode, mock.patch.object(
        transports.EkmServiceRestInterceptor, "post_update_ekm_config"
    ) as post, mock.patch.object(
        transports.EkmServiceRestInterceptor, "pre_update_ekm_config"
    ) as pre:
        pre.assert_not_called()
        post.assert_not_called()
        pb_message = ekm_service.UpdateEkmConfigRequest.pb(
            ekm_service.UpdateEkmConfigRequest()
        )
        transcode.return_value = {
            "method": "post",
            "uri": "my_uri",
            "body": pb_message,
            "query_params": pb_message,
        }

        req.return_value = Response()
        req.return_value.status_code = 200
        req.return_value.request = PreparedRequest()
        req.return_value._content = ekm_service.EkmConfig.to_json(
            ekm_service.EkmConfig()
        )

        request = ekm_service.UpdateEkmConfigRequest()
        metadata = [
            ("key", "val"),
            ("cephalopod", "squid"),
        ]
        pre.return_value = request, metadata
        post.return_value = ekm_service.EkmConfig()

        client.update_ekm_config(
            request,
            metadata=[
                ("key", "val"),
                ("cephalopod", "squid"),
            ],
        )

        pre.assert_called_once()
        post.assert_called_once()


def test_update_ekm_config_rest_bad_request(
    transport: str = "rest", request_type=ekm_service.UpdateEkmConfigRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # send a request that will satisfy transcoding
    request_init = {
        "ekm_config": {"name": "projects/sample1/locations/sample2/ekmConfig"}
    }
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.update_ekm_config(request)


def test_update_ekm_config_rest_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.EkmConfig()

        # get arguments that satisfy an http rule for this method
        sample_request = {
            "ekm_config": {"name": "projects/sample1/locations/sample2/ekmConfig"}
        }

        # get truthy value for each flattened field
        mock_args = dict(
            ekm_config=ekm_service.EkmConfig(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )
        mock_args.update(sample_request)

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.EkmConfig.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)
        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        client.update_ekm_config(**mock_args)

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(req.mock_calls) == 1
        _, args, _ = req.mock_calls[0]
        assert path_template.validate(
            "%s/v1/{ekm_config.name=projects/*/locations/*/ekmConfig}"
            % client.transport._host,
            args[1],
        )


def test_update_ekm_config_rest_flattened_error(transport: str = "rest"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.update_ekm_config(
            ekm_service.UpdateEkmConfigRequest(),
            ekm_config=ekm_service.EkmConfig(name="name_value"),
            update_mask=field_mask_pb2.FieldMask(paths=["paths_value"]),
        )


def test_update_ekm_config_rest_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(), transport="rest"
    )


@pytest.mark.parametrize(
    "request_type",
    [
        ekm_service.VerifyConnectivityRequest,
        dict,
    ],
)
def test_verify_connectivity_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # send a request that will satisfy transcoding
    request_init = {"name": "projects/sample1/locations/sample2/ekmConnections/sample3"}
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.VerifyConnectivityResponse()

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.VerifyConnectivityResponse.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value
        response = client.verify_connectivity(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, ekm_service.VerifyConnectivityResponse)


def test_verify_connectivity_rest_required_fields(
    request_type=ekm_service.VerifyConnectivityRequest,
):
    transport_class = transports.EkmServiceRestTransport

    request_init = {}
    request_init["name"] = ""
    request = request_type(**request_init)
    pb_request = request_type.pb(request)
    jsonified_request = json.loads(
        json_format.MessageToJson(
            pb_request,
            including_default_value_fields=False,
            use_integers_for_enums=False,
        )
    )

    # verify fields with default values are dropped

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).verify_connectivity._get_unset_required_fields(jsonified_request)
    jsonified_request.update(unset_fields)

    # verify required fields with default values are now present

    jsonified_request["name"] = "name_value"

    unset_fields = transport_class(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    ).verify_connectivity._get_unset_required_fields(jsonified_request)
    jsonified_request.update(unset_fields)

    # verify required fields with non-default values are left alone
    assert "name" in jsonified_request
    assert jsonified_request["name"] == "name_value"

    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request = request_type(**request_init)

    # Designate an appropriate value for the returned response.
    return_value = ekm_service.VerifyConnectivityResponse()
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(Session, "request") as req:
        # We need to mock transcode() because providing default values
        # for required fields will fail the real version if the http_options
        # expect actual values for those fields.
        with mock.patch.object(path_template, "transcode") as transcode:
            # A uri without fields and an empty body will force all the
            # request fields to show up in the query_params.
            pb_request = request_type.pb(request)
            transcode_result = {
                "uri": "v1/sample_method",
                "method": "get",
                "query_params": pb_request,
            }
            transcode.return_value = transcode_result

            response_value = Response()
            response_value.status_code = 200

            # Convert return value to protobuf type
            return_value = ekm_service.VerifyConnectivityResponse.pb(return_value)
            json_return_value = json_format.MessageToJson(return_value)

            response_value._content = json_return_value.encode("UTF-8")
            req.return_value = response_value

            response = client.verify_connectivity(request)

            expected_params = [("$alt", "json;enum-encoding=int")]
            actual_params = req.call_args.kwargs["params"]
            assert expected_params == actual_params


def test_verify_connectivity_rest_unset_required_fields():
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain
    )

    unset_fields = transport.verify_connectivity._get_unset_required_fields({})
    assert set(unset_fields) == (set(()) & set(("name",)))


@pytest.mark.parametrize("null_interceptor", [True, False])
def test_verify_connectivity_rest_interceptors(null_interceptor):
    transport = transports.EkmServiceRestTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        interceptor=None
        if null_interceptor
        else transports.EkmServiceRestInterceptor(),
    )
    client = EkmServiceClient(transport=transport)
    with mock.patch.object(
        type(client.transport._session), "request"
    ) as req, mock.patch.object(
        path_template, "transcode"
    ) as transcode, mock.patch.object(
        transports.EkmServiceRestInterceptor, "post_verify_connectivity"
    ) as post, mock.patch.object(
        transports.EkmServiceRestInterceptor, "pre_verify_connectivity"
    ) as pre:
        pre.assert_not_called()
        post.assert_not_called()
        pb_message = ekm_service.VerifyConnectivityRequest.pb(
            ekm_service.VerifyConnectivityRequest()
        )
        transcode.return_value = {
            "method": "post",
            "uri": "my_uri",
            "body": pb_message,
            "query_params": pb_message,
        }

        req.return_value = Response()
        req.return_value.status_code = 200
        req.return_value.request = PreparedRequest()
        req.return_value._content = ekm_service.VerifyConnectivityResponse.to_json(
            ekm_service.VerifyConnectivityResponse()
        )

        request = ekm_service.VerifyConnectivityRequest()
        metadata = [
            ("key", "val"),
            ("cephalopod", "squid"),
        ]
        pre.return_value = request, metadata
        post.return_value = ekm_service.VerifyConnectivityResponse()

        client.verify_connectivity(
            request,
            metadata=[
                ("key", "val"),
                ("cephalopod", "squid"),
            ],
        )

        pre.assert_called_once()
        post.assert_called_once()


def test_verify_connectivity_rest_bad_request(
    transport: str = "rest", request_type=ekm_service.VerifyConnectivityRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # send a request that will satisfy transcoding
    request_init = {"name": "projects/sample1/locations/sample2/ekmConnections/sample3"}
    request = request_type(**request_init)

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.verify_connectivity(request)


def test_verify_connectivity_rest_flattened():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )

    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = ekm_service.VerifyConnectivityResponse()

        # get arguments that satisfy an http rule for this method
        sample_request = {
            "name": "projects/sample1/locations/sample2/ekmConnections/sample3"
        }

        # get truthy value for each flattened field
        mock_args = dict(
            name="name_value",
        )
        mock_args.update(sample_request)

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        # Convert return value to protobuf type
        return_value = ekm_service.VerifyConnectivityResponse.pb(return_value)
        json_return_value = json_format.MessageToJson(return_value)
        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        client.verify_connectivity(**mock_args)

        # Establish that the underlying call was made with the expected
        # request object values.
        assert len(req.mock_calls) == 1
        _, args, _ = req.mock_calls[0]
        assert path_template.validate(
            "%s/v1/{name=projects/*/locations/*/ekmConnections/*}:verifyConnectivity"
            % client.transport._host,
            args[1],
        )


def test_verify_connectivity_rest_flattened_error(transport: str = "rest"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Attempting to call a method with both a request object and flattened
    # fields is an error.
    with pytest.raises(ValueError):
        client.verify_connectivity(
            ekm_service.VerifyConnectivityRequest(),
            name="name_value",
        )


def test_verify_connectivity_rest_error():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(), transport="rest"
    )


def test_credentials_transport_error():
    # It is an error to provide credentials and a transport instance.
    transport = transports.EkmServiceGrpcTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    with pytest.raises(ValueError):
        client = EkmServiceClient(
            credentials=_AnonymousCredentialsWithUniverseDomain(),
            transport=transport,
        )

    # It is an error to provide a credentials file and a transport instance.
    transport = transports.EkmServiceGrpcTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    with pytest.raises(ValueError):
        client = EkmServiceClient(
            client_options={"credentials_file": "credentials.json"},
            transport=transport,
        )

    # It is an error to provide an api_key and a transport instance.
    transport = transports.EkmServiceGrpcTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    options = client_options.ClientOptions()
    options.api_key = "api_key"
    with pytest.raises(ValueError):
        client = EkmServiceClient(
            client_options=options,
            transport=transport,
        )

    # It is an error to provide an api_key and a credential.
    options = client_options.ClientOptions()
    options.api_key = "api_key"
    with pytest.raises(ValueError):
        client = EkmServiceClient(
            client_options=options,
            credentials=_AnonymousCredentialsWithUniverseDomain(),
        )

    # It is an error to provide scopes and a transport instance.
    transport = transports.EkmServiceGrpcTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    with pytest.raises(ValueError):
        client = EkmServiceClient(
            client_options={"scopes": ["1", "2"]},
            transport=transport,
        )


def test_transport_instance():
    # A client may be instantiated with a custom transport instance.
    transport = transports.EkmServiceGrpcTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    client = EkmServiceClient(transport=transport)
    assert client.transport is transport


def test_transport_get_channel():
    # A client may be instantiated with a custom transport instance.
    transport = transports.EkmServiceGrpcTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    channel = transport.grpc_channel
    assert channel

    transport = transports.EkmServiceGrpcAsyncIOTransport(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    channel = transport.grpc_channel
    assert channel


@pytest.mark.parametrize(
    "transport_class",
    [
        transports.EkmServiceGrpcTransport,
        transports.EkmServiceGrpcAsyncIOTransport,
        transports.EkmServiceRestTransport,
    ],
)
def test_transport_adc(transport_class):
    # Test default credentials are used if not provided.
    with mock.patch.object(google.auth, "default") as adc:
        adc.return_value = (_AnonymousCredentialsWithUniverseDomain(), None)
        transport_class()
        adc.assert_called_once()


@pytest.mark.parametrize(
    "transport_name",
    [
        "grpc",
        "rest",
    ],
)
def test_transport_kind(transport_name):
    transport = EkmServiceClient.get_transport_class(transport_name)(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    assert transport.kind == transport_name


def test_transport_grpc_default():
    # A client should use the gRPC transport by default.
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    assert isinstance(
        client.transport,
        transports.EkmServiceGrpcTransport,
    )


def test_ekm_service_base_transport_error():
    # Passing both a credentials object and credentials_file should raise an error
    with pytest.raises(core_exceptions.DuplicateCredentialArgs):
        transport = transports.EkmServiceTransport(
            credentials=_AnonymousCredentialsWithUniverseDomain(),
            credentials_file="credentials.json",
        )


def test_ekm_service_base_transport():
    # Instantiate the base transport.
    with mock.patch(
        "google.cloud.kms_v1.services.ekm_service.transports.EkmServiceTransport.__init__"
    ) as Transport:
        Transport.return_value = None
        transport = transports.EkmServiceTransport(
            credentials=_AnonymousCredentialsWithUniverseDomain(),
        )

    # Every method on the transport should just blindly
    # raise NotImplementedError.
    methods = (
        "list_ekm_connections",
        "get_ekm_connection",
        "create_ekm_connection",
        "update_ekm_connection",
        "get_ekm_config",
        "update_ekm_config",
        "verify_connectivity",
        "set_iam_policy",
        "get_iam_policy",
        "test_iam_permissions",
        "get_location",
        "list_locations",
    )
    for method in methods:
        with pytest.raises(NotImplementedError):
            getattr(transport, method)(request=object())

    with pytest.raises(NotImplementedError):
        transport.close()

    # Catch all for all remaining methods and properties
    remainder = [
        "kind",
    ]
    for r in remainder:
        with pytest.raises(NotImplementedError):
            getattr(transport, r)()


def test_ekm_service_base_transport_with_credentials_file():
    # Instantiate the base transport with a credentials file
    with mock.patch.object(
        google.auth, "load_credentials_from_file", autospec=True
    ) as load_creds, mock.patch(
        "google.cloud.kms_v1.services.ekm_service.transports.EkmServiceTransport._prep_wrapped_messages"
    ) as Transport:
        Transport.return_value = None
        load_creds.return_value = (_AnonymousCredentialsWithUniverseDomain(), None)
        transport = transports.EkmServiceTransport(
            credentials_file="credentials.json",
            quota_project_id="octopus",
        )
        load_creds.assert_called_once_with(
            "credentials.json",
            scopes=None,
            default_scopes=(
                "https://www.googleapis.com/auth/cloud-platform",
                "https://www.googleapis.com/auth/cloudkms",
            ),
            quota_project_id="octopus",
        )


def test_ekm_service_base_transport_with_adc():
    # Test the default credentials are used if credentials and credentials_file are None.
    with mock.patch.object(google.auth, "default", autospec=True) as adc, mock.patch(
        "google.cloud.kms_v1.services.ekm_service.transports.EkmServiceTransport._prep_wrapped_messages"
    ) as Transport:
        Transport.return_value = None
        adc.return_value = (_AnonymousCredentialsWithUniverseDomain(), None)
        transport = transports.EkmServiceTransport()
        adc.assert_called_once()


def test_ekm_service_auth_adc():
    # If no credentials are provided, we should use ADC credentials.
    with mock.patch.object(google.auth, "default", autospec=True) as adc:
        adc.return_value = (_AnonymousCredentialsWithUniverseDomain(), None)
        EkmServiceClient()
        adc.assert_called_once_with(
            scopes=None,
            default_scopes=(
                "https://www.googleapis.com/auth/cloud-platform",
                "https://www.googleapis.com/auth/cloudkms",
            ),
            quota_project_id=None,
        )


@pytest.mark.parametrize(
    "transport_class",
    [
        transports.EkmServiceGrpcTransport,
        transports.EkmServiceGrpcAsyncIOTransport,
    ],
)
def test_ekm_service_transport_auth_adc(transport_class):
    # If credentials and host are not provided, the transport class should use
    # ADC credentials.
    with mock.patch.object(google.auth, "default", autospec=True) as adc:
        adc.return_value = (_AnonymousCredentialsWithUniverseDomain(), None)
        transport_class(quota_project_id="octopus", scopes=["1", "2"])
        adc.assert_called_once_with(
            scopes=["1", "2"],
            default_scopes=(
                "https://www.googleapis.com/auth/cloud-platform",
                "https://www.googleapis.com/auth/cloudkms",
            ),
            quota_project_id="octopus",
        )


@pytest.mark.parametrize(
    "transport_class",
    [
        transports.EkmServiceGrpcTransport,
        transports.EkmServiceGrpcAsyncIOTransport,
        transports.EkmServiceRestTransport,
    ],
)
def test_ekm_service_transport_auth_gdch_credentials(transport_class):
    host = "https://language.com"
    api_audience_tests = [None, "https://language2.com"]
    api_audience_expect = [host, "https://language2.com"]
    for t, e in zip(api_audience_tests, api_audience_expect):
        with mock.patch.object(google.auth, "default", autospec=True) as adc:
            gdch_mock = mock.MagicMock()
            type(gdch_mock).with_gdch_audience = mock.PropertyMock(
                return_value=gdch_mock
            )
            adc.return_value = (gdch_mock, None)
            transport_class(host=host, api_audience=t)
            gdch_mock.with_gdch_audience.assert_called_once_with(e)


@pytest.mark.parametrize(
    "transport_class,grpc_helpers",
    [
        (transports.EkmServiceGrpcTransport, grpc_helpers),
        (transports.EkmServiceGrpcAsyncIOTransport, grpc_helpers_async),
    ],
)
def test_ekm_service_transport_create_channel(transport_class, grpc_helpers):
    # If credentials and host are not provided, the transport class should use
    # ADC credentials.
    with mock.patch.object(
        google.auth, "default", autospec=True
    ) as adc, mock.patch.object(
        grpc_helpers, "create_channel", autospec=True
    ) as create_channel:
        creds = _AnonymousCredentialsWithUniverseDomain()
        adc.return_value = (creds, None)
        transport_class(quota_project_id="octopus", scopes=["1", "2"])

        create_channel.assert_called_with(
            "cloudkms.googleapis.com:443",
            credentials=creds,
            credentials_file=None,
            quota_project_id="octopus",
            default_scopes=(
                "https://www.googleapis.com/auth/cloud-platform",
                "https://www.googleapis.com/auth/cloudkms",
            ),
            scopes=["1", "2"],
            default_host="cloudkms.googleapis.com",
            ssl_credentials=None,
            options=[
                ("grpc.max_send_message_length", -1),
                ("grpc.max_receive_message_length", -1),
            ],
        )


@pytest.mark.parametrize(
    "transport_class",
    [transports.EkmServiceGrpcTransport, transports.EkmServiceGrpcAsyncIOTransport],
)
def test_ekm_service_grpc_transport_client_cert_source_for_mtls(transport_class):
    cred = _AnonymousCredentialsWithUniverseDomain()

    # Check ssl_channel_credentials is used if provided.
    with mock.patch.object(transport_class, "create_channel") as mock_create_channel:
        mock_ssl_channel_creds = mock.Mock()
        transport_class(
            host="squid.clam.whelk",
            credentials=cred,
            ssl_channel_credentials=mock_ssl_channel_creds,
        )
        mock_create_channel.assert_called_once_with(
            "squid.clam.whelk:443",
            credentials=cred,
            credentials_file=None,
            scopes=None,
            ssl_credentials=mock_ssl_channel_creds,
            quota_project_id=None,
            options=[
                ("grpc.max_send_message_length", -1),
                ("grpc.max_receive_message_length", -1),
            ],
        )

    # Check if ssl_channel_credentials is not provided, then client_cert_source_for_mtls
    # is used.
    with mock.patch.object(transport_class, "create_channel", return_value=mock.Mock()):
        with mock.patch("grpc.ssl_channel_credentials") as mock_ssl_cred:
            transport_class(
                credentials=cred,
                client_cert_source_for_mtls=client_cert_source_callback,
            )
            expected_cert, expected_key = client_cert_source_callback()
            mock_ssl_cred.assert_called_once_with(
                certificate_chain=expected_cert, private_key=expected_key
            )


def test_ekm_service_http_transport_client_cert_source_for_mtls():
    cred = _AnonymousCredentialsWithUniverseDomain()
    with mock.patch(
        "google.auth.transport.requests.AuthorizedSession.configure_mtls_channel"
    ) as mock_configure_mtls_channel:
        transports.EkmServiceRestTransport(
            credentials=cred, client_cert_source_for_mtls=client_cert_source_callback
        )
        mock_configure_mtls_channel.assert_called_once_with(client_cert_source_callback)


@pytest.mark.parametrize(
    "transport_name",
    [
        "grpc",
        "grpc_asyncio",
        "rest",
    ],
)
def test_ekm_service_host_no_port(transport_name):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        client_options=client_options.ClientOptions(
            api_endpoint="cloudkms.googleapis.com"
        ),
        transport=transport_name,
    )
    assert client.transport._host == (
        "cloudkms.googleapis.com:443"
        if transport_name in ["grpc", "grpc_asyncio"]
        else "https://cloudkms.googleapis.com"
    )


@pytest.mark.parametrize(
    "transport_name",
    [
        "grpc",
        "grpc_asyncio",
        "rest",
    ],
)
def test_ekm_service_host_with_port(transport_name):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        client_options=client_options.ClientOptions(
            api_endpoint="cloudkms.googleapis.com:8000"
        ),
        transport=transport_name,
    )
    assert client.transport._host == (
        "cloudkms.googleapis.com:8000"
        if transport_name in ["grpc", "grpc_asyncio"]
        else "https://cloudkms.googleapis.com:8000"
    )


@pytest.mark.parametrize(
    "transport_name",
    [
        "rest",
    ],
)
def test_ekm_service_client_transport_session_collision(transport_name):
    creds1 = _AnonymousCredentialsWithUniverseDomain()
    creds2 = _AnonymousCredentialsWithUniverseDomain()
    client1 = EkmServiceClient(
        credentials=creds1,
        transport=transport_name,
    )
    client2 = EkmServiceClient(
        credentials=creds2,
        transport=transport_name,
    )
    session1 = client1.transport.list_ekm_connections._session
    session2 = client2.transport.list_ekm_connections._session
    assert session1 != session2
    session1 = client1.transport.get_ekm_connection._session
    session2 = client2.transport.get_ekm_connection._session
    assert session1 != session2
    session1 = client1.transport.create_ekm_connection._session
    session2 = client2.transport.create_ekm_connection._session
    assert session1 != session2
    session1 = client1.transport.update_ekm_connection._session
    session2 = client2.transport.update_ekm_connection._session
    assert session1 != session2
    session1 = client1.transport.get_ekm_config._session
    session2 = client2.transport.get_ekm_config._session
    assert session1 != session2
    session1 = client1.transport.update_ekm_config._session
    session2 = client2.transport.update_ekm_config._session
    assert session1 != session2
    session1 = client1.transport.verify_connectivity._session
    session2 = client2.transport.verify_connectivity._session
    assert session1 != session2


def test_ekm_service_grpc_transport_channel():
    channel = grpc.secure_channel("http://localhost/", grpc.local_channel_credentials())

    # Check that channel is used if provided.
    transport = transports.EkmServiceGrpcTransport(
        host="squid.clam.whelk",
        channel=channel,
    )
    assert transport.grpc_channel == channel
    assert transport._host == "squid.clam.whelk:443"
    assert transport._ssl_channel_credentials == None


def test_ekm_service_grpc_asyncio_transport_channel():
    channel = aio.secure_channel("http://localhost/", grpc.local_channel_credentials())

    # Check that channel is used if provided.
    transport = transports.EkmServiceGrpcAsyncIOTransport(
        host="squid.clam.whelk",
        channel=channel,
    )
    assert transport.grpc_channel == channel
    assert transport._host == "squid.clam.whelk:443"
    assert transport._ssl_channel_credentials == None


# Remove this test when deprecated arguments (api_mtls_endpoint, client_cert_source) are
# removed from grpc/grpc_asyncio transport constructor.
@pytest.mark.parametrize(
    "transport_class",
    [transports.EkmServiceGrpcTransport, transports.EkmServiceGrpcAsyncIOTransport],
)
def test_ekm_service_transport_channel_mtls_with_client_cert_source(transport_class):
    with mock.patch(
        "grpc.ssl_channel_credentials", autospec=True
    ) as grpc_ssl_channel_cred:
        with mock.patch.object(
            transport_class, "create_channel"
        ) as grpc_create_channel:
            mock_ssl_cred = mock.Mock()
            grpc_ssl_channel_cred.return_value = mock_ssl_cred

            mock_grpc_channel = mock.Mock()
            grpc_create_channel.return_value = mock_grpc_channel

            cred = _AnonymousCredentialsWithUniverseDomain()
            with pytest.warns(DeprecationWarning):
                with mock.patch.object(google.auth, "default") as adc:
                    adc.return_value = (cred, None)
                    transport = transport_class(
                        host="squid.clam.whelk",
                        api_mtls_endpoint="mtls.squid.clam.whelk",
                        client_cert_source=client_cert_source_callback,
                    )
                    adc.assert_called_once()

            grpc_ssl_channel_cred.assert_called_once_with(
                certificate_chain=b"cert bytes", private_key=b"key bytes"
            )
            grpc_create_channel.assert_called_once_with(
                "mtls.squid.clam.whelk:443",
                credentials=cred,
                credentials_file=None,
                scopes=None,
                ssl_credentials=mock_ssl_cred,
                quota_project_id=None,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )
            assert transport.grpc_channel == mock_grpc_channel
            assert transport._ssl_channel_credentials == mock_ssl_cred


# Remove this test when deprecated arguments (api_mtls_endpoint, client_cert_source) are
# removed from grpc/grpc_asyncio transport constructor.
@pytest.mark.parametrize(
    "transport_class",
    [transports.EkmServiceGrpcTransport, transports.EkmServiceGrpcAsyncIOTransport],
)
def test_ekm_service_transport_channel_mtls_with_adc(transport_class):
    mock_ssl_cred = mock.Mock()
    with mock.patch.multiple(
        "google.auth.transport.grpc.SslCredentials",
        __init__=mock.Mock(return_value=None),
        ssl_credentials=mock.PropertyMock(return_value=mock_ssl_cred),
    ):
        with mock.patch.object(
            transport_class, "create_channel"
        ) as grpc_create_channel:
            mock_grpc_channel = mock.Mock()
            grpc_create_channel.return_value = mock_grpc_channel
            mock_cred = mock.Mock()

            with pytest.warns(DeprecationWarning):
                transport = transport_class(
                    host="squid.clam.whelk",
                    credentials=mock_cred,
                    api_mtls_endpoint="mtls.squid.clam.whelk",
                    client_cert_source=None,
                )

            grpc_create_channel.assert_called_once_with(
                "mtls.squid.clam.whelk:443",
                credentials=mock_cred,
                credentials_file=None,
                scopes=None,
                ssl_credentials=mock_ssl_cred,
                quota_project_id=None,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )
            assert transport.grpc_channel == mock_grpc_channel


def test_ekm_config_path():
    project = "squid"
    location = "clam"
    expected = "projects/{project}/locations/{location}/ekmConfig".format(
        project=project,
        location=location,
    )
    actual = EkmServiceClient.ekm_config_path(project, location)
    assert expected == actual


def test_parse_ekm_config_path():
    expected = {
        "project": "whelk",
        "location": "octopus",
    }
    path = EkmServiceClient.ekm_config_path(**expected)

    # Check that the path construction is reversible.
    actual = EkmServiceClient.parse_ekm_config_path(path)
    assert expected == actual


def test_ekm_connection_path():
    project = "oyster"
    location = "nudibranch"
    ekm_connection = "cuttlefish"
    expected = "projects/{project}/locations/{location}/ekmConnections/{ekm_connection}".format(
        project=project,
        location=location,
        ekm_connection=ekm_connection,
    )
    actual = EkmServiceClient.ekm_connection_path(project, location, ekm_connection)
    assert expected == actual


def test_parse_ekm_connection_path():
    expected = {
        "project": "mussel",
        "location": "winkle",
        "ekm_connection": "nautilus",
    }
    path = EkmServiceClient.ekm_connection_path(**expected)

    # Check that the path construction is reversible.
    actual = EkmServiceClient.parse_ekm_connection_path(path)
    assert expected == actual


def test_service_path():
    project = "scallop"
    location = "abalone"
    namespace = "squid"
    service = "clam"
    expected = "projects/{project}/locations/{location}/namespaces/{namespace}/services/{service}".format(
        project=project,
        location=location,
        namespace=namespace,
        service=service,
    )
    actual = EkmServiceClient.service_path(project, location, namespace, service)
    assert expected == actual


def test_parse_service_path():
    expected = {
        "project": "whelk",
        "location": "octopus",
        "namespace": "oyster",
        "service": "nudibranch",
    }
    path = EkmServiceClient.service_path(**expected)

    # Check that the path construction is reversible.
    actual = EkmServiceClient.parse_service_path(path)
    assert expected == actual


def test_common_billing_account_path():
    billing_account = "cuttlefish"
    expected = "billingAccounts/{billing_account}".format(
        billing_account=billing_account,
    )
    actual = EkmServiceClient.common_billing_account_path(billing_account)
    assert expected == actual


def test_parse_common_billing_account_path():
    expected = {
        "billing_account": "mussel",
    }
    path = EkmServiceClient.common_billing_account_path(**expected)

    # Check that the path construction is reversible.
    actual = EkmServiceClient.parse_common_billing_account_path(path)
    assert expected == actual


def test_common_folder_path():
    folder = "winkle"
    expected = "folders/{folder}".format(
        folder=folder,
    )
    actual = EkmServiceClient.common_folder_path(folder)
    assert expected == actual


def test_parse_common_folder_path():
    expected = {
        "folder": "nautilus",
    }
    path = EkmServiceClient.common_folder_path(**expected)

    # Check that the path construction is reversible.
    actual = EkmServiceClient.parse_common_folder_path(path)
    assert expected == actual


def test_common_organization_path():
    organization = "scallop"
    expected = "organizations/{organization}".format(
        organization=organization,
    )
    actual = EkmServiceClient.common_organization_path(organization)
    assert expected == actual


def test_parse_common_organization_path():
    expected = {
        "organization": "abalone",
    }
    path = EkmServiceClient.common_organization_path(**expected)

    # Check that the path construction is reversible.
    actual = EkmServiceClient.parse_common_organization_path(path)
    assert expected == actual


def test_common_project_path():
    project = "squid"
    expected = "projects/{project}".format(
        project=project,
    )
    actual = EkmServiceClient.common_project_path(project)
    assert expected == actual


def test_parse_common_project_path():
    expected = {
        "project": "clam",
    }
    path = EkmServiceClient.common_project_path(**expected)

    # Check that the path construction is reversible.
    actual = EkmServiceClient.parse_common_project_path(path)
    assert expected == actual


def test_common_location_path():
    project = "whelk"
    location = "octopus"
    expected = "projects/{project}/locations/{location}".format(
        project=project,
        location=location,
    )
    actual = EkmServiceClient.common_location_path(project, location)
    assert expected == actual


def test_parse_common_location_path():
    expected = {
        "project": "oyster",
        "location": "nudibranch",
    }
    path = EkmServiceClient.common_location_path(**expected)

    # Check that the path construction is reversible.
    actual = EkmServiceClient.parse_common_location_path(path)
    assert expected == actual


def test_client_with_default_client_info():
    client_info = gapic_v1.client_info.ClientInfo()

    with mock.patch.object(
        transports.EkmServiceTransport, "_prep_wrapped_messages"
    ) as prep:
        client = EkmServiceClient(
            credentials=_AnonymousCredentialsWithUniverseDomain(),
            client_info=client_info,
        )
        prep.assert_called_once_with(client_info)

    with mock.patch.object(
        transports.EkmServiceTransport, "_prep_wrapped_messages"
    ) as prep:
        transport_class = EkmServiceClient.get_transport_class()
        transport = transport_class(
            credentials=_AnonymousCredentialsWithUniverseDomain(),
            client_info=client_info,
        )
        prep.assert_called_once_with(client_info)


@pytest.mark.asyncio
async def test_transport_close_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="grpc_asyncio",
    )
    with mock.patch.object(
        type(getattr(client.transport, "grpc_channel")), "close"
    ) as close:
        async with client:
            close.assert_not_called()
        close.assert_called_once()


def test_get_location_rest_bad_request(
    transport: str = "rest", request_type=locations_pb2.GetLocationRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    request = request_type()
    request = json_format.ParseDict(
        {"name": "projects/sample1/locations/sample2"}, request
    )

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.get_location(request)


@pytest.mark.parametrize(
    "request_type",
    [
        locations_pb2.GetLocationRequest,
        dict,
    ],
)
def test_get_location_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request_init = {"name": "projects/sample1/locations/sample2"}
    request = request_type(**request_init)
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = locations_pb2.Location()

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        response = client.get_location(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, locations_pb2.Location)


def test_list_locations_rest_bad_request(
    transport: str = "rest", request_type=locations_pb2.ListLocationsRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    request = request_type()
    request = json_format.ParseDict({"name": "projects/sample1"}, request)

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.list_locations(request)


@pytest.mark.parametrize(
    "request_type",
    [
        locations_pb2.ListLocationsRequest,
        dict,
    ],
)
def test_list_locations_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request_init = {"name": "projects/sample1"}
    request = request_type(**request_init)
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = locations_pb2.ListLocationsResponse()

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        response = client.list_locations(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, locations_pb2.ListLocationsResponse)


def test_get_iam_policy_rest_bad_request(
    transport: str = "rest", request_type=iam_policy_pb2.GetIamPolicyRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    request = request_type()
    request = json_format.ParseDict(
        {"resource": "projects/sample1/locations/sample2/keyRings/sample3"}, request
    )

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.get_iam_policy(request)


@pytest.mark.parametrize(
    "request_type",
    [
        iam_policy_pb2.GetIamPolicyRequest,
        dict,
    ],
)
def test_get_iam_policy_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request_init = {"resource": "projects/sample1/locations/sample2/keyRings/sample3"}
    request = request_type(**request_init)
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = policy_pb2.Policy()

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        response = client.get_iam_policy(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, policy_pb2.Policy)


def test_set_iam_policy_rest_bad_request(
    transport: str = "rest", request_type=iam_policy_pb2.SetIamPolicyRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    request = request_type()
    request = json_format.ParseDict(
        {"resource": "projects/sample1/locations/sample2/keyRings/sample3"}, request
    )

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.set_iam_policy(request)


@pytest.mark.parametrize(
    "request_type",
    [
        iam_policy_pb2.SetIamPolicyRequest,
        dict,
    ],
)
def test_set_iam_policy_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request_init = {"resource": "projects/sample1/locations/sample2/keyRings/sample3"}
    request = request_type(**request_init)
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = policy_pb2.Policy()

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        response = client.set_iam_policy(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, policy_pb2.Policy)


def test_test_iam_permissions_rest_bad_request(
    transport: str = "rest", request_type=iam_policy_pb2.TestIamPermissionsRequest
):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    request = request_type()
    request = json_format.ParseDict(
        {"resource": "projects/sample1/locations/sample2/keyRings/sample3"}, request
    )

    # Mock the http request call within the method and fake a BadRequest error.
    with mock.patch.object(Session, "request") as req, pytest.raises(
        core_exceptions.BadRequest
    ):
        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 400
        response_value.request = Request()
        req.return_value = response_value
        client.test_iam_permissions(request)


@pytest.mark.parametrize(
    "request_type",
    [
        iam_policy_pb2.TestIamPermissionsRequest,
        dict,
    ],
)
def test_test_iam_permissions_rest(request_type):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport="rest",
    )
    request_init = {"resource": "projects/sample1/locations/sample2/keyRings/sample3"}
    request = request_type(**request_init)
    # Mock the http request call within the method and fake a response.
    with mock.patch.object(type(client.transport._session), "request") as req:
        # Designate an appropriate value for the returned response.
        return_value = iam_policy_pb2.TestIamPermissionsResponse()

        # Wrap the value into a proper Response obj
        response_value = Response()
        response_value.status_code = 200
        json_return_value = json_format.MessageToJson(return_value)

        response_value._content = json_return_value.encode("UTF-8")
        req.return_value = response_value

        response = client.test_iam_permissions(request)

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam_policy_pb2.TestIamPermissionsResponse)


def test_list_locations(transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = locations_pb2.ListLocationsRequest()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_locations), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = locations_pb2.ListLocationsResponse()
        response = client.list_locations(request)
        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, locations_pb2.ListLocationsResponse)


@pytest.mark.asyncio
async def test_list_locations_async(transport: str = "grpc_asyncio"):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = locations_pb2.ListLocationsRequest()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_locations), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            locations_pb2.ListLocationsResponse()
        )
        response = await client.list_locations(request)
        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, locations_pb2.ListLocationsResponse)


def test_list_locations_field_headers():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = locations_pb2.ListLocationsRequest()
    request.name = "locations"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_locations), "__call__") as call:
        call.return_value = locations_pb2.ListLocationsResponse()

        client.list_locations(request)
        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=locations",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_list_locations_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = locations_pb2.ListLocationsRequest()
    request.name = "locations"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_locations), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            locations_pb2.ListLocationsResponse()
        )
        await client.list_locations(request)
        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=locations",
    ) in kw["metadata"]


def test_list_locations_from_dict():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_locations), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = locations_pb2.ListLocationsResponse()

        response = client.list_locations(
            request={
                "name": "locations",
            }
        )
        call.assert_called()


@pytest.mark.asyncio
async def test_list_locations_from_dict_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_locations), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            locations_pb2.ListLocationsResponse()
        )
        response = await client.list_locations(
            request={
                "name": "locations",
            }
        )
        call.assert_called()


def test_get_location(transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = locations_pb2.GetLocationRequest()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_location), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = locations_pb2.Location()
        response = client.get_location(request)
        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, locations_pb2.Location)


@pytest.mark.asyncio
async def test_get_location_async(transport: str = "grpc_asyncio"):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = locations_pb2.GetLocationRequest()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_location), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            locations_pb2.Location()
        )
        response = await client.get_location(request)
        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, locations_pb2.Location)


def test_get_location_field_headers():
    client = EkmServiceClient(credentials=_AnonymousCredentialsWithUniverseDomain())

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = locations_pb2.GetLocationRequest()
    request.name = "locations/abc"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_location), "__call__") as call:
        call.return_value = locations_pb2.Location()

        client.get_location(request)
        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=locations/abc",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_get_location_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain()
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = locations_pb2.GetLocationRequest()
    request.name = "locations/abc"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_location), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            locations_pb2.Location()
        )
        await client.get_location(request)
        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "name=locations/abc",
    ) in kw["metadata"]


def test_get_location_from_dict():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_locations), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = locations_pb2.Location()

        response = client.get_location(
            request={
                "name": "locations/abc",
            }
        )
        call.assert_called()


@pytest.mark.asyncio
async def test_get_location_from_dict_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.list_locations), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            locations_pb2.Location()
        )
        response = await client.get_location(
            request={
                "name": "locations",
            }
        )
        call.assert_called()


def test_set_iam_policy(transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = iam_policy_pb2.SetIamPolicyRequest()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy(
            version=774,
            etag=b"etag_blob",
        )
        response = client.set_iam_policy(request)
        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]

        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, policy_pb2.Policy)

    assert response.version == 774

    assert response.etag == b"etag_blob"


@pytest.mark.asyncio
async def test_set_iam_policy_async(transport: str = "grpc_asyncio"):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = iam_policy_pb2.SetIamPolicyRequest()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            policy_pb2.Policy(
                version=774,
                etag=b"etag_blob",
            )
        )
        response = await client.set_iam_policy(request)
        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]

        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, policy_pb2.Policy)

    assert response.version == 774

    assert response.etag == b"etag_blob"


def test_set_iam_policy_field_headers():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.SetIamPolicyRequest()
    request.resource = "resource/value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        call.return_value = policy_pb2.Policy()

        client.set_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource/value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_set_iam_policy_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.SetIamPolicyRequest()
    request.resource = "resource/value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(policy_pb2.Policy())

        await client.set_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource/value",
    ) in kw["metadata"]


def test_set_iam_policy_from_dict():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy()

        response = client.set_iam_policy(
            request={
                "resource": "resource_value",
                "policy": policy_pb2.Policy(version=774),
            }
        )
        call.assert_called()


@pytest.mark.asyncio
async def test_set_iam_policy_from_dict_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.set_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(policy_pb2.Policy())

        response = await client.set_iam_policy(
            request={
                "resource": "resource_value",
                "policy": policy_pb2.Policy(version=774),
            }
        )
        call.assert_called()


def test_get_iam_policy(transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = iam_policy_pb2.GetIamPolicyRequest()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy(
            version=774,
            etag=b"etag_blob",
        )

        response = client.get_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]

        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, policy_pb2.Policy)

    assert response.version == 774

    assert response.etag == b"etag_blob"


@pytest.mark.asyncio
async def test_get_iam_policy_async(transport: str = "grpc_asyncio"):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = iam_policy_pb2.GetIamPolicyRequest()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            policy_pb2.Policy(
                version=774,
                etag=b"etag_blob",
            )
        )

        response = await client.get_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]

        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, policy_pb2.Policy)

    assert response.version == 774

    assert response.etag == b"etag_blob"


def test_get_iam_policy_field_headers():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.GetIamPolicyRequest()
    request.resource = "resource/value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        call.return_value = policy_pb2.Policy()

        client.get_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource/value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_get_iam_policy_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.GetIamPolicyRequest()
    request.resource = "resource/value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(policy_pb2.Policy())

        await client.get_iam_policy(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource/value",
    ) in kw["metadata"]


def test_get_iam_policy_from_dict():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = policy_pb2.Policy()

        response = client.get_iam_policy(
            request={
                "resource": "resource_value",
                "options": options_pb2.GetPolicyOptions(requested_policy_version=2598),
            }
        )
        call.assert_called()


@pytest.mark.asyncio
async def test_get_iam_policy_from_dict_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(type(client.transport.get_iam_policy), "__call__") as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(policy_pb2.Policy())

        response = await client.get_iam_policy(
            request={
                "resource": "resource_value",
                "options": options_pb2.GetPolicyOptions(requested_policy_version=2598),
            }
        )
        call.assert_called()


def test_test_iam_permissions(transport: str = "grpc"):
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = iam_policy_pb2.TestIamPermissionsRequest()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam_policy_pb2.TestIamPermissionsResponse(
            permissions=["permissions_value"],
        )

        response = client.test_iam_permissions(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]

        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam_policy_pb2.TestIamPermissionsResponse)

    assert response.permissions == ["permissions_value"]


@pytest.mark.asyncio
async def test_test_iam_permissions_async(transport: str = "grpc_asyncio"):
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
        transport=transport,
    )

    # Everything is optional in proto3 as far as the runtime is concerned,
    # and we are mocking out the actual API, so just send an empty request.
    request = iam_policy_pb2.TestIamPermissionsRequest()

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam_policy_pb2.TestIamPermissionsResponse(
                permissions=["permissions_value"],
            )
        )

        response = await client.test_iam_permissions(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]

        assert args[0] == request

    # Establish that the response is the type that we expect.
    assert isinstance(response, iam_policy_pb2.TestIamPermissionsResponse)

    assert response.permissions == ["permissions_value"]


def test_test_iam_permissions_field_headers():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.TestIamPermissionsRequest()
    request.resource = "resource/value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        call.return_value = iam_policy_pb2.TestIamPermissionsResponse()

        client.test_iam_permissions(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls) == 1
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource/value",
    ) in kw["metadata"]


@pytest.mark.asyncio
async def test_test_iam_permissions_field_headers_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )

    # Any value that is part of the HTTP/1.1 URI should be sent as
    # a field header. Set these to a non-empty value.
    request = iam_policy_pb2.TestIamPermissionsRequest()
    request.resource = "resource/value"

    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam_policy_pb2.TestIamPermissionsResponse()
        )

        await client.test_iam_permissions(request)

        # Establish that the underlying gRPC stub method was called.
        assert len(call.mock_calls)
        _, args, _ = call.mock_calls[0]
        assert args[0] == request

    # Establish that the field header was sent.
    _, _, kw = call.mock_calls[0]
    assert (
        "x-goog-request-params",
        "resource=resource/value",
    ) in kw["metadata"]


def test_test_iam_permissions_from_dict():
    client = EkmServiceClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = iam_policy_pb2.TestIamPermissionsResponse()

        response = client.test_iam_permissions(
            request={
                "resource": "resource_value",
                "permissions": ["permissions_value"],
            }
        )
        call.assert_called()


@pytest.mark.asyncio
async def test_test_iam_permissions_from_dict_async():
    client = EkmServiceAsyncClient(
        credentials=_AnonymousCredentialsWithUniverseDomain(),
    )
    # Mock the actual call within the gRPC stub, and fake the request.
    with mock.patch.object(
        type(client.transport.test_iam_permissions), "__call__"
    ) as call:
        # Designate an appropriate return value for the call.
        call.return_value = grpc_helpers_async.FakeUnaryUnaryCall(
            iam_policy_pb2.TestIamPermissionsResponse()
        )

        response = await client.test_iam_permissions(
            request={
                "resource": "resource_value",
                "permissions": ["permissions_value"],
            }
        )
        call.assert_called()


def test_transport_close():
    transports = {
        "rest": "_session",
        "grpc": "_grpc_channel",
    }

    for transport, close_name in transports.items():
        client = EkmServiceClient(
            credentials=_AnonymousCredentialsWithUniverseDomain(), transport=transport
        )
        with mock.patch.object(
            type(getattr(client.transport, close_name)), "close"
        ) as close:
            with client:
                close.assert_not_called()
            close.assert_called_once()


def test_client_ctx():
    transports = [
        "rest",
        "grpc",
    ]
    for transport in transports:
        client = EkmServiceClient(
            credentials=_AnonymousCredentialsWithUniverseDomain(), transport=transport
        )
        # Test client calls underlying transport.
        with mock.patch.object(type(client.transport), "close") as close:
            close.assert_not_called()
            with client:
                pass
            close.assert_called()


@pytest.mark.parametrize(
    "client_class,transport_class",
    [
        (EkmServiceClient, transports.EkmServiceGrpcTransport),
        (EkmServiceAsyncClient, transports.EkmServiceGrpcAsyncIOTransport),
    ],
)
def test_api_key_credentials(client_class, transport_class):
    with mock.patch.object(
        google.auth._default, "get_api_key_credentials", create=True
    ) as get_api_key_credentials:
        mock_cred = mock.Mock()
        get_api_key_credentials.return_value = mock_cred
        options = client_options.ClientOptions()
        options.api_key = "api_key"
        with mock.patch.object(transport_class, "__init__") as patched:
            patched.return_value = None
            client = client_class(client_options=options)
            patched.assert_called_once_with(
                credentials=mock_cred,
                credentials_file=None,
                host=client._DEFAULT_ENDPOINT_TEMPLATE.format(
                    UNIVERSE_DOMAIN=client._DEFAULT_UNIVERSE
                ),
                scopes=None,
                client_cert_source_for_mtls=None,
                quota_project_id=None,
                client_info=transports.base.DEFAULT_CLIENT_INFO,
                always_use_jwt_access=True,
                api_audience=None,
            )
