/**********************************************************************
 * $port_info example -- C source code using ACC PLI routines
 *
 * C source to scan through a module and list the names of all ports,
 * along with the vector size of the port and the type of signals 
 * connected internally and externally to the port.
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *
 * Usage: $port_info(<module_instance>);
 *
 * Routine definitions for a veriusertfs array:
 *  /* routine prototypes -/
 *   extern int PLIbook_PortInfo_checktf(),
 *              PLIbook_PortInfo_calltf();
 *  /* table entries -/
 *   {usertask,                       /* type of PLI routine -/
 *     0,                             /* user_data value -/
 *     PLIbook_PortInfo_checktf,      /* checktf routine -/
 *     0,                             /* sizetf routine -/
 *     PLIbook_PortInfo_calltf,       /* calltf routine -/
 *     0,                             /* misctf routine -/
 *     "$port_info",                  /* system task/function name -/
 *     1                              /* forward reference = true -/
 *   },
 *********************************************************************/

#include "veriuser.h"         /* IEEE 1364 PLI TF  routine library */
#include "acc_user.h"         /* IEEE 1364 PLI ACC routine library */
/**********************************************************************
 * checktf routine
 *********************************************************************/
void PLIbook_PortInfo_checktf()
{
  acc_initialize();
  if (tf_nump() != 1)
    tf_error("$port_info requires 1 argument");
  if (tf_typep(1) == TF_NULLPARAM)
    tf_error("$port_info arg cannot be null");
  if (acc_fetch_type(acc_handle_tfarg(1)) != accModule)
    tf_error("$port_info arg must be a module instance");
  acc_close();
  return;
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
void PLIbook_PortInfo_calltf() 
{
  handle mod_h, port_h, loconn_h, hiconn_h;

  acc_initialize();
  acc_configure(accDisplayWarnings, "true");

  mod_h = acc_handle_tfarg(1);

  io_printf("\nModule %s:\n", acc_fetch_defname(mod_h));
  io_printf("  Instance name: %s\n", acc_fetch_fullname(mod_h));

  switch ( acc_fetch_fulltype(mod_h) ) {
    case accTopModule:
      io_printf("  Module type: top-level\n");
      break;
    case accModuleInstance:
      io_printf("  Module type: module instance\n");
      break;
    case accCellInstance:
      io_printf("  Module type: cell module\n");
      break;
    default:
      io_printf("  Module type: unknown\n");
  }

  io_printf("  Ports:\n");
  port_h = null;
  while (port_h = acc_next_port(mod_h, port_h)) {
    io_printf("    %-8s", acc_fetch_name(port_h) );
    io_printf("%2d-bit ", acc_fetch_size(port_h) );
    switch ( acc_fetch_direction(port_h) ) {
      case accInput:
        io_printf("input   ");
        break;
      case accOutput:
        io_printf("output  ");
        break;
      case accInout:
        io_printf("inout   ");
        break;
      case accMixedIo:
        io_printf("mixed input/output  ");
        break;
      default:
        io_printf("unknown direction  ");
    }
    
    if (acc_object_of_type(port_h, accExpandedVector))
      io_printf("  Expanded=true   ");
    else
      io_printf("  Expanded=false  ");
    if (acc_object_of_type(port_h, accUnExpandedVector))
      io_printf("  Unexpanded=true\n");
    else
      io_printf("  Unexpanded=false\n");
    
    loconn_h = acc_handle_loconn(port_h);
    io_printf("      Loconn type = %s\n",
              acc_fetch_type_str(acc_fetch_fulltype(loconn_h)));
    
    hiconn_h = acc_handle_hiconn(port_h);
    if (hiconn_h)
      io_printf("      Hiconn type = %s\n\n",
                acc_fetch_type_str(acc_fetch_fulltype(hiconn_h)));
    else
      io_printf("      Hiconn type = none\n\n");
  } /* end of next port_h loop */
  acc_close();
  return;
}
/*********************************************************************/

