/*
 *	udev related utilities
 *
 *	Copyright (C) 2011 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *	MA  02110-1301  USA
 *
 *	Authors: Marius Tomaschewski <mt@suse.de>
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	You can find the wicked project at http://gitorious.org/wicked/.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>
#include <assert.h>

#include <sutils.h>
#include <futils.h>
#include <logging.h>

#define UDEV_NET_LOCK_MODE	S_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH
#define UDEV_NET_NAME_RULE	"/etc/udev/rules.d/70-persistent-net.rules"
#define UDEV_NET_NAME_TEMP	UDEV_NET_NAME_RULE".XXXXXX\0\0"
#define UDEV_NET_NAME_SECS	5  /* running system shouldn't need more */

#define NC_UDEV_RULE_CHUNK	16
#define NC_UDEV_RULE_INIT	{ .count = 0, .data = NULL }
#define	NC_UDEV_RULE_CMD_INIT	{ .op = 0, .key = NULL, .val = NULL }

enum {
	NC_UDEV_OP_NONE		= 0,	/* no op        */
	NC_UDEV_OP_EQ		= 1,	/* equal        */
	NC_UDEV_OP_NE		= 2,	/* not equal    */
	NC_UDEV_OP_ADD		= 3,	/* list add     */
	NC_UDEV_OP_SET		= 4,	/* set / reset	*/
	NC_UDEV_OP_FIN		= 5,	/* final assign */
};

static nc_intmap_t	__nc_udev_op_map[] = {
	{ "==",		NC_UDEV_OP_EQ		},
	{ "!=",		NC_UDEV_OP_NE		},
	{ "+=",		NC_UDEV_OP_ADD		},
	{ ":=",		NC_UDEV_OP_FIN		},
	{ "=",		NC_UDEV_OP_SET		},
	{ NULL,		NC_UDEV_OP_NONE		},
};

typedef struct nc_udev_rule_cmd {
	unsigned int		op;
	char *			key;
	char *			val;
} nc_udev_rule_cmd_t;

typedef struct nc_udev_rule {
	unsigned int		count;
	nc_udev_rule_cmd_t **	data;
} nc_udev_rule_t;
/*
typedef struct nc_udev_rule_set {
	unsigned int		count;
	nc_udev_rule_t **	rules;
} nc_udev_rule_set_t;
*/

/* currently all internal (no need to expose) */
static nc_udev_rule_cmd_t *	nc_udev_rule_cmd_new(unsigned int op, const char *key,
				                                      const char *val);
static void			nc_udev_rule_cmd_clear(nc_udev_rule_cmd_t *cmd);
static void			nc_udev_rule_cmd_free(nc_udev_rule_cmd_t *cmd);

static const char *		nc_udev_rule_op_format(unsigned int op);
static unsigned int		nc_udev_rule_op_parse(const char *str);

static void			nc_udev_rule_init(nc_udev_rule_t *rule);
static void			nc_udev_rule_destroy(nc_udev_rule_t *rule);
static int			nc_udev_rule_append(nc_udev_rule_t *rule, nc_udev_rule_cmd_t *cmd);

static int			nc_udev_rule_file_lock(const char *rule_file, unsigned int retry_sec);
static int			nc_udev_rule_file_unlock(const char *rule_file);

/* internal helpers */
static int			__nc_udev_rule_realloc(nc_udev_rule_t *rule, unsigned int newsize);
static char *			__strip_quotes(char *tok);
static int			__nc_udev_rule_parse(nc_udev_rule_t *rule, char *line);
static int			__nc_udev_net_rule_hwaddr_to_bus_id(nc_var_array_t *bus_id_map,
				                                    const char *rule_file,
				                                    char        rule_temp[]);


/* implementation of nc_udev_rule_cmd */
static nc_udev_rule_cmd_t *
nc_udev_rule_cmd_new(unsigned int op, const char *key, const char *val)
{
	nc_udev_rule_cmd_t *cmd;

	if(op > NC_UDEV_OP_FIN)
		return NULL;

	cmd = calloc(1, sizeof(*cmd));
	if(cmd) {
		cmd->op = op;
		if(nc_string_dup(&cmd->key, key) < 0) {
			nc_udev_rule_cmd_free(cmd);
			return NULL;
		}
		if(nc_string_dup(&cmd->val, val) < 0) {
			nc_udev_rule_cmd_free(cmd);
			return NULL;
		}
	}
	return cmd;
}

static void
nc_udev_rule_cmd_clear(nc_udev_rule_cmd_t *cmd)
{
	if(cmd) {
		cmd->op = NC_UDEV_OP_NONE;
		nc_string_free(&cmd->key);
		nc_string_free(&cmd->val);
	}
}

static void
nc_udev_rule_cmd_free(nc_udev_rule_cmd_t *cmd)
{
	if(cmd) {
		nc_udev_rule_cmd_clear(cmd);
		free(cmd);
	}
}


/* implementation of nc_udev_rule_op */
static const char *
nc_udev_rule_op_format(unsigned int op)
{
	return nc_format_int_mapped(op, __nc_udev_op_map);
}

static unsigned int
nc_udev_rule_op_parse(const char *str)
{
	nc_intmap_t *map;
	for(map = __nc_udev_op_map; map && map->name; ++map) {
		if(!strcmp(map->name, str)) {
			return map->value;
		}
	}
	return NC_UDEV_OP_NONE;
}

/* implementation of nc_udev_rule (cmd array) */
static void
nc_udev_rule_init(nc_udev_rule_t *rule)
{
	memset(rule, 0, sizeof(*rule));
}

static void
nc_udev_rule_destroy(nc_udev_rule_t *rule)
{
	if(rule) {
		while(rule->count--)
			nc_udev_rule_cmd_free(rule->data[rule->count]);
		free(rule->data);
		nc_udev_rule_init(rule);
	}
}

static int
__nc_udev_rule_realloc(nc_udev_rule_t *rule, unsigned int newsize)
{
	nc_udev_rule_cmd_t **newdata;

	newsize = (newsize + NC_UDEV_RULE_CHUNK);
	newdata = realloc(rule->data, newsize * sizeof(nc_udev_rule_cmd_t *));
	assert(newdata);
	if(!newdata)
		return -1;

	rule->data = newdata;
	memset( &rule->data[rule->count], 0,
		(newsize - rule->count) * sizeof(nc_udev_rule_cmd_t *));
	return 0;
}

static int
nc_udev_rule_append(nc_udev_rule_t *rule, nc_udev_rule_cmd_t *cmd)
{
	if(!rule || !cmd)
		return -1;

	if((rule->count % NC_UDEV_RULE_CHUNK) == 0) {
		if(__nc_udev_rule_realloc(rule, rule->count) != 0)
			return -1;
	}
	rule->data[rule->count++] = cmd;
	return 0;
}

static const char *
nc_udev_rule_lock_dir(void)
{
	static const char *lock_dirs[] = {
		"/run/udev",	/* systemd systems (>=SLE12) */
		"/dev/.udev",	/* sysvinit systems (SLES11) */
		NULL
	}, **dir;

	for (dir = lock_dirs; *dir; ++dir) {
		if (nc_isdir(*dir))
			return *dir;
	}
	return NULL;
}

static int
nc_udev_rule_lock_file(nc_stringbuf_t *rule_lock, const char *rule_file)
{
	const char *rule_name;
	const char *lock_dir;

	if (!rule_lock || !rule_file)
		return -1;

	rule_name = nc_basename(rule_file);
	if (nc_string_empty(rule_name))
		return -1;

	lock_dir = nc_udev_rule_lock_dir();
	if (nc_string_empty(lock_dir))
		return -1;

	nc_stringbuf_clear(rule_lock);
	nc_stringbuf_puts(rule_lock, lock_dir);
	nc_stringbuf_puts(rule_lock, "/.lock-");
	nc_stringbuf_puts(rule_lock, rule_name);
	if (rule_lock->len && rule_lock->string)
		return 0;

	nc_stringbuf_clear(rule_lock);
	return -1;
}

static int
nc_udev_rule_file_lock(const char *rule_file, unsigned int retry_sec)
{
	nc_stringbuf_t rule_lock = NC_STRINGBUF_INIT;

	if(nc_udev_rule_lock_file(&rule_lock, rule_file) != 0)
		return -1;

	while(mkdir(rule_lock.string, UDEV_NET_LOCK_MODE) != 0) {
		if(retry_sec > 0) {
			retry_sec--;
			sleep(1);
		} else {
			nc_stringbuf_destroy(&rule_lock);
			return 1;
		}
	}
	nc_stringbuf_destroy(&rule_lock);
	return 0;
}

static int
nc_udev_rule_file_unlock(const char *rule_file)
{
	nc_stringbuf_t rule_lock = NC_STRINGBUF_INIT;

	if(nc_udev_rule_lock_file(&rule_lock, rule_file) != 0)
		return -1;

	rmdir(rule_lock.string);
	nc_stringbuf_destroy(&rule_lock);
	return 0;
}

static char *
__strip_quotes(char *tok)
{
	/*
	** should be sufficient for us as we parse only
	** MAC address, interface name and bus-id ...
	*/
	size_t len = nc_string_len(tok);
	if(len > 0) {
		if(tok[0] == '"' || tok[0] == '\'') {
			if(len < 2 || tok[len-1] != tok[0])
				return NULL;
			tok[len-1] = '\0';
			tok++;
			len -= 2;
		}
		return tok;
	}
	return NULL;
}

static int
__nc_udev_rule_parse(nc_udev_rule_t *rule, char *line)
{
	char *tok, *_r = NULL;
	int ret = 0;

	if(!rule || !line)
		return -1;

	/* This will not work when there is a "," inside of a
	   quoted string, but we ignore the complete rule then */
	nc_udev_rule_destroy(rule);
	for(tok = strtok_r(line, ",", &_r); tok; tok = strtok_r(NULL, ",", &_r)) {
		nc_udev_rule_cmd_t *cmd;
		char *beg, *end, *ptr;

		cmd = nc_udev_rule_cmd_new(0, NULL, NULL);
		if(!cmd) {
			ret = -1;
			break;
		}

		tok = nc_string_strip_spaces(tok);
		beg = tok + strcspn(tok, "=!+:");
		if(!(beg > tok) || (nc_string_set(&cmd->key, tok, beg - tok) < 0)) {
			nc_udev_rule_cmd_free(cmd);
			ret = -1;
			break;
		}

		end = beg + strspn(beg,  "=!+:");
		if(!(end > beg)) {
			nc_udev_rule_cmd_free(cmd);
			ret = -1;
			break;
		}
		ptr = __strip_quotes(end);
		if(!ptr || nc_string_dup(&cmd->val, ptr) < 0) {
			nc_udev_rule_cmd_free(cmd);
			ret = -1;
			break;
		}
		*end = '\0'; /* op end */

		cmd->op = nc_udev_rule_op_parse(beg);
		if(cmd->op == NC_UDEV_OP_NONE) {
			nc_udev_rule_cmd_free(cmd);
			ret = -1;
			break;
		}
#if 0
		nc_trace("key: '%s',\n\top: '%s',\n\tval: '%s'",
			cmd->key, nc_udev_rule_op_format(cmd->op), cmd->val);
#endif
		if(nc_udev_rule_append(rule, cmd) < 0) {
			nc_udev_rule_cmd_free(cmd);
			ret = -1;
			break;
		}
	}

	if(ret) {
		nc_udev_rule_destroy(rule);
	}
	return ret;
}

static int
__nc_udev_net_rule_hwaddr_to_bus_id(nc_var_array_t *bus_id_map,
                                    const char *rule_file, char rule_temp[])
{
	char           rbuff[PATH_MAX + 1] = {'\0'};
	char           pbuff[PATH_MAX + 1] = {'\0'};
	unsigned int   replaced = 0;
	nc_udev_rule_t rule = NC_UDEV_RULE_INIT;
	FILE *       fpi;
	FILE *       fpo;
	int          fd;

	fpi = fopen(rule_file, "re");
	if(fpi == NULL) {
		if (errno != ENOENT) {
			nc_error("Unable to open %s: %m", rule_file);
			return -1;
		}
		return 0;
	}

	fd = mkstemp(rule_temp);
	if(fd == -1) {
		fclose(fpi);
		nc_error("Unable to create temp file %s: %m", rule_temp);
		return -1;
	}
	fpo = fdopen(fd, "we");
	if(fpo == NULL) {
		close(fd);
		fclose(fpi);
		nc_error("Unable to open %s for writing: %m", rule_temp);
		return -1;
	}

	while (fgets(rbuff, sizeof(rbuff), fpi) != NULL) {
		char               *lptr = pbuff;
		nc_udev_rule_cmd_t *name, *addr, *devid;
		nc_var_t           *v;
		unsigned int        i;

		memcpy(pbuff, rbuff, sizeof(pbuff));
		pbuff[strcspn(pbuff, "\r\n")] = '\0';

		while (isspace((unsigned char)*lptr))
			++lptr;

		if (*lptr == '#' || *lptr == '\0') {
	just_copy:
			fputs(rbuff, fpo);
			continue;
		}

		/* Don't fail but assume just we don't understand */
		if(__nc_udev_rule_parse(&rule, lptr) != 0)
			goto just_copy;

		/* Find the keys in the rule */
		name = NULL;
		addr = NULL;
		devid= NULL;
		for(i = rule.count; i-- > 0; ) {
			if(!name &&
			   nc_string_eq(rule.data[i]->key, "NAME") &&
			   (rule.data[i]->op == NC_UDEV_OP_SET ||
			    rule.data[i]->op == NC_UDEV_OP_FIN)) {
				name = rule.data[i];
				continue;
			}
			if(!addr &&
			   nc_string_eq(rule.data[i]->key, "ATTR{address}") &&
			   rule.data[i]->op == NC_UDEV_OP_EQ) {
				addr = rule.data[i];
				continue;
			}
			if(!devid &&
			   nc_string_eq(rule.data[i]->key, "ATTR{dev_id}") &&
			   rule.data[i]->op == NC_UDEV_OP_EQ) {
				devid = rule.data[i];
				continue;
			}
		}

		/*
		 * Replace addr by bus-id only, when devid has been found,
		 * because bus-id alone isn't sufficient for hardware with
		 * multiple port interfaces on one hw device (e.g mlx4).
		 * Take care, the rule matches dev_id (port number), too
		 * (or don't do any change).
		 */
		if(name && addr && devid &&
		   (v = nc_var_array_get(bus_id_map, name->val)) && v->value) {
			if(nc_string_dup(&addr->key, "KERNELS") < 0) {
				nc_udev_rule_destroy(&rule);
				goto just_copy;
			}
			if(nc_string_dup(&addr->val, v->value) < 0) {
				nc_udev_rule_destroy(&rule);
				goto just_copy;
			}

			// write modified rule
			for(i = 0; i < rule.count; ++i) {
				if(i)
					fputs(", ", fpo);
				fputs(rule.data[i]->key, fpo);
				fputs(nc_udev_rule_op_format(rule.data[i]->op), fpo);
				fputs("\"", fpo);
				fputs(rule.data[i]->val, fpo);
				fputs("\"", fpo);
			}
			fputs("\n", fpo);

			// mark this one done
			nc_string_free(&v->value);

			// drop the rule
			nc_udev_rule_destroy(&rule);

			// increase replaced counter
			replaced++;
		} else {
			nc_udev_rule_destroy(&rule);
			goto just_copy;
		}
	}
	nc_udev_rule_destroy(&rule);
	fclose(fpi);
	fflush(fpo);
	fclose(fpo);

	if(replaced == 0) {
		unlink(rule_temp);
	} else {
		if(rename(rule_temp, rule_file) != 0) {
			nc_error("Unable to rename modified '%s' to '%s': %m",
				rule_temp, rule_file);
			unlink(rule_temp);
			replaced = -1;
		}
	}
	return replaced;
}



/* public functions */
int
nc_udev_net_rule_hwaddr_for_name(const char *ifname, char **hwaddr)
{
	const char  *rule_file = UDEV_NET_NAME_RULE;
	char         rbuff[PATH_MAX + 1] = {'\0'};
	unsigned int line = 0;
	FILE *fp;

	fp = fopen(rule_file, "re");
	if(fp == NULL) {
		if (errno != ENOENT) {
			nc_error("Unable to open %s: %m", rule_file);
			return -1; /* error */
		}
		return 1; /* not found */
	}

	while (fgets(rbuff, sizeof(rbuff), fp) != NULL) {
		char *tok, *p  = NULL;
		char *addr_tok = NULL;
		char *name_tok = NULL;

		line++;
		rbuff[strcspn(rbuff, "#\r\n")] = '\0';

		for(tok = strtok_r(rbuff, ", \t", &p); tok; tok = strtok_r(NULL, ", \t", &p)) {
			if(!strncmp(tok, "ATTR{address}==", sizeof("ATTR{address}==")-1)) {
				tok += sizeof("ATTR{address}==")-1;
				addr_tok = __strip_quotes(tok);
				continue;
			}
			if(!strncmp(tok, "NAME:=", sizeof("NAME:=")-1)) {
				tok += sizeof("NAME:=")-1;
				name_tok = __strip_quotes(tok);
				break;
			}
			if(!strncmp(tok, "NAME=", sizeof("NAME=")-1)) {
				tok += sizeof("NAME=")-1;
				if(tok[0] == '=')
					continue;
				name_tok = __strip_quotes(tok);
				continue;
			}
		}

		if(name_tok && addr_tok && nc_string_eq(name_tok, ifname)) {
			/*
			** OK, that's it -- we have a match.
			*/
			fclose(fp);

			if(nc_string_dup(hwaddr, addr_tok) < 0)
				return -1; /* error */
			else
				return 0;  /* found & set */
		}
	}
	fclose(fp);
	return 1; /* not found */
}

int
nc_udev_net_rule_hwaddr_to_bus_id(nc_var_array_t *bus_id_map)
{
	const char * rule_file   = UDEV_NET_NAME_RULE;
	char         rule_temp[] = UDEV_NET_NAME_TEMP;
	int          ret;

	if(!bus_id_map)
		return -1;
	if(!bus_id_map->count)
		return 0;

	if(nc_udev_rule_file_lock(rule_file, UDEV_NET_NAME_SECS) != 0) {
		nc_error("Unable to acquire lock for %s: %m", rule_file);
		return 0; /* error, but we just report "not found" */
	}

	ret = __nc_udev_net_rule_hwaddr_to_bus_id(bus_id_map, rule_file, rule_temp);

	if(nc_udev_rule_file_unlock(rule_file) != 0) {
		nc_error("Unable to release lock for %s: %m", rule_file);
		/* Hmm... and now? */
	}
	return ret;
}

