/**********************************************************************
 * $realpow example -- PLI application using VPI routines
 *
 * C source to calculate the result of a number to the power of an
 * exponent. The result is returned as a double precision value.
 *
 * Usage: 
 *   real result;
 *   intiial
 *     result = $realpow(<base>,<exponent>);
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */
#include "veriuser.h"  /* IEEE 1364 PLI TF routine library    
                          (using TF routines for simulation control) */

/* prototypes of routines in this PLI application */
int PLIbook_RealPow_calltf(), PLIbook_RealPow_compiletf();

/**********************************************************************
 * $realpow_func Registration Data
 * (add this function name to the vlog_startup_array)
 *********************************************************************/
void PLIbook_RealPow_register()
{
  s_vpi_systf_data tf_data;
  tf_data.type        = vpiSysFunc;
  tf_data.sysfunctype = vpiSysFuncReal;
  tf_data.tfname      = "$realpow";
  tf_data.calltf      = PLIbook_RealPow_calltf;
  tf_data.compiletf   = PLIbook_RealPow_compiletf;
  tf_data.sizetf      = NULL; /*sizetf is not used for real functions*/
  vpi_register_systf(&tf_data);
}

/**********************************************************************
 * calltf to calculate base to power of exponent and return result.
 *********************************************************************/
#include <math.h>     /* ANSI C standard input/output library */
int PLIbook_RealPow_calltf(char *user_data)
{
  s_vpi_value value_s;
  vpiHandle   systf_handle, arg_itr, arg_handle;
  double      base, exp, result;

  value_s.format = vpiRealVal;

  /* obtain handle to system task arguments;
     compiletf has already verified only 2 args with correct types */
  systf_handle = vpi_handle(vpiSysTfCall, NULL);
  arg_itr = vpi_iterate(vpiArgument, systf_handle);

  /* read base value of system function arg 1 */
  arg_handle = vpi_scan(arg_itr);
  vpi_get_value(arg_handle, &value_s);
  base = value_s.value.real;

  /* read base value of system function arg 2 */
  arg_handle = vpi_scan(arg_itr);
  vpi_get_value(arg_handle, &value_s);
  exp = value_s.value.real;
  vpi_free_object(arg_itr); /* free iterator--did not scan till null */

  /* calculate result of base to power of exponent */
  result = pow(base,exp );

  /* write result to simulation as return value $realpow_func */
  value_s.value.real = result;
  vpi_put_value(systf_handle, &value_s, NULL, vpiNoDelay);
  return(0);
}

/**********************************************************************
 * compiletf application to verify valid systf args.
 *********************************************************************/
int PLIbook_RealPow_compiletf(char *user_data)
{
  vpiHandle systf_handle, arg_itr, arg_handle;
  int       tfarg_type;

  systf_handle = vpi_handle(vpiSysTfCall, NULL);
  arg_itr = vpi_iterate(vpiArgument, systf_handle);
  if (arg_itr == NULL) {
    vpi_printf("ERROR: $realpow requires 2 arguments\n");
    tf_dofinish();
    return(0);
  }
  arg_handle = vpi_scan(arg_itr);
  tfarg_type = vpi_get(vpiType, arg_handle);
  if ( (tfarg_type != vpiReg) &&
       (tfarg_type != vpiIntegerVar) &&
       (tfarg_type != vpiRealVar) &&
       (tfarg_type != vpiConstant)   ) {
    vpi_printf("ERR: $realpow arg1 must be number, variable or net\n");
    tf_dofinish();
    return(0);
  }
  arg_handle = vpi_scan(arg_itr);
  if (arg_handle == NULL) {
    vpi_printf("ERROR: $realpow requires 2nd argument\n");
    tf_dofinish();
    return(0);
  }
  tfarg_type = vpi_get(vpiType, arg_handle);
  if ( (tfarg_type != vpiReg) &&
       (tfarg_type != vpiIntegerVar) &&
       (tfarg_type != vpiRealVar) &&
       (tfarg_type != vpiConstant)   ) {
    vpi_printf("ERR: $realpow arg2 must be number, variable or net\n");
    tf_dofinish();
    return(0);
  }
  if (vpi_scan(arg_itr) != NULL) {
    vpi_printf("ERROR: $realpow requires only 2 arguments\n");
    vpi_free_object(arg_itr);
    tf_dofinish();
    return(0);
  }
}

/*********************************************************************/

