searchData={"items":[{"type":"module","title":"ExUnit","doc":"Unit testing framework for Elixir.","ref":"ExUnit.html"},{"type":"module","title":"Example - ExUnit","doc":"A basic setup for ExUnit is shown below:\n\n    # File: assertion_test.exs\n\n    # 1) Start ExUnit.\n    ExUnit.start()\n\n    # 2) Create a new test module (test case) and use \"ExUnit.Case\".\n    defmodule AssertionTest do\n      # 3) Note that we pass \"async: true\", this runs the test case\n      #    concurrently with other test cases. The individual tests\n      #    within each test case are still run serially.\n      use ExUnit.Case, async: true\n\n      # 4) Use the \"test\" macro instead of \"def\" for clarity.\n      test \"the truth\" do\n        assert true\n      end\n    end\n\nTo run the tests above, run the file using `elixir` from the\ncommand line. Assuming you named the file `assertion_test.exs`,\nyou can run it as:\n\n    $ elixir assertion_test.exs","ref":"ExUnit.html#module-example"},{"type":"module","title":"Case, Callbacks and Assertions - ExUnit","doc":"See `ExUnit.Case` and `ExUnit.Callbacks` for more information\nabout defining test cases and setting up callbacks.\n\nThe `ExUnit.Assertions` module contains a set of macros to\ngenerate assertions with appropriate error messages.","ref":"ExUnit.html#module-case-callbacks-and-assertions"},{"type":"module","title":"Integration with Mix - ExUnit","doc":"Mix is the project management and build tool for Elixir. Invoking `mix test`\nfrom the command line will run the tests in each file matching the pattern\n`*_test.exs` found in the `test` directory of your project.\n\nYou must create a `test_helper.exs` file inside the\n`test` directory and put the code common to all tests there.\n\nThe minimum example of a `test_helper.exs` file would be:\n\n    # test/test_helper.exs\n    ExUnit.start()\n\nMix will load the `test_helper.exs` file before executing the tests.\nIt is not necessary to `require` the `test_helper.exs` file in your test\nfiles. Run `mix help test` for more information.","ref":"ExUnit.html#module-integration-with-mix"},{"type":"function","title":"ExUnit.after_suite/1","doc":"Sets a callback to be executed after the completion of a test suite.\n\nCallbacks set with `after_suite/1` must accept a single argument, which is a\nmap containing the results of the test suite's execution.\n\nIf `after_suite/1` is called multiple times, the callbacks will be called in\nreverse order. In other words, the last callback set will be the first to be\ncalled.","ref":"ExUnit.html#after_suite/1"},{"type":"function","title":"ExUnit.async_run/0","doc":"Starts tests asynchronously while test cases are still loading.\n\nIt returns a task that must be given to `await_run/0` when a result\nis desired.","ref":"ExUnit.html#async_run/0"},{"type":"function","title":"ExUnit.await_run/1","doc":"Awaits for a test suite that has been started with `async_run/0`.","ref":"ExUnit.html#await_run/1"},{"type":"function","title":"ExUnit.configuration/0","doc":"Returns ExUnit configuration.\n\nFor the available configuration options, see `configure/1`.","ref":"ExUnit.html#configuration/0"},{"type":"function","title":"ExUnit.configure/1","doc":"Configures ExUnit.","ref":"ExUnit.html#configure/1"},{"type":"function","title":"Options - ExUnit.configure/1","doc":"ExUnit supports the following options:\n\n  * `:assert_receive_timeout` - the timeout to be used on `assert_receive`\n    calls in milliseconds, defaults to `100`;\n\n  * `:autorun` - if ExUnit should run by default on exit. Defaults to `true`;\n\n  * `:capture_log` - if ExUnit should default to keeping track of log messages\n    and print them on test failure. Can be overridden for individual tests via\n    `@tag capture_log: false`. Defaults to `false`;\n\n  * `:colors` - a keyword list of color options to be used by some formatters:\n    * `:enabled` - boolean option to enable colors, defaults to `IO.ANSI.enabled?/0`;\n\n    * `:success` - success message (defaults to `:green`)\n    * `:invalid` - invalid test message (defaults to `:yellow`)\n    * `:skipped` - skipped test message (defaults to `:yellow`)\n    * `:failure` - failed test message (defaults to `:red`)\n    * `:error_info` - display of actual error (defaults to `:red`)\n    * `:extra_info` - additional information (defaults to `:cyan`)\n    * `:location_info` - filename and tags (defaults to `[:bright, :black]`)\n    * `:diff_insert` - color of the insertions on diffs, defaults to `:green`;\n    * `:diff_insert_whitespace` - color of the whitespace insertions on diffs,\n      defaults to `IO.ANSI.color_background(2, 0, 0)`;\n    * `:diff_delete` - color of the deletions on diffs, defaults to `:red`;\n    * `:diff_delete_whitespace` - color of the whitespace deletions on diffs,\n      defaults to `IO.ANSI.color_background(0, 2, 0)`;\n\n  * `:exclude` - specifies which tests are run by skipping tests that match the\n    filter. See the \"Filters\" section in the documentation for `ExUnit.Case`;\n\n  * `:exit_status` - specifies an alternate exit status to use when the test\n    suite fails. Defaults to 2;\n\n  * `:failures_manifest_file` - specifies a path to the file used to store failures\n    between runs;\n\n  * `:formatters` - the formatters that will print results,\n    defaults to `[ExUnit.CLIFormatter]`;\n\n  * `:include` - specifies which tests are run by skipping tests that do not\n    match the filter. Keep in mind that all tests are included by default, so unless they are\n    excluded first, the `:include` option has no effect. To only run the tests\n    that match the `:include` filter, exclude the `:test` tag first (see the\n    documentation for `ExUnit.Case` for more information on tags and filters);\n\n  * `:max_cases` - maximum number of tests to run in parallel. Only tests from\n    different modules run in parallel. It defaults to `System.schedulers_online * 2`\n    to optimize both CPU-bound and IO-bound tests;\n\n  * `:max_failures` - the suite stops evaluating tests when this number of test failures\n    is reached. All tests within a module that fail when using the\n    [`setup_all/1,2`](`ExUnit.Callbacks.setup_all/1`) callbacks\n    are counted as failures. Defaults to `:infinity`;\n\n  * `:only_test_ids` - a list of `{module_name, test_name}` tuples that limits\n    what tests get run. This is typically used by Mix to filter which tests\n    should run;\n\n  * `:refute_receive_timeout` - the timeout to be used on `refute_receive`\n    calls in milliseconds, defaults to `100`;\n\n  * `:seed` - an integer seed value to randomize the test suite. This seed\n    is also mixed with the test module and name to create a new unique seed\n    on every test, which is automatically fed into the `:rand` module. This\n    provides randomness between tests, but predictable and reproducible\n    results. A `:seed` of `0` will disable randomization and the tests in each\n    file will always run in the order that they were defined in;\n\n  * `:slowest` - prints timing information for the N slowest tests. Running\n    ExUnit with slow test reporting automatically runs in `trace` mode. It\n    is disabled by default;\n\n  * `:stacktrace_depth` - configures the stacktrace depth to be used\n    on formatting and reporters, defaults to `20`;\n\n  * `:timeout` - sets the timeout for the tests in milliseconds, defaults to `60_000`;\n\n  * `:trace` - sets ExUnit into trace mode, this sets `:max_cases` to `1` and\n    prints each test case and test while running. Note that in trace mode test\n    timeouts will be ignored as timeout is set to `:infinity`;\n\n  * `:test_location_relative_path` - the test location is the file:line information\n    printed by tests as a shortcut to run a given test. When this value is set,\n    the value is used as a prefix for the test itself. This is typically used by\n    Mix to properly set-up umbrella projects;\n\nAny arbitrary configuration can also be passed to `configure/1` or `start/1`,\nand these options can then be used in places such as custom formatters. These\nother options will be ignored by ExUnit itself.","ref":"ExUnit.html#configure/1-options"},{"type":"function","title":"ExUnit.fetch_test_supervisor/0","doc":"Fetches the test supervisor for the current test.\n\nReturns `{:ok, supervisor_pid}` or `:error` if not called from the test process.\n\nThis is the same supervisor as used by `ExUnit.Callbacks.start_supervised/2`\nand similar, see `ExUnit.Callbacks` module documentation for more information.","ref":"ExUnit.html#fetch_test_supervisor/0"},{"type":"function","title":"ExUnit.plural_rule/1","doc":"Returns the pluralization for `word`.\n\nIf one is not registered, returns the word appended with an \"s\".","ref":"ExUnit.html#plural_rule/1"},{"type":"function","title":"ExUnit.plural_rule/2","doc":"Registers a `pluralization` for `word`.\n\nIf one is already registered, it is replaced.","ref":"ExUnit.html#plural_rule/2"},{"type":"function","title":"ExUnit.run/1","doc":"Runs the tests. It is invoked automatically\nif ExUnit is started via `start/1`.\n\nFrom Elixir v1.14, it accepts an optional list of modules to run\nas part of the suite. This is often used to rerun modules already\nloaded in memory.\n\nReturns a map containing the total number of tests, the number\nof failures, the number of excluded tests and the number of skipped tests.","ref":"ExUnit.html#run/1"},{"type":"function","title":"ExUnit.start/1","doc":"Starts ExUnit and automatically runs tests right before the\nVM terminates.\n\nIt accepts a set of `options` to configure `ExUnit`\n(the same ones accepted by `configure/1`).\n\nIf you want to run tests manually, you can set the `:autorun` option\nto `false` and use `run/0` to run tests.","ref":"ExUnit.html#start/1"},{"type":"type","title":"ExUnit.failed/0","doc":"The error state returned by `ExUnit.Test` and `ExUnit.TestModule`","ref":"ExUnit.html#t:failed/0"},{"type":"type","title":"ExUnit.state/0","doc":"All tests start with a state of `nil`.\n\nA finished test can be in one of five states:\n\n  1. Passed (also represented by `nil`)\n  2. Failed\n  3. Skipped (via @tag :skip)\n  4. Excluded (via :exclude filters)\n  5. Invalid (when setup_all fails)","ref":"ExUnit.html#t:state/0"},{"type":"type","title":"ExUnit.suite_result/0","doc":"A map representing the results of running a test suite","ref":"ExUnit.html#t:suite_result/0"},{"type":"type","title":"ExUnit.test_id/0","doc":"","ref":"ExUnit.html#t:test_id/0"},{"type":"module","title":"ExUnit.Assertions","doc":"This module contains a set of assertion functions that are\nimported by default into your test cases.\n\nIn general, a developer will want to use the general\n`assert` macro in tests. This macro introspects your code\nand provides good reporting whenever there is a failure.\nFor example, `assert some_fun() == 10` will fail (assuming\n`some_fun()` returns `13`):\n\n    Comparison (using ==) failed in:\n    code:  assert some_fun() == 10\n    left:  13\n    right: 10\n\nThis module also provides other convenience functions\nlike `assert_in_delta` and `assert_raise` to easily handle\nother common cases such as checking a floating-point number\nor handling exceptions.","ref":"ExUnit.Assertions.html"},{"type":"macro","title":"ExUnit.Assertions.assert/1","doc":"Asserts its argument is a truthy value.\n\n`assert` introspects the underlying expression and provides\ngood reporting whenever there is a failure. For example,\nif the expression uses the comparison operator, the message\nwill show the values of the two sides. The assertion\n\n    assert 1 + 2 + 3 + 4 > 15\n\n will fail with the message:\n\n    Assertion with > failed\n    code:  assert 1 + 2 + 3 + 4 > 15\n    left:  10\n    right: 15\n\nSimilarly, if a match expression is given, it will report\nany failure in terms of that match. Given\n\n    assert [1] = [2]\n\nyou'll see:\n\n    match (=) failed\n    code:  assert [1] = [2]\n    left: [1]\n    right: [2]\n\nKeep in mind that `assert` does not change its semantics\nbased on the expression. In other words, the expression\nis still required to return a truthy value. For example,\nthe following will fail:\n\n    assert nil = some_function_that_returns_nil()\n\nEven though the match works, `assert` still expects a truth\nvalue. In such cases, simply use `==/2` or `match?/2`.","ref":"ExUnit.Assertions.html#assert/1"},{"type":"function","title":"ExUnit.Assertions.assert/2","doc":"Asserts `value` is truthy, displaying the given `message` otherwise.","ref":"ExUnit.Assertions.html#assert/2"},{"type":"function","title":"Examples - ExUnit.Assertions.assert/2","doc":"assert false, \"it will never be true\"\n\n    assert x == :foo, \"expected x to be foo\"\n\n    assert match?({:ok, _}, x), \"expected x to match {:ok, _}\"","ref":"ExUnit.Assertions.html#assert/2-examples"},{"type":"function","title":"ExUnit.Assertions.assert_in_delta/4","doc":"Asserts that `value1` and `value2` differ by no more than `delta`.\n\nThis difference is inclusive, so the test will pass if the difference\nand the `delta` are equal.","ref":"ExUnit.Assertions.html#assert_in_delta/4"},{"type":"function","title":"Examples - ExUnit.Assertions.assert_in_delta/4","doc":"assert_in_delta 1.1, 1.5, 0.2\n    assert_in_delta 10, 15, 2\n    assert_in_delta 10, 15, 5","ref":"ExUnit.Assertions.html#assert_in_delta/4-examples"},{"type":"function","title":"ExUnit.Assertions.assert_raise/2","doc":"Asserts the `exception` is raised during `function` execution.\nReturns the rescued exception, fails otherwise.","ref":"ExUnit.Assertions.html#assert_raise/2"},{"type":"function","title":"Examples - ExUnit.Assertions.assert_raise/2","doc":"assert_raise ArithmeticError, fn ->\n      1 + \"test\"\n    end\n\n    assert_raise RuntimeError, fn ->\n      raise \"assertion will pass due to this raise\"\n    end","ref":"ExUnit.Assertions.html#assert_raise/2-examples"},{"type":"function","title":"ExUnit.Assertions.assert_raise/3","doc":"Asserts the `exception` is raised during `function` execution with\nthe expected `message`, which can be a `Regex` or an exact `String`.\nReturns the rescued exception, fails otherwise.","ref":"ExUnit.Assertions.html#assert_raise/3"},{"type":"function","title":"Examples - ExUnit.Assertions.assert_raise/3","doc":"assert_raise ArithmeticError, \"bad argument in arithmetic expression\", fn ->\n      1 + \"test\"\n    end\n\n    assert_raise RuntimeError, ~r/^today's lucky number is 0\\.\\d+!$/, fn ->\n      raise \"today's lucky number is #{:rand.uniform()}!\"\n    end","ref":"ExUnit.Assertions.html#assert_raise/3-examples"},{"type":"macro","title":"ExUnit.Assertions.assert_receive/3","doc":"Asserts that a message matching `pattern` was or is going to be received\nwithin the `timeout` period, specified in milliseconds.\n\nUnlike `assert_received`, it has a default `timeout`\nof 100 milliseconds.\n\nThe `pattern` argument must be a match pattern. Flunks with `failure_message`\nif a message matching `pattern` is not received.","ref":"ExUnit.Assertions.html#assert_receive/3"},{"type":"macro","title":"Examples - ExUnit.Assertions.assert_receive/3","doc":"assert_receive :hello\n\nAsserts against a larger timeout:\n\n    assert_receive :hello, 20_000\n\nYou can also match against specific patterns:\n\n    assert_receive {:hello, _}\n\n    x = 5\n    assert_receive {:count, ^x}","ref":"ExUnit.Assertions.html#assert_receive/3-examples"},{"type":"macro","title":"ExUnit.Assertions.assert_received/2","doc":"Asserts that a message matching `pattern` was received and is in the\ncurrent process' mailbox.\n\nThe `pattern` argument must be a match pattern. Flunks with `failure_message`\nif a message matching `pattern` was not received.\n\nTimeout is set to 0, so there is no waiting time.","ref":"ExUnit.Assertions.html#assert_received/2"},{"type":"macro","title":"Examples - ExUnit.Assertions.assert_received/2","doc":"send(self(), :hello)\n    assert_received :hello\n\n    send(self(), :bye)\n    assert_received :hello, \"Oh No!\"\n    ** (ExUnit.AssertionError) Oh No!\n\nYou can also match against specific patterns:\n\n    send(self(), {:hello, \"world\"})\n    assert_received {:hello, _}","ref":"ExUnit.Assertions.html#assert_received/2-examples"},{"type":"macro","title":"ExUnit.Assertions.catch_error/1","doc":"Asserts `expression` will cause an error.\n\nReturns the error or fails otherwise.","ref":"ExUnit.Assertions.html#catch_error/1"},{"type":"macro","title":"Examples - ExUnit.Assertions.catch_error/1","doc":"assert catch_error(error(1)) == 1","ref":"ExUnit.Assertions.html#catch_error/1-examples"},{"type":"macro","title":"ExUnit.Assertions.catch_exit/1","doc":"Asserts `expression` will exit.\n\nReturns the exit status/message of the current process or fails otherwise.","ref":"ExUnit.Assertions.html#catch_exit/1"},{"type":"macro","title":"Examples - ExUnit.Assertions.catch_exit/1","doc":"assert catch_exit(exit(1)) == 1\n\nTo assert exits from linked processes started from the test, trap exits\nwith `Process.flag/2` and assert the exit message with `assert_receive/2`.\n\n    Process.flag(:trap_exit, true)\n    pid = spawn_link(fn -> Process.exit(self(), :normal) end)\n    assert_receive {:EXIT, ^pid, :normal}","ref":"ExUnit.Assertions.html#catch_exit/1-examples"},{"type":"macro","title":"ExUnit.Assertions.catch_throw/1","doc":"Asserts `expression` will throw a value.\n\nReturns the thrown value or fails otherwise.","ref":"ExUnit.Assertions.html#catch_throw/1"},{"type":"macro","title":"Examples - ExUnit.Assertions.catch_throw/1","doc":"assert catch_throw(throw(1)) == 1","ref":"ExUnit.Assertions.html#catch_throw/1-examples"},{"type":"function","title":"ExUnit.Assertions.flunk/1","doc":"Fails with a message.","ref":"ExUnit.Assertions.html#flunk/1"},{"type":"function","title":"Examples - ExUnit.Assertions.flunk/1","doc":"flunk(\"This should raise an error\")","ref":"ExUnit.Assertions.html#flunk/1-examples"},{"type":"macro","title":"ExUnit.Assertions.refute/1","doc":"A negative assertion, expects the expression to be `false` or `nil`.\n\nKeep in mind that `refute` does not change the semantics of\nthe given expression. In other words, the following will fail:\n\n    refute {:ok, _} = some_function_that_returns_error_tuple()\n\nThe code above will fail because the `=` operator always fails\nwhen the sides do not match and `refute/2` does not change it.\n\nThe correct way to write the refutation above is to use `match?/2`:\n\n    refute match?({:ok, _}, some_function_that_returns_error_tuple())","ref":"ExUnit.Assertions.html#refute/1"},{"type":"macro","title":"Examples - ExUnit.Assertions.refute/1","doc":"refute age < 0","ref":"ExUnit.Assertions.html#refute/1-examples"},{"type":"function","title":"ExUnit.Assertions.refute/2","doc":"Asserts `value` is `nil` or `false` (that is, `value` is not truthy).","ref":"ExUnit.Assertions.html#refute/2"},{"type":"function","title":"Examples - ExUnit.Assertions.refute/2","doc":"refute true, \"This will obviously fail\"","ref":"ExUnit.Assertions.html#refute/2-examples"},{"type":"function","title":"ExUnit.Assertions.refute_in_delta/4","doc":"Asserts `value1` and `value2` are not within `delta`.\n\nThis difference is exclusive, so the test will fail if the difference\nand the delta are equal.\n\nIf you supply `message`, information about the values will\nautomatically be appended to it.","ref":"ExUnit.Assertions.html#refute_in_delta/4"},{"type":"function","title":"Examples - ExUnit.Assertions.refute_in_delta/4","doc":"refute_in_delta 1.1, 1.2, 0.2\n    refute_in_delta 10, 11, 2","ref":"ExUnit.Assertions.html#refute_in_delta/4-examples"},{"type":"macro","title":"ExUnit.Assertions.refute_receive/3","doc":"Asserts that a message matching `pattern` was not received (and won't be received)\nwithin the `timeout` period, specified in milliseconds.\n\nThe `pattern` argument must be a match pattern. Flunks with `failure_message`\nif a message matching `pattern` is received.","ref":"ExUnit.Assertions.html#refute_receive/3"},{"type":"macro","title":"Examples - ExUnit.Assertions.refute_receive/3","doc":"refute_receive :bye\n\nRefute received with an explicit timeout:\n\n    refute_receive :bye, 1000","ref":"ExUnit.Assertions.html#refute_receive/3-examples"},{"type":"macro","title":"ExUnit.Assertions.refute_received/2","doc":"Asserts a message matching `pattern` was not received (i.e. it is not in the\ncurrent process' mailbox).\n\nThe `pattern` argument must be a match pattern. Flunks with `failure_message`\nif a message matching `pattern` was received.\n\nTimeout is set to 0, so there is no waiting time.","ref":"ExUnit.Assertions.html#refute_received/2"},{"type":"macro","title":"Examples - ExUnit.Assertions.refute_received/2","doc":"send(self(), :hello)\n    refute_received :bye\n\n    send(self(), :hello)\n    refute_received :hello, \"Oh No!\"\n    ** (ExUnit.AssertionError) Oh No!","ref":"ExUnit.Assertions.html#refute_received/2-examples"},{"type":"module","title":"ExUnit.Callbacks","doc":"Defines ExUnit callbacks.\n\nThis module defines the `setup/1`, `setup/2`, `setup_all/1`, and\n`setup_all/2` callbacks, as well as the `on_exit/2`, `start_supervised/2`\nand `stop_supervised/1` functions.\n\nThe setup callbacks may be used to define [test fixtures](https://en.wikipedia.org/wiki/Test_fixture#Software)\nand run any initialization code which help bring the system into a known\nstate. They are defined via macros and each one can optionally receive a map\nwith test state and metadata, usually referred to as the `context`.\nOptionally, the context to be used in the tests can be extended by the\nsetup callbacks by returning a properly structured value (see below).\n\nThe `setup_all` callbacks are invoked only once per module, before any\ntest is run. All `setup` callbacks are run before each test. No callback\nis run if the test case has no tests or all tests have been filtered out.\n\n`setup` and `setup_all` callbacks can be defined by either a block, an atom\nnaming a local function, a `{module, function}` tuple, or a list of atoms/tuples.\n\nBoth can opt to receive the current context by specifying it\nas parameter if defined by a block. Functions used to define a test\nsetup must accept the context as single argument.\n\nA test module can define multiple `setup` and `setup_all` callbacks,\nand they are invoked in order of appearance.\n\n`start_supervised/2` is used to start processes under a supervisor. The\nsupervisor is linked to the current test process. The supervisor as well\nas all child processes are guaranteed to terminate before any `on_exit/2`\ncallback runs.\n\n`on_exit/2` callbacks are registered on demand, usually to undo an action\nperformed by a setup callback. `on_exit/2` may also take a reference,\nallowing the callback to be overridden in the future. A registered `on_exit/2`\ncallback will always run, while failures in `setup` and `setup_all` will stop\nall remaining setup callbacks from executing.\n\nFinally, `setup_all` callbacks run in a separate process per module, while\nall `setup` callbacks run in the same process as the test itself. `on_exit/2`\ncallbacks always run in a separate process, as implied by their name. The\ntest process always exits with reason `:shutdown`, which means any process\nlinked to the test process will also exit, although asynchronously. Therefore\nit is preferred to use `start_supervised/2` to guarantee synchronous termination.\n\nHere is a rundown of the life-cycle of the test process:\n\n  1. the test process is spawned\n  2. it runs `setup/2` callbacks\n  3. it runs the test itself\n  4. it stops all supervised processes\n  5. the test process exits with reason `:shutdown`\n  6. `on_exit/2` callbacks are executed in a separate process","ref":"ExUnit.Callbacks.html"},{"type":"module","title":"Context - ExUnit.Callbacks","doc":"If `setup_all` or `setup` return a keyword list, a map, or a tuple in the shape\nof `{:ok, keyword() | map()}`, the keyword list or map will be merged into the\ncurrent context and will be available in all subsequent `setup_all`,\n`setup`, and the `test` itself.\n\nReturning `:ok` leaves the context unchanged (in `setup` and `setup_all`\ncallbacks).\n\nReturning anything else from `setup_all` will force all tests to fail,\nwhile a bad response from `setup` causes the current test to fail.","ref":"ExUnit.Callbacks.html#module-context"},{"type":"module","title":"Examples - ExUnit.Callbacks","doc":"defmodule AssertionTest do\n      use ExUnit.Case, async: true\n\n      # \"setup_all\" is called once per module before any test runs\n      setup_all do\n        IO.puts(\"Starting AssertionTest\")\n\n        # Context is not updated here\n        :ok\n      end\n\n      # \"setup\" is called before each test\n      setup do\n        IO.puts(\"This is a setup callback for #{inspect(self())}\")\n\n        on_exit(fn ->\n          IO.puts(\"This is invoked once the test is done. Process: #{inspect(self())}\")\n        end)\n\n        # Returns extra metadata to be merged into context.\n        # Any of the following would also work:\n        #\n        #     {:ok, %{hello: \"world\"}}\n        #     {:ok, [hello: \"world\"]}\n        #     %{hello: \"world\"}\n        #\n        [hello: \"world\"]\n      end\n\n      # Same as above, but receives the context as argument\n      setup context do\n        IO.puts(\"Setting up: #{context.test}\")\n\n        # We can simply return :ok when we don't want to add any extra metadata\n        :ok\n      end\n\n      # Setups can also invoke a local or imported function that returns a context\n      setup :invoke_local_or_imported_function\n\n      test \"always pass\" do\n        assert true\n      end\n\n      test \"uses metadata from setup\", context do\n        assert context[:hello] == \"world\"\n        assert context[:from_named_setup] == true\n      end\n\n      defp invoke_local_or_imported_function(context) do\n        [from_named_setup: true]\n      end\n    end\n\nIt is also common to define your setup as a series of functions,\nwhich are put together by calling `setup` or `setup_all` with a\nlist of function names. Each of these functions receive the context and can\nreturn any of the values allowed in `setup` blocks:\n\n    defmodule ExampleContextTest do\n      use ExUnit.Case\n\n      setup [:step1, :step2, :step3, {OtherModule, :step4}]\n\n      defp step1(_context), do: [step_one: true]\n      defp step2(_context), do: {:ok, step_two: true} # return values with shape of {:ok, keyword() | map()} allowed\n      defp step3(_context), do: :ok # Context not modified\n\n      test \"context was modified\", context do\n        assert context[:step_one] == true\n        assert context[:step_two] == true\n      end\n    end\n\nFinally, as discussed in the `ExUnit.Case` documentation, remember\nthat the initial context metadata can also be set via `@tag`s, which\ncan then be accessed in the `setup` block:\n\n    defmodule ExampleTagModificationTest do\n      use ExUnit.Case\n\n      setup %{login_as: username} do\n        {:ok, current_user: username}\n      end\n\n      @tag login_as: \"max\"\n      test \"tags modify context\", context do\n        assert context[:login_as] == \"max\"\n        assert context[:current_user] == \"max\"\n      end\n    end","ref":"ExUnit.Callbacks.html#module-examples"},{"type":"function","title":"ExUnit.Callbacks.on_exit/2","doc":"Registers a callback that runs once the test exits.\n\n`callback` is a function that receives no arguments and\nruns in a separate process than the caller. Its return\nvalue is irrelevant and is discarded.\n\n`on_exit/2` is usually called from `setup/1` and `setup_all/1`\ncallbacks, often to undo the action performed during the setup.\n\nHowever, `on_exit/2` may also be called dynamically. An \"ID\" (the\n`name_or_ref` argument) can be used to guarantee that the callback\nwill be invoked only once. ExUnit uses this term to identify an\n`on_exit/2` handler: if you want to override a previous handler, for\nexample, use the same `name_or_ref` across multiple `on_exit/2`\ncalls.\n\nIf `on_exit/2` is called inside `setup/1` or inside a test, it's\nexecuted in a blocking fashion after the test exits and *before\nrunning the next test*. This means that no other test from the same\ntest case will be running while the `on_exit/2` callback for a\nprevious test is running. `on_exit/2` is executed in a different\nprocess than the test process. On the other hand, if `on_exit/2` is\ncalled inside a `setup_all/1` callback then `callback` is executed\nafter running *all tests* (see `setup_all/1` for more information).","ref":"ExUnit.Callbacks.html#on_exit/2"},{"type":"function","title":"Examples - ExUnit.Callbacks.on_exit/2","doc":"setup do\n      File.write!(\"fixture.json\", \"{}\")\n      on_exit(fn -> File.rm!(\"fixture.json\") end)\n    end\n\nYou can use the same `name_or_ref` across multiple `on_exit/2` calls\nto \"override\" the registered handler:\n\n    setup do\n      on_exit(:drop_table, fn ->\n        Database.drop_table()\n      end)\n    end\n\n    test \"a test that shouldn't drop the table\" do\n      on_exit(:drop_table, fn -> :ok end)\n    end\n\nRelying too much on overriding callbacks like this can lead to test\ncases that are hard to understand and with too many layers of\nindirection. However, it can be useful in some cases or for library\nauthors, for example.","ref":"ExUnit.Callbacks.html#on_exit/2-examples"},{"type":"macro","title":"ExUnit.Callbacks.setup/1","doc":"Defines a callback to be run before each test in a case.\n\nAccepts one of these:\n\n  * a block\n  * an atom naming a local function\n  * a `{module, function}` tuple\n  * a list of atoms and `{module, function}` tuples\n\nCan return values to be merged into the context, to set up the state for\ntests. For more details, see the \"Context\" section shown above.\n\n`setup/1` callbacks are executed in the same process as the test process.","ref":"ExUnit.Callbacks.html#setup/1"},{"type":"macro","title":"Examples - ExUnit.Callbacks.setup/1","doc":"defp clean_up_tmp_directory(context) do\n      # perform setup\n      :ok\n    end\n\n    setup :clean_up_tmp_directory\n\n    setup [:clean_up_tmp_directory, :another_setup]\n\n    setup do\n      [conn: Plug.Conn.build_conn()]\n    end\n\n    setup {MyModule, :my_setup_function}","ref":"ExUnit.Callbacks.html#setup/1-examples"},{"type":"macro","title":"ExUnit.Callbacks.setup/2","doc":"Defines a callback to be run before each test in a case.\n\nThis is similar to `setup/1`, but the first argument is the context.\nThe `block` argument can only be a block.\n\nFor more details, see the \"Context\" section shown above.","ref":"ExUnit.Callbacks.html#setup/2"},{"type":"macro","title":"Examples - ExUnit.Callbacks.setup/2","doc":"setup context do\n      [conn: Plug.Conn.build_conn()]\n    end","ref":"ExUnit.Callbacks.html#setup/2-examples"},{"type":"macro","title":"ExUnit.Callbacks.setup_all/1","doc":"Defines a callback to be run before all tests in a case.\n\nAccepts one of these:\n\n  * a block\n  * an atom naming a local function\n  * a `{module, function}` tuple\n  * a list of atoms and `{module, function}` tuples\n\nCan return values to be merged into the `context`, to set up the state for\ntests. For more details, see the \"Context\" section shown above.\n\n`setup_all/1` callbacks are executed in a separate process than tests.\nAll `setup_all/1` callbacks are executed in order in the same process.","ref":"ExUnit.Callbacks.html#setup_all/1"},{"type":"macro","title":"On-Exit Handlers - ExUnit.Callbacks.setup_all/1","doc":"On-exit handlers that you register inside `setup_all/1` callbacks\nare executed at once after all tests in the module have been run.\nThey are all executed *in the same process*, which is a separate\nprocess dedicated to running these handlers. These handlers are\nexecuted in the reverse order of their respective `setup_all/1`\ncallbacks.","ref":"ExUnit.Callbacks.html#setup_all/1-on-exit-handlers"},{"type":"macro","title":"Examples - ExUnit.Callbacks.setup_all/1","doc":"# One-arity function name\n    setup_all :clean_up_tmp_directory\n\n    # A module and function\n    setup_all {MyModule, :my_setup_function}\n\n    # A list of one-arity functions and module/function tuples\n    setup_all [:clean_up_tmp_directory, {MyModule, :my_setup_function}]\n\n    defp clean_up_tmp_directory(_context) do\n      # perform setup\n      :ok\n    end\n\n    # A block\n    setup_all do\n      [conn: Plug.Conn.build_conn()]\n    end\n\nThe context returned by `setup_all/1` will be available in all subsequent\n`setup_all`, `setup`, and the `test` itself. For instance, the `conn` from\nthe previous example can be accessed as:\n\n    test \"fetches current users\", %{conn: conn} do\n      # ...\n    end\n\n#","ref":"ExUnit.Callbacks.html#setup_all/1-examples"},{"type":"macro","title":"Handlers - ExUnit.Callbacks.setup_all/1","doc":"You can define \"global\" on-exit handlers in `setup_all/1` callbacks:\n\n    setup_all do\n      Database.create_table_for(__MODULE__)\n\n      on_exit(fn ->\n        Database.drop_table_for(__MODULE__)\n      end)\n\n      :ok\n    end\n\nThe handler in the example above will be executed only once, after\nrunning all tests in the module.","ref":"ExUnit.Callbacks.html#setup_all/1-handlers"},{"type":"macro","title":"ExUnit.Callbacks.setup_all/2","doc":"Defines a callback to be run before all tests in a case.\n\nSimilar as `setup_all/1` but also takes a context. The second argument\nmust be a block. See the \"Context\" section in the module documentation.","ref":"ExUnit.Callbacks.html#setup_all/2"},{"type":"macro","title":"Examples - ExUnit.Callbacks.setup_all/2","doc":"setup_all _context do\n      [conn: Plug.Conn.build_conn()]\n    end","ref":"ExUnit.Callbacks.html#setup_all/2-examples"},{"type":"function","title":"ExUnit.Callbacks.start_link_supervised!/2","doc":"Same as `start_supervised!/2` but links the started process to the test process.\n\nIf the process that was started crashes, the crash is propagated to the test process,\nfailing the test and printing the cause of the crash.\n\nNote that if the started terminates before it is linked to the test process,\nthis function will exit with reason `:noproc`.","ref":"ExUnit.Callbacks.html#start_link_supervised!/2"},{"type":"function","title":"ExUnit.Callbacks.start_supervised/2","doc":"Starts a child process under the test supervisor.\n\nIt expects a child specification or a module, similar to the ones\ngiven to `Supervisor.start_link/2`. For example, if your application\nstarts a supervision tree by running:\n\n    Supervisor.start_link([MyServer, {OtherSupervisor, ...}], ...)\n\nYou can start those processes under test in isolation by running:\n\n    start_supervised(MyServer)\n    start_supervised({OtherSupervisor, :initial_value})\n\nA keyword list can also be given if there is a need to change\nthe child specification for the given child process:\n\n    start_supervised({MyServer, :initial_value}, restart: :temporary)\n\nSee the `Supervisor` module for a discussion on child specifications\nand the available specification keys.\n\nThe advantage of starting a process under the test supervisor is that\nit is guaranteed to exit before the next test starts. Therefore, you\ndon't need to remove the process at the end of your tests via\n`stop_supervised/1`. You only need to use `stop_supervised/1 ` if you\nwant to remove a process from the supervision tree in the middle of a\ntest, as simply shutting down the process would cause it to be restarted\naccording to its `:restart` value.\n\nThe started process is not linked to the test process and a crash will\nnot necessarily fail the test. To start and link a process to guarantee\nthat any crash would also fail the test use `start_link_supervised!/2`.\n\nThis function returns `{:ok, pid}` in case of success, otherwise it\nreturns `{:error, reason}`.","ref":"ExUnit.Callbacks.html#start_supervised/2"},{"type":"function","title":"ExUnit.Callbacks.start_supervised!/2","doc":"Same as `start_supervised/2` but returns the PID on success and raises if\nnot started properly.","ref":"ExUnit.Callbacks.html#start_supervised!/2"},{"type":"function","title":"ExUnit.Callbacks.stop_supervised/1","doc":"Stops a child process started via `start_supervised/2`.\n\nThis function expects the `id` in the child specification.\nFor example:\n\n    {:ok, _} = start_supervised(MyServer)\n    :ok = stop_supervised(MyServer)\n\nIt returns `:ok` if there is a supervised process with such\n`id`, `{:error, :not_found}` otherwise.","ref":"ExUnit.Callbacks.html#stop_supervised/1"},{"type":"function","title":"ExUnit.Callbacks.stop_supervised!/1","doc":"Same as `stop_supervised/1` but raises if it cannot be stopped.","ref":"ExUnit.Callbacks.html#stop_supervised!/1"},{"type":"module","title":"ExUnit.CaptureIO","doc":"Functionality to capture IO for testing.","ref":"ExUnit.CaptureIO.html"},{"type":"module","title":"Examples - ExUnit.CaptureIO","doc":"defmodule AssertionTest do\n      use ExUnit.Case\n\n      import ExUnit.CaptureIO\n\n      test \"example\" do\n        assert capture_io(fn -> IO.puts(\"a\") end) == \"a\\n\"\n      end\n\n      test \"another example\" do\n        assert with_io(fn ->\n          IO.puts(\"a\")\n          IO.puts(\"b\")\n          2 + 2\n        end) == {4, \"a\\nb\\n\"}\n      end\n    end","ref":"ExUnit.CaptureIO.html#module-examples"},{"type":"function","title":"ExUnit.CaptureIO.capture_io/1","doc":"Captures IO generated when evaluating `fun`.\n\nReturns the binary which is the captured output.\n\nBy default, `capture_io` replaces the `group_leader` (`:stdio`)\nfor the current process. Capturing the group leader is done per\nprocess and therefore can be done concurrently.\n\nHowever, the capturing of any other named device, such as `:stderr`,\nhappens globally and persists until the function has ended. While this means\nit is safe to run your tests with `async: true` in many cases, captured output\nmay include output from a different test and care must be taken when using\n`capture_io` with a named process asynchronously.\n\nA developer can set a string as an input. The default input is an empty\nstring. If capturing a named device asynchronously, an input can only be given\nto the first capture. Any further capture that is given to a capture on that\ndevice will raise an exception and would indicate that the test should be run\nsynchronously.\n\nSimilarly, once a capture on a named device has begun, the encoding on that\ndevice cannot be changed in a subsequent concurrent capture. An error will\nbe raised in this case.","ref":"ExUnit.CaptureIO.html#capture_io/1"},{"type":"function","title":"IO devices - ExUnit.CaptureIO.capture_io/1","doc":"You may capture the IO from any registered IO device. The device name given\nmust be an atom representing the name of a registered process. In addition,\nElixir provides two shortcuts:\n\n  * `:stdio` - a shortcut for `:standard_io`, which maps to\n    the current `Process.group_leader/0` in Erlang\n\n  * `:stderr` - a shortcut for the named process `:standard_error`\n    provided in Erlang","ref":"ExUnit.CaptureIO.html#capture_io/1-io-devices"},{"type":"function","title":"Options - ExUnit.CaptureIO.capture_io/1","doc":"* `:input` - An input to the IO device, defaults to `\"\"`.\n\n  * `:capture_prompt` - Define if prompts (specified as arguments to\n    `IO.get*` functions) should be captured. Defaults to `true`. For\n    IO devices other than `:stdio`, the option is ignored.\n\n  * `:encoding` (since v1.10.0) - encoding of the IO device. Allowed\n    values are `:unicode` (default) and `:latin1`.","ref":"ExUnit.CaptureIO.html#capture_io/1-options"},{"type":"function","title":"Examples - ExUnit.CaptureIO.capture_io/1","doc":"To capture the standard io:\n\n    iex> capture_io(fn -> IO.write(\"john\") end) == \"john\"\n    true\n\n    iex> capture_io(\"this is input\", fn ->\n    ...>   input = IO.gets(\"> \")\n    ...>   IO.write(input)\n    ...> end) == \"> this is input\"\n    true\n\n    iex> capture_io([input: \"this is input\", capture_prompt: false], fn ->\n    ...>   input = IO.gets(\"> \")\n    ...>   IO.write(input)\n    ...> end) == \"this is input\"\n    true\n\nNote it is fine to use `==` with standard IO, because the content is captured\nper test process. However, `:stderr` is shared across all tests, so you will\nwant to use `=~` instead of `==` for assertions on `:stderr` if your tests\nare async:\n\n    iex> capture_io(:stderr, fn -> IO.write(:stderr, \"john\") end) =~ \"john\"\n    true\n\n    iex> capture_io(:standard_error, fn -> IO.write(:stderr, \"john\") end) =~ \"john\"\n    true\n\nIn particular, avoid empty captures on `:stderr` with async tests:\n\n    iex> capture_io(:stderr, fn -> :nothing end) == \"\"\n    true\n\nOtherwise, if the standard error of any other test is captured, the test will\nfail.","ref":"ExUnit.CaptureIO.html#capture_io/1-examples"},{"type":"function","title":"Returning values - ExUnit.CaptureIO.capture_io/1","doc":"As seen in the examples above, `capture_io` returns the captured output.\nIf you want to also capture the result of the function executed,\nuse `with_io/2`.","ref":"ExUnit.CaptureIO.html#capture_io/1-returning-values"},{"type":"function","title":"ExUnit.CaptureIO.capture_io/2","doc":"Captures IO generated when evaluating `fun`.\n\nSee `capture_io/1` for more information.","ref":"ExUnit.CaptureIO.html#capture_io/2"},{"type":"function","title":"ExUnit.CaptureIO.capture_io/3","doc":"Captures IO generated when evaluating `fun`.\n\nSee `capture_io/1` for more information.","ref":"ExUnit.CaptureIO.html#capture_io/3"},{"type":"function","title":"ExUnit.CaptureIO.with_io/1","doc":"Invokes the given `fun` and returns the result and captured output.\n\nIt accepts the same arguments and options as `capture_io/1`.","ref":"ExUnit.CaptureIO.html#with_io/1"},{"type":"function","title":"Examples - ExUnit.CaptureIO.with_io/1","doc":"{result, output} =\n      with_io(fn ->\n        IO.puts(\"a\")\n        IO.puts(\"b\")\n        2 + 2\n      end)\n\n    assert result == 4\n    assert output == \"a\\nb\\n\"","ref":"ExUnit.CaptureIO.html#with_io/1-examples"},{"type":"function","title":"ExUnit.CaptureIO.with_io/2","doc":"Invokes the given `fun` and returns the result and captured output.\n\nSee `with_io/1` for more information.","ref":"ExUnit.CaptureIO.html#with_io/2"},{"type":"function","title":"ExUnit.CaptureIO.with_io/3","doc":"Invokes the given `fun` and returns the result and captured output.\n\nSee `with_io/1` for more information.","ref":"ExUnit.CaptureIO.html#with_io/3"},{"type":"module","title":"ExUnit.CaptureLog","doc":"Functionality to capture logs for testing.","ref":"ExUnit.CaptureLog.html"},{"type":"module","title":"Examples - ExUnit.CaptureLog","doc":"defmodule AssertionTest do\n      use ExUnit.Case\n\n      import ExUnit.CaptureLog\n      require Logger\n\n      test \"example\" do\n        {result, log} =\n          with_log(fn ->\n            Logger.error(\"log msg\")\n            2 + 2\n          end)\n\n        assert result == 4\n        assert log =~ \"log msg\"\n      end\n\n      test \"check multiple captures concurrently\" do\n        fun = fn ->\n          for msg   Logger.error(msg) end) =~ msg\n          end\n\n          Logger.debug(\"testing\")\n        end\n\n        assert capture_log(fun) =~ \"hello\"\n        assert capture_log(fun) =~ \"testing\"\n      end\n    end","ref":"ExUnit.CaptureLog.html#module-examples"},{"type":"function","title":"ExUnit.CaptureLog.capture_log/2","doc":"Captures Logger messages generated when evaluating `fun`.\n\nReturns the binary which is the captured output.\n\nThis function mutes the `:console` backend and captures any log\nmessages sent to Logger from the calling processes. It is possible\nto ensure explicit log messages from other processes are captured\nby waiting for their exit or monitor signal.\n\nNote that when the `async` is set to `true`, the messages from another\ntest might be captured. This is OK as long you consider such cases in\nyour assertions.\n\nIt is possible to configure the level to capture with `:level`,\nwhich will set the capturing level for the duration of the\ncapture, for instance, if the log level is set to `:error`, then\nany message with the lower level will be ignored.\nThe default level is `nil`, which will capture all messages.\nNote this setting does not override the overall `Logger.level/0` value.\nTherefore, if `Logger.level/0` is set to a higher level than the one\nconfigured in this function, no message will be captured.\nThe behaviour is undetermined if async tests change Logger level.\n\nThe format, metadata and colors can be configured with `:format`,\n`:metadata` and `:colors` respectively. These three options\ndefaults to the `:console` backend configuration parameters.\n\nTo get the result of the evaluation along with the captured log,\nuse `with_log/2`.","ref":"ExUnit.CaptureLog.html#capture_log/2"},{"type":"function","title":"ExUnit.CaptureLog.with_log/2","doc":"Invokes the given `fun` and returns the result and captured log.\n\nIt accepts the same arguments and options as `capture_log/2`.","ref":"ExUnit.CaptureLog.html#with_log/2"},{"type":"function","title":"Examples - ExUnit.CaptureLog.with_log/2","doc":"{result, log} =\n      with_log(fn ->\n        Logger.error(\"log msg\")\n        2 + 2\n      end)\n\n    assert result == 4\n    assert log =~ \"log msg\"","ref":"ExUnit.CaptureLog.html#with_log/2-examples"},{"type":"module","title":"ExUnit.Case","doc":"Helpers for defining test cases.\n\nThis module must be used in other modules as a way to configure\nand prepare them for testing.\n\nWhen used, it accepts the following options:\n\n  * `:async` - configures tests in this module to run concurrently with\n    tests in other modules. Tests in the same module never run concurrently.\n    It should be enabled only if tests do not change any global state.\n    Defaults to `false`.\n\n  * `:register` - when `false`, does not register this module within\n    ExUnit server. This means the module won't run when ExUnit suite runs.\n\n> #### `use ExUnit.Case` {: .info}\n>\n> When you `use ExUnit.Case`, it will import the functionality\n> from `ExUnit.Assertions`, `ExUnit.Callbacks`, `ExUnit.DocTest`,\n> and this module itself.","ref":"ExUnit.Case.html"},{"type":"module","title":"Examples - ExUnit.Case","doc":"defmodule AssertionTest do\n      # Use the module\n      use ExUnit.Case, async: true\n\n      # The \"test\" macro is imported by ExUnit.Case\n      test \"always pass\" do\n        assert true\n      end\n    end","ref":"ExUnit.Case.html#module-examples"},{"type":"module","title":"Context - ExUnit.Case","doc":"All tests receive a context as an argument. The context is particularly\nuseful for sharing information between callbacks and tests:\n\n    defmodule KVTest do\n      use ExUnit.Case\n\n      setup do\n        {:ok, pid} = KV.start_link()\n        {:ok, pid: pid}\n      end\n\n      test \"stores key-value pairs\", context do\n        assert KV.put(context[:pid], :hello, :world) == :ok\n        assert KV.get(context[:pid], :hello) == :world\n      end\n    end\n\nAs the context is a map, it can be pattern matched on to extract\ninformation:\n\n    test \"stores key-value pairs\", %{pid: pid} = _context do\n      assert KV.put(pid, :hello, :world) == :ok\n      assert KV.get(pid, :hello) == :world\n    end","ref":"ExUnit.Case.html#module-context"},{"type":"module","title":"Tags - ExUnit.Case","doc":"The context is used to pass information from the callbacks to\nthe test. In order to pass information from the test to the\ncallback, ExUnit provides tags.\n\nBy tagging a test, the tag value can be accessed in the context,\nallowing the developer to customize the test. Let's see an\nexample:\n\n    defmodule FileTest do\n      # Changing directory cannot be async\n      use ExUnit.Case, async: false\n\n      setup context do\n        # Read the :cd tag value\n        if cd = context[:cd] do\n          prev_cd = File.cwd!()\n          File.cd!(cd)\n          on_exit(fn -> File.cd!(prev_cd) end)\n        end\n\n        :ok\n      end\n\n      @tag cd: \"fixtures\"\n      test \"reads UTF-8 fixtures\" do\n        File.read(\"README.md\")\n      end\n    end\n\nIn the example above, we have defined a tag called `:cd` that is\nread in the setup callback to configure the working directory the\ntest is going to run on.\n\nTags are also very effective when used with case templates\n(`ExUnit.CaseTemplate`) allowing callbacks in the case template\nto customize the test behaviour.\n\nNote a tag can be set in two different ways:\n\n    @tag key: value\n    @tag :key       # equivalent to setting @tag key: true\n\nIf a tag is given more than once, the last value wins.\n\n#","ref":"ExUnit.Case.html#module-tags"},{"type":"module","title":"Module and describe tags - ExUnit.Case","doc":"A tag can be set for all tests in a module or describe block by\nsetting `@moduletag` or `@describetag` inside each context\nrespectively:\n\n    defmodule ApiTest do\n      use ExUnit.Case\n      @moduletag :external\n\n      describe \"makes calls to the right endpoint\" do\n        @describetag :endpoint\n\n        # ...\n      end\n    end\n\nIf you are setting a `@moduletag` or `@describetag` attribute, you must\nset them after your call to `use ExUnit.Case` otherwise you will see\ncompilation errors.\n\nIf the same key is set via `@tag`, the `@tag` value has higher\nprecedence.\n\nThe `setup_all` blocks only receive tags that are set using `@moduletag`.\n\n#","ref":"ExUnit.Case.html#module-module-and-describe-tags"},{"type":"module","title":"Known tags - ExUnit.Case","doc":"The following tags are set automatically by ExUnit and are\ntherefore reserved:\n\n  * `:module` - the module on which the test was defined\n\n  * `:file` - the file on which the test was defined\n\n  * `:line` - the line on which the test was defined\n\n  * `:test` - the test name\n\n  * `:async` - if the test case is in async mode\n\n  * `:registered` - used for `ExUnit.Case.register_attribute/3` values\n\n  * `:describe` - the describe block the test belongs to\n\n  * `:describe_line` - the line the describe block begins on\n\n  * `:doctest` - the module or the file being doctested (if a doctest)\n\n  * `:doctest_line` - the line the doctest was defined (if a doctest)\n\n  * `:doctest_data` - additional metadata about doctests (if a doctest)\n\n  * `:test_type` - the test type used when printing test results.\n    It is set by ExUnit to `:test`, `:doctest` and so on, but is customizable.\n\nThe following tags customize how tests behave:\n\n  * `:capture_log` - see the \"Log Capture\" section below\n\n  * `:skip` - skips the test with the given reason\n\n  * `:timeout` - customizes the test timeout in milliseconds (defaults to 60000).\n    Accepts `:infinity` as a timeout value.\n\n  * `:tmp_dir` - (since v1.11.0) see the \"Tmp Dir\" section below","ref":"ExUnit.Case.html#module-known-tags"},{"type":"module","title":"Filters - ExUnit.Case","doc":"Tags can also be used to identify specific tests, which can then\nbe included or excluded using filters. The most common functionality\nis to exclude some particular tests from running, which can be done\nvia `ExUnit.configure/1`:\n\n    # Exclude all external tests from running\n    ExUnit.configure(exclude: [external: true])\n\nFrom now on, ExUnit will not run any test that has the `:external` option\nset to `true`. This behaviour can be reversed with the `:include` option\nwhich is usually passed through the command line:\n\n    $ mix test --include external:true\n\nRun `mix help test` for more information on how to run filters via Mix.\n\nAnother use case for tags and filters is to exclude all tests that have\na particular tag by default, regardless of its value, and include only\na certain subset:\n\n    ExUnit.configure(exclude: :os, include: [os: :unix])\n\nA given include/exclude filter can be given more than once:\n\n    ExUnit.configure(exclude: [os: :unix, os: :windows])\n\nKeep in mind that all tests are included by default, so unless they are\nexcluded first, the `include` option has no effect.","ref":"ExUnit.Case.html#module-filters"},{"type":"module","title":"Log Capture - ExUnit.Case","doc":"ExUnit can optionally suppress printing of log messages that are generated\nduring a test. Log messages generated while running a test are captured and\nonly if the test fails are they printed to aid with debugging.\n\nYou can opt into this behaviour for individual tests by tagging them with\n`:capture_log` or enable log capture for all tests in the ExUnit configuration:\n\n    ExUnit.start(capture_log: true)\n\nThis default can be overridden by `@tag capture_log: false` or\n`@moduletag capture_log: false`.\n\nSince `setup_all` blocks don't belong to a specific test, log messages generated\nin them (or between tests) are never captured. If you want to suppress these\nmessages as well, remove the console backend globally by setting:\n\n    config :logger, backends: []","ref":"ExUnit.Case.html#module-log-capture"},{"type":"module","title":"Tmp Dir - ExUnit.Case","doc":"ExUnit automatically creates a temporary directory for tests tagged with\n`:tmp_dir` and puts the path to that directory into the test context.\nThe directory is removed before being created to ensure we start with a blank\nslate.\n\nThe temporary directory path is unique (includes the test module and test name)\nand thus appropriate for running tests concurrently. You can customize the path\nfurther by setting the tag to a string, e.g.: `tmp_dir: \"my_path\"`, which would\nmake the final path to be: `tmp/ / /my_path`.\n\nExample:\n\n    defmodule MyTest do\n      use ExUnit.Case, async: true\n\n      @tag :tmp_dir\n      test \"with tmp_dir\", %{tmp_dir: tmp_dir} do\n        assert tmp_dir =~ \"with tmp_dir\"\n        assert File.dir?(tmp_dir)\n      end\n    end\n\nAs with other tags, `:tmp_dir` can also be set as `@moduletag` and\n`@describetag`.","ref":"ExUnit.Case.html#module-tmp-dir"},{"type":"macro","title":"ExUnit.Case.describe/2","doc":"Describes tests together.\n\nEvery describe block receives a name which is used as prefix for\nupcoming tests. Inside a block, `ExUnit.Callbacks.setup/1` may be\ninvoked and it will define a setup callback to run only for the\ncurrent block. The describe name is also added as a tag, allowing\ndevelopers to run tests for specific blocks.","ref":"ExUnit.Case.html#describe/2"},{"type":"macro","title":"Examples - ExUnit.Case.describe/2","doc":"defmodule StringTest do\n      use ExUnit.Case, async: true\n\n      describe \"String.capitalize/1\" do\n        test \"first grapheme is in uppercase\" do\n          assert String.capitalize(\"hello\") == \"Hello\"\n        end\n\n        test \"converts remaining graphemes to lowercase\" do\n          assert String.capitalize(\"HELLO\") == \"Hello\"\n        end\n      end\n    end\n\nWhen using Mix, you can run all tests in a describe block by name:\n\n    $ mix test --only describe:\"String.capitalize/1\"\n\nor by passing the exact line the describe block starts on:\n\n    $ mix test path/to/file:123\n\nNote describe blocks cannot be nested. Instead of relying on hierarchy\nfor composition, developers should build on top of named setups. For\nexample:\n\n    defmodule UserManagementTest do\n      use ExUnit.Case, async: true\n\n      describe \"when user is logged in and is an admin\" do\n        setup [:log_user_in, :set_type_to_admin]\n\n        test ...\n      end\n\n      describe \"when user is logged in and is a manager\" do\n        setup [:log_user_in, :set_type_to_manager]\n\n        test ...\n      end\n\n      defp log_user_in(context) do\n        # ...\n      end\n    end\n\nBy forbidding hierarchies in favor of named setups, it is straightforward\nfor the developer to glance at each describe block and know exactly the\nsetup steps involved.","ref":"ExUnit.Case.html#describe/2-examples"},{"type":"function","title":"ExUnit.Case.get_last_registered_test/1","doc":"Returns the most recently registered test case as an `%ExUnit.Test{}`\nstruct.\n\nThis is used by third-party utilities to allow compile-time configuration\nusing test tags without having to explicitly pass the test context at\nrun-time. It is intended to be invoked in macros before the test module\nis compiled.\n\nRaises if called with a module that has already been compiled.","ref":"ExUnit.Case.html#get_last_registered_test/1"},{"type":"function","title":"ExUnit.Case.register_attribute/3","doc":"Registers a new attribute to be used during `ExUnit.Case` tests.\n\nThe attribute values will be available through `context.registered`.\nRegistered values are cleared after each `test/3` similar\nto `@tag`.\n\nThis function takes the same options as `Module.register_attribute/3`.","ref":"ExUnit.Case.html#register_attribute/3"},{"type":"function","title":"Examples - ExUnit.Case.register_attribute/3","doc":"defmodule MyTest do\n      use ExUnit.Case\n\n      ExUnit.Case.register_attribute(__MODULE__, :fixtures, accumulate: true)\n\n      @fixtures :user\n      @fixtures {:post, insert: false}\n      test \"using custom attribute\", context do\n        assert context.registered.fixtures == [{:post, insert: false}, :user]\n      end\n\n      test \"custom attributes are cleared per test\", context do\n        assert context.registered.fixtures == []\n      end\n    end","ref":"ExUnit.Case.html#register_attribute/3-examples"},{"type":"function","title":"ExUnit.Case.register_describe_attribute/3","doc":"Registers a new describe attribute to be used during `ExUnit.Case` tests.\n\nThe attribute values will be available through `context.registered`.\nRegistered values are cleared after each `describe/2` similar\nto `@describetag`.\n\nThis function takes the same options as `Module.register_attribute/3`.","ref":"ExUnit.Case.html#register_describe_attribute/3"},{"type":"function","title":"Examples - ExUnit.Case.register_describe_attribute/3","doc":"defmodule MyTest do\n      use ExUnit.Case\n\n      ExUnit.Case.register_describe_attribute(__MODULE__, :describe_fixtures, accumulate: true)\n\n      describe \"using custom attribute\" do\n        @describe_fixtures :user\n        @describe_fixtures {:post, insert: false}\n\n        test \"has attribute\", context do\n          assert context.registered.describe_fixtures == [{:post, insert: false}, :user]\n        end\n      end\n\n      describe \"custom attributes are cleared per describe\" do\n        test \"doesn't have attributes\", context do\n          assert context.registered.describe_fixtures == []\n        end\n      end\n    end","ref":"ExUnit.Case.html#register_describe_attribute/3-examples"},{"type":"function","title":"ExUnit.Case.register_module_attribute/3","doc":"Registers a new module attribute to be used during `ExUnit.Case` tests.\n\nThe attribute values will be available through `context.registered`.\n\nThis function takes the same options as `Module.register_attribute/3`.","ref":"ExUnit.Case.html#register_module_attribute/3"},{"type":"function","title":"Examples - ExUnit.Case.register_module_attribute/3","doc":"defmodule MyTest do\n      use ExUnit.Case\n\n      ExUnit.Case.register_module_attribute(__MODULE__, :module_fixtures, accumulate: true)\n\n      @module_fixtures :user\n      @module_fixtures {:post, insert: false}\n\n      test \"using custom attribute\", context do\n        assert context.registered.module_fixtures == [{:post, insert: false}, :user]\n      end\n\n      test \"still using custom attribute\", context do\n        assert context.registered.module_fixtures == [{:post, insert: false}, :user]\n      end\n    end","ref":"ExUnit.Case.html#register_module_attribute/3-examples"},{"type":"function","title":"ExUnit.Case.register_test/4","doc":"Registers a test with the given environment.\n\nThis function is deprecated in favor of `register_test/6` which performs\nbetter under tight loops by avoiding `__ENV__`.","ref":"ExUnit.Case.html#register_test/4"},{"type":"function","title":"ExUnit.Case.register_test/6","doc":"Registers a function to run as part of this case.\n\nThis is used by third-party projects, like QuickCheck, to\nimplement macros like `property/3` that works like `test`\nbut instead defines a property. See `test/3` implementation\nfor an example of invoking this function.\n\nThe test type will be converted to a string and pluralized for\ndisplay. You can use `ExUnit.plural_rule/2` to set a custom\npluralization.","ref":"ExUnit.Case.html#register_test/6"},{"type":"macro","title":"ExUnit.Case.test/1","doc":"Defines a not implemented test with a string.\n\nProvides a convenient macro that allows a test to be defined\nwith a string, but not yet implemented. The resulting test will\nalways fail and print a \"Not implemented\" error message. The\nresulting test case is also tagged with `:not_implemented`.","ref":"ExUnit.Case.html#test/1"},{"type":"macro","title":"Examples - ExUnit.Case.test/1","doc":"test \"this will be a test in future\"","ref":"ExUnit.Case.html#test/1-examples"},{"type":"macro","title":"ExUnit.Case.test/3","doc":"Defines a test with `message`.\n\nThe test may also define a pattern, which will be matched\nagainst the test context. For more information on contexts, see\n`ExUnit.Callbacks`.","ref":"ExUnit.Case.html#test/3"},{"type":"macro","title":"Examples - ExUnit.Case.test/3","doc":"test \"true is equal to true\" do\n      assert true == true\n    end","ref":"ExUnit.Case.html#test/3-examples"},{"type":"type","title":"ExUnit.Case.env/0","doc":"","ref":"ExUnit.Case.html#t:env/0"},{"type":"module","title":"ExUnit.CaseTemplate","doc":"Defines a module template to be used throughout your test suite.\n\nThis is useful when there are a set of setup callbacks or a set\nof functions that should be shared between test modules.\n\nLet's imagine that you create a `MyCase` module that calls `use\nExUnit.CaseTemplate`. When a test case module calls `use MyCase`, the\nfollowing things hold true:\n\n  * All the functionality that `MyCase` would have had available from\n    `use ExUnit.Case` is available (same as if `MyCase` called `use\n    ExUnit.Case` directly)\n\n  * The `setup` and `setup_all` callbacks that you define in `MyCase`\n    get used in the test case module\n\nThe options that you pass to `use MyCase` get also passed to `use\nExUnit.Case` under the hood. This means you can do things like `use\nMyCase, async: true`. You can also access this options in `using/2`.\n\n> #### `use ExUnit.CaseTemplate` {: .info}\n>\n> When you `use ExUnit.CaseTemplate`, it will import the functionality\n> from `ExUnit.Assertions`, `ExUnit.Callbacks`, and this module itself.\n> It will also define a `__using__` callback, so the module itself can\n> be used as a template instead of `ExUnit.Case`.","ref":"ExUnit.CaseTemplate.html"},{"type":"module","title":"Example - ExUnit.CaseTemplate","doc":"defmodule MyCase do\n      use ExUnit.CaseTemplate\n\n      setup do\n        IO.puts(\"This will run before each test that uses this case\")\n      end\n    end\n\n    defmodule MyTest do\n      use MyCase, async: true\n\n      test \"truth\" do\n        assert true\n      end\n    end\n\nIf you need to \"hook\" into `use MyCase` and do other things as well,\nyou can use the `using/2` macro. See its documentation for more\ninformation and examples.\n\n    defmodule MyCase do\n      use ExUnit.CaseTemplate\n\n      using do\n        quote do\n          import MyApp.TestHelpers\n        end\n      end\n    end","ref":"ExUnit.CaseTemplate.html#module-example"},{"type":"macro","title":"ExUnit.CaseTemplate.using/2","doc":"Allows a developer to customize the using block\nwhen the case template is used.\n\nYou can use an optional `var` argument when calling `using/2`. ExUnit\nwill pass whatever argument you pass to `use MyCase` as this `var` argument. See the examples below for clarification.","ref":"ExUnit.CaseTemplate.html#using/2"},{"type":"macro","title":"Example - ExUnit.CaseTemplate.using/2","doc":"defmodule MyCase do\n      use ExUnit.CaseTemplate\n\n      using do\n        quote do\n          # This code is injected into every case that calls \"use MyCase\"\n          alias MyApp.FunModule\n        end\n      end\n    end\n\nYou can specify an argument to `using/2`:\n\n    defmodule MyCase do\n      use ExUnit.CaseTemplate\n\n      using options do\n        quote do\n          if unquote(options)[:import_helpers] do\n            import MyApp.TestHelpers\n          end\n        end\n      end\n    end\n\nThe second argument passed to `use MyCase` gets forwarded to `using/2` too:\n\n    defmodule SomeTestCase do\n      use MyCase, async: true, import_helpers: true\n\n      test \"the truth\" do\n        # truth/0 comes from MyApp.TestHelpers:\n        assert truth()\n      end\n    end\n\n> ##","ref":"ExUnit.CaseTemplate.html#using/2-example"},{"type":"macro","title":"Sharing options with `use ExUnit.Case` {: .warning} - ExUnit.CaseTemplate.using/2","doc":">\n> The second argument that you pass to `use MyCase` is *also* passed\n> as the second argument to `use ExUnit.Case`.","ref":"ExUnit.CaseTemplate.html#using/2-sharing-options-with-use-exunit-case-warning"},{"type":"module","title":"ExUnit.DocTest","doc":"Extract test cases from the documentation.\n\nDoctests allow us to generate tests from code examples found\nin `@moduledoc` and `@doc` attributes. To do this, invoke the\n`doctest/1` macro from within your test case and ensure your\ncode examples are written according to the syntax and guidelines\nbelow.","ref":"ExUnit.DocTest.html"},{"type":"module","title":"Syntax - ExUnit.DocTest","doc":"Every new test starts on a new line, with an `iex>` prefix.\nMultiline expressions can be used by prefixing subsequent lines\nwith either `...>` (recommended) or `iex>`.\n\nThe expected result should start the line after the `iex>`\nand `...>` line(s) and be terminated by a newline.","ref":"ExUnit.DocTest.html#module-syntax"},{"type":"module","title":"Examples - ExUnit.DocTest","doc":"To run doctests include them in an ExUnit case with a `doctest` macro:\n\n    defmodule MyModuleTest do\n      use ExUnit.Case, async: true\n      doctest MyModule\n    end\n\nThe `doctest` macro loops through all functions and\nmacros defined in `MyModule`, parsing their documentation in\nsearch of code examples.\n\nA very basic example is:\n\n    iex> 1 + 1\n    2\n\nExpressions on multiple lines are also supported:\n\n    iex> Enum.map([1, 2, 3], fn x ->\n    ...>   x * 2\n    ...> end)\n    [2, 4, 6]\n\nMultiple results can be checked within the same test:\n\n    iex> a = 1\n    1\n    iex> a + 1\n    2\n\nIf you want to keep any two tests separate,\nadd an empty line between them:\n\n    iex> a = 1\n    1\n\n    iex> a + 1 # will fail with a `undefined variable \"a\"` error\n    2\n\nIf you don't want to assert for every result in a doctest, you can omit\nthe result. You can do so between expressions:\n\n    iex> pid = spawn(fn -> :ok end)\n    iex> is_pid(pid)\n    true\n\nAs well as at the end:\n\n    iex> Mod.do_a_call_that_should_not_raise!(...)\n\nThis is useful when the result is something variable (like a PID in the\nexample above) or when the result is a complicated data structure and you\ndon't want to show it all, but just parts of it or some of its properties.\n\nSimilarly to IEx you can use numbers in your \"prompts\":\n\n    iex(1)> [1 + 2,\n    ...(1)>  3]\n    [3, 3]\n\nThis is useful in two cases:\n\n  * being able to refer to specific numbered scenarios\n  * copy-pasting examples from an actual IEx session\n\nYou can also select or skip functions when calling\n`doctest`. See the documentation on the `:except` and `:only` options below\nfor more information.","ref":"ExUnit.DocTest.html#module-examples"},{"type":"module","title":"Opaque types - ExUnit.DocTest","doc":"Some types' internal structures are kept hidden and instead show a\nuser-friendly structure when inspected. The idiom in\nElixir is to print those data types in the format `#Name `. Because those\nvalues are treated as comments in Elixir code due to the leading\n`#` sign, they require special care when being used in doctests.\n\nImagine you have a map that contains a MapSet and is printed as:\n\n    %{users: #MapSet }\n\nIf you try to match on such an expression, `doctest` will fail to compile.\nThere are two ways to resolve this.\n\nThe first is to rely on the fact that doctest can compare internal\nstructures as long as they are at the root. So one could write:\n\n    iex> map = %{users: Enum.into([:foo, :bar], MapSet.new())}\n    iex> map.users\n    #MapSet \n\nWhenever a doctest starts with \"#Name \"\n\nAlternatively, since doctest results are actually evaluated, you can have\nthe MapSet building expression as the doctest result:\n\n    iex> %{users: Enum.into([:foo, :bar], MapSet.new())}\n    %{users: Enum.into([:foo, :bar], MapSet.new())}\n\nThe downside of this approach is that the doctest result is not really\nwhat users would see in the terminal.","ref":"ExUnit.DocTest.html#module-opaque-types"},{"type":"module","title":"Exceptions - ExUnit.DocTest","doc":"You can also showcase expressions raising an exception, for example:\n\n    iex(1)> raise \"some error\"\n    ** (RuntimeError) some error\n\nDoctest will look for a line starting with `** (` and it will parse it\naccordingly to extract the exception name and message. The exception parser\nwill consider all following lines part of the exception message until there\nis an empty line or there is a new expression prefixed with `iex>`.\nTherefore, it is possible to match on multiline messages as long as there\nare no empty lines on the message itself.","ref":"ExUnit.DocTest.html#module-exceptions"},{"type":"module","title":"When not to use doctest - ExUnit.DocTest","doc":"In general, doctests are not recommended when your code examples contain\nside effects. For example, if a doctest prints to standard output, doctest\nwill not try to capture the output.\n\nSimilarly, doctests do not run in any kind of sandbox. So any module\ndefined in a code example is going to linger throughout the whole test\nsuite run.","ref":"ExUnit.DocTest.html#module-when-not-to-use-doctest"},{"type":"macro","title":"ExUnit.DocTest.doctest/2","doc":"Generate test cases from module documentation.\n\nCalling `doctest(Module)` will generate tests for all doctests found\nin the `module`.","ref":"ExUnit.DocTest.html#doctest/2"},{"type":"macro","title":"Options - ExUnit.DocTest.doctest/2","doc":"* `:except` - generates tests for all functions except those listed\n    (list of `{function, arity}` tuples, and/or `:moduledoc`).\n\n  * `:only` - generates tests only for functions listed\n    (list of `{function, arity}` tuples, and/or `:moduledoc`).\n\n  * `:import` - when `true`, one can test a function defined in the module\n    without referring to the module name. However, this is not feasible when\n    there is a clash with a module like `Kernel`. In these cases, `:import`\n    should be set to `false` and `Module.function(...)` should be used instead.\n\n  * `:tags` - a list of tags to apply to all generated doctests.","ref":"ExUnit.DocTest.html#doctest/2-options"},{"type":"macro","title":"Examples - ExUnit.DocTest.doctest/2","doc":"defmodule MyModuleTest do\n      use ExUnit.Case\n      doctest MyModule, except: [:moduledoc, trick_fun: 1]\n    end\n\nThis macro is auto-imported with every `ExUnit.Case`.","ref":"ExUnit.DocTest.html#doctest/2-examples"},{"type":"macro","title":"ExUnit.DocTest.doctest_file/2","doc":"Generate test cases from a markdown file.","ref":"ExUnit.DocTest.html#doctest_file/2"},{"type":"macro","title":"Options - ExUnit.DocTest.doctest_file/2","doc":"* `:tags` - a list of tags to apply to all generated doctests.","ref":"ExUnit.DocTest.html#doctest_file/2-options"},{"type":"macro","title":"Examples - ExUnit.DocTest.doctest_file/2","doc":"defmodule ReadmeTest do\n      use ExUnit.Case\n      doctest_file \"README.md\"\n    end\n\nThis macro is auto-imported with every `ExUnit.Case`.","ref":"ExUnit.DocTest.html#doctest_file/2-examples"},{"type":"module","title":"ExUnit.Filters","doc":"Conveniences for parsing and evaluating filters.","ref":"ExUnit.Filters.html"},{"type":"function","title":"ExUnit.Filters.eval/4","doc":"Evaluates the `include` and `exclude` filters against the given `tags` to\ndetermine if tests should be skipped or excluded.\n\nSome filters, like `:line`, may require the whole test `collection` to\nfind the closest line, that's why it must also be passed as an argument.\n\nFilters can either be a regular expression or any data structure\nthat implements the `String.Chars` protocol, which is invoked before comparing\nthe filter with the `:tag` value.","ref":"ExUnit.Filters.html#eval/4"},{"type":"function","title":"Precedence - ExUnit.Filters.eval/4","doc":"Tests are first excluded, then included, and then skipped (if any left).\n\nIf a `:skip` tag is found in `tags`, `{:skipped, message}` is returned if the test\nhas been left after the `exclude` and `include` filters. Otherwise `{:exclude, message}`\nis returned.\n\nThe only exception to this rule is that `:skip` is found in the `include` filter,\n`:ok` is returned regardless of whether the test was excluded or not.","ref":"ExUnit.Filters.html#eval/4-precedence"},{"type":"function","title":"Examples - ExUnit.Filters.eval/4","doc":"iex> ExUnit.Filters.eval([foo: \"bar\"], [:foo], %{foo: \"bar\"}, [])\n    :ok\n\n    iex> ExUnit.Filters.eval([foo: \"bar\"], [:foo], %{foo: \"baz\"}, [])\n    {:excluded, \"due to foo filter\"}","ref":"ExUnit.Filters.html#eval/4-examples"},{"type":"function","title":"ExUnit.Filters.failure_info/1","doc":"Returns a tuple containing useful information about test failures from the\nmanifest. The tuple contains:\n\n  * A set of files that contain tests that failed the last time they ran.\n    The paths are absolute paths.\n  * A set of test IDs that failed the last time they ran","ref":"ExUnit.Filters.html#failure_info/1"},{"type":"function","title":"ExUnit.Filters.normalize/2","doc":"Normalizes `include` and `exclude` filters to remove duplicates\nand keep precedence.","ref":"ExUnit.Filters.html#normalize/2"},{"type":"function","title":"Examples - ExUnit.Filters.normalize/2","doc":"iex> ExUnit.Filters.normalize(nil, nil)\n    {[], []}\n\n    iex> ExUnit.Filters.normalize([:foo, :bar, :bar], [:foo, :baz])\n    {[:foo, :bar], [:baz]}\n\n    iex> ExUnit.Filters.normalize([foo: \"true\"], [:foo])\n    {[foo: \"true\"], [:foo]}\n\n    iex> ExUnit.Filters.normalize([:foo], [foo: \"true\"])\n    {[:foo], []}\n\n    iex> ExUnit.Filters.normalize([foo: \"true\"], [foo: true])\n    {[foo: \"true\"], []}\n\n    iex> ExUnit.Filters.normalize([foo: true], [foo: \"true\"])\n    {[foo: true], []}\n\n    iex> ExUnit.Filters.normalize([foo: 1, foo: 1, foo: 2], [])\n    {[foo: 1, foo: 2], []}\n\n    iex> ExUnit.Filters.normalize([], [foo: 1, foo: 1, foo: 2])\n    {[], [foo: 1, foo: 2]}","ref":"ExUnit.Filters.html#normalize/2-examples"},{"type":"function","title":"ExUnit.Filters.parse/1","doc":"Parses the given filters, as one would receive from the command line.","ref":"ExUnit.Filters.html#parse/1"},{"type":"function","title":"Examples - ExUnit.Filters.parse/1","doc":"iex> ExUnit.Filters.parse([\"foo:bar\", \"baz\", \"line:9\", \"bool:true\"])\n    [{:foo, \"bar\"}, :baz, {:line, \"9\"}, {:bool, \"true\"}]","ref":"ExUnit.Filters.html#parse/1-examples"},{"type":"function","title":"ExUnit.Filters.parse_path/1","doc":"Parses filters out of a path.\n\nDetermines whether a given file path (supplied to ExUnit/Mix as arguments\non the command line) includes a line number filter, and if so returns the\nappropriate ExUnit configuration options.","ref":"ExUnit.Filters.html#parse_path/1"},{"type":"type","title":"ExUnit.Filters.t/0","doc":"","ref":"ExUnit.Filters.html#t:t/0"},{"type":"module","title":"ExUnit.Formatter","doc":"Helper functions for formatting and the formatting protocols.\n\nFormatters are `GenServer`s specified during ExUnit configuration\nthat receive a series of events as casts.\n\nThe following events are possible:\n\n  * `{:suite_started, opts}` -\n    the suite has started with the specified options to the runner.\n\n  * `{:suite_finished, times_us}` -\n    the suite has finished. Returns several measurements in microseconds\n    for running the suite. See `t:times_us` for more information.\n\n  * `{:module_started, test_module}` -\n    a test module has started. See `ExUnit.TestModule` for details.\n\n  * `{:module_finished, test_module}` -\n    a test module has finished. See `ExUnit.TestModule` for details.\n\n  * `{:test_started, test}` -\n    a test has started. See `ExUnit.Test` for details.\n\n  * `{:test_finished, test}` -\n    a test has finished. See `ExUnit.Test` for details.\n\n  * `{:sigquit, [test | test_module]}` -\n    the VM is going to shutdown. It receives the test cases (or test\n    module in case of `setup_all`) still running.\n\nThe formatter will also receive the following events but they are deprecated\nand should be ignored:\n\n  * `{:case_started, test_module}` -\n    a test module has started. See `ExUnit.TestModule` for details.\n\n  * `{:case_finished, test_module}` -\n    a test module has finished. See `ExUnit.TestModule` for details.\n\nThe full ExUnit configuration is passed as the argument to `c:GenServer.init/1`\ncallback when the formatters are started. If you need to do runtime configuration\nof a formatter, you can add any configuration needed by using `ExUnit.configure/1`\nor `ExUnit.start/1`, and this will then be included in the options passed to\nthe `c:GenServer.init/1` callback.","ref":"ExUnit.Formatter.html"},{"type":"function","title":"ExUnit.Formatter.format_assertion_diff/4","doc":"Formats `ExUnit.AssertionError` diff.\n\nIt returns a keyword list with diffing information\nfrom the left and right side of the assertion, if\nany exists.\n\nIt expects the assertion error, the `padding_size`\nfor formatted content, the width (may be `:infinity`),\nand the formatter callback function.","ref":"ExUnit.Formatter.html#format_assertion_diff/4"},{"type":"function","title":"ExUnit.Formatter.format_filters/2","doc":"Formats filters used to constrain cases to be run.","ref":"ExUnit.Formatter.html#format_filters/2"},{"type":"function","title":"Examples - ExUnit.Formatter.format_filters/2","doc":"iex> format_filters([run: true, slow: false], :include)\n    \"Including tags: [run: true, slow: false]\"","ref":"ExUnit.Formatter.html#format_filters/2-examples"},{"type":"function","title":"ExUnit.Formatter.format_test_all_failure/5","doc":"Receives a test module and formats its failure.","ref":"ExUnit.Formatter.html#format_test_all_failure/5"},{"type":"function","title":"ExUnit.Formatter.format_test_failure/5","doc":"Receives a test and formats its failure.","ref":"ExUnit.Formatter.html#format_test_failure/5"},{"type":"function","title":"ExUnit.Formatter.format_times/1","doc":"Formats time taken running the test suite.","ref":"ExUnit.Formatter.html#format_times/1"},{"type":"function","title":"Examples - ExUnit.Formatter.format_times/1","doc":"iex> format_times(%{run: 10000, async: nil, load: nil})\n    \"Finished in 0.01 seconds (0.00s async, 0.01s sync)\"\n\n    iex> format_times(%{run: 10000, async: nil, load: 20000})\n    \"Finished in 0.03 seconds (0.02s on load, 0.00s async, 0.01s sync)\"\n\n    iex> format_times(%{run: 10000, async: nil, load: 200_000})\n    \"Finished in 0.2 seconds (0.2s on load, 0.00s async, 0.01s sync)\"\n\n    iex> format_times(%{run: 100_000, async: 50000, load: 200_000})\n    \"Finished in 0.3 seconds (0.2s on load, 0.05s async, 0.05s sync)\"","ref":"ExUnit.Formatter.html#format_times/1-examples"},{"type":"type","title":"ExUnit.Formatter.id/0","doc":"","ref":"ExUnit.Formatter.html#t:id/0"},{"type":"type","title":"ExUnit.Formatter.test/0","doc":"","ref":"ExUnit.Formatter.html#t:test/0"},{"type":"type","title":"ExUnit.Formatter.times_us/0","doc":"The times spent on several parts of the test suite.\n\nThe following properties can be computed:\n\n    sync = run - (async || 0)\n    total = run + (load || 0)\n\n`async` is nil when there are no async tests.\n`load` is nil when the test suite is running and loading\ntests concurrently.","ref":"ExUnit.Formatter.html#t:times_us/0"},{"type":"module","title":"ExUnit.Test","doc":"A struct that keeps information about the test.\n\nIt is received by formatters and contains the following fields:\n\n  * `:name` - the test name\n  * `:module` - the test module\n  * `:state` - the finished test state (see `t:ExUnit.state/0`)\n  * `:time` - the duration in microseconds of the test's runtime\n  * `:tags` - the test tags\n  * `:logs` - the captured logs","ref":"ExUnit.Test.html"},{"type":"type","title":"ExUnit.Test.t/0","doc":"","ref":"ExUnit.Test.html#t:t/0"},{"type":"module","title":"ExUnit.TestModule","doc":"A struct that keeps information about the test module.\n\nIt is received by formatters and contains the following fields:\n\n  * `:file`  - (since v1.11.0) the file of the test module\n\n  * `:name`  - the test module name\n\n  * `:state` - the test error state (see `t:ExUnit.state/0`)\n\n  * `:tests` - all tests in this module","ref":"ExUnit.TestModule.html"},{"type":"type","title":"ExUnit.TestModule.t/0","doc":"","ref":"ExUnit.TestModule.html#t:t/0"},{"type":"exception","title":"ExUnit.AssertionError","doc":"Raised to signal an assertion error.","ref":"ExUnit.AssertionError.html"},{"type":"function","title":"ExUnit.AssertionError.no_value/0","doc":"Indicates no meaningful value for a field.","ref":"ExUnit.AssertionError.html#no_value/0"},{"type":"exception","title":"ExUnit.DocTest.Error","doc":"","ref":"ExUnit.DocTest.Error.html"},{"type":"exception","title":"ExUnit.DuplicateDescribeError","doc":"","ref":"ExUnit.DuplicateDescribeError.html"},{"type":"exception","title":"ExUnit.DuplicateTestError","doc":"","ref":"ExUnit.DuplicateTestError.html"},{"type":"exception","title":"ExUnit.MultiError","doc":"Raised to signal multiple errors happened in a test case.","ref":"ExUnit.MultiError.html"},{"type":"exception","title":"ExUnit.TimeoutError","doc":"","ref":"ExUnit.TimeoutError.html"}],"content_type":"text/markdown"}