searchData={"items":[{"type":"module","title":"Mix","doc":"Mix is a build tool that provides tasks for creating, compiling,\nand testing Elixir projects, managing its dependencies, and more.","ref":"Mix.html"},{"type":"module","title":"Mix.Project - Mix","doc":"The foundation of Mix is a project. A project can be defined by using\n`Mix.Project` in a module, usually placed in a file named `mix.exs`:\n\n    defmodule MyApp.MixProject do\n      use Mix.Project\n\n      def project do\n        [\n          app: :my_app,\n          version: \"1.0.0\"\n        ]\n      end\n    end\n\nSee the `Mix.Project` module for detailed documentation on Mix projects.\n\nOnce the project is defined, a number of default Mix tasks can be run\ndirectly from the command line:\n\n  * `mix compile` - compiles the current project\n  * `mix test` - runs tests for the given project\n  * `mix run` - runs a particular command inside the project\n\nEach task has its own options and sometimes specific configuration\nto be defined in the `project/0` function. You can use `mix help`\nto list all available tasks and `mix help NAME` to show help for\na particular task.\n\nThe best way to get started with your first project is by calling\n`mix new my_project` from the command line.","ref":"Mix.html#module-mix-project"},{"type":"module","title":"Mix.Task - Mix","doc":"Tasks are what make Mix extensible.\n\nProjects can extend Mix behaviour by adding their own tasks. For\nexample, adding the task below inside your project will\nmake it available to everyone that uses your project:\n\n    defmodule Mix.Tasks.Hello do\n      use Mix.Task\n\n      def run(_) do\n        Mix.shell().info(\"Hello world\")\n      end\n    end\n\nThe task can now be invoked with `mix hello`.\n\nSee the `Mix.Task` behaviour for detailed documentation on Mix tasks.","ref":"Mix.html#module-mix-task"},{"type":"module","title":"Dependencies - Mix","doc":"Mix also manages your dependencies and integrates nicely with the [Hex package\nmanager](https://hex.pm).\n\nIn order to use dependencies, you need to add a `:deps` key\nto your project configuration. We often extract the list of dependencies\ninto its own function:\n\n    defmodule MyApp.MixProject do\n      use Mix.Project\n\n      def project do\n        [\n          app: :my_app,\n          version: \"1.0.0\",\n          deps: deps()\n        ]\n      end\n\n      defp deps do\n        [\n          {:ecto, \"~> 2.0\"},\n          {:plug, github: \"elixir-lang/plug\"}\n        ]\n      end\n    end\n\nYou can run `mix help deps` to learn more about dependencies in Mix.","ref":"Mix.html#module-dependencies"},{"type":"module","title":"Environments - Mix","doc":"Mix supports different environments. Environments allow developers\nto prepare and organize their project specifically for different\nscenarios. By default, Mix provides three environments:\n\n  * `:dev` - the default environment\n  * `:test` - the environment `mix test` runs on\n  * `:prod` - the environment your dependencies run on\n\nThe environment can be changed via the command line by setting\nthe `MIX_ENV` environment variable, for example:\n\n```bash\n$ MIX_ENV=prod mix run server.exs\n```\n\nYou can also specify that certain dependencies are available only for\ncertain environments:\n\n    {:some_test_dependency, \"~> 1.0\", only: :test}\n\nWhen running Mix via the command line, you can configure the default\nenvironment or the preferred environment per task via the `def cli`\nfunction in your `mix.exs`. For example:\n\n    def cli do\n      [\n        default_env: :local,\n        preferred_envs: [docs: :docs]\n      ]\n    end\n\nThe environment can be read via `Mix.env/0`.","ref":"Mix.html#module-environments"},{"type":"module","title":"Targets - Mix","doc":"Besides environments, Mix supports targets. Targets are useful when a\nproject needs to compile to different architectures and some of the\ndependencies are only available to some of them. By default, the target\nis `:host` but it can be set via the `MIX_TARGET` environment variable.\n\nWhen running Mix via the command line, you can configure the default\ntarget or the preferred target per task via the `def cli` function\nin your `mix.exs`. For example:\n\n    def cli do\n      [\n        default_target: :local,\n        preferred_targets: [docs: :docs]\n      ]\n    end\n\nThe target can be read via `Mix.target/0`.","ref":"Mix.html#module-targets"},{"type":"module","title":"Configuration - Mix","doc":"Mix allows you configure the application environment of your application\nand of your dependencies. See the `Application` module to learn more about\nthe application environment. On this section, we will focus on how to configure\nit at two distinct moments: build-time and runtime.\n\n> ##","ref":"Mix.html#module-configuration"},{"type":"module","title":"Avoiding the application environment {: .warning} - Mix","doc":">\n> The application environment is discouraged for libraries. See Elixir's\n> [Library Guidelines](https://hexdocs.pm/elixir/library-guidelines.html) for\n> more information.\n\n#","ref":"Mix.html#module-avoiding-the-application-environment-warning"},{"type":"module","title":"Build-time configuration - Mix","doc":"Whenever you invoke a `mix` command, Mix loads the configuration\nin `config/config.exs`, if said file exists. It is common for the\n`config/config.exs` file itself to import other configuration based\non the current `MIX_ENV`, such as `config/dev.exs`, `config/test.exs`,\nand `config/prod.exs`, by calling `Config.import_config/1`:\n\n    import Config\n    import_config \"#{config_env()}.exs\"\n\nWe say `config/config.exs` and all imported files are build-time\nconfiguration as they are evaluated whenever you compile your code.\nIn other words, if your configuration does something like:\n\n    import Config\n    config :my_app, :secret_key, System.fetch_env!(\"MY_APP_SECRET_KEY\")\n\nThe `:secret_key` key under `:my_app` will be computed on the host\nmachine before your code compiles. This can be an issue if the machine\ncompiling your code does not have access to all environment variables\nused to run your code, as loading the config above will fail due to the\nmissing environment variable. Furthermore, even if the environment variable\nis set, changing the environment variable will require a full recompilation\nof your application by calling `mix compile --force` (otherwise your project\nwon't start). Luckily, Mix also provides runtime configuration, which is\npreferred in such cases and we will see next.\n\n#","ref":"Mix.html#module-build-time-configuration"},{"type":"module","title":"Runtime configuration - Mix","doc":"To enable runtime configuration in your release, all you need to do is\nto create a file named `config/runtime.exs`:\n\n    import Config\n    config :my_app, :secret_key, System.fetch_env!(\"MY_APP_SECRET_KEY\")\n\nThis file is executed whenever your project runs. If you assemble\na release with `mix release`, it also executes every time your release\nstarts.","ref":"Mix.html#module-runtime-configuration"},{"type":"module","title":"Aliases - Mix","doc":"Aliases are shortcuts or tasks specific to the current project.\n\nIn the [Mix.Task section](#module-mix-task), we have defined a task that would be\navailable to everyone using our project as a dependency. What if\nwe wanted the task to only be available for our project? Just\ndefine an alias:\n\n    defmodule MyApp.MixProject do\n      use Mix.Project\n\n      def project do\n        [\n          app: :my_app,\n          version: \"1.0.0\",\n          aliases: aliases()\n        ]\n      end\n\n      defp aliases do\n        [\n          c: \"compile\",\n          hello: &hello/1,\n          paid_task: &paid_task/1\n        ]\n      end\n\n      defp hello(_) do\n        Mix.shell().info(\"Hello world\")\n      end\n\n      defp paid_task(_) do\n        Mix.Task.run(\"paid.task\", [\n          \"first_arg\",\n          \"second_arg\",\n          \"--license-key\",\n          System.fetch_env!(\"SOME_LICENSE_KEY\")\n        ])\n      end\n    end\n\nIn the example above, we have defined three aliases. One is `mix c`\nwhich is a shortcut for `mix compile`. Another is named\n`mix hello` and the third is named `mix paid_task`, which executes\nthe code inside a custom function to invoke the `paid.task` task\nwith several arguments, including one pulled from an environment\nvariable.\n\nAliases may also be lists, specifying multiple tasks to be run\nconsecutively:\n\n    [all: [&hello/1, \"deps.get --only #{Mix.env()}\", \"compile\"]]\n\nIn the example above, we have defined an alias named `mix all`,\nthat prints \"Hello world\", then fetches dependencies specific\nto the current environment, and compiles the project.\n\nAliases can also be used to augment existing tasks. Let's suppose\nyou want to augment `mix clean` to clean another directory Mix does\nnot know about:\n\n    [clean: [\"clean\", &clean_extra/1]]\n\nWhere `&clean_extra/1` would be a function in your `mix.exs`\nwith extra cleanup logic.\n\nIf the alias is overriding an existing task, the arguments given\nto the alias will be forwarded to the original task in order to\npreserve semantics. Otherwise arguments given to the alias are\nappended to the arguments of the last task in the list.\n\nAnother use case of aliases is to run Elixir scripts and shell\ncommands, for example:\n\n    # priv/hello1.exs\n    IO.puts(\"Hello One\")\n\n    # priv/hello2.exs\n    IO.puts(\"Hello Two\")\n\n    # priv/world.sh\n    #!/bin/sh\n    echo \"world!\"\n\n    # mix.exs\n    defp aliases do\n      [\n        some_alias: [\"hex.info\", \"run priv/hello1.exs\", \"cmd priv/world.sh\"]\n      ]\n    end\n\nIn the example above we have created the alias `some_alias` that will\nrun the task `mix hex.info`, then `mix run` to run an Elixir script,\nthen `mix cmd` to execute a command line shell script. This shows how\npowerful aliases mixed with Mix tasks can be.\n\nOne commit pitfall of aliases comes when trying to invoke the same task\nmultiple times. Mix tasks are designed to run only once. This prevents\nthe same task from being executed multiple times. For example, if there\nare several tasks depending on `mix compile`, the code will be compiled\nonly once.\n\nSimilary, `mix format` can only be invoked once. So if you have an alias\nthat attempts to invoke `mix format` multiple times, it won't work unless\nit is explicitly reenabled using `Mix.Task.reenable/1`:\n\n    another_alias: [\n      \"format --check-formatted priv/hello1.exs\",\n      \"cmd priv/world.sh\",\n      fn _ -> Mix.Task.reenable(\"format\") end,\n      \"format --check-formatted priv/hello2.exs\"\n    ]\n\nSome tasks are automatically reenabled though, as they are expected to\nbe invoked multiple times, such as: `mix cmd`, `mix do`, `mix xref`, etc.\n\nFinally, aliases defined in the current project do not affect its\ndependencies and aliases defined in dependencies are not accessible\nfrom the current project, with the exception of umbrella projects.\nUmbrella projects will run the aliases of its children when the\numbrella project itself does not define said alias and there is no\ntask with said name.","ref":"Mix.html#module-aliases"},{"type":"module","title":"Environment variables - Mix","doc":"Several environment variables can be used to modify Mix's behaviour.\n\nMix responds to the following variables:\n\n  * `MIX_ARCHIVES` - specifies the directory into which the archives should be installed\n    (default: `~/.mix/archives`)\n\n  * `MIX_BUILD_PATH` - sets the project `Mix.Project.build_path/0` config.\n    This option must always point to a subdirectory inside a temporary directory.\n    For instance, never \"/tmp\" or \"_build\" but \"_build/PROD\" or \"/tmp/PROD\", as\n    required by Mix. This environment variable is used mostly by external build\n    tools. For your CI servers, you likely want to use `MIX_BUILD_ROOT` below.\n\n  * `MIX_BUILD_ROOT` - sets the root directory where build artifacts should be\n    written to. For example, \"_build\". If `MIX_BUILD_PATH` is set, this option\n    is ignored.\n\n  * `MIX_DEBUG` - outputs debug information about each task before running it\n\n  * `MIX_DEPS_PATH` - sets the project `Mix.Project.deps_path/0` config for the\n    current project (default: `deps`)\n\n  * `MIX_ENV` - specifies which environment should be used. See [Environments](#module-environments)\n\n  * `MIX_EXS` - changes the full path to the `mix.exs` file\n\n  * `MIX_HOME` - path to Mix's home directory, stores configuration files and scripts used by Mix\n    (default: `~/.mix`)\n\n  * `MIX_INSTALL_DIR` - (since v1.12.0) specifies directory where `Mix.install/2` keeps\n     install cache\n\n  * `MIX_INSTALL_FORCE` - (since v1.13.0) runs `Mix.install/2` with empty install cache\n\n  * `MIX_PATH` - appends extra code paths\n\n  * `MIX_QUIET` - does not print information messages to the terminal\n\n  * `MIX_REBAR3` - path to rebar3 command that overrides the one Mix installs\n    (default: `~/.mix/rebar3`)\n\n  * `MIX_TARGET` - specifies which target should be used. See [Targets](#module-targets)\n\n  * `MIX_XDG` - asks Mix to follow the [XDG Directory Specification](https://specifications.freedesktop.org/basedir-spec/basedir-spec-latest.html)\n    for its home directory and configuration files. This behaviour needs to\n    be opt-in due to backwards compatibility. `MIX_HOME` has higher preference\n    than `MIX_XDG`. If none of the variables are set, the default directory\n    `~/.mix` will be used\n\nEnvironment variables that are not meant to hold a value (and act basically as\nflags) should be set to either `1` or `true`, for example:\n\n    $ MIX_DEBUG=1 mix compile","ref":"Mix.html#module-environment-variables"},{"type":"function","title":"Mix.compilers/0","doc":"Returns the default compilers used by Mix.\n\nIt can be used in your `mix.exs` to prepend or\nappend new compilers to Mix:\n\n    def project do\n      [compilers: Mix.compilers() ++ [:foo, :bar]]\n    end","ref":"Mix.html#compilers/0"},{"type":"function","title":"Mix.debug/1","doc":"Sets Mix debug mode.","ref":"Mix.html#debug/1"},{"type":"function","title":"Mix.debug?/0","doc":"Returns `true` if Mix is in debug mode, `false` otherwise.","ref":"Mix.html#debug?/0"},{"type":"function","title":"Mix.ensure_application!/1","doc":"Ensures the given application from Erlang/OTP or Elixir and its dependencies\nare available in the path.\n\nThis is mostly used for Mix internal needs. In your own tasks, you should\nlist the Erlang application dependencies under the `:extra_applications`\nsection of your `mix.exs`.","ref":"Mix.html#ensure_application!/1"},{"type":"function","title":"Mix.env/0","doc":"Returns the current Mix environment.\n\nThis function should not be used at runtime in application code (as opposed\nto infrastructure and build code like Mix tasks). Mix is a build tool and may\nnot be available after the code is compiled (for example in a release).\n\nTo differentiate the program behavior depending on the environment, it is\nrecommended to use application environment through `Application.get_env/3`.\nProper configuration can be set in config files, often per-environment\n(see the `Config` module for more information).","ref":"Mix.html#env/0"},{"type":"function","title":"Mix.env/1","doc":"Changes the current Mix environment to `env`.\n\nBe careful when invoking this function as any project\nconfiguration won't be reloaded.\n\nThis function should not be used at runtime in application code\n(see `env/0` for more information).","ref":"Mix.html#env/1"},{"type":"function","title":"Mix.install/2","doc":"Installs and starts dependencies.\n\nThe given `deps` should be in the same format as defined in a regular Mix\nproject. See `mix help deps` for more information. As a shortcut, an atom\ncan be given as dependency to mean the latest version. In other words,\nspecifying `:decimal` is the same as `{:decimal, \">= 0.0.0\"}`.\n\nAfter each successful installation, a given set of dependencies is cached\nso starting another VM and calling `Mix.install/2` with the same dependencies\nwill avoid unnecessary downloads and compilations. The location of the cache\ndirectory can be controlled using the `MIX_INSTALL_DIR` environment variable.\n\nThis function can only be called outside of a Mix project and only with the\nsame dependencies in the given VM.\n\n**Note:** this feature is currently experimental and it may change\nin future releases.","ref":"Mix.html#install/2"},{"type":"function","title":"Options - Mix.install/2","doc":"* `:force` - if `true`, runs with empty install cache. This is useful when you want\n    to update your dependencies or your install got into an inconsistent state.\n    To use this option, you can also set the `MIX_INSTALL_FORCE` environment variable.\n    (Default: `false`)\n\n  * `:verbose` - if `true`, prints additional debugging information\n    (Default: `false`)\n\n  * `:consolidate_protocols` - if `true`, runs protocol\n    consolidation via the `mix compile.protocols` task (Default: `true`)\n\n  * `:elixir` - if set, ensures the current Elixir version matches the given\n    version requirement (Default: `nil`)\n\n  * `:system_env` (since v1.13.0) - a list or a map of system environment variable\n    names with respective values as binaries. The system environment is made part\n    of the `Mix.install/2` cache, so different configurations will lead to different apps\n\n  * `:config` (since v1.13.0) - a keyword list of keyword lists of compile-time\n    configuration. The configuration is part of the `Mix.install/2` cache, so\n    different configurations will lead to different apps. For this reason, you\n    want to minimize the amount of configuration set through this option.\n    Use `Application.put_all_env/4` for setting other runtime configuration.\n\n  * `:config_path` (since v1.14.0) - path to a configuration file. If a `runtime.exs`\n    file exists in the same directory as the given path, it is loaded too.\n\n  * `:lockfile` (since v1.14.0) - path to a lockfile to be used as a basis of\n    dependency resolution.\n\n  * `:start_applications` (since v1.15.3) - if `true`, ensures that installed app\n    and its dependencies are started after install (Default: `true`)","ref":"Mix.html#install/2-options"},{"type":"function","title":"Examples - Mix.install/2","doc":"Installing `:decimal` and `:jason`:\n\n    Mix.install([\n      :decimal,\n      {:jason, \"~> 1.0\"}\n    ])\n\nInstalling `:nx` and `:exla`, and configuring the underlying applications\nand environment variables:\n\n    Mix.install(\n      [:nx, :exla],\n      config: [\n        nx: [default_backend: EXLA]\n      ],\n      system_env: [\n        XLA_TARGET: \"cuda111\"\n      ]\n    )\n\nInstalling a Mix project as a path dependency along with its configuration\nand deps:\n\n    # $ git clone https://github.com/hexpm/hexpm /tmp/hexpm\n    # $ cd /tmp/hexpm && mix setup\n\n    Mix.install(\n      [\n        {:hexpm, path: \"/tmp/hexpm\", env: :dev},\n      ],\n      config_path: \"/tmp/hexpm/config/config.exs\",\n      lockfile: \"/tmp/hexpm/mix.lock\"\n    )\n\n    Hexpm.Repo.query!(\"SELECT COUNT(1) from packages\")\n    #=> ...\n\nThe example above can be simplified by passing the application\nname as an atom for `:config_path` and `:lockfile`:\n\n    Mix.install(\n      [\n        {:hexpm, path: \"/tmp/hexpm\", env: :dev},\n      ],\n      config_path: :hexpm,\n      lockfile: :hexpm\n    )","ref":"Mix.html#install/2-examples"},{"type":"function","title":"Limitations - Mix.install/2","doc":"There is one limitation to `Mix.install/2`, which is actually an Elixir\nbehaviour. If you are installing a dependency that defines a struct or\nmacro, you cannot use the struct or macro immediately after the install\ncall. For example, this won't work:\n\n    Mix.install([:decimal])\n    %Decimal{} = Decimal.new(42)\n\nThat's because Elixir first expands all structs and all macros, and then\nit executes the code. This means that, by the time Elixir tries to expand\nthe `%Decimal{}` struct, the dependency has not been installed yet.\n\nLuckily this has a straightforward solution, which is move the code to\ninside a module:\n\n    Mix.install([:decimal])\n\n    defmodule Script do\n      def run do\n        %Decimal{} = Decimal.new(42)\n      end\n    end\n\n    Script.run()\n\nThe contents inside `defmodule` will only be expanded and executed\nafter `Mix.install/2` runs, which means that any struct, macros,\nand imports will be correctly handled.","ref":"Mix.html#install/2-limitations"},{"type":"function","title":"Mix.installed?/0","doc":"Returns whether `Mix.install/2` was called in the current node.","ref":"Mix.html#installed?/0"},{"type":"function","title":"Mix.path_for/1","doc":"The path for local archives or escripts.","ref":"Mix.html#path_for/1"},{"type":"function","title":"Mix.raise/1","doc":"Raises a Mix error that is nicely formatted, defaulting to exit status `1`.","ref":"Mix.html#raise/1"},{"type":"function","title":"Mix.raise/2","doc":"Raises a Mix error that is nicely formatted.","ref":"Mix.html#raise/2"},{"type":"function","title":"Options - Mix.raise/2","doc":"* `:exit_status` - defines exit status, defaults to `1`","ref":"Mix.html#raise/2-options"},{"type":"function","title":"Mix.shell/0","doc":"Returns the current shell.\n\n`shell/0` can be used as a wrapper for the current shell. It contains\nconveniences for requesting information from the user, printing to the\nshell and so forth. The Mix shell is swappable (see `shell/1`), allowing\ndevelopers to use a test shell that simply sends messages to the current\nprocess instead of performing IO (see `Mix.Shell.Process`).\n\nBy default, this returns `Mix.Shell.IO`.","ref":"Mix.html#shell/0"},{"type":"function","title":"Examples - Mix.shell/0","doc":"Mix.shell().info(\"Preparing to do something dangerous...\")\n\n    if Mix.shell().yes?(\"Are you sure?\") do\n      # do something dangerous\n    end","ref":"Mix.html#shell/0-examples"},{"type":"function","title":"Mix.shell/1","doc":"Sets the current shell.\n\nAs an argument you may pass `Mix.Shell.IO`, `Mix.Shell.Process`,\n`Mix.Shell.Quiet`, or any module that implements the `Mix.Shell`\nbehaviour.\n\nAfter calling this function, `shell` becomes the shell that is\nreturned by `shell/0`.","ref":"Mix.html#shell/1"},{"type":"function","title":"Examples - Mix.shell/1","doc":"iex> Mix.shell(Mix.Shell.IO)\n    :ok\n\nYou can use `shell/0` and `shell/1` to temporarily switch shells,\nfor example, if you want to run a Mix Task that normally produces\na lot of output:\n\n    shell = Mix.shell()\n    Mix.shell(Mix.Shell.Quiet)\n\n    try do\n      Mix.Task.run(\"noisy.task\")\n    after\n      Mix.shell(shell)\n    end","ref":"Mix.html#shell/1-examples"},{"type":"function","title":"Mix.target/0","doc":"Returns the Mix target.","ref":"Mix.html#target/0"},{"type":"function","title":"Mix.target/1","doc":"Changes the current Mix target to `target`.\n\nBe careful when invoking this function as any project\nconfiguration won't be reloaded.","ref":"Mix.html#target/1"},{"type":"module","title":"Mix.Generator","doc":"Conveniences for working with paths and generating content.","ref":"Mix.Generator.html"},{"type":"function","title":"Mix.Generator.copy_file/3","doc":"Copies `source` to `target`.\n\nIf `target` already exists and the contents are not the same,\nit asks for user confirmation.","ref":"Mix.Generator.html#copy_file/3"},{"type":"function","title":"Options - Mix.Generator.copy_file/3","doc":"* `:force` - forces copying without a shell prompt\n  * `:quiet` - does not log command output","ref":"Mix.Generator.html#copy_file/3-options"},{"type":"function","title":"Examples - Mix.Generator.copy_file/3","doc":"iex> Mix.Generator.copy_file(\"source/gitignore\", \".gitignore\")\n    * creating .gitignore\n    true","ref":"Mix.Generator.html#copy_file/3-examples"},{"type":"function","title":"Mix.Generator.copy_template/4","doc":"Evaluates and copy templates at `source` to `target`.\n\nThe template in `source` is evaluated with the given `assigns`.\n\nIf `target` already exists and the contents are not the same,\nit asks for user confirmation.","ref":"Mix.Generator.html#copy_template/4"},{"type":"function","title":"Options - Mix.Generator.copy_template/4","doc":"* `:force` - forces copying without a shell prompt\n  * `:quiet` - does not log command output","ref":"Mix.Generator.html#copy_template/4-options"},{"type":"function","title":"Examples - Mix.Generator.copy_template/4","doc":"iex> assigns = [project_path: \"/Users/joe/newproject\"]\n    iex> Mix.Generator.copy_template(\"source/gitignore\", \".gitignore\", assigns)\n    * creating .gitignore\n    true","ref":"Mix.Generator.html#copy_template/4-examples"},{"type":"function","title":"Mix.Generator.create_directory/2","doc":"Creates a directory if one does not exist yet.\n\nThis function does nothing if the given directory already exists; in this\ncase, it still logs the directory creation.","ref":"Mix.Generator.html#create_directory/2"},{"type":"function","title":"Options - Mix.Generator.create_directory/2","doc":"* `:quiet` - does not log command output","ref":"Mix.Generator.html#create_directory/2-options"},{"type":"function","title":"Examples - Mix.Generator.create_directory/2","doc":"iex> Mix.Generator.create_directory(\"path/to/dir\")\n    * creating path/to/dir\n    true","ref":"Mix.Generator.html#create_directory/2-examples"},{"type":"function","title":"Mix.Generator.create_file/3","doc":"Creates a file with the given contents.\n\nIf the file already exists and the contents are not the same,\nit asks for user confirmation.","ref":"Mix.Generator.html#create_file/3"},{"type":"function","title":"Options - Mix.Generator.create_file/3","doc":"* `:force` - forces creation without a shell prompt\n  * `:quiet` - does not log command output","ref":"Mix.Generator.html#create_file/3-options"},{"type":"function","title":"Examples - Mix.Generator.create_file/3","doc":"iex> Mix.Generator.create_file(\".gitignore\", \"_build\\ndeps\\n\")\n    * creating .gitignore\n    true","ref":"Mix.Generator.html#create_file/3-examples"},{"type":"macro","title":"Mix.Generator.embed_template/2","doc":"Embeds a template given by `contents` into the current module.\n\nIt will define a private function with the `name` followed by\n`_template` that expects assigns as arguments.\n\nThis function must be invoked passing a keyword list.\nEach key in the keyword list can be accessed in the\ntemplate using the `@` macro.\n\nFor more information, check `EEx.SmartEngine`.","ref":"Mix.Generator.html#embed_template/2"},{"type":"macro","title":"Examples - Mix.Generator.embed_template/2","doc":"defmodule Mix.Tasks.MyTask do\n      require Mix.Generator\n      Mix.Generator.embed_template(:log, \"Log:  \")\n    end","ref":"Mix.Generator.html#embed_template/2-examples"},{"type":"macro","title":"Mix.Generator.embed_text/2","doc":"Embeds a text given by `contents` into the current module.\n\nIt will define a private function with the `name` followed by\n`_text` that expects no arguments.","ref":"Mix.Generator.html#embed_text/2"},{"type":"macro","title":"Examples - Mix.Generator.embed_text/2","doc":"defmodule Mix.Tasks.MyTask do\n      require Mix.Generator\n      Mix.Generator.embed_text(:error, \"There was an error!\")\n    end","ref":"Mix.Generator.html#embed_text/2-examples"},{"type":"function","title":"Mix.Generator.overwrite?/1","doc":"Prompts the user to overwrite the file if it exists.\n\nReturns false if the file exists and the user forbade\nto override it. Returns true otherwise.","ref":"Mix.Generator.html#overwrite?/1"},{"type":"function","title":"Mix.Generator.overwrite?/2","doc":"Prompts the user to overwrite the file if it exists.\n\nThe contents are compared to avoid asking the user to\noverride if the contents did not change. Returns false\nif the file exists and the content is the same or the\nuser forbade to override it. Returns true otherwise.","ref":"Mix.Generator.html#overwrite?/2"},{"type":"module","title":"Mix.Project","doc":"Defines and manipulates Mix projects.\n\nA Mix project is defined by calling `use Mix.Project` in a module, usually\nplaced in `mix.exs`:\n\n    defmodule MyApp.MixProject do\n      use Mix.Project\n\n      def project do\n        [\n          app: :my_app,\n          version: \"1.0.0\"\n        ]\n      end\n    end\n\n> #### `use Mix.Project` {: .info}\n>\n> When you `use Mix.Project`, it notifies Mix that a new project\n> has been defined, so all Mix tasks use your module as a starting\n> point.","ref":"Mix.Project.html"},{"type":"module","title":"Configuration - Mix.Project","doc":"In order to configure Mix, the module that `use`s `Mix.Project` should export\na `project/0` function that returns a keyword list representing configuration\nfor the project.\n\nThis configuration can be read using `Mix.Project.config/0`. Note that\n`config/0` won't fail if a project is not defined; this allows many Mix tasks\nto work without a project.\n\nIf a task requires a project to be defined or needs to access a\nspecial function within the project, the task can call `Mix.Project.get!/0`\nwhich fails with `Mix.NoProjectError` in the case a project is not\ndefined.\n\nThere isn't a comprehensive list of all the options that can be returned by\n`project/0` since many Mix tasks define their own options that they read from\nthis configuration. For example, look at the \"Configuration\" section in the\ndocumentation for the `Mix.Tasks.Compile` task.\n\nThese are a few options that are not used by just one Mix task (and will thus\nbe documented here):\n\n  * `:build_per_environment` - if `true`, builds will be *per-environment*. If\n    `false`, builds will go in `_build/shared` regardless of the Mix\n    environment. Defaults to `true`.\n\n  * `:aliases` - a list of task aliases. For more information, check out the\n    \"Aliases\" section in the documentation for the `Mix` module. Defaults to\n    `[]`.\n\n  * `:config_path` - a string representing the path of the main config\n    file. See `config_files/0` for more information. Defaults to\n    `\"config/config.exs\"`.\n\n  * `:deps` - a list of dependencies of this project. Refer to the\n    documentation for the `Mix.Tasks.Deps` task for more information. Defaults\n    to `[]`.\n\n  * `:deps_path` - directory where dependencies are stored. Also see\n    `deps_path/1`. Defaults to `\"deps\"`.\n\n  * `:lockfile` - the name of the lockfile used by the `mix deps.*` family of\n    tasks. Defaults to `\"mix.lock\"`.\n\nMix tasks may require their own configuration inside `def project`. For example,\ncheck the `Mix.Tasks.Compile` task and all the specific compiler tasks\n(such as `Mix.Tasks.Compile.Elixir` or `Mix.Tasks.Compile.Erlang`).\n\nNote that different tasks may share the same configuration option. For example,\nthe `:erlc_paths` configuration is used by `mix compile.erlang`, `mix compile.yecc`,\nand other tasks.","ref":"Mix.Project.html#module-configuration"},{"type":"module","title":"CLI configuration - Mix.Project","doc":"Mix is most often invoked from the command line. For this purpose, you may define\na specific `cli/0` function which customizes default values when executed from\nthe CLI. For example:\n\n    def cli do\n      [\n        default_task: \"phx.server\",\n        preferred_envs: [docs: :docs]\n      ]\n    end\n\nThe example above sets the default task (used by `iex -S mix` and `mix`) to\n`phx.server`. It also sets the default environment for the \"mix docs\" task to\nbe \"docs\".\n\nThe following CLI configuration are available:\n\n  * `:default_env` - the default environment to use when none is given\n    and `MIX_ENV` is not set\n\n  * `:default_target` - the default target to use when none is given\n    and `MIX_TARGET` is not set\n\n  * `:default_task` - the default task to invoke when none is given\n\n  * `:preferred_envs` - a keyword list of `{task, env}` tuples where `task`\n    is the task name as an atom (for example, `:\"deps.get\"`) and `env` is the\n    preferred environment (for example, `:test`)\n\n  * `:preferred_targets` - a keyword list of `{task, target}` tuples where\n    `task` is the task name as an atom (for example, `:test`) and `target`\n    is the preferred target (for example, `:host`)","ref":"Mix.Project.html#module-cli-configuration"},{"type":"module","title":"Erlang projects - Mix.Project","doc":"Mix can be used to manage Erlang projects that don't have any Elixir code. To\nensure Mix tasks work correctly for an Erlang project, `language: :erlang` has\nto be part of the configuration returned by `project/0`. This setting also\nmakes sure Elixir is not added as a dependency to the generated `.app` file or\nto the escript generated with `mix escript.build`, and so on.","ref":"Mix.Project.html#module-erlang-projects"},{"type":"module","title":"Invoking this module - Mix.Project","doc":"This module contains many functions that return project information and\nmetadata. However, since Mix is not included nor configured during releases,\nwe recommend using the functions in this module only inside Mix tasks.\nIf you need to configure your own app, consider using the application\nenvironment instead. For example, don't do this:\n\n    def some_config do\n      Mix.Project.config()[:some_config]\n    end\n\nNor this:\n\n    @some_config Mix.Project.config()[:some_config]\n\nInstead, do this:\n\n    def some_config do\n      Application.get_env(:my_app, :some_config)\n    end\n\nOr this:\n\n    @some_config Application.compile_env(:my_app, :some_config)","ref":"Mix.Project.html#module-invoking-this-module"},{"type":"function","title":"Mix.Project.app_path/1","doc":"Returns the application path inside the build.\n\nThe returned path will be expanded.","ref":"Mix.Project.html#app_path/1"},{"type":"function","title":"Examples - Mix.Project.app_path/1","doc":"If your project defines the app `my_app`:\n\n    Mix.Project.app_path()\n    #=> \"/path/to/project/_build/shared/lib/my_app\"","ref":"Mix.Project.html#app_path/1-examples"},{"type":"function","title":"Mix.Project.apps_paths/1","doc":"Returns a map with the umbrella child applications paths.\n\nThese paths are based on the `:apps_path` and `:apps` configurations.\n\nIf the given project configuration identifies an umbrella project, the return\nvalue is a map of `app => path` where `app` is a child app of the umbrella and\n`path` is its path relative to the root of the umbrella project.\n\nIf the given project configuration does not identify an umbrella project,\n`nil` is returned.","ref":"Mix.Project.html#apps_paths/1"},{"type":"function","title":"Examples - Mix.Project.apps_paths/1","doc":"Mix.Project.apps_paths()\n    #=> %{my_app1: \"apps/my_app1\", my_app2: \"apps/my_app2\"}","ref":"Mix.Project.html#apps_paths/1-examples"},{"type":"function","title":"Mix.Project.build_path/1","doc":"Returns the build path for the given project.\n\nThe build path is built based on the `:build_path` configuration\n(which defaults to `\"_build\"`) and a subdirectory. The subdirectory\nis built based on two factors:\n\n  * If `:build_per_environment` is set, the subdirectory is the value\n    of `Mix.env/0` (which can be set via `MIX_ENV`). Otherwise it is\n    set to \"shared\".\n\n  * If `Mix.target/0` is set (often via the `MIX_TARGET` environment\n    variable), it will be used as a prefix to the subdirectory.\n\nFinally, the environment variables `MIX_BUILD_ROOT` and `MIX_BUILD_PATH`\ncan be used to change the result of this function. `MIX_BUILD_ROOT`\noverwrites only the root `\"_build\"` directory while keeping the\nsubdirectory as is. It may be useful to change it for caching reasons,\ntypically during Continuous Integration (CI). `MIX_BUILD_PATH` overrides\nthe build path altogether and it typically used by other build tools\nthat invoke the `mix` CLI.\n\n> ##","ref":"Mix.Project.html#build_path/1"},{"type":"function","title":"Naming differences {: .info} - Mix.Project.build_path/1","doc":">\n> Ideally the configuration option `:build_path` would be called\n> `:build_root`, as it would fully mirror the environment variable.\n> However, its name is preserved for backwards compatibility.","ref":"Mix.Project.html#build_path/1-naming-differences-info"},{"type":"function","title":"Examples - Mix.Project.build_path/1","doc":"Mix.Project.build_path()\n    #=> \"/path/to/project/_build/shared\"\n\nIf `:build_per_environment` is set to `true`, it will create a new build per\nenvironment:\n\n    Mix.env()\n    #=> :dev\n    Mix.Project.build_path()\n    #=> \"/path/to/project/_build/dev\"","ref":"Mix.Project.html#build_path/1-examples"},{"type":"function","title":"Mix.Project.build_structure/2","doc":"Builds the project structure for the given application.","ref":"Mix.Project.html#build_structure/2"},{"type":"function","title":"Options - Mix.Project.build_structure/2","doc":"* `:symlink_ebin` - symlink ebin instead of copying it","ref":"Mix.Project.html#build_structure/2-options"},{"type":"function","title":"Mix.Project.clear_deps_cache/0","doc":"Clears the dependency for the current environment.\n\nUseful when dependencies need to be reloaded due to change of global state.\n\nFor example, Nerves uses this function to force all dependencies to be\nreloaded after it updates the system environment. It goes roughly like\nthis:\n\n  1. Nerves fetches all dependencies and looks for the system specific deps\n  2. Once the system specific dep is found, it loads it alongside env vars\n  3. Nerves then clears the cache, forcing dependencies to be loaded again\n  4. Dependencies are loaded again, now with an updated env environment","ref":"Mix.Project.html#clear_deps_cache/0"},{"type":"function","title":"Mix.Project.compile_path/1","doc":"Returns the paths the given project compiles to.\n\nIf no configuration is given, the one for the current project will be used.\n\nThe returned path will be expanded.","ref":"Mix.Project.html#compile_path/1"},{"type":"function","title":"Examples - Mix.Project.compile_path/1","doc":"If your project defines the app `my_app`:\n\n    Mix.Project.compile_path()\n    #=> \"/path/to/project/_build/dev/lib/my_app/ebin\"","ref":"Mix.Project.html#compile_path/1-examples"},{"type":"function","title":"Mix.Project.config/0","doc":"Returns the project configuration.\n\nIf there is no project defined, it still returns a keyword\nlist with default values. This allows many Mix tasks to work\nwithout the need for an underlying project.\n\nNote this configuration is cached once the project is\npushed onto the stack. Calling it multiple times won't\ncause it to be recomputed.\n\nDo not use `Mix.Project.config/0` to find the runtime configuration.\nUse it only to configure aspects of your project (like\ncompilation directories) and not your application runtime.","ref":"Mix.Project.html#config/0"},{"type":"function","title":"Mix.Project.config_files/0","doc":"Returns a list of project configuration files for this project.\n\nThis function is usually used in compilation tasks to trigger\na full recompilation whenever such configuration files change.\n\nIt returns the lock manifest, and all config files in the `config`\ndirectory that do not start with a leading period (for example,\n`.my_config.exs`).\n\nNote: before Elixir v1.13.0, the `mix.exs` file was also included\nas a config file, but since then it has been moved to its own\nfunction called `project_file/0`.","ref":"Mix.Project.html#config_files/0"},{"type":"function","title":"Mix.Project.config_mtime/0","doc":"Returns the latest modification time from config files.\n\nThis function is usually used in compilation tasks to trigger\na full recompilation whenever such configuration files change.\nFor this reason, the mtime is cached to avoid file system lookups.\n\nNote: before Elixir v1.13.0, the `mix.exs` file was also included\nin the mtimes, but not anymore. You can compute its modification\ndate by calling `project_file/0`.","ref":"Mix.Project.html#config_mtime/0"},{"type":"function","title":"Mix.Project.consolidation_path/1","doc":"Returns the path where protocol consolidations are stored.\n\nThe returned path will be expanded.","ref":"Mix.Project.html#consolidation_path/1"},{"type":"function","title":"Examples - Mix.Project.consolidation_path/1","doc":"If your project defines the app `my_app`:\n\n    Mix.Project.consolidation_path()\n    #=> \"/path/to/project/_build/dev/lib/my_app/consolidated\"\n\nInside umbrellas:\n\n    Mix.Project.consolidation_path()\n    #=> \"/path/to/project/_build/dev/consolidated\"","ref":"Mix.Project.html#consolidation_path/1-examples"},{"type":"function","title":"Mix.Project.deps_apps/0","doc":"Returns all dependencies app names.\n\nThe order they are returned is guaranteed to be sorted\nfor proper dependency resolution. For example, if A\ndepends on B, then B will listed before A.","ref":"Mix.Project.html#deps_apps/0"},{"type":"function","title":"Mix.Project.deps_path/1","doc":"Returns the path where dependencies are stored for the given project.\n\nIf no configuration is given, the one for the current project is used.\n\nThe returned path will be expanded.","ref":"Mix.Project.html#deps_path/1"},{"type":"function","title":"Examples - Mix.Project.deps_path/1","doc":"Mix.Project.deps_path()\n    #=> \"/path/to/project/deps\"","ref":"Mix.Project.html#deps_path/1-examples"},{"type":"function","title":"Mix.Project.deps_paths/1","doc":"Returns the full path of all dependencies as a map.","ref":"Mix.Project.html#deps_paths/1"},{"type":"function","title":"Options - Mix.Project.deps_paths/1","doc":"* `:depth` - only returns dependencies to the depth level,\n    a depth of `1` will only return top-level dependencies\n  * `:parents` - starts the dependency traversal from the\n    given parents instead of the application root","ref":"Mix.Project.html#deps_paths/1-options"},{"type":"function","title":"Examples - Mix.Project.deps_paths/1","doc":"Mix.Project.deps_paths()\n    #=> %{foo: \"deps/foo\", bar: \"custom/path/dep\"}","ref":"Mix.Project.html#deps_paths/1-examples"},{"type":"function","title":"Mix.Project.deps_scms/1","doc":"Returns the SCMs of all dependencies as a map.\n\nSee `Mix.SCM` module documentation to learn more about SCMs.","ref":"Mix.Project.html#deps_scms/1"},{"type":"function","title":"Options - Mix.Project.deps_scms/1","doc":"* `:depth` - only returns dependencies to the depth level,\n    a depth of `1` will only return top-level dependencies\n  * `:parents` - starts the dependency traversal from the\n    given parents instead of the application root","ref":"Mix.Project.html#deps_scms/1-options"},{"type":"function","title":"Examples - Mix.Project.deps_scms/1","doc":"Mix.Project.deps_scms()\n    #=> %{foo: Mix.SCM.Path, bar: Mix.SCM.Git}","ref":"Mix.Project.html#deps_scms/1-examples"},{"type":"function","title":"Mix.Project.deps_tree/1","doc":"Returns the dependencies of all dependencies as a map.","ref":"Mix.Project.html#deps_tree/1"},{"type":"function","title":"Options - Mix.Project.deps_tree/1","doc":"* `:depth` - only returns dependencies to the depth level,\n    a depth of `1` will only return top-level dependencies\n  * `:parents` - starts the dependency traversal from the\n    given parents instead of the application root","ref":"Mix.Project.html#deps_tree/1-options"},{"type":"function","title":"Examples - Mix.Project.deps_tree/1","doc":"Mix.Project.deps_tree()\n    #=> %{foo: [:bar, :baz], bar: [], baz: []}","ref":"Mix.Project.html#deps_tree/1-examples"},{"type":"function","title":"Mix.Project.ensure_structure/2","doc":"Ensures the project structure for the given project exists.\n\nIn case it does exist, it is a no-op. Otherwise, it is built.\n\n`opts` are the same options that can be passed to `build_structure/2`.","ref":"Mix.Project.html#ensure_structure/2"},{"type":"function","title":"Mix.Project.get/0","doc":"Retrieves the current project if there is one.\n\nIf there is no current project, `nil` is returned. This\nmay happen in cases there is no `mix.exs` in the current\ndirectory.\n\nIf you expect a project to be defined, i.e., it is a\nrequirement of the current task, you should call\n`get!/0` instead.","ref":"Mix.Project.html#get/0"},{"type":"function","title":"Mix.Project.get!/0","doc":"Same as `get/0`, but raises an exception if there is no current project.\n\nThis is usually called by tasks that need additional\nfunctions on the project to be defined. Since such\ntasks usually depend on a project being defined, this\nfunction raises a `Mix.NoProjectError` exception in\ncase no project is available.","ref":"Mix.Project.html#get!/0"},{"type":"function","title":"Mix.Project.in_project/4","doc":"Runs the given `fun` inside the given project.\n\nThis function changes the current working directory and\nloads the project at the given directory onto the project\nstack.\n\nA `post_config` can be passed that will be merged into\nthe project configuration.\n\n`fun` is called with the module name of the given `Mix.Project`.\nThe return value of this function is the return value of `fun`.","ref":"Mix.Project.html#in_project/4"},{"type":"function","title":"Examples - Mix.Project.in_project/4","doc":"Mix.Project.in_project(:my_app, \"/path/to/my_app\", fn module ->\n      \"Mix project is: #{inspect(module)}\"\n    end)\n    #=> \"Mix project is: MyApp.MixProject\"","ref":"Mix.Project.html#in_project/4-examples"},{"type":"function","title":"Mix.Project.load_paths/1","doc":"","ref":"Mix.Project.html#load_paths/1"},{"type":"function","title":"Mix.Project.manifest_path/1","doc":"Returns the path where manifests are stored.\n\nBy default they are stored in the app path inside\nthe build directory. Umbrella applications have\nthe manifest path set to the root of the build directory.\nDirectories may be changed in future releases.\n\nThe returned path will be expanded.","ref":"Mix.Project.html#manifest_path/1"},{"type":"function","title":"Examples - Mix.Project.manifest_path/1","doc":"If your project defines the app `my_app`:\n\n    Mix.Project.manifest_path()\n    #=> \"/path/to/project/_build/shared/lib/my_app/.mix\"","ref":"Mix.Project.html#manifest_path/1-examples"},{"type":"function","title":"Mix.Project.parent_umbrella_project_file/0","doc":"Returns the path to the file that defines the parent umbrella project, if one.\n\nThe majority of the time, it will point to a `mix.exs` file.\nReturns `nil` if not inside a project or not inside an umbrella.","ref":"Mix.Project.html#parent_umbrella_project_file/0"},{"type":"function","title":"Mix.Project.project_file/0","doc":"Returns the path to the file that defines the current project.\n\nThe majority of the time, it will point to a `mix.exs` file.\nReturns `nil` if not inside a project.","ref":"Mix.Project.html#project_file/0"},{"type":"function","title":"Mix.Project.umbrella?/1","doc":"Returns `true` if `config` is the configuration for an umbrella project.\n\nWhen called with no arguments, tells whether the current project is\nan umbrella project.","ref":"Mix.Project.html#umbrella?/1"},{"type":"module","title":"Mix.Release","doc":"Defines the release structure and convenience for assembling releases.","ref":"Mix.Release.html"},{"type":"function","title":"Mix.Release.__struct__/0","doc":"The Mix.Release struct has the following read-only fields:\n\n  * `:name` - the name of the release as an atom\n  * `:version` - the version of the release as a string\n  * `:path` - the path to the release root\n  * `:version_path` - the path to the release version inside the release\n  * `:applications` - a map of application with their definitions\n  * `:erts_source` - the ERTS source as a charlist (or nil)\n  * `:erts_version` - the ERTS version as a charlist\n\nThe following fields may be modified as long as they keep their defined types:\n\n  * `:boot_scripts` - a map of boot scripts with the boot script name\n    as key and a keyword list with **all** applications that are part of\n    it and their modes as value\n  * `:config_providers` - a list of `{config_provider, term}` tuples where the\n    first element is a module that implements the `Config.Provider` behaviour\n    and `term` is the value given to it on `c:Config.Provider.init/1`\n  * `:options` - a keyword list with all other user supplied release options\n  * `:overlays` - a list of extra files added to the release. If you have a custom\n    step adding extra files to a release, you can add these files to the `:overlays`\n    field so they are also considered on further commands, such as tar/zip. Each entry\n    in overlays is the relative path to the release root of each file\n  * `:steps` - a list of functions that receive the release and returns a release.\n    Must also contain the atom `:assemble` which is the internal assembling step.\n    May also contain the atom `:tar` to create a tarball of the release.","ref":"Mix.Release.html#__struct__/0"},{"type":"function","title":"Mix.Release.copy_app/2","doc":"Copies the given application specification into the release.\n\nIt assumes the application exists in the release.","ref":"Mix.Release.html#copy_app/2"},{"type":"function","title":"Mix.Release.copy_ebin/3","doc":"Copies the ebin directory at `source` to `target`\nrespecting release options such a `:strip_beams`.","ref":"Mix.Release.html#copy_ebin/3"},{"type":"function","title":"Mix.Release.copy_erts/1","doc":"Copies ERTS if the release is configured to do so.\n\nReturns true if the release was copied, false otherwise.","ref":"Mix.Release.html#copy_erts/1"},{"type":"function","title":"Mix.Release.make_boot_script/4","doc":"Makes boot scripts.\n\nIt receives a path to the boot file, without extension, such as\n`releases/0.1.0/start` and this command will write `start.rel`,\n`start.boot`, and `start.script` to the given path, returning\n`{:ok, rel_path}` or `{:error, message}`.\n\nThe boot script uses the RELEASE_LIB environment variable, which must\nbe accordingly set with `--boot-var` and point to the release lib dir.","ref":"Mix.Release.html#make_boot_script/4"},{"type":"function","title":"Mix.Release.make_cookie/2","doc":"Copies the cookie to the given path.\n\nIf a cookie option was given, we compare it with\nthe contents of the file (if any), and ask the user\nif they want to override.\n\nIf there is no option, we generate a random one\nthe first time.","ref":"Mix.Release.html#make_cookie/2"},{"type":"function","title":"Mix.Release.make_start_erl/2","doc":"Makes the start_erl.data file with the\nERTS version and release versions.","ref":"Mix.Release.html#make_start_erl/2"},{"type":"function","title":"Mix.Release.make_sys_config/3","doc":"Makes the `sys.config` structure.\n\nIf there are config providers, then a value is injected into\nthe `:elixir` application configuration in `sys_config` to be\nread during boot and trigger the providers.\n\nIt uses the following release options to customize its behaviour:\n\n  * `:reboot_system_after_config`\n  * `:start_distribution_during_config`\n  * `:prune_runtime_sys_config_after_boot`\n\nIn case there are no config providers, it doesn't change `sys_config`.","ref":"Mix.Release.html#make_sys_config/3"},{"type":"function","title":"Mix.Release.rel_templates_path/2","doc":"Finds a template path for the release.","ref":"Mix.Release.html#rel_templates_path/2"},{"type":"function","title":"Mix.Release.strip_beam/2","doc":"Strips a beam file for a release.\n\nThis keeps only significant chunks necessary for the VM operation,\ndiscarding documentation, debug info, compile information and others.\n\nThe exact chunks that are kept are not documented and may change in\nfuture versions.","ref":"Mix.Release.html#strip_beam/2"},{"type":"type","title":"Mix.Release.application/0","doc":"","ref":"Mix.Release.html#t:application/0"},{"type":"type","title":"Mix.Release.mode/0","doc":"","ref":"Mix.Release.html#t:mode/0"},{"type":"type","title":"Mix.Release.t/0","doc":"","ref":"Mix.Release.html#t:t/0"},{"type":"behaviour","title":"Mix.SCM","doc":"This module provides helper functions and defines the\nbehaviour required by any source code manager (SCM) used by Mix.","ref":"Mix.SCM.html"},{"type":"callback","title":"Mix.SCM.accepts_options/2","doc":"This behaviour function receives a keyword list of `opts`\nand should return an updated list in case the SCM consumes\nthe available options. For example, when a developer specifies\na dependency:\n\n    {:foo, \"0.1.0\", github: \"foo/bar\"}\n\nEach registered SCM will be asked if they consume this dependency,\nreceiving `[github: \"foo/bar\"]` as argument. Since this option makes\nsense for the Git SCM, it will return an update list of options\nwhile other SCMs would simply return `nil`.","ref":"Mix.SCM.html#c:accepts_options/2"},{"type":"function","title":"Mix.SCM.append/1","doc":"Appends the given SCM module to the list of available SCMs.","ref":"Mix.SCM.html#append/1"},{"type":"function","title":"Mix.SCM.available/0","doc":"Returns all available SCMs. Each SCM is tried in order\nuntil a matching one is found.","ref":"Mix.SCM.html#available/0"},{"type":"callback","title":"Mix.SCM.checked_out?/1","doc":"This behaviour function returns a boolean if the\ndependency is available.","ref":"Mix.SCM.html#c:checked_out?/1"},{"type":"callback","title":"Mix.SCM.checkout/1","doc":"This behaviour function checks out dependencies.\n\nIf the dependency is locked, a lock is received in `opts`\nand the repository must be check out at the lock. Otherwise,\nno lock is given and the repository can be checked out\nto the latest version.\n\nIt must return the current lock.","ref":"Mix.SCM.html#c:checkout/1"},{"type":"callback","title":"Mix.SCM.equal?/2","doc":"Receives two options and must return `true` if they refer to the\nsame repository. The options are guaranteed to belong to the\nsame SCM.","ref":"Mix.SCM.html#c:equal?/2"},{"type":"callback","title":"Mix.SCM.fetchable?/0","doc":"Returns a boolean if the dependency can be fetched\nor it is meant to be previously available in the\nfile system.\n\nLocal dependencies (i.e. non-fetchable ones) are automatically\nrecompiled every time the parent project is compiled.","ref":"Mix.SCM.html#c:fetchable?/0"},{"type":"callback","title":"Mix.SCM.format/1","doc":"Returns a string representing the SCM. This is used\nwhen printing the dependency and not for inspection,\nso the amount of information should be concise and\neasy to spot.","ref":"Mix.SCM.html#c:format/1"},{"type":"callback","title":"Mix.SCM.format_lock/1","doc":"Returns a string representing the SCM. This is used\nwhen printing the dependency and not for inspection,\nso the amount of information should be concise and\neasy to spot.\n\nIf nil is returned, it means no lock information is available.","ref":"Mix.SCM.html#c:format_lock/1"},{"type":"callback","title":"Mix.SCM.lock_status/1","doc":"This behaviour function checks the status of the lock. In\nparticular, it checks if the revision stored in the lock\nis the same as the repository it is currently in.\n\nIt may return:\n\n  * `:mismatch` - if the lock doesn't match and we need to\n    simply move to the latest lock\n\n  * `:outdated` - the repository options are outdated in the\n    lock and we need to trigger a full update\n\n  * `:ok` - everything is fine\n\nThe lock is sent via `opts[:lock]` but it may not always be\navailable. In such cases, if the SCM requires a lock, it must\nreturn `:mismatch`, otherwise simply `:ok`.\n\nNote the lock may also belong to another SCM and as such, an\nstructural check is required. A structural mismatch should always\nreturn `:outdated`.","ref":"Mix.SCM.html#c:lock_status/1"},{"type":"callback","title":"Mix.SCM.managers/1","doc":"Returns the usable managers for the dependency. This can be used\nif the SCM has extra knowledge of the dependency, otherwise it\nshould return an empty list.","ref":"Mix.SCM.html#c:managers/1"},{"type":"function","title":"Mix.SCM.prepend/1","doc":"Prepends the given SCM module to the list of available SCMs.","ref":"Mix.SCM.html#prepend/1"},{"type":"callback","title":"Mix.SCM.update/1","doc":"This behaviour function updates dependencies. It may be\ncalled by `deps.get` or `deps.update`.\n\nIn the first scenario, a lock is received in `opts` and\nthe repository must be updated to the lock. In the second,\nno lock is given and the repository can be updated freely.\n\nIt must return the current lock.","ref":"Mix.SCM.html#c:update/1"},{"type":"type","title":"Mix.SCM.opts/0","doc":"","ref":"Mix.SCM.html#t:opts/0"},{"type":"type","title":"Mix.SCM.t/0","doc":"A module implementing the `Mix.SCM` behaviour.","ref":"Mix.SCM.html#t:t/0"},{"type":"behaviour","title":"Mix.Shell","doc":"Defines `Mix.Shell` contract.","ref":"Mix.Shell.html"},{"type":"callback","title":"Mix.Shell.cmd/1","doc":"Executes the given command and returns its exit status.\n\nShortcut for `c:cmd/2` with empty options.","ref":"Mix.Shell.html#c:cmd/1"},{"type":"callback","title":"Mix.Shell.cmd/2","doc":"Executes the given command and returns its exit status.","ref":"Mix.Shell.html#c:cmd/2"},{"type":"callback","title":"Options - Mix.Shell.cmd/2","doc":"This callback should support the following options:\n\n  * `:print_app` - when `false`, does not print the app name\n    when the command outputs something\n\n  * `:stderr_to_stdout` - when `false`, does not redirect\n    stderr to stdout\n\n  * `:quiet` - when `true`, do not print the command output\n\n  * `:env` - environment options to the executed command\n\n  * `:cd` - (since v1.11.0) the directory to run the command in\n\nAll the built-in shells support these.","ref":"Mix.Shell.html#c:cmd/2-options"},{"type":"function","title":"Mix.Shell.cmd/3","doc":"Executes the given `command` as a shell command and\ninvokes the `callback` for the streamed response.\n\nThis is most commonly used by shell implementations\nbut can also be invoked directly.\n\n`callback` takes the output data of the command. Its\nreturn value is ignored.","ref":"Mix.Shell.html#cmd/3"},{"type":"function","title":"Options - Mix.Shell.cmd/3","doc":"* `:cd` - (since v1.11.0) the directory to run the command in\n\n  * `:stderr_to_stdout` - redirects stderr to stdout, defaults to true\n\n  * `:env` - a list of environment variables, defaults to `[]`\n\n  * `:quiet` - overrides the callback to no-op","ref":"Mix.Shell.html#cmd/3-options"},{"type":"callback","title":"Mix.Shell.error/1","doc":"Prints the given ANSI error to the shell.","ref":"Mix.Shell.html#c:error/1"},{"type":"callback","title":"Mix.Shell.info/1","doc":"Prints the given ANSI message to the shell.","ref":"Mix.Shell.html#c:info/1"},{"type":"callback","title":"Mix.Shell.print_app/0","doc":"Prints the current application to the shell if\nit was not printed yet.","ref":"Mix.Shell.html#c:print_app/0"},{"type":"function","title":"Mix.Shell.printable_app_name/0","doc":"Returns the printable app name.\n\nThis function returns the current application name,\nbut only if the application name should be printed.\n\nCalling this function automatically toggles its value\nto `false` until the current project is re-entered. The\ngoal is to avoid printing the application name\nmultiple times.","ref":"Mix.Shell.html#printable_app_name/0"},{"type":"callback","title":"Mix.Shell.prompt/1","doc":"Prompts the user for input.","ref":"Mix.Shell.html#c:prompt/1"},{"type":"callback","title":"Mix.Shell.yes?/1","doc":"Prompts the user for confirmation.\n\nShortcut for `yes?/2` with empty options.","ref":"Mix.Shell.html#c:yes?/1"},{"type":"callback","title":"Mix.Shell.yes?/2","doc":"Prompts the user for confirmation.","ref":"Mix.Shell.html#c:yes?/2"},{"type":"callback","title":"Options - Mix.Shell.yes?/2","doc":"* `:default` - `:yes` or `:no` (the default is `:yes`)","ref":"Mix.Shell.html#c:yes?/2-options"},{"type":"module","title":"Mix.Shell.IO","doc":"This is Mix's default shell.\n\nIt simply prints messages to stdio and stderr.","ref":"Mix.Shell.IO.html"},{"type":"function","title":"Mix.Shell.IO.cmd/2","doc":"Executes the given command and prints its output\nto stdout as it comes.","ref":"Mix.Shell.IO.html#cmd/2"},{"type":"function","title":"Mix.Shell.IO.error/1","doc":"Prints the given ANSI error to the shell followed by a newline.","ref":"Mix.Shell.IO.html#error/1"},{"type":"function","title":"Mix.Shell.IO.info/1","doc":"Prints the given ANSI message to the shell followed by a newline.","ref":"Mix.Shell.IO.html#info/1"},{"type":"function","title":"Mix.Shell.IO.print_app/0","doc":"Prints the current application to the shell if it\nwas not printed yet.","ref":"Mix.Shell.IO.html#print_app/0"},{"type":"function","title":"Mix.Shell.IO.prompt/1","doc":"Prints a message and prompts the user for input.\n\nInput will be consumed until Enter is pressed.","ref":"Mix.Shell.IO.html#prompt/1"},{"type":"function","title":"Mix.Shell.IO.yes?/2","doc":"Prints a message and asks the user to confirm if they\nwant to proceed. The user must type and submit one of\n\"y\", \"yes\", \"Y\", \"YES\" or \"Yes\".\n\nThe user may also press Enter; this can be configured\nto either accept or reject the prompt. The latter case\nmay be useful for a potentially dangerous operation that\nshould require explicit confirmation from the user.","ref":"Mix.Shell.IO.html#yes?/2"},{"type":"function","title":"Options - Mix.Shell.IO.yes?/2","doc":"* `:default` - (:yes or :no) if `:yes` pressing Enter\n    accepts the prompt; if `:no` pressing Enter rejects\n    the prompt instead. Defaults to `:yes`.","ref":"Mix.Shell.IO.html#yes?/2-options"},{"type":"function","title":"Examples - Mix.Shell.IO.yes?/2","doc":"if Mix.shell().yes?(\"Are you sure?\") do\n      # do something...\n    end","ref":"Mix.Shell.IO.html#yes?/2-examples"},{"type":"module","title":"Mix.Shell.Process","doc":"Mix shell that uses the current process mailbox for communication.\n\nThis module provides a Mix shell implementation that uses\nthe current process mailbox for communication instead of IO.\n\nAs an example, when `Mix.shell().info(\"hello\")` is called,\nthe following message will be sent to the calling process:\n\n    {:mix_shell, :info, [\"hello\"]}\n\nThis is mainly useful in tests, allowing us to assert\nif given messages were received or not instead of performing\nchecks on some captured IO. There is also a `flush/1` function\nresponsible for flushing all `:mix_shell` related messages\nfrom the process inbox.","ref":"Mix.Shell.Process.html"},{"type":"module","title":"Examples - Mix.Shell.Process","doc":"The first step is to set the Mix shell to this module:\n\n    Mix.shell(Mix.Shell.Process)\n\nThen if your Mix task calls:\n\n    Mix.shell().info(\"hello\")\n\nYou should be able to receive it in your tests as long as\nthey run in the same process:\n\n    assert_receive {:mix_shell, :info, [msg]}\n    assert msg == \"hello\"\n\nYou can respond to prompts in tests as well:\n\n    send(self(), {:mix_shell_input, :prompt, \"Pretty cool\"})\n    Mix.shell().prompt(\"How cool was that?!\")\n    #=> \"Pretty cool\"","ref":"Mix.Shell.Process.html#module-examples"},{"type":"function","title":"Mix.Shell.Process.cmd/2","doc":"Executes the given command and forwards its messages to\nthe current process.","ref":"Mix.Shell.Process.html#cmd/2"},{"type":"function","title":"Mix.Shell.Process.error/1","doc":"Forwards the error to the current process.","ref":"Mix.Shell.Process.html#error/1"},{"type":"function","title":"Mix.Shell.Process.flush/1","doc":"Flushes all `:mix_shell` and `:mix_shell_input` messages from the current process.\n\nIf a callback is given, it is invoked for each received message.","ref":"Mix.Shell.Process.html#flush/1"},{"type":"function","title":"Examples - Mix.Shell.Process.flush/1","doc":"flush(&IO.inspect/1)","ref":"Mix.Shell.Process.html#flush/1-examples"},{"type":"function","title":"Mix.Shell.Process.info/1","doc":"Forwards the message to the current process.","ref":"Mix.Shell.Process.html#info/1"},{"type":"function","title":"Mix.Shell.Process.print_app/0","doc":"Prints the current application if it\nwas not printed yet.","ref":"Mix.Shell.Process.html#print_app/0"},{"type":"function","title":"Mix.Shell.Process.prompt/1","doc":"Forwards the message to the current process.\n\nIt also checks the inbox for an input message matching:\n\n    {:mix_shell_input, :prompt, value}\n\nIf one does not exist, it will abort since there was no shell\nprocess inputs given. `value` must be a string.","ref":"Mix.Shell.Process.html#prompt/1"},{"type":"function","title":"Examples - Mix.Shell.Process.prompt/1","doc":"The following will answer with `\"Meg\"` to the prompt\n`\"What's your name?\"`:\n\n    # The response is sent before calling prompt/1 so that prompt/1 can read it\n    send(self(), {:mix_shell_input, :prompt, \"Meg\"})\n    Mix.shell().prompt(\"What's your name?\")","ref":"Mix.Shell.Process.html#prompt/1-examples"},{"type":"function","title":"Mix.Shell.Process.yes?/2","doc":"Forwards the message to the current process.\n\nIt also checks the inbox for an input message matching:\n\n    {:mix_shell_input, :yes?, value}\n\nIf one does not exist, it will abort since there was no shell\nprocess inputs given. `value` must be `true` or `false`.","ref":"Mix.Shell.Process.html#yes?/2"},{"type":"function","title":"Example - Mix.Shell.Process.yes?/2","doc":"# Send the response to self() first so that yes?/2 will be able to read it\n    send(self(), {:mix_shell_input, :yes?, true})\n    Mix.shell().yes?(\"Are you sure you want to continue?\")","ref":"Mix.Shell.Process.html#yes?/2-example"},{"type":"module","title":"Mix.Shell.Quiet","doc":"This is Mix's default shell when the `MIX_QUIET` environment\nvariable is set.\n\nIt's just like `Mix.Shell.IO`, but prints far less.","ref":"Mix.Shell.Quiet.html"},{"type":"function","title":"Mix.Shell.Quiet.cmd/2","doc":"Executes the given command quietly without outputting anything.","ref":"Mix.Shell.Quiet.html#cmd/2"},{"type":"function","title":"Mix.Shell.Quiet.error/1","doc":"Prints the error to the shell followed by a newline.","ref":"Mix.Shell.Quiet.html#error/1"},{"type":"function","title":"Mix.Shell.Quiet.info/1","doc":"Prints nothing to the shell.","ref":"Mix.Shell.Quiet.html#info/1"},{"type":"function","title":"Mix.Shell.Quiet.print_app/0","doc":"Prints the current application if it\nwas not printed yet.","ref":"Mix.Shell.Quiet.html#print_app/0"},{"type":"function","title":"Mix.Shell.Quiet.prompt/1","doc":"Prints a message and prompts the user for input.\n\nInput will be consumed until Enter is pressed.","ref":"Mix.Shell.Quiet.html#prompt/1"},{"type":"function","title":"Mix.Shell.Quiet.yes?/2","doc":"Prints a message and asks the user to confirm if they\nwant to proceed. The user must type and submit one of\n\"y\", \"yes\", \"Y\", \"YES\" or \"Yes\".\n\nThe user may also press Enter; this can be configured\nto either accept or reject the prompt. The latter case\nmay be useful for a potentially dangerous operation that\nshould require explicit confirmation from the user.","ref":"Mix.Shell.Quiet.html#yes?/2"},{"type":"function","title":"Options - Mix.Shell.Quiet.yes?/2","doc":"* `:default` - (:yes or :no) if `:yes` pressing Enter\n    accepts the prompt; if `:no` pressing Enter rejects\n    the prompt instead. Defaults to `:yes`.","ref":"Mix.Shell.Quiet.html#yes?/2-options"},{"type":"behaviour","title":"Mix.Task","doc":"Provides conveniences for creating, loading, and manipulating Mix tasks.\n\nA Mix task can be defined by `use Mix.Task` in a module whose name\nbegins with `Mix.Tasks.` and which defines the `run/1` function.\nTypically, task modules live inside the `lib/mix/tasks/` directory,\nand their file names use dot separators instead of underscores\n(e.g. `deps.clean.ex`) - although ultimately the file name is not\nrelevant.\n\nFor example:\n\n    # lib/mix/tasks/echo.ex\n    defmodule Mix.Tasks.Echo do\n      @moduledoc \"Printed when the user requests `mix help echo`\"\n      @shortdoc \"Echoes arguments\"\n\n      use Mix.Task\n\n      @impl Mix.Task\n      def run(args) do\n        Mix.shell().info(Enum.join(args, \" \"))\n      end\n    end\n\nThe command name will correspond to the portion of the module\nname following `Mix.Tasks.`. For example, a module name of\n`Mix.Tasks.Deps.Clean` corresponds to a task name of `deps.clean`.\n\nThe `run/1` function will receive a list of all command line\narguments passed, according to the user's terminal.\n\nFor example, if the `args` in the above `echo` task were\ninspected, you might see something like this:\n\n    $ mix echo 'A and B' C --test\n    [\"A and B\", \"C\", \"--test\"]\n\n> #### `use Mix.Task` {: .info}\n>\n> When you `use Mix.Task`, the `Mix.Task` module will\n> set `@behaviour Mix.Task` and define default values\n> for the module attributes documented in the section\n> below.","ref":"Mix.Task.html"},{"type":"behaviour","title":"Module attributes - Mix.Task","doc":"Define the `@shortdoc` attribute if you wish to make the task\npublicly visible on `mix help`. Omit this attribute if you do\nnot want your task to be listed via `mix help`.\nThe `@moduledoc` attribute may override `@shortdoc`. The task\nwill not appear in `mix help` if documentation for the entire\nmodule is hidden with `@moduledoc false`.\n\nIf a task has requirements, they can be listed using the\n`@requirements` attribute. For example:\n\n    @requirements [\"app.config\"]\n\nA task will typically depend on one of the following tasks:\n\n  * \"loadpaths\" - this ensures dependencies are available\n    and compiled. If you are publishing a task as part of\n    a library to be used by others, and your task does not\n    need to interact with the user code in any way, this is\n    the recommended requirement\n\n  * \"app.config\" - additionally compiles and loads the runtime\n    configuration for the current project. If you are creating\n    a task to be used within your application or as part of a\n    library, which must invoke or interact with the user code,\n    this is the minimum recommended requirement\n\n  * \"app.start\" - additionally starts the supervision tree of\n    the current project and its dependencies\n\nFinally, set `@recursive true` if you want the task to run\non each umbrella child in an umbrella project.","ref":"Mix.Task.html#module-module-attributes"},{"type":"behaviour","title":"Documentation - Mix.Task","doc":"Users can read the documentation for public Mix tasks by\nrunning `mix help my_task`. The documentation that will be\nshown is the `@moduledoc` of the task's module.","ref":"Mix.Task.html#module-documentation"},{"type":"function","title":"Mix.Task.alias?/1","doc":"Checks if the given `task` name is an alias.\n\nReturns false if the given name is not an alias or if it is not a task.\n\nFor more information about task aliasing, take a look at the\n[\"Aliases\"](https://hexdocs.pm/mix/Mix.html#module-aliases) section in the\ndocs for `Mix`.","ref":"Mix.Task.html#alias?/1"},{"type":"function","title":"Mix.Task.all_modules/0","doc":"Returns all loaded task modules.\n\nModules that are not yet loaded won't show up.\nCheck `load_all/0` if you want to preload all tasks.","ref":"Mix.Task.html#all_modules/0"},{"type":"function","title":"Mix.Task.clear/0","doc":"Clears all invoked tasks, allowing them to be reinvoked.\n\nThis operation is not recursive.","ref":"Mix.Task.html#clear/0"},{"type":"function","title":"Mix.Task.get/1","doc":"Receives a task name and returns the corresponding task module if one exists.\n\nReturns `nil` if the module cannot be found, if it is an alias, or if it is\nnot a valid `Mix.Task`.","ref":"Mix.Task.html#get/1"},{"type":"function","title":"Mix.Task.get!/1","doc":"Receives a task name and retrieves the corresponding task module.","ref":"Mix.Task.html#get!/1"},{"type":"function","title":"Exceptions - Mix.Task.get!/1","doc":"* `Mix.NoTaskError`      - raised if the task could not be found\n  * `Mix.InvalidTaskError` - raised if the task is not a valid `Mix.Task`","ref":"Mix.Task.html#get!/1-exceptions"},{"type":"function","title":"Mix.Task.load_all/0","doc":"Loads all tasks in all code paths.","ref":"Mix.Task.html#load_all/0"},{"type":"function","title":"Mix.Task.load_tasks/1","doc":"Loads all tasks in the given `paths`.","ref":"Mix.Task.html#load_tasks/1"},{"type":"function","title":"Mix.Task.moduledoc/1","doc":"Gets the moduledoc for the given task `module`.\n\nReturns the moduledoc or `nil`.","ref":"Mix.Task.html#moduledoc/1"},{"type":"function","title":"Mix.Task.preferred_cli_env/1","doc":"","ref":"Mix.Task.html#preferred_cli_env/1"},{"type":"function","title":"Mix.Task.recursing?/0","doc":"Indicates if the current task is recursing.\n\nThis returns true if a task is marked as recursive\nand it is being executed inside an umbrella project.","ref":"Mix.Task.html#recursing?/0"},{"type":"function","title":"Mix.Task.recursive/1","doc":"Checks if the task should be run recursively for all sub-apps in\numbrella projects.\n\nReturns `true` or `false`.","ref":"Mix.Task.html#recursive/1"},{"type":"function","title":"Mix.Task.reenable/1","doc":"Reenables a given task so it can be executed again down the stack.\n\nBoth alias and the regular stack are re-enabled when this function\nis called.\n\nIf an umbrella project reenables a task, it is re-enabled for all\nchild projects.","ref":"Mix.Task.html#reenable/1"},{"type":"function","title":"Mix.Task.requirements/1","doc":"Gets the list of requirements for the given task.\n\nReturns a list of strings, where the string is expected\nto be a task optionally followed by its arguments.","ref":"Mix.Task.html#requirements/1"},{"type":"function","title":"Mix.Task.rerun/2","doc":"Reruns `task` with the given arguments.\n\nThis function reruns the given task; to do that, it first re-enables the task\nand then runs it as normal.","ref":"Mix.Task.html#rerun/2"},{"type":"callback","title":"Mix.Task.run/1","doc":"A task needs to implement `run` which receives\na list of command line args.","ref":"Mix.Task.html#c:run/1"},{"type":"function","title":"Mix.Task.run/2","doc":"Conditionally runs the task (or alias) with the given `args`.\n\nIf there exists a task matching the given task name and it has not yet been\ninvoked, this will run the task with the given `args` and return the result.\n\nIf there is an [alias](Mix.html#module-aliases) defined\nfor the given task name, the alias will be invoked instead of the original\ntask.\n\nIf the task or alias has already been invoked, subsequent calls to `run/2`\nwill _abort_ without executing and return `:noop`.\n\nRemember: by default, tasks will only run _once_, even when called repeatedly!\nIf you need to run a task multiple times, you need to re-enable it via\n`reenable/1` or call it using `rerun/2`.\n\n`run/2` raises an exception if an alias or a task cannot be found or if the\ntask is invalid. See `get!/1` for more information.","ref":"Mix.Task.html#run/2"},{"type":"function","title":"Examples - Mix.Task.run/2","doc":"iex> Mix.Task.run(\"format\", [\"mix.exs\"])\n    :ok","ref":"Mix.Task.html#run/2-examples"},{"type":"function","title":"Mix.Task.run_in_apps/3","doc":"Runs recursive tasks in the specified list of children apps for umbrella projects.\n\nIf the task is not recursive (whose purpose is to be run in children\napplications), it runs at the project root level as usual. Calling\nthis function outside of an umbrella project root fails.","ref":"Mix.Task.html#run_in_apps/3"},{"type":"function","title":"Mix.Task.shortdoc/1","doc":"Gets the shortdoc for the given task `module`.\n\nReturns the shortdoc or `nil`.","ref":"Mix.Task.html#shortdoc/1"},{"type":"function","title":"Mix.Task.task?/1","doc":"Returns `true` if given module is a task.","ref":"Mix.Task.html#task?/1"},{"type":"function","title":"Mix.Task.task_name/1","doc":"Returns the task name for the given `module`.","ref":"Mix.Task.html#task_name/1"},{"type":"function","title":"Examples - Mix.Task.task_name/1","doc":"iex> Mix.Task.task_name(Mix.Tasks.Test)\n    \"test\"","ref":"Mix.Task.html#task_name/1-examples"},{"type":"type","title":"Mix.Task.task_module/0","doc":"","ref":"Mix.Task.html#t:task_module/0"},{"type":"type","title":"Mix.Task.task_name/0","doc":"","ref":"Mix.Task.html#t:task_name/0"},{"type":"behaviour","title":"Mix.Task.Compiler","doc":"This module defines the behaviour for a Mix task that does compilation.\n\nA Mix compiler task can be defined by simply using `Mix.Task.Compiler`\nin a module whose name starts with `Mix.Tasks.Compile.` and defining\nthe [`run/1`](`c:run/1`) function:\n\n    defmodule Mix.Tasks.Compile.MyLanguage do\n      use Mix.Task.Compiler\n\n      def run(_args) do\n        :ok\n      end\n    end\n\nThe [`run/1`](`c:run/1`) function returns an atom indicating the status of the\ncompilation, and optionally can also return a list of \"diagnostics\"\nsuch as warnings or compilation errors. Doing this enables code\neditors to display issues inline without having to analyze the\ncommand-line output.\n\nIf the compiler uses manifest files to track stale sources, it should\ndefine `manifests/0`, and if it writes any output to disk it should\nalso define `clean/0`.\n\nA compiler supports the same attributes for configuration and\ndocumentation as a regular Mix task. See `Mix.Task` for more information.","ref":"Mix.Task.Compiler.html"},{"type":"function","title":"Mix.Task.Compiler.after_compiler/2","doc":"Adds a callback that runs after a given compiler.\n\nThe callback is invoked after the compiler runs and\nit receives a tuple with current status and the list\nof diagnostic. It must return the updated status and\ndiagnostics.\n\nIf the given compiler does not run (for instance,\nbecause an earlier compiler in the stack has aborted),\nthe callback will not be executed.","ref":"Mix.Task.Compiler.html#after_compiler/2"},{"type":"callback","title":"Mix.Task.Compiler.clean/0","doc":"Removes build artifacts and manifests.","ref":"Mix.Task.Compiler.html#c:clean/0"},{"type":"callback","title":"Mix.Task.Compiler.manifests/0","doc":"Lists manifest files for the compiler.","ref":"Mix.Task.Compiler.html#c:manifests/0"},{"type":"callback","title":"Mix.Task.Compiler.run/1","doc":"Receives command-line arguments and performs compilation. If it\nproduces errors, warnings, or any other diagnostic information,\nit should return a tuple with the status and a list of diagnostics.","ref":"Mix.Task.Compiler.html#c:run/1"},{"type":"type","title":"Mix.Task.Compiler.status/0","doc":"","ref":"Mix.Task.Compiler.html#t:status/0"},{"type":"module","title":"Mix.Task.Compiler.Diagnostic","doc":"Diagnostic information such as a warning or compilation error.","ref":"Mix.Task.Compiler.Diagnostic.html"},{"type":"type","title":"Mix.Task.Compiler.Diagnostic.position/0","doc":"Where in a file the diagnostic applies. Can be either a line number,\na `{line, column}` tuple, a range specified as `{start_line, start_col,\nend_line, end_col}`. `0` line represents unknown.\n\nLine numbers are one-based, and column numbers in a range are zero-based and refer\nto the cursor position at the start of the character at that index. For example,\nto indicate that a diagnostic applies to the first `n` characters of the\nfirst line, the range would be `{1, 0, 1, n}`.","ref":"Mix.Task.Compiler.Diagnostic.html#t:position/0"},{"type":"type","title":"Mix.Task.Compiler.Diagnostic.severity/0","doc":"Severity of a diagnostic:\n\n  * `:error` - An issue that caused compilation to fail\n\n  * `:warning` - An issue that did not cause failure but suggests the\n    programmer may have made a mistake\n\n  * `:hint` - A suggestion for style or good practices that is not as\n    severe as a warning\n\n  * `:information` - Any other information relevant to compilation that\n    does not fit into the above categories","ref":"Mix.Task.Compiler.Diagnostic.html#t:severity/0"},{"type":"type","title":"Mix.Task.Compiler.Diagnostic.t/0","doc":"","ref":"Mix.Task.Compiler.Diagnostic.html#t:t/0"},{"type":"task","title":"mix app.config","doc":"Loads and configures all registered apps.\n\nThis is done by loading `config/runtime.exs` if one exists.\nThe application will be compiled if it was not compiled before.","ref":"Mix.Tasks.App.Config.html"},{"type":"task","title":"Command line options - mix app.config","doc":"* `--force` - forces compilation regardless of compilation times\n  * `--preload-modules` - preloads all modules defined in applications\n  * `--no-archives-check` - does not check archives\n  * `--no-app-loading` - does not load .app resource file after compilation\n  * `--no-compile` - does not compile even if files require compilation\n  * `--no-deps-check` - does not check dependencies\n  * `--no-elixir-version-check` - does not check Elixir version\n  * `--no-validate-compile-env` - does not validate the application compile environment","ref":"Mix.Tasks.App.Config.html#module-command-line-options"},{"type":"task","title":"mix app.start","doc":"Starts all registered apps.\n\nFirst, this task guarantees that all dependencies are in place\nand that the current project has been compiled. Then, the current\napplication is started as a temporary application, unless\n`:start_permanent` is set to `true` in your project configuration\nor the `--permanent` option is given. Setting it to permanent\nguarantees the node will shut down if the application terminates\n(typically because its root supervisor has terminated).","ref":"Mix.Tasks.App.Start.html"},{"type":"task","title":"Configuration - mix app.start","doc":"* `:start_permanent` - the application and all of its children\n    applications are started in permanent mode. Defaults to `false`.\n\n  * `:start_concurrently` - applications are started concurrently\n    whenever possible. This option only has an effect on Erlang/OTP 26+.\n    Defaults to `false`.","ref":"Mix.Tasks.App.Start.html#module-configuration"},{"type":"task","title":"Command line options - mix app.start","doc":"* `--force` - forces compilation regardless of compilation times\n  * `--temporary` - starts the application as temporary\n  * `--permanent` - starts the application as permanent\n  * `--preload-modules` - preloads all modules defined in applications\n  * `--no-archives-check` - does not check archives\n  * `--no-compile` - does not compile even if files require compilation\n  * `--no-deps-check` - does not check dependencies\n  * `--no-elixir-version-check` - does not check Elixir version\n  * `--no-start` - does not actually start applications, only compiles and loads code","ref":"Mix.Tasks.App.Start.html#module-command-line-options"},{"type":"task","title":"mix app.tree","doc":"Prints the application tree.\n\n    $ mix app.tree --exclude logger --exclude elixir\n\nIf no application is given, it uses the current application defined\nin the `mix.exs` file.","ref":"Mix.Tasks.App.Tree.html"},{"type":"task","title":"Command line options - mix app.tree","doc":"* `--exclude` - exclude applications which you do not want to see printed.\n    `kernel`, `stdlib` and `compiler` are always excluded from the tree.\n\n  * `--format` - Can be set to one of either:\n\n    * `pretty` - uses Unicode code points for formatting the tree.\n      This is the default except on Windows.\n\n    * `plain` - does not use Unicode code points for formatting the tree.\n      This is the default on Windows.\n\n    * `dot` - produces a DOT graph description of the application tree\n      in `app_tree.dot` in the current directory.\n      Warning: this will overwrite any previously generated file.","ref":"Mix.Tasks.App.Tree.html#module-command-line-options"},{"type":"task","title":"mix archive","doc":"Lists all installed archives.\n\nArchives are typically installed at `~/.mix/archives`\nalthough the installation path can be customized by\nsetting the `MIX_ARCHIVES` environment variable.\n\nSince archives are specific to Elixir versions, it is\nexpected from build tools to swap the `MIX_ARCHIVES`\nvariable to different locations based on a particular\nElixir installation.","ref":"Mix.Tasks.Archive.html"},{"type":"task","title":"mix archive.build","doc":"Builds an archive according to the specification of the\n[Erlang archive format](`:code`).\n\nArchives are meant to contain small projects, usually installed\nlocally. Archives may be installed into a Mix environment by\nrunning `mix archive.install`. Once installed, the archive is\navailable to all Mix projects. For this reason, the functionality\nbehind archives is limited. For instance, archives do not include\ndependencies, as those would conflict with any dependency in a\nMix project after the archive is installed. In general, we recommend\nthe usage of archives to be limited for extensions of Mix, such\nas custom SCMs, package managers, and the like. For general scripts to be\ndistributed to developers, please see `mix escript.build`.\n\nThe archive will be created in the current directory (which is\nexpected to be the project root), unless an argument `-o` is\nprovided with the file name.\n\nBy default, this command archives the current project but the `-i`\noption can be used to archive any directory. For example,\n`mix archive.build` with no options translates to:\n\n    $ mix archive.build -i _build/ENV/lib/APP -o APP-VERSION.ez","ref":"Mix.Tasks.Archive.Build.html"},{"type":"task","title":"Command line options - mix archive.build","doc":"* `-o` - specifies output file name.\n    If there is a `mix.exs`, defaults to \"APP-VERSION.ez\".\n\n  * `-i` - specifies the input directory to archive.\n    If there is a `mix.exs`, defaults to the current application build.\n\n  * `--no-compile` - skips compilation.\n    Only applies when `mix.exs` is available.\n\n  * `--include-dot-files` - adds dot files from priv directory to the archive.","ref":"Mix.Tasks.Archive.Build.html#module-command-line-options"},{"type":"task","title":"mix archive.check","doc":"Checks all archives are available.\n\nMix projects can specify required archives using\nthe `:archives` option:\n\n    archives: [{:foo, \"~> 1.0.0\"}]\n\nThis task guarantees this option is respected.","ref":"Mix.Tasks.Archive.Check.html"},{"type":"task","title":"mix archive.install","doc":"Installs an archive locally.\n\nIf no argument is supplied but there is an archive in the project's\nroot directory (created with `mix archive.build`), then the archive\nwill be installed locally. For example:\n\n    $ mix do archive.build + archive.install\n\nIf an argument is provided, it should be a local path to a\nprebuilt archive, a Git repository, a GitHub repository, or a Hex\npackage.\n\n    $ mix archive.install archive.ez\n    $ mix archive.install path/to/archive.ez\n    $ mix archive.install git https://path/to/git/repo\n    $ mix archive.install git https://path/to/git/repo branch git_branch\n    $ mix archive.install git https://path/to/git/repo tag git_tag\n    $ mix archive.install git https://path/to/git/repo ref git_ref\n    $ mix archive.install github user/project\n    $ mix archive.install github user/project branch git_branch\n    $ mix archive.install github user/project tag git_tag\n    $ mix archive.install github user/project ref git_ref\n    $ mix archive.install hex hex_package\n    $ mix archive.install hex hex_package 1.2.3\n\nAfter installation, the tasks in the archive are available locally:\n\n    $ mix some_task\n\nNote that installing via Git, GitHub, or Hex fetches the source\nof the archive and builds it, while using local path uses a pre-built archive.","ref":"Mix.Tasks.Archive.Install.html"},{"type":"task","title":"Command line options - mix archive.install","doc":"* `--sha512` - checks the archive matches the given SHA-512 checksum. Only\n    applies to installations via a local path\n\n  * `--force` - forces installation without a shell prompt; primarily\n    intended for automation in build systems like Make\n\n  * `--submodules` - fetches repository submodules before building archive from\n    Git or GitHub\n\n  * `--app` - specifies a custom app name to be used for building the archive\n    from Git, GitHub, or Hex\n\n  * `--organization` - set this for Hex private packages belonging to an\n    organization\n\n  * `--repo` - set this for self-hosted Hex instances, defaults to `hexpm`","ref":"Mix.Tasks.Archive.Install.html#module-command-line-options"},{"type":"task","title":"mix archive.uninstall","doc":"Uninstalls local archives.\n\n    $ mix archive.uninstall archive.ez","ref":"Mix.Tasks.Archive.Uninstall.html"},{"type":"task","title":"Command line options - mix archive.uninstall","doc":"* `--force` - forces uninstallation without a shell prompt; primarily\n    intended for automation","ref":"Mix.Tasks.Archive.Uninstall.html#module-command-line-options"},{"type":"task","title":"mix clean","doc":"Deletes generated application files.\n\nThis command deletes all build artifacts for the current project.\nDependencies' sources and build files are cleaned only if the\n`--deps` option is given.\n\nBy default this task works across all environments, unless `--only`\nis given.","ref":"Mix.Tasks.Clean.html"},{"type":"task","title":"mix cmd","doc":"Executes the given command.\n\nFor example, you can invoke an external command such as make:\n\n    $ mix cmd make\n\nBy passing the `--cd` flag before the command, you can also force\nthe command to run in a specific directory:\n\n    $ mix cmd --cd \"third-party\" make\n\nThis task is also useful in umbrella applications to execute a command\non each child app:\n\n    $ mix cmd pwd\n\nYou can limit which apps the cmd runs in by using `mix do` with the `--app`\noption:\n\n    $ mix do --app app1 --app app2 cmd pwd\n\nThe tasks aborts whenenever a command exits with a non-zero status.\n\nThis task is automatically re-enabled, so it can be called multiple times\nwith different arguments.","ref":"Mix.Tasks.Cmd.html"},{"type":"task","title":"Command line options - mix cmd","doc":"* `--app` - limit running the command to the given app.\n    This option is currently deprecated in favor of `mix do --app`\n\n  * `--cd` - (since v1.10.4) the directory to run the command in","ref":"Mix.Tasks.Cmd.html#module-command-line-options"},{"type":"task","title":"Zombie operating system processes - mix cmd","doc":"Beware that the Erlang VM does not terminate child processes\nwhen it shuts down. Therefore, if you use `mix cmd` to start\nlong running processes and then shut down the VM, it is likely\nthat those child processes won't be terminated with the VM.\n\nA solution is to make sure the child processes listen to the\nstandard input and terminate when standard input is closed.\nWe discuss this topic at length in the \"Zombie operating system processes\"\nof the `Port` module documentation.","ref":"Mix.Tasks.Cmd.html#module-zombie-operating-system-processes"},{"type":"task","title":"mix compile","doc":"The main entry point to compile source files.\n\nIt simply runs the compilers registered in your project and returns\na tuple with the compilation status and a list of diagnostics.\n\nBefore compiling code, it performs a series of checks to ensure all\ndependencies are compiled and the project is up to date. Then the\ncode path of your Elixir system is pruned to only contain the dependencies\nand applications that you have explicitly listed in your `mix.exs`.","ref":"Mix.Tasks.Compile.html"},{"type":"task","title":"Configuration - mix compile","doc":"* `:build_embedded` - this option was used to copy all code and\n    priv content to the `_build` directory. However, this option no\n    longer has an effect as Elixir will now copy those at release time\n\n  * `:compilers` - compilers to run, defaults to `Mix.compilers/0`,\n    which are `[:yecc, :leex, :erlang, :elixir, :app]`.\n\n  * `:consolidate_protocols` - when `true`, runs protocol\n    consolidation via the `mix compile.protocols` task. The default\n    value is `true`.\n\n  * `:build_path` - the directory where build artifacts\n    should be written to. This option is intended only for\n    child apps within a larger umbrella application so that\n    each child app can use the common `_build` directory of\n    the parent umbrella. In a non-umbrella context, configuring\n    this has undesirable side-effects (such as skipping some\n    compiler checks) and should be avoided.","ref":"Mix.Tasks.Compile.html#module-configuration"},{"type":"task","title":"Compilers - mix compile","doc":"To see documentation for each specific compiler, you must\ninvoke `help` directly for the compiler command:\n\n    $ mix help compile.elixir\n    $ mix help compile.erlang\n\nYou can get a list of all compilers by running:\n\n    $ mix compile --list","ref":"Mix.Tasks.Compile.html#module-compilers"},{"type":"task","title":"Command line options - mix compile","doc":"* `--erl-config` - path to an Erlang term file that will be loaded as Mix config\n  * `--force` - forces compilation\n  * `--list` - lists all enabled compilers\n  * `--no-all-warnings` - prints only warnings from files currently compiled (instead of all)\n  * `--no-app-loading` - does not load .app resource file after compilation\n  * `--no-archives-check` - skips checking of archives\n  * `--no-compile` - does not actually compile, only loads code and perform checks\n  * `--no-deps-check` - skips checking of dependencies\n  * `--no-elixir-version-check` - does not check Elixir version\n  * `--no-optional-deps` - does not compile or load optional deps. Useful for testing\n    if a library still successfully compiles without optional dependencies (which is the\n    default case with dependencies)\n  * `--no-prune-code-paths` - do not prune code paths before compilation, this keeps\n    the entirety of Erlang/OTP available on the project starts\n  * `--no-protocol-consolidation` - skips protocol consolidation\n  * `--no-validate-compile-env` - does not validate the application compile environment\n  * `--return-errors` - returns error status and diagnostics instead of exiting on error\n  * `--warnings-as-errors` - exit with non-zero status if compilation has one or more warnings","ref":"Mix.Tasks.Compile.html#module-command-line-options"},{"type":"function","title":"Mix.Tasks.Compile.compilers/1","doc":"Returns all compilers.","ref":"Mix.Tasks.Compile.html#compilers/1"},{"type":"task","title":"mix compile.app","doc":"Writes an .app file.\n\nAn `.app` file is a file containing Erlang terms that defines\nyour application. Mix automatically generates this file based on\nyour `mix.exs` configuration.\n\nIn order to generate the `.app` file, Mix expects your project\nto have both `:app` and `:version` keys. Furthermore, you can\nconfigure the generated application by defining an `application/0`\nfunction in your `mix.exs` that returns a keyword list.\n\nThe most commonly used keys are:\n\n  * `:extra_applications` - a list of OTP applications\n    your application depends on which are not included in `:deps`\n    (usually defined in `deps/0` in your `mix.exs`). For example,\n    here you can declare a dependency on applications that ship\n    with Erlang/OTP or Elixir, like `:crypto` or `:logger`.\n    Optional extra applications can be declared as a tuple, such\n    as `{:ex_unit, :optional}`. Mix guarantees all non-optional\n    applications are started before your application starts.\n\n  * `:registered` - the name of all registered processes in the\n    application. If your application defines a local GenServer\n    with name `MyServer`, it is recommended to add `MyServer`\n    to this list. It is most useful in detecting conflicts\n    between applications that register the same names.\n\n  * `:env` - the default values for the application environment.\n    The application environment is one of the most common ways\n    to configure applications. See the `Application` module for\n    mechanisms to read and write to the application environment.\n\nFor example:\n\n    def application do\n      [\n        extra_applications: [:logger, :crypto, ex_unit: :optional],\n        env: [key: :value],\n        registered: [MyServer]\n      ]\n    end\n\nOther options include:\n\n  * `:applications` - all applications your application depends\n    on at runtime. By default, this list is automatically inferred\n    from your dependencies. Mix and other tools use the application\n    list in order to start your dependencies before starting the\n    application itself.\n\n  * `:mod` - specifies a module to invoke when the application\n    is started. It must be in the format `{Mod, args}` where\n    args is often an empty list. The module specified must\n    implement the callbacks defined by the `Application`\n    module.\n\n  * `:start_phases` - specifies a list of phases and their arguments\n    to be called after the application is started. See the \"Phases\"\n    section below.\n\n  * `:included_applications` - specifies a list of applications\n    that will be included in the application. It is the responsibility of\n    the primary application to start the supervision tree of all included\n    applications, as only the primary application will be started. A process\n    in an included application considers itself belonging to the\n    primary application.\n\n  * `:maxT` - specifies the maximum time the application is allowed to run, in\n    milliseconds. Applications are stopped if `:maxT` is reached, and their\n    top-level supervisor terminated with reason `:normal`. This threshold is\n    technically valid in any resource file, but it is only effective for\n    applications with a callback module. Defaults to `:infinity`.\n\nBesides the options above, `.app` files also expect other options like\n`:modules` and `:vsn`, but these are automatically added by Mix.","ref":"Mix.Tasks.Compile.App.html"},{"type":"task","title":"Command line options - mix compile.app","doc":"* `--force` - forces compilation regardless of modification times\n  * `--compile-path` - where to find `.beam` files and write the\n    resulting `.app` file, defaults to `Mix.Project.compile_path/0`","ref":"Mix.Tasks.Compile.App.html#module-command-line-options"},{"type":"task","title":"Phases - mix compile.app","doc":"Applications provide a start phases mechanism which will be called,\nin order, for the application and all included applications. If a phase\nis not defined for an included application, that application is skipped.\n\nLet's see an example `MyApp.application/0` function:\n\n    def application do\n      [\n        start_phases: [init: [], go: [], finish: []],\n        included_applications: [:my_included_app]\n      ]\n    end\n\nAnd an example `:my_included_app` defines on its `mix.exs` the function:\n\n    def application do\n      [\n        mod: {MyIncludedApp, []},\n        start_phases: [go: []]\n      ]\n    end\n\nIn this example, the order that the application callbacks are called in is:\n\n    Application.start(MyApp)\n    MyApp.start(:normal, [])\n    MyApp.start_phase(:init, :normal, [])\n    MyApp.start_phase(:go, :normal, [])\n    MyIncludedApp.start_phase(:go, :normal, [])\n    MyApp.start_phase(:finish, :normal, [])","ref":"Mix.Tasks.Compile.App.html#module-phases"},{"type":"task","title":"mix compile.elixir","doc":"Compiles Elixir source files.\n\nElixir is smart enough to recompile only files that have changed\nand their dependencies. This means if `lib/a.ex` is invoking\na function defined over `lib/b.ex` at compile time, whenever\n`lib/b.ex` changes, `lib/a.ex` is also recompiled.\n\nNote Elixir considers a file as changed if its source file has\nchanged on disk since the last compilation AND its contents are\nno longer the same.\n\n## `@external_resource`\n\nIf a module depends on external files, those can be annotated\nwith the `@external_resource` module attribute. If these files\nchange, the Elixir module is automatically recompiled.\n\n## `__mix_recompile__?/0`\n\nA module may export a `__mix_recompile__?/0` function that can\ncause the module to be recompiled using custom rules. For example,\nto recompile whenever a file is changed in a given directory, you\ncan use a combination of `@external_resource` for existing files\nand a `__mix_recompile__?/0` check to verify when new entries are\nadded to the directory itself:\n\n    defmodule MyModule do\n      paths = Path.wildcard(\"*.txt\")\n      @paths_hash :erlang.md5(paths)\n\n      for path   :erlang.md5() != @paths_hash\n      end\n    end\n\nCompiler calls `__mix_recompile__?/0` for every module being\ncompiled (or previously compiled) and thus it is very important\nto do there as little work as possible to not slow down the\ncompilation.\n\nIf module has `@compile {:autoload, false}`, `__mix_recompile__?/0` will\nnot be used.","ref":"Mix.Tasks.Compile.Elixir.html"},{"type":"task","title":"Command line options - mix compile.elixir","doc":"* `--docs` (`--no-docs`) - attaches (or not) documentation to compiled modules\n  * `--debug-info` (`--no-debug-info`) - attaches (or not) debug info to compiled modules\n  * `--force` - forces compilation regardless of modification times\n  * `--ignore-module-conflict` - does not emit warnings if a module was previously defined\n  * `--long-compilation-threshold N` - sets the \"long compilation\" threshold\n    (in seconds) to `N` (see the docs for `Kernel.ParallelCompiler.compile/2`)\n  * `--purge-consolidation-path-if-stale PATH` - deletes and purges modules in the\n    given protocol consolidation path if compilation is required\n  * `--profile` - if set to `time`, outputs timing information of compilation steps\n  * `--tracer` - adds a compiler tracer in addition to any specified in the `mix.exs` file\n  * `--verbose` - prints each file being compiled\n  * `--warnings-as-errors` - treats warnings in the current project as errors and\n    return a non-zero exit status","ref":"Mix.Tasks.Compile.Elixir.html#module-command-line-options"},{"type":"task","title":"Configuration - mix compile.elixir","doc":"* `:elixirc_paths` - directories to find source files.\n    Defaults to `[\"lib\"]`.\n\n  * `:elixirc_options` - compilation options that apply to Elixir's compiler.\n    See `Code.put_compiler_option/2` for a complete list of options. These\n    options are often overridable from the command line using the switches\n    above.\n\n  * `[xref: [exclude: ...]]` - a list of `module` or `{module, function, arity}`\n    that should not be warned on in case on undefined modules or undefined\n    application warnings.","ref":"Mix.Tasks.Compile.Elixir.html#module-configuration"},{"type":"task","title":"mix compile.erlang","doc":"Compiles Erlang source files.\n\nWhen this task runs, it will first check the modification times of\nall files to be compiled and if they haven't been changed since the\nlast compilation, it will not compile them. If any of them have changed,\nit compiles everything.","ref":"Mix.Tasks.Compile.Erlang.html"},{"type":"task","title":"Command line options - mix compile.erlang","doc":"* `--force` - forces compilation regardless of modification times\n\n  * `--no-all-warnings` - prints only warnings from files currently compiled (instead of all)","ref":"Mix.Tasks.Compile.Erlang.html#module-command-line-options"},{"type":"task","title":"Configuration - mix compile.erlang","doc":"* `ERL_COMPILER_OPTIONS` - can be used to give default compile options.\n    The value must be a valid Erlang term. If the value is a list, it will\n    be used as is. If it is not a list, it will be put into a list.\n\n  * `:erlc_paths` - directories to find source files.\n    Defaults to `[\"src\"]`.\n\n  * `:erlc_include_path` - directory for adding include files.\n    Defaults to `\"include\"`.\n\n  * `:erlc_options` - compilation options that apply to Erlang's\n    compiler. Defaults to `[]`.\n\n    For a complete list of options, see `:compile.file/2`.\n    The option `:debug_info` is always added to the end of it.\n    You can disable that using:\n\n        erlc_options: [debug_info: false]","ref":"Mix.Tasks.Compile.Erlang.html#module-configuration"},{"type":"task","title":"mix compile.leex","doc":"Compiles Leex source files.\n\nWhen this task runs, it will check the modification time of every file, and\nif it has changed, the file will be compiled. Files will be\ncompiled in the same source directory with a .erl extension.\nYou can force compilation regardless of modification times by passing\nthe `--force` option.","ref":"Mix.Tasks.Compile.Leex.html"},{"type":"task","title":"Command line options - mix compile.leex","doc":"* `--force` - forces compilation regardless of modification times\n\n  * `--no-all-warnings` - prints only warnings from files currently compiled (instead of all)","ref":"Mix.Tasks.Compile.Leex.html#module-command-line-options"},{"type":"task","title":"Configuration - mix compile.leex","doc":"* `:erlc_paths` - directories to find source files. Defaults to `[\"src\"]`.\n\n  * `:leex_options` - compilation options that apply\n    to Leex's compiler.\n\n    For a complete list of options, see `:leex.file/2`.\n    Note that the `:report`, `:return_errors`, and `:return_warnings` options\n    are overridden by this compiler, thus setting them has no effect.","ref":"Mix.Tasks.Compile.Leex.html#module-configuration"},{"type":"task","title":"mix compile.protocols","doc":"Consolidates all protocols in all paths.\n\nThis task is automatically invoked unless the project\ndisables the `:consolidate_protocols` option in their\nconfiguration.","ref":"Mix.Tasks.Compile.Protocols.html"},{"type":"task","title":"Consolidation - mix compile.protocols","doc":"Protocol consolidation is useful in production when no\ndynamic code loading will happen, effectively optimizing\nprotocol dispatches by not accounting for code loading.\n\nThis task consolidates all protocols in the code path\nand outputs the new binary files to the given directory.\nDefaults to \"_build/MIX_ENV/lib/YOUR_APP/consolidated\"\nfor regular apps and \"_build/MIX_ENV/consolidated\" in\numbrella projects.\n\nIn case you are manually compiling protocols or building\nreleases, you need to take the generated protocols into\naccount. This can be done with:\n\n    $ elixir -pa _build/MIX_ENV/lib/YOUR_APP/consolidated -S mix run\n\nOr in umbrellas:\n\n    $ elixir -pa _build/MIX_ENV/consolidated -S mix run\n\nYou can verify a protocol is consolidated by checking\nits attributes:\n\n    iex> Protocol.consolidated?(Enumerable)\n    true","ref":"Mix.Tasks.Compile.Protocols.html#module-consolidation"},{"type":"function","title":"Mix.Tasks.Compile.Protocols.consolidated?/0","doc":"Returns if protocols have been consolidated at least once.","ref":"Mix.Tasks.Compile.Protocols.html#consolidated?/0"},{"type":"task","title":"mix compile.yecc","doc":"Compiles Yecc source files.\n\nWhen this task runs, it will check the modification time of every file, and\nif it has changed, the file will be compiled. Files will be\ncompiled in the same source directory with a .erl extension.\nYou can force compilation regardless of modification times by passing\nthe `--force` option.","ref":"Mix.Tasks.Compile.Yecc.html"},{"type":"task","title":"Command line options - mix compile.yecc","doc":"* `--force` - forces compilation regardless of modification times\n\n  * `--no-all-warnings` - prints only warnings from files currently compiled (instead of all)","ref":"Mix.Tasks.Compile.Yecc.html#module-command-line-options"},{"type":"task","title":"Configuration - mix compile.yecc","doc":"* `:erlc_paths` - directories to find source files. Defaults to `[\"src\"]`.\n\n  * `:yecc_options` - compilation options that apply\n    to Yecc's compiler.\n\n    For a complete list of options, see `:yecc.file/1`.\n    Note that the `:report`, `:return_errors`, and `:return_warnings` options\n    are overridden by this compiler, thus setting them has no effect.","ref":"Mix.Tasks.Compile.Yecc.html#module-configuration"},{"type":"task","title":"mix deps","doc":"Lists all dependencies and their status.\n\nDependencies must be specified in the `mix.exs` file in one of\nthe following formats:\n\n    {app, requirement}\n    {app, opts}\n    {app, requirement, opts}\n\nWhere:\n\n  * app is an atom\n  * requirement is a `Version` requirement or a regular expression\n  * opts is a keyword list of options\n\nFor example:\n\n    {:plug, \">= 0.4.0\"}\n    {:gettext, git: \"https://github.com/elixir-lang/gettext.git\", tag: \"0.1\"}\n    {:local_dependency, path: \"path/to/local_dependency\"}\n\nBy default, dependencies are fetched using the [Hex package manager](https://hex.pm/):\n\n    {:plug, \">= 0.4.0\"}\n\nBy specifying such dependencies, Mix will automatically install\nHex (if it wasn't previously installed) and download a package\nsuitable to your project. Note Hex expects the dependency\nrequirement to always be given and it will warn otherwise.\n\nMix also supports Git and path dependencies:\n\n    {:foobar, git: \"https://github.com/elixir-lang/foobar.git\", tag: \"0.1\"}\n    {:foobar, path: \"path/to/foobar\"}\n\nAnd also in umbrella dependencies:\n\n    {:my_app, in_umbrella: true}\n\nPath and in umbrella dependencies are automatically recompiled by\nthe parent project whenever they change. While fetchable dependencies,\nlike the ones using `:git`, are recompiled only when fetched/updated.\n\nThe dependencies' versions are expected to be formatted according to\nSemantic Versioning and the requirements must be specified as defined\nin the `Version` module.","ref":"Mix.Tasks.Deps.html"},{"type":"task","title":"Options - mix deps","doc":"Below we provide a more detailed look into the available options.\n\n#","ref":"Mix.Tasks.Deps.html#module-options"},{"type":"task","title":"Dependency definition options - mix deps","doc":"* `:app` - when set to `false`, does not read the app file for this\n    dependency. By default, the app file is read\n\n  * `:env` - the environment (as an atom) to run the dependency on; defaults to `:prod`\n\n  * `:compile` - a command (string) to compile the dependency; defaults to a `mix`,\n    `rebar` or `make` command\n\n  * `:optional` - marks the dependency as optional. In such cases, the\n    current project will always include the optional dependency but any\n    other project that depends on the current project won't be forced to\n    use the optional dependency. However, if the other project includes\n    the optional dependency on its own, the requirements and options\n    specified here will also be applied. Optional dependencies will _not_\n    be started by the application.\n\n  * `:only` - the dependency is made available only in the given environments,\n    useful when declaring dev- or test-only dependencies; by default the\n    dependency will be available in all environments. The value of this option\n    can either be a single environment (like `:dev`) or a list of environments\n    (like `[:dev, :test]`)\n\n  * `:targets` - the dependency is made available only for the given targets.\n    By default the dependency will be available in all environments. The value\n    of this option can either be a single target (like `:host`) or a list of\n    environments (like `[:host, :rpi3]`)\n\n  * `:override` - if set to `true` the dependency will override any other\n    definitions of itself by other dependencies\n\n  * `:manager` - Mix can also compile Rebar3 and makefile projects\n    and can fetch sub dependencies of Rebar3 projects. Mix will\n    try to infer the type of project but it can be overridden with this\n    option by setting it to `:mix`, `:rebar3`, or `:make`. In case\n    there are conflicting definitions, the first manager in the list above\n    will be picked up. For example, if a dependency is found with `:rebar3`\n    as a manager in different part of the trees, `:rebar3` will be automatically\n    picked. You can find the manager by running `mix deps` and override it by\n    setting the `:override` option in a top-level project.\n\n  * `:runtime` - whether the dependency is part of runtime applications.\n    If the `:applications` key is not provided in `def application` in your\n    `mix.exs` file, Mix will automatically include all dependencies as a runtime\n    application, except if `runtime: false` is given. Defaults to true.\n\n  * `:system_env` - an enumerable of key-value tuples of binaries to be set\n    as environment variables when loading or compiling the dependency\n\n#","ref":"Mix.Tasks.Deps.html#module-dependency-definition-options"},{"type":"task","title":"Git options (`:git`) - mix deps","doc":"* `:git` - the Git repository URI\n  * `:github` - a shortcut for specifying Git repos from GitHub, uses `:git`\n  * `:ref` - the reference to checkout (may be a branch, a commit SHA or a tag)\n  * `:branch` - the Git branch to checkout\n  * `:tag` - the Git tag to checkout\n  * `:submodules` - when `true`, initialize submodules for the repo\n  * `:sparse` - checkout a single directory inside the Git repository and use it\n    as your Mix dependency. Search \"sparse Git checkouts\" for more information.\n  * `:subdir` - (since v1.13.0) search for the project in the given directory\n    relative to the git checkout. This is similar to `:sparse` option but instead\n    of a doing a sparse checkout it does a full checkout.\n\nIf your Git repository requires authentication, such as basic username:password\nHTTP authentication via URLs, it can be achieved via Git configuration, keeping\nthe access rules outside of source control.\n\n    $ git config --global url.\"https://YOUR_USER:YOUR_PASS@example.com/\".insteadOf \"https://example.com/\"\n\nFor more information, see the `git config` documentation:\nhttps://git-scm.com/docs/git-config#git-config-urlltbasegtinsteadOf\n\n#","ref":"Mix.Tasks.Deps.html#module-git-options-git"},{"type":"task","title":"Path options (`:path`) - mix deps","doc":"* `:path`        - the path for the dependency\n  * `:in_umbrella` - when `true`, sets a path dependency pointing to\n    \"../#{app}\", sharing the same environment as the current application\n\n#","ref":"Mix.Tasks.Deps.html#module-path-options-path"},{"type":"task","title":"Hex options (`:hex`) - mix deps","doc":"See the [Hex usage documentation](https://hex.pm/docs/usage) for Hex options.","ref":"Mix.Tasks.Deps.html#module-hex-options-hex"},{"type":"task","title":"Deps task - mix deps","doc":"`mix deps` task lists all dependencies in the following format:\n\n    APP VERSION (SCM) (MANAGER)\n    [locked at REF]\n    STATUS\n\nFor dependencies satisfied by Hex, `REF` is the package checksum.\n\nFor dependencies satisfied by git, `REF` is the commit object name,\nand may include branch or tag information.\n\nIt supports the following options:\n\n  * `--all` - lists all dependencies, regardless of specified environment","ref":"Mix.Tasks.Deps.html#module-deps-task"},{"type":"task","title":"mix deps.clean","doc":"Deletes the given dependencies' files, including build artifacts and fetched\nsources.\n\nSince this is a destructive action, cleaning of dependencies\nonly occurs when passing arguments/options:\n\n  * `dep1 dep2` - the names of dependencies to be deleted separated by a space\n  * `--unlock` - also unlocks the deleted dependencies\n  * `--build` - deletes only compiled files (keeps source files)\n  * `--all` - deletes all dependencies\n  * `--unused` - deletes only unused dependencies\n    (i.e. dependencies no longer mentioned in `mix.exs`)\n\nBy default this task works across all environments,\nunless `--only` is given which will clean all dependencies\nfor the chosen environment.","ref":"Mix.Tasks.Deps.Clean.html"},{"type":"task","title":"mix deps.compile","doc":"Compiles dependencies.\n\nBy default, this task attempts to compile all dependencies.\nA list of dependencies can be given to compile multiple\ndependencies in order.\n\nThis task attempts to detect if the project contains one of\nthe following files and act accordingly:\n\n  * `mix.exs` - invokes `mix compile`\n  * `rebar.config` - invokes `rebar compile`\n  * `Makefile.win`- invokes `nmake /F Makefile.win` (only on Windows)\n  * `Makefile` - invokes `gmake` on DragonFlyBSD, FreeBSD, NetBSD, and OpenBSD,\n    invokes `make` on any other operating system (except on Windows)\n\nThe compilation can be customized by passing a `compile` option\nin the dependency:\n\n    {:some_dependency, \"0.1.0\", compile: \"command to compile\"}\n\nIf a list of dependencies is given, Mix will attempt to compile\nthem as is. For example, if project `a` depends on `b`, calling\n`mix deps.compile a` will compile `a` even if `b` is out of\ndate. This is to allow parts of the dependency tree to be\nrecompiled without propagating those changes upstream. To ensure\n`b` is included in the compilation step, pass `--include-children`.","ref":"Mix.Tasks.Deps.Compile.html"},{"type":"task","title":"Command line options - mix deps.compile","doc":"* `--force` - force compilation of deps\n  * `--skip-umbrella-children` - skips umbrella applications from compiling\n  * `--skip-local-deps` - skips non-remote dependencies, such as path deps, from compiling","ref":"Mix.Tasks.Deps.Compile.html#module-command-line-options"},{"type":"task","title":"mix deps.get","doc":"Gets all out of date dependencies, i.e. dependencies\nthat are not available or have an invalid lock.","ref":"Mix.Tasks.Deps.Get.html"},{"type":"task","title":"Command line options - mix deps.get","doc":"* `--check-locked` - raises if there are pending changes to the lockfile\n  * `--no-archives-check` - does not check archives before fetching deps\n  * `--only` - only fetches dependencies for given environment","ref":"Mix.Tasks.Deps.Get.html#module-command-line-options"},{"type":"task","title":"mix deps.loadpaths","doc":"Checks, compiles, and loads all dependencies along the way.\n\nIf there is an invalid dependency, its status is printed\nbefore aborting.\n\nAlthough this task does not show up in `mix help`, it is\npart of Mix public API and can be depended on.","ref":"Mix.Tasks.Deps.Loadpaths.html"},{"type":"task","title":"Command line options - mix deps.loadpaths","doc":"* `--no-archives-check` - does not check archives\n  * `--no-compile` - does not compile even if files require compilation\n  * `--no-deps-check` - does not check or compile deps, only load available ones\n  * `--no-elixir-version-check` - does not check Elixir version\n  * `--no-optional-deps` - does not compile or load optional deps","ref":"Mix.Tasks.Deps.Loadpaths.html#module-command-line-options"},{"type":"task","title":"mix deps.precompile","doc":"Extension point for precompiling dependencies.\n\nThis is a task that can be aliased by projects\nthat need to execute certain tasks before\ncompiling dependencies:\n\n    aliases: [\"deps.precompile\": [\"nerves.precompile\", \"deps.precompile\"]]","ref":"Mix.Tasks.Deps.Precompile.html"},{"type":"task","title":"mix deps.tree","doc":"Prints the dependency tree.\n\n    $ mix deps.tree\n\nIf no dependency is given, it uses the tree defined in the `mix.exs` file.","ref":"Mix.Tasks.Deps.Tree.html"},{"type":"task","title":"Command line options - mix deps.tree","doc":"* `--only` - the environment to show dependencies for\n\n  * `--target` - the target to show dependencies for\n\n  * `--exclude` - exclude dependencies which you do not want to see printed.\n\n  * `--format` - Can be set to one of either:\n\n    * `pretty` - uses Unicode code points for formatting the tree.\n      This is the default except on Windows.\n\n    * `plain` - does not use Unicode code points for formatting the tree.\n      This is the default on Windows.\n\n    * `dot` - produces a DOT graph description of the dependency tree\n      in `deps_tree.dot` in the current directory.\n      Warning: this will override any previously generated file.","ref":"Mix.Tasks.Deps.Tree.html#module-command-line-options"},{"type":"task","title":"mix deps.unlock","doc":"Unlocks the given dependencies.\n\nSince this is a destructive action, unlocking dependencies\nonly occurs when passing arguments/options:\n\n  * `dep1 dep2` - the name of dependencies to be unlocked\n  * `--all` - unlocks all dependencies\n  * `--filter` - unlocks only deps matching the given name\n  * `--unused` - unlocks only unused dependencies (no longer mentioned\n    in the `mix.exs` file)\n  * `--check-unused` - checks that the `mix.lock` file has no unused\n    dependencies. This is useful in pre-commit hooks and CI scripts\n    if you want to reject contributions with extra dependencies","ref":"Mix.Tasks.Deps.Unlock.html"},{"type":"task","title":"mix deps.update","doc":"Updates the given dependencies.\n\nThe given dependencies and the projects they depend on will\nbe unlocked and updated to the latest version according to their\nversion requirements.\n\nSince this is a destructive action, updating all dependencies\nonly occurs when the `--all` command line option is passed.\n\nAll dependencies are automatically recompiled after update.","ref":"Mix.Tasks.Deps.Update.html"},{"type":"task","title":"mix deps.unlock + mix deps.get - mix deps.update","doc":"Upgrading a dependency often requires the projects it depends on\nto upgrade too. If you would rather update a single dependency and\nnot touch its children, you can explicitly unlock the single dependency\nand run `mix deps.get`:\n\n    $ mix deps.unlock some_dep\n    $ mix deps.get","ref":"Mix.Tasks.Deps.Update.html#module-mix-deps-unlock-mix-deps-get"},{"type":"task","title":"Command line options - mix deps.update","doc":"* `--all` - updates all dependencies\n  * `--only` - only fetches dependencies for given environment\n  * `--target` - only fetches dependencies for given target\n  * `--no-archives-check` - does not check archives before fetching deps","ref":"Mix.Tasks.Deps.Update.html#module-command-line-options"},{"type":"task","title":"mix do","doc":"Executes the tasks separated by `+`:\n\n    $ mix do compile --list + deps\n\nThe plus should be followed by at least one space before and after.","ref":"Mix.Tasks.Do.html"},{"type":"task","title":"Examples - mix do","doc":"The example below prints the available compilers and\nthen the list of dependencies.\n\n    $ mix do compile --list + deps\n\nNote that the majority of Mix tasks are only executed once\nper invocation. So for example, the following command will\nonly compile once:\n\n    $ mix do compile + some_other_command + compile\n\nWhen `compile` is executed again, Mix will notice the task\nhas already ran, and skip it.\n\nInside umbrella projects, you can limit recursive tasks\n(the ones that run inside every app) by selecting the\ndesired application via the `--app` flag after `do` and\nbefore the first task:\n\n    $ mix do --app app1 --app app2 compile --list + deps\n\nElixir versions prior to v1.14 used the comma exclusively\nto separate commands:\n\n    $ mix do compile --list, deps\n\nSince then, the `+` operator has been introduced as a\nseparator for better support on Windows terminals.","ref":"Mix.Tasks.Do.html#module-examples"},{"type":"task","title":"Command line options - mix do","doc":"* `--app` - limit recursive tasks to the given apps.\n    This option may be given multiple times and must come\n    before any of the tasks.","ref":"Mix.Tasks.Do.html#module-command-line-options"},{"type":"task","title":"mix escript","doc":"Lists all installed escripts.\n\nEscripts are installed at `~/.mix/escripts`. Add that path to your `$PATH` environment variable\nto be able to run installed escripts from any directory.","ref":"Mix.Tasks.Escript.html"},{"type":"task","title":"mix escript.build","doc":"Builds an escript for the project.\n\nAn escript is an executable that can be invoked from the\ncommand line. An escript can run on any machine that has\nErlang/OTP installed and by default does not require Elixir to\nbe installed, as Elixir is embedded as part of the escript.\n\nThis task guarantees the project and its dependencies are\ncompiled and packages them inside an escript. Before invoking\n`mix escript.build`, it is only necessary to define a `:escript`\nkey with a `:main_module` option in your `mix.exs` file:\n\n    escript: [main_module: MyApp.CLI]\n\nEscripts should be used as a mechanism to share scripts between\ndevelopers and not as a deployment mechanism. For running live\nsystems, consider using `mix run` or building releases. See\nthe `Application` module for more information on systems\nlife-cycles.\n\nAll of the configuration defined in `config/config.exs` will\nbe included as part of the escript. `config/runtime.exs` is also\nincluded for Elixir escripts. Once the configuration is loaded,\nthis task starts the current application. If this is not desired,\nset the `:app` configuration to nil.\n\nThis task also removes documentation and debugging chunks from\nthe compiled `.beam` files to reduce the size of the escript.\nIf this is not desired, check the `:strip_beams` option.\n\n> #### `priv` directory support {: .warning}\n>\n> escripts do not support projects and dependencies\n> that need to store or read artifacts from the priv directory.","ref":"Mix.Tasks.Escript.Build.html"},{"type":"task","title":"Command line options - mix escript.build","doc":"Expects the same command line options as `mix compile`.","ref":"Mix.Tasks.Escript.Build.html#module-command-line-options"},{"type":"task","title":"Configuration - mix escript.build","doc":"The following option must be specified in your `mix.exs`\nunder the `:escript` key:\n\n  * `:main_module` - the module to be invoked once the escript starts.\n    The module must contain a function named `main/1` that will receive the\n    command line arguments. By default the arguments are given as a list of\n    binaries, but if project is configured with `language: :erlang` it will\n    be a list of charlists.\n\nThe remaining options can be specified to further customize the escript:\n\n  * `:name` - the name of the generated escript.\n    Defaults to app name.\n\n  * `:path` - the path to write the escript to.\n    Defaults to app name.\n\n  * `:app` - the app that starts with the escript.\n    Defaults to app name. Set it to `nil` if no application should\n    be started.\n\n  * `:strip_beams` - if `true` strips BEAM code in the escript to remove chunks\n    unnecessary at runtime, such as debug information and documentation.\n    Can be set to `[keep: [\"Docs\", \"Dbgi\"]]` to strip while keeping some chunks\n    that would otherwise be stripped, like docs, and debug info, for instance.\n    Defaults to `true`.\n\n  * `:embed_elixir` - if `true` embeds Elixir and its children apps\n    (`ex_unit`, `mix`, and the like) mentioned in the `:applications` list inside the\n    `application/0` function in `mix.exs`.\n\n    Defaults to `true` for Elixir projects, `false` for Erlang projects.\n\n    Note: if you set this to `false` for an Elixir project, you will have to add paths to Elixir's\n    `ebin` directories to `ERL_LIBS` environment variable when running the resulting escript, in\n    order for the code loader to be able to find `:elixir` application and its children\n    applications (if they are used).\n\n  * `:shebang` - shebang interpreter directive used to execute the escript.\n    Defaults to `\"#! /usr/bin/env escript\\n\"`.\n\n  * `:comment` - comment line to follow shebang directive in the escript.\n    Defaults to `\"\"`.\n\n  * `:emu_args` - emulator arguments to embed in the escript file.\n    Defaults to `\"\"`.\n\nThere is one project-level option that affects how the escript is generated:\n\n  * `language: :elixir | :erlang` - set it to `:erlang` for Erlang projects\n    managed by Mix. Doing so will ensure Elixir is not embedded by default.\n    Your app will still be started as part of escript loading, with the\n    config used during build.","ref":"Mix.Tasks.Escript.Build.html#module-configuration"},{"type":"task","title":"Example - mix escript.build","doc":"* `mix.exs`:\n\n      defmodule MyApp.MixProject do\n        use Mix.Project\n\n        def project do\n          [\n            app: :my_app,\n            version: \"0.0.1\",\n            escript: escript()\n          ]\n        end\n\n        def escript do\n          [main_module: MyApp.CLI]\n        end\n      end\n\n* `lib/cli.ex`:\n\n      defmodule MyApp.CLI do\n        def main(_args) do\n          IO.puts(\"Hello from MyApp!\")\n        end\n      end","ref":"Mix.Tasks.Escript.Build.html#module-example"},{"type":"task","title":"mix escript.install","doc":"Installs an escript locally.\n\nIf no argument is supplied but there is an escript in the project's root directory\n(created with `mix escript.build`), then the escript will be installed\nlocally. For example:\n\n    $ mix do escript.build + escript.install\n\nIf an argument is provided, it should be a local path to a prebuilt escript,\na Git repository, a GitHub repository, or a Hex package.\n\n    $ mix escript.install escript\n    $ mix escript.install path/to/escript\n    $ mix escript.install git https://path/to/git/repo\n    $ mix escript.install git https://path/to/git/repo branch git_branch\n    $ mix escript.install git https://path/to/git/repo tag git_tag\n    $ mix escript.install git https://path/to/git/repo ref git_ref\n    $ mix escript.install github user/project\n    $ mix escript.install github user/project branch git_branch\n    $ mix escript.install github user/project tag git_tag\n    $ mix escript.install github user/project ref git_ref\n    $ mix escript.install hex hex_package\n    $ mix escript.install hex hex_package 1.2.3\n\nAfter installation, the escript can be invoked as\n\n    $ ~/.mix/escripts/foo\n\nFor convenience, consider adding `~/.mix/escripts` directory to your\n`$PATH` environment variable. For more information, check the wikipedia\narticle on PATH: https://en.wikipedia.org/wiki/PATH_(variable)","ref":"Mix.Tasks.Escript.Install.html"},{"type":"task","title":"Command line options - mix escript.install","doc":"* `--sha512` - checks the escript matches the given SHA-512 checksum. Only\n    applies to installations via a local path\n\n  * `--force` - forces installation without a shell prompt; primarily\n    intended for automation in build systems like Make\n\n  * `--submodules` - fetches repository submodules before building escript from\n    Git or GitHub\n\n  * `--app` - specifies a custom app name to be used for building the escript\n    from Git, GitHub, or Hex\n\n  * `--organization` - set this for Hex private packages belonging to an\n    organization\n\n  * `--repo` - set this for self-hosted Hex instances, defaults to `hexpm`","ref":"Mix.Tasks.Escript.Install.html#module-command-line-options"},{"type":"task","title":"mix escript.uninstall","doc":"Uninstalls local escripts:\n\n    $ mix escript.uninstall escript_name","ref":"Mix.Tasks.Escript.Uninstall.html"},{"type":"task","title":"Command line options - mix escript.uninstall","doc":"* `--force` - forces uninstallation without a shell prompt; primarily\n    intended for automation","ref":"Mix.Tasks.Escript.Uninstall.html#module-command-line-options"},{"type":"task","title":"mix eval","doc":"Evaluates the given code within a configured application.\n\n    $ mix eval \"IO.puts(1 + 2)\"\n\nThe given code is evaluated after the current application\nhas been configured but without loading or starting them\n(some applications may be loaded as part of booting but\nthat's not guaranteed). See `mix run` for running your\napplication and scripts within a started application.\n\nThis task is designed to mirror the `bin/my_app eval` command\nin releases. It is typically used to invoke functions already\ndefined within your application. For example, you may have a\nmodule such as:\n\n    defmodule MyApp.ReleaseTasks do\n      def migrate_database do\n        Application.load(:my_app)\n        ...\n      end\n    end\n\nOnce defined, you can invoke this function either via `mix eval` or\nvia `bin/my_app eval` inside a release as follows:\n\n    $ mix eval MyApp.ReleaseTasks.migrate_database\n    $ bin/my_app eval MyApp.ReleaseTasks.migrate_database\n\nAs you can see, the current application has to be either explicitly\nloaded or started in your tasks, either by calling `Application.load/1`\nor `Application.ensure_all_started/1`. This gives you full control over\nthe application booting life-cycle. For more information, see the\n`Application` module.\n\nThis task is automatically re-enabled, so it can be called multiple\ntimes with different arguments.","ref":"Mix.Tasks.Eval.html"},{"type":"task","title":"Command-line options - mix eval","doc":"* `--no-archives-check` - does not check archives\n  * `--no-compile` - does not compile even if files require compilation\n  * `--no-deps-check` - does not check dependencies\n  * `--no-elixir-version-check` - does not check the Elixir version from mix.exs\n  * `--no-mix-exs` - allows the command to run even if there is no mix.exs","ref":"Mix.Tasks.Eval.html#module-command-line-options"},{"type":"task","title":"mix format","doc":"Formats the given files and patterns.\n\n    $ mix format mix.exs \"lib/**/*.{ex,exs}\" \"test/**/*.{ex,exs}\"\n\nIf any of the files is `-`, then the input is read from stdin and the output\nis written to stdout.","ref":"Mix.Tasks.Format.html"},{"type":"task","title":"Formatting options - mix format","doc":"The formatter will read a `.formatter.exs` file in the current directory for\nformatter configuration. Evaluating this file should return a keyword list.\n\nHere is an example of a `.formatter.exs` file that works as a starting point:\n\n    [\n      inputs: [\"{mix,.formatter}.exs\", \"{config,lib,test}/**/*.{ex,exs}\"]\n    ]\n\nBesides the options listed in `Code.format_string!/2`, the `.formatter.exs`\nfile supports the following options:\n\n  * `:inputs` (a list of paths and patterns) - specifies the default inputs\n    to be used by this task. For example, `[\"mix.exs\", \"{config,lib,test}/**/*.{ex,exs}\"]`.\n    Patterns are expanded with `Path.wildcard/2`.\n\n  * `:plugins` (a list of modules) (since v1.13.0) - specifies a list of\n    modules to customize how the formatter works. See the \"Plugins\" section\n    below for more information.\n\n  * `:subdirectories` (a list of paths and patterns) - specifies subdirectories\n    that have their own formatting rules. Each subdirectory should have a\n    `.formatter.exs` that configures how entries in that subdirectory should be\n    formatted as. Configuration between `.formatter.exs` are not shared nor\n    inherited. If a `.formatter.exs` lists \"lib/app\" as a subdirectory, the rules\n    in `.formatter.exs` won't be available in `lib/app/.formatter.exs`.\n    Note that the parent `.formatter.exs` must not specify files inside the \"lib/app\"\n    subdirectory in its `:inputs` configuration. If this happens, the behaviour of\n    which formatter configuration will be picked is unspecified.\n\n  * `:import_deps` (a list of dependencies as atoms) - specifies a list\n     of dependencies whose formatter configuration will be imported.\n     See the \"Importing dependencies configuration\" section below for more\n     information.\n\n  * `:export` (a keyword list) - specifies formatter configuration to be exported.\n    See the \"Importing dependencies configuration\" section below.","ref":"Mix.Tasks.Format.html#module-formatting-options"},{"type":"task","title":"Task-specific options - mix format","doc":"* `--check-formatted` - checks that the file is already formatted.\n    This is useful in pre-commit hooks and CI scripts if you want to\n    reject contributions with unformatted code. If the check fails,\n    the formatted contents are not written to disk. Keep in mind\n    that the formatted output may differ between Elixir versions as\n    improvements and fixes are applied to the formatter.\n\n  * `--no-exit` - only valid when used with `--check-formatted`.\n    Pass this if you don't want this Mix task to fail (and return a non-zero exit code),\n    but still want to check for format errors and print them to the console.\n\n  * `--dry-run` - does not save files after formatting.\n\n  * `--dot-formatter` - path to the file with formatter configuration.\n    Defaults to `.formatter.exs` if one is available. See the\n    \"Formatting options\" section above for more information.\n\n  * `--stdin-filename` - path to the file being formatted on stdin.\n    This is useful if you are using plugins to support custom filetypes such\n    as `.heex`. Without passing this flag, it is assumed that the code being\n    passed via stdin is valid Elixir code. Defaults to \"stdin.exs\".","ref":"Mix.Tasks.Format.html#module-task-specific-options"},{"type":"task","title":"When to format code - mix format","doc":"We recommend developers to format code directly in their editors, either\nautomatically when saving a file or via an explicit command or key binding. If\nsuch option is not available in your editor of choice, adding the required\nintegration is usually a matter of invoking:\n\n    $ cd $project && mix format $file\n\nwhere `$file` refers to the current file and `$project` is the root of your\nproject.\n\nIt is also possible to format code across the whole project by passing a list\nof patterns and files to `mix format`, as shown at the top of this task\ndocumentation. This list can also be set in the `.formatter.exs` file under the\n`:inputs` key.","ref":"Mix.Tasks.Format.html#module-when-to-format-code"},{"type":"task","title":"Plugins - mix format","doc":"It is possible to customize how the formatter behaves. Plugins must implement\nthe `Mix.Tasks.Format` behaviour. For example, imagine that your project uses\nMarkdown in two distinct ways: via a custom `~M` sigil and via files with the\n`.md` and `.markdown` extensions. A custom plugin would look like this:\n\n    defmodule MixMarkdownFormatter do\n      @behaviour Mix.Tasks.Format\n\n      def features(_opts) do\n        [sigils: [:M], extensions: [\".md\", \".markdown\"]]\n      end\n\n      def format(contents, opts) do\n        # logic that formats markdown\n      end\n    end\n\nThe `opts` passed to `format/2` contains all the formatting options and either:\n\n  * `:sigil` (atom) - the sigil being formatted, e.g. `:M`.\n\n  * `:modifiers` (charlist) - list of sigil modifiers.\n\n  * `:extension` (string) - the extension of the file being formatted, e.g. `\".md\"`.\n\nNow any application can use your formatter as follows:\n\n    # .formatter.exs\n    [\n      # Define the desired plugins\n      plugins: [MixMarkdownFormatter, AnotherMarkdownFormatter],\n      # Remember to update the inputs list to include the new extensions\n      inputs: [\"{mix,.formatter}.exs\", \"{config,lib,test}/**/*.{ex,exs}\", \"posts/*.{md,markdown}\"]\n    ]\n\nRemember that, when running the formatter with plugins, you must make\nsure that your dependencies and your application have been compiled,\nso the relevant plugin code can be loaded. Otherwise a warning is logged.\n\nIn addition, the order by which you input your plugins is the format order.\nSo, in the above `.formatter.exs`, the `MixMarkdownFormatter` will format\nthe markdown files and sigils before `AnotherMarkdownFormatter`.","ref":"Mix.Tasks.Format.html#module-plugins"},{"type":"task","title":"Importing dependencies configuration - mix format","doc":"This task supports importing formatter configuration from dependencies.\n\nA dependency that wants to export formatter configuration needs to have a\n`.formatter.exs` file at the root of the project. In this file, the dependency\ncan list an `:export` option with configuration to export. For now, only one\noption is supported under `:export`: `:locals_without_parens` (whose value has\nthe same shape as the value of the `:locals_without_parens` in `Code.format_string!/2`).\n\nThe functions listed under `:locals_without_parens` in the `:export` option of\na dependency can be imported in a project by listing that dependency in the\n`:import_deps` option of the formatter configuration file of the project.\n\nFor example, consider you have a project called `my_app` that depends on another one called `my_dep`.\n`my_dep` wants to export some configuration, so `my_dep/.formatter.exs`\nwould look like this:\n\n    # my_dep/.formatter.exs\n    [\n      # Regular formatter configuration for my_dep\n      # ...\n\n      export: [\n        locals_without_parens: [some_dsl_call: 2, some_dsl_call: 3]\n      ]\n    ]\n\nIn order to import configuration, `my_app`'s `.formatter.exs` would look like\nthis:\n\n    # my_app/.formatter.exs\n    [\n      import_deps: [:my_dep]\n    ]","ref":"Mix.Tasks.Format.html#module-importing-dependencies-configuration"},{"type":"function","title":"Mix.Tasks.Format.formatter_for_file/2","doc":"Returns a formatter function and the formatter options to\nbe used for the given file.\n\nThe function must be called with the contents of the file\nto be formatted. The options are returned for reflection\npurposes.","ref":"Mix.Tasks.Format.html#formatter_for_file/2"},{"type":"function","title":"Mix.Tasks.Format.formatter_opts_for_file/2","doc":"Returns formatter options to be used for the given file.","ref":"Mix.Tasks.Format.html#formatter_opts_for_file/2"},{"type":"task","title":"mix help","doc":"Lists all tasks and aliases or prints the documentation for a given task or alias.","ref":"Mix.Tasks.Help.html"},{"type":"task","title":"Arguments - mix help","doc":"$ mix help                  - prints all aliases, tasks and their short descriptions\n    $ mix help ALIAS            - prints the definition for the given alias\n    $ mix help TASK             - prints full docs for the given task\n    $ mix help --search PATTERN - prints all tasks and aliases that contain PATTERN in the name\n    $ mix help --names          - prints all task names and aliases\n                                (useful for autocompleting)","ref":"Mix.Tasks.Help.html#module-arguments"},{"type":"task","title":"Colors - mix help","doc":"When possible, `mix help` is going to use coloring for formatting\nthe help information. The formatting can be customized by configuring\nthe Mix application either inside your project (in `config/config.exs`)\nor by using the local config (in `~/.mix/config.exs`).\n\nFor example, to disable color, one may use the configuration:\n\n    [mix: [colors: [enabled: false]]]\n\nThe available color options are:\n\n  * `:enabled`         - shows ANSI formatting (defaults to `IO.ANSI.enabled?/0`)\n  * `:doc_code`        - the attributes for code blocks (cyan, bright)\n  * `:doc_inline_code` - inline code (cyan)\n  * `:doc_headings`    - h1 and h2 (yellow, bright)\n  * `:doc_title`       - the overall heading for the output (reverse, yellow, bright)\n  * `:doc_bold`        - (bright)\n  * `:doc_underline`   - (underline)","ref":"Mix.Tasks.Help.html#module-colors"},{"type":"task","title":"mix iex","doc":"A task that simply instructs users to run `iex -S mix`.","ref":"Mix.Tasks.Iex.html"},{"type":"task","title":"mix loadconfig","doc":"Loads and persists the given configuration.\n\n    $ mix loadconfig path/to/config.exs\n\nAny configuration file loaded with `loadconfig` is treated\nas a compile-time configuration.\n\nNote that \"config/config.exs\" is always loaded automatically\nby the Mix CLI when it boots. \"config/runtime.exs\" is loaded\nautomatically by `mix app.config` before starting the current\napplication. Therefore there is no need to load those config\nfiles directly.\n\nThis task is automatically re-enabled, so it can be called\nmultiple times to load different configs.","ref":"Mix.Tasks.Loadconfig.html"},{"type":"task","title":"mix loadpaths","doc":"Loads the application and its dependencies paths.\n\nThis task is never directly invoked from the command line,\nbut it is rather used as building block by other tasks.\n\nDependencies are checked, compiled, and loaded. Each step\ncan be explicitly disabled with flags.","ref":"Mix.Tasks.Loadpaths.html"},{"type":"task","title":"Configuration - mix loadpaths","doc":"* `:elixir` - matches the current Elixir version against the\n    given requirement","ref":"Mix.Tasks.Loadpaths.html#module-configuration"},{"type":"task","title":"Command line options - mix loadpaths","doc":"* `--no-archives-check` - does not check archives\n  * `--no-compile` - does not compile dependencies, only check and load them\n  * `--no-deps-check` - does not check dependencies, only load available ones\n  * `--no-elixir-version-check` - does not check Elixir version\n  * `--no-optional-deps` - does not compile or load optional deps","ref":"Mix.Tasks.Loadpaths.html#module-command-line-options"},{"type":"task","title":"mix local","doc":"Lists tasks installed locally via archives.","ref":"Mix.Tasks.Local.html"},{"type":"task","title":"mix local.hex","doc":"Installs Hex locally.\n\n    $ mix local.hex [version]\n\nBy default the latest compatible version of Hex will be installed, unless\n`version` is specified.\n\nIf installing a precompiled Hex does not work, you can compile and install\nHex directly with this command:\n\n    $ mix archive.install github hexpm/hex branch latest","ref":"Mix.Tasks.Local.Hex.html"},{"type":"task","title":"Command line options - mix local.hex","doc":"* `--force` - forces installation without a shell prompt; primarily\n    intended for automation in build systems like `make`\n\n  * `--if-missing` - performs installation only if Hex is not installed yet;\n    intended to avoid repeatedly reinstalling Hex in automation when a script\n    may be run multiple times\n\nIf both options are set, the shell prompt is skipped and Hex is not\nre-installed if it was already installed.","ref":"Mix.Tasks.Local.Hex.html#module-command-line-options"},{"type":"task","title":"Mirrors - mix local.hex","doc":"If you want to change the [default mirror](https://builds.hex.pm)\nused for fetching Hex, set the `HEX_BUILDS_URL` environment variable.","ref":"Mix.Tasks.Local.Hex.html#module-mirrors"},{"type":"task","title":"mix local.public_keys","doc":"Public keys are used by Mix to install packages like Rebar and Hex.\n\nMix by default ships with a public key but new ones can be added\non demand.\n\nTo list all available keys:\n\n    $ mix local.public_keys\n\nTo list all available keys showing the keys themselves:\n\n    $ mix local.public_keys --detailed\n\nTo add a new key:\n\n    $ mix local.public_keys local/path/to/key\n\nBe careful when adding new keys. Only add keys from sources you\ntrust.\n\nPublic keys are by default stored in your MIX_HOME under the\npublic_keys directory.","ref":"Mix.Tasks.Local.PublicKeys.html"},{"type":"task","title":"Command line options - mix local.public_keys","doc":"* `--force` - forces installation without a shell prompt; primarily\n    intended for automation in build systems like `make`","ref":"Mix.Tasks.Local.PublicKeys.html#module-command-line-options"},{"type":"task","title":"mix local.rebar","doc":"Fetches a copy of `rebar3` from the given path or URL.\n\nIt defaults to safely download a Rebar copy from Hex's CDN.\nHowever, a URL can be given as an argument, usually for an existing\nlocal copy of Rebar:\n\n    $ mix local.rebar rebar3 path/to/rebar\n\nThe local copy is stored in your `MIX_HOME` (defaults to `~/.mix`)\naccording to the current Elixir. The installed version of Rebar will\nbe used whenever required by `mix deps.compile`.","ref":"Mix.Tasks.Local.Rebar.html"},{"type":"task","title":"Command line options - mix local.rebar","doc":"* `rebar3 PATH` - specifies a path for `rebar3`\n\n  * `--sha512` - checks the Rebar script matches the given SHA-512 checksum\n\n  * `--force` - forces installation without a shell prompt; primarily\n    intended for automation in build systems like `make`\n\n  * `--if-missing` - performs installation only if not installed yet;\n    intended to avoid repeatedly reinstalling in automation when a script\n    may be run multiple times","ref":"Mix.Tasks.Local.Rebar.html#module-command-line-options"},{"type":"task","title":"Mirrors - mix local.rebar","doc":"If you want to change the [default mirror](https://builds.hex.pm)\nto use for fetching `rebar` please set the `HEX_BUILDS_URL` environment variable.","ref":"Mix.Tasks.Local.Rebar.html#module-mirrors"},{"type":"task","title":"mix new","doc":"Creates a new Elixir project.\nIt expects the path of the project as argument.\n\n    $ mix new PATH [--app APP] [--module MODULE] [--sup] [--umbrella]\n\nA project at the given PATH will be created. The\napplication name and module name will be retrieved\nfrom the path, unless `--module` or `--app` is given.\n\nAn `--app` option can be given in order to\nname the OTP application for the project.\n\nA `--module` option can be given in order\nto name the modules in the generated code skeleton.\n\nA `--sup` option can be given to generate an OTP application\nskeleton including a supervision tree. Normally an app is\ngenerated without a supervisor and without the app callback.\n\nAn `--umbrella` option can be given to generate an\numbrella project.","ref":"Mix.Tasks.New.html"},{"type":"task","title":"Examples - mix new","doc":"$ mix new hello_world\n\nIs equivalent to:\n\n    $ mix new hello_world --module HelloWorld\n\nTo generate an app with a supervision tree and an application callback:\n\n    $ mix new hello_world --sup\n\nTo generate an umbrella application with sub applications:\n\n    $ mix new hello_world --umbrella\n    $ cd hello_world/apps\n    $ mix new child_app","ref":"Mix.Tasks.New.html#module-examples"},{"type":"function","title":"Mix.Tasks.New.reserved_application_names/0","doc":"Returns a list of reserved application names.","ref":"Mix.Tasks.New.html#reserved_application_names/0"},{"type":"task","title":"mix profile.cprof","doc":"Profiles the given file or expression using Erlang's `cprof` tool.\n\n`cprof` can be useful when you want to discover the bottlenecks related\nto function calls.\n\nBefore running the code, it invokes the `app.start` task which compiles\nand loads your project. After that, the target expression is profiled together\nwith all matching function calls, by setting breakpoints containing\ncounters. These can only be set on BEAM code so BIFs cannot be call\ncount traced.\n\nTo profile the code, you can use syntax similar to the `mix run` task:\n\n    $ mix profile.cprof -e Hello.world\n    $ mix profile.cprof -e \"[1, 2, 3] |> Enum.reverse |> Enum.map(&Integer.to_string/1)\"\n    $ mix profile.cprof my_script.exs arg1 arg2 arg3\n\nThis task is automatically re-enabled, so you can profile multiple times\nin the same Mix invocation.","ref":"Mix.Tasks.Profile.Cprof.html"},{"type":"task","title":"Command line options - mix profile.cprof","doc":"* `--matching` - only profile calls matching the given `Module.function/arity` pattern\n  * `--limit` - filters out any results with a call count less than the limit\n  * `--module` - filters out any results not pertaining to the given module\n  * `--eval`, `-e` - evaluate the given code\n  * `--require`, `-r` - requires pattern before running the command\n  * `--parallel`, `-p` - makes all requires parallel\n  * `--no-compile` - does not compile even if files require compilation\n  * `--no-deps-check` - does not check dependencies\n  * `--no-archives-check` - does not check archives\n  * `--no-halt` - does not halt the system after running the command\n  * `--no-start` - does not start applications after compilation\n  * `--no-elixir-version-check` - does not check the Elixir version from mix.exs","ref":"Mix.Tasks.Profile.Cprof.html#module-command-line-options"},{"type":"task","title":"Profile output - mix profile.cprof","doc":"Example output:\n                                                                         CNT\n    Total                                                                 15\n    Enum                                                                   6  <--\n      Enum.\"-map/2-lists^map/1-0-\"/2                                       4\n      Enum.reverse/1                                                       1\n      Enum.map/2                                                           1\n    :elixir_compiler                                                       4  <--\n      anonymous fn/1 in :elixir_compiler.__FILE__/1                        3\n      anonymous fn/0 in :elixir_compiler.__FILE__/1                        1\n    String.Chars.Integer                                                   3  <--\n      String.Chars.Integer.to_string/1                                     3\n    :erlang                                                                2  <--\n      :erlang.trace_pattern/3                                              2\n    Profile done over 20229 matching functions\n\nThe default output contains data gathered from all matching functions. The left\ncolumn structures each module and its total call count trace is presented on the right.\nEach module has its count discriminated by function below. The `<--` symbol is meant to\nhelp visualize where a new module call count begins.\n\nThe first row (Total) is the sum of all function calls. In the last row the number of\nmatching functions that were considered for profiling is presented.\n\nWhen `--matching` option is specified, call count tracing will be started only for\nthe functions matching the given pattern:\n\n    String.Chars.Integer                                                   3  <--\n      String.Chars.Integer.to_string/1                                     3\n    Profile done over 1 matching functions\n\nThe pattern can be a module name, such as `String` to count all calls to that module,\na call without arity, such as `String.split`, to count all calls to that function\nregardless of arity, or a call with arity, such as `String.split/3`, to count all\ncalls to that exact module, function and arity.","ref":"Mix.Tasks.Profile.Cprof.html#module-profile-output"},{"type":"task","title":"Caveats - mix profile.cprof","doc":"You should be aware the profiler is stopped as soon as the code has finished running. This\nmay need special attention, when:  running asynchronous code as function calls which were\ncalled before the profiler stopped will not be counted; running synchronous code as long\nrunning computations and a profiler without a proper MFA trace pattern or filter may\nlead to a result set which is difficult to comprehend.\n\nOther caveats are the impossibility to call count trace BIFs, since breakpoints can\nonly be set on BEAM code; functions calls performed by `:cprof` are not traced; the\nmaximum size of a call counter is equal to the host machine's word size\n(for example, 2147483647 in a 32-bit host).","ref":"Mix.Tasks.Profile.Cprof.html#module-caveats"},{"type":"function","title":"Mix.Tasks.Profile.Cprof.profile/2","doc":"Allows to programmatically run the `cprof` profiler on expression in `fun`.\n\nReturns the return value of `fun`.","ref":"Mix.Tasks.Profile.Cprof.html#profile/2"},{"type":"function","title":"Options - Mix.Tasks.Profile.Cprof.profile/2","doc":"* `:matching` - only profile calls matching the given pattern in form of\n    `{module, function, arity}`, where each element may be replaced by `:_`\n    to allow any value\n  * `:limit` - filters out any results with a call count less than the limit\n  * `:module` - filters out any results not pertaining to the given module","ref":"Mix.Tasks.Profile.Cprof.html#profile/2-options"},{"type":"task","title":"mix profile.eprof","doc":"Profiles the given file or expression using Erlang's `eprof` tool.\n\n`:eprof` provides time information of each function call and can be useful\nwhen you want to discover the bottlenecks related to this.\n\nBefore running the code, it invokes the `app.start` task which compiles\nand loads your project. After that, the target expression is profiled together\nwith all matching function calls using the Erlang trace BIFs. The tracing of\nthe function calls for that is enabled when the profiling is begun, and\ndisabled when profiling is stopped.\n\nTo profile the code, you can use syntax similar to the `mix run` task:\n\n    $ mix profile.eprof -e Hello.world\n    $ mix profile.eprof -e \"[1, 2, 3] |> Enum.reverse |> Enum.map(&Integer.to_string/1)\"\n    $ mix profile.eprof my_script.exs arg1 arg2 arg3\n\nThis task is automatically re-enabled, so you can profile multiple times\nin the same Mix invocation.","ref":"Mix.Tasks.Profile.Eprof.html"},{"type":"task","title":"Command line options - mix profile.eprof","doc":"* `--matching` - only profile calls matching the given `Module.function/arity` pattern\n  * `--calls` - filters out any results with a call count lower than this\n  * `--time` - filters out any results that took lower than specified (in µs)\n  * `--sort` - sorts the results by `time` or `calls` (default: `time`)\n  * `--eval`, `-e` - evaluates the given code\n  * `--require`, `-r` - requires pattern before running the command\n  * `--parallel`, `-p` - makes all requires parallel\n  * `--no-warmup` - skips the warmup step before profiling\n  * `--no-compile` - does not compile even if files require compilation\n  * `--no-deps-check` - does not check dependencies\n  * `--no-archives-check` - does not check archives\n  * `--no-halt` - does not halt the system after running the command\n  * `--no-start` - does not start applications after compilation\n  * `--no-elixir-version-check` - does not check the Elixir version from mix.exs","ref":"Mix.Tasks.Profile.Eprof.html#module-command-line-options"},{"type":"task","title":"Profile output - mix profile.eprof","doc":"Example output:\n\n    #                                               CALLS     % TIME µS/CALL\n    Total                                              24 100.0   26    1.08\n    Enum.reduce_range_inc/4                             5  3.85    1    0.20\n    :erlang.make_fun/3                                  1  7.69    2    2.00\n    Enum.each/2                                         1  7.69    2    2.00\n    anonymous fn/0 in :elixir_compiler_0.__FILE__/1     1  7.69    2    2.00\n    :erlang.integer_to_binary/1                         5 15.39    4    0.80\n    :erlang.apply/2                                     1 15.39    4    4.00\n    anonymous fn/3 in Enum.each/2                       5 19.23    5    1.00\n    String.Chars.Integer.to_string/1                    5 23.08    6    1.20\n\n    Profile done over 8 matching functions\n\nThe default output contains data gathered from all matching functions. The first\nrow after the header contains the sums of the partial results and the average time\nfor all the function calls listed. The following rows contain the function call,\nfollowed by the number of times that the function was called, then by the percentage\nof time that the call uses, then the total time for that function in microseconds,\nand, finally, the average time per call in microseconds.\n\nWhen `--matching` option is specified, call count tracing will be started only for\nthe functions matching the given pattern:\n\n    #                                               CALLS     % TIME µS/CALL\n    Total                                               5 100.0    6    1.20\n    String.Chars.Integer.to_string/1                    5 100.0    6    1.20\n\n    Profile done over 1 matching functions\n\nThe pattern can be a module name, such as `String` to count all calls to that module,\na call without arity, such as `String.split`, to count all calls to that function\nregardless of arity, or a call with arity, such as `String.split/3`, to count all\ncalls to that exact module, function and arity.","ref":"Mix.Tasks.Profile.Eprof.html#module-profile-output"},{"type":"task","title":"Caveats - mix profile.eprof","doc":"You should be aware that the code being profiled is running in an anonymous\nfunction which is invoked by [`:eprof` module](https://www.erlang.org/doc/man/eprof.html).\nThus, you'll see some additional entries in your profile output. It is also\nimportant to note that the profiler is stopped as soon as the code has finished running,\nand this may need special attention, when: running asynchronous code as function calls which were\ncalled before the profiler stopped will not be counted; running synchronous code as long\nrunning computations and a profiler without a proper MFA trace pattern or filter may\nlead to a result set which is difficult to comprehend.\n\nYou should expect a slowdown in your code execution using this tool since `:eprof` has\nsome performance impact on the execution, but the impact is considerably lower than\n`Mix.Tasks.Profile.Fprof`. If you have a large system try to profile a limited\nscenario or focus on the main modules or processes. Another alternative is to use\n`Mix.Tasks.Profile.Cprof` that uses `:cprof` and has a low performance degradation effect.","ref":"Mix.Tasks.Profile.Eprof.html#module-caveats"},{"type":"function","title":"Mix.Tasks.Profile.Eprof.profile/2","doc":"Allows to programmatically run the `eprof` profiler on expression in `fun`.\n\nReturns the return value of `fun`.","ref":"Mix.Tasks.Profile.Eprof.html#profile/2"},{"type":"function","title":"Options - Mix.Tasks.Profile.Eprof.profile/2","doc":"* `:matching` - only profile calls matching the given pattern in form of\n    `{module, function, arity}`, where each element may be replaced by `:_`\n    to allow any value\n  * `:calls` - filters out any results with a call count lower than this\n  * `:time` - filters out any results that took lower than specified (in µs)\n  * `:sort` - sort the results by `:time` or `:calls` (default: `:time`)","ref":"Mix.Tasks.Profile.Eprof.html#profile/2-options"},{"type":"task","title":"mix profile.fprof","doc":"Profiles the given file or expression using Erlang's `fprof` tool.\n\n`fprof` can be useful when you want to discover the bottlenecks of a\nsequential code.\n\nBefore running the code, it invokes the `app.start` task which compiles\nand loads your project. After that, the target expression is profiled, together\nwith all processes which are spawned by it. Other processes (for example, those\nresiding in the OTP application supervision tree) are not profiled.\n\nTo profile the code, you can use syntax similar to the `mix run` task:\n\n    $ mix profile.fprof -e Hello.world\n    $ mix profile.fprof my_script.exs arg1 arg2 arg3\n\nThis task is automatically re-enabled, so you can profile multiple times\nin the same Mix invocation.","ref":"Mix.Tasks.Profile.Fprof.html"},{"type":"task","title":"Command line options - mix profile.fprof","doc":"* `--callers` - prints detailed information about immediate callers and called functions\n  * `--details` - includes profile data for each profiled process\n  * `--sort key` - sorts the output by given key: `acc` (default) or `own`\n  * `--trace-to-file` - uses a file to trace. Can improve performance and memory\n    usage for larger workloads\n  * `--eval`, `-e` - evaluates the given code\n  * `--require`, `-r` - requires pattern before running the command\n  * `--parallel`, `-p` - makes all requires parallel\n  * `--no-compile`    - does not compile even if files require compilation\n  * `--no-deps-check` - does not check dependencies\n  * `--no-archives-check` - does not check archives\n  * `--no-start` - does not start applications after compilation\n  * `--no-elixir-version-check` - does not check the Elixir version from mix.exs\n  * `--no-warmup` - does not execute code once before profiling","ref":"Mix.Tasks.Profile.Fprof.html#module-command-line-options"},{"type":"task","title":"Profile output - mix profile.fprof","doc":"Example output:\n    #                                        CNT    ACC (ms)    OWN (ms)\n    Total                                 200279    1972.188    1964.579\n    :fprof.apply_start_stop/4                  0    1972.188       0.012\n    anonymous fn/0 in :elixir_compiler_2       1    1972.167       0.001\n    Test.run/0                                 1    1972.166       0.007\n    Test.do_something/1                        3    1972.131       0.040\n    Test.bottleneck/0                          1    1599.490       0.007\n    ...\n\nThe default output contains data gathered from all profiled processes.\nAll times are wall clock milliseconds. The columns have the following meaning:\n\n  * CNT - total number of invocations of the given function\n  * ACC - total time spent in the function\n  * OWN - time spent in the function, excluding the time of called functions\n\nThe first row (Total) is the sum of all functions executed in all profiled\nprocesses. For the given output, we had a total of 200279 function calls and spent\nabout 2 seconds running the code.\n\nMore detailed information is returned if you provide the `--callers` and\n`--details` options.\n\nWhen `--callers` option is specified, you'll see expanded function entries:\n\n    Mod.caller1/0                             3     200.000       0.017\n    Mod.caller2/0                             2     100.000       0.017\n      Mod.some_function/0                     5     300.000       0.017  <--\n        Mod.called1/0                         4     250.000       0.010\n        Mod.called2/0                         1      50.000       0.030\n\nHere, the arrow (`<--`) indicates the __marked__ function - the function\ndescribed by this paragraph. You also see its immediate callers (above) and\ncalled functions (below).\n\nAll the values of caller functions describe the marked function. For example,\nthe first row means that `Mod.caller1/0` invoked `Mod.some_function/0` 3 times.\n200ms of the total time spent in `Mod.some_function/0` was spent processing\ncalls from this particular caller.\n\nIn contrast, the values for the called functions describe those functions, but\nin the context of the marked function. For example, the last row means that\n`Mod.called2/0` was called once by `Mod.some_function/0`, and in that case\nthe total time spent in the function was 50ms.\n\nFor a detailed explanation it's worth reading the analysis in\n[Erlang/OTP documentation for fprof](https://www.erlang.org/doc/man/fprof.html#analysis).","ref":"Mix.Tasks.Profile.Fprof.html#module-profile-output"},{"type":"task","title":"Caveats - mix profile.fprof","doc":"You should be aware that the code being profiled is running in an anonymous\nfunction which is invoked by [`:fprof` module](https://www.erlang.org/doc/man/fprof.html).\nThus, you'll see some additional entries in your profile output,\nsuch as `:fprof` calls, an anonymous\nfunction with high ACC time, or an `:undefined` function which represents\nthe outer caller (non-profiled code which started the profiler).\n\nAlso, keep in mind that profiling might significantly increase the running time\nof the profiled processes. This might skew your results if, for example, those\nprocesses perform some I/O operations, since running time of those operations\nwill remain unchanged, while CPU bound operations of the profiled processes\nmight take significantly longer. Thus, when profiling some intensive program,\ntry to reduce such dependencies, or be aware of the resulting bias.\n\nFinally, it's advised to profile your program with the `prod` environment, since\nthis should provide more realistic insights into bottlenecks.","ref":"Mix.Tasks.Profile.Fprof.html#module-caveats"},{"type":"function","title":"Mix.Tasks.Profile.Fprof.profile/2","doc":"Allows to programmatically run the `fprof` profiler on expression in `fun`.\n\nReturns the return value of `fun`.","ref":"Mix.Tasks.Profile.Fprof.html#profile/2"},{"type":"function","title":"Options - Mix.Tasks.Profile.Fprof.profile/2","doc":"* `:callers` - prints detailed information about immediate callers and called functions\n  * `:details` - includes profile data for each profiled process\n  * `:sort` - sorts the output by given key: `:acc` (default) or `:own`\n  * `:trace_to_file` - uses a file to trace. Can improve performance and memory\n    usage for larger workloads.","ref":"Mix.Tasks.Profile.Fprof.html#profile/2-options"},{"type":"task","title":"mix release","doc":"Assembles a self-contained release for the current project:\n\n    $ MIX_ENV=prod mix release\n    $ MIX_ENV=prod mix release NAME\n\nOnce a release is assembled, it can be packaged and deployed to a\ntarget, as long as the target runs on the same operating system (OS)\ndistribution and version as the machine running the `mix release`\ncommand.\n\nA release can be configured in your `mix.exs` file under the `:releases`\nkey inside `def project`:\n\n    def project do\n      [\n        releases: [\n          demo: [\n            include_executables_for: [:unix],\n            applications: [runtime_tools: :permanent]\n          ],\n\n          ...\n        ]\n      ]\n    end\n\nYou can specify multiple releases where the key is the release name\nand the value is a keyword list with the release configuration.\nReleasing a certain name is done with:\n\n    $ MIX_ENV=prod mix release demo\n\nIf the given name does not exist, an error is raised.\n\nIf `mix release` is invoked, without specifying a release name, and\nthere are multiple releases configured, an error will be raised\nunless you set `default_release: NAME` at the root of your project\nconfiguration.\n\nIf `mix release` is invoked and there are no releases configured, a\nrelease is assembled using the application name and default values.","ref":"Mix.Tasks.Release.html"},{"type":"task","title":"Why releases? - mix release","doc":"Releases allow developers to precompile and package all of their code\nand the runtime into a single unit. The benefits of releases are:\n\n  * Code preloading. The VM has two mechanisms for loading code:\n    interactive and embedded. By default, it runs in the interactive\n    mode which dynamically loads modules when they are used for the\n    first time. The first time your application calls `Enum.map/2`,\n    the VM will find the `Enum` module and load it. There's a downside:\n    when you start a new server in production, it may need to load\n    many other modules, causing the first requests to have an unusual\n    spike in response time. With releases, the system preloads\n    all modules and guarantees your system is ready to handle requests\n    after booting.\n\n  * Configuration and customization. Releases give developers fine\n    grained control over system configuration and the VM flags used\n    to start the system.\n\n  * Self-contained. A release does not require the source code to be\n    included in your production artifacts. All of the code is precompiled\n    and packaged. Releases do not even require Erlang or Elixir in your\n    servers, as it includes the Erlang VM and its runtime by default.\n    Furthermore, both Erlang and Elixir standard libraries are stripped\n    to bring only the parts you are actually using.\n\n  * Multiple releases. You can assemble different releases with\n    different configuration per application or even with different\n    applications altogether.\n\n  * Management scripts. Releases come with scripts to start, restart,\n    connect to the running system remotely, execute RPC calls, run as\n    daemon, run as a Windows service, and more.","ref":"Mix.Tasks.Release.html#module-why-releases"},{"type":"task","title":"Running the release - mix release","doc":"Once a release is assembled, you can start it by calling\n`bin/RELEASE_NAME start` inside the release. In production, you would do:\n\n    $ MIX_ENV=prod mix release\n    $ _build/prod/rel/my_app/bin/my_app start\n\n`bin/my_app start` will start the system connected to the current standard\ninput/output, where logs are also written to by default. This is the\npreferred way to run the system. Many tools, such as `systemd`, platforms\nas a service, such as Heroku, and many containers platforms, such as Docker,\nare capable of processing the standard input/output and redirecting\nthe log contents elsewhere. Those tools and platforms also take care\nof restarting the system in case it crashes.\n\nYou can also execute one-off commands, run the release as a daemon on\nUnix-like system, or install it as a service on Windows. We will take a\nlook at those next. You can also list all available commands by invoking\n`bin/RELEASE_NAME`.\n\n#","ref":"Mix.Tasks.Release.html#module-running-the-release"},{"type":"task","title":"One-off commands (eval and rpc) - mix release","doc":"If you want to invoke specific modules and functions in your release,\nyou can do so in two ways: using `eval` or `rpc`.\n\n    $ bin/RELEASE_NAME eval \"IO.puts(:hello)\"\n    $ bin/RELEASE_NAME rpc \"IO.puts(:hello)\"\n\nThe `eval` command starts its own instance of the VM but without\nstarting any of the applications in the release and without starting\ndistribution. For example, if you need to do some prep work before\nrunning the actual system, like migrating your database, `eval` can\nbe a good fit. Just keep in mind any application you may use during\neval has to be explicitly loaded and/or started.\n\nYou can start an application by calling `Application.ensure_all_started/1`.\nHowever, if for some reason you cannot start an application, maybe\nbecause it will run other services you do not want, you must at least\nload the application by calling `Application.load/1`. If you don't\nload the application, any attempt at reading its environment or\nconfiguration may fail. Note that if you start an application,\nit is automatically loaded before started.\n\nAnother way to run commands is with `rpc`, which will connect to the\nsystem currently running and instruct it to execute the given\nexpression. This means you need to guarantee the system was already\nstarted and be careful with the instructions you are executing.\nYou can also use `remote` to connect a remote IEx session to the\nsystem.\n\n##","ref":"Mix.Tasks.Release.html#module-one-off-commands-eval-and-rpc"},{"type":"task","title":"Helper module - mix release","doc":"As you operate your system, you may find yourself running some piece of code\nas a one-off command quite often. You may consider creating a module to group\nthese tasks:\n\n    # lib/my_app/release_tasks.ex\n    defmodule MyApp.ReleaseTasks do\n      def eval_purge_stale_data() do\n        # Eval commands needs to start the app before\n        # Or Application.load(:my_app) if you can't start it\n        Application.ensure_all_started(:my_app)\n\n        # Code that purges stale data\n        ...\n      end\n\n      def rpc_print_connected_users() do\n        # Code that print users connected to the current running system\n        ...\n      end\n    end\n\nIn the example above, we prefixed the function names with the command\nname used to execute them, but that is entirely optional.\n\nAnd to run them:\n\n    $ bin/RELEASE_NAME eval \"MyApp.ReleaseTasks.eval_purge_stale_data()\"\n    $ bin/RELEASE_NAME rpc \"MyApp.ReleaseTasks.rpc_print_connected_users()\"\n\n#","ref":"Mix.Tasks.Release.html#module-helper-module"},{"type":"task","title":"Daemon mode (Unix-like) - mix release","doc":"You can run the release in daemon mode with the command:\n\n    $ bin/RELEASE_NAME daemon\n\nIn daemon mode, the system is started on the background via\n[`run_erl`](https://www.erlang.org/doc/man/run_erl.html). You may also\nwant to enable [`heart`](https://www.erlang.org/doc/man/heart.html)\nin daemon mode so it automatically restarts the system in case\nof crashes. See the generated `releases/RELEASE_VSN/env.sh` file.\n\nThe daemon will write all of its standard output to the \"tmp/log/\"\ndirectory in the release root. You can watch the log file by doing\n`tail -f tmp/log/erlang.log.1` or similar. Once files get too large,\nthe index suffix will be incremented. A developer can also attach\nto the standard input of the daemon by invoking \"to_erl tmp/pipe/\"\nfrom the release root. However, note that attaching to the system\nshould be done with extreme care, since the usual commands for\nexiting an Elixir system, such as hitting Ctrl+C twice or Ctrl+\\,\nwill actually shut down the daemon. Therefore, using\n`bin/RELEASE_NAME remote` should be preferred, even in daemon mode.\n\nYou can customize the tmp directory used both for logging and for\npiping in daemon mode by setting the `RELEASE_TMP` environment\nvariable. See the \"Customization\" section.\n\n#","ref":"Mix.Tasks.Release.html#module-daemon-mode-unix-like"},{"type":"task","title":"Services mode (Windows) - mix release","doc":"While daemons are not available on Windows, it is possible to install a\nreleased system as a service on Windows with the help of\n[`erlsrv`](https://www.erlang.org/doc/man/erlsrv.html). This can be done by\nrunning:\n\n    $ bin/RELEASE_NAME install\n\nOnce installed, the service must be explicitly managed via the `erlsrv`\nexecutable, which is included in the `erts-VSN/bin` directory.\nThe service is not started automatically after installing.\n\nFor example, if you have a release named `demo`, you can install\nthe service and then start it from the release root as follows:\n\n    $ bin/demo install\n    $ erts-VSN/bin/erlsrv.exe start demo_demo\n\nThe name of the service is `demo_demo` because the name is built\nby concatenating the node name with the release name. Since Elixir\nautomatically uses the same name for both, the service will be\nreferenced as `demo_demo`.\n\nThe `install` command must be executed as an administrator.\n\n### `bin/RELEASE_NAME` commands\n\nThe following commands are supported by `bin/RELEASE_NAME`:\n\n```text\nstart        Starts the system\nstart_iex    Starts the system with IEx attached\ndaemon       Starts the system as a daemon (Unix-like only)\ndaemon_iex   Starts the system as a daemon with IEx attached (Unix-like only)\ninstall      Installs this system as a Windows service (Windows only)\neval \"EXPR\"  Executes the given expression on a new, non-booted system\nrpc \"EXPR\"   Executes the given expression remotely on the running system\nremote       Connects to the running system via a remote shell\nrestart      Restarts the running system via a remote command\nstop         Stops the running system via a remote command\npid          Prints the operating system PID of the running system via a remote command\nversion      Prints the release name and version to be booted\n```","ref":"Mix.Tasks.Release.html#module-services-mode-windows"},{"type":"task","title":"Deployments - mix release","doc":"#","ref":"Mix.Tasks.Release.html#module-deployments"},{"type":"task","title":"Requirements - mix release","doc":"A release is built on a **host**, a machine which contains Erlang, Elixir,\nand any other dependencies needed to compile your application. A release is\nthen deployed to a **target**, potentially the same machine as the host,\nbut usually separate, and often there are many targets (either multiple\ninstances, or the release is deployed to heterogeneous environments).\n\nTo deploy straight from a host to a separate target without cross-compilation,\nthe following must be the same between the host and the target:\n\n  * Target architecture (for example, x86_64 or ARM)\n  * Target vendor + operating system (for example, Windows, Linux, or Darwin/macOS)\n  * Target ABI (for example, musl or gnu)\n\nThis is often represented in the form of target triples, for example,\n`x86_64-unknown-linux-gnu`, `x86_64-unknown-linux-musl`, `x86_64-apple-darwin`.\n\nSo to be more precise, to deploy straight from a host to a separate target,\nthe Erlang Runtime System (ERTS), and any native dependencies (NIFs), must\nbe compiled for the same target triple. If you are building on a MacBook\n(`x86_64-apple-darwin`) and trying to deploy to a typical Ubuntu machine\n(`x86_64-unknown-linux-gnu`), the release will not work. Instead you should\nbuild the release on a `x86_64-unknown-linux-gnu` host. As we will see, this\ncan be done in multiple ways, such as releasing on the target itself, or by\nusing virtual machines or containers, usually as part of your release pipeline.\n\nIn addition to matching the target triple, it is also important that the\ntarget has all of the system packages that your application will need at\nruntime. A common one is the need for OpenSSL when building an application\nthat uses `:crypto` or `:ssl`, which is dynamically linked to ERTS. The other\ncommon source for native dependencies like this comes from dependencies\ncontaining NIFs (natively-implemented functions) which may expect to\ndynamically link to libraries they use.\n\nOf course, some operating systems and package managers can differ between\nversions, so if your goal is to have full compatibility between host and\ntarget, it is best to ensure the operating system and system package manager\nhave the same versions on host and target. This may even be a requirement in\nsome systems, especially so with package managers that try to create fully\nreproducible environments (Nix, Guix).\n\nSimilarly, when creating a stand-alone package and release for Windows, note\nthe Erlang Runtime System has a dependency to some Microsoft libraries\n(Visual C++ Redistributable Packages for Visual Studio 2013). These libraries\nare installed (if not present before) when Erlang is installed but it is not\npart of the standard Windows environment. Deploying a stand-alone release on\na computer without these libraries will result in a failure when trying to\nrun the release. One way to solve this is to download and install these\nMicrosoft libraries the first time a release is deployed (the Erlang installer\nversion 10.6 ships with “Microsoft Visual C++ 2013 Redistributable - 12.0.30501”).\n\nAlternatively, you can also bundle the compiled object files in the release,\nas long as they were compiled for the same target. If doing so, you need to\nupdate `LD_LIBRARY_PATH` environment variable with the paths containing the\nbundled objects on Unix-like systems or the `$PATH` environment variable on\nWindows systems.\n\nCurrently, there is no official way to cross-compile a release from one\ntarget triple to another, due to the complexities involved in the process.\n\n#","ref":"Mix.Tasks.Release.html#module-requirements"},{"type":"task","title":"Techniques - mix release","doc":"There are a couple of ways to guarantee that a release is built on a host with\nthe same properties as the target. A simple option is to fetch the source,\ncompile the code and assemble the release on the target itself. It would\nbe something like this:\n\n    $ git clone remote://path/to/my_app.git my_app_source\n    $ cd my_app_source\n    $ mix deps.get --only prod\n    $ MIX_ENV=prod mix release\n    $ _build/prod/rel/my_app/bin/my_app start\n\nIf you prefer, you can also compile the release to a separate directory,\nso you can erase all source after the release is assembled:\n\n    $ git clone remote://path/to/my_app.git my_app_source\n    $ cd my_app_source\n    $ mix deps.get --only prod\n    $ MIX_ENV=prod mix release --path ../my_app_release\n    $ cd ../my_app_release\n    $ rm -rf ../my_app_source\n    $ bin/my_app start\n\nHowever, this option can be expensive if you have multiple production\nnodes or if the release assembling process is a long one, as each node\nneeds to individually assemble the release.\n\nYou can automate this process in a couple different ways. One option\nis to make it part of your Continuous Integration (CI) / Continuous\nDeployment (CD) pipeline. When you have a CI/CD pipeline, it is common\nthat the machines in your CI/CD pipeline run on the exact same target\ntriple as your production servers (if they don't, they should).\nIn this case, you can assemble the release at the end of your CI/CD\npipeline by calling `MIX_ENV=prod mix release` and push the artifact\nto S3 or any other network storage. To perform the deployment, your\nproduction machines can fetch the deployment from the network storage\nand run `bin/my_app start`.\n\nAnother mechanism to automate deployments is to use images, such as\nAmazon Machine Images, or container platforms, such as Docker.\nFor instance, you can use Docker to run locally a system with the\nexact same target triple as your production servers. Inside the\ncontainer, you can invoke `MIX_ENV=prod mix release` and build\na complete image and/or container with the operating system, all\ndependencies as well as the releases.\n\nIn other words, there are multiple ways systems can be deployed and\nreleases can be automated and incorporated into all of them as long\nas you remember to build the system in the same target triple.\n\nOnce a system is deployed, shutting down the system can be done by\nsending SIGINT/SIGTERM to the system, which is what most containers,\nplatforms and tools do, or by explicitly invoking `bin/RELEASE_NAME stop`.\nOnce the system receives the shutdown request, each application and\ntheir respective supervision trees will stop, one by one, in the\nopposite order that they were started.","ref":"Mix.Tasks.Release.html#module-techniques"},{"type":"task","title":"Customization - mix release","doc":"There are a couple ways in which developers can customize the generated\nartifacts inside a release.\n\n#","ref":"Mix.Tasks.Release.html#module-customization"},{"type":"task","title":"Options - mix release","doc":"The following options can be set inside your `mix.exs` on each release definition:\n\n  * `:applications` - a keyword list with application names as keys and their\n    mode as value. By default `:applications` includes the current application and\n    all applications the current application depends on, recursively. You can include\n    new applications or change the mode of existing ones by listing them here.\n\n    The order of the applications given will be preserved as much as possible, with\n    only `:kernel`, `:stdlib`, `:sasl`, and `:elixir` listed before the given application\n    list. The supported values are:\n\n      * `:permanent` (default) - the application is started and the node shuts down\n        if the application terminates, regardless of reason\n      * `:transient` - the application is started and the node shuts down\n        if the application terminates abnormally\n      * `:temporary` - the application is started and the node does not\n        shut down if the application terminates\n      * `:load` - the application is only loaded\n      * `:none` - the application is part of the release but it is neither\n        loaded nor started\n\n    If you change the mode of an application, the mode will apply to all its child\n    applications. However, if an application has two parents, the mode of the parent\n    with highest priority wins (where `:permanent` has the highest priority, according\n    to the list above).\n\n  * `:strip_beams` - controls if BEAM files should have their debug information,\n    documentation chunks, and other non-essential metadata removed. Defaults to\n    `true`. May be set to `false` to disable stripping. Also accepts\n   `[keep: [\"Docs\", \"Dbgi\"]]` to keep certain chunks that are usually stripped.\n   You can also set the `:compress` option to true to enable individual\n   compression of BEAM files, although it is typically preferred to compress\n   the whole release instead.\n\n  * `:cookie` - a string representing the Erlang Distribution cookie. If this\n    option is not set, a random cookie is written to the `releases/COOKIE` file\n    when the first release is assembled. At runtime, we will first attempt\n    to fetch the cookie from the `RELEASE_COOKIE` environment variable and\n    then we'll read the `releases/COOKIE` file.\n\n    If you are setting this option manually, we recommend the cookie option\n    to be a long and randomly generated string, such as:\n    `Base.url_encode64(:crypto.strong_rand_bytes(40))`. We also recommend to restrict\n    the characters in the cookie to the subset returned by `Base.url_encode64/1`.\n\n  * `:validate_compile_env` - by default a release will match all runtime\n    configuration against any configuration that was marked at compile time\n    in your application of its dependencies via the `Application.compile_env/3`\n    function. If there is a mismatch between those, it means your system is\n    misconfigured and unable to boot. You can disable this check by setting\n    this option to false.\n\n  * `:path` - the path the release should be installed to.\n    Defaults to `\"_build/MIX_ENV/rel/RELEASE_NAME\"`.\n\n  * `:version` - the release version as a string or `{:from_app, app_name}`.\n    Defaults to the current application version. The `{:from_app, app_name}` format\n    can be used to easily reference the application version from another application.\n    This is particularly useful in umbrella applications.\n\n  * `:quiet` - a boolean that controls if releases should write steps to\n    the standard output. Defaults to `false`.\n\n  * `:include_erts` - a boolean, string, or anonymous function of arity zero.\n    If a boolean, it indicates whether the Erlang Runtime System (ERTS), which\n    includes the Erlang VM, should be included in the release. The default is\n    `true`, which is also the recommended value. If a string, it represents\n    the path to an existing ERTS installation. If an anonymous function of\n    arity zero, it's a function that returns any of the above (boolean or string).\n\n    You may also set this option to `false` if you desire to use the ERTS version installed\n    on the target. Note, however, that the ERTS version on the target must have **the\n    exact version** as the ERTS version used when the release is assembled. Setting it to\n    `false` also disables hot code upgrades. Therefore, `:include_erts` should be\n    set to `false` with caution and only if you are assembling the release on the\n    same server that runs it.\n\n  * `:include_executables_for` - a list of atoms detailing for which Operating\n    Systems executable files should be generated for. By default, it is set to\n    `[:unix, :windows]`. You can customize those as follows:\n\n        releases: [\n          demo: [\n            include_executables_for: [:unix] # Or [:windows] or []\n          ]\n        ]\n\n  * `:rel_templates_path` - the path to find template files that are copied to\n    the release, such as \"vm.args.eex\", \"remote.vm.args.eex\", \"env.sh.eex\"\n    (or \"env.bat.eex\"), and \"overlays\". Defaults to \"rel\" in the project root.\n\n  * `:overlays` - a list of directories with extra files to be copied\n    as is to the release. The \"overlays\" directory at `:rel_templates_path`\n    is always included in this list by default (typically at \"rel/overlays\").\n    See the \"Overlays\" section for more information.\n\n  * `:steps` - a list of steps to execute when assembling the release. See\n    the \"Steps\" section for more information.\n\n  * `:skip_mode_validation_for` - a list of application names\n    (atoms) specifying applications to skip strict validation of\n    \"unsafe\" modes. An \"unsafe\" case is when a parent application\n    mode is `:permanent` but one of the applications it depends on\n    is set to `:load`. Use this with care, as a release with\n    invalid modes may no longer boot without additional tweaks.\n    Defaults to `[]`.\n\nNote each release definition can be given as an anonymous function. This\nis useful if some release attributes are expensive to compute:\n\n    releases: [\n      demo: fn ->\n        [version: @version   \"+\"   git_ref()]\n      end\n    ]\n\nBesides the options above, it is possible to customize the generated\nrelease with custom files, by tweaking the release steps or by running\ncustom options and commands on boot. We will detail both approaches next.\n\n#","ref":"Mix.Tasks.Release.html#module-options"},{"type":"task","title":"Overlays - mix release","doc":"Often it is necessary to copy extra files to the release root after\nthe release is assembled. This can be easily done by placing such\nfiles in the `rel/overlays` directory. Any file in there is copied\nas is to the release root. For example, if you have placed a\n\"rel/overlays/Dockerfile\" file, the \"Dockerfile\" will be copied as\nis to the release root.\n\nIf you want to specify extra overlay directories, you can do so\nwith the `:overlays` option. If you need to copy files dynamically,\nsee the \"Steps\" section.\n\n#","ref":"Mix.Tasks.Release.html#module-overlays"},{"type":"task","title":"Steps - mix release","doc":"It is possible to add one or more steps before and after the release is\nassembled. This can be done with the `:steps` option:\n\n    releases: [\n      demo: [\n        steps: [&set_configs/1, :assemble, &copy_extra_files/1]\n      ]\n    ]\n\nThe `:steps` option must be a list and it must always include the\natom `:assemble`, which does most of the release assembling. You\ncan pass anonymous functions before and after the `:assemble` to\ncustomize your release assembling pipeline. Those anonymous functions\nwill receive a `Mix.Release` struct and must return the same or\nan updated `Mix.Release` struct. It is also possible to build a tarball\nof the release by passing the `:tar` step anywhere after `:assemble`.\nIf the release `:path` is not configured, the tarball is created in\n`_build/MIX_ENV/RELEASE_NAME-RELEASE_VSN.tar.gz` Otherwise it is\ncreated inside the configured `:path`.\n\nSee `Mix.Release` for more documentation on the struct and which\nfields can be modified. Note that the `:steps` field itself can be\nmodified and it is updated every time a step is called. Therefore,\nif you need to execute a command before and after assembling the\nrelease, you only need to declare the first steps in your pipeline\nand then inject the last step into the release struct. The steps\nfield can also be used to verify if the step was set before or\nafter assembling the release.\n\n#","ref":"Mix.Tasks.Release.html#module-steps"},{"type":"task","title":"vm.args and env.sh (env.bat) - mix release","doc":"Developers may want to customize the VM flags and environment variables\ngiven when the release starts. The simplest way to customize those files\nis by running `mix release.init`. The Mix task will copy custom\n`rel/vm.args.eex`, `rel/remote.vm.args.eex`,  `rel/env.sh.eex`, and\n`rel/env.bat.eex` files to your project root. You can modify those files\nand they will be evaluated every time you perform a new release. Those\nfiles are regular EEx templates and they have a single assign, called\n`@release`, with the `Mix.Release` struct.\n\nThe `vm.args` and `remote.vm.args` files may contain any of the VM flags\naccepted by the [`erl` command](https://www.erlang.org/doc/man/erl.html).\n\nThe `env.sh` and `env.bat` is used to set environment variables.\nIn there, you can set vars such as `RELEASE_NODE`, `RELEASE_COOKIE`,\nand `RELEASE_TMP` to customize your node name, cookie and tmp\ndirectory respectively. Whenever `env.sh` or `env.bat` is invoked,\nthe variables `RELEASE_ROOT`, `RELEASE_NAME`, `RELEASE_VSN`, and\n`RELEASE_COMMAND` have already been set, so you can rely on them.\nSee the section on environment variables for more information.\n\nFurthermore, while the `vm.args` files are static, you can use\n`env.sh` and `env.bat` to dynamically set VM options. For example,\nif you want to make sure the Erlang Distribution listens only on\na given port known at runtime, you can set the following:\n\n```bash\ncase $RELEASE_COMMAND in\n  start*|daemon*)\n    ELIXIR_ERL_OPTIONS=\"-kernel inet_dist_listen_min $BEAM_PORT inet_dist_listen_max $BEAM_PORT\"\n    export ELIXIR_ERL_OPTIONS\n    ;;\n  *)\n    ;;\nesac\n```\n\nNote we only set the port on start/daemon commands. If you also limit\nthe port on other commands, such as `rpc`, then you will be unable\nto establish a remote connection as the port will already be in use\nby the node.\n\nOn Windows, your `env.bat` would look like this:\n\n```bash\nIF NOT %RELEASE_COMMAND:start=%==%RELEASE_COMMAND% (\n  set ELIXIR_ERL_OPTIONS=\"-kernel inet_dist_listen_min %BEAM_PORT% inet_dist_listen_max %BEAM_PORT%\"\n)\n```","ref":"Mix.Tasks.Release.html#module-vm-args-and-env-sh-env-bat"},{"type":"task","title":"Application configuration - mix release","doc":"Mix provides two mechanisms for configuring the application environment\nof your application and your dependencies: build-time and runtime. On this\nsection, we will learn how those mechanisms apply to releases. An introduction\nto this topic can be found in the \"Configuration\" section of the `Mix` module.\n\n#","ref":"Mix.Tasks.Release.html#module-application-configuration"},{"type":"task","title":"Build-time configuration - mix release","doc":"Whenever you invoke a `mix` command, Mix loads the configuration in\n`config/config.exs`, if said file exists. We say that this configuration\nis a build-time configuration as it is evaluated whenever you compile your\ncode or whenever you assemble the release.\n\nIn other words, if your configuration does something like:\n\n    import Config\n    config :my_app, :secret_key, System.fetch_env!(\"MY_APP_SECRET_KEY\")\n\nThe `:secret_key` key under `:my_app` will be computed on the\nhost machine, whenever the release is built. Therefore if the machine\nassembling the release not have access to all environment variables used\nto run your code, loading the configuration will fail as the environment\nvariable is missing. Luckily, Mix also provides runtime configuration,\nwhich should be preferred and we will see next.\n\n#","ref":"Mix.Tasks.Release.html#module-build-time-configuration"},{"type":"task","title":"Runtime configuration - mix release","doc":"To enable runtime configuration in your release, all you need to do is\nto create a file named `config/runtime.exs`:\n\n    import Config\n    config :my_app, :secret_key, System.fetch_env!(\"MY_APP_SECRET_KEY\")\n\nThis file will be executed whenever your Mix project or your release\nstarts.\n\nYour `config/runtime.exs` file needs to follow three important rules:\n\n  * It MUST `import Config` at the top instead of the deprecated `use Mix.Config`\n  * It MUST NOT import any other configuration file via `import_config`\n  * It MUST NOT access `Mix` in any way, as `Mix` is a build tool and\n    it is not available inside releases\n\nIf a `config/runtime.exs` exists, it will be copied to your release\nand executed early in the boot process, when only Elixir and Erlang's\nmain applications have been started.\n\nYou can change the path to the runtime configuration file by setting\n`:runtime_config_path` inside each release configuration. This path is\nresolved at build time as the given configuration file is always copied\nto inside the release:\n\n    releases: [\n      demo: [\n        runtime_config_path: ...\n      ]\n    ]\n\nBy setting `:runtime_config_path` to `false` it can be used to prevent\na runtime configuration file to be included in the release.\n\n#","ref":"Mix.Tasks.Release.html#module-runtime-configuration"},{"type":"task","title":"Config providers - mix release","doc":"Releases also supports custom mechanisms, called config providers, to load\nany sort of runtime configuration to the system while it boots. For instance,\nif you need to access a vault or load configuration from a JSON file, it can\nbe achieved with config providers. The runtime configuration outlined in the\nprevious section is handled by the `Config.Reader` provider. See the\n`Config.Provider` module for more information and more examples.\n\nThe following options can be set inside your releases key in your `mix.exs`\nto control how config providers work:\n\n  * `:reboot_system_after_config` - reboot the system after configuration\n    so you can configure system applications, such as `:kernel` and `:stdlib`,\n    in your `config/runtime.exs`. Generally speaking, it is best to configure\n    `:kernel` and `:stdlib` using the `vm.args` file but this option is available\n    for those who need more complex configuration. When set to `true`, the\n    release will first boot in interactive mode to compute a config file and\n    write it to the \"tmp\" directory. Then it reboots in the configured `RELEASE_MODE`.\n    You can configure the \"tmp\" directory by setting the `RELEASE_TMP` environment\n    variable, either explicitly or inside your `releases/RELEASE_VSN/env.sh`\n    (or `env.bat` on Windows). Defaults to `true` if using the deprecated\n    `config/releases.exs`, `false` otherwise.\n\n  * `:prune_runtime_sys_config_after_boot` - if `:reboot_system_after_config`\n    is set, every time your system boots, the release will write a config file\n    to your tmp directory. These configuration files are generally small.\n    But if you are concerned with disk space or if you have other restrictions,\n    you can ask the system to remove said config files after boot. The downside\n    is that you will no longer be able to restart the system internally (neither\n    via `System.restart/0` nor `bin/RELEASE_NAME restart`). If you need a restart,\n    you will have to terminate the Operating System process and start a new\n    one. Defaults to `false`.\n\n  * `:start_distribution_during_config` - if `:reboot_system_after_config` is\n    set, releases only start the Erlang VM distribution features after the config\n    files are evaluated. You can set it to `true` if you need distribution during\n    configuration. Defaults to `false`.\n\n  * `:config_providers` - a list of tuples with custom config providers.\n    See `Config.Provider` for more information. Defaults to `[]`.\n\n#","ref":"Mix.Tasks.Release.html#module-config-providers"},{"type":"task","title":"Customization and configuration summary - mix release","doc":"Generally speaking, the following files are available for customizing\nand configuring the running system:\n\n  * `config/config.exs` (and `config/prod.exs`) - provides build-time\n    application configuration, which are executed when the release is\n    assembled\n\n  * `config/runtime.exs` - provides runtime application configuration.\n    It is executed every time your Mix project or your release boots\n    and is further extensible via config providers. If you want to\n    detect you are inside a release, you can check for release specific\n    environment variables, such as `RELEASE_NODE` or `RELEASE_MODE`\n\n  * `rel/vm.args.eex` and `rel/remote.vm.args.eex` - template files that\n    are copied into every release and provides static configuration of the\n    Erlang Virtual Machine and other runtime flags. `vm.args` runs on\n    `start`, `daemon`, and `eval` commands. `remote.vm.args` configures\n    the VM for `remote` and `rpc` commands\n\n  * `rel/env.sh.eex` and `rel/env.bat.eex` - template files that are copied\n    into every release and are executed on every command to set up environment\n    variables, including specific ones to the VM, and the general environment","ref":"Mix.Tasks.Release.html#module-customization-and-configuration-summary"},{"type":"task","title":"Directory structure - mix release","doc":"A release is organized as follows:\n\n```text\nbin/\n  RELEASE_NAME\nerts-ERTS_VSN/\nlib/\n  APP_NAME-APP_VSN/\n    ebin/\n    include/\n    priv/\nreleases/\n  RELEASE_VSN/\n    consolidated/\n    elixir\n    elixir.bat\n    env.bat\n    env.sh\n    iex\n    iex.bat\n    remote.vm.args\n    runtime.exs\n    start.boot\n    start.script\n    start_clean.boot\n    start_clean.script\n    sys.config\n    vm.args\n  COOKIE\n  start_erl.data\ntmp/\n```\n\nWe document this structure for completeness. In practice, developers\nshould not modify any of those files after the release is assembled.\nInstead use env scripts, custom config provider, overlays, and all\nother mechanisms described in this guide to configure how your release\nworks.","ref":"Mix.Tasks.Release.html#module-directory-structure"},{"type":"task","title":"Environment variables - mix release","doc":"The system sets different environment variables. The following variables\nare set early on and can only be read by `env.sh` and `env.bat`:\n\n  * `RELEASE_ROOT` - points to the root of the release. If the system\n    includes ERTS, then it is the same as `:code.root_dir/0`. This\n    variable is always computed and it cannot be set to a custom value\n\n  * `RELEASE_COMMAND` - the command given to the release, such as `\"start\"`,\n    `\"remote\"`, `\"eval\"`, and so on. This is typically accessed inside `env.sh`\n    and `env.bat` to set different environment variables under different\n    conditions. Note, however, that `RELEASE_COMMAND` has not been\n    validated by the time `env.sh` and `env.bat` are called, so it may\n    be empty or contain invalid values. This variable is always computed\n    and it cannot be set to a custom value\n\n  * `RELEASE_NAME` - the name of the release. It can be set to a custom\n    value when invoking the release\n\n  * `RELEASE_VSN` - the version of the release, otherwise the latest\n    version is used. It can be set to a custom value when invoking the\n    release. The custom value must be an existing release version in\n    the `releases/` directory\n\n  * `RELEASE_PROG` - the command line executable used to start the release\n\nThe following variables can be set before you invoke the release or\ninside `env.sh` and `env.bat`:\n\n  * `RELEASE_COOKIE` - the release cookie. By default uses the value\n    in `releases/COOKIE`. It can be set to a custom value\n\n  * `RELEASE_NODE` - the release node name, in the format `name` or\n    optionally `name@host` if running in distributed mode. It can be\n    set to a custom value. The name part must be made only of letters,\n    digits, underscores, and hyphens\n\n  * `RELEASE_SYS_CONFIG` - the location of the sys.config file. It can\n    be set to a custom path and it must not include the `.config` extension\n\n  * `RELEASE_VM_ARGS` - the location of the vm.args file. It can be set\n    to a custom path\n\n  * `RELEASE_REMOTE_VM_ARGS` - the location of the remote.vm.args file.\n    It can be set to a custom path\n\n  * `RELEASE_TMP` - the directory in the release to write temporary\n    files to. It can be set to a custom directory. It defaults to\n    `$RELEASE_ROOT/tmp`\n\n  * `RELEASE_MODE` - if the release should load code on demand (interactive)\n    or preload it (embedded). Defaults to \"embedded\", which increases boot\n    time but it means the runtime will respond faster as it doesn't have to\n    load code. Choose interactive if you need to decrease boot time and reduce\n    memory usage on boot. It applies only to start/daemon/install commands\n\n  * `RELEASE_DISTRIBUTION` - how do we want to run the distribution.\n    May be `name` (long names), `sname` (short names) or `none`\n    (distribution is not started automatically). Defaults to\n    `sname` which allows access only within the current system.\n    `name` allows external connections\n\n  * `RELEASE_BOOT_SCRIPT` - the name of the boot script to use when starting\n    the release. This script is used when running commands such as `start` and\n    `daemon`. The boot script is expected to be located at the\n    path `releases/RELEASE_VSN/RELEASE_BOOT_SCRIPT.boot`. Defaults to `start`\n\n  * `RELEASE_BOOT_SCRIPT_CLEAN` - the name of the boot script used when\n    starting the release clean, without your application or its dependencies.\n    This script is used by commands such as `eval`, `rpc`, and `remote`.\n    The boot script is expected to be located at the path\n    `releases/RELEASE_VSN/RELEASE_BOOT_SCRIPT_CLEAN.boot`. Defaults\n    to `start_clean`","ref":"Mix.Tasks.Release.html#module-environment-variables"},{"type":"task","title":"Umbrellas - mix release","doc":"Releases are well integrated with umbrella projects, allowing you to\nrelease one or more subsets of your umbrella children. The only difference\nbetween performing a release in the umbrella project compared to a\nregular application is that umbrellas require you to explicitly list\nyour release and the starting point for each release. For example,\nimagine this umbrella applications:\n\n```text\nmy_app_umbrella/\n  apps/\n    my_app_core/\n    my_app_event_processing/\n    my_app_web/\n```\n\nwhere both `my_app_event_processing` and `my_app_web` depend on\n`my_app_core` but they do not depend on each other.\n\nInside your umbrella, you can define multiple releases:\n\n    releases: [\n      web_and_event_processing: [\n        applications: [\n          my_app_event_processing: :permanent,\n          my_app_web: :permanent\n        ]\n      ],\n\n      web_only: [\n        applications: [my_app_web: :permanent]\n      ],\n\n      event_processing_only: [\n        applications: [my_app_event_processing: :permanent]\n      ]\n    ]\n\nNote you don't need to define all applications in `:applications`,\nonly the entry points. Also remember that the recommended mode\nfor all applications in the system is `:permanent`.\n\nFinally, keep in mind it is not required for you to assemble the\nrelease from the umbrella root. You can also assemble the release\nfrom each child application individually. Doing it from the root,\nhowever, allows you to include two applications that do not depend\non each other as part of the same release.","ref":"Mix.Tasks.Release.html#module-umbrellas"},{"type":"task","title":"Hot Code Upgrades - mix release","doc":"Erlang and Elixir are sometimes known for the capability of upgrading\na node that is running in production without shutting down that node.\nHowever, this feature is not supported out of the box by Elixir releases.\n\nThe reason we don't provide hot code upgrades is because they are very\ncomplicated to perform in practice, as they require careful coding of\nyour processes and applications as well as extensive testing. Given most\nteams can use other techniques that are language agnostic to upgrade\ntheir systems, such as Blue/Green deployments, Canary deployments,\nRolling deployments, and others, hot upgrades are rarely a viable\noption. Let's understand why.\n\nIn a hot code upgrade, you want to update a node from version A to\nversion B. To do so, the first step is to write recipes for every application\nthat changed between those two releases, telling exactly how the application\nchanged between versions, those recipes are called `.appup` files.\nWhile some of the steps in building `.appup` files can be automated,\nnot all of them can. Furthermore, each process in the application needs\nto be explicitly coded with hot code upgrades in mind. Let's see an example.\nImagine your application has a counter process as a GenServer:\n\n    defmodule Counter do\n      use GenServer\n\n      def start_link(_) do\n        GenServer.start_link(__MODULE__, :ok, name: __MODULE__)\n      end\n\n      def bump do\n        GenServer.call(__MODULE__, :bump)\n      end","ref":"Mix.Tasks.Release.html#module-hot-code-upgrades"},{"type":"task","title":"Callbacks - mix release","doc":"def init(:ok) do\n        {:ok, 0}\n      end\n\n      def handle_call(:bump, counter) do\n        {:reply, :ok, counter + 1}\n      end\n    end\n\nYou add this process as part of your supervision tree and ship version\n0.1.0 of your system. Now let's imagine that on version 0.2.0 you added\ntwo changes: instead of `bump/0`, that always increments the counter by\none, you introduce `bump/1` that passes the exact value to bump the\ncounter. You also change the state, because you want to store the maximum\nbump value:\n\n    defmodule Counter do\n      use GenServer\n\n      def start_link(_) do\n        GenServer.start_link(__MODULE__, :ok, name: __MODULE__)\n      end\n\n      def bump(by) do\n        GenServer.call(__MODULE__, {:bump, by})\n      end","ref":"Mix.Tasks.Release.html#module-callbacks"},{"type":"task","title":"Callbacks - mix release","doc":"def init(:ok) do\n        {:ok, {0, 0}}\n      end\n\n      def handle_call({:bump, by}, {counter, max}) do\n        {:reply, :ok, {counter + by, max(max, by)}}\n      end\n    end\n\nIf you were to perform a hot code upgrade in such an application, it would\ncrash, because in the initial version the state was just a counter\nbut in the new version the state is a tuple. Furthermore, you changed\nthe format of the `call` message from `:bump` to `{:bump, by}` and\nthe process may have both old and new messages temporarily mixed, so\nwe need to handle both. The final version would be:\n\n    defmodule Counter do\n      use GenServer\n\n      def start_link(_) do\n        GenServer.start_link(__MODULE__, :ok, name: __MODULE__)\n      end\n\n      def bump(by) do\n        GenServer.call(__MODULE__, {:bump, by})\n      end","ref":"Mix.Tasks.Release.html#module-callbacks"},{"type":"task","title":"Callbacks - mix release","doc":"def init(:ok) do\n        {:ok, {0, 0}}\n      end\n\n      def handle_call(:bump, {counter, max}) do\n        {:reply, :ok, {counter + 1, max(max, 1)}}\n      end\n\n      def handle_call({:bump, by}, {counter, max}) do\n        {:reply, :ok, {counter + by, max(max, by)}}\n      end\n\n      def code_change(_, counter, _) do\n        {:ok, {counter, 0}}\n      end\n    end\n\nNow you can proceed to list this process in the `.appup` file and\nhot code upgrade it. This is one of the many steps necessary\nto perform hot code upgrades and it must be taken into account by\nevery process and application being upgraded in the system.\nThe [`.appup` cookbook](https://www.erlang.org/doc/design_principles/appup_cookbook.html)\nprovides a good reference and more examples.\n\nOnce `.appup`s are created, the next step is to create a `.relup`\nfile with all instructions necessary to update the release itself.\nErlang documentation does provide a chapter on\n[Creating and upgrading a target system](https://www.erlang.org/doc/system_principles/create_target.html).\n[Learn You Some Erlang has a chapter on hot code upgrades](https://learnyousomeerlang.com/relups).\n\nOverall, there are many steps, complexities and assumptions made\nduring hot code upgrades, which is ultimately why they are not\nprovided by Elixir out of the box. However, hot code upgrades can\nstill be achieved by teams who desire to implement those steps\non top of `mix release` in their projects or as separate libraries.","ref":"Mix.Tasks.Release.html#module-callbacks"},{"type":"task","title":"Command line options - mix release","doc":"* `--force` - forces recompilation\n  * `--no-archives-check` - does not check archive\n  * `--no-deps-check` - does not check dependencies\n  * `--no-elixir-version-check` - does not check Elixir version\n  * `--no-compile` - does not compile before assembling the release\n  * `--overwrite` - if there is an existing release version, overwrite it\n  * `--path` - the path of the release\n  * `--quiet` - does not write progress to the standard output\n  * `--version` - the version of the release","ref":"Mix.Tasks.Release.html#module-command-line-options"},{"type":"task","title":"mix release.init","doc":"Generates sample files for releases.\n\n    $ mix release.init\n    * creating rel/vm.args.eex\n    * creating rel/remote.vm.args.eex\n    * creating rel/env.sh.eex\n    * creating rel/env.bat.eex","ref":"Mix.Tasks.Release.Init.html"},{"type":"task","title":"mix run","doc":"Runs the current application.\n\n`mix run` starts the current application dependencies and the\napplication itself. The application will be compiled if it has\nnot been compiled yet or it is outdated.\n\n`mix run` may also run code in the application context through\nadditional options. For example, to run a script within the\ncurrent application, you may pass a filename as argument:\n\n    $ mix run my_app_script.exs arg1 arg2 arg3\n\nCode to be executed can also be passed inline with the `-e` option:\n\n    $ mix run -e \"DbUtils.delete_old_records()\" -- arg1 arg2 arg3\n\nIn both cases, the command-line arguments for the script or expression\nare available in `System.argv/0`. This mirror the command line interface\nin the `elixir` executable.\n\nFor starting long running systems, one typically passes the `--no-halt`\noption:\n\n    $ mix run --no-halt\n\nThe `--no-start` option can also be given and the current application,\nnor its dependencies will be started. Alternatively, you may use\n`mix eval` to evaluate a single expression without starting the current\napplication.\n\nIf you need to pass options to the Elixir executable at the same time\nyou use `mix run`, it can be done as follows:\n\n    $ elixir --sname hello -S mix run --no-halt\n\nThis task is automatically re-enabled, so it can be called multiple times\nwith different arguments.","ref":"Mix.Tasks.Run.html"},{"type":"task","title":"Command-line options - mix run","doc":"* `--eval`, `-e` - evaluates the given code\n  * `--require`, `-r` - executes the given pattern/file\n  * `--parallel`, `-p` - makes all requires parallel\n  * `--preload-modules` - preloads all modules defined in applications\n  * `--no-archives-check` - does not check archives\n  * `--no-compile` - does not compile even if files require compilation\n  * `--no-deps-check` - does not check dependencies\n  * `--no-elixir-version-check` - does not check the Elixir version from mix.exs\n  * `--no-halt` - does not halt the system after running the command\n  * `--no-mix-exs` - allows the command to run even if there is no mix.exs\n  * `--no-start` - does not start applications after compilation","ref":"Mix.Tasks.Run.html#module-command-line-options"},{"type":"task","title":"mix test","doc":"Runs the tests for a project.\n\nThis task starts the current application, loads up\n`test/test_helper.exs` and then, requires all files matching the\n`test/**/*_test.exs` pattern in parallel.\n\nA list of files and/or directories can be given after the task\nname in order to select the files to run:\n\n    $ mix test test/some/particular/file_test.exs\n    $ mix test test/some/particular/dir\n\nTests in umbrella projects can be run from the root by specifying\nthe full suite path, including `apps/my_app/test`, in which case\nrecursive tests for other child apps will be skipped completely:\n\n    # To run all tests for my_app from the umbrella root\n    $ mix test apps/my_app/test\n\n    # To run a given test file on my_app from the umbrella root\n    $ mix test apps/my_app/test/some/particular/file_test.exs","ref":"Mix.Tasks.Test.html"},{"type":"task","title":"Understanding test results - mix test","doc":"When you run your test suite, it prints results as they run with\na summary at the end, as seen below:\n\n    $ mix test\n    ...\n\n      1) test greets the world (FooTest)\n         test/foo_test.exs:5\n         Assertion with == failed\n         code:  assert Foo.hello() == :world!\n         left:  :world\n         right: :world!\n         stacktrace:\n           test/foo_test.exs:6: (test)\n\n    ........\n\n    Finished in 0.05 seconds (0.00s async, 0.05s sync)\n    1 doctest, 11 tests, 1 failure\n\n    Randomized with seed 646219\n\nFor each test, the test suite will print a dot. Failed tests\nare printed immediately in the format described in the next\nsection.\n\nAfter all tests run, we print the suite summary. The first\nline contains the total time spent on the suite, followed\nby how much time was spent on async tests (defined with\n`use ExUnit.Case, async: true`) vs sync ones:\n\n    Finished in 0.05 seconds (0.00s async, 0.05s sync)\n\nDevelopers want to minimize the time spent on sync tests\nwhenever possible, as sync tests run serially and async\ntests run concurrently.\n\nFinally, how many tests we have run, how many of them\nfailed, how many were invalid, and so on.\n\n#","ref":"Mix.Tasks.Test.html#module-understanding-test-results"},{"type":"task","title":"Understanding test failures - mix test","doc":"First, it contains the failure counter, followed by the test\nname and the module the test was defined:\n\n    1) test greets the world (FooTest)\n\nThe next line contains the exact location of the test in the\n`FILE:LINE` format:\n\n    test/foo_test.exs:5\n\nIf you want to re-run only this test, all you need to do is to\ncopy the line above and paste it in front of `mix test`:\n\n    $ mix test test/foo_test.exs:5\n\nThen we show the error message, code snippet, and general information\nabout the failed test:\n\n    Assertion with == failed\n    code:  assert Foo.hello() == :world!\n    left:  :world\n    right: :world!\n\nIf your terminal supports coloring (see the  \"Coloring\" section below),\na diff is typically shown between `left` and `right` sides. Finally,\nwe print the stacktrace of the failure:\n\n    stacktrace:\n      test/foo_test.exs:6: (test)","ref":"Mix.Tasks.Test.html#module-understanding-test-failures"},{"type":"task","title":"Command line options - mix test","doc":"* `--color` - enables color in the output\n\n  * `--cover` - runs coverage tool. See \"Coverage\" section below\n\n  * `--exclude` - excludes tests that match the filter\n\n  * `--exit-status` - use an alternate exit status to use when the test suite\n    fails (default is 2).\n\n  * `--export-coverage` - the name of the file to export coverage results to.\n    Only has an effect when used with `--cover`\n\n  * `--failed` - runs only tests that failed the last time they ran\n\n  * `--force` - forces compilation regardless of modification times\n\n  * `--formatter` - sets the formatter module that will print the results.\n    Defaults to ExUnit's built-in CLI formatter\n\n  * `--include` - includes tests that match the filter\n\n  * `--listen-on-stdin` - runs tests, and then listens on stdin. It will\n    re-run tests once a newline is received. See the \"File system watchers\"\n    section below\n\n  * `--max-cases` - sets the maximum number of tests running asynchronously. Only tests from\n    different modules run in parallel. Defaults to twice the number of cores\n\n  * `--max-failures` - the suite stops evaluating tests when this number of test\n    failures is reached. It runs all tests if omitted\n\n  * `--no-all-warnings` - prints only warnings from files currently compiled (instead of all)\n\n  * `--no-archives-check` - does not check archives\n\n  * `--no-color` - disables color in the output\n\n  * `--no-compile` - does not compile, even if files require compilation\n\n  * `--no-deps-check` - does not check dependencies\n\n  * `--no-elixir-version-check` - does not check the Elixir version from `mix.exs`\n\n  * `--no-start` - does not start applications after compilation\n\n  * `--only` - runs only tests that match the filter\n\n  * `--partitions` - sets the amount of partitions to split tests in. It must be\n    a number greater than zero. If set to one, it acts a no-op. If more than one,\n    then you must also set the `MIX_TEST_PARTITION` environment variable with the\n    partition to use in the current test run. See the \"Operating system process\n    partitioning\" section for more information\n\n  * `--preload-modules` - preloads all modules defined in applications\n\n  * `--profile-require time` - profiles the time spent to require test files.\n    Used only for debugging. The test suite does not run.\n\n  * `--raise` - raises if the test suite failed\n\n  * `--seed` - seeds the random number generator used to randomize the order of tests;\n    `--seed 0` disables randomization so the tests in a single file will always be ran\n    in the same order they were defined in\n\n  * `--slowest` - prints timing information for the N slowest tests.\n    Automatically sets `--trace` and `--preload-modules`\n\n  * `--stale` - runs only tests which reference modules that changed since the\n    last time tests were ran with `--stale`. You can read more about this option\n    in the \"The --stale option\" section below\n\n  * `--timeout` - sets the timeout for the tests\n\n  * `--trace` - runs tests with detailed reporting. Automatically sets `--max-cases` to `1`.\n    Note that in trace mode test timeouts will be ignored as timeout is set to `:infinity`\n\n  * `--warnings-as-errors` - (since v1.12.0) treats warnings as errors and returns a non-zero\n    exit status. This option only applies to test files. To treat warnings as errors during\n    compilation and during tests, run:\n        MIX_ENV=test mix do compile --warnings-as-errors + test --warnings-as-errors","ref":"Mix.Tasks.Test.html#module-command-line-options"},{"type":"task","title":"Configuration - mix test","doc":"These configurations can be set in the `def project` section of your `mix.exs`:\n\n  * `:test_coverage` - a set of options to be passed down to the coverage\n    mechanism. See the \"Coverage\" section for more information\n\n  * `:test_elixirc_options` - the compiler options to used when\n    loading/compiling test files. By default it disables the debug chunk\n    and docs chunk\n\n  * `:test_paths` - list of paths containing test files. Defaults to\n    `[\"test\"]` if the `test` directory exists; otherwise, it defaults to `[]`.\n    It is expected that all test paths contain a `test_helper.exs` file\n\n  * `:test_pattern` - a pattern to load test files. Defaults to `*_test.exs`\n\n  * `:warn_test_pattern` - a pattern to match potentially misnamed test files\n    and display a warning. Defaults to `*_test.ex`","ref":"Mix.Tasks.Test.html#module-configuration"},{"type":"task","title":"Coloring - mix test","doc":"Coloring is enabled by default on most Unix terminals. They are also\navailable on Windows consoles from Windows 10, although it must be\nexplicitly enabled for the current user in the registry by running\nthe following command:\n\n    $ reg add HKCU\\Console /v VirtualTerminalLevel /t REG_DWORD /d 1\n\nAfter running the command above, you must restart your current console.","ref":"Mix.Tasks.Test.html#module-coloring"},{"type":"task","title":"Filters - mix test","doc":"ExUnit provides tags and filtering functionality that allow developers\nto select which tests to run. The most common functionality is to exclude\nsome particular tests from running by default in your test helper file:\n\n    # Exclude all external tests from running\n    ExUnit.configure(exclude: [external: true])\n\nThen, whenever desired, those tests could be included in the run via the\n`--include` option:\n\n    $ mix test --include external:true\n\nThe example above will run all tests that have the external option set to\n`true`. It is also possible to include all examples that have a given tag,\nregardless of its value:\n\n    $ mix test --include external\n\nNote that all tests are included by default, so unless they are excluded\nfirst (either in the test helper or via the `--exclude` option) the\n`--include` option has no effect.\n\nFor this reason, Mix also provides an `--only` option that excludes all\ntests and includes only the given ones:\n\n    $ mix test --only external\n\nWhich is similar to:\n\n    $ mix test --include external --exclude test\n\nIt differs in that the test suite will fail if no tests are executed when the `--only` option is used.\n\nIn case a single file is being tested, it is possible to pass one or more specific\nline numbers to run only those given tests:\n\n    $ mix test test/some/particular/file_test.exs:12\n\nWhich is equivalent to:\n\n    $ mix test --exclude test --include line:12 test/some/particular/file_test.exs\n\nOr:\n\n    $ mix test test/some/particular/file_test.exs:12:24\n\nWhich is equivalent to:\n\n    $ mix test --exclude test --include line:12 --include line:24 test/some/particular/file_test.exs\n\nIf a given line starts a `describe` block, that line filter runs all tests in it.\nOtherwise, it runs the closest test on or before the given line number.","ref":"Mix.Tasks.Test.html#module-filters"},{"type":"task","title":"Coverage - mix test","doc":"Elixir provides built-in line-based test coverage via the `--cover` flag.\nThe test coverages shows which lines of code and in which files were executed\nduring the test run.\n\n#","ref":"Mix.Tasks.Test.html#module-coverage"},{"type":"task","title":"Limitations - mix test","doc":"Coverage in Elixir has the following limitations:\n\n  * Literals, such as atoms, strings, and numbers, are not traced by coverage.\n    For example, if a function simply returns `:ok`, the atom `:ok` itself is\n    never taken into account by coverage;\n\n  * Macros, such as the ones defined by `defmacro/2` and `defguard/2`, and code\n    invoked only by macros are never considered as covered, unless they are also\n    invoked during the tests themselves. That's because macros are invoked at\n    compilation time, before the test coverage instrumentation begins;\n\n#","ref":"Mix.Tasks.Test.html#module-limitations"},{"type":"task","title":"Configuratiuon - mix test","doc":"The `:test_coverage` configures the coverage tool and accepts the following options:\n\n  * `:output` - the output directory for cover results. Defaults to `\"cover\"`.\n\n  * `:tool` - a module specifying the coverage tool to use.\n\n  * `:summary` - at the end of each coverage run, a summary of each\n    module is printed, with results in red or green depending on whether\n    the percentage is below or above a given threshold. The task will\n    exit with status of 1 if the total coverage is below the threshold.\n    The `:summary` option allows you to customize the summary generation\n    and defaults to `[threshold: 90]`, but it may be set to `false` to\n    disable such reports.\n\n  * `:export` - a filename to export results to instead of generating\n    the coverage result on the fly. The `.coverdata` extension is\n    automatically added to the given file. This option is automatically\n    set via the `--export-coverage` option or when using process partitioning.\n    See `mix test.coverage` to compile a report from multiple exports.\n\n  * `:ignore_modules` - modules to ignore from generating reports and\n    in summaries. It is a list of module names as atoms and regular\n    expressions that are matched against the module names.\n\n  * `:local_only` - by default coverage only tracks local calls,\n    set this option to false if you plan to run coverage across nodes.\n\nBy default, a wrapper around OTP's `cover` is used as the default coverage\ntool. You can learn more about how it works in the docs for\n`mix test.coverage`. Your tool of choice can be given as follows:\n\n    def project() do\n      [\n        ...\n        test_coverage: [tool: CoverModule]\n        ...\n      ]\n    end\n\n`CoverModule` can be any module that exports `start/2`, receiving the\ncompilation path and the `test_coverage` options as arguments.\nIt must return either `nil` or an anonymous function of zero arity that\nwill run after the test suite is done.","ref":"Mix.Tasks.Test.html#module-configuratiuon"},{"type":"task","title":"Operating system process partitioning - mix test","doc":"While ExUnit supports the ability to run tests concurrently within the same\nElixir instance, it is not always possible to run all tests concurrently. For\nexample, some tests may rely on global resources.\n\nFor this reason, `mix test` supports partitioning the test files across\ndifferent Elixir instances. This is done by setting the `--partitions` option\nto an integer, with the number of partitions, and setting the `MIX_TEST_PARTITION`\nenvironment variable to control which test partition that particular instance\nis running. This can also be useful if you want to distribute testing across\nmultiple machines.\n\nFor example, to split a test suite into 4 partitions and run them, you would\nuse the following commands:\n\n    $ MIX_TEST_PARTITION=1 mix test --partitions 4\n    $ MIX_TEST_PARTITION=2 mix test --partitions 4\n    $ MIX_TEST_PARTITION=3 mix test --partitions 4\n    $ MIX_TEST_PARTITION=4 mix test --partitions 4\n\nThe test files are sorted upfront in a round-robin fashion. Note the partition\nitself is given as an environment variable so it can be accessed in config files\nand test scripts. For example, it can be used to setup a different database instance\nper partition in `config/test.exs`.\n\nIf partitioning is enabled and `--cover` is used, no cover reports are generated,\nas they only contain a subset of the coverage data. Instead, the coverage data\nis exported to files such as `cover/MIX_TEST_PARTITION.coverdata`. Once you have\nthe results of all partitions inside `cover/`, you can run `mix test.coverage` to\nget the unified report.","ref":"Mix.Tasks.Test.html#module-operating-system-process-partitioning"},{"type":"task","title":"The --stale option - mix test","doc":"The `--stale` command line option attempts to run only the test files which\nreference modules that have changed since the last time you ran this task with\n`--stale`.\n\nThe first time this task is run with `--stale`, all tests are run and a manifest\nis generated. On subsequent runs, a test file is marked \"stale\" if any modules it\nreferences (and any modules those modules reference, recursively) were modified\nsince the last run with `--stale`. A test file is also marked \"stale\" if it has\nbeen changed since the last run with `--stale`.\n\nThe `--stale` option is extremely useful for software iteration, allowing you to\nrun only the relevant tests as you perform changes to the codebase.","ref":"Mix.Tasks.Test.html#module-the-stale-option"},{"type":"task","title":"File-system watchers - mix test","doc":"You can integrate `mix test` with filesystem watchers through the command line\nvia the `--listen-on-stdin` option. For example, you can use [fswatch](https://github.com/emcrisostomo/fswatch)\nor similar to emit newlines whenever there is a change, which will cause your test\nsuite to re-run:\n\n    $ fswatch lib test | mix test --listen-on-stdin\n\nThis can be combined with the `--stale` option to re-run only the test files that\nhave changed as well as the tests that have gone stale due to changes in `lib`.","ref":"Mix.Tasks.Test.html#module-file-system-watchers"},{"type":"task","title":"Aborting the suite - mix test","doc":"It is possible to abort the test suite with `Ctrl+\\ `, which sends a SIGQUIT\nsignal to the Erlang VM. ExUnit will intercept this signal to show all tests\nthat have been aborted and print the results collected so far.\n\nThis can be useful in case the suite gets stuck and you don't want to wait\nuntil the timeout times passes (which defaults to 30 seconds).","ref":"Mix.Tasks.Test.html#module-aborting-the-suite"},{"type":"task","title":"mix test.coverage","doc":"Build reports from exported test coverage.\n\nIn this moduledoc, we will describe how the default test\ncoverage works in Elixir and also explore how it is capable\nof exporting coverage results to group reports from multiple\ntest runs.","ref":"Mix.Tasks.Test.Coverage.html"},{"type":"task","title":"Line coverage - mix test.coverage","doc":"Elixir uses Erlang's [`:cover`](https://www.erlang.org/doc/man/cover.html)\nfor its default test coverage. Erlang coverage is done by tracking\n*executable lines of code*. This implies blank lines, code comments,\nfunction signatures, and patterns are not necessarily executable and\ntherefore won't be tracked in coverage reports. Code in macros are\nalso often executed at compilation time, and therefore may not be covered.\nSimilarly, Elixir AST literals, such as atoms, are not executable either.\n\nLet's see an example:\n\n    if some_condition? do\n      do_this()\n    else\n      do_that()\n    end\n\nIn the example above, if your tests exercise both `some_condition? == true`\nand `some_condition? == false`, all branches will be covered, as they all\nhave executable code. However, the following code\n\n    if some_condition? do\n      do_this()\n    else\n      :default\n    end\n\nwon't ever mark the `:default` branch as covered, as there is no executable\ncode in the `else` branch. Note, however, this issue does not happen on `case`\nor `cond`, as Elixir is able to mark the clause operator `->` as executable in\nsuch corner cases:\n\n    case some_condition? do\n      true ->\n        do_this()\n\n      false ->\n        :default\n    end\n\nIf the code above is tested with both conditions, you should see entries\nin both branches marked as covered.\n\nFinally, it is worth discussing that line coverage by itself has its own\nlimitations. For example, take the following code:\n\n    do_this() || do_that()\n\nLine coverage is not capable of expressing that both `do_this()` and\n`do_that()` have been executed, since as soon as `do_this()` is executed,\nthe whole line is covered. Other techniques, such as branch coverage,\ncan help spot those cases, but they are not currently supported by the\ndefault coverage tool.\n\nOverall, code coverage can be a great tool for finding flaws in our\ncode (such as functions that haven't been covered) but it can also lead\nteams into a false sense of security since 100% coverage never means all\ndifferent executions flows have been asserted, even with the most advanced\ncoverage techniques. It is up to you and your team to specify how much\nemphasis you want to place on it.","ref":"Mix.Tasks.Test.Coverage.html#module-line-coverage"},{"type":"task","title":"Exporting coverage - mix test.coverage","doc":"This task can be used when you need to group the coverage\nacross multiple test runs. Let's see some examples.\n\n#","ref":"Mix.Tasks.Test.Coverage.html#module-exporting-coverage"},{"type":"task","title":"Example: aggregating partitioned runs - mix test.coverage","doc":"If you partition your tests across multiple runs,\nyou can unify the report as shown below:\n\n    $ MIX_TEST_PARTITION=1 mix test --partitions 2 --cover\n    $ MIX_TEST_PARTITION=2 mix test --partitions 2 --cover\n    $ mix test.coverage\n\nThis works because the `--partitions` option\nautomatically exports the coverage results.\n\n#","ref":"Mix.Tasks.Test.Coverage.html#module-example-aggregating-partitioned-runs"},{"type":"task","title":"Example: aggregating coverage reports from all umbrella children - mix test.coverage","doc":"If you run `mix test.coverage` inside an umbrella,\nit will automatically gather exported cover results\nfrom all umbrella children - as long as the coverage\nresults have been exported, like this:\n\n    # from the umbrella root\n    $ mix test --cover --export-coverage default\n    $ mix test.coverage\n\nOf course, if you want to actually partition the tests,\nyou can also do:\n\n    # from the umbrella root\n    $ MIX_TEST_PARTITION=1 mix test --partitions 2 --cover\n    $ MIX_TEST_PARTITION=2 mix test --partitions 2 --cover\n    $ mix test.coverage\n\nOn the other hand, if you want partitioned tests but\nper-app reports, you can do:\n\n    # from the umbrella root\n    $ MIX_TEST_PARTITION=1 mix test --partitions 2 --cover\n    $ MIX_TEST_PARTITION=2 mix test --partitions 2 --cover\n    $ mix cmd mix test.coverage\n\nWhen running `test.coverage` from the umbrella root, it\nwill use the `:test_coverage` configuration from the umbrella\nroot.\n\nFinally, note the coverage itself is not measured across\nthe projects themselves. For example, if project B depends\non A, and if there is code in A that is only executed from\nproject B, those lines will not be marked as covered, which\nis important, as those projects should be developed and tested\nin isolation.\n\n#","ref":"Mix.Tasks.Test.Coverage.html#module-example-aggregating-coverage-reports-from-all-umbrella-children"},{"type":"task","title":"Other scenarios - mix test.coverage","doc":"There may be other scenarios where you want to export coverage.\nFor example, you may have broken your test suite into two, one\nfor unit tests and another for integration tests. In such scenarios,\nyou can explicitly use the `--export-coverage` command line option,\nor the `:export` option under `:test_coverage` in your `mix.exs` file.","ref":"Mix.Tasks.Test.Coverage.html#module-other-scenarios"},{"type":"task","title":"mix xref","doc":"Prints cross reference information between modules.\n\nThe `xref` task expects a mode as first argument:\n\n    $ mix xref MODE\n\nAll available modes are discussed below.\n\nThis task is automatically re-enabled, so you can print\ninformation multiple times in the same Mix invocation.","ref":"Mix.Tasks.Xref.html"},{"type":"task","title":"mix xref callers MODULE - mix xref","doc":"Prints all callers of the given module. Example:\n\n    $ mix xref callers MyMod","ref":"Mix.Tasks.Xref.html#module-mix-xref-callers-module"},{"type":"task","title":"mix xref trace FILE - mix xref","doc":"Compiles the given file listing all dependencies within the same app.\nIt includes the type and line for each one. Example:\n\n    $ mix xref trace lib/my_app/router.ex\n\nThe `--label` option may be given to keep only certain traces\n(compile, runtime or export):\n\n    $ mix xref trace lib/my_app/router.ex --label compile\n\nIf you have an umbrella application, we also recommend using the\n`--include-siblings` flag to see the dependencies from sibling\napplications. The `trace` command is not currently supported at the\numbrella root.\n\n#","ref":"Mix.Tasks.Xref.html#module-mix-xref-trace-file"},{"type":"task","title":"Example - mix xref","doc":"Imagine the given file lib/b.ex:\n\n    defmodule B do\n      import A\n      A.macro()\n      macro()\n      A.fun()\n      fun()\n      def calls_macro, do: A.macro()\n      def calls_fun, do: A.fun()\n      def calls_struct, do: %A{}\n    end\n\n`mix xref trace` will print:\n\n    lib/b.ex:2: require A (export)\n    lib/b.ex:3: call A.macro/0 (compile)\n    lib/b.ex:4: import A.macro/0 (compile)\n    lib/b.ex:5: call A.fun/0 (compile)\n    lib/b.ex:6: call A.fun/0 (compile)\n    lib/b.ex:6: import A.fun/0 (compile)\n    lib/b.ex:7: call A.macro/0 (compile)\n    lib/b.ex:8: call A.fun/0 (runtime)\n    lib/b.ex:9: struct A (export)","ref":"Mix.Tasks.Xref.html#module-example"},{"type":"task","title":"mix xref graph - mix xref","doc":"Prints a file dependency graph where an edge from `A` to `B` indicates\nthat `A` (source) depends on `B` (sink).\n\n    $ mix xref graph --format stats\n\nThe following options are accepted:\n\n  * `--exclude` - path to exclude. Can be repeated to exclude multiple paths.\n\n  * `--label` - only shows relationships with the given label.\n    The labels are \"compile\", \"export\" and \"runtime\". By default,\n    the `--label` option simply filters the printed graph to show\n    only relationships with the given label. You can pass `--only-direct`\n    to trim the graph to only the nodes that have the direct\n    relationship given by label. There is also a special label\n    called \"compile-connected\" that keeps only compile-time files\n    with at least one transitive dependency. See \"Dependency types\"\n    section below.\n\n  * `--group` - provide comma-separated paths to consider as a group. Dependencies\n    from and into multiple files of the group are considered a single dependency.\n    Dependencies between the group elements are ignored. This is useful when you\n    are computing compile and compile-connected dependencies and you want a\n    series of files to be treated as one. The group is printed using the first path,\n    with a `+` suffix. Can be repeated to create multiple groups.\n\n  * `--only-direct` - keeps only files with the direct relationship\n    given by `--label`\n\n  * `--only-nodes` - only shows the node names (no edges).\n    Generally useful with the `--sink` flag\n\n  * `--source` - displays all files that the given source file\n    references (directly or indirectly). Can be repeated to display\n    references from multiple sources.\n\n  * `--sink` - displays all files that reference the given file\n    (directly or indirectly). Can be repeated.\n\n  * `--min-cycle-size` - controls the minimum cycle size on formats\n    like `stats` and `cycles`\n\n  * `--format` - can be set to one of:\n\n    * `pretty` - prints the graph to the terminal using Unicode characters.\n      Each prints each file followed by the files it depends on. This is the\n      default except on Windows;\n\n    * `plain` - the same as pretty except ASCII characters are used instead of\n      Unicode characters. This is the default on Windows;\n\n    * `stats` - prints general statistics about the graph;\n\n    * `cycles` - prints all cycles in the graph;\n\n    * `dot` - produces a DOT graph description in `xref_graph.dot` in the\n      current directory. Warning: this will override any previously generated file\n\n  * `--output` (since v1.15.0) - can be set to one of\n\n    * `-` - prints the output to standard output;\n\n    * a path - writes the output graph to the given path\n\n    Defaults to `xref_graph.dot` in the current directory.\n\nThe `--source` and `--sink` options are particularly useful when trying to understand\nhow the modules in a particular file interact with the whole system. You can combine\nthose options with `--label` and `--only-nodes` to get all files that exhibit a certain\nproperty, for example:\n\n    # To show all compile-time relationships\n    $ mix xref graph --label compile\n\n    # To get the tree that depend on lib/foo.ex at compile time\n    $ mix xref graph --label compile --sink lib/foo.ex\n\n    # To get all files that depend on lib/foo.ex at compile time\n    $ mix xref graph --label compile --sink lib/foo.ex --only-nodes\n\n    # To get all paths between two files\n    $ mix xref graph --source lib/foo.ex --sink lib/bar.ex\n\n    # To show general statistics about the graph\n    $ mix xref graph --format stats\n\n#","ref":"Mix.Tasks.Xref.html#module-mix-xref-graph"},{"type":"task","title":"Understanding the printed graph - mix xref","doc":"When `mix xref graph` runs, it will print a tree of the following\nformat. Imagine the following code:\n\n    # lib/a.ex\n    defmodule A do\n      IO.puts B.hello()\n    end\n\n    # lib/b.ex\n    defmodule B do\n      def hello, do: C.world()\n    end\n\n    # lib/c.ex\n    defmodule C do\n      def world, do: \"hello world\"\n    end\n\nIt will print:\n\n    $ mix xref graph\n    lib/a.ex\n    └── lib/b.ex (compile)\n    lib/b.ex\n    └── lib/c.ex\n    lib/c.ex\n\nThis tree means that `lib/a.ex` depends on `lib/b.ex` at compile\ntime. And `lib/b.ex` depends on `lib/c.ex` at runtime. This is often\nproblematic because if `lib/c.ex` changes, `lib/a.ex` also has to\nrecompile due to this indirect compile time dependency. When you pass\n`--label compile`, the graph shows only the compile-time dependencies:\n\n    $ mix xref graph --label compile\n    lib/a.ex\n    └── lib/b.ex (compile)\n\nThe `--label compile` flag removes all non-compile dependencies. However,\nthis can be misleading because having direct compile time dependencies is\nnot necessarily an issue. The biggest concern, as mentioned above, are the\ntransitive compile time dependencies. You can get all compile time\ndependencies that cause transitive compile time dependencies by using\n`--label compile-connected`:\n\n    $ mix xref graph --label compile-connected\n    lib/a.ex\n    └── lib/b.ex (compile)\n\nThe above says `lib/a.ex` depends on `lib/b.ex` and that causes transitive\ncompile time dependencies - as we know, `lib/a.ex` also depends on `lib/c.ex`.\nWe can retrieve those transitive dependencies by passing `lib/b.ex` as\n`--source` to `mix xref graph`:\n\n    $ mix xref graph --source lib/b.ex\n    lib/b.ex\n    └── lib/c.ex\n\nSimilarly, you can use the `--label compile` and the `--sink` flag to find\nall compile time dependencies that will recompile once the sink changes:\n\n    $ mix xref graph --label compile --sink lib/c.ex\n    lib/a.ex\n    └── lib/b.ex (compile)\n\nIf you have an umbrella application, we also recommend using the\n`--include-siblings` flag to see the dependencies from sibling\napplications. When invoked at the umbrella root, the `graph`\ncommand will list all files from all umbrella children, without\nany namespacing.\n\n#","ref":"Mix.Tasks.Xref.html#module-understanding-the-printed-graph"},{"type":"task","title":"Dependency types - mix xref","doc":"Elixir tracks three types of dependencies between modules: compile,\nexports, and runtime. If a module has a compile time dependency on\nanother module, the caller module has to be recompiled whenever the\ncallee changes. Compile-time dependencies are typically added when\nusing macros or when invoking functions in the module body (outside\nof functions). You can list all dependencies in a file by running\n`mix xref trace path/to/file.ex`.\n\nExport dependencies are compile time dependencies on the module API,\nnamely structs and its public definitions. For example, if you import\na module but only use its functions, it is an export dependency. If\nyou use a struct, it is an export dependency too. Export dependencies\nare only recompiled if the module API changes. Note, however, that compile\ntime dependencies have higher precedence than exports. Therefore if\nyou import a module and use its macros, it is a compile time dependency.\n\nRuntime dependencies are added whenever you invoke another module\ninside a function. Modules with runtime dependencies do not have\nto be compiled when the callee changes, unless there is a transitive\ncompile or an outdated export time dependency between them. The option\n`--label compile-connected` can be used to find the first case.","ref":"Mix.Tasks.Xref.html#module-dependency-types"},{"type":"task","title":"Shared options - mix xref","doc":"Those options are shared across all modes:\n\n  * `--fail-above` - generates a failure if the relevant metric is above the\n    given threshold. Applies to all modes except `mix xref graph --format stats`.\n\n  * `--include-siblings` - includes dependencies that have `:in_umbrella` set\n    to true in the current project in the reports. This can be used to find\n    callers or to analyze graphs between projects (it applies only to `trace`\n    subcommand)\n\n  * `--no-compile` - does not compile even if files require compilation\n\n  * `--no-deps-check` - does not check dependencies\n\n  * `--no-archives-check` - does not check archives\n\n  * `--no-elixir-version-check` - does not check the Elixir version from mix.exs","ref":"Mix.Tasks.Xref.html#module-shared-options"},{"type":"function","title":"Mix.Tasks.Xref.calls/1","doc":"Returns a list of information of all the runtime function calls in the project.\n\nEach item in the list is a map with the following keys:\n\n  * `:callee` - a tuple containing the module, function, and arity of the call\n  * `:line` - an integer representing the line where the function is called\n  * `:file` - a binary representing the file where the function is called\n  * `:caller_module` - the module where the function is called\n\nThis function returns an empty list when used at the root of an umbrella\nproject because there is no compile manifest to extract the function call\ninformation from. To get the function calls of each child in an umbrella,\nexecute the function at the root of each individual application.","ref":"Mix.Tasks.Xref.html#calls/1"},{"type":"exception","title":"Mix.ElixirVersionError","doc":"","ref":"Mix.ElixirVersionError.html"},{"type":"exception","title":"Mix.Error","doc":"","ref":"Mix.Error.html"},{"type":"exception","title":"Mix.InvalidTaskError","doc":"","ref":"Mix.InvalidTaskError.html"},{"type":"exception","title":"Mix.NoProjectError","doc":"","ref":"Mix.NoProjectError.html"},{"type":"exception","title":"Mix.NoTaskError","doc":"","ref":"Mix.NoTaskError.html"}],"content_type":"text/markdown"}