/*
 * BRLTTY - Access software for Unix for a blind person
 *          using a soft Braille terminal
 *
 * Copyright (C) 1995-2000 by The BRLTTY Team, All rights reserved.
 *
 * Nicolas Pitre <nico@cam.org>
 * Stphane Doyon <s.doyon@videotron.ca>
 * Nikhil Nair <nn201@cus.cam.ac.uk>
 *
 * BRLTTY comes with ABSOLUTELY NO WARRANTY.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 */

/* CombiBraille/brl.c - Braille display library
 * For Tieman B.V.'s CombiBraille (serial interface only)
 * Maintained by Nikhil Nair <nn201@cus.cam.ac.uk>
 * $Id: brl.c,v 1.3 1996/09/24 01:04:29 nn201 Exp $
 */

#define __EXTENSIONS__          /* for termios.h */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <string.h>

#include "../brl.h"
#include "../brl_driver.h"
#include "../misc.h"
#include "brlconf.h"
#include "tables.h"             /* for keybindings */
#include "../sbllog.h"
unsigned char highbits[4] = { 128, 64, 32, 16 };

static unsigned int brl_dbg = 0;
unsigned char lowbits[4] = { 8, 4, 2, 1 };

unsigned char combitrans[256];  /* dot mapping table (output) */

int brl_fd;                     /* file descriptor for Braille display */

unsigned char *prevdata;        /* previously received data */

unsigned char status[5], oldstatus[5];  /* status cells - always five */

unsigned char *rawdata;         /* writebrl() buffer for raw Braille data */

short rawlen;                   /* length of rawdata buffer */

struct termios oldtio;          /* old terminal settings */

/* Function prototypes: */
int getbrlkey (void);           /* get a keystroke from the CombiBraille */

struct brlinfo identbrl (const char *name, const char *brldev)
{
  int devcnt;

  struct brlinfo retinfo;

  devcnt = sizeof (tiemandevs) / sizeof (struct brlinfo);
  retinfo.cols = 0;

  for (devnr = 0; devnr < devcnt; devnr++)
    if (!strcmp (tiemandevs[devnr].name, name))
     {
       printf (" %s on %s\n", tiemandevs[devnr].fullname, brldev);
       retinfo = tiemandevs[devnr];
       break;
     }

  return retinfo;

}

void brl_debug (unsigned int dbg)
{
  brl_dbg = dbg;
}

void initbrl (brldim * brl, const char *brldev)
{
  brldim res;                   /* return result */

  struct termios newtio;        /* new terminal settings */

  short i, n, success;          /* loop counters, flags, etc. */

  char *init_seq = INIT_SEQ;    /* bytewise accessible copies */

  char *init_ack = INIT_ACK;

  unsigned char c;

  char id = -1;
  unsigned char standard[8] = { 0, 1, 2, 3, 4, 5, 6, 7 };       /* BRLTTY
                                                                   standard
                                                                   mapping */
  unsigned char Tieman[8] = { 0, 7, 1, 6, 2, 5, 3, 4 }; /* Tieman standard */

  sbl_log ("init brl");
  brl->usb = res.usb = 0;
  res.disp = prevdata = rawdata = NULL; /* clear pointers */
  brl->x = res.x = BRLCOLS;     /* initialise size of display */
  brl->y = res.y = BRLROWS;

  /* No need to load translation tables, as these are now defined in tables.h 
   */

  /* Now open the Braille display device for random access */
  if (brldev != NULL)
    brl_fd = open (brldev, O_RDWR | O_NOCTTY);
  else
    brl_fd = open (BRLDEV, O_RDWR | O_NOCTTY);
  if (brl_fd < 0)
    goto failure;

  tcgetattr (brl_fd, &oldtio);  /* save current settings */

  /* Set bps, flow control and 8n1, enable reading */
  newtio.c_cflag = BAUDRATE | CRTSCTS | CS8 | CLOCAL | CREAD;

  /* Ignore bytes with parity errors and make terminal raw and dumb */
  newtio.c_iflag = IGNPAR;
  newtio.c_oflag = 0;           /* raw output */
  newtio.c_lflag = 0;           /* don't echo or generate signals */
  newtio.c_cc[VMIN] = 0;        /* set nonblocking read */
  newtio.c_cc[VTIME] = 0;
  tcflush (brl_fd, TCIFLUSH);   /* clean line */
  tcsetattr (brl_fd, TCSANOW, &newtio); /* activate new settings */

  /* CombiBraille initialisation procedure: */
  success = 0;
  /* Try MAX_ATTEMPTS times, or forever if MAX_ATTEMPTS is 0: */
#if MAX_ATTEMPTS == 0
  while (!success)
#else
  for (i = 0; i < MAX_ATTEMPTS && !success; i++)
#endif
   {
     if (init_seq[0])
       if (write (brl_fd, init_seq + 1, init_seq[0]) != init_seq[0])
         continue;
     timeout_yet (0);           /* initialise timeout testing */
     n = 0;
     do
      {
        usleep (20000);
        if (read (brl_fd, &c, 1) == 0)
          continue;
        if (n < init_ack[0] && c != init_ack[1 + n])
          continue;
        if (n == init_ack[0])
          id = c, success = 1;
        n++;
      }
     while (!timeout_yet (ACK_TIMEOUT) && n <= init_ack[0]);
   }
  if (!success)
   {
     tcsetattr (brl_fd, TCSANOW, &oldtio);
     goto failure;
   }

  if (res.x == -1)
    return;
  else
    brl->x = res.x;

  /* Allocate space for buffers */
  res.disp = (unsigned char *) malloc (res.x * res.y);
  prevdata = (unsigned char *) malloc (res.x * res.y);
  /* rawdata has to have room for the pre- and post-data sequences, the
     status cells, and escaped 0x1b's: */
  rawdata = (unsigned char *) malloc (20 + res.x * res.y * 2);
  if (!res.disp || !prevdata || !rawdata)
    goto failure;

  /* Generate dot mapping table: */
  memset (combitrans, 0, 256);
  for (n = 0; n < 256; n++)
    for (i = 0; i < 8; i++)
      if (n & 1 << standard[i])
        combitrans[n] |= 1 << Tieman[i];
  res.brl_fd = brl_fd;
  *brl = res;
  return;

failure:;
  if (res.disp)
    free (res.disp);
  if (prevdata)
    free (prevdata);
  if (rawdata)
    free (rawdata);
  if (brl_fd >= 0)
    close (brl_fd);
  return;
}

void closebrl (brldim * brl)
{
  char *pre_data = PRE_DATA;    /* bytewise accessible copies */

  char *post_data = POST_DATA;

  char *close_seq = CLOSE_SEQ;

  rawlen = 0;
  if (pre_data[0])
   {
     memcpy (rawdata + rawlen, pre_data + 1, pre_data[0]);
     rawlen += pre_data[0];
   }
  /* Clear the five status cells and the main display: */
  memset (rawdata + rawlen, 0, 5 + brl->x * brl->y);
  rawlen += 5 + brl->x * brl->y;
  if (post_data[0])
   {
     memcpy (rawdata + rawlen, post_data + 1, post_data[0]);
     rawlen += post_data[0];
   }

  /* Send closing sequence: */
  if (close_seq[0])
   {
     memcpy (rawdata + rawlen, close_seq + 1, close_seq[0]);
     rawlen += close_seq[0];
   }
  write (brl_fd, rawdata, rawlen);
  if (brl->disp)
   {
     free (brl->disp);
     brl->disp = NULL;
   }
  if (prevdata)
   {

     free (prevdata);
     prevdata = NULL;
   }
  if (rawdata)
   {
     free (rawdata);
     rawdata = NULL;
   }
#if 0
  tcsetattr (brl_fd, TCSANOW, &oldtio); /* restore terminal settings */
#endif
  close (brl_fd);

  brl->brl_fd = -1;
  brl_fd = -1;
}

void setbrlstat (const unsigned char *st)
{
  const char lowbits[10] = { 8 | 16 | 32, 4, 4 | 8, 4 | 32, 4 | 32 | 16,
    4 | 16, 4 | 8 | 32, 4 | 8 | 16 | 32, 4 | 8 | 16, 8 | 32
  };
  const char highbits[10] = { 2 | 64 | 128, 1, 1 | 2, 1 | 128, 1 | 128 | 64,
    1 | 64, 1 | 2 | 128, 1 | 2 | 128 | 64, 1 | 2 | 64, 2 | 128
  };
  status[0] = highbits[st[0] / 10] | lowbits[st[1] / 10];
  status[1] = highbits[st[0] % 10] | lowbits[st[1] % 10];
  status[2] = highbits[st[2] / 10];
  status[3] = highbits[st[2] % 10];

}

void writebrl (brldim * brl)
{
  short i;                      /* loop counter */

  char *pre_data = PRE_DATA;    /* bytewise accessible copies */

  char *post_data = POST_DATA;

  /* Only refresh display if the data has changed: */
  if (memcmp (brl->disp, prevdata, brl->x * brl->y)
      || memcmp (status, oldstatus, 5))
   {
     /* Save new Braille data: */
     memcpy (prevdata, brl->disp, brl->x * brl->y);

     /* Dot mapping from standard to CombiBraille: */
     for (i = 0; i < brl->x * brl->y;
          brl->disp[i] = combitrans[brl->disp[i]], i++);

     rawlen = 0;
     if (pre_data[0])
      {
        memcpy (rawdata + rawlen, pre_data + 1, pre_data[0]);
        rawlen += pre_data[0];
      }
     for (i = 0; i < 5; i++)
      {
        rawdata[rawlen++] = status[i];
        if (status[i] == 0x1b)  /* CombiBraille hack */
          rawdata[rawlen++] = 0x1b;
      }
     for (i = 0; i < brl->x * brl->y; i++)
      {
        rawdata[rawlen++] = brl->disp[i];
        if (brl->disp[i] == 0x1b)       /* CombiBraille hack */
          rawdata[rawlen++] = brl->disp[i];
      }
     if (post_data[0])
      {
        memcpy (rawdata + rawlen, post_data + 1, post_data[0]);
        rawlen += post_data[0];
      }
     write (brl_fd, rawdata, rawlen);
   }
}

int readbrl (int *type)
{
  int c;

  static int key = EOF;

  c = getbrlkey ();
  if (c == key)
   {
     *type = 0;
     return EOF;
   }

  if (c == EOF && key == EOF)
   {
     *type = 0;
     return EOF;
   }

  *type = 1;

  return c;
}

int getbrlkey (void)
{
  static short ptr = 0;         /* input queue pointer */

  unsigned char q[4];           /* input queue */

  unsigned char c;              /* character buffer */

  while (read (brl_fd, &c, 1))
   {
     if (ptr == 0 && c != 27)
      {
        continue;
      }
     if (ptr == 1 && c != 'K' && c != 'C')
      {
        ptr = 0;
        continue;
      }
     q[ptr++] = c;
     if (ptr < 3 || (ptr == 3 && q[1] == 'K' && !q[2]))
       continue;
     ptr = 0;
     if (q[1] == 'K')
       return (q[2] ? q[2] + 200 : q[3]);
     return ((int) q[2] + 100);
   }
  return EOF;
}
