/*
 *	Command pipe utilities
 *
 *	Copyright (C) 2003 SuSE Linux AG Nuernberg, Germany.
 *	Copyright (C) 2011 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *	MA  02110-1301  USA
 *
 *	Authors: Marius Tomaschewski <mt@suse.de>
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	You can find the wicked project at http://gitorious.org/wicked/.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <assert.h>
#include <unistd.h>

#include <cmd_pipe.h>
#include <logging.h>

int
nc_cmd_pipe_exec(nc_cmd_pipe_t *cmd)
{
	int rc, fd[2];

	assert(cmd);
	if(!cmd)
		return -1;

	cmd->pid  = -1;
	cmd->file = NULL;

	if(!cmd->exec)
		return -1;

	rc = socketpair(AF_UNIX, SOCK_STREAM, PF_LOCAL, fd);
	if(rc == -1) {
		nc_error("Unable to open a socketpair: %m");
		return -1;
	}

	rc = 2;
	do {
		cmd->pid = fork();
	} while(cmd->pid == -1 && EAGAIN == errno && 0 < --rc);

	switch(cmd->pid) {
		case -1:	/* error */
			nc_error("Unable to fork: %m");
			close(fd[0]);
			close(fd[1]);
		break;

		case 0:		/* child */
			close(fd[0]);

			rc = dup2(fd[1], fileno(stdin));
			if(rc == -1 || rc != fileno(stdin)) {
				close(fd[1]);
				exit(-1);
			}
			rc = dup2(fd[1], fileno(stdout));
			if(rc == -1 || rc != fileno(stdout)) {
				close(fd[1]);
				exit(-1);
			}
			rc = dup2(fd[1], fileno(stderr));
			if(rc == -1 || rc != fileno(stderr)) {
				close(fd[1]);
				exit(-1);
			}
			close(fd[1]);

			rc = cmd->exec(cmd->args);
			fclose(stdin);
			fclose(stdout);
			fclose(stderr);
			exit(rc);
		break;

		default:
			close(fd[1]);
			cmd->file = fdopen(fd[0], "r+e");
			return 0;
		break;
	}
	return -1;
}

pid_t
nc_cmd_pipe_wait(nc_cmd_pipe_t *cmd, int *status, int flags)
{
	pid_t pid;

	assert(status && cmd);
	do {
		pid = waitpid(cmd->pid, status, flags);
	} while(pid == -1 && errno == EINTR);

	return pid;
}

