/*
 *	Routines for loading and storing sysconfig files
 *
 *	Copyright (C) 2009-2010  Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *	MA  02110-1301  USA
 *
 *	Authors: Olaf Kirch <okir@suse.de>
 *	         Marius Tomaschewski <mt@suse.de>
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	You can find the wicked project at http://gitorious.org/wicked/.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>
#include <assert.h>

#include <futils.h>
#include <logging.h>
#include <sysconfig.h>

#define NC_SYSCONFIG_ARRAY_CHUNK	16
#define NC_SYSCONFIG_LINE_MAX		PATH_MAX


nc_sysconfig_t *
nc_sysconfig_new(const char *pathname)
{
	nc_sysconfig_t *nsc;

	nsc = calloc(1, sizeof(nc_sysconfig_t));
	assert(nsc != NULL);
	if(nsc != NULL) {
		if( nc_string_dup(&nsc->pathname, pathname) == 0)
			return nsc;
		free(nsc);
	}
	return NULL;
}

int
nc_sysconfig_init(nc_sysconfig_t *nsc, const char *pathname)
{
	memset(nsc, 0, sizeof(*nsc));
	if( nc_string_dup(&nsc->pathname, pathname) == 0)
		return 0;
	return -1;
}

void
nc_sysconfig_destroy(nc_sysconfig_t *nsc)
{
	nc_var_array_destroy(&nsc->vars);
	nc_string_free(&nsc->pathname);
	memset(nsc, 0, sizeof(*nsc));
}

void
nc_sysconfig_free(nc_sysconfig_t *nsc)
{
	nc_sysconfig_destroy(nsc);
	free(nsc);
}

static int
__unquote(char *string)
{
	char *src, *dst, qc = 0, lc = 0;
	unsigned char cc;
	int ret = 1;

	src = dst = string;
	if (*string == '"' || *string == '\'') {
		qc = *string;
		src++;
	}
	do {
		cc = *src;
		if (!cc) {
			ret = qc && lc == qc;
			break;
		}
		if (isspace(cc) && !qc)
			break;
		if (cc == qc)
			break;
		*dst = lc = cc;
		dst++;
		src++;
	} while (1);

	*dst = '\0';
	return ret;
}

static int
__quote(nc_stringbuf_t *nsb, const char *string)
{
	const char *p;

	if (!nsb || !string)
		return -1;

	/* FIXME: For now, just remove ticks and cntrls. We don't want them */
	for(p=string; *p; p++) {
		switch(*p) {
			case '\"':
			case '\'':
			case '\n':
				nc_stringbuf_putc(nsb, ' ');
			break;
			default:
				if(!iscntrl((unsigned char)*p))
					nc_stringbuf_putc(nsb, *p);
			break;
		}
	}
	return 0;
}

static int
__nc_sysconfig_parse(nc_sysconfig_t *nsc, const char *filename, const char **varnames)
{
	char linebuf[NC_SYSCONFIG_LINE_MAX];
	FILE *fp;

	if (!(fp = fopen(filename, "re")))
		goto error;

	if (!nsc)
		goto error;

	while (fgets(linebuf, sizeof(linebuf), fp) != NULL) {
		char *name, *value;
		char *sp = linebuf;

		while (isspace(*sp))
			++sp;
		if (*sp == '#')
			continue;

		/* Silently ignore fishy strings which do not seem to be
		   variables . */
		if (!isalpha(*sp))
			continue;
		name = sp;

		while (isalnum(*sp) || *sp == '_')
			++sp;
		if (*sp != '=')
			continue;
		*sp++ = '\0';

		/* If we were given a list of variable names to match
		 * against, ignore all variables not in this list. */
		if (varnames) {
			const char **vp = varnames, *match;

			while ((match = *vp++) != NULL) {
				if (!strcmp(match, name))
					 break;
			}
			if (!match)
				continue;
		}

		value = sp;
		if (!__unquote(value))
			continue;

		if (nc_sysconfig_set(nsc, name, value) < 0)
			goto error;
	}

	fclose(fp);
	return 0;
error:
	if(fp)
		fclose(fp);
	return -1;
}

nc_sysconfig_t *
nc_sysconfig_read(const char *filename)
{
	nc_sysconfig_t *sc;

	sc = nc_sysconfig_new(filename);
	if(sc) {
		if(__nc_sysconfig_parse(sc, sc->pathname, NULL) == 0)
			return sc;
		nc_sysconfig_free(sc);
	}
	return NULL;
}

static int
__nc_sysconfig_write_quoted(FILE *fpo, const nc_var_t *var)
{
	if (var->value != NULL) {
		nc_stringbuf_t buf = NC_STRINGBUF_INIT;

		if(__quote(&buf, var->value) < 0)
			return -1;

		fprintf(fpo, "%s='%s'\n", var->name,
			(buf.string ? buf.string : ""));

		nc_stringbuf_destroy(&buf);
	}
	/* NULL value -> deleted */
	return 0;
}

static int
__nc_sysconfig_dump(nc_sysconfig_t *sc, FILE *fpo)
{
	unsigned int i;

	for (i = 0; i < sc->vars.count; ++i) {
		nc_var_t *var = &sc->vars.data[i];

		__nc_sysconfig_write_quoted(fpo, var);
	}
	return 0;
}

static int
__nc_sysconfig_rewrite(FILE *fpi, FILE *fpo, const nc_sysconfig_t *sc)
{
	nc_string_array_t written = NC_STRING_ARRAY_INIT;
	char linebuf[NC_SYSCONFIG_LINE_MAX];
	unsigned int i;

	while (fgets(linebuf, sizeof(linebuf), fpi) != NULL) {
		char *sp = linebuf;
		char *name;
		nc_var_t *var;

		while (isspace(*sp))
			++sp;

		if (*sp == '#') {
	just_copy:
			fputs(linebuf, fpo);
			continue;
		}
		
		/* Silently ignore fishy strings which do not seem to be
		   variables . */
		if (!isalpha(*sp))
			goto just_copy;
		name = sp;

		while (isalnum(*sp) || *sp == '_')
			++sp;
		if (*sp != '=')
			goto just_copy;
		*sp++ = '\0';

		/* Skip variables we've written before. */
		if (nc_string_array_index(&written, name) >= 0)
			continue;

		var = nc_sysconfig_get(sc, name);
		if (var == NULL)
			continue;

		/* Write out the variable */
		__nc_sysconfig_write_quoted(fpo, var);

		/* Remember that we've written this one before */
		nc_string_array_append(&written, var->name);
	}

	/* Write out remaining variables */
	for (i = 0; i < sc->vars.count; ++i) {
		nc_var_t *var = &sc->vars.data[i];

		if (nc_string_array_index(&written, var->name) < 0)
			__nc_sysconfig_write_quoted(fpo, var);
	}

	nc_string_array_destroy(&written);
	return 0;
}

int
nc_sysconfig_overwrite(nc_sysconfig_t *sc, const char *dot_backup)
{
	nc_stringbuf_t old_path = NC_STRINGBUF_INIT;
	FILE *ofp;

	if(!(sc && sc->pathname && *sc->pathname))
		return -1;

	if(dot_backup &&
	   (nc_make_dot_suffix(&old_path, sc->pathname, dot_backup) < 0)) {
		nc_error("Unable to apply a backup suffix to filename: %s",
				sc->pathname);
		goto error;
	}

	if(old_path.string && nc_file_exists(sc->pathname)) {
		unlink(old_path.string);
		if(link(sc->pathname, old_path.string) != 0) {
			nc_error("Unable to create a backup of %s: %m",
					sc->pathname);
			goto error;
		}
	}

	ofp = fopen(sc->pathname, "we");
	if (ofp == NULL) {
		nc_error("Unable to open %s for writing: %m", sc->pathname);
		goto error;
	}

	if(__nc_sysconfig_dump(sc, ofp) < 0) {
		fclose(ofp);
		goto error;
	}

	nc_stringbuf_destroy(&old_path);
	fclose(ofp);
	return 0;

error:
	nc_stringbuf_destroy(&old_path);
	return -1;
}

/*
 * Rewrite a sysconfig file, replacing all variables with those found in
 * the changed sysconfig object.
 * Rewrite is a bit nicer than overwrite, in that it tries to preserve
 * all comments.
 */
int
nc_sysconfig_rewrite(nc_sysconfig_t *sc, const char *dot_backup)
{
	const char *   tmp_suffix = ".XXXXXX";
	nc_stringbuf_t tmp_path = NC_STRINGBUF_INIT;
	nc_stringbuf_t old_path = NC_STRINGBUF_INIT;
	FILE *ifp = NULL, *ofp = NULL;
	int  fd;

	/*
	** Hmm.. I don't think advisory locking makes sense here...
	** For netcontrol purposes, a dot-lock file, that write-locks
	** ifcfg-, ifroute-, routes, ...
	*/

	assert(sc && sc->pathname && *sc->pathname);

	if(nc_make_dot_suffix(&tmp_path, sc->pathname, tmp_suffix) < 0) {
		nc_error("Unable to apply a temp suffix to filename: %s",
				sc->pathname);
		goto error;
	}

	if(dot_backup && (nc_make_dot_suffix(&old_path,
				sc->pathname, dot_backup) < 0)) {
		nc_error("Unable to apply a backup suffix to filename: %s",
				sc->pathname);
		goto error;
	}

	fd = mkstemp(tmp_path.string);
	if(fd == -1) {
		nc_error("Unable to create temp file %s for writing: %m",
			tmp_path.string);
		goto error;
	}
	ofp = fdopen(fd, "we");
	if (ofp == NULL) {
		close(fd);
		nc_error("Unable to open %s for writing: %m", tmp_path.string);
		goto error;
	}

	ifp = fopen(sc->pathname, "re");
	if (ifp == NULL) {
		if (errno != ENOENT) {
			nc_error("Unable to open file %s: %m", sc->pathname);
			goto error;
		}
		if(__nc_sysconfig_dump(sc, ofp) < 0)
			goto error;
	} else {
		if(__nc_sysconfig_rewrite(ifp, ofp, sc) < 0)
			goto error;
	}
	fflush(ofp);

	if(ifp && old_path.string) {
		unlink(old_path.string);
		if(link(sc->pathname, old_path.string) != 0) {
			nc_warn("Unable to create a backup of %s: %m", sc->pathname);
			nc_stringbuf_clear(&old_path);
		}
	}
	if (rename(tmp_path.string, sc->pathname) < 0) {
		nc_error("Unable to rename %s to %s: %m", tmp_path.string, sc->pathname);
		goto error;
	}

	if(ifp) {
		fclose(ifp);
	}
	fclose(ofp);
	nc_stringbuf_destroy(&tmp_path);
	nc_stringbuf_destroy(&old_path);
	return 0;

error:
	if(ifp) {
		if(old_path.string)
			unlink(old_path.string);
		fclose(ifp);
	}
	if(ofp) {
		if(tmp_path.string)
			unlink(tmp_path.string);
		fclose(ofp);
	}
	nc_stringbuf_destroy(&tmp_path);
	nc_stringbuf_destroy(&old_path);
	return -1;
}

nc_var_t *
nc_sysconfig_get(const nc_sysconfig_t *sc, const char *name)
{
	nc_var_t *var = nc_var_array_get(&sc->vars, name);
	return var && var->value ? var : NULL;
}

int
nc_sysconfig_get_string(const nc_sysconfig_t *sc, const char *name, char **p)
{
	nc_var_t *var;

	nc_string_free(p);
	if ((var = nc_sysconfig_get(sc, name)) != NULL)
		return nc_string_dup(p, var->value);
	return 1;
}

int
nc_sysconfig_get_uint(const nc_sysconfig_t *sc, const char *name, unsigned int *p)
{
	nc_var_t *var;

	if ((var = nc_sysconfig_get(sc, name)) != NULL) {
		return nc_parse_uint(var->value, p, 0);
	}
	return 1;
}

int
nc_sysconfig_find_matching(nc_sysconfig_t *sc, const char *prefix,
                           nc_string_array_t *res)
{
	unsigned int i, pfxlen;
	nc_var_t *var;

	pfxlen = strlen(prefix);
	for (i = 0, var = sc->vars.data; i < sc->vars.count; ++i, ++var) {
		const char *value = var->value;

		if(!value || !*value)
			continue;

		if (!strncmp(var->name, prefix, pfxlen))
			nc_string_array_append(res, var->name);
	}
	return res->count;
}

int
nc_sysconfig_del_matching(nc_sysconfig_t *sc, const char *prefix)
{
	unsigned int i, pfxlen;
	nc_var_t *var;
	int count = 0;

	if(!sc || !prefix)
		return -1;

	pfxlen = strlen(prefix);
	for (i = 0, var = sc->vars.data; i < sc->vars.count; ++i, ++var) {
		const char *value = var->value;

		if(!value) /* deleted anyway */
			continue;

		if (!strncmp(var->name, prefix, pfxlen)) {
			nc_string_free(&var->value);
			count++;
		}
	}
	return count;
}

int
nc_sysconfig_del(nc_sysconfig_t *sc, const char *name)
{
	return nc_var_array_set(&sc->vars, name, NULL);
}

int
nc_sysconfig_set(nc_sysconfig_t *sc, const char *name, const char *value)
{
	return nc_var_array_set(&sc->vars, name, value);
}

int
nc_sysconfig_set_uint(nc_sysconfig_t *sc, const char *name, unsigned int value)
{
	return nc_var_array_set_uint(&sc->vars, name, value);
}


void
nc_sysconfig_array_init(nc_sysconfig_array_t *array)
{
	if(array) {
		memset(array, 0, sizeof(*array));
	}
}

void
nc_sysconfig_array_destroy(nc_sysconfig_array_t *array)
{
	if(array) {
		while(array->count--) {
			nc_sysconfig_free(array->data[array->count]);
		}
		free(array->data);
		nc_sysconfig_array_init(array);
	}
}

static int
__nc_sysconfig_array_realloc(nc_sysconfig_array_t *array, unsigned int newsize)
{
	nc_sysconfig_t **newdata;

	newsize = (newsize + NC_SYSCONFIG_ARRAY_CHUNK);
	newdata = realloc(array->data, newsize * sizeof(nc_sysconfig_t *));
	assert(newdata != NULL);
	if (!newdata)
		return -1;

	array->data = newdata;
	memset( &array->data[array->count], 0,
		(newsize - array->count) * sizeof(nc_sysconfig_t *));
/*
	for (i = array->count; i < newsize; ++i) {
		array->data[i] = NULL;
	}
*/
	return 0;
}

int
nc_sysconfig_array_append(nc_sysconfig_array_t *array, nc_sysconfig_t *sc)
{
	if(!array || !sc)
		return -1;

	if((array->count % NC_SYSCONFIG_ARRAY_CHUNK) == 0) {
		if(__nc_sysconfig_array_realloc(array, array->count) != 0)
			return -1;
	}
	array->data[array->count++] = sc;
	return 0;
}

