/*
 *	logging utilities
 *
 *	Copyright (C) 2010 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2021 SUSE LCC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Olaf Kirch
 *		Marius Tomaschewski
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <syslog.h>
#ifdef HAVE_PTHREAD
#include <pthread.h>
#endif
#include <assert.h>

#include <netcontrol/logger.h>
#include <logging.h>
#include <sutils.h>


#ifdef __GNUC__
#define ATTRIBUTE_UNUSED	__attribute__((__unused__))
#else
#define ATTRIBUTE_UNUSED	/* unused */
#endif


/* inline, just forwarding to pthread calls */
static int			__nc_logger_init(void);
static void			__nc_logger_init_once(void);
static int			__nc_logger_lock(void);
static int			__nc_logger_unlock(void);

static void			__nc_logger_close(void);

static void	__nc_stream_logger(	const char *category,
					int         priority,
					const char *func,
					const char *file,
					long long   line,
					const char *msg,
					size_t      len);
static void	__nc_syslog_logger(	const char *category,
					int         priority,
					const char *func,
					const char *file,
					long long   line,
					const char *msg,
					size_t      len);

#define DEFAULT_CATEGORY	"netcontrol"
static const nc_intmap_t	__nc_log_debug_flags_names[] = {
	{ "netcf",		NC_TRACE_NETCF		},
	{ "ifcfg",		NC_TRACE_IFCFG		},
	{ "wicked",		NC_TRACE_WICKED		},
	{ NULL,			0			},
};


#ifdef HAVE_PTHREAD
#ifdef PTHREAD_MUTEX_INITIALIZER
static pthread_mutex_t  	__nc_logger_mutex = PTHREAD_MUTEX_INITIALIZER;
#else
static pthread_mutex_t  	__nc_logger_mutex;
#endif
static pthread_once_t		__nc_logger_init_guard = PTHREAD_ONCE_INIT;
#else
static int                      __nc_logger_init_guard = 0;
#endif
static FILE *			__nc_logger_fd  = NULL;
static nc_logger_t		__nc_logger_drv = NULL;


static inline int
__nc_logger_init(void)
{
	int ret = 0;
#ifdef HAVE_PTHREAD
	ret = pthread_once(&__nc_logger_init_guard, __nc_logger_init_once);
	assert(ret == 0);
#else
	if(!__nc_logger_init_guard) {
		__nc_logger_init_guard = 1;
		__nc_logger_init_once();
	}
#endif
	return ret;
}

static void
__nc_logger_init_once(void)
{
#ifdef HAVE_PTHREAD
#ifndef PTHREAD_MUTEX_INITIALIZER
	pthread_mutexattr_t attr;
	int ret;

	ret = pthread_mutexattr_init(&attr);
	assert(ret == 0);
	if(ret == 0) {
		ret = pthread_mutex_init(&__nc_logger_mutex, attr);
		assert(ret == 0);
	}
#endif
#endif
}

static inline int
__nc_logger_lock(void)
{
	int ret = 0;
#ifdef HAVE_PTHREAD
	ret = pthread_mutex_lock(&__nc_logger_mutex);
	assert(ret == 0);
#endif
	return ret;
}

static int
__nc_logger_unlock(void)
{
	int ret = 0;
#ifdef HAVE_PTHREAD
	ret = pthread_mutex_unlock(&__nc_logger_mutex);
	assert(ret == 0);
#endif
	return ret;
}

static void
__nc_logger_close(void)
{
	if(__nc_logger_fd && __nc_logger_fd != stderr) {
		fclose(__nc_logger_fd);
	}
	__nc_logger_fd  = NULL;
	__nc_logger_drv = NULL;
}

int
nc_logger_redirect_to(nc_logger_t logger)
{
	int ret = -1;

	if(__nc_logger_init() == 0 && __nc_logger_lock() == 0) {

		/* first open wins -- until a close */
		ret = -2;
		if(__nc_logger_fd || __nc_logger_drv)
			goto unlock;

		__nc_logger_drv = logger;

		ret = 0;
		unlock:
			__nc_logger_unlock();
	}
	return ret;
}

int
nc_logger_open_file(const char *filename)
{
	int ret = -1;

	if(__nc_logger_init() == 0 && __nc_logger_lock() == 0) {

		/* first open wins -- until a close */
		ret = -2;
		if(__nc_logger_fd || __nc_logger_drv)
			goto unlock;

		if(filename && *filename) {
			__nc_logger_fd = fopen(filename, "ae");
			ret = 0;
		}
		if(!ret && __nc_logger_fd) {
			__nc_logger_drv = __nc_stream_logger;
		}

		unlock:
			__nc_logger_unlock();
	}
	return ret;
}

int
nc_logger_open_syslog(const char *ident)
{
	int ret = -1;

	if(__nc_logger_init() == 0 && __nc_logger_lock() == 0) {

		/* first open wins -- until close */
		ret = -2;
		if(__nc_logger_fd || __nc_logger_drv)
			goto unlock;

		ret = 0;
		(void)ident; /* we currently do not call openlog */
		__nc_logger_fd = NULL;
		__nc_logger_drv = __nc_syslog_logger;

		unlock:
			__nc_logger_unlock();
	}
	return ret;
}

int
nc_logger_open_stderr(void)
{
	int ret = -1;

	if(__nc_logger_init() == 0 && __nc_logger_lock() == 0) {

		/* first open wins -- until close */
		ret = -1;
		if(__nc_logger_fd || __nc_logger_drv)
			goto unlock;

		ret = 0;
		__nc_logger_fd = stderr;
		__nc_logger_drv = __nc_stream_logger;

		unlock:
			__nc_logger_unlock();
	}
	return ret;
}

int
nc_logger_close(void)
{
	int ret = -1;

	if(__nc_logger_init() == 0 && __nc_logger_lock() == 0) {

		__nc_logger_close();
		ret = 0;

		__nc_logger_unlock();
	}
	return ret;
}


static void	__nc_stream_logger(	const char *category ATTRIBUTE_UNUSED,
					int         priority,
					const char *func     ATTRIBUTE_UNUSED,
					const char *file     ATTRIBUTE_UNUSED,
					long long   line     ATTRIBUTE_UNUSED,
					const char *msg,
					size_t      len      ATTRIBUTE_UNUSED)
{
	switch(priority) {
		case NC_LOG_FATAL:
			fprintf(__nc_logger_fd, "Fatal: ");
		break;
		case NC_LOG_ERROR:
			fprintf(__nc_logger_fd, "Error: ");
		break;
		case NC_LOG_WARN:
			fprintf(__nc_logger_fd, "Warning: ");
		break;
		case NC_LOG_INFO:
			fprintf(__nc_logger_fd, "Info: ");
		break;
		case NC_LOG_DEBUG:
		default:
			fprintf(__nc_logger_fd, "%s: ",
				(category ? category : "::"));
		break;
	}
	fprintf(__nc_logger_fd, "%s", msg);
	fprintf(__nc_logger_fd, "\n");
	fflush(__nc_logger_fd);
}

static void	__nc_syslog_logger(	const char *category ATTRIBUTE_UNUSED,
					int         priority,
					const char *func     ATTRIBUTE_UNUSED,
					const char *file     ATTRIBUTE_UNUSED,
					long long   line     ATTRIBUTE_UNUSED,
					const char *msg,
					size_t      len      ATTRIBUTE_UNUSED)
{
	switch(priority) {
		case NC_LOG_FATAL:
			priority = LOG_CRIT;
		break;
		case NC_LOG_ERROR:
			priority = LOG_ERR;
		break;
		case NC_LOG_WARN:
			priority = LOG_WARNING;
		break;
		case NC_LOG_INFO:
			priority = LOG_INFO;
		break;
		case NC_LOG_DEBUG:
		default:
			priority = LOG_DEBUG;
		break;
	}
	syslog(priority, "%s", msg);
}

extern void
__nc_fatal(const char *func, const char *file, long long line,
		const char *fmt, ...)
{
	nc_stringbuf_t buf = NC_STRINGBUF_INIT;
	va_list ap;

	if(!(__nc_logger_init() == 0 && __nc_logger_lock() == 0))
		exit(1);

	if(__nc_logger_drv) {
		va_start(ap, fmt);
		nc_stringbuf_vprintf(&buf, fmt, ap);
		va_end(ap);

		if(buf.len > 0) {
			__nc_logger_drv(DEFAULT_CATEGORY,
					NC_LOG_FATAL,
					func, file, line,
					buf.string, buf.len);
		}
		nc_stringbuf_destroy(&buf);
	}
	__nc_logger_unlock();

	exit(1);
}

extern void
__nc_error(const char *func, const char *file, long long line,
		const char *fmt, ...)
{
	nc_stringbuf_t buf = NC_STRINGBUF_INIT;
	va_list ap;

	if(!(__nc_logger_init() == 0 && __nc_logger_lock() == 0))
		return;

	if(__nc_logger_drv) {
		va_start(ap, fmt);
		nc_stringbuf_vprintf(&buf, fmt, ap);
		va_end(ap);

		if(buf.len > 0) {
			__nc_logger_drv(DEFAULT_CATEGORY,
					NC_LOG_ERROR,
					func, file, line,
					buf.string, buf.len);
		}
		nc_stringbuf_destroy(&buf);
	}
	__nc_logger_unlock();
}

extern void
__nc_warn (const char *func, const char *file, long long line,
		const char *fmt, ...)
{
	nc_stringbuf_t buf = NC_STRINGBUF_INIT;
	va_list ap;

	if(!(__nc_logger_init() == 0 && __nc_logger_lock() == 0))
		return;

	if(__nc_logger_drv) {
		va_start(ap, fmt);
		nc_stringbuf_vprintf(&buf, fmt, ap);
		va_end(ap);

		if(buf.len > 0) {
			__nc_logger_drv(DEFAULT_CATEGORY,
					NC_LOG_WARN,
					func, file, line,
					buf.string, buf.len);
		}
		nc_stringbuf_destroy(&buf);
	}
	__nc_logger_unlock();
}

extern void
__nc_info (const char *func, const char *file, long long line,
		const char *fmt, ...)
{
	nc_stringbuf_t buf = NC_STRINGBUF_INIT;
	va_list ap;

	if(!(__nc_logger_init() == 0 && __nc_logger_lock() == 0))
		return;

	if(__nc_logger_drv) {
		va_start(ap, fmt);
		nc_stringbuf_vprintf(&buf, fmt, ap);
		va_end(ap);

		if(buf.len > 0) {
			__nc_logger_drv(DEFAULT_CATEGORY,
					NC_LOG_INFO,
					func, file, line,
					buf.string, buf.len);
		}
		nc_stringbuf_destroy(&buf);
	}
	__nc_logger_unlock();
}

extern void
__nc_trace(const char *func, const char *file, long long line,
		const char *fmt, ...)
{
	nc_stringbuf_t buf = NC_STRINGBUF_INIT;
	va_list ap;

	if(!(__nc_logger_init() == 0 && __nc_logger_lock() == 0))
		return;

	if(__nc_logger_drv) {
		va_start(ap, fmt);
		nc_stringbuf_vprintf(&buf, fmt, ap);
		va_end(ap);

		if(buf.len > 0) {
			__nc_logger_drv(NULL,
					NC_LOG_DEBUG,
					func, file, line,
					buf.string, buf.len);
		}
		nc_stringbuf_destroy(&buf);
	}
	__nc_logger_unlock();
}

extern void
__nc_debug(int facility, const char *func, const char *file,
		long long line, const char *fmt, ...)
{
	nc_stringbuf_t buf = NC_STRINGBUF_INIT;
	va_list ap;
	const char *category;

	if(!(__nc_logger_init() == 0 && __nc_logger_lock() == 0))
		return;

	if(__nc_logger_drv) {
		va_start(ap, fmt);
		nc_stringbuf_vprintf(&buf, fmt, ap);
		va_end(ap);

		if(buf.len > 0) {
			category = nc_format_int_mapped(facility,
					__nc_log_debug_flags_names);

			__nc_logger_drv(category,
					NC_LOG_DEBUG,
					func, file, line,
					buf.string, buf.len);
		}
		nc_stringbuf_destroy(&buf);
	}
	__nc_logger_unlock();
}

