use crate::{Int32, UInt32};
use himmelblau_red_asn1::{Asn1Object, OctetString};
use himmelblau_red_asn1_derive::Sequence;

/// (*EncryptedData*) Chunck of data that is encrypted in Kerberos exchanges.
/// ```asn1
/// EncryptedData   ::= SEQUENCE {
///        etype   [0] Int32 -- EncryptionType --,
///        kvno    [1] UInt32 OPTIONAL,
///        cipher  [2] OCTET STRING -- ciphertext
/// }
/// ```
#[derive(Sequence, Default, Debug, Clone, PartialEq)]
pub struct EncryptedData {
    #[seq_field(context_tag = 0)]
    pub etype: Int32,
    #[seq_field(context_tag = 1)]
    pub kvno: Option<UInt32>,
    #[seq_field(context_tag = 2)]
    pub cipher: OctetString,
}

impl EncryptedData {
    pub fn new(
        etype: Int32,
        kvno: Option<UInt32>,
        cipher: OctetString,
    ) -> Self {
        return Self {
            etype,
            kvno,
            cipher,
        };
    }
}

#[cfg(test)]
mod test {
    use super::*;
    use himmelblau_kerberos_constants::etypes::*;

    #[test]
    fn test_build_encrypted_data() {
        let enc_data = EncryptedData::new(
            AES256_CTS_HMAC_SHA1_96,
            None,
            vec![
                0x64, 0x67, 0x3f, 0x70, 0x45, 0x50, 0x57, 0xa5, 0x16, 0x16,
                0xf6, 0xa9, 0x0b, 0x8c, 0x04, 0xe6, 0xa9, 0x5d, 0x8e, 0x1d,
                0x95, 0xdf, 0x98, 0x67, 0x29, 0x16, 0x9a, 0x54, 0xbc, 0x66,
                0xae, 0x29, 0x9d, 0xd1, 0xec, 0x62, 0xbc, 0x99, 0xce, 0x2c,
                0x9f, 0x6a, 0x4e, 0xf1, 0xf0, 0x25, 0xf9, 0x9e, 0x13, 0xa5,
                0x94, 0xa2, 0x39, 0x80, 0x7f, 0xdf,
            ],
        );

        assert_eq!(
            vec![
                0x30, 0x41, 0xa0, 0x03, 0x02, 0x01, 0x12, 0xa2, 0x3a, 0x04,
                0x38, 0x64, 0x67, 0x3f, 0x70, 0x45, 0x50, 0x57, 0xa5, 0x16,
                0x16, 0xf6, 0xa9, 0x0b, 0x8c, 0x04, 0xe6, 0xa9, 0x5d, 0x8e,
                0x1d, 0x95, 0xdf, 0x98, 0x67, 0x29, 0x16, 0x9a, 0x54, 0xbc,
                0x66, 0xae, 0x29, 0x9d, 0xd1, 0xec, 0x62, 0xbc, 0x99, 0xce,
                0x2c, 0x9f, 0x6a, 0x4e, 0xf1, 0xf0, 0x25, 0xf9, 0x9e, 0x13,
                0xa5, 0x94, 0xa2, 0x39, 0x80, 0x7f, 0xdf
            ],
            enc_data.build()
        );
    }

    #[test]
    fn test_parse_encrypted_data() {
        let enc_data = EncryptedData::new(
            AES256_CTS_HMAC_SHA1_96,
            None,
            vec![
                0x64, 0x67, 0x3f, 0x70, 0x45, 0x50, 0x57, 0xa5, 0x16, 0x16,
                0xf6, 0xa9, 0x0b, 0x8c, 0x04, 0xe6, 0xa9, 0x5d, 0x8e, 0x1d,
                0x95, 0xdf, 0x98, 0x67, 0x29, 0x16, 0x9a, 0x54, 0xbc, 0x66,
                0xae, 0x29, 0x9d, 0xd1, 0xec, 0x62, 0xbc, 0x99, 0xce, 0x2c,
                0x9f, 0x6a, 0x4e, 0xf1, 0xf0, 0x25, 0xf9, 0x9e, 0x13, 0xa5,
                0x94, 0xa2, 0x39, 0x80, 0x7f, 0xdf,
            ],
        );

        assert_eq!(
            enc_data,
            EncryptedData::parse(&[
                0x30, 0x41, 0xa0, 0x03, 0x02, 0x01, 0x12, 0xa2, 0x3a, 0x04,
                0x38, 0x64, 0x67, 0x3f, 0x70, 0x45, 0x50, 0x57, 0xa5, 0x16,
                0x16, 0xf6, 0xa9, 0x0b, 0x8c, 0x04, 0xe6, 0xa9, 0x5d, 0x8e,
                0x1d, 0x95, 0xdf, 0x98, 0x67, 0x29, 0x16, 0x9a, 0x54, 0xbc,
                0x66, 0xae, 0x29, 0x9d, 0xd1, 0xec, 0x62, 0xbc, 0x99, 0xce,
                0x2c, 0x9f, 0x6a, 0x4e, 0xf1, 0xf0, 0x25, 0xf9, 0x9e, 0x13,
                0xa5, 0x94, 0xa2, 0x39, 0x80, 0x7f, 0xdf,
            ])
            .unwrap()
            .1
        );
    }
}
