/*
 * brass - Braille and speech server
 *
 * Copyright (C) 2001 by Roger Butenuth, All rights reserved.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 *
 * $Id: softspeech.c,v 1.16 2003/05/29 16:58:43 butenuth Exp $
 */
#define _GNU_SOURCE
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <malloc.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ioctl.h>

#include "synthesizer.h"
#include "softspeech.h"

static int s_close (synth_t * s);

static int s_synth (synth_t * s, unsigned char *buffer);

static int s_flush (synth_t * s);

static int s_clear (synth_t * s);

static int s_index_set (struct synth_struct *s);

static int s_index_wait (struct synth_struct *s, int id, int timeout);

static int s_get_param (struct synth_struct *s, synth_par_t par, int *value);

static int s_set_param (struct synth_struct *s, synth_par_t par, int value);

typedef struct synth_state
{
  int param[S_MAX];
  int initialized;
} synth_state_t;

static int start_process (void (*function) (int, int),
                          int *to, int *from, int *pid);
static void send_conf_vars (int language,
                            void *context, lookup_string_t lookup);

static synth_state_t german_state;

static synth_state_t english_state;

static int to_fd, from_fd, speech_pid;

static int fp_ref = 0;

static int current_language = -1;

                                                                        /* static int sync_mark_no = 0; *//* currently 
                                                                           used 
                                                                           number 
                                                                         */
                                                               /* static struct timeval mark; *//* time the
                                                                  mark has
                                                                  been set */

static synth_t german_softspeech = {
  &german_state,
  &languages[LANG_GERMAN],
  "Softspeech/German",
  NULL,
  s_close,
  s_synth,
  s_flush,
  s_clear,
  s_index_set,
  s_index_wait,
  s_get_param,
  s_set_param,
};

static synth_t english_softspeech = {
  &english_state,
  &languages[LANG_ENGLISH],
  "Softspeech/British English",
  NULL,
  s_close,
  s_synth,
  s_flush,
  s_clear,
  s_index_set,
  s_index_wait,
  s_get_param,
  s_set_param,
};

static char *conf_var[] = {
  "phoneme_generator_directory",
  "phoneme_generator_command",
  "synthesizer_directory",
  "synthesizer_command",
  "samples_per_second",
  NULL
};

/*
 * ----------------------------------------------------------------------
 * Called before library is loaded.
 * ----------------------------------------------------------------------
 */
/*
 * ----------------------------------------------------------------------
 * Called before library is unloaded.
 * ----------------------------------------------------------------------
 */
/*
 * ----------------------------------------------------------------------
 * Send a parameter block with data to the server process.
 * ----------------------------------------------------------------------
 */
static void send_cmd (cmd_block_t * cblock)
{
  cblock->cmd_length = (char *) &cblock->data[cblock->len] - (char *) cblock;
  write (to_fd, cblock, cblock->cmd_length);
}

/*
 * ----------------------------------------------------------------------
 * General open function for german and english synthesizer.
 * Second open increments refcount.
 * Return 0 on succes, 1 on error.
 * ----------------------------------------------------------------------
 */
synth_t *synth_open (void *context, lookup_string_t lookup)
{
  synth_t *s;

  int r = 0;

  char *language = (*lookup) (context, "language");

  if (fp_ref == 0)
   {
     r = start_process (server_process, &to_fd, &from_fd, &speech_pid);
     if (r != 0)
       return NULL;
     current_language = -1;
   }
  if (r == 0)
    fp_ref++;
  else
    return NULL;

  if (!strcasecmp (language, "english"))
   {
     if (!english_state.initialized)
      {
        english_state.param[S_SPEED] = 1000;
        english_state.param[S_PITCH] = 1000;
        english_state.param[S_VOLUME] = 1000;
        send_conf_vars (0, context, lookup);
        english_state.initialized = 1;
      }
     s = &english_softspeech;
   }
  else if (!strcasecmp (language, "german"))
   {
     if (!german_state.initialized)
      {
        german_state.param[S_SPEED] = 1000;
        german_state.param[S_PITCH] = 1000;
        german_state.param[S_VOLUME] = 1000;
        send_conf_vars (1, context, lookup);
        german_state.initialized = 1;
      }
     s = &german_softspeech;
   }
  else
   {
     s = NULL;
   }

  return s;
}

/*
 * ----------------------------------------------------------------------
 * General close. Decrement refcount, do real close when count reaches
 * zero.
 * ----------------------------------------------------------------------
 */
static int s_close (synth_t * s)
{
  int result = 0;

  int status;

  pid_t pid;

  assert (s->state->initialized);
  assert (fp_ref > 0);
  s->state->initialized = 0;

  fp_ref--;
  if (fp_ref == 0)
   {
     cmd_block_t c;

     c.cmd = C_TERMINATE;
     c.len = 0;
     send_cmd (&c);
     /* kill(speech_pid, SIGTERM); */
     pid = waitpid (speech_pid, &status, 0);
     if (WIFEXITED (status))
       result = WEXITSTATUS (status);
     else if (WIFSIGNALED (status))
       result = WTERMSIG (status);
     else
       result = 1;
   }

  return result;
}

/*
 * ----------------------------------------------------------------------
 * Verify that the synthesizer is set to the correct language.
 * Switch if necessary and reset parameters.
 * It is necessary to give a short break to Softspeech. Somewhere 
 * there is a timing bug left.
 * ----------------------------------------------------------------------
 */
static void verify_language (struct synth_struct *s)
{
  cmd_block_t c;

  int p;

  if (s->lang->lang == LANG_GERMAN && current_language != LANG_GERMAN)
   {
     c.cmd = C_SELECT_LANGUAGE;
     c.param_num = LANG_GERMAN;
     c.len = 0;
     send_cmd (&c);
     current_language = LANG_GERMAN;
     for (p = 0; p < S_MAX; p++)
      {
        s_set_param (s, p, s->state->param[p]);
      }
   }
  if (s->lang->lang == LANG_ENGLISH && current_language != LANG_ENGLISH)
   {

     c.cmd = C_SELECT_LANGUAGE;
     c.param_num = LANG_ENGLISH;
     c.len = 0;
     send_cmd (&c);
     current_language = LANG_ENGLISH;
     for (p = 0; p < S_MAX; p++)
      {
        s_set_param (s, p, s->state->param[p]);
      }
   }
}

/*
 * ----------------------------------------------------------------------
 * Copy Text to server. Text is not spoken until a flush command
 * arrives.
 * ----------------------------------------------------------------------
 */
static int s_synth (struct synth_struct *s, unsigned char *buffer)
{
  cmd_block_t c;

  assert (s->state->initialized);
  assert (strlen ((char *) buffer) + 1 <= MAX_DATA_LEN);

  verify_language (s);
  c.cmd = C_SYNTH;
  c.len = strlen ((char *) buffer);
  memcpy (c.data, buffer, strlen ((char *) buffer));
  send_cmd (&c);

  return 0;
}

/*
 * ----------------------------------------------------------------------
 * Flush synthesizer. This triggers the synthesizer and starts 
 * the synthesis.
 * ----------------------------------------------------------------------
 */
static int s_flush (synth_t * s)
{
  cmd_block_t c;

  assert (s->state->initialized);

  verify_language (s);
  c.cmd = C_FLUSH;
  c.len = 0;
  send_cmd (&c);

  return 0;
}

/*
 * ----------------------------------------------------------------------
 * Remove anything in the synthesizier speech queue.
 * ----------------------------------------------------------------------
 */
static int s_clear (synth_t * s)
{
  cmd_block_t c;

  assert (s->state->initialized);

  verify_language (s);
  c.cmd = C_CLEAR;
  c.len = 0;
  send_cmd (&c);

  return 0;
}

/*
 * ----------------------------------------------------------------------
 * ToDo
 * ----------------------------------------------------------------------
 */
static int s_index_set (struct synth_struct *s)
{
  return 0;
}

/*
 * ----------------------------------------------------------------------
 * ToDo
 * ----------------------------------------------------------------------
 */
static int s_index_wait (struct synth_struct *s, int id, int timeout)
{
  int res = 0;

  return res;
}

/*
 * ----------------------------------------------------------------------
 * Get a synthesizer parameter.
 * ----------------------------------------------------------------------
 */
static int s_get_param (struct synth_struct *s, synth_par_t par, int *value)
{
  if (par >= 0 && par < S_MAX)
   {
     *value = s->state->param[par];
     return 0;
   }
  else
    return 1;
}

/*
 * ----------------------------------------------------------------------
 * Set a parameter of the synthesizer.
 * ----------------------------------------------------------------------
 */
static int s_set_param (struct synth_struct *s, synth_par_t par, int value)
{
  if (par >= 0 && par < S_MAX)
   {
     cmd_block_t c;

     s->state->param[par] = value;

     verify_language (s);
     c.cmd = C_SET_PARAM;
     c.param_num = par;
     c.param_value = value;
     c.len = 0;
     send_cmd (&c);
   }
  else
   {
     return 1;
   }

  return 0;
}

/*
 * ----------------------------------------------------------------------
 * Start a function (int the background) and connect it with two pipes
 * to the caller:
 *  *to from caller to program
 *  *from from programm to caller
 * Process id of the program is returned in *pid.
 * ----------------------------------------------------------------------
 */
static int start_process (void (*function) (int, int),
                          int *to, int *from, int *pid)
{
  int to_pipe[2];               /* to client */

  int from_pipe[2];             /* from client */

  if (pipe (to_pipe) < 0 || pipe (from_pipe) < 0)
   {
     perror ("create pipes");
     return -1;
   }
  *pid = fork ();
  assert (*pid >= 0);
  if (*pid == 0)
   {                            /* child */
     close (from_pipe[0]);
     close (to_pipe[1]);
     (*function) (to_pipe[0], from_pipe[1]);
     kill (getppid (), SIGKILL);
     exit (1);
   }
  else
   {                            /* parent */
     *to = to_pipe[1];
     *from = from_pipe[0];
     close (to_pipe[0]);
     close (from_pipe[1]);
   }

  return 0;
}

/*
 * ----------------------------------------------------------------------
 * 
 * ----------------------------------------------------------------------
 */
static void send_conf_vars (int language, void *context,
                            lookup_string_t lookup)
{
  cmd_block_t c;

  int i;

  char *value;

  for (i = 0; conf_var[i] != NULL; i++)
   {
     c.cmd = C_CONFIG_VAR;
     c.param_num = language;
     strcpy (c.name, conf_var[i]);
     value = (*lookup) (context, conf_var[i]);
     if (value == NULL)
      {
        fprintf (stderr, "Language %d: Variable \"%s\" not defined\n",
                 language, conf_var[i]);
        exit (1);
      }
     c.len = strlen (value) + 1;
     strcpy (c.data, value);
     send_cmd (&c);
   }
}
