/*
 * This file is part of CycloneDX Rust Cargo.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 */

use crate::models::organization::{OrganizationalContact, OrganizationalEntity};
use crate::validation::{Validate, ValidationContext, ValidationPathComponent, ValidationResult};

/// Provides credits to organizations or individuals who contributed to a vulnerability.
#[derive(Clone, Debug, PartialEq, Eq)]
pub struct VulnerabilityCredits {
    pub organizations: Option<Vec<OrganizationalEntity>>,
    pub individuals: Option<Vec<OrganizationalContact>>,
}

impl Validate for VulnerabilityCredits {
    fn validate_with_context(&self, context: ValidationContext) -> ValidationResult {
        let mut results: Vec<ValidationResult> = vec![];

        if let Some(organizations) = &self.organizations {
            for (index, organization) in organizations.iter().enumerate() {
                let uri_context = context.extend_context(vec![
                    ValidationPathComponent::Struct {
                        struct_name: "VulnerabilityCredits".to_string(),
                        field_name: "organizations".to_string(),
                    },
                    ValidationPathComponent::Array { index },
                ]);
                results.push(organization.validate_with_context(uri_context));
            }
        }

        if let Some(individuals) = &self.individuals {
            for (index, individual) in individuals.iter().enumerate() {
                let uri_context = context.extend_context(vec![
                    ValidationPathComponent::Struct {
                        struct_name: "VulnerabilityCredits".to_string(),
                        field_name: "individuals".to_string(),
                    },
                    ValidationPathComponent::Array { index },
                ]);
                results.push(individual.validate_with_context(uri_context));
            }
        }

        results
            .into_iter()
            .fold(ValidationResult::default(), |acc, result| acc.merge(result))
    }
}

#[cfg(test)]
mod test {
    use crate::{external_models::normalized_string::NormalizedString, validation::FailureReason};

    use super::*;
    use pretty_assertions::assert_eq;

    #[test]
    fn valid_vulnerability_credits_should_pass_validation() {
        let validation_result = VulnerabilityCredits {
            organizations: Some(vec![OrganizationalEntity {
                name: Some(NormalizedString::new("name")),
                url: None,
                contact: None,
            }]),
            individuals: Some(vec![OrganizationalContact {
                name: Some(NormalizedString::new("name")),
                email: None,
                phone: None,
            }]),
        }
        .validate();

        assert_eq!(validation_result, ValidationResult::Passed);
    }

    #[test]
    fn invalid_vulnerability_credits_should_fail_validation() {
        let validation_result = VulnerabilityCredits {
            organizations: Some(vec![OrganizationalEntity {
                name: Some(NormalizedString("invalid\tname".to_string())),
                url: None,
                contact: None,
            }]),
            individuals: Some(vec![OrganizationalContact {
                name: Some(NormalizedString("invalid\tname".to_string())),
                email: None,
                phone: None,
            }]),
        }
        .validate();

        assert_eq!(
            validation_result,
            ValidationResult::Failed {
                reasons: vec![
                    FailureReason {
                        message:
                            "NormalizedString contains invalid characters \\r \\n \\t or \\r\\n"
                                .to_string(),
                        context: ValidationContext(vec![
                            ValidationPathComponent::Struct {
                                struct_name: "VulnerabilityCredits".to_string(),
                                field_name: "organizations".to_string()
                            },
                            ValidationPathComponent::Array { index: 0 },
                            ValidationPathComponent::Struct {
                                struct_name: "OrganizationalEntity".to_string(),
                                field_name: "name".to_string()
                            },
                        ])
                    },
                    FailureReason {
                        message:
                            "NormalizedString contains invalid characters \\r \\n \\t or \\r\\n"
                                .to_string(),
                        context: ValidationContext(vec![
                            ValidationPathComponent::Struct {
                                struct_name: "VulnerabilityCredits".to_string(),
                                field_name: "individuals".to_string()
                            },
                            ValidationPathComponent::Array { index: 0 },
                            ValidationPathComponent::Struct {
                                struct_name: "OrganizationalContact".to_string(),
                                field_name: "name".to_string()
                            },
                        ])
                    },
                ]
            }
        );
    }
}
