searchData={"items":[{"type":"module","title":"Kernel","doc":"`Kernel` is Elixir's default environment.\n\nIt mainly consists of:\n\n  * basic language primitives, such as arithmetic operators, spawning of processes,\n    data type handling, and others\n  * macros for control-flow and defining new functionality (modules, functions, and the like)\n  * guard checks for augmenting pattern matching\n\nYou can invoke `Kernel` functions and macros anywhere in Elixir code\nwithout the use of the `Kernel.` prefix since they have all been\nautomatically imported. For example, in IEx, you can call:\n\n    iex> is_number(13)\n    true\n\nIf you don't want to import a function or macro from `Kernel`, use the `:except`\noption and then list the function/macro by arity:\n\n    import Kernel, except: [if: 2, unless: 2]\n\nSee `Kernel.SpecialForms.import/2` for more information on importing.\n\nElixir also has special forms that are always imported and\ncannot be skipped. These are described in `Kernel.SpecialForms`.","ref":"Kernel.html"},{"type":"module","title":"The standard library - Kernel","doc":"`Kernel` provides the basic capabilities the Elixir standard library\nis built on top of. It is recommended to explore the standard library\nfor advanced functionality. Here are the main groups of modules in the\nstandard library (this list is not a complete reference, see the\ndocumentation sidebar for all entries).\n\n#","ref":"Kernel.html#module-the-standard-library"},{"type":"module","title":"Built-in types - Kernel","doc":"The following modules handle Elixir built-in data types:\n\n  * `Atom` - literal constants with a name (`true`, `false`, and `nil` are atoms)\n  * `Float` - numbers with floating point precision\n  * `Function` - a reference to code chunk, created with the `fn/1` special form\n  * `Integer` - whole numbers (not fractions)\n  * `List` - collections of a variable number of elements (linked lists)\n  * `Map` - collections of key-value pairs\n  * `Process` - light-weight threads of execution\n  * `Port` - mechanisms to interact with the external world\n  * `Tuple` - collections of a fixed number of elements\n\nThere are two data types without an accompanying module:\n\n  * Bitstring - a sequence of bits, created with `Kernel.SpecialForms. >/1`.\n    When the number of bits is divisible by 8, they are called binaries and can\n    be manipulated with Erlang's `:binary` module\n  * Reference - a unique value in the runtime system, created with `make_ref/0`\n\n#","ref":"Kernel.html#module-built-in-types"},{"type":"module","title":"Data types - Kernel","doc":"Elixir also provides other data types that are built on top of the types\nlisted above. Some of them are:\n\n  * `Date` - `year-month-day` structs in a given calendar\n  * `DateTime` - date and time with time zone in a given calendar\n  * `Exception` - data raised from errors and unexpected scenarios\n  * `MapSet` - unordered collections of unique elements\n  * `NaiveDateTime` - date and time without time zone in a given calendar\n  * `Keyword` - lists of two-element tuples, often representing optional values\n  * `Range` - inclusive ranges between two integers\n  * `Regex` - regular expressions\n  * `String` - UTF-8 encoded binaries representing characters\n  * `Time` - `hour:minute:second` structs in a given calendar\n  * `URI` - representation of URIs that identify resources\n  * `Version` - representation of versions and requirements\n\n#","ref":"Kernel.html#module-data-types"},{"type":"module","title":"System modules - Kernel","doc":"Modules that interface with the underlying system, such as:\n\n  * `IO` - handles input and output\n  * `File` - interacts with the underlying file system\n  * `Path` - manipulates file system paths\n  * `System` - reads and writes system information\n\n#","ref":"Kernel.html#module-system-modules"},{"type":"module","title":"Protocols - Kernel","doc":"Protocols add polymorphic dispatch to Elixir. They are contracts\nimplementable by data types. See `Protocol` for more information on\nprotocols. Elixir provides the following protocols in the standard library:\n\n  * `Collectable` - collects data into a data type\n  * `Enumerable` - handles collections in Elixir. The `Enum` module\n    provides eager functions for working with collections, the `Stream`\n    module provides lazy functions\n  * `Inspect` - converts data types into their programming language\n    representation\n  * `List.Chars` - converts data types to their outside world\n    representation as charlists (non-programming based)\n  * `String.Chars` - converts data types to their outside world\n    representation as strings (non-programming based)\n\n#","ref":"Kernel.html#module-protocols"},{"type":"module","title":"Process-based and application-centric functionality - Kernel","doc":"The following modules build on top of processes to provide concurrency,\nfault-tolerance, and more.\n\n  * `Agent` - a process that encapsulates mutable state\n  * `Application` - functions for starting, stopping and configuring\n    applications\n  * `GenServer` - a generic client-server API\n  * `Registry` - a key-value process-based storage\n  * `Supervisor` - a process that is responsible for starting,\n    supervising and shutting down other processes\n  * `Task` - a process that performs computations\n  * `Task.Supervisor` - a supervisor for managing tasks exclusively\n\n#","ref":"Kernel.html#module-process-based-and-application-centric-functionality"},{"type":"module","title":"Supporting documents - Kernel","doc":"Elixir documentation also includes supporting documents under the\n\"Pages\" section. Those are:\n\n  * [Compatibility and deprecations](compatibility-and-deprecations.md) - lists\n    compatibility between every Elixir version and Erlang/OTP, release schema;\n    lists all deprecated functions, when they were deprecated and alternatives\n  * [Library guidelines](library-guidelines.md) - general guidelines, anti-patterns,\n    and rules for those writing libraries\n  * [Naming conventions](naming-conventions.md) - naming conventions for Elixir code\n  * [Operators](operators.md) - lists all Elixir operators and their precedences\n  * [Patterns and guards](patterns-and-guards.md) - an introduction to patterns,\n    guards, and extensions\n  * [Syntax reference](syntax-reference.md) - the language syntax reference\n  * [Typespecs](typespecs.md)- types and function specifications, including list of types\n  * [Unicode syntax](unicode-syntax.md) - outlines Elixir support for Unicode\n  * [Writing documentation](writing-documentation.md) - guidelines for writing\n    documentation in Elixir","ref":"Kernel.html#module-supporting-documents"},{"type":"module","title":"Guards - Kernel","doc":"This module includes the built-in guards used by Elixir developers.\nThey are a predefined set of functions and macros that augment pattern\nmatching, typically invoked after the `when` operator. For example:\n\n    def drive(%User{age: age}) when age >= 16 do\n      ...\n    end\n\nThe clause above will only be invoked if the user's age is more than\nor equal to 16. Guards also support joining multiple conditions with\n`and` and `or`. The whole guard is true if all guard expressions will\nevaluate to `true`. A more complete introduction to guards is available\nin the [Patterns and guards](patterns-and-guards.md) page.","ref":"Kernel.html#module-guards"},{"type":"module","title":"Truthy and falsy values - Kernel","doc":"Besides the booleans `true` and `false`, Elixir has the\nconcept of a \"truthy\" or \"falsy\" value.\n\n  *  a value is truthy when it is neither `false` nor `nil`\n  *  a value is falsy when it is either `false` or `nil`\n\nElixir has functions, like `and/2`, that *only* work with\nbooleans, but also functions that work with these\ntruthy/falsy values, like `&&/2` and `!/1`.","ref":"Kernel.html#module-truthy-and-falsy-values"},{"type":"module","title":"Structural comparison - Kernel","doc":"The functions in this module perform structural comparison. This allows\ndifferent data types to be compared using comparison operators:\n\n```elixir\niex> 1  /2`, ` =/2`,\n`  ~D[2017-04-01]\n\nwill return `true` because structural comparison compares the `:day`\nfield before `:month` or `:year`. In order to perform semantic comparisons,\nthe relevant data-types provide a `compare/2` function, such as\n`Date.compare/2`:\n\n    iex> Date.compare(~D[2017-03-31], ~D[2017-04-01])\n    :lt\n\nAlternatively, you can use the functions in the `Enum` module to\nsort or compute a maximum/minimum:\n\n    iex> Enum.sort([~D[2017-03-31], ~D[2017-04-01]], Date)\n    [~D[2017-03-31], ~D[2017-04-01]]\n    iex> Enum.max([~D[2017-03-31], ~D[2017-04-01]], Date)\n    ~D[2017-04-01]\n\nThe second argument is precisely the module to be used for semantic\ncomparison. Keeping this distinction is important, because if semantic\ncomparison was used by default for implementing data structures and\nalgorithms, they could become orders of magnitude slower!\n\nFinally, note there is an overall structural sorting order, called\n\"Term Ordering\", defined below. This order is provided for reference\npurposes, it is not required by Elixir developers to know it by heart.\n\n#","ref":"Kernel.html#module-structural-comparison"},{"type":"module","title":"Term ordering - Kernel","doc":"```\nnumber < atom < reference < function < port < pid < tuple < map < list < bitstring\n```\n\nWhen comparing two numbers of different types (a number being either\nan integer or a float), a conversion to the type with greater precision\nwill always occur, unless the comparison operator used is either `===/2`\nor `!==`. A float will be considered more precise than an integer, unless\nthe float is greater/less than +/-9007199254740992.0 respectively,\nat which point all the significant figures of the float are to the left\nof the decimal point. This behavior exists so that the comparison of large\nnumbers remains transitive.\n\nThe collection types are compared using the following rules:\n\n* Tuples are compared by size, then element by element.\n* Maps are compared by size, then by keys in ascending term order,\n  then by values in key order. In the specific case of maps' key\n  ordering, integers are always considered to be less than floats.\n* Lists are compared element by element.\n* Bitstrings are compared byte by byte, incomplete bytes are compared bit by bit.\n* Atoms are compared using their string value, codepoint by codepoint.\n\n#","ref":"Kernel.html#module-term-ordering"},{"type":"module","title":"Examples - Kernel","doc":"We can check the truthiness of a value by using the `!/1`\nfunction twice.\n\nTruthy values:\n\n    iex> !!true\n    true\n    iex> !!5\n    true\n    iex> !![1,2]\n    true\n    iex> !!\"foo\"\n    true\n\nFalsy values (of which there are exactly two):\n\n    iex> !!false\n    false\n    iex> !!nil\n    false","ref":"Kernel.html#module-examples"},{"type":"module","title":"Inlining - Kernel","doc":"Some of the functions described in this module are inlined by\nthe Elixir compiler into their Erlang counterparts in the\n[`:erlang`](`:erlang`) module.\nThose functions are called BIFs (built-in internal functions)\nin Erlang-land and they exhibit interesting properties, as some\nof them are allowed in guards and others are used for compiler\noptimizations.\n\nMost of the inlined functions can be seen in effect when\ncapturing the function:\n\n    iex> &Kernel.is_atom/1\n    &:erlang.is_atom/1\n\nThose functions will be explicitly marked in their docs as\n\"inlined by the compiler\".","ref":"Kernel.html#module-inlining"},{"type":"macro","title":"Kernel.&&/2","doc":"Boolean \"and\" operator.\n\nProvides a short-circuit operator that evaluates and returns\nthe second expression only if the first one evaluates to a truthy value\n(neither `false` nor `nil`). Returns the first expression\notherwise.\n\nNot allowed in guard clauses.","ref":"Kernel.html#&&/2"},{"type":"macro","title":"Examples - Kernel.&&/2","doc":"iex> Enum.empty?([]) && Enum.empty?([])\n    true\n\n    iex> List.first([]) && true\n    nil\n\n    iex> Enum.empty?([]) && List.first([1])\n    1\n\n    iex> false && throw(:bad)\n    false\n\nNote that, unlike `and/2`, this operator accepts any expression\nas the first argument, not only booleans.","ref":"Kernel.html#&&/2-examples"},{"type":"function","title":"Kernel.**/2","doc":"Power operator.\n\nIt takes two numbers for input. If both are integers and the right-hand\nside (the `exponent`) is also greater than or equal to 0, then the result\nwill also be an integer. Otherwise it returns a float.","ref":"Kernel.html#**/2"},{"type":"function","title":"Examples - Kernel.**/2","doc":"iex> 2 ** 2\n    4\n    iex> 2 ** -4\n    0.0625\n\n    iex> 2.0 ** 2\n    4.0\n    iex> 2 ** 2.0\n    4.0","ref":"Kernel.html#**/2-examples"},{"type":"function","title":"Kernel.*/2","doc":"Arithmetic multiplication operator.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#*/2"},{"type":"function","title":"Examples - Kernel.*/2","doc":"iex> 1 * 2\n    2","ref":"Kernel.html#*/2-examples"},{"type":"function","title":"Kernel.++/2","doc":"List concatenation operator. Concatenates a proper list and a term, returning a list.\n\nThe complexity of `a ++ b` is proportional to `length(a)`, so avoid repeatedly\nappending to lists of arbitrary length, for example, `list ++ [element]`.\nInstead, consider prepending via `[element | rest]` and then reversing.\n\nIf the `right` operand is not a proper list, it returns an improper list.\nIf the `left` operand is not a proper list, it raises `ArgumentError`.\n\nInlined by the compiler.","ref":"Kernel.html#++/2"},{"type":"function","title":"Examples - Kernel.++/2","doc":"iex> [1] ++ [2, 3]\n    [1, 2, 3]\n\n    iex> ~c\"foo\" ++ ~c\"bar\"\n    ~c\"foobar\"\n\n    # a non-list on the right will return an improper list\n    # with said element at the end\n    iex> [1, 2] ++ 3\n    [1, 2 | 3]\n    iex> [1, 2] ++ {3, 4}\n    [1, 2 | {3, 4}]\n\n    # improper list on the right will return an improper list\n    iex> [1] ++ [2 | 3]\n    [1, 2 | 3]\n\nThe `++/2` operator is right associative, meaning:\n\n    iex> [1, 2, 3] -- [1] ++ [2]\n    [3]\n\nAs it is equivalent to:\n\n    iex> [1, 2, 3] -- ([1] ++ [2])\n    [3]","ref":"Kernel.html#++/2-examples"},{"type":"function","title":"Kernel.+/1","doc":"Arithmetic positive unary operator.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#+/1"},{"type":"function","title":"Examples - Kernel.+/1","doc":"iex> +1\n    1","ref":"Kernel.html#+/1-examples"},{"type":"function","title":"Kernel.+/2","doc":"Arithmetic addition operator.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#+/2"},{"type":"function","title":"Examples - Kernel.+/2","doc":"iex> 1 + 2\n    3","ref":"Kernel.html#+/2-examples"},{"type":"function","title":"Kernel.--/2","doc":"List subtraction operator. Removes the first occurrence of an element\non the left list for each element on the right.\n\nThis function is optimized so the complexity of `a -- b` is proportional\nto `length(a) * log(length(b))`. See also the [Erlang efficiency\nguide](https://www.erlang.org/doc/efficiency_guide/retired_myths.html).\n\nInlined by the compiler.","ref":"Kernel.html#--/2"},{"type":"function","title":"Examples - Kernel.--/2","doc":"iex> [1, 2, 3] -- [1, 2]\n    [3]\n\n    iex> [1, 2, 3, 2, 1] -- [1, 2, 2]\n    [3, 1]\n\nThe `--/2` operator is right associative, meaning:\n\n    iex> [1, 2, 3] -- [2] -- [3]\n    [1, 3]\n\nAs it is equivalent to:\n\n    iex> [1, 2, 3] -- ([2] -- [3])\n    [1, 3]","ref":"Kernel.html#--/2-examples"},{"type":"function","title":"Kernel.-/1","doc":"Arithmetic negative unary operator.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#-/1"},{"type":"function","title":"Examples - Kernel.-/1","doc":"iex> -2\n    -2","ref":"Kernel.html#-/1-examples"},{"type":"function","title":"Kernel.-/2","doc":"Arithmetic subtraction operator.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#-/2"},{"type":"function","title":"Examples - Kernel.-/2","doc":"iex> 1 - 2\n    -1","ref":"Kernel.html#-/2-examples"},{"type":"macro","title":"Kernel.../0","doc":"Creates the full-slice range `0..-1//1`.\n\nThis macro returns a range with the following properties:\n\n  * When enumerated, it is empty\n\n  * When used as a `slice`, it returns the sliced element as is\n\nSee `..///3` and the `Range` module for more information.","ref":"Kernel.html#../0"},{"type":"macro","title":"Examples - Kernel.../0","doc":"iex> Enum.to_list(..)\n    []\n\n    iex> String.slice(\"Hello world!\", ..)\n    \"Hello world!\"","ref":"Kernel.html#../0-examples"},{"type":"macro","title":"Kernel.../2","doc":"Creates a range from `first` to `last`.\n\nIf first is less than last, the range will be increasing from\nfirst to last. If first is equal to last, the range will contain\none element, which is the number itself.\n\nIf first is more than last, the range will be decreasing from first\nto last, albeit this behaviour is deprecated. Instead prefer to\nexplicitly list the step with `first..last//-1`.\n\nSee the `Range` module for more information.","ref":"Kernel.html#../2"},{"type":"macro","title":"Examples - Kernel.../2","doc":"iex> 0 in 1..3\n    false\n    iex> 2 in 1..3\n    true\n\n    iex> Enum.to_list(1..3)\n    [1, 2, 3]","ref":"Kernel.html#../2-examples"},{"type":"macro","title":"Kernel...///3","doc":"Creates a range from `first` to `last` with `step`.\n\nSee the `Range` module for more information.","ref":"Kernel.html#..///3"},{"type":"macro","title":"Examples - Kernel...///3","doc":"iex> 0 in 1..3//1\n    false\n    iex> 2 in 1..3//1\n    true\n    iex> 2 in 1..3//2\n    false\n\n    iex> Enum.to_list(1..3//1)\n    [1, 2, 3]\n    iex> Enum.to_list(1..3//2)\n    [1, 3]\n    iex> Enum.to_list(3..1//-1)\n    [3, 2, 1]\n    iex> Enum.to_list(1..0//1)\n    []","ref":"Kernel.html#..///3-examples"},{"type":"function","title":"Kernel.//2","doc":"Arithmetic division operator.\n\nThe result is always a float. Use `div/2` and `rem/2` if you want\nan integer division or the remainder.\n\nRaises `ArithmeticError` if `right` is 0 or 0.0.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#//2"},{"type":"function","title":"Examples - Kernel.//2","doc":"1 / 2\n    #=> 0.5\n\n    -3.0 / 2.0\n    #=> -1.5\n\n    5 / 1\n    #=> 5.0\n\n    7 / 0\n    ** (ArithmeticError) bad argument in arithmetic expression","ref":"Kernel.html#//2-examples"},{"type":"macro","title":"Kernel.!/1","doc":"Boolean \"not\" operator.\n\nReceives any value (not just booleans) and returns `true` if `value`\nis `false` or `nil`; returns `false` otherwise.\n\nNot allowed in guard clauses.","ref":"Kernel.html#!/1"},{"type":"macro","title":"Examples - Kernel.!/1","doc":"iex> !Enum.empty?([])\n    false\n\n    iex> !List.first([])\n    true","ref":"Kernel.html#!/1-examples"},{"type":"function","title":"Kernel.!=/2","doc":"Not equal to operator.\n\nReturns `true` if the two terms are not equal.\n\nThis operator considers 1 and 1.0 to be equal. For match\ncomparison, use `!==/2` instead.\n\nThis performs a structural comparison where all Elixir\nterms can be compared with each other. See the [\"Structural\ncomparison\" section](#module-structural-comparison) section\nfor more information.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#!=/2"},{"type":"function","title":"Examples - Kernel.!=/2","doc":"iex> 1 != 2\n    true\n\n    iex> 1 != 1.0\n    false","ref":"Kernel.html#!=/2-examples"},{"type":"function","title":"Kernel.!==/2","doc":"Strictly not equal to operator.\n\nReturns `true` if the two terms are not exactly equal.\nSee `===/2` for a definition of what is considered \"exactly equal\".\n\nThis performs a structural comparison where all Elixir\nterms can be compared with each other. See the [\"Structural\ncomparison\" section](#module-structural-comparison) section\nfor more information.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#!==/2"},{"type":"function","title":"Examples - Kernel.!==/2","doc":"iex> 1 !== 2\n    true\n\n    iex> 1 !== 1.0\n    true","ref":"Kernel.html#!==/2-examples"},{"type":"function","title":"Kernel.</2","doc":"Less-than operator.\n\nReturns `true` if `left` is less than `right`.\n\nThis performs a structural comparison where all Elixir\nterms can be compared with each other. See the [\"Structural\ncomparison\" section](#module-structural-comparison) section\nfor more information.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#%3C/2"},{"type":"function","title":"Examples - Kernel.</2","doc":"iex> 1 < 2\n    true","ref":"Kernel.html#%3C/2-examples"},{"type":"function","title":"Kernel.<=/2","doc":"Less-than or equal to operator.\n\nReturns `true` if `left` is less than or equal to `right`.\n\nThis performs a structural comparison where all Elixir\nterms can be compared with each other. See the [\"Structural\ncomparison\" section](#module-structural-comparison) section\nfor more information.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#%3C=/2"},{"type":"function","title":"Examples - Kernel.<=/2","doc":"iex> 1 <= 2\n    true","ref":"Kernel.html#%3C=/2-examples"},{"type":"macro","title":"Kernel.<>/2","doc":"Binary concatenation operator. Concatenates two binaries.\n\nRaises an `ArgumentError` if one of the sides aren't binaries.","ref":"Kernel.html#%3C%3E/2"},{"type":"macro","title":"Examples - Kernel.<>/2","doc":"iex> \"foo\"   \"bar\"\n    \"foobar\"\n\nThe ` /2` operator can also be used in pattern matching (and guard clauses) as\nlong as the left argument is a literal binary:\n\n    iex> \"foo\"   x = \"foobar\"\n    iex> x\n    \"bar\"\n\n`x   \"bar\" = \"foobar\"` would result in an `ArgumentError` exception.","ref":"Kernel.html#%3C%3E/2-examples"},{"type":"function","title":"Kernel.==/2","doc":"Equal to operator. Returns `true` if the two terms are equal.\n\nThis operator considers 1 and 1.0 to be equal. For stricter\nsemantics, use `===/2` instead.\n\nThis performs a structural comparison where all Elixir\nterms can be compared with each other. See the [\"Structural\ncomparison\" section](#module-structural-comparison) section\nfor more information.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#==/2"},{"type":"function","title":"Examples - Kernel.==/2","doc":"iex> 1 == 2\n    false\n\n    iex> 1 == 1.0\n    true","ref":"Kernel.html#==/2-examples"},{"type":"function","title":"Kernel.===/2","doc":"Strictly equal to operator.\n\nReturns `true` if the two terms are exactly equal.\n\nThe terms are only considered to be exactly equal if they\nhave the same value and are of the same type. For example,\n`1 == 1.0` returns `true`, but since they are of different\ntypes, `1 === 1.0` returns `false`.\n\nThis performs a structural comparison where all Elixir\nterms can be compared with each other. See the [\"Structural\ncomparison\" section](#module-structural-comparison) section\nfor more information.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#===/2"},{"type":"function","title":"Examples - Kernel.===/2","doc":"iex> 1 === 2\n    false\n\n    iex> 1 === 1.0\n    false","ref":"Kernel.html#===/2-examples"},{"type":"function","title":"Kernel.=~/2","doc":"Text-based match operator. Matches the term on the `left`\nagainst the regular expression or string on the `right`.\n\nIf `right` is a regular expression, returns `true` if `left` matches right.\n\nIf `right` is a string, returns `true` if `left` contains `right`.","ref":"Kernel.html#=~/2"},{"type":"function","title":"Examples - Kernel.=~/2","doc":"iex> \"abcd\" =~ ~r/c(d)/\n    true\n\n    iex> \"abcd\" =~ ~r/e/\n    false\n\n    iex> \"abcd\" =~ ~r//\n    true\n\n    iex> \"abcd\" =~ \"bc\"\n    true\n\n    iex> \"abcd\" =~ \"ad\"\n    false\n\n    iex> \"abcd\" =~ \"abcd\"\n    true\n\n    iex> \"abcd\" =~ \"\"\n    true\n\nFor more information about regular expressions, please check the `Regex` module.","ref":"Kernel.html#=~/2-examples"},{"type":"function","title":"Kernel.>/2","doc":"Greater-than operator.\n\nReturns `true` if `left` is more than `right`.\n\nThis performs a structural comparison where all Elixir\nterms can be compared with each other. See the [\"Structural\ncomparison\" section](#module-structural-comparison) section\nfor more information.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#%3E/2"},{"type":"function","title":"Examples - Kernel.>/2","doc":"iex> 1 > 2\n    false","ref":"Kernel.html#%3E/2-examples"},{"type":"function","title":"Kernel.>=/2","doc":"Greater-than or equal to operator.\n\nReturns `true` if `left` is more than or equal to `right`.\n\nThis performs a structural comparison where all Elixir\nterms can be compared with each other. See the [\"Structural\ncomparison\" section](#module-structural-comparison) section\nfor more information.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#%3E=/2"},{"type":"function","title":"Examples - Kernel.>=/2","doc":"iex> 1 >= 2\n    false","ref":"Kernel.html#%3E=/2-examples"},{"type":"macro","title":"Kernel.@/1","doc":"Module attribute unary operator.\n\nReads and writes attributes in the current module.\n\nThe canonical example for attributes is annotating that a module\nimplements an OTP behaviour, such as `GenServer`:\n\n    defmodule MyServer do\n      @behaviour GenServer\n      # ... callbacks ...\n    end\n\nBy default Elixir supports all the module attributes supported by Erlang, but\ncustom attributes can be used as well:\n\n    defmodule MyServer do\n      @my_data 13\n      IO.inspect(@my_data)\n      #=> 13\n    end\n\nUnlike Erlang, such attributes are not stored in the module by default since\nit is common in Elixir to use custom attributes to store temporary data that\nwill be available at compile-time. Custom attributes may be configured to\nbehave closer to Erlang by using `Module.register_attribute/3`.\n\n> ##","ref":"Kernel.html#@/1"},{"type":"macro","title":"Prefixing module attributes {: .tip} - Kernel.@/1","doc":">\n> Libraries and frameworks should consider prefixing any\n> module attributes that are private by underscore, such as `@_my_data`,\n> so code completion tools do not show them on suggestions and prompts.\n\nFinally, note that attributes can also be read inside functions:\n\n    defmodule MyServer do\n      @my_data 11\n      def first_data, do: @my_data\n      @my_data 13\n      def second_data, do: @my_data\n    end\n\n    MyServer.first_data()\n    #=> 11\n\n    MyServer.second_data()\n    #=> 13\n\nIt is important to note that reading an attribute takes a snapshot of\nits current value. In other words, the value is read at compilation\ntime and not at runtime. Check the `Module` module for other functions\nto manipulate module attributes.","ref":"Kernel.html#@/1-prefixing-module-attributes-tip"},{"type":"macro","title":"Attention! Multiple references of the same attribute - Kernel.@/1","doc":"As mentioned above, every time you read a module attribute, a snapshot\nof its current value is taken. Therefore, if you are storing large\nvalues inside module attributes (for example, embedding external files\nin module attributes), you should avoid referencing the same attribute\nmultiple times. For example, don't do this:\n\n    @files %{\n      example1: File.read!(\"lib/example1.data\"),\n      example2: File.read!(\"lib/example2.data\")\n    }\n\n    def example1, do: @files[:example1]\n    def example2, do: @files[:example2]\n\nIn the above, each reference to `@files` may end-up with a complete\nand individual copy of the whole `@files` module attribute. Instead,\nreference the module attribute once in a private function:\n\n    @files %{\n      example1: File.read!(\"lib/example1.data\"),\n      example2: File.read!(\"lib/example2.data\")\n    }\n\n    defp files(), do: @files\n    def example1, do: files()[:example1]\n    def example2, do: files()[:example2]","ref":"Kernel.html#@/1-attention-multiple-references-of-the-same-attribute"},{"type":"macro","title":"Attention! Compile-time dependencies - Kernel.@/1","doc":"Keep in mind references to other modules, even in module attributes,\ngenerate compile-time dependencies to said modules.\n\nFor example, take this common pattern:\n\n    @values [:foo, :bar, :baz]\n\n    def handle_arg(arg) when arg in @values do\n      ...\n    end\n\nWhile the above is fine, imagine if instead you have actual\nmodule names in the module attribute, like this:\n\n    @values [Foo, Bar, Baz]\n\n    def handle_arg(arg) when arg in @values do\n      ...\n    end\n\nThe code above will define a compile-time dependency on the modules\n`Foo`, `Bar`, and `Baz`, in a way that, if any of them change, the\ncurrent module will have to recompile. In such cases, it may be\npreferred to avoid the module attribute altogether:\n\n    def handle_arg(arg) when arg in [Foo, Bar, Baz] do\n      ...\n    end","ref":"Kernel.html#@/1-attention-compile-time-dependencies"},{"type":"function","title":"Kernel.abs/1","doc":"Returns an integer or float which is the arithmetical absolute value of `number`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#abs/1"},{"type":"function","title":"Examples - Kernel.abs/1","doc":"iex> abs(-3.33)\n    3.33\n\n    iex> abs(-3)\n    3","ref":"Kernel.html#abs/1-examples"},{"type":"macro","title":"Kernel.alias!/1","doc":"When used inside quoting, marks that the given alias should not\nbe hygienized. This means the alias will be expanded when\nthe macro is expanded.\n\nCheck `Kernel.SpecialForms.quote/2` for more information.","ref":"Kernel.html#alias!/1"},{"type":"macro","title":"Kernel.and/2","doc":"Strictly boolean \"and\" operator.\n\nIf `left` is `false`, returns `false`; otherwise returns `right`.\n\nRequires only the `left` operand to be a boolean since it short-circuits. If\nthe `left` operand is not a boolean, a `BadBooleanError` exception is raised.\n\nAllowed in guard tests.","ref":"Kernel.html#and/2"},{"type":"macro","title":"Examples - Kernel.and/2","doc":"iex> true and false\n    false\n\n    iex> true and \"yay!\"\n    \"yay!\"\n\n    iex> \"yay!\" and true\n    ** (BadBooleanError) expected a boolean on left-side of \"and\", got: \"yay!\"","ref":"Kernel.html#and/2-examples"},{"type":"function","title":"Kernel.apply/2","doc":"Invokes the given anonymous function `fun` with the list of\narguments `args`.\n\nIf the number of arguments is known at compile time, prefer\n`fun.(arg_1, arg_2, ..., arg_n)` as it is clearer than\n`apply(fun, [arg_1, arg_2, ..., arg_n])`.\n\nInlined by the compiler.","ref":"Kernel.html#apply/2"},{"type":"function","title":"Examples - Kernel.apply/2","doc":"iex> apply(fn x -> x * 2 end, [2])\n    4","ref":"Kernel.html#apply/2-examples"},{"type":"function","title":"Kernel.apply/3","doc":"Invokes the given function from `module` with the list of\narguments `args`.\n\n`apply/3` is used to invoke functions where the module, function\nname or arguments are defined dynamically at runtime. For this\nreason, you can't invoke macros using `apply/3`, only functions.\n\nIf the number of arguments and the function name are known at compile time,\nprefer `module.function(arg_1, arg_2, ..., arg_n)` as it is clearer than\n`apply(module, :function, [arg_1, arg_2, ..., arg_n])`.\n\n`apply/3` cannot be used to call private functions.\n\nInlined by the compiler.","ref":"Kernel.html#apply/3"},{"type":"function","title":"Examples - Kernel.apply/3","doc":"iex> apply(Enum, :reverse, [[1, 2, 3]])\n    [3, 2, 1]","ref":"Kernel.html#apply/3-examples"},{"type":"function","title":"Kernel.binary_part/3","doc":"Extracts the part of the binary starting at `start` with length `length`.\nBinaries are zero-indexed.\n\nIf `start` or `length` reference in any way outside the binary, an\n`ArgumentError` exception is raised.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#binary_part/3"},{"type":"function","title":"Examples - Kernel.binary_part/3","doc":"iex> binary_part(\"foo\", 1, 2)\n    \"oo\"\n\nA negative `length` can be used to extract bytes that come *before* the byte\nat `start`:\n\n    iex> binary_part(\"Hello\", 5, -3)\n    \"llo\"\n\nAn `ArgumentError` is raised when the length is outside of the binary:\n\n    binary_part(\"Hello\", 0, 10)\n    ** (ArgumentError) argument error","ref":"Kernel.html#binary_part/3-examples"},{"type":"function","title":"Kernel.binary_slice/2","doc":"Returns a binary from the offset given by the start of the\nrange to the offset given by the end of the range.\n\nIf the start or end of the range are negative, they are converted\ninto positive indices based on the binary size. For example,\n`-1` means the last byte of the binary.\n\nThis is similar to `binary_part/3` except that it works with ranges\nand it is not allowed in guards.\n\nThis function works with bytes. For a slicing operation that\nconsiders characters, see `String.slice/2`.","ref":"Kernel.html#binary_slice/2"},{"type":"function","title":"Examples - Kernel.binary_slice/2","doc":"iex> binary_slice(\"elixir\", 0..5)\n    \"elixir\"\n    iex> binary_slice(\"elixir\", 1..3)\n    \"lix\"\n    iex> binary_slice(\"elixir\", 1..10)\n    \"lixir\"\n\n    iex> binary_slice(\"elixir\", -4..-1)\n    \"ixir\"\n    iex> binary_slice(\"elixir\", -4..6)\n    \"ixir\"\n    iex> binary_slice(\"elixir\", -10..10)\n    \"elixir\"\n\nFor ranges where `start > stop`, you need to explicitly\nmark them as increasing:\n\n    iex> binary_slice(\"elixir\", 2..-1//1)\n    \"ixir\"\n    iex> binary_slice(\"elixir\", 1..-2//1)\n    \"lixi\"\n\nYou can use `../0` as a shortcut for `0..-1//1`, which returns\nthe whole binary as is:\n\n    iex> binary_slice(\"elixir\", ..)\n    \"elixir\"\n\nThe step can be any positive number. For example, to\nget every 2 characters of the binary:\n\n    iex> binary_slice(\"elixir\", 0..-1//2)\n    \"eii\"\n\nIf the first position is after the string ends or after\nthe last position of the range, it returns an empty string:\n\n    iex> binary_slice(\"elixir\", 10..3//1)\n    \"\"\n    iex> binary_slice(\"elixir\", -10..-7)\n    \"\"\n    iex> binary_slice(\"a\", 1..1500)\n    \"\"","ref":"Kernel.html#binary_slice/2-examples"},{"type":"function","title":"Kernel.binary_slice/3","doc":"Returns a binary starting at the offset `start` and of the given `size`.\n\nThis is similar to `binary_part/3` except that if `start + size`\nis greater than the binary size, it automatically clips it to\nthe binary size instead of raising. Opposite to `binary_part/3`,\nthis function is not allowed in guards.\n\nThis function works with bytes. For a slicing operation that\nconsiders characters, see `String.slice/3`.","ref":"Kernel.html#binary_slice/3"},{"type":"function","title":"Examples - Kernel.binary_slice/3","doc":"iex> binary_slice(\"elixir\", 0, 6)\n    \"elixir\"\n    iex> binary_slice(\"elixir\", 0, 5)\n    \"elixi\"\n    iex> binary_slice(\"elixir\", 1, 4)\n    \"lixi\"\n    iex> binary_slice(\"elixir\", 0, 10)\n    \"elixir\"\n\nIf `start` is negative, it is normalized against the binary\nsize and clamped to 0:\n\n    iex> binary_slice(\"elixir\", -3, 10)\n    \"xir\"\n    iex> binary_slice(\"elixir\", -10, 10)\n    \"elixir\"\n\nIf the `size` is zero, an empty binary is returned:\n\n    iex> binary_slice(\"elixir\", 1, 0)\n    \"\"\n\nIf `start` is greater than or equal to the binary size,\nan empty binary is returned:\n\n    iex> binary_slice(\"elixir\", 10, 10)\n    \"\"","ref":"Kernel.html#binary_slice/3-examples"},{"type":"macro","title":"Kernel.binding/1","doc":"Returns the binding for the given context as a keyword list.\n\nIn the returned result, keys are variable names and values are the\ncorresponding variable values.\n\nIf the given `context` is `nil` (by default it is), the binding for the\ncurrent context is returned.","ref":"Kernel.html#binding/1"},{"type":"macro","title":"Examples - Kernel.binding/1","doc":"iex> x = 1\n    iex> binding()\n    [x: 1]\n    iex> x = 2\n    iex> binding()\n    [x: 2]\n\n    iex> binding(:foo)\n    []\n    iex> var!(x, :foo) = 1\n    1\n    iex> binding(:foo)\n    [x: 1]","ref":"Kernel.html#binding/1-examples"},{"type":"function","title":"Kernel.bit_size/1","doc":"Returns an integer which is the size in bits of `bitstring`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#bit_size/1"},{"type":"function","title":"Examples - Kernel.bit_size/1","doc":"iex> bit_size( >)\n    19\n\n    iex> bit_size( >)\n    24","ref":"Kernel.html#bit_size/1-examples"},{"type":"function","title":"Kernel.byte_size/1","doc":"Returns the number of bytes needed to contain `bitstring`.\n\nThat is, if the number of bits in `bitstring` is not divisible by 8, the\nresulting number of bytes will be rounded up (by excess). This operation\nhappens in constant time.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#byte_size/1"},{"type":"function","title":"Examples - Kernel.byte_size/1","doc":"iex> byte_size( >)\n    3\n\n    iex> byte_size( >)\n    3","ref":"Kernel.html#byte_size/1-examples"},{"type":"function","title":"Kernel.ceil/1","doc":"Returns the smallest integer greater than or equal to `number`.\n\nIf you want to perform ceil operation on other decimal places,\nuse `Float.ceil/2` instead.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#ceil/1"},{"type":"function","title":"Examples - Kernel.ceil/1","doc":"iex> ceil(10)\n    10\n\n    iex> ceil(10.1)\n    11\n\n    iex> ceil(-10.1)\n    -10","ref":"Kernel.html#ceil/1-examples"},{"type":"macro","title":"Kernel.dbg/2","doc":"Debugs the given `code`.\n\n`dbg/2` can be used to debug the given `code` through a configurable debug function.\nIt returns the result of the given code.","ref":"Kernel.html#dbg/2"},{"type":"macro","title":"Examples - Kernel.dbg/2","doc":"Let's take this call to `dbg/2`:\n\n    dbg(Atom.to_string(:debugging))\n    #=> \"debugging\"\n\nIt returns the string `\"debugging\"`, which is the result of the `Atom.to_string/1` call.\nAdditionally, the call above prints:\n\n    [my_file.ex:10: MyMod.my_fun/0]\n    Atom.to_string(:debugging) #=> \"debugging\"\n\nThe default debugging function prints additional debugging info when dealing with\npipelines. It prints the values at every \"step\" of the pipeline.\n\n    \"Elixir is cool!\"\n    |> String.trim_trailing(\"!\")\n    |> String.split()\n    |> List.first()\n    |> dbg()\n    #=> \"Elixir\"\n\nThe code above prints:\n\n    [my_file.ex:10: MyMod.my_fun/0]\n    \"Elixir is cool!\" #=> \"Elixir is cool!\"\n    |> String.trim_trailing(\"!\") #=> \"Elixir is cool\"\n    |> String.split() #=> [\"Elixir\", \"is\", \"cool\"]\n    |> List.first() #=> \"Elixir\"\n\nWith no arguments, `dbg()` debugs information about the current binding. See `binding/1`.\n\n## `dbg` inside IEx\n\nYou can enable IEx to replace `dbg` with its `IEx.pry/0` backend by calling:\n\n    $ iex --dbg pry\n\nIn such cases, `dbg` will start a `pry` session where you can interact with\nthe imports, aliases, and variables of the current environment at the location\nof the `dbg` call.\n\nIf you call `dbg` at the end of a pipeline (using `|>`) within IEx, you are able\nto go through each step of the pipeline one by one by entering \"next\" (or \"n\").\n\nNote `dbg` only supports stepping for pipelines (in other words, it can only\nstep through the code it sees). For general stepping, you can set breakpoints\nusing `IEx.break!/4`.\n\nFor more information, [see IEx documentation](https://hexdocs.pm/iex/IEx.html#module-dbg-and-breakpoints).","ref":"Kernel.html#dbg/2-examples"},{"type":"macro","title":"Configuring the debug function - Kernel.dbg/2","doc":"One of the benefits of `dbg/2` is that its debugging logic is configurable,\nallowing tools to extend `dbg` with enhanced behaviour. This is done, for\nexample, by `IEx` which extends `dbg` with an interactive shell where you\ncan directly inspect and access values.\n\nThe debug function can be configured at compile time through the `:dbg_callback`\nkey of the `:elixir` application. The debug function must be a\n`{module, function, args}` tuple. The `function` function in `module` will be\ninvoked with three arguments *prepended* to `args`:\n\n  1. The AST of `code`\n  2. The AST of `options`\n  3. The `Macro.Env` environment of where `dbg/2` is invoked\n\nThe debug function is invoked at compile time and it must also return an AST.\nThe AST is expected to ultimately return the result of evaluating the debugged\nexpression.\n\nHere's a simple example:\n\n    defmodule MyMod do\n      def debug_fun(code, options, caller, device) do\n        quote do\n          result = unquote(code)\n          IO.inspect(unquote(device), result, label: unquote(Macro.to_string(code)))\n        end\n      end\n    end\n\nTo configure the debug function:\n\n    # In config/config.exs\n    config :elixir, :dbg_callback, {MyMod, :debug_fun, [:stdio]}\n\n#","ref":"Kernel.html#dbg/2-configuring-the-debug-function"},{"type":"macro","title":"Default debug function - Kernel.dbg/2","doc":"By default, the debug function we use is `Macro.dbg/3`. It just prints\ninformation about the code to standard output and returns the value\nreturned by evaluating `code`. `options` are used to control how terms\nare inspected. They are the same options accepted by `inspect/2`.","ref":"Kernel.html#dbg/2-default-debug-function"},{"type":"macro","title":"Kernel.def/2","doc":"Defines a public function with the given name and body.","ref":"Kernel.html#def/2"},{"type":"macro","title":"Examples - Kernel.def/2","doc":"defmodule Foo do\n      def bar, do: :baz\n    end\n\n    Foo.bar()\n    #=> :baz\n\nA function that expects arguments can be defined as follows:\n\n    defmodule Foo do\n      def sum(a, b) do\n        a + b\n      end\n    end\n\nIn the example above, a `sum/2` function is defined; this function receives\ntwo arguments and returns their sum.","ref":"Kernel.html#def/2-examples"},{"type":"macro","title":"Default arguments - Kernel.def/2","doc":"`\\\\` is used to specify a default value for a parameter of a function. For\nexample:\n\n    defmodule MyMath do\n      def multiply_by(number, factor \\\\ 2) do\n        number * factor\n      end\n    end\n\n    MyMath.multiply_by(4, 3)\n    #=> 12\n\n    MyMath.multiply_by(4)\n    #=> 8\n\nThe compiler translates this into multiple functions with different arities,\nhere `MyMath.multiply_by/1` and `MyMath.multiply_by/2`, that represent cases when\narguments for parameters with default values are passed or not passed.\n\nWhen defining a function with default arguments as well as multiple\nexplicitly declared clauses, you must write a function head that declares the\ndefaults. For example:\n\n    defmodule MyString do\n      def join(string1, string2 \\\\ nil, separator \\\\ \" \")\n\n      def join(string1, nil, _separator) do\n        string1\n      end\n\n      def join(string1, string2, separator) do\n        string1   separator   string2\n      end\n    end\n\nNote that `\\\\` can't be used with anonymous functions because they\ncan only have a sole arity.\n\n#","ref":"Kernel.html#def/2-default-arguments"},{"type":"macro","title":"Keyword lists with default arguments - Kernel.def/2","doc":"Functions containing many arguments can benefit from using `Keyword`\nlists to group and pass attributes as a single value.\n\n    defmodule MyConfiguration do\n      @default_opts [storage: \"local\"]\n\n      def configure(resource, opts \\\\ []) do\n        opts = Keyword.merge(@default_opts, opts)\n        storage = opts[:storage]\n        # ...\n      end\n    end\n\nThe difference between using `Map` and `Keyword` to store many\narguments is `Keyword`'s keys:\n\n  * must be atoms\n  * can be given more than once\n  * ordered, as specified by the developer","ref":"Kernel.html#def/2-keyword-lists-with-default-arguments"},{"type":"macro","title":"Function and variable names - Kernel.def/2","doc":"Function and variable names have the following syntax:\nA _lowercase ASCII letter_ or an _underscore_, followed by any number of\n_lowercase or uppercase ASCII letters_, _numbers_, or _underscores_.\nOptionally they can end in either an _exclamation mark_ or a _question mark_.\n\nFor variables, any identifier starting with an underscore should indicate an\nunused variable. For example:\n\n    def foo(bar) do\n      []\n    end\n    #=> warning: variable bar is unused\n\n    def foo(_bar) do\n      []\n    end\n    #=> no warning\n\n    def foo(_bar) do\n      _bar\n    end\n    #=> warning: the underscored variable \"_bar\" is used after being set\n\n## `rescue`/`catch`/`after`/`else`\n\nFunction bodies support `rescue`, `catch`, `after`, and `else` as `Kernel.SpecialForms.try/1`\ndoes (known as \"implicit try\"). For example, the following two functions are equivalent:\n\n    def convert(number) do\n      try do\n        String.to_integer(number)\n      rescue\n        e in ArgumentError -> {:error, e.message}\n      end\n    end\n\n    def convert(number) do\n      String.to_integer(number)\n    rescue\n      e in ArgumentError -> {:error, e.message}\n    end","ref":"Kernel.html#def/2-function-and-variable-names"},{"type":"macro","title":"Kernel.defdelegate/2","doc":"Defines a function that delegates to another module.\n\nFunctions defined with `defdelegate/2` are public and can be invoked from\noutside the module they're defined in, as if they were defined using `def/2`.\nTherefore, `defdelegate/2` is about extending the current module's public API.\nIf what you want is to invoke a function defined in another module without\nusing its full module name, then use `alias/2` to shorten the module name or use\n`import/2` to be able to invoke the function without the module name altogether.\n\nDelegation only works with functions; delegating macros is not supported.\n\nCheck `def/2` for rules on naming and default arguments.","ref":"Kernel.html#defdelegate/2"},{"type":"macro","title":"Options - Kernel.defdelegate/2","doc":"* `:to` - the module to dispatch to.\n\n  * `:as` - the function to call on the target given in `:to`.\n    This parameter is optional and defaults to the name being\n    delegated (`funs`).","ref":"Kernel.html#defdelegate/2-options"},{"type":"macro","title":"Examples - Kernel.defdelegate/2","doc":"defmodule MyList do\n      defdelegate reverse(list), to: Enum\n      defdelegate other_reverse(list), to: Enum, as: :reverse\n    end\n\n    MyList.reverse([1, 2, 3])\n    #=> [3, 2, 1]\n\n    MyList.other_reverse([1, 2, 3])\n    #=> [3, 2, 1]","ref":"Kernel.html#defdelegate/2-examples"},{"type":"macro","title":"Kernel.defexception/1","doc":"Defines an exception.\n\nExceptions are structs backed by a module that implements\nthe `Exception` behaviour. The `Exception` behaviour requires\ntwo functions to be implemented:\n\n  * [`exception/1`](`c:Exception.exception/1`) - receives the arguments given to `raise/2`\n    and returns the exception struct. The default implementation\n    accepts either a set of keyword arguments that is merged into\n    the struct or a string to be used as the exception's message.\n\n  * [`message/1`](`c:Exception.message/1`) - receives the exception struct and must return its\n    message. Most commonly exceptions have a message field which\n    by default is accessed by this function. However, if an exception\n    does not have a message field, this function must be explicitly\n    implemented.\n\nSince exceptions are structs, the API supported by `defstruct/1`\nis also available in `defexception/1`.","ref":"Kernel.html#defexception/1"},{"type":"macro","title":"Raising exceptions - Kernel.defexception/1","doc":"The most common way to raise an exception is via `raise/2`:\n\n    defmodule MyAppError do\n      defexception [:message]\n    end\n\n    value = [:hello]\n\n    raise MyAppError,\n      message: \"did not get what was expected, got: #{inspect(value)}\"\n\nIn many cases it is more convenient to pass the expected value to\n`raise/2` and generate the message in the `c:Exception.exception/1` callback:\n\n    defmodule MyAppError do\n      defexception [:message]\n\n      @impl true\n      def exception(value) do\n        msg = \"did not get what was expected, got: #{inspect(value)}\"\n        %MyAppError{message: msg}\n      end\n    end\n\n    raise MyAppError, value\n\nThe example above shows the preferred strategy for customizing\nexception messages.","ref":"Kernel.html#defexception/1-raising-exceptions"},{"type":"macro","title":"Kernel.defguard/1","doc":"Generates a macro suitable for use in guard expressions.\n\nIt raises at compile time if the definition uses expressions that aren't\nallowed in guards, and otherwise creates a macro that can be used both inside\nor outside guards.\n\nNote the convention in Elixir is to prefix all guards that return a boolean\nwith the `is_` prefix, such as `is_list/1`. If, however, the function/macro\nreturns a boolean and is not allowed in guards, it should have no prefix and\nend with a question mark, such as `Keyword.keyword?/1`.","ref":"Kernel.html#defguard/1"},{"type":"macro","title":"Example - Kernel.defguard/1","doc":"defmodule Integer.Guards do\n      defguard is_even(value) when is_integer(value) and rem(value, 2) == 0\n    end\n\n    defmodule Collatz do\n      @moduledoc \"Tools for working with the Collatz sequence.\"\n      import Integer.Guards\n\n      @doc \"Determines the number of steps `n` takes to reach `1`.\"\n      # If this function never converges, please let me know what `n` you used.\n      def converge(n) when n > 0, do: step(n, 0)\n\n      defp step(1, step_count) do\n        step_count\n      end\n\n      defp step(n, step_count) when is_even(n) do\n        step(div(n, 2), step_count + 1)\n      end\n\n      defp step(n, step_count) do\n        step(3 * n + 1, step_count + 1)\n      end\n    end","ref":"Kernel.html#defguard/1-example"},{"type":"macro","title":"Kernel.defguardp/1","doc":"Generates a private macro suitable for use in guard expressions.\n\nIt raises at compile time if the definition uses expressions that aren't\nallowed in guards, and otherwise creates a private macro that can be used\nboth inside or outside guards in the current module.\n\nSimilar to `defmacrop/2`, `defguardp/1` must be defined before its use\nin the current module.","ref":"Kernel.html#defguardp/1"},{"type":"macro","title":"Kernel.defimpl/3","doc":"Defines an implementation for the given protocol.\n\nSee the `Protocol` module for more information.","ref":"Kernel.html#defimpl/3"},{"type":"macro","title":"Kernel.defmacro/2","doc":"Defines a public macro with the given name and body.\n\nMacros must be defined before its usage.\n\nCheck `def/2` for rules on naming and default arguments.","ref":"Kernel.html#defmacro/2"},{"type":"macro","title":"Examples - Kernel.defmacro/2","doc":"defmodule MyLogic do\n      defmacro unless(expr, opts) do\n        quote do\n          if !unquote(expr), unquote(opts)\n        end\n      end\n    end\n\n    require MyLogic\n\n    MyLogic.unless false do\n      IO.puts(\"It works\")\n    end","ref":"Kernel.html#defmacro/2-examples"},{"type":"macro","title":"Kernel.defmacrop/2","doc":"Defines a private macro with the given name and body.\n\nPrivate macros are only accessible from the same module in which they are\ndefined.\n\nPrivate macros must be defined before its usage.\n\nCheck `defmacro/2` for more information, and check `def/2` for rules on\nnaming and default arguments.","ref":"Kernel.html#defmacrop/2"},{"type":"macro","title":"Kernel.defmodule/2","doc":"Defines a module given by name with the given contents.\n\nThis macro defines a module with the given `alias` as its name and with the\ngiven contents. It returns a tuple with four elements:\n\n  * `:module`\n  * the module name\n  * the binary contents of the module\n  * the result of evaluating the contents block","ref":"Kernel.html#defmodule/2"},{"type":"macro","title":"Examples - Kernel.defmodule/2","doc":"defmodule Number do\n      def one, do: 1\n      def two, do: 2\n    end\n    #=> {:module, Number,  >, {:two, 0}}\n\n    Number.one()\n    #=> 1\n\n    Number.two()\n    #=> 2","ref":"Kernel.html#defmodule/2-examples"},{"type":"macro","title":"Nesting - Kernel.defmodule/2","doc":"Nesting a module inside another module affects the name of the nested module:\n\n    defmodule Foo do\n      defmodule Bar do\n      end\n    end\n\nIn the example above, two modules - `Foo` and `Foo.Bar` - are created.\nWhen nesting, Elixir automatically creates an alias to the inner module,\nallowing the second module `Foo.Bar` to be accessed as `Bar` in the same\nlexical scope where it's defined (the `Foo` module). This only happens\nif the nested module is defined via an alias.\n\nIf the `Foo.Bar` module is moved somewhere else, the references to `Bar` in\nthe `Foo` module need to be updated to the fully-qualified name (`Foo.Bar`) or\nan alias has to be explicitly set in the `Foo` module with the help of\n`Kernel.SpecialForms.alias/2`.\n\n    defmodule Foo.Bar do\n      # code\n    end\n\n    defmodule Foo do\n      alias Foo.Bar\n      # code here can refer to \"Foo.Bar\" as just \"Bar\"\n    end","ref":"Kernel.html#defmodule/2-nesting"},{"type":"macro","title":"Dynamic names - Kernel.defmodule/2","doc":"Elixir module names can be dynamically generated. This is very\nuseful when working with macros. For instance, one could write:\n\n    defmodule String.to_atom(\"Foo#{1}\") do\n      # contents ...\n    end\n\nElixir will accept any module name as long as the expression passed as the\nfirst argument to `defmodule/2` evaluates to an atom.\nNote that, when a dynamic name is used, Elixir won't nest the name under\nthe current module nor automatically set up an alias.","ref":"Kernel.html#defmodule/2-dynamic-names"},{"type":"macro","title":"Reserved module names - Kernel.defmodule/2","doc":"If you attempt to define a module that already exists, you will get a\nwarning saying that a module has been redefined.\n\nThere are some modules that Elixir does not currently implement but it\nmay be implement in the future. Those modules are reserved and defining\nthem will result in a compilation error:\n\n    defmodule Any do\n      # code\n    end\n    ** (CompileError) iex:1: module Any is reserved and cannot be defined\n\nElixir reserves the following module names: `Elixir`, `Any`, `BitString`,\n`PID`, and `Reference`.","ref":"Kernel.html#defmodule/2-reserved-module-names"},{"type":"macro","title":"Kernel.defoverridable/1","doc":"Makes the given definitions in the current module overridable.\n\nIf the user defines a new function or macro with the same name\nand arity, then the overridable ones are discarded. Otherwise, the\noriginal definitions are used.\n\nIt is possible for the overridden definition to have a different visibility\nthan the original: a public function can be overridden by a private\nfunction and vice-versa.\n\nMacros cannot be overridden as functions and vice-versa.","ref":"Kernel.html#defoverridable/1"},{"type":"macro","title":"Example - Kernel.defoverridable/1","doc":"defmodule DefaultMod do\n      defmacro __using__(_opts) do\n        quote do\n          def test(x, y) do\n            x + y\n          end\n\n          defoverridable test: 2\n        end\n      end\n    end\n\n    defmodule ChildMod do\n      use DefaultMod\n\n      def test(x, y) do\n        x * y + super(x, y)\n      end\n    end\n\nAs seen as in the example above, `super` can be used to call the default\nimplementation.\n\n> ##","ref":"Kernel.html#defoverridable/1-example"},{"type":"macro","title":"Disclaimer {: .tip} - Kernel.defoverridable/1","doc":">\n> Use `defoverridable` with care. If you need to define multiple modules\n> with the same behaviour, it may be best to move the default implementation\n> to the caller, and check if a callback exists via `Code.ensure_loaded?/1` and\n> `function_exported?/3`.\n>\n> For example, in the example above, imagine there is a module that calls the\n> `test/2` function. This module could be defined as such:\n>\n>     defmodule CallsTest do\n>       def receives_module_and_calls_test(module, x, y) do\n>         if Code.ensure_loaded?(module) and function_exported?(module, :test, 2) do\n>           module.test(x, y)\n>         else\n>           x + y\n>         end\n>       end\n>     end","ref":"Kernel.html#defoverridable/1-disclaimer-tip"},{"type":"macro","title":"Example with behaviour - Kernel.defoverridable/1","doc":"You can also pass a behaviour to `defoverridable` and it will mark all of the\ncallbacks in the behaviour as overridable:\n\n\n    defmodule Behaviour do\n      @callback test(number(), number()) :: number()\n    end\n\n    defmodule DefaultMod do\n      defmacro __using__(_opts) do\n        quote do\n          @behaviour Behaviour\n\n          def test(x, y) do\n            x + y\n          end\n\n          defoverridable Behaviour\n        end\n      end\n    end\n\n    defmodule ChildMod do\n      use DefaultMod\n\n      def test(x, y) do\n        x * y + super(x, y)\n      end\n    end","ref":"Kernel.html#defoverridable/1-example-with-behaviour"},{"type":"macro","title":"Kernel.defp/2","doc":"Defines a private function with the given name and body.\n\nPrivate functions are only accessible from within the module in which they are\ndefined. Trying to access a private function from outside the module it's\ndefined in results in an `UndefinedFunctionError` exception.\n\nCheck `def/2` for more information.","ref":"Kernel.html#defp/2"},{"type":"macro","title":"Examples - Kernel.defp/2","doc":"defmodule Foo do\n      def bar do\n        sum(1, 2)\n      end\n\n      defp sum(a, b), do: a + b\n    end\n\n    Foo.bar()\n    #=> 3\n\n    Foo.sum(1, 2)\n    ** (UndefinedFunctionError) undefined function Foo.sum/2","ref":"Kernel.html#defp/2-examples"},{"type":"macro","title":"Kernel.defprotocol/2","doc":"Defines a protocol.\n\nSee the `Protocol` module for more information.","ref":"Kernel.html#defprotocol/2"},{"type":"macro","title":"Kernel.defstruct/1","doc":"Defines a struct.\n\nA struct is a tagged map that allows developers to provide\ndefault values for keys, tags to be used in polymorphic\ndispatches and compile time assertions. For more information\nabout structs, please check `Kernel.SpecialForms.%/2`.\n\nIt is only possible to define a struct per module, as the\nstruct is tied to the module itself. Calling `defstruct/1`\nalso defines a `__struct__/0` function that returns the\nstruct itself.","ref":"Kernel.html#defstruct/1"},{"type":"macro","title":"Examples - Kernel.defstruct/1","doc":"defmodule User do\n      defstruct name: nil, age: nil\n    end\n\nStruct fields are evaluated at compile-time, which allows\nthem to be dynamic. In the example below, `10 + 11` is\nevaluated at compile-time and the age field is stored\nwith value `21`:\n\n    defmodule User do\n      defstruct name: nil, age: 10 + 11\n    end\n\nThe `fields` argument is usually a keyword list with field names\nas atom keys and default values as corresponding values. `defstruct/1`\nalso supports a list of atoms as its argument: in that case, the atoms\nin the list will be used as the struct's field names and they will all\ndefault to `nil`.\n\n    defmodule Post do\n      defstruct [:title, :content, :author]\n    end\n\nAdd documentation to a struct with the `@doc` attribute, like a function.\n\n    defmodule Post do\n      @doc \"A post. The content should be valid Markdown.\"\n      defstruct [:title, :content, :author]\n    end","ref":"Kernel.html#defstruct/1-examples"},{"type":"macro","title":"Deriving - Kernel.defstruct/1","doc":"Although structs are maps, by default structs do not implement\nany of the protocols implemented for maps. For example, attempting\nto use a protocol with the `User` struct leads to an error:\n\n    john = %User{name: \"John\"}\n    MyProtocol.call(john)\n    ** (Protocol.UndefinedError) protocol MyProtocol not implemented for %User{...}\n\n`defstruct/1`, however, allows protocol implementations to be\n*derived*. This can be done by defining a `@derive` attribute as a\nlist before invoking `defstruct/1`:\n\n    defmodule User do\n      @derive MyProtocol\n      defstruct name: nil, age: nil\n    end\n\n    MyProtocol.call(john) # it works!\n\nA common example is to `@derive` the `Inspect` protocol to hide certain fields\nwhen the struct is printed:\n\n    defmodule User do\n      @derive {Inspect, only: :name}\n      defstruct name: nil, age: nil\n    end\n\nFor each protocol in `@derive`, Elixir will assert the protocol has\nbeen implemented for `Any`. If the `Any` implementation defines a\n`__deriving__/3` callback, the callback will be invoked and it should define\nthe implementation module. Otherwise an implementation that simply points to\nthe `Any` implementation is automatically derived. For more information on\nthe `__deriving__/3` callback, see `Protocol.derive/3`.","ref":"Kernel.html#defstruct/1-deriving"},{"type":"macro","title":"Enforcing keys - Kernel.defstruct/1","doc":"When building a struct, Elixir will automatically guarantee all keys\nbelong to the struct:\n\n    %User{name: \"john\", unknown: :key}\n    ** (KeyError) key :unknown not found in: %User{age: 21, name: nil}\n\nElixir also allows developers to enforce that certain keys must always be\ngiven when building the struct:\n\n    defmodule User do\n      @enforce_keys [:name]\n      defstruct name: nil, age: 10 + 11\n    end\n\nNow trying to build a struct without the name key will fail:\n\n    %User{age: 21}\n    ** (ArgumentError) the following keys must also be given when building struct User: [:name]\n\nKeep in mind `@enforce_keys` is a simple compile-time guarantee\nto aid developers when building structs. It is not enforced on\nupdates and it does not provide any sort of value-validation.","ref":"Kernel.html#defstruct/1-enforcing-keys"},{"type":"macro","title":"Types - Kernel.defstruct/1","doc":"It is recommended to define types for structs. By convention, such a type\nis called `t`. To define a struct inside a type, the struct literal syntax\nis used:\n\n    defmodule User do\n      defstruct name: \"John\", age: 25\n      @type t :: %__MODULE__{name: String.t(), age: non_neg_integer}\n    end\n\nIt is recommended to only use the struct syntax when defining the struct's\ntype. When referring to another struct, it's better to use `User.t()` instead of\n`%User{}`.\n\nThe types of the struct fields that are not included in `%User{}` default to\n`term()` (see `t:term/0`).\n\nStructs whose internal structure is private to the local module (pattern\nmatching them or directly accessing their fields should not be allowed) should\nuse the `@opaque` attribute. Structs whose internal structure is public should\nuse `@type`.","ref":"Kernel.html#defstruct/1-types"},{"type":"macro","title":"Kernel.destructure/2","doc":"Destructures two lists, assigning each term in the\nright one to the matching term in the left one.\n\nUnlike pattern matching via `=`, if the sizes of the left\nand right lists don't match, destructuring simply stops\ninstead of raising an error.","ref":"Kernel.html#destructure/2"},{"type":"macro","title":"Examples - Kernel.destructure/2","doc":"iex> destructure([x, y, z], [1, 2, 3, 4, 5])\n    iex> {x, y, z}\n    {1, 2, 3}\n\nIn the example above, even though the right list has more entries than the\nleft one, destructuring works fine. If the right list is smaller, the\nremaining elements are simply set to `nil`:\n\n    iex> destructure([x, y, z], [1])\n    iex> {x, y, z}\n    {1, nil, nil}\n\nThe left-hand side supports any expression you would use\non the left-hand side of a match:\n\n    x = 1\n    destructure([^x, y, z], [1, 2, 3])\n\nThe example above will only work if `x` matches the first value in the right\nlist. Otherwise, it will raise a `MatchError` (like the `=` operator would\ndo).","ref":"Kernel.html#destructure/2-examples"},{"type":"function","title":"Kernel.div/2","doc":"Performs an integer division.\n\nRaises an `ArithmeticError` exception if one of the arguments is not an\ninteger, or when the `divisor` is `0`.\n\n`div/2` performs *truncated* integer division. This means that\nthe result is always rounded towards zero.\n\nIf you want to perform floored integer division (rounding towards negative infinity),\nuse `Integer.floor_div/2` instead.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#div/2"},{"type":"function","title":"Examples - Kernel.div/2","doc":"div(5, 2)\n    #=> 2\n\n    div(6, -4)\n    #=> -1\n\n    div(-99, 2)\n    #=> -49\n\n    div(100, 0)\n    ** (ArithmeticError) bad argument in arithmetic expression","ref":"Kernel.html#div/2-examples"},{"type":"function","title":"Kernel.elem/2","doc":"Gets the element at the zero-based `index` in `tuple`.\n\nIt raises `ArgumentError` when index is negative or it is out of range of the tuple elements.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#elem/2"},{"type":"function","title":"Examples - Kernel.elem/2","doc":"tuple = {:foo, :bar, 3}\n    elem(tuple, 1)\n    #=> :bar\n\n    elem({}, 0)\n    ** (ArgumentError) argument error\n\n    elem({:foo, :bar}, 2)\n    ** (ArgumentError) argument error","ref":"Kernel.html#elem/2-examples"},{"type":"function","title":"Kernel.exit/1","doc":"Stops the execution of the calling process with the given reason.\n\nSince evaluating this function causes the process to terminate,\nit has no return value.\n\nInlined by the compiler.","ref":"Kernel.html#exit/1"},{"type":"function","title":"Examples - Kernel.exit/1","doc":"When a process reaches its end, by default it exits with\nreason `:normal`. You can also call `exit/1` explicitly if you\nwant to terminate a process but not signal any failure:\n\n    exit(:normal)\n\nIn case something goes wrong, you can also use `exit/1` with\na different reason:\n\n    exit(:seems_bad)\n\nIf the exit reason is not `:normal`, all the processes linked to the process\nthat exited will crash (unless they are trapping exits).","ref":"Kernel.html#exit/1-examples"},{"type":"function","title":"OTP exits - Kernel.exit/1","doc":"Exits are used by the OTP to determine if a process exited abnormally\nor not. The following exits are considered \"normal\":\n\n  * `exit(:normal)`\n  * `exit(:shutdown)`\n  * `exit({:shutdown, term})`\n\nExiting with any other reason is considered abnormal and treated\nas a crash. This means the default supervisor behaviour kicks in,\nerror reports are emitted, and so forth.\n\nThis behaviour is relied on in many different places. For example,\n`ExUnit` uses `exit(:shutdown)` when exiting the test process to\nsignal linked processes, supervision trees and so on to politely\nshut down too.","ref":"Kernel.html#exit/1-otp-exits"},{"type":"function","title":"CLI exits - Kernel.exit/1","doc":"Building on top of the exit signals mentioned above, if the\nprocess started by the command line exits with any of the three\nreasons above, its exit is considered normal and the Operating\nSystem process will exit with status 0.\n\nIt is, however, possible to customize the operating system exit\nsignal by invoking:\n\n    exit({:shutdown, integer})\n\nThis will cause the operating system process to exit with the status given by\n`integer` while signaling all linked Erlang processes to politely\nshut down.\n\nAny other exit reason will cause the operating system process to exit with\nstatus `1` and linked Erlang processes to crash.","ref":"Kernel.html#exit/1-cli-exits"},{"type":"function","title":"Kernel.floor/1","doc":"Returns the largest integer smaller than or equal to `number`.\n\nIf you want to perform floor operation on other decimal places,\nuse `Float.floor/2` instead.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#floor/1"},{"type":"function","title":"Examples - Kernel.floor/1","doc":"iex> floor(10)\n    10\n\n    iex> floor(9.7)\n    9\n\n    iex> floor(-9.7)\n    -10","ref":"Kernel.html#floor/1-examples"},{"type":"function","title":"Kernel.function_exported?/3","doc":"Returns `true` if `module` is loaded and contains a\npublic `function` with the given `arity`, otherwise `false`.\n\nNote that this function does not load the module in case\nit is not loaded. Check `Code.ensure_loaded/1` for more\ninformation.\n\nInlined by the compiler.","ref":"Kernel.html#function_exported?/3"},{"type":"function","title":"Examples - Kernel.function_exported?/3","doc":"iex> function_exported?(Enum, :map, 2)\n    true\n\n    iex> function_exported?(Enum, :map, 10)\n    false\n\n    iex> function_exported?(List, :to_string, 1)\n    true","ref":"Kernel.html#function_exported?/3-examples"},{"type":"macro","title":"Kernel.get_and_update_in/2","doc":"Gets a value and updates a nested data structure via the given `path`.\n\nThis is similar to `get_and_update_in/3`, except the path is extracted\nvia a macro rather than passing a list. For example:\n\n    get_and_update_in(opts[:foo][:bar], &{&1, &1 + 1})\n\nIs equivalent to:\n\n    get_and_update_in(opts, [:foo, :bar], &{&1, &1 + 1})\n\nThis also works with nested structs and the `struct.path.to.value` way to specify\npaths:\n\n    get_and_update_in(struct.foo.bar, &{&1, &1 + 1})\n\nNote that in order for this macro to work, the complete path must always\nbe visible by this macro. See the \"Paths\" section below.","ref":"Kernel.html#get_and_update_in/2"},{"type":"macro","title":"Examples - Kernel.get_and_update_in/2","doc":"iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> get_and_update_in(users[\"john\"].age, &{&1, &1 + 1})\n    {27, %{\"john\" => %{age: 28}, \"meg\" => %{age: 23}}}","ref":"Kernel.html#get_and_update_in/2-examples"},{"type":"macro","title":"Paths - Kernel.get_and_update_in/2","doc":"A path may start with a variable, local or remote call, and must be\nfollowed by one or more:\n\n  * `foo[bar]` - accesses the key `bar` in `foo`; in case `foo` is nil,\n    `nil` is returned\n\n  * `foo.bar` - accesses a map/struct field; in case the field is not\n    present, an error is raised\n\nHere are some valid paths:\n\n    users[\"john\"][:age]\n    users[\"john\"].age\n    User.all()[\"john\"].age\n    all_users()[\"john\"].age\n\nHere are some invalid ones:\n\n    # Does a remote call after the initial value\n    users[\"john\"].do_something(arg1, arg2)\n\n    # Does not access any key or field\n    users","ref":"Kernel.html#get_and_update_in/2-paths"},{"type":"function","title":"Kernel.get_and_update_in/3","doc":"Gets a value and updates a nested structure.\n\n`data` is a nested structure (that is, a map, keyword\nlist, or struct that implements the `Access` behaviour).\n\nThe `fun` argument receives the value of `key` (or `nil` if `key`\nis not present) and must return one of the following values:\n\n  * a two-element tuple `{current_value, new_value}`. In this case,\n    `current_value` is the retrieved value which can possibly be operated on before\n    being returned. `new_value` is the new value to be stored under `key`.\n\n  * `:pop`, which implies that the current value under `key`\n    should be removed from the structure and returned.\n\nThis function uses the `Access` module to traverse the structures\naccording to the given `keys`, unless the `key` is a function,\nwhich is detailed in a later section.","ref":"Kernel.html#get_and_update_in/3"},{"type":"function","title":"Examples - Kernel.get_and_update_in/3","doc":"This function is useful when there is a need to retrieve the current\nvalue (or something calculated in function of the current value) and\nupdate it at the same time. For example, it could be used to read the\ncurrent age of a user while increasing it by one in one pass:\n\n    iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> get_and_update_in(users, [\"john\", :age], &{&1, &1 + 1})\n    {27, %{\"john\" => %{age: 28}, \"meg\" => %{age: 23}}}\n\nNote the current value given to the anonymous function may be `nil`.\nIf any of the intermediate values are nil, it will raise:\n\n    iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> get_and_update_in(users, [\"jane\", :age], &{&1, &1 + 1})\n    ** (ArgumentError) could not put/update key :age on a nil value","ref":"Kernel.html#get_and_update_in/3-examples"},{"type":"function","title":"Functions as keys - Kernel.get_and_update_in/3","doc":"If a key is a function, the function will be invoked passing three\narguments:\n\n  * the operation (`:get_and_update`)\n  * the data to be accessed\n  * a function to be invoked next\n\nThis means `get_and_update_in/3` can be extended to provide custom\nlookups. The downside is that functions cannot be stored as keys\nin the accessed data structures.\n\nWhen one of the keys is a function, the function is invoked.\nIn the example below, we use a function to get and increment all\nages inside a list:\n\n    iex> users = [%{name: \"john\", age: 27}, %{name: \"meg\", age: 23}]\n    iex> all = fn :get_and_update, data, next ->\n    ...>   data |> Enum.map(next) |> Enum.unzip()\n    ...> end\n    iex> get_and_update_in(users, [all, :age], &{&1, &1 + 1})\n    {[27, 23], [%{name: \"john\", age: 28}, %{name: \"meg\", age: 24}]}\n\nIf the previous value before invoking the function is `nil`,\nthe function *will* receive `nil` as a value and must handle it\naccordingly (be it by failing or providing a sane default).\n\nThe `Access` module ships with many convenience accessor functions,\nlike the `all` anonymous function defined above. See `Access.all/0`,\n`Access.key/2`, and others as examples.","ref":"Kernel.html#get_and_update_in/3-functions-as-keys"},{"type":"function","title":"Kernel.get_in/2","doc":"Gets a value from a nested structure.\n\nUses the `Access` module to traverse the structures\naccording to the given `keys`, unless the `key` is a\nfunction, which is detailed in a later section.\n\nNote that if none of the given keys are functions,\nthere is rarely a reason to use `get_in` over\nwriting \"regular\" Elixir code using `[]`.","ref":"Kernel.html#get_in/2"},{"type":"function","title":"Examples - Kernel.get_in/2","doc":"iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> get_in(users, [\"john\", :age])\n    27\n    iex> # Equivalent to:\n    iex> users[\"john\"][:age]\n    27\n\n`get_in/2` can also use the accessors in the `Access` module\nto traverse more complex data structures. For example, here we\nuse `Access.all/0` to traverse a list:\n\n    iex> users = [%{name: \"john\", age: 27}, %{name: \"meg\", age: 23}]\n    iex> get_in(users, [Access.all(), :age])\n    [27, 23]\n\nIn case any of the components returns `nil`, `nil` will be returned\nand `get_in/2` won't traverse any further:\n\n    iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> get_in(users, [\"unknown\", :age])\n    nil\n    iex> # Equivalent to:\n    iex> users[\"unknown\"][:age]\n    nil\n\n    iex> users = nil\n    iex> get_in(users, [Access.all(), :age])\n    nil\n\nAlternatively, if you need to access complex data-structures, you can\nuse pattern matching:\n\n    case users do\n      %{\"john\" => %{age: age}} -> age\n      _ -> default_value\n    end","ref":"Kernel.html#get_in/2-examples"},{"type":"function","title":"Functions as keys - Kernel.get_in/2","doc":"If a key given to `get_in/2` is a function, the function will be invoked\npassing three arguments:\n\n  * the operation (`:get`)\n  * the data to be accessed\n  * a function to be invoked next\n\nThis means `get_in/2` can be extended to provide custom lookups.\nThat's precisely how the `Access.all/0` key in the previous section\nbehaves. For example, we can manually implement such traversal as\nfollows:\n\n    iex> users = [%{name: \"john\", age: 27}, %{name: \"meg\", age: 23}]\n    iex> all = fn :get, data, next -> Enum.map(data, next) end\n    iex> get_in(users, [all, :age])\n    [27, 23]\n\nThe `Access` module ships with many convenience accessor functions.\nSee `Access.all/0`, `Access.key/2`, and others as examples.","ref":"Kernel.html#get_in/2-functions-as-keys"},{"type":"function","title":"Working with structs - Kernel.get_in/2","doc":"By default, structs do not implement the `Access` behaviour required\nby this function. Therefore, you can't do this:\n\n    get_in(some_struct, [:some_key, :nested_key])\n\nThe good news is that structs have predefined shape. Therefore,\nyou can write instead:\n\n    some_struct.some_key.nested_key\n\nIf, by any chance, `some_key` can return nil, you can always\nfallback to pattern matching to provide nested struct handling:\n\n    case some_struct do\n      %{some_key: %{nested_key: value}} -> value\n      %{} -> nil\n    end","ref":"Kernel.html#get_in/2-working-with-structs"},{"type":"function","title":"Kernel.hd/1","doc":"Returns the head of a list. Raises `ArgumentError` if the list is empty.\n\nThe head of a list is its first element.\n\nIt works with improper lists.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#hd/1"},{"type":"function","title":"Examples - Kernel.hd/1","doc":"hd([1, 2, 3, 4])\n    #=> 1\n\n    hd([1 | 2])\n    #=> 1\n\nGiving it an empty list raises:\n\n    hd([])\n    ** (ArgumentError) argument error","ref":"Kernel.html#hd/1-examples"},{"type":"macro","title":"Kernel.if/2","doc":"Provides an `if/2` macro.\n\nThis macro expects the first argument to be a condition and the second\nargument to be a keyword list.","ref":"Kernel.html#if/2"},{"type":"macro","title":"One-liner examples - Kernel.if/2","doc":"if(foo, do: bar)\n\nIn the example above, `bar` will be returned if `foo` evaluates to\na truthy value (neither `false` nor `nil`). Otherwise, `nil` will be\nreturned.\n\nAn `else` option can be given to specify the opposite:\n\n    if(foo, do: bar, else: baz)","ref":"Kernel.html#if/2-one-liner-examples"},{"type":"macro","title":"Blocks examples - Kernel.if/2","doc":"It's also possible to pass a block to the `if/2` macro. The first\nexample above would be translated to:\n\n    if foo do\n      bar\n    end\n\nNote that `do`-`end` become delimiters. The second example would\ntranslate to:\n\n    if foo do\n      bar\n    else\n      baz\n    end\n\nIn order to compare more than two clauses, the `cond/1` macro has to be used.","ref":"Kernel.html#if/2-blocks-examples"},{"type":"macro","title":"Kernel.in/2","doc":"Membership operator.\n\nChecks if the element on the left-hand side is a member of the\ncollection on the right-hand side.","ref":"Kernel.html#in/2"},{"type":"macro","title":"Examples - Kernel.in/2","doc":"iex> x = 1\n    iex> x in [1, 2, 3]\n    true\n\nThis operator (which is a macro) simply translates to a call to\n`Enum.member?/2`. The example above would translate to:\n\n    Enum.member?([1, 2, 3], x)\n\nElixir also supports `left not in right`, which evaluates to\n`not(left in right)`:\n\n    iex> x = 1\n    iex> x not in [1, 2, 3]\n    false","ref":"Kernel.html#in/2-examples"},{"type":"macro","title":"Guards - Kernel.in/2","doc":"The `in/2` operator (as well as `not in`) can be used in guard clauses as\nlong as the right-hand side is a range or a list.\n\nIf the right-hand side is a list, Elixir will expand the operator to a valid\nguard expression which needs to check each value. For example:\n\n    when x in [1, 2, 3]\n\ntranslates to:\n\n    when x === 1 or x === 2 or x === 3\n\nHowever, this construct will be inefficient for large lists. In such cases, it\nis best to stop using guards and use a more appropriate data structure, such\nas `MapSet`.\n\nIf the right-hand side is a range, a more efficient comparison check will be\ndone. For example:\n\n    when x in 1..1000\n\ntranslates roughly to:\n\n    when x >= 1 and x <= 1000\n\n#","ref":"Kernel.html#in/2-guards"},{"type":"macro","title":"AST considerations - Kernel.in/2","doc":"`left not in right` is parsed by the compiler into the AST:\n\n    {:not, _, [{:in, _, [left, right]}]}\n\nThis is the same AST as `not(left in right)`.\n\nAdditionally, `Macro.to_string/2` and `Code.format_string!/2`\nwill translate all occurrences of this AST to `left not in right`.","ref":"Kernel.html#in/2-ast-considerations"},{"type":"function","title":"Kernel.inspect/2","doc":"Inspects the given argument according to the `Inspect` protocol.\nThe second argument is a keyword list with options to control\ninspection.","ref":"Kernel.html#inspect/2"},{"type":"function","title":"Options - Kernel.inspect/2","doc":"`inspect/2` accepts a list of options that are internally\ntranslated to an `Inspect.Opts` struct. Check the docs for\n`Inspect.Opts` to see the supported options.","ref":"Kernel.html#inspect/2-options"},{"type":"function","title":"Examples - Kernel.inspect/2","doc":"iex> inspect(:foo)\n    \":foo\"\n\n    iex> inspect([1, 2, 3, 4, 5], limit: 3)\n    \"[1, 2, 3, ...]\"\n\n    iex> inspect([1, 2, 3], pretty: true, width: 0)\n    \"[1,\\n 2,\\n 3]\"\n\n    iex> inspect(\"olá\"    >)\n    \" >\"\n\n    iex> inspect(\"olá\"    >, binaries: :as_strings)\n    \"\\\"olá\\\\0\\\"\"\n\n    iex> inspect(\"olá\", binaries: :as_binaries)\n    \" >\"\n\n    iex> inspect(~c\"bar\")\n    \"~c\\\"bar\\\"\"\n\n    iex> inspect([0 | ~c\"bar\"])\n    \"[0, 98, 97, 114]\"\n\n    iex> inspect(100, base: :octal)\n    \"0o144\"\n\n    iex> inspect(100, base: :hex)\n    \"0x64\"\n\nNote that the `Inspect` protocol does not necessarily return a valid\nrepresentation of an Elixir term. In such cases, the inspected result\nmust start with `#`. For example, inspecting a function will return:\n\n    inspect(fn a, b -> a + b end)\n    #=> #Function \n\nThe `Inspect` protocol can be derived to hide certain fields\nfrom structs, so they don't show up in logs, inspects and similar.\nSee the \"Deriving\" section of the documentation of the `Inspect`\nprotocol for more information.","ref":"Kernel.html#inspect/2-examples"},{"type":"function","title":"Kernel.is_atom/1","doc":"Returns `true` if `term` is an atom; otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_atom/1"},{"type":"function","title":"Examples - Kernel.is_atom/1","doc":"iex> is_atom(false)\n    true\n\n    iex> is_atom(:name)\n    true\n\n    iex> is_atom(AnAtom)\n    true\n\n    iex> is_atom(\"true\")\n    false","ref":"Kernel.html#is_atom/1-examples"},{"type":"function","title":"Kernel.is_binary/1","doc":"Returns `true` if `term` is a binary; otherwise returns `false`.\n\nA binary always contains a complete number of bytes.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_binary/1"},{"type":"function","title":"Examples - Kernel.is_binary/1","doc":"iex> is_binary(\"foo\")\n    true\n    iex> is_binary( >)\n    false","ref":"Kernel.html#is_binary/1-examples"},{"type":"function","title":"Kernel.is_bitstring/1","doc":"Returns `true` if `term` is a bitstring (including a binary); otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_bitstring/1"},{"type":"function","title":"Examples - Kernel.is_bitstring/1","doc":"iex> is_bitstring(\"foo\")\n    true\n    iex> is_bitstring( >)\n    true","ref":"Kernel.html#is_bitstring/1-examples"},{"type":"function","title":"Kernel.is_boolean/1","doc":"Returns `true` if `term` is either the atom `true` or the atom `false` (i.e.,\na boolean); otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_boolean/1"},{"type":"function","title":"Examples - Kernel.is_boolean/1","doc":"iex> is_boolean(false)\n    true\n\n    iex> is_boolean(true)\n    true\n\n    iex> is_boolean(:test)\n    false","ref":"Kernel.html#is_boolean/1-examples"},{"type":"macro","title":"Kernel.is_exception/1","doc":"Returns true if `term` is an exception; otherwise returns `false`.\n\nAllowed in guard tests.","ref":"Kernel.html#is_exception/1"},{"type":"macro","title":"Examples - Kernel.is_exception/1","doc":"iex> is_exception(%RuntimeError{})\n    true\n\n    iex> is_exception(%{})\n    false","ref":"Kernel.html#is_exception/1-examples"},{"type":"macro","title":"Kernel.is_exception/2","doc":"Returns true if `term` is an exception of `name`; otherwise returns `false`.\n\nAllowed in guard tests.","ref":"Kernel.html#is_exception/2"},{"type":"macro","title":"Examples - Kernel.is_exception/2","doc":"iex> is_exception(%RuntimeError{}, RuntimeError)\n    true\n\n    iex> is_exception(%RuntimeError{}, Macro.Env)\n    false","ref":"Kernel.html#is_exception/2-examples"},{"type":"function","title":"Kernel.is_float/1","doc":"Returns `true` if `term` is a floating-point number; otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_float/1"},{"type":"function","title":"Kernel.is_function/1","doc":"Returns `true` if `term` is a function; otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_function/1"},{"type":"function","title":"Examples - Kernel.is_function/1","doc":"iex> is_function(fn x -> x + x end)\n    true\n\n    iex> is_function(\"not a function\")\n    false","ref":"Kernel.html#is_function/1-examples"},{"type":"function","title":"Kernel.is_function/2","doc":"Returns `true` if `term` is a function that can be applied with `arity` number of arguments;\notherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_function/2"},{"type":"function","title":"Examples - Kernel.is_function/2","doc":"iex> is_function(fn x -> x * 2 end, 1)\n    true\n    iex> is_function(fn x -> x * 2 end, 2)\n    false","ref":"Kernel.html#is_function/2-examples"},{"type":"function","title":"Kernel.is_integer/1","doc":"Returns `true` if `term` is an integer; otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_integer/1"},{"type":"function","title":"Kernel.is_list/1","doc":"Returns `true` if `term` is a list with zero or more elements; otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_list/1"},{"type":"function","title":"Kernel.is_map/1","doc":"Returns `true` if `term` is a map; otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_map/1"},{"type":"function","title":"Kernel.is_map_key/2","doc":"Returns `true` if `key` is a key in `map`; otherwise returns `false`.\n\nIt raises `BadMapError` if the first element is not a map.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_map_key/2"},{"type":"function","title":"Examples - Kernel.is_map_key/2","doc":"iex> is_map_key(%{a: \"foo\", b: \"bar\"}, :a)\n    true\n\n    iex> is_map_key(%{a: \"foo\", b: \"bar\"}, :c)\n    false","ref":"Kernel.html#is_map_key/2-examples"},{"type":"macro","title":"Kernel.is_nil/1","doc":"Returns `true` if `term` is `nil`, `false` otherwise.\n\nAllowed in guard clauses.","ref":"Kernel.html#is_nil/1"},{"type":"macro","title":"Examples - Kernel.is_nil/1","doc":"iex> is_nil(1)\n    false\n\n    iex> is_nil(nil)\n    true","ref":"Kernel.html#is_nil/1-examples"},{"type":"function","title":"Kernel.is_number/1","doc":"Returns `true` if `term` is either an integer or a floating-point number;\notherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_number/1"},{"type":"function","title":"Kernel.is_pid/1","doc":"Returns `true` if `term` is a PID (process identifier); otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_pid/1"},{"type":"function","title":"Kernel.is_port/1","doc":"Returns `true` if `term` is a port identifier; otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_port/1"},{"type":"function","title":"Kernel.is_reference/1","doc":"Returns `true` if `term` is a reference; otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_reference/1"},{"type":"macro","title":"Kernel.is_struct/1","doc":"Returns true if `term` is a struct; otherwise returns `false`.\n\nAllowed in guard tests.","ref":"Kernel.html#is_struct/1"},{"type":"macro","title":"Examples - Kernel.is_struct/1","doc":"iex> is_struct(URI.parse(\"/\"))\n    true\n\n    iex> is_struct(%{})\n    false","ref":"Kernel.html#is_struct/1-examples"},{"type":"macro","title":"Kernel.is_struct/2","doc":"Returns true if `term` is a struct of `name`; otherwise returns `false`.\n\n`is_struct/2` does not check that `name` exists and is a valid struct.\nIf you want such validations, you must pattern match on the struct\ninstead, such as `match?(%URI{}, arg)`.\n\nAllowed in guard tests.","ref":"Kernel.html#is_struct/2"},{"type":"macro","title":"Examples - Kernel.is_struct/2","doc":"iex> is_struct(URI.parse(\"/\"), URI)\n    true\n\n    iex> is_struct(URI.parse(\"/\"), Macro.Env)\n    false","ref":"Kernel.html#is_struct/2-examples"},{"type":"function","title":"Kernel.is_tuple/1","doc":"Returns `true` if `term` is a tuple; otherwise returns `false`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#is_tuple/1"},{"type":"function","title":"Kernel.length/1","doc":"Returns the length of `list`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#length/1"},{"type":"function","title":"Examples - Kernel.length/1","doc":"iex> length([1, 2, 3, 4, 5, 6, 7, 8, 9])\n    9","ref":"Kernel.html#length/1-examples"},{"type":"function","title":"Kernel.macro_exported?/3","doc":"Returns `true` if `module` is loaded and contains a\npublic `macro` with the given `arity`, otherwise `false`.\n\nNote that this function does not load the module in case\nit is not loaded. Check `Code.ensure_loaded/1` for more\ninformation.\n\nIf `module` is an Erlang module (as opposed to an Elixir module), this\nfunction always returns `false`.","ref":"Kernel.html#macro_exported?/3"},{"type":"function","title":"Examples - Kernel.macro_exported?/3","doc":"iex> macro_exported?(Kernel, :use, 2)\n    true\n\n    iex> macro_exported?(:erlang, :abs, 1)\n    false","ref":"Kernel.html#macro_exported?/3-examples"},{"type":"function","title":"Kernel.make_ref/0","doc":"Returns an almost unique reference.\n\nThe returned reference will re-occur after approximately 2^82 calls;\ntherefore it is unique enough for practical purposes.\n\nInlined by the compiler.","ref":"Kernel.html#make_ref/0"},{"type":"function","title":"Examples - Kernel.make_ref/0","doc":"make_ref()\n    #=> #Reference","ref":"Kernel.html#make_ref/0-examples"},{"type":"function","title":"Kernel.map_size/1","doc":"Returns the size of a map.\n\nThe size of a map is the number of key-value pairs that the map contains.\n\nThis operation happens in constant time.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#map_size/1"},{"type":"function","title":"Examples - Kernel.map_size/1","doc":"iex> map_size(%{a: \"foo\", b: \"bar\"})\n    2","ref":"Kernel.html#map_size/1-examples"},{"type":"macro","title":"Kernel.match?/2","doc":"A convenience macro that checks if the right side (an expression) matches the\nleft side (a pattern).","ref":"Kernel.html#match?/2"},{"type":"macro","title":"Examples - Kernel.match?/2","doc":"iex> match?(1, 1)\n    true\n\n    iex> match?({1, _}, {1, 2})\n    true\n\n    iex> map = %{a: 1, b: 2}\n    iex> match?(%{a: _}, map)\n    true\n\n    iex> a = 1\n    iex> match?(^a, 1)\n    true\n\n`match?/2` is very useful when filtering or finding a value in an enumerable:\n\n    iex> list = [a: 1, b: 2, a: 3]\n    iex> Enum.filter(list, &match?({:a, _}, &1))\n    [a: 1, a: 3]\n\nGuard clauses can also be given to the match:\n\n    iex> list = [a: 1, b: 2, a: 3]\n    iex> Enum.filter(list, &match?({:a, x} when x   match?(_x, 1)\n    true\n    iex> binding()\n    []","ref":"Kernel.html#match?/2-examples"},{"type":"macro","title":"Values vs patterns - Kernel.match?/2","doc":"Remember the pin operator matches _values_, not _patterns_.\nPassing a variable as the pattern will always return `true` and will\nresult in a warning that the variable is unused:\n\n    # don't do this\n    pattern = %{a: :a}\n    match?(pattern, %{b: :b})\n\nSimilarly, moving an expression out the pattern may no longer preserve\nits semantics. For example:\n\n    match?([_ | _], [1, 2, 3])\n    #=> true\n\n    pattern = [_ | _]\n    match?(pattern, [1, 2, 3])\n    ** (CompileError) invalid use of _. _ can only be used inside patterns to ignore values and cannot be used in expressions. Make sure you are inside a pattern or change it accordingly\n\nAnother example is that a map as a pattern performs a subset match, but not\nonce assigned to a variable:\n\n    match?(%{x: 1}, %{x: 1, y: 2})\n    #=> true\n\n    attrs = %{x: 1}\n    match?(^attrs, %{x: 1, y: 2})\n    #=> false\n\nThe pin operator will check if the values are equal, using `===/2`, while\npatterns have their own rules when matching maps, lists, and so forth.\nSuch behaviour is not specific to `match?/2`. The following code also\nthrows an exception:\n\n    attrs = %{x: 1}\n    ^attrs = %{x: 1, y: 2}\n    #=> (MatchError) no match of right hand side value: %{x: 1, y: 2}","ref":"Kernel.html#match?/2-values-vs-patterns"},{"type":"function","title":"Kernel.max/2","doc":"Returns the biggest of the two given terms according to\ntheir structural comparison.\n\nIf the terms compare equal, the first one is returned.\n\nThis performs a structural comparison where all Elixir\nterms can be compared with each other. See the [\"Structural\ncomparison\" section](#module-structural-comparison) section\nfor more information.\n\nInlined by the compiler.","ref":"Kernel.html#max/2"},{"type":"function","title":"Examples - Kernel.max/2","doc":"iex> max(1, 2)\n    2\n    iex> max(:a, :b)\n    :b","ref":"Kernel.html#max/2-examples"},{"type":"function","title":"Kernel.min/2","doc":"Returns the smallest of the two given terms according to\ntheir structural comparison.\n\nIf the terms compare equal, the first one is returned.\n\nThis performs a structural comparison where all Elixir\nterms can be compared with each other. See the [\"Structural\ncomparison\" section](#module-structural-comparison) section\nfor more information.\n\nInlined by the compiler.","ref":"Kernel.html#min/2"},{"type":"function","title":"Examples - Kernel.min/2","doc":"iex> min(1, 2)\n    1\n    iex> min(\"foo\", \"bar\")\n    \"bar\"","ref":"Kernel.html#min/2-examples"},{"type":"function","title":"Kernel.node/0","doc":"Returns an atom representing the name of the local node.\nIf the node is not alive, `:nonode@nohost` is returned instead.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#node/0"},{"type":"function","title":"Kernel.node/1","doc":"Returns the node where the given argument is located.\nThe argument can be a PID, a reference, or a port.\nIf the local node is not alive, `:nonode@nohost` is returned.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#node/1"},{"type":"function","title":"Kernel.not/1","doc":"Strictly boolean \"not\" operator.\n\n`value` must be a boolean; if it's not, an `ArgumentError` exception is raised.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#not/1"},{"type":"function","title":"Examples - Kernel.not/1","doc":"iex> not false\n    true","ref":"Kernel.html#not/1-examples"},{"type":"macro","title":"Kernel.or/2","doc":"Strictly boolean \"or\" operator.\n\nIf `left` is `true`, returns `true`; otherwise returns `right`.\n\nRequires only the `left` operand to be a boolean since it short-circuits.\nIf the `left` operand is not a boolean, a `BadBooleanError` exception is\nraised.\n\nAllowed in guard tests.","ref":"Kernel.html#or/2"},{"type":"macro","title":"Examples - Kernel.or/2","doc":"iex> true or false\n    true\n\n    iex> false or 42\n    42\n\n    iex> 42 or false\n    ** (BadBooleanError) expected a boolean on left-side of \"or\", got: 42","ref":"Kernel.html#or/2-examples"},{"type":"macro","title":"Kernel.pop_in/1","doc":"Pops a key from the nested structure via the given `path`.\n\nThis is similar to `pop_in/2`, except the path is extracted via\na macro rather than passing a list. For example:\n\n    pop_in(opts[:foo][:bar])\n\nIs equivalent to:\n\n    pop_in(opts, [:foo, :bar])\n\nNote that in order for this macro to work, the complete path must always\nbe visible by this macro. For more information about the supported path\nexpressions, please check `get_and_update_in/2` docs.","ref":"Kernel.html#pop_in/1"},{"type":"macro","title":"Examples - Kernel.pop_in/1","doc":"iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> pop_in(users[\"john\"][:age])\n    {27, %{\"john\" => %{}, \"meg\" => %{age: 23}}}\n\n    iex> users = %{john: %{age: 27}, meg: %{age: 23}}\n    iex> pop_in(users.john[:age])\n    {27, %{john: %{}, meg: %{age: 23}}}\n\nIn case any entry returns `nil`, its key will be removed\nand the deletion will be considered a success.","ref":"Kernel.html#pop_in/1-examples"},{"type":"function","title":"Kernel.pop_in/2","doc":"Pops a key from the given nested structure.\n\nUses the `Access` protocol to traverse the structures\naccording to the given `keys`, unless the `key` is a\nfunction. If the key is a function, it will be invoked\nas specified in `get_and_update_in/3`.","ref":"Kernel.html#pop_in/2"},{"type":"function","title":"Examples - Kernel.pop_in/2","doc":"iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> pop_in(users, [\"john\", :age])\n    {27, %{\"john\" => %{}, \"meg\" => %{age: 23}}}\n\nIn case any entry returns `nil`, its key will be removed\nand the deletion will be considered a success.\n\n    iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> pop_in(users, [\"jane\", :age])\n    {nil, %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}}","ref":"Kernel.html#pop_in/2-examples"},{"type":"function","title":"Kernel.put_elem/3","doc":"Puts `value` at the given zero-based `index` in `tuple`.\n\nInlined by the compiler.","ref":"Kernel.html#put_elem/3"},{"type":"function","title":"Examples - Kernel.put_elem/3","doc":"iex> tuple = {:foo, :bar, 3}\n    iex> put_elem(tuple, 0, :baz)\n    {:baz, :bar, 3}","ref":"Kernel.html#put_elem/3-examples"},{"type":"macro","title":"Kernel.put_in/2","doc":"Puts a value in a nested structure via the given `path`.\n\nThis is similar to `put_in/3`, except the path is extracted via\na macro rather than passing a list. For example:\n\n    put_in(opts[:foo][:bar], :baz)\n\nIs equivalent to:\n\n    put_in(opts, [:foo, :bar], :baz)\n\nThis also works with nested structs and the `struct.path.to.value` way to specify\npaths:\n\n    put_in(struct.foo.bar, :baz)\n\nNote that in order for this macro to work, the complete path must always\nbe visible by this macro. For more information about the supported path\nexpressions, please check `get_and_update_in/2` docs.","ref":"Kernel.html#put_in/2"},{"type":"macro","title":"Examples - Kernel.put_in/2","doc":"iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> put_in(users[\"john\"][:age], 28)\n    %{\"john\" => %{age: 28}, \"meg\" => %{age: 23}}\n\n    iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> put_in(users[\"john\"].age, 28)\n    %{\"john\" => %{age: 28}, \"meg\" => %{age: 23}}","ref":"Kernel.html#put_in/2-examples"},{"type":"function","title":"Kernel.put_in/3","doc":"Puts a value in a nested structure.\n\nUses the `Access` module to traverse the structures\naccording to the given `keys`, unless the `key` is a\nfunction. If the key is a function, it will be invoked\nas specified in `get_and_update_in/3`.","ref":"Kernel.html#put_in/3"},{"type":"function","title":"Examples - Kernel.put_in/3","doc":"iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> put_in(users, [\"john\", :age], 28)\n    %{\"john\" => %{age: 28}, \"meg\" => %{age: 23}}\n\nIf any of the intermediate values are nil, it will raise:\n\n    iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> put_in(users, [\"jane\", :age], \"oops\")\n    ** (ArgumentError) could not put/update key :age on a nil value","ref":"Kernel.html#put_in/3-examples"},{"type":"macro","title":"Kernel.raise/1","doc":"Raises an exception.\n\nIf `message` is a string, it raises a `RuntimeError` exception with it.\n\nIf `message` is an atom, it just calls `raise/2` with the atom as the first\nargument and `[]` as the second one.\n\nIf `message` is an exception struct, it is raised as is.\n\nIf `message` is anything else, `raise` will fail with an `ArgumentError`\nexception.","ref":"Kernel.html#raise/1"},{"type":"macro","title":"Examples - Kernel.raise/1","doc":"iex> raise \"oops\"\n    ** (RuntimeError) oops\n\n    try do\n      1 + :foo\n    rescue\n      x in [ArithmeticError] ->\n        IO.puts(\"that was expected\")\n        raise x\n    end","ref":"Kernel.html#raise/1-examples"},{"type":"macro","title":"Kernel.raise/2","doc":"Raises an exception.\n\nCalls the `exception/1` function on the given argument (which has to be a\nmodule name like `ArgumentError` or `RuntimeError`) passing `attributes`\nin order to retrieve the exception struct.\n\nAny module that contains a call to the `defexception/1` macro automatically\nimplements the `c:Exception.exception/1` callback expected by `raise/2`.\nFor more information, see `defexception/1`.","ref":"Kernel.html#raise/2"},{"type":"macro","title":"Examples - Kernel.raise/2","doc":"iex> raise(ArgumentError, \"Sample\")\n    ** (ArgumentError) Sample","ref":"Kernel.html#raise/2-examples"},{"type":"function","title":"Kernel.rem/2","doc":"Computes the remainder of an integer division.\n\n`rem/2` uses truncated division, which means that\nthe result will always have the sign of the `dividend`.\n\nRaises an `ArithmeticError` exception if one of the arguments is not an\ninteger, or when the `divisor` is `0`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#rem/2"},{"type":"function","title":"Examples - Kernel.rem/2","doc":"iex> rem(5, 2)\n    1\n    iex> rem(6, -4)\n    2","ref":"Kernel.html#rem/2-examples"},{"type":"macro","title":"Kernel.reraise/2","doc":"Raises an exception preserving a previous stacktrace.\n\nWorks like `raise/1` but does not generate a new stacktrace.\n\nNote that `__STACKTRACE__` can be used inside catch/rescue\nto retrieve the current stacktrace.","ref":"Kernel.html#reraise/2"},{"type":"macro","title":"Examples - Kernel.reraise/2","doc":"try do\n      raise \"oops\"\n    rescue\n      exception ->\n        reraise exception, __STACKTRACE__\n    end","ref":"Kernel.html#reraise/2-examples"},{"type":"macro","title":"Kernel.reraise/3","doc":"Raises an exception preserving a previous stacktrace.\n\n`reraise/3` works like `reraise/2`, except it passes arguments to the\n`exception/1` function as explained in `raise/2`.","ref":"Kernel.html#reraise/3"},{"type":"macro","title":"Examples - Kernel.reraise/3","doc":"try do\n      raise \"oops\"\n    rescue\n      exception ->\n        reraise WrapperError, [exception: exception], __STACKTRACE__\n    end","ref":"Kernel.html#reraise/3-examples"},{"type":"function","title":"Kernel.round/1","doc":"Rounds a number to the nearest integer.\n\nIf the number is equidistant to the two nearest integers, rounds away from zero.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#round/1"},{"type":"function","title":"Examples - Kernel.round/1","doc":"iex> round(5.6)\n    6\n\n    iex> round(5.2)\n    5\n\n    iex> round(-9.9)\n    -10\n\n    iex> round(-9)\n    -9\n\n    iex> round(2.5)\n    3\n\n    iex> round(-2.5)\n    -3","ref":"Kernel.html#round/1-examples"},{"type":"function","title":"Kernel.self/0","doc":"Returns the PID (process identifier) of the calling process.\n\nAllowed in guard clauses. Inlined by the compiler.","ref":"Kernel.html#self/0"},{"type":"function","title":"Kernel.send/2","doc":"Sends a message to the given `dest` and returns the message.\n\n`dest` may be a remote or local PID, a local port, a locally\nregistered name, or a tuple in the form of `{registered_name, node}` for a\nregistered name at another node.\n\nInlined by the compiler.","ref":"Kernel.html#send/2"},{"type":"function","title":"Examples - Kernel.send/2","doc":"iex> send(self(), :hello)\n    :hello","ref":"Kernel.html#send/2-examples"},{"type":"macro","title":"Kernel.sigil_C/2","doc":"Handles the sigil `~C` for charlists.\n\nIt returns a charlist without interpolations and without escape\ncharacters, except for the escaping of the closing sigil character\nitself.\n\nA charlist is a list of integers where all the integers are valid code points.\nThe three expressions below are equivalent:\n\n    ~C\"foo\\n\"\n    [?f, ?o, ?o, ?\\\\, ?n]\n    [102, 111, 111, 92, 110]\n\nIn practice, charlists are mostly used in specific scenarios such as\ninterfacing with older Erlang libraries that do not accept binaries as arguments.","ref":"Kernel.html#sigil_C/2"},{"type":"macro","title":"Examples - Kernel.sigil_C/2","doc":"iex> ~C(foo)\n    ~c\"foo\"\n\n    iex> ~C(f#{o}o)\n    ~c\"f\\#{o}o\"\n\n    iex> ~C(foo\\n)\n    ~c\"foo\\\\n\"","ref":"Kernel.html#sigil_C/2-examples"},{"type":"macro","title":"Kernel.sigil_c/2","doc":"Handles the sigil `~c` for charlists.\n\nIt returns a charlist, unescaping characters and replacing interpolations.\n\nA charlist is a list of integers where all the integers are valid code points.\nThe three expressions below are equivalent:\n\n    ~c\"foo\"\n    [?f, ?o, ?o]\n    [102, 111, 111]\n\nIn practice, charlists are mostly used in specific scenarios such as\ninterfacing with older Erlang libraries that do not accept binaries as arguments.","ref":"Kernel.html#sigil_c/2"},{"type":"macro","title":"Examples - Kernel.sigil_c/2","doc":"iex> ~c(foo)\n    ~c\"foo\"\n\n    iex> ~c(f#{:o}o)\n    ~c\"foo\"\n\n    iex> ~c(f\\#{:o}o)\n    ~c\"f\\#{:o}o\"\n\nThe list is only printed as a `~c` sigil if all code points are within the\nASCII range:\n\n    iex> ~c\"hełło\"\n    [104, 101, 322, 322, 111]\n\n    iex> [104, 101, 108, 108, 111]\n    ~c\"hello\"\n\nSee `Inspect.Opts` for more information.","ref":"Kernel.html#sigil_c/2-examples"},{"type":"macro","title":"Kernel.sigil_D/2","doc":"Handles the sigil `~D` for dates.\n\nBy default, this sigil uses the built-in `Calendar.ISO`, which\nrequires dates to be written in the ISO8601 format:\n\n    ~D[yyyy-mm-dd]\n\nsuch as:\n\n    ~D[2015-01-13]\n\nIf you are using alternative calendars, any representation can\nbe used as long as you follow the representation by a single space\nand the calendar name:\n\n    ~D[SOME-REPRESENTATION My.Alternative.Calendar]\n\nThe lower case `~d` variant does not exist as interpolation\nand escape characters are not useful for date sigils.\n\nMore information on dates can be found in the `Date` module.","ref":"Kernel.html#sigil_D/2"},{"type":"macro","title":"Examples - Kernel.sigil_D/2","doc":"iex> ~D[2015-01-13]\n    ~D[2015-01-13]","ref":"Kernel.html#sigil_D/2-examples"},{"type":"macro","title":"Kernel.sigil_N/2","doc":"Handles the sigil `~N` for naive date times.\n\nBy default, this sigil uses the built-in `Calendar.ISO`, which\nrequires naive date times to be written in the ISO8601 format:\n\n    ~N[yyyy-mm-dd hh:mm:ss]\n    ~N[yyyy-mm-dd hh:mm:ss.ssssss]\n    ~N[yyyy-mm-ddThh:mm:ss.ssssss]\n\nsuch as:\n\n    ~N[2015-01-13 13:00:07]\n    ~N[2015-01-13T13:00:07.123]\n\nIf you are using alternative calendars, any representation can\nbe used as long as you follow the representation by a single space\nand the calendar name:\n\n    ~N[SOME-REPRESENTATION My.Alternative.Calendar]\n\nThe lower case `~n` variant does not exist as interpolation\nand escape characters are not useful for date time sigils.\n\nMore information on naive date times can be found in the\n`NaiveDateTime` module.","ref":"Kernel.html#sigil_N/2"},{"type":"macro","title":"Examples - Kernel.sigil_N/2","doc":"iex> ~N[2015-01-13 13:00:07]\n    ~N[2015-01-13 13:00:07]\n    iex> ~N[2015-01-13T13:00:07.001]\n    ~N[2015-01-13 13:00:07.001]","ref":"Kernel.html#sigil_N/2-examples"},{"type":"macro","title":"Kernel.sigil_R/2","doc":"Handles the sigil `~R` for regular expressions.\n\nIt returns a regular expression pattern without interpolations and\nwithout escape characters. Note it still supports escape of Regex\ntokens (such as escaping `+` or `?`) and it also requires you to\nescape the closing sigil character itself if it appears on the Regex.\n\nMore information on regexes can be found in the `Regex` module.","ref":"Kernel.html#sigil_R/2"},{"type":"macro","title":"Examples - Kernel.sigil_R/2","doc":"iex> Regex.match?(~R(f#{1,3}o), \"f#o\")\n    true","ref":"Kernel.html#sigil_R/2-examples"},{"type":"macro","title":"Kernel.sigil_r/2","doc":"Handles the sigil `~r` for regular expressions.\n\nIt returns a regular expression pattern, unescaping characters and replacing\ninterpolations.\n\nMore information on regular expressions can be found in the `Regex` module.","ref":"Kernel.html#sigil_r/2"},{"type":"macro","title":"Examples - Kernel.sigil_r/2","doc":"iex> Regex.match?(~r/foo/, \"foo\")\n    true\n\n    iex> Regex.match?(~r/a#{:b}c/, \"abc\")\n    true\n\nWhile the `~r` sigil allows parens and brackets to be used as delimiters,\nit is preferred to use `\"` or `/` to avoid escaping conflicts with reserved\nregex characters.","ref":"Kernel.html#sigil_r/2-examples"},{"type":"macro","title":"Kernel.sigil_S/2","doc":"Handles the sigil `~S` for strings.\n\nIt returns a string without interpolations and without escape\ncharacters, except for the escaping of the closing sigil character\nitself.","ref":"Kernel.html#sigil_S/2"},{"type":"macro","title":"Examples - Kernel.sigil_S/2","doc":"iex> ~S(foo)\n    \"foo\"\n    iex> ~S(f#{o}o)\n    \"f\\#{o}o\"\n    iex> ~S(\\o/)\n    \"\\\\o/\"\n\nHowever, if you want to re-use the sigil character itself on\nthe string, you need to escape it:\n\n    iex> ~S((\\))\n    \"()\"","ref":"Kernel.html#sigil_S/2-examples"},{"type":"macro","title":"Kernel.sigil_s/2","doc":"Handles the sigil `~s` for strings.\n\nIt returns a string as if it was a double quoted string, unescaping characters\nand replacing interpolations.","ref":"Kernel.html#sigil_s/2"},{"type":"macro","title":"Examples - Kernel.sigil_s/2","doc":"iex> ~s(foo)\n    \"foo\"\n\n    iex> ~s(f#{:o}o)\n    \"foo\"\n\n    iex> ~s(f\\#{:o}o)\n    \"f\\#{:o}o\"","ref":"Kernel.html#sigil_s/2-examples"},{"type":"macro","title":"Kernel.sigil_T/2","doc":"Handles the sigil `~T` for times.\n\nBy default, this sigil uses the built-in `Calendar.ISO`, which\nrequires times to be written in the ISO8601 format:\n\n    ~T[hh:mm:ss]\n    ~T[hh:mm:ss.ssssss]\n\nsuch as:\n\n    ~T[13:00:07]\n    ~T[13:00:07.123]\n\nIf you are using alternative calendars, any representation can\nbe used as long as you follow the representation by a single space\nand the calendar name:\n\n    ~T[SOME-REPRESENTATION My.Alternative.Calendar]\n\nThe lower case `~t` variant does not exist as interpolation\nand escape characters are not useful for time sigils.\n\nMore information on times can be found in the `Time` module.","ref":"Kernel.html#sigil_T/2"},{"type":"macro","title":"Examples - Kernel.sigil_T/2","doc":"iex> ~T[13:00:07]\n    ~T[13:00:07]\n    iex> ~T[13:00:07.001]\n    ~T[13:00:07.001]","ref":"Kernel.html#sigil_T/2-examples"},{"type":"macro","title":"Kernel.sigil_U/2","doc":"Handles the sigil `~U` to create a UTC `DateTime`.\n\nBy default, this sigil uses the built-in `Calendar.ISO`, which\nrequires UTC date times to be written in the ISO8601 format:\n\n    ~U[yyyy-mm-dd hh:mm:ssZ]\n    ~U[yyyy-mm-dd hh:mm:ss.ssssssZ]\n    ~U[yyyy-mm-ddThh:mm:ss.ssssss+00:00]\n\nsuch as:\n\n    ~U[2015-01-13 13:00:07Z]\n    ~U[2015-01-13T13:00:07.123+00:00]\n\nIf you are using alternative calendars, any representation can\nbe used as long as you follow the representation by a single space\nand the calendar name:\n\n    ~U[SOME-REPRESENTATION My.Alternative.Calendar]\n\nThe given `datetime_string` must include \"Z\" or \"00:00\" offset\nwhich marks it as UTC, otherwise an error is raised.\n\nThe lower case `~u` variant does not exist as interpolation\nand escape characters are not useful for date time sigils.\n\nMore information on date times can be found in the `DateTime` module.","ref":"Kernel.html#sigil_U/2"},{"type":"macro","title":"Examples - Kernel.sigil_U/2","doc":"iex> ~U[2015-01-13 13:00:07Z]\n    ~U[2015-01-13 13:00:07Z]\n    iex> ~U[2015-01-13T13:00:07.001+00:00]\n    ~U[2015-01-13 13:00:07.001Z]","ref":"Kernel.html#sigil_U/2-examples"},{"type":"macro","title":"Kernel.sigil_W/2","doc":"Handles the sigil `~W` for list of words.\n\nIt returns a list of \"words\" split by whitespace without interpolations\nand without escape characters, except for the escaping of the closing\nsigil character itself.","ref":"Kernel.html#sigil_W/2"},{"type":"macro","title":"Modifiers - Kernel.sigil_W/2","doc":"* `s`: words in the list are strings (default)\n  * `a`: words in the list are atoms\n  * `c`: words in the list are charlists","ref":"Kernel.html#sigil_W/2-modifiers"},{"type":"macro","title":"Examples - Kernel.sigil_W/2","doc":"iex> ~W(foo #{bar} baz)\n    [\"foo\", \"\\#{bar}\", \"baz\"]","ref":"Kernel.html#sigil_W/2-examples"},{"type":"macro","title":"Kernel.sigil_w/2","doc":"Handles the sigil `~w` for list of words.\n\nIt returns a list of \"words\" split by whitespace. Character unescaping and\ninterpolation happens for each word.","ref":"Kernel.html#sigil_w/2"},{"type":"macro","title":"Modifiers - Kernel.sigil_w/2","doc":"* `s`: words in the list are strings (default)\n  * `a`: words in the list are atoms\n  * `c`: words in the list are charlists","ref":"Kernel.html#sigil_w/2-modifiers"},{"type":"macro","title":"Examples - Kernel.sigil_w/2","doc":"iex> ~w(foo #{:bar} baz)\n    [\"foo\", \"bar\", \"baz\"]\n\n    iex> ~w(foo #{\" bar baz \"})\n    [\"foo\", \"bar\", \"baz\"]\n\n    iex> ~w(--source test/enum_test.exs)\n    [\"--source\", \"test/enum_test.exs\"]\n\n    iex> ~w(foo bar baz)a\n    [:foo, :bar, :baz]\n\n    iex> ~w(foo bar baz)c\n    [~c\"foo\", ~c\"bar\", ~c\"baz\"]","ref":"Kernel.html#sigil_w/2-examples"},{"type":"function","title":"Kernel.spawn/1","doc":"Spawns the given function and returns its PID.\n\nTypically developers do not use the `spawn` functions, instead they use\nabstractions such as `Task`, `GenServer` and `Agent`, built on top of\n`spawn`, that spawns processes with more conveniences in terms of\nintrospection and debugging.\n\nCheck the `Process` module for more process-related functions.\n\nThe anonymous function receives 0 arguments, and may return any value.\n\nInlined by the compiler.","ref":"Kernel.html#spawn/1"},{"type":"function","title":"Examples - Kernel.spawn/1","doc":"current = self()\n    child = spawn(fn -> send(current, {self(), 1 + 2}) end)\n\n    receive do\n      {^child, 3} -> IO.puts(\"Received 3 back\")\n    end","ref":"Kernel.html#spawn/1-examples"},{"type":"function","title":"Kernel.spawn/3","doc":"Spawns the given function `fun` from the given `module` passing it the given\n`args` and returns its PID.\n\nTypically developers do not use the `spawn` functions, instead they use\nabstractions such as `Task`, `GenServer` and `Agent`, built on top of\n`spawn`, that spawns processes with more conveniences in terms of\nintrospection and debugging.\n\nCheck the `Process` module for more process-related functions.\n\nInlined by the compiler.","ref":"Kernel.html#spawn/3"},{"type":"function","title":"Examples - Kernel.spawn/3","doc":"spawn(SomeModule, :function, [1, 2, 3])","ref":"Kernel.html#spawn/3-examples"},{"type":"function","title":"Kernel.spawn_link/1","doc":"Spawns the given function, links it to the current process, and returns its PID.\n\nTypically developers do not use the `spawn` functions, instead they use\nabstractions such as `Task`, `GenServer` and `Agent`, built on top of\n`spawn`, that spawns processes with more conveniences in terms of\nintrospection and debugging.\n\nCheck the `Process` module for more process-related functions. For more\ninformation on linking, check `Process.link/1`.\n\nThe anonymous function receives 0 arguments, and may return any value.\n\nInlined by the compiler.","ref":"Kernel.html#spawn_link/1"},{"type":"function","title":"Examples - Kernel.spawn_link/1","doc":"current = self()\n    child = spawn_link(fn -> send(current, {self(), 1 + 2}) end)\n\n    receive do\n      {^child, 3} -> IO.puts(\"Received 3 back\")\n    end","ref":"Kernel.html#spawn_link/1-examples"},{"type":"function","title":"Kernel.spawn_link/3","doc":"Spawns the given function `fun` from the given `module` passing it the given\n`args`, links it to the current process, and returns its PID.\n\nTypically developers do not use the `spawn` functions, instead they use\nabstractions such as `Task`, `GenServer` and `Agent`, built on top of\n`spawn`, that spawns processes with more conveniences in terms of\nintrospection and debugging.\n\nCheck the `Process` module for more process-related functions. For more\ninformation on linking, check `Process.link/1`.\n\nInlined by the compiler.","ref":"Kernel.html#spawn_link/3"},{"type":"function","title":"Examples - Kernel.spawn_link/3","doc":"spawn_link(SomeModule, :function, [1, 2, 3])","ref":"Kernel.html#spawn_link/3-examples"},{"type":"function","title":"Kernel.spawn_monitor/1","doc":"Spawns the given function, monitors it and returns its PID\nand monitoring reference.\n\nTypically developers do not use the `spawn` functions, instead they use\nabstractions such as `Task`, `GenServer` and `Agent`, built on top of\n`spawn`, that spawns processes with more conveniences in terms of\nintrospection and debugging.\n\nCheck the `Process` module for more process-related functions.\n\nThe anonymous function receives 0 arguments, and may return any value.\n\nInlined by the compiler.","ref":"Kernel.html#spawn_monitor/1"},{"type":"function","title":"Examples - Kernel.spawn_monitor/1","doc":"current = self()\n    spawn_monitor(fn -> send(current, {self(), 1 + 2}) end)","ref":"Kernel.html#spawn_monitor/1-examples"},{"type":"function","title":"Kernel.spawn_monitor/3","doc":"Spawns the given module and function passing the given args,\nmonitors it and returns its PID and monitoring reference.\n\nTypically developers do not use the `spawn` functions, instead they use\nabstractions such as `Task`, `GenServer` and `Agent`, built on top of\n`spawn`, that spawns processes with more conveniences in terms of\nintrospection and debugging.\n\nCheck the `Process` module for more process-related functions.\n\nInlined by the compiler.","ref":"Kernel.html#spawn_monitor/3"},{"type":"function","title":"Examples - Kernel.spawn_monitor/3","doc":"spawn_monitor(SomeModule, :function, [1, 2, 3])","ref":"Kernel.html#spawn_monitor/3-examples"},{"type":"function","title":"Kernel.struct/2","doc":"Creates and updates a struct.\n\nThe `struct` argument may be an atom (which defines `defstruct`)\nor a `struct` itself. The second argument is any `Enumerable` that\nemits two-element tuples (key-value pairs) during enumeration.\n\nKeys in the `Enumerable` that don't exist in the struct are automatically\ndiscarded. Note that keys must be atoms, as only atoms are allowed when\ndefining a struct. If keys in the `Enumerable` are duplicated, the last\nentry will be taken (same behaviour as `Map.new/1`).\n\nThis function is useful for dynamically creating and updating structs, as\nwell as for converting maps to structs; in the latter case, just inserting\nthe appropriate `:__struct__` field into the map may not be enough and\n`struct/2` should be used instead.","ref":"Kernel.html#struct/2"},{"type":"function","title":"Examples - Kernel.struct/2","doc":"defmodule User do\n      defstruct name: \"john\"\n    end\n\n    struct(User)\n    #=> %User{name: \"john\"}\n\n    opts = [name: \"meg\"]\n    user = struct(User, opts)\n    #=> %User{name: \"meg\"}\n\n    struct(user, unknown: \"value\")\n    #=> %User{name: \"meg\"}\n\n    struct(User, %{name: \"meg\"})\n    #=> %User{name: \"meg\"}\n\n    # String keys are ignored\n    struct(User, %{\"name\" => \"meg\"})\n    #=> %User{name: \"john\"}","ref":"Kernel.html#struct/2-examples"},{"type":"function","title":"Kernel.struct!/2","doc":"Similar to `struct/2` but checks for key validity.\n\nThe function `struct!/2` emulates the compile time behaviour\nof structs. This means that:\n\n  * when building a struct, as in `struct!(SomeStruct, key: :value)`,\n    it is equivalent to `%SomeStruct{key: :value}` and therefore this\n    function will check if every given key-value belongs to the struct.\n    If the struct is enforcing any key via `@enforce_keys`, those will\n    be enforced as well;\n\n  * when updating a struct, as in `struct!(%SomeStruct{}, key: :value)`,\n    it is equivalent to `%SomeStruct{struct | key: :value}` and therefore this\n    function will check if every given key-value belongs to the struct.\n    However, updating structs does not enforce keys, as keys are enforced\n    only when building;","ref":"Kernel.html#struct!/2"},{"type":"macro","title":"Kernel.tap/2","doc":"Pipes the first argument, `value`, into the second argument, a function `fun`,\nand returns `value` itself.\n\nUseful for running synchronous side effects in a pipeline, using the `|>/2` operator.","ref":"Kernel.html#tap/2"},{"type":"macro","title":"Examples - Kernel.tap/2","doc":"iex> tap(1, fn x -> x + 1 end)\n    1\n\nMost commonly, this is used in pipelines, using the `|>/2` operator.\nFor example, let's suppose you want to inspect part of a data structure.\nYou could write:\n\n    %{a: 1}\n    |> Map.update!(:a, & &1 + 2)\n    |> tap(&IO.inspect(&1.a))\n    |> Map.update!(:a, & &1 * 2)","ref":"Kernel.html#tap/2-examples"},{"type":"macro","title":"Kernel.then/2","doc":"Pipes the first argument, `value`, into the second argument, a function `fun`,\nand returns the result of calling `fun`.\n\nIn other words, it invokes the function `fun` with `value` as argument,\nand returns its result.\n\nThis is most commonly used in pipelines, using the `|>/2` operator, allowing you\nto pipe a value to a function outside of its first argument.\n\n#","ref":"Kernel.html#then/2"},{"type":"macro","title":"Examples - Kernel.then/2","doc":"iex> 1 |> then(fn x -> x * 2 end)\n    2\n\n    iex> 1 |> then(fn x -> Enum.drop([\"a\", \"b\", \"c\"], x) end)\n    [\"b\", \"c\"]","ref":"Kernel.html#then/2-examples"},{"type":"function","title":"Kernel.throw/1","doc":"A non-local return from a function.\n\nUsing `throw/1` is generally discouraged, as it allows a function\nto escape from its regular execution flow, which can make the code\nharder to read. Furthermore, all thrown values must be caught by\n`try/catch`. See `try/1` for more information.\n\nInlined by the compiler.","ref":"Kernel.html#throw/1"},{"type":"function","title":"Kernel.tl/1","doc":"Returns the tail of a list. Raises `ArgumentError` if the list is empty.\n\nThe tail of a list is the list without its first element.\n\nIt works with improper lists.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#tl/1"},{"type":"function","title":"Examples - Kernel.tl/1","doc":"tl([1, 2, 3, :go])\n    #=> [2, 3, :go]\n\n    tl([:one])\n    #=> []\n\n    tl([:a, :b | :improper_end])\n    #=> [:b | :improper_end]\n\n    tl([:a | %{b: 1}])\n    #=> %{b: 1}\n\nGiving it an empty list raises:\n\n    tl([])\n    ** (ArgumentError) argument error","ref":"Kernel.html#tl/1-examples"},{"type":"macro","title":"Kernel.to_charlist/1","doc":"Converts the given term to a charlist according to the `List.Chars` protocol.","ref":"Kernel.html#to_charlist/1"},{"type":"macro","title":"Examples - Kernel.to_charlist/1","doc":"iex> to_charlist(:foo)\n    ~c\"foo\"","ref":"Kernel.html#to_charlist/1-examples"},{"type":"macro","title":"Kernel.to_string/1","doc":"Converts the argument to a string according to the\n`String.Chars` protocol.\n\nThis is the function invoked when there is string interpolation.","ref":"Kernel.html#to_string/1"},{"type":"macro","title":"Examples - Kernel.to_string/1","doc":"iex> to_string(:foo)\n    \"foo\"","ref":"Kernel.html#to_string/1-examples"},{"type":"function","title":"Kernel.trunc/1","doc":"Returns the integer part of `number`.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#trunc/1"},{"type":"function","title":"Examples - Kernel.trunc/1","doc":"iex> trunc(5.4)\n    5\n\n    iex> trunc(-5.99)\n    -5\n\n    iex> trunc(-5)\n    -5","ref":"Kernel.html#trunc/1-examples"},{"type":"function","title":"Kernel.tuple_size/1","doc":"Returns the size of a tuple.\n\nThis operation happens in constant time.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Kernel.html#tuple_size/1"},{"type":"function","title":"Examples - Kernel.tuple_size/1","doc":"iex> tuple_size({:a, :b, :c})\n    3","ref":"Kernel.html#tuple_size/1-examples"},{"type":"macro","title":"Kernel.unless/2","doc":"Provides an `unless` macro.\n\nThis macro evaluates and returns the `do` block passed in as the second\nargument if `condition` evaluates to a falsy value (`false` or `nil`).\nOtherwise, it returns the value of the `else` block if present or `nil` if not.\n\nSee also `if/2`.","ref":"Kernel.html#unless/2"},{"type":"macro","title":"Examples - Kernel.unless/2","doc":"iex> unless(Enum.empty?([]), do: \"Hello\")\n    nil\n\n    iex> unless(Enum.empty?([1, 2, 3]), do: \"Hello\")\n    \"Hello\"\n\n    iex> unless Enum.sum([2, 2]) == 5 do\n    ...>   \"Math still works\"\n    ...> else\n    ...>   \"Math is broken\"\n    ...> end\n    \"Math still works\"","ref":"Kernel.html#unless/2-examples"},{"type":"macro","title":"Kernel.update_in/2","doc":"Updates a nested structure via the given `path`.\n\nThis is similar to `update_in/3`, except the path is extracted via\na macro rather than passing a list. For example:\n\n    update_in(opts[:foo][:bar], &(&1 + 1))\n\nIs equivalent to:\n\n    update_in(opts, [:foo, :bar], &(&1 + 1))\n\nThis also works with nested structs and the `struct.path.to.value` way to specify\npaths:\n\n    update_in(struct.foo.bar, &(&1 + 1))\n\nNote that in order for this macro to work, the complete path must always\nbe visible by this macro. For more information about the supported path\nexpressions, please check `get_and_update_in/2` docs.","ref":"Kernel.html#update_in/2"},{"type":"macro","title":"Examples - Kernel.update_in/2","doc":"iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> update_in(users[\"john\"][:age], &(&1 + 1))\n    %{\"john\" => %{age: 28}, \"meg\" => %{age: 23}}\n\n    iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> update_in(users[\"john\"].age, &(&1 + 1))\n    %{\"john\" => %{age: 28}, \"meg\" => %{age: 23}}","ref":"Kernel.html#update_in/2-examples"},{"type":"function","title":"Kernel.update_in/3","doc":"Updates a key in a nested structure.\n\nUses the `Access` module to traverse the structures\naccording to the given `keys`, unless the `key` is a\nfunction. If the key is a function, it will be invoked\nas specified in `get_and_update_in/3`.\n\n`data` is a nested structure (that is, a map, keyword\nlist, or struct that implements the `Access` behaviour).\nThe `fun` argument receives the value of `key` (or `nil`\nif `key` is not present) and the result replaces the value\nin the structure.","ref":"Kernel.html#update_in/3"},{"type":"function","title":"Examples - Kernel.update_in/3","doc":"iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> update_in(users, [\"john\", :age], &(&1 + 1))\n    %{\"john\" => %{age: 28}, \"meg\" => %{age: 23}}\n\nNote the current value given to the anonymous function may be `nil`.\nIf any of the intermediate values are nil, it will raise:\n\n    iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> update_in(users, [\"jane\", :age], & &1 + 1)\n    ** (ArgumentError) could not put/update key :age on a nil value","ref":"Kernel.html#update_in/3-examples"},{"type":"macro","title":"Kernel.use/2","doc":"Uses the given module in the current context.\n\nWhen calling:\n\n    use MyModule, some: :options\n\nElixir will invoke `MyModule.__using__/1` passing the second argument of\n`use` as its argument. Since `__using__/1` is typically a macro, all\nthe usual macro rules apply, and its return value should be quoted code\nthat is then inserted where `use/2` is called.\n\n> ##","ref":"Kernel.html#use/2"},{"type":"macro","title":"Code injection {: .warning} - Kernel.use/2","doc":">\n> `use MyModule` works as a **code-injection point** in the caller.\n> Given the caller of `use MyModule` has little control over how the\n> code is injected, `use/2` should be used with care. If you can,\n> avoid use in favor of `import/2` or `alias/2` whenever possible.","ref":"Kernel.html#use/2-code-injection-warning"},{"type":"macro","title":"Examples - Kernel.use/2","doc":"For example, to write test cases using the `ExUnit` framework provided\nwith Elixir, a developer should `use` the `ExUnit.Case` module:\n\n    defmodule AssertionTest do\n      use ExUnit.Case, async: true\n\n      test \"always pass\" do\n        assert true\n      end\n    end\n\nIn this example, Elixir will call the `__using__/1` macro in the\n`ExUnit.Case` module with the keyword list `[async: true]` as its\nargument.\n\nIn other words, `use/2` translates to:\n\n    defmodule AssertionTest do\n      require ExUnit.Case\n      ExUnit.Case.__using__(async: true)\n\n      test \"always pass\" do\n        assert true\n      end\n    end\n\nwhere `ExUnit.Case` defines the `__using__/1` macro:\n\n    defmodule ExUnit.Case do\n      defmacro __using__(opts) do\n        # do something with opts\n        quote do\n          # return some code to inject in the caller\n        end\n      end\n    end","ref":"Kernel.html#use/2-examples"},{"type":"macro","title":"Best practices - Kernel.use/2","doc":"`__using__/1` is typically used when there is a need to set some state\n(via module attributes) or callbacks (like `@before_compile`, see the\ndocumentation for `Module` for more information) into the caller.\n\n`__using__/1` may also be used to alias, require, or import functionality\nfrom different modules:\n\n    defmodule MyModule do\n      defmacro __using__(_opts) do\n        quote do\n          import MyModule.Foo\n          import MyModule.Bar\n          import MyModule.Baz\n\n          alias MyModule.Repo\n        end\n      end\n    end\n\nHowever, do not provide `__using__/1` if all it does is to import,\nalias or require the module itself. For example, avoid this:\n\n    defmodule MyModule do\n      defmacro __using__(_opts) do\n        quote do\n          import MyModule\n        end\n      end\n    end\n\nIn such cases, developers should instead import or alias the module\ndirectly, so that they can customize those as they wish,\nwithout the indirection behind `use/2`. Developers must also avoid\ndefining functions inside `__using__/1`.\n\nGiven `use MyModule` can generate any code, it may not be easy for\ndevelopers to understand the impact of `use MyModule`.\n\nFor this reason, to provide guidance and clarity, we recommend developers\nto include an admonition block in their `@moduledoc` that explains how\n`use MyModule` impacts their code. As an example, the `GenServer` documentation\noutlines:\n\n> #### `use GenServer` {: .info}\n>\n> When you `use GenServer`, the `GenServer` module will\n> set `@behaviour GenServer` and define a `child_spec/1`\n> function, so your module can be used as a child\n> in a supervision tree.\n\nThis provides a quick summary of how using a module impacts the user code.\nKeep in mind to only list changes made to the public API of the module.\nFor example, if `use MyModule` sets an internal attribute called\n`@_my_module_info` and this attribute is never meant to be public,\nit must not be listed.\n\nFor convenience, the markup notation to generate the admonition block\nabove is:\n\n```\n> #### `use GenServer` {: .info}\n>\n> When you `use GenServer`, the GenServer module will\n> set `@behaviour GenServer` and define a `child_spec/1`\n> function, so your module can be used as a child\n> in a supervision tree.\n```","ref":"Kernel.html#use/2-best-practices"},{"type":"macro","title":"Kernel.var!/2","doc":"Marks that the given variable should not be hygienized.\n\nThis macro expects a variable and it is typically invoked\ninside `Kernel.SpecialForms.quote/2` to mark that a variable\nshould not be hygienized. See `Kernel.SpecialForms.quote/2`\nfor more information.","ref":"Kernel.html#var!/2"},{"type":"macro","title":"Examples - Kernel.var!/2","doc":"iex> Kernel.var!(example) = 1\n    1\n    iex> Kernel.var!(example)\n    1","ref":"Kernel.html#var!/2-examples"},{"type":"macro","title":"Kernel.|>/2","doc":"Pipe operator.\n\nThis operator introduces the expression on the left-hand side as\nthe first argument to the function call on the right-hand side.","ref":"Kernel.html#%7C%3E/2"},{"type":"macro","title":"Examples - Kernel.|>/2","doc":"iex> [1, [2], 3] |> List.flatten()\n    [1, 2, 3]\n\nThe example above is the same as calling `List.flatten([1, [2], 3])`.\n\nThe `|>/2` operator is mostly useful when there is a desire to execute a series\nof operations resembling a pipeline:\n\n    iex> [1, [2], 3] |> List.flatten() |> Enum.map(fn x -> x * 2 end)\n    [2, 4, 6]\n\nIn the example above, the list `[1, [2], 3]` is passed as the first argument\nto the `List.flatten/1` function, then the flattened list is passed as the\nfirst argument to the `Enum.map/2` function which doubles each element of the\nlist.\n\nIn other words, the expression above simply translates to:\n\n    Enum.map(List.flatten([1, [2], 3]), fn x -> x * 2 end)","ref":"Kernel.html#%7C%3E/2-examples"},{"type":"macro","title":"Pitfalls - Kernel.|>/2","doc":"There are two common pitfalls when using the pipe operator.\n\nThe first one is related to operator precedence. For example,\nthe following expression:\n\n    String.graphemes \"Hello\" |> Enum.reverse\n\nTranslates to:\n\n    String.graphemes(\"Hello\" |> Enum.reverse())\n\nwhich results in an error as the `Enumerable` protocol is not defined\nfor binaries. Adding explicit parentheses resolves the ambiguity:\n\n    String.graphemes(\"Hello\") |> Enum.reverse()\n\nOr, even better:\n\n    \"Hello\" |> String.graphemes() |> Enum.reverse()\n\nThe second limitation is that Elixir always pipes to a function\ncall. Therefore, to pipe into an anonymous function, you need to\ninvoke it:\n\n    some_fun = &Regex.replace(~r/l/, &1, \"L\")\n    \"Hello\" |> some_fun.()\n\nAlternatively, you can use `then/2` for the same effect:\n\n    some_fun = &Regex.replace(~r/l/, &1, \"L\")\n    \"Hello\" |> then(some_fun)\n\n`then/2` is most commonly used when you want to pipe to a function\nbut the value is expected outside of the first argument, such as\nabove. By replacing `some_fun` by its value, we get:\n\n    \"Hello\" |> then(&Regex.replace(~r/l/, &1, \"L\"))","ref":"Kernel.html#%7C%3E/2-pitfalls"},{"type":"macro","title":"Kernel.||/2","doc":"Boolean \"or\" operator.\n\nProvides a short-circuit operator that evaluates and returns the second\nexpression only if the first one does not evaluate to a truthy value (that is,\nit is either `nil` or `false`). Returns the first expression otherwise.\n\nNot allowed in guard clauses.","ref":"Kernel.html#%7C%7C/2"},{"type":"macro","title":"Examples - Kernel.||/2","doc":"iex> Enum.empty?([1]) || Enum.empty?([1])\n    false\n\n    iex> List.first([]) || true\n    true\n\n    iex> Enum.empty?([1]) || 1\n    1\n\n    iex> Enum.empty?([]) || throw(:bad)\n    true\n\nNote that, unlike `or/2`, this operator accepts any expression\nas the first argument, not only booleans.","ref":"Kernel.html#%7C%7C/2-examples"},{"type":"module","title":"Kernel.SpecialForms","doc":"Special forms are the basic building blocks of Elixir, and therefore\ncannot be overridden by the developer.\n\nThe `Kernel.SpecialForms` module consists solely of macros that can be\ninvoked anywhere in Elixir code without the use of the\n`Kernel.SpecialForms.` prefix. This is possible because they all have\nbeen automatically imported, in the same fashion as the functions and\nmacros from the `Kernel` module.\n\nThese building blocks are defined in this module. Some of these special forms are lexical (such as\n`alias/2` and `case/2`). The macros `{}/1` and ` >/1` are also special\nforms used to define tuple and binary data structures respectively.\n\nThis module also documents macros that return information about Elixir's\ncompilation environment, such as (`__ENV__/0`, `__MODULE__/0`, `__DIR__/0`,\n`__STACKTRACE__/0`, and `__CALLER__/0`).\n\nAdditionally, it documents two special forms, `__block__/1` and\n`__aliases__/1`, which are not intended to be called directly by the\ndeveloper but they appear in quoted contents since they are essential\nin Elixir's constructs.","ref":"Kernel.SpecialForms.html"},{"type":"macro","title":"Kernel.SpecialForms.%/2","doc":"Matches on or builds a struct.\n\nA struct is a tagged map that allows developers to provide\ndefault values for keys, tags to be used in polymorphic\ndispatches and compile time assertions.\n\nStructs are usually defined with the `Kernel.defstruct/1` macro:\n\n    defmodule User do\n      defstruct name: \"john\", age: 27\n    end\n\nNow a struct can be created as follows:\n\n    %User{}\n\nUnderneath a struct is just a map with a `:__struct__` key\npointing to the `User` module:\n\n    %User{} == %{__struct__: User, name: \"john\", age: 27}\n\nThe struct fields can be given when building the struct:\n\n    %User{age: 31}\n    #=> %{__struct__: User, name: \"john\", age: 31}\n\nOr also on pattern matching to extract values out:\n\n    %User{age: age} = user\n\nAn update operation specific for structs is also available:\n\n    %User{user | age: 28}\n\nThe advantage of structs is that they validate that the given\nkeys are part of the defined struct. The example below will fail\nbecause there is no key `:full_name` in the `User` struct:\n\n    %User{full_name: \"john doe\"}\n\nThe syntax above will guarantee the given keys are valid at\ncompilation time and it will guarantee at runtime the given\nargument is a struct, failing with `BadStructError` otherwise.\n\nAlthough structs are maps, by default structs do not implement\nany of the protocols implemented for maps. Check\n`Kernel.defprotocol/2` for more information on how structs\ncan be used with protocols for polymorphic dispatch. Also\nsee `Kernel.struct/2` and `Kernel.struct!/2` for examples on\nhow to create and update structs dynamically.","ref":"Kernel.SpecialForms.html#%25/2"},{"type":"macro","title":"Pattern matching on struct names - Kernel.SpecialForms.%/2","doc":"Besides allowing pattern matching on struct fields, such as:\n\n    %User{age: age} = user\n\nStructs also allow pattern matching on the struct name:\n\n    %struct_name{} = user\n    struct_name #=> User\n\nYou can also assign the struct name to `_` when you want to\ncheck if something is a struct but you are not interested in\nits name:\n\n    %_{} = user","ref":"Kernel.SpecialForms.html#%25/2-pattern-matching-on-struct-names"},{"type":"macro","title":"Kernel.SpecialForms.%{}/1","doc":"Creates a map.\n\nSee the `Map` module for more information about maps, their syntax, and ways to\naccess and manipulate them.","ref":"Kernel.SpecialForms.html#%25%7B%7D/1"},{"type":"macro","title":"AST representation - Kernel.SpecialForms.%{}/1","doc":"Regardless of whether `=>` or the keyword syntax is used, key-value pairs in\nmaps are always represented internally as a list of two-element tuples for\nsimplicity:\n\n    iex> quote do\n    ...>   %{\"a\" => :b, c: :d}\n    ...> end\n    {:%{}, [], [{\"a\", :b}, {:c, :d}]}","ref":"Kernel.SpecialForms.html#%25%7B%7D/1-ast-representation"},{"type":"macro","title":"Kernel.SpecialForms.&/1","doc":"Capture operator. Captures or creates an anonymous function.","ref":"Kernel.SpecialForms.html#&/1"},{"type":"macro","title":"Capture - Kernel.SpecialForms.&/1","doc":"The capture operator is most commonly used to capture a\nfunction with given name and arity from a module:\n\n    iex> fun = &Kernel.is_atom/1\n    iex> fun.(:atom)\n    true\n    iex> fun.(\"string\")\n    false\n\nIn the example above, we captured `Kernel.is_atom/1` as an\nanonymous function and then invoked it.\n\nThe capture operator can also be used to capture local functions,\nincluding private ones, and imported functions by omitting the\nmodule name:\n\n    &local_function/1\n\nSee also `Function.capture/3`.","ref":"Kernel.SpecialForms.html#&/1-capture"},{"type":"macro","title":"Anonymous functions - Kernel.SpecialForms.&/1","doc":"The capture operator can also be used to partially apply\nfunctions, where `&1`, `&2` and so on can be used as value\nplaceholders. For example:\n\n    iex> double = &(&1 * 2)\n    iex> double.(2)\n    4\n\nIn other words, `&(&1 * 2)` is equivalent to `fn x -> x * 2 end`.\n\nWe can partially apply a remote function with placeholder:\n\n    iex> take_five = &Enum.take(&1, 5)\n    iex> take_five.(1..10)\n    [1, 2, 3, 4, 5]\n\nAnother example while using an imported or local function:\n\n    iex> first_elem = &elem(&1, 0)\n    iex> first_elem.({0, 1})\n    0\n\nThe `&` operator can be used with more complex expressions:\n\n    iex> fun = &(&1 + &2 + &3)\n    iex> fun.(1, 2, 3)\n    6\n\nAs well as with lists and tuples:\n\n    iex> fun = &{&1, &2}\n    iex> fun.(1, 2)\n    {1, 2}\n\n    iex> fun = &[&1 | &2]\n    iex> fun.(1, [2, 3])\n    [1, 2, 3]\n\nThe only restrictions when creating anonymous functions is that at\nleast one placeholder must be present, i.e. it must contain at least\n`&1`, and that block expressions are not supported:\n\n    # No placeholder, fails to compile.\n    &(:foo)\n\n    # Block expression, fails to compile.\n    &(&1; &2)","ref":"Kernel.SpecialForms.html#&/1-anonymous-functions"},{"type":"macro","title":"Kernel.SpecialForms../2","doc":"Dot operator. Defines a remote call, a call to an anonymous function, or an alias.\n\nThe dot (`.`) in Elixir can be used for remote calls:\n\n    iex> String.downcase(\"FOO\")\n    \"foo\"\n\nIn this example above, we have used `.` to invoke `downcase` in the\n`String` module, passing `\"FOO\"` as argument.\n\nThe dot may be used to invoke anonymous functions too:\n\n    iex> (fn n -> n end).(7)\n    7\n\nin which case there is a function on the left hand side.\n\nWe can also use the dot for creating aliases:\n\n    iex> Hello.World\n    Hello.World\n\nThis time, we have joined two aliases, defining the final alias\n`Hello.World`.","ref":"Kernel.SpecialForms.html#./2"},{"type":"macro","title":"Syntax - Kernel.SpecialForms../2","doc":"The right side of `.` may be a word starting with an uppercase letter, which represents\nan alias, a word starting with lowercase or underscore, any valid language\noperator or any name wrapped in single- or double-quotes. Those are all valid\nexamples:\n\n    iex> Kernel.Sample\n    Kernel.Sample\n\n    iex> Kernel.length([1, 2, 3])\n    3\n\n    iex> Kernel.+(1, 2)\n    3\n\n    iex> Kernel.\"+\"(1, 2)\n    3\n\nWrapping the function name in single- or double-quotes is always a\nremote call. Therefore `Kernel.\"Foo\"` will attempt to call the function \"Foo\"\nand not return the alias `Kernel.Foo`. This is done by design as module names\nare more strict than function names.\n\nWhen the dot is used to invoke an anonymous function there is only one\noperand, but it is still written using a postfix notation:\n\n    iex> negate = fn n -> -n end\n    iex> negate.(7)\n    -7","ref":"Kernel.SpecialForms.html#./2-syntax"},{"type":"macro","title":"Quoted expression - Kernel.SpecialForms../2","doc":"When `.` is used, the quoted expression may take two distinct\nforms. When the right side starts with a lowercase letter (or\nunderscore):\n\n    iex> quote do\n    ...>   String.downcase(\"FOO\")\n    ...> end\n    {{:., [], [{:__aliases__, [alias: false], [:String]}, :downcase]}, [], [\"FOO\"]}\n\nNote that we have an inner tuple, containing the atom `:.` representing\nthe dot as first element:\n\n    {:., [], [{:__aliases__, [alias: false], [:String]}, :downcase]}\n\nThis tuple follows the general quoted expression structure in Elixir,\nwith the name as first argument, some keyword list as metadata as second,\nand the list of arguments as third. In this case, the arguments are the\nalias `String` and the atom `:downcase`. The second argument in a remote call\nis **always** an atom.\n\nIn the case of calls to anonymous functions, the inner tuple with the dot\nspecial form has only one argument, reflecting the fact that the operator is\nunary:\n\n    iex> quote do\n    ...>   negate.(0)\n    ...> end\n    {{:., [], [{:negate, [], __MODULE__}]}, [], [0]}\n\nWhen the right side is an alias (i.e. starts with uppercase), we get instead:\n\n    iex> quote do\n    ...>   Hello.World\n    ...> end\n    {:__aliases__, [alias: false], [:Hello, :World]}\n\nWe go into more details about aliases in the `__aliases__/1` special form\ndocumentation.","ref":"Kernel.SpecialForms.html#./2-quoted-expression"},{"type":"macro","title":"Unquoting - Kernel.SpecialForms../2","doc":"We can also use unquote to generate a remote call in a quoted expression:\n\n    iex> x = :downcase\n    iex> quote do\n    ...>   String.unquote(x)(\"FOO\")\n    ...> end\n    {{:., [], [{:__aliases__, [alias: false], [:String]}, :downcase]}, [], [\"FOO\"]}\n\nSimilar to `Kernel.\"FUNCTION_NAME\"`, `unquote(x)` will always generate a remote call,\nindependent of the value of `x`. To generate an alias via the quoted expression,\none needs to rely on `Module.concat/2`:\n\n    iex> x = Sample\n    iex> quote do\n    ...>   Module.concat(String, unquote(x))\n    ...> end\n    {{:., [], [{:__aliases__, [alias: false], [:Module]}, :concat]}, [],\n     [{:__aliases__, [alias: false], [:String]}, Sample]}","ref":"Kernel.SpecialForms.html#./2-unquoting"},{"type":"macro","title":"Kernel.SpecialForms.__aliases__/1","doc":"Internal special form to hold aliases information.\n\nIt is usually compiled to an atom:\n\n    iex> quote do\n    ...>   Foo.Bar\n    ...> end\n    {:__aliases__, [alias: false], [:Foo, :Bar]}\n\nElixir represents `Foo.Bar` as `__aliases__` so calls can be\nunambiguously identified by the operator `:.`. For example:\n\n    iex> quote do\n    ...>   Foo.bar()\n    ...> end\n    {{:., [], [{:__aliases__, [alias: false], [:Foo]}, :bar]}, [], []}\n\nWhenever an expression iterator sees a `:.` as the tuple key,\nit can be sure that it represents a call and the second argument\nin the list is an atom.\n\nOn the other hand, aliases hold some properties:\n\n  1. The head element of aliases can be any term that must expand to\n     an atom at compilation time.\n\n  2. The tail elements of aliases are guaranteed to always be atoms.\n\n  3. When the head element of aliases is the atom `:Elixir`, no expansion happens.","ref":"Kernel.SpecialForms.html#__aliases__/1"},{"type":"macro","title":"Kernel.SpecialForms.__block__/1","doc":"Internal special form for block expressions.\n\nThis is the special form used whenever we have a block\nof expressions in Elixir. This special form is private\nand should not be invoked directly:\n\n    iex> quote do\n    ...>   1\n    ...>   2\n    ...>   3\n    ...> end\n    {:__block__, [], [1, 2, 3]}","ref":"Kernel.SpecialForms.html#__block__/1"},{"type":"macro","title":"Kernel.SpecialForms.__CALLER__/0","doc":"Returns the current calling environment as a `Macro.Env` struct.\n\nIn the environment you can access the filename, line numbers,\nset up aliases, the function and others.","ref":"Kernel.SpecialForms.html#__CALLER__/0"},{"type":"macro","title":"Kernel.SpecialForms.__DIR__/0","doc":"Returns the absolute path of the directory of the current file as a binary.\n\nAlthough the directory can be accessed as `Path.dirname(__ENV__.file)`,\nthis macro is a convenient shortcut.","ref":"Kernel.SpecialForms.html#__DIR__/0"},{"type":"macro","title":"Kernel.SpecialForms.__ENV__/0","doc":"Returns the current environment information as a `Macro.Env` struct.\n\nIn the environment you can access the current filename,\nline numbers, set up aliases, the current function and others.","ref":"Kernel.SpecialForms.html#__ENV__/0"},{"type":"macro","title":"Kernel.SpecialForms.__MODULE__/0","doc":"Returns the current module name as an atom or `nil` otherwise.\n\nAlthough the module can be accessed in the `__ENV__/0`, this macro\nis a convenient shortcut.","ref":"Kernel.SpecialForms.html#__MODULE__/0"},{"type":"macro","title":"Kernel.SpecialForms.__STACKTRACE__/0","doc":"Returns the stacktrace for the currently handled exception.\n\nIt is available only in the `catch` and `rescue` clauses of `try/1`\nexpressions.\n\nTo retrieve the stacktrace of the current process, use\n`Process.info(self(), :current_stacktrace)` instead.","ref":"Kernel.SpecialForms.html#__STACKTRACE__/0"},{"type":"macro","title":"Kernel.SpecialForms.::/2","doc":"Type operator. Used by types and bitstrings to specify types.\n\nThis operator is used in two distinct occasions in Elixir.\nIt is used in typespecs to specify the type of a variable,\nfunction or of a type itself:\n\n    @type number :: integer | float\n    @spec add(number, number) :: number\n\nIt may also be used in bit strings to specify the type\nof a given bit segment:\n\n     > = bits\n\nRead the documentation on the [Typespecs page](typespecs.md) and\n` >/1` for more information on typespecs and\nbitstrings respectively.","ref":"Kernel.SpecialForms.html#::/2"},{"type":"macro","title":"Kernel.SpecialForms.<<>>/1","doc":"Defines a new bitstring.","ref":"Kernel.SpecialForms.html#%3C%3C%3E%3E/1"},{"type":"macro","title":"Examples - Kernel.SpecialForms.<<>>/1","doc":"iex>  >\n     >","ref":"Kernel.SpecialForms.html#%3C%3C%3E%3E/1-examples"},{"type":"macro","title":"Types - Kernel.SpecialForms.<<>>/1","doc":"A bitstring is made of many segments and each segment has a\ntype. There are 9 types used in bitstrings:\n\n- `integer`\n- `float`\n- `bits` (alias for `bitstring`)\n- `bitstring`\n- `binary`\n- `bytes` (alias for `binary`)\n- `utf8`\n- `utf16`\n- `utf32`\n\nWhen no type is specified, the default is `integer`:\n\n    iex>  >\n     >\n\nElixir also accepts by default the segment to be a literal\nstring which expands to integers:\n\n    iex>  >\n     >\n\nBinaries need to be explicitly tagged as `binary`:\n\n    iex> rest = \"oo\"\n    iex>  >\n    \"foo\"\n\nThe `utf8`, `utf16`, and `utf32` types are for Unicode code points. They\ncan also be applied to literal strings and charlists:\n\n    iex>  >\n     >\n    iex>  >\n     >\n\nOtherwise we get an `ArgumentError` when constructing the binary:\n\n    rest = \"oo\"\n     >\n    ** (ArgumentError) argument error","ref":"Kernel.SpecialForms.html#%3C%3C%3E%3E/1-types"},{"type":"macro","title":"Options - Kernel.SpecialForms.<<>>/1","doc":"Many options can be given by using `-` as separator. Order is\narbitrary, so the following are all equivalent:\n\n     >\n     >\n     >\n     >\n     >\n     >\n     >\n\n#","ref":"Kernel.SpecialForms.html#%3C%3C%3E%3E/1-options"},{"type":"macro","title":"Unit and Size - Kernel.SpecialForms.<<>>/1","doc":"The length of the match is equal to the `unit` (a number of bits) times the\n`size` (the number of repeated segments of length `unit`).\n\nType      | Default Unit\n--------- | ------------\n`integer` | 1 bit\n`float`   | 1 bit\n`binary`  | 8 bits\n\nSizes for types are a bit more nuanced. The default size for integers is 8.\n\nFor floats, it is 64. For floats, `size * unit` must result in 16, 32, or 64,\ncorresponding to [IEEE 754](https://en.wikipedia.org/wiki/IEEE_floating_point)\nbinary16, binary32, and binary64, respectively.\n\nFor binaries, the default is the size of the binary. Only the last binary in a\nmatch can use the default size. All others must have their size specified\nexplicitly, even if the match is unambiguous. For example:\n\n    iex>  > =  >\n    \"Frank the Walrus\"\n    iex> {name, species}\n    {\"Frank\", \"Walrus\"}\n\nThe size can be a variable or any valid guard expression:\n\n    iex> name_size = 5\n    iex>  > =  >\n    iex> {name, species}\n    {\"Frank\", \"Walrus\"}\n\nThe size can access prior variables defined in the binary itself:\n\n    iex>  > =  >\n    iex> {name, species}\n    {\"Frank\", \"Walrus\"}\n\nHowever, it cannot access variables defined in the match outside of the binary/bitstring:\n\n    {name_size,  >} = {5,  >}\n    ** (CompileError): undefined variable \"name_size\" in bitstring segment\n\nFailing to specify the size for the non-last causes compilation to fail:\n\n     > =  >\n    ** (CompileError): a binary field without size is only allowed at the end of a binary pattern\n\n##","ref":"Kernel.SpecialForms.html#%3C%3C%3E%3E/1-unit-and-size"},{"type":"macro","title":"Shortcut Syntax - Kernel.SpecialForms.<<>>/1","doc":"Size and unit can also be specified using a syntax shortcut\nwhen passing integer values:\n\n    iex> x = 1\n    iex>  > ==  >\n    true\n    iex>  > ==  >\n    true\n\nThis syntax reflects the fact the effective size is given by\nmultiplying the size by the unit.\n\n#","ref":"Kernel.SpecialForms.html#%3C%3C%3E%3E/1-shortcut-syntax"},{"type":"macro","title":"Modifiers - Kernel.SpecialForms.<<>>/1","doc":"Some types have associated modifiers to clear up ambiguity in byte\nrepresentation.\n\nModifier             | Relevant Type(s)\n-------------------- | ----------------\n`signed`             | `integer`\n`unsigned` (default) | `integer`\n`little`             | `integer`, `float`, `utf16`, `utf32`\n`big` (default)      | `integer`, `float`, `utf16`, `utf32`\n`native`             | `integer`, `utf16`, `utf32`\n\n#","ref":"Kernel.SpecialForms.html#%3C%3C%3E%3E/1-modifiers"},{"type":"macro","title":"Sign - Kernel.SpecialForms.<<>>/1","doc":"Integers can be `signed` or `unsigned`, defaulting to `unsigned`.\n\n    iex>  > =  >\n     >\n    iex> int\n    156\n    iex>  > =  >\n     >\n    iex> int\n    -100\n\n`signed` and `unsigned` are only used for matching binaries (see below) and\nare only used for integers.\n\n    iex>  > =  >\n     >\n\n#","ref":"Kernel.SpecialForms.html#%3C%3C%3E%3E/1-sign"},{"type":"macro","title":"Endianness - Kernel.SpecialForms.<<>>/1","doc":"Elixir has three options for endianness: `big`, `little`, and `native`.\nThe default is `big`:\n\n    iex>  > =  >\n     >\n    iex> number\n    256\n    iex>  > =  >\n     >\n    iex> number\n    1\n\n`native` is determined by the VM at startup and will depend on the\nhost operating system.","ref":"Kernel.SpecialForms.html#%3C%3C%3E%3E/1-endianness"},{"type":"macro","title":"Binary/Bitstring Matching - Kernel.SpecialForms.<<>>/1","doc":"Binary matching is a powerful feature in Elixir that is useful for extracting\ninformation from binaries as well as pattern matching.\n\nBinary matching can be used by itself to extract information from binaries:\n\n    iex>  > = \"Hello, World\"\n    \"Hello, World\"\n    iex> place\n    \"World\"\n\nOr as a part of function definitions to pattern match:\n\n    defmodule ImageTyper do\n      @png_signature  >\n      @jpg_signature  >\n\n      def type( >), do: :png\n      def type( >), do: :jpg\n      def type(_), do: :unknown\n    end\n\n#","ref":"Kernel.SpecialForms.html#%3C%3C%3E%3E/1-binary-bitstring-matching"},{"type":"macro","title":"Performance & Optimizations - Kernel.SpecialForms.<<>>/1","doc":"The Erlang compiler can provide a number of optimizations on binary creation\nand matching. To see optimization output, set the `bin_opt_info` compiler\noption:\n\n    ERL_COMPILER_OPTIONS=bin_opt_info mix compile\n\nTo learn more about specific optimizations and performance considerations,\ncheck out the\n[\"Constructing and matching binaries\" chapter of the Erlang's Efficiency Guide](https://www.erlang.org/doc/efficiency_guide/binaryhandling.html).","ref":"Kernel.SpecialForms.html#%3C%3C%3E%3E/1-performance-optimizations"},{"type":"macro","title":"Kernel.SpecialForms.=/2","doc":"Match operator. Matches the value on the right against the pattern on the left.","ref":"Kernel.SpecialForms.html#=/2"},{"type":"macro","title":"Kernel.SpecialForms.alias/2","doc":"`alias/2` is used to set up aliases, often useful with modules' names.","ref":"Kernel.SpecialForms.html#alias/2"},{"type":"macro","title":"Examples - Kernel.SpecialForms.alias/2","doc":"`alias/2` can be used to set up an alias for any module:\n\n    defmodule Math do\n      alias MyKeyword, as: Keyword\n    end\n\nIn the example above, we have set up `MyKeyword` to be aliased\nas `Keyword`. So now, any reference to `Keyword` will be\nautomatically replaced by `MyKeyword`.\n\nIn case one wants to access the original `Keyword`, it can be done\nby accessing `Elixir`:\n\n    Keyword.values #=> uses MyKeyword.values\n    Elixir.Keyword.values #=> uses Keyword.values\n\nNote that calling `alias` without the `:as` option automatically\nsets an alias based on the last part of the module. For example:\n\n    alias Foo.Bar.Baz\n\nIs the same as:\n\n    alias Foo.Bar.Baz, as: Baz\n\nWe can also alias multiple modules in one line:\n\n    alias Foo.{Bar, Baz, Biz}\n\nIs the same as:\n\n    alias Foo.Bar\n    alias Foo.Baz\n    alias Foo.Biz","ref":"Kernel.SpecialForms.html#alias/2-examples"},{"type":"macro","title":"Lexical scope - Kernel.SpecialForms.alias/2","doc":"`import/2`, `require/2` and `alias/2` are called directives and all\nhave lexical scope. This means you can set up aliases inside\nspecific functions and it won't affect the overall scope.","ref":"Kernel.SpecialForms.html#alias/2-lexical-scope"},{"type":"macro","title":"Warnings - Kernel.SpecialForms.alias/2","doc":"If you alias a module and you don't use the alias, Elixir is\ngoing to issue a warning implying the alias is not being used.\n\nIn case the alias is generated automatically by a macro,\nElixir won't emit any warnings though, since the alias\nwas not explicitly defined.\n\nBoth warning behaviours could be changed by explicitly\nsetting the `:warn` option to `true` or `false`.","ref":"Kernel.SpecialForms.html#alias/2-warnings"},{"type":"macro","title":"Kernel.SpecialForms.case/2","doc":"Matches the given expression against the given clauses.","ref":"Kernel.SpecialForms.html#case/2"},{"type":"macro","title":"Examples - Kernel.SpecialForms.case/2","doc":"case File.read(file) do\n      {:ok, contents} when is_binary(contents) ->\n        String.split(contents, \"\\n\")\n\n      {:error, _reason} ->\n        Logger.warning \"could not find #{file}, assuming empty...\"\n        []\n    end\n\nIn the example above, we match the result of `File.read/1`\nagainst each clause \"head\" and execute the clause \"body\"\ncorresponding to the first clause that matches.\n\nIf no clause matches, an error is raised. For this reason,\nit may be necessary to add a final catch-all clause (like `_`)\nwhich will always match.\n\n    x = 10\n\n    case x do\n      0 ->\n        \"This clause won't match\"\n\n      _ ->\n        \"This clause would match any value (x = #{x})\"\n    end\n    #=> \"This clause would match any value (x = 10)\"","ref":"Kernel.SpecialForms.html#case/2-examples"},{"type":"macro","title":"Variable handling - Kernel.SpecialForms.case/2","doc":"Note that variables bound in a clause do not leak to the outer context:\n\n    case data do\n      {:ok, value} -> value\n      :error -> nil\n    end\n\n    value\n    #=> unbound variable value\n\nVariables in the outer context cannot be overridden either:\n\n    value = 7\n\n    case lucky? do\n      false -> value = 13\n      true -> true\n    end\n\n    value\n    #=> 7\n\nIn the example above, `value` is going to be `7` regardless of the value of\n`lucky?`. The variable `value` bound in the clause and the variable `value`\nbound in the outer context are two entirely separate variables.\n\nIf you want to pattern match against an existing variable,\nyou need to use the `^/1` operator:\n\n    x = 1\n\n    case 10 do\n      ^x -> \"Won't match\"\n      _ -> \"Will match\"\n    end\n    #=> \"Will match\"","ref":"Kernel.SpecialForms.html#case/2-variable-handling"},{"type":"macro","title":"Using guards to match against multiple values - Kernel.SpecialForms.case/2","doc":"While it is not possible to match against multiple patterns in a single\nclause, it's possible to match against multiple values by using guards:\n\n    case data do\n      value when value in [:one, :two] ->\n        \"#{value} has been matched\"\n\n      :three ->\n        \"three has been matched\"\n    end","ref":"Kernel.SpecialForms.html#case/2-using-guards-to-match-against-multiple-values"},{"type":"macro","title":"Kernel.SpecialForms.cond/1","doc":"Evaluates the expression corresponding to the first clause that\nevaluates to a truthy value.\n\n    cond do\n      hd([1, 2, 3]) ->\n        \"1 is considered as true\"\n    end\n    #=> \"1 is considered as true\"\n\nRaises an error if all conditions evaluate to `nil` or `false`.\nFor this reason, it may be necessary to add a final always-truthy condition\n(anything non-`false` and non-`nil`), which will always match.","ref":"Kernel.SpecialForms.html#cond/1"},{"type":"macro","title":"Examples - Kernel.SpecialForms.cond/1","doc":"cond do\n      1 + 1 == 1 ->\n        \"This will never match\"\n      2 * 2 != 4 ->\n        \"Nor this\"\n      true ->\n        \"This will\"\n    end\n    #=> \"This will\"","ref":"Kernel.SpecialForms.html#cond/1-examples"},{"type":"macro","title":"Kernel.SpecialForms.fn/1","doc":"Defines an anonymous function.\n\nSee `Function` for more information.","ref":"Kernel.SpecialForms.html#fn/1"},{"type":"macro","title":"Examples - Kernel.SpecialForms.fn/1","doc":"iex> add = fn a, b -> a + b end\n    iex> add.(1, 2)\n    3\n\nAnonymous functions can also have multiple clauses. All clauses\nshould expect the same number of arguments:\n\n    iex> negate = fn\n    ...>   true -> false\n    ...>   false -> true\n    ...> end\n    iex> negate.(false)\n    true","ref":"Kernel.SpecialForms.html#fn/1-examples"},{"type":"macro","title":"Kernel.SpecialForms.for/1","doc":"Comprehensions allow you to quickly build a data structure from\nan enumerable or a bitstring.\n\nLet's start with an example:\n\n    iex> for n   for n   for x   for n   users = [user: \"john\", admin: \"meg\", guest: \"barbara\"]\n    iex> for {type, name} when type != :guest     String.upcase(name)\n    ...> end\n    [\"JOHN\", \"MEG\"]\n\nBitstring generators are also supported and are very useful when you\nneed to organize bitstring streams:\n\n    iex> pixels =  >\n    iex> for  >, do: {r, g, b}\n    [{213, 45, 132}, {64, 76, 32}, {76, 0, 0}, {234, 32, 15}]\n\nVariable assignments inside the comprehension, be it in generators,\nfilters or inside the block, are not reflected outside of the\ncomprehension.\n\nVariable assignments inside filters must still return a truthy value,\notherwise values are discarded. Let's see an example. Imagine you have\na keyword list where the key is a programming language and the value\nis its direct parent. Then let's try to compute the grandparent of each\nlanguage. You could try this:\n\n    iex> languages = [elixir: :erlang, erlang: :prolog, prolog: nil]\n    iex> for {language, parent}   languages = [elixir: :erlang, erlang: :prolog, prolog: nil]\n    iex> for {language, parent}     grandparent = languages[parent]\n    ...>   {language, grandparent}\n    ...> end\n    [elixir: :prolog, erlang: nil, prolog: nil]\n\nHowever, such option is not always available, as you may have further\nfilters. An alternative is to convert the filter into a generator by\nwrapping the right side of `=` in a list:\n\n    iex> languages = [elixir: :erlang, erlang: :prolog, prolog: nil]\n    iex> for {language, parent} <- languages, grandparent <- [languages[parent]], do: {language, grandparent}\n    [elixir: :prolog, erlang: nil, prolog: nil]","ref":"Kernel.SpecialForms.html#for/1"},{"type":"macro","title":"The `:into` and `:uniq` options - Kernel.SpecialForms.for/1","doc":"In the examples above, the result returned by the comprehension was\nalways a list. The returned result can be configured by passing an\n`:into` option, that accepts any structure as long as it implements\nthe `Collectable` protocol.\n\nFor example, we can use bitstring generators with the `:into` option\nto easily remove all spaces in a string:\n\n    iex> for  >, c != ?\\s, into: \"\", do:  >\n    \"helloworld\"\n\nThe `IO` module provides streams, that are both `Enumerable` and\n`Collectable`, here is an upcase echo server using comprehensions:\n\n    for line   for x   for  >, uniq: true, into: \"\", do:  >\n    \"ABC\"","ref":"Kernel.SpecialForms.html#for/1-the-into-and-uniq-options"},{"type":"macro","title":"The `:reduce` option - Kernel.SpecialForms.for/1","doc":"While the `:into` option allows us to customize the comprehension behaviour\nto a given data type, such as putting all of the values inside a map or inside\na binary, it is not always enough.\n\nFor example, imagine that you have a binary with letters where you want to\ncount how many times each lowercase letter happens, ignoring all uppercase\nones. For instance, for the string `\"AbCabCABc\"`, we want to return the map\n`%{\"a\" => 1, \"b\" => 2, \"c\" => 1}`.\n\nIf we were to use `:into`, we would need a data type that computes the\nfrequency of each element it holds. While there is no such data type in\nElixir, you could implement one yourself.\n\nA simpler option would be to use comprehensions for the mapping and\nfiltering of letters, and then we invoke `Enum.reduce/3` to build a map,\nfor example:\n\n    iex> letters = for  >, x in ?a..?z, do:  >\n    iex> Enum.reduce(letters, %{}, fn x, acc -> Map.update(acc, x, 1, & &1 + 1) end)\n    %{\"a\" => 1, \"b\" => 2, \"c\" => 1}\n\nWhile the above is straight-forward, it has the downside of traversing the\ndata at least twice. If you are expecting long strings as inputs, this can\nbe quite expensive.\n\nLuckily, comprehensions also support the `:reduce` option, which would allow\nus to fuse both steps above into a single step:\n\n    iex> for  >, x in ?a..?z, reduce: %{} do\n    ...>   acc -> Map.update(acc,  >, 1, & &1 + 1)\n    ...> end\n    %{\"a\" => 1, \"b\" => 2, \"c\" => 1}\n\nWhen the `:reduce` key is given, its value is used as the initial accumulator\nand the `do` block must be changed to use `->` clauses, where the left side\nof `->` receives the accumulated value of the previous iteration and the\nexpression on the right side must return the new accumulator value. Once there are no more\nelements, the final accumulated value is returned. If there are no elements\nat all, then the initial accumulator value is returned.","ref":"Kernel.SpecialForms.html#for/1-the-reduce-option"},{"type":"macro","title":"Kernel.SpecialForms.import/2","doc":"Imports functions and macros from other modules.\n\n`import/2` allows one to easily access functions or macros from\nother modules without using the qualified name.","ref":"Kernel.SpecialForms.html#import/2"},{"type":"macro","title":"Examples - Kernel.SpecialForms.import/2","doc":"If you are using several functions from a given module, you can\nimport those functions and reference them as local functions,\nfor example:\n\n    iex> import List\n    iex> flatten([1, [2], 3])\n    [1, 2, 3]","ref":"Kernel.SpecialForms.html#import/2-examples"},{"type":"macro","title":"Selector - Kernel.SpecialForms.import/2","doc":"By default, Elixir imports functions and macros from the given\nmodule, except the ones starting with an underscore (which are\nusually callbacks):\n\n    import List\n\nA developer can filter to import only functions, macros, or sigils\n(which can be functions or macros) via the `:only` option:\n\n    import List, only: :functions\n    import List, only: :macros\n    import Kernel, only: :sigils\n\nAlternatively, Elixir allows a developer to pass pairs of\nname/arities to `:only` or `:except` as a fine grained control\non what to import (or not):\n\n    import List, only: [flatten: 1]\n    import String, except: [split: 2]\n\nImporting the same module again will erase the previous imports,\nexcept when the `except` option is used, which is always exclusive\non a previously declared `import/2`. If there is no previous import,\nthen it applies to all functions and macros in the module. For\nexample:\n\n    import List, only: [flatten: 1, keyfind: 4]\n    import List, except: [flatten: 1]\n\nAfter the two import calls above, only `List.keyfind/4` will be\nimported.","ref":"Kernel.SpecialForms.html#import/2-selector"},{"type":"macro","title":"Underscore functions - Kernel.SpecialForms.import/2","doc":"By default functions starting with `_` are not imported. If you really want\nto import a function starting with `_` you must explicitly include it in the\n`:only` selector.\n\n    import File.Stream, only: [__build__: 3]","ref":"Kernel.SpecialForms.html#import/2-underscore-functions"},{"type":"macro","title":"Lexical scope - Kernel.SpecialForms.import/2","doc":"It is important to note that `import/2` is lexical. This means you\ncan import specific macros inside specific functions:\n\n    defmodule Math do\n      def some_function do\n        # 1) Disable \"if/2\" from Kernel\n        import Kernel, except: [if: 2]\n\n        # 2) Require the new \"if/2\" macro from MyMacros\n        import MyMacros\n\n        # 3) Use the new macro\n        if do_something, it_works\n      end\n    end\n\nIn the example above, we imported macros from `MyMacros`,\nreplacing the original `if/2` implementation by our own\nwithin that specific function. All other functions in that\nmodule will still be able to use the original one.","ref":"Kernel.SpecialForms.html#import/2-lexical-scope"},{"type":"macro","title":"Warnings - Kernel.SpecialForms.import/2","doc":"If you import a module and you don't use any of the imported\nfunctions or macros from this module, Elixir is going to issue\na warning implying the import is not being used.\n\nIn case the import is generated automatically by a macro,\nElixir won't emit any warnings though, since the import\nwas not explicitly defined.\n\nBoth warning behaviours could be changed by explicitly\nsetting the `:warn` option to `true` or `false`.","ref":"Kernel.SpecialForms.html#import/2-warnings"},{"type":"macro","title":"Ambiguous function/macro names - Kernel.SpecialForms.import/2","doc":"If two modules `A` and `B` are imported and they both contain\na `foo` function with an arity of `1`, an error is only emitted\nif an ambiguous call to `foo/1` is actually made; that is, the\nerrors are emitted lazily, not eagerly.","ref":"Kernel.SpecialForms.html#import/2-ambiguous-function-macro-names"},{"type":"macro","title":"Kernel.SpecialForms.quote/2","doc":"Gets the representation of any expression.","ref":"Kernel.SpecialForms.html#quote/2"},{"type":"macro","title":"Examples - Kernel.SpecialForms.quote/2","doc":"iex> quote do\n    ...>   sum(1, 2, 3)\n    ...> end\n    {:sum, [], [1, 2, 3]}","ref":"Kernel.SpecialForms.html#quote/2-examples"},{"type":"macro","title":"Elixir's AST (Abstract Syntax Tree) - Kernel.SpecialForms.quote/2","doc":"Any Elixir code can be represented using Elixir data structures.\nThe building block of Elixir macros is a tuple with three elements,\nfor example:\n\n    {:sum, [], [1, 2, 3]}\n\nThe tuple above represents a function call to `sum` passing 1, 2 and\n3 as arguments. The tuple elements are:\n\n  * The first element of the tuple is always an atom or\n    another tuple in the same representation.\n\n  * The second element of the tuple represents [metadata](`t:Macro.metadata/0`).\n\n  * The third element of the tuple are the arguments for the\n    function call. The third argument may be an atom, which is\n    usually a variable (or a local call).\n\nBesides the tuple described above, Elixir has a few literals that\nare also part of its AST. Those literals return themselves when\nquoted. They are:\n\n    :sum         #=> Atoms\n    1            #=> Integers\n    2.0          #=> Floats\n    [1, 2]       #=> Lists\n    \"strings\"    #=> Strings\n    {key, value} #=> Tuples with two elements\n\nAny other value, such as a map or a four-element tuple, must be escaped\n(`Macro.escape/1`) before being introduced into an AST.","ref":"Kernel.SpecialForms.html#quote/2-elixir-s-ast-abstract-syntax-tree"},{"type":"macro","title":"Options - Kernel.SpecialForms.quote/2","doc":"* `:bind_quoted` - passes a binding to the macro. Whenever a binding is\n    given, `unquote/1` is automatically disabled.\n\n  * `:context` - sets the resolution context.\n\n  * `:generated` - marks the given chunk as generated so it does not emit warnings.\n    It is also useful to avoid dialyzer reporting errors when macros generate\n    unused clauses.\n\n  * `:file` - sets the quoted expressions to have the given file.\n\n  * `:line` - sets the quoted expressions to have the given line.\n\n  * `:location` - when set to `:keep`, keeps the current line and file from\n    quote. Read the \"Stacktrace information\" section below for more information.\n\n  * `:unquote` - when `false`, disables unquoting. This means any `unquote`\n    call will be kept as is in the AST, instead of replaced by the `unquote`\n    arguments. For example:\n\n        iex> quote do\n        ...>   unquote(\"hello\")\n        ...> end\n        \"hello\"\n\n        iex> quote unquote: false do\n        ...>   unquote(\"hello\")\n        ...> end\n        {:unquote, [], [\"hello\"]}","ref":"Kernel.SpecialForms.html#quote/2-options"},{"type":"macro","title":"Quote and macros - Kernel.SpecialForms.quote/2","doc":"`quote/2` is commonly used with macros for code generation. As an exercise,\nlet's define a macro that multiplies a number by itself (squared). In practice,\nthere is no reason to define such a macro (and it would actually be\nseen as a bad practice), but it is simple enough that it allows us to focus\non the important aspects of quotes and macros:\n\n    defmodule Math do\n      defmacro squared(x) do\n        quote do\n          unquote(x) * unquote(x)\n        end\n      end\n    end\n\nWe can invoke it as:\n\n    import Math\n    IO.puts(\"Got #{squared(5)}\")\n\nAt first, there is nothing in this example that actually reveals it is a\nmacro. But what is happening is that, at compilation time, `squared(5)`\nbecomes `5 * 5`. The argument `5` is duplicated in the produced code, we\ncan see this behaviour in practice though because our macro actually has\na bug:\n\n    import Math\n    my_number = fn ->\n      IO.puts(\"Returning 5\")\n      5\n    end\n    IO.puts(\"Got #{squared(my_number.())}\")\n\nThe example above will print:\n\n    Returning 5\n    Returning 5\n    Got 25\n\nNotice how \"Returning 5\" was printed twice, instead of just once. This is\nbecause a macro receives an expression and not a value (which is what we\nwould expect in a regular function). This means that:\n\n    squared(my_number.())\n\nActually expands to:\n\n    my_number.() * my_number.()\n\nWhich invokes the function twice, explaining why we get the printed value\ntwice! In the majority of the cases, this is actually unexpected behaviour,\nand that's why one of the first things you need to keep in mind when it\ncomes to macros is to **not unquote the same value more than once**.\n\nLet's fix our macro:\n\n    defmodule Math do\n      defmacro squared(x) do\n        quote do\n          x = unquote(x)\n          x * x\n        end\n      end\n    end\n\nNow invoking `squared(my_number.())` as before will print the value just\nonce.\n\nIn fact, this pattern is so common that most of the times you will want\nto use the `bind_quoted` option with `quote/2`:\n\n    defmodule Math do\n      defmacro squared(x) do\n        quote bind_quoted: [x: x] do\n          x * x\n        end\n      end\n    end\n\n`:bind_quoted` will translate to the same code as the example above.\n`:bind_quoted` can be used in many cases and is seen as good practice,\nnot only because it helps prevent us from running into common mistakes, but also\nbecause it allows us to leverage other tools exposed by macros, such as\nunquote fragments discussed in some sections below.\n\nBefore we finish this brief introduction, you will notice that, even though\nwe defined a variable `x` inside our quote:\n\n    quote do\n      x = unquote(x)\n      x * x\n    end\n\nWhen we call:\n\n    import Math\n    squared(5)\n    x\n    ** (CompileError) undefined variable \"x\"\n\nWe can see that `x` did not leak to the user context. This happens\nbecause Elixir macros are hygienic, a topic we will discuss at length\nin the next sections as well.","ref":"Kernel.SpecialForms.html#quote/2-quote-and-macros"},{"type":"macro","title":"Hygiene in variables - Kernel.SpecialForms.quote/2","doc":"Consider the following example:\n\n    defmodule Hygiene do\n      defmacro no_interference do\n        quote do\n          a = 1\n        end\n      end\n    end\n\n    require Hygiene\n\n    a = 10\n    Hygiene.no_interference()\n    a\n    #=> 10\n\nIn the example above, `a` returns 10 even if the macro\nis apparently setting it to 1 because variables defined\nin the macro do not affect the context the macro is executed in.\nIf you want to set or get a variable in the caller's context, you\ncan do it with the help of the `var!` macro:\n\n    defmodule NoHygiene do\n      defmacro interference do\n        quote do\n          var!(a) = 1\n        end\n      end\n    end\n\n    require NoHygiene\n\n    a = 10\n    NoHygiene.interference()\n    a\n    #=> 1\n\nYou cannot even access variables defined in the same module unless\nyou explicitly give it a context:\n\n    defmodule Hygiene do\n      defmacro write do\n        quote do\n          a = 1\n        end\n      end\n\n      defmacro read do\n        quote do\n          a\n        end\n      end\n    end\n\n    Hygiene.write()\n    Hygiene.read()\n    ** (RuntimeError) undefined variable a or undefined function a/0\n\nFor such, you can explicitly pass the current module scope as\nargument:\n\n    defmodule ContextHygiene do\n      defmacro write do\n        quote do\n          var!(a, ContextHygiene) = 1\n        end\n      end\n\n      defmacro read do\n        quote do\n          var!(a, ContextHygiene)\n        end\n      end\n    end\n\n    ContextHygiene.write()\n    ContextHygiene.read()\n    #=> 1\n\nThe contexts of a variable is identified by the third element of the tuple.\nThe default context is `nil` and `quote` assigns another context to all\nvariables within:\n\n    quote(do: var)\n    #=> {:var, [], Elixir}\n\nIn case of variables returned by macros, there may also be a `:counter` key\nin the metadata, which is used to further refine its contexts and guarantee\nisolation between macro invocations as seen in the previous example.","ref":"Kernel.SpecialForms.html#quote/2-hygiene-in-variables"},{"type":"macro","title":"Hygiene in aliases - Kernel.SpecialForms.quote/2","doc":"Aliases inside quote are hygienic by default.\nConsider the following example:\n\n    defmodule Hygiene do\n      alias Map, as: M\n\n      defmacro no_interference do\n        quote do\n          M.new()\n        end\n      end\n    end\n\n    require Hygiene\n    Hygiene.no_interference()\n    #=> %{}\n\nNote that, even though the alias `M` is not available\nin the context the macro is expanded, the code above works\nbecause `M` still expands to `Map`.\n\nSimilarly, even if we defined an alias with the same name\nbefore invoking a macro, it won't affect the macro's result:\n\n    defmodule Hygiene do\n      alias Map, as: M\n\n      defmacro no_interference do\n        quote do\n          M.new()\n        end\n      end\n    end\n\n    require Hygiene\n    alias SomethingElse, as: M\n    Hygiene.no_interference()\n    #=> %{}\n\nIn some cases, you want to access an alias or a module defined\nin the caller. For such, you can use the `alias!` macro:\n\n    defmodule Hygiene do\n      # This will expand to Elixir.Nested.hello()\n      defmacro no_interference do\n        quote do\n          Nested.hello()\n        end\n      end\n\n      # This will expand to Nested.hello() for\n      # whatever is Nested in the caller\n      defmacro interference do\n        quote do\n          alias!(Nested).hello()\n        end\n      end\n    end\n\n    defmodule Parent do\n      defmodule Nested do\n        def hello, do: \"world\"\n      end\n\n      require Hygiene\n      Hygiene.no_interference()\n      ** (UndefinedFunctionError) ...\n\n      Hygiene.interference()\n      #=> \"world\"\n    end","ref":"Kernel.SpecialForms.html#quote/2-hygiene-in-aliases"},{"type":"macro","title":"Hygiene in imports - Kernel.SpecialForms.quote/2","doc":"Similar to aliases, imports in Elixir are hygienic. Consider the\nfollowing code:\n\n    defmodule Hygiene do\n      defmacrop get_length do\n        quote do\n          length([1, 2, 3])\n        end\n      end\n\n      def return_length do\n        import Kernel, except: [length: 1]\n        get_length\n      end\n    end\n\n    Hygiene.return_length()\n    #=> 3\n\nNotice how `Hygiene.return_length/0` returns `3` even though the `Kernel.length/1`\nfunction is not imported. In fact, even if `return_length/0`\nimported a function with the same name and arity from another\nmodule, it wouldn't affect the function result:\n\n    def return_length do\n      import String, only: [length: 1]\n      get_length\n    end\n\nCalling this new `return_length/0` will still return `3` as result.\n\nElixir is smart enough to delay the resolution to the latest\npossible moment. So, if you call `length([1, 2, 3])` inside quote,\nbut no `length/1` function is available, it is then expanded in\nthe caller:\n\n    defmodule Lazy do\n      defmacrop get_length do\n        import Kernel, except: [length: 1]\n\n        quote do\n          length(\"hello\")\n        end\n      end\n\n      def return_length do\n        import Kernel, except: [length: 1]\n        import String, only: [length: 1]\n        get_length\n      end\n    end\n\n    Lazy.return_length()\n    #=> 5","ref":"Kernel.SpecialForms.html#quote/2-hygiene-in-imports"},{"type":"macro","title":"Stacktrace information - Kernel.SpecialForms.quote/2","doc":"When defining functions via macros, developers have the option of\nchoosing if runtime errors will be reported from the caller or from\ninside the quote. Let's see an example:\n\n    # adder.ex\n    defmodule Adder do\n      @doc \"Defines a function that adds two numbers\"\n      defmacro defadd do\n        quote location: :keep do\n          def add(a, b), do: a + b\n        end\n      end\n    end\n\n    # sample.ex\n    defmodule Sample do\n      import Adder\n      defadd\n    end\n\n    require Sample\n    Sample.add(:one, :two)\n    ** (ArithmeticError) bad argument in arithmetic expression\n        adder.ex:5: Sample.add/2\n\nWhen using `location: :keep` and invalid arguments are given to\n`Sample.add/2`, the stacktrace information will point to the file\nand line inside the quote. Without `location: :keep`, the error is\nreported to where `defadd` was invoked. `location: :keep` affects\nonly definitions inside the quote.\n\n> #### `location: :keep` and unquote {: .warning}\n>\n> Do not use `location: :keep` if the function definition\n> also `unquote`s some of the macro arguments. If you do so, Elixir\n> will store the file definition of the current location but the\n> unquoted arguments may contain line information of the macro caller,\n> leading to erroneous stacktraces.","ref":"Kernel.SpecialForms.html#quote/2-stacktrace-information"},{"type":"macro","title":"Binding and unquote fragments - Kernel.SpecialForms.quote/2","doc":"Elixir quote/unquote mechanisms provide a functionality called\nunquote fragments. Unquote fragments provide an easy way to generate\nfunctions on the fly. Consider this example:\n\n    kv = [foo: 1, bar: 2]\n    Enum.each(kv, fn {k, v} ->\n      def unquote(k)(), do: unquote(v)\n    end)\n\nIn the example above, we have generated the functions `foo/0` and\n`bar/0` dynamically. Now, imagine that we want to convert this\nfunctionality into a macro:\n\n    defmacro defkv(kv) do\n      Enum.map(kv, fn {k, v} ->\n        quote do\n          def unquote(k)(), do: unquote(v)\n        end\n      end)\n    end\n\nWe can invoke this macro as:\n\n    defkv [foo: 1, bar: 2]\n\nHowever, we can't invoke it as follows:\n\n    kv = [foo: 1, bar: 2]\n    defkv kv\n\nThis is because the macro is expecting its arguments to be a\nkeyword list at **compilation** time. Since in the example above\nwe are passing the representation of the variable `kv`, our\ncode fails.\n\nThis is actually a common pitfall when developing macros. We are\nassuming a particular shape in the macro. We can work around it\nby unquoting the variable inside the quoted expression:\n\n    defmacro defkv(kv) do\n      quote do\n        Enum.each(unquote(kv), fn {k, v} ->\n          def unquote(k)(), do: unquote(v)\n        end)\n      end\n    end\n\nIf you try to run our new macro, you will notice it won't\neven compile, complaining that the variables `k` and `v`\ndo not exist. This is because of the ambiguity: `unquote(k)`\ncan either be an unquote fragment, as previously, or a regular\nunquote as in `unquote(kv)`.\n\nOne solution to this problem is to disable unquoting in the\nmacro, however, doing that would make it impossible to inject the\n`kv` representation into the tree. That's when the `:bind_quoted`\noption comes to the rescue (again!). By using `:bind_quoted`, we\ncan automatically disable unquoting while still injecting the\ndesired variables into the tree:\n\n    defmacro defkv(kv) do\n      quote bind_quoted: [kv: kv] do\n        Enum.each(kv, fn {k, v} ->\n          def unquote(k)(), do: unquote(v)\n        end)\n      end\n    end\n\nIn fact, the `:bind_quoted` option is recommended every time\none desires to inject a value into the quote.","ref":"Kernel.SpecialForms.html#quote/2-binding-and-unquote-fragments"},{"type":"macro","title":"Kernel.SpecialForms.receive/1","doc":"Checks if there is a message matching the given clauses\nin the current process mailbox.\n\nIn case there is no such message, the current process hangs\nuntil a message arrives or waits until a given timeout value.","ref":"Kernel.SpecialForms.html#receive/1"},{"type":"macro","title":"Examples - Kernel.SpecialForms.receive/1","doc":"receive do\n      {:selector, number, name} when is_integer(number) ->\n        name\n      name when is_atom(name) ->\n        name\n      _ ->\n        IO.puts(:stderr, \"Unexpected message received\")\n    end\n\nAn optional `after` clause can be given in case the message was not\nreceived after the given timeout period, specified in milliseconds:\n\n    receive do\n      {:selector, number, name} when is_integer(number) ->\n        name\n      name when is_atom(name) ->\n        name\n      _ ->\n        IO.puts(:stderr, \"Unexpected message received\")\n    after\n      5000 ->\n        IO.puts(:stderr, \"No message in 5 seconds\")\n    end\n\nThe `after` clause can be specified even if there are no match clauses.\nThe timeout value given to `after` can be any expression evaluating to\none of the allowed values:\n\n  * `:infinity` - the process should wait indefinitely for a matching\n    message, this is the same as not using the after clause\n\n  * `0` - if there is no matching message in the mailbox, the timeout\n    will occur immediately\n\n  * positive integer smaller than or equal to `4_294_967_295` (`0xFFFFFFFF`\n    in hexadecimal notation) - it should be possible to represent the timeout\n    value as an unsigned 32-bit integer.","ref":"Kernel.SpecialForms.html#receive/1-examples"},{"type":"macro","title":"Variable handling - Kernel.SpecialForms.receive/1","doc":"The `receive/1` special form handles variables exactly as the `case/2`\nspecial macro. For more information, check the docs for `case/2`.","ref":"Kernel.SpecialForms.html#receive/1-variable-handling"},{"type":"macro","title":"Kernel.SpecialForms.require/2","doc":"Requires a module in order to use its macros.","ref":"Kernel.SpecialForms.html#require/2"},{"type":"macro","title":"Examples - Kernel.SpecialForms.require/2","doc":"Public functions in modules are globally available, but in order to use\nmacros, you need to opt-in by requiring the module they are defined in.\n\nLet's suppose you created your own `if/2` implementation in the module\n`MyMacros`. If you want to invoke it, you need to first explicitly\nrequire the `MyMacros`:\n\n    defmodule Math do\n      require MyMacros\n      MyMacros.if do_something, it_works\n    end\n\nAn attempt to call a macro that was not loaded will raise an error.","ref":"Kernel.SpecialForms.html#require/2-examples"},{"type":"macro","title":"Alias shortcut - Kernel.SpecialForms.require/2","doc":"`require/2` also accepts `:as` as an option so it automatically sets\nup an alias. Please check `alias/2` for more information.","ref":"Kernel.SpecialForms.html#require/2-alias-shortcut"},{"type":"macro","title":"Kernel.SpecialForms.super/1","doc":"Calls the overridden function when overriding it with `Kernel.defoverridable/1`.\n\nSee `Kernel.defoverridable/1` for more information and documentation.","ref":"Kernel.SpecialForms.html#super/1"},{"type":"macro","title":"Kernel.SpecialForms.try/1","doc":"Evaluates the given expressions and handles any error, exit,\nor throw that may have happened.","ref":"Kernel.SpecialForms.html#try/1"},{"type":"macro","title":"Examples - Kernel.SpecialForms.try/1","doc":"try do\n      do_something_that_may_fail(some_arg)\n    rescue\n      ArgumentError ->\n        IO.puts(\"Invalid argument given\")\n    catch\n      value ->\n        IO.puts(\"Caught #{inspect(value)}\")\n    else\n      value ->\n        IO.puts(\"Success! The result was #{inspect(value)}\")\n    after\n      IO.puts(\"This is printed regardless if it failed or succeeded\")\n    end\n\nThe `rescue` clause is used to handle exceptions while the `catch`\nclause can be used to catch thrown values and exits.\nThe `else` clause can be used to control flow based on the result of\nthe expression. `catch`, `rescue`, and `else` clauses work based on\npattern matching (similar to the `case` special form).\n\nCalls inside `try/1` are not tail recursive since the VM needs to keep\nthe stacktrace in case an exception happens. To retrieve the stacktrace,\naccess `__STACKTRACE__/0` inside the `rescue` or `catch` clause.\n\n## `rescue` clauses\n\nBesides relying on pattern matching, `rescue` clauses provide some\nconveniences around exceptions that allow one to rescue an\nexception by its name. All the following formats are valid patterns\nin `rescue` clauses:\n\n    # Rescue a single exception without binding the exception\n    # to a variable\n    try do\n      UndefinedModule.undefined_function\n    rescue\n      UndefinedFunctionError -> nil\n    end\n\n    # Rescue any of the given exception without binding\n    try do\n      UndefinedModule.undefined_function\n    rescue\n      [UndefinedFunctionError, ArgumentError] -> nil\n    end\n\n    # Rescue and bind the exception to the variable \"x\"\n    try do\n      UndefinedModule.undefined_function\n    rescue\n      x in [UndefinedFunctionError] -> nil\n    end\n\n    # Rescue all kinds of exceptions and bind the rescued exception\n    # to the variable \"x\"\n    try do\n      UndefinedModule.undefined_function\n    rescue\n      x -> nil\n    end\n\n#","ref":"Kernel.SpecialForms.html#try/1-examples"},{"type":"macro","title":"Erlang errors - Kernel.SpecialForms.try/1","doc":"Erlang errors are transformed into Elixir ones when rescuing:\n\n    try do\n      :erlang.error(:badarg)\n    rescue\n      ArgumentError -> :ok\n    end\n    #=> :ok\n\nThe most common Erlang errors will be transformed into their\nElixir counterpart. Those which are not will be transformed\ninto the more generic `ErlangError`:\n\n    try do\n      :erlang.error(:unknown)\n    rescue\n      ErlangError -> :ok\n    end\n    #=> :ok\n\nIn fact, `ErlangError` can be used to rescue any error that is\nnot a proper Elixir error. For example, it can be used to rescue\nthe earlier `:badarg` error too, prior to transformation:\n\n    try do\n      :erlang.error(:badarg)\n    rescue\n      ErlangError -> :ok\n    end\n    #=> :ok\n\n## `catch` clauses\n\nThe `catch` clause can be used to catch thrown values, exits, and errors.\n\n#","ref":"Kernel.SpecialForms.html#try/1-erlang-errors"},{"type":"macro","title":"Catching thrown values - Kernel.SpecialForms.try/1","doc":"`catch` can be used to catch values thrown by `Kernel.throw/1`:\n\n    try do\n      throw(:some_value)\n    catch\n      thrown_value ->\n        IO.puts(\"A value was thrown: #{inspect(thrown_value)}\")\n    end\n\n#","ref":"Kernel.SpecialForms.html#try/1-catching-thrown-values"},{"type":"macro","title":"Catching values of any kind - Kernel.SpecialForms.try/1","doc":"The `catch` clause also supports catching exits and errors. To do that, it\nallows matching on both the *kind* of the caught value as well as the value\nitself:\n\n    try do\n      exit(:shutdown)\n    catch\n      :exit, value ->\n        IO.puts(\"Exited with value #{inspect(value)}\")\n    end\n\n    try do\n      exit(:shutdown)\n    catch\n      kind, value when kind in [:exit, :throw] ->\n        IO.puts(\"Caught exit or throw with value #{inspect(value)}\")\n    end\n\nThe `catch` clause also supports `:error` alongside `:exit` and `:throw` as\nin Erlang, although this is commonly avoided in favor of `raise`/`rescue` control\nmechanisms. One reason for this is that when catching `:error`, the error is\nnot automatically transformed into an Elixir error:\n\n    try do\n      :erlang.error(:badarg)\n    catch\n      :error, :badarg -> :ok\n    end\n    #=> :ok\n\n## `after` clauses\n\nAn `after` clause allows you to define cleanup logic that will be invoked both\nwhen the block of code passed to `try/1` succeeds and also when an error is raised. Note\nthat the process will exit as usual when receiving an exit signal that causes\nit to exit abruptly and so the `after` clause is not guaranteed to be executed.\nLuckily, most resources in Elixir (such as open files, ETS tables, ports, sockets,\nand so on) are linked to or monitor the owning process and will automatically clean\nthemselves up if that process exits.\n\n    File.write!(\"tmp/story.txt\", \"Hello, World\")\n    try do\n      do_something_with(\"tmp/story.txt\")\n    after\n      File.rm(\"tmp/story.txt\")\n    end\n\nAlthough `after` clauses are invoked whether or not there was an error, they do not\nmodify the return value. All of the following examples return `:return_me`:\n\n    try do\n      :return_me\n    after\n      IO.puts(\"I will be printed\")\n      :not_returned\n    end\n\n    try do\n      raise \"boom\"\n    rescue\n      _ -> :return_me\n    after\n      IO.puts(\"I will be printed\")\n      :not_returned\n    end\n\n## `else` clauses\n\n`else` clauses allow the result of the body passed to `try/1` to be pattern\nmatched on:\n\n    x = 2\n    try do\n      1 / x\n    rescue\n      ArithmeticError ->\n        :infinity\n    else\n      y when y   -1 ->\n        :small\n      _ ->\n        :large\n    end\n\nIf an `else` clause is not present and no exceptions are raised,\nthe result of the expression will be returned:\n\n    x = 1\n    ^x =\n      try do\n        1 / x\n      rescue\n        ArithmeticError ->\n          :infinity\n      end\n\nHowever, when an `else` clause is present but the result of the expression\ndoes not match any of the patterns then an exception will be raised. This\nexception will not be caught by a `catch` or `rescue` in the same `try`:\n\n    x = 1\n    try do\n      try do\n        1 / x\n      rescue\n        # The TryClauseError cannot be rescued here:\n        TryClauseError ->\n          :error_a\n      else\n        0 ->\n          :small\n      end\n    rescue\n      # The TryClauseError is rescued here:\n      TryClauseError ->\n        :error_b\n    end\n\nSimilarly, an exception inside an `else` clause is not caught or rescued\ninside the same `try`:\n\n    try do\n      try do\n        nil\n      catch\n        # The exit(1) call below can not be caught here:\n        :exit, _ ->\n          :exit_a\n      else\n        _ ->\n          exit(1)\n      end\n    catch\n      # The exit is caught here:\n      :exit, _ ->\n        :exit_b\n    end\n\nThis means the VM no longer needs to keep the stacktrace once inside\nan `else` clause and so tail recursion is possible when using a `try`\nwith a tail call as the final call inside an `else` clause. The same\nis true for `rescue` and `catch` clauses.\n\nOnly the result of the tried expression falls down to the `else` clause.\nIf the `try` ends up in the `rescue` or `catch` clauses, their result\nwill not fall down to `else`:\n\n    try do\n      throw(:catch_this)\n    catch\n      :throw, :catch_this ->\n        :it_was_caught\n    else\n      # :it_was_caught will not fall down to this \"else\" clause.\n      other ->\n        {:else, other}\n    end","ref":"Kernel.SpecialForms.html#try/1-catching-values-of-any-kind"},{"type":"macro","title":"Variable handling - Kernel.SpecialForms.try/1","doc":"Since an expression inside `try` may not have been evaluated\ndue to an exception, any variable created inside `try` cannot\nbe accessed externally. For instance:\n\n    try do\n      x = 1\n      do_something_that_may_fail(same_arg)\n      :ok\n    catch\n      _, _ -> :failed\n    end\n\n    x\n    #=> unbound variable \"x\"\n\nIn the example above, `x` cannot be accessed since it was defined\ninside the `try` clause. A common practice to address this issue\nis to return the variables defined inside `try`:\n\n    x =\n      try do\n        x = 1\n        do_something_that_may_fail(same_arg)\n        x\n      catch\n        _, _ -> :failed\n      end","ref":"Kernel.SpecialForms.html#try/1-variable-handling"},{"type":"macro","title":"Kernel.SpecialForms.unquote/1","doc":"Unquotes the given expression inside a quoted expression.\n\nThis function expects a valid Elixir AST, also known as\nquoted expression, as argument. If you would like to `unquote`\nany value, such as a map or a four-element tuple, you should\ncall `Macro.escape/1` before unquoting.","ref":"Kernel.SpecialForms.html#unquote/1"},{"type":"macro","title":"Examples - Kernel.SpecialForms.unquote/1","doc":"Imagine the situation you have a quoted expression and\nyou want to inject it inside some quote. The first attempt\nwould be:\n\n    value =\n      quote do\n        13\n      end\n\n    quote do\n      sum(1, value, 3)\n    end\n\n\nWhich the argument for the `:sum` function call is not the\nexpected result:\n\n    {:sum, [], [1, {:value, [], Elixir}, 3]}\n\nFor this, we use `unquote`:\n\n    iex> value =\n    ...>   quote do\n    ...>     13\n    ...>   end\n    iex> quote do\n    ...>   sum(1, unquote(value), 3)\n    ...> end\n    {:sum, [], [1, 13, 3]}\n\nIf you want to unquote a value that is not a quoted expression,\nsuch as a map, you need to call `Macro.escape/1` before:\n\n    iex> value = %{foo: :bar}\n    iex> quote do\n    ...>   process_map(unquote(Macro.escape(value)))\n    ...> end\n    {:process_map, [], [{:%{}, [], [foo: :bar]}]}\n\nIf you forget to escape it, Elixir will raise an error\nwhen compiling the code.","ref":"Kernel.SpecialForms.html#unquote/1-examples"},{"type":"macro","title":"Kernel.SpecialForms.unquote_splicing/1","doc":"Unquotes the given list expanding its arguments.\n\nSimilar to `unquote/1`.","ref":"Kernel.SpecialForms.html#unquote_splicing/1"},{"type":"macro","title":"Examples - Kernel.SpecialForms.unquote_splicing/1","doc":"iex> values = [2, 3, 4]\n    iex> quote do\n    ...>   sum(1, unquote_splicing(values), 5)\n    ...> end\n    {:sum, [], [1, 2, 3, 4, 5]}","ref":"Kernel.SpecialForms.html#unquote_splicing/1-examples"},{"type":"macro","title":"Kernel.SpecialForms.with/1","doc":"Combine matching clauses.\n\nOne of the ways to understand with is to show which code\npatterns it improves. Imagine you have a map where the fields\n`width` and `height` are optional and you want to compute its\narea, as `{:ok, area}` or return `:error`. We could implement\nthis function as:\n\n    def area(map) do\n      case Map.fetch(opts, :width) do\n        {:ok, width} ->\n          case Map.fetch(opts, :height) do\n            {:ok, height} -> {:ok, width * height}\n            :error -> :error\n          end\n\n        :error ->\n          :error\n      end\n    end\n\nwhen called as `area(%{width: 10, height: 15})`, it should return\n`{:ok, 150}`. If any of the fields are missing, it returns `:error`.\n\nWhile the code above works, it is quite verbose. Using `with`,\nwe could rewrite it as:\n\n    def area(map) do\n      with {:ok, width}   opts = %{width: 10, height: 15}\n    iex> with {:ok, width}        {:ok, height}     {:ok, width * height}\n    ...> end\n    {:ok, 150}\n\nIf all clauses match, the `do` block is executed, returning its result.\nOtherwise the chain is aborted and the non-matched value is returned:\n\n    iex> opts = %{width: 10}\n    iex> with {:ok, width}        {:ok, height}     {:ok, width * height}\n    ...> end\n    :error\n\nGuards can be used in patterns as well:\n\n    iex> users = %{\"melany\" => \"guest\", \"bob\" => :admin}\n    iex> with {:ok, role} when not is_binary(role)     {:ok, to_string(role)}\n    ...> end\n    {:ok, \"admin\"}\n\nAs in `for/1`, variables bound inside `with/1` won't be accessible\noutside of `with/1`.\n\nExpressions without `  width = nil\n    iex> opts = %{width: 10, height: 15}\n    iex> with {:ok, width}        double_width = width * 2,\n    ...>      {:ok, height}     {:ok, double_width * height}\n    ...> end\n    {:ok, 300}\n    iex> width\n    nil\n\nThe behaviour of any expression in a clause is the same as if it was\nwritten outside of `with`. For example, `=` will raise a `MatchError`\ninstead of returning the non-matched value:\n\n    with :foo = :bar, do: :ok\n    ** (MatchError) no match of right hand side value: :bar\n\nAs with any other function or macro call in Elixir, explicit parens can\nalso be used around the arguments before the `do`-`end` block:\n\n    iex> opts = %{width: 10, height: 15}\n    iex> with(\n    ...>   {:ok, width}     {:ok, height}   ) do\n    ...>   {:ok, width * height}\n    ...> end\n    {:ok, 150}\n\nThe choice between parens and no parens is a matter of preference.","ref":"Kernel.SpecialForms.html#with/1"},{"type":"macro","title":"Else clauses - Kernel.SpecialForms.with/1","doc":"An `else` option can be given to modify what is being returned from\n`with` in the case of a failed match:\n\n    iex> opts = %{width: 10}\n    iex> with {:ok, width}        {:ok, height}     {:ok, width * height}\n    ...> else\n    ...>   :error ->\n    ...>     {:error, :wrong_data}\n    ...>\n    ...>   _other_error ->\n    ...>     :unexpected_error\n    ...> end\n    {:error, :wrong_data}\n\nThe `else` block works like a `case` clause: it can have multiple clauses,\nand the first match will be used. Variables bound inside `with` (such as\n`width` in this example) are not available in the `else` block.\n\nIf an `else` block is used and there are no matching clauses, a `WithClauseError`\nexception is raised.\n\n#","ref":"Kernel.SpecialForms.html#with/1-else-clauses"},{"type":"macro","title":"Beware! - Kernel.SpecialForms.with/1","doc":"Keep in mind that, one of potential drawback of `with` is that all\nfailure clauses are flattened into a single `else` block. For example,\ntake this code that checks if a given path points to an Elixir file\nand that it exists before creating a backup copy:\n\n    with \".ex\"   \".backup\"\n      File.cp!(path, backup_path)\n      {:ok, backup_path}\n    else\n      binary when is_binary(binary) ->\n        {:error, :invalid_extension}\n\n      false ->\n        {:error, :missing_file}\n    end\n\nNote how we are having to reconstruct the result types of `Path.extname/1`\nand `File.exists?/1` to build error messages. In this case, it is better\nto refactor the code so each `  \".backup\"\n      File.cp!(path, backup_path)\n      {:ok, backup_path}\n    end\n\n    defp validate_extension(path) do\n      if Path.extname(path) == \".ex\", do: :ok, else: {:error, :invalid_extension}\n    end\n\n    defp validate_exists(path) do\n      if File.exists?(path), do: :ok, else: {:error, :missing_file}\n    end\n\nNote how the code above is better organized and clearer once we\nmake sure each `<-` in `with` returns a normalized format.","ref":"Kernel.SpecialForms.html#with/1-beware"},{"type":"macro","title":"Kernel.SpecialForms.^/1","doc":"Pin operator. Accesses an already bound variable in match clauses.","ref":"Kernel.SpecialForms.html#%5E/1"},{"type":"macro","title":"Examples - Kernel.SpecialForms.^/1","doc":"Elixir allows variables to be rebound via static single assignment:\n\n    iex> x = 1\n    iex> x = x + 1\n    iex> x\n    2\n\nHowever, in some situations, it is useful to match against an existing\nvalue, instead of rebinding. This can be done with the `^` special form,\ncolloquially known as the pin operator:\n\n    iex> x = 1\n    iex> ^x = List.first([1])\n    iex> ^x = List.first([2])\n    ** (MatchError) no match of right hand side value: 2\n\nNote that `^x` always refers to the value of `x` prior to the match. The\nfollowing example will match:\n\n    iex> x = 0\n    iex> {x, ^x} = {1, 0}\n    iex> x\n    1","ref":"Kernel.SpecialForms.html#%5E/1-examples"},{"type":"macro","title":"Kernel.SpecialForms.{}/1","doc":"Creates a tuple.\n\nMore information about the tuple data type and about functions to manipulate\ntuples can be found in the `Tuple` module; some functions for working with\ntuples are also available in `Kernel` (such as `Kernel.elem/2` or\n`Kernel.tuple_size/1`).","ref":"Kernel.SpecialForms.html#%7B%7D/1"},{"type":"macro","title":"AST representation - Kernel.SpecialForms.{}/1","doc":"Only two-element tuples are considered literals in Elixir and return themselves\nwhen quoted. Therefore, all other tuples are represented in the AST as calls to\nthe `:{}` special form.\n\n    iex> quote do\n    ...>   {1, 2}\n    ...> end\n    {1, 2}\n\n    iex> quote do\n    ...>   {1, 2, 3}\n    ...> end\n    {:{}, [], [1, 2, 3]}","ref":"Kernel.SpecialForms.html#%7B%7D/1-ast-representation"},{"type":"module","title":"Atom","doc":"Atoms are constants whose values are their own name.\n\nThey are often useful to enumerate over distinct values, such as:\n\n    iex> :apple\n    :apple\n    iex> :orange\n    :orange\n    iex> :watermelon\n    :watermelon\n\nAtoms are equal if their names are equal.\n\n    iex> :apple == :apple\n    true\n    iex> :apple == :orange\n    false\n\nOften they are used to express the state of an operation, by using\nvalues such as `:ok` and `:error`.\n\nThe booleans `true` and `false` are also atoms:\n\n    iex> true == :true\n    true\n    iex> is_atom(false)\n    true\n    iex> is_boolean(:false)\n    true\n\nElixir allows you to skip the leading `:` for the atoms `false`, `true`,\nand `nil`.\n\nAtoms must be composed of Unicode characters such as letters, numbers,\nunderscore, and `@`. If the keyword has a character that does not\nbelong to the category above, such as spaces, you can wrap it in\nquotes:\n\n    iex> :\"this is an atom with spaces\"\n    :\"this is an atom with spaces\"","ref":"Atom.html"},{"type":"function","title":"Atom.to_charlist/1","doc":"Converts an atom to a charlist.\n\nInlined by the compiler.","ref":"Atom.html#to_charlist/1"},{"type":"function","title":"Examples - Atom.to_charlist/1","doc":"iex> Atom.to_charlist(:\"An atom\")\n    'An atom'","ref":"Atom.html#to_charlist/1-examples"},{"type":"function","title":"Atom.to_string/1","doc":"Converts an atom to a string.\n\nInlined by the compiler.","ref":"Atom.html#to_string/1"},{"type":"function","title":"Examples - Atom.to_string/1","doc":"iex> Atom.to_string(:foo)\n    \"foo\"","ref":"Atom.html#to_string/1-examples"},{"type":"module","title":"Base","doc":"This module provides data encoding and decoding functions\naccording to [RFC 4648](https://tools.ietf.org/html/rfc4648).\n\nThis document defines the commonly used base 16, base 32, and base\n64 encoding schemes.","ref":"Base.html"},{"type":"module","title":"Base 16 alphabet - Base","doc":"| Value | Encoding | Value | Encoding | Value | Encoding | Value | Encoding |\n|------:|:---------|------:|:---------|------:|:---------|------:|:---------|\n|     0 | 0        |     4 | 4        |     8 | 8        |    12 | C        |\n|     1 | 1        |     5 | 5        |     9 | 9        |    13 | D        |\n|     2 | 2        |     6 | 6        |    10 | A        |    14 | E        |\n|     3 | 3        |     7 | 7        |    11 | B        |    15 | F        |","ref":"Base.html#module-base-16-alphabet"},{"type":"module","title":"Base 32 alphabet - Base","doc":"| Value | Encoding | Value | Encoding | Value | Encoding | Value | Encoding |\n|------:|:---------|------:|:---------|------:|:---------|------:|:---------|\n|     0 | A        |     9 | J        |    18 | S        |    27 | 3        |\n|     1 | B        |    10 | K        |    19 | T        |    28 | 4        |\n|     2 | C        |    11 | L        |    20 | U        |    29 | 5        |\n|     3 | D        |    12 | M        |    21 | V        |    30 | 6        |\n|     4 | E        |    13 | N        |    22 | W        |    31 | 7        |\n|     5 | F        |    14 | O        |    23 | X        |       |          |\n|     6 | G        |    15 | P        |    24 | Y        | (pad) | =        |\n|     7 | H        |    16 | Q        |    25 | Z        |       |          |\n|     8 | I        |    17 | R        |    26 | 2        |       |          |","ref":"Base.html#module-base-32-alphabet"},{"type":"module","title":"Base 32 (extended hex) alphabet - Base","doc":"| Value | Encoding | Value | Encoding | Value | Encoding | Value | Encoding |\n|------:|:---------|------:|:---------|------:|:---------|------:|:---------|\n|     0 | 0        |     9 | 9        |    18 | I        |    27 | R        |\n|     1 | 1        |    10 | A        |    19 | J        |    28 | S        |\n|     2 | 2        |    11 | B        |    20 | K        |    29 | T        |\n|     3 | 3        |    12 | C        |    21 | L        |    30 | U        |\n|     4 | 4        |    13 | D        |    22 | M        |    31 | V        |\n|     5 | 5        |    14 | E        |    23 | N        |       |          |\n|     6 | 6        |    15 | F        |    24 | O        | (pad) | =        |\n|     7 | 7        |    16 | G        |    25 | P        |       |          |\n|     8 | 8        |    17 | H        |    26 | Q        |       |          |","ref":"Base.html#module-base-32-extended-hex-alphabet"},{"type":"module","title":"Base 64 alphabet - Base","doc":"| Value |  Encoding | Value | Encoding | Value | Encoding | Value | Encoding |\n|------:|:----------|------:|:---------|------:|:---------|------:|:---------|\n|     0 | A         |    17 | R        |    34 | i        |    51 | z        |\n|     1 | B         |    18 | S        |    35 | j        |    52 | 0        |\n|     2 | C         |    19 | T        |    36 | k        |    53 | 1        |\n|     3 | D         |    20 | U        |    37 | l        |    54 | 2        |\n|     4 | E         |    21 | V        |    38 | m        |    55 | 3        |\n|     5 | F         |    22 | W        |    39 | n        |    56 | 4        |\n|     6 | G         |    23 | X        |    40 | o        |    57 | 5        |\n|     7 | H         |    24 | Y        |    41 | p        |    58 | 6        |\n|     8 | I         |    25 | Z        |    42 | q        |    59 | 7        |\n|     9 | J         |    26 | a        |    43 | r        |    60 | 8        |\n|    10 | K         |    27 | b        |    44 | s        |    61 | 9        |\n|    11 | L         |    28 | c        |    45 | t        |    62 | +        |\n|    12 | M         |    29 | d        |    46 | u        |    63 | /        |\n|    13 | N         |    30 | e        |    47 | v        |       |          |\n|    14 | O         |    31 | f        |    48 | w        | (pad) | =        |\n|    15 | P         |    32 | g        |    49 | x        |       |          |\n|    16 | Q         |    33 | h        |    50 | y        |       |          |","ref":"Base.html#module-base-64-alphabet"},{"type":"module","title":"Base 64 (URL and filename safe) alphabet - Base","doc":"| Value | Encoding | Value | Encoding | Value | Encoding | Value | Encoding |\n|------:|:---------|------:|:---------|------:|:---------|------:|:---------|\n|     0 | A        |    17 | R        |    34 | i        |    51 | z        |\n|     1 | B        |    18 | S        |    35 | j        |    52 | 0        |\n|     2 | C        |    19 | T        |    36 | k        |    53 | 1        |\n|     3 | D        |    20 | U        |    37 | l        |    54 | 2        |\n|     4 | E        |    21 | V        |    38 | m        |    55 | 3        |\n|     5 | F        |    22 | W        |    39 | n        |    56 | 4        |\n|     6 | G        |    23 | X        |    40 | o        |    57 | 5        |\n|     7 | H        |    24 | Y        |    41 | p        |    58 | 6        |\n|     8 | I        |    25 | Z        |    42 | q        |    59 | 7        |\n|     9 | J        |    26 | a        |    43 | r        |    60 | 8        |\n|    10 | K        |    27 | b        |    44 | s        |    61 | 9        |\n|    11 | L        |    28 | c        |    45 | t        |    62 | -        |\n|    12 | M        |    29 | d        |    46 | u        |    63 | _        |\n|    13 | N        |    30 | e        |    47 | v        |       |          |\n|    14 | O        |    31 | f        |    48 | w        | (pad) | =        |\n|    15 | P        |    32 | g        |    49 | x        |       |          |\n|    16 | Q        |    33 | h        |    50 | y        |       |          |","ref":"Base.html#module-base-64-url-and-filename-safe-alphabet"},{"type":"function","title":"Base.decode16/2","doc":"Decodes a base 16 encoded string into a binary string.","ref":"Base.html#decode16/2"},{"type":"function","title":"Options - Base.decode16/2","doc":"The accepted options are:\n\n  * `:case` - specifies the character case to accept when decoding\n\nThe values for `:case` can be:\n\n  * `:upper` - only allows upper case characters (default)\n  * `:lower` - only allows lower case characters\n  * `:mixed` - allows mixed case characters","ref":"Base.html#decode16/2-options"},{"type":"function","title":"Examples - Base.decode16/2","doc":"iex> Base.decode16(\"666F6F626172\")\n    {:ok, \"foobar\"}\n\n    iex> Base.decode16(\"666f6f626172\", case: :lower)\n    {:ok, \"foobar\"}\n\n    iex> Base.decode16(\"666f6F626172\", case: :mixed)\n    {:ok, \"foobar\"}","ref":"Base.html#decode16/2-examples"},{"type":"function","title":"Base.decode16!/2","doc":"Decodes a base 16 encoded string into a binary string.","ref":"Base.html#decode16!/2"},{"type":"function","title":"Options - Base.decode16!/2","doc":"The accepted options are:\n\n  * `:case` - specifies the character case to accept when decoding\n\nThe values for `:case` can be:\n\n  * `:upper` - only allows upper case characters (default)\n  * `:lower` - only allows lower case characters\n  * `:mixed` - allows mixed case characters\n\nAn `ArgumentError` exception is raised if the padding is incorrect or\na non-alphabet character is present in the string.","ref":"Base.html#decode16!/2-options"},{"type":"function","title":"Examples - Base.decode16!/2","doc":"iex> Base.decode16!(\"666F6F626172\")\n    \"foobar\"\n\n    iex> Base.decode16!(\"666f6f626172\", case: :lower)\n    \"foobar\"\n\n    iex> Base.decode16!(\"666f6F626172\", case: :mixed)\n    \"foobar\"","ref":"Base.html#decode16!/2-examples"},{"type":"function","title":"Base.decode32/2","doc":"Decodes a base 32 encoded string into a binary string.","ref":"Base.html#decode32/2"},{"type":"function","title":"Options - Base.decode32/2","doc":"The accepted options are:\n\n  * `:case` - specifies the character case to accept when decoding\n  * `:padding` - specifies whether to require padding\n\nThe values for `:case` can be:\n\n  * `:upper` - only allows  upper case characters (default)\n  * `:lower` - only allows lower case characters\n  * `:mixed` - allows mixed case characters\n\nThe values for `:padding` can be:\n\n  * `true` - requires the input string to be padded to the nearest multiple of 8 (default)\n  * `false` - ignores padding from the input string","ref":"Base.html#decode32/2-options"},{"type":"function","title":"Examples - Base.decode32/2","doc":"iex> Base.decode32(\"MZXW6YTBOI======\")\n    {:ok, \"foobar\"}\n\n    iex> Base.decode32(\"mzxw6ytboi======\", case: :lower)\n    {:ok, \"foobar\"}\n\n    iex> Base.decode32(\"mzXW6ytBOi======\", case: :mixed)\n    {:ok, \"foobar\"}\n\n    iex> Base.decode32(\"MZXW6YTBOI\", padding: false)\n    {:ok, \"foobar\"}","ref":"Base.html#decode32/2-examples"},{"type":"function","title":"Base.decode32!/2","doc":"Decodes a base 32 encoded string into a binary string.\n\nAn `ArgumentError` exception is raised if the padding is incorrect or\na non-alphabet character is present in the string.","ref":"Base.html#decode32!/2"},{"type":"function","title":"Options - Base.decode32!/2","doc":"The accepted options are:\n\n  * `:case` - specifies the character case to accept when decoding\n  * `:padding` - specifies whether to require padding\n\nThe values for `:case` can be:\n\n  * `:upper` - only allows upper case characters (default)\n  * `:lower` - only allows lower case characters\n  * `:mixed` - allows mixed case characters\n\nThe values for `:padding` can be:\n\n  * `true` - requires the input string to be padded to the nearest multiple of 8 (default)\n  * `false` - ignores padding from the input string","ref":"Base.html#decode32!/2-options"},{"type":"function","title":"Examples - Base.decode32!/2","doc":"iex> Base.decode32!(\"MZXW6YTBOI======\")\n    \"foobar\"\n\n    iex> Base.decode32!(\"mzxw6ytboi======\", case: :lower)\n    \"foobar\"\n\n    iex> Base.decode32!(\"mzXW6ytBOi======\", case: :mixed)\n    \"foobar\"\n\n    iex> Base.decode32!(\"MZXW6YTBOI\", padding: false)\n    \"foobar\"","ref":"Base.html#decode32!/2-examples"},{"type":"function","title":"Base.decode64/2","doc":"Decodes a base 64 encoded string into a binary string.\n\nAccepts `ignore: :whitespace` option which will ignore all the\nwhitespace characters in the input string.\n\nAccepts `padding: false` option which will ignore padding from\nthe input string.","ref":"Base.html#decode64/2"},{"type":"function","title":"Examples - Base.decode64/2","doc":"iex> Base.decode64(\"Zm9vYmFy\")\n    {:ok, \"foobar\"}\n\n    iex> Base.decode64(\"Zm9vYmFy\\n\", ignore: :whitespace)\n    {:ok, \"foobar\"}\n\n    iex> Base.decode64(\"Zm9vYg==\")\n    {:ok, \"foob\"}\n\n    iex> Base.decode64(\"Zm9vYg\", padding: false)\n    {:ok, \"foob\"}","ref":"Base.html#decode64/2-examples"},{"type":"function","title":"Base.decode64!/2","doc":"Decodes a base 64 encoded string into a binary string.\n\nAccepts `ignore: :whitespace` option which will ignore all the\nwhitespace characters in the input string.\n\nAccepts `padding: false` option which will ignore padding from\nthe input string.\n\nAn `ArgumentError` exception is raised if the padding is incorrect or\na non-alphabet character is present in the string.","ref":"Base.html#decode64!/2"},{"type":"function","title":"Examples - Base.decode64!/2","doc":"iex> Base.decode64!(\"Zm9vYmFy\")\n    \"foobar\"\n\n    iex> Base.decode64!(\"Zm9vYmFy\\n\", ignore: :whitespace)\n    \"foobar\"\n\n    iex> Base.decode64!(\"Zm9vYg==\")\n    \"foob\"\n\n    iex> Base.decode64!(\"Zm9vYg\", padding: false)\n    \"foob\"","ref":"Base.html#decode64!/2-examples"},{"type":"function","title":"Base.encode16/2","doc":"Encodes a binary string into a base 16 encoded string.","ref":"Base.html#encode16/2"},{"type":"function","title":"Options - Base.encode16/2","doc":"The accepted options are:\n\n  * `:case` - specifies the character case to use when encoding\n\nThe values for `:case` can be:\n\n  * `:upper` - uses upper case characters (default)\n  * `:lower` - uses lower case characters","ref":"Base.html#encode16/2-options"},{"type":"function","title":"Examples - Base.encode16/2","doc":"iex> Base.encode16(\"foobar\")\n    \"666F6F626172\"\n\n    iex> Base.encode16(\"foobar\", case: :lower)\n    \"666f6f626172\"","ref":"Base.html#encode16/2-examples"},{"type":"function","title":"Base.encode32/2","doc":"Encodes a binary string into a base 32 encoded string.","ref":"Base.html#encode32/2"},{"type":"function","title":"Options - Base.encode32/2","doc":"The accepted options are:\n\n  * `:case` - specifies the character case to use when encoding\n  * `:padding` - specifies whether to apply padding\n\nThe values for `:case` can be:\n\n  * `:upper` - uses upper case characters (default)\n  * `:lower` - uses lower case characters\n\nThe values for `:padding` can be:\n\n  * `true` - pad the output string to the nearest multiple of 8 (default)\n  * `false` - omit padding from the output string","ref":"Base.html#encode32/2-options"},{"type":"function","title":"Examples - Base.encode32/2","doc":"iex> Base.encode32(\"foobar\")\n    \"MZXW6YTBOI======\"\n\n    iex> Base.encode32(\"foobar\", case: :lower)\n    \"mzxw6ytboi======\"\n\n    iex> Base.encode32(\"foobar\", padding: false)\n    \"MZXW6YTBOI\"","ref":"Base.html#encode32/2-examples"},{"type":"function","title":"Base.encode64/2","doc":"Encodes a binary string into a base 64 encoded string.\n\nAccepts `padding: false` option which will omit padding from\nthe output string.","ref":"Base.html#encode64/2"},{"type":"function","title":"Examples - Base.encode64/2","doc":"iex> Base.encode64(\"foobar\")\n    \"Zm9vYmFy\"\n\n    iex> Base.encode64(\"foob\")\n    \"Zm9vYg==\"\n\n    iex> Base.encode64(\"foob\", padding: false)\n    \"Zm9vYg\"","ref":"Base.html#encode64/2-examples"},{"type":"function","title":"Base.hex_decode32/2","doc":"Decodes a base 32 encoded string with extended hexadecimal alphabet\ninto a binary string.","ref":"Base.html#hex_decode32/2"},{"type":"function","title":"Options - Base.hex_decode32/2","doc":"The accepted options are:\n\n  * `:case` - specifies the character case to accept when decoding\n  * `:padding` - specifies whether to require padding\n\nThe values for `:case` can be:\n\n  * `:upper` - only allows upper case characters (default)\n  * `:lower` - only allows lower case characters\n  * `:mixed` - allows mixed case characters\n\nThe values for `:padding` can be:\n\n  * `true` - requires the input string to be padded to the nearest multiple of 8 (default)\n  * `false` - ignores padding from the input string","ref":"Base.html#hex_decode32/2-options"},{"type":"function","title":"Examples - Base.hex_decode32/2","doc":"iex> Base.hex_decode32(\"CPNMUOJ1E8======\")\n    {:ok, \"foobar\"}\n\n    iex> Base.hex_decode32(\"cpnmuoj1e8======\", case: :lower)\n    {:ok, \"foobar\"}\n\n    iex> Base.hex_decode32(\"cpnMuOJ1E8======\", case: :mixed)\n    {:ok, \"foobar\"}\n\n    iex> Base.hex_decode32(\"CPNMUOJ1E8\", padding: false)\n    {:ok, \"foobar\"}","ref":"Base.html#hex_decode32/2-examples"},{"type":"function","title":"Base.hex_decode32!/2","doc":"Decodes a base 32 encoded string with extended hexadecimal alphabet\ninto a binary string.\n\nAn `ArgumentError` exception is raised if the padding is incorrect or\na non-alphabet character is present in the string.","ref":"Base.html#hex_decode32!/2"},{"type":"function","title":"Options - Base.hex_decode32!/2","doc":"The accepted options are:\n\n  * `:case` - specifies the character case to accept when decoding\n  * `:padding` - specifies whether to require padding\n\nThe values for `:case` can be:\n\n  * `:upper` - only allows upper case characters (default)\n  * `:lower` - only allows lower case characters\n  * `:mixed` - allows mixed case characters\n\nThe values for `:padding` can be:\n\n  * `true` - requires the input string to be padded to the nearest multiple of 8 (default)\n  * `false` - ignores padding from the input string","ref":"Base.html#hex_decode32!/2-options"},{"type":"function","title":"Examples - Base.hex_decode32!/2","doc":"iex> Base.hex_decode32!(\"CPNMUOJ1E8======\")\n    \"foobar\"\n\n    iex> Base.hex_decode32!(\"cpnmuoj1e8======\", case: :lower)\n    \"foobar\"\n\n    iex> Base.hex_decode32!(\"cpnMuOJ1E8======\", case: :mixed)\n    \"foobar\"\n\n    iex> Base.hex_decode32!(\"CPNMUOJ1E8\", padding: false)\n    \"foobar\"","ref":"Base.html#hex_decode32!/2-examples"},{"type":"function","title":"Base.hex_encode32/2","doc":"Encodes a binary string into a base 32 encoded string with an\nextended hexadecimal alphabet.","ref":"Base.html#hex_encode32/2"},{"type":"function","title":"Options - Base.hex_encode32/2","doc":"The accepted options are:\n\n  * `:case` - specifies the character case to use when encoding\n  * `:padding` - specifies whether to apply padding\n\nThe values for `:case` can be:\n\n  * `:upper` - uses upper case characters (default)\n  * `:lower` - uses lower case characters\n\nThe values for `:padding` can be:\n\n  * `true` - pad the output string to the nearest multiple of 8 (default)\n  * `false` - omit padding from the output string","ref":"Base.html#hex_encode32/2-options"},{"type":"function","title":"Examples - Base.hex_encode32/2","doc":"iex> Base.hex_encode32(\"foobar\")\n    \"CPNMUOJ1E8======\"\n\n    iex> Base.hex_encode32(\"foobar\", case: :lower)\n    \"cpnmuoj1e8======\"\n\n    iex> Base.hex_encode32(\"foobar\", padding: false)\n    \"CPNMUOJ1E8\"","ref":"Base.html#hex_encode32/2-examples"},{"type":"function","title":"Base.url_decode64/2","doc":"Decodes a base 64 encoded string with URL and filename safe alphabet\ninto a binary string.\n\nAccepts `ignore: :whitespace` option which will ignore all the\nwhitespace characters in the input string.\n\nAccepts `padding: false` option which will ignore padding from\nthe input string.","ref":"Base.html#url_decode64/2"},{"type":"function","title":"Examples - Base.url_decode64/2","doc":"iex> Base.url_decode64(\"_3_-_A==\")\n    {:ok,  >}\n\n    iex> Base.url_decode64(\"_3_-_A==\\n\", ignore: :whitespace)\n    {:ok,  >}\n\n    iex> Base.url_decode64(\"_3_-_A\", padding: false)\n    {:ok,  >}","ref":"Base.html#url_decode64/2-examples"},{"type":"function","title":"Base.url_decode64!/2","doc":"Decodes a base 64 encoded string with URL and filename safe alphabet\ninto a binary string.\n\nAccepts `ignore: :whitespace` option which will ignore all the\nwhitespace characters in the input string.\n\nAccepts `padding: false` option which will ignore padding from\nthe input string.\n\nAn `ArgumentError` exception is raised if the padding is incorrect or\na non-alphabet character is present in the string.","ref":"Base.html#url_decode64!/2"},{"type":"function","title":"Examples - Base.url_decode64!/2","doc":"iex> Base.url_decode64!(\"_3_-_A==\")\n     >\n\n    iex> Base.url_decode64!(\"_3_-_A==\\n\", ignore: :whitespace)\n     >\n\n    iex> Base.url_decode64!(\"_3_-_A\", padding: false)\n     >","ref":"Base.html#url_decode64!/2-examples"},{"type":"function","title":"Base.url_encode64/2","doc":"Encodes a binary string into a base 64 encoded string with URL and filename\nsafe alphabet.\n\nAccepts `padding: false` option which will omit padding from\nthe output string.","ref":"Base.html#url_encode64/2"},{"type":"function","title":"Examples - Base.url_encode64/2","doc":"iex> Base.url_encode64( >)\n    \"_3_-_A==\"\n\n    iex> Base.url_encode64( >, padding: false)\n    \"_3_-_A\"","ref":"Base.html#url_encode64/2-examples"},{"type":"type","title":"Base.decode_case/0","doc":"","ref":"Base.html#t:decode_case/0"},{"type":"type","title":"Base.encode_case/0","doc":"","ref":"Base.html#t:encode_case/0"},{"type":"module","title":"Bitwise","doc":"A set of functions that perform calculations on bits.\n\nAll bitwise functions work only on integers; otherwise an\n`ArithmeticError` is raised. The functions `band/2`,\n`bor/2`, `bsl/2`, and `bsr/2` also have operators,\nrespectively: `&&&/2`, `|||/2`, ` >>/2`.","ref":"Bitwise.html"},{"type":"module","title":"Guards - Bitwise","doc":"All bitwise functions can be used in guards:\n\n    iex> odd? = fn\n    ...>   int when Bitwise.band(int, 1) == 1 -> true\n    ...>   _ -> false\n    ...> end\n    iex> odd?.(1)\n    true\n\nAll functions in this module are inlined by the compiler.","ref":"Bitwise.html#module-guards"},{"type":"function","title":"Bitwise.&&&/2","doc":"Bitwise AND operator.\n\nCalculates the bitwise AND of its arguments.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Bitwise.html#&&&/2"},{"type":"function","title":"Examples - Bitwise.&&&/2","doc":"iex> 9 &&& 3\n    1","ref":"Bitwise.html#&&&/2-examples"},{"type":"function","title":"Bitwise.<<</2","doc":"Arithmetic left bitshift operator.\n\nCalculates the result of an arithmetic left bitshift.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Bitwise.html#%3C%3C%3C/2"},{"type":"function","title":"Examples - Bitwise.<<</2","doc":"iex> 1   1   -1   -1 <<< -2\n    -1","ref":"Bitwise.html#%3C%3C%3C/2-examples"},{"type":"function","title":"Bitwise.>>>/2","doc":"Arithmetic right bitshift operator.\n\nCalculates the result of an arithmetic right bitshift.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Bitwise.html#%3E%3E%3E/2"},{"type":"function","title":"Examples - Bitwise.>>>/2","doc":"iex> 1 >>> 2\n    0\n\n    iex> 1 >>> -2\n    4\n\n    iex> -1 >>> 2\n    -1\n\n    iex> -1 >>> -2\n    -4","ref":"Bitwise.html#%3E%3E%3E/2-examples"},{"type":"function","title":"Bitwise.band/2","doc":"Calculates the bitwise AND of its arguments.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Bitwise.html#band/2"},{"type":"function","title":"Examples - Bitwise.band/2","doc":"iex> band(9, 3)\n    1","ref":"Bitwise.html#band/2-examples"},{"type":"function","title":"Bitwise.bnot/1","doc":"Calculates the bitwise NOT of the argument.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Bitwise.html#bnot/1"},{"type":"function","title":"Examples - Bitwise.bnot/1","doc":"iex> bnot(2)\n    -3\n\n    iex> bnot(2) &&& 3\n    1","ref":"Bitwise.html#bnot/1-examples"},{"type":"function","title":"Bitwise.bor/2","doc":"Calculates the bitwise OR of its arguments.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Bitwise.html#bor/2"},{"type":"function","title":"Examples - Bitwise.bor/2","doc":"iex> bor(9, 3)\n    11","ref":"Bitwise.html#bor/2-examples"},{"type":"function","title":"Bitwise.bsl/2","doc":"Calculates the result of an arithmetic left bitshift.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Bitwise.html#bsl/2"},{"type":"function","title":"Examples - Bitwise.bsl/2","doc":"iex> bsl(1, 2)\n    4\n\n    iex> bsl(1, -2)\n    0\n\n    iex> bsl(-1, 2)\n    -4\n\n    iex> bsl(-1, -2)\n    -1","ref":"Bitwise.html#bsl/2-examples"},{"type":"function","title":"Bitwise.bsr/2","doc":"Calculates the result of an arithmetic right bitshift.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Bitwise.html#bsr/2"},{"type":"function","title":"Examples - Bitwise.bsr/2","doc":"iex> bsr(1, 2)\n    0\n\n    iex> bsr(1, -2)\n    4\n\n    iex> bsr(-1, 2)\n    -1\n\n    iex> bsr(-1, -2)\n    -4","ref":"Bitwise.html#bsr/2-examples"},{"type":"function","title":"Bitwise.bxor/2","doc":"Calculates the bitwise XOR of its arguments.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Bitwise.html#bxor/2"},{"type":"function","title":"Examples - Bitwise.bxor/2","doc":"iex> bxor(9, 3)\n    10","ref":"Bitwise.html#bxor/2-examples"},{"type":"function","title":"Bitwise.|||/2","doc":"Bitwise OR operator.\n\nCalculates the bitwise OR of its arguments.\n\nAllowed in guard tests. Inlined by the compiler.","ref":"Bitwise.html#%7C%7C%7C/2"},{"type":"function","title":"Examples - Bitwise.|||/2","doc":"iex> 9 ||| 3\n    11","ref":"Bitwise.html#%7C%7C%7C/2-examples"},{"type":"module","title":"Date","doc":"A Date struct and functions.\n\nThe Date struct contains the fields year, month, day and calendar.\nNew dates can be built with the `new/3` function or using the\n`~D` (see `sigil_D/2`) sigil:\n\n    iex> ~D[2000-01-01]\n    ~D[2000-01-01]\n\nBoth `new/3` and sigil return a struct where the date fields can\nbe accessed directly:\n\n    iex> date = ~D[2000-01-01]\n    iex> date.year\n    2000\n    iex> date.month\n    1\n\nThe functions on this module work with the `Date` struct as well\nas any struct that contains the same fields as the `Date` struct,\nsuch as `NaiveDateTime` and `DateTime`. Such functions expect\n`t:Calendar.date/0` in their typespecs (instead of `t:t/0`).\n\nDevelopers should avoid creating the Date structs directly\nand instead rely on the functions provided by this module as well\nas the ones in third-party calendar libraries.","ref":"Date.html"},{"type":"module","title":"Comparing dates - Date","doc":"Comparisons in Elixir using `==/2`, `>/2`, `   Enum.min([~D[2017-03-31], ~D[2017-04-01]], Date)\n    ~D[2017-03-31]","ref":"Date.html#module-comparing-dates"},{"type":"module","title":"Using epochs - Date","doc":"The `add/2` and `diff/2` functions can be used for computing dates\nor retrieving the number of days between instants. For example, if there\nis an interest in computing the number of days from the Unix epoch\n(1970-01-01):\n\n    iex> Date.diff(~D[2010-04-17], ~D[1970-01-01])\n    14716\n\n    iex> Date.add(~D[1970-01-01], 14716)\n    ~D[2010-04-17]\n\nThose functions are optimized to deal with common epochs, such\nas the Unix Epoch above or the Gregorian Epoch (0000-01-01).","ref":"Date.html#module-using-epochs"},{"type":"function","title":"Date.add/2","doc":"Adds the number of days to the given `date`.\n\nThe days are counted as Gregorian days. The date is returned in the same\ncalendar as it was given in.","ref":"Date.html#add/2"},{"type":"function","title":"Examples - Date.add/2","doc":"iex> Date.add(~D[2000-01-03], -2)\n    ~D[2000-01-01]\n    iex> Date.add(~D[2000-01-01], 2)\n    ~D[2000-01-03]\n    iex> Date.add(~N[2000-01-01 09:00:00], 2)\n    ~D[2000-01-03]\n    iex> Date.add(~D[-0010-01-01], -2)\n    ~D[-0011-12-30]","ref":"Date.html#add/2-examples"},{"type":"function","title":"Date.after?/2","doc":"Returns true if the first date is strictly later than the second.","ref":"Date.html#after?/2"},{"type":"function","title":"Examples - Date.after?/2","doc":"iex> Date.after?(~D[2022-02-02], ~D[2021-01-01])\n    true\n    iex> Date.after?(~D[2021-01-01], ~D[2021-01-01])\n    false\n    iex> Date.after?(~D[2021-01-01], ~D[2022-02-02])\n    false","ref":"Date.html#after?/2-examples"},{"type":"function","title":"Date.before?/2","doc":"Returns true if the first date is strictly earlier than the second.","ref":"Date.html#before?/2"},{"type":"function","title":"Examples - Date.before?/2","doc":"iex> Date.before?(~D[2021-01-01], ~D[2022-02-02])\n    true\n    iex> Date.before?(~D[2021-01-01], ~D[2021-01-01])\n    false\n    iex> Date.before?(~D[2022-02-02], ~D[2021-01-01])\n    false","ref":"Date.html#before?/2-examples"},{"type":"function","title":"Date.beginning_of_month/1","doc":"Calculates a date that is the first day of the month for the given `date`.","ref":"Date.html#beginning_of_month/1"},{"type":"function","title":"Examples - Date.beginning_of_month/1","doc":"iex> Date.beginning_of_month(~D[2000-01-31])\n    ~D[2000-01-01]\n    iex> Date.beginning_of_month(~D[2000-01-01])\n    ~D[2000-01-01]\n    iex> Date.beginning_of_month(~N[2000-01-31 01:23:45])\n    ~D[2000-01-01]","ref":"Date.html#beginning_of_month/1-examples"},{"type":"function","title":"Date.beginning_of_week/2","doc":"Calculates a date that is the first day of the week for the given `date`.\n\nIf the day is already the first day of the week, it returns the\nday itself. For the built-in ISO calendar, the week starts on Monday.\nA weekday rather than `:default` can be given as `starting_on`.","ref":"Date.html#beginning_of_week/2"},{"type":"function","title":"Examples - Date.beginning_of_week/2","doc":"iex> Date.beginning_of_week(~D[2020-07-11])\n    ~D[2020-07-06]\n    iex> Date.beginning_of_week(~D[2020-07-06])\n    ~D[2020-07-06]\n    iex> Date.beginning_of_week(~D[2020-07-11], :sunday)\n    ~D[2020-07-05]\n    iex> Date.beginning_of_week(~D[2020-07-11], :saturday)\n    ~D[2020-07-11]\n    iex> Date.beginning_of_week(~N[2020-07-11 01:23:45])\n    ~D[2020-07-06]","ref":"Date.html#beginning_of_week/2-examples"},{"type":"function","title":"Date.compare/2","doc":"Compares two date structs.\n\nReturns `:gt` if first date is later than the second\nand `:lt` for vice versa. If the two dates are equal\n`:eq` is returned.","ref":"Date.html#compare/2"},{"type":"function","title":"Examples - Date.compare/2","doc":"iex> Date.compare(~D[2016-04-16], ~D[2016-04-28])\n    :lt\n\nThis function can also be used to compare across more\ncomplex calendar types by considering only the date fields:\n\n    iex> Date.compare(~D[2016-04-16], ~N[2016-04-28 01:23:45])\n    :lt\n    iex> Date.compare(~D[2016-04-16], ~N[2016-04-16 01:23:45])\n    :eq\n    iex> Date.compare(~N[2016-04-16 12:34:56], ~N[2016-04-16 01:23:45])\n    :eq","ref":"Date.html#compare/2-examples"},{"type":"function","title":"Date.convert/2","doc":"Converts the given `date` from its calendar to the given `calendar`.\n\nReturns `{:ok, date}` if the calendars are compatible,\nor `{:error, :incompatible_calendars}` if they are not.\n\nSee also `Calendar.compatible_calendars?/2`.","ref":"Date.html#convert/2"},{"type":"function","title":"Examples - Date.convert/2","doc":"Imagine someone implements `Calendar.Holocene`, a calendar based on the\nGregorian calendar that adds exactly 10,000 years to the current Gregorian\nyear:\n\n    iex> Date.convert(~D[2000-01-01], Calendar.Holocene)\n    {:ok, %Date{calendar: Calendar.Holocene, year: 12000, month: 1, day: 1}}","ref":"Date.html#convert/2-examples"},{"type":"function","title":"Date.convert!/2","doc":"Similar to `Date.convert/2`, but raises an `ArgumentError`\nif the conversion between the two calendars is not possible.","ref":"Date.html#convert!/2"},{"type":"function","title":"Examples - Date.convert!/2","doc":"Imagine someone implements `Calendar.Holocene`, a calendar based on the\nGregorian calendar that adds exactly 10,000 years to the current Gregorian\nyear:\n\n    iex> Date.convert!(~D[2000-01-01], Calendar.Holocene)\n    %Date{calendar: Calendar.Holocene, year: 12000, month: 1, day: 1}","ref":"Date.html#convert!/2-examples"},{"type":"function","title":"Date.day_of_era/1","doc":"Calculates the day-of-era and era for a given\ncalendar `date`.\n\nReturns a tuple `{day, era}` representing the\nday within the era and the era number.","ref":"Date.html#day_of_era/1"},{"type":"function","title":"Examples - Date.day_of_era/1","doc":"iex> Date.day_of_era(~D[0001-01-01])\n    {1, 1}\n\n    iex> Date.day_of_era(~D[0000-12-31])\n    {1, 0}","ref":"Date.html#day_of_era/1-examples"},{"type":"function","title":"Date.day_of_week/2","doc":"Calculates the day of the week of a given `date`.\n\nReturns the day of the week as an integer. For the ISO 8601\ncalendar (the default), it is an integer from 1 to 7, where\n1 is Monday and 7 is Sunday.\n\nAn optional `starting_on` value may be supplied, which\nconfigures the weekday the week starts on. The default value\nfor it is `:default`, which translates to `:monday` for the\nbuilt-in ISO calendar. Any other weekday may be given to.","ref":"Date.html#day_of_week/2"},{"type":"function","title":"Examples - Date.day_of_week/2","doc":"iex> Date.day_of_week(~D[2016-10-31])\n    1\n    iex> Date.day_of_week(~D[2016-11-01])\n    2\n    iex> Date.day_of_week(~N[2016-11-01 01:23:45])\n    2\n    iex> Date.day_of_week(~D[-0015-10-30])\n    3\n\n    iex> Date.day_of_week(~D[2016-10-31], :sunday)\n    2\n    iex> Date.day_of_week(~D[2016-11-01], :sunday)\n    3\n    iex> Date.day_of_week(~N[2016-11-01 01:23:45], :sunday)\n    3\n    iex> Date.day_of_week(~D[-0015-10-30], :sunday)\n    4","ref":"Date.html#day_of_week/2-examples"},{"type":"function","title":"Date.day_of_year/1","doc":"Calculates the day of the year of a given `date`.\n\nReturns the day of the year as an integer. For the ISO 8601\ncalendar (the default), it is an integer from 1 to 366.","ref":"Date.html#day_of_year/1"},{"type":"function","title":"Examples - Date.day_of_year/1","doc":"iex> Date.day_of_year(~D[2016-01-01])\n    1\n    iex> Date.day_of_year(~D[2016-11-01])\n    306\n    iex> Date.day_of_year(~D[-0015-10-30])\n    303\n    iex> Date.day_of_year(~D[2004-12-31])\n    366","ref":"Date.html#day_of_year/1-examples"},{"type":"function","title":"Date.days_in_month/1","doc":"Returns the number of days in the given `date` month.","ref":"Date.html#days_in_month/1"},{"type":"function","title":"Examples - Date.days_in_month/1","doc":"iex> Date.days_in_month(~D[1900-01-13])\n    31\n    iex> Date.days_in_month(~D[1900-02-09])\n    28\n    iex> Date.days_in_month(~N[2000-02-20 01:23:45])\n    29","ref":"Date.html#days_in_month/1-examples"},{"type":"function","title":"Date.diff/2","doc":"Calculates the difference between two dates, in a full number of days.\n\nIt returns the number of Gregorian days between the dates. Only `Date`\nstructs that follow the same or compatible calendars can be compared\nthis way. If two calendars are not compatible, it will raise.","ref":"Date.html#diff/2"},{"type":"function","title":"Examples - Date.diff/2","doc":"iex> Date.diff(~D[2000-01-03], ~D[2000-01-01])\n    2\n    iex> Date.diff(~D[2000-01-01], ~D[2000-01-03])\n    -2\n    iex> Date.diff(~D[0000-01-02], ~D[-0001-12-30])\n    3\n    iex> Date.diff(~D[2000-01-01], ~N[2000-01-03 09:00:00])\n    -2","ref":"Date.html#diff/2-examples"},{"type":"function","title":"Date.end_of_month/1","doc":"Calculates a date that is the last day of the month for the given `date`.","ref":"Date.html#end_of_month/1"},{"type":"function","title":"Examples - Date.end_of_month/1","doc":"iex> Date.end_of_month(~D[2000-01-01])\n    ~D[2000-01-31]\n    iex> Date.end_of_month(~D[2000-01-31])\n    ~D[2000-01-31]\n    iex> Date.end_of_month(~N[2000-01-01 01:23:45])\n    ~D[2000-01-31]","ref":"Date.html#end_of_month/1-examples"},{"type":"function","title":"Date.end_of_week/2","doc":"Calculates a date that is the last day of the week for the given `date`.\n\nIf the day is already the last day of the week, it returns the\nday itself. For the built-in ISO calendar, the week ends on Sunday.\nA weekday rather than `:default` can be given as `starting_on`.","ref":"Date.html#end_of_week/2"},{"type":"function","title":"Examples - Date.end_of_week/2","doc":"iex> Date.end_of_week(~D[2020-07-11])\n    ~D[2020-07-12]\n    iex> Date.end_of_week(~D[2020-07-05])\n    ~D[2020-07-05]\n    iex> Date.end_of_week(~D[2020-07-06], :sunday)\n    ~D[2020-07-11]\n    iex> Date.end_of_week(~D[2020-07-06], :saturday)\n    ~D[2020-07-10]\n    iex> Date.end_of_week(~N[2020-07-11 01:23:45])\n    ~D[2020-07-12]","ref":"Date.html#end_of_week/2-examples"},{"type":"function","title":"Date.from_erl/2","doc":"Converts an Erlang date tuple to a `Date` struct.\n\nOnly supports converting dates which are in the ISO calendar,\nor other calendars in which the days also start at midnight.\nAttempting to convert dates from other calendars will return an error tuple.","ref":"Date.html#from_erl/2"},{"type":"function","title":"Examples - Date.from_erl/2","doc":"iex> Date.from_erl({2000, 1, 1})\n    {:ok, ~D[2000-01-01]}\n    iex> Date.from_erl({2000, 13, 1})\n    {:error, :invalid_date}","ref":"Date.html#from_erl/2-examples"},{"type":"function","title":"Date.from_erl!/2","doc":"Converts an Erlang date tuple but raises for invalid dates.","ref":"Date.html#from_erl!/2"},{"type":"function","title":"Examples - Date.from_erl!/2","doc":"iex> Date.from_erl!({2000, 1, 1})\n    ~D[2000-01-01]\n    iex> Date.from_erl!({2000, 13, 1})\n    ** (ArgumentError) cannot convert {2000, 13, 1} to date, reason: :invalid_date","ref":"Date.html#from_erl!/2-examples"},{"type":"function","title":"Date.from_gregorian_days/2","doc":"Converts a number of gregorian days to a `Date` struct.","ref":"Date.html#from_gregorian_days/2"},{"type":"function","title":"Examples - Date.from_gregorian_days/2","doc":"iex> Date.from_gregorian_days(1)\n    ~D[0000-01-02]\n    iex> Date.from_gregorian_days(730_485)\n    ~D[2000-01-01]\n    iex> Date.from_gregorian_days(-1)\n    ~D[-0001-12-31]","ref":"Date.html#from_gregorian_days/2-examples"},{"type":"function","title":"Date.from_iso8601/2","doc":"Parses the extended \"Dates\" format described by\n[ISO 8601:2019](https://en.wikipedia.org/wiki/ISO_8601).\n\nThe year parsed by this function is limited to four digits.","ref":"Date.html#from_iso8601/2"},{"type":"function","title":"Examples - Date.from_iso8601/2","doc":"iex> Date.from_iso8601(\"2015-01-23\")\n    {:ok, ~D[2015-01-23]}\n\n    iex> Date.from_iso8601(\"2015:01:23\")\n    {:error, :invalid_format}\n\n    iex> Date.from_iso8601(\"2015-01-32\")\n    {:error, :invalid_date}","ref":"Date.html#from_iso8601/2-examples"},{"type":"function","title":"Date.from_iso8601!/2","doc":"Parses the extended \"Dates\" format described by\n[ISO 8601:2019](https://en.wikipedia.org/wiki/ISO_8601).\n\nRaises if the format is invalid.","ref":"Date.html#from_iso8601!/2"},{"type":"function","title":"Examples - Date.from_iso8601!/2","doc":"iex> Date.from_iso8601!(\"2015-01-23\")\n    ~D[2015-01-23]\n    iex> Date.from_iso8601!(\"2015:01:23\")\n    ** (ArgumentError) cannot parse \"2015:01:23\" as date, reason: :invalid_format","ref":"Date.html#from_iso8601!/2-examples"},{"type":"function","title":"Date.leap_year?/1","doc":"Returns `true` if the year in the given `date` is a leap year.","ref":"Date.html#leap_year?/1"},{"type":"function","title":"Examples - Date.leap_year?/1","doc":"iex> Date.leap_year?(~D[2000-01-01])\n    true\n    iex> Date.leap_year?(~D[2001-01-01])\n    false\n    iex> Date.leap_year?(~D[2004-01-01])\n    true\n    iex> Date.leap_year?(~D[1900-01-01])\n    false\n    iex> Date.leap_year?(~N[2004-01-01 01:23:45])\n    true","ref":"Date.html#leap_year?/1-examples"},{"type":"function","title":"Date.months_in_year/1","doc":"Returns the number of months in the given `date` year.","ref":"Date.html#months_in_year/1"},{"type":"function","title":"Example - Date.months_in_year/1","doc":"iex> Date.months_in_year(~D[1900-01-13])\n    12","ref":"Date.html#months_in_year/1-example"},{"type":"function","title":"Date.new/4","doc":"Builds a new ISO date.\n\nExpects all values to be integers. Returns `{:ok, date}` if each\nentry fits its appropriate range, returns `{:error, reason}` otherwise.","ref":"Date.html#new/4"},{"type":"function","title":"Examples - Date.new/4","doc":"iex> Date.new(2000, 1, 1)\n    {:ok, ~D[2000-01-01]}\n    iex> Date.new(2000, 13, 1)\n    {:error, :invalid_date}\n    iex> Date.new(2000, 2, 29)\n    {:ok, ~D[2000-02-29]}\n\n    iex> Date.new(2000, 2, 30)\n    {:error, :invalid_date}\n    iex> Date.new(2001, 2, 29)\n    {:error, :invalid_date}","ref":"Date.html#new/4-examples"},{"type":"function","title":"Date.new!/4","doc":"Builds a new ISO date.\n\nExpects all values to be integers. Returns `date` if each\nentry fits its appropriate range, raises if the date is invalid.","ref":"Date.html#new!/4"},{"type":"function","title":"Examples - Date.new!/4","doc":"iex> Date.new!(2000, 1, 1)\n    ~D[2000-01-01]\n    iex> Date.new!(2000, 13, 1)\n    ** (ArgumentError) cannot build date, reason: :invalid_date\n    iex> Date.new!(2000, 2, 29)\n    ~D[2000-02-29]","ref":"Date.html#new!/4-examples"},{"type":"function","title":"Date.quarter_of_year/1","doc":"Calculates the quarter of the year of a given `date`.\n\nReturns the day of the year as an integer. For the ISO 8601\ncalendar (the default), it is an integer from 1 to 4.","ref":"Date.html#quarter_of_year/1"},{"type":"function","title":"Examples - Date.quarter_of_year/1","doc":"iex> Date.quarter_of_year(~D[2016-10-31])\n    4\n    iex> Date.quarter_of_year(~D[2016-01-01])\n    1\n    iex> Date.quarter_of_year(~N[2016-04-01 01:23:45])\n    2\n    iex> Date.quarter_of_year(~D[-0015-09-30])\n    3","ref":"Date.html#quarter_of_year/1-examples"},{"type":"function","title":"Date.range/2","doc":"Returns a range of dates.\n\nA range of dates represents a discrete number of dates where\nthe first and last values are dates with matching calendars.\n\nRanges of dates can be either increasing (`first   last`). They are also always inclusive.","ref":"Date.html#range/2"},{"type":"function","title":"Examples - Date.range/2","doc":"iex> Date.range(~D[1999-01-01], ~D[2000-01-01])\n    Date.range(~D[1999-01-01], ~D[2000-01-01])\n\nA range of dates implements the `Enumerable` protocol, which means\nfunctions in the `Enum` module can be used to work with\nranges:\n\n    iex> range = Date.range(~D[2001-01-01], ~D[2002-01-01])\n    iex> range\n    Date.range(~D[2001-01-01], ~D[2002-01-01])\n    iex> Enum.count(range)\n    366\n    iex> ~D[2001-02-01] in range\n    true\n    iex> Enum.take(range, 3)\n    [~D[2001-01-01], ~D[2001-01-02], ~D[2001-01-03]]\n    iex> for d <- Date.range(~D[2023-03-01], ~D[2023-04-01]), Date.day_of_week(d) == 7, do: d\n    [~D[2023-03-05], ~D[2023-03-12], ~D[2023-03-19], ~D[2023-03-26]]","ref":"Date.html#range/2-examples"},{"type":"function","title":"Date.range/3","doc":"Returns a range of dates with a step.","ref":"Date.html#range/3"},{"type":"function","title":"Examples - Date.range/3","doc":"iex> range = Date.range(~D[2001-01-01], ~D[2002-01-01], 2)\n    iex> range\n    Date.range(~D[2001-01-01], ~D[2002-01-01], 2)\n    iex> Enum.count(range)\n    183\n    iex> ~D[2001-01-03] in range\n    true\n    iex> Enum.take(range, 3)\n    [~D[2001-01-01], ~D[2001-01-03], ~D[2001-01-05]]","ref":"Date.html#range/3-examples"},{"type":"function","title":"Date.to_erl/1","doc":"Converts the given `date` to an Erlang date tuple.\n\nOnly supports converting dates which are in the ISO calendar,\nor other calendars in which the days also start at midnight.\nAttempting to convert dates from other calendars will raise.","ref":"Date.html#to_erl/1"},{"type":"function","title":"Examples - Date.to_erl/1","doc":"iex> Date.to_erl(~D[2000-01-01])\n    {2000, 1, 1}\n\n    iex> Date.to_erl(~N[2000-01-01 00:00:00])\n    {2000, 1, 1}","ref":"Date.html#to_erl/1-examples"},{"type":"function","title":"Date.to_gregorian_days/1","doc":"Converts a `date` struct to a number of gregorian days.","ref":"Date.html#to_gregorian_days/1"},{"type":"function","title":"Examples - Date.to_gregorian_days/1","doc":"iex> Date.to_gregorian_days(~D[0000-01-02])\n    1\n    iex> Date.to_gregorian_days(~D[2000-01-01])\n    730_485\n    iex> Date.to_gregorian_days(~N[2000-01-01 00:00:00])\n    730_485","ref":"Date.html#to_gregorian_days/1-examples"},{"type":"function","title":"Date.to_iso8601/2","doc":"Converts the given `date` to\n[ISO 8601:2019](https://en.wikipedia.org/wiki/ISO_8601).\n\nBy default, `Date.to_iso8601/2` returns dates formatted in the \"extended\"\nformat, for human readability. It also supports the \"basic\" format through passing the `:basic` option.\n\nOnly supports converting dates which are in the ISO calendar,\nor other calendars in which the days also start at midnight.\nAttempting to convert dates from other calendars will raise an `ArgumentError`.\n\n#","ref":"Date.html#to_iso8601/2"},{"type":"function","title":"Examples - Date.to_iso8601/2","doc":"iex> Date.to_iso8601(~D[2000-02-28])\n    \"2000-02-28\"\n\n    iex> Date.to_iso8601(~D[2000-02-28], :basic)\n    \"20000228\"\n\n    iex> Date.to_iso8601(~N[2000-02-28 00:00:00])\n    \"2000-02-28\"","ref":"Date.html#to_iso8601/2-examples"},{"type":"function","title":"Date.to_string/1","doc":"Converts the given date to a string according to its calendar.\n\n#","ref":"Date.html#to_string/1"},{"type":"function","title":"Examples - Date.to_string/1","doc":"iex> Date.to_string(~D[2000-02-28])\n    \"2000-02-28\"\n    iex> Date.to_string(~N[2000-02-28 01:23:45])\n    \"2000-02-28\"\n    iex> Date.to_string(~D[-0100-12-15])\n    \"-0100-12-15\"","ref":"Date.html#to_string/1-examples"},{"type":"function","title":"Date.utc_today/1","doc":"Returns the current date in UTC.","ref":"Date.html#utc_today/1"},{"type":"function","title":"Examples - Date.utc_today/1","doc":"iex> date = Date.utc_today()\n    iex> date.year >= 2016\n    true","ref":"Date.html#utc_today/1-examples"},{"type":"function","title":"Date.year_of_era/1","doc":"Calculates the year-of-era and era for a given\ncalendar year.\n\nReturns a tuple `{year, era}` representing the\nyear within the era and the era number.","ref":"Date.html#year_of_era/1"},{"type":"function","title":"Examples - Date.year_of_era/1","doc":"iex> Date.year_of_era(~D[0001-01-01])\n    {1, 1}\n    iex> Date.year_of_era(~D[0000-12-31])\n    {1, 0}\n    iex> Date.year_of_era(~D[-0001-01-01])\n    {2, 0}","ref":"Date.html#year_of_era/1-examples"},{"type":"type","title":"Date.t/0","doc":"","ref":"Date.html#t:t/0"},{"type":"module","title":"DateTime","doc":"A datetime implementation with a time zone.\n\nThis datetime can be seen as a snapshot of a date and time\nat a given time zone. For such purposes, it also includes both\nUTC and Standard offsets, as well as the zone abbreviation\nfield used exclusively for formatting purposes. Note future\ndatetimes are not necessarily guaranteed to exist, as time\nzones may change any time in the future due to geopolitical\nreasons. See the \"Datetimes as snapshots\" section for more\ninformation.\n\nRemember, comparisons in Elixir using `==/2`, `>/2`, `  Enum.min([~U[2022-01-12 00:01:00.00Z], ~U[2021-01-12 00:01:00.00Z]], DateTime)\n    ~U[2021-01-12 00:01:00.00Z]\n\nDevelopers should avoid creating the `DateTime` struct directly\nand instead rely on the functions provided by this module as\nwell as the ones in third-party calendar libraries.","ref":"DateTime.html"},{"type":"module","title":"Time zone database - DateTime","doc":"Many functions in this module require a time zone database.\nBy default, it uses the default time zone database returned by\n`Calendar.get_time_zone_database/0`, which defaults to\n`Calendar.UTCOnlyTimeZoneDatabase` which only handles \"Etc/UTC\"\ndatetimes and returns `{:error, :utc_only_time_zone_database}`\nfor any other time zone.\n\nOther time zone databases can also be configured. Here are some\navailable options and libraries:\n\n  * [`time_zone_info`](https://github.com/hrzndhrn/time_zone_info)\n  * [`tz`](https://github.com/mathieuprog/tz)\n  * [`tzdata`](https://github.com/lau/tzdata)\n  * [`zoneinfo`](https://github.com/smartrent/zoneinfo) -\n    recommended for embedded devices\n\nTo use them, first make sure it is added as a dependency in `mix.exs`.\nIt can then be configured either via configuration:\n\n    config :elixir, :time_zone_database, Tz.TimeZoneDatabase\n\nor by calling `Calendar.put_time_zone_database/1`:\n\n    Calendar.put_time_zone_database(Tz.TimeZoneDatabase)\n\nSee the proper names in the library installation instructions.","ref":"DateTime.html#module-time-zone-database"},{"type":"module","title":"Datetimes as snapshots - DateTime","doc":"In the first section, we described datetimes as a \"snapshot of\na date and time at a given time zone\". To understand precisely\nwhat we mean, let's see an example.\n\nImagine someone in Poland wants to schedule a meeting with someone\nin Brazil in the next year. The meeting will happen at 2:30 AM\nin the Polish time zone. At what time will the meeting happen in\nBrazil?\n\nYou can consult the time zone database today, one year before,\nusing the API in this module and it will give you an answer that\nis valid right now. However, this answer may not be valid in the\nfuture. Why? Because both Brazil and Poland may change their timezone\nrules, ultimately affecting the result. For example, a country may\nchoose to enter or abandon \"Daylight Saving Time\", which is a\nprocess where we adjust the clock one hour forward or one hour\nback once per year. Whenener the rules change, the exact instant\nthat 2:30 AM in Polish time will be in Brazil may change.\n\nIn other words, whenever working with future DateTimes, there is\nno guarantee the results you get will always be correct, until\nthe event actually happens. Therefore, when you ask for a future\ntime, the answers you get are a snapshot that reflects the current\nstate of the time zone rules. For datetimes in the past, this is\nnot a problem, because time zone rules do not change for past\nevents.\n\nTo make matters worse, it may be that the 2:30 AM in Polish time\ndoes not actually even exist or it is ambiguous. If a certain\ntime zone observes \"Daylight Saving Time\", they will move their\nclock forward once a year. When this happens, there is a whole\nhour that does not exist. Then, when they move the clock back,\nthere is a certain hour that will happen twice. So if you want\nto schedule a meeting when this shift back happens, you would\nneed to explicitly say which of the 2:30 AM you precisely mean.\nApplications that are date and time sensitive, need to take\nthese scenarios into account and correctly communicate them to\nusers.\n\nThe good news is: Elixir contains all of the building blocks\nnecessary to tackle those problems. The default timezone database\nused by Elixir, `Calendar.UTCOnlyTimeZoneDatabase`, only works\nwith UTC, which does not observe those issues. Once you bring\na proper time zone database, the functions in this module will\nquery the database and return the relevant information. For\nexample, look at how `DateTime.new/4` returns different results\nbased on the scenarios described in this section.","ref":"DateTime.html#module-datetimes-as-snapshots"},{"type":"function","title":"DateTime.add/4","doc":"Adds a specified amount of time to a `DateTime`.\n\nAccepts an `amount_to_add` in any `unit`. `unit` can be `:day`,\n`:hour`, `:minute`, `:second` or any subsecond precision from\n`t:System.time_unit/0`. It defaults to `:second`. Negative values\nwill move backwards in time.\n\nThis function always consider the unit to be computed according\nto the `Calendar.ISO`.\n\nThis function uses relies on a contiguous representation of time,\nignoring the wall time and timezone changes. For example, if you add\none day when there are summer time/daylight saving time changes,\nit will also change the time forward or backward by one hour,\nso the elapsed time is precisely 24 hours. Similarly, adding just\na few seconds to a datetime just before \"spring forward\" can cause\nwall time to increase by more than an hour.\n\nWhile this means this function is precise in terms of elapsed time,\nits result may be misleading in certain use cases. For example, if a\nuser requests a meeting to happen every day at 15:00 and you use this\nfunction to compute all future meetings by adding day after day, this\nfunction may change the meeting time to 14:00 or 16:00 if there are\nchanges to the current timezone. Computing of recurring datetimes is\nnot currently supported in Elixir's standard library but it is available\nby third-party libraries.\n\n#","ref":"DateTime.html#add/4"},{"type":"function","title":"Examples - DateTime.add/4","doc":"iex> dt = DateTime.from_naive!(~N[2018-11-15 10:00:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    iex> dt |> DateTime.add(3600, :second, FakeTimeZoneDatabase)\n    #DateTime \n\n    iex> DateTime.add(~U[2018-11-15 10:00:00Z], 3600, :second)\n    ~U[2018-11-15 11:00:00Z]\n\nWhen adding 3 seconds just before \"spring forward\" we go from 1:59:59 to 3:00:02:\n\n    iex> dt = DateTime.from_naive!(~N[2019-03-31 01:59:59.123], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    iex> dt |> DateTime.add(3, :second, FakeTimeZoneDatabase)\n    #DateTime \n\nWhen adding 1 day during \"spring forward\", the hour also changes:\n\n    iex> dt = DateTime.from_naive!(~N[2019-03-31 01:00:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    iex> dt |> DateTime.add(1, :day, FakeTimeZoneDatabase)\n    #DateTime \n\nThis operation merges the precision of the naive date time with the given unit:\n\n    iex> result = DateTime.add(~U[2014-10-02 00:29:10Z], 21, :millisecond)\n    ~U[2014-10-02 00:29:10.021Z]\n    iex> result.microsecond\n    {21000, 3}","ref":"DateTime.html#add/4-examples"},{"type":"function","title":"DateTime.after?/2","doc":"Returns true if the first datetime is strictly later than the second.","ref":"DateTime.html#after?/2"},{"type":"function","title":"Examples - DateTime.after?/2","doc":"iex> DateTime.after?(~U[2022-02-02 11:00:00Z], ~U[2021-01-01 11:00:00Z])\n    true\n    iex> DateTime.after?(~U[2021-01-01 11:00:00Z], ~U[2021-01-01 11:00:00Z])\n    false\n    iex> DateTime.after?(~U[2021-01-01 11:00:00Z], ~U[2022-02-02 11:00:00Z])\n    false","ref":"DateTime.html#after?/2-examples"},{"type":"function","title":"DateTime.before?/2","doc":"Returns true if the first datetime is strictly earlier than the second.","ref":"DateTime.html#before?/2"},{"type":"function","title":"Examples - DateTime.before?/2","doc":"iex> DateTime.before?(~U[2021-01-01 11:00:00Z], ~U[2022-02-02 11:00:00Z])\n    true\n    iex> DateTime.before?(~U[2021-01-01 11:00:00Z], ~U[2021-01-01 11:00:00Z])\n    false\n    iex> DateTime.before?(~U[2022-02-02 11:00:00Z], ~U[2021-01-01 11:00:00Z])\n    false","ref":"DateTime.html#before?/2-examples"},{"type":"function","title":"DateTime.compare/2","doc":"Compares two datetime structs.\n\nReturns `:gt` if the first datetime is later than the second\nand `:lt` for vice versa. If the two datetimes are equal\n`:eq` is returned.\n\nNote that both UTC and Standard offsets will be taken into\naccount when comparison is done.","ref":"DateTime.html#compare/2"},{"type":"function","title":"Examples - DateTime.compare/2","doc":"iex> dt1 = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"AMT\",\n    ...>                 hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                 utc_offset: -14400, std_offset: 0, time_zone: \"America/Manaus\"}\n    iex> dt2 = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                 hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                 utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> DateTime.compare(dt1, dt2)\n    :gt","ref":"DateTime.html#compare/2-examples"},{"type":"function","title":"DateTime.convert/2","doc":"Converts a given `datetime` from one calendar to another.\n\nIf it is not possible to convert unambiguously between the calendars\n(see `Calendar.compatible_calendars?/2`), an `{:error, :incompatible_calendars}` tuple\nis returned.","ref":"DateTime.html#convert/2"},{"type":"function","title":"Examples - DateTime.convert/2","doc":"Imagine someone implements `Calendar.Holocene`, a calendar based on the\nGregorian calendar that adds exactly 10,000 years to the current Gregorian\nyear:\n\n    iex> dt1 = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"AMT\",\n    ...>                 hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                 utc_offset: -14400, std_offset: 0, time_zone: \"America/Manaus\"}\n    iex> DateTime.convert(dt1, Calendar.Holocene)\n    {:ok, %DateTime{calendar: Calendar.Holocene, day: 29, hour: 23,\n                    microsecond: {0, 0}, minute: 0, month: 2, second: 7, std_offset: 0,\n                    time_zone: \"America/Manaus\", utc_offset: -14400, year: 12000,\n                    zone_abbr: \"AMT\"}}","ref":"DateTime.html#convert/2-examples"},{"type":"function","title":"DateTime.convert!/2","doc":"Converts a given `datetime` from one calendar to another.\n\nIf it is not possible to convert unambiguously between the calendars\n(see `Calendar.compatible_calendars?/2`), an ArgumentError is raised.","ref":"DateTime.html#convert!/2"},{"type":"function","title":"Examples - DateTime.convert!/2","doc":"Imagine someone implements `Calendar.Holocene`, a calendar based on the\nGregorian calendar that adds exactly 10,000 years to the current Gregorian\nyear:\n\n    iex> dt1 = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"AMT\",\n    ...>                 hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                 utc_offset: -14400, std_offset: 0, time_zone: \"America/Manaus\"}\n    iex> DateTime.convert!(dt1, Calendar.Holocene)\n    %DateTime{calendar: Calendar.Holocene, day: 29, hour: 23,\n              microsecond: {0, 0}, minute: 0, month: 2, second: 7, std_offset: 0,\n              time_zone: \"America/Manaus\", utc_offset: -14400, year: 12000,\n              zone_abbr: \"AMT\"}","ref":"DateTime.html#convert!/2-examples"},{"type":"function","title":"DateTime.diff/3","doc":"Subtracts `datetime2` from `datetime1`.\n\nThe answer can be returned in any `:day`, `:hour`, `:minute`, or any `unit`\navailable from `t:System.time_unit/0`. The unit is measured according to\n`Calendar.ISO` and defaults to `:second`.\n\nFractional results are not supported and are truncated.","ref":"DateTime.html#diff/3"},{"type":"function","title":"Examples - DateTime.diff/3","doc":"iex> dt1 = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"AMT\",\n    ...>                 hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                 utc_offset: -14400, std_offset: 0, time_zone: \"America/Manaus\"}\n    iex> dt2 = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                 hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                 utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> DateTime.diff(dt1, dt2)\n    18000\n    iex> DateTime.diff(dt2, dt1)\n    -18000\n    iex> DateTime.diff(dt1, dt2, :hour)\n    5\n    iex> DateTime.diff(dt2, dt1, :hour)\n    -5","ref":"DateTime.html#diff/3-examples"},{"type":"function","title":"DateTime.from_gregorian_seconds/3","doc":"Converts a number of gregorian seconds to a `DateTime` struct.\n\nThe returned `DateTime` will have `UTC` timezone, if you want other timezone, please use\n`DateTime.shift_zone/3`.","ref":"DateTime.html#from_gregorian_seconds/3"},{"type":"function","title":"Examples - DateTime.from_gregorian_seconds/3","doc":"iex> DateTime.from_gregorian_seconds(1)\n    ~U[0000-01-01 00:00:01Z]\n    iex> DateTime.from_gregorian_seconds(63_755_511_991, {5000, 3})\n    ~U[2020-05-01 00:26:31.005Z]\n    iex> DateTime.from_gregorian_seconds(-1)\n    ~U[-0001-12-31 23:59:59Z]","ref":"DateTime.html#from_gregorian_seconds/3-examples"},{"type":"function","title":"DateTime.from_iso8601/2","doc":"Parses the extended \"Date and time of day\" format described by\n[ISO 8601:2019](https://en.wikipedia.org/wiki/ISO_8601).\n\nSince ISO 8601 does not include the proper time zone, the given\nstring will be converted to UTC and its offset in seconds will be\nreturned as part of this function. Therefore offset information\nmust be present in the string.\n\nAs specified in the standard, the separator \"T\" may be omitted if\ndesired as there is no ambiguity within this function.\n\nNote leap seconds are not supported by the built-in Calendar.ISO.","ref":"DateTime.html#from_iso8601/2"},{"type":"function","title":"Examples - DateTime.from_iso8601/2","doc":"iex> {:ok, datetime, 0} = DateTime.from_iso8601(\"2015-01-23T23:50:07Z\")\n    iex> datetime\n    ~U[2015-01-23 23:50:07Z]\n\n    iex> {:ok, datetime, 9000} = DateTime.from_iso8601(\"2015-01-23T23:50:07.123+02:30\")\n    iex> datetime\n    ~U[2015-01-23 21:20:07.123Z]\n\n    iex> {:ok, datetime, 9000} = DateTime.from_iso8601(\"2015-01-23T23:50:07,123+02:30\")\n    iex> datetime\n    ~U[2015-01-23 21:20:07.123Z]\n\n    iex> {:ok, datetime, 0} = DateTime.from_iso8601(\"-2015-01-23T23:50:07Z\")\n    iex> datetime\n    ~U[-2015-01-23 23:50:07Z]\n\n    iex> {:ok, datetime, 9000} = DateTime.from_iso8601(\"-2015-01-23T23:50:07,123+02:30\")\n    iex> datetime\n    ~U[-2015-01-23 21:20:07.123Z]\n\n    iex> {:ok, datetime, 9000} = DateTime.from_iso8601(\"20150123T235007.123+0230\", :basic)\n    iex> datetime\n    ~U[2015-01-23 21:20:07.123Z]\n\n    iex> DateTime.from_iso8601(\"2015-01-23P23:50:07\")\n    {:error, :invalid_format}\n    iex> DateTime.from_iso8601(\"2015-01-23T23:50:07\")\n    {:error, :missing_offset}\n    iex> DateTime.from_iso8601(\"2015-01-23 23:50:61\")\n    {:error, :invalid_time}\n    iex> DateTime.from_iso8601(\"2015-01-32 23:50:07\")\n    {:error, :invalid_date}\n    iex> DateTime.from_iso8601(\"2015-01-23T23:50:07.123-00:00\")\n    {:error, :invalid_format}","ref":"DateTime.html#from_iso8601/2-examples"},{"type":"function","title":"DateTime.from_iso8601/3","doc":"Converts to ISO8601 specifying both a calendar and a mode.\n\nSee `from_iso8601/2` for more information.","ref":"DateTime.html#from_iso8601/3"},{"type":"function","title":"Examples - DateTime.from_iso8601/3","doc":"iex> {:ok, datetime, 9000} = DateTime.from_iso8601(\"2015-01-23T23:50:07,123+02:30\", Calendar.ISO, :extended)\n    iex> datetime\n    ~U[2015-01-23 21:20:07.123Z]\n\n    iex> {:ok, datetime, 9000} = DateTime.from_iso8601(\"20150123T235007.123+0230\", Calendar.ISO, :basic)\n    iex> datetime\n    ~U[2015-01-23 21:20:07.123Z]","ref":"DateTime.html#from_iso8601/3-examples"},{"type":"function","title":"DateTime.from_naive/3","doc":"Converts the given `NaiveDateTime` to `DateTime`.\n\nIt expects a time zone to put the `NaiveDateTime` in.\nIf the time zone is \"Etc/UTC\", it always succeeds. Otherwise,\nthe NaiveDateTime is checked against the time zone database\ngiven as `time_zone_database`. See the \"Time zone database\"\nsection in the module documentation.","ref":"DateTime.html#from_naive/3"},{"type":"function","title":"Examples - DateTime.from_naive/3","doc":"iex> DateTime.from_naive(~N[2016-05-24 13:26:08.003], \"Etc/UTC\")\n    {:ok, ~U[2016-05-24 13:26:08.003Z]}\n\nWhen the datetime is ambiguous - for instance during changing from summer\nto winter time - the two possible valid datetimes are returned in a tuple.\nThe first datetime is also the one which comes first chronologically, while\nthe second one comes last.\n\n    iex> {:ambiguous, first_dt, second_dt} = DateTime.from_naive(~N[2018-10-28 02:30:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    iex> first_dt\n    #DateTime \n    iex> second_dt\n    #DateTime \n\nWhen there is a gap in wall time - for instance in spring when the clocks are\nturned forward - the latest valid datetime just before the gap and the first\nvalid datetime just after the gap.\n\n    iex> {:gap, just_before, just_after} = DateTime.from_naive(~N[2019-03-31 02:30:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    iex> just_before\n    #DateTime \n    iex> just_after\n    #DateTime \n\nMost of the time there is one, and just one, valid datetime for a certain\ndate and time in a certain time zone.\n\n    iex> {:ok, datetime} = DateTime.from_naive(~N[2018-07-28 12:30:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    iex> datetime\n    #DateTime \n\nThis function accepts any map or struct that contains at least the same fields as a `NaiveDateTime`\nstruct. The most common example of that is a `DateTime`. In this case the information about the time\nzone of that `DateTime` is completely ignored. This is the same principle as passing a `DateTime` to\n`Date.to_iso8601/2`. `Date.to_iso8601/2` extracts only the date-specific fields (calendar, year,\nmonth and day) of the given structure and ignores all others.\n\nThis way if you have a `DateTime` in one time zone, you can get the same wall time in another time zone.\nFor instance if you have 2018-08-24 10:00:00 in Copenhagen and want a `DateTime` for 2018-08-24 10:00:00\nin UTC you can do:\n\n    iex> cph_datetime = DateTime.from_naive!(~N[2018-08-24 10:00:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    iex> {:ok, utc_datetime} = DateTime.from_naive(cph_datetime, \"Etc/UTC\", FakeTimeZoneDatabase)\n    iex> utc_datetime\n    ~U[2018-08-24 10:00:00Z]\n\nIf instead you want a `DateTime` for the same point time in a different time zone see the\n`DateTime.shift_zone/3` function which would convert 2018-08-24 10:00:00 in Copenhagen\nto 2018-08-24 08:00:00 in UTC.","ref":"DateTime.html#from_naive/3-examples"},{"type":"function","title":"DateTime.from_naive!/3","doc":"Converts the given `NaiveDateTime` to `DateTime`.\n\nIt expects a time zone to put the NaiveDateTime in.\nIf the time zone is \"Etc/UTC\", it always succeeds. Otherwise,\nthe NaiveDateTime is checked against the time zone database\ngiven as `time_zone_database`. See the \"Time zone database\"\nsection in the module documentation.","ref":"DateTime.html#from_naive!/3"},{"type":"function","title":"Examples - DateTime.from_naive!/3","doc":"iex> DateTime.from_naive!(~N[2016-05-24 13:26:08.003], \"Etc/UTC\")\n    ~U[2016-05-24 13:26:08.003Z]\n\n    iex> DateTime.from_naive!(~N[2018-05-24 13:26:08.003], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    #DateTime","ref":"DateTime.html#from_naive!/3-examples"},{"type":"function","title":"DateTime.from_unix/3","doc":"Converts the given Unix time to `DateTime`.\n\nThe integer can be given in different unit\naccording to `System.convert_time_unit/3` and it will\nbe converted to microseconds internally. Up to\n253402300799 seconds is supported.\n\nUnix times are always in UTC and therefore the DateTime\nwill be returned in UTC.","ref":"DateTime.html#from_unix/3"},{"type":"function","title":"Examples - DateTime.from_unix/3","doc":"iex> {:ok, datetime} = DateTime.from_unix(1_464_096_368)\n    iex> datetime\n    ~U[2016-05-24 13:26:08Z]\n\n    iex> {:ok, datetime} = DateTime.from_unix(1_432_560_368_868_569, :microsecond)\n    iex> datetime\n    ~U[2015-05-25 13:26:08.868569Z]\n\n    iex> {:ok, datetime} = DateTime.from_unix(253_402_300_799)\n    iex> datetime\n    ~U[9999-12-31 23:59:59Z]\n\n    iex> {:error, :invalid_unix_time} = DateTime.from_unix(253_402_300_800)\n\nThe unit can also be an integer as in `t:System.time_unit/0`:\n\n    iex> {:ok, datetime} = DateTime.from_unix(143_256_036_886_856, 1024)\n    iex> datetime\n    ~U[6403-03-17 07:05:22.320312Z]\n\nNegative Unix times are supported up to -377705116800 seconds:\n\n    iex> {:ok, datetime} = DateTime.from_unix(-377_705_116_800)\n    iex> datetime\n    ~U[-9999-01-01 00:00:00Z]\n\n    iex> {:error, :invalid_unix_time} = DateTime.from_unix(-377_705_116_801)","ref":"DateTime.html#from_unix/3-examples"},{"type":"function","title":"DateTime.from_unix!/3","doc":"Converts the given Unix time to `DateTime`.\n\nThe integer can be given in different unit\naccording to `System.convert_time_unit/3` and it will\nbe converted to microseconds internally.\n\nUnix times are always in UTC and therefore the DateTime\nwill be returned in UTC.","ref":"DateTime.html#from_unix!/3"},{"type":"function","title":"Examples - DateTime.from_unix!/3","doc":"# An easy way to get the Unix epoch is passing 0 to this function\n    iex> DateTime.from_unix!(0)\n    ~U[1970-01-01 00:00:00Z]\n\n    iex> DateTime.from_unix!(1_464_096_368)\n    ~U[2016-05-24 13:26:08Z]\n\n    iex> DateTime.from_unix!(1_432_560_368_868_569, :microsecond)\n    ~U[2015-05-25 13:26:08.868569Z]\n\n    iex> DateTime.from_unix!(143_256_036_886_856, 1024)\n    ~U[6403-03-17 07:05:22.320312Z]","ref":"DateTime.html#from_unix!/3-examples"},{"type":"function","title":"DateTime.new/4","doc":"Builds a datetime from date and time structs.\n\nIt expects a time zone to put the `DateTime` in.\nIf the time zone is not passed it will default to `\"Etc/UTC\"`,\nwhich always succeeds. Otherwise, the `DateTime` is checked against the time zone database\ngiven as `time_zone_database`. See the \"Time zone database\"\nsection in the module documentation.","ref":"DateTime.html#new/4"},{"type":"function","title":"Examples - DateTime.new/4","doc":"iex> DateTime.new(~D[2016-05-24], ~T[13:26:08.003], \"Etc/UTC\")\n    {:ok, ~U[2016-05-24 13:26:08.003Z]}\n\nWhen the datetime is ambiguous - for instance during changing from summer\nto winter time - the two possible valid datetimes are returned in a tuple.\nThe first datetime is also the one which comes first chronologically, while\nthe second one comes last.\n\n    iex> {:ambiguous, first_dt, second_dt} = DateTime.new(~D[2018-10-28], ~T[02:30:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    iex> first_dt\n    #DateTime \n    iex> second_dt\n    #DateTime \n\nWhen there is a gap in wall time - for instance in spring when the clocks are\nturned forward - the latest valid datetime just before the gap and the first\nvalid datetime just after the gap.\n\n    iex> {:gap, just_before, just_after} = DateTime.new(~D[2019-03-31], ~T[02:30:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    iex> just_before\n    #DateTime \n    iex> just_after\n    #DateTime \n\nMost of the time there is one, and just one, valid datetime for a certain\ndate and time in a certain time zone.\n\n    iex> {:ok, datetime} = DateTime.new(~D[2018-07-28], ~T[12:30:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    iex> datetime\n    #DateTime","ref":"DateTime.html#new/4-examples"},{"type":"function","title":"DateTime.new!/4","doc":"Builds a datetime from date and time structs, raising on errors.\n\nIt expects a time zone to put the `DateTime` in.\nIf the time zone is not passed it will default to `\"Etc/UTC\"`,\nwhich always succeeds. Otherwise, the DateTime is checked against the time zone database\ngiven as `time_zone_database`. See the \"Time zone database\"\nsection in the module documentation.","ref":"DateTime.html#new!/4"},{"type":"function","title":"Examples - DateTime.new!/4","doc":"iex> DateTime.new!(~D[2016-05-24], ~T[13:26:08.003], \"Etc/UTC\")\n    ~U[2016-05-24 13:26:08.003Z]\n\nWhen the datetime is ambiguous - for instance during changing from summer\nto winter time - an error will be raised.\n\n    iex> DateTime.new!(~D[2018-10-28], ~T[02:30:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    ** (ArgumentError) cannot build datetime with ~D[2018-10-28] and ~T[02:30:00] because such instant is ambiguous in time zone Europe/Copenhagen as there is an overlap between #DateTime  and #DateTime \n\nWhen there is a gap in wall time - for instance in spring when the clocks are\nturned forward - an error will be raised.\n\n    iex> DateTime.new!(~D[2019-03-31], ~T[02:30:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    ** (ArgumentError) cannot build datetime with ~D[2019-03-31] and ~T[02:30:00] because such instant does not exist in time zone Europe/Copenhagen as there is a gap between #DateTime  and #DateTime \n\nMost of the time there is one, and just one, valid datetime for a certain\ndate and time in a certain time zone.\n\n    iex> datetime = DateTime.new!(~D[2018-07-28], ~T[12:30:00], \"Europe/Copenhagen\", FakeTimeZoneDatabase)\n    iex> datetime\n    #DateTime","ref":"DateTime.html#new!/4-examples"},{"type":"function","title":"DateTime.now/2","doc":"Returns the current datetime in the provided time zone.\n\nBy default, it uses the default time_zone returned by\n`Calendar.get_time_zone_database/0`, which defaults to\n`Calendar.UTCOnlyTimeZoneDatabase` which only handles \"Etc/UTC\" datetimes.\nOther time zone databases can be passed as argument or set globally.\nSee the \"Time zone database\" section in the module docs.","ref":"DateTime.html#now/2"},{"type":"function","title":"Examples - DateTime.now/2","doc":"iex> {:ok, datetime} = DateTime.now(\"Etc/UTC\")\n    iex> datetime.time_zone\n    \"Etc/UTC\"\n\n    iex> DateTime.now(\"Europe/Copenhagen\")\n    {:error, :utc_only_time_zone_database}\n\n    iex> DateTime.now(\"bad timezone\", FakeTimeZoneDatabase)\n    {:error, :time_zone_not_found}","ref":"DateTime.html#now/2-examples"},{"type":"function","title":"DateTime.now!/2","doc":"Returns the current datetime in the provided time zone or raises on errors\n\nSee `now/2` for more information.","ref":"DateTime.html#now!/2"},{"type":"function","title":"Examples - DateTime.now!/2","doc":"iex> datetime = DateTime.now!(\"Etc/UTC\")\n    iex> datetime.time_zone\n    \"Etc/UTC\"\n\n    iex> DateTime.now!(\"Europe/Copenhagen\")\n    ** (ArgumentError) cannot get current datetime in \"Europe/Copenhagen\" time zone, reason: :utc_only_time_zone_database\n\n    iex> DateTime.now!(\"bad timezone\", FakeTimeZoneDatabase)\n    ** (ArgumentError) cannot get current datetime in \"bad timezone\" time zone, reason: :time_zone_not_found","ref":"DateTime.html#now!/2-examples"},{"type":"function","title":"DateTime.shift_zone/3","doc":"Changes the time zone of a `DateTime`.\n\nReturns a `DateTime` for the same point in time, but instead at\nthe time zone provided. It assumes that `DateTime` is valid and\nexists in the given time zone and calendar.\n\nBy default, it uses the default time zone database returned by\n`Calendar.get_time_zone_database/0`, which defaults to\n`Calendar.UTCOnlyTimeZoneDatabase` which only handles \"Etc/UTC\" datetimes.\nOther time zone databases can be passed as argument or set globally.\nSee the \"Time zone database\" section in the module docs.","ref":"DateTime.html#shift_zone/3"},{"type":"function","title":"Examples - DateTime.shift_zone/3","doc":"iex> {:ok, pacific_datetime} = DateTime.shift_zone(~U[2018-07-16 10:00:00Z], \"America/Los_Angeles\", FakeTimeZoneDatabase)\n    iex> pacific_datetime\n    #DateTime \n\n    iex> DateTime.shift_zone(~U[2018-07-16 10:00:00Z], \"bad timezone\", FakeTimeZoneDatabase)\n    {:error, :time_zone_not_found}","ref":"DateTime.html#shift_zone/3-examples"},{"type":"function","title":"DateTime.shift_zone!/3","doc":"Changes the time zone of a `DateTime` or raises on errors.\n\nSee `shift_zone/3` for more information.","ref":"DateTime.html#shift_zone!/3"},{"type":"function","title":"Examples - DateTime.shift_zone!/3","doc":"iex> DateTime.shift_zone!(~U[2018-07-16 10:00:00Z], \"America/Los_Angeles\", FakeTimeZoneDatabase)\n    #DateTime \n\n    iex> DateTime.shift_zone!(~U[2018-07-16 10:00:00Z], \"bad timezone\", FakeTimeZoneDatabase)\n    ** (ArgumentError) cannot shift ~U[2018-07-16 10:00:00Z] to \"bad timezone\" time zone, reason: :time_zone_not_found","ref":"DateTime.html#shift_zone!/3-examples"},{"type":"function","title":"DateTime.to_date/1","doc":"Converts a `DateTime` into a `Date`.\n\nBecause `Date` does not hold time nor time zone information,\ndata will be lost during the conversion.","ref":"DateTime.html#to_date/1"},{"type":"function","title":"Examples - DateTime.to_date/1","doc":"iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> DateTime.to_date(dt)\n    ~D[2000-02-29]","ref":"DateTime.html#to_date/1-examples"},{"type":"function","title":"DateTime.to_gregorian_seconds/1","doc":"Converts a `DateTime` struct to a number of gregorian seconds and microseconds.","ref":"DateTime.html#to_gregorian_seconds/1"},{"type":"function","title":"Examples - DateTime.to_gregorian_seconds/1","doc":"iex> dt = %DateTime{year: 0000, month: 1, day: 1, zone_abbr: \"UTC\",\n    ...>                hour: 0, minute: 0, second: 1, microsecond: {0, 0},\n    ...>                utc_offset: 0, std_offset: 0, time_zone: \"Etc/UTC\"}\n    iex> DateTime.to_gregorian_seconds(dt)\n    {1, 0}\n\n    iex> dt = %DateTime{year: 2020, month: 5, day: 1, zone_abbr: \"UTC\",\n    ...>                hour: 0, minute: 26, second: 31, microsecond: {5000, 0},\n    ...>                utc_offset: 0, std_offset: 0, time_zone: \"Etc/UTC\"}\n    iex> DateTime.to_gregorian_seconds(dt)\n    {63_755_511_991, 5000}\n\n    iex> dt = %DateTime{year: 2020, month: 5, day: 1, zone_abbr: \"CET\",\n    ...>                hour: 1, minute: 26, second: 31, microsecond: {5000, 0},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> DateTime.to_gregorian_seconds(dt)\n    {63_755_511_991, 5000}","ref":"DateTime.html#to_gregorian_seconds/1-examples"},{"type":"function","title":"DateTime.to_iso8601/3","doc":"Converts the given datetime to\n[ISO 8601:2019](https://en.wikipedia.org/wiki/ISO_8601) format.\n\nBy default, `DateTime.to_iso8601/2` returns datetimes formatted in the \"extended\"\nformat, for human readability. It also supports the \"basic\" format through passing the `:basic` option.\n\nOnly supports converting datetimes which are in the ISO calendar,\nattempting to convert datetimes from other calendars will raise.\nYou can also optionally specify an offset for the formatted string.\n\nWARNING: the ISO 8601 datetime format does not contain the time zone nor\nits abbreviation, which means information is lost when converting to such\nformat.\n\n#","ref":"DateTime.html#to_iso8601/3"},{"type":"function","title":"Examples - DateTime.to_iso8601/3","doc":"iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> DateTime.to_iso8601(dt)\n    \"2000-02-29T23:00:07+01:00\"\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"UTC\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: 0, std_offset: 0, time_zone: \"Etc/UTC\"}\n    iex> DateTime.to_iso8601(dt)\n    \"2000-02-29T23:00:07Z\"\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"AMT\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: -14400, std_offset: 0, time_zone: \"America/Manaus\"}\n    iex> DateTime.to_iso8601(dt, :extended)\n    \"2000-02-29T23:00:07-04:00\"\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"AMT\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: -14400, std_offset: 0, time_zone: \"America/Manaus\"}\n    iex> DateTime.to_iso8601(dt, :basic)\n    \"20000229T230007-0400\"\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"AMT\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: -14400, std_offset: 0, time_zone: \"America/Manaus\"}\n    iex> DateTime.to_iso8601(dt, :extended, 3600)\n    \"2000-03-01T04:00:07+01:00\"\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"AMT\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: -14400, std_offset: 0, time_zone: \"America/Manaus\"}\n    iex> DateTime.to_iso8601(dt, :extended, 0)\n    \"2000-03-01T03:00:07+00:00\"\n\n    iex> dt = %DateTime{year: 2000, month: 3, day: 01, zone_abbr: \"UTC\",\n    ...>                hour: 03, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: 0, std_offset: 0, time_zone: \"Etc/UTC\"}\n    iex> DateTime.to_iso8601(dt, :extended, 0)\n    \"2000-03-01T03:00:07Z\"\n\n    iex> {:ok, dt, offset} = DateTime.from_iso8601(\"2000-03-01T03:00:07Z\")\n    iex> \"2000-03-01T03:00:07Z\" = DateTime.to_iso8601(dt, :extended, offset)","ref":"DateTime.html#to_iso8601/3-examples"},{"type":"function","title":"DateTime.to_naive/1","doc":"Converts the given `datetime` into a `NaiveDateTime`.\n\nBecause `NaiveDateTime` does not hold time zone information,\nany time zone related data will be lost during the conversion.","ref":"DateTime.html#to_naive/1"},{"type":"function","title":"Examples - DateTime.to_naive/1","doc":"iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 1},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> DateTime.to_naive(dt)\n    ~N[2000-02-29 23:00:07.0]","ref":"DateTime.html#to_naive/1-examples"},{"type":"function","title":"DateTime.to_string/1","doc":"Converts the given `datetime` to a string according to its calendar.\n\n#","ref":"DateTime.html#to_string/1"},{"type":"function","title":"Examples - DateTime.to_string/1","doc":"iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> DateTime.to_string(dt)\n    \"2000-02-29 23:00:07+01:00 CET Europe/Warsaw\"\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"UTC\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: 0, std_offset: 0, time_zone: \"Etc/UTC\"}\n    iex> DateTime.to_string(dt)\n    \"2000-02-29 23:00:07Z\"\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"AMT\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: -14400, std_offset: 0, time_zone: \"America/Manaus\"}\n    iex> DateTime.to_string(dt)\n    \"2000-02-29 23:00:07-04:00 AMT America/Manaus\"\n\n    iex> dt = %DateTime{year: -100, month: 12, day: 19, zone_abbr: \"CET\",\n    ...>                hour: 3, minute: 20, second: 31, microsecond: {0, 0},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Stockholm\"}\n    iex> DateTime.to_string(dt)\n    \"-0100-12-19 03:20:31+01:00 CET Europe/Stockholm\"","ref":"DateTime.html#to_string/1-examples"},{"type":"function","title":"DateTime.to_time/1","doc":"Converts a `DateTime` into `Time`.\n\nBecause `Time` does not hold date nor time zone information,\ndata will be lost during the conversion.","ref":"DateTime.html#to_time/1"},{"type":"function","title":"Examples - DateTime.to_time/1","doc":"iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 1},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> DateTime.to_time(dt)\n    ~T[23:00:07.0]","ref":"DateTime.html#to_time/1-examples"},{"type":"function","title":"DateTime.to_unix/2","doc":"Converts the given `datetime` to Unix time.\n\nThe `datetime` is expected to be using the ISO calendar\nwith a year greater than or equal to 0.\n\nIt will return the integer with the given unit,\naccording to `System.convert_time_unit/3`.","ref":"DateTime.html#to_unix/2"},{"type":"function","title":"Examples - DateTime.to_unix/2","doc":"iex> 1_464_096_368 |> DateTime.from_unix!() |> DateTime.to_unix()\n    1464096368\n\n    iex> dt = %DateTime{calendar: Calendar.ISO, day: 20, hour: 18, microsecond: {273806, 6},\n    ...>                minute: 58, month: 11, second: 19, time_zone: \"America/Montevideo\",\n    ...>                utc_offset: -10800, std_offset: 3600, year: 2014, zone_abbr: \"UYST\"}\n    iex> DateTime.to_unix(dt)\n    1416517099\n\n    iex> flamel = %DateTime{calendar: Calendar.ISO, day: 22, hour: 8, microsecond: {527771, 6},\n    ...>                minute: 2, month: 3, second: 25, std_offset: 0, time_zone: \"Etc/UTC\",\n    ...>                utc_offset: 0, year: 1418, zone_abbr: \"UTC\"}\n    iex> DateTime.to_unix(flamel)\n    -17412508655","ref":"DateTime.html#to_unix/2-examples"},{"type":"function","title":"DateTime.truncate/2","doc":"Returns the given datetime with the microsecond field truncated to the given\nprecision (`:microsecond`, `:millisecond` or `:second`).\n\nThe given datetime is returned unchanged if it already has lower precision than\nthe given precision.","ref":"DateTime.html#truncate/2"},{"type":"function","title":"Examples - DateTime.truncate/2","doc":"iex> dt1 = %DateTime{year: 2017, month: 11, day: 7, zone_abbr: \"CET\",\n    ...>                 hour: 11, minute: 45, second: 18, microsecond: {123456, 6},\n    ...>                 utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Paris\"}\n    iex> DateTime.truncate(dt1, :microsecond)\n    #DateTime \n\n    iex> dt2 = %DateTime{year: 2017, month: 11, day: 7, zone_abbr: \"CET\",\n    ...>                 hour: 11, minute: 45, second: 18, microsecond: {123456, 6},\n    ...>                 utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Paris\"}\n    iex> DateTime.truncate(dt2, :millisecond)\n    #DateTime \n\n    iex> dt3 = %DateTime{year: 2017, month: 11, day: 7, zone_abbr: \"CET\",\n    ...>                 hour: 11, minute: 45, second: 18, microsecond: {123456, 6},\n    ...>                 utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Paris\"}\n    iex> DateTime.truncate(dt3, :second)\n    #DateTime","ref":"DateTime.html#truncate/2-examples"},{"type":"function","title":"DateTime.utc_now/1","doc":"Returns the current datetime in UTC.\n\nIf you want the current time in Unix seconds,\nuse `System.os_time/1` instead.\n\nYou can also pass a time unit to automatically\ntruncate the resulting datetime. This is available\nsince v1.15.0.","ref":"DateTime.html#utc_now/1"},{"type":"function","title":"Examples - DateTime.utc_now/1","doc":"iex> datetime = DateTime.utc_now()\n    iex> datetime.time_zone\n    \"Etc/UTC\"\n\n    iex> datetime = DateTime.utc_now(:second)\n    iex> datetime.microsecond\n    {0, 0}","ref":"DateTime.html#utc_now/1-examples"},{"type":"function","title":"DateTime.utc_now/2","doc":"Returns the current datetime in UTC, supporting \na specific calendar and precision.\n\nIf you want the current time in Unix seconds,\nuse `System.os_time/1` instead.","ref":"DateTime.html#utc_now/2"},{"type":"function","title":"Examples - DateTime.utc_now/2","doc":"iex> datetime = DateTime.utc_now(:microsecond, Calendar.ISO)\n    iex> datetime.time_zone\n    \"Etc/UTC\"\n\n    iex> datetime = DateTime.utc_now(:second, Calendar.ISO)\n    iex> datetime.microsecond\n    {0, 0}","ref":"DateTime.html#utc_now/2-examples"},{"type":"type","title":"DateTime.t/0","doc":"","ref":"DateTime.html#t:t/0"},{"type":"behaviour","title":"Exception","doc":"Functions to format throw/catch/exit and exceptions.\n\nNote that stacktraces in Elixir are only available inside\ncatch and rescue by using the `__STACKTRACE__/0` variable.\n\nDo not rely on the particular format returned by the `format*`\nfunctions in this module. They may be changed in future releases\nin order to better suit Elixir's tool chain. In other words,\nby using the functions in this module it is guaranteed you will\nformat exceptions as in the current Elixir version being used.","ref":"Exception.html"},{"type":"callback","title":"Exception.blame/2","doc":"Called from `Exception.blame/3` to augment the exception struct.\n\nCan be used to collect additional information about the exception\nor do some additional expensive computation.","ref":"Exception.html#c:blame/2"},{"type":"function","title":"Exception.blame/3","doc":"Attaches information to exceptions for extra debugging.\n\nThis operation is potentially expensive, as it reads data\nfrom the file system, parses beam files, evaluates code and\nso on.\n\nIf the exception module implements the optional `c:blame/2`\ncallback, it will be invoked to perform the computation.","ref":"Exception.html#blame/3"},{"type":"function","title":"Exception.blame_mfa/3","doc":"Blames the invocation of the given module, function and arguments.\n\nThis function will retrieve the available clauses from bytecode\nand evaluate them against the given arguments. The clauses are\nreturned as a list of `{args, guards}` pairs where each argument\nand each top-level condition in a guard separated by `and`/`or`\nis wrapped in a tuple with blame metadata.\n\nThis function returns either `{:ok, definition, clauses}` or `:error`.\nWhere `definition` is `:def`, `:defp`, `:defmacro` or `:defmacrop`.","ref":"Exception.html#blame_mfa/3"},{"type":"callback","title":"Exception.exception/1","doc":"","ref":"Exception.html#c:exception/1"},{"type":"function","title":"Exception.format/3","doc":"Normalizes and formats throw/errors/exits and stacktraces.\n\nIt relies on `format_banner/3` and `format_stacktrace/1`\nto generate the final format.\n\nIf `kind` is `{:EXIT, pid}`, it does not generate a stacktrace,\nas such exits are retrieved as messages without stacktraces.","ref":"Exception.html#format/3"},{"type":"function","title":"Exception.format_banner/3","doc":"Normalizes and formats any throw/error/exit.\n\nThe message is formatted and displayed in the same\nformat as used by Elixir's CLI.\n\nThe third argument is the stacktrace which is used to enrich\na normalized error with more information. It is only used when\nthe kind is an error.","ref":"Exception.html#format_banner/3"},{"type":"function","title":"Exception.format_exit/1","doc":"Formats an exit. It returns a string.\n\nOften there are errors/exceptions inside exits. Exits are often\nwrapped by the caller and provide stacktraces too. This function\nformats exits in a way to nicely show the exit reason, caller\nand stacktrace.","ref":"Exception.html#format_exit/1"},{"type":"function","title":"Exception.format_fa/2","doc":"Receives an anonymous function and arity and formats it as\nshown in stacktraces. The arity may also be a list of arguments.","ref":"Exception.html#format_fa/2"},{"type":"function","title":"Examples - Exception.format_fa/2","doc":"Exception.format_fa(fn -> nil end, 1)\n    #=> \"#Function /1\"","ref":"Exception.html#format_fa/2-examples"},{"type":"function","title":"Exception.format_file_line/3","doc":"Formats the given `file` and `line` as shown in stacktraces.\n\nIf any of the values are `nil`, they are omitted.","ref":"Exception.html#format_file_line/3"},{"type":"function","title":"Examples - Exception.format_file_line/3","doc":"iex> Exception.format_file_line(\"foo\", 1)\n    \"foo:1:\"\n\n    iex> Exception.format_file_line(\"foo\", nil)\n    \"foo:\"\n\n    iex> Exception.format_file_line(nil, nil)\n    \"\"","ref":"Exception.html#format_file_line/3-examples"},{"type":"function","title":"Exception.format_file_line_column/4","doc":"Formats the given `file`, `line`, and `column` as shown in stacktraces.\n\nIf any of the values are `nil`, they are omitted.","ref":"Exception.html#format_file_line_column/4"},{"type":"function","title":"Examples - Exception.format_file_line_column/4","doc":"iex> Exception.format_file_line_column(\"foo\", 1, 2)\n    \"foo:1:2:\"\n\n    iex> Exception.format_file_line_column(\"foo\", 1, nil)\n    \"foo:1:\"\n\n    iex> Exception.format_file_line_column(\"foo\", nil, nil)\n    \"foo:\"\n\n    iex> Exception.format_file_line_column(\"foo\", nil, 2)\n    \"foo:\"\n\n    iex> Exception.format_file_line_column(nil, nil, nil)\n    \"\"","ref":"Exception.html#format_file_line_column/4-examples"},{"type":"function","title":"Exception.format_mfa/3","doc":"Receives a module, fun and arity and formats it\nas shown in stacktraces. The arity may also be a list\nof arguments.","ref":"Exception.html#format_mfa/3"},{"type":"function","title":"Examples - Exception.format_mfa/3","doc":"iex> Exception.format_mfa(Foo, :bar, 1)\n    \"Foo.bar/1\"\n\n    iex> Exception.format_mfa(Foo, :bar, [])\n    \"Foo.bar()\"\n\n    iex> Exception.format_mfa(nil, :bar, [])\n    \"nil.bar()\"\n\nAnonymous functions are reported as -func/arity-anonfn-count-,\nwhere func is the name of the enclosing function. Convert to\n\"anonymous fn in func/arity\"","ref":"Exception.html#format_mfa/3-examples"},{"type":"function","title":"Exception.format_stacktrace/1","doc":"Formats the stacktrace.\n\nA stacktrace must be given as an argument. If not, the stacktrace\nis retrieved from `Process.info/2`.","ref":"Exception.html#format_stacktrace/1"},{"type":"function","title":"Exception.format_stacktrace_entry/1","doc":"Receives a stacktrace entry and formats it into a string.","ref":"Exception.html#format_stacktrace_entry/1"},{"type":"function","title":"Exception.message/1","doc":"Gets the message for an `exception`.","ref":"Exception.html#message/1"},{"type":"callback","title":"Exception.message/1","doc":"","ref":"Exception.html#c:message/1"},{"type":"function","title":"Exception.normalize/3","doc":"Normalizes an exception, converting Erlang exceptions\nto Elixir exceptions.\n\nIt takes the `kind` spilled by `catch` as an argument and\nnormalizes only `:error`, returning the untouched payload\nfor others.\n\nThe third argument is the stacktrace which is used to enrich\na normalized error with more information. It is only used when\nthe kind is an error.","ref":"Exception.html#normalize/3"},{"type":"type","title":"Exception.arity_or_args/0","doc":"","ref":"Exception.html#t:arity_or_args/0"},{"type":"type","title":"Exception.kind/0","doc":"The kind handled by formatting functions","ref":"Exception.html#t:kind/0"},{"type":"type","title":"Exception.location/0","doc":"","ref":"Exception.html#t:location/0"},{"type":"type","title":"Exception.non_error_kind/0","doc":"","ref":"Exception.html#t:non_error_kind/0"},{"type":"type","title":"Exception.stacktrace/0","doc":"","ref":"Exception.html#t:stacktrace/0"},{"type":"type","title":"Exception.stacktrace_entry/0","doc":"","ref":"Exception.html#t:stacktrace_entry/0"},{"type":"type","title":"Exception.t/0","doc":"The exception type","ref":"Exception.html#t:t/0"},{"type":"module","title":"Float","doc":"Functions for working with floating-point numbers.\n\nFor mathematical operations on top of floating-points,\nsee Erlang's [`:math`](`:math`) module.","ref":"Float.html"},{"type":"module","title":"Kernel functions - Float","doc":"There are functions related to floating-point numbers on the `Kernel` module\ntoo. Here is a list of them:\n\n  * `Kernel.round/1`: rounds a number to the nearest integer.\n  * `Kernel.trunc/1`: returns the integer part of a number.","ref":"Float.html#module-kernel-functions"},{"type":"module","title":"Known issues - Float","doc":"There are some very well known problems with floating-point numbers\nand arithmetic due to the fact most decimal fractions cannot be\nrepresented by a floating-point binary and most operations are not exact,\nbut operate on approximations. Those issues are not specific\nto Elixir, they are a property of floating point representation itself.\n\nFor example, the numbers 0.1 and 0.01 are two of them, what means the result\nof squaring 0.1 does not give 0.01 neither the closest representable. Here is\nwhat happens in this case:\n\n  * The closest representable number to 0.1 is 0.1000000014\n  * The closest representable number to 0.01 is 0.0099999997\n  * Doing 0.1 * 0.1 should return 0.01, but because 0.1 is actually 0.1000000014,\n    the result is 0.010000000000000002, and because this is not the closest\n    representable number to 0.01, you'll get the wrong result for this operation\n\nThere are also other known problems like flooring or rounding numbers. See\n`round/2` and `floor/2` for more details about them.\n\nTo learn more about floating-point arithmetic visit:\n\n  * [0.30000000000000004.com](http://0.30000000000000004.com/)\n  * [What Every Programmer Should Know About Floating-Point Arithmetic](https://floating-point-gui.de/)","ref":"Float.html#module-known-issues"},{"type":"function","title":"Float.ceil/2","doc":"Rounds a float to the smallest integer greater than or equal to `num`.\n\n`ceil/2` also accepts a precision to round a floating-point value down\nto an arbitrary number of fractional digits (between 0 and 15).\n\nThe operation is performed on the binary floating point, without a\nconversion to decimal.\n\nThe behaviour of `ceil/2` for floats can be surprising. For example:\n\n    iex> Float.ceil(-12.52, 2)\n    -12.51\n\nOne may have expected it to ceil to -12.52. This is not a bug.\nMost decimal fractions cannot be represented as a binary floating point\nand therefore the number above is internally represented as -12.51999999,\nwhich explains the behaviour above.\n\nThis function always returns floats. `Kernel.trunc/1` may be used instead to\ntruncate the result to an integer afterwards.","ref":"Float.html#ceil/2"},{"type":"function","title":"Examples - Float.ceil/2","doc":"iex> Float.ceil(34.25)\n    35.0\n    iex> Float.ceil(-56.5)\n    -56.0\n    iex> Float.ceil(34.251, 2)\n    34.26","ref":"Float.html#ceil/2-examples"},{"type":"function","title":"Float.floor/2","doc":"Rounds a float to the largest number less than or equal to `num`.\n\n`floor/2` also accepts a precision to round a floating-point value down\nto an arbitrary number of fractional digits (between 0 and 15).\nThe operation is performed on the binary floating point, without a\nconversion to decimal.\n\nThis function always returns a float. `Kernel.trunc/1` may be used instead to\ntruncate the result to an integer afterwards.","ref":"Float.html#floor/2"},{"type":"function","title":"Known issues - Float.floor/2","doc":"The behaviour of `floor/2` for floats can be surprising. For example:\n\n    iex> Float.floor(12.52, 2)\n    12.51\n\nOne may have expected it to floor to 12.52. This is not a bug.\nMost decimal fractions cannot be represented as a binary floating point\nand therefore the number above is internally represented as 12.51999999,\nwhich explains the behaviour above.","ref":"Float.html#floor/2-known-issues"},{"type":"function","title":"Examples - Float.floor/2","doc":"iex> Float.floor(34.25)\n    34.0\n    iex> Float.floor(-56.5)\n    -57.0\n    iex> Float.floor(34.259, 2)\n    34.25","ref":"Float.html#floor/2-examples"},{"type":"function","title":"Float.max_finite/0","doc":"Returns the maximum finite value for a float.","ref":"Float.html#max_finite/0"},{"type":"function","title":"Examples - Float.max_finite/0","doc":"iex> Float.max_finite()\n    1.7976931348623157e308","ref":"Float.html#max_finite/0-examples"},{"type":"function","title":"Float.min_finite/0","doc":"Returns the minimum finite value for a float.","ref":"Float.html#min_finite/0"},{"type":"function","title":"Examples - Float.min_finite/0","doc":"iex> Float.min_finite()\n    -1.7976931348623157e308","ref":"Float.html#min_finite/0-examples"},{"type":"function","title":"Float.parse/1","doc":"Parses a binary into a float.\n\nIf successful, returns a tuple in the form of `{float, remainder_of_binary}`;\nwhen the binary cannot be coerced into a valid float, the atom `:error` is\nreturned.\n\nIf the size of float exceeds the maximum size of `1.7976931348623157e+308`,\n`:error` is returned even though the textual representation itself might be\nwell formed.\n\nIf you want to convert a string-formatted float directly to a float,\n`String.to_float/1` can be used instead.","ref":"Float.html#parse/1"},{"type":"function","title":"Examples - Float.parse/1","doc":"iex> Float.parse(\"34\")\n    {34.0, \"\"}\n    iex> Float.parse(\"34.25\")\n    {34.25, \"\"}\n    iex> Float.parse(\"56.5xyz\")\n    {56.5, \"xyz\"}\n\n    iex> Float.parse(\"pi\")\n    :error\n    iex> Float.parse(\"1.7976931348623159e+308\")\n    :error","ref":"Float.html#parse/1-examples"},{"type":"function","title":"Float.pow/2","doc":"Computes `base` raised to power of `exponent`.\n\n`base` must be a float and `exponent` can be any number.\nHowever, if a negative base and a fractional exponent\nare given, it raises `ArithmeticError`.\n\nIt always returns a float. See `Integer.pow/2` for\nexponentiation that returns integers.","ref":"Float.html#pow/2"},{"type":"function","title":"Examples - Float.pow/2","doc":"iex> Float.pow(2.0, 0)\n    1.0\n    iex> Float.pow(2.0, 1)\n    2.0\n    iex> Float.pow(2.0, 10)\n    1024.0\n    iex> Float.pow(2.0, -1)\n    0.5\n    iex> Float.pow(2.0, -3)\n    0.125\n\n    iex> Float.pow(3.0, 1.5)\n    5.196152422706632\n\n    iex> Float.pow(-2.0, 3)\n    -8.0\n    iex> Float.pow(-2.0, 4)\n    16.0\n\n    iex> Float.pow(-1.0, 0.5)\n    ** (ArithmeticError) bad argument in arithmetic expression","ref":"Float.html#pow/2-examples"},{"type":"function","title":"Float.ratio/1","doc":"Returns a pair of integers whose ratio is exactly equal\nto the original float and with a positive denominator.","ref":"Float.html#ratio/1"},{"type":"function","title":"Examples - Float.ratio/1","doc":"iex> Float.ratio(0.0)\n    {0, 1}\n    iex> Float.ratio(3.14)\n    {7070651414971679, 2251799813685248}\n    iex> Float.ratio(-3.14)\n    {-7070651414971679, 2251799813685248}\n    iex> Float.ratio(1.5)\n    {3, 2}\n    iex> Float.ratio(-1.5)\n    {-3, 2}\n    iex> Float.ratio(16.0)\n    {16, 1}\n    iex> Float.ratio(-16.0)\n    {-16, 1}","ref":"Float.html#ratio/1-examples"},{"type":"function","title":"Float.round/2","doc":"Rounds a floating-point value to an arbitrary number of fractional\ndigits (between 0 and 15).\n\nThe rounding direction always ties to half up. The operation is\nperformed on the binary floating point, without a conversion to decimal.\n\nThis function only accepts floats and always returns a float. Use\n`Kernel.round/1` if you want a function that accepts both floats\nand integers and always returns an integer.","ref":"Float.html#round/2"},{"type":"function","title":"Known issues - Float.round/2","doc":"The behaviour of `round/2` for floats can be surprising. For example:\n\n    iex> Float.round(5.5675, 3)\n    5.567\n\nOne may have expected it to round to the half up 5.568. This is not a bug.\nMost decimal fractions cannot be represented as a binary floating point\nand therefore the number above is internally represented as 5.567499999,\nwhich explains the behaviour above. If you want exact rounding for decimals,\nyou must use a decimal library. The behaviour above is also in accordance\nto reference implementations, such as \"Correctly Rounded Binary-Decimal and\nDecimal-Binary Conversions\" by David M. Gay.","ref":"Float.html#round/2-known-issues"},{"type":"function","title":"Examples - Float.round/2","doc":"iex> Float.round(12.5)\n    13.0\n    iex> Float.round(5.5674, 3)\n    5.567\n    iex> Float.round(5.5675, 3)\n    5.567\n    iex> Float.round(-5.5674, 3)\n    -5.567\n    iex> Float.round(-5.5675)\n    -6.0\n    iex> Float.round(12.341444444444441, 15)\n    12.341444444444441","ref":"Float.html#round/2-examples"},{"type":"function","title":"Float.to_charlist/1","doc":"Returns a charlist which corresponds to the shortest text representation\nof the given float.\n\nThe underlying algorithm changes depending on the Erlang/OTP version:\n\n  * For OTP >= 24, it uses the algorithm presented in \"Ryū: fast\n    float-to-string conversion\" in Proceedings of the SIGPLAN '2018\n    Conference on Programming Language Design and Implementation.\n\n  * For OTP < 24, it uses the algorithm presented in \"Printing Floating-Point\n    Numbers Quickly and Accurately\" in Proceedings of the SIGPLAN '1996\n    Conference on Programming Language Design and Implementation.\n\nFor a configurable representation, use `:erlang.float_to_list/2`.","ref":"Float.html#to_charlist/1"},{"type":"function","title":"Examples - Float.to_charlist/1","doc":"iex> Float.to_charlist(7.0)\n    '7.0'","ref":"Float.html#to_charlist/1-examples"},{"type":"function","title":"Float.to_string/1","doc":"Returns a binary which corresponds to the shortest text representation\nof the given float.\n\nThe underlying algorithm changes depending on the Erlang/OTP version:\n\n  * For OTP >= 24, it uses the algorithm presented in \"Ryū: fast\n    float-to-string conversion\" in Proceedings of the SIGPLAN '2018\n    Conference on Programming Language Design and Implementation.\n\n  * For OTP < 24, it uses the algorithm presented in \"Printing Floating-Point\n    Numbers Quickly and Accurately\" in Proceedings of the SIGPLAN '1996\n    Conference on Programming Language Design and Implementation.\n\nFor a configurable representation, use `:erlang.float_to_binary/2`.","ref":"Float.html#to_string/1"},{"type":"function","title":"Examples - Float.to_string/1","doc":"iex> Float.to_string(7.0)\n    \"7.0\"","ref":"Float.html#to_string/1-examples"},{"type":"type","title":"Float.precision_range/0","doc":"","ref":"Float.html#t:precision_range/0"},{"type":"module","title":"Function","doc":"A set of functions for working with functions.\n\nAnonymous functions are typically created by using `fn`:\n\n    iex> add = fn a, b -> a + b end\n    iex> add.(1, 2)\n    3\n\nAnonymous functions can also have multiple clauses. All clauses\nshould expect the same number of arguments:\n\n    iex> negate = fn\n    ...>   true -> false\n    ...>   false -> true\n    ...> end\n    iex> negate.(false)\n    true","ref":"Function.html"},{"type":"module","title":"The capture operator - Function","doc":"It is also possible to capture public module functions and pass them\naround as if they were anonymous functions by using the capture\noperator `&/1`:\n\n    iex> add = &Kernel.+/2\n    iex> add.(1, 2)\n    3\n\n    iex> length = &String.length/1\n    iex> length.(\"hello\")\n    5\n\nTo capture a definition within the current module, you can skip the\nmodule prefix, such as `&my_fun/2`. In those cases, the captured\nfunction can be public (`def`) or private (`defp`).\n\nThe capture operator can also be used to create anonymous functions\nthat expect at least one argument:\n\n    iex> add = &(&1 + &2)\n    iex> add.(1, 2)\n    3\n\nIn such cases, using the capture operator is no different than using `fn`.","ref":"Function.html#module-the-capture-operator"},{"type":"module","title":"Internal and external functions - Function","doc":"We say that functions that point to definitions residing in modules, such\nas `&String.length/1`, are **external** functions. All other functions are\n**local** and they are always bound to the file or module that defined them.\n\nBesides the functions in this module to work with functions, `Kernel` also\nhas an `apply/2` function that invokes a function with a dynamic number of\narguments, as well as `is_function/1` and `is_function/2`, to check\nrespectively if a given value is a function or a function of a given arity.","ref":"Function.html#module-internal-and-external-functions"},{"type":"function","title":"Function.capture/3","doc":"Captures the given function.\n\nInlined by the compiler.","ref":"Function.html#capture/3"},{"type":"function","title":"Examples - Function.capture/3","doc":"iex> Function.capture(String, :length, 1)\n    &String.length/1","ref":"Function.html#capture/3-examples"},{"type":"function","title":"Function.identity/1","doc":"Returns its input `value`. This function can be passed as an anonymous function\nto transformation functions.","ref":"Function.html#identity/1"},{"type":"function","title":"Examples - Function.identity/1","doc":"iex> Function.identity(\"Hello world!\")\n    \"Hello world!\"\n\n    iex> ~c\"abcdaabccc\" |> Enum.sort() |> Enum.chunk_by(&Function.identity/1)\n    [~c\"aaa\", ~c\"bb\", ~c\"cccc\", ~c\"d\"]\n\n    iex> Enum.group_by(~c\"abracadabra\", &Function.identity/1)\n    %{97 => ~c\"aaaaa\", 98 => ~c\"bb\", 99 => ~c\"c\", 100 => ~c\"d\", 114 => ~c\"rr\"}\n\n    iex> Enum.map([1, 2, 3, 4], &Function.identity/1)\n    [1, 2, 3, 4]","ref":"Function.html#identity/1-examples"},{"type":"function","title":"Function.info/1","doc":"Returns a keyword list with information about a function.\n\nThe returned keys (with the corresponding possible values) for\nall types of functions (local and external) are the following:\n\n  * `:type` - `:local` (for anonymous functions) or `:external` (for\n    named functions).\n\n  * `:module` - an atom which is the module where the function is defined when\n  anonymous or the module which the function refers to when it's a named function.\n\n  * `:arity` - (integer) the number of arguments the function is to be called with.\n\n  * `:name` - (atom) the name of the function.\n\n  * `:env` - a list of the environment or free variables. For named\n    functions, the returned list is always empty.\n\nWhen `fun` is an anonymous function (that is, the type is `:local`), the following\nadditional keys are returned:\n\n  * `:pid` - PID of the process that originally created the function.\n\n  * `:index` - (integer) an index into the module function table.\n\n  * `:new_index` - (integer) an index into the module function table.\n\n  * `:new_uniq` - (binary) a unique value for this function. It's\n    calculated from the compiled code for the entire module.\n\n  * `:uniq` - (integer) a unique value for this function. This integer is\n    calculated from the compiled code for the entire module.\n\n**Note**: this function must be used only for debugging purposes.\n\nInlined by the compiler.","ref":"Function.html#info/1"},{"type":"function","title":"Examples - Function.info/1","doc":"iex> fun = fn x -> x end\n    iex> info = Function.info(fun)\n    iex> Keyword.get(info, :arity)\n    1\n    iex> Keyword.get(info, :type)\n    :local\n\n    iex> fun = &String.length/1\n    iex> info = Function.info(fun)\n    iex> Keyword.get(info, :type)\n    :external\n    iex> Keyword.get(info, :name)\n    :length","ref":"Function.html#info/1-examples"},{"type":"function","title":"Function.info/2","doc":"Returns a specific information about the function.\n\nThe returned information is a two-element tuple in the shape of\n`{info, value}`.\n\nFor any function, the information asked for can be any of the atoms\n`:module`, `:name`, `:arity`, `:env`, or `:type`.\n\nFor anonymous functions, there is also information about any of the\natoms `:index`, `:new_index`, `:new_uniq`, `:uniq`, and `:pid`.\nFor a named function, the value of any of these items is always the\natom `:undefined`.\n\nFor more information on each of the possible returned values, see\n`info/1`.\n\nInlined by the compiler.","ref":"Function.html#info/2"},{"type":"function","title":"Examples - Function.info/2","doc":"iex> f = fn x -> x end\n    iex> Function.info(f, :arity)\n    {:arity, 1}\n    iex> Function.info(f, :type)\n    {:type, :local}\n\n    iex> fun = &String.length/1\n    iex> Function.info(fun, :name)\n    {:name, :length}\n    iex> Function.info(fun, :pid)\n    {:pid, :undefined}","ref":"Function.html#info/2-examples"},{"type":"type","title":"Function.information/0","doc":"","ref":"Function.html#t:information/0"},{"type":"module","title":"Integer","doc":"Functions for working with integers.\n\nSome functions that work on integers are found in `Kernel`:\n\n  * `Kernel.abs/1`\n  * `Kernel.div/2`\n  * `Kernel.max/2`\n  * `Kernel.min/2`\n  * `Kernel.rem/2`","ref":"Integer.html"},{"type":"function","title":"Integer.digits/2","doc":"Returns the ordered digits for the given `integer`.\n\nAn optional `base` value may be provided representing the radix for the returned\ndigits. This one must be an integer >= 2.","ref":"Integer.html#digits/2"},{"type":"function","title":"Examples - Integer.digits/2","doc":"iex> Integer.digits(123)\n    [1, 2, 3]\n\n    iex> Integer.digits(170, 2)\n    [1, 0, 1, 0, 1, 0, 1, 0]\n\n    iex> Integer.digits(-170, 2)\n    [-1, 0, -1, 0, -1, 0, -1, 0]","ref":"Integer.html#digits/2-examples"},{"type":"function","title":"Integer.extended_gcd/2","doc":"Returns the extended greatest common divisor of the two given integers.\n\nThis function uses the extended Euclidean algorithm to return a three-element tuple with the `gcd`\nand the coefficients `m` and `n` of Bézout's identity such that:\n\n    gcd(a, b) = m*a + n*b\n\nBy convention, `extended_gcd(0, 0)` returns `{0, 0, 0}`.","ref":"Integer.html#extended_gcd/2"},{"type":"function","title":"Examples - Integer.extended_gcd/2","doc":"iex> Integer.extended_gcd(240, 46)\n    {2, -9, 47}\n    iex> Integer.extended_gcd(46, 240)\n    {2, 47, -9}\n    iex> Integer.extended_gcd(-46, 240)\n    {2, -47, -9}\n    iex> Integer.extended_gcd(-46, -240)\n    {2, -47, 9}\n\n    iex> Integer.extended_gcd(14, 21)\n    {7, -1, 1}\n\n    iex> Integer.extended_gcd(10, 0)\n    {10, 1, 0}\n    iex> Integer.extended_gcd(0, 10)\n    {10, 0, 1}\n    iex> Integer.extended_gcd(0, 0)\n    {0, 0, 0}","ref":"Integer.html#extended_gcd/2-examples"},{"type":"function","title":"Integer.floor_div/2","doc":"Performs a floored integer division.\n\nRaises an `ArithmeticError` exception if one of the arguments is not an\ninteger, or when the `divisor` is `0`.\n\nThis function performs a *floored* integer division, which means that\nthe result will always be rounded towards negative infinity.\n\nIf you want to perform truncated integer division (rounding towards zero),\nuse `Kernel.div/2` instead.","ref":"Integer.html#floor_div/2"},{"type":"function","title":"Examples - Integer.floor_div/2","doc":"iex> Integer.floor_div(5, 2)\n    2\n    iex> Integer.floor_div(6, -4)\n    -2\n    iex> Integer.floor_div(-99, 2)\n    -50","ref":"Integer.html#floor_div/2-examples"},{"type":"function","title":"Integer.gcd/2","doc":"Returns the greatest common divisor of the two given integers.\n\nThe greatest common divisor (GCD) of `integer1` and `integer2` is the largest positive\ninteger that divides both `integer1` and `integer2` without leaving a remainder.\n\nBy convention, `gcd(0, 0)` returns `0`.","ref":"Integer.html#gcd/2"},{"type":"function","title":"Examples - Integer.gcd/2","doc":"iex> Integer.gcd(2, 3)\n    1\n\n    iex> Integer.gcd(8, 12)\n    4\n\n    iex> Integer.gcd(8, -12)\n    4\n\n    iex> Integer.gcd(10, 0)\n    10\n\n    iex> Integer.gcd(7, 7)\n    7\n\n    iex> Integer.gcd(0, 0)\n    0","ref":"Integer.html#gcd/2-examples"},{"type":"macro","title":"Integer.is_even/1","doc":"Determines if an `integer` is even.\n\nReturns `true` if the given `integer` is an even number,\notherwise it returns `false`.\n\nAllowed in guard clauses.","ref":"Integer.html#is_even/1"},{"type":"macro","title":"Examples - Integer.is_even/1","doc":"iex> Integer.is_even(10)\n    true\n\n    iex> Integer.is_even(5)\n    false\n\n    iex> Integer.is_even(-10)\n    true\n\n    iex> Integer.is_even(0)\n    true","ref":"Integer.html#is_even/1-examples"},{"type":"macro","title":"Integer.is_odd/1","doc":"Determines if `integer` is odd.\n\nReturns `true` if the given `integer` is an odd number,\notherwise it returns `false`.\n\nAllowed in guard clauses.","ref":"Integer.html#is_odd/1"},{"type":"macro","title":"Examples - Integer.is_odd/1","doc":"iex> Integer.is_odd(5)\n    true\n\n    iex> Integer.is_odd(6)\n    false\n\n    iex> Integer.is_odd(-5)\n    true\n\n    iex> Integer.is_odd(0)\n    false","ref":"Integer.html#is_odd/1-examples"},{"type":"function","title":"Integer.mod/2","doc":"Computes the modulo remainder of an integer division.\n\nThis function performs a [floored division](`floor_div/2`), which means that\nthe result will always have the sign of the `divisor`.\n\nRaises an `ArithmeticError` exception if one of the arguments is not an\ninteger, or when the `divisor` is `0`.","ref":"Integer.html#mod/2"},{"type":"function","title":"Examples - Integer.mod/2","doc":"iex> Integer.mod(5, 2)\n    1\n    iex> Integer.mod(6, -4)\n    -2","ref":"Integer.html#mod/2-examples"},{"type":"function","title":"Integer.parse/2","doc":"Parses a text representation of an integer.\n\nAn optional `base` to the corresponding integer can be provided.\nIf `base` is not given, 10 will be used.\n\nIf successful, returns a tuple in the form of `{integer, remainder_of_binary}`.\nOtherwise `:error`.\n\nRaises an error if `base` is less than 2 or more than 36.\n\nIf you want to convert a string-formatted integer directly to an integer,\n`String.to_integer/1` or `String.to_integer/2` can be used instead.","ref":"Integer.html#parse/2"},{"type":"function","title":"Examples - Integer.parse/2","doc":"iex> Integer.parse(\"34\")\n    {34, \"\"}\n\n    iex> Integer.parse(\"34.5\")\n    {34, \".5\"}\n\n    iex> Integer.parse(\"three\")\n    :error\n\n    iex> Integer.parse(\"34\", 10)\n    {34, \"\"}\n\n    iex> Integer.parse(\"f4\", 16)\n    {244, \"\"}\n\n    iex> Integer.parse(\"Awww++\", 36)\n    {509216, \"++\"}\n\n    iex> Integer.parse(\"fab\", 10)\n    :error\n\n    iex> Integer.parse(\"a2\", 38)\n    ** (ArgumentError) invalid base 38","ref":"Integer.html#parse/2-examples"},{"type":"function","title":"Integer.pow/2","doc":"Computes `base` raised to power of `exponent`.\n\nBoth `base` and `exponent` must be integers.\nThe exponent must be zero or positive.\n\nSee `Float.pow/2` for exponentiation of negative\nexponents as well as floats.","ref":"Integer.html#pow/2"},{"type":"function","title":"Examples - Integer.pow/2","doc":"iex> Integer.pow(2, 0)\n    1\n    iex> Integer.pow(2, 1)\n    2\n    iex> Integer.pow(2, 10)\n    1024\n    iex> Integer.pow(2, 11)\n    2048\n    iex> Integer.pow(2, 64)\n    0x10000000000000000\n\n    iex> Integer.pow(3, 4)\n    81\n    iex> Integer.pow(4, 3)\n    64\n\n    iex> Integer.pow(-2, 3)\n    -8\n    iex> Integer.pow(-2, 4)\n    16\n\n    iex> Integer.pow(2, -2)\n    ** (ArithmeticError) bad argument in arithmetic expression","ref":"Integer.html#pow/2-examples"},{"type":"function","title":"Integer.to_charlist/2","doc":"Returns a charlist which corresponds to the text representation\nof `integer` in the given `base`.\n\n`base` can be an integer between 2 and 36. If no `base` is given,\nit defaults to `10`.\n\nInlined by the compiler.","ref":"Integer.html#to_charlist/2"},{"type":"function","title":"Examples - Integer.to_charlist/2","doc":"iex> Integer.to_charlist(123)\n    '123'\n\n    iex> Integer.to_charlist(+456)\n    '456'\n\n    iex> Integer.to_charlist(-789)\n    '-789'\n\n    iex> Integer.to_charlist(0123)\n    '123'\n\n    iex> Integer.to_charlist(100, 16)\n    '64'\n\n    iex> Integer.to_charlist(-100, 16)\n    '-64'\n\n    iex> Integer.to_charlist(882_681_651, 36)\n    'ELIXIR'","ref":"Integer.html#to_charlist/2-examples"},{"type":"function","title":"Integer.to_string/2","doc":"Returns a binary which corresponds to the text representation\nof `integer` in the given `base`.\n\n`base` can be an integer between 2 and 36. If no `base` is given,\nit defaults to `10`.\n\nInlined by the compiler.","ref":"Integer.html#to_string/2"},{"type":"function","title":"Examples - Integer.to_string/2","doc":"iex> Integer.to_string(123)\n    \"123\"\n\n    iex> Integer.to_string(+456)\n    \"456\"\n\n    iex> Integer.to_string(-789)\n    \"-789\"\n\n    iex> Integer.to_string(0123)\n    \"123\"\n\n    iex> Integer.to_string(100, 16)\n    \"64\"\n\n    iex> Integer.to_string(-100, 16)\n    \"-64\"\n\n    iex> Integer.to_string(882_681_651, 36)\n    \"ELIXIR\"","ref":"Integer.html#to_string/2-examples"},{"type":"function","title":"Integer.undigits/2","doc":"Returns the integer represented by the ordered `digits`.\n\nAn optional `base` value may be provided representing the radix for the `digits`.\nBase has to be an integer greater than or equal to `2`.","ref":"Integer.html#undigits/2"},{"type":"function","title":"Examples - Integer.undigits/2","doc":"iex> Integer.undigits([1, 2, 3])\n    123\n\n    iex> Integer.undigits([1, 4], 16)\n    20\n\n    iex> Integer.undigits([])\n    0","ref":"Integer.html#undigits/2-examples"},{"type":"behaviour","title":"Module","doc":"Provides functions to deal with modules during compilation time.\n\nIt allows a developer to dynamically add, delete and register\nattributes, attach documentation and so forth.\n\nAfter a module is compiled, using many of the functions in\nthis module will raise errors, since it is out of their scope\nto inspect runtime data. Most of the runtime data can be inspected\nvia the [`__info__/1`](`c:Module.__info__/1`) function attached to\neach compiled module.","ref":"Module.html"},{"type":"behaviour","title":"Module attributes - Module","doc":"Each module can be decorated with one or more attributes. The following ones\nare currently defined by Elixir:\n\n### `@after_compile`\n\nA hook that will be invoked right after the current module is compiled.\nAccepts a module or a `{module, function_name}`. See the \"Compile callbacks\"\nsection below.\n\n### `@after_verify` (since v1.14.0)\n\nA hook that will be invoked right after the current module is verified for\nundefined functions, deprecations, etc. Accepts a module or a `{module, function_name}`.\nSee the \"Compile callbacks\" section below.\n\n### `@before_compile`\n\nA hook that will be invoked before the module is compiled.\nAccepts a module or a `{module, function_or_macro_name}` tuple.\nSee the \"Compile callbacks\" section below.\n\n### `@behaviour`\n\nNote the British spelling!\n\nBehaviours can be referenced by modules to ensure they implement\nrequired specific function signatures defined by `@callback`.\n\nFor example, you could specify a `URI.Parser` behaviour as follows:\n\n    defmodule URI.Parser do\n      @doc \"Defines a default port\"\n      @callback default_port() :: integer\n\n      @doc \"Parses the given URL\"\n      @callback parse(uri_info :: URI.t()) :: URI.t()\n    end\n\nAnd then a module may use it as:\n\n    defmodule URI.HTTP do\n      @behaviour URI.Parser\n      def default_port(), do: 80\n      def parse(info), do: info\n    end\n\nIf the behaviour changes or `URI.HTTP` does not implement\none of the callbacks, a warning will be raised.\n\nFor detailed documentation, see the\n[behaviour typespec documentation](typespecs.md#behaviours).\n\n### `@impl` (since v1.5.0)\n\nTo aid in the correct implementation of behaviours, you may optionally declare\n`@impl` for implemented callbacks of a behaviour. This makes callbacks\nexplicit and can help you to catch errors in your code. The compiler will warn\nin these cases:\n\n  * if you mark a function with `@impl` when that function is not a callback.\n\n  * if you don't mark a function with `@impl` when other functions are marked\n    with `@impl`. If you mark one function with `@impl`, you must mark all\n    other callbacks for that behaviour as `@impl`.\n\n`@impl` works on a per-context basis. If you generate a function through a macro\nand mark it with `@impl`, that won't affect the module where that function is\ngenerated in.\n\n`@impl` also helps with maintainability by making it clear to other developers\nthat the function is implementing a callback.\n\nUsing `@impl`, the example above can be rewritten as:\n\n    defmodule URI.HTTP do\n      @behaviour URI.Parser\n\n      @impl true\n      def default_port(), do: 80\n\n      @impl true\n      def parse(info), do: info\n    end\n\nYou may pass either `false`, `true`, or a specific behaviour to `@impl`.\n\n    defmodule Foo do\n      @behaviour Bar\n      @behaviour Baz\n\n      # Will warn if neither Bar nor Baz specify a callback named bar/0.\n      @impl true\n      def bar(), do: :ok\n\n      # Will warn if Baz does not specify a callback named baz/0.\n      @impl Baz\n      def baz(), do: :ok\n    end\n\nThe code is now more readable, as it is now clear which functions are\npart of your API and which ones are callback implementations. To reinforce this\nidea, `@impl true` automatically marks the function as `@doc false`, disabling\ndocumentation unless `@doc` is explicitly set.\n\n### `@compile`\n\nDefines options for module compilation. This is used to configure\nboth Elixir and Erlang compilers, as any other compilation pass\nadded by external tools. For example:\n\n    defmodule MyModule do\n      @compile {:inline, my_fun: 1}\n\n      def my_fun(arg) do\n        to_string(arg)\n      end\n    end\n\nMultiple uses of `@compile` will accumulate instead of overriding\nprevious ones. See the \"Compile options\" section below.\n\n### `@deprecated` (since v1.6.0)\n\nProvides the deprecation reason for a function. For example:\n\n    defmodule Keyword do\n      @deprecated \"Use Kernel.length/1 instead\"\n      def size(keyword) do\n        length(keyword)\n      end\n    end\n\nThe Mix compiler automatically looks for calls to deprecated modules\nand emit warnings during compilation.\n\nUsing the `@deprecated` attribute will also be reflected in the\ndocumentation of the given function and macro. You can choose between\nthe `@deprecated` attribute and the documentation metadata to provide\nhard-deprecations (with warnings) and soft-deprecations (without warnings):\n\nThis is a soft-deprecation as it simply annotates the documentation\nas deprecated:\n\n    @doc deprecated: \"Use Kernel.length/1 instead\"\n    def size(keyword)\n\nThis is a hard-deprecation as it emits warnings and annotates the\ndocumentation as deprecated:\n\n    @deprecated \"Use Kernel.length/1 instead\"\n    def size(keyword)\n\nCurrently `@deprecated` only supports functions and macros. However\nyou can use the `:deprecated` key in the annotation metadata to\nannotate the docs of modules, types and callbacks too.\n\nWe recommend using this feature with care, especially library authors.\nDeprecating code always pushes the burden towards library users. We\nalso recommend for deprecated functionality to be maintained for long\nperiods of time, even after deprecation, giving developers plenty of\ntime to update (except for cases where keeping the deprecated API is\nundesired, such as in the presence of security issues).\n\n### `@doc` and `@typedoc`\n\nProvides documentation for the entity that follows the attribute.\n`@doc` is to be used with a function, macro, callback, or\nmacrocallback, while `@typedoc` with a type (public or opaque).\n\nAccepts one of these:\n\n  * a string (often a heredoc)\n  * `false`, which will make the entity invisible to documentation-extraction\n    tools like [`ExDoc`](https://hexdocs.pm/ex_doc/)\n  * a keyword list, since Elixir 1.7.0\n\nFor example:\n\n    defmodule MyModule do\n      @typedoc \"This type\"\n      @typedoc since: \"1.1.0\"\n      @type t :: term\n\n      @doc \"Hello world\"\n      @doc since: \"1.1.0\"\n      def hello do\n        \"world\"\n      end\n\n      @doc \"\"\"\n      Sums `a` to `b`.\n      \"\"\"\n      def sum(a, b) do\n        a + b\n      end\n    end\n\nAs can be seen in the example above, since Elixir 1.7.0 `@doc` and `@typedoc`\nalso accept a keyword list that serves as a way to provide arbitrary metadata\nabout the entity. Tools like [`ExDoc`](https://hexdocs.pm/ex_doc/) and\n`IEx` may use this information to display annotations. A common use\ncase is the `:since` key, which may be used to annotate in which version the\nfunction was introduced.\n\nAs illustrated in the example, it is possible to use these attributes\nmore than once before an entity. However, the compiler will warn if\nused twice with binaries as that replaces the documentation text from\nthe preceding use. Multiple uses with keyword lists will merge the\nlists into one.\n\nNote that since the compiler also defines some additional metadata,\nthere are a few reserved keys that will be ignored and warned if used.\nCurrently these are: `:opaque` and `:defaults`.\n\nOnce this module is compiled, this information becomes available via\nthe `Code.fetch_docs/1` function.\n\n### `@dialyzer`\n\nDefines warnings to request or suppress when using `:dialyzer`.\n\nAccepts an atom, a tuple, or a list of atoms and tuples. For example:\n\n    defmodule MyModule do\n      @dialyzer {:nowarn_function, [my_fun: 1]}\n\n      def my_fun(arg) do\n        M.not_a_function(arg)\n      end\n    end\n\nFor the list of supported warnings, see [`:dialyzer` module](`:dialyzer`).\n\nMultiple uses of `@dialyzer` will accumulate instead of overriding\nprevious ones.\n\n### `@external_resource`\n\nSpecifies an external resource for the current module.\n\nSometimes a module embeds information from an external file. This\nattribute allows the module to annotate which external resources\nhave been used.\n\nTools may use this information to ensure the module is recompiled\nin case any of the external resources change, see for example:\n[`mix compile.elixir`](https://hexdocs.pm/mix/Mix.Tasks.Compile.Elixir.html).\n\nThe specified file path provided is interpreted as relative to\nthe folder containing the project's `mix.exs`, which is the\ncurrent working directory, not the file where `@external_resource`\nis declared.\n\nIf the external resource does not exist, the module still has\na dependency on it, causing the module to be recompiled as soon\nas the file is added.\n\n### `@file`\n\nChanges the filename used in stacktraces for the function or macro that\nfollows the attribute, such as:\n\n    defmodule MyModule do\n      @doc \"Hello world\"\n      @file \"hello.ex\"\n      def hello do\n        \"world\"\n      end\n    end\n\n### `@moduledoc`\n\nProvides documentation for the current module.\n\n    defmodule MyModule do\n      @moduledoc \"\"\"\n      A very useful module.\n      \"\"\"\n      @moduledoc authors: [\"Alice\", \"Bob\"]\n    end\n\nAccepts a string (often a heredoc) or `false` where `@moduledoc false`\nwill make the module invisible to documentation extraction tools like\n[`ExDoc`](https://hexdocs.pm/ex_doc/).\n\nSimilarly to `@doc` also accepts a keyword list to provide metadata\nabout the module. For more details, see the documentation of `@doc`\nabove.\n\nOnce this module is compiled, this information becomes available via\nthe `Code.fetch_docs/1` function.\n\n### `@on_definition`\n\nA hook that will be invoked when each function or macro in the current\nmodule is defined. Useful when annotating functions.\n\nAccepts a module or a `{module, function_name}` tuple. The function\nmust take 6 arguments:\n\n  * the module environment\n  * the kind of the function/macro: `:def`, `:defp`, `:defmacro`, or `:defmacrop`\n  * the function/macro name\n  * the list of quoted arguments\n  * the list of quoted guards\n  * the quoted function body\n\nIf the function/macro being defined has multiple clauses, the hook will\nbe called for each clause.\n\nUnlike other hooks, `@on_definition` will only invoke functions and\nnever macros. This is to avoid `@on_definition` callbacks from\nredefining functions that have just been defined in favor of more\nexplicit approaches.\n\nWhen just a module is provided, the function is assumed to be\n`__on_definition__/6`.\n\n##","ref":"Module.html#module-module-attributes"},{"type":"behaviour","title":"Example - Module","doc":"defmodule Hooks do\n      def on_def(_env, kind, name, args, guards, body) do\n        IO.puts(\"Defining #{kind} named #{name} with args:\")\n        IO.inspect(args)\n        IO.puts(\"and guards\")\n        IO.inspect(guards)\n        IO.puts(\"and body\")\n        IO.puts(Macro.to_string(body))\n      end\n    end\n\n    defmodule MyModule do\n      @on_definition {Hooks, :on_def}\n\n      def hello(arg) when is_binary(arg) or is_list(arg) do\n        \"Hello\"   to_string(arg)\n      end\n\n      def hello(_) do\n        :ok\n      end\n    end\n\n### `@on_load`\n\nA hook that will be invoked whenever the module is loaded.\n\nAccepts the function name (as an atom) of a function in the current module.\nThe function must have an arity of 0 (no arguments). If the function does\nnot return `:ok`, the loading of the module will be aborted.\nFor example:\n\n    defmodule MyModule do\n      @on_load :load_check\n\n      def load_check do\n        if some_condition() do\n          :ok\n        else\n          :abort\n        end\n      end\n\n      def some_condition do\n        false\n      end\n    end\n\n### `@vsn`\n\nSpecify the module version. Accepts any valid Elixir value, for example:\n\n    defmodule MyModule do\n      @vsn \"1.0\"\n    end\n\n#","ref":"Module.html#module-example"},{"type":"behaviour","title":"Struct attributes - Module","doc":"* `@derive` - derives an implementation for the given protocol for the\n    struct defined in the current module\n\n  * `@enforce_keys` - ensures the given keys are always set when building\n    the struct defined in the current module\n\nSee `defstruct/1` for more information on building and using structs.\n\n#","ref":"Module.html#module-struct-attributes"},{"type":"behaviour","title":"Typespec attributes - Module","doc":"The following attributes are part of typespecs and are also built-in in\nElixir:\n\n  * `@type` - defines a type to be used in `@spec`\n  * `@typep` - defines a private type to be used in `@spec`\n  * `@opaque` - defines an opaque type to be used in `@spec`\n  * `@spec` - provides a specification for a function\n  * `@callback` - provides a specification for a behaviour callback\n  * `@macrocallback` - provides a specification for a macro behaviour callback\n  * `@optional_callbacks` - specifies which behaviour callbacks and macro\n    behaviour callbacks are optional\n  * `@impl` - declares an implementation of a callback function or macro\n\nFor detailed documentation, see the [typespec documentation](typespecs.md).\n\n#","ref":"Module.html#module-typespec-attributes"},{"type":"behaviour","title":"Custom attributes - Module","doc":"In addition to the built-in attributes outlined above, custom attributes may\nalso be added. Custom attributes are expressed using the `@/1` operator followed\nby a valid variable name. The value given to the custom attribute must be a valid\nElixir value:\n\n    defmodule MyModule do\n      @custom_attr [some: \"stuff\"]\n    end\n\nFor more advanced options available when defining custom attributes, see\n`register_attribute/3`.","ref":"Module.html#module-custom-attributes"},{"type":"behaviour","title":"Compile callbacks - Module","doc":"There are three compilation callbacks, invoked in this order:\n`@before_compile`, `@after_compile`, and `@after_verify`.\nThey are described next.\n\n### `@before_compile`\n\nA hook that will be invoked before the module is compiled. This is\noften used to change how the current module is being compiled.\n\nAccepts a module or a `{module, function_or_macro_name}` tuple. The\nfunction/macro must take one argument: the module environment. If\nit's a macro, its returned value will be injected at the end of the\nmodule definition before the compilation starts.\n\nWhen just a module is provided, the function/macro is assumed to be\n`__before_compile__/1`.\n\nCallbacks will run in the order they are registered. Any overridable\ndefinition will be made concrete before the first callback runs.\nA definition may be made overridable again in another before compile\ncallback and it will be made concrete one last time after all callbacks\nrun.\n\n*Note*: the callback function/macro must be placed in a separate module\n(because when the callback is invoked, the current module does not yet exist).\n\n##","ref":"Module.html#module-compile-callbacks"},{"type":"behaviour","title":"Example - Module","doc":"defmodule A do\n      defmacro __before_compile__(_env) do\n        quote do\n          def hello, do: \"world\"\n        end\n      end\n    end\n\n    defmodule B do\n      @before_compile A\n    end\n\n    B.hello()\n    #=> \"world\"\n\n### `@after_compile`\n\nA hook that will be invoked right after the current module is compiled.\n\nAccepts a module or a `{module, function_name}` tuple. The function\nmust take two arguments: the module environment and its bytecode.\nWhen just a module is provided, the function is assumed to be\n`__after_compile__/2`.\n\nCallbacks will run in the order they are registered.\n\n`Module` functions expecting not yet compiled modules (such as `definitions_in/1`)\nare still available at the time `@after_compile` is invoked.\n\n##","ref":"Module.html#module-example"},{"type":"behaviour","title":"Example - Module","doc":"defmodule MyModule do\n      @after_compile __MODULE__\n\n      def __after_compile__(env, _bytecode) do\n        IO.inspect(env)\n      end\n    end\n\n### `@after_verify`\n\nA hook that will be invoked right after the current module is verified for\nundefined functions, deprecations, etc. A module is always verified after\nit is compiled. In Mix projects, a module is also verified when any of its\nruntime dependencies change. Therefore this is useful to perform verification\nof the current module while avoiding compile-time dependencies.\n\nAccepts a module or a `{module, function_name}` tuple. The function\nmust take one argument: the module name. When just a module is provided,\nthe function is assumed to be `__after_verify__/1`.\n\nCallbacks will run in the order they are registered.\n\n`Module` functions expecting not yet compiled modules are no longer available\nat the time `@after_verify` is invoked.\n\n##","ref":"Module.html#module-example"},{"type":"behaviour","title":"Example - Module","doc":"defmodule MyModule do\n      @after_verify __MODULE__\n\n      def __after_verify__(module) do\n        IO.inspect(module)\n        :ok\n      end\n    end","ref":"Module.html#module-example"},{"type":"behaviour","title":"Compile options - Module","doc":"The `@compile` attribute accepts different options that are used by both\nElixir and Erlang compilers. Some of the common use cases are documented\nbelow:\n\n  * `@compile :debug_info` - includes `:debug_info` regardless of the\n    corresponding setting in `Code.get_compiler_option/1`\n\n  * `@compile {:debug_info, false}` - disables `:debug_info` regardless\n    of the corresponding setting in `Code.get_compiler_option/1`. Note\n    disabling `:debug_info` is not recommended as it removes the ability\n    of the Elixir compiler and other tools to static analyse the code.\n    If you want to remove the `:debug_info` while deploying, tools like\n    `mix release` already do such by default.\n\n  * `@compile {:inline, some_fun: 2, other_fun: 3}` - inlines the given\n    name/arity pairs. Inlining is applied locally, calls from another\n    module are not affected by this option\n\n  * `@compile {:autoload, false}` - disables automatic loading of\n    modules after compilation. Instead, the module will be loaded after\n    it is dispatched to\n\n  * `@compile {:no_warn_undefined, Mod}` or\n    `@compile {:no_warn_undefined, {Mod, fun, arity}}` - does not warn if\n    the given module or the given `Mod.fun/arity` are not defined","ref":"Module.html#module-compile-options"},{"type":"callback","title":"Module.__info__/1","doc":"Provides runtime information about functions, macros, and other information\ndefined by the module.\n\nEach module gets an `__info__/1` function when it's compiled. The function\ntakes one of the following items:\n\n  * `:attributes` - a keyword list with all persisted attributes\n\n  * `:compile` - a list with compiler metadata\n\n  * `:functions` - a keyword list of public functions and their arities\n\n  * `:macros` - a keyword list of public macros and their arities\n\n  * `:md5` - the MD5 of the module\n\n  * `:module` - the module atom name\n\n  * `:struct` - (since v1.14.0) if the module defines a struct and if so each field in order","ref":"Module.html#c:__info__/1"},{"type":"function","title":"Module.attributes_in/1","doc":"Returns all module attributes names defined in `module`.\n\nThis function can only be used on modules that have not yet been compiled.","ref":"Module.html#attributes_in/1"},{"type":"function","title":"Examples - Module.attributes_in/1","doc":"defmodule Example do\n      @foo 1\n      Module.register_attribute(__MODULE__, :bar, accumulate: true)\n\n      :foo in Module.attributes_in(__MODULE__)\n      #=> true\n\n      :bar in Module.attributes_in(__MODULE__)\n      #=> true\n    end","ref":"Module.html#attributes_in/1-examples"},{"type":"function","title":"Module.concat/1","doc":"Concatenates a list of aliases and returns a new alias.\n\nIt handles binaries and atoms.","ref":"Module.html#concat/1"},{"type":"function","title":"Examples - Module.concat/1","doc":"iex> Module.concat([Foo, Bar])\n    Foo.Bar\n\n    iex> Module.concat([Foo, \"Bar\"])\n    Foo.Bar","ref":"Module.html#concat/1-examples"},{"type":"function","title":"Module.concat/2","doc":"Concatenates two aliases and returns a new alias.\n\nIt handles binaries and atoms.","ref":"Module.html#concat/2"},{"type":"function","title":"Examples - Module.concat/2","doc":"iex> Module.concat(Foo, Bar)\n    Foo.Bar\n\n    iex> Module.concat(Foo, \"Bar\")\n    Foo.Bar","ref":"Module.html#concat/2-examples"},{"type":"function","title":"Module.create/3","doc":"Creates a module with the given name and defined by\nthe given quoted expressions.\n\nThe line where the module is defined and its file **must**\nbe passed as options.\n\nIt returns a tuple of shape `{:module, module, binary, term}`\nwhere `module` is the module name, `binary` is the module\nbytecode and `term` is the result of the last expression in\n`quoted`.\n\nSimilar to `Kernel.defmodule/2`, the binary will only be\nwritten to disk as a `.beam` file if `Module.create/3` is\ninvoked in a file that is currently being compiled.","ref":"Module.html#create/3"},{"type":"function","title":"Examples - Module.create/3","doc":"contents =\n      quote do\n        def world, do: true\n      end\n\n    Module.create(Hello, contents, Macro.Env.location(__ENV__))\n\n    Hello.world()\n    #=> true","ref":"Module.html#create/3-examples"},{"type":"function","title":"Differences from `defmodule` - Module.create/3","doc":"`Module.create/3` works similarly to `Kernel.defmodule/2`\nand return the same results. While one could also use\n`Kernel.defmodule/2` to define modules dynamically, this function\nis preferred when the module body is given by a quoted\nexpression.\n\nAnother important distinction is that `Module.create/3`\nallows you to control the environment variables used\nwhen defining the module, while `Kernel.defmodule/2`\nautomatically uses the environment it is invoked at.","ref":"Module.html#create/3-differences-from-defmodule"},{"type":"function","title":"Module.defines?/2","doc":"Checks if the module defines the given function or macro.\n\nUse `defines?/3` to assert for a specific type.\n\nThis function can only be used on modules that have not yet been compiled.\nUse `Kernel.function_exported?/3` and `Kernel.macro_exported?/3` to check for\npublic functions and macros respectively in compiled modules.\n\nNote that `defines?` returns false for functions and macros that have\nbeen defined but then marked as overridable and no other implementation\nhas been provided. You can check the overridable status by calling\n`overridable?/2`.","ref":"Module.html#defines?/2"},{"type":"function","title":"Examples - Module.defines?/2","doc":"defmodule Example do\n      Module.defines?(__MODULE__, {:version, 0}) #=> false\n      def version, do: 1\n      Module.defines?(__MODULE__, {:version, 0}) #=> true\n    end","ref":"Module.html#defines?/2-examples"},{"type":"function","title":"Module.defines?/3","doc":"Checks if the module defines a function or macro of the\ngiven `kind`.\n\n`kind` can be any of `:def`, `:defp`, `:defmacro`, or `:defmacrop`.\n\nThis function can only be used on modules that have not yet been compiled.\nUse `Kernel.function_exported?/3` and `Kernel.macro_exported?/3` to check for\npublic functions and macros respectively in compiled modules.","ref":"Module.html#defines?/3"},{"type":"function","title":"Examples - Module.defines?/3","doc":"defmodule Example do\n      Module.defines?(__MODULE__, {:version, 0}, :def) #=> false\n      def version, do: 1\n      Module.defines?(__MODULE__, {:version, 0}, :def) #=> true\n    end","ref":"Module.html#defines?/3-examples"},{"type":"function","title":"Module.defines_type?/2","doc":"Checks if the current module defines the given type (private, opaque or not).\n\nThis function is only available for modules being compiled.","ref":"Module.html#defines_type?/2"},{"type":"function","title":"Module.definitions_in/1","doc":"Returns all functions and macros defined in `module`.\n\nIt returns a list with all defined functions and macros, public and private,\nin the shape of `[{name, arity}, ...]`.\n\nThis function can only be used on modules that have not yet been compiled.\nUse the `c:Module.__info__/1` callback to get the public functions and macros in\ncompiled modules.","ref":"Module.html#definitions_in/1"},{"type":"function","title":"Examples - Module.definitions_in/1","doc":"defmodule Example do\n      def version, do: 1\n      defmacrop test(arg), do: arg\n      Module.definitions_in(__MODULE__) #=> [{:version, 0}, {:test, 1}]\n    end","ref":"Module.html#definitions_in/1-examples"},{"type":"function","title":"Module.definitions_in/2","doc":"Returns all functions defined in `module`, according\nto its kind.\n\nThis function can only be used on modules that have not yet been compiled.\nUse the `c:Module.__info__/1` callback to get the public functions and macros in\ncompiled modules.","ref":"Module.html#definitions_in/2"},{"type":"function","title":"Examples - Module.definitions_in/2","doc":"defmodule Example do\n      def version, do: 1\n      Module.definitions_in(__MODULE__, :def)  #=> [{:version, 0}]\n      Module.definitions_in(__MODULE__, :defp) #=> []\n    end","ref":"Module.html#definitions_in/2-examples"},{"type":"function","title":"Module.delete_attribute/2","doc":"Deletes the entry (or entries) for the given module attribute.\n\nIt returns the deleted attribute value. If the attribute has not\nbeen set nor configured to accumulate, it returns `nil`.\n\nIf the attribute is set to accumulate, then this function always\nreturns a list. Deleting the attribute removes existing entries\nbut the attribute will still accumulate.","ref":"Module.html#delete_attribute/2"},{"type":"function","title":"Examples - Module.delete_attribute/2","doc":"defmodule MyModule do\n      Module.put_attribute(__MODULE__, :custom_threshold_for_lib, 10)\n      Module.delete_attribute(__MODULE__, :custom_threshold_for_lib)\n    end","ref":"Module.html#delete_attribute/2-examples"},{"type":"function","title":"Module.delete_definition/2","doc":"Deletes a definition from a module.\n\nIt returns true if the definition exists and it was removed,\notherwise it returns false.","ref":"Module.html#delete_definition/2"},{"type":"function","title":"Module.eval_quoted/4","doc":"Evaluates the quoted contents in the given module's context.\n\nA list of environment options can also be given as argument.\nSee `Code.eval_string/3` for more information.\n\nRaises an error if the module was already compiled.","ref":"Module.html#eval_quoted/4"},{"type":"function","title":"Examples - Module.eval_quoted/4","doc":"defmodule Foo do\n      contents =\n        quote do\n          def sum(a, b), do: a + b\n        end\n\n      Module.eval_quoted(__MODULE__, contents)\n    end\n\n    Foo.sum(1, 2)\n    #=> 3\n\nFor convenience, you can pass any `Macro.Env` struct, such\nas  `__ENV__/0`, as the first argument or as options. Both\nthe module and all options will be automatically extracted\nfrom the environment:\n\n    defmodule Foo do\n      contents =\n        quote do\n          def sum(a, b), do: a + b\n        end\n\n      Module.eval_quoted(__ENV__, contents)\n    end\n\n    Foo.sum(1, 2)\n    #=> 3\n\nNote that if you pass a `Macro.Env` struct as first argument\nwhile also passing `opts`, they will be merged with `opts`\nhaving precedence.","ref":"Module.html#eval_quoted/4-examples"},{"type":"function","title":"Module.get_attribute/3","doc":"Gets the given attribute from a module.\n\nIf the attribute was marked with `accumulate` with\n`Module.register_attribute/3`, a list is always returned.\n`nil` is returned if the attribute has not been marked with\n`accumulate` and has not been set to any value.\n\nThe `@` macro compiles to a call to this function. For example,\nthe following code:\n\n    @foo\n\nExpands to something akin to:\n\n    Module.get_attribute(__MODULE__, :foo)\n\nThis function can only be used on modules that have not yet been compiled.\nUse the `c:Module.__info__/1` callback to get all persisted attributes, or\n`Code.fetch_docs/1` to retrieve all documentation related attributes in\ncompiled modules.","ref":"Module.html#get_attribute/3"},{"type":"function","title":"Examples - Module.get_attribute/3","doc":"defmodule Foo do\n      Module.put_attribute(__MODULE__, :value, 1)\n      Module.get_attribute(__MODULE__, :value) #=> 1\n\n      Module.get_attribute(__MODULE__, :value, :default) #=> 1\n      Module.get_attribute(__MODULE__, :not_found, :default) #=> :default\n\n      Module.register_attribute(__MODULE__, :value, accumulate: true)\n      Module.put_attribute(__MODULE__, :value, 1)\n      Module.get_attribute(__MODULE__, :value) #=> [1]\n    end","ref":"Module.html#get_attribute/3-examples"},{"type":"function","title":"Module.get_definition/3","doc":"Returns the definition for the given name-arity pair.\n\nIt returns a tuple with the `version`, the `kind`,\nthe definition `metadata`, and a list with each clause.\nEach clause is a four-element tuple with metadata,\nthe arguments, the guards, and the clause AST.\n\nThe clauses are returned in the Elixir AST but a subset\nthat has already been expanded and normalized. This makes\nit useful for analyzing code but it cannot be reinjected\ninto the module as it will have lost some of its original\ncontext. Given this AST representation is mostly internal,\nit is versioned and it may change at any time. Therefore,\n**use this API with caution**.","ref":"Module.html#get_definition/3"},{"type":"function","title":"Options - Module.get_definition/3","doc":"* `:skip_clauses` (since v1.14.0) - returns `[]` instead\n    of returning the clauses. This is useful when there is\n    only an interest in fetching the kind and the metadata","ref":"Module.html#get_definition/3-options"},{"type":"function","title":"Module.get_last_attribute/3","doc":"Gets the last set value of a given attribute from a module.\n\nIf the attribute was marked with `accumulate` with\n`Module.register_attribute/3`, the previous value to have been set will be\nreturned. If the attribute does not accumulate, this call is the same as\ncalling `Module.get_attribute/3`.\n\nThis function can only be used on modules that have not yet been compiled.\nUse the `c:Module.__info__/1` callback to get all persisted attributes, or\n`Code.fetch_docs/1` to retrieve all documentation related attributes in\ncompiled modules.","ref":"Module.html#get_last_attribute/3"},{"type":"function","title":"Examples - Module.get_last_attribute/3","doc":"defmodule Foo do\n      Module.put_attribute(__MODULE__, :value, 1)\n      Module.get_last_attribute(__MODULE__, :value) #=> 1\n\n      Module.get_last_attribute(__MODULE__, :not_found, :default) #=> :default\n\n      Module.register_attribute(__MODULE__, :acc, accumulate: true)\n      Module.put_attribute(__MODULE__, :acc, 1)\n      Module.get_last_attribute(__MODULE__, :acc) #=> 1\n      Module.put_attribute(__MODULE__, :acc, 2)\n      Module.get_last_attribute(__MODULE__, :acc) #=> 2\n    end","ref":"Module.html#get_last_attribute/3-examples"},{"type":"function","title":"Module.has_attribute?/2","doc":"Checks if the given attribute has been defined.\n\nAn attribute is defined if it has been registered with `register_attribute/3`\nor assigned a value. If an attribute has been deleted with `delete_attribute/2`\nit is no longer considered defined.\n\nThis function can only be used on modules that have not yet been compiled.","ref":"Module.html#has_attribute?/2"},{"type":"function","title":"Examples - Module.has_attribute?/2","doc":"defmodule MyModule do\n      @value 1\n      Module.register_attribute(__MODULE__, :other_value)\n      Module.put_attribute(__MODULE__, :another_value, 1)\n\n      Module.has_attribute?(__MODULE__, :value) #=> true\n      Module.has_attribute?(__MODULE__, :other_value) #=> true\n      Module.has_attribute?(__MODULE__, :another_value) #=> true\n\n      Module.has_attribute?(__MODULE__, :undefined) #=> false\n\n      Module.delete_attribute(__MODULE__, :value)\n      Module.has_attribute?(__MODULE__, :value) #=> false\n    end","ref":"Module.html#has_attribute?/2-examples"},{"type":"function","title":"Module.make_overridable/2","doc":"Makes the given functions in `module` overridable.\n\nAn overridable function is lazily defined, allowing a\ndeveloper to customize it. See `Kernel.defoverridable/1` for\nmore information and documentation.\n\nOnce a function or a macro is marked as overridable, it will\nno longer be listed under `definitions_in/1` or return true\nwhen given to `defines?/2` until another implementation is\ngiven.","ref":"Module.html#make_overridable/2"},{"type":"function","title":"Module.open?/1","doc":"Checks if a module is open.\n\nA module is \"open\" if it is currently being defined and its attributes and\nfunctions can be modified.","ref":"Module.html#open?/1"},{"type":"function","title":"Module.overridable?/2","doc":"Returns `true` if `tuple` in `module` was marked as overridable\nat some point.\n\nNote `overridable?/2` returns true even if the definition was\nalready overridden. You can use `defines?/2` to see if a definition\nexists or one is pending.","ref":"Module.html#overridable?/2"},{"type":"function","title":"Module.overridables_in/1","doc":"Returns all overridable definitions in `module`.\n\nNote a definition is included even if it was was already overridden.\nYou can use `defines?/2` to see if a definition exists or one is pending.\n\nThis function can only be used on modules that have not yet been compiled.","ref":"Module.html#overridables_in/1"},{"type":"function","title":"Examples - Module.overridables_in/1","doc":"defmodule Example do\n      def foo, do: 1\n      def bar, do: 2\n\n      defoverridable foo: 0, bar: 0\n      def foo, do: 3\n\n      [bar: 0, foo: 0] = Module.overridables_in(__MODULE__) |> Enum.sort()\n    end","ref":"Module.html#overridables_in/1-examples"},{"type":"function","title":"Module.put_attribute/3","doc":"Puts a module attribute with `key` and `value` in the given `module`.","ref":"Module.html#put_attribute/3"},{"type":"function","title":"Examples - Module.put_attribute/3","doc":"defmodule MyModule do\n      Module.put_attribute(__MODULE__, :custom_threshold_for_lib, 10)\n    end","ref":"Module.html#put_attribute/3-examples"},{"type":"function","title":"Module.register_attribute/3","doc":"Registers an attribute.\n\nBy registering an attribute, a developer is able to customize\nhow Elixir will store and accumulate the attribute values.","ref":"Module.html#register_attribute/3"},{"type":"function","title":"Options - Module.register_attribute/3","doc":"When registering an attribute, two options can be given:\n\n  * `:accumulate` - several calls to the same attribute will\n    accumulate instead of overriding the previous one. New attributes\n    are always added to the top of the accumulated list.\n\n  * `:persist` - the attribute will be persisted in the Erlang\n    Abstract Format. Useful when interfacing with Erlang libraries.\n\nBy default, both options are `false`. Once an attribute has been\nset to accumulate or persist, the behaviour cannot be reverted.","ref":"Module.html#register_attribute/3-options"},{"type":"function","title":"Examples - Module.register_attribute/3","doc":"defmodule MyModule do\n      Module.register_attribute(__MODULE__, :custom_threshold_for_lib, accumulate: true)\n\n      @custom_threshold_for_lib 10\n      @custom_threshold_for_lib 20\n      @custom_threshold_for_lib #=> [20, 10]\n    end","ref":"Module.html#register_attribute/3-examples"},{"type":"function","title":"Module.reserved_attributes/0","doc":"Returns information about module attributes used by Elixir.\n\nSee the \"Module attributes\" section in the module documentation for more\ninformation on each attribute.","ref":"Module.html#reserved_attributes/0"},{"type":"function","title":"Examples - Module.reserved_attributes/0","doc":"iex> map = Module.reserved_attributes()\n    iex> Map.has_key?(map, :moduledoc)\n    true\n    iex> Map.has_key?(map, :doc)\n    true","ref":"Module.html#reserved_attributes/0-examples"},{"type":"function","title":"Module.safe_concat/1","doc":"Concatenates a list of aliases and returns a new alias only if the alias\nwas already referenced.\n\nIf the alias was not referenced yet, fails with `ArgumentError`.\nIt handles binaries and atoms.","ref":"Module.html#safe_concat/1"},{"type":"function","title":"Examples - Module.safe_concat/1","doc":"iex> Module.safe_concat([List, Chars])\n    List.Chars","ref":"Module.html#safe_concat/1-examples"},{"type":"function","title":"Module.safe_concat/2","doc":"Concatenates two aliases and returns a new alias only if the alias was\nalready referenced.\n\nIf the alias was not referenced yet, fails with `ArgumentError`.\nIt handles binaries and atoms.","ref":"Module.html#safe_concat/2"},{"type":"function","title":"Examples - Module.safe_concat/2","doc":"iex> Module.safe_concat(List, Chars)\n    List.Chars","ref":"Module.html#safe_concat/2-examples"},{"type":"function","title":"Module.spec_to_callback/2","doc":"Copies the given spec as a callback.\n\nReturns `true` if there is such a spec and it was copied as a callback.\nIf the function associated to the spec has documentation defined prior to\ninvoking this function, the docs are copied too.","ref":"Module.html#spec_to_callback/2"},{"type":"function","title":"Module.split/1","doc":"Splits the given module name into binary parts.\n\n`module` has to be an Elixir module, as `split/1` won't work with Erlang-style\nmodules (for example, `split(:lists)` raises an error).\n\n`split/1` also supports splitting the string representation of Elixir modules\n(that is, the result of calling `Atom.to_string/1` with the module name).","ref":"Module.html#split/1"},{"type":"function","title":"Examples - Module.split/1","doc":"iex> Module.split(Very.Long.Module.Name.And.Even.Longer)\n    [\"Very\", \"Long\", \"Module\", \"Name\", \"And\", \"Even\", \"Longer\"]\n    iex> Module.split(\"Elixir.String.Chars\")\n    [\"String\", \"Chars\"]","ref":"Module.html#split/1-examples"},{"type":"type","title":"Module.def_kind/0","doc":"","ref":"Module.html#t:def_kind/0"},{"type":"type","title":"Module.definition/0","doc":"","ref":"Module.html#t:definition/0"},{"type":"module","title":"NaiveDateTime","doc":"A NaiveDateTime struct (without a time zone) and functions.\n\nThe NaiveDateTime struct contains the fields year, month, day, hour,\nminute, second, microsecond and calendar. New naive datetimes can be\nbuilt with the `new/2` and `new/8` functions or using the\n`~N` (see `sigil_N/2`) sigil:\n\n    iex> ~N[2000-01-01 23:00:07]\n    ~N[2000-01-01 23:00:07]\n\nThe date and time fields in the struct can be accessed directly:\n\n    iex> naive = ~N[2000-01-01 23:00:07]\n    iex> naive.year\n    2000\n    iex> naive.second\n    7\n\nWe call them \"naive\" because this datetime representation does not\nhave a time zone. This means the datetime may not actually exist in\ncertain areas in the world even though it is valid.\n\nFor example, when daylight saving changes are applied by a region,\nthe clock typically moves forward or backward by one hour. This means\ncertain datetimes never occur or may occur more than once. Since\n`NaiveDateTime` is not validated against a time zone, such errors\nwould go unnoticed.\n\nDevelopers should avoid creating the NaiveDateTime structs directly\nand instead, rely on the functions provided by this module as well\nas the ones in third-party calendar libraries.","ref":"NaiveDateTime.html"},{"type":"module","title":"Comparing naive date times - NaiveDateTime","doc":"Comparisons in Elixir using `==/2`, `>/2`, `  Enum.min([~N[2020-01-01 23:00:07], ~N[2000-01-01 23:00:07]], NaiveDateTime)\n    ~N[2000-01-01 23:00:07]","ref":"NaiveDateTime.html#module-comparing-naive-date-times"},{"type":"module","title":"Using epochs - NaiveDateTime","doc":"The `add/3` and `diff/3` functions can be used for computing date\ntimes or retrieving the number of seconds between instants.\nFor example, if there is an interest in computing the number of\nseconds from the Unix epoch (1970-01-01 00:00:00):\n\n    iex> NaiveDateTime.diff(~N[2010-04-17 14:00:00], ~N[1970-01-01 00:00:00])\n    1271512800\n\n    iex> NaiveDateTime.add(~N[1970-01-01 00:00:00], 1_271_512_800)\n    ~N[2010-04-17 14:00:00]\n\nThose functions are optimized to deal with common epochs, such\nas the Unix Epoch above or the Gregorian Epoch (0000-01-01 00:00:00).","ref":"NaiveDateTime.html#module-using-epochs"},{"type":"function","title":"NaiveDateTime.add/3","doc":"Adds a specified amount of time to a `NaiveDateTime`.\n\nAccepts an `amount_to_add` in any `unit`. `unit` can be `:day`,\n`:hour`, `:minute`, `:second` or any subsecond precision from\n`t:System.time_unit/0`. It defaults to `:second`. Negative values\nwill move backwards in time.\n\nThis function always consider the unit to be computed according\nto the `Calendar.ISO`.","ref":"NaiveDateTime.html#add/3"},{"type":"function","title":"Examples - NaiveDateTime.add/3","doc":"It uses seconds by default:\n\n    # adds seconds by default\n    iex> NaiveDateTime.add(~N[2014-10-02 00:29:10], 2)\n    ~N[2014-10-02 00:29:12]\n\n    # accepts negative offsets\n    iex> NaiveDateTime.add(~N[2014-10-02 00:29:10], -2)\n    ~N[2014-10-02 00:29:08]\n\nIt can also work with subsecond precisions:\n\n    iex> NaiveDateTime.add(~N[2014-10-02 00:29:10], 2_000, :millisecond)\n    ~N[2014-10-02 00:29:12.000]\n\nAs well as days/hours/minutes:\n\n    iex> NaiveDateTime.add(~N[2015-02-28 00:29:10], 2, :day)\n    ~N[2015-03-02 00:29:10]\n    iex> NaiveDateTime.add(~N[2015-02-28 00:29:10], 36, :hour)\n    ~N[2015-03-01 12:29:10]\n    iex> NaiveDateTime.add(~N[2015-02-28 00:29:10], 60, :minute)\n    ~N[2015-02-28 01:29:10]\n\nThis operation merges the precision of the naive date time with the given unit:\n\n    iex> result = NaiveDateTime.add(~N[2014-10-02 00:29:10], 21, :millisecond)\n    ~N[2014-10-02 00:29:10.021]\n    iex> result.microsecond\n    {21000, 3}\n\nOperations on top of gregorian seconds or the Unix epoch are optimized:\n\n    # from Gregorian seconds\n    iex> NaiveDateTime.add(~N[0000-01-01 00:00:00], 63_579_428_950)\n    ~N[2014-10-02 00:29:10]\n\nPassing a `DateTime` automatically converts it to `NaiveDateTime`,\ndiscarding the time zone information:\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> NaiveDateTime.add(dt, 21, :second)\n    ~N[2000-02-29 23:00:28]","ref":"NaiveDateTime.html#add/3-examples"},{"type":"function","title":"NaiveDateTime.after?/2","doc":"Returns true if the first `NaiveDateTime` is strictly later than the second.","ref":"NaiveDateTime.html#after?/2"},{"type":"function","title":"Examples - NaiveDateTime.after?/2","doc":"iex> NaiveDateTime.after?(~N[2022-02-02 11:00:00], ~N[2021-01-01 11:00:00])\n    true\n    iex> NaiveDateTime.after?(~N[2021-01-01 11:00:00], ~N[2021-01-01 11:00:00])\n    false\n    iex> NaiveDateTime.after?(~N[2021-01-01 11:00:00], ~N[2022-02-02 11:00:00])\n    false","ref":"NaiveDateTime.html#after?/2-examples"},{"type":"function","title":"NaiveDateTime.before?/2","doc":"Returns true if the first `NaiveDateTime` is strictly earlier than the second.","ref":"NaiveDateTime.html#before?/2"},{"type":"function","title":"Examples - NaiveDateTime.before?/2","doc":"iex> NaiveDateTime.before?(~N[2021-01-01 11:00:00], ~N[2022-02-02 11:00:00])\n    true\n    iex> NaiveDateTime.before?(~N[2021-01-01 11:00:00], ~N[2021-01-01 11:00:00])\n    false\n    iex> NaiveDateTime.before?(~N[2022-02-02 11:00:00], ~N[2021-01-01 11:00:00])\n    false","ref":"NaiveDateTime.html#before?/2-examples"},{"type":"function","title":"NaiveDateTime.beginning_of_day/1","doc":"Calculates a `NaiveDateTime` that is the first moment for the given `NaiveDateTime`.\n\nTo calculate the beginning of day of a `DateTime`, call this function, then convert back to a `DateTime`:\n\n    datetime\n    |> NaiveDateTime.beginning_of_day()\n    |> DateTime.from_naive(datetime.timezone)\n\nNote that the beginning of the day may not exist or be ambiguous\nin a given timezone, so you must handle those cases accordingly.","ref":"NaiveDateTime.html#beginning_of_day/1"},{"type":"function","title":"Examples - NaiveDateTime.beginning_of_day/1","doc":"iex> NaiveDateTime.beginning_of_day(~N[2000-01-01 23:00:07.123456])\n    ~N[2000-01-01 00:00:00.000000]","ref":"NaiveDateTime.html#beginning_of_day/1-examples"},{"type":"function","title":"NaiveDateTime.compare/2","doc":"Compares two `NaiveDateTime` structs.\n\nReturns `:gt` if first is later than the second\nand `:lt` for vice versa. If the two NaiveDateTime\nare equal `:eq` is returned.","ref":"NaiveDateTime.html#compare/2"},{"type":"function","title":"Examples - NaiveDateTime.compare/2","doc":"iex> NaiveDateTime.compare(~N[2016-04-16 13:30:15], ~N[2016-04-28 16:19:25])\n    :lt\n    iex> NaiveDateTime.compare(~N[2016-04-16 13:30:15.1], ~N[2016-04-16 13:30:15.01])\n    :gt\n\nThis function can also be used to compare a DateTime without\nthe time zone information:\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> NaiveDateTime.compare(dt, ~N[2000-02-29 23:00:07])\n    :eq\n    iex> NaiveDateTime.compare(dt, ~N[2000-01-29 23:00:07])\n    :gt\n    iex> NaiveDateTime.compare(dt, ~N[2000-03-29 23:00:07])\n    :lt","ref":"NaiveDateTime.html#compare/2-examples"},{"type":"function","title":"NaiveDateTime.convert/2","doc":"Converts the given `naive_datetime` from one calendar to another.\n\nIf it is not possible to convert unambiguously between the calendars\n(see `Calendar.compatible_calendars?/2`), an `{:error, :incompatible_calendars}` tuple\nis returned.","ref":"NaiveDateTime.html#convert/2"},{"type":"function","title":"Examples - NaiveDateTime.convert/2","doc":"Imagine someone implements `Calendar.Holocene`, a calendar based on the\nGregorian calendar that adds exactly 10,000 years to the current Gregorian\nyear:\n\n    iex> NaiveDateTime.convert(~N[2000-01-01 13:30:15], Calendar.Holocene)\n    {:ok, %NaiveDateTime{calendar: Calendar.Holocene, year: 12000, month: 1, day: 1,\n                         hour: 13, minute: 30, second: 15, microsecond: {0, 0}}}","ref":"NaiveDateTime.html#convert/2-examples"},{"type":"function","title":"NaiveDateTime.convert!/2","doc":"Converts the given `naive_datetime` from one calendar to another.\n\nIf it is not possible to convert unambiguously between the calendars\n(see `Calendar.compatible_calendars?/2`), an ArgumentError is raised.","ref":"NaiveDateTime.html#convert!/2"},{"type":"function","title":"Examples - NaiveDateTime.convert!/2","doc":"Imagine someone implements `Calendar.Holocene`, a calendar based on the\nGregorian calendar that adds exactly 10,000 years to the current Gregorian\nyear:\n\n    iex> NaiveDateTime.convert!(~N[2000-01-01 13:30:15], Calendar.Holocene)\n    %NaiveDateTime{calendar: Calendar.Holocene, year: 12000, month: 1, day: 1,\n                   hour: 13, minute: 30, second: 15, microsecond: {0, 0}}","ref":"NaiveDateTime.html#convert!/2-examples"},{"type":"function","title":"NaiveDateTime.diff/3","doc":"Subtracts `naive_datetime2` from `naive_datetime1`.\n\nThe answer can be returned in any `:day`, `:hour`, `:minute`, or any `unit`\navailable from `t:System.time_unit/0`. The unit is measured according to\n`Calendar.ISO` and defaults to `:second`.\n\nFractional results are not supported and are truncated.","ref":"NaiveDateTime.html#diff/3"},{"type":"function","title":"Examples - NaiveDateTime.diff/3","doc":"iex> NaiveDateTime.diff(~N[2014-10-02 00:29:12], ~N[2014-10-02 00:29:10])\n    2\n    iex> NaiveDateTime.diff(~N[2014-10-02 00:29:12], ~N[2014-10-02 00:29:10], :microsecond)\n    2_000_000\n\n    iex> NaiveDateTime.diff(~N[2014-10-02 00:29:10.042], ~N[2014-10-02 00:29:10.021])\n    0\n    iex> NaiveDateTime.diff(~N[2014-10-02 00:29:10.042], ~N[2014-10-02 00:29:10.021], :millisecond)\n    21\n\n    iex> NaiveDateTime.diff(~N[2014-10-02 00:29:10], ~N[2014-10-02 00:29:12])\n    -2\n    iex> NaiveDateTime.diff(~N[-0001-10-02 00:29:10], ~N[-0001-10-02 00:29:12])\n    -2\n\nIt can also compute the difference in days, hours, or minutes:\n\n    iex> NaiveDateTime.diff(~N[2014-10-10 00:29:10], ~N[2014-10-02 00:29:10], :day)\n    8\n    iex> NaiveDateTime.diff(~N[2014-10-02 12:29:10], ~N[2014-10-02 00:29:10], :hour)\n    12\n    iex> NaiveDateTime.diff(~N[2014-10-02 00:39:10], ~N[2014-10-02 00:29:10], :minute)\n    10\n\nBut it also rounds incomplete days to zero:\n\n    iex> NaiveDateTime.diff(~N[2014-10-10 00:29:09], ~N[2014-10-02 00:29:10], :day)\n    7","ref":"NaiveDateTime.html#diff/3-examples"},{"type":"function","title":"NaiveDateTime.end_of_day/1","doc":"Calculates a `NaiveDateTime` that is the last moment for the given `NaiveDateTime`.\n\nTo calculate the end of day of a `DateTime`, call this function, then convert back to a `DateTime`:\n\n    datetime\n    |> NaiveDateTime.end_of_day()\n    |> DateTime.from_naive(datetime.timezone)\n\nNote that the end of the day may not exist or be ambiguous\nin a given timezone, so you must handle those cases accordingly.","ref":"NaiveDateTime.html#end_of_day/1"},{"type":"function","title":"Examples - NaiveDateTime.end_of_day/1","doc":"iex> NaiveDateTime.end_of_day(~N[2000-01-01 23:00:07.123456])\n    ~N[2000-01-01 23:59:59.999999]","ref":"NaiveDateTime.html#end_of_day/1-examples"},{"type":"function","title":"NaiveDateTime.from_erl/3","doc":"Converts an Erlang datetime tuple to a `NaiveDateTime` struct.\n\nAttempting to convert an invalid ISO calendar date will produce an error tuple.","ref":"NaiveDateTime.html#from_erl/3"},{"type":"function","title":"Examples - NaiveDateTime.from_erl/3","doc":"iex> NaiveDateTime.from_erl({{2000, 1, 1}, {13, 30, 15}})\n    {:ok, ~N[2000-01-01 13:30:15]}\n    iex> NaiveDateTime.from_erl({{2000, 1, 1}, {13, 30, 15}}, {5000, 3})\n    {:ok, ~N[2000-01-01 13:30:15.005]}\n    iex> NaiveDateTime.from_erl({{2000, 13, 1}, {13, 30, 15}})\n    {:error, :invalid_date}\n    iex> NaiveDateTime.from_erl({{2000, 13, 1}, {13, 30, 15}})\n    {:error, :invalid_date}","ref":"NaiveDateTime.html#from_erl/3-examples"},{"type":"function","title":"NaiveDateTime.from_erl!/3","doc":"Converts an Erlang datetime tuple to a `NaiveDateTime` struct.\n\nRaises if the datetime is invalid.\nAttempting to convert an invalid ISO calendar date will produce an error tuple.","ref":"NaiveDateTime.html#from_erl!/3"},{"type":"function","title":"Examples - NaiveDateTime.from_erl!/3","doc":"iex> NaiveDateTime.from_erl!({{2000, 1, 1}, {13, 30, 15}})\n    ~N[2000-01-01 13:30:15]\n    iex> NaiveDateTime.from_erl!({{2000, 1, 1}, {13, 30, 15}}, {5000, 3})\n    ~N[2000-01-01 13:30:15.005]\n    iex> NaiveDateTime.from_erl!({{2000, 13, 1}, {13, 30, 15}})\n    ** (ArgumentError) cannot convert {{2000, 13, 1}, {13, 30, 15}} to naive datetime, reason: :invalid_date","ref":"NaiveDateTime.html#from_erl!/3-examples"},{"type":"function","title":"NaiveDateTime.from_gregorian_seconds/3","doc":"Converts a number of gregorian seconds to a `NaiveDateTime` struct.","ref":"NaiveDateTime.html#from_gregorian_seconds/3"},{"type":"function","title":"Examples - NaiveDateTime.from_gregorian_seconds/3","doc":"iex> NaiveDateTime.from_gregorian_seconds(1)\n    ~N[0000-01-01 00:00:01]\n    iex> NaiveDateTime.from_gregorian_seconds(63_755_511_991, {5000, 3})\n    ~N[2020-05-01 00:26:31.005]\n    iex> NaiveDateTime.from_gregorian_seconds(-1)\n    ~N[-0001-12-31 23:59:59]","ref":"NaiveDateTime.html#from_gregorian_seconds/3-examples"},{"type":"function","title":"NaiveDateTime.from_iso8601/2","doc":"Parses the extended \"Date and time of day\" format described by\n[ISO 8601:2019](https://en.wikipedia.org/wiki/ISO_8601).\n\nTime zone offset may be included in the string but they will be\nsimply discarded as such information is not included in naive date\ntimes.\n\nAs specified in the standard, the separator \"T\" may be omitted if\ndesired as there is no ambiguity within this function.\n\nNote leap seconds are not supported by the built-in Calendar.ISO.","ref":"NaiveDateTime.html#from_iso8601/2"},{"type":"function","title":"Examples - NaiveDateTime.from_iso8601/2","doc":"iex> NaiveDateTime.from_iso8601(\"2015-01-23 23:50:07\")\n    {:ok, ~N[2015-01-23 23:50:07]}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23T23:50:07\")\n    {:ok, ~N[2015-01-23 23:50:07]}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23T23:50:07Z\")\n    {:ok, ~N[2015-01-23 23:50:07]}\n\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23 23:50:07.0\")\n    {:ok, ~N[2015-01-23 23:50:07.0]}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23 23:50:07,0123456\")\n    {:ok, ~N[2015-01-23 23:50:07.012345]}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23 23:50:07.0123456\")\n    {:ok, ~N[2015-01-23 23:50:07.012345]}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23T23:50:07.123Z\")\n    {:ok, ~N[2015-01-23 23:50:07.123]}\n\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23P23:50:07\")\n    {:error, :invalid_format}\n    iex> NaiveDateTime.from_iso8601(\"2015:01:23 23-50-07\")\n    {:error, :invalid_format}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23 23:50:07A\")\n    {:error, :invalid_format}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23 23:50:61\")\n    {:error, :invalid_time}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-32 23:50:07\")\n    {:error, :invalid_date}\n\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23T23:50:07.123+02:30\")\n    {:ok, ~N[2015-01-23 23:50:07.123]}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23T23:50:07.123+00:00\")\n    {:ok, ~N[2015-01-23 23:50:07.123]}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23T23:50:07.123-02:30\")\n    {:ok, ~N[2015-01-23 23:50:07.123]}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23T23:50:07.123-00:00\")\n    {:error, :invalid_format}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23T23:50:07.123-00:60\")\n    {:error, :invalid_format}\n    iex> NaiveDateTime.from_iso8601(\"2015-01-23T23:50:07.123-24:00\")\n    {:error, :invalid_format}","ref":"NaiveDateTime.html#from_iso8601/2-examples"},{"type":"function","title":"NaiveDateTime.from_iso8601!/2","doc":"Parses the extended \"Date and time of day\" format described by\n[ISO 8601:2019](https://en.wikipedia.org/wiki/ISO_8601).\n\nRaises if the format is invalid.","ref":"NaiveDateTime.html#from_iso8601!/2"},{"type":"function","title":"Examples - NaiveDateTime.from_iso8601!/2","doc":"iex> NaiveDateTime.from_iso8601!(\"2015-01-23T23:50:07.123Z\")\n    ~N[2015-01-23 23:50:07.123]\n    iex> NaiveDateTime.from_iso8601!(\"2015-01-23T23:50:07,123Z\")\n    ~N[2015-01-23 23:50:07.123]\n    iex> NaiveDateTime.from_iso8601!(\"2015-01-23P23:50:07\")\n    ** (ArgumentError) cannot parse \"2015-01-23P23:50:07\" as naive datetime, reason: :invalid_format","ref":"NaiveDateTime.html#from_iso8601!/2-examples"},{"type":"function","title":"NaiveDateTime.local_now/1","doc":"Returns the \"local time\" for the machine the Elixir program is running on.\n\nWARNING: This function can cause insidious bugs. It depends on the time zone\nconfiguration at run time. This can changed and be set to a time zone that has\ndaylight saving jumps (spring forward or fall back).\n\nThis function can be used to display what the time is right now for the time\nzone configuration that the machine happens to have. An example would be a\ndesktop program displaying a clock to the user. For any other uses it is\nprobably a bad idea to use this function.\n\nFor most cases, use `DateTime.now/2` or `DateTime.utc_now/1` instead.\n\nDoes not include fractional seconds.","ref":"NaiveDateTime.html#local_now/1"},{"type":"function","title":"Examples - NaiveDateTime.local_now/1","doc":"iex> naive_datetime = NaiveDateTime.local_now()\n    iex> naive_datetime.year >= 2019\n    true","ref":"NaiveDateTime.html#local_now/1-examples"},{"type":"function","title":"NaiveDateTime.new/2","doc":"Builds a naive datetime from date and time structs.","ref":"NaiveDateTime.html#new/2"},{"type":"function","title":"Examples - NaiveDateTime.new/2","doc":"iex> NaiveDateTime.new(~D[2010-01-13], ~T[23:00:07.005])\n    {:ok, ~N[2010-01-13 23:00:07.005]}","ref":"NaiveDateTime.html#new/2-examples"},{"type":"function","title":"NaiveDateTime.new/8","doc":"Builds a new ISO naive datetime.\n\nExpects all values to be integers. Returns `{:ok, naive_datetime}`\nif each entry fits its appropriate range, returns `{:error, reason}`\notherwise.","ref":"NaiveDateTime.html#new/8"},{"type":"function","title":"Examples - NaiveDateTime.new/8","doc":"iex> NaiveDateTime.new(2000, 1, 1, 0, 0, 0)\n    {:ok, ~N[2000-01-01 00:00:00]}\n    iex> NaiveDateTime.new(2000, 13, 1, 0, 0, 0)\n    {:error, :invalid_date}\n    iex> NaiveDateTime.new(2000, 2, 29, 0, 0, 0)\n    {:ok, ~N[2000-02-29 00:00:00]}\n    iex> NaiveDateTime.new(2000, 2, 30, 0, 0, 0)\n    {:error, :invalid_date}\n    iex> NaiveDateTime.new(2001, 2, 29, 0, 0, 0)\n    {:error, :invalid_date}\n\n    iex> NaiveDateTime.new(2000, 1, 1, 23, 59, 59, {0, 1})\n    {:ok, ~N[2000-01-01 23:59:59.0]}\n    iex> NaiveDateTime.new(2000, 1, 1, 23, 59, 59, 999_999)\n    {:ok, ~N[2000-01-01 23:59:59.999999]}\n    iex> NaiveDateTime.new(2000, 1, 1, 24, 59, 59, 999_999)\n    {:error, :invalid_time}\n    iex> NaiveDateTime.new(2000, 1, 1, 23, 60, 59, 999_999)\n    {:error, :invalid_time}\n    iex> NaiveDateTime.new(2000, 1, 1, 23, 59, 60, 999_999)\n    {:error, :invalid_time}\n    iex> NaiveDateTime.new(2000, 1, 1, 23, 59, 59, 1_000_000)\n    {:error, :invalid_time}\n\n    iex> NaiveDateTime.new(2000, 1, 1, 23, 59, 59, {0, 1}, Calendar.ISO)\n    {:ok, ~N[2000-01-01 23:59:59.0]}","ref":"NaiveDateTime.html#new/8-examples"},{"type":"function","title":"NaiveDateTime.new!/2","doc":"Builds a naive datetime from date and time structs.","ref":"NaiveDateTime.html#new!/2"},{"type":"function","title":"Examples - NaiveDateTime.new!/2","doc":"iex> NaiveDateTime.new!(~D[2010-01-13], ~T[23:00:07.005])\n    ~N[2010-01-13 23:00:07.005]","ref":"NaiveDateTime.html#new!/2-examples"},{"type":"function","title":"NaiveDateTime.new!/8","doc":"Builds a new ISO naive datetime.\n\nExpects all values to be integers. Returns `naive_datetime`\nif each entry fits its appropriate range, raises if\ntime or date is invalid.","ref":"NaiveDateTime.html#new!/8"},{"type":"function","title":"Examples - NaiveDateTime.new!/8","doc":"iex> NaiveDateTime.new!(2000, 1, 1, 0, 0, 0)\n    ~N[2000-01-01 00:00:00]\n    iex> NaiveDateTime.new!(2000, 2, 29, 0, 0, 0)\n    ~N[2000-02-29 00:00:00]\n    iex> NaiveDateTime.new!(2000, 1, 1, 23, 59, 59, {0, 1})\n    ~N[2000-01-01 23:59:59.0]\n    iex> NaiveDateTime.new!(2000, 1, 1, 23, 59, 59, 999_999)\n    ~N[2000-01-01 23:59:59.999999]\n    iex> NaiveDateTime.new!(2000, 1, 1, 23, 59, 59, {0, 1}, Calendar.ISO)\n    ~N[2000-01-01 23:59:59.0]\n    iex> NaiveDateTime.new!(2000, 1, 1, 24, 59, 59, 999_999)\n    ** (ArgumentError) cannot build naive datetime, reason: :invalid_time","ref":"NaiveDateTime.html#new!/8-examples"},{"type":"function","title":"NaiveDateTime.to_date/1","doc":"Converts a `NaiveDateTime` into a `Date`.\n\nBecause `Date` does not hold time information,\ndata will be lost during the conversion.","ref":"NaiveDateTime.html#to_date/1"},{"type":"function","title":"Examples - NaiveDateTime.to_date/1","doc":"iex> NaiveDateTime.to_date(~N[2002-01-13 23:00:07])\n    ~D[2002-01-13]","ref":"NaiveDateTime.html#to_date/1-examples"},{"type":"function","title":"NaiveDateTime.to_erl/1","doc":"Converts a `NaiveDateTime` struct to an Erlang datetime tuple.\n\nOnly supports converting naive datetimes which are in the ISO calendar,\nattempting to convert naive datetimes from other calendars will raise.\n\nWARNING: Loss of precision may occur, as Erlang time tuples only store\nhour/minute/second.","ref":"NaiveDateTime.html#to_erl/1"},{"type":"function","title":"Examples - NaiveDateTime.to_erl/1","doc":"iex> NaiveDateTime.to_erl(~N[2000-01-01 13:30:15])\n    {{2000, 1, 1}, {13, 30, 15}}\n\nThis function can also be used to convert a DateTime to an Erlang\ndatetime tuple without the time zone information:\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> NaiveDateTime.to_erl(dt)\n    {{2000, 2, 29}, {23, 00, 07}}","ref":"NaiveDateTime.html#to_erl/1-examples"},{"type":"function","title":"NaiveDateTime.to_gregorian_seconds/1","doc":"Converts a `NaiveDateTime` struct to a number of gregorian seconds and microseconds.","ref":"NaiveDateTime.html#to_gregorian_seconds/1"},{"type":"function","title":"Examples - NaiveDateTime.to_gregorian_seconds/1","doc":"iex> NaiveDateTime.to_gregorian_seconds(~N[0000-01-01 00:00:01])\n    {1, 0}\n    iex> NaiveDateTime.to_gregorian_seconds(~N[2020-05-01 00:26:31.005])\n    {63_755_511_991, 5000}","ref":"NaiveDateTime.html#to_gregorian_seconds/1-examples"},{"type":"function","title":"NaiveDateTime.to_iso8601/2","doc":"Converts the given naive datetime to\n[ISO 8601:2019](https://en.wikipedia.org/wiki/ISO_8601).\n\nBy default, `NaiveDateTime.to_iso8601/2` returns naive datetimes formatted in the \"extended\"\nformat, for human readability. It also supports the \"basic\" format through passing the `:basic` option.\n\nOnly supports converting naive datetimes which are in the ISO calendar,\nattempting to convert naive datetimes from other calendars will raise.\n\n#","ref":"NaiveDateTime.html#to_iso8601/2"},{"type":"function","title":"Examples - NaiveDateTime.to_iso8601/2","doc":"iex> NaiveDateTime.to_iso8601(~N[2000-02-28 23:00:13])\n    \"2000-02-28T23:00:13\"\n\n    iex> NaiveDateTime.to_iso8601(~N[2000-02-28 23:00:13.001])\n    \"2000-02-28T23:00:13.001\"\n\n    iex> NaiveDateTime.to_iso8601(~N[2000-02-28 23:00:13.001], :basic)\n    \"20000228T230013.001\"\n\nThis function can also be used to convert a DateTime to ISO 8601 without\nthe time zone information:\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> NaiveDateTime.to_iso8601(dt)\n    \"2000-02-29T23:00:07\"","ref":"NaiveDateTime.html#to_iso8601/2-examples"},{"type":"function","title":"NaiveDateTime.to_string/1","doc":"Converts the given naive datetime to a string according to its calendar.\n\n#","ref":"NaiveDateTime.html#to_string/1"},{"type":"function","title":"Examples - NaiveDateTime.to_string/1","doc":"iex> NaiveDateTime.to_string(~N[2000-02-28 23:00:13])\n    \"2000-02-28 23:00:13\"\n    iex> NaiveDateTime.to_string(~N[2000-02-28 23:00:13.001])\n    \"2000-02-28 23:00:13.001\"\n    iex> NaiveDateTime.to_string(~N[-0100-12-15 03:20:31])\n    \"-0100-12-15 03:20:31\"\n\nThis function can also be used to convert a DateTime to a string without\nthe time zone information:\n\n    iex> dt = %DateTime{year: 2000, month: 2, day: 29, zone_abbr: \"CET\",\n    ...>                hour: 23, minute: 0, second: 7, microsecond: {0, 0},\n    ...>                utc_offset: 3600, std_offset: 0, time_zone: \"Europe/Warsaw\"}\n    iex> NaiveDateTime.to_string(dt)\n    \"2000-02-29 23:00:07\"","ref":"NaiveDateTime.html#to_string/1-examples"},{"type":"function","title":"NaiveDateTime.to_time/1","doc":"Converts a `NaiveDateTime` into `Time`.\n\nBecause `Time` does not hold date information,\ndata will be lost during the conversion.","ref":"NaiveDateTime.html#to_time/1"},{"type":"function","title":"Examples - NaiveDateTime.to_time/1","doc":"iex> NaiveDateTime.to_time(~N[2002-01-13 23:00:07])\n    ~T[23:00:07]","ref":"NaiveDateTime.html#to_time/1-examples"},{"type":"function","title":"NaiveDateTime.truncate/2","doc":"Returns the given naive datetime with the microsecond field truncated to the\ngiven precision (`:microsecond`, `:millisecond` or `:second`).\n\nThe given naive datetime is returned unchanged if it already has lower precision\nthan the given precision.","ref":"NaiveDateTime.html#truncate/2"},{"type":"function","title":"Examples - NaiveDateTime.truncate/2","doc":"iex> NaiveDateTime.truncate(~N[2017-11-06 00:23:51.123456], :microsecond)\n    ~N[2017-11-06 00:23:51.123456]\n\n    iex> NaiveDateTime.truncate(~N[2017-11-06 00:23:51.123456], :millisecond)\n    ~N[2017-11-06 00:23:51.123]\n\n    iex> NaiveDateTime.truncate(~N[2017-11-06 00:23:51.123456], :second)\n    ~N[2017-11-06 00:23:51]","ref":"NaiveDateTime.html#truncate/2-examples"},{"type":"function","title":"NaiveDateTime.utc_now/1","doc":"Returns the current naive datetime in UTC.\n\nPrefer using `DateTime.utc_now/0` when possible as, opposite\nto `NaiveDateTime`, it will keep the time zone information.\n\nYou can also provide a time unit to automatically truncate\nthe naive datetime. This is available since v1.15.0.","ref":"NaiveDateTime.html#utc_now/1"},{"type":"function","title":"Examples - NaiveDateTime.utc_now/1","doc":"iex> naive_datetime = NaiveDateTime.utc_now()\n    iex> naive_datetime.year >= 2016\n    true\n\n    iex> naive_datetime = NaiveDateTime.utc_now(:second)\n    iex> naive_datetime.microsecond\n    {0, 0}","ref":"NaiveDateTime.html#utc_now/1-examples"},{"type":"function","title":"NaiveDateTime.utc_now/2","doc":"Returns the current naive datetime in UTC, supporting a specific\ncalendar and precision.\n\nPrefer using `DateTime.utc_now/2` when possible as, opposite\nto `NaiveDateTime`, it will keep the time zone information.","ref":"NaiveDateTime.html#utc_now/2"},{"type":"function","title":"Examples - NaiveDateTime.utc_now/2","doc":"iex> naive_datetime = NaiveDateTime.utc_now(:second, Calendar.ISO)\n    iex> naive_datetime.year >= 2016\n    true\n\n    iex> naive_datetime = NaiveDateTime.utc_now(:second, Calendar.ISO)\n    iex> naive_datetime.microsecond\n    {0, 0}","ref":"NaiveDateTime.html#utc_now/2-examples"},{"type":"type","title":"NaiveDateTime.t/0","doc":"","ref":"NaiveDateTime.html#t:t/0"},{"type":"module","title":"Record","doc":"Module to work with, define, and import records.\n\nRecords are simply tuples where the first element is an atom:\n\n    iex> Record.is_record({User, \"john\", 27})\n    true\n\nThis module provides conveniences for working with records at\ncompilation time, where compile-time field names are used to\nmanipulate the tuples, providing fast operations on top of\nthe tuples' compact structure.\n\nIn Elixir, records are used mostly in two situations:\n\n  1. to work with short, internal data\n  2. to interface with Erlang records\n\nThe macros `defrecord/3` and `defrecordp/3` can be used to create records\nwhile `extract/2` and `extract_all/1` can be used to extract records from\nErlang files.","ref":"Record.html"},{"type":"module","title":"Types - Record","doc":"Types can be defined for tuples with the `record/2` macro (only available in\ntypespecs). This macro will expand to a tuple as seen in the example below:\n\n    defmodule MyModule do\n      require Record\n      Record.defrecord(:user, name: \"john\", age: 25)\n\n      @type user :: record(:user, name: String.t(), age: integer)\n      # expands to: \"@type user :: {:user, String.t(), integer}\"\n    end","ref":"Record.html#module-types"},{"type":"module","title":"Reflection - Record","doc":"A list of all records in a module, if any, can be retrieved by reading the\n`@__records__` module attribute. It returns a list of maps with the record\nkind, name, tag, and fields. The attribute is only available inside the\nmodule definition.","ref":"Record.html#module-reflection"},{"type":"macro","title":"Record.defrecord/3","doc":"Defines a set of macros to create, access, and pattern match\non a record.\n\nThe name of the generated macros will be `name` (which has to be an\natom). `tag` is also an atom and is used as the \"tag\" for the record (i.e.,\nthe first element of the record tuple); by default (if `nil`), it's the same\nas `name`. `kv` is a keyword list of `name: default_value` fields for the\nnew record.\n\nThe following macros are generated:\n\n  * `name/0` to create a new record with default values for all fields\n  * `name/1` to create a new record with the given fields and values,\n    to get the zero-based index of the given field in a record or to\n    convert the given record to a keyword list\n  * `name/2` to update an existing record with the given fields and values\n    or to access a given field in a given record\n\nAll these macros are public macros (as defined by `defmacro`).\n\nSee the \"Examples\" section for examples on how to use these macros.","ref":"Record.html#defrecord/3"},{"type":"macro","title":"Examples - Record.defrecord/3","doc":"defmodule User do\n      require Record\n      Record.defrecord(:user, name: \"meg\", age: \"25\")\n    end\n\nIn the example above, a set of macros named `user` but with different\narities will be defined to manipulate the underlying record.\n\n    # Import the module to make the user macros locally available\n    import User\n\n    # To create records\n    record = user()        #=> {:user, \"meg\", 25}\n    record = user(age: 26) #=> {:user, \"meg\", 26}\n\n    # To get a field from the record\n    user(record, :name) #=> \"meg\"\n\n    # To update the record\n    user(record, age: 26) #=> {:user, \"meg\", 26}\n\n    # To get the zero-based index of the field in record tuple\n    # (index 0 is occupied by the record \"tag\")\n    user(:name) #=> 1\n\n    # Convert a record to a keyword list\n    user(record) #=> [name: \"meg\", age: 26]\n\nThe generated macros can also be used in order to pattern match on records and\nto bind variables during the match:\n\n    record = user() #=> {:user, \"meg\", 25}\n\n    user(name: name) = record\n    name #=> \"meg\"\n\nBy default, Elixir uses the record name as the first element of the tuple (the \"tag\").\nHowever, a different tag can be specified when defining a record,\nas in the following example, in which we use `Customer` as the second argument of `defrecord/3`:\n\n    defmodule User do\n      require Record\n      Record.defrecord(:user, Customer, name: nil)\n    end\n\n    require User\n    User.user() #=> {Customer, nil}","ref":"Record.html#defrecord/3-examples"},{"type":"macro","title":"Defining extracted records with anonymous functions in the values - Record.defrecord/3","doc":"If a record defines an anonymous function in the default values, an\n`ArgumentError` will be raised. This can happen unintentionally when defining\na record after extracting it from an Erlang library that uses anonymous\nfunctions for defaults.\n\n    Record.defrecord(:my_rec, Record.extract(...))\n    ** (ArgumentError) invalid value for record field fun_field,\n        cannot escape #Function .\n\nTo work around this error, redefine the field with your own &M.f/a function,\nlike so:\n\n    defmodule MyRec do\n      require Record\n      Record.defrecord(:my_rec, Record.extract(...) |> Keyword.merge(fun_field: &__MODULE__.foo/2))\n      def foo(bar, baz), do: IO.inspect({bar, baz})\n    end","ref":"Record.html#defrecord/3-defining-extracted-records-with-anonymous-functions-in-the-values"},{"type":"macro","title":"Record.defrecordp/3","doc":"Same as `defrecord/3` but generates private macros.","ref":"Record.html#defrecordp/3"},{"type":"function","title":"Record.extract/2","doc":"Extracts record information from an Erlang file.\n\nReturns a quoted expression containing the fields as a list\nof tuples.\n\n`name`, which is the name of the extracted record, is expected to be an atom\n*at compile time*.","ref":"Record.html#extract/2"},{"type":"function","title":"Options - Record.extract/2","doc":"This function requires one of the following options, which are exclusive to each\nother (i.e., only one of them can be used in the same call):\n\n  * `:from` - (binary representing a path to a file) path to the Erlang file\n    that contains the record definition to extract; with this option, this\n    function uses the same path lookup used by the `-include` attribute used in\n    Erlang modules.\n\n  * `:from_lib` - (binary representing a path to a file) path to the Erlang\n    file that contains the record definition to extract; with this option,\n    this function uses the same path lookup used by the `-include_lib`\n    attribute used in Erlang modules.\n\nIt additionally accepts the following optional, non-exclusive options:\n\n  * `:includes` - (a list of directories as binaries) if the record being\n    extracted depends on relative includes, this option allows developers\n    to specify the directory where those relative includes exist.\n\n  * `:macros` - (keyword list of macro names and values) if the record\n    being extracted depends on the values of macros, this option allows\n    the value of those macros to be set.\n\nThese options are expected to be literals (including the binary values) at\ncompile time.","ref":"Record.html#extract/2-options"},{"type":"function","title":"Examples - Record.extract/2","doc":"iex> Record.extract(:file_info, from_lib: \"kernel/include/file.hrl\")\n    [\n      size: :undefined,\n      type: :undefined,\n      access: :undefined,\n      atime: :undefined,\n      mtime: :undefined,\n      ctime: :undefined,\n      mode: :undefined,\n      links: :undefined,\n      major_device: :undefined,\n      minor_device: :undefined,\n      inode: :undefined,\n      uid: :undefined,\n      gid: :undefined\n    ]","ref":"Record.html#extract/2-examples"},{"type":"function","title":"Record.extract_all/1","doc":"Extracts all records information from an Erlang file.\n\nReturns a keyword list of `{record_name, fields}` tuples where `record_name`\nis the name of an extracted record and `fields` is a list of `{field, value}`\ntuples representing the fields for that record.","ref":"Record.html#extract_all/1"},{"type":"function","title":"Options - Record.extract_all/1","doc":"Accepts the same options as listed for `Record.extract/2`.","ref":"Record.html#extract_all/1-options"},{"type":"macro","title":"Record.is_record/1","doc":"Checks if the given `data` is a record.\n\nThis is implemented as a macro so it can be used in guard clauses.","ref":"Record.html#is_record/1"},{"type":"macro","title":"Examples - Record.is_record/1","doc":"Record.is_record({User, \"john\", 27})\n    #=> true\n\n    Record.is_record({})\n    #=> false","ref":"Record.html#is_record/1-examples"},{"type":"macro","title":"Record.is_record/2","doc":"Checks if the given `data` is a record of kind `kind`.\n\nThis is implemented as a macro so it can be used in guard clauses.","ref":"Record.html#is_record/2"},{"type":"macro","title":"Examples - Record.is_record/2","doc":"iex> record = {User, \"john\", 27}\n    iex> Record.is_record(record, User)\n    true","ref":"Record.html#is_record/2-examples"},{"type":"module","title":"Regex","doc":"Provides regular expressions for Elixir.\n\nRegex is based on PCRE (Perl Compatible Regular Expressions) and\nbuilt on top of Erlang's `:re` module. More information can be found\nin the [`:re` module documentation](`:re`).\n\nRegular expressions in Elixir can be created using the sigils\n`~r` (see `sigil_r/2`) or `~R` (see `sigil_R/2`):\n\n    # A simple regular expression that matches foo anywhere in the string\n    ~r/foo/\n\n    # A regular expression with case insensitive and Unicode options\n    ~r/foo/iu\n\nRegular expressions created via sigils are pre-compiled and stored\nin the `.beam` file. Note that this may be a problem if you are precompiling\nElixir, see the \"Precompilation\" section for more information.\n\nA Regex is represented internally as the `Regex` struct. Therefore,\n`%Regex{}` can be used whenever there is a need to match on them.\nKeep in mind that all of the structs fields are private. There is\nalso not guarantee two regular expressions from the same source are\nequal, for example:\n\n    ~r/(? .)(? .)/ == ~r/(? .)(? .)/\n\nmay return `true` or `false` depending on your machine, endianness,\navailable optimizations and others. You can, however, retrieve the source\nof a compiled regular expression by accessing the `source` field, and then\ncompare those directly:\n\n    ~r/(? .)(? .)/.source == ~r/(? .)(? .)/.source","ref":"Regex.html"},{"type":"module","title":"Modifiers - Regex","doc":"The modifiers available when creating a Regex are:\n\n  * `:unicode` (u) - enables Unicode specific patterns like `\\p` and causes\n    character classes like `\\w`, `\\W`, `\\s`, and the like to also match on Unicode\n    (see examples below in \"Character classes\"). It expects valid Unicode\n    strings to be given on match\n\n  * `:caseless` (i) - adds case insensitivity\n\n  * `:dotall` (s) - causes dot to match newlines and also set newline to\n    anycrlf; the new line setting can be overridden by setting `(*CR)` or\n    `(*LF)` or `(*CRLF)` or `(*ANY)` according to `:re` documentation\n\n  * `:multiline` (m) - causes `^` and `$` to mark the beginning and end of\n    each line; use `\\A` and `\\z` to match the end or beginning of the string\n\n  * `:extended` (x) - whitespace characters are ignored except when escaped\n    and allow `#` to delimit comments\n\n  * `:firstline` (f) - forces the unanchored pattern to match before or at the\n    first newline, though the matched text may continue over the newline\n\n  * `:ungreedy` (U) - inverts the \"greediness\" of the regexp\n    (the previous `r` option is deprecated in favor of `U`)\n\nThe options not available are:\n\n  * `:anchored` - not available, use `^` or `\\A` instead\n  * `:dollar_endonly` - not available, use `\\z` instead\n  * `:no_auto_capture` - not available, use `?:` instead\n  * `:newline` - not available, use `(*CR)` or `(*LF)` or `(*CRLF)` or\n    `(*ANYCRLF)` or `(*ANY)` at the beginning of the regexp according to the\n    `:re` documentation","ref":"Regex.html#module-modifiers"},{"type":"module","title":"Captures - Regex","doc":"Many functions in this module handle what to capture in a regex\nmatch via the `:capture` option. The supported values are:\n\n  * `:all` - all captured subpatterns including the complete matching string\n    (this is the default)\n\n  * `:first` - only the first captured subpattern, which is always the\n    complete matching part of the string; all explicitly captured subpatterns\n    are discarded\n\n  * `:all_but_first` - all but the first matching subpattern, i.e. all\n    explicitly captured subpatterns, but not the complete matching part of\n    the string\n\n  * `:none` - does not return matching subpatterns at all\n\n  * `:all_names` - captures all named subpattern matches in the Regex as a list\n    ordered **alphabetically** by the names of the subpatterns\n\n  * `list(binary)` - a list of named captures to capture","ref":"Regex.html#module-captures"},{"type":"module","title":"Character classes - Regex","doc":"Regex supports several built in named character classes. These are used by\nenclosing the class name in `[: :]` inside a group. For example:\n\n    iex> String.match?(\"123\", ~r/^[[:alnum:]]+$/)\n    true\n    iex> String.match?(\"123 456\", ~r/^[[:alnum:][:blank:]]+$/)\n    true\n\nThe supported class names are:\n\n  * alnum - Letters and digits\n  * alpha - Letters\n  * blank - Space or tab only\n  * cntrl - Control characters\n  * digit - Decimal digits (same as \\\\d)\n  * graph - Printing characters, excluding space\n  * lower - Lowercase letters\n  * print - Printing characters, including space\n  * punct - Printing characters, excluding letters, digits, and space\n  * space - Whitespace (the same as \\s from PCRE 8.34)\n  * upper - Uppercase letters\n  * word  - \"Word\" characters (same as \\w)\n  * xdigit - Hexadecimal digits\n\nThere is another character class, `ascii`, that erroneously matches\nLatin-1 characters instead of the 0-127 range specified by POSIX. This\ncannot be fixed without altering the behaviour of other classes, so we\nrecommend matching the range with `[\\\\0-\\x7f]` instead.\n\nNote the behaviour of those classes may change according to the Unicode\nand other modifiers:\n\n    iex> String.match?(\"josé\", ~r/^[[:lower:]]+$/)\n    false\n    iex> String.match?(\"josé\", ~r/^[[:lower:]]+$/u)\n    true\n    iex> Regex.replace(~r/\\s/, \"Unicode\\u00A0spaces\", \"-\")\n    \"Unicode spaces\"\n    iex> Regex.replace(~r/\\s/u, \"Unicode\\u00A0spaces\", \"-\")\n    \"Unicode-spaces\"","ref":"Regex.html#module-character-classes"},{"type":"module","title":"Precompilation - Regex","doc":"Regular expressions built with sigil are precompiled and stored in `.beam`\nfiles. Precompiled regexes will be checked in runtime and may work slower\nbetween operating systems and OTP releases. This is rarely a problem, as most Elixir code\nshared during development is compiled on the target (such as dependencies,\narchives, and escripts) and, when running in production, the code must either\nbe compiled on the target (via `mix compile` or similar) or released on the\nhost (via `mix releases` or similar) with a matching OTP, operating system\nand architecture as the target.\n\nIf you know you are running on a different system than the current one and\nyou are doing multiple matches with the regex, you can manually invoke\n`Regex.recompile/1` or `Regex.recompile!/1` to perform a runtime version\ncheck and recompile the regex if necessary.","ref":"Regex.html#module-precompilation"},{"type":"function","title":"Regex.compile/2","doc":"Compiles the regular expression.\n\nThe given options can either be a binary with the characters\nrepresenting the same regex options given to the\n`~r` (see `sigil_r/2`) sigil, or a list of options, as\nexpected by the Erlang's [`:re`](`:re`) module.\n\nIt returns `{:ok, regex}` in case of success,\n`{:error, reason}` otherwise.","ref":"Regex.html#compile/2"},{"type":"function","title":"Examples - Regex.compile/2","doc":"iex> Regex.compile(\"foo\")\n    {:ok, ~r/foo/}\n\n    iex> Regex.compile(\"*foo\")\n    {:error, {'nothing to repeat', 0}}\n\n    iex> Regex.compile(\"foo\", \"i\")\n    {:ok, ~r/foo/i}\n\n    iex> Regex.compile(\"foo\", [:caseless])\n    {:ok, Regex.compile!(\"foo\", [:caseless])}","ref":"Regex.html#compile/2-examples"},{"type":"function","title":"Regex.compile!/2","doc":"Compiles the regular expression and raises `Regex.CompileError` in case of errors.","ref":"Regex.html#compile!/2"},{"type":"function","title":"Regex.escape/1","doc":"Escapes a string to be literally matched in a regex.","ref":"Regex.html#escape/1"},{"type":"function","title":"Examples - Regex.escape/1","doc":"iex> Regex.escape(\".\")\n    \"\\\\.\"\n\n    iex> Regex.escape(\"\\\\what if\")\n    \"\\\\\\\\what\\\\ if\"","ref":"Regex.html#escape/1-examples"},{"type":"function","title":"Regex.match?/2","doc":"Returns a boolean indicating whether there was a match or not.","ref":"Regex.html#match?/2"},{"type":"function","title":"Examples - Regex.match?/2","doc":"iex> Regex.match?(~r/foo/, \"foo\")\n    true\n\n    iex> Regex.match?(~r/foo/, \"bar\")\n    false\n\nElixir also provides text-based match operator `=~/2` and function `String.match?/2` as\nan alternative to test strings against regular expressions and\nstrings.","ref":"Regex.html#match?/2-examples"},{"type":"function","title":"Regex.named_captures/3","doc":"Returns the given captures as a map or `nil` if no captures are found.","ref":"Regex.html#named_captures/3"},{"type":"function","title":"Options - Regex.named_captures/3","doc":"* `:return` - when set to `:index`, returns byte index and match length.\n    Defaults to `:binary`.","ref":"Regex.html#named_captures/3-options"},{"type":"function","title":"Examples - Regex.named_captures/3","doc":"iex> Regex.named_captures(~r/c(? d)/, \"abcd\")\n    %{\"foo\" => \"d\"}\n\n    iex> Regex.named_captures(~r/a(? b)c(? d)/, \"abcd\")\n    %{\"bar\" => \"d\", \"foo\" => \"b\"}\n\n    iex> Regex.named_captures(~r/a(? b)c(? d)/, \"efgh\")\n    nil","ref":"Regex.html#named_captures/3-examples"},{"type":"function","title":"Regex.names/1","doc":"Returns a list of names in the regex.","ref":"Regex.html#names/1"},{"type":"function","title":"Examples - Regex.names/1","doc":"iex> Regex.names(~r/(? bar)/)\n    [\"foo\"]","ref":"Regex.html#names/1-examples"},{"type":"function","title":"Regex.opts/1","doc":"Returns the regex options, as a string or list depending on how\nit was compiled.\n\nSee the documentation of `Regex.compile/2` for more information.","ref":"Regex.html#opts/1"},{"type":"function","title":"Examples - Regex.opts/1","doc":"iex> Regex.opts(~r/foo/m)\n    \"m\"\n\n    iex> Regex.opts(Regex.compile!(\"foo\", [:caseless]))\n    [:caseless]","ref":"Regex.html#opts/1-examples"},{"type":"function","title":"Regex.re_pattern/1","doc":"Returns the underlying `re_pattern` in the regular expression.","ref":"Regex.html#re_pattern/1"},{"type":"function","title":"Regex.recompile/1","doc":"Recompiles the existing regular expression if necessary.\n\nThis checks the version stored in the regular expression\nand recompiles the regex in case of version mismatch.","ref":"Regex.html#recompile/1"},{"type":"function","title":"Regex.recompile!/1","doc":"Recompiles the existing regular expression and raises `Regex.CompileError` in case of errors.","ref":"Regex.html#recompile!/1"},{"type":"function","title":"Regex.replace/4","doc":"Receives a regex, a binary and a replacement, returns a new\nbinary where all matches are replaced by the replacement.\n\nThe replacement can be either a string or a function that returns a string.\nThe resulting string is used as a replacement for every match.\n\nWhen the replacement is a string, it allows specific captures of the match\nusing brackets at the regex expression and accessing them in the replacement\nvia `\\N` or `\\g{N}`, where `N` is the number of the capture. In case `\\0` is\nused, the whole match is inserted. Note that in regexes the backslash needs\nto be escaped, hence in practice you'll need to use `\\\\N` and `\\\\g{N}`.\n\nWhen the replacement is a function, it allows specific captures too.\nThe function may have arity N where each argument maps to a capture,\nwith the first argument being the whole match. If the function expects more\narguments than captures found, the remaining arguments will receive `\"\"`.","ref":"Regex.html#replace/4"},{"type":"function","title":"Options - Regex.replace/4","doc":"* `:global` - when `false`, replaces only the first occurrence\n    (defaults to `true`)","ref":"Regex.html#replace/4-options"},{"type":"function","title":"Examples - Regex.replace/4","doc":"iex> Regex.replace(~r/d/, \"abc\", \"d\")\n    \"abc\"\n\n    iex> Regex.replace(~r/b/, \"abc\", \"d\")\n    \"adc\"\n\n    iex> Regex.replace(~r/b/, \"abc\", \"[\\\\0]\")\n    \"a[b]c\"\n\n    iex> Regex.replace(~r/a(b|d)c/, \"abcadc\", \"[\\\\1]\")\n    \"[b][d]\"\n\n    iex> Regex.replace(~r/\\.(\\d)$/, \"500.5\", \".\\\\g{1}0\")\n    \"500.50\"\n\n    iex> Regex.replace(~r/a(b|d)c/, \"abcadc\", fn _, x -> \"[#{x}]\" end)\n    \"[b][d]\"\n\n    iex> Regex.replace(~r/(\\w+)@(\\w+).(\\w+)/, \"abc@def.com\", fn _full, _c1, _c2, c3 -> \"TLD: #{c3}\" end)\n    \"TLD: com\"\n\n    iex> Regex.replace(~r/a/, \"abcadc\", \"A\", global: false)\n    \"Abcadc\"","ref":"Regex.html#replace/4-examples"},{"type":"function","title":"Regex.run/3","doc":"Runs the regular expression against the given string until the first match.\nIt returns a list with all captures or `nil` if no match occurred.","ref":"Regex.html#run/3"},{"type":"function","title":"Options - Regex.run/3","doc":"* `:return` - when set to `:index`, returns byte index and match length.\n    Defaults to `:binary`.\n  * `:capture` - what to capture in the result. Check the moduledoc for `Regex`\n    to see the possible capture values.\n  * `:offset` - (since v1.12.0) specifies the starting offset to match in the given string.\n    Defaults to zero.","ref":"Regex.html#run/3-options"},{"type":"function","title":"Examples - Regex.run/3","doc":"iex> Regex.run(~r/c(d)/, \"abcd\")\n    [\"cd\", \"d\"]\n\n    iex> Regex.run(~r/e/, \"abcd\")\n    nil\n\n    iex> Regex.run(~r/c(d)/, \"abcd\", return: :index)\n    [{2, 2}, {3, 1}]","ref":"Regex.html#run/3-examples"},{"type":"function","title":"Regex.scan/3","doc":"Same as `run/3`, but scans the target several times collecting all\nmatches of the regular expression.\n\nA list of lists is returned, where each entry in the primary list represents a\nmatch and each entry in the secondary list represents the captured contents.","ref":"Regex.html#scan/3"},{"type":"function","title":"Options - Regex.scan/3","doc":"* `:return` - when set to `:index`, returns byte index and match length.\n    Defaults to `:binary`.\n  * `:capture` - what to capture in the result. Check the moduledoc for `Regex`\n    to see the possible capture values.\n  * `:offset` - (since v1.12.0) specifies the starting offset to match in the given string.\n    Defaults to zero.","ref":"Regex.html#scan/3-options"},{"type":"function","title":"Examples - Regex.scan/3","doc":"iex> Regex.scan(~r/c(d|e)/, \"abcd abce\")\n    [[\"cd\", \"d\"], [\"ce\", \"e\"]]\n\n    iex> Regex.scan(~r/c(?:d|e)/, \"abcd abce\")\n    [[\"cd\"], [\"ce\"]]\n\n    iex> Regex.scan(~r/e/, \"abcd\")\n    []\n\n    iex> Regex.scan(~r/\\p{Sc}/u, \"$, £, and €\")\n    [[\"$\"], [\"£\"], [\"€\"]]\n\n    iex> Regex.scan(~r/=+/, \"=ü†ƒ8===\", return: :index)\n    [[{0, 1}], [{9, 3}]]","ref":"Regex.html#scan/3-examples"},{"type":"function","title":"Regex.source/1","doc":"Returns the regex source as a binary.","ref":"Regex.html#source/1"},{"type":"function","title":"Examples - Regex.source/1","doc":"iex> Regex.source(~r/foo/)\n    \"foo\"","ref":"Regex.html#source/1-examples"},{"type":"function","title":"Regex.split/3","doc":"Splits the given target based on the given pattern and in the given number of\nparts.","ref":"Regex.html#split/3"},{"type":"function","title":"Options - Regex.split/3","doc":"* `:parts` - when specified, splits the string into the given number of\n    parts. If not specified, `:parts` defaults to `:infinity`, which will\n    split the string into the maximum number of parts possible based on the\n    given pattern.\n\n  * `:trim` - when `true`, removes empty strings (`\"\"`) from the result.\n    Defaults to `false`.\n\n  * `:on` - specifies which captures to split the string on, and in what\n    order. Defaults to `:first` which means captures inside the regex do not\n    affect the splitting process.\n\n  * `:include_captures` - when `true`, includes in the result the matches of\n    the regular expression. The matches are not counted towards the maximum\n    number of parts if combined with the `:parts` option. Defaults to `false`.","ref":"Regex.html#split/3-options"},{"type":"function","title":"Examples - Regex.split/3","doc":"iex> Regex.split(~r{-}, \"a-b-c\")\n    [\"a\", \"b\", \"c\"]\n\n    iex> Regex.split(~r{-}, \"a-b-c\", parts: 2)\n    [\"a\", \"b-c\"]\n\n    iex> Regex.split(~r{-}, \"abc\")\n    [\"abc\"]\n\n    iex> Regex.split(~r{}, \"abc\")\n    [\"\", \"a\", \"b\", \"c\", \"\"]\n\n    iex> Regex.split(~r{a(? b)c}, \"abc\")\n    [\"\", \"\"]\n\n    iex> Regex.split(~r{a(? b)c}, \"abc\", on: [:second])\n    [\"a\", \"c\"]\n\n    iex> Regex.split(~r{(x)}, \"Elixir\", include_captures: true)\n    [\"Eli\", \"x\", \"ir\"]\n\n    iex> Regex.split(~r{a(? b)c}, \"abc\", on: [:second], include_captures: true)\n    [\"a\", \"b\", \"c\"]","ref":"Regex.html#split/3-examples"},{"type":"function","title":"Regex.version/0","doc":"Returns the version of the underlying Regex engine.","ref":"Regex.html#version/0"},{"type":"type","title":"Regex.t/0","doc":"","ref":"Regex.html#t:t/0"},{"type":"module","title":"String","doc":"Strings in Elixir are UTF-8 encoded binaries.\n\nStrings in Elixir are a sequence of Unicode characters,\ntypically written between double quoted strings, such\nas `\"hello\"` and `\"héllò\"`.\n\nIn case a string must have a double-quote in itself,\nthe double quotes must be escaped with a backslash,\nfor example: `\"this is a string with \\\"double quotes\\\"\"`.\n\nYou can concatenate two strings with the ` /2` operator:\n\n    iex> \"hello\"   \" \"   \"world\"\n    \"hello world\"\n\nThe functions in this module act according to\n[The Unicode Standard, Version 15.0.0](http://www.unicode.org/versions/Unicode15.0.0/).","ref":"String.html"},{"type":"module","title":"Interpolation - String","doc":"Strings in Elixir also support interpolation. This allows\nyou to place some value in the middle of a string by using\nthe `#{}` syntax:\n\n    iex> name = \"joe\"\n    iex> \"hello #{name}\"\n    \"hello joe\"\n\nAny Elixir expression is valid inside the interpolation.\nIf a string is given, the string is interpolated as is.\nIf any other value is given, Elixir will attempt to convert\nit to a string using the `String.Chars` protocol. This\nallows, for example, to output an integer from the interpolation:\n\n    iex> \"2 + 2 = #{2 + 2}\"\n    \"2 + 2 = 4\"\n\nIn case the value you want to interpolate cannot be\nconverted to a string, because it doesn't have a human\ntextual representation, a protocol error will be raised.","ref":"String.html#module-interpolation"},{"type":"module","title":"Escape characters - String","doc":"Besides allowing double-quotes to be escaped with a backslash,\nstrings also support the following escape characters:\n\n  * `\\0` - Null byte\n  * `\\a` - Bell\n  * `\\b` - Backspace\n  * `\\t` - Horizontal tab\n  * `\\n` - Line feed (New lines)\n  * `\\v` - Vertical tab\n  * `\\f` - Form feed\n  * `\\r` - Carriage return\n  * `\\e` - Command Escape\n  * `\\s` - Space\n  * `\\#` - Returns the `#` character itself, skipping interpolation\n  * `\\\\` - Single backslash\n  * `\\xNN` - A byte represented by the hexadecimal `NN`\n  * `\\uNNNN` - A Unicode code point represented by `NNNN`\n  * `\\u{NNNNNN}` - A Unicode code point represented by `NNNNNN`\n\nNote it is generally not advised to use `\\xNN` in Elixir\nstrings, as introducing an invalid byte sequence would\nmake the string invalid. If you have to introduce a\ncharacter by its hexadecimal representation, it is best\nto work with Unicode code points, such as `\\uNNNN`. In fact,\nunderstanding Unicode code points can be essential when doing\nlow-level manipulations of string, so let's explore them in\ndetail next.","ref":"String.html#module-escape-characters"},{"type":"module","title":"Unicode and code points - String","doc":"In order to facilitate meaningful communication between computers\nacross multiple languages, a standard is required so that the ones\nand zeros on one machine mean the same thing when they are transmitted\nto another. The Unicode Standard acts as an official registry of\nvirtually all the characters we know: this includes characters from\nclassical and historical texts, emoji, and formatting and control\ncharacters as well.\n\nUnicode organizes all of the characters in its repertoire into code\ncharts, and each character is given a unique numerical index. This\nnumerical index is known as a Code Point.\n\nIn Elixir you can use a `?` in front of a character literal to reveal\nits code point:\n\n    iex> ?a\n    97\n    iex> ?ł\n    322\n\nNote that most Unicode code charts will refer to a code point by its\nhexadecimal (hex) representation, e.g. `97` translates to `0061` in hex,\nand we can represent any Unicode character in an Elixir string by\nusing the `\\u` escape character followed by its code point number:\n\n    iex> \"\\u0061\" === \"a\"\n    true\n    iex> 0x0061 = 97 = ?a\n    97\n\nThe hex representation will also help you look up information about a\ncode point, e.g. [https://codepoints.net/U+0061](https://codepoints.net/U+0061)\nhas a data sheet all about the lower case `a`, a.k.a. code point 97.\nRemember you can get the hex presentation of a number by calling\n`Integer.to_string/2`:\n\n    iex> Integer.to_string(?a, 16)\n    \"61\"","ref":"String.html#module-unicode-and-code-points"},{"type":"module","title":"UTF-8 encoded and encodings - String","doc":"Now that we understand what the Unicode standard is and what code points\nare, we can finally talk about encodings. Whereas the code point is **what**\nwe store, an encoding deals with **how** we store it: encoding is an\nimplementation. In other words, we need a mechanism to convert the code\npoint numbers into bytes so they can be stored in memory, written to disk, and such.\n\nElixir uses UTF-8 to encode its strings, which means that code points are\nencoded as a series of 8-bit bytes. UTF-8 is a **variable width** character\nencoding that uses one to four bytes to store each code point. It is capable\nof encoding all valid Unicode code points. Let's see an example:\n\n    iex> string = \"héllo\"\n    \"héllo\"\n    iex> String.length(string)\n    5\n    iex> byte_size(string)\n    6\n\nAlthough the string above has 5 characters, it uses 6 bytes, as two bytes\nare used to represent the character `é`.","ref":"String.html#module-utf-8-encoded-and-encodings"},{"type":"module","title":"Grapheme clusters - String","doc":"This module also works with the concept of grapheme cluster\n(from now on referenced as graphemes). Graphemes can consist\nof multiple code points that may be perceived as a single character\nby readers. For example, \"é\" can be represented either as a single\n\"e with acute\" code point, as seen above in the string `\"héllo\"`,\nor as the letter \"e\" followed by a \"combining acute accent\"\n(two code points):\n\n    iex> string = \"\\u0065\\u0301\"\n    \"é\"\n    iex> byte_size(string)\n    3\n    iex> String.length(string)\n    1\n    iex> String.codepoints(string)\n    [\"e\", \"́\"]\n    iex> String.graphemes(string)\n    [\"é\"]\n\nAlthough it looks visually the same as before, the example above\nis made of two characters, it is perceived by users as one.\n\nGraphemes can also be two characters that are interpreted as one\nby some languages. For example, some languages may consider \"ch\"\nas a single character. However, since this information depends on\nthe locale, it is not taken into account by this module.\n\nIn general, the functions in this module rely on the Unicode\nStandard, but do not contain any of the locale specific behaviour.\nMore information about graphemes can be found in the [Unicode\nStandard Annex #29](https://www.unicode.org/reports/tr29/).\n\nFor converting a binary to a different encoding and for Unicode\nnormalization mechanisms, see Erlang's `:unicode` module.","ref":"String.html#module-grapheme-clusters"},{"type":"module","title":"String and binary operations - String","doc":"To act according to the Unicode Standard, many functions\nin this module run in linear time, as they need to traverse\nthe whole string considering the proper Unicode code points.\n\nFor example, `String.length/1` will take longer as\nthe input grows. On the other hand, `Kernel.byte_size/1` always runs\nin constant time (i.e. regardless of the input size).\n\nThis means often there are performance costs in using the\nfunctions in this module, compared to the more low-level\noperations that work directly with binaries:\n\n  * `Kernel.binary_part/3` - retrieves part of the binary\n  * `Kernel.bit_size/1` and `Kernel.byte_size/1` - size related functions\n  * `Kernel.is_bitstring/1` and `Kernel.is_binary/1` - type-check function\n  * Plus a number of functions for working with binaries (bytes)\n    in the [`:binary` module](`:binary`)\n\nA `utf8` modifier is also available inside the binary syntax ` >`.\nIt can be used to match code points out of a binary/string:\n\n    iex>  > = \"é\"\n    iex> eacute\n    233\n\nYou can also fully convert a string into a list of integer code points,\nknown as \"charlists\" in Elixir, by calling `String.to_charlist/1`:\n\n    iex> String.to_charlist(\"héllo\")\n    [104, 233, 108, 108, 111]\n\nIf you would rather see the underlying bytes of a string, instead of\nits codepoints, a common trick is to concatenate the null byte ` >`\nto it:\n\n    iex> \"héllo\"    >\n     >\n\nAlternatively, you can view a string's binary representation by\npassing an option to `IO.inspect/2`:\n\n    IO.inspect(\"héllo\", binaries: :as_binaries)\n    #=>  >","ref":"String.html#module-string-and-binary-operations"},{"type":"module","title":"Self-synchronization - String","doc":"The UTF-8 encoding is self-synchronizing. This means that\nif malformed data (i.e., data that is not possible according\nto the definition of the encoding) is encountered, only one\ncode point needs to be rejected.\n\nThis module relies on this behaviour to ignore such invalid\ncharacters. For example, `length/1` will return\na correct result even if an invalid code point is fed into it.\n\nIn other words, this module expects invalid data to be detected\nelsewhere, usually when retrieving data from the external source.\nFor example, a driver that reads strings from a database will be\nresponsible to check the validity of the encoding. `String.chunk/2`\ncan be used for breaking a string into valid and invalid parts.","ref":"String.html#module-self-synchronization"},{"type":"module","title":"Compile binary patterns - String","doc":"Many functions in this module work with patterns. For example,\n`String.split/3` can split a string into multiple strings given\na pattern. This pattern can be a string, a list of strings or\na compiled pattern:\n\n    iex> String.split(\"foo bar\", \" \")\n    [\"foo\", \"bar\"]\n\n    iex> String.split(\"foo bar!\", [\" \", \"!\"])\n    [\"foo\", \"bar\", \"\"]\n\n    iex> pattern = :binary.compile_pattern([\" \", \"!\"])\n    iex> String.split(\"foo bar!\", pattern)\n    [\"foo\", \"bar\", \"\"]\n\nThe compiled pattern is useful when the same match will\nbe done over and over again. Note though that the compiled\npattern cannot be stored in a module attribute as the pattern\nis generated at runtime and does not survive compile time.","ref":"String.html#module-compile-binary-patterns"},{"type":"function","title":"String.at/2","doc":"Returns the grapheme at the `position` of the given UTF-8 `string`.\nIf `position` is greater than `string` length, then it returns `nil`.","ref":"String.html#at/2"},{"type":"function","title":"Examples - String.at/2","doc":"iex> String.at(\"elixir\", 0)\n    \"e\"\n\n    iex> String.at(\"elixir\", 1)\n    \"l\"\n\n    iex> String.at(\"elixir\", 10)\n    nil\n\n    iex> String.at(\"elixir\", -1)\n    \"r\"\n\n    iex> String.at(\"elixir\", -10)\n    nil","ref":"String.html#at/2-examples"},{"type":"function","title":"String.bag_distance/2","doc":"Computes the bag distance between two strings.\n\nReturns a float value between 0 and 1 representing the bag\ndistance between `string1` and `string2`.\n\nThe bag distance is meant to be an efficient approximation\nof the distance between two strings to quickly rule out strings\nthat are largely different.\n\nThe algorithm is outlined in the \"String Matching with Metric\nTrees Using an Approximate Distance\" paper by Ilaria Bartolini,\nPaolo Ciaccia, and Marco Patella.","ref":"String.html#bag_distance/2"},{"type":"function","title":"Examples - String.bag_distance/2","doc":"iex> String.bag_distance(\"abc\", \"\")\n    0.0\n    iex> String.bag_distance(\"abcd\", \"a\")\n    0.25\n    iex> String.bag_distance(\"abcd\", \"ab\")\n    0.5\n    iex> String.bag_distance(\"abcd\", \"abc\")\n    0.75\n    iex> String.bag_distance(\"abcd\", \"abcd\")\n    1.0","ref":"String.html#bag_distance/2-examples"},{"type":"function","title":"String.capitalize/2","doc":"Converts the first character in the given string to\nuppercase and the remainder to lowercase according to `mode`.\n\n`mode` may be `:default`, `:ascii`, `:greek` or `:turkic`. The `:default` mode considers\nall non-conditional transformations outlined in the Unicode standard. `:ascii`\ncapitalizes only the letters A to Z. `:greek` includes the context sensitive\nmappings found in Greek. `:turkic` properly handles the letter i with the dotless variant.","ref":"String.html#capitalize/2"},{"type":"function","title":"Examples - String.capitalize/2","doc":"iex> String.capitalize(\"abcd\")\n    \"Abcd\"\n\n    iex> String.capitalize(\"ﬁn\")\n    \"Fin\"\n\n    iex> String.capitalize(\"olá\")\n    \"Olá\"","ref":"String.html#capitalize/2-examples"},{"type":"function","title":"String.chunk/2","doc":"Splits the string into chunks of characters that share a common trait.\n\nThe trait can be one of two options:\n\n  * `:valid` - the string is split into chunks of valid and invalid\n    character sequences\n\n  * `:printable` - the string is split into chunks of printable and\n    non-printable character sequences\n\nReturns a list of binaries each of which contains only one kind of\ncharacters.\n\nIf the given string is empty, an empty list is returned.","ref":"String.html#chunk/2"},{"type":"function","title":"Examples - String.chunk/2","doc":"iex> String.chunk( >, :valid)\n    [\"abc\\0\"]\n\n    iex> String.chunk( >, :valid)\n    [\"abc\\0\",  >]\n\n    iex> String.chunk( >, :printable)\n    [\"abc\",  >]","ref":"String.html#chunk/2-examples"},{"type":"function","title":"String.codepoints/1","doc":"Returns a list of code points encoded as strings.\n\nTo retrieve code points in their natural integer\nrepresentation, see `to_charlist/1`. For details about\ncode points and graphemes, see the `String` module\ndocumentation.","ref":"String.html#codepoints/1"},{"type":"function","title":"Examples - String.codepoints/1","doc":"iex> String.codepoints(\"olá\")\n    [\"o\", \"l\", \"á\"]\n\n    iex> String.codepoints(\"оптими зации\")\n    [\"о\", \"п\", \"т\", \"и\", \"м\", \"и\", \" \", \"з\", \"а\", \"ц\", \"и\", \"и\"]\n\n    iex> String.codepoints(\"ἅἪῼ\")\n    [\"ἅ\", \"Ἢ\", \"ῼ\"]\n\n    iex> String.codepoints(\"\\u00e9\")\n    [\"é\"]\n\n    iex> String.codepoints(\"\\u0065\\u0301\")\n    [\"e\", \"́\"]","ref":"String.html#codepoints/1-examples"},{"type":"function","title":"String.contains?/2","doc":"Searches if `string` contains any of the given `contents`.\n\n`contents` can be either a string, a list of strings,\nor a compiled pattern. If `contents` is a list, this\nfunction will search if any of the strings in `contents`\nare part of `string`.\n\n> ##","ref":"String.html#contains?/2"},{"type":"function","title":"Searching for a string in a list {: .tip} - String.contains?/2","doc":">\n> If you want to check if `string` is listed in `contents`,\n> where `contents` is a list, use `Enum.member?(contents, string)`\n> instead.","ref":"String.html#contains?/2-searching-for-a-string-in-a-list-tip"},{"type":"function","title":"Examples - String.contains?/2","doc":"iex> String.contains?(\"elixir of life\", \"of\")\n    true\n    iex> String.contains?(\"elixir of life\", [\"life\", \"death\"])\n    true\n    iex> String.contains?(\"elixir of life\", [\"death\", \"mercury\"])\n    false\n\nThe argument can also be a compiled pattern:\n\n    iex> pattern = :binary.compile_pattern([\"life\", \"death\"])\n    iex> String.contains?(\"elixir of life\", pattern)\n    true\n\nAn empty string will always match:\n\n    iex> String.contains?(\"elixir of life\", \"\")\n    true\n    iex> String.contains?(\"elixir of life\", [\"\", \"other\"])\n    true\n\nAn empty list will never match:\n\n    iex> String.contains?(\"elixir of life\", [])\n    false\n\n    iex> String.contains?(\"\", [])\n    false\n\nBe aware that this function can match within or across grapheme boundaries.\nFor example, take the grapheme \"é\" which is made of the characters\n\"e\" and the acute accent. The following returns `true`:\n\n    iex> String.contains?(String.normalize(\"é\", :nfd), \"e\")\n    true\n\nHowever, if \"é\" is represented by the single character \"e with acute\"\naccent, then it will return `false`:\n\n    iex> String.contains?(String.normalize(\"é\", :nfc), \"e\")\n    false","ref":"String.html#contains?/2-examples"},{"type":"function","title":"String.downcase/2","doc":"Converts all characters in the given string to lowercase according to `mode`.\n\n`mode` may be `:default`, `:ascii`, `:greek` or `:turkic`. The `:default` mode considers\nall non-conditional transformations outlined in the Unicode standard. `:ascii`\nlowercases only the letters A to Z. `:greek` includes the context sensitive\nmappings found in Greek. `:turkic` properly handles the letter i with the dotless variant.","ref":"String.html#downcase/2"},{"type":"function","title":"Examples - String.downcase/2","doc":"iex> String.downcase(\"ABCD\")\n    \"abcd\"\n\n    iex> String.downcase(\"AB 123 XPTO\")\n    \"ab 123 xpto\"\n\n    iex> String.downcase(\"OLÁ\")\n    \"olá\"\n\nThe `:ascii` mode ignores Unicode characters and provides a more\nperformant implementation when you know the string contains only\nASCII characters:\n\n    iex> String.downcase(\"OLÁ\", :ascii)\n    \"olÁ\"\n\nThe `:greek` mode properly handles the context sensitive sigma in Greek:\n\n    iex> String.downcase(\"ΣΣ\")\n    \"σσ\"\n\n    iex> String.downcase(\"ΣΣ\", :greek)\n    \"σς\"\n\nAnd `:turkic` properly handles the letter i with the dotless variant:\n\n    iex> String.downcase(\"Iİ\")\n    \"ii̇\"\n\n    iex> String.downcase(\"Iİ\", :turkic)\n    \"ıi\"","ref":"String.html#downcase/2-examples"},{"type":"function","title":"String.duplicate/2","doc":"Returns a string `subject` repeated `n` times.\n\nInlined by the compiler.","ref":"String.html#duplicate/2"},{"type":"function","title":"Examples - String.duplicate/2","doc":"iex> String.duplicate(\"abc\", 0)\n    \"\"\n\n    iex> String.duplicate(\"abc\", 1)\n    \"abc\"\n\n    iex> String.duplicate(\"abc\", 2)\n    \"abcabc\"","ref":"String.html#duplicate/2-examples"},{"type":"function","title":"String.ends_with?/2","doc":"Returns `true` if `string` ends with any of the suffixes given.\n\n`suffixes` can be either a single suffix or a list of suffixes.","ref":"String.html#ends_with?/2"},{"type":"function","title":"Examples - String.ends_with?/2","doc":"iex> String.ends_with?(\"language\", \"age\")\n    true\n    iex> String.ends_with?(\"language\", [\"youth\", \"age\"])\n    true\n    iex> String.ends_with?(\"language\", [\"youth\", \"elixir\"])\n    false\n\nAn empty suffix will always match:\n\n    iex> String.ends_with?(\"language\", \"\")\n    true\n    iex> String.ends_with?(\"language\", [\"\", \"other\"])\n    true","ref":"String.html#ends_with?/2-examples"},{"type":"function","title":"String.equivalent?/2","doc":"Returns `true` if `string1` is canonically equivalent to `string2`.\n\nIt performs Normalization Form Canonical Decomposition (NFD) on the\nstrings before comparing them. This function is equivalent to:\n\n    String.normalize(string1, :nfd) == String.normalize(string2, :nfd)\n\nIf you plan to compare multiple strings, multiple times in a row, you\nmay normalize them upfront and compare them directly to avoid multiple\nnormalization passes.","ref":"String.html#equivalent?/2"},{"type":"function","title":"Examples - String.equivalent?/2","doc":"iex> String.equivalent?(\"abc\", \"abc\")\n    true\n\n    iex> String.equivalent?(\"man\\u0303ana\", \"mañana\")\n    true\n\n    iex> String.equivalent?(\"abc\", \"ABC\")\n    false\n\n    iex> String.equivalent?(\"nø\", \"nó\")\n    false","ref":"String.html#equivalent?/2-examples"},{"type":"function","title":"String.first/1","doc":"Returns the first grapheme from a UTF-8 string,\n`nil` if the string is empty.","ref":"String.html#first/1"},{"type":"function","title":"Examples - String.first/1","doc":"iex> String.first(\"elixir\")\n    \"e\"\n\n    iex> String.first(\"եոգլի\")\n    \"ե\"\n\n    iex> String.first(\"\")\n    nil","ref":"String.html#first/1-examples"},{"type":"function","title":"String.graphemes/1","doc":"Returns Unicode graphemes in the string as per Extended Grapheme\nCluster algorithm.\n\nThe algorithm is outlined in the [Unicode Standard Annex #29,\nUnicode Text Segmentation](https://www.unicode.org/reports/tr29/).\n\nFor details about code points and graphemes, see the `String` module documentation.","ref":"String.html#graphemes/1"},{"type":"function","title":"Examples - String.graphemes/1","doc":"iex> String.graphemes(\"Ńaïve\")\n    [\"Ń\", \"a\", \"ï\", \"v\", \"e\"]\n\n    iex> String.graphemes(\"\\u00e9\")\n    [\"é\"]\n\n    iex> String.graphemes(\"\\u0065\\u0301\")\n    [\"é\"]","ref":"String.html#graphemes/1-examples"},{"type":"function","title":"String.jaro_distance/2","doc":"Computes the Jaro distance (similarity) between two strings.\n\nReturns a float value between `0.0` (equates to no similarity) and `1.0`\n(is an exact match) representing [Jaro](https://en.wikipedia.org/wiki/Jaro-Winkler_distance)\ndistance between `string1` and `string2`.\n\nThe Jaro distance metric is designed and best suited for short\nstrings such as person names. Elixir itself uses this function\nto provide the \"did you mean?\" functionality. For instance, when you\nare calling a function in a module and you have a typo in the\nfunction name, we attempt to suggest the most similar function\nname available, if any, based on the `jaro_distance/2` score.","ref":"String.html#jaro_distance/2"},{"type":"function","title":"Examples - String.jaro_distance/2","doc":"iex> String.jaro_distance(\"Dwayne\", \"Duane\")\n    0.8222222222222223\n    iex> String.jaro_distance(\"even\", \"odd\")\n    0.0\n    iex> String.jaro_distance(\"same\", \"same\")\n    1.0","ref":"String.html#jaro_distance/2-examples"},{"type":"function","title":"String.last/1","doc":"Returns the last grapheme from a UTF-8 string,\n`nil` if the string is empty.\n\nIt traverses the whole string to find its last grapheme.","ref":"String.html#last/1"},{"type":"function","title":"Examples - String.last/1","doc":"iex> String.last(\"\")\n    nil\n\n    iex> String.last(\"elixir\")\n    \"r\"\n\n    iex> String.last(\"եոգլի\")\n    \"ի\"","ref":"String.html#last/1-examples"},{"type":"function","title":"String.length/1","doc":"Returns the number of Unicode graphemes in a UTF-8 string.","ref":"String.html#length/1"},{"type":"function","title":"Examples - String.length/1","doc":"iex> String.length(\"elixir\")\n    6\n\n    iex> String.length(\"եոգլի\")\n    5","ref":"String.html#length/1-examples"},{"type":"function","title":"String.match?/2","doc":"Checks if `string` matches the given regular expression.","ref":"String.html#match?/2"},{"type":"function","title":"Examples - String.match?/2","doc":"iex> String.match?(\"foo\", ~r/foo/)\n    true\n\n    iex> String.match?(\"bar\", ~r/foo/)\n    false\n\nElixir also provides text-based match operator `=~/2` and function `Regex.match?/2` as\nalternatives to test strings against regular expressions.","ref":"String.html#match?/2-examples"},{"type":"function","title":"String.myers_difference/2","doc":"Returns a keyword list that represents an edit script.\n\nCheck `List.myers_difference/2` for more information.","ref":"String.html#myers_difference/2"},{"type":"function","title":"Examples - String.myers_difference/2","doc":"iex> string1 = \"fox hops over the dog\"\n    iex> string2 = \"fox jumps over the lazy cat\"\n    iex> String.myers_difference(string1, string2)\n    [eq: \"fox \", del: \"ho\", ins: \"jum\", eq: \"ps over the \", del: \"dog\", ins: \"lazy cat\"]","ref":"String.html#myers_difference/2-examples"},{"type":"function","title":"String.next_codepoint/1","doc":"Returns the next code point in a string.\n\nThe result is a tuple with the code point and the\nremainder of the string or `nil` in case\nthe string reached its end.\n\nAs with other functions in the `String` module, `next_codepoint/1`\nworks with binaries that are invalid UTF-8. If the string starts\nwith a sequence of bytes that is not valid in UTF-8 encoding, the\nfirst element of the returned tuple is a binary with the first byte.","ref":"String.html#next_codepoint/1"},{"type":"function","title":"Examples - String.next_codepoint/1","doc":"iex> String.next_codepoint(\"olá\")\n    {\"o\", \"lá\"}\n\n    iex> invalid = \"\\x80\\x80OK\" # first two bytes are invalid in UTF-8\n    iex> {_, rest} = String.next_codepoint(invalid)\n    { >,  >}\n    iex> String.next_codepoint(rest)\n    { >, \"OK\"}","ref":"String.html#next_codepoint/1-examples"},{"type":"function","title":"Comparison with binary pattern matching - String.next_codepoint/1","doc":"Binary pattern matching provides a similar way to decompose\na string:\n\n    iex>  > = \"Elixir\"\n    \"Elixir\"\n    iex> codepoint\n    69\n    iex> rest\n    \"lixir\"\n\nthough not entirely equivalent because `codepoint` comes as\nan integer, and the pattern won't match invalid UTF-8.\n\nBinary pattern matching, however, is simpler and more efficient,\nso pick the option that better suits your use case.","ref":"String.html#next_codepoint/1-comparison-with-binary-pattern-matching"},{"type":"function","title":"String.next_grapheme/1","doc":"Returns the next grapheme in a string.\n\nThe result is a tuple with the grapheme and the\nremainder of the string or `nil` in case\nthe String reached its end.","ref":"String.html#next_grapheme/1"},{"type":"function","title":"Examples - String.next_grapheme/1","doc":"iex> String.next_grapheme(\"olá\")\n    {\"o\", \"lá\"}\n\n    iex> String.next_grapheme(\"\")\n    nil","ref":"String.html#next_grapheme/1-examples"},{"type":"function","title":"String.normalize/2","doc":"Converts all characters in `string` to Unicode normalization\nform identified by `form`.\n\nInvalid Unicode codepoints are skipped and the remaining of\nthe string is converted. If you want the algorithm to stop\nand return on invalid codepoint, use `:unicode.characters_to_nfd_binary/1`,\n`:unicode.characters_to_nfc_binary/1`, `:unicode.characters_to_nfkd_binary/1`,\nand `:unicode.characters_to_nfkc_binary/1` instead.\n\nNormalization forms `:nfkc` and `:nfkd` should not be blindly applied\nto arbitrary text. Because they erase many formatting distinctions,\nthey will prevent round-trip conversion to and from many legacy\ncharacter sets.","ref":"String.html#normalize/2"},{"type":"function","title":"Forms - String.normalize/2","doc":"The supported forms are:\n\n  * `:nfd` - Normalization Form Canonical Decomposition.\n    Characters are decomposed by canonical equivalence, and\n    multiple combining characters are arranged in a specific\n    order.\n\n  * `:nfc` - Normalization Form Canonical Composition.\n    Characters are decomposed and then recomposed by canonical equivalence.\n\n  * `:nfkd` - Normalization Form Compatibility Decomposition.\n    Characters are decomposed by compatibility equivalence, and\n    multiple combining characters are arranged in a specific\n    order.\n\n  * `:nfkc` - Normalization Form Compatibility Composition.\n    Characters are decomposed and then recomposed by compatibility equivalence.","ref":"String.html#normalize/2-forms"},{"type":"function","title":"Examples - String.normalize/2","doc":"iex> String.normalize(\"yêṩ\", :nfd)\n    \"yêṩ\"\n\n    iex> String.normalize(\"leña\", :nfc)\n    \"leña\"\n\n    iex> String.normalize(\"ﬁ\", :nfkd)\n    \"fi\"\n\n    iex> String.normalize(\"fi\", :nfkc)\n    \"fi\"","ref":"String.html#normalize/2-examples"},{"type":"function","title":"String.pad_leading/3","doc":"Returns a new string padded with a leading filler\nwhich is made of elements from the `padding`.\n\nPassing a list of strings as `padding` will take one element of the list\nfor every missing entry. If the list is shorter than the number of inserts,\nthe filling will start again from the beginning of the list.\nPassing a string `padding` is equivalent to passing the list of graphemes in it.\nIf no `padding` is given, it defaults to whitespace.\n\nWhen `count` is less than or equal to the length of `string`,\ngiven `string` is returned.\n\nRaises `ArgumentError` if the given `padding` contains a non-string element.","ref":"String.html#pad_leading/3"},{"type":"function","title":"Examples - String.pad_leading/3","doc":"iex> String.pad_leading(\"abc\", 5)\n    \"  abc\"\n\n    iex> String.pad_leading(\"abc\", 4, \"12\")\n    \"1abc\"\n\n    iex> String.pad_leading(\"abc\", 6, \"12\")\n    \"121abc\"\n\n    iex> String.pad_leading(\"abc\", 5, [\"1\", \"23\"])\n    \"123abc\"","ref":"String.html#pad_leading/3-examples"},{"type":"function","title":"String.pad_trailing/3","doc":"Returns a new string padded with a trailing filler\nwhich is made of elements from the `padding`.\n\nPassing a list of strings as `padding` will take one element of the list\nfor every missing entry. If the list is shorter than the number of inserts,\nthe filling will start again from the beginning of the list.\nPassing a string `padding` is equivalent to passing the list of graphemes in it.\nIf no `padding` is given, it defaults to whitespace.\n\nWhen `count` is less than or equal to the length of `string`,\ngiven `string` is returned.\n\nRaises `ArgumentError` if the given `padding` contains a non-string element.","ref":"String.html#pad_trailing/3"},{"type":"function","title":"Examples - String.pad_trailing/3","doc":"iex> String.pad_trailing(\"abc\", 5)\n    \"abc  \"\n\n    iex> String.pad_trailing(\"abc\", 4, \"12\")\n    \"abc1\"\n\n    iex> String.pad_trailing(\"abc\", 6, \"12\")\n    \"abc121\"\n\n    iex> String.pad_trailing(\"abc\", 5, [\"1\", \"23\"])\n    \"abc123\"","ref":"String.html#pad_trailing/3-examples"},{"type":"function","title":"String.printable?/2","doc":"Checks if a string contains only printable characters up to `character_limit`.\n\nTakes an optional `character_limit` as a second argument. If `character_limit` is `0`, this\nfunction will return `true`.","ref":"String.html#printable?/2"},{"type":"function","title":"Examples - String.printable?/2","doc":"iex> String.printable?(\"abc\")\n    true\n\n    iex> String.printable?(\"abc\"    >)\n    false\n\n    iex> String.printable?(\"abc\"    >, 2)\n    true\n\n    iex> String.printable?(\"abc\"    >, 0)\n    true","ref":"String.html#printable?/2-examples"},{"type":"function","title":"String.replace/4","doc":"Returns a new string created by replacing occurrences of `pattern` in\n`subject` with `replacement`.\n\nThe `subject` is always a string.\n\nThe `pattern` may be a string, a list of strings, a regular expression, or a\ncompiled pattern.\n\nThe `replacement` may be a string or a function that receives the matched\npattern and must return the replacement as a string or iodata.\n\nBy default it replaces all occurrences but this behaviour can be controlled\nthrough the `:global` option; see the \"Options\" section below.","ref":"String.html#replace/4"},{"type":"function","title":"Options - String.replace/4","doc":"* `:global` - (boolean) if `true`, all occurrences of `pattern` are replaced\n    with `replacement`, otherwise only the first occurrence is\n    replaced. Defaults to `true`","ref":"String.html#replace/4-options"},{"type":"function","title":"Examples - String.replace/4","doc":"iex> String.replace(\"a,b,c\", \",\", \"-\")\n    \"a-b-c\"\n\n    iex> String.replace(\"a,b,c\", \",\", \"-\", global: false)\n    \"a-b,c\"\n\nThe pattern may also be a list of strings and the replacement may also\nbe a function that receives the matches:\n\n    iex> String.replace(\"a,b,c\", [\"a\", \"c\"], fn  > ->  > end)\n    \"b,b,d\"\n\nWhen the pattern is a regular expression, one can give `\\N` or\n`\\g{N}` in the `replacement` string to access a specific capture in the\nregular expression:\n\n    iex> String.replace(\"a,b,c\", ~r/,(.)/, \",\\\\1\\\\g{1}\")\n    \"a,bb,cc\"\n\nNote that we had to escape the backslash escape character (i.e., we used `\\\\N`\ninstead of just `\\N` to escape the backslash; same thing for `\\\\g{N}`). By\ngiving `\\0`, one can inject the whole match in the replacement string.\n\nA compiled pattern can also be given:\n\n    iex> pattern = :binary.compile_pattern(\",\")\n    iex> String.replace(\"a,b,c\", pattern, \"[]\")\n    \"a[]b[]c\"\n\nWhen an empty string is provided as a `pattern`, the function will treat it as\nan implicit empty string between each grapheme and the string will be\ninterspersed. If an empty string is provided as `replacement` the `subject`\nwill be returned:\n\n    iex> String.replace(\"ELIXIR\", \"\", \".\")\n    \".E.L.I.X.I.R.\"\n\n    iex> String.replace(\"ELIXIR\", \"\", \"\")\n    \"ELIXIR\"\n\nBe aware that this function can replace within or across grapheme boundaries.\nFor example, take the grapheme \"é\" which is made of the characters\n\"e\" and the acute accent. The following will replace only the letter \"e\",\nmoving the accent to the letter \"o\":\n\n    iex> String.replace(String.normalize(\"é\", :nfd), \"e\", \"o\")\n    \"ó\"\n\nHowever, if \"é\" is represented by the single character \"e with acute\"\naccent, then it won't be replaced at all:\n\n    iex> String.replace(String.normalize(\"é\", :nfc), \"e\", \"o\")\n    \"é\"","ref":"String.html#replace/4-examples"},{"type":"function","title":"String.replace_leading/3","doc":"Replaces all leading occurrences of `match` by `replacement` of `match` in `string`.\n\nReturns the string untouched if there are no occurrences.\n\nIf `match` is `\"\"`, this function raises an `ArgumentError` exception: this\nhappens because this function replaces **all** the occurrences of `match` at\nthe beginning of `string`, and it's impossible to replace \"multiple\"\noccurrences of `\"\"`.","ref":"String.html#replace_leading/3"},{"type":"function","title":"Examples - String.replace_leading/3","doc":"iex> String.replace_leading(\"hello world\", \"hello \", \"\")\n    \"world\"\n    iex> String.replace_leading(\"hello hello world\", \"hello \", \"\")\n    \"world\"\n\n    iex> String.replace_leading(\"hello world\", \"hello \", \"ola \")\n    \"ola world\"\n    iex> String.replace_leading(\"hello hello world\", \"hello \", \"ola \")\n    \"ola ola world\"\n\nThis function can replace across grapheme boundaries. See `replace/3`\nfor more information and examples.","ref":"String.html#replace_leading/3-examples"},{"type":"function","title":"String.replace_prefix/3","doc":"Replaces prefix in `string` by `replacement` if it matches `match`.\n\nReturns the string untouched if there is no match. If `match` is an empty\nstring (`\"\"`), `replacement` is just prepended to `string`.","ref":"String.html#replace_prefix/3"},{"type":"function","title":"Examples - String.replace_prefix/3","doc":"iex> String.replace_prefix(\"world\", \"hello \", \"\")\n    \"world\"\n    iex> String.replace_prefix(\"hello world\", \"hello \", \"\")\n    \"world\"\n    iex> String.replace_prefix(\"hello hello world\", \"hello \", \"\")\n    \"hello world\"\n\n    iex> String.replace_prefix(\"world\", \"hello \", \"ola \")\n    \"world\"\n    iex> String.replace_prefix(\"hello world\", \"hello \", \"ola \")\n    \"ola world\"\n    iex> String.replace_prefix(\"hello hello world\", \"hello \", \"ola \")\n    \"ola hello world\"\n\n    iex> String.replace_prefix(\"world\", \"\", \"hello \")\n    \"hello world\"\n\nThis function can replace across grapheme boundaries. See `replace/3`\nfor more information and examples.","ref":"String.html#replace_prefix/3-examples"},{"type":"function","title":"String.replace_suffix/3","doc":"Replaces suffix in `string` by `replacement` if it matches `match`.\n\nReturns the string untouched if there is no match. If `match` is an empty\nstring (`\"\"`), `replacement` is just appended to `string`.","ref":"String.html#replace_suffix/3"},{"type":"function","title":"Examples - String.replace_suffix/3","doc":"iex> String.replace_suffix(\"hello\", \" world\", \"\")\n    \"hello\"\n    iex> String.replace_suffix(\"hello world\", \" world\", \"\")\n    \"hello\"\n    iex> String.replace_suffix(\"hello world world\", \" world\", \"\")\n    \"hello world\"\n\n    iex> String.replace_suffix(\"hello\", \" world\", \" mundo\")\n    \"hello\"\n    iex> String.replace_suffix(\"hello world\", \" world\", \" mundo\")\n    \"hello mundo\"\n    iex> String.replace_suffix(\"hello world world\", \" world\", \" mundo\")\n    \"hello world mundo\"\n\n    iex> String.replace_suffix(\"hello\", \"\", \" world\")\n    \"hello world\"\n\nThis function can replace across grapheme boundaries. See `replace/3`\nfor more information and examples.","ref":"String.html#replace_suffix/3-examples"},{"type":"function","title":"String.replace_trailing/3","doc":"Replaces all trailing occurrences of `match` by `replacement` in `string`.\n\nReturns the string untouched if there are no occurrences.\n\nIf `match` is `\"\"`, this function raises an `ArgumentError` exception: this\nhappens because this function replaces **all** the occurrences of `match` at\nthe end of `string`, and it's impossible to replace \"multiple\" occurrences of\n`\"\"`.","ref":"String.html#replace_trailing/3"},{"type":"function","title":"Examples - String.replace_trailing/3","doc":"iex> String.replace_trailing(\"hello world\", \" world\", \"\")\n    \"hello\"\n    iex> String.replace_trailing(\"hello world world\", \" world\", \"\")\n    \"hello\"\n\n    iex> String.replace_trailing(\"hello world\", \" world\", \" mundo\")\n    \"hello mundo\"\n    iex> String.replace_trailing(\"hello world world\", \" world\", \" mundo\")\n    \"hello mundo mundo\"\n\nThis function can replace across grapheme boundaries. See `replace/3`\nfor more information and examples.","ref":"String.html#replace_trailing/3-examples"},{"type":"function","title":"String.reverse/1","doc":"Reverses the graphemes in given string.","ref":"String.html#reverse/1"},{"type":"function","title":"Examples - String.reverse/1","doc":"iex> String.reverse(\"abcd\")\n    \"dcba\"\n\n    iex> String.reverse(\"hello world\")\n    \"dlrow olleh\"\n\n    iex> String.reverse(\"hello ∂og\")\n    \"go∂ olleh\"\n\nKeep in mind reversing the same string twice does\nnot necessarily yield the original string:\n\n    iex> \"̀e\"\n    \"̀e\"\n    iex> String.reverse(\"̀e\")\n    \"è\"\n    iex> String.reverse(String.reverse(\"̀e\"))\n    \"è\"\n\nIn the first example the accent is before the vowel, so\nit is considered two graphemes. However, when you reverse\nit once, you have the vowel followed by the accent, which\nbecomes one grapheme. Reversing it again will keep it as\none single grapheme.","ref":"String.html#reverse/1-examples"},{"type":"function","title":"String.slice/2","doc":"Returns a substring from the offset given by the start of the\nrange to the offset given by the end of the range.\n\nIf the start of the range is not a valid offset for the given\nstring or if the range is in reverse order, returns `\"\"`.\n\nIf the start or end of the range is negative, the whole string\nis traversed first in order to convert the negative indices into\npositive ones.\n\nRemember this function works with Unicode graphemes and considers\nthe slices to represent grapheme offsets. If you want to split\non raw bytes, check `Kernel.binary_part/3` or\n`Kernel.binary_slice/2` instead","ref":"String.html#slice/2"},{"type":"function","title":"Examples - String.slice/2","doc":"iex> String.slice(\"elixir\", 1..3)\n    \"lix\"\n    iex> String.slice(\"elixir\", 1..10)\n    \"lixir\"\n\n    iex> String.slice(\"elixir\", -4..-1)\n    \"ixir\"\n    iex> String.slice(\"elixir\", -4..6)\n    \"ixir\"\n    iex> String.slice(\"elixir\", -100..100)\n    \"elixir\"\n\nFor ranges where `start > stop`, you need to explicitly\nmark them as increasing:\n\n    iex> String.slice(\"elixir\", 2..-1//1)\n    \"ixir\"\n    iex> String.slice(\"elixir\", 1..-2//1)\n    \"lixi\"\n\nYou can use `../0` as a shortcut for `0..-1//1`, which returns\nthe whole string as is:\n\n    iex> String.slice(\"elixir\", ..)\n    \"elixir\"\n\nThe step can be any positive number. For example, to\nget every 2 characters of the string:\n\n    iex> String.slice(\"elixir\", 0..-1//2)\n    \"eii\"\n\nIf the first position is after the string ends or after\nthe last position of the range, it returns an empty string:\n\n    iex> String.slice(\"elixir\", 10..3)\n    \"\"\n    iex> String.slice(\"a\", 1..1500)\n    \"\"","ref":"String.html#slice/2-examples"},{"type":"function","title":"String.slice/3","doc":"Returns a substring starting at the offset `start`, and of the given `length`.\n\nIf the offset is greater than string length, then it returns `\"\"`.\n\nRemember this function works with Unicode graphemes and considers\nthe slices to represent grapheme offsets. If you want to split\non raw bytes, check `Kernel.binary_part/3` or `Kernel.binary_slice/3`\ninstead.","ref":"String.html#slice/3"},{"type":"function","title":"Examples - String.slice/3","doc":"iex> String.slice(\"elixir\", 1, 3)\n    \"lix\"\n\n    iex> String.slice(\"elixir\", 1, 10)\n    \"lixir\"\n\n    iex> String.slice(\"elixir\", 10, 3)\n    \"\"\n\nIf the start position is negative, it is normalized\nagainst the string length and clamped to 0:\n\n    iex> String.slice(\"elixir\", -4, 4)\n    \"ixir\"\n\n    iex> String.slice(\"elixir\", -10, 3)\n    \"eli\"\n\nIf start is more than the string length, an empty\nstring is returned:\n\n    iex> String.slice(\"elixir\", 10, 1500)\n    \"\"","ref":"String.html#slice/3-examples"},{"type":"function","title":"String.split/1","doc":"Divides a string into substrings at each Unicode whitespace\noccurrence with leading and trailing whitespace ignored. Groups\nof whitespace are treated as a single occurrence. Divisions do\nnot occur on non-breaking whitespace.","ref":"String.html#split/1"},{"type":"function","title":"Examples - String.split/1","doc":"iex> String.split(\"foo bar\")\n    [\"foo\", \"bar\"]\n\n    iex> String.split(\"foo\"    >   \"bar\")\n    [\"foo\", \"bar\"]\n\n    iex> String.split(\" foo   bar \")\n    [\"foo\", \"bar\"]\n\n    iex> String.split(\"no\\u00a0break\")\n    [\"no\\u00a0break\"]","ref":"String.html#split/1-examples"},{"type":"function","title":"String.split/3","doc":"Divides a string into parts based on a pattern.\n\nReturns a list of these parts.\n\nThe `pattern` may be a string, a list of strings, a regular expression, or a\ncompiled pattern.\n\nThe string is split into as many parts as possible by\ndefault, but can be controlled via the `:parts` option.\n\nEmpty strings are only removed from the result if the\n`:trim` option is set to `true`.\n\nWhen the pattern used is a regular expression, the string is\nsplit using `Regex.split/3`.","ref":"String.html#split/3"},{"type":"function","title":"Options - String.split/3","doc":"* `:parts` (positive integer or `:infinity`) - the string\n    is split into at most as many parts as this option specifies.\n    If `:infinity`, the string will be split into all possible\n    parts. Defaults to `:infinity`.\n\n  * `:trim` (boolean) - if `true`, empty strings are removed from\n    the resulting list.\n\nThis function also accepts all options accepted by `Regex.split/3`\nif `pattern` is a regular expression.","ref":"String.html#split/3-options"},{"type":"function","title":"Examples - String.split/3","doc":"Splitting with a string pattern:\n\n    iex> String.split(\"a,b,c\", \",\")\n    [\"a\", \"b\", \"c\"]\n\n    iex> String.split(\"a,b,c\", \",\", parts: 2)\n    [\"a\", \"b,c\"]\n\n    iex> String.split(\" a b c \", \" \", trim: true)\n    [\"a\", \"b\", \"c\"]\n\nA list of patterns:\n\n    iex> String.split(\"1,2 3,4\", [\" \", \",\"])\n    [\"1\", \"2\", \"3\", \"4\"]\n\nA regular expression:\n\n    iex> String.split(\"a,b,c\", ~r{,})\n    [\"a\", \"b\", \"c\"]\n\n    iex> String.split(\"a,b,c\", ~r{,}, parts: 2)\n    [\"a\", \"b,c\"]\n\n    iex> String.split(\" a b c \", ~r{\\s}, trim: true)\n    [\"a\", \"b\", \"c\"]\n\n    iex> String.split(\"abc\", ~r{b}, include_captures: true)\n    [\"a\", \"b\", \"c\"]\n\nA compiled pattern:\n\n    iex> pattern = :binary.compile_pattern([\" \", \",\"])\n    iex> String.split(\"1,2 3,4\", pattern)\n    [\"1\", \"2\", \"3\", \"4\"]\n\nSplitting on empty string returns graphemes:\n\n    iex> String.split(\"abc\", \"\")\n    [\"\", \"a\", \"b\", \"c\", \"\"]\n\n    iex> String.split(\"abc\", \"\", trim: true)\n    [\"a\", \"b\", \"c\"]\n\n    iex> String.split(\"abc\", \"\", parts: 1)\n    [\"abc\"]\n\n    iex> String.split(\"abc\", \"\", parts: 3)\n    [\"\", \"a\", \"bc\"]\n\nBe aware that this function can split within or across grapheme boundaries.\nFor example, take the grapheme \"é\" which is made of the characters\n\"e\" and the acute accent. The following will split the string into two parts:\n\n    iex> String.split(String.normalize(\"é\", :nfd), \"e\")\n    [\"\", \"́\"]\n\nHowever, if \"é\" is represented by the single character \"e with acute\"\naccent, then it will split the string into just one part:\n\n    iex> String.split(String.normalize(\"é\", :nfc), \"e\")\n    [\"é\"]","ref":"String.html#split/3-examples"},{"type":"function","title":"String.split_at/2","doc":"Splits a string into two at the specified offset. When the offset given is\nnegative, location is counted from the end of the string.\n\nThe offset is capped to the length of the string. Returns a tuple with\ntwo elements.\n\nNote: keep in mind this function splits on graphemes and for such it\nhas to linearly traverse the string. If you want to split a string or\na binary based on the number of bytes, use `Kernel.binary_part/3`\ninstead.","ref":"String.html#split_at/2"},{"type":"function","title":"Examples - String.split_at/2","doc":"iex> String.split_at(\"sweetelixir\", 5)\n    {\"sweet\", \"elixir\"}\n\n    iex> String.split_at(\"sweetelixir\", -6)\n    {\"sweet\", \"elixir\"}\n\n    iex> String.split_at(\"abc\", 0)\n    {\"\", \"abc\"}\n\n    iex> String.split_at(\"abc\", 1000)\n    {\"abc\", \"\"}\n\n    iex> String.split_at(\"abc\", -1000)\n    {\"\", \"abc\"}","ref":"String.html#split_at/2-examples"},{"type":"function","title":"String.splitter/3","doc":"Returns an enumerable that splits a string on demand.\n\nThis is in contrast to `split/3` which splits the\nentire string upfront.\n\nThis function does not support regular expressions\nby design. When using regular expressions, it is often\nmore efficient to have the regular expressions traverse\nthe string at once than in parts, like this function does.","ref":"String.html#splitter/3"},{"type":"function","title":"Options - String.splitter/3","doc":"* :trim - when `true`, does not emit empty patterns","ref":"String.html#splitter/3-options"},{"type":"function","title":"Examples - String.splitter/3","doc":"iex> String.splitter(\"1,2 3,4 5,6 7,8,...,99999\", [\" \", \",\"]) |> Enum.take(4)\n    [\"1\", \"2\", \"3\", \"4\"]\n\n    iex> String.splitter(\"abcd\", \"\") |> Enum.take(10)\n    [\"\", \"a\", \"b\", \"c\", \"d\", \"\"]\n\n    iex> String.splitter(\"abcd\", \"\", trim: true) |> Enum.take(10)\n    [\"a\", \"b\", \"c\", \"d\"]\n\nA compiled pattern can also be given:\n\n    iex> pattern = :binary.compile_pattern([\" \", \",\"])\n    iex> String.splitter(\"1,2 3,4 5,6 7,8,...,99999\", pattern) |> Enum.take(4)\n    [\"1\", \"2\", \"3\", \"4\"]","ref":"String.html#splitter/3-examples"},{"type":"function","title":"String.starts_with?/2","doc":"Returns `true` if `string` starts with any of the prefixes given.\n\n`prefix` can be either a string, a list of strings, or a compiled\npattern.","ref":"String.html#starts_with?/2"},{"type":"function","title":"Examples - String.starts_with?/2","doc":"iex> String.starts_with?(\"elixir\", \"eli\")\n    true\n    iex> String.starts_with?(\"elixir\", [\"erlang\", \"elixir\"])\n    true\n    iex> String.starts_with?(\"elixir\", [\"erlang\", \"ruby\"])\n    false\n\nAn empty string will always match:\n\n    iex> String.starts_with?(\"elixir\", \"\")\n    true\n    iex> String.starts_with?(\"elixir\", [\"\", \"other\"])\n    true\n\nAn empty list will never match:\n\n    iex> String.starts_with?(\"elixir\", [])\n    false\n\n    iex> String.starts_with?(\"\", [])\n    false","ref":"String.html#starts_with?/2-examples"},{"type":"function","title":"String.to_atom/1","doc":"Converts a string to an existing atom or creates a new one.\n\nWarning: this function creates atoms dynamically and atoms are\nnot garbage-collected. Therefore, `string` should not be an\nuntrusted value, such as input received from a socket or during\na web request. Consider using `to_existing_atom/1` instead.\n\nBy default, the maximum number of atoms is `1_048_576`. This limit\ncan be raised or lowered using the VM option `+t`.\n\nThe maximum atom size is of 255 Unicode code points.\n\nInlined by the compiler.","ref":"String.html#to_atom/1"},{"type":"function","title":"Examples - String.to_atom/1","doc":"iex> String.to_atom(\"my_atom\")\n    :my_atom","ref":"String.html#to_atom/1-examples"},{"type":"function","title":"String.to_charlist/1","doc":"Converts a string into a charlist.\n\nSpecifically, this function takes a UTF-8 encoded binary and returns a list of its integer\ncode points. It is similar to `codepoints/1` except that the latter returns a list of code points as\nstrings.\n\nIn case you need to work with bytes, take a look at the\n[`:binary` module](`:binary`).","ref":"String.html#to_charlist/1"},{"type":"function","title":"Examples - String.to_charlist/1","doc":"iex> String.to_charlist(\"foo\")\n    ~c\"foo\"","ref":"String.html#to_charlist/1-examples"},{"type":"function","title":"String.to_existing_atom/1","doc":"Converts a string to an existing atom or raises if\nthe atom does not exist.\n\nThe maximum atom size is of 255 Unicode code points.\nRaises an `ArgumentError` if the atom does not exist.\n\nInlined by the compiler.\n\n> ##","ref":"String.html#to_existing_atom/1"},{"type":"function","title":"Atoms and modules {: .info} - String.to_existing_atom/1","doc":">\n> Since Elixir is a compiled language, the atoms defined in a module\n> will only exist after said module is loaded, which typically happens\n> whenever a function in the module is executed. Therefore, it is\n> generally recommended to call `String.to_existing_atom/1` only to\n> convert atoms defined within the module making the function call\n> to `to_existing_atom/1`.\n>\n> To create a module name itself from a string safely,\n> it is recommended to use `Module.safe_concat/1`.","ref":"String.html#to_existing_atom/1-atoms-and-modules-info"},{"type":"function","title":"Examples - String.to_existing_atom/1","doc":"iex> _ = :my_atom\n    iex> String.to_existing_atom(\"my_atom\")\n    :my_atom","ref":"String.html#to_existing_atom/1-examples"},{"type":"function","title":"String.to_float/1","doc":"Returns a float whose text representation is `string`.\n\n`string` must be the string representation of a float including a decimal point.\nIn order to parse a string without decimal point as a float then `Float.parse/1`\nshould be used. Otherwise, an `ArgumentError` will be raised.\n\nInlined by the compiler.","ref":"String.html#to_float/1"},{"type":"function","title":"Examples - String.to_float/1","doc":"iex> String.to_float(\"2.2017764e+0\")\n    2.2017764\n\n    iex> String.to_float(\"3.0\")\n    3.0\n\n    String.to_float(\"3\")\n    ** (ArgumentError) argument error","ref":"String.html#to_float/1-examples"},{"type":"function","title":"String.to_integer/1","doc":"Returns an integer whose text representation is `string`.\n\n`string` must be the string representation of an integer.\nOtherwise, an `ArgumentError` will be raised. If you want\nto parse a string that may contain an ill-formatted integer,\nuse `Integer.parse/1`.\n\nInlined by the compiler.","ref":"String.html#to_integer/1"},{"type":"function","title":"Examples - String.to_integer/1","doc":"iex> String.to_integer(\"123\")\n    123\n\nPassing a string that does not represent an integer leads to an error:\n\n    String.to_integer(\"invalid data\")\n    ** (ArgumentError) argument error","ref":"String.html#to_integer/1-examples"},{"type":"function","title":"String.to_integer/2","doc":"Returns an integer whose text representation is `string` in base `base`.\n\nInlined by the compiler.","ref":"String.html#to_integer/2"},{"type":"function","title":"Examples - String.to_integer/2","doc":"iex> String.to_integer(\"3FF\", 16)\n    1023","ref":"String.html#to_integer/2-examples"},{"type":"function","title":"String.trim/1","doc":"Returns a string where all leading and trailing Unicode whitespaces\nhave been removed.","ref":"String.html#trim/1"},{"type":"function","title":"Examples - String.trim/1","doc":"iex> String.trim(\"\\n  abc\\n  \")\n    \"abc\"","ref":"String.html#trim/1-examples"},{"type":"function","title":"String.trim/2","doc":"Returns a string where all leading and trailing `to_trim` characters have been\nremoved.","ref":"String.html#trim/2"},{"type":"function","title":"Examples - String.trim/2","doc":"iex> String.trim(\"a  abc  a\", \"a\")\n    \"  abc  \"","ref":"String.html#trim/2-examples"},{"type":"function","title":"String.trim_leading/1","doc":"Returns a string where all leading Unicode whitespaces\nhave been removed.","ref":"String.html#trim_leading/1"},{"type":"function","title":"Examples - String.trim_leading/1","doc":"iex> String.trim_leading(\"\\n  abc   \")\n    \"abc   \"","ref":"String.html#trim_leading/1-examples"},{"type":"function","title":"String.trim_leading/2","doc":"Returns a string where all leading `to_trim` characters have been removed.","ref":"String.html#trim_leading/2"},{"type":"function","title":"Examples - String.trim_leading/2","doc":"iex> String.trim_leading(\"__ abc _\", \"_\")\n    \" abc _\"\n\n    iex> String.trim_leading(\"1 abc\", \"11\")\n    \"1 abc\"","ref":"String.html#trim_leading/2-examples"},{"type":"function","title":"String.trim_trailing/1","doc":"Returns a string where all trailing Unicode whitespaces\nhas been removed.","ref":"String.html#trim_trailing/1"},{"type":"function","title":"Examples - String.trim_trailing/1","doc":"iex> String.trim_trailing(\"   abc\\n  \")\n    \"   abc\"","ref":"String.html#trim_trailing/1-examples"},{"type":"function","title":"String.trim_trailing/2","doc":"Returns a string where all trailing `to_trim` characters have been removed.","ref":"String.html#trim_trailing/2"},{"type":"function","title":"Examples - String.trim_trailing/2","doc":"iex> String.trim_trailing(\"_ abc __\", \"_\")\n    \"_ abc \"\n\n    iex> String.trim_trailing(\"abc 1\", \"11\")\n    \"abc 1\"","ref":"String.html#trim_trailing/2-examples"},{"type":"function","title":"String.upcase/2","doc":"Converts all characters in the given string to uppercase according to `mode`.\n\n`mode` may be `:default`, `:ascii`, `:greek` or `:turkic`. The `:default` mode considers\nall non-conditional transformations outlined in the Unicode standard. `:ascii`\nuppercases only the letters a to z. `:greek` includes the context sensitive\nmappings found in Greek. `:turkic` properly handles the letter i with the dotless variant.","ref":"String.html#upcase/2"},{"type":"function","title":"Examples - String.upcase/2","doc":"iex> String.upcase(\"abcd\")\n    \"ABCD\"\n\n    iex> String.upcase(\"ab 123 xpto\")\n    \"AB 123 XPTO\"\n\n    iex> String.upcase(\"olá\")\n    \"OLÁ\"\n\nThe `:ascii` mode ignores Unicode characters and provides a more\nperformant implementation when you know the string contains only\nASCII characters:\n\n    iex> String.upcase(\"olá\", :ascii)\n    \"OLá\"\n\nAnd `:turkic` properly handles the letter i with the dotless variant:\n\n    iex> String.upcase(\"ıi\")\n    \"II\"\n\n    iex> String.upcase(\"ıi\", :turkic)\n    \"Iİ\"","ref":"String.html#upcase/2-examples"},{"type":"function","title":"String.valid?/2","doc":"Checks whether `string` contains only valid characters.\n\n`algorithm` may be `:default` or `:fast_ascii`. Both algorithms are equivalent\nfrom a validation perspective (they will always produce the same output), but\n`:fast_ascii` can yield significant performance benefits in specific scenarios.\n\nIf all of the following conditions are true, you may want to experiment with\nthe `:fast_ascii` algorithm to see if it yields performance benefits in your\nspecific scenario:\n\n* You are running Erlang/OTP 26 or newer on a 64 bit platform\n* You expect most of your strings to be longer than ~64 bytes\n* You expect most of your strings to contain mostly ASCII codepoints\n\nNote that the `:fast_ascii` algorithm does not affect correctness, you can\nexpect the output of `String.valid?/2` to be the same regardless of algorithm.\nThe only difference to be expected is one of performance, which can be\nexpected to improve roughly linearly in string length compared to the\n`:default` algorithm.","ref":"String.html#valid?/2"},{"type":"function","title":"Examples - String.valid?/2","doc":"iex> String.valid?(\"a\")\n    true\n\n    iex> String.valid?(\"ø\")\n    true\n\n    iex> String.valid?( >)\n    false\n\n    iex> String.valid?( >)\n    true\n\n    iex> String.valid?(\"asd\"    >)\n    false\n\n    iex> String.valid?(\"a\", :fast_ascii)\n    true\n\n    iex> String.valid?(4)\n    ** (FunctionClauseError) no function clause matching in String.valid?/2","ref":"String.html#valid?/2-examples"},{"type":"type","title":"String.codepoint/0","doc":"A single Unicode code point encoded in UTF-8. It may be one or more bytes.","ref":"String.html#t:codepoint/0"},{"type":"type","title":"String.grapheme/0","doc":"Multiple code points that may be perceived as a single character by readers","ref":"String.html#t:grapheme/0"},{"type":"type","title":"String.pattern/0","doc":"Pattern used in functions like `replace/4` and `split/3`.\n\nIt must be one of:\n\n  * a string\n  * an empty list\n  * a list containing non-empty strings\n  * a compiled search pattern created by `:binary.compile_pattern/1`","ref":"String.html#t:pattern/0"},{"type":"type","title":"String.t/0","doc":"A UTF-8 encoded binary.\n\nThe types `String.t()` and `binary()` are equivalent to analysis tools.\nAlthough, for those reading the documentation, `String.t()` implies\nit is a UTF-8 encoded binary.","ref":"String.html#t:t/0"},{"type":"module","title":"Time","doc":"A Time struct and functions.\n\nThe Time struct contains the fields hour, minute, second and microseconds.\nNew times can be built with the `new/4` function or using the\n`~T` (see `sigil_T/2`) sigil:\n\n    iex> ~T[23:00:07.001]\n    ~T[23:00:07.001]\n\nBoth `new/4` and sigil return a struct where the time fields can\nbe accessed directly:\n\n    iex> time = ~T[23:00:07.001]\n    iex> time.hour\n    23\n    iex> time.microsecond\n    {1000, 3}\n\nThe functions on this module work with the `Time` struct as well\nas any struct that contains the same fields as the `Time` struct,\nsuch as `NaiveDateTime` and `DateTime`. Such functions expect\n`t:Calendar.time/0` in their typespecs (instead of `t:t/0`).\n\nDevelopers should avoid creating the Time structs directly\nand instead rely on the functions provided by this module as well\nas the ones in third-party calendar libraries.","ref":"Time.html"},{"type":"module","title":"Comparing times - Time","doc":"Comparisons in Elixir using `==/2`, `>/2`, `  Enum.min([~T[23:00:07.001], ~T[10:00:07.001]], Time)\n    ~T[10:00:07.001]","ref":"Time.html#module-comparing-times"},{"type":"function","title":"Time.add/3","doc":"Adds the `amount_to_add` of `unit`s to the given `time`.\n\nAccepts an `amount_to_add` in any `unit`. `unit` can be\n`:hour`, `:minute`, `:second` or any subsecond precision from\n`t:System.time_unit/0`. It defaults to `:second`. Negative values\nwill move backwards in time.\n\nThis function always consider the unit to be computed according\nto the `Calendar.ISO`.\n\nNote the result value represents the time of day, meaning that it is cyclic,\nfor instance, it will never go over 24 hours for the ISO calendar.","ref":"Time.html#add/3"},{"type":"function","title":"Examples - Time.add/3","doc":"iex> Time.add(~T[10:00:00], 27000)\n    ~T[17:30:00]\n    iex> Time.add(~T[11:00:00.005], 2400)\n    ~T[11:40:00.005]\n    iex> Time.add(~T[00:00:00.000], 86_399_999, :millisecond)\n    ~T[23:59:59.999]\n\nNegative values are allowed:\n\n    iex> Time.add(~T[23:00:00], -60)\n    ~T[22:59:00]\n\nNote that the time is cyclic:\n\n    iex> Time.add(~T[17:10:05], 86400)\n    ~T[17:10:05]\n\nHours and minutes are also supported:\n\n    iex> Time.add(~T[17:10:05], 2, :hour)\n    ~T[19:10:05]\n    iex> Time.add(~T[17:10:05], 30, :minute)\n    ~T[17:40:05]\n\nThis operation merges the precision of the time with the given unit:\n\n    iex> result = Time.add(~T[00:29:10], 21, :millisecond)\n    ~T[00:29:10.021]\n    iex> result.microsecond\n    {21000, 3}","ref":"Time.html#add/3-examples"},{"type":"function","title":"Time.after?/2","doc":"Returns true if the first time is strictly later than the second.","ref":"Time.html#after?/2"},{"type":"function","title":"Examples - Time.after?/2","doc":"iex> Time.after?(~T[16:04:28], ~T[16:04:16])\n    true\n    iex> Time.after?(~T[16:04:16], ~T[16:04:16])\n    false\n    iex> Time.after?(~T[16:04:16.001], ~T[16:04:16.01])\n    false","ref":"Time.html#after?/2-examples"},{"type":"function","title":"Time.before?/2","doc":"Returns true if the first time is strictly earlier than the second.","ref":"Time.html#before?/2"},{"type":"function","title":"Examples - Time.before?/2","doc":"iex> Time.before?(~T[16:04:16], ~T[16:04:28])\n    true\n    iex> Time.before?(~T[16:04:16], ~T[16:04:16])\n    false\n    iex> Time.before?(~T[16:04:16.01], ~T[16:04:16.001])\n    false","ref":"Time.html#before?/2-examples"},{"type":"function","title":"Time.compare/2","doc":"Compares two time structs.\n\nReturns `:gt` if first time is later than the second\nand `:lt` for vice versa. If the two times are equal\n`:eq` is returned.","ref":"Time.html#compare/2"},{"type":"function","title":"Examples - Time.compare/2","doc":"iex> Time.compare(~T[16:04:16], ~T[16:04:28])\n    :lt\n    iex> Time.compare(~T[16:04:16], ~T[16:04:16])\n    :eq\n    iex> Time.compare(~T[16:04:16.01], ~T[16:04:16.001])\n    :gt\n\nThis function can also be used to compare across more\ncomplex calendar types by considering only the time fields:\n\n    iex> Time.compare(~N[1900-01-01 16:04:16], ~N[2015-01-01 16:04:16])\n    :eq\n    iex> Time.compare(~N[2015-01-01 16:04:16], ~N[2015-01-01 16:04:28])\n    :lt\n    iex> Time.compare(~N[2015-01-01 16:04:16.01], ~N[2000-01-01 16:04:16.001])\n    :gt","ref":"Time.html#compare/2-examples"},{"type":"function","title":"Time.convert/2","doc":"Converts given `time` to a different calendar.\n\nReturns `{:ok, time}` if the conversion was successful,\nor `{:error, reason}` if it was not, for some reason.","ref":"Time.html#convert/2"},{"type":"function","title":"Examples - Time.convert/2","doc":"Imagine someone implements `Calendar.Holocene`, a calendar based on the\nGregorian calendar that adds exactly 10,000 years to the current Gregorian\nyear:\n\n    iex> Time.convert(~T[13:30:15], Calendar.Holocene)\n    {:ok, %Time{calendar: Calendar.Holocene, hour: 13, minute: 30, second: 15, microsecond: {0, 0}}}","ref":"Time.html#convert/2-examples"},{"type":"function","title":"Time.convert!/2","doc":"Similar to `Time.convert/2`, but raises an `ArgumentError`\nif the conversion between the two calendars is not possible.","ref":"Time.html#convert!/2"},{"type":"function","title":"Examples - Time.convert!/2","doc":"Imagine someone implements `Calendar.Holocene`, a calendar based on the\nGregorian calendar that adds exactly 10,000 years to the current Gregorian\nyear:\n\n    iex> Time.convert!(~T[13:30:15], Calendar.Holocene)\n    %Time{calendar: Calendar.Holocene, hour: 13, minute: 30, second: 15, microsecond: {0, 0}}","ref":"Time.html#convert!/2-examples"},{"type":"function","title":"Time.diff/3","doc":"Returns the difference between two times, considering only the hour, minute,\nsecond and microsecond.\n\nAs with the `compare/2` function both `Time` structs and other structures\ncontaining time can be used. If for instance a `NaiveDateTime` or `DateTime`\nis passed, only the hour, minute, second, and microsecond is considered. Any\nadditional information about a date or time zone is ignored when calculating\nthe difference.\n\nThe answer can be returned in any `:hour`, `:minute`, `:second` or any\nsubsecond `unit` available from `t:System.time_unit/0`. If the first time\nvalue is earlier than the second, a negative number is returned.\n\nThe unit is measured according to `Calendar.ISO` and defaults to `:second`.\nFractional results are not supported and are truncated.","ref":"Time.html#diff/3"},{"type":"function","title":"Examples - Time.diff/3","doc":"iex> Time.diff(~T[00:29:12], ~T[00:29:10])\n    2\n\n    # When passing a `NaiveDateTime` the date part is ignored.\n    iex> Time.diff(~N[2017-01-01 00:29:12], ~T[00:29:10])\n    2\n\n    # Two `NaiveDateTime` structs could have big differences in the date\n    # but only the time part is considered.\n    iex> Time.diff(~N[2017-01-01 00:29:12], ~N[1900-02-03 00:29:10])\n    2\n\n    iex> Time.diff(~T[00:29:12], ~T[00:29:10], :microsecond)\n    2_000_000\n    iex> Time.diff(~T[00:29:10], ~T[00:29:12], :microsecond)\n    -2_000_000\n\n    iex> Time.diff(~T[02:29:10], ~T[00:29:10], :hour)\n    2\n    iex> Time.diff(~T[02:29:10], ~T[00:29:11], :hour)\n    1","ref":"Time.html#diff/3-examples"},{"type":"function","title":"Time.from_erl/3","doc":"Converts an Erlang time tuple to a `Time` struct.","ref":"Time.html#from_erl/3"},{"type":"function","title":"Examples - Time.from_erl/3","doc":"iex> Time.from_erl({23, 30, 15}, {5000, 3})\n    {:ok, ~T[23:30:15.005]}\n    iex> Time.from_erl({24, 30, 15})\n    {:error, :invalid_time}","ref":"Time.html#from_erl/3-examples"},{"type":"function","title":"Time.from_erl!/3","doc":"Converts an Erlang time tuple to a `Time` struct.","ref":"Time.html#from_erl!/3"},{"type":"function","title":"Examples - Time.from_erl!/3","doc":"iex> Time.from_erl!({23, 30, 15})\n    ~T[23:30:15]\n    iex> Time.from_erl!({23, 30, 15}, {5000, 3})\n    ~T[23:30:15.005]\n    iex> Time.from_erl!({24, 30, 15})\n    ** (ArgumentError) cannot convert {24, 30, 15} to time, reason: :invalid_time","ref":"Time.html#from_erl!/3-examples"},{"type":"function","title":"Time.from_iso8601/2","doc":"Parses the extended \"Local time\" format described by\n[ISO 8601:2019](https://en.wikipedia.org/wiki/ISO_8601).\n\nTime zone offset may be included in the string but they will be\nsimply discarded as such information is not included in times.\n\nAs specified in the standard, the separator \"T\" may be omitted if\ndesired as there is no ambiguity within this function.","ref":"Time.html#from_iso8601/2"},{"type":"function","title":"Examples - Time.from_iso8601/2","doc":"iex> Time.from_iso8601(\"23:50:07\")\n    {:ok, ~T[23:50:07]}\n    iex> Time.from_iso8601(\"23:50:07Z\")\n    {:ok, ~T[23:50:07]}\n    iex> Time.from_iso8601(\"T23:50:07Z\")\n    {:ok, ~T[23:50:07]}\n\n    iex> Time.from_iso8601(\"23:50:07,0123456\")\n    {:ok, ~T[23:50:07.012345]}\n    iex> Time.from_iso8601(\"23:50:07.0123456\")\n    {:ok, ~T[23:50:07.012345]}\n    iex> Time.from_iso8601(\"23:50:07.123Z\")\n    {:ok, ~T[23:50:07.123]}\n\n    iex> Time.from_iso8601(\"2015:01:23 23-50-07\")\n    {:error, :invalid_format}\n    iex> Time.from_iso8601(\"23:50:07A\")\n    {:error, :invalid_format}\n    iex> Time.from_iso8601(\"23:50:07.\")\n    {:error, :invalid_format}\n    iex> Time.from_iso8601(\"23:50:61\")\n    {:error, :invalid_time}","ref":"Time.html#from_iso8601/2-examples"},{"type":"function","title":"Time.from_iso8601!/2","doc":"Parses the extended \"Local time\" format described by\n[ISO 8601:2019](https://en.wikipedia.org/wiki/ISO_8601).\n\nRaises if the format is invalid.","ref":"Time.html#from_iso8601!/2"},{"type":"function","title":"Examples - Time.from_iso8601!/2","doc":"iex> Time.from_iso8601!(\"23:50:07,123Z\")\n    ~T[23:50:07.123]\n    iex> Time.from_iso8601!(\"23:50:07.123Z\")\n    ~T[23:50:07.123]\n    iex> Time.from_iso8601!(\"2015:01:23 23-50-07\")\n    ** (ArgumentError) cannot parse \"2015:01:23 23-50-07\" as time, reason: :invalid_format","ref":"Time.html#from_iso8601!/2-examples"},{"type":"function","title":"Time.from_seconds_after_midnight/3","doc":"Converts a number of seconds after midnight to a `Time` struct.","ref":"Time.html#from_seconds_after_midnight/3"},{"type":"function","title":"Examples - Time.from_seconds_after_midnight/3","doc":"iex> Time.from_seconds_after_midnight(10_000)\n    ~T[02:46:40]\n    iex> Time.from_seconds_after_midnight(30_000, {5000, 3})\n    ~T[08:20:00.005]\n    iex> Time.from_seconds_after_midnight(-1)\n    ~T[23:59:59]\n    iex> Time.from_seconds_after_midnight(100_000)\n    ~T[03:46:40]","ref":"Time.html#from_seconds_after_midnight/3-examples"},{"type":"function","title":"Time.new/5","doc":"Builds a new time.\n\nExpects all values to be integers. Returns `{:ok, time}` if each\nentry fits its appropriate range, returns `{:error, reason}` otherwise.\n\nMicroseconds can also be given with a precision, which must be an\ninteger between 0 and 6.\n\nThe built-in calendar does not support leap seconds.","ref":"Time.html#new/5"},{"type":"function","title":"Examples - Time.new/5","doc":"iex> Time.new(0, 0, 0, 0)\n    {:ok, ~T[00:00:00.000000]}\n    iex> Time.new(23, 59, 59, 999_999)\n    {:ok, ~T[23:59:59.999999]}\n\n    iex> Time.new(24, 59, 59, 999_999)\n    {:error, :invalid_time}\n    iex> Time.new(23, 60, 59, 999_999)\n    {:error, :invalid_time}\n    iex> Time.new(23, 59, 60, 999_999)\n    {:error, :invalid_time}\n    iex> Time.new(23, 59, 59, 1_000_000)\n    {:error, :invalid_time}\n\n    # Invalid precision\n    Time.new(23, 59, 59, {999_999, 10})\n    {:error, :invalid_time}","ref":"Time.html#new/5-examples"},{"type":"function","title":"Time.new!/5","doc":"Builds a new time.\n\nExpects all values to be integers. Returns `time` if each\nentry fits its appropriate range, raises if the time is invalid.\n\nMicroseconds can also be given with a precision, which must be an\ninteger between 0 and 6.\n\nThe built-in calendar does not support leap seconds.","ref":"Time.html#new!/5"},{"type":"function","title":"Examples - Time.new!/5","doc":"iex> Time.new!(0, 0, 0, 0)\n    ~T[00:00:00.000000]\n    iex> Time.new!(23, 59, 59, 999_999)\n    ~T[23:59:59.999999]\n    iex> Time.new!(24, 59, 59, 999_999)\n    ** (ArgumentError) cannot build time, reason: :invalid_time","ref":"Time.html#new!/5-examples"},{"type":"function","title":"Time.to_erl/1","doc":"Converts given `time` to an Erlang time tuple.\n\nWARNING: Loss of precision may occur, as Erlang time tuples\nonly contain hours/minutes/seconds.","ref":"Time.html#to_erl/1"},{"type":"function","title":"Examples - Time.to_erl/1","doc":"iex> Time.to_erl(~T[23:30:15.999])\n    {23, 30, 15}\n\n    iex> Time.to_erl(~N[2010-04-17 23:30:15.999])\n    {23, 30, 15}","ref":"Time.html#to_erl/1-examples"},{"type":"function","title":"Time.to_iso8601/2","doc":"Converts the given time to\n[ISO 8601:2019](https://en.wikipedia.org/wiki/ISO_8601).\n\nBy default, `Time.to_iso8601/2` returns times formatted in the \"extended\"\nformat, for human readability. It also supports the \"basic\" format through\npassing the `:basic` option.\n\n#","ref":"Time.html#to_iso8601/2"},{"type":"function","title":"Examples - Time.to_iso8601/2","doc":"iex> Time.to_iso8601(~T[23:00:13])\n    \"23:00:13\"\n\n    iex> Time.to_iso8601(~T[23:00:13.001])\n    \"23:00:13.001\"\n\n    iex> Time.to_iso8601(~T[23:00:13.001], :basic)\n    \"230013.001\"\n\n    iex> Time.to_iso8601(~N[2010-04-17 23:00:13])\n    \"23:00:13\"","ref":"Time.html#to_iso8601/2-examples"},{"type":"function","title":"Time.to_seconds_after_midnight/1","doc":"Converts a `Time` struct to a number of seconds after midnight.\n\nThe returned value is a two-element tuple with the number of seconds and microseconds.","ref":"Time.html#to_seconds_after_midnight/1"},{"type":"function","title":"Examples - Time.to_seconds_after_midnight/1","doc":"iex> Time.to_seconds_after_midnight(~T[23:30:15])\n    {84615, 0}\n    iex> Time.to_seconds_after_midnight(~N[2010-04-17 23:30:15.999])\n    {84615, 999000}","ref":"Time.html#to_seconds_after_midnight/1-examples"},{"type":"function","title":"Time.to_string/1","doc":"Converts the given `time` to a string.\n\n#","ref":"Time.html#to_string/1"},{"type":"function","title":"Examples - Time.to_string/1","doc":"iex> Time.to_string(~T[23:00:00])\n    \"23:00:00\"\n    iex> Time.to_string(~T[23:00:00.001])\n    \"23:00:00.001\"\n    iex> Time.to_string(~T[23:00:00.123456])\n    \"23:00:00.123456\"\n\n    iex> Time.to_string(~N[2015-01-01 23:00:00.001])\n    \"23:00:00.001\"\n    iex> Time.to_string(~N[2015-01-01 23:00:00.123456])\n    \"23:00:00.123456\"","ref":"Time.html#to_string/1-examples"},{"type":"function","title":"Time.truncate/2","doc":"Returns the given time with the microsecond field truncated to the given\nprecision (`:microsecond`, `millisecond` or `:second`).\n\nThe given time is returned unchanged if it already has lower precision than\nthe given precision.","ref":"Time.html#truncate/2"},{"type":"function","title":"Examples - Time.truncate/2","doc":"iex> Time.truncate(~T[01:01:01.123456], :microsecond)\n    ~T[01:01:01.123456]\n\n    iex> Time.truncate(~T[01:01:01.123456], :millisecond)\n    ~T[01:01:01.123]\n\n    iex> Time.truncate(~T[01:01:01.123456], :second)\n    ~T[01:01:01]","ref":"Time.html#truncate/2-examples"},{"type":"function","title":"Time.utc_now/1","doc":"Returns the current time in UTC.","ref":"Time.html#utc_now/1"},{"type":"function","title":"Examples - Time.utc_now/1","doc":"iex> time = Time.utc_now()\n    iex> time.hour >= 0\n    true","ref":"Time.html#utc_now/1-examples"},{"type":"type","title":"Time.t/0","doc":"","ref":"Time.html#t:t/0"},{"type":"module","title":"Tuple","doc":"Functions for working with tuples.\n\nPlease note the following functions for tuples are found in `Kernel`:\n\n  * `elem/2` - accesses a tuple by index\n  * `put_elem/3` - inserts a value into a tuple by index\n  * `tuple_size/1` - gets the number of elements in a tuple\n\nTuples are intended as fixed-size containers for multiple elements.\nTo manipulate a collection of elements, use a list instead. `Enum`\nfunctions do not work on tuples.\n\nTuples are denoted with curly braces:\n\n    iex> {}\n    {}\n    iex> {1, :two, \"three\"}\n    {1, :two, \"three\"}\n\nA tuple may contain elements of different types, which are stored\ncontiguously in memory. Accessing any element takes constant time,\nbut modifying a tuple, which produces a shallow copy, takes linear time.\nTuples are good for reading data while lists are better for traversals.\n\nTuples are typically used either when a function has multiple return values\nor for error handling. `File.read/1` returns `{:ok, contents}` if reading\nthe given file is successful, or else `{:error, reason}` such as when\nthe file does not exist.\n\nThe functions in this module that add and remove elements from tuples are\nrarely used in practice, as they typically imply tuples are being used as\ncollections. To append to a tuple, it is preferable to extract the elements\nfrom the old tuple with pattern matching, and then create a new tuple:\n\n    tuple = {:ok, :example}\n\n    # Avoid\n    result = Tuple.insert_at(tuple, 2, %{})\n\n    # Prefer\n    {:ok, atom} = tuple\n    result = {:ok, atom, %{}}","ref":"Tuple.html"},{"type":"function","title":"Tuple.append/2","doc":"Inserts an element at the end of a tuple.\n\nReturns a new tuple with the element appended at the end, and contains\nthe elements in `tuple` followed by `value` as the last element.\n\nInlined by the compiler.","ref":"Tuple.html#append/2"},{"type":"function","title":"Examples - Tuple.append/2","doc":"iex> tuple = {:foo, :bar}\n    iex> Tuple.append(tuple, :baz)\n    {:foo, :bar, :baz}","ref":"Tuple.html#append/2-examples"},{"type":"function","title":"Tuple.delete_at/2","doc":"Removes an element from a tuple.\n\nDeletes the element at the given `index` from `tuple`.\nRaises an `ArgumentError` if `index` is negative or greater than\nor equal to the length of `tuple`. Index is zero-based.\n\nInlined by the compiler.","ref":"Tuple.html#delete_at/2"},{"type":"function","title":"Examples - Tuple.delete_at/2","doc":"iex> tuple = {:foo, :bar, :baz}\n    iex> Tuple.delete_at(tuple, 0)\n    {:bar, :baz}","ref":"Tuple.html#delete_at/2-examples"},{"type":"function","title":"Tuple.duplicate/2","doc":"Creates a new tuple.\n\nCreates a tuple of `size` containing the\ngiven `data` at every position.\n\nInlined by the compiler.","ref":"Tuple.html#duplicate/2"},{"type":"function","title":"Examples - Tuple.duplicate/2","doc":"iex> Tuple.duplicate(:hello, 3)\n    {:hello, :hello, :hello}","ref":"Tuple.html#duplicate/2-examples"},{"type":"function","title":"Tuple.insert_at/3","doc":"Inserts an element into a tuple.\n\nInserts `value` into `tuple` at the given `index`.\nRaises an `ArgumentError` if `index` is negative or greater than the\nlength of `tuple`. Index is zero-based.\n\nInlined by the compiler.","ref":"Tuple.html#insert_at/3"},{"type":"function","title":"Examples - Tuple.insert_at/3","doc":"iex> tuple = {:bar, :baz}\n    iex> Tuple.insert_at(tuple, 0, :foo)\n    {:foo, :bar, :baz}\n    iex> Tuple.insert_at(tuple, 2, :bong)\n    {:bar, :baz, :bong}","ref":"Tuple.html#insert_at/3-examples"},{"type":"function","title":"Tuple.product/1","doc":"Computes a product of tuple elements.","ref":"Tuple.html#product/1"},{"type":"function","title":"Examples - Tuple.product/1","doc":"iex> Tuple.product({255, 255})\n    65025\n    iex> Tuple.product({255, 1.0})\n    255.0\n    iex> Tuple.product({})\n    1","ref":"Tuple.html#product/1-examples"},{"type":"function","title":"Tuple.sum/1","doc":"Computes a sum of tuple elements.","ref":"Tuple.html#sum/1"},{"type":"function","title":"Examples - Tuple.sum/1","doc":"iex> Tuple.sum({255, 255})\n    510\n    iex> Tuple.sum({255, 0.0})\n    255.0\n    iex> Tuple.sum({})\n    0","ref":"Tuple.html#sum/1-examples"},{"type":"function","title":"Tuple.to_list/1","doc":"Converts a tuple to a list.\n\nReturns a new list with all the tuple elements.\n\nInlined by the compiler.","ref":"Tuple.html#to_list/1"},{"type":"function","title":"Examples - Tuple.to_list/1","doc":"iex> tuple = {:foo, :bar, :baz}\n    iex> Tuple.to_list(tuple)\n    [:foo, :bar, :baz]","ref":"Tuple.html#to_list/1-examples"},{"type":"module","title":"URI","doc":"Utilities for working with URIs.\n\nThis module provides functions for working with URIs (for example, parsing\nURIs or encoding query strings). The functions in this module are implemented\naccording to [RFC 3986](https://tools.ietf.org/html/rfc3986).","ref":"URI.html"},{"type":"function","title":"URI.__struct__/0","doc":"The URI struct.\n\nThe fields are defined to match the following URI representation\n(with field names between brackets):\n\n    [scheme]://[userinfo]@[host]:[port][path]?[query]#[fragment]\n\n\nNote the `authority` field is deprecated. `parse/1` will still\npopulate it for backwards compatibility but you should generally\navoid setting or getting it.","ref":"URI.html#__struct__/0"},{"type":"function","title":"URI.append_path/2","doc":"Appends `path` to the given `uri`.\n\nPath must start with `/` and cannot contain additional URL components like\nfragments or query strings. This function further assumes the path is valid and\nit does not contain a query string or fragment parts.","ref":"URI.html#append_path/2"},{"type":"function","title":"Examples - URI.append_path/2","doc":"iex> URI.append_path(URI.parse(\"http://example.com/foo/?x=1\"), \"/my-path\") |> URI.to_string()\n    \"http://example.com/foo/my-path?x=1\"\n\n    iex> URI.append_path(URI.parse(\"http://example.com\"), \"my-path\")\n    ** (ArgumentError) path must start with \"/\", got: \"my-path\"","ref":"URI.html#append_path/2-examples"},{"type":"function","title":"URI.append_query/2","doc":"Appends `query` to the given `uri`.\n\nThe given `query` is not automatically encoded, use `encode/2` or `encode_www_form/1`.","ref":"URI.html#append_query/2"},{"type":"function","title":"Examples - URI.append_query/2","doc":"iex> URI.append_query(URI.parse(\"http://example.com/\"), \"x=1\") |> URI.to_string()\n    \"http://example.com/?x=1\"\n\n    iex> URI.append_query(URI.parse(\"http://example.com/?x=1\"), \"y=2\") |> URI.to_string()\n    \"http://example.com/?x=1&y=2\"\n\n    iex> URI.append_query(URI.parse(\"http://example.com/?x=1\"), \"x=2\") |> URI.to_string()\n    \"http://example.com/?x=1&x=2\"","ref":"URI.html#append_query/2-examples"},{"type":"function","title":"URI.char_reserved?/1","doc":"Checks if `character` is a reserved one in a URI.\n\nAs specified in [RFC 3986, section 2.2](https://tools.ietf.org/html/rfc3986#section-2.2),\nthe following characters are reserved: `:`, `/`, `?`, `#`, `[`, `]`, `@`, `!`, `$`, `&`, `'`, `(`, `)`, `*`, `+`, `,`, `;`, `=`","ref":"URI.html#char_reserved?/1"},{"type":"function","title":"Examples - URI.char_reserved?/1","doc":"iex> URI.char_reserved?(?+)\n    true","ref":"URI.html#char_reserved?/1-examples"},{"type":"function","title":"URI.char_unescaped?/1","doc":"Checks if `character` is allowed unescaped in a URI.\n\nThis is the default used by `URI.encode/2` where both\n[reserved](`char_reserved?/1`) and [unreserved characters](`char_unreserved?/1`)\nare kept unescaped.","ref":"URI.html#char_unescaped?/1"},{"type":"function","title":"Examples - URI.char_unescaped?/1","doc":"iex> URI.char_unescaped?(?{)\n    false","ref":"URI.html#char_unescaped?/1-examples"},{"type":"function","title":"URI.char_unreserved?/1","doc":"Checks if `character` is an unreserved one in a URI.\n\nAs specified in [RFC 3986, section 2.3](https://tools.ietf.org/html/rfc3986#section-2.3),\nthe following characters are unreserved:\n\n  * Alphanumeric characters: `A-Z`, `a-z`, `0-9`\n  * `~`, `_`, `-`, `.`","ref":"URI.html#char_unreserved?/1"},{"type":"function","title":"Examples - URI.char_unreserved?/1","doc":"iex> URI.char_unreserved?(?_)\n    true","ref":"URI.html#char_unreserved?/1-examples"},{"type":"function","title":"URI.decode/1","doc":"Percent-unescapes a URI.","ref":"URI.html#decode/1"},{"type":"function","title":"Examples - URI.decode/1","doc":"iex> URI.decode(\"https%3A%2F%2Felixir-lang.org\")\n    \"https://elixir-lang.org\"","ref":"URI.html#decode/1-examples"},{"type":"function","title":"URI.decode_query/3","doc":"Decodes `query` into a map.\n\nGiven a query string in the form of `key1=value1&key2=value2...`, this\nfunction inserts each key-value pair in the query string as one entry in the\ngiven `map`. Keys and values in the resulting map will be binaries. Keys and\nvalues will be percent-unescaped.\n\nYou can specify one of the following `encoding` options:\n\n  * `:www_form` - (default, since v1.12.0) keys and values are decoded as per\n    `decode_www_form/1`. This is the format typically used by browsers on\n    query strings and form data. It decodes \"+\" as \" \".\n\n  * `:rfc3986` - (since v1.12.0) keys and values are decoded as per\n    `decode/1`. The result is the same as `:www_form` except for leaving \"+\"\n    as is in line with [RFC 3986](https://tools.ietf.org/html/rfc3986).\n\nEncoding defaults to `:www_form` for backward compatibility.\n\nUse `query_decoder/1` if you want to iterate over each value manually.","ref":"URI.html#decode_query/3"},{"type":"function","title":"Examples - URI.decode_query/3","doc":"iex> URI.decode_query(\"foo=1&bar=2\")\n    %{\"bar\" => \"2\", \"foo\" => \"1\"}\n\n    iex> URI.decode_query(\"percent=oh+yes%21\", %{\"starting\" => \"map\"})\n    %{\"percent\" => \"oh yes!\", \"starting\" => \"map\"}\n\n    iex> URI.decode_query(\"percent=oh+yes%21\", %{}, :rfc3986)\n    %{\"percent\" => \"oh+yes!\"}","ref":"URI.html#decode_query/3-examples"},{"type":"function","title":"URI.decode_www_form/1","doc":"Decodes `string` as \"x-www-form-urlencoded\".\n\nNote \"x-www-form-urlencoded\" is not specified as part of\nRFC 3986. However, it is a commonly used format to encode\nquery strings and form data by browsers.","ref":"URI.html#decode_www_form/1"},{"type":"function","title":"Examples - URI.decode_www_form/1","doc":"iex> URI.decode_www_form(\"%3Call+in%2F\")\n    \"<all in/\"","ref":"URI.html#decode_www_form/1-examples"},{"type":"function","title":"URI.default_port/1","doc":"Returns the default port for a given `scheme`.\n\nIf the scheme is unknown to the `URI` module, this function returns\n`nil`. The default port for any scheme can be configured globally\nvia `default_port/2`.","ref":"URI.html#default_port/1"},{"type":"function","title":"Examples - URI.default_port/1","doc":"iex> URI.default_port(\"ftp\")\n    21\n\n    iex> URI.default_port(\"ponzi\")\n    nil","ref":"URI.html#default_port/1-examples"},{"type":"function","title":"URI.default_port/2","doc":"Registers the default `port` for the given `scheme`.\n\nAfter this function is called, `port` will be returned by\n`default_port/1` for the given scheme `scheme`. Note that this function\nchanges the default port for the given `scheme` *globally*, meaning for\nevery application.\n\nIt is recommended for this function to be invoked in your\napplication's start callback in case you want to register\nnew URIs.","ref":"URI.html#default_port/2"},{"type":"function","title":"URI.encode/2","doc":"Percent-escapes all characters that require escaping in `string`.\n\nThis means reserved characters, such as `:` and `/`, and the\nso-called unreserved characters, which have the same meaning both\nescaped and unescaped, won't be escaped by default.\n\nSee `encode_www_form/1` if you are interested in escaping reserved\ncharacters too.\n\nThis function also accepts a `predicate` function as an optional\nargument. If passed, this function will be called with each byte\nin `string` as its argument and should return a truthy value (anything other\nthan `false` or `nil`) if the given byte should be left as is, or return a\nfalsy value (`false` or `nil`) if the character should be escaped. Defaults\nto `URI.char_unescaped?/1`.","ref":"URI.html#encode/2"},{"type":"function","title":"Examples - URI.encode/2","doc":"iex> URI.encode(\"ftp://s-ite.tld/?value=put it+й\")\n    \"ftp://s-ite.tld/?value=put%20it+%D0%B9\"\n\n    iex> URI.encode(\"a string\", &(&1 != ?i))\n    \"a str%69ng\"","ref":"URI.html#encode/2-examples"},{"type":"function","title":"URI.encode_query/2","doc":"Encodes `enumerable` into a query string using `encoding`.\n\nTakes an enumerable that enumerates as a list of two-element\ntuples (for instance, a map or a keyword list) and returns a string\nin the form of `key1=value1&key2=value2...`.\n\nKeys and values can be any term that implements the `String.Chars`\nprotocol with the exception of lists, which are explicitly forbidden.\n\nYou can specify one of the following `encoding` strategies:\n\n  * `:www_form` - (default, since v1.12.0) keys and values are URL encoded as\n    per `encode_www_form/1`. This is the format typically used by browsers on\n    query strings and form data. It encodes \" \" as \"+\".\n\n  * `:rfc3986` - (since v1.12.0) the same as `:www_form` except it encodes\n    \" \" as \"%20\" according [RFC 3986](https://tools.ietf.org/html/rfc3986).\n    This is the best option if you are encoding in a non-browser situation,\n    since encoding spaces as \"+\" can be ambiguous to URI parsers. This can\n    inadvertently lead to spaces being interpreted as literal plus signs.\n\nEncoding defaults to `:www_form` for backward compatibility.","ref":"URI.html#encode_query/2"},{"type":"function","title":"Examples - URI.encode_query/2","doc":"iex> query = %{\"foo\" => 1, \"bar\" => 2}\n    iex> URI.encode_query(query)\n    \"bar=2&foo=1\"\n\n    iex> query = %{\"key\" => \"value with spaces\"}\n    iex> URI.encode_query(query)\n    \"key=value+with+spaces\"\n\n    iex> query = %{\"key\" => \"value with spaces\"}\n    iex> URI.encode_query(query, :rfc3986)\n    \"key=value%20with%20spaces\"\n\n    iex> URI.encode_query(%{key: [:a, :list]})\n    ** (ArgumentError) encode_query/2 values cannot be lists, got: [:a, :list]","ref":"URI.html#encode_query/2-examples"},{"type":"function","title":"URI.encode_www_form/1","doc":"Encodes `string` as \"x-www-form-urlencoded\".\n\nNote \"x-www-form-urlencoded\" is not specified as part of\nRFC 3986. However, it is a commonly used format to encode\nquery strings and form data by browsers.","ref":"URI.html#encode_www_form/1"},{"type":"function","title":"Example - URI.encode_www_form/1","doc":"iex> URI.encode_www_form(\"put: it+й\")\n    \"put%3A+it%2B%D0%B9\"","ref":"URI.html#encode_www_form/1-example"},{"type":"function","title":"URI.merge/2","doc":"Merges two URIs.\n\nThis function merges two URIs as per\n[RFC 3986, section 5.2](https://tools.ietf.org/html/rfc3986#section-5.2).","ref":"URI.html#merge/2"},{"type":"function","title":"Examples - URI.merge/2","doc":"iex> URI.merge(URI.parse(\"http://google.com\"), \"/query\") |> to_string()\n    \"http://google.com/query\"\n\n    iex> URI.merge(\"http://example.com\", \"http://google.com\") |> to_string()\n    \"http://google.com\"","ref":"URI.html#merge/2-examples"},{"type":"function","title":"URI.new/1","doc":"Creates a new URI struct from a URI or a string.\n\nIf a `%URI{}` struct is given, it returns `{:ok, uri}`. If a string is\ngiven, it will parse and validate it. If the string is valid, it returns\n`{:ok, uri}`, otherwise it returns `{:error, part}` with the invalid part\nof the URI. For parsing URIs without further validation, see `parse/1`.\n\nThis function can parse both absolute and relative URLs. You can check\nif a URI is absolute or relative by checking if the `scheme` field is\n`nil` or not.\n\nWhen a URI is given without a port, the value returned by `URI.default_port/1`\nfor the URI's scheme is used for the `:port` field. The scheme is also\nnormalized to lowercase.","ref":"URI.html#new/1"},{"type":"function","title":"Examples - URI.new/1","doc":"iex> URI.new(\"https://elixir-lang.org/\")\n    {:ok, %URI{\n      fragment: nil,\n      host: \"elixir-lang.org\",\n      path: \"/\",\n      port: 443,\n      query: nil,\n      scheme: \"https\",\n      userinfo: nil\n    }}\n\n    iex> URI.new(\"//elixir-lang.org/\")\n    {:ok, %URI{\n      fragment: nil,\n      host: \"elixir-lang.org\",\n      path: \"/\",\n      port: nil,\n      query: nil,\n      scheme: nil,\n      userinfo: nil\n    }}\n\n    iex> URI.new(\"/foo/bar\")\n    {:ok, %URI{\n      fragment: nil,\n      host: nil,\n      path: \"/foo/bar\",\n      port: nil,\n      query: nil,\n      scheme: nil,\n      userinfo: nil\n    }}\n\n    iex> URI.new(\"foo/bar\")\n    {:ok, %URI{\n      fragment: nil,\n      host: nil,\n      path: \"foo/bar\",\n      port: nil,\n      query: nil,\n      scheme: nil,\n      userinfo: nil\n    }}\n\n    iex> URI.new(\"//[fe80::]/\")\n    {:ok, %URI{\n      fragment: nil,\n      host: \"fe80::\",\n      path: \"/\",\n      port: nil,\n      query: nil,\n      scheme: nil,\n      userinfo: nil\n    }}\n\n    iex> URI.new(\"https:?query\")\n    {:ok, %URI{\n      fragment: nil,\n      host: nil,\n      path: nil,\n      port: 443,\n      query: \"query\",\n      scheme: \"https\",\n      userinfo: nil\n    }}\n\n    iex> URI.new(\"/invalid_greater_than_in_path/>\")\n    {:error, \">\"}\n\nGiving an existing URI simply returns it wrapped in a tuple:\n\n    iex> {:ok, uri} = URI.new(\"https://elixir-lang.org/\")\n    iex> URI.new(uri)\n    {:ok, %URI{\n      fragment: nil,\n      host: \"elixir-lang.org\",\n      path: \"/\",\n      port: 443,\n      query: nil,\n      scheme: \"https\",\n      userinfo: nil\n    }}","ref":"URI.html#new/1-examples"},{"type":"function","title":"URI.new!/1","doc":"Similar to `new/1` but raises `URI.Error` if an invalid string is given.","ref":"URI.html#new!/1"},{"type":"function","title":"Examples - URI.new!/1","doc":"iex> URI.new!(\"https://elixir-lang.org/\")\n    %URI{\n      fragment: nil,\n      host: \"elixir-lang.org\",\n      path: \"/\",\n      port: 443,\n      query: nil,\n      scheme: \"https\",\n      userinfo: nil\n    }\n\n    iex> URI.new!(\"/invalid_greater_than_in_path/>\")\n    ** (URI.Error) cannot parse due to reason invalid_uri: \">\"\n\nGiving an existing URI simply returns it:\n\n    iex> uri = URI.new!(\"https://elixir-lang.org/\")\n    iex> URI.new!(uri)\n    %URI{\n      fragment: nil,\n      host: \"elixir-lang.org\",\n      path: \"/\",\n      port: 443,\n      query: nil,\n      scheme: \"https\",\n      userinfo: nil\n    }","ref":"URI.html#new!/1-examples"},{"type":"function","title":"URI.parse/1","doc":"Parses a URI into its components, without further validation.\n\nThis function can parse both absolute and relative URLs. You can check\nif a URI is absolute or relative by checking if the `scheme` field is\nnil or not. Furthermore, this function expects both absolute and\nrelative URIs to be well-formed and does not perform any validation.\nSee the \"Examples\" section below. Use `new/1` if you want to validate\nthe URI fields after parsing.\n\nWhen a URI is given without a port, the value returned by `URI.default_port/1`\nfor the URI's scheme is used for the `:port` field. The scheme is also\nnormalized to lowercase.\n\nIf a `%URI{}` struct is given to this function, this function returns it\nunmodified.\n\n> #### `:authority` field {: .info}\n>\n> This function sets the field `:authority` for backwards-compatibility reasons\n> but it is deprecated.","ref":"URI.html#parse/1"},{"type":"function","title":"Examples - URI.parse/1","doc":"iex> URI.parse(\"https://elixir-lang.org/\")\n    %URI{\n      authority: \"elixir-lang.org\",\n      fragment: nil,\n      host: \"elixir-lang.org\",\n      path: \"/\",\n      port: 443,\n      query: nil,\n      scheme: \"https\",\n      userinfo: nil\n    }\n\n    iex> URI.parse(\"//elixir-lang.org/\")\n    %URI{\n      authority: \"elixir-lang.org\",\n      fragment: nil,\n      host: \"elixir-lang.org\",\n      path: \"/\",\n      port: nil,\n      query: nil,\n      scheme: nil,\n      userinfo: nil\n    }\n\n    iex> URI.parse(\"/foo/bar\")\n    %URI{\n      fragment: nil,\n      host: nil,\n      path: \"/foo/bar\",\n      port: nil,\n      query: nil,\n      scheme: nil,\n      userinfo: nil\n    }\n\n    iex> URI.parse(\"foo/bar\")\n    %URI{\n      fragment: nil,\n      host: nil,\n      path: \"foo/bar\",\n      port: nil,\n      query: nil,\n      scheme: nil,\n      userinfo: nil\n    }\n\nIn contrast to `URI.new/1`, this function will parse poorly-formed\nURIs, for example:\n\n    iex> URI.parse(\"/invalid_greater_than_in_path/>\")\n    %URI{\n      fragment: nil,\n      host: nil,\n      path: \"/invalid_greater_than_in_path/>\",\n      port: nil,\n      query: nil,\n      scheme: nil,\n      userinfo: nil\n    }\n\nAnother example is a URI with brackets in query strings. It is accepted\nby `parse/1`, it is commonly accepted by browsers, but it will be refused\nby `new/1`:\n\n    iex> URI.parse(\"/?foo[bar]=baz\")\n    %URI{\n      fragment: nil,\n      host: nil,\n      path: \"/\",\n      port: nil,\n      query: \"foo[bar]=baz\",\n      scheme: nil,\n      userinfo: nil\n    }","ref":"URI.html#parse/1-examples"},{"type":"function","title":"URI.query_decoder/2","doc":"Returns a stream of two-element tuples representing key-value pairs in the\ngiven `query`.\n\nKey and value in each tuple will be binaries and will be percent-unescaped.\n\nYou can specify one of the following `encoding` options:\n\n  * `:www_form` - (default, since v1.12.0) keys and values are decoded as per\n    `decode_www_form/1`. This is the format typically used by browsers on\n    query strings and form data. It decodes \"+\" as \" \".\n\n  * `:rfc3986` - (since v1.12.0) keys and values are decoded as per\n    `decode/1`. The result is the same as `:www_form` except for leaving \"+\"\n    as is in line with [RFC 3986](https://tools.ietf.org/html/rfc3986).\n\nEncoding defaults to `:www_form` for backward compatibility.","ref":"URI.html#query_decoder/2"},{"type":"function","title":"Examples - URI.query_decoder/2","doc":"iex> URI.query_decoder(\"foo=1&bar=2\") |> Enum.to_list()\n    [{\"foo\", \"1\"}, {\"bar\", \"2\"}]\n\n    iex> URI.query_decoder(\"food=bread%26butter&drinks=tap%20water+please\") |> Enum.to_list()\n    [{\"food\", \"bread&butter\"}, {\"drinks\", \"tap water please\"}]\n\n    iex> URI.query_decoder(\"food=bread%26butter&drinks=tap%20water+please\", :rfc3986) |> Enum.to_list()\n    [{\"food\", \"bread&butter\"}, {\"drinks\", \"tap water+please\"}]","ref":"URI.html#query_decoder/2-examples"},{"type":"function","title":"URI.to_string/1","doc":"Returns the string representation of the given [URI struct](`t:t/0`).","ref":"URI.html#to_string/1"},{"type":"function","title":"Examples - URI.to_string/1","doc":"iex> uri = URI.parse(\"http://google.com\")\n    iex> URI.to_string(uri)\n    \"http://google.com\"\n\n    iex> uri = URI.parse(\"foo://bar.baz\")\n    iex> URI.to_string(uri)\n    \"foo://bar.baz\"","ref":"URI.html#to_string/1-examples"},{"type":"opaque","title":"URI.authority/0","doc":"","ref":"URI.html#t:authority/0"},{"type":"type","title":"URI.t/0","doc":"","ref":"URI.html#t:t/0"},{"type":"module","title":"Version","doc":"Functions for parsing and matching versions against requirements.\n\nA version is a string in a specific format or a `Version`\ngenerated after parsing via `Version.parse/1`.\n\nAlthough Elixir projects are not required to follow SemVer,\nthey must follow the format outlined on [SemVer 2.0 schema](https://semver.org/).","ref":"Version.html"},{"type":"module","title":"Versions - Version","doc":"In a nutshell, a version is represented by three numbers:\n\n    MAJOR.MINOR.PATCH\n\nPre-releases are supported by optionally appending a hyphen and a series of\nperiod-separated identifiers immediately following the patch version.\nIdentifiers consist of only ASCII alphanumeric characters and hyphens (`[0-9A-Za-z-]`):\n\n    \"1.0.0-alpha.3\"\n\nBuild information can be added by appending a plus sign and a series of\ndot-separated identifiers immediately following the patch or pre-release version.\nIdentifiers consist of only ASCII alphanumeric characters and hyphens (`[0-9A-Za-z-]`):\n\n    \"1.0.0-alpha.3+20130417140000.amd64\"","ref":"Version.html#module-versions"},{"type":"module","title":"Requirements - Version","doc":"Requirements allow you to specify which versions of a given\ndependency you are willing to work against. Requirements support the common\ncomparison operators such as `>`, `>=`, ` ` described in detail\nfurther below.\n\n    # Only version 2.0.0\n    \"== 2.0.0\"\n\n    # Anything later than 2.0.0\n    \"> 2.0.0\"\n\nRequirements also support `and` and `or` for complex conditions:\n\n    # 2.0.0 and later until 2.1.0\n    \">= 2.0.0 and   2.0.0\"\n\n`~>` will never include pre-release versions of its upper bound,\nregardless of the usage of the `:allow_pre` option, or whether the operand\nis a pre-release version. It can also be used to set an upper bound on only the major\nversion part. See the table below for `~>` requirements and\ntheir corresponding translations.\n\n`~>`           | Translation\n:------------- | :---------------------\n`~> 2.0.0`     | `>= 2.0.0 and   2.1.2`     | `>= 2.1.2 and   2.1.3-dev` | `>= 2.1.3-dev and   2.0`       | `>= 2.0.0 and   2.1`       | `>= 2.1.0 and  ` is allowed to omit the patch version,\nallowing us to express `~> 2.1` or `~> 2.1-dev`, something that wouldn't be allowed\nwhen using the common comparison operators.\n\nWhen the `:allow_pre` option is set `false` in `Version.match?/3`, the requirement\nwill not match a pre-release version unless the operand is a pre-release version.\nThe default is to always allow pre-releases but note that in\nHex `:allow_pre` is set to `false`. See the table below for examples.\n\nRequirement    | Version     | `:allow_pre`      | Matches\n:------------- | :---------- | :---------------- | :------\n`~> 2.0`       | `2.1.0`     | `true` or `false` | `true`\n`~> 2.0`       | `3.0.0`     | `true` or `false` | `false`\n`~> 2.0.0`     | `2.0.5`     | `true` or `false` | `true`\n`~> 2.0.0`     | `2.1.0`     | `true` or `false` | `false`\n`~> 2.1.2`     | `2.1.6-dev` | `true`            | `true`\n`~> 2.1.2`     | `2.1.6-dev` | `false`           | `false`\n`~> 2.1-dev`   | `2.2.0-dev` | `true` or `false` | `true`\n`~> 2.1.2-dev` | `2.1.6-dev` | `true` or `false` | `true`\n`>= 2.1.0`     | `2.2.0-dev` | `true`            | `true`\n`>= 2.1.0`     | `2.2.0-dev` | `false`           | `false`\n`>= 2.1.0-dev` | `2.2.6-dev` | `true` or `false` | `true`","ref":"Version.html#module-requirements"},{"type":"function","title":"Version.__struct__/0","doc":"The Version struct.\n\nIt contains the fields `:major`, `:minor`, `:patch`, `:pre`, and\n`:build` according to SemVer 2.0, where `:pre` is a list.\n\nYou can read those fields but you should not create a new `Version`\ndirectly via the struct syntax. Instead use the functions in this\nmodule.","ref":"Version.html#__struct__/0"},{"type":"function","title":"Version.compare/2","doc":"Compares two versions.\n\nReturns `:gt` if the first version is greater than the second one, and `:lt`\nfor vice versa. If the two versions are equal, `:eq` is returned.\n\nPre-releases are strictly less than their corresponding release versions.\n\nPatch segments are compared lexicographically if they are alphanumeric, and\nnumerically otherwise.\n\nBuild segments are ignored: if two versions differ only in their build segment\nthey are considered to be equal.\n\nRaises a `Version.InvalidVersionError` exception if any of the two given\nversions are not parsable. If given an already parsed version this function\nwon't raise.","ref":"Version.html#compare/2"},{"type":"function","title":"Examples - Version.compare/2","doc":"iex> Version.compare(\"2.0.1-alpha1\", \"2.0.0\")\n    :gt\n\n    iex> Version.compare(\"1.0.0-beta\", \"1.0.0-rc1\")\n    :lt\n\n    iex> Version.compare(\"1.0.0-10\", \"1.0.0-2\")\n    :gt\n\n    iex> Version.compare(\"2.0.1+build0\", \"2.0.1\")\n    :eq\n\n    iex> Version.compare(\"invalid\", \"2.0.1\")\n    ** (Version.InvalidVersionError) invalid version: \"invalid\"","ref":"Version.html#compare/2-examples"},{"type":"function","title":"Version.compile_requirement/1","doc":"Compiles a requirement to an internal representation that may optimize matching.\n\nThe internal representation is opaque.","ref":"Version.html#compile_requirement/1"},{"type":"function","title":"Version.match?/3","doc":"Checks if the given version matches the specification.\n\nReturns `true` if `version` satisfies `requirement`, `false` otherwise.\nRaises a `Version.InvalidRequirementError` exception if `requirement` is not\nparsable, or a `Version.InvalidVersionError` exception if `version` is not parsable.\nIf given an already parsed version and requirement this function won't\nraise.","ref":"Version.html#match?/3"},{"type":"function","title":"Options - Version.match?/3","doc":"* `:allow_pre` (boolean) - when `false`, pre-release versions will not match\n    unless the operand is a pre-release version. Defaults to `true`.\n    For examples, please refer to the table above under the \"Requirements\" section.","ref":"Version.html#match?/3-options"},{"type":"function","title":"Examples - Version.match?/3","doc":"iex> Version.match?(\"2.0.0\", \"> 1.0.0\")\n    true\n\n    iex> Version.match?(\"2.0.0\", \"== 1.0.0\")\n    false\n\n    iex> Version.match?(\"2.1.6-dev\", \"~> 2.1.2\")\n    true\n\n    iex> Version.match?(\"2.1.6-dev\", \"~> 2.1.2\", allow_pre: false)\n    false\n\n    iex> Version.match?(\"foo\", \"== 1.0.0\")\n    ** (Version.InvalidVersionError) invalid version: \"foo\"\n\n    iex> Version.match?(\"2.0.0\", \"== == 1.0.0\")\n    ** (Version.InvalidRequirementError) invalid requirement: \"== == 1.0.0\"","ref":"Version.html#match?/3-examples"},{"type":"function","title":"Version.parse/1","doc":"Parses a version string into a `Version` struct.","ref":"Version.html#parse/1"},{"type":"function","title":"Examples - Version.parse/1","doc":"iex> Version.parse(\"2.0.1-alpha1\")\n    {:ok, %Version{major: 2, minor: 0, patch: 1, pre: [\"alpha1\"]}}\n\n    iex> Version.parse(\"2.0-alpha1\")\n    :error","ref":"Version.html#parse/1-examples"},{"type":"function","title":"Version.parse!/1","doc":"Parses a version string into a `Version`.\n\nIf `string` is an invalid version, a `Version.InvalidVersionError` is raised.","ref":"Version.html#parse!/1"},{"type":"function","title":"Examples - Version.parse!/1","doc":"iex> Version.parse!(\"2.0.1-alpha1\")\n    %Version{major: 2, minor: 0, patch: 1, pre: [\"alpha1\"]}\n\n    iex> Version.parse!(\"2.0-alpha1\")\n    ** (Version.InvalidVersionError) invalid version: \"2.0-alpha1\"","ref":"Version.html#parse!/1-examples"},{"type":"function","title":"Version.parse_requirement/1","doc":"Parses a version requirement string into a `Version.Requirement` struct.","ref":"Version.html#parse_requirement/1"},{"type":"function","title":"Examples - Version.parse_requirement/1","doc":"iex> {:ok, requirement} = Version.parse_requirement(\"== 2.0.1\")\n    iex> requirement\n    Version.parse_requirement!(\"== 2.0.1\")\n\n    iex> Version.parse_requirement(\"== == 2.0.1\")\n    :error","ref":"Version.html#parse_requirement/1-examples"},{"type":"function","title":"Version.parse_requirement!/1","doc":"Parses a version requirement string into a `Version.Requirement` struct.\n\nIf `string` is an invalid requirement, a `Version.InvalidRequirementError` is raised.\n\n# Examples\n\n    iex> Version.parse_requirement!(\"== 2.0.1\")\n    Version.parse_requirement!(\"== 2.0.1\")\n\n    iex> Version.parse_requirement!(\"== == 2.0.1\")\n    ** (Version.InvalidRequirementError) invalid requirement: \"== == 2.0.1\"","ref":"Version.html#parse_requirement!/1"},{"type":"function","title":"Version.to_string/1","doc":"Converts the given version to a string.\n\n#","ref":"Version.html#to_string/1"},{"type":"function","title":"Examples - Version.to_string/1","doc":"iex> Version.to_string(%Version{major: 1, minor: 2, patch: 3})\n    \"1.2.3\"\n    iex> Version.to_string(Version.parse!(\"1.14.0-rc.0+build0\"))\n    \"1.14.0-rc.0+build0\"","ref":"Version.html#to_string/1-examples"},{"type":"type","title":"Version.build/0","doc":"","ref":"Version.html#t:build/0"},{"type":"type","title":"Version.major/0","doc":"","ref":"Version.html#t:major/0"},{"type":"type","title":"Version.minor/0","doc":"","ref":"Version.html#t:minor/0"},{"type":"type","title":"Version.patch/0","doc":"","ref":"Version.html#t:patch/0"},{"type":"type","title":"Version.pre/0","doc":"","ref":"Version.html#t:pre/0"},{"type":"type","title":"Version.requirement/0","doc":"","ref":"Version.html#t:requirement/0"},{"type":"type","title":"Version.t/0","doc":"","ref":"Version.html#t:t/0"},{"type":"type","title":"Version.version/0","doc":"","ref":"Version.html#t:version/0"},{"type":"module","title":"Version.Requirement","doc":"A struct that holds version requirement information.\n\nThe struct fields are private and should not be accessed.\n\nSee the \"Requirements\" section in the `Version` module\nfor more information.","ref":"Version.Requirement.html"},{"type":"opaque","title":"Version.Requirement.t/0","doc":"","ref":"Version.Requirement.html#t:t/0"},{"type":"behaviour","title":"Access","doc":"Key-based access to data structures.\n\nThe `Access` module defines a behaviour for dynamically accessing\nkeys of any type in a data structure via the `data[key]` syntax.\n\n`Access` supports keyword lists (`Keyword`) and maps (`Map`) out\nof the box. Keywords supports only atoms keys, keys for maps can\nbe of any type. Both return `nil` if the key does not exist:\n\n    iex> keywords = [a: 1, b: 2]\n    iex> keywords[:a]\n    1\n    iex> keywords[:c]\n    nil\n\n    iex> map = %{a: 1, b: 2}\n    iex> map[:a]\n    1\n\n    iex> star_ratings = %{1.0 => \"★\", 1.5 => \"★☆\", 2.0 => \"★★\"}\n    iex> star_ratings[1.5]\n    \"★☆\"\n\nThis syntax is very convenient as it can be nested arbitrarily:\n\n    iex> keywords = [a: 1, b: 2]\n    iex> keywords[:c][:unknown]\n    nil\n\nThis works because accessing anything on a `nil` value, returns\n`nil` itself:\n\n    iex> nil[:a]\n    nil\n\nThe access syntax can also be used with the `Kernel.put_in/2`,\n`Kernel.update_in/2` and `Kernel.get_and_update_in/2` macros\nto allow values to be set in nested data structures:\n\n    iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> put_in(users[\"john\"][:age], 28)\n    %{\"john\" => %{age: 28}, \"meg\" => %{age: 23}}","ref":"Access.html"},{"type":"behaviour","title":"Maps and structs - Access","doc":"While the access syntax is allowed in maps via `map[key]`,\nif your map is made of predefined atom keys, you should prefer\nto access those atom keys with `map.key` instead of `map[key]`,\nas `map.key` will raise if the key is missing (which is not\nsupposed to happen if the keys are predefined).\n\nSimilarly, since structs are maps and structs have predefined\nkeys, they only allow the `struct.key` syntax and they do not\nallow the `struct[key]` access syntax. `Access.key/1` can also\nbe used to construct dynamic access to structs and maps.\n\nIn a nutshell, when using `put_in/2` and friends:\n\n    put_in(struct_or_map.key, :value)\n    put_in(keyword_or_map[:key], :value)\n\nWhen using `put_in/3` and friends:\n\n    put_in(struct_or_map, [Access.key!(:key)], :value)\n    put_in(keyword_or_map, [:key], :value)\n\nThis covers the dual nature of maps in Elixir, as they can be\neither for structured data or as a key-value store. See the `Map`\nmodule for more information.","ref":"Access.html#module-maps-and-structs"},{"type":"behaviour","title":"Nested data structures - Access","doc":"Both key-based access syntaxes can be used with the nested update\nfunctions and macros in `Kernel`, such as `Kernel.get_in/2`,\n`Kernel.put_in/3`, `Kernel.update_in/3`, `Kernel.pop_in/2`, and\n`Kernel.get_and_update_in/3`.\n\nFor example, to update a map inside another map:\n\n    iex> users = %{\"john\" => %{age: 27}, \"meg\" => %{age: 23}}\n    iex> put_in(users[\"john\"].age, 28)\n    %{\"john\" => %{age: 28}, \"meg\" => %{age: 23}}\n\nThis module provides convenience functions for traversing other\nstructures, like tuples and lists. These functions can be used\nin all the `Access`-related functions and macros in `Kernel`.\n\nFor instance, given a user map with the `:name` and `:languages`\nkeys, here is how to deeply traverse the map and convert all\nlanguage names to uppercase:\n\n    iex> languages = [\n    ...>   %{name: \"elixir\", type: :functional},\n    ...>   %{name: \"c\", type: :procedural}\n    ...> ]\n    iex> user = %{name: \"john\", languages: languages}\n    iex> update_in(user, [:languages, Access.all(), :name], &String.upcase/1)\n    %{\n      name: \"john\",\n      languages: [\n        %{name: \"ELIXIR\", type: :functional},\n        %{name: \"C\", type: :procedural}\n      ]\n    }\n\nSee the functions `key/1`, `key!/1`, `elem/1`, and `all/0` for\nsome of the available accessors.","ref":"Access.html#module-nested-data-structures"},{"type":"function","title":"Access.all/0","doc":"Returns a function that accesses all the elements in a list.\n\nThe returned function is typically passed as an accessor to `Kernel.get_in/2`,\n`Kernel.get_and_update_in/3`, and friends.","ref":"Access.html#all/0"},{"type":"function","title":"Examples - Access.all/0","doc":"iex> list = [%{name: \"john\"}, %{name: \"mary\"}]\n    iex> get_in(list, [Access.all(), :name])\n    [\"john\", \"mary\"]\n    iex> get_and_update_in(list, [Access.all(), :name], fn prev ->\n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {[\"john\", \"mary\"], [%{name: \"JOHN\"}, %{name: \"MARY\"}]}\n    iex> pop_in(list, [Access.all(), :name])\n    {[\"john\", \"mary\"], [%{}, %{}]}\n\nHere is an example that traverses the list dropping even\nnumbers and multiplying odd numbers by 2:\n\n    iex> require Integer\n    iex> get_and_update_in([1, 2, 3, 4, 5], [Access.all()], fn num ->\n    ...>   if Integer.is_even(num), do: :pop, else: {num, num * 2}\n    ...> end)\n    {[1, 2, 3, 4, 5], [2, 6, 10]}\n\nAn error is raised if the accessed structure is not a list:\n\n    iex> get_in(%{}, [Access.all()])\n    ** (RuntimeError) Access.all/0 expected a list, got: %{}","ref":"Access.html#all/0-examples"},{"type":"function","title":"Access.at/1","doc":"Returns a function that accesses the element at `index` (zero based) of a list.\n\nKeep in mind that index lookups in lists take linear time: the larger the list,\nthe longer it will take to access its index. Therefore index-based operations\nare generally avoided in favor of other functions in the `Enum` module.\n\nThe returned function is typically passed as an accessor to `Kernel.get_in/2`,\n`Kernel.get_and_update_in/3`, and friends.","ref":"Access.html#at/1"},{"type":"function","title":"Examples - Access.at/1","doc":"iex> list = [%{name: \"john\"}, %{name: \"mary\"}]\n    iex> get_in(list, [Access.at(1), :name])\n    \"mary\"\n    iex> get_in(list, [Access.at(-1), :name])\n    \"mary\"\n    iex> get_and_update_in(list, [Access.at(0), :name], fn prev ->\n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {\"john\", [%{name: \"JOHN\"}, %{name: \"mary\"}]}\n    iex> get_and_update_in(list, [Access.at(-1), :name], fn prev ->\n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {\"mary\", [%{name: \"john\"}, %{name: \"MARY\"}]}\n\n`at/1` can also be used to pop elements out of a list or\na key inside of a list:\n\n    iex> list = [%{name: \"john\"}, %{name: \"mary\"}]\n    iex> pop_in(list, [Access.at(0)])\n    {%{name: \"john\"}, [%{name: \"mary\"}]}\n    iex> pop_in(list, [Access.at(0), :name])\n    {\"john\", [%{}, %{name: \"mary\"}]}\n\nWhen the index is out of bounds, `nil` is returned and the update function is never called:\n\n    iex> list = [%{name: \"john\"}, %{name: \"mary\"}]\n    iex> get_in(list, [Access.at(10), :name])\n    nil\n    iex> get_and_update_in(list, [Access.at(10), :name], fn prev ->\n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {nil, [%{name: \"john\"}, %{name: \"mary\"}]}\n\nAn error is raised if the accessed structure is not a list:\n\n    iex> get_in(%{}, [Access.at(1)])\n    ** (RuntimeError) Access.at/1 expected a list, got: %{}","ref":"Access.html#at/1-examples"},{"type":"function","title":"Access.at!/1","doc":"Same as `at/1` except that it raises `Enum.OutOfBoundsError`\nif the given index is out of bounds.","ref":"Access.html#at!/1"},{"type":"function","title":"Examples - Access.at!/1","doc":"iex> get_in([:a, :b, :c], [Access.at!(2)])\n    :c\n    iex> get_in([:a, :b, :c], [Access.at!(3)])\n    ** (Enum.OutOfBoundsError) out of bounds error","ref":"Access.html#at!/1-examples"},{"type":"function","title":"Access.elem/1","doc":"Returns a function that accesses the element at the given index in a tuple.\n\nThe returned function is typically passed as an accessor to `Kernel.get_in/2`,\n`Kernel.get_and_update_in/3`, and friends.\n\nThe returned function raises if `index` is out of bounds.\n\nNote that popping elements out of tuples is not possible and raises an\nerror.","ref":"Access.html#elem/1"},{"type":"function","title":"Examples - Access.elem/1","doc":"iex> map = %{user: {\"john\", 27}}\n    iex> get_in(map, [:user, Access.elem(0)])\n    \"john\"\n    iex> get_and_update_in(map, [:user, Access.elem(0)], fn prev ->\n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {\"john\", %{user: {\"JOHN\", 27}}}\n    iex> pop_in(map, [:user, Access.elem(0)])\n    ** (RuntimeError) cannot pop data from a tuple\n\nAn error is raised if the accessed structure is not a tuple:\n\n    iex> get_in(%{}, [Access.elem(0)])\n    ** (RuntimeError) Access.elem/1 expected a tuple, got: %{}","ref":"Access.html#elem/1-examples"},{"type":"function","title":"Access.fetch/2","doc":"Fetches the value for the given key in a container (a map, keyword\nlist, or struct that implements the `Access` behaviour).\n\nReturns `{:ok, value}` where `value` is the value under `key` if there is such\na key, or `:error` if `key` is not found.","ref":"Access.html#fetch/2"},{"type":"function","title":"Examples - Access.fetch/2","doc":"iex> Access.fetch(%{name: \"meg\", age: 26}, :name)\n    {:ok, \"meg\"}\n\n    iex> Access.fetch([ordered: true, on_timeout: :exit], :timeout)\n    :error","ref":"Access.html#fetch/2-examples"},{"type":"callback","title":"Access.fetch/2","doc":"Invoked in order to access the value stored under `key` in the given term `term`.\n\nThis function should return `{:ok, value}` where `value` is the value under\n`key` if the key exists in the term, or `:error` if the key does not exist in\nthe term.\n\nMany of the functions defined in the `Access` module internally call this\nfunction. This function is also used when the square-brackets access syntax\n(`structure[key]`) is used: the `fetch/2` callback implemented by the module\nthat defines the `structure` struct is invoked and if it returns `{:ok,\nvalue}` then `value` is returned, or if it returns `:error` then `nil` is\nreturned.\n\nSee the `Map.fetch/2` and `Keyword.fetch/2` implementations for examples of\nhow to implement this callback.","ref":"Access.html#c:fetch/2"},{"type":"function","title":"Access.fetch!/2","doc":"Same as `fetch/2` but returns the value directly,\nor raises a `KeyError` exception if `key` is not found.","ref":"Access.html#fetch!/2"},{"type":"function","title":"Examples - Access.fetch!/2","doc":"iex> Access.fetch!(%{name: \"meg\", age: 26}, :name)\n    \"meg\"","ref":"Access.html#fetch!/2-examples"},{"type":"function","title":"Access.filter/1","doc":"Returns a function that accesses all elements of a list that match the provided predicate.\n\nThe returned function is typically passed as an accessor to `Kernel.get_in/2`,\n`Kernel.get_and_update_in/3`, and friends.","ref":"Access.html#filter/1"},{"type":"function","title":"Examples - Access.filter/1","doc":"iex> list = [%{name: \"john\", salary: 10}, %{name: \"francine\", salary: 30}]\n    iex> get_in(list, [Access.filter(&(&1.salary > 20)), :name])\n    [\"francine\"]\n    iex> get_and_update_in(list, [Access.filter(&(&1.salary  \n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {[\"john\"], [%{name: \"JOHN\", salary: 10}, %{name: \"francine\", salary: 30}]}\n\n`filter/1` can also be used to pop elements out of a list or\na key inside of a list:\n\n    iex> list = [%{name: \"john\", salary: 10}, %{name: \"francine\", salary: 30}]\n    iex> pop_in(list, [Access.filter(&(&1.salary >= 20))])\n    {[%{name: \"francine\", salary: 30}], [%{name: \"john\", salary: 10}]}\n    iex> pop_in(list, [Access.filter(&(&1.salary >= 20)), :name])\n    {[\"francine\"], [%{name: \"john\", salary: 10}, %{salary: 30}]}\n\nWhen no match is found, an empty list is returned and the update function is never called\n\n    iex> list = [%{name: \"john\", salary: 10}, %{name: \"francine\", salary: 30}]\n    iex> get_in(list, [Access.filter(&(&1.salary >= 50)), :name])\n    []\n    iex> get_and_update_in(list, [Access.filter(&(&1.salary >= 50)), :name], fn prev ->\n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {[], [%{name: \"john\", salary: 10}, %{name: \"francine\", salary: 30}]}\n\nAn error is raised if the predicate is not a function or is of the incorrect arity:\n\n    iex> get_in([], [Access.filter(5)])\n    ** (FunctionClauseError) no function clause matching in Access.filter/1\n\nAn error is raised if the accessed structure is not a list:\n\n    iex> get_in(%{}, [Access.filter(fn a -> a == 10 end)])\n    ** (RuntimeError) Access.filter/1 expected a list, got: %{}","ref":"Access.html#filter/1-examples"},{"type":"function","title":"Access.get/3","doc":"Gets the value for the given key in a container (a map, keyword\nlist, or struct that implements the `Access` behaviour).\n\nReturns the value under `key` if there is such a key, or `default` if `key` is\nnot found.","ref":"Access.html#get/3"},{"type":"function","title":"Examples - Access.get/3","doc":"iex> Access.get(%{name: \"john\"}, :name, \"default name\")\n    \"john\"\n    iex> Access.get(%{name: \"john\"}, :age, 25)\n    25\n\n    iex> Access.get([ordered: true], :timeout)\n    nil","ref":"Access.html#get/3-examples"},{"type":"function","title":"Access.get_and_update/3","doc":"Gets and updates the given key in a `container` (a map, a keyword list,\na struct that implements the `Access` behaviour).\n\nThe `fun` argument receives the value of `key` (or `nil` if `key` is not\npresent in `container`) and must return a two-element tuple `{current_value, new_value}`:\nthe \"get\" value `current_value` (the retrieved value, which can be operated on before\nbeing returned) and the new value to be stored under `key` (`new_value`).\n`fun` may also return `:pop`, which means the current value\nshould be removed from the container and returned.\n\nThe returned value is a two-element tuple with the \"get\" value returned by\n`fun` and a new container with the updated value under `key`.","ref":"Access.html#get_and_update/3"},{"type":"function","title":"Examples - Access.get_and_update/3","doc":"iex> Access.get_and_update([a: 1], :a, fn current_value ->\n    ...>   {current_value, current_value + 1}\n    ...> end)\n    {1, [a: 2]}","ref":"Access.html#get_and_update/3-examples"},{"type":"callback","title":"Access.get_and_update/3","doc":"Invoked in order to access the value under `key` and update it at the same time.\n\nThe implementation of this callback should invoke `fun` with the value under\n`key` in the passed structure `data`, or with `nil` if `key` is not present in it.\nThis function must return either `{current_value, new_value}` or `:pop`.\n\nIf the passed function returns `{current_value, new_value}`,\nthe return value of this callback should be `{current_value, new_data}`, where:\n\n  * `current_value` is the retrieved value (which can be operated on before being returned)\n\n  * `new_value` is the new value to be stored under `key`\n\n  * `new_data` is `data` after updating the value of `key` with `new_value`.\n\nIf the passed function returns `:pop`, the return value of this callback\nmust be `{value, new_data}` where `value` is the value under `key`\n(or `nil` if not present) and `new_data` is `data` without `key`.\n\nSee the implementations of `Map.get_and_update/3` or `Keyword.get_and_update/3`\nfor more examples.","ref":"Access.html#c:get_and_update/3"},{"type":"function","title":"Access.key/2","doc":"Returns a function that accesses the given key in a map/struct.\n\nThe returned function is typically passed as an accessor to `Kernel.get_in/2`,\n`Kernel.get_and_update_in/3`, and friends.\n\nThe returned function uses the default value if the key does not exist.\nThis can be used to specify defaults and safely traverse missing keys:\n\n    iex> get_in(%{}, [Access.key(:user, %{}), Access.key(:name, \"meg\")])\n    \"meg\"\n\nSuch is also useful when using update functions, allowing us to introduce\nvalues as we traverse the data structure for updates:\n\n    iex> put_in(%{}, [Access.key(:user, %{}), Access.key(:name)], \"Mary\")\n    %{user: %{name: \"Mary\"}}","ref":"Access.html#key/2"},{"type":"function","title":"Examples - Access.key/2","doc":"iex> map = %{user: %{name: \"john\"}}\n    iex> get_in(map, [Access.key(:unknown, %{}), Access.key(:name, \"john\")])\n    \"john\"\n    iex> get_and_update_in(map, [Access.key(:user), Access.key(:name)], fn prev ->\n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {\"john\", %{user: %{name: \"JOHN\"}}}\n    iex> pop_in(map, [Access.key(:user), Access.key(:name)])\n    {\"john\", %{user: %{}}}\n\nAn error is raised if the accessed structure is not a map or a struct:\n\n    iex> get_in([], [Access.key(:foo)])\n    ** (BadMapError) expected a map, got: []","ref":"Access.html#key/2-examples"},{"type":"function","title":"Access.key!/1","doc":"Returns a function that accesses the given key in a map/struct.\n\nThe returned function is typically passed as an accessor to `Kernel.get_in/2`,\n`Kernel.get_and_update_in/3`, and friends.\n\nSimilar to `key/2`, but the returned function raises if the key does not exist.","ref":"Access.html#key!/1"},{"type":"function","title":"Examples - Access.key!/1","doc":"iex> map = %{user: %{name: \"john\"}}\n    iex> get_in(map, [Access.key!(:user), Access.key!(:name)])\n    \"john\"\n    iex> get_and_update_in(map, [Access.key!(:user), Access.key!(:name)], fn prev ->\n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {\"john\", %{user: %{name: \"JOHN\"}}}\n    iex> pop_in(map, [Access.key!(:user), Access.key!(:name)])\n    {\"john\", %{user: %{}}}\n    iex> get_in(map, [Access.key!(:user), Access.key!(:unknown)])\n    ** (KeyError) key :unknown not found in: %{name: \"john\"}\n\nThe examples above could be partially written as:\n\n    iex> map = %{user: %{name: \"john\"}}\n    iex> map.user.name\n    \"john\"\n    iex> get_and_update_in(map.user.name, fn prev ->\n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {\"john\", %{user: %{name: \"JOHN\"}}}\n\nHowever, it is not possible to remove fields using the dot notation,\nas it is implified those fields must also be present. In any case,\n`Access.key!/1` is useful when the key is not known in advance\nand must be accessed dynamically.\n\nAn error is raised if the accessed structure is not a map/struct:\n\n    iex> get_in([], [Access.key!(:foo)])\n    ** (RuntimeError) Access.key!/1 expected a map/struct, got: []","ref":"Access.html#key!/1-examples"},{"type":"function","title":"Access.pop/2","doc":"Removes the entry with a given key from a container (a map, keyword\nlist, or struct that implements the `Access` behaviour).\n\nReturns a tuple containing the value associated with the key and the\nupdated container. `nil` is returned for the value if the key isn't\nin the container.","ref":"Access.html#pop/2"},{"type":"function","title":"Examples - Access.pop/2","doc":"With a map:\n\n    iex> Access.pop(%{name: \"Elixir\", creator: \"Valim\"}, :name)\n    {\"Elixir\", %{creator: \"Valim\"}}\n\nA keyword list:\n\n    iex> Access.pop([name: \"Elixir\", creator: \"Valim\"], :name)\n    {\"Elixir\", [creator: \"Valim\"]}\n\nAn unknown key:\n\n    iex> Access.pop(%{name: \"Elixir\", creator: \"Valim\"}, :year)\n    {nil, %{creator: \"Valim\", name: \"Elixir\"}}","ref":"Access.html#pop/2-examples"},{"type":"callback","title":"Access.pop/2","doc":"Invoked to \"pop\" the value under `key` out of the given data structure.\n\nWhen `key` exists in the given structure `data`, the implementation should\nreturn a `{value, new_data}` tuple where `value` is the value that was under\n`key` and `new_data` is `term` without `key`.\n\nWhen `key` is not present in the given structure, a tuple `{value, data}`\nshould be returned, where `value` is implementation-defined.\n\nSee the implementations for `Map.pop/3` or `Keyword.pop/3` for more examples.","ref":"Access.html#c:pop/2"},{"type":"function","title":"Access.slice/1","doc":"Returns a function that accesses all items of a list that are within the provided range.\n\nThe range will be normalized following the same rules from `Enum.slice/2`.\n\nThe returned function is typically passed as an accessor to `Kernel.get_in/2`,\n`Kernel.get_and_update_in/3`, and friends.","ref":"Access.html#slice/1"},{"type":"function","title":"Examples - Access.slice/1","doc":"iex> list = [%{name: \"john\", salary: 10}, %{name: \"francine\", salary: 30}, %{name: \"vitor\", salary: 25}]\n    iex> get_in(list, [Access.slice(1..2), :name])\n    [\"francine\", \"vitor\"]\n    iex> get_and_update_in(list, [Access.slice(1..3//2), :name], fn prev ->\n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {[\"francine\"], [%{name: \"john\", salary: 10}, %{name: \"FRANCINE\", salary: 30}, %{name: \"vitor\", salary: 25}]}\n\n`slice/1` can also be used to pop elements out of a list or\na key inside of a list:\n\n    iex> list = [%{name: \"john\", salary: 10}, %{name: \"francine\", salary: 30}, %{name: \"vitor\", salary: 25}]\n    iex> pop_in(list, [Access.slice(-2..-1)])\n    {[%{name: \"francine\", salary: 30}, %{name: \"vitor\", salary: 25}], [%{name: \"john\", salary: 10}]}\n    iex> pop_in(list, [Access.slice(-2..-1), :name])\n    {[\"francine\", \"vitor\"], [%{name: \"john\", salary: 10}, %{salary: 30}, %{salary: 25}]}\n\nWhen no match is found, an empty list is returned and the update function is never called\n\n    iex> list = [%{name: \"john\", salary: 10}, %{name: \"francine\", salary: 30}, %{name: \"vitor\", salary: 25}]\n    iex> get_in(list, [Access.slice(5..10//2), :name])\n    []\n    iex> get_and_update_in(list, [Access.slice(5..10//2), :name], fn prev ->\n    ...>   {prev, String.upcase(prev)}\n    ...> end)\n    {[], [%{name: \"john\", salary: 10}, %{name: \"francine\", salary: 30}, %{name: \"vitor\", salary: 25}]}\n\nAn error is raised if the accessed structure is not a list:\n\n    iex> get_in(%{}, [Access.slice(2..10//3)])\n    ** (ArgumentError) Access.slice/1 expected a list, got: %{}\n\nAn error is raised if the step of the range is negative:\n\n    iex> get_in([], [Access.slice(2..10//-1)])\n    ** (ArgumentError) Access.slice/1 does not accept ranges with negative steps, got: 2..10//-1","ref":"Access.html#slice/1-examples"},{"type":"type","title":"Access.access_fun/2","doc":"","ref":"Access.html#t:access_fun/2"},{"type":"type","title":"Access.container/0","doc":"","ref":"Access.html#t:container/0"},{"type":"type","title":"Access.get_and_update_fun/2","doc":"","ref":"Access.html#t:get_and_update_fun/2"},{"type":"type","title":"Access.get_fun/1","doc":"","ref":"Access.html#t:get_fun/1"},{"type":"type","title":"Access.key/0","doc":"","ref":"Access.html#t:key/0"},{"type":"type","title":"Access.nil_container/0","doc":"","ref":"Access.html#t:nil_container/0"},{"type":"type","title":"Access.t/0","doc":"","ref":"Access.html#t:t/0"},{"type":"type","title":"Access.value/0","doc":"","ref":"Access.html#t:value/0"},{"type":"module","title":"Date.Range","doc":"Returns an inclusive range between dates.\n\nRanges must be created with the `Date.range/2` or `Date.range/3` function.\n\nThe following fields are public:\n\n  * `:first` - the initial date on the range\n  * `:last` - the last date on the range\n  * `:step` - (since v1.12.0) the step\n\nThe remaining fields are private and should not be accessed.","ref":"Date.Range.html"},{"type":"type","title":"Date.Range.t/0","doc":"","ref":"Date.Range.html#t:t/0"},{"type":"module","title":"Enum","doc":"Functions for working with collections (known as enumerables).\n\nIn Elixir, an enumerable is any data type that implements the\n`Enumerable` protocol. `List`s (`[1, 2, 3]`), `Map`s (`%{foo: 1, bar: 2}`)\nand `Range`s (`1..3`) are common data types used as enumerables:\n\n    iex> Enum.map([1, 2, 3], fn x -> x * 2 end)\n    [2, 4, 6]\n\n    iex> Enum.sum([1, 2, 3])\n    6\n\n    iex> Enum.map(1..3, fn x -> x * 2 end)\n    [2, 4, 6]\n\n    iex> Enum.sum(1..3)\n    6\n\n    iex> map = %{\"a\" => 1, \"b\" => 2}\n    iex> Enum.map(map, fn {k, v} -> {k, v * 2} end)\n    [{\"a\", 2}, {\"b\", 4}]\n\nHowever, many other enumerables exist in the language, such as `MapSet`s\nand the data type returned by `File.stream!/3` which allows a file to be\ntraversed as if it was an enumerable.\n\nThe functions in this module work in linear time. This means that, the\ntime it takes to perform an operation grows at the same rate as the length\nof the enumerable. This is expected on operations such as `Enum.map/2`.\nAfter all, if we want to traverse every element on a list, the longer the\nlist, the more elements we need to traverse, and the longer it will take.\n\nThis linear behaviour should also be expected on operations like `count/1`,\n`member?/2`, `at/2` and similar. While Elixir does allow data types to\nprovide performant variants for such operations, you should not expect it\nto always be available, since the `Enum` module is meant to work with a\nlarge variety of data types and not all data types can provide optimized\nbehaviour.\n\nFinally, note the functions in the `Enum` module are eager: they will\ntraverse the enumerable as soon as they are invoked. This is particularly\ndangerous when working with infinite enumerables. In such cases, you should\nuse the `Stream` module, which allows you to lazily express computations,\nwithout traversing collections, and work with possibly infinite collections.\nSee the `Stream` module for examples and documentation.","ref":"Enum.html"},{"type":"function","title":"Enum.all?/1","doc":"Returns `true` if all elements in `enumerable` are truthy.\n\nWhen an element has a falsy value (`false` or `nil`) iteration stops immediately\nand `false` is returned. In all other cases `true` is returned.","ref":"Enum.html#all?/1"},{"type":"function","title":"Examples - Enum.all?/1","doc":"iex> Enum.all?([1, 2, 3])\n    true\n\n    iex> Enum.all?([1, nil, 3])\n    false\n\n    iex> Enum.all?([])\n    true","ref":"Enum.html#all?/1-examples"},{"type":"function","title":"Enum.all?/2","doc":"Returns `true` if `fun.(element)` is truthy for all elements in `enumerable`.\n\nIterates over `enumerable` and invokes `fun` on each element. If `fun` ever\nreturns a falsy value (`false` or `nil`), iteration stops immediately and\n`false` is returned. Otherwise, `true` is returned.","ref":"Enum.html#all?/2"},{"type":"function","title":"Examples - Enum.all?/2","doc":"iex> Enum.all?([2, 4, 6], fn x -> rem(x, 2) == 0 end)\n    true\n\n    iex> Enum.all?([2, 3, 4], fn x -> rem(x, 2) == 0 end)\n    false\n\n    iex> Enum.all?([], fn _ -> nil end)\n    true\n\nAs the last example shows, `Enum.all?/2` returns `true` if `enumerable` is\nempty, regardless of `fun`. In an empty enumerable there is no element for\nwhich `fun` returns a falsy value, so the result must be `true`. This is a\nwell-defined logical argument for empty collections.","ref":"Enum.html#all?/2-examples"},{"type":"function","title":"Enum.any?/1","doc":"Returns `true` if at least one element in `enumerable` is truthy.\n\nWhen an element has a truthy value (neither `false` nor `nil`) iteration stops\nimmediately and `true` is returned. In all other cases `false` is returned.","ref":"Enum.html#any?/1"},{"type":"function","title":"Examples - Enum.any?/1","doc":"iex> Enum.any?([false, false, false])\n    false\n\n    iex> Enum.any?([false, true, false])\n    true\n\n    iex> Enum.any?([])\n    false","ref":"Enum.html#any?/1-examples"},{"type":"function","title":"Enum.any?/2","doc":"Returns `true` if `fun.(element)` is truthy for at least one element in `enumerable`.\n\nIterates over the `enumerable` and invokes `fun` on each element. When an invocation\nof `fun` returns a truthy value (neither `false` nor `nil`) iteration stops\nimmediately and `true` is returned. In all other cases `false` is returned.","ref":"Enum.html#any?/2"},{"type":"function","title":"Examples - Enum.any?/2","doc":"iex> Enum.any?([2, 4, 6], fn x -> rem(x, 2) == 1 end)\n    false\n\n    iex> Enum.any?([2, 3, 4], fn x -> rem(x, 2) == 1 end)\n    true\n\n    iex> Enum.any?([], fn x -> x > 0 end)\n    false","ref":"Enum.html#any?/2-examples"},{"type":"function","title":"Enum.at/3","doc":"Finds the element at the given `index` (zero-based).\n\nReturns `default` if `index` is out of bounds.\n\nA negative `index` can be passed, which means the `enumerable` is\nenumerated once and the `index` is counted from the end (for example,\n`-1` finds the last element).","ref":"Enum.html#at/3"},{"type":"function","title":"Examples - Enum.at/3","doc":"iex> Enum.at([2, 4, 6], 0)\n    2\n\n    iex> Enum.at([2, 4, 6], 2)\n    6\n\n    iex> Enum.at([2, 4, 6], 4)\n    nil\n\n    iex> Enum.at([2, 4, 6], 4, :none)\n    :none","ref":"Enum.html#at/3-examples"},{"type":"function","title":"Enum.chunk_by/2","doc":"Splits enumerable on every element for which `fun` returns a new\nvalue.\n\nReturns a list of lists.","ref":"Enum.html#chunk_by/2"},{"type":"function","title":"Examples - Enum.chunk_by/2","doc":"iex> Enum.chunk_by([1, 2, 2, 3, 4, 4, 6, 7, 7], &(rem(&1, 2) == 1))\n    [[1], [2, 2], [3], [4, 4, 6], [7, 7]]","ref":"Enum.html#chunk_by/2-examples"},{"type":"function","title":"Enum.chunk_every/2","doc":"Shortcut to `chunk_every(enumerable, count, count)`.","ref":"Enum.html#chunk_every/2"},{"type":"function","title":"Enum.chunk_every/4","doc":"Returns list of lists containing `count` elements each, where\neach new chunk starts `step` elements into the `enumerable`.\n\n`step` is optional and, if not passed, defaults to `count`, i.e.\nchunks do not overlap. Chunking will stop as soon as the collection\nends or when we emit an incomplete chunk.\n\nIf the last chunk does not have `count` elements to fill the chunk,\nelements are taken from `leftover` to fill in the chunk. If `leftover`\ndoes not have enough elements to fill the chunk, then a partial chunk\nis returned with less than `count` elements.\n\nIf `:discard` is given in `leftover`, the last chunk is discarded\nunless it has exactly `count` elements.","ref":"Enum.html#chunk_every/4"},{"type":"function","title":"Examples - Enum.chunk_every/4","doc":"iex> Enum.chunk_every([1, 2, 3, 4, 5, 6], 2)\n    [[1, 2], [3, 4], [5, 6]]\n\n    iex> Enum.chunk_every([1, 2, 3, 4, 5, 6], 3, 2, :discard)\n    [[1, 2, 3], [3, 4, 5]]\n\n    iex> Enum.chunk_every([1, 2, 3, 4, 5, 6], 3, 2, [7])\n    [[1, 2, 3], [3, 4, 5], [5, 6, 7]]\n\n    iex> Enum.chunk_every([1, 2, 3, 4], 3, 3, [])\n    [[1, 2, 3], [4]]\n\n    iex> Enum.chunk_every([1, 2, 3, 4], 10)\n    [[1, 2, 3, 4]]\n\n    iex> Enum.chunk_every([1, 2, 3, 4, 5], 2, 3, [])\n    [[1, 2], [4, 5]]\n\n    iex> Enum.chunk_every([1, 2, 3, 4], 3, 3, Stream.cycle([0]))\n    [[1, 2, 3], [4, 0, 0]]","ref":"Enum.html#chunk_every/4-examples"},{"type":"function","title":"Enum.chunk_while/4","doc":"Chunks the `enumerable` with fine grained control when every chunk is emitted.\n\n`chunk_fun` receives the current element and the accumulator and must return:\n\n  * `{:cont, chunk, acc}` to emit a chunk and continue with the accumulator\n  * `{:cont, acc}` to not emit any chunk and continue with the accumulator\n  * `{:halt, acc}` to halt chunking over the `enumerable`.\n\n`after_fun` is invoked with the final accumulator when iteration is\nfinished (or `halt`ed) to handle any trailing elements that were returned\nas part of an accumulator, but were not emitted as a chunk by `chunk_fun`.\nIt must return:\n\n  * `{:cont, chunk, acc}` to emit a chunk. The chunk will be appended to the\n    list of already emitted chunks.\n  * `{:cont, acc}` to not emit a chunk\n\nThe `acc` in `after_fun` is required in order to mirror the tuple format\nfrom `chunk_fun` but it will be discarded since the traversal is complete.\n\nReturns a list of emitted chunks.","ref":"Enum.html#chunk_while/4"},{"type":"function","title":"Examples - Enum.chunk_while/4","doc":"iex> chunk_fun = fn element, acc ->\n    ...>   if rem(element, 2) == 0 do\n    ...>     {:cont, Enum.reverse([element | acc]), []}\n    ...>   else\n    ...>     {:cont, [element | acc]}\n    ...>   end\n    ...> end\n    iex> after_fun = fn\n    ...>   [] -> {:cont, []}\n    ...>   acc -> {:cont, Enum.reverse(acc), []}\n    ...> end\n    iex> Enum.chunk_while(1..10, [], chunk_fun, after_fun)\n    [[1, 2], [3, 4], [5, 6], [7, 8], [9, 10]]\n    iex> Enum.chunk_while([1, 2, 3, 5, 7], [], chunk_fun, after_fun)\n    [[1, 2], [3, 5, 7]]","ref":"Enum.html#chunk_while/4-examples"},{"type":"function","title":"Enum.concat/1","doc":"Given an enumerable of enumerables, concatenates the `enumerables` into\na single list.","ref":"Enum.html#concat/1"},{"type":"function","title":"Examples - Enum.concat/1","doc":"iex> Enum.concat([1..3, 4..6, 7..9])\n    [1, 2, 3, 4, 5, 6, 7, 8, 9]\n\n    iex> Enum.concat([[1, [2], 3], [4], [5, 6]])\n    [1, [2], 3, 4, 5, 6]","ref":"Enum.html#concat/1-examples"},{"type":"function","title":"Enum.concat/2","doc":"Concatenates the enumerable on the `right` with the enumerable on the\n`left`.\n\nThis function produces the same result as the `++/2` operator\nfor lists.","ref":"Enum.html#concat/2"},{"type":"function","title":"Examples - Enum.concat/2","doc":"iex> Enum.concat(1..3, 4..6)\n    [1, 2, 3, 4, 5, 6]\n\n    iex> Enum.concat([1, 2, 3], [4, 5, 6])\n    [1, 2, 3, 4, 5, 6]","ref":"Enum.html#concat/2-examples"},{"type":"function","title":"Enum.count/1","doc":"Returns the size of the `enumerable`.","ref":"Enum.html#count/1"},{"type":"function","title":"Examples - Enum.count/1","doc":"iex> Enum.count([1, 2, 3])\n    3","ref":"Enum.html#count/1-examples"},{"type":"function","title":"Enum.count/2","doc":"Returns the count of elements in the `enumerable` for which `fun` returns\na truthy value.","ref":"Enum.html#count/2"},{"type":"function","title":"Examples - Enum.count/2","doc":"iex> Enum.count([1, 2, 3, 4, 5], fn x -> rem(x, 2) == 0 end)\n    2","ref":"Enum.html#count/2-examples"},{"type":"function","title":"Enum.count_until/2","doc":"Counts the enumerable stopping at `limit`.\n\nThis is useful for checking certain properties of the count of an enumerable\nwithout having to actually count the entire enumerable. For example, if you\nwanted to check that the count was exactly, at least, or more than a value.\n\nIf the enumerable implements `c:Enumerable.count/1`, the enumerable is\nnot traversed and we return the lower of the two numbers. To force\nenumeration, use `count_until/3` with `fn _ -> true end` as the second\nargument.","ref":"Enum.html#count_until/2"},{"type":"function","title":"Examples - Enum.count_until/2","doc":"iex> Enum.count_until(1..20, 5)\n    5\n    iex> Enum.count_until(1..20, 50)\n    20\n    iex> Enum.count_until(1..10, 10) == 10 # At least 10\n    true\n    iex> Enum.count_until(1..11, 10 + 1) > 10 # More than 10\n    true\n    iex> Enum.count_until(1..5, 10)   Enum.count_until(1..10, 10 + 1) == 10 # Exactly ten\n    true","ref":"Enum.html#count_until/2-examples"},{"type":"function","title":"Enum.count_until/3","doc":"Counts the elements in the enumerable for which `fun` returns a truthy value, stopping at `limit`.\n\nSee `count/2` and `count_until/2` for more information.","ref":"Enum.html#count_until/3"},{"type":"function","title":"Examples - Enum.count_until/3","doc":"iex> Enum.count_until(1..20, fn x -> rem(x, 2) == 0 end, 7)\n    7\n    iex> Enum.count_until(1..20, fn x -> rem(x, 2) == 0 end, 11)\n    10","ref":"Enum.html#count_until/3-examples"},{"type":"function","title":"Enum.dedup/1","doc":"Enumerates the `enumerable`, returning a list where all consecutive\nduplicated elements are collapsed to a single element.\n\nElements are compared using `===/2`.\n\nIf you want to remove all duplicated elements, regardless of order,\nsee `uniq/1`.","ref":"Enum.html#dedup/1"},{"type":"function","title":"Examples - Enum.dedup/1","doc":"iex> Enum.dedup([1, 2, 3, 3, 2, 1])\n    [1, 2, 3, 2, 1]\n\n    iex> Enum.dedup([1, 1, 2, 2.0, :three, :three])\n    [1, 2, 2.0, :three]","ref":"Enum.html#dedup/1-examples"},{"type":"function","title":"Enum.dedup_by/2","doc":"Enumerates the `enumerable`, returning a list where all consecutive\nduplicated elements are collapsed to a single element.\n\nThe function `fun` maps every element to a term which is used to\ndetermine if two elements are duplicates.","ref":"Enum.html#dedup_by/2"},{"type":"function","title":"Examples - Enum.dedup_by/2","doc":"iex> Enum.dedup_by([{1, :a}, {2, :b}, {2, :c}, {1, :a}], fn {x, _} -> x end)\n    [{1, :a}, {2, :b}, {1, :a}]\n\n    iex> Enum.dedup_by([5, 1, 2, 3, 2, 1], fn x -> x > 2 end)\n    [5, 1, 3, 2]","ref":"Enum.html#dedup_by/2-examples"},{"type":"function","title":"Enum.drop/2","doc":"Drops the `amount` of elements from the `enumerable`.\n\nIf a negative `amount` is given, the `amount` of last values will be dropped.\nThe `enumerable` will be enumerated once to retrieve the proper index and\nthe remaining calculation is performed from the end.","ref":"Enum.html#drop/2"},{"type":"function","title":"Examples - Enum.drop/2","doc":"iex> Enum.drop([1, 2, 3], 2)\n    [3]\n\n    iex> Enum.drop([1, 2, 3], 10)\n    []\n\n    iex> Enum.drop([1, 2, 3], 0)\n    [1, 2, 3]\n\n    iex> Enum.drop([1, 2, 3], -1)\n    [1, 2]","ref":"Enum.html#drop/2-examples"},{"type":"function","title":"Enum.drop_every/2","doc":"Returns a list of every `nth` element in the `enumerable` dropped,\nstarting with the first element.\n\nThe first element is always dropped, unless `nth` is 0.\n\nThe second argument specifying every `nth` element must be a non-negative\ninteger.","ref":"Enum.html#drop_every/2"},{"type":"function","title":"Examples - Enum.drop_every/2","doc":"iex> Enum.drop_every(1..10, 2)\n    [2, 4, 6, 8, 10]\n\n    iex> Enum.drop_every(1..10, 0)\n    [1, 2, 3, 4, 5, 6, 7, 8, 9, 10]\n\n    iex> Enum.drop_every([1, 2, 3], 1)\n    []","ref":"Enum.html#drop_every/2-examples"},{"type":"function","title":"Enum.drop_while/2","doc":"Drops elements at the beginning of the `enumerable` while `fun` returns a\ntruthy value.","ref":"Enum.html#drop_while/2"},{"type":"function","title":"Examples - Enum.drop_while/2","doc":"iex> Enum.drop_while([1, 2, 3, 2, 1], fn x -> x < 3 end)\n    [3, 2, 1]","ref":"Enum.html#drop_while/2-examples"},{"type":"function","title":"Enum.each/2","doc":"Invokes the given `fun` for each element in the `enumerable`.\n\nReturns `:ok`.","ref":"Enum.html#each/2"},{"type":"function","title":"Examples - Enum.each/2","doc":"Enum.each([\"some\", \"example\"], fn x -> IO.puts(x) end)\n    \"some\"\n    \"example\"\n    #=> :ok","ref":"Enum.html#each/2-examples"},{"type":"function","title":"Enum.empty?/1","doc":"Determines if the `enumerable` is empty.\n\nReturns `true` if `enumerable` is empty, otherwise `false`.","ref":"Enum.html#empty?/1"},{"type":"function","title":"Examples - Enum.empty?/1","doc":"iex> Enum.empty?([])\n    true\n\n    iex> Enum.empty?([1, 2, 3])\n    false","ref":"Enum.html#empty?/1-examples"},{"type":"function","title":"Enum.fetch/2","doc":"Finds the element at the given `index` (zero-based).\n\nReturns `{:ok, element}` if found, otherwise `:error`.\n\nA negative `index` can be passed, which means the `enumerable` is\nenumerated once and the `index` is counted from the end (for example,\n`-1` fetches the last element).","ref":"Enum.html#fetch/2"},{"type":"function","title":"Examples - Enum.fetch/2","doc":"iex> Enum.fetch([2, 4, 6], 0)\n    {:ok, 2}\n\n    iex> Enum.fetch([2, 4, 6], -3)\n    {:ok, 2}\n\n    iex> Enum.fetch([2, 4, 6], 2)\n    {:ok, 6}\n\n    iex> Enum.fetch([2, 4, 6], 4)\n    :error","ref":"Enum.html#fetch/2-examples"},{"type":"function","title":"Enum.fetch!/2","doc":"Finds the element at the given `index` (zero-based).\n\nRaises `OutOfBoundsError` if the given `index` is outside the range of\nthe `enumerable`.","ref":"Enum.html#fetch!/2"},{"type":"function","title":"Examples - Enum.fetch!/2","doc":"iex> Enum.fetch!([2, 4, 6], 0)\n    2\n\n    iex> Enum.fetch!([2, 4, 6], 2)\n    6\n\n    iex> Enum.fetch!([2, 4, 6], 4)\n    ** (Enum.OutOfBoundsError) out of bounds error","ref":"Enum.html#fetch!/2-examples"},{"type":"function","title":"Enum.filter/2","doc":"Filters the `enumerable`, i.e. returns only those elements\nfor which `fun` returns a truthy value.\n\nSee also `reject/2` which discards all elements where the\nfunction returns a truthy value.","ref":"Enum.html#filter/2"},{"type":"function","title":"Examples - Enum.filter/2","doc":"iex> Enum.filter([1, 2, 3], fn x -> rem(x, 2) == 0 end)\n    [2]\n\nKeep in mind that `filter` is not capable of filtering and\ntransforming an element at the same time. If you would like\nto do so, consider using `flat_map/2`. For example, if you\nwant to convert all strings that represent an integer and\ndiscard the invalid one in one pass:\n\n    strings = [\"1234\", \"abc\", \"12ab\"]\n\n    Enum.flat_map(strings, fn string ->\n      case Integer.parse(string) do\n        # transform to integer\n        {int, _rest} -> [int]\n        # skip the value\n        :error -> []\n      end\n    end)","ref":"Enum.html#filter/2-examples"},{"type":"function","title":"Enum.find/3","doc":"Returns the first element for which `fun` returns a truthy value.\nIf no such element is found, returns `default`.","ref":"Enum.html#find/3"},{"type":"function","title":"Examples - Enum.find/3","doc":"iex> Enum.find([2, 3, 4], fn x -> rem(x, 2) == 1 end)\n    3\n\n    iex> Enum.find([2, 4, 6], fn x -> rem(x, 2) == 1 end)\n    nil\n    iex> Enum.find([2, 4, 6], 0, fn x -> rem(x, 2) == 1 end)\n    0","ref":"Enum.html#find/3-examples"},{"type":"function","title":"Enum.find_index/2","doc":"Similar to `find/3`, but returns the index (zero-based)\nof the element instead of the element itself.","ref":"Enum.html#find_index/2"},{"type":"function","title":"Examples - Enum.find_index/2","doc":"iex> Enum.find_index([2, 4, 6], fn x -> rem(x, 2) == 1 end)\n    nil\n\n    iex> Enum.find_index([2, 3, 4], fn x -> rem(x, 2) == 1 end)\n    1","ref":"Enum.html#find_index/2-examples"},{"type":"function","title":"Enum.find_value/3","doc":"Similar to `find/3`, but returns the value of the function\ninvocation instead of the element itself.\n\nThe return value is considered to be found when the result is truthy\n(neither `nil` nor `false`).","ref":"Enum.html#find_value/3"},{"type":"function","title":"Examples - Enum.find_value/3","doc":"iex> Enum.find_value([2, 3, 4], fn x ->\n    ...>   if x > 2, do: x * x\n    ...> end)\n    9\n\n    iex> Enum.find_value([2, 4, 6], fn x -> rem(x, 2) == 1 end)\n    nil\n\n    iex> Enum.find_value([2, 3, 4], fn x -> rem(x, 2) == 1 end)\n    true\n\n    iex> Enum.find_value([1, 2, 3], \"no bools!\", &is_boolean/1)\n    \"no bools!\"","ref":"Enum.html#find_value/3-examples"},{"type":"function","title":"Enum.flat_map/2","doc":"Maps the given `fun` over `enumerable` and flattens the result.\n\nThis function returns a new enumerable built by appending the result of invoking `fun`\non each element of `enumerable` together; conceptually, this is similar to a\ncombination of `map/2` and `concat/1`.","ref":"Enum.html#flat_map/2"},{"type":"function","title":"Examples - Enum.flat_map/2","doc":"iex> Enum.flat_map([:a, :b, :c], fn x -> [x, x] end)\n    [:a, :a, :b, :b, :c, :c]\n\n    iex> Enum.flat_map([{1, 3}, {4, 6}], fn {x, y} -> x..y end)\n    [1, 2, 3, 4, 5, 6]\n\n    iex> Enum.flat_map([:a, :b, :c], fn x -> [[x]] end)\n    [[:a], [:b], [:c]]","ref":"Enum.html#flat_map/2-examples"},{"type":"function","title":"Enum.flat_map_reduce/3","doc":"Maps and reduces an `enumerable`, flattening the given results (only one level deep).\n\nIt expects an accumulator and a function that receives each enumerable\nelement, and must return a tuple containing a new enumerable (often a list)\nwith the new accumulator or a tuple with `:halt` as first element and\nthe accumulator as second.","ref":"Enum.html#flat_map_reduce/3"},{"type":"function","title":"Examples - Enum.flat_map_reduce/3","doc":"iex> enumerable = 1..100\n    iex> n = 3\n    iex> Enum.flat_map_reduce(enumerable, 0, fn x, acc ->\n    ...>   if acc   end)\n    {[1, 2, 3], 3}\n\n    iex> Enum.flat_map_reduce(1..5, 0, fn x, acc -> {[[x]], acc + x} end)\n    {[[1], [2], [3], [4], [5]], 15}","ref":"Enum.html#flat_map_reduce/3-examples"},{"type":"function","title":"Enum.frequencies/1","doc":"Returns a map with keys as unique elements of `enumerable` and values\nas the count of every element.","ref":"Enum.html#frequencies/1"},{"type":"function","title":"Examples - Enum.frequencies/1","doc":"iex> Enum.frequencies(~w{ant buffalo ant ant buffalo dingo})\n    %{\"ant\" => 3, \"buffalo\" => 2, \"dingo\" => 1}","ref":"Enum.html#frequencies/1-examples"},{"type":"function","title":"Enum.frequencies_by/2","doc":"Returns a map with keys as unique elements given by `key_fun` and values\nas the count of every element.","ref":"Enum.html#frequencies_by/2"},{"type":"function","title":"Examples - Enum.frequencies_by/2","doc":"iex> Enum.frequencies_by(~w{aa aA bb cc}, &String.downcase/1)\n    %{\"aa\" => 2, \"bb\" => 1, \"cc\" => 1}\n\n    iex> Enum.frequencies_by(~w{aaa aA bbb cc c}, &String.length/1)\n    %{3 => 2, 2 => 2, 1 => 1}","ref":"Enum.html#frequencies_by/2-examples"},{"type":"function","title":"Enum.group_by/3","doc":"Splits the `enumerable` into groups based on `key_fun`.\n\nThe result is a map where each key is given by `key_fun`\nand each value is a list of elements given by `value_fun`.\nThe order of elements within each list is preserved from the `enumerable`.\nHowever, like all maps, the resulting map is unordered.","ref":"Enum.html#group_by/3"},{"type":"function","title":"Examples - Enum.group_by/3","doc":"iex> Enum.group_by(~w{ant buffalo cat dingo}, &String.length/1)\n    %{3 => [\"ant\", \"cat\"], 5 => [\"dingo\"], 7 => [\"buffalo\"]}\n\n    iex> Enum.group_by(~w{ant buffalo cat dingo}, &String.length/1, &String.first/1)\n    %{3 => [\"a\", \"c\"], 5 => [\"d\"], 7 => [\"b\"]}\n\nThe key can be any Elixir value. For example, you may use a tuple\nto group by multiple keys:\n\n    iex> collection = [\n    ...>   %{id: 1, lang: \"Elixir\", seq: 1},\n    ...>   %{id: 1, lang: \"Java\", seq: 1},\n    ...>   %{id: 1, lang: \"Ruby\", seq: 2},\n    ...>   %{id: 2, lang: \"Python\", seq: 1},\n    ...>   %{id: 2, lang: \"C#\", seq: 2},\n    ...>   %{id: 2, lang: \"Haskell\", seq: 2},\n    ...> ]\n    iex> Enum.group_by(collection, &{&1.id, &1.seq})\n    %{\n      {1, 1} => [%{id: 1, lang: \"Elixir\", seq: 1}, %{id: 1, lang: \"Java\", seq: 1}],\n      {1, 2} => [%{id: 1, lang: \"Ruby\", seq: 2}],\n      {2, 1} => [%{id: 2, lang: \"Python\", seq: 1}],\n      {2, 2} => [%{id: 2, lang: \"C#\", seq: 2}, %{id: 2, lang: \"Haskell\", seq: 2}]\n    }\n    iex> Enum.group_by(collection, &{&1.id, &1.seq}, &{&1.id, &1.lang})\n    %{\n      {1, 1} => [{1, \"Elixir\"}, {1, \"Java\"}],\n      {1, 2} => [{1, \"Ruby\"}],\n      {2, 1} => [{2, \"Python\"}],\n      {2, 2} => [{2, \"C#\"}, {2, \"Haskell\"}]\n    }","ref":"Enum.html#group_by/3-examples"},{"type":"function","title":"Enum.intersperse/2","doc":"Intersperses `separator` between each element of the enumeration.","ref":"Enum.html#intersperse/2"},{"type":"function","title":"Examples - Enum.intersperse/2","doc":"iex> Enum.intersperse([1, 2, 3], 0)\n    [1, 0, 2, 0, 3]\n\n    iex> Enum.intersperse([1], 0)\n    [1]\n\n    iex> Enum.intersperse([], 0)\n    []","ref":"Enum.html#intersperse/2-examples"},{"type":"function","title":"Enum.into/2","doc":"Inserts the given `enumerable` into a `collectable`.\n\nNote that passing a non-empty list as the `collectable` is deprecated.\nIf you're collecting into a non-empty keyword list, consider using\n`Keyword.merge(collectable, Enum.to_list(enumerable))`. If you're collecting\ninto a non-empty list, consider something like `Enum.to_list(enumerable) ++ collectable`.","ref":"Enum.html#into/2"},{"type":"function","title":"Examples - Enum.into/2","doc":"iex> Enum.into([1, 2], [])\n    [1, 2]\n\n    iex> Enum.into([a: 1, b: 2], %{})\n    %{a: 1, b: 2}\n\n    iex> Enum.into(%{a: 1}, %{b: 2})\n    %{a: 1, b: 2}\n\n    iex> Enum.into([a: 1, a: 2], %{})\n    %{a: 2}","ref":"Enum.html#into/2-examples"},{"type":"function","title":"Enum.into/3","doc":"Inserts the given `enumerable` into a `collectable` according to the\ntransformation function.","ref":"Enum.html#into/3"},{"type":"function","title":"Examples - Enum.into/3","doc":"iex> Enum.into([1, 2, 3], [], fn x -> x * 3 end)\n    [3, 6, 9]\n\n    iex> Enum.into(%{a: 1, b: 2}, %{c: 3}, fn {k, v} -> {k, v * 2} end)\n    %{a: 2, b: 4, c: 3}","ref":"Enum.html#into/3-examples"},{"type":"function","title":"Enum.join/2","doc":"Joins the given `enumerable` into a string using `joiner` as a\nseparator.\n\nIf `joiner` is not passed at all, it defaults to an empty string.\n\nAll elements in the `enumerable` must be convertible to a string,\notherwise an error is raised.","ref":"Enum.html#join/2"},{"type":"function","title":"Examples - Enum.join/2","doc":"iex> Enum.join([1, 2, 3])\n    \"123\"\n\n    iex> Enum.join([1, 2, 3], \" = \")\n    \"1 = 2 = 3\"\n\n    iex> Enum.join([[\"a\", \"b\"], [\"c\", \"d\", \"e\", [\"f\", \"g\"]], \"h\", \"i\"], \" \")\n    \"ab cdefg h i\"","ref":"Enum.html#join/2-examples"},{"type":"function","title":"Enum.map/2","doc":"Returns a list where each element is the result of invoking\n`fun` on each corresponding element of `enumerable`.\n\nFor maps, the function expects a key-value tuple.","ref":"Enum.html#map/2"},{"type":"function","title":"Examples - Enum.map/2","doc":"iex> Enum.map([1, 2, 3], fn x -> x * 2 end)\n    [2, 4, 6]\n\n    iex> Enum.map([a: 1, b: 2], fn {k, v} -> {k, -v} end)\n    [a: -1, b: -2]","ref":"Enum.html#map/2-examples"},{"type":"function","title":"Enum.map_every/3","doc":"Returns a list of results of invoking `fun` on every `nth`\nelement of `enumerable`, starting with the first element.\n\nThe first element is always passed to the given function, unless `nth` is `0`.\n\nThe second argument specifying every `nth` element must be a non-negative\ninteger.\n\nIf `nth` is `0`, then `enumerable` is directly converted to a list,\nwithout `fun` being ever applied.","ref":"Enum.html#map_every/3"},{"type":"function","title":"Examples - Enum.map_every/3","doc":"iex> Enum.map_every(1..10, 2, fn x -> x + 1000 end)\n    [1001, 2, 1003, 4, 1005, 6, 1007, 8, 1009, 10]\n\n    iex> Enum.map_every(1..10, 3, fn x -> x + 1000 end)\n    [1001, 2, 3, 1004, 5, 6, 1007, 8, 9, 1010]\n\n    iex> Enum.map_every(1..5, 0, fn x -> x + 1000 end)\n    [1, 2, 3, 4, 5]\n\n    iex> Enum.map_every([1, 2, 3], 1, fn x -> x + 1000 end)\n    [1001, 1002, 1003]","ref":"Enum.html#map_every/3-examples"},{"type":"function","title":"Enum.map_intersperse/3","doc":"Maps and intersperses the given enumerable in one pass.","ref":"Enum.html#map_intersperse/3"},{"type":"function","title":"Examples - Enum.map_intersperse/3","doc":"iex> Enum.map_intersperse([1, 2, 3], :a, &(&1 * 2))\n    [2, :a, 4, :a, 6]","ref":"Enum.html#map_intersperse/3-examples"},{"type":"function","title":"Enum.map_join/3","doc":"Maps and joins the given `enumerable` in one pass.\n\nIf `joiner` is not passed at all, it defaults to an empty string.\n\nAll elements returned from invoking the `mapper` must be convertible to\na string, otherwise an error is raised.","ref":"Enum.html#map_join/3"},{"type":"function","title":"Examples - Enum.map_join/3","doc":"iex> Enum.map_join([1, 2, 3], &(&1 * 2))\n    \"246\"\n\n    iex> Enum.map_join([1, 2, 3], \" = \", &(&1 * 2))\n    \"2 = 4 = 6\"","ref":"Enum.html#map_join/3-examples"},{"type":"function","title":"Enum.map_reduce/3","doc":"Invokes the given function to each element in the `enumerable` to reduce\nit to a single element, while keeping an accumulator.\n\nReturns a tuple where the first element is the mapped enumerable and\nthe second one is the final accumulator.\n\nThe function, `fun`, receives two arguments: the first one is the\nelement, and the second one is the accumulator. `fun` must return\na tuple with two elements in the form of `{result, accumulator}`.\n\nFor maps, the first tuple element must be a `{key, value}` tuple.","ref":"Enum.html#map_reduce/3"},{"type":"function","title":"Examples - Enum.map_reduce/3","doc":"iex> Enum.map_reduce([1, 2, 3], 0, fn x, acc -> {x * 2, x + acc} end)\n    {[2, 4, 6], 6}","ref":"Enum.html#map_reduce/3-examples"},{"type":"function","title":"Enum.max/3","doc":"Returns the maximal element in the `enumerable` according\nto Erlang's term ordering.\n\nBy default, the comparison is done with the `>=` sorter function.\nIf multiple elements are considered maximal, the first one that\nwas found is returned. If you want the last element considered\nmaximal to be returned, the sorter function should not return true\nfor equal elements.\n\nIf the enumerable is empty, the provided `empty_fallback` is called.\nThe default `empty_fallback` raises `Enum.EmptyError`.","ref":"Enum.html#max/3"},{"type":"function","title":"Examples - Enum.max/3","doc":"iex> Enum.max([1, 2, 3])\n    3\n\nThe fact this function uses Erlang's term ordering means that the comparison\nis structural and not semantic. For example:\n\n    iex> Enum.max([~D[2017-03-31], ~D[2017-04-01]])\n    ~D[2017-03-31]\n\nIn the example above, `max/2` returned March 31st instead of April 1st\nbecause the structural comparison compares the day before the year.\nFor this reason, most structs provide a \"compare\" function, such as\n`Date.compare/2`, which receives two structs and returns `:lt` (less-than),\n`:eq` (equal to), and `:gt` (greater-than). If you pass a module as the\nsorting function, Elixir will automatically use the `compare/2` function\nof said module:\n\n    iex> Enum.max([~D[2017-03-31], ~D[2017-04-01]], Date)\n    ~D[2017-04-01]\n\nFinally, if you don't want to raise on empty enumerables, you can pass\nthe empty fallback:\n\n    iex> Enum.max([], &>=/2, fn -> 0 end)\n    0","ref":"Enum.html#max/3-examples"},{"type":"function","title":"Enum.max_by/4","doc":"Returns the maximal element in the `enumerable` as calculated\nby the given `fun`.\n\nBy default, the comparison is done with the `>=` sorter function.\nIf multiple elements are considered maximal, the first one that\nwas found is returned. If you want the last element considered\nmaximal to be returned, the sorter function should not return true\nfor equal elements.\n\nCalls the provided `empty_fallback` function and returns its value if\n`enumerable` is empty. The default `empty_fallback` raises `Enum.EmptyError`.","ref":"Enum.html#max_by/4"},{"type":"function","title":"Examples - Enum.max_by/4","doc":"iex> Enum.max_by([\"a\", \"aa\", \"aaa\"], fn x -> String.length(x) end)\n    \"aaa\"\n\n    iex> Enum.max_by([\"a\", \"aa\", \"aaa\", \"b\", \"bbb\"], &String.length/1)\n    \"aaa\"\n\nThe fact this function uses Erlang's term ordering means that the\ncomparison is structural and not semantic. Therefore, if you want\nto compare structs, most structs provide a \"compare\" function, such as\n`Date.compare/2`, which receives two structs and returns `:lt` (less-than),\n`:eq` (equal to), and `:gt` (greater-than). If you pass a module as the\nsorting function, Elixir will automatically use the `compare/2` function\nof said module:\n\n    iex> users = [\n    ...>   %{name: \"Ellis\", birthday: ~D[1943-05-11]},\n    ...>   %{name: \"Lovelace\", birthday: ~D[1815-12-10]},\n    ...>   %{name: \"Turing\", birthday: ~D[1912-06-23]}\n    ...> ]\n    iex> Enum.max_by(users, &(&1.birthday), Date)\n    %{name: \"Ellis\", birthday: ~D[1943-05-11]}\n\nFinally, if you don't want to raise on empty enumerables, you can pass\nthe empty fallback:\n\n    iex> Enum.max_by([], &String.length/1, fn -> nil end)\n    nil","ref":"Enum.html#max_by/4-examples"},{"type":"function","title":"Enum.member?/2","doc":"Checks if `element` exists within the `enumerable`.\n\nMembership is tested with the match (`===/2`) operator.","ref":"Enum.html#member?/2"},{"type":"function","title":"Examples - Enum.member?/2","doc":"iex> Enum.member?(1..10, 5)\n    true\n    iex> Enum.member?(1..10, 5.0)\n    false\n\n    iex> Enum.member?([1.0, 2.0, 3.0], 2)\n    false\n    iex> Enum.member?([1.0, 2.0, 3.0], 2.000)\n    true\n\n    iex> Enum.member?([:a, :b, :c], :d)\n    false\n\n\nWhen called outside guards, the [`in`](`in/2`) and [`not in`](`in/2`)\noperators work by using this function.","ref":"Enum.html#member?/2-examples"},{"type":"function","title":"Enum.min/3","doc":"Returns the minimal element in the `enumerable` according\nto Erlang's term ordering.\n\nBy default, the comparison is done with the `<=` sorter function.\nIf multiple elements are considered minimal, the first one that\nwas found is returned. If you want the last element considered\nminimal to be returned, the sorter function should not return true\nfor equal elements.\n\nIf the enumerable is empty, the provided `empty_fallback` is called.\nThe default `empty_fallback` raises `Enum.EmptyError`.","ref":"Enum.html#min/3"},{"type":"function","title":"Examples - Enum.min/3","doc":"iex> Enum.min([1, 2, 3])\n    1\n\nThe fact this function uses Erlang's term ordering means that the comparison\nis structural and not semantic. For example:\n\n    iex> Enum.min([~D[2017-03-31], ~D[2017-04-01]])\n    ~D[2017-04-01]\n\nIn the example above, `min/2` returned April 1st instead of March 31st\nbecause the structural comparison compares the day before the year.\nFor this reason, most structs provide a \"compare\" function, such as\n`Date.compare/2`, which receives two structs and returns `:lt` (less-than),\n`:eq` (equal to), and `:gt` (greater-than). If you pass a module as the\nsorting function, Elixir will automatically use the `compare/2` function\nof said module:\n\n    iex> Enum.min([~D[2017-03-31], ~D[2017-04-01]], Date)\n    ~D[2017-03-31]\n\nFinally, if you don't want to raise on empty enumerables, you can pass\nthe empty fallback:\n\n    iex> Enum.min([], fn -> 0 end)\n    0","ref":"Enum.html#min/3-examples"},{"type":"function","title":"Enum.min_by/4","doc":"Returns the minimal element in the `enumerable` as calculated\nby the given `fun`.\n\nBy default, the comparison is done with the `<=` sorter function.\nIf multiple elements are considered minimal, the first one that\nwas found is returned. If you want the last element considered\nminimal to be returned, the sorter function should not return true\nfor equal elements.\n\nCalls the provided `empty_fallback` function and returns its value if\n`enumerable` is empty. The default `empty_fallback` raises `Enum.EmptyError`.","ref":"Enum.html#min_by/4"},{"type":"function","title":"Examples - Enum.min_by/4","doc":"iex> Enum.min_by([\"a\", \"aa\", \"aaa\"], fn x -> String.length(x) end)\n    \"a\"\n\n    iex> Enum.min_by([\"a\", \"aa\", \"aaa\", \"b\", \"bbb\"], &String.length/1)\n    \"a\"\n\nThe fact this function uses Erlang's term ordering means that the\ncomparison is structural and not semantic. Therefore, if you want\nto compare structs, most structs provide a \"compare\" function, such as\n`Date.compare/2`, which receives two structs and returns `:lt` (less-than),\n`:eq` (equal to), and `:gt` (greater-than). If you pass a module as the\nsorting function, Elixir will automatically use the `compare/2` function\nof said module:\n\n    iex> users = [\n    ...>   %{name: \"Ellis\", birthday: ~D[1943-05-11]},\n    ...>   %{name: \"Lovelace\", birthday: ~D[1815-12-10]},\n    ...>   %{name: \"Turing\", birthday: ~D[1912-06-23]}\n    ...> ]\n    iex> Enum.min_by(users, &(&1.birthday), Date)\n    %{name: \"Lovelace\", birthday: ~D[1815-12-10]}\n\nFinally, if you don't want to raise on empty enumerables, you can pass\nthe empty fallback:\n\n    iex> Enum.min_by([], &String.length/1, fn -> nil end)\n    nil","ref":"Enum.html#min_by/4-examples"},{"type":"function","title":"Enum.min_max/2","doc":"Returns a tuple with the minimal and the maximal elements in the\nenumerable according to Erlang's term ordering.\n\nIf multiple elements are considered maximal or minimal, the first one\nthat was found is returned.\n\nCalls the provided `empty_fallback` function and returns its value if\n`enumerable` is empty. The default `empty_fallback` raises `Enum.EmptyError`.","ref":"Enum.html#min_max/2"},{"type":"function","title":"Examples - Enum.min_max/2","doc":"iex> Enum.min_max([2, 3, 1])\n    {1, 3}\n\n    iex> Enum.min_max([], fn -> {nil, nil} end)\n    {nil, nil}","ref":"Enum.html#min_max/2-examples"},{"type":"function","title":"Enum.min_max_by/4","doc":"Returns a tuple with the minimal and the maximal elements in the\nenumerable as calculated by the given function.\n\nIf multiple elements are considered maximal or minimal, the first one\nthat was found is returned.","ref":"Enum.html#min_max_by/4"},{"type":"function","title":"Examples - Enum.min_max_by/4","doc":"iex> Enum.min_max_by([\"aaa\", \"bb\", \"c\"], fn x -> String.length(x) end)\n    {\"c\", \"aaa\"}\n\n    iex> Enum.min_max_by([\"aaa\", \"a\", \"bb\", \"c\", \"ccc\"], &String.length/1)\n    {\"a\", \"aaa\"}\n\n    iex> Enum.min_max_by([], &String.length/1, fn -> {nil, nil} end)\n    {nil, nil}\n\nThe fact this function uses Erlang's term ordering means that the\ncomparison is structural and not semantic. Therefore, if you want\nto compare structs, most structs provide a \"compare\" function, such as\n`Date.compare/2`, which receives two structs and returns `:lt` (less-than),\n`:eq` (equal to), and `:gt` (greater-than). If you pass a module as the\nsorting function, Elixir will automatically use the `compare/2` function\nof said module:\n\n    iex> users = [\n    ...>   %{name: \"Ellis\", birthday: ~D[1943-05-11]},\n    ...>   %{name: \"Lovelace\", birthday: ~D[1815-12-10]},\n    ...>   %{name: \"Turing\", birthday: ~D[1912-06-23]}\n    ...> ]\n    iex> Enum.min_max_by(users, &(&1.birthday), Date)\n    {\n      %{name: \"Lovelace\", birthday: ~D[1815-12-10]},\n      %{name: \"Ellis\", birthday: ~D[1943-05-11]}\n    }\n\nFinally, if you don't want to raise on empty enumerables, you can pass\nthe empty fallback:\n\n    iex> Enum.min_max_by([], &String.length/1, fn -> nil end)\n    nil","ref":"Enum.html#min_max_by/4-examples"},{"type":"function","title":"Enum.product/1","doc":"Returns the product of all elements.\n\nRaises `ArithmeticError` if `enumerable` contains a non-numeric value.","ref":"Enum.html#product/1"},{"type":"function","title":"Examples - Enum.product/1","doc":"iex> Enum.product([])\n    1\n    iex> Enum.product([2, 3, 4])\n    24\n    iex> Enum.product([2.0, 3.0, 4.0])\n    24.0","ref":"Enum.html#product/1-examples"},{"type":"function","title":"Enum.random/1","doc":"Returns a random element of an `enumerable`.\n\nRaises `Enum.EmptyError` if `enumerable` is empty.\n\nThis function uses Erlang's [`:rand` module](`:rand`) to calculate\nthe random value. Check its documentation for setting a\ndifferent random algorithm or a different seed.\n\nThe implementation is based on the\n[reservoir sampling](https://en.wikipedia.org/wiki/Reservoir_sampling#Relation_to_Fisher-Yates_shuffle)\nalgorithm.\nIt assumes that the sample being returned can fit into memory;\nthe input `enumerable` doesn't have to, as it is traversed just once.\n\nIf a range is passed into the function, this function will pick a\nrandom value between the range limits, without traversing the whole\nrange (thus executing in constant time and constant memory).","ref":"Enum.html#random/1"},{"type":"function","title":"Examples - Enum.random/1","doc":"The examples below use the `:exsss` pseudorandom algorithm since it's\nthe default from Erlang/OTP 22:\n\n    # Although not necessary, let's seed the random algorithm\n    iex> :rand.seed(:exsss, {100, 101, 102})\n    iex> Enum.random([1, 2, 3])\n    2\n    iex> Enum.random([1, 2, 3])\n    1\n    iex> Enum.random(1..1_000)\n    309","ref":"Enum.html#random/1-examples"},{"type":"function","title":"Enum.reduce/2","doc":"Invokes `fun` for each element in the `enumerable` with the\naccumulator.\n\nRaises `Enum.EmptyError` if `enumerable` is empty.\n\nThe first element of the `enumerable` is used as the initial value\nof the accumulator. Then, the function is invoked with the next\nelement and the accumulator. The result returned by the function\nis used as the accumulator for the next iteration, recursively.\nWhen the `enumerable` is done, the last accumulator is returned.\n\nSince the first element of the enumerable is used as the initial\nvalue of the accumulator, `fun` will only be executed `n - 1` times\nwhere `n` is the length of the enumerable. This function won't call\nthe specified function for enumerables that are one-element long.\n\nIf you wish to use another value for the accumulator, use\n`Enum.reduce/3`.","ref":"Enum.html#reduce/2"},{"type":"function","title":"Examples - Enum.reduce/2","doc":"iex> Enum.reduce([1, 2, 3, 4], fn x, acc -> x * acc end)\n    24","ref":"Enum.html#reduce/2-examples"},{"type":"function","title":"Enum.reduce/3","doc":"Invokes `fun` for each element in the `enumerable` with the accumulator.\n\nThe initial value of the accumulator is `acc`. The function is invoked for\neach element in the enumerable with the accumulator. The result returned\nby the function is used as the accumulator for the next iteration.\nThe function returns the last accumulator.","ref":"Enum.html#reduce/3"},{"type":"function","title":"Examples - Enum.reduce/3","doc":"iex> Enum.reduce([1, 2, 3], 0, fn x, acc -> x + acc end)\n    6\n\n    iex> Enum.reduce(%{a: 2, b: 3, c: 4}, 0, fn {_key, val}, acc -> acc + val end)\n    9","ref":"Enum.html#reduce/3-examples"},{"type":"function","title":"Reduce as a building block - Enum.reduce/3","doc":"Reduce (sometimes called `fold`) is a basic building block in functional\nprogramming. Almost all of the functions in the `Enum` module can be\nimplemented on top of reduce. Those functions often rely on other operations,\nsuch as `Enum.reverse/1`, which are optimized by the runtime.\n\nFor example, we could implement `map/2` in terms of `reduce/3` as follows:\n\n    def my_map(enumerable, fun) do\n      enumerable\n      |> Enum.reduce([], fn x, acc -> [fun.(x) | acc] end)\n      |> Enum.reverse()\n    end\n\nIn the example above, `Enum.reduce/3` accumulates the result of each call\nto `fun` into a list in reverse order, which is correctly ordered at the\nend by calling `Enum.reverse/1`.\n\nImplementing functions like `map/2`, `filter/2` and others are a good\nexercise for understanding the power behind `Enum.reduce/3`. When an\noperation cannot be expressed by any of the functions in the `Enum`\nmodule, developers will most likely resort to `reduce/3`.","ref":"Enum.html#reduce/3-reduce-as-a-building-block"},{"type":"function","title":"Enum.reduce_while/3","doc":"Reduces `enumerable` until `fun` returns `{:halt, term}`.\n\nThe return value for `fun` is expected to be\n\n  * `{:cont, acc}` to continue the reduction with `acc` as the new\n    accumulator or\n  * `{:halt, acc}` to halt the reduction\n\nIf `fun` returns `{:halt, acc}` the reduction is halted and the function\nreturns `acc`. Otherwise, if the enumerable is exhausted, the function returns\nthe accumulator of the last `{:cont, acc}`.","ref":"Enum.html#reduce_while/3"},{"type":"function","title":"Examples - Enum.reduce_while/3","doc":"iex> Enum.reduce_while(1..100, 0, fn x, acc ->\n    ...>   if x       {:cont, acc + x}\n    ...>   else\n    ...>     {:halt, acc}\n    ...>   end\n    ...> end)\n    10\n    iex> Enum.reduce_while(1..100, 0, fn x, acc ->\n    ...>   if x > 0 do\n    ...>     {:cont, acc + x}\n    ...>   else\n    ...>     {:halt, acc}\n    ...>   end\n    ...> end)\n    5050","ref":"Enum.html#reduce_while/3-examples"},{"type":"function","title":"Enum.reject/2","doc":"Returns a list of elements in `enumerable` excluding those for which the function `fun` returns\na truthy value.\n\nSee also `filter/2`.","ref":"Enum.html#reject/2"},{"type":"function","title":"Examples - Enum.reject/2","doc":"iex> Enum.reject([1, 2, 3], fn x -> rem(x, 2) == 0 end)\n    [1, 3]","ref":"Enum.html#reject/2-examples"},{"type":"function","title":"Enum.reverse/1","doc":"Returns a list of elements in `enumerable` in reverse order.","ref":"Enum.html#reverse/1"},{"type":"function","title":"Examples - Enum.reverse/1","doc":"iex> Enum.reverse([1, 2, 3])\n    [3, 2, 1]","ref":"Enum.html#reverse/1-examples"},{"type":"function","title":"Enum.reverse/2","doc":"Reverses the elements in `enumerable`, appends the `tail`, and returns\nit as a list.\n\nThis is an optimization for\n`enumerable |> Enum.reverse() |> Enum.concat(tail)`.","ref":"Enum.html#reverse/2"},{"type":"function","title":"Examples - Enum.reverse/2","doc":"iex> Enum.reverse([1, 2, 3], [4, 5, 6])\n    [3, 2, 1, 4, 5, 6]","ref":"Enum.html#reverse/2-examples"},{"type":"function","title":"Enum.reverse_slice/3","doc":"Reverses the `enumerable` in the range from initial `start_index`\nthrough `count` elements.\n\nIf `count` is greater than the size of the rest of the `enumerable`,\nthen this function will reverse the rest of the enumerable.","ref":"Enum.html#reverse_slice/3"},{"type":"function","title":"Examples - Enum.reverse_slice/3","doc":"iex> Enum.reverse_slice([1, 2, 3, 4, 5, 6], 2, 4)\n    [1, 2, 6, 5, 4, 3]","ref":"Enum.html#reverse_slice/3-examples"},{"type":"function","title":"Enum.scan/2","doc":"Applies the given function to each element in the `enumerable`,\nstoring the result in a list and passing it as the accumulator\nfor the next computation. Uses the first element in the `enumerable`\nas the starting value.","ref":"Enum.html#scan/2"},{"type":"function","title":"Examples - Enum.scan/2","doc":"iex> Enum.scan(1..5, &(&1 + &2))\n    [1, 3, 6, 10, 15]","ref":"Enum.html#scan/2-examples"},{"type":"function","title":"Enum.scan/3","doc":"Applies the given function to each element in the `enumerable`,\nstoring the result in a list and passing it as the accumulator\nfor the next computation. Uses the given `acc` as the starting value.","ref":"Enum.html#scan/3"},{"type":"function","title":"Examples - Enum.scan/3","doc":"iex> Enum.scan(1..5, 0, &(&1 + &2))\n    [1, 3, 6, 10, 15]","ref":"Enum.html#scan/3-examples"},{"type":"function","title":"Enum.shuffle/1","doc":"Returns a list with the elements of `enumerable` shuffled.\n\nThis function uses Erlang's [`:rand` module](`:rand`) to calculate\nthe random value. Check its documentation for setting a\ndifferent random algorithm or a different seed.","ref":"Enum.html#shuffle/1"},{"type":"function","title":"Examples - Enum.shuffle/1","doc":"The examples below use the `:exsss` pseudorandom algorithm since it's\nthe default from Erlang/OTP 22:\n\n    # Although not necessary, let's seed the random algorithm\n    iex> :rand.seed(:exsss, {1, 2, 3})\n    iex> Enum.shuffle([1, 2, 3])\n    [3, 2, 1]\n    iex> Enum.shuffle([1, 2, 3])\n    [2, 1, 3]","ref":"Enum.html#shuffle/1-examples"},{"type":"function","title":"Enum.slice/2","doc":"Returns a subset list of the given `enumerable` by `index_range`.\n\n`index_range` must be a `Range`. Given an `enumerable`, it drops\nelements before `index_range.first` (zero-base), then it takes elements\nuntil element `index_range.last` (inclusively).\n\nIndexes are normalized, meaning that negative indexes will be counted\nfrom the end (for example, `-1` means the last element of the `enumerable`).\n\nIf `index_range.last` is out of bounds, then it is assigned as the index\nof the last element.\n\nIf the normalized `index_range.first` is out of bounds of the given\n`enumerable`, or this one is greater than the normalized `index_range.last`,\nthen `[]` is returned.\n\nIf a step `n` (other than `1`) is used in `index_range`, then it takes\nevery `n`th element from `index_range.first` to `index_range.last`\n(according to the same rules described above).","ref":"Enum.html#slice/2"},{"type":"function","title":"Examples - Enum.slice/2","doc":"iex> Enum.slice([1, 2, 3, 4, 5], 1..3)\n    [2, 3, 4]\n\n    iex> Enum.slice([1, 2, 3, 4, 5], 3..10)\n    [4, 5]\n\n    # Last three elements (negative indexes)\n    iex> Enum.slice([1, 2, 3, 4, 5], -3..-1)\n    [3, 4, 5]\n\nFor ranges where `start > stop`, you need to explicit\nmark them as increasing:\n\n    iex> Enum.slice([1, 2, 3, 4, 5], 1..-2//1)\n    [2, 3, 4]\n\nThe step can be any positive number. For example, to\nget every 2 elements of the collection:\n\n    iex> Enum.slice([1, 2, 3, 4, 5], 0..-1//2)\n    [1, 3, 5]\n\nTo get every third element of the first ten elements:\n\n    iex> integers = Enum.to_list(1..20)\n    iex> Enum.slice(integers, 0..9//3)\n    [1, 4, 7, 10]\n\nIf the first position is after the end of the enumerable\nor after the last position of the range, it returns an\nempty list:\n\n    iex> Enum.slice([1, 2, 3, 4, 5], 6..10)\n    []\n\n    # first is greater than last\n    iex> Enum.slice([1, 2, 3, 4, 5], 6..5)\n    []","ref":"Enum.html#slice/2-examples"},{"type":"function","title":"Enum.slice/3","doc":"Returns a subset list of the given `enumerable`, from `start_index` (zero-based)\nwith `amount` number of elements if available.\n\nGiven an `enumerable`, it drops elements right before element `start_index`;\nthen, it takes `amount` of elements, returning as many elements as possible if\nthere are not enough elements.\n\nA negative `start_index` can be passed, which means the `enumerable` is\nenumerated once and the index is counted from the end (for example,\n`-1` starts slicing from the last element).\n\nIt returns `[]` if `amount` is `0` or if `start_index` is out of bounds.","ref":"Enum.html#slice/3"},{"type":"function","title":"Examples - Enum.slice/3","doc":"iex> Enum.slice(1..100, 5, 10)\n    [6, 7, 8, 9, 10, 11, 12, 13, 14, 15]\n\n    # amount to take is greater than the number of elements\n    iex> Enum.slice(1..10, 5, 100)\n    [6, 7, 8, 9, 10]\n\n    iex> Enum.slice(1..10, 5, 0)\n    []\n\n    # using a negative start index\n    iex> Enum.slice(1..10, -6, 3)\n    [5, 6, 7]\n    iex> Enum.slice(1..10, -11, 5)\n    [1, 2, 3, 4, 5]\n\n    # out of bound start index\n    iex> Enum.slice(1..10, 10, 5)\n    []","ref":"Enum.html#slice/3-examples"},{"type":"function","title":"Enum.slide/3","doc":"Slides a single or multiple elements given by `range_or_single_index` from `enumerable`\nto `insertion_index`.\n\nThe semantics of the range to be moved match the semantics of `Enum.slice/2`.\nSpecifically, that means:\n\n * Indices are normalized, meaning that negative indexes will be counted from the end\n    (for example, -1 means the last element of the enumerable). This will result in *two*\n    traversals of your enumerable on types like lists that don't provide a constant-time count.\n\n  * If the normalized index range's `last` is out of bounds, the range is truncated to the last element.\n\n  * If the normalized index range's `first` is out of bounds, the selected range for sliding\n    will be empty, so you'll get back your input list.\n\n  * Decreasing ranges (such as `5..0//1`) also select an empty range to be moved,\n    so you'll get back your input list.\n\n  * Ranges with any step but 1 will raise an error.","ref":"Enum.html#slide/3"},{"type":"function","title":"Examples - Enum.slide/3","doc":"# Slide a single element\n    iex> Enum.slide([:a, :b, :c, :d, :e, :f, :g], 5, 1)\n    [:a, :f, :b, :c, :d, :e, :g]\n\n    # Slide a range of elements backward\n    iex> Enum.slide([:a, :b, :c, :d, :e, :f, :g], 3..5, 1)\n    [:a, :d, :e, :f, :b, :c, :g]\n\n    # Slide a range of elements forward\n    iex> Enum.slide([:a, :b, :c, :d, :e, :f, :g], 1..3, 5)\n    [:a, :e, :f, :b, :c, :d, :g]\n\n    # Slide with negative indices (counting from the end)\n    iex> Enum.slide([:a, :b, :c, :d, :e, :f, :g], 3..-1//1, 2)\n    [:a, :b, :d, :e, :f, :g, :c]\n    iex> Enum.slide([:a, :b, :c, :d, :e, :f, :g], -4..-2, 1)\n    [:a, :d, :e, :f, :b, :c, :g]\n\n    # Insert at negative indices (counting from the end)\n    iex> Enum.slide([:a, :b, :c, :d, :e, :f, :g], 3, -1)\n    [:a, :b, :c, :e, :f, :g, :d]","ref":"Enum.html#slide/3-examples"},{"type":"function","title":"Enum.sort/1","doc":"Sorts the `enumerable` according to Erlang's term ordering.\n\nThis function uses the merge sort algorithm. Do not use this\nfunction to sort structs, see `sort/2` for more information.","ref":"Enum.html#sort/1"},{"type":"function","title":"Examples - Enum.sort/1","doc":"iex> Enum.sort([3, 2, 1])\n    [1, 2, 3]","ref":"Enum.html#sort/1-examples"},{"type":"function","title":"Enum.sort/2","doc":"Sorts the `enumerable` by the given function.\n\nThis function uses the merge sort algorithm. The given function should compare\ntwo arguments, and return `true` if the first argument precedes or is in the\nsame place as the second one.","ref":"Enum.html#sort/2"},{"type":"function","title":"Examples - Enum.sort/2","doc":"iex> Enum.sort([1, 2, 3], &(&1 >= &2))\n    [3, 2, 1]\n\nThe sorting algorithm will be stable as long as the given function\nreturns `true` for values considered equal:\n\n    iex> Enum.sort([\"some\", \"kind\", \"of\", \"monster\"], &(byte_size(&1)   Enum.sort([\"some\", \"kind\", \"of\", \"monster\"], &(byte_size(&1) < byte_size(&2)))\n    [\"of\", \"kind\", \"some\", \"monster\"]","ref":"Enum.html#sort/2-examples"},{"type":"function","title":"Ascending and descending (since v1.10.0) - Enum.sort/2","doc":"`sort/2` allows a developer to pass `:asc` or `:desc` as the sorter, which is a convenience for\n[`& =/2`](`>=/2`) respectively.\n\n    iex> Enum.sort([2, 3, 1], :asc)\n    [1, 2, 3]\n    iex> Enum.sort([2, 3, 1], :desc)\n    [3, 2, 1]","ref":"Enum.html#sort/2-ascending-and-descending-since-v1-10-0"},{"type":"function","title":"Sorting structs - Enum.sort/2","doc":"Do not use ` /2`, `>=/2` and friends when sorting structs.\nThat's because the built-in operators above perform structural comparison\nand not a semantic one. Imagine we sort the following list of dates:\n\n    iex> dates = [~D[2019-01-01], ~D[2020-03-02], ~D[2019-06-06]]\n    iex> Enum.sort(dates)\n    [~D[2019-01-01], ~D[2020-03-02], ~D[2019-06-06]]\n\nNote that the returned result is incorrect, because `sort/1` by default uses\n`  dates = [~D[2019-01-01], ~D[2020-03-02], ~D[2019-06-06]]\n    iex> Enum.sort(dates, Date)\n    [~D[2019-01-01], ~D[2019-06-06], ~D[2020-03-02]]\n\nTo retrieve all dates in descending order, you can wrap the module in\na tuple with `:asc` or `:desc` as first element:\n\n    iex> dates = [~D[2019-01-01], ~D[2020-03-02], ~D[2019-06-06]]\n    iex> Enum.sort(dates, {:asc, Date})\n    [~D[2019-01-01], ~D[2019-06-06], ~D[2020-03-02]]\n    iex> dates = [~D[2019-01-01], ~D[2020-03-02], ~D[2019-06-06]]\n    iex> Enum.sort(dates, {:desc, Date})\n    [~D[2020-03-02], ~D[2019-06-06], ~D[2019-01-01]]","ref":"Enum.html#sort/2-sorting-structs"},{"type":"function","title":"Enum.sort_by/3","doc":"Sorts the mapped results of the `enumerable` according to the provided `sorter`\nfunction.\n\nThis function maps each element of the `enumerable` using the\nprovided `mapper` function. The enumerable is then sorted by\nthe mapped elements using the `sorter`, which defaults to `:asc`\nand sorts the elements ascendingly.\n\n`sort_by/3` differs from `sort/2` in that it only calculates the\ncomparison value for each element in the enumerable once instead of\nonce for each element in each comparison. If the same function is\nbeing called on both elements, it's more efficient to use `sort_by/3`.","ref":"Enum.html#sort_by/3"},{"type":"function","title":"Ascending and descending (since v1.10.0) - Enum.sort_by/3","doc":"`sort_by/3` allows a developer to pass `:asc` or `:desc` as the sorter,\nwhich is a convenience for [`& =/2`](`>=/2`) respectively:\n    iex> Enum.sort_by([2, 3, 1], &(&1), :asc)\n    [1, 2, 3]\n\n    iex> Enum.sort_by([2, 3, 1], &(&1), :desc)\n    [3, 2, 1]","ref":"Enum.html#sort_by/3-ascending-and-descending-since-v1-10-0"},{"type":"function","title":"Examples - Enum.sort_by/3","doc":"Using the default `sorter` of `:asc` :\n\n    iex> Enum.sort_by([\"some\", \"kind\", \"of\", \"monster\"], &byte_size/1)\n    [\"of\", \"some\", \"kind\", \"monster\"]\n\nSorting by multiple properties - first by size, then by first letter\n(this takes advantage of the fact that tuples are compared element-by-element):\n\n    iex> Enum.sort_by([\"some\", \"kind\", \"of\", \"monster\"], &{byte_size(&1), String.first(&1)})\n    [\"of\", \"kind\", \"some\", \"monster\"]\n\nSimilar to `sort/2`, you can pass a custom sorter:\n\n    iex> Enum.sort_by([\"some\", \"kind\", \"of\", \"monster\"], &byte_size/1, :desc)\n    [\"monster\", \"some\", \"kind\", \"of\"]\n\nAs in `sort/2`, avoid using the default sorting function to sort\nstructs, as by default it performs structural comparison instead of\na semantic one. In such cases, you shall pass a sorting function as\nthird element or any module that implements a `compare/2` function.\nFor example, to sort users by their birthday in both ascending and\ndescending order respectively:\n\n    iex> users = [\n    ...>   %{name: \"Ellis\", birthday: ~D[1943-05-11]},\n    ...>   %{name: \"Lovelace\", birthday: ~D[1815-12-10]},\n    ...>   %{name: \"Turing\", birthday: ~D[1912-06-23]}\n    ...> ]\n    iex> Enum.sort_by(users, &(&1.birthday), Date)\n    [\n      %{name: \"Lovelace\", birthday: ~D[1815-12-10]},\n      %{name: \"Turing\", birthday: ~D[1912-06-23]},\n      %{name: \"Ellis\", birthday: ~D[1943-05-11]}\n    ]\n    iex> Enum.sort_by(users, &(&1.birthday), {:desc, Date})\n    [\n      %{name: \"Ellis\", birthday: ~D[1943-05-11]},\n      %{name: \"Turing\", birthday: ~D[1912-06-23]},\n      %{name: \"Lovelace\", birthday: ~D[1815-12-10]}\n    ]","ref":"Enum.html#sort_by/3-examples"},{"type":"function","title":"Performance characteristics - Enum.sort_by/3","doc":"As detailed in the initial section, `sort_by/3` calculates the comparison\nvalue for each element in the enumerable once instead of once for each\nelement in each comparison. This implies `sort_by/3` must do an initial\npass on the data to compute those values.\n\nHowever, if those values are cheap to compute, for example, you have\nalready extracted the field you want to sort by into a tuple, then those\nextra passes become overhead. In such cases, consider using `List.keysort/3`\ninstead.\n\nLet's see an example. Imagine you have a list of products and you have a\nlist of IDs. You want to keep all products that are in the given IDs and\nreturn their names sorted by their price. You could write it like this:\n\n    for(\n      product   Enum.sort_by(& &1.price)\n    |> Enum.map(& &1.name)\n\nHowever, you could also write it like this:\n\n    for(\n      product   List.keysort(1)\n    |> Enum.map(&elem(&1, 0))\n\nUsing `List.keysort/3` will be a better choice for performance sensitive\ncode as it avoids additional traversals.","ref":"Enum.html#sort_by/3-performance-characteristics"},{"type":"function","title":"Enum.split/2","doc":"Splits the `enumerable` into two enumerables, leaving `count`\nelements in the first one.\n\nIf `count` is a negative number, it starts counting from the\nback to the beginning of the `enumerable`.\n\nBe aware that a negative `count` implies the `enumerable`\nwill be enumerated twice: once to calculate the position, and\na second time to do the actual splitting.","ref":"Enum.html#split/2"},{"type":"function","title":"Examples - Enum.split/2","doc":"iex> Enum.split([1, 2, 3], 2)\n    {[1, 2], [3]}\n\n    iex> Enum.split([1, 2, 3], 10)\n    {[1, 2, 3], []}\n\n    iex> Enum.split([1, 2, 3], 0)\n    {[], [1, 2, 3]}\n\n    iex> Enum.split([1, 2, 3], -1)\n    {[1, 2], [3]}\n\n    iex> Enum.split([1, 2, 3], -5)\n    {[], [1, 2, 3]}","ref":"Enum.html#split/2-examples"},{"type":"function","title":"Enum.split_while/2","doc":"Splits enumerable in two at the position of the element for which\n`fun` returns a falsy value (`false` or `nil`) for the first time.\n\nIt returns a two-element tuple with two lists of elements.\nThe element that triggered the split is part of the second list.","ref":"Enum.html#split_while/2"},{"type":"function","title":"Examples - Enum.split_while/2","doc":"iex> Enum.split_while([1, 2, 3, 4], fn x -> x   Enum.split_while([1, 2, 3, 4], fn x -> x   Enum.split_while([1, 2, 3, 4], fn x -> x > 0 end)\n    {[1, 2, 3, 4], []}","ref":"Enum.html#split_while/2-examples"},{"type":"function","title":"Enum.split_with/2","doc":"Splits the `enumerable` in two lists according to the given function `fun`.\n\nSplits the given `enumerable` in two lists by calling `fun` with each element\nin the `enumerable` as its only argument. Returns a tuple with the first list\ncontaining all the elements in `enumerable` for which applying `fun` returned\na truthy value, and a second list with all the elements for which applying\n`fun` returned a falsy value (`false` or `nil`).\n\nThe elements in both the returned lists are in the same relative order as they\nwere in the original enumerable (if such enumerable was ordered, like a\nlist). See the examples below.","ref":"Enum.html#split_with/2"},{"type":"function","title":"Examples - Enum.split_with/2","doc":"iex> Enum.split_with([5, 4, 3, 2, 1, 0], fn x -> rem(x, 2) == 0 end)\n    {[4, 2, 0], [5, 3, 1]}\n\n    iex> Enum.split_with([a: 1, b: -2, c: 1, d: -3], fn {_k, v} -> v   Enum.split_with([a: 1, b: -2, c: 1, d: -3], fn {_k, v} -> v > 50 end)\n    {[], [a: 1, b: -2, c: 1, d: -3]}\n\n    iex> Enum.split_with([], fn {_k, v} -> v > 50 end)\n    {[], []}","ref":"Enum.html#split_with/2-examples"},{"type":"function","title":"Enum.sum/1","doc":"Returns the sum of all elements.\n\nRaises `ArithmeticError` if `enumerable` contains a non-numeric value.","ref":"Enum.html#sum/1"},{"type":"function","title":"Examples - Enum.sum/1","doc":"iex> Enum.sum([1, 2, 3])\n    6\n\n    iex> Enum.sum(1..10)\n    55\n\n    iex> Enum.sum(1..10//2)\n    25","ref":"Enum.html#sum/1-examples"},{"type":"function","title":"Enum.take/2","doc":"Takes an `amount` of elements from the beginning or the end of the `enumerable`.\n\nIf a positive `amount` is given, it takes the `amount` elements from the\nbeginning of the `enumerable`.\n\nIf a negative `amount` is given, the `amount` of elements will be taken from the end.\nThe `enumerable` will be enumerated once to retrieve the proper index and\nthe remaining calculation is performed from the end.\n\nIf amount is `0`, it returns `[]`.","ref":"Enum.html#take/2"},{"type":"function","title":"Examples - Enum.take/2","doc":"iex> Enum.take([1, 2, 3], 2)\n    [1, 2]\n\n    iex> Enum.take([1, 2, 3], 10)\n    [1, 2, 3]\n\n    iex> Enum.take([1, 2, 3], 0)\n    []\n\n    iex> Enum.take([1, 2, 3], -1)\n    [3]","ref":"Enum.html#take/2-examples"},{"type":"function","title":"Enum.take_every/2","doc":"Returns a list of every `nth` element in the `enumerable`,\nstarting with the first element.\n\nThe first element is always included, unless `nth` is 0.\n\nThe second argument specifying every `nth` element must be a non-negative\ninteger.","ref":"Enum.html#take_every/2"},{"type":"function","title":"Examples - Enum.take_every/2","doc":"iex> Enum.take_every(1..10, 2)\n    [1, 3, 5, 7, 9]\n\n    iex> Enum.take_every(1..10, 0)\n    []\n\n    iex> Enum.take_every([1, 2, 3], 1)\n    [1, 2, 3]","ref":"Enum.html#take_every/2-examples"},{"type":"function","title":"Enum.take_random/2","doc":"Takes `count` random elements from `enumerable`.\n\nNote that this function will traverse the whole `enumerable` to\nget the random sublist.\n\nSee `random/1` for notes on implementation and random seed.","ref":"Enum.html#take_random/2"},{"type":"function","title":"Examples - Enum.take_random/2","doc":"# Although not necessary, let's seed the random algorithm\n    iex> :rand.seed(:exsss, {1, 2, 3})\n    iex> Enum.take_random(1..10, 2)\n    [3, 1]\n    iex> Enum.take_random(?a..?z, 5)\n    'mikel'","ref":"Enum.html#take_random/2-examples"},{"type":"function","title":"Enum.take_while/2","doc":"Takes the elements from the beginning of the `enumerable` while `fun` returns\na truthy value.","ref":"Enum.html#take_while/2"},{"type":"function","title":"Examples - Enum.take_while/2","doc":"iex> Enum.take_while([1, 2, 3], fn x -> x < 3 end)\n    [1, 2]","ref":"Enum.html#take_while/2-examples"},{"type":"function","title":"Enum.to_list/1","doc":"Converts `enumerable` to a list.","ref":"Enum.html#to_list/1"},{"type":"function","title":"Examples - Enum.to_list/1","doc":"iex> Enum.to_list(1..3)\n    [1, 2, 3]","ref":"Enum.html#to_list/1-examples"},{"type":"function","title":"Enum.uniq/1","doc":"Enumerates the `enumerable`, removing all duplicated elements.","ref":"Enum.html#uniq/1"},{"type":"function","title":"Examples - Enum.uniq/1","doc":"iex> Enum.uniq([1, 2, 3, 3, 2, 1])\n    [1, 2, 3]","ref":"Enum.html#uniq/1-examples"},{"type":"function","title":"Enum.uniq_by/2","doc":"Enumerates the `enumerable`, by removing the elements for which\nfunction `fun` returned duplicate elements.\n\nThe function `fun` maps every element to a term. Two elements are\nconsidered duplicates if the return value of `fun` is equal for\nboth of them.\n\nThe first occurrence of each element is kept.","ref":"Enum.html#uniq_by/2"},{"type":"function","title":"Example - Enum.uniq_by/2","doc":"iex> Enum.uniq_by([{1, :x}, {2, :y}, {1, :z}], fn {x, _} -> x end)\n    [{1, :x}, {2, :y}]\n\n    iex> Enum.uniq_by([a: {:tea, 2}, b: {:tea, 2}, c: {:coffee, 1}], fn {_, y} -> y end)\n    [a: {:tea, 2}, c: {:coffee, 1}]","ref":"Enum.html#uniq_by/2-example"},{"type":"function","title":"Enum.unzip/1","doc":"Opposite of `zip/2`. Extracts two-element tuples from the\ngiven `enumerable` and groups them together.\n\nIt takes an `enumerable` with elements being two-element tuples and returns\na tuple with two lists, each of which is formed by the first and\nsecond element of each tuple, respectively.\n\nThis function fails unless `enumerable` is or can be converted into a\nlist of tuples with *exactly* two elements in each tuple.","ref":"Enum.html#unzip/1"},{"type":"function","title":"Examples - Enum.unzip/1","doc":"iex> Enum.unzip([{:a, 1}, {:b, 2}, {:c, 3}])\n    {[:a, :b, :c], [1, 2, 3]}\n\n    iex> Enum.unzip(%{a: 1, b: 2})\n    {[:a, :b], [1, 2]}","ref":"Enum.html#unzip/1-examples"},{"type":"function","title":"Enum.with_index/2","doc":"Returns the `enumerable` with each element wrapped in a tuple\nalongside its index.\n\nMay receive a function or an integer offset.\n\nIf an `offset` is given, it will index from the given offset instead of from\nzero.\n\nIf a `function` is given, it will index by invoking the function for each\nelement and index (zero-based) of the enumerable.","ref":"Enum.html#with_index/2"},{"type":"function","title":"Examples - Enum.with_index/2","doc":"iex> Enum.with_index([:a, :b, :c])\n    [a: 0, b: 1, c: 2]\n\n    iex> Enum.with_index([:a, :b, :c], 3)\n    [a: 3, b: 4, c: 5]\n\n    iex> Enum.with_index([:a, :b, :c], fn element, index -> {index, element} end)\n    [{0, :a}, {1, :b}, {2, :c}]","ref":"Enum.html#with_index/2-examples"},{"type":"function","title":"Enum.zip/1","doc":"Zips corresponding elements from a finite collection of enumerables\ninto a list of tuples.\n\nThe zipping finishes as soon as any enumerable in the given collection completes.","ref":"Enum.html#zip/1"},{"type":"function","title":"Examples - Enum.zip/1","doc":"iex> Enum.zip([[1, 2, 3], [:a, :b, :c], [\"foo\", \"bar\", \"baz\"]])\n    [{1, :a, \"foo\"}, {2, :b, \"bar\"}, {3, :c, \"baz\"}]\n\n    iex> Enum.zip([[1, 2, 3, 4, 5], [:a, :b, :c]])\n    [{1, :a}, {2, :b}, {3, :c}]","ref":"Enum.html#zip/1-examples"},{"type":"function","title":"Enum.zip/2","doc":"Zips corresponding elements from two enumerables into a list\nof tuples.\n\nThe zipping finishes as soon as either enumerable completes.","ref":"Enum.html#zip/2"},{"type":"function","title":"Examples - Enum.zip/2","doc":"iex> Enum.zip([1, 2, 3], [:a, :b, :c])\n    [{1, :a}, {2, :b}, {3, :c}]\n\n    iex> Enum.zip([1, 2, 3, 4, 5], [:a, :b, :c])\n    [{1, :a}, {2, :b}, {3, :c}]","ref":"Enum.html#zip/2-examples"},{"type":"function","title":"Enum.zip_reduce/3","doc":"Reduces over all of the given enumerables, halting as soon as any enumerable is\nempty.\n\nThe reducer will receive 2 args: a list of elements (one from each enum) and the\naccumulator.\n\nIn practice, the behaviour provided by this function can be achieved with:\n\n    Enum.reduce(Stream.zip(enums), acc, reducer)\n\nBut `zip_reduce/3` exists for convenience purposes.","ref":"Enum.html#zip_reduce/3"},{"type":"function","title":"Examples - Enum.zip_reduce/3","doc":"iex> enums = [[1, 1], [2, 2], [3, 3]]\n    ...>  Enum.zip_reduce(enums, [], fn elements, acc ->\n    ...>    [List.to_tuple(elements) | acc]\n    ...> end)\n    [{1, 2, 3}, {1, 2, 3}]\n\n    iex> enums = [[1, 2], %{a: 3, b: 4}, [5, 6]]\n    ...> Enum.zip_reduce(enums, [], fn elements, acc ->\n    ...>   [List.to_tuple(elements) | acc]\n    ...> end)\n    [{2, {:b, 4}, 6}, {1, {:a, 3}, 5}]","ref":"Enum.html#zip_reduce/3-examples"},{"type":"function","title":"Enum.zip_reduce/4","doc":"Reduces over two enumerables halting as soon as either enumerable is empty.\n\nIn practice, the behaviour provided by this function can be achieved with:\n\n    Enum.reduce(Stream.zip(left, right), acc, reducer)\n\nBut `zip_reduce/4` exists for convenience purposes.","ref":"Enum.html#zip_reduce/4"},{"type":"function","title":"Examples - Enum.zip_reduce/4","doc":"iex> Enum.zip_reduce([1, 2], [3, 4], 0, fn x, y, acc -> x + y + acc end)\n    10\n\n    iex> Enum.zip_reduce([1, 2], [3, 4], [], fn x, y, acc -> [x + y | acc] end)\n    [6, 4]","ref":"Enum.html#zip_reduce/4-examples"},{"type":"function","title":"Enum.zip_with/2","doc":"Zips corresponding elements from a finite collection of enumerables\ninto list, transforming them with the `zip_fun` function as it goes.\n\nThe first element from each of the enums in `enumerables` will be put\ninto a list which is then passed to the one-arity `zip_fun` function.\nThen, the second elements from each of the enums are put into a list\nand passed to `zip_fun`, and so on until any one of the enums in\n`enumerables` runs out of elements.\n\nReturns a list with all the results of calling `zip_fun`.","ref":"Enum.html#zip_with/2"},{"type":"function","title":"Examples - Enum.zip_with/2","doc":"iex> Enum.zip_with([[1, 2], [3, 4], [5, 6]], fn [x, y, z] -> x + y + z end)\n    [9, 12]\n\n    iex> Enum.zip_with([[1, 2], [3, 4]], fn [x, y] -> x + y end)\n    [4, 6]","ref":"Enum.html#zip_with/2-examples"},{"type":"function","title":"Enum.zip_with/3","doc":"Zips corresponding elements from two enumerables into a list, transforming them with\nthe `zip_fun` function as it goes.\n\nThe corresponding elements from each collection are passed to the provided two-arity `zip_fun`\nfunction in turn. Returns a list that contains the result of calling `zip_fun` for each pair of\nelements.\n\nThe zipping finishes as soon as either enumerable runs out of elements.","ref":"Enum.html#zip_with/3"},{"type":"function","title":"Zipping Maps - Enum.zip_with/3","doc":"It's important to remember that zipping inherently relies on order.\nIf you zip two lists you get the element at the index from each list in turn.\nIf we zip two maps together it's tempting to think that you will get the given\nkey in the left map and the matching key in the right map, but there is no such\nguarantee because map keys are not ordered! Consider the following:\n\n    left =  %{:a => 1, 1 => 3}\n    right = %{:a => 1, :b => :c}\n    Enum.zip(left, right)\n    # [{{1, 3}, {:a, 1}}, {{:a, 1}, {:b, :c}}]\n\nAs you can see `:a` does not get paired with `:a`. If this is what you want,\nyou should use `Map.merge/3`.","ref":"Enum.html#zip_with/3-zipping-maps"},{"type":"function","title":"Examples - Enum.zip_with/3","doc":"iex> Enum.zip_with([1, 2], [3, 4], fn x, y -> x + y end)\n    [4, 6]\n\n    iex> Enum.zip_with([1, 2], [3, 4, 5, 6], fn x, y -> x + y end)\n    [4, 6]\n\n    iex> Enum.zip_with([1, 2, 5, 6], [3, 4], fn x, y -> x + y end)\n    [4, 6]","ref":"Enum.html#zip_with/3-examples"},{"type":"type","title":"Enum.acc/0","doc":"","ref":"Enum.html#t:acc/0"},{"type":"type","title":"Enum.default/0","doc":"","ref":"Enum.html#t:default/0"},{"type":"type","title":"Enum.element/0","doc":"","ref":"Enum.html#t:element/0"},{"type":"type","title":"Enum.index/0","doc":"Zero-based index. It can also be a negative integer.","ref":"Enum.html#t:index/0"},{"type":"type","title":"Enum.t/0","doc":"","ref":"Enum.html#t:t/0"},{"type":"module","title":"Keyword","doc":"A keyword list is a list that consists exclusively of two-element tuples.\n\nThe first element of these tuples is known as the *key*, and it must be an atom.\nThe second element, known as the *value*, can be any term.\n\nKeywords are mostly used to work with optional values.","ref":"Keyword.html"},{"type":"module","title":"Examples - Keyword","doc":"For example, the following is a keyword list:\n\n    [{:exit_on_close, true}, {:active, :once}, {:packet_size, 1024}]\n\nElixir provides a special and more concise syntax for keyword lists:\n\n    [exit_on_close: true, active: :once, packet_size: 1024]\n\nThe two syntaxes return the exact same value.\n\nA *key* can be any atom, consisting of Unicode letters, numbers,\nan underscore or the `@` sign. If the *key* should have any other\ncharacters, such as spaces, you can wrap it in quotes:\n\n    iex> [\"exit on close\": true]\n    [\"exit on close\": true]\n\nWrapping an atom in quotes does not make it a string. Keyword list\n*keys* are always atoms. Quotes should only be used when necessary\nor Elixir will issue a warning.","ref":"Keyword.html#module-examples"},{"type":"module","title":"Duplicate keys and ordering - Keyword","doc":"A keyword may have duplicate keys so it is not strictly a key-value\ndata type. However, most of the functions in this module work on a\nkey-value structure and behave similar to the functions you would\nfind in the `Map` module. For example, `Keyword.get/3` will get the first\nentry matching the given key, regardless if duplicate entries exist.\nSimilarly, `Keyword.put/3` and `Keyword.delete/2` ensure all duplicate\nentries for a given key are removed when invoked. Note, however, that\nkeyword list operations need to traverse the whole list in order to find\nkeys, so these operations are slower than their map counterparts.\n\nA handful of functions exist to handle duplicate keys, for example,\n`get_values/2` returns all values for a given key and `delete_first/2`\ndeletes just the first entry of the existing ones.\n\nEven though lists preserve the existing order, the functions in\n`Keyword` do not guarantee any ordering. For example, if you invoke\n`Keyword.put(opts, new_key, new_value)`, there is no guarantee for\nwhere `new_key` will be added to (the front, the end or anywhere else).\n\nGiven ordering is not guaranteed, it is not recommended to pattern\nmatch on keyword lists either. For example, a function such as:\n\n    def my_function([some_key: value, another_key: another_value])\n\nwill match\n\n    my_function([some_key: :foo, another_key: :bar])\n\nbut it won't match\n\n    my_function([another_key: :bar, some_key: :foo])\n\nMost of the functions in this module work in linear time. This means\nthat the time it takes to perform an operation grows at the same\nrate as the length of the list.","ref":"Keyword.html#module-duplicate-keys-and-ordering"},{"type":"module","title":"Call syntax - Keyword","doc":"When keyword lists are passed as the last argument to a function,\nthe square brackets around the keyword list can be omitted. For\nexample, the keyword list syntax:\n\n    String.split(\"1-0\", \"-\", [trim: true, parts: 2])\n\ncan be written without the enclosing brackets whenever it is the last\nargument of a function call:\n\n    String.split(\"1-0\", \"-\", trim: true, parts: 2)\n\nSince tuples, lists and maps are treated similarly to function\narguments in Elixir syntax, this property is also available to them:\n\n    iex> {1, 2, foo: :bar}\n    {1, 2, [{:foo, :bar}]}\n\n    iex> [1, 2, foo: :bar]\n    [1, 2, {:foo, :bar}]\n\n    iex> %{1 => 2, foo: :bar}\n    %{1 => 2, :foo => :bar}","ref":"Keyword.html#module-call-syntax"},{"type":"function","title":"Keyword.delete/2","doc":"Deletes the entries in the keyword list under a specific `key`.\n\nIf the `key` does not exist, it returns the keyword list unchanged.\nUse `delete_first/2` to delete just the first entry in case of\nduplicate keys.","ref":"Keyword.html#delete/2"},{"type":"function","title":"Examples - Keyword.delete/2","doc":"iex> Keyword.delete([a: 1, b: 2], :a)\n    [b: 2]\n    iex> Keyword.delete([a: 1, b: 2, a: 3], :a)\n    [b: 2]\n    iex> Keyword.delete([b: 2], :a)\n    [b: 2]","ref":"Keyword.html#delete/2-examples"},{"type":"function","title":"Keyword.delete_first/2","doc":"Deletes the first entry in the keyword list under a specific `key`.\n\nIf the `key` does not exist, it returns the keyword list unchanged.","ref":"Keyword.html#delete_first/2"},{"type":"function","title":"Examples - Keyword.delete_first/2","doc":"iex> Keyword.delete_first([a: 1, b: 2, a: 3], :a)\n    [b: 2, a: 3]\n    iex> Keyword.delete_first([b: 2], :a)\n    [b: 2]","ref":"Keyword.html#delete_first/2-examples"},{"type":"function","title":"Keyword.drop/2","doc":"Drops the given `keys` from the keyword list.\n\nRemoves duplicate keys from the new keyword list.","ref":"Keyword.html#drop/2"},{"type":"function","title":"Examples - Keyword.drop/2","doc":"iex> Keyword.drop([a: 1, a: 2], [:a])\n    []\n    iex> Keyword.drop([a: 1, b: 2, c: 3], [:b, :d])\n    [a: 1, c: 3]\n    iex> Keyword.drop([a: 1, b: 2, b: 3, c: 3, a: 5], [:b, :d])\n    [a: 1, c: 3, a: 5]","ref":"Keyword.html#drop/2-examples"},{"type":"function","title":"Keyword.equal?/2","doc":"Checks if two keywords are equal.\n\nConsiders two keywords to be equal if they contain\nthe same keys and those keys contain the same values.","ref":"Keyword.html#equal?/2"},{"type":"function","title":"Examples - Keyword.equal?/2","doc":"iex> Keyword.equal?([a: 1, b: 2], [b: 2, a: 1])\n    true\n    iex> Keyword.equal?([a: 1, b: 2], [b: 1, a: 2])\n    false\n    iex> Keyword.equal?([a: 1, b: 2, a: 3], [b: 2, a: 3, a: 1])\n    true\n\nComparison between values is done with `===/3`,\nwhich means integers are not equivalent to floats:\n\n    iex> Keyword.equal?([a: 1.0], [a: 1])\n    false","ref":"Keyword.html#equal?/2-examples"},{"type":"function","title":"Keyword.fetch/2","doc":"Fetches the value for a specific `key` and returns it in a tuple.\n\nIf the `key` does not exist, it returns `:error`.","ref":"Keyword.html#fetch/2"},{"type":"function","title":"Examples - Keyword.fetch/2","doc":"iex> Keyword.fetch([a: 1], :a)\n    {:ok, 1}\n    iex> Keyword.fetch([a: 1], :b)\n    :error","ref":"Keyword.html#fetch/2-examples"},{"type":"function","title":"Keyword.fetch!/2","doc":"Fetches the value for specific `key`.\n\nIf the `key` does not exist, it raises a `KeyError`.","ref":"Keyword.html#fetch!/2"},{"type":"function","title":"Examples - Keyword.fetch!/2","doc":"iex> Keyword.fetch!([a: 1], :a)\n    1\n    iex> Keyword.fetch!([a: 1], :b)\n    ** (KeyError) key :b not found in: [a: 1]","ref":"Keyword.html#fetch!/2-examples"},{"type":"function","title":"Keyword.filter/2","doc":"Returns a keyword list containing only the entries from `keywords`\nfor which the function `fun` returns a truthy value.\n\nSee also `reject/2` which discards all entries where the function\nreturns a truthy value.","ref":"Keyword.html#filter/2"},{"type":"function","title":"Examples - Keyword.filter/2","doc":"iex> Keyword.filter([one: 1, two: 2, three: 3], fn {_key, val} -> rem(val, 2) == 1 end)\n    [one: 1, three: 3]","ref":"Keyword.html#filter/2-examples"},{"type":"function","title":"Keyword.from_keys/2","doc":"Builds a keyword from the given `keys` and the fixed `value`.","ref":"Keyword.html#from_keys/2"},{"type":"function","title":"Examples - Keyword.from_keys/2","doc":"iex> Keyword.from_keys([:foo, :bar, :baz], :atom)\n    [foo: :atom, bar: :atom, baz: :atom]","ref":"Keyword.html#from_keys/2-examples"},{"type":"function","title":"Keyword.get/3","doc":"Gets the value under the given `key`.\n\nReturns the default value if `key` does not exist\n(`nil` if no default value is provided).\n\nIf duplicate entries exist, it returns the first one.\nUse `get_values/2` to retrieve all entries.","ref":"Keyword.html#get/3"},{"type":"function","title":"Examples - Keyword.get/3","doc":"iex> Keyword.get([], :a)\n    nil\n    iex> Keyword.get([a: 1], :a)\n    1\n    iex> Keyword.get([a: 1], :b)\n    nil\n    iex> Keyword.get([a: 1], :b, 3)\n    3\n\nWith duplicate keys:\n\n    iex> Keyword.get([a: 1, a: 2], :a, 3)\n    1\n    iex> Keyword.get([a: 1, a: 2], :b, 3)\n    3","ref":"Keyword.html#get/3-examples"},{"type":"function","title":"Keyword.get_and_update/3","doc":"Gets the value from `key` and updates it, all in one pass.\n\nThe `fun` argument receives the value of `key` (or `nil` if `key`\nis not present) and must return a two-element tuple: the current value\n(the retrieved value, which can be operated on before being returned)\nand the new value to be stored under `key`. The `fun` may also\nreturn `:pop`, implying the current value shall be removed from the\nkeyword list and returned.\n\nReturns a tuple that contains the current value returned by\n`fun` and a new keyword list with the updated value under `key`.","ref":"Keyword.html#get_and_update/3"},{"type":"function","title":"Examples - Keyword.get_and_update/3","doc":"iex> Keyword.get_and_update([a: 1], :a, fn current_value ->\n    ...>   {current_value, \"new value!\"}\n    ...> end)\n    {1, [a: \"new value!\"]}\n\n    iex> Keyword.get_and_update([a: 1], :b, fn current_value ->\n    ...>   {current_value, \"new value!\"}\n    ...> end)\n    {nil, [b: \"new value!\", a: 1]}\n\n    iex> Keyword.get_and_update([a: 2], :a, fn number ->\n    ...>   {2 * number, 3 * number}\n    ...> end)\n    {4, [a: 6]}\n\n    iex> Keyword.get_and_update([a: 1], :a, fn _ -> :pop end)\n    {1, []}\n\n    iex> Keyword.get_and_update([a: 1], :b, fn _ -> :pop end)\n    {nil, [a: 1]}","ref":"Keyword.html#get_and_update/3-examples"},{"type":"function","title":"Keyword.get_and_update!/3","doc":"Gets the value under `key` and updates it. Raises if there is no `key`.\n\nThe `fun` argument receives the value under `key` and must return a\ntwo-element tuple: the current value (the retrieved value, which can be\noperated on before being returned) and the new value to be stored under\n`key`.\n\nReturns a tuple that contains the current value returned by\n`fun` and a new keyword list with the updated value under `key`.","ref":"Keyword.html#get_and_update!/3"},{"type":"function","title":"Examples - Keyword.get_and_update!/3","doc":"iex> Keyword.get_and_update!([a: 1], :a, fn current_value ->\n    ...>   {current_value, \"new value!\"}\n    ...> end)\n    {1, [a: \"new value!\"]}\n\n    iex> Keyword.get_and_update!([a: 1], :b, fn current_value ->\n    ...>   {current_value, \"new value!\"}\n    ...> end)\n    ** (KeyError) key :b not found in: [a: 1]\n\n    iex> Keyword.get_and_update!([a: 1], :a, fn _ ->\n    ...>   :pop\n    ...> end)\n    {1, []}","ref":"Keyword.html#get_and_update!/3-examples"},{"type":"function","title":"Keyword.get_lazy/3","doc":"Gets the value under the given `key`.\n\nIf `key` does not exist, lazily evaluates `fun` and returns its result.\n\nThis is useful if the default value is very expensive to calculate or\ngenerally difficult to set up and tear down again.\n\nIf duplicate entries exist, it returns the first one.\nUse `get_values/2` to retrieve all entries.","ref":"Keyword.html#get_lazy/3"},{"type":"function","title":"Examples - Keyword.get_lazy/3","doc":"iex> keyword = [a: 1]\n    iex> fun = fn ->\n    ...>   # some expensive operation here\n    ...>   13\n    ...> end\n    iex> Keyword.get_lazy(keyword, :a, fun)\n    1\n    iex> Keyword.get_lazy(keyword, :b, fun)\n    13","ref":"Keyword.html#get_lazy/3-examples"},{"type":"function","title":"Keyword.get_values/2","doc":"Gets all values under a specific `key`.","ref":"Keyword.html#get_values/2"},{"type":"function","title":"Examples - Keyword.get_values/2","doc":"iex> Keyword.get_values([], :a)\n    []\n    iex> Keyword.get_values([a: 1], :a)\n    [1]\n    iex> Keyword.get_values([a: 1, a: 2], :a)\n    [1, 2]","ref":"Keyword.html#get_values/2-examples"},{"type":"function","title":"Keyword.has_key?/2","doc":"Returns whether a given `key` exists in the given `keywords`.","ref":"Keyword.html#has_key?/2"},{"type":"function","title":"Examples - Keyword.has_key?/2","doc":"iex> Keyword.has_key?([a: 1], :a)\n    true\n    iex> Keyword.has_key?([a: 1], :b)\n    false","ref":"Keyword.html#has_key?/2-examples"},{"type":"function","title":"Keyword.keys/1","doc":"Returns all keys from the keyword list.\n\nKeeps duplicate keys in the resulting list of keys.","ref":"Keyword.html#keys/1"},{"type":"function","title":"Examples - Keyword.keys/1","doc":"iex> Keyword.keys(a: 1, b: 2)\n    [:a, :b]\n\n    iex> Keyword.keys(a: 1, b: 2, a: 3)\n    [:a, :b, :a]\n\n    iex> Keyword.keys([{:a, 1}, {\"b\", 2}, {:c, 3}])\n    ** (ArgumentError) expected a keyword list, but an entry in the list is not a two-element tuple with an atom as its first element, got: {\"b\", 2}","ref":"Keyword.html#keys/1-examples"},{"type":"function","title":"Keyword.keyword?/1","doc":"Returns `true` if `term` is a keyword list, otherwise `false`.\n\nWhen `term` is a list it is traversed to the end.","ref":"Keyword.html#keyword?/1"},{"type":"function","title":"Examples - Keyword.keyword?/1","doc":"iex> Keyword.keyword?([])\n    true\n    iex> Keyword.keyword?(a: 1)\n    true\n    iex> Keyword.keyword?([{Foo, 1}])\n    true\n    iex> Keyword.keyword?([{}])\n    false\n    iex> Keyword.keyword?([:key])\n    false\n    iex> Keyword.keyword?(%{})\n    false","ref":"Keyword.html#keyword?/1-examples"},{"type":"function","title":"Keyword.merge/2","doc":"Merges two keyword lists into one.\n\nAdds all keys, including duplicate keys, given in `keywords2`\nto `keywords1`, overriding any existing ones.\n\nThere are no guarantees about the order of the keys in the returned keyword.","ref":"Keyword.html#merge/2"},{"type":"function","title":"Examples - Keyword.merge/2","doc":"iex> Keyword.merge([a: 1, b: 2], [a: 3, d: 4])\n    [b: 2, a: 3, d: 4]\n\n    iex> Keyword.merge([a: 1, b: 2], [a: 3, d: 4, a: 5])\n    [b: 2, a: 3, d: 4, a: 5]\n\n    iex> Keyword.merge([a: 1], [2, 3])\n    ** (ArgumentError) expected a keyword list as the second argument, got: [2, 3]","ref":"Keyword.html#merge/2-examples"},{"type":"function","title":"Keyword.merge/3","doc":"Merges two keyword lists into one.\n\nAdds all keys, including duplicate keys, given in `keywords2`\nto `keywords1`. Invokes the given function to solve conflicts.\n\nIf `keywords2` has duplicate keys, it invokes the given function\nfor each matching pair in `keywords1`.\n\nThere are no guarantees about the order of the keys in the returned keyword.","ref":"Keyword.html#merge/3"},{"type":"function","title":"Examples - Keyword.merge/3","doc":"iex> Keyword.merge([a: 1, b: 2], [a: 3, d: 4], fn _k, v1, v2 ->\n    ...>   v1 + v2\n    ...> end)\n    [b: 2, a: 4, d: 4]\n\n    iex> Keyword.merge([a: 1, b: 2], [a: 3, d: 4, a: 5], fn :a, v1, v2 ->\n    ...>   v1 + v2\n    ...> end)\n    [b: 2, a: 4, d: 4, a: 5]\n\n    iex> Keyword.merge([a: 1, b: 2, a: 3], [a: 3, d: 4, a: 5], fn :a, v1, v2 ->\n    ...>   v1 + v2\n    ...> end)\n    [b: 2, a: 4, d: 4, a: 8]\n\n    iex> Keyword.merge([a: 1, b: 2], [:a, :b], fn :a, v1, v2 ->\n    ...>   v1 + v2\n    ...> end)\n    ** (ArgumentError) expected a keyword list as the second argument, got: [:a, :b]","ref":"Keyword.html#merge/3-examples"},{"type":"function","title":"Keyword.new/0","doc":"Returns an empty keyword list, i.e. an empty list.","ref":"Keyword.html#new/0"},{"type":"function","title":"Examples - Keyword.new/0","doc":"iex> Keyword.new()\n    []","ref":"Keyword.html#new/0-examples"},{"type":"function","title":"Keyword.new/1","doc":"Creates a keyword list from an enumerable.\n\nRemoves duplicate entries and the last one prevails.\nUnlike `Enum.into(enumerable, [])`, `Keyword.new(enumerable)`\nguarantees the keys are unique.","ref":"Keyword.html#new/1"},{"type":"function","title":"Examples - Keyword.new/1","doc":"iex> Keyword.new([{:b, 1}, {:a, 2}])\n    [b: 1, a: 2]\n\n    iex> Keyword.new([{:a, 1}, {:a, 2}, {:a, 3}])\n    [a: 3]","ref":"Keyword.html#new/1-examples"},{"type":"function","title":"Keyword.new/2","doc":"Creates a keyword list from an enumerable via the transformation function.\n\nRemoves duplicate entries and the last one prevails.\nUnlike `Enum.into(enumerable, [], fun)`,\n`Keyword.new(enumerable, fun)` guarantees the keys are unique.","ref":"Keyword.html#new/2"},{"type":"function","title":"Examples - Keyword.new/2","doc":"iex> Keyword.new([:a, :b], fn x -> {x, x} end)\n    [a: :a, b: :b]","ref":"Keyword.html#new/2-examples"},{"type":"function","title":"Keyword.pop/3","doc":"Returns the first value for `key` and removes all associated entries in the keyword list.\n\nIt returns a tuple where the first element is the first value for `key` and the\nsecond element is a keyword list with all entries associated with `key` removed.\nIf the `key` is not present in the keyword list, it returns `{default, keyword_list}`.\n\nIf you don't want to remove all the entries associated with `key` use `pop_first/3`\ninstead, which will remove only the first entry.","ref":"Keyword.html#pop/3"},{"type":"function","title":"Examples - Keyword.pop/3","doc":"iex> Keyword.pop([a: 1], :a)\n    {1, []}\n    iex> Keyword.pop([a: 1], :b)\n    {nil, [a: 1]}\n    iex> Keyword.pop([a: 1], :b, 3)\n    {3, [a: 1]}\n    iex> Keyword.pop([a: 1, a: 2], :a)\n    {1, []}","ref":"Keyword.html#pop/3-examples"},{"type":"function","title":"Keyword.pop!/2","doc":"Returns the first value for `key` and removes all associated entries in the keyword list,\nraising if `key` is not present.\n\nThis function behaves like `pop/3`, but raises in case the `key` is not present in the\ngiven `keywords`.","ref":"Keyword.html#pop!/2"},{"type":"function","title":"Examples - Keyword.pop!/2","doc":"iex> Keyword.pop!([a: 1], :a)\n    {1, []}\n    iex> Keyword.pop!([a: 1, a: 2], :a)\n    {1, []}\n    iex> Keyword.pop!([a: 1], :b)\n    ** (KeyError) key :b not found in: [a: 1]","ref":"Keyword.html#pop!/2-examples"},{"type":"function","title":"Keyword.pop_first/3","doc":"Returns and removes the first value associated with `key` in the keyword list.\n\nKeeps duplicate keys in the resulting keyword list.","ref":"Keyword.html#pop_first/3"},{"type":"function","title":"Examples - Keyword.pop_first/3","doc":"iex> Keyword.pop_first([a: 1], :a)\n    {1, []}\n    iex> Keyword.pop_first([a: 1], :b)\n    {nil, [a: 1]}\n    iex> Keyword.pop_first([a: 1], :b, 3)\n    {3, [a: 1]}\n    iex> Keyword.pop_first([a: 1, a: 2], :a)\n    {1, [a: 2]}","ref":"Keyword.html#pop_first/3-examples"},{"type":"function","title":"Keyword.pop_lazy/3","doc":"Lazily returns and removes all values associated with `key` in the keyword list.\n\nThis is useful if the default value is very expensive to calculate or\ngenerally difficult to set up and tear down again.\n\nRemoves all duplicate keys. See `pop_first/3` for removing only the first entry.","ref":"Keyword.html#pop_lazy/3"},{"type":"function","title":"Examples - Keyword.pop_lazy/3","doc":"iex> keyword = [a: 1]\n    iex> fun = fn ->\n    ...>   # some expensive operation here\n    ...>   13\n    ...> end\n    iex> Keyword.pop_lazy(keyword, :a, fun)\n    {1, []}\n    iex> Keyword.pop_lazy(keyword, :b, fun)\n    {13, [a: 1]}","ref":"Keyword.html#pop_lazy/3-examples"},{"type":"function","title":"Keyword.pop_values/2","doc":"Returns all values for `key` and removes all associated entries in the keyword list.\n\nIt returns a tuple where the first element is a list of values for `key` and the\nsecond element is a keyword list with all entries associated with `key` removed.\nIf the `key` is not present in the keyword list, it returns `{[], keyword_list}`.\n\nIf you don't want to remove all the entries associated with `key` use `pop_first/3`\ninstead, which will remove only the first entry.","ref":"Keyword.html#pop_values/2"},{"type":"function","title":"Examples - Keyword.pop_values/2","doc":"iex> Keyword.pop_values([a: 1], :a)\n    {[1], []}\n    iex> Keyword.pop_values([a: 1], :b)\n    {[], [a: 1]}\n    iex> Keyword.pop_values([a: 1, a: 2], :a)\n    {[1, 2], []}","ref":"Keyword.html#pop_values/2-examples"},{"type":"function","title":"Keyword.put/3","doc":"Puts the given `value` under the specified `key`.\n\nIf a value under `key` already exists, it overrides the value\nand removes all duplicate entries.","ref":"Keyword.html#put/3"},{"type":"function","title":"Examples - Keyword.put/3","doc":"iex> Keyword.put([a: 1], :b, 2)\n    [b: 2, a: 1]\n    iex> Keyword.put([a: 1, b: 2], :a, 3)\n    [a: 3, b: 2]\n    iex> Keyword.put([a: 1, b: 2, a: 4], :a, 3)\n    [a: 3, b: 2]","ref":"Keyword.html#put/3-examples"},{"type":"function","title":"Keyword.put_new/3","doc":"Puts the given `value` under `key`, unless the entry `key` already exists.","ref":"Keyword.html#put_new/3"},{"type":"function","title":"Examples - Keyword.put_new/3","doc":"iex> Keyword.put_new([a: 1], :b, 2)\n    [b: 2, a: 1]\n    iex> Keyword.put_new([a: 1, b: 2], :a, 3)\n    [a: 1, b: 2]","ref":"Keyword.html#put_new/3-examples"},{"type":"function","title":"Keyword.put_new_lazy/3","doc":"Evaluates `fun` and puts the result under `key`\nin keyword list unless `key` is already present.\n\nThis is useful if the value is very expensive to calculate or\ngenerally difficult to set up and tear down again.","ref":"Keyword.html#put_new_lazy/3"},{"type":"function","title":"Examples - Keyword.put_new_lazy/3","doc":"iex> keyword = [a: 1]\n    iex> fun = fn ->\n    ...>   # some expensive operation here\n    ...>   13\n    ...> end\n    iex> Keyword.put_new_lazy(keyword, :a, fun)\n    [a: 1]\n    iex> Keyword.put_new_lazy(keyword, :b, fun)\n    [b: 13, a: 1]","ref":"Keyword.html#put_new_lazy/3-examples"},{"type":"function","title":"Keyword.reject/2","doc":"Returns a keyword list excluding the entries from `keywords`\nfor which the function `fun` returns a truthy value.\n\nSee also `filter/2`.","ref":"Keyword.html#reject/2"},{"type":"function","title":"Examples - Keyword.reject/2","doc":"iex> Keyword.reject([one: 1, two: 2, three: 3], fn {_key, val} -> rem(val, 2) == 1 end)\n    [two: 2]","ref":"Keyword.html#reject/2-examples"},{"type":"function","title":"Keyword.replace/3","doc":"Puts a value under `key` only if the `key` already exists in `keywords`.\n\nIn case a key exists multiple times in the keyword list,\nit removes later occurrences.","ref":"Keyword.html#replace/3"},{"type":"function","title":"Examples - Keyword.replace/3","doc":"iex> Keyword.replace([a: 1, b: 2, a: 4], :a, 3)\n    [a: 3, b: 2]\n\n    iex> Keyword.replace([a: 1], :b, 2)\n    [a: 1]","ref":"Keyword.html#replace/3-examples"},{"type":"function","title":"Keyword.replace!/3","doc":"Puts a value under `key` only if the `key` already exists in `keywords`.\n\nIf `key` is not present in `keywords`, it raises a `KeyError`.","ref":"Keyword.html#replace!/3"},{"type":"function","title":"Examples - Keyword.replace!/3","doc":"iex> Keyword.replace!([a: 1, b: 2, a: 3], :a, :new)\n    [a: :new, b: 2]\n    iex> Keyword.replace!([a: 1, b: 2, c: 3, b: 4], :b, :new)\n    [a: 1, b: :new, c: 3]\n\n    iex> Keyword.replace!([a: 1], :b, 2)\n    ** (KeyError) key :b not found in: [a: 1]","ref":"Keyword.html#replace!/3-examples"},{"type":"function","title":"Keyword.replace_lazy/3","doc":"Replaces the value under `key` using the given function only if\n`key` already exists in `keywords`.\n\nIn comparison to `replace/3`, this can be useful when it's expensive to calculate the value.\n\nIf `key` does not exist, the original keyword list is returned unchanged.","ref":"Keyword.html#replace_lazy/3"},{"type":"function","title":"Examples - Keyword.replace_lazy/3","doc":"iex> Keyword.replace_lazy([a: 1, b: 2], :a, fn v -> v * 4 end)\n    [a: 4, b: 2]\n\n    iex> Keyword.replace_lazy([a: 2, b: 2, a: 1], :a, fn v -> v * 4 end)\n    [a: 8, b: 2]\n\n    iex> Keyword.replace_lazy([a: 1, b: 2], :c, fn v -> v * 4 end)\n    [a: 1, b: 2]","ref":"Keyword.html#replace_lazy/3-examples"},{"type":"function","title":"Keyword.split/2","doc":"Takes all entries corresponding to the given `keys` and extracts them into a\nseparate keyword list.\n\nReturns a tuple with the new list and the old list with removed keys.\n\nIgnores keys for which there are no entries in the keyword list.\n\nEntries with duplicate keys end up in the same keyword list.","ref":"Keyword.html#split/2"},{"type":"function","title":"Examples - Keyword.split/2","doc":"iex> Keyword.split([a: 1, b: 2, c: 3], [:a, :c, :e])\n    {[a: 1, c: 3], [b: 2]}\n    iex> Keyword.split([a: 1, b: 2, c: 3, a: 4], [:a, :c, :e])\n    {[a: 1, c: 3, a: 4], [b: 2]}","ref":"Keyword.html#split/2-examples"},{"type":"function","title":"Keyword.split_with/2","doc":"Splits the `keywords` into two keyword lists according to the given function\n`fun`.\n\nThe provided `fun` receives each `{key, value}` pair in the `keywords` as its only\nargument. Returns a tuple with the first keyword list containing all the\nelements in `keywords` for which applying `fun` returned a truthy value, and\na second keyword list with all the elements for which applying `fun` returned\na falsy value (`false` or `nil`).","ref":"Keyword.html#split_with/2"},{"type":"function","title":"Examples - Keyword.split_with/2","doc":"iex> Keyword.split_with([a: 1, b: 2, c: 3], fn {_k, v} -> rem(v, 2) == 0 end)\n    {[b: 2], [a: 1, c: 3]}\n\n    iex> Keyword.split_with([a: 1, b: 2, c: 3, b: 4], fn {_k, v} -> rem(v, 2) == 0 end)\n    {[b: 2, b: 4], [a: 1, c: 3]}\n\n    iex> Keyword.split_with([a: 1, b: 2, c: 3, b: 4], fn {k, v} -> k in [:a, :c] and rem(v, 2) == 0 end)\n    {[], [a: 1, b: 2, c: 3, b: 4]}\n\n    iex> Keyword.split_with([], fn {_k, v} -> rem(v, 2) == 0 end)\n    {[], []}","ref":"Keyword.html#split_with/2-examples"},{"type":"function","title":"Keyword.take/2","doc":"Takes all entries corresponding to the given `keys` and returns them as a new\nkeyword list.\n\nPreserves duplicate keys in the new keyword list.","ref":"Keyword.html#take/2"},{"type":"function","title":"Examples - Keyword.take/2","doc":"iex> Keyword.take([a: 1, b: 2, c: 3], [:a, :c, :e])\n    [a: 1, c: 3]\n    iex> Keyword.take([a: 1, b: 2, c: 3, a: 5], [:a, :c, :e])\n    [a: 1, c: 3, a: 5]","ref":"Keyword.html#take/2-examples"},{"type":"function","title":"Keyword.to_list/1","doc":"Returns the keyword list itself.","ref":"Keyword.html#to_list/1"},{"type":"function","title":"Examples - Keyword.to_list/1","doc":"iex> Keyword.to_list(a: 1)\n    [a: 1]","ref":"Keyword.html#to_list/1-examples"},{"type":"function","title":"Keyword.update/4","doc":"Updates the value under `key` in `keywords` using the given function.\n\nIf the `key` does not exist, it inserts the given `default` value.\nDoes not pass the `default` value through the update function.\n\nRemoves all duplicate keys and only updates the first one.","ref":"Keyword.html#update/4"},{"type":"function","title":"Examples - Keyword.update/4","doc":"iex> Keyword.update([a: 1], :a, 13, fn existing_value -> existing_value * 2 end)\n    [a: 2]\n\n    iex> Keyword.update([a: 1, a: 2], :a, 13, fn existing_value -> existing_value * 2 end)\n    [a: 2]\n\n    iex> Keyword.update([a: 1], :b, 11, fn existing_value -> existing_value * 2 end)\n    [a: 1, b: 11]","ref":"Keyword.html#update/4-examples"},{"type":"function","title":"Keyword.update!/3","doc":"Updates the value under `key` using the given function.\n\nRaises `KeyError` if the `key` does not exist.\n\nRemoves all duplicate keys and only updates the first one.","ref":"Keyword.html#update!/3"},{"type":"function","title":"Examples - Keyword.update!/3","doc":"iex> Keyword.update!([a: 1, b: 2, a: 3], :a, &(&1 * 2))\n    [a: 2, b: 2]\n    iex> Keyword.update!([a: 1, b: 2, c: 3], :b, &(&1 * 2))\n    [a: 1, b: 4, c: 3]\n\n    iex> Keyword.update!([a: 1], :b, &(&1 * 2))\n    ** (KeyError) key :b not found in: [a: 1]","ref":"Keyword.html#update!/3-examples"},{"type":"function","title":"Keyword.validate/2","doc":"Ensures the given `keyword` has only the keys given in `values`.\n\nThe second argument must be a list of atoms, specifying\na given key, or tuples specifying a key and a default value.\n\nIf the keyword list has only the given keys, it returns\n`{:ok, keyword}` with default values applied. Otherwise it\nreturns `{:error, invalid_keys}` with invalid keys.\n\nSee also: `validate!/2`.","ref":"Keyword.html#validate/2"},{"type":"function","title":"Examples - Keyword.validate/2","doc":"iex> {:ok, result} = Keyword.validate([], [one: 1, two: 2])\n    iex> Enum.sort(result)\n    [one: 1, two: 2]\n\n    iex> {:ok, result} = Keyword.validate([two: 3], [one: 1, two: 2])\n    iex> Enum.sort(result)\n    [one: 1, two: 3]\n\nIf atoms are given, they are supported as keys but do not\nprovide a default value:\n\n    iex> {:ok, result} = Keyword.validate([], [:one, two: 2])\n    iex> Enum.sort(result)\n    [two: 2]\n\n    iex> {:ok, result} = Keyword.validate([one: 1], [:one, two: 2])\n    iex> Enum.sort(result)\n    [one: 1, two: 2]\n\nPassing unknown keys returns an error:\n\n    iex> Keyword.validate([three: 3, four: 4], [one: 1, two: 2])\n    {:error, [:four, :three]}\n\nPassing the same key multiple times also errors:\n\n    iex> Keyword.validate([one: 1, two: 2, one: 1], [:one, :two])\n    {:error, [:one]}","ref":"Keyword.html#validate/2-examples"},{"type":"function","title":"Keyword.validate!/2","doc":"Similar to `validate/2` but returns the keyword or raises an error.","ref":"Keyword.html#validate!/2"},{"type":"function","title":"Examples - Keyword.validate!/2","doc":"iex> Keyword.validate!([], [one: 1, two: 2]) |> Enum.sort()\n    [one: 1, two: 2]\n    iex> Keyword.validate!([two: 3], [one: 1, two: 2]) |> Enum.sort()\n    [one: 1, two: 3]\n\nIf atoms are given, they are supported as keys but do not\nprovide a default value:\n\n    iex> Keyword.validate!([], [:one, two: 2]) |> Enum.sort()\n    [two: 2]\n    iex> Keyword.validate!([one: 1], [:one, two: 2]) |> Enum.sort()\n    [one: 1, two: 2]\n\nPassing unknown keys raises an error:\n\n    iex> Keyword.validate!([three: 3], [one: 1, two: 2])\n    ** (ArgumentError) unknown keys [:three] in [three: 3], the allowed keys are: [:one, :two]\n\nPassing the same key multiple times also errors:\n\n    iex> Keyword.validate!([one: 1, two: 2, one: 1], [:one, :two])\n    ** (ArgumentError) duplicate keys [:one] in [one: 1, two: 2, one: 1]","ref":"Keyword.html#validate!/2-examples"},{"type":"function","title":"Keyword.values/1","doc":"Returns all values from the keyword list.\n\nKeeps values from duplicate keys in the resulting list of values.","ref":"Keyword.html#values/1"},{"type":"function","title":"Examples - Keyword.values/1","doc":"iex> Keyword.values(a: 1, b: 2)\n    [1, 2]\n    iex> Keyword.values(a: 1, b: 2, a: 3)\n    [1, 2, 3]","ref":"Keyword.html#values/1-examples"},{"type":"type","title":"Keyword.key/0","doc":"","ref":"Keyword.html#t:key/0"},{"type":"type","title":"Keyword.t/0","doc":"","ref":"Keyword.html#t:t/0"},{"type":"type","title":"Keyword.t/1","doc":"","ref":"Keyword.html#t:t/1"},{"type":"type","title":"Keyword.value/0","doc":"","ref":"Keyword.html#t:value/0"},{"type":"module","title":"List","doc":"Linked lists hold zero, one, or more elements in the chosen order.\n\nLists in Elixir are specified between square brackets:\n\n    iex> [1, \"two\", 3, :four]\n    [1, \"two\", 3, :four]\n\nTwo lists can be concatenated and subtracted using the\n`++/2` and `--/2` operators:\n\n    iex> [1, 2, 3] ++ [4, 5, 6]\n    [1, 2, 3, 4, 5, 6]\n    iex> [1, true, 2, false, 3, true] -- [true, false]\n    [1, 2, 3, true]\n\nAn element can be prepended to a list using `|`:\n\n    iex> new = 0\n    iex> list = [1, 2, 3]\n    iex> [new | list]\n    [0, 1, 2, 3]\n\nLists in Elixir are effectively linked lists, which means\nthey are internally represented in pairs containing the\nhead and the tail of a list:\n\n    iex> [head | tail] = [1, 2, 3]\n    iex> head\n    1\n    iex> tail\n    [2, 3]\n\nSimilarly, we could write the list `[1, 2, 3]` using only\nsuch pairs (called cons cells):\n\n    iex> [1 | [2 | [3 | []]]]\n    [1, 2, 3]\n\nSome lists, called improper lists, do not have an empty list as\nthe second element in the last cons cell:\n\n    iex> [1 | [2 | [3 | 4]]]\n    [1, 2, 3 | 4]\n\nAlthough improper lists are generally avoided, they are used in some\nspecial circumstances like iodata and chardata entities (see the `IO` module).\n\nDue to their cons cell based representation, prepending an element\nto a list is always fast (constant time), while appending becomes\nslower as the list grows in size (linear time):\n\n    iex> list = [1, 2, 3]\n    iex> [0 | list] # fast\n    [0, 1, 2, 3]\n    iex> list ++ [4] # slow\n    [1, 2, 3, 4]\n\nMost of the functions in this module work in linear time. This means that,\nthat the time it takes to perform an operation grows at the same rate as the\nlength of the list. For example `length/1` and `last/1` will run in linear\ntime because they need to iterate through every element of the list, but\n`first/1` will run in constant time because it only needs the first element.\n\nLists also implement the `Enumerable` protocol, so many functions to work with\nlists are found in the `Enum` module. Additionally, the following functions and\noperators for lists are found in `Kernel`:\n\n  * `++/2`\n  * `--/2`\n  * `hd/1`\n  * `tl/1`\n  * `in/2`\n  * `length/1`","ref":"List.html"},{"type":"module","title":"Charlists - List","doc":"If a list is made of non-negative integers, where each integer represents a\nUnicode code point, the list can also be called a charlist. These integers\nmust:\n\n  * be within the range `0..0x10FFFF` (`0..1_114_111`);\n  * and be out of the range `0xD800..0xDFFF` (`55_296..57_343`), which is\n    reserved in Unicode for UTF-16 surrogate pairs.\n\nElixir uses the [`~c` sigil](`sigil_c/2`) to define charlists:\n\n    iex> ~c\"héllo\"\n    [104, 233, 108, 108, 111]\n\nIn particular, charlists will be printed back by default with the `~c`\nsigil if they contain only printable ASCII characters:\n\n    iex> ~c\"abc\"\n    ~c\"abc\"\n\nEven though the representation changed, the raw data does remain a list of\nintegers, which can be handled as such:\n\n    iex> inspect(~c\"abc\", charlists: :as_list)\n    \"[97, 98, 99]\"\n    iex> Enum.map(~c\"abc\", fn num -> 1000 + num end)\n    [1097, 1098, 1099]\n\nYou can use the `IEx.Helpers.i/1` helper to get a condensed rundown on\ncharlists in IEx when you encounter them, which shows you the type, description\nand also the raw representation in one single summary.\n\nThe rationale behind this behaviour is to better support\nErlang libraries which may return text as charlists\ninstead of Elixir strings. In Erlang, charlists are the default\nway of handling strings, while in Elixir it's binaries. One\nexample of such functions is `Application.loaded_applications/0`:\n\n    Application.loaded_applications()\n    #=>  [\n    #=>    {:stdlib, ~c\"ERTS  CXC 138 10\", ~c\"2.6\"},\n    #=>    {:compiler, ~c\"ERTS  CXC 138 10\", ~c\"6.0.1\"},\n    #=>    {:elixir, ~c\"elixir\", ~c\"1.0.0\"},\n    #=>    {:kernel, ~c\"ERTS  CXC 138 10\", ~c\"4.1\"},\n    #=>    {:logger, ~c\"logger\", ~c\"1.0.0\"}\n    #=>  ]\n\nA list can be checked if it is made of only printable ASCII\ncharacters with `ascii_printable?/2`.\n\nImproper lists are never deemed as charlists.","ref":"List.html#module-charlists"},{"type":"function","title":"List.ascii_printable?/2","doc":"Checks if `list` is a charlist made only of printable ASCII characters.\n\nTakes an optional `limit` as a second argument. `ascii_printable?/2` only\nchecks the printability of the list up to the `limit`.\n\nA printable charlist in Elixir contains only the printable characters in the\nstandard seven-bit ASCII character encoding, which are characters ranging from\n32 to 126 in decimal notation, plus the following control characters:\n\n  * `?\\a` - Bell\n  * `?\\b` - Backspace\n  * `?\\t` - Horizontal tab\n  * `?\\n` - Line feed\n  * `?\\v` - Vertical tab\n  * `?\\f` - Form feed\n  * `?\\r` - Carriage return\n  * `?\\e` - Escape\n\nFor more information read the [Character groups](https://en.wikipedia.org/wiki/ASCII#Character_groups)\nsection in the Wikipedia article of the [ASCII](https://en.wikipedia.org/wiki/ASCII) standard.","ref":"List.html#ascii_printable?/2"},{"type":"function","title":"Examples - List.ascii_printable?/2","doc":"iex> List.ascii_printable?(~c\"abc\")\n    true\n\n    iex> List.ascii_printable?(~c\"abc\" ++ [0])\n    false\n\n    iex> List.ascii_printable?(~c\"abc\" ++ [0], 2)\n    true\n\nImproper lists are not printable, even if made only of ASCII characters:\n\n    iex> List.ascii_printable?(~c\"abc\" ++ ?d)\n    false","ref":"List.html#ascii_printable?/2-examples"},{"type":"function","title":"List.delete/2","doc":"Deletes the given `element` from the `list`. Returns a new list without\nthe element.\n\nIf the `element` occurs more than once in the `list`, just\nthe first occurrence is removed.","ref":"List.html#delete/2"},{"type":"function","title":"Examples - List.delete/2","doc":"iex> List.delete([:a, :b, :c], :a)\n    [:b, :c]\n\n    iex> List.delete([:a, :b, :c], :d)\n    [:a, :b, :c]\n\n    iex> List.delete([:a, :b, :b, :c], :b)\n    [:a, :b, :c]\n\n    iex> List.delete([], :b)\n    []","ref":"List.html#delete/2-examples"},{"type":"function","title":"List.delete_at/2","doc":"Produces a new list by removing the value at the specified `index`.\n\nNegative indices indicate an offset from the end of the `list`.\nIf `index` is out of bounds, the original `list` is returned.","ref":"List.html#delete_at/2"},{"type":"function","title":"Examples - List.delete_at/2","doc":"iex> List.delete_at([1, 2, 3], 0)\n    [2, 3]\n\n    iex> List.delete_at([1, 2, 3], 10)\n    [1, 2, 3]\n\n    iex> List.delete_at([1, 2, 3], -1)\n    [1, 2]","ref":"List.html#delete_at/2-examples"},{"type":"function","title":"List.duplicate/2","doc":"Duplicates the given element `n` times in a list.\n\n`n` is an integer greater than or equal to `0`.\n\nIf `n` is `0`, an empty list is returned.","ref":"List.html#duplicate/2"},{"type":"function","title":"Examples - List.duplicate/2","doc":"iex> List.duplicate(\"hello\", 0)\n    []\n\n    iex> List.duplicate(\"hi\", 1)\n    [\"hi\"]\n\n    iex> List.duplicate(\"bye\", 2)\n    [\"bye\", \"bye\"]\n\n    iex> List.duplicate([1, 2], 3)\n    [[1, 2], [1, 2], [1, 2]]","ref":"List.html#duplicate/2-examples"},{"type":"function","title":"List.first/2","doc":"Returns the first element in `list` or `default` if `list` is empty.\n\n`first/2` has been introduced in Elixir v1.12.0, while `first/1` has been available since v1.0.0.","ref":"List.html#first/2"},{"type":"function","title":"Examples - List.first/2","doc":"iex> List.first([])\n    nil\n\n    iex> List.first([], 1)\n    1\n\n    iex> List.first([1])\n    1\n\n    iex> List.first([1, 2, 3])\n    1","ref":"List.html#first/2-examples"},{"type":"function","title":"List.flatten/1","doc":"Flattens the given `list` of nested lists.\n\nEmpty list elements are discarded.","ref":"List.html#flatten/1"},{"type":"function","title":"Examples - List.flatten/1","doc":"iex> List.flatten([1, [[2], 3]])\n    [1, 2, 3]\n\n    iex> List.flatten([[], [[], []]])\n    []","ref":"List.html#flatten/1-examples"},{"type":"function","title":"List.flatten/2","doc":"Flattens the given `list` of nested lists.\nThe list `tail` will be added at the end of\nthe flattened list.\n\nEmpty list elements from `list` are discarded,\nbut not the ones from `tail`.","ref":"List.html#flatten/2"},{"type":"function","title":"Examples - List.flatten/2","doc":"iex> List.flatten([1, [[2], 3]], [4, 5])\n    [1, 2, 3, 4, 5]\n\n    iex> List.flatten([1, [], 2], [3, [], 4])\n    [1, 2, 3, [], 4]","ref":"List.html#flatten/2-examples"},{"type":"function","title":"List.foldl/3","doc":"Folds (reduces) the given list from the left with\na function. Requires an accumulator, which can be any value.","ref":"List.html#foldl/3"},{"type":"function","title":"Examples - List.foldl/3","doc":"iex> List.foldl([5, 5], 10, fn x, acc -> x + acc end)\n    20\n\n    iex> List.foldl([1, 2, 3, 4], 0, fn x, acc -> x - acc end)\n    2\n\n    iex> List.foldl([1, 2, 3], {0, 0}, fn x, {a1, a2} -> {a1 + x, a2 - x} end)\n    {6, -6}","ref":"List.html#foldl/3-examples"},{"type":"function","title":"List.foldr/3","doc":"Folds (reduces) the given list from the right with\na function. Requires an accumulator, which can be any value.","ref":"List.html#foldr/3"},{"type":"function","title":"Examples - List.foldr/3","doc":"iex> List.foldr([1, 2, 3, 4], 0, fn x, acc -> x - acc end)\n    -2\n\n    iex> List.foldr([1, 2, 3, 4], %{sum: 0, product: 1}, fn x, %{sum: a1, product: a2} -> %{sum: a1 + x, product: a2 * x} end)\n    %{product: 24, sum: 10}","ref":"List.html#foldr/3-examples"},{"type":"function","title":"List.improper?/1","doc":"Returns `true` if `list` is an improper list. Otherwise returns `false`.","ref":"List.html#improper?/1"},{"type":"function","title":"Examples - List.improper?/1","doc":"iex> List.improper?([1, 2 | 3])\n    true\n\n    iex> List.improper?([1, 2, 3])\n    false","ref":"List.html#improper?/1-examples"},{"type":"function","title":"List.insert_at/3","doc":"Returns a list with `value` inserted at the specified `index`.\n\nNote that `index` is capped at the list length. Negative indices\nindicate an offset from the end of the `list`.","ref":"List.html#insert_at/3"},{"type":"function","title":"Examples - List.insert_at/3","doc":"iex> List.insert_at([1, 2, 3, 4], 2, 0)\n    [1, 2, 0, 3, 4]\n\n    iex> List.insert_at([1, 2, 3], 10, 0)\n    [1, 2, 3, 0]\n\n    iex> List.insert_at([1, 2, 3], -1, 0)\n    [1, 2, 3, 0]\n\n    iex> List.insert_at([1, 2, 3], -10, 0)\n    [0, 1, 2, 3]","ref":"List.html#insert_at/3-examples"},{"type":"function","title":"List.keydelete/3","doc":"Receives a `list` of tuples and deletes the first tuple\nwhere the element at `position` matches the\ngiven `key`. Returns the new list.","ref":"List.html#keydelete/3"},{"type":"function","title":"Examples - List.keydelete/3","doc":"iex> List.keydelete([a: 1, b: 2], :a, 0)\n    [b: 2]\n\n    iex> List.keydelete([a: 1, b: 2], 2, 1)\n    [a: 1]\n\n    iex> List.keydelete([a: 1, b: 2], :c, 0)\n    [a: 1, b: 2]\n\nThis function works for any list of tuples:\n\n    iex> List.keydelete([{22, \"SSH\"}, {80, \"HTTP\"}], 80, 0)\n    [{22, \"SSH\"}]","ref":"List.html#keydelete/3-examples"},{"type":"function","title":"List.keyfind/4","doc":"Receives a list of tuples and returns the first tuple\nwhere the element at `position` in the tuple matches the\ngiven `key`.\n\nIf no matching tuple is found, `default` is returned.","ref":"List.html#keyfind/4"},{"type":"function","title":"Examples - List.keyfind/4","doc":"iex> List.keyfind([a: 1, b: 2], :a, 0)\n    {:a, 1}\n\n    iex> List.keyfind([a: 1, b: 2], 2, 1)\n    {:b, 2}\n\n    iex> List.keyfind([a: 1, b: 2], :c, 0)\n    nil\n\nThis function works for any list of tuples:\n\n    iex> List.keyfind([{22, \"SSH\"}, {80, \"HTTP\"}], 22, 0)\n    {22, \"SSH\"}","ref":"List.html#keyfind/4-examples"},{"type":"function","title":"List.keyfind!/3","doc":"Receives a list of tuples and returns the first tuple\nwhere the element at `position` in the tuple matches the\ngiven `key`.\n\nIf no matching tuple is found, an error is raised.","ref":"List.html#keyfind!/3"},{"type":"function","title":"Examples - List.keyfind!/3","doc":"iex> List.keyfind!([a: 1, b: 2], :a, 0)\n    {:a, 1}\n\n    iex> List.keyfind!([a: 1, b: 2], 2, 1)\n    {:b, 2}\n\n    iex> List.keyfind!([a: 1, b: 2], :c, 0)\n    ** (KeyError) key :c at position 0 not found in: [a: 1, b: 2]\n\nThis function works for any list of tuples:\n\n    iex> List.keyfind!([{22, \"SSH\"}, {80, \"HTTP\"}], 22, 0)\n    {22, \"SSH\"}","ref":"List.html#keyfind!/3-examples"},{"type":"function","title":"List.keymember?/3","doc":"Receives a list of tuples and returns `true` if there is\na tuple where the element at `position` in the tuple matches\nthe given `key`.","ref":"List.html#keymember?/3"},{"type":"function","title":"Examples - List.keymember?/3","doc":"iex> List.keymember?([a: 1, b: 2], :a, 0)\n    true\n\n    iex> List.keymember?([a: 1, b: 2], 2, 1)\n    true\n\n    iex> List.keymember?([a: 1, b: 2], :c, 0)\n    false\n\nThis function works for any list of tuples:\n\n    iex> List.keymember?([{22, \"SSH\"}, {80, \"HTTP\"}], 22, 0)\n    true","ref":"List.html#keymember?/3-examples"},{"type":"function","title":"List.keyreplace/4","doc":"Receives a list of tuples and if the identified element by `key` at `position`\nexists, it is replaced with `new_tuple`.","ref":"List.html#keyreplace/4"},{"type":"function","title":"Examples - List.keyreplace/4","doc":"iex> List.keyreplace([a: 1, b: 2], :a, 0, {:a, 3})\n    [a: 3, b: 2]\n\n    iex> List.keyreplace([a: 1, b: 2], :a, 1, {:a, 3})\n    [a: 1, b: 2]\n\nThis function works for any list of tuples:\n\n    iex> List.keyreplace([{22, \"SSH\"}, {80, \"HTTP\"}], 22, 0, {22, \"Secure Shell\"})\n    [{22, \"Secure Shell\"}, {80, \"HTTP\"}]","ref":"List.html#keyreplace/4-examples"},{"type":"function","title":"List.keysort/3","doc":"Receives a list of tuples and sorts the elements\nat `position` of the tuples.\n\nThe sort is stable.\n\nA `sorter` argument is available since Elixir v1.14.0. Similar to\n`Enum.sort/2`, the sorter can be an anonymous function, the atoms\n`:asc` or `:desc`, or module that implements a compare function.","ref":"List.html#keysort/3"},{"type":"function","title":"Examples - List.keysort/3","doc":"iex> List.keysort([a: 5, b: 1, c: 3], 1)\n    [b: 1, c: 3, a: 5]\n\n    iex> List.keysort([a: 5, c: 1, b: 3], 0)\n    [a: 5, b: 3, c: 1]\n\nTo sort in descending order:\n\n    iex> List.keysort([a: 5, c: 1, b: 3], 0, :desc)\n    [c: 1, b: 3, a: 5]\n\nAs in `Enum.sort/2`, avoid using the default sorting function to sort\nstructs, as by default it performs structural comparison instead of a\nsemantic one. In such cases, you shall pass a sorting function as third\nelement or any module that implements a `compare/2` function. For example,\nif you have tuples with user names and their birthday, and you want to\nsort on their birthday, in both ascending and descending order, you should\ndo:\n\n    iex> users = [\n    ...>   {\"Ellis\", ~D[1943-05-11]},\n    ...>   {\"Lovelace\", ~D[1815-12-10]},\n    ...>   {\"Turing\", ~D[1912-06-23]}\n    ...> ]\n    iex> List.keysort(users, 1, Date)\n    [\n      {\"Lovelace\", ~D[1815-12-10]},\n      {\"Turing\", ~D[1912-06-23]},\n      {\"Ellis\", ~D[1943-05-11]}\n    ]\n    iex> List.keysort(users, 1, {:desc, Date})\n    [\n      {\"Ellis\", ~D[1943-05-11]},\n      {\"Turing\", ~D[1912-06-23]},\n      {\"Lovelace\", ~D[1815-12-10]}\n    ]","ref":"List.html#keysort/3-examples"},{"type":"function","title":"List.keystore/4","doc":"Receives a `list` of tuples and replaces the element\nidentified by `key` at `position` with `new_tuple`.\n\nIf the element does not exist, it is added to the end of the `list`.","ref":"List.html#keystore/4"},{"type":"function","title":"Examples - List.keystore/4","doc":"iex> List.keystore([a: 1, b: 2], :a, 0, {:a, 3})\n    [a: 3, b: 2]\n\n    iex> List.keystore([a: 1, b: 2], :c, 0, {:c, 3})\n    [a: 1, b: 2, c: 3]\n\nThis function works for any list of tuples:\n\n    iex> List.keystore([{22, \"SSH\"}], 80, 0, {80, \"HTTP\"})\n    [{22, \"SSH\"}, {80, \"HTTP\"}]","ref":"List.html#keystore/4-examples"},{"type":"function","title":"List.keytake/3","doc":"Receives a `list` of tuples and returns the first tuple\nwhere the element at `position` in the tuple matches the\ngiven `key`, as well as the `list` without found tuple.\n\nIf such a tuple is not found, `nil` will be returned.","ref":"List.html#keytake/3"},{"type":"function","title":"Examples - List.keytake/3","doc":"iex> List.keytake([a: 1, b: 2], :a, 0)\n    {{:a, 1}, [b: 2]}\n\n    iex> List.keytake([a: 1, b: 2], 2, 1)\n    {{:b, 2}, [a: 1]}\n\n    iex> List.keytake([a: 1, b: 2], :c, 0)\n    nil\n\nThis function works for any list of tuples:\n\n    iex> List.keytake([{22, \"SSH\"}, {80, \"HTTP\"}], 80, 0)\n    {{80, \"HTTP\"}, [{22, \"SSH\"}]}","ref":"List.html#keytake/3-examples"},{"type":"function","title":"List.last/2","doc":"Returns the last element in `list` or `default` if `list` is empty.\n\n`last/2` has been introduced in Elixir v1.12.0, while `last/1` has been available since v1.0.0.","ref":"List.html#last/2"},{"type":"function","title":"Examples - List.last/2","doc":"iex> List.last([])\n    nil\n\n    iex> List.last([], 1)\n    1\n\n    iex> List.last([1])\n    1\n\n    iex> List.last([1, 2, 3])\n    3","ref":"List.html#last/2-examples"},{"type":"function","title":"List.myers_difference/2","doc":"Returns a keyword list that represents an *edit script*.\n\nThe algorithm is outlined in the\n\"An O(ND) Difference Algorithm and Its Variations\" paper by E. Myers.\n\nAn *edit script* is a keyword list. Each key describes the \"editing action\" to\ntake in order to bring `list1` closer to being equal to `list2`; a key can be\n`:eq`, `:ins`, or `:del`. Each value is a sublist of either `list1` or `list2`\nthat should be inserted (if the corresponding key `:ins`), deleted (if the\ncorresponding key is `:del`), or left alone (if the corresponding key is\n`:eq`) in `list1` in order to be closer to `list2`.\n\nSee `myers_difference/3` if you want to handle nesting in the diff scripts.","ref":"List.html#myers_difference/2"},{"type":"function","title":"Examples - List.myers_difference/2","doc":"iex> List.myers_difference([1, 4, 2, 3], [1, 2, 3, 4])\n    [eq: [1], del: [4], eq: [2, 3], ins: [4]]","ref":"List.html#myers_difference/2-examples"},{"type":"function","title":"List.myers_difference/3","doc":"Returns a keyword list that represents an *edit script* with nested diffs.\n\nThis is an extension of `myers_difference/2` where a `diff_script` function\ncan be given in case it is desired to compute nested differences. The function\nmay return a list with the inner edit script or `nil` in case there is no\nsuch script. The returned inner edit script will be under the `:diff` key.","ref":"List.html#myers_difference/3"},{"type":"function","title":"Examples - List.myers_difference/3","doc":"iex> List.myers_difference([\"a\", \"db\", \"c\"], [\"a\", \"bc\"], &String.myers_difference/2)\n    [eq: [\"a\"], diff: [del: \"d\", eq: \"b\", ins: \"c\"], del: [\"c\"]]","ref":"List.html#myers_difference/3-examples"},{"type":"function","title":"List.pop_at/3","doc":"Returns and removes the value at the specified `index` in the `list`.\n\nNegative indices indicate an offset from the end of the `list`.\nIf `index` is out of bounds, the original `list` is returned.","ref":"List.html#pop_at/3"},{"type":"function","title":"Examples - List.pop_at/3","doc":"iex> List.pop_at([1, 2, 3], 0)\n    {1, [2, 3]}\n    iex> List.pop_at([1, 2, 3], 5)\n    {nil, [1, 2, 3]}\n    iex> List.pop_at([1, 2, 3], 5, 10)\n    {10, [1, 2, 3]}\n    iex> List.pop_at([1, 2, 3], -1)\n    {3, [1, 2]}","ref":"List.html#pop_at/3-examples"},{"type":"function","title":"List.replace_at/3","doc":"Returns a list with a replaced value at the specified `index`.\n\nNegative indices indicate an offset from the end of the `list`.\nIf `index` is out of bounds, the original `list` is returned.","ref":"List.html#replace_at/3"},{"type":"function","title":"Examples - List.replace_at/3","doc":"iex> List.replace_at([1, 2, 3], 0, 0)\n    [0, 2, 3]\n\n    iex> List.replace_at([1, 2, 3], 10, 0)\n    [1, 2, 3]\n\n    iex> List.replace_at([1, 2, 3], -1, 0)\n    [1, 2, 0]\n\n    iex> List.replace_at([1, 2, 3], -10, 0)\n    [1, 2, 3]","ref":"List.html#replace_at/3-examples"},{"type":"function","title":"List.starts_with?/2","doc":"Returns `true` if `list` starts with the given `prefix` list; otherwise returns `false`.\n\nIf `prefix` is an empty list, it returns `true`.\n\n#","ref":"List.html#starts_with?/2"},{"type":"function","title":"Examples - List.starts_with?/2","doc":"iex> List.starts_with?([1, 2, 3], [1, 2])\n    true\n\n    iex> List.starts_with?([1, 2], [1, 2, 3])\n    false\n\n    iex> List.starts_with?([:alpha], [])\n    true\n\n    iex> List.starts_with?([], [:alpha])\n    false","ref":"List.html#starts_with?/2-examples"},{"type":"function","title":"List.to_atom/1","doc":"Converts a charlist to an atom.\n\nElixir supports conversions from charlists which contains any Unicode\ncode point.\n\nInlined by the compiler.","ref":"List.html#to_atom/1"},{"type":"function","title":"Examples - List.to_atom/1","doc":"iex> List.to_atom(~c\"Elixir\")\n    :Elixir\n\n    iex> List.to_atom(~c\"🌢 Elixir\")\n    :\"🌢 Elixir\"","ref":"List.html#to_atom/1-examples"},{"type":"function","title":"List.to_charlist/1","doc":"Converts a list of integers representing Unicode code points, lists or\nstrings into a charlist.\n\nNote that this function expects a list of integers representing\nUnicode code points. If you have a list of bytes, you must instead use\nthe [`:binary` module](`:binary`).","ref":"List.html#to_charlist/1"},{"type":"function","title":"Examples - List.to_charlist/1","doc":"iex> ~c\"æß\" = List.to_charlist([0x00E6, 0x00DF])\n    [230, 223]\n\n    iex> List.to_charlist([0x0061, \"bc\"])\n    ~c\"abc\"\n\n    iex> List.to_charlist([0x0064, \"ee\", [~c\"p\"]])\n    ~c\"deep\"","ref":"List.html#to_charlist/1-examples"},{"type":"function","title":"List.to_existing_atom/1","doc":"Converts a charlist to an existing atom.\n\nElixir supports conversions from charlists which contains any Unicode\ncode point. Raises an `ArgumentError` if the atom does not exist.\n\nInlined by the compiler.\n\n> ##","ref":"List.html#to_existing_atom/1"},{"type":"function","title":"Atoms and modules {: .info} - List.to_existing_atom/1","doc":">\n> Since Elixir is a compiled language, the atoms defined in a module\n> will only exist after said module is loaded, which typically happens\n> whenever a function in the module is executed. Therefore, it is\n> generally recommended to call `List.to_existing_atom/1` only to\n> convert atoms defined within the module making the function call\n> to `to_existing_atom/1`.","ref":"List.html#to_existing_atom/1-atoms-and-modules-info"},{"type":"function","title":"Examples - List.to_existing_atom/1","doc":"iex> _ = :my_atom\n    iex> List.to_existing_atom(~c\"my_atom\")\n    :my_atom\n\n    iex> _ = :\"🌢 Elixir\"\n    iex> List.to_existing_atom(~c\"🌢 Elixir\")\n    :\"🌢 Elixir\"","ref":"List.html#to_existing_atom/1-examples"},{"type":"function","title":"List.to_float/1","doc":"Returns the float whose text representation is `charlist`.\n\nInlined by the compiler.","ref":"List.html#to_float/1"},{"type":"function","title":"Examples - List.to_float/1","doc":"iex> List.to_float(~c\"2.2017764e+0\")\n    2.2017764","ref":"List.html#to_float/1-examples"},{"type":"function","title":"List.to_integer/1","doc":"Returns an integer whose text representation is `charlist`.\n\nInlined by the compiler.","ref":"List.html#to_integer/1"},{"type":"function","title":"Examples - List.to_integer/1","doc":"iex> List.to_integer(~c\"123\")\n    123","ref":"List.html#to_integer/1-examples"},{"type":"function","title":"List.to_integer/2","doc":"Returns an integer whose text representation is `charlist` in base `base`.\n\nInlined by the compiler.\n\nThe base needs to be between `2` and `36`.","ref":"List.html#to_integer/2"},{"type":"function","title":"Examples - List.to_integer/2","doc":"iex> List.to_integer(~c\"3FF\", 16)\n    1023","ref":"List.html#to_integer/2-examples"},{"type":"function","title":"List.to_string/1","doc":"Converts a list of integers representing code points, lists or\nstrings into a string.\n\nTo be converted to a string, a list must either be empty or only\ncontain the following elements:\n\n  * strings\n  * integers representing Unicode code points\n  * a list containing one of these three elements\n\nNote that this function expects a list of integers representing\nUnicode code points. If you have a list of bytes, you must instead use\nthe [`:binary` module](`:binary`).","ref":"List.html#to_string/1"},{"type":"function","title":"Examples - List.to_string/1","doc":"iex> List.to_string([0x00E6, 0x00DF])\n    \"æß\"\n\n    iex> List.to_string([0x0061, \"bc\"])\n    \"abc\"\n\n    iex> List.to_string([0x0064, \"ee\", [~c\"p\"]])\n    \"deep\"\n\n    iex> List.to_string([])\n    \"\"","ref":"List.html#to_string/1-examples"},{"type":"function","title":"List.to_tuple/1","doc":"Converts a list to a tuple.\n\nInlined by the compiler.","ref":"List.html#to_tuple/1"},{"type":"function","title":"Examples - List.to_tuple/1","doc":"iex> List.to_tuple([:share, [:elixir, 163]])\n    {:share, [:elixir, 163]}","ref":"List.html#to_tuple/1-examples"},{"type":"function","title":"List.update_at/3","doc":"Returns a list with an updated value at the specified `index`.\n\nNegative indices indicate an offset from the end of the `list`.\nIf `index` is out of bounds, the original `list` is returned.","ref":"List.html#update_at/3"},{"type":"function","title":"Examples - List.update_at/3","doc":"iex> List.update_at([1, 2, 3], 0, &(&1 + 10))\n    [11, 2, 3]\n\n    iex> List.update_at([1, 2, 3], 10, &(&1 + 10))\n    [1, 2, 3]\n\n    iex> List.update_at([1, 2, 3], -1, &(&1 + 10))\n    [1, 2, 13]\n\n    iex> List.update_at([1, 2, 3], -10, &(&1 + 10))\n    [1, 2, 3]","ref":"List.html#update_at/3-examples"},{"type":"function","title":"List.wrap/1","doc":"Wraps `term` in a list if this is not list.\n\nIf `term` is already a list, it returns the list.\nIf `term` is `nil`, it returns an empty list.","ref":"List.html#wrap/1"},{"type":"function","title":"Examples - List.wrap/1","doc":"iex> List.wrap(\"hello\")\n    [\"hello\"]\n\n    iex> List.wrap([1, 2, 3])\n    [1, 2, 3]\n\n    iex> List.wrap(nil)\n    []","ref":"List.html#wrap/1-examples"},{"type":"function","title":"List.zip/1","doc":"Zips corresponding elements from each list in `list_of_lists`.\n\nThe zipping finishes as soon as any list terminates.","ref":"List.html#zip/1"},{"type":"function","title":"Examples - List.zip/1","doc":"iex> List.zip([[1, 2], [3, 4], [5, 6]])\n    [{1, 3, 5}, {2, 4, 6}]\n\n    iex> List.zip([[1, 2], [3], [5, 6]])\n    [{1, 3, 5}]","ref":"List.html#zip/1-examples"},{"type":"module","title":"Map","doc":"Maps are the \"go to\" key-value data structure in Elixir.\n\nMaps can be created with the `%{}` syntax, and key-value pairs can be\nexpressed as `key => value`:\n\n    iex> %{}\n    %{}\n    iex> %{\"one\" => :two, 3 => \"four\"}\n    %{3 => \"four\", \"one\" => :two}\n\nKey-value pairs in a map do not follow any order (that's why the printed map\nin the example above has a different order than the map that was created).\n\nMaps do not impose any restriction on the key type: anything can be a key in a\nmap. As a key-value structure, maps do not allow duplicated keys. Keys are\ncompared using the exact-equality operator (`===/2`). If colliding keys are defined\nin a map literal, the last one prevails.\n\nWhen the key in a key-value pair is an atom, the `key: value` shorthand syntax\ncan be used (as in many other special forms):\n\n    iex> %{a: 1, b: 2}\n    %{a: 1, b: 2}\n\nIf you want to mix the shorthand syntax with `=>`, the shorthand syntax must come\nat the end:\n\n    iex> %{\"hello\" => \"world\", a: 1, b: 2}\n    %{:a => 1, :b => 2, \"hello\" => \"world\"}\n\nKeys in maps can be accessed through some of the functions in this module\n(such as `Map.get/3` or `Map.fetch/2`) or through the `map[]` syntax provided\nby the `Access` module:\n\n    iex> map = %{a: 1, b: 2}\n    iex> Map.fetch(map, :a)\n    {:ok, 1}\n    iex> map[:b]\n    2\n    iex> map[\"non_existing_key\"]\n    nil\n\nTo access atom keys, one may also use the `map.key` notation. Note that `map.key`\nwill raise a `KeyError` if the `map` doesn't contain the key `:key`, compared to\n`map[:key]`, that would return `nil`.\n\n    map = %{foo: \"bar\", baz: \"bong\"}\n    map.foo\n    #=> \"bar\"\n    map.non_existing_key\n    ** (KeyError) key :non_existing_key not found in: %{baz: \"bong\", foo: \"bar\"}\n\n> ##","ref":"Map.html"},{"type":"module","title":"Avoid parentheses {: .warning} - Map","doc":">\n> Do not add parentheses when accessing fields, such as in `data.key()`.\n> If parentheses are used, Elixir will expect `data` to be an atom representing\n> a module and attempt to call the *function* `key/0` in it.\n\nThe two syntaxes for accessing keys reveal the dual nature of maps. The `map[key]`\nsyntax is used for dynamically created maps that may have any key, of any type.\n`map.key` is used with maps that hold a predetermined set of atoms keys, which are\nexpected to always be present. Structs, defined via `defstruct/1`, are one example\nof such \"static maps\", where the keys can also be checked during compile time.\n\nMaps can be pattern matched on. When a map is on the left-hand side of a\npattern match, it will match if the map on the right-hand side contains the\nkeys on the left-hand side and their values match the ones on the left-hand\nside. This means that an empty map matches every map.\n\n    iex> %{} = %{foo: \"bar\"}\n    %{foo: \"bar\"}\n    iex> %{a: a} = %{:a => 1, \"b\" => 2, [:c, :e, :e] => 3}\n    iex> a\n    1\n\nBut this will raise a `MatchError` exception:\n\n    %{:c => 3} = %{:a => 1, 2 => :b}\n\nVariables can be used as map keys both when writing map literals as well as\nwhen matching:\n\n    iex> n = 1\n    1\n    iex> %{n => :one}\n    %{1 => :one}\n    iex> %{^n => :one} = %{1 => :one, 2 => :two, 3 => :three}\n    %{1 => :one, 2 => :two, 3 => :three}\n\nMaps also support a specific update syntax to update the value stored under\n*existing* keys. You can update using the atom keys syntax:\n\n    iex> map = %{one: 1, two: 2}\n    iex> %{map | one: \"one\"}\n    %{one: \"one\", two: 2}\n\nOr any other key:\n\n    iex> other_map = %{\"three\" => 3, \"four\" => 4}\n    iex> %{other_map | \"three\" => \"three\"}\n    %{\"four\" => 4, \"three\" => \"three\"}\n\nWhen a key that does not exist in the map is updated a `KeyError` exception will be raised:\n\n    %{map | three: 3}\n\nThe functions in this module that need to find a specific key work in logarithmic time.\nThis means that the time it takes to find keys grows as the map grows, but it's not\ndirectly proportional to the map size. In comparison to finding an element in a list,\nit performs better because lists have a linear time complexity. Some functions,\nsuch as `keys/1` and `values/1`, run in linear time because they need to get to every\nelement in the map.\n\nMaps also implement the `Enumerable` protocol, so many functions to work with maps\nare found in the `Enum` module. Additionally, the following functions for maps are\nfound in `Kernel`:\n\n  * `map_size/1`","ref":"Map.html#module-avoid-parentheses-warning"},{"type":"function","title":"Map.delete/2","doc":"Deletes the entry in `map` for a specific `key`.\n\nIf the `key` does not exist, returns `map` unchanged.\n\nInlined by the compiler.","ref":"Map.html#delete/2"},{"type":"function","title":"Examples - Map.delete/2","doc":"iex> Map.delete(%{a: 1, b: 2}, :a)\n    %{b: 2}\n    iex> Map.delete(%{b: 2}, :a)\n    %{b: 2}","ref":"Map.html#delete/2-examples"},{"type":"function","title":"Map.drop/2","doc":"Drops the given `keys` from `map`.\n\nIf `keys` contains keys that are not in `map`, they're simply ignored.","ref":"Map.html#drop/2"},{"type":"function","title":"Examples - Map.drop/2","doc":"iex> Map.drop(%{a: 1, b: 2, c: 3}, [:b, :d])\n    %{a: 1, c: 3}","ref":"Map.html#drop/2-examples"},{"type":"function","title":"Map.equal?/2","doc":"Checks if two maps are equal.\n\nTwo maps are considered to be equal if they contain\nthe same keys and those keys contain the same values.\n\nNote this function exists for completeness so the `Map`\nand `Keyword` modules provide similar APIs. In practice,\ndevelopers often compare maps using `==/2` or `===/2`\ndirectly.","ref":"Map.html#equal?/2"},{"type":"function","title":"Examples - Map.equal?/2","doc":"iex> Map.equal?(%{a: 1, b: 2}, %{b: 2, a: 1})\n    true\n    iex> Map.equal?(%{a: 1, b: 2}, %{b: 1, a: 2})\n    false\n\nComparison between keys and values is done with `===/3`,\nwhich means integers are not equivalent to floats:\n\n    iex> Map.equal?(%{a: 1.0}, %{a: 1})\n    false","ref":"Map.html#equal?/2-examples"},{"type":"function","title":"Map.fetch/2","doc":"Fetches the value for a specific `key` in the given `map`.\n\nIf `map` contains the given `key` then its value is returned in the shape of `{:ok, value}`.\nIf `map` doesn't contain `key`, `:error` is returned.\n\nInlined by the compiler.","ref":"Map.html#fetch/2"},{"type":"function","title":"Examples - Map.fetch/2","doc":"iex> Map.fetch(%{a: 1}, :a)\n    {:ok, 1}\n    iex> Map.fetch(%{a: 1}, :b)\n    :error","ref":"Map.html#fetch/2-examples"},{"type":"function","title":"Map.fetch!/2","doc":"Fetches the value for a specific `key` in the given `map`, erroring out if\n`map` doesn't contain `key`.\n\nIf `map` contains `key`, the corresponding value is returned. If\n`map` doesn't contain `key`, a `KeyError` exception is raised.\n\nInlined by the compiler.","ref":"Map.html#fetch!/2"},{"type":"function","title":"Examples - Map.fetch!/2","doc":"iex> Map.fetch!(%{a: 1}, :a)\n    1","ref":"Map.html#fetch!/2-examples"},{"type":"function","title":"Map.filter/2","doc":"Returns a map containing only those pairs from `map`\nfor which `fun` returns a truthy value.\n\n`fun` receives the key and value of each of the\nelements in the map as a key-value pair.\n\nSee also `reject/2` which discards all elements where the\nfunction returns a truthy value.\n\n> ##","ref":"Map.html#filter/2"},{"type":"function","title":"Performance considerations {: .tip} - Map.filter/2","doc":">\n> If you find yourself doing multiple calls to `Map.filter/2`\n> and `Map.reject/2` in a pipeline, it is likely more efficient\n> to use `Enum.map/2` and `Enum.filter/2` instead and convert to\n> a map at the end using `Map.new/1`.","ref":"Map.html#filter/2-performance-considerations-tip"},{"type":"function","title":"Examples - Map.filter/2","doc":"iex> Map.filter(%{one: 1, two: 2, three: 3}, fn {_key, val} -> rem(val, 2) == 1 end)\n    %{one: 1, three: 3}","ref":"Map.html#filter/2-examples"},{"type":"function","title":"Map.from_keys/2","doc":"Builds a map from the given `keys` and the fixed `value`.","ref":"Map.html#from_keys/2"},{"type":"function","title":"Examples - Map.from_keys/2","doc":"iex> Map.from_keys([1, 2, 3], :number)\n    %{1 => :number, 2 => :number, 3 => :number}","ref":"Map.html#from_keys/2-examples"},{"type":"function","title":"Map.from_struct/1","doc":"Converts a `struct` to map.\n\nIt accepts the struct module or a struct itself and\nsimply removes the `__struct__` field from the given struct\nor from a new struct generated from the given module.","ref":"Map.html#from_struct/1"},{"type":"function","title":"Example - Map.from_struct/1","doc":"defmodule User do\n      defstruct [:name]\n    end\n\n    Map.from_struct(User)\n    #=> %{name: nil}\n\n    Map.from_struct(%User{name: \"john\"})\n    #=> %{name: \"john\"}","ref":"Map.html#from_struct/1-example"},{"type":"function","title":"Map.get/3","doc":"Gets the value for a specific `key` in `map`.\n\nIf `key` is present in `map` then its value `value` is\nreturned. Otherwise, `default` is returned.\n\nIf `default` is not provided, `nil` is used.","ref":"Map.html#get/3"},{"type":"function","title":"Examples - Map.get/3","doc":"iex> Map.get(%{}, :a)\n    nil\n    iex> Map.get(%{a: 1}, :a)\n    1\n    iex> Map.get(%{a: 1}, :b)\n    nil\n    iex> Map.get(%{a: 1}, :b, 3)\n    3\n    iex> Map.get(%{a: nil}, :a, 1)\n    nil","ref":"Map.html#get/3-examples"},{"type":"function","title":"Map.get_and_update/3","doc":"Gets the value from `key` and updates it, all in one pass.\n\n`fun` is called with the current value under `key` in `map` (or `nil` if `key`\nis not present in `map`) and must return a two-element tuple: the current value\n(the retrieved value, which can be operated on before being returned) and the\nnew value to be stored under `key` in the resulting new map. `fun` may also\nreturn `:pop`, which means the current value shall be removed from `map` and\nreturned (making this function behave like `Map.pop(map, key)`).\n\nThe returned value is a two-element tuple with the current value returned by\n`fun` and a new map with the updated value under `key`.","ref":"Map.html#get_and_update/3"},{"type":"function","title":"Examples - Map.get_and_update/3","doc":"iex> Map.get_and_update(%{a: 1}, :a, fn current_value ->\n    ...>   {current_value, \"new value!\"}\n    ...> end)\n    {1, %{a: \"new value!\"}}\n\n    iex> Map.get_and_update(%{a: 1}, :b, fn current_value ->\n    ...>   {current_value, \"new value!\"}\n    ...> end)\n    {nil, %{a: 1, b: \"new value!\"}}\n\n    iex> Map.get_and_update(%{a: 1}, :a, fn _ -> :pop end)\n    {1, %{}}\n\n    iex> Map.get_and_update(%{a: 1}, :b, fn _ -> :pop end)\n    {nil, %{a: 1}}","ref":"Map.html#get_and_update/3-examples"},{"type":"function","title":"Map.get_and_update!/3","doc":"Gets the value from `key` and updates it, all in one pass. Raises if there is no `key`.\n\nBehaves exactly like `get_and_update/3`, but raises a `KeyError` exception if\n`key` is not present in `map`.","ref":"Map.html#get_and_update!/3"},{"type":"function","title":"Examples - Map.get_and_update!/3","doc":"iex> Map.get_and_update!(%{a: 1}, :a, fn current_value ->\n    ...>   {current_value, \"new value!\"}\n    ...> end)\n    {1, %{a: \"new value!\"}}\n\n    iex> Map.get_and_update!(%{a: 1}, :b, fn current_value ->\n    ...>   {current_value, \"new value!\"}\n    ...> end)\n    ** (KeyError) key :b not found in: %{a: 1}\n\n    iex> Map.get_and_update!(%{a: 1}, :a, fn _ ->\n    ...>   :pop\n    ...> end)\n    {1, %{}}","ref":"Map.html#get_and_update!/3-examples"},{"type":"function","title":"Map.get_lazy/3","doc":"Gets the value for a specific `key` in `map`.\n\nIf `key` is present in `map` then its value `value` is\nreturned. Otherwise, `fun` is evaluated and its result is returned.\n\nThis is useful if the default value is very expensive to calculate or\ngenerally difficult to setup and teardown again.","ref":"Map.html#get_lazy/3"},{"type":"function","title":"Examples - Map.get_lazy/3","doc":"iex> map = %{a: 1}\n    iex> fun = fn ->\n    ...>   # some expensive operation here\n    ...>   13\n    ...> end\n    iex> Map.get_lazy(map, :a, fun)\n    1\n    iex> Map.get_lazy(map, :b, fun)\n    13","ref":"Map.html#get_lazy/3-examples"},{"type":"function","title":"Map.has_key?/2","doc":"Returns whether the given `key` exists in the given `map`.\n\nInlined by the compiler.","ref":"Map.html#has_key?/2"},{"type":"function","title":"Examples - Map.has_key?/2","doc":"iex> Map.has_key?(%{a: 1}, :a)\n    true\n    iex> Map.has_key?(%{a: 1}, :b)\n    false","ref":"Map.html#has_key?/2-examples"},{"type":"function","title":"Map.intersect/2","doc":"Intersects two maps, returning a map with the common keys.\n\nThe values in the returned map are the values of the intersected keys in `map2`.\n\nInlined by the compiler.","ref":"Map.html#intersect/2"},{"type":"function","title":"Examples - Map.intersect/2","doc":"iex> Map.intersect(%{a: 1, b: 2}, %{b: \"b\", c: \"c\"})\n    %{b: \"b\"}","ref":"Map.html#intersect/2-examples"},{"type":"function","title":"Map.intersect/3","doc":"Intersects two maps, returning a map with the common keys and resolving conflicts through a function.\n\nThe given function will be invoked when there are duplicate keys; its\narguments are `key` (the duplicate key), `value1` (the value of `key` in\n`map1`), and `value2` (the value of `key` in `map2`). The value returned by\n`fun` is used as the value under `key` in the resulting map.","ref":"Map.html#intersect/3"},{"type":"function","title":"Examples - Map.intersect/3","doc":"iex> Map.intersect(%{a: 1, b: 2}, %{b: 2, c: 3}, fn _k, v1, v2 ->\n    ...>   v1 + v2\n    ...> end)\n    %{b: 4}","ref":"Map.html#intersect/3-examples"},{"type":"function","title":"Map.keys/1","doc":"Returns all keys from `map`.\n\nInlined by the compiler.","ref":"Map.html#keys/1"},{"type":"function","title":"Examples - Map.keys/1","doc":"iex> Map.keys(%{a: 1, b: 2})\n    [:a, :b]","ref":"Map.html#keys/1-examples"},{"type":"function","title":"Map.merge/2","doc":"Merges two maps into one.\n\nAll keys in `map2` will be added to `map1`, overriding any existing one\n(i.e., the keys in `map2` \"have precedence\" over the ones in `map1`).\n\nIf you have a struct and you would like to merge a set of keys into the\nstruct, do not use this function, as it would merge all keys on the right\nside into the struct, even if the key is not part of the struct. Instead,\nuse `struct/2`.\n\nInlined by the compiler.","ref":"Map.html#merge/2"},{"type":"function","title":"Examples - Map.merge/2","doc":"iex> Map.merge(%{a: 1, b: 2}, %{a: 3, d: 4})\n    %{a: 3, b: 2, d: 4}","ref":"Map.html#merge/2-examples"},{"type":"function","title":"Map.merge/3","doc":"Merges two maps into one, resolving conflicts through the given `fun`.\n\nAll keys in `map2` will be added to `map1`. The given function will be invoked\nwhen there are duplicate keys; its arguments are `key` (the duplicate key),\n`value1` (the value of `key` in `map1`), and `value2` (the value of `key` in\n`map2`). The value returned by `fun` is used as the value under `key` in\nthe resulting map.","ref":"Map.html#merge/3"},{"type":"function","title":"Examples - Map.merge/3","doc":"iex> Map.merge(%{a: 1, b: 2}, %{a: 3, d: 4}, fn _k, v1, v2 ->\n    ...>   v1 + v2\n    ...> end)\n    %{a: 4, b: 2, d: 4}","ref":"Map.html#merge/3-examples"},{"type":"function","title":"Map.new/0","doc":"Returns a new empty map.","ref":"Map.html#new/0"},{"type":"function","title":"Examples - Map.new/0","doc":"iex> Map.new()\n    %{}","ref":"Map.html#new/0-examples"},{"type":"function","title":"Map.new/1","doc":"Creates a map from an `enumerable`.\n\nDuplicated keys are removed; the latest one prevails.","ref":"Map.html#new/1"},{"type":"function","title":"Examples - Map.new/1","doc":"iex> Map.new([{:b, 1}, {:a, 2}])\n    %{a: 2, b: 1}\n    iex> Map.new(a: 1, a: 2, a: 3)\n    %{a: 3}","ref":"Map.html#new/1-examples"},{"type":"function","title":"Map.new/2","doc":"Creates a map from an `enumerable` via the given transformation function.\n\nDuplicated keys are removed; the latest one prevails.","ref":"Map.html#new/2"},{"type":"function","title":"Examples - Map.new/2","doc":"iex> Map.new([:a, :b], fn x -> {x, x} end)\n    %{a: :a, b: :b}\n\n    iex> Map.new(%{a: 2, b: 3, c: 4}, fn {key, val} -> {key, val * 2} end)\n    %{a: 4, b: 6, c: 8}","ref":"Map.html#new/2-examples"},{"type":"function","title":"Map.pop/3","doc":"Removes the value associated with `key` in `map` and returns the value and the updated map.\n\nIf `key` is present in `map`, it returns `{value, updated_map}` where `value` is the value of\nthe key and `updated_map` is the result of removing `key` from `map`. If `key`\nis not present in `map`, `{default, map}` is returned.","ref":"Map.html#pop/3"},{"type":"function","title":"Examples - Map.pop/3","doc":"iex> Map.pop(%{a: 1}, :a)\n    {1, %{}}\n    iex> Map.pop(%{a: 1}, :b)\n    {nil, %{a: 1}}\n    iex> Map.pop(%{a: 1}, :b, 3)\n    {3, %{a: 1}}","ref":"Map.html#pop/3-examples"},{"type":"function","title":"Map.pop!/2","doc":"Removes the value associated with `key` in `map` and returns the value\nand the updated map, or it raises if `key` is not present.\n\nBehaves the same as `pop/3` but raises if `key` is not present in `map`.","ref":"Map.html#pop!/2"},{"type":"function","title":"Examples - Map.pop!/2","doc":"iex> Map.pop!(%{a: 1}, :a)\n    {1, %{}}\n    iex> Map.pop!(%{a: 1, b: 2}, :a)\n    {1, %{b: 2}}\n    iex> Map.pop!(%{a: 1}, :b)\n    ** (KeyError) key :b not found in: %{a: 1}","ref":"Map.html#pop!/2-examples"},{"type":"function","title":"Map.pop_lazy/3","doc":"Lazily returns and removes the value associated with `key` in `map`.\n\nIf `key` is present in `map`, it returns `{value, new_map}` where `value` is the value of\nthe key and `new_map` is the result of removing `key` from `map`. If `key`\nis not present in `map`, `{fun_result, map}` is returned, where `fun_result`\nis the result of applying `fun`.\n\nThis is useful if the default value is very expensive to calculate or\ngenerally difficult to setup and teardown again.","ref":"Map.html#pop_lazy/3"},{"type":"function","title":"Examples - Map.pop_lazy/3","doc":"iex> map = %{a: 1}\n    iex> fun = fn ->\n    ...>   # some expensive operation here\n    ...>   13\n    ...> end\n    iex> Map.pop_lazy(map, :a, fun)\n    {1, %{}}\n    iex> Map.pop_lazy(map, :b, fun)\n    {13, %{a: 1}}","ref":"Map.html#pop_lazy/3-examples"},{"type":"function","title":"Map.put/3","doc":"Puts the given `value` under `key` in `map`.\n\nInlined by the compiler.","ref":"Map.html#put/3"},{"type":"function","title":"Examples - Map.put/3","doc":"iex> Map.put(%{a: 1}, :b, 2)\n    %{a: 1, b: 2}\n    iex> Map.put(%{a: 1, b: 2}, :a, 3)\n    %{a: 3, b: 2}","ref":"Map.html#put/3-examples"},{"type":"function","title":"Map.put_new/3","doc":"Puts the given `value` under `key` unless the entry `key`\nalready exists in `map`.","ref":"Map.html#put_new/3"},{"type":"function","title":"Examples - Map.put_new/3","doc":"iex> Map.put_new(%{a: 1}, :b, 2)\n    %{a: 1, b: 2}\n    iex> Map.put_new(%{a: 1, b: 2}, :a, 3)\n    %{a: 1, b: 2}","ref":"Map.html#put_new/3-examples"},{"type":"function","title":"Map.put_new_lazy/3","doc":"Evaluates `fun` and puts the result under `key`\nin `map` unless `key` is already present.\n\nThis function is useful in case you want to compute the value to put under\n`key` only if `key` is not already present, as for example, when the value is expensive to\ncalculate or generally difficult to setup and teardown again.","ref":"Map.html#put_new_lazy/3"},{"type":"function","title":"Examples - Map.put_new_lazy/3","doc":"iex> map = %{a: 1}\n    iex> fun = fn ->\n    ...>   # some expensive operation here\n    ...>   3\n    ...> end\n    iex> Map.put_new_lazy(map, :a, fun)\n    %{a: 1}\n    iex> Map.put_new_lazy(map, :b, fun)\n    %{a: 1, b: 3}","ref":"Map.html#put_new_lazy/3-examples"},{"type":"function","title":"Map.reject/2","doc":"Returns map excluding the pairs from `map` for which `fun` returns\na truthy value.\n\nSee also `filter/2`.","ref":"Map.html#reject/2"},{"type":"function","title":"Examples - Map.reject/2","doc":"iex> Map.reject(%{one: 1, two: 2, three: 3}, fn {_key, val} -> rem(val, 2) == 1 end)\n    %{two: 2}","ref":"Map.html#reject/2-examples"},{"type":"function","title":"Map.replace/3","doc":"Puts a value under `key` only if the `key` already exists in `map`.","ref":"Map.html#replace/3"},{"type":"function","title":"Examples - Map.replace/3","doc":"iex> Map.replace(%{a: 1, b: 2}, :a, 3)\n    %{a: 3, b: 2}\n\n    iex> Map.replace(%{a: 1}, :b, 2)\n    %{a: 1}","ref":"Map.html#replace/3-examples"},{"type":"function","title":"Map.replace!/3","doc":"Puts a value under `key` only if the `key` already exists in `map`.\n\nIf `key` is not present in `map`, a `KeyError` exception is raised.\n\nInlined by the compiler.","ref":"Map.html#replace!/3"},{"type":"function","title":"Examples - Map.replace!/3","doc":"iex> Map.replace!(%{a: 1, b: 2}, :a, 3)\n    %{a: 3, b: 2}\n\n    iex> Map.replace!(%{a: 1}, :b, 2)\n    ** (KeyError) key :b not found in: %{a: 1}","ref":"Map.html#replace!/3-examples"},{"type":"function","title":"Map.replace_lazy/3","doc":"Replaces the value under `key` using the given function only if\n`key` already exists in `map`.\n\nIn comparison to `replace/3`, this can be useful when it's expensive to calculate the value.\n\nIf `key` does not exist, the original map is returned unchanged.","ref":"Map.html#replace_lazy/3"},{"type":"function","title":"Examples - Map.replace_lazy/3","doc":"iex> Map.replace_lazy(%{a: 1, b: 2}, :a, fn v -> v * 4 end)\n    %{a: 4, b: 2}\n\n    iex> Map.replace_lazy(%{a: 1, b: 2}, :c, fn v -> v * 4 end)\n    %{a: 1, b: 2}","ref":"Map.html#replace_lazy/3-examples"},{"type":"function","title":"Map.split/2","doc":"Takes all entries corresponding to the given `keys` in `map` and extracts\nthem into a separate map.\n\nReturns a tuple with the new map and the old map with removed keys.\n\nKeys for which there are no entries in `map` are ignored.","ref":"Map.html#split/2"},{"type":"function","title":"Examples - Map.split/2","doc":"iex> Map.split(%{a: 1, b: 2, c: 3}, [:a, :c, :e])\n    {%{a: 1, c: 3}, %{b: 2}}","ref":"Map.html#split/2-examples"},{"type":"function","title":"Map.split_with/2","doc":"Splits the `map` into two maps according to the given function `fun`.\n\n`fun` receives each `{key, value}` pair in the `map` as its only argument. Returns\na tuple with the first map containing all the elements in `map` for which\napplying `fun` returned a truthy value, and a second map with all the elements\nfor which applying `fun` returned a falsy value (`false` or `nil`).","ref":"Map.html#split_with/2"},{"type":"function","title":"Examples - Map.split_with/2","doc":"iex> Map.split_with(%{a: 1, b: 2, c: 3, d: 4}, fn {_k, v} -> rem(v, 2) == 0 end)\n    {%{b: 2, d: 4}, %{a: 1, c: 3}}\n\n    iex> Map.split_with(%{a: 1, b: -2, c: 1, d: -3}, fn {k, _v} -> k in [:b, :d] end)\n    {%{b: -2, d: -3}, %{a: 1, c: 1}}\n\n    iex> Map.split_with(%{a: 1, b: -2, c: 1, d: -3}, fn {_k, v} -> v > 50 end)\n    {%{}, %{a: 1, b: -2, c: 1, d: -3}}\n\n    iex> Map.split_with(%{}, fn {_k, v} -> v > 50 end)\n    {%{}, %{}}","ref":"Map.html#split_with/2-examples"},{"type":"function","title":"Map.take/2","doc":"Returns a new map with all the key-value pairs in `map` where the key\nis in `keys`.\n\nIf `keys` contains keys that are not in `map`, they're simply ignored.","ref":"Map.html#take/2"},{"type":"function","title":"Examples - Map.take/2","doc":"iex> Map.take(%{a: 1, b: 2, c: 3}, [:a, :c, :e])\n    %{a: 1, c: 3}","ref":"Map.html#take/2-examples"},{"type":"function","title":"Map.to_list/1","doc":"Converts `map` to a list.\n\nEach key-value pair in the map is converted to a two-element tuple `{key,\nvalue}` in the resulting list.\n\nInlined by the compiler.","ref":"Map.html#to_list/1"},{"type":"function","title":"Examples - Map.to_list/1","doc":"iex> Map.to_list(%{a: 1})\n    [a: 1]\n    iex> Map.to_list(%{1 => 2})\n    [{1, 2}]","ref":"Map.html#to_list/1-examples"},{"type":"function","title":"Map.update/4","doc":"Updates the `key` in `map` with the given function.\n\nIf `key` is present in `map` then the existing value is passed to `fun` and its result is\nused as the updated value of `key`. If `key` is\nnot present in `map`, `default` is inserted as the value of `key`. The default\nvalue will not be passed through the update function.","ref":"Map.html#update/4"},{"type":"function","title":"Examples - Map.update/4","doc":"iex> Map.update(%{a: 1}, :a, 13, fn existing_value -> existing_value * 2 end)\n    %{a: 2}\n    iex> Map.update(%{a: 1}, :b, 11, fn existing_value -> existing_value * 2 end)\n    %{a: 1, b: 11}","ref":"Map.html#update/4-examples"},{"type":"function","title":"Map.update!/3","doc":"Updates `key` with the given function.\n\nIf `key` is present in `map` then the existing value is passed to `fun` and its result is\nused as the updated value of `key`. If `key` is\nnot present in `map`, a `KeyError` exception is raised.","ref":"Map.html#update!/3"},{"type":"function","title":"Examples - Map.update!/3","doc":"iex> Map.update!(%{a: 1}, :a, &(&1 * 2))\n    %{a: 2}\n\n    iex> Map.update!(%{a: 1}, :b, &(&1 * 2))\n    ** (KeyError) key :b not found in: %{a: 1}","ref":"Map.html#update!/3-examples"},{"type":"function","title":"Map.values/1","doc":"Returns all values from `map`.\n\nInlined by the compiler.","ref":"Map.html#values/1"},{"type":"function","title":"Examples - Map.values/1","doc":"iex> Map.values(%{a: 1, b: 2})\n    [1, 2]","ref":"Map.html#values/1-examples"},{"type":"type","title":"Map.key/0","doc":"","ref":"Map.html#t:key/0"},{"type":"type","title":"Map.value/0","doc":"","ref":"Map.html#t:value/0"},{"type":"module","title":"MapSet","doc":"Functions that work on sets.\n\nA set is a data structure that can contain unique elements of any kind,\nwithout any particular order. `MapSet` is the \"go to\" set data structure in Elixir.\n\nA set can be constructed using `MapSet.new/0`:\n\n    iex> MapSet.new()\n    MapSet.new([])\n\nElements in a set don't have to be of the same type and they can be\npopulated from an [enumerable](`t:Enumerable.t/0`) using `MapSet.new/1`:\n\n    iex> MapSet.new([1, :two, {\"three\"}])\n    MapSet.new([1, :two, {\"three\"}])\n\nElements can be inserted using `MapSet.put/2`:\n\n    iex> MapSet.new([2]) |> MapSet.put(4) |> MapSet.put(0)\n    MapSet.new([0, 2, 4])\n\nBy definition, sets can't contain duplicate elements: when\ninserting an element in a set where it's already present, the insertion is\nsimply a no-op.\n\n    iex> map_set = MapSet.new()\n    iex> MapSet.put(map_set, \"foo\")\n    MapSet.new([\"foo\"])\n    iex> map_set |> MapSet.put(\"foo\") |> MapSet.put(\"foo\")\n    MapSet.new([\"foo\"])\n\nA `MapSet` is represented internally using the `%MapSet{}` struct. This struct\ncan be used whenever there's a need to pattern match on something being a `MapSet`:\n\n    iex> match?(%MapSet{}, MapSet.new())\n    true\n\nNote that, however, the struct fields are private and must not be accessed\ndirectly; use the functions in this module to perform operations on sets.\n\n`MapSet`s can also be constructed starting from other collection-type data\nstructures: for example, see `MapSet.new/1` or `Enum.into/2`.\n\n`MapSet` is built on top of Erlang's\n[`:sets`](https://www.erlang.org/doc/man/sets.html) (version 2). This means\nthat they share many properties, including logarithmic time complexity. Erlang\n`:sets` (version 2) are implemented on top of maps, so see the documentation\nfor `Map` for more information on its execution time complexity.","ref":"MapSet.html"},{"type":"function","title":"MapSet.delete/2","doc":"Deletes `value` from `map_set`.\n\nReturns a new set which is a copy of `map_set` but without `value`.","ref":"MapSet.html#delete/2"},{"type":"function","title":"Examples - MapSet.delete/2","doc":"iex> map_set = MapSet.new([1, 2, 3])\n    iex> MapSet.delete(map_set, 4)\n    MapSet.new([1, 2, 3])\n    iex> MapSet.delete(map_set, 2)\n    MapSet.new([1, 3])","ref":"MapSet.html#delete/2-examples"},{"type":"function","title":"MapSet.difference/2","doc":"Returns a set that is `map_set1` without the members of `map_set2`.","ref":"MapSet.html#difference/2"},{"type":"function","title":"Examples - MapSet.difference/2","doc":"iex> MapSet.difference(MapSet.new([1, 2]), MapSet.new([2, 3, 4]))\n    MapSet.new([1])","ref":"MapSet.html#difference/2-examples"},{"type":"function","title":"MapSet.disjoint?/2","doc":"Checks if `map_set1` and `map_set2` have no members in common.","ref":"MapSet.html#disjoint?/2"},{"type":"function","title":"Examples - MapSet.disjoint?/2","doc":"iex> MapSet.disjoint?(MapSet.new([1, 2]), MapSet.new([3, 4]))\n    true\n    iex> MapSet.disjoint?(MapSet.new([1, 2]), MapSet.new([2, 3]))\n    false","ref":"MapSet.html#disjoint?/2-examples"},{"type":"function","title":"MapSet.equal?/2","doc":"Checks if two sets are equal.\n\nThe comparison between elements is done using `===/2`,\nwhich a set with `1` is not equivalent to a set with\n`1.0`.","ref":"MapSet.html#equal?/2"},{"type":"function","title":"Examples - MapSet.equal?/2","doc":"iex> MapSet.equal?(MapSet.new([1, 2]), MapSet.new([2, 1, 1]))\n    true\n    iex> MapSet.equal?(MapSet.new([1, 2]), MapSet.new([3, 4]))\n    false\n    iex> MapSet.equal?(MapSet.new([1]), MapSet.new([1.0]))\n    false","ref":"MapSet.html#equal?/2-examples"},{"type":"function","title":"MapSet.filter/2","doc":"Filters the set by returning only the elements from `map_set` for which invoking\n`fun` returns a truthy value.\n\nAlso see `reject/2` which discards all elements where the function returns\na truthy value.\n\n> ##","ref":"MapSet.html#filter/2"},{"type":"function","title":"Performance considerations {: .tip} - MapSet.filter/2","doc":">\n> If you find yourself doing multiple calls to `MapSet.filter/2`\n> and `MapSet.reject/2` in a pipeline, it is likely more efficient\n> to use `Enum.map/2` and `Enum.filter/2` instead and convert to\n> a map at the end using `MapSet.new/1`.","ref":"MapSet.html#filter/2-performance-considerations-tip"},{"type":"function","title":"Examples - MapSet.filter/2","doc":"iex> MapSet.filter(MapSet.new(1..5), fn x -> x > 3 end)\n    MapSet.new([4, 5])\n\n    iex> MapSet.filter(MapSet.new([\"a\", :b, \"c\"]), &is_atom/1)\n    MapSet.new([:b])","ref":"MapSet.html#filter/2-examples"},{"type":"function","title":"MapSet.intersection/2","doc":"Returns a set containing only members that `map_set1` and `map_set2` have in common.","ref":"MapSet.html#intersection/2"},{"type":"function","title":"Examples - MapSet.intersection/2","doc":"iex> MapSet.intersection(MapSet.new([1, 2]), MapSet.new([2, 3, 4]))\n    MapSet.new([2])\n\n    iex> MapSet.intersection(MapSet.new([1, 2]), MapSet.new([3, 4]))\n    MapSet.new([])","ref":"MapSet.html#intersection/2-examples"},{"type":"function","title":"MapSet.member?/2","doc":"Checks if `map_set` contains `value`.","ref":"MapSet.html#member?/2"},{"type":"function","title":"Examples - MapSet.member?/2","doc":"iex> MapSet.member?(MapSet.new([1, 2, 3]), 2)\n    true\n    iex> MapSet.member?(MapSet.new([1, 2, 3]), 4)\n    false","ref":"MapSet.html#member?/2-examples"},{"type":"function","title":"MapSet.new/0","doc":"Returns a new set.","ref":"MapSet.html#new/0"},{"type":"function","title":"Examples - MapSet.new/0","doc":"iex> MapSet.new()\n    MapSet.new([])","ref":"MapSet.html#new/0-examples"},{"type":"function","title":"MapSet.new/1","doc":"Creates a set from an enumerable.","ref":"MapSet.html#new/1"},{"type":"function","title":"Examples - MapSet.new/1","doc":"iex> MapSet.new([:b, :a, 3])\n    MapSet.new([3, :a, :b])\n    iex> MapSet.new([3, 3, 3, 2, 2, 1])\n    MapSet.new([1, 2, 3])","ref":"MapSet.html#new/1-examples"},{"type":"function","title":"MapSet.new/2","doc":"Creates a set from an enumerable via the transformation function.","ref":"MapSet.html#new/2"},{"type":"function","title":"Examples - MapSet.new/2","doc":"iex> MapSet.new([1, 2, 1], fn x -> 2 * x end)\n    MapSet.new([2, 4])","ref":"MapSet.html#new/2-examples"},{"type":"function","title":"MapSet.put/2","doc":"Inserts `value` into `map_set` if `map_set` doesn't already contain it.","ref":"MapSet.html#put/2"},{"type":"function","title":"Examples - MapSet.put/2","doc":"iex> MapSet.put(MapSet.new([1, 2, 3]), 3)\n    MapSet.new([1, 2, 3])\n    iex> MapSet.put(MapSet.new([1, 2, 3]), 4)\n    MapSet.new([1, 2, 3, 4])","ref":"MapSet.html#put/2-examples"},{"type":"function","title":"MapSet.reject/2","doc":"Returns a set by excluding the elements from `map_set` for which invoking `fun`\nreturns a truthy value.\n\nSee also `filter/2`.","ref":"MapSet.html#reject/2"},{"type":"function","title":"Examples - MapSet.reject/2","doc":"iex> MapSet.reject(MapSet.new(1..5), fn x -> rem(x, 2) != 0 end)\n    MapSet.new([2, 4])\n\n    iex> MapSet.reject(MapSet.new([\"a\", :b, \"c\"]), &is_atom/1)\n    MapSet.new([\"a\", \"c\"])","ref":"MapSet.html#reject/2-examples"},{"type":"function","title":"MapSet.size/1","doc":"Returns the number of elements in `map_set`.","ref":"MapSet.html#size/1"},{"type":"function","title":"Examples - MapSet.size/1","doc":"iex> MapSet.size(MapSet.new([1, 2, 3]))\n    3","ref":"MapSet.html#size/1-examples"},{"type":"function","title":"MapSet.split_with/2","doc":"Splits the `map_set` into two `MapSet`s according to the given function `fun`.\n\n`fun` receives each element in the `map_set` as its only argument. Returns\na tuple with the first `MapSet` containing all the elements in `map_set` for which\napplying `fun` returned a truthy value, and a second `MapSet` with all the elements\nfor which applying `fun` returned a falsy value (`false` or `nil`).","ref":"MapSet.html#split_with/2"},{"type":"function","title":"Examples - MapSet.split_with/2","doc":"iex> {while_true, while_false} = MapSet.split_with(MapSet.new([1, 2, 3, 4]), fn v -> rem(v, 2) == 0 end)\n    iex> while_true\n    MapSet.new([2, 4])\n    iex> while_false\n    MapSet.new([1, 3])\n\n    iex> {while_true, while_false} = MapSet.split_with(MapSet.new(), fn {_k, v} -> v > 50 end)\n    iex> while_true\n    MapSet.new([])\n    iex> while_false\n    MapSet.new([])","ref":"MapSet.html#split_with/2-examples"},{"type":"function","title":"MapSet.subset?/2","doc":"Checks if `map_set1`'s members are all contained in `map_set2`.\n\nThis function checks if `map_set1` is a subset of `map_set2`.","ref":"MapSet.html#subset?/2"},{"type":"function","title":"Examples - MapSet.subset?/2","doc":"iex> MapSet.subset?(MapSet.new([1, 2]), MapSet.new([1, 2, 3]))\n    true\n    iex> MapSet.subset?(MapSet.new([1, 2, 3]), MapSet.new([1, 2]))\n    false","ref":"MapSet.html#subset?/2-examples"},{"type":"function","title":"MapSet.symmetric_difference/2","doc":"Returns a set with elements that are present in only one but not both sets.","ref":"MapSet.html#symmetric_difference/2"},{"type":"function","title":"Examples - MapSet.symmetric_difference/2","doc":"iex> MapSet.symmetric_difference(MapSet.new([1, 2, 3]), MapSet.new([2, 3, 4]))\n    MapSet.new([1, 4])","ref":"MapSet.html#symmetric_difference/2-examples"},{"type":"function","title":"MapSet.to_list/1","doc":"Converts `map_set` to a list.","ref":"MapSet.html#to_list/1"},{"type":"function","title":"Examples - MapSet.to_list/1","doc":"iex> MapSet.to_list(MapSet.new([1, 2, 3]))\n    [1, 2, 3]","ref":"MapSet.html#to_list/1-examples"},{"type":"function","title":"MapSet.union/2","doc":"Returns a set containing all members of `map_set1` and `map_set2`.","ref":"MapSet.html#union/2"},{"type":"function","title":"Examples - MapSet.union/2","doc":"iex> MapSet.union(MapSet.new([1, 2]), MapSet.new([2, 3, 4]))\n    MapSet.new([1, 2, 3, 4])","ref":"MapSet.html#union/2-examples"},{"type":"opaque","title":"MapSet.internal/1","doc":"","ref":"MapSet.html#t:internal/1"},{"type":"type","title":"MapSet.t/0","doc":"","ref":"MapSet.html#t:t/0"},{"type":"type","title":"MapSet.t/1","doc":"","ref":"MapSet.html#t:t/1"},{"type":"type","title":"MapSet.value/0","doc":"","ref":"MapSet.html#t:value/0"},{"type":"module","title":"Range","doc":"Ranges represent a sequence of zero, one or many, ascending\nor descending integers with a common difference called step.\n\nThe most common form of creating and matching on ranges is\nvia the [`first..last`](`../2`) and [`first..last//step`](`..///3`)\nnotations, auto-imported from `Kernel`:\n\n    iex> 1 in 1..10\n    true\n    iex> 5 in 1..10\n    true\n    iex> 10 in 1..10\n    true\n\nRanges are always inclusive in Elixir. When a step is defined,\nintegers will only belong to the range if they match the step:\n\n    iex> 5 in 1..10//2\n    true\n    iex> 4 in 1..10//2\n    false\n\nWhen defining a range without a step, the step will be\ndefined based on the first and last position of the\nrange, If `last >= first`, it will be an increasing range\nwith a step of 1. Otherwise, it is a decreasing range.\nNote, however, implicit decreasing ranges are deprecated.\nTherefore, if you need a decreasing range from `3` to `1`,\nprefer to write `3..1//-1` instead.\n\n`../0` can also be used as a shortcut to create the range `0..-1//1`,\nalso known as the full-slice range:\n\n    iex> ..\n    0..-1//1","ref":"Range.html"},{"type":"module","title":"Use cases - Range","doc":"Ranges typically have two uses in Elixir: as a collection or\nto represent a slice of another data structure.\n\n#","ref":"Range.html#module-use-cases"},{"type":"module","title":"Ranges as collections - Range","doc":"Ranges in Elixir are enumerables and therefore can be used\nwith the `Enum` module:\n\n    iex> Enum.to_list(1..3)\n    [1, 2, 3]\n    iex> Enum.to_list(3..1//-1)\n    [3, 2, 1]\n    iex> Enum.to_list(1..5//2)\n    [1, 3, 5]\n\nRanges may also have a single element:\n\n    iex> Enum.to_list(1..1)\n    [1]\n    iex> Enum.to_list(1..1//2)\n    [1]\n\nOr even no elements at all:\n\n    iex> Enum.to_list(10..0//1)\n    []\n    iex> Enum.to_list(0..10//-1)\n    []\n\nThe full-slice range, returned by `../0`, is an empty collection:\n\n    iex> Enum.to_list(..)\n    []\n\n#","ref":"Range.html#module-ranges-as-collections"},{"type":"module","title":"Ranges as slices - Range","doc":"Ranges are also frequently used to slice collections.\nYou can slice strings or any enumerable:\n\n    iex> String.slice(\"elixir\", 1..4)\n    \"lixi\"\n    iex> Enum.slice([0, 1, 2, 3, 4, 5], 1..4)\n    [1, 2, 3, 4]\n\nIn those cases, the first and last values of the range\nare mapped to positions in the collections.\n\nIf a negative number is given, it maps to a position\nfrom the back:\n\n    iex> String.slice(\"elixir\", 1..-2//1)\n    \"lixi\"\n    iex> Enum.slice([0, 1, 2, 3, 4, 5], 1..-2//1)\n    [1, 2, 3, 4]\n\nThe range `0..-1//1`, returned by `../0`, returns the\ncollection as is, which is why it is called the full-slice\nrange:\n\n    iex> String.slice(\"elixir\", ..)\n    \"elixir\"\n    iex> Enum.slice([0, 1, 2, 3, 4, 5], ..)\n    [0, 1, 2, 3, 4, 5]","ref":"Range.html#module-ranges-as-slices"},{"type":"module","title":"Definition - Range","doc":"An increasing range `first..last//step` is a range from `first`\nto `last` increasing by `step` where  `step` must be a positive\ninteger and all values `v` must be `first  = v and v >= last`.\nTherefore, a range `0..10//-1` is an empty range because there\nis no value `v` that is `0 >= v and v >= 10`.","ref":"Range.html#module-definition"},{"type":"module","title":"Representation - Range","doc":"Internally, ranges are represented as structs:\n\n    iex> range = 1..9//2\n    1..9//2\n    iex> first..last//step = range\n    iex> first\n    1\n    iex> last\n    9\n    iex> step\n    2\n    iex> range.step\n    2\n\nYou can access the range fields (`first`, `last`, and `step`)\ndirectly but you should not modify nor create ranges by hand.\nInstead use the proper operators or `new/2` and `new/3`.\n\nRanges implement the `Enumerable` protocol with memory\nefficient versions of all `Enumerable` callbacks:\n\n    iex> range = 1..10\n    1..10\n    iex> Enum.reduce(range, 0, fn i, acc -> i * i + acc end)\n    385\n    iex> Enum.count(range)\n    10\n    iex> Enum.member?(range, 11)\n    false\n    iex> Enum.member?(range, 8)\n    true\n\nSuch function calls are efficient memory-wise no matter the\nsize of the range. The implementation of the `Enumerable`\nprotocol uses logic based solely on the endpoints and does\nnot materialize the whole list of integers.","ref":"Range.html#module-representation"},{"type":"function","title":"Range.disjoint?/2","doc":"Checks if two ranges are disjoint.","ref":"Range.html#disjoint?/2"},{"type":"function","title":"Examples - Range.disjoint?/2","doc":"iex> Range.disjoint?(1..5, 6..9)\n    true\n    iex> Range.disjoint?(5..1, 6..9)\n    true\n    iex> Range.disjoint?(1..5, 5..9)\n    false\n    iex> Range.disjoint?(1..5, 2..7)\n    false\n\nSteps are also considered when computing the ranges to be disjoint:\n\n    iex> Range.disjoint?(1..10//2, 2..10//2)\n    true\n\n    # First element in common is 29\n    iex> Range.disjoint?(1..100//14, 8..100//21)\n    false\n    iex> Range.disjoint?(57..-1//-14, 8..100//21)\n    false\n    iex> Range.disjoint?(1..100//14, 50..8//-21)\n    false\n    iex> Range.disjoint?(1..28//14, 8..28//21)\n    true\n\n    # First element in common is 14\n    iex> Range.disjoint?(2..28//3, 9..28//5)\n    false\n    iex> Range.disjoint?(26..2//-3, 29..9//-5)\n    false\n\n    # Starting from the back without alignment\n    iex> Range.disjoint?(27..11//-3, 30..0//-7)\n    true","ref":"Range.html#disjoint?/2-examples"},{"type":"function","title":"Range.new/2","doc":"Creates a new range.\n\nIf `first` is less than `last`, the range will be increasing from\n`first` to `last`. If `first` is equal to `last`, the range will contain\none element, which is the number itself.\n\nIf `first` is greater than `last`, the range will be decreasing from `first`\nto `last`, albeit this behaviour is deprecated. Therefore, it is advised to\nexplicitly list the step with `new/3`.","ref":"Range.html#new/2"},{"type":"function","title":"Examples - Range.new/2","doc":"iex> Range.new(-100, 100)\n    -100..100","ref":"Range.html#new/2-examples"},{"type":"function","title":"Range.new/3","doc":"Creates a new range with `step`.","ref":"Range.html#new/3"},{"type":"function","title":"Examples - Range.new/3","doc":"iex> Range.new(-100, 100, 2)\n    -100..100//2","ref":"Range.html#new/3-examples"},{"type":"function","title":"Range.shift/2","doc":"Shifts a range by the given number of steps.","ref":"Range.html#shift/2"},{"type":"function","title":"Examples - Range.shift/2","doc":"iex> Range.shift(0..10, 1)\n    1..11\n    iex> Range.shift(0..10, 2)\n    2..12\n\n    iex> Range.shift(0..10//2, 2)\n    4..14//2\n    iex> Range.shift(10..0//-2, 2)\n    6..-4//-2","ref":"Range.html#shift/2-examples"},{"type":"function","title":"Range.size/1","doc":"Returns the size of `range`.","ref":"Range.html#size/1"},{"type":"function","title":"Examples - Range.size/1","doc":"iex> Range.size(1..10)\n    10\n    iex> Range.size(1..10//2)\n    5\n    iex> Range.size(1..10//3)\n    4\n    iex> Range.size(1..10//-1)\n    0\n\n    iex> Range.size(10..1)\n    10\n    iex> Range.size(10..1//-1)\n    10\n    iex> Range.size(10..1//-2)\n    5\n    iex> Range.size(10..1//-3)\n    4\n    iex> Range.size(10..1//1)\n    0","ref":"Range.html#size/1-examples"},{"type":"function","title":"Range.split/2","doc":"Splits a range in two.\n\nIt returns a tuple of two elements.\n\nIf `split` is less than the number of elements in the range, the first\nelement in the range will have `split` entries and the second will have\nall remaining entries.\n\nIf `split` is more than the number of elements in the range, the second\nrange in the tuple will emit zero elements.","ref":"Range.html#split/2"},{"type":"function","title":"Examples - Range.split/2","doc":"Increasing ranges:\n\n    iex> Range.split(1..5, 2)\n    {1..2, 3..5}\n\n    iex> Range.split(1..5//2, 2)\n    {1..3//2, 5..5//2}\n\n    iex> Range.split(1..5//2, 0)\n    {1..-1//2, 1..5//2}\n\n    iex> Range.split(1..5//2, 10)\n    {1..5//2, 7..5//2}\n\nDecreasing ranges can also be split:\n\n    iex> Range.split(5..1//-1, 2)\n    {5..4//-1, 3..1//-1}\n\n    iex> Range.split(5..1//-2, 2)\n    {5..3//-2, 1..1//-2}\n\n    iex> Range.split(5..1//-2, 0)\n    {5..7//-2, 5..1//-2}\n\n    iex> Range.split(5..1//-2, 10)\n    {5..1//-2, -1..1//-2}\n\nEmpty ranges preserve their property but still return empty ranges:\n\n    iex> Range.split(2..5//-1, 2)\n    {2..3//-1, 4..5//-1}\n\n    iex> Range.split(2..5//-1, 10)\n    {2..3//-1, 4..5//-1}\n\n    iex> Range.split(5..2//1, 2)\n    {5..4//1, 3..2//1}\n\n    iex> Range.split(5..2//1, 10)\n    {5..4//1, 3..2//1}\n\nIf the number to split is negative, it splits from the back:\n\n    iex> Range.split(1..5, -2)\n    {1..3, 4..5}\n\n    iex> Range.split(5..1//-1, -2)\n    {5..3//-1, 2..1//-1}\n\nIf it is negative and greater than the elements in the range,\nthe first element of the tuple will be an empty range:\n\n    iex> Range.split(1..5, -10)\n    {1..0//1, 1..5}\n\n    iex> Range.split(5..1//-1, -10)\n    {5..6//-1, 5..1//-1}","ref":"Range.html#split/2-examples"},{"type":"function","title":"Properties - Range.split/2","doc":"When a range is split, the following properties are observed.\nGiven `split(input)` returns `{left, right}`, we have:\n\n    assert input.first == left.first\n    assert input.last == right.last\n    assert input.step == left.step\n    assert input.step == right.step\n    assert Range.size(input) == Range.size(left) + Range.size(right)","ref":"Range.html#split/2-properties"},{"type":"function","title":"Range.to_list/1","doc":"Converts a range to a list.","ref":"Range.html#to_list/1"},{"type":"type","title":"Range.limit/0","doc":"","ref":"Range.html#t:limit/0"},{"type":"type","title":"Range.step/0","doc":"","ref":"Range.html#t:step/0"},{"type":"type","title":"Range.t/0","doc":"","ref":"Range.html#t:t/0"},{"type":"type","title":"Range.t/2","doc":"","ref":"Range.html#t:t/2"},{"type":"module","title":"Stream","doc":"Functions for creating and composing streams.\n\nStreams are composable, lazy enumerables (for an introduction on\nenumerables, see the `Enum` module). Any enumerable that generates\nelements one by one during enumeration is called a stream. For example,\nElixir's `Range` is a stream:\n\n    iex> range = 1..5\n    1..5\n    iex> Enum.map(range, &(&1 * 2))\n    [2, 4, 6, 8, 10]\n\nIn the example above, as we mapped over the range, the elements being\nenumerated were created one by one, during enumeration. The `Stream`\nmodule allows us to map the range, without triggering its enumeration:\n\n    iex> range = 1..3\n    iex> stream = Stream.map(range, &(&1 * 2))\n    iex> Enum.map(stream, &(&1 + 1))\n    [3, 5, 7]\n\nNote that we started with a range and then we created a stream that is\nmeant to multiply each element in the range by 2. At this point, no\ncomputation was done. Only when `Enum.map/2` is called we actually\nenumerate over each element in the range, multiplying it by 2 and adding 1.\nWe say the functions in `Stream` are *lazy* and the functions in `Enum`\nare *eager*.\n\nDue to their laziness, streams are useful when working with large\n(or even infinite) collections. When chaining many operations with `Enum`,\nintermediate lists are created, while `Stream` creates a recipe of\ncomputations that are executed at a later moment. Let's see another\nexample:\n\n    1..3\n    |> Enum.map(&IO.inspect(&1))\n    |> Enum.map(&(&1 * 2))\n    |> Enum.map(&IO.inspect(&1))\n    1\n    2\n    3\n    2\n    4\n    6\n    #=> [2, 4, 6]\n\nNote that we first printed each element in the list, then multiplied each\nelement by 2 and finally printed each new value. In this example, the list\nwas enumerated three times. Let's see an example with streams:\n\n    stream = 1..3\n    |> Stream.map(&IO.inspect(&1))\n    |> Stream.map(&(&1 * 2))\n    |> Stream.map(&IO.inspect(&1))\n    Enum.to_list(stream)\n    1\n    2\n    2\n    4\n    3\n    6\n    #=> [2, 4, 6]\n\nAlthough the end result is the same, the order in which the elements were\nprinted changed! With streams, we print the first element and then print\nits double. In this example, the list was enumerated just once!\n\nThat's what we meant when we said earlier that streams are composable,\nlazy enumerables. Note that we could call `Stream.map/2` multiple times,\neffectively composing the streams and keeping them lazy. The computations\nare only performed when you call a function from the `Enum` module.\n\nLike with `Enum`, the functions in this module work in linear time. This\nmeans that, the time it takes to perform an operation grows at the same\nrate as the length of the list. This is expected on operations such as\n`Stream.map/2`. After all, if we want to traverse every element on a\nstream, the longer the stream, the more elements we need to traverse,\nand the longer it will take.","ref":"Stream.html"},{"type":"module","title":"Creating Streams - Stream","doc":"There are many functions in Elixir's standard library that return\nstreams, some examples are:\n\n  * `IO.stream/2`         - streams input lines, one by one\n  * `URI.query_decoder/1` - decodes a query string, pair by pair\n\nThis module also provides many convenience functions for creating streams,\nlike `Stream.cycle/1`, `Stream.unfold/2`, `Stream.resource/3` and more.\n\nNote the functions in this module are guaranteed to return enumerables.\nSince enumerables can have different shapes (structs, anonymous functions,\nand so on), the functions in this module may return any of those shapes\nand this may change at any time. For example, a function that today\nreturns an anonymous function may return a struct in future releases.","ref":"Stream.html#module-creating-streams"},{"type":"function","title":"Stream.chunk_by/2","doc":"Chunks the `enum` by buffering elements for which `fun` returns the same value.\n\nElements are only emitted when `fun` returns a new value or the `enum` finishes.","ref":"Stream.html#chunk_by/2"},{"type":"function","title":"Examples - Stream.chunk_by/2","doc":"iex> stream = Stream.chunk_by([1, 2, 2, 3, 4, 4, 6, 7, 7], &(rem(&1, 2) == 1))\n    iex> Enum.to_list(stream)\n    [[1], [2, 2], [3], [4, 4, 6], [7, 7]]","ref":"Stream.html#chunk_by/2-examples"},{"type":"function","title":"Stream.chunk_every/2","doc":"Shortcut to `chunk_every(enum, count, count)`.","ref":"Stream.html#chunk_every/2"},{"type":"function","title":"Stream.chunk_every/4","doc":"Streams the enumerable in chunks, containing `count` elements each,\nwhere each new chunk starts `step` elements into the enumerable.\n\n`step` is optional and, if not passed, defaults to `count`, i.e.\nchunks do not overlap. Chunking will stop as soon as the collection\nends or when we emit an incomplete chunk.\n\nIf the last chunk does not have `count` elements to fill the chunk,\nelements are taken from `leftover` to fill in the chunk. If `leftover`\ndoes not have enough elements to fill the chunk, then a partial chunk\nis returned with less than `count` elements.\n\nIf `:discard` is given in `leftover`, the last chunk is discarded\nunless it has exactly `count` elements.","ref":"Stream.html#chunk_every/4"},{"type":"function","title":"Examples - Stream.chunk_every/4","doc":"iex> Stream.chunk_every([1, 2, 3, 4, 5, 6], 2) |> Enum.to_list()\n    [[1, 2], [3, 4], [5, 6]]\n\n    iex> Stream.chunk_every([1, 2, 3, 4, 5, 6], 3, 2, :discard) |> Enum.to_list()\n    [[1, 2, 3], [3, 4, 5]]\n\n    iex> Stream.chunk_every([1, 2, 3, 4, 5, 6], 3, 2, [7]) |> Enum.to_list()\n    [[1, 2, 3], [3, 4, 5], [5, 6, 7]]\n\n    iex> Stream.chunk_every([1, 2, 3, 4, 5, 6], 3, 3, []) |> Enum.to_list()\n    [[1, 2, 3], [4, 5, 6]]\n\n    iex> Stream.chunk_every([1, 2, 3, 4], 3, 3, Stream.cycle([0])) |> Enum.to_list()\n    [[1, 2, 3], [4, 0, 0]]","ref":"Stream.html#chunk_every/4-examples"},{"type":"function","title":"Stream.chunk_while/4","doc":"Chunks the `enum` with fine grained control when every chunk is emitted.\n\n`chunk_fun` receives the current element and the accumulator and\nmust return `{:cont, element, acc}` to emit the given chunk and\ncontinue with accumulator or `{:cont, acc}` to not emit any chunk\nand continue with the return accumulator.\n\n`after_fun` is invoked when iteration is done and must also return\n`{:cont, element, acc}` or `{:cont, acc}`.","ref":"Stream.html#chunk_while/4"},{"type":"function","title":"Examples - Stream.chunk_while/4","doc":"iex> chunk_fun = fn element, acc ->\n    ...>   if rem(element, 2) == 0 do\n    ...>     {:cont, Enum.reverse([element | acc]), []}\n    ...>   else\n    ...>     {:cont, [element | acc]}\n    ...>   end\n    ...> end\n    iex> after_fun = fn\n    ...>   [] -> {:cont, []}\n    ...>   acc -> {:cont, Enum.reverse(acc), []}\n    ...> end\n    iex> stream = Stream.chunk_while(1..10, [], chunk_fun, after_fun)\n    iex> Enum.to_list(stream)\n    [[1, 2], [3, 4], [5, 6], [7, 8], [9, 10]]","ref":"Stream.html#chunk_while/4-examples"},{"type":"function","title":"Stream.concat/1","doc":"Creates a stream that enumerates each enumerable in an enumerable.","ref":"Stream.html#concat/1"},{"type":"function","title":"Examples - Stream.concat/1","doc":"iex> stream = Stream.concat([1..3, 4..6, 7..9])\n    iex> Enum.to_list(stream)\n    [1, 2, 3, 4, 5, 6, 7, 8, 9]","ref":"Stream.html#concat/1-examples"},{"type":"function","title":"Stream.concat/2","doc":"Creates a stream that enumerates the first argument, followed by the second.","ref":"Stream.html#concat/2"},{"type":"function","title":"Examples - Stream.concat/2","doc":"iex> stream = Stream.concat(1..3, 4..6)\n    iex> Enum.to_list(stream)\n    [1, 2, 3, 4, 5, 6]\n\n    iex> stream1 = Stream.cycle([1, 2, 3])\n    iex> stream2 = Stream.cycle([4, 5, 6])\n    iex> stream = Stream.concat(stream1, stream2)\n    iex> Enum.take(stream, 6)\n    [1, 2, 3, 1, 2, 3]","ref":"Stream.html#concat/2-examples"},{"type":"function","title":"Stream.cycle/1","doc":"Creates a stream that cycles through the given enumerable,\ninfinitely.","ref":"Stream.html#cycle/1"},{"type":"function","title":"Examples - Stream.cycle/1","doc":"iex> stream = Stream.cycle([1, 2, 3])\n    iex> Enum.take(stream, 5)\n    [1, 2, 3, 1, 2]","ref":"Stream.html#cycle/1-examples"},{"type":"function","title":"Stream.dedup/1","doc":"Creates a stream that only emits elements if they are different from the last emitted element.\n\nThis function only ever needs to store the last emitted element.\n\nElements are compared using `===/2`.","ref":"Stream.html#dedup/1"},{"type":"function","title":"Examples - Stream.dedup/1","doc":"iex> Stream.dedup([1, 2, 3, 3, 2, 1]) |> Enum.to_list()\n    [1, 2, 3, 2, 1]","ref":"Stream.html#dedup/1-examples"},{"type":"function","title":"Stream.dedup_by/2","doc":"Creates a stream that only emits elements if the result of calling `fun` on the element is\ndifferent from the (stored) result of calling `fun` on the last emitted element.","ref":"Stream.html#dedup_by/2"},{"type":"function","title":"Examples - Stream.dedup_by/2","doc":"iex> Stream.dedup_by([{1, :x}, {2, :y}, {2, :z}, {1, :x}], fn {x, _} -> x end) |> Enum.to_list()\n    [{1, :x}, {2, :y}, {1, :x}]","ref":"Stream.html#dedup_by/2-examples"},{"type":"function","title":"Stream.drop/2","doc":"Lazily drops the next `n` elements from the enumerable.\n\nIf a negative `n` is given, it will drop the last `n` elements from\nthe collection. Note that the mechanism by which this is implemented\nwill delay the emission of any element until `n` additional elements have\nbeen emitted by the enum.","ref":"Stream.html#drop/2"},{"type":"function","title":"Examples - Stream.drop/2","doc":"iex> stream = Stream.drop(1..10, 5)\n    iex> Enum.to_list(stream)\n    [6, 7, 8, 9, 10]\n\n    iex> stream = Stream.drop(1..10, -5)\n    iex> Enum.to_list(stream)\n    [1, 2, 3, 4, 5]","ref":"Stream.html#drop/2-examples"},{"type":"function","title":"Stream.drop_every/2","doc":"Creates a stream that drops every `nth` element from the enumerable.\n\nThe first element is always dropped, unless `nth` is 0.\n\n`nth` must be a non-negative integer.","ref":"Stream.html#drop_every/2"},{"type":"function","title":"Examples - Stream.drop_every/2","doc":"iex> stream = Stream.drop_every(1..10, 2)\n    iex> Enum.to_list(stream)\n    [2, 4, 6, 8, 10]\n\n    iex> stream = Stream.drop_every(1..1000, 1)\n    iex> Enum.to_list(stream)\n    []\n\n    iex> stream = Stream.drop_every([1, 2, 3, 4, 5], 0)\n    iex> Enum.to_list(stream)\n    [1, 2, 3, 4, 5]","ref":"Stream.html#drop_every/2-examples"},{"type":"function","title":"Stream.drop_while/2","doc":"Lazily drops elements of the enumerable while the given\nfunction returns a truthy value.","ref":"Stream.html#drop_while/2"},{"type":"function","title":"Examples - Stream.drop_while/2","doc":"iex> stream = Stream.drop_while(1..10, &(&1   Enum.to_list(stream)\n    [6, 7, 8, 9, 10]","ref":"Stream.html#drop_while/2-examples"},{"type":"function","title":"Stream.duplicate/2","doc":"Duplicates the given element `n` times in a stream.\n\n`n` is an integer greater than or equal to `0`.\n\nIf `n` is `0`, an empty stream is returned.","ref":"Stream.html#duplicate/2"},{"type":"function","title":"Examples - Stream.duplicate/2","doc":"iex> stream = Stream.duplicate(\"hello\", 0)\n    iex> Enum.to_list(stream)\n    []\n\n    iex> stream = Stream.duplicate(\"hi\", 1)\n    iex> Enum.to_list(stream)\n    [\"hi\"]\n\n    iex> stream = Stream.duplicate(\"bye\", 2)\n    iex> Enum.to_list(stream)\n    [\"bye\", \"bye\"]\n\n    iex> stream = Stream.duplicate([1, 2], 3)\n    iex> Enum.to_list(stream)\n    [[1, 2], [1, 2], [1, 2]]","ref":"Stream.html#duplicate/2-examples"},{"type":"function","title":"Stream.each/2","doc":"Executes the given function for each element.\n\nThe values in the stream do not change, therefore this\nfunction is useful for adding side effects (like printing)\nto a stream. See `map/2` if producing a different stream\nis desired.","ref":"Stream.html#each/2"},{"type":"function","title":"Examples - Stream.each/2","doc":"iex> stream = Stream.each([1, 2, 3], fn x -> send(self(), x) end)\n    iex> Enum.to_list(stream)\n    iex> receive do: (x when is_integer(x) -> x)\n    1\n    iex> receive do: (x when is_integer(x) -> x)\n    2\n    iex> receive do: (x when is_integer(x) -> x)\n    3","ref":"Stream.html#each/2-examples"},{"type":"function","title":"Stream.filter/2","doc":"Creates a stream that filters elements according to\nthe given function on enumeration.","ref":"Stream.html#filter/2"},{"type":"function","title":"Examples - Stream.filter/2","doc":"iex> stream = Stream.filter([1, 2, 3], fn x -> rem(x, 2) == 0 end)\n    iex> Enum.to_list(stream)\n    [2]","ref":"Stream.html#filter/2-examples"},{"type":"function","title":"Stream.flat_map/2","doc":"Maps the given `fun` over `enumerable` and flattens the result.\n\nThis function returns a new stream built by appending the result of invoking `fun`\non each element of `enumerable` together.","ref":"Stream.html#flat_map/2"},{"type":"function","title":"Examples - Stream.flat_map/2","doc":"iex> stream = Stream.flat_map([1, 2, 3], fn x -> [x, x * 2] end)\n    iex> Enum.to_list(stream)\n    [1, 2, 2, 4, 3, 6]\n\n    iex> stream = Stream.flat_map([1, 2, 3], fn x -> [[x]] end)\n    iex> Enum.to_list(stream)\n    [[1], [2], [3]]","ref":"Stream.html#flat_map/2-examples"},{"type":"function","title":"Stream.intersperse/2","doc":"Lazily intersperses `intersperse_element` between each element of the enumeration.","ref":"Stream.html#intersperse/2"},{"type":"function","title":"Examples - Stream.intersperse/2","doc":"iex> Stream.intersperse([1, 2, 3], 0) |> Enum.to_list()\n    [1, 0, 2, 0, 3]\n\n    iex> Stream.intersperse([1], 0) |> Enum.to_list()\n    [1]\n\n    iex> Stream.intersperse([], 0) |> Enum.to_list()\n    []","ref":"Stream.html#intersperse/2-examples"},{"type":"function","title":"Stream.interval/1","doc":"Creates a stream that emits a value after the given period `n`\nin milliseconds.\n\nThe values emitted are an increasing counter starting at `0`.\nThis operation will block the caller by the given interval\nevery time a new element is streamed.\n\nDo not use this function to generate a sequence of numbers.\nIf blocking the caller process is not necessary, use\n`Stream.iterate(0, & &1 + 1)` instead.","ref":"Stream.html#interval/1"},{"type":"function","title":"Examples - Stream.interval/1","doc":"iex> Stream.interval(10) |> Enum.take(10)\n    [0, 1, 2, 3, 4, 5, 6, 7, 8, 9]","ref":"Stream.html#interval/1-examples"},{"type":"function","title":"Stream.into/3","doc":"Injects the stream values into the given collectable as a side-effect.\n\nThis function is often used with `run/1` since any evaluation\nis delayed until the stream is executed. See `run/1` for an example.","ref":"Stream.html#into/3"},{"type":"function","title":"Stream.iterate/2","doc":"Emits a sequence of values, starting with `start_value`. Successive\nvalues are generated by calling `next_fun` on the previous value.","ref":"Stream.html#iterate/2"},{"type":"function","title":"Examples - Stream.iterate/2","doc":"iex> Stream.iterate(0, &(&1 + 1)) |> Enum.take(5)\n    [0, 1, 2, 3, 4]","ref":"Stream.html#iterate/2-examples"},{"type":"function","title":"Stream.map/2","doc":"Creates a stream that will apply the given function on\nenumeration.","ref":"Stream.html#map/2"},{"type":"function","title":"Examples - Stream.map/2","doc":"iex> stream = Stream.map([1, 2, 3], fn x -> x * 2 end)\n    iex> Enum.to_list(stream)\n    [2, 4, 6]","ref":"Stream.html#map/2-examples"},{"type":"function","title":"Stream.map_every/3","doc":"Creates a stream that will apply the given function on\nevery `nth` element from the enumerable.\n\nThe first element is always passed to the given function.\n\n`nth` must be a non-negative integer.","ref":"Stream.html#map_every/3"},{"type":"function","title":"Examples - Stream.map_every/3","doc":"iex> stream = Stream.map_every(1..10, 2, fn x -> x * 2 end)\n    iex> Enum.to_list(stream)\n    [2, 2, 6, 4, 10, 6, 14, 8, 18, 10]\n\n    iex> stream = Stream.map_every([1, 2, 3, 4, 5], 1, fn x -> x * 2 end)\n    iex> Enum.to_list(stream)\n    [2, 4, 6, 8, 10]\n\n    iex> stream = Stream.map_every(1..5, 0, fn x -> x * 2 end)\n    iex> Enum.to_list(stream)\n    [1, 2, 3, 4, 5]","ref":"Stream.html#map_every/3-examples"},{"type":"function","title":"Stream.reject/2","doc":"Creates a stream that will reject elements according to\nthe given function on enumeration.","ref":"Stream.html#reject/2"},{"type":"function","title":"Examples - Stream.reject/2","doc":"iex> stream = Stream.reject([1, 2, 3], fn x -> rem(x, 2) == 0 end)\n    iex> Enum.to_list(stream)\n    [1, 3]","ref":"Stream.html#reject/2-examples"},{"type":"function","title":"Stream.repeatedly/1","doc":"Returns a stream generated by calling `generator_fun` repeatedly.","ref":"Stream.html#repeatedly/1"},{"type":"function","title":"Examples - Stream.repeatedly/1","doc":"# Although not necessary, let's seed the random algorithm\n    iex> :rand.seed(:exsss, {1, 2, 3})\n    iex> Stream.repeatedly(&:rand.uniform/0) |> Enum.take(3)\n    [0.5455598952593053, 0.6039309974353404, 0.6684893034823949]","ref":"Stream.html#repeatedly/1-examples"},{"type":"function","title":"Stream.resource/3","doc":"Emits a sequence of values for the given resource.\n\nSimilar to `transform/3` but the initial accumulated value is\ncomputed lazily via `start_fun` and executes an `after_fun` at\nthe end of enumeration (both in cases of success and failure).\n\nSuccessive values are generated by calling `next_fun` with the\nprevious accumulator (the initial value being the result returned\nby `start_fun`) and it must return a tuple containing a list\nof elements to be emitted and the next accumulator. The enumeration\nfinishes if it returns `{:halt, acc}`.\n\nAs the function name suggests, this function is useful to stream values from\nresources.","ref":"Stream.html#resource/3"},{"type":"function","title":"Examples - Stream.resource/3","doc":"Stream.resource(\n      fn -> File.open!(\"sample\") end,\n      fn file ->\n        case IO.read(file, :line) do\n          data when is_binary(data) -> {[data], file}\n          _ -> {:halt, file}\n        end\n      end,\n      fn file -> File.close(file) end\n    )\n\n    iex> Stream.resource(\n    ...>  fn ->\n    ...>    {:ok, pid} = StringIO.open(\"string\")\n    ...>    pid\n    ...>  end,\n    ...>  fn pid ->\n    ...>    case IO.getn(pid, \"\", 1) do\n    ...>      :eof -> {:halt, pid}\n    ...>      char -> {[char], pid}\n    ...>    end\n    ...>  end,\n    ...>  fn pid -> StringIO.close(pid) end\n    ...> ) |> Enum.to_list()\n    [\"s\", \"t\", \"r\", \"i\", \"n\", \"g\"]","ref":"Stream.html#resource/3-examples"},{"type":"function","title":"Stream.run/1","doc":"Runs the given stream.\n\nThis is useful when a stream needs to be run, for side effects,\nand there is no interest in its return result.","ref":"Stream.html#run/1"},{"type":"function","title":"Examples - Stream.run/1","doc":"Open up a file, replace all `#` by `%` and stream to another file\nwithout loading the whole file in memory:\n\n    File.stream!(\"/path/to/file\")\n    |> Stream.map(&String.replace(&1, \"#\", \"%\"))\n    |> Stream.into(File.stream!(\"/path/to/other/file\"))\n    |> Stream.run()\n\nNo computation will be done until we call one of the `Enum` functions\nor `run/1`.","ref":"Stream.html#run/1-examples"},{"type":"function","title":"Stream.scan/2","doc":"Creates a stream that applies the given function to each\nelement, emits the result and uses the same result as the accumulator\nfor the next computation. Uses the first element in the enumerable\nas the starting value.","ref":"Stream.html#scan/2"},{"type":"function","title":"Examples - Stream.scan/2","doc":"iex> stream = Stream.scan(1..5, &(&1 + &2))\n    iex> Enum.to_list(stream)\n    [1, 3, 6, 10, 15]","ref":"Stream.html#scan/2-examples"},{"type":"function","title":"Stream.scan/3","doc":"Creates a stream that applies the given function to each\nelement, emits the result and uses the same result as the accumulator\nfor the next computation. Uses the given `acc` as the starting value.","ref":"Stream.html#scan/3"},{"type":"function","title":"Examples - Stream.scan/3","doc":"iex> stream = Stream.scan(1..5, 0, &(&1 + &2))\n    iex> Enum.to_list(stream)\n    [1, 3, 6, 10, 15]","ref":"Stream.html#scan/3-examples"},{"type":"function","title":"Stream.take/2","doc":"Lazily takes the next `count` elements from the enumerable and stops\nenumeration.\n\nIf a negative `count` is given, the last `count` values will be taken.\nFor such, the collection is fully enumerated keeping up to `2 * count`\nelements in memory. Once the end of the collection is reached,\nthe last `count` elements will be executed. Therefore, using\na negative `count` on an infinite collection will never return.","ref":"Stream.html#take/2"},{"type":"function","title":"Examples - Stream.take/2","doc":"iex> stream = Stream.take(1..100, 5)\n    iex> Enum.to_list(stream)\n    [1, 2, 3, 4, 5]\n\n    iex> stream = Stream.take(1..100, -5)\n    iex> Enum.to_list(stream)\n    [96, 97, 98, 99, 100]\n\n    iex> stream = Stream.cycle([1, 2, 3]) |> Stream.take(5)\n    iex> Enum.to_list(stream)\n    [1, 2, 3, 1, 2]","ref":"Stream.html#take/2-examples"},{"type":"function","title":"Stream.take_every/2","doc":"Creates a stream that takes every `nth` element from the enumerable.\n\nThe first element is always included, unless `nth` is 0.\n\n`nth` must be a non-negative integer.","ref":"Stream.html#take_every/2"},{"type":"function","title":"Examples - Stream.take_every/2","doc":"iex> stream = Stream.take_every(1..10, 2)\n    iex> Enum.to_list(stream)\n    [1, 3, 5, 7, 9]\n\n    iex> stream = Stream.take_every([1, 2, 3, 4, 5], 1)\n    iex> Enum.to_list(stream)\n    [1, 2, 3, 4, 5]\n\n    iex> stream = Stream.take_every(1..1000, 0)\n    iex> Enum.to_list(stream)\n    []","ref":"Stream.html#take_every/2-examples"},{"type":"function","title":"Stream.take_while/2","doc":"Lazily takes elements of the enumerable while the given\nfunction returns a truthy value.","ref":"Stream.html#take_while/2"},{"type":"function","title":"Examples - Stream.take_while/2","doc":"iex> stream = Stream.take_while(1..100, &(&1   Enum.to_list(stream)\n    [1, 2, 3, 4, 5]","ref":"Stream.html#take_while/2-examples"},{"type":"function","title":"Stream.timer/1","doc":"Creates a stream that emits a single value after `n` milliseconds.\n\nThe value emitted is `0`. This operation will block the caller by\nthe given time until the element is streamed.","ref":"Stream.html#timer/1"},{"type":"function","title":"Examples - Stream.timer/1","doc":"iex> Stream.timer(10) |> Enum.to_list()\n    [0]","ref":"Stream.html#timer/1-examples"},{"type":"function","title":"Stream.transform/3","doc":"Transforms an existing stream.\n\nIt expects an accumulator and a function that receives two arguments,\nthe stream element and the updated accumulator. It must return a tuple,\nwhere the first element is a new stream (often a list) or the atom `:halt`,\nand the second element is the accumulator to be used by the next element.\n\nNote: this function is equivalent to `Enum.flat_map_reduce/3`, except this\nfunction does not return the accumulator once the stream is processed.","ref":"Stream.html#transform/3"},{"type":"function","title":"Examples - Stream.transform/3","doc":"`Stream.transform/3` is useful as it can be used as the basis to implement\nmany of the functions defined in this module. For example, we can implement\n`Stream.take(enum, n)` as follows:\n\n    iex> enum = 1001..9999\n    iex> n = 3\n    iex> stream = Stream.transform(enum, 0, fn i, acc ->\n    ...>   if acc   end)\n    iex> Enum.to_list(stream)\n    [1001, 1002, 1003]\n\n`Stream.transform/5` further generalizes this function to allow wrapping\naround resources.","ref":"Stream.html#transform/3-examples"},{"type":"function","title":"Stream.transform/4","doc":"Similar to `Stream.transform/5`, except `last_fun` is not supplied.\n\nThis function can be seen as a combination of `Stream.resource/3` with\n`Stream.transform/3`.","ref":"Stream.html#transform/4"},{"type":"function","title":"Stream.transform/5","doc":"Transforms an existing stream with function-based start, last, and after\ncallbacks.\n\nOnce transformation starts, `start_fun` is invoked to compute the initial\naccumulator. Then, for each element in the enumerable, the `reducer` function\nis invoked with the element and the accumulator, returning new elements and a\nnew accumulator, as in `transform/3`.\n\nOnce the collection is done, `last_fun` is invoked with the accumulator to\nemit any remaining items. Then `after_fun` is invoked, to close any resource,\nbut not emitting any new items. `last_fun` is only invoked if the given\nenumerable terminates successfully (either because it is done or it halted\nitself). `after_fun` is always invoked, therefore `after_fun` must be the\none used for closing resources.","ref":"Stream.html#transform/5"},{"type":"function","title":"Stream.unfold/2","doc":"Emits a sequence of values for the given accumulator.\n\nSuccessive values are generated by calling `next_fun` with the previous\naccumulator and it must return a tuple with the current value and next\naccumulator. The enumeration finishes if it returns `nil`.","ref":"Stream.html#unfold/2"},{"type":"function","title":"Examples - Stream.unfold/2","doc":"To create a stream that counts down and stops before zero:\n\n    iex> Stream.unfold(5, fn\n    ...>   0 -> nil\n    ...>   n -> {n, n - 1}\n    ...> end) |> Enum.to_list()\n    [5, 4, 3, 2, 1]\n\nIf `next_fun` never returns `nil`, the returned stream is *infinite*:\n\n    iex> Stream.unfold(0, fn\n    ...>   n -> {n, n + 1}\n    ...> end) |> Enum.take(10)\n    [0, 1, 2, 3, 4, 5, 6, 7, 8, 9]\n\n    iex> Stream.unfold(1, fn\n    ...>   n -> {n, n * 2}\n    ...> end) |> Enum.take(10)\n    [1, 2, 4, 8, 16, 32, 64, 128, 256, 512]","ref":"Stream.html#unfold/2-examples"},{"type":"function","title":"Stream.uniq/1","doc":"Creates a stream that only emits elements if they are unique.\n\nKeep in mind that, in order to know if an element is unique\nor not, this function needs to store all unique values emitted\nby the stream. Therefore, if the stream is infinite, the number\nof elements stored will grow infinitely, never being garbage-collected.","ref":"Stream.html#uniq/1"},{"type":"function","title":"Examples - Stream.uniq/1","doc":"iex> Stream.uniq([1, 2, 3, 3, 2, 1]) |> Enum.to_list()\n    [1, 2, 3]","ref":"Stream.html#uniq/1-examples"},{"type":"function","title":"Stream.uniq_by/2","doc":"Creates a stream that only emits elements if they are unique, by removing the\nelements for which function `fun` returned duplicate elements.\n\nThe function `fun` maps every element to a term which is used to\ndetermine if two elements are duplicates.\n\nKeep in mind that, in order to know if an element is unique\nor not, this function needs to store all unique values emitted\nby the stream. Therefore, if the stream is infinite, the number\nof elements stored will grow infinitely, never being garbage-collected.","ref":"Stream.html#uniq_by/2"},{"type":"function","title":"Example - Stream.uniq_by/2","doc":"iex> Stream.uniq_by([{1, :x}, {2, :y}, {1, :z}], fn {x, _} -> x end) |> Enum.to_list()\n    [{1, :x}, {2, :y}]\n\n    iex> Stream.uniq_by([a: {:tea, 2}, b: {:tea, 2}, c: {:coffee, 1}], fn {_, y} -> y end) |> Enum.to_list()\n    [a: {:tea, 2}, c: {:coffee, 1}]","ref":"Stream.html#uniq_by/2-example"},{"type":"function","title":"Stream.with_index/2","doc":"Creates a stream where each element in the enumerable will\nbe wrapped in a tuple alongside its index.\n\nIf an `offset` is given, we will index from the given offset instead of from zero.","ref":"Stream.html#with_index/2"},{"type":"function","title":"Examples - Stream.with_index/2","doc":"iex> stream = Stream.with_index([1, 2, 3])\n    iex> Enum.to_list(stream)\n    [{1, 0}, {2, 1}, {3, 2}]\n\n    iex> stream = Stream.with_index([1, 2, 3], 3)\n    iex> Enum.to_list(stream)\n    [{1, 3}, {2, 4}, {3, 5}]","ref":"Stream.html#with_index/2-examples"},{"type":"function","title":"Stream.zip/1","doc":"Zips corresponding elements from a finite collection of enumerables\ninto one stream of tuples.\n\nThe zipping finishes as soon as any enumerable in the given collection completes.","ref":"Stream.html#zip/1"},{"type":"function","title":"Examples - Stream.zip/1","doc":"iex> concat = Stream.concat(1..3, 4..6)\n    iex> cycle = Stream.cycle([\"foo\", \"bar\", \"baz\"])\n    iex> Stream.zip([concat, [:a, :b, :c], cycle]) |> Enum.to_list()\n    [{1, :a, \"foo\"}, {2, :b, \"bar\"}, {3, :c, \"baz\"}]","ref":"Stream.html#zip/1-examples"},{"type":"function","title":"Stream.zip/2","doc":"Zips two enumerables together, lazily.\n\nThe zipping finishes as soon as either enumerable completes.","ref":"Stream.html#zip/2"},{"type":"function","title":"Examples - Stream.zip/2","doc":"iex> concat = Stream.concat(1..3, 4..6)\n    iex> cycle = Stream.cycle([:a, :b, :c])\n    iex> Stream.zip(concat, cycle) |> Enum.to_list()\n    [{1, :a}, {2, :b}, {3, :c}, {4, :a}, {5, :b}, {6, :c}]","ref":"Stream.html#zip/2-examples"},{"type":"function","title":"Stream.zip_with/2","doc":"Lazily zips corresponding elements from a finite collection of enumerables into a new\nenumerable, transforming them with the `zip_fun` function as it goes.\n\nThe first element from each of the enums in `enumerables` will be put into a list which is then passed to\nthe one-arity `zip_fun` function. Then, the second elements from each of the enums are put into a list and passed to\n`zip_fun`, and so on until any one of the enums in `enumerables` completes.\n\nReturns a new enumerable with the results of calling `zip_fun`.","ref":"Stream.html#zip_with/2"},{"type":"function","title":"Examples - Stream.zip_with/2","doc":"iex> concat = Stream.concat(1..3, 4..6)\n    iex> Stream.zip_with([concat, concat], fn [a, b] -> a + b end) |> Enum.to_list()\n    [2, 4, 6, 8, 10, 12]\n\n    iex> concat = Stream.concat(1..3, 4..6)\n    iex> Stream.zip_with([concat, concat, 1..3], fn [a, b, c] -> a + b + c end) |> Enum.to_list()\n    [3, 6, 9]","ref":"Stream.html#zip_with/2-examples"},{"type":"function","title":"Stream.zip_with/3","doc":"Lazily zips corresponding elements from two enumerables into a new one, transforming them with\nthe `zip_fun` function as it goes.\n\nThe `zip_fun` will be called with the first element from `enumerable1` and the first\nelement from `enumerable2`, then with the second element from each, and so on until\neither one of the enumerables completes.","ref":"Stream.html#zip_with/3"},{"type":"function","title":"Examples - Stream.zip_with/3","doc":"iex> concat = Stream.concat(1..3, 4..6)\n    iex> Stream.zip_with(concat, concat, fn a, b -> a + b end) |> Enum.to_list()\n    [2, 4, 6, 8, 10, 12]","ref":"Stream.html#zip_with/3-examples"},{"type":"type","title":"Stream.acc/0","doc":"","ref":"Stream.html#t:acc/0"},{"type":"type","title":"Stream.default/0","doc":"","ref":"Stream.html#t:default/0"},{"type":"type","title":"Stream.element/0","doc":"","ref":"Stream.html#t:element/0"},{"type":"type","title":"Stream.index/0","doc":"Zero-based index.","ref":"Stream.html#t:index/0"},{"type":"type","title":"Stream.timer/0","doc":"","ref":"Stream.html#t:timer/0"},{"type":"module","title":"File","doc":"This module contains functions to manipulate files.\n\nSome of those functions are low-level, allowing the user\nto interact with files or IO devices, like `open/2`,\n`copy/3` and others. This module also provides higher\nlevel functions that work with filenames and have their naming\nbased on Unix variants. For example, one can copy a file\nvia `cp/3` and remove files and directories recursively\nvia `rm_rf/1`.\n\nPaths given to functions in this module can be either relative to the\ncurrent working directory (as returned by `File.cwd/0`), or absolute\npaths. Shell conventions like `~` are not expanded automatically.\nTo use paths like `~/Downloads`, you can use `Path.expand/1` or\n`Path.expand/2` to expand your path to an absolute path.","ref":"File.html"},{"type":"module","title":"Encoding - File","doc":"In order to write and read files, one must use the functions\nin the `IO` module. By default, a file is opened in binary mode,\nwhich requires the functions `IO.binread/2` and `IO.binwrite/2`\nto interact with the file. A developer may pass `:utf8` as an\noption when opening the file, then the slower `IO.read/2` and\n`IO.write/2` functions must be used as they are responsible for\ndoing the proper conversions and providing the proper data guarantees.\n\nNote that filenames when given as charlists in Elixir are\nalways treated as UTF-8. In particular, we expect that the\nshell and the operating system are configured to use UTF-8\nencoding. Binary filenames are considered raw and passed\nto the operating system as is.","ref":"File.html#module-encoding"},{"type":"module","title":"API - File","doc":"Most of the functions in this module return `:ok` or\n`{:ok, result}` in case of success, `{:error, reason}`\notherwise. Those functions also have a variant\nthat ends with `!` which returns the result (instead of the\n`{:ok, result}` tuple) in case of success or raises an\nexception in case it fails. For example:\n\n    File.read(\"hello.txt\")\n    #=> {:ok, \"World\"}\n\n    File.read(\"invalid.txt\")\n    #=> {:error, :enoent}\n\n    File.read!(\"hello.txt\")\n    #=> \"World\"\n\n    File.read!(\"invalid.txt\")\n    #=> raises File.Error\n\nIn general, a developer should use the former in case they want\nto react if the file does not exist. The latter should be used\nwhen the developer expects their software to fail in case the\nfile cannot be read (i.e. it is literally an exception).","ref":"File.html#module-api"},{"type":"module","title":"Processes and raw files - File","doc":"Every time a file is opened, Elixir spawns a new process. Writing\nto a file is equivalent to sending messages to the process that\nwrites to the file descriptor.\n\nThis means files can be passed between nodes and message passing\nguarantees they can write to the same file in a network.\n\nHowever, you may not always want to pay the price for this abstraction.\nIn such cases, a file can be opened in `:raw` mode. The options `:read_ahead`\nand `:delayed_write` are also useful when operating on large files or\nworking with files in tight loops.\n\nCheck `:file.open/2` for more information about such options and\nother performance considerations.","ref":"File.html#module-processes-and-raw-files"},{"type":"function","title":"File.cd/1","doc":"Sets the current working directory.\n\nThe current working directory is set for the BEAM globally. This can lead to\nrace conditions if multiple processes are changing the current working\ndirectory concurrently. To run an external command in a given directory\nwithout changing the global current working directory, use the `:cd` option\nof `System.cmd/3` and `Port.open/2`.\n\nReturns `:ok` if successful, `{:error, reason}` otherwise.","ref":"File.html#cd/1"},{"type":"function","title":"File.cd!/1","doc":"The same as `cd/1`, but raises a `File.Error` exception if it fails.","ref":"File.html#cd!/1"},{"type":"function","title":"File.cd!/2","doc":"Changes the current directory to the given `path`,\nexecutes the given function and then reverts back\nto the previous path regardless of whether there is an exception.\n\nThe current working directory is temporarily set for the BEAM globally. This\ncan lead to race conditions if multiple processes are changing the current\nworking directory concurrently. To run an external command in a given\ndirectory without changing the global current working directory, use the\n`:cd` option of `System.cmd/3` and `Port.open/2`.\n\nRaises an error if retrieving or changing the current\ndirectory fails.","ref":"File.html#cd!/2"},{"type":"function","title":"File.chgrp/2","doc":"Changes the group given by the group ID `gid`\nfor a given `file`. Returns `:ok` on success, or\n`{:error, reason}` on failure.","ref":"File.html#chgrp/2"},{"type":"function","title":"File.chgrp!/2","doc":"Same as `chgrp/2`, but raises a `File.Error` exception in case of failure.\nOtherwise `:ok`.","ref":"File.html#chgrp!/2"},{"type":"function","title":"File.chmod/2","doc":"Changes the `mode` for a given `file`.\n\nReturns `:ok` on success, or `{:error, reason}` on failure.","ref":"File.html#chmod/2"},{"type":"function","title":"Permissions - File.chmod/2","doc":"File permissions are specified by adding together the following octal modes:\n\n  * `0o400` - read permission: owner\n  * `0o200` - write permission: owner\n  * `0o100` - execute permission: owner\n\n  * `0o040` - read permission: group\n  * `0o020` - write permission: group\n  * `0o010` - execute permission: group\n\n  * `0o004` - read permission: other\n  * `0o002` - write permission: other\n  * `0o001` - execute permission: other\n\nFor example, setting the mode `0o755` gives it\nwrite, read and execute permission to the owner\nand both read and execute permission to group\nand others.","ref":"File.html#chmod/2-permissions"},{"type":"function","title":"File.chmod!/2","doc":"Same as `chmod/2`, but raises a `File.Error` exception in case of failure.\nOtherwise `:ok`.","ref":"File.html#chmod!/2"},{"type":"function","title":"File.chown/2","doc":"Changes the owner given by the user ID `uid`\nfor a given `file`. Returns `:ok` on success,\nor `{:error, reason}` on failure.","ref":"File.html#chown/2"},{"type":"function","title":"File.chown!/2","doc":"Same as `chown/2`, but raises a `File.Error` exception in case of failure.\nOtherwise `:ok`.","ref":"File.html#chown!/2"},{"type":"function","title":"File.close/1","doc":"Closes the file referenced by `io_device`. It mostly returns `:ok`, except\nfor some severe errors such as out of memory.\n\nNote that if the option `:delayed_write` was used when opening the file,\n`close/1` might return an old write error and not even try to close the file.\nSee `open/2` for more information.","ref":"File.html#close/1"},{"type":"function","title":"File.copy/3","doc":"Copies the contents of `source` to `destination`.\n\nBoth parameters can be a filename or an IO device opened\nwith `open/2`. `bytes_count` specifies the number of\nbytes to copy, the default being `:infinity`.\n\nIf file `destination` already exists, it is overwritten\nby the contents in `source`.\n\nReturns `{:ok, bytes_copied}` if successful,\n`{:error, reason}` otherwise.\n\nCompared to the `cp/3`, this function is more low-level,\nallowing a copy from device to device limited by a number of\nbytes. On the other hand, `cp/3` performs more extensive\nchecks on both source and destination and it also preserves\nthe file mode after copy.\n\nTypical error reasons are the same as in `open/2`,\n`read/1` and `write/3`.","ref":"File.html#copy/3"},{"type":"function","title":"File.copy!/3","doc":"The same as `copy/3` but raises a `File.CopyError` exception if it fails.\nReturns the `bytes_copied` otherwise.","ref":"File.html#copy!/3"},{"type":"function","title":"File.cp/3","doc":"Copies the contents of `source_file` to `destination_file` preserving its modes.\n\n`source_file` must be a file or a symbolic link to one. `destination_file` must\nbe a path to a non-existent file. If either is a directory, `{:error, :eisdir}`\nwill be returned.\n\nThe function returns `:ok` in case of success. Otherwise, it returns\n`{:error, reason}`.\n\nIf you want to copy contents from an IO device to another device\nor do a straight copy from a source to a destination without\npreserving modes, check `copy/3` instead.\n\nNote: The command `cp` in Unix-like systems behaves differently depending on\nwhether the destination is an existing directory or not. We have chosen to\nexplicitly disallow copying to a destination which is a directory,\nand an error will be returned if tried.","ref":"File.html#cp/3"},{"type":"function","title":"Options - File.cp/3","doc":"* `:on_conflict` - (since v1.14.0) Invoked when a file already exists in the destination.\n    The function receives arguments for `source_file` and `destination_file`. It should\n    return `true` if the existing file should be overwritten, `false` if otherwise.\n    The default callback returns `true`. On earlier versions, this callback could be\n    given as third argument, but such behaviour is now deprecated.","ref":"File.html#cp/3-options"},{"type":"function","title":"File.cp!/3","doc":"The same as `cp/3`, but raises a `File.CopyError` exception if it fails.\nReturns `:ok` otherwise.","ref":"File.html#cp!/3"},{"type":"function","title":"File.cp_r/3","doc":"Copies the contents in `source` to `destination` recursively, maintaining the\nsource directory structure and modes.\n\nIf `source` is a file or a symbolic link to it, `destination` must be a path\nto an existent file, a symbolic link to one, or a path to a non-existent file.\n\nIf `source` is a directory, or a symbolic link to it, then `destination` must\nbe an existent `directory` or a symbolic link to one, or a path to a non-existent directory.\n\nIf the source is a file, it copies `source` to `destination`. If the `source`\nis a directory, it copies the contents inside source into the `destination` directory.\n\nIf a file already exists in the destination, it invokes the optional `on_conflict`\ncallback given as an option. See \"Options\" for more information.\n\nThis function may fail while copying files, in such cases, it will leave the\ndestination directory in a dirty state, where file which have already been\ncopied won't be removed.\n\nThe function returns `{:ok, files_and_directories}` in case of\nsuccess, `files_and_directories` lists all files and directories copied in no\nspecific order. It returns `{:error, reason, file}` otherwise.\n\nNote: The command `cp` in Unix-like systems behaves differently depending on\nwhether `destination` is an existing directory or not. We have chosen to\nexplicitly disallow this behaviour. If `source` is a `file` and `destination`\nis a directory, `{:error, :eisdir}` will be returned.","ref":"File.html#cp_r/3"},{"type":"function","title":"Options - File.cp_r/3","doc":"* `:on_conflict` - (since v1.14.0) Invoked when a file already exists in the destination.\n    The function receives arguments for `source` and `destination`. It should return\n    `true` if the existing file should be overwritten, `false` if otherwise. The default\n    callback returns `true`. On earlier versions, this callback could be given as third\n    argument, but such behaviour is now deprecated.\n\n  * `:dereference_symlinks` - (since v1.14.0) By default, this function will copy symlinks\n    by creating symlinks that point to the same location. This option forces symlinks to be\n    dereferenced and have their contents copied instead when set to `true`. If the dereferenced\n    files do not exist, than the operation fails. The default is `false`.","ref":"File.html#cp_r/3-options"},{"type":"function","title":"Examples - File.cp_r/3","doc":"# Copies file \"a.txt\" to \"b.txt\"\n    File.cp_r(\"a.txt\", \"b.txt\")\n\n    # Copies all files in \"samples\" to \"tmp\"\n    File.cp_r(\"samples\", \"tmp\")\n\n    # Same as before, but asks the user how to proceed in case of conflicts\n    File.cp_r(\"samples\", \"tmp\", on_conflict: fn source, destination ->\n      IO.gets(\"Overwriting #{destination} by #{source}. Type y to confirm. \") == \"y\\n\"\n    end)","ref":"File.html#cp_r/3-examples"},{"type":"function","title":"File.cp_r!/3","doc":"The same as `cp_r/3`, but raises a `File.CopyError` exception if it fails.\nReturns the list of copied files otherwise.","ref":"File.html#cp_r!/3"},{"type":"function","title":"File.cwd/0","doc":"Gets the current working directory.\n\nIn rare circumstances, this function can fail on Unix-like systems. It may happen\nif read permissions do not exist for the parent directories of the\ncurrent directory. For this reason, returns `{:ok, cwd}` in case\nof success, `{:error, reason}` otherwise.","ref":"File.html#cwd/0"},{"type":"function","title":"File.cwd!/0","doc":"The same as `cwd/0`, but raises a `File.Error` exception if it fails.","ref":"File.html#cwd!/0"},{"type":"function","title":"File.dir?/2","doc":"Returns `true` if the given path is a directory.\n\nThis function follows symbolic links, so if a symbolic link points to a\ndirectory, `true` is returned.","ref":"File.html#dir?/2"},{"type":"function","title":"Options - File.dir?/2","doc":"The supported options are:\n\n  * `:raw` - a single atom to bypass the file server and only check\n    for the file locally","ref":"File.html#dir?/2-options"},{"type":"function","title":"Examples - File.dir?/2","doc":"File.dir?(\"./test\")\n    #=> true\n\n    File.dir?(\"test\")\n    #=> true\n\n    File.dir?(\"/usr/bin\")\n    #=> true\n\n    File.dir?(\"~/Downloads\")\n    #=> false\n\n    \"~/Downloads\" |> Path.expand() |> File.dir?()\n    #=> true","ref":"File.html#dir?/2-examples"},{"type":"function","title":"File.exists?/2","doc":"Returns `true` if the given path exists.\n\nIt can be a regular file, directory, socket, symbolic link, named pipe, or device file.\nReturns `false` for symbolic links pointing to non-existing targets.","ref":"File.html#exists?/2"},{"type":"function","title":"Options - File.exists?/2","doc":"The supported options are:\n\n  * `:raw` - a single atom to bypass the file server and only check\n    for the file locally","ref":"File.html#exists?/2-options"},{"type":"function","title":"Examples - File.exists?/2","doc":"File.exists?(\"test/\")\n    #=> true\n\n    File.exists?(\"missing.txt\")\n    #=> false\n\n    File.exists?(\"/dev/null\")\n    #=> true","ref":"File.html#exists?/2-examples"},{"type":"function","title":"File.ln/2","doc":"Creates a hard link `new` to the file `existing`.\n\nReturns `:ok` if successful, `{:error, reason}` otherwise.\nIf the operating system does not support hard links, returns\n`{:error, :enotsup}`.","ref":"File.html#ln/2"},{"type":"function","title":"File.ln!/2","doc":"Same as `ln/2` but raises a `File.LinkError` exception if it fails.\nReturns `:ok` otherwise.","ref":"File.html#ln!/2"},{"type":"function","title":"File.ln_s/2","doc":"Creates a symbolic link `new` to the file or directory `existing`.\n\nReturns `:ok` if successful, `{:error, reason}` otherwise.\nIf the operating system does not support symlinks, returns\n`{:error, :enotsup}`.","ref":"File.html#ln_s/2"},{"type":"function","title":"File.ln_s!/2","doc":"Same as `ln_s/2` but raises a `File.LinkError` exception if it fails.\nReturns `:ok` otherwise.","ref":"File.html#ln_s!/2"},{"type":"function","title":"File.ls/1","doc":"Returns the list of files in the given directory.\n\nReturns `{:ok, files}` in case of success,\n`{:error, reason}` otherwise.","ref":"File.html#ls/1"},{"type":"function","title":"File.ls!/1","doc":"The same as `ls/1` but raises a `File.Error` exception in case of an error.","ref":"File.html#ls!/1"},{"type":"function","title":"File.lstat/2","doc":"Returns information about the `path`. If the file is a symlink, sets\nthe `type` to `:symlink` and returns a `File.Stat` struct for the link. For any\nother file, returns exactly the same values as `stat/2`.\n\nFor more details, see `:file.read_link_info/2`.","ref":"File.html#lstat/2"},{"type":"function","title":"Options - File.lstat/2","doc":"The accepted options are:\n\n  * `:time` - configures how the file timestamps are returned\n\nThe values for `:time` can be:\n\n  * `:universal` - returns a `{date, time}` tuple in UTC (default)\n  * `:local` - returns a `{date, time}` tuple using the machine time\n  * `:posix` - returns the time as integer seconds since epoch\n\nNote: Since file times are stored in POSIX time format on most operating systems,\nit is faster to retrieve file information with the `time: :posix` option.","ref":"File.html#lstat/2-options"},{"type":"function","title":"File.lstat!/2","doc":"Same as `lstat/2` but returns the `File.Stat` struct directly,\nor raises a `File.Error` exception if an error is returned.","ref":"File.html#lstat!/2"},{"type":"function","title":"File.mkdir/1","doc":"Tries to create the directory `path`.\n\nMissing parent directories are not created.\nReturns `:ok` if successful, or `{:error, reason}` if an error occurs.\n\nTypical error reasons are:\n\n  * `:eacces`  - missing search or write permissions for the parent\n    directories of `path`\n  * `:eexist`  - there is already a file or directory named `path`\n  * `:enoent`  - a component of `path` does not exist\n  * `:enospc`  - there is no space left on the device\n  * `:enotdir` - a component of `path` is not a directory;\n    on some platforms, `:enoent` is returned instead","ref":"File.html#mkdir/1"},{"type":"function","title":"File.mkdir!/1","doc":"Same as `mkdir/1`, but raises a `File.Error` exception in case of failure.\nOtherwise `:ok`.","ref":"File.html#mkdir!/1"},{"type":"function","title":"File.mkdir_p/1","doc":"Tries to create the directory `path`.\n\nMissing parent directories are created. Returns `:ok` if successful, or\n`{:error, reason}` if an error occurs.\n\nTypical error reasons are:\n\n  * `:eacces`  - missing search or write permissions for the parent\n    directories of `path`\n  * `:enospc`  - there is no space left on the device\n  * `:enotdir` - a component of `path` is not a directory","ref":"File.html#mkdir_p/1"},{"type":"function","title":"File.mkdir_p!/1","doc":"Same as `mkdir_p/1`, but raises a `File.Error` exception in case of failure.\nOtherwise `:ok`.","ref":"File.html#mkdir_p!/1"},{"type":"function","title":"File.open/2","doc":"Opens the given `path`.\n\nIn order to write and read files, one must use the functions\nin the `IO` module. By default, a file is opened in `:binary` mode,\nwhich requires the functions `IO.binread/2` and `IO.binwrite/2`\nto interact with the file. A developer may pass `:utf8` as an\noption when opening the file and then all other functions from\n`IO` are available, since they work directly with Unicode data.\n\n`modes_or_function` can either be a list of modes or a function. If it's a\nlist, it's considered to be a list of modes (that are documented below). If\nit's a function, then it's equivalent to calling `open(path, [],\nmodes_or_function)`. See the documentation for `open/3` for more information\non this function.\n\nThe allowed modes:\n\n  * `:binary` - opens the file in binary mode, disabling special handling of Unicode sequences\n    (default mode).\n\n  * `:read` - the file, which must exist, is opened for reading.\n\n  * `:write` - the file is opened for writing. It is created if it does not\n    exist.\n\n    If the file does exist, and if write is not combined with read, the file\n    will be truncated.\n\n  * `:append` - the file will be opened for writing, and it will be created\n    if it does not exist. Every write operation to a file opened with append\n    will take place at the end of the file.\n\n  * `:exclusive` - the file, when opened for writing, is created if it does\n    not exist. If the file exists, open will return `{:error, :eexist}`.\n\n  * `:charlist` - when this term is given, read operations on the file will\n    return charlists rather than binaries.\n\n  * `:compressed` - makes it possible to read or write gzip compressed files.\n\n    The compressed option must be combined with either read or write, but not\n    both. Note that the file size obtained with `stat/1` will most probably\n    not match the number of bytes that can be read from a compressed file.\n\n  * `:utf8` - this option denotes how data is actually stored in the disk\n    file and makes the file perform automatic translation of characters to\n    and from UTF-8.\n\n    If data is sent to a file in a format that cannot be converted to the\n    UTF-8 or if data is read by a function that returns data in a format that\n    cannot cope with the character range of the data, an error occurs and the\n    file will be closed.\n\n  * `:delayed_write`, `:raw`, `:ram`, `:read_ahead`, `:sync`, `{:encoding, ...}`,\n    `{:read_ahead, pos_integer}`, `{:delayed_write, non_neg_integer, non_neg_integer}` -\n    for more information about these options see `:file.open/2`.\n\nThis function returns:\n\n  * `{:ok, io_device}` - the file has been opened in the requested mode.\n\n    `io_device` is actually the PID of the process which handles the file.\n    This process monitors the process that originally opened the file (the\n    owner process). If the owner process terminates, the file is closed and\n    the process itself terminates too. If any process to which the `io_device`\n    is linked terminates, the file will be closed and the process itself will\n    be terminated.\n\n    An `io_device` returned from this call can be used as an argument to the\n    `IO` module functions.\n\n  * `{:error, reason}` - the file could not be opened.","ref":"File.html#open/2"},{"type":"function","title":"Examples - File.open/2","doc":"{:ok, file} = File.open(\"foo.tar.gz\", [:read, :compressed])\n    IO.read(file, :line)\n    File.close(file)","ref":"File.html#open/2-examples"},{"type":"function","title":"File.open/3","doc":"Similar to `open/2` but expects a function as its last argument.\n\nThe file is opened, given to the function as an argument and\nautomatically closed after the function returns, regardless\nif there was an error when executing the function.\n\nReturns `{:ok, function_result}` in case of success,\n`{:error, reason}` otherwise.\n\nThis function expects the file to be closed with success,\nwhich is usually the case unless the `:delayed_write` option\nis given. For this reason, we do not recommend passing\n`:delayed_write` to this function.","ref":"File.html#open/3"},{"type":"function","title":"Examples - File.open/3","doc":"File.open(\"file.txt\", [:read, :write], fn file ->\n      IO.read(file, :line)\n    end)\n\nSee `open/2` for the list of available `modes`.","ref":"File.html#open/3-examples"},{"type":"function","title":"File.open!/2","doc":"Similar to `open/2` but raises a `File.Error` exception if the file\ncould not be opened. Returns the IO device otherwise.\n\nSee `open/2` for the list of available modes.","ref":"File.html#open!/2"},{"type":"function","title":"File.open!/3","doc":"Similar to `open/3` but raises a `File.Error` exception if the file\ncould not be opened.\n\nIf it succeeds opening the file, it returns the `function` result on the IO device.\n\nSee `open/2` for the list of available `modes`.","ref":"File.html#open!/3"},{"type":"function","title":"File.read/1","doc":"Returns `{:ok, binary}`, where `binary` is a binary data object that contains the contents\nof `path`, or `{:error, reason}` if an error occurs.\n\nTypical error reasons:\n\n  * `:enoent`  - the file does not exist\n  * `:eacces`  - missing permission for reading the file,\n    or for searching one of the parent directories\n  * `:eisdir`  - the named file is a directory\n  * `:enotdir` - a component of the file name is not a directory;\n    on some platforms, `:enoent` is returned instead\n  * `:enomem`  - there is not enough memory for the contents of the file\n\nYou can use `:file.format_error/1` to get a descriptive string of the error.","ref":"File.html#read/1"},{"type":"function","title":"File.read!/1","doc":"Returns a binary with the contents of the given filename,\nor raises a `File.Error` exception if an error occurs.","ref":"File.html#read!/1"},{"type":"function","title":"File.read_link/1","doc":"Reads the symbolic link at `path`.\n\nIf `path` exists and is a symlink, returns `{:ok, target}`, otherwise returns\n`{:error, reason}`.\n\nFor more details, see `:file.read_link/1`.\n\nTypical error reasons are:\n\n  * `:einval` - path is not a symbolic link\n  * `:enoent` - path does not exist\n  * `:enotsup` - symbolic links are not supported on the current platform","ref":"File.html#read_link/1"},{"type":"function","title":"File.read_link!/1","doc":"Same as `read_link/1` but returns the target directly,\nor raises a `File.Error` exception if an error is returned.","ref":"File.html#read_link!/1"},{"type":"function","title":"File.regular?/2","doc":"Returns `true` if the path is a regular file.\n\nThis function follows symbolic links, so if a symbolic link points to a\nregular file, `true` is returned.","ref":"File.html#regular?/2"},{"type":"function","title":"Options - File.regular?/2","doc":"The supported options are:\n\n  * `:raw` - a single atom to bypass the file server and only check\n    for the file locally","ref":"File.html#regular?/2-options"},{"type":"function","title":"Examples - File.regular?/2","doc":"File.regular?(__ENV__.file)\n    #=> true","ref":"File.html#regular?/2-examples"},{"type":"function","title":"File.rename/2","doc":"Renames the `source` file to `destination` file.  It can be used to move files\n(and directories) between directories.  If moving a file, you must fully\nspecify the `destination` filename, it is not sufficient to simply specify\nits directory.\n\nReturns `:ok` in case of success, `{:error, reason}` otherwise.\n\nNote: The command `mv` in Unix-like systems behaves differently depending on\nwhether `source` is a file and the `destination` is an existing directory.\nWe have chosen to explicitly disallow this behaviour.","ref":"File.html#rename/2"},{"type":"function","title":"Examples - File.rename/2","doc":"# Rename file \"a.txt\" to \"b.txt\"\n    File.rename(\"a.txt\", \"b.txt\")\n\n    # Rename directory \"samples\" to \"tmp\"\n    File.rename(\"samples\", \"tmp\")","ref":"File.html#rename/2-examples"},{"type":"function","title":"File.rename!/2","doc":"The same as `rename/2` but raises a `File.RenameError` exception if it fails.\nReturns `:ok` otherwise.","ref":"File.html#rename!/2"},{"type":"function","title":"File.rm/1","doc":"Tries to delete the file `path`.\n\nReturns `:ok` if successful, or `{:error, reason}` if an error occurs.\n\nNote the file is deleted even if in read-only mode.\n\nTypical error reasons are:\n\n  * `:enoent`  - the file does not exist\n  * `:eacces`  - missing permission for the file or one of its parents\n  * `:eperm`   - the file is a directory and user is not super-user\n  * `:enotdir` - a component of the file name is not a directory;\n    on some platforms, `:enoent` is returned instead\n  * `:einval`  - filename had an improper type, such as tuple","ref":"File.html#rm/1"},{"type":"function","title":"Examples - File.rm/1","doc":"File.rm(\"file.txt\")\n    #=> :ok\n\n    File.rm(\"tmp_dir/\")\n    #=> {:error, :eperm}","ref":"File.html#rm/1-examples"},{"type":"function","title":"File.rm!/1","doc":"Same as `rm/1`, but raises a `File.Error` exception in case of failure.\nOtherwise `:ok`.","ref":"File.html#rm!/1"},{"type":"function","title":"File.rm_rf/1","doc":"Removes files and directories recursively at the given `path`.\nSymlinks are not followed but simply removed, non-existing\nfiles are simply ignored (i.e. doesn't make this function fail).\n\nReturns `{:ok, files_and_directories}` with all files and\ndirectories removed in no specific order, `{:error, reason, file}`\notherwise.","ref":"File.html#rm_rf/1"},{"type":"function","title":"Examples - File.rm_rf/1","doc":"File.rm_rf(\"samples\")\n    #=> {:ok, [\"samples\", \"samples/1.txt\"]}\n\n    File.rm_rf(\"unknown\")\n    #=> {:ok, []}","ref":"File.html#rm_rf/1-examples"},{"type":"function","title":"File.rm_rf!/1","doc":"Same as `rm_rf/1` but raises a `File.Error` exception in case of failures,\notherwise the list of files or directories removed.","ref":"File.html#rm_rf!/1"},{"type":"function","title":"File.rmdir/1","doc":"Tries to delete the dir at `path`.\n\nReturns `:ok` if successful, or `{:error, reason}` if an error occurs.\nIt returns `{:error, :eexist}` if the directory is not empty.","ref":"File.html#rmdir/1"},{"type":"function","title":"Examples - File.rmdir/1","doc":"File.rmdir(\"tmp_dir\")\n    #=> :ok\n\n    File.rmdir(\"non_empty_dir\")\n    #=> {:error, :eexist}\n\n    File.rmdir(\"file.txt\")\n    #=> {:error, :enotdir}","ref":"File.html#rmdir/1-examples"},{"type":"function","title":"File.rmdir!/1","doc":"Same as `rmdir/1`, but raises a `File.Error` exception in case of failure.\nOtherwise `:ok`.","ref":"File.html#rmdir!/1"},{"type":"function","title":"File.stat/2","doc":"Returns information about the `path`. If it exists, it\nreturns a `{:ok, info}` tuple, where info is a\n`File.Stat` struct. Returns `{:error, reason}` with\nthe same reasons as `read/1` if a failure occurs.","ref":"File.html#stat/2"},{"type":"function","title":"Options - File.stat/2","doc":"The accepted options are:\n\n  * `:time` - configures how the file timestamps are returned\n\nThe values for `:time` can be:\n\n  * `:universal` - returns a `{date, time}` tuple in UTC (default)\n  * `:local` - returns a `{date, time}` tuple using the same time zone as the\n    machine\n  * `:posix` - returns the time as integer seconds since epoch\n\nNote: Since file times are stored in POSIX time format on most operating systems,\nit is faster to retrieve file information with the `time: :posix` option.","ref":"File.html#stat/2-options"},{"type":"function","title":"File.stat!/2","doc":"Same as `stat/2` but returns the `File.Stat` directly,\nor raises a `File.Error` exception if an error is returned.","ref":"File.html#stat!/2"},{"type":"function","title":"File.stream!/3","doc":"Returns a `File.Stream` for the given `path` with the given `modes`.\n\nThe stream implements both `Enumerable` and `Collectable` protocols,\nwhich means it can be used both for read and write.\n\nThe `line_or_bytes` argument configures how the file is read when\nstreaming, by `:line` (default) or by a given number of bytes. When\nusing the `:line` option, CRLF line breaks (`\"\\r\\n\"`) are normalized\nto LF (`\"\\n\"`).\n\nSimilar to other file operations, a stream can be created in one node\nand forwarded to another node. Once the stream is opened in another node,\na request will be sent to the creator node to spawn a process for file\nstreaming.\n\nOperating the stream can fail on open for the same reasons as\n`File.open!/2`. Note that the file is automatically opened each time streaming\nbegins. There is no need to pass `:read` and `:write` modes, as those are\nautomatically set by Elixir.","ref":"File.html#stream!/3"},{"type":"function","title":"Raw files - File.stream!/3","doc":"Since Elixir controls when the streamed file is opened, the underlying\ndevice cannot be shared and as such it is convenient to open the file\nin raw mode for performance reasons. Therefore, Elixir **will** open\nstreams in `:raw` mode with the `:read_ahead` option if the stream is\nopen in the same node as it is created and no encoding has been specified.\nThis means any data streamed into the file must be converted to `t:iodata/0`\ntype. If you pass, for example, `[encoding: :utf8]` or\n`[encoding: {:utf16, :little}]` in the modes parameter, the underlying stream\nwill use `IO.write/2` and the `String.Chars` protocol to convert the data.\nSee `IO.binwrite/2` and `IO.write/2` .\n\nOne may also consider passing the `:delayed_write` option if the stream\nis meant to be written to under a tight loop.","ref":"File.html#stream!/3-raw-files"},{"type":"function","title":"Byte order marks - File.stream!/3","doc":"If you pass `:trim_bom` in the modes parameter, the stream will\ntrim UTF-8, UTF-16 and UTF-32 byte order marks when reading from file.\n\nNote that this function does not try to discover the file encoding\nbased on BOM.","ref":"File.html#stream!/3-byte-order-marks"},{"type":"function","title":"Examples - File.stream!/3","doc":"# Read in 2048 byte chunks rather than lines\n    File.stream!(\"./test/test.data\", [], 2048)\n    #=> %File.Stream{line_or_bytes: 2048, modes: [:raw, :read_ahead, :binary],\n    #=>   path: \"./test/test.data\", raw: true}\n\nSee `Stream.run/1` for an example of streaming into a file.","ref":"File.html#stream!/3-examples"},{"type":"function","title":"File.touch/2","doc":"Updates modification time (mtime) and access time (atime) of\nthe given file.\n\nThe file is created if it doesn't exist. Requires datetime in UTC\n(as returned by `:erlang.universaltime()`) or an integer\nrepresenting the POSIX timestamp (as returned by `System.os_time(:second)`).\n\nIn Unix-like systems, changing the modification time may require\nyou to be either `root` or the owner of the file. Having write\naccess may not be enough. In those cases, touching the file the\nfirst time (to create it) will succeed, but touching an existing\nfile with fail with `{:error, :eperm}`.","ref":"File.html#touch/2"},{"type":"function","title":"Examples - File.touch/2","doc":"File.touch(\"/tmp/a.txt\", {{2018, 1, 30}, {13, 59, 59}})\n    #=> :ok\n    File.touch(\"/fakedir/b.txt\", {{2018, 1, 30}, {13, 59, 59}})\n    {:error, :enoent}\n\n    File.touch(\"/tmp/a.txt\", 1544519753)\n    #=> :ok","ref":"File.html#touch/2-examples"},{"type":"function","title":"File.touch!/2","doc":"Same as `touch/2` but raises a `File.Error` exception if it fails.\nReturns `:ok` otherwise.\n\nThe file is created if it doesn't exist. Requires datetime in UTC\n(as returned by `:erlang.universaltime()`) or an integer\nrepresenting the POSIX timestamp (as returned by `System.os_time(:second)`).","ref":"File.html#touch!/2"},{"type":"function","title":"Examples - File.touch!/2","doc":"File.touch!(\"/tmp/a.txt\", {{2018, 1, 30}, {13, 59, 59}})\n    #=> :ok\n    File.touch!(\"/fakedir/b.txt\", {{2018, 1, 30}, {13, 59, 59}})\n    ** (File.Error) could not touch \"/fakedir/b.txt\": no such file or directory\n\n    File.touch!(\"/tmp/a.txt\", 1544519753)","ref":"File.html#touch!/2-examples"},{"type":"function","title":"File.write/3","doc":"Writes `content` to the file `path`.\n\nThe file is created if it does not exist. If it exists, the previous\ncontents are overwritten. Returns `:ok` if successful, or `{:error, reason}`\nif an error occurs.\n\n`content` must be `iodata` (a list of bytes or a binary). Setting the\nencoding for this function has no effect.\n\n**Warning:** Every time this function is invoked, a file descriptor is opened\nand a new process is spawned to write to the file. For this reason, if you are\ndoing multiple writes in a loop, opening the file via `File.open/2` and using\nthe functions in `IO` to write to the file will yield much better performance\nthan calling this function multiple times.\n\nTypical error reasons are:\n\n  * `:enoent`  - a component of the file name does not exist\n  * `:enotdir` - a component of the file name is not a directory;\n    on some platforms, `:enoent` is returned instead\n  * `:enospc`  - there is no space left on the device\n  * `:eacces`  - missing permission for writing the file or searching one of\n    the parent directories\n  * `:eisdir`  - the named file is a directory\n\nCheck `File.open/2` for other available options.","ref":"File.html#write/3"},{"type":"function","title":"File.write!/3","doc":"Same as `write/3` but raises a `File.Error` exception if it fails.\nReturns `:ok` otherwise.","ref":"File.html#write!/3"},{"type":"function","title":"File.write_stat/3","doc":"Writes the given `File.Stat` back to the file system at the given\npath. Returns `:ok` or `{:error, reason}`.","ref":"File.html#write_stat/3"},{"type":"function","title":"File.write_stat!/3","doc":"Same as `write_stat/3` but raises a `File.Error` exception if it fails.\nReturns `:ok` otherwise.","ref":"File.html#write_stat!/3"},{"type":"type","title":"File.encoding_mode/0","doc":"","ref":"File.html#t:encoding_mode/0"},{"type":"type","title":"File.erlang_time/0","doc":"","ref":"File.html#t:erlang_time/0"},{"type":"type","title":"File.io_device/0","doc":"","ref":"File.html#t:io_device/0"},{"type":"type","title":"File.mode/0","doc":"","ref":"File.html#t:mode/0"},{"type":"type","title":"File.on_conflict_callback/0","doc":"","ref":"File.html#t:on_conflict_callback/0"},{"type":"type","title":"File.posix/0","doc":"","ref":"File.html#t:posix/0"},{"type":"type","title":"File.posix_time/0","doc":"","ref":"File.html#t:posix_time/0"},{"type":"type","title":"File.stat_options/0","doc":"","ref":"File.html#t:stat_options/0"},{"type":"type","title":"File.stream_mode/0","doc":"","ref":"File.html#t:stream_mode/0"},{"type":"module","title":"File.Stat","doc":"A struct that holds file information.\n\nIn Erlang, this struct is represented by a `:file_info` record.\nTherefore this module also provides functions for converting\nbetween the Erlang record and the Elixir struct.\n\nIts fields are:\n\n  * `size` - size of file in bytes.\n\n  * `type` - `:device | :directory | :regular | :other | :symlink`; the type of the\n    file.\n\n  * `access` - `:read | :write | :read_write | :none`; the current system\n    access to the file.\n\n  * `atime` - the last time the file was read.\n\n  * `mtime` - the last time the file was written.\n\n  * `ctime` - the interpretation of this time field depends on the operating\n    system. On Unix-like operating systems, it is the last time the file or the inode was changed.\n    In Windows, it is the time of creation.\n\n  * `mode` - the file permissions.\n\n  * `links` - the number of links to this file. This is always 1 for file\n    systems which have no concept of links.\n\n  * `major_device` - identifies the file system where the file is located.\n    In Windows, the number indicates a drive as follows: 0 means A:, 1 means\n    B:, and so on.\n\n  * `minor_device` - only valid for character devices on Unix-like systems. In all other\n    cases, this field is zero.\n\n  * `inode` - gives the inode number. On non-Unix-like file systems, this field\n    will be zero.\n\n  * `uid` - indicates the owner of the file. Will be zero for non-Unix-like file\n    systems.\n\n  * `gid` - indicates the group that owns the file. Will be zero for\n    non-Unix-like file systems.\n\nThe time type returned in `atime`, `mtime`, and `ctime` is dependent on the\ntime type set in options. `{:time, type}` where type can be `:local`,\n`:universal`, or `:posix`. Default is `:universal`.","ref":"File.Stat.html"},{"type":"function","title":"File.Stat.from_record/1","doc":"Converts a `:file_info` record into a `File.Stat`.","ref":"File.Stat.html#from_record/1"},{"type":"function","title":"File.Stat.to_record/1","doc":"Converts a `File.Stat` struct to a `:file_info` record.","ref":"File.Stat.html#to_record/1"},{"type":"type","title":"File.Stat.t/0","doc":"","ref":"File.Stat.html#t:t/0"},{"type":"module","title":"File.Stream","doc":"Defines a `File.Stream` struct returned by `File.stream!/3`.\n\nThe following fields are public:\n\n  * `path`          - the file path\n  * `modes`         - the file modes\n  * `raw`           - a boolean indicating if bin functions should be used\n  * `line_or_bytes` - if reading should read lines or a given number of bytes\n  * `node`          - the node the file belongs to","ref":"File.Stream.html"},{"type":"type","title":"File.Stream.t/0","doc":"","ref":"File.Stream.html#t:t/0"},{"type":"module","title":"IO","doc":"Functions handling input/output (IO).\n\nMany functions in this module expect an IO device as an argument.\nAn IO device must be a PID or an atom representing a process.\nFor convenience, Elixir provides `:stdio` and `:stderr` as\nshortcuts to Erlang's `:standard_io` and `:standard_error`.\n\nThe majority of the functions expect chardata. In case another type is given,\nfunctions will convert those types to string via the `String.Chars` protocol\n(as shown in typespecs). For more information on chardata, see the\n\"IO data\" section below.","ref":"IO.html"},{"type":"module","title":"IO devices - IO","doc":"An IO device may be an atom or a PID. In case it is an atom,\nthe atom must be the name of a registered process. In addition,\nElixir provides two shortcuts:\n\n  * `:stdio` - a shortcut for `:standard_io`, which maps to\n    the current `Process.group_leader/0` in Erlang\n\n  * `:stderr` - a shortcut for the named process `:standard_error`\n    provided in Erlang\n\nIO devices maintain their position, which means subsequent calls to any\nreading or writing functions will start from the place where the device\nwas last accessed. The position of files can be changed using the\n`:file.position/2` function.","ref":"IO.html#module-io-devices"},{"type":"module","title":"IO data - IO","doc":"IO data is a data type that can be used as a more efficient alternative to binaries\nin certain situations.\n\nA term of type **IO data** is a binary or a list containing bytes (integers within the `0..255` range)\nor nested IO data. The type is recursive. Let's see an example of one of\nthe possible IO data representing the binary `\"hello\"`:\n\n    [?h, \"el\", [\"l\", [?o]]]\n\nThe built-in `t:iodata/0` type is defined in terms of `t:iolist/0`. An IO list is\nthe same as IO data but it doesn't allow for a binary at the top level (but binaries\nare still allowed in the list itself).\n\n#","ref":"IO.html#module-io-data"},{"type":"module","title":"Use cases for IO data - IO","doc":"IO data exists because often you need to do many append operations\non smaller chunks of binaries in order to create a bigger binary. However, in\nErlang and Elixir concatenating binaries will copy the concatenated binaries\ninto a new binary.\n\n    def email(username, domain) do\n      username   \"@\"   domain\n    end\n\nIn this function, creating the email address will copy the `username` and `domain`\nbinaries. Now imagine you want to use the resulting email inside another binary:\n\n    def welcome_message(name, username, domain) do\n      \"Welcome #{name}, your email is: #{email(username, domain)}\"\n    end\n\n    IO.puts(welcome_message(\"Meg\", \"meg\", \"example.com\"))\n    #=> \"Welcome Meg, your email is: meg@example.com\"\n\nEvery time you concatenate binaries or use interpolation (`#{}`) you are making\ncopies of those binaries. However, in many cases you don't need the complete\nbinary while you create it, but only at the end to print it out or send it\nsomewhere. In such cases, you can construct the binary by creating IO data:\n\n    def email(username, domain) do\n      [username, ?@, domain]\n    end\n\n    def welcome_message(name, username, domain) do\n      [\"Welcome \", name, \", your email is: \", email(username, domain)]\n    end\n\n    IO.puts(welcome_message(\"Meg\", \"meg\", \"example.com\"))\n    #=> \"Welcome Meg, your email is: meg@example.com\"\n\nBuilding IO data is cheaper than concatenating binaries. Concatenating multiple\npieces of IO data just means putting them together inside a list since IO data\ncan be arbitrarily nested, and that's a cheap and efficient operation. Most of\nthe IO-based APIs, such as `:gen_tcp` and `IO`, receive IO data and write it\nto the socket directly without converting it to binary.\n\nOne drawback of IO data is that you can't do things like pattern match on the\nfirst part of a piece of IO data like you can with a binary, because you usually\ndon't know the shape of the IO data. In those cases, you may need to convert it\nto a binary by calling `iodata_to_binary/1`, which is reasonably efficient\nsince it's implemented natively in C. Other functionality, like computing the\nlength of IO data, can be computed directly on the iodata by calling `iodata_length/1`.\n\n#","ref":"IO.html#module-use-cases-for-io-data"},{"type":"module","title":"Chardata - IO","doc":"Erlang and Elixir also have the idea of `t:chardata/0`. Chardata is very\nsimilar to IO data: the only difference is that integers in IO data represent\nbytes while integers in chardata represent Unicode code points. Bytes\n(`t:byte/0`) are integers within the `0..255` range, while Unicode code points\n(`t:char/0`) are integers within the `0..0x10FFFF` range. The `IO` module provides\nthe `chardata_to_string/1` function for chardata as the \"counter-part\" of the\n`iodata_to_binary/1` function for IO data.\n\nIf you try to use `iodata_to_binary/1` on chardata, it will result in an\nargument error. For example, let's try to put a code point that is not\nrepresentable with one byte, like `?π`, inside IO data:\n\n    IO.iodata_to_binary([\"The symbol for pi is: \", ?π])\n    #=> ** (ArgumentError) argument error\n\nIf we use chardata instead, it will work as expected:\n\n    iex> IO.chardata_to_string([\"The symbol for pi is: \", ?π])\n    \"The symbol for pi is: π\"","ref":"IO.html#module-chardata"},{"type":"function","title":"IO.binread/2","doc":"Reads from the IO `device`. The operation is Unicode unsafe.\n\nThe `device` is iterated as specified by the `line_or_chars` argument:\n\n  * if `line_or_chars` is an integer, it represents a number of bytes. The device is\n    iterated by that number of bytes.\n\n  * if `line_or_chars` is `:line`, the device is iterated line by line.\n\n  * if `line_or_chars` is `:eof`, the device is iterated until `:eof`. `line_or_chars`\n    can only be `:eof` since Elixir 1.13.0. `:eof` replaces the deprecated `:all`,\n    with the difference that `:all` returns `\"\"` on end of file, while `:eof` returns\n    `:eof` itself.\n\nIt returns:\n\n  * `data` - the output bytes\n\n  * `:eof` - end of file was encountered\n\n  * `{:error, reason}` - other (rare) error condition;\n    for instance, `{:error, :estale}` if reading from an\n    NFS volume\n\nNote: do not use this function on IO devices in Unicode mode\nas it will return the wrong result.","ref":"IO.html#binread/2"},{"type":"function","title":"IO.binstream/0","doc":"Returns a raw, line-based `IO.Stream` on `:stdio`. The operation is Unicode unsafe.\n\nThis is equivalent to:\n\n    IO.binstream(:stdio, :line)","ref":"IO.html#binstream/0"},{"type":"function","title":"IO.binstream/2","doc":"Converts the IO `device` into an `IO.Stream`. The operation is Unicode unsafe.\n\nAn `IO.Stream` implements both `Enumerable` and\n`Collectable`, allowing it to be used for both read\nand write.\n\nThe `device` is iterated by the given number of bytes or line by line if\n`:line` is given. This reads from the IO device as a raw binary.\n\nNote that an IO stream has side effects and every time\nyou go over the stream you may get different results.\n\nFinally, do not use this function on IO devices in Unicode\nmode as it will return the wrong result.\n\n`binstream/0` has been introduced in Elixir v1.12.0,\nwhile `binstream/2` has been available since v1.0.0.","ref":"IO.html#binstream/2"},{"type":"function","title":"IO.binwrite/2","doc":"Writes `iodata` to the given `device`.\n\nThis operation is meant to be used with \"raw\" devices\nthat are started without an encoding. The given `iodata`\nis written as is to the device, without conversion. For\nmore information on IO data, see the \"IO data\" section in\nthe module documentation.\n\nUse `write/2` for devices with encoding.\n\nImportant: do **not** use this function on IO devices in\nUnicode mode as it will write the wrong data. In particular,\nthe standard IO device is set to Unicode by default, so writing\nto stdio with this function will likely result in the wrong data\nbeing sent down the wire.","ref":"IO.html#binwrite/2"},{"type":"function","title":"IO.chardata_to_string/1","doc":"Converts chardata into a string.\n\nFor more information about chardata, see the [\"Chardata\"](#module-chardata)\nsection in the module documentation.\n\nIn case the conversion fails, it raises an `UnicodeConversionError`.\nIf a string is given, it returns the string itself.","ref":"IO.html#chardata_to_string/1"},{"type":"function","title":"Examples - IO.chardata_to_string/1","doc":"iex> IO.chardata_to_string([0x00E6, 0x00DF])\n    \"æß\"\n\n    iex> IO.chardata_to_string([0x0061, \"bc\"])\n    \"abc\"\n\n    iex> IO.chardata_to_string(\"string\")\n    \"string\"","ref":"IO.html#chardata_to_string/1-examples"},{"type":"function","title":"IO.getn/2","doc":"Gets a number of bytes from IO device `:stdio`.\n\nIf `:stdio` is a Unicode device, `count` implies\nthe number of Unicode code points to be retrieved.\nOtherwise, `count` is the number of raw bytes to be retrieved.\n\nSee `IO.getn/3` for a description of return values.","ref":"IO.html#getn/2"},{"type":"function","title":"IO.getn/3","doc":"Gets a number of bytes from the IO `device`.\n\nIf the IO `device` is a Unicode device, `count` implies\nthe number of Unicode code points to be retrieved.\nOtherwise, `count` is the number of raw bytes to be retrieved.\n\nIt returns:\n\n  * `data` - the input characters\n\n  * `:eof` - end of file was encountered\n\n  * `{:error, reason}` - other (rare) error condition;\n    for instance, `{:error, :estale}` if reading from an\n    NFS volume","ref":"IO.html#getn/3"},{"type":"function","title":"IO.gets/2","doc":"Reads a line from the IO `device`.\n\nIt returns:\n\n  * `data` - the characters in the line terminated\n    by a line-feed (LF) or end of file (EOF)\n\n  * `:eof` - end of file was encountered\n\n  * `{:error, reason}` - other (rare) error condition;\n    for instance, `{:error, :estale}` if reading from an\n    NFS volume","ref":"IO.html#gets/2"},{"type":"function","title":"Examples - IO.gets/2","doc":"To display \"What is your name?\" as a prompt and await user input:\n\n    IO.gets(\"What is your name?\\n\")","ref":"IO.html#gets/2-examples"},{"type":"function","title":"IO.inspect/2","doc":"Inspects and writes the given `item` to the device.\n\nIt's important to note that it returns the given `item` unchanged.\nThis makes it possible to \"spy\" on values by inserting an\n`IO.inspect/2` call almost anywhere in your code, for example,\nin the middle of a pipeline.\n\nIt enables pretty printing by default with width of\n80 characters. The width can be changed by explicitly\npassing the `:width` option.\n\nThe output can be decorated with a label, by providing the `:label`\noption to easily distinguish it from other `IO.inspect/2` calls.\nThe label will be printed before the inspected `item`.\n\nSee `Inspect.Opts` for a full list of remaining formatting options.","ref":"IO.html#inspect/2"},{"type":"function","title":"Examples - IO.inspect/2","doc":"IO.inspect( >, width: 40)\n\nPrints:\n\n     >\n\nWe can use the `:label` option to decorate the output:\n\n    IO.inspect(1..100, label: \"a wonderful range\")\n\nPrints:\n\n    a wonderful range: 1..100\n\nThe `:label` option is especially useful with pipelines:\n\n    [1, 2, 3]\n    |> IO.inspect(label: \"before\")\n    |> Enum.map(&(&1 * 2))\n    |> IO.inspect(label: \"after\")\n    |> Enum.sum()\n\nPrints:\n\n    before: [1, 2, 3]\n    after: [2, 4, 6]","ref":"IO.html#inspect/2-examples"},{"type":"function","title":"IO.inspect/3","doc":"Inspects `item` according to the given options using the IO `device`.\n\nSee `inspect/2` for a full list of options.","ref":"IO.html#inspect/3"},{"type":"function","title":"IO.iodata_length/1","doc":"Returns the size of an IO data.\n\nFor more information about IO data, see the [\"IO data\"](#module-io-data)\nsection in the module documentation.\n\nInlined by the compiler.","ref":"IO.html#iodata_length/1"},{"type":"function","title":"Examples - IO.iodata_length/1","doc":"iex> IO.iodata_length([1, 2 |  >])\n    4","ref":"IO.html#iodata_length/1-examples"},{"type":"function","title":"IO.iodata_to_binary/1","doc":"Converts IO data into a binary\n\nThe operation is Unicode unsafe.\n\nNote that this function treats integers in the given IO data as\nraw bytes and does not perform any kind of encoding conversion.\nIf you want to convert from a charlist to a UTF-8-encoded string,\nuse `chardata_to_string/1` instead. For more information about\nIO data and chardata, see the [\"IO data\"](#module-io-data) section in the\nmodule documentation.\n\nIf this function receives a binary, the same binary is returned.\n\nInlined by the compiler.","ref":"IO.html#iodata_to_binary/1"},{"type":"function","title":"Examples - IO.iodata_to_binary/1","doc":"iex> bin1 =  >\n    iex> bin2 =  >\n    iex> bin3 =  >\n    iex> IO.iodata_to_binary([bin1, 1, [2, 3, bin2], 4 | bin3])\n     >\n\n    iex> bin =  >\n    iex> IO.iodata_to_binary(bin)\n     >","ref":"IO.html#iodata_to_binary/1-examples"},{"type":"function","title":"IO.puts/2","doc":"Writes `item` to the given `device`, similar to `write/2`,\nbut adds a newline at the end.\n\nBy default, the `device` is the standard output. It returns `:ok`\nif it succeeds.","ref":"IO.html#puts/2"},{"type":"function","title":"Examples - IO.puts/2","doc":"IO.puts(\"Hello World!\")\n    #=> Hello World!\n\n    IO.puts(:stderr, \"error\")\n    #=> error","ref":"IO.html#puts/2-examples"},{"type":"function","title":"IO.read/2","doc":"Reads from the IO `device`.\n\nThe `device` is iterated by the given number of characters, line by line if\n`:line` is given, or until `:eof`.\n\nIt returns:\n\n  * `data` - the output characters\n\n  * `:eof` - end of file was encountered\n\n  * `{:error, reason}` - other (rare) error condition;\n    for instance, `{:error, :estale}` if reading from an\n    NFS volume","ref":"IO.html#read/2"},{"type":"function","title":"IO.stream/0","doc":"Returns a line-based `IO.Stream` on `:stdio`.\n\nThis is equivalent to:\n\n    IO.stream(:stdio, :line)","ref":"IO.html#stream/0"},{"type":"function","title":"IO.stream/2","doc":"Converts the IO `device` into an `IO.Stream`.\n\nAn `IO.Stream` implements both `Enumerable` and\n`Collectable`, allowing it to be used for both read\nand write.\n\nThe `device` is iterated by the given number of characters or line by line if\n`:line` is given.\n\nThis reads from the IO as UTF-8. Check out\n`IO.binstream/2` to handle the IO as a raw binary.\n\nNote that an IO stream has side effects and every time\nyou go over the stream you may get different results.\n\n`stream/0` has been introduced in Elixir v1.12.0,\nwhile `stream/2` has been available since v1.0.0.","ref":"IO.html#stream/2"},{"type":"function","title":"Examples - IO.stream/2","doc":"Here is an example on how we mimic an echo server\nfrom the command line:\n\n    Enum.each(IO.stream(:stdio, :line), &IO.write(&1))\n\nAnother example where you might want to collect a user input\nevery new line and break on an empty line, followed by removing\nredundant new line characters (`\"\\n\"`):\n\n    IO.stream(:stdio, :line)\n    |> Enum.take_while(&(&1 != \"\\n\"))\n    |> Enum.map(&String.replace(&1, \"\\n\", \"\"))","ref":"IO.html#stream/2-examples"},{"type":"function","title":"IO.warn/1","doc":"Writes a `message` to stderr, along with the current stacktrace.\n\nIt returns `:ok` if it succeeds.\n\nDo not call this function at the tail of another function. Due to tail\ncall optimization, a stacktrace entry would not be added and the\nstacktrace would be incorrectly trimmed. Therefore make sure at least\none expression (or an atom such as `:ok`) follows the `IO.warn/1` call.","ref":"IO.html#warn/1"},{"type":"function","title":"Examples - IO.warn/1","doc":"IO.warn(\"variable bar is unused\")\n    #=> warning: variable bar is unused\n    #=>   (iex) evaluator.ex:108: IEx.Evaluator.eval/4","ref":"IO.html#warn/1-examples"},{"type":"function","title":"IO.warn/2","doc":"Writes a `message` to stderr, along with the given `stacktrace_info`.\n\nThe `stacktrace_info` must be one of:\n\n  * a `__STACKTRACE__`, where all entries in the stacktrace will be\n    included in the error message\n\n  * a `Macro.Env` structure (since v1.14.0), where a single stacktrace\n    entry from the compilation environment will be used\n\n  * a keyword list with at least the `:file` option representing\n    a single stacktrace entry (since v1.14.0). The `:line`, `:module`,\n    `:function` options are also supported\n\nThis function also notifies the compiler a warning was printed\n(in case --warnings-as-errors was enabled). It returns `:ok`\nif it succeeds.","ref":"IO.html#warn/2"},{"type":"function","title":"Examples - IO.warn/2","doc":"stacktrace = [{MyApp, :main, 1, [file: 'my_app.ex', line: 4]}]\n    IO.warn(\"variable bar is unused\", stacktrace)\n    #=> warning: variable bar is unused\n    #=>   my_app.ex:4: MyApp.main/1","ref":"IO.html#warn/2-examples"},{"type":"function","title":"IO.write/2","doc":"Writes `chardata` to the given `device`.\n\nBy default, the `device` is the standard output.","ref":"IO.html#write/2"},{"type":"function","title":"Examples - IO.write/2","doc":"IO.write(\"sample\")\n    #=> sample\n\n    IO.write(:stderr, \"error\")\n    #=> error","ref":"IO.html#write/2-examples"},{"type":"type","title":"IO.chardata/0","doc":"","ref":"IO.html#t:chardata/0"},{"type":"type","title":"IO.device/0","doc":"","ref":"IO.html#t:device/0"},{"type":"type","title":"IO.nodata/0","doc":"","ref":"IO.html#t:nodata/0"},{"type":"module","title":"IO.ANSI","doc":"Functionality to render ANSI escape sequences.\n\n[ANSI escape sequences](https://en.wikipedia.org/wiki/ANSI_escape_code)\nare characters embedded in text used to control formatting, color, and\nother output options on video text terminals.\n\nANSI escapes are typically enabled on all Unix terminals. They are also\navailable on Windows consoles from Windows 10, although it must be\nexplicitly enabled for the current user in the registry by running the\nfollowing command:\n\n    reg add HKCU\\Console /v VirtualTerminalLevel /t REG_DWORD /d 1\n\nAfter running the command above, you must restart your current console.","ref":"IO.ANSI.html"},{"type":"module","title":"Examples - IO.ANSI","doc":"Because the ANSI escape sequences are embedded in text, the normal usage of\nthese functions is to concatenate their output with text.\n\n    formatted_text = IO.ANSI.blue_background()   \"Example\"   IO.ANSI.reset()\n    IO.puts(formatted_text)\n\nA higher level and more convenient API is also available via `IO.ANSI.format/1`,\nwhere you use atoms to represent each ANSI escape sequence and by default\nchecks if ANSI is enabled:\n\n    IO.puts(IO.ANSI.format([:blue_background, \"Example\"]))\n\nIn case ANSI is disabled, the ANSI escape sequences are simply discarded.","ref":"IO.ANSI.html#module-examples"},{"type":"function","title":"IO.ANSI.black/0","doc":"Sets foreground color to black.","ref":"IO.ANSI.html#black/0"},{"type":"function","title":"IO.ANSI.black_background/0","doc":"Sets background color to black.","ref":"IO.ANSI.html#black_background/0"},{"type":"function","title":"IO.ANSI.blink_off/0","doc":"Blink: off.","ref":"IO.ANSI.html#blink_off/0"},{"type":"function","title":"IO.ANSI.blink_rapid/0","doc":"Blink: rapid. MS-DOS ANSI.SYS; 150 per minute or more; not widely supported.","ref":"IO.ANSI.html#blink_rapid/0"},{"type":"function","title":"IO.ANSI.blink_slow/0","doc":"Blink: slow. Less than 150 per minute.","ref":"IO.ANSI.html#blink_slow/0"},{"type":"function","title":"IO.ANSI.blue/0","doc":"Sets foreground color to blue.","ref":"IO.ANSI.html#blue/0"},{"type":"function","title":"IO.ANSI.blue_background/0","doc":"Sets background color to blue.","ref":"IO.ANSI.html#blue_background/0"},{"type":"function","title":"IO.ANSI.bright/0","doc":"Bright (increased intensity) or bold.","ref":"IO.ANSI.html#bright/0"},{"type":"function","title":"IO.ANSI.clear/0","doc":"Clears screen.","ref":"IO.ANSI.html#clear/0"},{"type":"function","title":"IO.ANSI.clear_line/0","doc":"Clears line.","ref":"IO.ANSI.html#clear_line/0"},{"type":"function","title":"IO.ANSI.color/1","doc":"Sets foreground color.","ref":"IO.ANSI.html#color/1"},{"type":"function","title":"IO.ANSI.color/3","doc":"Sets the foreground color from individual RGB values.\n\nValid values for each color are in the range 0 to 5.","ref":"IO.ANSI.html#color/3"},{"type":"function","title":"IO.ANSI.color_background/1","doc":"Sets background color.","ref":"IO.ANSI.html#color_background/1"},{"type":"function","title":"IO.ANSI.color_background/3","doc":"Sets the background color from individual RGB values.\n\nValid values for each color are in the range 0 to 5.","ref":"IO.ANSI.html#color_background/3"},{"type":"function","title":"IO.ANSI.conceal/0","doc":"Conceal. Not widely supported.","ref":"IO.ANSI.html#conceal/0"},{"type":"function","title":"IO.ANSI.crossed_out/0","doc":"Crossed-out. Characters legible, but marked for deletion. Not widely supported.","ref":"IO.ANSI.html#crossed_out/0"},{"type":"function","title":"IO.ANSI.cursor/2","doc":"Sends cursor to the absolute position specified by `line` and `column`.\n\nLine `0` and column `0` would mean the top left corner.","ref":"IO.ANSI.html#cursor/2"},{"type":"function","title":"IO.ANSI.cursor_down/1","doc":"Sends cursor `lines` down.","ref":"IO.ANSI.html#cursor_down/1"},{"type":"function","title":"IO.ANSI.cursor_left/1","doc":"Sends cursor `columns` to the left.","ref":"IO.ANSI.html#cursor_left/1"},{"type":"function","title":"IO.ANSI.cursor_right/1","doc":"Sends cursor `columns` to the right.","ref":"IO.ANSI.html#cursor_right/1"},{"type":"function","title":"IO.ANSI.cursor_up/1","doc":"Sends cursor `lines` up.","ref":"IO.ANSI.html#cursor_up/1"},{"type":"function","title":"IO.ANSI.cyan/0","doc":"Sets foreground color to cyan.","ref":"IO.ANSI.html#cyan/0"},{"type":"function","title":"IO.ANSI.cyan_background/0","doc":"Sets background color to cyan.","ref":"IO.ANSI.html#cyan_background/0"},{"type":"function","title":"IO.ANSI.default_background/0","doc":"Default background color.","ref":"IO.ANSI.html#default_background/0"},{"type":"function","title":"IO.ANSI.default_color/0","doc":"Default text color.","ref":"IO.ANSI.html#default_color/0"},{"type":"function","title":"IO.ANSI.enabled?/0","doc":"Checks if ANSI coloring is supported and enabled on this machine.\n\nThis function simply reads the configuration value for\n`:ansi_enabled` in the `:elixir` application. The value is by\ndefault `false` unless Elixir can detect during startup that\nboth `stdout` and `stderr` are terminals.","ref":"IO.ANSI.html#enabled?/0"},{"type":"function","title":"IO.ANSI.encircled/0","doc":"Encircled.","ref":"IO.ANSI.html#encircled/0"},{"type":"function","title":"IO.ANSI.faint/0","doc":"Faint (decreased intensity). Not widely supported.","ref":"IO.ANSI.html#faint/0"},{"type":"function","title":"IO.ANSI.font_1/0","doc":"Sets alternative font 1.","ref":"IO.ANSI.html#font_1/0"},{"type":"function","title":"IO.ANSI.font_2/0","doc":"Sets alternative font 2.","ref":"IO.ANSI.html#font_2/0"},{"type":"function","title":"IO.ANSI.font_3/0","doc":"Sets alternative font 3.","ref":"IO.ANSI.html#font_3/0"},{"type":"function","title":"IO.ANSI.font_4/0","doc":"Sets alternative font 4.","ref":"IO.ANSI.html#font_4/0"},{"type":"function","title":"IO.ANSI.font_5/0","doc":"Sets alternative font 5.","ref":"IO.ANSI.html#font_5/0"},{"type":"function","title":"IO.ANSI.font_6/0","doc":"Sets alternative font 6.","ref":"IO.ANSI.html#font_6/0"},{"type":"function","title":"IO.ANSI.font_7/0","doc":"Sets alternative font 7.","ref":"IO.ANSI.html#font_7/0"},{"type":"function","title":"IO.ANSI.font_8/0","doc":"Sets alternative font 8.","ref":"IO.ANSI.html#font_8/0"},{"type":"function","title":"IO.ANSI.font_9/0","doc":"Sets alternative font 9.","ref":"IO.ANSI.html#font_9/0"},{"type":"function","title":"IO.ANSI.format/2","doc":"Formats a chardata-like argument by converting named ANSI sequences into actual\nANSI codes.\n\nThe named sequences are represented by atoms.\n\nIt will also append an `IO.ANSI.reset/0` to the chardata when a conversion is\nperformed. If you don't want this behaviour, use `format_fragment/2`.\n\nAn optional boolean parameter can be passed to enable or disable\nemitting actual ANSI codes. When `false`, no ANSI codes will be emitted.\nBy default checks if ANSI is enabled using the `enabled?/0` function.","ref":"IO.ANSI.html#format/2"},{"type":"function","title":"Examples - IO.ANSI.format/2","doc":"iex> IO.ANSI.format([\"Hello, \", :red, :bright, \"world!\"], true)\n    [[[[[[], \"Hello, \"] | \"\\e[31m\"] | \"\\e[1m\"], \"world!\"] | \"\\e[0m\"]","ref":"IO.ANSI.html#format/2-examples"},{"type":"function","title":"IO.ANSI.format_fragment/2","doc":"Formats a chardata-like argument by converting named ANSI sequences into actual\nANSI codes.\n\nThe named sequences are represented by atoms.\n\nAn optional boolean parameter can be passed to enable or disable\nemitting actual ANSI codes. When `false`, no ANSI codes will be emitted.\nBy default checks if ANSI is enabled using the `enabled?/0` function.","ref":"IO.ANSI.html#format_fragment/2"},{"type":"function","title":"Examples - IO.ANSI.format_fragment/2","doc":"iex> IO.ANSI.format_fragment([:bright, 'Word'], true)\n    [[[[[[] | \"\\e[1m\"], 87], 111], 114], 100]","ref":"IO.ANSI.html#format_fragment/2-examples"},{"type":"function","title":"IO.ANSI.framed/0","doc":"Framed.","ref":"IO.ANSI.html#framed/0"},{"type":"function","title":"IO.ANSI.green/0","doc":"Sets foreground color to green.","ref":"IO.ANSI.html#green/0"},{"type":"function","title":"IO.ANSI.green_background/0","doc":"Sets background color to green.","ref":"IO.ANSI.html#green_background/0"},{"type":"function","title":"IO.ANSI.home/0","doc":"Sends cursor home.","ref":"IO.ANSI.html#home/0"},{"type":"function","title":"IO.ANSI.inverse/0","doc":"Image: negative. Swap foreground and background.","ref":"IO.ANSI.html#inverse/0"},{"type":"function","title":"IO.ANSI.inverse_off/0","doc":"Image: positive. Normal foreground and background.","ref":"IO.ANSI.html#inverse_off/0"},{"type":"function","title":"IO.ANSI.italic/0","doc":"Italic: on. Not widely supported. Sometimes treated as inverse.","ref":"IO.ANSI.html#italic/0"},{"type":"function","title":"IO.ANSI.light_black/0","doc":"Sets foreground color to light black.","ref":"IO.ANSI.html#light_black/0"},{"type":"function","title":"IO.ANSI.light_black_background/0","doc":"Sets background color to light black.","ref":"IO.ANSI.html#light_black_background/0"},{"type":"function","title":"IO.ANSI.light_blue/0","doc":"Sets foreground color to light blue.","ref":"IO.ANSI.html#light_blue/0"},{"type":"function","title":"IO.ANSI.light_blue_background/0","doc":"Sets background color to light blue.","ref":"IO.ANSI.html#light_blue_background/0"},{"type":"function","title":"IO.ANSI.light_cyan/0","doc":"Sets foreground color to light cyan.","ref":"IO.ANSI.html#light_cyan/0"},{"type":"function","title":"IO.ANSI.light_cyan_background/0","doc":"Sets background color to light cyan.","ref":"IO.ANSI.html#light_cyan_background/0"},{"type":"function","title":"IO.ANSI.light_green/0","doc":"Sets foreground color to light green.","ref":"IO.ANSI.html#light_green/0"},{"type":"function","title":"IO.ANSI.light_green_background/0","doc":"Sets background color to light green.","ref":"IO.ANSI.html#light_green_background/0"},{"type":"function","title":"IO.ANSI.light_magenta/0","doc":"Sets foreground color to light magenta.","ref":"IO.ANSI.html#light_magenta/0"},{"type":"function","title":"IO.ANSI.light_magenta_background/0","doc":"Sets background color to light magenta.","ref":"IO.ANSI.html#light_magenta_background/0"},{"type":"function","title":"IO.ANSI.light_red/0","doc":"Sets foreground color to light red.","ref":"IO.ANSI.html#light_red/0"},{"type":"function","title":"IO.ANSI.light_red_background/0","doc":"Sets background color to light red.","ref":"IO.ANSI.html#light_red_background/0"},{"type":"function","title":"IO.ANSI.light_white/0","doc":"Sets foreground color to light white.","ref":"IO.ANSI.html#light_white/0"},{"type":"function","title":"IO.ANSI.light_white_background/0","doc":"Sets background color to light white.","ref":"IO.ANSI.html#light_white_background/0"},{"type":"function","title":"IO.ANSI.light_yellow/0","doc":"Sets foreground color to light yellow.","ref":"IO.ANSI.html#light_yellow/0"},{"type":"function","title":"IO.ANSI.light_yellow_background/0","doc":"Sets background color to light yellow.","ref":"IO.ANSI.html#light_yellow_background/0"},{"type":"function","title":"IO.ANSI.magenta/0","doc":"Sets foreground color to magenta.","ref":"IO.ANSI.html#magenta/0"},{"type":"function","title":"IO.ANSI.magenta_background/0","doc":"Sets background color to magenta.","ref":"IO.ANSI.html#magenta_background/0"},{"type":"function","title":"IO.ANSI.no_underline/0","doc":"Underline: none.","ref":"IO.ANSI.html#no_underline/0"},{"type":"function","title":"IO.ANSI.normal/0","doc":"Normal color or intensity.","ref":"IO.ANSI.html#normal/0"},{"type":"function","title":"IO.ANSI.not_framed_encircled/0","doc":"Not framed or encircled.","ref":"IO.ANSI.html#not_framed_encircled/0"},{"type":"function","title":"IO.ANSI.not_italic/0","doc":"Not italic.","ref":"IO.ANSI.html#not_italic/0"},{"type":"function","title":"IO.ANSI.not_overlined/0","doc":"Not overlined.","ref":"IO.ANSI.html#not_overlined/0"},{"type":"function","title":"IO.ANSI.overlined/0","doc":"Overlined.","ref":"IO.ANSI.html#overlined/0"},{"type":"function","title":"IO.ANSI.primary_font/0","doc":"Sets primary (default) font.","ref":"IO.ANSI.html#primary_font/0"},{"type":"function","title":"IO.ANSI.red/0","doc":"Sets foreground color to red.","ref":"IO.ANSI.html#red/0"},{"type":"function","title":"IO.ANSI.red_background/0","doc":"Sets background color to red.","ref":"IO.ANSI.html#red_background/0"},{"type":"function","title":"IO.ANSI.reset/0","doc":"Resets all attributes.","ref":"IO.ANSI.html#reset/0"},{"type":"function","title":"IO.ANSI.reverse/0","doc":"Image: negative. Swap foreground and background.","ref":"IO.ANSI.html#reverse/0"},{"type":"function","title":"IO.ANSI.reverse_off/0","doc":"Image: positive. Normal foreground and background.","ref":"IO.ANSI.html#reverse_off/0"},{"type":"function","title":"IO.ANSI.syntax_colors/0","doc":"Syntax colors to be used by `Inspect`.\n\nThose colors are used throughout Elixir's standard library,\nsuch as `dbg/2` and `IEx`.\n\nThe colors can be changed by setting the `:ansi_syntax_colors`\nin the `:elixir` application configuration. Configuration for\nmost built-in data types are supported: `:atom`, `:binary`,\n`:boolean`, `:charlist`, `:list`, `:map`, `:nil`, `:number`,\n`:string`, and `:tuple`. The default is:\n\n    [\n      atom: :cyan\n      boolean: :magenta,\n      charlist: :yellow,\n      nil: :magenta,\n      number: :yellow,\n      string: :green\n    ]","ref":"IO.ANSI.html#syntax_colors/0"},{"type":"function","title":"IO.ANSI.underline/0","doc":"Underline: single.","ref":"IO.ANSI.html#underline/0"},{"type":"function","title":"IO.ANSI.white/0","doc":"Sets foreground color to white.","ref":"IO.ANSI.html#white/0"},{"type":"function","title":"IO.ANSI.white_background/0","doc":"Sets background color to white.","ref":"IO.ANSI.html#white_background/0"},{"type":"function","title":"IO.ANSI.yellow/0","doc":"Sets foreground color to yellow.","ref":"IO.ANSI.html#yellow/0"},{"type":"function","title":"IO.ANSI.yellow_background/0","doc":"Sets background color to yellow.","ref":"IO.ANSI.html#yellow_background/0"},{"type":"type","title":"IO.ANSI.ansicode/0","doc":"","ref":"IO.ANSI.html#t:ansicode/0"},{"type":"type","title":"IO.ANSI.ansidata/0","doc":"","ref":"IO.ANSI.html#t:ansidata/0"},{"type":"type","title":"IO.ANSI.ansilist/0","doc":"","ref":"IO.ANSI.html#t:ansilist/0"},{"type":"module","title":"IO.Stream","doc":"Defines an `IO.Stream` struct returned by `IO.stream/2` and `IO.binstream/2`.\n\nThe following fields are public:\n\n  * `device`        - the IO device\n  * `raw`           - a boolean indicating if bin functions should be used\n  * `line_or_bytes` - if reading should read lines or a given number of bytes\n\nIt is worth noting that an IO stream has side effects and every time you go\nover the stream you may get different results.","ref":"IO.Stream.html"},{"type":"type","title":"IO.Stream.t/0","doc":"","ref":"IO.Stream.html#t:t/0"},{"type":"module","title":"OptionParser","doc":"Functions for parsing command line arguments.\n\nWhen calling a command, it's possible to pass command line options\nto modify what the command does. In this documentation, those are\ncalled \"switches\", in other situations they may be called \"flags\"\nor simply \"options\". A switch can be given a value, also called an\n\"argument\".\n\nThe main function in this module is `parse/2`, which parses a list\nof command line options and arguments into a keyword list:\n\n    iex> OptionParser.parse([\"--debug\"], strict: [debug: :boolean])\n    {[debug: true], [], []}\n\n`OptionParser` provides some conveniences out of the box,\nsuch as aliases and automatic handling of negation switches.\n\nThe `parse_head/2` function is an alternative to `parse/2`\nwhich stops parsing as soon as it finds a value that is not\na switch nor a value for a previous switch.\n\nThis module also provides low-level functions, such as `next/2`,\nfor parsing switches manually, as well as `split/1` and `to_argv/1`\nfor parsing from and converting switches to strings.","ref":"OptionParser.html"},{"type":"function","title":"OptionParser.next/2","doc":"Low-level function that parses one option.\n\nIt accepts the same options as `parse/2` and `parse_head/2`\nas both functions are built on top of this function. This function\nmay return:\n\n  * `{:ok, key, value, rest}` - the option `key` with `value` was\n    successfully parsed\n\n  * `{:invalid, key, value, rest}` - the option `key` is invalid with `value`\n    (returned when the value cannot be parsed according to the switch type)\n\n  * `{:undefined, key, value, rest}` - the option `key` is undefined\n    (returned in strict mode when the switch is unknown or on nonexistent atoms)\n\n  * `{:error, rest}` - there are no switches at the head of the given `argv`","ref":"OptionParser.html#next/2"},{"type":"function","title":"OptionParser.parse/2","doc":"Parses `argv` into a keyword list.\n\nIt returns a three-element tuple with the form `{parsed, args, invalid}`, where:\n\n  * `parsed` is a keyword list of parsed switches with `{switch_name, value}`\n    tuples in it; `switch_name` is the atom representing the switch name while\n    `value` is the value for that switch parsed according to `opts` (see the\n    \"Examples\" section for more information)\n  * `args` is a list of the remaining arguments in `argv` as strings\n  * `invalid` is a list of invalid options as `{option_name, value}` where\n    `option_name` is the raw option and `value` is `nil` if the option wasn't\n    expected or the string value if the value didn't have the expected type for\n    the corresponding option\n\nElixir converts switches to underscored atoms, so `--source-path` becomes\n`:source_path`. This is done to better suit Elixir conventions. However, this\nmeans that switches can't contain underscores and switches that do contain\nunderscores are always returned in the list of invalid switches.\n\nWhen parsing, it is common to list switches and their expected types:\n\n    iex> OptionParser.parse([\"--debug\"], strict: [debug: :boolean])\n    {[debug: true], [], []}\n\n    iex> OptionParser.parse([\"--source\", \"lib\"], strict: [source: :string])\n    {[source: \"lib\"], [], []}\n\n    iex> OptionParser.parse(\n    ...>   [\"--source-path\", \"lib\", \"test/enum_test.exs\", \"--verbose\"],\n    ...>   strict: [source_path: :string, verbose: :boolean]\n    ...> )\n    {[source_path: \"lib\", verbose: true], [\"test/enum_test.exs\"], []}\n\nWe will explore the valid switches and operation modes of option parser below.","ref":"OptionParser.html#parse/2"},{"type":"function","title":"Options - OptionParser.parse/2","doc":"The following options are supported:\n\n  * `:switches` or `:strict` - see the \"Switch definitions\" section below\n  * `:allow_nonexistent_atoms` - see the \"Parsing unknown switches\" section below\n  * `:aliases` - see the \"Aliases\" section below\n  * `:return_separator` - see the \"Return separator\" section below","ref":"OptionParser.html#parse/2-options"},{"type":"function","title":"Switch definitions - OptionParser.parse/2","doc":"Switches can be specified via one of two options:\n\n  * `:strict` - defines strict switches and their types. Any switch\n    in `argv` that is not specified in the list is returned in the\n    invalid options list. This is the preferred way to parse options.\n\n  * `:switches` - defines switches and their types. This function\n    still attempts to parse switches that are not in this list.\n\nBoth these options accept a keyword list where the key is an atom\ndefining the name of the switch and value is the `type` of the\nswitch (see the \"Types\" section below for more information).\n\nNote that you should only supply the `:switches` or the `:strict` option.\nIf you supply both, an `ArgumentError` exception will be raised.\n\n#","ref":"OptionParser.html#parse/2-switch-definitions"},{"type":"function","title":"Types - OptionParser.parse/2","doc":"Switches parsed by `OptionParser` may take zero or one arguments.\n\nThe following switches types take no arguments:\n\n  * `:boolean` - sets the value to `true` when given (see also the\n    \"Negation switches\" section below)\n  * `:count` - counts the number of times the switch is given\n\nThe following switches take one argument:\n\n  * `:integer` - parses the value as an integer\n  * `:float` - parses the value as a float\n  * `:string` - parses the value as a string\n\nIf a switch can't be parsed according to the given type, it is\nreturned in the invalid options list.\n\n#","ref":"OptionParser.html#parse/2-types"},{"type":"function","title":"Modifiers - OptionParser.parse/2","doc":"Switches can be specified with modifiers, which change how\nthey behave. The following modifiers are supported:\n\n  * `:keep` - keeps duplicated elements instead of overriding them;\n    works with all types except `:count`. Specifying `switch_name: :keep`\n    assumes the type of `:switch_name` will be `:string`.\n\nTo use `:keep` with a type other than `:string`, use a list as the type\nfor the switch. For example: `[foo: [:integer, :keep]]`.\n\n#","ref":"OptionParser.html#parse/2-modifiers"},{"type":"function","title":"Negation switches - OptionParser.parse/2","doc":"In case a switch `SWITCH` is specified to have type `:boolean`, it may be\npassed as `--no-SWITCH` as well which will set the option to `false`:\n\n    iex> OptionParser.parse([\"--no-op\", \"path/to/file\"], switches: [op: :boolean])\n    {[op: false], [\"path/to/file\"], []}\n\n#","ref":"OptionParser.html#parse/2-negation-switches"},{"type":"function","title":"Parsing unknown switches - OptionParser.parse/2","doc":"When the `:switches` option is given, `OptionParser` will attempt to parse\nunknown switches:\n\n    iex> OptionParser.parse([\"--debug\"], switches: [key: :string])\n    {[debug: true], [], []}\n\nEven though we haven't specified `--debug` in the list of switches, it is part\nof the returned options. This would also work:\n\n    iex> OptionParser.parse([\"--debug\", \"value\"], switches: [key: :string])\n    {[debug: \"value\"], [], []}\n\nSwitches followed by a value will be assigned the value, as a string. Switches\nwithout an argument will be set automatically to `true`. Since we cannot assert\nthe type of the switch value, it is preferred to use the `:strict` option that\naccepts only known switches and always verify their types.\n\nIf you do want to parse unknown switches, remember that Elixir converts switches\nto atoms. Since atoms are not garbage-collected, OptionParser will only parse\nswitches that translate to atoms used by the runtime to avoid leaking atoms.\nFor instance, the code below will discard the `--option-parser-example` switch\nbecause the `:option_parser_example` atom is never used anywhere:\n\n    OptionParser.parse([\"--option-parser-example\"], switches: [debug: :boolean])\n    # The :option_parser_example atom is not used anywhere below\n\nHowever, the code below would work as long as `:option_parser_example` atom is\nused at some point later (or earlier) **in the same module**. For example:\n\n    {opts, _, _} = OptionParser.parse([\"--option-parser-example\"], switches: [debug: :boolean])\n    # ... then somewhere in the same module you access it ...\n    opts[:option_parser_example]\n\nIn other words, Elixir will only parse options that are used by the runtime,\nignoring all others. If you would like to parse all switches, regardless if\nthey exist or not, you can force creation of atoms by passing\n`allow_nonexistent_atoms: true` as option. Use this option with care. It is\nonly useful when you are building command-line applications that receive\ndynamically-named arguments and must be avoided in long-running systems.","ref":"OptionParser.html#parse/2-parsing-unknown-switches"},{"type":"function","title":"Aliases - OptionParser.parse/2","doc":"A set of aliases can be specified in the `:aliases` option:\n\n    iex> OptionParser.parse([\"-d\"], aliases: [d: :debug], strict: [debug: :boolean])\n    {[debug: true], [], []}","ref":"OptionParser.html#parse/2-aliases"},{"type":"function","title":"Examples - OptionParser.parse/2","doc":"Here are some examples of working with different types and modifiers:\n\n    iex> OptionParser.parse([\"--unlock\", \"path/to/file\"], strict: [unlock: :boolean])\n    {[unlock: true], [\"path/to/file\"], []}\n\n    iex> OptionParser.parse(\n    ...>   [\"--unlock\", \"--limit\", \"0\", \"path/to/file\"],\n    ...>   strict: [unlock: :boolean, limit: :integer]\n    ...> )\n    {[unlock: true, limit: 0], [\"path/to/file\"], []}\n\n    iex> OptionParser.parse([\"--limit\", \"3\"], strict: [limit: :integer])\n    {[limit: 3], [], []}\n\n    iex> OptionParser.parse([\"--limit\", \"xyz\"], strict: [limit: :integer])\n    {[], [], [{\"--limit\", \"xyz\"}]}\n\n    iex> OptionParser.parse([\"--verbose\"], switches: [verbose: :count])\n    {[verbose: 1], [], []}\n\n    iex> OptionParser.parse([\"-v\", \"-v\"], aliases: [v: :verbose], strict: [verbose: :count])\n    {[verbose: 2], [], []}\n\n    iex> OptionParser.parse([\"--unknown\", \"xyz\"], strict: [])\n    {[], [\"xyz\"], [{\"--unknown\", nil}]}\n\n    iex> OptionParser.parse(\n    ...>   [\"--limit\", \"3\", \"--unknown\", \"xyz\"],\n    ...>   switches: [limit: :integer]\n    ...> )\n    {[limit: 3, unknown: \"xyz\"], [], []}\n\n    iex> OptionParser.parse(\n    ...>   [\"--unlock\", \"path/to/file\", \"--unlock\", \"path/to/another/file\"],\n    ...>   strict: [unlock: :keep]\n    ...> )\n    {[unlock: \"path/to/file\", unlock: \"path/to/another/file\"], [], []}","ref":"OptionParser.html#parse/2-examples"},{"type":"function","title":"Return separator - OptionParser.parse/2","doc":"The separator `--` implies options should no longer be processed.\nBy default, the separator is not returned as parts of the arguments,\nbut that can be changed via the `:return_separator` option:\n\n    iex> OptionParser.parse([\"--\", \"lib\"], return_separator: true, strict: [])\n    {[], [\"--\", \"lib\"], []}\n\n    iex> OptionParser.parse([\"--no-halt\", \"--\", \"lib\"], return_separator: true, switches: [halt: :boolean])\n    {[halt: false], [\"--\", \"lib\"], []}\n\n    iex> OptionParser.parse([\"script.exs\", \"--no-halt\", \"--\", \"foo\"], return_separator: true, switches: [halt: :boolean])\n    {[{:halt, false}], [\"script.exs\", \"--\", \"foo\"], []}","ref":"OptionParser.html#parse/2-return-separator"},{"type":"function","title":"OptionParser.parse!/2","doc":"The same as `parse/2` but raises an `OptionParser.ParseError`\nexception if any invalid options are given.\n\nIf there are no errors, returns a `{parsed, rest}` tuple where:\n\n  * `parsed` is the list of parsed switches (same as in `parse/2`)\n  * `rest` is the list of arguments (same as in `parse/2`)","ref":"OptionParser.html#parse!/2"},{"type":"function","title":"Examples - OptionParser.parse!/2","doc":"iex> OptionParser.parse!([\"--debug\", \"path/to/file\"], strict: [debug: :boolean])\n    {[debug: true], [\"path/to/file\"]}\n\n    iex> OptionParser.parse!([\"--limit\", \"xyz\"], strict: [limit: :integer])\n    ** (OptionParser.ParseError) 1 error found!\n    --limit : Expected type integer, got \"xyz\"\n\n    iex> OptionParser.parse!([\"--unknown\", \"xyz\"], strict: [])\n    ** (OptionParser.ParseError) 1 error found!\n    --unknown : Unknown option\n\n    iex> OptionParser.parse!(\n    ...>   [\"-l\", \"xyz\", \"-f\", \"bar\"],\n    ...>   switches: [limit: :integer, foo: :integer],\n    ...>   aliases: [l: :limit, f: :foo]\n    ...> )\n    ** (OptionParser.ParseError) 2 errors found!\n    -l : Expected type integer, got \"xyz\"\n    -f : Expected type integer, got \"bar\"","ref":"OptionParser.html#parse!/2-examples"},{"type":"function","title":"OptionParser.parse_head/2","doc":"Similar to `parse/2` but only parses the head of `argv`;\nas soon as it finds a non-switch, it stops parsing.\n\nSee `parse/2` for more information.","ref":"OptionParser.html#parse_head/2"},{"type":"function","title":"Example - OptionParser.parse_head/2","doc":"iex> OptionParser.parse_head(\n    ...>   [\"--source\", \"lib\", \"test/enum_test.exs\", \"--verbose\"],\n    ...>   switches: [source: :string, verbose: :boolean]\n    ...> )\n    {[source: \"lib\"], [\"test/enum_test.exs\", \"--verbose\"], []}\n\n    iex> OptionParser.parse_head(\n    ...>   [\"--verbose\", \"--source\", \"lib\", \"test/enum_test.exs\", \"--unlock\"],\n    ...>   switches: [source: :string, verbose: :boolean, unlock: :boolean]\n    ...> )\n    {[verbose: true, source: \"lib\"], [\"test/enum_test.exs\", \"--unlock\"], []}","ref":"OptionParser.html#parse_head/2-example"},{"type":"function","title":"OptionParser.parse_head!/2","doc":"The same as `parse_head/2` but raises an `OptionParser.ParseError`\nexception if any invalid options are given.\n\nIf there are no errors, returns a `{parsed, rest}` tuple where:\n\n  * `parsed` is the list of parsed switches (same as in `parse_head/2`)\n  * `rest` is the list of arguments (same as in `parse_head/2`)","ref":"OptionParser.html#parse_head!/2"},{"type":"function","title":"Examples - OptionParser.parse_head!/2","doc":"iex> OptionParser.parse_head!(\n    ...>   [\"--source\", \"lib\", \"path/to/file\", \"--verbose\"],\n    ...>   switches: [source: :string, verbose: :boolean]\n    ...> )\n    {[source: \"lib\"], [\"path/to/file\", \"--verbose\"]}\n\n    iex> OptionParser.parse_head!(\n    ...>   [\"--number\", \"lib\", \"test/enum_test.exs\", \"--verbose\"],\n    ...>   strict: [number: :integer]\n    ...> )\n    ** (OptionParser.ParseError) 1 error found!\n    --number : Expected type integer, got \"lib\"\n\n    iex> OptionParser.parse_head!(\n    ...>   [\"--verbose\", \"--source\", \"lib\", \"test/enum_test.exs\", \"--unlock\"],\n    ...>   strict: [verbose: :integer, source: :integer]\n    ...> )\n    ** (OptionParser.ParseError) 2 errors found!\n    --verbose : Missing argument of type integer\n    --source : Expected type integer, got \"lib\"","ref":"OptionParser.html#parse_head!/2-examples"},{"type":"function","title":"OptionParser.split/1","doc":"Splits a string into `t:argv/0` chunks.\n\nThis function splits the given `string` into a list of strings in a similar\nway to many shells.","ref":"OptionParser.html#split/1"},{"type":"function","title":"Examples - OptionParser.split/1","doc":"iex> OptionParser.split(\"foo bar\")\n    [\"foo\", \"bar\"]\n\n    iex> OptionParser.split(\"foo \\\"bar baz\\\"\")\n    [\"foo\", \"bar baz\"]","ref":"OptionParser.html#split/1-examples"},{"type":"function","title":"OptionParser.to_argv/2","doc":"Receives a key-value enumerable and converts it to `t:argv/0`.\n\nKeys must be atoms. Keys with `nil` value are discarded,\nboolean values are converted to `--key` or `--no-key`\n(if the value is `true` or `false`, respectively),\nand all other values are converted using `to_string/1`.\n\nIt is advised to pass to `to_argv/2` the same set of `options`\ngiven to `parse/2`. Some switches can only be reconstructed\ncorrectly with the `:switches` information in hand.","ref":"OptionParser.html#to_argv/2"},{"type":"function","title":"Examples - OptionParser.to_argv/2","doc":"iex> OptionParser.to_argv(foo_bar: \"baz\")\n    [\"--foo-bar\", \"baz\"]\n    iex> OptionParser.to_argv(bool: true, bool: false, discarded: nil)\n    [\"--bool\", \"--no-bool\"]\n\nSome switches will output different values based on the switches\ntypes:\n\n    iex> OptionParser.to_argv([number: 2], switches: [])\n    [\"--number\", \"2\"]\n    iex> OptionParser.to_argv([number: 2], switches: [number: :count])\n    [\"--number\", \"--number\"]","ref":"OptionParser.html#to_argv/2-examples"},{"type":"type","title":"OptionParser.argv/0","doc":"","ref":"OptionParser.html#t:argv/0"},{"type":"type","title":"OptionParser.errors/0","doc":"","ref":"OptionParser.html#t:errors/0"},{"type":"type","title":"OptionParser.options/0","doc":"","ref":"OptionParser.html#t:options/0"},{"type":"type","title":"OptionParser.parsed/0","doc":"","ref":"OptionParser.html#t:parsed/0"},{"type":"module","title":"Path","doc":"This module provides conveniences for manipulating or\nretrieving file system paths.\n\nThe functions in this module may receive chardata as\narguments and will always return a string encoded in UTF-8. Chardata\nis a string or a list of characters and strings, see `t:IO.chardata/0`.\nIf a binary is given, in whatever encoding, its encoding will be kept.\n\nThe majority of the functions in this module do not\ninteract with the file system, except for a few functions\nthat require it (like `wildcard/2` and `expand/1`).","ref":"Path.html"},{"type":"function","title":"Path.absname/1","doc":"Converts the given path to an absolute one.\n\nUnlike `expand/1`, no attempt is made to resolve `..`, `.`, or `~`.","ref":"Path.html#absname/1"},{"type":"function","title":"Examples - Path.absname/1","doc":"#","ref":"Path.html#absname/1-examples"},{"type":"function","title":"Unix-like operating systems - Path.absname/1","doc":"Path.absname(\"foo\")\n    #=> \"/usr/local/foo\"\n\n    Path.absname(\"../x\")\n    #=> \"/usr/local/../x\"\n\n#","ref":"Path.html#absname/1-unix-like-operating-systems"},{"type":"function","title":"Windows - Path.absname/1","doc":"Path.absname(\"foo\")\n    #=> \"D:/usr/local/foo\"\n\n    Path.absname(\"../x\")\n    #=> \"D:/usr/local/../x\"","ref":"Path.html#absname/1-windows"},{"type":"function","title":"Path.absname/2","doc":"Builds a path from `relative_to` to `path`.\n\nIf `path` is already an absolute path, `relative_to` is ignored. See also\n`relative_to/2`.\n\nUnlike `expand/2`, no attempt is made to\nresolve `..`, `.` or `~`.","ref":"Path.html#absname/2"},{"type":"function","title":"Examples - Path.absname/2","doc":"iex> Path.absname(\"foo\", \"bar\")\n    \"bar/foo\"\n\n    iex> Path.absname(\"../x\", \"bar\")\n    \"bar/../x\"","ref":"Path.html#absname/2-examples"},{"type":"function","title":"Path.basename/1","doc":"Returns the last component of the path or the path\nitself if it does not contain any directory separators.","ref":"Path.html#basename/1"},{"type":"function","title":"Examples - Path.basename/1","doc":"iex> Path.basename(\"foo\")\n    \"foo\"\n\n    iex> Path.basename(\"foo/bar\")\n    \"bar\"\n\n    iex> Path.basename(\"lib/module/submodule.ex\")\n    \"submodule.ex\"\n\n    iex> Path.basename(\"/\")\n    \"\"","ref":"Path.html#basename/1-examples"},{"type":"function","title":"Path.basename/2","doc":"Returns the last component of `path` with the `extension`\nstripped.\n\nThis function should be used to remove a specific\nextension which may or may not be there.","ref":"Path.html#basename/2"},{"type":"function","title":"Examples - Path.basename/2","doc":"iex> Path.basename(\"~/foo/bar.ex\", \".ex\")\n    \"bar\"\n\n    iex> Path.basename(\"~/foo/bar.exs\", \".ex\")\n    \"bar.exs\"\n\n    iex> Path.basename(\"~/foo/bar.old.ex\", \".ex\")\n    \"bar.old\"","ref":"Path.html#basename/2-examples"},{"type":"function","title":"Path.dirname/1","doc":"Returns the directory component of `path`.","ref":"Path.html#dirname/1"},{"type":"function","title":"Examples - Path.dirname/1","doc":"iex> Path.dirname(\"/foo/bar.ex\")\n    \"/foo\"\n\n    iex> Path.dirname(\"/foo/bar/baz.ex\")\n    \"/foo/bar\"\n\n    iex> Path.dirname(\"/foo/bar/\")\n    \"/foo/bar\"\n\n    iex> Path.dirname(\"bar.ex\")\n    \".\"","ref":"Path.html#dirname/1-examples"},{"type":"function","title":"Path.expand/1","doc":"Converts the path to an absolute one, expanding\nany `.` and `..` components and a leading `~`.","ref":"Path.html#expand/1"},{"type":"function","title":"Examples - Path.expand/1","doc":"Path.expand(\"/foo/bar/../baz\")\n    #=> \"/foo/baz\"","ref":"Path.html#expand/1-examples"},{"type":"function","title":"Path.expand/2","doc":"Expands the path relative to the path given as the second argument\nexpanding any `.` and `..` characters.\n\nIf the path is already an absolute path, `relative_to` is ignored.\n\nNote that this function treats a `path` with a leading `~` as\nan absolute one.\n\nThe second argument is first expanded to an absolute path.","ref":"Path.html#expand/2"},{"type":"function","title":"Examples - Path.expand/2","doc":"# Assuming that the absolute path to baz is /quux/baz\n    Path.expand(\"foo/bar/../bar\", \"baz\")\n    #=> \"/quux/baz/foo/bar\"\n\n    Path.expand(\"foo/bar/../bar\", \"/baz\")\n    #=> \"/baz/foo/bar\"\n\n    Path.expand(\"/foo/bar/../bar\", \"/baz\")\n    #=> \"/foo/bar\"","ref":"Path.html#expand/2-examples"},{"type":"function","title":"Path.extname/1","doc":"Returns the extension of the last component of `path`.\n\nFor filenames starting with a dot and without an extension, it returns\nan empty string.\n\nSee `basename/1` and `rootname/1` for related functions to extract\ninformation from paths.","ref":"Path.html#extname/1"},{"type":"function","title":"Examples - Path.extname/1","doc":"iex> Path.extname(\"foo.erl\")\n    \".erl\"\n\n    iex> Path.extname(\"~/foo/bar\")\n    \"\"\n\n    iex> Path.extname(\".gitignore\")\n    \"\"","ref":"Path.html#extname/1-examples"},{"type":"function","title":"Path.join/1","doc":"Joins a list of paths.\n\nThis function should be used to convert a list of paths to a path.\nNote that any trailing slash is removed when joining.\n\nRaises an error if the given list of paths is empty.","ref":"Path.html#join/1"},{"type":"function","title":"Examples - Path.join/1","doc":"iex> Path.join([\"~\", \"foo\"])\n    \"~/foo\"\n\n    iex> Path.join([\"foo\"])\n    \"foo\"\n\n    iex> Path.join([\"/\", \"foo\", \"bar/\"])\n    \"/foo/bar\"","ref":"Path.html#join/1-examples"},{"type":"function","title":"Path.join/2","doc":"Joins two paths.\n\nThe right path will always be expanded to its relative format\nand any trailing slash will be removed when joining.","ref":"Path.html#join/2"},{"type":"function","title":"Examples - Path.join/2","doc":"iex> Path.join(\"foo\", \"bar\")\n    \"foo/bar\"\n\n    iex> Path.join(\"/foo\", \"/bar/\")\n    \"/foo/bar\"\n\nThe functions in this module support chardata, so giving a list will\ntreat it as a single entity:\n\n    iex> Path.join(\"foo\", [\"bar\", \"fiz\"])\n    \"foo/barfiz\"\n\n    iex> Path.join([\"foo\", \"bar\"], \"fiz\")\n    \"foobar/fiz\"\n\nUse `join/1` if you need to join a list of paths instead.","ref":"Path.html#join/2-examples"},{"type":"function","title":"Path.relative/1","doc":"Forces the path to be a relative path.","ref":"Path.html#relative/1"},{"type":"function","title":"Examples - Path.relative/1","doc":"#","ref":"Path.html#relative/1-examples"},{"type":"function","title":"Unix-like operating systems - Path.relative/1","doc":"Path.relative(\"/usr/local/bin\")   #=> \"usr/local/bin\"\n    Path.relative(\"usr/local/bin\")    #=> \"usr/local/bin\"\n    Path.relative(\"../usr/local/bin\") #=> \"../usr/local/bin\"\n\n#","ref":"Path.html#relative/1-unix-like-operating-systems"},{"type":"function","title":"Windows - Path.relative/1","doc":"Path.relative(\"D:/usr/local/bin\") #=> \"usr/local/bin\"\n    Path.relative(\"usr/local/bin\")    #=> \"usr/local/bin\"\n    Path.relative(\"D:bar.ex\")         #=> \"bar.ex\"\n    Path.relative(\"/bar/foo.ex\")      #=> \"bar/foo.ex\"","ref":"Path.html#relative/1-windows"},{"type":"function","title":"Path.relative_to/2","doc":"Returns the direct relative path from `path` in relation to `from`.\n\nIn other words, this function tries to strip the `from` prefix from `path`.\n\nThis function does not query the file system, so it assumes\nno symlinks between the paths.\n\nIn case a direct relative path cannot be found, it returns\nthe original path.","ref":"Path.html#relative_to/2"},{"type":"function","title":"Examples - Path.relative_to/2","doc":"iex> Path.relative_to(\"/usr/local/foo\", \"/usr/local\")\n    \"foo\"\n\n    iex> Path.relative_to(\"/usr/local/foo\", \"/\")\n    \"usr/local/foo\"\n\n    iex> Path.relative_to(\"/usr/local/foo\", \"/etc\")\n    \"/usr/local/foo\"\n\n    iex> Path.relative_to(\"/usr/local/foo\", \"/usr/local/foo\")\n    \".\"","ref":"Path.html#relative_to/2-examples"},{"type":"function","title":"Path.relative_to_cwd/1","doc":"Convenience to get the path relative to the current working\ndirectory.\n\nIf, for some reason, the current working directory\ncannot be retrieved, this function returns the given `path`.","ref":"Path.html#relative_to_cwd/1"},{"type":"function","title":"Path.rootname/1","doc":"Returns the `path` with the `extension` stripped.","ref":"Path.html#rootname/1"},{"type":"function","title":"Examples - Path.rootname/1","doc":"iex> Path.rootname(\"/foo/bar\")\n    \"/foo/bar\"\n\n    iex> Path.rootname(\"/foo/bar.ex\")\n    \"/foo/bar\"","ref":"Path.html#rootname/1-examples"},{"type":"function","title":"Path.rootname/2","doc":"Returns the `path` with the `extension` stripped.\n\nThis function should be used to remove a specific extension which may\nor may not be there.","ref":"Path.html#rootname/2"},{"type":"function","title":"Examples - Path.rootname/2","doc":"iex> Path.rootname(\"/foo/bar.erl\", \".erl\")\n    \"/foo/bar\"\n\n    iex> Path.rootname(\"/foo/bar.erl\", \".ex\")\n    \"/foo/bar.erl\"","ref":"Path.html#rootname/2-examples"},{"type":"function","title":"Path.safe_relative/1","doc":"Returns a path relative to the current working directory that is\nprotected from directory-traversal attacks.\n\nSame as `safe_relative_to/2` with the current working directory as\nthe second argument. If there is an issue retrieving the current working\ndirectory, this function raises an error.","ref":"Path.html#safe_relative/1"},{"type":"function","title":"Examples - Path.safe_relative/1","doc":"iex> Path.safe_relative(\"foo\")\n    {:ok, \"foo\"}\n\n    iex> Path.safe_relative(\"foo/../bar\")\n    {:ok, \"bar\"}\n\n    iex> Path.safe_relative(\"foo/../..\")\n    :error\n\n    iex> Path.safe_relative(\"/usr/local\")\n    :error","ref":"Path.html#safe_relative/1-examples"},{"type":"function","title":"Path.safe_relative_to/2","doc":"Returns a relative path that is protected from directory-traversal attacks.\n\nThe given relative path is sanitized by eliminating `..` and `.` components.\n\nThis function checks that, after expanding those components, the path is still \"safe\".\nPaths are considered unsafe if either of these is true:\n\n  * The path is not relative, such as `\"/foo/bar\"`.\n\n  * A `..` component would make it so that the path would travers up above\n    the root of `relative_to`.\n\n  * A symbolic link in the path points to something above the root of `relative_to`.","ref":"Path.html#safe_relative_to/2"},{"type":"function","title":"Examples - Path.safe_relative_to/2","doc":"iex> Path.safe_relative_to(\"deps/my_dep/app.beam\", \"deps\")\n    {:ok, \"deps/my_dep/app.beam\"}\n\n    iex> Path.safe_relative_to(\"deps/my_dep/./build/../app.beam\", \"deps\")\n    {:ok, \"deps/my_dep/app.beam\"}\n\n    iex> Path.safe_relative_to(\"my_dep/../..\", \"deps\")\n    :error\n\n    iex> Path.safe_relative_to(\"/usr/local\", \".\")\n    :error","ref":"Path.html#safe_relative_to/2-examples"},{"type":"function","title":"Path.split/1","doc":"Splits the path into a list at the path separator.\n\nIf an empty string is given, returns an empty list.\n\nOn Windows, path is split on both `\"\\\"` and `\"/\"` separators\nand the driver letter, if there is one, is always returned\nin lowercase.","ref":"Path.html#split/1"},{"type":"function","title":"Examples - Path.split/1","doc":"iex> Path.split(\"\")\n    []\n\n    iex> Path.split(\"foo\")\n    [\"foo\"]\n\n    iex> Path.split(\"/foo/bar\")\n    [\"/\", \"foo\", \"bar\"]","ref":"Path.html#split/1-examples"},{"type":"function","title":"Path.type/1","doc":"Returns the path type.","ref":"Path.html#type/1"},{"type":"function","title":"Examples - Path.type/1","doc":"#","ref":"Path.html#type/1-examples"},{"type":"function","title":"Unix-like operating systems - Path.type/1","doc":"Path.type(\"/\")                #=> :absolute\n    Path.type(\"/usr/local/bin\")   #=> :absolute\n    Path.type(\"usr/local/bin\")    #=> :relative\n    Path.type(\"../usr/local/bin\") #=> :relative\n    Path.type(\"~/file\")           #=> :relative\n\n#","ref":"Path.html#type/1-unix-like-operating-systems"},{"type":"function","title":"Windows - Path.type/1","doc":"Path.type(\"D:/usr/local/bin\") #=> :absolute\n    Path.type(\"usr/local/bin\")    #=> :relative\n    Path.type(\"D:bar.ex\")         #=> :volumerelative\n    Path.type(\"/bar/foo.ex\")      #=> :volumerelative","ref":"Path.html#type/1-windows"},{"type":"function","title":"Path.wildcard/2","doc":"Traverses paths according to the given `glob` expression and returns a\nlist of matches.\n\nThe wildcard looks like an ordinary path, except that the following\n\"wildcard characters\" are interpreted in a special way:\n\n  * `?` - matches one character.\n\n  * `*` - matches any number of characters up to the end of the filename, the\n    next dot, or the next slash.\n\n  * `**` - two adjacent `*`'s used as a single pattern will match all\n    files and zero or more directories and subdirectories.\n\n  * `[char1,char2,...]` - matches any of the characters listed; two\n    characters separated by a hyphen will match a range of characters.\n    Do not add spaces before and after the comma as it would then match\n    paths containing the space character itself.\n\n  * `{item1,item2,...}` - matches one of the alternatives.\n    Do not add spaces before and after the comma as it would then match\n    paths containing the space character itself.\n\nOther characters represent themselves. Only paths that have\nexactly the same character in the same position will match. Note\nthat matching is case-sensitive: `\"a\"` will not match `\"A\"`.\n\nDirectory separators must always be written as `/`, even on Windows.\nYou may call `Path.expand/1` to normalize the path before invoking\nthis function.\n\nA character preceded by  loses its special meaning.\nNote that  must be written as \\ in a string literal.\nFor example, \"\\?*\" will match any filename starting with ?.\n\nBy default, the patterns `*` and `?` do not match files starting\nwith a dot `.`. See the `:match_dot` option in the \"Options\" section\nbelow.","ref":"Path.html#wildcard/2"},{"type":"function","title":"Options - Path.wildcard/2","doc":"* `:match_dot` - (boolean) if `false`, the special wildcard characters `*` and `?`\n    will not match files starting with a dot (`.`). If `true`, files starting with\n    a `.` will not be treated specially. Defaults to `false`.","ref":"Path.html#wildcard/2-options"},{"type":"function","title":"Examples - Path.wildcard/2","doc":"Imagine you have a directory called `projects` with three Elixir projects\ninside of it: `elixir`, `ex_doc`, and `plug`. You can find all `.beam` files\ninside the `ebin` directory of each project as follows:\n\n    Path.wildcard(\"projects/*/ebin/**/*.beam\")\n\nIf you want to search for both `.beam` and `.app` files, you could do:\n\n    Path.wildcard(\"projects/*/ebin/**/*.{beam,app}\")","ref":"Path.html#wildcard/2-examples"},{"type":"type","title":"Path.t/0","doc":"A path.","ref":"Path.html#t:t/0"},{"type":"module","title":"Port","doc":"Functions for interacting with the external world through ports.\n\nPorts provide a mechanism to start operating system processes external\nto the Erlang VM and communicate with them via message passing.","ref":"Port.html"},{"type":"module","title":"Example - Port","doc":"iex> port = Port.open({:spawn, \"cat\"}, [:binary])\n    iex> send(port, {self(), {:command, \"hello\"}})\n    iex> send(port, {self(), {:command, \"world\"}})\n    iex> flush()\n    {#Port , {:data, \"hello\"}}\n    {#Port , {:data, \"world\"}}\n    iex> send(port, {self(), :close})\n    :ok\n    iex> flush()\n    {#Port , :closed}\n    :ok\n\nIn the example above, we have created a new port that executes the\nprogram `cat`. `cat` is a program available on Unix-like operating systems that\nreceives data from multiple inputs and concatenates them in the output.\n\nAfter the port was created, we sent it two commands in the form of\nmessages using `send/2`. The first command has the binary payload\nof \"hello\" and the second has \"world\".\n\nAfter sending those two messages, we invoked the IEx helper `flush()`,\nwhich printed all messages received from the port, in this case we got\n\"hello\" and \"world\" back. Note that the messages are in binary because we\npassed the `:binary` option when opening the port in `Port.open/2`. Without\nsuch option, it would have yielded a list of bytes.\n\nOnce everything was done, we closed the port.\n\nElixir provides many conveniences for working with ports and some drawbacks.\nWe will explore those below.","ref":"Port.html#module-example"},{"type":"module","title":"Message and function APIs - Port","doc":"There are two APIs for working with ports. It can be either asynchronous via\nmessage passing, as in the example above, or by calling the functions on this\nmodule.\n\nThe messages supported by ports and their counterpart function APIs are\nlisted below:\n\n  * `{pid, {:command, binary}}` - sends the given data to the port.\n    See `command/3`.\n\n  * `{pid, :close}` - closes the port. Unless the port is already closed,\n    the port will reply with `{port, :closed}` message once it has flushed\n    its buffers and effectively closed. See `close/1`.\n\n  * `{pid, {:connect, new_pid}}` - sets the `new_pid` as the new owner of\n    the port. Once a port is opened, the port is linked and connected to the\n    caller process and communication to the port only happens through the\n    connected process. This message makes `new_pid` the new connected processes.\n    Unless the port is dead, the port will reply to the old owner with\n    `{port, :connected}`. See `connect/2`.\n\nOn its turn, the port will send the connected process the following messages:\n\n  * `{port, {:data, data}}` - data sent by the port\n  * `{port, :closed}` - reply to the `{pid, :close}` message\n  * `{port, :connected}` - reply to the `{pid, {:connect, new_pid}}` message\n  * `{:EXIT, port, reason}` - exit signals in case the port crashes. If reason\n    is not `:normal`, this message will only be received if the owner process\n    is trapping exits","ref":"Port.html#module-message-and-function-apis"},{"type":"module","title":"Open mechanisms - Port","doc":"The port can be opened through four main mechanisms.\n\nAs a short summary, prefer to using the `:spawn` and `:spawn_executable`\noptions mentioned below. The other two options, `:spawn_driver` and `:fd`\nare for advanced usage within the VM. Also consider using `System.cmd/3`\nif all you want is to execute a program and retrieve its return value.\n\n#","ref":"Port.html#module-open-mechanisms"},{"type":"module","title":"spawn - Port","doc":"The `:spawn` tuple receives a binary that is going to be executed as a\nfull invocation. For example, we can use it to invoke \"echo hello\" directly:\n\n    iex> port = Port.open({:spawn, \"echo hello\"}, [:binary])\n    iex> flush()\n    {#Port , {:data, \"hello\\n\"}}\n\n`:spawn` will retrieve the program name from the argument and traverse your\noperating system `$PATH` environment variable looking for a matching program.\n\nAlthough the above is handy, it means it is impossible to invoke an executable\nthat has whitespaces on its name or in any of its arguments. For those reasons,\nmost times it is preferable to execute `:spawn_executable`.\n\n#","ref":"Port.html#module-spawn"},{"type":"module","title":"spawn_executable - Port","doc":"Spawn executable is a more restricted and explicit version of spawn. It expects\nfull file paths to the executable you want to execute. If they are in your `$PATH`,\nthey can be retrieved by calling `System.find_executable/1`:\n\n    iex> path = System.find_executable(\"echo\")\n    iex> port = Port.open({:spawn_executable, path}, [:binary, args: [\"hello world\"]])\n    iex> flush()\n    {#Port , {:data, \"hello world\\n\"}}\n\nWhen using `:spawn_executable`, the list of arguments can be passed via\nthe `:args` option as done above. For the full list of options, see the\ndocumentation for the Erlang function `:erlang.open_port/2`.\n\n#","ref":"Port.html#module-spawn_executable"},{"type":"module","title":"fd - Port","doc":"The `:fd` name option allows developers to access `in` and `out` file\ndescriptors used by the Erlang VM. You would use those only if you are\nreimplementing core part of the Runtime System, such as the `:user` and\n`:shell` processes.","ref":"Port.html#module-fd"},{"type":"module","title":"Zombie operating system processes - Port","doc":"A port can be closed via the `close/1` function or by sending a `{pid, :close}`\nmessage. However, if the VM crashes, a long-running program started by the port\nwill have its stdin and stdout channels closed but **it won't be automatically\nterminated**.\n\nWhile most Unix command line tools will exit once its communication channels\nare closed, not all command line applications will do so. You can easily check\nthis by starting the port and then shutting down the VM and inspecting your\noperating system to see if the port process is still running.\n\nWhile we encourage graceful termination by detecting if stdin/stdout has been\nclosed, we do not always have control over how third-party software terminates.\nIn those cases, you can wrap the application in a script that checks for stdin.\nHere is such script that has been verified to work on bash shells:\n\n    #!/usr/bin/env bash\n\n    # Start the program in the background\n    exec \"$@\" &\n    pid1=$!\n\n    # Silence warnings from here on\n    exec >/dev/null 2>&1\n\n    # Read from stdin in the background and\n    # kill running program when stdin closes\n    exec 0<&0 $(\n      while read; do :; done\n      kill -KILL $pid1\n    ) &\n    pid2=$!\n\n    # Clean up\n    wait $pid1\n    ret=$?\n    kill -KILL $pid2\n    exit $ret\n\nNote the program above hijacks stdin, so you won't be able to communicate\nwith the underlying software via stdin (on the positive side, software that\nreads from stdin typically terminates when stdin closes).\n\nNow instead of:\n\n    Port.open(\n      {:spawn_executable, \"/path/to/program\"},\n      args: [\"a\", \"b\", \"c\"]\n    )\n\nYou may invoke:\n\n    Port.open(\n      {:spawn_executable, \"/path/to/wrapper\"},\n      args: [\"/path/to/program\", \"a\", \"b\", \"c\"]\n    )","ref":"Port.html#module-zombie-operating-system-processes"},{"type":"function","title":"Port.close/1","doc":"Closes the `port`.\n\nFor more information, see `:erlang.port_close/1`.\n\nInlined by the compiler.","ref":"Port.html#close/1"},{"type":"function","title":"Port.command/3","doc":"Sends `data` to the port driver `port`.\n\nFor more information, see `:erlang.port_command/3`.\n\nInlined by the compiler.","ref":"Port.html#command/3"},{"type":"function","title":"Port.connect/2","doc":"Associates the `port` identifier with a `pid`.\n\nFor more information, see `:erlang.port_connect/2`.\n\nInlined by the compiler.","ref":"Port.html#connect/2"},{"type":"function","title":"Port.demonitor/2","doc":"Demonitors the monitor identified by the given `reference`.\n\nIf `monitor_ref` is a reference which the calling process\nobtained by calling `monitor/1`, that monitoring is turned off.\nIf the monitoring is already turned off, nothing happens.\n\nSee `:erlang.demonitor/2` for more information.\n\nInlined by the compiler.","ref":"Port.html#demonitor/2"},{"type":"function","title":"Port.info/1","doc":"Returns information about the `port` (or `nil` if the port is closed).\n\nFor more information, see `:erlang.port_info/1`.","ref":"Port.html#info/1"},{"type":"function","title":"Port.info/2","doc":"Returns information about a specific field within\nthe `port` (or `nil` if the port is closed).\n\nFor more information, see `:erlang.port_info/2`.","ref":"Port.html#info/2"},{"type":"function","title":"Port.list/0","doc":"Returns a list of all ports in the current node.\n\nInlined by the compiler.","ref":"Port.html#list/0"},{"type":"function","title":"Port.monitor/1","doc":"Starts monitoring the given `port` from the calling process.\n\nOnce the monitored port process dies, a message is delivered to the\nmonitoring process in the shape of:\n\n    {:DOWN, ref, :port, object, reason}\n\nwhere:\n\n  * `ref` is a monitor reference returned by this function;\n  * `object` is either the `port` being monitored (when monitoring by port ID)\n  or `{name, node}` (when monitoring by a port name);\n  * `reason` is the exit reason.\n\nSee `:erlang.monitor/2` for more information.\n\nInlined by the compiler.","ref":"Port.html#monitor/1"},{"type":"function","title":"Port.open/2","doc":"Opens a port given a tuple `name` and a list of `options`.\n\nThe module documentation above contains documentation and examples\nfor the supported `name` values, summarized below:\n\n  * `{:spawn, command}` - runs an external program. `command` must contain\n    the program name and optionally a list of arguments separated by space.\n    If passing programs or arguments with space in their name, use the next option.\n  * `{:spawn_executable, filename}` - runs the executable given by the absolute\n    file name `filename`. Arguments can be passed via the `:args` option.\n  * `{:spawn_driver, command}` - spawns so-called port drivers.\n  * `{:fd, fd_in, fd_out}` - accesses file descriptors, `fd_in` and `fd_out`\n    opened by the VM.\n\nFor more information and the list of options, see `:erlang.open_port/2`.\n\nInlined by the compiler.","ref":"Port.html#open/2"},{"type":"type","title":"Port.name/0","doc":"","ref":"Port.html#t:name/0"},{"type":"module","title":"StringIO","doc":"Controls an IO device process that wraps a string.\n\nA `StringIO` IO device can be passed as a \"device\" to\nmost of the functions in the `IO` module.","ref":"StringIO.html"},{"type":"module","title":"Examples - StringIO","doc":"iex> {:ok, pid} = StringIO.open(\"foo\")\n    iex> IO.read(pid, 2)\n    \"fo\"","ref":"StringIO.html#module-examples"},{"type":"function","title":"StringIO.close/1","doc":"Stops the IO device and returns the remaining input/output\nbuffers.","ref":"StringIO.html#close/1"},{"type":"function","title":"Examples - StringIO.close/1","doc":"iex> {:ok, pid} = StringIO.open(\"in\")\n    iex> IO.write(pid, \"out\")\n    iex> StringIO.close(pid)\n    {:ok, {\"in\", \"out\"}}","ref":"StringIO.html#close/1-examples"},{"type":"function","title":"StringIO.contents/1","doc":"Returns the current input/output buffers for the given IO\ndevice.","ref":"StringIO.html#contents/1"},{"type":"function","title":"Examples - StringIO.contents/1","doc":"iex> {:ok, pid} = StringIO.open(\"in\")\n    iex> IO.write(pid, \"out\")\n    iex> StringIO.contents(pid)\n    {\"in\", \"out\"}","ref":"StringIO.html#contents/1-examples"},{"type":"function","title":"StringIO.flush/1","doc":"Flushes the output buffer and returns its current contents.","ref":"StringIO.html#flush/1"},{"type":"function","title":"Examples - StringIO.flush/1","doc":"iex> {:ok, pid} = StringIO.open(\"in\")\n    iex> IO.write(pid, \"out\")\n    iex> StringIO.flush(pid)\n    \"out\"\n    iex> StringIO.contents(pid)\n    {\"in\", \"\"}","ref":"StringIO.html#flush/1-examples"},{"type":"function","title":"StringIO.open/2","doc":"Creates an IO device.\n\n`string` will be the initial input of the newly created\ndevice.\n\n`options_or_function` can be a keyword list of options or\na function.\n\nIf options are provided, the result will be `{:ok, pid}`, returning the\nIO device created. The option `:capture_prompt`, when set to `true`, causes\nprompts (which are specified as arguments to `IO.get*` functions) to be\nincluded in the device's output.\n\nIf a function is provided, the device will be created and sent to the\nfunction. When the function returns, the device will be closed. The final\nresult will be a tuple with `:ok` and the result of the function.","ref":"StringIO.html#open/2"},{"type":"function","title":"Examples - StringIO.open/2","doc":"iex> {:ok, pid} = StringIO.open(\"foo\")\n    iex> IO.gets(pid, \">\")\n    \"foo\"\n    iex> StringIO.contents(pid)\n    {\"\", \"\"}\n\n    iex> {:ok, pid} = StringIO.open(\"foo\", capture_prompt: true)\n    iex> IO.gets(pid, \">\")\n    \"foo\"\n    iex> StringIO.contents(pid)\n    {\"\", \">\"}\n\n    iex> StringIO.open(\"foo\", fn pid ->\n    ...>   input = IO.gets(pid, \">\")\n    ...>   IO.write(pid, \"The input was #{input}\")\n    ...>   StringIO.contents(pid)\n    ...> end)\n    {:ok, {\"\", \"The input was foo\"}}","ref":"StringIO.html#open/2-examples"},{"type":"function","title":"StringIO.open/3","doc":"Creates an IO device.\n\n`string` will be the initial input of the newly created\ndevice.\n\nThe device will be created and sent to the function given.\nWhen the function returns, the device will be closed. The final\nresult will be a tuple with `:ok` and the result of the function.","ref":"StringIO.html#open/3"},{"type":"function","title":"Options - StringIO.open/3","doc":"* `:capture_prompt` - if set to `true`, prompts (specified as\n    arguments to `IO.get*` functions) are captured in the output.\n    Defaults to `false`.\n\n  * `:encoding` (since v1.10.0) - encoding of the IO device. Allowed\n    values are `:unicode` (default) and `:latin1`.","ref":"StringIO.html#open/3-options"},{"type":"function","title":"Examples - StringIO.open/3","doc":"iex> StringIO.open(\"foo\", [], fn pid ->\n    ...>   input = IO.gets(pid, \">\")\n    ...>   IO.write(pid, \"The input was #{input}\")\n    ...>   StringIO.contents(pid)\n    ...> end)\n    {:ok, {\"\", \"The input was foo\"}}\n\n    iex> StringIO.open(\"foo\", [capture_prompt: true], fn pid ->\n    ...>   input = IO.gets(pid, \">\")\n    ...>   IO.write(pid, \"The input was #{input}\")\n    ...>   StringIO.contents(pid)\n    ...> end)\n    {:ok, {\"\", \">The input was foo\"}}","ref":"StringIO.html#open/3-examples"},{"type":"module","title":"System","doc":"The `System` module provides functions that interact directly\nwith the VM or the host system.","ref":"System.html"},{"type":"module","title":"Time - System","doc":"The `System` module also provides functions that work with time,\nreturning different times kept by the system with support for\ndifferent time units.\n\nOne of the complexities in relying on system times is that they\nmay be adjusted. For example, when you enter and leave daylight\nsaving time, the system clock will be adjusted, often adding\nor removing one hour. We call such changes \"time warps\". In\norder to understand how such changes may be harmful, imagine\nthe following code:","ref":"System.html#module-time"},{"type":"module","title":"DO NOT DO THIS - System","doc":"prev = System.os_time()\n    # ... execute some code ...\n    next = System.os_time()\n    diff = next - prev\n\nIf, while the code is executing, the system clock changes,\nsome code that executed in 1 second may be reported as taking\nover 1 hour! To address such concerns, the VM provides a\nmonotonic time via `System.monotonic_time/0` which never\ndecreases and does not leap:","ref":"System.html#module-do-not-do-this"},{"type":"module","title":"DO THIS - System","doc":"prev = System.monotonic_time()\n    # ... execute some code ...\n    next = System.monotonic_time()\n    diff = next - prev\n\nGenerally speaking, the VM provides three time measurements:\n\n  * `os_time/0` - the time reported by the operating system (OS). This time may be\n    adjusted forwards or backwards in time with no limitation;\n\n  * `system_time/0` - the VM view of the `os_time/0`. The system time and operating\n    system time may not match in case of time warps although the VM works towards\n    aligning them. This time is not monotonic (i.e., it may decrease)\n    as its behaviour is configured [by the VM time warp\n    mode](https://www.erlang.org/doc/apps/erts/time_correction.html#Time_Warp_Modes);\n\n  * `monotonic_time/0` - a monotonically increasing time provided\n    by the Erlang VM. This is not strictly monotonically increasing. Multiple\n    sequential calls of the function may return the same value.\n\nThe time functions in this module work in the `:native` unit\n(unless specified otherwise), which is operating system dependent. Most of\nthe time, all calculations are done in the `:native` unit, to\navoid loss of precision, with `convert_time_unit/3` being\ninvoked at the end to convert to a specific time unit like\n`:millisecond` or `:microsecond`. See the `t:time_unit/0` type for\nmore information.\n\nFor a more complete rundown on the VM support for different\ntimes, see the [chapter on time and time\ncorrection](https://www.erlang.org/doc/apps/erts/time_correction.html)\nin the Erlang docs.","ref":"System.html#module-do-this"},{"type":"function","title":"System.argv/0","doc":"Lists command line arguments.\n\nReturns the list of command line arguments passed to the program.","ref":"System.html#argv/0"},{"type":"function","title":"System.argv/1","doc":"Modifies command line arguments.\n\nChanges the list of command line arguments. Use it with caution,\nas it destroys any previous argv information.","ref":"System.html#argv/1"},{"type":"function","title":"System.at_exit/1","doc":"Registers a program exit handler function.\n\nRegisters a function that will be invoked at the end of an Elixir script.\nA script is typically started via the command line via the `elixir` and\n`mix` executables.\n\nThe handler always executes in a different process from the one it was\nregistered in. As a consequence, any resources managed by the calling process\n(ETS tables, open files, and others) won't be available by the time the handler\nfunction is invoked.\n\nThe function must receive the exit status code as an argument.\n\nIf the VM terminates programmatically, via `System.stop/1`, `System.halt/1`,\nor exit signals, the `at_exit/1` callbacks are not guaranteed to be executed.","ref":"System.html#at_exit/1"},{"type":"function","title":"System.build_info/0","doc":"Elixir build information.\n\nReturns a map with the Elixir version, the Erlang/OTP release it was compiled\nwith, a short Git revision hash and the date and time it was built.\n\nEvery value in the map is a string, and these are:\n\n  * `:build` - the Elixir version, short Git revision hash and\n    Erlang/OTP release it was compiled with\n  * `:date` - a string representation of the ISO8601 date and time it was built\n  * `:otp_release` - OTP release it was compiled with\n  * `:revision` - short Git revision hash. If Git was not available at building\n    time, it is set to `\"\"`\n  * `:version` - the Elixir version\n\nOne should not rely on the specific formats returned by each of those fields.\nInstead one should use specialized functions, such as `version/0` to retrieve\nthe Elixir version and `otp_release/0` to retrieve the Erlang/OTP release.","ref":"System.html#build_info/0"},{"type":"function","title":"Examples - System.build_info/0","doc":"iex> System.build_info()\n    %{\n      build: \"1.9.0-dev (772a00a0c) (compiled with Erlang/OTP 21)\",\n      date: \"2018-12-24T01:09:21Z\",\n      otp_release: \"21\",\n      revision: \"772a00a0c\",\n      version: \"1.9.0-dev\"\n    }","ref":"System.html#build_info/0-examples"},{"type":"function","title":"System.cmd/3","doc":"Executes the given `command` with `args`.\n\n`command` is expected to be an executable available in PATH\nunless an absolute path is given.\n\n`args` must be a list of binaries which the executable will receive\nas its arguments as is. This means that:\n\n  * environment variables will not be interpolated\n  * wildcard expansion will not happen (unless `Path.wildcard/2` is used\n    explicitly)\n  * arguments do not need to be escaped or quoted for shell safety\n\nThis function returns a tuple containing the collected result\nand the command exit status.\n\nInternally, this function uses a `Port` for interacting with the\noutside world. However, if you plan to run a long-running program,\nports guarantee stdin/stdout devices will be closed but it does not\nautomatically terminate the program. The documentation for the\n`Port` module describes this problem and possible solutions under\nthe \"Zombie processes\" section.","ref":"System.html#cmd/3"},{"type":"function","title":"Examples - System.cmd/3","doc":"iex> System.cmd(\"echo\", [\"hello\"])\n    {\"hello\\n\", 0}\n\n    iex> System.cmd(\"echo\", [\"hello\"], env: [{\"MIX_ENV\", \"test\"}])\n    {\"hello\\n\", 0}\n\nIf you want to stream the output to Standard IO as it arrives:\n\n    iex> System.cmd(\"echo\", [\"hello\"], into: IO.stream())\n    hello\n    {%IO.Stream{}, 0}\n\nIf you want to read lines:\n\n    iex> System.cmd(\"echo\", [\"hello\\nworld\"], into: [], lines: 1024)\n    {[\"hello\", \"world\"], 0}","ref":"System.html#cmd/3-examples"},{"type":"function","title":"Options - System.cmd/3","doc":"* `:into` - injects the result into the given collectable, defaults to `\"\"`\n\n  * `:lines` - (since v1.15.0) reads the output by lines instead of in bytes. It expects a\n    number of maximum bytes to buffer internally (1024 is a reasonable default).\n    The collectable will be called with each finished line (regardless of buffer\n    size) and without the EOL character\n\n  * `:cd` - the directory to run the command in\n\n  * `:env` - an enumerable of tuples containing environment key-value as\n    binary. The child process inherits all environment variables from its\n    parent process, the Elixir application, except those overwritten or\n    cleared using this option. Specify a value of `nil` to clear (unset) an\n    environment variable, which is useful for preventing credentials passed\n    to the application from leaking into child processes\n\n  * `:arg0` - sets the command arg0\n\n  * `:stderr_to_stdout` - redirects stderr to stdout when `true`\n\n  * `:parallelism` - when `true`, the VM will schedule port tasks to improve\n    parallelism in the system. If set to `false`, the VM will try to perform\n    commands immediately, improving latency at the expense of parallelism.\n    The default is `false`, and can be set on system startup by passing the \n    [`+spp`](https://www.erlang.org/doc/man/erl.html#+spp) flag to `--erl`. \n    Use `:erlang.system_info(:port_parallelism)` to check if enabled.","ref":"System.html#cmd/3-options"},{"type":"function","title":"Error reasons - System.cmd/3","doc":"If invalid arguments are given, `ArgumentError` is raised by\n`System.cmd/3`. `System.cmd/3` also expects a strict set of\noptions and will raise if unknown or invalid options are given.\n\nFurthermore, `System.cmd/3` may fail with one of the POSIX reasons\ndetailed below:\n\n  * `:system_limit` - all available ports in the Erlang emulator are in use\n\n  * `:enomem` - there was not enough memory to create the port\n\n  * `:eagain` - there are no more available operating system processes\n\n  * `:enametoolong` - the external command given was too long\n\n  * `:emfile` - there are no more available file descriptors\n    (for the operating system process that the Erlang emulator runs in)\n\n  * `:enfile` - the file table is full (for the entire operating system)\n\n  * `:eacces` - the command does not point to an executable file\n\n  * `:enoent` - the command does not point to an existing file","ref":"System.html#cmd/3-error-reasons"},{"type":"function","title":"Shell commands - System.cmd/3","doc":"If you desire to execute a trusted command inside a shell, with pipes,\nredirecting and so on, please check `shell/2`.","ref":"System.html#cmd/3-shell-commands"},{"type":"function","title":"System.compiled_endianness/0","doc":"Returns the endianness the system was compiled with.","ref":"System.html#compiled_endianness/0"},{"type":"function","title":"System.convert_time_unit/3","doc":"Converts `time` from time unit `from_unit` to time unit `to_unit`.\n\nThe result is rounded via the floor function.\n\n`convert_time_unit/3` accepts an additional time unit (other than the\nones in the `t:time_unit/0` type) called `:native`. `:native` is the time\nunit used by the Erlang runtime system. It's determined when the runtime\nstarts and stays the same until the runtime is stopped, but could differ\nthe next time the runtime is started on the same machine. For this reason,\nyou should use this function to convert `:native` time units to a predictable\nunit before you display them to humans.\n\nTo determine how many seconds the `:native` unit represents in your current\nruntime, you can call this function to convert 1 second to the `:native`\ntime unit: `System.convert_time_unit(1, :second, :native)`.","ref":"System.html#convert_time_unit/3"},{"type":"function","title":"System.cwd/0","doc":"Current working directory.\n\nReturns the current working directory or `nil` if one\nis not available.","ref":"System.html#cwd/0"},{"type":"function","title":"System.cwd!/0","doc":"Current working directory, exception on error.\n\nReturns the current working directory or raises `RuntimeError`.","ref":"System.html#cwd!/0"},{"type":"function","title":"System.delete_env/1","doc":"Deletes an environment variable.\n\nRemoves the variable `varname` from the environment.","ref":"System.html#delete_env/1"},{"type":"function","title":"System.endianness/0","doc":"Returns the endianness.","ref":"System.html#endianness/0"},{"type":"function","title":"System.fetch_env/1","doc":"Returns the value of the given environment variable or `:error` if not found.\n\nIf the environment variable `varname` is set, then `{:ok, value}` is returned\nwhere `value` is a string. If `varname` is not set, `:error` is returned.","ref":"System.html#fetch_env/1"},{"type":"function","title":"Examples - System.fetch_env/1","doc":"iex> System.fetch_env(\"PORT\")\n    {:ok, \"4000\"}\n\n    iex> System.fetch_env(\"NOT_SET\")\n    :error","ref":"System.html#fetch_env/1-examples"},{"type":"function","title":"System.fetch_env!/1","doc":"Returns the value of the given environment variable or raises if not found.\n\nSame as `get_env/1` but raises instead of returning `nil` when the variable is\nnot set.","ref":"System.html#fetch_env!/1"},{"type":"function","title":"Examples - System.fetch_env!/1","doc":"iex> System.fetch_env!(\"PORT\")\n    \"4000\"\n\n    iex> System.fetch_env!(\"NOT_SET\")\n    ** (System.EnvError) could not fetch environment variable \"NOT_SET\" because it is not set","ref":"System.html#fetch_env!/1-examples"},{"type":"function","title":"System.find_executable/1","doc":"Locates an executable on the system.\n\nThis function looks up an executable program given\nits name using the environment variable PATH on Windows and Unix-like\noperating systems. It also considers the proper executable\nextension for each operating system, so for Windows it will try to\nlookup files with `.com`, `.cmd` or similar extensions.","ref":"System.html#find_executable/1"},{"type":"function","title":"System.get_env/0","doc":"Returns all system environment variables.\n\nThe returned value is a map containing name-value pairs.\nVariable names and their values are strings.","ref":"System.html#get_env/0"},{"type":"function","title":"System.get_env/2","doc":"Returns the value of the given environment variable.\n\nThe returned value of the environment variable\n`varname` is a string. If the environment variable\nis not set, returns the string specified in `default` or\n`nil` if none is specified.","ref":"System.html#get_env/2"},{"type":"function","title":"Examples - System.get_env/2","doc":"iex> System.get_env(\"PORT\")\n    \"4000\"\n\n    iex> System.get_env(\"NOT_SET\")\n    nil\n\n    iex> System.get_env(\"NOT_SET\", \"4001\")\n    \"4001\"","ref":"System.html#get_env/2-examples"},{"type":"function","title":"System.get_pid/0","doc":"Erlang VM process identifier.\n\nReturns the process identifier of the current Erlang emulator\nin the format most commonly used by the operating system environment.\n\nFor more information, see `:os.getpid/0`.","ref":"System.html#get_pid/0"},{"type":"function","title":"System.halt/1","doc":"Immediately halts the Erlang runtime system.\n\nTerminates the Erlang runtime system without properly shutting down\napplications and ports. Please see `stop/1` for a careful shutdown of the\nsystem.\n\n`status` must be a non-negative integer, the atom `:abort` or a binary.\n\n  * If an integer, the runtime system exits with the integer value which\n    is returned to the operating system.\n\n  * If `:abort`, the runtime system aborts producing a core dump, if that is\n    enabled in the operating system.\n\n  * If a string, an Erlang crash dump is produced with status as slogan,\n    and then the runtime system exits with status code 1.\n\nNote that on many platforms, only the status codes 0-255 are supported\nby the operating system.\n\nFor more information, see `:erlang.halt/1`.","ref":"System.html#halt/1"},{"type":"function","title":"Examples - System.halt/1","doc":"System.halt(0)\n    System.halt(1)\n    System.halt(:abort)","ref":"System.html#halt/1-examples"},{"type":"function","title":"System.monotonic_time/0","doc":"Returns the current monotonic time in the `:native` time unit.\n\nThis time is monotonically increasing and starts in an unspecified\npoint in time. This is not strictly monotonically increasing. Multiple\nsequential calls of the function may return the same value.\n\nInlined by the compiler.","ref":"System.html#monotonic_time/0"},{"type":"function","title":"System.monotonic_time/1","doc":"Returns the current monotonic time in the given time unit.\n\nThis time is monotonically increasing and starts in an unspecified\npoint in time.","ref":"System.html#monotonic_time/1"},{"type":"function","title":"System.no_halt/0","doc":"Checks if the system will halt or not at the end of ARGV processing.","ref":"System.html#no_halt/0"},{"type":"function","title":"System.no_halt/1","doc":"Marks if the system should halt or not at the end of ARGV processing.","ref":"System.html#no_halt/1"},{"type":"function","title":"System.os_time/0","doc":"Returns the current operating system (OS) time.\n\nThe result is returned in the `:native` time unit.\n\nThis time may be adjusted forwards or backwards in time\nwith no limitation and is not monotonic.\n\nInlined by the compiler.","ref":"System.html#os_time/0"},{"type":"function","title":"System.os_time/1","doc":"Returns the current operating system (OS) time in the given time `unit`.\n\nThis time may be adjusted forwards or backwards in time\nwith no limitation and is not monotonic.","ref":"System.html#os_time/1"},{"type":"function","title":"System.otp_release/0","doc":"Returns the Erlang/OTP release number.","ref":"System.html#otp_release/0"},{"type":"function","title":"System.pid/0","doc":"Returns the operating system PID for the current Erlang runtime system instance.\n\nReturns a string containing the (usually) numerical identifier for a process.\nOn Unix-like operating systems, this is typically the return value of the `getpid()` system call.\nOn Windows, the process ID as returned by the `GetCurrentProcessId()` system\ncall is used.","ref":"System.html#pid/0"},{"type":"function","title":"Examples - System.pid/0","doc":"System.pid()","ref":"System.html#pid/0-examples"},{"type":"function","title":"System.put_env/1","doc":"Sets multiple environment variables.\n\nSets a new value for each environment variable corresponding\nto each `{key, value}` pair in `enum`. Keys are automatically\nconverted to strings, values are sent as is. `nil` values erase\nthe given keys.","ref":"System.html#put_env/1"},{"type":"function","title":"System.put_env/2","doc":"Sets an environment variable value.\n\nSets a new `value` for the environment variable `varname`.","ref":"System.html#put_env/2"},{"type":"function","title":"System.restart/0","doc":"Restarts all applications in the Erlang runtime system.\n\nAll applications are taken down smoothly, all code is unloaded, and all ports\nare closed before the system starts all applications once again.","ref":"System.html#restart/0"},{"type":"function","title":"Examples - System.restart/0","doc":"System.restart()","ref":"System.html#restart/0-examples"},{"type":"function","title":"System.schedulers/0","doc":"Returns the number of schedulers in the VM.","ref":"System.html#schedulers/0"},{"type":"function","title":"System.schedulers_online/0","doc":"Returns the number of schedulers online in the VM.","ref":"System.html#schedulers_online/0"},{"type":"function","title":"System.shell/2","doc":"Executes the given `command` in the OS shell.\n\nIt uses `sh` for Unix-like systems and `cmd` for Windows.\n\n> ##","ref":"System.html#shell/2"},{"type":"function","title":"Watch out {: .warning} - System.shell/2","doc":">\n> Use this function with care. In particular, **never\n> pass untrusted user input to this function**, as the user would be\n> able to perform \"command injection attacks\" by executing any code\n> directly on the machine. Generally speaking, prefer to use `cmd/3`\n> over this function.","ref":"System.html#shell/2-watch-out-warning"},{"type":"function","title":"Examples - System.shell/2","doc":"iex> System.shell(\"echo hello\")\n    {\"hello\\n\", 0}\n\nIf you want to stream the output to Standard IO as it arrives:\n\n    iex> System.shell(\"echo hello\", into: IO.stream())\n    hello\n    {%IO.Stream{}, 0}","ref":"System.html#shell/2-examples"},{"type":"function","title":"Options - System.shell/2","doc":"It accepts the same options as `cmd/3` (except for `arg0`).\nIt also accepts the following exclusive options:\n\n  * `:close_stdin` (since v1.14.1) - if the stdin should be closed\n    on Unix systems, forcing any command that waits on stdin to\n    immediately terminate. Defaults to false.","ref":"System.html#shell/2-options"},{"type":"function","title":"System.stacktrace/0","doc":"Deprecated mechanism to retrieve the last exception stacktrace.\n\nIt always return an empty list.","ref":"System.html#stacktrace/0"},{"type":"function","title":"System.stop/1","doc":"Asynchronously and carefully stops the Erlang runtime system.\n\nAll applications are taken down smoothly, all code is unloaded, and all ports\nare closed before the system terminates by calling `halt/1`.\n\n`status` must be a non-negative integer or a binary.\n\n  * If an integer, the runtime system exits with the integer value which is\n    returned to the operating system. On many platforms, only the status codes\n    0-255 are supported by the operating system.\n\n  * If a binary, an Erlang crash dump is produced with status as slogan, and\n    then the runtime system exits with status code 1.\n\nNote this function is asynchronous and the current process will continue\nexecuting after this function is invoked. In case you want to block the\ncurrent process until the system effectively shuts down, you can invoke\n`Process.sleep(:infinity)`.","ref":"System.html#stop/1"},{"type":"function","title":"Examples - System.stop/1","doc":"System.stop(0)\n    System.stop(1)","ref":"System.html#stop/1-examples"},{"type":"function","title":"System.system_time/0","doc":"Returns the current system time in the `:native` time unit.\n\nIt is the VM view of the `os_time/0`. They may not match in\ncase of time warps although the VM works towards aligning\nthem. This time is not monotonic.\n\nInlined by the compiler.","ref":"System.html#system_time/0"},{"type":"function","title":"System.system_time/1","doc":"Returns the current system time in the given time unit.\n\nIt is the VM view of the `os_time/0`. They may not match in\ncase of time warps although the VM works towards aligning\nthem. This time is not monotonic.","ref":"System.html#system_time/1"},{"type":"function","title":"System.time_offset/0","doc":"Returns the current time offset between the Erlang VM monotonic\ntime and the Erlang VM system time.\n\nThe result is returned in the `:native` time unit.\n\nSee `time_offset/1` for more information.\n\nInlined by the compiler.","ref":"System.html#time_offset/0"},{"type":"function","title":"System.time_offset/1","doc":"Returns the current time offset between the Erlang VM monotonic\ntime and the Erlang VM system time.\n\nThe result is returned in the given time unit `unit`. The returned\noffset, added to an Erlang monotonic time (for instance, one obtained with\n`monotonic_time/1`), gives the Erlang system time that corresponds\nto that monotonic time.","ref":"System.html#time_offset/1"},{"type":"function","title":"System.tmp_dir/0","doc":"Writable temporary directory.\n\nReturns a writable temporary directory.\nSearches for directories in the following order:\n\n  1. the directory named by the TMPDIR environment variable\n  2. the directory named by the TEMP environment variable\n  3. the directory named by the TMP environment variable\n  4. `C:\\TMP` on Windows or `/tmp` on Unix-like operating systems\n  5. as a last resort, the current working directory\n\nReturns `nil` if none of the above are writable.","ref":"System.html#tmp_dir/0"},{"type":"function","title":"System.tmp_dir!/0","doc":"Writable temporary directory, exception on error.\n\nSame as `tmp_dir/0` but raises `RuntimeError`\ninstead of returning `nil` if no temp dir is set.","ref":"System.html#tmp_dir!/0"},{"type":"function","title":"System.trap_signal/3","doc":"Traps the given `signal` to execute the `fun`.\n\n> ##","ref":"System.html#trap_signal/3"},{"type":"function","title":"Avoid setting traps in libraries {: .warning} - System.trap_signal/3","doc":">\n> Trapping signals may have strong implications\n> on how a system shuts down and behaves in production and\n> therefore it is extremely discouraged for libraries to\n> set their own traps. Instead, they should redirect users\n> to configure them themselves. The only cases where it is\n> acceptable for libraries to set their own traps is when\n> using Elixir in script mode, such as in `.exs` files and\n> via Mix tasks.\n\nAn optional `id` that uniquely identifies the function\ncan be given, otherwise a unique one is automatically\ngenerated. If a previously registered `id` is given,\nthis function returns an error tuple. The `id` can be\nused to remove a registered signal by calling\n`untrap_signal/2`.\n\nThe given `fun` receives no arguments and it must return\n`:ok`.\n\nIt returns `{:ok, id}` in case of success,\n`{:error, :already_registered}` in case the id has already\nbeen registered for the given signal, or `{:error, :not_sup}`\nin case trapping exists is not supported by the current OS.\n\nThe first time a signal is trapped, it will override the\ndefault behaviour from the operating system. If the same\nsignal is trapped multiple times, subsequent functions\ngiven to `trap_signal` will execute *first*. In other\nwords, you can consider each function is prepended to\nthe signal handler.\n\nBy default, the Erlang VM register traps to the three\nsignals:\n\n  * `:sigstop` - gracefully shuts down the VM with `stop/0`\n  * `:sigquit` - halts the VM via `halt/0`\n  * `:sigusr1` - halts the VM via status code of 1\n\nTherefore, if you add traps to the signals above, the\ndefault behaviour above will be executed after all user\nsignals.","ref":"System.html#trap_signal/3-avoid-setting-traps-in-libraries-warning"},{"type":"function","title":"Implementation notes - System.trap_signal/3","doc":"All signals run from a single process. Therefore, blocking the\n`fun` will block subsequent traps. It is also not possible to add\nor remove traps from within a trap itself.\n\nInternally, this functionality is built on top of `:os.set_signal/2`.\nWhen you register a trap, Elixir automatically sets it to `:handle`\nand it reverts it back to `:default` once all traps are removed\n(except for `:sigquit`, `:sigterm`, and `:sigusr1` which are always\nhandled). If you or a library call `:os.set_signal/2` directly,\nit may disable Elixir traps (or Elixir may override your configuration).","ref":"System.html#trap_signal/3-implementation-notes"},{"type":"function","title":"System.unique_integer/1","doc":"Generates and returns an integer that is unique in the current runtime\ninstance.\n\n\"Unique\" means that this function, called with the same list of `modifiers`,\nwill never return the same integer more than once on the current runtime\ninstance.\n\nIf `modifiers` is `[]`, then a unique integer (that can be positive or negative) is returned.\nOther modifiers can be passed to change the properties of the returned integer:\n\n  * `:positive` - the returned integer is guaranteed to be positive.\n  * `:monotonic` - the returned integer is monotonically increasing. This\n    means that, on the same runtime instance (but even on different\n    processes), integers returned using the `:monotonic` modifier will always\n    be strictly less than integers returned by successive calls with the\n    `:monotonic` modifier.\n\nAll modifiers listed above can be combined; repeated modifiers in `modifiers`\nwill be ignored.\n\nInlined by the compiler.","ref":"System.html#unique_integer/1"},{"type":"function","title":"System.untrap_signal/2","doc":"Removes a previously registered `signal` with `id`.","ref":"System.html#untrap_signal/2"},{"type":"function","title":"System.user_home/0","doc":"User home directory.\n\nReturns the user home directory (platform independent).","ref":"System.html#user_home/0"},{"type":"function","title":"System.user_home!/0","doc":"User home directory, exception on error.\n\nSame as `user_home/0` but raises `RuntimeError`\ninstead of returning `nil` if no user home is set.","ref":"System.html#user_home!/0"},{"type":"function","title":"System.version/0","doc":"Elixir version information.\n\nReturns Elixir's version as binary.","ref":"System.html#version/0"},{"type":"function","title":"System.wait_until_booted/0","doc":"Waits until the system boots.\n\nCalling this function blocks until all of ARGV is processed.\nInside a release, this means the boot script and then ARGV\nhave been processed. This is only useful for those implementing\ncustom shells/consoles on top of Elixir.\n\nHowever, be careful to not invoke this command from within\nthe process that is processing the command line arguments,\nas doing so would lead to a deadlock.","ref":"System.html#wait_until_booted/0"},{"type":"type","title":"System.signal/0","doc":"","ref":"System.html#t:signal/0"},{"type":"type","title":"System.time_unit/0","doc":"The time unit to be passed to functions like `monotonic_time/1` and others.\n\nThe `:second`, `:millisecond`, `:microsecond` and `:nanosecond` time\nunits controls the return value of the functions that accept a time unit.\n\nA time unit can also be a strictly positive integer. In this case, it\nrepresents the \"parts per second\": the time will be returned in `1 /\nparts_per_second` seconds. For example, using the `:millisecond` time unit\nis equivalent to using `1000` as the time unit (as the time will be returned\nin 1/1000 seconds - milliseconds).","ref":"System.html#t:time_unit/0"},{"type":"behaviour","title":"Calendar","doc":"This module defines the responsibilities for working with\ncalendars, dates, times and datetimes in Elixir.\n\nIt defines types and the minimal implementation\nfor a calendar behaviour in Elixir. The goal of the calendar\nfeatures in Elixir is to provide a base for interoperability\nrather than a full-featured datetime API.\n\nFor the actual date, time and datetime structs, see `Date`,\n`Time`, `NaiveDateTime`, and `DateTime`.\n\nTypes for year, month, day, and more are *overspecified*.\nFor example, the `t:month/0` type is specified as an integer\ninstead of `1..12`. This is because different calendars may\nhave a different number of days per month.","ref":"Calendar.html"},{"type":"function","title":"Calendar.compatible_calendars?/2","doc":"Returns `true` if two calendars have the same moment of starting a new day,\n`false` otherwise.\n\nIf two calendars are not compatible, we can only convert datetimes and times\nbetween them. If they are compatible, this means that we can also convert\ndates as well as naive datetimes between them.","ref":"Calendar.html#compatible_calendars?/2"},{"type":"callback","title":"Calendar.date_to_string/3","doc":"Converts the date into a string according to the calendar.","ref":"Calendar.html#c:date_to_string/3"},{"type":"callback","title":"Calendar.datetime_to_string/11","doc":"Converts the datetime (with time zone) into a string according to the calendar.","ref":"Calendar.html#c:datetime_to_string/11"},{"type":"callback","title":"Calendar.day_of_era/3","doc":"Calculates the day and era from the given `year`, `month`, and `day`.","ref":"Calendar.html#c:day_of_era/3"},{"type":"callback","title":"Calendar.day_of_week/4","doc":"Calculates the day of the week from the given `year`, `month`, and `day`.\n\n`starting_on` represents the starting day of the week. All\ncalendars must support at least the `:default` value. They may\nalso support other values representing their days of the week.","ref":"Calendar.html#c:day_of_week/4"},{"type":"callback","title":"Calendar.day_of_year/3","doc":"Calculates the day of the year from the given `year`, `month`, and `day`.","ref":"Calendar.html#c:day_of_year/3"},{"type":"callback","title":"Calendar.day_rollover_relative_to_midnight_utc/0","doc":"Define the rollover moment for the calendar.\n\nThis is the moment, in your calendar, when the current day ends\nand the next day starts.\n\nThe result of this function is used to check if two calendars roll over at\nthe same time of day. If they do not, we can only convert datetimes and times\nbetween them. If they do, this means that we can also convert dates as well\nas naive datetimes between them.\n\nThis day fraction should be in its most simplified form possible, to make comparisons fast.","ref":"Calendar.html#c:day_rollover_relative_to_midnight_utc/0"},{"type":"callback","title":"Examples - Calendar.day_rollover_relative_to_midnight_utc/0","doc":"* If in your calendar a new day starts at midnight, return `{0, 1}`.\n  * If in your calendar a new day starts at sunrise, return `{1, 4}`.\n  * If in your calendar a new day starts at noon, return `{1, 2}`.\n  * If in your calendar a new day starts at sunset, return `{3, 4}`.","ref":"Calendar.html#c:day_rollover_relative_to_midnight_utc/0-examples"},{"type":"callback","title":"Calendar.days_in_month/2","doc":"Returns how many days there are in the given month of the given year.","ref":"Calendar.html#c:days_in_month/2"},{"type":"function","title":"Calendar.get_time_zone_database/0","doc":"Gets the current time zone database.","ref":"Calendar.html#get_time_zone_database/0"},{"type":"callback","title":"Calendar.iso_days_to_beginning_of_day/1","doc":"Converts the given `t:iso_days/0` to the first moment of the day.","ref":"Calendar.html#c:iso_days_to_beginning_of_day/1"},{"type":"callback","title":"Calendar.iso_days_to_end_of_day/1","doc":"Converts the given `t:iso_days/0` to the last moment of the day.","ref":"Calendar.html#c:iso_days_to_end_of_day/1"},{"type":"callback","title":"Calendar.leap_year?/1","doc":"Returns `true` if the given year is a leap year.\n\nA leap year is a year of a longer length than normal. The exact meaning\nis up to the calendar. A calendar must return `false` if it does not support\nthe concept of leap years.","ref":"Calendar.html#c:leap_year?/1"},{"type":"callback","title":"Calendar.months_in_year/1","doc":"Returns how many months there are in the given year.","ref":"Calendar.html#c:months_in_year/1"},{"type":"callback","title":"Calendar.naive_datetime_from_iso_days/1","doc":"Converts `t:iso_days/0` to the calendar's datetime format.","ref":"Calendar.html#c:naive_datetime_from_iso_days/1"},{"type":"callback","title":"Calendar.naive_datetime_to_iso_days/7","doc":"Converts the datetime (without time zone) into the `t:iso_days/0` format.","ref":"Calendar.html#c:naive_datetime_to_iso_days/7"},{"type":"callback","title":"Calendar.naive_datetime_to_string/7","doc":"Converts the naive datetime (without time zone) into a string according to the calendar.","ref":"Calendar.html#c:naive_datetime_to_string/7"},{"type":"callback","title":"Calendar.parse_date/1","doc":"Parses the string representation for a date returned by `c:date_to_string/3`\ninto a date tuple.","ref":"Calendar.html#c:parse_date/1"},{"type":"callback","title":"Calendar.parse_naive_datetime/1","doc":"Parses the string representation for a naive datetime returned by\n`c:naive_datetime_to_string/7` into a naive datetime tuple.\n\nThe given string may contain a timezone offset but it is ignored.","ref":"Calendar.html#c:parse_naive_datetime/1"},{"type":"callback","title":"Calendar.parse_time/1","doc":"Parses the string representation for a time returned by `c:time_to_string/4`\ninto a time tuple.","ref":"Calendar.html#c:parse_time/1"},{"type":"callback","title":"Calendar.parse_utc_datetime/1","doc":"Parses the string representation for a datetime returned by\n`c:datetime_to_string/11` into a datetime tuple.\n\nThe returned datetime must be in UTC. The original `utc_offset`\nit was written in must be returned in the result.","ref":"Calendar.html#c:parse_utc_datetime/1"},{"type":"function","title":"Calendar.put_time_zone_database/1","doc":"Sets the current time zone database.","ref":"Calendar.html#put_time_zone_database/1"},{"type":"callback","title":"Calendar.quarter_of_year/3","doc":"Calculates the quarter of the year from the given `year`, `month`, and `day`.","ref":"Calendar.html#c:quarter_of_year/3"},{"type":"function","title":"Calendar.strftime/3","doc":"Formats the given date, time, or datetime into a string.\n\nThe datetime can be any of the `Calendar` types (`Time`, `Date`,\n`NaiveDateTime`, and `DateTime`) or any map, as long as they\ncontain all of the relevant fields necessary for formatting.\nFor example, if you use `%Y` to format the year, the datetime\nmust have the `:year` field. Therefore, if you pass a `Time`,\nor a map without the `:year` field to a format that expects `%Y`,\nan error will be raised.\n\nExamples of common usage:\n\n    iex> Calendar.strftime(~U[2019-08-26 13:52:06.0Z], \"%y-%m-%d %I:%M:%S %p\")\n    \"19-08-26 01:52:06 PM\"\n\n    iex> Calendar.strftime(~U[2019-08-26 13:52:06.0Z], \"%a, %B %d %Y\")\n    \"Mon, August 26 2019\"","ref":"Calendar.html#strftime/3"},{"type":"function","title":"User Options - Calendar.strftime/3","doc":"* `:preferred_datetime` - a string for the preferred format to show datetimes,\n    it can't contain the `%c` format and defaults to `\"%Y-%m-%d %H:%M:%S\"`\n    if the option is not received\n\n  * `:preferred_date` - a string for the preferred format to show dates,\n    it can't contain the `%x` format and defaults to `\"%Y-%m-%d\"`\n    if the option is not received\n\n  * `:preferred_time` - a string for the preferred format to show times,\n    it can't contain the `%X` format and defaults to `\"%H:%M:%S\"`\n    if the option is not received\n\n  * `:am_pm_names` - a function that receives either `:am` or `:pm` and returns\n    the name of the period of the day, if the option is not received it defaults\n    to a function that returns `\"am\"` and `\"pm\"`, respectively\n\n  *  `:month_names` - a function that receives a number and returns the name of\n    the corresponding month, if the option is not received it defaults to a\n    function that returns the month names in English\n\n  * `:abbreviated_month_names` - a function that receives a number and returns the\n    abbreviated name of the corresponding month, if the option is not received it\n    defaults to a function that returns the abbreviated month names in English\n\n  * `:day_of_week_names` - a function that receives a number and returns the name of\n    the corresponding day of week, if the option is not received it defaults to a\n    function that returns the day of week names in English\n\n  * `:abbreviated_day_of_week_names` - a function that receives a number and returns\n    the abbreviated name of the corresponding day of week, if the option is not received\n    it defaults to a function that returns the abbreviated day of week names in English","ref":"Calendar.html#strftime/3-user-options"},{"type":"function","title":"Formatting syntax - Calendar.strftime/3","doc":"The formatting syntax for the `string_format` argument is a sequence of characters in\nthe following format:\n\n    %   \n\nwhere:\n\n  * `%`: indicates the start of a formatted section\n  * ` `: set the padding (see below)\n  * ` `: a number indicating the minimum size of the formatted section\n  * ` `: the format itself (see below)\n\n#","ref":"Calendar.html#strftime/3-formatting-syntax"},{"type":"function","title":"Accepted padding options - Calendar.strftime/3","doc":"* `-`: no padding, removes all padding from the format\n  * `_`: pad with spaces\n  * `0`: pad with zeroes\n\n#","ref":"Calendar.html#strftime/3-accepted-padding-options"},{"type":"function","title":"Accepted string formats - Calendar.strftime/3","doc":"The accepted formats for `string_format` are:\n\nFormat | Description                                                             | Examples (in ISO)\n:----- | :-----------------------------------------------------------------------| :------------------------\na      | Abbreviated name of day                                                 | Mon\nA      | Full name of day                                                        | Monday\nb      | Abbreviated month name                                                  | Jan\nB      | Full month name                                                         | January\nc      | Preferred date+time representation                                      | 2018-10-17 12:34:56\nd      | Day of the month                                                        | 01, 31\nf      | Microseconds *(does not support width and padding modifiers)*           | 000000, 999999, 0123\nH      | Hour using a 24-hour clock                                              | 00, 23\nI      | Hour using a 12-hour clock                                              | 01, 12\nj      | Day of the year                                                         | 001, 366\nm      | Month                                                                   | 01, 12\nM      | Minute                                                                  | 00, 59\np      | \"AM\" or \"PM\" (noon is \"PM\", midnight as \"AM\")                           | AM, PM\nP      | \"am\" or \"pm\" (noon is \"pm\", midnight as \"am\")                           | am, pm\nq      | Quarter                                                                 | 1, 2, 3, 4\ns      | Number of seconds since the Epoch, 1970-01-01 00:00:00+0000 (UTC)       | 1565888877\nS      | Second                                                                  | 00, 59, 60\nu      | Day of the week                                                         | 1 (Monday), 7 (Sunday)\nx      | Preferred date (without time) representation                            | 2018-10-17\nX      | Preferred time (without date) representation                            | 12:34:56\ny      | Year as 2-digits                                                        | 01, 01, 86, 18\nY      | Year                                                                    | -0001, 0001, 1986\nz      | +hhmm/-hhmm time zone offset from UTC (empty string if naive)           | +0300, -0530\nZ      | Time zone abbreviation (empty string if naive)                          | CET, BRST\n%      | Literal \"%\" character                                                   | %\n\nAny other character will be interpreted as an invalid format and raise an error.","ref":"Calendar.html#strftime/3-accepted-string-formats"},{"type":"function","title":"Examples - Calendar.strftime/3","doc":"Without user options:\n\n    iex> Calendar.strftime(~U[2019-08-26 13:52:06.0Z], \"%y-%m-%d %I:%M:%S %p\")\n    \"19-08-26 01:52:06 PM\"\n\n    iex> Calendar.strftime(~U[2019-08-26 13:52:06.0Z], \"%a, %B %d %Y\")\n    \"Mon, August 26 2019\"\n\n    iex> Calendar.strftime(~U[2020-04-02 13:52:06.0Z], \"%B %-d, %Y\")\n    \"April 2, 2020\"\n\n    iex> Calendar.strftime(~U[2019-08-26 13:52:06.0Z], \"%c\")\n    \"2019-08-26 13:52:06\"\n\nWith user options:\n\n    iex> Calendar.strftime(~U[2019-08-26 13:52:06.0Z], \"%c\", preferred_datetime: \"%H:%M:%S %d-%m-%y\")\n    \"13:52:06 26-08-19\"\n\n    iex> Calendar.strftime(\n    ...>  ~U[2019-08-26 13:52:06.0Z],\n    ...>  \"%A\",\n    ...>  day_of_week_names: fn day_of_week ->\n    ...>    {\"segunda-feira\", \"terça-feira\", \"quarta-feira\", \"quinta-feira\",\n    ...>    \"sexta-feira\", \"sábado\", \"domingo\"}\n    ...>    |> elem(day_of_week - 1)\n    ...>  end\n    ...>)\n    \"segunda-feira\"\n\n    iex> Calendar.strftime(\n    ...>  ~U[2019-08-26 13:52:06.0Z],\n    ...>  \"%B\",\n    ...>  month_names: fn month ->\n    ...>    {\"січень\", \"лютий\", \"березень\", \"квітень\", \"травень\", \"червень\",\n    ...>    \"липень\", \"серпень\", \"вересень\", \"жовтень\", \"листопад\", \"грудень\"}\n    ...>    |> elem(month - 1)\n    ...>  end\n    ...>)\n    \"серпень\"","ref":"Calendar.html#strftime/3-examples"},{"type":"callback","title":"Calendar.time_from_day_fraction/1","doc":"Converts `t:day_fraction/0` to the calendar's time format.","ref":"Calendar.html#c:time_from_day_fraction/1"},{"type":"callback","title":"Calendar.time_to_day_fraction/4","doc":"Converts the given time to the `t:day_fraction/0` format.","ref":"Calendar.html#c:time_to_day_fraction/4"},{"type":"callback","title":"Calendar.time_to_string/4","doc":"Converts the time into a string according to the calendar.","ref":"Calendar.html#c:time_to_string/4"},{"type":"function","title":"Calendar.truncate/2","doc":"Returns a microsecond tuple truncated to a given precision (`:microsecond`,\n`:millisecond`, or `:second`).","ref":"Calendar.html#truncate/2"},{"type":"callback","title":"Calendar.valid_date?/3","doc":"Should return `true` if the given date describes a proper date in the calendar.","ref":"Calendar.html#c:valid_date?/3"},{"type":"callback","title":"Calendar.valid_time?/4","doc":"Should return `true` if the given time describes a proper time in the calendar.","ref":"Calendar.html#c:valid_time?/4"},{"type":"callback","title":"Calendar.year_of_era/3","doc":"Calculates the year and era from the given `year`.","ref":"Calendar.html#c:year_of_era/3"},{"type":"type","title":"Calendar.calendar/0","doc":"A calendar implementation.","ref":"Calendar.html#t:calendar/0"},{"type":"type","title":"Calendar.date/0","doc":"Any map or struct that contains the date fields.","ref":"Calendar.html#t:date/0"},{"type":"type","title":"Calendar.datetime/0","doc":"Any map or struct that contains the datetime fields.","ref":"Calendar.html#t:datetime/0"},{"type":"type","title":"Calendar.day/0","doc":"","ref":"Calendar.html#t:day/0"},{"type":"type","title":"Calendar.day_fraction/0","doc":"The internal time format is used when converting between calendars.\n\nIt represents time as a fraction of a day (starting from midnight).\n`parts_in_day` specifies how much of the day is already passed,\nwhile `parts_per_day` signifies how many parts are there in a day.","ref":"Calendar.html#t:day_fraction/0"},{"type":"type","title":"Calendar.day_of_era/0","doc":"A tuple representing the `day` and the `era`.","ref":"Calendar.html#t:day_of_era/0"},{"type":"type","title":"Calendar.day_of_week/0","doc":"","ref":"Calendar.html#t:day_of_week/0"},{"type":"type","title":"Calendar.era/0","doc":"","ref":"Calendar.html#t:era/0"},{"type":"type","title":"Calendar.hour/0","doc":"","ref":"Calendar.html#t:hour/0"},{"type":"type","title":"Calendar.iso_days/0","doc":"The internal date format that is used when converting between calendars.\n\nThis is the number of days including the fractional part that has passed of\nthe last day since `0000-01-01+00:00T00:00.000000` in ISO 8601 notation (also\nknown as *midnight 1 January BC 1* of the proleptic Gregorian calendar).","ref":"Calendar.html#t:iso_days/0"},{"type":"type","title":"Calendar.microsecond/0","doc":"Microseconds with stored precision.\n\nThe precision represents the number of digits that must be used when\nrepresenting the microseconds to external format. If the precision is `0`,\nit means microseconds must be skipped.","ref":"Calendar.html#t:microsecond/0"},{"type":"type","title":"Calendar.minute/0","doc":"","ref":"Calendar.html#t:minute/0"},{"type":"type","title":"Calendar.month/0","doc":"","ref":"Calendar.html#t:month/0"},{"type":"type","title":"Calendar.naive_datetime/0","doc":"Any map or struct that contains the naive datetime fields.","ref":"Calendar.html#t:naive_datetime/0"},{"type":"type","title":"Calendar.second/0","doc":"","ref":"Calendar.html#t:second/0"},{"type":"type","title":"Calendar.std_offset/0","doc":"The time zone standard offset in seconds (typically not zero in summer times).\n\nIt must be added to `t:utc_offset/0` to get the total offset from UTC used for \"wall time\".","ref":"Calendar.html#t:std_offset/0"},{"type":"type","title":"Calendar.time/0","doc":"Any map or struct that contains the time fields.","ref":"Calendar.html#t:time/0"},{"type":"type","title":"Calendar.time_zone/0","doc":"The time zone ID according to the IANA tz database (for example, `Europe/Zurich`).","ref":"Calendar.html#t:time_zone/0"},{"type":"type","title":"Calendar.time_zone_database/0","doc":"Specifies the time zone database for calendar operations.\n\nMany functions in the `DateTime` module require a time zone database.\nBy default, this module uses the default time zone database returned by\n`Calendar.get_time_zone_database/0`, which defaults to\n`Calendar.UTCOnlyTimeZoneDatabase`. This database only handles `Etc/UTC`\ndatetimes and returns `{:error, :utc_only_time_zone_database}`\nfor any other time zone.\n\nOther time zone databases (including ones provided by packages)\ncan be configured as default either via configuration:\n\n    config :elixir, :time_zone_database, CustomTimeZoneDatabase\n\nor by calling `Calendar.put_time_zone_database/1`.\n\nSee `Calendar.TimeZoneDatabase` for more information on custom\ntime zone databases.","ref":"Calendar.html#t:time_zone_database/0"},{"type":"type","title":"Calendar.utc_offset/0","doc":"The time zone UTC offset in seconds for standard time.\n\nSee also `t:std_offset/0`.","ref":"Calendar.html#t:utc_offset/0"},{"type":"type","title":"Calendar.week/0","doc":"","ref":"Calendar.html#t:week/0"},{"type":"type","title":"Calendar.year/0","doc":"","ref":"Calendar.html#t:year/0"},{"type":"type","title":"Calendar.zone_abbr/0","doc":"The time zone abbreviation (for example, `CET` or `CEST` or `BST`).","ref":"Calendar.html#t:zone_abbr/0"},{"type":"module","title":"Calendar.ISO","doc":"The default calendar implementation, a Gregorian calendar following ISO 8601.\n\nThis calendar implements a proleptic Gregorian calendar and\nis therefore compatible with the calendar used in most countries\ntoday. The proleptic means the Gregorian rules for leap years are\napplied for all time, consequently the dates give different results\nbefore the year 1583 from when the Gregorian calendar was adopted.","ref":"Calendar.ISO.html"},{"type":"module","title":"ISO 8601 compliance - Calendar.ISO","doc":"The ISO 8601 specification is feature-rich, but allows applications\nto selectively implement most parts of it. The choices Elixir makes\nare catalogued below.\n\n#","ref":"Calendar.ISO.html#module-iso-8601-compliance"},{"type":"module","title":"Features - Calendar.ISO","doc":"The standard library supports a minimal set of possible ISO 8601 features.\nSpecifically, the parser only supports calendar dates and does not support\nordinal and week formats.\n\nBy default Elixir only parses extended-formatted date/times. You can opt-in\nto parse basic-formatted date/times.\n\n`NaiveDateTime.to_iso8601/2` and `DateTime.to_iso8601/2` allow you to produce\neither basic or extended formatted strings, and `Calendar.strftime/2` allows\nyou to format datetimes however else you desire.\n\nElixir does not support reduced accuracy formats (for example, a date without\nthe day component) nor decimal precisions in the lowest component (such as\n`10:01:25,5`). No functions exist to parse ISO 8601 durations or time intervals.\n\n##","ref":"Calendar.ISO.html#module-features"},{"type":"module","title":"Examples - Calendar.ISO","doc":"Elixir expects the extended format by default when parsing:\n\n    iex> Calendar.ISO.parse_naive_datetime(\"2015-01-23T23:50:07\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {0, 0}}}\n    iex> Calendar.ISO.parse_naive_datetime(\"20150123T235007\")\n    {:error, :invalid_format}\n\nParsing can be restricted to basic if desired:\n\n    iex> Calendar.ISO.parse_naive_datetime(\"20150123T235007Z\", :basic)\n    {:ok, {2015, 1, 23, 23, 50, 7, {0, 0}}}\n    iex> Calendar.ISO.parse_naive_datetime(\"20150123T235007Z\", :extended)\n    {:error, :invalid_format}\n\nOnly calendar dates are supported in parsing; ordinal and week dates are not.\n\n    iex> Calendar.ISO.parse_date(\"2015-04-15\")\n    {:ok, {2015, 4, 15}}\n    iex> Calendar.ISO.parse_date(\"2015-105\")\n    {:error, :invalid_format}\n    iex> Calendar.ISO.parse_date(\"2015-W16\")\n    {:error, :invalid_format}\n    iex> Calendar.ISO.parse_date(\"2015-W016-3\")\n    {:error, :invalid_format}\n\nYears, months, days, hours, minutes, and seconds must be fully specified:\n\n    iex> Calendar.ISO.parse_date(\"2015-04-15\")\n    {:ok, {2015, 4, 15}}\n    iex> Calendar.ISO.parse_date(\"2015-04\")\n    {:error, :invalid_format}\n    iex> Calendar.ISO.parse_date(\"2015\")\n    {:error, :invalid_format}\n\n    iex> Calendar.ISO.parse_time(\"23:50:07.0123456\")\n    {:ok, {23, 50, 7, {12345, 6}}}\n    iex> Calendar.ISO.parse_time(\"23:50:07\")\n    {:ok, {23, 50, 7, {0, 0}}}\n    iex> Calendar.ISO.parse_time(\"23:50\")\n    {:error, :invalid_format}\n    iex> Calendar.ISO.parse_time(\"23\")\n    {:error, :invalid_format}\n\n#","ref":"Calendar.ISO.html#module-examples"},{"type":"module","title":"Extensions - Calendar.ISO","doc":"The parser and formatter adopt one ISO 8601 extension: extended year notation.\n\nThis allows dates to be prefixed with a `+` or `-` sign, extending the range of\nexpressible years from the default (`0000..9999`) to `-9999..9999`. Elixir still\nrestricts years in this format to four digits.\n\n##","ref":"Calendar.ISO.html#module-extensions"},{"type":"module","title":"Examples - Calendar.ISO","doc":"iex> Calendar.ISO.parse_date(\"-2015-01-23\")\n    {:ok, {-2015, 1, 23}}\n    iex> Calendar.ISO.parse_date(\"+2015-01-23\")\n    {:ok, {2015, 1, 23}}\n\n    iex> Calendar.ISO.parse_naive_datetime(\"-2015-01-23 23:50:07\")\n    {:ok, {-2015, 1, 23, 23, 50, 7, {0, 0}}}\n    iex> Calendar.ISO.parse_naive_datetime(\"+2015-01-23 23:50:07\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {0, 0}}}\n\n    iex> Calendar.ISO.parse_utc_datetime(\"-2015-01-23 23:50:07Z\")\n    {:ok, {-2015, 1, 23, 23, 50, 7, {0, 0}}, 0}\n    iex> Calendar.ISO.parse_utc_datetime(\"+2015-01-23 23:50:07Z\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {0, 0}}, 0}\n\n#","ref":"Calendar.ISO.html#module-examples"},{"type":"module","title":"Additions - Calendar.ISO","doc":"ISO 8601 does not allow a whitespace instead of `T` as a separator\nbetween date and times, both when parsing and formatting.\nThis is a common enough representation, Elixir allows it during parsing.\n\nThe formatting of dates in `NaiveDateTime.to_iso8601/1` and `DateTime.to_iso8601/1`\ndo produce specification-compliant string representations using the `T` separator.\n\n##","ref":"Calendar.ISO.html#module-additions"},{"type":"module","title":"Examples - Calendar.ISO","doc":"iex> Calendar.ISO.parse_naive_datetime(\"2015-01-23 23:50:07.0123456\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {12345, 6}}}\n    iex> Calendar.ISO.parse_naive_datetime(\"2015-01-23T23:50:07.0123456\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {12345, 6}}}\n\n    iex> Calendar.ISO.parse_utc_datetime(\"2015-01-23 23:50:07.0123456Z\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {12345, 6}}, 0}\n    iex> Calendar.ISO.parse_utc_datetime(\"2015-01-23T23:50:07.0123456Z\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {12345, 6}}, 0}","ref":"Calendar.ISO.html#module-examples"},{"type":"function","title":"Calendar.ISO.date_to_string/4","doc":"Converts the given date into a string.\n\nBy default, returns dates formatted in the \"extended\" format,\nfor human readability. It also supports the \"basic\" format\nby passing the `:basic` option.","ref":"Calendar.ISO.html#date_to_string/4"},{"type":"function","title":"Examples - Calendar.ISO.date_to_string/4","doc":"iex> Calendar.ISO.date_to_string(2015, 2, 28)\n    \"2015-02-28\"\n    iex> Calendar.ISO.date_to_string(2017, 8, 1)\n    \"2017-08-01\"\n    iex> Calendar.ISO.date_to_string(-99, 1, 31)\n    \"-0099-01-31\"\n\n    iex> Calendar.ISO.date_to_string(2015, 2, 28, :basic)\n    \"20150228\"\n    iex> Calendar.ISO.date_to_string(-99, 1, 31, :basic)\n    \"-00990131\"","ref":"Calendar.ISO.html#date_to_string/4-examples"},{"type":"function","title":"Calendar.ISO.datetime_to_string/12","doc":"Converts the datetime (with time zone) into a string.\n\nBy default, returns datetimes formatted in the \"extended\" format,\nfor human readability. It also supports the \"basic\" format\nby passing the `:basic` option.","ref":"Calendar.ISO.html#datetime_to_string/12"},{"type":"function","title":"Examples - Calendar.ISO.datetime_to_string/12","doc":"iex> time_zone = \"Etc/UTC\"\n    iex> Calendar.ISO.datetime_to_string(2017, 8, 1, 1, 2, 3, {4, 5}, time_zone, \"UTC\", 0, 0)\n    \"2017-08-01 01:02:03.00000Z\"\n    iex> Calendar.ISO.datetime_to_string(2017, 8, 1, 1, 2, 3, {4, 5}, time_zone, \"UTC\", 3600, 0)\n    \"2017-08-01 01:02:03.00000+01:00\"\n    iex> Calendar.ISO.datetime_to_string(2017, 8, 1, 1, 2, 3, {4, 5}, time_zone, \"UTC\", 3600, 3600)\n    \"2017-08-01 01:02:03.00000+02:00\"\n\n    iex> time_zone = \"Europe/Berlin\"\n    iex> Calendar.ISO.datetime_to_string(2017, 8, 1, 1, 2, 3, {4, 5}, time_zone, \"CET\", 3600, 0)\n    \"2017-08-01 01:02:03.00000+01:00 CET Europe/Berlin\"\n    iex> Calendar.ISO.datetime_to_string(2017, 8, 1, 1, 2, 3, {4, 5}, time_zone, \"CDT\", 3600, 3600)\n    \"2017-08-01 01:02:03.00000+02:00 CDT Europe/Berlin\"\n\n    iex> time_zone = \"America/Los_Angeles\"\n    iex> Calendar.ISO.datetime_to_string(2015, 2, 28, 1, 2, 3, {4, 5}, time_zone, \"PST\", -28800, 0)\n    \"2015-02-28 01:02:03.00000-08:00 PST America/Los_Angeles\"\n    iex> Calendar.ISO.datetime_to_string(2015, 2, 28, 1, 2, 3, {4, 5}, time_zone, \"PDT\", -28800, 3600)\n    \"2015-02-28 01:02:03.00000-07:00 PDT America/Los_Angeles\"\n\n    iex> time_zone = \"Europe/Berlin\"\n    iex> Calendar.ISO.datetime_to_string(2017, 8, 1, 1, 2, 3, {4, 5}, time_zone, \"CET\", 3600, 0, :basic)\n    \"20170801 010203.00000+0100 CET Europe/Berlin\"","ref":"Calendar.ISO.html#datetime_to_string/12-examples"},{"type":"function","title":"Calendar.ISO.day_of_era/3","doc":"Calculates the day and era from the given `year`, `month`, and `day`.","ref":"Calendar.ISO.html#day_of_era/3"},{"type":"function","title":"Examples - Calendar.ISO.day_of_era/3","doc":"iex> Calendar.ISO.day_of_era(0, 1, 1)\n    {366, 0}\n    iex> Calendar.ISO.day_of_era(1, 1, 1)\n    {1, 1}\n    iex> Calendar.ISO.day_of_era(0, 12, 31)\n    {1, 0}\n    iex> Calendar.ISO.day_of_era(0, 12, 30)\n    {2, 0}\n    iex> Calendar.ISO.day_of_era(-1, 12, 31)\n    {367, 0}","ref":"Calendar.ISO.html#day_of_era/3-examples"},{"type":"function","title":"Calendar.ISO.day_of_week/4","doc":"Calculates the day of the week from the given `year`, `month`, and `day`.\n\nIt is an integer from 1 to 7, where 1 is the given `starting_on` weekday.\nFor example, if `starting_on` is set to `:monday`, then 1 is Monday and\n7 is Sunday.\n\n`starting_on` can also be `:default`, which is equivalent to `:monday`.","ref":"Calendar.ISO.html#day_of_week/4"},{"type":"function","title":"Examples - Calendar.ISO.day_of_week/4","doc":"iex> Calendar.ISO.day_of_week(2016, 10, 31, :monday)\n    {1, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 1, :monday)\n    {2, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 2, :monday)\n    {3, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 3, :monday)\n    {4, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 4, :monday)\n    {5, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 5, :monday)\n    {6, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 6, :monday)\n    {7, 1, 7}\n    iex> Calendar.ISO.day_of_week(-99, 1, 31, :monday)\n    {4, 1, 7}\n\n    iex> Calendar.ISO.day_of_week(2016, 10, 31, :sunday)\n    {2, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 1, :sunday)\n    {3, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 2, :sunday)\n    {4, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 3, :sunday)\n    {5, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 4, :sunday)\n    {6, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 5, :sunday)\n    {7, 1, 7}\n    iex> Calendar.ISO.day_of_week(2016, 11, 6, :sunday)\n    {1, 1, 7}\n    iex> Calendar.ISO.day_of_week(-99, 1, 31, :sunday)\n    {5, 1, 7}\n\n    iex> Calendar.ISO.day_of_week(2016, 10, 31, :saturday)\n    {3, 1, 7}","ref":"Calendar.ISO.html#day_of_week/4-examples"},{"type":"function","title":"Calendar.ISO.day_of_year/3","doc":"Calculates the day of the year from the given `year`, `month`, and `day`.\n\nIt is an integer from 1 to 366.","ref":"Calendar.ISO.html#day_of_year/3"},{"type":"function","title":"Examples - Calendar.ISO.day_of_year/3","doc":"iex> Calendar.ISO.day_of_year(2016, 1, 31)\n    31\n    iex> Calendar.ISO.day_of_year(-99, 2, 1)\n    32\n    iex> Calendar.ISO.day_of_year(2018, 2, 28)\n    59","ref":"Calendar.ISO.html#day_of_year/3-examples"},{"type":"function","title":"Calendar.ISO.day_rollover_relative_to_midnight_utc/0","doc":"See `c:Calendar.day_rollover_relative_to_midnight_utc/0` for documentation.","ref":"Calendar.ISO.html#day_rollover_relative_to_midnight_utc/0"},{"type":"function","title":"Calendar.ISO.days_in_month/2","doc":"Returns how many days there are in the given year-month.","ref":"Calendar.ISO.html#days_in_month/2"},{"type":"function","title":"Examples - Calendar.ISO.days_in_month/2","doc":"iex> Calendar.ISO.days_in_month(1900, 1)\n    31\n    iex> Calendar.ISO.days_in_month(1900, 2)\n    28\n    iex> Calendar.ISO.days_in_month(2000, 2)\n    29\n    iex> Calendar.ISO.days_in_month(2001, 2)\n    28\n    iex> Calendar.ISO.days_in_month(2004, 2)\n    29\n    iex> Calendar.ISO.days_in_month(2004, 4)\n    30\n    iex> Calendar.ISO.days_in_month(-1, 5)\n    31","ref":"Calendar.ISO.html#days_in_month/2-examples"},{"type":"function","title":"Calendar.ISO.iso_days_to_beginning_of_day/1","doc":"Converts the `t:Calendar.iso_days/0` to the first moment of the day.","ref":"Calendar.ISO.html#iso_days_to_beginning_of_day/1"},{"type":"function","title":"Examples - Calendar.ISO.iso_days_to_beginning_of_day/1","doc":"iex> Calendar.ISO.iso_days_to_beginning_of_day({0, {0, 86400000000}})\n    {0, {0, 86400000000}}\n    iex> Calendar.ISO.iso_days_to_beginning_of_day({730485, {43200000000, 86400000000}})\n    {730485, {0, 86400000000}}\n    iex> Calendar.ISO.iso_days_to_beginning_of_day({730485, {46800000000, 86400000000}})\n    {730485, {0, 86400000000}}","ref":"Calendar.ISO.html#iso_days_to_beginning_of_day/1-examples"},{"type":"function","title":"Calendar.ISO.iso_days_to_end_of_day/1","doc":"Converts the `t:Calendar.iso_days/0` to the last moment of the day.","ref":"Calendar.ISO.html#iso_days_to_end_of_day/1"},{"type":"function","title":"Examples - Calendar.ISO.iso_days_to_end_of_day/1","doc":"iex> Calendar.ISO.iso_days_to_end_of_day({0, {0, 86400000000}})\n    {0, {86399999999, 86400000000}}\n    iex> Calendar.ISO.iso_days_to_end_of_day({730485, {43200000000, 86400000000}})\n    {730485, {86399999999, 86400000000}}\n    iex> Calendar.ISO.iso_days_to_end_of_day({730485, {46800000000, 86400000000}})\n    {730485, {86399999999, 86400000000}}","ref":"Calendar.ISO.html#iso_days_to_end_of_day/1-examples"},{"type":"function","title":"Calendar.ISO.leap_year?/1","doc":"Returns if the given year is a leap year.","ref":"Calendar.ISO.html#leap_year?/1"},{"type":"function","title":"Examples - Calendar.ISO.leap_year?/1","doc":"iex> Calendar.ISO.leap_year?(2000)\n    true\n    iex> Calendar.ISO.leap_year?(2001)\n    false\n    iex> Calendar.ISO.leap_year?(2004)\n    true\n    iex> Calendar.ISO.leap_year?(1900)\n    false\n    iex> Calendar.ISO.leap_year?(-4)\n    true","ref":"Calendar.ISO.html#leap_year?/1-examples"},{"type":"function","title":"Calendar.ISO.months_in_year/1","doc":"Returns how many months there are in the given year.","ref":"Calendar.ISO.html#months_in_year/1"},{"type":"function","title":"Example - Calendar.ISO.months_in_year/1","doc":"iex> Calendar.ISO.months_in_year(2004)\n    12","ref":"Calendar.ISO.html#months_in_year/1-example"},{"type":"function","title":"Calendar.ISO.naive_datetime_from_iso_days/1","doc":"Converts the `t:Calendar.iso_days/0` format to the datetime format specified by this calendar.","ref":"Calendar.ISO.html#naive_datetime_from_iso_days/1"},{"type":"function","title":"Examples - Calendar.ISO.naive_datetime_from_iso_days/1","doc":"iex> Calendar.ISO.naive_datetime_from_iso_days({0, {0, 86400}})\n    {0, 1, 1, 0, 0, 0, {0, 6}}\n    iex> Calendar.ISO.naive_datetime_from_iso_days({730_485, {0, 86400}})\n    {2000, 1, 1, 0, 0, 0, {0, 6}}\n    iex> Calendar.ISO.naive_datetime_from_iso_days({730_485, {43200, 86400}})\n    {2000, 1, 1, 12, 0, 0, {0, 6}}\n    iex> Calendar.ISO.naive_datetime_from_iso_days({-365, {0, 86400000000}})\n    {-1, 1, 1, 0, 0, 0, {0, 6}}","ref":"Calendar.ISO.html#naive_datetime_from_iso_days/1-examples"},{"type":"function","title":"Calendar.ISO.naive_datetime_to_iso_days/7","doc":"Returns the `t:Calendar.iso_days/0` format of the specified date.","ref":"Calendar.ISO.html#naive_datetime_to_iso_days/7"},{"type":"function","title":"Examples - Calendar.ISO.naive_datetime_to_iso_days/7","doc":"iex> Calendar.ISO.naive_datetime_to_iso_days(0, 1, 1, 0, 0, 0, {0, 6})\n    {0, {0, 86400000000}}\n    iex> Calendar.ISO.naive_datetime_to_iso_days(2000, 1, 1, 12, 0, 0, {0, 6})\n    {730485, {43200000000, 86400000000}}\n    iex> Calendar.ISO.naive_datetime_to_iso_days(2000, 1, 1, 13, 0, 0, {0, 6})\n    {730485, {46800000000, 86400000000}}\n    iex> Calendar.ISO.naive_datetime_to_iso_days(-1, 1, 1, 0, 0, 0, {0, 6})\n    {-365, {0, 86400000000}}","ref":"Calendar.ISO.html#naive_datetime_to_iso_days/7-examples"},{"type":"function","title":"Calendar.ISO.naive_datetime_to_string/8","doc":"Converts the datetime (without time zone) into a string.\n\nBy default, returns datetimes formatted in the \"extended\" format,\nfor human readability. It also supports the \"basic\" format\nby passing the `:basic` option.","ref":"Calendar.ISO.html#naive_datetime_to_string/8"},{"type":"function","title":"Examples - Calendar.ISO.naive_datetime_to_string/8","doc":"iex> Calendar.ISO.naive_datetime_to_string(2015, 2, 28, 1, 2, 3, {4, 6})\n    \"2015-02-28 01:02:03.000004\"\n    iex> Calendar.ISO.naive_datetime_to_string(2017, 8, 1, 1, 2, 3, {4, 5})\n    \"2017-08-01 01:02:03.00000\"\n\n    iex> Calendar.ISO.naive_datetime_to_string(2015, 2, 28, 1, 2, 3, {4, 6}, :basic)\n    \"20150228 010203.000004\"","ref":"Calendar.ISO.html#naive_datetime_to_string/8-examples"},{"type":"function","title":"Calendar.ISO.parse_date/1","doc":"Parses a date `string` in the `:extended` format.\n\nFor more information on supported strings, see how this\nmodule implements [ISO 8601](#module-iso-8601-compliance).","ref":"Calendar.ISO.html#parse_date/1"},{"type":"function","title":"Examples - Calendar.ISO.parse_date/1","doc":"iex> Calendar.ISO.parse_date(\"2015-01-23\")\n    {:ok, {2015, 1, 23}}\n\n    iex> Calendar.ISO.parse_date(\"2015:01:23\")\n    {:error, :invalid_format}\n    iex> Calendar.ISO.parse_date(\"2015-01-32\")\n    {:error, :invalid_date}","ref":"Calendar.ISO.html#parse_date/1-examples"},{"type":"function","title":"Calendar.ISO.parse_date/2","doc":"Parses a date `string` according to a given `format`.\n\nThe `format` can either be `:basic` or `:extended`.\n\nFor more information on supported strings, see how this\nmodule implements [ISO 8601](#module-iso-8601-compliance).","ref":"Calendar.ISO.html#parse_date/2"},{"type":"function","title":"Examples - Calendar.ISO.parse_date/2","doc":"iex> Calendar.ISO.parse_date(\"20150123\", :basic)\n    {:ok, {2015, 1, 23}}\n    iex> Calendar.ISO.parse_date(\"20150123\", :extended)\n    {:error, :invalid_format}","ref":"Calendar.ISO.html#parse_date/2-examples"},{"type":"function","title":"Calendar.ISO.parse_naive_datetime/1","doc":"Parses a naive datetime `string` in the `:extended` format.\n\nFor more information on supported strings, see how this\nmodule implements [ISO 8601](#module-iso-8601-compliance).","ref":"Calendar.ISO.html#parse_naive_datetime/1"},{"type":"function","title":"Examples - Calendar.ISO.parse_naive_datetime/1","doc":"iex> Calendar.ISO.parse_naive_datetime(\"2015-01-23 23:50:07\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {0, 0}}}\n    iex> Calendar.ISO.parse_naive_datetime(\"2015-01-23 23:50:07Z\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {0, 0}}}\n    iex> Calendar.ISO.parse_naive_datetime(\"2015-01-23 23:50:07-02:30\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {0, 0}}}\n\n    iex> Calendar.ISO.parse_naive_datetime(\"2015-01-23 23:50:07.0\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {0, 1}}}\n    iex> Calendar.ISO.parse_naive_datetime(\"2015-01-23 23:50:07,0123456\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {12345, 6}}}","ref":"Calendar.ISO.html#parse_naive_datetime/1-examples"},{"type":"function","title":"Calendar.ISO.parse_naive_datetime/2","doc":"Parses a naive datetime `string` according to a given `format`.\n\nThe `format` can either be `:basic` or `:extended`.\n\nFor more information on supported strings, see how this\nmodule implements [ISO 8601](#module-iso-8601-compliance).","ref":"Calendar.ISO.html#parse_naive_datetime/2"},{"type":"function","title":"Examples - Calendar.ISO.parse_naive_datetime/2","doc":"iex> Calendar.ISO.parse_naive_datetime(\"20150123 235007\", :basic)\n    {:ok, {2015, 1, 23, 23, 50, 7, {0, 0}}}\n    iex> Calendar.ISO.parse_naive_datetime(\"20150123 235007\", :extended)\n    {:error, :invalid_format}","ref":"Calendar.ISO.html#parse_naive_datetime/2-examples"},{"type":"function","title":"Calendar.ISO.parse_time/1","doc":"Parses a time `string` in the `:extended` format.\n\nFor more information on supported strings, see how this\nmodule implements [ISO 8601](#module-iso-8601-compliance).","ref":"Calendar.ISO.html#parse_time/1"},{"type":"function","title":"Examples - Calendar.ISO.parse_time/1","doc":"iex> Calendar.ISO.parse_time(\"23:50:07\")\n    {:ok, {23, 50, 7, {0, 0}}}\n\n    iex> Calendar.ISO.parse_time(\"23:50:07Z\")\n    {:ok, {23, 50, 7, {0, 0}}}\n    iex> Calendar.ISO.parse_time(\"T23:50:07Z\")\n    {:ok, {23, 50, 7, {0, 0}}}","ref":"Calendar.ISO.html#parse_time/1-examples"},{"type":"function","title":"Calendar.ISO.parse_time/2","doc":"Parses a time `string` according to a given `format`.\n\nThe `format` can either be `:basic` or `:extended`.\n\nFor more information on supported strings, see how this\nmodule implements [ISO 8601](#module-iso-8601-compliance).","ref":"Calendar.ISO.html#parse_time/2"},{"type":"function","title":"Examples - Calendar.ISO.parse_time/2","doc":"iex> Calendar.ISO.parse_time(\"235007\", :basic)\n    {:ok, {23, 50, 7, {0, 0}}}\n    iex> Calendar.ISO.parse_time(\"235007\", :extended)\n    {:error, :invalid_format}","ref":"Calendar.ISO.html#parse_time/2-examples"},{"type":"function","title":"Calendar.ISO.parse_utc_datetime/1","doc":"Parses a UTC datetime `string` in the `:extended` format.\n\nFor more information on supported strings, see how this\nmodule implements [ISO 8601](#module-iso-8601-compliance).","ref":"Calendar.ISO.html#parse_utc_datetime/1"},{"type":"function","title":"Examples - Calendar.ISO.parse_utc_datetime/1","doc":"iex> Calendar.ISO.parse_utc_datetime(\"2015-01-23 23:50:07Z\")\n    {:ok, {2015, 1, 23, 23, 50, 7, {0, 0}}, 0}\n\n    iex> Calendar.ISO.parse_utc_datetime(\"2015-01-23 23:50:07+02:30\")\n    {:ok, {2015, 1, 23, 21, 20, 7, {0, 0}}, 9000}\n\n    iex> Calendar.ISO.parse_utc_datetime(\"2015-01-23 23:50:07\")\n    {:error, :missing_offset}","ref":"Calendar.ISO.html#parse_utc_datetime/1-examples"},{"type":"function","title":"Calendar.ISO.parse_utc_datetime/2","doc":"Parses a UTC datetime `string` according to a given `format`.\n\nThe `format` can either be `:basic` or `:extended`.\n\nFor more information on supported strings, see how this\nmodule implements [ISO 8601](#module-iso-8601-compliance).","ref":"Calendar.ISO.html#parse_utc_datetime/2"},{"type":"function","title":"Examples - Calendar.ISO.parse_utc_datetime/2","doc":"iex> Calendar.ISO.parse_utc_datetime(\"20150123 235007Z\", :basic)\n    {:ok, {2015, 1, 23, 23, 50, 7, {0, 0}}, 0}\n    iex> Calendar.ISO.parse_utc_datetime(\"20150123 235007Z\", :extended)\n    {:error, :invalid_format}","ref":"Calendar.ISO.html#parse_utc_datetime/2-examples"},{"type":"function","title":"Calendar.ISO.quarter_of_year/3","doc":"Calculates the quarter of the year from the given `year`, `month`, and `day`.\n\nIt is an integer from 1 to 4.","ref":"Calendar.ISO.html#quarter_of_year/3"},{"type":"function","title":"Examples - Calendar.ISO.quarter_of_year/3","doc":"iex> Calendar.ISO.quarter_of_year(2016, 1, 31)\n    1\n    iex> Calendar.ISO.quarter_of_year(2016, 4, 3)\n    2\n    iex> Calendar.ISO.quarter_of_year(-99, 9, 31)\n    3\n    iex> Calendar.ISO.quarter_of_year(2018, 12, 28)\n    4","ref":"Calendar.ISO.html#quarter_of_year/3-examples"},{"type":"function","title":"Calendar.ISO.time_from_day_fraction/1","doc":"Converts a day fraction to this Calendar's representation of time.","ref":"Calendar.ISO.html#time_from_day_fraction/1"},{"type":"function","title":"Examples - Calendar.ISO.time_from_day_fraction/1","doc":"iex> Calendar.ISO.time_from_day_fraction({1, 2})\n    {12, 0, 0, {0, 6}}\n    iex> Calendar.ISO.time_from_day_fraction({13, 24})\n    {13, 0, 0, {0, 6}}","ref":"Calendar.ISO.html#time_from_day_fraction/1-examples"},{"type":"function","title":"Calendar.ISO.time_to_day_fraction/4","doc":"Returns the normalized day fraction of the specified time.","ref":"Calendar.ISO.html#time_to_day_fraction/4"},{"type":"function","title":"Examples - Calendar.ISO.time_to_day_fraction/4","doc":"iex> Calendar.ISO.time_to_day_fraction(0, 0, 0, {0, 6})\n    {0, 86400000000}\n    iex> Calendar.ISO.time_to_day_fraction(12, 34, 56, {123, 6})\n    {45296000123, 86400000000}","ref":"Calendar.ISO.html#time_to_day_fraction/4-examples"},{"type":"function","title":"Calendar.ISO.time_to_string/5","doc":"Converts the given time into a string.\n\nBy default, returns times formatted in the \"extended\" format,\nfor human readability. It also supports the \"basic\" format\nby passing the `:basic` option.","ref":"Calendar.ISO.html#time_to_string/5"},{"type":"function","title":"Examples - Calendar.ISO.time_to_string/5","doc":"iex> Calendar.ISO.time_to_string(2, 2, 2, {2, 6})\n    \"02:02:02.000002\"\n    iex> Calendar.ISO.time_to_string(2, 2, 2, {2, 2})\n    \"02:02:02.00\"\n    iex> Calendar.ISO.time_to_string(2, 2, 2, {2, 0})\n    \"02:02:02\"\n\n    iex> Calendar.ISO.time_to_string(2, 2, 2, {2, 6}, :basic)\n    \"020202.000002\"\n    iex> Calendar.ISO.time_to_string(2, 2, 2, {2, 6}, :extended)\n    \"02:02:02.000002\"","ref":"Calendar.ISO.html#time_to_string/5-examples"},{"type":"function","title":"Calendar.ISO.time_unit_to_precision/1","doc":"Converts a `t:System.time_unit/0` to precision.\n\nInteger-based time units always get maximum precision.","ref":"Calendar.ISO.html#time_unit_to_precision/1"},{"type":"function","title":"Examples - Calendar.ISO.time_unit_to_precision/1","doc":"iex> Calendar.ISO.time_unit_to_precision(:nanosecond)\n    6\n\n    iex> Calendar.ISO.time_unit_to_precision(:second)\n    0\n\n    iex> Calendar.ISO.time_unit_to_precision(1)\n    6","ref":"Calendar.ISO.html#time_unit_to_precision/1-examples"},{"type":"function","title":"Calendar.ISO.valid_date?/3","doc":"Determines if the date given is valid according to the proleptic Gregorian calendar.","ref":"Calendar.ISO.html#valid_date?/3"},{"type":"function","title":"Examples - Calendar.ISO.valid_date?/3","doc":"iex> Calendar.ISO.valid_date?(2015, 2, 28)\n    true\n    iex> Calendar.ISO.valid_date?(2015, 2, 30)\n    false\n    iex> Calendar.ISO.valid_date?(-1, 12, 31)\n    true\n    iex> Calendar.ISO.valid_date?(-1, 12, 32)\n    false","ref":"Calendar.ISO.html#valid_date?/3-examples"},{"type":"function","title":"Calendar.ISO.valid_time?/4","doc":"Determines if the date given is valid according to the proleptic Gregorian calendar.\n\nLeap seconds are not supported by the built-in Calendar.ISO.","ref":"Calendar.ISO.html#valid_time?/4"},{"type":"function","title":"Examples - Calendar.ISO.valid_time?/4","doc":"iex> Calendar.ISO.valid_time?(10, 50, 25, {3006, 6})\n    true\n    iex> Calendar.ISO.valid_time?(23, 59, 60, {0, 0})\n    false\n    iex> Calendar.ISO.valid_time?(24, 0, 0, {0, 0})\n    false","ref":"Calendar.ISO.html#valid_time?/4-examples"},{"type":"function","title":"Calendar.ISO.year_of_era/1","doc":"Calculates the year and era from the given `year`.\n\nThe ISO calendar has two eras: the \"current era\" (CE) which\nstarts in year `1` and is defined as era `1`. And \"before the current\nera\" (BCE) for those years less than `1`, defined as era `0`.","ref":"Calendar.ISO.html#year_of_era/1"},{"type":"function","title":"Examples - Calendar.ISO.year_of_era/1","doc":"iex> Calendar.ISO.year_of_era(1)\n    {1, 1}\n    iex> Calendar.ISO.year_of_era(2018)\n    {2018, 1}\n    iex> Calendar.ISO.year_of_era(0)\n    {1, 0}\n    iex> Calendar.ISO.year_of_era(-1)\n    {2, 0}","ref":"Calendar.ISO.html#year_of_era/1-examples"},{"type":"function","title":"Calendar.ISO.year_of_era/3","doc":"Calendar callback to compute the year and era from the\ngiven `year`, `month` and `day`.\n\nIn the ISO calendar, the new year coincides with the new era,\nso the `month` and `day` arguments are discarded. If you only\nhave the year available, you can `year_of_era/1` instead.","ref":"Calendar.ISO.html#year_of_era/3"},{"type":"function","title":"Examples - Calendar.ISO.year_of_era/3","doc":"iex> Calendar.ISO.year_of_era(1, 1, 1)\n    {1, 1}\n    iex> Calendar.ISO.year_of_era(2018, 12, 1)\n    {2018, 1}\n    iex> Calendar.ISO.year_of_era(0, 1, 1)\n    {1, 0}\n    iex> Calendar.ISO.year_of_era(-1, 12, 1)\n    {2, 0}","ref":"Calendar.ISO.html#year_of_era/3-examples"},{"type":"type","title":"Calendar.ISO.bce/0","doc":"\"Before the Current Era\" or \"Before the Common Era\" (BCE), for those years less than `1`.","ref":"Calendar.ISO.html#t:bce/0"},{"type":"type","title":"Calendar.ISO.ce/0","doc":"The \"Current Era\" or the \"Common Era\" (CE) which starts in year `1`.","ref":"Calendar.ISO.html#t:ce/0"},{"type":"type","title":"Calendar.ISO.day/0","doc":"","ref":"Calendar.ISO.html#t:day/0"},{"type":"type","title":"Calendar.ISO.day_of_week/0","doc":"Integer that represents the day of the week, where 1 is Monday and 7 is Sunday.","ref":"Calendar.ISO.html#t:day_of_week/0"},{"type":"type","title":"Calendar.ISO.day_of_year/0","doc":"","ref":"Calendar.ISO.html#t:day_of_year/0"},{"type":"type","title":"Calendar.ISO.era/0","doc":"The calendar era.\n\nThe ISO calendar has two eras:\n* [CE](`t:ce/0`) - which starts in year `1` and is defined as era `1`.\n* [BCE](`t:bce/0`) - for those years less than `1` and is defined as era `0`.","ref":"Calendar.ISO.html#t:era/0"},{"type":"type","title":"Calendar.ISO.format/0","doc":"","ref":"Calendar.ISO.html#t:format/0"},{"type":"type","title":"Calendar.ISO.hour/0","doc":"","ref":"Calendar.ISO.html#t:hour/0"},{"type":"type","title":"Calendar.ISO.microsecond/0","doc":"Microseconds with stored precision.\n\nThe precision represents the number of digits that must be used when\nrepresenting the microseconds to external format. If the precision is 0,\nit means microseconds must be skipped.","ref":"Calendar.ISO.html#t:microsecond/0"},{"type":"type","title":"Calendar.ISO.minute/0","doc":"","ref":"Calendar.ISO.html#t:minute/0"},{"type":"type","title":"Calendar.ISO.month/0","doc":"","ref":"Calendar.ISO.html#t:month/0"},{"type":"type","title":"Calendar.ISO.quarter_of_year/0","doc":"","ref":"Calendar.ISO.html#t:quarter_of_year/0"},{"type":"type","title":"Calendar.ISO.second/0","doc":"","ref":"Calendar.ISO.html#t:second/0"},{"type":"type","title":"Calendar.ISO.utc_offset/0","doc":"","ref":"Calendar.ISO.html#t:utc_offset/0"},{"type":"type","title":"Calendar.ISO.weekday/0","doc":"","ref":"Calendar.ISO.html#t:weekday/0"},{"type":"type","title":"Calendar.ISO.year/0","doc":"","ref":"Calendar.ISO.html#t:year/0"},{"type":"type","title":"Calendar.ISO.year_of_era/0","doc":"","ref":"Calendar.ISO.html#t:year_of_era/0"},{"type":"behaviour","title":"Calendar.TimeZoneDatabase","doc":"This module defines a behaviour for providing time zone data.\n\nIANA provides time zone data that includes data about different\nUTC offsets and standard offsets for time zones.","ref":"Calendar.TimeZoneDatabase.html"},{"type":"callback","title":"Calendar.TimeZoneDatabase.time_zone_period_from_utc_iso_days/2","doc":"Time zone period for a point in time in UTC for a specific time zone.\n\nTakes a time zone name and a point in time for UTC and returns a\n`time_zone_period` for that point in time.","ref":"Calendar.TimeZoneDatabase.html#c:time_zone_period_from_utc_iso_days/2"},{"type":"callback","title":"Calendar.TimeZoneDatabase.time_zone_periods_from_wall_datetime/2","doc":"Possible time zone periods for a certain time zone and wall clock date and time.\n\nWhen the provided naive datetime is ambiguous, return a tuple with `:ambiguous`\nand the two possible periods. The periods in the tuple must be sorted with the\nfirst element being the one that begins first.\n\nWhen the provided naive datetime is in a gap, such as during the \"spring forward\" when going\nfrom winter time to summer time, return a tuple with `:gap` and two periods with limits\nin a nested tuple. The first nested two-tuple is the period before the gap and a naive datetime\nwith a limit for when the period ends (wall time). The second nested two-tuple is the period\njust after the gap and a datetime (wall time) for when the period begins just after the gap.\n\nIf there is only a single possible period for the provided `datetime`, then return a tuple\nwith `:ok` and the `time_zone_period`.","ref":"Calendar.TimeZoneDatabase.html#c:time_zone_periods_from_wall_datetime/2"},{"type":"type","title":"Calendar.TimeZoneDatabase.time_zone_period/0","doc":"A period where a certain combination of UTC offset, standard offset, and zone\nabbreviation is in effect.\n\nFor example, one period could be the summer of 2018 in the `Europe/London` timezone,\nwhere summer time/daylight saving time is in effect and lasts from spring to autumn.\nIn autumn, the `std_offset` changes along with the `zone_abbr` so a different\nperiod is needed during winter.","ref":"Calendar.TimeZoneDatabase.html#t:time_zone_period/0"},{"type":"type","title":"Calendar.TimeZoneDatabase.time_zone_period_limit/0","doc":"Limit for when a certain time zone period begins or ends.\n\nA beginning is inclusive. An ending is exclusive. For example, if a period is from\n`2015-03-29 01:00:00` and until `2015-10-25 01:00:00`, the period includes and\nbegins from the beginning of `2015-03-29 01:00:00` and lasts until just before\n`2015-10-25 01:00:00`.\n\nA beginning or end for certain periods are infinite, such as the latest\nperiod for time zones without DST or plans to change. However, for the purpose\nof this behaviour, they are only used for gaps in wall time where the needed\nperiod limits are at a certain time.","ref":"Calendar.TimeZoneDatabase.html#t:time_zone_period_limit/0"},{"type":"module","title":"Calendar.UTCOnlyTimeZoneDatabase","doc":"Built-in time zone database that works only in the `Etc/UTC` timezone.\n\nFor all other time zones, it returns `{:error, :utc_only_time_zone_database}`.","ref":"Calendar.UTCOnlyTimeZoneDatabase.html"},{"type":"module","title":"Agent","doc":"Agents are a simple abstraction around state.\n\nOften in Elixir there is a need to share or store state that\nmust be accessed from different processes or by the same process\nat different points in time.\n\nThe `Agent` module provides a basic server implementation that\nallows state to be retrieved and updated via a simple API.","ref":"Agent.html"},{"type":"module","title":"Examples - Agent","doc":"For example, the following agent implements a counter:\n\n    defmodule Counter do\n      use Agent\n\n      def start_link(initial_value) do\n        Agent.start_link(fn -> initial_value end, name: __MODULE__)\n      end\n\n      def value do\n        Agent.get(__MODULE__, & &1)\n      end\n\n      def increment do\n        Agent.update(__MODULE__, &(&1 + 1))\n      end\n    end\n\nUsage would be:\n\n    Counter.start_link(0)\n    #=> {:ok, #PID }\n\n    Counter.value()\n    #=> 0\n\n    Counter.increment()\n    #=> :ok\n\n    Counter.increment()\n    #=> :ok\n\n    Counter.value()\n    #=> 2\n\nThanks to the agent server process, the counter can be safely incremented\nconcurrently.\n\n> #### `use Agent` {: .info}\n>\n> When you `use Agent`, the `Agent` module will define a\n> `child_spec/1` function, so your module can be used\n> as a child in a supervision tree.\n\nAgents provide a segregation between the client and server APIs (similar to\n`GenServer`s). In particular, the functions passed as arguments to the calls to\n`Agent` functions are invoked inside the agent (the server). This distinction\nis important because you may want to avoid expensive operations inside the\nagent, as they will effectively block the agent until the request is\nfulfilled.\n\nConsider these two examples:\n\n    # Compute in the agent/server\n    def get_something(agent) do\n      Agent.get(agent, fn state -> do_something_expensive(state) end)\n    end\n\n    # Compute in the agent/client\n    def get_something(agent) do\n      Agent.get(agent, & &1) |> do_something_expensive()\n    end\n\nThe first function blocks the agent. The second function copies all the state\nto the client and then executes the operation in the client. One aspect to\nconsider is whether the data is large enough to require processing in the server,\nat least initially, or small enough to be sent to the client cheaply. Another\nfactor is whether the data needs to be processed atomically: getting the\nstate and calling `do_something_expensive(state)` outside of the agent means\nthat the agent's state can be updated in the meantime. This is specially\nimportant in case of updates as computing the new state in the client rather\nthan in the server can lead to race conditions if multiple clients are trying\nto update the same state to different values.","ref":"Agent.html#module-examples"},{"type":"module","title":"How to supervise - Agent","doc":"An `Agent` is most commonly started under a supervision tree.\nWhen we invoke `use Agent`, it automatically defines a `child_spec/1`\nfunction that allows us to start the agent directly under a supervisor.\nTo start an agent under a supervisor with an initial counter of 0,\none may do:\n\n    children = [\n      {Counter, 0}\n    ]\n\n    Supervisor.start_link(children, strategy: :one_for_all)\n\nWhile one could also simply pass the `Counter` as a child to the supervisor,\nsuch as:\n\n    children = [\n      Counter # Same as {Counter, []}\n    ]\n\n    Supervisor.start_link(children, strategy: :one_for_all)\n\nThe definition above wouldn't work for this particular example,\nas it would attempt to start the counter with an initial value\nof an empty list. However, this may be a viable option in your\nown agents. A common approach is to use a keyword list, as that\nwould allow setting the initial value and giving a name to the\ncounter process, for example:\n\n    def start_link(opts) do\n      {initial_value, opts} = Keyword.pop(opts, :initial_value, 0)\n      Agent.start_link(fn -> initial_value end, opts)\n    end\n\nand then you can use `Counter`, `{Counter, name: :my_counter}` or\neven `{Counter, initial_value: 0, name: :my_counter}` as a child\nspecification.\n\n`use Agent` also accepts a list of options which configures the\nchild specification and therefore how it runs under a supervisor.\nThe generated `child_spec/1` can be customized with the following options:\n\n  * `:id` - the child specification identifier, defaults to the current module\n  * `:restart` - when the child should be restarted, defaults to `:permanent`\n  * `:shutdown` - how to shut down the child, either immediately or by giving it time to shut down\n\nFor example:\n\n    use Agent, restart: :transient, shutdown: 10_000\n\nSee the \"Child specification\" section in the `Supervisor` module for more\ndetailed information. The `@doc` annotation immediately preceding\n`use Agent` will be attached to the generated `child_spec/1` function.","ref":"Agent.html#module-how-to-supervise"},{"type":"module","title":"Name registration - Agent","doc":"An agent is bound to the same name registration rules as GenServers.\nRead more about it in the `GenServer` documentation.","ref":"Agent.html#module-name-registration"},{"type":"module","title":"A word on distributed agents - Agent","doc":"It is important to consider the limitations of distributed agents. Agents\nprovide two APIs, one that works with anonymous functions and another\nthat expects an explicit module, function, and arguments.\n\nIn a distributed setup with multiple nodes, the API that accepts anonymous\nfunctions only works if the caller (client) and the agent have the same\nversion of the caller module.\n\nKeep in mind this issue also shows up when performing \"rolling upgrades\"\nwith agents. By rolling upgrades we mean the following situation: you wish\nto deploy a new version of your software by *shutting down* some of your\nnodes and replacing them with nodes running a new version of the software.\nIn this setup, part of your environment will have one version of a given\nmodule and the other part another version (the newer one) of the same module.\n\nThe best solution is to simply use the explicit module, function, and arguments\nAPIs when working with distributed agents.","ref":"Agent.html#module-a-word-on-distributed-agents"},{"type":"module","title":"Hot code swapping - Agent","doc":"An agent can have its code hot swapped live by simply passing a module,\nfunction, and arguments tuple to the update instruction. For example, imagine\nyou have an agent named `:sample` and you want to convert its inner state\nfrom a keyword list to a map. It can be done with the following\ninstruction:\n\n    {:update, :sample, {:advanced, {Enum, :into, [%{}]}}}\n\nThe agent's state will be added to the given list of arguments (`[%{}]`) as\nthe first argument.","ref":"Agent.html#module-hot-code-swapping"},{"type":"function","title":"Agent.cast/2","doc":"Performs a cast (*fire and forget*) operation on the agent state.\n\nThe function `fun` is sent to the `agent` which invokes the function\npassing the agent state. The return value of `fun` becomes the new\nstate of the agent.\n\nNote that `cast` returns `:ok` immediately, regardless of whether `agent` (or\nthe node it should live on) exists.","ref":"Agent.html#cast/2"},{"type":"function","title":"Examples - Agent.cast/2","doc":"iex> {:ok, pid} = Agent.start_link(fn -> 42 end)\n    iex> Agent.cast(pid, fn state -> state + 1 end)\n    :ok\n    iex> Agent.get(pid, fn state -> state end)\n    43","ref":"Agent.html#cast/2-examples"},{"type":"function","title":"Agent.cast/4","doc":"Performs a cast (*fire and forget*) operation on the agent state.\n\nSame as `cast/2` but a module, function, and arguments are expected\ninstead of an anonymous function. The state is added as first\nargument to the given list of arguments.","ref":"Agent.html#cast/4"},{"type":"function","title":"Examples - Agent.cast/4","doc":"iex> {:ok, pid} = Agent.start_link(fn -> 42 end)\n    iex> Agent.cast(pid, Kernel, :+, [12])\n    :ok\n    iex> Agent.get(pid, fn state -> state end)\n    54","ref":"Agent.html#cast/4-examples"},{"type":"function","title":"Agent.child_spec/1","doc":"Returns a specification to start an agent under a supervisor.\n\nSee the \"Child specification\" section in the `Supervisor` module for more detailed information.","ref":"Agent.html#child_spec/1"},{"type":"function","title":"Agent.get/3","doc":"Gets an agent value via the given anonymous function.\n\nThe function `fun` is sent to the `agent` which invokes the function\npassing the agent state. The result of the function invocation is\nreturned from this function.\n\n`timeout` is an integer greater than zero which specifies how many\nmilliseconds are allowed before the agent executes the function and returns\nthe result value, or the atom `:infinity` to wait indefinitely. If no result\nis received within the specified time, the function call fails and the caller\nexits.","ref":"Agent.html#get/3"},{"type":"function","title":"Examples - Agent.get/3","doc":"iex> {:ok, pid} = Agent.start_link(fn -> 42 end)\n    iex> Agent.get(pid, fn state -> state end)\n    42","ref":"Agent.html#get/3-examples"},{"type":"function","title":"Agent.get/5","doc":"Gets an agent value via the given function.\n\nSame as `get/3` but a module, function, and arguments are expected\ninstead of an anonymous function. The state is added as first\nargument to the given list of arguments.","ref":"Agent.html#get/5"},{"type":"function","title":"Agent.get_and_update/3","doc":"Gets and updates the agent state in one operation via the given anonymous\nfunction.\n\nThe function `fun` is sent to the `agent` which invokes the function\npassing the agent state. The function must return a tuple with two\nelements, the first being the value to return (that is, the \"get\" value)\nand the second one being the new state of the agent.\n\n`timeout` is an integer greater than zero which specifies how many\nmilliseconds are allowed before the agent executes the function and returns\nthe result value, or the atom `:infinity` to wait indefinitely. If no result\nis received within the specified time, the function call fails and the caller\nexits.","ref":"Agent.html#get_and_update/3"},{"type":"function","title":"Examples - Agent.get_and_update/3","doc":"iex> {:ok, pid} = Agent.start_link(fn -> 42 end)\n    iex> Agent.get_and_update(pid, fn state -> {state, state + 1} end)\n    42\n    iex> Agent.get(pid, fn state -> state end)\n    43","ref":"Agent.html#get_and_update/3-examples"},{"type":"function","title":"Agent.get_and_update/5","doc":"Gets and updates the agent state in one operation via the given function.\n\nSame as `get_and_update/3` but a module, function, and arguments are expected\ninstead of an anonymous function. The state is added as first\nargument to the given list of arguments.","ref":"Agent.html#get_and_update/5"},{"type":"function","title":"Agent.start/2","doc":"Starts an agent process without links (outside of a supervision tree).\n\nSee `start_link/2` for more information.","ref":"Agent.html#start/2"},{"type":"function","title":"Examples - Agent.start/2","doc":"iex> {:ok, pid} = Agent.start(fn -> 42 end)\n    iex> Agent.get(pid, fn state -> state end)\n    42","ref":"Agent.html#start/2-examples"},{"type":"function","title":"Agent.start/4","doc":"Starts an agent without links with the given module, function, and arguments.\n\nSee `start_link/4` for more information.","ref":"Agent.html#start/4"},{"type":"function","title":"Agent.start_link/2","doc":"Starts an agent linked to the current process with the given function.\n\nThis is often used to start the agent as part of a supervision tree.\n\nOnce the agent is spawned, the given function `fun` is invoked in the server\nprocess, and should return the initial agent state. Note that `start_link/2`\ndoes not return until the given function has returned.","ref":"Agent.html#start_link/2"},{"type":"function","title":"Options - Agent.start_link/2","doc":"The `:name` option is used for registration as described in the module\ndocumentation.\n\nIf the `:timeout` option is present, the agent is allowed to spend at most\nthe given number of milliseconds on initialization or it will be terminated\nand the start function will return `{:error, :timeout}`.\n\nIf the `:debug` option is present, the corresponding function in the\n[`:sys` module](`:sys`) will be invoked.\n\nIf the `:spawn_opt` option is present, its value will be passed as options\nto the underlying process as in `Process.spawn/4`.","ref":"Agent.html#start_link/2-options"},{"type":"function","title":"Return values - Agent.start_link/2","doc":"If the server is successfully created and initialized, the function returns\n`{:ok, pid}`, where `pid` is the PID of the server. If an agent with the\nspecified name already exists, the function returns\n`{:error, {:already_started, pid}}` with the PID of that process.\n\nIf the given function callback fails, the function returns `{:error, reason}`.","ref":"Agent.html#start_link/2-return-values"},{"type":"function","title":"Examples - Agent.start_link/2","doc":"iex> {:ok, pid} = Agent.start_link(fn -> 42 end)\n    iex> Agent.get(pid, fn state -> state end)\n    42\n\n    iex> {:error, {exception, _stacktrace}} = Agent.start(fn -> raise \"oops\" end)\n    iex> exception\n    %RuntimeError{message: \"oops\"}","ref":"Agent.html#start_link/2-examples"},{"type":"function","title":"Agent.start_link/4","doc":"Starts an agent linked to the current process.\n\nSame as `start_link/2` but a module, function, and arguments are expected\ninstead of an anonymous function; `fun` in `module` will be called with the\ngiven arguments `args` to initialize the state.","ref":"Agent.html#start_link/4"},{"type":"function","title":"Agent.stop/3","doc":"Synchronously stops the agent with the given `reason`.\n\nIt returns `:ok` if the agent terminates with the given\nreason. If the agent terminates with another reason, the call will\nexit.\n\nThis function keeps OTP semantics regarding error reporting.\nIf the reason is any other than `:normal`, `:shutdown` or\n`{:shutdown, _}`, an error report will be logged.","ref":"Agent.html#stop/3"},{"type":"function","title":"Examples - Agent.stop/3","doc":"iex> {:ok, pid} = Agent.start_link(fn -> 42 end)\n    iex> Agent.stop(pid)\n    :ok","ref":"Agent.html#stop/3-examples"},{"type":"function","title":"Agent.update/3","doc":"Updates the agent state via the given anonymous function.\n\nThe function `fun` is sent to the `agent` which invokes the function\npassing the agent state. The return value of `fun` becomes the new\nstate of the agent.\n\nThis function always returns `:ok`.\n\n`timeout` is an integer greater than zero which specifies how many\nmilliseconds are allowed before the agent executes the function and returns\nthe result value, or the atom `:infinity` to wait indefinitely. If no result\nis received within the specified time, the function call fails and the caller\nexits.","ref":"Agent.html#update/3"},{"type":"function","title":"Examples - Agent.update/3","doc":"iex> {:ok, pid} = Agent.start_link(fn -> 42 end)\n    iex> Agent.update(pid, fn state -> state + 1 end)\n    :ok\n    iex> Agent.get(pid, fn state -> state end)\n    43","ref":"Agent.html#update/3-examples"},{"type":"function","title":"Agent.update/5","doc":"Updates the agent state via the given function.\n\nSame as `update/3` but a module, function, and arguments are expected\ninstead of an anonymous function. The state is added as first\nargument to the given list of arguments.","ref":"Agent.html#update/5"},{"type":"function","title":"Examples - Agent.update/5","doc":"iex> {:ok, pid} = Agent.start_link(fn -> 42 end)\n    iex> Agent.update(pid, Kernel, :+, [12])\n    :ok\n    iex> Agent.get(pid, fn state -> state end)\n    54","ref":"Agent.html#update/5-examples"},{"type":"type","title":"Agent.agent/0","doc":"The agent reference","ref":"Agent.html#t:agent/0"},{"type":"type","title":"Agent.name/0","doc":"The agent name","ref":"Agent.html#t:name/0"},{"type":"type","title":"Agent.on_start/0","doc":"Return values of `start*` functions","ref":"Agent.html#t:on_start/0"},{"type":"type","title":"Agent.state/0","doc":"The agent state","ref":"Agent.html#t:state/0"},{"type":"behaviour","title":"Application","doc":"A module for working with applications and defining application callbacks.\n\nApplications are the idiomatic way to package software in Erlang/OTP. To get\nthe idea, they are similar to the \"library\" concept common in other\nprogramming languages, but with some additional characteristics.\n\nAn application is a component implementing some specific functionality, with a\nstandardized directory structure, configuration, and life cycle. Applications\nare *loaded*, *started*, and *stopped*. Each application also has its own\nenvironment, which provides a unified API for configuring each application.\n\nDevelopers typically interact with the application environment and its\ncallback module. Therefore those will be the topics we will cover first\nbefore jumping into details about the application resource file and life-cycle.","ref":"Application.html"},{"type":"behaviour","title":"The application environment - Application","doc":"Each application has its own environment. The environment is a keyword list\nthat maps atoms to terms. Note that this environment is unrelated to the\noperating system environment.\n\nBy default, the environment of an application is an empty list. In a Mix\nproject's `mix.exs` file, you can set the `:env` key in `application/0`:\n\n    def application do\n      [env: [db_host: \"localhost\"]]\n    end\n\nNow, in your application, you can read this environment by using functions\nsuch as `fetch_env!/2` and friends:\n\n    defmodule MyApp.DBClient do\n      def start_link() do\n        SomeLib.DBClient.start_link(host: db_host())\n      end\n\n      defp db_host do\n        Application.fetch_env!(:my_app, :db_host)\n      end\n    end\n\nIn Mix projects, the environment of the application and its dependencies can\nbe overridden via the `config/config.exs` and `config/runtime.exs` files. The\nformer is loaded at build-time, before your code compiles, and the latter at\nruntime, just before your app starts. For example, someone using your application\ncan override its `:db_host` environment variable as follows:\n\n    import Config\n    config :my_app, :db_host, \"db.local\"\n\nSee the \"Configuration\" section in the `Mix` module for more information.\nYou can also change the application environment dynamically by using functions\nsuch as `put_env/3` and `delete_env/2`.\n\n> ##","ref":"Application.html#module-the-application-environment"},{"type":"behaviour","title":"Environment in libraries {: .tip} - Application","doc":">\n> The config files `config/config.exs` and `config/runtime.exs`\n> are rarely used by libraries. Libraries typically define their environment\n> in the `application/0` function of their `mix.exs`. Configuration files\n> are rather used by applications to configure their libraries.\n\n> ##","ref":"Application.html#module-environment-in-libraries-tip"},{"type":"behaviour","title":"Reading the environment of other applications {: .warning} - Application","doc":">\n> Each application is responsible for its own environment. Do not\n> use the functions in this module for directly accessing or modifying\n> the environment of other applications. Whenever you change the application\n> environment, Elixir's build tool will only recompile the files that\n> belong to that application. So if you read the application environment\n> of another application, there is a chance you will be depending on\n> outdated configuration, as your file won't be recompiled as it changes.","ref":"Application.html#module-reading-the-environment-of-other-applications-warning"},{"type":"behaviour","title":"Compile-time environment - Application","doc":"In the previous example, we read the application environment at runtime:\n\n    defmodule MyApp.DBClient do\n      def start_link() do\n        SomeLib.DBClient.start_link(host: db_host())\n      end\n\n      defp db_host do\n        Application.fetch_env!(:my_app, :db_host)\n      end\n    end\n\nIn other words, the environment key `:db_host` for application `:my_app`\nwill only be read when `MyApp.DBClient` effectively starts. While reading\nthe application environment at runtime is the preferred approach, in some\nrare occasions you may want to use the application environment to configure\nthe compilation of a certain project. However, if you try to access\n`Application.fetch_env!/2` outside of a function:\n\n    defmodule MyApp.DBClient do\n      @db_host Application.fetch_env!(:my_app, :db_host)\n\n      def start_link() do\n        SomeLib.DBClient.start_link(host: @db_host)\n      end\n    end\n\nYou might see warnings and errors:\n\n    warning: Application.fetch_env!/2 is discouraged in the module body,\n    use Application.compile_env/3 instead\n      iex:3: MyApp.DBClient\n\n    ** (ArgumentError) could not fetch application environment :db_host\n    for application :my_app because the application was not loaded nor\n    configured\n\nThis happens because, when defining modules, the application environment\nis not yet available. Luckily, the warning tells us how to solve this\nissue, by using `Application.compile_env/3` instead:\n\n    defmodule MyApp.DBClient do\n      @db_host Application.compile_env(:my_app, :db_host, \"db.local\")\n\n      def start_link() do\n        SomeLib.DBClient.start_link(host: @db_host)\n      end\n    end\n\nThe difference here is that `compile_env` expects the default value to be\ngiven as an argument, instead of using the `def application` function of\nyour `mix.exs`. Furthermore, by using `compile_env/3`, tools like Mix will\nstore the values used during compilation and compare the compilation values\nwith the runtime values whenever your system starts, raising an error in\ncase they differ.\n\nIn any case, compile-time environments should be avoided. Whenever possible,\nreading the application environment at runtime should be the first choice.","ref":"Application.html#module-compile-time-environment"},{"type":"behaviour","title":"The application callback module - Application","doc":"Applications can be loaded, started, and stopped. Generally, build tools\nlike Mix take care of starting an application and all of its dependencies\nfor you, but you can also do it manually by calling:\n\n    {:ok, _} = Application.ensure_all_started(:some_app)\n\nWhen an application starts, developers may configure a callback module\nthat executes custom code. Developers use this callback to start the\napplication supervision tree.\n\nThe first step to do so is to add a `:mod` key to the `application/0`\ndefinition in your `mix.exs` file. It expects a tuple, with the application\ncallback module and start argument (commonly an empty list):\n\n    def application do\n      [mod: {MyApp, []}]\n    end\n\nThe `MyApp` module given to `:mod` needs to implement the `Application` behaviour.\nThis can be done by putting `use Application` in that module and implementing the\n`c:start/2` callback, for example:\n\n    defmodule MyApp do\n      use Application\n\n      def start(_type, _args) do\n        children = []\n        Supervisor.start_link(children, strategy: :one_for_one)\n      end\n    end\n\n> #### `use Application` {: .info}\n>\n> When you `use Application`, the `Application` module will\n> set `@behaviour Application` and define an overridable\n> definition for the `c:stop/1` function, which is required\n> by Erlang/OTP.\n\nThe `c:start/2` callback has to spawn and link a supervisor and return `{:ok,\npid}` or `{:ok, pid, state}`, where `pid` is the PID of the supervisor, and\n`state` is an optional application state. `args` is the second element of the\ntuple given to the `:mod` option.\n\nThe `type` argument passed to `c:start/2` is usually `:normal` unless in a\ndistributed setup where application takeovers and failovers are configured.\nDistributed applications are beyond the scope of this documentation.\n\nWhen an application is shutting down, its `c:stop/1` callback is called after\nthe supervision tree has been stopped by the runtime. This callback allows the\napplication to do any final cleanup. The argument is the state returned by\n`c:start/2`, if it did, or `[]` otherwise. The return value of `c:stop/1` is\nignored.\n\nBy using `Application`, modules get a default implementation of `c:stop/1`\nthat ignores its argument and returns `:ok`, but it can be overridden.\n\nApplication callback modules may also implement the optional callback\n`c:prep_stop/1`. If present, `c:prep_stop/1` is invoked before the supervision\ntree is terminated. Its argument is the state returned by `c:start/2`, if it did,\nor `[]` otherwise, and its return value is passed to `c:stop/1`.","ref":"Application.html#module-the-application-callback-module"},{"type":"behaviour","title":"The application resource file - Application","doc":"In the sections above, we have configured an application in the\n`application/0` section of the `mix.exs` file. Ultimately, Mix will use\nthis configuration to create an [*application resource\nfile*](https://www.erlang.org/doc/man/application.html), which is a file called\n`APP_NAME.app`. For example, the application resource file of the OTP\napplication `ex_unit` is called `ex_unit.app`.\n\nYou can learn more about the generation of application resource files in\nthe documentation of `Mix.Tasks.Compile.App`, available as well by running\n`mix help compile.app`.","ref":"Application.html#module-the-application-resource-file"},{"type":"behaviour","title":"The application life cycle - Application","doc":"#","ref":"Application.html#module-the-application-life-cycle"},{"type":"behaviour","title":"Loading applications - Application","doc":"Applications are *loaded*, which means that the runtime finds and processes\ntheir resource files:\n\n    Application.load(:ex_unit)\n    #=> :ok\n\nWhen an application is loaded, the environment specified in its resource file\nis merged with any overrides from config files.\n\nLoading an application *does not* load its modules.\n\nIn practice, you rarely load applications by hand because that is part of the\nstart process, explained next.\n\n#","ref":"Application.html#module-loading-applications"},{"type":"behaviour","title":"Starting applications - Application","doc":"Applications are also *started*:\n\n    Application.start(:ex_unit)\n    #=> :ok\n\nOnce your application is compiled, running your system is a matter of starting\nyour current application and its dependencies. Differently from other languages,\nElixir does not have a `main` procedure that is responsible for starting your\nsystem. Instead, you start one or more applications, each with their own\ninitialization and termination logic.\n\nWhen an application is started, the `Application.load/1` is automatically\ninvoked if it hasn't been done yet. Then, it checks if the dependencies listed\nin the `applications` key of the resource file are already started. Having at\nleast one dependency not started is an error condition. Functions like\n`ensure_all_started/1` takes care of starting an application and all of its\ndependencies for you.\n\nIf the application does not have a callback module configured, starting is\ndone at this point. Otherwise, its `c:start/2` callback is invoked. The PID of\nthe top-level supervisor returned by this function is stored by the runtime\nfor later use, and the returned application state is saved too, if any.\n\n#","ref":"Application.html#module-starting-applications"},{"type":"behaviour","title":"Stopping applications - Application","doc":"Started applications are, finally, *stopped*:\n\n    Application.stop(:ex_unit)\n    #=> :ok\n\nStopping an application without a callback module is defined, but except for\nsome system tracing, it is in practice a no-op.\n\nStopping an application with a callback module has three steps:\n\n  1. If present, invoke the optional callback `c:prep_stop/1`.\n  2. Terminate the top-level supervisor.\n  3. Invoke the required callback `c:stop/1`.\n\nThe arguments passed to the callbacks are related to the state optionally\nreturned by `c:start/2`, and are documented in the section about the callback\nmodule above.\n\nIt is important to highlight that step 2 is a blocking one. Termination of a\nsupervisor triggers a recursive chain of children terminations, therefore\norderly shutting down all descendant processes. The `c:stop/1` callback is\ninvoked only after termination of the whole supervision tree.\n\nShutting down a live system cleanly can be done by calling `System.stop/1`. It\nwill shut down every application in the opposite order they had been started.\n\nBy default, a SIGTERM from the operating system will automatically translate to\n`System.stop/0`. You can also have more explicit control over operating system\nsignals via the `:os.set_signal/2` function.","ref":"Application.html#module-stopping-applications"},{"type":"behaviour","title":"Tooling - Application","doc":"The Mix build tool automates most of the application management tasks. For example,\n`mix test` automatically starts your application dependencies and your application\nitself before your test runs. `mix run --no-halt` boots your current project and\ncan be used to start a long running system. See `mix help run`.\n\nDevelopers can also use `mix release` to build **releases**. Releases are able to\npackage all of your source code as well as the Erlang VM into a single directory.\nReleases also give you explicit control over how each application is started and in\nwhich order. They also provide a more streamlined mechanism for starting and\nstopping systems, debugging, logging, as well as system monitoring.\n\nFinally, Elixir provides tools such as escripts and archives, which are\ndifferent mechanisms for packaging your application. Those are typically used\nwhen tools must be shared between developers and not as deployment options.\nSee `mix help archive.build` and `mix help escript.build` for more detail.","ref":"Application.html#module-tooling"},{"type":"behaviour","title":"Further information - Application","doc":"For further details on applications please check the documentation of the\n[`:application` Erlang module](`:application`), and the\n[Applications](https://www.erlang.org/doc/design_principles/applications.html)\nsection of the [OTP Design Principles User's\nGuide](https://www.erlang.org/doc/design_principles/users_guide.html).","ref":"Application.html#module-further-information"},{"type":"function","title":"Application.app_dir/1","doc":"Gets the directory for app.\n\nThis information is returned based on the code path. Here is an\nexample:\n\n    File.mkdir_p!(\"foo/ebin\")\n    Code.prepend_path(\"foo/ebin\")\n    Application.app_dir(:foo)\n    #=> \"foo\"\n\nEven though the directory is empty and there is no `.app` file\nit is considered the application directory based on the name\n\"foo/ebin\". The name may contain a dash `-` which is considered\nto be the app version and it is removed for the lookup purposes:\n\n    File.mkdir_p!(\"bar-123/ebin\")\n    Code.prepend_path(\"bar-123/ebin\")\n    Application.app_dir(:bar)\n    #=> \"bar-123\"\n\nFor more information on code paths, check the `Code` module in\nElixir and also Erlang's [`:code` module](`:code`).","ref":"Application.html#app_dir/1"},{"type":"function","title":"Application.app_dir/2","doc":"Returns the given path inside `app_dir/1`.\n\nIf `path` is a string, then it will be used as the path inside `app_dir/1`. If\n`path` is a list of strings, it will be joined (see `Path.join/1`) and the result\nwill be used as the path inside `app_dir/1`.","ref":"Application.html#app_dir/2"},{"type":"function","title":"Examples - Application.app_dir/2","doc":"File.mkdir_p!(\"foo/ebin\")\n    Code.prepend_path(\"foo/ebin\")\n\n    Application.app_dir(:foo, \"my_path\")\n    #=> \"foo/my_path\"\n\n    Application.app_dir(:foo, [\"my\", \"nested\", \"path\"])\n    #=> \"foo/my/nested/path\"","ref":"Application.html#app_dir/2-examples"},{"type":"macro","title":"Application.compile_env/3","doc":"Reads the application environment at compilation time.\n\nSimilar to `get_env/3`, except it must be used to read values\nat compile time. This allows Elixir to track when configuration\nvalues change between compile time and runtime.\n\nThe first argument is the application name. The second argument\n`key_or_path` is either an atom key or a path to traverse in\nsearch of the configuration, starting with an atom key.\n\nFor example, imagine the following configuration:\n\n    config :my_app, :key, [foo: [bar: :baz]]\n\nWe can access it during compile time as:\n\n    Application.compile_env(:my_app, :key)\n    #=> [foo: [bar: :baz]]\n\n    Application.compile_env(:my_app, [:key, :foo])\n    #=> [bar: :baz]\n\n    Application.compile_env(:my_app, [:key, :foo, :bar])\n    #=> :baz\n\nA default value can also be given as third argument. If\nany of the keys in the path along the way is missing, the\ndefault value is used:\n\n    Application.compile_env(:my_app, [:unknown, :foo, :bar], :default)\n    #=> :default\n\n    Application.compile_env(:my_app, [:key, :unknown, :bar], :default)\n    #=> :default\n\n    Application.compile_env(:my_app, [:key, :foo, :unknown], :default)\n    #=> :default\n\nGiving a path is useful to let Elixir know that only certain paths\nin a large configuration are compile time dependent.","ref":"Application.html#compile_env/3"},{"type":"function","title":"Application.compile_env/4","doc":"Reads the application environment at compilation time from a macro.\n\nTypically, developers will use `compile_env/3`. This function must\nonly be invoked from macros which aim to read the compilation environment\ndynamically.\n\nIt expects a `Macro.Env` as first argument, where the `Macro.Env` is\ntypically the `__CALLER__` in a macro. It raises if `Macro.Env` comes\nfrom a function.","ref":"Application.html#compile_env/4"},{"type":"macro","title":"Application.compile_env!/2","doc":"Reads the application environment at compilation time or raises.\n\nThis is the same as `compile_env/3` but it raises an\n`ArgumentError` if the configuration is not available.","ref":"Application.html#compile_env!/2"},{"type":"function","title":"Application.compile_env!/3","doc":"Reads the application environment at compilation time from a macro\nor raises.\n\nTypically, developers will use `compile_env!/2`. This function must\nonly be invoked from macros which aim to read the compilation environment\ndynamically.\n\nIt expects a `Macro.Env` as first argument, where the `Macro.Env` is\ntypically the `__CALLER__` in a macro. It raises if `Macro.Env` comes\nfrom a function.","ref":"Application.html#compile_env!/3"},{"type":"callback","title":"Application.config_change/3","doc":"Callback invoked after code upgrade, if the application environment\nhas changed.\n\n`changed` is a keyword list of keys and their changed values in the\napplication environment. `new` is a keyword list with all new keys\nand their values. `removed` is a list with all removed keys.","ref":"Application.html#c:config_change/3"},{"type":"function","title":"Application.delete_env/3","doc":"Deletes the `key` from the given `app` environment.\n\nIt receives the same options as `put_env/4`. Returns `:ok`.","ref":"Application.html#delete_env/3"},{"type":"function","title":"Application.ensure_all_started/2","doc":"Ensures the given `app` or `apps` and their child applications are started.\n\nThe second argument is either the `t:restart_type/1` (for consistency with\n`start/2`) or a keyword list.","ref":"Application.html#ensure_all_started/2"},{"type":"function","title":"Options - Application.ensure_all_started/2","doc":"* `:type` - if the application should be started in `:permanent`,\n    `:temporary`, or `:transient`. See `t:restart_type/1` for more information.\n\n  * `:mode` - (since v1.15.0) if the applications should be started serially\n    or concurrently. This option requires Erlang/OTP 26+.","ref":"Application.html#ensure_all_started/2-options"},{"type":"function","title":"Application.ensure_loaded/1","doc":"Ensures the given `app` is loaded.\n\nSame as `load/1` but returns `:ok` if the application was already\nloaded.","ref":"Application.html#ensure_loaded/1"},{"type":"function","title":"Application.ensure_started/2","doc":"Ensures the given `app` is started with `t:restart_type/0`.\n\nSame as `start/2` but returns `:ok` if the application was already\nstarted.","ref":"Application.html#ensure_started/2"},{"type":"function","title":"Application.fetch_env/2","doc":"Returns the value for `key` in `app`'s environment in a tuple.\n\nIf the configuration parameter does not exist, the function returns `:error`.\n\n> ##","ref":"Application.html#fetch_env/2"},{"type":"function","title":"Warning {: .warning} - Application.fetch_env/2","doc":">\n> You must use this function to read only your own application\n> environment. Do not read the environment of other applications.\n\n> ##","ref":"Application.html#fetch_env/2-warning-warning"},{"type":"function","title":"Application environment in info - Application.fetch_env/2","doc":">\n> If you are writing a library to be used by other developers,\n> it is generally recommended to avoid the application environment, as the\n> application environment is effectively a global storage. For more information,\n> read our [library guidelines](library-guidelines.md).","ref":"Application.html#fetch_env/2-application-environment-in-info"},{"type":"function","title":"Application.fetch_env!/2","doc":"Returns the value for `key` in `app`'s environment.\n\nIf the configuration parameter does not exist, raises `ArgumentError`.\n\n> ##","ref":"Application.html#fetch_env!/2"},{"type":"function","title":"Warning {: .warning} - Application.fetch_env!/2","doc":">\n> You must use this function to read only your own application\n> environment. Do not read the environment of other applications.\n\n> ##","ref":"Application.html#fetch_env!/2-warning-warning"},{"type":"function","title":"Application environment in info - Application.fetch_env!/2","doc":">\n> If you are writing a library to be used by other developers,\n> it is generally recommended to avoid the application environment, as the\n> application environment is effectively a global storage. For more information,\n> read our [library guidelines](library-guidelines.md).","ref":"Application.html#fetch_env!/2-application-environment-in-info"},{"type":"function","title":"Application.format_error/1","doc":"Formats the error reason returned by `start/2`,\n`ensure_started/2`, `stop/1`, `load/1` and `unload/