/*
 *  suse-blinux - Braille-display support for linux
 *  Author: Marco Skambraks <marco@suse.de>
 *  SuSE GmbH Nuernberg
 *
 *
 * suse-blinux based on brltty
 * special thanks to the Brltty-Team
 * Nicolas Pitre <nico@cam.org>
 * Stphane Doyon <s.doyon@videotron.ca>
 * Nikhil Nair <nn201@cus.cam.ac.uk>
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
*/

#define FIFO_IN "/var/run/sbl.fifo.in"
#define FIFO_OUT "/var/run/sbl.fifo.out"
#define SPELL_WORD_TO 500
#define SLEEP sblconf.sleep
#define REPEATDELAY sblconf.repeatdelay
#define PROFILE1 sblconf.profile1
#define PROFILE2 sblconf.profile2
#define PROFILE3 sblconf.profile3
#define PROFILE4 sblconf.profile4
#define ROWS scr.rows
#define COLS scr.cols
#include <errno.h>
#include <utmp.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <wait.h>
#include <sys/stat.h>
#include <errno.h>
#include <linux/vt.h>
#include <sys/ioctl.h>
#include <dlfcn.h>
#include <getopt.h>
#include "../brld/libbrld.h"
#include "sbllog.h"
#include "functions.h"
#include "system.h"
#include "ldconf.h"
#include "config.h"
#include "selkey.h"
#include "brl.h"
#include "scr.h"
#include "inskey.h"
#include "spk.h"
#include "beeps.h"
#include "cut-n-paste.h"
#include "misc.h"
#include "kbdsniff.h"

param_file env;

keymap_file keymap;

sbl_config sblconf;

char sblconf_file[255] = "";

/*
 * Some useful macros: 
 */
#define POSX env.csrmode?attrpos.x+1:scr.posx+1
#define POSY env.csrmode?attrpos.y+1:scr.posy+1
#define MIN(a, b) (((a) < (b)) ? (a) : (b))
#define MAX(a, b) (((a) > (b)) ? (a) : (b))
#define TOGGLEPLAY(var) play ((var) ? snd_toggleon : snd_toggleoff)
#define TOGGLE(var) \
  (var = (keypress & VAL_SWITCHON) ? 1 \
   : (keypress & VAL_SWITCHOFF) ? 0 \
   : var ^ 1)

/*
 * Global variables 
 */

int sblpaus = 0, kbd_sniff_on;

int press = 0, confd, kbdfd;

int connr = 1;

struct vt_stat vterm;

unsigned char attrbuf[1024];

unsigned char **oldscr;

int scrx, scry;                 /* current screen-size */

int upd_st_cells = 1;

int fcells = -1;                /* offset for screenfull() */

int vertdisp = 0;               /* only for fhp */

mask showattr;

char profile[80] = "";

struct csrxy
{
  short x, y;
}
hardcsr, attrpos, spkcsrpos;

/* struct definition for volatile parameters */
sbl_param initparam = {
  1, 0, 0, 0, 0, 0
};

sbl_param *scrparam[NBR_SCR + 1];

int spk = 0, spkupdate = 0, spkmode = 0;

char spkstr[MAXSPKSTR] = "";

char attrstr[MAXSPKSTR] = "", old_attrstr[MAXSPKSTR] = "";

char brlstr[MAXSPKSTR] = "", old_brlstr[MAXSPKSTR] = "";

int csrpos = -1, old_csrpos = -1;

int need_brl_upd = 0;

int fifo_ok = 1;

int jmpmark = 0;

sbl_param *p;                   /* pointer to current param structure */

int curscr = 0;                 /* current screen number */

/* Misc param variables */
short dispmd = LIVE_SCRN;       /* freeze screen on/off */

short csr_offright;             /* used for sliding window */

short hwinshift;                /* Half window horizontal distance */

short fwinshift;                /* Full window horizontal distance */

short vwinshift;                /* Window vertical distance */

brldim brl;                     /* For the Braille routines */
scrstat scr = { 0, 0, 0, 0, 0 };        /* For screen statistics */

/*
 * Output translation tables - the files *.auto.h are generated at *
 * compile-time: 
 */
unsigned char *curtbl = NULL;   /* currently active translation table */

volatile sig_atomic_t keep_going = 1;   /* 

                                         * controls program termination 
                                         */
char home[100];

short homedir_found = 0;        /* CWD status */

/*
 * Status cells support 
 * remark: the Papenmeier has a column with 22 cells, 
 * all other terminals use up to 5 bytes
 */
unsigned char statcells[22];    /* status cell buffer */

/* 
 * Number dot translation for status cells 
 */
const unsigned char num[10] = { 14, 1, 5, 3, 11, 9, 7, 15, 13, 6 };

/*
 * for csrjmp subprocess 
 */
volatile int csr_active = 0;

pid_t csr_pid, spk_pid = 0;

/*
 * Function prototypes: 
 */

void chkscrsize ();

void allocoldscr ();

void cleanoldscr ();

void cpytooldscr (unsigned char *src);

void ggetproc (void);

int getsoftcsr (mask softcsr, range limit);

void screenfull ();

void initattr ();

void startbrl ();

void switchto (unsigned int scrno);     /* activate params for specified
                                           screen */
void csrjmp (int x, int y);     /* move cursor to (x,y) */

void csrjmp_sub (int x, int y); /* cursor routing subprocess */

void setwinxy (int x, int y);   /* move window to include (x,y) */

void message (char *s);         /* write literal message on

                                 * Braille display 
                                 */
void clrbrlstat (void);

void termination_handler (int signum);  /* clean up before termination */

void stop_child (int signum);   /* called at end of cursor routing */

void loadconfig (void);

int nice (int);                 /* should really be in a header file ... */

int brlcode = -1, autoprof = 1;

/* variables for fifo check */
fd_set w_fds;

int read_fd, write_fd;

brld_interface *brld;

static void *libbrld;

int elapsed_sec (struct timeval *start);

int elapsed_ms (struct timeval *start);

/* options */
struct option long_options[] = {
  {"version", 0, 0, 'v'},
  {"sppkdev", 1, 0, 'd'},
  {"log", 0, 0, 'l'},
  {"kbdid", 1, 0, 'i'},
  {"help", 0, 0, 'h'},
  {"auth", 1, 0, 'a'},
  {"port", 1, 0, 'p'},
  {"kbdport", 1, 0, 'k'},
  {"config", 1, 0, 'c'},
  {"speech", 1, 0, 's'},
  {0, 0, 0, 0}
};

int main (int argc, char *argv[])
{
  int lastkey = 0;

  int repeat = 0;

  int keypress;                 /* character received from braille display */

  int i, c = 0;

  struct timeval brld_down_time, kbd_time;

  int brld_down_event = 0;

  short csron = 1;              /* display cursor on (toggled during blink) */

  short csrcntr = 1;

  short oldwinx, oldwiny;

/* load sbl.conf */
  sprintf (sblconf_file, "%s/sbl.conf", CONFDIR);
  getldconf (&sblconf, sblconf_file);

/* parse cmdline options */
  while (1)
   {
     int option_index = 0;

     c =
       getopt_long (argc, argv, "a:c:d:hi:k:lp:s:", long_options,
                    &option_index);
     if (c == -1)
       break;

     switch (c)
      {
      case 0:
        printf (" option %s", long_options[option_index].name);
        if (optarg)
          printf (" with arg %s", optarg);
        printf ("\n");
        break;

      case 'h':
        printf ("%s\n%s\n%s\nconfdir=%s progpath=%s\n%s", VERSION, AUTHOR,
                COPYRIGHT, CONFDIR, PROGPATH, USAGE);
        return 0;

      case 'i':
        strcpy (sblconf.kbd_key, optarg);
        break;

      case 'a':
        strcpy (sblconf.brld_auth_key, optarg);
        break;
      case 'p':
        sblconf.brld_port = atoi (optarg);
        break;
      case 'k':
        sblconf.kbd_port = atoi (optarg);
        break;
      case 's':
        strcpy (sblconf.spkname, optarg);
        break;
      case 'd':
        strcpy (sblconf.spkport, optarg);
        break;
      case 'c':
        strcpy (sblconf_file, optarg);
        getldconf (&sblconf, sblconf_file);
        break;
      case 'l':
        sblconf.debug = 1;
        break;
      case '?':
        return 2;
        break;

      }
   }

  if (optind < argc)
   {
     fprintf (stderr, " non - option: ");
     return 2;
     while (optind < argc)
       fprintf (stderr, "%s", argv[optind++]);
     fprintf (stderr, "\n");

     return 2;
   }
  /* set default values if needed */
  if (!sblconf.sleep)
    sblconf.sleep = 45;
  if (!sblconf.repeatdelay)
    sblconf.repeatdelay = 6;
  if (!sblconf.brld_auth_key[0])
    strcpy (sblconf.brld_auth_key, "default");
  if (!sblconf.kbd_key[0])
    strcpy (sblconf.kbd_key, "default");
  if (!sblconf.spkname[0])
    strcpy (sblconf.spkname, "none");

  env.sound = 1;

  sbl_open_log (sblconf.debug);

  getprofile (&env, CONFDIR, home, PROFILE1);
  soundstat (env.sound);

  /* initialize braille */
  startbrl ();
  brld->cursor (env.csrsize);
  initattr ();

  if (strncmp (sblconf.spkname, "none", 4))
   {
     getspkctrl (CONFDIR, sblconf.spkname);
     getspkfilter (CONFDIR, env.profname, home, env.spklang);
   }

  hardcsr.x = -1;
  hardcsr.y = -1;
  puts (VERSION);
  sbl_log ("%s\n", VERSION);
  puts (AUTHOR);

/* external speechoutput must be initialized before we fork */
  if (strcmp (sblconf.spkname, "ttsynth")
      && strcmp (sblconf.spkname, "speechd"))
   {
     spkinit (sblconf.spkport);
     lang (env.spklang);
     setspk ();
   }

  /* 
   * Initialize screen library 
   */
  if (initscr ())
   {
     /* initialise screen reading */
   }

  /* allocate the first screen information structures */
  p = (sbl_param *) malloc (sizeof (sbl_param));
  if (!p)
   {
     sbl_log ("error: cannot allocate memory for screen\n");
     closescr ();
     sbl_close_log ();
     exit (-1);
   }
  *p = initparam;
  scrparam[0] = p;
  for (i = 1; i <= NBR_SCR; i++)
    scrparam[i] = 0;
  curscr = 0;

  /* 
   * Become a daemon: 
   */
  switch (fork ())
   {
   case -1:                    /* can't fork */
     perror ("fork()");
     closescr ();
     exit (3);
   case 0:                     /* child, process becomes a daemon: */
     close (STDIN_FILENO);
     close (STDOUT_FILENO);
     close (STDERR_FILENO);
     sbl_log ("become a daemon\n");
     if (setsid () == -1)
      {
        /* request a new session (job control) */
        closescr ();
        sbl_log ("setsid: %s", strerror (errno));
        sbl_close_log ();
        exit (4);
      }
     break;
   default:                    /* parent returns to calling process: */
     return 0;
   }

  getkeymap (&keymap, CONFDIR, sblconf, env.profname, home);

  sbl_log ("sniffercmd: on1=%d on2=%d off=%d\n", keymap.kbdsniffon1.kbd,
           keymap.kbdsniffon2.kbd, keymap.kbdsniffoff.kbd);
  if (kbd_open (sblconf.kbd_port) < 0)
    sbl_log ("can't connect to kbdsniffd\n");
  else if (kbd_auth (sblconf.kbd_key))
    sbl_log ("kbdsniffd auth failed\n");
  else if (kbd_setkey (keymap.kbdsniffon1.kbd))
    sbl_log ("kbdsniffd setkey failed\n");
  else
    kbd_sniff_on = 1;

  if (!kbd_sniff_on)
    sbl_log ("can't connect to kbdsniffd\n");

/* we have to load the ttsynth lib after fork */
  if (!strcmp (sblconf.spkname, "ttsynth")
      || !strcmp (sblconf.spkname, "speechd"))
   {
     spkinit (sblconf.spkport);
     lang (env.spklang);
     setspk ();
   }

  sbl_log ("braille display started\n");

  /* 
   * Establish signal handler to clean up before termination: 
   */
  if (signal (SIGTERM, termination_handler) == SIG_IGN)
    signal (SIGTERM, SIG_IGN);
  signal (SIGINT, SIG_IGN);
  signal (SIGHUP, SIG_IGN);
  signal (SIGPIPE, SIG_IGN);
  signal (SIGCHLD, stop_child);

  sbl_log ("signals set - brltbl loaded\n");
  message (VERSION);            /* display initialisation message */
  sbl_log ("wrote version to braille\n");
/* set braille translation to text */
  brld->texttbl ();
  sleep (1);                    /* sleep for a while */

  sbl_log ("get screenstatus and size\n");
  getstat (&scr);
  chkscrsize ();
  switchto (scr.no);            /* allocate current screen params */
  sbl_log ("finished - and current screen allocated\n");
  sbl_log ("set braille to xy\n");
  setwinxy (scr.posx, scr.posy);        /* set initial window position */
  oldwinx = p->winx;
  oldwiny = p->winy;
  p->cox = scr.posx;
  p->coy = scr.posy;
  sbl_log ("finished\n");
  allocoldscr ();
  sbl_log ("oldscreen allocated\n");
  /* 
   * Main program loop 
   */

  confd = open ("/dev/tty0", O_RDONLY);

  openfifo ();
  sbl_log ("go into mainloop\n");
  env.csrvis = 1;
  gettimeofday (&kbd_time, NULL);

  while (keep_going)
   {

     usleep (SLEEP * 1000);

     ioctl (confd, VT_GETSTATE, &vterm);

     if (connr != vterm.v_active)
      {
        connr = vterm.v_active;
      }

     if (autoprof)
       ggetproc ();

     soundstat (env.sound);

     if (kbd_sniff_on)
      {
        int key = 0;

        key = kbd_get ();
        if (key == 999)
          spkstop ();
        if (key > 0 && key != 999)
          selkey (keymap, 10000, key);
        else if (key < 0)
          kbd_sniff_on = 0;

        key = 0;
      }
     else
      {
        if (elapsed_sec (&kbd_time) > 3)
         {
           gettimeofday (&kbd_time, NULL);
           if (!kbd_sniffd_connect ())
            {
              sbl_log ("kbdsniffd reconnected\n");
              kbd_sniff_on = 1;
            }

         }                      /* if elapsed_sec */

      }                         /* else */

     if (repeat && lastkey)
      {
        repeat++;
        if (repeat > REPEATDELAY)
         {
           selkey (keymap, lastkey, 1);
           repeat = 1;
         }
      }

     keypress = 0;
     keypress = brld->getkey (&press);
     if (keypress == BRLD_NO_CONN)
      {
        int ret = 0;

        if (!brld_down_event)
         {
           gettimeofday (&brld_down_time, NULL);
           brld_down_event = 1;
         }
        else if ((ret = elapsed_sec (&brld_down_time)) > 2)
         {
           if (sblconf.brld_reconnect)
            {
              ret = brld_reconnect ();
              if (!ret)
               {
                 play (snd_detected);
                 getkeymap (&keymap, CONFDIR, sblconf, env.profname, home);
               }

              sbl_log ("brld_reconnect=%d\n", ret);
            }
           brld_down_event = 0;
         }
        keypress = 0;
        press = 0;
      }

     if (keypress > 0)
      {
        sbl_log ("key=%d p=%d l=%d\n", keypress, press, lastkey);
        if (!lastkey && press)
         {
           int ret = 0;

           lastkey = keypress;
           ret = selkey (keymap, lastkey, 1);
           if (ret == 0)
             lastkey = 0;
           else if (ret == 2)
             repeat = 1;

         }
        else if (lastkey == keypress && !press)
         {
           lastkey = (lastkey * 1000) + keypress;
           sbl_log ("release first key=%d\n", selkey (keymap, lastkey, 1));
           lastkey = 0;
           repeat = 0;
           env.csrvis = 1;
         }
        else if (lastkey && lastkey != keypress && press)
         {
           lastkey = (lastkey * 1000) + keypress;
           sbl_log ("press second key=%d\n", selkey (keymap, lastkey, 1));
           lastkey = 0;
           env.csrvis = 1;
         }

      }

     getstat (&scr);
     chkscrsize ();

     if (!(dispmd & (HELP_SCRN | FROZ_SCRN)) && curscr != scr.no)
       switchto (scr.no);

     /* cursor tracking */
     if (p->csrtrk)
      {

        /* If cursor moves while blinking is on */
        if (env.csrblink)
         {
           if ((env.csrmode ? attrpos.y : scr.posy) != p->coy)
            {
              /* turn off cursor to see what's under it while changing lines */
              csron = 0;
              csrcntr = env.csroffcnt;
            }
           if ((env.csrmode ? attrpos.x : scr.posx) != p->cox)
            {
              /* turn on cursor to see it moving on the line */
              csron = 1;
              csrcntr = env.csroncnt;
            }
         }
        /* If the cursor moves in cursor tracking mode: */
        switch (env.csrmode)
         {
         case 0:               /* folow the systemcursor */
           if ((scr.posx != p->cox || scr.posy != p->coy))
            {
              setwinxy (scr.posx, scr.posy);
              p->cox = scr.posx;
              p->coy = scr.posy;
              hardcsr.x = p->cox;
              hardcsr.y = p->coy;
            }
           break;
         case 1:               /* follow attributes */
         case 2:               /* follow systemcursor first then attributes */
          {
            if ((hardcsr.x != scr.posx || hardcsr.y != scr.posy)
                && env.csrmode == 2 && (scr.posx >= env.syscsrlimit.x1
                                        && scr.posx <= env.syscsrlimit.x2)
                && (scr.posy >= env.syscsrlimit.y1
                    && scr.posy <= env.syscsrlimit.y2))
             {

               setwinxy (scr.posx, scr.posy);
               hardcsr.x = scr.posx;
               hardcsr.y = scr.posy;
               attrpos = hardcsr;
               p->cox = scr.posx;
               p->coy = scr.posy;
             }
            else
             {
               if (!getsoftcsr (env.softcsr1, env.limitcsr1))
                 if (!getsoftcsr (env.softcsr2, env.limitcsr2))
                  {
                    int dum;

                    cleanoldscr ();
                    dum = getsoftcsr (env.softcsr1, env.limitcsr1);
                  }
               if (attrpos.x != p->cox || attrpos.y != p->coy)
                {
                  hardcsr = attrpos;
                  setwinxy (attrpos.x, attrpos.y);
                  p->cox = attrpos.x;
                  p->coy = attrpos.y;
                }
             }                  /* else */
          }

           break;
         case 3:               /* follow attributes first then systemcursor */
          {

            if (!getsoftcsr (env.softcsr1, env.limitcsr1))
              if (!getsoftcsr (env.softcsr2, env.limitcsr2))
               {
                 cleanoldscr ();
                 getsoftcsr (env.softcsr1, env.limitcsr1);
               }
            if (attrpos.x != p->cox || attrpos.y != p->coy)
             {
               setwinxy (attrpos.x, attrpos.y);
               p->cox = attrpos.x;
               p->coy = attrpos.y;
             }
            else if (hardcsr.x != scr.posx || hardcsr.y != scr.posy)
             {
               setwinxy (scr.posx, scr.posy);
               hardcsr.x = scr.posx;
               hardcsr.y = scr.posy;
               attrpos = hardcsr;
               p->cox = scr.posx;
               p->coy = scr.posy;
             }
          }

           break;
         }                      /* switch */
      }
     oldwinx = p->winx;
     oldwiny = p->winy;
/*  if(fcells>-1) */
     {
       statcells[0] = p->winy + 1;      /* y-pos of brailledisplay */
       statcells[1] = POSX;     /* x-pos of current cursor */
       statcells[2] = connr;    /* number of current console */
     }

     if (vertdisp)
       screenfull ();

     /* update the braille-status-cells */

     memset (spkstr, 0, sizeof (spkstr));       /* clean spkstr */
     getscr ((winpos)
             {

             0, p->winy, scr.cols, brl.y}
             , (unsigned char *) spkstr, p->dispmode ? SCR_ATTRIB : SCR_TEXT);

     /* Attribute underlining: if viewing text (not attributes), attribute
        underlining is active and visible and we're not in help, then we get
        the attributes for the current region and OR the underline. */

     memset (attrstr, 0, sizeof (attrstr));
     if (!p->dispmode && env.attrvis)
      {
        getscr ((winpos)
                {
                0, p->winy, scr.cols, brl.y}
                , attrbuf, SCR_ATTRIB);

        /* create the attribute string */
        for (i = p->winx; i < (p->winx + brl.x) * brl.y; i++)
         {
           unsigned char ma = attrbuf[i] & showattr.mask;

           unsigned char mv = showattr.mask & showattr.val;

           if (ma == mv)
             attrstr[i - p->winx] |= 0xC0;
           else
             attrstr[i - p->winx] = ' ';
         }

      }

     memcpy (brlstr, spkstr + p->winx, brl.x * brl.y);
     if (env.csrmode
         && (attrpos.x >= p->winx && attrpos.x < p->winx + brl.x
             && attrpos.y >= p->winy && attrpos.y < p->winy + brl.y))
      {
        csrpos =
          env.csrvis ? (attrpos.y - p->winy) * brl.x + attrpos.x -
          p->winx : -1;
        if (csrpos != old_csrpos || attrpos.y != spkcsrpos.y)
          spkcsrpos = attrpos;
      }
     else if (scr.posx >= p->winx && scr.posx < p->winx + brl.x
              && scr.posy >= p->winy && scr.posy < p->winy + brl.y)
      {
        csrpos =
          env.csrvis ? (scr.posy - p->winy) * brl.x + scr.posx - p->winx : -1;
        if (csrpos != old_csrpos || scr.posy != spkcsrpos.y)
         {
           spkcsrpos.x = scr.posx;
           spkcsrpos.y = scr.posy;
         }
      }
     else
       csrpos = -1;

     if (need_brl_upd || csrpos != old_csrpos
         || memcmp (old_brlstr, brlstr, sizeof (brlstr))
         || memcmp (old_attrstr, attrstr, sizeof (attrstr)))
      {
        sbl_log ("brlwrite: %d\n",
                 brld->write (csrpos, brlstr, env.attrvis ? attrstr : ""));

        need_brl_upd = 0;

        memcpy (old_attrstr, attrstr, sizeof (attrstr));
        memcpy (old_brlstr, brlstr, sizeof (brlstr));
        old_csrpos = csrpos;
      }

     spkline (spkcsr (0));

     do
      {
        chkfifo ();
      }
     while (sblpaus);

   }                            /* while (mainloop) */

  spkclose ();
  if (confd >= 0)
    close (confd);
  closefifo ();
  clrbrlstat ();
  message ("end of sbl");
  closescr ();

  brld->close ();
  if (kbd_sniff_on)
    kbd_close (1);
  if (libbrld)
    dlclose (libbrld);

  sleep (1);
  play (snd_brloff);
  for (i = 0; i <= NBR_SCR; i++)
   {
     free (scrparam[i]);
     scrparam[i] = 0;
     /* p is freed by this code */
   }

  sbl_close_log ();
  return 0;
}

void startbrl ()
{

  char *err = NULL;

  libbrld = dlopen ("./libbrld.so.1", RTLD_LAZY);
  if (libbrld == NULL)
    libbrld = dlopen ("libbrld.so.1", RTLD_LAZY);

  if (libbrld == NULL)
   {
     fprintf (stderr, "error: libbrld open failed\n");
     exit (2);
   }

  brld = (brld_interface *) dlsym (libbrld, "brld");

  err = dlerror ();

  if (err)
   {
     fprintf (stderr, "error: %s\n", err);
     dlclose (libbrld);
     exit (3);
   }

  if (brld->open ("localhost", sblconf.brld_port) < 0)
   {
     sbl_log ("error: connection to brld %s:%d failed\n",
              "localhost", sblconf.brld_port);

     brl.x = 80;
     brl.y = 1;
     strcpy (sblconf.brlname, "none");
   }
  else if (brld->reg (TEXT, sblconf.brld_auth_key))
   {
     sbl_log ("error: brld registration failed\n");
     brl.x = 80;
     brl.y = 1;
     strcpy (sblconf.brlname, "none");
   }
  else
   {
     brld->getxy (&brl.x, &brl.y);
     if (brl.x <= 0)
       brl.x = 80;
     if (brl.y <= 0)
       brl.y = 1;
     brld->getalias (sblconf.brlname);
     play (snd_detected);
     /* clrbrlstat (); */
   }

}

void switchto (unsigned int scrno)
{
  curscr = scrno;
  if (scrno > NBR_SCR)
    scrno = 0;
  if (!scrparam[scrno])
   {                            /* if not already allocated... */
     if (!(scrparam[scrno] = (sbl_param *) malloc (sizeof (*p))))
       scrno = 0;               /* unable to allocate a new structure */
     else
       *scrparam[scrno] = initparam;
   }
  p = scrparam[scrno];
}

void setwinxy (int x, int y)
{
  if (env.slidewin)
   {
     /* Change position only if the coordonates are not already displayed */
     if (x < p->winx || x >= p->winx + brl.x ||
         y < p->winy || y >= p->winy + brl.y)
      {
        p->winy = y < brl.y - 1 ? 0 : y - (brl.y - 1);
        if (x < brl.x)
          p->winx = 0;
        else if (x >= scr.cols - csr_offright)
          p->winx = scr.cols - brl.x;
        else
          p->winx = x - (brl.x - csr_offright);
      }
   }
  else
   {
     if (x < p->winx || x >= p->winx + brl.x)
       p->winx = x >= (scr.cols / brl.x) * brl.x ? scr.cols - brl.x :
         (x / brl.x) * brl.x;
     if (y < p->winy || y >= p->winy + brl.y)
       p->winy = y < brl.y - 1 ? 0 : y - (brl.y - 1);
   }
}

void csrjmp (int x, int y)
{
  /* 
   * Fork cursor routing subprocess. * First, we must check if a
   * subprocess is already running: if so, we * send it a SIGUSR1 and
   * wait for it to die. 
   */
  signal (SIGCHLD, SIG_IGN);    /* ignore SIGCHLD for the moment */
  if (csr_active)
   {
     kill (csr_pid, SIGUSR1);
     wait (NULL);
     csr_active = 0;
   }
  signal (SIGCHLD, stop_child); /* re-establish handler */

  csr_active = 1;
  switch (csr_pid = fork ())
   {
   case -1:                    /* fork failed */
     csr_active = 0;
     break;
   case 0:                     /* child, cursor routing process */
     nice (CSRJMP_NICENESS);    /* reduce scheduling priority */
     csrjmp_sub (x, y);
     exit (0);                  /* terminate child process */
   default:                    /* parent waits for child to return */
     break;
   }
}

void csrjmp_sub (int x, int y)
{
  int curx, cury;               /* current cursor position */

  int dif, t = 0;

  sigset_t mask;                /* for blocking of SIGUSR1 */

  /* Set up signal mask: */
  sigemptyset (&mask);
  sigaddset (&mask, SIGUSR1);

  /* Initialise second thread of screen reading: */
  if (initscr_phys ())
    return;

  getstat_phys (&scr);
  chkscrsize ();
  /* Deal with vertical movement first, ignoring horizontal jumping ... */
  dif = y - scr.posy;
  while (dif != 0 && curscr == scr.no)
   {
     timeout_yet (0);           /* initialise stop-watch */
     sigprocmask (SIG_BLOCK, &mask, NULL);      /* block SIGUSR1 */
     inskey (dif > 0 ? (unsigned char *) DN_CSR : (unsigned char *) UP_CSR);
     sigprocmask (SIG_UNBLOCK, &mask, NULL);    /* unblock SIGUSR1 */
     do
      {
#if CSRJMP_LOOP_DELAY > 0
        delay (CSRJMP_LOOP_DELAY);      /* sleep a while ... */
#endif
        cury = scr.posy;
        curx = scr.posx;
        getstat_phys (&scr);
        chkscrsize ();
      }
     while (!(t = timeout_yet (CSRJMP_TIMEOUT)) &&
            scr.posy == cury && scr.posx == curx);
     if (t)
       break;
     if ((scr.posy == cury && (scr.posx - curx) * dif <= 0) ||
         (scr.posy != cury && (y - scr.posy) * (y - scr.posy) >= dif * dif))
      {
        delay (CSRJMP_SETTLE_DELAY);
        getstat_phys (&scr);
        chkscrsize ();
        if ((scr.posy == cury && (scr.posx - curx) * dif <= 0) ||
            (scr.posy != cury
             && (y - scr.posy) * (y - scr.posy) >= dif * dif))
         {
           /* We are getting farther from our target... Let's try to go back
              to the previous position wich was obviously the nearest ever
              reached before gibing up. */
           sigprocmask (SIG_BLOCK, &mask, NULL);        /* block SIGUSR1 */
           inskey ((unsigned char *) (dif < 0 ? DN_CSR : UP_CSR));
           sigprocmask (SIG_UNBLOCK, &mask, NULL);      /* unblock SIGUSR1 */
           break;
         }
      }
     dif = y - scr.posy;
   }

  if (x >= 0)
   {                            /* don't do this for vertical-only routing
                                   (x=-1) */
     /* Now horizontal movement, quitting if the vertical position is wrong: */
     dif = x - scr.posx;
     while (dif != 0 && scr.posy == y && curscr == scr.no)
      {
        timeout_yet (0);        /* initialise stop-watch */
        sigprocmask (SIG_BLOCK, &mask, NULL);   /* block SIGUSR1 */
        inskey (dif >
                0 ? (unsigned char *) RT_CSR : (unsigned char *) LT_CSR);
        sigprocmask (SIG_UNBLOCK, &mask, NULL); /* unblock SIGUSR1 */
        do
         {
#if CSRJMP_LOOP_DELAY > 0
           delay (CSRJMP_LOOP_DELAY);   /* sleep a while ... */
#endif
           curx = scr.posx;
           getstat_phys (&scr);
           chkscrsize ();
         }
        while (!(t = timeout_yet (CSRJMP_TIMEOUT)) &&
               scr.posx == curx && scr.posy == y);
        if (t)
          break;
        if (scr.posy != y || (x - scr.posx) * (x - scr.posx) >= dif * dif)
         {
           delay (CSRJMP_SETTLE_DELAY);
           getstat_phys (&scr);
           chkscrsize ();
           if (scr.posy != y || (x - scr.posx) * (x - scr.posx) >= dif * dif)
            {
              /* We probably wrapped on a short line... or are getting
                 farther from our target. Try to get back to the previous
                 position which was obviously the nearest ever reached before 
                 we exit. */
              sigprocmask (SIG_BLOCK, &mask, NULL);     /* block SIGUSR1 */
              inskey (dif > 0 ? (unsigned char *) LT_CSR : (unsigned char *)
                      RT_CSR);
              sigprocmask (SIG_UNBLOCK, &mask, NULL);   /* unblock SIGUSR1 */
              break;
            }
         }
        dif = x - scr.posx;
      }
   }

  closescr_phys ();             /* close second thread of screen reading */
}

void message (char *s)
{
  brld->write (-1, s, NULL);
}

void clrbrlstat (void)
{
  memset (statcells, 0, sizeof (statcells));
/*####  braille->setstatus (statcells);*/
}

void termination_handler (int signum)
{
  keep_going = 0;
  signal (signum, termination_handler);
}

void stop_child (int signum)
{
  signal (signum, stop_child);
  wait (NULL);
  csr_active = 0;
}

void chkscrsize ()
{
  if ((scrx != COLS) || (scry != ROWS))
   {
     sbl_log ("oldx=%i oldy=%i newx=%i newy=%i\n", scrx, scry, COLS, ROWS);
     scrx = COLS;
     scry = ROWS;
     free (oldscr);
     oldscr = 0;
     sbl_log ("free oldscr\n");
     allocoldscr ();
     sbl_log ("new alloc oldscr\n");
   }

}

void allocoldscr ()
{
  int i;

  oldscr = malloc (sizeof (unsigned char *) * ROWS);

  for (i = 0; i < ROWS; i++)
   {
     oldscr[i] = (unsigned char *) malloc (COLS);
     memset (oldscr[i], 0, COLS);
   }

}

void cpytooldscr (unsigned char *src)
{
  int y;

  for (y = 0; y < ROWS; y++)
    memcpy (oldscr[y], src + y * COLS, COLS);
}

void cleanoldscr ()
{

  int i;

  for (i = 0; i < ROWS; i++)
   {
     sbl_log ("#%i# %s \n", i, oldscr[i]);
     memset (oldscr[i], 0, COLS);
   }

}

int getsoftcsr (mask softcsr, range limit)
{
  short int i, hit = 0, j;

  unsigned char line[ROWS][COLS];

  unsigned char tmpline[COLS * ROWS];

  if (limit.y1 > ROWS)
    limit.y1 = ROWS;
  if (limit.y2 > ROWS)
    limit.y2 = ROWS;
  if (limit.x1 > COLS)
    limit.x1 = COLS;
  if (limit.x2 > COLS)
    limit.x2 = COLS;

  getscr ((winpos)
          {
          0, 0, COLS, ROWS}
          , tmpline, SCR_ATTRIB);
  memcpy (line, tmpline, sizeof (line));
  for (i = limit.y1; i < limit.y2; i++)
    for (j = limit.x1; j < limit.x2; j++)
      if (oldscr[i][j] != line[i][j])
       {
         unsigned char ma = line[i][j] & softcsr.mask;

         unsigned char mv = softcsr.mask & softcsr.val;

         short t;

         hit = 1;
         if (ma == mv)
          {
            attrpos.y = i;
            for (t = j;
                 t > limit.x1
                 && (line[i][t] & softcsr.mask) ==
                 (softcsr.mask & softcsr.val); t--);
            if (t == limit.x1
                && (line[i][t] & softcsr.mask) ==
                (softcsr.mask & softcsr.val))
              attrpos.x = t;
            else
              attrpos.x = t + 1;
            cpytooldscr (line[0]);
            return 1;
          }
       }

/*    memcpy(oldscr,line,scr)); */
  if (hit)
    return 0;
  else
    return 1;

}

void screenfull ()
{
  int dots1[] = { 64, 4, 2, 1 };
  int dots2[] = { 128, 32, 16, 8 };
  char line[22];

  int i, y, parts, cell = 0, celldisp = 0;

  line[21] = 0;
  for (y = 0; y < scr.rows; y++)
   {

     if (!(y % 2))
      {
        cell++;
        celldisp = 0;
      }
     for (parts = 0; parts < 4; parts++)
      {
        celldisp = 0;

        line[0] = 0;
        getscr ((winpos)
                {
                20 * parts, y, 20, 1}
                , (unsigned char *) line, SCR_TEXT);
        for (i = 0; i <= 20 && strlen (line); i++)
          if (line[i] > 32)
           {
             if (!(y % 2))
               celldisp += dots1[parts];
             else
               celldisp += dots2[parts];
             i = 21;
           }

      }
/*   statcells[-1+cell]=celldisp; */
     statcells[fcells + cell] = celldisp;
   }

}

void initattr ()
{
  switch (env.attrnr)
   {
   case 1:
     showattr = env.attr1;
     break;
   case 2:
     showattr = env.attr2;
     break;
   case 3:
     showattr = env.attr3;
     break;
   case 4:
     showattr = env.attr4;
   }
}

void ggetproc (void)
{
  char str[100] = "";

  getfgproc (str, connr);
  if (!strcmp (str, env.profname))
    return;
  sbl_log ("applicatoon changed on console=%i\n", connr);

  gethome (home, connr);
  sbl_log ("home=%s ", home);

  getprofile (&env, CONFDIR, home, str);
  brld->cursor (env.csrsize);
  sbl_log ("application=%s profile=%s csr=%d\n", env.profname, str,
           env.csrsize);
  getkeymap (&keymap, CONFDIR, sblconf, env.profname, home);
  setspk ();
  getspkfilter (CONFDIR, env.profname, home, env.spklang);

  initattr ();

}

void openfifo ()
{
/* Create fifo if it does not exist yet. -KK */
  (void) mkfifo(FIFO_IN, S_IRUSR|S_IWUSR);
  (void) mkfifo(FIFO_OUT, S_IWUSR|S_IRUSR);
      
  read_fd = open (FIFO_IN, O_NDELAY);
  write_fd = open (FIFO_OUT, O_RDWR);

  if (read_fd < 0 || write_fd < 0)
   {
     fifo_ok = 0;
     sbl_log ("fifo error: read=%d write=%d\n", read_fd, write_fd);
     return;
   }
  FD_ZERO (&w_fds);
  FD_SET (write_fd, &w_fds);

}                               /* openfifo */

void chkfifo ()
{
  int len;                      /* length of input-string */

  char writebuf[100];           /* writebuffer */

  char readbuf[100];            /* input-buffer */

  if (!fifo_ok)
    return;
  len = read (read_fd, readbuf, 80);
  if (len > 1)
   {
     readbuf[len - 1] = 0;
     fifocommands (writebuf, readbuf);
     readbuf[0] = 0;

     write (write_fd, writebuf, strlen (writebuf));
   }                            /* if */
}                               /* chkfifo */

void fifocommands (char *out, char *cmd)
{
  if (!strcmp (cmd, "nextvol"))
   {
     char tmpstr[80] = "1 2 3 4";

     nextvol ();
     spkwrite (tmpstr);
     sprintf (out, "01: nextvol\n");
     return;
   }
  else if (!strcmp (cmd, "prevvol"))
   {
     char tmpstr[80] = "1 2 3 4";

     prevvol ();
     spkwrite (tmpstr);
     sprintf (out, "01: prevvol\n");
     return;
   }
  else if (!strcmp (cmd, "prevspd"))
   {
     char tmpstr[80] = "1 2 3 4";

     prevspd ();
     spkwrite (tmpstr);
     sprintf (out, "01: prevspd\n");
     return;
   }
  else if (!strcmp (cmd, "nextspd"))
   {
     char tmpstr[80] = "1 2 3 4";

     nextspd ();
     spkwrite (tmpstr);
     sprintf (out, "01: nextspd\n");
     return;
   }
  else if (!strcmp (cmd, "spkscrfromln"))
   {
     spkmode = SPKNORM;
     spkscrfromln ();
     sprintf (out, "01: spkscrfromln\n");
     return;
   }
  else if (!strcmp (cmd, "lnup"))
   {
     lnup ();
     sprintf (out, "01: %s done\n", cmd);
     return;
   }
  else if (!strcmp (cmd, "lndn"))
   {
     lndn ();
     sprintf (out, "01: %s done\n", cmd);
     return;
   }
  else if (!strcmp (cmd, "lnlft"))
   {
     lnlft ();
     sprintf (out, "01: %s done\n", cmd);
     return;
   }
  else if (!strcmp (cmd, "lnrgt"))
   {
     lnrgt ();
     sprintf (out, "01: %s done\n", cmd);
     return;
   }
  else if (!strcmp (cmd, "botleft"))
   {
     botleft ();
     sprintf (out, "01: %s done\n", cmd);
     return;
   }
  else if (!strcmp (cmd, "topleft"))
   {
     topleft ();
     sprintf (out, "01: %s done\n", cmd);
     return;
   }
  else if (!strcmp (cmd, "spkcurln"))
   {
     spkcurln ();
     sprintf (out, "01: %s done\n", cmd);
     return;
   }
  else if (!strcmp (cmd, "csrtrk"))
   {
     csrtrk ();
     sprintf (out, "01: %s done\n", cmd);
     return;
   }
  else if (!strcmp (cmd, "csrblink on"))
   {
     env.csrblink = 1;
     sprintf (out, "01: %s done\n", cmd);
     return;
   }

  if (!strcmp (cmd, "csrblink off"))
   {
     env.csrblink = 0;
     sprintf (out, "01: %s done\n", cmd);
     return;
   }
  if (!strncmp (cmd, "resetbrl", 8))
   {
     resetbrl ();
     sprintf (out, "01: %s done\n", cmd);
     return;
   }
  if (!strncmp (cmd, "sblpaus", 7))
   {
     /* TOD: send brl_dev_close to brld */
     play (snd_brloff);
     sprintf (out, "01: %s done\n", cmd);
     sblpaus = 1;
     return;
   }
  if (!strncmp (cmd, "sblresume", 9) && sblpaus)
   {
     startbrl ();
     sprintf (out, "01: %s done\n", cmd);
     sblpaus = 0;
     return;
   }
  if (!strncmp (cmd, "setline", 7))
   {
     int nr = atoi (cmd + 7);

     if (nr >= 0 && nr < ROWS)
      {
        p->winy = nr;
      }
     sprintf (out, "01: setline\n");
     return;
   }
  else if (!strncmp (cmd, "spkoff_setline", 14))
   {
     int nr = atoi (cmd + 14);

     if (nr >= 0 && nr < ROWS)
      {
        p->winy = nr;
/*   spkmode=SPKOFF;*/
      }
     sprintf (out, "01: spkoff_setline\n");
     return;
   }
  else if (!strncmp (cmd, "spkoff", 6))
   {
     spkmode = SPKOFF;
     sprintf (out, "01: spkoff\n");
     return;
   }
  else if (!strncmp (cmd, "spktoggle", 6))
   {
     if (spkmode == SPKOFF)
       spkmode = SPKNORM;
     else
       spkmode = SPKOFF;
     sprintf (out, "01: spktoggle\n");
     return;
   }
  else if (!strncmp (cmd, "spkon", 5))
   {
     spkmode = SPKNORM;
     sprintf (out, "01: spkon\n");
     return;
   }
  else if (!strncmp (cmd, "insert", 6))
   {
     if (strlen (cmd) > 7)
      {
        inskey ((unsigned char *) cmd + 7);
        sprintf (out, "01: %s done\n", cmd);
      }
     else
       sprintf (out, "02: %s failed\n", cmd);
     return;
   }
  if (!strcmp (cmd, "getcurline"))
   {

     sprintf (out, "01: %s\n", spkstr);
     return;
   }
  if (!strncmp (cmd, "getline", 7))
   {
     int number = 0;

     char linestr[COLS + 1];

     memset (linestr, 0, sizeof (linestr));
     if (strlen (cmd) < 8)
      {
        sprintf (out, "03: %s failed\n", cmd);
        return;
      }

     if (!(number = atoi (cmd + 8)))
      {
        sprintf (out, "04: %s failed %s %i\n", cmd, cmd + 8, atoi (cmd + 8));
        return;
      }

     if (number == 0 || number > ROWS)
      {
        sprintf (out, "02: line number %i not in 1..%i\n", number, ROWS);
        return;
      }
     getscr ((winpos)
             {
             0, number - 1, COLS, 1}
             , (unsigned char *) linestr, SCR_TEXT);
     sprintf (out, "01: %s\n", linestr);
     return;
   }

  else
    sprintf (out, "02: error %s is not a sbl command\n", cmd);
  return;
}                               /* fifo commands */

void closefifo ()
{

  if (read_fd >= 0)
    close (read_fd);
  if (write_fd >= 0)
    close (write_fd);
}

/* functions for navigation */

void autoprofonoff ()
{

  if (autoprof)
   {
     play (snd_toggleoff);
     autoprof = 0;
   }
  else
   {
     play (snd_toggleon);
     autoprof = 1;
   }
}

void resetbrl ()
{
  play (snd_brloff);
  brld->reset ();
  need_brl_upd = 1;
  play (snd_detected);
}

void reset_speech ()
{
  play (snd_brloff);
  spkclose ();
  usleep(10000);
  if (!strcmp (sblconf.spkname, "ttsynth")
      || !strcmp (sblconf.spkname, "speechd"))
   {
     spkinit (sblconf.spkport);
     lang (env.spklang);
     setspk ();
   }
  play (snd_detected);
}

void line01 ()
{
  p->winy = 0;
  spkcsrpos.y = 0;
}

void topleft ()
{
  p->winy = 0;
  p->winx = 0;
  spkcsrpos.y = 0;
}

void botleft ()
{
  p->winy = scr.rows - brl.y;
  p->winx = 0;
  spkcsrpos.y = p->winy;
}

void winup ()
{

  if (p->winy == 0)
    play (snd_bounce);
  p->winy = MAX (p->winy - vwinshift, 0);
  spkcsrpos.y = p->winy;
}

void windn ()
{
  if (p->winy == scr.rows - brl.y)
    play (snd_bounce);
  p->winy = MIN (p->winy + vwinshift, scr.rows - brl.y);
  spkcsrpos.y = p->winy;
}

void lnlft ()
{
  if (p->winx)
   {
     if (p->winx >= brl.x)
       p->winx -= brl.x;
     else
       p->winx = 0;
   }
  else if (p->winy)
   {
     p->winy--;
     p->winx = COLS - brl.x;
   }
  spkcsrpos.y = p->winy;
}

void lnrgt ()
{
  if (p->winx < (COLS - brl.x))
   {
     if ((p->winx + brl.x) <= (COLS - brl.x))
       p->winx += brl.x;
     else
       p->winx = COLS - brl.x;
   }
  else if (p->winy < ROWS - brl.y)
   {
     p->winy++;
     p->winx = 0;
   }
  spkcsrpos.y = p->winy;
}

void lnup ()
{

  spk = 1;
  if (p->winy == 0)
    play (snd_bounce);

  else
   {
     p->winy--;
     spkcsrpos.y = p->winy;
   }
}

void lndn ()
{
  spk = 1;
  if (p->winy == scr.rows - brl.y)
    play (snd_bounce);
  else
   {
     p->winy++;
     spkcsrpos.y = p->winy;
   }
}

void csrtrk ()
{

  switch (env.csrmode)
   {
   case 0:
     setwinxy (scr.posx, scr.posy);
     spkcsrpos.x = scr.posx;
     spkcsrpos.y = scr.posy;
     break;
   case 1:
   case 2:
   case 3:
     setwinxy (attrpos.x, attrpos.y);
     spkcsrpos = attrpos;
     break;
   }
  p->csrtrk = 1;
  spkcsr (1);
}

void syscsr ()
{
  env.csrmode = 0;
  p->csrtrk = 1;
}

void softcsr ()
{
  env.csrmode = 1;
  p->csrtrk = 1;
}

void chrlft ()
{
  if (p->winx == 0)
    play (snd_bounce);
  p->winx = MAX (p->winx - 1, 0);
}

void chrrgt ()
{
  if (p->winx == scr.cols - brl.x)
    play (snd_bounce);
  p->winx = MIN (p->winx + 1, scr.cols - brl.x);
}

void hwinlft ()
{
  if (p->winx == 0)
    play (snd_bounce);
  p->winx = MAX (p->winx - hwinshift, 0);
}

void hwinrgt ()
{
  if (p->winx == scr.cols - brl.x)
    play (snd_bounce);
  p->winx = MIN (p->winx + hwinshift, scr.cols - brl.x);
}

void fwinlft ()
{
  if (p->winx == 0 && p->winy > 0)
   {
     p->winx = scr.cols - brl.x;
     p->winy--;
     play (snd_wrap_up);
   }
  else if (p->winx == 0 && p->winy == 0)
    play (snd_bounce);
  else
    p->winx = MAX (p->winx - fwinshift, 0);
  spkcsrpos.y = p->winy;
}

void fwinrgt ()
{
  if (p->winx == scr.cols - brl.x && p->winy < scr.rows - brl.y)
   {
     p->winx = 0;
     p->winy++;
     play (snd_wrap_down);
   }
  else if (p->winx == scr.cols - brl.x && p->winy == scr.rows - brl.y)
    play (snd_bounce);
  else
    p->winx = MIN (p->winx + fwinshift, scr.cols - brl.x);
  spkcsrpos.y = p->winy;
}

void csrjump ()
{
  if ((dispmd & HELP_SCRN) != HELP_SCRN)
    csrjmp (p->winx, p->winy);
}

void csrjmpvert ()
{
  if ((dispmd & HELP_SCRN) != HELP_SCRN)
    csrjmp (-1, p->winy);
}

void keyup ()
{
  if ((dispmd & HELP_SCRN) != HELP_SCRN)
    inskey ((unsigned char *) UP_CSR);
}

void keydn ()
{
  if ((dispmd & HELP_SCRN) != HELP_SCRN)
    inskey ((unsigned char *) DN_CSR);
}

void keyrgt ()
{
  if ((dispmd & HELP_SCRN) != HELP_SCRN)
    inskey ((unsigned char *) RT_CSR);
}

void keylft ()
{
  if ((dispmd & HELP_SCRN) != HELP_SCRN)
    inskey ((unsigned char *) LT_CSR);
}

void keyenter ()
{
  if ((dispmd & HELP_SCRN) != HELP_SCRN)
    inskey ((unsigned char *) KEY_RETURN);
}

void keyesc ()
{
  char esc[2];

  sprintf (esc, "%c", 27);
  inskey ((unsigned char *) esc);
}

void csrvisoff ()
{
  spkstop ();
  env.csrvis = 0;
}

void jmpmark1 ()
{
  p->winy = env.mark1 - 1;
  spkcsrpos.y = p->winy;
}

void jmpmark2 ()
{
  p->winy = env.mark2 - 1;
  spkcsrpos.y = p->winy;
}

void jmpmark3 ()
{
  p->winy = env.mark3 - 1;
  spkcsrpos.y = p->winy;
}

void jmpmark4 ()
{
  p->winy = env.mark4 - 1;
  env.csrvis = 1;
  spkcsrpos.y = p->winy;
}

void attr1 ()
{
  env.attrvis = 1;
  showattr = env.attr1;
}

void attr2 ()
{
  env.attrvis = 1;
  showattr = env.attr2;
}

void attr3 ()
{
  env.attrvis = 1;
  showattr = env.attr3;
}

void attr4 ()
{
  env.attrvis = 1;
  showattr = env.attr4;
}

void csrtrkoff ()
{
  p->csrtrk = 0;
  play (snd_toggleoff);
}

void prof1 ()
{
  int snd[] = { 1000, 80, 0 };

  autoprof = 0;
  getprofile (&env, CONFDIR, home, PROFILE1);
  brld->cursor (env.csrsize);
  initattr ();
  getkeymap (&keymap, CONFDIR, sblconf, env.profname, home);
  setspk ();
  getspkfilter (CONFDIR, env.profname, home, env.spklang);

  play (snd);

}

void prof2 ()
{
  int snd[] = { 1000, 80, 1, 40, 1000, 80, 0 };

  autoprof = 0;
  getprofile (&env, CONFDIR, home, PROFILE2);
  brld->cursor (env.csrsize);
  initattr ();
  getkeymap (&keymap, CONFDIR, sblconf, env.profname, home);

  setspk ();
  getspkfilter (CONFDIR, env.profname, home, env.spklang);

  play (snd);
}

void prof3 ()
{
  int snd[] = { 1000, 80, 1, 40, 1000, 80, 1, 40, 1000, 80, 0 };

  autoprof = 0;
  getprofile (&env, CONFDIR, home, PROFILE3);
  brld->cursor (env.csrsize);
  initattr ();
  getkeymap (&keymap, CONFDIR, sblconf, env.profname, home);
  setspk ();
  getspkfilter (CONFDIR, env.profname, home, env.spklang);

  play (snd);
}

void prof4 ()
{
  int snd[] = { 1000, 80, 1, 40, 1000, 80,
    1, 40, 1000, 80, 1, 40, 1000, 80, 0
  };

  autoprof = 0;
  getprofile (&env, CONFDIR, home, PROFILE4);
  brld->cursor (env.csrsize);
  initattr ();
  getkeymap (&keymap, CONFDIR, sblconf, env.profname, home);
  setspk ();
  getspkfilter (CONFDIR, env.profname, home, env.spklang);

  play (snd);
}

void attroff ()
{
  env.attrvis = 0;
}

void attrmodonoff ()
{
  if (p->dispmode)
   {
     p->dispmode = 0;
     brld->texttbl ();
     need_brl_upd = 1;
   }
  else
   {
     brld->attrtbl ();
     need_brl_upd = 1;
     p->dispmode = 1;
   }
}

void sixdotsonoff ()
{
  if (env.sixdots)
    env.sixdots = 0;
  else
    env.sixdots = 1;
}

void csrblinkonoff ()
{
  if (env.csrblink)
    env.csrblink = 0;
  else
    env.csrblink = 1;
}

void csrblockonoff ()
{
  if (env.csrsize)
   {
     env.csrsize = 0;
     brld->cursor (env.csrsize);
   }
  else
   {
     env.csrsize = 1;
     brld->cursor (env.csrsize);
   }

  need_brl_upd = 1;
}

void cutbegin ()
{
  cut_begin (p->winx, p->winy);
}

void cutend ()
{
  cut_end (p->winx + brl.x - 1, p->winy + brl.y - 1);
}

void paste ()
{
  if ((dispmd & HELP_SCRN) != HELP_SCRN)
    cut_paste ();
}

int linerouting (int key)
{
  int val = -1;

/* is it a lineroutingkey */
  if (key >= keymap.lineroutbeg && key <= keymap.lineroutend)
    val = key - keymap.lineroutbeg;

/* is val in range of linrouting */
  if (val >= 0 && val <= 13)
   {
     if (p->winy != val * 2 || val == 13)
       p->winy = val * 2;
     else
       p->winy = val * 2 + 1;
     if (p->winy > ROWS)
       p->winy = ROWS;
     return 1;
   }

  spkcsrpos.y = p->winy;
  return 0;
}

int csrrouting (int key)
{
  int val = -1;

/* is key a csrroutingkey */
  if (key >= keymap.csrroutbeg && key <= keymap.csrroutend)
    val = key - keymap.csrroutbeg;

/* is val in range of csrjmp */
  if (val >= 0 && val <= brl.x)
   {
     csrjmp (p->winx + val, p->winy);
     return 1;
   }

  spkcsrpos.y = p->winy;
  return 0;

}

void loadkeymap ()
{
  play (snd_toggleon);
  getkeymap (&keymap, CONFDIR, sblconf, env.profname, home);
}

void soundonoff ()
{
  if (env.sound)
   {
     play (snd_toggleoff);
     env.sound = 0;
   }
  else
   {
     env.sound = 1;
     soundstat (env.sound);
     play (snd_toggleon);
   }
}

void spkline (int spk)
{
  static int y = 0;

  if (y == p->winy)
    return;
  y = p->winy;
  if (spk)
    return;
  spkwrite (spkstr);
}

void spkmod ()
{
  if (spkmode == SPKNORM)
    spkmode = SPKSPELL;
  else
    spkmode = SPKNORM;

  play (snd_toggleon);

}

void spkoff ()
{
  spkmode = SPKOFF;
  play (snd_toggleoff);
}

void spkstop ()
{
  spkwrite ("");
}

int spkcsr (int mode)
{

#define SPK_MOVE_NORM 1
#define SPK_MOVE_BS 2
#define SPK_MOVE_DEL 4
  int i, j = 0;

  char str[MAXSPKSTR];

  int move = 0;

  char ch = 0;

  static unsigned char oldstr[500] = "";

  static int x = 0, y = 0;
  mask spkattr = { 0, 0 };;

  if (mode || (memcmp (oldstr, spkstr, COLS) && p->winy == p->coy)
      || p->cox != x || p->coy != y)
   {
     if (memcmp (oldstr, spkstr, COLS))
      {
        move = SPK_MOVE_NORM;
/* check for char left from cursor is deleted (backspace) */
        if (p->cox + 1 == x)

          if (!memcmp (oldstr, spkstr, p->cox)
              && !memcmp (oldstr + x, spkstr + p->cox, COLS - x))
           {
             move = SPK_MOVE_BS;
             ch = oldstr[p->cox];
           }

        if (x == p->cox)
          if (!memcmp (oldstr, spkstr, x)
              && !memcmp (oldstr + x + 1, spkstr + x, COLS - x - 1))
           {
             move = SPK_MOVE_DEL;
             ch = oldstr[x];
           }

        if (x + 1 == p->cox)
          if (!memcmp (oldstr, spkstr, x)
              && !memcmp (oldstr + x, spkstr + x + 1, COLS - x - 1))
           {
             move = SPK_MOVE_DEL;
             ch = spkstr[x];
           }
      }

     memcpy (oldstr, spkstr, COLS);
     x = p->cox;
     if (x < 0)
       x = 0;
     if (y != p->coy)
       move = SPK_MOVE_NORM;
     y = p->coy;
   }

  else
   {
     memcpy (oldstr, spkstr, COLS);
     return 0;
   }

  memset (str, 0, MAXSPKSTR);
  if (env.spkcharmod)
   {
/*   if(move) sbl_log("charmod: %d\n",move);*/
     switch (move)
      {
      case 0:
        strncpy (str, spkstr + p->cox, 1);
        spkpunctuation (1);
        if (str[0] >= 'A' && str[0] <= 'Z')
          frq (8);
        spkwrite (str);
        if (str[0] >= 'A' && str[0] <= 'Z')
          frq (env.spkfrq);
        spkpunctuation (0);
        return 1;
      case SPK_MOVE_NORM:
        break;
      case SPK_MOVE_BS:
      case SPK_MOVE_DEL:
        sprintf (str, "%c\n", ch);
        spkpunctuation (1);
        if (str[0] >= 'A' && str[0] <= 'Z')
          frq (8);
        spkwrite (str);
        if (str[0] >= 'A' && str[0] <= 'Z')
          frq (env.spkfrq);
        spkpunctuation (0);
        return 1;
      }
   }

  if ((attrbuf[x] & env.attr1.mask) == (env.attr1.mask & env.attr1.val))
    spkattr = env.attr1;
  else if ((attrbuf[x] & env.attr2.mask) == (env.attr2.mask & env.attr2.val))
    spkattr = env.attr2;
  else if ((attrbuf[x] & env.attr3.mask) == (env.attr3.mask & env.attr3.val))
    spkattr = env.attr3;
  else if ((attrbuf[x] & env.attr4.mask) == (env.attr4.mask & env.attr4.val))
    spkattr = env.attr4;

  if (env.spkcharmod && move)
   {
     spkwrite (spkstr);
     return 1;
   }

  for (i = x; i < COLS; i++)
   {
     unsigned char ma = attrbuf[i] & spkattr.mask;

     unsigned char mv = spkattr.mask & spkattr.val;

     if (ma == mv)
      {
        str[j] = spkstr[i];
        j++;
      }
   }

  spkwrite (str);
  return 1;
}

void setmark ()
{

  jmpmark = p->winy;
  play (snd_toggleon);
}

void jmptomark ()
{

  p->winy = jmpmark;
  spkcsrpos.y = p->winy;
}

void nextfrq ()
{
  spkstop ();
  if (frq (env.spkfrq + 1))
   {
     env.spkfrq++;
   }

}

void prevfrq ()
{
  spkstop ();
  if (frq (env.spkfrq - 1))
   {
     env.spkfrq--;
   }
}

void nextspd ()
{
  spkstop ();
  if (speed (env.spkspd + 1))
   {
     env.spkspd++;
   }
}

void prevspd ()
{
  spkstop ();
  if (speed (env.spkspd - 1))
   {
     env.spkspd--;
   }
}

void nextlang ()
{
  spkstop ();
  if (lang (env.spklang + 1))
   {
     env.spklang++;
     setspk ();
     getspkfilter (CONFDIR, env.profname, home, env.spklang);
   }
}

void nextvol ()
{
  spkstop ();
  if (volume (env.spkvolume + 1))
   {
     env.spkvolume++;
   }
}

void nextvoice ()
{
  spkstop ();
  if (voice (env.spkvoice + 1))
   {
     env.spkvoice++;
   }
}

void nextspec ()
{
  spkstop ();
  if (special (env.spkspec + 1))
   {
     env.spkspec++;
   }
}

void prevlang ()
{

  spkstop ();
  if (env.spklang > 0)
    if (lang (env.spklang - 1))
     {
       env.spklang--;
       setspk ();
       getspkfilter (CONFDIR, env.profname, home, env.spklang);
     }
}

void prevvol ()
{

  spkstop ();
  if (env.spkvolume > 0)
    if (volume (env.spkvolume - 1))
     {
       env.spkvolume--;
     }
}

void prevvoice ()
{

  spkstop ();
  if (env.spkvoice > 0)
    if (voice (env.spkvoice - 1))
     {
       env.spkvoice--;
     }
}

void prevspec ()
{

  spkstop ();
  if (env.spkspec > 0)
    if (special (env.spkspec - 1))
     {
       env.spkspec--;
     }
}

void spkcurln ()
{
  spkwrite (spkstr);
}

void spkscrtocsr ()
{
  char tmpstr[COLS * ROWS];

  memset (tmpstr, 0, COLS * ROWS);
  getscr ((winpos)
          {
          0, 0, COLS, spkcsrpos.y + 1}
          , (unsigned char *) tmpstr, SCR_TEXT);
/*  tmpstr[strlen (tmpstr) - (COLS - p->cox)] = 0; */
  spkwrite (tmpstr);
}

void spkscrfromcsr ()
{
  char tmpstr[(COLS + 1) * (ROWS + 1)];

  spkstop ();
  spkscrfromln ();
  return;
  memset (tmpstr, 0, (COLS + 1) * (ROWS + 1));
  getscr ((winpos)
          {
          0, 0, COLS, ROWS}
          , (unsigned char *) tmpstr, SCR_TEXT);
  spkwrite (tmpstr + (p->coy * COLS) + p->cox);
}

void spkscrfromln ()
{
  char tmpstr[(COLS + 1) * (ROWS + 1)];

  memset (tmpstr, 0, (COLS + 1) * (ROWS + 1));
  getscr ((winpos)
          {
          0, 0, COLS, ROWS - 1}
          , (unsigned char *) tmpstr, SCR_TEXT);
  spkwrite (tmpstr + (spkcsrpos.y * COLS));
}

void spkcharmod ()
{
  if (env.spkcharmod)
   {
     play (snd_toggleoff);
     env.spkcharmod = 0;
   }
  else
   {
     play (snd_toggleon);
     env.spkcharmod = 1;
   }
}

void spktocsr ()
{

  char str[MAXSPKSTR];

  memset (str, 0, MAXSPKSTR);
  memcpy (str, spkstr, spkcsrpos.x);
  spkwrite (str);
}

void spkfromcsr ()
{
  char str[MAXSPKSTR];

  memset (str, 0, MAXSPKSTR);
  memcpy (str, spkstr + spkcsrpos.x, (COLS - spkcsrpos.x));
  spkwrite (str);
}

void setspk ()
{
  static int nr = -1;

  sbl_log ("setspk\n");
  spkstop ();
  if (nr != env.spklang)
   {
     nr = env.spklang;
     sbl_log ("lang changed to %d\n", nr);
     lang (env.spklang);
   }

/* use voice and volume in this order
 * with some synths voice will overwrite the volume setting
 */
  voice (env.spkvoice);
  volume (env.spkvolume);
  spkmode = env.spkmod;
  speed (env.spkspd);
  frq (env.spkfrq);
  special (env.spkspec);
  sbl_log ("finished setspk\n");
}

int spkchar ()
{
  char tmp[10] = "";

  static int pos = 0, ch = 0;

  if (pos == (POSX) && spkstr[pos] == ch)
   {
     sbl_log ("speak true\n");
     return 1;
   }
  pos = (POSX);
  sbl_log ("speak: %d\n", pos);
  tmp[0] = spkstr[pos - 1];
  tmp[1] = '\n';
  tmp[2] = 0;
  spkwrite (tmp);

  return 0;
}

int brld_reconnect ()
{
  if (brld->open ("localhost", sblconf.brld_port) < 0)
   {
     return 1;
   }

  if (brld->reg (TEXT, sblconf.brld_auth_key))
   {
     return 2;
   }
  brld->getxy (&brl.x, &brl.y);
  if (brl.x < 00)
    brl.x = 80;
  if (brl.y <= 0)
    brl.y = 1;
  brld->getalias (sblconf.brlname);
  memset (brlstr, 0, sizeof (brlstr));

  getkeymap (&keymap, CONFDIR, sblconf, env.profname, home);
  return 0;
}

int elapsed_sec (struct timeval *start)
{
  struct timeval current_time;

  gettimeofday (&current_time, NULL);
  return (current_time.tv_sec - start->tv_sec);
}

int elapsed_ms (struct timeval *start)
{
  struct timeval current_time;

  int diff = 0;

  gettimeofday (&current_time, NULL);
  if (start->tv_sec > current_time.tv_sec)
    return -1;
  diff = (current_time.tv_sec - start->tv_sec) * 1000;
  if (diff == 0 && start->tv_usec > current_time.tv_usec)
    return -2;
  diff += (current_time.tv_usec - start->tv_usec) / 1000;
  return diff;
}

int kbd_sniffd_connect ()
{
  if (kbd_open (sblconf.kbd_port) >= 0)
    if (!kbd_auth (sblconf.kbd_key))
      if (!kbd_setkey (keymap.kbdsniffon1.kbd))
       {
         kbd_sniff_on = 1;
         return 0;
       }

  kbd_close (1);
  return 1;

}

void spkcsrlft ()
{
  char str[20] = "";

  memset (str, 0, sizeof (str));
  if (spkcsrpos.x > 0)
    spkcsrpos.x--;
  else
   {
     play (snd_bounce);
     return;
   }
  strncpy (str, spkstr + spkcsrpos.x, 1);
  spkpunctuation (1);
  if (str[0] >= 'A' && str[0] <= 'Z')
    frq (8);
  spkwrite (str);
  if (str[0] >= 'A' && str[0] <= 'Z')
    frq (env.spkfrq);
  spkpunctuation (0);
}

void spkcsrrgt ()
{
  char str[20] = "";

  memset (str, 0, sizeof (str));
  if ((unsigned) spkcsrpos.x < strlen (spkstr) - 1)
    spkcsrpos.x++;
  else
   {
     play (snd_bounce);
     return;
   }

  strncpy (str, spkstr + spkcsrpos.x, 1);
  spkpunctuation (1);
  if (str[0] >= 'A' && str[0] <= 'Z')
    frq (8);
  spkwrite (str);
  if (str[0] >= 'A' && str[0] <= 'Z')
    frq (env.spkfrq);
  spkpunctuation (0);
}

void spkcsrxy ()
{
  char str_xy[20] = "";

  sprintf (str_xy, "x %d y %d TTY %d\n", spkcsrpos.x + 1, spkcsrpos.y + 1,
           connr);
  spkwrite (str_xy);
}

void spkcsrchar ()
{
  char str[3] = "";

  str[0] = spkstr[spkcsrpos.x];
  str[1] = '\n';
  str[2] = 0;
  spkpunctuation (1);
  if (str[0] >= 'A' && str[0] <= 'Z')
    frq (8);
  spkwrite (str);
  if (str[0] >= 'A' && str[0] <= 'Z')
    frq (env.spkfrq);
  spkpunctuation (0);
}

void spkcsrword ()
{
  char word[255] = "";

  int start = 0;

  int i = 0;

  static struct timeval last_spk_word;

  int last_event = 0;

  last_event = elapsed_ms (&last_spk_word);
  if (last_event < 0 || last_event > SPELL_WORD_TO)
   {
     last_spk_word.tv_sec = 0;
     last_spk_word.tv_usec = 0;
   }
  if (!last_spk_word.tv_sec && !last_spk_word.tv_usec)
   {
     gettimeofday (&last_spk_word, NULL);
   }
  if (spkcsrpos.x == 0 || spkstr[spkcsrpos.x - 1] == ' ')
   {

     memset (word, 0, sizeof (word));
     for (i = spkcsrpos.x; i < (int) strlen (spkstr); i++)
      {
        if (spkstr[i] == ' ')
         {
           strncpy (word, spkstr + spkcsrpos.x, i - spkcsrpos.x);
           break;
         }
      }
   }
  else
   {

     memset (word, 0, sizeof (word));
     for (start = spkcsrpos.x; start && spkstr[start] != ' '; start--);
     if (spkstr[start] == ' ')
       start++;
     for (i = start; i < (int) strlen (spkstr) && spkstr[i] != ' '; i++);

     if (spkstr[spkcsrpos.x] == ' ')
       strcpy (word, " \n");
     else if (i != start && i < (int) strlen (spkstr))
       strncpy (word, spkstr + start, i - start);
   }

  if (word[0])
   {

     last_event = elapsed_ms (&last_spk_word);
     if (last_event > 0 && last_event < SPELL_WORD_TO)
      {

        spkmode = SPKSPELL;
        last_spk_word.tv_sec = 0;
        last_spk_word.tv_usec = 0;
      }

     spkwrite (word);
     spkmode = SPKNORM;
   }
}

void spkwordlft ()
{
  int i, end = 0, start = 0;

  char word[255];

  memset (word, 0, sizeof (word));
  /* check for start of word */
  for (i = spkcsrpos.x; i > 0; i--)
   {
     if (spkstr[i] != ' ' && spkstr[i - 1] == ' ' && spkcsrpos.x != i)
      {
        start = i;
        break;
      }

   }

/* word end */
  for (i = start; i < (int) strlen (spkstr); i++)
   {
     if (spkstr[i] == ' ')
       break;

   }                            /* for */
  end = i;

  if (end >= start)
   {
     spkcsrpos.x = start;
     strncpy (word, spkstr + start, (end - start) ? end - start : 1);

     spkwrite (word);
   }

}

void spkwordrgt ()
{
  int i, end = 0, start = -1, blank = 0;

  char word[255];

  memset (word, 0, sizeof (word));
/* check for next word */
  for (i = spkcsrpos.x; i < (int) strlen (spkstr); i++)
   {
     if (spkstr[i] == ' ')
       blank++;
     else if (blank)
      {
        start = i;
        break;
      }

   }

  if (blank && start >= 0)
   {

/* word end */
     for (i = start; i < (int) strlen (spkstr); i++)
      {
        if (spkstr[i] == ' ')
          break;

      }                         /* for */
     end = i;
   }                            /* if */
  if (end >= start && start >= 0)
   {
     spkcsrpos.x = start;
     strncpy (word, spkstr + start, (end - start) ? end - start : 1);
     spkwrite (word);
   }

}
