/*
 *  suse-blinux - Braille-display support for linux
 *  Author: Marco Skambraks <marco@suse.de>
 *  Modifications for new HT protocol by Klaus Knopper <knoppix@knopper.net>
 *  SuSE GmbH Nuernberg
 *
 *
 * suse-blinux based on brltty
 * special thanks to the Brltty-Team
 * Nicolas Pitre <nico@cam.org>
 * Stphane Doyon <s.doyon@videotron.ca>
 * Nikhil Nair <nn201@cus.cam.ac.uk>
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
*/
#include <stdio.h>
#include <termios.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>          /* ioctl */
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <dlfcn.h>
#include "variolow.h"
#include "misc.h"

        /* The filedescriptor of the open port, sorry to say, wee need a
           global one here */

/* Imported from brld.c */
extern unsigned int brl_dbg;

unsigned char st_disp[4];

static int devfd = -1;

static int devid = 0;           /* Device ID needed for new HT protocol -KK */
static int use_frames = 0;      /* != 0 indicates new protocol -KK */
static int use_hid = 1;         /* != 0 indicates to use HID frames -KK */

/* Old protocol */
#define ID_USB_HID_ADAPTER 0x03
#define ID_BRAILLEWAVE 0x05
#define ID_MODULAREVOLUTION64 0x36
#define ID_MODULAREVOLUTION88 0x38
#define	ID_EASYBRAILLE 0x44
#define ID_BRAILLINO 0x72
#define ID_BRAILLESTAR40 0x74
#define ID_BRAILELSTAR80 0x78
#define	ID_MODULAR20 0x80
#define ID_MODULAR80 0x88
#define ID_MODULAR40 0x89
#define	ID_BOOKWORM 0x90

/* New protocol */
#define ID_ACTIVEBRAILLE 0x54
#define ID_ACTIVESTAR    0x64
#define ID_BASICBRAILLE  0x84

static void *libbt = NULL;

/* This is the new protocols frame */
typedef struct _ht_frame
{
  unsigned char frame_start;    /* always 0x79 */
  unsigned char devid;          /* Device ID of target */
  unsigned char size;           /* size of command + command-data */
  unsigned char cmd;            /* command */
  unsigned char data[101];      /* variable size between 0 and 100 bytes,
                                   plus a frame_end indicator (0x16) */
} ht_frame;

/* cmd + data without frame_end byte */
static const unsigned int ht_frame_max_size = 101;

/* see spec */
static const unsigned int ht_hid_frame_size = 61;

static int write_hid (int fd, unsigned char *data, int size)
{
  int written = 0;
  unsigned char hid_frame[ht_hid_frame_size];

  while (size > 0)
   {
     int rc, current_size;

     memset (hid_frame, 0, ht_hid_frame_size);
     current_size =
       (unsigned int) size >
       (ht_hid_frame_size - 2) ? (int) (ht_hid_frame_size - 2) : size;
     /* 0x01 = Braille to PC, 0x02 = PC to Braille */
     hid_frame[0] = 0x02;
     /* Size of real data contained in frame */
     hid_frame[1] = current_size;
     memcpy (&hid_frame[2], &data[written], current_size);
     rc = write (fd, hid_frame, ht_hid_frame_size);
     if (rc < 0)
      {
        if (brl_dbg >= 1)
          fprintf (stderr, "Error %d writing HID frame with %d user bytes.\n",
                   rc, current_size);
        return rc;
      }
     else if (brl_dbg >= 3)
       fprintf (stderr, "Wrote HID frame of %d bytes (payload %d bytes).\n",
                rc, current_size);
     written += current_size;
     size -= current_size;
   }
  return 0;
}

/* data must be able to hold at least 59 bytes */
static int read_hid (int fd, unsigned char *data, int *size)
{
  unsigned char hid_frame[ht_hid_frame_size];
  int rc = 0, current_size;

  /* We simplify reading here by assuming that braille device */
  /* will not return data > 59 bytes that has to be split into */
  /* multiple frames.  */
  rc = read (fd, hid_frame, ht_hid_frame_size);
  if (rc < (int) ht_hid_frame_size)
   {
     if (brl_dbg >= 1)
       fprintf (stderr, "Error %d reading HID frame (received %d bytes).\n",
                errno, rc);
     return -1;
   }
  current_size =
    (hid_frame[1] >
     (ht_hid_frame_size - 2)) ? (ht_hid_frame_size - 2) : hid_frame[1];
  memcpy (data, &hid_frame[2], current_size);
  *size = current_size;
  if (brl_dbg >= 3)
    fprintf (stderr, "Read HID frame with %d user bytes.\n", current_size);
  return 0;
}

static ht_frame *alloc_ht_frame ()
{
  ht_frame *htf = malloc (sizeof (ht_frame));

  if (!htf)
    return NULL;
  memset (htf, 0, sizeof (ht_frame));
  htf->frame_start = 0x79;      /* start of frame */
  htf->devid = devid;           /* From global variable */
  htf->size = 1;                /* One (dummy) command byte */
  htf->data[0] = 0x16;          /* Frame end */
  return htf;
}

static void add_ht_frame_data (ht_frame * htf, unsigned char *data,
                               unsigned int size)
{
  if (htf->size + size >= ht_frame_max_size)
    size = ht_frame_max_size - htf->size;
  if (size > 0)
   {
     memcpy (&htf->data[htf->size - 1], data, size);
     htf->size += size;
   }
  htf->data[htf->size - 1] = 0x16;
}

static void set_ht_frame_cmd (ht_frame * htf, unsigned char *cmd)
{
  htf->cmd = *cmd;
}

static int write_ht_frame (int fd, ht_frame * frame)
{
  int size = 4 + frame->size;

  if (use_hid)                  /* Need to use HID frames */
    return write_hid (fd, (unsigned char *) frame, size);
  else
    return write (fd, frame, size) != size;
}

static ht_frame *read_ht_frame (int fd)
{
  unsigned char data = 0;
  ht_frame *htf = alloc_ht_frame ();

  if (htf == NULL)
    return NULL;
  memset (htf, 0, sizeof (ht_frame));

  /* HID device variant */
  if (use_hid)
   {
     int hid_data_size = sizeof (ht_frame);

     if (!read_hid (fd, (unsigned char *) htf, &hid_data_size))
       return htf;
     free (htf);
     return NULL;
   }

  /* Non-HID variant */
  /* Search for Frame start in stream */
  while (data != 0x79)
   {                            /* Search frame start */
     if (read (fd, &data, 1) != 1)
       return NULL;
   }

  if ((read (fd, &data, 1) != 1) || (data != devid) ||
      (read (fd, &data, 1) != 1) || (data == 0) || (data > ht_frame_max_size))
   {
     if (brl_dbg >= 1)
       fprintf (stderr, "HT protocol error.\n");
     free (htf);
     return NULL;
   }

  /* Now we have reached the cmd field in the input */
  htf->frame_start = 0x79;
  htf->devid = devid;
  htf->size = data;
  if ((read (fd, &htf->cmd, 1) == 1) &&
      ((htf->size <= 1)
       || (read (fd, &htf->data, htf->size - 1) == (htf->size - 1)))
      && (read (fd, &htf->data[htf->size], 1) == 1)
      && htf->data[htf->size] == 0x16)
    return htf;

  /* If we reach this point, something went wrong. */
  free (htf);
  return NULL;
}

int varioinit (char *dev)
{

  struct termios tiodata;

  /* Idiot check one */
  if (!dev)
   {
     return -1;
   }

  /* Check for "hid" in filename, like in "/dev/hidraw0" */
  use_hid = strstr (dev, "hid") != NULL;

  if (dev[0] != '/')            /* EXPERIMENTAL direct bluetooth support
                                   (tested only with brlstar40) */
   {
     if (brl_dbg >= 1)
       fprintf (stderr, "connecting to bluetooth device using addr: %s\n",
                dev);
     devfd = btconnect (dev);
   }
  else
    devfd = open (dev, O_RDWR | O_NOCTTY | O_NONBLOCK);

  if (devfd == -1)
   {
     if (brl_dbg >= 1)
       fprintf (stderr, "Port open failed on %s: %s\n", dev,
                strerror (errno));
     return -1;
   }

  usleep (500000);

  if (!libbt)
   {
     /* If we got it open, get the old attributes of the port */
     if (tcgetattr (devfd, &tiodata))
      {
        if (brl_dbg >= 4)
          fprintf (stderr, "NOTICE: tcgetattr() on %s returned %s\n", dev,
                   strerror (errno));
      }
     else
      {
        tiodata.c_cflag = (CLOCAL | PARODD | PARENB | CREAD | CS8);
        tiodata.c_iflag = IGNPAR;
        tiodata.c_oflag = 0;
        tiodata.c_lflag = 0;
        tiodata.c_cc[VMIN] = 0;
        tiodata.c_cc[VTIME] = 0;
        /* Force down DTR, flush any pending data and then set the port to
           what we want it to be */
        cfsetispeed (&tiodata, B0);
        cfsetospeed (&tiodata, B0);
        tcsetattr (devfd, TCSANOW, &tiodata);
        tcflush (devfd, TCIOFLUSH);
        cfsetispeed (&tiodata, B19200);
        cfsetospeed (&tiodata, B19200);
        if (tcsetattr (devfd, TCSANOW, &tiodata))
         {
           if (brl_dbg >= 4)
             fprintf (stderr, "NOTICE: tcsetattr() on %s returned: %s\n", dev,
                      strerror (errno));
         }
      }

     /* Pause 500ms to let them take effect */
     usleep (500000);
     /* resetbrl */
     /* write(devfd,0xff,1); */
   }
  if (!varioreset ())
   {
     variogetid ();
     variosetpcmode ();
     if (devid == ID_ACTIVEBRAILLE)
       variosetatc (1);
   }
  return devfd;
}

int varioclose ()
{
  /* Flush all pending output and then close the port */
  if (devfd >= 0)
   {
     if (libbt == NULL)
       tcflush (devfd, TCIOFLUSH);
     close (devfd);
     devfd = -1;

     if (libbt)
      {
        close (devfd);
        devfd = -1;
        dlclose (libbt);
        libbt = NULL;
      }
     return 0;
   }
  return -1;
}

int varioreset ()
{
  unsigned char c = VARIO_RESET;

  if (devfd < 0)
   {
     return -1;
   }
  /* This is the only command which is not framed in the new protocol. -KK */

  return use_hid ? write_hid (devfd, &c, sizeof (c)) : write (devfd, &c,
                                                              sizeof (c)) !=
    sizeof (c);
}

int variogetid ()
{
  int i;
  unsigned char id1 = 0, id2 = 0;

  for (i = 0; i < 20; i++)
   {
     if (variocheckwaiting (500000))
      {
        continue;
      }
     break;
   }
  if (use_hid)
   {
     unsigned char data[80];
     int size = 0;

     while (id1 != 0xfe)
      {
        if (read_hid (devfd, data, &size))
          return -1;
        id1 = data[0];
        id2 = data[1];
      }
   }
  else
   {
     while (id1 != 0xfe)
      {
        if (read (devfd, &id1, 1) != 1)
          return -1;
      }
     if (read (devfd, &id2, 1) != 1)
       return -1;
   }
  devid = (int) id2;
  switch (devid)
   {
     /* old protocol */
   case ID_USB_HID_ADAPTER:
   case ID_BRAILLEWAVE:
   case ID_MODULAREVOLUTION64:
   case ID_MODULAREVOLUTION88:
   case ID_EASYBRAILLE:
   case ID_BRAILLINO:
   case ID_BRAILLESTAR40:
   case ID_BRAILELSTAR80:
   case ID_MODULAR20:
   case ID_MODULAR80:
   case ID_MODULAR40:
   case ID_BOOKWORM:
     use_frames = 0;
     break;
     /* new protocol */
   case ID_ACTIVEBRAILLE:
   case ID_ACTIVESTAR:
   case ID_BASICBRAILLE:
   default:
     use_frames = 1;
     break;
   }
  if (brl_dbg >= 1)
    fprintf (stderr, "braille device responds with %d, id=%d, %s.\n", id1,
             id2,
             use_frames ? "using new HT protocol with frames" :
             "using old HT protocol");
  return 0;
}

int variosetatc (int onoff)
{
  unsigned char atc_cmd[] = { 0x50, 0x00 };
  unsigned char atc_onoff = (onoff != 0);
  ht_frame *frame = alloc_ht_frame ();
  int rc = 0, i;

  if (!frame)
    return -1;
  set_ht_frame_cmd (frame, atc_cmd);
  add_ht_frame_data (frame, &atc_onoff, 1);
  /* Write framed data */
  rc = write_ht_frame (devfd, frame);
  free (frame);
  frame = NULL;
  if (rc != 0)
    return -1;
  for (i = 0; i < 20; i++)
   {
     if (variocheckwaiting (500000))
      {
        continue;
      }
     /* Read answer */
     frame = read_ht_frame (devfd);
     if (frame != NULL)
      {
        if (brl_dbg >= 2)
          fprintf (stderr, "Answer received to SET_ATC_MODE: 0x%x 0x%x\n",
                   (int) frame->cmd, (int) frame->data[0]);
        free (frame);
      }
     break;
   }
  return 0;
}

int variosetpcmode ()
{
  if (use_frames)
   {
     ht_frame *frame = alloc_ht_frame ();
     int rc = 0, i;

     if (!frame)
       return -1;
     set_ht_frame_cmd (frame, (unsigned char *) VARIO_SET_PC_MODE);
     /* Write framed data */
     rc = write_ht_frame (devfd, frame);
     free (frame);
     frame = NULL;
     if (rc != 0)
       return -1;
     for (i = 0; i < 20; i++)
      {
        if (variocheckwaiting (500000))
         {
           continue;
         }
        /* Read answer */
        frame = read_ht_frame (devfd);
        if (frame != NULL)
         {
           if (brl_dbg >= 2)
             fprintf (stderr, "Answer received to SET_PC_MODE: 0x%x 0x%x\n",
                      (int) frame->cmd, (int) frame->data[0]);
           free (frame);
         }
        break;
      }
     /* Should we do something here if braille device did not send an
        acknowledgement? */
   }
  else
   {
     int i, rc = write (devfd, VARIO_SET_PC_MODE, VARIO_SET_PC_MODE_LEN)
       != VARIO_SET_PC_MODE_LEN;
     unsigned char answer[2];

     if (rc)
       return -1;
     for (i = 0; i < 20; i++)
      {
        if (variocheckwaiting (500000))
         {
           continue;
         }
        if (read (devfd, answer, 2) == 2)
          if (brl_dbg >= 1)
            fprintf (stderr, "Answer received to SET_PC_MODE: 0x%x 0x%x\n",
                     answer[0], answer[1]);
        break;
      }
     /* Should we do something here if braille device did not send an
        acknowledgement? */
   }
  return 0;
}

int variodisplay (char *buff)
{

  /* fprintf(stderr, "variodisplay() called with use_hid=%d, buff=0x%x,
     devfd=%d.\n", use_hid, buff, devfd); */
  if (!buff || (devfd < 0))
   {
     return -1;
   }
  if (!use_frames)
   {                            /* old protocol -KK */
     /* Write header and then data ... */
     if (write (devfd, VARIO_DISPLAY_DATA, VARIO_DISPLAY_DATA_LEN) < 0)
       return -1;
     else if (write (devfd, st_disp, BRL_ST_CELLS) < 0)
       return -1;
     else if (write (devfd, buff, BRLCOLS) < 0)
       return -1;
   }
  else
   {                            /* new protocol */
     int rc;
     ht_frame *frame = alloc_ht_frame ();

     if (!frame)
       return -1;
     set_ht_frame_cmd (frame, (unsigned char *) VARIO_DISPLAY_DATA);
     add_ht_frame_data (frame, st_disp, BRL_ST_CELLS);
     add_ht_frame_data (frame, (unsigned char *) buff, BRLCOLS);
     /* Write framed data */
     rc = write_ht_frame (devfd, frame);
     free (frame);
     if (rc != 0)
       return -1;
   }

  return 0;
}

/* Returns 0 if data is available for being read */
int variocheckwaiting (int usec)
{
  fd_set checkset;

  struct timeval tval;

  tval.tv_sec = 0;
  tval.tv_usec = usec;

  FD_ZERO (&checkset);
  FD_SET (devfd, &checkset);
  return !select (devfd + 1, &checkset, NULL, NULL, &tval);
}

int varioget ()
{

  /* Just read and return one byte */

  if (use_frames)
   {                            /* new protocol -KK */
     int data;
     ht_frame *htf = read_ht_frame (devfd);

     if (htf == NULL)
       return -1;
     if (htf->cmd == 0x04)      /* Got a key code */
       data = htf->data[0];
     else if (htf->cmd == 0x55 && htf->data[0] != 0xff)
       /* Got ATC finger reading position */
       data = htf->data[0] | 0x100;     /* Set ATC indication flag */
     else
       data = -1;
     free (htf);
     return data;
   }
  else
   {                            /* old protocol */
     unsigned char data = 0;

     if (read (devfd, &data, 1) == 1)
       return data;
     else
       return -1;
   }
}

int variotranslate (unsigned char *frombuff, char *tobuff, int count)
{
  if (!frombuff | !tobuff)
   {
     return -1;
   }
  /* Just apply the nify macro for all chars */
  for (; count > 0; count--)
   {
     tobuff[count - 1] = CHAR_TO_VARIO_CHAR (frombuff[count - 1]);
   }

  return 0;
}

int btconnect (char *dev)
{
  void (*str2ba) (const char *str, bdaddr_t * ba);

  struct sockaddr_rc addr;

  int status, fd;

  libbt = dlopen ("libbluetooth.so.2", RTLD_NOW | RTLD_GLOBAL);
  if (libbt == NULL)
    libbt = dlopen ("libbluetooth.so.3", RTLD_NOW | RTLD_GLOBAL);
  if (libbt == NULL)
    return -1;
  str2ba = dlsym (libbt, "str2ba");

  memset (&addr, 0, sizeof (addr));

  fd = socket (AF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);

  /* set the connection parameters */
  addr.rc_family = AF_BLUETOOTH;
  addr.rc_channel = (uint8_t) 1;
  str2ba (dev, &addr.rc_bdaddr);

  /* connect to device */
  status = connect (fd, (struct sockaddr *) &addr, sizeof (struct sockaddr));
  if (!status)
   {
     fcntl (fd, F_SETFL, O_NONBLOCK);
     if (brl_dbg >= 1)
       fprintf (stderr, "btconnect successful.\n");
     return fd;
   }
  dlclose (libbt);
  if (brl_dbg >= 1)
    fprintf (stderr, "btconnect  to %s failed.\n", dev);
  libbt = NULL;

  return -1;
}
