searchData={"items":[{"type":"module","title":"Logger","doc":"A logger for Elixir applications.\n\nThis application is mostly a wrapper around Erlang's\n[`:logger`](`:logger`) functionality, to provide message\ntranslation and formatting to Elixir terms.\n\nOverall, you will find that `Logger`:\n\n  * Provides all 7 syslog levels\n    (although debug, info, warning, and error are the most commonly used).\n\n  * Supports both message-based and structural logging.\n\n  * Formats and truncates messages on the client\n    to avoid clogging `Logger` backends.\n\n  * Alternates between sync and async modes to remain\n    performant when required but also apply backpressure\n    when under stress.\n\n  * Support for custom filters and handlers as provided by\n    Erlang's `:logger`.\n\n  * Allows overriding the logging level for a specific module,\n    application or process.\n\nLogging is useful for tracking when an event of interest happens in your\nsystem. For example, it may be helpful to log whenever a user is deleted.\n\n    def delete_user(user) do\n      Logger.info(\"Deleting user from the system: #{inspect(user)}\")\n      # ...\n    end\n\nThe `Logger.info/2` macro emits the provided message at the `:info`\nlevel. Note the arguments given to `info/2` will only be evaluated\nif a message is logged. For instance, if the Logger level is\nset to `:warning`, `:info` messages are never logged and therefore\nthe arguments given above won't even be executed.\n\nThere are additional macros for other levels.\n\nLogger also allows log commands to be removed altogether via the\n`:compile_time_purge_matching` option (see below).\n\nFor dynamically logging messages, see `bare_log/3`. But note that\n`bare_log/3` always evaluates its arguments (unless the argument\nis an anonymous function).","ref":"Logger.html"},{"type":"module","title":"Levels - Logger","doc":"The supported levels, ordered by importance, are:\n\n  * `:emergency` - when system is unusable, panics\n  * `:alert` - for alerts, actions that must be taken immediately,\n    ex. corrupted database\n  * `:critical` - for critical conditions\n  * `:error` - for errors\n  * `:warning` - for warnings\n  * `:notice` - for normal, but significant, messages\n  * `:info` - for information of any kind\n  * `:debug` - for debug-related messages\n\nFor example, `:info` takes precedence over `:debug`. If your log\nlevel is set to `:info`, then all `:info`, `:notice` and above will\nbe passed to backends. If your log level is set to `:alert`, only\n`:alert` and `:emergency` will be printed.","ref":"Logger.html#module-levels"},{"type":"module","title":"Message - Logger","doc":"Logger can be used for logging both unstructured and structured data.\n\nUnstructured data is a string or a list of strings:\n\n    Logger.info(\"hello world!\")\n    Logger.info([\"hello \", \"world!\"])\n\nStructured data, also known as reports, are keyword lists and maps:\n\n    Logger.info([new_user: user.id, account_type: :admin])\n    Logger.info(%{new_user: user.id, account_type: :admin})\n\nLog functions also accept a zero-arity anonymous function as a message:\n\n    Logger.info(fn -> \"hello world!\" end)\n\nThe anonymous function can return a message or a tuple containing\nthe message and additional metadata (to be described in the next\nsection).\n\nIn all cases, the arguments given to the `Logger` macros are only\nevaluated if required by the current log level. The exception is\nthe `bare_log/3` function, which is the raw mechanism for logging.","ref":"Logger.html#module-message"},{"type":"module","title":"Metadata - Logger","doc":"Whenever a message is logged, additional information can be given\nvia metadata. Each log operation, such as `Logger.info/2`, allows\nmetadata to be given as an argument.\n\nFurthermore, metadata can be set per process with `Logger.metadata/1`.\n\nSome metadata, however, is always added automatically by Logger\nwhenever possible. Those are:\n\n  * `:application` - the current application\n\n  * `:mfa` - the current module, function and arity\n\n  * `:file` - the current file\n\n  * `:line` - the current line\n\n  * `:pid` - the current process identifier\n\n  * `:initial_call` - the initial call that started the process\n\n  * `:registered_name` - the process registered name as an atom\n\n  * `:domain` - a list of domains for the logged message. For example,\n    all Elixir reports default to `[:elixir]`. Erlang reports may start\n    with `[:otp]` or `[:sasl]`\n\n  * `:crash_reason` - a two-element tuple with the throw/error/exit reason\n    as first argument and the stacktrace as second. A throw will always be\n    `{:nocatch, term}`. An error is always an `Exception` struct. All other\n    entries are exits. The console backend ignores this metadata by default\n    but it can be useful to other backends, such as the ones that report\n    errors to third-party services\n\nThere are two special metadata keys, `:module` and `:function`, which\nextract the relevant bits from `:mfa`.\n\nNote that all metadata is optional and may not always be available.\nThe `:mfa`, `:file`, `:line`, and similar metadata are automatically\nincluded when using `Logger` macros. `Logger.bare_log/3` does not include\nany metadata beyond the `:pid` by default. Other metadata, such as\n`:crash_reason`, `:initial_call`, and `:registered_name` are available\nonly inside behaviours such as GenServer, Supervisor, and others.\n\nFor example, you might wish to include a custom `:error_code` metadata in\nyour logs:\n\n    Logger.error(\"We have a problem\", [error_code: :pc_load_letter])\n\nBy default, no metadata is logged. We will learn how to enable that\nover the next sections.","ref":"Logger.html#module-metadata"},{"type":"module","title":"Configuration - Logger","doc":"`Logger` supports a wide range of configurations.\n\nThis configuration is split in three categories:\n\n  * Boot configuration - this configuration is read when logger\n    starts and configures how Elixir hooks into Erlang's own logger\n\n  * Compile configuration - this must be set before your code\n    is compiled\n\n  * Runtime configuration - can be set before the `:logger`\n    application is started, but may be changed during runtime\n\n#","ref":"Logger.html#module-configuration"},{"type":"module","title":"Boot configuration - Logger","doc":"When `Logger` starts, it configures the `:default` log handler from\nErlang to translate and format Elixir terms. As a developer, you\nare able to customize the default handler, the default formatter,\nand many other options.\n\nThe following configuration must be set via config files (such as\n`config/config.exs`), under the `:logger` key, before your application\nis started:\n\n  * `:default_formatter` - a keyword list which configures the\n    default formatter used by the default handler. See `Logger.Formatter`\n    for the full list of configuration.\n\n  * `:default_handler` - this option configures the default handler\n    used for logging. The default handler is a [`:logger_std_h`](`:logger_std_h`)\n    instance which also supports file logging and log rotation.\n    You can set it to `false` to disable the default logging altogether.\n    See the examples below for more information.\n\n  * `:handle_otp_reports` - if Erlang/OTP message should be logged.\n    Defaults to `true`.\n\n  * `:handle_sasl_reports` - if supervisor, crash, and progress reports\n    should be logged. Defaults to `false`. This option only has an effect\n    if `:handle_otp_reports` is true.\n\n   * `:metadata` - global primary metadata to be included in all log messages.\n    Defaults to `[]`. This can be overridden at the process level with `metadata/1`\n    or each on log call as desired.\n\nFor example, to configure `Logger` to redirect all Erlang messages using a\n`config/config.exs` file:\n\n    config :logger,\n      handle_otp_reports: true,\n      handle_sasl_reports: true\n\nTo configure the default formatter, for example, to use a different format\nand include some metadata:\n\n    config :logger, :default_formatter,\n      format: \"[$level] $message $metadata\\n\",\n      metadata: [:error_code, :file]\n\nOr to configure default handler, for instance, to log into a file with\nbuilt-in support for log rotation and compression:\n\n    config :logger, :default_handler,\n      config: [\n        file: ~c\"system.log\",\n        filesync_repeat_interval: 5000,\n        file_check: 5000,\n        max_no_bytes: 10_000_000,\n        max_no_files: 5,\n        compress_on_rotate: true\n      ]\n\nSee [`:logger_std_h`](`:logger_std_h`) for all relevant configuration,\nincluding overload protection. Or set `:default_handler` to false to\ndisable the default logging altogether:\n\n    config :logger, :default_handler, false\n\nHow to add new handlers is covered in later sections.\n\n> ##","ref":"Logger.html#module-boot-configuration"},{"type":"module","title":"Keywords or maps {: .tip} - Logger","doc":">\n> While Erlang's logger expects `:config` to be a map, Elixir's Logger\n> allows the default handler configuration to be set with keyword lists.\n> For example, this allows your `config/*.exs` files, such as `config/dev.exs`,\n> to override individual keys defined in `config/config.exs`.\n>\n> When reading the handler configuration using Erlang's APIs,\n> the configuration will always be read (and written) as a map.\n\n#","ref":"Logger.html#module-keywords-or-maps-tip"},{"type":"module","title":"Compile configuration - Logger","doc":"The following configuration must be set via config files (such as\n`config/config.exs`) under the `:logger` application before your code\nis compiled:\n\n  * `:always_evaluate_messages` - if messages should be *evaluated* even if\n    the log level is lower than the minimum configured level. Defaults to `false`.\n    This is useful for cases where the log level in your *test environment*\n    is high (such as `:error`), which is common in order to avoid logs mixed\n    with the test output. In such, cases, you might discover log messages\n    that contain runtime errors only when your code is deployed to production,\n    where the log level is lower (such as `:info`). These runtime errors could\n    be caused by, for example, interpolating something that doesn't implement\n    the `String.Chars` protocol in the log message, such as `\"PID: #{self()}\"`\n    (since PIDs cannot be converted to strings with `String.Chars`).\n\n  * `:compile_time_application` - sets the `:application` metadata value\n    to the configured value at compilation time. This configuration is\n    automatically set by Mix and made available as metadata when logging.\n\n  * `:compile_time_purge_matching` - purges *at compilation time* all calls\n    that match the given conditions. This means that `Logger` calls with\n    level lower than this option will be completely removed at compile time,\n    accruing no overhead at runtime. This configuration expects a list of\n    keyword lists. Each keyword list contains a metadata key and the matching\n    value that should be purged. Some special keys are supported:\n\n      * `:level_lower_than` - purges all messages with a lower logger level\n      * `:module` - purges all messages with the matching module\n      * `:function` - purges all messages with the \"function/arity\"\n\n    Remember that if you want to purge log calls from a dependency, the\n    dependency must be recompiled.\n\nFor example, to configure the `:backends` and purge all calls that happen\nat compile time with level lower than `:info` in a `config/config.exs` file:\n\n    config :logger,\n      compile_time_purge_matching: [\n        [level_lower_than: :info]\n      ]\n\nIf you want to purge all log calls from an application named `:foo` and only\nkeep errors from `Bar.foo/3`, you can set up two different matches:\n\n    config :logger,\n      compile_time_purge_matching: [\n        [application: :foo],\n        [module: Bar, function: \"foo/3\", level_lower_than: :error]\n      ]\n\n#","ref":"Logger.html#module-compile-configuration"},{"type":"module","title":"Runtime Configuration - Logger","doc":"All configuration below can be set via config files (such as\n`config/config.exs`) but also changed dynamically during runtime via\n`Logger.configure/1`.\n\n  * `:level` - the logging level. Attempting to log any message\n    with severity less than the configured level will simply\n    cause the message to be ignored. Keep in mind that each backend\n    may have its specific level, too. In addition to levels mentioned\n    above it also supports 2 \"meta-levels\":\n\n      - `:all` - all messages will be logged, conceptually identical to\n        `:debug`\n      - `:none` - no messages will be logged at all\n\n  * `:translator_inspect_opts` - when translating OTP reports and\n    errors, the last message and state must be inspected in the\n    error reports. This configuration allow developers to change\n    how much and how the data should be inspected.\n\nFor example, to configure the `:level` options in a `config/config.exs`\nfile:\n\n    config :logger, level: :warning\n\nFurthermore, `Logger` allows messages sent by Erlang to be translated\ninto an Elixir format via translators. Translators can be added at any\ntime with the `add_translator/1` and `remove_translator/1` APIs. Check\n`Logger.Translator` for more information.","ref":"Logger.html#module-runtime-configuration"},{"type":"module","title":"Erlang/OTP handlers - Logger","doc":"Handlers represent the ability to integrate into the logging system to\nhandle each logged message/event. Elixir automatically configures the\ndefault handler, but you can use Erlang's [`:logger`](`:logger`) module\nto add other handlers too.\n\nErlang/OTP handlers must be listed under your own application.\nFor example, to setup an additional handler, so you write to\nconsole and file:\n\n    config :my_app, :logger, [\n      {:handler, :file_log, :logger_std_h, %{\n         config: %{\n           file: ~c\"system.log\",\n           filesync_repeat_interval: 5000,\n           file_check: 5000,\n           max_no_bytes: 10_000_000,\n           max_no_files: 5,\n           compress_on_rotate: true\n         },\n         formatter: Logger.Formatter.new()\n       }}\n    ]\n\nEach handler has the shape `{:handler, name, handler_module, config_map}`.\nOnce defined, a handler can be explicitly attached in your\n`c:Application.start/2` callback with `add_handlers/1`:\n\n    Logger.add_handlers(:my_app)\n\nYou can also develop your own handlers. Handlers run in the same\nprocess as the process logging the message/event. This gives developers\nflexibility but they should avoid performing any long running action in\nsuch handlers, as it may slow down the action being executed considerably.\nAt the moment, there is no built-in overload protection for Erlang handlers,\nso it is your responsibility to implement it.\n\nAlternatively, you can use the\n[`:logger_backends`](https://github.com/elixir-lang/logger_backends) project.\nIt sets up a log handler with overload protection and allows incoming events\nto be dispatched to multiple backends.","ref":"Logger.html#module-erlang-otp-handlers"},{"type":"module","title":"Backends and backwards compatibility - Logger","doc":"Prior to Elixir v1.15, custom logging could be achieved with Logger\nbackends. The main API for writing Logger backends have been moved to\nthe [`:logger_backends`](https://github.com/elixir-lang/logger_backends)\nproject. However, the backends are still part of Elixir for backwards\ncompatibility.\n\nImportant remarks:\n\n  * If the `:backends` key is set and it doesn't have the `:console` entry,\n    we assume that you want to disable the built-in logging. You can force\n    logging by setting `config :logger, :default_handler, []`\n\n  * The `:console` backend configuration is automatically maped to the default\n    handler and default formatter. Previously, you would set:\n\n        config :logger, :console,\n          level: :error,\n          format: \"$time $message $metadata\"\n\n    This is now equivalent to:\n\n        config :logger, :default_handler,\n          level: :error\n\n        config :logger, :default_formatter,\n          format: \"$time $message $metadata\"\n\n    All previous console configuration, except for `:level`, now go under\n    `:default_formatter`.\n\n  * If you want to use the previous `:console` implementation based on Logger\n    Backends, you can still set `backends: [Logger.Backends.Console]` and place\n    the configuration under `config :logger, Logger.Backends.Console`. Although\n    consider using the [`:logger_backends`](https://github.com/elixir-lang/logger_backends)\n    project in such case, as `Logger.Backends.Console` itself will be deprecated\n    in future releases","ref":"Logger.html#module-backends-and-backwards-compatibility"},{"type":"function","title":"Logger.add_backend/2","doc":"Adds a new backend.","ref":"Logger.html#add_backend/2"},{"type":"function","title":"Logger.add_handlers/1","doc":"Adds the handlers configured in the `:logger` application parameter\nof the given `app`.\n\nThis is used to register new handlers into the logging system.\nSee [the module documentation](#module-erlang-otp-handlers) for\nmore information.","ref":"Logger.html#add_handlers/1"},{"type":"function","title":"Logger.add_translator/1","doc":"Adds a new translator.","ref":"Logger.html#add_translator/1"},{"type":"macro","title":"Logger.alert/2","doc":"Logs a alert message.\n\nReturns `:ok`.\n\nThis is reported as \"error\" in Elixir's\nlogger backends for backwards compatibility reasons.","ref":"Logger.html#alert/2"},{"type":"macro","title":"Examples - Logger.alert/2","doc":"Logging a message (string or iodata):\n\n    Logger.alert(\"this is a alert message\")\n\nReport message (maps or keywords):\n\n    # as keyword list\n    Logger.alert([something: :reported, this: :alert])\n\n    # as map\n    Logger.alert(%{this: :alert, something: :reported})","ref":"Logger.html#alert/2-examples"},{"type":"function","title":"Logger.bare_log/3","doc":"Logs a message dynamically.\n\nOpposite to `log/3`, `debug/2`, `info/2`, and friends, the arguments\ngiven to `bare_log/3` are always evaluated. However, you can pass\nanonymous functions to `bare_log/3` and they will only be evaluated\nif there is something to be logged.","ref":"Logger.html#bare_log/3"},{"type":"function","title":"Logger.compare_levels/2","doc":"Compares log levels.\n\nReceives two log levels and compares the `left` level\nagainst the `right` level and returns:\n\n  * `:lt` if `left` is less than `right`\n  * `:eq` if `left` and `right` are equal\n  * `:gt` if `left` is greater than `right`","ref":"Logger.html#compare_levels/2"},{"type":"function","title":"Examples - Logger.compare_levels/2","doc":"iex> Logger.compare_levels(:debug, :warning)\n    :lt\n    iex> Logger.compare_levels(:error, :info)\n    :gt","ref":"Logger.html#compare_levels/2-examples"},{"type":"function","title":"Logger.configure/1","doc":"Configures the logger.\n\nSee the \"Runtime Configuration\" section in the `Logger` module\ndocumentation for the available options. The changes done here\nare automatically persisted to the `:logger` application\nenvironment.","ref":"Logger.html#configure/1"},{"type":"function","title":"Logger.configure_backend/2","doc":"Configures the given backend.","ref":"Logger.html#configure_backend/2"},{"type":"macro","title":"Logger.critical/2","doc":"Logs a critical message.\n\nReturns `:ok`.\n\nThis is reported as \"error\" in Elixir's\nlogger backends for backwards compatibility reasons.","ref":"Logger.html#critical/2"},{"type":"macro","title":"Examples - Logger.critical/2","doc":"Logging a message (string or iodata):\n\n    Logger.critical(\"this is a critical message\")\n\nReport message (maps or keywords):\n\n    # as keyword list\n    Logger.critical([something: :reported, this: :critical])\n\n    # as map\n    Logger.critical(%{this: :critical, something: :reported})","ref":"Logger.html#critical/2-examples"},{"type":"macro","title":"Logger.debug/2","doc":"Logs a debug message.\n\nReturns `:ok`.","ref":"Logger.html#debug/2"},{"type":"macro","title":"Examples - Logger.debug/2","doc":"Logging a message (string or iodata):\n\n    Logger.debug(\"this is a debug message\")\n\nReport message (maps or keywords):\n\n    # as keyword list\n    Logger.debug([something: :reported, this: :debug])\n\n    # as map\n    Logger.debug(%{this: :debug, something: :reported})","ref":"Logger.html#debug/2-examples"},{"type":"function","title":"Logger.default_formatter/1","doc":"Returns the default formatter used by Logger.\n\nIt returns a `Logger.Formatter` built on the `:default_formatter` configuration:\n\n    config :logger, :default_formatter,\n      format: \"\\n$time $metadata[$level] $message\\n\",\n      metadata: [:user_id]\n\nIn case of a list, a set of `overrides` can be given to merge into the list.\nSee `Logger.Formatter.new/1` for all options.","ref":"Logger.html#default_formatter/1"},{"type":"function","title":"Examples - Logger.default_formatter/1","doc":"`Logger` will automatically load a default formatter into the default handler\non boot. However, you can use this function if you wish to programatically replace\na handler formatter. For example, inside tests, you might want to change the formatter\nsettings:\n\n    setup tags do\n      formatter = Logger.default_formatter(colors: [enabled: false])\n      :logger.update_handler_config(:default, :formatter, formatter)\n\n      on_exit(fn ->\n        :logger.update_handler_config(:default, :formatter, Logger.default_formatter())\n      end)\n    end\n\nHowever, note you should not invoke this function inside `config` files,\nas this function expects `Logger` to already be configured and started.\nTo start a brand new handler with this formatter, use `Logger.Formatter.new/1`\ninstead.","ref":"Logger.html#default_formatter/1-examples"},{"type":"function","title":"Logger.delete_all_module_levels/0","doc":"Resets the logging level for all modules to the primary level.","ref":"Logger.html#delete_all_module_levels/0"},{"type":"function","title":"Logger.delete_application_level/1","doc":"Resets logging level for all modules in the given application to the primary level.\n\nEquivalent of:\n\n    appname |> Application.spec(:modules) |> Logger.delete_module_level()","ref":"Logger.html#delete_application_level/1"},{"type":"function","title":"Logger.delete_module_level/1","doc":"Resets the logging level for a given module to the primary level.","ref":"Logger.html#delete_module_level/1"},{"type":"function","title":"Logger.delete_process_level/1","doc":"Resets logging level for the current process to the primary level.\n\nCurrently the only accepted PID is `self()`.","ref":"Logger.html#delete_process_level/1"},{"type":"function","title":"Logger.disable/1","doc":"Disables logging for the current process.\n\nCurrently the only accepted PID is `self()`.\n\nEquivalent of:\n\n    put_process_level(pid, :none)","ref":"Logger.html#disable/1"},{"type":"macro","title":"Logger.emergency/2","doc":"Logs a emergency message.\n\nReturns `:ok`.\n\nThis is reported as \"error\" in Elixir's\nlogger backends for backwards compatibility reasons.","ref":"Logger.html#emergency/2"},{"type":"macro","title":"Examples - Logger.emergency/2","doc":"Logging a message (string or iodata):\n\n    Logger.emergency(\"this is a emergency message\")\n\nReport message (maps or keywords):\n\n    # as keyword list\n    Logger.emergency([something: :reported, this: :emergency])\n\n    # as map\n    Logger.emergency(%{this: :emergency, something: :reported})","ref":"Logger.html#emergency/2-examples"},{"type":"function","title":"Logger.enable/1","doc":"Enables logging for the current process.\n\nCurrently the only accepted PID is `self()`.\n\nEquivalent of:\n\n    delete_process_level(pid)","ref":"Logger.html#enable/1"},{"type":"function","title":"Logger.enabled?/1","doc":"Returns whether the logging is enabled for a given process.\n\nCurrently the only accepted PID is `self()`.","ref":"Logger.html#enabled?/1"},{"type":"macro","title":"Logger.error/2","doc":"Logs a error message.\n\nReturns `:ok`.","ref":"Logger.html#error/2"},{"type":"macro","title":"Examples - Logger.error/2","doc":"Logging a message (string or iodata):\n\n    Logger.error(\"this is a error message\")\n\nReport message (maps or keywords):\n\n    # as keyword list\n    Logger.error([something: :reported, this: :error])\n\n    # as map\n    Logger.error(%{this: :error, something: :reported})","ref":"Logger.html#error/2-examples"},{"type":"function","title":"Logger.flush/0","doc":"Flushes the logger.\n\nThis guarantees all log handlers are flushed. This is useful\nfor testing and it should not be called in production code.","ref":"Logger.html#flush/0"},{"type":"function","title":"Logger.get_module_level/1","doc":"Gets logging level for given module.\n\nThe returned value will be the effective value used. If no value\nwas set for a given module, then it will not be present in\nthe returned list.","ref":"Logger.html#get_module_level/1"},{"type":"function","title":"Logger.get_process_level/1","doc":"Gets logging level for the current process.\n\nCurrently the only accepted PID is `self()`.\n\nThe returned value will be the effective value used. If no value\nwas set for a given process, then `nil` is returned.","ref":"Logger.html#get_process_level/1"},{"type":"macro","title":"Logger.info/2","doc":"Logs a info message.\n\nReturns `:ok`.","ref":"Logger.html#info/2"},{"type":"macro","title":"Examples - Logger.info/2","doc":"Logging a message (string or iodata):\n\n    Logger.info(\"this is a info message\")\n\nReport message (maps or keywords):\n\n    # as keyword list\n    Logger.info([something: :reported, this: :info])\n\n    # as map\n    Logger.info(%{this: :info, something: :reported})","ref":"Logger.html#info/2-examples"},{"type":"function","title":"Logger.level/0","doc":"Retrieves the `Logger` level.\n\nThe `Logger` level can be changed via `configure/1`.","ref":"Logger.html#level/0"},{"type":"macro","title":"Logger.log/3","doc":"Logs a message with the given `level`.\n\nReturns `:ok`.\n\nThe macros `debug/2`, `info/2`, `notice/2`, `warning/2`,\n`error/2`, `critical/2`, `alert/2`, and `emergency/2` are\npreferred over this macro as they can automatically eliminate\nthe call to `Logger` altogether at compile time if desired\n(see the documentation for the `Logger` module).","ref":"Logger.html#log/3"},{"type":"function","title":"Logger.metadata/0","doc":"Reads the current process metadata.\n\nThis does not return the \"global\" logger metadata (set via the `:metadata` key in the\n`:logger` application config), but only the process metadata.","ref":"Logger.html#metadata/0"},{"type":"function","title":"Logger.metadata/1","doc":"Alters the current process metadata according to the given keyword list.\n\nThis function will merge the given keyword list into the existing metadata,\nwith the exception of setting a key to `nil`, which will remove that key\nfrom the metadata.\n\nNote not all keys can be set as metadata. The metadata automatically added\nby Logger, as declared in the module documentation, will always override\ncustom one.","ref":"Logger.html#metadata/1"},{"type":"macro","title":"Logger.notice/2","doc":"Logs a notice message.\n\nReturns `:ok`.\n\nThis is reported as \"info\" in Elixir's\nlogger backends for backwards compatibility reasons.","ref":"Logger.html#notice/2"},{"type":"macro","title":"Examples - Logger.notice/2","doc":"Logging a message (string or iodata):\n\n    Logger.notice(\"this is a notice message\")\n\nReport message (maps or keywords):\n\n    # as keyword list\n    Logger.notice([something: :reported, this: :notice])\n\n    # as map\n    Logger.notice(%{this: :notice, something: :reported})","ref":"Logger.html#notice/2-examples"},{"type":"function","title":"Logger.put_application_level/2","doc":"Puts logging level for modules in a given application.\n\nThis will take priority over the primary level set, so it can be\nused to increase or decrease verbosity of some parts of the project.\n\nEquivalent of:\n\n    appname |> Application.spec(:modules) |> Logger.put_module_level(level)","ref":"Logger.html#put_application_level/2"},{"type":"function","title":"Logger.put_module_level/2","doc":"Puts logging level for given module.\n\nThis will take priority over the primary level set, so it can be\nused to increase or decrease verbosity of some parts of the project.","ref":"Logger.html#put_module_level/2"},{"type":"function","title":"Example - Logger.put_module_level/2","doc":"defmodule Foo do\n      require Logger\n\n      def log, do: Logger.debug(\"foo\")\n    end\n\n    Logger.configure(level: :error)\n    Logger.put_module_level(Foo, :all)\n\n    Foo.log()\n    # This will print the message even if global level is :error","ref":"Logger.html#put_module_level/2-example"},{"type":"function","title":"Logger.put_process_level/2","doc":"Puts logging level for the current process.\n\nCurrently the only accepted PID is `self()`.\n\nThis will take priority over the primary level set, so it can be\nused to increase or decrease verbosity of some parts of the running system.","ref":"Logger.html#put_process_level/2"},{"type":"function","title":"Logger.remove_backend/2","doc":"Removes a backend.","ref":"Logger.html#remove_backend/2"},{"type":"function","title":"Logger.remove_translator/1","doc":"Removes a translator.","ref":"Logger.html#remove_translator/1"},{"type":"function","title":"Logger.reset_metadata/1","doc":"Resets the current process metadata to the given keyword list.","ref":"Logger.html#reset_metadata/1"},{"type":"macro","title":"Logger.warn/2","doc":"","ref":"Logger.html#warn/2"},{"type":"macro","title":"Logger.warning/2","doc":"Logs a warning message.\n\nReturns `:ok`.","ref":"Logger.html#warning/2"},{"type":"macro","title":"Examples - Logger.warning/2","doc":"Logging a message (string or iodata):\n\n    Logger.warning(\"this is a warning message\")\n\nReport message (maps or keywords):\n\n    # as keyword list\n    Logger.warning([something: :reported, this: :warning])\n\n    # as map\n    Logger.warning(%{this: :warning, something: :reported})","ref":"Logger.html#warning/2-examples"},{"type":"type","title":"Logger.level/0","doc":"","ref":"Logger.html#t:level/0"},{"type":"type","title":"Logger.message/0","doc":"","ref":"Logger.html#t:message/0"},{"type":"type","title":"Logger.metadata/0","doc":"","ref":"Logger.html#t:metadata/0"},{"type":"type","title":"Logger.report/0","doc":"","ref":"Logger.html#t:report/0"},{"type":"module","title":"Logger.Formatter","doc":"Conveniences and built-in formatter for logs.\n\nThis modules defines a suitable `:logger` formatter which formats\nmessages and reports as Elixir terms and also provides additional\nfunctionality, such as timezone conversion, truncation, and coloring.\nThis formatter is used by default by `Logger` and you can configure it\nusing:\n\n    config :logger, :default_formatter,\n      format: \"\\n$time $metadata[$level] $message\\n\",\n      metadata: [:user_id]\n\nSee `Logger.Formatter.new/1` for all configuration options.\n\nYou can also build your own instances of this formatter by calling\n`new/1` and setting at the formatter of any `:logger` handler by\nsettings its `:formatter` key to `Logger.Formatter.new(options)`.\n\nThis module also provides several conveniences for those who wish\nto [write their custom logger formatters](https://www.erlang.org/doc/apps/kernel/logger_chapter.html#formatters).","ref":"Logger.Formatter.html"},{"type":"module","title":"Formatting - Logger.Formatter","doc":"The log messages can be controlled by a formatting string. Here is\nan example:\n\n    config :logger, :default_formatter,\n      format: \"\\n$time $metadata[$level] $message\\n\",\n      metadata: [:user_id]\n\nThe above will print error messages as:\n\n    18:43:12.439 user_id=13 [error] Hello\\n\n\nThe valid parameters you can use are:\n\n  * `$time`     - the time the log message was sent\n  * `$date`     - the date the log message was sent\n  * `$message`  - the log message\n  * `$level`    - the log level\n  * `$node`     - the node that prints the message\n  * `$metadata` - user controlled data presented in `\"key=val key2=val2 \"` format\n\n#","ref":"Logger.Formatter.html#module-formatting"},{"type":"module","title":"Formatting function - Logger.Formatter","doc":"You can also customize the format of your log messages with\na `{module, function_name}` tuple if you wish to change how messages\nare formatted but keep all other features provided by `Logger.Formatter`\nsuch as truncation and coloring. However, if you want to get full\ncontrol of formatting, consider writing a custom\n[`:logger` formatter](https://www.erlang.org/doc/apps/kernel/logger_chapter.html#formatters)\ninstead, which has complete access to all events and metadata.\n\nWhen using a `{module, function_name}`, the function will be invoked\nwith the level, the message, the timestamp, and metadata, like this:\n\n    defmodule MyConsoleLogger do\n      @spec format(atom, chardata, Logger.Formatter.date_time_ms(), keyword()) :: IO.chardata()\n      def format(level, message, timestamp, metadata) do\n        # Custom formatting logic that must return chardata.\n        # ...\n      end\n    end\n\n#","ref":"Logger.Formatter.html#module-formatting-function"},{"type":"module","title":"Metadata - Logger.Formatter","doc":"Metadata to be sent to the logger can be read and written with\nthe `Logger.metadata/0` and `Logger.metadata/1` functions. For example,\nyou can set `Logger.metadata([user_id: 13])` to add user_id metadata\nto the current process. The user can configure the backend to choose\nwhich metadata it wants to print and it will replace the `$metadata`\nvalue.","ref":"Logger.Formatter.html#module-metadata"},{"type":"function","title":"Logger.Formatter.compile/1","doc":"Compiles a pattern or function into a data structure that `format/5` can handle.\n\nCheck the module doc for documentation on the valid parameters that\nwill be interpolated in the pattern. If you pass `nil` as the pattern,\nthe pattern defaults to:\n\n    \"\\n$time $metadata[$level] $message\\n\"\n\nIf you want to customize formatting with a custom function, you can\npass a `{module, function_name}` tuple.\n\nThis function, alongside `format/5`, is the main building block used\nby `Logger.Formatter.new/1` for formatting messages. It can also be used\nby those interested in building custom formatters.","ref":"Logger.Formatter.html#compile/1"},{"type":"function","title":"Examples - Logger.Formatter.compile/1","doc":"iex> Logger.Formatter.compile(\"$time $metadata [$level] $message\\n\")\n    [:time, \" \", :metadata, \" [\", :level, \"] \", :message, \"\\n\"]\n\n    iex> Logger.Formatter.compile({MyLoggerFormatter, :format})\n    {MyLoggerFormatter, :format}","ref":"Logger.Formatter.html#compile/1-examples"},{"type":"function","title":"Logger.Formatter.format/5","doc":"Formats a `pattern_or_function` returned by `compile/1`.\n\nIt takes a compiled format and injects the level, timestamp, message,\nand metadata keyword list and returns a properly formatted string.\n\nIf `pattern_or_function` is a `{module, function_name}` tuple,\nthen `module.function_name(level, message, timestamp, metadata)` is\ninvoked to get the message.\n\nThis function, alongside `compile/1`, is the main building block used\nby `Logger.Formatter.new/1` for formatting messages. It can also be used\nby those interested in building custom formatters.","ref":"Logger.Formatter.html#format/5"},{"type":"function","title":"Examples - Logger.Formatter.format/5","doc":"iex> pattern = Logger.Formatter.compile(\"[$level] $message\")\n    iex> timestamp = {{1977, 01, 28}, {13, 29, 00, 000}}\n    iex> formatted = Logger.Formatter.format(pattern, :info, \"hello\", timestamp, [])\n    iex> IO.chardata_to_string(formatted)\n    \"[info] hello\"","ref":"Logger.Formatter.html#format/5-examples"},{"type":"function","title":"Logger.Formatter.format_date/1","doc":"Formats date as chardata.","ref":"Logger.Formatter.html#format_date/1"},{"type":"function","title":"Logger.Formatter.format_event/2","doc":"Formats the message of a log event.","ref":"Logger.Formatter.html#format_event/2"},{"type":"function","title":"Logger.Formatter.format_time/1","doc":"Formats time as chardata.","ref":"Logger.Formatter.html#format_time/1"},{"type":"function","title":"Logger.Formatter.new/1","doc":"Initializes a formatter for `:logger` handlers.\n\nThe supported options are:\n\n  * `:colors` - a keyword list of coloring options.\n\n  * `:format` - the format message used to print logs.\n    Defaults to: `\"\\n$time $metadata[$level] $message\\n\"`.\n    It may also be a `{module, function_name}` tuple that is invoked\n    with the log level, the message, the current timestamp and\n    the metadata and must return `t:IO.chardata/0`.\n    See the module docs for more information on `:format`.\n\n  * `:metadata` - the metadata to be printed by `$metadata`.\n    Defaults to an empty list (no metadata).\n    Setting `:metadata` to `:all` prints all metadata. See\n    the \"Metadata\" section in the `Logger` documentation for\n    more information.\n\n  * `:truncate` - the maximum message size to be logged (in bytes).\n    Defaults to 8192 bytes. Note this configuration is approximate.\n    Truncated messages will have `\" (truncated)\"` at the end.\n    The atom `:infinity` can be passed to disable this behavior.\n\n  * `:utc_log` - when `true`, uses UTC in logs. By default it uses\n    local time (as it defaults to `false`).\n\nThe supported keys in the `:colors` keyword list are:\n\n  * `:enabled` - boolean value that allows for switching the\n    coloring on and off. Defaults to: `IO.ANSI.enabled?/0`\n\n  * `:debug` - color for debug messages. Defaults to: `:cyan`\n\n  * `:info` - color for info and notice messages. Defaults to: `:normal`\n\n  * `:warning` - color for warning messages. Defaults to: `:yellow`\n\n  * `:error` - color for error and higher messages. Defaults to: `:red`\n\nSee the `IO.ANSI` module for a list of colors and attributes.\nThe color of the message can also be configured per message via\nthe `:ansi_color` metadata.","ref":"Logger.Formatter.html#new/1"},{"type":"function","title":"Logger.Formatter.prune/1","doc":"Prunes invalid Unicode code points from lists and invalid UTF-8 bytes.\n\nTypically called after formatting when the data cannot be printed.","ref":"Logger.Formatter.html#prune/1"},{"type":"function","title":"Logger.Formatter.system_time_to_date_time_ms/2","doc":"Converts the system time (in microseconds) from metadata into a `date_time_ms` tuple.","ref":"Logger.Formatter.html#system_time_to_date_time_ms/2"},{"type":"function","title":"Logger.Formatter.truncate/2","doc":"Truncates a `chardata` into `n` bytes.\n\nThere is a chance we truncate in the middle of a grapheme\ncluster but we never truncate in the middle of a binary\ncode point. For this reason, truncation is not exact.","ref":"Logger.Formatter.html#truncate/2"},{"type":"type","title":"Logger.Formatter.date/0","doc":"","ref":"Logger.Formatter.html#t:date/0"},{"type":"type","title":"Logger.Formatter.date_time_ms/0","doc":"","ref":"Logger.Formatter.html#t:date_time_ms/0"},{"type":"type","title":"Logger.Formatter.pattern/0","doc":"","ref":"Logger.Formatter.html#t:pattern/0"},{"type":"type","title":"Logger.Formatter.time_ms/0","doc":"","ref":"Logger.Formatter.html#t:time_ms/0"},{"type":"behaviour","title":"Logger.Translator","doc":"Default translation for Erlang log messages.\n\nLogger allows developers to rewrite log messages provided by\nOTP applications into a format more compatible with Elixir\nlog messages by providing a translator.\n\nA translator is simply a tuple containing a module and a function\nthat can be added and removed via the `Logger.add_translator/1` and\n`Logger.remove_translator/1` functions and is invoked for every Erlang\nmessage above the minimum log level with four arguments:\n\n  * `min_level` - the current Logger level\n  * `level` - the level of the message being translated\n  * `kind` - if the message is a `:report` or `:format`\n  * `message` - the message to format. If it is `:report`, it is a tuple\n    with `{report_type, report_data}`, if it is `:format`, it is a\n    tuple with `{format_message, format_args}`.\n\nThe function must return:\n\n  * `{:ok, chardata, metadata}` - if the message translation with its metadata\n  * `{:ok, chardata}` - the translated message\n  * `:skip` - if the message is not meant to be translated nor logged\n  * `:none` - if there is no translation, which triggers the next translator\n\nSee the function `translate/4` in this module for an example implementation\nand the default messages translated by Logger.","ref":"Logger.Translator.html"},{"type":"function","title":"Logger.Translator.translate/4","doc":"Built-in translation function.","ref":"Logger.Translator.html#translate/4"},{"type":"callback","title":"Logger.Translator.translate/4","doc":"Callback for translating a logger message.","ref":"Logger.Translator.html#c:translate/4"},{"type":"module","title":"Logger.Backends.Console","doc":"A logger backend that logs messages by printing them to the console.\n\nThis backend was typically configured as `config :logger, :console`,\nbut it has been deprecated in favor of `:default_handler` and\n`:default_formatter`. However, for backwards compatibility, you can\nstill add it as:\n\n    config :logger, :backends, [Logger.Backends.Console]\n\nHowever, if you plan to continue using Logger backends in the long\nterm, consider using the [`:logger_backends`](https://github.com/elixir-lang/logger_backends)\nproject.","ref":"Logger.Backends.Console.html"}],"content_type":"text/markdown"}