/* ====================================================================
 *
 *
 *   BSD LICENSE
 *
 *   Copyright(c) 2016-2024 Intel Corporation.
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in
 *       the documentation and/or other materials provided with the
 *       distribution.
 *     * Neither the name of Intel Corporation nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * ====================================================================
 */

/*
 * This file is based on modified code from OpenSSL.
 * This is needed because the constant time functions are not exported
 * from OpenSSL forcing engines to have their own copy of the
 * functionality.
 * The code based on OpenSSL code is subject to the following license:
 */

/*
 * Copyright 2014-2016 The OpenSSL Project Authors. All Rights Reserved.
 *
 * Licensed under the OpenSSL license (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */

/*****************************************************************************
 * @file qat_constant_time.h
 *
 * This file provides constant time functions
 *
 *****************************************************************************/

#ifndef QAT_CONST_TIME_H
# define QAT_CONST_TIME_H

#ifdef __cplusplus
extern "C" {
#endif

#pragma GCC push_options
#pragma GCC optimize ("O0")

static inline unsigned int value_barrier(unsigned int a)
{
    volatile unsigned int r = a;
    return r;
}

static inline unsigned int qat_constant_time_select(unsigned int mask,
                                                     unsigned int a,
                                                     unsigned int b)
{
    return (value_barrier(mask) & a) | (value_barrier(~mask) & b);
}

static inline unsigned char qat_constant_time_select_8(unsigned char mask,
                                                        unsigned char a,
                                                        unsigned char b)
{
    return (unsigned char)qat_constant_time_select(mask, a, b);
}

static inline unsigned int qat_constant_time_msb(unsigned int a)
{
    return 0 - (a >> (sizeof(a) * 8 - 1));
}

static inline unsigned int qat_constant_time_lt(unsigned int a,
                                                unsigned int b)
{
    return qat_constant_time_msb(a ^ ((a ^ b) | ((a - b) ^ b)));
}

static inline unsigned int qat_constant_time_ge(unsigned int a,
                                                unsigned int b)
{
    return ~qat_constant_time_lt(a, b);
}

static inline unsigned char qat_constant_time_ge_8(unsigned int a,
                                                   unsigned int b)
{
    return (unsigned char)(qat_constant_time_ge(a, b));
}

static inline unsigned int qat_constant_time_is_zero(unsigned int a)
{
    return qat_constant_time_msb(~a & (a - 1));
}

static inline unsigned int qat_constant_time_eq(unsigned int a,
                                                unsigned int b)
{
    return qat_constant_time_is_zero(a ^ b);
}

static inline unsigned char qat_constant_time_is_zero_8(unsigned int a)
{
    return (unsigned char)qat_constant_time_is_zero(a);
}

static inline size_t value_barrier_s(size_t a)
{
    volatile size_t r = a;
    return r;
}

static inline size_t qat_constant_time_select_s(size_t mask,
                                                size_t a,
                                                size_t b)
{
    return (value_barrier_s(mask) & a) | (value_barrier_s(~mask) & b);
}

static inline size_t qat_constant_time_msb_s(size_t a)
{
    return 0 - (a >> (sizeof(a) * 8 - 1));
}

static inline int qat_constant_time_select_int(unsigned int mask, int a,
                                               int b)
{
    return (int)qat_constant_time_select(mask, (unsigned)(a), (unsigned)(b));
}

/* This function works like qat_constant_time_select but
 * operates on pointer types. This would return either
 * "a" (if "i" is 1) or "b" (if "i" is 0). */
static inline void *qat_constant_time_select_ptr(int i, void *a, void *b)
{
    /* i should be 0 or 1 to avoid running into pointer issues. */
    i = i & 1;
    intptr_t time_int_result = ((~(i - 1)) & (intptr_t)a) | ((i - 1) & (intptr_t)b);
    return (void *)time_int_result;
}

/* This function returns 1 in case of a<=b else 0. */
static inline int qat_constant_time_le_int(int a, int b)
{
    return !(((b + (~a + 1)) >> (sizeof(int) * 8 - 1)) & 1);
}

#pragma GCC pop_options
#ifdef __cplusplus
}
#endif

#endif                          /* QAT_CONST_TIME_H */
