# -*- coding: utf-8 -*-
"""\
XBM related tests.
"""
from __future__ import unicode_literals, absolute_import
import os
import pytest
import pyqrcodeng as pyqrcode

#Create by:
#   First I ran: pyqrcode.create('Test', scale=1).png('test.png')
#   Next, I used GIMP to convert it to a XBM file.
expected = '''#define im_width 29
#define im_height 29
static unsigned char im_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0xf0, 0x17, 0xfd, 0x01, 0x10, 0xe4, 0x05, 0x01,
   0xd0, 0xb5, 0x75, 0x01, 0xd0, 0xc5, 0x74, 0x01, 0xd0, 0xb5, 0x74, 0x01,
   0x10, 0x24, 0x05, 0x01, 0xf0, 0x57, 0xfd, 0x01, 0x00, 0x70, 0x01, 0x00,
   0x00, 0x06, 0x55, 0x01, 0xf0, 0xf0, 0x76, 0x00, 0xc0, 0xb5, 0xf3, 0x00,
   0x80, 0x82, 0x70, 0x00, 0x40, 0xd4, 0x06, 0x01, 0x00, 0x50, 0xcc, 0x00,
   0xf0, 0x47, 0xac, 0x00, 0x10, 0xb4, 0x7b, 0x01, 0xd0, 0x45, 0xaa, 0x00,
   0xd0, 0xe5, 0x66, 0x00, 0xd0, 0x25, 0xe3, 0x01, 0x10, 0x64, 0x57, 0x00,
   0xf0, 0xa7, 0xd5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };'''


def decompose_xbm(s):
    import re
    
    width = re.search('width ([0-9]+)', s).group(1)
    height = re.search('height ([0-9]+)', s).group(1)
    bits = re.findall(r'(0x\d{2,})', s)
    return width, height, bits


def test_xbm():
    """Test the xbm render against a known good.
    
    This test checks the *values* contained in the XBM, not the text.
    """
    c = pyqrcode.create('Test').xbm(scale=1)
    
    #Testing number-by-number to get more useful failure message
    c_width, c_height, c_bits = decompose_xbm(c)
    e_width, e_height, e_bits = decompose_xbm(expected)
    
    #Check the there is the same width and height
    assert c_width == e_width
    assert c_height == e_height
    
    #Check that there is the same number of bits
    assert len(c_bits) == len(e_bits)
    
    #Check the bit values
    for i in range(len(e_bits)):
        assert c_bits[i] == e_bits[i], "Wrong value at {0}: {1} != {2}".format(i, c_bits[i], e_bits[i])


def test_xbm_scale():
    """Test the xbm render against a known good.

    This test checks the *values* contained in the XBM, not the text.
    """
    expected_with_scale = '''#define im_width 116
#define im_height 116
static unsigned char im_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x00, 0x0f, 0xff, 0xff, 0xff,
   0x0f, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x00, 0x0f,
   0xff, 0xff, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x0f,
   0x0f, 0x00, 0x0f, 0xff, 0xff, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xff,
   0xff, 0xff, 0x0f, 0x0f, 0x00, 0x0f, 0xff, 0xff, 0xff, 0x0f, 0x00, 0x00,
   0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f, 0xf0, 0xff, 0x0f, 0x0f, 0x00, 0x00,
   0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f, 0xf0, 0xff, 0x0f,
   0x0f, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f,
   0xf0, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f,
   0x00, 0x00, 0x0f, 0xf0, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x0f, 0x00, 0x00,
   0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0xff, 0xf0, 0x0f, 0x0f, 0xff, 0x0f,
   0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0xff, 0xf0, 0x0f,
   0x0f, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f,
   0xff, 0xf0, 0x0f, 0x0f, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f,
   0xff, 0x0f, 0x0f, 0xff, 0xf0, 0x0f, 0x0f, 0xff, 0x0f, 0x0f, 0x00, 0x00,
   0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0x00, 0xff, 0x00, 0x0f, 0xff, 0x0f,
   0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0x00, 0xff, 0x00,
   0x0f, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f,
   0x00, 0xff, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f,
   0xff, 0x0f, 0x0f, 0x00, 0xff, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0x00, 0x00,
   0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0xff, 0xf0, 0x00, 0x0f, 0xff, 0x0f,
   0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0xff, 0xf0, 0x00,
   0x0f, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f,
   0xff, 0xf0, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f,
   0xff, 0x0f, 0x0f, 0xff, 0xf0, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0x00, 0x00,
   0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x0f, 0x0f, 0x00, 0x00,
   0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f, 0xf0, 0x00, 0x0f,
   0x0f, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f,
   0xf0, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f,
   0x00, 0x00, 0x0f, 0xf0, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x0f, 0x00, 0x00,
   0x00, 0x00, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff,
   0x0f, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
   0xff, 0xff, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x0f,
   0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xff,
   0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0x0f, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x0f, 0x0f,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0xf0, 0x0f, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
   0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0x0f, 0x00, 0x00, 0x0f,
   0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0x0f,
   0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0xf0, 0x0f, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00,
   0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xff, 0xff, 0xf0, 0x0f, 0xff, 0x0f,
   0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xff, 0xff, 0xf0,
   0x0f, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00,
   0xff, 0xff, 0xf0, 0x0f, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
   0xff, 0x00, 0x00, 0xff, 0xff, 0xf0, 0x0f, 0xff, 0x0f, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xff, 0x0f, 0x0f, 0xff, 0xf0, 0xff, 0x00, 0xff, 0xff,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x0f, 0x0f, 0xff, 0xf0, 0xff,
   0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x0f, 0x0f,
   0xff, 0xf0, 0xff, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0x0f, 0x0f, 0xff, 0xf0, 0xff, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xf0, 0xf0, 0x00, 0x00, 0xf0, 0x00, 0x00, 0xff, 0x0f,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0x00, 0x00, 0xf0, 0x00,
   0x00, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0x00,
   0x00, 0xf0, 0x00, 0x00, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xf0, 0xf0, 0x00, 0x00, 0xf0, 0x00, 0x00, 0xff, 0x0f, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x0f, 0x00, 0x0f, 0x0f, 0xff, 0xf0, 0x0f, 0x00, 0x00,
   0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x0f, 0x0f, 0xff, 0xf0,
   0x0f, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x0f,
   0x0f, 0xff, 0xf0, 0x0f, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x0f, 0x00, 0x0f, 0x0f, 0xff, 0xf0, 0x0f, 0x00, 0x00, 0x0f, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0xff, 0x00, 0xff,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00,
   0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x0f, 0x0f, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0xff, 0xff, 0xff, 0x0f, 0x00, 0x0f, 0x00, 0xff, 0xf0, 0xf0,
   0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x0f, 0x00, 0x0f, 0x00,
   0xff, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x0f,
   0x00, 0x0f, 0x00, 0xff, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
   0xff, 0xff, 0x0f, 0x00, 0x0f, 0x00, 0xff, 0xf0, 0xf0, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f, 0xff, 0xf0, 0xff, 0xf0, 0xff, 0x0f,
   0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f, 0xff, 0xf0, 0xff,
   0xf0, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f,
   0xff, 0xf0, 0xff, 0xf0, 0xff, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f,
   0x00, 0x00, 0x0f, 0xff, 0xf0, 0xff, 0xf0, 0xff, 0x0f, 0x0f, 0x00, 0x00,
   0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0x00, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0x00, 0x0f, 0xf0,
   0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f,
   0x00, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f,
   0xff, 0x0f, 0x0f, 0x00, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0xf0, 0xff, 0xf0, 0x0f, 0xf0, 0x0f,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0xf0, 0xff, 0xf0,
   0x0f, 0xf0, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f,
   0xf0, 0xff, 0xf0, 0x0f, 0xf0, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f,
   0xff, 0x0f, 0x0f, 0xf0, 0xff, 0xf0, 0x0f, 0xf0, 0x0f, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0xf0, 0x00, 0xff, 0x00, 0xf0, 0xff,
   0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f, 0xf0, 0x00, 0xff,
   0x00, 0xf0, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x0f, 0x0f,
   0xf0, 0x00, 0xff, 0x00, 0xf0, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f,
   0xff, 0x0f, 0x0f, 0xf0, 0x00, 0xff, 0x00, 0xf0, 0xff, 0x0f, 0x00, 0x00,
   0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f, 0xf0, 0x0f, 0xff, 0x0f, 0x0f, 0x0f,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f, 0xf0, 0x0f, 0xff,
   0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x0f,
   0xf0, 0x0f, 0xff, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f,
   0x00, 0x00, 0x0f, 0xf0, 0x0f, 0xff, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00,
   0x00, 0x00, 0xff, 0xff, 0xff, 0x0f, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0xff,
   0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x0f, 0xf0, 0xf0, 0x0f,
   0x0f, 0x0f, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x0f,
   0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
   0xff, 0xff, 0x0f, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
'''

    c = pyqrcode.create('Test', error='h').xbm(scale=4)

    # Testing number-by-number to get more useful failure message
    c_width, c_height, c_bits = decompose_xbm(c)
    expected_width, expected_height, expected_bits = decompose_xbm(expected_with_scale)

    # Check the there is the same width and height
    assert c_width == expected_width
    assert c_height == expected_height

    # Check that there is the same number of bits
    assert len(expected_bits) == len(c_bits)

    # Check the bit values
    for i in range(len(expected_bits)):
        assert c_bits[i] == expected_bits[i], "Wrong value at {0}: {1} != {2}".format(i, c_bits[i], expected_bits[i])


@pytest.mark.skipif(not os.getenv('DISPLAY'), reason='No screen environment')
def test_xbm_with_tkinter():
    """Test XBM renderer is compatible with Tkinter
    """
    #Python 2 vs 3
    try:
        import Tkinter as tkinter
    except:
        import tkinter

    code = pyqrcode.create('Test')
    width, height = code.symbol_size(scale=1)
    code_xbm = code.xbm(scale=1)
    
    top = tkinter.Tk()
    bitmap = tkinter.BitmapImage(data=code_xbm)

    assert bitmap.width() == width
    assert bitmap.height() == height


if __name__ == '__main__':
    pytest.main([__file__])

