/* This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "viocons.h"
#include <stdio.h>
#include <ctype.h>
#include <termio.h>
#include <setjmp.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>
#include <stdlib.h>
#include <stdlib.h>


#define BUFSIZE		256

#define DEFAULT_COLS	80
#define DEFAULT_ROWS	24
#define TSET		"/usr/bin/tset"
#define STTY		"/bin/stty"
#define ALLOWED		2

/* useful bits */
#define SAVE_CURSOR	"\0337"
#define RESTORE_CURSOR	"\0338"
#define POSITION_CURSOR	"\033[999;999H"
#define REPORT_CURSOR	"\033[6n"
#define CURSOR_OFF	"\033[?25l"
#define CURSOR_ON	"\033[?25h"
#define BCE_ON		"\033[=0L"
#define BS_IS_DEL	"\033[?67l"

/* probes */
#define TERM_DA		"\033[c"
#define EXTENDED_DA	"\033[?1;1c"
#define SECONDARY_DA	"\033[>c"
#define OLD_TERMID	"\033Z"
#define ANSWERBACK	"\005"
#define TITLE		"\033[21;t"

/* replies */
#define PUTTY		"PuTTY"
#define PUTTY_SECDA	"\033[>0;136;0c"	/* we'll compare only the first bytes of these, */
#define GNOME_SECDA	"\033[>1;0;0c"		/* as the middle number will vary (version) */
#define KTERM_SECDA	"\033[?1;2c"
#define TERMAPP_SECDA	"\033[?1;2c"
#define SCREEN_SECDA	"\033[>83;$version;0c"
#define RXVT_SECDA	"\033[>82;$version;0c"
#define TERA_SECDA	"\033[>32;$version;2c"
#define XTERM_SECDA	"\033[>0;"


/* for Darwin 8.6 */
#ifndef TCSETA
#define TCSETA TIOCSETA
#endif
#ifndef TCGETA
#define TCGETA TIOCGETA
#endif

#ifndef IUCLC 		/* IUCLC (translate uppercase characters to lowercase) */
#define IUCLC 0 	/* is not know on Darwin 8.6 */
#endif


/* terminal modes */
enum { RAW = 0,	COOKED = 1, TOGGLE = 2 };

enum { linuxcons, xterm, kterm, putty, konsole, gnome, rxvt, mlterm, screen, teraterm, win, termapp, vt100, vt220, vt52, unknown = 255 };

static struct {
	int	termid;			
	char	terminfoname [32];		/* terminal name (as in terminfo name) */
} termname [] = {				/* this is what $TERM will be set to */
	{linuxcons,	"linux"},		
	{xterm,		"xterm"},	
	{kterm,		"kterm"},	
	{mlterm,	"mlterm"},	
	{konsole,	"xterm"},
	{gnome,		"gnome"},
	{rxvt,		"rxvt"},
	{win,		"vt100"},
	{putty,		"xterm"},	
	{screen,	"screen"},	
	{teraterm,	"vt100"},	
	{termapp, 	"xterm-color"},	
	{vt220,		"vt220"},
	{vt100,		"vt100"},
	{vt52,		"vt52"},	
	{255, 		"unknown"}
};

static struct {
	char	reply [32];		/* answerback response (primary device attr (DA) */
	int	termid;			/* terminal name (not the terminfo name) */
} primaryda [] = {
	{"\033[>0;115;0c",	konsole},
	{"\033[?1;0c",		win},		/* Base vt100 */
	{"\033[?1;2c",		xterm},		/* ANSI/VT100 Clone */
	{"\033[?63;",		vt220},
	{"\033[?6c",		linuxcons},	/* vt102 */
	{"\033[?1;",		vt100},
	{"\033[?1;1c",		vt100},		/* vt100 with STP */
	{"\033[?1;3c",		vt100},		/* vt100 with AVO and STP */
	{"\033[?1;4c",		vt100},		/* vt100 with GPO */
	{"\033[?1;5c",		vt100},		/* vt100 with GPO and STP */
	{"\033[?1;6c",		vt100},		/* vt100 with GPO and AVO */
	{"\033[?1;7c",		vt100},		/* vt100 with GPO, STP, and AVO */
	{"\033[?8c",		vt100},		/* TeleVideo 970 */
	{"\033/Z",		vt52},		/* Generic vt52 */
	{"\033[?64;1;2;6;7;8;9;15;18;19;21c",   vt100}, /* WebSM console */
	{"",0}
};

void usage(void);
void strascii();
void settermmode(int);
int readreply(int, char *, int, int);
int termsend(int, char *, char *);
int sendenq(int, char *);
int sendescz(int, char *);
int sendsecda(int, char *);
int querytitle(int, char *);
int queryscreensize(int, int *, int *, char *);
int debug_show(char *);
int check_multiple(int, char *, char *);
int findcompatible(char *, char *);
int resolve_terminfo(int, char *);
void warn_mutiple(void);

int Debug;
int Quiet;
int Igncr;

/*****************************************************************************
	Main 
*****************************************************************************/
int main(int argc, char *argv[])
{
	char da    [BUFSIZE+1] = "";	/* reply to DA (device attributes) */
	char secda [BUFSIZE+1] = ""; 	/* reply to secondary DA */
	char enq   [BUFSIZE+1] = "";	/* reply to old termid */
	char escz  [BUFSIZE+1] = "";	/* reply to old termid */

	int opt;			/* for getopt() */
	int should_init = 1;		/* shall we do any initialization? */
	int output_term_for_eval = 0;	/* print TERM for eval? */
	int output_size_for_eval = 0;	/* print LINES+COLUMNS for eval? */
	int csh_user = 0;		/* the eval oputput belongs to csh */
	int fd;
	struct stat statbuf;

	char device[128] = "";
	char termstr[128] = "";
	int multiple_detected = 0;
	int foundterm = 0;
	int cols = DEFAULT_COLS;
	int lines = DEFAULT_ROWS;
	char lines_s[4], cols_s[4];
	char model[42];

	/* initviocons breaks fb on ps3, so disable it */
	fd = open("/proc/device-tree/model", O_RDONLY);
	if (fd >=0) {
		opt = read(fd, model, sizeof(model) - 1);
		if (opt > 0) {
			model[opt] = '\0';
			if (strcmp("PLAYSTATION 3", model) == 0)
				exit(0);
			if (strcmp("SonyPS3", model) == 0) {
				exit(0);
			}
		}
		close(fd);
	}

	Debug = 0;
	Quiet = 0;
	Igncr = 0;

	while ((opt = getopt(argc, argv, "dqpesF:ch")) != -1) {
		switch (opt) {
			case 'd': Debug++; break;
			case 'q': Quiet++; break;
			case 'p': should_init=0; break;
			case 'e': output_term_for_eval++; break;
			case 's': output_size_for_eval++; break;
			case 'F': sprintf(device, "%s", optarg); break;
			case ':': printf("option needs a value\n"); break;
			case 'c': csh_user++; break;
			case 'h': usage(); break;
			case '?': usage(); break;
		}
	}
	if (Debug) Quiet = 0;

	/* find the actual terminal */
	if (device[0] == 0) {
		if ( readlink("/proc/self/fd/0", device, sizeof(device)) == -1 )
			sprintf(device, "%s", ctermid(NULL)); 
	}
	if (Debug) printf("%s\n", device); 

	if ((fd = open(device, O_RDWR | O_NDELAY)) == -1) {
		if ( errno == EACCES ) {
			fprintf(stderr, "%s: can't initialize terminal if we do not own it.\n", PROGRAM); 
		} else {
			perror("main: open");
			fprintf(stderr, "Try -F `tty`\n");
		}
		exit(1);
	}

	settermmode(RAW);
	termsend(fd, CURSOR_OFF, "switching cursor off\n"); 

	if (!Quiet) {
		settermmode(TOGGLE);
		fprintf(stderr, "Probing connected terminal...\n");
		settermmode(TOGGLE);
	}


	/* are we running on IBM iSeries? */
	if (stat("/proc/iSeries", &statbuf) == 0) {

		/* check whether multiple terminals are connected on the same line 
		 * (this can happen on IBM iSeries where a terminal server is used for the 
		 * virtual console). 
		 *
		 * Other architectures can save time here. (Over the network, it can easily take a second.)
		 *
		 * We will probe for primary device attributes (DA). If the response is valid, 
		 * we can use it below when we begin the identification. 
		 */
		if (check_multiple(fd, "Probing for multiple terminals\n", da)) {
			warn_mutiple();
			multiple_detected = 1;
			foundterm = vt100;
		}
	}

	if (!multiple_detected) {

		/* probe for primary device attributes (terminal id) with ESC [ c */

		if (!*da) {
			termsend(fd, TERM_DA, "asking for device attributes: "); 
			readreply(fd, da, 1, 0);
		}
		if (*da) 
			foundterm = findcompatible(da, termstr);
		else 
			foundterm = vt100;

		if (Debug) { settermmode(TOGGLE); printf("foundterm 1st catch: %d\n", foundterm); settermmode(TOGGLE); }

		/* determine screen size */
		if ( !queryscreensize(fd, &cols, &lines, "\n") ) {
			if (Debug) printf("could not get screen size\n");
			cols=DEFAULT_COLS; lines=DEFAULT_ROWS;
		}
		if (Debug) printf ("%d cols, %d lines \n", cols, lines);

		/*
		 * a terminal size smaller than the defaults doesnt make sense
		 * this will happen on the POWER4/POWER5 hvc console
		 */
		if (DEFAULT_COLS >= cols)
			cols = DEFAULT_COLS;
		if (DEFAULT_ROWS >= lines)
			lines = DEFAULT_ROWS;

		/* get more info to discriminate terminals */
		switch (foundterm) {
			case konsole:
			case win: 
			case vt100: 
				/* we know enough */
				break;

			case linuxcons:
				sendescz(fd, escz);
				if (!*escz) foundterm=gnome; 
				else {
					sendenq(fd, enq);
					if (*enq) {
						if (strncmp(enq, PUTTY, strlen(PUTTY)) == 0)
						foundterm=putty;
					} else foundterm=linuxcons;
				}
				break;

			case xterm:
				sendsecda(fd, secda);
				if (*secda) {
					if (strncmp(secda, TERMAPP_SECDA, 7) == 0) 
						foundterm=termapp; 
					else if (strncmp(secda, SCREEN_SECDA, 6) == 0)
						foundterm=screen; 
					else if (strncmp(secda, RXVT_SECDA, 6) == 0)
						foundterm=rxvt; 
					else if (strncmp(secda, TERA_SECDA, 6) == 0)
						foundterm=teraterm; 
					else if (strncmp(secda, XTERM_SECDA, 5) == 0)
						foundterm=xterm; 
				} 
				else foundterm=mlterm; 
				break;
		} 

		if (Debug) { settermmode(TOGGLE); printf("foundterm: %d\n", foundterm); settermmode(TOGGLE); }


	} /* end if (!multipledetected) */


	/* initialize the terminal */
	if (should_init) {
		if (!Quiet) { settermmode(TOGGLE); fprintf(stderr, "\nInitializing virtual console...\n"); settermmode(TOGGLE); }
		if (Debug) { settermmode(TOGGLE); printf("\nrunning tset\n"); settermmode(TOGGLE); }
		{
			int ret;
			int child = fork();

			if (child == 0) {
				execl(TSET, "-I", "-Q", NULL);
				_exit(127);
			}
			if (child > 0) {
				int status;
				do
					status = waitpid(child, &ret, 0);
				while (status == -1 && errno == EINTR);
			}
			else ret = -1;
		}

		if (Debug) { 
			settermmode(TOGGLE); 
			printf(STTY " sane rows %d cols %d", lines, cols);
			printf("\nrunning stty\n"); 
			settermmode(TOGGLE); 
		}

		sprintf(lines_s, "%d", lines);
		sprintf(cols_s, "%d", cols);

		{
			int ret;
			int child = fork();

			if (child == 0) {
				execl(STTY, "sane", "rows", lines_s, "cols", cols_s, NULL);
				_exit(127);
			}
			if (child > 0) {
				int status;
				do
					status = waitpid(child, &ret, 0);
				while (status == -1 && errno == EINTR);
			}
			else ret = -1;
		}



		/* special treatments */
		if (foundterm == putty) termsend(fd, BCE_ON, "switching BCE on for PuTTY\n");
		if (foundterm == xterm) termsend(fd, BS_IS_DEL, "setting erase = DEL for xterm\n");

		/* PuTTY and Windows telnet clients usually send CRLF when Enter is pressed...
		 * Before MF28151 and MF***** [non approved v5r1 ptf.], we could ignore carriage return 
		 * because the telnet server would not read it as NVT newline but pass both through instead 
		 */ 
		/* if ((foundterm == win) || (foundterm == putty)) Igncr = 1; */


	}
	termsend(fd, CURSOR_ON, "switching cursor on\n"); 
	settermmode(COOKED);

	/* resolve to the actual terminfo entry */
	foundterm = resolve_terminfo(foundterm, termstr);
		if (Debug) { settermmode(TOGGLE); printf("foundterm: %d\n", foundterm); settermmode(TOGGLE); }

	if (!Quiet) {
		fprintf(stderr, "\nFound a");
		switch (foundterm) {
			case xterm: 	fprintf(stderr, "n xterm"); break;
			case linuxcons:	fprintf(stderr, " Linux console"); break;
			case putty: 	fprintf(stderr, " PuTTY"); break;
			case win: 	fprintf(stderr, " Windows"); break;
			case gnome: 	fprintf(stderr, " Gnome"); break;
			case konsole: 	fprintf(stderr, " KDE konsole"); break;
			case kterm: 	fprintf(stderr, " kterm"); break;
			case mlterm: 	fprintf(stderr, " mlterm"); break;
			case teraterm: 	fprintf(stderr, " TeraTerm (vt100)"); break;
			case termapp:	fprintf(stderr, " Terminal.app (OSX)"); break;
			case screen: 	fprintf(stderr, " screen"); break;
			case rxvt: 	fprintf(stderr, " rxvt"); break;
			default: 	fprintf(stderr, "n unknown"); break;
		}
		fprintf(stderr, " terminal on %s (%d columns x %d lines).\n", device, cols, lines);

		/* additional bla, not to be shown in the instsys */
		if ( (stat("/sbin/inst_setup", &statbuf) == -1) &&
		     (stat("/var/lib/YaST2/runme_at_boot", &statbuf) == -1)) {
			fprintf(stderr, "Use the 'initviocons' command whenever the terminal has been resized.\n");
		}

	}

	/* to be evaled by the shell */
	if (output_term_for_eval) {
		if (csh_user) {
			printf ("setenv TERM %s;\n", termstr);
		} else {
			printf ("TERM=%s;\n", termstr);
		}
	}
	if (output_size_for_eval) {
		if (csh_user) {
			printf ("setenv COLUMNS %d;\n", cols);
			printf ("setenv LINES %d\n", lines);
		} else {
			printf ("COLUMNS=%d;\n", cols);
			printf ("LINES=%d;\n", lines);
			printf ("export COLUMNS LINES\n");
		}
	}

	/* FIXME: signal handling */
	/* FIXME: atexit: stty -igncr */

	close(fd);
	if (multiple_detected) exit(1);
	exit(0);
}	/* end of main */


/*****************************************************************************
	termsend
*****************************************************************************/
int termsend(int fd, char *str, char *text)
{
	if (Debug) {
		settermmode(TOGGLE);
		printf("\033[100D%-50s ", text);
		settermmode(TOGGLE);
	}
	if (write(fd, str, strlen(str)) == -1) {
		settermmode(COOKED);
		perror("main: write");
		exit(1);
	}
	return (0);
}
/*****************************************************************************
	check_multiple
*****************************************************************************/
int check_multiple(int fd, char *text, char *da)
{
	char	buf1 [BUFSIZE+1];
	char	buf2 [BUFSIZE+1];
	char	buf3 [BUFSIZE+1];
	char	buf4 [BUFSIZE+1];

	if (Debug) {
		settermmode(TOGGLE);
		printf("%s", text);
		settermmode(TOGGLE);
	}
	
	termsend(fd, TERM_DA, ""); 
	readreply(fd, buf1, 2, 0);

	/* note: it might be cooler to just concatenate all strings and count all 'c' */
	if (*buf1) { 
		if ( buf1[0] == 0x0a ) {
			/* a windows client is connected -- it sent CR LF when user pressed enter, 
			 * and the LF leaked through. So we repeat the first read. */
			readreply(fd, buf1, 0, 700);
		} 
		if ( strlen(buf1) > 7 ) { 
			/* everything longer than 7 chars must either be KDE konsole, or come 
			 * from more than one terminal */
			if ( strncmp(buf1, "\033[>0;115;0c", sizeof(buf1)) != 0) 
				return (1); 
		}
	}
	/* catch further replies... but try to avoid wasting too much time */
	readreply(fd, buf2, 0, 500);
	readreply(fd, buf3, 0, 200);
	readreply(fd, buf4, 0, 100);

	/* by allowing up to 2 characters in each buffer, we try to be somewhat robust
	 * against some user injecting characters because he is already typing */
	if ( (*buf2 && (strlen(buf2)>ALLOWED)) ||
	     (*buf3 && (strlen(buf3)>ALLOWED)) ||
	     (*buf4 && (strlen(buf4)>ALLOWED)) ) { 
		return (1); 
	}

	/* save the valid reply for later use */
	strncpy(da, buf1, strlen(buf1));

	return(0);
}

/*****************************************************************************
	findcompatible
*****************************************************************************/
int findcompatible(char *buf, char *str)
{
	int i=0, j;

	while (primaryda[i].reply) {
		if (strncmp(primaryda[i].reply, buf, strlen(primaryda[i].reply)) == 0) {
			j = 0;
			while (termname[j].termid <= unknown) {
				if (primaryda[i].termid == termname[j].termid) 
					break;
				j++;
			}
			strncpy(str, termname[j].terminfoname, 
				sizeof(termname[j].terminfoname));
			if (Debug) {
				settermmode(TOGGLE);
				puts(str);
				settermmode(TOGGLE);
			}
			return (termname[j].termid);
		}
		i++;
	}
	return (unknown);
}

/*****************************************************************************
	resolve_terminfo
*****************************************************************************/
int resolve_terminfo(int foundterm, char *str)
{
	int i=0;

	while (termname[i].termid <= unknown) {
		if (termname[i].termid == foundterm) 
			break;
		i++;
	}
	strncpy(str, termname[i].terminfoname, 
		sizeof(termname[i].terminfoname));
	if (!Quiet && Debug) {
		settermmode(TOGGLE);
		puts(str);
		settermmode(TOGGLE);
	}
	return (termname[i].termid);
}

/*****************************************************************************
	sendenq
*****************************************************************************/
int sendenq(int fd, char *buf)
{
	termsend(fd, ANSWERBACK, "asking for ANSWERBACK string (ENQ): "); 
	readreply(fd, buf, 1, 0);
	return (0);
}

/*****************************************************************************
	sendescz
*****************************************************************************/
int sendescz(int fd, char *buf)
{
	termsend(fd, OLD_TERMID, "asking with obsolete ESC Z: "); 
	readreply(fd, buf, 1, 0);
	return (0);
}

/*****************************************************************************
	sendsecda
*****************************************************************************/
int sendsecda(int fd, char *buf)
{
	termsend(fd, SECONDARY_DA, "asking for secondary DA: "); 
	readreply(fd, buf, 1, 0);
	return (0);
}

/*****************************************************************************
	querytitle
*****************************************************************************/
int querytitle(int fd, char *buf)
{
	termsend(fd, TITLE, "asking for window title: "); 
	readreply(fd, buf, 0, 500);
	return (0);
}

/*****************************************************************************
	queryscreensize
*****************************************************************************/
int queryscreensize(int fd, int *cols, int *lines, char *text)
{
	char	buf [BUFSIZE+1];

	if (Debug) {
		settermmode(TOGGLE);
		printf("%s", text);
		settermmode(TOGGLE);
	}

	termsend(fd, SAVE_CURSOR "\033[r" POSITION_CURSOR REPORT_CURSOR, "\033[100DQuerying screen size");
	readreply(fd, buf, 1, 0);
	if (!Debug) termsend(fd, RESTORE_CURSOR, "");
	if (*buf) {
		if (sscanf(buf, "%*c %*c %d ; %d R", lines, cols)) {
			return (1);
		}
	}
	return (0);
}


/*****************************************************************************
	readreply
*****************************************************************************/
int readreply(int fd, char *str, int s, int ms)
{
	int 	n = 0;
	int     r;                              /* return value of select() */
	fd_set  rset;                           /* file descriptor set for select() */
	struct timeval tv;                      /* timeout for select() */

	FD_ZERO(&rset);
	FD_SET(fd, &rset); // sets the bit for this descriptor in the read set
	tv.tv_sec = s; tv.tv_usec = ms*1000; 
	r = select(fd+1, &rset, NULL, NULL, &tv);

	if (FD_ISSET(fd, &rset)) 
		n = read(fd, str, BUFSIZE);

	str[n] = '\0';

	if (n < 0)
		return(-1);

	strascii(str);

	if (Debug) debug_show(str);
	//tcflush(0,TCIFLUSH);

	return (0);

}	

/*****************************************************************************
	debug_show
*****************************************************************************/
int debug_show(char *str)
{
	int u = 0;

	settermmode(TOGGLE);
	printf("%d bytes: ", strlen(str));
	while (str[u]) printf("%02X ",str[u++]);
	putchar(' ');
	u = 0;
	while (str[u]) {
		printf("%c", ( isprint(str[u]) && (!iscntrl(str[u])) ) ? str[u] : '.');
		u++;
	}
	putchar('\n');
	settermmode(TOGGLE);

	return(0);
}

/*****************************************************************************
	Strascii
*****************************************************************************/
void strascii(register unsigned char *p)	/* (r/w) the string to convert */
{

	while (*p) {
		*p &= 0x7F;
		p++;
	}
		
	return;
}	


/*****************************************************************************
	settermmode
*****************************************************************************/
void settermmode(int mode)
{
	static struct termio
		oldmode;
	struct termio
		newmode;
	static int currentmode = 1;

	if (mode == TOGGLE) {
		mode = abs(currentmode-1);
	}

	if (mode) {

		/* before restoring: */
		/* windows clients tend to send CR and LF, and the OS/400 telnet server does 
		 * not translate them properly to a telnet virtual linefeed.
		 * Thus, we want to ignore CR */
		if ( Igncr ) oldmode.c_iflag |= IGNCR;
		else oldmode.c_iflag &= ~IGNCR;

		ioctl(0,TCSETA,&oldmode);		  /* reset original mode */
		currentmode = COOKED;
		//printf("-"); 
	}
	else {
		//printf("+"); 
		ioctl(0,TCGETA,&oldmode);			/* save mode */
		memcpy(&newmode,&oldmode,sizeof(struct termio));
		newmode.c_iflag |= (BRKINT|IGNPAR);
		newmode.c_iflag &= ~(IGNBRK|INLCR|IGNCR|ICRNL|IUCLC|IXON);
		newmode.c_lflag &= ~(ISIG|ICANON|ECHO);
		newmode.c_oflag &= ~(ONLCR|OCRNL|ONLRET);
		newmode.c_cc[4] = 1;
		newmode.c_cc[5] = 1;
		ioctl(0,TCSETA,&newmode);		  /* set raw mode */
		currentmode = RAW;
	}

	return;

}


/*****************************************************************************
	warn_mutiple
*****************************************************************************/
void warn_mutiple(void)
{
	settermmode(TOGGLE);
	fprintf(stderr, "\n\
\n\
###############################################################\n\
More than one terminal on the virtual console detected.\n\
Only one telnet client should be connected at a time, otherwise\n\
terminal type/size cannot be determined unambiguously.\n\
\n\
Recommendation: detach other terminals.\n\
###############################################################\n");
	settermmode(TOGGLE);

}


