#! /bin/bash
#
# Johannes Meixner <jsmeix@suse.de>, 2010

#set -x

# Make sure to have a clean environment:
export PATH="/sbin:/usr/sbin:/usr/bin:/bin"
export LC_ALL="POSIX"
export LANG="POSIX"
umask 022
# Disable bash file name globbing to avoid side-effects
# with characters like '*' and '?' (see bnc #575056):
set -f

INPUT="drivers.xml"
OUTPUT="drivers_to_be_considered"
cat /dev/null >$OUTPUT
DIGEST="drivers.digest"
GS_DRIVERS="drivers.ghostscript.available"

# Function to output one entry:
Output()
{ echo "$TYPE $NAME" >>$OUTPUT
}

# Function to test and output one entry:
TestAndOutput()
{ if [ -z "$NAME" ]
  then echo "Skipping entry (empty name) of type '$TYPE' from url '$URL'" 1>&2
       return 1
  fi
  if [ -n "$NONFREE" ]
  then if [ "$NAME" = "splix" ]
       then echo "Keeping $NAME (does not implement JBIG but marked as '$NONFREE') of type '$TYPE' from url '$URL'" 1>&2
       else echo "Skipping $NAME ($NONFREE) of type '$TYPE' from url '$URL'" 1>&2
            return 1
       fi
  fi
  for D in foo2hbpl2 foo2hiperc foo2hp foo2kyo foo2lava foo2oak foo2oak-z1 foo2qpdl foo2slx foo2xqx foo2zjs foo2zjs-z1 foo2zjs-z2 foo2zjs-z3 rastertokmXXXXdl
  do if [ "$NAME" = "$D" ]
     then echo "Skipping $NAME (implements JBIG but marked as 'nopatents') of type '$TYPE' from url '$URL'" 1>&2
          return 1
     fi
  done
  if [ -n "$OBSOLETE" ]
  then for D in gutenprint hplip lxm3200 oki4w pcl3 stcolor $( cat $GS_DRIVERS )
       do if [ "$OBSOLETE" = "replace=$D" ]
          then echo "Skipping $NAME (obsoleted by $D) of type '$TYPE' from url '$URL'" 1>&2
               return 1
          fi
       done
  fi
  if [ "$NAME" = "omni" ]
  then echo "Skipping $NAME (deprecated) of type '$TYPE' from url '$URL'" 1>&2
       return 1
  fi
  for D in gutenprint hplip splix m2300w m2400w
  do if [ "$NAME" = "$D" ]
     then echo "Skipping $NAME (has distinct RPM with PPDs) of type '$TYPE' from url '$URL'" 1>&2
          return 1
     fi
  done
  for D in capt dplix epson-escpr iwlo iwlq lbp660 lbp800 lxx74 lz11-V2 pbm2ppa pegg ptouch slap
  do if [ "$NAME" = "$D" ]
     then echo "Skipping $NAME (no PPDs are or can be generated) of type '$TYPE' from url '$URL'" 1>&2
          return 1
     fi
  done
  case "$TYPE" in
       cups) Output
             return ;;
       filter) Output
               return ;;
       ghostscript) if grep -q "^$NAME\$" $GS_DRIVERS
                    then Output
                         return
                    fi
                    for M in Gestetner InfoPrint Infotec Lanier NRG Ricoh Savin
                    do for D in pxlcolor pxlmono 
                       do if [ "$NAME" = "$D-$M" ]
                          then Output
                               return
                           fi
                       done
                    done
                    echo -n "Skipping $NAME (not listed in 'gs -h') of type $TYPE" 1>&2
                    [ -n "$URL" ] && echo " from $URL" 1>&2 || echo 1>&2
                    return 1 ;;
       ijs) if [ "$URL" = "http://hplipopensource.com/" ]
            then Output
                 return
            fi
            if [ -n "$URL" ]
            then echo "Skipping $NAME (unexpected URL) of type $TYPE from $URL" 1>&2
            else echo "Skipping $NAME (empty URL) of type $TYPE" 1>&2
            fi
            return 1 ;;
       postscript) Output
                   return ;;
       uniprint) FILE_PATTERN="$( echo "$NAME" | tr -s 'XYZ' '*' )"
                 # Re-enable bash file name globbing in a sub-shell only for this particular ls command:
                 if ( set +f ; ls /usr/share/ghostscript/*/lib/$FILE_PATTERN &>/dev/null )
                 then Output
                      return
                 fi
                 echo -n "Skipping $NAME (no /usr/share/ghostscript/*/lib/$FILE_PATTERN) of type $TYPE" 1>&2
                 [ -n "$URL" ] && echo " from $URL" 1>&2 || echo 1>&2
                 return 1 ;;
       *) if [ -n "$TYPE" ]
          then echo "Skipping $NAME (unexpected type $TYPE)" 1>&2
          else echo "Skipping $NAME (empty type)" 1>&2
          fi
          return 1 ;;
  esac
}

# Make a list of Ghostscript's built-in drivers.
gs -h \
 | sed -n -e '/^Available devices:/,/^Search path:/p' \
 | egrep -v '^Available devices:|^Search path:' \
 | tr -s '[:space:]' '\n' \
 >$GS_DRIVERS

# Process the input:
# The useless use of cat is intentional to make it more obvious
# what the input, the processing, and the output is.
# The leading spaces in the 'egrep' search expression
# avoid to get also such XML tags from nested lower levels
# (same XML tags like <url> are used for same kind of content
# but with different meaning/semantics on different levels).
cat $INPUT \
 | sed -e '/<execution>/{N;s/<execution>\n/<execution>/}' \
 | egrep '^    <name>|^    <url>|^    <obsolete replace=|freesoftware />|patents />|^    <execution>' \
 | tr -d '"' \
 | sed -e 's/<\/name>//' \
       -e 's/<\/url>//' \
       -e 's/[[:space:]][[:space:]]*/ /g' \
       -e 's/^[[:space:]]*//' \
       -e 's/[[:space:]]*$//' \
       -e 's/<execution> </<execution>/' \
       -e 's/ \/>//' \
       -e 's/^<//' \
       -e 's/>/ /' \
 | cat >$DIGEST
# The entries in the drivers digest have a form like
# ----------------------------------------------------
# name ljet4
# url http://www.ghostscript.com/
# freesoftware
# nopatents
# execution ghostscript
# ----------------------------------------------------
# or with an optional obsolete entry like
# ----------------------------------------------------
# name bjc6000X1.upp
# url http://www.ph.tum.de/~thaller/bjc6000/
# obsolete replace=gutenprint
# freesoftware
# nopatents
# execution uniprint
# ----------------------------------------------------
# There exists 'freesoftware' versus 'nonfreesoftware'
# and 'nopatents' versus 'patents' and one of
# ----------------------------------------------------
# execution cups
# execution filter
# execution ghostscript
# execution ijs
# execution postscript
# execution uniprint
# ----------------------------------------------------
# and optionally one of
# ----------------------------------------------------
# obsolete replace=cZ11somsom
# obsolete replace=foo2zjs
# obsolete replace=gutenprint
# obsolete replace=hplip
# obsolete replace=lxm3200
# obsolete replace=lxm3200-tweaked
# obsolete replace=lz11
# obsolete replace=oki4w
# obsolete replace=pcl3
# obsolete replace=pnm2ppa
# obsolete replace=ppmtomd
# obsolete replace=pxldpl
# obsolete replace=stcolor
# ----------------------------------------------------
exec <$DIGEST
NAME=""
URL=""
OBSOLETE=""
NONFREE=""
TYPE=""
while read KEY VALUE
do case "$KEY" in
        name) NAME="$VALUE"
              URL=""
              OBSOLETE=""
              NONFREE=""
              TYPE="" ;;
        url) URL="$VALUE" ;;
        obsolete) OBSOLETE="$VALUE" ;;
        freesoftware) ;;
        nonfreesoftware) [ -z "$NONFREE" ] && NONFREE="nonfreesoftware" || NONFREE="$NONFREE nonfreesoftware" ;;
        nopatents) ;;
        patents) [ -z "$NONFREE" ] && NONFREE="patents" || NONFREE="$NONFREE patents" ;;
        execution) TYPE="$VALUE"
                   TestAndOutput ;;
        *) echo "Ignoring key $KEY" 1>&2 ;;
   esac
done

# Sort the list of drivers to be considered:
mv -f $OUTPUT $OUTPUT.unsorted
sort -b -d -f -o $OUTPUT $OUTPUT.unsorted && rm $OUTPUT.unsorted

