/*
 * livepatch_bsc1200266
 *
 * Fix for CVE-2022-1972, bsc#1200266
 *
 *  Upstream commit:
 *  fecf31ee395b ("netfilter: nf_tables: sanitize nft_set_desc_concat_parse()")
 *
 *  SLE12-SP3 commit:
 *  not affected
 *
 *  SLE12-SP4, SLE12-SP5, SLE15 and SLE15-SP1 commit:
 *  not affected
 *
 *  SLE15-SP2 commit:
 *  not affected
 *
 *  SLE15-SP3 commit:
 *  323e1668e3f692457813512021a3f9869e24d6e7
 *
 *  SLE15-SP4 commit:
 *  fb312f53a1abadd33c8bfb02775be220149e3074
 *
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if !IS_MODULE(CONFIG_NF_TABLES)
#error "Live patch supports only CONFIG_NF_TABLES=m"
#endif

/* klp-ccp: from net/netfilter/nf_tables_api.c */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/skbuff.h>
#include <linux/netlink.h>
#include <linux/vmalloc.h>
#include <linux/rhashtable.h>
#include <linux/audit.h>
#include <linux/netfilter.h>
#include <linux/netfilter/nfnetlink.h>
#include <linux/netfilter/nf_tables.h>
#include <net/netfilter/nf_flow_table.h>
#include <net/netfilter/nf_tables_core.h>
#include <net/netfilter/nf_tables.h>
#include <net/net_namespace.h>

static const struct nla_policy (*klpe_nft_set_desc_policy)[NFTA_SET_DESC_MAX + 1];

static const struct nla_policy (*klpe_nft_concat_policy)[NFTA_SET_FIELD_MAX + 1];

static int klpp_nft_set_desc_concat_parse(const struct nlattr *attr,
				     struct nft_set_desc *desc)
{
	struct nlattr *tb[NFTA_SET_FIELD_MAX + 1];
	u32 len;
	int err;

	/*
	 * Fix CVE-2022-1972
	 *  +2 lines
	 */
	if (desc->field_count >= ARRAY_SIZE(desc->field_len))
		return -E2BIG;

	err = nla_parse_nested_deprecated(tb, NFTA_SET_FIELD_MAX, attr,
					  (*klpe_nft_concat_policy), NULL);
	if (err < 0)
		return err;

	if (!tb[NFTA_SET_FIELD_LEN])
		return -EINVAL;

	len = ntohl(nla_get_be32(tb[NFTA_SET_FIELD_LEN]));

	/*
	 * Fix CVE-2022-1972
	 *  -2 lines, +2 lines
	 */
	if (!len || len > U8_MAX)
		return -EINVAL;

	desc->field_len[desc->field_count++] = len;

	return 0;
}

static int klpp_nft_set_desc_concat(struct nft_set_desc *desc,
			       const struct nlattr *nla)
{
	struct nlattr *attr;
	/*
	 * Fix CVE-2022-1972
	 *  -1 line, +2 lines
	 */
	u32 num_regs = 0;
	int rem, err, i;

	nla_for_each_nested(attr, nla, rem) {
		if (nla_type(attr) != NFTA_LIST_ELEM)
			return -EINVAL;

		err = klpp_nft_set_desc_concat_parse(attr, desc);
		if (err < 0)
			return err;
	}

	/*
	 * Fix CVE-2022-1972
	 *  +5 lines
	 */
	for (i = 0; i < desc->field_count; i++)
		num_regs += DIV_ROUND_UP(desc->field_len[i], sizeof(u32));

	if (num_regs > NFT_REG32_COUNT)
		return -E2BIG;

	return 0;
}

int klpp_nf_tables_set_desc_parse(struct nft_set_desc *desc,
				    const struct nlattr *nla)
{
	struct nlattr *da[NFTA_SET_DESC_MAX + 1];
	int err;

	err = nla_parse_nested_deprecated(da, NFTA_SET_DESC_MAX, nla,
					  (*klpe_nft_set_desc_policy), NULL);
	if (err < 0)
		return err;

	if (da[NFTA_SET_DESC_SIZE] != NULL)
		desc->size = ntohl(nla_get_be32(da[NFTA_SET_DESC_SIZE]));
	if (da[NFTA_SET_DESC_CONCAT])
		err = klpp_nft_set_desc_concat(desc, da[NFTA_SET_DESC_CONCAT]);

	return err;
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1200266.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "nf_tables"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "nft_concat_policy", (void *)&klpe_nft_concat_policy, "nf_tables" },
	{ "nft_set_desc_policy", (void *)&klpe_nft_set_desc_policy,
	  "nf_tables" },
};

static int livepatch_bsc1200266_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1200266_module_nb = {
	.notifier_call = livepatch_bsc1200266_module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1200266_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&livepatch_bsc1200266_module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LIVEPATCHED_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						  ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&livepatch_bsc1200266_module_nb);

	module_put(mod);
	return ret;
}

void livepatch_bsc1200266_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1200266_module_nb);
}
