/*
 * bsc1202087_fbcon
 *
 * Fix for CVE-2021-33655, bsc#1202087
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Marcos Paulo de Souza <mpdesouza@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_FRAMEBUFFER_CONSOLE)

/* klp-ccp: from drivers/video/fbdev/core/fbcon.c */
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/kernel.h>

#include <linux/console.h>

/* klp-ccp: from drivers/video/fbdev/core/fbcon.c */
#include <linux/string.h>
#include <linux/kd.h>
#include <linux/slab.h>
#include <linux/fb.h>
#include <linux/vt_kern.h>

/* klp-ccp: from include/linux/font.h */
#define REFCOUNT(fd)	(((int *)(fd))[-1])
#define FNTSIZE(fd)	(((int *)(fd))[-2])
#define FNTCHARCNT(fd)	(((int *)(fd))[-3])
#define FNTSUM(fd)	(((int *)(fd))[-4])

#define FONT_EXTRA_WORDS 4

/* klp-ccp: from drivers/video/fbdev/core/fbcon.c */
#include <linux/smp.h>
#include <linux/init.h>

/* klp-ccp: from drivers/video/fbdev/core/fbcon.c */
#include <linux/crc32.h> /* For counting font checksums */
#include <linux/uaccess.h>
#include <asm/irq.h>
/* klp-ccp: from drivers/video/fbdev/core/fbcon.h */
#include <linux/types.h>
#include <linux/vt_buffer.h>
#include <linux/vt_kern.h>
#include <asm/io.h>

struct fbcon_display {
    /* Filled in by the low-level console driver */
    const u_char *fontdata;
    int userfont;                   /* != 0 if fontdata kmalloc()ed */
    u_short scrollmode;             /* Scroll Method */
    u_short inverse;                /* != 0 text black on white as default */
    short yscroll;                  /* Hardware scrolling */
    int vrows;                      /* number of virtual rows */
    int cursor_shape;
    int con_rotate;
    u32 xres_virtual;
    u32 yres_virtual;
    u32 height;
    u32 width;
    u32 bits_per_pixel;
    u32 grayscale;
    u32 nonstd;
    u32 accel_flags;
    u32 rotate;
    struct fb_bitfield red;
    struct fb_bitfield green;
    struct fb_bitfield blue;
    struct fb_bitfield transp;
    const struct fb_videomode *mode;
};

/* klp-ccp: from drivers/video/fbdev/core/fbcon.c */
static struct fbcon_display (*klpe_fb_display)[MAX_NR_CONSOLES];

static signed char (*klpe_con2fb_map)[MAX_NR_CONSOLES];

static int (*klpe_first_fb_vc);
static int (*klpe_last_fb_vc);

#ifdef CONFIG_FB_TILEBLITTING

static int fbcon_invalid_charcount(struct fb_info *info, unsigned charcount)
{
	int err = 0;

	if (info->flags & FBINFO_MISC_TILEBLITTING &&
	    info->tileops->fb_get_tilemax(info) < charcount)
		err = 1;

	return err;
}
#else
static int fbcon_invalid_charcount(struct fb_info *info, unsigned charcount)
{
	return 0;
}
#endif /* CONFIG_MISC_TILEBLITTING */

#define PITCH(w) (((w) + 7) >> 3)
#define CALC_FONTSZ(h, p, c) ((h) * (p) * (c)) /* size = height * pitch * charcount */

static int (*klpe_fbcon_do_set_font)(struct vc_data *vc, int w, int h,
			     const u8 * data, int userfont);

/* klp-ccp: from drivers/video/fbdev/core/fbcon.h */
#define FBCON_SWAP(i,r,v) ({ \
        typeof(r) _r = (r);  \
        typeof(v) _v = (v);  \
        (void) (&_r == &_v); \
        (i == FB_ROTATE_UR || i == FB_ROTATE_UD) ? _r : _v; })

#if defined(CONFIG_X86_64) || defined(CONFIG_PPC64)
#define klpr_registered_fb registered_fb
#elif defined(CONFIG_S390)
static struct fb_info *(*klpe_registered_fb)[FB_MAX];

#define klpr_registered_fb (*klpe_registered_fb)
#else
#error "klp-ccp: non-taken branch"
#endif

int klpp_fbcon_set_font(struct vc_data *vc, struct console_font *font,
			  unsigned int flags)
{
	struct fb_info *info = klpr_registered_fb[(*klpe_con2fb_map)[vc->vc_num]];
	unsigned charcount = font->charcount;
	int w = font->width;
	int h = font->height;
	int size;
	int i, csum;
	u8 *new_data, *data = font->data;
	int pitch = PITCH(font->width);

	/* Is there a reason why fbconsole couldn't handle any charcount >256?
	 * If not this check should be changed to charcount < 256 */
	if (charcount != 256 && charcount != 512)
		return -EINVAL;

	/* font bigger than screen resolution ? */
	if (w > FBCON_SWAP(info->var.rotate, info->var.xres, info->var.yres) ||
	    h > FBCON_SWAP(info->var.rotate, info->var.yres, info->var.xres))
		return -EINVAL;

	/* Make sure drawing engine can handle the font */
	if (!(info->pixmap.blit_x & (1 << (font->width - 1))) ||
	    !(info->pixmap.blit_y & (1 << (font->height - 1))))
		return -EINVAL;

	/* Make sure driver can handle the font length */
	if (fbcon_invalid_charcount(info, charcount))
		return -EINVAL;

	size = CALC_FONTSZ(h, pitch, charcount);

	new_data = kmalloc(FONT_EXTRA_WORDS * sizeof(int) + size, GFP_USER);

	if (!new_data)
		return -ENOMEM;

	new_data += FONT_EXTRA_WORDS * sizeof(int);
	FNTSIZE(new_data) = size;
	FNTCHARCNT(new_data) = charcount;
	REFCOUNT(new_data) = 0;	/* usage counter */
	for (i=0; i< charcount; i++) {
		memcpy(new_data + i*h*pitch, data +  i*32*pitch, h*pitch);
	}

	/* Since linux has a nice crc32 function use it for counting font
	 * checksums. */
	csum = crc32(0, new_data, size);

	FNTSUM(new_data) = csum;
	/* Check if the same font is on some other console already */
	for (i = (*klpe_first_fb_vc); i <= (*klpe_last_fb_vc); i++) {
		struct vc_data *tmp = vc_cons[i].d;

		if ((*klpe_fb_display)[i].userfont &&
		    (*klpe_fb_display)[i].fontdata &&
		    FNTSUM((*klpe_fb_display)[i].fontdata) == csum &&
		    FNTSIZE((*klpe_fb_display)[i].fontdata) == size &&
		    tmp->vc_font.width == w &&
		    !memcmp((*klpe_fb_display)[i].fontdata, new_data, size)) {
			kfree(new_data - FONT_EXTRA_WORDS * sizeof(int));
			new_data = (u8 *)(*klpe_fb_display)[i].fontdata;
			break;
		}
	}
	return (*klpe_fbcon_do_set_font)(vc, font->width, font->height, new_data, 1);
}

/* let fbcon check if it supports a new screen resolution */
int klpp_fbcon_modechange_possible(struct fb_info *info, struct fb_var_screeninfo *var)
{
	struct fbcon_ops *ops = info->fbcon_par;
	struct vc_data *vc;
	unsigned int i;

	WARN_CONSOLE_UNLOCKED();

	if (!ops)
		return 0;

	/* prevent setting a screen size which is smaller than font size */
	for (i = (*klpe_first_fb_vc); i <= (*klpe_last_fb_vc); i++) {
		vc = vc_cons[i].d;
		if (!vc || vc->vc_mode != KD_TEXT ||
			   klpr_registered_fb[(*klpe_con2fb_map)[i]] != info)
			continue;

		if (vc->vc_font.width  > FBCON_SWAP(var->rotate, var->xres, var->yres) ||
		    vc->vc_font.height > FBCON_SWAP(var->rotate, var->yres, var->xres))
			return -EINVAL;
	}

	return 0;
}



#include <linux/kernel.h>
#include "livepatch_bsc1202087.h"
#include "../kallsyms_relocs.h"

#if defined(CONFIG_S390)
static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "con2fb_map", (void *)&klpe_con2fb_map, "fb" },
	{ "fb_display", (void *)&klpe_fb_display, "fb" },
	{ "fbcon_do_set_font", (void *)&klpe_fbcon_do_set_font, "fb" },
	{ "first_fb_vc", (void *)&klpe_first_fb_vc, "fb" },
	{ "last_fb_vc", (void *)&klpe_last_fb_vc, "fb" },
	{ "registered_fb", (void *)&klpe_registered_fb, "fb" },
};
#elif defined(CONFIG_X86_64) || defined(CONFIG_PPC64)
static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "con2fb_map", (void *)&klpe_con2fb_map },
	{ "fb_display", (void *)&klpe_fb_display },
	{ "fbcon_do_set_font", (void *)&klpe_fbcon_do_set_font },
	{ "first_fb_vc", (void *)&klpe_first_fb_vc },
	{ "last_fb_vc", (void *)&klpe_last_fb_vc },
};
#else
#error "klp-ccp: non-taken branch"
#endif

#if defined(CONFIG_S390)
#define LP_MODULE "fb"

#include <linux/module.h>

static int bsc1202087_module_notify(struct notifier_block *nb,
					unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;

	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = bsc1202087_module_notify,
	.priority = INT_MIN+1,
};

int bsc1202087_fbcon_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LP_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void bsc1202087_fbcon_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}
#elif defined(CONFIG_X86_64) || defined(CONFIG_PPC64)
int bsc1202087_fbcon_init(void)
{
	return __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

void bsc1202087_fbcon_cleanup(void)
{
}
#else
#error "klp-ccp: non-taken branch"
#endif

#endif /* IS_ENABLED(CONFIG_FRAMEBUFFER_CONSOLE) */
