/*
 * bsc1185901_sctp_socket
 *
 * Fix for CVE-2021-23133, bsc#1185901 (net/sctp/socket.c part)
 *
 *
 *  Copyright (c) 2021 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if !IS_MODULE(CONFIG_IP_SCTP)
#error "Live patch supports only CONFIG_IP_SCTP=m"
#endif

#include "bsc1185901_common.h"

/* klp-ccp: from net/sctp/socket.c */
#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <crypto/hash.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/wait.h>
#include <linux/time.h>
#include <linux/sched/signal.h>
#include <linux/ip.h>
#include <linux/capability.h>
#include <linux/fcntl.h>
#include <linux/poll.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/file.h>
#include <linux/compat.h>
#include <linux/rhashtable.h>
#include <net/ip.h>
#include <net/route.h>
#include <net/ipv6.h>
#include <linux/socket.h> /* for sa_family_t */
#include <linux/export.h>
#include <net/sock.h>
#include <net/sctp/sctp.h>

static void (*klpe_sctp_destroy_sock)(struct sock *sk);

/* New. */
bool klpp_is_sctp_sock(struct sock *sk)
{
	void (* const destroy)(struct sock *) = sk->sk_prot->destroy;

	return destroy && destroy == READ_ONCE(klpe_sctp_destroy_sock);
}

/* New. */
void klpp_sctp_disable_asconf(struct sock *sk)
{
	struct net *net = sock_net(sk);
	struct sctp_sock *sp = sctp_sk(sk);

	spin_lock_bh(&net->sctp.addr_wq_lock);
	if (sp->do_auto_asconf) {
		sp->do_auto_asconf = 0;
		list_del(&sp->auto_asconf_list);
	}
	spin_unlock_bh(&net->sctp.addr_wq_lock);
}

static int sctp_wait_for_accept(struct sock *sk, long timeo);

static int (*klpe_sctp_sock_migrate)(struct sock *oldsk, struct sock *newsk,
			     struct sctp_association *assoc,
			     enum sctp_socket_type type);

struct sock *klpp_sctp_accept(struct sock *sk, int flags, int *err, bool kern)
{
	struct sctp_sock *sp;
	struct sctp_endpoint *ep;
	struct sock *newsk = NULL;
	struct sctp_association *asoc;
	long timeo;
	int error = 0;

	lock_sock(sk);

	sp = sctp_sk(sk);
	ep = sp->ep;

	if (!sctp_style(sk, TCP)) {
		error = -EOPNOTSUPP;
		goto out;
	}

	if (!sctp_sstate(sk, LISTENING)) {
		error = -EINVAL;
		goto out;
	}

	timeo = sock_rcvtimeo(sk, flags & O_NONBLOCK);

	error = sctp_wait_for_accept(sk, timeo);
	if (error)
		goto out;

	/* We treat the list of associations on the endpoint as the accept
	 * queue and pick the first association on the list.
	 */
	asoc = list_entry(ep->asocs.next, struct sctp_association, asocs);

	newsk = sp->pf->create_accept_sk(sk, asoc, kern);
	if (!newsk) {
		error = -ENOMEM;
		goto out;
	}

	/* Populate the fields of the newsk from the oldsk and migrate the
	 * asoc to the newsk.
	 */
	error = (*klpe_sctp_sock_migrate)(sk, newsk, asoc, SCTP_SOCKET_TCP);
	/*
	 * Fix CVE-2021-23133
	 *  -4 lines, +2 lines
	 */
	if (error)
		goto out_migrate_err;

out:
	release_sock(sk);
	*err = error;
	return newsk;

	/*
	 * Fix CVE-2021-23133
	 *  +10 lines
	 */
out_migrate_err:
	/*
	 * sctp_disable_asconf() grabs net->sctp.addr_wq_lock. Unlock
	 * the socket first in order to avoid deadlocks.
	 */
	release_sock(sk);
	*err = error;
	klpp_sctp_disable_asconf(newsk);
	sk_common_release(newsk);
	return NULL;
}

static int sctp_wait_for_accept(struct sock *sk, long timeo)
{
	struct sctp_endpoint *ep;
	int err = 0;
	DEFINE_WAIT(wait);

	ep = sctp_sk(sk)->ep;


	for (;;) {
		prepare_to_wait_exclusive(sk_sleep(sk), &wait,
					  TASK_INTERRUPTIBLE);

		if (list_empty(&ep->asocs)) {
			release_sock(sk);
			timeo = schedule_timeout(timeo);
			lock_sock(sk);
		}

		err = -EINVAL;
		if (!sctp_sstate(sk, LISTENING))
			break;

		err = 0;
		if (!list_empty(&ep->asocs))
			break;

		err = sock_intr_errno(timeo);
		if (signal_pending(current))
			break;

		err = -EAGAIN;
		if (!timeo)
			break;
	}

	finish_wait(sk_sleep(sk), &wait);

	return err;
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1185901.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "sctp"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "sctp_sock_migrate", (void *)&klpe_sctp_sock_migrate, "sctp" },
	{ "sctp_destroy_sock", (void *)&klpe_sctp_destroy_sock, "sctp" },
};

static int livepatch_bsc1185901_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action == MODULE_STATE_GOING &&
	    !strcmp(mod->name, LIVEPATCHED_MODULE)) {
		/*
		 * Clear out klpe_sctp_destroy_sock still used for
		 * comparisons in livepatched inet_create()
		 * resp. inet6_create().
		 */
		WRITE_ONCE(klpe_sctp_destroy_sock, NULL);
	}

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1185901_module_nb = {
	.notifier_call = livepatch_bsc1185901_module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1185901_sctp_socket_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LIVEPATCHED_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&livepatch_bsc1185901_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void livepatch_bsc1185901_sctp_socket_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1185901_module_nb);
}
