#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdatomic.h>

#ifndef _VERSION
# error "_VERSION macro not defined."
#endif

#define STRINGFY(s) #s
#define STRINGFY_VALUE(s) STRINGFY(s)

#define LP_SUFFIX "-lp-" STRINGFY_VALUE(_VERSION)
#define ARR_LEN(v) (sizeof(v)/sizeof(*(v)))

/* Avoid ugly casting on code.  */
static void *add_long_to_ptr(void *ptr, long val)
{
  return (void *) ((char *)ptr + val);
}

#define INSN_ENDBR64 0xf3, 0x0f, 0x1e, 0xfa

/* Skip the ULP prologue of a function, so that when 'func' is called it runs
   its original code.  This should save us from including libc headers and copy
   & pasting code from libc itself.  */
static void *skip_ulp_redirect_insns(void *func)
{
  /* Check if function contain the jump trampoline.  */
  const unsigned char *as_bytes = (const unsigned char *) func;
  const unsigned char insn_endbr64[] = {INSN_ENDBR64};
  int i, bias = 0;

  for (i = 0; i < sizeof(insn_endbr64); i++) {
    if (as_bytes[i] != insn_endbr64[i])
      break;
  }

  if (i == sizeof(insn_endbr64)) {
    /* Comparison successful.  */
    as_bytes += sizeof(insn_endbr64);
    bias += sizeof(insn_endbr64);
  }

  if (as_bytes[0] != 0xEB) {
    /* JMP rel8 opcode not found. Function definitely not livepatched.
       Check if it is livepatchable*/

    if (as_bytes[1] == 0x90) {
      if (as_bytes [0] == 0x90 || as_bytes[0] == 0x66) {
        goto add;
      }
    }

    /* Function not livepatchable.  */
    return func;
  }

  /* On x86_64, the JMP insns used for redirecting the old function
     into the new one takes 2 bytes.  So add 2 bytes to skip it.  */
add:
  return add_long_to_ptr(func, 2 + bias);
}

const char *OpenSSL_version(int);

# define OPENSSL_VERSION          0

static const char *Build_OpenSSL_LP_Version_String(const char *str)
{
  char ver[32];
  int pos = 0;

  if (str == NULL)
    return NULL;

  int n = sscanf(str, "OpenSSL %31s%n", (char *)&ver, &pos);
  if (n == 1) {
    size_t size = strlen(str) + ARR_LEN(LP_SUFFIX) + 1;
    char *new_ver_str = (char *) calloc(size, sizeof(char));
    if (new_ver_str == NULL) {
      return str;
    }
    sprintf(new_ver_str, "OpenSSL %s", ver);
    strcat(new_ver_str, LP_SUFFIX);
    strcat(new_ver_str, &str[pos]);

    return new_ver_str;
  }

  return str;
}

const char *OpenSSL_version_lp(int type)
{
  static volatile const char *new_ver_str;
  static const char *(*volatile old_OpenSSL_version)(int) = NULL;
  static atomic_int lock = 0;

  const int _0 = 0;

  /* Initialize variables.  */
  if (atomic_compare_exchange_strong(&lock, &_0, 1)) {
    old_OpenSSL_version = skip_ulp_redirect_insns(OpenSSL_version);
    const char *str = old_OpenSSL_version(OPENSSL_VERSION);
    new_ver_str = Build_OpenSSL_LP_Version_String(str);
    atomic_store(&lock, 2);
  }

  while (atomic_load(&lock) < 2)
    ; /* Wait for variable initialization to end.  */

  if (type == OPENSSL_VERSION) {
    return (const char *) new_ver_str;
  }

  return old_OpenSSL_version(type);
}
