/*
 * bsc1202087_fbmem
 *
 * Fix for CVE-2021-33655, bsc#1202087
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Marcos Paulo de Souza <mpdesouza@suse.com>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_FRAMEBUFFER_CONSOLE)

/* klp-ccp: from drivers/video/fbdev/core/fbmem.c */
#include <linux/compat.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/init.h>

/* klp-ccp: from drivers/video/fbdev/core/fbmem.c */
#include <linux/seq_file.h>
#include <linux/console.h>
#include <linux/kmod.h>
#include <linux/err.h>
#include <linux/device.h>
#include <linux/efi.h>
#include <linux/fb.h>

static void (*klpe_fb_delete_videomode)(const struct fb_videomode *mode,
				struct list_head *head);

static int (*klpe_fb_cmap_to_user)(const struct fb_cmap *from, struct fb_cmap_user *to);

static int (*klpe_fb_set_user_cmap)(struct fb_cmap_user *cmap, struct fb_info *fb_info);

/* klp-ccp: from drivers/video/fbdev/core/fbmem.c */
#include <linux/fbcon.h>

static int (*klpe_fbcon_mode_deleted)(struct fb_info *info,
		       struct fb_videomode *mode);

static void (*klpe_fbcon_get_requirement)(struct fb_info *info,
			   struct fb_blit_caps *caps);
static void (*klpe_fbcon_fb_blanked)(struct fb_info *info, int blank);

static int (*klpe_fbcon_set_con2fb_map_ioctl)(void __user *argp);
static int (*klpe_fbcon_get_con2fb_map_ioctl)(void __user *argp);

/* klp-ccp: from drivers/video/fbdev/core/fbmem.c */
#include <linux/mem_encrypt.h>

#if defined(CONFIG_X86_64) || defined(CONFIG_PPC64)
int fb_blank(struct fb_info *info, int blank);
int fb_pan_display(struct fb_info *info, struct fb_var_screeninfo *var);

#define klpr_fb_blank fb_blank
#define klpr_fb_set_cmap fb_set_cmap
#define klpr_fb_mode_is_equal fb_mode_is_equal
#define klpr_fb_var_to_videomode fb_var_to_videomode
#define klpr_fbcon_update_vcs fbcon_update_vcs
#define klpr_fb_add_videomode fb_add_videomode
#define klpr_fb_pan_display fb_pan_display
#elif defined(CONFIG_S390)
static int (*klpe_fb_blank)(struct fb_info *info, int blank);
static int (*klpe_fb_set_cmap)(struct fb_cmap *cmap, struct fb_info *info);
static int (*klpe_fb_mode_is_equal)(const struct fb_videomode *mode1,
				const struct fb_videomode *mode2);
static void (*klpe_fb_var_to_videomode)(struct fb_videomode *mode,
				const struct fb_var_screeninfo *var);
static void (*klpe_fbcon_update_vcs)(struct fb_info *info, bool all);
static int (*klpe_fb_add_videomode)(const struct fb_videomode *mode,
				struct list_head *head);
static int (*klpe_fb_pan_display)(struct fb_info *info,
				struct fb_var_screeninfo *var);

#define klpr_fb_blank (*klpe_fb_blank)
#define klpr_fb_set_cmap (*klpe_fb_set_cmap)
#define klpr_fb_mode_is_equal (*klpe_fb_mode_is_equal)
#define klpr_fb_var_to_videomode (*klpe_fb_var_to_videomode)
#define klpr_fbcon_update_vcs (*klpe_fbcon_update_vcs)
#define klpr_fb_add_videomode (*klpe_fb_add_videomode)
#define klpr_fb_pan_display (*klpe_fb_pan_display)
#else
#error "klp-ccp: non-taken branch"
#endif

static int klpr_fb_check_caps(struct fb_info *info, struct fb_var_screeninfo *var,
			 u32 activate)
{
	struct fb_blit_caps caps, fbcaps;
	int err = 0;

	memset(&caps, 0, sizeof(caps));
	memset(&fbcaps, 0, sizeof(fbcaps));
	caps.flags = (activate & FB_ACTIVATE_ALL) ? 1 : 0;
	(*klpe_fbcon_get_requirement)(info, &caps);
	info->fbops->fb_get_caps(info, &fbcaps, var);

	if (((fbcaps.x ^ caps.x) & caps.x) ||
	    ((fbcaps.y ^ caps.y) & caps.y) ||
	    (fbcaps.len < caps.len))
		err = -EINVAL;

	return err;
}

int
klpp_fb_set_var(struct fb_info *info, struct fb_var_screeninfo *var)
{
	int ret = 0;
	u32 activate;
	struct fb_var_screeninfo old_var;
	struct fb_videomode mode;
	struct fb_event event;
	u32 unused;

	if (var->activate & FB_ACTIVATE_INV_MODE) {
		struct fb_videomode mode1, mode2;

		klpr_fb_var_to_videomode(&mode1, var);
		klpr_fb_var_to_videomode(&mode2, &info->var);
		/* make sure we don't delete the videomode of current var */
		ret = klpr_fb_mode_is_equal(&mode1, &mode2);
		if (!ret) {
			ret = (*klpe_fbcon_mode_deleted)(info, &mode1);
			if (!ret)
				(*klpe_fb_delete_videomode)(&mode1, &info->modelist);
		}

		return ret ? -EINVAL : 0;
	}

	if (!(var->activate & FB_ACTIVATE_FORCE) &&
	    !memcmp(&info->var, var, sizeof(struct fb_var_screeninfo)))
		return 0;

	activate = var->activate;

	/* When using FOURCC mode, make sure the red, green, blue and
	 * transp fields are set to 0.
	 */
	if ((info->fix.capabilities & FB_CAP_FOURCC) &&
	    var->grayscale > 1) {
		if (var->red.offset     || var->green.offset    ||
		    var->blue.offset    || var->transp.offset   ||
		    var->red.length     || var->green.length    ||
		    var->blue.length    || var->transp.length   ||
		    var->red.msb_right  || var->green.msb_right ||
		    var->blue.msb_right || var->transp.msb_right)
			return -EINVAL;
	}

	if (!info->fbops->fb_check_var) {
		*var = info->var;
		return 0;
	}

	/* bitfill_aligned() assumes that it's at least 8x8 */
	if (var->xres < 8 || var->yres < 8)
		return -EINVAL;

	/* Too huge resolution causes multiplication overflow. */
	if (check_mul_overflow(var->xres, var->yres, &unused) ||
	    check_mul_overflow(var->xres_virtual, var->yres_virtual, &unused))
		return -EINVAL;

	ret = info->fbops->fb_check_var(var, info);

	if (ret)
		return ret;

	/* verify that virtual resolution >= physical resolution */
	if (var->xres_virtual < var->xres ||
	    var->yres_virtual < var->yres) {
		pr_warn("WARNING: fbcon: Driver '%s' missed to adjust virtual screen size (%ux%u vs. %ux%u)\n",
			info->fix.id,
			var->xres_virtual, var->yres_virtual,
			var->xres, var->yres);
		return -EINVAL;
	}

	if ((var->activate & FB_ACTIVATE_MASK) != FB_ACTIVATE_NOW)
		return 0;

	if (info->fbops->fb_get_caps) {
		ret = klpr_fb_check_caps(info, var, activate);

		if (ret)
			return ret;
	}

	old_var = info->var;
	info->var = *var;

	if (info->fbops->fb_set_par) {
		ret = info->fbops->fb_set_par(info);

		if (ret) {
			info->var = old_var;
			printk(KERN_WARNING "detected "
				"fb_set_par error, "
				"error code: %d\n", ret);
			return ret;
		}
	}

	klpr_fb_pan_display(info, &info->var);
	klpr_fb_set_cmap(&info->cmap, info);
	klpr_fb_var_to_videomode(&mode, &info->var);

	if (info->modelist.prev && info->modelist.next &&
	    !list_empty(&info->modelist))
		ret = klpr_fb_add_videomode(&mode, &info->modelist);

	if (ret)
		return ret;

	event.info = info;
	event.data = &mode;
	fb_notifier_call_chain(FB_EVENT_MODE_CHANGE, &event);

	return 0;
}

int klpp_fbcon_modechange_possible(struct fb_info *info, struct fb_var_screeninfo *var);

long klpp_do_fb_ioctl(struct fb_info *info, unsigned int cmd,
			unsigned long arg)
{
	const struct fb_ops *fb;
	struct fb_var_screeninfo var;
	struct fb_fix_screeninfo fix;
	struct fb_cmap cmap_from;
	struct fb_cmap_user cmap;
	void __user *argp = (void __user *)arg;
	long ret = 0;

	switch (cmd) {
	case FBIOGET_VSCREENINFO:
		lock_fb_info(info);
		var = info->var;
		unlock_fb_info(info);

		ret = copy_to_user(argp, &var, sizeof(var)) ? -EFAULT : 0;
		break;
	case FBIOPUT_VSCREENINFO:
		if (copy_from_user(&var, argp, sizeof(var)))
			return -EFAULT;
		console_lock();
		lock_fb_info(info);
		ret = klpp_fbcon_modechange_possible(info, &var);
		if (!ret)
			ret = klpp_fb_set_var(info, &var);
		if (!ret)
			klpr_fbcon_update_vcs(info, var.activate & FB_ACTIVATE_ALL);
		unlock_fb_info(info);
		console_unlock();
		if (!ret && copy_to_user(argp, &var, sizeof(var)))
			ret = -EFAULT;
		break;
	case FBIOGET_FSCREENINFO:
		lock_fb_info(info);
		memcpy(&fix, &info->fix, sizeof(fix));
		if (info->flags & FBINFO_HIDE_SMEM_START)
			fix.smem_start = 0;
		unlock_fb_info(info);

		ret = copy_to_user(argp, &fix, sizeof(fix)) ? -EFAULT : 0;
		break;
	case FBIOPUTCMAP:
		if (copy_from_user(&cmap, argp, sizeof(cmap)))
			return -EFAULT;
		ret = (*klpe_fb_set_user_cmap)(&cmap, info);
		break;
	case FBIOGETCMAP:
		if (copy_from_user(&cmap, argp, sizeof(cmap)))
			return -EFAULT;
		lock_fb_info(info);
		cmap_from = info->cmap;
		unlock_fb_info(info);
		ret = (*klpe_fb_cmap_to_user)(&cmap_from, &cmap);
		break;
	case FBIOPAN_DISPLAY:
		if (copy_from_user(&var, argp, sizeof(var)))
			return -EFAULT;
		console_lock();
		lock_fb_info(info);
		ret = klpr_fb_pan_display(info, &var);
		unlock_fb_info(info);
		console_unlock();
		if (ret == 0 && copy_to_user(argp, &var, sizeof(var)))
			return -EFAULT;
		break;
	case FBIO_CURSOR:
		ret = -EINVAL;
		break;
	case FBIOGET_CON2FBMAP:
		ret = (*klpe_fbcon_get_con2fb_map_ioctl)(argp);
		break;
	case FBIOPUT_CON2FBMAP:
		ret = (*klpe_fbcon_set_con2fb_map_ioctl)(argp);
		break;
	case FBIOBLANK:
		console_lock();
		lock_fb_info(info);
		ret = klpr_fb_blank(info, arg);
		/* might again call into fb_blank */
		(*klpe_fbcon_fb_blanked)(info, arg);
		unlock_fb_info(info);
		console_unlock();
		break;
	default:
		lock_fb_info(info);
		fb = info->fbops;
		if (fb->fb_ioctl)
			ret = fb->fb_ioctl(info, cmd, arg);
		else
			ret = -ENOTTY;
		unlock_fb_info(info);
	}
	return ret;
}



#include <linux/kernel.h>
#include "livepatch_bsc1202087.h"
#include "../kallsyms_relocs.h"

#if defined(CONFIG_S390)
static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "fb_cmap_to_user", (void *)&klpe_fb_cmap_to_user, "fb" },
	{ "fb_delete_videomode", (void *)&klpe_fb_delete_videomode, "fb" },
	{ "fb_set_user_cmap", (void *)&klpe_fb_set_user_cmap, "fb" },
	{ "fbcon_fb_blanked", (void *)&klpe_fbcon_fb_blanked, "fb" },
	{ "fbcon_get_con2fb_map_ioctl",
	  (void *)&klpe_fbcon_get_con2fb_map_ioctl, "fb" },
	{ "fbcon_get_requirement", (void *)&klpe_fbcon_get_requirement, "fb" },
	{ "fbcon_mode_deleted", (void *)&klpe_fbcon_mode_deleted, "fb" },
	{ "fbcon_set_con2fb_map_ioctl",
	  (void *)&klpe_fbcon_set_con2fb_map_ioctl, "fb" },
	{ "fb_blank", (void *)&klpe_fb_blank, "fb" },
	{ "fb_set_cmap", (void *)&klpe_fb_set_cmap, "fb" },
	{ "fb_mode_is_equal", (void *)&klpe_fb_mode_is_equal, "fb" },
	{ "fb_var_to_videomode", (void *)&klpe_fb_var_to_videomode, "fb" },
	{ "fbcon_update_vcs", (void *)&klpe_fbcon_update_vcs, "fb" },
	{ "fb_add_videomode", (void *)&klpe_fb_add_videomode, "fb" },
	{ "fb_pan_display", (void *)&klpe_fb_pan_display, "fb" },
};
#elif defined(CONFIG_X86_64) || defined(CONFIG_PPC64)
static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "fb_cmap_to_user", (void *)&klpe_fb_cmap_to_user },
	{ "fb_delete_videomode", (void *)&klpe_fb_delete_videomode },
	{ "fb_set_user_cmap", (void *)&klpe_fb_set_user_cmap },
	{ "fbcon_fb_blanked", (void *)&klpe_fbcon_fb_blanked },
	{ "fbcon_get_con2fb_map_ioctl",
	  (void *)&klpe_fbcon_get_con2fb_map_ioctl },
	{ "fbcon_get_requirement", (void *)&klpe_fbcon_get_requirement },
	{ "fbcon_mode_deleted", (void *)&klpe_fbcon_mode_deleted },
	{ "fbcon_set_con2fb_map_ioctl",
	  (void *)&klpe_fbcon_set_con2fb_map_ioctl },
};
#else
#error "klp-ccp: non-taken branch"
#endif

#if defined(CONFIG_S390)
#define LP_MODULE "fb"

#include <linux/module.h>

static int bsc1202087_module_notify(struct notifier_block *nb,
					unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LP_MODULE))
		return 0;
	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));

	WARN(ret, "%s: delayed kallsyms lookup failed. System is broken and can crash.\n",
		__func__);

	return ret;
}

static struct notifier_block module_nb = {
	.notifier_call = bsc1202087_module_notify,
	.priority = INT_MIN+1,
};

int bsc1202087_fbmem_init(void)
{
	int ret;

	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LP_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&module_nb);
	module_put(mod);

	return ret;
}

void bsc1202087_fbmem_cleanup(void)
{
	unregister_module_notifier(&module_nb);
}
#elif defined(CONFIG_X86_64) || defined(CONFIG_PPC64)
int bsc1202087_fbmem_init(void)
{
	return klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
}

void bsc1202087_fbmem_cleanup(void)
{
}
#else
#error "klp-ccp: non-taken branch"
#endif

#endif /* IS_ENABLED(CONFIG_FRAMEBUFFER_CONSOLE) */
