/*
 * bsc1203994_mac80211_util
 *
 * Fix for CVE-2022-41674, CVE-2022-42719, CVE-2022-42720 and CVE-2022-42721,
 * bsc#1203994 (net/mac80211/util.c part)
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_CFG80211)

#if !IS_MODULE(CONFIG_MAC80211)
#error "Live patch supports only CONFIG_MAC80211=m"
#endif

#include "bsc1203994_common.h"

/* klp-ccp: from include/net/mac80211.h */
#include <linux/if_ether.h>
#include <linux/skbuff.h>
#include <linux/ieee80211.h>
#include <net/cfg80211.h>

/* klp-ccp: from include/net/cfg80211.h */
static const struct element *
(*klpe_cfg80211_find_elem_match)(u8 eid, const u8 *ies, unsigned int len,
			 const u8 *match, unsigned int match_len,
			 unsigned int match_offset);

static inline const u8 *
klpr_cfg80211_find_ie_match(u8 eid, const u8 *ies, unsigned int len,
		       const u8 *match, unsigned int match_len,
		       unsigned int match_offset)
{
	/* match_offset can't be smaller than 2, unless match_len is
	 * zero, in which case match_offset must be zero as well.
	 */
	if (WARN_ON((match_len && match_offset < 2) ||
		    (!match_len && match_offset)))
		return NULL;

	return (void *)(*klpe_cfg80211_find_elem_match)(eid, ies, len,
						match, match_len,
						match_offset ?
							match_offset - 2 : 0);
}

static inline const u8 *klpr_cfg80211_find_ie(u8 eid, const u8 *ies, int len)
{
	return klpr_cfg80211_find_ie_match(eid, ies, len, NULL, 0, 0);
}

static inline const struct element *
klpr_cfg80211_find_ext_elem(u8 ext_eid, const u8 *ies, int len)
{
	return (*klpe_cfg80211_find_elem_match)(WLAN_EID_EXTENSION, ies, len,
					&ext_eid, 1, 0);
}

static bool (*klpe_cfg80211_is_element_inherited)(const struct element *element,
				   const struct element *non_inherit_element);

static size_t (*klpe_cfg80211_merge_profile)(const u8 *ie, size_t ielen,
			      const struct element *mbssid_elem,
			      const struct element *sub_elem,
			      u8 *merged_ie, size_t max_copy_len);

/* klp-ccp: from include/net/mac80211.h */
#include <asm/unaligned.h>
/* klp-ccp: from net/mac80211/util.c */
#include <linux/netdevice.h>
#include <linux/export.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/skbuff.h>
#include <linux/etherdevice.h>
#include <linux/bitmap.h>
#include <linux/crc32.h>
#include <net/net_namespace.h>
#include <net/cfg80211.h>
#include <net/rtnetlink.h>

/* klp-ccp: from net/mac80211/driver-ops.h */
#include <net/mac80211.h>

/* klp-ccp: from net/mac80211/rate.h */
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/types.h>
#include <net/mac80211.h>
/* klp-ccp: from net/mac80211/mesh.h */
#include <linux/types.h>
#include <linux/jhash.h>
/* klp-ccp: from net/mac80211/wme.h */
#include <linux/netdevice.h>
/* klp-ccp: from net/mac80211/led.h */
#include <linux/list.h>
#include <linux/spinlock.h>
#include <linux/leds.h>
/* klp-ccp: from net/mac80211/wep.h */
#include <linux/skbuff.h>
#include <linux/types.h>

/* klp-ccp: from net/mac80211/util.c */
static void ieee80211_parse_extension_element(u32 *crc,
					      const struct element *elem,
					      struct ieee802_11_elems *elems)
{
	const void *data = elem->data + 1;
	u8 len;

	if (!elem->datalen)
		return;

	len = elem->datalen - 1;

	switch (elem->data[0]) {
	case WLAN_EID_EXT_HE_MU_EDCA:
		if (len == sizeof(*elems->mu_edca_param_set)) {
			elems->mu_edca_param_set = data;
			if (crc)
				*crc = crc32_be(*crc, (void *)elem,
						elem->datalen + 2);
		}
		break;
	case WLAN_EID_EXT_HE_CAPABILITY:
		elems->he_cap = data;
		elems->he_cap_len = len;
		break;
	case WLAN_EID_EXT_HE_OPERATION:
		if (len >= sizeof(*elems->he_operation) &&
		    len >= ieee80211_he_oper_size(data) - 1) {
			if (crc)
				*crc = crc32_be(*crc, (void *)elem,
						elem->datalen + 2);
			elems->he_operation = data;
		}
		break;
	case WLAN_EID_EXT_UORA:
		if (len == 1)
			elems->uora_element = data;
		break;
	case WLAN_EID_EXT_MAX_CHANNEL_SWITCH_TIME:
		if (len == 3)
			elems->max_channel_switch_time = data;
		break;
	case WLAN_EID_EXT_MULTIPLE_BSSID_CONFIGURATION:
		if (len == sizeof(*elems->mbssid_config_ie))
			elems->mbssid_config_ie = data;
		break;
	case WLAN_EID_EXT_HE_SPR:
		if (len >= sizeof(*elems->he_spr) &&
		    len >= ieee80211_he_spr_size(data))
			elems->he_spr = data;
		break;
	case WLAN_EID_EXT_HE_6GHZ_CAPA:
		if (len == sizeof(*elems->he_6ghz_capa))
			elems->he_6ghz_capa = data;
		break;
	}
}

/*
 * Fix CVE-2022-42719
 *  -5 lines, +4 lines
 */
static u32
klpp__ieee802_11_parse_elems_full(struct ieee80211_elems_parse_params *params,
			     struct ieee802_11_elems *elems,
			     const struct element *check_inherit)
{
	const struct element *elem;
	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +1 line
	 */
	bool calc_crc = params->filter != 0;
	DECLARE_BITMAP(seen_elems, 256);
	/*
	 * Fix CVE-2022-42719
	 *  +1 line
	 */
	u32 crc = params->crc;
	const u8 *ie;

	bitmap_zero(seen_elems, 256);

	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +1 line
	 */
	for_each_element(elem, params->start, params->len) {
		bool elem_parse_failed;
		u8 id = elem->id;
		u8 elen = elem->datalen;
		const u8 *pos = elem->data;

		if (check_inherit &&
		    !(*klpe_cfg80211_is_element_inherited)(elem,
						   check_inherit))
			continue;

		switch (id) {
		case WLAN_EID_SSID:
		case WLAN_EID_SUPP_RATES:
		case WLAN_EID_FH_PARAMS:
		case WLAN_EID_DS_PARAMS:
		case WLAN_EID_CF_PARAMS:
		case WLAN_EID_TIM:
		case WLAN_EID_IBSS_PARAMS:
		case WLAN_EID_CHALLENGE:
		case WLAN_EID_RSN:
		case WLAN_EID_ERP_INFO:
		case WLAN_EID_EXT_SUPP_RATES:
		case WLAN_EID_HT_CAPABILITY:
		case WLAN_EID_HT_OPERATION:
		case WLAN_EID_VHT_CAPABILITY:
		case WLAN_EID_VHT_OPERATION:
		case WLAN_EID_MESH_ID:
		case WLAN_EID_MESH_CONFIG:
		case WLAN_EID_PEER_MGMT:
		case WLAN_EID_PREQ:
		case WLAN_EID_PREP:
		case WLAN_EID_PERR:
		case WLAN_EID_RANN:
		case WLAN_EID_CHANNEL_SWITCH:
		case WLAN_EID_EXT_CHANSWITCH_ANN:
		case WLAN_EID_COUNTRY:
		case WLAN_EID_PWR_CONSTRAINT:
		case WLAN_EID_TIMEOUT_INTERVAL:
		case WLAN_EID_SECONDARY_CHANNEL_OFFSET:
		case WLAN_EID_WIDE_BW_CHANNEL_SWITCH:
		case WLAN_EID_CHAN_SWITCH_PARAM:
		case WLAN_EID_EXT_CAPABILITY:
		case WLAN_EID_CHAN_SWITCH_TIMING:
		case WLAN_EID_LINK_ID:
		case WLAN_EID_BSS_MAX_IDLE_PERIOD:
		case WLAN_EID_RSNX:
		/*
		 * not listing WLAN_EID_CHANNEL_SWITCH_WRAPPER -- it seems possible
		 * that if the content gets bigger it might be needed more than once
		 */
			if (test_bit(id, seen_elems)) {
				elems->parse_error = true;
				continue;
			}
			break;
		}

		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		if (calc_crc && id < 64 && (params->filter & (1ULL << id)))
			crc = crc32_be(crc, pos - 2, elen + 2);

		elem_parse_failed = false;

		switch (id) {
		case WLAN_EID_LINK_ID:
			if (elen + 2 != sizeof(struct ieee80211_tdls_lnkie)) {
				elem_parse_failed = true;
				break;
			}
			elems->lnk_id = (void *)(pos - 2);
			break;
		case WLAN_EID_CHAN_SWITCH_TIMING:
			if (elen != sizeof(struct ieee80211_ch_switch_timing)) {
				elem_parse_failed = true;
				break;
			}
			elems->ch_sw_timing = (void *)pos;
			break;
		case WLAN_EID_EXT_CAPABILITY:
			elems->ext_capab = pos;
			elems->ext_capab_len = elen;
			break;
		case WLAN_EID_SSID:
			elems->ssid = pos;
			elems->ssid_len = elen;
			break;
		case WLAN_EID_SUPP_RATES:
			elems->supp_rates = pos;
			elems->supp_rates_len = elen;
			break;
		case WLAN_EID_DS_PARAMS:
			if (elen >= 1)
				elems->ds_params = pos;
			else
				elem_parse_failed = true;
			break;
		case WLAN_EID_TIM:
			if (elen >= sizeof(struct ieee80211_tim_ie)) {
				elems->tim = (void *)pos;
				elems->tim_len = elen;
			} else
				elem_parse_failed = true;
			break;
		/*
		 * Fix CVE-2022-42719
		 *  -4 lines
		 */
		case WLAN_EID_VENDOR_SPECIFIC:
			if (elen >= 4 && pos[0] == 0x00 && pos[1] == 0x50 &&
			    pos[2] == 0xf2) {
				/* Microsoft OUI (00:50:F2) */

				if (calc_crc)
					crc = crc32_be(crc, pos - 2, elen + 2);

				if (elen >= 5 && pos[3] == 2) {
					/* OUI Type 2 - WMM IE */
					if (pos[4] == 0) {
						elems->wmm_info = pos;
						elems->wmm_info_len = elen;
					} else if (pos[4] == 1) {
						elems->wmm_param = pos;
						elems->wmm_param_len = elen;
					}
				}
			}
			break;
		case WLAN_EID_RSN:
			elems->rsn = pos;
			elems->rsn_len = elen;
			break;
		case WLAN_EID_ERP_INFO:
			if (elen >= 1)
				elems->erp_info = pos;
			else
				elem_parse_failed = true;
			break;
		case WLAN_EID_EXT_SUPP_RATES:
			elems->ext_supp_rates = pos;
			elems->ext_supp_rates_len = elen;
			break;
		case WLAN_EID_HT_CAPABILITY:
			if (elen >= sizeof(struct ieee80211_ht_cap))
				elems->ht_cap_elem = (void *)pos;
			else
				elem_parse_failed = true;
			break;
		case WLAN_EID_HT_OPERATION:
			if (elen >= sizeof(struct ieee80211_ht_operation))
				elems->ht_operation = (void *)pos;
			else
				elem_parse_failed = true;
			break;
		case WLAN_EID_VHT_CAPABILITY:
			if (elen >= sizeof(struct ieee80211_vht_cap))
				elems->vht_cap_elem = (void *)pos;
			else
				elem_parse_failed = true;
			break;
		case WLAN_EID_VHT_OPERATION:
			if (elen >= sizeof(struct ieee80211_vht_operation)) {
				elems->vht_operation = (void *)pos;
				if (calc_crc)
					crc = crc32_be(crc, pos - 2, elen + 2);
				break;
			}
			elem_parse_failed = true;
			break;
		case WLAN_EID_OPMODE_NOTIF:
			if (elen > 0) {
				elems->opmode_notif = pos;
				if (calc_crc)
					crc = crc32_be(crc, pos - 2, elen + 2);
				break;
			}
			elem_parse_failed = true;
			break;
		case WLAN_EID_MESH_ID:
			elems->mesh_id = pos;
			elems->mesh_id_len = elen;
			break;
		case WLAN_EID_MESH_CONFIG:
			if (elen >= sizeof(struct ieee80211_meshconf_ie))
				elems->mesh_config = (void *)pos;
			else
				elem_parse_failed = true;
			break;
		case WLAN_EID_PEER_MGMT:
			elems->peering = pos;
			elems->peering_len = elen;
			break;
		case WLAN_EID_MESH_AWAKE_WINDOW:
			if (elen >= 2)
				elems->awake_window = (void *)pos;
			break;
		case WLAN_EID_PREQ:
			elems->preq = pos;
			elems->preq_len = elen;
			break;
		case WLAN_EID_PREP:
			elems->prep = pos;
			elems->prep_len = elen;
			break;
		case WLAN_EID_PERR:
			elems->perr = pos;
			elems->perr_len = elen;
			break;
		case WLAN_EID_RANN:
			if (elen >= sizeof(struct ieee80211_rann_ie))
				elems->rann = (void *)pos;
			else
				elem_parse_failed = true;
			break;
		case WLAN_EID_CHANNEL_SWITCH:
			if (elen != sizeof(struct ieee80211_channel_sw_ie)) {
				elem_parse_failed = true;
				break;
			}
			elems->ch_switch_ie = (void *)pos;
			break;
		case WLAN_EID_EXT_CHANSWITCH_ANN:
			if (elen != sizeof(struct ieee80211_ext_chansw_ie)) {
				elem_parse_failed = true;
				break;
			}
			elems->ext_chansw_ie = (void *)pos;
			break;
		case WLAN_EID_SECONDARY_CHANNEL_OFFSET:
			if (elen != sizeof(struct ieee80211_sec_chan_offs_ie)) {
				elem_parse_failed = true;
				break;
			}
			elems->sec_chan_offs = (void *)pos;
			break;
		case WLAN_EID_CHAN_SWITCH_PARAM:
			if (elen !=
			    sizeof(*elems->mesh_chansw_params_ie)) {
				elem_parse_failed = true;
				break;
			}
			elems->mesh_chansw_params_ie = (void *)pos;
			break;
		case WLAN_EID_WIDE_BW_CHANNEL_SWITCH:
			/*
			 * Fix CVE-2022-42719
			 *  -1 line, +1 line
			 */
			if (!params->action ||
			    elen != sizeof(*elems->wide_bw_chansw_ie)) {
				elem_parse_failed = true;
				break;
			}
			elems->wide_bw_chansw_ie = (void *)pos;
			break;
		case WLAN_EID_CHANNEL_SWITCH_WRAPPER:
			/*
			 * Fix CVE-2022-42719
			 *  -1 line, +1 line
			 */
			if (params->action) {
				elem_parse_failed = true;
				break;
			}
			/*
			 * This is a bit tricky, but as we only care about
			 * the wide bandwidth channel switch element, so
			 * just parse it out manually.
			 */
			ie = klpr_cfg80211_find_ie(WLAN_EID_WIDE_BW_CHANNEL_SWITCH,
					      pos, elen);
			if (ie) {
				if (ie[1] == sizeof(*elems->wide_bw_chansw_ie))
					elems->wide_bw_chansw_ie =
						(void *)(ie + 2);
				else
					elem_parse_failed = true;
			}
			break;
		case WLAN_EID_COUNTRY:
			elems->country_elem = pos;
			elems->country_elem_len = elen;
			break;
		case WLAN_EID_PWR_CONSTRAINT:
			if (elen != 1) {
				elem_parse_failed = true;
				break;
			}
			elems->pwr_constr_elem = pos;
			break;
		case WLAN_EID_CISCO_VENDOR_SPECIFIC:
			/* Lots of different options exist, but we only care
			 * about the Dynamic Transmit Power Control element.
			 * First check for the Cisco OUI, then for the DTPC
			 * tag (0x00).
			 */
			if (elen < 4) {
				elem_parse_failed = true;
				break;
			}

			if (pos[0] != 0x00 || pos[1] != 0x40 ||
			    pos[2] != 0x96 || pos[3] != 0x00)
				break;

			if (elen != 6) {
				elem_parse_failed = true;
				break;
			}

			if (calc_crc)
				crc = crc32_be(crc, pos - 2, elen + 2);

			elems->cisco_dtpc_elem = pos;
			break;
		case WLAN_EID_ADDBA_EXT:
			if (elen != sizeof(struct ieee80211_addba_ext_ie)) {
				elem_parse_failed = true;
				break;
			}
			elems->addba_ext_ie = (void *)pos;
			break;
		case WLAN_EID_TIMEOUT_INTERVAL:
			if (elen >= sizeof(struct ieee80211_timeout_interval_ie))
				elems->timeout_int = (void *)pos;
			else
				elem_parse_failed = true;
			break;
		case WLAN_EID_BSS_MAX_IDLE_PERIOD:
			if (elen >= sizeof(*elems->max_idle_period_ie))
				elems->max_idle_period_ie = (void *)pos;
			break;
		case WLAN_EID_RSNX:
			elems->rsnx = pos;
			elems->rsnx_len = elen;
			break;
		case WLAN_EID_EXTENSION:
			ieee80211_parse_extension_element(calc_crc ?
								&crc : NULL,
							  elem, elems);
			break;
		default:
			break;
		}

		if (elem_parse_failed)
			elems->parse_error = true;
		else
			__set_bit(id, seen_elems);
	}

	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +1 line
	 */
	if (!for_each_element_completed(elem, params->start, params->len))
		elems->parse_error = true;

	return crc;
}

static size_t klpp_ieee802_11_find_bssid_profile(const u8 *start, size_t len,
					    struct ieee802_11_elems *elems,
					    /*
					     * Fix CVE-2022-42719
					     *  -2 lines, +2 lines
					     */
					    const u8 *transmitter_bssid,
					    const u8 *bss_bssid,
					    u8 *nontransmitted_profile)
{
	const struct element *elem, *sub;
	size_t profile_len = 0;
	bool found = false;

	if (!bss_bssid || !transmitter_bssid)
		return profile_len;

	for_each_element_id(elem, WLAN_EID_MULTIPLE_BSSID, start, len) {
		if (elem->datalen < 2)
			continue;
		/*
		 * Fix CVE-2022-41674
		 *  +2 lines
		 */
		if (elem->data[0] < 1 || elem->data[0] > 8)
			continue;

		for_each_element(sub, elem->data + 1, elem->datalen - 1) {
			u8 new_bssid[ETH_ALEN];
			const u8 *index;

			if (sub->id != 0 || sub->datalen < 4) {
				/* not a valid BSS profile */
				continue;
			}

			if (sub->data[0] != WLAN_EID_NON_TX_BSSID_CAP ||
			    sub->data[1] != 2) {
				/* The first element of the
				 * Nontransmitted BSSID Profile is not
				 * the Nontransmitted BSSID Capability
				 * element.
				 */
				continue;
			}

			memset(nontransmitted_profile, 0, len);
			profile_len = (*klpe_cfg80211_merge_profile)(start, len,
							     elem,
							     sub,
							     nontransmitted_profile,
							     len);

			/* found a Nontransmitted BSSID Profile */
			index = klpr_cfg80211_find_ie(WLAN_EID_MULTI_BSSID_IDX,
						 nontransmitted_profile,
						 profile_len);
			if (!index || index[1] < 1 || index[2] == 0) {
				/* Invalid MBSSID Index element */
				continue;
			}

			cfg80211_gen_new_bssid(transmitter_bssid,
					       elem->data[0],
					       index[2],
					       new_bssid);
			if (ether_addr_equal(new_bssid, bss_bssid)) {
				found = true;
				elems->bssid_index_len = index[1];
				elems->bssid_index = (void *)&index[2];
				break;
			}
		}
	}

	return found ? profile_len : 0;
}

/*
 * Fix CVE-2022-42719
 *  -4 lines, +2 lines
 */
struct ieee802_11_elems *
klpp_ieee802_11_parse_elems_full(struct ieee80211_elems_parse_params *params)
{
	/*
	 * Fix CVE-2022-42719
	 *  +2 lines
	 */
	struct klpp_ieee802_11_elems *klpp_elems;
	struct ieee802_11_elems *elems;
	const struct element *non_inherit = NULL;
	u8 *nontransmitted_profile;
	int nontransmitted_profile_len = 0;
	/*
	 * Fix CVE-2022-42719
	 *  +1 line
	 */
	size_t scratch_len = params->len;

	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +4 lines
	 */
	klpp_elems = kzalloc(sizeof(*klpp_elems) + scratch_len, GFP_ATOMIC);
	if (!klpp_elems)
		return NULL;
	elems = &klpp_elems->orig;
	/*
	 * Fix CVE-2022-42719
	 *  -2 lines, +4 lines
	 */
	elems->ie_start = params->start;
	elems->total_len = params->len;
	klpp_elems->scratch_len = scratch_len;
	klpp_elems->scratch_pos = klpp_elems->scratch;

	/*
	 * Fix CVE-2022-42719
	 *  -16 lines, +12 lines
	 */
	nontransmitted_profile = klpp_elems->scratch_pos;
	nontransmitted_profile_len =
		klpp_ieee802_11_find_bssid_profile(params->start, params->len,
					      elems,
					      params->transmitter_bssid,
					      params->bss_bssid,
					      nontransmitted_profile);
	klpp_elems->scratch_pos += nontransmitted_profile_len;
	klpp_elems->scratch_len -= nontransmitted_profile_len;
	non_inherit = klpr_cfg80211_find_ext_elem(WLAN_EID_EXT_NON_INHERITANCE,
					     nontransmitted_profile,
					     nontransmitted_profile_len);

	/*
	 * Fix CVE-2022-42719
	 *  -2 lines, +1 line
	 */
	klpp_elems->crc = klpp__ieee802_11_parse_elems_full(params, elems, non_inherit);

	/* Override with nontransmitted profile, if found */
	/*
	 * Fix CVE-2022-42719
	 *  -4 lines, +9 lines
	 */
	if (nontransmitted_profile_len) {
		struct ieee80211_elems_parse_params sub = {
			.start = nontransmitted_profile,
			.len = nontransmitted_profile_len,
			.action = params->action,
		};

		klpp__ieee802_11_parse_elems_full(&sub, elems, NULL);
	}

	if (elems->tim && !elems->parse_error) {
		const struct ieee80211_tim_ie *tim_ie = elems->tim;

		elems->dtim_period = tim_ie->dtim_period;
		elems->dtim_count = tim_ie->dtim_count;
	}

	/* Override DTIM period and count if needed */
	if (elems->bssid_index &&
	    elems->bssid_index_len >=
	    offsetofend(struct ieee80211_bssid_index, dtim_period))
		elems->dtim_period = elems->bssid_index->dtim_period;

	if (elems->bssid_index &&
	    elems->bssid_index_len >=
	    offsetofend(struct ieee80211_bssid_index, dtim_count))
		elems->dtim_count = elems->bssid_index->dtim_count;

	/*
	 * Fix CVE-2022-42719
	 *  -1 line
	 */

	return elems;
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1203994.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "mac80211"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "cfg80211_find_elem_match", (void *)&klpe_cfg80211_find_elem_match,
	  "cfg80211" },
	{ "cfg80211_is_element_inherited",
	  (void *)&klpe_cfg80211_is_element_inherited, "cfg80211" },
	{ "cfg80211_merge_profile", (void *)&klpe_cfg80211_merge_profile,
	  "cfg80211" },
};

static int livepatch_bsc1203994_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1203994_module_nb = {
	.notifier_call = livepatch_bsc1203994_module_notify,
	.priority = INT_MIN+1,
};

int bsc1203994_mac80211_util_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LIVEPATCHED_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&livepatch_bsc1203994_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void bsc1203994_mac80211_util_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1203994_module_nb);
}

#endif /* IS_ENABLED(CONFIG_CFG80211) */
