/*
 * bsc1203994_mac80211_mlme
 *
 * Fix for CVE-2022-41674, CVE-2022-42719, CVE-2022-42720 and CVE-2022-42721,
 * bsc#1203994 (net/mac80211/mlme.c part)
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_CFG80211)

#if !IS_MODULE(CONFIG_MAC80211)
#error "Live patch supports only CONFIG_MAC80211=m"
#endif

#include "bsc1203994_common.h"

/* klp-ccp: from net/mac80211/mlme.c */
#include <linux/delay.h>
#include <linux/if_ether.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/etherdevice.h>
#include <linux/moduleparam.h>
#include <linux/rtnetlink.h>
#include <linux/crc32.h>
#include <linux/slab.h>
#include <linux/export.h>
#include <net/mac80211.h>

/* klp-ccp: from include/net/cfg80211.h */
static bool (*klpe_cfg80211_chandef_valid)(const struct cfg80211_chan_def *chandef);

static int (*klpe_ieee80211_freq_khz_to_channel)(u32 freq);

static inline int
klpr_ieee80211_frequency_to_channel(int freq)
{
	return (*klpe_ieee80211_freq_khz_to_channel)(MHZ_TO_KHZ(freq));
}

static struct ieee80211_channel *
(*klpe_ieee80211_get_channel_khz)(struct wiphy *wiphy, u32 freq);

static inline struct ieee80211_channel *
klpr_ieee80211_get_channel(struct wiphy *wiphy, int freq)
{
	return (*klpe_ieee80211_get_channel_khz)(wiphy, MHZ_TO_KHZ(freq));
}

static void (*klpe_cfg80211_rx_mlme_mgmt)(struct net_device *dev, const u8 *buf, size_t len);

static void (*klpe_cfg80211_rx_assoc_resp)(struct net_device *dev,
			    struct cfg80211_bss *bss,
			    const u8 *buf, size_t len,
			    int uapsd_queues,
			    const u8 *req_ies, size_t req_ies_len);

static void (*klpe_cfg80211_assoc_timeout)(struct net_device *dev, struct cfg80211_bss *bss);

static void (*klpe_cfg80211_abandon_assoc)(struct net_device *dev, struct cfg80211_bss *bss);

static void (*klpe_cfg80211_ch_switch_notify)(struct net_device *dev,
			       struct cfg80211_chan_def *chandef);

static int (*klpe_cfg80211_get_p2p_attr)(const u8 *ies, unsigned int len,
			  enum ieee80211_p2p_attr_id attr,
			  u8 *buf, unsigned int bufsize);

static const struct element *
(*klpe_cfg80211_find_elem_match)(u8 eid, const u8 *ies, unsigned int len,
			 const u8 *match, unsigned int match_len,
			 unsigned int match_offset);

static inline const struct element *
klpr_cfg80211_find_elem(u8 eid, const u8 *ies, int len)
{
	return (*klpe_cfg80211_find_elem_match)(eid, ies, len, NULL, 0, 0);
}

/* klp-ccp: from include/net/mac80211.h */
static void (*klpe_ieee80211_queue_work)(struct ieee80211_hw *hw, struct work_struct *work);

static void (*klpe_ieee80211_cqm_rssi_notify)(struct ieee80211_vif *vif,
			       enum nl80211_cqm_rssi_threshold_event rssi_event,
			       s32 rssi_level,
			       gfp_t gfp);

/* klp-ccp: from net/mac80211/mlme.c */
#include <asm/unaligned.h>

/* klp-ccp: from net/mac80211/sta_info.h */
static int (*klpe_sta_info_move_state)(struct sta_info *sta,
			enum ieee80211_sta_state new_state);

static struct sta_info *(*klpe_sta_info_get)(struct ieee80211_sub_if_data *sdata,
			      const u8 *addr);

static int __must_check (*klpe___sta_info_destroy)(struct sta_info *sta);
static int (*klpe_sta_info_destroy_addr)(struct ieee80211_sub_if_data *sdata,
			  const u8 *addr);


/* klp-ccp: from net/mac80211/ieee80211_i.h */
static const u8 (*klpe_ieee80211_ac_to_qos_mask)[IEEE80211_NUM_ACS];

static int (*klpe_ieee80211_hw_config)(struct ieee80211_local *local, u32 changed);

static void (*klpe_ieee80211_bss_info_change_notify)(struct ieee80211_sub_if_data *sdata,
				      u32 changed);

static void (*klpe_ieee80211_send_pspoll)(struct ieee80211_local *local,
			   struct ieee80211_sub_if_data *sdata);
static void (*klpe_ieee80211_recalc_ps)(struct ieee80211_local *local);
static void (*klpe_ieee80211_recalc_ps_vif)(struct ieee80211_sub_if_data *sdata);

void klpp_ieee80211_sta_rx_queued_mgmt(struct ieee80211_sub_if_data *sdata,
				  struct sk_buff *skb);
static void (*klpe_ieee80211_sta_reset_beacon_monitor)(struct ieee80211_sub_if_data *sdata);

static bool (*klpe___ieee80211_recalc_txpower)(struct ieee80211_sub_if_data *sdata);

static bool (*klpe_ieee80211_ht_cap_ie_to_sta_ht_cap)(struct ieee80211_sub_if_data *sdata,
				       struct ieee80211_supported_band *sband,
				       const struct ieee80211_ht_cap *ht_cap_ie,
				       struct sta_info *sta);

static void
(*klpe_ieee80211_vht_cap_ie_to_sta_vht_cap)(struct ieee80211_sub_if_data *sdata,
				    struct ieee80211_supported_band *sband,
				    const struct ieee80211_vht_cap *vht_cap_ie,
				    struct sta_info *sta);

static void (*klpe_ieee80211_vht_handle_opmode)(struct ieee80211_sub_if_data *sdata,
				 struct sta_info *sta, u8 opmode,
				 enum nl80211_band band);

static void
(*klpe_ieee80211_he_cap_ie_to_sta_he_cap)(struct ieee80211_sub_if_data *sdata,
				  struct ieee80211_supported_band *sband,
				  const u8 *he_cap_ie, u8 he_cap_len,
				  const struct ieee80211_he_6ghz_capa *he_6ghz_capa,
				  struct sta_info *sta);
static void
(*klpe_ieee80211_he_spr_ie_to_bss_conf)(struct ieee80211_vif *vif,
				const struct ieee80211_he_spr *he_spr_ie_elem);

static void
(*klpe_ieee80211_he_op_ie_to_bss_conf)(struct ieee80211_vif *vif,
			const struct ieee80211_he_operation *he_op_ie_elem);

static void (*klpe_ieee80211_set_wmm_default)(struct ieee80211_sub_if_data *sdata,
			       bool bss_notify, bool enable_qos);

static void (*klpe___ieee80211_tx_skb_tid_band)(struct ieee80211_sub_if_data *sdata,
				 struct sk_buff *skb, int tid,
				 enum nl80211_band band);

static inline void klpr_ieee80211_tx_skb_tid(struct ieee80211_sub_if_data *sdata,
					struct sk_buff *skb, int tid)
{
	struct ieee80211_chanctx_conf *chanctx_conf;

	rcu_read_lock();
	chanctx_conf = rcu_dereference(sdata->vif.chanctx_conf);
	if (WARN_ON(!chanctx_conf)) {
		rcu_read_unlock();
		kfree_skb(skb);
		return;
	}

	(*klpe___ieee80211_tx_skb_tid_band)(sdata, skb, tid,
				    chanctx_conf->def.chan->band);
	rcu_read_unlock();
}

static inline void klpr_ieee80211_tx_skb(struct ieee80211_sub_if_data *sdata,
				    struct sk_buff *skb)
{
	/* Send all internal mgmt frames on VO. Accordingly set TID to 7. */
	klpr_ieee80211_tx_skb_tid(sdata, skb, 7);
}

static void (*klpe_ieee80211_send_nullfunc)(struct ieee80211_local *local,
			     struct ieee80211_sub_if_data *sdata,
			     bool powersave);
static void (*klpe_ieee80211_sta_rx_notify)(struct ieee80211_sub_if_data *sdata,
			     struct ieee80211_hdr *hdr);

static void (*klpe_ieee80211_wake_vif_queues)(struct ieee80211_local *local,
			       struct ieee80211_sub_if_data *sdata,
			       enum queue_stop_reason reason);

static void (*klpe_ieee80211_send_auth)(struct ieee80211_sub_if_data *sdata,
			 u16 transaction, u16 auth_alg, u16 status,
			 const u8 *extra, size_t extra_len, const u8 *bssid,
			 const u8 *da, const u8 *key, u8 key_len, u8 key_idx,
			 u32 tx_flags);

static void (*klpe_ieee80211_recalc_smps)(struct ieee80211_sub_if_data *sdata);

static u32 (*klpe_ieee80211_chandef_downgrade)(struct cfg80211_chan_def *c);

static int __must_check
(*klpe_ieee80211_vif_change_bandwidth)(struct ieee80211_sub_if_data *sdata,
			       const struct cfg80211_chan_def *chandef,
			       u32 *changed);
static void (*klpe_ieee80211_vif_release_channel)(struct ieee80211_sub_if_data *sdata);

static void (*klpe_ieee80211_tdls_handle_disconnect)(struct ieee80211_sub_if_data *sdata,
				      const u8 *peer, u16 reason);
static const char *(*klpe_ieee80211_get_reason_code_string)(u16 reason_code);

/* klp-ccp: from net/mac80211/driver-ops.h */
#include <net/mac80211.h>

/* klp-ccp: from net/mac80211/trace.h */
#include <linux/tracepoint.h>
#include <net/mac80211.h>

KLPR_TRACE_EVENT(drv_return_int,
	TP_PROTO(struct ieee80211_local *local, int ret),
	TP_ARGS(local, ret)
);

KLPR_DEFINE_EVENT(local_sdata_evt, drv_post_channel_switch,
	     TP_PROTO(struct ieee80211_local *local,
		      struct ieee80211_sub_if_data *sdata),
	     TP_ARGS(local, sdata)
);

/* klp-ccp: from net/mac80211/driver-ops.h */
#define check_sdata_in_driver(sdata)	({					\
	!WARN_ONCE(!(sdata->flags & IEEE80211_SDATA_IN_DRIVER),			\
		   "%s: Failed check-sdata-in-driver check, flags: 0x%x\n",	\
		   sdata->dev ? sdata->dev->name : sdata->name, sdata->flags);	\
})

static void (*klpe_drv_event_callback)(struct ieee80211_local *local,
				      struct ieee80211_sub_if_data *sdata,
				      const struct ieee80211_event *event);

static void (*klpe_drv_mgd_prepare_tx)(struct ieee80211_local *local,
				      struct ieee80211_sub_if_data *sdata,
				      u16 duration);

static inline int
klpr_drv_post_channel_switch(struct ieee80211_sub_if_data *sdata)
{
	struct ieee80211_local *local = sdata->local;
	int ret = 0;

	if (!check_sdata_in_driver(sdata))
		return -EIO;

	klpr_trace_drv_post_channel_switch(local, sdata);
	if (local->ops->post_channel_switch)
		ret = local->ops->post_channel_switch(&local->hw, &sdata->vif);
	klpr_trace_drv_return_int(local, ret);
	return ret;
}

/* klp-ccp: from net/mac80211/rate.h */
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/types.h>
#include <net/mac80211.h>

static void (*klpe_rate_control_rate_init)(struct sta_info *sta);
static void (*klpe_rate_control_rate_update)(struct ieee80211_local *local,
				    struct ieee80211_supported_band *sband,
				    struct sta_info *sta, u32 changed);

static void (*klpe_ieee80211_check_rate_mask)(struct ieee80211_sub_if_data *sdata);

/* klp-ccp: from net/mac80211/led.h */
#include <linux/list.h>
#include <linux/spinlock.h>
#include <linux/leds.h>

#ifdef CONFIG_MAC80211_LEDS
static void (*klpe_ieee80211_led_assoc)(struct ieee80211_local *local,
			 bool associated);

#else
#error "klp-ccp: non-taken branch"
#endif

/* klp-ccp: from net/mac80211/fils_aead.h */
static int (*klpe_fils_decrypt_assoc_resp)(struct ieee80211_sub_if_data *sdata,
			    u8 *frame, size_t *frame_len,
			    struct ieee80211_mgd_assoc_data *assoc_data);

/* klp-ccp: from net/mac80211/mlme.c */
#define IEEE80211_ASSOC_TIMEOUT		(HZ / 5)

static int (*klpe_beacon_loss_count);

#define IEEE80211_SIGNAL_AVE_MIN_COUNT	4

static void run_again(struct ieee80211_sub_if_data *sdata,
		      unsigned long timeout)
{
	sdata_assert_lock(sdata);

	if (!timer_pending(&sdata->u.mgd.timer) ||
	    time_before(timeout, sdata->u.mgd.timer.expires))
		mod_timer(&sdata->u.mgd.timer, timeout);
}

static u32
(*klpe_ieee80211_determine_chantype)(struct ieee80211_sub_if_data *sdata,
			     struct ieee80211_supported_band *sband,
			     struct ieee80211_channel *channel,
			     u32 vht_cap_info,
			     const struct ieee80211_ht_operation *ht_oper,
			     const struct ieee80211_vht_operation *vht_oper,
			     const struct ieee80211_he_operation *he_oper,
			     struct cfg80211_chan_def *chandef, bool tracking);

static int klpr_ieee80211_config_bw(struct ieee80211_sub_if_data *sdata,
			       struct sta_info *sta,
			       const struct ieee80211_ht_cap *ht_cap,
			       const struct ieee80211_vht_cap *vht_cap,
			       const struct ieee80211_ht_operation *ht_oper,
			       const struct ieee80211_vht_operation *vht_oper,
			       const struct ieee80211_he_operation *he_oper,
			       const u8 *bssid, u32 *changed)
{
	struct ieee80211_local *local = sdata->local;
	struct ieee80211_if_managed *ifmgd = &sdata->u.mgd;
	struct ieee80211_channel *chan = sdata->vif.bss_conf.chandef.chan;
	struct ieee80211_supported_band *sband =
		local->hw.wiphy->bands[chan->band];
	struct cfg80211_chan_def chandef;
	u16 ht_opmode;
	u32 flags;
	enum ieee80211_sta_rx_bandwidth new_sta_bw;
	u32 vht_cap_info = 0;
	int ret;

	/* if HT was/is disabled, don't track any bandwidth changes */
	if (ifmgd->flags & IEEE80211_STA_DISABLE_HT || !ht_oper)
		return 0;

	/* don't check VHT if we associated as non-VHT station */
	if (ifmgd->flags & IEEE80211_STA_DISABLE_VHT)
		vht_oper = NULL;

	/* don't check HE if we associated as non-HE station */
	if (ifmgd->flags & IEEE80211_STA_DISABLE_HE ||
	    !ieee80211_get_he_sta_cap(sband))
		he_oper = NULL;

	if (WARN_ON_ONCE(!sta))
		return -EINVAL;

	/*
	 * if bss configuration changed store the new one -
	 * this may be applicable even if channel is identical
	 */
	ht_opmode = le16_to_cpu(ht_oper->operation_mode);
	if (sdata->vif.bss_conf.ht_operation_mode != ht_opmode) {
		*changed |= BSS_CHANGED_HT;
		sdata->vif.bss_conf.ht_operation_mode = ht_opmode;
	}

	if (vht_cap)
		vht_cap_info = le32_to_cpu(vht_cap->vht_cap_info);

	/* calculate new channel (type) based on HT/VHT/HE operation IEs */
	flags = (*klpe_ieee80211_determine_chantype)(sdata, sband, chan, vht_cap_info,
					     ht_oper, vht_oper, he_oper,
					     &chandef, true);

	/*
	 * Downgrade the new channel if we associated with restricted
	 * capabilities. For example, if we associated as a 20 MHz STA
	 * to a 40 MHz AP (due to regulatory, capabilities or config
	 * reasons) then switching to a 40 MHz channel now won't do us
	 * any good -- we couldn't use it with the AP.
	 */
	if (ifmgd->flags & IEEE80211_STA_DISABLE_80P80MHZ &&
	    chandef.width == NL80211_CHAN_WIDTH_80P80)
		flags |= (*klpe_ieee80211_chandef_downgrade)(&chandef);
	if (ifmgd->flags & IEEE80211_STA_DISABLE_160MHZ &&
	    chandef.width == NL80211_CHAN_WIDTH_160)
		flags |= (*klpe_ieee80211_chandef_downgrade)(&chandef);
	if (ifmgd->flags & IEEE80211_STA_DISABLE_40MHZ &&
	    chandef.width > NL80211_CHAN_WIDTH_20)
		flags |= (*klpe_ieee80211_chandef_downgrade)(&chandef);

	if (cfg80211_chandef_identical(&chandef, &sdata->vif.bss_conf.chandef))
		return 0;

	sdata_info(sdata,
		   "AP %pM changed bandwidth, new config is %d.%03d MHz, "
		   "width %d (%d.%03d/%d MHz)\n",
		   ifmgd->bssid, chandef.chan->center_freq,
		   chandef.chan->freq_offset, chandef.width,
		   chandef.center_freq1, chandef.freq1_offset,
		   chandef.center_freq2);

	if (flags != (ifmgd->flags & (IEEE80211_STA_DISABLE_HT |
				      IEEE80211_STA_DISABLE_VHT |
				      IEEE80211_STA_DISABLE_HE |
				      IEEE80211_STA_DISABLE_40MHZ |
				      IEEE80211_STA_DISABLE_80P80MHZ |
				      IEEE80211_STA_DISABLE_160MHZ)) ||
	    !(*klpe_cfg80211_chandef_valid)(&chandef)) {
		sdata_info(sdata,
			   "AP %pM changed bandwidth in a way we can't support - disconnect\n",
			   ifmgd->bssid);
		return -EINVAL;
	}

	switch (chandef.width) {
	case NL80211_CHAN_WIDTH_20_NOHT:
	case NL80211_CHAN_WIDTH_20:
		new_sta_bw = IEEE80211_STA_RX_BW_20;
		break;
	case NL80211_CHAN_WIDTH_40:
		new_sta_bw = IEEE80211_STA_RX_BW_40;
		break;
	case NL80211_CHAN_WIDTH_80:
		new_sta_bw = IEEE80211_STA_RX_BW_80;
		break;
	case NL80211_CHAN_WIDTH_80P80:
	case NL80211_CHAN_WIDTH_160:
		new_sta_bw = IEEE80211_STA_RX_BW_160;
		break;
	default:
		return -EINVAL;
	}

	if (new_sta_bw > sta->cur_max_bandwidth)
		new_sta_bw = sta->cur_max_bandwidth;

	if (new_sta_bw < sta->sta.bandwidth) {
		sta->sta.bandwidth = new_sta_bw;
		(*klpe_rate_control_rate_update)(local, sband, sta,
					 IEEE80211_RC_BW_CHANGED);
	}

	ret = (*klpe_ieee80211_vif_change_bandwidth)(sdata, &chandef, changed);
	if (ret) {
		sdata_info(sdata,
			   "AP %pM changed bandwidth to incompatible one - disconnect\n",
			   ifmgd->bssid);
		return ret;
	}

	if (new_sta_bw > sta->sta.bandwidth) {
		sta->sta.bandwidth = new_sta_bw;
		(*klpe_rate_control_rate_update)(local, sband, sta,
					 IEEE80211_RC_BW_CHANGED);
	}

	return 0;
}
static void klpr_ieee80211_send_4addr_nullfunc(struct ieee80211_local *local,
					  struct ieee80211_sub_if_data *sdata)
{
	struct sk_buff *skb;
	struct ieee80211_hdr *nullfunc;
	__le16 fc;

	if (WARN_ON(sdata->vif.type != NL80211_IFTYPE_STATION))
		return;

	skb = dev_alloc_skb(local->hw.extra_tx_headroom + 30);
	if (!skb)
		return;

	skb_reserve(skb, local->hw.extra_tx_headroom);

	nullfunc = skb_put_zero(skb, 30);
	fc = cpu_to_le16(IEEE80211_FTYPE_DATA | IEEE80211_STYPE_NULLFUNC |
			 IEEE80211_FCTL_FROMDS | IEEE80211_FCTL_TODS);
	nullfunc->frame_control = fc;
	memcpy(nullfunc->addr1, sdata->u.mgd.bssid, ETH_ALEN);
	memcpy(nullfunc->addr2, sdata->vif.addr, ETH_ALEN);
	memcpy(nullfunc->addr3, sdata->u.mgd.bssid, ETH_ALEN);
	memcpy(nullfunc->addr4, sdata->vif.addr, ETH_ALEN);

	IEEE80211_SKB_CB(skb)->flags |= IEEE80211_TX_INTFL_DONT_ENCRYPT;
	klpr_ieee80211_tx_skb(sdata, skb);
}

static void klpr_ieee80211_chswitch_post_beacon(struct ieee80211_sub_if_data *sdata)
{
	struct ieee80211_local *local = sdata->local;
	struct ieee80211_if_managed *ifmgd = &sdata->u.mgd;
	int ret;

	sdata_assert_lock(sdata);

	WARN_ON(!sdata->vif.csa_active);

	if (sdata->csa_block_tx) {
		(*klpe_ieee80211_wake_vif_queues)(local, sdata,
					  IEEE80211_QUEUE_STOP_REASON_CSA);
		sdata->csa_block_tx = false;
	}

	sdata->vif.csa_active = false;
	ifmgd->csa_waiting_bcn = false;
	/*
	 * If the CSA IE is still present on the beacon after the switch,
	 * we need to consider it as a new CSA (possibly to self).
	 */
	ifmgd->beacon_crc_valid = false;

	ret = klpr_drv_post_channel_switch(sdata);
	if (ret) {
		sdata_info(sdata,
			   "driver post channel switch failed, disconnecting\n");
		(*klpe_ieee80211_queue_work)(&local->hw,
				     &ifmgd->csa_connection_drop_work);
		return;
	}

	(*klpe_cfg80211_ch_switch_notify)(sdata->dev, &sdata->reserved_chandef);
}

static void
(*klpe_ieee80211_sta_process_chanswitch)(struct ieee80211_sub_if_data *sdata,
				 u64 timestamp, u32 device_timestamp,
				 struct ieee802_11_elems *elems,
				 bool beacon);

static bool
klpr_ieee80211_find_80211h_pwr_constr(struct ieee80211_sub_if_data *sdata,
				 struct ieee80211_channel *channel,
				 const u8 *country_ie, u8 country_ie_len,
				 const u8 *pwr_constr_elem,
				 int *chan_pwr, int *pwr_reduction)
{
	struct ieee80211_country_ie_triplet *triplet;
	int chan = klpr_ieee80211_frequency_to_channel(channel->center_freq);
	int i, chan_increment;
	bool have_chan_pwr = false;

	/* Invalid IE */
	if (country_ie_len % 2 || country_ie_len < IEEE80211_COUNTRY_IE_MIN_LEN)
		return false;

	triplet = (void *)(country_ie + 3);
	country_ie_len -= 3;

	switch (channel->band) {
	default:
		WARN_ON_ONCE(1);
		fallthrough;
	case NL80211_BAND_2GHZ:
	case NL80211_BAND_60GHZ:
		chan_increment = 1;
		break;
	case NL80211_BAND_5GHZ:
	case NL80211_BAND_6GHZ:
		chan_increment = 4;
		break;
	}

	/* find channel */
	while (country_ie_len >= 3) {
		u8 first_channel = triplet->chans.first_channel;

		if (first_channel >= IEEE80211_COUNTRY_EXTENSION_ID)
			goto next;

		for (i = 0; i < triplet->chans.num_channels; i++) {
			if (first_channel + i * chan_increment == chan) {
				have_chan_pwr = true;
				*chan_pwr = triplet->chans.max_power;
				break;
			}
		}
		if (have_chan_pwr)
			break;

 next:
		triplet++;
		country_ie_len -= 3;
	}

	if (have_chan_pwr && pwr_constr_elem)
		*pwr_reduction = *pwr_constr_elem;
	else
		*pwr_reduction = 0;

	return have_chan_pwr;
}

static void ieee80211_find_cisco_dtpc(struct ieee80211_sub_if_data *sdata,
				      struct ieee80211_channel *channel,
				      const u8 *cisco_dtpc_ie,
				      int *pwr_level)
{
	/* From practical testing, the first data byte of the DTPC element
	 * seems to contain the requested dBm level, and the CLI on Cisco
	 * APs clearly state the range is -127 to 127 dBm, which indicates
	 * a signed byte, although it seemingly never actually goes negative.
	 * The other byte seems to always be zero.
	 */
	*pwr_level = (__s8)cisco_dtpc_ie[4];
}

static u32 klpr_ieee80211_handle_pwr_constr(struct ieee80211_sub_if_data *sdata,
				       struct ieee80211_channel *channel,
				       struct ieee80211_mgmt *mgmt,
				       const u8 *country_ie, u8 country_ie_len,
				       const u8 *pwr_constr_ie,
				       const u8 *cisco_dtpc_ie)
{
	bool has_80211h_pwr = false, has_cisco_pwr = false;
	int chan_pwr = 0, pwr_reduction_80211h = 0;
	int pwr_level_cisco, pwr_level_80211h;
	int new_ap_level;
	__le16 capab = mgmt->u.probe_resp.capab_info;

	if (country_ie &&
	    (capab & cpu_to_le16(WLAN_CAPABILITY_SPECTRUM_MGMT) ||
	     capab & cpu_to_le16(WLAN_CAPABILITY_RADIO_MEASURE))) {
		has_80211h_pwr = klpr_ieee80211_find_80211h_pwr_constr(
			sdata, channel, country_ie, country_ie_len,
			pwr_constr_ie, &chan_pwr, &pwr_reduction_80211h);
		pwr_level_80211h =
			max_t(int, 0, chan_pwr - pwr_reduction_80211h);
	}

	if (cisco_dtpc_ie) {
		ieee80211_find_cisco_dtpc(
			sdata, channel, cisco_dtpc_ie, &pwr_level_cisco);
		has_cisco_pwr = true;
	}

	if (!has_80211h_pwr && !has_cisco_pwr)
		return 0;

	/* If we have both 802.11h and Cisco DTPC, apply both limits
	 * by picking the smallest of the two power levels advertised.
	 */
	if (has_80211h_pwr &&
	    (!has_cisco_pwr || pwr_level_80211h <= pwr_level_cisco)) {
		new_ap_level = pwr_level_80211h;

		if (sdata->ap_power_level == new_ap_level)
			return 0;

		sdata_dbg(sdata,
			  "Limiting TX power to %d (%d - %d) dBm as advertised by %pM\n",
			  pwr_level_80211h, chan_pwr, pwr_reduction_80211h,
			  sdata->u.mgd.bssid);
	} else {  /* has_cisco_pwr is always true here. */
		new_ap_level = pwr_level_cisco;

		if (sdata->ap_power_level == new_ap_level)
			return 0;

		sdata_dbg(sdata,
			  "Limiting TX power to %d dBm as advertised by %pM\n",
			  pwr_level_cisco, sdata->u.mgd.bssid);
	}

	sdata->ap_power_level = new_ap_level;
	if ((*klpe___ieee80211_recalc_txpower)(sdata))
		return BSS_CHANGED_TXPOWER;
	return 0;
}

static bool
(*klpe_ieee80211_sta_wmm_params)(struct ieee80211_local *local,
			 struct ieee80211_sub_if_data *sdata,
			 const u8 *wmm_param, size_t wmm_param_len,
			 const struct ieee80211_mu_edca_param_set *mu_edca);

static void (*klpe_ieee80211_stop_poll)(struct ieee80211_sub_if_data *sdata);

static u32 (*klpe_ieee80211_handle_bss_capability)(struct ieee80211_sub_if_data *sdata,
					   u16 capab, bool erp_valid, u8 erp);

static void klpr_ieee80211_set_associated(struct ieee80211_sub_if_data *sdata,
				     struct cfg80211_bss *cbss,
				     u32 bss_info_changed)
{
	struct ieee80211_bss *bss = (void *)cbss->priv;
	struct ieee80211_local *local = sdata->local;
	struct ieee80211_bss_conf *bss_conf = &sdata->vif.bss_conf;

	bss_info_changed |= BSS_CHANGED_ASSOC;
	bss_info_changed |= (*klpe_ieee80211_handle_bss_capability)(sdata,
		bss_conf->assoc_capability, bss->has_erp_value, bss->erp_value);

	sdata->u.mgd.beacon_timeout = usecs_to_jiffies(ieee80211_tu_to_usec(
		(*klpe_beacon_loss_count) * bss_conf->beacon_int));

	sdata->u.mgd.associated = cbss;
	memcpy(sdata->u.mgd.bssid, cbss->bssid, ETH_ALEN);

	(*klpe_ieee80211_check_rate_mask)(sdata);

	sdata->u.mgd.flags |= IEEE80211_STA_RESET_SIGNAL_AVE;

	if (sdata->vif.p2p ||
	    sdata->vif.driver_flags & IEEE80211_VIF_GET_NOA_UPDATE) {
		const struct cfg80211_bss_ies *ies;

		rcu_read_lock();
		ies = rcu_dereference(cbss->ies);
		if (ies) {
			int ret;

			ret = (*klpe_cfg80211_get_p2p_attr)(
					ies->data, ies->len,
					IEEE80211_P2P_ATTR_ABSENCE_NOTICE,
					(u8 *) &bss_conf->p2p_noa_attr,
					sizeof(bss_conf->p2p_noa_attr));
			if (ret >= 2) {
				sdata->u.mgd.p2p_noa_index =
					bss_conf->p2p_noa_attr.index;
				bss_info_changed |= BSS_CHANGED_P2P_PS;
			}
		}
		rcu_read_unlock();
	}

	/* just to be sure */
	(*klpe_ieee80211_stop_poll)(sdata);

	(*klpe_ieee80211_led_assoc)(local, 1);

	if (sdata->u.mgd.have_beacon) {
		/*
		 * If the AP is buggy we may get here with no DTIM period
		 * known, so assume it's 1 which is the only safe assumption
		 * in that case, although if the TIM IE is broken powersave
		 * probably just won't work at all.
		 */
		bss_conf->dtim_period = sdata->u.mgd.dtim_period ?: 1;
		bss_conf->beacon_rate = bss->beacon_rate;
		bss_info_changed |= BSS_CHANGED_BEACON_INFO;
	} else {
		bss_conf->beacon_rate = NULL;
		bss_conf->dtim_period = 0;
	}

	bss_conf->assoc = 1;

	/* Tell the driver to monitor connection quality (if supported) */
	if (sdata->vif.driver_flags & IEEE80211_VIF_SUPPORTS_CQM_RSSI &&
	    bss_conf->cqm_rssi_thold)
		bss_info_changed |= BSS_CHANGED_CQM;

	/* Enable ARP filtering */
	if (bss_conf->arp_addr_cnt)
		bss_info_changed |= BSS_CHANGED_ARP_FILTER;

	(*klpe_ieee80211_bss_info_change_notify)(sdata, bss_info_changed);

	mutex_lock(&local->iflist_mtx);
	(*klpe_ieee80211_recalc_ps)(local);
	mutex_unlock(&local->iflist_mtx);

	(*klpe_ieee80211_recalc_smps)(sdata);
	(*klpe_ieee80211_recalc_ps_vif)(sdata);

	netif_carrier_on(sdata->dev);
}

static void (*klpe_ieee80211_set_disassoc)(struct ieee80211_sub_if_data *sdata,
				   u16 stype, u16 reason, bool tx,
				   u8 *frame_buf);

static void (*klpe_ieee80211_reset_ap_probe)(struct ieee80211_sub_if_data *sdata);

static void (*klpe_ieee80211_report_disconnect)(struct ieee80211_sub_if_data *sdata,
					const u8 *buf, size_t len, bool tx,
					u16 reason);

static void (*klpe_ieee80211_destroy_auth_data)(struct ieee80211_sub_if_data *sdata,
					bool assoc);

static void klpr_ieee80211_destroy_assoc_data(struct ieee80211_sub_if_data *sdata,
					 bool assoc, bool abandon)
{
	struct ieee80211_mgd_assoc_data *assoc_data = sdata->u.mgd.assoc_data;

	sdata_assert_lock(sdata);

	if (!assoc) {
		/*
		 * we are not associated yet, the only timer that could be
		 * running is the timeout for the association response which
		 * which is not relevant anymore.
		 */
		del_timer_sync(&sdata->u.mgd.timer);
		(*klpe_sta_info_destroy_addr)(sdata, assoc_data->bss->bssid);

		eth_zero_addr(sdata->u.mgd.bssid);
		(*klpe_ieee80211_bss_info_change_notify)(sdata, BSS_CHANGED_BSSID);
		sdata->u.mgd.flags = 0;
		sdata->vif.mu_mimo_owner = false;

		mutex_lock(&sdata->local->mtx);
		(*klpe_ieee80211_vif_release_channel)(sdata);
		mutex_unlock(&sdata->local->mtx);

		if (abandon)
			(*klpe_cfg80211_abandon_assoc)(sdata->dev, assoc_data->bss);
	}

	kfree(assoc_data);
	sdata->u.mgd.assoc_data = NULL;
}

static void klpp_ieee80211_auth_challenge(struct ieee80211_sub_if_data *sdata,
				     struct ieee80211_mgmt *mgmt, size_t len)
{
	struct ieee80211_local *local = sdata->local;
	struct ieee80211_mgd_auth_data *auth_data = sdata->u.mgd.auth_data;
	/*
	 * Fix CVE-2022-42719
	 *  +1 line
	 */
	const struct element *challenge;
	u8 *pos;
	/*
	 * Fix CVE-2022-42719
	 *  -1 line
	 */

	u32 tx_flags = 0;

	pos = mgmt->u.auth.variable;
	/*
	 * Fix CVE-2022-42719
	 *  -3 lines, +3 lines
	 */
	challenge = klpr_cfg80211_find_elem(WLAN_EID_CHALLENGE, pos,
				       len - (pos - (u8 *)mgmt));
	if (!challenge)
		return;
	auth_data->expected_transaction = 4;
	(*klpe_drv_mgd_prepare_tx)(sdata->local, sdata, 0);
	if (ieee80211_hw_check(&local->hw, REPORTS_TX_ACK_STATUS))
		tx_flags = IEEE80211_TX_CTL_REQ_TX_STATUS |
			   IEEE80211_TX_INTFL_MLME_CONN_TX;
	(*klpe_ieee80211_send_auth)(sdata, 3, auth_data->algorithm, 0,
			    /*
			     * Fix CVE-2022-42719
			     *  -1 line, +2 lines
			     */
			    (void *)challenge,
			    challenge->datalen + sizeof(*challenge),
			    auth_data->bss->bssid, auth_data->bss->bssid,
			    auth_data->key, auth_data->key_len,
			    auth_data->key_idx, tx_flags);
}

static bool (*klpe_ieee80211_mark_sta_auth)(struct ieee80211_sub_if_data *sdata,
				    const u8 *bssid);

static void klpr_ieee80211_rx_mgmt_auth(struct ieee80211_sub_if_data *sdata,
				   struct ieee80211_mgmt *mgmt, size_t len)
{
	struct ieee80211_if_managed *ifmgd = &sdata->u.mgd;
	u8 bssid[ETH_ALEN];
	u16 auth_alg, auth_transaction, status_code;
	struct ieee80211_event event = {
		.type = MLME_EVENT,
		.u.mlme.data = AUTH_EVENT,
	};

	sdata_assert_lock(sdata);

	if (len < 24 + 6)
		return;

	if (!ifmgd->auth_data || ifmgd->auth_data->done)
		return;

	memcpy(bssid, ifmgd->auth_data->bss->bssid, ETH_ALEN);

	if (!ether_addr_equal(bssid, mgmt->bssid))
		return;

	auth_alg = le16_to_cpu(mgmt->u.auth.auth_alg);
	auth_transaction = le16_to_cpu(mgmt->u.auth.auth_transaction);
	status_code = le16_to_cpu(mgmt->u.auth.status_code);

	if (auth_alg != ifmgd->auth_data->algorithm ||
	    (auth_alg != WLAN_AUTH_SAE &&
	     auth_transaction != ifmgd->auth_data->expected_transaction) ||
	    (auth_alg == WLAN_AUTH_SAE &&
	     (auth_transaction < ifmgd->auth_data->expected_transaction ||
	      auth_transaction > 2))) {
		sdata_info(sdata, "%pM unexpected authentication state: alg %d (expected %d) transact %d (expected %d)\n",
			   mgmt->sa, auth_alg, ifmgd->auth_data->algorithm,
			   auth_transaction,
			   ifmgd->auth_data->expected_transaction);
		return;
	}

	if (status_code != WLAN_STATUS_SUCCESS) {
		(*klpe_cfg80211_rx_mlme_mgmt)(sdata->dev, (u8 *)mgmt, len);

		if (auth_alg == WLAN_AUTH_SAE &&
		    (status_code == WLAN_STATUS_ANTI_CLOG_REQUIRED ||
		     (auth_transaction == 1 &&
		      (status_code == WLAN_STATUS_SAE_HASH_TO_ELEMENT ||
		       status_code == WLAN_STATUS_SAE_PK))))
			return;

		sdata_info(sdata, "%pM denied authentication (status %d)\n",
			   mgmt->sa, status_code);
		(*klpe_ieee80211_destroy_auth_data)(sdata, false);
		event.u.mlme.status = MLME_DENIED;
		event.u.mlme.reason = status_code;
		(*klpe_drv_event_callback)(sdata->local, sdata, &event);
		return;
	}

	switch (ifmgd->auth_data->algorithm) {
	case WLAN_AUTH_OPEN:
	case WLAN_AUTH_LEAP:
	case WLAN_AUTH_FT:
	case WLAN_AUTH_SAE:
	case WLAN_AUTH_FILS_SK:
	case WLAN_AUTH_FILS_SK_PFS:
	case WLAN_AUTH_FILS_PK:
		break;
	case WLAN_AUTH_SHARED_KEY:
		if (ifmgd->auth_data->expected_transaction != 4) {
			klpp_ieee80211_auth_challenge(sdata, mgmt, len);
			/* need another frame */
			return;
		}
		break;
	default:
		WARN_ONCE(1, "invalid auth alg %d",
			  ifmgd->auth_data->algorithm);
		return;
	}

	event.u.mlme.status = MLME_SUCCESS;
	(*klpe_drv_event_callback)(sdata->local, sdata, &event);
	if (ifmgd->auth_data->algorithm != WLAN_AUTH_SAE ||
	    (auth_transaction == 2 &&
	     ifmgd->auth_data->expected_transaction == 2)) {
		if (!(*klpe_ieee80211_mark_sta_auth)(sdata, bssid))
			return; /* ignore frame -- wait for timeout */
	} else if (ifmgd->auth_data->algorithm == WLAN_AUTH_SAE &&
		   auth_transaction == 2) {
		sdata_info(sdata, "SAE peer confirmed\n");
		ifmgd->auth_data->peer_confirmed = true;
	}

	(*klpe_cfg80211_rx_mlme_mgmt)(sdata->dev, (u8 *)mgmt, len);
}

static void klpr_ieee80211_rx_mgmt_deauth(struct ieee80211_sub_if_data *sdata,
				     struct ieee80211_mgmt *mgmt, size_t len)
{
	struct ieee80211_if_managed *ifmgd = &sdata->u.mgd;
	u16 reason_code = le16_to_cpu(mgmt->u.deauth.reason_code);

	sdata_assert_lock(sdata);

	if (len < 24 + 2)
		return;

	if (!ether_addr_equal(mgmt->bssid, mgmt->sa)) {
		(*klpe_ieee80211_tdls_handle_disconnect)(sdata, mgmt->sa, reason_code);
		return;
	}

	if (ifmgd->associated &&
	    ether_addr_equal(mgmt->bssid, ifmgd->associated->bssid)) {
		const u8 *bssid = ifmgd->associated->bssid;

		sdata_info(sdata, "deauthenticated from %pM (Reason: %u=%s)\n",
			   bssid, reason_code,
			   (*klpe_ieee80211_get_reason_code_string)(reason_code));

		(*klpe_ieee80211_set_disassoc)(sdata, 0, 0, false, NULL);

		(*klpe_ieee80211_report_disconnect)(sdata, (u8 *)mgmt, len, false,
					    reason_code);
		return;
	}

	if (ifmgd->assoc_data &&
	    ether_addr_equal(mgmt->bssid, ifmgd->assoc_data->bss->bssid)) {
		const u8 *bssid = ifmgd->assoc_data->bss->bssid;

		sdata_info(sdata,
			   "deauthenticated from %pM while associating (Reason: %u=%s)\n",
			   bssid, reason_code,
			   (*klpe_ieee80211_get_reason_code_string)(reason_code));

		klpr_ieee80211_destroy_assoc_data(sdata, false, true);

		(*klpe_cfg80211_rx_mlme_mgmt)(sdata->dev, (u8 *)mgmt, len);
		return;
	}
}

static void klpr_ieee80211_rx_mgmt_disassoc(struct ieee80211_sub_if_data *sdata,
				       struct ieee80211_mgmt *mgmt, size_t len)
{
	struct ieee80211_if_managed *ifmgd = &sdata->u.mgd;
	u16 reason_code;

	sdata_assert_lock(sdata);

	if (len < 24 + 2)
		return;

	if (!ifmgd->associated ||
	    !ether_addr_equal(mgmt->bssid, ifmgd->associated->bssid))
		return;

	reason_code = le16_to_cpu(mgmt->u.disassoc.reason_code);

	if (!ether_addr_equal(mgmt->bssid, mgmt->sa)) {
		(*klpe_ieee80211_tdls_handle_disconnect)(sdata, mgmt->sa, reason_code);
		return;
	}

	sdata_info(sdata, "disassociated from %pM (Reason: %u=%s)\n",
		   mgmt->sa, reason_code,
		   (*klpe_ieee80211_get_reason_code_string)(reason_code));

	(*klpe_ieee80211_set_disassoc)(sdata, 0, 0, false, NULL);

	(*klpe_ieee80211_report_disconnect)(sdata, (u8 *)mgmt, len, false, reason_code);
}

static bool ieee80211_twt_req_supported(const struct sta_info *sta,
					const struct ieee802_11_elems *elems)
{
	if (elems->ext_capab_len < 10)
		return false;

	if (!(elems->ext_capab[9] & WLAN_EXT_CAPA10_TWT_RESPONDER_SUPPORT))
		return false;

	return sta->sta.he_cap.he_cap_elem.mac_cap_info[0] &
		IEEE80211_HE_MAC_CAP0_TWT_RES;
}

static int ieee80211_recalc_twt_req(struct ieee80211_sub_if_data *sdata,
				    struct sta_info *sta,
				    struct ieee802_11_elems *elems)
{
	bool twt = ieee80211_twt_req_supported(sta, elems);

	if (sdata->vif.bss_conf.twt_requester != twt) {
		sdata->vif.bss_conf.twt_requester = twt;
		return BSS_CHANGED_TWT;
	}
	return 0;
}

static bool klpp_ieee80211_assoc_success(struct ieee80211_sub_if_data *sdata,
				    struct cfg80211_bss *cbss,
				    struct ieee80211_mgmt *mgmt, size_t len,
				    struct ieee802_11_elems *elems)
{
	struct ieee80211_if_managed *ifmgd = &sdata->u.mgd;
	struct ieee80211_local *local = sdata->local;
	struct ieee80211_supported_band *sband;
	struct sta_info *sta;
	u16 capab_info, aid;
	struct ieee80211_bss_conf *bss_conf = &sdata->vif.bss_conf;
	const struct cfg80211_bss_ies *bss_ies = NULL;
	struct ieee80211_mgd_assoc_data *assoc_data = ifmgd->assoc_data;
	bool is_6ghz = cbss->channel->band == NL80211_BAND_6GHZ;
	u32 changed = 0;
	int err;
	bool ret;

	/* AssocResp and ReassocResp have identical structure */

	aid = le16_to_cpu(mgmt->u.assoc_resp.aid);
	capab_info = le16_to_cpu(mgmt->u.assoc_resp.capab_info);

	/*
	 * The 5 MSB of the AID field are reserved
	 * (802.11-2016 9.4.1.8 AID field)
	 */
	aid &= 0x7ff;

	ifmgd->broken_ap = false;

	if (aid == 0 || aid > IEEE80211_MAX_AID) {
		sdata_info(sdata, "invalid AID value %d (out of range), turn off PS\n",
			   aid);
		aid = 0;
		ifmgd->broken_ap = true;
	}

	if (!elems->supp_rates) {
		sdata_info(sdata, "no SuppRates element in AssocResp\n");
		return false;
	}

	sdata->vif.bss_conf.aid = aid;
	ifmgd->tdls_chan_switch_prohibited =
		elems->ext_capab && elems->ext_capab_len >= 5 &&
		(elems->ext_capab[4] & WLAN_EXT_CAPA5_TDLS_CH_SW_PROHIBITED);

	/*
	 * Some APs are erroneously not including some information in their
	 * (re)association response frames. Try to recover by using the data
	 * from the beacon or probe response. This seems to afflict mobile
	 * 2G/3G/4G wifi routers, reported models include the "Onda PN51T",
	 * "Vodafone PocketWiFi 2", "ZTE MF60" and a similar T-Mobile device.
	 */
	if (!is_6ghz &&
	    ((assoc_data->wmm && !elems->wmm_param) ||
	     (!(ifmgd->flags & IEEE80211_STA_DISABLE_HT) &&
	      (!elems->ht_cap_elem || !elems->ht_operation)) ||
	     (!(ifmgd->flags & IEEE80211_STA_DISABLE_VHT) &&
	      (!elems->vht_cap_elem || !elems->vht_operation)))) {
		const struct cfg80211_bss_ies *ies;
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		struct ieee802_11_elems *bss_elems;

		rcu_read_lock();
		ies = rcu_dereference(cbss->ies);
		if (ies)
			bss_ies = kmemdup(ies, sizeof(*ies) + ies->len,
					  GFP_ATOMIC);
		rcu_read_unlock();
		/*
		 * Fix CVE-2022-42719
		 *  -2 lines, +4 lines
		 */
		if (!bss_ies) {
			ret = false;
			goto out;
		}

		/*
		 * Fix CVE-2022-42719
		 *  -4 lines, +8 lines
		 */
		bss_elems = klpp_ieee802_11_parse_elems(bss_ies->data, bss_ies->len,
						   false, mgmt->bssid,
						   assoc_data->bss->bssid);
		if (!bss_elems) {
			ret = false;
			goto out;
		}
#define bss_elems (*bss_elems)

		if (assoc_data->wmm &&
		    !elems->wmm_param && bss_elems.wmm_param) {
			elems->wmm_param = bss_elems.wmm_param;
			sdata_info(sdata,
				   "AP bug: WMM param missing from AssocResp\n");
		}

		/*
		 * Also check if we requested HT/VHT, otherwise the AP doesn't
		 * have to include the IEs in the (re)association response.
		 */
		if (!elems->ht_cap_elem && bss_elems.ht_cap_elem &&
		    !(ifmgd->flags & IEEE80211_STA_DISABLE_HT)) {
			elems->ht_cap_elem = bss_elems.ht_cap_elem;
			sdata_info(sdata,
				   "AP bug: HT capability missing from AssocResp\n");
		}
		if (!elems->ht_operation && bss_elems.ht_operation &&
		    !(ifmgd->flags & IEEE80211_STA_DISABLE_HT)) {
			elems->ht_operation = bss_elems.ht_operation;
			sdata_info(sdata,
				   "AP bug: HT operation missing from AssocResp\n");
		}
		if (!elems->vht_cap_elem && bss_elems.vht_cap_elem &&
		    !(ifmgd->flags & IEEE80211_STA_DISABLE_VHT)) {
			elems->vht_cap_elem = bss_elems.vht_cap_elem;
			sdata_info(sdata,
				   "AP bug: VHT capa missing from AssocResp\n");
		}
		if (!elems->vht_operation && bss_elems.vht_operation &&
		    !(ifmgd->flags & IEEE80211_STA_DISABLE_VHT)) {
			elems->vht_operation = bss_elems.vht_operation;
			sdata_info(sdata,
				   "AP bug: VHT operation missing from AssocResp\n");
		}
/*
 * Fix CVE-2022-42719
 *  +1 line
 */
#undef bss_elems
	}

	/*
	 * We previously checked these in the beacon/probe response, so
	 * they should be present here. This is just a safety net.
	 */
	if (!is_6ghz && !(ifmgd->flags & IEEE80211_STA_DISABLE_HT) &&
	    (!elems->wmm_param || !elems->ht_cap_elem || !elems->ht_operation)) {
		sdata_info(sdata,
			   "HT AP is missing WMM params or HT capability/operation\n");
		ret = false;
		goto out;
	}

	if (!is_6ghz && !(ifmgd->flags & IEEE80211_STA_DISABLE_VHT) &&
	    (!elems->vht_cap_elem || !elems->vht_operation)) {
		sdata_info(sdata,
			   "VHT AP is missing VHT capability/operation\n");
		ret = false;
		goto out;
	}

	if (is_6ghz && !(ifmgd->flags & IEEE80211_STA_DISABLE_HE) &&
	    !elems->he_6ghz_capa) {
		sdata_info(sdata,
			   "HE 6 GHz AP is missing HE 6 GHz band capability\n");
		ret = false;
		goto out;
	}

	mutex_lock(&sdata->local->sta_mtx);
	/*
	 * station info was already allocated and inserted before
	 * the association and should be available to us
	 */
	sta = (*klpe_sta_info_get)(sdata, cbss->bssid);
	if (WARN_ON(!sta)) {
		mutex_unlock(&sdata->local->sta_mtx);
		ret = false;
		goto out;
	}

	sband = ieee80211_get_sband(sdata);
	if (!sband) {
		mutex_unlock(&sdata->local->sta_mtx);
		ret = false;
		goto out;
	}

	if (!(ifmgd->flags & IEEE80211_STA_DISABLE_HE) &&
	    (!elems->he_cap || !elems->he_operation)) {
		mutex_unlock(&sdata->local->sta_mtx);
		sdata_info(sdata,
			   "HE AP is missing HE capability/operation\n");
		ret = false;
		goto out;
	}

	/* Set up internal HT/VHT capabilities */
	if (elems->ht_cap_elem && !(ifmgd->flags & IEEE80211_STA_DISABLE_HT))
		(*klpe_ieee80211_ht_cap_ie_to_sta_ht_cap)(sdata, sband,
						  elems->ht_cap_elem, sta);

	if (elems->vht_cap_elem && !(ifmgd->flags & IEEE80211_STA_DISABLE_VHT))
		(*klpe_ieee80211_vht_cap_ie_to_sta_vht_cap)(sdata, sband,
						    elems->vht_cap_elem, sta);

	if (elems->he_operation && !(ifmgd->flags & IEEE80211_STA_DISABLE_HE) &&
	    elems->he_cap) {
		(*klpe_ieee80211_he_cap_ie_to_sta_he_cap)(sdata, sband,
						  elems->he_cap,
						  elems->he_cap_len,
						  elems->he_6ghz_capa,
						  sta);

		bss_conf->he_support = sta->sta.he_cap.has_he;
		if (elems->rsnx && elems->rsnx_len &&
		    (elems->rsnx[0] & WLAN_RSNX_CAPA_PROTECTED_TWT) &&
		    wiphy_ext_feature_isset(local->hw.wiphy,
					    NL80211_EXT_FEATURE_PROTECTED_TWT))
			bss_conf->twt_protected = true;
		else
			bss_conf->twt_protected = false;

		changed |= ieee80211_recalc_twt_req(sdata, sta, elems);
	} else {
		bss_conf->he_support = false;
		bss_conf->twt_requester = false;
		bss_conf->twt_protected = false;
	}

	if (bss_conf->he_support) {
		bss_conf->he_bss_color.color =
			le32_get_bits(elems->he_operation->he_oper_params,
				      IEEE80211_HE_OPERATION_BSS_COLOR_MASK);
		bss_conf->he_bss_color.partial =
			le32_get_bits(elems->he_operation->he_oper_params,
				      IEEE80211_HE_OPERATION_PARTIAL_BSS_COLOR);
		bss_conf->he_bss_color.enabled =
			!le32_get_bits(elems->he_operation->he_oper_params,
				       IEEE80211_HE_OPERATION_BSS_COLOR_DISABLED);

		if (bss_conf->he_bss_color.enabled)
			changed |= BSS_CHANGED_HE_BSS_COLOR;

		bss_conf->htc_trig_based_pkt_ext =
			le32_get_bits(elems->he_operation->he_oper_params,
			      IEEE80211_HE_OPERATION_DFLT_PE_DURATION_MASK);
		bss_conf->frame_time_rts_th =
			le32_get_bits(elems->he_operation->he_oper_params,
			      IEEE80211_HE_OPERATION_RTS_THRESHOLD_MASK);

		bss_conf->multi_sta_back_32bit =
			sta->sta.he_cap.he_cap_elem.mac_cap_info[2] &
			IEEE80211_HE_MAC_CAP2_32BIT_BA_BITMAP;

		bss_conf->ack_enabled =
			sta->sta.he_cap.he_cap_elem.mac_cap_info[2] &
			IEEE80211_HE_MAC_CAP2_ACK_EN;

		bss_conf->uora_exists = !!elems->uora_element;
		if (elems->uora_element)
			bss_conf->uora_ocw_range = elems->uora_element[0];

		(*klpe_ieee80211_he_op_ie_to_bss_conf)(&sdata->vif, elems->he_operation);
		(*klpe_ieee80211_he_spr_ie_to_bss_conf)(&sdata->vif, elems->he_spr);
		/* TODO: OPEN: what happens if BSS color disable is set? */
	}

	if (cbss->transmitted_bss) {
		bss_conf->nontransmitted = true;
		ether_addr_copy(bss_conf->transmitter_bssid,
				cbss->transmitted_bss->bssid);
		bss_conf->bssid_indicator = cbss->max_bssid_indicator;
		bss_conf->bssid_index = cbss->bssid_index;
	}

	/*
	 * Some APs, e.g. Netgear WNDR3700, report invalid HT operation data
	 * in their association response, so ignore that data for our own
	 * configuration. If it changed since the last beacon, we'll get the
	 * next beacon and update then.
	 */

	/*
	 * If an operating mode notification IE is present, override the
	 * NSS calculation (that would be done in rate_control_rate_init())
	 * and use the # of streams from that element.
	 */
	if (elems->opmode_notif &&
	    !(*elems->opmode_notif & IEEE80211_OPMODE_NOTIF_RX_NSS_TYPE_BF)) {
		u8 nss;

		nss = *elems->opmode_notif & IEEE80211_OPMODE_NOTIF_RX_NSS_MASK;
		nss >>= IEEE80211_OPMODE_NOTIF_RX_NSS_SHIFT;
		nss += 1;
		sta->sta.rx_nss = nss;
	}

	(*klpe_rate_control_rate_init)(sta);

	if (ifmgd->flags & IEEE80211_STA_MFP_ENABLED) {
		set_sta_flag(sta, WLAN_STA_MFP);
		sta->sta.mfp = true;
	} else {
		sta->sta.mfp = false;
	}

	sta->sta.wme = elems->wmm_param && local->hw.queues >= IEEE80211_NUM_ACS;

	err = (*klpe_sta_info_move_state)(sta, IEEE80211_STA_ASSOC);
	if (!err && !(ifmgd->flags & IEEE80211_STA_CONTROL_PORT))
		err = (*klpe_sta_info_move_state)(sta, IEEE80211_STA_AUTHORIZED);
	if (err) {
		sdata_info(sdata,
			   "failed to move station %pM to desired state\n",
			   sta->sta.addr);
		WARN_ON((*klpe___sta_info_destroy)(sta));
		mutex_unlock(&sdata->local->sta_mtx);
		ret = false;
		goto out;
	}

	mutex_unlock(&sdata->local->sta_mtx);

	/*
	 * Always handle WMM once after association regardless
	 * of the first value the AP uses. Setting -1 here has
	 * that effect because the AP values is an unsigned
	 * 4-bit value.
	 */
	ifmgd->wmm_last_param_set = -1;
	ifmgd->mu_edca_last_param_set = -1;

	if (ifmgd->flags & IEEE80211_STA_DISABLE_WMM) {
		(*klpe_ieee80211_set_wmm_default)(sdata, false, false);
	} else if (!(*klpe_ieee80211_sta_wmm_params)(local, sdata, elems->wmm_param,
					     elems->wmm_param_len,
					     elems->mu_edca_param_set)) {
		/* still enable QoS since we might have HT/VHT */
		(*klpe_ieee80211_set_wmm_default)(sdata, false, true);
		/* set the disable-WMM flag in this case to disable
		 * tracking WMM parameter changes in the beacon if
		 * the parameters weren't actually valid. Doing so
		 * avoids changing parameters very strangely when
		 * the AP is going back and forth between valid and
		 * invalid parameters.
		 */
		ifmgd->flags |= IEEE80211_STA_DISABLE_WMM;
	}
	changed |= BSS_CHANGED_QOS;

	if (elems->max_idle_period_ie) {
		bss_conf->max_idle_period =
			le16_to_cpu(elems->max_idle_period_ie->max_idle_period);
		bss_conf->protected_keep_alive =
			!!(elems->max_idle_period_ie->idle_options &
			   WLAN_IDLE_OPTIONS_PROTECTED_KEEP_ALIVE);
		changed |= BSS_CHANGED_KEEP_ALIVE;
	} else {
		bss_conf->max_idle_period = 0;
		bss_conf->protected_keep_alive = false;
	}

	/* set assoc capability (AID was already set earlier),
	 * ieee80211_set_associated() will tell the driver */
	bss_conf->assoc_capability = capab_info;
	klpr_ieee80211_set_associated(sdata, cbss, changed);

	/*
	 * If we're using 4-addr mode, let the AP know that we're
	 * doing so, so that it can create the STA VLAN on its side
	 */
	if (ifmgd->use_4addr)
		klpr_ieee80211_send_4addr_nullfunc(local, sdata);

	/*
	 * Start timer to probe the connection to the AP now.
	 * Also start the timer that will detect beacon loss.
	 */
	(*klpe_ieee80211_sta_rx_notify)(sdata, (struct ieee80211_hdr *)mgmt);
	(*klpe_ieee80211_sta_reset_beacon_monitor)(sdata);

	ret = true;
 out:
	kfree(bss_ies);
	return ret;
}

static void klpp_ieee80211_rx_mgmt_assoc_resp(struct ieee80211_sub_if_data *sdata,
					 struct ieee80211_mgmt *mgmt,
					 size_t len)
{
	struct ieee80211_if_managed *ifmgd = &sdata->u.mgd;
	struct ieee80211_mgd_assoc_data *assoc_data = ifmgd->assoc_data;
	u16 capab_info, status_code, aid;
	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +1 line
	 */
	struct ieee802_11_elems *elems;
	int ac, uapsd_queues = -1;
	u8 *pos;
	bool reassoc;
	struct cfg80211_bss *bss;
	struct ieee80211_event event = {
		.type = MLME_EVENT,
		.u.mlme.data = ASSOC_EVENT,
	};

	sdata_assert_lock(sdata);

	if (!assoc_data)
		return;
	if (!ether_addr_equal(assoc_data->bss->bssid, mgmt->bssid))
		return;

	/*
	 * AssocResp and ReassocResp have identical structure, so process both
	 * of them in this function.
	 */

	if (len < 24 + 6)
		return;

	reassoc = ieee80211_is_reassoc_resp(mgmt->frame_control);
	capab_info = le16_to_cpu(mgmt->u.assoc_resp.capab_info);
	status_code = le16_to_cpu(mgmt->u.assoc_resp.status_code);
	aid = le16_to_cpu(mgmt->u.assoc_resp.aid);

	sdata_info(sdata,
		   "RX %sssocResp from %pM (capab=0x%x status=%d aid=%d)\n",
		   reassoc ? "Rea" : "A", mgmt->sa,
		   capab_info, status_code, (u16)(aid & ~(BIT(15) | BIT(14))));

	if (assoc_data->fils_kek_len &&
	    (*klpe_fils_decrypt_assoc_resp)(sdata, (u8 *)mgmt, &len, assoc_data) < 0)
		return;

	pos = mgmt->u.assoc_resp.variable;
	/*
	 * Fix CVE-2022-42719
	 *  -2 lines, +6 lines
	 */
	elems = klpp_ieee802_11_parse_elems(pos, len - (pos - (u8 *)mgmt), false,
			       mgmt->bssid, assoc_data->bss->bssid);
	if (!elems)
		return;
#define elems (*elems)

	if (status_code == WLAN_STATUS_ASSOC_REJECTED_TEMPORARILY &&
	    elems.timeout_int &&
	    elems.timeout_int->type == WLAN_TIMEOUT_ASSOC_COMEBACK) {
		u32 tu, ms;
		tu = le32_to_cpu(elems.timeout_int->value);
		ms = tu * 1024 / 1000;
		sdata_info(sdata,
			   "%pM rejected association temporarily; comeback duration %u TU (%u ms)\n",
			   mgmt->sa, tu, ms);
		assoc_data->timeout = jiffies + msecs_to_jiffies(ms);
		assoc_data->timeout_started = true;
		if (ms > IEEE80211_ASSOC_TIMEOUT)
			run_again(sdata, assoc_data->timeout);
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		goto out;
	}

	bss = assoc_data->bss;

	if (status_code != WLAN_STATUS_SUCCESS) {
		sdata_info(sdata, "%pM denied association (code=%d)\n",
			   mgmt->sa, status_code);
		klpr_ieee80211_destroy_assoc_data(sdata, false, false);
		event.u.mlme.status = MLME_DENIED;
		event.u.mlme.reason = status_code;
		(*klpe_drv_event_callback)(sdata->local, sdata, &event);
	} else {
		if (!klpp_ieee80211_assoc_success(sdata, bss, mgmt, len, &elems)) {
			/* oops -- internal error -- send timeout for now */
			klpr_ieee80211_destroy_assoc_data(sdata, false, false);
			(*klpe_cfg80211_assoc_timeout)(sdata->dev, bss);
			/*
			 * Fix CVE-2022-42719
			 *  -1 line, +1 line
			 */
			goto out;
		}
		event.u.mlme.status = MLME_SUCCESS;
		(*klpe_drv_event_callback)(sdata->local, sdata, &event);
		sdata_info(sdata, "associated\n");

		/*
		 * destroy assoc_data afterwards, as otherwise an idle
		 * recalc after assoc_data is NULL but before associated
		 * is set can cause the interface to go idle
		 */
		klpr_ieee80211_destroy_assoc_data(sdata, true, false);

		/* get uapsd queues configuration */
		uapsd_queues = 0;
		for (ac = 0; ac < IEEE80211_NUM_ACS; ac++)
			if (sdata->tx_conf[ac].uapsd)
				uapsd_queues |= (*klpe_ieee80211_ac_to_qos_mask)[ac];
	}

	(*klpe_cfg80211_rx_assoc_resp)(sdata->dev, bss, (u8 *)mgmt, len, uapsd_queues,
			       ifmgd->assoc_req_ies, ifmgd->assoc_req_ies_len);
/*
 * Fix CVE-2022-42719
 *  +3 lines
 */
out:
#undef elems
	kfree(elems);
}

static void (*klpe_ieee80211_rx_bss_info)(struct ieee80211_sub_if_data *sdata,
				  struct ieee80211_mgmt *mgmt, size_t len,
				  struct ieee80211_rx_status *rx_status);

static void klpr_ieee80211_rx_mgmt_probe_resp(struct ieee80211_sub_if_data *sdata,
					 struct sk_buff *skb)
{
	struct ieee80211_mgmt *mgmt = (void *)skb->data;
	struct ieee80211_if_managed *ifmgd;
	struct ieee80211_rx_status *rx_status = (void *) skb->cb;
	struct ieee80211_channel *channel;
	size_t baselen, len = skb->len;

	ifmgd = &sdata->u.mgd;

	sdata_assert_lock(sdata);

	/*
	 * According to Draft P802.11ax D6.0 clause 26.17.2.3.2:
	 * "If a 6 GHz AP receives a Probe Request frame  and responds with
	 * a Probe Response frame [..], the Address 1 field of the Probe
	 * Response frame shall be set to the broadcast address [..]"
	 * So, on 6GHz band we should also accept broadcast responses.
	 */
	channel = klpr_ieee80211_get_channel(sdata->local->hw.wiphy,
					rx_status->freq);
	if (!channel)
		return;

	if (!ether_addr_equal(mgmt->da, sdata->vif.addr) &&
	    (channel->band != NL80211_BAND_6GHZ ||
	     !is_broadcast_ether_addr(mgmt->da)))
		return; /* ignore ProbeResp to foreign address */

	baselen = (u8 *) mgmt->u.probe_resp.variable - (u8 *) mgmt;
	if (baselen > len)
		return;

	(*klpe_ieee80211_rx_bss_info)(sdata, mgmt, len, rx_status);

	if (ifmgd->associated &&
	    ether_addr_equal(mgmt->bssid, ifmgd->associated->bssid))
		(*klpe_ieee80211_reset_ap_probe)(sdata);
}

static const u64 care_about_ies =
	(1ULL << WLAN_EID_COUNTRY) |
	(1ULL << WLAN_EID_ERP_INFO) |
	(1ULL << WLAN_EID_CHANNEL_SWITCH) |
	(1ULL << WLAN_EID_PWR_CONSTRAINT) |
	(1ULL << WLAN_EID_HT_CAPABILITY) |
	(1ULL << WLAN_EID_HT_OPERATION) |
	(1ULL << WLAN_EID_EXT_CHANSWITCH_ANN);

static void klpr_ieee80211_handle_beacon_sig(struct ieee80211_sub_if_data *sdata,
					struct ieee80211_if_managed *ifmgd,
					struct ieee80211_bss_conf *bss_conf,
					struct ieee80211_local *local,
					struct ieee80211_rx_status *rx_status)
{
	/* Track average RSSI from the Beacon frames of the current AP */

	if (ifmgd->flags & IEEE80211_STA_RESET_SIGNAL_AVE) {
		ifmgd->flags &= ~IEEE80211_STA_RESET_SIGNAL_AVE;
		ewma_beacon_signal_init(&ifmgd->ave_beacon_signal);
		ifmgd->last_cqm_event_signal = 0;
		ifmgd->count_beacon_signal = 1;
		ifmgd->last_ave_beacon_signal = 0;
	} else {
		ifmgd->count_beacon_signal++;
	}

	ewma_beacon_signal_add(&ifmgd->ave_beacon_signal, -rx_status->signal);

	if (ifmgd->rssi_min_thold != ifmgd->rssi_max_thold &&
	    ifmgd->count_beacon_signal >= IEEE80211_SIGNAL_AVE_MIN_COUNT) {
		int sig = -ewma_beacon_signal_read(&ifmgd->ave_beacon_signal);
		int last_sig = ifmgd->last_ave_beacon_signal;
		struct ieee80211_event event = {
			.type = RSSI_EVENT,
		};

		/*
		 * if signal crosses either of the boundaries, invoke callback
		 * with appropriate parameters
		 */
		if (sig > ifmgd->rssi_max_thold &&
		    (last_sig <= ifmgd->rssi_min_thold || last_sig == 0)) {
			ifmgd->last_ave_beacon_signal = sig;
			event.u.rssi.data = RSSI_EVENT_HIGH;
			(*klpe_drv_event_callback)(local, sdata, &event);
		} else if (sig < ifmgd->rssi_min_thold &&
			   (last_sig >= ifmgd->rssi_max_thold ||
			   last_sig == 0)) {
			ifmgd->last_ave_beacon_signal = sig;
			event.u.rssi.data = RSSI_EVENT_LOW;
			(*klpe_drv_event_callback)(local, sdata, &event);
		}
	}

	if (bss_conf->cqm_rssi_thold &&
	    ifmgd->count_beacon_signal >= IEEE80211_SIGNAL_AVE_MIN_COUNT &&
	    !(sdata->vif.driver_flags & IEEE80211_VIF_SUPPORTS_CQM_RSSI)) {
		int sig = -ewma_beacon_signal_read(&ifmgd->ave_beacon_signal);
		int last_event = ifmgd->last_cqm_event_signal;
		int thold = bss_conf->cqm_rssi_thold;
		int hyst = bss_conf->cqm_rssi_hyst;

		if (sig < thold &&
		    (last_event == 0 || sig < last_event - hyst)) {
			ifmgd->last_cqm_event_signal = sig;
			(*klpe_ieee80211_cqm_rssi_notify)(
				&sdata->vif,
				NL80211_CQM_RSSI_THRESHOLD_EVENT_LOW,
				sig, GFP_KERNEL);
		} else if (sig > thold &&
			   (last_event == 0 || sig > last_event + hyst)) {
			ifmgd->last_cqm_event_signal = sig;
			(*klpe_ieee80211_cqm_rssi_notify)(
				&sdata->vif,
				NL80211_CQM_RSSI_THRESHOLD_EVENT_HIGH,
				sig, GFP_KERNEL);
		}
	}

	if (bss_conf->cqm_rssi_low &&
	    ifmgd->count_beacon_signal >= IEEE80211_SIGNAL_AVE_MIN_COUNT) {
		int sig = -ewma_beacon_signal_read(&ifmgd->ave_beacon_signal);
		int last_event = ifmgd->last_cqm_event_signal;
		int low = bss_conf->cqm_rssi_low;
		int high = bss_conf->cqm_rssi_high;

		if (sig < low &&
		    (last_event == 0 || last_event >= low)) {
			ifmgd->last_cqm_event_signal = sig;
			(*klpe_ieee80211_cqm_rssi_notify)(
				&sdata->vif,
				NL80211_CQM_RSSI_THRESHOLD_EVENT_LOW,
				sig, GFP_KERNEL);
		} else if (sig > high &&
			   (last_event == 0 || last_event <= high)) {
			ifmgd->last_cqm_event_signal = sig;
			(*klpe_ieee80211_cqm_rssi_notify)(
				&sdata->vif,
				NL80211_CQM_RSSI_THRESHOLD_EVENT_HIGH,
				sig, GFP_KERNEL);
		}
	}
}

static bool ieee80211_rx_our_beacon(const u8 *tx_bssid,
				    struct cfg80211_bss *bss)
{
	if (ether_addr_equal(tx_bssid, bss->bssid))
		return true;
	if (!bss->transmitted_bss)
		return false;
	return ether_addr_equal(tx_bssid, bss->transmitted_bss->bssid);
}

static void klpp_ieee80211_rx_mgmt_beacon(struct ieee80211_sub_if_data *sdata,
				     struct ieee80211_mgmt *mgmt, size_t len,
				     struct ieee80211_rx_status *rx_status)
{
	struct ieee80211_if_managed *ifmgd = &sdata->u.mgd;
	struct ieee80211_bss_conf *bss_conf = &sdata->vif.bss_conf;
	size_t baselen;
	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +1 line
	 */
	struct ieee802_11_elems *elems;
	struct ieee80211_local *local = sdata->local;
	struct ieee80211_chanctx_conf *chanctx_conf;
	struct ieee80211_channel *chan;
	struct sta_info *sta;
	u32 changed = 0;
	bool erp_valid;
	u8 erp_value = 0;
	u32 ncrc;
	u8 *bssid;
	u8 deauth_buf[IEEE80211_DEAUTH_FRAME_LEN];

	sdata_assert_lock(sdata);

	/* Process beacon from the current BSS */
	baselen = (u8 *) mgmt->u.beacon.variable - (u8 *) mgmt;
	if (baselen > len)
		return;

	rcu_read_lock();
	chanctx_conf = rcu_dereference(sdata->vif.chanctx_conf);
	if (!chanctx_conf) {
		rcu_read_unlock();
		return;
	}

	if (ieee80211_rx_status_to_khz(rx_status) !=
	    ieee80211_channel_to_khz(chanctx_conf->def.chan)) {
		rcu_read_unlock();
		return;
	}
	chan = chanctx_conf->def.chan;
	rcu_read_unlock();

	if (ifmgd->assoc_data && ifmgd->assoc_data->need_beacon &&
	    ieee80211_rx_our_beacon(mgmt->bssid, ifmgd->assoc_data->bss)) {
		/*
		 * Fix CVE-2022-42719
		 *  -4 lines, +7 lines
		 */
		elems = klpp_ieee802_11_parse_elems(mgmt->u.beacon.variable,
					   len - baselen, false,
					   mgmt->bssid,
					   ifmgd->assoc_data->bss->bssid);
		if (!elems)
			return;
#define elems (*elems)

		(*klpe_ieee80211_rx_bss_info)(sdata, mgmt, len, rx_status);

		if (elems.dtim_period)
			ifmgd->dtim_period = elems.dtim_period;
		ifmgd->have_beacon = true;
		ifmgd->assoc_data->need_beacon = false;
		if (ieee80211_hw_check(&local->hw, TIMING_BEACON_ONLY)) {
			sdata->vif.bss_conf.sync_tsf =
				le64_to_cpu(mgmt->u.beacon.timestamp);
			sdata->vif.bss_conf.sync_device_ts =
				rx_status->device_timestamp;
			sdata->vif.bss_conf.sync_dtim_count = elems.dtim_count;
		}

		if (elems.mbssid_config_ie)
			bss_conf->profile_periodicity =
				elems.mbssid_config_ie->profile_periodicity;
		else
			bss_conf->profile_periodicity = 0;

		if (elems.ext_capab_len >= 11 &&
		    (elems.ext_capab[10] & WLAN_EXT_CAPA11_EMA_SUPPORT))
			bss_conf->ema_ap = true;
		else
			bss_conf->ema_ap = false;

		/* continue assoc process */
		ifmgd->assoc_data->timeout = jiffies;
		ifmgd->assoc_data->timeout_started = true;
		run_again(sdata, ifmgd->assoc_data->timeout);
/*
 * Fix CVE-2022-42719
 *  +2 lines
 */
#undef elems
		kfree(elems);
		return;
	}

	if (!ifmgd->associated ||
	    !ieee80211_rx_our_beacon(mgmt->bssid,  ifmgd->associated))
		return;
	bssid = ifmgd->associated->bssid;

	if (!(rx_status->flag & RX_FLAG_NO_SIGNAL_VAL))
		klpr_ieee80211_handle_beacon_sig(sdata, ifmgd, bss_conf,
					    local, rx_status);

	if (ifmgd->flags & IEEE80211_STA_CONNECTION_POLL) {
		mlme_dbg_ratelimited(sdata,
				     "cancelling AP probe due to a received beacon\n");
		(*klpe_ieee80211_reset_ap_probe)(sdata);
	}

	/*
	 * Push the beacon loss detection into the future since
	 * we are processing a beacon from the AP just now.
	 */
	(*klpe_ieee80211_sta_reset_beacon_monitor)(sdata);

	ncrc = crc32_be(0, (void *)&mgmt->u.beacon.beacon_int, 4);
	/*
	 * Fix CVE-2022-42719
	 *  -4 lines, +7 lines
	 */
	elems = klpp_ieee802_11_parse_elems_crc(mgmt->u.beacon.variable,
					   len - baselen, false,
					   care_about_ies, ncrc,
					   mgmt->bssid, bssid);
	if (!elems)
		return;
#define elems (*elems)
	ncrc = ((struct klpp_ieee802_11_elems *)&elems)->crc;

	if (ieee80211_hw_check(&local->hw, PS_NULLFUNC_STACK) &&
	    ieee80211_check_tim(elems.tim, elems.tim_len, bss_conf->aid)) {
		if (local->hw.conf.dynamic_ps_timeout > 0) {
			if (local->hw.conf.flags & IEEE80211_CONF_PS) {
				local->hw.conf.flags &= ~IEEE80211_CONF_PS;
				(*klpe_ieee80211_hw_config)(local,
						    IEEE80211_CONF_CHANGE_PS);
			}
			(*klpe_ieee80211_send_nullfunc)(local, sdata, false);
		} else if (!local->pspolling && sdata->u.mgd.powersave) {
			local->pspolling = true;

			/*
			 * Here is assumed that the driver will be
			 * able to send ps-poll frame and receive a
			 * response even though power save mode is
			 * enabled, but some drivers might require
			 * to disable power save here. This needs
			 * to be investigated.
			 */
			(*klpe_ieee80211_send_pspoll)(local, sdata);
		}
	}

	if (sdata->vif.p2p ||
	    sdata->vif.driver_flags & IEEE80211_VIF_GET_NOA_UPDATE) {
		struct ieee80211_p2p_noa_attr noa = {};
		int ret;

		ret = (*klpe_cfg80211_get_p2p_attr)(mgmt->u.beacon.variable,
					    len - baselen,
					    IEEE80211_P2P_ATTR_ABSENCE_NOTICE,
					    (u8 *) &noa, sizeof(noa));
		if (ret >= 2) {
			if (sdata->u.mgd.p2p_noa_index != noa.index) {
				/* valid noa_attr and index changed */
				sdata->u.mgd.p2p_noa_index = noa.index;
				memcpy(&bss_conf->p2p_noa_attr, &noa, sizeof(noa));
				changed |= BSS_CHANGED_P2P_PS;
				/*
				 * make sure we update all information, the CRC
				 * mechanism doesn't look at P2P attributes.
				 */
				ifmgd->beacon_crc_valid = false;
			}
		} else if (sdata->u.mgd.p2p_noa_index != -1) {
			/* noa_attr not found and we had valid noa_attr before */
			sdata->u.mgd.p2p_noa_index = -1;
			memset(&bss_conf->p2p_noa_attr, 0, sizeof(bss_conf->p2p_noa_attr));
			changed |= BSS_CHANGED_P2P_PS;
			ifmgd->beacon_crc_valid = false;
		}
	}

	if (ifmgd->csa_waiting_bcn)
		klpr_ieee80211_chswitch_post_beacon(sdata);

	/*
	 * Update beacon timing and dtim count on every beacon appearance. This
	 * will allow the driver to use the most updated values. Do it before
	 * comparing this one with last received beacon.
	 * IMPORTANT: These parameters would possibly be out of sync by the time
	 * the driver will use them. The synchronized view is currently
	 * guaranteed only in certain callbacks.
	 */
	if (ieee80211_hw_check(&local->hw, TIMING_BEACON_ONLY)) {
		sdata->vif.bss_conf.sync_tsf =
			le64_to_cpu(mgmt->u.beacon.timestamp);
		sdata->vif.bss_conf.sync_device_ts =
			rx_status->device_timestamp;
		sdata->vif.bss_conf.sync_dtim_count = elems.dtim_count;
	}

	if (ncrc == ifmgd->beacon_crc && ifmgd->beacon_crc_valid)
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		goto free;
	ifmgd->beacon_crc = ncrc;
	ifmgd->beacon_crc_valid = true;

	(*klpe_ieee80211_rx_bss_info)(sdata, mgmt, len, rx_status);

	(*klpe_ieee80211_sta_process_chanswitch)(sdata, rx_status->mactime,
					 rx_status->device_timestamp,
					 &elems, true);

	if (!(ifmgd->flags & IEEE80211_STA_DISABLE_WMM) &&
	    (*klpe_ieee80211_sta_wmm_params)(local, sdata, elems.wmm_param,
				     elems.wmm_param_len,
				     elems.mu_edca_param_set))
		changed |= BSS_CHANGED_QOS;

	/*
	 * If we haven't had a beacon before, tell the driver about the
	 * DTIM period (and beacon timing if desired) now.
	 */
	if (!ifmgd->have_beacon) {
		/* a few bogus AP send dtim_period = 0 or no TIM IE */
		bss_conf->dtim_period = elems.dtim_period ?: 1;

		changed |= BSS_CHANGED_BEACON_INFO;
		ifmgd->have_beacon = true;

		mutex_lock(&local->iflist_mtx);
		(*klpe_ieee80211_recalc_ps)(local);
		mutex_unlock(&local->iflist_mtx);

		(*klpe_ieee80211_recalc_ps_vif)(sdata);
	}

	if (elems.erp_info) {
		erp_valid = true;
		erp_value = elems.erp_info[0];
	} else {
		erp_valid = false;
	}
	changed |= (*klpe_ieee80211_handle_bss_capability)(sdata,
			le16_to_cpu(mgmt->u.beacon.capab_info),
			erp_valid, erp_value);

	mutex_lock(&local->sta_mtx);
	sta = (*klpe_sta_info_get)(sdata, bssid);

	changed |= ieee80211_recalc_twt_req(sdata, sta, &elems);

	if (klpr_ieee80211_config_bw(sdata, sta, elems.ht_cap_elem,
				elems.vht_cap_elem, elems.ht_operation,
				elems.vht_operation, elems.he_operation,
				bssid, &changed)) {
		mutex_unlock(&local->sta_mtx);
		sdata_info(sdata,
			   "failed to follow AP %pM bandwidth change, disconnect\n",
			   bssid);
		(*klpe_ieee80211_set_disassoc)(sdata, IEEE80211_STYPE_DEAUTH,
				       WLAN_REASON_DEAUTH_LEAVING,
				       true, deauth_buf);
		(*klpe_ieee80211_report_disconnect)(sdata, deauth_buf,
					    sizeof(deauth_buf), true,
					    WLAN_REASON_DEAUTH_LEAVING);
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		goto free;
	}

	if (sta && elems.opmode_notif)
		(*klpe_ieee80211_vht_handle_opmode)(sdata, sta, *elems.opmode_notif,
					    rx_status->band);
	mutex_unlock(&local->sta_mtx);

	changed |= klpr_ieee80211_handle_pwr_constr(sdata, chan, mgmt,
					       elems.country_elem,
					       elems.country_elem_len,
					       elems.pwr_constr_elem,
					       elems.cisco_dtpc_elem);

	(*klpe_ieee80211_bss_info_change_notify)(sdata, changed);
/*
 * Fix CVE-2022-42719
 *  +3 lines
 */
free:
#undef elems
	kfree(elems);
}

void klpp_ieee80211_sta_rx_queued_mgmt(struct ieee80211_sub_if_data *sdata,
				  struct sk_buff *skb)
{
	struct ieee80211_rx_status *rx_status;
	struct ieee80211_mgmt *mgmt;
	u16 fc;
	/*
	 * Fix CVE-2022-42719
	 *  -1 line
	 */

	int ies_len;

	rx_status = (struct ieee80211_rx_status *) skb->cb;
	mgmt = (struct ieee80211_mgmt *) skb->data;
	fc = le16_to_cpu(mgmt->frame_control);

	sdata_lock(sdata);

	switch (fc & IEEE80211_FCTL_STYPE) {
	case IEEE80211_STYPE_BEACON:
		klpp_ieee80211_rx_mgmt_beacon(sdata, mgmt, skb->len, rx_status);
		break;
	case IEEE80211_STYPE_PROBE_RESP:
		klpr_ieee80211_rx_mgmt_probe_resp(sdata, skb);
		break;
	case IEEE80211_STYPE_AUTH:
		klpr_ieee80211_rx_mgmt_auth(sdata, mgmt, skb->len);
		break;
	case IEEE80211_STYPE_DEAUTH:
		klpr_ieee80211_rx_mgmt_deauth(sdata, mgmt, skb->len);
		break;
	case IEEE80211_STYPE_DISASSOC:
		klpr_ieee80211_rx_mgmt_disassoc(sdata, mgmt, skb->len);
		break;
	case IEEE80211_STYPE_ASSOC_RESP:
	case IEEE80211_STYPE_REASSOC_RESP:
		klpp_ieee80211_rx_mgmt_assoc_resp(sdata, mgmt, skb->len);
		break;
	case IEEE80211_STYPE_ACTION:
		if (mgmt->u.action.category == WLAN_CATEGORY_SPECTRUM_MGMT) {
			/*
			 * Fix CVE-2022-42719
			 *  +1 line
			 */
			struct ieee802_11_elems *elems;

			ies_len = skb->len -
				  offsetof(struct ieee80211_mgmt,
					   u.action.u.chan_switch.variable);

			if (ies_len < 0)
				break;

			/* CSA IE cannot be overridden, no need for BSSID */
			/*
			 * Fix CVE-2022-42719
			 *  -3 lines, +3 lines
			 */
			elems = klpp_ieee802_11_parse_elems(
				mgmt->u.action.u.chan_switch.variable,
				ies_len, true, mgmt->bssid, NULL);

			/*
			 * Fix CVE-2022-42719
			 *  -7 lines, +6 lines
			 */
			if (elems && !elems->parse_error)
				(*klpe_ieee80211_sta_process_chanswitch)(sdata,
								 rx_status->mactime,
								 rx_status->device_timestamp,
								 elems, false);
			kfree(elems);
		} else if (mgmt->u.action.category == WLAN_CATEGORY_PUBLIC) {
			/*
			 * Fix CVE-2022-42719
			 *  +1 line
			 */
			struct ieee802_11_elems *elems;

			ies_len = skb->len -
				  offsetof(struct ieee80211_mgmt,
					   u.action.u.ext_chan_switch.variable);

			if (ies_len < 0)
				break;

			/*
			 * extended CSA IE can't be overridden, no need for
			 * BSSID
			 */
			/*
			 * Fix CVE-2022-42719
			 *  -3 lines, +3 lines
			 */
			elems = klpp_ieee802_11_parse_elems(
					mgmt->u.action.u.ext_chan_switch.variable,
					ies_len, true, mgmt->bssid, NULL);

			/*
			 * Fix CVE-2022-42719
			 *  -10 lines, +11 lines
			 */
			if (elems && !elems->parse_error) {
				/* for the handling code pretend this was also an IE */
				elems->ext_chansw_ie =
					&mgmt->u.action.u.ext_chan_switch.data;

				(*klpe_ieee80211_sta_process_chanswitch)(sdata,
							 rx_status->mactime,
							 rx_status->device_timestamp,
							 elems, false);
			}
			kfree(elems);
		}
		break;
	}
	sdata_unlock(sdata);
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1203994.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "mac80211"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "cfg80211_abandon_assoc", (void *)&klpe_cfg80211_abandon_assoc,
	  "cfg80211" },
	{ "cfg80211_assoc_timeout", (void *)&klpe_cfg80211_assoc_timeout,
	  "cfg80211" },
	{ "cfg80211_ch_switch_notify", (void *)&klpe_cfg80211_ch_switch_notify,
	  "cfg80211" },
	{ "cfg80211_chandef_valid", (void *)&klpe_cfg80211_chandef_valid,
	  "cfg80211" },
	{ "cfg80211_get_p2p_attr", (void *)&klpe_cfg80211_get_p2p_attr,
	  "cfg80211" },
	{ "cfg80211_rx_assoc_resp", (void *)&klpe_cfg80211_rx_assoc_resp,
	  "cfg80211" },
	{ "cfg80211_rx_mlme_mgmt", (void *)&klpe_cfg80211_rx_mlme_mgmt,
	  "cfg80211" },
	{ "cfg80211_find_elem_match", (void *)&klpe_cfg80211_find_elem_match,
	  "cfg80211" },
	{ "ieee80211_freq_khz_to_channel",
	  (void *)&klpe_ieee80211_freq_khz_to_channel, "cfg80211" },
	{ "ieee80211_get_channel_khz", (void *)&klpe_ieee80211_get_channel_khz,
	  "cfg80211" },
	{ "__ieee80211_recalc_txpower",
	  (void *)&klpe___ieee80211_recalc_txpower, "mac80211" },
	{ "__ieee80211_tx_skb_tid_band",
	  (void *)&klpe___ieee80211_tx_skb_tid_band, "mac80211" },
	{ "__sta_info_destroy", (void *)&klpe___sta_info_destroy, "mac80211" },
	{ "__tracepoint_drv_post_channel_switch",
	  (void *)&klpe___tracepoint_drv_post_channel_switch, "mac80211" },
	{ "__tracepoint_drv_return_int",
	  (void *)&klpe___tracepoint_drv_return_int, "mac80211" },
	{ "beacon_loss_count", (void *)&klpe_beacon_loss_count, "mac80211" },
	{ "drv_event_callback", (void *)&klpe_drv_event_callback, "mac80211" },
	{ "drv_mgd_prepare_tx", (void *)&klpe_drv_mgd_prepare_tx, "mac80211" },
	{ "fils_decrypt_assoc_resp", (void *)&klpe_fils_decrypt_assoc_resp,
	  "mac80211" },
	{ "ieee80211_ac_to_qos_mask", (void *)&klpe_ieee80211_ac_to_qos_mask,
	  "mac80211" },
	{ "ieee80211_bss_info_change_notify",
	  (void *)&klpe_ieee80211_bss_info_change_notify, "mac80211" },
	{ "ieee80211_chandef_downgrade",
	  (void *)&klpe_ieee80211_chandef_downgrade, "mac80211" },
	{ "ieee80211_check_rate_mask", (void *)&klpe_ieee80211_check_rate_mask,
	  "mac80211" },
	{ "ieee80211_cqm_rssi_notify", (void *)&klpe_ieee80211_cqm_rssi_notify,
	  "mac80211" },
	{ "ieee80211_destroy_auth_data",
	  (void *)&klpe_ieee80211_destroy_auth_data, "mac80211" },
	{ "ieee80211_determine_chantype",
	  (void *)&klpe_ieee80211_determine_chantype, "mac80211" },
	{ "ieee80211_get_reason_code_string",
	  (void *)&klpe_ieee80211_get_reason_code_string, "mac80211" },
	{ "ieee80211_handle_bss_capability",
	  (void *)&klpe_ieee80211_handle_bss_capability, "mac80211" },
	{ "ieee80211_he_cap_ie_to_sta_he_cap",
	  (void *)&klpe_ieee80211_he_cap_ie_to_sta_he_cap, "mac80211" },
	{ "ieee80211_he_op_ie_to_bss_conf",
	  (void *)&klpe_ieee80211_he_op_ie_to_bss_conf, "mac80211" },
	{ "ieee80211_he_spr_ie_to_bss_conf",
	  (void *)&klpe_ieee80211_he_spr_ie_to_bss_conf, "mac80211" },
	{ "ieee80211_ht_cap_ie_to_sta_ht_cap",
	  (void *)&klpe_ieee80211_ht_cap_ie_to_sta_ht_cap, "mac80211" },
	{ "ieee80211_hw_config", (void *)&klpe_ieee80211_hw_config,
	  "mac80211" },
	{ "ieee80211_led_assoc", (void *)&klpe_ieee80211_led_assoc,
	  "mac80211" },
	{ "ieee80211_mark_sta_auth", (void *)&klpe_ieee80211_mark_sta_auth,
	  "mac80211" },
	{ "ieee80211_queue_work", (void *)&klpe_ieee80211_queue_work,
	  "mac80211" },
	{ "ieee80211_recalc_ps", (void *)&klpe_ieee80211_recalc_ps,
	  "mac80211" },
	{ "ieee80211_recalc_ps_vif", (void *)&klpe_ieee80211_recalc_ps_vif,
	  "mac80211" },
	{ "ieee80211_recalc_smps", (void *)&klpe_ieee80211_recalc_smps,
	  "mac80211" },
	{ "ieee80211_report_disconnect",
	  (void *)&klpe_ieee80211_report_disconnect, "mac80211" },
	{ "ieee80211_reset_ap_probe", (void *)&klpe_ieee80211_reset_ap_probe,
	  "mac80211" },
	{ "ieee80211_rx_bss_info", (void *)&klpe_ieee80211_rx_bss_info,
	  "mac80211" },
	{ "ieee80211_send_auth", (void *)&klpe_ieee80211_send_auth,
	  "mac80211" },
	{ "ieee80211_send_nullfunc", (void *)&klpe_ieee80211_send_nullfunc,
	  "mac80211" },
	{ "ieee80211_send_pspoll", (void *)&klpe_ieee80211_send_pspoll,
	  "mac80211" },
	{ "ieee80211_set_disassoc", (void *)&klpe_ieee80211_set_disassoc,
	  "mac80211" },
	{ "ieee80211_set_wmm_default", (void *)&klpe_ieee80211_set_wmm_default,
	  "mac80211" },
	{ "ieee80211_sta_process_chanswitch",
	  (void *)&klpe_ieee80211_sta_process_chanswitch, "mac80211" },
	{ "ieee80211_sta_reset_beacon_monitor",
	  (void *)&klpe_ieee80211_sta_reset_beacon_monitor, "mac80211" },
	{ "ieee80211_sta_rx_notify", (void *)&klpe_ieee80211_sta_rx_notify,
	  "mac80211" },
	{ "ieee80211_sta_wmm_params", (void *)&klpe_ieee80211_sta_wmm_params,
	  "mac80211" },
	{ "ieee80211_stop_poll", (void *)&klpe_ieee80211_stop_poll,
	  "mac80211" },
	{ "ieee80211_tdls_handle_disconnect",
	  (void *)&klpe_ieee80211_tdls_handle_disconnect, "mac80211" },
	{ "ieee80211_vht_cap_ie_to_sta_vht_cap",
	  (void *)&klpe_ieee80211_vht_cap_ie_to_sta_vht_cap, "mac80211" },
	{ "ieee80211_vht_handle_opmode",
	  (void *)&klpe_ieee80211_vht_handle_opmode, "mac80211" },
	{ "ieee80211_vif_change_bandwidth",
	  (void *)&klpe_ieee80211_vif_change_bandwidth, "mac80211" },
	{ "ieee80211_vif_release_channel",
	  (void *)&klpe_ieee80211_vif_release_channel, "mac80211" },
	{ "ieee80211_wake_vif_queues", (void *)&klpe_ieee80211_wake_vif_queues,
	  "mac80211" },
	{ "rate_control_rate_init", (void *)&klpe_rate_control_rate_init,
	  "mac80211" },
	{ "rate_control_rate_update", (void *)&klpe_rate_control_rate_update,
	  "mac80211" },
	{ "sta_info_destroy_addr", (void *)&klpe_sta_info_destroy_addr,
	  "mac80211" },
	{ "sta_info_get", (void *)&klpe_sta_info_get, "mac80211" },
	{ "sta_info_move_state", (void *)&klpe_sta_info_move_state,
	  "mac80211" }
};

static int livepatch_bsc1203994_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1203994_module_nb = {
	.notifier_call = livepatch_bsc1203994_module_notify,
	.priority = INT_MIN+1,
};

int bsc1203994_mac80211_mlme_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LIVEPATCHED_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&livepatch_bsc1203994_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void bsc1203994_mac80211_mlme_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1203994_module_nb);
}

#endif /* IS_ENABLED(CONFIG_CFG80211) */
