/*
 * bsc1203994_mac80211_mesh
 *
 * Fix for CVE-2022-41674, CVE-2022-42719, CVE-2022-42720 and CVE-2022-42721,
 * bsc#1203994 (net/mac80211/mesh.c part)
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_CFG80211)

#if !IS_MODULE(CONFIG_MAC80211)
#error "Live patch supports only CONFIG_MAC80211=m"
#endif

#include "bsc1203994_common.h"

/* klp-ccp: from net/mac80211/mesh.c */
#include <asm/unaligned.h>
/* klp-ccp: from include/net/cfg80211.h */
static u32 (*klpe_ieee80211_channel_to_freq_khz)(int chan, enum nl80211_band band);

static inline int
klpr_ieee80211_channel_to_frequency(int chan, enum nl80211_band band)
{
	return KHZ_TO_MHZ((*klpe_ieee80211_channel_to_freq_khz)(chan, band));
}

static struct ieee80211_channel *
(*klpe_ieee80211_get_channel_khz)(struct wiphy *wiphy, u32 freq);

static inline struct ieee80211_channel *
klpr_ieee80211_get_channel(struct wiphy *wiphy, int freq)
{
	return (*klpe_ieee80211_get_channel_khz)(wiphy, MHZ_TO_KHZ(freq));
}

/* klp-ccp: from net/mac80211/ieee80211_i.h */
static void (*klpe___ieee80211_tx_skb_tid_band)(struct ieee80211_sub_if_data *sdata,
				 struct sk_buff *skb, int tid,
				 enum nl80211_band band);

static inline void klpr_ieee80211_tx_skb_tid(struct ieee80211_sub_if_data *sdata,
					struct sk_buff *skb, int tid)
{
	struct ieee80211_chanctx_conf *chanctx_conf;

	rcu_read_lock();
	chanctx_conf = rcu_dereference(sdata->vif.chanctx_conf);
	if (WARN_ON(!chanctx_conf)) {
		rcu_read_unlock();
		kfree_skb(skb);
		return;
	}

	(*klpe___ieee80211_tx_skb_tid_band)(sdata, skb, tid,
				    chanctx_conf->def.chan->band);
	rcu_read_unlock();
}

static inline void klpr_ieee80211_tx_skb(struct ieee80211_sub_if_data *sdata,
				    struct sk_buff *skb)
{
	/* Send all internal mgmt frames on VO. Accordingly set TID to 7. */
	klpr_ieee80211_tx_skb_tid(sdata, skb, 7);
}

/* klp-ccp: from net/mac80211/mesh.h */
#include <linux/types.h>
#include <linux/jhash.h>

static bool (*klpe_mesh_matches_local)(struct ieee80211_sub_if_data *sdata,
			struct ieee802_11_elems *ie);

static void (*klpe_mesh_neighbour_update)(struct ieee80211_sub_if_data *sdata,
			   u8 *hw_addr, struct ieee802_11_elems *ie,
			   struct ieee80211_rx_status *rx_status);

/* klp-ccp: from net/mac80211/driver-ops.h */
#include <net/mac80211.h>

/* klp-ccp: from net/mac80211/mesh.c */
static bool
(*klpe_ieee80211_mesh_process_chnswitch)(struct ieee80211_sub_if_data *sdata,
				 struct ieee802_11_elems *elems, bool beacon);

void
klpp_ieee80211_mesh_rx_probe_req(struct ieee80211_sub_if_data *sdata,
			    struct ieee80211_mgmt *mgmt, size_t len)
{
	struct ieee80211_local *local = sdata->local;
	struct ieee80211_if_mesh *ifmsh = &sdata->u.mesh;
	struct sk_buff *presp;
	struct beacon_data *bcn;
	struct ieee80211_mgmt *hdr;
	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +1 line
	 */
	struct ieee802_11_elems *elems;
	size_t baselen;
	u8 *pos;

	pos = mgmt->u.probe_req.variable;
	baselen = (u8 *) pos - (u8 *) mgmt;
	if (baselen > len)
		return;

	/*
	 * Fix CVE-2022-42719
	 *  -2 lines, +5 lines
	 */
	elems = klpp_ieee802_11_parse_elems(pos, len - baselen, false, mgmt->bssid,
			       NULL);
	if (!elems)
		return;
#define elems (*elems)

	if (!elems.mesh_id)
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		goto free;

	/* 802.11-2012 10.1.4.3.2 */
	if ((!ether_addr_equal(mgmt->da, sdata->vif.addr) &&
	     !is_broadcast_ether_addr(mgmt->da)) ||
	    elems.ssid_len != 0)
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		goto free;

	/*
	 * Fix CVE-2022-42719
	 *  -4 lines, +4 lines
	 */
	if (elems.mesh_id_len != 0 &&
	    (elems.mesh_id_len != ifmsh->mesh_id_len ||
	     memcmp(elems.mesh_id, ifmsh->mesh_id, ifmsh->mesh_id_len)))
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		goto free;

	rcu_read_lock();
	bcn = rcu_dereference(ifmsh->beacon);

	if (!bcn)
		goto out;

	presp = dev_alloc_skb(local->tx_headroom +
			      bcn->head_len + bcn->tail_len);
	if (!presp)
		goto out;

	skb_reserve(presp, local->tx_headroom);
	skb_put_data(presp, bcn->head, bcn->head_len);
	skb_put_data(presp, bcn->tail, bcn->tail_len);
	hdr = (struct ieee80211_mgmt *) presp->data;
	hdr->frame_control = cpu_to_le16(IEEE80211_FTYPE_MGMT |
					 IEEE80211_STYPE_PROBE_RESP);
	memcpy(hdr->da, mgmt->sa, ETH_ALEN);
	IEEE80211_SKB_CB(presp)->flags |= IEEE80211_TX_INTFL_DONT_ENCRYPT;
	klpr_ieee80211_tx_skb(sdata, presp);
out:
	rcu_read_unlock();
/*
 * Fix CVE-2022-42719
 *  +3 lines
 */
free:
#undef elems
	kfree(elems);
}

void klpp_ieee80211_mesh_rx_bcn_presp(struct ieee80211_sub_if_data *sdata,
					u16 stype,
					struct ieee80211_mgmt *mgmt,
					size_t len,
					struct ieee80211_rx_status *rx_status)
{
	struct ieee80211_local *local = sdata->local;
	struct ieee80211_if_mesh *ifmsh = &sdata->u.mesh;
	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +1 line
	 */
	struct ieee802_11_elems *elems;
	struct ieee80211_channel *channel;
	size_t baselen;
	int freq;
	enum nl80211_band band = rx_status->band;

	/* ignore ProbeResp to foreign address */
	if (stype == IEEE80211_STYPE_PROBE_RESP &&
	    !ether_addr_equal(mgmt->da, sdata->vif.addr))
		return;

	baselen = (u8 *) mgmt->u.probe_resp.variable - (u8 *) mgmt;
	if (baselen > len)
		return;

	/*
	 * Fix CVE-2022-42719
	 *  -2 lines, +6 lines
	 */
	elems = klpp_ieee802_11_parse_elems(mgmt->u.probe_resp.variable,
				       len - baselen,
				       false, mgmt->bssid, NULL);
	if (!elems)
		return;
#define elems (*elems)

	/* ignore non-mesh or secure / unsecure mismatch */
	if ((!elems.mesh_id || !elems.mesh_config) ||
	    (elems.rsn && sdata->u.mesh.security == IEEE80211_MESH_SEC_NONE) ||
	    (!elems.rsn && sdata->u.mesh.security != IEEE80211_MESH_SEC_NONE))
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		goto free;

	if (elems.ds_params)
		freq = klpr_ieee80211_channel_to_frequency(elems.ds_params[0], band);
	else
		freq = rx_status->freq;

	channel = klpr_ieee80211_get_channel(local->hw.wiphy, freq);

	if (!channel || channel->flags & IEEE80211_CHAN_DISABLED)
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		goto free;

	if ((*klpe_mesh_matches_local)(sdata, &elems)) {
		mpl_dbg(sdata, "rssi_threshold=%d,rx_status->signal=%d\n",
			sdata->u.mesh.mshcfg.rssi_threshold, rx_status->signal);
		if (!sdata->u.mesh.user_mpm ||
		    sdata->u.mesh.mshcfg.rssi_threshold == 0 ||
		    sdata->u.mesh.mshcfg.rssi_threshold < rx_status->signal)
			(*klpe_mesh_neighbour_update)(sdata, mgmt->sa, &elems,
					      rx_status);

		if (ifmsh->csa_role != IEEE80211_MESH_CSA_ROLE_INIT &&
		    !sdata->vif.csa_active)
			(*klpe_ieee80211_mesh_process_chnswitch)(sdata, &elems, true);
	}

	if (ifmsh->sync_ops)
		ifmsh->sync_ops->rx_bcn_presp(sdata,
			stype, mgmt, &elems, rx_status);

/*
 * Fix CVE-2022-42719
 *  +3 lines
 */
free:
#undef elems
	kfree(elems);
}

static int klpr_mesh_fwd_csa_frame(struct ieee80211_sub_if_data *sdata,
			       struct ieee80211_mgmt *mgmt, size_t len,
			       struct ieee802_11_elems *elems)
{
	struct ieee80211_mgmt *mgmt_fwd;
	struct sk_buff *skb;
	struct ieee80211_local *local = sdata->local;

	skb = dev_alloc_skb(local->tx_headroom + len);
	if (!skb)
		return -ENOMEM;
	skb_reserve(skb, local->tx_headroom);
	mgmt_fwd = skb_put(skb, len);

	elems->mesh_chansw_params_ie->mesh_ttl--;
	elems->mesh_chansw_params_ie->mesh_flags &=
		~WLAN_EID_CHAN_SWITCH_PARAM_INITIATOR;

	memcpy(mgmt_fwd, mgmt, len);
	eth_broadcast_addr(mgmt_fwd->da);
	memcpy(mgmt_fwd->sa, sdata->vif.addr, ETH_ALEN);
	memcpy(mgmt_fwd->bssid, sdata->vif.addr, ETH_ALEN);

	klpr_ieee80211_tx_skb(sdata, skb);
	return 0;
}

void klpp_mesh_rx_csa_frame(struct ieee80211_sub_if_data *sdata,
			      struct ieee80211_mgmt *mgmt, size_t len)
{
	struct ieee80211_if_mesh *ifmsh = &sdata->u.mesh;
	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +1 line
	 */
	struct ieee802_11_elems *elems;
	u16 pre_value;
	bool fwd_csa = true;
	size_t baselen;
	u8 *pos;

	if (mgmt->u.action.u.measurement.action_code !=
	    WLAN_ACTION_SPCT_CHL_SWITCH)
		return;

	pos = mgmt->u.action.u.chan_switch.variable;
	baselen = offsetof(struct ieee80211_mgmt,
			   u.action.u.chan_switch.variable);
	/*
	 * Fix CVE-2022-42719
	 *  -2 lines, +5 lines
	 */
	elems = klpp_ieee802_11_parse_elems(pos, len - baselen, true,
			       mgmt->bssid, NULL);
	if (!elems)
		return;
#define elems (*elems)

	if (!(*klpe_mesh_matches_local)(sdata, &elems))
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		goto free;

	ifmsh->chsw_ttl = elems.mesh_chansw_params_ie->mesh_ttl;
	if (!--ifmsh->chsw_ttl)
		fwd_csa = false;

	pre_value = le16_to_cpu(elems.mesh_chansw_params_ie->mesh_pre_value);
	if (ifmsh->pre_value >= pre_value)
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		goto free;

	ifmsh->pre_value = pre_value;

	if (!sdata->vif.csa_active &&
	    !(*klpe_ieee80211_mesh_process_chnswitch)(sdata, &elems, false)) {
		mcsa_dbg(sdata, "Failed to process CSA action frame");
		/*
		 * Fix CVE-2022-42719
		 *  -1 line, +1 line
		 */
		goto free;
	}

	/* forward or re-broadcast the CSA frame */
	if (fwd_csa) {
		if (klpr_mesh_fwd_csa_frame(sdata, mgmt, len, &elems) < 0)
			mcsa_dbg(sdata, "Failed to forward the CSA frame");
	}
/*
 * Fix CVE-2022-42719
 *  +3 lines
 */
free:
#undef elems
	kfree(elems);
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1203994.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "mac80211"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "ieee80211_channel_to_freq_khz",
	  (void *)&klpe_ieee80211_channel_to_freq_khz, "cfg80211" },
	{ "ieee80211_get_channel_khz", (void *)&klpe_ieee80211_get_channel_khz,
	  "cfg80211" },
	{ "__ieee80211_tx_skb_tid_band",
	  (void *)&klpe___ieee80211_tx_skb_tid_band, "mac80211" },
	{ "ieee80211_mesh_process_chnswitch",
	  (void *)&klpe_ieee80211_mesh_process_chnswitch, "mac80211" },
	{ "mesh_matches_local", (void *)&klpe_mesh_matches_local, "mac80211" },
	{ "mesh_neighbour_update", (void *)&klpe_mesh_neighbour_update,
	  "mac80211" },
};

static int livepatch_bsc1203994_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1203994_module_nb = {
	.notifier_call = livepatch_bsc1203994_module_notify,
	.priority = INT_MIN+1,
};

int bsc1203994_mac80211_mesh_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LIVEPATCHED_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&livepatch_bsc1203994_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void bsc1203994_mac80211_mesh_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1203994_module_nb);
}

#endif /* IS_ENABLED(CONFIG_CFG80211) */
