/*
 * bsc1203994_mac80211_ibss
 *
 * Fix for CVE-2022-41674, CVE-2022-42719, CVE-2022-42720 and CVE-2022-42721,
 * bsc#1203994 (net/mac80211/ibss.c part)
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_CFG80211)

#if !IS_MODULE(CONFIG_MAC80211)
#error "Live patch supports only CONFIG_MAC80211=m"
#endif

#include "bsc1203994_common.h"

/* klp-ccp: from net/mac80211/ibss.c */
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/if_ether.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/etherdevice.h>
#include <linux/rtnetlink.h>
#include <net/mac80211.h>

/* klp-ccp: from include/net/cfg80211.h */
static void (*klpe_cfg80211_chandef_create)(struct cfg80211_chan_def *chandef,
			     struct ieee80211_channel *channel,
			     enum nl80211_channel_type chantype);

static const struct cfg80211_chan_def *
(*klpe_cfg80211_chandef_compatible)(const struct cfg80211_chan_def *chandef1,
			    const struct cfg80211_chan_def *chandef2);

static int (*klpe_cfg80211_chandef_dfs_required)(struct wiphy *wiphy,
				  const struct cfg80211_chan_def *chandef,
				  enum nl80211_iftype iftype);

static struct ieee80211_channel *
(*klpe_ieee80211_get_channel_khz)(struct wiphy *wiphy, u32 freq);

static inline struct ieee80211_channel *
klpr_ieee80211_get_channel(struct wiphy *wiphy, int freq)
{
	return (*klpe_ieee80211_get_channel_khz)(wiphy, MHZ_TO_KHZ(freq));
}

static u32 (*klpe_ieee80211_mandatory_rates)(struct ieee80211_supported_band *sband,
			      enum nl80211_bss_scan_width scan_width);

static bool (*klpe_cfg80211_reg_can_beacon)(struct wiphy *wiphy,
			     struct cfg80211_chan_def *chandef,
			     enum nl80211_iftype iftype);

/* klp-ccp: from include/net/mac80211.h */
static void (*klpe_ieee80211_queue_work)(struct ieee80211_hw *hw, struct work_struct *work);

/* klp-ccp: from net/mac80211/sta_info.h */
static struct sta_info *(*klpe_sta_info_get)(struct ieee80211_sub_if_data *sdata,
			      const u8 *addr);

static int (*klpe_sta_info_destroy_addr)(struct ieee80211_sub_if_data *sdata,
			  const u8 *addr);

/* klp-ccp: from net/mac80211/ieee80211_i.h */
static u64 (*klpe_ieee80211_calculate_rx_timestamp)(struct ieee80211_local *local,
				     struct ieee80211_rx_status *status,
				     unsigned int mpdu_len,
				     unsigned int mpdu_offset);

void klpp_ieee80211_ibss_rx_queued_mgmt(struct ieee80211_sub_if_data *sdata,
				   struct sk_buff *skb);

static void (*klpe_ieee80211_rx_bss_put)(struct ieee80211_local *local,
			  struct ieee80211_bss *bss);

static int (*klpe_ieee80211_channel_switch)(struct wiphy *wiphy, struct net_device *dev,
			     struct cfg80211_csa_settings *params);

static void (*klpe_ieee80211_check_fast_xmit)(struct sta_info *sta);

static bool (*klpe_ieee80211_ht_cap_ie_to_sta_ht_cap)(struct ieee80211_sub_if_data *sdata,
				       struct ieee80211_supported_band *sband,
				       const struct ieee80211_ht_cap *ht_cap_ie,
				       struct sta_info *sta);

static void
(*klpe_ieee80211_vht_cap_ie_to_sta_vht_cap)(struct ieee80211_sub_if_data *sdata,
				    struct ieee80211_supported_band *sband,
				    const struct ieee80211_vht_cap *vht_cap_ie,
				    struct sta_info *sta);

static int (*klpe_ieee80211_parse_ch_switch_ie)(struct ieee80211_sub_if_data *sdata,
				 struct ieee802_11_elems *elems,
				 enum nl80211_band current_band,
				 u32 vht_cap_info,
				 u32 sta_flags, u8 *bssid,
				 struct ieee80211_csa_ie *csa_ie);

static void (*klpe___ieee80211_tx_skb_tid_band)(struct ieee80211_sub_if_data *sdata,
				 struct sk_buff *skb, int tid,
				 enum nl80211_band band);

static inline void klpr_ieee80211_tx_skb_tid(struct ieee80211_sub_if_data *sdata,
					struct sk_buff *skb, int tid)
{
	struct ieee80211_chanctx_conf *chanctx_conf;

	rcu_read_lock();
	chanctx_conf = rcu_dereference(sdata->vif.chanctx_conf);
	if (WARN_ON(!chanctx_conf)) {
		rcu_read_unlock();
		kfree_skb(skb);
		return;
	}

	(*klpe___ieee80211_tx_skb_tid_band)(sdata, skb, tid,
				    chanctx_conf->def.chan->band);
	rcu_read_unlock();
}

static inline void klpr_ieee80211_tx_skb(struct ieee80211_sub_if_data *sdata,
				    struct sk_buff *skb)
{
	/* Send all internal mgmt frames on VO. Accordingly set TID to 7. */
	klpr_ieee80211_tx_skb_tid(sdata, skb, 7);
}

static void (*klpe_ieee80211_send_auth)(struct ieee80211_sub_if_data *sdata,
			 u16 transaction, u16 auth_alg, u16 status,
			 const u8 *extra, size_t extra_len, const u8 *bssid,
			 const u8 *da, const u8 *key, u8 key_len, u8 key_idx,
			 u32 tx_flags);

static u32 (*klpe_ieee80211_sta_get_rates)(struct ieee80211_sub_if_data *sdata,
			    struct ieee802_11_elems *elems,
			    enum nl80211_band band, u32 *basic_rates);

static bool (*klpe_ieee80211_chandef_ht_oper)(const struct ieee80211_ht_operation *ht_oper,
			       struct cfg80211_chan_def *chandef);
static bool (*klpe_ieee80211_chandef_vht_oper)(struct ieee80211_hw *hw, u32 vht_cap_info,
				const struct ieee80211_vht_operation *oper,
				const struct ieee80211_ht_operation *htop,
				struct cfg80211_chan_def *chandef);

/* klp-ccp: from net/mac80211/driver-ops.h */
#include <net/mac80211.h>

/* klp-ccp: from net/mac80211/trace.h */
#include <linux/tracepoint.h>
#include <net/mac80211.h>

KLPR_TRACE_EVENT(drv_return_int,
	TP_PROTO(struct ieee80211_local *local, int ret),
	TP_ARGS(local, ret)
);

KLPR_DEFINE_EVENT(local_only_evt, drv_tx_last_beacon,
	TP_PROTO(struct ieee80211_local *local),
	TP_ARGS(local)
);

/* klp-ccp: from net/mac80211/driver-ops.h */
static void (*klpe_drv_sta_rc_update)(struct ieee80211_local *local,
		       struct ieee80211_sub_if_data *sdata,
		       struct ieee80211_sta *sta, u32 changed);

static u64 (*klpe_drv_get_tsf)(struct ieee80211_local *local,
		struct ieee80211_sub_if_data *sdata);

/* klp-ccp: from net/mac80211/driver-ops.h */
static inline int klpr_drv_tx_last_beacon(struct ieee80211_local *local)
{
	int ret = 0; /* default unsupported op for less congestion */

	might_sleep();

	klpr_trace_drv_tx_last_beacon(local);
	if (local->ops->tx_last_beacon)
		ret = local->ops->tx_last_beacon(&local->hw);
	klpr_trace_drv_return_int(local, ret);
	return ret;
}

/* klp-ccp: from net/mac80211/rate.h */
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/types.h>
#include <net/mac80211.h>

static void (*klpe_rate_control_rate_init)(struct sta_info *sta);

/* klp-ccp: from net/mac80211/ibss.c */
static void (*klpe_ieee80211_sta_join_ibss)(struct ieee80211_sub_if_data *sdata,
				    struct ieee80211_bss *bss);

static struct sta_info *
(*klpe_ieee80211_ibss_add_sta)(struct ieee80211_sub_if_data *sdata, const u8 *bssid,
		       const u8 *addr, u32 supp_rates);

static void (*klpe_ieee80211_ibss_csa_mark_radar)(struct ieee80211_sub_if_data *sdata);

static bool
klpr_ieee80211_ibss_process_chanswitch(struct ieee80211_sub_if_data *sdata,
				  struct ieee802_11_elems *elems,
				  bool beacon)
{
	struct cfg80211_csa_settings params;
	struct ieee80211_csa_ie csa_ie;
	struct ieee80211_if_ibss *ifibss = &sdata->u.ibss;
	enum nl80211_channel_type ch_type;
	int err;
	u32 sta_flags;
	u32 vht_cap_info = 0;

	sdata_assert_lock(sdata);

	sta_flags = IEEE80211_STA_DISABLE_VHT;
	switch (ifibss->chandef.width) {
	case NL80211_CHAN_WIDTH_5:
	case NL80211_CHAN_WIDTH_10:
	case NL80211_CHAN_WIDTH_20_NOHT:
		sta_flags |= IEEE80211_STA_DISABLE_HT;
		fallthrough;
	case NL80211_CHAN_WIDTH_20:
		sta_flags |= IEEE80211_STA_DISABLE_40MHZ;
		break;
	default:
		break;
	}

	if (elems->vht_cap_elem)
		vht_cap_info = le32_to_cpu(elems->vht_cap_elem->vht_cap_info);

	memset(&params, 0, sizeof(params));
	err = (*klpe_ieee80211_parse_ch_switch_ie)(sdata, elems,
					   ifibss->chandef.chan->band,
					   vht_cap_info,
					   sta_flags, ifibss->bssid, &csa_ie);
	/* can't switch to destination channel, fail */
	if (err < 0)
		goto disconnect;

	/* did not contain a CSA */
	if (err)
		return false;

	/* channel switch is not supported, disconnect */
	if (!(sdata->local->hw.wiphy->flags & WIPHY_FLAG_HAS_CHANNEL_SWITCH))
		goto disconnect;

	params.count = csa_ie.count;
	params.chandef = csa_ie.chandef;

	switch (ifibss->chandef.width) {
	case NL80211_CHAN_WIDTH_20_NOHT:
	case NL80211_CHAN_WIDTH_20:
	case NL80211_CHAN_WIDTH_40:
		/* keep our current HT mode (HT20/HT40+/HT40-), even if
		 * another mode  has been announced. The mode is not adopted
		 * within the beacon while doing CSA and we should therefore
		 * keep the mode which we announce.
		 */
		ch_type = cfg80211_get_chandef_type(&ifibss->chandef);
		(*klpe_cfg80211_chandef_create)(&params.chandef, params.chandef.chan,
					ch_type);
		break;
	case NL80211_CHAN_WIDTH_5:
	case NL80211_CHAN_WIDTH_10:
		if (params.chandef.width != ifibss->chandef.width) {
			sdata_info(sdata,
				   "IBSS %pM received channel switch from incompatible channel width (%d MHz, width:%d, CF1/2: %d/%d MHz), disconnecting\n",
				   ifibss->bssid,
				   params.chandef.chan->center_freq,
				   params.chandef.width,
				   params.chandef.center_freq1,
				   params.chandef.center_freq2);
			goto disconnect;
		}
		break;
	default:
		/* should not happen, sta_flags should prevent VHT modes. */
		WARN_ON(1);
		goto disconnect;
	}

	if (!(*klpe_cfg80211_reg_can_beacon)(sdata->local->hw.wiphy, &params.chandef,
				     NL80211_IFTYPE_ADHOC)) {
		sdata_info(sdata,
			   "IBSS %pM switches to unsupported channel (%d MHz, width:%d, CF1/2: %d/%d MHz), disconnecting\n",
			   ifibss->bssid,
			   params.chandef.chan->center_freq,
			   params.chandef.width,
			   params.chandef.center_freq1,
			   params.chandef.center_freq2);
		goto disconnect;
	}

	err = (*klpe_cfg80211_chandef_dfs_required)(sdata->local->hw.wiphy,
					    &params.chandef,
					    NL80211_IFTYPE_ADHOC);
	if (err < 0)
		goto disconnect;
	if (err > 0 && !ifibss->userspace_handles_dfs) {
		/* IBSS-DFS only allowed with a control program */
		goto disconnect;
	}

	params.radar_required = err;

	if (cfg80211_chandef_identical(&params.chandef,
				       &sdata->vif.bss_conf.chandef)) {
		ibss_dbg(sdata,
			 "received csa with an identical chandef, ignoring\n");
		return true;
	}

	/* all checks done, now perform the channel switch. */
	ibss_dbg(sdata,
		 "received channel switch announcement to go to channel %d MHz\n",
		 params.chandef.chan->center_freq);

	params.block_tx = !!csa_ie.mode;

	if ((*klpe_ieee80211_channel_switch)(sdata->local->hw.wiphy, sdata->dev,
				     &params))
		goto disconnect;

	(*klpe_ieee80211_ibss_csa_mark_radar)(sdata);

	return true;
disconnect:
	ibss_dbg(sdata, "Can't handle channel switch, disconnect\n");
	(*klpe_ieee80211_queue_work)(&sdata->local->hw,
			     &ifibss->csa_connection_drop_work);

	(*klpe_ieee80211_ibss_csa_mark_radar)(sdata);

	return true;
}

static void
klpr_ieee80211_rx_mgmt_spectrum_mgmt(struct ieee80211_sub_if_data *sdata,
				struct ieee80211_mgmt *mgmt, size_t len,
				struct ieee80211_rx_status *rx_status,
				struct ieee802_11_elems *elems)
{
	int required_len;

	if (len < IEEE80211_MIN_ACTION_SIZE + 1)
		return;

	/* CSA is the only action we handle for now */
	if (mgmt->u.action.u.measurement.action_code !=
	    WLAN_ACTION_SPCT_CHL_SWITCH)
		return;

	required_len = IEEE80211_MIN_ACTION_SIZE +
		       sizeof(mgmt->u.action.u.chan_switch);
	if (len < required_len)
		return;

	if (!sdata->vif.csa_active)
		klpr_ieee80211_ibss_process_chanswitch(sdata, elems, false);
}

static void klpr_ieee80211_rx_mgmt_deauth_ibss(struct ieee80211_sub_if_data *sdata,
					  struct ieee80211_mgmt *mgmt,
					  size_t len)
{
	u16 reason = le16_to_cpu(mgmt->u.deauth.reason_code);

	if (len < IEEE80211_DEAUTH_FRAME_LEN)
		return;

	ibss_dbg(sdata, "RX DeAuth SA=%pM DA=%pM\n", mgmt->sa, mgmt->da);
	ibss_dbg(sdata, "\tBSSID=%pM (reason: %d)\n", mgmt->bssid, reason);
	(*klpe_sta_info_destroy_addr)(sdata, mgmt->sa);
}

static void klpr_ieee80211_rx_mgmt_auth_ibss(struct ieee80211_sub_if_data *sdata,
					struct ieee80211_mgmt *mgmt,
					size_t len)
{
	u16 auth_alg, auth_transaction;

	sdata_assert_lock(sdata);

	if (len < 24 + 6)
		return;

	auth_alg = le16_to_cpu(mgmt->u.auth.auth_alg);
	auth_transaction = le16_to_cpu(mgmt->u.auth.auth_transaction);

	ibss_dbg(sdata, "RX Auth SA=%pM DA=%pM\n", mgmt->sa, mgmt->da);
	ibss_dbg(sdata, "\tBSSID=%pM (auth_transaction=%d)\n",
		 mgmt->bssid, auth_transaction);

	if (auth_alg != WLAN_AUTH_OPEN || auth_transaction != 1)
		return;

	/*
	 * IEEE 802.11 standard does not require authentication in IBSS
	 * networks and most implementations do not seem to use it.
	 * However, try to reply to authentication attempts if someone
	 * has actually implemented this.
	 */
	(*klpe_ieee80211_send_auth)(sdata, 2, WLAN_AUTH_OPEN, 0, NULL, 0,
			    mgmt->sa, sdata->u.ibss.bssid, NULL, 0, 0, 0);
}

static void klpr_ieee80211_update_sta_info(struct ieee80211_sub_if_data *sdata,
				      struct ieee80211_mgmt *mgmt, size_t len,
				      struct ieee80211_rx_status *rx_status,
				      struct ieee802_11_elems *elems,
				      struct ieee80211_channel *channel)
{
	struct sta_info *sta;
	enum nl80211_band band = rx_status->band;
	enum nl80211_bss_scan_width scan_width;
	struct ieee80211_local *local = sdata->local;
	struct ieee80211_supported_band *sband;
	bool rates_updated = false;
	u32 supp_rates = 0;

	if (sdata->vif.type != NL80211_IFTYPE_ADHOC)
		return;

	if (!ether_addr_equal(mgmt->bssid, sdata->u.ibss.bssid))
		return;

	sband = local->hw.wiphy->bands[band];
	if (WARN_ON(!sband))
		return;

	rcu_read_lock();
	sta = (*klpe_sta_info_get)(sdata, mgmt->sa);

	if (elems->supp_rates) {
		supp_rates = (*klpe_ieee80211_sta_get_rates)(sdata, elems,
						     band, NULL);
		if (sta) {
			u32 prev_rates;

			prev_rates = sta->sta.supp_rates[band];
			/* make sure mandatory rates are always added */
			scan_width = NL80211_BSS_CHAN_WIDTH_20;
			if (rx_status->bw == RATE_INFO_BW_5)
				scan_width = NL80211_BSS_CHAN_WIDTH_5;
			else if (rx_status->bw == RATE_INFO_BW_10)
				scan_width = NL80211_BSS_CHAN_WIDTH_10;

			sta->sta.supp_rates[band] = supp_rates |
				(*klpe_ieee80211_mandatory_rates)(sband, scan_width);
			if (sta->sta.supp_rates[band] != prev_rates) {
				ibss_dbg(sdata,
					 "updated supp_rates set for %pM based on beacon/probe_resp (0x%x -> 0x%x)\n",
					 sta->sta.addr, prev_rates,
					 sta->sta.supp_rates[band]);
				rates_updated = true;
			}
		} else {
			rcu_read_unlock();
			sta = (*klpe_ieee80211_ibss_add_sta)(sdata, mgmt->bssid,
						     mgmt->sa, supp_rates);
		}
	}

	if (sta && !sta->sta.wme &&
	    elems->wmm_info && local->hw.queues >= IEEE80211_NUM_ACS) {
		sta->sta.wme = true;
		(*klpe_ieee80211_check_fast_xmit)(sta);
	}

	if (sta && elems->ht_operation && elems->ht_cap_elem &&
	    sdata->u.ibss.chandef.width != NL80211_CHAN_WIDTH_20_NOHT &&
	    sdata->u.ibss.chandef.width != NL80211_CHAN_WIDTH_5 &&
	    sdata->u.ibss.chandef.width != NL80211_CHAN_WIDTH_10) {
		/* we both use HT */
		struct ieee80211_ht_cap htcap_ie;
		struct cfg80211_chan_def chandef;
		enum ieee80211_sta_rx_bandwidth bw = sta->sta.bandwidth;

		(*klpe_cfg80211_chandef_create)(&chandef, channel, NL80211_CHAN_NO_HT);
		(*klpe_ieee80211_chandef_ht_oper)(elems->ht_operation, &chandef);

		memcpy(&htcap_ie, elems->ht_cap_elem, sizeof(htcap_ie));
		rates_updated |= (*klpe_ieee80211_ht_cap_ie_to_sta_ht_cap)(sdata, sband,
								   &htcap_ie,
								   sta);

		if (elems->vht_operation && elems->vht_cap_elem &&
		    sdata->u.ibss.chandef.width != NL80211_CHAN_WIDTH_20 &&
		    sdata->u.ibss.chandef.width != NL80211_CHAN_WIDTH_40) {
			/* we both use VHT */
			struct ieee80211_vht_cap cap_ie;
			struct ieee80211_sta_vht_cap cap = sta->sta.vht_cap;
			u32 vht_cap_info =
				le32_to_cpu(elems->vht_cap_elem->vht_cap_info);

			(*klpe_ieee80211_chandef_vht_oper)(&local->hw, vht_cap_info,
						   elems->vht_operation,
						   elems->ht_operation,
						   &chandef);
			memcpy(&cap_ie, elems->vht_cap_elem, sizeof(cap_ie));
			(*klpe_ieee80211_vht_cap_ie_to_sta_vht_cap)(sdata, sband,
							    &cap_ie, sta);
			if (memcmp(&cap, &sta->sta.vht_cap, sizeof(cap)))
				rates_updated |= true;
		}

		if (bw != sta->sta.bandwidth)
			rates_updated |= true;

		if (!(*klpe_cfg80211_chandef_compatible)(&sdata->u.ibss.chandef,
						 &chandef))
			WARN_ON_ONCE(1);
	}

	if (sta && rates_updated) {
		u32 changed = IEEE80211_RC_SUPP_RATES_CHANGED;
		u8 rx_nss = sta->sta.rx_nss;

		/* Force rx_nss recalculation */
		sta->sta.rx_nss = 0;
		(*klpe_rate_control_rate_init)(sta);
		if (sta->sta.rx_nss != rx_nss)
			changed |= IEEE80211_RC_NSS_CHANGED;

		(*klpe_drv_sta_rc_update)(local, sdata, &sta->sta, changed);
	}

	rcu_read_unlock();
}

static void klpr_ieee80211_rx_bss_info(struct ieee80211_sub_if_data *sdata,
				  struct ieee80211_mgmt *mgmt, size_t len,
				  struct ieee80211_rx_status *rx_status,
				  struct ieee802_11_elems *elems)
{
	struct ieee80211_local *local = sdata->local;
	struct cfg80211_bss *cbss;
	struct ieee80211_bss *bss;
	struct ieee80211_channel *channel;
	u64 beacon_timestamp, rx_timestamp;
	u32 supp_rates = 0;
	enum nl80211_band band = rx_status->band;

	channel = klpr_ieee80211_get_channel(local->hw.wiphy, rx_status->freq);
	if (!channel)
		return;

	klpr_ieee80211_update_sta_info(sdata, mgmt, len, rx_status, elems, channel);

	bss = klpp_ieee80211_bss_info_update(local, rx_status, mgmt, len, channel);
	if (!bss)
		return;

	cbss = container_of((void *)bss, struct cfg80211_bss, priv);

	/* same for beacon and probe response */
	beacon_timestamp = le64_to_cpu(mgmt->u.beacon.timestamp);

	/* check if we need to merge IBSS */

	/* not an IBSS */
	if (!(cbss->capability & WLAN_CAPABILITY_IBSS))
		goto put_bss;

	/* different channel */
	if (sdata->u.ibss.fixed_channel &&
	    sdata->u.ibss.chandef.chan != cbss->channel)
		goto put_bss;

	/* different SSID */
	if (elems->ssid_len != sdata->u.ibss.ssid_len ||
	    memcmp(elems->ssid, sdata->u.ibss.ssid,
				sdata->u.ibss.ssid_len))
		goto put_bss;

	/* process channel switch */
	if (sdata->vif.csa_active ||
	    klpr_ieee80211_ibss_process_chanswitch(sdata, elems, true))
		goto put_bss;

	/* same BSSID */
	if (ether_addr_equal(cbss->bssid, sdata->u.ibss.bssid))
		goto put_bss;

	/* we use a fixed BSSID */
	if (sdata->u.ibss.fixed_bssid)
		goto put_bss;

	if (ieee80211_have_rx_timestamp(rx_status)) {
		/* time when timestamp field was received */
		rx_timestamp =
			(*klpe_ieee80211_calculate_rx_timestamp)(local, rx_status,
							 len + FCS_LEN, 24);
	} else {
		/*
		 * second best option: get current TSF
		 * (will return -1 if not supported)
		 */
		rx_timestamp = (*klpe_drv_get_tsf)(local, sdata);
	}

	ibss_dbg(sdata, "RX beacon SA=%pM BSSID=%pM TSF=0x%llx\n",
		 mgmt->sa, mgmt->bssid,
		 (unsigned long long)rx_timestamp);
	ibss_dbg(sdata, "\tBCN=0x%llx diff=%lld @%lu\n",
		 (unsigned long long)beacon_timestamp,
		 (unsigned long long)(rx_timestamp - beacon_timestamp),
		 jiffies);

	if (beacon_timestamp > rx_timestamp) {
		ibss_dbg(sdata,
			 "beacon TSF higher than local TSF - IBSS merge with BSSID %pM\n",
			 mgmt->bssid);
		(*klpe_ieee80211_sta_join_ibss)(sdata, bss);
		supp_rates = (*klpe_ieee80211_sta_get_rates)(sdata, elems, band, NULL);
		(*klpe_ieee80211_ibss_add_sta)(sdata, mgmt->bssid, mgmt->sa,
				       supp_rates);
		rcu_read_unlock();
	}

 put_bss:
	(*klpe_ieee80211_rx_bss_put)(local, bss);
}

static void klpr_ieee80211_rx_mgmt_probe_req(struct ieee80211_sub_if_data *sdata,
					struct sk_buff *req)
{
	struct ieee80211_mgmt *mgmt = (void *)req->data;
	struct ieee80211_if_ibss *ifibss = &sdata->u.ibss;
	struct ieee80211_local *local = sdata->local;
	int tx_last_beacon, len = req->len;
	struct sk_buff *skb;
	struct beacon_data *presp;
	u8 *pos, *end;

	sdata_assert_lock(sdata);

	presp = rcu_dereference_protected(ifibss->presp,
					  lockdep_is_held(&sdata->wdev.mtx));

	if (ifibss->state != IEEE80211_IBSS_MLME_JOINED ||
	    len < 24 + 2 || !presp)
		return;

	tx_last_beacon = klpr_drv_tx_last_beacon(local);

	ibss_dbg(sdata, "RX ProbeReq SA=%pM DA=%pM\n", mgmt->sa, mgmt->da);
	ibss_dbg(sdata, "\tBSSID=%pM (tx_last_beacon=%d)\n",
		 mgmt->bssid, tx_last_beacon);

	if (!tx_last_beacon && is_multicast_ether_addr(mgmt->da))
		return;

	if (!ether_addr_equal(mgmt->bssid, ifibss->bssid) &&
	    !is_broadcast_ether_addr(mgmt->bssid))
		return;

	end = ((u8 *) mgmt) + len;
	pos = mgmt->u.probe_req.variable;
	if (pos[0] != WLAN_EID_SSID ||
	    pos + 2 + pos[1] > end) {
		ibss_dbg(sdata, "Invalid SSID IE in ProbeReq from %pM\n",
			 mgmt->sa);
		return;
	}
	if (pos[1] != 0 &&
	    (pos[1] != ifibss->ssid_len ||
	     memcmp(pos + 2, ifibss->ssid, ifibss->ssid_len))) {
		/* Ignore ProbeReq for foreign SSID */
		return;
	}

	/* Reply with ProbeResp */
	skb = dev_alloc_skb(local->tx_headroom + presp->head_len);
	if (!skb)
		return;

	skb_reserve(skb, local->tx_headroom);
	skb_put_data(skb, presp->head, presp->head_len);

	memcpy(((struct ieee80211_mgmt *) skb->data)->da, mgmt->sa, ETH_ALEN);
	ibss_dbg(sdata, "Sending ProbeResp to %pM\n", mgmt->sa);
	IEEE80211_SKB_CB(skb)->flags |= IEEE80211_TX_INTFL_DONT_ENCRYPT;

	/* avoid excessive retries for probe request to wildcard SSIDs */
	if (pos[1] == 0)
		IEEE80211_SKB_CB(skb)->flags |= IEEE80211_TX_CTL_NO_ACK;

	klpr_ieee80211_tx_skb(sdata, skb);
}

static
void klpp_ieee80211_rx_mgmt_probe_beacon(struct ieee80211_sub_if_data *sdata,
				    struct ieee80211_mgmt *mgmt, size_t len,
				    struct ieee80211_rx_status *rx_status)
{
	size_t baselen;
	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +1 line
	 */
	struct ieee802_11_elems *elems;

	BUILD_BUG_ON(offsetof(typeof(mgmt->u.probe_resp), variable) !=
		     offsetof(typeof(mgmt->u.beacon), variable));

	/*
	 * either beacon or probe_resp but the variable field is at the
	 * same offset
	 */
	baselen = (u8 *) mgmt->u.probe_resp.variable - (u8 *) mgmt;
	if (baselen > len)
		return;

	/*
	 * Fix CVE-2022-42719
	 *  -2 lines, +3 line
	 */
	elems = klpp_ieee802_11_parse_elems(mgmt->u.probe_resp.variable,
				       len - baselen, false,
				       mgmt->bssid, NULL);

	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +4 lines
	 */
	if (elems) {
		klpr_ieee80211_rx_bss_info(sdata, mgmt, len, rx_status, elems);
		kfree(elems);
	}
}

void klpp_ieee80211_ibss_rx_queued_mgmt(struct ieee80211_sub_if_data *sdata,
				   struct sk_buff *skb)
{
	struct ieee80211_rx_status *rx_status;
	struct ieee80211_mgmt *mgmt;
	u16 fc;
	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +1 line
	 */
	struct ieee802_11_elems *elems;
	int ies_len;

	rx_status = IEEE80211_SKB_RXCB(skb);
	mgmt = (struct ieee80211_mgmt *) skb->data;
	fc = le16_to_cpu(mgmt->frame_control);

	sdata_lock(sdata);

	if (!sdata->u.ibss.ssid_len)
		goto mgmt_out; /* not ready to merge yet */

	switch (fc & IEEE80211_FCTL_STYPE) {
	case IEEE80211_STYPE_PROBE_REQ:
		klpr_ieee80211_rx_mgmt_probe_req(sdata, skb);
		break;
	case IEEE80211_STYPE_PROBE_RESP:
	case IEEE80211_STYPE_BEACON:
		klpp_ieee80211_rx_mgmt_probe_beacon(sdata, mgmt, skb->len,
					       rx_status);
		break;
	case IEEE80211_STYPE_AUTH:
		klpr_ieee80211_rx_mgmt_auth_ibss(sdata, mgmt, skb->len);
		break;
	case IEEE80211_STYPE_DEAUTH:
		klpr_ieee80211_rx_mgmt_deauth_ibss(sdata, mgmt, skb->len);
		break;
	case IEEE80211_STYPE_ACTION:
		switch (mgmt->u.action.category) {
		case WLAN_CATEGORY_SPECTRUM_MGMT:
			ies_len = skb->len -
				  offsetof(struct ieee80211_mgmt,
					   u.action.u.chan_switch.variable);

			if (ies_len < 0)
				break;

			/*
			 * Fix CVE-2022-42719
			 *  -3 lines, +3 lines
			 */
			elems = klpp_ieee802_11_parse_elems(
				mgmt->u.action.u.chan_switch.variable,
				ies_len, true, mgmt->bssid, NULL);

			/*
			 * Fix CVE-2022-42719
			 *  -5 lines, +5 lines
			 */
			if (elems && !elems->parse_error)
				klpr_ieee80211_rx_mgmt_spectrum_mgmt(sdata,
								 mgmt, skb->len,
								 rx_status,
								 elems);
			kfree(elems);
			break;
		}
	}

 mgmt_out:
	sdata_unlock(sdata);
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1203994.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "mac80211"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "cfg80211_chandef_compatible",
	  (void *)&klpe_cfg80211_chandef_compatible, "cfg80211" },
	{ "cfg80211_chandef_create", (void *)&klpe_cfg80211_chandef_create,
	  "cfg80211" },
	{ "cfg80211_chandef_dfs_required",
	  (void *)&klpe_cfg80211_chandef_dfs_required, "cfg80211" },
	{ "cfg80211_reg_can_beacon", (void *)&klpe_cfg80211_reg_can_beacon,
	  "cfg80211" },
	{ "ieee80211_get_channel_khz", (void *)&klpe_ieee80211_get_channel_khz,
	  "cfg80211" },
	{ "ieee80211_mandatory_rates", (void *)&klpe_ieee80211_mandatory_rates,
	  "cfg80211" },
	{ "__ieee80211_tx_skb_tid_band",
	  (void *)&klpe___ieee80211_tx_skb_tid_band, "mac80211" },
	{ "__tracepoint_drv_return_int",
	  (void *)&klpe___tracepoint_drv_return_int, "mac80211" },
	{ "__tracepoint_drv_tx_last_beacon",
	  (void *)&klpe___tracepoint_drv_tx_last_beacon, "mac80211" },
	{ "drv_get_tsf", (void *)&klpe_drv_get_tsf, "mac80211" },
	{ "drv_sta_rc_update", (void *)&klpe_drv_sta_rc_update, "mac80211" },
	{ "ieee80211_calculate_rx_timestamp",
	  (void *)&klpe_ieee80211_calculate_rx_timestamp, "mac80211" },
	{ "ieee80211_chandef_ht_oper", (void *)&klpe_ieee80211_chandef_ht_oper,
	  "mac80211" },
	{ "ieee80211_chandef_vht_oper",
	  (void *)&klpe_ieee80211_chandef_vht_oper, "mac80211" },
	{ "ieee80211_channel_switch", (void *)&klpe_ieee80211_channel_switch,
	  "mac80211" },
	{ "ieee80211_check_fast_xmit", (void *)&klpe_ieee80211_check_fast_xmit,
	  "mac80211" },
	{ "ieee80211_ht_cap_ie_to_sta_ht_cap",
	  (void *)&klpe_ieee80211_ht_cap_ie_to_sta_ht_cap, "mac80211" },
	{ "ieee80211_ibss_add_sta", (void *)&klpe_ieee80211_ibss_add_sta,
	  "mac80211" },
	{ "ieee80211_ibss_csa_mark_radar",
	  (void *)&klpe_ieee80211_ibss_csa_mark_radar, "mac80211" },
	{ "ieee80211_parse_ch_switch_ie",
	  (void *)&klpe_ieee80211_parse_ch_switch_ie, "mac80211" },
	{ "ieee80211_queue_work", (void *)&klpe_ieee80211_queue_work,
	  "mac80211" },
	{ "ieee80211_rx_bss_put", (void *)&klpe_ieee80211_rx_bss_put,
	  "mac80211" },
	{ "ieee80211_send_auth", (void *)&klpe_ieee80211_send_auth,
	  "mac80211" },
	{ "ieee80211_sta_get_rates", (void *)&klpe_ieee80211_sta_get_rates,
	  "mac80211" },
	{ "ieee80211_sta_join_ibss", (void *)&klpe_ieee80211_sta_join_ibss,
	  "mac80211" },
	{ "ieee80211_vht_cap_ie_to_sta_vht_cap",
	  (void *)&klpe_ieee80211_vht_cap_ie_to_sta_vht_cap, "mac80211" },
	{ "rate_control_rate_init", (void *)&klpe_rate_control_rate_init,
	  "mac80211" },
	{ "sta_info_destroy_addr", (void *)&klpe_sta_info_destroy_addr,
	  "mac80211" },
	{ "sta_info_get", (void *)&klpe_sta_info_get, "mac80211" },
};

static int livepatch_bsc1203994_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1203994_module_nb = {
	.notifier_call = livepatch_bsc1203994_module_notify,
	.priority = INT_MIN+1,
};

int bsc1203994_mac80211_ibss_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LIVEPATCHED_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&livepatch_bsc1203994_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void bsc1203994_mac80211_ibss_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1203994_module_nb);
}

#endif /* IS_ENABLED(CONFIG_CFG80211) */
