/*
 * bsc1203994_mac80211_agg_rx
 *
 * Fix for CVE-2022-41674, CVE-2022-42719, CVE-2022-42720 and CVE-2022-42721,
 * bsc#1203994 (net/mac80211/agg-rx.c part)
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_CFG80211)

#if !IS_MODULE(CONFIG_MAC80211)
#error "Live patch supports only CONFIG_MAC80211=m"
#endif

#include "bsc1203994_common.h"

/* klp-ccp: from net/mac80211/agg-rx.c */
#include <linux/ieee80211.h>
#include <linux/slab.h>
#include <linux/export.h>
#include <net/mac80211.h>
/* klp-ccp: from net/mac80211/ieee80211_i.h */
static void (*klpe____ieee80211_start_rx_ba_session)(struct sta_info *sta,
				      u8 dialog_token, u16 timeout,
				      u16 start_seq_num, u16 ba_policy, u16 tid,
				      u16 buf_size, bool tx, bool auto_seq,
				      const struct ieee80211_addba_ext_ie *addbaext);

void klpp_ieee80211_process_addba_request(struct ieee80211_local *local,
				     struct sta_info *sta,
				     struct ieee80211_mgmt *mgmt,
				     size_t len);

/* klp-ccp: from net/mac80211/driver-ops.h */
#include <net/mac80211.h>

/* klp-ccp: from net/mac80211/agg-rx.c */
static void klpr___ieee80211_start_rx_ba_session(struct sta_info *sta,
					    u8 dialog_token, u16 timeout,
					    u16 start_seq_num, u16 ba_policy,
					    u16 tid, u16 buf_size, bool tx,
					    bool auto_seq,
					    const struct ieee80211_addba_ext_ie *addbaext)
{
	mutex_lock(&sta->ampdu_mlme.mtx);
	(*klpe____ieee80211_start_rx_ba_session)(sta, dialog_token, timeout,
					 start_seq_num, ba_policy, tid,
					 buf_size, tx, auto_seq, addbaext);
	mutex_unlock(&sta->ampdu_mlme.mtx);
}

void klpp_ieee80211_process_addba_request(struct ieee80211_local *local,
				     struct sta_info *sta,
				     struct ieee80211_mgmt *mgmt,
				     size_t len)
{
	u16 capab, tid, timeout, ba_policy, buf_size, start_seq_num;
	/*
	 * Fix CVE-2022-42719
	 *  -1 line, +1 line
	 */
	struct ieee802_11_elems *elems = NULL;
	u8 dialog_token;
	int ies_len;

	/* extract session parameters from addba request frame */
	dialog_token = mgmt->u.action.u.addba_req.dialog_token;
	timeout = le16_to_cpu(mgmt->u.action.u.addba_req.timeout);
	start_seq_num =
		le16_to_cpu(mgmt->u.action.u.addba_req.start_seq_num) >> 4;

	capab = le16_to_cpu(mgmt->u.action.u.addba_req.capab);
	ba_policy = (capab & IEEE80211_ADDBA_PARAM_POLICY_MASK) >> 1;
	tid = (capab & IEEE80211_ADDBA_PARAM_TID_MASK) >> 2;
	buf_size = (capab & IEEE80211_ADDBA_PARAM_BUF_SIZE_MASK) >> 6;

	ies_len = len - offsetof(struct ieee80211_mgmt,
				 u.action.u.addba_req.variable);
	if (ies_len) {
		/*
		 * Fix CVE-2022-42719
		 *  -3 lines, +3 lines
		 */
		elems = klpp_ieee802_11_parse_elems(mgmt->u.action.u.addba_req.variable,
					ies_len, true, mgmt->bssid, NULL);
		if (!elems || elems->parse_error)
			goto free;
	}

	/*
	 * Fix CVE-2022-42719
	 *  -4 lines, +6 lines
	 */
	klpr___ieee80211_start_rx_ba_session(sta, dialog_token, timeout,
					start_seq_num, ba_policy, tid,
					buf_size, true, false,
					elems ? elems->addba_ext_ie : NULL);
free:
	kfree(elems);
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1203994.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "mac80211"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "___ieee80211_start_rx_ba_session",
	  (void *)&klpe____ieee80211_start_rx_ba_session, "mac80211" },
};

static int livepatch_bsc1203994_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	mutex_lock(&module_mutex);
	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	mutex_unlock(&module_mutex);
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1203994_module_nb = {
	.notifier_call = livepatch_bsc1203994_module_notify,
	.priority = INT_MIN+1,
};

int bsc1203994_mac80211_agg_rx_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LIVEPATCHED_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&livepatch_bsc1203994_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void bsc1203994_mac80211_agg_rx_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1203994_module_nb);
}

#endif /* IS_ENABLED(CONFIG_CFG80211) */
