// Copyright 2020-2024 the Pinniped contributors. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0

package timeouts

import (
	"time"

	"github.com/ory/fosite"
)

// StorageLifetime is a function that can, given a request, decide how long it should live in session storage.
type StorageLifetime func(requester fosite.Requester) time.Duration

// OverrideLifespan is a function that, given a request, can suggest to override the default lifespan
// by returning true along with a new lifespan. When false is returned, the returned duration should be ignored.
type OverrideLifespan func(accessRequest fosite.AccessRequester) (time.Duration, bool)

type Configuration struct {
	// The length of time that our state param that we encrypt and pass to the upstream OIDC IDP should be considered
	// valid. If a state param generated by the authorize endpoint is sent to the callback endpoint after this much
	// time has passed, then the callback endpoint should reject it. This allows us to set a limit on how long
	// the end user has to finish their login with the upstream IDP, including the time that it takes to fumble
	// with password manager and two-factor authenticator apps, and also accounting for taking a coffee break while
	// the browser is sitting at the upstream IDP's login page.
	UpstreamStateParamLifespan time.Duration

	// How long an authcode issued by the callback endpoint is valid. This determines how much time the end user
	// has to come back to exchange the authcode for tokens at the token endpoint.
	AuthorizeCodeLifespan time.Duration

	// The lifetime of an downstream access token issued by the token endpoint. Access tokens should generally
	// be fairly short-lived.
	AccessTokenLifespan time.Duration

	// Optionally override the default AccessTokenLifespan depending on the specific request.
	// Note that access tokens can be issued by authcode exchanges and refreshes (with different grant types on the
	// request), so implementations of this method should handle choosing lifespans for both cases as desired.
	// Note that fosite offers the fosite.ClientWithCustomTokenLifespans interface, but that interface does not
	// pass the full request details to the GetEffectiveLifespan() function, so it does not suit our needs,
	// and we use this technique instead.
	OverrideDefaultAccessTokenLifespan OverrideLifespan

	// The lifetime of an downstream ID token issued by the token endpoint. This should generally be the same
	// as the AccessTokenLifespan, or longer if it would be useful for the user's proof of identity to be valid
	// for longer than their proof of authorization.
	IDTokenLifespan time.Duration

	// Optionally override the default IDTokenLifespan depending on the specific request.
	// Note that ID tokens can be issued by authcode exchanges, refreshes, and RFC8693 token exchanges
	// (with different grant types on the request), so implementations of this method should handle choosing
	// lifespans for all three cases as desired.
	// Note that fosite offers the fosite.ClientWithCustomTokenLifespans interface, but that interface does not
	// pass the full request details to the GetEffectiveLifespan() function, so it does not suit our needs,
	// and we use this technique instead.
	OverrideDefaultIDTokenLifespan OverrideLifespan

	// The lifetime of an downstream refresh token issued by the token endpoint. This should generally be
	// significantly longer than the access token lifetime, so it can be used to refresh the access token
	// multiple times. Once the refresh token expires, the user's session is over and they will need
	// to start a new authorization request, which will require them to log in again with the upstream IDP
	// in their web browser.
	RefreshTokenLifespan time.Duration

	// AuthorizationCodeSessionStorageLifetime is the length of time after which an authcode is allowed to be garbage
	// collected from storage. Authcodes are kept in storage after they are redeemed to allow the system to mark the
	// authcode as already used, so it can reject any future uses of the same authcode with special case handling which
	// include revoking the access and refresh tokens associated with the session. Therefore, this should be
	// significantly longer than the AuthorizeCodeLifespan, and there is probably no reason to make it longer than
	// the sum of the AuthorizeCodeLifespan and the RefreshTokenLifespan.
	AuthorizationCodeSessionStorageLifetime StorageLifetime

	// PKCESessionStorageLifetime is the length of time after which PKCE data is allowed to be garbage collected from
	// storage. PKCE sessions are closely related to authorization code sessions. After the authcode is successfully
	// redeemed, the PKCE session is explicitly deleted. After the authcode expires, the PKCE session is no longer needed,
	// but it is not explicitly deleted. Therefore, this can be just slightly longer than the AuthorizeCodeLifespan. We'll
	// avoid making it exactly the same as AuthorizeCodeLifespan to avoid any chance of the garbage collector deleting it
	// while it is being used.
	PKCESessionStorageLifetime StorageLifetime

	// OIDCSessionStorageLifetime is the length of time after which the OIDC session data related to an authcode
	// is allowed to be garbage collected from storage. After the authcode is successfully redeemed, the OIDC session is
	// explicitly deleted. Similar to the PKCE session, they are not needed anymore after the corresponding authcode has expired.
	// Therefore, this can be just slightly longer than the AuthorizeCodeLifespan. We'll avoid making it exactly the same
	// as AuthorizeCodeLifespan to avoid any chance of the garbage collector deleting it while it is being used.
	OIDCSessionStorageLifetime StorageLifetime

	// AccessTokenSessionStorageLifetime is the length of time after which an access token's session data is allowed
	// to be garbage collected from storage.  These must exist in storage for as long as the refresh token is valid
	// or else the refresh flow will not work properly. So this must be longer than RefreshTokenLifespan.
	AccessTokenSessionStorageLifetime StorageLifetime

	// RefreshTokenSessionStorageLifetime is the length of time after which a refresh token's session data is allowed
	// to be garbage collected from storage. These must exist in storage for as long as the refresh token is valid.
	// Therefore, this can be just slightly longer than the RefreshTokenLifespan. We'll avoid making it exactly the same
	// as RefreshTokenLifespan to avoid any chance of the garbage collector deleting it while it is being used.
	// If an expired token is still stored when the user tries to refresh it, then they will get a more specific
	// error message telling them that the token is expired, rather than a more generic error that is returned
	// when the token does not exist. If this is desirable, then the RefreshTokenSessionStorageLifetime can be made
	// to be significantly larger than RefreshTokenLifespan, at the cost of slower cleanup.
	RefreshTokenSessionStorageLifetime StorageLifetime
}
