// Copyright Project Harbor Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
import {
    ChangeDetectionStrategy,
    Component,
    ElementRef,
    Input,
    OnChanges,
    SimpleChanges,
    ViewChild,
} from '@angular/core';
import { DangerousArtifact } from '../../../../../../../ng-swagger-gen/models/dangerous-artifact';
import * as echarts from 'echarts/core';
import { SeverityColors } from '../security-hub.interface';
import { TranslateService } from '@ngx-translate/core';

@Component({
    selector: 'app-single-bar',
    templateUrl: './single-bar.component.html',
    styleUrls: ['./single-bar.component.scss'],
    changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SingleBarComponent implements OnChanges {
    @Input()
    dangerousArtifact: DangerousArtifact;

    @ViewChild('container', { static: true })
    container: ElementRef;

    constructor(private translate: TranslateService) {}

    ngOnChanges(changes: SimpleChanges) {
        if (changes && changes['dangerousArtifact']) {
            this.initChart();
        }
    }

    initChart() {
        const chart = echarts.init(this.container.nativeElement);
        const [severity, c, h, m] = [
            'VULNERABILITY.GRID.COLUMN_SEVERITY',
            'VULNERABILITY.SEVERITY.CRITICAL',
            'VULNERABILITY.SEVERITY.HIGH',
            'VULNERABILITY.SEVERITY.MEDIUM',
        ];
        this.translate.get([severity, c, h, m]).subscribe(res => {
            chart.setOption({
                color: [
                    SeverityColors.CRITICAL,
                    SeverityColors.HIGH,
                    SeverityColors.MEDIUM,
                ],
                title: {
                    text: '',
                },
                tooltip: {
                    formatter: '{b}: {c}',
                    textStyle: {
                        fontSize: '12px',
                        whiteSpace: 'nowrap',
                    },
                },
                legend: {
                    show: false,
                },
                series: [
                    {
                        type: 'pie',
                        radius: '65%',
                        name: res[severity],
                        // adjust the start angle
                        startAngle: 180,
                        itemStyle: {
                            borderRadius: 2,
                            borderWidth: 1,
                        },
                        labelLine: {
                            show: false,
                        },
                        label: {
                            show: true,
                            position: 'inside',
                            formatter: '{c}',
                            fontSize: '9px',
                        },
                        data: [
                            {
                                name: res[c],
                                value:
                                    this.dangerousArtifact?.critical_cnt || 0,
                            },
                            {
                                name: res[h],
                                value: this.dangerousArtifact?.high_cnt || 0,
                            },
                            {
                                name: res[m],
                                value: this.dangerousArtifact?.medium_cnt || 0,
                            },
                            {
                                // make a record to fill the bottom 50%
                                value:
                                    this.dangerousArtifact?.critical_cnt +
                                        this.dangerousArtifact?.high_cnt +
                                        this.dangerousArtifact?.medium_cnt || 0,
                                itemStyle: {
                                    // stop the chart from rendering this piece
                                    color: 'none',
                                    decal: {
                                        symbol: 'none',
                                    },
                                },
                                label: {
                                    show: false,
                                },
                            },
                        ],
                    },
                ],
            });
        });
    }
}
