// Copyright Project Harbor Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
import { VULNERABILITY_SEVERITY } from '../../../../shared/units/utils';

export const SEVERITY_OPTIONS = [
    {
        severity: 'Critical',
        severityLevel: 'VULNERABILITY.SEVERITY.CRITICAL',
    },
    { severity: 'High', severityLevel: 'VULNERABILITY.SEVERITY.HIGH' },
    { severity: 'Medium', severityLevel: 'VULNERABILITY.SEVERITY.MEDIUM' },
    { severity: 'Low', severityLevel: 'VULNERABILITY.SEVERITY.LOW' },
    { severity: 'Unknown', severityLevel: 'UNKNOWN' },
    { severity: 'None', severityLevel: 'VULNERABILITY.SEVERITY.NONE' },
];

export enum OptionType {
    ALL = 'all',
    CVE_ID = 'cve_id',
    SEVERITY = 'severity',
    CVSS3 = 'cvss_score_v3',
    REPO = 'repository_name',
    PACKAGE = 'package',
    TAG = 'tag',
    PROJECT_ID = 'project_id',
    DIGEST = 'digest',
}

export const OptionType_I18n_Map = {
    [OptionType.ALL]: 'SECURITY_HUB.OPTION_ALL',
    [OptionType.CVE_ID]: 'SECURITY_HUB.CVE_ID',
    [OptionType.SEVERITY]: 'VULNERABILITY.GRID.COLUMN_SEVERITY',
    [OptionType.CVSS3]: 'VULNERABILITY.GRID.CVSS3',
    [OptionType.REPO]: 'SECURITY_HUB.REPO_NAME',
    [OptionType.PACKAGE]: 'VULNERABILITY.GRID.COLUMN_PACKAGE',
    [OptionType.TAG]: 'REPLICATION.TAG',
    [OptionType.PROJECT_ID]: 'SECURITY_HUB.OPTION_PROJECT_ID_NAME',
    [OptionType.DIGEST]: 'P2P_PROVIDER.DIGEST',
};

export interface OptionTypeValueMap {
    [key: string]: any;
}

export interface SearchEventData {
    normal: string[];
    projectId: string;
}

export const VUL_ID: string = 'vulnerabilities';

export function severityText(severity: string): string {
    switch (severity) {
        case VULNERABILITY_SEVERITY.CRITICAL:
            return 'VULNERABILITY.SEVERITY.CRITICAL';
        case VULNERABILITY_SEVERITY.HIGH:
            return 'VULNERABILITY.SEVERITY.HIGH';
        case VULNERABILITY_SEVERITY.MEDIUM:
            return 'VULNERABILITY.SEVERITY.MEDIUM';
        case VULNERABILITY_SEVERITY.LOW:
            return 'VULNERABILITY.SEVERITY.LOW';
        case VULNERABILITY_SEVERITY.NONE:
            return 'VULNERABILITY.SEVERITY.NONE';
        default:
            return 'UNKNOWN';
    }
}

export function getDigestLink(
    proId: number | string,
    repoName: string,
    digest: string
): any {
    const projectName = repoName?.split('/')[0];
    const realRepoName = projectName
        ? repoName?.substring(projectName.length + 1)
        : repoName;
    return [
        '/harbor/projects',
        proId,
        'repositories',
        realRepoName,
        'artifacts-tab',
        'artifacts',
        digest,
    ];
}

export function getRepoLink(proId: number | string, repoName: string): any {
    const projectName = repoName?.split('/')[0];
    const realRepoName = projectName
        ? repoName?.substring(projectName.length + 1)
        : repoName;
    return ['/harbor/projects', proId, 'repositories', realRepoName];
}

export const CVSS3_REG = /^([0-9]|10)(\.\d)?$/;

export enum SeverityColors {
    CRITICAL = '#FF4D2E',
    HIGH = '#FF8F3D',
    MEDIUM = '#FFCE66',
    LOW = '#FFF1AD',
    NONE = '#2EC0FF',
    NA = 'gray',
}
