# -----------------------------------------------------------------------------
# Copyright (c) 2020, 2023, Oracle and/or its affiliates.
#
# This software is dual-licensed to you under the Universal Permissive License
# (UPL) 1.0 as shown at https://oss.oracle.com/licenses/upl and Apache License
# 2.0 as shown at http://www.apache.org/licenses/LICENSE-2.0. You may choose
# either license.
#
# If you elect to accept the software under the Apache License, Version 2.0,
# the following applies:
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# -----------------------------------------------------------------------------

# -----------------------------------------------------------------------------
# exceptions.py
#
# Contains the exception classes mandated by the Python Database API.
# -----------------------------------------------------------------------------


class Warning(Exception):
    """
    Exception raised for warnings.

    Exception messages of this class will have the prefix DPY and an error
    number in the range 9000 - 9999.
    """


class Error(Exception):
    """
    Exception that is the base class of all other exceptions defined by
    python-oracledb.
    """


class DatabaseError(Error):
    """
    Exception raised for errors that are related to the database. It is a
    subclass of Error.

    Exception messages of this class will have the prefix DPY and an error
    number in the range 4000 - 4999.
    """


class DataError(DatabaseError):
    """
    Exception raised for errors that are due to problems with the processed
    data. It is a subclass of DatabaseError.

    Exception messages of this class are generated by the database and will
    have a prefix such as ORA.
    """


class IntegrityError(DatabaseError):
    """
    Exception raised when the relational integrity of the database is affected.
    It is a subclass of DatabaseError.

    Exception messages of this class are generated by the database and will
    have a prefix such as ORA.
    """


class InterfaceError(Error):
    """
    Exception raised for errors that are related to the database interface
    rather than the database itself. It is a subclass of Error.

    Exception messages of this class will have the prefix DPY and an error
    number in the range 1000 - 1999.
    """


class InternalError(DatabaseError):
    """
    Exception raised when the database encounters an internal error. It is a
    subclass of DatabaseError.

    Exception messages of this class will have the prefix DPY and an error
    number in the range 5000 - 5999.
    """


class NotSupportedError(DatabaseError):
    """
    Exception raised when a method or database API was used which is not
    supported by the database. It is a subclass of DatabaseError.

    Exception messages of this class will have the prefix DPY and an error
    number in the range 3000 - 3999.
    """


class OperationalError(DatabaseError):
    """
    Exception raised for errors that are related to the operation of the
    database but are not necessarily under the control of the programmer. It is
    a subclass of DatabaseError.

    Exception messages of this class will have the prefix DPY and an error
    number in the range 6000 - 6999.
    """


class ProgrammingError(DatabaseError):
    """
    Exception raised for programming errors. It is a subclass of DatabaseError.

    Exception messages of this class will have the prefix DPY and an error
    number in the range 2000 - 2999.
    """
