/* Copyright (c) 2023, Google LLC
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE. */

#define OPENSSL_UNSTABLE_EXPERIMENTAL_DILITHIUM
#include <openssl/experimental/dilithium.h>

#include <memory>
#include <vector>

#include <gtest/gtest.h>

#include <openssl/bytestring.h>
#include <openssl/ctrdrbg.h>
#include <openssl/span.h>

#include "../internal.h"
#include "../test/file_test.h"
#include "../test/test_util.h"
#include "./internal.h"

OPENSSL_BEGIN_ALLOW_DEPRECATED

// This test is very slow, so it is disabled by default.
TEST(DilithiumTest, DISABLED_BitFlips) {
  std::vector<uint8_t> encoded_public_key(DILITHIUM_PUBLIC_KEY_BYTES);
  auto priv = std::make_unique<DILITHIUM_private_key>();
  EXPECT_TRUE(DILITHIUM_generate_key(encoded_public_key.data(), priv.get()));

  std::vector<uint8_t> encoded_signature(DILITHIUM_SIGNATURE_BYTES);
  static const uint8_t kMessage[] = {'H', 'e', 'l', 'l', 'o', ' ',
                                     'w', 'o', 'r', 'l', 'd'};
  EXPECT_TRUE(DILITHIUM_sign(encoded_signature.data(), priv.get(), kMessage,
                             sizeof(kMessage)));

  auto pub = std::make_unique<DILITHIUM_public_key>();
  CBS cbs = bssl::MakeConstSpan(encoded_public_key);
  ASSERT_TRUE(DILITHIUM_parse_public_key(pub.get(), &cbs));

  EXPECT_EQ(DILITHIUM_verify(pub.get(), encoded_signature.data(), kMessage,
                             sizeof(kMessage)),
            1);

  for (size_t i = 0; i < DILITHIUM_SIGNATURE_BYTES; i++) {
    for (int j = 0; j < 8; j++) {
      encoded_signature[i] ^= 1 << j;
      EXPECT_EQ(DILITHIUM_verify(pub.get(), encoded_signature.data(), kMessage,
                                 sizeof(kMessage)),
                0)
          << "Bit flip in signature at byte " << i << " bit " << j
          << " didn't cause a verification failure";
      encoded_signature[i] ^= 1 << j;
    }
  }
}

TEST(DilithiumTest, SignatureIsRandomized) {
  std::vector<uint8_t> encoded_public_key(DILITHIUM_PUBLIC_KEY_BYTES);
  auto priv = std::make_unique<DILITHIUM_private_key>();
  EXPECT_TRUE(DILITHIUM_generate_key(encoded_public_key.data(), priv.get()));

  auto pub = std::make_unique<DILITHIUM_public_key>();
  CBS cbs = bssl::MakeConstSpan(encoded_public_key);
  ASSERT_TRUE(DILITHIUM_parse_public_key(pub.get(), &cbs));

  std::vector<uint8_t> encoded_signature1(DILITHIUM_SIGNATURE_BYTES);
  std::vector<uint8_t> encoded_signature2(DILITHIUM_SIGNATURE_BYTES);
  static const uint8_t kMessage[] = {'H', 'e', 'l', 'l', 'o', ' ',
                                     'w', 'o', 'r', 'l', 'd'};
  EXPECT_TRUE(DILITHIUM_sign(encoded_signature1.data(), priv.get(), kMessage,
                             sizeof(kMessage)));
  EXPECT_TRUE(DILITHIUM_sign(encoded_signature2.data(), priv.get(), kMessage,
                             sizeof(kMessage)));

  EXPECT_NE(Bytes(encoded_signature1), Bytes(encoded_signature2));

  // Even though the signatures are different, they both verify.
  EXPECT_EQ(DILITHIUM_verify(pub.get(), encoded_signature1.data(), kMessage,
                             sizeof(kMessage)),
            1);
  EXPECT_EQ(DILITHIUM_verify(pub.get(), encoded_signature2.data(), kMessage,
                             sizeof(kMessage)),
            1);
}

TEST(DilithiumTest, PublicFromPrivateIsConsistent) {
  std::vector<uint8_t> encoded_public_key(DILITHIUM_PUBLIC_KEY_BYTES);
  auto priv = std::make_unique<DILITHIUM_private_key>();
  EXPECT_TRUE(DILITHIUM_generate_key(encoded_public_key.data(), priv.get()));

  auto pub = std::make_unique<DILITHIUM_public_key>();
  EXPECT_TRUE(DILITHIUM_public_from_private(pub.get(), priv.get()));

  std::vector<uint8_t> encoded_public_key2(DILITHIUM_PUBLIC_KEY_BYTES);

  CBB cbb;
  CBB_init_fixed(&cbb, encoded_public_key2.data(), encoded_public_key2.size());
  ASSERT_TRUE(DILITHIUM_marshal_public_key(&cbb, pub.get()));

  EXPECT_EQ(Bytes(encoded_public_key2), Bytes(encoded_public_key));
}

TEST(DilithiumTest, InvalidPublicKeyEncodingLength) {
  // Encode a public key with a trailing 0 at the end.
  std::vector<uint8_t> encoded_public_key(DILITHIUM_PUBLIC_KEY_BYTES + 1);
  auto priv = std::make_unique<DILITHIUM_private_key>();
  EXPECT_TRUE(DILITHIUM_generate_key(encoded_public_key.data(), priv.get()));

  // Public key is 1 byte too short.
  CBS cbs = bssl::MakeConstSpan(encoded_public_key)
                .first(DILITHIUM_PUBLIC_KEY_BYTES - 1);
  auto parsed_pub = std::make_unique<DILITHIUM_public_key>();
  EXPECT_FALSE(DILITHIUM_parse_public_key(parsed_pub.get(), &cbs));

  // Public key has the correct length.
  cbs =
      bssl::MakeConstSpan(encoded_public_key).first(DILITHIUM_PUBLIC_KEY_BYTES);
  EXPECT_TRUE(DILITHIUM_parse_public_key(parsed_pub.get(), &cbs));

  // Public key is 1 byte too long.
  cbs = bssl::MakeConstSpan(encoded_public_key);
  EXPECT_FALSE(DILITHIUM_parse_public_key(parsed_pub.get(), &cbs));
}

TEST(DilithiumTest, InvalidPrivateKeyEncodingLength) {
  std::vector<uint8_t> encoded_public_key(DILITHIUM_PUBLIC_KEY_BYTES);
  auto priv = std::make_unique<DILITHIUM_private_key>();
  EXPECT_TRUE(DILITHIUM_generate_key(encoded_public_key.data(), priv.get()));

  CBB cbb;
  std::vector<uint8_t> malformed_private_key(DILITHIUM_PRIVATE_KEY_BYTES + 1,
                                             0);
  CBB_init_fixed(&cbb, malformed_private_key.data(),
                 DILITHIUM_PRIVATE_KEY_BYTES);
  ASSERT_TRUE(DILITHIUM_marshal_private_key(&cbb, priv.get()));

  CBS cbs;
  auto parsed_priv = std::make_unique<DILITHIUM_private_key>();

  // Private key is 1 byte too short.
  CBS_init(&cbs, malformed_private_key.data(), DILITHIUM_PRIVATE_KEY_BYTES - 1);
  EXPECT_FALSE(DILITHIUM_parse_private_key(parsed_priv.get(), &cbs));

  // Private key has the correct length.
  CBS_init(&cbs, malformed_private_key.data(), DILITHIUM_PRIVATE_KEY_BYTES);
  EXPECT_TRUE(DILITHIUM_parse_private_key(parsed_priv.get(), &cbs));

  // Private key is 1 byte too long.
  CBS_init(&cbs, malformed_private_key.data(), DILITHIUM_PRIVATE_KEY_BYTES + 1);
  EXPECT_FALSE(DILITHIUM_parse_private_key(parsed_priv.get(), &cbs));
}

static void DilithiumFileTest(FileTest *t) {
  std::vector<uint8_t> seed, message, public_key_expected, private_key_expected,
      signed_message_expected;
  t->IgnoreAttribute("count");
  ASSERT_TRUE(t->GetBytes(&seed, "seed"));
  t->IgnoreAttribute("mlen");
  ASSERT_TRUE(t->GetBytes(&message, "msg"));
  ASSERT_TRUE(t->GetBytes(&public_key_expected, "pk"));
  ASSERT_TRUE(t->GetBytes(&private_key_expected, "sk"));
  t->IgnoreAttribute("smlen");
  ASSERT_TRUE(t->GetBytes(&signed_message_expected, "sm"));

  uint8_t gen_key_entropy[DILITHIUM_GENERATE_KEY_ENTROPY];
  // The test vectors provide a CTR-DRBG seed which is used to generate the
  // input entropy.
  ASSERT_EQ(seed.size(), size_t{CTR_DRBG_ENTROPY_LEN});
  {
    bssl::UniquePtr<CTR_DRBG_STATE> state(
        CTR_DRBG_new(seed.data(), nullptr, 0));
    ASSERT_TRUE(state);
    ASSERT_TRUE(CTR_DRBG_generate(state.get(), gen_key_entropy,
                                  DILITHIUM_GENERATE_KEY_ENTROPY, nullptr, 0));
    CONSTTIME_SECRET(gen_key_entropy, sizeof(gen_key_entropy));
  }

  // Reproduce key generation.
  auto parsed_pub = std::make_unique<DILITHIUM_public_key>();
  auto priv = std::make_unique<DILITHIUM_private_key>();
  std::vector<uint8_t> encoded_private_key(DILITHIUM_PRIVATE_KEY_BYTES);
  std::vector<uint8_t> encoded_public_key(DILITHIUM_PUBLIC_KEY_BYTES);

  EXPECT_TRUE(DILITHIUM_generate_key_external_entropy(
      encoded_public_key.data(), priv.get(), gen_key_entropy));

  CBB cbb;
  CBB_init_fixed(&cbb, encoded_private_key.data(), encoded_private_key.size());
  ASSERT_TRUE(DILITHIUM_marshal_private_key(&cbb, priv.get()));
  CONSTTIME_DECLASSIFY(encoded_private_key.data(), encoded_private_key.size());

  EXPECT_EQ(Bytes(encoded_public_key), Bytes(public_key_expected));
  EXPECT_EQ(Bytes(encoded_private_key), Bytes(private_key_expected));

  // Reproduce signature.
  std::vector<uint8_t> encoded_signature(DILITHIUM_SIGNATURE_BYTES);
  EXPECT_TRUE(DILITHIUM_sign_deterministic(encoded_signature.data(), priv.get(),
                                           message.data(), message.size()));

  ASSERT_GE(signed_message_expected.size(), size_t{DILITHIUM_SIGNATURE_BYTES});
  EXPECT_EQ(Bytes(encoded_signature),
            Bytes(bssl::MakeConstSpan(signed_message_expected)
                      .first(DILITHIUM_SIGNATURE_BYTES)));
  EXPECT_EQ(Bytes(message), Bytes(bssl::MakeConstSpan(signed_message_expected)
                                      .subspan(DILITHIUM_SIGNATURE_BYTES)));

  // Check that verification matches.
  CBS cbs;
  CBS_init(&cbs, encoded_public_key.data(), encoded_public_key.size());
  ASSERT_TRUE(DILITHIUM_parse_public_key(parsed_pub.get(), &cbs));
  EXPECT_EQ(DILITHIUM_verify(parsed_pub.get(), encoded_signature.data(),
                             message.data(), message.size()),
            1);

  // Test that parsing the encoded private key yields a functional object.
  auto parsed_priv = std::make_unique<DILITHIUM_private_key>();
  std::vector<uint8_t> encoded_signature2(DILITHIUM_SIGNATURE_BYTES);

  CBS_init(&cbs, encoded_private_key.data(), encoded_private_key.size());
  ASSERT_TRUE(DILITHIUM_parse_private_key(parsed_priv.get(), &cbs));

  EXPECT_TRUE(DILITHIUM_sign_deterministic(encoded_signature2.data(),
                                           parsed_priv.get(), message.data(),
                                           message.size()));
  EXPECT_EQ(Bytes(encoded_signature2), Bytes(encoded_signature));

  // Test that parsing + encoding is idempotent.
  std::vector<uint8_t> encoded_private_key2(DILITHIUM_PRIVATE_KEY_BYTES);
  std::vector<uint8_t> encoded_public_key2(DILITHIUM_PUBLIC_KEY_BYTES);

  CBB_init_fixed(&cbb, encoded_private_key2.data(),
                 encoded_private_key2.size());
  ASSERT_TRUE(DILITHIUM_marshal_private_key(&cbb, parsed_priv.get()));
  CBB_init_fixed(&cbb, encoded_public_key2.data(), encoded_public_key2.size());
  ASSERT_TRUE(DILITHIUM_marshal_public_key(&cbb, parsed_pub.get()));

  EXPECT_EQ(Bytes(encoded_public_key2), Bytes(encoded_public_key));
  EXPECT_EQ(Bytes(encoded_private_key2), Bytes(encoded_private_key));
}

static void DilithiumSignFileTest(FileTest *t) {
  std::string description, valid;
  std::vector<uint8_t> message, private_key, signed_message_expected;
  ASSERT_TRUE(t->GetAttribute(&description, "description"));
  t->IgnoreAttribute("mlen");
  ASSERT_TRUE(t->GetBytes(&message, "msg"));
  ASSERT_TRUE(t->GetBytes(&private_key, "sk"));
  ASSERT_TRUE(t->GetAttribute(&valid, "valid"));
  t->IgnoreAttribute("smlen");
  ASSERT_TRUE(t->GetBytes(&signed_message_expected, "sm"));

  bool is_valid = valid == "true";

  // Parse private key.
  DILITHIUM_private_key priv;

  CBS cbs;
  CBS_init(&cbs, private_key.data(), private_key.size());
  if (!DILITHIUM_parse_private_key(&priv, &cbs)) {
    EXPECT_FALSE(is_valid) << "Unexpected signing result for edge case: "
                           << description;
    return;
  }

  // Reproduce signature.
  uint8_t encoded_signature[DILITHIUM_SIGNATURE_BYTES];
  DILITHIUM_sign_deterministic(encoded_signature, &priv, message.data(),
                               message.size());

  ASSERT_GE(signed_message_expected.size(), size_t{DILITHIUM_SIGNATURE_BYTES});
  EXPECT_EQ(Bytes(encoded_signature),
            Bytes(signed_message_expected.data(), DILITHIUM_SIGNATURE_BYTES))
      << "Unexpected signing result for edge case: " << description;
  EXPECT_EQ(Bytes(message),
            Bytes(&signed_message_expected[DILITHIUM_SIGNATURE_BYTES],
                  signed_message_expected.size() - DILITHIUM_SIGNATURE_BYTES))
      << "Unexpected signing result for edge case: " << description;
  EXPECT_TRUE(is_valid) << "Unexpected signing result for edge case: "
                        << description;
}

static void DilithiumVerifyFileTest(FileTest *t) {
  std::string description, valid;
  std::vector<uint8_t> message, public_key, signed_message;
  ASSERT_TRUE(t->GetAttribute(&description, "description"));
  t->IgnoreAttribute("mlen");
  ASSERT_TRUE(t->GetBytes(&message, "msg"));
  ASSERT_TRUE(t->GetBytes(&public_key, "pk"));
  ASSERT_TRUE(t->GetAttribute(&valid, "valid"));
  t->IgnoreAttribute("smlen");
  ASSERT_TRUE(t->GetBytes(&signed_message, "sm"));

  bool is_valid = valid == "true";

  // Parse public key.
  DILITHIUM_public_key pub;

  CBS cbs;
  CBS_init(&cbs, public_key.data(), public_key.size());
  if (!DILITHIUM_parse_public_key(&pub, &cbs)) {
    EXPECT_FALSE(is_valid) << "Unexpected verification result for edge case: "
                           << description;
    return;
  }

  // Verify signature.
  ASSERT_GE(signed_message.size(), size_t{DILITHIUM_SIGNATURE_BYTES});
  EXPECT_EQ(DILITHIUM_verify(&pub, signed_message.data(),
                             &signed_message[DILITHIUM_SIGNATURE_BYTES],
                             signed_message.size() - DILITHIUM_SIGNATURE_BYTES),
            is_valid)
      << "Unexpected verification result for edge case: " << description;
}

TEST(DilithiumTest, TestVectors) {
  FileTestGTest("crypto/dilithium/dilithium_tests.txt", DilithiumFileTest);
}

TEST(DilithiumTest, EdgeCaseSigningTests) {
  FileTestGTest("crypto/dilithium/edge_cases_draft_dilithium3_sign.txt",
                DilithiumSignFileTest);
}

TEST(DilithiumTest, EdgeCaseVerifyTests) {
  FileTestGTest("crypto/dilithium/edge_cases_draft_dilithium3_verify.txt",
                DilithiumVerifyFileTest);
}

TEST(DilithiumTest, KeyGenerationHardCodedNIST) {
  // Published on
  // https://csrc.nist.gov/Projects/post-quantum-cryptography/post-quantum-cryptography-standardization/example-files
  uint8_t entropy[DILITHIUM_GENERATE_KEY_ENTROPY] = {
      0x72, 0xC3, 0xC5, 0xE0, 0xCC, 0x9F, 0x33, 0x2F, 0x49, 0xD0, 0xFC,
      0x0F, 0xD6, 0x39, 0x9D, 0xA7, 0x56, 0x45, 0xA3, 0xE3, 0x3D, 0xBF,
      0x56, 0xF1, 0xE9, 0x68, 0x97, 0x66, 0x2D, 0x0A, 0x9B, 0x37};
  std::vector<uint8_t> encoded_public_key(DILITHIUM_PUBLIC_KEY_BYTES);
  auto priv = std::make_unique<DILITHIUM_private_key>();
  EXPECT_TRUE(DILITHIUM_generate_key_external_entropy(encoded_public_key.data(),
                                                      priv.get(), entropy));

  static const uint8_t expected_public_key[DILITHIUM_PUBLIC_KEY_BYTES] = {
      0x88, 0xC9, 0x3D, 0xEB, 0x90, 0x1A, 0x24, 0xA9, 0x45, 0xFA, 0x94, 0xAE,
      0x54, 0xDB, 0xAF, 0x46, 0xD8, 0x1C, 0xAC, 0xA9, 0x43, 0xFB, 0xE3, 0x20,
      0x42, 0x1C, 0x61, 0xB3, 0x32, 0x04, 0x84, 0x9F, 0x65, 0xE8, 0x33, 0xB5,
      0xEB, 0x94, 0xC6, 0x0E, 0x74, 0xA6, 0x0D, 0xD2, 0xC8, 0xEA, 0x1D, 0xE7,
      0xFE, 0x99, 0x60, 0x93, 0x55, 0x87, 0x5F, 0xFC, 0x40, 0x0E, 0xCD, 0x83,
      0xF1, 0x2C, 0xDD, 0xD1, 0xB0, 0x15, 0xAC, 0x75, 0x0D, 0xF7, 0x75, 0x9D,
      0x3D, 0x0F, 0xEC, 0xE7, 0x95, 0x42, 0xF1, 0x2B, 0xE3, 0x0A, 0x47, 0x39,
      0x21, 0x49, 0x9A, 0x92, 0x7C, 0xDE, 0xED, 0xF9, 0xC6, 0x2F, 0x0A, 0x75,
      0x58, 0x35, 0xB9, 0xFF, 0x46, 0x40, 0x40, 0xEF, 0x40, 0x97, 0x84, 0xBA,
      0x7D, 0x50, 0x91, 0xBC, 0x92, 0xC7, 0xA0, 0xC6, 0x23, 0xDE, 0xC0, 0x17,
      0xE9, 0x5C, 0x3A, 0xD4, 0x3C, 0x86, 0x5B, 0x59, 0x5D, 0x04, 0x09, 0x16,
      0x29, 0xA5, 0x61, 0x14, 0xA3, 0x19, 0x7B, 0x0A, 0x21, 0x17, 0xE9, 0xEC,
      0xC8, 0xF3, 0xE4, 0x8E, 0x1E, 0x9C, 0x77, 0x2B, 0xA6, 0x1E, 0x5A, 0xE3,
      0x13, 0x81, 0x89, 0xDF, 0x4A, 0x9E, 0x14, 0x92, 0x91, 0xA6, 0x69, 0x4A,
      0xBB, 0x10, 0x05, 0xCF, 0xA8, 0xED, 0x76, 0x6D, 0xCD, 0x8C, 0x8C, 0xF7,
      0x2A, 0xE9, 0xC4, 0x55, 0x03, 0xBB, 0x71, 0x65, 0xB8, 0x41, 0x40, 0xB0,
      0xCB, 0x5C, 0xBD, 0x33, 0xFF, 0xC4, 0x3E, 0x63, 0x84, 0xCD, 0x90, 0x59,
      0x95, 0xB5, 0xA7, 0x87, 0xFE, 0x48, 0x4B, 0xFE, 0x17, 0xD2, 0xBF, 0x30,
      0x25, 0x0F, 0x78, 0xBE, 0x5C, 0xEA, 0x26, 0xE2, 0x01, 0xFF, 0x1F, 0x20,
      0x57, 0x4B, 0xB6, 0x77, 0xF4, 0xD2, 0x08, 0xE7, 0xC5, 0xB9, 0x6C, 0xAB,
      0xD6, 0x41, 0x6E, 0xEC, 0x1E, 0x21, 0xE0, 0x81, 0x6F, 0x35, 0x2C, 0x84,
      0x8E, 0xEA, 0xA4, 0x7C, 0xDD, 0x8E, 0x02, 0xB4, 0x71, 0x58, 0xBC, 0xB4,
      0x44, 0x51, 0xD3, 0x60, 0xAC, 0xBD, 0x9D, 0xCF, 0x52, 0x3E, 0xB9, 0x69,
      0x4F, 0xD5, 0xE1, 0x55, 0x7D, 0xA8, 0xF2, 0xBE, 0xED, 0x0A, 0x0A, 0xEE,
      0x52, 0xE2, 0x3A, 0xEF, 0x1B, 0xEB, 0x44, 0x9D, 0x91, 0x32, 0x76, 0xE8,
      0x8A, 0xF1, 0x88, 0x15, 0x1F, 0x68, 0x76, 0xE4, 0xA5, 0xF5, 0x70, 0xA2,
      0x40, 0xA6, 0x95, 0xB0, 0x6E, 0x4D, 0x60, 0x13, 0x2D, 0xA2, 0xB6, 0x2F,
      0x4C, 0xAB, 0x5C, 0x88, 0x1F, 0x72, 0x77, 0xAC, 0xD1, 0xDA, 0x48, 0x68,
      0x73, 0x68, 0xCB, 0x8E, 0xD4, 0x9F, 0xAE, 0x2A, 0x4B, 0x2E, 0x5E, 0xF3,
      0xDE, 0x02, 0x60, 0xA7, 0x31, 0xD6, 0x59, 0xA3, 0xF4, 0x2A, 0x19, 0xF4,
      0x66, 0xC9, 0xF0, 0xD4, 0x3A, 0x10, 0x2B, 0x12, 0x9C, 0xDA, 0xE4, 0xA6,
      0xBF, 0x08, 0x41, 0x96, 0xD6, 0xFE, 0xA7, 0xEE, 0xB3, 0x83, 0x36, 0xEF,
      0x1F, 0x25, 0x65, 0x02, 0xEE, 0xC8, 0x0A, 0x49, 0x2F, 0x0A, 0x4F, 0x8A,
      0x18, 0x23, 0x45, 0xCC, 0x78, 0x97, 0x8C, 0x06, 0x1C, 0x56, 0x3D, 0x03,
      0xE0, 0x7A, 0xC6, 0xFC, 0x2A, 0x1C, 0xF5, 0xCB, 0x8D, 0x25, 0xC0, 0x48,
      0x87, 0xE5, 0xD1, 0x13, 0x91, 0xE0, 0xB3, 0xCD, 0x1D, 0xB1, 0x05, 0xC7,
      0x34, 0x5C, 0xB3, 0xC8, 0x26, 0xFF, 0x05, 0x8C, 0x30, 0xE0, 0x22, 0x8C,
      0x3A, 0x85, 0x7B, 0x7C, 0xE2, 0xF0, 0x64, 0xA5, 0x08, 0xAE, 0x80, 0xE3,
      0x3C, 0xE2, 0xCA, 0x82, 0xFE, 0x36, 0x24, 0x6D, 0x61, 0x57, 0x8C, 0x52,
      0xD6, 0x9A, 0x8F, 0x8B, 0x84, 0x4F, 0x99, 0x7D, 0x8B, 0x4A, 0x33, 0x41,
      0xC9, 0x16, 0x2F, 0xF5, 0x32, 0x02, 0x28, 0xDD, 0x59, 0xC0, 0x91, 0x1F,
      0xE4, 0x79, 0xE7, 0x2F, 0xF6, 0xAB, 0x43, 0xF4, 0xA7, 0xB9, 0xF8, 0x05,
      0xA8, 0x6F, 0x98, 0x45, 0x12, 0xF9, 0x73, 0xD0, 0x91, 0x37, 0x0F, 0x68,
      0x1F, 0x49, 0x82, 0x72, 0x96, 0x96, 0x43, 0x41, 0xAC, 0xAE, 0xD3, 0xE3,
      0x11, 0xA3, 0x2B, 0xEC, 0x51, 0x74, 0xA7, 0x76, 0x89, 0xA4, 0x92, 0x1D,
      0x02, 0x9F, 0x21, 0xF6, 0xC4, 0x0E, 0xA5, 0x5F, 0x1E, 0x73, 0xC1, 0xAF,
      0xF2, 0x14, 0x3A, 0xCD, 0x3E, 0x50, 0x37, 0xFC, 0xD5, 0xB3, 0x11, 0xCD,
      0x82, 0x5F, 0x33, 0x26, 0x71, 0x0C, 0x32, 0x63, 0xA0, 0xCF, 0x60, 0x16,
      0xD9, 0xEA, 0xE3, 0xE5, 0x0F, 0xE2, 0x57, 0x5E, 0x0A, 0xAB, 0xED, 0xFC,
      0xB3, 0x49, 0x5B, 0x9E, 0x77, 0x76, 0x4E, 0x97, 0xA2, 0x63, 0xD0, 0x94,
      0x80, 0x6F, 0xA1, 0x24, 0xA8, 0xC1, 0x53, 0xE5, 0x97, 0x5D, 0x35, 0xFA,
      0x42, 0x59, 0x76, 0x83, 0xC9, 0x5D, 0x83, 0x80, 0x67, 0x02, 0x8E, 0xBD,
      0xA9, 0x76, 0x58, 0xE6, 0x44, 0x78, 0x58, 0x0C, 0xEC, 0xAA, 0x77, 0x2E,
      0xC6, 0xCA, 0x31, 0x9F, 0x11, 0x80, 0x3A, 0x64, 0x8D, 0x6A, 0xF0, 0xF9,
      0x48, 0x4B, 0x15, 0x82, 0xDE, 0x0C, 0x72, 0xAF, 0x6C, 0x49, 0x12, 0xEA,
      0x79, 0x15, 0xFF, 0xEC, 0x90, 0x0A, 0x95, 0xDE, 0x23, 0x5D, 0xEF, 0xFC,
      0x4F, 0x75, 0x4B, 0xED, 0x72, 0x4E, 0x36, 0x12, 0xF0, 0x69, 0x6A, 0x1A,
      0xE1, 0xFF, 0x05, 0xE8, 0xBE, 0x25, 0x90, 0x02, 0x9C, 0xC0, 0xD3, 0x4E,
      0xDD, 0x83, 0x98, 0x25, 0x0B, 0xC0, 0xD3, 0x91, 0xDE, 0xE8, 0x71, 0xA5,
      0x4F, 0x15, 0x68, 0x5D, 0x9C, 0xD6, 0x7F, 0x85, 0xEB, 0x1B, 0xB5, 0x0F,
      0x39, 0xA8, 0xA1, 0x94, 0x15, 0xF8, 0x63, 0x52, 0x74, 0x30, 0xF4, 0xEF,
      0xC2, 0x97, 0xBC, 0x08, 0xA0, 0x3D, 0xB9, 0xDF, 0x86, 0xD2, 0x4C, 0x6D,
      0x0F, 0x63, 0xD0, 0x39, 0x9D, 0x2E, 0x86, 0x42, 0x63, 0x31, 0x57, 0x4D,
      0x1C, 0x67, 0x98, 0xFB, 0xAF, 0x46, 0x33, 0xCF, 0xBE, 0x10, 0xB1, 0x78,
      0x90, 0x33, 0x09, 0x65, 0x90, 0x6E, 0x5F, 0x8B, 0x01, 0xDB, 0x12, 0x5C,
      0x0F, 0xEB, 0xAE, 0x1F, 0x3A, 0x49, 0x8F, 0x45, 0xBF, 0x24, 0x32, 0x67,
      0xD9, 0xBA, 0xD0, 0x96, 0xAE, 0x0F, 0x9C, 0x15, 0x9A, 0x2B, 0x4D, 0x11,
      0xBF, 0x6A, 0xDB, 0x35, 0xAA, 0x7C, 0xCC, 0x3A, 0x17, 0xE7, 0xCE, 0x14,
      0xB5, 0x64, 0x82, 0x81, 0xF8, 0x56, 0xDC, 0x7D, 0x29, 0xBE, 0x92, 0xF5,
      0x60, 0xC1, 0x05, 0xDF, 0xC2, 0x68, 0x4D, 0x2D, 0x5D, 0x16, 0x25, 0x7E,
      0x22, 0xDB, 0xA9, 0x9C, 0x93, 0x00, 0x2B, 0xE2, 0x4F, 0xC1, 0x45, 0x44,
      0x0E, 0xF8, 0xF7, 0x7B, 0x2A, 0xDA, 0xE7, 0xEF, 0xD6, 0x76, 0x62, 0x00,
      0xBE, 0xD5, 0xAF, 0x94, 0xFD, 0x9F, 0x23, 0xAD, 0xEB, 0x7E, 0xF3, 0xF3,
      0xE7, 0x86, 0xE5, 0xD5, 0xFB, 0xC8, 0x3F, 0x09, 0x12, 0x0B, 0xD1, 0x59,
      0x35, 0xDA, 0xC6, 0x88, 0x47, 0xF5, 0x07, 0xE4, 0x56, 0x10, 0x69, 0x91,
      0x5E, 0x02, 0xA6, 0x53, 0x96, 0xAC, 0xAB, 0xCB, 0xDB, 0xA9, 0x48, 0x12,
      0x27, 0xC0, 0x26, 0x1C, 0x7F, 0xA8, 0x03, 0x92, 0xEE, 0xCE, 0x16, 0x2F,
      0xAF, 0x98, 0xE7, 0xCC, 0x53, 0x56, 0xFD, 0x0A, 0xCB, 0x3F, 0x5F, 0x9B,
      0x7B, 0x60, 0xEA, 0x43, 0x66, 0x4D, 0x05, 0x3C, 0xA4, 0x18, 0x65, 0xD0,
      0x68, 0x85, 0x4A, 0xEE, 0xF2, 0xAD, 0x80, 0xA5, 0x62, 0x6D, 0xF6, 0x6A,
      0xC0, 0x03, 0x2B, 0x22, 0x51, 0x97, 0xD5, 0x2C, 0x7A, 0x55, 0xF6, 0xDD,
      0x56, 0xD3, 0x75, 0xA4, 0x4D, 0xC2, 0x34, 0xE3, 0x55, 0xE6, 0xB7, 0xF1,
      0x00, 0xCD, 0x5D, 0xF2, 0xE3, 0x07, 0xC2, 0x0C, 0x8E, 0xF4, 0x4C, 0x77,
      0x97, 0x6A, 0xBF, 0xC4, 0x54, 0x2A, 0x0E, 0xBF, 0xBD, 0xB7, 0x9A, 0xFF,
      0x5E, 0x79, 0xB4, 0x9A, 0x3F, 0x08, 0xEE, 0x63, 0xC0, 0xF7, 0x8D, 0x12,
      0x4B, 0x62, 0x1B, 0x7B, 0xF4, 0xBD, 0x35, 0x50, 0xAD, 0x63, 0x24, 0x6F,
      0xAF, 0x74, 0x8C, 0x46, 0xE9, 0x40, 0x91, 0xE4, 0xCC, 0x1A, 0x98, 0x1B,
      0x41, 0x46, 0x4B, 0xD2, 0x7C, 0xF6, 0xED, 0xA5, 0xDE, 0xD5, 0xE9, 0x2C,
      0x0A, 0x83, 0x8E, 0x59, 0xE2, 0xEC, 0x6F, 0xD2, 0x0C, 0x02, 0x59, 0x0E,
      0xA2, 0x80, 0x7E, 0x1B, 0xEE, 0xBA, 0xA9, 0x1B, 0x5A, 0x5F, 0x03, 0x12,
      0x54, 0x3F, 0x6F, 0xC6, 0x3C, 0x0B, 0x82, 0x24, 0x15, 0x32, 0xD2, 0xC4,
      0x97, 0xEC, 0xEE, 0x54, 0xC0, 0xE6, 0x5E, 0x4F, 0x93, 0x83, 0x6A, 0x8A,
      0x69, 0x09, 0x42, 0x67, 0x3A, 0x6C, 0x61, 0x11, 0x39, 0x66, 0xCA, 0x56,
      0x6C, 0x2C, 0xE9, 0x60, 0x6B, 0xD8, 0x97, 0xB9, 0x00, 0xD7, 0x66, 0x1B,
      0x9A, 0x0C, 0xB5, 0x36, 0x0C, 0x0A, 0x8D, 0x65, 0x5F, 0x7A, 0x5D, 0x99,
      0x91, 0x36, 0xAD, 0x8E, 0xCD, 0xDA, 0xE0, 0x6C, 0x11, 0xF2, 0x14, 0x4D,
      0xD6, 0x45, 0x29, 0xB6, 0x2C, 0xB1, 0xF1, 0x29, 0x4D, 0xF2, 0x22, 0xBF,
      0xA6, 0xE5, 0x78, 0x0F, 0xEC, 0x07, 0xFB, 0xA3, 0xF7, 0xEC, 0xA8, 0x6B,
      0x59, 0x3C, 0x26, 0x11, 0xA7, 0x45, 0xBC, 0x20, 0x6F, 0xE3, 0x07, 0xD4,
      0x10, 0x3A, 0x3B, 0x68, 0xD3, 0xCA, 0x41, 0xA7, 0x72, 0x95, 0x41, 0x1D,
      0x44, 0x85, 0x3B, 0x08, 0xF5, 0x2B, 0xC7, 0x46, 0x78, 0xE6, 0xC2, 0x2C,
      0x17, 0x00, 0x76, 0x45, 0x64, 0xF3, 0xA3, 0x91, 0xF7, 0x87, 0xFF, 0x5F,
      0x97, 0xA9, 0x50, 0xC7, 0x2F, 0x27, 0x1F, 0x49, 0x3C, 0xF6, 0x4E, 0x80,
      0x89, 0x25, 0x23, 0x12, 0x09, 0xB8, 0xBE, 0x77, 0xC8, 0x62, 0xC7, 0x7B,
      0x7C, 0x04, 0xC3, 0x61, 0x1C, 0x86, 0x39, 0x7A, 0xF4, 0x9D, 0xAF, 0x0B,
      0xAA, 0x4E, 0xDE, 0x81, 0x71, 0x45, 0x86, 0xC0, 0xA6, 0xC5, 0x21, 0x0E,
      0x46, 0xAB, 0x49, 0xAA, 0x56, 0xAD, 0x0D, 0xC8, 0x6B, 0x02, 0x07, 0xE5,
      0xA2, 0xF8, 0xDE, 0x98, 0x85, 0xB8, 0x40, 0x76, 0xC3, 0x10, 0xD0, 0x9C,
      0xE3, 0x24, 0xE9, 0xCC, 0x27, 0x56, 0x87, 0xB0, 0x54, 0xA0, 0xB4, 0xC6,
      0x19, 0x76, 0x0A, 0x5F, 0x3C, 0x4E, 0x73, 0xA6, 0x1C, 0xDF, 0x37, 0xC2,
      0x76, 0x26, 0x1D, 0xB1, 0xE0, 0x7F, 0x53, 0x61, 0x04, 0x4E, 0xEC, 0x1E,
      0x69, 0xCC, 0xA9, 0xBD, 0x48, 0xC3, 0xAD, 0xCF, 0x68, 0x0B, 0x9E, 0x47,
      0xD1, 0x5A, 0x2F, 0x39, 0x23, 0x54, 0x37, 0xCD, 0x40, 0xE7, 0xC9, 0x54,
      0x1A, 0x06, 0xC0, 0xDA, 0xB1, 0x3A, 0x30, 0x29, 0x43, 0x38, 0x56, 0x1F,
      0xB3, 0x26, 0xE8, 0x27, 0x5E, 0xD2, 0x83, 0x2F, 0x77, 0xD9, 0x5C, 0x63,
      0x91, 0x5C, 0x06, 0x9F, 0xD4, 0xF5, 0x9A, 0xFF, 0x38, 0x84, 0x58, 0xB9,
      0xB3, 0x9E, 0x4E, 0x0E, 0x24, 0xC9, 0x93, 0x24, 0x77, 0xDD, 0x52, 0xFF,
      0x83, 0xDB, 0xDD, 0x24, 0xF2, 0xB6, 0x39, 0xFF, 0x9A, 0x32, 0x15, 0x66,
      0x73, 0xDF, 0x05, 0x1C, 0x35, 0xB6, 0xF4, 0xF3, 0x2F, 0x21, 0x78, 0xB2,
      0xEC, 0xD0, 0x9E, 0x7E, 0x9A, 0xFD, 0x4B, 0x62, 0xDD, 0x55, 0x16, 0xA0,
      0xE2, 0xBE, 0x2C, 0x89, 0x53, 0x87, 0x28, 0x4B, 0xAF, 0x82, 0x8F, 0x88,
      0x6C, 0x0C, 0xF4, 0x67, 0x40, 0x5B, 0x0A, 0x67, 0xFA, 0xF7, 0x6E, 0x4D,
      0x1A, 0xD9, 0x33, 0x11, 0xC8, 0xAF, 0x07, 0xEE, 0xF1, 0x6F, 0x17, 0x29,
      0xF6, 0x06, 0x46, 0xF3, 0xDB, 0xB9, 0x2D, 0xB0, 0x20, 0x44, 0xF9, 0xFE,
      0x56, 0x33, 0x38, 0xDA, 0xC3, 0x0E, 0xC1, 0x65, 0x8A, 0xFF, 0xEA, 0xAC,
      0xC7, 0xBC, 0x42, 0x12, 0x58, 0x8C, 0x32, 0x25, 0x19, 0xB3, 0xB9, 0x0B,
      0x66, 0x55, 0xC2, 0xEC, 0x50, 0x9C, 0x73, 0xE9, 0x9E, 0x4B, 0x27, 0x50,
      0xBB, 0x71, 0x1B, 0xBC, 0xD3, 0x9F, 0xBD, 0xC5, 0x39, 0x9C, 0xA9, 0x3E,
      0xFE, 0xEF, 0x0C, 0xE8, 0x2E, 0x51, 0xD0, 0x8A, 0xC3, 0x50, 0xBC, 0x0B,
      0x5E, 0x50, 0x38, 0x35, 0xEB, 0xE1, 0xB6, 0xF8, 0x4C, 0x42, 0xE1, 0xAB,
      0x43, 0xC9, 0xD8, 0x86, 0xD1, 0x4A, 0xFA, 0xAB, 0x89, 0xA0, 0xBA, 0x4A,
      0x7C, 0x94, 0x9B, 0x9D, 0x11, 0x5C, 0x7F, 0x83, 0x7D, 0x74, 0x95, 0xA7,
      0x80, 0x10, 0x3B, 0xE8, 0xBE, 0x9F, 0x51, 0x4A, 0x34, 0x97, 0x03, 0xFD,
      0x0E, 0x41, 0x8D, 0xBE, 0x88, 0xF2, 0x27, 0x88, 0x0C, 0xE5, 0xB2, 0xDD,
      0x54, 0xC0, 0xBC, 0xE1, 0xFF, 0x61, 0x89, 0xC3, 0xCA, 0x89, 0x21, 0x6C,
      0xDD, 0x23, 0x0D, 0x59, 0x87, 0xBC, 0xD4, 0x6C, 0x3B, 0x6C, 0x2D, 0xE7,
      0xEE, 0x54, 0xB7, 0xDC, 0x96, 0x12, 0x3B, 0xAD, 0x7A, 0x9E, 0x2F, 0x92,
      0x6A, 0xF1, 0x95, 0x2F, 0x42, 0x4F, 0x39, 0xC8, 0x90, 0xD6, 0x1A, 0x9D,
      0xA8, 0xCF, 0xEB, 0xE1, 0x77, 0x08, 0x5D, 0x1A, 0x87, 0x60, 0x87, 0x6F,
      0x68, 0x64, 0x8A, 0x82, 0xDD, 0xB7, 0x99, 0x95, 0x1A, 0x10, 0x1B, 0x82,
      0x8F, 0x57, 0xE0, 0xCF, 0xE0, 0xFA, 0xC6, 0xD9, 0x08, 0x59, 0x62, 0xCF,
      0x45, 0x11, 0x3C, 0xC5, 0xBF, 0x6D, 0x7F, 0xC2, 0xAC, 0xBC, 0xEE, 0xB5,
      0x56, 0xDF, 0x2B, 0xC3, 0xD2, 0xFE, 0x67, 0xA8, 0x63, 0x31, 0x21, 0x34,
      0xE7, 0x3E, 0x50, 0x6D, 0xA9, 0xDC, 0x78, 0xB7, 0xC9, 0x43, 0x47, 0x21,
      0x7E, 0xAB, 0xB7, 0x3E, 0x24, 0x98, 0xD5, 0xF3, 0xA5, 0x29, 0x7F, 0x18,
      0x59, 0xBD, 0x13, 0x74, 0x8C, 0xDB, 0x3B, 0x70, 0xE5, 0x19, 0x71, 0xCC,
      0x01, 0xF0, 0x5B, 0x04, 0xEB, 0x08, 0x02, 0x28, 0xD5, 0x95, 0x9C, 0xC8,
      0xFE, 0x59, 0x61, 0xA0, 0x20, 0x92, 0xEC, 0x8D, 0x0A, 0x4D, 0x16, 0x80,
      0x6E, 0xB5, 0xD6, 0x08, 0x64, 0xC3, 0xEC, 0xF8, 0x01, 0x24, 0x42, 0x94,
      0x02, 0x2A, 0x08, 0x23, 0x87, 0x3D, 0xBE, 0x8D, 0xFB, 0x6D, 0x06, 0xB6,
      0xBA, 0xF6, 0x9C, 0xDD, 0x10, 0x7C, 0x96, 0x0A, 0xA4, 0xB2, 0xC3, 0x53,
      0x6B, 0xB4, 0x42, 0x46, 0x13, 0x74, 0xF8, 0xFC, 0xB0, 0x48, 0x32, 0x64,
      0x8B, 0x60, 0x03, 0x28, 0x64, 0xFA, 0x6F, 0x0A, 0xF7, 0x6F, 0x34, 0x35,
      0x5B, 0x93, 0x43, 0xDC, 0xF3, 0xC8, 0xF9, 0x9F, 0xED, 0xD8, 0x8C, 0xA3,
      0xD8, 0x3C, 0xAD, 0x67, 0x35, 0x1B, 0x45, 0x62, 0x06, 0x4E, 0xA1, 0x32,
      0x51, 0x80, 0x60, 0x63, 0x62, 0xD7, 0x9A, 0x76, 0xA1, 0xB3, 0xC6, 0x76,
      0x15, 0x2B, 0xA8, 0x0E, 0x82, 0xB8, 0x43, 0xD8, 0x5D, 0x53, 0x7E, 0xCE,
      0xFF, 0xDD, 0xCC, 0x95, 0xB4, 0x66, 0x1C, 0xFA, 0x4A, 0x70, 0xB0, 0x6F,
      0x92, 0x4D, 0x69, 0x2D, 0x87, 0xE1, 0xFB, 0x8E, 0x1C, 0xED, 0xD8, 0x5F,
      0x7A, 0xB9, 0x32, 0x4F, 0xF5, 0xD0, 0xA7, 0x53, 0xD8, 0x62, 0x59, 0x1B,
      0xE2, 0x39, 0xAB, 0xC9, 0x63, 0xB5, 0x0A, 0x7E, 0xF8, 0x70, 0xB0, 0x5F,
      0x27, 0x21, 0x21, 0x5E, 0xB9, 0x0A, 0x60, 0xE4, 0x66, 0x95, 0x5D, 0x52,
      0x2C, 0xA4, 0x21, 0x69, 0x44, 0x69, 0x35, 0x46};
  EXPECT_EQ(Bytes(encoded_public_key), Bytes(expected_public_key));

  static const uint8_t expected_private_key[DILITHIUM_PRIVATE_KEY_BYTES] = {
      0x88, 0xC9, 0x3D, 0xEB, 0x90, 0x1A, 0x24, 0xA9, 0x45, 0xFA, 0x94, 0xAE,
      0x54, 0xDB, 0xAF, 0x46, 0xD8, 0x1C, 0xAC, 0xA9, 0x43, 0xFB, 0xE3, 0x20,
      0x42, 0x1C, 0x61, 0xB3, 0x32, 0x04, 0x84, 0x9F, 0x9B, 0x2E, 0x34, 0x8E,
      0x48, 0x65, 0x61, 0x66, 0x80, 0xFC, 0x18, 0x43, 0x76, 0xA8, 0xCC, 0xB8,
      0xDB, 0xE5, 0x98, 0x36, 0x5A, 0x23, 0xA2, 0x72, 0x39, 0x44, 0xF5, 0x35,
      0x1F, 0x42, 0xF4, 0x74, 0x20, 0x77, 0x58, 0x21, 0x02, 0xF6, 0xF3, 0x13,
      0xC2, 0x25, 0x64, 0xF8, 0x41, 0x4E, 0x37, 0x66, 0x70, 0x4F, 0xFD, 0x18,
      0x01, 0x7F, 0xDC, 0xAE, 0x57, 0x0D, 0xB8, 0xC8, 0xFC, 0xB5, 0x67, 0xE4,
      0x13, 0x21, 0x9C, 0xB4, 0xB4, 0x51, 0x75, 0xB8, 0x3E, 0x6B, 0xA5, 0xE3,
      0x22, 0xA1, 0x12, 0xD1, 0x8C, 0xF9, 0xF3, 0x56, 0xB5, 0x76, 0x97, 0x23,
      0x03, 0x50, 0xD2, 0x59, 0x1E, 0xE0, 0x2A, 0xD2, 0x65, 0x45, 0x15, 0x84,
      0x13, 0x23, 0x22, 0x25, 0x80, 0x32, 0x10, 0x40, 0x12, 0x06, 0x42, 0x60,
      0x83, 0x76, 0x18, 0x85, 0x50, 0x17, 0x66, 0x34, 0x73, 0x15, 0x10, 0x68,
      0x33, 0x33, 0x26, 0x40, 0x78, 0x28, 0x76, 0x15, 0x32, 0x51, 0x27, 0x24,
      0x30, 0x53, 0x38, 0x53, 0x35, 0x76, 0x47, 0x50, 0x65, 0x14, 0x11, 0x35,
      0x28, 0x86, 0x41, 0x24, 0x47, 0x44, 0x18, 0x54, 0x65, 0x64, 0x26, 0x58,
      0x21, 0x77, 0x85, 0x68, 0x11, 0x38, 0x50, 0x02, 0x13, 0x68, 0x41, 0x86,
      0x37, 0x85, 0x47, 0x26, 0x14, 0x48, 0x56, 0x44, 0x20, 0x72, 0x84, 0x86,
      0x37, 0x46, 0x27, 0x51, 0x16, 0x02, 0x12, 0x44, 0x36, 0x24, 0x72, 0x07,
      0x64, 0x32, 0x85, 0x42, 0x17, 0x18, 0x40, 0x10, 0x42, 0x17, 0x75, 0x18,
      0x41, 0x64, 0x58, 0x51, 0x48, 0x08, 0x67, 0x10, 0x34, 0x85, 0x07, 0x31,
      0x53, 0x28, 0x02, 0x75, 0x33, 0x63, 0x25, 0x54, 0x45, 0x72, 0x72, 0x07,
      0x26, 0x37, 0x13, 0x53, 0x03, 0x01, 0x86, 0x07, 0x52, 0x84, 0x46, 0x15,
      0x58, 0x53, 0x61, 0x12, 0x35, 0x02, 0x72, 0x21, 0x85, 0x45, 0x01, 0x56,
      0x24, 0x00, 0x84, 0x53, 0x48, 0x26, 0x87, 0x40, 0x11, 0x86, 0x36, 0x42,
      0x31, 0x30, 0x11, 0x26, 0x00, 0x00, 0x83, 0x04, 0x71, 0x15, 0x28, 0x54,
      0x73, 0x10, 0x53, 0x04, 0x48, 0x63, 0x57, 0x43, 0x45, 0x67, 0x14, 0x11,
      0x47, 0x36, 0x04, 0x42, 0x38, 0x01, 0x36, 0x01, 0x57, 0x58, 0x56, 0x42,
      0x58, 0x14, 0x67, 0x56, 0x33, 0x20, 0x16, 0x46, 0x05, 0x40, 0x71, 0x68,
      0x66, 0x05, 0x73, 0x62, 0x11, 0x15, 0x10, 0x76, 0x63, 0x74, 0x13, 0x35,
      0x77, 0x14, 0x20, 0x05, 0x55, 0x14, 0x82, 0x06, 0x32, 0x82, 0x75, 0x44,
      0x84, 0x86, 0x82, 0x08, 0x15, 0x27, 0x30, 0x68, 0x58, 0x18, 0x65, 0x81,
      0x38, 0x88, 0x74, 0x24, 0x10, 0x73, 0x35, 0x78, 0x47, 0x81, 0x45, 0x57,
      0x57, 0x84, 0x51, 0x04, 0x87, 0x41, 0x34, 0x88, 0x66, 0x60, 0x62, 0x63,
      0x48, 0x76, 0x80, 0x01, 0x64, 0x63, 0x70, 0x16, 0x83, 0x64, 0x08, 0x30,
      0x32, 0x11, 0x67, 0x66, 0x60, 0x76, 0x36, 0x37, 0x80, 0x83, 0x56, 0x74,
      0x70, 0x17, 0x07, 0x51, 0x60, 0x43, 0x12, 0x37, 0x86, 0x37, 0x06, 0x55,
      0x60, 0x44, 0x58, 0x23, 0x55, 0x11, 0x47, 0x60, 0x78, 0x62, 0x72, 0x84,
      0x45, 0x64, 0x61, 0x55, 0x14, 0x50, 0x28, 0x82, 0x77, 0x14, 0x88, 0x68,
      0x51, 0x56, 0x51, 0x84, 0x16, 0x55, 0x02, 0x55, 0x35, 0x11, 0x16, 0x13,
      0x13, 0x80, 0x72, 0x21, 0x68, 0x04, 0x02, 0x07, 0x81, 0x75, 0x41, 0x32,
      0x10, 0x26, 0x10, 0x53, 0x32, 0x37, 0x25, 0x13, 0x58, 0x85, 0x60, 0x48,
      0x66, 0x66, 0x76, 0x36, 0x05, 0x15, 0x34, 0x00, 0x68, 0x60, 0x36, 0x78,
      0x17, 0x73, 0x53, 0x23, 0x03, 0x58, 0x45, 0x85, 0x58, 0x53, 0x82, 0x55,
      0x61, 0x45, 0x87, 0x32, 0x31, 0x70, 0x26, 0x51, 0x82, 0x16, 0x01, 0x61,
      0x54, 0x75, 0x61, 0x34, 0x35, 0x45, 0x32, 0x05, 0x25, 0x36, 0x46, 0x50,
      0x30, 0x75, 0x12, 0x43, 0x75, 0x01, 0x63, 0x18, 0x31, 0x46, 0x24, 0x23,
      0x27, 0x77, 0x66, 0x38, 0x01, 0x87, 0x84, 0x26, 0x42, 0x48, 0x37, 0x80,
      0x77, 0x47, 0x44, 0x45, 0x30, 0x33, 0x57, 0x43, 0x84, 0x02, 0x53, 0x72,
      0x45, 0x33, 0x07, 0x68, 0x11, 0x41, 0x51, 0x26, 0x52, 0x23, 0x20, 0x80,
      0x13, 0x07, 0x36, 0x12, 0x22, 0x36, 0x42, 0x61, 0x37, 0x63, 0x00, 0x38,
      0x48, 0x61, 0x83, 0x31, 0x34, 0x62, 0x33, 0x05, 0x05, 0x00, 0x75, 0x16,
      0x64, 0x18, 0x50, 0x62, 0x05, 0x17, 0x58, 0x68, 0x74, 0x55, 0x34, 0x51,
      0x43, 0x30, 0x68, 0x72, 0x81, 0x11, 0x88, 0x60, 0x01, 0x63, 0x50, 0x37,
      0x11, 0x86, 0x66, 0x33, 0x28, 0x18, 0x83, 0x10, 0x83, 0x07, 0x45, 0x27,
      0x36, 0x50, 0x72, 0x01, 0x05, 0x18, 0x27, 0x37, 0x22, 0x37, 0x25, 0x38,
      0x56, 0x35, 0x30, 0x52, 0x11, 0x02, 0x41, 0x78, 0x61, 0x38, 0x26, 0x58,
      0x37, 0x74, 0x83, 0x10, 0x43, 0x54, 0x81, 0x13, 0x61, 0x72, 0x60, 0x26,
      0x48, 0x85, 0x05, 0x78, 0x38, 0x04, 0x07, 0x60, 0x30, 0x21, 0x10, 0x70,
      0x18, 0x21, 0x74, 0x06, 0x81, 0x05, 0x68, 0x80, 0x32, 0x63, 0x75, 0x66,
      0x43, 0x82, 0x26, 0x40, 0x12, 0x81, 0x11, 0x60, 0x47, 0x85, 0x03, 0x28,
      0x83, 0x44, 0x26, 0x67, 0x41, 0x72, 0x22, 0x67, 0x08, 0x30, 0x04, 0x32,
      0x21, 0x70, 0x54, 0x50, 0x55, 0x37, 0x65, 0x28, 0x86, 0x20, 0x25, 0x65,
      0x42, 0x74, 0x50, 0x64, 0x14, 0x12, 0x77, 0x38, 0x25, 0x11, 0x84, 0x76,
      0x61, 0x85, 0x61, 0x47, 0x37, 0x48, 0x80, 0x47, 0x08, 0x58, 0x80, 0x18,
      0x24, 0x12, 0x04, 0x54, 0x36, 0x44, 0x21, 0x30, 0x81, 0x14, 0x56, 0x75,
      0x28, 0x43, 0x20, 0x21, 0x50, 0x13, 0x87, 0x55, 0x72, 0x76, 0x08, 0x85,
      0x64, 0x82, 0x62, 0x28, 0x38, 0x42, 0x44, 0x02, 0x84, 0x45, 0x87, 0x23,
      0x37, 0x74, 0x17, 0x33, 0x41, 0x67, 0x31, 0x10, 0x07, 0x04, 0x83, 0x01,
      0x40, 0x68, 0x73, 0x02, 0x77, 0x18, 0x71, 0x16, 0x36, 0x73, 0x74, 0x84,
      0x05, 0x80, 0x22, 0x27, 0x78, 0x65, 0x58, 0x65, 0x75, 0x27, 0x56, 0x18,
      0x61, 0x71, 0x86, 0x80, 0x68, 0x26, 0x76, 0x23, 0x57, 0x03, 0x40, 0x72,
      0x80, 0x87, 0x44, 0x02, 0x62, 0x06, 0x44, 0x13, 0x32, 0x17, 0x84, 0x40,
      0x75, 0x30, 0x43, 0x26, 0x75, 0x40, 0x41, 0x53, 0x51, 0x02, 0x22, 0x80,
      0x11, 0x27, 0x82, 0x85, 0x78, 0x28, 0x63, 0x28, 0x62, 0x02, 0x45, 0x51,
      0x44, 0x84, 0x03, 0x80, 0x18, 0x55, 0x48, 0x16, 0x20, 0x82, 0x05, 0x50,
      0x44, 0x56, 0x24, 0x22, 0x20, 0x26, 0x06, 0x50, 0x73, 0x06, 0x45, 0x14,
      0x67, 0x23, 0x06, 0x21, 0x31, 0x03, 0x67, 0x24, 0x55, 0x50, 0x02, 0x75,
      0x23, 0x42, 0x57, 0x20, 0x57, 0x67, 0x25, 0x66, 0x44, 0x07, 0x84, 0x21,
      0x53, 0x65, 0x57, 0x40, 0x84, 0x14, 0x50, 0x12, 0x46, 0x68, 0x56, 0x04,
      0x45, 0x03, 0x71, 0x26, 0x82, 0x57, 0x45, 0x44, 0x18, 0x22, 0x67, 0x47,
      0x83, 0x48, 0x53, 0x31, 0x03, 0x22, 0x21, 0x53, 0x48, 0x05, 0x26, 0x41,
      0x54, 0x58, 0x04, 0x33, 0x51, 0x33, 0x24, 0x32, 0x38, 0x85, 0x44, 0x76,
      0x28, 0x82, 0x58, 0x17, 0x14, 0x03, 0x40, 0x65, 0x84, 0x25, 0x06, 0x05,
      0x21, 0x21, 0x85, 0x74, 0x00, 0x51, 0x60, 0x21, 0x22, 0x76, 0x58, 0x74,
      0x65, 0x82, 0x61, 0x23, 0x53, 0x40, 0x53, 0x07, 0x05, 0x72, 0x67, 0x33,
      0x83, 0x44, 0x88, 0x86, 0x77, 0x34, 0x40, 0x61, 0x74, 0x27, 0x14, 0x32,
      0x81, 0x42, 0x33, 0x32, 0x52, 0x66, 0x33, 0x47, 0x63, 0x21, 0x68, 0x32,
      0x48, 0x01, 0x20, 0x57, 0x42, 0x56, 0x48, 0x51, 0x86, 0x74, 0x11, 0x26,
      0x84, 0x02, 0x23, 0x36, 0x26, 0x28, 0x45, 0x78, 0x36, 0x50, 0x81, 0x13,
      0x52, 0x03, 0x38, 0x83, 0x17, 0x58, 0x53, 0x81, 0x76, 0x86, 0x78, 0x34,
      0x14, 0x67, 0x53, 0x68, 0x21, 0x88, 0x71, 0x33, 0x77, 0x66, 0x15, 0x06,
      0x74, 0x77, 0x50, 0x77, 0x72, 0x08, 0x87, 0x88, 0x47, 0x52, 0x23, 0x16,
      0x01, 0x38, 0x80, 0x47, 0x15, 0x72, 0x85, 0x35, 0x78, 0x60, 0x23, 0x26,
      0x06, 0x46, 0x82, 0x60, 0x21, 0x88, 0x57, 0x12, 0x30, 0x62, 0x81, 0x64,
      0x78, 0x12, 0x06, 0x23, 0x72, 0x81, 0x12, 0x86, 0x36, 0x36, 0x11, 0x04,
      0x67, 0x27, 0x80, 0x80, 0x57, 0x72, 0x04, 0x42, 0x77, 0x61, 0x32, 0x78,
      0x10, 0x67, 0x71, 0x63, 0x52, 0x48, 0x83, 0x61, 0x06, 0x27, 0x33, 0x25,
      0x18, 0x66, 0x05, 0x46, 0x85, 0x85, 0x78, 0x40, 0x45, 0x74, 0x20, 0x54,
      0x20, 0x23, 0x54, 0x51, 0x55, 0x76, 0x53, 0x77, 0x85, 0x70, 0x84, 0x58,
      0x22, 0x44, 0x50, 0x84, 0x65, 0x13, 0x50, 0x35, 0x17, 0x24, 0x58, 0x26,
      0x18, 0x16, 0x55, 0x43, 0x34, 0x25, 0x43, 0x17, 0x16, 0x61, 0x71, 0x31,
      0x38, 0x24, 0x37, 0x22, 0x88, 0x33, 0x04, 0x86, 0x34, 0x88, 0x30, 0x23,
      0x21, 0x35, 0x36, 0x87, 0x17, 0x23, 0x26, 0x33, 0x20, 0x44, 0x80, 0x24,
      0x53, 0x62, 0x72, 0x43, 0x25, 0x70, 0x22, 0x61, 0x53, 0x61, 0x28, 0x25,
      0x20, 0x07, 0x64, 0x13, 0x20, 0x83, 0x14, 0x67, 0x77, 0x68, 0x02, 0x11,
      0x48, 0x87, 0x48, 0x84, 0x65, 0x41, 0x08, 0x26, 0x05, 0x36, 0x02, 0x54,
      0x30, 0x10, 0x76, 0x26, 0x01, 0x42, 0x74, 0x24, 0x20, 0x86, 0x71, 0x53,
      0x23, 0x71, 0x63, 0x62, 0x10, 0x14, 0x72, 0x24, 0x85, 0x01, 0x33, 0x52,
      0x00, 0x33, 0x28, 0x18, 0x70, 0x04, 0x55, 0x74, 0x58, 0x77, 0x35, 0x04,
      0x05, 0x33, 0x70, 0x47, 0x83, 0x16, 0x54, 0x12, 0x68, 0x00, 0x78, 0x73,
      0x11, 0x57, 0x47, 0x70, 0x51, 0x15, 0x26, 0x14, 0x32, 0x45, 0x17, 0x58,
      0x10, 0x88, 0x43, 0x17, 0x85, 0x41, 0x72, 0x75, 0x16, 0x26, 0x32, 0x40,
      0x14, 0x73, 0x32, 0x10, 0x83, 0x81, 0x56, 0x78, 0x85, 0x65, 0x45, 0x74,
      0x67, 0x76, 0x37, 0x48, 0x74, 0x30, 0x48, 0x00, 0x62, 0x60, 0x07, 0x43,
      0x38, 0x63, 0x04, 0x83, 0x05, 0x24, 0x21, 0x68, 0x84, 0x71, 0x28, 0x48,
      0x51, 0x57, 0x78, 0x54, 0x60, 0x12, 0x72, 0x41, 0x37, 0x36, 0x80, 0x12,
      0x30, 0x31, 0x32, 0x05, 0x10, 0x75, 0x35, 0x57, 0x47, 0x64, 0x55, 0x61,
      0x84, 0x82, 0x25, 0x76, 0x71, 0x27, 0x10, 0x40, 0x40, 0x30, 0x50, 0x18,
      0x20, 0x53, 0x63, 0x54, 0x46, 0x61, 0x18, 0x42, 0x37, 0x17, 0x82, 0x00,
      0x33, 0x85, 0x20, 0x64, 0x25, 0x68, 0x63, 0x24, 0x50, 0x62, 0x78, 0x74,
      0x38, 0x43, 0x60, 0x87, 0x60, 0x16, 0x76, 0x47, 0x05, 0x28, 0x12, 0x75,
      0x64, 0x34, 0x85, 0x01, 0x50, 0x30, 0x40, 0x62, 0x12, 0x38, 0x35, 0x64,
      0x52, 0x44, 0x00, 0x05, 0x55, 0x33, 0x48, 0x54, 0x84, 0x03, 0x04, 0x84,
      0x60, 0x70, 0x52, 0x26, 0x72, 0x40, 0x60, 0x04, 0x82, 0x55, 0x81, 0x02,
      0x14, 0x71, 0x43, 0x53, 0x63, 0x61, 0x11, 0x18, 0x44, 0x17, 0x85, 0x68,
      0x30, 0x41, 0x81, 0x40, 0x20, 0x15, 0x45, 0x70, 0x74, 0x43, 0x30, 0x53,
      0x37, 0x42, 0x11, 0x38, 0x03, 0x62, 0x72, 0x62, 0x11, 0x24, 0x18, 0x47,
      0x4C, 0x22, 0x1B, 0x25, 0xDF, 0xE6, 0x5E, 0x91, 0x3E, 0x32, 0x54, 0x5D,
      0xD2, 0xC3, 0x33, 0x5E, 0x6C, 0x6C, 0xFF, 0x3F, 0x73, 0xD2, 0xA4, 0xD2,
      0xEE, 0x3C, 0xCC, 0xF8, 0x15, 0x0F, 0x4A, 0x4A, 0xA3, 0x4F, 0xE4, 0xDE,
      0xEE, 0x18, 0x8A, 0xDF, 0x12, 0x8F, 0xB7, 0xA2, 0xB5, 0x49, 0xA9, 0x6D,
      0x67, 0xE4, 0xAB, 0xE2, 0x80, 0x08, 0x9C, 0xF1, 0x6D, 0xE0, 0x78, 0x48,
      0x3C, 0x9D, 0x88, 0xCC, 0xB2, 0x92, 0x17, 0x0D, 0x58, 0x5A, 0x43, 0x82,
      0xBF, 0x1C, 0x69, 0xFE, 0x4E, 0x93, 0x86, 0xCA, 0x1D, 0x5B, 0xBA, 0x67,
      0x3B, 0x92, 0xAB, 0x78, 0xAE, 0xC2, 0x93, 0x54, 0x52, 0x21, 0xDE, 0xFF,
      0x40, 0x19, 0x08, 0x1B, 0xDF, 0xB9, 0x37, 0x44, 0x41, 0xC9, 0x32, 0x8E,
      0x45, 0xA3, 0x06, 0xB3, 0x19, 0x3B, 0xC8, 0xF2, 0x07, 0xF9, 0x14, 0xA5,
      0xE4, 0xEE, 0x76, 0xC4, 0xFD, 0x52, 0x13, 0xCD, 0x17, 0x95, 0x29, 0x4A,
      0x0C, 0xF5, 0x70, 0x15, 0x5D, 0x12, 0x41, 0x48, 0x65, 0x7B, 0x5E, 0x20,
      0x9A, 0x64, 0xBC, 0xEB, 0x01, 0x7F, 0x58, 0xC3, 0xF8, 0x51, 0xCB, 0x79,
      0x16, 0xBF, 0x22, 0xA0, 0xF9, 0x4D, 0x24, 0x8D, 0xB7, 0xF6, 0xE3, 0x75,
      0x85, 0x45, 0xAA, 0x63, 0xF8, 0x97, 0x57, 0xBF, 0xB1, 0xC0, 0x3F, 0x43,
      0xBB, 0xA7, 0x00, 0xEE, 0xBA, 0xA8, 0x4E, 0xDD, 0xE9, 0x61, 0xCC, 0x20,
      0x51, 0x54, 0x41, 0x1C, 0xA1, 0x04, 0x81, 0xA8, 0x8F, 0xE1, 0x01, 0xDB,
      0x0B, 0xA3, 0x75, 0xCF, 0xBC, 0xC6, 0xE7, 0x77, 0xAC, 0xD7, 0x2F, 0x00,
      0x20, 0x2B, 0xFB, 0xA2, 0x2F, 0xE5, 0xC0, 0x96, 0x4C, 0xA2, 0x87, 0xD0,
      0x0C, 0xB9, 0x3A, 0x01, 0xC2, 0x35, 0xA3, 0xA6, 0xE7, 0x31, 0x68, 0x70,
      0x39, 0xA8, 0x6F, 0xF1, 0xC5, 0xC2, 0x11, 0x86, 0x8D, 0xD4, 0x1B, 0x32,
      0x61, 0x4A, 0x48, 0xE1, 0x09, 0x15, 0x60, 0x79, 0xF7, 0x66, 0x8D, 0xF0,
      0x88, 0x79, 0xAF, 0xC7, 0xD0, 0xE2, 0x31, 0x97, 0xAA, 0xF7, 0xE6, 0x30,
      0x16, 0x9E, 0x53, 0xD7, 0x37, 0x97, 0x42, 0x7F, 0x0E, 0x8C, 0x47, 0x27,
      0xED, 0xAC, 0xE5, 0xE7, 0xB6, 0x93, 0x0D, 0x5E, 0x1A, 0x85, 0x91, 0xB6,
      0x00, 0x76, 0x80, 0xB1, 0x97, 0x64, 0x2A, 0x71, 0xDC, 0x64, 0x5D, 0x1A,
      0x24, 0x0D, 0xA6, 0x93, 0x0F, 0xDD, 0xC2, 0xAE, 0x6A, 0x5A, 0xF0, 0x9C,
      0xDA, 0x16, 0x20, 0x1E, 0x0B, 0x41, 0x48, 0x4B, 0xFB, 0x1C, 0xD9, 0x27,
      0x7C, 0xA6, 0x76, 0xD4, 0x6A, 0x48, 0xD6, 0x66, 0x37, 0xFC, 0xC3, 0x2D,
      0x2E, 0xF5, 0xDE, 0x2E, 0xD4, 0x97, 0x42, 0x38, 0x34, 0x7C, 0x81, 0xBF,
      0x18, 0x3F, 0x5B, 0xBD, 0xE7, 0x90, 0x1D, 0x56, 0xF8, 0xA8, 0xF1, 0x94,
      0xE8, 0x36, 0x43, 0x26, 0x97, 0xBD, 0xDD, 0x22, 0xB0, 0xC8, 0x2B, 0x37,
      0x47, 0x68, 0x93, 0xF5, 0x2F, 0xD2, 0xA3, 0xA4, 0xE3, 0xD7, 0xD7, 0x01,
      0xFE, 0x37, 0x01, 0xB5, 0x94, 0x57, 0x36, 0xD9, 0x7F, 0x5C, 0xD0, 0x29,
      0x30, 0x2D, 0x9C, 0x14, 0xF0, 0xFF, 0x98, 0x1A, 0xDF, 0x7C, 0xC0, 0xB1,
      0xA5, 0x2E, 0x04, 0x90, 0xD5, 0xE8, 0xB3, 0xEB, 0x8D, 0x5E, 0x95, 0xC8,
      0xF2, 0x59, 0x91, 0x43, 0x84, 0x9A, 0xDA, 0x2D, 0x1A, 0x9F, 0x2A, 0xE8,
      0x8C, 0x69, 0x56, 0x35, 0x69, 0x5C, 0x1F, 0x86, 0x0D, 0x5E, 0x59, 0x19,
      0x11, 0xE7, 0x9E, 0xE6, 0x6D, 0x74, 0x2C, 0x14, 0x41, 0x97, 0x4F, 0xCB,
      0xC2, 0x9C, 0x42, 0xD9, 0xBB, 0xFA, 0xC0, 0xBC, 0x71, 0xED, 0x75, 0x1E,
      0x56, 0xEF, 0xAF, 0x0C, 0x87, 0xD3, 0x15, 0x20, 0xCC, 0x17, 0xAA, 0x91,
      0x0E, 0xC9, 0x0C, 0xFF, 0x31, 0xD9, 0x15, 0x27, 0xDC, 0xD9, 0x40, 0xDC,
      0x02, 0x80, 0xEF, 0xCC, 0x3C, 0x2B, 0xD2, 0xB9, 0xBC, 0x8F, 0x0B, 0x47,
      0xE6, 0xBA, 0x57, 0x5E, 0xD4, 0xC5, 0xA4, 0x73, 0xB2, 0x6C, 0x54, 0x7E,
      0x82, 0xED, 0x73, 0x7F, 0x46, 0x76, 0xCF, 0x6E, 0x2E, 0xE6, 0xF9, 0x45,
      0xBE, 0xCE, 0x53, 0xDF, 0x76, 0x4E, 0x6F, 0xFF, 0xAB, 0x31, 0x39, 0x4D,
      0x4E, 0x0D, 0x56, 0x21, 0x9D, 0xE2, 0xB6, 0x04, 0xA0, 0x4D, 0x4D, 0xFF,
      0x79, 0xC1, 0x11, 0x5D, 0x79, 0x76, 0xD5, 0x71, 0xD6, 0x70, 0x03, 0x61,
      0x9C, 0x80, 0x1F, 0x05, 0x28, 0x8E, 0x2F, 0xCB, 0xDE, 0x51, 0x9A, 0x5E,
      0xD1, 0x0C, 0x17, 0x42, 0xC7, 0x80, 0xB2, 0xEA, 0xC2, 0x80, 0x15, 0xCE,
      0x8E, 0xC3, 0xCC, 0xC5, 0xD1, 0xAF, 0x22, 0xA8, 0xAC, 0x54, 0x20, 0x87,
      0x1E, 0xC6, 0xE8, 0x0E, 0x85, 0x27, 0xB3, 0xFF, 0xC5, 0xD2, 0xAB, 0x8C,
      0xE7, 0x20, 0x15, 0x13, 0x47, 0x6D, 0x0A, 0xC4, 0x4B, 0x35, 0x15, 0xB3,
      0x7A, 0xB1, 0xE6, 0xD9, 0x23, 0x79, 0xB1, 0x00, 0xD7, 0xE3, 0x61, 0x56,
      0xDF, 0xB2, 0x58, 0x6C, 0xCE, 0x1E, 0x2F, 0xDD, 0xA6, 0xFF, 0x51, 0x64,
      0x4A, 0x0B, 0x1B, 0x71, 0xB4, 0xB9, 0x76, 0x57, 0x8D, 0x05, 0xD4, 0xBB,
      0x8F, 0x2B, 0xC0, 0xE9, 0x16, 0x0A, 0x0F, 0x71, 0x4F, 0xE2, 0x70, 0xDF,
      0x89, 0x94, 0x74, 0x77, 0xD1, 0xCF, 0x24, 0x43, 0x6A, 0x16, 0x65, 0x88,
      0xD4, 0x65, 0xCE, 0xBA, 0x4E, 0x7A, 0xC8, 0x53, 0x12, 0xB4, 0x81, 0x93,
      0x84, 0xC9, 0xD0, 0x32, 0xAF, 0x59, 0xA6, 0x8E, 0x08, 0xAD, 0x10, 0x6A,
      0x3A, 0xA0, 0x0C, 0xCE, 0xA3, 0xF1, 0xC0, 0xB1, 0xF1, 0x32, 0x50, 0x42,
      0x51, 0x60, 0xD5, 0xB4, 0x46, 0xEA, 0x5D, 0x17, 0x98, 0xA2, 0xB9, 0xA2,
      0x81, 0x34, 0x45, 0xBC, 0x70, 0xEC, 0x44, 0x5E, 0x13, 0xE3, 0x71, 0xFA,
      0xB6, 0x3B, 0x6A, 0x71, 0x15, 0x54, 0xC4, 0xB4, 0xBE, 0xC8, 0xBE, 0x67,
      0x20, 0x12, 0xE9, 0xC0, 0x56, 0xA0, 0x89, 0x8A, 0x86, 0x58, 0x38, 0x3E,
      0x09, 0x3B, 0x2F, 0x57, 0x15, 0xD0, 0xC0, 0x26, 0xF8, 0xF5, 0xD7, 0x10,
      0x3B, 0x9B, 0x8C, 0x8C, 0xDF, 0x4A, 0x52, 0xEE, 0x26, 0xE5, 0x37, 0x17,
      0x59, 0x57, 0xEC, 0xE2, 0x87, 0x73, 0x3D, 0x17, 0xDF, 0x40, 0x3A, 0x57,
      0x1E, 0x8B, 0xE5, 0x39, 0x79, 0x7E, 0x54, 0x6B, 0xBB, 0xB2, 0x87, 0x19,
      0xDD, 0xB1, 0x48, 0xAC, 0xB2, 0x9E, 0xB0, 0x44, 0xBF, 0x5F, 0x6D, 0x4D,
      0x24, 0xF0, 0x85, 0x24, 0xEB, 0x54, 0x09, 0xF8, 0xDB, 0xBA, 0x7B, 0x64,
      0x72, 0xD6, 0x8E, 0x94, 0x74, 0x66, 0x06, 0x17, 0xCF, 0x5D, 0x6C, 0xDC,
      0x1E, 0xDF, 0x94, 0x17, 0xE6, 0x3C, 0xB3, 0x8B, 0x8B, 0x00, 0xE4, 0xB0,
      0xFA, 0xB4, 0x13, 0x47, 0x7F, 0x23, 0x4A, 0x48, 0xB4, 0x3D, 0x10, 0x7F,
      0xD9, 0x93, 0x1B, 0x7F, 0x7B, 0x50, 0x5E, 0xD5, 0x88, 0x2A, 0x50, 0xF0,
      0x9A, 0x59, 0xB6, 0x58, 0x65, 0xC0, 0x6B, 0xEC, 0x97, 0x91, 0x5F, 0xF3,
      0xB0, 0x4D, 0xCF, 0x95, 0xCB, 0xA4, 0x10, 0xE9, 0x30, 0xE9, 0xD5, 0x28,
      0x4B, 0x36, 0x6C, 0x60, 0x6F, 0xBC, 0x9B, 0xCD, 0xD0, 0x77, 0x64, 0xD6,
      0xAE, 0x69, 0x95, 0xB5, 0x1B, 0x18, 0x40, 0x7D, 0xDD, 0x91, 0x0B, 0x77,
      0x4D, 0xC4, 0x5A, 0x7F, 0x69, 0x2D, 0xE6, 0x6E, 0xB8, 0x7F, 0x54, 0xBB,
      0xD8, 0x35, 0xF9, 0x5E, 0x49, 0x12, 0xA4, 0x34, 0x9F, 0x8B, 0x89, 0x8A,
      0xA5, 0xE0, 0x80, 0x4A, 0xAB, 0x46, 0x07, 0xFF, 0xFE, 0x3C, 0xF4, 0x5F,
      0x40, 0xFA, 0x1E, 0x4E, 0xFF, 0xC0, 0xA4, 0x25, 0x20, 0xC7, 0x85, 0x94,
      0x42, 0xCE, 0x7F, 0xC6, 0x85, 0xF4, 0x5A, 0x9C, 0xCB, 0xC1, 0x5A, 0x5A,
      0xB5, 0xB9, 0x92, 0x7B, 0xDA, 0x21, 0xF8, 0x59, 0x08, 0xD2, 0xDE, 0x4B,
      0x9A, 0x8A, 0x82, 0xB7, 0xC0, 0xFE, 0xDF, 0x5B, 0x52, 0xF1, 0x24, 0xD5,
      0x77, 0x0A, 0x08, 0x0B, 0x07, 0xFA, 0xDE, 0xB5, 0x52, 0x46, 0x47, 0x08,
      0xDC, 0xD9, 0x3D, 0x89, 0xD9, 0xBB, 0x35, 0x25, 0xA5, 0x25, 0xB9, 0xF6,
      0x3D, 0xEF, 0x3A, 0xFE, 0xC9, 0x7D, 0xD3, 0xF7, 0x37, 0xF6, 0x12, 0xCF,
      0x54, 0x45, 0xD6, 0xBD, 0xB0, 0x89, 0xB2, 0xFF, 0xD1, 0x27, 0x6E, 0x11,
      0x25, 0xA3, 0x8E, 0xA1, 0xB1, 0x37, 0x3E, 0x28, 0xAF, 0x98, 0x78, 0x45,
      0xED, 0xB8, 0xCC, 0xAC, 0x32, 0x04, 0xC9, 0x88, 0xDD, 0x64, 0x36, 0x01,
      0x81, 0xD3, 0x4B, 0x5F, 0x2B, 0xD2, 0x11, 0x04, 0xFB, 0xC8, 0x6F, 0x19,
      0xDB, 0x1D, 0x73, 0x2E, 0xD4, 0x29, 0x8C, 0x72, 0xE0, 0x8C, 0x1D, 0x75,
      0xD8, 0x80, 0x09, 0x78, 0xE4, 0xCB, 0xAA, 0x3F, 0xA8, 0xE8, 0xB2, 0x99,
      0x19, 0x22, 0x88, 0x2C, 0xC8, 0x84, 0x1D, 0x33, 0xF8, 0xBA, 0x9F, 0xE1,
      0xC2, 0x49, 0x31, 0xA5, 0x41, 0x21, 0xB9, 0xB3, 0xC2, 0x45, 0x23, 0x79,
      0x7B, 0x58, 0x5F, 0x78, 0xBB, 0x38, 0xFF, 0xAB, 0x9F, 0x29, 0x1E, 0xD8,
      0x6E, 0x79, 0xB3, 0x63, 0x63, 0x72, 0x96, 0x32, 0xBB, 0x62, 0x64, 0xA9,
      0x5E, 0x24, 0xF1, 0x96, 0x65, 0x02, 0x97, 0x64, 0x99, 0x49, 0x03, 0xE3,
      0x33, 0x0E, 0xD3, 0x96, 0x7A, 0x6B, 0x1A, 0xDE, 0xB5, 0x03, 0x88, 0x85,
      0xB6, 0x18, 0x81, 0x86, 0xCF, 0xC9, 0x79, 0xC2, 0x07, 0x7B, 0xDB, 0x2A,
      0xE5, 0x31, 0xC7, 0x13, 0xCD, 0x58, 0xC2, 0x58, 0xD5, 0xD7, 0x67, 0xFB,
      0x05, 0x87, 0xC0, 0xF2, 0xC3, 0x77, 0x30, 0xEF, 0x32, 0x6B, 0x98, 0x25,
      0xA1, 0x85, 0x78, 0xB9, 0x52, 0x2C, 0x71, 0x08, 0x80, 0x35, 0x53, 0xE5,
      0x4A, 0x97, 0x24, 0x30, 0x0C, 0x69, 0xB5, 0x34, 0x2F, 0xFD, 0x88, 0x7D,
      0x1D, 0x7A, 0xE7, 0x44, 0xE7, 0x21, 0xE1, 0x61, 0x00, 0x1D, 0x2F, 0xAD,
      0x52, 0xAF, 0xA9, 0x3A, 0x63, 0xF9, 0xE1, 0xDD, 0x06, 0xA1, 0xAC, 0xD7,
      0x53, 0x2D, 0x46, 0x5E, 0xD1, 0x5B, 0x07, 0x57, 0x5C, 0x5B, 0x09, 0x91,
      0x7C, 0xB5, 0x5B, 0x7B, 0x17, 0x4A, 0x03, 0x7E, 0x0A, 0x03, 0xB5, 0x14,
      0x84, 0x66, 0x53, 0x1D, 0x5D, 0x31, 0x5D, 0xB5, 0x15, 0x6A, 0x58, 0x87,
      0x5C, 0x0C, 0xA2, 0x59, 0x65, 0x54, 0xD6, 0xA9, 0x7B, 0x3A, 0xE6, 0x8D,
      0xB6, 0xBC, 0x27, 0xF3, 0x2E, 0x2E, 0xF5, 0x4F, 0x8E, 0x18, 0x1C, 0xAE,
      0xF3, 0xB9, 0xF3, 0x29, 0x64, 0x3C, 0x82, 0x91, 0x22, 0x59, 0x75, 0x00,
      0x90, 0x32, 0xF9, 0xDC, 0x9C, 0x40, 0x9B, 0x97, 0x59, 0xBE, 0xE8, 0x75,
      0x9E, 0xAB, 0x07, 0x99, 0x5D, 0xAE, 0x0D, 0x87, 0xA1, 0x0B, 0x2B, 0x11,
      0x57, 0x36, 0x46, 0x66, 0xC9, 0xC9, 0x74, 0x09, 0x8C, 0x0A, 0x52, 0xB4,
      0x6C, 0x2A, 0xDE, 0x69, 0xAC, 0xE0, 0x8D, 0x1E, 0x92, 0x72, 0x6B, 0x33,
      0xCA, 0x12, 0x7C, 0xC2, 0x6E, 0x77, 0x5D, 0x1B, 0x41, 0x23, 0xFD, 0x48,
      0x32, 0xB3, 0x95, 0x4F, 0x25, 0x55, 0x01, 0x01, 0x57, 0xAB, 0x56, 0x64,
      0x41, 0x06, 0x67, 0xE8, 0x28, 0xA7, 0x65, 0xCC, 0x56, 0xFE, 0x2B, 0x7D,
      0x2A, 0xAC, 0xB2, 0x7B, 0x58, 0xB6, 0x44, 0xDB, 0x35, 0xAD, 0xD1, 0x63,
      0x8C, 0x9E, 0x52, 0x2B, 0xDF, 0x74, 0x81, 0xF1, 0xE3, 0x26, 0x5C, 0x8D,
      0x91, 0x08, 0x11, 0x39, 0xB3, 0xB8, 0x2A, 0x4E, 0x05, 0xDC, 0x44, 0x9E,
      0xA9, 0x4E, 0xCD, 0xC6, 0x39, 0x07, 0x4B, 0x5E, 0x61, 0x7C, 0x6B, 0x21,
      0xAD, 0x33, 0xF7, 0x95, 0x27, 0xD3, 0xD1, 0x0B, 0xEB, 0x57, 0x5A, 0xCD,
      0x38, 0x16, 0xA9, 0x93, 0x9B, 0xFE, 0x76, 0x6B, 0x8F, 0x4F, 0x07, 0xA1,
      0xAC, 0x9C, 0xA2, 0x19, 0x27, 0xD6, 0xC6, 0x09, 0xD7, 0x1A, 0x82, 0x19,
      0xC0, 0xD3, 0x9A, 0x22, 0x9C, 0x8E, 0xB2, 0x31, 0xFB, 0xDD, 0x53, 0xE7,
      0x56, 0xD8, 0x92, 0x50, 0xBC, 0x03, 0x81, 0x10, 0x41, 0xA8, 0xBD, 0xD2,
      0xA6, 0x0C, 0x39, 0x2D, 0xAE, 0x89, 0x9F, 0x6F, 0x5C, 0x57, 0x93, 0x56,
      0x50, 0x64, 0x9B, 0x6F, 0x21, 0xA2, 0x10, 0x46, 0x97, 0x8D, 0x4F, 0x2A,
      0x5D, 0xD1, 0x05, 0xFF, 0x16, 0x78, 0x0D, 0x54, 0xDF, 0x3F, 0x7B, 0x0D,
      0x1A, 0x6D, 0x68, 0xFD, 0x09, 0x15, 0xD1, 0x31, 0x81, 0x29, 0x2A, 0xD4,
      0x63, 0x35, 0x4B, 0x90, 0xDA, 0x2C, 0x77, 0x09, 0xFE, 0x3F, 0x24, 0x8C,
      0x4A, 0x46, 0x9E, 0xAE, 0x9C, 0xA5, 0x7A, 0x8E, 0xAB, 0xAD, 0xFD, 0x0E,
      0xB2, 0xBA, 0xAD, 0x54, 0x82, 0xB1, 0x45, 0x53, 0xDB, 0xA0, 0x53, 0x77,
      0xCB, 0x62, 0xAE, 0x9A, 0xD7, 0x6E, 0xD0, 0x7A, 0xDB, 0x93, 0x1E, 0x43,
      0x84, 0xFD, 0xD1, 0xE5, 0x5E, 0x4A, 0xE8, 0x29, 0xB9, 0x75, 0x09, 0xB3,
      0xC8, 0x4B, 0x67, 0x3D, 0xFF, 0x72, 0x68, 0x63, 0x65, 0x32, 0x28, 0x59,
      0xC5, 0x21, 0xF3, 0x66, 0xF3, 0x8A, 0xFC, 0x2E, 0x8F, 0x49, 0x26, 0xF4,
      0xD1, 0xD0, 0x04, 0x31, 0x12, 0x01, 0x27, 0xF1, 0xFD, 0xE2, 0x54, 0x67,
      0xDE, 0x45, 0x42, 0xD8, 0xEB, 0x43, 0x69, 0x03, 0x37, 0xA7, 0xB4, 0x05,
      0xB8, 0x8C, 0x3D, 0xB5, 0x2A, 0x46, 0x0D, 0x1B, 0x0B, 0xB8, 0xF8, 0x7E,
      0x1E, 0xB7, 0xDB, 0xC6, 0xD9, 0xBB, 0x50, 0xDC, 0x56, 0x1D, 0x76, 0xF2,
      0x3A, 0x5C, 0x4A, 0x74, 0x34, 0x12, 0xAD, 0xEF, 0x08, 0x39, 0x37, 0x64,
      0xFE, 0xAF, 0xFB, 0x3E, 0xE2, 0x72, 0xDA, 0xC0, 0x42, 0x06, 0x49, 0x79,
      0x0A, 0xFB, 0x07, 0x0F, 0x5F, 0xB8, 0x3B, 0x62, 0x83, 0xA6, 0xC6, 0x4C,
      0xBF, 0x69, 0xC4, 0x9D, 0x9A, 0x0C, 0xDC, 0xD6, 0x05, 0xE6, 0x9F, 0xBE,
      0xAA, 0x2A, 0x26, 0x72, 0xDC, 0x7D, 0xD1, 0xB4, 0x54, 0xC8, 0xE3, 0x0E,
      0x00, 0x22, 0xEF, 0x2A, 0xAB, 0x91, 0xC7, 0x86, 0x6A, 0xF4, 0x3B, 0x25,
      0x45, 0xB7, 0xD8, 0x66, 0x10, 0x0E, 0x56, 0xE6, 0x7D, 0xF1, 0x2A, 0x63,
      0x18, 0xCA, 0x07, 0xDE, 0xC5, 0x72, 0x6A, 0x14, 0xCC, 0xDC, 0x97, 0x28,
      0xB1, 0x69, 0xA1, 0xD0, 0x39, 0x50, 0x9C, 0x72, 0xC0, 0x29, 0xAE, 0x04,
      0xC2, 0x8C, 0x1D, 0x6F, 0x04, 0x02, 0xE7, 0x81, 0x9A, 0xCB, 0x65, 0x1E,
      0x03, 0xD0, 0x35, 0x87, 0x94, 0xB0, 0xD3, 0x25, 0x76, 0x33, 0x1B, 0x9C,
      0x00, 0x04, 0x24, 0x38, 0x19, 0xB0, 0xD4, 0xA2, 0x00, 0x6E, 0x81, 0xA3,
      0x6F, 0xE2, 0xA6, 0x7F, 0x70, 0x47, 0x3C, 0x90, 0xA1, 0x2D, 0xD6, 0xBA,
      0x66, 0x9D, 0x5A, 0x1F, 0x4D, 0xA3, 0xD5, 0x77, 0xFE, 0x9A, 0x8E, 0xBA,
      0x72, 0x63, 0xD5, 0x83, 0x7E, 0xB6, 0x00, 0x97, 0x64, 0x61, 0xE5, 0xDD,
      0x64, 0xB9, 0x3F, 0x48, 0x45, 0x11, 0xEE, 0x28, 0x81, 0x94, 0x9A, 0x59,
      0x74, 0xFC, 0xE1, 0x57, 0x5B, 0xF5, 0xD5, 0x28, 0xCA, 0xB6, 0x54, 0xCE,
      0xDC, 0x7C, 0x5E, 0x6E, 0x47, 0x52, 0x02, 0x4E, 0xC7, 0x11, 0x7E, 0xDF,
      0x5B, 0xC1, 0xED, 0x2E, 0xE7, 0xF6, 0xB3, 0xC2, 0x87, 0x3B, 0xD4, 0xEB,
      0x23, 0xEF, 0xE0, 0x02, 0x91, 0xDC, 0x81, 0xEF, 0x1B, 0x17, 0xCB, 0xB6,
      0x07, 0x23, 0xEA, 0xB8, 0x43, 0x5A, 0xC2, 0x79, 0x4E, 0x9A, 0x13, 0xD1,
      0x45, 0x43, 0x60, 0xFB, 0x18, 0xCB, 0x0B, 0x9B, 0x07, 0x33, 0x34, 0x3F,
      0xA0, 0x08, 0xDC, 0x45, 0x04, 0x6C, 0x82, 0x67, 0x28, 0xC3, 0x83, 0xFA,
      0x67, 0x20, 0xA9, 0xDC, 0x2C, 0xE1, 0x7A, 0xEF, 0xA0, 0xAB, 0x54, 0x8A,
      0xA0, 0x13, 0x8A, 0x18, 0x47, 0x51, 0xE7, 0xCC, 0x9F, 0x17, 0xFA, 0xF7,
      0x59, 0xEB, 0x18, 0x79, 0x3F, 0x04, 0x0D, 0xA4, 0x65, 0xAB, 0x67, 0x06,
      0xAF, 0xD8, 0xE0, 0x2C, 0x7A, 0xF5, 0x18, 0x07, 0xE2, 0xF5, 0x0D, 0x90,
      0xE8, 0x72, 0xA4, 0xDD, 0x03, 0x98, 0x63, 0x82, 0x9C, 0x1D, 0xFF, 0xEB,
      0x16, 0x84, 0xE6, 0x94, 0x51, 0x27, 0x88, 0x1B, 0xFF, 0x2E, 0x42, 0x10,
      0x7D, 0x64, 0x3B, 0xEB, 0xB2, 0x54, 0xA2, 0xCE, 0x48, 0x9E, 0x32, 0xD7,
      0xD5, 0x2A, 0x99, 0x76, 0x62, 0x51, 0x90, 0x77, 0x70, 0xA2, 0xA1, 0xF4,
      0xCE, 0xF9, 0xC1, 0x17, 0x13, 0x40, 0xBD, 0xED, 0xF1, 0xD5, 0x39, 0xB4,
      0xC3, 0x04, 0xF8, 0xB2, 0x24, 0x08, 0xAC, 0x24, 0xDD, 0x1E, 0x30, 0x7C,
      0xB7, 0x47, 0xEF, 0x11, 0xF6, 0xBB, 0x35, 0xEE, 0xD0, 0x4B, 0x9C, 0x98,
      0xE4, 0x77, 0xB9, 0x18, 0xA9, 0xF4, 0xC5, 0xEE, 0xA6, 0x50, 0x2F, 0x3D,
      0x06, 0x87, 0xEC, 0x5B, 0xBE, 0x77, 0xF2, 0xA5, 0xC7, 0xDD, 0x17, 0xBE,
      0xCF, 0x5E, 0x1F, 0x2D, 0xCF, 0xD1, 0x8B, 0x1C, 0x49, 0xC8, 0xC9, 0xA1,
      0xA7, 0xCC, 0xDE, 0x14, 0x43, 0xF3, 0xF7, 0x1D, 0x77, 0xFE, 0x07, 0x73,
      0x2C, 0x9C, 0xBA, 0x7A, 0x43, 0x29, 0x01, 0x85, 0xCA, 0xD7, 0xCD, 0x67,
      0x41, 0xB9, 0xDA, 0x4F, 0xE0, 0xC9, 0x84, 0xAC, 0x6E, 0xE6, 0xC1, 0x75,
      0xC7, 0xD6, 0xDE, 0x9E, 0xC6, 0x5E, 0xD3, 0xC0, 0xA6, 0xEE, 0xFE, 0x71,
      0x3D, 0xBC, 0xC2, 0xD6, 0x47, 0x8D, 0x65, 0x7A, 0x1E, 0x41, 0x7B, 0x8F,
      0x94, 0x81, 0x46, 0x3A, 0x79, 0x98, 0x79, 0x53, 0x1E, 0x99, 0x38, 0x3F,
      0xD3, 0xF7, 0xD4, 0xED, 0x22, 0xD0, 0x80, 0x43, 0x10, 0x5F, 0xFA, 0x67,
      0x61, 0xC0, 0xAC, 0x5B, 0x54, 0xCF, 0xEA, 0xCD, 0x88, 0x56, 0x64, 0xE5,
      0x50, 0x3E, 0xAE, 0x79, 0xA9, 0x69, 0x70, 0x13, 0xB6, 0x27, 0x2D, 0x69,
      0xE4, 0x8A, 0x60, 0xDC, 0xCF, 0xEB, 0x3A, 0xD2, 0x8E, 0x40, 0x23, 0x34,
      0xE7, 0x72, 0x14, 0x72, 0x93, 0x0C, 0xBD, 0xC9, 0x45, 0xA9, 0x8E, 0xD5,
      0xD7, 0x8C, 0xF8, 0xFC, 0xC6, 0x8F, 0xB2, 0x67, 0xC0, 0xD4, 0x97, 0x92,
      0x5A, 0x9C, 0x33, 0x82, 0x9B, 0x3C, 0x6B, 0x87, 0xA0, 0x17, 0x3D, 0xB9,
      0xBC, 0x55, 0xA7, 0x4A, 0x9F, 0x2C, 0x86, 0x85, 0x9A, 0xCA, 0x54, 0xB1,
      0x58, 0x98, 0x4B, 0x8F, 0xF7, 0x1F, 0x63, 0x61, 0x69, 0x1A, 0xB6, 0xEA,
      0x01, 0xB5, 0xA9, 0x22, 0x74, 0x90, 0x58, 0xD0, 0xE1, 0x88, 0x74, 0x7E,
      0x08, 0x83, 0x87, 0xFD, 0x42, 0xB6, 0x87, 0x9E, 0x87, 0x9F, 0x0F, 0x66,
      0xB2, 0x76, 0xFC, 0xFD, 0x55, 0x80, 0x11, 0x80, 0x82, 0x12, 0x53, 0x7A,
      0xE4, 0xA3, 0x5B, 0x30, 0x8A, 0xF2, 0x76, 0x87, 0xE6, 0xFB, 0x1A, 0xBE,
      0x6C, 0x90, 0x17, 0xCA, 0xA6, 0xE4, 0x77, 0x88, 0x92, 0x54, 0x68, 0x92,
      0x6F, 0xEF, 0x07, 0x84, 0x1D, 0x1B, 0xF4, 0x99, 0x6F, 0xB9, 0xD3, 0xBB,
      0x0E, 0x81, 0x97, 0x73, 0xE8, 0x49, 0x29, 0xE4, 0x3C, 0x28, 0xE4, 0xB5,
      0x06, 0x96, 0xC0, 0x8E, 0x1C, 0x7E, 0xF5, 0x1E, 0x46, 0xC9, 0x4C, 0x99,
      0x40, 0x1F, 0xE6, 0xB2, 0x81, 0x25, 0xDA, 0x89, 0xA6, 0x6D, 0x65, 0xD3,
      0x8B, 0x7A, 0x9D, 0xE8, 0xB0, 0x2B, 0x59, 0x57, 0x2A, 0x0F, 0x72, 0x46,
      0x67, 0x35, 0x8C, 0x69, 0x6B, 0x9E, 0xBE, 0x5D, 0xFC, 0x51, 0x1B, 0x1A};

  bssl::ScopedCBB cbb;
  EXPECT_TRUE(CBB_init(cbb.get(), DILITHIUM_PRIVATE_KEY_BYTES));
  EXPECT_TRUE(DILITHIUM_marshal_private_key(cbb.get(), priv.get()));

  EXPECT_EQ(CBB_len(cbb.get()), size_t{DILITHIUM_PRIVATE_KEY_BYTES});
  EXPECT_EQ(Bytes(CBB_data(cbb.get()), CBB_len(cbb.get())),
            Bytes(expected_private_key));
}

TEST(DilithiumTest, SignatureHardCodedNIST) {
  // Published on
  // https://csrc.nist.gov/Projects/post-quantum-cryptography/post-quantum-cryptography-standardization/example-files
  static const uint8_t encoded_private_key[DILITHIUM_PRIVATE_KEY_BYTES] = {
      0xD2, 0x6E, 0xE5, 0x9A, 0x89, 0xF6, 0x7C, 0x98, 0xB2, 0x0F, 0x89, 0x0B,
      0x03, 0x42, 0x2C, 0x80, 0x27, 0xB7, 0x76, 0xFC, 0x30, 0x5B, 0xEF, 0x42,
      0x2C, 0xDE, 0xD4, 0x03, 0xAA, 0x70, 0x5D, 0xA5, 0xDC, 0x38, 0x0A, 0xD9,
      0x09, 0xE3, 0xC1, 0x3A, 0x76, 0xF7, 0xF5, 0x9B, 0xDD, 0x27, 0xA8, 0x43,
      0xF5, 0xF9, 0xA1, 0xB8, 0xE5, 0xB7, 0xB5, 0x24, 0xEC, 0x44, 0x5A, 0x91,
      0x6B, 0x7A, 0x44, 0xA1, 0x51, 0x67, 0x71, 0x6B, 0xC6, 0x09, 0xB3, 0xEC,
      0x0C, 0x65, 0x3B, 0x21, 0x0C, 0x65, 0x69, 0xBA, 0x2F, 0x82, 0x33, 0xE7,
      0xF6, 0xE4, 0x04, 0xB9, 0x2E, 0xB2, 0x1F, 0x39, 0x72, 0x9F, 0xEB, 0xC0,
      0x6A, 0x64, 0x40, 0x87, 0xC7, 0x5F, 0xA7, 0xDA, 0x3D, 0xDC, 0x5B, 0x85,
      0xA8, 0x9F, 0x55, 0xC9, 0xF8, 0x7A, 0x15, 0x01, 0x4E, 0x27, 0xAB, 0x32,
      0x5E, 0xBE, 0xE0, 0xFE, 0x47, 0x2B, 0x2F, 0x56, 0x20, 0x58, 0x44, 0x86,
      0x03, 0x82, 0x53, 0x40, 0x18, 0x82, 0x88, 0x66, 0x44, 0x14, 0x76, 0x57,
      0x56, 0x85, 0x64, 0x12, 0x03, 0x64, 0x80, 0x87, 0x66, 0x55, 0x85, 0x18,
      0x36, 0x25, 0x51, 0x65, 0x55, 0x42, 0x00, 0x58, 0x87, 0x17, 0x51, 0x60,
      0x27, 0x17, 0x71, 0x12, 0x46, 0x11, 0x56, 0x35, 0x71, 0x18, 0x01, 0x27,
      0x36, 0x01, 0x12, 0x78, 0x06, 0x46, 0x80, 0x05, 0x02, 0x11, 0x61, 0x76,
      0x81, 0x10, 0x48, 0x15, 0x65, 0x56, 0x43, 0x13, 0x80, 0x40, 0x86, 0x15,
      0x62, 0x24, 0x77, 0x52, 0x53, 0x36, 0x50, 0x85, 0x72, 0x66, 0x76, 0x70,
      0x25, 0x42, 0x12, 0x87, 0x77, 0x01, 0x52, 0x17, 0x65, 0x63, 0x84, 0x22,
      0x14, 0x66, 0x16, 0x70, 0x18, 0x54, 0x17, 0x12, 0x56, 0x07, 0x75, 0x65,
      0x33, 0x05, 0x35, 0x67, 0x88, 0x35, 0x20, 0x67, 0x34, 0x65, 0x43, 0x72,
      0x73, 0x38, 0x42, 0x64, 0x71, 0x42, 0x52, 0x81, 0x41, 0x50, 0x38, 0x53,
      0x32, 0x02, 0x65, 0x61, 0x28, 0x85, 0x16, 0x71, 0x30, 0x01, 0x17, 0x75,
      0x55, 0x13, 0x45, 0x23, 0x07, 0x35, 0x13, 0x15, 0x37, 0x08, 0x54, 0x30,
      0x18, 0x72, 0x37, 0x21, 0x57, 0x58, 0x73, 0x03, 0x15, 0x60, 0x81, 0x11,
      0x03, 0x21, 0x11, 0x22, 0x58, 0x73, 0x64, 0x04, 0x43, 0x82, 0x64, 0x70,
      0x52, 0x44, 0x30, 0x87, 0x45, 0x33, 0x05, 0x57, 0x35, 0x84, 0x55, 0x16,
      0x04, 0x73, 0x00, 0x45, 0x31, 0x65, 0x04, 0x42, 0x42, 0x77, 0x24, 0x30,
      0x37, 0x53, 0x27, 0x36, 0x43, 0x54, 0x55, 0x46, 0x15, 0x12, 0x41, 0x08,
      0x15, 0x47, 0x41, 0x24, 0x33, 0x07, 0x82, 0x43, 0x31, 0x13, 0x43, 0x70,
      0x37, 0x23, 0x51, 0x85, 0x81, 0x38, 0x85, 0x73, 0x23, 0x48, 0x83, 0x00,
      0x25, 0x24, 0x13, 0x82, 0x03, 0x11, 0x88, 0x18, 0x10, 0x67, 0x12, 0x72,
      0x62, 0x75, 0x00, 0x51, 0x64, 0x48, 0x58, 0x46, 0x70, 0x34, 0x35, 0x42,
      0x31, 0x80, 0x31, 0x60, 0x86, 0x65, 0x26, 0x47, 0x17, 0x43, 0x38, 0x63,
      0x73, 0x85, 0x10, 0x27, 0x77, 0x33, 0x11, 0x12, 0x45, 0x23, 0x28, 0x02,
      0x70, 0x48, 0x82, 0x61, 0x36, 0x83, 0x36, 0x04, 0x02, 0x76, 0x23, 0x52,
      0x05, 0x67, 0x67, 0x85, 0x61, 0x00, 0x74, 0x43, 0x30, 0x01, 0x18, 0x02,
      0x64, 0x17, 0x76, 0x21, 0x72, 0x47, 0x04, 0x14, 0x10, 0x05, 0x67, 0x68,
      0x42, 0x36, 0x75, 0x52, 0x76, 0x75, 0x02, 0x65, 0x03, 0x88, 0x07, 0x02,
      0x36, 0x54, 0x31, 0x35, 0x30, 0x50, 0x02, 0x01, 0x74, 0x63, 0x32, 0x58,
      0x67, 0x62, 0x13, 0x76, 0x86, 0x70, 0x04, 0x31, 0x22, 0x42, 0x03, 0x77,
      0x78, 0x68, 0x42, 0x62, 0x34, 0x60, 0x75, 0x01, 0x37, 0x58, 0x07, 0x60,
      0x61, 0x18, 0x64, 0x36, 0x43, 0x34, 0x31, 0x82, 0x85, 0x44, 0x87, 0x11,
      0x80, 0x67, 0x37, 0x57, 0x17, 0x21, 0x67, 0x63, 0x68, 0x62, 0x34, 0x63,
      0x14, 0x45, 0x72, 0x74, 0x70, 0x52, 0x12, 0x25, 0x61, 0x58, 0x65, 0x01,
      0x26, 0x24, 0x30, 0x41, 0x63, 0x88, 0x51, 0x46, 0x65, 0x06, 0x45, 0x16,
      0x83, 0x24, 0x15, 0x67, 0x57, 0x12, 0x50, 0x27, 0x63, 0x17, 0x61, 0x18,
      0x13, 0x78, 0x33, 0x53, 0x67, 0x28, 0x64, 0x26, 0x40, 0x52, 0x58, 0x17,
      0x84, 0x63, 0x76, 0x70, 0x46, 0x74, 0x43, 0x46, 0x40, 0x53, 0x85, 0x55,
      0x86, 0x07, 0x41, 0x63, 0x65, 0x31, 0x23, 0x11, 0x60, 0x76, 0x55, 0x78,
      0x30, 0x32, 0x61, 0x41, 0x71, 0x11, 0x80, 0x36, 0x42, 0x63, 0x33, 0x18,
      0x26, 0x24, 0x23, 0x67, 0x47, 0x86, 0x48, 0x07, 0x65, 0x63, 0x53, 0x82,
      0x32, 0x46, 0x57, 0x70, 0x01, 0x57, 0x65, 0x47, 0x60, 0x15, 0x36, 0x13,
      0x72, 0x45, 0x64, 0x81, 0x67, 0x88, 0x15, 0x48, 0x35, 0x57, 0x03, 0x44,
      0x16, 0x16, 0x63, 0x53, 0x65, 0x66, 0x15, 0x35, 0x12, 0x78, 0x64, 0x12,
      0x16, 0x15, 0x46, 0x02, 0x26, 0x47, 0x53, 0x08, 0x62, 0x88, 0x56, 0x17,
      0x33, 0x21, 0x32, 0x16, 0x10, 0x87, 0x45, 0x62, 0x55, 0x35, 0x06, 0x78,
      0x47, 0x23, 0x73, 0x53, 0x85, 0x38, 0x64, 0x63, 0x05, 0x63, 0x70, 0x23,
      0x50, 0x73, 0x23, 0x68, 0x22, 0x68, 0x07, 0x83, 0x60, 0x63, 0x65, 0x57,
      0x51, 0x67, 0x37, 0x24, 0x23, 0x48, 0x13, 0x70, 0x27, 0x37, 0x87, 0x32,
      0x17, 0x06, 0x27, 0x47, 0x56, 0x64, 0x05, 0x15, 0x83, 0x27, 0x55, 0x31,
      0x75, 0x24, 0x41, 0x76, 0x43, 0x52, 0x71, 0x31, 0x78, 0x27, 0x76, 0x73,
      0x30, 0x23, 0x62, 0x65, 0x63, 0x22, 0x62, 0x84, 0x34, 0x44, 0x14, 0x66,
      0x48, 0x14, 0x18, 0x07, 0x51, 0x87, 0x30, 0x17, 0x47, 0x36, 0x06, 0x63,
      0x05, 0x68, 0x44, 0x76, 0x04, 0x50, 0x58, 0x22, 0x47, 0x82, 0x45, 0x47,
      0x25, 0x45, 0x71, 0x77, 0x04, 0x48, 0x76, 0x14, 0x71, 0x48, 0x72, 0x06,
      0x02, 0x04, 0x43, 0x17, 0x05, 0x60, 0x44, 0x77, 0x51, 0x26, 0x22, 0x75,
      0x52, 0x08, 0x75, 0x41, 0x11, 0x64, 0x41, 0x14, 0x53, 0x35, 0x55, 0x05,
      0x70, 0x70, 0x67, 0x40, 0x13, 0x75, 0x67, 0x56, 0x61, 0x30, 0x28, 0x60,
      0x28, 0x34, 0x61, 0x35, 0x07, 0x52, 0x87, 0x05, 0x64, 0x32, 0x18, 0x23,
      0x64, 0x56, 0x37, 0x83, 0x37, 0x31, 0x78, 0x81, 0x60, 0x12, 0x64, 0x76,
      0x27, 0x52, 0x72, 0x10, 0x64, 0x33, 0x46, 0x75, 0x02, 0x74, 0x66, 0x16,
      0x45, 0x47, 0x38, 0x10, 0x66, 0x46, 0x66, 0x54, 0x32, 0x45, 0x75, 0x82,
      0x55, 0x20, 0x88, 0x26, 0x82, 0x27, 0x75, 0x58, 0x86, 0x11, 0x26, 0x80,
      0x86, 0x52, 0x84, 0x20, 0x67, 0x24, 0x27, 0x06, 0x50, 0x26, 0x02, 0x82,
      0x21, 0x35, 0x41, 0x38, 0x06, 0x76, 0x11, 0x50, 0x41, 0x22, 0x41, 0x62,
      0x31, 0x42, 0x13, 0x20, 0x62, 0x85, 0x45, 0x61, 0x46, 0x20, 0x05, 0x28,
      0x24, 0x63, 0x52, 0x70, 0x54, 0x22, 0x20, 0x67, 0x08, 0x42, 0x32, 0x46,
      0x38, 0x34, 0x70, 0x30, 0x22, 0x56, 0x33, 0x03, 0x12, 0x06, 0x40, 0x70,
      0x06, 0x13, 0x02, 0x82, 0x48, 0x18, 0x61, 0x35, 0x86, 0x22, 0x67, 0x80,
      0x75, 0x14, 0x51, 0x70, 0x34, 0x80, 0x88, 0x71, 0x88, 0x70, 0x28, 0x65,
      0x13, 0x76, 0x60, 0x33, 0x57, 0x05, 0x34, 0x71, 0x35, 0x68, 0x48, 0x23,
      0x24, 0x60, 0x12, 0x37, 0x20, 0x23, 0x66, 0x23, 0x60, 0x48, 0x68, 0x15,
      0x70, 0x88, 0x11, 0x73, 0x31, 0x70, 0x80, 0x32, 0x61, 0x25, 0x44, 0x63,
      0x34, 0x66, 0x02, 0x74, 0x87, 0x88, 0x10, 0x11, 0x82, 0x25, 0x77, 0x84,
      0x02, 0x13, 0x58, 0x46, 0x03, 0x83, 0x86, 0x88, 0x16, 0x74, 0x82, 0x87,
      0x45, 0x77, 0x53, 0x40, 0x23, 0x60, 0x86, 0x84, 0x71, 0x75, 0x51, 0x17,
      0x80, 0x60, 0x87, 0x86, 0x05, 0x80, 0x70, 0x13, 0x60, 0x73, 0x71, 0x14,
      0x68, 0x61, 0x81, 0x01, 0x11, 0x45, 0x21, 0x18, 0x25, 0x28, 0x51, 0x72,
      0x75, 0x46, 0x66, 0x41, 0x16, 0x44, 0x27, 0x25, 0x71, 0x86, 0x66, 0x62,
      0x77, 0x45, 0x56, 0x57, 0x06, 0x12, 0x80, 0x63, 0x34, 0x63, 0x37, 0x83,
      0x71, 0x25, 0x85, 0x05, 0x68, 0x44, 0x48, 0x71, 0x52, 0x13, 0x47, 0x47,
      0x12, 0x36, 0x26, 0x63, 0x50, 0x16, 0x16, 0x74, 0x30, 0x60, 0x83, 0x25,
      0x14, 0x15, 0x44, 0x80, 0x31, 0x24, 0x24, 0x78, 0x88, 0x10, 0x77, 0x30,
      0x72, 0x31, 0x46, 0x81, 0x43, 0x32, 0x85, 0x43, 0x38, 0x07, 0x06, 0x26,
      0x34, 0x87, 0x80, 0x23, 0x48, 0x51, 0x28, 0x56, 0x65, 0x85, 0x22, 0x88,
      0x16, 0x06, 0x46, 0x00, 0x85, 0x14, 0x77, 0x11, 0x84, 0x76, 0x60, 0x26,
      0x57, 0x87, 0x21, 0x40, 0x85, 0x34, 0x48, 0x12, 0x64, 0x01, 0x16, 0x71,
      0x36, 0x86, 0x70, 0x15, 0x44, 0x01, 0x88, 0x82, 0x50, 0x86, 0x06, 0x02,
      0x53, 0x75, 0x43, 0x64, 0x00, 0x48, 0x57, 0x04, 0x78, 0x88, 0x68, 0x34,
      0x45, 0x67, 0x58, 0x37, 0x40, 0x38, 0x68, 0x11, 0x25, 0x64, 0x20, 0x86,
      0x78, 0x77, 0x16, 0x60, 0x77, 0x71, 0x24, 0x32, 0x13, 0x62, 0x88, 0x33,
      0x10, 0x62, 0x38, 0x11, 0x78, 0x45, 0x03, 0x76, 0x52, 0x23, 0x16, 0x15,
      0x35, 0x20, 0x15, 0x37, 0x26, 0x88, 0x06, 0x35, 0x20, 0x61, 0x25, 0x15,
      0x75, 0x17, 0x87, 0x48, 0x21, 0x07, 0x86, 0x32, 0x83, 0x15, 0x18, 0x11,
      0x00, 0x02, 0x57, 0x84, 0x34, 0x62, 0x86, 0x77, 0x27, 0x72, 0x56, 0x81,
      0x11, 0x81, 0x64, 0x11, 0x33, 0x08, 0x33, 0x03, 0x10, 0x84, 0x74, 0x04,
      0x18, 0x85, 0x82, 0x13, 0x15, 0x17, 0x72, 0x37, 0x03, 0x44, 0x15, 0x74,
      0x02, 0x62, 0x87, 0x77, 0x35, 0x53, 0x02, 0x52, 0x15, 0x56, 0x63, 0x68,
      0x36, 0x21, 0x05, 0x03, 0x63, 0x81, 0x23, 0x82, 0x06, 0x27, 0x03, 0x54,
      0x16, 0x36, 0x07, 0x20, 0x37, 0x88, 0x81, 0x30, 0x65, 0x14, 0x37, 0x72,
      0x74, 0x53, 0x16, 0x42, 0x55, 0x77, 0x75, 0x04, 0x34, 0x66, 0x25, 0x37,
      0x42, 0x21, 0x71, 0x76, 0x66, 0x35, 0x53, 0x41, 0x02, 0x14, 0x61, 0x64,
      0x23, 0x00, 0x01, 0x81, 0x61, 0x63, 0x55, 0x51, 0x23, 0x25, 0x53, 0x76,
      0x77, 0x82, 0x84, 0x34, 0x83, 0x04, 0x02, 0x36, 0x11, 0x18, 0x80, 0x70,
      0x12, 0x45, 0x70, 0x01, 0x54, 0x01, 0x82, 0x76, 0x32, 0x18, 0x53, 0x45,
      0x42, 0x61, 0x14, 0x48, 0x02, 0x27, 0x72, 0x61, 0x07, 0x20, 0x16, 0x65,
      0x63, 0x36, 0x14, 0x70, 0x62, 0x50, 0x08, 0x50, 0x10, 0x70, 0x86, 0x11,
      0x15, 0x51, 0x06, 0x70, 0x05, 0x22, 0x78, 0x66, 0x65, 0x48, 0x12, 0x55,
      0x15, 0x66, 0x68, 0x04, 0x14, 0x04, 0x26, 0x12, 0x30, 0x73, 0x70, 0x31,
      0x68, 0x34, 0x00, 0x12, 0x06, 0x56, 0x02, 0x31, 0x70, 0x66, 0x68, 0x64,
      0x16, 0x15, 0x31, 0x78, 0x57, 0x42, 0x53, 0x05, 0x31, 0x34, 0x02, 0x76,
      0x87, 0x58, 0x03, 0x10, 0x66, 0x34, 0x68, 0x12, 0x50, 0x22, 0x14, 0x18,
      0x78, 0x12, 0x01, 0x67, 0x74, 0x12, 0x10, 0x74, 0x44, 0x80, 0x86, 0x64,
      0x13, 0x71, 0x28, 0x05, 0x64, 0x24, 0x01, 0x51, 0x56, 0x62, 0x27, 0x54,
      0x34, 0x42, 0x57, 0x03, 0x06, 0x10, 0x10, 0x83, 0x51, 0x38, 0x62, 0x87,
      0x36, 0x85, 0x63, 0x14, 0x07, 0x57, 0x60, 0x10, 0x24, 0x52, 0x81, 0x71,
      0x28, 0x30, 0x88, 0x34, 0x56, 0x30, 0x14, 0x34, 0x25, 0x53, 0x51, 0x01,
      0x2F, 0xAD, 0xA6, 0x07, 0x97, 0x6F, 0x68, 0x37, 0xB0, 0xDB, 0xB6, 0x81,
      0x27, 0xC5, 0xD2, 0xCC, 0xA1, 0xE3, 0xBF, 0xE9, 0xCE, 0x04, 0x41, 0xDA,
      0xA2, 0xFC, 0x7F, 0x01, 0x45, 0xF9, 0x6B, 0xFA, 0x47, 0x30, 0x54, 0x00,
      0x9E, 0xFB, 0x11, 0x18, 0x4E, 0x47, 0x18, 0xB0, 0x45, 0x75, 0x8D, 0xAB,
      0x9E, 0x41, 0xCC, 0x03, 0xA8, 0xA5, 0x81, 0xF8, 0x6F, 0x05, 0xC9, 0xD7,
      0x6F, 0x3B, 0xBE, 0xFC, 0xEE, 0x4A, 0x67, 0xDC, 0xD2, 0xCE, 0x92, 0x0C,
      0xD1, 0x5B, 0x72, 0x41, 0xD7, 0xBF, 0x64, 0x1B, 0x46, 0x2A, 0x72, 0xE2,
      0x87, 0xC3, 0xDB, 0x90, 0xAF, 0xF4, 0x54, 0xED, 0x1C, 0x8B, 0x07, 0x94,
      0xAC, 0x10, 0x55, 0xBC, 0x6B, 0x4B, 0xE8, 0x16, 0x69, 0xFE, 0x71, 0x54,
      0xED, 0x8F, 0xAB, 0x43, 0xBB, 0x21, 0xE2, 0xD7, 0x87, 0xFE, 0xD3, 0x16,
      0xB8, 0x1C, 0xE0, 0x3D, 0xA5, 0x01, 0x9A, 0x09, 0x48, 0xBF, 0x4F, 0x19,
      0xCE, 0x09, 0xD3, 0x17, 0x73, 0x6C, 0xF9, 0x1F, 0xDD, 0xD4, 0xB4, 0x2B,
      0x29, 0xB9, 0xAA, 0x0C, 0x87, 0xD5, 0x8C, 0x81, 0xD2, 0xA7, 0x4F, 0x22,
      0xF4, 0x4A, 0xFE, 0xFD, 0xF6, 0xDD, 0x0B, 0xB2, 0x68, 0x9B, 0xC4, 0xE5,
      0x4C, 0x42, 0xDE, 0x56, 0xF5, 0xFC, 0x67, 0x96, 0x94, 0x16, 0x31, 0x93,
      0x15, 0x25, 0xD7, 0xB5, 0x1F, 0xED, 0x90, 0x89, 0x4F, 0x88, 0xB2, 0x0B,
      0xC8, 0x19, 0xBE, 0x6F, 0x19, 0xD0, 0x80, 0xEF, 0xC0, 0x3E, 0x23, 0xCB,
      0x03, 0x4D, 0x06, 0xEA, 0xA4, 0x1D, 0x55, 0x2C, 0xCB, 0x07, 0xB4, 0x2A,
      0xE7, 0x94, 0x86, 0x82, 0x5E, 0xBE, 0xAE, 0xE2, 0x2C, 0xE1, 0x93, 0xD5,
      0xB9, 0x17, 0x4E, 0x20, 0x49, 0x1E, 0x23, 0xF3, 0x76, 0xE3, 0xEA, 0x97,
      0x31, 0x98, 0x1F, 0xDE, 0xE6, 0xE0, 0xC8, 0x37, 0xA0, 0x7C, 0x9A, 0x65,
      0x75, 0xA0, 0x38, 0x52, 0x2C, 0x8F, 0x36, 0xFF, 0x0E, 0x48, 0x16, 0x25,
      0xE1, 0xD6, 0x93, 0xC6, 0x9E, 0x2B, 0x4B, 0x49, 0x10, 0xB3, 0x5C, 0xFC,
      0xD7, 0xAF, 0xF7, 0x63, 0x01, 0xF1, 0x5E, 0x58, 0x0C, 0x5F, 0x2E, 0x41,
      0x18, 0xCB, 0x97, 0x54, 0xD9, 0x0A, 0xDC, 0x41, 0x40, 0xE4, 0xAA, 0x8D,
      0xA0, 0x2F, 0xF7, 0x50, 0x25, 0x14, 0x65, 0x2F, 0x07, 0xD1, 0x4B, 0x0A,
      0xFD, 0xD6, 0xDE, 0x58, 0xEB, 0x89, 0x6C, 0xF5, 0x1E, 0x56, 0x2E, 0x0B,
      0x79, 0x72, 0xE8, 0x5D, 0x33, 0x4E, 0xE7, 0x7E, 0x09, 0xE7, 0x88, 0x2E,
      0xF0, 0x80, 0x27, 0x57, 0x86, 0xB9, 0x1E, 0x1B, 0xDB, 0x94, 0x9E, 0x94,
      0xBC, 0xFB, 0xE4, 0x53, 0x33, 0xD0, 0xCF, 0xD6, 0x30, 0x57, 0x46, 0xA2,
      0x4B, 0x4B, 0x2A, 0xF2, 0x31, 0x1F, 0x96, 0x41, 0x51, 0xD4, 0xF8, 0x9D,
      0x12, 0xD1, 0xD8, 0x8A, 0x61, 0x49, 0x35, 0x1C, 0xB3, 0x29, 0x49, 0x26,
      0xA3, 0xB0, 0x50, 0x51, 0xFA, 0x1F, 0x61, 0x3F, 0x48, 0x54, 0xA1, 0x72,
      0x7E, 0x3F, 0xC1, 0x95, 0xF4, 0xF5, 0xF9, 0x9D, 0x7A, 0x6F, 0x67, 0x66,
      0xA2, 0x50, 0x83, 0x9C, 0x09, 0x0B, 0x4D, 0x81, 0xE7, 0x1C, 0x24, 0x2A,
      0x58, 0xDE, 0xBE, 0x5D, 0xDC, 0xA5, 0x56, 0xB9, 0x7D, 0xE9, 0x79, 0x0D,
      0xB8, 0x9E, 0xD1, 0x5C, 0xED, 0xCF, 0x7E, 0xBC, 0x6D, 0x44, 0xDC, 0x61,
      0x88, 0x1B, 0xE0, 0x60, 0xF9, 0x05, 0xCB, 0xB7, 0x70, 0x25, 0x69, 0x4F,
      0xB3, 0x82, 0xA6, 0x6A, 0x28, 0x09, 0xA7, 0x59, 0xAE, 0x66, 0xDE, 0x10,
      0x2C, 0xF0, 0x59, 0xE9, 0x33, 0x9C, 0x26, 0x11, 0x2F, 0x76, 0xC5, 0x0B,
      0x39, 0xEF, 0x61, 0xAD, 0xF3, 0x93, 0xB9, 0x2B, 0x69, 0xBD, 0x88, 0xF6,
      0xAC, 0x5E, 0x7E, 0xED, 0x1F, 0xA1, 0xA8, 0xBA, 0x72, 0x01, 0x69, 0x6D,
      0xBE, 0xCC, 0x8F, 0xB9, 0xAF, 0xC1, 0x94, 0x6C, 0x74, 0x4D, 0x92, 0xF3,
      0xED, 0x30, 0xF8, 0x83, 0x4C, 0x39, 0x8F, 0x42, 0xC5, 0x37, 0x20, 0x51,
      0x0D, 0x16, 0x81, 0x7A, 0x80, 0x74, 0xA2, 0xF1, 0x2D, 0x27, 0xFF, 0x98,
      0xCF, 0xCF, 0xFD, 0xFF, 0x7D, 0x94, 0xFF, 0x79, 0x39, 0x93, 0x99, 0x1F,
      0xD4, 0x4C, 0x79, 0x9D, 0x8B, 0xAA, 0x5E, 0x0B, 0xCC, 0x78, 0x9D, 0x3F,
      0x3E, 0xF4, 0x67, 0x10, 0x4B, 0xDB, 0xF2, 0x09, 0x25, 0x09, 0xE5, 0x30,
      0xF9, 0x20, 0xDA, 0x1F, 0x66, 0x3A, 0x5E, 0x58, 0xAE, 0xCA, 0x50, 0xEC,
      0x6F, 0x1D, 0xC8, 0x29, 0x86, 0xD3, 0x7E, 0xE6, 0xC1, 0x95, 0xD6, 0x4E,
      0x2D, 0x49, 0x60, 0xBE, 0xFB, 0x09, 0xD3, 0xEE, 0xAA, 0xDB, 0xF9, 0x87,
      0x33, 0x40, 0xA5, 0xB8, 0x6D, 0x29, 0xDE, 0x38, 0x0D, 0xFC, 0x41, 0xA9,
      0x9D, 0x73, 0x50, 0xD9, 0xA5, 0x05, 0x57, 0x06, 0xE3, 0xC2, 0x03, 0x72,
      0x45, 0xDF, 0x35, 0xB9, 0xDF, 0x14, 0xB8, 0x10, 0x77, 0x2C, 0x3A, 0x40,
      0x88, 0x7C, 0xA0, 0x2C, 0xF5, 0xCD, 0x42, 0x99, 0x3E, 0x36, 0x08, 0x52,
      0x21, 0x9A, 0x9A, 0x38, 0x46, 0x47, 0xCE, 0x91, 0x2D, 0x95, 0xDF, 0xF0,
      0xE8, 0x07, 0x06, 0xA2, 0x69, 0x62, 0x96, 0xC7, 0x29, 0x02, 0x14, 0xCE,
      0xC7, 0x2E, 0xEB, 0x01, 0x6D, 0x34, 0xE0, 0x3F, 0xBF, 0x5E, 0x65, 0xF4,
      0x79, 0x5E, 0x71, 0x4B, 0x35, 0xDE, 0x18, 0x3F, 0xEE, 0x6C, 0xC6, 0xC3,
      0x5E, 0x34, 0x92, 0x89, 0xCC, 0x8D, 0xBB, 0x37, 0xAE, 0x9F, 0x96, 0x8C,
      0xF0, 0xBC, 0x36, 0x54, 0xD3, 0xB8, 0x21, 0x75, 0xB0, 0xC5, 0xAF, 0xFD,
      0x9B, 0x1D, 0x42, 0x1B, 0x7D, 0xCE, 0x80, 0x15, 0xAF, 0xD7, 0x75, 0x20,
      0xA9, 0x83, 0xE1, 0xC1, 0xE9, 0xEB, 0x07, 0x08, 0x11, 0x8F, 0xA4, 0xBB,
      0xEE, 0x07, 0x21, 0x57, 0xE7, 0xCD, 0x8C, 0xE9, 0x38, 0x10, 0x91, 0xD3,
      0xF1, 0xD4, 0xFC, 0x39, 0x80, 0xE5, 0xF8, 0xC9, 0x5D, 0x54, 0xF4, 0xE6,
      0xBB, 0x7A, 0xF3, 0x68, 0xAF, 0x6B, 0xC9, 0xB3, 0x32, 0xB8, 0x0D, 0x7F,
      0x8E, 0x76, 0xC2, 0xE2, 0x03, 0x04, 0x9C, 0x33, 0x80, 0xC5, 0xE0, 0xF5,
      0x29, 0x9D, 0xD6, 0xE5, 0x7C, 0xD5, 0x9A, 0xED, 0x1C, 0xCD, 0xC9, 0x98,
      0xF2, 0xFA, 0xE7, 0x2C, 0x29, 0xA0, 0xC1, 0xD4, 0x02, 0x5C, 0x34, 0xE5,
      0x06, 0xAF, 0x77, 0x76, 0x02, 0x28, 0xDC, 0x3B, 0x8B, 0x68, 0x0A, 0xC8,
      0xA2, 0xFA, 0xD7, 0x84, 0xEA, 0x6B, 0x86, 0xEF, 0x34, 0xF1, 0x84, 0x2E,
      0x66, 0xF4, 0x6F, 0xB0, 0xF6, 0x8F, 0x01, 0x01, 0x42, 0x5D, 0x16, 0x34,
      0x18, 0x25, 0xDD, 0x66, 0x05, 0xDF, 0x63, 0x10, 0xE7, 0xC9, 0x81, 0x1F,
      0x4C, 0xB8, 0x1E, 0xC3, 0xC7, 0xAE, 0x7C, 0xC6, 0xF5, 0x28, 0xC7, 0x82,
      0xC6, 0xD3, 0x3D, 0x68, 0x83, 0xE8, 0x95, 0xB3, 0xA0, 0x6E, 0x3D, 0x3E,
      0xA8, 0x4B, 0x03, 0xE3, 0x60, 0x33, 0x94, 0xB4, 0x4A, 0x1A, 0x3C, 0x28,
      0x98, 0x71, 0xA5, 0x92, 0x69, 0xF4, 0xB8, 0x0F, 0x4C, 0x7C, 0xF4, 0x04,
      0xA4, 0xB9, 0x9A, 0x52, 0x9A, 0x57, 0x4B, 0xE4, 0x6A, 0xA6, 0xB5, 0x2B,
      0x4E, 0x54, 0x4F, 0xB7, 0x02, 0x2C, 0x90, 0x90, 0x4B, 0xE1, 0xC7, 0x67,
      0xFB, 0x8D, 0xB7, 0x44, 0xB9, 0xC6, 0x9F, 0xBC, 0x50, 0x3C, 0x23, 0x4F,
      0x66, 0x18, 0x79, 0x4F, 0x71, 0xAD, 0x8C, 0x95, 0xE5, 0x08, 0x7F, 0x54,
      0xB4, 0xF8, 0x21, 0x54, 0x45, 0x4C, 0x0E, 0x73, 0x86, 0xF2, 0x1E, 0xEC,
      0xC0, 0xDD, 0xBA, 0xF4, 0x54, 0x38, 0x2A, 0x97, 0x3B, 0xE4, 0xEC, 0xEF,
      0xE8, 0x46, 0xE0, 0xC2, 0x1B, 0x60, 0xB5, 0x64, 0x48, 0xE6, 0xE5, 0xED,
      0xDB, 0xAE, 0x08, 0x41, 0x32, 0xA1, 0x39, 0x7F, 0x16, 0x31, 0x40, 0x47,
      0x14, 0x4A, 0xCA, 0xDD, 0x6C, 0xC3, 0x42, 0x44, 0x7D, 0xB6, 0xA6, 0x53,
      0x65, 0x74, 0x7A, 0x63, 0xC4, 0x04, 0xAE, 0x62, 0x3B, 0xC3, 0xEF, 0x17,
      0x03, 0xC1, 0x8B, 0x14, 0xFD, 0xA1, 0x78, 0xAB, 0x73, 0x92, 0xE9, 0xEB,
      0x6D, 0x2C, 0x90, 0x45, 0xEE, 0xFC, 0x8A, 0x73, 0x68, 0xDB, 0xCB, 0xA7,
      0xF0, 0x9C, 0xC8, 0x8D, 0x6E, 0xA5, 0x45, 0x05, 0xBF, 0xB8, 0x23, 0x73,
      0x81, 0x9F, 0xB9, 0x90, 0xF0, 0xF6, 0x4C, 0x6F, 0xBE, 0x42, 0x1E, 0x35,
      0xCE, 0xBD, 0x9B, 0x79, 0xCE, 0x77, 0xBD, 0xF3, 0xC8, 0x94, 0x6C, 0x58,
      0xC6, 0x07, 0x17, 0x57, 0x28, 0x4E, 0x5B, 0x53, 0xB5, 0x6D, 0xFE, 0xDB,
      0x2E, 0xEF, 0x92, 0x8D, 0xC9, 0xAE, 0x02, 0xBA, 0x21, 0xA2, 0xB5, 0xE2,
      0x84, 0xE2, 0x85, 0xC9, 0xA6, 0xCB, 0x05, 0x71, 0x0E, 0x26, 0x34, 0xCB,
      0x36, 0x27, 0xC5, 0x2E, 0x57, 0x7F, 0x7E, 0x5C, 0xA7, 0x47, 0x56, 0x07,
      0xA7, 0xD2, 0xCC, 0xF1, 0x45, 0x2F, 0x9B, 0xA7, 0x03, 0xD3, 0x22, 0xF6,
      0x9A, 0x47, 0xF1, 0x6B, 0xD9, 0x85, 0x1C, 0x75, 0x28, 0xD8, 0x9B, 0x59,
      0xC0, 0x3B, 0x42, 0x6A, 0xEB, 0xB3, 0xB3, 0x72, 0x32, 0x8C, 0xA9, 0x10,
      0x2F, 0x62, 0x4E, 0x90, 0x1D, 0xC8, 0xA3, 0x15, 0x52, 0x4B, 0x94, 0x17,
      0x06, 0xEC, 0xFE, 0x0A, 0xA8, 0xAE, 0xB0, 0xAF, 0x8E, 0xC3, 0x3D, 0x2E,
      0x1C, 0xA2, 0x03, 0xB4, 0x89, 0xAE, 0x23, 0xC0, 0x3D, 0x88, 0x0B, 0x83,
      0xD7, 0x38, 0xCC, 0x27, 0x7E, 0xB6, 0xFF, 0x1C, 0x7D, 0xD3, 0x18, 0x86,
      0xC6, 0x39, 0xC6, 0xC6, 0x58, 0x39, 0x6F, 0x60, 0x92, 0x0D, 0x8A, 0x0A,
      0x4E, 0x58, 0x27, 0x8D, 0x63, 0xE5, 0xB3, 0x93, 0x29, 0x5F, 0xE8, 0x88,
      0x7E, 0x8F, 0x9F, 0x6A, 0x64, 0xC1, 0x0C, 0xA2, 0x41, 0xF6, 0x99, 0xB9,
      0x9D, 0x72, 0xD1, 0x37, 0x22, 0xE4, 0x2D, 0x46, 0x15, 0xD9, 0xF3, 0xD0,
      0x54, 0x5E, 0x32, 0x0D, 0x38, 0xEF, 0x80, 0x90, 0x76, 0xF4, 0x21, 0x64,
      0x28, 0xC2, 0x9D, 0x13, 0x0F, 0x2F, 0x83, 0x05, 0x2B, 0x9D, 0x0B, 0xA4,
      0xA8, 0xA1, 0x87, 0x2A, 0x24, 0x2B, 0xDC, 0x59, 0x72, 0x22, 0x81, 0xDD,
      0xEC, 0xE5, 0x40, 0x7C, 0xBB, 0xDD, 0xBD, 0xB0, 0xB8, 0xC0, 0x4E, 0x93,
      0x85, 0xAD, 0x96, 0x60, 0xD2, 0x52, 0xDD, 0x28, 0xAA, 0x88, 0x11, 0x66,
      0x69, 0x42, 0x07, 0xBB, 0x04, 0xF9, 0xA2, 0x4F, 0xA9, 0x22, 0x16, 0xAC,
      0xAF, 0x26, 0x4C, 0x9F, 0x1C, 0xC2, 0x00, 0xA5, 0xE2, 0xCB, 0xD2, 0x69,
      0x4C, 0x91, 0x23, 0x8A, 0x89, 0x72, 0x49, 0x60, 0x97, 0xB3, 0xBD, 0x37,
      0x85, 0x34, 0x5E, 0xE7, 0x17, 0x1A, 0x9E, 0xB9, 0x66, 0x19, 0x72, 0x2C,
      0x64, 0x57, 0x1D, 0xE6, 0x32, 0x7C, 0xD0, 0x34, 0x5B, 0x78, 0x88, 0x71,
      0xD7, 0xD5, 0xD0, 0xD8, 0x4E, 0xF2, 0xA4, 0x7D, 0xDE, 0x66, 0x80, 0xB7,
      0x74, 0x3A, 0xD6, 0x1B, 0x16, 0x4E, 0x55, 0x8B, 0x67, 0x2E, 0x4C, 0xC4,
      0xB4, 0x6D, 0x6E, 0x6C, 0x23, 0x6C, 0x21, 0x33, 0x96, 0x5C, 0x8B, 0x3C,
      0x7D, 0x0F, 0x24, 0x0A, 0xB0, 0xED, 0xEC, 0x78, 0xE8, 0xEA, 0xB4, 0xC8,
      0x00, 0x40, 0xA0, 0x37, 0xF0, 0x08, 0x0F, 0xA3, 0xB1, 0x5A, 0xCC, 0x3E,
      0xEA, 0x34, 0x20, 0x80, 0x65, 0xEB, 0x6E, 0x6E, 0x21, 0xDA, 0x65, 0x60,
      0x14, 0x54, 0xE6, 0xDB, 0x03, 0x7C, 0x08, 0xB0, 0x51, 0xA1, 0x94, 0x2D,
      0xAB, 0x90, 0xC9, 0x2B, 0x0B, 0xDC, 0x8C, 0x57, 0x2C, 0x52, 0xF0, 0xAE,
      0xF8, 0x4B, 0x85, 0xF0, 0x44, 0x4A, 0x6C, 0x15, 0xE8, 0x48, 0x01, 0x44,
      0x8D, 0xE8, 0x14, 0x78, 0x04, 0xF7, 0x96, 0xA1, 0xA0, 0x53, 0x41, 0xE7,
      0x5D, 0xF0, 0x15, 0xFE, 0xEA, 0x7E, 0x19, 0x1C, 0x46, 0x1D, 0x3D, 0x3D,
      0x7B, 0x2D, 0x74, 0xA9, 0xC4, 0xC7, 0xE1, 0x8E, 0xAF, 0xE0, 0x96, 0x6C,
      0x26, 0x73, 0x1F, 0x0A, 0x96, 0x04, 0x61, 0xDE, 0x03, 0x87, 0xA9, 0x17,
      0xA6, 0xBB, 0xD6, 0x45, 0xC4, 0x66, 0x80, 0x30, 0x64, 0x2D, 0x70, 0xDE,
      0x30, 0xD9, 0x95, 0x45, 0x56, 0xCD, 0xAC, 0xDE, 0x00, 0xF8, 0x22, 0x19,
      0xE7, 0xA3, 0xEE, 0xCC, 0x83, 0x54, 0xFC, 0x5A, 0x43, 0x88, 0x19, 0x5C,
      0xE5, 0x68, 0x01, 0x6C, 0x7C, 0x63, 0x66, 0x15, 0x59, 0xEF, 0x65, 0xE2,
      0xFF, 0x09, 0xF1, 0xC9, 0x8A, 0x66, 0x68, 0xC2, 0x3A, 0xE4, 0x86, 0x3B,
      0x04, 0x28, 0x7A, 0x3B, 0xB6, 0x21, 0x0B, 0x3F, 0x56, 0x89, 0xA3, 0xE7,
      0x97, 0x85, 0x17, 0xB9, 0x5B, 0x0D, 0x06, 0xA8, 0x31, 0xF2, 0x8A, 0x3F,
      0xF5, 0x5B, 0x60, 0x53, 0xE3, 0x85, 0x35, 0xB2, 0xAA, 0x97, 0xFD, 0x66,
      0xB4, 0x29, 0x80, 0xCD, 0xAF, 0x9E, 0x5D, 0xB6, 0xF1, 0x21, 0x2B, 0xC2,
      0x3C, 0xC7, 0x40, 0xBB, 0x7D, 0xDF, 0x8C, 0xD4, 0x03, 0x91, 0xC0, 0x19,
      0x0F, 0x62, 0xA0, 0xA2, 0xA2, 0x87, 0xDC, 0xC2, 0xAB, 0x8F, 0xA6, 0x5C,
      0x71, 0x67, 0x56, 0x19, 0xC4, 0x51, 0xEC, 0x07, 0xF3, 0xD6, 0x7E, 0xE8,
      0x1A, 0x63, 0x70, 0xEA, 0x21, 0x44, 0xD8, 0x2C, 0xE6, 0x3C, 0xAE, 0xB4,
      0xFD, 0x76, 0xE6, 0xFE, 0x49, 0x79, 0xFB, 0xC1, 0x49, 0xBC, 0x54, 0xA9,
      0x23, 0x6D, 0x4C, 0xF2, 0x3C, 0xD5, 0xC0, 0x9E, 0x83, 0xDE, 0x5B, 0xD8,
      0x88, 0xF2, 0xEB, 0x03, 0x00, 0x8B, 0xB9, 0xFB, 0xA0, 0xCD, 0x5C, 0xB7,
      0x9D, 0xDB, 0x62, 0xE2, 0xF9, 0xF8, 0xF2, 0xE1, 0x1B, 0x22, 0x10, 0xDD,
      0xD1, 0x15, 0x90, 0xE2, 0x12, 0x14, 0x01, 0x50, 0x85, 0xA3, 0xCD, 0x8D,
      0xD1, 0x66, 0x5A, 0xD8, 0x1D, 0x28, 0xD5, 0x52, 0xB2, 0x14, 0xD9, 0x15,
      0x50, 0xFF, 0xB3, 0x01, 0x01, 0x02, 0x09, 0x8E, 0x6F, 0x76, 0x49, 0xA5,
      0xE0, 0x60, 0x57, 0x99, 0x8E, 0x25, 0xB4, 0xF7, 0x89, 0xB2, 0xFF, 0x35,
      0x62, 0x99, 0xC5, 0x89, 0xB1, 0xCE, 0x03, 0xD4, 0xAA, 0xDD, 0xC9, 0x17,
      0x7B, 0x47, 0x97, 0x4F, 0xBE, 0x20, 0x55, 0x16, 0xB3, 0xA9, 0xE8, 0x8F,
      0x0B, 0x39, 0x55, 0xF1, 0x33, 0xC2, 0x9C, 0x1E, 0xDB, 0x94, 0x96, 0x9E,
      0x89, 0xB2, 0xDA, 0x7B, 0x64, 0xEE, 0x42, 0x3A, 0x51, 0x67, 0x25, 0x21,
      0xCB, 0x4E, 0x10, 0x08, 0xD0, 0xAF, 0xCF, 0x68, 0xD6, 0x17, 0x2A, 0x17,
      0x4F, 0x0E, 0xCA, 0xBB, 0x5F, 0xB7, 0x64, 0x84, 0x44, 0x03, 0x2E, 0x64,
      0x43, 0xCF, 0x9C, 0x7F, 0xA4, 0xED, 0x4E, 0x17, 0x28, 0x42, 0x7C, 0x28,
      0x04, 0x69, 0x9A, 0x14, 0x3E, 0xDF, 0x11, 0xCD, 0x6C, 0xF9, 0xF5, 0xAF,
      0xF6, 0x34, 0x1A, 0x69, 0xDE, 0x7B, 0x77, 0x10, 0xC2, 0x23, 0xC7, 0x99,
      0x86, 0x12, 0x42, 0xC0, 0xD6, 0x88, 0x41, 0xF3, 0x6D, 0x92, 0x00, 0x12,
      0x23, 0x81, 0xEB, 0xF9, 0x4D, 0x22, 0x25, 0x8B, 0x38, 0x37, 0x5C, 0x0D,
      0x25, 0xDC, 0x1E, 0x3E, 0xEE, 0x00, 0xE3, 0x12, 0x43, 0x3C, 0xF8, 0x03,
      0x2F, 0x46, 0xEA, 0x0A, 0x0B, 0xCA, 0x36, 0xC6, 0x79, 0xF5, 0xFF, 0xC8,
      0x28, 0x98, 0x37, 0x2D, 0x63, 0x1F, 0xA7, 0x36, 0x85, 0x95, 0xED, 0x7C,
      0xB6, 0x57, 0x1E, 0x8C, 0x00, 0xC1, 0x6A, 0xFA, 0x4F, 0x47, 0x91, 0xE6,
      0x38, 0x13, 0x14, 0x1D, 0x80, 0x01, 0x11, 0xC1, 0xED, 0xCE, 0xFD, 0x88,
      0x72, 0xBE, 0xA4, 0x1B, 0x2D, 0xD1, 0x59, 0x67, 0x6F, 0x23, 0xBF, 0x85,
      0xC7, 0xDE, 0x07, 0xA3, 0x0D, 0x24, 0x59, 0xEB, 0x16, 0xB2, 0x4B, 0xDA,
      0xD3, 0x22, 0xC1, 0x77, 0xB1, 0x74, 0xBA, 0x83, 0xEA, 0xEB, 0x35, 0x2B,
      0x2C, 0x0D, 0x81, 0x40, 0xB3, 0x3D, 0x50, 0xB0, 0xC9, 0xF2, 0xDD, 0xD3,
      0x9F, 0x30, 0x5F, 0x9A, 0xB4, 0x56, 0x5D, 0x3A, 0xDF, 0x35, 0x55, 0x0F,
      0x02, 0x3F, 0x34, 0x69, 0x00, 0xC5, 0x99, 0x03, 0xCB, 0x97, 0x25, 0x79,
      0x39, 0x58, 0x40, 0xD6, 0xA8, 0x91, 0x69, 0xDA, 0x96, 0xD7, 0x01, 0x2A,
      0x82, 0x16, 0x32, 0xE6, 0x68, 0x21, 0x59, 0x1F, 0xD3, 0x73, 0x1E, 0xB0,
      0x68, 0xCB, 0xAF, 0x5F, 0x18, 0x58, 0x9E, 0x75, 0x41, 0xCB, 0x4A, 0xEB,
      0x0C, 0x48, 0x59, 0x02, 0x0D, 0x0A, 0x3B, 0x99, 0x88, 0x7D, 0x12, 0xE7,
      0x8F, 0x69, 0x00, 0x73, 0xD2, 0x9A, 0x24, 0x26, 0x82, 0xD4, 0x67, 0x7E,
      0x2F, 0x90, 0x4D, 0xFD, 0x72, 0x66, 0xE0, 0x51, 0x4F, 0x6E, 0x46, 0x0A,
      0xD2, 0xF1, 0xFC, 0x5C, 0xE8, 0x75, 0xD1, 0x06, 0x43, 0x2C, 0x5F, 0x91,
      0xD6, 0x24, 0x6D, 0x80, 0xAE, 0x5C, 0x7D, 0x8C, 0x4E, 0x4E, 0xE1, 0x22,
      0xF5, 0xFC, 0x58, 0x38, 0x03, 0x91, 0xED, 0x53, 0x6C, 0xA4, 0xF0, 0x73,
      0xE2, 0xB5, 0x5F, 0x5E, 0xD1, 0x97, 0x51, 0x47, 0x9C, 0xB3, 0xA7, 0x1E,
      0x72, 0xE8, 0xF3, 0x30, 0xE0, 0x86, 0x98, 0x11, 0x69, 0x14, 0x0A, 0x8A,
      0xD0, 0x69, 0xA4, 0x9B, 0x1F, 0xD4, 0xEB, 0x95, 0x0F, 0x9F, 0xB9, 0x08,
      0x4D, 0x1E, 0x70, 0x2B, 0x95, 0x7D, 0x08, 0x17, 0x09, 0xFC, 0x51, 0x0D,
      0x31, 0xCE, 0x2E, 0xB1, 0xE4, 0x2A, 0xB4, 0x74, 0xD7, 0x3A, 0x79, 0x32,
      0x31, 0xEC, 0x27, 0x4E, 0xE3, 0x99, 0x25, 0x7D, 0xEC, 0x4B, 0x9D, 0x07,
      0x04, 0x26, 0x5A, 0x77, 0xE7, 0x45, 0xF4, 0xCD, 0x8F, 0x9E, 0x49, 0x09,
      0xFB, 0x92, 0x99, 0x30, 0x4C, 0xE6, 0xAC, 0x7F, 0xA9, 0x5C, 0x3F, 0x47,
      0xB2, 0x10, 0xDA, 0x38, 0x8E, 0xE7, 0x48, 0xAB, 0xD4, 0x1C, 0xDE, 0xB3,
      0xCB, 0xB1, 0x87, 0xA4, 0x0B, 0xB3, 0xC3, 0xF4, 0x15, 0x0A, 0xDA, 0x34,
      0x72, 0xDD, 0xB3, 0x98, 0xDF, 0x8D, 0x8D, 0x58, 0x65, 0xEF, 0x27, 0xF5,
      0x4C, 0xBD, 0x41, 0x64, 0xDB, 0x04, 0x65, 0xDF, 0x74, 0x43, 0xE1, 0xD7,
      0x78, 0x2A, 0x24, 0x03, 0x38, 0x7E, 0xF0, 0x11, 0x8E, 0x8E, 0xB1, 0x02,
      0x83, 0xA3, 0x90, 0x72, 0xC3, 0x6A, 0x46, 0xE7, 0x24, 0x16, 0x5F, 0x49,
      0xED, 0x17, 0x0A, 0x8F, 0xCC, 0x00, 0xFA, 0xDF, 0x03, 0x7E, 0xBF, 0xC6,
      0x52, 0x81, 0x06, 0x97, 0xCD, 0x71, 0x08, 0xA5, 0x69, 0x89, 0x49, 0x1B,
      0xA2, 0x23, 0x2B, 0xED, 0x54, 0xBA, 0x16, 0x7D, 0xB9, 0xD5, 0x26, 0xFA,
      0xD3, 0x48, 0xB7, 0x78, 0x7A, 0xE7, 0x15, 0x57, 0xC5, 0x9B, 0x38, 0xD1,
      0x05, 0x4C, 0xBE, 0x9B, 0xCB, 0x45, 0xE1, 0x7B, 0xB1, 0x46, 0xAD, 0xFC,
      0x04, 0xF6, 0x81, 0x63, 0x34, 0xD3, 0xD8, 0x15, 0xBF, 0x73, 0xFB, 0xEC,
      0xF1, 0xC9, 0xD4, 0x37, 0xD3, 0xA4, 0x7A, 0x78, 0x62, 0xC2, 0x86, 0x88,
      0x4B, 0x63, 0x8D, 0xD7, 0xFB, 0x64, 0xBE, 0x01, 0xB7, 0x98, 0x1D, 0x7C,
      0xC8, 0x2A, 0x90, 0x93, 0x8E, 0xBD, 0xD9, 0x66, 0x4D, 0xEB, 0xAF, 0x0E,
      0x03, 0xA5, 0x95, 0x36, 0x8B, 0x89, 0x33, 0x8A, 0xB0, 0x76, 0xBD, 0x22,
      0xAD, 0x57, 0x9D, 0x04, 0xF1, 0xC7, 0xEA, 0x07, 0x5D, 0x22, 0x0F, 0x8C,
      0xCD, 0xC0, 0xB9, 0xE3, 0x72, 0xF5, 0x19, 0x4A, 0xFE, 0x83, 0x4D, 0x28};

  CBS cbs;
  auto priv = std::make_unique<DILITHIUM_private_key>();
  CBS_init(&cbs, encoded_private_key, sizeof(encoded_private_key));
  ASSERT_TRUE(DILITHIUM_parse_private_key(priv.get(), &cbs));

  static const uint8_t message[] = {
      0x4D, 0x3C, 0x4D, 0x95, 0x2A, 0x1D, 0xAC, 0x15, 0x17, 0x36, 0xAE, 0x9D,
      0x0A, 0xD8, 0x1C, 0xD3, 0x7F, 0x7C, 0x49, 0x25, 0x39, 0xFC, 0xC9, 0x16,
      0xA4, 0xB2, 0x25, 0x13, 0x09, 0xE0, 0x6C, 0xED, 0x54, 0xD7, 0x14, 0x58,
      0x68, 0xD1, 0x45, 0xBC, 0x8D, 0x16, 0xF8, 0xB3, 0x64, 0xE5, 0xD6, 0x02,
      0x6E, 0x11, 0x3B, 0xEC, 0x48, 0x24, 0xF6, 0xBF, 0xAE, 0x3C, 0x9F, 0x6C,
      0x8B, 0x88, 0x8E, 0xC6, 0xB8, 0x25, 0x4C, 0xE5, 0x9E, 0x9E, 0x8E, 0x15,
      0x8E, 0xB0, 0x40, 0x77, 0x84, 0x12, 0x43, 0xEE, 0x3A, 0x89, 0xD2, 0x5C,
      0x0B, 0x30, 0xB4, 0xA6, 0x4B, 0x72, 0x36, 0x97, 0x33, 0x0A, 0xA8, 0x76,
      0x63, 0xB2, 0x0E, 0xA4, 0x51, 0x31, 0x89, 0x55, 0x84, 0x66, 0xD2, 0x64,
      0xC7, 0x90, 0x88, 0x41, 0x5E, 0xFA, 0x09, 0xDE, 0x9B, 0x64, 0xAB, 0x01,
      0xB6, 0x40, 0xB2, 0x88, 0xD1, 0xD5, 0x56, 0x2E, 0xD7, 0x44, 0xE8, 0x0C,
      0xAF, 0x0B, 0x6E, 0x2C, 0x29, 0xFF, 0xA1, 0xC3, 0x21, 0x12, 0x9E, 0xBC,
      0x70, 0xCA, 0xA7, 0x3D, 0xCA, 0x7D, 0xCE, 0x89, 0x95, 0x71, 0x26, 0x0B,
      0x2E, 0x7D, 0xA1, 0xB4, 0x7C, 0xEC, 0x5B, 0xB0, 0x0F, 0xD6, 0x6A, 0xF8,
      0xB8, 0x9B, 0xB8, 0x43, 0x92, 0x89, 0x7B, 0xDD, 0x1A, 0x5E, 0xC7, 0x18,
      0x66, 0x70, 0xDB, 0x81, 0x60, 0x4E, 0x0B, 0x94, 0x9B, 0x5A, 0xB1, 0xD1,
      0x95, 0x22, 0xA3, 0xF0, 0x82, 0x8D, 0x0E, 0x77, 0xB3, 0x00, 0xA6, 0x3E,
      0xDC, 0xB7, 0x4C, 0x48, 0xF9, 0xA0, 0x1C, 0xA9, 0x18, 0x16, 0xFB, 0xEE,
      0x01, 0x1A, 0x5B, 0x01, 0x45, 0xC6, 0xA0, 0xB4, 0xEF, 0xD5, 0xA3, 0x00,
      0x8A, 0x91, 0xE0, 0xB1, 0xAC, 0x96, 0xDD, 0x8C, 0xB9, 0x41, 0x0D, 0x56,
      0xA9, 0x52, 0xE7, 0x4B, 0x7F, 0xB6, 0x84, 0xA0, 0xB1, 0x66, 0xE5, 0xF0,
      0xDF, 0x9A, 0xD1, 0x00, 0xEE, 0x1C, 0x9E, 0x3B, 0x7B, 0x85, 0x1B, 0x4C,
      0xBF, 0x33, 0x7A, 0x33, 0x28, 0xDB, 0x13, 0x97, 0x74, 0x51, 0xE9, 0x92,
      0xC4, 0xA0, 0xB4, 0x54, 0xBA, 0xC5, 0xE3, 0xA2, 0xD0, 0xE6, 0x4D, 0x10,
      0x6D, 0xC4, 0x7C, 0x21, 0x6E, 0x0C, 0x0A, 0xC1, 0x1B, 0x0F, 0x7C, 0x48,
      0x58, 0x9E, 0x57, 0x30, 0x51, 0x13, 0x3D, 0x01, 0xF2, 0x95, 0x79, 0xC7,
      0xE6, 0x59, 0xF8, 0xFB, 0x8E, 0x65, 0x39, 0xB0, 0xF6, 0x18, 0xD0, 0x53,
      0xF4, 0x32, 0x3A, 0xD0, 0x23, 0x7A, 0x58, 0x94, 0x9F, 0x68, 0x94, 0x58,
      0xAC, 0xC1, 0x09, 0xBE, 0xB7, 0xC8, 0x19, 0x52, 0xA8, 0x79, 0xA1, 0x0B,
      0xF8, 0xBC, 0x01, 0xCE, 0x05, 0x7E, 0x93, 0x6B, 0x62, 0x01, 0x7E, 0x22,
      0x6D, 0xD6, 0x6A, 0xB2, 0x47, 0x08, 0x52, 0xC0, 0x00, 0xA6, 0x18, 0xB9,
      0xB1, 0x30, 0x05, 0x0F, 0x6B, 0x04, 0x77, 0x46, 0xD0, 0x68, 0xFE, 0x7D,
      0x3C, 0xC1, 0x27, 0x7E, 0x4F, 0xF4, 0x40, 0x0F, 0x37, 0x2D, 0x50, 0xA8,
      0x5F, 0x46, 0x9B, 0x4C, 0x7A, 0x35, 0xAA, 0x7F, 0xE8, 0x8D, 0xD9, 0x23,
      0xAC, 0xEF, 0x5F, 0xFF, 0x47, 0xA9, 0x5F, 0x97, 0x1E, 0xFE, 0xA6, 0xB6,
      0x5D, 0x33, 0xE6, 0x5F, 0x20, 0x17, 0x3C, 0x5B, 0xA5, 0xA3, 0x20, 0xDD,
      0x16, 0xCA, 0xAB, 0x59, 0xAA, 0x7A, 0x7B, 0xB9, 0x99, 0x56, 0xEB, 0x5E,
      0xD5, 0x4E, 0xB9, 0x3E, 0x89, 0xC8, 0x87, 0xA3, 0x0D, 0x6D, 0x38, 0x43,
      0x45, 0xE2, 0x12, 0x91, 0xB8, 0xE9, 0xC6, 0xCD, 0xCB, 0xE2, 0x67, 0x1F,
      0x81, 0x0A, 0x05, 0x8B, 0x85, 0x82, 0x20, 0x14, 0x6A, 0x21, 0x61, 0xB7,
      0x41, 0x2C, 0xE2, 0xDB, 0x81, 0x7C, 0xC8, 0xC9, 0x95, 0x28, 0x19, 0x5F,
      0x56, 0x7A, 0x92, 0xE1, 0x98, 0x33, 0x0A, 0xAA, 0x41, 0x8A, 0x5F, 0xAB,
      0x75, 0x4E, 0x27, 0x22, 0x1A, 0x21, 0x65, 0xEC, 0x4D, 0x50, 0xB3, 0x91,
      0xBB, 0x8A, 0xB2, 0xBE, 0xCC, 0xAA, 0x44, 0x9C, 0xA1, 0xD4, 0xD3, 0xD5,
      0xC1, 0x7E, 0x28, 0x57, 0x30, 0x48, 0x45, 0x2E, 0xE4, 0x12, 0x06, 0x93,
      0x50, 0x55, 0x15, 0xCF, 0x8D, 0x1E, 0x2F, 0x89, 0x84, 0xD3, 0x63, 0x46,
      0x4D, 0xB7, 0x86, 0xE9, 0xB8, 0xB7, 0xBD, 0xED, 0xFA, 0x23, 0x01, 0xA2,
      0x1F, 0x3C, 0x46, 0x9F, 0x01, 0x3E, 0x27, 0x74, 0x24, 0x05, 0xCF, 0xD0,
      0xD1, 0x5E, 0x70, 0xBE, 0x05, 0x60, 0xE8, 0xB1, 0x8C, 0x82, 0x32, 0x78,
      0x61, 0xA8, 0xDF, 0x64, 0x20, 0x0A, 0x51, 0xE6, 0x9D, 0x27, 0x91, 0xD4,
      0x1E, 0x1E, 0x12, 0xBF, 0x37, 0x16, 0x64, 0x27, 0x0A, 0xFB, 0xBE, 0x1C,
      0x84, 0xFF, 0x19, 0xD1, 0x1D, 0x4C, 0x23, 0x3E, 0xCD, 0xB8, 0x03, 0x3B,
      0xC7, 0x9B, 0xC7, 0x96, 0x7C, 0x41, 0x0C, 0x13, 0xE8, 0xE3, 0xF9, 0x34,
      0x0F, 0x84, 0x85, 0x10, 0x76, 0x0D, 0xAB, 0xCC, 0xAB, 0x6E, 0xF7, 0xBB,
      0x87, 0x39, 0x8B, 0x14, 0x21, 0xE5, 0x80, 0x4F, 0x6D, 0x41, 0xAB, 0xC0,
      0x9E, 0x65, 0x42, 0xD7, 0x2B, 0x87, 0x11, 0x5F, 0x12, 0x81, 0x45, 0x0C,
      0x66, 0xAF, 0x3B, 0xF4, 0xFD, 0xC2, 0x1C, 0xB9, 0xB8, 0xC2, 0x7D, 0x0F,
      0xC6, 0xDD, 0xFA, 0x73, 0x73, 0x2A, 0x87, 0x7C, 0xB7, 0xD7, 0xC0, 0x9B,
      0x8E, 0x9F, 0x46, 0x03, 0xD9, 0x86, 0x5E, 0x4A, 0xF9, 0x1C, 0x58, 0xE9,
      0xEF, 0x6F, 0xD9, 0xA3, 0x2C, 0x91, 0xEE, 0xAA, 0xEF, 0x17, 0x0B, 0xF4,
      0xF4, 0x4B, 0xF5, 0x86, 0xD7, 0x89, 0xEF, 0x45, 0x71, 0xDE, 0xBE, 0x6F,
      0xFF, 0x0C, 0x43, 0xE8, 0x42, 0x61, 0x7F, 0xA5, 0x0F, 0xA6, 0xEA, 0xC8,
      0xC7, 0xF7, 0x0E, 0xB6, 0x80, 0xBE, 0x71, 0x76, 0x88, 0x18, 0x8A, 0x43,
      0x04, 0x84, 0xF3, 0x38, 0x8F, 0x62, 0x19, 0xA7, 0x0E, 0xA3, 0x8B, 0xF3,
      0xC6, 0xDB, 0xBC, 0x7F, 0xF3, 0xA7, 0x49, 0xD8, 0xC8, 0xE5, 0x1A, 0x13,
      0x65, 0xAE, 0x37, 0x29, 0x88, 0xD2, 0x48, 0xF9, 0xFC, 0x0F, 0x2B, 0x67,
      0xE8, 0x96, 0xCB, 0xC8, 0x6D, 0x2A, 0xED, 0x35, 0xF7, 0xAD, 0x8D, 0x4E,
      0x93, 0xC2, 0x1D, 0xA0, 0x9E, 0x02, 0x04, 0xC7, 0xDB, 0x92, 0xFA, 0xFA,
      0x71, 0x19, 0xA6, 0x3B, 0xD0, 0xCC, 0x00, 0xA0, 0x29, 0xCE, 0x7D, 0xB9,
      0x3D, 0xF3, 0x4C, 0x33, 0xEC, 0xF6, 0x73, 0xD4, 0x8C, 0xE3, 0xF4, 0x09,
      0x48, 0xDE, 0x86, 0xCA, 0x54, 0x67, 0xCC, 0x15, 0x99, 0x60, 0xB3, 0x29,
      0xF5, 0x60, 0xB2, 0x0E, 0x5C, 0x8A, 0xC3, 0x7A, 0xC4, 0xC8, 0x96, 0x0B,
      0x49, 0xB4, 0x55, 0x07, 0x54, 0x89, 0xE7, 0x21, 0x7D, 0x6B, 0xC5, 0x3A,
      0xEE, 0x7C, 0x19, 0xB3, 0xFD, 0xE9, 0xFF, 0xF6, 0x67, 0xEA, 0xE8, 0x9A,
      0xA6, 0xCE, 0x8C, 0x41, 0x5E, 0x80, 0x7B, 0x34, 0x8A, 0x14, 0x59, 0x83,
      0x29, 0xA3, 0x2D, 0x9E, 0x50, 0x8C, 0x34, 0x09, 0xDF, 0x18, 0xEB, 0x42,
      0x6B, 0xDB, 0xFD, 0xDD, 0x88, 0xAE, 0xB3, 0xA9, 0xBC, 0xB9, 0xD8, 0x4C,
      0x23, 0x88, 0xAD, 0x79, 0xAA, 0xBF, 0xA3, 0x44, 0x5F, 0x80, 0xA9, 0xA6,
      0x8E, 0xF7, 0x40, 0x96, 0xE3, 0x45, 0x16, 0xCB, 0x53, 0x16, 0xDF, 0x59,
      0x9E, 0x58, 0x36, 0xF1, 0x23, 0x27, 0x1C, 0xF7, 0x29, 0x47, 0x29, 0x52,
      0xE0, 0x6C, 0x81, 0x7A, 0x67, 0xC2, 0x19, 0x38, 0x2F, 0x22, 0xC4, 0xB0,
      0xA1, 0x6F, 0x7B, 0x7D, 0xDE, 0x9E, 0xCF, 0x4A, 0xF6, 0x92, 0x7B, 0xBF,
      0x44, 0x24, 0xF4, 0x3F, 0xCE, 0xB1, 0xA4, 0x8F, 0x1F, 0x2C, 0xBD, 0x98,
      0xE7, 0x59, 0x34, 0x64, 0xEB, 0xDE, 0x45, 0xD6, 0x04, 0xB6, 0xAF, 0x95,
      0xE3, 0x2A, 0xE7, 0x28, 0xD9, 0x21, 0xB3, 0x90, 0xF7, 0x12, 0xAD, 0xC2,
      0x80, 0x25, 0x7F, 0x44, 0x20, 0x74, 0x1C, 0x0D, 0x29, 0xE0, 0xE6, 0xB9,
      0x8E, 0xC1, 0x70, 0x85, 0x69, 0x38, 0x1B, 0xC8, 0x45, 0x42, 0x0D, 0xB0,
      0xC1, 0xBD, 0xC9, 0xA2, 0x3D, 0xAE, 0xC2, 0x09, 0x37, 0x8A, 0x01, 0x70,
      0x3B, 0x9C, 0x16, 0xC1, 0x0F, 0xC2, 0x97, 0x63, 0xA5, 0x2F, 0xEE, 0x64,
      0xF0, 0xA7, 0x28, 0x19, 0x85, 0x13, 0x36, 0x1C, 0xD9, 0x81, 0xDA, 0xC7,
      0x39, 0x93, 0x7F, 0xC4, 0x72, 0xCC, 0xCF, 0x66, 0x6E, 0x61, 0xDA, 0x3D,
      0x90, 0xC3, 0x65, 0x6F, 0x0B, 0xBD, 0xE6, 0xB9, 0x42, 0xD8, 0xA0, 0x49,
      0x6C, 0x9B, 0xC7, 0x03, 0xF7, 0x42, 0x4B, 0xF0, 0x08, 0x48, 0x80, 0x0A,
      0xAD, 0x51, 0x93, 0x47, 0xA4, 0x74, 0x76, 0x76, 0xC8, 0x10, 0x88, 0x10,
      0x9A, 0x2B, 0x67, 0x84, 0x8E, 0xE0, 0xA0, 0x8B, 0xD1, 0x40, 0x9E, 0x77,
      0x44, 0x34, 0xEF, 0xFE, 0x75, 0x4B, 0xDE, 0xCA, 0xFF, 0x23, 0x28, 0xED,
      0xC3, 0x0F, 0xDB, 0x85, 0xB4, 0x54, 0x88, 0xDE, 0x0A, 0x5D, 0x90, 0x52,
      0x39, 0x7F, 0x9D, 0xDB, 0x0F, 0x13, 0x86, 0x5D, 0xCA, 0x85, 0xA3, 0x6F,
      0x6B, 0x43, 0xB8, 0x4E, 0xD0, 0x44, 0xE5, 0xF4, 0x0B, 0xB9, 0x60, 0xE5,
      0x6D, 0xD0, 0xCD, 0x6E, 0xC0, 0x8C, 0x37, 0xB0, 0x17, 0xFE, 0xED, 0xBA,
      0x4A, 0x7F, 0x20, 0xA1, 0x74, 0xE1, 0x14, 0x6C, 0x44, 0x1D, 0xB9, 0x9A,
      0x2B, 0xF3, 0x4B, 0xB8, 0xD8, 0xFC, 0xAE, 0xB8, 0x74, 0xB2, 0x31, 0x3A,
      0xF2, 0xE2, 0x0B, 0x01, 0xD3, 0x8B, 0x56, 0xB4, 0x24, 0x51, 0x04, 0x7E,
      0xAF, 0xF5, 0x92, 0x23, 0x6C, 0xB2, 0x41, 0x85, 0x77, 0x54, 0xF6, 0xE2,
      0xDA, 0x3E, 0x4F, 0xAD, 0x07, 0x0F, 0xD1, 0x95, 0xB6, 0x41, 0x56, 0x73,
      0x75, 0xB6, 0xF2, 0x74, 0x4F, 0xE1, 0x69, 0x96, 0xD7, 0xA7, 0x3D, 0x09,
      0x5F, 0x5C, 0xFF, 0xDD, 0x50, 0xA9, 0x65, 0x08, 0xFB, 0xEC, 0xF8, 0x02,
      0x62, 0x5E, 0x5F, 0x16, 0x8D, 0xA0, 0x8C, 0x8D, 0x51, 0x98, 0xBB, 0x27,
      0x02, 0xF1, 0xCB, 0xDE, 0xA8, 0x22, 0x1B, 0xCF, 0xC3, 0x01, 0x19, 0x5B,
      0x71, 0x7A, 0x0D, 0x17, 0x4E, 0xEC, 0x9B, 0xB9, 0x6E, 0x7C, 0x49, 0xE1,
      0x5C, 0xD2, 0xA2, 0x6B, 0x93, 0x3E, 0x0C, 0x6B, 0xB1, 0x1D, 0x70, 0x2F,
      0x1A, 0x21, 0xD3, 0x3B, 0xE4, 0x80, 0x39, 0xE8, 0x9E, 0x38, 0xC2, 0xA5,
      0xE9, 0x6F, 0x55, 0xA5, 0x57, 0x35, 0x9B, 0xBC, 0x8E, 0x19, 0x17, 0x19,
      0xEC, 0x5E, 0x56, 0x69, 0xB2, 0xF1, 0x68, 0x91, 0x2C, 0x91, 0x77, 0xB3,
      0x20, 0xA5, 0x76, 0x24, 0x74, 0xCB, 0x23, 0x03, 0xF5, 0x4E, 0x78, 0xF4,
      0xDB, 0xB9, 0x14, 0xEA, 0x35, 0x4F, 0x8C, 0x9A, 0x3E, 0x81, 0x34, 0x2A,
      0x10, 0xAC, 0xF5, 0xFD, 0x4A, 0x3D, 0x8C, 0xD6, 0xF8, 0x5A, 0x20, 0x4E,
      0x16, 0x5D, 0x59, 0xE4, 0xFD, 0x12, 0xD5, 0xE5, 0xE9, 0x00, 0xAC, 0x94,
      0xDB, 0x98, 0x4D, 0xD8, 0xA5, 0xA7, 0x23, 0x72, 0xD8, 0xB5, 0xD8, 0x95,
      0x8E, 0xC7, 0xF9, 0xE6, 0x4C, 0xEA, 0x4E, 0x56, 0x0A, 0x8D, 0x95, 0x25,
      0xCB, 0x24, 0xD3, 0xB2, 0xF3, 0x6E, 0xD7, 0xD4, 0x6F, 0x89, 0x72, 0x29,
      0xA7, 0x76, 0xE7, 0xE4, 0x22, 0x83, 0x8D, 0x4E, 0x2B, 0x55, 0x1B, 0x28,
      0xD0, 0x12, 0x7F, 0x2B, 0x76, 0x44, 0xF3, 0x5F, 0xBC, 0x35, 0xBE, 0xF9,
      0xD8, 0x56, 0xEF, 0x44, 0x7F, 0x69, 0x91, 0x51, 0xF7, 0x3A, 0xA7, 0x0C,
      0xEE, 0xA5, 0x5F, 0x61, 0x93, 0x06, 0x07, 0x62, 0xB2, 0x8F, 0xA1, 0x50,
      0xA8, 0x33, 0x44, 0xA2, 0xE2, 0x3B, 0x9E, 0x5E, 0x1B, 0x10, 0x38, 0x05,
      0xDE, 0x85, 0xC5, 0x20, 0xEA, 0xCC, 0xFD, 0x67, 0xA5, 0xC3, 0x90, 0x33,
      0x4B, 0x03, 0x60, 0x28, 0xD4, 0xB6, 0x96, 0xAE, 0x3D, 0x6F, 0x6D, 0xD8,
      0x03, 0x04, 0xF7, 0xCF, 0x91, 0x5F, 0x56, 0x44, 0x09, 0xAE, 0x65, 0xAB,
      0x75, 0x04, 0xD8, 0xFE, 0xD4, 0x34, 0x96, 0x6A, 0xA4, 0xD4, 0xDB, 0x3A,
      0x3C, 0x90, 0xB1, 0x60, 0x33, 0x2F, 0xB8, 0xE6, 0xA3, 0x3A, 0x39, 0x21,
      0x22, 0xE3, 0xC0, 0x2F, 0x27, 0xC3, 0x8E, 0x37, 0xB8, 0xB2, 0x17, 0x95,
      0x89, 0x12, 0xD0, 0xAB, 0x98, 0x7D, 0x23, 0x2B, 0x65, 0xD7, 0xFD, 0xF3,
      0xF2, 0x8C, 0xD5, 0x07, 0xD5, 0x3F, 0x5C, 0xAA, 0xCE, 0x75, 0x22, 0x4A,
      0xA3, 0xF6, 0x3A, 0x4D, 0x45, 0x70, 0x62, 0x75, 0x61, 0xAA, 0x14, 0x6A,
      0x32, 0x64, 0x5C, 0x89, 0xBA, 0x18, 0xCD, 0x4C, 0x7D, 0x2A, 0xC0, 0xD9,
      0x12, 0x3C, 0x89, 0xAA, 0x7D, 0x4A, 0xEE, 0x9C, 0xF8, 0x11, 0xDC, 0x87,
      0x7D, 0x7D, 0x03, 0x21, 0xAB, 0x5C, 0x7E, 0xAE, 0x86, 0xF2, 0xEB, 0xD6,
      0x1A, 0xAE, 0x68, 0x1E, 0xB5, 0x3E, 0xEA, 0x73, 0xD0, 0xD7, 0x53, 0x29,
      0x37, 0xAA, 0xF8, 0xF8, 0xA2, 0x3E, 0x0A, 0xE6, 0x4F, 0xCB, 0xAF, 0x78,
      0x21, 0x69, 0x42, 0x2A, 0xDB, 0x6C, 0xE8, 0xF3, 0xF0, 0x03, 0xCD, 0x2E,
      0xE9, 0xEF, 0xC3, 0x7E, 0x5E, 0xA1, 0xA8, 0x90, 0x2D, 0x3D, 0x12, 0x46,
      0xA8, 0x97, 0x36, 0xC6, 0xC7, 0xC9, 0xA4, 0x6F, 0xE2, 0xD2, 0xDB, 0x3B,
      0xEC, 0x00, 0xE6, 0x91, 0x66, 0xAB, 0xD2, 0x13, 0xA6, 0xD3, 0x04, 0x94,
      0x24, 0x4F, 0xEE, 0x28, 0x05, 0x0B, 0xFA, 0xE5, 0x66, 0xDC, 0xC9, 0x1C,
      0x95, 0x28, 0x88, 0x30, 0x11, 0xF2, 0x15, 0x99, 0x5D, 0xC9, 0x2E, 0x57,
      0x74, 0x21, 0xC9, 0x64, 0x54, 0x78, 0xFD, 0x3C, 0x3D, 0x7D, 0x8F, 0xE0,
      0x6B, 0xD1, 0x82, 0x4A, 0xA2, 0x0B, 0x4A, 0x67, 0xCF, 0x9E, 0x8C, 0x2A,
      0x69, 0xC0, 0x31, 0xE4, 0xE1, 0xFC, 0xBC, 0xBB, 0x61, 0x70, 0xD6, 0x08,
      0x38, 0xC6, 0x1A, 0xF3, 0x73, 0x1C, 0x7F, 0xD4, 0xDE, 0x4D, 0x62, 0x77,
      0xFD, 0x66, 0x52, 0xB6, 0x33, 0x18, 0xCF, 0x09, 0x3F, 0x7D, 0xF4, 0x6D,
      0x80, 0x03, 0x68, 0xA7, 0x13, 0x33, 0xBA, 0x65, 0x71, 0x71, 0xAD, 0xCE,
      0x4D, 0x36, 0x0E, 0x24, 0x9B, 0x32, 0xF4, 0x9F, 0x93, 0xE3, 0x8C, 0x22,
      0xEC, 0xF7, 0x9C, 0x89, 0x64, 0x75, 0x64, 0xB7, 0xF8, 0x3E, 0x26, 0xA5,
      0x9D, 0x40, 0x3B, 0x23, 0x9A, 0x25, 0xF7, 0x62, 0x9C, 0xEC, 0x50, 0xED,
      0xE1, 0xEF, 0xB5, 0x8D, 0x90, 0x17, 0xED, 0x44, 0xCE, 0x34, 0xA3, 0x13,
      0x65, 0x18, 0xFA, 0x00, 0x9C, 0x8B, 0xEC, 0x05, 0x7B, 0xA0, 0x54, 0xC7,
      0x94, 0xEA, 0x0F, 0x49, 0xF3, 0xC5, 0x81, 0xFD, 0x91, 0x00, 0x98, 0x35,
      0x94, 0x67, 0x42, 0xB8, 0x06, 0xD1, 0x15, 0xD4, 0xE2, 0xDD, 0x10, 0x9E,
      0x17, 0x97, 0xF7, 0xDB, 0x55, 0x75, 0xCF, 0x23, 0x39, 0x80, 0xB2, 0xD0,
      0x72, 0xD2, 0x47, 0xE7, 0x04, 0xEF, 0x40, 0xA6, 0xDD, 0x66, 0x51, 0x36,
      0x9C, 0x67, 0xB5, 0x76, 0xFA, 0x7D, 0xE4, 0xC7, 0x94, 0xE4, 0x69, 0xEB,
      0x2D, 0xB8, 0xC0, 0xE4, 0x24, 0xD0, 0x7C, 0xA0, 0x17, 0x53, 0x67, 0x11,
      0x16, 0xFD, 0xCB, 0x9B, 0x37, 0xC0, 0x51, 0x76, 0x86, 0xD8, 0x51, 0x11,
      0xB8, 0xE1, 0x2A, 0x53, 0x0C, 0x25, 0xF0, 0xD0, 0x1E, 0xCB, 0x20, 0xE2,
      0x59, 0xCC, 0xCD, 0xB0, 0xA0, 0x81, 0xAC, 0xA5, 0xCD, 0x06, 0xE9, 0xFE,
      0x50, 0xD7, 0x9B, 0x9A, 0x74, 0x12, 0xE0, 0x2F, 0x41, 0xB5, 0x3C, 0xE3,
      0x55, 0x5E, 0x98, 0x68, 0x33, 0xD2, 0x53, 0x53, 0x2C, 0xA5, 0xD2, 0x7A,
      0x4F, 0x0C, 0x72, 0xFB, 0x69, 0x8C, 0xAD, 0xE0, 0x1B, 0x36, 0xCF, 0x94,
      0xC5, 0xA0, 0x59, 0x07, 0xAB, 0x3A, 0xC6, 0x31, 0x45, 0x5B, 0x4A, 0xE9,
      0x18, 0x03, 0xF2, 0xA3, 0xA8, 0x3E, 0x9E, 0xE2, 0x3E, 0xB0, 0x53, 0x88,
      0x40, 0xE1, 0xC9, 0x40, 0x8F, 0x73, 0x20, 0x5D, 0x79, 0x30, 0xAF, 0x7C,
      0xAA, 0x7B, 0x4E, 0xEC, 0x2D, 0x47, 0xA7, 0x1B, 0x99, 0xC0, 0x7B, 0x09,
      0xFB, 0x4B, 0xA1, 0xEC, 0xC0, 0x36, 0xBA, 0xAF, 0x52, 0x30, 0xD6, 0x00,
      0x55, 0x0C, 0xEB, 0xFC, 0xAF, 0x49, 0xA1, 0x9D, 0x63, 0xB3, 0xB5, 0x00,
      0x16, 0xCC, 0x80, 0x77, 0xD9, 0x7C, 0x27, 0x37, 0x6E, 0x2A, 0xAA, 0x37,
      0x0B, 0xB3, 0xCA, 0x21, 0x39, 0x68, 0xD2, 0x42, 0xC1, 0x1A, 0xC0, 0xA3,
      0x41};

  std::vector<uint8_t> encoded_signature(DILITHIUM_SIGNATURE_BYTES);
  EXPECT_TRUE(DILITHIUM_sign_deterministic(encoded_signature.data(), priv.get(),
                                           message, sizeof(message)));

  static const uint8_t expected_signature[DILITHIUM_SIGNATURE_BYTES] = {
      0x76, 0x5A, 0xB6, 0x8A, 0xC2, 0x04, 0xB2, 0x55, 0x69, 0x96, 0x18, 0x46,
      0x3D, 0x65, 0xD6, 0x93, 0xA1, 0x84, 0x5D, 0x57, 0x41, 0x81, 0xD1, 0x40,
      0x39, 0x72, 0x88, 0x8A, 0x76, 0xB3, 0xA5, 0x53, 0xF7, 0x9F, 0xEF, 0xD0,
      0x5F, 0xF2, 0x2F, 0x37, 0xA9, 0x9C, 0x52, 0x5F, 0xCE, 0x21, 0xF2, 0x83,
      0xE4, 0x37, 0xFD, 0xC3, 0x6F, 0x8E, 0xF4, 0xE9, 0xC3, 0x62, 0x8B, 0xDA,
      0xA7, 0xCE, 0xB6, 0xEB, 0x22, 0x8C, 0x23, 0x11, 0x80, 0xB6, 0x8D, 0xE6,
      0x34, 0x21, 0x23, 0x81, 0x41, 0x66, 0x4A, 0xE3, 0x00, 0x58, 0x60, 0xA4,
      0x22, 0x1A, 0xFA, 0xB1, 0x8D, 0xA8, 0x3F, 0x3A, 0x58, 0x64, 0x91, 0xB1,
      0x92, 0xC8, 0x52, 0x01, 0x77, 0x81, 0xFE, 0x44, 0x9A, 0x54, 0xE5, 0xAB,
      0x16, 0x6D, 0x97, 0x2A, 0x18, 0x2D, 0x3B, 0x4B, 0xD3, 0xD0, 0xAA, 0x5D,
      0x0B, 0xE6, 0x3C, 0xFF, 0xA7, 0x8D, 0x3B, 0x1E, 0xC7, 0x01, 0x4D, 0x1D,
      0x65, 0xBB, 0x94, 0x15, 0x3C, 0xC4, 0xAA, 0xF1, 0xCD, 0xAD, 0x92, 0xDD,
      0x3F, 0x94, 0x60, 0xE9, 0x12, 0x38, 0xE5, 0x51, 0xA6, 0x7D, 0xA3, 0x98,
      0x0C, 0xBC, 0x95, 0x1F, 0x58, 0x25, 0x0D, 0x32, 0x98, 0x4B, 0xDF, 0x98,
      0x83, 0x74, 0x86, 0xAA, 0x9E, 0x1D, 0x7F, 0x00, 0x72, 0xBB, 0x73, 0x1C,
      0xA6, 0x2C, 0x23, 0x16, 0xDB, 0xF8, 0x52, 0xC8, 0xAC, 0xC3, 0x69, 0x6C,
      0xAC, 0x30, 0x37, 0x87, 0x19, 0x59, 0xB7, 0xC4, 0x63, 0x3E, 0x17, 0x47,
      0x9D, 0x53, 0x61, 0x1F, 0x8A, 0xB8, 0x02, 0x57, 0xA8, 0xD2, 0x7A, 0x3C,
      0x32, 0xEC, 0x62, 0xD6, 0xE6, 0x8C, 0xDE, 0x16, 0xCF, 0x4E, 0xD8, 0x67,
      0xD3, 0x3C, 0xEB, 0x4A, 0x96, 0x3C, 0x91, 0x95, 0x13, 0x33, 0x86, 0x27,
      0x88, 0xD4, 0x21, 0x62, 0xBE, 0x81, 0x73, 0x16, 0x6A, 0x10, 0x42, 0xE6,
      0x7F, 0xC8, 0x1F, 0x37, 0xE4, 0xB3, 0x3D, 0x68, 0xE3, 0x90, 0x6F, 0xB5,
      0x1F, 0x09, 0xEF, 0x15, 0xB0, 0x83, 0xF2, 0xF2, 0x6B, 0xBE, 0xB1, 0x4B,
      0xDE, 0x76, 0x5A, 0x9C, 0xC8, 0x30, 0x44, 0x5C, 0x1B, 0x8D, 0x4C, 0x3B,
      0xBC, 0x43, 0x8F, 0xCB, 0x76, 0xAC, 0xC8, 0xB4, 0x3D, 0x43, 0xA9, 0x4E,
      0x85, 0xEC, 0x2D, 0x64, 0xCE, 0x02, 0x76, 0xA3, 0xBE, 0x11, 0xD5, 0x69,
      0x6A, 0xC2, 0x9F, 0xE8, 0x62, 0x4C, 0x30, 0x61, 0x9A, 0x7F, 0x9E, 0x71,
      0x6F, 0x06, 0xD0, 0x04, 0xFC, 0xB9, 0xBE, 0x45, 0x13, 0xAC, 0x17, 0x19,
      0xA6, 0x88, 0xFE, 0xBB, 0xED, 0x40, 0xB7, 0x78, 0x13, 0x40, 0xE6, 0xAF,
      0xCE, 0xAF, 0x52, 0x3A, 0x33, 0x7B, 0x90, 0xDC, 0x96, 0x4F, 0x15, 0x80,
      0xC7, 0x8E, 0x26, 0x09, 0x3C, 0xD9, 0x52, 0xE1, 0xF9, 0x4D, 0xAE, 0xFC,
      0x15, 0xF0, 0xE5, 0xB2, 0x89, 0xC0, 0x70, 0x49, 0x64, 0xD9, 0xDE, 0x70,
      0xBD, 0x8D, 0xFD, 0xC6, 0xF8, 0x57, 0xD9, 0x25, 0xC5, 0x24, 0x98, 0x06,
      0xA1, 0x6D, 0x42, 0x33, 0x04, 0x0B, 0x04, 0x9D, 0x31, 0xE7, 0x20, 0x1B,
      0x52, 0x75, 0x6F, 0x86, 0x38, 0x36, 0xE6, 0x14, 0x2A, 0x8C, 0xC6, 0xA2,
      0xAB, 0xD1, 0x1C, 0xF4, 0x05, 0xF0, 0x63, 0x68, 0x35, 0x35, 0xBF, 0xB9,
      0xB8, 0xA4, 0x95, 0x40, 0x4B, 0x98, 0xE6, 0xEF, 0x53, 0xE3, 0x70, 0x65,
      0x44, 0x0F, 0xE8, 0x68, 0xFB, 0x78, 0xE8, 0x8E, 0x5E, 0x89, 0x98, 0x61,
      0x5F, 0x78, 0x8E, 0x1C, 0xB6, 0xA3, 0xAA, 0x37, 0x07, 0x08, 0x4F, 0x17,
      0xC3, 0x14, 0xC4, 0x03, 0x76, 0x64, 0x4E, 0x2D, 0xF4, 0x61, 0x20, 0x31,
      0x03, 0xDE, 0x0E, 0x65, 0x20, 0x38, 0xF9, 0x09, 0x23, 0xF4, 0x2C, 0x31,
      0xC4, 0xA0, 0x16, 0x9E, 0xF5, 0xD3, 0xA0, 0x2F, 0xC5, 0x75, 0xCE, 0x3F,
      0x73, 0x1D, 0x7B, 0x1A, 0x9D, 0xB4, 0x44, 0x38, 0x67, 0x08, 0xCA, 0x4A,
      0x1A, 0xC2, 0x39, 0x08, 0x1A, 0x3F, 0x66, 0xE2, 0x69, 0xA9, 0x07, 0x0B,
      0x7A, 0x3A, 0xEB, 0x7B, 0x76, 0xA0, 0xFD, 0x76, 0xC2, 0x6B, 0x1F, 0xD9,
      0x18, 0x23, 0xF8, 0x4D, 0x7C, 0x9F, 0x89, 0x05, 0xE4, 0x72, 0x5B, 0x52,
      0x88, 0x96, 0x22, 0x1C, 0xE7, 0xDB, 0x23, 0x53, 0x0C, 0x62, 0x34, 0x3F,
      0x5E, 0xA3, 0xDB, 0x1F, 0x90, 0xEB, 0x14, 0x52, 0xD5, 0x4E, 0xE1, 0xAB,
      0x1E, 0x10, 0x9E, 0xF9, 0x62, 0xF0, 0x41, 0x9C, 0xA0, 0x11, 0x1F, 0x95,
      0xC1, 0xA8, 0x03, 0x4F, 0x96, 0x9F, 0xBD, 0xC1, 0x07, 0xF5, 0xEF, 0xAD,
      0x0C, 0x2B, 0xD5, 0xBC, 0x17, 0x4B, 0x29, 0xD6, 0x59, 0x75, 0xFA, 0xD9,
      0x8A, 0x61, 0x27, 0x28, 0x42, 0x27, 0x35, 0x6A, 0x42, 0x86, 0x6C, 0xAF,
      0x9A, 0xFA, 0x81, 0x63, 0x41, 0xC8, 0xE7, 0xC4, 0xC8, 0xE4, 0x86, 0x2B,
      0x71, 0x40, 0xB6, 0x5F, 0xDA, 0x45, 0x7E, 0x9B, 0x1E, 0xC1, 0x3F, 0x78,
      0xEE, 0xB0, 0x51, 0xD3, 0x71, 0x21, 0x31, 0x34, 0x18, 0x69, 0x69, 0x32,
      0x7D, 0x0E, 0x45, 0x16, 0x36, 0x38, 0xE9, 0x39, 0xE6, 0xA4, 0x9C, 0x10,
      0xB0, 0xE0, 0xAC, 0xC5, 0xBA, 0xC9, 0x04, 0xE2, 0x4F, 0x22, 0xA1, 0xF2,
      0xB1, 0x29, 0x8F, 0x4F, 0x59, 0xBC, 0xD0, 0x85, 0xEB, 0x08, 0x30, 0x2C,
      0x55, 0xDE, 0xCD, 0x53, 0xFA, 0x84, 0xE1, 0x28, 0x64, 0x7D, 0x3B, 0xA8,
      0x4D, 0x06, 0xED, 0x09, 0x2A, 0x0A, 0x41, 0x60, 0x85, 0xBC, 0xB2, 0x2F,
      0x21, 0xA5, 0x16, 0xBA, 0xC0, 0x1F, 0x49, 0xC5, 0x44, 0x88, 0xBF, 0x91,
      0xE2, 0x49, 0xC0, 0x0A, 0x65, 0x5E, 0xAB, 0xF6, 0x52, 0x5F, 0x0A, 0xB9,
      0x9B, 0xBF, 0x15, 0xDB, 0x35, 0x57, 0x82, 0xAB, 0xB7, 0x62, 0x6A, 0xD6,
      0x0B, 0x13, 0xED, 0xB2, 0xE4, 0x13, 0x71, 0x5C, 0x12, 0x9E, 0xDF, 0x21,
      0x54, 0x30, 0x59, 0xA3, 0xA5, 0x64, 0x6F, 0x9A, 0x55, 0xE9, 0x25, 0x70,
      0xA6, 0x83, 0xE6, 0xEF, 0x5C, 0x7E, 0x09, 0x7C, 0x4D, 0xFA, 0x5B, 0x4A,
      0x72, 0x5C, 0x82, 0xC0, 0xF6, 0x0E, 0x94, 0x14, 0x9A, 0x5D, 0x1E, 0x6E,
      0x14, 0x05, 0xDF, 0x20, 0x4B, 0xE7, 0x49, 0xA9, 0xA4, 0x03, 0xD0, 0xA2,
      0xEE, 0x26, 0xB7, 0x11, 0x3B, 0x24, 0x39, 0x90, 0x41, 0x90, 0xBE, 0xAA,
      0xCE, 0x77, 0xF2, 0x5E, 0x3F, 0x62, 0xC1, 0x22, 0xF7, 0x92, 0x5F, 0xD8,
      0x91, 0x3A, 0x33, 0x31, 0xAD, 0x43, 0x13, 0xB6, 0xC3, 0x47, 0x90, 0xE5,
      0x94, 0x09, 0x90, 0x76, 0x15, 0x19, 0xEA, 0x6B, 0x17, 0x39, 0x4F, 0x51,
      0xFA, 0x7C, 0xE8, 0x9C, 0xB7, 0x7A, 0xD9, 0xA1, 0x5A, 0x18, 0x42, 0x61,
      0xB4, 0xE5, 0xF2, 0xF9, 0x58, 0x71, 0xE9, 0xCA, 0x59, 0xEB, 0x99, 0x6B,
      0x93, 0x51, 0x83, 0x12, 0xEA, 0x1F, 0xA3, 0xB6, 0xDE, 0x79, 0x90, 0x36,
      0x3F, 0xE6, 0x52, 0x59, 0xA7, 0xDE, 0x67, 0x25, 0x6E, 0x3E, 0x8D, 0xC8,
      0x57, 0xE3, 0x50, 0x82, 0x6E, 0x95, 0xC1, 0x95, 0x36, 0x89, 0xFB, 0x6A,
      0x5B, 0xAD, 0x82, 0x6A, 0x1E, 0x9C, 0x6A, 0xDE, 0xD1, 0xAE, 0xEE, 0x12,
      0x49, 0x1A, 0x97, 0x8D, 0xE2, 0xEC, 0x35, 0xFD, 0x1D, 0xC7, 0x6B, 0x48,
      0x32, 0x2B, 0x87, 0x72, 0xA7, 0xB7, 0xE5, 0x5E, 0x8B, 0xFC, 0x5A, 0x8B,
      0xD0, 0x40, 0x94, 0x11, 0xF0, 0xFF, 0x3B, 0xE6, 0x93, 0x54, 0x9F, 0x04,
      0xB3, 0x46, 0x3C, 0xD0, 0x46, 0x69, 0xDF, 0x95, 0xCD, 0xE0, 0x4D, 0x7E,
      0xB6, 0xBD, 0x13, 0xDA, 0xBB, 0x39, 0xAA, 0x39, 0x16, 0xA5, 0x03, 0xFD,
      0x15, 0x70, 0xD5, 0xC5, 0xD4, 0xDF, 0xE2, 0x94, 0xA9, 0xE8, 0x2B, 0xE5,
      0x75, 0xB2, 0x7F, 0x9A, 0xB1, 0xEB, 0x21, 0xBA, 0xE0, 0x64, 0x07, 0xE2,
      0x69, 0x8C, 0xC4, 0xDD, 0x53, 0x3C, 0xBB, 0x78, 0xC9, 0x2E, 0x22, 0x7E,
      0x9E, 0x81, 0x36, 0x10, 0x49, 0xB9, 0x2E, 0xC1, 0x99, 0x9D, 0xC3, 0x0B,
      0xC9, 0xEC, 0x30, 0xE9, 0xBB, 0xB5, 0x72, 0x55, 0xBE, 0xE7, 0x50, 0xF8,
      0x84, 0x44, 0x9E, 0xFD, 0xFD, 0x28, 0x91, 0xD6, 0x97, 0x06, 0xC3, 0xCE,
      0x2A, 0x19, 0xD4, 0xB6, 0x4D, 0x26, 0xFC, 0xCD, 0xA9, 0x24, 0x54, 0x4D,
      0x69, 0xBB, 0x87, 0xD5, 0x0D, 0x17, 0xF6, 0x40, 0xD5, 0xFC, 0x5F, 0xDE,
      0xAF, 0xC6, 0xE4, 0xD6, 0x1F, 0x29, 0x11, 0xAC, 0x63, 0x17, 0xD1, 0x8A,
      0xD1, 0x2C, 0xCA, 0x6D, 0x6A, 0x61, 0x39, 0xEE, 0x34, 0x20, 0xE9, 0xE0,
      0x82, 0x52, 0x7F, 0xEF, 0x6D, 0x11, 0x1A, 0x88, 0x58, 0x41, 0x22, 0x72,
      0x07, 0xBB, 0x79, 0x33, 0x1A, 0xC0, 0xA2, 0x0A, 0xE1, 0x31, 0x7D, 0x09,
      0xC7, 0x53, 0x7D, 0xFF, 0xD1, 0x82, 0xDB, 0xD4, 0xB2, 0x5E, 0xE4, 0x83,
      0x16, 0x6F, 0x6C, 0xE0, 0x3A, 0xC5, 0xE1, 0x95, 0x60, 0xF2, 0x82, 0x16,
      0x96, 0x61, 0xF9, 0xF8, 0x61, 0xDD, 0x59, 0x33, 0x59, 0x7E, 0xAA, 0x06,
      0xDD, 0x34, 0xB2, 0x49, 0x24, 0x02, 0x1D, 0x1D, 0x35, 0xCA, 0x30, 0x64,
      0x01, 0x47, 0xFA, 0x2A, 0xAF, 0x1B, 0xA7, 0x82, 0x65, 0x0D, 0xDE, 0xCE,
      0x73, 0xA7, 0x28, 0x7F, 0xCB, 0x16, 0xE6, 0x44, 0x4E, 0xD5, 0x31, 0x31,
      0xB9, 0xD9, 0x16, 0x9B, 0xF6, 0xA9, 0xE0, 0x97, 0x76, 0xD2, 0x2C, 0x05,
      0xEF, 0xC7, 0x40, 0xA7, 0x6F, 0x7F, 0x2F, 0xBB, 0xDE, 0x0B, 0xD4, 0xAC,
      0x63, 0xF5, 0x0B, 0x3F, 0xA5, 0xC7, 0x18, 0x28, 0x55, 0x67, 0xE8, 0xEC,
      0xAD, 0x96, 0xDA, 0xF8, 0x9F, 0x48, 0xFE, 0x1F, 0x6C, 0x4B, 0x2D, 0x34,
      0xE0, 0x19, 0x14, 0x66, 0x9C, 0x13, 0x46, 0xF0, 0xAE, 0x30, 0xF5, 0x49,
      0xED, 0xE7, 0x37, 0x5C, 0x73, 0x12, 0x03, 0xEE, 0x4A, 0xA2, 0x90, 0x47,
      0xBC, 0x31, 0xD9, 0x8F, 0x7C, 0x2E, 0x47, 0x2D, 0xA8, 0x22, 0x4D, 0xFD,
      0x0D, 0x05, 0xED, 0xFA, 0xD5, 0x89, 0xD9, 0xAB, 0x17, 0x81, 0x15, 0x59,
      0xC8, 0xC3, 0x38, 0x64, 0x19, 0x33, 0xC7, 0x78, 0xFE, 0x57, 0x22, 0x02,
      0xBC, 0x41, 0x14, 0xED, 0xAE, 0x55, 0x2F, 0xD9, 0xA9, 0xA4, 0x4E, 0xAF,
      0x8B, 0xC6, 0x6B, 0xC6, 0x43, 0xA6, 0xCC, 0xA7, 0x08, 0xCB, 0x9F, 0xFB,
      0x8E, 0xFC, 0xD6, 0xCB, 0x5C, 0x87, 0xF1, 0x6F, 0x82, 0x1D, 0x88, 0x95,
      0x9B, 0x40, 0xAA, 0xEE, 0xD4, 0xED, 0x6A, 0x6D, 0x1F, 0x0E, 0xB7, 0xF5,
      0x0D, 0x90, 0xE1, 0xD4, 0x0D, 0xC3, 0x23, 0xC3, 0x8E, 0x66, 0x67, 0x19,
      0x5F, 0xAC, 0xD3, 0x59, 0x07, 0x45, 0xD7, 0xA2, 0x89, 0x9E, 0x55, 0x81,
      0x32, 0x20, 0x10, 0x32, 0xC5, 0x78, 0x2B, 0x34, 0x3E, 0xC1, 0x62, 0xC7,
      0x10, 0x66, 0xD8, 0xB7, 0x52, 0xE5, 0xDF, 0xA9, 0x11, 0x4B, 0x55, 0xAA,
      0xF7, 0x0A, 0x8C, 0x29, 0x17, 0x9C, 0x88, 0x52, 0xA6, 0x03, 0x08, 0xC7,
      0x32, 0x6E, 0xF0, 0xB4, 0xFD, 0xE2, 0x0C, 0xAB, 0xFF, 0xBC, 0x7D, 0x22,
      0xAF, 0xC9, 0x40, 0x09, 0x94, 0x66, 0x93, 0x12, 0xEB, 0x78, 0xF0, 0xFD,
      0x45, 0x84, 0x3A, 0x6B, 0xA5, 0xA0, 0xDA, 0xBD, 0x7B, 0x3A, 0xB5, 0x85,
      0xB7, 0x33, 0x8A, 0x84, 0x8C, 0x85, 0xE8, 0x0B, 0x53, 0x6E, 0xE9, 0x59,
      0x9B, 0xF7, 0xBA, 0xDE, 0x58, 0x67, 0xA2, 0x8B, 0x12, 0xB4, 0xEF, 0xE4,
      0x64, 0x63, 0x98, 0x97, 0xE6, 0xB6, 0xFC, 0x79, 0xF4, 0x2F, 0x78, 0x5B,
      0xB4, 0xA7, 0xDC, 0xFC, 0x0C, 0xA5, 0x46, 0xD8, 0x8C, 0xD8, 0xF9, 0x8B,
      0x93, 0x0F, 0x07, 0x20, 0x62, 0x99, 0x4F, 0xBD, 0xE9, 0x05, 0x10, 0xE6,
      0xD6, 0x1A, 0x7C, 0xE1, 0xBA, 0x63, 0xA6, 0x6C, 0x46, 0x45, 0xCF, 0x99,
      0x5F, 0x2C, 0x77, 0x77, 0xE4, 0x1D, 0x28, 0x76, 0x01, 0x0F, 0xD7, 0xCF,
      0x88, 0x7E, 0x9D, 0xB6, 0x02, 0xD9, 0x6E, 0x86, 0x4B, 0x5B, 0x9F, 0x64,
      0x10, 0x2E, 0x00, 0xF1, 0x98, 0x68, 0x77, 0x9A, 0x78, 0x0D, 0x51, 0x6B,
      0x69, 0x45, 0xF5, 0x82, 0x8E, 0x98, 0x07, 0xD4, 0x15, 0xC7, 0x68, 0x82,
      0x2A, 0x3B, 0x9A, 0x36, 0x3C, 0xFF, 0x26, 0x3E, 0x04, 0xBA, 0x72, 0xFF,
      0x41, 0xD5, 0x08, 0xB7, 0x59, 0x24, 0xD1, 0x7C, 0xCA, 0x35, 0x95, 0x5E,
      0xD1, 0x02, 0xEC, 0xCF, 0x93, 0x4F, 0xD2, 0x20, 0x86, 0x56, 0x75, 0x6B,
      0x8D, 0xDA, 0x6F, 0xFD, 0x5A, 0x2B, 0x18, 0x89, 0x07, 0x2E, 0x13, 0x14,
      0x1B, 0x9D, 0xBC, 0xF8, 0xB8, 0xA6, 0x9E, 0x8D, 0xAC, 0x83, 0x40, 0x3E,
      0x7D, 0x42, 0x45, 0xA3, 0x11, 0x8E, 0xC5, 0x1B, 0xEE, 0x51, 0x96, 0x69,
      0xB8, 0x49, 0x21, 0xE9, 0xEB, 0x9B, 0x85, 0x45, 0x0A, 0x45, 0x04, 0xAA,
      0x47, 0x63, 0xB5, 0xC1, 0xFF, 0x07, 0x83, 0x55, 0x9B, 0xF7, 0xD0, 0x4A,
      0x38, 0x32, 0xD5, 0xC1, 0xE6, 0x90, 0xC4, 0xA2, 0x7B, 0x6F, 0x0E, 0x6D,
      0xCB, 0xB4, 0xDA, 0x80, 0x24, 0x9C, 0xA9, 0x82, 0x5F, 0xAA, 0x25, 0x56,
      0x32, 0x67, 0xFA, 0x02, 0x37, 0x24, 0x4B, 0x2D, 0xD0, 0xAE, 0x1A, 0x8F,
      0x10, 0x10, 0xAB, 0xDF, 0x3D, 0x05, 0x27, 0x07, 0xDF, 0x8D, 0xB9, 0xBD,
      0x3B, 0x20, 0x6B, 0x61, 0xD2, 0xB2, 0x62, 0xB7, 0xBB, 0x0C, 0x0F, 0x43,
      0x72, 0x15, 0x5F, 0x6E, 0x27, 0xF1, 0xE0, 0x36, 0x84, 0x9D, 0xA9, 0x2A,
      0x79, 0x22, 0x49, 0xE2, 0x9D, 0x87, 0x6E, 0x0C, 0x97, 0x63, 0xA2, 0x44,
      0xCF, 0x58, 0x39, 0x50, 0xD6, 0xCD, 0xEC, 0x34, 0x8D, 0xA4, 0x83, 0x60,
      0xF9, 0xB7, 0xCE, 0xEE, 0xC0, 0x46, 0xBD, 0xFC, 0x7E, 0xE4, 0x01, 0xD7,
      0x08, 0x59, 0x07, 0x13, 0xE6, 0xF1, 0x45, 0xC9, 0x06, 0x10, 0xEB, 0xD1,
      0xBE, 0xC8, 0xFB, 0x95, 0x18, 0xC3, 0x9A, 0x2F, 0xA9, 0xEF, 0xED, 0xBC,
      0x61, 0xDE, 0xB4, 0xD0, 0xB6, 0x11, 0x01, 0xE3, 0x73, 0x93, 0x0F, 0x2D,
      0x57, 0x76, 0xBE, 0xE5, 0xE1, 0xAE, 0x6D, 0xFB, 0x2D, 0x8E, 0xAF, 0x3D,
      0xD0, 0x71, 0x75, 0x3B, 0xB7, 0x00, 0xE0, 0xC8, 0xCD, 0x99, 0x79, 0x84,
      0x90, 0x54, 0x17, 0xAA, 0x2B, 0xF8, 0x05, 0xC1, 0xB9, 0x44, 0xFA, 0x67,
      0xBC, 0x0C, 0xB9, 0x61, 0xD8, 0x48, 0xAD, 0x3B, 0x07, 0x25, 0xC2, 0x69,
      0xCA, 0xE7, 0x7B, 0x61, 0xF0, 0x1D, 0xB3, 0xBB, 0x16, 0x0F, 0x8F, 0xA2,
      0x8B, 0x04, 0x61, 0x36, 0xBE, 0xAC, 0xAC, 0x81, 0x17, 0x4B, 0xAA, 0x18,
      0xB7, 0xE4, 0xDE, 0x4E, 0x21, 0x08, 0x32, 0x23, 0x2A, 0xDA, 0x0B, 0xCE,
      0x80, 0x05, 0xB4, 0x50, 0xC2, 0x49, 0xE0, 0x5B, 0x79, 0x6E, 0x31, 0x2B,
      0x69, 0x1F, 0xF1, 0x36, 0x5E, 0x9A, 0x39, 0x76, 0x05, 0xDF, 0xFA, 0x63,
      0xA2, 0xF6, 0x12, 0xCB, 0x46, 0xE9, 0xF2, 0xA1, 0xB5, 0xC9, 0xE0, 0xBC,
      0xF1, 0x23, 0x5D, 0x1C, 0x55, 0x1F, 0xE5, 0x68, 0xB1, 0xF2, 0x63, 0x4F,
      0x90, 0x15, 0x92, 0xBD, 0x54, 0x7B, 0x8C, 0x60, 0x68, 0x32, 0x4E, 0xF6,
      0x60, 0xEB, 0xFF, 0xB6, 0xCE, 0xBB, 0x74, 0xAD, 0x03, 0x3A, 0x61, 0x42,
      0x59, 0x14, 0x6E, 0x30, 0xE0, 0x2B, 0x45, 0x6F, 0xFA, 0x03, 0x8C, 0x2A,
      0x0A, 0x28, 0xFF, 0xE5, 0xF8, 0x82, 0xBF, 0xFC, 0xA4, 0xE7, 0xE3, 0x6B,
      0x07, 0xA6, 0xCB, 0x9B, 0xED, 0x8F, 0xDF, 0x77, 0x5D, 0x4B, 0xCD, 0xE8,
      0x54, 0xDE, 0xBA, 0xA1, 0x8B, 0x21, 0x9F, 0x15, 0x8E, 0xD2, 0x49, 0x10,
      0x28, 0xD2, 0xA4, 0x18, 0xA1, 0x90, 0xA8, 0xDB, 0x4F, 0x07, 0x90, 0x04,
      0x67, 0xF9, 0xB6, 0x12, 0xB4, 0xEF, 0x96, 0x42, 0xAA, 0x1F, 0x7C, 0x09,
      0xE0, 0x8E, 0xE8, 0xB1, 0x24, 0x40, 0x1B, 0xAF, 0xB5, 0xBA, 0xCA, 0xC2,
      0x25, 0x04, 0x74, 0x83, 0xE5, 0x36, 0x2A, 0x99, 0xF3, 0x9C, 0xBB, 0x22,
      0xD2, 0xDB, 0x63, 0xE3, 0x88, 0x38, 0x12, 0xB8, 0x42, 0x53, 0x60, 0xF7,
      0x9B, 0x1C, 0x35, 0x18, 0xC1, 0xD1, 0x0C, 0x3E, 0x1D, 0x64, 0xDC, 0x1F,
      0xC2, 0xC5, 0x1F, 0xC2, 0x0E, 0x34, 0x6E, 0xCA, 0x82, 0x94, 0xE3, 0x7C,
      0x91, 0x16, 0x87, 0x14, 0x85, 0xDB, 0x47, 0x39, 0x5F, 0x8B, 0xC0, 0xD7,
      0x08, 0xEB, 0x2A, 0x52, 0xEE, 0x68, 0xBB, 0x2A, 0xCE, 0x0C, 0xB0, 0x70,
      0xAB, 0x99, 0x38, 0xC3, 0x3F, 0xB3, 0x41, 0xB8, 0x14, 0x80, 0x41, 0x8F,
      0x8D, 0x39, 0xCF, 0x0E, 0x6F, 0x92, 0x3E, 0xB6, 0x52, 0x06, 0xF8, 0x0D,
      0x3A, 0xE5, 0x39, 0x13, 0x92, 0x78, 0x2F, 0x14, 0x8E, 0xDB, 0x6E, 0xE3,
      0x93, 0x94, 0x31, 0xDF, 0x8F, 0xEE, 0x91, 0xDE, 0x62, 0x5F, 0xD0, 0xAD,
      0x89, 0x53, 0xD3, 0xA5, 0x0D, 0x33, 0x81, 0x79, 0xB4, 0xD1, 0xF7, 0xF4,
      0x2A, 0xBA, 0x9F, 0x51, 0x89, 0xC7, 0xDD, 0xE4, 0x72, 0xB1, 0xF8, 0x9B,
      0xFA, 0x60, 0x58, 0x6B, 0x75, 0x0B, 0x22, 0x17, 0x58, 0x0F, 0xA5, 0x18,
      0x54, 0x9E, 0x50, 0x2A, 0x30, 0x18, 0x1A, 0xAB, 0xA3, 0x21, 0x7C, 0x1A,
      0x5D, 0xC9, 0x6D, 0x2A, 0x86, 0x2D, 0xD8, 0xC9, 0xD9, 0x1F, 0x5A, 0x55,
      0x63, 0x67, 0xC2, 0xD4, 0x44, 0x9D, 0x07, 0x60, 0x60, 0x69, 0x21, 0x1E,
      0xE6, 0x5A, 0xBE, 0x35, 0xCD, 0x87, 0x02, 0x5F, 0x89, 0xBF, 0x53, 0xF0,
      0x64, 0xFD, 0xDE, 0x4C, 0xB5, 0xB4, 0xC1, 0xFB, 0x6B, 0xAF, 0x6D, 0xF3,
      0x14, 0x97, 0x84, 0xB5, 0x43, 0x82, 0xF1, 0x00, 0x18, 0x8E, 0x62, 0xEF,
      0x5B, 0x56, 0x60, 0x04, 0x12, 0x54, 0x1F, 0x21, 0x01, 0x94, 0x35, 0x65,
      0x7E, 0x53, 0x26, 0x6F, 0x86, 0x5D, 0x86, 0x23, 0xE2, 0x18, 0x29, 0x45,
      0xBF, 0x30, 0xD1, 0x33, 0xB2, 0xF6, 0x95, 0xD1, 0xB7, 0x5D, 0xE1, 0x1A,
      0x33, 0x28, 0xFF, 0xEE, 0xFF, 0x04, 0x1F, 0x4B, 0x5C, 0x3E, 0x96, 0x42,
      0x04, 0x0E, 0x13, 0xA8, 0x25, 0xF1, 0x17, 0x67, 0x6A, 0xD5, 0xD4, 0x0B,
      0x82, 0x7A, 0xAE, 0x31, 0x56, 0x4F, 0x91, 0x43, 0xE8, 0x58, 0x6F, 0x39,
      0x4B, 0xFF, 0xC0, 0x50, 0x73, 0x29, 0xDC, 0x6A, 0xE4, 0x96, 0xA6, 0xE0,
      0xE0, 0xC1, 0x66, 0x4E, 0x1C, 0xDF, 0x30, 0x7C, 0x39, 0x4D, 0x19, 0x51,
      0x5F, 0xC1, 0xD6, 0xB8, 0x05, 0xF5, 0xED, 0x3F, 0xE9, 0xFF, 0x20, 0xDA,
      0xD5, 0x70, 0x3C, 0x43, 0x04, 0xDE, 0xA9, 0x7E, 0xFD, 0xED, 0x61, 0x7C,
      0x20, 0xDF, 0xB8, 0x2B, 0xE3, 0x00, 0x22, 0x92, 0x27, 0xEF, 0xC3, 0x36,
      0x45, 0xA4, 0x69, 0xE2, 0xEB, 0xF8, 0xB8, 0x0B, 0x4D, 0x15, 0x7B, 0xE5,
      0xCB, 0xF9, 0xAB, 0x9B, 0xFD, 0x27, 0x35, 0x06, 0x87, 0xBA, 0x5F, 0xF5,
      0xBD, 0x6B, 0xCA, 0x72, 0x84, 0x8E, 0x6E, 0x30, 0xC9, 0x5E, 0xA3, 0x55,
      0x2A, 0x7F, 0xBD, 0x46, 0x62, 0x35, 0x8B, 0x38, 0xC5, 0xEE, 0xCF, 0xD5,
      0x48, 0x8A, 0x1F, 0xB3, 0xE9, 0x8A, 0xFE, 0xA4, 0x79, 0xB9, 0xDD, 0xAC,
      0x0E, 0x91, 0xC1, 0x6F, 0xCE, 0x75, 0xFE, 0x37, 0xCA, 0xAF, 0x08, 0x9E,
      0x6F, 0x84, 0xB8, 0xCE, 0x63, 0x31, 0xE6, 0xB5, 0x78, 0x28, 0xA6, 0x7A,
      0x12, 0xB4, 0xD2, 0x61, 0x66, 0x55, 0xE5, 0x94, 0xC4, 0x61, 0x83, 0x4B,
      0x22, 0x8D, 0x38, 0xD3, 0x23, 0xC5, 0x09, 0xC2, 0x70, 0x1D, 0x03, 0x27,
      0xEB, 0xD8, 0xE9, 0x76, 0x0A, 0x3C, 0xC8, 0xFD, 0x98, 0x92, 0xF1, 0x0C,
      0x84, 0xBF, 0xDF, 0x00, 0xA1, 0xF9, 0x2D, 0xB6, 0x97, 0x6F, 0x23, 0x6B,
      0xA7, 0xF4, 0x30, 0xCE, 0xB0, 0xC1, 0x3C, 0x28, 0xBF, 0x5D, 0x1B, 0x7C,
      0x92, 0xE1, 0x66, 0x37, 0xB6, 0x99, 0x76, 0x0A, 0x04, 0xC8, 0x87, 0x15,
      0x21, 0x71, 0x93, 0x78, 0x29, 0x2E, 0xDF, 0xF0, 0x85, 0x90, 0x0F, 0x36,
      0x4D, 0x1D, 0x2F, 0xD1, 0x4B, 0xBA, 0x8F, 0xD4, 0x37, 0x56, 0x4B, 0x86,
      0x30, 0x09, 0x90, 0x0F, 0x4C, 0x80, 0xCB, 0x86, 0x97, 0xDB, 0x7D, 0xD8,
      0x59, 0x41, 0xE8, 0xF9, 0xF4, 0x59, 0xFA, 0x6C, 0x2D, 0xA2, 0x6F, 0x2F,
      0x78, 0x3D, 0xA5, 0x19, 0x86, 0xB3, 0x44, 0xC5, 0xB8, 0xE8, 0xA8, 0xB9,
      0x93, 0xF4, 0x0C, 0x7E, 0xA2, 0x95, 0x04, 0x65, 0x35, 0xB9, 0xAC, 0x02,
      0xFC, 0x56, 0x76, 0x9D, 0xC1, 0x41, 0xE0, 0xED, 0x3A, 0x6F, 0xC5, 0xC9,
      0xAC, 0xE3, 0x86, 0x11, 0x04, 0x70, 0x42, 0xAC, 0xD4, 0x3A, 0xE4, 0x0C,
      0xC1, 0x9C, 0xF1, 0x10, 0x34, 0x90, 0xB0, 0x10, 0x7B, 0x34, 0x97, 0x47,
      0x1B, 0x2F, 0xBA, 0x1F, 0x07, 0x8B, 0x77, 0x93, 0xC6, 0xED, 0xC9, 0x20,
      0xDA, 0x77, 0x73, 0x55, 0x25, 0x1C, 0x7C, 0xC3, 0x5A, 0x01, 0x24, 0xAB,
      0x3C, 0x86, 0x37, 0x1D, 0x7A, 0xB1, 0x1C, 0x8B, 0xF0, 0x83, 0xBF, 0xB8,
      0x79, 0xEE, 0x9C, 0xF0, 0x2F, 0xE5, 0x57, 0x0D, 0xD0, 0x50, 0x39, 0x83,
      0xE2, 0x4C, 0x82, 0xEB, 0x21, 0x1E, 0x47, 0xBB, 0xB0, 0x46, 0x5B, 0x4B,
      0xD4, 0x7E, 0x2E, 0xCA, 0x81, 0x4B, 0x68, 0x96, 0x0C, 0xCD, 0xAE, 0xCC,
      0x51, 0x72, 0xC8, 0x7F, 0x04, 0x88, 0xA0, 0x27, 0x64, 0x99, 0x59, 0x1F,
      0xEE, 0x17, 0x63, 0x98, 0x73, 0xF2, 0x11, 0x46, 0x19, 0x8D, 0x16, 0x4C,
      0x5C, 0x20, 0xDE, 0x69, 0xD1, 0x10, 0xE8, 0x76, 0x64, 0x40, 0x67, 0x45,
      0xF6, 0x79, 0xE6, 0x76, 0x26, 0xFD, 0xCF, 0x22, 0x02, 0x76, 0x1A, 0x2C,
      0x5C, 0x96, 0xEB, 0xE8, 0x30, 0x3F, 0x05, 0xA9, 0x74, 0x69, 0x73, 0x17,
      0x0A, 0x7C, 0xC1, 0x10, 0x5F, 0x9A, 0xE5, 0x0A, 0x53, 0x30, 0x79, 0x01,
      0xAC, 0x66, 0xCE, 0x27, 0xA6, 0x5C, 0xF9, 0xA6, 0xAC, 0x0D, 0x2F, 0x40,
      0xAA, 0x0D, 0x15, 0x8A, 0xFE, 0xAD, 0x73, 0x7E, 0x16, 0x11, 0x84, 0x79,
      0x95, 0xDE, 0xE8, 0x4B, 0xC7, 0x48, 0xA5, 0xED, 0x57, 0xEA, 0x2A, 0xC5,
      0x8B, 0x2F, 0xD3, 0xD2, 0xD9, 0x28, 0x66, 0x6E, 0xA0, 0x9D, 0x07, 0x0A,
      0x1E, 0x87, 0x95, 0x3F, 0xAF, 0xB3, 0xAB, 0x8D, 0x3D, 0x46, 0x8A, 0x15,
      0x3E, 0xF8, 0x7C, 0xBC, 0xA1, 0xCE, 0x7A, 0x6C, 0x27, 0x8A, 0xC8, 0x90,
      0x09, 0x8D, 0x2B, 0xE5, 0x2B, 0x3A, 0xF9, 0xF0, 0x21, 0x87, 0xEB, 0x63,
      0x88, 0x3D, 0x52, 0x64, 0x17, 0x1D, 0x48, 0x5F, 0xFB, 0x1E, 0xCB, 0x27,
      0x2F, 0x18, 0x38, 0x5C, 0x6B, 0x8E, 0x81, 0x78, 0x8F, 0x37, 0xB6, 0x34,
      0xE0, 0xEC, 0x96, 0x72, 0x8A, 0xBD, 0xCF, 0xC8, 0x76, 0x9A, 0x5F, 0x2D,
      0xEB, 0x8B, 0xB1, 0xFC, 0xA8, 0x44, 0xDA, 0xFA, 0xF6, 0xD6, 0x3A, 0xE1,
      0x1E, 0x8E, 0x9A, 0x1A, 0x1B, 0xD7, 0x0E, 0x14, 0xD0, 0x58, 0x3B, 0xB1,
      0x2B, 0x7E, 0x0C, 0x13, 0xC7, 0xA9, 0x88, 0x05, 0x4E, 0xED, 0xC5, 0xDD,
      0xDB, 0x57, 0xBC, 0x25, 0x5F, 0x6E, 0xA9, 0x0D, 0x66, 0x19, 0xB5, 0x42,
      0xFE, 0xDC, 0x56, 0x53, 0x07, 0x82, 0x32, 0x07, 0x04, 0x10, 0xAD, 0x75,
      0x14, 0xFB, 0x50, 0xE9, 0xE7, 0x4D, 0x51, 0xC6, 0xF8, 0x16, 0x2A, 0xD2,
      0x5A, 0xD6, 0x15, 0x97, 0x70, 0x76, 0x2F, 0x36, 0xF7, 0x45, 0x87, 0xE1,
      0xAB, 0x3E, 0x9F, 0x50, 0xF7, 0xC0, 0x4C, 0xE7, 0x5C, 0xBC, 0x78, 0xBA,
      0xCB, 0x26, 0xFC, 0xE2, 0x43, 0xE1, 0x3C, 0xE7, 0x4C, 0x28, 0x67, 0x11,
      0x57, 0x59, 0x96, 0x51, 0x01, 0x6B, 0xB0, 0x41, 0x45, 0x9D, 0xF4, 0xEA,
      0x0B, 0x6F, 0x6F, 0x35, 0xE3, 0x0E, 0xC1, 0xEA, 0x6F, 0xEA, 0x94, 0xBB,
      0x53, 0xD3, 0x47, 0xE2, 0x6C, 0xE7, 0xBE, 0x6E, 0x4C, 0x56, 0x9E, 0x60,
      0xF9, 0xA4, 0x28, 0xEB, 0x55, 0xC0, 0x2F, 0x65, 0x1A, 0x0B, 0x84, 0xE2,
      0xD6, 0x52, 0xA2, 0x42, 0xA5, 0x51, 0x2A, 0xCB, 0x29, 0x43, 0xF0, 0x29,
      0xBE, 0xE7, 0x79, 0x3A, 0xC3, 0x85, 0xBF, 0x7E, 0xA0, 0x77, 0xF2, 0x00,
      0x65, 0x4E, 0xDA, 0x0A, 0x23, 0x18, 0x2A, 0x1F, 0xAB, 0xB5, 0x00, 0x10,
      0x87, 0xAC, 0x7B, 0x45, 0xDF, 0xA7, 0x9F, 0xAB, 0x72, 0xBA, 0xD1, 0x34,
      0x90, 0x93, 0xEC, 0x40, 0x87, 0x0B, 0xC8, 0x8D, 0xA3, 0xCD, 0x66, 0x0E,
      0x6E, 0x61, 0x71, 0x1B, 0xFB, 0xCB, 0xCD, 0x50, 0x7A, 0x29, 0x47, 0xDD,
      0xC4, 0x11, 0x4F, 0xE2, 0x7E, 0x93, 0xA3, 0x46, 0x18, 0x25, 0x43, 0xF7,
      0xC5, 0x70, 0x6C, 0x5C, 0x8E, 0x58, 0x80, 0x4F, 0x42, 0x1A, 0xA7, 0x75,
      0xBC, 0x82, 0x4D, 0x9B, 0x27, 0xC7, 0x58, 0xA6, 0x90, 0x42, 0x69, 0x7F,
      0x04, 0x6A, 0x17, 0xC8, 0xB9, 0x97, 0xA8, 0xFD, 0xA0, 0x96, 0xA2, 0xA3,
      0x73, 0x19, 0x5D, 0xA3, 0xE5, 0xFF, 0x02, 0xFA, 0x1A, 0xFE, 0x51, 0x66,
      0xF8, 0xC2, 0x43, 0xFF, 0x65, 0x49, 0x7F, 0x67, 0x3D, 0x6F, 0x95, 0x97,
      0xBF, 0xD9, 0x11, 0x23, 0x4D, 0x7C, 0x81, 0xC3, 0xC5, 0xF1, 0x79, 0x8F,
      0xB4, 0xCA, 0xEA, 0x4F, 0x7D, 0x84, 0xE9, 0xF3, 0x10, 0x34, 0x3B, 0x51,
      0xA9, 0xB4, 0x4D, 0x93, 0xA2, 0xC0, 0xF9, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x06, 0x0E, 0x13, 0x18, 0x1E, 0x23};
  EXPECT_EQ(Bytes(encoded_signature), Bytes(expected_signature));
}
