/*---------------------------------------------------------------------\
|                          ____ _   __ __ ___                          |
|                         |__  / \ / / . \ . \                         |
|                           / / \ V /|  _/  _/                         |
|                          / /__ | | | | | |                           |
|                         /_____||_| |_| |_|                           |
|                                                                      |
-----------------------------------------------------------------------/
*
* This file contains private API, this might break at any time between releases.
* You have been warned!
*
*/

#ifndef ZYPP_CORE_ZYPPNG_RPC_MESSAGESTREAM_H_INCLUDED
#define ZYPP_CORE_ZYPPNG_RPC_MESSAGESTREAM_H_INCLUDED

#include <zypp-core/zyppng/base/Base>
#include <zypp-core/zyppng/base/Signals>
#include <zypp-core/zyppng/base/Timer>
#include <zypp-core/zyppng/io/IODevice>
#include <zypp-core/zyppng/pipelines/expected.h>
#include <zypp-core/zyppng/rpc/rpc.h>

#include <deque>
#include <optional>

namespace zypp::proto
{
  class Envelope;
}

namespace zyppng {

  class RpcMessageStream;

  class InvalidMessageReceivedException : public zypp::Exception
  {
  public:
    InvalidMessageReceivedException( const std::string &msg = {});
  };

  /*!
   * Implement this Base class for all types that should serialized into
   * a \ref RpcMessage
   */
  class RpcBaseType {
  public:
    RpcBaseType() = default;
    virtual ~RpcBaseType() = default;
    RpcBaseType(const RpcBaseType &) = default;
    RpcBaseType(RpcBaseType &&) = default;
    RpcBaseType &operator=(const RpcBaseType &) = default;
    RpcBaseType &operator=(RpcBaseType &&) = default;

    virtual const std::string &typeName() const = 0;
    virtual bool deserialize( const std::string &data ) = 0;
    virtual void serializeInto( std::string &str ) const = 0;
    virtual std::string serialize() const;
  };


  class RpcMessage {

  public:
    friend class RpcMessageStream;
    RpcMessage( );
    RpcMessage( zypp::proto::Envelope data );

    RpcMessage(const RpcMessage &) = default;
    RpcMessage(RpcMessage &&) = default;
    RpcMessage &operator=(const RpcMessage &) = default;
    RpcMessage &operator=(RpcMessage &&) = default;

    void set_messagetypename( std::string name );
    const std::string &messagetypename() const;

    void set_value( std::string name );
    const std::string &value() const;

    std::string serialize() const;

  public:
    zypp::RWCOW_pointer<zypp::proto::Envelope> _data;
  };

  namespace rpc {
    /*!
     * Helper function to get the type name of a given RPC message type.
     * Sadly Protobuf does not offer a static function to get the types FQN we
     * cache it after asking for it the first time. So we need a dummy object just once.
     */
    template <typename T>
    const std::string & messageTypeName() {
      static std::string name = T().GetTypeName();
      return name;
    }

    template <typename T>
    expected<void> deserializeMessageInto( const RpcMessage &message, T &target )
    {
      if ( !target.deserialize( message.value() ) ) {
        const std::string &msg = zypp::str::Str() << "Failed to parse " << message.messagetypename() << " message.";
        ERR << msg << std::endl ;
        return expected<void>::error( ZYPP_EXCPT_PTR ( InvalidMessageReceivedException(msg) ) );
      }
      return expected<void>::success();
    }

    template <typename T>
    expected<T> deserializeMessage( const RpcMessage &message )
    {
      T target;
      auto res = deserializeMessageInto (message, target);
      if ( !res )
        return expected<T>::error( res.error() );
      return expected<T>::success(std::move(target));
    }

    template <typename T>
    RpcMessage serializeIntoMessage( const T& data )
    {
      RpcMessage env;
      env.set_messagetypename( data.typeName() );
      env.set_value( data.serialize() );
      return env;
    }

  }

  /*!
   *
   * Implements the basic protocol for sending zypp RPC messages over a IODevice
   *
   * Communication Format:
   * ---------------------
   * Each message is serialized into a zypp.proto.Envelope and sent over the communication medium in binary
   * format. The binary format looks like:
   *
   * +--------------------------------+---------------------------------+
   * | msglen ( 32 bit unsigned int ) | binary zypp.proto.Envelope data |
   * +--------------------------------+---------------------------------+
   *
   * The header defines the size in bytes of the following data trailer. The header type is a 32 bit uint, endianess is defined by
   * the underlying CPU arch. The data portion is directly generated by libprotobuf via SerializeToZeroCopyStream() to generate
   * the binary representation of the message.
   *
   */
  class RpcMessageStream : public zyppng::Base
  {
    public:

      using Ptr = std::shared_ptr<RpcMessageStream>;

      /*!
       * Uses the given iostream to send and receive messages.
       * If the device is already open and readable tries to read messages right away.
       * So make sure to check if messages have already been received via \ref nextMessage
       */
      static Ptr create( IODevice::Ptr iostr ) {
        return Ptr( new RpcMessageStream( std::move(iostr) ) );
      }

      /*!
       * Returns the next message in the queue, wait for the \ref sigMessageReceived signal
       * to know when new messages have arrived.
       * If \a msgName is specified returns the next message in the queue that matches the msgName
       */
      std::optional<RpcMessage> nextMessage ( const std::string &msgName = "" );

      /*!
       * Waits until at least one message is in the queue and returns it. Will return a empty
       * optional if a error occurs.
       *
       * If \a msgName is set this will block until a message with the given message name arrives and returns it
       *
       * \note Make sure to check if there are more than one messages in the queue after this function returns
       */
      std::optional<RpcMessage> nextMessageWait ( const std::string &msgName = "" );

      /*!
       * Send out a RpcMessage to the other side, depending on the underlying device state
       * this will be buffered and send when the device is writeable again.
       */
      bool sendMessage ( const RpcMessage &env );

      /*!
       * Reads all messages from the underlying IO Device, this is usually called automatically
       * but when shutting down this can be used to process all remaining messages.
       */
      void readAllMessages ();

      /*!
       * Send a messagee to the server side, it will be enclosed in a Envelope
       * and immediately sent out.
       */
      template <typename T>
      std::enable_if_t< !std::is_same_v<T, RpcMessage>, bool> sendMessage ( const T &m ) {
        return sendMessage( rpc::serializeIntoMessage(m) );
      }

      /*!
       * Emitted when new messages have arrived. This will continuously be emitted
       * as long as messages are in the queue.
       */
      SignalProxy<void()> sigMessageReceived ();

      /*!
       * Signal is emitted every time there was data on the line that could not be parsed
       */
      SignalProxy<void()> sigInvalidMessageReceived ();

      template<class T>
      static expected< T > parseMessage ( const RpcMessage &m ) {
        return rpc::deserializeMessage<T>(m);
      }

      template<class T>
      static expected< void > parseMessageInto ( const RpcMessage &m, T &target ) {
        if ( !target.ParseFromString( m.value() ) ) {
          const std::string &msg = zypp::str::Str() << "Failed to parse " << m.messagetypename() << " message.";
          ERR << msg << std::endl ;
          return expected<void>::error( ZYPP_EXCPT_PTR ( InvalidMessageReceivedException(msg) ) );
        }
        return expected<void>::success( );
      }

    private:
      RpcMessageStream( IODevice::Ptr iostr );
      bool readNextMessage ();
      void timeout( const zyppng::Timer &);

      IODevice::Ptr _ioDev;
      Timer::Ptr _nextMessageTimer = Timer::create();
      zyppng::rpc::HeaderSizeType _pendingMessageSize = 0;
      std::deque<RpcMessage> _messages;
      Signal<void()> _sigNextMessage;
      Signal<void()> _sigInvalidMessageReceived;

  };
}

namespace zypp {
  template<> zypp::proto::Envelope* rwcowClone<zypp::proto::Envelope>( const zypp::proto::Envelope * rhs );
}

/*!
 * Helper macro to be added into the class declaration
 * for a \ref zyppng::RpcBase subclass
 */
#define ZYPP_RPCBASE \
  public: \
    static const std::string &staticTypeName(); \
    const std::string &typeName() const override; \
    bool deserialize(const std::string &data) override; \
    void serializeInto(std::string &str) const override; \
    std::string serialize( ) const override; \
  private: \

/*!
 * Helper macro to be added into the class cc file
 * for a \ref zyppng::RpcBase subclass. Generates the
 * default implementation for the virtual functions if
 * the impl is a protobuf type
 */
#define ZYPP_IMPL_RPCBASE(Class, ImplClass, implVar) \
  const std::string &Class::staticTypeName()  \
  {  \
    return rpc::messageTypeName<ImplClass>();  \
  }  \
  \
  const std::string &Class::typeName() const  \
  { \
    return staticTypeName(); \
  } \
   \
  bool Class::deserialize(const std::string &data) \
  { \
    return implVar->ParseFromString( data ); \
  } \
 \
  void Class::serializeInto(std::string &str) const \
  { \
    implVar->SerializeToString( &str ); \
  } \
  \
  std::string Class::serialize( ) const \
  { \
   return implVar->SerializeAsString( ); \
  }



#endif // ZYPP_CORE_ZYPPNG_RPC_MESSAGESTREAM_H_INCLUDED
