// Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License"). You may not
// use this file except in compliance with the License. A copy of the
// License is located at
//
// http://aws.amazon.com/apache2.0/
//
// or in the "license" file accompanying this file. This file is distributed
// on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
// either express or implied. See the License for the specific language governing
// permissions and limitations under the License.

// Usage:
// go run riputil-gen.go
package main

import (
	"bytes"
	"fmt"
	"go/format"
	"io/ioutil"
	"os"
	"path/filepath"
	"sort"
	"strings"

	"github.com/aws/amazon-ssm-agent/agent/jsonutil"
)

const (
	startMarker   = "//AUTOGEN_START"
	endMarker     = "//AUTOGEN_END"
	serviceName   = "s3"
	serviceStatus = "GA"
)

var desiredAccessibilityAttribute = []string{"PUBLICLY_ANNOUNCED"}
var avoidedAccessibilityAttribute = []string{"ISOLATED", "RETAIL", "TEST"}

var ripConfigJsonFilePath = filepath.Join("agent", "s3util", "rip_static_config.json")
var ripUtilFilePath = filepath.Join("agent", "s3util", "riputil.go")

// riputil-gen is a simple program that generates the riputil file,
// riputil file contains information about S3 endpoints
func main() {
	var ripConfig AutoGeneratedRipConfigSchema
	if err := jsonutil.UnmarshalFile(ripConfigJsonFilePath, &ripConfig); err != nil {
		fmt.Printf("Error in opening/parsing RIP config file %v\n", err)
		os.Exit(1)
	}
	var mapEntries []string
	for i, region := range ripConfig.Regions {
		if allStringsInSlice(desiredAccessibilityAttribute, region.AccessibilityAttributes) &&
			!anyStringInSlice(avoidedAccessibilityAttribute, region.AccessibilityAttributes) {
			for _, service := range ripConfig.Regions[i].Services {
				if service.Name == serviceName && service.Status == serviceStatus {
					mapEntries = append(mapEntries, "\""+region.RegionName+"\" :\t\""+service.Endpoint+"\",\n")
				}
			}
		}
	}
	sort.Strings(mapEntries)

	var newContent bytes.Buffer
	for _, mapEntry := range mapEntries {
		newContent.WriteString(mapEntry)
	}

	if err := replaceFileContents(newContent.String()); err != nil {
		fmt.Printf("Error in updating the riputil file %v\n", err)
	}
}

func replaceFileContents(newContent string) (err error) {
	if oldContent, err := ioutil.ReadFile(ripUtilFilePath); err == nil {
		parts := strings.Split(string(oldContent), startMarker)
		oldContentPart1 := parts[0]
		moreParts := strings.Split(string(parts[1]), endMarker)
		oldContentPart2 := moreParts[1]
		fileNewContent := oldContentPart1 + startMarker + "\n" + newContent + endMarker + oldContentPart2
		if newContentFormatted, err := format.Source([]byte(fileNewContent)); err == nil {
			err = ioutil.WriteFile(ripUtilFilePath, newContentFormatted, 0644)
		}
	}
	return
}

func anyStringInSlice(keyWords []string, searchList []string) bool {
	for _, listItem := range searchList {
		for _, keyword := range keyWords {
			if keyword == listItem {
				return true
			}
		}
	}
	return false
}

func allStringsInSlice(keyWords []string, searchList []string) bool {
	foundCount := 0
	for _, listItem := range searchList {
		for _, keyword := range keyWords {
			if keyword == listItem {
				foundCount++
				break
			}
		}
	}
	return len(keyWords) == foundCount
}

// Auto generated using https://mholt.github.io/json-to-go/
type AutoGeneratedRipConfigSchema struct {
	Regions []struct {
		AccessibilityAttributes []string `json:"accessibility_attributes"`
		RegionName              string   `json:"region_name"`
		Services                []struct {
			Name     string `json:"name"`
			Endpoint string `json:"endpoint"`
			Status   string `json:"status"`
		} `json:"services"`
	} `json:"regions"`
}
