/*
 *	string related utilities
 *
 *	Copyright (C) 2010 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2021 SUSE LCC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Olaf Kirch
 *		Marius Tomaschewski
 *		Clemens Famulla-Conrad
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stddef.h>
#include <limits.h>
#include <errno.h>
#include <ctype.h>
#include <assert.h>
#include <regex.h>

#include <sutils.h>
#include <logging.h>


#define NC_STRING_ARRAY_CHUNK	16
#define NC_VAR_ARRAY_CHUNK	16
#define NC_STRINGBUF_CHUNK	64


int
nc_string_dup(char **pp, const char *str)
{
	if(!pp)
		return -1;

	if(*pp)
		free(*pp);
	if(str) {
		*pp = strdup(str);
		assert(*pp != NULL);
		if( !*pp)
			return -1;
	} else {
		*pp = NULL;
	}
	return 0;
}

int
nc_string_set(char **pp, const char *buf, size_t len)
{
	if(!pp)
		return -1;

	if(*pp)
		free(*pp);
	if(buf && len) {
		*pp = calloc(1, len + 1);
		assert(*pp != NULL);
		if(!*pp)
			return -1;
		strncat(*pp, buf, len);
	} else {
		*pp = NULL;
	}
	return 0;
}

void
nc_string_free(char **pp)
{
	if(pp) {
		if(*pp)
			free(*pp);
		*pp = NULL;
	}
}

size_t
nc_string_len(const char *str)
{
	return str ? strlen(str) : 0;
}

int /* bool */
nc_string_empty(const char *str)
{
	return !str || !*str;
}

int /* bool */
nc_string_eq(const char *a, const char *b)
{
	if (a == NULL || b == NULL)
		return a == b;
	return strcmp(a, b) == 0;
}

char *
nc_string_strip_spaces(char *str)
{
	char *end;

	if(!str || !*str)
		return str;

	end = str + strlen(str);
	while(end-- > str) {
		switch(*end) {
			case ' ':
			case '\t':
			case '\n':
			case '\r':
				*end = '\0';
			break;
			default:
				end = str;
			break;
		}
	}

	while(*str == ' ' || *str == '\t')
		str++;

	return str;
}

int
nc_string_prefix_eq(const char *pfx, const char *str)
{
	if (pfx == NULL || str == NULL)
		return pfx == str;
	return !strncmp(str, pfx, strlen(pfx));
}

int
nc_string_suffix_eq(const char *sfx, const char *str)
{
	size_t sfx_len, str_len;

	if (sfx == NULL || str == NULL)
		return sfx == str;

	sfx_len = strlen(sfx);
	str_len = strlen(str);
	return (str_len >= sfx_len) && !strcmp(sfx, str + (str_len - sfx_len));
}

void
nc_stringbuf_init(nc_stringbuf_t *nsb)
{
	memset(nsb, 0, sizeof(*nsb));
}

void
nc_stringbuf_destroy(nc_stringbuf_t *nsb)
{
	nc_stringbuf_clear(nsb);
}

void
nc_stringbuf_clear(nc_stringbuf_t *nsb)
{
	if (nsb->string)
		free(nsb->string);
	nsb->string = NULL;
	nsb->len = 0;
}

int
nc_stringbuf_empty(const nc_stringbuf_t *nsb)
{
	return nsb->len == 0; /* bool */
}

inline static size_t
__nc_stringbuf_size(nc_stringbuf_t *nsb, size_t len)
{
	return ((nsb->len + len) | (NC_STRINGBUF_CHUNK - 1)) + 1;
}

static int
__nc_stringbuf_realloc(nc_stringbuf_t *nsb, size_t len)
{
	size_t newsize;
	char  *newdata;

	if (!nsb->string ||
	    ((nsb->len + len + 1) > __nc_stringbuf_size(nsb, 0))) {
		newsize = __nc_stringbuf_size(nsb, len);
		newdata = realloc(nsb->string, newsize);
		assert(newdata != NULL);
		if(!newdata)
			return -1;
		nsb->string = newdata;
		memset(nsb->string + nsb->len, 0, newsize - nsb->len);
	}
	return 0;
}

int
nc_stringbuf_grow(nc_stringbuf_t *nsb, size_t len)
{
	if (__nc_stringbuf_realloc(nsb, len) != 0)
		return -1;
	return 0;
	
}

static int
__nc_stringbuf_put(nc_stringbuf_t *nsb, const char *ptr, size_t len)
{
	if (__nc_stringbuf_realloc(nsb, len) != 0)
		return -1;
	memcpy(nsb->string + nsb->len, ptr, len);
	nsb->string[nsb->len + len] = '\0';
	nsb->len += len;
	return 0;
}

int
nc_stringbuf_puts_n(nc_stringbuf_t *nsb, const char *s, size_t n)
{
	if(s) {
		size_t len = strlen(s);
		return __nc_stringbuf_put(nsb, s, len > n ? n : len);
	}
	return -1;
}

int
nc_stringbuf_puts(nc_stringbuf_t *nsb, const char *s)
{
	if(!s)
		return -1;
	return __nc_stringbuf_put(nsb, s, strlen(s));
}

int
nc_stringbuf_putc(nc_stringbuf_t *nsb, char cc)
{
	return __nc_stringbuf_put(nsb, &cc, 1);
}

int
nc_stringbuf_printf(nc_stringbuf_t *nsb, const char *fmt, ...)
{
	va_list ap;
	int ret;

	va_start(ap, fmt);
	ret = nc_stringbuf_vprintf(nsb, fmt, ap);
	va_end(ap);

	return ret;
}

int
nc_stringbuf_vprintf(nc_stringbuf_t *nsb, const char *fmt, va_list ap)
{
	va_list cp;
	int size = (NC_STRINGBUF_CHUNK * 4) - 1;
	int n = 0;

	if(!nsb || !fmt)
		return -1;

	nsb->len = 0;
	if(nc_stringbuf_grow(nsb, size++) != 0) {
		nc_stringbuf_clear(nsb);
		return -1;
	}

	while(1) {
		va_copy(cp, ap);
 		n = vsnprintf(nsb->string, size, fmt, cp);
		va_end(cp);

		if(n > -1 && n < size) {
			nsb->len = n;
			break;
		}

		if (n > -1)
			size = n;
		else
			size *= 2;

		if(nc_stringbuf_grow(nsb, size++) != 0) {
			nc_stringbuf_clear(nsb);
			return -1;
		}
	}
	return nsb->len;
}

void
nc_stringbuf_move(nc_stringbuf_t *dst, nc_stringbuf_t *src)
{
	nc_stringbuf_clear(dst);
	*dst = *src;
	src->string = NULL;
	src->len = 0;
}

void
nc_stringbuf_trim_empty_lines(nc_stringbuf_t *nsb)
{
	char *str = nsb->string;
	unsigned int n, trim;

	/* trim tail */
	for (trim = n = nsb->len; n; --n) {
		char cc = str[n-1];

		if (cc == '\r' || cc == '\n')
			trim = n;
		else if (cc != ' ' && cc != '\t')
			break;
	}
	nsb->string[trim] = '\0';
	nsb->len = trim;

	/* trim head */
	for (trim = n = 0; n < nsb->len; ) {
		char cc = str[n++];

		if (cc == '\r' || cc == '\n')
			trim = n;
		else if (cc != ' ' && cc != '\t')
			break;
	}
	if(trim) {
		nsb->len -= trim;
		memmove(nsb->string, nsb->string + trim, nsb->len + 1);
	}
}

const char *
nc_stringbuf_join(nc_stringbuf_t *buf, const nc_string_array_t *nsa, const char *sep)
{
	unsigned int i;
	size_t len;

	if (!buf || !nsa)
		return NULL;

	len = buf->len;
	for (i = 0; i < nsa->count; ++i) {
		if (sep && buf->len > len)
			nc_stringbuf_puts(buf, sep);
		nc_stringbuf_puts(buf, nsa->data[i]);
	}
	return buf->string ? buf->string + len : NULL;
}

void
nc_string_array_init(nc_string_array_t *nsa)
{
	memset(nsa, 0, sizeof(*nsa));
}

void
nc_string_array_destroy(nc_string_array_t *nsa)
{
	while (nsa->count--)
		free(nsa->data[nsa->count]);
	free(nsa->data);
	memset(nsa, 0, sizeof(*nsa));
}

static int
__nc_string_array_realloc(nc_string_array_t *nsa, unsigned int newsize)
{
	char **newdata;
	unsigned int i;

	/* 1 more for a NULL element at the end */
	newsize = (newsize + NC_STRING_ARRAY_CHUNK) + 1;
	newdata = realloc(nsa->data, newsize * sizeof(char *));
	assert(newdata != NULL);
	if (!newdata)
		return -1;

	nsa->data = newdata;
	for (i = nsa->count; i < newsize; ++i)
		nsa->data[i] = NULL;

	return 0;
}

int
nc_string_array_copy(nc_string_array_t *dst, const nc_string_array_t *src)
{
	unsigned int i;

	nc_string_array_destroy(dst);
	for (i = 0; i < src->count; ++i) {
		if (nc_string_array_append(dst, src->data[i]) < 0)
			return -1;
	}
	return 0;
}

void
nc_string_array_move(nc_string_array_t *dst, nc_string_array_t *src)
{
	nc_string_array_destroy(dst);
	*dst = *src;
	memset(src, 0, sizeof(*src));
}

static int
__nc_string_array_append(nc_string_array_t *nsa, char *str)
{
	if ((nsa->count % NC_STRING_ARRAY_CHUNK) == 0)
		if(__nc_string_array_realloc(nsa, nsa->count) < 0)
			return -1;

	nsa->data[nsa->count++] = str;
	return 0;
}

int
nc_string_array_append(nc_string_array_t *nsa, const char *str)
{
	char *newstr;

	newstr = strdup(str);
	if (!newstr)
		return -1;

	if (__nc_string_array_append(nsa, newstr) < 0) {
		free(newstr);
		return -1;
	}
	return 0;
}

static int
__nc_string_array_insert(nc_string_array_t *nsa, unsigned int pos, char *str)
{
	if ((nsa->count % NC_STRING_ARRAY_CHUNK) == 0)
		if(__nc_string_array_realloc(nsa, nsa->count) < 0)
			return -1;
	
	if (pos >= nsa->count) {
		nsa->data[nsa->count++] = str;
	} else {
		memmove(&nsa->data[pos + 1], &nsa->data[pos],
			(nsa->count - pos) * sizeof(char *));
		nsa->data[pos] = str;
		nsa->count++;
	}
	return 0;
}

int
nc_string_array_insert(nc_string_array_t *nsa, unsigned int pos, const char *str)
{
	char *newstr;

	newstr = strdup(str);
	if (!newstr)
		return -1;

	if (__nc_string_array_insert(nsa, pos, newstr) < 0) {
		free(newstr);
		return -1;
	}
	return 0;
}

int
nc_string_array_remove_index(nc_string_array_t *nsa, unsigned int pos)
{
	if (pos >= nsa->count)
		return -1;
	
	free(nsa->data[pos]);
	/* Note: this also copies the NULL pointer following the last element */
	memmove(&nsa->data[pos], &nsa->data[pos + 1],
		(nsa->count - pos) * sizeof(char *));
	nsa->count--;

	/* Don't bother with shrinking the array. It's not worth the trouble */
	return 0;
}

int
nc_string_array_remove_match(nc_string_array_t *nsa, const char *str, unsigned int maxkill)
{
	unsigned int i, j, killed = 0;

	if (!maxkill)
		maxkill = nsa->count;

	for (i = j = 0; i < nsa->count; ++i) {
		if (killed < maxkill && !strcmp(nsa->data[i], str)) {
			free(nsa->data[i]);
			killed++;
		} else {
			nsa->data[j++] = nsa->data[i];
		}
	}

	/* assert(j + killed == nsa->count); */
	memset(&nsa->data[j], 0, killed * sizeof(char *));
	nsa->count = j;

	/* Don't bother with shrinking the array. It's not worth the trouble */
	return killed;
}

int
nc_string_array_set(nc_string_array_t *nsa, unsigned int pos, const char *str)
{
	if (pos >= nsa->count)
		return -1;

	nc_string_dup(&nsa->data[pos], str);

	return nsa->data[pos] ? 0 : -1;
}

int
nc_string_array_get(nc_string_array_t *nsa, unsigned int pos, char **str)
{
	if (pos >= nsa->count)
		return -1;

	nc_string_dup(str, nsa->data[pos]);

	return *str ? 0 : -1;
}

int
nc_string_array_index(const nc_string_array_t *nsa, const char *str)
{
	unsigned int i;

	for (i = 0; i < nsa->count; ++i) {
		if (!strcmp(nsa->data[i], str))
			return i;
	}
	return -1;
}

int /* bool */
nc_string_array_is_uniq(const nc_string_array_t *nsa)
{
	unsigned int i, j;
	for (i = 0; i < nsa->count; ++i) {
		char *val_a = nsa->data[i];

		for (j = i + 1; j < nsa->count && val_a; ++j) {
			char *val_b = nsa->data[j];

			if (!strcmp(val_a, val_b))
				return 0;
		}
	}
	return 1;
}

void
nc_var_array_init(nc_var_array_t *nva)
{
	memset(nva, 0, sizeof(*nva));
}

void
nc_var_array_destroy(nc_var_array_t *nva)
{
	unsigned int i;
	for (i = 0; i < nva->count; ++i) {
		free(nva->data[i].name);
		free(nva->data[i].value);
	}
	free(nva->data);
	memset(nva, 0, sizeof(*nva));
}

static int
__nc_var_array_realloc(nc_var_array_t *nva, unsigned int newsize)
{
	nc_var_t *newdata;
	unsigned int i;

	newsize = (newsize + NC_VAR_ARRAY_CHUNK);
	newdata = realloc(nva->data, newsize * sizeof(nc_var_t));
	assert(newdata != NULL);
	if (!newdata)
		return -1;

	nva->data = newdata;
	for (i = nva->count; i < newsize; ++i) {
		nva->data[i].name = NULL;
		nva->data[i].value = NULL;
	}
	return 0;
}

nc_var_t *
nc_var_array_get(const nc_var_array_t *nva, const char *name)
{
	unsigned int i;
	nc_var_t *var;

	for (i = 0, var = nva->data; i < nva->count; ++i, ++var) {
		if (!strcmp(var->name, name))
			return var;
	}
	return NULL;
}

int
nc_var_array_copy(nc_var_array_t *dst, const nc_var_array_t *src)
{
	unsigned int i;
	nc_var_t *var;

	for (i = 0, var = src->data; i < src->count; ++i, ++var) {
		if( nc_var_array_set(dst, var->name, var->value) < 0)
			return -1;
	}
	return 0;
}

int
nc_var_array_set(nc_var_array_t *nva, const char *name, const char *value)
{
	nc_var_t *var;

	if ((var = nc_var_array_get(nva, name)) == NULL) {
		if ((nva->count % NC_VAR_ARRAY_CHUNK) == 0)
			if(__nc_var_array_realloc(nva, nva->count) < 0)
				return -1;

		var = &nva->data[nva->count];
		var->name = NULL;
		var->value = NULL;
		if(nc_string_dup(&var->name, name) < 0) {
			return -1;
		}
		if(nc_string_dup(&var->value, value) < 0) {
			nc_string_free(&var->name);
			return -1;
		}
		nva->count++;
	} else if(nc_string_dup(&var->value, value) < 0) {
		return -1;
	}
	return 0;
}

int
nc_var_array_set_uint(nc_var_array_t *nva, const char *name, unsigned int value)
{
	char buffer[32];
	snprintf(buffer, sizeof(buffer), "%u", value);
	return nc_var_array_set(nva, name, buffer);
}

int
nc_var_array_set_ulong(nc_var_array_t *nva, const char *name, unsigned long value)
{
	char buffer[32];
	snprintf(buffer, sizeof(buffer), "%lu", value);
	return nc_var_array_set(nva, name, buffer);
}

int
nc_var_array_set_double(nc_var_array_t *nva, const char *name, double value)
{
	char buffer[32];
	snprintf(buffer, sizeof(buffer), "%g", value);
	return nc_var_array_set(nva, name, buffer);
}


int
nc_parse_int(const char *input, int *result, int base)
{
	char *end = NULL;

	if (!input || !input[0])
		return -1;

	*result = strtol(input, (char **) &end, base);
	if (*end == '\0')
		return 0;

	return -1;
}

int
nc_parse_uint(const char *input, unsigned int *result, int base)
{
	char *end = NULL;

	if (!input || !input[0])
		return -1;

	*result = strtoul(input, (char **) &end, base);
	if (*end == '\0')
		return 0;

	return -1;
}

int
nc_parse_long(const char *input, long *result, int base)
{
	char *end = NULL;

	if (!input || !input[0])
		return -1;

	*result = strtol(input, (char **) &end, base);
	if (*end == '\0')
		return 0;

	return -1;
}

int
nc_parse_ulong(const char *input, unsigned long *result, int base)
{
	char *end = NULL;

	if (!input || !input[0])
		return -1;

	*result = strtoul(input, (char **) &end, base);
	if (*end == '\0')
		return 0;

	return -1;
}

int
nc_parse_int_mapped(const char *input, const struct nc_intmap *map, unsigned int *result)
{
	if (!input || !input[0])
		return -1;

	if (isdigit(input[0])) {
		if( nc_parse_uint(input, result, 0) == 0)
			return 0;
	}

	if (!map)
		return -1;
	for (; map->name; ++map) {
		if (!strcasecmp(map->name, input)) {
			*result = map->value;
			return 0;
		}
	}
	return -1;
}

const char *
nc_format_int_mapped(unsigned int value, const nc_intmap_t *map)
{
	if(map) {
		for (; map->name; ++map) {
			if (map->value == value)
				return map->name;
		}
	}
	return NULL;
}

int
nc_parse_hex(const char *string, unsigned char *data, unsigned int datasize)
{
	unsigned int len = 0;

	while(datasize > len) {
		unsigned int octet;

		octet = strtoul(string, (char **) &string, 16);
		if (octet > 255)
			return -1;

		data[len++] = octet;
		if(*string == '\0')
			break;
		if(*string++ != ':')
			return -1;
		if (len >= datasize)
			return -1;
	}
	return len;
}

const char *
nc_format_hex(const unsigned char *data, unsigned int datalen,
              char *namebuf, size_t namelen)
{
	unsigned int i, j;

	if(!data || !namebuf)
		return NULL;

	memset(namebuf, 0, namelen);
	for (i = j = 0; i < datalen; ++i) {
		if (j + 4 >= namelen)
			break;
		if (i)
			namebuf[j++] = ':';
		snprintf(namebuf + j, namelen - j, "%02x", data[i]);
		j += 2;
	}
	return namebuf;
}

int
nc_parse_bool(const char *string, int *result)
{
	static const nc_intmap_t map[] = {
		{ "true",	1	},
		{ "false",	0	},
		{ "yes",	1	},
		{ "no",		0	},
		{ "on",		1	},
		{ "off",	0	},
		{ NULL,		0	}
	};
	unsigned int mapped = 0;

	if (nc_parse_int_mapped(string, map, &mapped))
		return -1;

	*result = mapped;
	return 0;
}

