/*
 *	Translation between internal representation and netcf XML
 *
 *	Copyright (C) 2010 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2013 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *	MA  02110-1301  USA
 *
 *	Authors: Olaf Kirch <okir@suse.de>
 *	         Marius Tomaschewski <mt@suse.de>
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	You can find the wicked project at http://gitorious.org/wicked/.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if_arp.h>
#include <arpa/inet.h>

#include <sutils.h>
#include <logging.h>
#include <xml.h>
#include <nutils.h>
#include <ntypes.h>
#include <syntax.h>
#include <handle.h>
#include <vlan.h>
#include <bridge.h>
#include <bonding.h>

nc_syntax_t *			nc__syntax_xml_netcf(const char *pathname);

static int			__nc_netcf_supported_inteface(
						nc_syntax_t          *syntax,
						nc_handle_t          *handle,
						const nc_interface_t *ifp);

static xml_node_t *		__nc_netcf_xml_from_interface(
						nc_syntax_t          *syntax,
						nc_handle_t          *handle,
						nc_interface_t       *ifp,
						xml_node_t           *parent);

static nc_interface_t *		__nc_netcf_xml_to_interface(
						nc_syntax_t          *syntax,
						nc_handle_t          *handle,
						const xml_node_t     *ifnode);


static nc_intmap_t      __netcf_type_names_map[] = {
	{ "unknown",		NC_IFTYPE_UNKNOWN		},
	{ "ethernet",		NC_IFTYPE_ETHERNET		},
	{ "bridge",		NC_IFTYPE_BRIDGE		},
	{ "bond",		NC_IFTYPE_BOND			},
	{ "vlan",		NC_IFTYPE_VLAN			},
	{ "ethernet",		NC_IFTYPE_INFINIBAND		},
	{ NULL,			0				},
};
static nc_intmap_t	__bonding_modes_map[] = {
	{ "balance-rr",		NC_BOND_MODE_BALANCE_RR		},
	{ "active-backup",	NC_BOND_MODE_ACTIVE_BACKUP	},
	{ "balance-xor",	NC_BOND_MODE_BALANCE_XOR	},
	{ "broadcast",		NC_BOND_MODE_BROADCAST		},
	{ "802.3ad",		NC_BOND_MODE_802_3AD		},
	{ "balance-tlb",	NC_BOND_MODE_BALANCE_TLB	},
	{ "balance-alb",	NC_BOND_MODE_BALANCE_ALB	},
	{ NULL,			0				},
};
static nc_intmap_t	__arpmon_validation_map[] = {
	{ "none",		NC_BOND_VALIDATE_NONE		},	/* !strict */
	{ "active",		NC_BOND_VALIDATE_ACTIVE		},
	{ "backup",		NC_BOND_VALIDATE_BACKUP		},
	{ "all",		NC_BOND_VALIDATE_ALL		},
	{ NULL,			0				},
};
static nc_intmap_t	__bridge_cfg_attr_map[] = {
	{ "stp",		NC_BRIDGE_STP_ENABLED		},
	{ "delay",		NC_BRIDGE_FORWARD_DELAY		},
	{ "ageing-time",	NC_BRIDGE_AGEING_TIME		},	/* !strict */
	{ "hello-time",		NC_BRIDGE_HELLO_TIME		},	/* !strict */
	{ "max-age",		NC_BRIDGE_MAX_AGE		},	/* !strict */
	{ "priority",		NC_BRIDGE_PRIORITY		},	/* !strict */
	{ NULL,			0				},
};
static nc_intmap_t	__bridge_port_cfg_attr_map[] = {
	{ "priority",		NC_BRIDGE_PORT_PRIORITY		},	/* !strict */
	{ "path-cost",		NC_BRIDGE_PORT_PATH_COST	},	/* !strict */
	{ NULL,			0				},
};
static nc_intmap_t	__start_mode_map[] = {
	{ "none",		NC_STARTMODE_MANUAL		},
	{ "onboot",		NC_STARTMODE_AUTO		},
	{ "hotplug",		NC_STARTMODE_HOTPLUG		},
	{ NULL,			0				},
};
static nc_intmap_t	__proto_child_node_map[] = {
	{ "autoconf",		NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF)	},
	{ "dhcp",		NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP)	},
	{ "ip",			NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC)	},
	{ NULL,			0					},
};

static const char *
__get_iftype_name(unsigned int type)
{
	const char *n = nc_format_int_mapped(type, __netcf_type_names_map);
	return n ? n : __netcf_type_names_map[0].name;
}

static unsigned int
__map_iftype_to_netcf(const nc_interface_t *ifp, unsigned int strict)
{
	switch(ifp ? ifp->type : NC_IFTYPE_UNKNOWN) {
		case NC_IFTYPE_ETHERNET:
		case NC_IFTYPE_BRIDGE:
		case NC_IFTYPE_VLAN:
			return ifp->type;

		case NC_IFTYPE_BOND:
			if (strict && ifp->bonding &&
			    ifp->bonding->slave_type != NC_IFTYPE_ETHERNET)
				break;
			return ifp->type;

		case NC_IFTYPE_LOOPBACK:	/* as netcf does ... */
		case NC_IFTYPE_TOKENRING:
		case NC_IFTYPE_INFINIBAND:
			if (strict)
				break;
			return NC_IFTYPE_ETHERNET;
		case NC_IFTYPE_DUMMY:
		case NC_IFTYPE_TAP:
			return NC_IFTYPE_ETHERNET;

		default:
			break;
	}
	return NC_IFTYPE_UNKNOWN;
}

static const char *
__get_start_mode(unsigned int mode)
{
	return nc_format_int_mapped(mode, __start_mode_map);
}

static const char *
__get_bonding_mode(unsigned int mode)
{
	const char *m = nc_format_int_mapped(mode, __bonding_modes_map);
	return m ? m : __bonding_modes_map[0].name;
}
static int
__set_bonding_mode(const char *value, unsigned int *var)
{
	return nc_parse_int_mapped(value, __bonding_modes_map, var);
}

static const char *
__get_arpmon_validation(unsigned int mode)
{
	const char *m = nc_format_int_mapped(mode, __arpmon_validation_map);
	return m ? m : __arpmon_validation_map[0].name;
}
static int
__set_arpmon_validation(const char *value, unsigned int *var)
{
	return nc_parse_int_mapped(value, __arpmon_validation_map, var);
}
static int
__xml_get_boolean_attr(const xml_node_t *node, const char *attrname, int *var)
{
	const char *attrval;

	assert(node && attrname && var);

	*var = 0;

	if (!(attrval = xml_node_get_attr(node, attrname)))
		return -1;

	if (nc_string_eq(attrval, "on") || nc_string_eq(attrval, "yes"))
		*var = 1;
	else
	if (nc_string_eq(attrval, "off") || nc_string_eq(attrval, "no"))
		*var = 0;
	else {
		nc_error("unexpected boolean value <%s %s=\"%s\"> ignored",
			node->name, attrname, attrval);
		return -1;
	}
	return 0;
}

/* ========================================================================= */

nc_syntax_t *
nc__syntax_xml_netcf(const char *pathname)
{
	nc_syntax_t *syntax;

	syntax = calloc(1, sizeof(*syntax));
	if(syntax) {
		syntax->schema = "netcf";
		syntax->strict = 1;
		if(pathname && nc_string_dup(&syntax->base_path, pathname) != 0)
			goto cleanup;

		syntax->supported_inteface = __nc_netcf_supported_inteface;

		syntax->xml_from_interface = __nc_netcf_xml_from_interface;
		syntax->xml_to_interface   = __nc_netcf_xml_to_interface;
	}
	return syntax;

cleanup:
	nc_syntax_free(syntax);
	return NULL;
}


/* ========================================================================= */
static int
__nc_netcf_supported_inteface(	nc_syntax_t          *syntax,
				nc_handle_t          *handle,
				const nc_interface_t *ifp)
{
	if(!syntax || !handle || !ifp || !ifp->name)
		return -1;

	if(syntax->strict && ifp->ifindex && ifp->arp_type != ARPHRD_ETHER)
		return 1;

	if(__map_iftype_to_netcf(ifp, syntax->strict) == NC_IFTYPE_UNKNOWN)
		return 1;

	if(ifp->startmode == NC_STARTMODE_NFSROOT ||
	   ifp->startmode == NC_STARTMODE_IFPLUGD)
		return 1;

	if(NC_ADDRCONF_TEST(ifp->ipv4.addrconf, NC_ADDRCONF_IBFT) ||
	   NC_ADDRCONF_TEST(ifp->ipv6.addrconf, NC_ADDRCONF_IBFT))
		return 1;

	return 0;
}


/* ========================================================================= */
static xml_node_t *		__netcf_xml_from_interface(
						nc_syntax_t          *syntax,
						nc_handle_t          *handle,
						nc_interface_t       *ifp,
						xml_node_t           *parent,
						nc_interface_array_t *guard);
static xml_node_t *		__netcf_xml_from_vlan(
						nc_syntax_t          *syntax,
						nc_handle_t          *handle,
						nc_interface_t       *ifp,
						xml_node_t           *ifnode,
						nc_interface_array_t *guard);
static xml_node_t *		__netcf_xml_from_bridge(
						nc_syntax_t          *syntax,
						nc_handle_t          *handle,
						nc_interface_t       *ifp,
						xml_node_t           *ifnode,
						nc_interface_array_t *guard);
static xml_node_t *		__netcf_xml_from_bonding(
						nc_syntax_t          *syntax,
						nc_handle_t          *handle,
						nc_interface_t       *ifp,
						xml_node_t           *ifnode,
						nc_interface_array_t *guard);
static int			__xml_from_address_config(
						nc_syntax_t          *syntax,
						nc_handle_t          *handle,
						const nc_afinfo_t    *afi,
						const nc_interface_t *ifp,
						xml_node_t           *ifnode);
static xml_node_t *		__netcf_xml_from_ifname_ref(
						xml_node_t           *parent,
						const char           *ifname,
						unsigned int          iftype);

static xml_node_t *
__nc_netcf_xml_from_interface(	nc_syntax_t          *syntax,
				nc_handle_t          *handle,
				nc_interface_t       *ifp,
				xml_node_t           *parent)
{
	nc_interface_array_t guard = NC_INTERFACE_ARRAY_INIT;
	xml_node_t *ifnode = NULL;

	ifnode = __netcf_xml_from_interface(syntax, handle, ifp, parent, &guard);
	nc_interface_array_destroy(&guard);

	return ifnode;
}

static int /* bool */
check_interface_loop(nc_interface_array_t *guard, const char *ifname)
{
	nc_stringbuf_t buf = NC_STRINGBUF_INIT;
	unsigned int i;

	if(guard->count > 0) {
		for(i = 0; i < guard->count; ++i) {
			nc_interface_t *p = guard->data[i];
			if(i > 0)
				nc_stringbuf_puts(&buf, " -> ");
			nc_stringbuf_puts(&buf, p->name);
		}
		nc_stringbuf_puts(&buf, " -> ");
		nc_stringbuf_puts(&buf, ifname);
	}
	if(nc_interface_array_index_by_name(guard, ifname) != -1) {
		nc_error("recursive loop in interface chain: %s", buf.string);
		nc_stringbuf_destroy(&buf);
		return 1;
	} else {
		nc_debug_netcf("interface chain: %s",
			buf.string ? buf.string : ifname);
		nc_stringbuf_destroy(&buf);
		return 0;
	}
}

static xml_node_t *
__netcf_xml_from_interface(	nc_syntax_t          *syntax,
				nc_handle_t          *handle,
				nc_interface_t       *ifp,
				xml_node_t           *parent,
				nc_interface_array_t *guard)
{
	xml_node_t *ifnode, *node;
	const char *ptr;
	unsigned int type;

	if(!syntax || !handle || !ifp || !ifp->name)
		return NULL;

	assert(ifp && ifp->name);
	type = __map_iftype_to_netcf(ifp, syntax->strict);
	if(type == NC_IFTYPE_UNKNOWN)
		return NULL;

	if(check_interface_loop(guard, ifp->name))
		return NULL;
	if(nc_interface_array_append(guard, nc_interface_ref(ifp)) < 0)
		return NULL;

	ifnode = xml_node_new("interface", NULL);
	xml_node_add_attr(ifnode, "type", __get_iftype_name(type));
	xml_node_add_attr(ifnode, "name", ifp->name);

	ptr = __get_start_mode(ifp->startmode);
	if(ptr) {
		node = xml_node_new("start", ifnode);
		xml_node_add_attr(node, "mode", ptr);
	}

	/* Hmm... show the MAC regardless of the interface type
	 * Libvirt and backend will drop it while write as needed
	 */
	if(ifp->hwaddr.len) {
		nc_stringbuf_t nsb = NC_STRINGBUF_INIT;
		if(nc_hw_address_format(&ifp->hwaddr, &nsb)) {
			node = xml_node_new("mac", ifnode);
			xml_node_add_attr(node, "address", nsb.string);
			nc_stringbuf_destroy(&nsb);
		}
	}

	if(ifp->mtu != 0) {
		node = xml_node_new("mtu", ifnode);
		xml_node_add_attr_uint(node, "size", ifp->mtu);
	}

	/* this should be not needed -> backend job */
	if(!ifp->parent) {
		__xml_from_address_config(syntax, handle, &ifp->ipv4, ifp, ifnode);
		__xml_from_address_config(syntax, handle, &ifp->ipv6, ifp, ifnode);
	}

	switch(type) {
	case NC_IFTYPE_VLAN:
		if(!__netcf_xml_from_vlan(syntax, handle, ifp, ifnode, guard))
			goto failure;
	break;

	case NC_IFTYPE_BOND:
		if(!__netcf_xml_from_bonding(syntax, handle, ifp, ifnode, guard))
			goto failure;
	break;

	case NC_IFTYPE_BRIDGE:
		if(!__netcf_xml_from_bridge(syntax, handle, ifp, ifnode, guard))
			goto failure;
	break;

	default:
	break;
	}

	nc_interface_array_remove_index(guard, guard->count - 1);

	/* add to parent when given and return ifnode */
	if(!parent || xml_node_append_child_node(parent, ifnode))
		return ifnode;
failure:
	xml_node_free(ifnode);
	return NULL;
}

static xml_node_t *
__netcf_xml_from_vlan(		nc_syntax_t          *syntax,
				nc_handle_t          *handle,
				nc_interface_t       *ifp,
				xml_node_t           *ifnode,
				nc_interface_array_t *guard)
{
	const nc_vlan_t *vlan;
	xml_node_t *vlnode;

	assert(syntax && handle && ifnode && ifp);
	vlan = ifp->vlan;
	if(!vlan)
		return NULL;

	if(ifp->type != NC_IFTYPE_VLAN && ifp->type != NC_IFTYPE_INFINIBAND)
		return NULL;

	vlnode = xml_node_new("vlan", ifnode);
	xml_node_add_attr_uint(vlnode, "tag", vlan->tag);

	if(vlan->interface_name) {
		if(check_interface_loop(guard, vlan->interface_name))
			return NULL;

		if(__netcf_xml_from_ifname_ref(vlnode,
				vlan->interface_name, NC_IFTYPE_UNKNOWN))
			return vlnode;
	}
	return NULL;
}

static void
__netcf_xml_from_bridge_config(nc_bridge_t *bridge, const char *attr,
				xml_node_t *node)
{
	unsigned int opt;
	char *value = NULL;

	if (nc_parse_int_mapped(attr, __bridge_cfg_attr_map, &opt) < 0)
		return;

	if (nc_bridge_get(bridge, opt, &value) > 0) {
		xml_node_add_attr(node, attr, value);
		nc_string_free(&value);
	}
}

static void
__netcf_xml_from_bridge_port_config(nc_bridge_t *bridge, const char *port,
				const char *attr, xml_node_t *node)
{
	unsigned int opt;
	char *value = NULL;

	if (nc_parse_int_mapped(attr, __bridge_port_cfg_attr_map, &opt) < 0)
		return;

	if (nc_bridge_port_get(bridge, port, opt, &value) > 0) {
		xml_node_add_attr(node, attr, value);
		nc_string_free(&value);
	}
}

static xml_node_t *
__netcf_xml_from_bridge(nc_syntax_t          *syntax,
			nc_handle_t          *handle,
			nc_interface_t       *ifp,
			xml_node_t           *ifnode,
			nc_interface_array_t *guard)
{
	nc_bridge_t *bridge;
	xml_node_t *brnode;
	unsigned int i;

	assert(syntax && handle && ifnode && ifp);
	bridge = ifp->bridge;
	if(!bridge)
		return NULL;

	brnode = xml_node_new("bridge", ifnode);
	__netcf_xml_from_bridge_config(bridge, "stp", brnode);
	__netcf_xml_from_bridge_config(bridge, "delay", brnode);
	if (!syntax->strict) {
		__netcf_xml_from_bridge_config(bridge, "ageing-time", brnode);
		__netcf_xml_from_bridge_config(bridge, "hello-time", brnode);
		__netcf_xml_from_bridge_config(bridge, "priority", brnode);
		__netcf_xml_from_bridge_config(bridge, "max-age", brnode);
	}
	for (i = 0; i < bridge->ports.count; ++i) {
		nc_interface_t *port_ifp;
		xml_node_t *	port_node;
		char       *	port_name;

		/* when this happens, it is a bug */
		assert(bridge->ports.data[i]);

		port_name = bridge->ports.data[i]->name;
		port_ifp = nc_interface_by_name(handle, port_name);
		if(port_ifp) {
			port_node = __netcf_xml_from_interface(syntax,
				handle,	port_ifp, brnode, guard);
		} else {
			port_node = __netcf_xml_from_ifname_ref(brnode,
						port_name, NC_IFTYPE_ETHERNET);
		}
		if(!port_node)
			return NULL;

		if(!syntax->strict) {
			__netcf_xml_from_bridge_port_config(bridge,
				port_name, "priority", port_node);
			__netcf_xml_from_bridge_port_config(bridge,
				port_name, "path-cost", port_node);
		}
	}
	return brnode;
}

static xml_node_t *
__netcf_xml_from_bonding(nc_syntax_t          *syntax,
			 nc_handle_t          *handle,
			 nc_interface_t       *ifp,
			 xml_node_t           *ifnode,
			 nc_interface_array_t *guard)
{
	const nc_bonding_t *bonding;
	xml_node_t *bdnode;
	unsigned int i, j;

	assert(syntax && handle && ifnode && ifp);
	bonding = ifp->bonding;
	if(!bonding)
		return NULL;

	if(ifp->type != NC_IFTYPE_BOND)
		return NULL;

	bdnode = xml_node_new("bond", ifnode);
	xml_node_add_attr(bdnode, "mode", __get_bonding_mode(bonding->mode));

	if(bonding->monitoring == NC_BOND_MONITOR_ARP) {
		nc_stringbuf_t targets = NC_STRINGBUF_INIT;
		unsigned int max_targets = 16; /* multiple possible here! */
		xml_node_t *arpnode;

		arpnode = xml_node_new("arpmon", bdnode);
		xml_node_add_attr_uint(arpnode, "interval", bonding->arpmon.interval);

		for (j = 0; j < bonding->arpmon.targets.count; ++j) {
			if(max_targets && j >= max_targets)
				break;
			if(j > 0)
				nc_stringbuf_putc(&targets, ',');
			nc_stringbuf_puts(&targets, bonding->arpmon.targets.data[j]);
		}
		if(nc_string_len(targets.string))
			xml_node_add_attr(arpnode, "target", targets.string);
		nc_stringbuf_destroy(&targets);

		if(bonding->arpmon.validate != NC_BOND_VALIDATE_NONE) {
			xml_node_add_attr(arpnode, "validate",
				__get_arpmon_validation(bonding->arpmon.validate));
		}
	} else
	if(bonding->monitoring == NC_BOND_MONITOR_MII) {
		xml_node_t *miinode;

		miinode = xml_node_new("miimon", bdnode);
		xml_node_add_attr_uint(miinode, "freq", bonding->miimon.frequency);

		if (bonding->miimon.updelay)
			xml_node_add_attr_uint(miinode, "updelay",
				bonding->miimon.updelay);

		if (bonding->miimon.downdelay)
			xml_node_add_attr_uint(miinode, "downdelay",
				bonding->miimon.downdelay);

		if (bonding->miimon.carrier_detect == NC_BOND_CARRIER_DETECT_IOCTL)
			xml_node_add_attr(miinode, "carrier", "ioctl");
		else
			xml_node_add_attr(miinode, "carrier", "netif");
	} else {
		nc_error("unsupported bonding monitoring mode %d", bonding->monitoring);
		return NULL;
	}

	for (i = 0; i < bonding->slave_names.count; ++i) {
		nc_interface_t *	slave_ifp;
		const char *		slave_name;
		xml_node_t *		slave_node;

		slave_name  = bonding->slave_names.data[i];
		slave_ifp = nc_interface_by_name(handle, slave_name);
		if(slave_ifp) {
			slave_node = __netcf_xml_from_interface(syntax,
				handle,	slave_ifp, bdnode, guard);
		} else {
			unsigned int slave_type = bonding->slave_type;

			/* assume ethernet -- could be also infiniband */
			if(slave_type == NC_IFTYPE_UNKNOWN)
				slave_type = NC_IFTYPE_ETHERNET;

			slave_node = __netcf_xml_from_ifname_ref(bdnode,
						slave_name, slave_type);
		}
		if(slave_node == NULL)
			return NULL;

		if(!syntax->strict) {
			if(bonding->primary_slave &&
			   nc_string_eq(bonding->primary_slave, slave_name)) {
				xml_node_add_attr(slave_node, "primary", "yes");
			}
		}
	}

	return bdnode;
}

static xml_node_t *
__netcf_xml_from_ifname_ref(xml_node_t *parent,
				const char * name,
				unsigned int type)
{
	xml_node_t *ifnode;

	assert(name && *name && parent);

	ifnode = xml_node_new("interface", parent);
	xml_node_add_attr(ifnode, "name", name);

	if(type != NC_IFTYPE_UNKNOWN) {
		xml_node_add_attr(ifnode, "type", __get_iftype_name(type));
	}
	return ifnode;
}

static inline const char *
__get_af_name(int af)
{
	switch (af) {
	case AF_INET:
		return "ipv4";
	case AF_INET6:
		return "ipv6";
	default:
	break;
	}
	return NULL;
}

static inline xml_node_t *
__make_protocol_node(xml_node_t *ifnode, int af)
{
	xml_node_t *node = NULL;
	const char *name = __get_af_name(af);
	if(name) {
		node = xml_node_new("protocol", ifnode);
		xml_node_add_attr(node, "family", name);
	}
	return node;
}

static int
__xml_from_route(nc_route_t *rp, xml_node_t *protnode)
{
	nc_stringbuf_t dest = NC_STRINGBUF_INIT;
	nc_stringbuf_t gway = NC_STRINGBUF_INIT;
	xml_node_t *routenode;

	if(rp->prefixlen != 0) {
		if(nc_address_format(&rp->destination, &dest) == NULL)
			goto failure;
	}
	if(nc_address_format(&rp->nh.gateway, &gway) == NULL)
		goto failure;

	routenode = xml_node_new("route", protnode);
	if(!routenode)
		goto failure;

	if (rp->prefixlen != 0) {
		xml_node_add_attr(routenode, "address", dest.string);
		xml_node_add_attr_uint(routenode, "prefix", rp->prefixlen);
	}
	if(rp->nh.gateway.ss_family != AF_UNSPEC) {
		xml_node_add_attr(routenode, "gateway", gway.string);
	}

	nc_stringbuf_destroy(&dest);
	nc_stringbuf_destroy(&gway);
	return 0;
failure:
	nc_stringbuf_destroy(&dest);
	nc_stringbuf_destroy(&gway);
	return -1;
}

static int
__xml_from_address_config(nc_syntax_t *syntax, nc_handle_t *handle,
				const nc_afinfo_t *afi,
				const nc_interface_t *ifp,
				xml_node_t *ifnode)
{
	xml_node_t *proto_node = NULL;

	assert(syntax && handle && ifnode);
	if(!afi || !ifp)
		return -1;

	if( !afi->enabled || !afi->addrconf)
		return 0;

	if( __get_af_name(afi->family) == NULL)
		return -1;

	if(NC_ADDRCONF_TEST(afi->addrconf, NC_ADDRCONF_AUTOCONF)) {
		if(!proto_node)
			proto_node = __make_protocol_node(ifnode, afi->family);
		if(!proto_node)
			return -1;
		xml_node_new("autoconf", proto_node);
	}
	if(NC_ADDRCONF_TEST(afi->addrconf, NC_ADDRCONF_DHCP)) {
		if(!proto_node)
			proto_node = __make_protocol_node(ifnode, afi->family);
		if(!proto_node)
			return -1;
		/* node = */ xml_node_new("dhcp", proto_node);
		/* xml_node_add_attr(node, "peerdns", "yes" or "no") */
	}
	if(NC_ADDRCONF_TEST(afi->addrconf, NC_ADDRCONF_STATIC)) {
		nc_address_t *ap;
		unsigned int i;

		for(i = 0; i < ifp->addrs.count; ++i) {
			nc_stringbuf_t addr_buf = NC_STRINGBUF_INIT;
			xml_node_t *addr_node;

			ap = ifp->addrs.data[i];
			if (ap->family != afi->family /* || ap->config_method != NC_ADDRCONF_STATIC */)
				continue;

			if(nc_address_format(&ap->local_addr, &addr_buf) == NULL)
				continue;

			if(!proto_node)
				proto_node = __make_protocol_node(ifnode, afi->family);
			if(!proto_node) {
				nc_stringbuf_destroy(&addr_buf);
				return -1;
			}

			addr_node = xml_node_new("ip", proto_node);
			xml_node_add_attr(addr_node, "address", addr_buf.string);
			xml_node_add_attr_uint(addr_node, "prefix", ap->prefixlen);

			nc_stringbuf_destroy(&addr_buf);
		}

		for(i = 0; i < ifp->routes.count; ++i) {
			nc_route_t *rp = ifp->routes.data[i];

			if(!rp || rp->family != afi->family)
				continue;

			if(rp->prefixlen != 0)
				continue;
			if(rp->nh.gateway.ss_family == AF_UNSPEC)
				continue;

			if(!proto_node)
				proto_node = __make_protocol_node(ifnode, afi->family);
			if(!proto_node)
				return -1;

			__xml_from_route(rp, proto_node);
		}
#if defined(NC_OPT_TRACE)
		for(i = 0; i < handle->routes.count; ++i) {
			nc_route_t *rp = handle->routes.data[i];
			nc_stringbuf_t dest = NC_STRINGBUF_INIT;
			nc_stringbuf_t gway = NC_STRINGBUF_INIT;

			if(nc_address_format(&rp->destination, &dest) == NULL)
				continue;

			if(nc_address_format(&rp->nh.gateway, &gway) == NULL) {
				nc_stringbuf_destroy(&dest);
				continue;
			}

			nc_trace("skipping route to %s/%u via %s %s%s",
				dest.string, rp->prefixlen, gway.string,
				(rp->nh.device ? "dev " : ""),
				(rp->nh.device ? rp->nh.device : ""));
			nc_stringbuf_destroy(&dest);
			nc_stringbuf_destroy(&gway);
		}
#endif
	}
	return 0;
}


/* ========================================================================= */
static nc_interface_t *		__netcf_xml_to_interface(
						nc_syntax_t      *syntax,
						nc_handle_t      *handle,
						const xml_node_t *ifnode,
						nc_interface_t   *parent);
static int			__netcf_xml_to_ipconfig(
						nc_syntax_t       *syntax,
						nc_handle_t       *handle,
						const xml_node_t  *ifnode,
						nc_interface_t    *ifp,
						nc_interface_t   *parent);
static int			__netcf_xml_to_vlan(
						nc_syntax_t       *syntax,
						nc_handle_t       *handle,
						const xml_node_t  *ifnode,
						nc_interface_t    *ifp);
static int			__netcf_xml_to_bridge(
						nc_syntax_t       *syntax,
						nc_handle_t       *handle,
						const xml_node_t  *ifnode,
						nc_interface_t    *ifp);
static int			__netcf_xml_to_bonding(
						nc_syntax_t       *syntax,
						nc_handle_t       *handle,
						const xml_node_t  *ifnode,
						nc_interface_t    *ifp);


static nc_interface_t *
__nc_netcf_xml_to_interface(	nc_syntax_t      *syntax,
				nc_handle_t      *handle,
				const xml_node_t *ifnode)
{
	nc_interface_t *top;
	nc_interface_t *ifp;
	unsigned int i;

#if defined(NC_OPT_TRACE)
	for(i=0; i< handle->ifaces.count; ++i) {
		ifp = handle->ifaces.data[i];
		if(ifp) {
			nc_trace("cfg iface[%u]: users=%u, parent=%s ifname=%s%s%s",
				i, ifp->users, ifp->parent, ifp->name,
				(ifp->modified ? " modified" : ""),
				(ifp->deleted ? " deleted" : ""));
		} else {
			nc_trace("cfg iface[%u]: NULL", i);
		}
	}
#endif
	/* reset modify/delete flags in handle */
	for(ifp=nc_interface_first(handle, &i); ifp; ifp=nc_interface_next(handle, &i)) {
		ifp->deleted = 0;
		ifp->modified = 0;
	}

	top = __netcf_xml_to_interface(syntax, handle, ifnode, NULL);
	if(top == NULL) {
		for(ifp=nc_interface_first(handle, &i); ifp; ifp=nc_interface_next(handle, &i)) {
			/* remove all deletion marks */
			ifp->deleted = 0;

			/* drop all modified interfaces (== parsed from xml) */
			if(ifp->modified)
				nc_interface_array_remove_index(&handle->ifaces, i--);
		}
	}
#if defined(NC_OPT_TRACE)
	for(i=0; i< handle->ifaces.count; ++i) {
		ifp = handle->ifaces.data[i];
		if(ifp) {
			nc_trace("new iface[%u]: users=%u, parent=%s ifname=%s%s%s",
				i, ifp->users, ifp->parent, ifp->name,
				(ifp->modified ? " modified" : ""),
				(ifp->deleted ? " deleted" : ""));
		} else {
			nc_trace("new iface[%u]: NULL", i);
		}
	}
#endif
	return top;
}

static nc_interface_t *
__netcf_xml_to_interface(	nc_syntax_t      *syntax,
				nc_handle_t      *handle,
				const xml_node_t *ifnode,
				nc_interface_t   *parent)
{
	nc_interface_t *ifp = NULL;
	const char *attrval;
	xml_node_t *node;

	if((attrval = xml_node_get_attr(ifnode, "name")) == NULL || !nc_string_len(attrval)) {
		nc_error_detail_fmt(handle, "<interface> element without valid name attribute");
		goto failure;
	}

	if(nc_interface_find(handle, NULL, -1, attrval, 1, 0) != NULL) {
		nc_error_detail_fmt(handle, "interface %s: loop in the interface tree",
			attrval);
		goto failure;
	}

	ifp = nc_interface_new(attrval, 0);
	if(!ifp) {
		nc_error_detail_fmt(handle, "interface %s: unable to allocate interface structure",
			attrval);
		goto failure;
	}
	/*
	 * mark it modified
	 */
	ifp->modified = 1;
	if(parent) {
		assert(parent->name);
		nc_string_dup(&ifp->parent, parent->name);
	}

	if(nc_interfaces_add(handle, nc_interface_ref(ifp)) < 0) {
		nc_error_detail_fmt(handle, "interface %s: unable to add to interface repository",
			ifp->name);
		goto failure;
	}

	if((attrval = xml_node_get_attr(ifnode, "type")) == NULL) {
		nc_error_detail_fmt(handle, "interface %s: element without type attribute",
			ifp->name);
		goto failure;
	}
	if(nc_parse_int_mapped(attrval, __netcf_type_names_map, &ifp->type) < 0) {
		nc_error_detail_fmt(handle, "interface %s: unknown/unsupported interface type %s",
				ifp->name, attrval);
		goto failure;
	}

	/*
	 * parent may override mode of slave/port, e.g. to auto or hotplug
	 */
	node = xml_node_get_child(ifnode, "start");
	if (node && (attrval = xml_node_get_attr(node, "mode")) != NULL) {
		if(nc_parse_int_mapped(attrval, __start_mode_map, &ifp->startmode) < 0) {
			nc_error_detail_fmt(handle, "interface %s: unknown/unsupported start mode %s",
				ifp->name, attrval);
			goto failure;
		}
	}

	/*
	 * vlan: inherits from base-if, but mtu of differ (vlan-if <= mtu of base-if)
	 * bond: sets own mtu on all its slaves
	 * bridge: bridge + all ports share same mtu
	 */
	node = xml_node_get_child(ifnode, "mtu");
	if (node && (attrval = xml_node_get_attr(node, "size")) != NULL) {
		if(nc_parse_uint(attrval, &ifp->mtu, 10) < 0) {
			nc_error_detail_fmt(handle, "interface %s: cannot interpret mtu size %s",
				ifp->name, attrval);
			goto failure;
		}
	}

	/*
	 * vlan: inherits mac from base-if _by_ _default_
	 * bond: sets mac depending on the mode [e.g. active slave]
	 * bridge: is using mac of first local port
	 */
	if(ifp->type != NC_IFTYPE_BRIDGE) {
		node = xml_node_get_child(ifnode, "mac");
		if (node && (attrval = xml_node_get_attr(node, "address")) != NULL) {
			if(nc_hw_address_parse(&ifp->hwaddr, ifp->type, attrval) < 0) {
				nc_error_detail_fmt(handle,
					"interface %s: cannot parse hardware address (MAC) %s",
					ifp->name, attrval);
				goto failure;
			}
		}
	}

	if(__netcf_xml_to_ipconfig(syntax, handle, ifnode, ifp, parent) < 0)
		goto failure;

	switch (ifp->type) {
		case NC_IFTYPE_VLAN:
			if(parent &&	!(parent->type == NC_IFTYPE_BRIDGE)) {
				nc_error_detail_fmt(handle,
					"invalid interface chain: %s %s in %s %s",
					__get_iftype_name(ifp->type), ifp->name,
					__get_iftype_name(parent->type), parent->name);
				goto failure;
			}

			if(__netcf_xml_to_vlan(syntax, handle, ifnode, ifp))
				goto failure;
		break;
		case NC_IFTYPE_BRIDGE:
			if(parent &&	!(parent->type == NC_IFTYPE_VLAN)) {
				nc_error_detail_fmt(handle,
					"invalid interface chain: %s %s in %s %s",
					__get_iftype_name(ifp->type), ifp->name,
					__get_iftype_name(parent->type), parent->name);
				goto failure;
			}

			if(__netcf_xml_to_bridge(syntax, handle, ifnode, ifp))
				goto failure;
		break;
		case NC_IFTYPE_BOND:
			/* basically bridge only as vlan does not recurse */
			if(parent &&	!(parent->type == NC_IFTYPE_VLAN ||
					  parent->type == NC_IFTYPE_BRIDGE)) {
				nc_error_detail_fmt(handle,
					"invalid interface chain: %s %s in %s %s",
					__get_iftype_name(ifp->type), ifp->name,
					__get_iftype_name(parent->type), parent->name);
				goto failure;
			}

			if(__netcf_xml_to_bonding(syntax, handle, ifnode, ifp))
				goto failure;
		break;

		case NC_IFTYPE_ETHERNET:
		default:
			/* nothing special */
		break;
	}

	return ifp;

failure:
	if(ifp)
		nc_interface_free(ifp);
	return NULL;
}

static int
__netcf_xml_to_ipconfig(
				nc_syntax_t       *syntax,
				nc_handle_t       *handle,
				const xml_node_t  *ifnode,
				nc_interface_t    *ifp,
				nc_interface_t   *parent)
{
	const char *attrval;
	xml_node_t *node;

	(void)syntax;
	(void)handle;

	/*
	 * when we're a slave/port, just ignore all ip config
	 * [vlan does not recurse / is not a parent]
	 */
	if(parent) {
		if(parent->type == NC_IFTYPE_BOND) {
			/* bootproto "none" */
			ifp->ipv4.addrconf = 0;
			ifp->ipv6.addrconf = 0;
		}
		/* else use defaults (static) */
		return 0;
	}

	for(node = ifnode->children; node; node = node->next) {
		nc_afinfo_t *afi;
		xml_node_t *child;
		const char *afn;
	
		if (!nc_string_eq(node->name, "protocol"))
			continue;

		attrval = xml_node_get_attr(node, "family");
		if (!attrval) {
			nc_error_detail_fmt(handle,
				"interface %s: protocol node without family attribute",
				ifp->name);
			goto failure;
		}
		if (nc_string_eq(attrval, "ipv4")) {
			afn = attrval;
			afi = &ifp->ipv4;
		} else
		if (nc_string_eq(attrval, "ipv6")) {
			afn = attrval;
			afi = &ifp->ipv6;
		} else {
			nc_info("ignoring unknown address family %s", attrval);
			continue;
		}

		afi->enabled = 1;
		afi->addrconf = 0;

		for (child = node->children; child; child = child->next) {
			nc_intmap_t *map;
			nc_address_t *ap;
			nc_sockaddr_t addr;
			unsigned int plen = 0;

			for(map = __proto_child_node_map; map->name; map++) {

				if(!nc_string_eq(map->name, child->name))
					continue;

				switch(map->value) {
				case NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC):
					afi->addrconf |= map->value;
					attrval = xml_node_get_attr(child, "address");
					if(!attrval) {
						nc_error_detail_fmt(handle,
							"ip node without address attribute");
						goto failure;
					}
					if (nc_address_parse(&addr, attrval, afi->family) < 0) {
						nc_error_detail_fmt(handle,
							"ip node with invalid %s address '%s'",
							afn, attrval);
						goto failure;
					}

					attrval = xml_node_get_attr(child, "prefix");
					if(attrval) {
						nc_parse_uint(attrval, &plen, 10);
					}
					if(afi->family == AF_INET) {
						if(plen == 0 || plen > 32)
							plen = 32;
					} else {
						if(plen == 0 || plen > 128)
							plen = 128;
					}

					ap = nc_address_new(afi->family, plen, &addr);
					if(ap) {
						nc_address_array_append(&ifp->addrs, ap);
					}
				break;
				case NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP):
					/* peerdns */
				case NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF):
				default:
					afi->addrconf |= map->value;
				break;
				}
			}

			if(nc_string_eq("route", child->name)) {
				nc_route_t *rp = NULL;
				nc_sockaddr_t gway_addr;
#if 0
				nc_sockaddr_t dest_addr;

				memset(&dest_addr, 0, sizeof(dest_addr));
				attrval = xml_node_get_attr(child, "destination");
				if(attrval && !nc_string_eq("default")) {
					if (nc_address_parse(&dest_addr, attrval, AF_UNSPEC) < 0) {
						nc_error("Unable to parse destination address \"%s\"",
								attrval);
						goto failure;
					}
				}
#endif

				attrval = xml_node_get_attr(child, "gateway");
				if(!attrval) {
					nc_error_detail_fmt(handle,
						"currently only default route with gateway supported");
					goto failure;
				}

				if (nc_address_parse(&gway_addr, attrval, AF_UNSPEC) < 0) {
					nc_error_detail_fmt(handle,
						"Unable to parse gateway address \"%s\"", attrval);
					goto failure;
				}

				if(afi->family != gway_addr.ss_family) {
					nc_error_detail_fmt(handle,
						"gateway (%s) does not match protocol node family (%s)",
							attrval, afn);
					goto failure;
				}

				rp = nc_route_create(0, NULL, &gway_addr, ifp->name, NULL);
				if(!rp) {
					/* should never happen :-) */
					nc_error_detail_fmt(handle,
						"Unable to create route via %s", attrval);
					goto failure;
				}
				if( nc_route_array_append(&ifp->routes, rp) < 0) {
					/* invalid parameter or oom (-> assert) only ... */
					nc_route_free(rp);
					nc_error_detail_fmt(handle, "Unable to add route to array");
					goto failure;
				}
				nc_trace("applied default route via %s dev %s", attrval, ifp->name);
			}
		}
	}

	return 0;
failure:
	return -1;
}

static int
__netcf_xml_to_vlan(		nc_syntax_t       *syntax,
				nc_handle_t       *handle,
				const xml_node_t  *ifnode,
				nc_interface_t    *ifp)
{
	xml_node_t *vnode, *rnode;
	const char *attrval;
	unsigned int tag;

	assert(syntax && handle && ifp && ifnode);
	assert(ifp->type == NC_IFTYPE_VLAN);

	if ((vnode = xml_node_get_child(ifnode, "vlan")) == NULL) {
		nc_error_detail_fmt(handle,
			"VLAN interface %s: xml config has no vlan element",
			ifp->name);
		goto failure;
	}
	if ((attrval = xml_node_get_attr(vnode, "tag")) == NULL) {
		nc_error_detail_fmt(handle,
			"VLAN interface %s: vlan element has no tag attribute",
			ifp->name);
		goto failure;
	}
	if(nc_parse_uint(attrval, &tag, 10) < 0) {
		nc_error_detail_fmt(handle,
			"VLAN interface %s: unable to parse vlan tag %s",
			ifp->name, attrval);
		goto failure;
	}

	if ((rnode = xml_node_get_child(vnode, "interface")) == NULL) {
		nc_error_detail_fmt(handle,
			"VLAN interface %s: vlan element has no interface element",
			ifp->name);
		goto failure;
	}
	if ((attrval = xml_node_get_attr(rnode, "name")) == NULL) {
		nc_error_detail_fmt(handle,
			"VLAN interface %s: vlan interface element has no name attribute",
			ifp->name);
		goto failure;
	}

	if(nc_string_eq(attrval, ifp->name)) {
		nc_error_detail_fmt(handle,
			"VLAN interface %s: refers to itself (%s)",
			ifp->name, attrval);
		goto failure;
	}

	if( (ifp->vlan = nc_vlan_new(attrval, tag)) == NULL) {
		nc_error_detail_fmt(handle,
			"VLAN interface %s: cannot allocate memory for %s vlan structure",
			ifp->name, attrval);
		goto failure;
	}

	/*
	 * Note: no recursion here!
	 */
	return 0;
failure:
	return -1;
}

static int
__netcf_xml_to_bridge(		nc_syntax_t       *syntax,
				nc_handle_t       *handle,
				const xml_node_t  *ifnode,
				nc_interface_t    *ifp)
{
	xml_node_t *brnode, *child;
	nc_bridge_t *bridge;
	nc_interface_t *nfp;

	assert(syntax && handle && ifp && ifnode);
	assert(ifp->type == NC_IFTYPE_BRIDGE);

	if (!(brnode = xml_node_get_child(ifnode, "bridge"))) {
		nc_error_detail_fmt(handle,
			"Bridge interface %s: xml config has no bridge element",
		       	ifp->name);
		return -1;
	}

	bridge = nc_bridge_new();
	if(!bridge) {
		nc_error_detail_fmt(handle,
			"Bridge interface %s: cannot allocate bridge structure",
		       	ifp->name);
		return -1;
	}

	/* stp disabled as default ?! */
	if (nc_bridge_set_stp(bridge, xml_node_get_attr(brnode, "stp")) < 0) {
		nc_error_detail_fmt(handle,
			"Bridge interface %s: bridge element lacks stp attribute",
		       	ifp->name);
		goto failure;
	}

	nc_bridge_set_forward_delay(bridge, xml_node_get_attr(brnode, "delay"));
	if (!syntax->strict) {
		nc_bridge_set_ageing_time(bridge, xml_node_get_attr(brnode, "ageing-time"));
		nc_bridge_set_hello_time(bridge, xml_node_get_attr(brnode, "hello-time"));
		nc_bridge_set_max_age(bridge, xml_node_get_attr(brnode, "max-age"));
		nc_bridge_set_priority(bridge, xml_node_get_attr(brnode, "priority"));
	}

	for (child = brnode->children; child; child = child->next) {
		const char *ifname;

		if (!nc_string_eq(child->name, "interface"))
			continue;

		if (!(ifname = xml_node_get_attr(child, "name"))) {
			nc_error_detail_fmt(handle,
				"Bridge interface %s: interface element lacks name attribute",
				ifp->name);
			goto failure;
		}

		if(nc_string_eq(ifname, ifp->name)) {
			nc_error_detail_fmt(handle,
				"Bridge interface %s: contains a port with name %s",
				ifp->name, ifname);
			goto failure;
		}

		if(nc_bridge_add_port(bridge, ifname) < 0) {
			nc_error_detail_fmt(handle,
				"Bridge interface %s: bridge port %s specified twice",
				ifp->name, ifname);
			goto failure;
		}
		if (!syntax->strict) {
			nc_bridge_port_set_priority(bridge, ifname,
				xml_node_get_attr(child, "priority"));
			nc_bridge_port_set_path_cost(bridge, ifname,
				xml_node_get_attr(child, "path-cost"));
		}

		/* recurse */
		nfp = __netcf_xml_to_interface(syntax, handle, child, ifp);
		if(!nfp) {
			/* error already reported */
			goto failure;
		}

		if(nfp->startmode == NC_STARTMODE_MANUAL)
			nfp->startmode = NC_STARTMODE_AUTO;

		/* release the reference */
		nc_interface_free(nfp);
	}

	ifp->bridge = bridge;
	return 0;

failure:
	if(bridge) {
		nc_bridge_free(bridge);
	}
	return -1;
}

static int
__netcf_xml_to_bonding(		nc_syntax_t       *syntax,
				nc_handle_t       *handle,
				const xml_node_t  *ifnode,
				nc_interface_t    *ifp)
{
	xml_node_t *bdnode, *child;
	nc_bonding_t *bonding;
	const char *attrval;
	nc_interface_t    *nfp;

	assert(syntax && handle && ifp && ifnode);
	assert(ifp->type == NC_IFTYPE_BOND);

	bdnode = xml_node_get_child(ifnode, "bond");
	if(!bdnode) {
		nc_error_detail_fmt(handle,
			"Bond interface %s: xml config has no bond element",
		       	ifp->name);
		return -1;
	}

	bonding = nc_bonding_new();

	if(!(attrval = xml_node_get_attr(bdnode, "mode"))) {
		/* active-backup is always usable -- it does not need any switch support */
		nc_warn("Bond interface %s: bond element lacks mode attribute"
			" -- using active-backup", ifp->name);

		attrval = "active-backup";
	}
	if(__set_bonding_mode(attrval, &bonding->mode) < 0) {
		nc_error_detail_fmt(handle,
			"Bond interface %s: unsupported bonding mode \"%s\"",
		       	ifp->name, attrval);
		goto failure;
	}

	if ((child = xml_node_get_child(bdnode, "arpmon")) != NULL) {
		bonding->monitoring = NC_BOND_MONITOR_ARP;

		if (xml_node_get_attr_uint(child, "interval", &bonding->arpmon.interval) < 0) {
			nc_error_detail_fmt(handle,
				"Bond interface %s: incomplete arpmon definition",
			       	ifp->name);
			goto failure;
		}

		bonding->arpmon.validate = NC_BOND_VALIDATE_NONE;
		if ((attrval = xml_node_get_attr(child, "validate")) != NULL &&
			__set_arpmon_validation(attrval, &bonding->arpmon.validate) < 0) {
			nc_error_detail_fmt(handle,
				"Bond interface %s: arpmon validate=\"%s\" not supported",
				ifp->name, attrval);
			goto failure;
		}
		if ((attrval = xml_node_get_attr(child, "target")) != NULL) {
			char *s, *t, *p = NULL;
			if( (s = strdup(attrval)) != NULL) {
				for(t = strtok_r(s, ", ", &p); t; t = strtok_r(NULL, ", ", &p)) {
					struct in_addr dummy;
					if (inet_aton(t, &dummy) == 0) {
						free(s);
						nc_error_detail_fmt(handle,
							"bond interface %s: bad arpmon target \"%s\"",
							ifp->name, attrval);
						goto failure;
					}
					if(nc_string_array_index(&bonding->arpmon.targets, t) == -1)
						nc_string_array_append(&bonding->arpmon.targets, t);
				}
				free(s);
			}
		}
		if(bonding->arpmon.interval == 0 || bonding->arpmon.targets.count == 0) {
			nc_error_detail_fmt(handle,
				"Bond interface %s: arpmon interval (%u) and target (%u) required",
				ifp->name, bonding->arpmon.interval, bonding->arpmon.targets.count);
			goto failure;
		}
	} else
	if ((child = xml_node_get_child(bdnode, "miimon")) != NULL) {
		bonding->monitoring = NC_BOND_MONITOR_MII;

		if ((xml_node_get_attr_uint(child, "freq", &bonding->miimon.frequency) < 0) ||
		    (xml_node_get_attr_uint(child, "validate", &bonding->arpmon.validate) < 0)) {
			nc_error_detail_fmt(handle,
				"Bond interface %s: incomplete miimon definition",
			       	ifp->name);
			goto failure;
		}

		xml_node_get_attr_uint(child, "updelay", &bonding->miimon.updelay);
		xml_node_get_attr_uint(child, "downdelay", &bonding->miimon.downdelay);

		if ((attrval = xml_node_get_attr(child, "carrier")) != NULL) {
			if (nc_string_eq(attrval, "ioctl"))
				bonding->miimon.carrier_detect = NC_BOND_CARRIER_DETECT_IOCTL;
			else
				bonding->miimon.carrier_detect = NC_BOND_CARRIER_DETECT_NETIF;
		}

		if(bonding->miimon.frequency == 0) {
			nc_error_detail_fmt(handle,
				"Bond interface %s: miimon frequency (%u) greater 0 required",
				ifp->name, bonding->miimon.frequency);
			goto failure;
		}
	} else {
		/*
		 * request to use system defaults ... but:
		 * "It is critical that either the miimon or arp_interval and
		 *  arp_ip_target parameters be specified, otherwise serious
		 *  network degradation will occur during link failures."
		 */
		bonding->monitoring = NC_BOND_MONITOR_MII;
		bonding->miimon.frequency = 100;
	}

	/* add all slave interfaces */
	for (child = bdnode->children; child; child = child->next) {
		const char *ifname;
		int primary = 0;

		if (!nc_string_eq(child->name, "interface"))
			continue;

		if (!(ifname = xml_node_get_attr(child, "name"))) {
			nc_error_detail_fmt(handle,
				"Bond interface %s: interface element lacks name attribute",
			       	ifp->name);
			goto failure;
		}

		if(nc_string_eq(ifname, ifp->name)) {
			nc_error_detail_fmt(handle,
				"Bond interface %s: contains a slave with name %s",
				ifp->name, ifname);
			goto failure;
		}

		if (__xml_get_boolean_attr(child, "primary", &primary) >= 0 && primary) {
			if (bonding->primary_slave != NULL) {
				nc_error_detail_fmt(handle,
					"Bond interface %s: more than one primary slave specified",
					ifp->name);
				goto failure;
			}
			nc_string_dup(&bonding->primary_slave, ifname);
		}

		if(nc_bonding_add_slave(bonding, ifname) < 0) {
			nc_error_detail_fmt(handle,
				"Bond interface %s: slave infterface %s specified twice",
				ifp->name, ifname);
			goto failure;
		}

		nfp = __netcf_xml_to_interface(syntax, handle, child, ifp);
		if(!nfp) {
			/* error already reported */
			goto failure;
		}

		if(nfp->startmode == NC_STARTMODE_MANUAL)
			nfp->startmode = NC_STARTMODE_HOTPLUG;

		/* release the reference */
		nc_interface_free(nfp);
	}

	if(bonding->slave_names.count == 0) {
		nc_error_detail_fmt(handle,
			"Bond interface %s: at least one slave interface is required",
		       	ifp->name);
		goto failure;
	}

	ifp->bonding = bonding;
	return 0;

failure:
	nc_bonding_free(bonding);
	return -1;
}

