/*
 * Copyright 2018-2020 Raffaello Giulietti
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package com.fasterxml.jackson.core.io.schubfach;

import java.math.BigDecimal;
import java.util.Random;

import static java.lang.Double.*;
import static java.lang.Long.numberOfTrailingZeros;
import static java.lang.StrictMath.scalb;
import static com.fasterxml.jackson.core.io.schubfach.MathUtils.flog10pow2;

public class DoubleToDecimalChecker extends ToDecimalChecker {

    static final int P =
            numberOfTrailingZeros(doubleToRawLongBits(3)) + 2;
    private static final int W = (SIZE - 1) - (P - 1);
    static final int Q_MIN = (-1 << W - 1) - P + 3;
    static final int Q_MAX = (1 << W - 1) - P;
    static final long C_MIN = 1L << P - 1;
    static final long C_MAX = (1L << P) - 1;

    static final int K_MIN = flog10pow2(Q_MIN);
    static final int K_MAX = flog10pow2(Q_MAX);
    static final int H = flog10pow2(P) + 2;

    static final double MIN_VALUE = scalb(1.0, Q_MIN);
    static final double MIN_NORMAL = scalb((double) C_MIN, Q_MIN);
    static final double MAX_VALUE = scalb((double) C_MAX, Q_MAX);

    static final int E_MIN = e(MIN_VALUE);
    static final int E_MAX = e(MAX_VALUE);

    static final long C_TINY = cTiny(Q_MIN, K_MIN);

    private double v;
    private final long originalBits;

    private DoubleToDecimalChecker(double v, String s) {
        super(s);
        this.v = v;
        originalBits = doubleToRawLongBits(v);
    }

    @Override
    BigDecimal toBigDecimal() {
        return new BigDecimal(v);
    }

    @Override
    boolean recovers(BigDecimal b) {
        return b.doubleValue() == v;
    }

    @Override
    boolean recovers(String s) {
        return parseDouble(s) == v;
    }

    @Override
    String hexBits() {
        return String.format("0x%01X__%03X__%01X_%04X_%04X_%04X",
                (int) (originalBits >>> 63) & 0x1,
                (int) (originalBits >>> 52) & 0x7FF,
                (int) (originalBits >>> 48) & 0xF,
                (int) (originalBits >>> 32) & 0xFFFF,
                (int) (originalBits >>> 16) & 0xFFFF,
                (int) originalBits & 0xFFFF);
    }

    @Override
    int minExp() {
        return E_MIN;
    }

    @Override
    int maxExp() {
        return E_MAX;
    }

    @Override
    int maxLen10() {
        return H;
    }

    @Override
    boolean isZero() {
        return v == 0;
    }

    @Override
    boolean isInfinity() {
        return v == POSITIVE_INFINITY;
    }

    @Override
    void negate() {
        v = -v;
    }

    @Override
    boolean isNegative() {
        return originalBits < 0;
    }

    @Override
    boolean isNaN() {
        return Double.isNaN(v);
    }

    static void toDec(double v) {
//        String s = Double.toString(v);
        String s = DoubleToDecimal.toString(v);
        new DoubleToDecimalChecker(v, s).validate();
    }

    /*
    There are tons of doubles that are rendered incorrectly by the JDK.
    While the renderings correctly round back to the original value,
    they are longer than needed or are not the closest decimal to the double.
    Here are just a very few examples.
     */
    static final String[] Anomalies = {
            // JDK renders these, and others, with 18 digits!
            "2.82879384806159E17", "1.387364135037754E18",
            "1.45800632428665E17",

            // JDK renders these longer than needed.
            "1.6E-322", "6.3E-322",
            "7.3879E20", "2.0E23", "7.0E22", "9.2E22",
            "9.5E21", "3.1E22", "5.63E21", "8.41E21",

            // JDK does not render these, and many others, as the closest.
            "9.9E-324", "9.9E-323",
            "1.9400994884341945E25", "3.6131332396758635E25",
            "2.5138990223946153E25",
    };

    /*
    Values are from
    Paxson V, "A Program for Testing IEEE Decimal-Binary Conversion"
    tables 3 and 4
     */
    static final double[] PaxsonSignificands = {
            8_511_030_020_275_656L,
            5_201_988_407_066_741L,
            6_406_892_948_269_899L,
            8_431_154_198_732_492L,
            6_475_049_196_144_587L,
            8_274_307_542_972_842L,
            5_381_065_484_265_332L,
            6_761_728_585_499_734L,
            7_976_538_478_610_756L,
            5_982_403_858_958_067L,
            5_536_995_190_630_837L,
            7_225_450_889_282_194L,
            7_225_450_889_282_194L,
            8_703_372_741_147_379L,
            8_944_262_675_275_217L,
            7_459_803_696_087_692L,
            6_080_469_016_670_379L,
            8_385_515_147_034_757L,
            7_514_216_811_389_786L,
            8_397_297_803_260_511L,
            6_733_459_239_310_543L,
            8_091_450_587_292_794L,

            6_567_258_882_077_402L,
            6_712_731_423_444_934L,
            6_712_731_423_444_934L,
            5_298_405_411_573_037L,
            5_137_311_167_659_507L,
            6_722_280_709_661_868L,
            5_344_436_398_034_927L,
            8_369_123_604_277_281L,
            8_995_822_108_487_663L,
            8_942_832_835_564_782L,
            8_942_832_835_564_782L,
            8_942_832_835_564_782L,
            6_965_949_469_487_146L,
            6_965_949_469_487_146L,
            6_965_949_469_487_146L,
            7_487_252_720_986_826L,
            5_592_117_679_628_511L,
            8_887_055_249_355_788L,
            6_994_187_472_632_449L,
            8_797_576_579_012_143L,
            7_363_326_733_505_337L,
            8_549_497_411_294_502L,
    };

    static final int[] PaxsonExponents = {
            -342,
            -824,
            237,
            72,
            99,
            726,
            -456,
            -57,
            376,
            377,
            93,
            710,
            709,
            117,
            -1,
            -707,
            -381,
            721,
            -828,
            -345,
            202,
            -473,

            952,
            535,
            534,
            -957,
            -144,
            363,
            -169,
            -853,
            -780,
            -383,
            -384,
            -385,
            -249,
            -250,
            -251,
            548,
            164,
            665,
            690,
            588,
            272,
            -448,
    };

    /*
    Random doubles over the whole range
     */
    private static void testRandom(int randomCount, Random r) {
        for (int i = 0; i < randomCount; ++i) {
            toDec(longBitsToDouble(r.nextLong()));
        }
    }

    /*
    Random doubles over the integer range [0, 2^52).
    These are all exact doubles and exercise the fast path (except 0).
     */
    private static void testRandomUnit(int randomCount, Random r) {
        for (int i = 0; i < randomCount; ++i) {
            toDec(r.nextLong() & (1L << P - 1));
        }
    }

    /*
    Random doubles over the range [0, 10^15) as "multiples" of 1e-3
     */
    private static void testRandomMilli(int randomCount, Random r) {
        for (int i = 0; i < randomCount; ++i) {
            toDec(r.nextLong() % 1_000_000_000_000_000_000L / 1e3);
        }
    }

    /*
    Random doubles over the range [0, 10^15) as "multiples" of 1e-6
     */
    private static void testRandomMicro(int randomCount, Random r) {
        for (int i = 0; i < randomCount; ++i) {
            toDec((r.nextLong() & 0x7FFF_FFFF_FFFF_FFFFL) / 1e6);
        }
    }

    static void randomNumberTests(int randomCount, Random r) {
        testRandom(randomCount, r);
        testRandomUnit(randomCount, r);
        testRandomMilli(randomCount, r);
        testRandomMicro(randomCount, r);
    }

    /*
     * The following 19'545 doubles were kindly provided by
     * Paul Zimmermann of INRIA
     * and are used here with permission.
     *
     * They were generated on purpose as hard test cases
     * and were obtained from the continued fraction of 2^q*10^(-k),
     * with k as in the Schubfach paper.
     *
     * The values are split into different methods to overcome a JVM .class
     * limit of 64 KiB for the bytecode of any method, including <clinit>.
     */
    static double[] hard0() {
        return new double[] {
                0x1.00003eaba12cap-804,
                0x1.00003eaba12cap-805,
                0x1.00003eaba12cap-806,
                0x1.000617d435ddap426,
                0x1.000626f884978p641,
                0x1.00068e6f99769p-559,
                0x1.00085e2dc7963p-775,
                0x1.000ff55a6p-1038,
                0x1.0016956dfca36p295,
                0x1.0016956dfca36p+296,
                0x1.00171dfbe1d2fp-881,
                0x1.001b7fdbb5c3bp-707,
                0x1.002174594573dp-733,
                0x1.003d8e0b200cep+280,
                0x1.004052146aed2p-472,
                0x1.004052146aed2p-473,
                0x1.004351fdb5d5p171,
                0x1.00444244d7cabp485,
                0x1.00444244d7cacp485,
                0x1.0045a0a6e58e8p+532,
                0x1.0045a0a6e58e8p533,
                0x1.0045a0a6e58e8p534,
                0x1.0045a0a6e58e8p535,
                0x1.0045af4fa2278p578,
                0x1.0045af4fa2278p579,
                0x1.004a2e06fe26ep841,
                0x1.00591ff7983b6p-458,
                0x1.00591ff7983b6p-459,
                0x1.00591ff7983b6p-460,
                0x1.00591ff7983b6p-461,
                0x1.00591ff7983b6p-462,
                0x1.005e1a42f5e92p979,
                0x1.005e1a42f5e92p+980,
                0x1.005f01458a534p-392,
                0x1.005f24c16923cp+600,
                0x1.005f24c16923cp601,
                0x1.005f24c16923cp602,
                0x1.005f24c16923cp603,
                0x1.00601aab28c7ep785,
                0x1.00683a21de855p-150,
                0x1.00683a21de855p-151,
                0x1.00683a21de855p-152,
                0x1.0068e4e484464p-56,
                0x1.006b5a40d1c46p-582,
                0x1.006b5a40d1c46p-583,
                0x1.006b5a40d1c46p-584,
                0x1.006de725c61ap-343,
                0x1.0073bceb2f3aep302,
                0x1.0073c0d74c096p-1008,
                0x1.0073c0d74c096p-1009,
                0x1.0073c0d74c096p-1010,
                0x1.007575c3d9p-1034,
                0x1.0079b37caa968p-469,
                0x1.007e00a23b6a8p-478,
                0x1.00804197592a6p335,
                0x1.00867155f6e2fp-242,
                0x1.008896bcf54fap969,
                0x1.008896bcf54fap970,
                0x1.0089d20bc457ep-155,
                0x1.00904842a1a5ep-185,
                0x1.0091238bec71ep493,
                0x1.009655c795954p167,
                0x1.0098ac5d7659ap-693,
                0x1.0098ac5d7659ap-694,
                0x1.0098ac5d7659ap-695,
                0x1.009e0f5f18863p+940,
                0x1.009e7d474572ap109,
                0x1.00a12168f6b1ap-570,
                0x1.00a19a3ffd982p-160,
                0x1.00a19a3ffd982p-161,
                0x1.00a19a3ffd982p-162,
                0x1.00a19a3ffd982p-163,
                0x1.00a19a3ffd982p-164,
                0x1.00a19a3ffd982p-165,
                0x1.00a19a3ffd982p-166,
                0x1.00a19a3ffd982p-167,
                0x1.00a19a3ffd982p-168,
                0x1.00a19a3ffd982p-169,
                0x1.00a572e7842b4p981,
                0x1.00a8130b6beb1p589,
                0x1.00ad7e5a1a383p-156,
                0x1.00b1d36707b02p-694,
                0x1.00b1d36707b02p-695,
                0x1.00b1f2d1d7f44p526,
                0x1.00b576fbfbebap-222,
                0x1.00b576fbfbebap-223,
                0x1.00b576fbfbebap-224,
                0x1.00b576fbfbebap-225,
                0x1.00bp-1062,
                0x1.00cc8129ba75cp-604,
                0x1.00d7b2e28c65cp-681,
                0x1.00d7b2e28c65cp-682,
                0x1.00d7b2e28c65cp-683,
                0x1.00d7e8a7c6814p-973,
                0x1.00d7e8a7c6814p-974,
                0x1.00d88df79e596p-37,
                0x1.00d8ec639913ep-677,
                0x1.00d8ec639913ep-678,
                0x1.00d904cfeaa7p-602,
                0x1.00d904cfeaa7p-603,
                0x1.00dc674f8146ep373,
                0x1.00dc674f8146ep374,
                0x1.00dee745d8d1ap-78,
                0x1.00df313bbb0bap-346,
                0x1.00e36f5a455bcp-1005,
                0x1.00e4879c182b4p951,
                0x1.00e7db3b3f242p-180,
                0x1.00e7db3b3f242p-181,
                0x1.00e7db3b3f242p-182,
                0x1.00e9133f376f6p658,
                0x1.00e9e81ca036dp-243,
                0x1.00edde7d381ddp-142,
                0x1.00f2abb60d8ddp-630,
                0x1.00f4e76160e2ep-640,
                0x1.00f4e76160e2ep-641,
                0x1.00f567652a292p-208,
                0x1.00f663c660dacp-896,
                0x1.00f95f0691d7p-184,
                0x1.00f9dcef38067p-410,
                0x1.00f9dcef38067p-411,
                0x1.00fd149576383p+592,
                0x1.0100da7d9ea1dp-546,
                0x1.010169a08e0b2p-462,
                0x1.010169a08e0b2p-463,
                0x1.01028c751bbd2p-777,
                0x1.01080b8617498p+756,
                0x1.010ae5d5226eap+476,
                0x1.011c2eaabe7d8p-196,
                0x1.011c669d373cap-187,
                0x1.011f2d73116f4p539,
                0x1.0127955a7b5ecp594,
                0x1.01305db9ec12p-622,
                0x1.01305db9ec12p-623,
                0x1.01305db9ec12p-624,
                0x1.01314be1645fdp694,
                0x1.01314be1645fdp695,
                0x1.01314be1645fdp+696,
                0x1.01325cc41cf5ap738,
                0x1.01325cc41cf5ap739,
                0x1.0141c6cf446a8p-466,
                0x1.01422850451a3p-716,
                0x1.01484b7153aep-979,
                0x1.014a3b5674cd6p873,
                0x1.015156db67819p+220,
                0x1.01542p-1054,
                0x1.01542p-1055,
                0x1.01586f9850ebp+384,
                0x1.015b0246dff08p571,
                0x1.0160bcb58c16cp289,
                0x1.01635e073d1a8p+676,
                0x1.01700b6a974dap-899,
                0x1.0174103a14p-1036,
                0x1.018341c1f40aap685,
                0x1.01891fc4717fdp-485,
                0x1.01891fc4717fdp-486,
                0x1.01891fc4717fdp-487,
                0x1.0189a26df575fp+700,
                0x1.0189a26df575fp701,
                0x1.0189a26df575fp702,
                0x1.0189a26df575fp703,
                0x1.018d081aee1d8p-102,
                0x1.018ea74578544p487,
                0x1.01931f4607cd9p-124,
                0x1.019b74d2172bap821,
                0x1.01a103abb2e7cp-48,
                0x1.01a216bfffb34p-1005,
                0x1.01a216bfffb34p-1006,
                0x1.01a512a3d335ep234,
                0x1.01a55d07d39cfp774,
                0x1.01a7454b57d6ep170,
                0x1.01ae0b01bb6aap-692,
                0x1.01b3352e8793p-936,
                0x1.01b3352e8793p-937,
                0x1.01b8a3f1f6f8fp619,
                0x1.01b9365e9db56p342,
                0x1.01b94c2bec388p-584,
                0x1.01bc7762a0a8p-1029,
                0x1.01c27581dd914p837,
                0x1.01c7b1b493039p-63,
                0x1.01cea42f6503ap-708,
                0x1.01cfd3f2d416bp-400,
                0x1.01d3cd544c2b4p-1024,
                0x1.01d75948a78bcp-482,
                0x1.01e25b4f8ce5cp-338,
                0x1.01e54a87eaad8p-877,
                0x1.01e74952ee74ep+148,
                0x1.01e921453dc34p150,
                0x1.01e921453dc34p151,
                0x1.01ee1334a32c6p623,
                0x1.01f0490d0fb57p213,
                0x1.01f4d0ff1039p-877,
                0x1.01f4d0ff1039p-878,
                0x1.01f96b8bce4dep-555,
                0x1.01faac0a20762p+476,
                0x1.01fc1933951f9p591,
                0x1.01fc1933951f9p+592,
                0x1.01fc1a33e1013p-250,
                0x1.01fc57a9c7822p-542,
                0x1.020149d6b47aep193,
                0x1.020149d6b47aep194,
                0x1.0207cad683bbep254,
                0x1.0207cad683bbep255,
                0x1.020d4eb5c15aep730,
                0x1.02141417e860ep-730,
                0x1.0216be5ec380ep181,
                0x1.0216be5ec380ep182,
                0x1.021d5b74cdbfdp666,
                0x1.021f14ed7b3fap293,
                0x1.021f14ed7b3fap294,
                0x1.021f14ed7b3fap295,
                0x1.02230390cea48p891,
                0x1.022613713bbfep-1018,
                0x1.0234a352626a2p-190,
                0x1.023998cd10537p-392,
                0x1.023b10ec8184ep-916,
                0x1.023c31461359fp945,
                0x1.02433b8a1de7ep-466,
                0x1.02488e401c4f2p-865,
                0x1.02498ea6df0c4p+144,
                0x1.02498ea6df0c4p145,
                0x1.02498ea6df0c4p146,
                0x1.0249dfa32be9ap413,
                0x1.024b815d4db06p+128,
                0x1.024eb59839cp-386,
                0x1.024eb59839cp-387,
                0x1.024eb59839cp-388,
                0x1.02502cdc147a4p-315,
                0x1.025085b33027bp470,
                0x1.02510a0b202bbp-52,
                0x1.0255efac502f3p634,
                0x1.025774a630305p-62,
                0x1.025a29555f9p-739,
                0x1.025a30f20789ep810,
                0x1.025a8cc77c8a2p+876,
                0x1.025a8cc77c8a2p877,
                0x1.025a8cc77c8a2p878,
                0x1.02696aa6b1ef1p-755,
                0x1.027269a067068p283,
                0x1.027269a067068p+284,
                0x1.0275be6f8dc32p586,
                0x1.027e72f1f1281p93,
                0x1.027f52909935p-341,
                0x1.027f52909935p-342,
                0x1.028127630b608p-1008,
                0x1.028127630b608p-1009,
                0x1.0288a6b326f66p-160,
                0x1.028917d937f7ep-448,
                0x1.028917d937f7ep-449,
                0x1.028917d937f7ep-450,
                0x1.028917d937f7ep-451,
                0x1.028917d937f7ep-452,
                0x1.028bc1fc8eeeap321,
                0x1.028c676ab7b8ap705,
                0x1.028c676ab7b8ap706,
                0x1.028p-1065,
                0x1.02963bcd28d44p-974,
                0x1.02a051d714c2ep-406,
                0x1.02a6b2dce4944p-505,
                0x1.02a6b2dce4944p-506,
                0x1.02a967922569p459,
                0x1.02a9ae7e66e24p-739,
                0x1.02a9ea467fb42p-435,
                0x1.02a9ea467fb42p-436,
                0x1.02a9ea467fb42p-437,
                0x1.02ab3170c2274p+660,
                0x1.02adcad689ba8p-551,
                0x1.02ae1cep-1047,
                0x1.02bad5d0865bbp-363,
                0x1.02bd4e3a4e1a2p-953,
                0x1.02bd8b1ffa014p-531,
                0x1.02bd8b1ffa014p-532,
                0x1.02bd8b1ffa014p-533,
                0x1.02bf0e816482dp-926,
                0x1.02c35f729689ep578,
                0x1.02c4203793ddap707,
                0x1.02c5ap-1052,
                0x1.02c6362022cf2p-338,
                0x1.02c80269eddeap283,
                0x1.02caec241129ap913,
                0x1.02caec241129ap914,
                0x1.02cbd1567218cp-642,
                0x1.02cbd1567218cp-643,
                0x1.02cc60cdef1acp-888,
                0x1.02d0c3d2663c4p-218,
                0x1.02d0c3d2663c4p-219,
                0x1.02d0c3d2663c4p-220,
                0x1.02d0c3d2663c4p-221,
                0x1.02d38a682d671p+460,
                0x1.02db8079a1b02p-609,
                0x1.02ddfe110f66p673,
                0x1.02e09f9de65a8p203,
                0x1.02eab645fde46p662,
                0x1.02eab645fde46p663,
                0x1.02ebaf189d5f2p867,
                0x1.02ebaf189d5f2p+868,
                0x1.02ebaf189d5f2p869,
                0x1.02ec537974571p-982,
                0x1.02ec537974571p-983,
                0x1.02eed0e147a1bp1015,
                0x1.02faeaa4ca5p-814,
                0x1.02ff8fc076acep857,
                0x1.030106c8d72b1p353,
                0x1.0304c23f123c4p-863,
                0x1.030aa463886ecp-854,
                0x1.030b291a619c6p+312,
                0x1.030b291a619c6p313,
                0x1.030b291a619c6p314,
                0x1.030b291a619c6p315,
                0x1.030b291a619c6p+316,
                0x1.030e9728p-1045,
                0x1.031315c7cabccp925,
                0x1.0321281d1cb03p-461,
                0x1.0327894c6b121p421,
                0x1.0327894c6b121p422,
                0x1.032a1a72ba0eap137,
                0x1.032c3ffac5ea3p-120,
                0x1.03348b9a6344ap911,
                0x1.033b429919e23p-887,
                0x1.033b45p-1048,
                0x1.033b45p-1049,
                0x1.033b45p-1050,
                0x1.033c53f9c02d4p-711,
                0x1.033c53f9c02d4p-712,
                0x1.033d7eca0adeep155,
                0x1.033d7eca0adeep+156,
                0x1.033d7eca0adeep157,
                0x1.033d7eca0adeep158,
                0x1.033d7eca0adeep159,
                0x1.033d7eca0adeep+160,
                0x1.033d7eca0adeep161,
                0x1.033d7eca0adeep162,
                0x1.033d7eca0adeep163,
                0x1.033d7eca0adeep+164,
                0x1.033d7eca0adeep165,
                0x1.0344105b846a8p-541,
                0x1.03471364434cbp-216,
                0x1.03583fc527ab3p-588,
                0x1.035adb1fca12p749,
                0x1.035e7b5183922p-535,
                0x1.035e7b5183922p-536,
                0x1.035e7b5183922p-537,
                0x1.035e7b5183922p-538,
                0x1.035e7b5183922p-539,
                0x1.035e7b5183922p-540,
                0x1.0363fc817c0c1p-833,
                0x1.0363fc817c0c1p-834,
                0x1.036ade388446ep874,
                0x1.036ade388446ep875,
                0x1.0371a609147dap-992,
                0x1.0371bd506b1p-356,
                0x1.0372db8ca4eb6p-847,
                0x1.037590c4b1b7dp-963,
                0x1.03771e36c0d04p-117,
                0x1.037d447715b54p-387,
                0x1.037f272d17fecp+928,
                0x1.0381889cfea2cp-985,
                0x1.038918f77011ap326,
                0x1.038b267642c08p717,
                0x1.038b9f8d4e39ep-815,
                0x1.038b9f8d4e39ep-816,
                0x1.038e4348ac1dfp-960,
                0x1.038e5a6f898bap233,
                0x1.039c7c92726e5p553,
                0x1.039d66589688p-102,
                0x1.039d66589688p-103,
                0x1.03a58276bc3a2p286,
                0x1.03a58276bc3a2p287,
                0x1.03aad6010e566p-901,
                0x1.03aad6010e566p-902,
                0x1.03b032a94171p845,
                0x1.03b151acc10cfp-465,
                0x1.03b301b05c96p-253,
                0x1.03b4e7c172b58p113,
                0x1.03b50a7957856p958,
                0x1.03b50a7957856p959,
                0x1.03b50a7957856p+960,
                0x1.03b50a7957856p961,
                0x1.03b77d2b969dap-644,
                0x1.03b77d2b969dap-645,
                0x1.03b77d2b969dap-646,
                0x1.03b77d2b969dap-647,
                0x1.03b77d2b969dap-648,
                0x1.03b77d2b969dap-649,
                0x1.03bb086fb444ap-15,
                0x1.03bddf5cbbee1p-571,
                0x1.03bddf5cbbee1p-572,
                0x1.03c4d29f23d67p-621,
                0x1.03c96d5c3c15fp-66,
                0x1.03cafd2p-1047,
                0x1.03cba409a2bbcp+952,
                0x1.03cba409a2bbcp953,
                0x1.03cea3586452ep-795,
                0x1.03cea3586452ep-796,
                0x1.03cea3586452ep-797,
                0x1.03cea3586452ep-798,
                0x1.03cea3586452ep-799,
                0x1.03cea3586452ep-800,
                0x1.03cea3586452ep-801,
                0x1.03cea3586452ep-802,
                0x1.03cea3586452ep-803,
                0x1.03d7cb98edc58p866,
                0x1.03d7cb98edc58p867,
                0x1.03d7cb98edc58p+868,
                0x1.03d7cb98edc58p869,
                0x1.03d7cb98edc58p870,
                0x1.03d7cb98edc58p871,
                0x1.03d7cb98edc58p+872,
                0x1.03d7cb98edc58p873,
                0x1.03d7cb98edc58p874,
                0x1.03d8184p-1048,
                0x1.03df4bd7292f6p655,
                0x1.03e27ebc2931ap-360,
                0x1.03e27ebc2931ap-361,
                0x1.03e27ebc2931ap-362,
                0x1.03e27ebc2931ap-363,
                0x1.03e27ebc2931ap-364,
                0x1.03e29f5c2b18cp382,
                0x1.03eb8baf6572ap-333,
                0x1.03ee7a10ba90cp-984,
                0x1.03ef25a107692p-113,
                0x1.03ef5bb464346p-935,
                0x1.03f83a0df0abap355,
                0x1.03f83a0df0abap+356,
                0x1.03f877795be78p+1020,
                0x1.03f877795be78p1021,
                0x1.04054bb4f2156p-615,
                0x1.04057944d176p-900,
                0x1.0405c84ccf4aap175,
                0x1.04117c040bdep185,
                0x1.04138b5e17dcep+264,
                0x1.04139cc0a2b38p+208,
                0x1.0413d2476f054p-246,
                0x1.0418af6146fdep-1014,
                0x1.0419cdefbfc42p-482,
                0x1.0419cdefbfc42p-483,
                0x1.041a89dd8f00cp814,
                0x1.042062e46a678p+216,
                0x1.0424e536d82e8p-935,
                0x1.04273b18918bp-793,
                0x1.04273b18918bp-794,
                0x1.04273b18918bp-795,
                0x1.04273b18918bp-796,
                0x1.04273b18918bp-797,
                0x1.04273b18918bp-798,
                0x1.04273b18918bp-799,
                0x1.04273b18918bp-800,
                0x1.0427ead4cfed6p867,
                0x1.0429cb8f8ddb8p-779,
                0x1.0429cb8f8ddb8p-780,
                0x1.0429cb8f8ddb8p-781,
                0x1.0429cb8f8ddb8p-782,
                0x1.0429cb8f8ddb8p-783,
                0x1.042a34ac61664p574,
                0x1.0430c2d075c08p-804,
                0x1.0430c2d075c08p-805,
                0x1.0430c2d075c08p-806,
                0x1.0430c2d075c08p-807,
                0x1.0430c2d075c08p-808,
                0x1.0430c2d075c08p-809,
                0x1.0430c2d075c08p-810,
                0x1.0430c2d075c08p-811,
                0x1.0430c2d075c08p-812,
                0x1.0430c2d075c08p-813,
                0x1.0436f6f632d9cp-225,
                0x1.0445534deb29ep+492,
                0x1.04466796b2cc8p-25,
                0x1.0447ad1ee1d03p-36,
                0x1.0448f426d0dcp-752,
                0x1.044ce18a22ccbp-678,
                0x1.044e943b92568p-58,
                0x1.044f3324fcd15p921,
                0x1.045e2b89b065ep586,
                0x1.04606019b7edcp385,
                0x1.0461d16ee750cp-590,
                0x1.0465e9d2cec8ep-339,
                0x1.046b52b88p-1041,
                0x1.0471fc733cb3cp518,
                0x1.047475d761e06p+724,
                0x1.047475d761e06p725,
                0x1.047475d761e06p726,
                0x1.047475d761e06p727,
                0x1.047824f2bb6dap-784,
                0x1.04796d0ea4171p-869,
                0x1.047ac3323007fp+552,
                0x1.047ac3323007fp553,
                0x1.047c6881a00dbp-379,
                0x1.047df126a59ccp799,
                0x1.047df126a59ccp+800,
                0x1.047df126a59ccp801,
                0x1.047efe362a376p-715,
                0x1.047efe362a376p-716,
                0x1.047efe362a376p-717,
                0x1.0488f57782279p-801,
                0x1.048cb468bc208p-691,
                0x1.048cb468bc208p-692,
                0x1.048cb468bc208p-693,
                0x1.048cb468bc208p-694,
                0x1.048cb468bc208p-695,
                0x1.048cb468bc208p-696,
                0x1.048dda6d2bae5p+584,
                0x1.0492a4a8a37fdp-699,
                0x1.0492a4a8a37fdp-700,
                0x1.0492a4a8a37fdp-701,
                0x1.0492a4a8a37fdp-702,
                0x1.0492abe88bdap898,
                0x1.0493f056e9ef3p-1018,
                0x1.0493f056e9ef3p-1019,
                0x1.0493f056e9ef3p-1020,
                0x1.0496bdb90c31cp955,
                0x1.049a044ac2708p-837,
                0x1.049c68dfac75p-547,
                0x1.049c68dfac75p-548,
                0x1.049c68dfac75p-549,
                0x1.049c68dfac75p-550,
                0x1.049c68dfac75p-551,
                0x1.049c68dfac75p-552,
                0x1.049c68dfac75p-553,
                0x1.049d522795f68p-228,
                0x1.04a616046e074p-369,
                0x1.04a616046e074p-370,
                0x1.04a616046e074p-371,
                0x1.04a616046e074p-372,
                0x1.04a616046e074p-373,
                0x1.04a81d35952ffp-947,
                0x1.04a81d35952ffp-948,
                0x1.04a81d35952ffp-949,
                0x1.04aa33f5741aep-876,
                0x1.04aa33f5741aep-877,
                0x1.04aa33f5741aep-878,
                0x1.04aac642d0b24p+836,
                0x1.04aac642d0b24p837,
                0x1.04b942fbf81c8p-772,
                0x1.04b98ab4ffd62p393,
                0x1.04bb945117b42p-964,
                0x1.04bbd35d3151ap-421,
                0x1.04bd984990e6fp-299,
                0x1.04bd984990e6fp-300,
                0x1.04c2f8ad9f409p+180,
                0x1.04cb77d0138fep770,
                0x1.04cea72108cf2p-392,
                0x1.04d2b347eabfep-305,
                0x1.04d85bddfb4acp-1024,
                0x1.04dc5d8167a57p-274,
                0x1.04dc92e64b5d2p+580,
                0x1.04dc92e64b5d2p581,
                0x1.04dcb406211b8p-693,
                0x1.04e825259118ep-113,
                0x1.04eb8ab1543b1p-377,
                0x1.04ebfa4bd41adp397,
                0x1.04f0a0813f705p297,
                0x1.04f3d254df3d8p789,
                0x1.05021e11702d4p1017,
                0x1.05021e11702d4p1018,
                0x1.05031e2503da8p186,
                0x1.05031e2503da8p187,
                0x1.05079bc6a9b8ep+608,
                0x1.050b5928358b8p-922,
                0x1.05155f3539915p-592,
                0x1.0523643a025ccp+448,
                0x1.0525b81f7f742p454,
                0x1.0525b81f7f742p455,
                0x1.0525b81f7f742p+456,
                0x1.0525b81f7f742p457,
                0x1.05265011cbcc4p833,
                0x1.05265011cbcc4p834,
                0x1.05265011cbcc4p835,
                0x1.05265011cbcc4p+836,
                0x1.052c30c0b487dp341,
                0x1.0532934495b9cp+672,
                0x1.0539c46b451aep+1000,
                0x1.053ap-1059,
                0x1.05439b6669e36p-105,
                0x1.0543e9a7b2626p-436,
                0x1.0543e9a7b2626p-437,
                0x1.0543e9a7b2626p-438,
                0x1.0543e9a7b2626p-439,
                0x1.05445eec8d1b8p198,
                0x1.0545def3e125ep+404,
                0x1.0547529a85bc1p+168,
                0x1.0548b68a044d6p670,
                0x1.0548b68a044d6p671,
                0x1.0548b68a044d6p+672,
                0x1.0548b68a044d6p673,
                0x1.05558a03b3a47p1018,
                0x1.05563c4ffe222p-128,
                0x1.05572a2a68f92p-508,
                0x1.0557dd789805bp-320,
                0x1.0562e2a2c8649p-168,
                0x1.0562e2a2c8649p-169,
                0x1.05666c527ba14p711,
                0x1.05666c527ba14p+712,
                0x1.05666c527ba14p713,
                0x1.0570faa313e47p-886,
                0x1.0571992bf430cp387,
                0x1.057670fff6246p-761,
                0x1.057e84ee48773p-636,
                0x1.058649a08dffap-723,
                0x1.0586e956b3a5ep-105,
                0x1.058b0b33deb25p-140,
                0x1.058b811a93c04p+872,
                0x1.058b811a93c04p873,
                0x1.0594194b591b1p-857,
                0x1.059949b708f28p-980,
                0x1.059d2c1c4c808p-773,
                0x1.059d9e0059db2p-312,
                0x1.059d9e0059db2p-313,
                0x1.059d9e0059db2p-314,
                0x1.059d9e0059db2p-315,
                0x1.059d9e0059db2p-316,
                0x1.05a28db5562adp527,
                0x1.05a89ac04014p-1028,
                0x1.05b2a61b14befp222,
                0x1.05b9c5e6d32e7p-795,
                0x1.05bea4eb35c82p-636,
                0x1.05bf5fc34804cp946,
                0x1.05cb22e9e0c38p770,
                0x1.05d4c20b6ff76p659,
                0x1.05d4c20b6ff76p+660,
                0x1.05d807e05c941p762,
                0x1.05dba0541c4bep-112,
                0x1.05df0a267bcc9p-495,
                0x1.05e705c196079p-668,
                0x1.05ef48771f5e6p+796,
                0x1.05ef48771f5e6p797,
                0x1.05f2a22ccp-1037,
                0x1.05f2a22ccp-1038,
                0x1.05f43c01a994ap-958,
                0x1.05f43c01a994ap-959,
                0x1.05f88e3aa6604p219,
                0x1.05fe63c38fc1p725,
                0x1.05fe63c38fc1p726,
                0x1.05fe63c38fc1p727,
                0x1.0601712a7128ap-863,
                0x1.0608b9f466228p-892,
                0x1.0611b4f97d256p-274,
                0x1.061bcfa1335fp-455,
                0x1.061bcfa1335fp-456,
                0x1.061d138f2986fp-39,
                0x1.062445e7ba5a2p-430,
                0x1.062445e7ba5a2p-431,
                0x1.062445e7ba5a2p-432,
                0x1.062445e7ba5a2p-433,
                0x1.062445e7ba5a2p-434,
                0x1.062445e7ba5a2p-435,
                0x1.062445e7ba5a2p-436,
                0x1.0624dd2f1a9fcp-10,
                0x1.06250a94856aep+880,
                0x1.06250a94856aep881,
                0x1.06272d20bp-1038,
                0x1.06273e03ffc1cp-251,
                0x1.062782a310e1ap521,
                0x1.06326a8bbcf6cp-82,
                0x1.06333c496d5fbp311,
                0x1.0636a63eaa16ap186,
                0x1.0636a63eaa16ap187,
                0x1.063c0e825a94cp969,
                0x1.06497ae765322p342,
                0x1.064cbd2e8b0b8p437,
                0x1.064eb061df7f2p-153,
                0x1.0651011b48546p835,
                0x1.0653c5c249c86p923,
                0x1.0653c5c249c86p+924,
                0x1.065947845b9aep-627,
                0x1.0665dc4b9fab2p131,
                0x1.066a8eaba3a86p845,
                0x1.066a8eaba3a86p846,
                0x1.066a8eaba3a86p847,
                0x1.066a8eaba3a86p+848,
                0x1.066ac2d5daec4p475,
                0x1.0672c67fac1f1p-287,
                0x1.0673f5a99c07bp945,
                0x1.06756dfde8824p-284,
                0x1.0678579eb370ap-324,
                0x1.067b94e27ad68p567,
                0x1.0680f42d75b45p-1016,
                0x1.0681e94a7242ap-849,
                0x1.0681e94a7242ap-850,
                0x1.0684ec3219cbap361,
                0x1.0684ec3219cbap362,
                0x1.0685c70ea3298p-295,
                0x1.06889999671f6p515,
                0x1.068cb90fded28p-785,
                0x1.068cb90fded28p-786,
                0x1.068cb90fded28p-787,
                0x1.068cb90fded28p-788,
                0x1.068cb90fded28p-789,
                0x1.069672b9f176ep1022,
                0x1.069672b9f176ep1023,
                0x1.06a094fdc7254p-429,
                0x1.06a094fdc7254p-430,
                0x1.06a573469374ep831,
                0x1.06a5ebf77349p-124,
                0x1.06a68313fc5bfp-955,
                0x1.06b0bb1fb384cp+960,
                0x1.06b8b768a69cap-716,
                0x1.06bb2b40f85ep623,
                0x1.06bb2b40f85ep+624,
                0x1.06bd46f57005p429,
                0x1.06be538795656p161,
                0x1.06c0eaa65b554p783,
                0x1.06c0eaa65b554p+784,
                0x1.06c0eaa65b554p785,
                0x1.06c0eaa65b554p786,
                0x1.06c3e3b95cc33p290,
                0x1.06c3f03862bdep526,
                0x1.06c3f03862bdep527,
                0x1.06c4182fea9ccp-820,
                0x1.06c776866677ap-377,
                0x1.06c776866677ap-378,
                0x1.06c776866677ap-379,
                0x1.06c776866677ap-380,
                0x1.06c776866677ap-381,
                0x1.06c91eebc31f4p-938,
                0x1.06c91eebc31f4p-939,
                0x1.06ce99d8050dbp+1000,
                0x1.06ce99d8050dbp999,
                0x1.06d683a2771f4p339,
                0x1.06d7c18521296p375,
                0x1.06d7c18521296p+376,
                0x1.06dd49479884ap-932,
                0x1.06dd49479884ap-933,
                0x1.06dd49479884ap-934,
                0x1.06dd5700aa5p253,
                0x1.06e2cp-1056,
                0x1.06e98f5ec8f38p405,
                0x1.06e98f5ec8f38p406,
                0x1.06e98f5ec8f38p407,
                0x1.06e98f5ec8f38p+408,
                0x1.06e98f5ec8f38p409,
                0x1.06e98f5ec8f38p410,
                0x1.06ef4147b8a9ep-109,
                0x1.06ef4147b8a9ep-110,
                0x1.06ef4147b8a9ep-111,
                0x1.06ef4147b8a9ep-112,
                0x1.06ef4147b8a9ep-113,
                0x1.06f0b3ff6c2b4p-459,
                0x1.06f0b3ff6c2b4p-460,
                0x1.06f8798a7457ap461,
                0x1.06f8bd3c51ee6p-114,
                0x1.06f8bd3c51ee6p-115,
                0x1.06f9bf65p-1042,
                0x1.06p-1067,
                0x1.0700630c75267p-65,
                0x1.070118p-1051,
                0x1.0701bd527b498p-688,
                0x1.0701bd527b498p-689,
                0x1.0701bd527b498p-690,
                0x1.0701bd527b498p-691,
                0x1.0701bd527b498p-692,
                0x1.070388dfae0ccp-1014,
                0x1.0703ee6df2d6ep-925,
                0x1.070462197efa2p646,
                0x1.070462197efa2p647,
                0x1.070462197efa2p+648,
                0x1.070462197efa2p649,
                0x1.07095d408119ep939,
                0x1.070b15bfbaeep-70,
                0x1.071113aa823b6p-388,
                0x1.0711fed5b19a4p-153,
                0x1.0711fed5b19a4p-154,
                0x1.0716b7a1e2518p919,
                0x1.0716de7ada994p-625,
                0x1.071d384p-1047,
                0x1.071d384p-1048,
                0x1.071ef5166497p777,
                0x1.07270a4b06bfbp-97,
                0x1.072804b19ff8p582,
                0x1.0728ef5edc36ap357,
                0x1.072f3819c1321p911,
                0x1.0730417dadc44p277,
                0x1.07335b58p-1043,
                0x1.07335b58p-1044,
                0x1.0733bfeed34ebp347,
                0x1.07477419c873dp+924,
                0x1.0747ddddf22a8p-206,
                0x1.0748be80a4224p354,
                0x1.074f8c4d3cd7bp129,
                0x1.0752275b0f2fcp-154,
                0x1.0752b3ab5f47ap719,
                0x1.0753fc345e4p-1032,
                0x1.07557e183cd52p-1005,
                0x1.0756147e662a4p406,
                0x1.075823b6c498ap+332,
                0x1.075a9052e0e1dp-204,
                0x1.07684982f634fp119,
                0x1.0768bf91ae844p+844,
                0x1.076cd135cf623p289,
                0x1.07757fac58b8dp-63,
                0x1.077a8p-1057,
                0x1.077aa33e13f82p218,
                0x1.077ae28459ae8p-974,
                0x1.0782189b336aep567,
                0x1.0782189b336aep+568,
                0x1.0782189b336aep569,
                0x1.0782189b336aep570,
                0x1.0782189b336aep571,
                0x1.0787d2dae2fe5p873,
                0x1.07887656840f2p713,
                0x1.07887656840f2p714,
                0x1.078a6b3e8deecp-900,
                0x1.078bac98c76cep-211,
                0x1.078e111c3556cp279,
                0x1.079bb4f896766p-520,
                0x1.079ef227a7af8p118,
                0x1.079ef227a7af8p119,
                0x1.07aac4974f504p255,
                0x1.07ac23fca339cp875,
                0x1.07ad7bd288c83p238,
                0x1.07b540bc422bp399,
                0x1.07b5f2f3147e9p-124,
                0x1.07ba001f38cb7p-272,
                0x1.07ba76b76a4e8p-354,
                0x1.07ba97c65f8fap+588,
                0x1.07bb8914a047ap-947,
                0x1.07bb8914a047ap-948,
                0x1.07c2d27a5b989p-306,
                0x1.07c2d27a5b989p-307,
                0x1.07c578576cf8ep331,
                0x1.07c578576cf8ep+332,
                0x1.07cb489bf5b1p-545,
                0x1.07ce30dc1d288p438,
                0x1.07d08f6742a34p-477,
                0x1.07d457124123cp+764,
                0x1.07d457124123cp765,
                0x1.07d457124123cp766,
                0x1.07d9b380bae0cp+428,
                0x1.07d9b380bae0cp429,
                0x1.07d9b380bae0cp430,
                0x1.07d9d0341d7ep+320,
                0x1.07daffd0797eep-407,
                0x1.07e1fe91b0b7p-35,
                0x1.07ee6b237a489p686,
                0x1.07f256f319187p+804,
                0x1.07ff684e4c2a2p666,
                0x1.080055cf8d00bp-496,
                0x1.08046619bcbbbp313,
                0x1.080682eaff3d1p129,
                0x1.0806ecd817346p+144,
                0x1.080d763b36932p883,
                0x1.080d763b36932p+884,
                0x1.080d763b36932p885,
                0x1.0811a0d254548p655,
                0x1.0811a0d254548p+656,
                0x1.0811a0d254548p657,
                0x1.0811a0d254548p658,
                0x1.0815612bfcb62p+116,
                0x1.08194b6fd600bp794,
                0x1.0819bcf94d72ep-542,
                0x1.081c316fe21f1p-984,
                0x1.081e47ad9d4b8p-236,
                0x1.081f4621192bcp-426,
                0x1.08217425512c4p138,
                0x1.08217425512c4p139,
                0x1.08217425512c4p+140,
                0x1.08217425512c4p141,
                0x1.08217425512c4p142,
                0x1.0825b3323dabp-887,
                0x1.0825b3323dabp-888,
                0x1.08285ae88982ep450,
                0x1.082c635d4af63p+176,
                0x1.0831eb2616p-1035,
                0x1.08343e9cd58b6p-408,
                0x1.083921a0455cp+244,
                0x1.0839fc19dd47ep-815,
                0x1.083a0f634d3bep-678,
                0x1.083a0f634d3bep-679,
                0x1.08474fb7dee51p-442,
                0x1.0847822f765b2p-255,
                0x1.0847822f765b2p-256,
                0x1.0847822f765b2p-257,
                0x1.0847822f765b2p-258,
                0x1.0847822f765b2p-259,
                0x1.0847822f765b2p-260,
                0x1.0847822f765b2p-261,
                0x1.0847822f765b2p-262,
                0x1.0848e7313e2e6p-681,
                0x1.084a949ad8612p-657,
                0x1.084da19e1ea6ap-538,
                0x1.085732f63c44fp-889,
                0x1.08621c4199208p-174,
                0x1.08621c4199208p-175,
                0x1.086c219697e2cp-402,
                0x1.086df80401797p-1021,
                0x1.086df80401797p-1022,
                0x1.086f8e78ec348p-19,
                0x1.086f8e78ec348p-20,
                0x1.0872e73ea1718p-961,
                0x1.0873d88ccc7e6p-67,
                0x1.087b3p-1054,
                0x1.087ba40d1f2fp-409,
                0x1.088295e0cc54ep-263,
                0x1.08864936c67eep358,
                0x1.08871f4c0dbb9p+472,
                0x1.0889eb89fab62p-410,
                0x1.088aa25038e34p-411,
                0x1.088b591677107p-411,
                0x1.088d7d693197ep-403,
                0x1.088d7d693197ep-404,
                0x1.088d7d693197ep-405,
                0x1.088d7d693197ep-406,
                0x1.088d7d693197ep-407,
                0x1.088d7d693197ep-408,
                0x1.088d7d693197ep-409,
                0x1.088p-1064,
                0x1.089956e5d3143p721,
                0x1.089a82fa60186p-419,
                0x1.089f8d00fa3aep670,
                0x1.08a037fc63261p585,
                0x1.08a037fc63261p586,
                0x1.08a22474b236p974,
                0x1.08a4be5ab3e1bp-568,
                0x1.08a884f678593p911,
                0x1.08b0ba47f164fp-201,
                0x1.08b25a8c8db4cp258,
                0x1.08b25a8c8db4cp259,
                0x1.08b2a2c28029p83,
                0x1.08b2a2c28029p+84,
                0x1.08b591212f23p-928,
                0x1.08c03fc472aadp-211,
                0x1.08c0a8174285ep382,
                0x1.08c0a8174285ep383,
                0x1.08c2ff075b8ep621,
                0x1.08c5c5231800fp+256,
                0x1.08cc49a1b2ac2p841,
                0x1.08cc49a1b2ac2p842,
                0x1.08ce499519ce4p245,
                0x1.08d338e046544p-524,
                0x1.08d338e046544p-525,
                0x1.08d5d9bbc62eap-805,
                0x1.08d6d747aa858p-284,
                0x1.08dea0b5578bep+680,
                0x1.08dea0b5578bep681,
                0x1.08e0e94169f9p610,
                0x1.08e147161e896p-391,
                0x1.08e147161e896p-392,
                0x1.08e147161e896p-393,
                0x1.08e3150e39d98p-283,
                0x1.08e3150e39d98p-284,
                0x1.08e3150e39d98p-285,
                0x1.08e98b22ff9fap-688,
                0x1.08e98b22ff9fap-689,
                0x1.08e98b22ff9fap-690,
                0x1.08f07912a94ap+180,
                0x1.08f936baf85c1p+568,
                0x1.08fbd69563f84p622,
                0x1.08fbd69563f84p623,
                0x1.08fbd69563f84p+624,
                0x1.08fbd69563f84p625,
                0x1.08fc561be390ep-297,
                0x1.08fe2015d015p-867,
                0x1.08fe2015d015p-868,
                0x1.08fe2015d015p-869,
                0x1.08fe2015d015p-870,
                0x1.09010dc3940fdp-159,
                0x1.090684f5fe998p-645,
                0x1.090684f5fe998p-646,
                0x1.090684f5fe998p-647,
                0x1.090684f5fe998p-648,
                0x1.090684f5fe998p-649,
                0x1.090684f5fe998p-650,
                0x1.0909a7f64fd2cp-909,
                0x1.0909c975913bfp-704,
                0x1.090b32a2cc7a2p-386,
                0x1.090d951009a02p-823,
                0x1.09112368e6c18p-578,
                0x1.091152e12f9a8p858,
                0x1.091152e12f9a8p859,
                0x1.091152e12f9a8p+860,
                0x1.091152e12f9a8p861,
                0x1.0911ac4363228p-802,
                0x1.0911ac4363228p-803,
                0x1.09133531391ap787,
                0x1.09133531391ap+788,
                0x1.09133531391ap789,
                0x1.09133531391ap790,
                0x1.09133531391ap791,
                0x1.0914a4c025ap-1030,
                0x1.09189461770aep-883,
                0x1.09189461770aep-884,
                0x1.091a4ddf352fep-621,
                0x1.092b9407dcc74p850,
                0x1.092d6c8ff7fa6p-661,
                0x1.09355f8050c02p-768,
                0x1.0936c6a7c200ep+852,
                0x1.094c291e364p-1032,
                0x1.094f8a44b87eap+468,
                0x1.094f8a44b87eap469,
                0x1.094f8a44b87eap470,
                0x1.094f8a44b87eap471,
                0x1.0952aaa23e9dap-748,
                0x1.0952aaa23e9dap-749,
                0x1.095cdaa3ac4ap-665,
                0x1.095d7ddc2a6d8p-947,
                0x1.096310cb3dd4dp-210,
                0x1.096310cb3dd4dp-211,
                0x1.096dec97a2c86p-1014,
                0x1.096dec97a2c86p-1015,
                0x1.096f7e4dabd2cp-31,
                0x1.096f7e4dabd2cp-32,
                0x1.096f7e4dabd2cp-33,
                0x1.0973767cf492ep579,
                0x1.0981029b4ab46p-105,
                0x1.0990e5c3406c4p-208,
                0x1.0990f7083f99fp-848,
                0x1.0991a9bfa58c8p-598,
                0x1.0996db7de3ad8p603,
                0x1.099a98cc0683p-1005,
                0x1.099a98cc0683p-1006,
                0x1.099b393b84832p-585,
                0x1.099b393b84832p-586,
                0x1.099b393b84832p-587,
                0x1.099b393b84832p-588,
                0x1.099b393b84832p-589,
                0x1.099b393b84832p-590,
                0x1.099b393b84832p-591,
                0x1.099b393b84832p-592,
                0x1.099b393b84832p-593,
                0x1.099b393b84832p-594,
                0x1.099b393b84832p-595,
                0x1.099e2f0d84ddap-885,
                0x1.09afdc438245ep-558,
                0x1.09bdf8054f836p-694,
                0x1.09bf6429a15e4p-587,
                0x1.09c8eaa8ce898p-871,
                0x1.09c8eaa8ce898p-872,
                0x1.09ce27037d143p-462,
                0x1.09ce57af299cep-254,
                0x1.09cee3a2c538cp649,
                0x1.09d57b6ff1d48p-860,
                0x1.09d57b6ff1d48p-861,
                0x1.09d57b6ff1d48p-862,
                0x1.09d57b6ff1d48p-863,
                0x1.09d57b6ff1d48p-864,
                0x1.09d8792fb4c49p-113,
                0x1.09d8792fb4c4ap-112,
                0x1.09d8792fb4c4ap-113,
                0x1.09dc2dae6dc66p-479,
                0x1.09e161e1168p-1033,
                0x1.09e244fbfd4c4p+288,
                0x1.09e63b656efp-911,
                0x1.09e63b656efp-912,
                0x1.09e9758a90126p170,
                0x1.09ee6a6b5a619p-704,
                0x1.09efa4597a878p-619,
                0x1.09f06ddb8205ap-84,
                0x1.09f8c354fb201p-425,
                0x1.09fd865182866p-162,
                0x1.0a0b71eb5d526p-733,
                0x1.0a0cb57d239c6p-986,
                0x1.0a0cb57d239c6p-987,
                0x1.0a0ef1b973164p-147,
                0x1.0a14ed8d45fcp341,
                0x1.0a15d46cc7c85p-394,
                0x1.0a1f5b8132466p+372,
                0x1.0a21f9902af7p743,
                0x1.0a261125762c5p-374,
                0x1.0a2792a8ccabp-696,
                0x1.0a2792a8ccabp-697,
                0x1.0a2792a8ccabp-698,
                0x1.0a2792a8ccabp-699,
                0x1.0a2792a8ccabp-700,
                0x1.0a2792a8ccabp-701,
                0x1.0a2792a8ccabp-702,
                0x1.0a2833945c43p-451,
                0x1.0a2c446b448cbp-557,
                0x1.0a2f6d66ee4eep+436,
                0x1.0a2f7d0c14ee7p558,
                0x1.0a30c47b35df6p278,
                0x1.0a30c47b35df6p279,
                0x1.0a30c47b35df6p+280,
                0x1.0a30c47b35df6p281,
                0x1.0a31a03822bcap109,
                0x1.0a31a03822bcap110,
                0x1.0a31a03822bcap111,
                0x1.0a3670d7d2448p-280,
                0x1.0a3bb33903d55p146,
                0x1.0a3ea66a5cd96p-937,
                0x1.0a4a1bfc750e5p-366,
                0x1.0a4ac49201c42p899,
                0x1.0a5261d41e826p-248,
                0x1.0a5261d41e826p-249,
                0x1.0a59fff60f511p481,
                0x1.0a59fff60f511p482,
                0x1.0a6242e2f79a9p-316,
                0x1.0a6650b926d66p857,
                0x1.0a67f801519c7p-883,
                0x1.0a6b7c3035522p+544,
                0x1.0a6edd9c0d117p773,
                0x1.0a705550ed29p-241,
                0x1.0a705550ed29p-242,
                0x1.0a742d78f2966p-736,
                0x1.0a754e53d3d9cp-347,
                0x1.0a75ccf91ce04p-373,
                0x1.0a75f770bc557p-828,
                0x1.0a775dcd32633p-544,
                0x1.0a7932b55b97p-944,
                0x1.0a7ec0dde51cap145,
                0x1.0a7f825f049fcp175,
                0x1.0a81e489d3778p-741,
                0x1.0a861327e230ep-439,
                0x1.0a861327e230ep-440,
                0x1.0a861327e230ep-441,
                0x1.0a861327e230ep-442,
                0x1.0a861327e230ep-443,
                0x1.0a8747a4c17c8p-294,
                0x1.0a8747a4c17c8p-295,
                0x1.0a88d1a6cc952p-97,
                0x1.0a8f40d74a702p-351,
                0x1.0a95a3c55d282p+560,
                0x1.0a95e06cfe54cp-578,
                0x1.0a9964a6f6734p237,
                0x1.0a9964a6f6734p238,
                0x1.0a9964a6f6734p239,
                0x1.0a997d7dd85d5p-867,
                0x1.0a99d4f4ccf31p-1009,
                0x1.0a9a269c5a414p-745,
                0x1.0a9bb99p-1046,
                0x1.0aa220ad63f48p-612,
                0x1.0aa69e243d4dcp879,
                0x1.0aabe0d19e1aap435,
                0x1.0ab5be61d3c34p+468,
                0x1.0ab877c142ffap-793,
                0x1.0ab877c142ffap-794,
                0x1.0ab8868e446fep-810,
                0x1.0abde7a187b61p858,
                0x1.0ac03fe1bba65p-459,
                0x1.0ac0f10a7ef64p-453,
                0x1.0ac0f10a7ef64p-454,
                0x1.0ac2e7f90b8a2p-161,
                0x1.0ac55aa1bbae4p+476,
                0x1.0ac55aa1bbae4p477,
                0x1.0ac573b4a35e3p858,
                0x1.0ac96b966adc6p-926,
                0x1.0ac9f2f6c6954p-434,
                0x1.0acca82ad8bf6p-926,
                0x1.0acde6a98eb4cp502,
                0x1.0acde6a98eb4cp503,
                0x1.0acde6a98eb4cp+504,
                0x1.0acde6a98eb4cp505,
                0x1.0acde6a98eb4cp506,
                0x1.0acde6a98eb4cp507,
                0x1.0acde6a98eb4cp+508,
                0x1.0acde6a98eb4cp509,
                0x1.0ad053d745fdep954,
                0x1.0ad053d745fdep955,
                0x1.0ad30b6123873p-706,
                0x1.0ad358c0ede54p-344,
                0x1.0ad5ad71cc38cp-210,
                0x1.0ad836f269a17p-323,
                0x1.0ad836f269a17p-324,
                0x1.0ad836f269a17p-325,
                0x1.0ad856d0e17bep-1009,
                0x1.0ad8840c15a4cp515,
                0x1.0ad9bf4a352fcp-167,
                0x1.0adb8c389269cp+264,
                0x1.0adc8dad355b2p222,
                0x1.0adc8dad355b2p223,
                0x1.0adc8dad355b2p+224,
                0x1.0adc8dad355b2p225,
                0x1.0adc8dad355b2p226,
                0x1.0ae113b32c292p-115,
                0x1.0ae113b32c292p-116,
                0x1.0ae113b32c292p-117,
                0x1.0ae34e56411d4p+712,
                0x1.0ae45b698552ep-517,
                0x1.0af1da9fffab4p-495,
                0x1.0af1da9fffab4p-496,
                0x1.0af2d6e59d77ep759,
                0x1.0af9db0f0575p-194,
                0x1.0af9db0f0575p-195,
                0x1.0af9db0f0575p-196,
                0x1.0afa4102aba1cp559,
                0x1.0afbb37bd967bp242,
                0x1.0afd89a99fd1ep+676,
                0x1.0afd89a99fd1ep677,
                0x1.0aff95cc5b092p-307,
                0x1.0aff95cc5b092p-308,
                0x1.0aff95cc5b092p-309,
                0x1.0aff95cc5b092p-310,
                0x1.0affb869cb9p-88,
                0x1.0b03737c6a7fp-820,
                0x1.0b03737c6a7fp-821,
                0x1.0b061654e4c1dp-369,
                0x1.0b066c5807cbp-868,
                0x1.0b1340f06c8e4p911,
                0x1.0b1340f06c8e4p+912,
                0x1.0b1be07fefc8ep-567,
                0x1.0b1f74a323dc2p939,
                0x1.0b260db2bac7ap-707,
                0x1.0b260db2bac7ap-708,
                0x1.0b29b18d4958dp241,
                0x1.0b38e07c745aep-473,
                0x1.0b38e07c745aep-474,
                0x1.0b38e07c745aep-475,
                0x1.0b38e07c745aep-476,
                0x1.0b38e07c745aep-477,
                0x1.0b3e6d5bf755ap-669,
                0x1.0b3e6d5bf755ap-670,
                0x1.0b46c6cdd6e3ep+176,
                0x1.0b4cc5da43251p-656,
                0x1.0b4cc5da43251p-657,
                0x1.0b4d9f2c45e66p-383,
                0x1.0b4fef741892bp-712,
                0x1.0b517dc5d3212p-1012,
                0x1.0b517dc5d3212p-1013,
                0x1.0b517dc5d3212p-1014,
                0x1.0b5298786d22p+292,
                0x1.0b552d2edcdeap-248,
                0x1.0b552d2edcdeap-249,
                0x1.0b552d2edcdeap-250,
                0x1.0b552d2edcdeap-251,
                0x1.0b552d2edcdeap-252,
                0x1.0b552d2edcdeap-253,
                0x1.0b552d2edcdeap-254,
                0x1.0b552d2edcdeap-255,
                0x1.0b55f47dec66cp+956,
                0x1.0b5659d928459p-552,
                0x1.0b56a05de9587p-101,
                0x1.0b5773cc11fc2p-405,
                0x1.0b5773cc11fc2p-406,
                0x1.0b5bfadfb0cecp169,
                0x1.0b60b6a188436p-913,
                0x1.0b6a9166663d1p+428,
                0x1.0b74604cf8bc7p-712,
                0x1.0b751e5085a0ep798,
                0x1.0b751e5085a0ep799,
                0x1.0b7cb60c994dap889,
                0x1.0b7cb60c994dap890,
                0x1.0b7cb60c994dap891,
                0x1.0b7cb60c994dap+892,
                0x1.0b7cb60c994dap893,
                0x1.0b7cb60c994dap894,
                0x1.0b7cb60c994dap895,
                0x1.0b7cb60c994dap+896,
                0x1.0b7cb60c994dap897,
                0x1.0b7cb60c994dap898,
                0x1.0b7cb60c994dap899,
                0x1.0b7cb60c994dap+900,
                0x1.0b7cb60c994dap901,
                0x1.0b7cb60c994dap902,
                0x1.0b814b2b445fap339,
                0x1.0b841cbfa3c6p-1014,
                0x1.0b877cab94bd7p-213,
                0x1.0b88257e31093p731,
                0x1.0b8e0acac4eaep+660,
                0x1.0b8e0acac4eaep661,
                0x1.0b8e0acac4eaep662,
                0x1.0b8e0acac4eaep663,
                0x1.0b8e0acac4eaep+664,
                0x1.0b93fbd82353cp+304,
                0x1.0b93fbd82353cp305,
                0x1.0b93fbd82353cp306,
                0x1.0b98eeaaf578cp-983,
                0x1.0b98eeaaf578cp-984,
                0x1.0b98f5e11c998p603,
                0x1.0b9be3a5e16aap-138,
                0x1.0ba5e37f1caf7p-163,
                0x1.0baac78726bp-179,
                0x1.0bab78cb07d72p-876,
                0x1.0bab78cb07d72p-877,
                0x1.0bae035fb0e43p446,
                0x1.0baefaf3459b7p+632,
                0x1.0bafc10f1d01cp965,
                0x1.0bafc10f1d01cp966,
                0x1.0bafc10f1d01cp967,
                0x1.0bb74648c898p-896,
                0x1.0bb74648c898p-897,
                0x1.0bb76a8353c5ap-633,
                0x1.0bbb1ad85b0e1p690,
                0x1.0bc1afeae18fp-642,
                0x1.0bc1afeae18fp-643,
                0x1.0bc60e6896718p723,
                0x1.0bc60e6896718p+724,
                0x1.0bc60e6896718p725,
                0x1.0bc60e6896718p726,
                0x1.0bc60e6896718p727,
                0x1.0bc60e6896718p+728,
                0x1.0bc7b7603a2cap-118,
                0x1.0bc7b7603a2cap-119,
                0x1.0bc7b7603a2cap-120,
                0x1.0bc88e309d908p749,
                0x1.0bc88e309d908p750,
                0x1.0bc88e309d908p751,
                0x1.0bc88e309d908p+752,
                0x1.0bc88e309d908p753,
                0x1.0bc88e309d908p754,
                0x1.0bc88e309d908p755,
                0x1.0bc88e309d908p+756,
                0x1.0bc9623677a26p-475,
                0x1.0bc9623677a26p-476,
                0x1.0bd2bfd34f98bp202,
                0x1.0bd2bfd34f98bp203,
                0x1.0bd2bfd34f98bp+204,
                0x1.0bd4bd88c86d7p+372,
                0x1.0bd4cdc5e3c5ap+188,
                0x1.0bdcf1096c91p-310,
                0x1.0be08d0527e1dp-990,
                0x1.0be08d0527e1ep-989,
                0x1.0be08d0527e1ep-990,
                0x1.0be1e55424c4p965,
                0x1.0be1e55424c4p966,
                0x1.0be4b45b00971p-998,
                0x1.0be4fb9af66d4p601,
                0x1.0be7437bd14a2p-45,
                0x1.0be7437bd14a2p-46,
                0x1.0be7437bd14a2p-47,
                0x1.0be7437bd14a2p-48,
                0x1.0bf38a812cd68p-552,
                0x1.0bf3a86e8a286p771,
                0x1.0c0f48457644p-1028,
                0x1.0c1671b73d898p-111,
                0x1.0c27fa028b0eep-505,
                0x1.0c2ded8770bbp-358,
                0x1.0c2ded8770bbp-359,
                0x1.0c2ded8770bbp-360,
                0x1.0c2ded8770bbp-361,
                0x1.0c34a747abb8ap-805,
                0x1.0c35312de9d1p553,
                0x1.0c35312de9d1p554,
                0x1.0c394252de353p709,
                0x1.0c3e64e7p-1036,
                0x1.0c3e64e7p-1037,
                0x1.0c42fb8dbd91cp-838,
                0x1.0c42fb8dbd91cp-839,
                0x1.0c458231910d4p782,
                0x1.0c4f9921c3f8fp-38,
                0x1.0c52fe6dc6a1cp170,
                0x1.0c52fe6dc6a1cp171,
                0x1.0c52fe6dc6a1cp+172,
                0x1.0c52fe6dc6a1cp173,
                0x1.0c52fe6dc6a1cp174,
                0x1.0c52fe6dc6a1cp175,
                0x1.0c52fe6dc6a1cp+176,
                0x1.0c52fe6dc6a1cp177,
                0x1.0c52fe6dc6a1cp178,
                0x1.0c52fe6dc6a1cp179,
                0x1.0c52fe6dc6a1cp+180,
                0x1.0c624dc976612p-975,
                0x1.0c6cc2e52c168p497,
                0x1.0c6cc2e52c168p498,
                0x1.0c6f7a0b5ed8dp-20,
                0x1.0c71f3ed7b542p289,
                0x1.0c7f2e55776a3p225,
                0x1.0c7f2e55776a3p226,
                0x1.0c80c65a3d5f6p+268,
                0x1.0c80c65a3d5f6p269,
                0x1.0c819b445642p-471,
                0x1.0c83c75865542p-515,
                0x1.0c8dfc920ef17p-824,
                0x1.0c94bf931bd56p-477,
                0x1.0c94bf931bd56p-478,
                0x1.0c94bf931bd56p-479,
                0x1.0c94bf931bd56p-480,
                0x1.0c94bf931bd56p-481,
                0x1.0c94bf931bd56p-482,
                0x1.0c96417cff7f8p694,
                0x1.0c96417cff7f8p695,
                0x1.0c96417cff7f8p+696,
                0x1.0c9b665385bc9p-446,
                0x1.0c9p-1062,
                0x1.0ca004c60cf14p-849,
                0x1.0cab388842182p763,
                0x1.0cac2a2aa6df1p-926,
                0x1.0cb70d24b7378p465,
                0x1.0cbdc63089476p681,
                0x1.0cbdc63089476p682,
                0x1.0cbf9fb95fbb2p-730,
                0x1.0cc4f55eecfeap-330,
                0x1.0cc4f55eecfeap-331,
                0x1.0cc4f55eecfeap-332,
                0x1.0cc4f55eecfeap-333,
                0x1.0cc4f55eecfeap-334,
                0x1.0ccfa547ba5b6p-876,
                0x1.0ccfa547ba5b6p-877,
                0x1.0ccfa547ba5b6p-878,
                0x1.0ccfa547ba5b6p-879,
                0x1.0ccfa547ba5b6p-880,
                0x1.0ccfa547ba5b6p-881,
                0x1.0ccfa547ba5b6p-882,
                0x1.0ccfa547ba5b6p-883,
                0x1.0cd45fbd85168p166,
                0x1.0cd59b3cd417ep505,
                0x1.0cd59b3cd417ep506,
                0x1.0cd5b474ad1bap-457,
                0x1.0cd5c1ae42ed6p-322,
                0x1.0cda75482dcacp+916,
                0x1.0cda75482dcacp917,
                0x1.0cda75482dcacp918,
                0x1.0cda75482dcacp919,
                0x1.0cda75482dcacp+920,
                0x1.0ce37bded79e4p-595,
                0x1.0ce3850bdd6e2p-294,
                0x1.0cef3dcd7711cp-754,
                0x1.0cef3dcd7711cp-755,
                0x1.0cf17a101a50cp-259,
                0x1.0cf3ab81f1226p567,
                0x1.0cfeb353a97dap950,
                0x1.0cfeb353a97dbp950,
                0x1.0d00a63e9dd56p289,
                0x1.0d022c6214dacp-811,
                0x1.0d0395f769a1p474,
                0x1.0d043c678a228p-605,
                0x1.0d058ba53e146p+284,
                0x1.0d058ba53e146p285,
                0x1.0d05df5f3fffp-80,
                0x1.0d05df5f3fffp-81,
                0x1.0d084988441a2p-984,
                0x1.0d084988441a2p-985,
                0x1.0d084988441a2p-986,
                0x1.0d16b1682f19p-549,
                0x1.0d1c77507eb1ap775,
                0x1.0d1f3580d3e39p-998,
                0x1.0d21f1fda33aep649,
                0x1.0d21f1fda33aep650,
                0x1.0d24f59626544p910,
                0x1.0d2980dabf6b6p-190,
                0x1.0d2bf8f790642p993,
                0x1.0d2d04904fd39p-526,
                0x1.0d2fffb823e9ep-686,
                0x1.0d47021912b06p-369,
                0x1.0d4c83df27458p959,
                0x1.0d4da405f7fb8p-42,
                0x1.0d51a73deee2cp-700,
                0x1.0d51a73deee2cp-701,
                0x1.0d545c4311626p+636,
                0x1.0d5982c6a6ef6p-366,
                0x1.0d5a6cac26bd9p635,
                0x1.0d686c58afa56p-121,
                0x1.0d686c58afa56p-122,
                0x1.0d69cf0b5fce1p861,
                0x1.0d73ae895a6a5p-60,
                0x1.0d781033e681ep659,
                0x1.0d7cea01a40aep-614,
                0x1.0d7cea01a40aep-615,
                0x1.0d7cea01a40aep-616,
                0x1.0d8afc9bfd35p+272,
                0x1.0d9976a5d5297p-216,
                0x1.0d9976a5d5298p-215,
                0x1.0d9976a5d5298p-216,
                0x1.0d9a630f441d6p-802,
                0x1.0d9b828199006p-907,
                0x1.0da6e6db887a6p-308,
                0x1.0da6e6db887a6p-309,
                0x1.0da985b572807p-865,
                0x1.0db20eade0209p+836,
                0x1.0db354366446ep+204,
                0x1.0db7990792f39p135,
                0x1.0db7f066a422bp-591,
                0x1.0db9e2cd88d68p241,
                0x1.0dc7f22382c16p+212,
                0x1.0dcbf420977cep-837,
                0x1.0dcbf420977cep-838,
                0x1.0dcc3baead672p-1021,
                0x1.0dcd5b3b2ad26p+984,
                0x1.0dd3b8c41a10cp-1004,
                0x1.0dd8f2788d69ap414,
                0x1.0dd8f2788d69ap415,
                0x1.0dd8f2788d69ap+416,
                0x1.0dda81b80b2bbp-741,
                0x1.0dde45cb9f076p-753,
                0x1.0dde45cb9f076p-754,
                0x1.0dde45cb9f076p-755,
                0x1.0dde6d69a39ap+536,
                0x1.0dde6d69a39ap537,
                0x1.0de1593369d1bp269,
                0x1.0de1593369d1cp269,
                0x1.0de1593369d1cp270,
                0x1.0de6f53b56217p-663,
                0x1.0de6f53b56217p-664,
                0x1.0df274ff5c342p-759,
                0x1.0df355ba5435ap-984,
                0x1.0df47b09c75b2p707,
                0x1.0df571ed66883p+408,
                0x1.0dfdc476cb097p-887,
                0x1.0dffdb147397ap-106,
                0x1.0e00b5afad15ap-77,
                0x1.0e033ac5c2327p-385,
                0x1.0e07ecd40a61ep495,
                0x1.0e085b7debebp357,
                0x1.0e085b7debebp358,
                0x1.0e085b7debebp359,
                0x1.0e097217c58c7p-594,
                0x1.0e12419736d2dp-859,
                0x1.0e12419736d2dp-860,
                0x1.0e15ee1d7539dp-146,
                0x1.0e162b336df3dp+476,
                0x1.0e17efcd98e5cp-824,
                0x1.0e18b3365ba8ap-636,
                0x1.0e1df34dbe8ccp-888,
                0x1.0e294eebc7d2cp753,
                0x1.0e294eebc7d2cp754,
                0x1.0e294eebc7d2cp755,
                0x1.0e294eebc7d2cp+756,
                0x1.0e303a3450fd4p195,
                0x1.0e32f36012af2p597,
                0x1.0e32f36012af2p598,
                0x1.0e337464faa08p-57,
                0x1.0e35d83efadc8p+1016,
                0x1.0e35d83efadc8p1017,
                0x1.0e38c6d4ea61dp-127,
                0x1.0e3b0e8b721e2p+680,
                0x1.0e3dee9c9d9f5p-487,
                0x1.0e3dee9c9d9f5p-488,
                0x1.0e3ed49f2f982p+132,
                0x1.0e3f2c8b6cb18p290,
                0x1.0e488cfaf3518p445,
                0x1.0e53cc32c28e6p463,
                0x1.0e53cc32c28e6p+464,
                0x1.0e53cc32c28e6p465,
                0x1.0e53cc32c28e6p466,
                0x1.0e53cc32c28e6p467,
                0x1.0e53f14866427p+856,
                0x1.0e53f14866427p857,
                0x1.0e59605c8c22ep194,
                0x1.0e5990c48e797p-419,
                0x1.0e5eb59544079p-92,
                0x1.0e61652926caap+368,
                0x1.0e61652926caap369,
                0x1.0e61652926caap370,
                0x1.0e61652926caap371,
                0x1.0e658f38481dbp-710,
                0x1.0e66c2ce08c6dp833,
                0x1.0e66c2ce08c6dp834,
                0x1.0e6a0de73ba1p339,
                0x1.0e6b02afc2a3ap-862,
                0x1.0e6b5843bf0ddp-650,
                0x1.0e7473896f5acp+856,
                0x1.0e7473896f5acp857,
                0x1.0e7473896f5acp858,
                0x1.0e7473896f5acp859,
                0x1.0e7c9eebc444ap-897,
                0x1.0e7d85a43d43p-253,
                0x1.0e82a0207d872p559,
                0x1.0e90d83da47cp913,
                0x1.0e935752cabbap-53,
                0x1.0e935752cabbap-54,
                0x1.0e935752cabbap-55,
                0x1.0e97b786ea336p-609,
                0x1.0e97b786ea336p-610,
                0x1.0e97b786ea336p-611,
                0x1.0e9e18ec89c3ap+480,
                0x1.0e9fa98aba3ccp225,
                0x1.0ea797658b976p-536,
                0x1.0ea865aeb9b6ep853,
                0x1.0ea865aeb9b6ep854,
                0x1.0eab7c5a44fd9p-161,
                0x1.0eaf1d48b625dp-215,
                0x1.0eb7afd8c96ecp-791,
                0x1.0eb7afd8c96ecp-792,
                0x1.0ebc64246897p+136,
                0x1.0ec4be0ad8f89p-412,
                0x1.0ec4be0ad8f8ap-412,
                0x1.0ec776cae162ep925,
                0x1.0ec9ef9b1b68p-338,
                0x1.0ec9ef9b1b68p-339,
                0x1.0ec9ef9b1b68p-340,
                0x1.0ecc411cb96f5p-877,
                0x1.0ecca46737cd2p-75,
                0x1.0ecca46737cd2p-76,
                0x1.0ecca46737cd2p-77,
                0x1.0ecf2b36ef5a4p-146,
                0x1.0ecf2b36ef5a4p-147,
                0x1.0ed2e61e9fa84p-112,
                0x1.0ed2e61e9fa84p-113,
                0x1.0ed2e61e9fa84p-114,
                0x1.0ed2e61e9fa84p-115,
                0x1.0ed94fd382ea8p621,
                0x1.0ed9bd6bfd004p-234,
                0x1.0ed9d7668fb46p-920,
                0x1.0edcf1f873c64p319,
                0x1.0ee52a4c1865p+712,
                0x1.0ee54b5818908p-903,
                0x1.0ee54b5818908p-904,
                0x1.0ee54b5818908p-905,
                0x1.0ee70c3703628p-269,
                0x1.0ee7811241ad2p643,
                0x1.0ee9b7d7a9c1bp655,
                0x1.0ef8f4004a3e6p-643,
                0x1.0ef9e24f0e7ap783,
                0x1.0efb8cfe76fap959,
                0x1.0f00b81cebe3ep970,
                0x1.0f0470e6930eep325,
                0x1.0f0470e6930eep326,
                0x1.0f067de0dfd1ap321,
                0x1.0f0b1dea4edeep347,
                0x1.0f0cf064dd592p73,
                0x1.0f0d091cd4027p-505,
                0x1.0f0df842a5b6cp562,
                0x1.0f0df842a5b6cp563,
                0x1.0f1451ea55ddap-406,
                0x1.0f1b9266f4b6dp685,
                0x1.0f259cb734e18p-20,
                0x1.0f2bc2790aa9cp113,
                0x1.0f2bc2790aa9cp114,
                0x1.0f329f74fc43ep-542,
                0x1.0f48p-1058,
                0x1.0f48p-1059,
                0x1.0f54eab3adf37p190,
                0x1.0f54eab3adf37p191,
                0x1.0f5535fef2084p558,
                0x1.0f5a867cfcb73p-822,
                0x1.0f5a867cfcb73p-823,
                0x1.0f5cea4297adap+256,
                0x1.0f5cea4297adap257,
                0x1.0f5cea4297adap258,
                0x1.0f62f6461b974p-177,
                0x1.0f62f6461b974p-178,
                0x1.0f6f227097c28p-426,
                0x1.0f6f227097c28p-427,
                0x1.0f73be1dff9adp+760,
                0x1.0f73be1dff9adp761,
                0x1.0f73be1dff9adp762,
                0x1.0f73e1e90e3f7p182,
                0x1.0f73e1e90e3f7p183,
                0x1.0f749da49c9ebp-46,
                0x1.0f7c31f31dfc6p-612,
                0x1.0f7c31f31dfc6p-613,
                0x1.0f7c31f31dfc6p-614,
                0x1.0f86ae323cf84p-730,
                0x1.0f912f4dff7f2p699,
                0x1.0f912f4dff7f2p+700,
                0x1.0f9451ba34cd6p-32,
                0x1.0f95833ccf0d6p+392,
                0x1.0f95833ccf0d6p393,
                0x1.0f95833ccf0d6p394,
                0x1.0f95833ccf0d6p395,
                0x1.0f95833ccf0d6p+396,
                0x1.0f98e1e937254p-505,
                0x1.0f98e1e937254p-506,
                0x1.0f98e1e937254p-507,
                0x1.0fa0e62f26e8cp665,
                0x1.0fa0e62f26e8cp666,
                0x1.0fa0e62f26e8cp667,
                0x1.0fa0e62f26e8cp+668,
                0x1.0fa790623e15p-437,
                0x1.0fa9251566e39p-978,
                0x1.0fa9251566e39p-979,
                0x1.0fa9251566e39p-980,
                0x1.0fad4f2d3fe3dp+424,
                0x1.0fb2580647d96p-209,
                0x1.0fb490863fbc9p1007,
                0x1.0fb557f87e774p+144,
                0x1.0fc23505aca18p-477,
                0x1.0fc23505aca18p-478,
                0x1.0fc36b80bc6b8p466,
                0x1.0fc36b80bc6b8p467,
                0x1.0fc36b80bc6b8p+468,
                0x1.0fc36b80bc6b8p469,
                0x1.0fcf3b7c5b4e4p955,
                0x1.0fcfbcea6ee7cp-224,
                0x1.0fcfbcea6ee7cp-225,
                0x1.0fd07ed297f8p670,
                0x1.0fd07ed297f8p671,
                0x1.0fd07ed297f8p+672,
                0x1.0fd07ed297f8p673,
                0x1.0fd07ed297f8p674,
                0x1.0fd07ed297f8p675,
                0x1.0fd07ed297f8p+676,
                0x1.0fd07ed297f8p677,
                0x1.0fd07ed297f8p678,
                0x1.0fd07ed297f8p679,
                0x1.0fd97bcd5fdaap-122,
                0x1.0fde34c996086p233,
                0x1.0fe2efddacb97p689,
                0x1.0fe2efddacb97p690,
                0x1.0ff151a99f482p-608,
                0x1.0ff151a99f483p-608,
                0x1.0ff151a99f483p-609,
                0x1.0ff40d6f1bbfp729,
                0x1.1001a77de2346p-819,
                0x1.100223c57ccd4p369,
                0x1.1002b9b3e246cp-695,
                0x1.100565f15729ap993,
                0x1.1009fd836acf5p-792,
                0x1.100d12e19dc9p865,
                0x1.100f0a4d72016p-601,
                0x1.101663d6ef728p-369,
                0x1.101c5a850796ap-80,
                0x1.101e0e5f014e5p949,
                0x1.101f07813e5e8p287,
                0x1.101f07813e5e8p+288,
                0x1.101f07813e5e8p289,
                0x1.101f07813e5e8p290,
                0x1.101f07813e5e8p291,
                0x1.101f649f50d9cp-45,
                0x1.101f66b4f8ed9p123,
                0x1.1022075e308dfp-1018,
                0x1.10234e1ada8d8p702,
                0x1.10250b4443d49p-694,
                0x1.102520120a0cap+608,
                0x1.102b47e4af988p+660,
                0x1.102b47e4af988p661,
                0x1.102b47e4af988p662,
                0x1.102c663e4701ep809,
                0x1.102ca067d3598p-90,
                0x1.102e52fe11ef8p221,
                0x1.1033e45bb752cp-750,
                0x1.1033edabef16ep+284,
                0x1.1033edabef16ep285,
                0x1.1033edabef16ep286,
                0x1.1033edabef16ep287,
                0x1.1033edabef16ep+288,
                0x1.103512b413936p+136,
                0x1.103512b413936p137,
                0x1.103512b413936p138,
                0x1.103512b413936p139,
                0x1.103512b413936p+140,
                0x1.103512b413936p141,
                0x1.1039d428a8b8ep-123,
                0x1.1045da2e7e4dfp-757,
                0x1.1047eae6d8p-921,
                0x1.1047eae6d8p-922,
                0x1.1048866925476p489,
                0x1.104a9d81132c6p977,
                0x1.104fa3e44072dp914,
                0x1.104fdc8f03409p-587,
                0x1.1059dcf354e3ep-35,
                0x1.1059dcf354e3ep-36,
                0x1.1059dcf354e3ep-37,
                0x1.1059dcf354e3ep-38,
                0x1.1059dcf354e3ep-39,
                0x1.1059dcf354e3ep-40,
                0x1.105d1874d2098p+1000,
                0x1.105d1874d2098p999,
                0x1.106d07164e85dp281,
                0x1.106da80bb9582p491,
                0x1.106da8a23a9fep733,
                0x1.106da8a23a9fep734,
                0x1.106ee4f150dafp-336,
                0x1.1071a7531ee48p-925,
                0x1.1072c5f7db431p-57,
                0x1.1072c5f7db431p-58,
                0x1.1079997e424d4p177,
                0x1.107c64a0ebfc1p-534,
                0x1.1080f1d827bcep+480,
                0x1.108269fd210cbp362,
                0x1.108287947a80ap743,
                0x1.10836667dfc5p+796,
                0x1.108d5d557553cp+184,
                0x1.108e2a08729dcp-601,
                0x1.109341dc24914p+900,
                0x1.109cf84e0fbbp741,
                0x1.109cf84e0fbbp742,
                0x1.109d00e1d1d78p827,
                0x1.109d00e1d1d78p+828,
                0x1.109d00e1d1d78p829,
                0x1.109d00e1d1d78p830,
                0x1.109d00e1d1d78p831,
                0x1.109d718p-1048,
                0x1.10a16688e23ddp402,
                0x1.10a2bac862232p-957,
                0x1.10a6e6d35a361p727,
                0x1.10adc6ae1c36cp-778,
                0x1.10b0281a4304cp-731,
                0x1.10b1c35adec56p-235,
                0x1.10bd2c4e0dcb2p-303,
                0x1.10c20303fe0fp+448,
                0x1.10c20303fe0fp449,
                0x1.10c20303fe0fp450,
                0x1.10c20303fe0fp451,
                0x1.10c20303fe0fp+452,
                0x1.10c20303fe0fp453,
                0x1.10c20303fe0fp454,
                0x1.10c20303fe0fp455,
                0x1.10c20303fe0fp+456,
                0x1.10c20303fe0fp457,
                0x1.10c20303fe0fp458,
                0x1.10cb132c2ff63p846,
                0x1.10cb132c2ff63p847,
                0x1.10cbcf545967ep-112,
                0x1.10cp-1064,
                0x1.10d4d7f171772p623,
                0x1.10d4d7f171772p+624,
                0x1.10d4d7f171772p625,
                0x1.10e1662d07c1p-343,
                0x1.10e8302245571p-38,
                0x1.10ee60a2b5a9ep+156,
                0x1.10f452d2063dcp287,
                0x1.10f452d2063dcp+288,
                0x1.10f4ff883696p-769,
                0x1.11005012f46b6p-719,
                0x1.11005012f46b6p-720,
                0x1.11005012f46b6p-721,
                0x1.110050d2c17e3p-97,
                0x1.11078e301a2bdp259,
                0x1.11078e301a2bdp+260,
                0x1.110a410c7948cp-85,
                0x1.110d81e40b6cp-572,
                0x1.110ddf795b767p-284,
                0x1.110ecb3dfec44p-563,
                0x1.110ecb3dfec44p-564,
                0x1.1112188164531p-462,
                0x1.1112188164531p-463,
                0x1.11147fe608602p441,
                0x1.1115820895658p-193,
                0x1.1115f679d8e2bp-215,
                0x1.1119b4f208fa4p-542,
                0x1.1119b4f208fa4p-543,
                0x1.111ce7c6e9488p879,
                0x1.111ce7c6e9488p+880,
                0x1.111ce7c6e9488p881,
                0x1.111ce7c6e9488p882,
                0x1.111ce7c6e9488p883,
                0x1.111f32f2f4bcp-804,
                0x1.1123fb2391327p-333,
                0x1.112c423b711c9p-440,
                0x1.112ee1287f7f2p-205,
                0x1.112f983e853bfp471,
                0x1.1137ce535d062p461,
                0x1.1144ecb565ddcp+144,
                0x1.1149c3ef38bd9p654,
                0x1.114e977cef86ap-630,
                0x1.114e977cef86ap-631,
                0x1.114e977cef86ap-632,
                0x1.1153331830571p-627,
                0x1.1153331830571p-628,
                0x1.115762b066028p-171,
                0x1.115762b066028p-172,
                0x1.1159a8bd8a24p+148,
                0x1.115d533a46694p-724,
                0x1.1165b6a7a956dp-755,
                0x1.1167cf9686bf6p-740,
                0x1.1167cf9686bf6p-741,
                0x1.116805effaeaap-317,
                0x1.116805effaeaap-318,
                0x1.116805effaeaap-319,
                0x1.116805effaeaap-320,
                0x1.116ae12ae467fp-807,
                0x1.1174913e55424p-795,
                0x1.1174ea3324624p-30,
                0x1.117577a61d9b8p866,
                0x1.117577a61d9b8p867,
                0x1.1176804bda8b8p814,
                0x1.1177416feb874p547,
                0x1.117ecf506965p813,
                0x1.1180fbe4120d2p595,
                0x1.11894b202e9f4p527,
                0x1.11894b202e9f4p+528,
                0x1.11894b202e9f4p529,
                0x1.11894b202e9f4p530,
                0x1.11896a5e11ce2p-702,
                0x1.1189cf0f4e14ep757,
                0x1.118b7a6a1ad1ap-179,
                0x1.1191c40b7f2f4p447,
                0x1.11922a5330f5ap718,
                0x1.11926d079e00ap+304,
                0x1.11926d079e00ap305,
                0x1.11926d079e00ap306,
                0x1.11926d079e00ap307,
                0x1.1193034aa6472p-534,
                0x1.1193034aa6472p-535,
                0x1.1193034aa6472p-536,
                0x1.1193034aa6472p-537,
                0x1.1199e3bcd6978p-75,
                0x1.11a45d654a618p-301,
                0x1.11ae1c456e4p-1029,
                0x1.11b0ec57e649ap166,
                0x1.11b0ec57e649ap167,
                0x1.11b6602329204p982,
                0x1.11b6602329204p983,
                0x1.11b6602329204p+984,
                0x1.11b960ce5306p+384,
                0x1.11bf34594138p-287,
                0x1.11bf34594138p-288,
                0x1.11c102f8690c9p-173,
                0x1.11c102f8690c9p-174,
                0x1.11c2c456b0f3p+1012,
                0x1.11c2c456b0f3p1013,
                0x1.11c2c456b0f3p1014,
                0x1.11c2c456b0f3p1015,
                0x1.11c2c456b0f3p+1016,
                0x1.11cde9e58a944p+416,
                0x1.11d89a8b5c142p-14,
                0x1.11dbf428a7c5cp-664,
                0x1.11e2e497ade39p-435,
                0x1.11e2e497ade39p-436,
                0x1.11e3a60a66503p-966,
                0x1.11e52404e5673p+580,
                0x1.11f9e62fe4448p651,
                0x1.11fa8b3a62ab8p-90,
                0x1.120453766862ap-635,
                0x1.120e6f8adc69ap+912,
                0x1.120e6f8adc69ap913,
                0x1.120e6f8adc69ap914,
                0x1.12104ad2a4162p-551,
                0x1.12104ad2a4162p-552,
                0x1.12104ad2a4162p-553,
                0x1.12104ad2a4162p-554,
                0x1.121b9ce1e172ap-155,
                0x1.121c87b90c472p+804,
                0x1.121c87b90c472p805,
                0x1.122319db25184p-804,
                0x1.122748a5ea3dcp-745,
                0x1.1237c5db93fb4p327,
                0x1.124cbc5593b64p+380,
                0x1.124cbc5593b64p381,
                0x1.124d49ae24d56p871,
                0x1.124d49ae24d56p+872,
                0x1.124d49ae24d56p873,
                0x1.124dbcab1a356p-734,
                0x1.124e63593f5ep-1000,
                0x1.1251c23dc7aa3p-1004,
                0x1.1254cebad2576p662,
                0x1.125782ec15cbep102,
                0x1.125d1eb3d1932p-890,
                0x1.125d3259e158ap-387,
                0x1.125d3259e158ap-388,
                0x1.12641e58be566p-530,
                0x1.127657a23a2b4p307,
                0x1.1278deb83fbe1p+664,
                0x1.1279f72804954p-639,
                0x1.1284d7a14fa34p314,
                0x1.1284d7a14fa34p315,
                0x1.1284d7a14fa34p+316,
                0x1.12966629982fdp594,
                0x1.12970b6f3500ep+944,
                0x1.12970b6f3500ep945,
                0x1.12970b6f3500ep946,
                0x1.12970b6f3500ep947,
                0x1.1297872d9cbb4p-515,
                0x1.1297872d9cbb5p-515,
                0x1.1297872d9cbb5p-516,
                0x1.129a49c44991ap822,
                0x1.12a0ac22e6a4ap-968,
                0x1.12a13daa46fe4p+576,
                0x1.12a13daa46fe4p577,
                0x1.12a13daa46fe4p578,
                0x1.12a13daa46fe4p579,
                0x1.12a13daa46fe4p+580,
                0x1.12a208ba25133p489,
                0x1.12b2321d9caap258,
                0x1.12b2b904673d9p-972,
                0x1.12bbd4e3bc794p-700,
                0x1.12bbd4e3bc794p-701,
                0x1.12bc948584086p618,
                0x1.12bc948584086p619,
                0x1.12bc948584086p+620,
                0x1.12bc948584086p621,
                0x1.12be2041a79fcp-848,
                0x1.12be2041a79fcp-849,
                0x1.12be2041a79fcp-850,
                0x1.12be2041a79fcp-851,
                0x1.12be2041a79fcp-852,
                0x1.12be2041a79fcp-853,
                0x1.12be2041a79fcp-854,
                0x1.12be2041a79fcp-855,
                0x1.12c4b88a13278p-115,
                0x1.12cc6b1579e74p+360,
                0x1.12cee2ea4b33p598,
                0x1.12cee2ea4b33p599,
                0x1.12cee2ea4b33p+600,
                0x1.12cee2ea4b33p601,
                0x1.12d3cc5aef816p-707,
                0x1.12d3cc5aef816p-708,
                0x1.12d3cc5aef816p-709,
                0x1.12d3cc5aef816p-710,
                0x1.12d3cc5aef816p-711,
                0x1.12d3cc5aef816p-712,
                0x1.12de7f9578af2p658,
                0x1.12de7f9578af2p659,
                0x1.12e0be826d694p-30,
                0x1.12e4468289e63p-807,
                0x1.12e8641d786f6p246,
                0x1.12e8c39029d19p305,
                0x1.12ec77e09ff38p-276,
                0x1.12f1044ad4826p891,
                0x1.1301f03a410e2p-819,
                0x1.130b1dcb98c9fp-582,
                0x1.130b1dcb98c9fp-583,
                0x1.130b7ec1010fdp-142,
                0x1.13118f0129cd8p763,
                0x1.1320daaa300c6p-575,
                0x1.1323025cfbd68p-738,
                0x1.1323025cfbd68p-739,
                0x1.1323025cfbd68p-740,
                0x1.1323025cfbd68p-741,
                0x1.1324p-1060,
                0x1.1325bad72deb7p235,
                0x1.1328309f174e3p395,
                0x1.132a095ce493p455,
                0x1.132a16cb2afa4p527,
                0x1.132c1ce3e3226p977,
                0x1.132ecc7614058p-1025,
                0x1.132f8d19ba3c2p753,
                0x1.132f8d19ba3c2p754,
                0x1.1338470923d5ap-344,
                0x1.133b1a33a1108p405,
                0x1.133f0a1781beep-424,
                0x1.133f0a1781beep-425,
                0x1.1346666a8788p-1027,
                0x1.1349532ff3cecp495,
                0x1.134d951f961c4p-568,
                0x1.13528ca816576p149,
                0x1.135314090b052p-272,
                0x1.135314090b052p-273,
                0x1.1353dea3d4237p959,
                0x1.1358930c57d3dp-283,
                0x1.135972630774cp-158,
                0x1.135972630774cp-159,
                0x1.135972630774cp-160,
                0x1.135972630774cp-161,
                0x1.135972630774cp-162,
                0x1.135972630774cp-163,
                0x1.135972630774cp-164,
                0x1.135972630774cp-165,
                0x1.135b0901c33c4p931,
                0x1.135b0901c33c4p+932,
                0x1.135b0901c33c4p933,
                0x1.135b0901c33c4p934,
                0x1.135b0901c33c4p935,
                0x1.1361d5297d06p-747,
                0x1.1361fe1dd06e4p-648,
                0x1.136291682af6ap+944,
                0x1.136291682af6ap945,
                0x1.136791e54f1cbp+356,
                0x1.136e6aa1474f6p-320,
                0x1.136fa936994d8p-382,
                0x1.1373095755fcap-261,
                0x1.137367c236c65p+940,
                0x1.1373e7e2014bfp1009,
                0x1.13770474911bcp-567,
                0x1.13770474911bcp-568,
                0x1.13770474911bcp-569,
                0x1.13770474911bcp-570,
                0x1.13770474911bcp-571,
                0x1.137e11e885367p+524,
                0x1.137e5674e7d6cp-1015,
                0x1.137edb4c7844cp-661,
                0x1.137f09d9c9bacp-253,
                0x1.1381a93a87848p141,
                0x1.1381a93a87848p142,
                0x1.138643a016132p235,
                0x1.138643a016132p+236,
                0x1.138643a016132p237,
                0x1.138812a0bafb5p386,
                0x1.138812a0bafb5p387,
                0x1.1388ea3ea25aap554,
                0x1.13893a628bfe3p1015,
                0x1.13893a628bfe3p+1016,
                0x1.138eaf741cbf1p-159,
                0x1.138feb6fb8625p-658,
                0x1.13918353bbc47p-623,
                0x1.13918353bbc47p-624,
                0x1.13918353bbc47p-625,
                0x1.1392b9ecfb848p+980,
                0x1.1392b9ecfb848p981,
                0x1.1392b9ecfb848p982,
                0x1.13957b96cf7a5p-585,
                0x1.1398f075c4f72p-839,
                0x1.139e0a26f5b03p-69,
                0x1.139e40187c841p-327,
                0x1.13a26646723e7p+816,
                0x1.13a884d94f2c6p+708,
                0x1.13a884d94f2c6p709,
                0x1.13a884d94f2c6p710,
                0x1.13a884d94f2c6p711,
                0x1.13a884d94f2c6p+712,
                0x1.13ad689b07f87p153,
                0x1.13b39c010f3dep775,
                0x1.13c3268357478p774,
                0x1.13c3a650b7405p-579,
                0x1.13c85955f2923p-711,
                0x1.13c85955f2924p-710,
                0x1.13c85955f2924p-711,
                0x1.13c85955f2924p-712,
                0x1.13c9b2de46c9ap-61,
                0x1.13c9b2de46c9ap-62,
                0x1.13c9b2de46c9ap-63,
                0x1.13c9b2de46c9ap-64,
                0x1.13c9b2de46c9ap-65,
                0x1.13c9b2de46c9ap-66,
                0x1.13d0652588202p-376,
                0x1.13d0652588202p-377,
                0x1.13d5336164e5p245,
                0x1.13d84c54f0e9ep975,
                0x1.13ddabe3b520cp-850,
                0x1.13e04050dba28p503,
                0x1.13e04050dba28p+504,
                0x1.13e385f9d335ap-139,
                0x1.13ee8e95fc3ccp-362,
                0x1.13f7a84c503cp-512,
                0x1.13f7a84c503cp-513,
                0x1.13f7a84c503cp-514,
                0x1.13f7a84c503cp-515,
                0x1.13f7a84c503cp-516,
                0x1.13f8c02ea103p-505,
                0x1.13f8c02ea103p-506,
                0x1.13fc6fd662f18p-976,
                0x1.14025d6879059p-349,
                0x1.1402b640cfd58p402,
                0x1.14048c0c8e4aap657,
                0x1.14048c0c8e4aap658,
                0x1.14048c0c8e4aap659,
                0x1.1408a803cbf34p+924,
                0x1.140e95c10f357p-404,
                0x1.140e95c10f357p-405,
                0x1.1411e1f17e1e2p-226,
                0x1.1419956470536p455,
                0x1.1419aff3c39cep-784,
                0x1.1419aff3c39cep-785,
                0x1.1419aff3c39cep-786,
                0x1.1419aff3c39cep-787,
                0x1.14202b9d7c86p-1025,
                0x1.1421eaf1bc64ep+840,
                0x1.142cdfe0e5a1p-263,
                0x1.143116f045c55p-601,
                0x1.1433154d58ecp231,
                0x1.14343eba6204bp153,
                0x1.143aa13731a72p521,
                0x1.143aa13731a72p522,
                0x1.143aa13731a72p523,
                0x1.143e581b21d4ep-918,
                0x1.143e581b21d4ep-919,
                0x1.143e581b21d4ep-920,
                0x1.14588e750fe87p-226,
                0x1.14588e750fe87p-227,
                0x1.145b7e285bf98p259,
                0x1.145b7e285bf99p258,
                0x1.145b7e285bf99p259,
                0x1.14636a414b32ap-941,
                0x1.14638e42f4aafp593,
                0x1.14638e42f4aafp594,
                0x1.1469cba39ee6ep-540,
                0x1.1470083f89d48p-658,
                0x1.1470083f89d48p-659,
                0x1.1475d5c0c3179p-331,
                0x1.147ebcaaa62d6p505,
                0x1.147ebcaaa62d6p506,
                0x1.14828ee39c852p-758,
                0x1.14828ee39c852p-759,
                0x1.14828ee39c852p-760,
                0x1.14828ee39c852p-761,
                0x1.14828ee39c852p-762,
                0x1.14942c3fdac96p-645,
                0x1.14942c3fdac96p-646,
                0x1.1497516e07833p243,
                0x1.1497b54c02893p677,
                0x1.1498764e8e4edp945,
                0x1.14a508f351d66p-315,
                0x1.14a52dffc6799p+744,
                0x1.14b289f08fcc8p-329,
                0x1.14b37f4b51f71p-55,
                0x1.14b9bdc9204dbp-60,
                0x1.14bfa5630bbbap-154,
                0x1.14bfa5630bbbap-155,
                0x1.14bfa5630bbbap-156,
                0x1.14bfa5630bbbap-157,
                0x1.14bfa5630bbbap-158,
                0x1.14c4a2ef6851ap+784,
                0x1.14c4a2ef6851ap785,
                0x1.14c958d381ff6p-770,
                0x1.14d5be95a50f2p250,
                0x1.14d5be95a50f2p251,
                0x1.14d5be95a50f2p+252,
                0x1.14d5be95a50f2p253,
                0x1.14d717ccb05e2p-665,
                0x1.14d717ccb05e2p-666,
                0x1.14dc489a3e146p-352,
                0x1.14dc489a3e146p-353,
                0x1.14dc489a3e146p-354,
                0x1.14dc489a3e146p-355,
                0x1.14dda13615cap-256,
                0x1.14e55a2a8ac2dp533,
                0x1.14e55a2a8ac2dp534,
                0x1.14ea0793fafd2p-444,
                0x1.14ef3e205c817p-337,
                0x1.14ef3e205c817p-338,
                0x1.14f12046aca7bp-881,
                0x1.14f5967adaba2p-965,
                0x1.14fa7ddefe3ap-904,
                0x1.14fa7ddefe3ap-905,
                0x1.14fa7ddefe3ap-906,
                0x1.14fa7ddefe3ap-907,
                0x1.14fa7ddefe3ap-908,
                0x1.14fe7daf8f3aep-645,
                0x1.14p-1068,
                0x1.15001f1e20cbap850,
                0x1.150096c2ed8acp-554,
                0x1.150096c2ed8acp-555,
                0x1.1502fc46a0b78p727,
                0x1.1502fc46a0b78p+728,
                0x1.1502fc46a0b78p729,
                0x1.1502fc46a0b78p730,
                0x1.150ecaac1fd66p-350,
                0x1.1515874d3120cp710,
                0x1.1515874d3120cp711,
                0x1.1515874d3120cp+712,
                0x1.1515874d3120cp713,
                0x1.151a76a1570aep542,
                0x1.151a76a1570aep543,
                0x1.151a76a1570aep+544,
                0x1.151a76a1570aep545,
                0x1.151a76a1570aep546,
                0x1.151b29d1a6b9p-371,
                0x1.151b29d1a6b9p-372,
                0x1.15202393a2ab3p+932,
                0x1.1525ab8972cf2p442,
                0x1.1525ab8972cf2p443,
                0x1.1525ab8972cf2p+444,
                0x1.1525ab8972cf2p445,
                0x1.1528c7897c828p-919,
                0x1.1528eb225551cp-785,
                0x1.152959c84c1aap-710,
                0x1.152e371f4a686p-517,
                0x1.152e371f4a686p-518,
                0x1.15380d61c7fd8p+900,
                0x1.153e9dac7c8bdp-633,
                0x1.153e9dac7c8bdp-634,
                0x1.1544581b7dec2p-422,
                0x1.1544f226fc22cp-499,
                0x1.154721945b881p915,
                0x1.154da32cc4e7ep133,
                0x1.1554acd6fab8fp374,
                0x1.1554acd6fab8fp375,
                0x1.155515fd37265p-718,
                0x1.155515fd37265p-719,
                0x1.155515fd37265p-720,
                0x1.155630d497c18p305,
                0x1.155813edbafb6p+884,
                0x1.155c3e21b3dbep866,
                0x1.155c762828fa7p-488,
                0x1.155c9f6646642p705,
                0x1.155c9f6646642p706,
                0x1.155c9f6646642p707,
                0x1.15665f50b23ccp-394,
                0x1.15681ff9ca6e2p+160,
                0x1.1570d819c3f53p-512,
                0x1.1575e20c31112p-673,
                0x1.157a604ed01ap-39,
                0x1.157a604ed01ap-40,
                0x1.157a604ed01ap-41,
                0x1.1581609d62131p-772,
                0x1.1581609d62131p-773,
                0x1.158418eb10232p562,
                0x1.158e460913dp63,
                0x1.1591c31be2d2ep782,
                0x1.1592429784915p-50,
                0x1.1592ee3b29dc8p714,
                0x1.159345e553c64p533,
                0x1.159ca364e6f1cp-734,
                0x1.159ca364e6f1cp-735,
                0x1.159ca364e6f1cp-736,
                0x1.159ca364e6f1cp-737,
                0x1.15a1097e87f3bp+608,
                0x1.15a2e819a0088p+792,
                0x1.15a2e819a0088p793,
                0x1.15a2e819a0088p794,
                0x1.15a2e819a0088p795,
                0x1.15a2e819a0088p+796,
                0x1.15b564e27384ap-810,
                0x1.15b564e27384ap-811,
                0x1.15b7104a718efp950,
                0x1.15b80f6dfc4a4p211,
                0x1.15b80f6dfc4a4p+212,
                0x1.15b80f6dfc4a4p213,
                0x1.15b80f6dfc4a4p214,
                0x1.15b80f6dfc4a4p215,
                0x1.15be4615e2edap-552,
                0x1.15be4615e2edap-553,
                0x1.15c17ba3ab543p906,
                0x1.15cb665bc97fap-69,
                0x1.15cc14a2aceb6p1002,
                0x1.15cc14a2aceb6p1003,
                0x1.15cc14a2aceb6p+1004,
                0x1.15d4491c0146ap465,
                0x1.15d847ad00087p+548,
                0x1.15d847ad00088p+548,
                0x1.15d847ad00088p549,
                0x1.15d847ad00088p550,
                0x1.15d847ad00088p551,
                0x1.15d847ad00088p+552,
                0x1.15d847ad00088p553,
                0x1.15dc8e5ed41b4p+936,
                0x1.15dd53cbdef8ep-267,
                0x1.15df038dcd5c8p-717,
                0x1.15e20edf8f553p-330,
                0x1.15e20edf8f553p-331,
                0x1.15e6a8dd577cep-251,
                0x1.15e93c1bae21dp-498,
                0x1.15ee17fdfdf08p694,
                0x1.15f5d6c5f0c4cp741,
                0x1.15fc31275351ep-578,
                0x1.15fc31275351ep-579,
                0x1.15fc31275351ep-580,
                0x1.15fc31275351ep-581,
                0x1.15fc31275351ep-582,
                0x1.15fc31275351ep-583,
                0x1.15fc31275351ep-584,
                0x1.15fc31275351ep-585,
                0x1.15fc834e216bep+476,
                0x1.15fd1aab923b4p-789,
                0x1.15fd982c0022ep+332,
                0x1.15fd982c0022ep333,
                0x1.15fd982c0022ep334,
                0x1.15fd982c0022ep335,
                0x1.15fdc1aa8ca22p243,
                0x1.1603ca9857344p+240,
                0x1.1604d4ba2c14cp-601,
                0x1.1604d4ba2c14cp-602,
                0x1.16235e4ep-1041,
                0x1.1630c2126556ep234,
                0x1.16393a6d3f2dfp-164,
                0x1.163d17fd98926p-737,
                0x1.163fa24acd7a4p517,
                0x1.16401a5a2df69p-791,
                0x1.16429e8b8f9eap131,
                0x1.16475692d8a28p-594,
                0x1.164995317cd4cp375,
                0x1.164995317cd4cp+376,
                0x1.164995317cd4cp377,
                0x1.164af33362d16p+840,
                0x1.164b567ab5631p-952,
                0x1.164b567ab5631p-953,
                0x1.1653377498584p338,
                0x1.1659dcd7eb4cp899,
                0x1.165a98d2d6b86p-128,
                0x1.165a98d2d6b86p-129,
                0x1.165bca067f70ap289,
                0x1.165bca067f70ap290,
                0x1.165c3007bb89p-181,
                0x1.165ee4883b122p+848,
                0x1.166031ebccc4ap-248,
                0x1.166031ebccc4ap-249,
                0x1.166244fb75ca3p-833,
                0x1.166244fb75ca3p-834,
                0x1.16631a1d29fbap-133,
                0x1.16631a1d29fbap-134,
                0x1.16663e6824bd1p839,
                0x1.1667a24ef9754p+280,
                0x1.166b9ba449a73p-140,
                0x1.166e3f6ab1148p375,
                0x1.166e984d49783p+436,
                0x1.1672c5aa48476p+572,
                0x1.167694529a965p825,
                0x1.1678227871f3ep-616,
                0x1.1678227871f3ep-617,
                0x1.1678227871f3ep-618,
                0x1.167cef770e718p398,
                0x1.1680134e0852p-623,
                0x1.1680134e0852p-624,
                0x1.168289c11315p709,
                0x1.168927756df67p-172,
                0x1.169070daad282p-965,
                0x1.169e9b6071ad8p-241,
                0x1.16a3f451e6d46p-462,
                0x1.16a3f451e6d46p-463,
                0x1.16a55ccbcd46cp-761,
                0x1.16a55ccbcd46cp-762,
                0x1.16a67f3f6c08cp634,
                0x1.16a67f3f6c08cp635,
                0x1.16a67f3f6c08cp+636,
                0x1.16a67f3f6c08cp637,
                0x1.16a67f3f6c08cp638,
                0x1.16a7dc6717d52p657,
                0x1.16a8093df66a7p-29,
                0x1.16a8b49d30458p-362,
                0x1.16ac3469fab2cp547,
                0x1.16add48fd03dep-849,
                0x1.16b2443b4b249p-410,
                0x1.16bd9553274d6p-929,
                0x1.16bd9553274d6p-930,
                0x1.16c262777579cp-132,
                0x1.16c262777579cp-133,
                0x1.16c315cab39d5p-315,
                0x1.16c7696f8674ep-383,
                0x1.16c7696f8674ep-384,
                0x1.16cc335bbc0fep-100,
                0x1.16cc335bbc0fep-101,
                0x1.16cc335bbc0fep-102,
                0x1.16d97e64d6755p-872,
                0x1.16d9c31c4da8ep438,
                0x1.16d9c31c4da8ep439,
                0x1.16e27be5ca6f1p345,
                0x1.16e43e9fa729p-412,
                0x1.16ea804faf00fp-933,
                0x1.170170a435b53p673,
                0x1.1701b8fc1b334p715,
                0x1.1707c02068784p-898,
                0x1.1707c02068784p-899,
                0x1.170b07e7b13a4p+292,
                0x1.170b3516afdcp470,
                0x1.170cb642b133ep351,
                0x1.170cb642b133ep+352,
                0x1.170cb642b133ep353,
                0x1.170cb642b133ep354,
                0x1.170f0378af01p1022,
                0x1.17100fb4f4bc5p-360,
                0x1.1716267a4214ap1017,
                0x1.171b2ba3b52c3p-421,
                0x1.171d50c54d9f2p954,
                0x1.17243f9713dd4p-981,
                0x1.1724847955f9ep895,
                0x1.1724847955f9ep+896,
                0x1.172b70eababaap222,
                0x1.172b70eababaap223,
                0x1.172b70eababaap+224,
                0x1.172b70eababaap225,
                0x1.172b70eababaap226,
                0x1.172c417c771efp-23,
                0x1.172fd8f381a16p-683,
                0x1.1736b35ba47p-225,
                0x1.17390afb040f3p713,
                0x1.173f48217a47p911,
                0x1.173f76c63b792p934,
                0x1.173f76c63b792p935,
                0x1.173f76c63b792p+936,
                0x1.173f76c63b792p937,
                0x1.173f76c63b792p938,
                0x1.173f76c63b792p939,
                0x1.173f76c63b792p+940,
                0x1.173f76c63b792p941,
                0x1.1752aa924e55dp+736,
                0x1.1752aa924e55dp737,
                0x1.1752aa924e55dp738,
                0x1.17571ddf6c814p835,
                0x1.17571ddf6c814p+836,
                0x1.17571ddf6c814p837,
                0x1.17571ddf6c814p838,
                0x1.17571ddf6c814p839,
                0x1.17571ddf6c814p+840,
                0x1.17571ddf6c814p841,
                0x1.1758ef60ba8eap-939,
                0x1.175905cp-1048,
                0x1.175d77fd116dcp+928,
                0x1.1765d7a9903b9p-470,
                0x1.1769eea5557ccp+152,
                0x1.1769eea5557ccp153,
                0x1.176ad27483b6cp598,
                0x1.176ad27483b6cp599,
                0x1.176ad27483b6cp+600,
                0x1.176ad27483b6cp601,
                0x1.176b31593543p-719,
                0x1.176b31593543p-720,
                0x1.1771d927a2fe1p-687,
                0x1.1775458b899cap+644,
                0x1.17787ef98b4edp622,
                0x1.177c07867e9cfp383,
                0x1.178187fd3e01p-602,
                0x1.1786ae5e745c5p-524,
                0x1.1786ae5e745c5p-525,
                0x1.1789ac1ac8304p-34,
                0x1.178cacfa5fba8p217,
                0x1.17908a885203ep729,
                0x1.179c3010ca5b7p+672,
                0x1.17a397ff0d604p118,
                0x1.17a397ff0d604p119,
                0x1.17a397ff0d604p+120,
                0x1.17a9edfcd2e0cp482,
                0x1.17aa0d0c35ae8p-596,
                0x1.17aa0d0c35ae8p-597,
                0x1.17aa0d0c35ae8p-598,
                0x1.17aa0d0c35ae8p-599,
                0x1.17aa0d0c35ae8p-600,
                0x1.17accc17d1d53p+876,
                0x1.17accc17d1d53p877,
                0x1.17ad428200c08p-813,
                0x1.17ad428200c08p-814,
                0x1.17af9eb35b098p-452,
                0x1.17af9eb35b098p-453,
                0x1.17af9eb35b098p-454,
                0x1.17af9eb35b098p-455,
                0x1.17af9eb35b098p-456,
                0x1.17b4a2219ef67p-481,
                0x1.17b529c232d58p-344,
                0x1.17b56b677adb5p-479,
                0x1.17bbbc2c4f56p-827,
                0x1.17bbbc2c4f56p-828,
                0x1.17c1b42848e1ap989,
                0x1.17c1b42848e1ap990,
                0x1.17c1b42848e1ap991,
                0x1.17c1b42848e1ap+992,
                0x1.17c1b42848e1ap993,
                0x1.17c1b42848e1ap994,
                0x1.17c1b42848e1ap995,
                0x1.17c1b42848e1ap+996,
                0x1.17cda3e0e8d5dp-310,
                0x1.17cda3e0e8d5dp-311,
                0x1.17cda3e0e8d5dp-312,
                0x1.17ce8c7c7e082p-341,
                0x1.17ce8c7c7e082p-342,
                0x1.17ce8c7c7e082p-343,
                0x1.17ce8c7c7e082p-344,
                0x1.17ceacf74cfb1p722,
                0x1.17ceade74428p-1029,
                0x1.17d361aec2532p-31,
                0x1.17d65f9f7ecb4p-707,
                0x1.17d65f9f7ecb4p-708,
                0x1.17d65f9f7ecb4p-709,
                0x1.17dea0942bcb2p-152,
                0x1.17e05a8e5c56cp-357,
                0x1.17ece9f77p-1035,
                0x1.17f1928c70ff8p-359,
                0x1.17f7d4ed8c33ep-329,
                0x1.180072606941ap-133,
                0x1.1807686bc9ebep-159,
                0x1.1807686bc9ebep-160,
                0x1.180b086b7f042p-907,
                0x1.1810e9d1306c4p186,
                0x1.18177abb4a5c8p-704,
                0x1.1818ab41d410dp939,
                0x1.1820d636c4881p582,
                0x1.182316ef9c262p+756,
                0x1.182316ef9c262p757,
                0x1.1823408adf3cp294,
                0x1.182baf96e0fcap+776,
                0x1.182baf96e0fcap777,
                0x1.182c487a6bbe8p355,
                0x1.1836466bae15ep534,
                0x1.1840d929743c8p+916,
                0x1.18427b3b4a05cp+156,
                0x1.18452f93667d3p-237,
                0x1.18452f93667d3p-238,
                0x1.184a0b1b9fb4ap-713,
                0x1.184cc09ecfdb7p-779,
                0x1.184d7c8dc20d8p679,
                0x1.184d7c8dc20d8p+680,
                0x1.184d8b28ac51cp-755,
                0x1.184e29fe41dcap+376,
                0x1.185793412ad84p919,
                0x1.1858a3f7e1cebp-653,
                0x1.185a894293b62p-579,
                0x1.185d778bfaf6ap226,
                0x1.185e40c12aea1p-1003,
                0x1.18640e490b087p-24,
                0x1.18657b11ca8p-1033,
                0x1.186c3ab67a857p-546,
                0x1.18705a0e02404p566,
                0x1.1877fe312a6c8p-229,
                0x1.18824812a55fcp709,
                0x1.188a8144087e7p-544,
                0x1.188d357087712p641,
                0x1.188d79c402798p-444,
                0x1.188dc0a7df32ep-22,
                0x1.1890784f0d8f8p-174,
                0x1.1894043285abep-448,
                0x1.1895e1141dfd4p-411,
                0x1.189b69273cbe3p-111,
                0x1.189ea7ccf701ep-524,
                0x1.189ea7ccf701ep-525,
                0x1.189ea7ccf701ep-526,
                0x1.189ea7ccf701ep-527,
                0x1.18a423a897a4p-564,
                0x1.18a423a897a4p-565,
                0x1.18a7f36172332p434,
                0x1.18a7f36172332p435,
                0x1.18aa91de8d154p839,
                0x1.18aa91de8d154p+840,
                0x1.18abbc2f0420fp-608,
                0x1.18acb4dbcbe96p-536,
                0x1.18acb4dbcbe96p-537,
                0x1.18ae6851e84afp-991,
                0x1.18b27a759405dp551,
                0x1.18c03f725c2e8p-92,
                0x1.18c1f966304f8p735,
                0x1.18c3b89731f3dp-21,
                0x1.18c6d4d3b9c42p794,
                0x1.18d28218b7f2cp-253,
                0x1.18e0ec3e30ea9p+284,
                0x1.18e1da59807d1p238,
                0x1.18e3b9b374169p-1010,
                0x1.18e3bb6cab09ep+544,
                0x1.18e5936aaa459p431,
                0x1.18e6a1befep-1035,
                0x1.18e7dea3444dap+804,
                0x1.18ed57a80a254p-502,
                0x1.18f13fedec7c6p-331,
                0x1.18f5162b5ed57p-472,
                0x1.18f5162b5ed57p-473,
                0x1.18f6110c779b1p+648,
                0x1.18f7eafc5c7a6p+172,
                0x1.1904b88c6978ap729,
                0x1.1904b88c6978ap730,
                0x1.19078a913357ep-41,
                0x1.19078a913357ep-42,
                0x1.19099e254c45p910,
                0x1.1913ca1d87308p-406,
                0x1.1913ca1d87308p-407,
                0x1.19197f95b1cedp877,
                0x1.1922b49ee5a85p553,
                0x1.19261497e586bp-950,
                0x1.1926c7a7122bap963,
                0x1.1926c7a7122bap+964,
                0x1.1926c7a7122bap965,
                0x1.1926c7a7122bap966,
                0x1.192747365bb29p-692,
                0x1.192789300f9c2p-655,
                0x1.1929f2a1e0f3ap-998,
                0x1.192e9ee706e4ap-525,
                0x1.192f73ff299ep-602,
                0x1.192f73ff299ep-603,
                0x1.192f73ff299ep-604,
                0x1.19317272fb766p+812,
                0x1.1939e65fa4ebep361,
                0x1.193aff1f1c8e3p545,
                0x1.193aff1f1c8e3p546,
                0x1.193aff1f1c8e3p547,
                0x1.19459855997b8p-749,
                0x1.19492aa4cf38cp+880,
                0x1.194a955d5f05ep-482,
                0x1.194a955d5f05ep-483,
                0x1.194b925b7ed8ep-419,
                0x1.194e3014d914dp-230,
                0x1.194e67d67a32ep-900,
                0x1.194e97e756fccp-485,
                0x1.1958b36c5102cp305,
                0x1.1958b36c5102cp306,
                0x1.1958b36c5102cp307,
                0x1.1958b36c5102cp+308,
                0x1.1958b36c5102cp309,
                0x1.1958b36c5102cp310,
                0x1.1958b36c5102cp311,
                0x1.1958b36c5102cp+312,
                0x1.1958b36c5102cp313,
                0x1.1958b36c5102cp314,
                0x1.1959a0e2c294ap234,
                0x1.1959a0e2c294ap235,
                0x1.195c52ea77cf2p-675,
                0x1.195c52ea77cf2p-676,
                0x1.195c8ea8d6fdap706,
                0x1.195c8ea8d6fdap707,
                0x1.195cc1d3a6a2ep259,
                0x1.195cc1d3a6a2ep+260,
                0x1.195cc1d3a6a2ep261,
                0x1.195cc1d3a6a2ep262,
                0x1.196b613c69364p-644,
                0x1.196b613c69364p-645,
                0x1.196b613c69364p-646,
                0x1.196fb5c9c019ap-1017,
                0x1.19710dc581911p-967,
                0x1.197270d80000ap-77,
                0x1.197270d80000ap-78,
                0x1.19799812dea11p-40,
                0x1.197b290631476p-468,
                0x1.197b290631476p-469,
                0x1.197b290631476p-470,
                0x1.197b290631476p-471,
                0x1.197b290631476p-472,
                0x1.19846b59ebc16p702,
                0x1.19860d7a88992p+800,
                0x1.19860d7a88992p801,
                0x1.19860d7a88992p802,
                0x1.19860d7a88992p803,
                0x1.19860d7a88992p+804,
                0x1.19860d7a88992p805,
                0x1.19883163a4666p-235,
                0x1.19883163a4666p-236,
                0x1.198bb8596bb71p510,
                0x1.198fd0024033p797,
                0x1.198fd0024033p798,
                0x1.19903351494a2p-302,
                0x1.199035a075f8ap-660,
                0x1.1995f04a5c034p1013,
                0x1.1995f04a5c034p1014,
                0x1.199dbf6cf8a42p751,
                0x1.199dbf6cf8a42p+752,
                0x1.19a90e4f70b66p-732,
                0x1.19ac7e5b79a5ep638,
                0x1.19ac7e5b79a5ep639,
                0x1.19ac7e5b79a5ep+640,
                0x1.19ac7e5b79a5ep641,
                0x1.19ac7e5b79a5ep642,
                0x1.19aedc8347409p+128,
                0x1.19b7bff34aa4ep-386,
                0x1.19b8744033457p-738,
                0x1.19b8744033457p-739,
                0x1.19b8744033457p-740,
                0x1.19b96f36ec68bp-19,
                0x1.19b96f36ec68bp-20,
                0x1.19b96f36ec68bp-21,
                0x1.19ba27a59b496p+528,
                0x1.19c03c40b218p225,
                0x1.19c21261f844ep510,
                0x1.19c21261f844ep511,
                0x1.19c21261f844ep+512,
                0x1.19c21261f844ep513,
                0x1.19c4a53c4e697p445,
                0x1.19c4a53c4e698p445,
                0x1.19c4a53c4e698p446,
                0x1.19cbb887923e9p-147,
                0x1.19ccddc08f73ap+780,
                0x1.19ce075f6fd22p-8,
                0x1.19cfef312de86p-53,
                0x1.19d15d3821788p345,
                0x1.19dc138f8a8f5p433,
                0x1.19dc138f8a8f5p434,
                0x1.19dc32fe7c524p674,
                0x1.19e841fd5769ap-839,
                0x1.19e841fd5769ap-840,
                0x1.19e841fd5769ap-841,
                0x1.19ea208178105p254,
                0x1.19eed3d929188p-136,
                0x1.19fb2b5b153d6p-574,
                0x1.19fb2b5b153d6p-575,
                0x1.1a01d8067815ep-752,
                0x1.1a01d8067815ep-753,
                0x1.1a05622ace3bcp-268,
                0x1.1a0fc668aac7p930,
                0x1.1a103a7b52a7ep350,
                0x1.1a149dd13c3c6p-84,
                0x1.1a151ddbd3c4ap+264,
                0x1.1a151ddbd3c4ap265,
                0x1.1a1d9135cca54p287,
                0x1.1a1d9135cca54p+288,
                0x1.1a1d9135cca54p289,
                0x1.1a1d9135cca54p290,
                0x1.1a1d9135cca54p291,
                0x1.1a1e5f7753796p131,
                0x1.1a247a7d801a4p-1006,
                0x1.1a26871e1fd9cp+620,
                0x1.1a26eccdf70f2p-920,
                0x1.1a2f647f6b546p-644,
                0x1.1a379bf0f90c6p906,
                0x1.1a40839cfc202p487,
                0x1.1a40839cfc202p+488,
                0x1.1a47c20818216p+724,
                0x1.1a498083c888dp595,
                0x1.1a498083c888dp+596,
                0x1.1a4fbf063815p595,
                0x1.1a58cb30c6a1ep795,
                0x1.1a58cb30c6a1ep+796,
                0x1.1a5a539ed4e98p1017,
                0x1.1a5a539ed4e98p1018,
                0x1.1a5dfc073cb38p-102,
                0x1.1a5dfc073cb38p-103,
                0x1.1a61eed452c0bp-922,
                0x1.1a6462c2142ebp962,
                0x1.1a66c1e139edcp-720,
                0x1.1a66c1e139edcp-721,
                0x1.1a6795e5c2fbcp95,
                0x1.1a6cc949607d8p+700,
                0x1.1a730031c099cp163,
                0x1.1a730031c099cp+164,
                0x1.1a746167ce55ep-833,
                0x1.1a746167ce55ep-834,
                0x1.1a78f9f97d515p469,
                0x1.1a7b5e3e39ff1p591,
                0x1.1a7d848f98ccp-1028,
                0x1.1a7eab4f5ebecp-819,
                0x1.1a7eab4f5ebecp-820,
                0x1.1a7eab4f5ebecp-821,
                0x1.1a7eab4f5ebecp-822,
                0x1.1a7eab4f5ebecp-823,
                0x1.1a7f08dfda4adp923,
                0x1.1a83acdce9c5ep-723,
                0x1.1a87ee30ec2ecp1018,
                0x1.1a885b37d81ccp622,
                0x1.1a89b5616366ap+656,
                0x1.1a8b4b519a2bdp+720,
                0x1.1a8be8307d3e8p-844,
                0x1.1a8c1c10a1fc6p-57,
                0x1.1a8c1c10a1fc6p-58,
                0x1.1a8e2afb7b5d3p-736,
                0x1.1a90bbebea43p-976,
                0x1.1a90bbebea43p-977,
                0x1.1aa3bfc064646p-137,
                0x1.1aa820d48ac0ep+108,
                0x1.1aa85d60fa25ep-114,
                0x1.1aa85d60fa25ep-115,
                0x1.1aaf9a1ca25b4p-388,
                0x1.1ab20e472914ap-234,
                0x1.1ab20e472914ap-235,
                0x1.1ab20e472914ap-236,
                0x1.1ab4daf21d8d6p-600,
                0x1.1ab4daf21d8d6p-601,
                0x1.1aba8129d24ap365,
                0x1.1ac0afbbe4p-1035,
                0x1.1ac0afbbe4p-1036,
                0x1.1ac3662843f7ap-748,
                0x1.1ac3662843f7ap-749,
                0x1.1ac48cd134013p-329,
                0x1.1ac63ad922ff8p-321,
                0x1.1ac63ad922ff8p-322,
                0x1.1ad293b46ad91p+428,
                0x1.1ad8944d4c67ep-262,
                0x1.1adc84fe5a7ccp731,
                0x1.1adef745cf426p125,
                0x1.1adf22c915a94p-300,
                0x1.1adfa2d11f994p367,
                0x1.1adfa2d11f994p+368,
                0x1.1ae15e87423a3p-726,
                0x1.1ae27b3b6c2f8p-235,
                0x1.1ae27b3b6c2f8p-236,
                0x1.1ae27b3b6c2f8p-237,
                0x1.1ae601cad6dep427,
                0x1.1ae6ff4ee9d83p214,
                0x1.1ae7c467331e3p-481,
                0x1.1ae7c67d05324p966,
                0x1.1ae7c67d05324p967,
                0x1.1ae7db7dc138ap-925,
                0x1.1aee84a1ea904p622,
                0x1.1af219f15a3dbp-115,
                0x1.1af3dd1f1a8b8p275,
                0x1.1af664915a7d8p+760,
                0x1.1af994ac12d14p-529,
                0x1.1afd6ec0e1411p249,
                0x1.1afd6ec0e1412p249,
                0x1.1afd6ec0e1412p250,
                0x1.1b0c141c489a8p-550,
                0x1.1b0cp-1059,
                0x1.1b13af62ea26p-492,
                0x1.1b148ca59d9aep-341,
                0x1.1b1a2f57121d4p-549,
                0x1.1b1a398d23a1p-1026,
                0x1.1b1fd4f8d287cp-726,
                0x1.1b20c5e0b1506p-937,
                0x1.1b20c5e0b1506p-938,
                0x1.1b20c5e0b1506p-939,
                0x1.1b20c5e0b1506p-940,
                0x1.1b20c5e0b1506p-941,
                0x1.1b21c65f4d10dp+796,
                0x1.1b21fb2c71324p-285,
                0x1.1b25ac4f5037p-510,
                0x1.1b25b967f8ce9p-869,
                0x1.1b280d38d706fp-356,
                0x1.1b2a72c9d60dep-622,
                0x1.1b2a72c9d60dep-623,
                0x1.1b324968b342ep-49,
                0x1.1b348772e70e2p-802,
                0x1.1b3bfbad72f8p390,
                0x1.1b3c5c7008bdfp686,
                0x1.1b3c7f0b566e6p-637,
                0x1.1b3c7f0b566e6p-638,
                0x1.1b3efddcf9235p777,
                0x1.1b445de6d1d5cp+912,
                0x1.1b4596e77f06dp654,
                0x1.1b48ab1beef26p-163,
                0x1.1b48e353bce7p734,
                0x1.1b5087b3c6473p333,
                0x1.1b5087b3c6473p334,
                0x1.1b56f9c090dfcp+116,
                0x1.1b56f9c090dfcp117,
                0x1.1b56f9c090dfcp118,
                0x1.1b56f9c090dfcp119,
                0x1.1b56f9c090dfcp+120,
                0x1.1b578c96db19ap-63,
                0x1.1b578c96db19ap-64,
                0x1.1b578c96db19ap-65,
                0x1.1b578c96db19ap-66,
                0x1.1b57f08946678p857,
                0x1.1b5a3f0eba1aap133,
                0x1.1b5cc0d0f30b8p839,
                0x1.1b5cc0d0f30b8p+840,
                0x1.1b5cc0d0f30b8p841,
                0x1.1b5e951966cb1p-887,
                0x1.1b6231e18c5cap433,
                0x1.1b6231e18c5cap434,
                0x1.1b6231e18c5cap435,
                0x1.1b6231e18c5cap+436,
                0x1.1b6231e18c5cap437,
                0x1.1b6231e18c5cap438,
                0x1.1b6231e18c5cap439,
                0x1.1b6231e18c5cap+440,
                0x1.1b6231e18c5cap441,
                0x1.1b6231e18c5cap442,
                0x1.1b6231e18c5cap443,
                0x1.1b6231e18c5cap+444,
                0x1.1b6c61e534819p346,
                0x1.1b7622bbe7954p-205,
                0x1.1b7622bbe7954p-206,
                0x1.1b778e6b97852p-599,
                0x1.1b778e6b97852p-600,
                0x1.1b77b17026708p298,
                0x1.1b77b17026708p299,
                0x1.1b7bf91348859p-423,
                0x1.1b7c71d2a7c88p-83,
                0x1.1b7c71d2a7c88p-84,
                0x1.1b7c71d2a7c88p-85,
                0x1.1b7c71d2a7c88p-86,
                0x1.1b7eb61c2851ep+488,
                0x1.1b7eb61c2851ep489,
                0x1.1b7eb61c2851ep490,
                0x1.1b7eb61c2851ep491,
                0x1.1b8f6bb7b9b8ap+844,
                0x1.1b9563d59ce8cp-423,
                0x1.1b9563d59ce8cp-424,
                0x1.1b9563d59ce8cp-425,
                0x1.1b96458445d07p-343,
                0x1.1b96458445d07p-344,
                0x1.1ba03f5b21p-913,
                0x1.1ba03f5b21p-914,
                0x1.1ba03f5b21p-915,
                0x1.1ba03f5b21p-916,
                0x1.1ba03f5b21p-917,
                0x1.1ba03f5b21p-918,
                0x1.1ba1418a326cp974,
                0x1.1ba1418a326cp975,
                0x1.1ba15ba3b51d2p-64,
                0x1.1ba15ba3b51d2p-65,
                0x1.1ba15ba3b51d2p-66,
                0x1.1ba15ba3b51d2p-67,
                0x1.1ba31930f783p+420,
                0x1.1ba3c16eaafccp-24,
                0x1.1ba650f4ce6aap-282,
                0x1.1ba650f4ce6aap-283,
                0x1.1ba92f1aa703cp-763,
                0x1.1ba92f1aa703cp-764,
                0x1.1bad0b63018c4p926,
                0x1.1bb058be9684cp555,
                0x1.1bb64ecf0574ap+1004,
                0x1.1bbb6b796ab12p585,
                0x1.1bbc0269ef752p-105,
                0x1.1bc6d8p-1052,
                0x1.1bc6d8p-1053,
                0x1.1bcb58017924bp-714,
                0x1.1bcfa78a8d00ap-327,
                0x1.1bd2d90732b8ap842,
                0x1.1be00b39d8513p-761,
                0x1.1beac18dd37fcp793,
                0x1.1beac18dd37fcp794,
                0x1.1bebdf578b2f4p-430,
                0x1.1bebdf578b2f4p-431,
                0x1.1bebdf578b2f4p-432,
                0x1.1becbfee4413ap-793,
                0x1.1bef1535ca0c4p265,
                0x1.1befa0880f8d7p-1008,
                0x1.1bf49682e5c24p-361,
                0x1.1c12b9f6dad55p129,
                0x1.1c131d5cfbb74p121,
                0x1.1c131d5cfbb74p122,
                0x1.1c1feb690caf6p-725,
                0x1.1c1feb690caf6p-726,
                0x1.1c229e90c739p-663,
                0x1.1c24e7192622ep403,
                0x1.1c2ac735358d5p538,
                0x1.1c33150e14b59p419,
                0x1.1c34e4648ef1bp+656,
                0x1.1c37937e08p53,
                0x1.1c3a41e5117cep-602,
                0x1.1c401d16b44d2p-424,
                0x1.1c4daf7578294p-645,
                0x1.1c4daf7578294p-646,
                0x1.1c57338da1ffp174,
                0x1.1c57338da1ffp175,
                0x1.1c57338da1ffp+176,
                0x1.1c57338da1ffp177,
                0x1.1c57ebdacc8e7p395,
                0x1.1c5cb19ef3452p+408,
                0x1.1c60600769ba7p982,
                0x1.1c6e11f836908p-567,
                0x1.1c731209617e2p230,
                0x1.1c731209617e2p231,
                0x1.1c760a1e13d56p861,
                0x1.1c760a1e13d56p862,
                0x1.1c798b94af66ep390,
                0x1.1c7cb60c3aa42p-936,
                0x1.1c7cb60c3aa42p-937,
                0x1.1c835bd3f7d78p538,
                0x1.1c84894d0f761p965,
                0x1.1c88024fc6894p610,
                0x1.1c88024fc6894p611,
                0x1.1c8d66ce9677p857,
                0x1.1c926ec5ef03ep-750,
                0x1.1c988c8bc082ep285,
                0x1.1cb209aeab9f6p255,
                0x1.1cc242c9faddep-947,
                0x1.1cc6f46ac5cffp-887,
                0x1.1ccbdb8fdbdc7p+540,
                0x1.1ccf385ebc8ap1022,
                0x1.1ccf385ebc8ap1023,
                0x1.1ccfb049f7b87p-364,
                0x1.1ccfb049f7b87p-365,
                0x1.1cd31704d458ep557,
                0x1.1cd5aa423b0b4p511,
                0x1.1cd5cd90cb2p-636,
                0x1.1cd5cd90cb2p-637,
                0x1.1cd5cd90cb2p-638,
                0x1.1cd5cd90cb2p-639,
                0x1.1cd5cd90cb2p-640,
                0x1.1cd5cd90cb2p-641,
                0x1.1cdbfc322619p695,
                0x1.1ce107a71a97ap995,
                0x1.1ce51268b3b58p-161,
                0x1.1ce51268b3b58p-162,
                0x1.1ce51268b3b58p-163,
                0x1.1ce51268b3b58p-164,
                0x1.1cea04f5074cp422,
                0x1.1ceba2bec8p-1035,
                0x1.1ceba2bec8p-1036,
                0x1.1ceba2bec8p-1037,
                0x1.1cebb55ad72a8p-809,
                0x1.1cebb55ad72a8p-810,
                0x1.1cee9191a5292p993,
                0x1.1cefb1da33c29p-294,
                0x1.1cf1f3f6b7882p-93,
                0x1.1cf1f3f6b7882p-94,
                0x1.1cf27772f980ep-516,
                0x1.1cf280d13df82p+876,
                0x1.1cf2f378779acp851,
                0x1.1cf2f378779acp+852,
                0x1.1cf5268f02ac4p+364,
                0x1.1cf9aecb75301p-869,
                0x1.1cf9c513f54fep+468,
                0x1.1cfd009f3399ep-639,
                0x1.1cfd009f3399ep-640,
                0x1.1cfd009f3399ep-641,
                0x1.1cfd009f3399ep-642,
                0x1.1cfd009f3399ep-643,
                0x1.1d076f4784bb7p+684,
                0x1.1d09a23c527b2p118,
                0x1.1d0aae9ef70bcp-876,
                0x1.1d0b4ec3b6fa3p-173,
                0x1.1d0c83f6378a6p-367,
                0x1.1d0c83f6378a6p-368,
                0x1.1d0c83f6378a6p-369,
                0x1.1d0c83f6378a6p-370,
                0x1.1d12f62879591p290,
                0x1.1d15ce7d933bep-81,
                0x1.1d15ce7d933bep-82,
                0x1.1d15ce7d933bep-83,
                0x1.1d15ce7d933bep-84,
                0x1.1d15ce7d933bep-85,
                0x1.1d15ce7d933bep-86,
                0x1.1d19a81649b7dp335,
                0x1.1d270cc51055ep-627,
                0x1.1d270cc51055ep-628,
                0x1.1d33699e6303ap+256,
                0x1.1d349ede0e7ebp+760,
                0x1.1d3b7f802854ep-888,
                0x1.1d467e94b856ep-751,
                0x1.1d48f66fbd13fp841,
                0x1.1d48f66fbd13fp842,
                0x1.1d5125d904b14p-673,
                0x1.1d51727564daap-508,
                0x1.1d57e619dd508p110,
                0x1.1d5f3472ec322p-336,
                0x1.1d5ffd52f681cp774,
                0x1.1d6224aec26e8p-675,
                0x1.1d669acc69901p-297,
                0x1.1d6e5688419ap-616,
                0x1.1d70eff7edadfp718,
                0x1.1d7314f534b6p-142,
                0x1.1d7314f534b6p-143,
                0x1.1d7c967ad1f7p-596,
                0x1.1d7cb7eae05d8p+796,
                0x1.1d81dae6e221cp-942,
                0x1.1d8672cd4f908p-80,
                0x1.1d8813536e0ep898,
                0x1.1d8813536e0ep899,
                0x1.1d907568758f6p-728,
                0x1.1d907568758f6p-729,
                0x1.1d95c1268e8p-422,
                0x1.1d9f2a813bcf1p-780,
                0x1.1d9fc3b4bd166p-405,
                0x1.1da9cfeda1e4ap-763,
                0x1.1dab5e589d09cp658,
                0x1.1dab5e589d09cp659,
                0x1.1dab5e589d09cp+660,
                0x1.1dap-1063,
                0x1.1db0438b265d1p-675,
                0x1.1db7324818284p-956,
                0x1.1db7324818284p-957,
                0x1.1dba345e938f4p558,
                0x1.1dbcff8ad7442p547,
                0x1.1dbf316b346e8p+340,
                0x1.1dbf316b346e8p341,
                0x1.1dbf316b346e8p342,
                0x1.1dc1c98da80f4p-455,
                0x1.1dc1c98da80f4p-456,
                0x1.1dc5ca4ca9193p-657,
                0x1.1dc8db2bfec2dp639,
                0x1.1dcdfb4d490dap-457,
                0x1.1dcefcc37deap767,
                0x1.1dcefcc37deap+768,
                0x1.1dcefcc37deap769,
                0x1.1dcefcc37deap770,
                0x1.1dcefcc37deap771,
                0x1.1dcp-1064,
                0x1.1dd1e30f9bd3ap249,
                0x1.1dd1e30f9bd3ap250,
                0x1.1dd1e30f9bd3ap251,
                0x1.1dd7d8b44c714p-45,
                0x1.1dd87f2fcaa9ap+412,
                0x1.1ddd4169222f9p-638,
                0x1.1de3ac85cp-1040,
                0x1.1de3e5d1de59cp403,
                0x1.1de3e5d1de59cp+404,
                0x1.1de88961e3664p-152,
                0x1.1deaf40ead9ap+1000,
                0x1.1deaf40ead9ap1001,
                0x1.1deaf40ead9ap999,
                0x1.1df00d1bd53c7p-686,
                0x1.1df435c00551ep354,
                0x1.1df435c00551ep355,
                0x1.1df6d0a57d51ap-824,
                0x1.1dfa463d58d4ep-214,
                0x1.1e05cb2a12026p210,
                0x1.1e05cb2a12026p211,
                0x1.1e0705479b944p327,
                0x1.1e0705479b944p+328,
                0x1.1e07f320c19adp-205,
                0x1.1e08d97d9506ap474,
                0x1.1e08d97d9506ap475,
                0x1.1e0b622c774dp827,
                0x1.1e0b622c774dp+828,
                0x1.1e0d3ed9d59p-949,
                0x1.1e10bcd3e35fcp566,
                0x1.1e10bcd3e35fcp567,
                0x1.1e1c4eccd3a28p-828,
                0x1.1e1fa8a031c94p-974,
                0x1.1e22fd4f8d48cp701,
                0x1.1e22fd4f8d48cp702,
                0x1.1e22fd4f8d48cp703,
                0x1.1e22fd4f8d48cp+704,
                0x1.1e22fd4f8d48cp705,
                0x1.1e22fd4f8d48cp706,
                0x1.1e24cd17931c3p-580,
                0x1.1e24cd17931c3p-581,
                0x1.1e2a0935a8fcfp-445,
                0x1.1e342d9713198p+856,
                0x1.1e3904b9d3a4dp885,
                0x1.1e3ce223a6f1ap-727,
                0x1.1e3ce223a6f1ap-728,
                0x1.1e48a6a442a24p965,
                0x1.1e4a643a64628p-47,
                0x1.1e4b5dbf1c307p-101,
                0x1.1e4d24e551bfep-458,
                0x1.1e4ee41180c0ap-427,
                0x1.1e515ae15dc16p-202,
                0x1.1e515ae15dc16p-203,
                0x1.1e515ae15dc16p-204,
                0x1.1e53a6993cc6ap+516,
                0x1.1e5bcc3aac1c9p+920,
                0x1.1e6398126f5cbp-824,
                0x1.1e63ceaf17bcap-639,
                0x1.1e63ceaf17bcap-640,
                0x1.1e63ceaf17bcap-641,
                0x1.1e63ceaf17bcap-642,
                0x1.1e63ceaf17bcap-643,
                0x1.1e667904707f4p513,
                0x1.1e6679fa9143ep-981,
                0x1.1e69396e7f662p-167,
                0x1.1e6e67e9ad08fp402,
                0x1.1e6ed42198c28p+848,
                0x1.1e6faba464f02p-896,
                0x1.1e6faba464f02p-897,
                0x1.1e6faba464f02p-898,
                0x1.1e6faba464f02p-899,
                0x1.1e7073f95f871p789,
                0x1.1e72df5607993p-467,
                0x1.1e73ac3d9ae56p-685,
                0x1.1e744e6027ca2p-154,
                0x1.1e754b8ed271cp-285,
                0x1.1e754b8ed271cp-286,
                0x1.1e7dd30f1065cp+732,
                0x1.1e8e8e09ffd5fp+692,
                0x1.1e90a8711440fp-639,
                0x1.1e92fdf361d3cp166,
                0x1.1e94bac61d478p711,
                0x1.1ea4ff85530f6p-813,
                0x1.1eac9447ec001p431,
                0x1.1eaf66ca0cda4p-662,
                0x1.1eaf6c9303323p-704,
                0x1.1eafd0ec348f8p-596,
                0x1.1eaff4a98553dp-339,
                0x1.1eaff4a98553ep-338,
                0x1.1eaff4a98553ep-339,
                0x1.1eb851eb851ecp-4,
                0x1.1eba8443e5974p-27,
                0x1.1ebd823310c54p-637,
                0x1.1ebd823310c54p-638,
                0x1.1ebd823310c54p-639,
                0x1.1ec01018a2344p-235,
                0x1.1ec7281776151p-825,
                0x1.1ec853afdbf04p749,
                0x1.1ec853afdbf04p750,
                0x1.1ece45eba5c04p-526,
                0x1.1ed894f6dbf89p301,
                0x1.1edbb48a08d59p+832,
                0x1.1edc686e2fcdep498,
                0x1.1eddf57877658p218,
                0x1.1ee1bf8daf76bp-660,
                0x1.1ee43b351600cp-721,
                0x1.1ee4a23ed6ce8p-1003,
                0x1.1ee57e0e12482p618,
                0x1.1ee57e0e12482p619,
                0x1.1ee8a323894ecp-584,
                0x1.1eed9dfda98dap-182,
                0x1.1eeffba37eeecp-64,
                0x1.1ef2c9de78946p-815,
                0x1.1ef58cb634beap-311,
                0x1.1ef7d320fffap-639,
                0x1.1ef8136dd12adp+848,
                0x1.1efa2f739abap387,
                0x1.1efc659cf7d4cp145,
                0x1.1efc659cf7d4cp146,
                0x1.1efc659cf7d4cp147,
                0x1.1efc659cf7d4cp+148,
                0x1.1efc659cf7d4cp149,
                0x1.1f08fdd5fd4dp-694,
                0x1.1f09c2d4e883p-940,
                0x1.1f0b965a6aba8p326,
                0x1.1f0d2aa29c0dcp-80,
                0x1.1f0d2aa29c0dcp-81,
                0x1.1f0d2aa29c0dcp-82,
                0x1.1f1323b34d718p869,
                0x1.1f17d7b619dafp582,
                0x1.1f1cfdc2e1f3p-115,
                0x1.1f258cd512aa1p+696,
                0x1.1f2a14e4b8456p-251,
                0x1.1f2ca926ecbbdp-1007,
                0x1.1f31a9d056451p-590,
                0x1.1f324d11d4862p354,
                0x1.1f324d11d4862p355,
                0x1.1f36578dec81p+216,
                0x1.1f38bdfb8acdp294,
                0x1.1f38bdfb8acdp295,
                0x1.1f38bdfb8acdp+296,
                0x1.1f3990b617e8dp-382,
                0x1.1f450fcaf95bcp-601,
                0x1.1f450fcaf95bcp-602,
                0x1.1f476cb8fe988p-376,
                0x1.1f476cb8fe988p-377,
                0x1.1f48eaf234ad4p631,
                0x1.1f48eaf234ad4p+632,
                0x1.1f4ac8aadcc3cp-505,
                0x1.1f4b373601d8ap-979,
                0x1.1f4cfde3d6de6p423,
                0x1.1f4e501033af8p-978,
                0x1.1f4e814c85ab8p+696,
                0x1.1f4e814c85ab8p697,
                0x1.1f4f88984e5eap+380,
                0x1.1f4f88984e5eap381,
                0x1.1f526574f1974p-490,
                0x1.1f526574f1974p-491,
                0x1.1f55b23c8bf2dp-40,
                0x1.1f58c500b4cbp+740,
                0x1.1f59e1721cb14p-955,
                0x1.1f5d4ece08832p-510,
                0x1.1f5d4ece08832p-511,
                0x1.1f5d4ece08832p-512,
                0x1.1f6020e014a12p-704,
                0x1.1f63d824b4ce2p275,
                0x1.1f66a83c56948p-965,
                0x1.1f66a83c56948p-966,
                0x1.1f66a83c56948p-967,
                0x1.1f699597d1886p671,
                0x1.1f699597d1886p+672,
                0x1.1f699597d1886p673,
                0x1.1f699597d1886p674,
                0x1.1f6b84e99de72p-594,
                0x1.1f6c11f5d4216p-404,
                0x1.1f6e3b205f7edp-61,
                0x1.1f6e62b0ea63ep-172,
                0x1.1f6e62b0ea63ep-173,
                0x1.1f6e62b0ea63ep-174,
                0x1.1f704p-1056,
                0x1.1f71d8921e08cp915,
                0x1.1f71fcbf372c9p90,
                0x1.1f7329d89e0ap-840,
                0x1.1f7329d89e0ap-841,
                0x1.1f76a66857a68p234,
                0x1.1f775866420e2p-1023,
                0x1.1f7865de2824ep-27,
                0x1.1f83fae786ef1p-993,
                0x1.1f875459c1d4fp-108,
                0x1.1f88d6d9e592p-398,
                0x1.1f88d6d9e592p-399,
                0x1.1f88d6d9e592p-400,
                0x1.1f88d6d9e592p-401,
                0x1.1f904934c0328p893,
                0x1.1f930b676d244p421,
                0x1.1f951a8f0f096p-337,
                0x1.1f98323eb8baep1002,
                0x1.1f98323eb8baep1003,
                0x1.1f98323eb8baep+1004,
                0x1.1f98323eb8baep1005,
                0x1.1f98323eb8baep1006,
                0x1.1f98323eb8baep1007,
                0x1.1f99e11ea0a24p-60,
                0x1.1f99e11ea0a24p-61,
                0x1.1f99e11ea0a24p-62,
                0x1.1f99e11ea0a24p-63,
                0x1.1f99e11ea0a24p-64,
                0x1.1f99e11ea0a24p-65,
                0x1.1f9dc2fcff5bp-637,
                0x1.1f9f2950a41a5p-245,
                0x1.1f9f8ac077e4p+792,
                0x1.1f9fcda46e026p-454,
                0x1.1f9fcda46e026p-455,
                0x1.1fa08501f3b16p390,
                0x1.1fa08501f3b16p391,
                0x1.1fa182c40c60dp-1020,
                0x1.1fa182c40c60ep-1017,
                0x1.1fa182c40c60ep-1018,
                0x1.1fa182c40c60ep-1019,
                0x1.1fa182c40c60ep-1020,
                0x1.1fa182c40c60ep-1021,
                0x1.1fa47cabp-1042,
                0x1.1fa58042809e2p-476,
                0x1.1fa58042809e2p-477,
                0x1.1fa58042809e2p-478,
                0x1.1faaab01dd162p450,
                0x1.1faf3ee7c360ap-553,
                0x1.1fb606e10bdaap741,
                0x1.1fbdc269b43f1p818,
                0x1.1fc0d1664b236p783,
                0x1.1fc29abaa8badp-317,
                0x1.1fca9cbefbdf6p-643,
                0x1.1fca9cbefbdf6p-644,
                0x1.1fca9cbefbdf6p-645,
                0x1.1fca9cbefbdf6p-646,
                0x1.1fe1202728376p529,
                0x1.1fe88401b9306p319,
                0x1.1fe88401b9306p+320,
                0x1.1fee341fc585cp-535,
                0x1.1fee341fc585dp-535,
                0x1.1fee5ffe29edep978,
                0x1.1fee5ffe29edep979,
                0x1.1ff674b5894ccp1014,
                0x1.1ff674b5894ccp1015,
                0x1.1ff69aaac0de8p-200,
                0x1.1ff79a795a851p-360,
                0x1.1ffc49dcd53aap-503,
                0x1.1ffe2b877138cp945,
                0x1.1ffe2b877138cp946,
                0x1.1ffe2b877138cp947,
                0x1.1ffe2b877138cp+948,
                0x1.1p-1067,
                0x1.1p-1068,
                0x1.1p-1070,
                0x1.200ec0588d84p-910,
                0x1.201271a34fd8ep-31,
                0x1.201271a34fd8ep-32,
                0x1.201271a34fd8ep-33,
                0x1.201271a34fd8ep-34,
                0x1.20223f2b3a882p933,
                0x1.20223f2b3a882p934,
                0x1.20223f2b3a882p935,
                0x1.20223f2b3a882p+936,
                0x1.20223f2b3a882p937,
                0x1.20223f2b3a882p938,
                0x1.20223f2b3a882p939,
                0x1.20223f2b3a882p+940,
                0x1.20223f2b3a882p941,
                0x1.20223f2b3a882p942,
                0x1.20223f2b3a882p943,
                0x1.20223f2b3a882p+944,
                0x1.2026dbb1294acp249,
                0x1.202baae891d4ap+324,
                0x1.202c523f7e592p+960,
                0x1.203af9ee75616p-49,
                0x1.203af9ee75616p-50,
                0x1.204039e52d09ep-639,
                0x1.204039e52d09ep-640,
                0x1.204039e52d09ep-641,
                0x1.204039e52d09ep-642,
                0x1.204039e52d09ep-643,
                0x1.204039e52d09ep-644,
                0x1.20418854f82cp-583,
                0x1.204631777584ep421,
                0x1.204631777584ep422,
                0x1.204631777584ep423,
                0x1.204631777584ep+424,
                0x1.204631777584ep425,
                0x1.204da525e190bp501,
                0x1.2051e272bfb86p-317,
                0x1.2051e272bfb86p-318,
                0x1.2051e272bfb86p-319,
                0x1.2051e272bfb86p-320,
                0x1.2051e272bfb86p-321,
                0x1.20583668b4c4dp797,
                0x1.2068dfe15452ap-429,
                0x1.206d307f6e5acp-329,
                0x1.206d307f6e5acp-330,
                0x1.206d307f6e5acp-331,
                0x1.207627e5cd547p-265,
                0x1.207843efe9f9ep-68,
                0x1.207843efe9f9ep-69,
                0x1.207843efe9f9ep-70,
                0x1.207843efe9f9ep-71,
                0x1.207843efe9f9ep-72,
                0x1.20811f528378cp783,
                0x1.20811f528378cp+784,
                0x1.20811f528378cp785,
                0x1.2087d4358fc82p434,
                0x1.2087d4358fc82p435,
                0x1.2087d4358fc82p+436,
                0x1.2087d4358fc82p437,
                0x1.208a94435d698p369,
                0x1.208b2522f897ep-748,
                0x1.208b2522f897ep-749,
                0x1.208b2522f897ep-750,
                0x1.208b2522f897ep-751,
                0x1.208b2522f897ep-752,
                0x1.20924f5680133p193,
                0x1.2092ac5f2019fp+96,
                0x1.2096df6a591b4p-253,
                0x1.209922c421db1p+912,
                0x1.209cb5cab2ap-1028,
                0x1.209d973e48d1fp847,
                0x1.209d973e48d1fp+848,
                0x1.209eff8e4fbf1p-698,
                0x1.20a07aedf13ecp+900,
                0x1.20a2b188f4be6p595,
                0x1.20a2b188f4be6p+596,
                0x1.20a336f98af45p-951,
                0x1.20a36daecb402p811,
                0x1.20aae03563488p+156,
                0x1.20ab3ddd80dbp-483,
                0x1.20b08a08454acp-992,
                0x1.20b08a08454acp-993,
                0x1.20b08a08454acp-994,
                0x1.20b08a08454acp-995,
                0x1.20b08a08454acp-996,
                0x1.20b4ce47f060ap-347,
                0x1.20b761d16601cp-1012,
                0x1.20b7e8ba33efp402,
                0x1.20b7e8ba33efp403,
                0x1.20b99b3c15f74p-792,
                0x1.20c0351ed6573p-150,
                0x1.20c32fa6210cp-870,
                0x1.20c4ad868f1e9p-630,
                0x1.20cbe1ee50a92p+728,
                0x1.20cbe1ee50a92p729,
                0x1.20cbe1ee50a92p730,
                0x1.20d4c2fa8a031p919,
                0x1.20d4c2fa8a031p+920,
                0x1.20d4c2fa8a03p+920,
                0x1.20e3b5b9ad8aap121,
                0x1.20e5a5b1247dfp+828,
                0x1.20e7a84682bbep827,
                0x1.20e8ac3310427p453,
                0x1.20eae6f682781p-524,
                0x1.20eb38ea91b0cp-577,
                0x1.20eb9fff46a84p-886,
                0x1.20eb9fff46a84p-887,
                0x1.20f1b03d58918p846,
                0x1.20f1b03d58918p847,
                0x1.20f850ee7c3eap-444,
                0x1.20f850ee7c3eap-445,
                0x1.20f850ee7c3eap-446,
                0x1.20f87527a532p-998,
                0x1.20ff32caa77aep829,
                0x1.210343ab9ebeep+660,
                0x1.2103c0b895d7dp898,
                0x1.210c23ede2d39p+300,
                0x1.210d2c8d61105p311,
                0x1.210f2fa09defcp+712,
                0x1.210f2fa09defcp713,
                0x1.211e60abcbd53p-559,
                0x1.21231c19271aep486,
                0x1.21239a56e050dp563,
                0x1.212dd4de70913p-731,
                0x1.212dd4de70913p-732,
                0x1.21416d3ebbce1p+780,
                0x1.214a9e905970bp-171,
                0x1.216292ba4aaf2p-895,
                0x1.216543658f056p563,
                0x1.21743e1b83556p-199,
                0x1.21773af7a3a52p-535,
                0x1.2179e68e3edf3p639,
                0x1.217aefe690777p-246,
                0x1.217cffdce15b8p-470,
                0x1.2182f7d91ea58p477,
                0x1.21884f264ca1ap-578,
                0x1.2189eb3f6p-1039,
                0x1.2189eb4p-1045,
                0x1.218cd2d13af42p-530,
                0x1.218cd2d13af42p-531,
                0x1.218eebb99338cp-259,
                0x1.2192374299178p735,
                0x1.2192374299178p+736,
                0x1.2192374299178p737,
                0x1.219d2bc3da25cp-915,
                0x1.21a03b5db2977p+216,
                0x1.21b0f3000cad2p+148,
                0x1.21b2433541d04p635,
                0x1.21b6ce226f8e9p-973,
                0x1.21bb72e3d8db7p889,
                0x1.21c81f7dd43a7p239,
                0x1.21c81f7dd43a8p239,
                0x1.21cp-1064,
                0x1.21da08a8ce109p777,
                0x1.21e553794e715p-961,
                0x1.21e553794e715p-962,
                0x1.21eb8e61c0d1p237,
                0x1.21ebc30118b68p-813,
                0x1.21ebc30118b68p-814,
                0x1.21f1059e1b678p818,
                0x1.21f15374309c4p-644,
                0x1.21f15374309c4p-645,
                0x1.21f7e2527daa4p-934,
                0x1.21f7e2527daa4p-935,
                0x1.21fb1cdb8748ep1005,
                0x1.21fb1cdb8748ep1006,
                0x1.22025be7d1eb8p-72,
                0x1.2207072a8197cp663,
                0x1.2207072a8197cp+664,
                0x1.2207072a8197cp665,
                0x1.220ffddaba0ffp963,
                0x1.221563a9b7322p723,
                0x1.221563a9b7322p+724,
                0x1.221563a9b7322p725,
                0x1.221bd871d2cf4p330,
                0x1.221bd871d2cf4p331,
                0x1.221e875776b1ep-401,
                0x1.221f8f250b9dp-690,
                0x1.22246700e05bdp-75,
                0x1.22248091c5aecp-629,
                0x1.2226dfed80262p790,
                0x1.2226dfed80262p791,
                0x1.2229bb342c14p+828,
                0x1.2229bb342c14p829,
                0x1.22329b0206f7p791,
                0x1.22435fbd7575ep-723,
                0x1.22435fbd7575ep-724,
                0x1.2247bd13f4e55p170,
                0x1.22493bdea4e8p-566,
                0x1.225337dedd6d2p243,
                0x1.225367bbff574p-722,
                0x1.2254edcff6c94p271,
                0x1.2254edcff6c94p+272,
                0x1.225fee0e6897cp-111,
                0x1.226429302875ep+936,
                0x1.226429302875ep937,
                0x1.226636c16c78ep606,
                0x1.226c684c87261p+264,
                0x1.226c684c87261p265,
                0x1.226c684c87261p266,
                0x1.226ed86db3332p-927,
                0x1.22733c07f08c2p-643,
                0x1.22733c07f08c2p-644,
                0x1.2280cf53299f4p-269,
                0x1.2280cf53299f4p-270,
                0x1.22815078cb97cp+992,
                0x1.22815078cb97cp993,
                0x1.22815078cb97cp994,
                0x1.22815078cb97cp995,
                0x1.22815078cb97cp+996,
                0x1.22815078cb97cp997,
                0x1.2290f6439d71dp+812,
                0x1.22990393f2a28p-693,
                0x1.229dbd1507b0fp543,
                0x1.22a0bc0a49194p970,
                0x1.22ac43006f6f9p-991,
                0x1.22b31adf958dcp-77,
                0x1.22b31adf958dcp-78,
                0x1.22bc490dde65ap-442,
                0x1.22bfc0d71e0e8p-604,
                0x1.22bfc0d71e0e8p-605,
                0x1.22bfc0d71e0e8p-606,
                0x1.22bfc0d71e0e8p-607,
                0x1.22bfc0d71e0e8p-608,
                0x1.22bfc0d71e0e8p-609,
                0x1.22bfc0d71e0e8p-610,
                0x1.22bfc0d71e0e8p-611,
                0x1.22c8612a2e468p-544,
                0x1.22cea327fa99dp-770,
                0x1.22cea327fa99dp-771,
                0x1.22d12c0df0cf6p438,
                0x1.22d12c0df0cf6p439,
                0x1.22d36e6a52aap889,
                0x1.22d3d0f6f1568p-906,
                0x1.22d6e9bcad8dcp354,
                0x1.22d99aec65067p-874,
                0x1.22dd5825a713p-401,
                0x1.22dd5825a713p-402,
                0x1.22df4f08ed948p-680,
                0x1.22df4f08ed948p-681,
                0x1.22df4f08ed948p-682,
                0x1.22df4f08ed948p-683,
                0x1.22e41a7ec8db6p-493,
                0x1.22e44545012bep-519,
                0x1.22e9f902696a1p123,
                0x1.22eabba029abap-17,
                0x1.22f2b2ce50c18p957,
                0x1.22f2b2ce50c18p958,
                0x1.22f761717c97cp-182,
                0x1.2309ce54p43,
                0x1.230ac6942c5aep687,
                0x1.230ac6942c5aep+688,
                0x1.230ac6942c5aep689,
                0x1.231205237f048p594,
                0x1.231205237f048p595,
                0x1.23127a14ccbfcp163,
                0x1.2318de5174f9p-756,
                0x1.2320722245ccp-656,
                0x1.2322ad878d5d7p265,
                0x1.232895608290cp-144,
                0x1.232c8bf029934p-460,
                0x1.2330b31f9b401p590,
                0x1.23335afa48f54p-464,
                0x1.23335afa48f54p-465,
                0x1.23335afa48f54p-466,
                0x1.23355feff1506p-866,
                0x1.23355feff1506p-867,
                0x1.23355feff1506p-868,
                0x1.23355feff1506p-869,
                0x1.23355feff1506p-870,
                0x1.2339035a075c6p-663,
                0x1.233f16de49e96p-188,
                0x1.2346ea580bd24p-308,
                0x1.2348a4f251cf8p-734,
                0x1.234d191d76ffbp-924,
                0x1.2357684599702p+528,
                0x1.2357684599702p529,
                0x1.235dc06fefa69p747,
                0x1.236028a75265ep529,
                0x1.23642dbfa6413p-1016,
                0x1.236c0f972b274p966,
                0x1.236ec767851bcp-557,
                0x1.2373143284c24p-895,
                0x1.237a387dfe9fep313,
                0x1.237a387dfe9fep314,
                0x1.237a387dfe9fep315,
                0x1.237f306f0a428p513,
                0x1.237f306f0a428p514,
                0x1.2382570364ce2p785,
                0x1.2382570364ce2p786,
                0x1.238890bfa2baap-919,
                0x1.2389370b7d817p831,
                0x1.238abe794d33dp498,
                0x1.23a4ad20748a2p-75,
                0x1.23a4ad20748a2p-76,
                0x1.23a516e82d9bap1013,
                0x1.23b6669d3fb04p295,
                0x1.23b6669d3fb04p+296,
                0x1.23b9f0bd714d3p-675,
                0x1.23c476c221392p-223,
                0x1.23cc906326f3p202,
                0x1.23ce9d055d6bcp-103,
                0x1.23ce9d055d6bcp-104,
                0x1.23ce9d055d6bcp-105,
                0x1.23ce9d055d6bcp-106,
                0x1.23d1b5eb1d778p-38,
                0x1.23d1b5eb1d778p-39,
                0x1.23d3840ca92e3p+616,
                0x1.23d8f48679ff3p-707,
                0x1.23dae5e943aecp-352,
                0x1.23dc4badd1ddp-113,
                0x1.23e0e7aa505d5p107,
                0x1.23e0e7aa505d5p+108,
                0x1.23e6cb90ba23p611,
                0x1.23e6cb90ba23p+612,
                0x1.23e731b1eb118p+644,
                0x1.23f0a8a4ff1aep430,
                0x1.23f7c8ab5e36dp910,
                0x1.23fa9e6fcf47ep-359,
                0x1.23fa9e6fcf47ep-360,
                0x1.23fa9e6fcf47ep-361,
                0x1.23fa9e6fcf47ep-362,
                0x1.23fa9e6fcf47ep-363,
                0x1.23fa9e6fcf47ep-364,
                0x1.23fc914067368p+700,
                0x1.23fc914067368p701,
                0x1.23ff06eea847ap-632,
                0x1.23ff06eea847ap-633,
                0x1.23ff06eea847ap-634,
                0x1.23ff06eea847ap-635,
                0x1.23ff06eea847ap-636,
                0x1.23ff06eea847ap-637,
                0x1.23ff06eea847ap-638,
                0x1.23ff06eea847ap-639,
                0x1.23ff06eea847ap-640,
                0x1.2401f64e5d0ccp699,
                0x1.2405b773fbdf3p-91,
                0x1.2405b773fbdf3p-92,
                0x1.2405b773fbdf3p-93,
                0x1.240b6a1945e35p-795,
                0x1.240bf61af444ap-936,
                0x1.240eca6a943fep-26,
                0x1.240f1d12a3532p381,
                0x1.2410a19cf639ep426,
                0x1.2410a19cf639ep427,
                0x1.241296cadda77p490,
                0x1.241c4a1389bd7p-335,
                0x1.2431c9f61853bp642,
                0x1.2435b800934ccp+332,
                0x1.243d93f26f385p-421,
                0x1.243dcde3ed0b2p997,
                0x1.24484056b3698p-980,
                0x1.24484056b3698p-981,
                0x1.24484056b3698p-982,
                0x1.24484056b3698p-983,
                0x1.2448a3ab8110cp626,
                0x1.244ce242c5561p-153,
                0x1.244ce242c556p-153,
                0x1.24504d2d0f3fdp615,
                0x1.245654c0d6566p+540,
                0x1.2458a53fca424p-496,
                0x1.245a195e25a05p+240,
                0x1.245cef1eb72b4p-939,
                0x1.2461c6e1026b2p-934,
                0x1.2466b5bp-1046,
                0x1.24683834012b9p-517,
                0x1.246ccec94f52ap505,
                0x1.2471056a78898p923,
                0x1.2471d7ef1df52p-997,
                0x1.247884b855e75p637,
                0x1.247d4155fa08p-912,
                0x1.247e2bc9b3644p-845,
                0x1.2482e58454bafp633,
                0x1.2483a0f12569ap-419,
                0x1.2483a0f12569ap-420,
                0x1.2486b18c42cecp-714,
                0x1.248b1d46f08bcp882,
                0x1.2494b8351287cp-159,
                0x1.2494b8351287cp-160,
                0x1.2494b8351287cp-161,
                0x1.24975e1127a04p-197,
                0x1.249ad2594c37cp+332,
                0x1.249ad2594c37dp+332,
                0x1.249cff79e195p-748,
                0x1.249cff79e195p-749,
                0x1.249cff79e195p-750,
                0x1.24a10172532dcp-381,
                0x1.24a10172532dcp-382,
                0x1.24a30bbc7da68p835,
                0x1.24a68225e0fbep-344,
                0x1.24aa3361bd6bap-884,
                0x1.24aa3361bd6bap-885,
                0x1.24b032a942cc6p-180,
                0x1.24b032a942cc6p-181,
                0x1.24b1aef8f2236p409,
                0x1.24c06843730cp393,
                0x1.24c78db50609cp414,
                0x1.24c9669be2675p-65,
                0x1.24cad88c9fc36p-706,
                0x1.24d23c932ad4fp+468,
                0x1.24d2e2923242p474,
                0x1.24d2ff9022756p-816,
                0x1.24d41206d1b45p-571,
                0x1.24d9c270834bcp551,
                0x1.24dcp-1060,
                0x1.24df97d524402p-590,
                0x1.24df97d524402p-591,
                0x1.24e8d737c5f0ap817,
                0x1.24e8d737c5f0ap818,
                0x1.24ebed977b296p-367,
                0x1.24f41db9591ffp789,
                0x1.24fe7950357p-1030,
                0x1.24p-1068,
                0x1.250137f090754p-230,
                0x1.250534daa27d1p699,
                0x1.2507299112db5p-781,
                0x1.2507d0eccf548p630,
                0x1.2509f38666074p442,
                0x1.250e55470878ap818,
                0x1.25128038e71c2p777,
                0x1.251777e5cc8fep+420,
                0x1.251ac9a570beap561,
                0x1.2529df80db713p921,
                0x1.252aceb16f6c6p915,
                0x1.25305ae6e7dd3p-860,
                0x1.25305ae6e7dd3p-861,
                0x1.25305ae6e7dd3p-862,
                0x1.2533ea586a2e5p-731,
                0x1.2534164491406p-887,
                0x1.2535d090ac00bp+748,
                0x1.253acb830bcdfp181,
                0x1.253ec0e16142p-456,
                0x1.253ec0e16142p-457,
                0x1.253ec0e16142p-458,
                0x1.253ec0e16142p-459,
                0x1.253ec0e16142p-460,
                0x1.25432b14ecea2p-834,
                0x1.25432b14ecea3p-834,
                0x1.254ef1b4b17f6p706,
                0x1.25544519f7a05p-531,
                0x1.255aab96c9198p-605,
                0x1.255aab96c9198p-606,
                0x1.255aab96c9198p-607,
                0x1.255aab96c9198p-608,
                0x1.256799cdae7c4p-71,
                0x1.256799cdae7c4p-72,
                0x1.2569befbd280cp-310,
                0x1.256e444127d84p-843,
                0x1.25799582d3bbep-283,
                0x1.257a30d33c8cp167,
                0x1.257a30d33c8cp+168,
                0x1.25818c7294f27p-688,
                0x1.25818c7294f27p-689,
                0x1.25818c7294f27p-690,
                0x1.2588ee39716e2p+252,
                0x1.2589f7987ecbap591,
                0x1.2589ff8e7d214p789,
                0x1.258b0fe3de08ep261,
                0x1.258b0fe3de08ep262,
                0x1.25915cd68c9f9p-349,
                0x1.2596856d074dcp654,
                0x1.2596856d074dcp655,
                0x1.2596856d074dcp+656,
                0x1.2596856d074dcp657,
                0x1.2596856d074dcp658,
                0x1.25995393a1d0cp-366,
                0x1.2599ed7c6fbd2p-14,
                0x1.25a1360ae1b92p263,
                0x1.25a3d9e7e48dcp-38,
                0x1.25a3d9e7e48dcp-39,
                0x1.25a3d9e7e48dcp-40,
                0x1.25a3d9e7e48dcp-41,
                0x1.25aa26bfa9fc3p-881,
                0x1.25b4537a5004p-435,
                0x1.25b8bc771d72p-586,
                0x1.25bfbc2154772p-421,
                0x1.25c83f79d1e72p-776,
                0x1.25cd4ceae43d2p-210,
                0x1.25cf33eae3cd9p-787,
                0x1.25d342b1e33e6p-126,
                0x1.25d342b1e33e6p-127,
                0x1.25d342b1e33e6p-128,
                0x1.25dd1c4e67fecp-594,
                0x1.25dfa371a19e6p+136,
                0x1.25dfa371a19e7p135,
                0x1.25dfa371a19e7p+136,
                0x1.25dp-1062,
                0x1.25e5555c7e293p+436,
                0x1.25e836602864cp471,
                0x1.25ec18d7014a3p389,
                0x1.25ede7a171e06p909,
                0x1.25f3b053a21e2p291,
                0x1.25f8720dc98p-1033,
                0x1.25f936b20c97dp-769,
                0x1.25fe40eac2448p-591,
                0x1.26043bd352716p-516,
                0x1.26104f8f53efap401,
                0x1.2617a2e8a3086p-215,
                0x1.2617a2e8a3086p-216,
                0x1.2617a2e8a3086p-217,
                0x1.2617a2e8a3086p-218,
                0x1.2617a2e8a3086p-219,
                0x1.262d8af74d7a3p723,
                0x1.262d8af74d7a3p+724,
                0x1.262dfeebbb0f9p621,
                0x1.262dfeebbb0fap+620,
                0x1.262dfeebbb0fap621,
                0x1.262dfeebbb0fap622,
                0x1.262dfeebbb0fap623,
                0x1.262dfeebbb0fap+624,
                0x1.263a7736662e8p609,
                0x1.263a7736662e8p610,
                0x1.263a7736662e8p611,
                0x1.263a7736662e8p+612,
                0x1.263a7736662e8p613,
                0x1.26474a23b8e16p-953,
                0x1.264a25bcee1dap-902,
                0x1.264a25bcee1dap-903,
                0x1.2651fdde60d36p785,
                0x1.265734a4617d7p-461,
                0x1.2657e16ccaa36p-497,
                0x1.265e371d134ddp-805,
                0x1.265e64da14bbdp997,
                0x1.2662c776bc6p790,
                0x1.2662c776bc6p791,
                0x1.2662c776bc6p+792,
                0x1.26636f80a42acp-893,
                0x1.2682160cc250dp+944,
                0x1.2683599859f14p-631,
                0x1.2686a35724608p-297,
                0x1.2688b70e62b1p-1026,
                0x1.2688b70e62bp-1030,
                0x1.2689fffcb28eep-427,
                0x1.268a0be6cd02p-18,
                0x1.2694e06022c22p-444,
                0x1.2699652196262p-64,
                0x1.2699652196262p-65,
                0x1.2699ccc34a1a8p666,
                0x1.2699ccc34a1a8p667,
                0x1.269b93ada1409p919,
                0x1.26a93e9993124p285,
                0x1.26b22fb68b34cp-622,
                0x1.26b3894b1d0ffp+144,
                0x1.26b3894b1d0ffp145,
                0x1.26b4e5cf2b5c5p486,
                0x1.26b507ea5c2acp1014,
                0x1.26b507ea5c2acp1015,
                0x1.26b875d075608p-573,
                0x1.26bc92333a618p-1010,
                0x1.26bc92333a618p-1011,
                0x1.26bc92333a618p-1012,
                0x1.26c9155a05f44p-925,
                0x1.26c9155a05f44p-926,
                0x1.26d506eca4677p-115,
                0x1.26d5755ecc00cp-787,
                0x1.26d73f9d764b9p-545,
                0x1.26dae7bbeda75p-552,
                0x1.26dae7bbeda75p-553,
                0x1.26dae7bbeda75p-554,
                0x1.26dae7bbeda75p-555,
                0x1.26dcba072a935p-305,
                0x1.26e52c4d4977ap585,
                0x1.26f05846a99e1p-521,
                0x1.26f893cf6ap-919,
                0x1.26f893cf6ap-920,
                0x1.26fa0ff395226p410,
                0x1.26fa194a4463cp-355,
                0x1.26fe74bcd1df1p214,
                0x1.26p-1066,
                0x1.2701a3f1a962cp155,
                0x1.2701a3f1a962cp+156,
                0x1.27024a488c834p-776,
                0x1.27024a488c834p-777,
                0x1.270411b6d91a8p-443,
                0x1.270411b6d91a8p-444,
                0x1.270685439e7f5p-938,
                0x1.2706d1df3d66cp+768,
                0x1.2706d1df3d66cp769,
                0x1.270af76e7e284p-266,
                0x1.270af76e7e284p-267,
                0x1.270af76e7e284p-268,
                0x1.270af76e7e284p-269,
                0x1.2714723f791ap-754,
                0x1.2715128df0dd8p-644,
                0x1.27165f96d808p-711,
                0x1.27177de325162p511,
                0x1.2725a8f3e9e22p-987,
                0x1.2725a8f3e9e22p-988,
                0x1.2725dd1d243acp-59,
                0x1.2725dd1d243acp-60,
                0x1.2726d8deb3b55p+584,
                0x1.272a47124ff7p-306,
                0x1.272cb64d97d81p549,
                0x1.272ed2307f56ap+968,
                0x1.272ed2307f56ap969,
                0x1.273584ec38d92p839,
                0x1.273584ec38d92p+840,
                0x1.273584ec38d92p841,
                0x1.27383c5456c5ep-712,
                0x1.27383c5456c5ep-713,
                0x1.2743dca07e78dp+780,
                0x1.274b4504c5c2ap-850,
                0x1.274b4504c5c2ap-851,
                0x1.274c2c14b49c1p-142,
                0x1.274fccd00a21dp-791,
                0x1.27565b6f24c9p-317,
                0x1.2756a8be22154p+256,
                0x1.2758ab4e64836p-96,
                0x1.275a717fadb7bp-608,
                0x1.275c51591352cp117,
                0x1.276276ade098p-275,
                0x1.27670450881c9p-301,
                0x1.276cf9c18e069p-801,
                0x1.276cf9c18e069p-802,
                0x1.27703c89c7322p534,
                0x1.277252b0e89c3p157,
                0x1.27737be05a67cp918,
                0x1.27748f9301d32p425,
                0x1.27748f9301d32p426,
                0x1.27777f7f049a2p473,
                0x1.2784fb5bd62fap817,
                0x1.2784fb5bd62fap818,
                0x1.2784fb5bd62fap819,
                0x1.2784fb5bd62fap+820,
                0x1.2796fca48b918p799,
                0x1.279b2c40fddb6p809,
                0x1.279e2a5d09c37p-865,
                0x1.27a0ae1365974p1018,
                0x1.27a84db40700ap-280,
                0x1.27a9c875c9f7p-941,
                0x1.27b120ba08c48p898,
                0x1.27b2cd0f21b46p198,
                0x1.27b2cd0f21b46p199,
                0x1.27b2cd0f21b46p+200,
                0x1.27b2e4f210075p-771,
                0x1.27b2e4f210075p-772,
                0x1.27b2e4f210075p-773,
                0x1.27b2e4f210075p-774,
                0x1.27b31685fa48ap-614,
                0x1.27ba055fc4e34p-115,
                0x1.27c3191c89c43p691,
                0x1.27c35704a5e67p910,
                0x1.27c35704a5e68p910,
                0x1.27c35704a5e68p911,
                0x1.27c3a89e5dfc8p314,
                0x1.27c3a89e5dfc8p315,
                0x1.27cf82591d57ep-791,
                0x1.27cfcf4ecde02p+1016,
                0x1.27cfcf4ecde02p1017,
                0x1.27d0dd1bd29dap+932,
                0x1.27d0ef0bb970ap-873,
                0x1.27d0ef0bb970ap-874,
                0x1.27d929582cd41p-66,
                0x1.27d929582cd41p-67,
                0x1.27dfeea7f3751p-903,
                0x1.27dfeea7f3751p-904,
                0x1.27e1f4fccab2ap-687,
                0x1.27e1f4fccab2ap-688,
                0x1.27e29123c5eb3p-446,
                0x1.27e8d55a6ef62p-140,
                0x1.27e8d55a6ef62p-141,
                0x1.27ebe44bce004p-854,
                0x1.27edd0dbb8c08p746,
                0x1.27edd0dbb8c08p747,
                0x1.27edd0dbb8c08p+748,
                0x1.27ee7d248b041p-491,
                0x1.27ee7d248b041p-492,
                0x1.2801f11a92896p206,
                0x1.280461b856ec6p221,
                0x1.280461b856ec6p222,
                0x1.280461b856ec6p223,
                0x1.280461b856ec6p+224,
                0x1.280461b856ec6p225,
                0x1.2805e80f73d58p-306,
                0x1.281885422f7ddp-86,
                0x1.281b52p-1051,
                0x1.281e8c275cbdap-741,
                0x1.28223b3d9de32p489,
                0x1.28223b3d9de32p490,
                0x1.2826a3e7044bcp874,
                0x1.2826a3e7044bcp875,
                0x1.28292a81fb49fp-205,
                0x1.283167f97dbccp-177,
                0x1.28437300a37ap+524,
                0x1.2844165c41eedp-127,
                0x1.284895d29101ap415,
                0x1.284895d29101ap+416,
                0x1.284af7c90027bp775,
                0x1.28557bff1fe88p378,
                0x1.28557bff1fe88p379,
                0x1.2856303c221e4p551,
                0x1.2856303c221e4p+552,
                0x1.2856303c221e4p553,
                0x1.2858bf0bf97e6p-35,
                0x1.2858bf0bf97e6p-36,
                0x1.2858bf0bf97e6p-37,
                0x1.2858bf0bf97e6p-38,
                0x1.2858bf0bf97e6p-39,
                0x1.2858bf0bf97e6p-40,
                0x1.2858bf0bf97e6p-41,
                0x1.285cbda237e2p273,
                0x1.285ee642ce5b6p759,
                0x1.285ee642ce5b6p+760,
                0x1.28670c91f1716p+468,
                0x1.2869d59p-1044,
                0x1.286bf6ebf042ap-260,
                0x1.286d80ec190dcp-255,
                0x1.286d80ec190dcp-256,
                0x1.287cep-1055,
                0x1.2880bc98bbdc3p-774,
                0x1.288734ac2fd8dp-366,
                0x1.2887893b26b3dp-779,
                0x1.288a3443b320ap543,
                0x1.288a3443b320ap+544,
                0x1.288f35b66a7a6p-212,
                0x1.288f35b66a7a6p-213,
                0x1.288f35b66a7a6p-214,
                0x1.288f35b66a7a6p-215,
                0x1.288f35b66a7a6p-216,
                0x1.288f35b66a7a6p-217,
                0x1.28904dfba9c1dp+740,
                0x1.2890a9c42ce54p871,
                0x1.2896359325ad5p257,
                0x1.289640510be9dp710,
                0x1.289640510be9dp711,
                0x1.28984a1019e3p915,
                0x1.2898c89a68efp-643,
                0x1.2898c89a68efp-644,
                0x1.2898c89a68efp-645,
                0x1.2898c89a68efp-646,
                0x1.289dee4a7ba41p975,
                0x1.28a2e6d020a3p602,
                0x1.28a61cf9483b7p521,
                0x1.28a61cf9483b7p522,
                0x1.28a61cf9483b7p523,
                0x1.28a61cf9483b7p+524,
                0x1.28aa0a4d2e54fp-836,
                0x1.28aa9725967bcp-279,
                0x1.28aa9725967bcp-280,
                0x1.28bb3e31754fcp-148,
                0x1.28bc8abe49f64p229,
                0x1.28bc8abe49f64p230,
                0x1.28c02dca3103dp-981,
                0x1.28c424437b03cp-330,
                0x1.28cd9dfe9181p-403,
                0x1.28cdc5e6c8fc6p-393,
                0x1.28cddba0110fep899,
                0x1.28d37b23e8f6fp-54,
                0x1.28d710e3d2893p+960,
                0x1.28dae78f5f24ep-426,
                0x1.28df61ad53e3ap-174,
                0x1.28df61ad53e3ap-175,
                0x1.28df61ad53e3ap-176,
                0x1.28df61ad53e3ap-177,
                0x1.28df61ad53e3ap-178,
                0x1.28e9bddd30518p-960,
                0x1.28e9bddd30518p-961,
                0x1.28e9bddd30518p-962,
                0x1.28e9bddd30518p-963,
                0x1.28e9bddd30518p-964,
                0x1.28ebc253668cp-911,
                0x1.28edea80ece3p943,
                0x1.28eebc9a3da5ap-630,
                0x1.28f03f6baep-1034,
                0x1.28f9a38209514p+152,
                0x1.28fd294e6b81cp-868,
                0x1.29000a229ae4bp309,
                0x1.290088722707ep855,
                0x1.290088722707ep+856,
                0x1.2904b551e66c2p230,
                0x1.290ba9a38c7d1p714,
                0x1.290ba9a38c7d2p713,
                0x1.290ba9a38c7d2p714,
                0x1.290ba9a38c7d2p715,
                0x1.290ba9a38c7d2p+716,
                0x1.290ba9a38c7d2p717,
                0x1.290ba9a38c7d2p718,
                0x1.2912e2a4b5264p587,
                0x1.2920f96e7f9efp+560,
                0x1.2920f96e7f9efp561,
                0x1.2920f96e7f9efp562,
                0x1.292103653c764p445,
                0x1.292103653c764p446,
                0x1.2925969a04f72p629,
                0x1.29268fef0abe6p-278,
                0x1.292d7069c234ap754,
                0x1.292d7069c234ap755,
                0x1.29386a175d625p770,
                0x1.293939d21aba8p-881,
                0x1.293939d21aba8p-882,
                0x1.2941a7dad3318p+988,
                0x1.294ed4227ead4p1005,
                0x1.294ed4227ead4p1006,
                0x1.295c5aa1ea43p731,
                0x1.295c5aa1ea43p+732,
                0x1.295fdce22ef5ep-660,
                0x1.29624bfa16edcp-188,
                0x1.29674405d6388p-937,
                0x1.2978fabfe5eb4p-741,
                0x1.2978fabfe5eb4p-742,
                0x1.2978fabfe5eb4p-743,
                0x1.2978fabfe5eb4p-744,
                0x1.297988dfc5702p-162,
                0x1.297b0e4e7a05ap-463,
                0x1.2993e0f6f9eb6p579,
                0x1.29987c9016feep-60,
                0x1.299985953f9f2p701,
                0x1.299a5367715bdp-702,
                0x1.299cf1ead57dcp-404,
                0x1.299cf1ead57dcp-405,
                0x1.299cf1ead57dcp-406,
                0x1.29a0c45ceca7ap+120,
                0x1.29a0c45ceca7ap121,
                0x1.29b01b6885d36p-937,
                0x1.29b01b6885d36p-938,
                0x1.29b01b6885d36p-939,
                0x1.29b01b6885d36p-940,
                0x1.29b01b6885d36p-941,
                0x1.29b01b6885d36p-942,
                0x1.29b01b6885d36p-943,
                0x1.29b01b6885d36p-944,
                0x1.29b01b6885d36p-945,
                0x1.29b23e4994404p+416,
                0x1.29b69070816e2p-452,
                0x1.29b69070816e3p-452,
                0x1.29b69070816e3p-453,
                0x1.29b69070816e3p-454,
                0x1.29b694bde768p-415,
                0x1.29b694bde768p-416,
                0x1.29b706b74f2e7p-1001,
                0x1.29bd5ba63ecfep182,
                0x1.29bef63455bbp-960,
                0x1.29c07a82eae4cp-904,
                0x1.29c07a82eae4cp-905,
                0x1.29c19b9e77196p-625,
                0x1.29c19b9e77196p-626,
                0x1.29c19b9e77196p-627,
                0x1.29c19b9e77196p-628,
                0x1.29c19b9e77196p-629,
                0x1.29c19b9e77196p-630,
                0x1.29c542774d9ap+728,
                0x1.29c6a2eefea62p+160,
                0x1.29ccef5cf2b1cp-306,
                0x1.29ccef5cf2b1cp-307,
                0x1.29ccef5cf2b1cp-308,
                0x1.29cea8ec7c4p-914,
                0x1.29cea8ec7c4p-915,
                0x1.29cea8ec7c4p-916,
                0x1.29cf9c7370489p425,
                0x1.29d4p-1060,
                0x1.29e0194ad8416p-705,
                0x1.29e08180410a3p-275,
                0x1.29e2dc3b54484p-254,
                0x1.29e2dc3b54484p-255,
                0x1.29e2dc3b54484p-256,
                0x1.29e2dc3b54484p-257,
                0x1.29e2dc3b54484p-258,
                0x1.29e3dadace9c4p537,
                0x1.29e418883159ep121,
                0x1.29eb2134f511cp-502,
                0x1.29fc38ac70598p-592,
                0x1.29fc38ac70598p-593,
                0x1.29fc67fbb89c4p-709,
                0x1.2a044328bc75cp522,
                0x1.2a044328bc75cp523,
                0x1.2a044328bc75cp+524,
                0x1.2a05f2p33,
                0x1.2a0622145180ap+232,
                0x1.2a07291abf912p257,
                0x1.2a12adf058324p-1023,
                0x1.2a12d3b81a724p-189,
                0x1.2a166b59f4172p-390,
                0x1.2a166b59f4172p-391,
                0x1.2a222436bec99p-1003,
                0x1.2a27cb0f0d53cp-228,
                0x1.2a2d9cdfcefe2p-675,
                0x1.2a3a96026f751p426,
                0x1.2a3f1622ba6b4p-584,
                0x1.2a3f1622ba6b4p-585,
                0x1.2a3f1622ba6b4p-586,
                0x1.2a3f1622ba6b4p-587,
                0x1.2a3f1622ba6b4p-588,
                0x1.2a444p-1056,
                0x1.2a4f83f487894p-335,
                0x1.2a53aac0aa992p-591,
                0x1.2a53aac0aa992p-592,
                0x1.2a53aac0aa992p-593,
                0x1.2a5568b9f52f4p518,
                0x1.2a58234374d66p195,
                0x1.2a58234374d66p+196,
                0x1.2a58234374d66p197,
                0x1.2a58234374d66p198,
                0x1.2a58234374d66p199,
                0x1.2a58234374d66p+200,
                0x1.2a58234374d66p201,
                0x1.2a58f69d4c61ep+440,
                0x1.2a5daf9226f04p981,
                0x1.2a5daf9226f04p982,
                0x1.2a5daf9226f04p983,
                0x1.2a5daf9226f04p+984,
                0x1.2a5daf9226f04p985,
                0x1.2a62c584d512ap206,
                0x1.2a63606e1ab7ep-307,
                0x1.2a63a08e81dcep1017,
                0x1.2a67bafa22a2cp-226,
                0x1.2a6804a5c98aap939,
                0x1.2a692e3086d6ap+412,
                0x1.2a6de00466c7ep+536,
                0x1.2a6fa70ec822ap-118,
                0x1.2a743576226f4p-650,
                0x1.2a743576226f4p-651,
                0x1.2a74f5469a8aep331,
                0x1.2a7a911e68ec8p-178,
                0x1.2a7a911e68ec8p-179,
                0x1.2a81770b22b92p-306,
                0x1.2a81770b22b92p-307,
                0x1.2a81770b22b92p-308,
                0x1.2a81770b22b92p-309,
                0x1.2a8869849918p+820,
                0x1.2a8869849918p821,
                0x1.2a88ce7d4dd4ep-913,
                0x1.2a88ce7d4dd4ep-914,
                0x1.2a88ce7d4dd4ep-915,
                0x1.2a88ce7d4dd4ep-916,
                0x1.2a8a2452b80dep538,
                0x1.2a8a2452b80dep539,
                0x1.2a8ba9462ac88p-627,
                0x1.2a8ba9462ac88p-628,
                0x1.2a96f8d8d44afp382,
                0x1.2a9bc22527cdep-386,
                0x1.2a9bc22527cdep-387,
                0x1.2a9bc22527cdep-388,
                0x1.2a9e0fb46a168p-901,
                0x1.2aa4f4a405be2p1003,
                0x1.2aa4f4a405be2p+1004,
                0x1.2aa4f4a405be2p1005,
                0x1.2aa5e5baa732ap-99,
                0x1.2aa7df854e25cp-399,
                0x1.2aab573f4ae37p-304,
                0x1.2aac70665485ep-27,
                0x1.2aace0e76ba2cp-870,
                0x1.2aaf1af1687eep603,
                0x1.2ab344e1e3c1ap-947,
                0x1.2ab469676c41p203,
                0x1.2ab469676c41p+204,
                0x1.2ac6f1cc146c6p-487,
                0x1.2ad9f34p-1048,
                0x1.2adc0333020eap-750,
                0x1.2adc0333020eap-751,
                0x1.2adc0333020eap-752,
                0x1.2adc1772595f4p-757,
                0x1.2adc1772595f4p-758,
                0x1.2add36fa66f36p941,
                0x1.2add36fa66f36p942,
                0x1.2ade7f659ec57p681,
                0x1.2ae1051903b68p-203,
                0x1.2ae1051903b68p-204,
                0x1.2ae1bed06a9f4p574,
                0x1.2ae1bed06a9f4p575,
                0x1.2ae5f0555f2e7p309,
                0x1.2af26e041ca04p-858,
                0x1.2af730d873d7ap-706,
                0x1.2afa630cfc8ffp-779,
                0x1.2afb2ddb43304p-168,
                0x1.2afe7f45e2533p-555,
                0x1.2b010d3e1cf56p-643,
                0x1.2b010d3e1cf56p-644,
                0x1.2b010d3e1cf56p-645,
                0x1.2b010d3e1cf56p-646,
                0x1.2b010d3e1cf56p-647,
                0x1.2b010d3e1cf56p-648,
                0x1.2b010d3e1cf56p-649,
                0x1.2b010d3e1cf56p-650,
                0x1.2b0480dd426bcp511,
                0x1.2b04a62a34ebep-222,
                0x1.2b0a474431b74p919,
                0x1.2b0a474431b74p+920,
                0x1.2b0a474431b74p921,
                0x1.2b0a474431b74p922,
                0x1.2b0a474431b74p923,
                0x1.2b101134a64ap+564,
                0x1.2b101134a64ap565,
                0x1.2b1ac0384c59ep-962,
                0x1.2b1ac0384c59ep-963,
                0x1.2b1ac0384c59ep-964,
                0x1.2b1ac0384c59ep-965,
                0x1.2b1ac0384c59ep-966,
                0x1.2b1d7e071d8e3p-563,
                0x1.2b1e0aa2098c5p398,
                0x1.2b22d9ea9c3bcp-1023,
                0x1.2b2302d21ccep-907,
                0x1.2b2302d21ccep-908,
                0x1.2b2302d21ccep-909,
                0x1.2b2302d21ccep-910,
                0x1.2b29f56a87752p+680,
                0x1.2b29f56a87752p681,
                0x1.2b29f56a87752p682,
                0x1.2b29f56a87752p683,
                0x1.2b29f56a87752p+684,
                0x1.2b2b0e3ea5b58p358,
                0x1.2b2db8986fcb4p-261,
                0x1.2b2db8986fcb4p-262,
                0x1.2b2e01c90388ep-410,
                0x1.2b2ee151df738p-538,
                0x1.2b334482e5a34p377,
                0x1.2b334482e5a34p378,
                0x1.2b334482e5a34p379,
                0x1.2b353b789cbd4p603,
                0x1.2b35feb952c06p-313,
                0x1.2b35feb952c06p-314,
                0x1.2b45d0a9b6d2ep+448,
                0x1.2b50c6ec4f313p-163,
                0x1.2b50c6ec4f314p-162,
                0x1.2b50c6ec4f314p-163,
                0x1.2b5175c85c524p-279,
                0x1.2b58ac09ef644p-573,
                0x1.2b6419c06fc7ap630,
                0x1.2b6dcd3b9c654p357,
                0x1.2b6ff587cd132p-888,
                0x1.2b6ff587cd132p-889,
                0x1.2b6ff587cd132p-890,
                0x1.2b6ff587cd132p-891,
                0x1.2b738395538e9p-511,
                0x1.2b738395538e9p-512,
                0x1.2b7515c4bde0ep-364,
                0x1.2b7515c4bde0ep-365,
                0x1.2b7ba4e5e9484p-903,
                0x1.2b860d173720ap-16,
                0x1.2b86d958bae3p-844,
                0x1.2b88e6969eb4p129,
                0x1.2b88e6969eb4p130,
                0x1.2b88e6969eb4p131,
                0x1.2b88e6969eb4p+132,
                0x1.2b88e6969eb4p133,
                0x1.2b88e6969eb4p134,
                0x1.2b94761ea7a8p-913,
                0x1.2b9dcde4fcbafp-1015,
                0x1.2ba095dc7701ep322,
                0x1.2ba095dc7701ep323,
                0x1.2ba14bed0c672p+984,
                0x1.2ba14bed0c672p985,
                0x1.2ba14bed0c672p986,
                0x1.2ba953c461aafp657,
                0x1.2bad504ed4bf6p+344,
                0x1.2bad504ed4bf6p345,
                0x1.2bad504ed4bf6p346,
                0x1.2bad504ed4bf6p347,
                0x1.2bb0af7b8c2f6p-583,
                0x1.2bb0af7b8c2f6p-584,
                0x1.2bb0af7b8c2f6p-585,
                0x1.2bb0af7b8c2f6p-586,
                0x1.2bb0af7b8c2f6p-587,
                0x1.2bb0af7b8c2f6p-588,
                0x1.2bb0af7b8c2f6p-589,
                0x1.2bb0af7b8c2f6p-590,
                0x1.2bb3fa4347f7ap-949,
                0x1.2bb61f1f64e06p-339,
                0x1.2bbc7c9b7029ep1007,
                0x1.2bbc7c9b7029ep+1008,
                0x1.2bcb8d64e8716p-249,
                0x1.2bcb8d64e8716p-250,
                0x1.2bcee08710614p-211,
                0x1.2bcee08710614p-212,
                0x1.2bcee08710614p-213,
                0x1.2bcee08710614p-214,
                0x1.2bd29e07c0f0bp547,
                0x1.2bd7e517b0d35p395,
                0x1.2bda89c335804p597,
                0x1.2be06dfb5f8e6p-28,
                0x1.2be06dfb5f8e6p-29,
                0x1.2be76eaa03a6cp-202,
                0x1.2be78af2b727ap-662,
                0x1.2be79148e78e5p-226,
                0x1.2bea6d5de970dp+848,
                0x1.2bea6d5de970dp849,
                0x1.2beda2da9936ep157,
                0x1.2bf07a143f6d4p807,
                0x1.2bf07a143f6d4p+808,
                0x1.2bfabf61d9332p-449,
                0x1.2bfde1cc6c07fp-290,
                0x1.2bffabcbd231ep659,
                0x1.2c00a65cb8cd6p-359,
                0x1.2c00a65cb8cd6p-360,
                0x1.2c095ba4561fp-354,
                0x1.2c0c6779a25fap-816,
                0x1.2c0fd39ac2cf8p-588,
                0x1.2c0fd39ac2cf8p-589,
                0x1.2c1022e3516ffp-562,
                0x1.2c15bf7ac316p-456,
                0x1.2c15bf7ac316p-457,
                0x1.2c15bf7ac316p-458,
                0x1.2c1d0edbcb2fcp553,
                0x1.2c1d0edbcb2fcp554,
                0x1.2c29611fe2ed2p245,
                0x1.2c333b6027507p-650,
                0x1.2c34b3fa8ce3ep886,
                0x1.2c34b3fa8ce3ep887,
                0x1.2c34b3fa8ce3ep+888,
                0x1.2c34b3fa8ce3ep889,
                0x1.2c34b3fa8ce3ep890,
                0x1.2c3865d5425fp346,
                0x1.2c3865d5425fp347,
                0x1.2c3ad0c7dbb2ep-526,
                0x1.2c3ad0c7dbb2ep-527,
                0x1.2c3e72d179606p-631,
                0x1.2c3e72d179606p-632,
                0x1.2c3e72d179606p-633,
                0x1.2c3e72d179606p-634,
                0x1.2c3e72d179606p-635,
                0x1.2c3e72d179606p-636,
                0x1.2c3e72d179606p-637,
                0x1.2c3e72d179606p-638,
                0x1.2c3ea31177efp-909,
                0x1.2c41f90901c48p289,
                0x1.2c42521558a05p362,
                0x1.2c43305a0acap134,
                0x1.2c4428eff4fc7p-514,
                0x1.2c49d7e177dfbp-863,
                0x1.2c4cf8ea6b6ecp-841,
                0x1.2c4cf8ea6b6ecp-842,
                0x1.2c4cf8ea6b6ecp-843,
                0x1.2c4cf8ea6b6ecp-844,
                0x1.2c4ed70752ad8p525,
                0x1.2c4ed70752ad8p526,
                0x1.2c593a17687a1p-902,
                0x1.2c5a05a4621dp-378,
                0x1.2c5a05a4621dp-379,
                0x1.2c5a5b7843e58p-228,
                0x1.2c5b68ad7d5c6p174,
                0x1.2c5ba438dd3b9p+300,
                0x1.2c5d34375c0fep-387,
                0x1.2c5d34375c0fep-388,
                0x1.2c5d66f5e7f14p+504,
                0x1.2c5d66f5e7f14p505,
                0x1.2c5f31d8fcfa8p-584,
                0x1.2c5f31d8fcfa8p-585,
                0x1.2c5f31d8fcfa8p-586,
                0x1.2c5f31d8fcfa8p-587,
                0x1.2c5f31d8fcfa8p-588,
                0x1.2c5f31d8fcfa8p-589,
                0x1.2c5f31d8fcfa8p-590,
                0x1.2c5f31d8fcfa8p-591,
                0x1.2c626460b87e1p917,
                0x1.2c73bc6c421fdp+964,
                0x1.2c775cb7bd5cp-912,
                0x1.2c791ae276ap-853,
                0x1.2c7f16b8af0c2p-833,
                0x1.2c83be3a6b4f4p-720,
                0x1.2c83be3a6b4f4p-721,
                0x1.2c83be3a6b4f4p-722,
                0x1.2c858b214eb74p-905,
                0x1.2c858b214eb74p-906,
                0x1.2c8ad270592d4p-586,
                0x1.2c92fee3a3868p-486,
                0x1.2c963b623c69ep-773,
                0x1.2c963b623c69ep-774,
                0x1.2c963b623c69ep-775,
                0x1.2c963b623c69ep-776,
                0x1.2c963b623c69ep-777,
                0x1.2c99e813689bp+872,
                0x1.2c9a0b17a052p-273,
                0x1.2c9a0b17a052p-274,
                0x1.2c9a0b17a052p-275,
                0x1.2c9a0b17a052p-276,
                0x1.2c9a0b17a052p-277,
                0x1.2c9d0b1923744p-359,
                0x1.2c9d0b1923745p-359,
                0x1.2c9ee7f8928dep-462,
                0x1.2ca010a56f795p978,
                0x1.2ca4be098e8p-915,
                0x1.2ca4be098e8p-916,
                0x1.2ca4be098e8p-917,
                0x1.2ca4be098e8p-918,
                0x1.2ca6bb09c4882p271,
                0x1.2cace06eeacaap-748,
                0x1.2cb58d36c64a4p+240,
                0x1.2cb58d36c64a4p241,
                0x1.2cb804b61b898p-56,
                0x1.2cb804b61b898p-57,
                0x1.2cbf39bb2d9bcp-316,
                0x1.2cbf39bb2d9bcp-317,
                0x1.2cbf39bb2d9bcp-318,
                0x1.2cbf39bb2d9bcp-319,
                0x1.2cbf39bb2d9bcp-320,
                0x1.2cc32a38642cdp-932,
                0x1.2cc3fcdc7d75ap763,
                0x1.2cc8fdccab336p-537,
                0x1.2cc8fdccab336p-538,
                0x1.2cc8fdccab336p-539,
                0x1.2ccc7331257f8p-908,
                0x1.2cccae96ae676p-466,
                0x1.2cccae96ae676p-467,
                0x1.2ccd539a42604p969,
                0x1.2cd210cf8852ep479,
                0x1.2cd6ab14afcap-436,
                0x1.2cd6ab14afcap-437,
                0x1.2cd6ab14afcap-438,
                0x1.2cd6ab14afcap-439,
                0x1.2cd6ab14afcap-440,
                0x1.2cd875b288ad6p+380,
                0x1.2cde2d351b318p-904,
                0x1.2ce2d0fbf17bp+528,
                0x1.2ce2d0fbf17bp529,
                0x1.2ce5c89f1ab64p195,
                0x1.2ce7e09a3b966p671,
                0x1.2ce8d0044836p-911,
                0x1.2ced32a16a1b1p126,
                0x1.2ced951320ddap-589,
                0x1.2cf5e7bb9a312p-989,
                0x1.2cf5e7bb9a312p-990,
                0x1.2cf5e7bb9a312p-991,
                0x1.2cf80eb893588p-556,
                0x1.2cff80ad30c8p-914,
                0x1.2d02e03f494e4p701,
                0x1.2d032b72e9fb2p634,
                0x1.2d0a7e3f016ebp-903,
                0x1.2d0db4366dc5ap-590,
                0x1.2d0db4366dc5ap-591,
                0x1.2d106b2d489fep810,
                0x1.2d11280ffb90fp-89,
                0x1.2d11280ffb90fp-90,
                0x1.2d12f9cb44b98p657,
                0x1.2d12f9cb44b98p658,
                0x1.2d130bc954a87p-692,
                0x1.2d135b40fc47cp-906,
                0x1.2d135b40fc47cp-907,
                0x1.2d1d1a054f3d1p-271,
                0x1.2d1fe75827665p735,
                0x1.2d2189aa8da3p-908,
                0x1.2d2189aa8da3p-909,
                0x1.2d2189aa8da3p-910,
                0x1.2d251544f1f9cp-904,
                0x1.2d251544f1f9cp-905,
                0x1.2d27e0b45ddcdp-320,
                0x1.2d28a0df5650ap-907,
                0x1.2d28dc1e6d8acp-233,
                0x1.2d2909e201058p-479,
                0x1.2d2991ed526c3p+924,
                0x1.2d2b34074dbd8p686,
                0x1.2d2c2c79baa76p-905,
                0x1.2d2c2c79baa76p-906,
                0x1.2d2ce1ff01ecp-908,
                0x1.2d2ce1ff01ecp-909,
                0x1.2d2ce1ff01ecp-910,
                0x1.2d2ce1ff01ecp-911,
                0x1.2d2ce1ff01ecp-912,
                0x1.2d2ce1ff01ecp-913,
                0x1.2d2ce1ff01ecp-914,
                0x1.2d2ce1ff01ecp-915,
                0x1.2d2d978449308p-906,
                0x1.2d2d978449308p-907,
                0x1.2d2df246ecd2dp-904,
                0x1.2d2df246ecd2dp-905,
                0x1.2d2df246ecd2dp-906,
                0x1.2d2e4d0990752p-908,
                0x1.2d2fb8141efe4p-907,
                0x1.2d2fb8141efe4p-908,
                0x1.2d30578d6107ap797,
                0x1.2d36cf48e7abep-899,
                0x1.2d36cf48e7abep-900,
                0x1.2d36cf48e7abep-901,
                0x1.2d36cf48e7abep-902,
                0x1.2d36cf48e7abep-903,
                0x1.2d36cf48e7abep-904,
                0x1.2d36cf48e7abep-905,
                0x1.2d36cf48e7abep-906,
                0x1.2d3d6f88f0b3cp611,
                0x1.2d3d6f88f0b3dp611,
                0x1.2d3f27b22b686p386,
                0x1.2d44edcc51304p415,
                0x1.2d4b9df0a7a84p-984,
                0x1.2d4d06b0173b1p-188,
                0x1.2d4d06b0173b1p-189,
                0x1.2d53854d423bcp+488,
                0x1.2d5a4350d31p-908,
                0x1.2d5a4350d31p-909,
                0x1.2d5a4350d31p-910,
                0x1.2d5a4350d31p-911,
                0x1.2d5a4350d31p-912,
                0x1.2d5a4350d31p-913,
                0x1.2d5a4350d31p-914,
                0x1.2d5a4350d31p-915,
                0x1.2d5c5281ea0fep-68,
                0x1.2d656a90d35d4p-878,
                0x1.2d656a90d35d4p-879,
                0x1.2d6b183fe4b55p305,
                0x1.2d6b183fe4b55p306,
                0x1.2d6b183fe4b55p307,
                0x1.2d6d371dfe95p378,
                0x1.2d6d371dfe95p379,
                0x1.2d73088f4050ap-91,
                0x1.2d73088f4050ap-92,
                0x1.2d73088f4050ap-93,
                0x1.2d756dfcb1bb4p-75,
                0x1.2d756dfcb1bb4p-76,
                0x1.2d7667c02fafcp+308,
                0x1.2d76932f0eb76p611,
                0x1.2d78e20567ecap898,
                0x1.2d78e20567ecap899,
                0x1.2d78e20567ecap+900,
                0x1.2d79467444ac4p-259,
                0x1.2d7cb22502b6cp-189,
                0x1.2d7cb22502b6cp-190,
                0x1.2d86e5820a8cap+224,
                0x1.2d88811e4613cp-800,
                0x1.2d8fd3b99df86p-579,
                0x1.2d90626eaa244p-1006,
                0x1.2d90626eaa244p-1007,
                0x1.2d90626eaa244p-1008,
                0x1.2d90ea02c6b1ep-83,
                0x1.2d926942bff1ep773,
                0x1.2d994ce5e566cp-33,
                0x1.2d9a550c8p-1040,
                0x1.2d9a550cbp-1038,
                0x1.2d9a550ccp-1040,
                0x1.2da3628ca6504p-388,
                0x1.2da3628ca6504p-389,
                0x1.2da3628ca6504p-390,
                0x1.2db17685245dep163,
                0x1.2db71ca581b3p-931,
                0x1.2db9e43d106bp+268,
                0x1.2db9e43d106bp269,
                0x1.2db9e43d106bp270,
                0x1.2dbcffc441aa6p297,
                0x1.2dbfed8b38875p-217,
                0x1.2dc37152a6d12p-256,
                0x1.2dc37152a6d12p-257,
                0x1.2dc594423c889p303,
                0x1.2dc6f314427b2p398,
                0x1.2dc6f314427b2p399,
                0x1.2dd9ff9b76aeap403,
                0x1.2ddc59586acebp149,
                0x1.2dded6c91f99fp147,
                0x1.2de007f1a38d4p438,
                0x1.2de8137080a08p-907,
                0x1.2de9eb36b4b12p-642,
                0x1.2deac01e2b4f6p-555,
                0x1.2deac01e2b4f7p-555,
                0x1.2deac01e2b4f7p-556,
                0x1.2deac01e2b4f7p-557,
                0x1.2ded05b961738p-148,
                0x1.2df490af544cp+424,
                0x1.2df702389435ep-108,
                0x1.2e03d07375e56p618,
                0x1.2e086ed876ab6p-845,
                0x1.2e0ce115ed916p-930,
                0x1.2e173b60d01f4p-826,
                0x1.2e1ae779befcep-360,
                0x1.2e1e06bb9c04ap-809,
                0x1.2e1e06bb9c04ap-810,
                0x1.2e1e06bb9c04ap-811,
                0x1.2e1e06bb9c04ap-812,
                0x1.2e1e06bb9c04ap-813,
                0x1.2e1e06bb9c04ap-814,
                0x1.2e1e06bb9c04ap-815,
                0x1.2e1e06bb9c04ap-816,
                0x1.2e2a251461a68p+1004,
                0x1.2e2a251461a68p1005,
                0x1.2e2a251461a68p1006,
                0x1.2e2a251461a68p1007,
                0x1.2e2a251461a68p+1008,
                0x1.2e2a251461a68p1009,
                0x1.2e2a251461a68p1010,
                0x1.2e2a251461a68p1011,
                0x1.2e2dd64fdcc12p+696,
                0x1.2e3096b8b1a7p747,
                0x1.2e347647a0ceep-278,
                0x1.2e347647a0ceep-279,
                0x1.2e347647a0ceep-280,
                0x1.2e34f3f4b322ap+972,
                0x1.2e3b40a0e9b4fp-70,
                0x1.2e3b40a0e9b5p-66,
                0x1.2e3b40a0e9b5p-67,
                0x1.2e3b40a0e9b5p-68,
                0x1.2e3b40a0e9b5p-69,
                0x1.2e3b40a0e9b5p-70,
                0x1.2e3b40a0e9b5p-71,
                0x1.2e3b40a0e9b5p-72,
                0x1.2e51a84a3c6ap374,
                0x1.2e51a84a3c6ap375,
                0x1.2e51a84a3c6ap+376,
                0x1.2e5d6a8b3aadap+308,
                0x1.2e5d6a8b3aadap309,
                0x1.2e5d6a8b3aadap310,
                0x1.2e5d6a8b3aadap311,
                0x1.2e5f5dfa4fe9dp-285,
                0x1.2e6493c2a2b7p-122,
                0x1.2e6493c2a2b7p-123,
                0x1.2e6895d83d1fap349,
                0x1.2e6a986c0e5aep-161,
                0x1.2e704c1a7e725p558,
                0x1.2e704c1a7e725p559,
                0x1.2e75fc65d0a06p-744,
                0x1.2e767221e3e7fp134,
                0x1.2e79a035b9714p699,
                0x1.2e79a035b9714p+700,
                0x1.2e79a035b9714p701,
                0x1.2e79a035b9714p702,
                0x1.2e79a035b9714p703,
                0x1.2e79a035b9714p+704,
                0x1.2e85dbc737298p311,
                0x1.2e8bd69aa19ccp415,
                0x1.2e8bd69aa19ccp+416,
                0x1.2e8f2957c93a3p-918,
                0x1.2e94aa4fc115p750,
                0x1.2e94aa4fc115p751,
                0x1.2e94aa4fc115p+752,
                0x1.2e94aa4fc115p753,
                0x1.2e9a13b7f1c9cp-678,
                0x1.2e9a13b7f1c9cp-679,
                0x1.2e9f67db3899cp-609,
                0x1.2ea2925d20032p825,
                0x1.2ea2925d20032p826,
                0x1.2ea2925d20032p827,
                0x1.2ea2925d20032p+828,
                0x1.2ea2925d20032p829,
                0x1.2ea4533a61a64p-49,
                0x1.2ea6a7d59908p-1029,
                0x1.2ea8d4f0bff94p1017,
                0x1.2eac644e8402cp-634,
                0x1.2eac644e8402cp-635,
                0x1.2eaf6b0891456p+476,
                0x1.2eaf6b0891456p477,
                0x1.2eb0012b707ccp-129,
                0x1.2ec5d8bf3ce41p-205,
                0x1.2ed1934d5d373p-759,
                0x1.2ed29f6a0922ep-555,
                0x1.2ed3f3353472p-762,
                0x1.2ed524880a675p-503,
                0x1.2ed6d460ba20fp-699,
                0x1.2ed71b686bcc6p245,
                0x1.2edc82110c2f9p+900,
                0x1.2edc82110c2fap+900,
                0x1.2ee36e5140446p-559,
                0x1.2ee36e5140446p-560,
                0x1.2ee8d0079661ap-729,
                0x1.2ef1f9abe60f6p-898,
                0x1.2ef1f9abe60f6p-899,
                0x1.2ef1f9abe60f6p-900,
                0x1.2ef1f9abe60f6p-901,
                0x1.2ef1f9abe60f6p-902,
                0x1.2ef21357a0df8p-526,
                0x1.2ef3d501a8dd4p842,
                0x1.2ef7ccad0bb0cp895,
                0x1.2efb6e943e428p-120,
                0x1.2efbbddf53f03p426,
                0x1.2efc987c579bbp779,
                0x1.2f128c9921d82p-788,
                0x1.2f128p-1054,
                0x1.2f192081f941ep-672,
                0x1.2f19f28c3d0fep381,
                0x1.2f20550df5e68p-816,
                0x1.2f2839319072ap+912,
                0x1.2f2f844b9bd59p303,
                0x1.2f38c37d8c102p+444,
                0x1.2f392a212fd77p-429,
                0x1.2f39779b5c7c8p-138,
                0x1.2f39779b5c7c8p-139,
                0x1.2f39779b5c7c8p-140,
                0x1.2f39779b5c7c8p-141,
                0x1.2f39779b5c7c8p-142,
                0x1.2f39779b5c7c8p-143,
                0x1.2f39e794d9d8bp-751,
                0x1.2f39e794d9d8cp-750,
                0x1.2f39e794d9d8cp-751,
                0x1.2f424cd9080fcp-524,
                0x1.2f42c05e11ba4p-416,
                0x1.2f498a49405f6p-154,
                0x1.2f581c0fa608ap-296,
                0x1.2f5c6244f13a7p-711,
                0x1.2f5e0f8aabe86p-832,
                0x1.2f5ebc8c41dp-1030,
                0x1.2f6b714745212p479,
                0x1.2f6b714745212p+480,
                0x1.2f6b714745212p481,
                0x1.2f6b714745212p482,
                0x1.2f6d8325918ap225,
                0x1.2f78a15058dc3p+260,
                0x1.2f8786d48f70ap-231,
                0x1.2f888838e5851p-507,
                0x1.2f8ac174d6123p-266,
                0x1.2f9057d5b10f8p-999,
                0x1.2f90c56a57758p902,
                0x1.2f93272b84ee6p841,
                0x1.2f93272b84ee6p842,
                0x1.2f93d3d63f138p335,
                0x1.2f95a86012d37p698,
                0x1.2f974d8p-1049,
                0x1.2f9777721f2dap-349,
                0x1.2f9777721f2dap-350,
                0x1.2f9c6e3c0a9bap-630,
                0x1.2fa82eccae40ap+936,
                0x1.2fb1548bea1eep453,
                0x1.2fb1548bea1eep454,
                0x1.2fbc744cea168p-137,
                0x1.2fc1cd426a0cep197,
                0x1.2fc1cd426a0cep198,
                0x1.2fc1cd426a0cep199,
                0x1.2fc2c1a2ab162p-460,
                0x1.2fc2c1a2ab162p-461,
                0x1.2fc2c1a2ab162p-462,
                0x1.2fc2c1a2ab162p-463,
                0x1.2fc2c1a2ab162p-464,
                0x1.2fc2c1a2ab162p-465,
                0x1.2fc2c1a2ab162p-466,
                0x1.2fc80c0d2d1b3p+812,
                0x1.2fcb6ce74621cp-1006,
                0x1.2fcb6ce74621cp-1007,
                0x1.2fcbc91078fefp-116,
                0x1.2fcbc91078fefp-117,
                0x1.2fcbc91078fefp-118,
                0x1.2fdbb0e39fb47p219,
                0x1.2fdbb0e39fb48p219,
                0x1.2fdef8e1e394ep-495,
                0x1.2fe1598ebef54p-143,
                0x1.2fe8618794a9cp175,
                0x1.2fec8b240e94fp379,
                0x1.2fee78c940cd1p+756,
                0x1.2fee78c940cd1p757,
                0x1.2ff9864a5a956p-404,
                0x1.2ffa354c1ef48p190,
                0x1.2p-1071,
                0x1.3006a46769a7fp-544,
                0x1.3006a46769a7fp-545,
                0x1.3008431152424p771,
                0x1.300d6db5fa8cap-820,
                0x1.300d6db5fa8cap-821,
                0x1.300ea21f3bf5cp515,
                0x1.301067f4b1b67p962,
                0x1.301296619bf51p578,
                0x1.30131c1f1a2b6p1002,
                0x1.30131c1f1a2b6p1003,
                0x1.30131c1f1a2b6p+1004,
                0x1.301a43a50593cp-478,
                0x1.301a43a50593cp-479,
                0x1.3023b360fe1dap-286,
                0x1.30246db97025p-907,
                0x1.3029be27a33a4p899,
                0x1.302ad82e83fd3p369,
                0x1.302cb5e6f642ap703,
                0x1.302cb5e6f642ap+704,
                0x1.302cb5e6f642ap705,
                0x1.302cb5e6f642ap706,
                0x1.302cb5e6f642ap707,
                0x1.302cb5e6f642ap+708,
                0x1.302f9b53f1a5ap725,
                0x1.303314704f14dp751,
                0x1.30334b1c9109ap-171,
                0x1.3038d680d46c1p211,
                0x1.303c73efd0676p-94,
                0x1.303c73efd0676p-95,
                0x1.303dec3198p-1037,
                0x1.303ecebf0f0d5p-661,
                0x1.303fa4f5c490ap818,
                0x1.3040f585c435ap+388,
                0x1.3049324b71932p-519,
                0x1.304c32c93ccd3p-894,
                0x1.3051d9d6a2c5p+440,
                0x1.3053c9ef60c77p-526,
                0x1.3059e188e6fdbp-592,
                0x1.30622f599bb7dp-975,
                0x1.30622f599bb7dp-976,
                0x1.30707ec1ecce6p-182,
                0x1.30707ec1ecce6p-183,
                0x1.3072b371f3ce8p-560,
                0x1.3072d0d60c01cp-152,
                0x1.3072d0d60c01cp-153,
                0x1.307fcada17cbp607,
                0x1.307fcada17cbp+608,
                0x1.307fcada17cbp609,
                0x1.307fcada17cbp610,
                0x1.307fcada17cbp611,
                0x1.308619efb5f47p682,
                0x1.30897f09aa283p-281,
                0x1.308a831868ac8p-946,
                0x1.308a831868ac8p-947,
                0x1.308d92c00281p389,
                0x1.308d92c00281p390,
                0x1.3096d272030ddp647,
                0x1.3096d272030ddp+648,
                0x1.30988165f2196p-972,
                0x1.30a6411b49c6p-345,
                0x1.30aa0837e0b06p-936,
                0x1.30ad240ee473p-906,
                0x1.30ad240ee473p-907,
                0x1.30adc928405ep774,
                0x1.30b1e036bf908p-252,
                0x1.30baa4ed2b746p+108,
                0x1.30baa4ed2b746p109,
                0x1.30c7f250ac79fp163,
                0x1.30c7f53323f91p-849,
                0x1.30c7f53323f91p-850,
                0x1.30d32e3938b9ep479,
                0x1.30d4b1580a7bp-612,
                0x1.30d4b1580a7bp-613,
                0x1.30d5c2775c00cp-703,
                0x1.30d5c2775c00cp-704,
                0x1.30d7e4a8b43cep222,
                0x1.30dbb6b8d674ep-462,
                0x1.30dbb6b8d674fp-462,
                0x1.30e265ea39dd1p-749,
                0x1.30e49bb2a87d3p-127,
                0x1.30e49bb2a87d3p-128,
                0x1.30e6c7d4e348p+280,
                0x1.30e6c7d4e348p281,
                0x1.30e6c7d4e348p282,
                0x1.30e6c7d4e348p283,
                0x1.30e6c7d4e348p+284,
                0x1.30e9b9db2a42ap-162,
                0x1.30f53d811a4b2p-870,
                0x1.30f53d811a4b2p-871,
                0x1.30f792d6cd9c2p965,
                0x1.30f792d6cd9c2p966,
                0x1.30f792d6cd9c2p967,
                0x1.30f792d6cd9c2p+968,
                0x1.30fe5ac9242d6p-745,
                0x1.310204090cc6cp833,
                0x1.310d84ad1b516p769,
                0x1.3113637da3b14p439,
                0x1.3113637da3b14p+440,
                0x1.311475e5f81cap-87,
                0x1.31167f74289cep-693,
                0x1.3117f602b5986p791,
                0x1.3117f602b5986p+792,
                0x1.3117f602b5986p793,
                0x1.3125daa057ec4p342,
                0x1.3125daa057ec4p343,
                0x1.31296b36e579ep-325,
                0x1.31296b36e579ep-326,
                0x1.31296b36e579ep-327,
                0x1.31296b36e579ep-328,
                0x1.312dp23,
                0x1.3139a133c728ap718,
                0x1.3139a133c728ap719,
                0x1.3139a133c728ap+720,
                0x1.3139a133c728ap721,
                0x1.3139a133c728ap722,
                0x1.3144be1d2a5fcp681,
                0x1.31484742c0852p754,
                0x1.31484742c0852p755,
                0x1.3149b769a2608p-360,
                0x1.3149b769a2608p-361,
                0x1.3149b769a2608p-362,
                0x1.314a200f69abap-795,
                0x1.314d381f2c31ep-35,
                0x1.3154a9abad08ap-847,
                0x1.315977b12d222p-254,
                0x1.315b6f65f1726p379,
                0x1.315b6f65f1726p+380,
                0x1.315dabaea549fp-837,
                0x1.31606ccf32c12p791,
                0x1.3165633977bcap-825,
                0x1.3165633977bcap-826,
                0x1.3165633977bcap-827,
                0x1.3165633977bcap-828,
                0x1.3166a79009109p-159,
                0x1.3168747c0c3c6p-522,
                0x1.316a951647a9p+740,
                0x1.316a951647a9p741,
                0x1.316a951647a9p742,
                0x1.316a951647a9p743,
                0x1.316e0b9bfc156p-505,
                0x1.31798a6093cd3p-664,
                0x1.317a34ed1de37p657,
                0x1.317c9646507cp-603,
                0x1.317e5ef3ab327p505,
                0x1.317e5ef3ab327p506,
                0x1.317e5ef3ab327p507,
                0x1.317e5ef3ab327p+508,
                0x1.318006a75e499p-251,
                0x1.31841a554bde4p850,
                0x1.318c2cb5fffd6p573,
                0x1.318e2e7635ceap-291,
                0x1.318e4df26d852p465,
                0x1.3191fcc916bcp+212,
                0x1.3191fcc916bcp213,
                0x1.3191fcc916bcp214,
                0x1.3191fcc916bcp215,
                0x1.31952f9cc3718p665,
                0x1.319d89aba172cp-681,
                0x1.319d89aba172cp-682,
        };
    }

    static double[] hard1() {
        return new double[] {
                0x1.31af0075d42a6p111,
                0x1.31b3f52209558p785,
                0x1.31b4aff7f21b6p501,
                0x1.31b7112c17bcep1017,
                0x1.31b9470c3ddf9p277,
                0x1.31bd06d61f9e3p-397,
                0x1.31bf974a62452p-878,
                0x1.31bf974a62452p-879,
                0x1.31c4156c8a32cp685,
                0x1.31c72c1bfbdbcp-507,
                0x1.31c72c1bfbdbcp-508,
                0x1.31c72c1bfbdbcp-509,
                0x1.31c72c1bfbdbcp-510,
                0x1.31c72c1bfbdbcp-511,
                0x1.31c8192f1a5p746,
                0x1.31c8192f1a5p747,
                0x1.31c8192f1a5p+748,
                0x1.31c8192f1a5p749,
                0x1.31cfd3999f7bp993,
                0x1.31da02fcd9858p-164,
                0x1.31dfa75360842p194,
                0x1.31dfa75360842p195,
                0x1.31e0cf0b3e774p-403,
                0x1.31e0cf0b3e774p-404,
                0x1.31e0cf0b3e774p-405,
                0x1.31e0cf0b3e774p-406,
                0x1.31e29be19ef4cp-899,
                0x1.31ec0e2c32854p-417,
                0x1.31ee972971c31p431,
                0x1.31f2a19595fp-497,
                0x1.31f46bcf7b452p-42,
                0x1.31p-1066,
                0x1.3204b48a0a50cp+780,
                0x1.3204b48a0a50cp781,
                0x1.3204b48a0a50cp782,
                0x1.320bdee96e8a2p945,
                0x1.322138ea3de5bp122,
                0x1.322e220a5b17ep-655,
                0x1.322e220a5b17ep-656,
                0x1.322e220a5b17ep-657,
                0x1.322e220a5b17ep-658,
                0x1.322e220a5b17ep-659,
                0x1.3235245e602fp-977,
                0x1.323ebad3c2b77p911,
                0x1.3240293e95c3cp-43,
                0x1.3240293e95c3cp-44,
                0x1.324872e9c704p-976,
                0x1.324872e9c704p-977,
                0x1.324c99p-1049,
                0x1.3250be60e18p-1033,
                0x1.3250e843b3p-919,
                0x1.325424053c56dp857,
                0x1.325eep-1055,
                0x1.32645e1ba93fp98,
                0x1.32645e1ba93fp99,
                0x1.326947b949f98p882,
                0x1.326cd14f71c23p-211,
                0x1.326cd14f71c23p-212,
                0x1.326db292p-1043,
                0x1.3270636e76897p943,
                0x1.3272114e5b4e2p-533,
                0x1.3272a7788fd82p-736,
                0x1.3272a7788fd82p-737,
                0x1.3272a7788fd82p-738,
                0x1.3272e17ea7c58p557,
                0x1.3274280201a8ap283,
                0x1.3274280201a8ap+284,
                0x1.32778bc28286cp-494,
                0x1.32778bc28286cp-495,
                0x1.3279e4a7ef173p-88,
                0x1.327e88e28f3b3p+884,
                0x1.327fc58da0f7p-173,
                0x1.327fc58da0f7p-174,
                0x1.3281288abc664p-441,
                0x1.3281c721f41eap-590,
                0x1.3282566129622p-623,
                0x1.3286fb5758873p-433,
                0x1.329012f488455p-627,
                0x1.3291ce5b58bccp545,
                0x1.3292b73cc52e4p365,
                0x1.329336899daa7p741,
                0x1.329336899daa7p742,
                0x1.3293b377ebae6p-843,
                0x1.3293b377ebae6p-844,
                0x1.32964f2944b06p125,
                0x1.32964f2944b06p126,
                0x1.329c0568a8ac3p449,
                0x1.329ed43378edbp-770,
                0x1.32a215bc403d4p755,
                0x1.32a3cf309095p-1025,
                0x1.32b7f3e7ee913p-117,
                0x1.32be82ec7c29ep-775,
                0x1.32c44e976f938p-503,
                0x1.32c853f13960ep527,
                0x1.32c853f13960ep+528,
                0x1.32c853f13960ep529,
                0x1.32c853f13960ep530,
                0x1.32c853f13960ep531,
                0x1.32cec70a53f22p-40,
                0x1.32d0612ee673ep202,
                0x1.32d0612ee673ep203,
                0x1.32d17ed577d0cp+312,
                0x1.32d24c6791794p266,
                0x1.32d2fd0c60df6p843,
                0x1.32d761e295701p+932,
                0x1.32d8b68b56c41p537,
                0x1.32d9b943b6f84p393,
                0x1.32d9b943b6f84p394,
                0x1.32d9b943b6f84p395,
                0x1.32dab56f2081ap-954,
                0x1.32dab56f2081ap-955,
                0x1.32e246f62d12p-795,
                0x1.32e366538c768p981,
                0x1.32e366538c768p982,
                0x1.32e3d18e5b1d4p-963,
                0x1.32f209f9852bp-613,
                0x1.32fa69a69bd6ep-624,
                0x1.32fa69a69bd6ep-625,
                0x1.32fa69a69bd6ep-626,
                0x1.32fa69a69bd6ep-627,
                0x1.32fa69a69bd6ep-628,
                0x1.32fa69a69bd6ep-629,
                0x1.32fa69a69bd6ep-630,
                0x1.32fa69a69bd6ep-631,
                0x1.32fb4488ea4fp-894,
                0x1.32fbf3db0445fp-965,
                0x1.32fc01500e56ap979,
                0x1.32fc01500e56ap+980,
                0x1.32fc75ef6f1d9p-499,
                0x1.33057606449fap+1004,
                0x1.3306ac14dab55p293,
                0x1.330776d9fbc79p935,
                0x1.330894de1cfc1p-210,
                0x1.3308fd767ce4ep+852,
                0x1.33094ec35de5fp-92,
                0x1.330a962603082p-134,
                0x1.330a962603082p-135,
                0x1.330a962603082p-136,
                0x1.331351037407p875,
                0x1.3318ab6924fc7p587,
                0x1.33190feb39882p137,
                0x1.331a84dfb84adp-105,
                0x1.33234de7ad7e2p797,
                0x1.33234de7ad7e3p797,
                0x1.332366e586805p-55,
                0x1.332fcabc3f0bcp-485,
                0x1.332fcabc3f0bcp-486,
                0x1.3339131c46f8bp-1004,
                0x1.3340a350c714dp899,
                0x1.33411f2eaeddp+364,
                0x1.3341c3cfd1e24p945,
                0x1.3345744c56ebp+668,
                0x1.33462ff967d44p+536,
                0x1.33462ff967d44p537,
                0x1.3348ad97c35d4p927,
                0x1.3349ccb37f3c4p287,
                0x1.3354acffe253ep-868,
                0x1.3356f81907b6ap+480,
                0x1.33673d49f819ep-1011,
                0x1.33673d49f819ep-1012,
                0x1.33673d49f819ep-1013,
                0x1.336928d9c2d8ap-529,
                0x1.336b0f7f5e0d7p529,
                0x1.3370d8714d5f5p+664,
                0x1.3375987038dap-50,
                0x1.33787de1f187dp-563,
                0x1.337b58454b8d4p849,
                0x1.33815d9045244p-996,
                0x1.33819e4e8999fp-474,
                0x1.338205089f29cp-854,
                0x1.338b3p-1054,
                0x1.3394d711b084fp697,
                0x1.33a01038c705p-913,
                0x1.33a6d655bfc41p-134,
                0x1.33a9c6baa200cp-340,
                0x1.33a9c6baa200cp-341,
                0x1.33a9c6baa200cp-342,
                0x1.33a9c6baa200cp-343,
                0x1.33ab601710c27p-885,
                0x1.33bb38ea396dep407,
                0x1.33bd7732c0454p-540,
                0x1.33bd7732c0454p-541,
                0x1.33be26a0261c4p883,
                0x1.33d4032c2c7f4p-366,
                0x1.33d4032c2c7f4p-367,
                0x1.33d4032c2c7f4p-368,
                0x1.33d4032c2c7f4p-369,
                0x1.33d4032c2c7f4p-370,
                0x1.33d7dbad8aa1p537,
                0x1.33e3ab156d127p-194,
                0x1.33e925726b56p-245,
                0x1.33e925726b56p-246,
                0x1.33e925726b56p-247,
                0x1.33e925726b56p-248,
                0x1.33ea5afbe9c9p-322,
                0x1.33f19972247d5p-957,
                0x1.33fb42745fa14p274,
                0x1.33fbe435bff56p753,
                0x1.3409ab20c48eap-371,
                0x1.3409ab20c48eap-372,
                0x1.340e1d4d0d451p-219,
                0x1.3417ecba7c1f2p391,
                0x1.3417ecba7c1f2p+392,
                0x1.3417ecba7c1f2p393,
                0x1.3417ecba7c1f2p394,
                0x1.34188214f18dap279,
                0x1.34188214f18dap+280,
                0x1.341d41ca6e29p-429,
                0x1.341d6d840a26cp+628,
                0x1.3421d14721948p-763,
                0x1.342265ea2d4d2p758,
                0x1.3426172c74d82p+116,
                0x1.342acd09310acp270,
                0x1.343555d0194cap382,
                0x1.343609dca87a2p-683,
                0x1.34360d0b8e1a6p-584,
                0x1.34360d0b8e1a6p-585,
                0x1.343f0070262eap-850,
                0x1.344449599b5bcp205,
                0x1.344449599b5bcp206,
                0x1.344a3d5f42bebp+204,
                0x1.344eb36097b4ap-185,
                0x1.344eb36097b4ap-186,
                0x1.344eb36097b4ap-187,
                0x1.344eb36097b4ap-188,
                0x1.34506efed33fbp-188,
                0x1.3452248911254p-425,
                0x1.34564f2e1157ep+292,
                0x1.345b46397df2ep-132,
                0x1.345b46397df2ep-133,
                0x1.345b46397df2ep-134,
                0x1.345b46397df2ep-135,
                0x1.346ded2dc2958p-359,
                0x1.346ded2dc2958p-360,
                0x1.3477d6f21c383p-96,
                0x1.3478410f4c7ecp601,
                0x1.3478410f4c7ecp602,
                0x1.3478410f4c7ecp603,
                0x1.3478410f4c7ecp+604,
                0x1.347c6640b51d3p829,
                0x1.347d8b4bc33ecp-354,
                0x1.3483175f1fa44p-671,
                0x1.3484a84b270fcp-259,
                0x1.3484a84b270fcp-260,
                0x1.3484a84b270fcp-261,
                0x1.3486ee3a4b56fp-523,
                0x1.348e02b28315ap789,
                0x1.348e02b28315ap790,
                0x1.348e5692db0ddp-934,
                0x1.348e5692db0ddp-935,
                0x1.3496128b66f3p-659,
                0x1.3496128b66f3p-660,
                0x1.3496128b66f3p-661,
                0x1.3496128b66f3p-662,
                0x1.3496128b66f3p-663,
                0x1.3496128b66f3p-664,
                0x1.3496128b66f3p-665,
                0x1.3496128b66f3p-666,
                0x1.349a6290342ffp970,
                0x1.349a6290342ffp971,
                0x1.349a6ap-1045,
                0x1.34a815b2b16e4p879,
                0x1.34b2dcd360d4ap+356,
                0x1.34b365f379dfcp-19,
                0x1.34b38c5980dc7p-813,
                0x1.34b4b46c34636p-738,
                0x1.34b4b46c34636p-739,
                0x1.34b4b46c34636p-740,
                0x1.34b8a069d92ddp-248,
                0x1.34ba8p-1057,
                0x1.34ce9bc05659p505,
                0x1.34d761p-1050,
                0x1.34e33177e794ap+244,
                0x1.34e33177e794ap245,
                0x1.34e3e1bf59b4ep-647,
                0x1.34eb2882074f4p899,
                0x1.34ebf78a20341p371,
                0x1.34ef66dcb6e9ep871,
                0x1.34f6588f0bb28p907,
                0x1.34f6588f0bb28p+908,
                0x1.34fb158be096p+996,
                0x1.3505d0ac73p-1034,
                0x1.3509c358d40fep-991,
                0x1.350ce8943c10ep+584,
                0x1.35101ea32178bp-497,
                0x1.3516b5876ac27p-334,
                0x1.351b4febafea7p-677,
                0x1.3521d3cce3c1fp-743,
                0x1.3524972dp-1038,
                0x1.3524972dp-1039,
                0x1.3524972dp-1040,
                0x1.3529217bdce6cp139,
                0x1.3529ba7d19eafp-565,
                0x1.352cd6096455fp+772,
                0x1.352d98f76c755p-854,
                0x1.352d98f76c755p-855,
                0x1.353bd097b74b2p-594,
                0x1.353bd097b74b2p-595,
                0x1.353bd097b74b2p-596,
                0x1.353bd097b74b2p-597,
                0x1.354c30e828f9cp-845,
                0x1.354d1eec466b2p-75,
                0x1.35604548816dep-1012,
                0x1.35708bb61dd1cp-746,
                0x1.35708bb61dd1cp-747,
                0x1.3578ea80868c4p-669,
                0x1.357c299a88ea7p-80,
                0x1.357c299a88ea8p-79,
                0x1.357c299a88ea8p-80,
                0x1.3584219bd91fbp-215,
                0x1.35853122f65f3p254,
                0x1.35853122f65f3p255,
                0x1.3588688e0a0d4p950,
                0x1.3588c42be6bb2p+1020,
                0x1.358b96a0c4cbap287,
                0x1.358bc7811da44p607,
                0x1.358cdec9f3072p+532,
                0x1.358e76513f848p519,
                0x1.359ac0f608508p-158,
                0x1.35a0ff35b68p+712,
                0x1.35a0ff35b68p713,
                0x1.35aac9ae48337p-217,
                0x1.35aac9ae48337p-218,
                0x1.35ac97a0d0c1ap-611,
                0x1.35ac97a0d0c1ap-612,
                0x1.35ac97a0d0c1ap-613,
                0x1.35ac97a0d0c1ap-614,
                0x1.35b288f43676p+488,
                0x1.35b651c840b25p-547,
                0x1.35b651c840b25p-548,
                0x1.35b8a046a8b73p-580,
                0x1.35bc835980a78p841,
                0x1.35bd26a53f1b2p939,
                0x1.35c59e2555388p274,
                0x1.35c6bccdf5081p-881,
                0x1.35ceaeb2d28c1p+404,
                0x1.35ceaeb2d28c1p405,
                0x1.35ceaeb2d28cp405,
                0x1.35d4a81c6685dp239,
                0x1.35d4ffc62ed16p+532,
                0x1.35d50e95f7e96p-963,
                0x1.35d50e95f7e96p-964,
                0x1.35dc57a56212ep182,
                0x1.35de69195a374p381,
                0x1.35e02714cd7dap+572,
                0x1.35e311a87b966p+284,
                0x1.35e7c1c4ffa41p-59,
                0x1.35e8bc370c2fcp169,
                0x1.35f7b57395a1cp-437,
                0x1.3602cb845363p-499,
                0x1.3602cb845363p-500,
                0x1.36071dcae4565p-46,
                0x1.360a74f6dc268p-461,
                0x1.360a74f6dc268p-462,
                0x1.360b6f67d6efcp+968,
                0x1.360c38b6p-1042,
                0x1.361b439cdd628p+720,
                0x1.362149cbd3226p890,
                0x1.362910ecc299ap571,
                0x1.3637454bac18p-910,
                0x1.3637454bac18p-911,
                0x1.363c1ddd3dcd2p250,
                0x1.363c1ddd3dcd2p251,
                0x1.363c1ddd3dcd2p+252,
                0x1.363c1ddd3dcd2p253,
                0x1.363c1ddd3dcd2p254,
                0x1.3641347a78569p305,
                0x1.3645a3c4c3ddap+196,
                0x1.3647e76a9d682p554,
                0x1.3647e76a9d682p555,
                0x1.36485e50f9c6p-60,
                0x1.3649f314e523bp783,
                0x1.364c8d0585b16p-919,
                0x1.3653ccc1108dp-947,
                0x1.36565af754cf6p+516,
                0x1.36565af754cf6p517,
                0x1.36565af754cf6p518,
                0x1.367068ccd4b3p614,
                0x1.3674819514f18p-755,
                0x1.3680ed23aff88p-760,
                0x1.3680ed23aff88p-761,
                0x1.36816ab53f086p222,
                0x1.368646d8fb8aap949,
                0x1.36a37fe3fcafep+640,
                0x1.36a40ee603dcap-817,
                0x1.36a40ee603dcap-818,
                0x1.36a4984f9d4ep-634,
                0x1.36a6adfe4c5eep-680,
                0x1.36aed8cd94438p506,
                0x1.36af8128bf658p755,
                0x1.36b2251718e18p841,
                0x1.36b2251718e18p842,
                0x1.36c242313c289p-42,
                0x1.36c4c4c96eb3cp315,
                0x1.36c4e759e0a04p+352,
                0x1.36c4e759e0a04p353,
                0x1.36c82c14e824p922,
                0x1.36ce8d5ef8e48p-749,
                0x1.36d3b7c36a91ap-276,
                0x1.36d49560e49b2p-928,
                0x1.36d7926b6d4ccp-184,
                0x1.36d7926b6d4ccp-185,
                0x1.36d7926b6d4ccp-186,
                0x1.36d7926b6d4ccp-187,
                0x1.36d7926b6d4ccp-188,
                0x1.36dcp-1060,
                0x1.36e0bcf558ef9p+876,
                0x1.36e7b66f66a25p-803,
                0x1.36eb12cfc621p134,
                0x1.36eb12cfc621p135,
                0x1.36eb12cfc621p+136,
                0x1.36eb12cfc621p137,
                0x1.36f0813736b42p890,
                0x1.36f0f753aafe8p+328,
                0x1.36f7b37edfebbp-904,
                0x1.36f7c1e681323p711,
                0x1.36fc994a726c7p782,
                0x1.370727b5144p-1032,
                0x1.370c0a289afbap-270,
                0x1.370fa840771cfp342,
                0x1.3713e7da92ef2p+360,
                0x1.37148ece45774p+728,
                0x1.371ada372ff18p-56,
                0x1.37200fbf846aap+644,
                0x1.37200fbf846aap645,
                0x1.3720e5d54d97cp+816,
                0x1.3722384360192p353,
                0x1.3722384360192p354,
                0x1.3722384360192p355,
                0x1.3722384360192p+356,
                0x1.372698766619p209,
                0x1.373b4b668febep-516,
                0x1.373b4b668febep-517,
                0x1.373b4b668febep-518,
                0x1.373b4b668febep-519,
                0x1.373ce3592afa9p325,
                0x1.37419bf371bb2p-960,
                0x1.37419bf371bb2p-961,
                0x1.374636e9edb55p-940,
                0x1.374b5faa428f3p399,
                0x1.374ba17a6f204p679,
                0x1.374ba17a6f204p+680,
                0x1.374ba17a6f204p681,
                0x1.374ba17a6f204p682,
                0x1.3752659bad47ap+792,
                0x1.3752659bad47ap793,
                0x1.3752659bad47ap794,
                0x1.3758af0da1396p853,
                0x1.3758af0da1396p854,
                0x1.375b558e199edp273,
                0x1.375c4886d382ep253,
                0x1.3761db483bcf8p-435,
                0x1.3761db483bcf8p-436,
                0x1.376352949785dp-735,
                0x1.376352949785dp-736,
                0x1.3768e4eb6fce6p+888,
                0x1.3768e4eb6fce6p889,
                0x1.3768e4eb6fce6p890,
                0x1.3768e4eb6fce6p891,
                0x1.3768e4eb6fce6p+892,
                0x1.3768e4eb6fce6p893,
                0x1.376bd29d05404p-838,
                0x1.376cb30bdeb92p-1018,
                0x1.376cb30bdeb92p-1019,
                0x1.37764009f924ep+208,
                0x1.37764009f924ep209,
                0x1.37764009f924ep210,
                0x1.37764009f924ep211,
                0x1.37764009f924ep+212,
                0x1.3777b02cfd46cp-730,
                0x1.3777b02cfd46cp-731,
                0x1.37798f428563p694,
                0x1.37798f428563p695,
                0x1.37798f428563p+696,
                0x1.377f0770e75dep-223,
                0x1.377f61c42c471p-446,
                0x1.377f8c0bcbc38p238,
                0x1.3789ae03e7d66p-105,
                0x1.378d0c23df509p630,
                0x1.3796945a1f91cp793,
                0x1.3796945a1f91cp794,
                0x1.3796945a1f91cp795,
                0x1.3796945a1f91cp+796,
                0x1.3799cd9ade014p-904,
                0x1.379f099a86228p-314,
                0x1.379f099a86228p-315,
                0x1.379f099a86228p-316,
                0x1.379f099a86228p-317,
                0x1.379f099a86228p-318,
                0x1.37a6188823dcp-375,
                0x1.37a7d16adaf6cp+344,
                0x1.37a7d16adaf6cp345,
                0x1.37a95f2adb142p-598,
                0x1.37a95f2adb142p-599,
                0x1.37a95f2adb142p-600,
                0x1.37ab2483d7a39p297,
                0x1.37ab4be4b2e21p225,
                0x1.37af3cd807edap994,
                0x1.37b4cf0ccff8cp-915,
                0x1.37c177e0e9e2cp403,
                0x1.37c177e0e9e2cp+404,
                0x1.37cb099f74ac8p-820,
                0x1.37d85e2158c21p+636,
                0x1.37d88ba4b43e4p138,
                0x1.37d99cc506d58p-957,
                0x1.37dab66db46fcp354,
                0x1.37de5e23baba4p530,
                0x1.37de5e23baba4p531,
                0x1.37e09da6218a6p-324,
                0x1.37e0c6f82798ep1007,
                0x1.37e2b5f789dep-622,
                0x1.37e7eab97b24dp-84,
                0x1.37e8492d5b05dp518,
                0x1.37f7f1792cc86p-247,
                0x1.37f91e5271988p-38,
                0x1.37fb53447ab95p871,
                0x1.37fd127dd78p-917,
                0x1.38052965327fep-250,
                0x1.3808d72e39b88p+184,
                0x1.3808d72e39b88p185,
                0x1.3808e7e748c9ap-173,
                0x1.380a1f1aa075bp+684,
                0x1.380a1f1aa075bp685,
                0x1.380ed8p-1052,
                0x1.380ed8p-1053,
                0x1.38173fa77e19ep+1008,
                0x1.38192f7693e02p647,
                0x1.38192f7693e02p+648,
                0x1.381c5d507b734p-468,
                0x1.381c5d507b734p-469,
                0x1.38211f715c034p-619,
                0x1.3823fd5a1e876p-76,
                0x1.38245fafc63b6p+484,
                0x1.38245fafc63b6p485,
                0x1.3825b9718p-1040,
                0x1.3829a6ee0f598p498,
                0x1.382cc34ca2428p-472,
                0x1.382fe698f9834p865,
                0x1.383094p-1049,
                0x1.383413d84de55p+640,
                0x1.383dd003fb8fp129,
                0x1.3844bccf34b5ap+492,
                0x1.384d7b6c0fa22p-310,
                0x1.384d7b6c0fa22p-311,
                0x1.384d7b6c0fa22p-312,
                0x1.384d7b6c0fa22p-313,
                0x1.384ed65afcap-1030,
                0x1.386b46c662664p147,
                0x1.38743172c17dp271,
                0x1.3874801dfbc84p327,
                0x1.38750489b0d4ep-809,
                0x1.38750489b0d4ep-810,
                0x1.38750489b0d4ep-811,
                0x1.38750489b0d4ep-812,
                0x1.3878aeb8abf4p+332,
                0x1.387cf9cb4ad4ep-257,
                0x1.387cf9cb4ad4ep-258,
                0x1.387cf9cb4ad4ep-259,
                0x1.387cf9cb4ad4ep-260,
                0x1.387cf9cb4ad4ep-261,
                0x1.387cf9cb4ad4ep-262,
                0x1.387cf9cb4ad4ep-263,
                0x1.38857e6517013p162,
                0x1.38857e6517013p163,
                0x1.3886ac82e7d22p697,
                0x1.38877181620bcp-42,
                0x1.3888f725232b4p+744,
                0x1.388p13,
                0x1.3897268e6981p567,
                0x1.3897268e6981p+568,
                0x1.3897268e6981p569,
                0x1.38a7b43a9e5f2p-526,
                0x1.38aa388f7273ep706,
                0x1.38aa388f7273ep707,
                0x1.38aa388f7273ep+708,
                0x1.38aa388f7273ep709,
                0x1.38c251d77d8c3p-785,
                0x1.38c2p-1059,
                0x1.38d0132705fcp-252,
                0x1.38d0132705fcp-253,
                0x1.38d0132705fcp-254,
                0x1.38d2d0ed2cc42p+176,
                0x1.38d352e5096afp498,
                0x1.38d54a79c1e7ap-63,
                0x1.38d8fc91bd332p-867,
                0x1.38da479b51e2p-1008,
                0x1.38da479b51e2p-1009,
                0x1.38db5181ab3bcp-694,
                0x1.38e6edd48f2a2p-310,
                0x1.38e6edd48f2a2p-311,
                0x1.38e6edd48f2a2p-312,
                0x1.38e6edd48f2a2p-313,
                0x1.38e6edd48f2a2p-314,
                0x1.38e6edd48f2a2p-315,
                0x1.38e6edd48f2a2p-316,
                0x1.38e6edd48f2a2p-317,
                0x1.38f19a49aa27ap-895,
                0x1.38f19a49aa27ap-896,
                0x1.38f7221ac774cp-126,
                0x1.38f9cd37b8818p+520,
                0x1.39015973ac7aap-433,
                0x1.391b1fcb2fd58p-908,
                0x1.391d6ad06b0cp-929,
                0x1.3920c1a5adc3p-489,
                0x1.39223225de922p-257,
                0x1.39223225de922p-258,
                0x1.3926bc01a973bp983,
                0x1.3935505d1a394p-394,
                0x1.3935505d1a394p-395,
                0x1.39382cbb82031p-286,
                0x1.393b456eef178p-989,
                0x1.393b456eef178p-990,
                0x1.393b456eef178p-991,
                0x1.393b456eef178p-992,
                0x1.393b456eef178p-993,
                0x1.393d869eeaf86p-273,
                0x1.3940145cdb0b4p519,
                0x1.3940145cdb0b4p+520,
                0x1.3944fafb1083dp-825,
                0x1.3945063a2d43ap-301,
                0x1.394cbee428ea4p-29,
                0x1.394cbee428ea4p-30,
                0x1.394fc0f33536cp+536,
                0x1.394fc0f33536cp537,
                0x1.3953c6404cb1p-159,
                0x1.3957d158d5936p+960,
                0x1.3957d158d5936p961,
                0x1.3957d158d5936p962,
                0x1.396dbfe7a009p-1026,
                0x1.3976bb09217dap126,
                0x1.397d3c9745d2fp-389,
                0x1.397d3c9745d2fp-390,
                0x1.397d3c9745d2fp-391,
                0x1.397d3c9745d2fp-392,
                0x1.3980db75d629ap298,
                0x1.3980db75d629ap299,
                0x1.39874ddd8c623p-668,
                0x1.39874ddd8c624p-668,
                0x1.39878892a1098p-327,
                0x1.3989fbee7e624p-613,
                0x1.398a74a59ec34p669,
                0x1.398a74a59ec34p670,
                0x1.3990120d2d3a8p715,
                0x1.39957d204c5f4p414,
                0x1.39a68p-1057,
                0x1.39aabedap-1043,
                0x1.39aaf4725ec4ep-628,
                0x1.39ac1ce2cc95fp327,
                0x1.39ac1ce2cc95fp+328,
                0x1.39b03f1ba3fedp511,
                0x1.39b402c0cp-1040,
                0x1.39b88441329c2p-416,
                0x1.39b94b7960cd9p850,
                0x1.39b94b7960cd9p851,
                0x1.39c70e9be6ccfp+260,
                0x1.39d6eb1d7a46dp-570,
                0x1.39dae6f76d883p-183,
                0x1.39dae6f76d883p-184,
                0x1.39e1aa2a821a1p-687,
                0x1.39e492ae602f2p835,
                0x1.39e492ae602f2p+836,
                0x1.39e492ae602f2p837,
                0x1.39e882b5b1022p-377,
                0x1.39eb2afd8c6f5p886,
                0x1.39f3e5d7fd76bp121,
                0x1.39f61e74a4ed3p-932,
                0x1.39f91a7575e96p-948,
                0x1.39fa8c27041e6p-725,
                0x1.39fb399b93293p578,
                0x1.39fba82317009p178,
                0x1.3a021a34da814p519,
                0x1.3a0399ce2ed1ep-788,
                0x1.3a19b0e4ae236p909,
                0x1.3a1bc91cf5ce6p-524,
                0x1.3a1bc91cf5ce6p-525,
                0x1.3a22ddaf37accp+740,
                0x1.3a2ae84e2b9fap946,
                0x1.3a2d93e5ad254p-128,
                0x1.3a2dbfa8ca4fap-510,
                0x1.3a2e965b9d81cp301,
                0x1.3a2e965b9d81cp302,
                0x1.3a2e965b9d81cp303,
                0x1.3a2e965b9d81cp+304,
                0x1.3a34bbaa78df2p+756,
                0x1.3a37c0f8e99d8p637,
                0x1.3a37c0f8e99d8p638,
                0x1.3a3bd86e52492p862,
                0x1.3a3d0ce5a1f1ap707,
                0x1.3a3d0ce5a1f1ap+708,
                0x1.3a3d0ce5a1f1ap709,
                0x1.3a3d0ce5a1f1ap710,
                0x1.3a3d0ce5a1f1ap711,
                0x1.3a50e0ap-1047,
                0x1.3a570bcd47808p-144,
                0x1.3a570bcd47808p-145,
                0x1.3a570bcd47808p-146,
                0x1.3a570bcd47808p-147,
                0x1.3a5d15594798ep+664,
                0x1.3a6162ccc2fc4p377,
                0x1.3a6594ed321fcp-380,
                0x1.3a715a611269cp-720,
                0x1.3a7302561bee4p826,
                0x1.3a795459b25afp-968,
                0x1.3a825c100dd11p787,
                0x1.3a825c100dd12p787,
                0x1.3a85368352fb4p545,
                0x1.3a85368352fb4p546,
                0x1.3a8be44c15cdbp846,
                0x1.3a92a30553262p-10,
                0x1.3a92a30553262p-11,
                0x1.3a92a30553262p-12,
                0x1.3a92a30553262p-13,
                0x1.3a9dfe6956e28p881,
                0x1.3aa1ad778f23cp121,
                0x1.3aa1ad778f23cp122,
                0x1.3aa1ad778f23cp123,
                0x1.3aa1ad778f23cp+124,
                0x1.3aa39e855d62ap599,
                0x1.3aa39e855d62ap+600,
                0x1.3aa39e855d62ap601,
                0x1.3aa39e855d62ap602,
                0x1.3aa59d45fd098p-787,
                0x1.3aaa842aa784ap+736,
                0x1.3ab17128db9e8p-125,
                0x1.3abcbc09ebefap693,
                0x1.3abde2775e9b5p143,
                0x1.3abde2775e9b5p+144,
                0x1.3abde2775e9b5p145,
                0x1.3abde2775e9b5p146,
                0x1.3ad4c35c903fp998,
                0x1.3ad775629c44ep-924,
                0x1.3ada3fb0de578p+636,
                0x1.3ada3fb0de578p637,
                0x1.3ae1ac87a740dp442,
                0x1.3ae3591f5b4d9p-864,
                0x1.3ae60753af6cap758,
                0x1.3ae60753af6cap759,
                0x1.3ae60753af6cap+760,
                0x1.3ae60753af6cap761,
                0x1.3ae60753af6cap762,
                0x1.3ae60753af6cap763,
                0x1.3ae60753af6cap+764,
                0x1.3ae6836706b52p473,
                0x1.3ae6836706b52p474,
                0x1.3ae8ae20e2df8p-376,
                0x1.3af048feeeb63p-464,
                0x1.3af048feeeb63p-465,
                0x1.3af84b7040464p803,
                0x1.3afd748234046p+380,
                0x1.3b063946e10aep+956,
                0x1.3b063946e10aep957,
                0x1.3b063946e10aep958,
                0x1.3b063946e10aep959,
                0x1.3b063946e10aep+960,
                0x1.3b0727477a7f9p-824,
                0x1.3b097022327b2p287,
                0x1.3b180db83377bp+860,
                0x1.3b1dfe4854d2p-1004,
                0x1.3b1dfe4854d2p-1005,
                0x1.3b282abb82821p726,
                0x1.3b28b27523ea6p-228,
                0x1.3b28b27523ea6p-229,
                0x1.3b2a11dd840ep-173,
                0x1.3b2a11dd840ep-174,
                0x1.3b2c52029b332p-495,
                0x1.3b3650092611p-334,
                0x1.3b374f06526dep-379,
                0x1.3b391f453bf46p-243,
                0x1.3b391f453bf46p-244,
                0x1.3b391f453bf46p-245,
                0x1.3b391f453bf46p-246,
                0x1.3b3c2df518856p-490,
                0x1.3b4df9703927p+372,
                0x1.3b4fc9defaba6p266,
                0x1.3b4fc9defaba6p267,
                0x1.3b4fc9defaba6p+268,
                0x1.3b57ab4cc0e47p+884,
                0x1.3b57ab4cc0e47p885,
                0x1.3b57ab4cc0e47p886,
                0x1.3b57ab4cc0e47p887,
                0x1.3b57bfb3fb712p-964,
                0x1.3b58a49e52c19p-566,
                0x1.3b644f596522dp-305,
                0x1.3b6c0b352094p+628,
                0x1.3b6c0b352094p629,
                0x1.3b73917e0d658p671,
                0x1.3b73917e0d658p+672,
                0x1.3b78ba35778c8p-405,
                0x1.3b796ab9bd48ap-364,
                0x1.3b8292a7aca04p-357,
                0x1.3b8292a7aca04p-358,
                0x1.3b8b5b5056e16p106,
                0x1.3b916f49ef229p537,
                0x1.3b916f49ef229p538,
                0x1.3b91d813bbd5dp-772,
                0x1.3b9b0b05928d5p682,
                0x1.3b9d7bf9ea154p-501,
                0x1.3b9d7bf9ea154p-502,
                0x1.3b9d7bf9ea154p-503,
                0x1.3ba056799fae5p819,
                0x1.3ba707638e73p434,
                0x1.3baac6c570f2fp+396,
                0x1.3baac6c570f2fp397,
                0x1.3bc37521a726ap-230,
                0x1.3bc60e9dd92fbp-847,
                0x1.3bcc59cfee7e2p437,
                0x1.3bd0586101272p230,
                0x1.3bd0586101272p231,
                0x1.3bd0586101272p+232,
                0x1.3bd0586101272p233,
                0x1.3bd0586101272p234,
                0x1.3bd0586101272p235,
                0x1.3bdc85a96fd9ap-953,
                0x1.3bde3fee5474p-582,
                0x1.3bdf7e0360c36p591,
                0x1.3be3f9b78b582p-519,
                0x1.3be71cd154c0fp309,
                0x1.3beac33947372p-859,
                0x1.3bebcdb3af369p350,
                0x1.3befd70a55ccap-208,
                0x1.3bf38d881da96p-291,
                0x1.3bf38d881da96p-292,
                0x1.3bfc18c728b42p-401,
                0x1.3bp-1066,
                0x1.3c0368d739f1cp630,
                0x1.3c0368d739f1cp631,
                0x1.3c053ca96bb3ap+376,
                0x1.3c06bd7fee3acp-248,
                0x1.3c06bd7fee3acp-249,
                0x1.3c08708fd5028p+728,
                0x1.3c0cfe5a2e8fep315,
                0x1.3c0fce8c35594p-897,
                0x1.3c12582a2d975p615,
                0x1.3c169e6205486p-761,
                0x1.3c1b566c6e315p323,
                0x1.3c234b6e28f6ap-801,
                0x1.3c2da276e51e4p-108,
                0x1.3c313e2f4d26ep-827,
                0x1.3c35205f3e6d5p-374,
                0x1.3c35205f3e6d5p-375,
                0x1.3c396de6d1406p-787,
                0x1.3c41p-1058,
                0x1.3c4414884001cp277,
                0x1.3c4804f1a58f2p+772,
                0x1.3c4d67b33c789p839,
                0x1.3c4d67b33c789p+840,
                0x1.3c4p-1060,
                0x1.3c50efd573adcp743,
                0x1.3c51693638f18p-803,
                0x1.3c560ae1ecdfp-208,
                0x1.3c560ae1ecdfp-209,
                0x1.3c560ae1ecdfp-210,
                0x1.3c5753de06caep-527,
                0x1.3c5753de06caep-528,
                0x1.3c597fdbde1eap+860,
                0x1.3c597fdbde1eap861,
                0x1.3c64dca474042p-1020,
                0x1.3c64dca474042p-1021,
                0x1.3c6a941587498p219,
                0x1.3c6c9cc58f9bep-506,
                0x1.3c7535bbd85a2p-24,
                0x1.3c7a0e66e3e3cp275,
                0x1.3c7a0e66e3e3cp+276,
                0x1.3c7a0e66e3e3cp277,
                0x1.3c7a0e66e3e3cp278,
                0x1.3c7a0e66e3e3cp279,
                0x1.3c7f1d01156a6p-351,
                0x1.3c89575559469p498,
                0x1.3c898e32e4ecfp619,
                0x1.3c8b13bd97aa8p710,
                0x1.3c8b13bd97aa8p711,
                0x1.3c8b13bd97aa8p+712,
                0x1.3c8c2d576518ep674,
                0x1.3c8eeb77b8d04p1007,
                0x1.3c8eeb77b8d04p+1008,
                0x1.3c8eeb77b8d04p1009,
                0x1.3c8eeb77b8d04p1010,
                0x1.3c9539d82aec8p-575,
                0x1.3c97e460131d2p-307,
                0x1.3c9856f47d71dp-204,
                0x1.3c9856f47d71dp-205,
                0x1.3c9e13cf6da9p514,
                0x1.3c9e13cf6da9p515,
                0x1.3ca73573b2a52p-127,
                0x1.3caba046a110cp449,
                0x1.3cae4bb77e49ap+580,
                0x1.3cb302bd975ep+416,
                0x1.3cb31cb20b968p-935,
                0x1.3cb45798aab1p-343,
                0x1.3cbc88697dfc3p-743,
                0x1.3cbc96df057c2p587,
                0x1.3cc7p-1057,
                0x1.3ccb09feadfb5p645,
                0x1.3ccc96ab69604p-412,
                0x1.3cd1824be4c4p-351,
                0x1.3cd1824be4c4p-352,
                0x1.3cda841a6e18ep+548,
                0x1.3cdb48507057p-1026,
                0x1.3cdde53f85e9p-965,
                0x1.3cdde53f85e9p-966,
                0x1.3cdde53f85e9p-967,
                0x1.3cdde53f85e9p-968,
                0x1.3cdde53f85e9p-969,
                0x1.3ce772f4aa41ep401,
                0x1.3ce772f4aa41ep402,
                0x1.3ce772f4aa41ep403,
                0x1.3ce772f4aa41ep+404,
                0x1.3ce9a36f23c1p-90,
                0x1.3cea189d41d94p227,
                0x1.3cf2e271a01ffp245,
                0x1.3cfa0a3c4a006p-886,
                0x1.3cfa0a3c4a006p-887,
                0x1.3cfa0a3c4a006p-888,
                0x1.3cfa0a3c4a006p-889,
                0x1.3cfa0a3c4a006p-890,
                0x1.3cfa0ffa4a1e6p335,
                0x1.3cfef9714bfb8p-684,
                0x1.3cffbf89598a8p-595,
                0x1.3d058aeb22b88p131,
                0x1.3d086870ae831p749,
                0x1.3d0b8fe910d9ap-125,
                0x1.3d0b8fe910d9ap-126,
                0x1.3d0b8fe910d9ap-127,
                0x1.3d16f179d166p-333,
                0x1.3d16f179d166p-334,
                0x1.3d19d0c8ef6fap810,
                0x1.3d19d0c8ef6fap811,
                0x1.3d1a36872b0bep507,
                0x1.3d1e1530e4231p-849,
                0x1.3d3b71680892cp-625,
                0x1.3d3e2388029bbp395,
                0x1.3d3e2388029bcp395,
                0x1.3d3e895a36edap+732,
                0x1.3d3f2b805009cp-641,
                0x1.3d4855ac79b81p-219,
                0x1.3d4ae77399726p599,
                0x1.3d4e8eb4b6435p-404,
                0x1.3d4f9ea580c2ep+300,
                0x1.3d524bd8ab92bp-743,
                0x1.3d5783e85fcf8p-67,
                0x1.3d587d9f0bcafp-81,
                0x1.3d5ab9c5c7dacp225,
                0x1.3d61b4f557c22p-1007,
                0x1.3d68a1e25bf64p206,
                0x1.3d69b021b174dp-714,
                0x1.3d6b6decf86cdp+468,
                0x1.3d6b6decf86cdp469,
                0x1.3d6c1088ae40ep-62,
                0x1.3d6fea5e6f0e3p-130,
                0x1.3d73f3bc2d993p231,
                0x1.3d8905fb346e3p173,
                0x1.3d8b9733af506p411,
                0x1.3d8b9733af506p+412,
                0x1.3d8b9733af506p413,
                0x1.3d8b9733af506p414,
                0x1.3d8b9733af506p415,
                0x1.3d92ba28c7d14p+880,
                0x1.3d92ba28c7d14p881,
                0x1.3d94a1f635f3ap795,
                0x1.3d94a1f635f3ap+796,
                0x1.3d961ea70d24cp-445,
                0x1.3d99dc5e1a224p-981,
                0x1.3d99dc5e1a224p-982,
                0x1.3d9a651ddaea3p-564,
                0x1.3d9b25eeb22c8p-102,
                0x1.3da26dc116b3p490,
                0x1.3da93cb7fcp-921,
                0x1.3db11ac608108p-594,
                0x1.3db11ac608108p-595,
                0x1.3db11ac608108p-596,
                0x1.3db11ac608108p-597,
                0x1.3db11ac608108p-598,
                0x1.3db11ac608108p-599,
                0x1.3db11ac608108p-600,
                0x1.3db11ac608108p-601,
                0x1.3db11ac608108p-602,
                0x1.3db11ac608108p-603,
                0x1.3db11ac608108p-604,
                0x1.3dc434ad61f92p-467,
                0x1.3dc6227de9148p-408,
                0x1.3dc6227de9148p-409,
                0x1.3dc6227de9148p-410,
                0x1.3dc6227de9148p-411,
                0x1.3dc6227de9148p-412,
                0x1.3dc6227de9148p-413,
                0x1.3dcf9de19cf8ap259,
                0x1.3dcf9de19cf8ap+260,
                0x1.3dcf9de19cf8ap261,
                0x1.3dcf9de19cf8ap262,
                0x1.3dd1580d9a06p271,
                0x1.3dd1580d9a06p+272,
                0x1.3dd1580d9a06p273,
                0x1.3dd1580d9a06p274,
                0x1.3dd1580d9a06p275,
                0x1.3dd3b55eef32p917,
                0x1.3dd3b55eef32p918,
                0x1.3dd3b55eef32p919,
                0x1.3dd3b55eef32p+920,
                0x1.3dd3b55eef32p921,
                0x1.3dddd337a9f6ep-1018,
                0x1.3ddded52eb41ap-619,
                0x1.3de005bd620dfp217,
                0x1.3de591768d6e4p-1006,
                0x1.3de591768d6e4p-1007,
                0x1.3de591768d6e4p-1008,
                0x1.3de591768d6e4p-1009,
                0x1.3de843048f8f8p-439,
                0x1.3df4a91a4dcb4p-771,
                0x1.3df4a91a4dcb5p-771,
                0x1.3df53e597b704p-128,
                0x1.3df644bf1b8e8p151,
                0x1.3df644bf1b8e8p+152,
                0x1.3df644bf1b8e8p153,
                0x1.3df644bf1b8e8p154,
                0x1.3df644bf1b8e8p155,
                0x1.3df644bf1b8e8p+156,
                0x1.3df644bf1b8e8p157,
                0x1.3df644bf1b8e8p158,
                0x1.3df7db46c3a1bp566,
                0x1.3dfba8p-1053,
                0x1.3dff9309d6bcep-294,
                0x1.3dfffe1fe3c24p873,
                0x1.3e04313026519p373,
                0x1.3e0489db0f526p-836,
                0x1.3e0489db0f526p-837,
                0x1.3e0b56f1b0198p990,
                0x1.3e0c0587fa27ep-84,
                0x1.3e1197641dfdcp-481,
                0x1.3e176fdcccabdp-757,
                0x1.3e185f078969ep375,
                0x1.3e185f078969ep+376,
                0x1.3e1948bcd2eb8p+840,
                0x1.3e1948bcd2eb8p841,
                0x1.3e1b85f2ad4bep+956,
                0x1.3e1b85f2ad4bep957,
                0x1.3e1b85f2ad4bep958,
                0x1.3e1b85f2ad4bep959,
                0x1.3e1b85f2ad4bep+960,
                0x1.3e1b85f2ad4bep961,
                0x1.3e1d895a1165ep-30,
                0x1.3e272a77478e8p274,
                0x1.3e272a77478e8p275,
                0x1.3e272a77478e8p+276,
                0x1.3e272a77478e8p277,
                0x1.3e272a77478e8p278,
                0x1.3e272a77478e8p279,
                0x1.3e272a77478e8p+280,
                0x1.3e272a77478e8p281,
                0x1.3e272a77478e8p282,
                0x1.3e272a77478e8p283,
                0x1.3e344207a0e4ep850,
                0x1.3e344207a0e4ep851,
                0x1.3e3dae0dd2b9p-400,
                0x1.3e44cd8409782p-503,
                0x1.3e497065cd61ep-283,
                0x1.3e497065cd61ep-284,
                0x1.3e497065cd61ep-285,
                0x1.3e497065cd61ep-286,
                0x1.3e497065cd61ep-287,
                0x1.3e559482a72eap-920,
                0x1.3e57942acc1d7p-797,
                0x1.3e59c4f17403ep-827,
                0x1.3e59c4f17403ep-828,
                0x1.3e605ec6d2d3bp-967,
                0x1.3e605ec6d2d3bp-968,
                0x1.3e60b1694e5cep702,
                0x1.3e65240e56e58p605,
                0x1.3e65240e56e58p606,
                0x1.3e65240e56e58p607,
                0x1.3e65240e56e58p+608,
                0x1.3e65240e56e58p609,
                0x1.3e6d969fd1adcp-546,
                0x1.3e6d969fd1adcp-547,
                0x1.3e72ac4b80d0cp+836,
                0x1.3e7373510d9e4p405,
                0x1.3e75be18757ep511,
                0x1.3e75be18757ep+512,
                0x1.3e75be18757ep513,
                0x1.3e75be18757ep514,
                0x1.3e75be18757ep515,
                0x1.3e75c93b07db4p-805,
                0x1.3e7cfce0f5172p275,
                0x1.3e7d6a8c8c342p159,
                0x1.3e7f981992556p-100,
                0x1.3e80436a00d03p-875,
                0x1.3e92ad06c9474p-202,
                0x1.3e92de9f17c78p-525,
                0x1.3e92de9f17c78p-526,
                0x1.3e996a6536e9ap501,
                0x1.3e9e4e4c2f344p199,
                0x1.3e9fd85768e8cp-743,
                0x1.3ea312f0ba70ep753,
                0x1.3ea312f0ba70ep754,
                0x1.3eaecbb2c6a8ap-598,
                0x1.3eaecbb2c6a8ap-599,
                0x1.3eaecbb2c6a8ap-600,
                0x1.3eaecbb2c6a8ap-601,
                0x1.3eb65b8c6b15ep-314,
                0x1.3ec0c87eaf4bp337,
                0x1.3ec0c87eaf4bp338,
                0x1.3ec20952c6186p641,
                0x1.3ec2890a69738p239,
                0x1.3ec2a5b81bffap-710,
                0x1.3ec35b39ff514p-923,
                0x1.3ec80669009cap-453,
                0x1.3ec80669009cap-454,
                0x1.3ecb2c03c98e5p567,
                0x1.3eccfa23b067dp+652,
                0x1.3ecee3a32e98dp-468,
                0x1.3ecf22ea07862p-781,
                0x1.3ecf22ea07862p-782,
                0x1.3ecf22ea07862p-783,
                0x1.3ecf22ea07862p-784,
                0x1.3ecf22ea07862p-785,
                0x1.3ecf22ea07862p-786,
                0x1.3ecf22ea07862p-787,
                0x1.3ecffe7b6407ap-218,
                0x1.3ed333d60753ep-544,
                0x1.3ed7f999e1f5bp363,
                0x1.3ed91a958d636p711,
                0x1.3edd2ff1e0062p530,
                0x1.3edd2ff1e0062p531,
                0x1.3edd2ff1e0062p+532,
                0x1.3eecaa6b8ef64p+808,
                0x1.3eecaa6b8ef64p809,
                0x1.3eecaa6b8ef64p810,
                0x1.3eecaa6b8ef64p811,
                0x1.3eece916107acp-383,
                0x1.3ef342d37a408p+684,
                0x1.3ef37fa5acffcp866,
                0x1.3f16cbbb69a64p-232,
                0x1.3f16cbbb69a64p-233,
                0x1.3f1750b50bf7dp309,
                0x1.3f17873aafea6p+724,
                0x1.3f190452a29f4p-1021,
                0x1.3f2013d6fac76p217,
                0x1.3f2013d6fac76p218,
                0x1.3f30c32e50d38p453,
                0x1.3f30c32e50d38p454,
                0x1.3f30da347bde6p162,
                0x1.3f30da347bde6p163,
                0x1.3f311e5f35118p+972,
                0x1.3f3b182f6f32ap662,
                0x1.3f3b7f7c862f9p954,
                0x1.3f4144b0178f1p+812,
                0x1.3f52p-1059,
                0x1.3f549ed8afeecp595,
                0x1.3f549ed8afeecp+596,
                0x1.3f549ed8afeecp597,
                0x1.3f559aed56932p434,
                0x1.3f559aed56932p435,
                0x1.3f559aed56932p+436,
                0x1.3f559aed56932p437,
                0x1.3f559e7bee6c1p-967,
                0x1.3f58d434a2baep369,
                0x1.3f58d434a2baep370,
                0x1.3f5d7a39bf73ep982,
                0x1.3f695b427c952p-429,
                0x1.3f6d3441186a1p-170,
                0x1.3f726a259d405p-500,
                0x1.3f74bc0d4fc8ep-351,
                0x1.3f75f55d669b8p445,
                0x1.3f79ee12451ebp-927,
                0x1.3f7c544d9c57cp915,
                0x1.3f7ced916872bp-5,
                0x1.3f81eaf3d45e2p-192,
                0x1.3f846e311d269p+364,
                0x1.3f856485793e1p491,
                0x1.3f856485793e1p+492,
                0x1.3f866137dbf72p+180,
                0x1.3f866137dbf72p181,
                0x1.3f8a3d7a3b923p-715,
                0x1.3f8fb7791bc6ep385,
                0x1.3f93f1d2133cfp-258,
                0x1.3f93f1d2133cfp-259,
                0x1.3f9dab04433c9p-172,
                0x1.3fa93e92f9e9ap+212,
                0x1.3faac3e3fa1f3p-482,
                0x1.3faac3e3fa1f4p-478,
                0x1.3faac3e3fa1f4p-479,
                0x1.3faac3e3fa1f4p-480,
                0x1.3faac3e3fa1f4p-481,
                0x1.3faac3e3fa1f4p-482,
                0x1.3faac3e3fa1f4p-483,
                0x1.3fadfa77c8348p855,
                0x1.3fb069f6b0fa4p-12,
                0x1.3fb0a3ffa045cp-524,
                0x1.3fb0a3ffa045cp-525,
                0x1.3fb4177907d62p-147,
                0x1.3fb5a68b4de2ap+444,
                0x1.3fc4efb205912p-114,
                0x1.3fc4efb205912p-115,
                0x1.3fc4efb205912p-116,
                0x1.3fc4efb205912p-117,
                0x1.3fc4efb205912p-118,
                0x1.3fc4efb205912p-119,
                0x1.3fc9e3c6be3p-425,
                0x1.3fcp-1064,
                0x1.3fd22628c4ba7p-253,
                0x1.3fd377f413acap-80,
                0x1.3fd44687ab396p+276,
                0x1.3fd843ec75edbp-484,
                0x1.3fdea1769703p501,
                0x1.3fee0f647a7eap437,
                0x1.3ff541b2e6136p-381,
                0x1.3p-1070,
                0x1.4001b585e860ep-429,
                0x1.4007b0b6a5bd6p+644,
                0x1.400c1d408aeacp-687,
                0x1.4023f66ec3fd2p-592,
                0x1.402ce32246e16p+920,
                0x1.402ce32246e16p921,
                0x1.402db7e9ee90bp-730,
                0x1.402fce9be95fep278,
                0x1.402fce9be95fep279,
                0x1.40318114d243ap-546,
                0x1.403dbd70c65eap753,
                0x1.403dbd70c65eap754,
                0x1.40474ed1d1811p-279,
                0x1.404baf54249bcp311,
                0x1.404cf18de8102p283,
                0x1.4050669985a86p-466,
                0x1.4050669985a86p-467,
                0x1.4050669985a86p-468,
                0x1.4050669985a86p-469,
                0x1.405552d60dbd6p+488,
                0x1.405deef4bdef6p-41,
                0x1.4060a6a2d1ca6p+588,
                0x1.4066d88c7cab8p919,
                0x1.406b0cd17fd56p143,
                0x1.406b0cd17fd56p+144,
                0x1.406b0cd17fd56p145,
                0x1.406cfad0678ep+832,
                0x1.406cfad0678ep833,
                0x1.406cfad0678ep834,
                0x1.406cfad0678ep835,
                0x1.406cfad0678ep+836,
                0x1.4070e8c84997dp319,
                0x1.40782155f2f9ep+788,
                0x1.407b99ed0da9p273,
                0x1.407b99ed0da9p274,
                0x1.40804d8a1d7cp499,
                0x1.40804d8a1d7cp+500,
                0x1.4086fa9ee1098p-408,
                0x1.4091f3f80688p1021,
                0x1.4091f3f80688p1022,
                0x1.4092be0960b27p-132,
                0x1.4092be0960b27p-133,
                0x1.4092d334cf4p-1032,
                0x1.40942000b01d1p955,
                0x1.4098205bd53c2p-466,
                0x1.40986f362aacap-222,
                0x1.40986f362aacap-223,
                0x1.40986f362aacap-224,
                0x1.40986f362aacap-225,
                0x1.40986f362aacap-226,
                0x1.40986f362aacap-227,
                0x1.409883759c288p-526,
                0x1.409883759c288p-527,
                0x1.409883759c288p-528,
                0x1.409d80caca452p-475,
                0x1.409e3a188c43p-275,
                0x1.409e3a188c43p-276,
                0x1.40a5f7a9b9a0ap330,
                0x1.40a5f7a9b9a0ap331,
                0x1.40a5f7a9b9a0ap+332,
                0x1.40a5f7a9b9a0ap333,
                0x1.40a619730719p-161,
                0x1.40aa0d0120ceap795,
                0x1.40aa329d226e6p773,
                0x1.40aabc6c32a38p973,
                0x1.40aabc6c32a38p974,
                0x1.40aabc6c32a38p975,
                0x1.40b6177fda395p949,
                0x1.40bfda1f51cbap-116,
                0x1.40c746f8dc2a3p-1004,
                0x1.40c969c3345ep-567,
                0x1.40caacd9e198ap-744,
                0x1.40caacd9e198ap-745,
                0x1.40caacd9e198ap-746,
                0x1.40caacd9e198ap-747,
                0x1.40ce696028c4p-527,
                0x1.40d76077b648ep-971,
                0x1.40d76077b648ep-972,
                0x1.40d76077b648ep-973,
                0x1.40d76077b648ep-974,
                0x1.40d76077b648ep-975,
                0x1.40d76077b648ep-976,
                0x1.40d76077b648ep-977,
                0x1.40de6f864df15p529,
                0x1.40dfb625a8f86p-776,
                0x1.40f232256e982p86,
                0x1.40f232256e982p87,
                0x1.40f232256e982p+88,
                0x1.40f40f0a26d32p+800,
                0x1.40f40f0a26d32p801,
                0x1.40f40f0a26d32p802,
                0x1.40f40f0a26d32p803,
                0x1.40f40f0a26d32p+804,
                0x1.40f8d54ff639cp-941,
                0x1.40ffa17429133p-601,
                0x1.4106b6e9c2787p+232,
                0x1.4106b6e9c2787p233,
                0x1.410d9f9b2f7f2p-678,
                0x1.410d9f9b2f7f3p-678,
                0x1.4110fbf95f5dcp-270,
                0x1.4112241c3fcf2p998,
                0x1.4116a1174f06p-74,
                0x1.4116a1174f06p-75,
                0x1.4116d591ef1fap-638,
                0x1.4116d591ef1fap-639,
                0x1.4116d591ef1fap-640,
                0x1.4116d591ef1fap-641,
                0x1.4116d591ef1fap-642,
                0x1.4116fd8aa9ce8p-343,
                0x1.4121d20a0eed2p-172,
                0x1.4121d20a0eed2p-173,
                0x1.4121d20a0eed2p-174,
                0x1.4121d20a0eed2p-175,
                0x1.4121d20a0eed2p-176,
                0x1.4121d20a0eed2p-177,
                0x1.4123580f054b4p661,
                0x1.4125d959e023ap383,
                0x1.4127216d831c3p713,
                0x1.4127216d831c3p714,
                0x1.412efae44a7aap-640,
                0x1.412efae44a7aap-641,
                0x1.412efae44a7aap-642,
                0x1.412efae44a7aap-643,
                0x1.412efae44a7aap-644,
                0x1.4132c13e74b36p-204,
                0x1.4132c13e74b36p-205,
                0x1.4139a6b17b224p-753,
                0x1.4139a6b17b224p-754,
                0x1.4139a6b17b224p-755,
                0x1.4139a6b17b224p-756,
                0x1.4139a6b17b224p-757,
                0x1.4140b5a50567ep-238,
                0x1.4140b5a50567ep-239,
                0x1.414d8ec363013p+224,
                0x1.414f4ee1e940ep-157,
                0x1.4151db545f25cp-465,
                0x1.4151db545f25cp-466,
                0x1.41633a556e1cep-193,
                0x1.4166f8cfd5cb1p542,
                0x1.4169819f1f872p-766,
                0x1.416f67c3db856p798,
                0x1.41717ab11a367p597,
                0x1.4171e186cdb6ap-819,
                0x1.41724063a9e44p-314,
                0x1.4174a6cadc654p639,
                0x1.4174a6cadc654p+640,
                0x1.4177a9915fbf8p271,
                0x1.4177a9915fbf8p+272,
                0x1.417ac36dbdacbp617,
                0x1.418bf1e207406p486,
                0x1.4191129273992p825,
                0x1.41ae8b7e6525cp387,
                0x1.41afp-1058,
                0x1.41b148d95c771p-795,
                0x1.41b8ebe2ef1c7p+292,
                0x1.41c593344ebcp381,
                0x1.41c6320ee0322p-510,
                0x1.41c9926973ab8p-507,
                0x1.41cb01be557e2p-422,
                0x1.41cb01be557e2p-423,
                0x1.41dc9fc32bfc2p+848,
                0x1.41dd49fce05p-83,
                0x1.41e01d5d3ee48p682,
                0x1.41e41232710d4p+688,
                0x1.41e41232710d4p689,
                0x1.41e436c3bf068p409,
                0x1.41e436c3bf068p410,
                0x1.41e5a5ee5c992p-328,
                0x1.41e5a5ee5c992p-329,
                0x1.41e9224f5da24p-1008,
                0x1.41ee49f0935b4p-960,
                0x1.41f04a21a9a4ep-99,
                0x1.41f25116d6695p490,
                0x1.41f704e06038p-1025,
                0x1.41f704e06038p-1026,
                0x1.41f704e06038p-1027,
                0x1.420252069cf68p+824,
                0x1.420705090619ap-343,
                0x1.420944969fa1bp-45,
                0x1.420d42d2b499cp-897,
                0x1.420e4899395b4p-293,
                0x1.420eb449c8842p777,
                0x1.420eb449c8843p777,
                0x1.421160c512f9p-96,
                0x1.421160c512f9p-97,
                0x1.4220027a2977cp-933,
                0x1.42271b8de8242p666,
                0x1.4229e766998p-1030,
                0x1.42300b14bd484p-819,
                0x1.42300b14bd484p-820,
                0x1.42300b14bd484p-821,
                0x1.423312e254f59p+840,
                0x1.423538c968ae2p+636,
                0x1.42498ae0d13cbp239,
                0x1.42568ae58ep-1035,
                0x1.425af223701f3p-335,
                0x1.425e9d29e558ep-874,
                0x1.425fe9f576799p-36,
                0x1.4266193744e9dp802,
                0x1.426d460536202p861,
                0x1.426d460536202p862,
                0x1.426d460536202p863,
                0x1.426d460536202p+864,
                0x1.426d460536202p865,
                0x1.426dd14c2ed3ep-491,
                0x1.4272053ed4474p-874,
                0x1.427542f8dbdc4p463,
                0x1.42756ea2c06fp-612,
                0x1.427b6d943962ap-539,
                0x1.428095033db3p-612,
                0x1.428610e59e90cp921,
                0x1.428610e59e90cp922,
                0x1.428610e59e90cp923,
                0x1.428a8e99f51f4p397,
                0x1.428a8e99f51f4p398,
                0x1.4291a1ae6f824p-746,
                0x1.4296af63d168p-834,
                0x1.429b844d95088p-30,
                0x1.429d71a263148p-479,
                0x1.429d9e5c5200dp-19,
                0x1.42a62384e681ep271,
                0x1.42a6da28da0f8p298,
                0x1.42a6da28da0f8p299,
                0x1.42a6da28da0f8p+300,
                0x1.42a6da28da0f8p301,
                0x1.42a6da28da0f8p302,
                0x1.42a6da28da0f8p303,
                0x1.42ac4eb92829dp115,
                0x1.42af984d8aafep-1015,
                0x1.42b710bc73dcap621,
                0x1.42c330299993p-835,
                0x1.42c7ff0054684p-389,
                0x1.42c7ff0054685p-389,
                0x1.42c995e60680ap682,
                0x1.42cbdeac842cdp981,
                0x1.42cf31b24c3e6p-682,
                0x1.42cf5795eba18p-338,
                0x1.42cf5795eba18p-339,
                0x1.42cf5795eba18p-340,
                0x1.42cf5795eba18p-341,
                0x1.42d3a1747edfap-399,
                0x1.42d3a1747edfap-400,
                0x1.42d3d932d6422p953,
                0x1.42d40a6ca561ep-463,
                0x1.42dad4f5c1572p281,
                0x1.42dbf25096cf5p149,
                0x1.42de8b7b0018ap-446,
                0x1.42de8b7b0018ap-447,
                0x1.42f12ff95bacap881,
                0x1.42f12ff95bacap882,
                0x1.42f12ff95bacap883,
                0x1.42f12ff95bacap+884,
                0x1.42f12ff95bacap885,
                0x1.42f12ff95bacap886,
                0x1.42f1cff3f0f2ep733,
                0x1.42fb5b6e2040ep339,
                0x1.42fb5b6e2040ep+340,
                0x1.42fc185bb4cf6p-710,
                0x1.42fc185bb4cf6p-711,
                0x1.430af6d6136d6p+268,
                0x1.431e0fae6d721p+96,
                0x1.431e0fae6d722p+100,
                0x1.431e0fae6d722p95,
                0x1.431e0fae6d722p+96,
                0x1.431e0fae6d722p97,
                0x1.431e0fae6d722p98,
                0x1.431e0fae6d722p99,
                0x1.4323c02bd15fep-661,
                0x1.4323c02bd15fep-662,
                0x1.432ac03ebea7ap-991,
                0x1.432ac03ebea7ap-992,
                0x1.432ac03ebea7ap-993,
                0x1.432ac03ebea7ap-994,
                0x1.432ac03ebea7ap-995,
                0x1.432ac03ebea7ap-996,
                0x1.432e7ea8728fp270,
                0x1.432e7ea8728fp271,
                0x1.432e7ea8728fp+272,
                0x1.432e7ea8728fp273,
                0x1.432e7ea8728fp274,
                0x1.432eb27bb2aa4p+324,
                0x1.432eb27bb2aa4p325,
                0x1.432ec8b0b8436p-700,
                0x1.432ec8b0b8436p-701,
                0x1.432ec8b0b8436p-702,
                0x1.432ec8b0b8436p-703,
                0x1.432ec8b0b8436p-704,
                0x1.433a4f950417dp817,
                0x1.433c88a16f9ecp-859,
                0x1.433c88a16f9ecp-860,
                0x1.433c88a16f9ecp-861,
                0x1.43419ee1459afp887,
                0x1.4342962764977p-279,
                0x1.4342962764977p-280,
                0x1.434c40da2c8dap-444,
                0x1.434c40da2c8dap-445,
                0x1.4350f08d39302p447,
                0x1.4353c176aec34p462,
                0x1.43553763cef4ap+620,
                0x1.43553763cef4ap621,
                0x1.43553763cef4ap622,
                0x1.43553763cef4ap623,
                0x1.43593d8c2c292p-548,
                0x1.435b005dc8456p314,
                0x1.436883e62863ep733,
                0x1.436883e62863ep734,
                0x1.436883e62863ep735,
                0x1.436883e62863ep+736,
                0x1.436883e62863ep737,
                0x1.436b194f1d87ep541,
                0x1.436c0c0a487f2p-465,
                0x1.436c0c0a487f2p-466,
                0x1.436c0c0a487f2p-467,
                0x1.436c0c0a487f2p-468,
                0x1.436c0c0a487f2p-469,
                0x1.436c8850b7968p+332,
                0x1.4374374f3c2c6p581,
                0x1.4374374f3c2c6p582,
                0x1.4374d8b87ac63p+612,
                0x1.4374d8b87ac63p613,
                0x1.4374d8b87ac63p614,
                0x1.4375b2214e1b5p+612,
                0x1.4375b2214e1b5p613,
                0x1.4375b2214e1b5p614,
                0x1.4377c3a82b82ep-335,
                0x1.437c66a37f447p802,
                0x1.437c66a37f447p803,
                0x1.437c66a37f447p+804,
                0x1.43809b76628c6p-60,
                0x1.438acfb8f7e99p-656,
                0x1.438c4c325e18cp199,
                0x1.4390c68b888cep786,
                0x1.439202b66724ep-1000,
                0x1.439202b66724ep-1001,
                0x1.439202b66724ep-1002,
                0x1.439202b66724ep-999,
                0x1.43a3c0e7d9bdcp363,
                0x1.43a669d28020cp-666,
                0x1.43a69adec4b6ep+872,
                0x1.43a69adec4b6ep873,
                0x1.43a69adec4b6ep874,
                0x1.43a69adec4b6ep875,
                0x1.43a69adec4b6ep+876,
                0x1.43a69adec4b6ep877,
                0x1.43b8daf891822p797,
                0x1.43b9802363ebap853,
                0x1.43b9802363ebap854,
                0x1.43b9a54dfce4p-811,
                0x1.43bb70d1be571p239,
                0x1.43bbb7bb63c2dp1013,
                0x1.43bf0c1a50ddp227,
                0x1.43bf73b094582p+860,
                0x1.43bf780c0ba9ap791,
                0x1.43cd4d7c6a8a7p-851,
                0x1.43d96f466f861p-826,
                0x1.43e72fcd3aeb2p639,
                0x1.43e72fcd3aeb2p+640,
                0x1.43e72fcd3aeb2p641,
                0x1.43ebfaa3be368p-532,
                0x1.43f35e0b2a0c9p619,
                0x1.4401ae80fc15cp914,
                0x1.44088ad226f94p445,
                0x1.44088ad226f94p446,
                0x1.44088ad226f94p447,
                0x1.44088ad226f94p+448,
                0x1.440a164p-1045,
                0x1.4412be23755e2p-365,
                0x1.4412be23755e2p-366,
                0x1.4412be23755e2p-367,
                0x1.4415147265852p-538,
                0x1.441aae1c3984ap199,
                0x1.441aae1c3984ap+200,
                0x1.441aae1c3984ap201,
                0x1.441ed79830182p-438,
                0x1.441ed79830182p-439,
                0x1.441ed79830182p-440,
                0x1.441ed79830182p-441,
                0x1.441ed79830182p-442,
                0x1.441ed79830182p-443,
                0x1.441ed79830182p-444,
                0x1.441ed79830182p-445,
                0x1.441fc7672f80cp831,
                0x1.442c4cf422f98p819,
                0x1.442c4cf422f98p+820,
                0x1.442e4fb67196p-585,
                0x1.442e4fb67196p-586,
                0x1.443191e7bc968p+752,
                0x1.443191e7bc968p753,
                0x1.44379b716bda6p-368,
                0x1.4437c5023b671p-250,
                0x1.44398cc893d69p-473,
                0x1.443a315c561e6p545,
                0x1.44430f0142be1p-786,
                0x1.4443c52d36cap-652,
                0x1.44443d019604ap-624,
                0x1.4449185a4c82ap-30,
                0x1.4449185a4c82ap-31,
                0x1.444b34a6fb3ecp202,
                0x1.444b34a6fb3ecp203,
                0x1.444b34a6fb3ecp+204,
                0x1.444b34a6fb3ecp205,
                0x1.444b34a6fb3ecp206,
                0x1.444b34a6fb3ecp207,
                0x1.444b34a6fb3ecp+208,
                0x1.444b34a6fb3ecp209,
                0x1.444e0f8b599dp-988,
                0x1.444e0f8b599dp-989,
                0x1.444e2ca485d4p-353,
                0x1.444f926fce264p-844,
                0x1.44526421007dap390,
                0x1.4454e5c471045p-114,
                0x1.445ef0f85dfe7p931,
                0x1.44604b669dc1p155,
                0x1.446b5f354c16p329,
                0x1.446df013d370ap+720,
                0x1.4471f10b6bee8p+236,
                0x1.4471f10b6bee8p237,
                0x1.4472ba9550e86p-820,
                0x1.4472ba9550e86p-821,
                0x1.4472ba9550e86p-822,
                0x1.4472ba9550e86p-823,
                0x1.4472ba9550e86p-824,
                0x1.4472ba9550e86p-825,
                0x1.4472ba9550e86p-826,
                0x1.4472ba9550e86p-827,
                0x1.4472ba9550e86p-828,
                0x1.447ee7e718c9bp-638,
                0x1.4484bfeebc2ap-100,
                0x1.4484bfeebc2ap-101,
                0x1.4484bfeebc2ap-96,
                0x1.4484bfeebc2ap-97,
                0x1.4484bfeebc2ap-98,
                0x1.4484bfeebc2ap-99,
                0x1.4484ec6aa76f2p425,
                0x1.4486f9e5085c7p-358,
                0x1.448e2p-1055,
                0x1.448e37c020e55p235,
                0x1.4492a2059f706p+860,
                0x1.4492a2059f706p861,
                0x1.4492a2059f706p862,
                0x1.4492a2059f706p863,
                0x1.4492a2059f706p+864,
                0x1.4492a2059f706p865,
                0x1.4492a2059f706p866,
                0x1.4495736b8d90ep441,
                0x1.44958b8151ebfp-899,
                0x1.4496bce738e9ap738,
                0x1.449faddf8bb4bp-704,
                0x1.44a2b250cbe2cp-769,
                0x1.44a9ca8ba155cp-12,
                0x1.44b5ba35f4f9ap-230,
                0x1.44ba73e85628p-859,
                0x1.44ba73e85628p-860,
                0x1.44c2c1a37b1a4p-301,
                0x1.44c76719a8002p-688,
                0x1.44ce1e5p-1046,
                0x1.44d4253ecp-1040,
                0x1.44db473335deep385,
                0x1.44db473335defp385,
                0x1.44ea48cab9764p+684,
                0x1.44eb32a17d418p-932,
                0x1.44f179959b88p-888,
                0x1.44f179959b88p-889,
                0x1.44fa72414dbc6p+440,
                0x1.44fb644e466b4p+408,
                0x1.44fb644e466b4p409,
                0x1.44fb644e466b4p410,
                0x1.44fbfe94c0ec1p939,
                0x1.44fbfe94c0ec1p+940,
                0x1.44p-1068,
                0x1.450030e26c6dcp-32,
                0x1.4500de008957cp954,
                0x1.450803d29a81fp927,
                0x1.450efdc9c4da9p-12,
                0x1.450efdc9c4da9p-13,
                0x1.451207f052e0ap-851,
                0x1.45152b6f85e0ap-15,
                0x1.4518c6d94ac69p-243,
                0x1.451d12456223fp+360,
                0x1.4522c9e2190cp-987,
                0x1.4529a28d5c17ep382,
                0x1.4529a28d5c17ep383,
                0x1.452cbf7193bfp-975,
                0x1.4531e58a03e8cp870,
                0x1.45325cea62d07p325,
                0x1.453b4a9ab273dp-87,
                0x1.4544b4b3bf903p-222,
                0x1.45457ee24abd2p-526,
                0x1.45457ee24abd2p-527,
                0x1.45457ee24abd2p-528,
                0x1.45457ee24abd2p-529,
                0x1.45457ee24abd2p-530,
                0x1.45469af0e9b74p-727,
                0x1.45469af0e9b74p-728,
                0x1.45469af0e9b74p-729,
                0x1.45532dff6d621p-728,
                0x1.45532dff6d621p-729,
                0x1.455848ce9ea4p+660,
                0x1.4562394a76ec2p-55,
                0x1.45688b34f0d9p622,
                0x1.45688b34f0d9p623,
                0x1.456f1300d6a3fp-25,
                0x1.456f1300d6a3fp-26,
                0x1.4575b66c1c7f6p589,
                0x1.457a45caa124fp-587,
                0x1.45822ddab8dcep597,
                0x1.45862766ap-1039,
                0x1.458e7b900be0bp521,
                0x1.4590a268307b2p830,
                0x1.459131c8p-1045,
                0x1.45962e2f6a49p-781,
                0x1.4599ae61a15d2p787,
                0x1.4599b679c8bfp-362,
                0x1.459ab6e193558p574,
                0x1.459e40fe15264p-378,
                0x1.459e40fe15264p-379,
                0x1.459ebda8cb90ep-169,
                0x1.459ebda8cb90ep-170,
                0x1.45a41a9e1b2c8p865,
                0x1.45b54ff67f8f6p-48,
                0x1.45bc6d274f3e3p958,
                0x1.45bd4a96d3936p-64,
                0x1.45c70b0709e36p-115,
                0x1.45ccbc24b6959p-561,
                0x1.45cdd2013b548p623,
                0x1.45cdd2013b548p+624,
                0x1.45d4c0f2d121ap-869,
                0x1.45d4c0f2d121ap-870,
                0x1.45d4c0f2d121ap-871,
                0x1.45d4c0f2d121ap-872,
                0x1.45d4c0f2d121ap-873,
                0x1.45d83c92f49edp325,
                0x1.45dafe22da70ep+988,
                0x1.45dafe22da70ep989,
                0x1.45dafe22da70ep990,
                0x1.45dafe22da70ep991,
                0x1.45e352dc4013fp-883,
                0x1.45ea79655da12p-961,
                0x1.45ecfe5bf520ap-296,
                0x1.45ecfe5bf520bp-296,
                0x1.45f8eedbab58ap782,
                0x1.45fe55c41873ep773,
                0x1.45fe55c41873ep774,
                0x1.45fe5b136a4c2p417,
                0x1.460dd6b5d9defp227,
                0x1.4614c3219891fp-160,
                0x1.4614c3219891fp-161,
                0x1.4614c3219891fp-162,
                0x1.4616b04e33b2bp645,
                0x1.461e708e78826p418,
                0x1.4620e084ff46ap-693,
                0x1.4620e084ff46ap-694,
                0x1.46258b0743459p101,
                0x1.462664df08c96p-691,
                0x1.462664df08c96p-692,
                0x1.462b3b8ec24d3p-437,
                0x1.462d9aecdc2bap-837,
                0x1.462f791429ed5p-741,
                0x1.462f791429ed5p-742,
                0x1.462p-1063,
                0x1.463417a258f36p+592,
                0x1.4635d9f81a678p-573,
                0x1.4635d9f81a678p-574,
                0x1.4635d9f81a678p-575,
                0x1.4635d9f81a678p-576,
                0x1.4635d9f81a678p-577,
                0x1.4635d9f81a678p-578,
                0x1.4638298e8p-1040,
                0x1.463b5e2b98p-1037,
                0x1.4642a595cc389p1021,
                0x1.4643e5ae44d12p189,
                0x1.464517f346f2fp317,
                0x1.464d6a8b642bbp+524,
                0x1.46593deb0d42ap+96,
                0x1.46593deb0d42ap97,
                0x1.465b6562f61bap-678,
                0x1.465b6562f61bap-679,
                0x1.465b6562f61bap-680,
                0x1.465b6562f61bap-681,
                0x1.465b896c2452p-951,
                0x1.465b896c2452p-952,
                0x1.465b896c2452p-953,
                0x1.465b896c2452p-954,
                0x1.465b896c2452p-955,
                0x1.465b896c2452p-956,
                0x1.465b896c2452p-957,
                0x1.465b896c2452p-958,
                0x1.465b896c2452p-959,
                0x1.465b896c2452p-960,
                0x1.465b896c2452p-961,
                0x1.4662731d3f3e1p-810,
                0x1.46667800f1348p+168,
                0x1.46708fa963826p-1002,
                0x1.46708fa963826p-1003,
                0x1.46708fa963826p-1004,
                0x1.46708fa963826p-1005,
                0x1.46708fa963826p-1006,
                0x1.4677812d3a606p729,
                0x1.4677812d3a606p730,
                0x1.4689b4a5fa202p538,
                0x1.4689b4a5fa202p539,
                0x1.4689b4a5fa202p+540,
                0x1.4689b4a5fa202p541,
                0x1.4689b4a5fa202p542,
                0x1.4689b4a5fa202p543,
                0x1.4689b4a5fa202p+544,
                0x1.4689b4a5fa202p545,
                0x1.469010e924cf2p194,
                0x1.469a573606a06p238,
                0x1.469ae42c8560cp674,
                0x1.469bd67a9d47bp885,
                0x1.46a3418629ef6p113,
                0x1.46a3418629ef6p114,
                0x1.46a3418629ef6p115,
                0x1.46a3418629ef6p+116,
                0x1.46a3418629ef6p117,
                0x1.46a3418629ef6p118,
                0x1.46a3418629ef6p119,
                0x1.46a3418629ef6p+120,
                0x1.46a46df56660cp-990,
                0x1.46af58750cb32p-153,
                0x1.46af58750cb32p-154,
                0x1.46af58750cb32p-155,
                0x1.46af58750cb32p-156,
                0x1.46af58750cb32p-157,
                0x1.46b16979236fp+152,
                0x1.46b2137e90646p-290,
                0x1.46b2137e90646p-291,
                0x1.46b2137e90646p-292,
                0x1.46b2137e90646p-293,
                0x1.46b2137e90646p-294,
                0x1.46b2137e90646p-295,
                0x1.46b2137e90646p-296,
                0x1.46b80088ee1b1p170,
                0x1.46b8544c36d26p-195,
                0x1.46c428088e665p-851,
                0x1.46d09a1bbd048p-493,
                0x1.46d33091a37ap-522,
                0x1.46d631c98a2adp-207,
                0x1.46df2aa8fb754p-72,
                0x1.46df4b737381ap-560,
                0x1.46e01567c342p-926,
                0x1.46e01567c342p-927,
                0x1.46e7dc08b17f8p-720,
                0x1.46ea80225b54p-996,
                0x1.46ea80225b54p-997,
                0x1.46eb27fce1757p-611,
                0x1.46eb27fce1757p-612,
                0x1.46eb27fce1757p-613,
                0x1.46ef23fbf84p-1031,
                0x1.46ef23fbf84p-1032,
                0x1.46f7ee209483ap-404,
                0x1.46f7ee209483ap-405,
                0x1.46fe7cce7292bp999,
                0x1.46ff9c24cb2f2p-977,
                0x1.46ff9c24cb2f3p-977,
                0x1.46p-1067,
                0x1.4709267befac6p-642,
                0x1.470fbbf78bb24p-451,
                0x1.471295bf9e338p-587,
                0x1.4712c1705019p-1026,
                0x1.47139f8eca91ap574,
                0x1.47139f8eca91ap575,
                0x1.47139f8eca91ap+576,
                0x1.47178p-1057,
                0x1.4724cad338716p-936,
                0x1.4724cad338716p-937,
                0x1.4724cad338716p-938,
                0x1.472e8ee9f2318p-322,
                0x1.472e8ee9f2318p-323,
                0x1.473104dfd0578p+928,
                0x1.47326e213daf8p-170,
                0x1.47326e213daf8p-171,
                0x1.47326e213daf8p-172,
                0x1.47326e213daf8p-173,
                0x1.47326e213daf8p-174,
                0x1.4738090d75438p-72,
                0x1.4738090d75438p-73,
                0x1.474146b188d3p947,
                0x1.47476ad989494p614,
                0x1.4749d40074f09p182,
                0x1.474c802e307f4p+884,
                0x1.474dbf02e56c2p651,
                0x1.4756ccb01abfbp-492,
                0x1.4756ccb01abfcp-491,
                0x1.4756ccb01abfcp-492,
                0x1.475e6a5875eep-232,
                0x1.4761a1ce3c46bp-484,
                0x1.4763e2d29291ep-509,
                0x1.4763e2d29291ep-510,
                0x1.47726460f332ep889,
                0x1.478071ce3bc53p210,
                0x1.4781cd750d72cp-859,
                0x1.4781cd750d72cp-860,
                0x1.4782d38513d9ap-322,
                0x1.478dccaa9185bp-709,
                0x1.478dccaa9185bp-710,
                0x1.47962237dc6ecp-271,
                0x1.47966fc77f444p719,
                0x1.479c2dde4p-1040,
                0x1.479ee09260e26p-256,
                0x1.47a537d168982p+216,
                0x1.47a5f0249692cp577,
                0x1.47a5f0249692cp578,
                0x1.47ae147ae147ap-7,
                0x1.47ae147ae147bp-7,
                0x1.47aed2f2bcc01p657,
                0x1.47b0f868dcp-1036,
                0x1.47b8ca7809b6p-782,
                0x1.47be4ef775c1ap-736,
                0x1.47bf052eac347p-79,
                0x1.47c8bf78p-1045,
                0x1.47cb1222f139fp+972,
                0x1.47d51527ac532p+464,
                0x1.47dbd9a13e7eap345,
                0x1.47ddab541e288p889,
                0x1.47ddab541e288p890,
                0x1.47dfd29e829aep-553,
                0x1.47e17e7814ce8p901,
                0x1.47e17e7814ce8p902,
                0x1.47e17e7814ce8p903,
                0x1.47e17e7814ce8p+904,
                0x1.47e17e7814ce8p905,
                0x1.47e17e7814ce8p906,
                0x1.47e3ce1871d7ap-48,
                0x1.47ec27dadcf38p863,
                0x1.47ec27dadcf38p+864,
                0x1.47ed0d61fd4eap-28,
                0x1.47fafa6c61958p-619,
                0x1.4801da861f2dfp639,
                0x1.4805738b51a74p478,
                0x1.4805738b51a75p478,
                0x1.480f0586255f8p813,
                0x1.48166d86604ccp-320,
                0x1.48166d86604ccp-321,
                0x1.481a7a1b198c2p570,
                0x1.481ae13be236ap858,
                0x1.481ae13be236ap859,
                0x1.4820b6f15ff21p-374,
                0x1.482a87ed2727ep103,
                0x1.4844b3ac8a6a4p247,
                0x1.4844b3ac8a6a4p+248,
                0x1.4848f247491e1p887,
                0x1.484f66f5501b4p-121,
                0x1.4856a2e4p-1041,
                0x1.48590cda34c17p141,
                0x1.48590cda34c17p142,
                0x1.485aa138de80cp-401,
                0x1.485aa138de80cp-402,
                0x1.485ce9e7a065ep963,
                0x1.485ce9e7a065ep+964,
                0x1.48648e7fd69fap-706,
                0x1.486950beb4a23p-688,
                0x1.486ad9f25a5e2p-991,
                0x1.486d41578e48cp+204,
                0x1.486de8697abecp+164,
                0x1.486de8697abecp165,
                0x1.486f8835d185dp+360,
                0x1.488bab69b0db2p-804,
                0x1.488bab69b0db2p-805,
                0x1.488bab69b0db2p-806,
                0x1.488bab69b0db2p-807,
                0x1.488cf6ef7f8d6p-759,
                0x1.488db1e66973cp379,
                0x1.488db1e66973cp+380,
                0x1.488db1e66973cp381,
                0x1.4897e27bfee55p-910,
                0x1.4897e27bfee55p-911,
                0x1.489f3f37c7634p729,
                0x1.48a28d730a1a8p-398,
                0x1.48a36992e27c7p-114,
                0x1.48a926a196302p-466,
                0x1.48ace0ff47361p-456,
                0x1.48ba959b43ed6p382,
                0x1.48bf49af2f401p-901,
                0x1.48c22ca71a1bdp-688,
                0x1.48c22ca71a1bep-686,
                0x1.48c22ca71a1bep-687,
                0x1.48c22ca71a1bep-688,
                0x1.48c2bd60a1524p235,
                0x1.48c2bd60a1524p+236,
                0x1.48c2bd60a1524p237,
                0x1.48c2bd60a1524p238,
                0x1.48c2bd60a1524p239,
                0x1.48c2bd60a1524p+240,
                0x1.48c2bd60a1524p241,
                0x1.48c46b17998ffp-1011,
                0x1.48ca715f69262p-715,
                0x1.48cddb2fa9a98p-67,
                0x1.48d67e8b1e00dp-150,
                0x1.48dd1cf3afc3cp-516,
                0x1.48e40dd4978ecp-390,
                0x1.48e4865p-1044,
                0x1.48e5bd5b51583p-831,
                0x1.48e6b25bfdbccp+780,
                0x1.48e8326d02c4p-215,
                0x1.48e8e55042a03p614,
                0x1.48f205de07f6p585,
                0x1.4905472ec783dp707,
                0x1.4906db5bb891ap+628,
                0x1.490abad057752p1005,
                0x1.49105982e9c9ep-280,
                0x1.4913e3672f05p-465,
                0x1.4913e3672f05p-466,
                0x1.4919d5556eb52p-203,
                0x1.491daad0ba28p531,
                0x1.491daad0ba28p+532,
                0x1.4920a9900443cp-93,
                0x1.492e2ca475becp335,
                0x1.492p-1062,
                0x1.492p-1063,
                0x1.4936decf62077p185,
                0x1.493eed4273024p-687,
                0x1.493eed4273024p-688,
                0x1.493eed4273024p-689,
                0x1.493eed4273024p-690,
                0x1.4942ef761a255p847,
                0x1.4944b4974f4f1p-467,
                0x1.49551857ec587p-392,
                0x1.49599b25701a2p-971,
                0x1.4961a6976ac07p129,
                0x1.4965a87ecd591p-203,
                0x1.496a21e15d4b8p783,
                0x1.496e97bef9482p-208,
                0x1.4971956342ac8p282,
                0x1.497303a88b96dp-241,
                0x1.49775b628cc35p625,
                0x1.497b6df0b5b5fp883,
                0x1.4981285e98e7ap-17,
                0x1.4981285e98e7ap-18,
                0x1.4982a236bc14p-515,
                0x1.4982a236bc14p-516,
                0x1.4982a236bc14p-517,
                0x1.498822d319d2ap689,
                0x1.498bd54860434p-752,
                0x1.498bd54860434p-753,
                0x1.49938182e52afp+296,
                0x1.4996d2ee771aap-93,
                0x1.4997a298b2f2ep979,
                0x1.4997a298b2f2ep+980,
                0x1.4997a298b2f2ep981,
                0x1.499ff0fdeee9p339,
                0x1.49a9146544e22p-351,
                0x1.49ab9cf797b34p-514,
                0x1.49ae93b146p-1035,
                0x1.49b58ddd02429p-535,
                0x1.49b5aea721869p-133,
                0x1.49b8002e2a826p925,
                0x1.49c1bd132472ap441,
                0x1.49c96cd6d16ccp767,
                0x1.49d0444124dd8p323,
                0x1.49d04683dbb3cp942,
                0x1.49d5c23fdd18ap605,
                0x1.49d5c23fdd18ap606,
                0x1.49d847305b26ap+448,
                0x1.49d847305b26ap449,
                0x1.49d847305b26ap450,
                0x1.49d847305b26ap451,
                0x1.49d847305b26ap+452,
                0x1.49d847305b26ap453,
                0x1.49da7e361ce4cp-32,
                0x1.49ff4261df34ap669,
                0x1.49ff4261df34ap670,
                0x1.49ff4261df34ap671,
                0x1.4a007b2ca09b6p-414,
                0x1.4a02b8f4e1395p-949,
                0x1.4a03bd121f7fap850,
                0x1.4a04a6b78e4bep+408,
                0x1.4a04a6b78e4bep409,
                0x1.4a04a6b78e4bep410,
                0x1.4a0ebbe2324d2p551,
                0x1.4a12ead7b2f9dp-120,
                0x1.4a1b40adf93p-884,
                0x1.4a1b40adf93p-885,
                0x1.4a261e42dc16ap-85,
                0x1.4a2717a95f76bp-423,
                0x1.4a28ad132d1bcp-868,
                0x1.4a2f1ffecd15cp-884,
                0x1.4a31281548db3p975,
                0x1.4a3271a2abe3ap453,
                0x1.4a43c98572c54p-956,
                0x1.4a43c98572c54p-957,
                0x1.4a476a0856b3p247,
                0x1.4a48933c208aep-671,
                0x1.4a48933c208aep-672,
                0x1.4a48933c208aep-673,
                0x1.4a48933c208aep-674,
                0x1.4a48933c208aep-675,
                0x1.4a4bb1ecb299ap805,
                0x1.4a53b8476f3c9p-747,
                0x1.4a59295db3f0ap-197,
                0x1.4a593f89f4194p111,
                0x1.4a5b20fd8dbap-676,
                0x1.4a5b20fd8dbap-677,
                0x1.4a5b20fd8dbap-678,
                0x1.4a5b539db1d11p-839,
                0x1.4a5b539db1d11p-840,
                0x1.4a5d39c18e796p-652,
                0x1.4a5d39c18e796p-653,
                0x1.4a5d39c18e796p-654,
                0x1.4a5fdae7cfefep-227,
                0x1.4a610a05a6504p-535,
                0x1.4a6cb85d1e1c2p-316,
                0x1.4a6cb85d1e1c2p-317,
                0x1.4a749130931d6p699,
                0x1.4a749130931d6p+700,
                0x1.4a749130931d6p701,
                0x1.4a749130931d6p702,
                0x1.4a749130931d6p703,
                0x1.4a749130931d6p+704,
                0x1.4a759f0dcba5dp714,
                0x1.4a75eb58df47cp579,
                0x1.4a75eb58df47cp+580,
                0x1.4a75eb58df47cp581,
                0x1.4a75eb58df47cp582,
                0x1.4a75eb58df47cp583,
                0x1.4a7b40921fb4cp+480,
                0x1.4a7b40921fb4cp481,
                0x1.4a7f2e371714p-683,
                0x1.4a8729fc3ddb7p-399,
                0x1.4a89e54e8ffe4p946,
                0x1.4a8a67061071p+428,
                0x1.4a8a67061071p429,
                0x1.4a8a67061071p430,
                0x1.4a8a67061071p431,
                0x1.4a8a7c82a8213p938,
                0x1.4a8fa10e49cdep-958,
                0x1.4aa79f943cd91p551,
                0x1.4aab73b5febe2p-941,
                0x1.4aac666c7963ap-407,
                0x1.4aac86e827f22p-772,
                0x1.4aac86e827f22p-773,
                0x1.4aac86e827f22p-774,
                0x1.4aac86e827f22p-775,
                0x1.4aadd3f8f5bep-408,
                0x1.4aadd3f8f5bep-409,
                0x1.4ab695fa6d1cfp-647,
                0x1.4ab695fa6d1cfp-648,
                0x1.4acaad91dec38p977,
                0x1.4acdd5a3e01c8p-832,
                0x1.4acdd5a3e01c8p-833,
                0x1.4ace16f47d584p721,
                0x1.4ace16f47d584p722,
                0x1.4ace16f47d584p723,
                0x1.4adf4b7320334p86,
                0x1.4ae2f5697aebcp-925,
                0x1.4ae7e2ce50cfcp-1005,
                0x1.4ae8f2b920c38p421,
                0x1.4ae91eff68bbcp-988,
                0x1.4af06b33efe2ap-710,
                0x1.4af06b33efe2ap-711,
                0x1.4af06b33efe2ap-712,
                0x1.4af06b33efe2ap-713,
                0x1.4af06b33efe2ap-714,
                0x1.4af1d3ddb7c8ap-696,
                0x1.4af1d3ddb7c8ap-697,
                0x1.4af1d3ddb7c8ap-698,
                0x1.4af1d3ddb7c8ap-699,
                0x1.4af1d3ddb7c8ap-700,
                0x1.4af58111820dcp-766,
                0x1.4af7fd0369628p-1010,
                0x1.4aff5c0a1f572p845,
                0x1.4aff5c0a1f572p846,
                0x1.4aff5c0a1f572p847,
                0x1.4aff5c0a1f572p+848,
                0x1.4b05798085cd2p-77,
                0x1.4b05798085cd2p-78,
                0x1.4b059ccfc7d36p+656,
                0x1.4b0b502ab7ba4p-802,
                0x1.4b0c8d199526ep-281,
                0x1.4b0dded431e48p-231,
                0x1.4b1124879cdfbp142,
                0x1.4b17705029b7ap979,
                0x1.4b193ff0a9d76p691,
                0x1.4b1ebbf15659ep-534,
                0x1.4b2c9757539c8p183,
                0x1.4b2fcef8bd75bp393,
                0x1.4b378469b6732p570,
                0x1.4b378469b6732p571,
                0x1.4b378469b6732p+572,
                0x1.4b378469b6732p573,
                0x1.4b378469b6732p574,
                0x1.4b378469b6732p575,
                0x1.4b489cff67924p-525,
                0x1.4b4c11f3e3c77p-906,
                0x1.4b4c3647e832p453,
                0x1.4b4dff4b7f98ap-382,
                0x1.4b4dff4b7f98ap-383,
                0x1.4b5002578d339p401,
                0x1.4b5278eb0e795p774,
                0x1.4b53c2516484ep+364,
                0x1.4b53c2516484ep365,
                0x1.4b53c2516484ep366,
                0x1.4b53c2516484ep367,
                0x1.4b53c2516484ep+368,
                0x1.4b5475269fa2dp-634,
                0x1.4b56612b6f188p678,
                0x1.4b583bb5ff9e2p867,
                0x1.4b5ad35814588p-593,
                0x1.4b5cf544fde47p+196,
                0x1.4b5d2de36f2c3p609,
                0x1.4b5d2de36f2c3p610,
                0x1.4b5d2de36f2c3p611,
                0x1.4b5eb979d4cdap-878,
                0x1.4b5eb979d4cdap-879,
                0x1.4b5eb979d4cdap-880,
                0x1.4b60a8c16e3a4p-745,
                0x1.4b60a8c16e3a4p-746,
                0x1.4b60a8c16e3a4p-747,
                0x1.4b60a8c16e3a4p-748,
                0x1.4b60e157027bep-612,
                0x1.4b60e157027bep-613,
                0x1.4b60e157027bep-614,
                0x1.4b60e157027bep-615,
                0x1.4b60e157027bep-616,
                0x1.4b60e157027bep-617,
                0x1.4b60e157027bep-618,
                0x1.4b693f908a54ep-743,
                0x1.4b693f908a54ep-744,
                0x1.4b6c367b9fed4p-1018,
                0x1.4b6e1142a1eacp521,
                0x1.4b701501ebcc3p621,
                0x1.4b77624928574p+340,
                0x1.4b792daba9962p-490,
                0x1.4b792daba9962p-491,
                0x1.4b792daba9962p-492,
                0x1.4b792daba9962p-493,
                0x1.4b82b76064f02p-761,
                0x1.4b82b76064f02p-762,
                0x1.4b82b76064f02p-763,
                0x1.4b82b76064f02p-764,
                0x1.4b82b76064f02p-765,
                0x1.4b914b5096548p321,
                0x1.4b914b5096548p322,
                0x1.4b94697e06248p910,
                0x1.4b99a833a956ap+292,
                0x1.4b9c1011b87b8p982,
                0x1.4b9c1011b87b8p983,
                0x1.4b9d09b6d73ebp593,
                0x1.4b9d09b6d73ebp594,
                0x1.4ba29692319fcp119,
                0x1.4ba329e0a0cadp889,
                0x1.4ba6bb67a8bep-731,
                0x1.4ba6bb67a8bep-732,
                0x1.4ba6bb67a8bep-733,
                0x1.4ba7554ace45p-743,
                0x1.4ba7554ace45p-744,
                0x1.4ba7554ace45p-745,
                0x1.4ba9c6e06e85dp-382,
                0x1.4bbea44de189ap-640,
                0x1.4bc30e5e69e2bp555,
                0x1.4bcb7e109c9d3p-526,
                0x1.4bd7ed9a2bafep+392,
                0x1.4bd7ed9a2bafep393,
                0x1.4bd7ed9a2bafep394,
                0x1.4bd980b81b80ap670,
                0x1.4be143421d618p-102,
                0x1.4be4be55f679p-175,
                0x1.4be4be55f679p-176,
                0x1.4be4be55f679p-177,
                0x1.4be4be55f679p-178,
                0x1.4be4be55f679p-179,
                0x1.4be4be55f679p-180,
                0x1.4beaba36147b9p-361,
                0x1.4bf51f3410875p-205,
                0x1.4bf6142f8eefap-595,
                0x1.4bf96a66215ebp742,
                0x1.4bfc925d5c98ep606,
                0x1.4bff8841d72cep-404,
                0x1.4c05bad0e615p-882,
                0x1.4c0b3a63c1444p-683,
                0x1.4c0b3a63c1444p-684,
                0x1.4c0b3a63c1444p-685,
                0x1.4c0b3a63c1444p-686,
                0x1.4c0b3a63c1444p-687,
                0x1.4c17116f185bap-126,
                0x1.4c1bc1138223bp303,
                0x1.4c1beb48d810cp689,
                0x1.4c1cee9cd666bp89,
                0x1.4c21ec866bc7ep-485,
                0x1.4c25506806192p369,
                0x1.4c29aea67f1e6p1009,
                0x1.4c2d7606a3644p-690,
                0x1.4c2d7606a3644p-691,
                0x1.4c33abb8b49c7p+584,
                0x1.4c4135c6f7658p309,
                0x1.4c41ed9af4042p-251,
                0x1.4c4e977ba1f5cp-110,
                0x1.4c584978f6ad6p225,
                0x1.4c5fca3ecaacp-909,
                0x1.4c6d499a661p379,
                0x1.4c764cda167dap-324,
                0x1.4c764cda167dap-325,
                0x1.4c7c42546081ap859,
                0x1.4c7c42546081ap+860,
                0x1.4c88f86f92b8cp919,
                0x1.4c88f86f92b8cp+920,
                0x1.4c8e4e6634a7p-729,
                0x1.4c8e4e6634a7p-730,
                0x1.4c92ae27cfdbdp-144,
                0x1.4c973e07a07c7p975,
                0x1.4c98fce16152ep-199,
                0x1.4c98fce16152ep-200,
                0x1.4c98fce16152ep-201,
                0x1.4ca08ee3577ep-433,
                0x1.4ca08ee3577ep-434,
                0x1.4ca732617ed8p375,
                0x1.4caa77f435b4cp746,
                0x1.4cad43a54fb86p710,
                0x1.4cad43a54fb86p711,
                0x1.4cb0e6f49d58ep710,
                0x1.4cb869ba971b6p-423,
                0x1.4cb869ba971b6p-424,
                0x1.4cbb48c0a9e2ap439,
                0x1.4cbb48c0a9e2ap+440,
                0x1.4cbfd492e30dp202,
                0x1.4cce5004f40fcp-933,
                0x1.4cce5004f40fcp-934,
                0x1.4ccef25667e6dp+160,
                0x1.4cd3a1cefb3b4p542,
                0x1.4cd785501f416p850,
                0x1.4cd785501f416p851,
                0x1.4cdd7550b7644p-614,
                0x1.4ce6fa492623p-243,
                0x1.4ce6fa492623p-244,
                0x1.4ce6fa492623p-245,
                0x1.4ceaebb809b8ap-199,
                0x1.4ceaebb809b8ap-200,
                0x1.4ceaebb809b8ap-201,
                0x1.4cf41fe9977eep637,
                0x1.4cffe4e7708cp+860,
                0x1.4cffe4e7708cp861,
                0x1.4cffe4e7708cp862,
                0x1.4cp-1068,
                0x1.4d1389b5d6b5ep235,
                0x1.4d1389b5d6b5ep+236,
                0x1.4d1389b5d6b5ep237,
                0x1.4d1389b5d6b5ep238,
                0x1.4d1389b5d6b5ep239,
                0x1.4d14ef280617ap283,
                0x1.4d1cb824aa1fep474,
                0x1.4d1e71155e63cp+148,
                0x1.4d1eefcd53e8fp657,
                0x1.4d1eefcd53e8fp658,
                0x1.4d1f62f6c5c7bp178,
                0x1.4d1fc2bb65286p-662,
                0x1.4d1fc2bb65286p-663,
                0x1.4d1fc2bb65286p-664,
                0x1.4d22d30e1f458p-531,
                0x1.4d22d30e1f458p-532,
                0x1.4d2444854bda5p-262,
                0x1.4d25c245f3cdfp+900,
                0x1.4d262d496953fp487,
                0x1.4d262d496953fp+488,
                0x1.4d2de2fa2cp-1036,
                0x1.4d3794ce2fc26p111,
                0x1.4d397d1fb192ep423,
                0x1.4d3b0cb6b4722p563,
                0x1.4d3b0cb6b4722p+564,
                0x1.4d3b58883de9fp-575,
                0x1.4d3cda8eb21e7p-203,
                0x1.4d4a1a3157dc8p-27,
                0x1.4d5045ad4ca13p882,
                0x1.4d52e0aa57922p397,
                0x1.4d52e0aa57922p398,
                0x1.4d52e0aa57922p399,
                0x1.4d52e0aa57922p+400,
                0x1.4d632dfa48b41p471,
                0x1.4d64d4079150cp771,
                0x1.4d64d4079150cp+772,
                0x1.4d64d4079150cp773,
                0x1.4d64d4079150cp774,
                0x1.4d64d4079150cp775,
                0x1.4d64d4079150cp+776,
                0x1.4d64d4079150cp777,
                0x1.4d64d4079150cp778,
                0x1.4d6695b193bf8p-791,
                0x1.4d6695b193bf8p-792,
                0x1.4d70e45c2c2fap-174,
                0x1.4d7c14bc2c2b2p635,
                0x1.4d7c6fb4783a9p-431,
                0x1.4d8296f693526p159,
                0x1.4d8468cd177d6p958,
                0x1.4d8468cd177d6p959,
                0x1.4d8468cd177d6p+960,
                0x1.4d8468cd177d6p961,
                0x1.4d8bda76bdcecp705,
                0x1.4d8ea50f1b0dfp518,
                0x1.4d902f1cc27bbp-164,
                0x1.4d933a6584477p+608,
                0x1.4d964820c9fbcp-464,
                0x1.4d971170ed056p-24,
                0x1.4d9b79p-1050,
                0x1.4da9dcda9b9ccp-74,
                0x1.4dab846e19e24p-494,
                0x1.4dab846e19e24p-495,
                0x1.4dab846e19e24p-496,
                0x1.4dab846e19e24p-497,
                0x1.4dab846e19e24p-498,
                0x1.4db2ddcb341f2p689,
                0x1.4db5cc7c0f2fp-970,
                0x1.4db7757d1ba78p+280,
                0x1.4db851d2c6d24p-191,
                0x1.4dba14465b291p245,
                0x1.4dbf7b3f71cb7p-306,
                0x1.4dc5ee85b9b6ep-234,
                0x1.4dc625bb93dbap-589,
                0x1.4dc66fc14d2d4p-1010,
                0x1.4dc8076e09bdcp-865,
                0x1.4dcbf8bc2aad4p639,
                0x1.4dcbf8bc2aad4p+640,
                0x1.4dcc75e3376ccp+948,
                0x1.4deb4d307aef4p+588,
                0x1.4df19de36599ep-878,
                0x1.4df2effb30cp-884,
                0x1.4df2effb30cp-885,
                0x1.4dfa324517621p-876,
                0x1.4dfa324517621p-877,
                0x1.4dfb2fd9ec5f2p-883,
                0x1.4dfe790d6fb3ep633,
                0x1.4dfe790d6fb3ep634,
                0x1.4dfe790d6fb3ep635,
                0x1.4dfe790d6fb3ep+636,
                0x1.4dfe790d6fb3ep637,
                0x1.4e12733b19661p+888,
                0x1.4e1878814c9cep179,
                0x1.4e1878814c9cep+180,
                0x1.4e1878814c9cep181,
                0x1.4e1e38088753p601,
                0x1.4e2106f7576p-379,
                0x1.4e2106f7576p-380,
                0x1.4e2270705d399p-748,
                0x1.4e2270705d399p-749,
                0x1.4e25dd3747e96p-1005,
                0x1.4e25dd3747e96p-1006,
                0x1.4e25dd3747e96p-1007,
                0x1.4e25dd3747e96p-1008,
                0x1.4e25dd3747e96p-1009,
                0x1.4e273e96886a8p295,
                0x1.4e2f9b1fcbd3cp-365,
                0x1.4e38e449ea544p-909,
                0x1.4e38e449ea544p-910,
                0x1.4e3ada9563d2ep-405,
                0x1.4e3ada9563d2ep-406,
                0x1.4e3dd1009a137p-982,
                0x1.4e48ae566529cp-862,
                0x1.4e48ae566529cp-863,
                0x1.4e4a2c9167fc9p422,
                0x1.4e4da79a6eb88p-218,
                0x1.4e4e41a3ffafcp-839,
                0x1.4e61f991c4edp-412,
                0x1.4e61f991c4edp-413,
                0x1.4e61f991c4edp-414,
                0x1.4e61f991c4edp-415,
                0x1.4e61f991c4edp-416,
                0x1.4e61f991c4edp-417,
                0x1.4e61f991c4edp-418,
                0x1.4e6523ef8cb78p-1011,
                0x1.4e6ab27c1c40cp-493,
                0x1.4e6b045a2123p-112,
                0x1.4e718d7d7625ap+664,
                0x1.4e7c99338915ep-865,
                0x1.4e7c99338915ep-866,
                0x1.4e7c99338915ep-867,
                0x1.4e7c99338915ep-868,
                0x1.4e7c99338915ep-869,
                0x1.4e7f335963bfep606,
                0x1.4e8fe9e43a8a7p647,
                0x1.4e99158432d4p854,
                0x1.4e9a85889ede4p-622,
                0x1.4e9a85889ede4p-623,
                0x1.4e9c8bb5820ccp-17,
                0x1.4e9c8bb5820ccp-18,
                0x1.4ea0a613d62eap+452,
                0x1.4ea517dafabep-893,
                0x1.4ea5452428b7p-629,
                0x1.4ea5452428b7p-630,
                0x1.4ea5b31caf274p-410,
                0x1.4ea83577a78d7p-509,
                0x1.4ea8e5ee320b4p-95,
                0x1.4ebbbac4158bp-470,
                0x1.4ebbbac4158bp-471,
                0x1.4ebbbac4158bp-472,
                0x1.4ebcaf75815a4p174,
                0x1.4ec8fd70420c8p486,
                0x1.4eca01375cb7p191,
                0x1.4ecd5bd06aa65p+172,
                0x1.4ed7b6c005902p-42,
                0x1.4ed7b6c005902p-43,
                0x1.4ed809f9cd1d8p-200,
                0x1.4ed8b04671da4p-987,
                0x1.4ed8b04671da5p-987,
                0x1.4ede3a81b4089p+604,
                0x1.4ee33d1104d5p-600,
                0x1.4ee33d1104d5p-601,
                0x1.4ee5d56b32958p-919,
                0x1.4ee5d56b32958p-920,
                0x1.4ee5d56b32958p-921,
                0x1.4ee5d56b32958p-922,
                0x1.4ee5d56b32958p-923,
                0x1.4ee5d56b32958p-924,
                0x1.4ee5d56b32958p-925,
                0x1.4ee5d56b32958p-926,
                0x1.4ee77bf5d8ee8p+348,
                0x1.4eef73d15c44ep282,
                0x1.4ef06d21780c2p-549,
                0x1.4ef48b1b6281cp+528,
                0x1.4ef48b1b6281cp529,
                0x1.4ef79a5e7a0b5p149,
                0x1.4efe461248f32p-812,
                0x1.4efe461248f32p-813,
                0x1.4f0578097ac77p457,
                0x1.4f102324aefe1p-186,
                0x1.4f122374ebac3p+352,
                0x1.4f14348a4c5dcp-297,
                0x1.4f14348a4c5dcp-298,
                0x1.4f14348a4c5dcp-299,
                0x1.4f14348a4c5dcp-300,
                0x1.4f14348a4c5dcp-301,
                0x1.4f14348a4c5dcp-302,
                0x1.4f14348a4c5dcp-303,
                0x1.4f14348a4c5dcp-304,
                0x1.4f153f3cd2602p-105,
                0x1.4f1c0e250cebep-108,
                0x1.4f21b08894ea4p-440,
                0x1.4f26a73752272p-803,
                0x1.4f2b15f6a8fb8p-756,
                0x1.4f2dc65d3a4a6p634,
                0x1.4f31f8832dd2ap-501,
                0x1.4f31f8832dd2ap-502,
                0x1.4f3d34a0e780ep-25,
                0x1.4f3d34a0e780ep-26,
                0x1.4f3d34a0e780ep-27,
                0x1.4f427d7964454p557,
                0x1.4f53b4ed9f139p+612,
                0x1.4f53ba712cf63p-835,
                0x1.4f54d3d869396p-563,
                0x1.4f54d3d869396p-564,
                0x1.4f553489af5c5p373,
                0x1.4f5adc031e664p417,
                0x1.4f6493f34a0bdp-300,
                0x1.4f6493f34a0bdp-301,
                0x1.4f6493f34a0bdp-302,
                0x1.4f67755c9bdfdp-255,
                0x1.4f7a098588ab5p191,
                0x1.4f7d27935da62p-127,
                0x1.4f8b588e368fp-14,
                0x1.4f8b588e368fp-15,
                0x1.4f8b588e368fp-16,
                0x1.4f8b588e368fp-17,
                0x1.4f8b588e368fp-18,
                0x1.4f9299346ff9bp-678,
                0x1.4f9522b97b0c5p317,
                0x1.4f9522b97b0c5p318,
                0x1.4f9afd8b1c52ep+688,
                0x1.4f9afd8b1c52ep689,
                0x1.4f9afd8b1c52ep690,
                0x1.4f9afd8b1c52ep691,
                0x1.4f9cb6359a80ap-816,
                0x1.4f9cb6359a80ap-817,
                0x1.4f9f5612c6d21p-286,
                0x1.4faa98c7eb787p-593,
                0x1.4faa98c7eb787p-594,
                0x1.4faba79ea92ecp-119,
                0x1.4faba79ea92ecp-120,
                0x1.4faba79ea92ecp-121,
                0x1.4faba79ea92ecp-122,
                0x1.4faba79ea92ecp-123,
                0x1.4fb146a42ed2ep-392,
                0x1.4fb766b1520ecp493,
                0x1.4fc16f5cad938p+428,
                0x1.4fc16f5cad938p429,
                0x1.4fc16f5cad938p430,
                0x1.4fc16f5cad938p431,
                0x1.4fc16f5cad938p+432,
                0x1.4fd606aa529e2p766,
                0x1.4fd606aa529e2p767,
                0x1.4fdabe3f83328p466,
                0x1.4fde580e1dab7p-891,
                0x1.4fe4d06de5056p+468,
                0x1.4fe4d06de5057p+468,
                0x1.4feba2ab109eep-880,
                0x1.4feba2ab109eep-881,
                0x1.4feba2ab109eep-882,
                0x1.4p-1068,
                0x1.4p-1070,
                0x1.4p-1072,
                0x1.4p3,
                0x1.50059a4e30b61p-425,
                0x1.5012268287d34p-644,
                0x1.5012268287d34p-645,
                0x1.5012268287d34p-646,
                0x1.5012268287d34p-647,
                0x1.501369741313cp-171,
                0x1.501369741313cp-172,
                0x1.501c5ad68d85dp-592,
                0x1.50268403fe71dp-31,
                0x1.50295bb19676cp1007,
                0x1.50295bb19676cp+1008,
                0x1.502db46c4ced5p-599,
                0x1.503289990fb52p445,
                0x1.503289990fb52p446,
                0x1.5035503d21e1p-233,
                0x1.503ad892e700ep491,
                0x1.503e602893dd2p+952,
                0x1.503e602893dd2p953,
                0x1.503e602893dd2p954,
                0x1.503e602893dd2p955,
                0x1.5043b64e56c72p546,
                0x1.5043b64e56c72p547,
                0x1.5043b64e56c72p+548,
                0x1.5043b64e56c72p549,
                0x1.5043b64e56c72p550,
                0x1.5043b64e56c72p551,
                0x1.50447b7544094p477,
                0x1.50498f385409ap-768,
                0x1.50498f385409ap-769,
                0x1.50545f835aa1ep+860,
                0x1.50545f835aa1ep861,
                0x1.505aa4b8c33d6p-30,
                0x1.505c5dc23adf4p-546,
                0x1.505e4332cea03p-408,
                0x1.505e4332cea03p-409,
                0x1.505e8844c2582p-143,
                0x1.505e8b2fa8dd3p262,
                0x1.50649bf5c9074p993,
                0x1.50649bf5c9074p994,
                0x1.506a7522f8a8p-451,
                0x1.506a7522f8a8p-452,
                0x1.506d4c4165702p-596,
                0x1.506e71b630a42p+412,
                0x1.506e71b630a42p413,
                0x1.506e71b630a42p414,
                0x1.506e988847746p734,
                0x1.506f34a79facp-524,
                0x1.5073e1116f464p-187,
                0x1.507a05f7aba21p-439,
                0x1.5089ca72f3ab6p359,
                0x1.508a94c8c9fbcp-321,
                0x1.508e70cfccdfap-120,
                0x1.508e70cfccdfap-121,
                0x1.5098c29f575c8p-366,
                0x1.509d8be905c46p938,
                0x1.509fa4d6f116ep962,
                0x1.50a246ecd44f3p-63,
                0x1.50a246ecd44f3p-64,
                0x1.50a6110d6a9b8p-698,
                0x1.50a97353d5bbp638,
                0x1.50a97353d5bbp639,
                0x1.50ae3849bc406p649,
                0x1.50afb61581ef2p-544,
                0x1.50aff653ffe8ap566,
                0x1.50aff653ffe8ap567,
                0x1.50aff653ffe8ap+568,
                0x1.50aff653ffe8ap569,
                0x1.50aff653ffe8ap570,
                0x1.50aff653ffe8ap571,
                0x1.50b27e36539bcp-728,
                0x1.50b27e36539bcp-729,
                0x1.50b27e36539bcp-730,
                0x1.50b3a2e0aff14p-30,
                0x1.50b3a2e0aff14p-31,
                0x1.50c00f69b6c45p-497,
                0x1.50d5387118be7p-732,
                0x1.50d5dda9faa77p433,
                0x1.50e843df4434ep-897,
                0x1.50ffd44f4a73dp-213,
                0x1.5100fbd31524cp-799,
                0x1.51017ec246bbap795,
                0x1.51017ec246bbap+796,
                0x1.51017ec246bbap797,
                0x1.51017ec246bbap798,
                0x1.51017ec246bbap799,
                0x1.51026321ff408p-904,
                0x1.510cce5e9fd4bp+252,
                0x1.51108055c0976p-276,
                0x1.511125b6444cbp+340,
                0x1.5111dbced8fe2p+848,
                0x1.511267ce37d12p203,
                0x1.511b399dd08c1p-980,
                0x1.512337c5ed58p-142,
                0x1.512337c5ed58p-143,
                0x1.51233c2076fc9p179,
                0x1.51248e4b70cbep482,
                0x1.51248e4b70cbep483,
                0x1.51248e4b70cbep+484,
                0x1.51248e4b70cbep485,
                0x1.51248e4b70cbep486,
                0x1.512d5273e62e8p+996,
                0x1.512d5273e62e8p997,
                0x1.512d5273e62e8p998,
                0x1.5132524961072p+188,
                0x1.513f4a9a58c0ep-1018,
                0x1.5141bc5610e53p489,
                0x1.5149ba58329c2p-919,
                0x1.514e737e8493cp-783,
                0x1.514ed3107b7c2p599,
                0x1.5159af8044462p+272,
                0x1.5159db3ca011ep493,
                0x1.51647836782e2p+752,
                0x1.5165f7c1536c7p-461,
                0x1.5168e9c45d19fp+172,
                0x1.516f123f33412p-756,
                0x1.517e79e35b5aep-513,
                0x1.5180e31b985bp-74,
                0x1.51858db3257a3p-623,
                0x1.518a725d66e5cp362,
                0x1.5193e2a9c2ebp-955,
                0x1.5198918d7da5p766,
                0x1.51a4678d343bep-828,
                0x1.51a56a21ea85bp-310,
                0x1.51b3a2a6b9c76p+756,
                0x1.51b3a2a6b9c76p757,
                0x1.51b3a2a6b9c76p758,
                0x1.51b3a2a6b9c76p759,
                0x1.51cef7ae47ddep293,
                0x1.51d0f44c0239cp275,
                0x1.51d69807b8dcap-841,
                0x1.51d8d1c8ec436p-335,
                0x1.51de5662139ep785,
                0x1.51efb873af2bap197,
                0x1.51efb873af2bap198,
                0x1.51efb873af2bap199,
                0x1.51efb873af2bap+200,
                0x1.51f08c2708202p-584,
                0x1.51f08c2708202p-585,
                0x1.51f4c0a2a7bf8p526,
                0x1.51f508b287ae7p314,
                0x1.51f508b287ae7p315,
                0x1.51f508b287ae7p+316,
                0x1.51f742d8acdfp-566,
                0x1.51f742d8acdfp-567,
                0x1.51p-1065,
                0x1.51p-1066,
                0x1.52033879f9499p499,
                0x1.5205c35bb34c8p-858,
                0x1.5205c35bb34c8p-859,
                0x1.52081afdbb31ap806,
                0x1.52081afdbb31ap807,
                0x1.5211081722e44p-731,
                0x1.5211081722e44p-732,
                0x1.5211081722e44p-733,
                0x1.5211081722e44p-734,
                0x1.5216ep-1055,
                0x1.521bc6a6b555cp-894,
                0x1.522268e4c0d4p873,
                0x1.522268e4c0d4p874,
                0x1.5225a3f8d459p-556,
                0x1.5225a3f8d459p-557,
                0x1.5225a3f8d459p-558,
                0x1.5225a3f8d459p-559,
                0x1.5225a3f8d459p-560,
                0x1.522dbc74057acp-373,
                0x1.52302ae21c699p787,
                0x1.52302ae21c699p+788,
                0x1.523b919fb6bd8p303,
                0x1.523b919fb6bd8p+304,
                0x1.523b919fb6bd8p305,
                0x1.523b919fb6bd8p306,
                0x1.523d4372f52abp-71,
                0x1.5246e1209cb04p149,
                0x1.52518e8c64394p-1024,
                0x1.52543a43d3664p-453,
                0x1.5258e1cd039ap698,
                0x1.52697c76159ep-853,
                0x1.526be9c22eb17p+628,
                0x1.526be9c22eb17p629,
                0x1.526be9c22eb17p630,
                0x1.526f061ca9054p901,
                0x1.526f061ca9054p902,
                0x1.526f061ca9054p903,
                0x1.526f061ca9054p+904,
                0x1.526f061ca9054p905,
                0x1.526f061ca9054p906,
                0x1.526f061ca9054p907,
                0x1.5271ae1cf732ap-912,
                0x1.5271ae1cf732ap-913,
                0x1.5271ae1cf732ap-914,
                0x1.5271ae1cf732ap-915,
                0x1.5275ed8d8f36cp-408,
                0x1.5275ed8d8f36cp-409,
                0x1.52762b6b94796p525,
                0x1.5279547d99bbap+928,
                0x1.52797a5929fdcp+920,
                0x1.527a4ff6f0e7p-27,
                0x1.528350722ccdcp953,
                0x1.528350722ccdcp954,
                0x1.528350722ccdcp955,
                0x1.528350722ccdcp+956,
                0x1.52847dad80453p-270,
                0x1.52847dad80453p-271,
                0x1.52847dad80453p-272,
                0x1.5295d01fbdd7p-177,
                0x1.529d924068908p-414,
                0x1.529e74df1e7e4p715,
                0x1.52aad91366adep-775,
                0x1.52b09466e4f48p-168,
                0x1.52b457057be32p977,
                0x1.52bc95023ad33p610,
                0x1.52bf915ed63eep-273,
                0x1.52c81d5917c6p+324,
                0x1.52c8370c85754p-60,
                0x1.52ca8b08f29a8p-396,
                0x1.52ca8b08f29a8p-397,
                0x1.52cafc7065989p-134,
                0x1.52cc86af3df65p649,
                0x1.52cc86af3df65p650,
                0x1.52d02c7e14af6p+100,
                0x1.52d02c7e14af6p101,
                0x1.52d02c7e14af6p102,
                0x1.52d02c7e14af6p103,
                0x1.52d02c7e14af6p+104,
                0x1.52d02c7e14af6p105,
                0x1.52d02c7e14af6p106,
                0x1.52d02c7e14af6p107,
                0x1.52d02c7e14af6p+108,
                0x1.52d02c7e14af6p109,
                0x1.52d02c7e14af6p110,
                0x1.52d02c7e14af6p111,
                0x1.52d02c7e14af6p+112,
                0x1.52d02c7e14af6p113,
                0x1.52d02c7e14af6p114,
                0x1.52d02c7e14af6p115,
                0x1.52d02c7e14af6p+116,
                0x1.52d02c7e14af6p117,
                0x1.52d02c7e14af6p118,
                0x1.52d02c7e14af6p119,
                0x1.52d02c7e14af6p+120,
                0x1.52d02c7e14af6p121,
                0x1.52d02c7e14af6p122,
                0x1.52d02c7e14af6p123,
                0x1.52d02c7e14af6p+124,
                0x1.52d02c7e14af6p125,
                0x1.52d02c7e14af6p126,
                0x1.52d02c7e14af6p127,
                0x1.52d02c7e14af6p+128,
                0x1.52d02c7e14af6p129,
                0x1.52d02c7e14af6p130,
                0x1.52d02c7e14af6p131,
                0x1.52d02c7e14af6p+132,
                0x1.52d02c7e14af6p+76,
                0x1.52d02c7e14af6p77,
                0x1.52d02c7e14af6p78,
                0x1.52d02c7e14af6p79,
                0x1.52d02c7e14af6p+80,
                0x1.52d02c7e14af6p81,
                0x1.52d02c7e14af6p82,
                0x1.52d02c7e14af6p83,
                0x1.52d02c7e14af6p+84,
                0x1.52d02c7e14af6p85,
                0x1.52d02c7e14af6p86,
                0x1.52d02c7e14af6p87,
                0x1.52d02c7e14af6p+88,
                0x1.52d02c7e14af6p89,
                0x1.52d02c7e14af6p90,
                0x1.52d02c7e14af6p91,
                0x1.52d02c7e14af6p+92,
                0x1.52d02c7e14af6p93,
                0x1.52d02c7e14af6p94,
                0x1.52d02c7e14af6p95,
                0x1.52d02c7e14af6p+96,
                0x1.52d02c7e14af6p97,
                0x1.52d02c7e14af6p98,
                0x1.52d02c7e14af6p99,
                0x1.52d2b933p-1042,
                0x1.52d399df2184ap-903,
                0x1.52da0f1396493p410,
                0x1.52da0f1396493p411,
                0x1.52e1c9e04ee06p-394,
                0x1.52e1c9e04ee06p-395,
                0x1.52e1c9e04ee06p-396,
                0x1.52e1c9e04ee06p-397,
                0x1.52e1c9e04ee06p-398,
                0x1.52e51d7a0eca4p614,
                0x1.52e548405c99fp-26,
                0x1.52ed5643ed34cp610,
                0x1.52ed5643ed34cp611,
                0x1.52efda0de45c6p-303,
                0x1.52efda0de45c6p-304,
                0x1.52fe60b4aa49ap-639,
                0x1.5305d9f3b6024p-111,
                0x1.5305d9f3b6024p-112,
                0x1.5309bedf1700ap929,
                0x1.530e2f6e5e87p-292,
                0x1.53183f609ca0dp-505,
                0x1.531f287544534p-134,
                0x1.5320f63778cdep942,
                0x1.5320f63778cdep943,
                0x1.53279a41f522ap733,
                0x1.5328364054e66p997,
                0x1.5328364054e66p998,
                0x1.532a837eae8a5p561,
                0x1.532a837eae8a6p561,
                0x1.532a837eae8a6p562,
                0x1.53382709422bap-784,
                0x1.53385f9da90abp-729,
                0x1.533b95661fc76p-795,
                0x1.533b95661fc76p-796,
                0x1.533bb16be2334p+204,
                0x1.533bb16be2334p205,
                0x1.533d22b01b854p507,
                0x1.533d22b01b854p+508,
                0x1.5343d7cb33296p-647,
                0x1.53477566c8c25p-706,
                0x1.534aeb0cbdb32p-423,
                0x1.534d35e0858f4p-781,
                0x1.5350d75eb4facp299,
                0x1.5350d75eb4facp+300,
                0x1.5356937fa0f75p525,
                0x1.535f3110e5d3dp291,
                0x1.536859becc365p-727,
                0x1.536bbc7875eeep-979,
                0x1.537780a6c6ea6p606,
                0x1.5377847b48a4p-775,
                0x1.5377847b48a4p-776,
                0x1.538b443508cebp342,
                0x1.538e07a2a62bap+360,
                0x1.539616f2362cap+640,
                0x1.539616f2362cap641,
                0x1.539684e774b48p247,
                0x1.539eee07d9cfcp-205,
                0x1.539eee07d9cfcp-206,
                0x1.53a07f6040d22p-29,
                0x1.53a9985dbde6cp-50,
                0x1.53a9985dbde6cp-51,
                0x1.53a9985dbde6cp-52,
                0x1.53a9985dbde6cp-53,
                0x1.53a9985dbde6cp-54,
                0x1.53a9985dbde6cp-55,
                0x1.53a9985dbde6cp-56,
                0x1.53a9985dbde6cp-57,
                0x1.53a9985dbde6cp-58,
                0x1.53a9985dbde6cp-59,
                0x1.53a9985dbde6cp-60,
                0x1.53aa59be9de08p201,
                0x1.53aa59be9de08p202,
                0x1.53abb82852ceep+660,
                0x1.53abb82852ceep661,
                0x1.53b153369d376p-669,
                0x1.53b98cb253ba8p-732,
                0x1.53b98cb253ba8p-733,
                0x1.53b98cb253ba8p-734,
                0x1.53b98cb253ba8p-735,
                0x1.53b98cb253ba8p-736,
                0x1.53b98cb253ba8p-737,
                0x1.53b98cb253ba8p-738,
                0x1.53b98cb253ba8p-739,
                0x1.53ba9a3eba3fcp-370,
                0x1.53baf8edecfbfp-94,
                0x1.53c30a5b7221dp958,
                0x1.53c3ac250aa1bp-221,
                0x1.53ceb74871f4ap-512,
                0x1.53cec130dfef3p-867,
                0x1.53db2fea1ea31p-605,
                0x1.53db2fea1ea31p-606,
                0x1.53ecc1523ceep-855,
                0x1.53eda614071a4p-605,
                0x1.53f5faa894b1bp-465,
                0x1.53fbbe12921a4p-948,
                0x1.53fbee06104e2p-960,
                0x1.53ff5e185b1p921,
                0x1.53ff5e185b1p922,
                0x1.540178c23bedap-302,
                0x1.5402115d5ac18p-815,
                0x1.5402115d5ac18p-816,
                0x1.5402acb6dc009p+372,
                0x1.54036820dad87p-692,
                0x1.5406bf6dacf4p+996,
                0x1.5407bd2d1f57ap+456,
                0x1.5407bd2d1f57ap457,
                0x1.54084147b9138p+340,
                0x1.54084147b9138p341,
                0x1.54084147b9138p342,
                0x1.54084147b9138p343,
                0x1.540c942f467a4p-315,
                0x1.5411ae784f28p425,
                0x1.541ecdfd48694p-351,
                0x1.541ecdfd48694p-352,
                0x1.541ecdfd48694p-353,
                0x1.541ecdfd48694p-354,
                0x1.541ecdfd48694p-355,
                0x1.541ecdfd48694p-356,
                0x1.541ecdfd48694p-357,
                0x1.541ecdfd48694p-358,
                0x1.541ecdfd48694p-359,
                0x1.541ecdfd48694p-360,
                0x1.542c78ae70ebp-610,
                0x1.542c78ae70ebp-611,
                0x1.542c78ae70ebp-612,
                0x1.542c78ae70ebp-613,
                0x1.542c78ae70ebp-614,
                0x1.542e68168c8fcp611,
                0x1.542e68168c8fcp+612,
                0x1.54315cd3a98b4p693,
                0x1.543619dddb7eap665,
                0x1.54377fcdd8c26p+812,
                0x1.543821de4a0e3p758,
                0x1.5439cccbc0276p-429,
                0x1.543ad68442bdp938,
                0x1.543ad68442bdp939,
                0x1.544214922fae8p-172,
                0x1.544214922fae8p-173,
                0x1.544214922fae8p-174,
                0x1.5442a5d58f916p-185,
                0x1.5446ab8b1b9eep-468,
                0x1.5446ab8b1b9eep-469,
                0x1.5446ab8b1b9eep-470,
                0x1.54484932d2e72p-119,
                0x1.54484932d2e72p-120,
                0x1.544e6daee2a18p674,
                0x1.544e6daee2a18p675,
                0x1.544e6daee2a18p+676,
                0x1.544e6daee2a18p677,
                0x1.544e6daee2a18p678,
                0x1.544e6daee2a18p679,
                0x1.544e6daee2a18p+680,
                0x1.544e6daee2a18p681,
                0x1.544e6daee2a18p682,
                0x1.544e6daee2a18p683,
                0x1.544e6daee2a18p+684,
                0x1.5457a514e6068p943,
                0x1.5457a514e6068p+944,
                0x1.5457a514e6068p945,
                0x1.5457a514e6068p946,
                0x1.5457a514e6068p947,
                0x1.5459e5a08ep-919,
                0x1.545aa8036e994p+492,
                0x1.545aa8036e994p493,
                0x1.545d44e157f78p+980,
                0x1.5468e6111c48ap130,
                0x1.5468e6111c48ap131,
                0x1.546c6714d7c6ep+572,
                0x1.546c6714d7c6ep573,
                0x1.546f6d5a28944p153,
                0x1.547166a3a2b1p+792,
                0x1.547166a3a2b1p793,
                0x1.547166a3a2b1p794,
                0x1.547166a3a2b1p795,
                0x1.547166a3a2b1p+796,
                0x1.547166a3a2b1p797,
                0x1.547166a3a2b1p798,
                0x1.54758f30cdfaep-299,
                0x1.547a36cd4e80cp158,
                0x1.547bdd6eec133p526,
                0x1.5487df5aae14ep-691,
                0x1.548a1e1826becp-167,
                0x1.548e1127e69dap-922,
                0x1.5497ffddd2e09p+180,
                0x1.54a0f56a2206ap737,
                0x1.54a3047c694fep365,
                0x1.54a44001d7b64p799,
                0x1.54a92p-1055,
                0x1.54ab43366c046p-89,
                0x1.54ab9b113b1ap-381,
                0x1.54acf4b0c1e44p+700,
                0x1.54b0b4aad2a8bp187,
                0x1.54b1b48a8f453p-598,
                0x1.54bbe6dba91ap-730,
                0x1.54bbe6dba91ap-731,
                0x1.54be965445928p-100,
                0x1.54be965445928p-101,
                0x1.54be965445928p-94,
                0x1.54be965445928p-95,
                0x1.54be965445928p-96,
                0x1.54be965445928p-97,
                0x1.54be965445928p-98,
                0x1.54be965445928p-99,
                0x1.54c4366193a9cp745,
                0x1.54c8cfb2187bep+464,
                0x1.54c8cfb2187bep465,
                0x1.54cb1273aa0cp-944,
                0x1.54cb84b82ce18p-1014,
                0x1.54d1c9b8af674p327,
                0x1.54d6f4b8cc0c4p-854,
                0x1.54d93859a3447p-775,
                0x1.54de34319676cp-232,
                0x1.54ef633cb317p-835,
                0x1.54ef633cb317p-836,
                0x1.54ef633cb317p-837,
                0x1.54ef633cb317p-838,
                0x1.54ef633cb317p-839,
                0x1.54ef633cb317p-840,
                0x1.54f0e30343c54p462,
                0x1.54f604775c2b8p774,
                0x1.54f604775c2b8p775,
                0x1.54fafb7392004p-872,
                0x1.54fdd7f73bf3cp850,
                0x1.54fec3296fc1ep-108,
                0x1.54fec3296fc1ep-109,
                0x1.54fp-1062,
                0x1.54p-1068,
                0x1.5504dab04eb5cp802,
                0x1.550a0dedcdd4ep+628,
                0x1.550a0dedcdd4ep629,
                0x1.550a0dedcdd4ep630,
                0x1.550a0dedcdd4ep631,
                0x1.550a0dedcdd4ep+632,
                0x1.550a0dedcdd4ep633,
                0x1.551603777f799p-951,
                0x1.551603777f799p-952,
                0x1.5516eda00942ap-27,
                0x1.5516eda00942ap-28,
                0x1.551767ef8a9a4p-818,
                0x1.551767ef8a9a4p-819,
                0x1.551767ef8a9a4p-820,
                0x1.551767ef8a9a4p-821,
                0x1.551767ef8a9a4p-822,
                0x1.5519ac5142aaap-965,
                0x1.5519ac5142aaap-966,
                0x1.5519ac5142aaap-967,
                0x1.5519ac5142aaap-968,
                0x1.5519ac5142aaap-969,
                0x1.5519ac5142aaap-970,
                0x1.5519ac5142aaap-971,
                0x1.5519ac5142aaap-972,
                0x1.5519ac5142aaap-973,
                0x1.5519ac5142aaap-974,
                0x1.5519ac5142aaap-975,
                0x1.5519ac5142aaap-976,
                0x1.551cd88d06c8ep+784,
                0x1.551cd88d06c8ep785,
                0x1.55223c34b3042p-25,
                0x1.55236a9fc359ap453,
                0x1.55265219cfaf6p-562,
                0x1.55265219cfaf6p-563,
                0x1.55323f6a443b8p-766,
                0x1.553f985920abbp619,
                0x1.5542f2b09e8afp917,
                0x1.5542f2b09e8afp918,
                0x1.554439e9298fcp+820,
                0x1.5546d747e3f94p-860,
                0x1.5547ef788978ep+852,
                0x1.554dbf59b6d54p-169,
                0x1.554dbf59b6d54p-170,
                0x1.554dbf59b6d54p-171,
                0x1.555ae28ababeep-190,
                0x1.555cfa91d56cap731,
                0x1.5564fb098c956p+200,
                0x1.5564fb098c956p201,
                0x1.5564fb098c956p202,
                0x1.5564fb098c956p203,
                0x1.5566ffafb1ebp-801,
                0x1.5570f59bd178cp-460,
                0x1.5570f59bd178cp-461,
                0x1.5570f59bd178cp-462,
                0x1.5570f59bd178cp-463,
                0x1.5570f59bd178cp-464,
                0x1.5570f59bd178cp-465,
                0x1.5570f59bd178cp-466,
                0x1.5570f59bd178cp-467,
                0x1.5570f59bd178cp-468,
                0x1.557ef1a5e11e4p-553,
                0x1.55868a28f61ap1006,
                0x1.55868a28f61ap1007,
                0x1.5588e47e7ad7ep-998,
                0x1.558eed83dd832p-861,
                0x1.558eed83dd832p-862,
                0x1.558fcb48ceb3p-1016,
                0x1.559627e2bf553p147,
                0x1.559a20aaee7d3p+220,
                0x1.55a527b1a8eecp-710,
                0x1.55a8d4585ep-1032,
                0x1.55b082a0f89cp-1027,
                0x1.55b526337a1b4p+956,
                0x1.55bca012f5ebcp-316,
                0x1.55bcf72fd10f8p+504,
                0x1.55bcf72fd10f8p505,
                0x1.55bcf72fd10f8p506,
                0x1.55bcf72fd10f8p507,
                0x1.55bcf72fd10f8p+508,
                0x1.55bcf72fd10f8p509,
                0x1.55bcf72fd10f8p510,
                0x1.55bcf72fd10f8p511,
                0x1.55bcf72fd10f8p+512,
                0x1.55be4104fe79ap-728,
                0x1.55be4104fe79ap-729,
                0x1.55be4104fe79ap-730,
                0x1.55be4104fe79ap-731,
                0x1.55be4104fe79ap-732,
                0x1.55c1c37c286f4p-735,
                0x1.55c1c37c286f4p-736,
                0x1.55c1c37c286f4p-737,
                0x1.55c2076bf9a55p-316,
                0x1.55cf5c3fdc5f8p-474,
                0x1.55d0895ee92a3p-82,
                0x1.55d394bd7a689p-170,
                0x1.55d753fa93af1p+188,
                0x1.55ddc94f5bcfbp-552,
                0x1.55de832483be4p+816,
                0x1.55e12b3658e44p-653,
                0x1.55e8b2f22f502p-1001,
                0x1.55ec16f04d785p509,
                0x1.55ee5904a186p-173,
                0x1.55ee5904a186p-174,
                0x1.55ee5904a186p-175,
                0x1.55ee5904a186p-176,
                0x1.55f31e4a735e4p346,
                0x1.55f57c3159fc3p-907,
                0x1.55f6b4e7fd33p721,
                0x1.55f6b4e7fd33p722,
                0x1.55fc86be6eb44p-725,
                0x1.55ffed5ee8aaep933,
                0x1.560ba23ba8024p-324,
                0x1.5612ddcdcf894p-609,
                0x1.5612ddcdcf894p-610,
                0x1.5614c117f07ccp-996,
                0x1.5617970e7e3bep137,
                0x1.561a550da0f44p-729,
                0x1.561d179fd2c6ap+892,
                0x1.561d276ddfdcp+168,
                0x1.561d276ddfdcp169,
                0x1.561def4a9ee32p381,
                0x1.561def4a9ee32p382,
                0x1.561def4a9ee32p383,
                0x1.561def4a9ee32p+384,
                0x1.561def4a9ee32p385,
                0x1.561def4a9ee32p386,
                0x1.561def4a9ee32p387,
                0x1.561def4a9ee32p+388,
                0x1.561def4a9ee32p389,
                0x1.561def4a9ee32p390,
                0x1.561def4a9ee32p391,
                0x1.561def4a9ee32p+392,
                0x1.561ec3b2e1f5ep261,
                0x1.561f82f71b94ep-639,
                0x1.56215f494942p-262,
                0x1.56215f494942p-263,
                0x1.5623e192efc1p-171,
                0x1.5623e192efc1p-172,
                0x1.562722aa824dp+504,
                0x1.563091a9d905fp-531,
                0x1.5637dae8c920ap-732,
                0x1.56421504411a2p+904,
                0x1.5648210e33d6dp863,
                0x1.564a387be519p-1026,
                0x1.564caeb7201aep-718,
                0x1.564caeb7201aep-719,
                0x1.564d7b53674f8p-825,
                0x1.564d7b53674f8p-826,
                0x1.564d7b53674f8p-827,
                0x1.5655b255bca3ep757,
                0x1.5655b255bca3ep758,
                0x1.5655b255bca3ep759,
                0x1.56596a213dfbep-169,
                0x1.56596a213dfbep-170,
                0x1.56596a213dfbep-171,
                0x1.56596a213dfbep-172,
                0x1.56596a213dfbep-173,
                0x1.5660bad9cd0e7p-113,
                0x1.566554cd85edcp-1024,
                0x1.56731f3fe5ec6p-187,
                0x1.56785fbbdd55ap654,
                0x1.56792e08fb566p-87,
                0x1.567ae2ce80274p386,
                0x1.567ae2ce80274p387,
                0x1.5680b5559c035p-719,
                0x1.5685978b7c329p-668,
                0x1.568a19263f40ep-143,
                0x1.568a19263f40ep-144,
                0x1.568a19263f40ep-145,
                0x1.568cbf13a36bcp-873,
                0x1.568d5bdfd1b3p-26,
                0x1.568d5bdfd1b3p-27,
                0x1.568d5bdfd1b3p-28,
                0x1.568d5bdfd1b3p-29,
                0x1.568d5bdfd1b3p-30,
                0x1.5692def8c351ap-704,
                0x1.569626d68b521p-184,
                0x1.5698931438eabp+728,
                0x1.569bda5e56a2cp257,
                0x1.56a55a10bfa4bp787,
                0x1.56a55a10bfa4bp+788,
                0x1.56ac7338655abp+980,
                0x1.56b95680c122cp-749,
                0x1.56c8b6d1398c4p-681,
                0x1.56cf76605855p175,
                0x1.56cf76605855p+176,
                0x1.56d2de1fb8dep365,
                0x1.56d589dc3d0e2p-74,
                0x1.56d589dc3d0e2p-75,
                0x1.56d589dc3d0e2p-76,
                0x1.56d589dc3d0e2p-77,
                0x1.56d589dc3d0e2p-78,
                0x1.56e1fc2f8f358p-997,
                0x1.56e1fc2f8f359p-997,
                0x1.56ea026986ed4p665,
                0x1.56ea93d1afbbp-55,
                0x1.56ea93d1afbbp-56,
                0x1.56eb956ea1e8ep-928,
                0x1.56eb956ea1e8ep-929,
                0x1.56f46660c5f7ep-887,
                0x1.56f636fea5b12p-639,
                0x1.56f95ee60f856p134,
                0x1.56f95ee60f856p135,
                0x1.5712c14d83bdap+556,
                0x1.5712c14d83bdap557,
                0x1.5712c14d83bdap558,
                0x1.573968b841eb8p-164,
                0x1.573c4de0b0254p649,
                0x1.573d68f903ea2p-512,
                0x1.573ef9b8d48cbp-626,
                0x1.5740dc355bf6p825,
                0x1.5740dc355bf6p826,
                0x1.5740dc355bf6p827,
                0x1.57414e289e384p-845,
                0x1.57473feaf2f47p-805,
                0x1.5747ab143e354p+88,
                0x1.5747ab143e354p89,
                0x1.5747ab143e354p90,
                0x1.5747ab143e354p91,
                0x1.5747ab143e354p+92,
                0x1.5747ab143e354p93,
                0x1.5748d72ba04dcp-964,
                0x1.5748d72ba04dcp-965,
                0x1.574d787d87eacp141,
                0x1.574d787d87eacp142,
                0x1.574d787d87eacp143,
                0x1.575927f546957p-549,
                0x1.575e4c0df6f84p-381,
                0x1.575e4c0df6f84p-382,
                0x1.5760e7672f2bp905,
                0x1.576b12e90c3f1p+492,
                0x1.576b12e90c3f1p493,
                0x1.577833a4b1e3ep687,
                0x1.577833a4b1e3ep+688,
                0x1.577833a4b1e3ep689,
                0x1.577833a4b1e3ep690,
                0x1.577833a4b1e3ep691,
                0x1.577833a4b1e3ep+692,
                0x1.5779b67a1dbccp-385,
                0x1.578a057b85e2cp297,
                0x1.5798ee2308c3ap-27,
                0x1.579b41d949c39p+564,
                0x1.57a27d24d68b4p249,
                0x1.57a340eb5d4f1p-758,
                0x1.57a340eb5d4f1p-759,
                0x1.57a832196cf86p625,
                0x1.57aab67c85a27p-826,
                0x1.57ad455d89a3p894,
                0x1.57ad455d89a3p895,
                0x1.57adf634c359ep-439,
                0x1.57aeb28204d1ap-578,
                0x1.57b0304b1649cp547,
                0x1.57b0304b1649cp+548,
                0x1.57c520e1c5eaep597,
                0x1.57c520e1c5eaep598,
                0x1.57c520e1c5eaep599,
                0x1.57c8b36d64e8ep-307,
                0x1.57d2c16f14024p589,
                0x1.57d52ce4fb8d6p-723,
                0x1.57d52ce4fb8d6p-724,
                0x1.57d52ce4fb8d6p-725,
                0x1.57d52ce4fb8d6p-726,
                0x1.57d52ce4fb8d6p-727,
                0x1.57d52ce4fb8d6p-728,
                0x1.57d5f2c17440ep766,
                0x1.57eb8ad52a5c9p182,
                0x1.57eb8ad52a5c9p183,
                0x1.57eb8ad52a5c9p+184,
                0x1.57eca77fdd646p-433,
                0x1.57f48bb41db7cp458,
                0x1.57fb706028cb2p757,
                0x1.57fb706028cb2p758,
                0x1.57fb706028cb2p759,
                0x1.57fb706028cb2p+760,
                0x1.57fb706028cb2p761,
                0x1.580328378cfe8p+480,
                0x1.580328378cfe8p481,
                0x1.580658cb6ccbp-341,
                0x1.580658cb6ccbp-342,
                0x1.58098637f1284p786,
                0x1.580b0c35f9b9ap365,
                0x1.580cce42d3c76p191,
                0x1.58137558c2766p+188,
                0x1.58137558c2766p189,
                0x1.5816199625103p602,
                0x1.5818e87da9e6p-904,
                0x1.581ba7fbf0c27p498,
                0x1.5820c3849fd45p187,
                0x1.5820fa677ba35p-565,
                0x1.583a7da54489dp-645,
                0x1.584453f70120cp+136,
                0x1.584453f70120cp137,
                0x1.5844d0f74d7e9p-199,
                0x1.584b93843fa0ep-379,
                0x1.584d09a4c7ef8p-951,
                0x1.584fcbad2b7bcp-258,
                0x1.585041b2c477ep942,
                0x1.585041b2c477ep943,
                0x1.585041b2c477ep+944,
                0x1.585041b2c477ep945,
                0x1.585041b2c477ep946,
                0x1.585c8714d20dp-993,
                0x1.585c8714d20dp-994,
                0x1.586409ed54d12p697,
                0x1.5867d4881b97ep+240,
                0x1.5867d4881b97ep241,
                0x1.5867d4881b97ep242,
                0x1.5867d4881b97ep243,
                0x1.5867d4881b97ep+244,
                0x1.5867d4881b97ep245,
                0x1.586a907ba0574p-450,
                0x1.586b24ce4af14p557,
                0x1.586b24ce4af14p558,
                0x1.5870988bd007fp190,
                0x1.5874bb0e0b994p543,
                0x1.587601ded2a04p-444,
                0x1.587691b7e3c02p491,
                0x1.58772d8ca9448p+568,
                0x1.587b54b6936dap-854,
                0x1.588af345b103dp673,
                0x1.588f5bccfac5p-693,
                0x1.5890d1ef6a0dap+120,
                0x1.5890d1ef6a0dap121,
                0x1.5891205812a6bp-157,
                0x1.5891f6f0a7bd3p+848,
                0x1.5894e89baf4fcp-127,
                0x1.5895abc356a2bp977,
                0x1.58a009397dd92p-675,
                0x1.58a274b4f2126p-604,
                0x1.58a57dff1dc34p+1008,
                0x1.58ad6f5d0a854p-44,
                0x1.58ba6fab6f36cp-708,
                0x1.58c3cf6ba79d1p202,
                0x1.58c47e6eea282p-370,
                0x1.58c47e6eea282p-371,
                0x1.58c47e6eea282p-372,
                0x1.58c47e6eea282p-373,
                0x1.58ca8039be1e4p+248,
                0x1.58cb7c5724514p-249,
                0x1.58cbb1d9b87a6p881,
                0x1.58cd0beda7ecap-95,
                0x1.58cd0beda7ecap-96,
                0x1.58ce5f6a2d246p978,
                0x1.58d3f517f263p589,
                0x1.58d3f517f263p590,
                0x1.58d3f517f263p591,
                0x1.58d516dca268fp-847,
                0x1.58dc1055d276bp+948,
                0x1.58e1a273b405ap-668,
                0x1.58p-1069,
                0x1.5901681dadcc8p-586,
                0x1.5901681dadcc8p-587,
                0x1.590363d103caep405,
                0x1.5908369e56afep+800,
                0x1.590c3621cdd4p-239,
                0x1.59165a6ddda5bp-223,
                0x1.59165a6ddda5cp-222,
                0x1.59165a6ddda5cp-223,
                0x1.5916b681b1cc8p-615,
                0x1.5916b681b1cc8p-616,
                0x1.591ffabd8c684p458,
                0x1.5924f765d6e7p1018,
                0x1.5925507b55105p+660,
                0x1.5925507b55105p661,
                0x1.592953e47443ep670,
                0x1.592f9932c06bdp-291,
                0x1.592f9932c06bdp-292,
                0x1.592f9932c06bdp-293,
                0x1.59374a773e107p-426,
                0x1.593f96c07818ap-438,
                0x1.5946e1e98535p-177,
                0x1.5946e1e98535p-178,
                0x1.5949a070d8b3ep-751,
                0x1.594e4392f3197p641,
                0x1.594e4392f3197p642,
                0x1.59593bf527a9dp-337,
                0x1.59644266c9f6ap-178,
                0x1.59644266c9f6ap-179,
                0x1.59725db272f7fp262,
                0x1.597b3694accbap-297,
                0x1.598182cd56c7cp702,
                0x1.59843e8c86a09p-537,
                0x1.59843e8c86a09p-538,
                0x1.5985d956631f8p+224,
                0x1.598c0a4f6c49ap-655,
                0x1.5993a2fe24258p-1004,
                0x1.59963a1d7b506p-665,
                0x1.59963a1d7b506p-666,
                0x1.5999a564c6db9p302,
                0x1.599e0fa9e1d6cp-1003,
                0x1.599e6bd54fb8cp509,
                0x1.599e6bd54fb8cp510,
                0x1.59a2783ce70abp-328,
                0x1.59a2783ce70abp-329,
                0x1.59a920ac84a2p-865,
                0x1.59a920ac84a2p-866,
                0x1.59a95b60edd47p-135,
                0x1.59b83b3f9115cp-55,
                0x1.59ba1152531ap+860,
                0x1.59bd3ad46e346p+192,
                0x1.59bd3ad46e346p193,
                0x1.59bd3ad46e346p194,
                0x1.59c2e58bb7622p+236,
                0x1.59ce797fb817fp747,
                0x1.59ce797fb818p746,
                0x1.59ce797fb818p747,
                0x1.59ce797fb818p+748,
                0x1.59ce797fb818p749,
                0x1.59d16e02d5bb8p853,
                0x1.59da4cad9bfcep451,
                0x1.59da4cad9bfcep+452,
                0x1.59df2c6cb3bfap-326,
                0x1.59e3859fb664fp-996,
                0x1.59e5c78e6dbbp-1025,
                0x1.59f0a4823d0dfp286,
                0x1.59f732b2972d8p214,
                0x1.59f732b2972d8p215,
                0x1.59f732b2972d8p+216,
                0x1.59f732b2972d8p217,
                0x1.59f732b2972d8p218,
                0x1.59f732b2972d8p219,
                0x1.5a00f828744a6p-535,
                0x1.5a090d99d8ae7p805,
                0x1.5a0b06b67ffffp-346,
                0x1.5a13d7174d9p-1028,
                0x1.5a22c1390d582p177,
                0x1.5a248978f9bc6p-440,
                0x1.5a248978f9bc6p-441,
                0x1.5a2b8f45a18f4p-56,
                0x1.5a30ccae0dd44p-558,
                0x1.5a33ec5f9e0aep-444,
                0x1.5a33ec5f9e0aep-445,
                0x1.5a33ec5f9e0aep-446,
                0x1.5a33ec5f9e0aep-447,
                0x1.5a33ec5f9e0aep-448,
                0x1.5a340d037c7cp967,
                0x1.5a340d037c7cp+968,
                0x1.5a340d037c7cp969,
                0x1.5a340d037c7cp970,
                0x1.5a391d56bdc87p-904,
                0x1.5a391d56bdc88p-903,
                0x1.5a391d56bdc88p-904,
                0x1.5a39ec0dfa234p750,
                0x1.5a3e1d1b7309ap-639,
                0x1.5a3e1d1b7309ap-640,
                0x1.5a3e1d1b7309ap-641,
                0x1.5a3e1d1b7309ap-642,
                0x1.5a4337446ef2ap-565,
                0x1.5a4337446ef2ap-566,
                0x1.5a4337446ef2ap-567,
                0x1.5a4337446ef2ap-568,
                0x1.5a4337446ef2ap-569,
                0x1.5a4337446ef2ap-570,
                0x1.5a4337446ef2ap-571,
                0x1.5a462d91c6ab4p-819,
                0x1.5a462d91c6ab4p-820,
                0x1.5a462d91c6ab4p-821,
                0x1.5a462d91c6ab4p-822,
                0x1.5a462d91c6ab4p-823,
                0x1.5a462d91c6ab4p-824,
                0x1.5a462d91c6ab4p-825,
                0x1.5a462d91c6ab4p-826,
                0x1.5a462d91c6ab4p-827,
                0x1.5a462d91c6ab4p-828,
                0x1.5a462d91c6ab4p-829,
                0x1.5a462d91c6ab4p-830,
                0x1.5a462d91c6ab4p-831,
                0x1.5a462d91c6ab4p-832,
                0x1.5a527d5727ccp-347,
                0x1.5a607ba27e088p-864,
                0x1.5a73b03a5f214p-706,
                0x1.5a73b03a5f214p-707,
                0x1.5a75e44c5d803p-326,
                0x1.5a79c4e71d028p-511,
                0x1.5a79c4e71d028p-512,
                0x1.5a79c4e71d028p-513,
                0x1.5a79c4e71d028p-514,
                0x1.5a7d6d7c8fb8cp802,
                0x1.5a857d2ad9f1ep125,
                0x1.5a85e8ac901f6p349,
                0x1.5a85e8ac901f6p350,
                0x1.5a8610ba39fcep903,
                0x1.5a88153f5a96ap-171,
                0x1.5a88896ef9a77p713,
                0x1.5a956e225d672p-419,
                0x1.5aa56c3be286bp399,
                0x1.5aabbd09bdb1ep+308,
                0x1.5ab6d2c8170dp+996,
                0x1.5ac299488d23cp485,
                0x1.5ac299488d23cp486,
                0x1.5ac299488d23cp487,
                0x1.5acc2053064c2p-959,
                0x1.5acc2053064c2p-960,
                0x1.5acc2053064c2p-961,
                0x1.5ad084696d644p-516,
                0x1.5ad35a8f3d556p-668,
                0x1.5ad35a8f3d556p-669,
                0x1.5ad35a8f3d556p-670,
                0x1.5ad80aebd34dp321,
                0x1.5ad80aebd34dp322,
                0x1.5ad80aebd34dp323,
                0x1.5ade0681e0613p-543,
                0x1.5ae45ad573bc7p718,
                0x1.5af09d162b664p+580,
                0x1.5af1d78b58c4p66,
                0x1.5af26ef0313bfp607,
                0x1.5af633e2db87ap785,
                0x1.5af7693607903p-290,
                0x1.5af7693607903p-291,
                0x1.5af8175ea8b7dp+536,
                0x1.5afcb3f6d7bd4p-839,
                0x1.5afd65538df02p-387,
                0x1.5afffe018f47ep-324,
                0x1.5afffe018f47ep-325,
                0x1.5b074931360eap603,
                0x1.5b074931360eap+604,
                0x1.5b0e92315be3p-352,
                0x1.5b0e92315be3p-353,
                0x1.5b1051df943a8p381,
                0x1.5b1051df943a8p382,
                0x1.5b1051df943a8p383,
                0x1.5b1051df943a8p+384,
                0x1.5b1051df943a8p385,
                0x1.5b1051df943a8p386,
                0x1.5b1051df943a8p387,
                0x1.5b1051df943a8p+388,
                0x1.5b1630a6057bap855,
                0x1.5b194ad85f919p106,
                0x1.5b1aa5bc89ca6p299,
                0x1.5b1c9fcab6b1p-257,
                0x1.5b1c9fcab6b1p-258,
                0x1.5b1c9fcab6b1p-259,
                0x1.5b1f374e536b8p-249,
                0x1.5b2796ff8p-1041,
                0x1.5b29df55793f4p-333,
                0x1.5b34be9487c22p-25,
                0x1.5b34be9487c22p-26,
                0x1.5b34be9487c22p-27,
                0x1.5b3b7c4a6dbf6p-214,
                0x1.5b3e3ff2bbdf8p-66,
                0x1.5b4606f1d7d61p813,
                0x1.5b490d7f6b9b9p-226,
                0x1.5b490d7f6b9b9p-227,
                0x1.5b490d7f6b9b9p-228,
                0x1.5b495b6301984p+468,
                0x1.5b495b6301984p469,
                0x1.5b4e5998400a9p551,
                0x1.5b4e5998400aap551,
                0x1.5b4e5998400aap+552,
                0x1.5b51b8c860fd3p961,
                0x1.5b51e773a39cdp-209,
                0x1.5b538bc7e72b2p-568,
                0x1.5b538bc7e72b2p-569,
                0x1.5b5c0fc2706ddp-59,
                0x1.5b636a1a86b16p+296,
                0x1.5b7ba421a9c6ep479,
                0x1.5b7ba421a9c6ep+480,
                0x1.5b809505b9432p-80,
                0x1.5b84bd3e6d015p243,
                0x1.5b85863f0fbbap+912,
                0x1.5b85a30c141bap154,
                0x1.5b8b0aeb5cad4p+864,
                0x1.5b8fc34f0bb94p-233,
                0x1.5b8fc34f0bb94p-234,
                0x1.5bb10fdb09f8ep-217,
                0x1.5bb122ab9ca24p-410,
                0x1.5bbd8bf744628p-726,
                0x1.5bbd8bf744628p-727,
                0x1.5bbd8bf744628p-728,
                0x1.5bbd8bf744628p-729,
                0x1.5bbd8bf744628p-730,
                0x1.5bbd8bf744628p-731,
                0x1.5bbd8bf744628p-732,
                0x1.5bbd8bf744628p-733,
                0x1.5bbf869c005eep541,
                0x1.5bc471d5456c8p1007,
                0x1.5bc471d5456c8p+1008,
                0x1.5bc471d5456c8p1009,
                0x1.5bc471d5456c8p1010,
                0x1.5bc471d5456c8p1011,
                0x1.5bcef1ccbd285p-562,
                0x1.5bcf8add80d8dp+520,
                0x1.5bd3462e73864p134,
                0x1.5bd740c7c1daap-470,
                0x1.5bdb13e8714bp266,
                0x1.5be1p-1058,
                0x1.5be2a4f52f6a8p-74,
                0x1.5be383a7b9b0ap-184,
                0x1.5be383a7b9b0ap-185,
                0x1.5bec757abc7f2p-305,
                0x1.5bf2d22dde8c4p+876,
                0x1.5bf2d22dde8c4p877,
                0x1.5bf2d22dde8c4p878,
                0x1.5bf2d22dde8c4p879,
                0x1.5bf2d22dde8c4p+880,
                0x1.5bf83e66bff5cp-243,
                0x1.5bf83e66bff5cp-244,
                0x1.5bf83e66bff5cp-245,
                0x1.5bfa50e326e74p+720,
                0x1.5c0fc374eb30ap-436,
                0x1.5c162b168e70ep-615,
                0x1.5c162b168e70ep-616,
                0x1.5c19aacba8a2ap721,
                0x1.5c19aacba8a2ap722,
                0x1.5c19aacba8a2ap723,
                0x1.5c19aacba8a2ap+724,
                0x1.5c19aacba8a2ap725,
                0x1.5c19aacba8a2ap726,
                0x1.5c1fc1dce5f8ep137,
                0x1.5c2432f0736e9p+136,
                0x1.5c30007fb13eep-771,
                0x1.5c30007fb13eep-772,
                0x1.5c3a432d9802fp-506,
                0x1.5c3a96fd75be7p-899,
                0x1.5c47da84ed2e9p-435,
                0x1.5c47da84ed2e9p-436,
                0x1.5c51d380ddca6p+660,
                0x1.5c5203c0aad53p-54,
                0x1.5c55dcdeb00c1p-418,
                0x1.5c5925eefc199p-217,
                0x1.5c61d4cefe96ep-812,
                0x1.5c6275de3e68bp695,
                0x1.5c648bb3e8dcbp-803,
                0x1.5c6714def374cp-32,
                0x1.5c6714def374cp-33,
                0x1.5c6714def374cp-34,
                0x1.5c6714def374cp-35,
                0x1.5c6714def374cp-36,
                0x1.5c6714def374cp-37,
                0x1.5c6714def374cp-38,
                0x1.5c6714def374cp-39,
                0x1.5c6bd36a5dfd8p+852,
                0x1.5c71ea82b1f36p-590,
                0x1.5c7243423ca38p669,
                0x1.5c72fb1552d83p-130,
                0x1.5c72fb1552d84p-128,
                0x1.5c72fb1552d84p-129,
                0x1.5c72fb1552d84p-130,
                0x1.5c72fb1552d84p-131,
                0x1.5c75115009c78p-239,
                0x1.5c75115009c78p-240,
                0x1.5c7d1db5f9cd8p-248,
                0x1.5c7d8337d0c1ep-327,
                0x1.5c872f700d98ep857,
                0x1.5c87356f12d2cp-68,
                0x1.5c87356f12d2cp-69,
                0x1.5c87356f12d2cp-70,
                0x1.5c8884892cdd9p+408,
                0x1.5c95e9dd809acp-838,
                0x1.5c95e9dd809acp-839,
                0x1.5c9bfd1cf02b8p-432,
                0x1.5c9f6ace68e4p-175,
                0x1.5ca430f6f3a25p-884,
                0x1.5cb62b980a683p-361,
                0x1.5cb62b980a683p-362,
                0x1.5cbf6bc0de98p-576,
                0x1.5cbf6bc0de98p-577,
                0x1.5cbf6bc0de98p-578,
                0x1.5cc529c27675p122,
                0x1.5cc9442d2b209p986,
                0x1.5cc9442d2b209p987,
                0x1.5ccfe3d35d80ep355,
                0x1.5cd7a285061b6p-109,
                0x1.5cda11174db68p689,
                0x1.5cda11174db68p690,
                0x1.5cda11174db68p691,
                0x1.5cdbb018d299cp+1020,
                0x1.5cdd04dd38eap931,
                0x1.5cdfad1e6cdb8p109,
                0x1.5cdfad1e6cdb8p110,
                0x1.5cdfad1e6cdb8p111,
                0x1.5cdfad1e6cdb8p+112,
                0x1.5ce1f143d7ad2p-444,
                0x1.5ce1f143d7ad2p-445,
                0x1.5ce239c700459p235,
                0x1.5ce363b7821a9p-214,
                0x1.5ce98357dc88ep-935,
                0x1.5ce98357dc88ep-936,
                0x1.5ced4f7cd8d49p-978,
                0x1.5cef89a4f1a84p-242,
                0x1.5cfab65116b1cp-453,
                0x1.5cfec09f0dce2p347,
                0x1.5d00436d5436ep793,
                0x1.5d0a70c57481p-738,
                0x1.5d0b928c5eaap801,
                0x1.5d0f1a29d8d8cp914,
                0x1.5d12c4dff62e8p-265,
                0x1.5d15b26b80e3p-1008,
                0x1.5d1bd863c2f4cp563,
                0x1.5d2ce55747a18p+840,
                0x1.5d2ce55747a18p841,
                0x1.5d2f2b38e9324p-934,
                0x1.5d2f2b38e9324p-935,
                0x1.5d2f2b38e9324p-936,
                0x1.5d35e72852ac8p+812,
                0x1.5d45f71773cf6p-959,
                0x1.5d45fdaf8293cp-716,
                0x1.5d4cb547aa3f6p-542,
                0x1.5d4fe53afec65p-726,
                0x1.5d4fe53afec65p-727,
                0x1.5d50974bf315p-114,
                0x1.5d50974bf315p-115,
                0x1.5d50974bf315p-116,
                0x1.5d5e371c32478p-910,
                0x1.5d6ceab33d056p909,
                0x1.5d6ceab33d056p910,
                0x1.5d6fd838f7a92p+220,
                0x1.5d735edd582dep870,
                0x1.5d74ad2a6684ep+732,
                0x1.5d776b4f4c012p-957,
                0x1.5d7e4021d7be4p-803,
                0x1.5d8156a0f65dp-1025,
                0x1.5d8156a0f65dp-1026,
                0x1.5d8400e4ae754p+148,
                0x1.5d8fb4bf36c5ap-918,
                0x1.5d94697c0798fp485,
                0x1.5d95a38df767p871,
                0x1.5d95a38df767p+872,
                0x1.5d98932280f0ap-809,
                0x1.5d98932280f0ap-810,
                0x1.5d98932280f0ap-811,
                0x1.5d98932280f0ap-812,
                0x1.5d9b17e503077p530,
                0x1.5d9c16b9eb536p526,
                0x1.5dae2ebbd7e3dp-273,
                0x1.5db182b98f89cp949,
                0x1.5db182b98f89cp950,
                0x1.5db182b98f89cp951,
                0x1.5db182b98f89cp+952,
                0x1.5db182b98f89cp953,
                0x1.5db872d9ba744p214,
                0x1.5db872d9ba744p215,
                0x1.5db872d9ba744p+216,
                0x1.5db872d9ba744p217,
                0x1.5db872d9ba744p218,
                0x1.5db872d9ba744p219,
                0x1.5db872d9ba744p+220,
                0x1.5db872d9ba744p221,
                0x1.5dc3bde87be37p785,
                0x1.5dc83a1a72e7ep-28,
                0x1.5dc98655bd4dp+540,
                0x1.5dd48bf11cabep+516,
                0x1.5dd5b5e4d514dp-648,
                0x1.5dd648b936bdep-149,
                0x1.5dd90b559375ap795,
                0x1.5dd90b559375ap+796,
                0x1.5dd90b559375ap797,
                0x1.5dd90b559375ap798,
                0x1.5de04d788485p834,
                0x1.5dee6e625311ep-367,
                0x1.5df5ca28ef40dp-326,
                0x1.5df5ca28ef40ep-325,
                0x1.5df5ca28ef40ep-326,
                0x1.5df937f03c22ep502,
                0x1.5df937f03c22ep503,
                0x1.5dfa71b53448ep641,
                0x1.5dfa71b53448ep642,
                0x1.5dfec0a291b5cp-885,
                0x1.5dff9192680ap+496,
                0x1.5dff9192680ap497,
                0x1.5dff9192680ap498,
                0x1.5e003faead3cp274,
                0x1.5e003faead3cp275,
                0x1.5e003faead3cp+276,
                0x1.5e003faead3cp277,
                0x1.5e003faead3cp278,
                0x1.5e008ef88392p-130,
                0x1.5e0164a317246p-240,
                0x1.5e0262d9e21dp-574,
                0x1.5e094286bc66ep-153,
                0x1.5e094286bc66ep-154,
                0x1.5e094286bc66ep-155,
                0x1.5e094286bc66ep-156,
                0x1.5e0a2180361dcp203,
                0x1.5e0a2180361dcp+204,
                0x1.5e0a2180361dcp205,
                0x1.5e0a2180361dcp206,
                0x1.5e0a2180361dcp207,
                0x1.5e0a2180361dcp+208,
                0x1.5e0dca865ec52p-903,
                0x1.5e0dca865ec52p-904,
                0x1.5e1f27b4c2a74p-134,
                0x1.5e1f27b4c2a74p-135,
                0x1.5e1f27b4c2a74p-136,
                0x1.5e2381cdf8fd3p+500,
                0x1.5e2c10ad970bp297,
                0x1.5e304951356d6p307,
                0x1.5e304951356d6p+308,
                0x1.5e304951356d6p309,
                0x1.5e31e72cfb985p+192,
                0x1.5e42p-1059,
                0x1.5e531a0a1c872p159,
                0x1.5e60dbb13290ep683,
                0x1.5e60e2742ef6p-596,
                0x1.5e60e2742ef6p-597,
                0x1.5e66dc3d6bdb4p-100,
                0x1.5e66dc3d6bdb4p-101,
                0x1.5e66dc3d6bdb4p-102,
                0x1.5e66dc3d6bdb4p-103,
                0x1.5e66dc3d6bdb4p-104,
                0x1.5e66dc3d6bdb4p-97,
                0x1.5e66dc3d6bdb4p-98,
                0x1.5e66dc3d6bdb4p-99,
                0x1.5e68c12ad60f6p697,
                0x1.5e6b1ed360536p483,
                0x1.5e7309797c034p-263,
                0x1.5e74d56ef9b44p229,
                0x1.5e7896eb77298p654,
                0x1.5e88efcp-1048,
                0x1.5e8ac17419b12p+520,
                0x1.5e96d325e70a8p+472,
                0x1.5ead46e6d3675p-356,
                0x1.5eb082cca94d7p+644,
                0x1.5eb082cca94d8p+644,
                0x1.5eb082cca94d8p645,
                0x1.5eb0d8350317ep-441,
                0x1.5eb130d1d6ffap-19,
                0x1.5eb2a525511c8p-48,
                0x1.5eb2a525511c8p-49,
                0x1.5eb49662d0f36p-171,
                0x1.5eb9053f2716ep-443,
                0x1.5eb9053f2716ep-444,
                0x1.5eb9053f2716ep-445,
                0x1.5ebff46312d8ep-715,
                0x1.5ebff46312d8ep-716,
                0x1.5ec6de64b4efcp-622,
                0x1.5ec6de64b4efcp-623,
                0x1.5ec6de64b4efcp-624,
                0x1.5ec6de64b4efcp-625,
                0x1.5ec6de64b4efcp-626,
                0x1.5ecacb546c52cp+560,
                0x1.5ecacb546c52cp561,
                0x1.5ecacb546c52cp562,
                0x1.5ecacb546c52cp563,
                0x1.5ecacb546c52cp+564,
                0x1.5ecd2c92bd8dp-561,
                0x1.5ecf6d9b58999p483,
                0x1.5ed0e0db97b1cp-583,
                0x1.5ed0e0db97b1cp-584,
                0x1.5ed0e0db97b1cp-585,
                0x1.5ed1f039cebfep438,
                0x1.5ed1f039cebfep439,
                0x1.5ed1f039cebfep+440,
                0x1.5ed598bcd6a1dp457,
                0x1.5ed6797f0cb0ap+480,
                0x1.5ed8p-1054,
                0x1.5ed8p-1055,
                0x1.5edae6e5d57a2p-863,
                0x1.5ef04f4ef33a2p-89,
                0x1.5ef6b8f65b336p873,
                0x1.5f01a85b67cbdp-879,
                0x1.5f0300d03ce2cp+404,
                0x1.5f0300d03ce2cp405,
                0x1.5f0300d03ce2cp406,
                0x1.5f0300d03ce2cp407,
                0x1.5f0300d03ce2cp+408,
                0x1.5f07229ee5ef7p-250,
                0x1.5f0d408362a72p802,
                0x1.5f0d408362a72p803,
                0x1.5f0d408362a72p+804,
                0x1.5f0d408362a72p805,
                0x1.5f0d408362a72p806,
                0x1.5f0e002250a64p635,
                0x1.5f140b101365bp-1000,
                0x1.5f14ad485e23ep-507,
                0x1.5f1b3f973963p+864,
                0x1.5f1c8p-1056,
                0x1.5f1ca820511c3p-1007,
                0x1.5f1ca820511c4p-1005,
                0x1.5f1ca820511c4p-1006,
                0x1.5f1ca820511c4p-1007,
                0x1.5f1ca820511c4p-1008,
                0x1.5f1d7a64fd7f2p707,
                0x1.5f265fbb59685p-176,
                0x1.5f299cd1888c4p-767,
                0x1.5f2df5e675a0fp-454,
                0x1.5f2df5e675a0fp-455,
                0x1.5f2df5e675a0fp-456,
                0x1.5f2df5e675a0fp-457,
                0x1.5f37dcc7b9a02p-612,
                0x1.5f3aa454p-1042,
                0x1.5f3f392f6035cp+860,
                0x1.5f3f3d6c3f9eap-508,
                0x1.5f4389df9edf6p+204,
                0x1.5f4389df9edf6p205,
                0x1.5f5af4ca8e80ep-1022,
                0x1.5f6a4b5e2ca9ap-293,
                0x1.5f6de9d5d6b5bp445,
                0x1.5f6de9d5d6b5bp446,
                0x1.5f6de9d5d6b5bp447,
                0x1.5f716f733b446p-349,
                0x1.5f73fce7fd0cep799,
                0x1.5f73fce7fd0cep+800,
                0x1.5f7a46a0c89ddp-522,
                0x1.5f7a46a0c89dep-521,
                0x1.5f7a46a0c89dep-522,
                0x1.5f7b7ace6ba7bp370,
                0x1.5f7b7ace6ba7bp371,
                0x1.5f7be21275cb5p-138,
                0x1.5f7be21275cb5p-139,
                0x1.5f7be21275cb5p-140,
                0x1.5f7dcf0fba54p814,
                0x1.5f7dcf0fba54p815,
                0x1.5f885ff78e26ep+364,
                0x1.5f8d402a52368p90,
                0x1.5f9a6bf029e3dp-264,
                0x1.5f9b823b41018p-162,
                0x1.5f9e76f25e8f2p-416,
                0x1.5fafa20c82156p570,
                0x1.5fafa20c82156p571,
                0x1.5fb367a515c2ep-671,
                0x1.5fb367a515c2ep-672,
                0x1.5fb41f690e47p-122,
                0x1.5fbd64f48cb16p-663,
                0x1.5fc26b24fe582p-165,
                0x1.5fc26b24fe582p-166,
                0x1.5fc26b24fe582p-167,
                0x1.5fc26b24fe582p-168,
                0x1.5fc26b24fe582p-169,
                0x1.5fc27d5775e8ap-674,
                0x1.5fc439d80e118p194,
                0x1.5fc439d80e118p195,
                0x1.5fc439d80e118p+196,
                0x1.5fc439d80e118p197,
                0x1.5fc439d80e118p198,
                0x1.5fd5p-1057,
                0x1.5fd7aa44d9477p967,
                0x1.5fd7aa44d9477p+968,
                0x1.5fd7aa44d9477p969,
                0x1.5fd7fe1796495p-37,
                0x1.5fd7fe1796496p-36,
                0x1.5fd7fe1796496p-37,
                0x1.5fdab94c9772cp+800,
                0x1.5fdab94c9772cp801,
                0x1.5fdab94c9772cp802,
                0x1.5fdab94c9772cp803,
                0x1.5fdfe248fc327p-442,
                0x1.5fe1c381b5b32p607,
                0x1.5fe5863066b1cp705,
                0x1.5fea33b2b5e5dp-836,
                0x1.5fedbd41aa985p+500,
                0x1.5fedbd41aa985p501,
                0x1.5ff2df047d53cp-389,
                0x1.5ff3c402d03fcp801,
                0x1.5ff459c7768d8p-322,
                0x1.5ff459c7768d8p-323,
                0x1.5ff5dcc65b2fep-884,
                0x1.5ff5dcc65b2fep-885,
                0x1.5ff5dcc65b2fep-886,
                0x1.5ff801bfe4f5cp253,
                0x1.5ff9224a521c6p+360,
                0x1.5ffb6c5cf3041p1017,
                0x1.5ffeb886d5eb8p-659,
                0x1.5ffeb886d5eb8p-660,
                0x1.5p-1067,
                0x1.60073fd3586b1p-525,
                0x1.60073fd3586b1p-526,
                0x1.60073fd3586b1p-527,
                0x1.600e177ee4a5ap803,
                0x1.6011cea0e66d4p-858,
                0x1.601351e34ce4p-729,
                0x1.6016ac97db3afp-433,
                0x1.601b2e50c87b3p-178,
                0x1.6026b2e07ec07p-35,
                0x1.602d43cc0493cp+432,
                0x1.602d43cc0493cp433,
                0x1.602d43cc0493cp434,
                0x1.602d43cc0493cp435,
                0x1.60304b50de9ep+228,
                0x1.6031e3c370b7bp-472,
                0x1.603343e1578f2p+980,
                0x1.60358a0edd4bbp113,
                0x1.6035ce8b6203dp+448,
                0x1.603c663f85129p186,
                0x1.603c663f85129p187,
                0x1.603ef04dc7741p339,
                0x1.604175b131d88p+800,
                0x1.604189374bc6ap-5,
                0x1.60480901758c4p-351,
                0x1.604b9381dea22p985,
                0x1.604b9381dea22p986,
                0x1.604b9381dea22p987,
                0x1.604b9381dea22p+988,
                0x1.605a4b7e0f526p-156,
                0x1.6061245105274p+172,
                0x1.606a88cf735eap-133,
                0x1.60700ab59ecccp-529,
                0x1.6076d7a1a4301p687,
                0x1.6076d7a1a4301p+688,
                0x1.608491c95b06p+348,
                0x1.608491c95b06p349,
                0x1.6086bab581cabp-265,
                0x1.608c21def3722p+848,
                0x1.608c21def3722p849,
                0x1.60927ee212233p343,
                0x1.60927ee212233p+344,
                0x1.6093b802d578cp933,
                0x1.6095eee76f85ap-685,
                0x1.6097c7008422p-634,
                0x1.6097c7008422p-635,
                0x1.6097c7008422p-636,
                0x1.6097c7008422p-637,
                0x1.6097c7008422p-638,
                0x1.609934e7741adp-605,
                0x1.609a6552c8b5cp+268,
                0x1.609a6552c8b5cp269,
                0x1.609a6552c8b5cp270,
                0x1.60a374ab3d7ecp-387,
                0x1.60a5f7552857cp134,
                0x1.60aba5f00289ep-704,
                0x1.60aba5f00289ep-705,
                0x1.60b0ed0c36082p613,
                0x1.60b5757287deap-589,
                0x1.60b5757287deap-590,
                0x1.60b5757287deap-591,
                0x1.60b5757287deap-592,
                0x1.60bb92fea10acp558,
                0x1.60c1ce66d7c46p-262,
                0x1.60c1ce66d7c46p-263,
                0x1.60c1ce66d7c46p-264,
                0x1.60c23623f702ap-790,
                0x1.60c23623f702ap-791,
                0x1.60c23623f702ap-792,
                0x1.60cf8f38b0464p782,
                0x1.60cf8f38b0464p783,
                0x1.60cf8f38b0464p+784,
                0x1.60cf8f38b0464p785,
                0x1.60cf8f38b0464p786,
                0x1.60d84fd627a58p-104,
                0x1.60dbc9399671ep467,
                0x1.60dbc9399671ep+468,
                0x1.60dd21bb19664p-532,
                0x1.60df7679d36dep435,
                0x1.60e1c88193145p-658,
                0x1.60e5922b166bep-460,
                0x1.60e983160dd06p-1002,
                0x1.60e983160dd06p-1003,
                0x1.60ed707e41ea3p898,
                0x1.6100725988694p-718,
                0x1.6107fb9bb89cep703,
                0x1.610eee7a66a42p798,
                0x1.610eee7a66a42p799,
                0x1.610eee7a66a42p+800,
                0x1.610eee7a66a42p801,
                0x1.610f4292f788ap-567,
                0x1.6128418a0b388p103,
                0x1.6133914965f01p529,
                0x1.613a15225ef97p-921,
                0x1.613a15225ef97p-922,
                0x1.61481cc0ccf86p-987,
                0x1.614d8010124bp650,
                0x1.614f5b4c13e66p-247,
                0x1.615742abc6cd8p883,
                0x1.615742abc6cd8p+884,
                0x1.615b80a3caf21p-385,
                0x1.615d7d3697162p-892,
                0x1.615d7d3697162p-893,
                0x1.615d7d3697162p-894,
                0x1.615d7d3697162p-895,
                0x1.615d7d3697162p-896,
                0x1.615e91d8f359dp-233,
                0x1.615e91d8f359dp-234,
                0x1.616e03784339ep+948,
                0x1.616e03784339ep949,
                0x1.616e03784339ep950,
                0x1.616e03784339ep951,
                0x1.616e03784339ep+952,
                0x1.616e03784339ep953,
                0x1.616e03784339ep954,
                0x1.616e03784339ep955,
                0x1.61751ee51a90ep-646,
                0x1.6175acca65dc8p-95,
                0x1.617af47af63dcp-84,
                0x1.617f1418c7316p977,
                0x1.61880e468a7ccp762,
                0x1.61880e468a7ccp763,
                0x1.6188568e7bf96p-42,
                0x1.6188568e7bf96p-43,
                0x1.618a188caeb56p-725,
                0x1.618a188caeb56p-726,
                0x1.618a188caeb56p-727,
                0x1.619b3dcda5ffdp-296,
                0x1.619b519dd6833p+120,
                0x1.619c529ffc4a2p-960,
                0x1.619c529ffc4a2p-961,
                0x1.619c529ffc4a2p-962,
                0x1.619c529ffc4a2p-963,
                0x1.619e922cb74ccp+1016,
                0x1.61a6c588a3562p387,
                0x1.61a6c588a3562p+388,
                0x1.61ade21cdfd0cp-176,
                0x1.61afe79be027ap-660,
                0x1.61afe79be027ap-661,
                0x1.61b65b816553cp-632,
                0x1.61bcca7119916p+252,
                0x1.61c4aaeffa34ep-392,
                0x1.61c4aaeffa34ep-393,
                0x1.61c69a11ffab6p+932,
                0x1.61c892223d692p-201,
                0x1.61c8f73bc9e74p-197,
                0x1.61d7efada37ecp-479,
                0x1.61d89b3ba4af8p-489,
                0x1.61e30b008abd9p910,
                0x1.61e4c701a823ap-123,
                0x1.61e69a49b25b8p442,
                0x1.61ea79f78d7edp-282,
                0x1.61edaec5f8ec6p806,
                0x1.61ef17632444cp-507,
                0x1.61f024bec451bp986,
                0x1.61f024bec451bp987,
                0x1.61f50f7c4b916p-617,
                0x1.61f50f7c4b916p-618,
                0x1.61f50f7c4b916p-619,
                0x1.61f605ccd807ap-500,
                0x1.61f8db91cc838p-306,
                0x1.62009d3f99926p-368,
                0x1.62009d3f99926p-369,
                0x1.6202598194beep1003,
                0x1.6202598194beep+1004,
                0x1.6202598194beep1005,
                0x1.620cf88129626p-384,
                0x1.6211dd9ce9d71p-519,
                0x1.6212a96f5c6p-688,
                0x1.62139e5d402a9p434,
                0x1.6214e82d0eb18p-415,
                0x1.6214e82d0eb18p-416,
                0x1.62157560864b3p915,
                0x1.621b111dad022p-956,
                0x1.621b1c28ac20cp737,
                0x1.622b4d1570a3ep-193,
                0x1.622b4d1570a3ep-194,
                0x1.622decab98016p+860,
                0x1.6230ced268a14p+136,
                0x1.6230ced268a14p137,
                0x1.623879362821dp343,
                0x1.623879362821dp+344,
                0x1.6238961f14ap-1031,
                0x1.623a3a27fd7f8p-122,
                0x1.623ba072b8cdep-739,
                0x1.623ba072b8cdep-740,
                0x1.6241a2af8ca4cp-379,
                0x1.625572067d666p-593,
                0x1.625572067d666p-594,
                0x1.625572067d666p-595,
                0x1.625572067d666p-596,
                0x1.62569d9ce4dffp-314,
                0x1.625c99a3ad3a8p429,
                0x1.625c99a3ad3a8p430,
                0x1.625f010cb9922p117,
                0x1.625f010cb9922p118,
                0x1.6261aec16f1d8p633,
                0x1.6261cp-1056,
                0x1.62644c61d41aap881,
                0x1.62644c61d41aap882,
                0x1.62644c61d41aap883,
                0x1.62644c61d41aap+884,
                0x1.62644c61d41aap885,
                0x1.62644c61d41aap886,
                0x1.62644c61d41aap887,
                0x1.62644c61d41aap+888,
                0x1.62644c61d41aap889,
                0x1.62644c61d41aap890,
                0x1.62644c61d41aap891,
                0x1.62644c61d41aap+892,
                0x1.62644c61d41aap893,
                0x1.62644c61d41aap894,
                0x1.626f1c480545cp401,
                0x1.626f1c480545cp402,
                0x1.626f1c480545cp403,
                0x1.626f1c480545cp+404,
                0x1.626f1c480545cp405,
                0x1.626f1c480545cp406,
                0x1.626f1c480545cp407,
                0x1.62816bb925473p+576,
                0x1.62816bb925473p577,
                0x1.62816bb925473p578,
                0x1.62884f31e93ffp-914,
                0x1.62884f31e94p-913,
                0x1.62884f31e94p-914,
                0x1.628bdf7d3563cp422,
                0x1.628bdf7d3563cp423,
                0x1.628bdf7d3563cp+424,
                0x1.628bdf7d3563cp425,
                0x1.62937ae150c4bp-760,
                0x1.629b72f368e82p-301,
                0x1.629b72f368e82p-302,
                0x1.62a7875fe1a86p239,
                0x1.62a7875fe1a86p+240,
                0x1.62ab03046b526p-102,
                0x1.62b88ep-1049,
                0x1.62befb1c083b2p-1001,
                0x1.62befb1c083b2p-1002,
                0x1.62c0fb3437b6ep973,
                0x1.62c0fb3437b6ep974,
                0x1.62c3916d3040cp-324,
                0x1.62c78f48ff66cp845,
                0x1.62d2b5b9871a4p410,
                0x1.62d2b5b9871a4p411,
                0x1.62d2b5b9871a4p+412,
                0x1.62d2b5b9871a4p413,
                0x1.62e6d72d6dfbp-426,
                0x1.62e6d72d6dfbp-427,
                0x1.62e6d72d6dfbp-428,
                0x1.62e6d72d6dfbp-429,
                0x1.62e6d72d6dfbp-430,
                0x1.62ea686bfa47cp+908,
                0x1.62eada833c8f5p+268,
                0x1.62f131ed1dcdep-94,
                0x1.62f6761b867ap-950,
                0x1.62f81ca3f2e92p-960,
                0x1.62f81ca3f2e92p-961,
                0x1.62f81ca3f2e92p-962,
                0x1.62f81ca3f2e92p-963,
                0x1.62fba9df27fcdp606,
                0x1.62fba9df27fcdp607,
                0x1.6302e5e03b696p+612,
                0x1.63036bb2590dcp-52,
                0x1.6317e4b43aa51p125,
                0x1.6329bfe32726cp727,
                0x1.6329bfe32726cp+728,
                0x1.632b4634f9074p-660,
                0x1.632c3ffb4d1cap-540,
                0x1.6333c7ce550dap659,
                0x1.6335d802de455p-636,
                0x1.6335d802de455p-637,
                0x1.6335d802de455p-638,
                0x1.633d4e6f5bad4p-793,
                0x1.633d4e6f5bad4p-794,
                0x1.634139fcbd767p-122,
                0x1.6345785d8ap+56,
                0x1.634c2def8b2dcp-768,
                0x1.6350f3e14fb76p749,
                0x1.6350f3e14fb76p750,
                0x1.6353b76671f07p-673,
                0x1.6355557262ff1p+652,
                0x1.6355557262ff1p653,
                0x1.6363c444a1c0fp-283,
                0x1.636584bec7568p579,
                0x1.636691946a278p913,
                0x1.636691946a278p914,
                0x1.636691946a278p915,
                0x1.636691946a278p+916,
                0x1.6379ae73c48ep941,
                0x1.6379ae73c48ep942,
                0x1.6379ae73c48ep943,
                0x1.6381dae63506p-22,
                0x1.6381dae63506p-23,
                0x1.6381dae63506p-24,
                0x1.6384ef2ad17ccp+444,
                0x1.6388ac82b48fp-992,
                0x1.6388ac82b48fp-993,
                0x1.6398c9fcd3252p+348,
                0x1.6398c9fcd3252p349,
                0x1.6398c9fcd3252p350,
                0x1.6398c9fcd3252p351,
                0x1.6398c9fcd3252p+352,
                0x1.63997ac1ae183p543,
                0x1.63a432c8f5cd6p541,
                0x1.63a432c8f5cd6p542,
                0x1.63a9855f9fb4bp+264,
                0x1.63a9d195e1064p-784,
                0x1.63a9d195e1064p-785,
                0x1.63ae2caed4528p+376,
                0x1.63ae2caed4528p377,
                0x1.63ae930ef4469p+876,
                0x1.63b96644e12dbp-306,
                0x1.63bae24a9144ap+168,
                0x1.63bae24a9144ap169,
                0x1.63c531affee9ep+892,
                0x1.63c56cfb5d11cp-244,
                0x1.63c56cfb5d11cp-245,
                0x1.63c56cfb5d11cp-246,
                0x1.63c9p-1057,
                0x1.63cc667244f94p166,
                0x1.63d9cafe1a03cp-153,
                0x1.63e0ad8ed22p-602,
                0x1.63e0ad8ed22p-603,
                0x1.63e0ad8ed22p-604,
                0x1.63e0ad8ed22p-605,
                0x1.63ed3cfb4a946p-688,
                0x1.63ed4a60c9c9p+324,
                0x1.63ef6f560afe7p-351,
                0x1.63ef6f560afe7p-352,
                0x1.64044fcp-1048,
                0x1.6404b5f1623acp-509,
                0x1.6404b5f1623acp-510,
                0x1.6407dcc6096f2p+560,
                0x1.640879bfd94e6p+940,
                0x1.640a62f3a83dep338,
                0x1.640a62f3a83dep339,
                0x1.640a62f3a83dep+340,
                0x1.640a62f3a83dep341,
                0x1.640a62f3a83dep342,
                0x1.640a62f3a83dep343,
                0x1.640a62f3a83dep+344,
                0x1.640a62f3a83dep345,
                0x1.640a62f3a83dep346,
                0x1.640a62f3a83dep347,
                0x1.640a62f3a83dep+348,
                0x1.640a62f3a83dep349,
                0x1.640a62f3a83dep350,
                0x1.6415734abf278p227,
                0x1.6415734abf278p+228,
                0x1.64222b2871d36p1010,
                0x1.6425180b60afdp-498,
                0x1.64268b6e7ap-1032,
                0x1.6426f4cd95dap-308,
                0x1.64291808440f4p-940,
                0x1.642a35f60e736p+996,
                0x1.642a35f60e736p997,
                0x1.642a35f60e736p998,
                0x1.642ba19a8c352p270,
                0x1.64375cc3b446ap-405,
                0x1.643a41de1cecep-834,
                0x1.643a41de1cecep-835,
                0x1.644e04ed4848ep+588,
                0x1.64568acf62f96p275,
                0x1.645ecbd4ea424p747,
                0x1.645ecbd4ea424p+748,
                0x1.64642836a539p+368,
                0x1.6470cff6546b6p-625,
                0x1.647665084bee2p497,
                0x1.647697f925cbdp-884,
                0x1.6483f5b7e7f4ap930,
                0x1.6483f5b7e7f4ap931,
                0x1.648a5f60326a2p-885,
                0x1.648ffb617bd5cp-909,
                0x1.6494732202a5ep-1002,
                0x1.64981e39e66cap-748,
                0x1.64991ac0d260ap-1000,
                0x1.649caba8fb1acp394,
                0x1.649caba8fb1acp395,
                0x1.649caba8fb1acp+396,
                0x1.649caba8fb1acp397,
                0x1.649caba8fb1acp398,
                0x1.64a56f4f45dd9p-670,
                0x1.64a5cf12be114p-505,
                0x1.64ae1f6559f08p-780,
                0x1.64b166995a9b8p-42,
                0x1.64baadda730a2p-672,
                0x1.64bfd7cf2e4dep909,
                0x1.64c1b9d377cddp363,
                0x1.64c37e1e2eed6p-395,
                0x1.64c8a4da900fep-558,
                0x1.64c8a4da900fep-559,
                0x1.64c8a4da900fep-560,
                0x1.64c9ec2a52008p-613,
                0x1.64cdc6850a18ap-650,
                0x1.64cfda3281e38p-140,
                0x1.64cfda3281e39p-140,
                0x1.64d0e4a14ef9cp-415,
                0x1.64d649848325p-391,
                0x1.64da43369b44cp195,
                0x1.64dbf0b2e5bb5p497,
                0x1.64dbf0b2e5bb5p498,
                0x1.64e1300c1100ap-66,
                0x1.64e1300c1100ap-67,
                0x1.64e1300c1100ap-68,
                0x1.64e1300c1100ap-69,
                0x1.64e1300c1100ap-70,
                0x1.64e1300c1100ap-71,
                0x1.64e1300c1100ap-72,
                0x1.64e251a09aaa3p-939,
                0x1.64e4690e47258p-470,
                0x1.64e4ac88758fcp529,
                0x1.64e4ac88758fcp530,
                0x1.64e4ac88758fcp531,
                0x1.64e4ac88758fcp+532,
                0x1.64e5f4b4f5f59p-1011,
                0x1.64ea182849918p902,
                0x1.64eb2a270d859p+616,
                0x1.64ec106d80d26p-949,
                0x1.64f249bb8d559p-261,
                0x1.64f492c292f34p-723,
                0x1.64f492c292f34p-724,
                0x1.64f492c292f34p-725,
                0x1.64f569ff6e276p+520,
                0x1.64f5cff60f1eep-635,
                0x1.64f86cb9cefbp-100,
                0x1.64fcc10aa2198p-173,
                0x1.6508p-1061,
                0x1.6524feda1e325p-953,
                0x1.65280597b8faap838,
                0x1.652efdc6018a2p345,
                0x1.65304c124be1ep-815,
                0x1.65304c124be1ep-816,
                0x1.65308e415a65fp-276,
                0x1.65308e415a65fp-277,
                0x1.653p-1062,
                0x1.6548faf8b223p602,
                0x1.6548faf8b223p603,
                0x1.6548faf8b223p+604,
                0x1.6548faf8b223p605,
                0x1.6548faf8b223p606,
                0x1.654dcee15f8d9p-42,
                0x1.654e9efbbd54p415,
                0x1.655a059dd1042p411,
                0x1.655a98aa02ep-900,
                0x1.655b1837afa2cp-468,
                0x1.655f62947aad8p495,
                0x1.6565b11259008p+1004,
                0x1.656d9e86abf74p-841,
                0x1.656d9e86abf74p-842,
                0x1.656d9e86abf74p-843,
                0x1.657077f156deep831,
                0x1.6570b189aa6d6p-471,
                0x1.6571433006a66p358,
                0x1.6571433006a66p359,
                0x1.6571433006a66p+360,
                0x1.65755201a7cdap653,
                0x1.6578d7ccaf0a2p-211,
                0x1.65857e29b02d6p-919,
                0x1.6585ca1365ab6p851,
                0x1.6585ca1365ab6p+852,
                0x1.65873df49683p214,
                0x1.65873df49683p215,
                0x1.65873df49683p+216,
                0x1.65873df49683p217,
                0x1.6588c69982795p331,
                0x1.658b0fdcfa484p478,
                0x1.658b0fdcfa484p479,
                0x1.658c5a4fba388p-891,
                0x1.658e3ab795204p+828,
                0x1.658e3ab795204p829,
                0x1.658e3ab795204p830,
                0x1.658e3ab795204p831,
                0x1.658e3ab795204p+832,
                0x1.658e3ab795204p833,
                0x1.658e3ab795204p834,
                0x1.658e3ab795204p835,
                0x1.658e3ab795204p+836,
                0x1.658fc73b81b02p-526,
                0x1.658fc73b81b02p-527,
                0x1.65908e904af4p-946,
                0x1.6599e6f7a7811p+592,
                0x1.659a22f59b8eap377,
                0x1.659a22f59b8eap378,
                0x1.659a22f59b8eap379,
                0x1.65a86a2p-1047,
                0x1.65b33e3d9784cp-768,
                0x1.65b527ebfbcfep-773,
                0x1.65b527ebfbcfep-774,
                0x1.65b92ccfef5fp-1026,
                0x1.65c25ae018e34p890,
                0x1.65dad04d534adp+968,
                0x1.65dde8e688ba6p-54,
                0x1.65dde8e688ba6p-55,
                0x1.65dde8e688ba6p-56,
                0x1.65dde8e688ba6p-57,
                0x1.65dde8e688ba6p-58,
                0x1.65e06e1ea62fep-455,
                0x1.65e46d8d8d1b5p722,
                0x1.65e79e4643059p841,
                0x1.65e79e4643059p842,
                0x1.65e8903f8bf84p519,
                0x1.65e8903f8bf84p+520,
                0x1.65f728addc94ap775,
                0x1.65f728addc94ap+776,
                0x1.65fc7e170b33ep-821,
                0x1.66074a0f006dap-28,
                0x1.660a8929fef32p851,
                0x1.660aaacc2dbd4p475,
                0x1.661606e2e0ceap+940,
                0x1.661d47d2d47f3p735,
                0x1.66288f327cf92p141,
                0x1.6637ef9cccf61p-468,
                0x1.665144e7723dap-828,
                0x1.665bc52741b36p-593,
                0x1.665bf1d3e6a8cp-336,
                0x1.665f8bf806e42p1001,
                0x1.665f9b5237886p-576,
                0x1.66675bd915276p-674,
                0x1.66675bd915276p-675,
                0x1.666e37102b4b8p+460,
                0x1.666e37102b4b8p461,
                0x1.666e37102b4b8p462,
                0x1.667a689bd2ec5p753,
                0x1.667afa252e8dcp-199,
                0x1.667afa252e8dcp-200,
                0x1.667afa252e8dcp-201,
                0x1.667afa252e8dcp-202,
                0x1.667afa252e8dcp-203,
                0x1.667afa252e8dcp-204,
                0x1.6684059c1d1c9p-822,
                0x1.668a0c6c20eaep502,
                0x1.668d6cp-1052,
                0x1.668e5db55bde8p-105,
                0x1.668e5db55bde8p-106,
                0x1.668e5db55bde8p-107,
                0x1.668e5db55bde8p-108,
                0x1.668e5db55bde8p-109,
                0x1.6699a39e22cd8p-765,
                0x1.6699a39e22cd8p-766,
                0x1.669db7803c9e4p-798,
                0x1.669db7803c9e4p-799,
                0x1.669edd9196da2p622,
                0x1.669edd9196da2p623,
                0x1.669edd9196da2p+624,
                0x1.669edd9196da2p625,
                0x1.66a00a69c6c34p+224,
                0x1.66a00a69c6c34p225,
                0x1.66af7c5616b98p-812,
                0x1.66bae5f8c55cdp+332,
                0x1.66bb7f0435c9ep149,
                0x1.66bb7f0435c9ep150,
                0x1.66bb7f0435c9ep151,
                0x1.66cb3d4b7ca04p-691,
                0x1.66d1ca38d1822p-256,
                0x1.66d3f4dd3c67cp-299,
                0x1.66e40c2057274p-226,
                0x1.66e4f1756df84p-674,
                0x1.66e4f1756df84p-675,
                0x1.66e4f1756df84p-676,
                0x1.66e4f1756df84p-677,
                0x1.66efa2a7af778p-387,
                0x1.66f8e68fc30d8p+1012,
                0x1.66fee05649a7ap358,
                0x1.66fee05649a7ap359,
                0x1.66fee05649a7ap+360,
                0x1.66fee05649a7ap361,
                0x1.6704b82218c42p-601,
                0x1.6704b82218c42p-602,
                0x1.6704b82218c42p-603,
                0x1.6708547567644p-629,
                0x1.671b25aec1d88p633,
                0x1.671b25aec1d88p634,
                0x1.671b25aec1d88p635,
                0x1.671b25aec1d88p+636,
                0x1.671bdac73c42ep-710,
                0x1.671bdac73c42ep-711,
                0x1.671e0503824ecp-976,
                0x1.67203d5ccc96p426,
                0x1.67203d5ccc96p427,
                0x1.6723dc69a0dcp-455,
                0x1.6723dc69a0dcp-456,
                0x1.67259fd5576f8p969,
                0x1.67259fd5576f8p970,
                0x1.6726fed22dfd1p-487,
                0x1.672ef640e1fdcp743,
                0x1.674acd87a9102p-629,
                0x1.674c5p-1054,
                0x1.67518397c3p-1032,
                0x1.67518397c3p-1033,
                0x1.67567f55b22e2p-24,
                0x1.6757e687dc459p831,
                0x1.675ba8426832ap145,
                0x1.67745b81f03f2p+896,
                0x1.677a6132d2cbcp-334,
                0x1.678533bc3f31cp-634,
                0x1.6788a642709dcp394,
                0x1.6788a642709dcp395,
                0x1.6788a642709dcp+396,
                0x1.6788a642709dcp397,
                0x1.6789e3750f791p-1017,
                0x1.6789e3750f79p-1017,
                0x1.678d80a22868ap+320,
                0x1.678d9bd5cp-1040,
                0x1.678f87a10cf92p526,
                0x1.679c85949b331p-782,
                0x1.67a3c8dc4bc9cp-889,
                0x1.67a3c8dc4bc9cp-890,
                0x1.67a5da743e788p+904,
                0x1.67a8488b396cep-852,
                0x1.67b105bfddec4p786,
                0x1.67b105bfddec4p787,
                0x1.67b1b04f3b068p-389,
                0x1.67b1b04f3b068p-390,
                0x1.67b1b04f3b068p-391,
                0x1.67b1b04f3b068p-392,
                0x1.67b2c55bac4d8p-977,
                0x1.67b2c55bac4d8p-978,
                0x1.67c0ed5965318p757,
                0x1.67c30198f7d5bp-656,
                0x1.67c4023fba672p554,
                0x1.67c78da99ac2p-417,
                0x1.67c78da99ac2p-418,
                0x1.67c78da99ac2p-419,
                0x1.67c78da99ac2p-420,
                0x1.67cf7596968d4p894,
                0x1.67cf7596968d4p895,
                0x1.67cf7596968d4p+896,
                0x1.67d9735144ae3p93,
                0x1.67dce12d75d66p+864,
                0x1.67e1aa7e6aebp499,
                0x1.67e6667f253fap+844,
                0x1.67e6667f253fap845,
                0x1.67e9c127b6e74p-532,
                0x1.67ebbdaa5b07p370,
                0x1.67ed5f4b7f4fep805,
                0x1.67fb96a64f11bp645,
                0x1.6804ec3ca424cp-142,
                0x1.6811f9f8f18c7p-138,
                0x1.6812adf4ea394p-490,
                0x1.6812adf4ea394p-491,
                0x1.6812adf4ea394p-492,
                0x1.681671c70e742p-769,
                0x1.681671c70e742p-770,
                0x1.682660452e792p1015,
                0x1.6830929d739d7p+252,
                0x1.683786b3b9dfp-653,
                0x1.683f632df77b4p-1001,
                0x1.683f632df77b4p-1002,
                0x1.683f632df77b4p-1003,
                0x1.683f632df77b4p-1004,
                0x1.683f632df77b4p-1005,
                0x1.683f632df77b4p-1006,
                0x1.683f632df77b4p-1007,
                0x1.684404ddcbfp+852,
                0x1.684404ddcbfp853,
                0x1.684404ddcbfp854,
                0x1.684404ddcbfp855,
                0x1.6849b86a12b9bp-47,
                0x1.6849b86a12b9bp-48,
                0x1.6849b86a12b9bp-49,
                0x1.6851264b146cep431,
                0x1.6851264b146cep+432,
                0x1.6851264b146cep433,
                0x1.6851264b146cep434,
                0x1.6851938d1d6p-1030,
                0x1.685f7683c20cdp-363,
                0x1.68632e03b013fp435,
                0x1.68638801b6886p610,
                0x1.68638801b6886p611,
                0x1.68638801b6886p+612,
                0x1.6866845b45d44p-209,
                0x1.6866845b45d44p-210,
                0x1.68674e88d802cp-994,
                0x1.68674e88d802cp-995,
                0x1.686c84ecf2483p-958,
                0x1.686f435fe6b6cp-61,
                0x1.686f435fe6b6cp-62,
                0x1.6872f757a1093p-684,
                0x1.68757bbea482cp-326,
                0x1.687daec7ffd32p-608,
                0x1.687fe0e547d02p258,
                0x1.6880c6b31d8ccp-715,
                0x1.688bdf708a1d8p-217,
                0x1.688e746a1a72p+700,
                0x1.688e746a1a72p701,
                0x1.688e746a1a72p702,
                0x1.688e746a1a72p703,
                0x1.6891b9fbcb534p-518,
                0x1.68a0e924542dcp+756,
                0x1.68a0e924542dcp757,
                0x1.68a0e924542dcp758,
                0x1.68a7a5b7b3fdbp+1000,
                0x1.68a7a5b7b3fdbp1001,
                0x1.68a8a2355d381p483,
                0x1.68a8a2355d381p+484,
                0x1.68a9c942f3ba3p438,
                0x1.68ad395434c3ep+372,
                0x1.68af802294925p-889,
                0x1.68b690d9b8235p-812,
                0x1.68b690d9b8235p-813,
                0x1.68b80990297ddp-592,
                0x1.68b8e03b2aecbp-231,
                0x1.68bc9744ef621p-250,
                0x1.68bd95e904898p-638,
                0x1.68bd95e904898p-639,
                0x1.68c013f9b8f8p+976,
                0x1.68c013f9b8f8p977,
                0x1.68c899a96d8e7p903,
                0x1.68d1ffa0777c8p537,
                0x1.68d6d77360508p-276,
                0x1.68dd4a4a6d73ap-390,
                0x1.68dfc77c5d9p-143,
                0x1.68e6f997356dep475,
                0x1.68e85dd024402p-491,
                0x1.68f08add9cedep+736,
                0x1.68f7532bf067cp-887,
                0x1.68f7532bf067cp-888,
                0x1.68fb1dcb2ep-1035,
                0x1.68fe2af35a862p-532,
                0x1.6904c405d5e5ap-515,
                0x1.6904c405d5e5ap-516,
                0x1.6909eaa3f3486p-383,
                0x1.6909f3b92c83dp923,
                0x1.690e7c1eb7cc3p+784,
                0x1.690e7c1eb7cc3p785,
                0x1.6915a65676986p801,
                0x1.6915a65676986p802,
                0x1.6917bcd01d456p-556,
                0x1.6917bcd01d456p-557,
                0x1.6918b1f994016p-625,
                0x1.6918da1b72a8ep-762,
                0x1.691ca32818ed4p+124,
                0x1.691ca32818ed4p125,
                0x1.69219258236aep830,
                0x1.69239f38fb691p-462,
                0x1.69239f38fb691p-463,
                0x1.69260725361cfp-574,
                0x1.69284b76f5bbap982,
                0x1.69284b76f5bbap983,
                0x1.69429222e15cp-536,
                0x1.69441496866eap663,
                0x1.6948c8517929fp-482,
                0x1.694a9636aba1fp206,
                0x1.694c9b7a09414p-838,
                0x1.69514112d650cp-897,
                0x1.6953eb06e4a1ep-471,
                0x1.69568814cfcb3p-29,
                0x1.69593f03dba8ap415,
                0x1.69630a9521552p-1009,
                0x1.69630a9521552p-1010,
                0x1.69630a9521552p-1011,
                0x1.696786cffe22ap-209,
                0x1.69696f2d9ee36p541,
                0x1.696a31a0dabc1p-911,
                0x1.696a31a0dabc1p-912,
                0x1.6976912c0a40ep-169,
                0x1.69794a160cb58p-728,
                0x1.697a7e6ac4c82p-84,
                0x1.697b361f89299p-916,
                0x1.697cec3deda49p609,
                0x1.697fe29386f96p923,
                0x1.6989a197b27fep+832,
                0x1.6989a197b27fep833,
                0x1.6989a197b27fep834,
                0x1.6989a197b27fep835,
                0x1.698ccdc60015ap-107,
                0x1.698ccdc60015ap-108,
                0x1.698ccdc60015ap-109,
                0x1.698ccdc60015ap-110,
                0x1.698ccdc60015ap-111,
                0x1.698ccdc60015ap-112,
                0x1.698ccdc60015ap-113,
                0x1.698ccdc60015ap-114,
                0x1.698ccdc60015ap-115,
                0x1.698ccdc60015ap-116,
                0x1.698ccdc60015ap-117,
                0x1.698ccdc60015ap-118,
                0x1.698ccdc60015ap-119,
                0x1.698ccdc60015ap-120,
                0x1.69a270307f924p-801,
                0x1.69a270307f924p-802,
                0x1.69bb3768dd5aep-891,
                0x1.69bb3768dd5aep-892,
                0x1.69bcb27c77d58p-116,
                0x1.69be71893e849p+368,
                0x1.69c1007bf7101p-745,
                0x1.69d0e73101615p-923,
                0x1.69d7e84d9bfaep302,
                0x1.69d9abe034955p-243,
                0x1.69d9abe034955p-244,
                0x1.69ddc0f86b71ap+968,
                0x1.69ddc0f86b71ap969,
                0x1.69ddc0f86b71ap970,
                0x1.69ddc0f86b71ap971,
                0x1.69ddc0f86b71ap+972,
                0x1.69e31f39687e5p-581,
                0x1.69ed4c3fa9f0ep-730,
                0x1.69fce21c46c92p-261,
                0x1.69fce21c46c92p-262,
                0x1.69p-1066,
                0x1.6a17f7778cb42p-407,
                0x1.6a1d89b8c304cp-553,
                0x1.6a1da09cc24d9p626,
                0x1.6a250102b41d6p306,
                0x1.6a250102b41d6p307,
                0x1.6a250102b41d6p+308,
                0x1.6a250102b41d6p309,
                0x1.6a35392e2c91p-536,
                0x1.6a3a275d49491p242,
                0x1.6a3c3bb6a4529p-242,
                0x1.6a3c3bb6a4529p-243,
                0x1.6a3c41cd3368p-1029,
                0x1.6a3efd21fb96ap182,
                0x1.6a3f61edd7a86p+272,
                0x1.6a3f61edd7a86p273,
                0x1.6a47d4e7ec632p+300,
                0x1.6a4d8a0042b22p-662,
                0x1.6a50bc3c42b12p-991,
                0x1.6a635786c9576p282,
                0x1.6a635786c9576p283,
                0x1.6a635786c9576p+284,
                0x1.6a70e7657329bp-182,
                0x1.6a70e7657329bp-183,
                0x1.6a77bba221186p-770,
                0x1.6a77bba221186p-771,
                0x1.6a77bba221186p-772,
                0x1.6a829855d1b6ep-154,
                0x1.6a87a1456b3d1p-1020,
                0x1.6a90288fd752ap455,
                0x1.6a90288fd752ap+456,
                0x1.6a938ad510e46p-772,
                0x1.6a938ad510e46p-773,
                0x1.6a938ad510e46p-774,
                0x1.6a938ad510e46p-775,
                0x1.6a938ad510e46p-776,
                0x1.6a9abc9424febp727,
                0x1.6aa2ce8e47831p334,
                0x1.6aa61b0bd9b0cp-372,
                0x1.6aa6a2c276691p-409,
                0x1.6aab6b6905827p541,
                0x1.6aace2094131dp242,
                0x1.6ab097e8e02fap794,
                0x1.6ab097e8e02fap795,
                0x1.6ab42a013719p+832,
                0x1.6abad3d08924ap+292,
                0x1.6abad3d08924ap293,
                0x1.6ac3768c0d1bp+684,
                0x1.6ac3b9bd19904p+352,
                0x1.6accbfbf218ap+380,
                0x1.6accbfbf218ap381,
                0x1.6ad5063014b59p-938,
                0x1.6ad737c7e8858p-358,
                0x1.6ae51b0e16a58p-177,
                0x1.6ae51b0e16a58p-178,
                0x1.6ae805d694c86p246,
                0x1.6af237b3ed8eep894,
                0x1.6ap-1059,
                0x1.6ap-1060,
                0x1.6ap-1061,
                0x1.6b0a8e891ffffp-924,
                0x1.6b0a8e892p-923,
                0x1.6b0a8e892p-924,
                0x1.6b0b2ed2f3ce6p-977,
                0x1.6b0e34b98ebc2p413,
                0x1.6b100b09ecaf2p-639,
                0x1.6b100b09ecaf2p-640,
                0x1.6b100e18e5c18p-766,
                0x1.6b100e18e5c18p-767,
                0x1.6b100e18e5c18p-768,
                0x1.6b100e18e5c18p-769,
                0x1.6b20c2458ba2cp523,
                0x1.6b20c2458ba2cp+524,
                0x1.6b20c2458ba2cp525,
                0x1.6b20c2458ba2cp526,
                0x1.6b210327f4071p-266,
                0x1.6b35cb49d0ddcp+960,
                0x1.6b35cb49d0ddcp961,
                0x1.6b366467a1925p-366,
                0x1.6b3cbc1048fcap-370,
                0x1.6b3f4478ef4b2p-690,
                0x1.6b42832bc3b82p-723,
                0x1.6b492p-1055,
                0x1.6b4b6c9ec806p+432,
                0x1.6b4d3213fae58p-883,
                0x1.6b4d3213fae58p-884,
                0x1.6b5c37544816p-768,
                0x1.6b6b5b5155ffp-439,
                0x1.6b6bddb0b85b5p-532,
                0x1.6b6fdc1c4b968p-924,
                0x1.6b739292cc6cbp+760,
                0x1.6b82ccec56bbcp679,
                0x1.6b82ccec56bbcp+680,
                0x1.6b90cf7d32472p821,
                0x1.6ba0c5957bf1ep-881,
                0x1.6ba460f42ab4p-964,
                0x1.6ba56a8834168p-14,
                0x1.6ba84efd94208p213,
                0x1.6ba8608faa6a8p-769,
                0x1.6ba8608faa6a8p-770,
                0x1.6bbf6ccd542b5p-407,
                0x1.6bc06c64a399p-653,
                0x1.6bc06d67b8eccp-354,
                0x1.6bcc41e9p46,
                0x1.6bd71899ffefbp166,
                0x1.6bde99deb93fp-837,
                0x1.6bdf15e5d2374p-753,
                0x1.6be3e115ebf18p+428,
                0x1.6be3e115ebf18p429,
                0x1.6be3e115ebf18p430,
                0x1.6be8643fd2edcp873,
                0x1.6be88eaad73fp-653,
                0x1.6be9058b1b09bp-763,
                0x1.6be9058b1b09bp-764,
                0x1.6bec3ad347b5ep-835,
                0x1.6bf7a5f444945p790,
                0x1.6bfa6e2e2611ap-603,
                0x1.6c07d87cp-1044,
                0x1.6c09e88a42ccdp-1022,
                0x1.6c11a7c9e58dfp-232,
                0x1.6c18a4ee0ec6dp-305,
                0x1.6c1966f18308dp382,
                0x1.6c19cc49f7eaep-433,
                0x1.6c19cc49f7eaep-434,
                0x1.6c19cc49f7eaep-435,
                0x1.6c20c00def739p-515,
                0x1.6c2cc8fd484b3p-795,
                0x1.6c2d4256ffcc2p531,
                0x1.6c2d4256ffcc3p530,
                0x1.6c2d4256ffcc3p531,
                0x1.6c309024bab4bp+288,
                0x1.6c309024bab4bp289,
                0x1.6c309024bab4bp290,
                0x1.6c362a3dc506fp-113,
                0x1.6c37e25d23192p866,
                0x1.6c3832d126ff6p+532,
                0x1.6c408dca27b3p-757,
                0x1.6c42ade232ff4p823,
                0x1.6c46f3972904dp-967,
                0x1.6c46f3972904dp-968,
                0x1.6c46f3972904dp-969,
                0x1.6c4de344489c2p+808,
                0x1.6c4fd93f25f2dp-892,
                0x1.6c74a426fa60cp-371,
                0x1.6c7acb6de725fp-427,
                0x1.6c7ae7ecd6d78p737,
                0x1.6c7c558d0fdcep431,
                0x1.6c7eafe6c9f3p-939,
                0x1.6c7eafe6c9f3p-940,
                0x1.6c8dd86891acap-69,
        };
    }

    static double[] hard2() {
        return new double[] {
                0x1.6c8dd86891acap-70,
                0x1.6c8dd86891acap-71,
                0x1.6c8dd86891acap-72,
                0x1.6c8e5ca239028p+1016,
                0x1.6c8e5ca239029p+1016,
                0x1.6c9693a2ab21bp489,
                0x1.6c99537666142p625,
                0x1.6c9e14b7c22c4p703,
                0x1.6c9e14b7c22c4p+704,
                0x1.6c9e14b7c22c4p705,
                0x1.6c9e14b7c22c4p706,
                0x1.6c9e14b7c22c4p707,
                0x1.6c9e254d33b6ep+244,
                0x1.6c9e254d33b6ep245,
                0x1.6ca642213a95fp-443,
                0x1.6ca80027a0accp-748,
                0x1.6cadf794bc86cp+220,
                0x1.6cadf794bc86cp221,
                0x1.6cb0c6372349ep255,
                0x1.6cb0c6372349ep+256,
                0x1.6cb0c6372349ep257,
                0x1.6cb0c6372349ep258,
                0x1.6cb67ce362c4p985,
                0x1.6cb67ce362c4p986,
                0x1.6cbfb47bf0afcp205,
                0x1.6cc62365e4d56p-35,
                0x1.6cc88fc8a1d2ap+432,
                0x1.6cc88fc8a1d2ap433,
                0x1.6cd15f3e9fe2ep-400,
                0x1.6cd35e9946544p-110,
                0x1.6cdc8b1d1175ep-998,
                0x1.6cdc8b1d1175ep-999,
                0x1.6ce0fe1e65d5ep647,
                0x1.6ce584e9fa7edp-687,
                0x1.6ce94febdc7a5p197,
                0x1.6ce94febdc7a5p198,
                0x1.6ce94febdc7a5p199,
                0x1.6ce94febdc7a5p+200,
                0x1.6ced3d56a463cp-254,
                0x1.6ced3d56a463cp-255,
                0x1.6ced3d56a463cp-256,
                0x1.6cf2b01f2f00ap946,
                0x1.6cfc8601948bcp-114,
                0x1.6cfec8aa52598p-635,
                0x1.6cp-1068,
                0x1.6d0e30427ea66p+900,
                0x1.6d127d05394fep-22,
                0x1.6d127d05394fep-23,
                0x1.6d14ca0433c86p430,
                0x1.6d14ca0433c86p431,
                0x1.6d14ca0433c86p+432,
                0x1.6d1879144d84ap-73,
                0x1.6d229d4a06692p-992,
                0x1.6d24299b76b64p471,
                0x1.6d255f9703514p-83,
                0x1.6d2e8d2d1e6cap-915,
                0x1.6d2e8d2d1e6cap-916,
                0x1.6d2e8d2d1e6cap-917,
                0x1.6d2e8d2d1e6cap-918,
                0x1.6d2e8d2d1e6cap-919,
                0x1.6d2e8d2d1e6cap-920,
                0x1.6d350d631074p-515,
                0x1.6d360b06631bep+156,
                0x1.6d43cbe81445bp-697,
                0x1.6d475aa413588p653,
                0x1.6d4d415ce84dep+1000,
                0x1.6d4d415ce84dep1001,
                0x1.6d58e218c028ap-254,
                0x1.6d5acc966154fp629,
                0x1.6d5ce4104a11ap301,
                0x1.6d601ad376ab9p-150,
                0x1.6d65a465d3d6fp-82,
                0x1.6d660e23e487ap379,
                0x1.6d660e23e487ap+380,
                0x1.6d660e23e487ap381,
                0x1.6d660e23e487ap382,
                0x1.6d660e23e487ap383,
                0x1.6d660e23e487ap+384,
                0x1.6d660e23e487ap385,
                0x1.6d6c23da854efp973,
                0x1.6d6c45bbee09p-603,
                0x1.6d7303b6e4024p-949,
                0x1.6d8d46c516abep926,
                0x1.6d8e4deae5726p-994,
                0x1.6d8fa6fba7732p-511,
                0x1.6d996bda852c4p961,
                0x1.6d996bda852c4p962,
                0x1.6d99cd229a79ap+188,
                0x1.6d99cd229a79ap189,
                0x1.6d99cd229a79ap190,
                0x1.6d99cd229a79ap191,
                0x1.6d99cd229a79ap+192,
                0x1.6d99cd229a79ap193,
                0x1.6d99cd229a79ap194,
                0x1.6d99cd229a79ap195,
                0x1.6d99cd229a79ap+196,
                0x1.6d9a093e7e062p990,
                0x1.6d9a093e7e062p991,
                0x1.6d9f521bdf66p441,
                0x1.6da2b6e9abf05p461,
                0x1.6da4892d6ec4p-412,
                0x1.6da4892d6ec4p-413,
                0x1.6da4892d6ec4p-414,
                0x1.6da4892d6ec4p-415,
                0x1.6da4892d6ec4p-416,
                0x1.6da628de3fddp-770,
                0x1.6dae1c0c46d27p-634,
                0x1.6db3e4a6d19dcp478,
                0x1.6db99f3de43e6p-946,
                0x1.6db99f3de43e6p-947,
                0x1.6db99f3de43e6p-948,
                0x1.6dc186ef9f45cp335,
                0x1.6dca59d03582p-202,
                0x1.6dca59d03582p-203,
                0x1.6dca59d03582p-204,
                0x1.6dcbceab9d102p838,
                0x1.6dd022af593aep-341,
                0x1.6dd1ac037266ap445,
                0x1.6dd563b69759cp+964,
                0x1.6dd563b69759cp965,
                0x1.6dd6db306f14ap638,
                0x1.6dd885beb2134p-896,
                0x1.6ddb450e73129p897,
                0x1.6dde1ab72eac4p+412,
                0x1.6de020754cb4ap-1014,
                0x1.6de262e569d42p278,
                0x1.6de2eea43a6f2p-447,
                0x1.6de8b0a498b44p-384,
                0x1.6df082544fcfp+396,
                0x1.6df3p-1057,
                0x1.6df4b236ec0d9p-605,
                0x1.6df59e4154d18p+544,
                0x1.6df59e4154d18p545,
                0x1.6df88f3f3500bp+728,
                0x1.6df8dce42674dp-523,
                0x1.6df8dce42674dp-524,
                0x1.6e00747f54d3ap678,
                0x1.6e00eedb8c8bbp819,
                0x1.6e07bf742b12cp-812,
                0x1.6e07bf742b12cp-813,
                0x1.6e09aa6abd0ecp-766,
                0x1.6e09aa6abd0ecp-767,
                0x1.6e09aa6abd0ecp-768,
                0x1.6e09aa6abd0ecp-769,
                0x1.6e0c94774af95p317,
                0x1.6e0cd275fa00ep-40,
                0x1.6e0fa1f5a7ca5p-995,
                0x1.6e17ae0bbef8cp-878,
                0x1.6e17ae0bbef8cp-879,
                0x1.6e17ae0bbef8cp-880,
                0x1.6e17ae0bbef8cp-881,
                0x1.6e212c473e0cep774,
                0x1.6e22db4568793p-246,
                0x1.6e230d05b76cdp+820,
                0x1.6e230d05b76cep+820,
                0x1.6e26e8fd59f3cp-364,
                0x1.6e3cb6defb168p187,
                0x1.6e42d02eb0438p-618,
                0x1.6e4406f32dcddp143,
                0x1.6e4406f32dcddp+144,
                0x1.6e446862f8464p929,
                0x1.6e45b2f27b9f4p430,
                0x1.6e45b2f27b9f4p431,
                0x1.6e45b2f27b9f4p+432,
                0x1.6e48951ec6897p-87,
                0x1.6e4b3bc19df2ep238,
                0x1.6e4e34e354061p+936,
                0x1.6e4e34e354061p937,
                0x1.6e50adf81d4c8p866,
                0x1.6e50adf81d4c8p867,
                0x1.6e50adf81d4c8p+868,
                0x1.6e57204720e32p+780,
                0x1.6e5d4fa2da77ep-320,
                0x1.6e5d55df3fb3ep423,
                0x1.6e5de7ff45a0cp-845,
                0x1.6e617c0eccee4p+564,
                0x1.6e617c0eccee4p565,
                0x1.6e617c0eccee4p566,
                0x1.6e6b056f7f3p+580,
                0x1.6e6b056f7f3p581,
                0x1.6e6d113aabdfcp-217,
                0x1.6e70cc7f234d2p-279,
                0x1.6e70cc7f234d2p-280,
                0x1.6e70cc7f234d2p-281,
                0x1.6e811bd5b5908p-883,
                0x1.6e811bd5b5908p-884,
                0x1.6e93f5da2824cp-830,
                0x1.6e93f5da2824cp-831,
                0x1.6e9952aa36bd7p-960,
                0x1.6ea99ccb6f32p105,
                0x1.6ea99ccb6f32p106,
                0x1.6eb18493d5d6cp721,
                0x1.6eb18493d5d6cp722,
                0x1.6ec9d55171ce5p-840,
                0x1.6ecdc2a58fc65p353,
                0x1.6ecdc2a58fc65p354,
                0x1.6ecdc2a58fc65p355,
                0x1.6ed79c3030542p238,
                0x1.6ed79c3030542p239,
                0x1.6edcb4c4e161cp-983,
                0x1.6ee73b7b47f43p-930,
                0x1.6ee73b7b47f43p-931,
                0x1.6ee8e00b557e6p-320,
                0x1.6ee9a3db22fc5p-809,
                0x1.6eec7f721c699p+792,
                0x1.6ef2247dd9e11p478,
                0x1.6ef425fa9d15ep974,
                0x1.6ef5b40c2fc77p-346,
                0x1.6ef5b40c2fc78p-343,
                0x1.6ef5b40c2fc78p-344,
                0x1.6ef5b40c2fc78p-345,
                0x1.6ef5b40c2fc78p-346,
                0x1.6ef5b40c2fc78p-347,
                0x1.6ef7dd43b9d0ap-761,
                0x1.6f02089fa0f8fp-77,
                0x1.6f02bee68670dp+480,
                0x1.6f02bee68670dp481,
                0x1.6f09e5bb5f476p-376,
                0x1.6f09e5bb5f476p-377,
                0x1.6f136951b1624p634,
                0x1.6f178b128d6dep154,
                0x1.6f1fbc37efe19p-712,
                0x1.6f20a732aafb6p215,
                0x1.6f20a732aafb6p+216,
                0x1.6f26eb94e4ce8p-583,
                0x1.6f2aad4094c1ep265,
                0x1.6f2d67f9a3ea4p-72,
                0x1.6f2d67f9a3ea4p-73,
                0x1.6f3173215394ap994,
                0x1.6f3963998faeap542,
                0x1.6f3a4f584660ep-773,
                0x1.6f492ca4696d1p-266,
                0x1.6f578c4e0a06p139,
                0x1.6f6afde45fa7p319,
                0x1.6f6e645cd37p-1028,
                0x1.6f709c688aa5ap294,
                0x1.6f8111fca2e74p-236,
                0x1.6f854ac8270dcp-511,
                0x1.6f854ac8270dcp-512,
                0x1.6f854ac8270dcp-513,
                0x1.6f8d7d3dd798ap-500,
                0x1.6f8ef791dc758p+936,
                0x1.6f8ef791dc758p937,
                0x1.6fa09386c702ep-190,
                0x1.6fa09386c702ep-191,
                0x1.6faeb24042228p647,
                0x1.6faf99c361762p1017,
                0x1.6faf99c361762p1018,
                0x1.6fb94f9510662p633,
                0x1.6fb94f9510662p634,
                0x1.6fb94f9510662p635,
                0x1.6fb97ea6a9d38p+624,
                0x1.6fbeb59e83acp+416,
                0x1.6fc0a580ab19p689,
                0x1.6fc2d61c556bdp434,
                0x1.6fcc86a9f45bap-175,
                0x1.6fcf1e819158ap+932,
                0x1.6fd4d575270d2p+256,
                0x1.6fd91caca719cp-949,
                0x1.6fd91caca719cp-950,
                0x1.6fdcp-1057,
                0x1.6fe34ed6cec3p663,
                0x1.6fe34ed6cec3p+664,
                0x1.6fe67d639ee3p-411,
                0x1.6fedd9c7fd4c4p-491,
                0x1.6fedd9c7fd4c4p-492,
                0x1.6fedd9c7fd4c4p-493,
                0x1.6fedd9c7fd4c4p-494,
                0x1.6ff0327ba5e84p887,
                0x1.6ff0327ba5e84p+888,
                0x1.6ff187f95eb68p170,
                0x1.6ff406c702454p763,
                0x1.6ff497b2ac10dp-661,
                0x1.6ffb9e4524414p-565,
                0x1.6ffb9e4524414p-566,
                0x1.6ffd85d36be97p-342,
                0x1.6ffd85d36be97p-343,
                0x1.6ffd85d36be97p-344,
                0x1.6fff5c88a7856p379,
                0x1.6p-1069,
                0x1.6p-1071,
                0x1.70094d570f7c8p227,
                0x1.700a7114fce42p-747,
                0x1.700a7114fce42p-748,
                0x1.70105df3d47cbp149,
                0x1.70105df3d47cbp150,
                0x1.70105df3d47cbp151,
                0x1.70105df3d47cbp+152,
                0x1.7014ed387c473p+704,
                0x1.7024c96ca3ce4p517,
                0x1.7024c96ca3ce4p518,
                0x1.70284c2ced78ap-294,
                0x1.702ae4d1fb5cp-1027,
                0x1.702ae4d1fb5d4p-1024,
                0x1.702ae4d1fb5dp-1026,
                0x1.702ae4d1fb5ep-1027,
                0x1.702e98335c942p310,
                0x1.702e98335c942p311,
                0x1.7035a2a6ac9a1p-692,
                0x1.703856844bdbep-47,
                0x1.703856844bdbep-48,
                0x1.703856844bdbep-49,
                0x1.703856844bdbep-50,
                0x1.703856844bdbep-51,
                0x1.703856844bdbep-52,
                0x1.703856844bdbep-53,
                0x1.703856844bdbep-54,
                0x1.703856844bdbep-55,
                0x1.70403ff41ca12p670,
                0x1.704204bbfb2f4p-917,
                0x1.704824a974d99p-958,
                0x1.70507036a9024p218,
                0x1.7051986dbd401p-713,
                0x1.70538e3ff7d6dp+288,
                0x1.70628606586dap878,
                0x1.7062978a32c5fp+716,
                0x1.7065ccf697d76p-1023,
                0x1.7066934492b8ap-570,
                0x1.7068b12497302p343,
                0x1.706feccf0d2afp-799,
                0x1.707cbe50acbp+948,
                0x1.707cbe50acbp949,
                0x1.707eb8aa6ec64p282,
                0x1.707eb8aa6ec64p283,
                0x1.707f8671088a5p-509,
                0x1.7088e901814f6p-454,
                0x1.7088e901814f6p-455,
                0x1.708b1eb0ee29ap-673,
                0x1.708b1eb0ee29ap-674,
                0x1.708b1eb0ee29ap-675,
                0x1.708b1eb0ee29ap-676,
                0x1.708b1eb0ee29ap-677,
                0x1.708b1eb0ee29ap-678,
                0x1.708b1eb0ee29ap-679,
                0x1.708b1eb0ee29ap-680,
                0x1.708d0f84d3de7p-542,
                0x1.708d0f84d3de8p-535,
                0x1.708d0f84d3de8p-536,
                0x1.708d0f84d3de8p-537,
                0x1.708d0f84d3de8p-538,
                0x1.708d0f84d3de8p-539,
                0x1.708d0f84d3de8p-540,
                0x1.708d0f84d3de8p-541,
                0x1.708d0f84d3de8p-542,
                0x1.708d0f84d3de8p-543,
                0x1.708d0f84d3de8p-544,
                0x1.708d0f84d3de8p-545,
                0x1.708p-1064,
                0x1.7090aa6e99ad6p699,
                0x1.7090aa6e99ad6p+700,
                0x1.7097d036b96ebp-207,
                0x1.7099183b5b942p+432,
                0x1.709c74b07b9d6p-482,
                0x1.70a784cf0b4dp427,
                0x1.70aa8f4bc1d83p617,
                0x1.70b6eb2e63fddp-803,
                0x1.70b6eb2e63fddp-804,
                0x1.70b86386ea064p-488,
                0x1.70c94544855a9p689,
                0x1.70cc285f2d20ap-808,
                0x1.70cc285f2d20ap-809,
                0x1.70d1745dcd1e2p-1010,
                0x1.70d1745dcd1e2p-1011,
                0x1.70d1745dcd1e2p-1012,
                0x1.70d1745dcd1e2p-1013,
                0x1.70d1745dcd1e2p-1014,
                0x1.70dda0a3a77aap+644,
                0x1.70dda0a3a77aap645,
                0x1.70e319a5b4517p-232,
                0x1.70ec44b0dd104p989,
                0x1.70ef54646d496p-54,
                0x1.70ef54646d496p-55,
                0x1.70ef54646d496p-56,
                0x1.70ef54646d496p-57,
                0x1.70ef54646d496p-58,
                0x1.70f4d8d6e3f4cp-303,
                0x1.70fa86bc9f2c4p+972,
                0x1.70fa86bc9f2c4p973,
                0x1.70fa86bc9f2c4p974,
                0x1.70fa86bc9f2c4p975,
                0x1.71026c70befeep-855,
                0x1.71026c70befeep-856,
                0x1.710621fcbbdc4p481,
                0x1.710b7a2ef18b8p-629,
                0x1.7123726f69592p-68,
                0x1.7127e05b09cdap638,
                0x1.7127e05b09cdap639,
                0x1.7127e05b09cdap+640,
                0x1.7127e05b09cdap641,
                0x1.7127e05b09cdap642,
                0x1.7127e05b09cdap643,
                0x1.7127e05b09cdap+644,
                0x1.7129b8fe527acp-829,
                0x1.7129b8fe527acp-830,
                0x1.7129b8fe527acp-831,
                0x1.7129b8fe527acp-832,
                0x1.712ed621fda44p-93,
                0x1.713365af58277p+120,
                0x1.7135979c65121p-642,
                0x1.7135979c65121p-643,
                0x1.713b145958bep-272,
                0x1.713d481a48034p813,
                0x1.7143ec73378ep-948,
                0x1.7143ec73378ep-949,
                0x1.71454bf079998p990,
                0x1.71454bf079998p991,
                0x1.7149307be8e02p-406,
                0x1.714a3afe55cb5p-505,
                0x1.714c4bac38feap537,
                0x1.714c4bac38feap538,
                0x1.7151b377c247ep426,
                0x1.7151b377c247ep427,
                0x1.7151b377c247ep+428,
                0x1.715541d6746a6p1023,
                0x1.7168p-1060,
                0x1.7168p-1061,
                0x1.716abb4bdb90ep-999,
                0x1.71760b3c0bc14p287,
                0x1.71760b3c0bc14p+288,
                0x1.717725f822a36p-369,
                0x1.717ba576aa169p167,
                0x1.717bb2127bec6p+468,
                0x1.7186cf76319b4p+404,
                0x1.718bad7c8796ep-458,
                0x1.719105576565fp622,
                0x1.71942837ff59ap+344,
                0x1.7198c86f5ebfbp285,
                0x1.71a0510b88p-1037,
                0x1.71ade19a25ea9p-398,
                0x1.71b01bd52fb1cp589,
                0x1.71b1cac3b3c84p-312,
                0x1.71b30a53ccf8fp-17,
                0x1.71b42cc5cf601p913,
                0x1.71b42cc5cf602p913,
                0x1.71bb51ae0d0e8p559,
                0x1.71bb51ae0d0e8p+560,
                0x1.71bb51ae0d0e8p561,
                0x1.71c362ef64adep-788,
                0x1.71c51462c745p935,
                0x1.71c51462c745p+936,
                0x1.71c74f0225d02p114,
                0x1.71cc364c5bfbap-623,
                0x1.71d6f198cf306p-665,
                0x1.71ddf7605bcb6p538,
                0x1.71e05a1efe658p698,
                0x1.71e30ab10ab3ap-135,
                0x1.71e30ab10ab3ap-136,
                0x1.71e30ab10ab3ap-137,
                0x1.71e72d36c1c96p-712,
                0x1.71eb1a739e7acp697,
                0x1.71ec6760446c9p613,
                0x1.7206bfc4ccabep-230,
                0x1.7206bfc4ccabep-231,
                0x1.721267424c96p-566,
                0x1.721267424c96p-567,
                0x1.7215099871088p-298,
                0x1.7216e054baafap105,
                0x1.721785aa67232p+572,
                0x1.721785aa67232p573,
                0x1.721cf1a6f9c93p-933,
                0x1.721cf1a6f9c93p-934,
                0x1.721cf1a6f9c93p-935,
                0x1.7221a79cdd1d8p347,
                0x1.7221a79cdd1d8p+348,
                0x1.7221a79cdd1d8p349,
                0x1.7221a79cdd1d8p350,
                0x1.7221a79cdd1d8p351,
                0x1.7221a79cdd1d8p+352,
                0x1.7223136cf99a8p945,
                0x1.72262f3133edp-738,
                0x1.72285e7ca1a22p-742,
                0x1.72369916132a9p-492,
                0x1.72369916132a9p-493,
                0x1.723a61ce5af56p-641,
                0x1.723d10001c8bcp-200,
                0x1.723e221b86629p+620,
                0x1.723fed7b523p-1030,
                0x1.7244de75fdd33p-861,
                0x1.7246f653ba877p-668,
                0x1.7247ae95cb544p-850,
                0x1.7247ae95cb544p-851,
                0x1.72496296e4ac2p-863,
                0x1.72496296e4ac2p-864,
                0x1.724f48275f17cp-459,
                0x1.725fd8d75c9f2p590,
                0x1.725fd8d75c9f2p591,
                0x1.725fd8d75c9f2p+592,
                0x1.725fd8d75c9f2p593,
                0x1.725fd8d75c9f2p594,
                0x1.726515fd1bbadp-142,
                0x1.7265d55a05021p-329,
                0x1.7272f3dc60edap-836,
                0x1.7272f3dc60edap-837,
                0x1.72852443155aep826,
                0x1.7286f4a6ec534p-256,
                0x1.7286f4a6ec534p-257,
                0x1.72878d82dfd6p+632,
                0x1.7288e1271f513p-253,
                0x1.7288e1271f514p-250,
                0x1.7288e1271f514p-251,
                0x1.7288e1271f514p-252,
                0x1.7288e1271f514p-253,
                0x1.7288e1271f514p-254,
                0x1.728b35fe80cd8p161,
                0x1.728b35fe80cd8p162,
                0x1.7296f445e597ap-212,
                0x1.72a23f9197193p-956,
                0x1.72ac1fb748445p-168,
                0x1.72b2d8344eb92p275,
                0x1.72b2d8344eb92p+276,
                0x1.72b2d8344eb92p277,
                0x1.72b2d8344eb92p278,
                0x1.72b4ad455ea8p-63,
                0x1.72b4d435381e9p874,
                0x1.72bc8db36340dp-849,
                0x1.72cc3e456c74cp810,
                0x1.72cdbb3c06a7ep-726,
                0x1.72cef1ebbca44p-49,
                0x1.72cfd8978422ep-906,
                0x1.72cfd8978422ep-907,
                0x1.72cfd8978422ep-908,
                0x1.72cfd8978422ep-909,
                0x1.72cfd8978422ep-910,
                0x1.72d206123324ep-129,
                0x1.72d53ceefc1abp-276,
                0x1.72d9299449e74p-797,
                0x1.72d9299449e74p-798,
                0x1.72ddb5305e58cp+440,
                0x1.72ddb5305e58cp441,
                0x1.72e131e85753ap-523,
                0x1.72e4d6e22530dp-754,
                0x1.72ebad6ddc73cp231,
                0x1.72ebad6ddc73cp+232,
                0x1.72ebad6ddc73cp233,
                0x1.72ebad6ddc73cp234,
                0x1.72ebad6ddc73cp235,
                0x1.72f7a1831ad71p357,
                0x1.72f7a1831ad71p358,
                0x1.7300289f1fb97p946,
                0x1.730137607b3b8p-389,
                0x1.730137607b3b8p-390,
                0x1.7303f18c4271ap139,
                0x1.73057a9b3a924p-819,
                0x1.73057a9b3a924p-820,
                0x1.730a527544032p-369,
                0x1.7311a17336ee2p-423,
                0x1.7312f78267921p482,
                0x1.73142ea4cfaf3p-154,
                0x1.73159b3ec4864p-143,
                0x1.73189da1f22c3p-877,
                0x1.731e0c298a89cp+804,
                0x1.731e0c298a89cp805,
                0x1.731e0c298a89cp806,
                0x1.731e0c298a89cp807,
                0x1.732556f7a6eeap886,
                0x1.732d792c1e1b6p-633,
                0x1.732d800f48b82p+284,
                0x1.7337840e7e4f4p-58,
                0x1.7337840e7e4f4p-59,
                0x1.7337840e7e4f4p-60,
                0x1.7337840e7e4f4p-61,
                0x1.7337840e7e4f4p-62,
                0x1.734e940c6f9c6p717,
                0x1.73505f40f7b56p-225,
                0x1.73579b4de26fdp590,
                0x1.7358086f5437cp821,
                0x1.73583f54cf314p+332,
                0x1.73583f54cf314p333,
                0x1.7361cb863de62p-76,
                0x1.7361cb863de62p-77,
                0x1.7361cb863de62p-78,
                0x1.7361cb863de62p-79,
                0x1.7361cb863de62p-80,
                0x1.7361cb863de62p-81,
                0x1.7361cb863de62p-82,
                0x1.7361cb863de62p-83,
                0x1.7361cb863de62p-84,
                0x1.737402c5d5ff8p390,
                0x1.737aa2567167cp127,
                0x1.7382f1df55911p-836,
                0x1.738cca4c5ef7ep-435,
                0x1.738cca4c5ef7ep-436,
                0x1.738cca4c5ef7ep-437,
                0x1.738cca4c5ef7ep-438,
                0x1.73998940ee03p+600,
                0x1.73998940ee03p601,
                0x1.73998940ee03p602,
                0x1.73998940ee03p603,
                0x1.739a961760598p-124,
                0x1.739a961760598p-125,
                0x1.739a961760598p-126,
                0x1.739a961760598p-127,
                0x1.739a961760598p-128,
                0x1.739f473b1984ep+876,
                0x1.739f473b1984ep877,
                0x1.73b1512b9e52ep-434,
                0x1.73b1512b9e52ep-435,
                0x1.73b1512b9e52ep-436,
                0x1.73b1512b9e52ep-437,
                0x1.73c115074bc6ap-934,
                0x1.73c4d1a3630e8p402,
                0x1.73c4d1a3630e8p403,
                0x1.73c4d1a3630e8p+404,
                0x1.73c4d1a3630e8p405,
                0x1.73c4d1a3630e8p406,
                0x1.73c4d1a3630e8p407,
                0x1.73cdf8db7a7bcp-24,
                0x1.73cdf8db7a7bcp-25,
                0x1.73cdf8db7a7bcp-26,
                0x1.73cdf8db7a7bcp-27,
                0x1.73cdf8db7a7bcp-28,
                0x1.73cdf8db7a7bcp-29,
                0x1.73cdf8db7a7bcp-30,
                0x1.73cdf8db7a7bcp-31,
                0x1.73d80f3bc596ep-396,
                0x1.73db6fb186d5bp-496,
                0x1.73db9939ce4a8p827,
                0x1.73eb2fd54f818p289,
                0x1.73eb5a842dde6p-894,
                0x1.73f3248dee69ap-43,
                0x1.73f3248dee69ap-44,
                0x1.73f9614840dabp+736,
                0x1.73f9614840dabp737,
                0x1.73f9614840dabp738,
                0x1.73fce30e6ba96p-771,
                0x1.73fd5b708ec9ep551,
                0x1.73fd5b708ec9ep+552,
                0x1.740032c6de418p662,
                0x1.7403ac66be0e3p-869,
                0x1.7404d1dcffac8p-734,
                0x1.7404d1dcffac8p-735,
                0x1.740b97c9b37ffp175,
                0x1.740b97c9b37ffp+176,
                0x1.740fb1ec13c2p-998,
                0x1.740fb1ec13c2p-999,
                0x1.7419080b18e37p315,
                0x1.7424348ca1c9cp-449,
                0x1.74352fa5ae52p-398,
                0x1.7446111044dbep+424,
                0x1.744685e678d2ep-52,
                0x1.7448ae4e36b88p425,
                0x1.7448ae4e36b88p426,
                0x1.7448ae4e36b88p427,
                0x1.744e4304cda1fp-220,
                0x1.745653c73774ap+988,
                0x1.74581f9d8e51cp-702,
                0x1.745a794f7fdf7p-247,
                0x1.745a794f7fdf7p-248,
                0x1.747b46d78c6fep-589,
                0x1.747b81faa6c35p-706,
                0x1.74876e8p+36,
                0x1.7488e99eb909ap974,
                0x1.7488e99eb909ap975,
                0x1.748ab9c8d88f2p-579,
                0x1.748c5812d97d4p155,
                0x1.748deb444b4eap467,
                0x1.7493d664f1354p-415,
                0x1.7493d664f1354p-416,
                0x1.7493d664f1354p-417,
                0x1.7493d664f1354p-418,
                0x1.749ab5aef4768p+996,
                0x1.749ab62ad32a4p-763,
                0x1.749d69a3ddb2ap678,
                0x1.749de580f5374p-707,
                0x1.749de580f5374p-708,
                0x1.74a75a2da0842p654,
                0x1.74bff50de2308p-89,
                0x1.74c1826f3010cp-18,
                0x1.74c45a2d69ccap506,
                0x1.74c507ae34e8p-1024,
                0x1.74c687cbe11d4p-935,
                0x1.74d74a0923d3p-946,
                0x1.74d74a0923d3p-947,
                0x1.74e364f1a96b9p-332,
                0x1.74e907164b2e4p-806,
                0x1.74e907164b2e4p-807,
                0x1.74e907164b2e4p-808,
                0x1.74e907164b2e4p-809,
                0x1.74e907164b2e4p-810,
                0x1.74eac2e8727b1p521,
                0x1.74eb7827e14d8p878,
                0x1.74f03c80f9034p226,
                0x1.74fac06b29e03p-590,
                0x1.74fc3889a165ep-304,
                0x1.7501a9b8ab4b7p-223,
                0x1.750379bca88c4p415,
                0x1.750379bca88c4p+416,
                0x1.7504c1eb95166p551,
                0x1.750958058079ep539,
                0x1.750958058079ep+540,
                0x1.750958058079ep541,
                0x1.7509e3c4d6824p918,
                0x1.751277687bccp+320,
                0x1.751e26b1580f6p+852,
                0x1.75224ab64f00ep-497,
                0x1.75224ab64f00ep-498,
                0x1.75224ab64f00ep-499,
                0x1.752a83e5bf5ep+824,
                0x1.75316a0e5c41p253,
                0x1.754e31cd072dap1003,
                0x1.754e31cd072dap+1004,
                0x1.754e31cd072dap1005,
                0x1.754e31cd072dap1006,
                0x1.754e8c980a4afp-231,
                0x1.754e8c980a4afp-232,
                0x1.754f5f2950ff4p-95,
                0x1.754f5f2950ff4p-96,
                0x1.7554a3e2fcc8cp-182,
                0x1.75581921468b7p-867,
                0x1.755ae1adc29eap606,
                0x1.755ae1adc29eap607,
                0x1.756183c147514p207,
                0x1.7578ae3f19878p-484,
                0x1.757fbbfd28d1cp-840,
                0x1.7580a6c6f8a91p255,
                0x1.7587165e5b94fp-75,
                0x1.758ae0dd68ba3p886,
                0x1.7590701p-1046,
                0x1.7592f4d3c9242p786,
                0x1.759a3d44p-1044,
                0x1.759b1cacbb725p-995,
                0x1.75b4fd0e90cd8p-702,
                0x1.75b4fd0e90cd8p-703,
                0x1.75b9f95213fc5p-165,
                0x1.75bc216e9ee44p-942,
                0x1.75bc216e9ee44p-943,
                0x1.75bc216e9ee44p-944,
                0x1.75bc216e9ee44p-945,
                0x1.75bc216e9ee44p-946,
                0x1.75c1508da432ap-645,
                0x1.75c1508da432bp-645,
                0x1.75c5cfb4c226ep-219,
                0x1.75cc381e190cdp-310,
                0x1.75d184c8b82cbp238,
                0x1.75d184c8b82cbp239,
                0x1.75d226331d03ap719,
                0x1.75d226331d03ap+720,
                0x1.75d226331d03ap721,
                0x1.75d226331d03ap722,
                0x1.75d226331d03ap723,
                0x1.75d226331d03ap+724,
                0x1.75d8bc5b4e467p-942,
                0x1.75d8bc5b4e467p-943,
                0x1.75ec48af6689p290,
                0x1.75eeb6a6d6b84p-923,
                0x1.75eeb6a6d6b84p-924,
                0x1.75f7304b36faap-718,
                0x1.75f7304b36faap-719,
                0x1.75fa99a657506p-535,
                0x1.76075fa16910ap925,
                0x1.760d19483fb6ep-568,
                0x1.761744d42487ap451,
                0x1.761744d42487ap+452,
                0x1.7624f8a762fd8p-160,
                0x1.7625d33a7366dp-276,
                0x1.76267e5277356p-86,
                0x1.76285d9b7d518p-401,
                0x1.762ed70c6b3d5p-570,
                0x1.7631921b4d22p-377,
                0x1.7631921b4d22p-378,
                0x1.7631921b4d22p-379,
                0x1.7631921b4d22p-380,
                0x1.7631921b4d22p-381,
                0x1.7631921b4d22p-382,
                0x1.7631921b4d22p-383,
                0x1.7631921b4d22p-384,
                0x1.7635b25f99028p123,
                0x1.7635b25f99028p+124,
                0x1.7648fd3da9008p387,
                0x1.764f84d5d3aa8p-158,
                0x1.764fdb7359874p565,
                0x1.765447987844p-1004,
                0x1.765a884ecdf5dp327,
                0x1.765a884ecdf5dp+328,
                0x1.765f6c4acc6a6p645,
                0x1.7667905d04e8cp-12,
                0x1.7667905d04e8cp-13,
                0x1.76688faee99bcp-841,
                0x1.76715b1b382fep-146,
                0x1.7671ccdd1fd0fp709,
                0x1.76851b6f03996p+364,
                0x1.76851b6f03996p365,
                0x1.768569f81b791p-27,
                0x1.7688bb5394c25p323,
                0x1.7688bb5394c25p+324,
                0x1.7688bb5394c25p325,
                0x1.768bf06b08418p457,
                0x1.7690a5b10e4fp-389,
                0x1.769fe1845e9d8p863,
                0x1.76a0f8d419f58p-946,
                0x1.76a4cc9bc4acbp675,
                0x1.76ab9bc24c346p1011,
                0x1.76ab9bc24c346p+1012,
                0x1.76ae79cea3b93p-875,
                0x1.76b1f9c623084p+196,
                0x1.76ba749d21993p+740,
                0x1.76bc172a76d83p+444,
                0x1.76be128473d44p869,
                0x1.76be128473d44p870,
                0x1.76c2291f2e6b4p-988,
                0x1.76c2291f2e6b4p-989,
                0x1.76c2291f2e6b4p-990,
                0x1.76c2496f477ddp-883,
                0x1.76c4ad5be5812p-976,
                0x1.76c4ad5be5812p-977,
                0x1.76c4ad5be5812p-978,
                0x1.76c4ad5be5812p-979,
                0x1.76c4ad5be5812p-980,
                0x1.76c84f8e12553p650,
                0x1.76ca8be2021d6p-121,
                0x1.76cbd60a94744p-40,
                0x1.76d12c3402e05p+600,
                0x1.76da012fbffbap-852,
                0x1.76da012fbffbap-853,
                0x1.76ea949db859ep-88,
                0x1.76ec98994f488p+808,
                0x1.76ec98994f488p809,
                0x1.76ec98994f488p810,
                0x1.76ec98994f488p811,
                0x1.76ec98994f488p+812,
                0x1.76ec98994f488p813,
                0x1.76ec98994f488p814,
                0x1.76ec98994f488p815,
                0x1.76f3e490ee71ep-360,
                0x1.76f96f3a4f7fep-446,
                0x1.76fe639c69825p-495,
                0x1.7702957c5f36ap-927,
                0x1.770f81580af78p-812,
                0x1.770f81580af78p-813,
                0x1.77132e58f922cp225,
                0x1.7716fa45a151p435,
                0x1.77196b9c45979p647,
                0x1.771a9805ecab6p-509,
                0x1.771f62f7c536bp-138,
                0x1.771f62f7c536bp-139,
                0x1.771f87c833e7ep-95,
                0x1.772bd39926632p-537,
                0x1.773a6ea8c76ffp425,
                0x1.774456697fd7ep+1000,
                0x1.774456697fd7ep1001,
                0x1.7744c9fdea9e9p571,
                0x1.77467f4a92f6cp350,
                0x1.7753fc708d7c8p137,
                0x1.775636d20d08ap-764,
                0x1.775b2985a4b9ap+312,
                0x1.775b2985a4b9ap313,
                0x1.775b2985a4b9ap314,
                0x1.775b2985a4b9ap315,
                0x1.775b77f1bdcd8p-666,
                0x1.775b77f1bdcd8p-667,
                0x1.775b77f1bdcd8p-668,
                0x1.77603725064a8p-839,
                0x1.77603725064a8p-840,
                0x1.77603725064a8p-841,
                0x1.7763fd12d819fp+496,
                0x1.7764ad224e24ap110,
                0x1.77652a33ea1dcp+900,
                0x1.77652a33ea1dcp901,
                0x1.776e3afb70aeep+276,
                0x1.776e6947c7146p778,
                0x1.776ed35c9135p-291,
                0x1.776ed35c9135p-292,
                0x1.7772e1f9aee1ap+692,
                0x1.7774c0b361ed9p+508,
                0x1.777e442b2c4f4p813,
                0x1.777e442b2c4f4p814,
                0x1.778698671312fp610,
                0x1.778b21df59e1p-823,
                0x1.778b21df59e1p-824,
                0x1.778d12p-1051,
                0x1.778p-1065,
                0x1.7791804859b56p-482,
                0x1.77945d1ec1269p545,
                0x1.7797619b1448p-850,
                0x1.779f071b03a64p206,
                0x1.779f071b03a64p207,
                0x1.779f54fb1071dp-746,
                0x1.779f54fb1071dp-747,
                0x1.77ad870c6f789p-583,
                0x1.77b30c1a7bc8ap-129,
                0x1.77bc3d4ca153p254,
                0x1.77c44ddf6c516p-356,
                0x1.77c44ddf6c516p-357,
                0x1.77c44ddf6c516p-358,
                0x1.77c96c3a0913p847,
                0x1.77d8188aa57d4p-743,
                0x1.77d8188aa57d4p-744,
                0x1.77d8188aa57d4p-745,
                0x1.77dadf69f79aap493,
                0x1.77dda9169a83ep-505,
                0x1.77dda9169a83ep-506,
                0x1.77dda9169a83ep-507,
                0x1.77dda9169a83ep-508,
                0x1.77dda9169a83ep-509,
                0x1.77e2f08477dcdp+244,
                0x1.77e38910b4bfap-106,
                0x1.77eb06d7586bp-714,
                0x1.77f1a5cfa3778p893,
                0x1.7800a880d2f85p+972,
                0x1.780695036a67ap482,
                0x1.780695036a67ap483,
                0x1.780695036a67ap+484,
                0x1.780d098c870c8p691,
                0x1.781569e633p-1034,
                0x1.7817033fc4fdfp-69,
                0x1.781e6b0ffbe4ep905,
                0x1.78260a3b3b4f8p-242,
                0x1.78260c1c37a56p-841,
                0x1.78287f49c4a1dp129,
                0x1.78287f49c4a1ep129,
                0x1.78287f49c4a1ep130,
                0x1.7828fa57e915p-585,
                0x1.7828fa57e915p-586,
                0x1.782933ae7e4d4p-463,
                0x1.782933ae7e4d4p-464,
                0x1.782933ae7e4d4p-465,
                0x1.782933ae7e4d4p-466,
                0x1.782933ae7e4d4p-467,
                0x1.782933ae7e4d4p-468,
                0x1.782adbb4cdd3dp+920,
                0x1.782adbb4cdd3dp921,
                0x1.7831ac830f56ap911,
                0x1.7831ac830f56ap+912,
                0x1.783361aa80bd6p-983,
                0x1.783361aa80bd6p-984,
                0x1.783361aa80bd6p-985,
                0x1.783361aa80bd6p-986,
                0x1.78358811cbc96p422,
                0x1.78358811cbc96p423,
                0x1.78358811cbc96p+424,
                0x1.78358811cbc96p425,
                0x1.78358811cbc96p426,
                0x1.78358811cbc96p427,
                0x1.784366184a4e4p-694,
                0x1.784366184a4e4p-695,
                0x1.784366184a4e4p-696,
                0x1.784df6366f95ap-137,
                0x1.78561d3bd2876p194,
                0x1.785e78ac221acp438,
                0x1.7866b217631f8p-931,
                0x1.786a7aa8a1184p406,
                0x1.786aa79f1018p-948,
                0x1.786ecb774d68dp+224,
                0x1.7884008c19e5cp-123,
                0x1.788ccb6b2ce0cp614,
                0x1.78947479f53adp-983,
                0x1.78947479f53adp-984,
                0x1.7896293f657c5p418,
                0x1.789e856cd1925p-981,
                0x1.78a623b1aa53ap-480,
                0x1.78b169dcb5a25p761,
                0x1.78b2a0c191856p+908,
                0x1.78b6516c2b478p247,
                0x1.78b6516c2b478p+248,
                0x1.78bde3cdff2fcp486,
                0x1.78bf76c490f1ep-775,
                0x1.78bf76c490f1ep-776,
                0x1.78c26839554cp-504,
                0x1.78c26839554cp-505,
                0x1.78c4d7b81a327p347,
                0x1.78c9e90a50618p-697,
                0x1.78c9e90a50618p-698,
                0x1.78cfcab31064cp-83,
                0x1.78cfcab31064cp-84,
                0x1.78cfcab31064cp-85,
                0x1.78cfcab31064cp-86,
                0x1.78cfcab31064cp-87,
                0x1.78cfcab31064cp-88,
                0x1.78d2c97bde2a1p-72,
                0x1.78d430d0365bep935,
                0x1.78d437fad2654p614,
                0x1.78d437fad2654p615,
                0x1.78d43b2f15ba3p-142,
                0x1.78d7981155d75p-256,
                0x1.78db025b0a21fp-151,
                0x1.78dbdeae43647p-186,
                0x1.78e3cdacbf9c7p-818,
                0x1.78f1e51d8f74ap182,
                0x1.78f2bd602e742p-706,
                0x1.78f2bd602e742p-707,
                0x1.78f52483785e6p-80,
                0x1.78fa19890c9cp709,
                0x1.78fa19890c9cp710,
                0x1.7900ea4fd8p-1037,
                0x1.7900ea4fda8p-1033,
                0x1.7904b46df0356p+300,
                0x1.7904b46df0356p301,
                0x1.79066d2f692e2p-788,
                0x1.790e6d080feefp442,
                0x1.791544a044dacp-424,
                0x1.791544a044dacp-425,
                0x1.791dd4266d756p166,
                0x1.791dd4266d756p167,
                0x1.791de464ep-1039,
                0x1.791e3d106d488p+884,
                0x1.791e3d106d488p885,
                0x1.791e3d106d488p886,
                0x1.791f7c7f6210ap-653,
                0x1.7924e3cee21fcp-928,
                0x1.792c3fb55215p+300,
                0x1.793e0e9e0091p-639,
                0x1.794987f4d717ap867,
                0x1.794987f4d717ap+868,
                0x1.79527f1688938p+564,
                0x1.795a531b37b5bp-795,
                0x1.79657025b6234p-552,
                0x1.796640d9e8692p743,
                0x1.79693b4676faep785,
                0x1.796ca1f978f97p-451,
                0x1.796ca1f978f97p-452,
                0x1.79840c8b6f1f4p-125,
                0x1.79892eb1730a2p950,
                0x1.798b0a9c7dc16p381,
                0x1.798b0a9c7dc16p382,
                0x1.798b0a9c7dc16p383,
                0x1.798b0a9c7dc16p+384,
                0x1.798b0a9c7dc16p385,
                0x1.79918a35d32f7p1023,
                0x1.799c22d9502fep-409,
                0x1.799c22d9502fep-410,
                0x1.799c22d9502fep-411,
                0x1.799e78ff3824ep919,
                0x1.799e78ff3824ep+920,
                0x1.79a58ba6dc708p274,
                0x1.79ae3593295efp-783,
                0x1.79ae3593295efp-784,
                0x1.79b94be3d3f16p699,
                0x1.79bd03a514dc6p-844,
                0x1.79ca10c924223p-67,
                0x1.79ca10c924224p-66,
                0x1.79ca10c924224p-67,
                0x1.79d0124345806p581,
                0x1.79d0a8ff65bdp-375,
                0x1.79e6125ccb848p379,
                0x1.79e618a1311f4p-171,
                0x1.79ecc161883f2p+180,
                0x1.79f4476d372a3p-25,
                0x1.79f53a2bacabep-983,
                0x1.79f53a2bacabep-984,
                0x1.79faa1bcadf93p-479,
                0x1.79fd204e9c959p551,
                0x1.79ff78414f47fp-351,
                0x1.7a01d4ce4edb8p602,
                0x1.7a02bb4e4c678p+352,
                0x1.7a053e8711f1ap-158,
                0x1.7a06a6dad042fp326,
                0x1.7a06a6dad042fp327,
                0x1.7a07b216ae5cfp397,
                0x1.7a1bdc59730c1p-777,
                0x1.7a1bdc59730c1p-778,
                0x1.7a29552d7cc79p-415,
                0x1.7a2ecc414a03fp418,
                0x1.7a2ecc414a04p414,
                0x1.7a2ecc414a04p415,
                0x1.7a2ecc414a04p+416,
                0x1.7a2ecc414a04p417,
                0x1.7a2ecc414a04p418,
                0x1.7a2ecc414a04p419,
                0x1.7a2ecc414a04p+420,
                0x1.7a2ecc414a04p421,
                0x1.7a2ecc414a04p422,
                0x1.7a2ecc414a04p423,
                0x1.7a2ecc414a04p+424,
                0x1.7a2ecc414a04p425,
                0x1.7a2ecc414a04p426,
                0x1.7a2ecc414a04p427,
                0x1.7a2ecc414a04p+428,
                0x1.7a30f4f9e707cp-980,
                0x1.7a3913ebe08b5p191,
                0x1.7a3a4d079c81cp-379,
                0x1.7a3bdf1842802p203,
                0x1.7a55300ed7936p+840,
                0x1.7a697a33e4d7ap-534,
                0x1.7a697a33e4d7ap-535,
                0x1.7a697a33e4d7ap-536,
                0x1.7a786374b32afp986,
                0x1.7a8747448b6bap-551,
                0x1.7a8747448b6bap-552,
                0x1.7a937f034e15p-446,
                0x1.7a93a2954f3b8p903,
                0x1.7a93a2954f3b8p+904,
                0x1.7a93a2954f3b8p905,
                0x1.7a9c49e590558p-554,
                0x1.7a9c49e590558p-555,
                0x1.7a9c49e590558p-556,
                0x1.7aa52901df2a4p-359,
                0x1.7aa95a0ac1446p+768,
                0x1.7aa95a0ac1446p769,
                0x1.7ab11aadf9f8p-1028,
                0x1.7ab946e29886p999,
                0x1.7abb6833a379ap+640,
                0x1.7abb6833a379ap641,
                0x1.7abb6833a379ap642,
                0x1.7abb6833a379ap643,
                0x1.7abb6833a379ap+644,
                0x1.7abc6036020fep-439,
                0x1.7abe66de389bcp-536,
                0x1.7ac5cc8482294p+712,
                0x1.7aceba8bf8176p777,
                0x1.7acfdfeeb70c5p-237,
                0x1.7ad2a3bb8038ep+312,
                0x1.7ad4444cf5e8cp+260,
                0x1.7ad995c510599p933,
                0x1.7ae06f2f4c53ep+384,
                0x1.7ae19a05263d4p681,
                0x1.7ae19a05263d4p682,
                0x1.7ae3a09ead5dep-1020,
                0x1.7ae3a09ead5dep-1021,
                0x1.7ae86a5173602p-813,
                0x1.7ae9bff2f264ep-528,
                0x1.7ae9bff2f264ep-529,
                0x1.7ae9bff2f264ep-530,
                0x1.7ae9bff2f264ep-531,
                0x1.7aea7ba0cfb99p-460,
                0x1.7aeda07725b04p535,
                0x1.7aeda07725b04p+536,
                0x1.7aee14f386e91p709,
                0x1.7af2477df48f4p915,
                0x1.7af5904accd62p746,
                0x1.7af5904accd62p747,
                0x1.7af9f832d208ap1006,
                0x1.7afa3ff71ad7ap-882,
                0x1.7afa3ff71ad7ap-883,
                0x1.7afa3ff71ad7ap-884,
                0x1.7afa3ff71ad7ap-885,
                0x1.7afa3ff71ad7ap-886,
                0x1.7afe8b4e46311p-921,
                0x1.7b044p-1056,
                0x1.7b079fc677785p835,
                0x1.7b08617a104eep-748,
                0x1.7b12e00f4a13bp-521,
                0x1.7b1370759628dp-413,
                0x1.7b1bbd2f96f8ep-113,
                0x1.7b1bbd2f96f8ep-114,
                0x1.7b2085166381p-184,
                0x1.7b240ac5e9ffap-199,
                0x1.7b2749dff1e32p-835,
                0x1.7b2749dff1e32p-836,
                0x1.7b2749dff1e32p-837,
                0x1.7b2749dff1e32p-838,
                0x1.7b2749dff1e32p-839,
                0x1.7b35936d56e28p-829,
                0x1.7b48e3eef5ec8p+228,
                0x1.7b49906154ebp-691,
                0x1.7b49906154ebp-692,
                0x1.7b49906154ebp-693,
                0x1.7b59eea35f58cp-313,
                0x1.7b59eea35f58cp-314,
                0x1.7b59eea35f58cp-315,
                0x1.7b59eea35f58cp-316,
                0x1.7b59eea35f58cp-317,
                0x1.7b5c213f300cbp-546,
                0x1.7b5c213f300cbp-547,
                0x1.7b6723926ed3p-736,
                0x1.7b696889b34ccp-228,
                0x1.7b69f6317e7b1p-222,
                0x1.7b6d71d20b96cp-263,
                0x1.7b6d71d20b96cp-264,
                0x1.7b6d71d20b96cp-265,
                0x1.7b6fd0c6fb376p-217,
                0x1.7b6fd0c6fb376p-218,
                0x1.7b6fd0c6fb376p-219,
                0x1.7b6fd0c6fb376p-220,
                0x1.7b6fdb5912f5cp-284,
                0x1.7b72b184edec2p173,
                0x1.7b72b184edec2p174,
                0x1.7b72b184edec2p175,
                0x1.7b72b184edec2p+176,
                0x1.7b72b184edec2p177,
                0x1.7b746dcb1d536p-996,
                0x1.7b74f6c4ed52ep905,
                0x1.7b78c8cbced86p338,
                0x1.7b8389cb0d428p-626,
                0x1.7b8389cb0d428p-627,
                0x1.7b84e5704cceep274,
                0x1.7b84e5704cceep275,
                0x1.7b84e5704cceep+276,
                0x1.7b84e5704cceep277,
                0x1.7b8c2763f3be9p-727,
                0x1.7b923a7fd9d0cp939,
                0x1.7b923a7fd9d0cp+940,
                0x1.7b923a7fd9d0cp941,
                0x1.7b976fd49b8eap626,
                0x1.7b976fd49b8eap627,
                0x1.7b976fd49b8eap+628,
                0x1.7b976fd49b8eap629,
                0x1.7b97b9cf186f4p-239,
                0x1.7b97b9cf186f4p-240,
                0x1.7b9902ccca8d7p681,
                0x1.7b9c26991290ap699,
                0x1.7b9fad34efe87p410,
                0x1.7ba37c4fbe994p221,
                0x1.7ba37c4fbe994p222,
                0x1.7ba37c4fbe994p223,
                0x1.7bab9160249c2p-134,
                0x1.7baebda91f07p-286,
                0x1.7bb096545db81p+992,
                0x1.7bb69b8499e8ap-494,
                0x1.7bc2237434d8cp-599,
                0x1.7bc939293ca5ep861,
                0x1.7bd1326e6649p-933,
                0x1.7bd29d1c87a19p222,
                0x1.7bd6b71a5c7a2p-490,
                0x1.7bd6b71a5c7a2p-491,
                0x1.7bd6b71a5c7a2p-492,
                0x1.7bdf432f9f33cp-231,
                0x1.7be02b675eac4p-773,
                0x1.7be02b675eac4p-774,
                0x1.7be02b675eac4p-775,
                0x1.7be279e979d43p178,
                0x1.7beeb325f3eabp-297,
                0x1.7bef248c358d3p-617,
                0x1.7bef248c358d3p-618,
                0x1.7bf1a561ed082p621,
                0x1.7bf1a561ed082p622,
                0x1.7bf1a561ed082p623,
                0x1.7bf775f0f3e9p-536,
                0x1.7bf775f0f3e9p-537,
                0x1.7bf7c6965ec76p+620,
                0x1.7bf7c6965ec76p621,
                0x1.7bf7c6965ec76p622,
                0x1.7bf7c6965ec76p623,
                0x1.7bf7c6965ec76p+624,
                0x1.7bf7c6965ec76p625,
                0x1.7bfc277dbe0a3p198,
                0x1.7c026b84ba3ap+724,
                0x1.7c0747bd76fa1p-813,
                0x1.7c0747bd76fa1p-814,
                0x1.7c10c923792fcp-815,
                0x1.7c10c923792fcp-816,
                0x1.7c10c923792fcp-817,
                0x1.7c137a2043e38p-649,
                0x1.7c16ec67934f1p-541,
                0x1.7c28222acf75p-284,
                0x1.7c28222acf75p-285,
                0x1.7c2d8927cc2e4p-904,
                0x1.7c2f36d1d8f2dp-963,
                0x1.7c342db18c08dp902,
                0x1.7c37e360b3d35p707,
                0x1.7c3b8228ee0fp+728,
                0x1.7c3b8228ee0fp729,
                0x1.7c3b8228ee0fp730,
                0x1.7c3bdc2b317c3p-979,
                0x1.7c3bdc2b317c3p-980,
                0x1.7c481b871a8abp+724,
                0x1.7c49092d1eb4fp-787,
                0x1.7c4b90ebc4813p-92,
                0x1.7c4f8e3335b4cp821,
                0x1.7c4f8e3335b4cp822,
                0x1.7c4f8e3335b4cp823,
                0x1.7c50a6ae69026p-895,
                0x1.7c5132e73543p391,
                0x1.7c52358ea9b4cp-979,
                0x1.7c554e5e94f9ep605,
                0x1.7c5b7ede4df7ep-516,
                0x1.7c81a74936afep-207,
                0x1.7c8489b06ea54p907,
                0x1.7c8489b06ea54p+908,
                0x1.7c8cd77503081p585,
                0x1.7c8f604e70c22p-557,
                0x1.7c938c44bf62ap-228,
                0x1.7c938c44bf62ap-229,
                0x1.7c938c44bf62ap-230,
                0x1.7c938c44bf62ap-231,
                0x1.7c938c44bf62ap-232,
                0x1.7c938c44bf62ap-233,
                0x1.7c938c44bf62ap-234,
                0x1.7c938c44bf62ap-235,
                0x1.7c9c579647e47p853,
                0x1.7c9d52bd5315ap-942,
                0x1.7ca18a876c5fp226,
                0x1.7ca18a876c5fp227,
                0x1.7ca18a876c5fp+228,
                0x1.7ca18a876c5fp229,
                0x1.7ca18a876c5fp230,
                0x1.7ca18a876c5fp231,
                0x1.7ca857bb602f6p+496,
                0x1.7ca89391d923cp-347,
                0x1.7ca89391d923cp-348,
                0x1.7ca89391d923cp-349,
                0x1.7ca89391d923cp-350,
                0x1.7ca8e3d68578ep521,
                0x1.7ca8e3d68578ep522,
                0x1.7ca8e3d68578ep523,
                0x1.7ca8e3d68578ep+524,
                0x1.7ca8e3d68578ep525,
                0x1.7ca8e3d68578ep526,
                0x1.7ca8e3d68578ep527,
                0x1.7ca8e3d68578ep+528,
                0x1.7ca8e3d68578ep529,
                0x1.7ca8e3d68578ep530,
                0x1.7ca8e3d68578ep531,
                0x1.7cad23de82d7ap-944,
                0x1.7cad23de82d7bp-944,
                0x1.7cb0f77003214p393,
                0x1.7cbea1bf6e9fcp-969,
                0x1.7ccabfc07c223p977,
                0x1.7ccb99298a178p927,
                0x1.7ccb99298a178p+928,
                0x1.7ccd9a7586d29p209,
                0x1.7ccfc72347b28p-654,
                0x1.7ccfc73126788p-71,
                0x1.7ccfc73126788p-72,
                0x1.7cd97642af427p169,
                0x1.7cde58446f74ap-249,
                0x1.7ce3f85c2f999p-424,
                0x1.7ce714a05d3bcp-852,
                0x1.7ce714a05d3bcp-853,
                0x1.7ce714a05d3bcp-854,
                0x1.7ce714a05d3bcp-855,
                0x1.7ce714a05d3bcp-856,
                0x1.7ce714a05d3bcp-857,
                0x1.7ce714a05d3bcp-858,
                0x1.7ce714a05d3bcp-859,
                0x1.7d01d4691ddbep393,
                0x1.7d07f9c786e86p482,
                0x1.7d07f9c786e86p483,
                0x1.7d07f9c786e86p+484,
                0x1.7d08433119118p-135,
                0x1.7d0d9de038344p+208,
                0x1.7d12a4670c122p-457,
                0x1.7d12a4670c122p-458,
                0x1.7d12a4670c122p-459,
                0x1.7d12a4670c122p-460,
                0x1.7d15bf318608p291,
                0x1.7d15bf318608p+292,
                0x1.7d15bf318608p293,
                0x1.7d15bf318608p294,
                0x1.7d15bf318608p295,
                0x1.7d15bf318608p+296,
                0x1.7d15bf318608p297,
                0x1.7d15bf318608p298,
                0x1.7d1678fdbc94cp878,
                0x1.7d26082060686p-330,
                0x1.7d2950dc76da4p137,
                0x1.7d2950dc76da4p138,
                0x1.7d2db3bc1625ap-878,
                0x1.7d42850b4ff87p+836,
                0x1.7d459a7157161p-272,
                0x1.7d4b57562e71p-265,
                0x1.7d4b57562e71p-266,
                0x1.7d4be648e175ap853,
                0x1.7d4be648e175ap854,
                0x1.7d50e5d86225cp+772,
                0x1.7d50e5d86225cp773,
                0x1.7d57c8ad4d28cp805,
                0x1.7d57c8ad4d28cp806,
                0x1.7d57c8ad4d28cp807,
                0x1.7d57c8ad4d28cp+808,
                0x1.7d57c8ad4d28cp809,
                0x1.7d583c5d0c9d9p443,
                0x1.7d70b5d624e8ap-78,
                0x1.7d736b91a07p671,
                0x1.7d736b91a07p+672,
                0x1.7d736b91a07p673,
                0x1.7d784p26,
                0x1.7d7c189577b6p883,
                0x1.7d7c189577b6p+884,
                0x1.7d83612f2e45p361,
                0x1.7d88f54a5bd9cp421,
                0x1.7d88f54a5bd9cp422,
                0x1.7d8bfe1f908d9p-773,
                0x1.7d93193f78fc6p+584,
                0x1.7d93193f78fc6p585,
                0x1.7d93193f78fc6p586,
                0x1.7d93193f78fc6p587,
                0x1.7d93193f78fc6p+588,
                0x1.7d9821bcb150ep771,
                0x1.7d9821bcb150ep+772,
                0x1.7d9821bcb150ep773,
                0x1.7d9821bcb150ep774,
                0x1.7d9821bcb150ep775,
                0x1.7d9a2c698239cp-72,
                0x1.7d9a2c698239cp-73,
                0x1.7d9ca81344168p-792,
                0x1.7daa4a3ccd559p265,
                0x1.7dafd59d786aap-250,
                0x1.7dafd59d786aap-251,
                0x1.7db744c0de168p-62,
                0x1.7db88802ff716p794,
                0x1.7dc2919b0f4b8p-519,
                0x1.7dc8p-1061,
                0x1.7dcd962873b36p-730,
                0x1.7dcd962873b36p-731,
                0x1.7dd0664a99caap-478,
                0x1.7dd8ccd104372p-453,
                0x1.7ddbbbd7e49bp-600,
                0x1.7ddcf64621755p467,
                0x1.7dddf6b095ff1p511,
                0x1.7dddf6b095ffp511,
                0x1.7de84c3d8979cp-817,
                0x1.7dead8296a2c6p-799,
                0x1.7dead8296a2c6p-800,
                0x1.7def37e37ffccp+576,
                0x1.7df1e16f08e66p+468,
                0x1.7df3842d6aa4ep-705,
                0x1.7df3842d6aa4ep-706,
                0x1.7dfa7b83f44dep+668,
                0x1.7dfef47bb4115p-257,
                0x1.7e0a7b01323dap607,
                0x1.7e165aef889bp-316,
                0x1.7e165aef889bp-317,
                0x1.7e194e0c5ca1cp-706,
                0x1.7e1aba124e35bp-629,
                0x1.7e1ac0934935p114,
                0x1.7e1e0f1c7a4acp413,
                0x1.7e1e0f1c7a4acp414,
                0x1.7e1e0f1c7a4acp415,
                0x1.7e1e0f1c7a4acp+416,
                0x1.7e2ed3362d34ep998,
                0x1.7e351ac7acbf7p+688,
                0x1.7e3987916a69ep-248,
                0x1.7e3987916a69ep-249,
                0x1.7e3987916a69ep-250,
                0x1.7e43c8800759cp+996,
                0x1.7e43c8800759cp997,
                0x1.7e5131fc7a364p-441,
                0x1.7e5395940ae48p+212,
                0x1.7e53c0ba6b87dp390,
                0x1.7e553bf9ddc4ap-899,
                0x1.7e574b5f481f1p+932,
                0x1.7e57912838a52p197,
                0x1.7e5902ce0e151p-400,
                0x1.7e5b42da06b1fp-896,
                0x1.7e6f49fafb6cp-494,
                0x1.7e7a6e4b5c702p455,
                0x1.7e7a6e4b5c702p+456,
                0x1.7e7b072be2d84p962,
                0x1.7e813b27e3968p-669,
                0x1.7e8d6fde3bdep674,
                0x1.7e92e21b88f82p+148,
                0x1.7e98e913063e8p345,
                0x1.7e98e913063e8p346,
                0x1.7ea7119371409p-1022,
                0x1.7eb2279e12c66p473,
                0x1.7eb84080b6d12p-198,
                0x1.7eb84080b6d12p-199,
                0x1.7eb84080b6d12p-200,
                0x1.7eb84080b6d12p-201,
                0x1.7eb84080b6d12p-202,
                0x1.7eb9aa8cf1ddep-655,
                0x1.7ec0a895c1f16p-556,
                0x1.7ec0a895c1f16p-557,
                0x1.7ec26d75f83acp-974,
                0x1.7ec51c867ec96p+960,
                0x1.7ec51c867ec96p961,
                0x1.7ec51c867ec96p962,
                0x1.7ec51c867ec96p963,
                0x1.7ed3972fe3a9ap-460,
                0x1.7ed3972fe3a9ap-461,
                0x1.7eda8fa438c5p-973,
                0x1.7ede11f4a0b6p-570,
                0x1.7ede11f4a0b6p-571,
                0x1.7ee2e00570566p-945,
                0x1.7ee2e00570566p-946,
                0x1.7ee86f08bf3cap257,
                0x1.7ee86f08bf3cap258,
                0x1.7eef8b74b0228p-543,
                0x1.7ef943af7889ep-482,
                0x1.7efa70dcfd822p285,
                0x1.7efd75a2938ecp102,
                0x1.7f018e20e4802p749,
                0x1.7f037708d96b2p-885,
                0x1.7f037708d96b2p-886,
                0x1.7f0d6cf257aa1p-235,
                0x1.7f0e95a1f221ap-530,
                0x1.7f1fb6f10934cp-170,
                0x1.7f1p-1062,
                0x1.7f203339c9628p338,
                0x1.7f203339c9628p339,
                0x1.7f203339c9628p+340,
                0x1.7f203339c9628p341,
                0x1.7f203339c9628p342,
                0x1.7f203339c9628p343,
                0x1.7f203339c9628p+344,
                0x1.7f203339c9628p345,
                0x1.7f2096c30588p-286,
                0x1.7f2172ad6b7fdp-438,
                0x1.7f2ae47b0f33ep165,
                0x1.7f2ae47b0f33ep166,
                0x1.7f2e75d8f0213p-531,
                0x1.7f2e75d8f0213p-532,
                0x1.7f2e75d8f0213p-533,
                0x1.7f39122e7e29dp927,
                0x1.7f3b569a27632p-794,
                0x1.7f3b569a27632p-795,
                0x1.7f3b569a27632p-796,
                0x1.7f3f19a1bb8c8p-176,
                0x1.7f487b1782f1ep755,
                0x1.7f4a9b2b504c9p758,
                0x1.7f4b476441877p-130,
                0x1.7f5bf9d80cb22p663,
                0x1.7f6275a78e902p+452,
                0x1.7f65c1125fd92p818,
                0x1.7f65c1125fd92p819,
                0x1.7f6e23a79b346p-772,
                0x1.7f75b721c5cf6p+232,
                0x1.7f76dccb9957ap-88,
                0x1.7f76dccb9957ap-89,
                0x1.7f76dccb9957ap-90,
                0x1.7f76dccb9957ap-91,
                0x1.7f8278cce8eeap-37,
                0x1.7f85de8ad5c4ep315,
                0x1.7f85de8ad5c4fp315,
                0x1.7f86df8175d79p269,
                0x1.7f87bc4f79174p846,
                0x1.7f901b694fed1p-152,
                0x1.7f910f0fd92ccp650,
                0x1.7f9162cae8a2p-950,
                0x1.7f9162cae8a2p-951,
                0x1.7f965f952e3fp110,
                0x1.7f99b7de5f58ap-484,
                0x1.7f9ad8b3b8568p-792,
                0x1.7f9c3fe86f942p985,
                0x1.7f9c92093c12ep+688,
                0x1.7f9cc5f1f1e49p-960,
                0x1.7fa81668f8ac2p590,
                0x1.7fb17a9bb3a76p355,
                0x1.7fb2721acb6f4p-750,
                0x1.7fb2721acb6f4p-751,
                0x1.7fb2721acb6f4p-752,
                0x1.7fb7d4bb2feeep570,
                0x1.7fbbf7b43651ep+252,
                0x1.7fbbf7b43651ep253,
                0x1.7fbd555e31ee6p879,
                0x1.7fbdac4698b3dp+500,
                0x1.7fcb3cd41c1e2p855,
                0x1.7fd49915ec4c6p-83,
                0x1.7fd49915ec4c6p-84,
                0x1.7fd49915ec4c6p-85,
                0x1.7fd49915ec4c6p-86,
                0x1.7fd49915ec4c6p-87,
                0x1.7fec216198ddcp+800,
                0x1.7fec216198ddcp801,
                0x1.7ff4706a1ea94p-73,
                0x1.7ff4706a1ea94p-74,
                0x1.7ff4706a1ea94p-75,
                0x1.7ffb354456642p-291,
                0x1.7ffea023cd734p934,
                0x1.7p-1070,
                0x1.8005e5b081d73p-983,
                0x1.800ade999cd1dp-309,
                0x1.801234c3c65adp+948,
                0x1.8016d15f6ca5cp671,
                0x1.801787724f2cp425,
                0x1.801787724f2cp426,
                0x1.801ad8fdb4349p930,
                0x1.801cd0333fd18p-233,
                0x1.801e768c642f9p-605,
                0x1.801e768c642f9p-606,
                0x1.8020197dc0c48p230,
                0x1.802cb61f49a44p483,
                0x1.803adf26f3442p-105,
                0x1.804e50987a989p-967,
                0x1.804e50987a989p-968,
                0x1.804e50987a989p-969,
                0x1.8052fe8c47108p-47,
                0x1.805c19e680456p-1004,
                0x1.805c19e680456p-1005,
                0x1.805c19e680456p-1006,
                0x1.805c19e680456p-1007,
                0x1.805p-1060,
                0x1.805p-1061,
                0x1.8062864ac6f43p-851,
                0x1.8066636743b01p486,
                0x1.807dac70ccb76p-715,
                0x1.807dac70ccb76p-716,
                0x1.807dac70ccb76p-717,
                0x1.807dac70ccb76p-718,
                0x1.807dac70ccb76p-719,
                0x1.807dac70ccb76p-720,
                0x1.807dac70ccb76p-721,
                0x1.807dac70ccb76p-722,
                0x1.8082b9b827ea6p-978,
                0x1.8082b9b827ea6p-979,
                0x1.8082b9b827ea6p-980,
                0x1.8082b9b827ea6p-981,
                0x1.8082b9b827ea6p-982,
                0x1.8082b9b827ea6p-983,
                0x1.8082b9b827ea6p-984,
                0x1.808bbc654f39fp+932,
                0x1.808e3b72c108cp-811,
                0x1.80a940dd62104p542,
                0x1.80aa0724c7c96p410,
                0x1.80aae1496ccfcp+168,
                0x1.80bea1492517ap-565,
                0x1.80bea1492517ap-566,
                0x1.80bea1492517ap-567,
                0x1.80c903f7379f2p-365,
                0x1.80c903f7379f2p-366,
                0x1.80d65aba590b8p817,
                0x1.80db54e2bf028p-581,
                0x1.80e37857aeb74p-873,
                0x1.80e4f1bae43b4p-319,
                0x1.80e6521d47831p453,
                0x1.80eac3dd0ecc8p-128,
                0x1.80eac3dd0ecc8p-129,
                0x1.80eac3dd0ecc8p-130,
                0x1.80eac3dd0ecc8p-131,
                0x1.80ed6efdd86afp+612,
                0x1.80f482fff8902p-741,
                0x1.80f482fff8902p-742,
                0x1.80fa131177899p277,
                0x1.811da1cc7922ap141,
                0x1.811da1cc7922ap142,
                0x1.811da1cc7922ap143,
                0x1.811da1cc7922ap+144,
                0x1.8124c8e50cb07p631,
                0x1.812aff64b8a06p761,
                0x1.812aff64b8a06p762,
                0x1.812aff64b8a06p763,
                0x1.812c337427f8fp-233,
                0x1.812f9cf7920e2p119,
                0x1.8135804b7d4d7p273,
                0x1.813ad9e873343p454,
                0x1.81511e1b2c2dcp593,
                0x1.81511e1b2c2dcp594,
                0x1.8152a22d8dc9ap-677,
                0x1.8152a22d8dc9ap-678,
                0x1.8158a0835ed48p275,
                0x1.815d20db2857fp+968,
                0x1.816d4ca43169p-181,
                0x1.816d4ca43169p-182,
                0x1.817702671d445p717,
                0x1.81841a1113d93p-303,
                0x1.8192f34ace08bp-864,
                0x1.8192f34ace08bp-865,
                0x1.819651531f9e8p+604,
                0x1.819651531f9e8p605,
                0x1.819651531f9e8p606,
                0x1.8198976dbb235p-317,
                0x1.819f00e4daa8p533,
                0x1.819f00e4daa8p534,
                0x1.819f33c0ccfb4p-131,
                0x1.81a24d0830406p-462,
                0x1.81a24d0830406p-463,
                0x1.81a24d0830406p-464,
                0x1.81a24d0830406p-465,
                0x1.81b1835f23dbp793,
                0x1.81b1835f23dbp794,
                0x1.81b1835f23dbp795,
                0x1.81c1e790f9c02p229,
                0x1.81c1e790f9c02p230,
                0x1.81c1e790f9c02p231,
                0x1.81c2b0cc9a9bep-301,
                0x1.81c53342160a2p685,
                0x1.81c53342160a2p686,
                0x1.81c71edaff916p374,
                0x1.81c71edaff916p375,
                0x1.81dd109127526p769,
                0x1.81defe1294c7bp-824,
                0x1.81df94083909cp359,
                0x1.81df94083909cp+360,
                0x1.81e03f705857bp-16,
                0x1.81e431a4dd7a8p-701,
                0x1.81e92p-1055,
                0x1.81f656a84b0aep437,
                0x1.820bf6c360007p-579,
                0x1.820d39ap-1047,
                0x1.821e910a9aedfp-400,
                0x1.822307659cff2p-462,
                0x1.822507db6a8fep-870,
                0x1.822507db6a8fep-871,
                0x1.822507db6a8fep-872,
                0x1.822507db6a8fep-873,
                0x1.822507db6a8fep-874,
                0x1.8225f2a289231p902,
                0x1.8229bda003342p514,
                0x1.82390f5fe20b6p-204,
                0x1.823bd19a23531p313,
                0x1.824c342f0913ep-987,
                0x1.824c342f0913ep-988,
                0x1.825022b94b152p587,
                0x1.825022b94b152p+588,
                0x1.82576b21e37e7p729,
                0x1.82576b21e37e7p730,
                0x1.825ed60f3717bp-429,
                0x1.825ed60f3717bp-430,
                0x1.82701a0f5ecaep-342,
                0x1.82701a0f5ecaep-343,
                0x1.8274291c6065ap-562,
                0x1.8274291c6065bp-562,
                0x1.82782afe1869ep-70,
                0x1.82782afe1869ep-71,
                0x1.82782afe1869ep-72,
                0x1.82782afe1869ep-73,
                0x1.82782afe1869ep-74,
                0x1.82782afe1869ep-75,
                0x1.82782afe1869ep-76,
                0x1.82782afe1869ep-77,
                0x1.82782afe1869ep-78,
                0x1.82782afe1869ep-79,
                0x1.82782afe1869ep-80,
                0x1.827a35cfaf516p721,
                0x1.827a35cfaf516p722,
                0x1.827adc2058ec2p+800,
                0x1.827adc2058ec2p801,
                0x1.827adc2058ec2p802,
                0x1.827adc2058ec2p803,
                0x1.8280111754a8ap-569,
                0x1.8280111754a8ap-570,
                0x1.828a570aacbdep-72,
                0x1.8296a9ce8036ep-748,
                0x1.8296a9ce8036ep-749,
                0x1.8296a9ce8036ep-750,
                0x1.8296a9ce8036ep-751,
                0x1.8299381ab68bap678,
                0x1.829cbb02fa842p145,
                0x1.829cbb02fa842p146,
                0x1.829cbb02fa842p147,
                0x1.829cbb02fa842p+148,
                0x1.82b1fe41ac9e1p-558,
                0x1.82b1fe41ac9e1p-559,
                0x1.82b22e9088e4ep-367,
                0x1.82b22e9088e4ep-368,
                0x1.82b7798b1f23dp291,
                0x1.82b8569aa1c96p-1009,
                0x1.82bd749858bcap983,
                0x1.82bd749858bcap+984,
                0x1.82be7fc5475dep+184,
                0x1.82be7fc5475dep185,
                0x1.82be7fc5475dep186,
                0x1.82c02d6b5654dp929,
                0x1.82c53a573e931p-62,
                0x1.82ca08144dc4p689,
                0x1.82d72520a82f5p-666,
                0x1.82da42d81ac14p269,
                0x1.82daf1cbc780ep670,
                0x1.82db34012b251p-77,
                0x1.82db34012b252p-77,
                0x1.82e7ab8e5dbc8p-257,
                0x1.82ea82b18c909p953,
                0x1.83010aba78a54p966,
                0x1.83010aba78a54p967,
                0x1.83010aba78a54p+968,
                0x1.830171338a64ap-155,
                0x1.83073119f21d8p-36,
                0x1.83073119f21d8p-37,
                0x1.830889cf8c7cep-969,
                0x1.831919a316da1p-73,
                0x1.831f2b3144138p491,
                0x1.83238beab8p-1037,
                0x1.8326f87d4caep127,
                0x1.8327d2d7b3182p-702,
                0x1.832ba42fe0d16p+844,
                0x1.832f9c5d0f68fp+132,
                0x1.83380b12a69d3p-578,
                0x1.834036762ddeap+456,
                0x1.834036762ddeap457,
                0x1.83425a5f872f1p+408,
                0x1.8349fbf37e59cp694,
                0x1.834a523b75e3cp-957,
                0x1.834a523b75e3cp-958,
                0x1.834a523b75e3cp-959,
                0x1.834a523b75e3cp-960,
                0x1.834c5e3fa94f8p813,
                0x1.834c5e3fa94f8p814,
                0x1.834c5e3fa94f8p815,
                0x1.834c5e3fa94f8p+816,
                0x1.834c5e3fa94f8p817,
                0x1.834c5e3fa94f8p818,
                0x1.834c5e3fa94f8p819,
                0x1.8354941ea621p-686,
                0x1.8355f7eae78d9p-103,
                0x1.8355f7eae78d9p-104,
                0x1.8355f7eae78d9p-105,
                0x1.8356b7ccf0032p950,
                0x1.835b4cccd487p-448,
                0x1.835b4cccd487p-449,
                0x1.835b87bbdef89p-778,
                0x1.835c92b9d72ccp1021,
                0x1.836fe529afb5p469,
                0x1.837853eedbacap-181,
                0x1.837853eedbacap-182,
                0x1.837853eedbacap-183,
                0x1.837853eedbacap-184,
                0x1.837853eedbacap-185,
                0x1.83871863c88e9p994,
                0x1.83877627a875ap521,
                0x1.838a78e97324cp-466,
                0x1.838a78e97324cp-467,
                0x1.838c4dea98fbp267,
                0x1.8395688592fafp-519,
                0x1.8395688592fafp-520,
                0x1.83a0cbbec4a1fp-846,
                0x1.83a2ed7753daap573,
                0x1.83a2ed7753daap574,
                0x1.83a3e6a56ad12p990,
                0x1.83a99c3ec7ebp893,
                0x1.83ac0589ab9fdp842,
                0x1.83aeeda6f2c17p-1018,
                0x1.83aeeda6f2c17p-1019,
                0x1.83b01023952bbp322,
                0x1.83b0cf0abea1ap-923,
                0x1.83b0cf0abea1ap-924,
                0x1.83b0cf0abea1ap-925,
                0x1.83b35527f34p574,
                0x1.83ba3f00053b8p+1008,
                0x1.83c606402d74ap-484,
                0x1.83c920de69ad8p-369,
                0x1.83d56e8b5857p518,
                0x1.83e1167d4ad9ep581,
                0x1.83e1167d4ad9ep582,
                0x1.83e8bff154b04p-944,
                0x1.83eb3afa0371ap-910,
                0x1.83eb3afa0371ap-911,
                0x1.83eb3afa0371ap-912,
                0x1.83f4618fa262fp438,
                0x1.83f840b3815cp374,
                0x1.840c830009dfcp617,
                0x1.8411a1fa5a2dep-752,
                0x1.8414c9d448da2p-807,
                0x1.8416375d191e5p-278,
                0x1.84169118ea0b8p-197,
                0x1.841d5ec9d1afp-525,
                0x1.841d5ec9d1afp-526,
                0x1.841d5ec9d1afp-527,
                0x1.8421286c9bf6ap-758,
                0x1.8421286c9bf6bp-758,
                0x1.8421c5628eca8p225,
                0x1.8421c670905e9p737,
                0x1.8422010f3cb9p-870,
                0x1.84250f2be1ceep579,
                0x1.842c7b59b04a9p-423,
                0x1.842c7b59b04a9p-424,
                0x1.843e10734fa56p+276,
                0x1.843e10734fa56p277,
                0x1.8454da6f8ba85p733,
                0x1.8454da6f8ba85p734,
                0x1.8457aacf13abcp-383,
                0x1.8457aacf13abcp-384,
                0x1.8457aacf13abcp-385,
                0x1.8457aacf13abcp-386,
                0x1.8457aacf13abcp-387,
                0x1.845d606749a3cp-270,
                0x1.8470f34fc5b66p-129,
                0x1.84710f443502ap-76,
                0x1.8478bf5dd5a3dp-989,
                0x1.847d8e8ab03b5p-854,
                0x1.847e9f2d9711ap-290,
                0x1.8481de363ab04p-884,
                0x1.8481de363ab04p-885,
                0x1.8486eb10c3b64p+736,
                0x1.8488a5b44536p-273,
                0x1.8489c96711a8cp543,
                0x1.8489c96711a8cp+544,
                0x1.8497e9d060f5cp-476,
                0x1.84b58c7bcae6cp719,
                0x1.84b713111fc3ap-485,
                0x1.84b713111fc3ap-486,
                0x1.84b713111fc3ap-487,
                0x1.84b713111fc3ap-488,
                0x1.84b713111fc3ap-489,
                0x1.84c8f1a2595p-1030,
                0x1.84d53b525c50cp-126,
                0x1.84d690cc3915ep-328,
                0x1.84d8e1d137aaep363,
                0x1.84d8e1d137aaep+364,
                0x1.84e6190647e75p-950,
                0x1.84e91f4aa0fdbp819,
                0x1.84ea14c952148p-755,
                0x1.84eb2cc21f781p-355,
                0x1.84ebf82e41c32p649,
                0x1.84ebf82e41c32p650,
                0x1.84f03e93ff9f4p+212,
                0x1.84f03e93ff9f5p+212,
                0x1.84f0e4abd321ap-1013,
                0x1.84f0e4abd321ap-1014,
                0x1.84f0e4abd321ap-1015,
                0x1.84f0e4abd321ap-1016,
                0x1.84f0e4abd321ap-1017,
                0x1.84f0e4abd321ap-1018,
                0x1.84f0e4abd321ap-1019,
                0x1.85087075f8ecep-734,
                0x1.85087075f8ecep-735,
                0x1.851115f98844ep-246,
                0x1.8519645daeac1p+520,
                0x1.8519645daeac1p521,
                0x1.851b96f835326p+916,
                0x1.8520125e4cc48p-724,
                0x1.8526006e2014cp766,
                0x1.852707f200dc8p-243,
                0x1.85372e3c4a75p995,
                0x1.853a521a4ac36p-432,
                0x1.853cd079ed9cp-30,
                0x1.853cd079ed9cp-31,
                0x1.8544dddfa2472p469,
                0x1.8557f31326bbbp697,
                0x1.8557f31326bbcp694,
                0x1.8557f31326bbcp695,
                0x1.8557f31326bbcp+696,
                0x1.8557f31326bbcp697,
                0x1.8557f31326bbcp698,
                0x1.8557f31326bbcp699,
                0x1.8557f31326bbcp+700,
                0x1.8557f31326bbcp701,
                0x1.8557f31326bbcp702,
                0x1.8557f31326bbcp703,
                0x1.8557f31326bbcp+704,
                0x1.8557f31326bbcp705,
                0x1.8557f31326bbcp706,
                0x1.855ec94d21356p-219,
                0x1.855ec94d21356p-220,
                0x1.85617887e68e8p218,
                0x1.85617887e68e8p219,
                0x1.856c1984e1ccp-102,
                0x1.857da97a29d39p-752,
                0x1.8580d017619cap-964,
                0x1.858423699d07p-626,
                0x1.85a8760d4a48ap+876,
                0x1.85ab729479f3bp-775,
                0x1.85ac473d8870ap-763,
                0x1.85ap-1063,
                0x1.85b581342dabcp617,
                0x1.85bb23fd563c3p-529,
                0x1.85bb23fd563c3p-530,
                0x1.85bcf03e9f2b2p+792,
                0x1.85bcf03e9f2b2p793,
                0x1.85c483fde5efp317,
                0x1.85c95e1ca5ee4p-208,
                0x1.85cdf194p-1044,
                0x1.85ceae8de14ddp141,
                0x1.85d003f6488aep-954,
                0x1.85d003f6488afp-954,
                0x1.85d4b10934fc4p-854,
                0x1.85deecbd7f08p875,
                0x1.85e25136d9db8p-1025,
                0x1.85ebfef8a00a2p+208,
                0x1.85f281bcfffbep-857,
                0x1.85f56e95bfb07p555,
                0x1.85f5edd777fa8p-243,
                0x1.85f5edd777fa8p-244,
                0x1.86128ep-1049,
                0x1.8617e5901868cp431,
                0x1.8617e5901868cp+432,
                0x1.8617e5901868cp433,
                0x1.8617e5901868cp434,
                0x1.86241a0a1ad3dp473,
                0x1.86241a0a1ad3dp474,
                0x1.8629674db3041p-616,
                0x1.862c8p-1057,
                0x1.862e4764d6d1ap-210,
                0x1.8637a9424429cp733,
                0x1.8637a9424429cp734,
                0x1.8637f41fcad32p-469,
                0x1.863abce6b82ecp-398,
                0x1.863be03f37e42p867,
                0x1.863be03f37e42p+868,
                0x1.863be03f37e42p869,
                0x1.863be03f37e42p870,
                0x1.863be03f37e42p871,
                0x1.8650d8093a5c6p971,
                0x1.8650d8093a5c6p+972,
                0x1.86528c022c128p+388,
                0x1.8655ec0301e3p495,
                0x1.8655ec0301e3p+496,
                0x1.86566a53ece25p-652,
                0x1.86573ccca10eap989,
                0x1.865a2a368p-1041,
                0x1.865e84372e0f2p+388,
                0x1.8674e7f7de5fep990,
                0x1.867be78b38f16p-771,
                0x1.867be78b38f16p-772,
                0x1.8691a0257aba5p330,
                0x1.8696da66d6f1p335,
                0x1.869bd839421b6p333,
                0x1.869bf335e2c3ep937,
                0x1.869bf335e2c3ep938,
                0x1.869bf335e2c3ep939,
                0x1.869bf335e2c3ep+940,
                0x1.869bf335e2c3ep941,
                0x1.869bf335e2c3ep942,
                0x1.86ad022aa2d88p-436,
                0x1.86ad022aa2d88p-437,
                0x1.86ap+16,
                0x1.86b677fdb29fbp+288,
                0x1.86b871c906a2p-55,
                0x1.86b871c906a2p-56,
                0x1.86bcf03203e14p+572,
                0x1.86d37296ca126p-515,
                0x1.86d37296ca126p-516,
                0x1.86d37296ca126p-517,
                0x1.86d37296ca126p-518,
                0x1.86dc5b8283844p219,
                0x1.86f50aadceedfp-489,
                0x1.86f653140a659p-20,
                0x1.86f653140a659p-21,
                0x1.8708279e4bc5ap501,
                0x1.8708279e4bc5bp501,
                0x1.8713fd72dacfep+184,
                0x1.871794e6c7ba5p-640,
                0x1.87203d29dbc06p-720,
                0x1.872f563ae0cc9p-56,
                0x1.8734eaa17951fp-123,
                0x1.8739b24fc1bd6p-440,
                0x1.8739d11afbffdp+892,
                0x1.873cb93afbad8p293,
                0x1.8741afd097994p-430,
                0x1.87504926f2d32p-267,
                0x1.8750ea732fdaep242,
                0x1.8750ea732fdaep243,
                0x1.8750ea732fdaep+244,
                0x1.8750ea732fdaep245,
                0x1.8750ea732fdaep246,
                0x1.8750ea732fdaep247,
                0x1.875250f843bp877,
                0x1.875840fff19b4p-535,
                0x1.875840fff19b4p-536,
                0x1.875840fff19b4p-537,
                0x1.875840fff19b4p-538,
                0x1.875840fff19b4p-539,
                0x1.875840fff19b4p-540,
                0x1.875840fff19b4p-541,
                0x1.875840fff19b4p-542,
                0x1.875840fff19b4p-543,
                0x1.875d15965a912p455,
                0x1.875d32fa48c08p-751,
                0x1.875d32fa48c08p-752,
                0x1.877015b7f0772p-693,
                0x1.877015b7f0772p-694,
                0x1.87706b0213d0ap986,
                0x1.877d9d459cc03p805,
                0x1.877fa0260beb2p-78,
                0x1.877fa0260beb2p-79,
                0x1.8784f12575803p+216,
                0x1.8789782272cb2p+500,
                0x1.878ce846a5b68p-269,
                0x1.878ce846a5b68p-270,
                0x1.87916c48b9197p-516,
                0x1.87a3b13002847p+540,
                0x1.87a599a9bbce2p-588,
                0x1.87a7016ccf102p-685,
                0x1.87a7016ccf102p-686,
                0x1.87a8b7d05fdd4p-156,
                0x1.87a938df46469p+212,
                0x1.87acdd49b2538p-483,
                0x1.87b3859f3c926p157,
                0x1.87b3859f3c926p158,
                0x1.87b809b494507p206,
                0x1.87b94f9cdb2b4p-60,
                0x1.87ba197c5d406p602,
                0x1.87bb73e4262c2p741,
                0x1.87bd9dd38f873p225,
                0x1.87bd9dd38f873p226,
                0x1.87be96d099726p335,
                0x1.87c92fe1880b4p-1024,
                0x1.87cc4ec675p-1034,
                0x1.87cfd6fe6c386p854,
                0x1.87d469cb69ddp129,
                0x1.87d469cb69ddp130,
                0x1.87d469cb69ddp131,
                0x1.87d7d8b78b142p-431,
                0x1.87d7d8b78b142p-432,
                0x1.87d8bdfa72d94p-222,
                0x1.87dcbf6ad5cf8p425,
                0x1.87dcbf6ad5cf8p426,
                0x1.87dcbf6ad5cf8p427,
                0x1.87dcbf6ad5cf8p+428,
                0x1.87e112534bb4p302,
                0x1.87e112534bb4p303,
                0x1.87e112534bb4p+304,
                0x1.87e92154ef7acp-665,
                0x1.87e96ab7494bep-324,
                0x1.8801ddb39a7afp-607,
                0x1.88036854fa76ap-564,
                0x1.88036854fa76ap-565,
                0x1.880a3d515e849p-87,
                0x1.880a3d515e849p-88,
                0x1.880d446ade4a6p-832,
                0x1.880d446ade4a6p-833,
                0x1.880d446ade4a6p-834,
                0x1.880d446ade4a6p-835,
                0x1.88102p-1055,
                0x1.88156e908p-1040,
                0x1.8822870bd6d61p581,
                0x1.8823a57adbef8p-492,
                0x1.8823a57adbef8p-493,
                0x1.8823a57adbef8p-494,
                0x1.8823a57adbef8p-495,
                0x1.8823a57adbef8p-496,
                0x1.8823a57adbef8p-497,
                0x1.8823a57adbef8p-498,
                0x1.8823a57adbef8p-499,
                0x1.88294716519b6p+252,
                0x1.883597689cdeep+512,
                0x1.88399036476a2p-806,
                0x1.88399036476a2p-807,
                0x1.883eb5469b9dcp-741,
                0x1.883fc1824856p-523,
                0x1.883fc1824856p-524,
                0x1.8843ec274e479p119,
                0x1.88487d0a8f64fp365,
                0x1.8849dd33c95afp-48,
                0x1.8849dd33c95afp-49,
                0x1.884a84d44914bp-216,
                0x1.884e8f9fc5fa2p-497,
                0x1.884e8f9fc5fa2p-498,
                0x1.884e8f9fc5fa2p-499,
                0x1.884e8f9fc5fa2p-500,
                0x1.884e8f9fc5fa2p-501,
                0x1.884e8f9fc5fa2p-502,
                0x1.8851a0b548ea4p-180,
                0x1.8859a9e36328p789,
                0x1.8859a9e36328p790,
                0x1.8865ee928d6bdp-979,
                0x1.8869e04dc931p358,
                0x1.886c20f56c4ebp-467,
                0x1.8871a8c9269fdp-378,
                0x1.88736696231dap630,
                0x1.88736696231dap631,
                0x1.88736696231dap+632,
                0x1.88736696231dap633,
                0x1.88736696231dap634,
                0x1.88792f30c526p-721,
                0x1.88792f30c526p-722,
                0x1.8882a0c211219p522,
                0x1.888cd3b23c89ap1018,
                0x1.888cd3b23c89ap1019,
                0x1.888f58233badap-772,
                0x1.8891a2f029f6cp-408,
                0x1.889a0937677dp-696,
                0x1.889a0937677dp-697,
                0x1.889adep-1051,
                0x1.889d46c2edceep-411,
                0x1.88a07ebcfcb18p-71,
                0x1.88a4036fa081dp-690,
                0x1.88b17558fd59cp885,
                0x1.88ba3bf284e24p305,
                0x1.88bb4313ed68p259,
                0x1.88c1ea951716ep759,
                0x1.88c1ea951716ep+760,
                0x1.88d4ca5a2f1e2p-637,
                0x1.88e29a9d7c5b9p-45,
                0x1.88e9d00be8a7cp-736,
                0x1.88e9d00be8a7cp-737,
                0x1.88f6a1b543ad8p-181,
                0x1.88f96dc9e51dp918,
                0x1.88fd2fe46a76ep+268,
                0x1.8902d40d3df4bp469,
                0x1.8909d470cbb85p298,
                0x1.890bb55fe1388p+272,
                0x1.890bb55fe1388p273,
                0x1.890bb55fe1388p274,
                0x1.890bb55fe1388p275,
                0x1.8922d6942f82ap-770,
                0x1.8922f31411456p790,
                0x1.8922f31411456p791,
                0x1.8926842427ba1p549,
                0x1.892cedc62e548p329,
                0x1.892cedc62e548p330,
                0x1.892d04a288664p231,
                0x1.8934a75f8ada8p-702,
                0x1.893b3f6956116p-585,
                0x1.89434286cd7dep789,
                0x1.89434286cd7dep790,
                0x1.89434286cd7dep791,
                0x1.89434286cd7dep+792,
                0x1.8943e1d185c7ap198,
                0x1.8943e1d185c7ap199,
                0x1.8943e1d185c7ap+200,
                0x1.8943e1d185c7ap201,
                0x1.8943e23c2d4b7p366,
                0x1.89457e03ac9b2p+884,
                0x1.894ca6883ce4ep+732,
                0x1.894d3661a88aep315,
                0x1.896428f9ecfafp658,
                0x1.89683f7016aa6p-523,
                0x1.896beb0c66eb8p+696,
                0x1.896d32c1b8d9dp+344,
                0x1.897915fb21ca3p-470,
                0x1.897fa529e3c78p613,
                0x1.89821bbc321e6p-170,
                0x1.89821bbc321e6p-171,
                0x1.899c2f673221p-857,
                0x1.899c2f673221p-858,
                0x1.899c2f673221p-859,
                0x1.899c2f673221p-860,
                0x1.899c2f673221p-861,
                0x1.899c2f673221p-862,
                0x1.899c2f673221p-863,
                0x1.899c454225b96p-628,
                0x1.899ccbf754f6dp-544,
                0x1.89a2d9a91b976p-373,
                0x1.89acc91a18846p+372,
                0x1.89acc91a18846p373,
                0x1.89ace98b239b6p-520,
                0x1.89c090a3dd54ep-692,
                0x1.89c85d1e1a7b4p+788,
                0x1.89cbcc2abf19ep290,
                0x1.89ce220f304d2p583,
                0x1.89cp-1064,
                0x1.89d0af534aa09p182,
                0x1.89d3f26b8334cp-220,
                0x1.89d3f26b8334cp-221,
                0x1.89d3f26b8334cp-222,
                0x1.89deefe30d9a6p+928,
                0x1.89dff9bb6d9ap-970,
                0x1.89ecbbcaa7132p1013,
                0x1.89eee4206e21ep-314,
                0x1.89f1498321f5ap-255,
                0x1.89f1498321f5ap-256,
                0x1.89f1498321f5ap-257,
                0x1.89f2e02fb58e6p625,
                0x1.89f2e02fb58e6p626,
                0x1.89f2e02fb58e6p627,
                0x1.89f7397p-1046,
                0x1.8a02da961ba87p+632,
                0x1.8a0522c7e7095p-376,
                0x1.8a0918af8d472p961,
                0x1.8a0a1af490b9ap1018,
                0x1.8a0c50792930ep-1002,
                0x1.8a0c50792930ep-1003,
                0x1.8a0c50792930ep-1004,
                0x1.8a0c8a54f2542p675,
                0x1.8a10a1b4047b2p-41,
                0x1.8a31e261b91d6p-359,
                0x1.8a49f1e308b8bp-965,
                0x1.8a4a4db17a6a4p202,
                0x1.8a4a4db17a6a4p203,
                0x1.8a5075dd90beep675,
                0x1.8a63375197c85p-354,
                0x1.8a6e32246c99cp109,
                0x1.8a6e32246c99cp110,
                0x1.8a6ff9e12a876p-835,
                0x1.8a6ff9e12a876p-836,
                0x1.8a7a1f7cdfedep1017,
                0x1.8a7a1f7cdfedep1018,
                0x1.8a7a1f7cdfedep1019,
                0x1.8a7a1f7cdfedep+1020,
                0x1.8a7a1f7cdfedep1021,
                0x1.8a7a1f7cdfedep1022,
                0x1.8a85eb277e644p+268,
                0x1.8a85eb277e644p269,
                0x1.8a85eb277e644p270,
                0x1.8a85eb277e644p271,
                0x1.8a85eb277e644p+272,
                0x1.8a85eb277e644p273,
                0x1.8a85eb277e644p274,
                0x1.8a85eb277e644p275,
                0x1.8a85eb277e644p+276,
                0x1.8a861c9e2b55bp-367,
                0x1.8a8ce5d081b8ap+776,
                0x1.8a8f9809371bep959,
                0x1.8a94bb619856cp1011,
                0x1.8a9d24c1f25f2p+580,
                0x1.8aa55f4bff13fp585,
                0x1.8aa591e997f2dp929,
                0x1.8abf7043ea1dap+440,
                0x1.8ac755e26e9dcp-243,
                0x1.8ac79f8487566p+812,
                0x1.8ac8c79e1ff18p-894,
                0x1.8ac8c79e1ff18p-895,
                0x1.8ac8c79e1ff18p-896,
                0x1.8ac8c79e1ff18p-897,
                0x1.8ac8c79e1ff18p-898,
                0x1.8ac8c79e1ff18p-899,
                0x1.8ad6d5b0c7f6p-103,
                0x1.8ad6d5b0c7f6p-104,
                0x1.8ad74fda888e5p-89,
                0x1.8ad75d8438f43p594,
                0x1.8ae2e29288b03p929,
                0x1.8ae555821bffcp+460,
                0x1.8ae574079904ep-854,
                0x1.8ae574079904ep-855,
                0x1.8ae574079904ep-856,
                0x1.8ae7862e3a28ep463,
                0x1.8af070ea2513cp-286,
                0x1.8af070ea2513cp-287,
                0x1.8af070ea2513cp-288,
                0x1.8af46ec9d7c4p-613,
                0x1.8af46ec9d7c4p-614,
                0x1.8af46ec9d7c4p-615,
                0x1.8af46ec9d7c4p-616,
                0x1.8af7cb0c01b42p-540,
                0x1.8af7cb0c01b42p-541,
                0x1.8af7cb0c01b42p-542,
                0x1.8af7cb0c01b42p-543,
                0x1.8af7cb0c01b42p-544,
                0x1.8afd2657c083p-928,
                0x1.8afd2657c083p-929,
                0x1.8afd2657c083p-930,
                0x1.8afe10a2a66aap-572,
                0x1.8afe10a2a66aap-573,
                0x1.8aff2d1b663bbp827,
                0x1.8aff2d1b663bbp+828,
                0x1.8b03fd6894b66p-912,
                0x1.8b03fd6894b66p-913,
                0x1.8b04430d086e3p634,
                0x1.8b09b9798eabdp-316,
                0x1.8b0a8cb3ca432p731,
                0x1.8b0ade88519d8p+232,
                0x1.8b0bc363f3718p-669,
                0x1.8b13f4c3a2122p+256,
                0x1.8b13f4c3a2122p257,
                0x1.8b1c45fa869a8p-756,
                0x1.8b1c45fa869a8p-757,
                0x1.8b1c45fa869a8p-758,
                0x1.8b21ab9ccbbf6p575,
                0x1.8b35e39c9e38dp281,
                0x1.8b3bd05a11cecp-882,
                0x1.8b3bd05a11cecp-883,
                0x1.8b3bd05a11cecp-884,
                0x1.8b3bd05a11cecp-885,
                0x1.8b47c96085908p-784,
                0x1.8b512p-1055,
                0x1.8b51p-1057,
                0x1.8b5519aa50023p+280,
                0x1.8b5ca9081de68p-722,
                0x1.8b5f573f03268p-630,
                0x1.8b65c5a5e95a6p1015,
                0x1.8b6fdfd2d5a64p+864,
                0x1.8b6fdfd2d5a64p865,
                0x1.8b6fdfd2d5a64p866,
                0x1.8b70114f7f58dp-918,
                0x1.8b7e13cd91052p-1015,
                0x1.8b7e13cd91052p-1016,
                0x1.8b7e13cd91052p-1017,
                0x1.8b827fa1a0cf2p-7,
                0x1.8b8f6ef1003b1p807,
                0x1.8b8f6ef1003b1p+808,
                0x1.8b8feda2f6dd2p-696,
                0x1.8b92832ace70ap-21,
                0x1.8b98b9b019d74p+192,
                0x1.8b98b9b019d74p193,
                0x1.8b98b9b019d74p194,
                0x1.8ba5f18b2837ap-488,
                0x1.8ba5f18b2837ap-489,
                0x1.8ba786cd63a31p-243,
                0x1.8badd8acfd952p715,
                0x1.8baf38ad3e5f2p677,
                0x1.8baf38ad3e5f2p678,
                0x1.8bb04ef8c689bp706,
                0x1.8bb9869627bbep+588,
                0x1.8bba884e35a7ap-572,
                0x1.8bc58881addb2p-988,
                0x1.8bc613fd39835p-237,
                0x1.8bc9c59d2613ep-766,
                0x1.8bc9c59d2613ep-767,
                0x1.8bc9c59d2613ep-768,
                0x1.8bc9c59d2613ep-769,
                0x1.8bc9c59d2613ep-770,
                0x1.8bc9c59d2613ep-771,
                0x1.8bc9c59d2613ep-772,
                0x1.8bc9c59d2613ep-773,
                0x1.8bd102d09f4e6p-123,
                0x1.8bd102d09f4e6p-124,
                0x1.8bdfc36cfd358p419,
                0x1.8be02c582d264p-487,
                0x1.8be494f1c992p-427,
                0x1.8be494f1c992p-428,
                0x1.8bf5e5272ebd2p943,
                0x1.8bf5e5272ebd2p+944,
                0x1.8bf5e5272ebd2p945,
                0x1.8bf5e5272ebd2p946,
                0x1.8bf5e5272ebd2p947,
                0x1.8bf6cbe14305cp725,
                0x1.8bf7e7fa6f02ap-196,
                0x1.8bf7e7fa6f02ap-197,
                0x1.8bf7e7fa6f02ap-198,
                0x1.8bf7e7fa6f02ap-199,
                0x1.8bf7e7fa6f02ap-200,
                0x1.8bf7e7fa6f02ap-201,
                0x1.8bf8f81f63e07p-341,
                0x1.8c112521099f2p551,
                0x1.8c17d05939eeep-679,
                0x1.8c1d0c2b3a6dp+772,
                0x1.8c1ee26f699bp+296,
                0x1.8c240c4aecb14p-87,
                0x1.8c249ec4924f9p230,
                0x1.8c2bb4c3405b6p-874,
                0x1.8c2c2f6b8b00dp-925,
                0x1.8c2dcd94f9b7fp-630,
                0x1.8c3eb7cd9efa6p-681,
                0x1.8c4df0b2c179ap-274,
                0x1.8c50d3f049fe6p-461,
                0x1.8c50d3f049fe6p-462,
                0x1.8c58789a35539p-50,
                0x1.8c5cadd845bf8p-330,
                0x1.8c631a5471f27p-661,
                0x1.8c631a5471f27p-662,
                0x1.8c63e20e965f4p558,
                0x1.8c6c359db6848p542,
                0x1.8c74ed9d2fd2dp313,
                0x1.8c7cbbb955136p+688,
                0x1.8c7cbbb955136p689,
                0x1.8c81382p-1046,
                0x1.8c81382p-1047,
                0x1.8c84a32c66ec6p-590,
                0x1.8c84a32c66ec6p-591,
                0x1.8c84a32c66ec6p-592,
                0x1.8c84a32c66ec6p-593,
                0x1.8c84a32c66ec6p-594,
                0x1.8c87154dff6c6p-73,
                0x1.8c87154dff6c6p-74,
                0x1.8c87154dff6c6p-75,
                0x1.8c87154dff6c6p-76,
                0x1.8c8dac6a0342ap398,
                0x1.8c9b331e4ae51p-780,
                0x1.8c9da1507fcfp602,
                0x1.8ca3864ee0f3ap303,
                0x1.8ca79cb79c5dap-402,
                0x1.8ca79cb79c5dap-403,
                0x1.8ca79cb79c5dap-404,
                0x1.8ca79cb79c5dap-405,
                0x1.8ca79cb79c5dap-406,
                0x1.8ca891d088ccep-35,
                0x1.8ca891d088ccep-36,
                0x1.8ca891d088ccep-37,
                0x1.8ca891d088ccep-38,
                0x1.8cad7bd2aabfcp829,
                0x1.8cad7bd2aabfcp830,
                0x1.8cae82c8dc46cp-964,
                0x1.8cae82c8dc46cp-965,
                0x1.8cb36e128da0cp1014,
                0x1.8cbab63fcd62bp137,
                0x1.8cc714aad9d12p-59,
                0x1.8cc8fac4386e5p-153,
                0x1.8cd33ba19cef2p675,
                0x1.8cd634052a68ep-865,
                0x1.8cd634052a68ep-866,
                0x1.8cd7eaa5c0b22p-916,
                0x1.8cd7eaa5c0b22p-917,
                0x1.8cdecf812ba1ap1010,
                0x1.8cdf015fec86cp-505,
                0x1.8cdf015fec86cp-506,
                0x1.8cdf015fec86cp-507,
                0x1.8ce0769d4ce3dp574,
                0x1.8cf768b2f9c5ap883,
                0x1.8cf9ca73c3708p799,
                0x1.8cf9ca73c3708p+800,
                0x1.8cf9ca73c3708p801,
                0x1.8cfc43eaf0496p-636,
                0x1.8cfc43eaf0496p-637,
                0x1.8cfd2cc245478p-890,
                0x1.8cfd2cc245478p-891,
                0x1.8cfd2cc245478p-892,
                0x1.8cfd2cc245478p-893,
                0x1.8cfd2cc245478p-894,
                0x1.8cp-1065,
                0x1.8cp-1066,
                0x1.8cp-1067,
                0x1.8d015d580354p+564,
                0x1.8d01ef6a5eb7ap-99,
                0x1.8d0b09315c5fcp493,
                0x1.8d10a1ced69p-1030,
                0x1.8d1914a5540f4p-561,
                0x1.8d1914a5540f4p-562,
                0x1.8d2452aace1aep283,
                0x1.8d2452aace1aep+284,
                0x1.8d31ebd493ee3p+208,
                0x1.8d35148208588p+548,
                0x1.8d6253c5b3736p-436,
                0x1.8d6a8535ccp-1035,
                0x1.8d6a8672a0c78p978,
                0x1.8d6a8672a0c78p979,
                0x1.8d6a8672a0c78p+980,
                0x1.8d6a8672a0c78p981,
                0x1.8d71d360e13e2p-768,
                0x1.8d728defda4c5p-125,
                0x1.8d7f77cc4c6c2p-290,
                0x1.8d7f77cc4c6c2p-291,
                0x1.8d8b586fb3e8dp-361,
                0x1.8d9ceb789b57p850,
                0x1.8da4ebb095bf6p-27,
                0x1.8dac9aa0bdc7p499,
                0x1.8db1b7c16e48ep-313,
                0x1.8dbb0e7428472p198,
                0x1.8dc48bc8ce82cp730,
                0x1.8dcae8b90cda2p+296,
                0x1.8dcd199147674p-397,
                0x1.8dcdcd745585fp-871,
                0x1.8dcdcd745585fp-872,
                0x1.8dcdf93242308p-959,
                0x1.8dd600e50bd62p-500,
                0x1.8dda663d4f004p373,
                0x1.8ddbcc7f40ba6p-283,
                0x1.8de02685b6f4ep557,
                0x1.8de2f977a8cbbp-956,
                0x1.8de61f1b48c18p-397,
                0x1.8df8ddc3a1f42p705,
                0x1.8e09df9ca4876p867,
                0x1.8e0bfb98864c8p-300,
                0x1.8e0bfb98864c8p-301,
                0x1.8e0bfb98864c8p-302,
                0x1.8e0bfb98864c8p-303,
                0x1.8e0bfb98864c8p-304,
                0x1.8e0bfb98864c8p-305,
                0x1.8e0bfb98864c8p-306,
                0x1.8e0bfb98864c8p-307,
                0x1.8e0f575e6104fp839,
                0x1.8e1050255105dp+408,
                0x1.8e14120438825p-105,
                0x1.8e153010ed393p755,
                0x1.8e1cc52faf412p162,
                0x1.8e23a7e6b9ed5p-70,
                0x1.8e3758487b991p-199,
                0x1.8e3d84cc2e9dp637,
                0x1.8e3fc4fe84a4p+504,
                0x1.8e3fc4fe84a4p505,
                0x1.8e3fc4fe84a4p506,
                0x1.8e45e1df3b015p202,
                0x1.8e45e1df3b016p202,
                0x1.8e45e1df3b016p203,
                0x1.8e61a3c7defe6p543,
                0x1.8e61a3c7defe6p+544,
                0x1.8e63f26f85db6p-310,
                0x1.8e63f26f85db6p-311,
                0x1.8e63f7cf5313cp-26,
                0x1.8e63f7cf5313cp-27,
                0x1.8e63f7cf5313cp-28,
                0x1.8e63f7cf5313cp-29,
                0x1.8e63f7cf5313cp-30,
                0x1.8e6fc219054d8p-1024,
                0x1.8e6fc219054d8p-1025,
                0x1.8e700fd35d32dp-540,
                0x1.8e712ab9d3cdfp715,
                0x1.8e72db14ac828p-871,
                0x1.8e72db14ac828p-872,
                0x1.8e732b4d03d06p242,
                0x1.8e734f2622ff8p-706,
                0x1.8e734f2622ff8p-707,
                0x1.8e7d54e964c08p183,
                0x1.8e835895fb578p478,
                0x1.8e83fe1a3d098p-214,
                0x1.8e83fe1a3d098p-215,
                0x1.8e95e3244733ep-520,
                0x1.8ea26ca9ae52ap-310,
                0x1.8eb0138858d0ap687,
                0x1.8eb3638c413p-80,
                0x1.8ec4b5c78f26ep-112,
                0x1.8ec4b5c78f26ep-113,
                0x1.8ecf7037c025p-175,
                0x1.8ecf7037c025p-176,
                0x1.8ed4a637e260fp-980,
                0x1.8ed4a637e260fp-981,
                0x1.8ed7f893aeffep-779,
                0x1.8edf45674b471p-1018,
                0x1.8ee272a676957p-644,
                0x1.8eea820bc5e82p194,
                0x1.8efc5dba963d7p-281,
                0x1.8efc5dba963d7p-282,
                0x1.8f00cbfad6712p581,
                0x1.8f0385717ee84p655,
                0x1.8f1ec7de79dbcp-702,
                0x1.8f1ec7de79dbcp-703,
                0x1.8f1f3p-1054,
                0x1.8f27abf3c4c82p718,
                0x1.8f27abf3c4c82p719,
                0x1.8f27abf3c4c82p+720,
                0x1.8f27abf3c4c82p721,
                0x1.8f27abf3c4c82p722,
                0x1.8f27abf3c4c82p723,
                0x1.8f2b061aea072p-960,
                0x1.8f2b061aea072p-961,
                0x1.8f2b061aea072p-962,
                0x1.8f2b061aea072p-963,
                0x1.8f2b061aea072p-964,
                0x1.8f2b061aea072p-965,
                0x1.8f2b168624646p167,
                0x1.8f2b168624646p+168,
                0x1.8f2b168624646p169,
                0x1.8f2f0941cb69ap373,
                0x1.8f2f0941cb69ap374,
                0x1.8f2f0941cb69ap375,
                0x1.8f2f0941cb69ap+376,
                0x1.8f34d8c82f50ep985,
                0x1.8f34d8c82f50ep986,
                0x1.8f425130a5e1ap-294,
                0x1.8f43b2131bba6p-426,
                0x1.8f477ee0618a7p+796,
                0x1.8f477ee0618a7p797,
                0x1.8f4b6774575b2p-351,
                0x1.8f4effba763f4p-52,
                0x1.8f4effba763f4p-53,
                0x1.8f51eb10a3bb2p-347,
                0x1.8f51eb10a3bb2p-348,
                0x1.8f5372b4c0426p+448,
                0x1.8f5b2995a60f7p-731,
                0x1.8f5b6961036dbp918,
                0x1.8f5b813b701a3p181,
                0x1.8f65feb9ae786p+904,
                0x1.8f65feb9ae786p905,
                0x1.8f8b4dd16f1dfp-11,
                0x1.8f8d79bd707a9p-815,
                0x1.8f938e37b864p215,
                0x1.8f938e37b864p+216,
                0x1.8f9574dcf8a7p-478,
                0x1.8f9574dcf8a7p-479,
                0x1.8f997915ba41ap858,
                0x1.8f9cccff88573p-521,
                0x1.8fa113ff82398p-204,
                0x1.8fa3102e215b4p447,
                0x1.8fa63afcbb9p117,
                0x1.8fa63afcbb9p118,
                0x1.8fa63afcbb9p119,
                0x1.8fa63afcbb9p+120,
                0x1.8fa63afcbb9p121,
                0x1.8fa63afcbb9p122,
                0x1.8fa63afcbb9p123,
                0x1.8fa63afcbb9p+124,
                0x1.8fa9fb83132p+712,
                0x1.8fa9fb83132p713,
                0x1.8fa9fb83132p714,
                0x1.8fa9fb83132p715,
                0x1.8fa9fb83132p+716,
                0x1.8faa34fdae498p-906,
                0x1.8faa66a9fc496p+612,
                0x1.8faa66a9fc496p613,
                0x1.8fbca5753334cp-929,
                0x1.8fc855f11897cp-76,
                0x1.8fc855f11897cp-77,
                0x1.8fd649d43cc3cp+504,
                0x1.8fd82085edb2bp610,
                0x1.8fe462137668ap+824,
                0x1.8fe77496cd7a6p-234,
                0x1.8ff131b82e08cp-41,
                0x1.8ff131b82e08cp-42,
                0x1.8ff131b82e08cp-43,
                0x1.8ffc60cb8a922p-183,
                0x1.8ffc8ffb01296p-664,
                0x1.8ffc8ffb01296p-665,
                0x1.8p-1069,
                0x1.8p-1070,
                0x1.8p-1071,
                0x1.8p-1072,
                0x1.900098b603fa2p-33,
                0x1.900222e762792p-426,
                0x1.9014b3a1e47f6p-792,
                0x1.9014b3a1e47f6p-793,
                0x1.901af52cd39eap351,
                0x1.9024a0eb2a8bep-82,
                0x1.9024a0eb2a8bep-83,
                0x1.9025d959654c6p370,
                0x1.902d375306a58p-369,
                0x1.90381bead899cp+924,
                0x1.90381bead899cp925,
                0x1.90381bead899cp926,
                0x1.90381bead899cp927,
                0x1.9039d13d5d862p-335,
                0x1.9039d13d5d862p-336,
                0x1.904293d7b683ap-1016,
                0x1.904d2cccf7f64p757,
                0x1.904d2cccf7f64p758,
                0x1.904d2cccf7f64p759,
                0x1.904d2cccf7f64p+760,
                0x1.904d2cccf7f64p761,
                0x1.90501445d7ff8p-847,
                0x1.90529a37b7e22p925,
                0x1.90529a37b7e22p926,
                0x1.906aa78b912ccp491,
                0x1.906aa78b912ccp+492,
                0x1.90738d4050332p-544,
                0x1.90738edb4a17p-1016,
                0x1.9078d04b863dp591,
                0x1.9078d04b863dp+592,
                0x1.90996bc796978p-209,
                0x1.90a3df6673688p866,
                0x1.90aa5e8b8d344p-646,
                0x1.90aa5e8b8d344p-647,
                0x1.90aa5e8b8d344p-648,
                0x1.90aa5e8b8d344p-649,
                0x1.90aa5e8b8d344p-650,
                0x1.90af93e98dc8bp+660,
                0x1.90af93e98dc8bp661,
                0x1.90b42ca40e85dp-584,
                0x1.90b42ca40e85dp-585,
                0x1.90bae27ae3616p-750,
                0x1.90bd7a1276a12p547,
                0x1.90bd7a1276a12p+548,
                0x1.90bd7a1276a12p549,
                0x1.90bd7a1276a12p550,
                0x1.90bd7a1276a12p551,
                0x1.90c5c89eaf53cp-272,
                0x1.90cf9fcfc8df4p-158,
                0x1.90d320c9cfb04p-335,
                0x1.90d4904169024p798,
                0x1.90d56b873f4c6p975,
                0x1.90d56b873f4c6p+976,
                0x1.90d56b873f4c6p977,
                0x1.90d56b873f4c6p978,
                0x1.90d56b873f4c6p979,
                0x1.90d6b04fea9cep-59,
                0x1.90e1831fba6b8p639,
                0x1.90e2c4b9e1778p523,
                0x1.90ed765c2181p-349,
                0x1.90fa448312899p-459,
                0x1.90ffef7fb037ep711,
                0x1.90ffef7fb037ep+712,
                0x1.9102f1cffda8ep715,
                0x1.9117a3af13368p-773,
                0x1.9117c0889f9c4p217,
                0x1.912a355f172b2p813,
                0x1.912a355f172b2p814,
                0x1.912a355f172b2p815,
                0x1.912e6e5f86c6cp-59,
                0x1.91373da2c65dap798,
                0x1.91373da2c65dap799,
                0x1.9139399b5f07cp990,
                0x1.9139399b5f07cp991,
                0x1.913af49aee4dfp-702,
                0x1.9147b9330eaaep170,
                0x1.9147b9330eaaep171,
                0x1.9147b9330eaaep+172,
                0x1.9147b9330eaaep173,
                0x1.9147b9330eaaep174,
                0x1.914e03c9260eep1018,
                0x1.914e03c9260eep1019,
                0x1.91510781fb5fp-675,
                0x1.91553af7b7353p-267,
                0x1.916f4fb0582c8p386,
                0x1.916f4fb0582c8p387,
                0x1.9170c11a0994cp+432,
                0x1.9170c11a0994cp433,
                0x1.9174ec0efbabcp151,
                0x1.9177a4948362cp-964,
                0x1.917c9ec81bb3cp-738,
                0x1.91812924a25d9p599,
                0x1.91869e3843599p-81,
                0x1.918cffb2e2c6cp729,
                0x1.918cffb2e2c6cp730,
                0x1.9190e30e46c1ep-233,
                0x1.9190e30e46c1ep-234,
                0x1.9190e30e46c1ep-235,
                0x1.9199b3cedbf88p673,
                0x1.919ebcce7cca2p-888,
                0x1.919ebcce7cca2p-889,
                0x1.91a24aacd13bfp-187,
                0x1.91b5e751f5854p-274,
                0x1.91b5e751f5854p-275,
                0x1.91b5e751f5854p-276,
                0x1.91b5e751f5854p-277,
                0x1.91b9de4d5cf3p-254,
                0x1.91b9de4d5cf3p-255,
                0x1.91b9de4d5cf3p-256,
                0x1.91bc08eac9a41p-190,
                0x1.91bc08eac9a42p-190,
                0x1.91c7d54a54746p759,
                0x1.91c8cf76a7b0ep-1015,
                0x1.91cb41b4d266cp801,
                0x1.91cb504e0594ap-171,
                0x1.91cc88f3371ep-631,
                0x1.91cc88f3371ep-632,
                0x1.91d4d2872e3b6p-742,
                0x1.91d593f5b7af6p275,
                0x1.91d63f05687ccp-234,
                0x1.91dc1a1bd405p734,
                0x1.91dc8cad978f6p-654,
                0x1.91ee36063668p511,
                0x1.91ee36063668p+512,
                0x1.91f23abac7676p-91,
                0x1.91f8331aee8ebp299,
                0x1.9204996ce1c3cp-737,
                0x1.9205d2fd4796p-702,
                0x1.9205d2fd4796p-703,
                0x1.9205d2fd4796p-704,
                0x1.9205d2fd4796p-705,
                0x1.92085d052d888p-435,
                0x1.92090f9ae114cp-810,
                0x1.92135e414161cp-488,
                0x1.92135e414161cp-489,
                0x1.92135e414161cp-490,
                0x1.92135e414161cp-491,
                0x1.921f4c23e675ep810,
                0x1.921f4c23e675ep811,
                0x1.92238f798777ap-472,
                0x1.92238f798777ap-473,
                0x1.922726dbaae39p294,
                0x1.922726dbaae39p295,
                0x1.922b526a3783cp+780,
                0x1.922b526a3783cp781,
                0x1.9237be92983eap-507,
                0x1.923bd746a3528p-14,
                0x1.923bf703d0966p-504,
                0x1.923dc22deaddap-417,
                0x1.923dc22deaddap-418,
                0x1.923dc22deaddap-419,
                0x1.9247fcfd0b21cp-804,
                0x1.924bc9052ab98p-133,
                0x1.924c03c27b348p511,
                0x1.924c03c27b348p+512,
                0x1.924e9c33cd5p-1030,
                0x1.9253c7b3f6fb2p851,
                0x1.9253c7b3f6fb2p+852,
                0x1.92588c06468a7p+724,
                0x1.925a0aabcdc68p-365,
                0x1.925a0aabcdc68p-366,
                0x1.925a0aabcdc68p-367,
                0x1.925a0aabcdc68p-368,
                0x1.925a0aabcdc68p-369,
                0x1.925a0aabcdc68p-370,
                0x1.925a0aabcdc68p-371,
                0x1.925a0aabcdc68p-372,
                0x1.9266087f67467p573,
                0x1.92672b6fb61f9p-102,
                0x1.9269dc6cb8321p-957,
                0x1.926c5caa140e2p-96,
                0x1.9273c780a9882p+900,
                0x1.927d97238a3dep-337,
                0x1.928a5f50f4f72p-338,
                0x1.928a5f50f4f72p-339,
                0x1.9292615c3aa54p+780,
                0x1.9292615c3aa54p781,
                0x1.92975b84329ap-155,
                0x1.92a285f508f0ep+776,
                0x1.92a80318b3d5bp-930,
                0x1.92a93f491e0d8p1002,
                0x1.92b0e271622d2p669,
                0x1.92b0e271622d2p670,
                0x1.92b1ca5f5e525p881,
                0x1.92c286fbc2d9ap639,
                0x1.92c286fbc2d9ap+640,
                0x1.92d1266b0553ep-593,
                0x1.92dc8a374ee92p+600,
                0x1.92e321eb15122p-995,
                0x1.92ee6a9c835eap702,
                0x1.92f644745eaf2p-869,
                0x1.92f644745eaf2p-870,
                0x1.92f644745eaf2p-871,
                0x1.9300a22aa2043p983,
                0x1.93020d43062d4p-167,
                0x1.93020d43062d4p-168,
                0x1.93020d43062d4p-169,
                0x1.93020d43062d4p-170,
                0x1.93020d43062d4p-171,
                0x1.93023p-1054,
                0x1.9302e0e26447p629,
                0x1.93040de93e71cp+264,
                0x1.93040de93e71cp265,
                0x1.93040de93e71cp266,
                0x1.93040de93e71cp267,
                0x1.93040de93e71cp+268,
                0x1.9306c793217c6p-597,
                0x1.9306c793217c6p-598,
                0x1.9306c793217c6p-599,
                0x1.9308e86a2861dp-792,
                0x1.9309459f3a88ep-488,
                0x1.930e868e89591p-871,
                0x1.930e868e8959p-871,
                0x1.9310a8c89e0c2p486,
                0x1.931293b712d35p466,
                0x1.9317f0393a62p393,
                0x1.9317f0393a62p394,
                0x1.9320ba440d1fcp-609,
                0x1.932405bf6e4e4p739,
                0x1.9324fb1cc3fbep574,
                0x1.9325b3b1f3eddp-501,
                0x1.932b5763146aep646,
                0x1.932d06be21f33p945,
                0x1.93360a1a0b62dp-743,
                0x1.933c5b3cc5c2p-831,
                0x1.933ddf3042499p+868,
                0x1.93428eee94258p187,
                0x1.9344ce0afbd9ap-476,
                0x1.9346ada546bebp634,
                0x1.9351f49p-1046,
                0x1.9364d4eb90d3cp+624,
                0x1.9364d4eb90d3cp625,
                0x1.9371a2de609b8p-970,
                0x1.9379fec069826p-386,
                0x1.9379fec069826p-387,
                0x1.93845419f117dp-945,
                0x1.93845419f117dp-946,
                0x1.938889d19e978p-394,
                0x1.938889d19e978p-395,
                0x1.938889d19e978p-396,
                0x1.9388cf7f8bd2ap+956,
                0x1.93918a3331acep+284,
                0x1.939214213dce2p-464,
                0x1.939214213dce2p-465,
                0x1.93941aaa32fe4p491,
                0x1.93a6d11077292p-67,
                0x1.93a6d11077292p-68,
                0x1.93a6d11077292p-69,
                0x1.93a6d11077292p-70,
                0x1.93a6d11077292p-71,
                0x1.93a6d11077292p-72,
                0x1.93a6d11077292p-73,
                0x1.93b2e5192a1d8p+852,
                0x1.93b2e5192a1d8p853,
                0x1.93b47546ea19ep-119,
                0x1.93b47546ea19ep-120,
                0x1.93b47546ea19ep-121,
                0x1.93b47546ea19ep-122,
                0x1.93b47546ea19ep-123,
                0x1.93ba3249a8512p343,
                0x1.93ba3249a8512p+344,
                0x1.93ba3249a8512p345,
                0x1.93bb1e72a2034p-705,
                0x1.93bb1e72a2034p-706,
                0x1.93bb1e72a2034p-707,
                0x1.93cdb48b9848cp271,
                0x1.93dd452395e84p513,
                0x1.93dd452395e84p514,
                0x1.93dfef58eeb6ap507,
                0x1.93e2fbc66a793p-492,
                0x1.93e5939a08ceap99,
                0x1.93e6decb9b64cp-991,
                0x1.93e6decb9b64cp-992,
                0x1.93e9c661f6e9ep-151,
                0x1.94099785e1a4p303,
                0x1.9410893df9f04p-544,
                0x1.941251e5d9b21p207,
                0x1.941277e8812ecp-212,
                0x1.941401cca2bfep111,
                0x1.941790b25aa31p-378,
                0x1.941a445a18a5ep-759,
                0x1.9427c042b96eap94,
                0x1.9427f19fcc04p-183,
                0x1.94286a5f96f6ap-658,
                0x1.94286a5f96f6ap-659,
                0x1.94286a5f96f6ap-660,
                0x1.94286a5f96f6ap-661,
                0x1.9428b1d45a741p465,
                0x1.942f8cef37336p-545,
                0x1.9431c0753a56cp317,
                0x1.943517f173fc2p-339,
                0x1.9437e08c52bafp-27,
                0x1.9438a2e07a1bcp-79,
                0x1.9439c914ff3d6p-924,
                0x1.943f67e3510a2p-322,
                0x1.943f67e3510a2p-323,
                0x1.944055b7bb63ep-870,
                0x1.944072c923fc2p773,
                0x1.9445dfa2e4e9ep+544,
                0x1.9447aa64e57c2p335,
                0x1.94501707ab9e8p-738,
                0x1.945145230b378p583,
                0x1.945145230b378p+584,
                0x1.945c5dad9cca4p515,
                0x1.945c5dad9cca4p+516,
                0x1.945c5dad9cca4p517,
                0x1.945c5dad9cca4p518,
                0x1.945c5dad9cca4p519,
                0x1.945c5dad9cca4p+520,
                0x1.945c5dad9cca4p521,
                0x1.945ebc08e6946p131,
                0x1.9463b59b8f2bdp926,
                0x1.9463b59b8f2bdp927,
                0x1.9463b59b8f2bdp+928,
                0x1.946502ead3d3p-167,
                0x1.946631f8bfbe3p-215,
                0x1.946f5f3ef59efp202,
                0x1.948cb121d02d3p366,
                0x1.948dd06de561fp974,
                0x1.948dd06de561fp975,
                0x1.949004472028fp-663,
                0x1.94997308727adp+1008,
                0x1.949f3b7a4dbe8p+352,
                0x1.949f3b7a4dbe8p353,
                0x1.949f3b7a4dbe8p354,
                0x1.949f3b7a4dbe8p355,
                0x1.94a0fdea5aee9p-906,
                0x1.94a7edfe2abefp779,
                0x1.94a80ea17c1dp-808,
                0x1.94a94c7bdea3ep446,
                0x1.94af560f0e94p794,
                0x1.94b2bf8db071ep-601,
                0x1.94bf9bbb8d6c2p-515,
                0x1.94bf9bbb8d6c2p-516,
                0x1.94c988f217e52p-19,
                0x1.94d205cd1eba9p270,
                0x1.94d3fa74144a6p-672,
                0x1.94d521e393fd2p-221,
                0x1.94ed4ba9a7d5ap-56,
                0x1.94f0809343b34p218,
                0x1.94f11b2f8ee3ap-638,
                0x1.94f11b2f8ee3ap-639,
                0x1.94f11b2f8ee3ap-640,
                0x1.94f8979917479p-295,
                0x1.94f8979917479p-296,
                0x1.94fb3d51632d4p-351,
                0x1.94fb3d51632d4p-352,
                0x1.94fb3d51632d4p-353,
                0x1.94fb3d51632d4p-354,
                0x1.94fb3d51632d4p-355,
                0x1.94p-1067,
                0x1.94p-1068,
                0x1.9500d7197e3f2p155,
                0x1.9500d7197e3f2p+156,
                0x1.9507d80147609p847,
                0x1.950d2fe818f68p-1004,
                0x1.95106fecb6f56p219,
                0x1.95106fecb6f56p+220,
                0x1.95133c9b2c3a4p+928,
                0x1.95133c9b2c3a4p929,
                0x1.95133c9b2c3a4p930,
                0x1.95133c9b2c3a4p931,
                0x1.9515528a8b79ep-574,
                0x1.9515528a8b79ep-575,
                0x1.951a598efee66p-535,
                0x1.9539e3a40dfb8p-582,
                0x1.9545824dc6d1p-364,
                0x1.9545824dc6d1p-365,
                0x1.954b88b26687cp895,
                0x1.954e95d18e945p219,
                0x1.955c89c723db6p-82,
                0x1.9562fe1579582p-848,
                0x1.956304ff5db9ap814,
                0x1.9565223b1461cp-621,
                0x1.9565223b1461cp-622,
                0x1.9569bdbdb11b5p-69,
                0x1.956bfcd6e963bp494,
                0x1.95845196da652p527,
                0x1.95845196da652p+528,
                0x1.95845196da652p529,
                0x1.95845196da652p530,
                0x1.95845196da652p531,
                0x1.958546f63372cp-264,
                0x1.95896c18c84ccp723,
                0x1.95a5efea6b347p-97,
                0x1.95a5efea6b348p-96,
                0x1.95a5efea6b348p-97,
                0x1.95ad1185480bep-54,
                0x1.95ad1185480bep-55,
                0x1.95ad1185480bep-56,
                0x1.95ad1185480bep-57,
                0x1.95ad1185480bep-58,
                0x1.95ad1185480bep-59,
                0x1.95bad04670f52p+444,
                0x1.95c39c48cc644p+472,
                0x1.95c39c48cc644p473,
                0x1.95c39c48cc644p474,
                0x1.95c39c48cc644p475,
                0x1.95c5330c17416p-823,
                0x1.95c5330c17416p-824,
                0x1.95d01bd723355p-381,
                0x1.95d4100811d0cp-57,
                0x1.95d61cadf266cp-403,
                0x1.95d668d038745p-312,
                0x1.95d7d6dcce8p-1033,
                0x1.95dd37863fd5cp599,
                0x1.95dd37863fd5cp+600,
                0x1.95dd37863fd5cp601,
                0x1.95dd37863fd5cp602,
                0x1.95dd37863fd5cp603,
                0x1.95e328c37238p-227,
                0x1.95e910e26bb2p-856,
                0x1.95f9282e3c5f6p690,
                0x1.95f9282e3c5f6p691,
                0x1.95fd4c88d4b1cp-547,
                0x1.95fd4c88d4b1cp-548,
                0x1.95fd4c88d4b1cp-549,
                0x1.95fd4c88d4b1cp-550,
                0x1.95fd4c88d4b1cp-551,
                0x1.95fd4c88d4b1cp-552,
                0x1.95fd4c88d4b1cp-553,
                0x1.95fd4c88d4b1cp-554,
                0x1.95fd4c88d4b1cp-555,
                0x1.95feaf0eb3984p-345,
                0x1.95feaf0eb3984p-346,
                0x1.95feb79a5663cp827,
                0x1.95feb79a5663cp+828,
                0x1.95feb79a5663cp829,
                0x1.95feb79a5663cp830,
                0x1.95p-1066,
                0x1.960b7318p-1045,
                0x1.960f49108cefap671,
                0x1.960fe08d5847ep-124,
                0x1.960fe08d5847ep-125,
                0x1.960fe08d5847ep-126,
                0x1.960fe08d5847ep-127,
                0x1.960fe08d5847ep-128,
                0x1.961219000356ap+388,
                0x1.961219000356ap389,
                0x1.961a595d93523p-323,
                0x1.96257d7cd0434p297,
                0x1.96257d7cd0434p298,
                0x1.96271d104574ep-411,
                0x1.9628e26b610f7p775,
                0x1.9634ca8c9904ap853,
                0x1.96390ed1a12b8p443,
                0x1.964f900699e61p+708,
                0x1.964f900699e61p709,
                0x1.96572b6108db2p-237,
                0x1.966065e2c4eecp157,
                0x1.966b154f094fap-605,
                0x1.966b154f094fap-606,
                0x1.966b154f094fap-607,
                0x1.966b154f094fap-608,
                0x1.96740b30b31dep386,
                0x1.96740b30b31dep387,
                0x1.96740b30b31dep+388,
                0x1.96771f5143a14p+956,
                0x1.96771f5143a14p957,
                0x1.96771f5143a14p958,
                0x1.96771f5143a14p959,
                0x1.96771f5143a14p+960,
                0x1.96771f5143a14p961,
                0x1.96771f5143a14p962,
                0x1.9677825a0c0eap+148,
                0x1.9677825a0c0eap149,
                0x1.96779d8643578p+400,
                0x1.967e5eec84e2ep873,
                0x1.967e5eec84e2fp873,
                0x1.9698b8fc23e03p231,
                0x1.96a7b88961ae4p-845,
                0x1.96a7b88961ae4p-846,
                0x1.96a7b88961ae4p-847,
                0x1.96c1adbe0c72ep-493,
                0x1.96c1adbe0c72ep-494,
                0x1.96d60f4c9b90bp+908,
                0x1.96dad5a1bd27bp953,
                0x1.96e4421b4001cp-522,
                0x1.96e4a899f5ed4p-593,
                0x1.96f33e462566p778,
                0x1.96f4cb023c99ep833,
                0x1.96f53ae823102p-600,
                0x1.96fbb9bb44db4p-778,
                0x1.96fce5de7929p563,
                0x1.96fce5de7929p+564,
                0x1.96fce5de7929p565,
                0x1.96fce5de7929p566,
                0x1.96fce5de7929p567,
                0x1.96ffd0fe6b0c6p559,
                0x1.96ffd0fe6b0c6p+560,
                0x1.96ffd0fe6b0c6p561,
                0x1.96fff21c6e954p-293,
                0x1.970804fe5182p865,
                0x1.970804fe5182p866,
                0x1.9709527e8e482p415,
                0x1.9710a1f84ff82p+632,
                0x1.9712768881374p463,
                0x1.971431ca4p-1038,
                0x1.971431ca4p-1039,
                0x1.971431ca4p-1040,
                0x1.9719958a1c846p-931,
                0x1.9719958a1c846p-932,
                0x1.971e3de09f648p+580,
                0x1.97294ec1bc228p-774,
                0x1.972c5b787a0fap-992,
                0x1.972c5b787a0fap-993,
                0x1.972c9d3c88784p-61,
                0x1.9738cdc8cc5c4p-112,
                0x1.973945a9fb517p-830,
                0x1.9749e464815cdp-64,
                0x1.974b2fdc4b517p-596,
                0x1.97683df2f268dp-293,
                0x1.97683df2f268ep-289,
                0x1.97683df2f268ep-290,
                0x1.97683df2f268ep-291,
                0x1.97683df2f268ep-292,
                0x1.97683df2f268ep-293,
                0x1.97683df2f268ep-294,
                0x1.977b65c576e1fp-820,
                0x1.977c5eb212aa7p639,
                0x1.9780ccf3560a9p929,
                0x1.978e01583788p-1029,
                0x1.97964ba0034c8p-667,
                0x1.97964ba0034c8p-668,
                0x1.97969144a272fp-253,
                0x1.9799ae0ec67cap+496,
                0x1.9799ae0ec67cap497,
                0x1.9799ae0ec67cap498,
                0x1.9799ae0ec67cap499,
                0x1.9799ae0ec67cap+500,
                0x1.9799ae0ec67cap501,
                0x1.9799ae0ec67cap502,
                0x1.979bb2c62786p+308,
                0x1.97a2a205f591ep-291,
                0x1.97a2a205f591ep-292,
                0x1.97a2a205f591ep-293,
                0x1.97a2a205f591ep-294,
                0x1.97a8p-1060,
                0x1.97a918a63f184p-688,
                0x1.97a918a63f184p-689,
                0x1.97a918a63f184p-690,
                0x1.97bb77bf3838fp-550,
                0x1.97bb77bf3838fp-551,
                0x1.97bf5559b31fep+124,
                0x1.97bf5559b31fep125,
                0x1.97c35707b6cfap707,
                0x1.97c35707b6cfap+708,
                0x1.97cb4cee928a6p645,
                0x1.97d4df19d6057p+192,
                0x1.97d4df19d6058p+192,
                0x1.97d4df19d6058p193,
                0x1.97daf0d95f858p127,
                0x1.97ed81baf097p+552,
                0x1.97ed81baf097p553,
                0x1.97ed81baf097p554,
                0x1.97ed81baf097p555,
                0x1.97fb5f280da74p-577,
                0x1.97fb5f280da74p-578,
                0x1.97fde023a0c76p-633,
                0x1.97fde023a0c76p-634,
                0x1.98033ec5a5b29p+232,
                0x1.9813653f62db8p513,
                0x1.9815617888f88p733,
                0x1.9815617888f88p734,
                0x1.9815617888f88p735,
                0x1.9815617888f88p+736,
                0x1.9815a946180c7p553,
                0x1.982afddf17a9p-677,
                0x1.982cb1be59f0cp965,
                0x1.98419d37a6b8fp677,
                0x1.984d8972bff8fp-987,
                0x1.985010a0f46fap961,
                0x1.985010a0f46fap962,
                0x1.985010a0f46fap963,
                0x1.9853c0e57f046p151,
                0x1.9856be3cfd156p-122,
                0x1.9868f6d84f78cp+1000,
                0x1.9868f6d84f78cp1001,
                0x1.9868f6d84f78cp1002,
                0x1.9868f6d84f78cp1003,
                0x1.9868f6d84f78cp+1004,
                0x1.9868f6d84f78cp999,
                0x1.986964ecfc14ep-668,
                0x1.986a0ecc52359p833,
                0x1.987152p-1048,
                0x1.987152p-1049,
                0x1.987265e5e34a3p462,
                0x1.9875f13c4778cp735,
                0x1.987acd0367e6fp555,
                0x1.987acd0367e6fp+556,
                0x1.987fb27132404p-902,
                0x1.9884c0a2ac45ap-490,
                0x1.989f8742204bap-379,
                0x1.98a2c810f3f07p-651,
                0x1.98a576a3f532p805,
                0x1.98a7fa6f52d94p-577,
                0x1.98a8848429302p-242,
                0x1.98aaecfaf65p-1028,
                0x1.98b8efcddef4cp603,
                0x1.98b8f10cee22fp-34,
                0x1.98be3bf6a3c1p+588,
                0x1.98bf832dfdfbp-973,
                0x1.98bf832dfdfbp-974,
                0x1.98c44192ebbb2p+680,
                0x1.98c78e2214ff9p815,
                0x1.98ce80d792f5cp-913,
                0x1.98ce80d792f5cp-914,
                0x1.98d26b4d788b8p+760,
                0x1.98d3aaf56e9edp-448,
                0x1.98d98deeabe95p+908,
                0x1.98dd89db75cd5p-687,
                0x1.98de7da400ac6p-1010,
                0x1.98e3445512a6ep-932,
                0x1.98e3445512a6ep-933,
                0x1.98e3445512a6ep-934,
                0x1.98e3445512a6ep-935,
                0x1.98e3445512a6ep-936,
                0x1.98e3445512a6ep-937,
                0x1.98e3445512a6ep-938,
                0x1.98e6e3133adcep-109,
                0x1.98e6e3133adcep-110,
                0x1.98e6e3133adcep-111,
                0x1.98e6e3133adcep-112,
                0x1.98e6e3133adcep-113,
                0x1.98e6e3133adcep-114,
                0x1.98fa32a46ebdep-319,
                0x1.990021a012ceep+440,
                0x1.9901f196d8b03p+948,
                0x1.99090446a4d98p-350,
                0x1.990ecbb9a97bap-675,
                0x1.991d00e76af2p411,
                0x1.991d00e76af2p+412,
                0x1.991d00e76af2p413,
                0x1.991d00e76af2p414,
                0x1.9929ae2fce2c1p-861,
                0x1.9929e60504af4p-88,
                0x1.9929e60504af4p-89,
                0x1.9929e60504af4p-90,
                0x1.9929e60504af4p-91,
                0x1.9929e60504af4p-92,
                0x1.9929e60504af4p-93,
                0x1.9929e60504af4p-94,
                0x1.9929e60504af4p-95,
                0x1.992b0f69cf51ep167,
                0x1.992c7fdc216fap-489,
                0x1.9931213e39006p-141,
                0x1.9931213e39006p-142,
                0x1.99418eab49a8bp-338,
                0x1.9948d2b1b9df1p-735,
                0x1.9949819f693d7p-51,
                0x1.9949819f693d7p-52,
                0x1.994efd792fffap+892,
                0x1.994efd792fffap893,
                0x1.99669a4c67e9bp251,
                0x1.997388018306cp-564,
                0x1.99745a8d99d4p158,
                0x1.9983b6254741bp-970,
                0x1.999376f8cfc78p+744,
                0x1.999376f8cfc78p745,
                0x1.999376f8cfc78p746,
                0x1.999999999999ap-3,
                0x1.999999999999ap-4,
                0x1.999a1bb37c8e7p-914,
                0x1.99a41718f58a8p-563,
                0x1.99a58ce36ab48p331,
                0x1.99a58ce36ab48p+332,
                0x1.99a83532d8251p774,
                0x1.99ade2b55332p-733,
                0x1.99aecc0396c2ap825,
                0x1.99aecc0396c2ap826,
                0x1.99b858e76d78ap-446,
                0x1.99b8a9e2632b8p-840,
                0x1.99ba446c5af53p890,
                0x1.99ba6e3ca3cp-838,
                0x1.99bdbbe32dd24p291,
                0x1.99beec936a12ep-685,
                0x1.99beec936a12ep-686,
                0x1.99c7d89db518p-1029,
                0x1.99ca5a719767ep467,
                0x1.99ca5a719767ep+468,
                0x1.99e0a642f9da1p-458,
                0x1.99e46124954a4p-243,
                0x1.99e6a0e1c81cep+692,
                0x1.99e6ce1831fd6p+776,
                0x1.99ec513f2c503p-360,
                0x1.99ec513f2c503p-361,
                0x1.99ee5d4b7fdaep927,
                0x1.99ef223faa1p379,
                0x1.99f5278f401bep+416,
                0x1.9a04c3f8451a1p222,
                0x1.9a04c3f8451a1p223,
                0x1.9a05efa037b1ep314,
                0x1.9a06d06e26112p481,
                0x1.9a09187f69d8cp574,
                0x1.9a0db3bdfa672p614,
                0x1.9a0db3bdfa672p615,
                0x1.9a0db3bdfa672p+616,
                0x1.9a0db3bdfa672p617,
                0x1.9a2028368022ep1017,
                0x1.9a2028368022ep1018,
                0x1.9a2028368022ep1019,
                0x1.9a2c2a34ac2fap234,
                0x1.9a2c2a34ac2fap235,
                0x1.9a2c2a34ac2fap+236,
                0x1.9a2c2a34ac2fap237,
                0x1.9a2c2a34ac2fap238,
                0x1.9a2c2a34ac2fap239,
                0x1.9a3415eb0478ep865,
                0x1.9a3529e870f1ep106,
                0x1.9a3529e870f1ep107,
                0x1.9a36417cfca31p915,
                0x1.9a364d0d139d9p950,
                0x1.9a455d3482d3cp-588,
                0x1.9a4815e0ad0c6p+84,
                0x1.9a4815e0ad0c6p85,
                0x1.9a493c90e67c6p702,
                0x1.9a493c90e67c6p703,
                0x1.9a4971d609c34p-347,
                0x1.9a52c7deb1f7cp298,
                0x1.9a547f75bd0aap213,
                0x1.9a5b527b3ddacp-903,
                0x1.9a5ed85a8a99p-784,
                0x1.9a5ed85a8a99p-785,
                0x1.9a6340b2a4221p-118,
                0x1.9a66cf588eaa4p331,
                0x1.9a686ffeb807bp-579,
                0x1.9a686ffeb807bp-580,
                0x1.9a70b55657d18p-246,
                0x1.9a7149871620fp-398,
                0x1.9a742461887f6p966,
                0x1.9a7c5bdd34166p347,
                0x1.9a7c5bdd34166p+348,
                0x1.9a7c5bdd34166p349,
                0x1.9a807a2dd1e12p549,
                0x1.9a807a2dd1e12p550,
                0x1.9a81d2797a4fcp489,
                0x1.9a84bd2b2278fp-444,
                0x1.9a896283d96e6p167,
                0x1.9a8a22d8ef554p163,
                0x1.9a8c60b32b5dfp-629,
                0x1.9a8c60b32b5dfp-630,
                0x1.9a8f5722007e6p878,
                0x1.9a967efe8da38p+936,
                0x1.9a9d36d3e4eeep-500,
                0x1.9ab034ab5f70cp-756,
                0x1.9ab61f0b3f804p-699,
                0x1.9ab745a97b0fap+940,
                0x1.9ab8261990292p759,
                0x1.9ab8261990292p+760,
                0x1.9ab8261990292p761,
                0x1.9ab8261990292p762,
                0x1.9ab8261990292p763,
                0x1.9abb5d1218287p-47,
                0x1.9ac0db8a5c3d2p433,
                0x1.9ac0db8a5c3d2p434,
                0x1.9ac0db8a5c3d2p435,
                0x1.9ac0db8a5c3d2p+436,
                0x1.9ad37049fbbc2p-462,
                0x1.9ad37049fbbc2p-463,
                0x1.9ad5a5685899ep214,
                0x1.9ad6b62ep-1043,
                0x1.9ae0cea92a56p-608,
                0x1.9ae57da459546p-327,
                0x1.9aeb2b2f0c6e4p-430,
                0x1.9af2b7d0e0a2cp-685,
                0x1.9af2b7d0e0a2dp-685,
                0x1.9af4ad6c281fcp-682,
                0x1.9af9ac7810a2fp-873,
                0x1.9afeb52c5e79p-350,
                0x1.9aff88d82ff74p-517,
                0x1.9b0151fb9413ep-64,
                0x1.9b0fe7c145afap-346,
                0x1.9b0fe7c145afap-347,
                0x1.9b0fe7c145afap-348,
                0x1.9b0fe7c145afap-349,
                0x1.9b1464309bb4bp-513,
                0x1.9b1de6df7ec51p-406,
                0x1.9b223f084375p-212,
                0x1.9b27552fe4eaap-428,
                0x1.9b2a5c4041e4cp-173,
                0x1.9b2a5c4041e4cp-174,
                0x1.9b2a5c4041e4cp-175,
                0x1.9b2a5c4041e4cp-176,
                0x1.9b2a5c4041e4cp-177,
                0x1.9b2a5c4041e4cp-178,
                0x1.9b2a5c4041e4cp-179,
                0x1.9b2a5c4041e4cp-180,
                0x1.9b2e875589f38p+588,
                0x1.9b3a9ca34532bp-786,
                0x1.9b3dd3316e694p462,
                0x1.9b3e026893253p+180,
                0x1.9b445b63da1fap-963,
                0x1.9b4d2092ffb92p786,
                0x1.9b4d2092ffb92p787,
                0x1.9b546fe3a43c6p-276,
                0x1.9b546fe3a43c6p-277,
                0x1.9b604aaaca626p-200,
                0x1.9b651584e8b2p535,
                0x1.9b68p-1059,
                0x1.9b759505df0d1p-999,
                0x1.9b8095f646834p-628,
                0x1.9b85858f2d91ep553,
                0x1.9b8fba12c969dp-660,
                0x1.9b9353902b16ap-334,
                0x1.9b9353902b16ap-335,
                0x1.9ba7c0bdc4769p-1017,
                0x1.9ba7c0bdc4769p-1018,
                0x1.9bac5eb6a855ep506,
                0x1.9bafc6df6336ep467,
                0x1.9bb001eecc20ap-968,
                0x1.9bcb0a8dcf952p-397,
                0x1.9bcdfabc1357ap285,
                0x1.9bceff7ac027ap538,
                0x1.9bceff7ac027ap539,
                0x1.9bceff7ac027ap+540,
                0x1.9bceff7ac027ap541,
                0x1.9bcf774ac3e31p-296,
                0x1.9bd7fe213576p-121,
                0x1.9bd7fe213576p-122,
                0x1.9bd7fe213576p-123,
                0x1.9bd886afba00fp-274,
                0x1.9bd886afba00fp-275,
                0x1.9bda1b0973346p-572,
                0x1.9bda1b0973346p-573,
                0x1.9beb2c3dcbd18p723,
                0x1.9beb2c3dcbd18p+724,
                0x1.9bf56d038acdap-841,
                0x1.9bf56d038acdap-842,
                0x1.9bf56d038acdap-843,
                0x1.9bf5df60e226fp257,
                0x1.9bfc87aa14e14p-90,
                0x1.9bfd8b3edfafap+984,
                0x1.9bfd8b3edfafap985,
                0x1.9bfd8b3edfafap986,
                0x1.9bfd8b3edfafap987,
                0x1.9bfd8b3edfafap+988,
                0x1.9bfd8b3edfafap989,
                0x1.9bfd8b3edfafap990,
                0x1.9c0536032011p681,
                0x1.9c05b894b1066p-550,
                0x1.9c0bbbd1e0fbcp-471,
                0x1.9c0bbbd1e0fbcp-472,
                0x1.9c0bbbd1e0fbcp-473,
                0x1.9c0bbbd1e0fbcp-474,
                0x1.9c0bbbd1e0fbcp-475,
                0x1.9c0bbbd1e0fbcp-476,
                0x1.9c0bbbd1e0fbcp-477,
                0x1.9c0f4bb204095p-273,
                0x1.9c13597e961aap-347,
                0x1.9c13597e961aap-348,
                0x1.9c1684357da01p-511,
                0x1.9c265d1b2d188p-864,
                0x1.9c2ca63b13fdp-277,
                0x1.9c2ca63b13fdp-278,
                0x1.9c322c57ed8f4p+444,
                0x1.9c322c57ed8f4p445,
                0x1.9c3bc80c85c7ep770,
                0x1.9c3bc80c85c7ep771,
                0x1.9c41590c40304p682,
                0x1.9c41590c40304p683,
                0x1.9c4455516e14ep326,
                0x1.9c4p-1062,
                0x1.9c4p-1063,
                0x1.9c511dc3a41dfp-29,
                0x1.9c60bf04344p-1032,
                0x1.9c611cb3edfd1p-222,
                0x1.9c62e852569bp-729,
                0x1.9c62e852569bp-730,
                0x1.9c62e852569bp-731,
                0x1.9c62e852569bp-732,
                0x1.9c8099f7c8c24p-411,
                0x1.9c84e878c8f26p-380,
                0x1.9c861553ad12p-1027,
                0x1.9c926adabee06p554,
                0x1.9c95125d3e503p-347,
                0x1.9c9c8c4fcad2dp-154,
                0x1.9ca210d9777cp-881,
                0x1.9ca96e042aa96p658,
                0x1.9cb2d857f862bp-865,
                0x1.9cbae7fe805b3p-881,
                0x1.9cbd06456ee4ep-310,
                0x1.9cbd06456ee4ep-311,
                0x1.9cc69053019b8p-254,
                0x1.9ccd15f6cd355p-59,
                0x1.9cd4bbe6cf769p-953,
                0x1.9cd7b86981bdfp855,
                0x1.9cd9448a6c5fcp250,
                0x1.9cde9e67df4p+808,
                0x1.9cde9e67df4p809,
                0x1.9cded3f290ceep+184,
                0x1.9cef8f6c711f9p114,
                0x1.9cfe557d4a658p-83,
                0x1.9d16cb3be685cp-349,
                0x1.9d16cb3be685cp-350,
                0x1.9d16cb3be685cp-351,
                0x1.9d16cb3be685cp-352,
                0x1.9d1b8e299d156p-141,
                0x1.9d2692b79cd78p-981,
                0x1.9d27f3b2b775ap649,
                0x1.9d28f47b4d524p-396,
                0x1.9d28f47b4d525p-396,
                0x1.9d2d8366baffap-784,
                0x1.9d312bed124dp519,
                0x1.9d314a074c176p-1008,
                0x1.9d338951dc416p-955,
                0x1.9d4abc26c2cccp-392,
                0x1.9d557148fc913p511,
                0x1.9d557148fc913p+512,
                0x1.9d5650a37e6dap-938,
                0x1.9d5d5041feed6p601,
                0x1.9d5d5041feed6p602,
                0x1.9d5d5041feed6p603,
                0x1.9d5d5041feed6p+604,
                0x1.9d5d5041feed6p605,
                0x1.9d5d5041feed6p606,
                0x1.9d5d5041feed6p607,
                0x1.9d5d5041feed6p+608,
                0x1.9d5d5041feed6p609,
                0x1.9d5d5041feed6p610,
                0x1.9d5d5041feed6p611,
                0x1.9d5d5041feed6p+612,
                0x1.9d5df640a33cap+852,
                0x1.9d61723a45622p387,
                0x1.9d61723a45622p+388,
                0x1.9d63b7dfb8388p-436,
                0x1.9d69455a53bd9p-137,
                0x1.9d69455a53bd9p-138,
                0x1.9d69455a53bd9p-139,
                0x1.9d6e7911a528fp-963,
                0x1.9d79022ea821ep-799,
                0x1.9d7d58f656746p+980,
                0x1.9d7e57598b149p-710,
                0x1.9d814b0cd823ap-829,
                0x1.9d8f9fc2808d2p321,
                0x1.9d8f9fc2808d2p322,
                0x1.9d971e4fe8402p89,
                0x1.9d97553a730c2p-231,
                0x1.9d97553a730c2p-232,
                0x1.9d97553a730c2p-233,
                0x1.9d97553a730c2p-234,
                0x1.9d97553a730c2p-235,
                0x1.9d98841a8ebb4p-346,
                0x1.9d9bb2c3d9a6bp-922,
                0x1.9d9d9b538c19cp277,
                0x1.9d9fb797aeb4ep163,
                0x1.9da366bf42eabp-985,
                0x1.9da8eed9e9122p937,
                0x1.9da8eed9e9122p938,
                0x1.9da8eed9e9122p939,
                0x1.9da8eed9e9122p+940,
                0x1.9dad15f8d4356p450,
                0x1.9dad15f8d4356p451,
                0x1.9dad15f8d4356p+452,
                0x1.9db6df6c3d2fp-658,
                0x1.9db6df6c3d2fp-659,
                0x1.9db7da0d0c9f2p499,
                0x1.9dc9f20a40a25p+328,
                0x1.9dd10ad90ef42p-975,
                0x1.9dd10ad90ef42p-976,
                0x1.9dd78494a0ed4p-510,
                0x1.9dd8f4fcc04e3p-308,
                0x1.9ddc3d1071a9ep915,
                0x1.9ddc3d1071a9ep+916,
                0x1.9ddc3d1071a9ep917,
                0x1.9ddc3d1071a9ep918,
                0x1.9ddc3d1071a9ep919,
                0x1.9ddc3d1071a9ep+920,
                0x1.9ddc3d1071a9ep921,
                0x1.9ddc3d1071a9ep922,
                0x1.9ddc3d1071a9ep923,
                0x1.9ddf8fecd44d4p694,
                0x1.9de66aedabf06p-531,
                0x1.9de744f2bec07p245,
                0x1.9df7bd2d2883ap186,
                0x1.9dfc11ccc3354p-537,
                0x1.9dfe7d9bd4048p-930,
                0x1.9e030341e561ap954,
                0x1.9e056584240fep574,
                0x1.9e09f0336ae5p-342,
                0x1.9e09p-1056,
                0x1.9e09p-1057,
                0x1.9e09p-1058,
                0x1.9e117541d04e6p-741,
                0x1.9e117541d04e6p-742,
                0x1.9e117541d04e6p-743,
                0x1.9e117541d04e6p-744,
                0x1.9e117541d04e6p-745,
                0x1.9e117541d04e6p-746,
                0x1.9e1ac43f4176dp-522,
                0x1.9e1ad2ea3d2d4p-225,
                0x1.9e1ad2ea3d2d4p-226,
                0x1.9e1f43d9e23e8p+456,
                0x1.9e2e4aa37f85ap870,
                0x1.9e2eddap-1045,
                0x1.9e30b20f59e62p149,
                0x1.9e4711ca60dc6p+880,
                0x1.9e4d03130410cp247,
                0x1.9e50e313bbe82p+204,
                0x1.9e50e313bbe82p205,
                0x1.9e50e313bbe82p206,
                0x1.9e50e313bbe82p207,
                0x1.9e50e313bbe82p+208,
                0x1.9e5110745c822p+144,
                0x1.9e5430aaf5643p-818,
                0x1.9e5ad7ea9d9bep-433,
                0x1.9e5b5e6c4741cp-282,
                0x1.9e5b5e6c4741cp-283,
                0x1.9e5b5e6c4741cp-284,
                0x1.9e5f5a23f9b16p-983,
                0x1.9e5f5a23f9b16p-984,
                0x1.9e6304f2ae9ddp-176,
                0x1.9e637678f390ep+752,
                0x1.9e680adaf1de8p349,
                0x1.9e6c44532c272p-919,
                0x1.9e6c44532c272p-920,
                0x1.9e70e1e09c632p841,
                0x1.9e740c7157ee6p-330,
                0x1.9e740c7157ee6p-331,
                0x1.9e76c3a3a82f3p-196,
                0x1.9e76c3a3a82f3p-197,
                0x1.9e80124093ac4p295,
                0x1.9e80124093ac4p+296,
                0x1.9e8164b838325p311,
                0x1.9e8fd850a7c26p+948,
                0x1.9e8fd850a7c26p949,
                0x1.9e958e6a526d8p787,
                0x1.9ea6424991d8p706,
                0x1.9ea6424991d8p707,
                0x1.9eab1a859a02bp-789,
                0x1.9eab1a859a02bp-790,
                0x1.9ead332ad6438p-124,
                0x1.9eb434f17p-1037,
                0x1.9eb6dadeaadfep-263,
                0x1.9eb6e350f1bc9p-452,
                0x1.9ec6ecc2cd154p-716,
                0x1.9ecde900b69bep397,
                0x1.9ecde900b69bep398,
                0x1.9ecde900b69bep399,
                0x1.9ecde900b69bep+400,
                0x1.9ecde900b69bep401,
                0x1.9ed2f8bb00613p1002,
                0x1.9ed2f8bb00613p1003,
                0x1.9ed34d5f3a44p-545,
                0x1.9ef3993b72ab8p-591,
                0x1.9ef3993b72ab8p-592,
                0x1.9f004f9fea2f8p-21,
                0x1.9f05c9bf29db6p398,
                0x1.9f1cd5cade728p-123,
                0x1.9f1d89636a8ccp551,
                0x1.9f1d89636a8ccp+552,
                0x1.9f3b83fcf11eap321,
                0x1.9f3b83fcf11eap322,
                0x1.9f434812cdf42p369,
                0x1.9f4373e14ffcep-997,
                0x1.9f4373e14ffcep-998,
                0x1.9f4373e14ffcep-999,
                0x1.9f450a56d134p713,
                0x1.9f4d5a4680acfp483,
                0x1.9f4d5a4680acfp+484,
                0x1.9f4d5e35b434cp281,
                0x1.9f518338b53eep+312,
                0x1.9f5ed7e04a904p-1014,
                0x1.9f623d5a8a732p-106,
                0x1.9f623d5a8a732p-107,
                0x1.9f6abf7a52299p862,
                0x1.9f6c298548ad2p-783,
                0x1.9f6c298548ad2p-784,
                0x1.9f6e5bd73458cp+228,
                0x1.9f741e709ddd6p-615,
                0x1.9f74b92470e76p+356,
                0x1.9f74b92470e76p357,
                0x1.9f890622ab7acp906,
                0x1.9f890622ab7acp907,
                0x1.9f96b6dddb375p-988,
                0x1.9fa150fe9fbd8p-625,
                0x1.9fa43aab4ad7ap254,
                0x1.9fa974b2efa6fp+672,
                0x1.9fbe1a4661228p943,
                0x1.9fbe1a4661228p+944,
                0x1.9fbf3c19b9a7ap-190,
                0x1.9fbf3c19b9a7ap-191,
                0x1.9fbf3c19b9a7ap-192,
                0x1.9fbf3c19b9a7ap-193,
                0x1.9fbf3c19b9a7ap-194,
                0x1.9fbf3c19b9a7ap-195,
                0x1.9fbf3c19b9a7ap-196,
                0x1.9fc09d5ece99fp-629,
                0x1.9fc2e5a1cc61ap+844,
                0x1.9fc70edfce001p-353,
                0x1.9fcd4a38e197fp-580,
                0x1.9fd0fef9de8ep378,
                0x1.9fd20b53b9e3cp-389,
                0x1.9fd4c4b68c5eep137,
                0x1.9fd4c4b68c5eep138,
                0x1.9fd4c4b68c5eep139,
                0x1.9fe771b2350dap989,
                0x1.9fe771b2350dap990,
                0x1.9fe771b2350dap991,
                0x1.9fe771b2350dap+992,
                0x1.9fe9542235b24p958,
                0x1.9ffa539cd68aap770,
                0x1.9p6,
                0x1.a00782911e9ap-1027,
                0x1.a0088a42ba0a1p545,
                0x1.a0221bce9b6bep402,
                0x1.a025f4cf00d36p721,
                0x1.a0288b6fedd06p563,
                0x1.a02c899718681p410,
                0x1.a02c899718681p411,
                0x1.a02c899718681p+412,
                0x1.a032dab8e04c9p-203,
                0x1.a03be2c2b89ecp-384,
                0x1.a03be2c2b89ecp-385,
                0x1.a03be2c2b89ecp-386,
                0x1.a03be2c2b89ecp-387,
                0x1.a03be2c2b89ecp-388,
                0x1.a03fde214cafp862,
                0x1.a03fde214cafp863,
                0x1.a03fde214cafp+864,
                0x1.a03fde214cafp865,
                0x1.a03fde214cafp866,
                0x1.a04f802bebb6dp157,
                0x1.a04f802bebb6dp158,
                0x1.a04f802bebb6dp159,
                0x1.a055dd68f3e3cp105,
                0x1.a057f189eaf6p-229,
                0x1.a05d2132ec87ep-287,
                0x1.a05d541553fap-836,
                0x1.a06a1024b95e1p-476,
                0x1.a06a1024b95e1p-477,
                0x1.a075376b51f68p923,
                0x1.a075376b51f68p+924,
                0x1.a075376b51f68p925,
                0x1.a0785abdc615ap-419,
                0x1.a0785abdc615ap-420,
                0x1.a0785abdc615ap-421,
                0x1.a0785abdc615ap-422,
                0x1.a0785abdc615ap-423,
                0x1.a07f799a10a64p195,
                0x1.a0808p-1057,
                0x1.a08528b253eep-1027,
                0x1.a0857a01bbb3p114,
                0x1.a087dc679df7ap426,
                0x1.a087dc679df7ap427,
                0x1.a096ac0f80a3p582,
                0x1.a0acecc60373cp613,
                0x1.a0b7fe70072cep-892,
                0x1.a0bd64e56c606p-793,
                0x1.a0bd64e56c606p-794,
                0x1.a0c03b1df8af6p-788,
                0x1.a0c53e485e236p-457,
                0x1.a0c53e485e236p-458,
                0x1.a0c981d76f614p795,
                0x1.a0cb3056a9f24p-721,
                0x1.a0d6bac09d78ep-176,
                0x1.a0db19eb3735ep638,
                0x1.a0e120a793674p-700,
                0x1.a0e120a793674p-701,
                0x1.a0e120a793674p-702,
                0x1.a0e2f71512b08p+580,
                0x1.a0e5c0547b34ep-759,
                0x1.a0e5c0547b34ep-760,
                0x1.a0e5c0547b34ep-761,
                0x1.a0ecbba52f5c5p+696,
                0x1.a0eeaaa72eec2p-87,
                0x1.a0eeaaa72eec2p-88,
                0x1.a0f28c1aa858bp+852,
                0x1.a0feb7830bfcep305,
                0x1.a109bcd3e33ecp-377,
                0x1.a109bcd3e33ecp-378,
                0x1.a10e85a05974dp-201,
                0x1.a11b9cdb7d4a1p-166,
                0x1.a12666477886dp-188,
                0x1.a126da260c9d7p-74,
                0x1.a126da260c9d7p-75,
                0x1.a12c02028eef5p793,
                0x1.a12f5a0f4e3e4p-303,
                0x1.a12f5a0f4e3e5p-303,
                0x1.a13190c78e046p-725,
                0x1.a1370e53fd4bbp931,
                0x1.a1376a272824ap-230,
                0x1.a1376a272824ap-231,
                0x1.a13a09498c2d3p-862,
                0x1.a13a9e89fdca9p-1014,
                0x1.a13a9e89fdca9p-1015,
                0x1.a13f935c0547fp951,
                0x1.a149ee19691c4p-143,
                0x1.a149ee19691c4p-144,
                0x1.a149ee19691c4p-145,
                0x1.a149ee19691c4p-146,
                0x1.a149ee19691c4p-147,
                0x1.a14c9398e1e5cp-259,
                0x1.a14e874520cbp+568,
                0x1.a14e874520cbp569,
                0x1.a14e874520cbp570,
                0x1.a1589a54bf75ap+992,
                0x1.a1589a54bf75ap993,
                0x1.a1589a54bf75ap994,
                0x1.a15a2c965edep-1027,
                0x1.a15b782d554ep-490,
                0x1.a16e055c3f006p-874,
                0x1.a16e055c3f006p-875,
                0x1.a171f3728fda5p-563,
                0x1.a17896938fe16p-796,
                0x1.a19e96a19fc41p182,
                0x1.a19e96a19fc4p182,
                0x1.a19f2c0f17efbp741,
                0x1.a1a006e134287p-666,
                0x1.a1aaf66e6c578p+924,
                0x1.a1b522455259bp794,
                0x1.a1b84f64c4p-1036,
                0x1.a1c0ccbf2ac0ep+176,
                0x1.a1c37b3ee2829p-482,
                0x1.a1c37b3ee2829p-483,
                0x1.a1c37b3ee2829p-484,
                0x1.a1ca85e218cdcp-457,
                0x1.a1ca85e218cdcp-458,
                0x1.a1ca85e218cdcp-459,
                0x1.a1ca85e218cdcp-460,
                0x1.a1ca85e218cdcp-461,
                0x1.a1ca85e218cdcp-462,
                0x1.a1ce1338fb982p222,
                0x1.a1cee1fd536b4p626,
                0x1.a1cee1fd536b4p627,
                0x1.a1cfa5028602fp958,
                0x1.a1d0994a5d05ep-289,
                0x1.a1d0994a5d05ep-290,
                0x1.a1d5f36598b9cp450,
                0x1.a1d6e682dfad4p+828,
                0x1.a1d6e682dfad4p829,
                0x1.a1e1d20cff9bbp-836,
                0x1.a1eccf4e0992ep-45,
                0x1.a1f8342f12b7ap-163,
                0x1.a1fd4dcb00974p-152,
                0x1.a1fd4dcb00974p-153,
                0x1.a1fe6ceb6fe56p-1008,
                0x1.a2020f1cd499p-518,
                0x1.a2020f1cd499p-519,
                0x1.a2020f1cd499p-520,
                0x1.a2020f1cd499p-521,
                0x1.a2055f1b2350fp-490,
                0x1.a20a3ccfa672dp315,
                0x1.a20df0dcd3afp667,
                0x1.a21f002fbcafep609,
                0x1.a21f002fbcafep610,
                0x1.a22393b33036ap-130,
                0x1.a22393b33036ap-131,
                0x1.a22393b33036ap-132,
                0x1.a22393b33036ap-133,
                0x1.a2265fe87a892p-83,
                0x1.a2265fe87a892p-84,
                0x1.a230b67d28012p583,
                0x1.a235f25da543p-857,
                0x1.a23a09d97p-1038,
                0x1.a240b7cd59aa2p341,
                0x1.a2468282f67bcp-783,
                0x1.a2468282f67bcp-784,
                0x1.a2480fff6ddddp645,
                0x1.a24e5dd1b96d8p-890,
                0x1.a24f1fe3daf11p-407,
                0x1.a24fb383f573p757,
                0x1.a24fb383f573p758,
                0x1.a254bb0805b6ep614,
                0x1.a254bb0805b6ep615,
                0x1.a25ad2030b594p-129,
                0x1.a25ad2030b594p-130,
                0x1.a26f973b7529p-582,
                0x1.a26f973b7529p-583,
                0x1.a27fbf3877722p1022,
                0x1.a28290bc66eabp-659,
                0x1.a2884176a8db2p-769,
                0x1.a28877a09a4ep-447,
                0x1.a28edc580e50ep-984,
                0x1.a28edc580e50ep-985,
                0x1.a2a01352f3178p-541,
                0x1.a2a7491778375p-314,
                0x1.a2a9bccd20d29p-345,
                0x1.a2ac8869d60f2p-545,
                0x1.a2ac8869d60f2p-546,
                0x1.a2af6a835d7f6p937,
                0x1.a2b13b10459c8p-613,
                0x1.a2c3ef1a09472p+968,
                0x1.a2cf604c30d4p-610,
                0x1.a2cf604c30d4p-611,
                0x1.a2cf604c30d4p-612,
                0x1.a2cf604c30d4p-613,
                0x1.a2cf604c30d4p-614,
                0x1.a2cf604c30d4p-615,
                0x1.a2dc66e56c7a9p819,
                0x1.a2deb6b858013p-502,
                0x1.a2ea1caaba24dp-437,
                0x1.a2edd09bb5acap+860,
                0x1.a2edd09bb5acap861,
                0x1.a2edd09bb5acap862,
                0x1.a2edd09bb5acap863,
                0x1.a2edd09bb5acap+864,
                0x1.a2ee03457ff24p655,
                0x1.a2f09e1fe6a5cp507,
                0x1.a2f09e1fe6a5cp+508,
                0x1.a2f0bc1f15988p617,
                0x1.a2fe76a3f9474p-499,
                0x1.a2fe76a3f9475p-499,
                0x1.a2fe76a3f9475p-500,
                0x1.a2p-1067,
                0x1.a304a17dc6228p-481,
                0x1.a308bf09144b2p511,
                0x1.a30a2a5767c56p-818,
                0x1.a30e450b26d9cp-107,
                0x1.a30e450b26d9cp-108,
                0x1.a30e450b26d9cp-109,
                0x1.a3131cd7bd569p+560,
                0x1.a328faf68bdb4p-64,
                0x1.a32ac316fb3acp185,
                0x1.a32ac316fb3acp186,
                0x1.a32ac316fb3acp187,
                0x1.a32ac316fb3acp+188,
                0x1.a32ac316fb3acp189,
                0x1.a3319303e5ffdp+420,
                0x1.a3333729809c6p677,
                0x1.a334b454f459dp-418,
                0x1.a336dfacbc442p578,
                0x1.a3398ce28662ep-713,
                0x1.a3398ce28662ep-714,
                0x1.a341298709d0cp-896,
                0x1.a3470da4ff23ap986,
                0x1.a3570110750bep-846,
                0x1.a357d155dbb3bp-220,
                0x1.a35c7178350fap-123,
                0x1.a35c7178350fap-124,
                0x1.a363ae4803bc7p341,
                0x1.a3647483d00a8p+520,
                0x1.a364p-1060,
                0x1.a366a3394c8ecp-165,
                0x1.a36812p-1051,
                0x1.a36c2d39caabcp-658,
                0x1.a36e2eb1c432cp-13,
                0x1.a36e2eb1c432cp-14,
                0x1.a378eca4d2796p-571,
                0x1.a378eca4d2796p-572,
                0x1.a378eca4d2796p-573,
                0x1.a378eca4d2796p-574,
                0x1.a37935f3b71c8p823,
                0x1.a37935f3b71c8p+824,
                0x1.a37935f3b71c8p825,
                0x1.a37935f3b71c8p826,
                0x1.a37935f3b71c8p827,
                0x1.a37935f3b71c8p+828,
                0x1.a383e3c30120cp-811,
                0x1.a383e3c30120cp-812,
                0x1.a383e3c30120cp-813,
                0x1.a388fb6506d3ap537,
                0x1.a399af5908778p-465,
                0x1.a399af5908778p-466,
                0x1.a39d984d3a87ap-389,
                0x1.a39db6674c27ap-784,
                0x1.a3a2f950baf58p+488,
                0x1.a3ae1d4a3ca6ep-258,
                0x1.a3ae1d4a3ca6ep-259,
                0x1.a3ae1d4a3ca6ep-260,
                0x1.a3ae1d4a3ca6ep-261,
                0x1.a3c2d2bcfcc9ep-844,
                0x1.a3c69e17e0ed2p186,
                0x1.a3c69e17e0ed2p187,
                0x1.a3c69e17e0ed2p+188,
                0x1.a3c69e17e0ed2p189,
                0x1.a3c69e17e0ed2p190,
                0x1.a3cb730b344f2p+488,
                0x1.a3d8d5e503e59p-974,
                0x1.a3ddb1129f73cp841,
                0x1.a3de04895e46cp471,
                0x1.a3de04895e46cp+472,
                0x1.a3ead732acfe8p-291,
                0x1.a3ecbc42933f8p+940,
                0x1.a3ecbc42933f8p941,
                0x1.a3fc894a07fap+612,
                0x1.a3fc894a07fap613,
                0x1.a4009bd50054ap-732,
                0x1.a407e0502945cp357,
                0x1.a412000ce725fp247,
                0x1.a41a0894eb95dp697,
                0x1.a42bd18f305b9p+564,
                0x1.a42bd18f305b9p565,
                0x1.a42bd18f305b9p566,
                0x1.a4498eb7a0c12p494,
                0x1.a4498eb7a0c12p495,
                0x1.a44b4ec8c93c2p-896,
                0x1.a44df832b8d46p+956,
                0x1.a45038bbc114ap475,
                0x1.a451935e4200ap-515,
                0x1.a4559a52950b9p+480,
                0x1.a461dcfb1f1eap430,
                0x1.a461dcfb1f1eap431,
                0x1.a461dcfb1f1eap+432,
                0x1.a461dcfb1f1eap433,
                0x1.a4625ee9e536p-592,
                0x1.a46d070e06809p+176,
                0x1.a46db10409cd4p201,
                0x1.a46db10409cd4p202,
                0x1.a4737532c9971p-543,
                0x1.a4779cb5787b2p-112,
                0x1.a47f989c67f5p405,
                0x1.a4818627b26a8p595,
                0x1.a4818627b26a8p+596,
                0x1.a4818627b26a8p597,
                0x1.a482c7c4429aep+576,
                0x1.a482d744dp-1037,
                0x1.a482d744dp-1038,
                0x1.a48a3eaa59518p737,
                0x1.a48dff814168dp-588,
                0x1.a49ep-1059,
                0x1.a4a8e56474b8cp399,
                0x1.a4a8e56474b8cp+400,
                0x1.a4a8e56474b8cp401,
                0x1.a4ad39fafc7abp-318,
                0x1.a4bc3a90ce418p-573,
                0x1.a4c4eee347358p941,
                0x1.a4c4eee347358p942,
                0x1.a4c78e0cad5cap965,
                0x1.a4ce72013e99ap-768,
                0x1.a4ce72013e99ap-769,
                0x1.a4cf9550c5426p-695,
                0x1.a4d3d028cb29cp642,
                0x1.a4d9c65c2b508p+652,
                0x1.a4d9c65c2b508p653,
                0x1.a4de8932c4bp-74,
                0x1.a4e1ddc1621ccp674,
                0x1.a4e31c75549d9p-648,
                0x1.a4f6b3ce2fb18p-232,
                0x1.a4f7f464d4356p-253,
                0x1.a4f92db74e6c6p-773,
                0x1.a5006d695fefp-30,
                0x1.a5006d695fefp-31,
                0x1.a500d3286441ap+684,
                0x1.a500d3286441ap685,
                0x1.a5052b90e9e9p-505,
                0x1.a506848dbc75cp-719,
                0x1.a506848dbc75cp-720,
                0x1.a506848dbc75cp-721,
                0x1.a506848dbc75cp-722,
                0x1.a50a259963ca6p182,
                0x1.a50b43ab3dff8p-101,
                0x1.a51859c2ceb68p905,
                0x1.a51859c2ceb68p906,
                0x1.a51859c2ceb68p907,
                0x1.a518f00f47e4fp-736,
                0x1.a518f00f47e4fp-737,
                0x1.a532592669de4p-349,
                0x1.a532592669de4p-350,
                0x1.a532592669de4p-351,
                0x1.a53bb31b369a2p217,
                0x1.a53bb31b369a2p218,
                0x1.a53bb31b369a2p219,
                0x1.a53bb31b369a2p+220,
                0x1.a53bb31b369a2p221,
                0x1.a53fc9631d10cp-205,
                0x1.a53fc9631d10cp-206,
                0x1.a53fc9631d10cp-207,
                0x1.a53fc9631d10cp-208,
                0x1.a53fc9631d10cp-209,
                0x1.a53fc9631d10cp-210,
                0x1.a53fc9631d10cp-211,
                0x1.a53fc9631d10cp-212,
                0x1.a54c13aec7bf8p125,
                0x1.a5622e27480fap-408,
                0x1.a5631e877df4ap470,
                0x1.a5635f732c81ap-169,
                0x1.a578a710dfba2p1001,
                0x1.a582d82d507cbp-607,
                0x1.a58973ecbede6p-47,
                0x1.a5907b02eeb93p-311,
                0x1.a596a6eb949f9p-335,
                0x1.a59c28ee3f432p-916,
                0x1.a5a2105e25b8bp-780,
                0x1.a5a6851e306bap-128,
                0x1.a5a78142b38c2p-912,
                0x1.a5b01b605557ap275,
                0x1.a5b01b605557bp275,
                0x1.a5c5ee5a8a57p-524,
                0x1.a5c5ee5a8a57p-525,
                0x1.a5cb1d0c3f7f4p114,
                0x1.a5cdfb5db1fe6p-562,
                0x1.a5cdfb5db1fe6p-563,
                0x1.a5db812948281p-310,
                0x1.a5db812948281p-311,
                0x1.a5db812948281p-312,
                0x1.a5ed0ef4c09f3p365,
                0x1.a5f0a59772387p-919,
                0x1.a5f8db5433a5cp-952,
                0x1.a608c08be18e4p327,
                0x1.a60c174548412p-928,
                0x1.a60c174548412p-929,
                0x1.a60c174548412p-930,
                0x1.a6208b5068394p+760,
                0x1.a6233e49694fp+352,
                0x1.a6233e49694fp353,
                0x1.a624147ee62d2p-477,
                0x1.a6294d202f3p+316,
                0x1.a63e7711cp-1040,
                0x1.a642b599d9d56p+296,
                0x1.a642b92f07ccbp-750,
                0x1.a645315f02c83p278,
                0x1.a645598d05989p-54,
                0x1.a645598d05989p-55,
                0x1.a64c3e09a713cp-838,
                0x1.a64f063b27544p-332,
                0x1.a64fdcbf2f373p-814,
                0x1.a6508b1e8e8d8p+800,
                0x1.a662c350f37f3p794,
                0x1.a662c350f37f3p795,
                0x1.a66573b079ddp662,
                0x1.a66caf30ca282p-579,
                0x1.a66caf30ca282p-580,
                0x1.a66caf30ca282p-581,
                0x1.a673df6471ee6p+472,
                0x1.a67ac9f7ea21ep-835,
                0x1.a67bf05ebdb84p879,
                0x1.a681c00c7c274p178,
                0x1.a6829aea20874p650,
                0x1.a6829aea20874p651,
                0x1.a689922a382c8p389,
                0x1.a68a21bd29fep810,
                0x1.a68a21bd29fep811,
                0x1.a68a21bd29fep+812,
                0x1.a68e0b654a254p-424,
                0x1.a693b912b76aap-425,
                0x1.a696e4bdcb7b4p+632,
                0x1.a6a2b85062ab3p-891,
                0x1.a6a2b85062ab4p-891,
                0x1.a6a6f7da759e3p446,
                0x1.a6a8cb9e9572cp374,
                0x1.a6a8cb9e9572cp375,
                0x1.a6a8cb9e9572cp+376,
                0x1.a6a8cb9e9572cp377,
                0x1.a6ae9432bc966p-537,
                0x1.a6afc26fb2386p-365,
                0x1.a6c1cf66d56p+240,
                0x1.a6ccc143672fbp+476,
                0x1.a6d14d3946c5p-508,
                0x1.a6d14d3946c5p-509,
                0x1.a6d14d3946c5p-510,
                0x1.a6d14d3946c5p-511,
                0x1.a6d14d3946c5p-512,
                0x1.a6d33f5ccbccp-574,
                0x1.a6d33f5ccbccp-575,
                0x1.a6d33f5ccbccp-576,
                0x1.a6d33f5ccbccp-577,
                0x1.a6d69df13eec6p353,
                0x1.a6daa70457156p298,
                0x1.a6dac3fcddfc8p769,
                0x1.a6db0b81fd17p-685,
                0x1.a6ddba915a35p-661,
                0x1.a6ddba915a35p-662,
                0x1.a6dfe448f675bp-993,
                0x1.a6e111634bb66p-916,
                0x1.a6e111634bb66p-917,
                0x1.a6e111634bb66p-918,
                0x1.a6e4cf866aefap-78,
                0x1.a6e4cf866aefap-79,
                0x1.a6e948d4c83fdp-450,
                0x1.a6f1eb2393a18p-893,
                0x1.a6f1eb2393a18p-894,
                0x1.a6fcf48c89cap707,
                0x1.a7036068f500cp-179,
                0x1.a706bb70379f6p926,
                0x1.a70bb7812fae4p-427,
                0x1.a70c60fe76c3ep-879,
                0x1.a7103b9e58112p318,
                0x1.a7103b9e58112p319,
                0x1.a71368f0f3046p-404,
                0x1.a71368f0f3046p-405,
                0x1.a71368f0f3046p-406,
                0x1.a71368f0f3046p-407,
                0x1.a715a314cb599p+192,
                0x1.a718e3f4ad20cp-24,
                0x1.a7208f410547fp391,
                0x1.a72aef348f77cp-228,
                0x1.a72b8p-1057,
                0x1.a7373591200bcp-146,
                0x1.a7373591200bcp-147,
                0x1.a737adf87c45ap-1018,
                0x1.a73bdc372bcd8p870,
                0x1.a740aeb153645p-757,
                0x1.a740aeb153645p-758,
                0x1.a7439edbb8d62p-785,
                0x1.a744c91ef3296p-416,
                0x1.a744c91ef3296p-417,
                0x1.a745f6f4843f8p-619,
                0x1.a745f6f4843f8p-620,
                0x1.a745f6f4843f8p-621,
                0x1.a745f6f4843f8p-622,
                0x1.a7461216e61ddp718,
                0x1.a747c58494c36p970,
                0x1.a747c58494c36p971,
                0x1.a7485931ba3efp943,
                0x1.a749df8f45d07p-669,
                0x1.a74a7f50e492ap-421,
                0x1.a74a7f50e492ap-422,
                0x1.a752a3a51297ap-102,
                0x1.a75cb9809e31ap-165,
                0x1.a75f7ba23e3e1p-698,
                0x1.a760b389de2d6p458,
                0x1.a760b389de2d6p459,
                0x1.a7616cc6dadbep+980,
                0x1.a76581fc9c4fp-324,
                0x1.a766719b869eap789,
                0x1.a769d3ede9fp970,
                0x1.a77581053543cp+632,
                0x1.a77581053543cp633,
                0x1.a784379d99db4p79,
                0x1.a788e8351838p-932,
                0x1.a78ce1807f5f8p+132,
                0x1.a78ce1807f5f8p133,
                0x1.a7a344c5f1f0ep-904,
                0x1.a7ad4d03bff06p982,
                0x1.a7ae6a012306ep-912,
                0x1.a7aee9d901bd7p325,
                0x1.a7aee9d901bd7p326,
                0x1.a7b02159aa69cp-531,
                0x1.a7b02159aa69cp-532,
                0x1.a7b02159aa69cp-533,
                0x1.a7b45fdd20a93p119,
                0x1.a7b85b007086cp-529,
                0x1.a7bb2c9ac4aebp-215,
                0x1.a7bc045582f63p521,
                0x1.a7bc8f92d6b1p-809,
                0x1.a7c75070a382dp-108,
                0x1.a7cadd0c95702p-857,
                0x1.a7cadd0c95702p-858,
                0x1.a7cadd0c95702p-859,
                0x1.a7cadd0c95702p-860,
                0x1.a7cf7a5ad1febp+380,
                0x1.a7d1bb49f628cp-289,
                0x1.a7d808d1eac8ap269,
                0x1.a7d808d1eac8ap270,
                0x1.a7d808d1eac8ap271,
                0x1.a7d87003a4b4ep855,
                0x1.a7dc24346a1e8p-306,
                0x1.a7e6f29295681p-131,
                0x1.a7eaeb17c874ep586,
                0x1.a7eaeb17c874ep587,
                0x1.a7eaeb17c874ep+588,
                0x1.a7f180d2726b4p+736,
                0x1.a7f5245e5a2cep+564,
                0x1.a801af9f534c8p-163,
                0x1.a80342b3f4beep111,
                0x1.a80a6e566428cp-654,
                0x1.a80a6e566428cp-655,
                0x1.a80a6e566428cp-656,
                0x1.a80a9dc6dac01p+208,
                0x1.a814cdbdfff3cp-644,
                0x1.a81bb7ceb290cp853,
                0x1.a81bb7ceb290cp854,
                0x1.a81da5cfed1fep-418,
                0x1.a81da5cfed1fep-419,
                0x1.a81da5cfed1fep-420,
                0x1.a82415f593934p773,
                0x1.a82415f593934p774,
                0x1.a82a1631eeb3p-625,
                0x1.a82a1631eeb3p-626,
                0x1.a82a1631eeb3p-627,
                0x1.a82c5572c8617p463,
                0x1.a830d7476a046p237,
                0x1.a830d7476a046p238,
                0x1.a830d7476a046p239,
                0x1.a830d7476a046p+240,
                0x1.a830d7476a046p241,
                0x1.a832bbc136a9fp-482,
                0x1.a832bbc136a9fp-483,
                0x1.a83dc8a9e9116p-617,
                0x1.a83dc8a9e9116p-618,
                0x1.a846ab96936aap-976,
                0x1.a850056826498p-488,
                0x1.a850056826498p-489,
                0x1.a850056826498p-490,
                0x1.a855659a1acdp-772,
                0x1.a855f64889ef8p625,
                0x1.a857a43f9cce4p+848,
                0x1.a85c6f7fba05ep977,
                0x1.a85c6f7fba05ep978,
                0x1.a867e2add9b8ep185,
                0x1.a867e2add9b8ep186,
                0x1.a8713b1802806p315,
                0x1.a8713b1802806p+316,
                0x1.a8714903ab7e1p-729,
                0x1.a879db638ap-1035,
                0x1.a87b9caec3b7cp+644,
                0x1.a87b9caec3b7cp645,
                0x1.a87b9caec3b7cp646,
                0x1.a87b9caec3b7cp647,
                0x1.a87c262151e1ap250,
                0x1.a88634b2d433ep965,
                0x1.a88634b2d433ep966,
                0x1.a88634b2d433ep967,
                0x1.a89628a046fdcp674,
                0x1.a89628a046fdcp675,
                0x1.a89da80444854p-666,
                0x1.a8a1cdcf5ef96p229,
                0x1.a8a222d16f626p+932,
                0x1.a8a222d16f626p933,
                0x1.a8a222d16f626p934,
                0x1.a8a222d16f626p935,
                0x1.a8af0b45d9532p-12,
                0x1.a8b14755df1c6p+888,
                0x1.a8b14755df1c6p889,
                0x1.a8b14755df1c6p890,
                0x1.a8b263af79514p-37,
                0x1.a8c1c0593d566p-500,
                0x1.a8de5563b49a2p707,
                0x1.a8de5563b49a2p+708,
                0x1.a8e0b66a67c1ep865,
                0x1.a8e2ec05da002p758,
                0x1.a8e7f1a6cc298p-852,
                0x1.a8e90f9908e0cp-601,
                0x1.a8e90f9908e0cp-602,
                0x1.a8ecff003bf3fp814,
                0x1.a8eff5720f9cp-452,
                0x1.a8f55baccp-1040,
                0x1.a8f9d4bbfd916p-87,
                0x1.a90077c8d8dc3p-757,
                0x1.a9023b0ed9301p303,
                0x1.a909ac78672d8p+460,
                0x1.a909ac78672d8p461,
                0x1.a909ac78672d8p462,
                0x1.a91a1555ce676p+208,
                0x1.a9234c34a96c6p-812,
                0x1.a9269ac3b96acp-381,
                0x1.a92cf61098e32p+636,
                0x1.a934960280451p-639,
                0x1.a93f5ef033f6ap-587,
                0x1.a93f5ef033f6ap-588,
                0x1.a943a055525e4p+668,
                0x1.a943a055525e4p669,
                0x1.a943a055525e4p670,
                0x1.a943a055525e4p671,
                0x1.a943a055525e4p+672,
                0x1.a943a055525e4p673,
                0x1.a94be791248d8p-721,
                0x1.a94be791248d8p-722,
                0x1.a94ead5f77f8p-957,
                0x1.a94ead5f77f8p-958,
                0x1.a9527d1d99f4fp-825,
                0x1.a9534f4af375cp-182,
                0x1.a95394c43297cp385,
                0x1.a959ad46e15e6p-934,
                0x1.a959ad46e15e6p-935,
                0x1.a95a5b7f87a0ep-117,
                0x1.a95a5b7f87a0fp-117,
                0x1.a95a5b7f87a0fp-118,
                0x1.a978239b270d6p-449,
                0x1.a97cc5e9176f6p-865,
                0x1.a9831f95635acp134,
                0x1.a9831f95635acp135,
                0x1.a9831f95635acp+136,
                0x1.a98b48b0b2b95p+156,
                0x1.a992f2fd0179ap-296,
                0x1.a998c480a220fp161,
                0x1.a99974096154p-442,
                0x1.a99974096154p-443,
                0x1.a999ddec72acap+600,
                0x1.a999ddec72acap601,
                0x1.a9a0ec1051eeap-612,
                0x1.a9a48ddf509p-1030,
                0x1.a9a5067bb1936p461,
                0x1.a9ace4a9d76fap527,
                0x1.a9b375b76c188p-74,
                0x1.a9b375b76c188p-75,
                0x1.a9b375b76c188p-76,
                0x1.a9c932c4aa884p+740,
                0x1.a9c932c4aa884p741,
                0x1.a9ca01f84e929p-950,
                0x1.a9cbc3802d486p942,
                0x1.a9cbc59b83a3dp+368,
                0x1.a9cbc59b83a3ep+368,
                0x1.a9cd50024da3dp802,
                0x1.a9cf99974a56ap-664,
                0x1.a9d01955161dap-151,
                0x1.a9d01955161dap-152,
                0x1.a9d681d589e08p-378,
                0x1.a9dba5190c7ebp-152,
                0x1.a9dba5190c7ebp-153,
                0x1.a9e616e7f922ep906,
                0x1.a9e616e7f922ep907,
                0x1.a9e616e7f922ep+908,
                0x1.a9e90059d12dcp105,
                0x1.a9f543f9f8943p+748,
                0x1.a9fdd710948fp-941,
                0x1.aa00af1a9c59ap138,
                0x1.aa03dc85128a9p+416,
                0x1.aa063c26db411p330,
                0x1.aa0cb1e6ff0f5p-851,
                0x1.aa102cc721b08p-370,
                0x1.aa102cc721b08p-371,
                0x1.aa257624a8834p-200,
                0x1.aa257624a8834p-201,
                0x1.aa257624a8834p-202,
                0x1.aa257624a8834p-203,
                0x1.aa273f141989bp-696,
                0x1.aa28be6471dc5p-1002,
                0x1.aa2d1bc414b69p465,
                0x1.aa2f2342ed1ccp-93,
                0x1.aa365786eb2f6p-979,
                0x1.aa365786eb2f6p-980,
                0x1.aa365786eb2f6p-981,
                0x1.aa365786eb2f6p-982,
                0x1.aa3d4df50af0ap853,
                0x1.aa3d4df50af0bp853,
                0x1.aa4eb9cbep-1039,
                0x1.aa65b58639e69p-909,
                0x1.aa65b58639e69p-910,
                0x1.aa65b58639e69p-911,
                0x1.aa65b58639e69p-912,
                0x1.aa6acb41dfc52p-22,
                0x1.aa6d4cfaf4b6ep-844,
                0x1.aa6d4cfaf4b6ep-845,
                0x1.aa6db6eaf5d48p+592,
                0x1.aa6dc3bda13e7p-1006,
                0x1.aa6dc3bda13e7p-1007,
                0x1.aa7aa38bfd6b6p-717,
                0x1.aa7aa38bfd6b6p-718,
                0x1.aa7c959b6a667p-34,
                0x1.aa890608950dp+556,
                0x1.aa903dc6f6cecp-749,
                0x1.aa92c28p-1049,
                0x1.aa92cde11f332p-922,
                0x1.aa99eb56abd72p855,
                0x1.aa99eb56abd72p+856,
                0x1.aa9bc2521106ap+612,
                0x1.aa9bc2521106ap613,
                0x1.aa9bc2521106ap614,
                0x1.aaa7b04985fc8p581,
                0x1.aaa7b04985fc8p582,
                0x1.aab19b2d696eap-187,
                0x1.aab439364ac7cp734,
                0x1.aac0bf9b9e65cp-798,
                0x1.aac509bdf3c34p539,
                0x1.aac509bdf3c34p+540,
                0x1.aac509bdf3c34p541,
                0x1.aac9729c0c568p854,
                0x1.aacb0a2ef7883p-563,
                0x1.aacfcb88c92d6p-391,
                0x1.aacfcb88c92d6p-392,
                0x1.aacfcb88c92d6p-393,
                0x1.aad6db1ea7ab9p-880,
                0x1.aadd46c314ac5p315,
                0x1.aae9193c42218p178,
                0x1.aaf2d98f0c0dcp+196,
                0x1.aaf2d98f0c0dcp197,
                0x1.aaf2d98f0c0dcp198,
                0x1.aaf2d98f0c0dcp199,
                0x1.aaf2d98f0c0dcp+200,
                0x1.aaf2d98f0c0dcp201,
                0x1.aaf2d98f0c0dcp202,
                0x1.aaf2d98f0c0dcp203,
                0x1.aaf3b43c2f7e9p+956,
                0x1.aaf686a3479bap-241,
                0x1.ab03b5db07289p-109,
                0x1.ab063d450dabap-534,
                0x1.ab0b6089d61bep-635,
                0x1.ab10c016c34b8p110,
                0x1.ab161a1b57ee6p-312,
                0x1.ab161a1b57ee6p-313,
                0x1.ab161a1b57ee6p-314,
                0x1.ab161a1b57ee6p-315,
                0x1.ab226fc058a21p959,
                0x1.ab2bc817b366bp-313,
                0x1.ab2c525fc23f8p238,
                0x1.ab328946f80eap-313,
                0x1.ab38b8c710cb4p-825,
                0x1.ab437686a176dp665,
                0x1.ab5ff45dbf0f3p-23,
                0x1.ab631cacb2884p-272,
                0x1.ab631cacb2884p-273,
                0x1.ab75e8e20ef48p237,
                0x1.ab7fe8b6a2d5ap+936,
                0x1.ab82f4dfb2868p-968,
                0x1.ab92ec91be736p-512,
                0x1.aba0ea5109315p-726,
                0x1.aba4714957d3p+172,
                0x1.aba550908ebb2p354,
                0x1.abb881ff25b12p+872,
                0x1.abba93279880ep-627,
                0x1.abba93279880ep-628,
                0x1.abbb989f44b46p-812,
                0x1.abc8e236e5a5ep-951,
                0x1.abcdf102739fp-917,
                0x1.abcf4041a6fc1p361,
                0x1.abd29a455160ap907,
                0x1.abdf8c9a549d3p726,
                0x1.abdfb47c570a6p783,
                0x1.abdfb47c570a6p+784,
                0x1.abe78ebd708ep715,
                0x1.abe78ebd708ep+716,
                0x1.abfb350bc97a2p-561,
                0x1.abfb350bc97a2p-562,
                0x1.abfc227ab1026p-29,
                0x1.abfc227ab1026p-30,
                0x1.ac00d02b531c9p-788,
                0x1.ac0fe70fdf678p-184,
                0x1.ac13b9p-1050,
                0x1.ac1677aad4ab1p657,
                0x1.ac1677aad4abp657,
                0x1.ac17798b3a2cp-83,
                0x1.ac17798b3a2cp-84,
                0x1.ac199b8220311p390,
                0x1.ac1e69cb437e5p769,
                0x1.ac292c6b8a05ap162,
                0x1.ac292c6b8a05ap163,
                0x1.ac2c9f6fcf111p-142,
                0x1.ac2feed88c46bp-870,
                0x1.ac3796b6f426p-700,
                0x1.ac3796b6f426p-701,
                0x1.ac42d0f5ec4b7p+260,
                0x1.ac4b2b1ed5c58p+628,
                0x1.ac4f2dcea051ep+944,
                0x1.ac690ac5cffa2p-288,
                0x1.ac83a7a460f5p-741,
                0x1.ac8795a7a7158p+368,
                0x1.ac879978b6938p-620,
                0x1.ac8c4a1fb036p-664,
                0x1.ac9a7b3b7302fp-994,
                0x1.ac9a7b3b7302fp-995,
                0x1.ac9ca2203ad34p961,
                0x1.ac9d95b405e9cp-42,
                0x1.ac9d95b405e9cp-43,
                0x1.aca538c61ba9cp-52,
                0x1.acb014c936a4fp174,
                0x1.acb068cd9b9cp729,
                0x1.acb3c4be4f1d6p-636,
                0x1.acbf0d011a63ep-182,
                0x1.acc0541ca09c8p+812,
                0x1.acc0541ca09c8p813,
                0x1.acc46749dccfep-867,
                0x1.acc46749dccfep-868,
                0x1.acc46749dccfep-869,
                0x1.acc4f13cc3d3dp-549,
                0x1.acd1049e90cbp-432,
                0x1.acdd663b7e21ap-147,
                0x1.acdd663b7e21ap-148,
                0x1.ace5406f23ap-1031,
                0x1.acf26c7ad3242p-711,
                0x1.acp-1067,
                0x1.ad00d700c51e4p-594,
                0x1.ad00d700c51e4p-595,
                0x1.ad00d700c51e4p-596,
                0x1.ad042855a6ca8p+752,
                0x1.ad0b6158dc2e9p+652,
                0x1.ad0ca35324c2cp-651,
                0x1.ad0ca35324c2cp-652,
                0x1.ad0cc33744e4ap-509,
                0x1.ad111342b2f38p+828,
                0x1.ad11a1b2c5c65p-842,
                0x1.ad1c7ff0e1dep+264,
                0x1.ad1f2a6f5963dp-929,
                0x1.ad202dec3487fp-357,
                0x1.ad202dec3487fp-358,
                0x1.ad23a26d1fa2ap-991,
                0x1.ad2e3fee284b1p429,
                0x1.ad40471f0d58bp-843,
                0x1.ad4d59ee32e2ap+364,
                0x1.ad4d59ee32e2ap365,
                0x1.ad5ae00cce5d2p+928,
                0x1.ad7f29abcaf48p-23,
                0x1.ad7f29abcaf48p-24,
                0x1.ad9973c1f4306p-436,
                0x1.ad9c3c5ddbdc3p551,
                0x1.ada8d31f3abffp790,
                0x1.adac22e7a1dd1p-637,
                0x1.adb78446aa59ep627,
                0x1.adb78446aa59ep+628,
                0x1.adb9318bf4345p-96,
                0x1.adbd3594cbff4p690,
                0x1.adc771cad902dp+592,
                0x1.adcb6f71d1512p769,
                0x1.adcb6f71d1512p770,
                0x1.adcb6f71d1512p771,
                0x1.add4af52p-1042,
                0x1.add4af52p-1043,
                0x1.addac3f165154p-403,
                0x1.addac3f165154p-404,
                0x1.ade1e7afd2d8ep-407,
                0x1.ade59c2ee2f16p406,
                0x1.ade841151659cp1011,
                0x1.adf1aea12525ap461,
                0x1.adf1aea12525bp461,
                0x1.adfc704da8724p677,
                0x1.ae03f245703e2p+484,
                0x1.ae07eefe47fdcp-338,
                0x1.ae0ada618ecb9p-50,
                0x1.ae0be7c5ed725p789,
                0x1.ae0dcf437828p+368,
                0x1.ae1335d60c85ep-303,
                0x1.ae1852aef314p+192,
                0x1.ae1852aef314p193,
                0x1.ae1852aef314p194,
                0x1.ae22ba54482c4p-461,
                0x1.ae22cf7d37e24p-326,
                0x1.ae36e78b9a9e1p201,
                0x1.ae4c091e08b6ap-450,
                0x1.ae4c091e08b6ap-451,
                0x1.ae4c4c979c6fp+460,
                0x1.ae4c63a932c39p-735,
                0x1.ae52465147cc6p-41,
                0x1.ae52465147cc6p-42,
                0x1.ae5354f4ad774p-733,
                0x1.ae5354f4ad774p-734,
                0x1.ae5568f4c168fp+140,
                0x1.ae5c0fef20e75p-158,
                0x1.ae5eda569b0a2p-943,
                0x1.ae5eda569b0a2p-944,
                0x1.ae5eda569b0a2p-945,
                0x1.ae5eda569b0a2p-946,
                0x1.ae5eda569b0a2p-947,
                0x1.ae5eda569b0a2p-948,
                0x1.ae6148e3902b3p-430,
                0x1.ae6148e3902b3p-431,
                0x1.ae6148e3902b3p-432,
                0x1.ae64521f7595ep946,
                0x1.ae6d10905cab9p929,
                0x1.ae73a8da06904p-990,
                0x1.ae73fd1c3f4bp-747,
                0x1.ae73fd1c3f4bp-748,
                0x1.ae76ea50ae6f5p-884,
                0x1.ae7b358f31bb6p462,
                0x1.ae82d3d8ded41p-855,
                0x1.ae83da9c69d5ep177,
                0x1.ae91e9d18e7f9p546,
                0x1.ae93e923b7fe2p-339,
                0x1.ae9839e53223ep-377,
                0x1.aeac2d1ea2696p506,
                0x1.aeb332c039764p-690,
                0x1.aeb37d43d3106p130,
                0x1.aec7aa75404bap-984,
                0x1.aec7aa75404bap-985,
                0x1.aec7aa75404bap-986,
                0x1.aecd0ac2d595p+356,
                0x1.aecedd7ee5341p1011,
                0x1.aed1d03b19f85p350,
                0x1.aed2981d135b6p939,
                0x1.aed2981d135b6p+940,
                0x1.aed39c3919d72p+864,
                0x1.aed8c3c041b5cp-445,
                0x1.aed8c3c041b5cp-446,
                0x1.aed8c3c041b5cp-447,
                0x1.aedaa0fc32ac8p-222,
                0x1.aee0d2fea53cp955,
                0x1.aee90b964b047p-705,
                0x1.aee90b964b048p-704,
                0x1.aee90b964b048p-705,
                0x1.aee9c93097396p-177,
                0x1.aeed606b4f03ep+632,
                0x1.aef00755011cep-93,
                0x1.aef71446a4628p631,
                0x1.aefc5209fe4cbp-243,
                0x1.aefc5209fe4cbp-244,
                0x1.aefd20482da5dp251,
                0x1.aep-1067,
                0x1.af027f48dfe8ep230,
                0x1.af027f48dfe8ep231,
                0x1.af027f48dfe8ep+232,
                0x1.af027f48dfe8ep233,
                0x1.af027f48dfe8ep234,
                0x1.af027f48dfe8ep235,
                0x1.af0811889659ap-68,
                0x1.af14abb25b3cap890,
                0x1.af14abb25b3cap891,
                0x1.af158e8c3700ep-937,
                0x1.af1bfaa13611ap-515,
                0x1.af1f588619072p825,
                0x1.af34ddf3227ccp477,
                0x1.af34ddf3227ccp478,
                0x1.af34ddf3227ccp479,
                0x1.af34ddf3227ccp+480,
                0x1.af34ddf3227ccp481,
                0x1.af443cc544bdap+408,
                0x1.af45a715b54d2p-362,
                0x1.af45a715b54d2p-363,
                0x1.af4f1443431f7p-871,
                0x1.af53a6cf81bcp+704,
                0x1.af53a6cf81bcp705,
                0x1.af54ca8b8e0cep-343,
                0x1.af54ca8b8e0cep-344,
                0x1.af54ca8b8e0cep-345,
                0x1.af54ca8b8e0cep-346,
                0x1.af54ca8b8e0cep-347,
                0x1.af5b5378aa2e6p+448,
                0x1.af5b5378aa2e6p449,
                0x1.af5b5378aa2e6p450,
                0x1.af5b5378aa2e6p451,
                0x1.af5b5378aa2e6p+452,
                0x1.af5b5378aa2e6p453,
                0x1.af5bf109550f2p-220,
                0x1.af5dd5b7795p942,
                0x1.af6886d7d6c83p115,
                0x1.af71715f40c3cp-313,
                0x1.af739b1347e6p-411,
                0x1.af7d91d9259d7p-671,
                0x1.af7d91d9259d7p-672,
                0x1.af802a8e0c2dep177,
                0x1.af85bbc2ab145p-639,
                0x1.af8723a782752p-238,
                0x1.af8c280c1e528p131,
                0x1.af94b6d37417ap-207,
                0x1.af94b6d37417ap-208,
                0x1.af94b6d37417ap-209,
                0x1.af94b6d37417ap-210,
                0x1.af968de8f3677p457,
                0x1.af968de8f3677p458,
                0x1.af9c088d0ee0ep-748,
                0x1.afa1c0ce149a5p218,
                0x1.afacb9cdbf2e4p-842,
                0x1.afc183f41575cp410,
                0x1.afc8a494604c7p-664,
                0x1.afca48a905c34p+532,
                0x1.afcef51f0fb5ep265,
                0x1.afcef51f0fb5fp262,
                0x1.afcef51f0fb5fp263,
                0x1.afcef51f0fb5fp+264,
                0x1.afcef51f0fb5fp265,
                0x1.afe07a4af606cp839,
                0x1.afe07a4af606cp+840,
                0x1.afe07a4af606cp841,
                0x1.afef0ce3475cp-650,
                0x1.afef0ce3475cp-651,
                0x1.afef0ce3475cp-652,
                0x1.afef1caf0a738p403,
                0x1.afef1caf0a738p+404,
                0x1.aff30d11dbf48p-373,
                0x1.aff861a3e9759p290,
                0x1.affc6d8ade758p-891,
                0x1.affee27b02068p-969,
                0x1.affee27b02068p-970,
                0x1.ap-1070,
                0x1.ap-1071,
                0x1.b00593945a4c7p-1000,
                0x1.b008215a8ad24p143,
                0x1.b0156502066a4p-454,
                0x1.b01d1817c4618p787,
                0x1.b0231695885c8p-150,
                0x1.b0231695885c8p-151,
                0x1.b02c898989c18p197,
                0x1.b02c898989c18p198,
                0x1.b02c898989c18p199,
                0x1.b02c898989c18p+200,
                0x1.b02c898989c18p201,
                0x1.b02e4e50c5036p698,
                0x1.b03365f0b479dp-815,
                0x1.b04161e29445ep-514,
                0x1.b04217dfa61dfp750,
                0x1.b04217dfa61ep750,
                0x1.b04427ef7a632p986,
                0x1.b04da9165d03ap-470,
                0x1.b04da9165d03ap-471,
                0x1.b04e5acf997bp+776,
                0x1.b04f54dcdabp-844,
                0x1.b04f54dcdabp-845,
                0x1.b050dfd902fc2p455,
                0x1.b050dfd902fc2p+456,
                0x1.b050dfd902fc2p457,
                0x1.b050dfd902fc2p458,
                0x1.b056f787e0af8p-323,
                0x1.b05876e5b012p-49,
                0x1.b05943f62349p-474,
                0x1.b05943f62349p-475,
                0x1.b05943f62349p-476,
                0x1.b05943f62349p-477,
                0x1.b05ad7bb109c8p-131,
                0x1.b05ad7bb109c8p-132,
                0x1.b05e5f43943c2p657,
                0x1.b0618805a90a6p782,
                0x1.b067be3c1981fp-680,
                0x1.b073435156486p-810,
                0x1.b073435156486p-811,
                0x1.b075bec74fc28p535,
                0x1.b077ee7615d9ep285,
                0x1.b077ee7615d9ep286,
                0x1.b08f0094136bp189,
                0x1.b08f0094136bp190,
                0x1.b0a682510dee8p-280,
                0x1.b0a682510dee8p-281,
                0x1.b0a682510dee8p-282,
                0x1.b0a682510dee8p-283,
                0x1.b0a682510dee8p-284,
                0x1.b0a682510dee8p-285,
                0x1.b0a81bda1bdf6p-229,
                0x1.b0ab27b786313p934,
                0x1.b0ab718750ae2p+180,
                0x1.b0abd4ca63312p-280,
                0x1.b0b236c2944f2p-57,
                0x1.b0bc7ecd3fbe2p-826,
                0x1.b0c764ac6d3a9p-901,
                0x1.b0c764ac6d3aap-901,
                0x1.b0d330b7ea47p-200,
                0x1.b0d330b7ea47p-201,
                0x1.b0d330b7ea47p-202,
                0x1.b0d330b7ea47p-203,
                0x1.b0d330b7ea47p-204,
                0x1.b0d330b7ea47p-205,
                0x1.b0d330b7ea47p-206,
                0x1.b0de9b39eddbap325,
                0x1.b0f9e79c295a5p-764,
                0x1.b113a00813ce7p-720,
                0x1.b11722a3e6531p814,
                0x1.b11af85af7cb9p975,
                0x1.b11cc8dbb3a6fp805,
                0x1.b12269990ae1dp683,
                0x1.b12406eaa0aaap-43,
                0x1.b125d1a3a17b6p163,
                0x1.b12762d7b4274p353,
                0x1.b12762d7b4274p354,
                0x1.b12762d7b4274p355,
                0x1.b12794e8c87c2p906,
                0x1.b1285ea2114a6p-828,
                0x1.b13ac9aaf4c0ep-416,
                0x1.b13ac9aaf4c0fp-416,
                0x1.b156ac4c2d1e6p311,
                0x1.b156ac4c2d1e6p+312,
                0x1.b156ac4c2d1e6p313,
                0x1.b156ac4c2d1e6p314,
                0x1.b15b62cbac36bp-315,
                0x1.b16edef35a80cp-909,
                0x1.b1733f9ab06cbp490,
                0x1.b17684a92b8f9p-504,
                0x1.b17684a92b8f9p-505,
                0x1.b17af588292e6p+200,
                0x1.b17af588292e6p201,
                0x1.b17f2867c7df2p-951,
                0x1.b17f2867c7df2p-952,
                0x1.b17f2867c7df2p-953,
                0x1.b17f2867c7df2p-954,
                0x1.b17f2867c7df2p-955,
                0x1.b17f2867c7df2p-956,
                0x1.b1814691bad49p-741,
                0x1.b182da4721d6p-296,
                0x1.b1862e85e0ff4p991,
                0x1.b18fd07e7d9p779,
                0x1.b1acc45bb63fdp583,
                0x1.b1ae4d6e2ef5p69,
                0x1.b1aff39dd5f14p558,
                0x1.b1b6409348786p571,
                0x1.b1b6409348786p+572,
                0x1.b1bcbea8716c2p-384,
                0x1.b1bffd81f319ep-318,
                0x1.b1bffd81f319ep-319,
                0x1.b1bffd81f319ep-320,
                0x1.b1bffd81f319ep-321,
                0x1.b1c91b7d83924p607,
                0x1.b1c91b7d83924p+608,
                0x1.b1c91b7d83924p609,
                0x1.b1d70c9f5e405p-835,
                0x1.b1df536c13eeep-306,
                0x1.b1e0b8739b8e6p+236,
                0x1.b1e14f2bac3dp302,
                0x1.b1e14f2bac3dp303,
                0x1.b1e805b69f17fp1013,
                0x1.b1fb907c137b2p-426,
                0x1.b20275cfa6c7p342,
                0x1.b20275cfa6c7p343,
                0x1.b20275cfa6c7p+344,
                0x1.b20275cfa6c7p345,
                0x1.b20275cfa6c7p346,
                0x1.b20c2f4f8d4ap-137,
                0x1.b20c2f4f8d4ap-138,
                0x1.b221effe500d4p554,
                0x1.b2261dd214e61p371,
                0x1.b2286eb9e0f5ep-564,
                0x1.b2286eb9e0f5ep-565,
                0x1.b230651263115p101,
                0x1.b237f0702c254p-182,
                0x1.b23b50fc204dap-438,
                0x1.b23b50fc204dap-439,
                0x1.b23e56c698b8p461,
                0x1.b23e56c698b8p462,
                0x1.b23e56c698b8p463,
                0x1.b23f33c28762ap606,
                0x1.b23f33c28762ap607,
                0x1.b241e51e895f7p145,
                0x1.b244e6d40b218p-93,
                0x1.b244e6d40b218p-94,
                0x1.b244e6d40b218p-95,
                0x1.b244e6d40b218p-96,
                0x1.b248728b9c116p-441,
                0x1.b248728b9c116p-442,
                0x1.b248728b9c116p-443,
                0x1.b248728b9c116p-444,
                0x1.b248728b9c116p-445,
                0x1.b248728b9c116p-446,
                0x1.b248728b9c116p-447,
                0x1.b248728b9c116p-448,
                0x1.b24ac909da376p+284,
                0x1.b24ac909da376p285,
                0x1.b260ba472793ep-76,
                0x1.b260ba472793ep-77,
                0x1.b2613cb179dccp+808,
                0x1.b2613cb179dccp809,
                0x1.b2613cb179dccp810,
                0x1.b2613cb179dccp811,
                0x1.b2613cb179dccp+812,
                0x1.b263b7159b3c9p745,
                0x1.b26c5d8cff612p-713,
                0x1.b2722682970aep-758,
                0x1.b28e3641f1d54p-511,
                0x1.b29090076b353p510,
                0x1.b2925695b8a16p-1011,
                0x1.b2925695b8a16p-1012,
                0x1.b297cad9f70b6p95,
                0x1.b297cad9f70b6p+96,
                0x1.b297cad9f70b6p97,
                0x1.b2a130e176b6fp346,
                0x1.b2ab18615fcc6p-575,
                0x1.b2ab18615fcc6p-576,
                0x1.b2ab18615fcc6p-577,
                0x1.b2ac2f3cbe57bp+240,
                0x1.b2b312893ba92p-70,
                0x1.b2b8ddcabf771p575,
                0x1.b2c2a8c7e621fp+172,
                0x1.b2c40b4959d36p-581,
                0x1.b2c7a6b79bcb3p589,
                0x1.b2d15310039cbp-525,
                0x1.b2db9d4223feep-439,
                0x1.b2db9d4223feep-440,
                0x1.b2db9d4223feep-441,
                0x1.b2db9d4223feep-442,
                0x1.b2db9d4223feep-443,
                0x1.b2dc6491a81ccp-180,
                0x1.b2dc6491a81ccp-181,
                0x1.b2e32e464e9c3p+544,
                0x1.b2eed32d4b5b2p134,
                0x1.b2f37e8d56999p699,
                0x1.b2fd6d3da2442p+560,
                0x1.b2p-1066,
                0x1.b2p-1067,
                0x1.b301cb231bce5p-531,
                0x1.b30a93190c55p-287,
                0x1.b313b45225fccp-432,
                0x1.b313b45225fccp-433,
                0x1.b316cfa69b6e6p435,
                0x1.b31bb5dc320d2p-610,
                0x1.b31bb5dc320d2p-611,
                0x1.b31bb5dc320d2p-612,
                0x1.b31cb2c5ba345p-553,
                0x1.b31f17b95c96cp-865,
                0x1.b327b2541f772p+140,
                0x1.b339803e79262p-892,
                0x1.b33b0e06231edp-37,
                0x1.b34bcb6227facp-436,
                0x1.b364d8ceca30cp282,
                0x1.b364d8ceca30cp283,
                0x1.b3707abe12c45p-849,
                0x1.b37a4a02be3cap-809,
                0x1.b37c4b7928317p+280,
                0x1.b37c4b7928317p281,
                0x1.b386c844f57cep855,
                0x1.b38b7299a05a4p-914,
                0x1.b38fb9daa78e4p-127,
                0x1.b39ab78718832p-153,
                0x1.b39ab78718832p-154,
                0x1.b39ae1909c31bp-686,
                0x1.b39ae1909c31bp-687,
                0x1.b3a644a48ccccp-926,
                0x1.b3a7e659d282ap-84,
                0x1.b3a7e659d282ap-85,
                0x1.b3ab4e1b1a5a8p-978,
                0x1.b3abb0fcbeeccp885,
                0x1.b3ad8c00d8ed3p-830,
                0x1.b3c57a4fad61bp509,
                0x1.b3c82721500f4p995,
                0x1.b3cb71dd41a23p507,
                0x1.b3e2a769f7664p729,
                0x1.b3e9cb8095923p-773,
                0x1.b3ee0a03efc68p331,
                0x1.b3f3ce1763f6cp-616,
                0x1.b3f3ce1763f6cp-617,
                0x1.b3f66bac7c8a6p-359,
                0x1.b403dcc834e12p358,
                0x1.b4113d89f1564p-457,
        };
    }

    static double[] hard3() {
        return new double[] {
                0x1.b4113d89f1564p-458,
                0x1.b41a6d94cd986p-438,
                0x1.b41a6d94cd986p-439,
                0x1.b41a6d94cd986p-440,
                0x1.b41a6d94cd986p-441,
                0x1.b41b7a23a65e1p882,
                0x1.b42861ee34304p-669,
                0x1.b42a73640d943p-957,
                0x1.b42a73640d943p-958,
                0x1.b42a73640d943p-959,
                0x1.b42b6c0e2e125p-239,
                0x1.b42f1cp-1052,
                0x1.b43561f846f3ap97,
                0x1.b436f55e36c45p-534,
                0x1.b4422ae23c827p733,
                0x1.b44b361e06ee4p-880,
                0x1.b44b521b4d2a4p-921,
                0x1.b44c69e2b0842p-747,
                0x1.b44c69e2b0842p-748,
                0x1.b44d0cf6d1a14p-735,
                0x1.b44e13e551864p-115,
                0x1.b44e13e551864p-116,
                0x1.b44e13e551864p-117,
                0x1.b45066e9e82bdp+936,
                0x1.b452e0b44f0efp917,
                0x1.b4542206bdcaep669,
                0x1.b4542206bdcaep670,
                0x1.b460cba481266p-123,
                0x1.b464779b81366p715,
                0x1.b4781ead1989ep841,
                0x1.b4781ead1989ep842,
                0x1.b4781ead1989ep843,
                0x1.b4781ead1989ep+844,
                0x1.b4781ead1989ep845,
                0x1.b4781ead1989ep846,
                0x1.b4781ead1989ep847,
                0x1.b47ee893ba2a2p279,
                0x1.b4977d1b6338bp-713,
                0x1.b4b0f10f6ce1ep+368,
                0x1.b4b3c681d3b1bp-387,
                0x1.b4ba1e1cd6c94p283,
                0x1.b4c68207cbec7p866,
                0x1.b4cd4ceb20abcp-725,
                0x1.b4cd4e1e02638p-101,
                0x1.b4cfd1cf9b342p+812,
                0x1.b4d03694ae396p873,
                0x1.b4d37fa06864bp110,
                0x1.b4dd34e0c20ep-89,
                0x1.b4e29d47928fdp227,
                0x1.b4e29d47928fdp+228,
                0x1.b4ec58352128ap165,
                0x1.b4f3a1ef0477p-913,
                0x1.b4f3a1ef0477p-914,
                0x1.b4f3a1ef0477p-915,
                0x1.b4f8b64fa125ep370,
                0x1.b4f8b64fa125ep371,
                0x1.b4f8b64fa125ep+372,
                0x1.b4f8b64fa125ep373,
                0x1.b4fa6497ef8e3p503,
                0x1.b4fb0c717540cp875,
                0x1.b4feb7eb212cdp-808,
                0x1.b4feb7eb212cdp-809,
                0x1.b5031c6866284p529,
                0x1.b50a079009b09p-473,
                0x1.b512f5313aaaap-250,
                0x1.b5155dd5417fap118,
                0x1.b5171098b24a7p-515,
                0x1.b5188cd42fd02p-598,
                0x1.b523a9001ce77p-473,
                0x1.b525916b7692cp-696,
                0x1.b532efb95a7ecp-1021,
                0x1.b55c00c25a586p-221,
                0x1.b56517ae9031bp+400,
                0x1.b56517ae9031bp401,
                0x1.b566398407754p193,
                0x1.b566398407754p194,
                0x1.b566398407754p195,
                0x1.b566398407754p+196,
                0x1.b5705802c43d5p+376,
                0x1.b5733cb32b11p-323,
                0x1.b577a25c44698p1014,
                0x1.b577a25c44698p1015,
                0x1.b5790e22815b2p645,
                0x1.b5790e22815b2p646,
                0x1.b5790e22815b2p647,
                0x1.b5790e22815b2p+648,
                0x1.b5864ca6b0b2bp722,
                0x1.b588bf702f5f4p862,
                0x1.b58ada67d48d2p289,
                0x1.b58caeb7793fp211,
                0x1.b591611a4528ep825,
                0x1.b591611a4528ep826,
                0x1.b591611a4528ep827,
                0x1.b599c65315262p-103,
                0x1.b599c65315262p-104,
                0x1.b599c65315262p-105,
                0x1.b59a56c2fc99p+408,
                0x1.b59a56c2fc99p409,
                0x1.b5a01c20b200dp862,
                0x1.b5a4fa8bfa66ap-283,
                0x1.b5a863b965083p-925,
                0x1.b5abf7102ae9p-183,
                0x1.b5abf7102ae9p-184,
                0x1.b5b60678cb7ecp443,
                0x1.b5b714d8fccdcp+300,
                0x1.b5b778d134a26p859,
                0x1.b5c0ec0d2c126p469,
                0x1.b5c0ec0d2c126p470,
                0x1.b5c0ec0d2c126p471,
                0x1.b5c0ec0d2c126p+472,
                0x1.b5cd8bbdd877p-390,
                0x1.b5e7e08ca3a8fp162,
                0x1.b5e7e08ca3a9p162,
                0x1.b5e7e08ca3a9p163,
                0x1.b5f099d1db66cp978,
                0x1.b5f507b493a96p411,
                0x1.b5f56250bbeeep222,
                0x1.b5f56250bbeeep223,
                0x1.b5f56250bbeeep+224,
                0x1.b5f9129d7f352p686,
                0x1.b5f9129d7f352p687,
                0x1.b5f9129d7f352p+688,
                0x1.b5f9129d7f352p689,
                0x1.b5fa3def48147p-300,
                0x1.b5fd8ee2bc871p865,
                0x1.b5fe80ebc926ap-607,
                0x1.b5fe80ebc926ap-608,
                0x1.b5fe80ebc926ap-609,
                0x1.b5ff848d4e246p773,
                0x1.b5ff848d4e246p774,
                0x1.b5ff848d4e246p775,
                0x1.b5ff848d4e246p+776,
                0x1.b60372eec480cp579,
                0x1.b6046d578184cp1007,
                0x1.b6046d578184cp+1008,
                0x1.b620c2b5bb372p-954,
                0x1.b620c2b5bb372p-955,
                0x1.b622366a7c386p210,
                0x1.b62b6d75c1316p709,
                0x1.b62d71d1201d6p523,
                0x1.b62d71d1201d6p+524,
                0x1.b62dbe2f93212p-342,
                0x1.b6300d36aadb8p-1011,
                0x1.b6300d36aadb8p-1012,
                0x1.b63d87dbec0b6p509,
                0x1.b63d87dbec0b6p510,
                0x1.b63d87dbec0b6p511,
                0x1.b63d87dbec0b6p+512,
                0x1.b63d87dbec0b6p513,
                0x1.b641b0b522db4p-743,
                0x1.b648f7286036ep215,
                0x1.b648f7286036ep+216,
                0x1.b65ca37fd3a0dp647,
                0x1.b6725e5549aeep858,
                0x1.b67b91de5d248p+744,
                0x1.b67b91de5d248p745,
                0x1.b67b91de5d248p746,
                0x1.b67b91de5d248p747,
                0x1.b67b91de5d248p+748,
                0x1.b67d7f4493dc4p+908,
                0x1.b6806df2a6d42p449,
                0x1.b68077b76cf04p-558,
                0x1.b68880dc3a85fp-819,
                0x1.b68e30f38bf4cp+272,
                0x1.b6940c5b46d84p+800,
                0x1.b69e2c321bfe8p-859,
                0x1.b6a78355314c7p318,
                0x1.b6a78355314c7p319,
                0x1.b6ac6322b008ap-86,
                0x1.b6b03ec710bf4p294,
                0x1.b6b46733f2004p+876,
                0x1.b6c80d95b3ed4p-977,
                0x1.b6c80d95b3ed4p-978,
                0x1.b6c80d95b3ed4p-979,
                0x1.b6c80d95b3ed4p-980,
                0x1.b6c80d95b3ed4p-981,
                0x1.b6cd4p-1056,
                0x1.b6cece299460ep-668,
                0x1.b6cece299460ep-669,
                0x1.b6d6a7c5182b6p+564,
                0x1.b6e20f7d07bbcp867,
                0x1.b6e3d22865634p-1004,
                0x1.b7017c4142ee6p-288,
                0x1.b7017c4142ee6p-289,
                0x1.b705f52ae6422p+916,
                0x1.b705f52ae6422p917,
                0x1.b705f52ae6422p918,
                0x1.b705f52ae6422p919,
                0x1.b705f52ae6422p+920,
                0x1.b70f0cc74f864p-504,
                0x1.b70f0cc74f864p-505,
                0x1.b71241dcc155ap-836,
                0x1.b71257f2b2e6dp+728,
                0x1.b72be61c11c6p189,
                0x1.b72c729c9d26cp311,
                0x1.b72d6e36b9cbep-349,
                0x1.b72d6e36b9cbep-350,
                0x1.b72d6e36b9cbep-351,
                0x1.b72d6e36b9cbep-352,
                0x1.b72d6e36b9cbep-353,
                0x1.b731b1fd32212p-1019,
                0x1.b736326e879fap-540,
                0x1.b73af29bb29ecp310,
                0x1.b7457ae60bcb8p-282,
                0x1.b746cf242410cp-105,
                0x1.b746cf242410cp-106,
                0x1.b746cf242410cp-107,
                0x1.b746cf242410cp-108,
                0x1.b746cf242410cp-109,
                0x1.b746cf242410cp-110,
                0x1.b746cf242410cp-111,
                0x1.b746cf242410cp-112,
                0x1.b74dcb500a158p-346,
                0x1.b75788bea5632p251,
                0x1.b758d848fac54p-519,
                0x1.b75d42d3a8e9p818,
                0x1.b764f577dd3d1p161,
                0x1.b764f577dd3d1p162,
                0x1.b766ac9e286ep+724,
                0x1.b7709034e6c42p93,
                0x1.b7738011e75fep-50,
                0x1.b7738011e75fep-51,
                0x1.b7738011e75fep-52,
                0x1.b7738011e75fep-53,
                0x1.b7738011e75fep-54,
                0x1.b77418a411de2p-59,
                0x1.b783836294434p254,
                0x1.b78761da54b1ep-269,
                0x1.b79193573a187p377,
                0x1.b792ee392d8ecp-705,
                0x1.b7a1274351d8cp-119,
                0x1.b7aa3d73f6658p-1007,
                0x1.b7aa3d73f6658p-1008,
                0x1.b7aa3d73f6658p-1009,
                0x1.b7aa3d73f6658p-1010,
                0x1.b7aa3d73f6658p-1011,
                0x1.b7aa3d73f6658p-1012,
                0x1.b7aa3d73f6658p-1013,
                0x1.b7ab7aaee960ap-693,
                0x1.b7abd9258492cp463,
                0x1.b7abd9258492cp+464,
                0x1.b7acbe31afddcp-660,
                0x1.b7ae8e09b834p-634,
                0x1.b7b33723f683ap529,
                0x1.b7cc78fa80cbp415,
                0x1.b7cd4b6817326p+212,
                0x1.b7cdfd9d7bdbap-34,
                0x1.b7db22884a5b6p567,
                0x1.b7db8bbbb861ep-362,
                0x1.b7de3f2c509c2p+804,
                0x1.b7f0b503844fbp+804,
                0x1.b7f76adce6a38p363,
                0x1.b7f76adce6a38p+364,
                0x1.b7f76adce6a38p365,
                0x1.b7f96fa2f14d2p-358,
                0x1.b8119d5e9dcfp806,
                0x1.b8119d5e9dcfp807,
                0x1.b813dc1bcecacp327,
                0x1.b81c1801dc7cp758,
                0x1.b81c1801dc7cp759,
                0x1.b81eafa785e61p-76,
                0x1.b81eafa785e61p-77,
                0x1.b8376f4b0b13cp981,
                0x1.b83a13bb3547bp-42,
                0x1.b83c5e2516458p231,
                0x1.b83fa373c2d1ep463,
                0x1.b843422e3a84cp450,
                0x1.b843422e3a84cp451,
                0x1.b843422e3a84cp+452,
                0x1.b843422e3a84cp453,
                0x1.b848a3ee9807ep-122,
                0x1.b848a3ee9807ep-123,
                0x1.b8501a7ce024cp786,
                0x1.b850e2ef56fc8p-576,
                0x1.b850e2ef56fc8p-577,
                0x1.b870de5d9327p-153,
                0x1.b877a957ac118p922,
                0x1.b8784611f2a73p-913,
                0x1.b8796d6546711p175,
                0x1.b87a66b3d720ap893,
                0x1.b87a66b3d720ap894,
                0x1.b8852b0350364p-129,
                0x1.b8852b0350364p-130,
                0x1.b888aae5ed712p-216,
                0x1.b88bd24adf7d3p-723,
                0x1.b88c0d63067ffp-526,
                0x1.b88e26508p-1041,
                0x1.b88ea9bf11bbap-563,
                0x1.b89916d942f31p675,
                0x1.b89c0f2b4a40ep+996,
                0x1.b89c884261a34p-751,
                0x1.b89cf2ba5ae5ep151,
                0x1.b89d67c5b6d25p+596,
                0x1.b89d67c5b6d25p597,
                0x1.b89d67c5b6d25p598,
                0x1.b8a5b63bb1c78p+352,
                0x1.b8a9a40499258p741,
                0x1.b8af4ae944d6p741,
                0x1.b8b8a6038ad6ep+936,
                0x1.b8b9873ba9dc2p205,
                0x1.b8bb6aa14b67p-682,
                0x1.b8be32187c592p-480,
                0x1.b8c0b9e4a2e36p+148,
                0x1.b8c22f001c496p-48,
                0x1.b8c22f001c496p-49,
                0x1.b8c22f001c496p-50,
                0x1.b8c22f001c496p-51,
                0x1.b8c5f33b8d3a4p-260,
                0x1.b8c5f33b8d3a4p-261,
                0x1.b8c5f33b8d3a4p-262,
                0x1.b8c5f33b8d3a4p-263,
                0x1.b8c5f33b8d3a4p-264,
                0x1.b8d68f6c032c6p-698,
                0x1.b8d68f6c032c6p-699,
                0x1.b8d68f6c032c6p-700,
                0x1.b8d68f6c032c6p-701,
                0x1.b8d7e32be6396p-860,
                0x1.b8d7e32be6396p-861,
                0x1.b8d7e32be6396p-862,
                0x1.b8d7e32be6396p-863,
                0x1.b8d7e32be6396p-864,
                0x1.b8d7e32be6396p-865,
                0x1.b8d7e32be6396p-866,
                0x1.b8d7e32be6396p-867,
                0x1.b8d96c408abbp753,
                0x1.b8dp-1061,
                0x1.b8e04156df814p942,
                0x1.b8ea77be494d7p561,
                0x1.b8f75020ad58dp750,
                0x1.b8p-1069,
                0x1.b9106ec657e36p-277,
                0x1.b9106ec657e36p-278,
                0x1.b9106ec657e36p-279,
                0x1.b9106ec657e36p-280,
                0x1.b9106ec657e36p-281,
                0x1.b9146a29dfbfdp-529,
                0x1.b917541848496p438,
                0x1.b9182c3f0adcep823,
                0x1.b91d317c8ebe8p-723,
                0x1.b92afecd1710cp-240,
                0x1.b92afecd1710cp-241,
                0x1.b92c5ccd59338p-927,
                0x1.b92c5ccd59338p-928,
                0x1.b92c5ccd59338p-929,
                0x1.b92c5ccd59338p-930,
                0x1.b92c5ccd59338p-931,
                0x1.b93d96bdbbfcbp-379,
                0x1.b9408eefea838p-715,
                0x1.b9551f023b08p241,
                0x1.b96e18b9346dbp495,
                0x1.b96e18b9346dbp+496,
                0x1.b979c828dc5dbp-869,
                0x1.b97a8f2b552f8p-445,
                0x1.b993afed217d2p-392,
                0x1.b993afed217d2p-393,
                0x1.b993afed217d2p-394,
                0x1.b993afed217d2p-395,
                0x1.b993afed217d2p-396,
                0x1.b99d6240c1a28p-353,
                0x1.b99d6240c1a28p-354,
                0x1.b9a0e01416ddcp653,
                0x1.b9a3448ecp-1038,
                0x1.b9a53a899a67p-387,
                0x1.b9a742ebda418p-961,
                0x1.b9a9fcaa90f3cp857,
                0x1.b9b26efc803b6p-86,
                0x1.b9b6364f30304p-230,
                0x1.b9d2669e61352p-643,
                0x1.b9d317fcff53ap-92,
                0x1.b9dcede9b9bd2p-364,
                0x1.b9ded91ef8fdcp+980,
                0x1.b9ded91ef8fdcp981,
                0x1.b9e1e4ae88243p-364,
                0x1.b9e85b56edd7fp-349,
                0x1.b9ea11d82d1bfp766,
                0x1.ba061afdf92bp-820,
                0x1.ba10d818fdafep814,
                0x1.ba10d818fdafep815,
                0x1.ba10d818fdafep+816,
                0x1.ba10d818fdafep817,
                0x1.ba10d818fdafep818,
                0x1.ba10d818fdafep819,
                0x1.ba11735607191p317,
                0x1.ba1be182d8318p-653,
                0x1.ba1be182d8318p-654,
                0x1.ba1be182d8318p-655,
                0x1.ba1be182d8318p-656,
                0x1.ba1be182d8318p-657,
                0x1.ba200f7000e85p-170,
                0x1.ba23f261bea8bp-629,
                0x1.ba2bfd0d5ff5bp255,
                0x1.ba305660bb937p626,
                0x1.ba35d5abf8c22p-387,
                0x1.ba35d5abf8c22p-388,
                0x1.ba35d5abf8c22p-389,
                0x1.ba3840967f964p935,
                0x1.ba3a4794cdf06p+948,
                0x1.ba43ef00dbe64p831,
                0x1.ba43ef00dbe64p+832,
                0x1.ba43ef00dbe64p833,
                0x1.ba4deb990c5e7p-476,
                0x1.ba5622ce04f28p-335,
                0x1.ba6008389068ap-670,
                0x1.ba6040dc1ef26p445,
                0x1.ba6381983d8fdp-422,
                0x1.ba67e81b9f0e4p+352,
                0x1.ba67e81b9f0e4p353,
                0x1.ba6add3bed55fp-504,
                0x1.ba6cd5df5a395p609,
                0x1.ba730d61aba2dp-747,
                0x1.ba75a3aace5e4p650,
                0x1.ba80c48f7ff7p-362,
                0x1.ba80c48f7ff7p-363,
                0x1.ba80c48f7ff7p-364,
                0x1.ba80c48f7ff7p-365,
                0x1.ba89467f781acp-941,
                0x1.ba94f7308ab9fp-281,
                0x1.ba96aa8c3486ep938,
                0x1.ba96aa8c3486ep939,
                0x1.ba96aa8c3486ep+940,
                0x1.baa1e332d728ep+740,
                0x1.baa1e332d728ep741,
                0x1.baaa0653ff3e9p806,
                0x1.bab8cbabb6582p-59,
                0x1.bac6f7d64d119p315,
                0x1.bac6f7d64d119p+316,
                0x1.bac71ac02dfcep634,
                0x1.bad3ec3b55b4ap365,
                0x1.bad437e573b5cp+780,
                0x1.badf0e1cac368p-964,
                0x1.bae07e90253ep-561,
                0x1.bae07e90253ep-562,
                0x1.bae07e90253ep-563,
                0x1.bae07e90253ep-564,
                0x1.bae07e90253ep-565,
                0x1.baf5bfaa86059p+472,
                0x1.baf88b4323bbfp-330,
                0x1.baf91ac23563p-714,
                0x1.baf91ac23563p-715,
                0x1.bafa1a71cae4ep+636,
                0x1.bb033a44739c5p-117,
                0x1.bb033a44739c5p-118,
                0x1.bb0d7c68f3756p-379,
                0x1.bb0d7c68f3756p-380,
                0x1.bb17c3398a01fp-535,
                0x1.bb1871614ab46p887,
                0x1.bb1871614ab46p+888,
                0x1.bb2a62fe638ffp-911,
                0x1.bb2a62fe639p-911,
                0x1.bb2ad233d59b2p+296,
                0x1.bb2ed75c82bcap426,
                0x1.bb30c918e52bp-649,
                0x1.bb30c918e52bp-650,
                0x1.bb4c6bd8601bep-385,
                0x1.bb4c6bd8601bep-386,
                0x1.bb4c6bd8601bep-387,
                0x1.bb4c6bd8601bep-388,
                0x1.bb4c6bd8601bep-389,
                0x1.bb55a5484e9acp705,
                0x1.bb55a5484e9acp706,
                0x1.bb66b18p-1046,
                0x1.bb6972029ba64p-263,
                0x1.bb6979ae39c48p167,
                0x1.bb6eb9e30a49ep-996,
                0x1.bb6eb9e30a49ep-997,
                0x1.bb6eb9e30a49ep-998,
                0x1.bb77e2b018336p-746,
                0x1.bb77e2b018336p-747,
                0x1.bb77e2b018336p-748,
                0x1.bb7c57f03765dp-239,
                0x1.bb7c57f03765dp-240,
                0x1.bb9f1d016be08p+708,
                0x1.bb9f1d016be08p709,
                0x1.bb9f1d016be08p710,
                0x1.bba08cf8c979cp-425,
                0x1.bba08cf8c979cp-426,
                0x1.bbaa1e9e42af4p-230,
                0x1.bbb8800bb9c05p186,
                0x1.bbbd1aedbf9cp+300,
                0x1.bbbd1aedbf9cp301,
                0x1.bbbf0d455e75ep125,
                0x1.bbc7657070a04p701,
                0x1.bbced4f69402ep+320,
                0x1.bbdb9281f636p-1018,
                0x1.bbdda0bb4fd5ep-117,
                0x1.bbdda8055264p+552,
                0x1.bbdf405a79202p269,
                0x1.bbefd01381b5p-677,
                0x1.bbefd01381b5p-678,
                0x1.bbf543bea3763p-600,
                0x1.bbf543bea3763p-601,
                0x1.bbfcea34e0d76p-911,
                0x1.bbfcea34e0d76p-912,
                0x1.bbfcea34e0d76p-913,
                0x1.bc00b9c1ea51p662,
                0x1.bc053791b0e34p-160,
                0x1.bc053791b0e34p-161,
                0x1.bc07b922032a7p771,
                0x1.bc16d674ec8p59,
                0x1.bc1f396b6df93p-765,
                0x1.bc2b8765feff6p350,
                0x1.bc2b8765feff6p351,
                0x1.bc324934c012cp597,
                0x1.bc329308b1aeep202,
                0x1.bc4909a5f7bd5p-316,
                0x1.bc4b180b07445p-623,
                0x1.bc4b180b07445p-624,
                0x1.bc52ec15cebbp+168,
                0x1.bc581a10b5b18p946,
                0x1.bc59f7dcdddcap982,
                0x1.bc5f32e89ce64p777,
                0x1.bc681146ea6c2p950,
                0x1.bc6ec488255aep-414,
                0x1.bc7dc964446c6p-923,
                0x1.bc8262dd0eb3p-679,
                0x1.bc8262dd0eb3p-680,
                0x1.bc8d3f7b3340cp542,
                0x1.bc8d3f7b3340cp543,
                0x1.bc8d3f7b3340cp+544,
                0x1.bc9445fb5947dp-781,
                0x1.bc948d999ac12p-15,
                0x1.bca441622594ap-255,
                0x1.bcb7d682937f3p+648,
                0x1.bcbfa78f71b96p+236,
                0x1.bcc0e9b305e6cp-783,
                0x1.bcc205acffe85p413,
                0x1.bcc56f2ed5896p-742,
                0x1.bcc60266a98e1p-818,
                0x1.bcc60266a98e1p-819,
                0x1.bccc8adf13591p-574,
                0x1.bcdbb0403432ap-215,
                0x1.bcdbb0403432ap-216,
                0x1.bcdbb0403432ap-217,
                0x1.bcdbb0403432ap-218,
                0x1.bcdbb0403432ap-219,
                0x1.bcdbb0403432ap-220,
                0x1.bcea0ec21e25p-38,
                0x1.bcfc62cb120d9p+556,
                0x1.bd0563bp-1045,
                0x1.bd11b99472d9ap-455,
                0x1.bd16055178506p549,
                0x1.bd16055178506p550,
                0x1.bd16055178506p551,
                0x1.bd16055178506p+552,
                0x1.bd16055178506p553,
                0x1.bd1ad01d6ef16p230,
                0x1.bd2ab5f28e484p1013,
                0x1.bd2bcbab466a8p-525,
                0x1.bd302e4a188p-1029,
                0x1.bd4533f4a1584p-401,
                0x1.bd4533f4a1584p-402,
                0x1.bd4812c1a6a81p739,
                0x1.bd48d255a4282p-829,
                0x1.bd48d255a4282p-830,
                0x1.bd48d255a4282p-831,
                0x1.bd4c15a3287b8p-312,
                0x1.bd4c15a3287b8p-313,
                0x1.bd4c41a1b66c2p+580,
                0x1.bd5fb10ec50d4p-649,
                0x1.bd5fb10ec50d4p-650,
                0x1.bd64899d821e4p634,
                0x1.bd66d32aef254p-427,
                0x1.bd66d32aef254p-428,
                0x1.bd7293cea2966p146,
                0x1.bd7d32444e874p371,
                0x1.bd8429e4d6a14p-550,
                0x1.bd8429e4d6a14p-551,
                0x1.bd8a86ea90f08p821,
                0x1.bd8d03f3e9864p-622,
                0x1.bd93fe4a5ee9ap+500,
                0x1.bd93fe4a5ee9ap501,
                0x1.bd99b87cda1ccp-628,
                0x1.bda80a9978912p117,
                0x1.bdaf295acc0d8p+448,
                0x1.bdaf295acc0d8p449,
                0x1.bdb90e62a8cbdp-700,
                0x1.bdb90e62a8cbdp-701,
                0x1.bdb90e62a8cbdp-702,
                0x1.bdc2417bf7787p-837,
                0x1.bdc2417bf7787p-838,
                0x1.bdc2417bf7787p-839,
                0x1.bdc4114397ff4p-373,
                0x1.bdc41968680cep454,
                0x1.bdcb7da4b4ac4p-435,
                0x1.bdcb7da4b4ac4p-436,
                0x1.bdd9a73eb06cap-777,
                0x1.bdd9a86323dd8p-33,
                0x1.bdd9a86323dd8p-34,
                0x1.bddca507e4384p-74,
                0x1.bddca507e4384p-75,
                0x1.bddca507e4384p-76,
                0x1.bddca507e4384p-77,
                0x1.bddca507e4384p-78,
                0x1.bde4737b09a86p-282,
                0x1.bde4737b09a86p-283,
                0x1.bde6e98847c0fp-343,
                0x1.bde959510fccap-668,
                0x1.bde959510fccap-669,
                0x1.bdf142478c707p-670,
                0x1.bdf142478c707p-671,
                0x1.bdfc6734e680ap-610,
                0x1.be01ef949bcd2p+448,
                0x1.be03d0bf225c6p-137,
                0x1.be03d0bf225c7p-137,
                0x1.be03d0bf225c7p-138,
                0x1.be051dc9a2b83p-412,
                0x1.be0ade587b92bp-539,
                0x1.be0bdbe5a3ee4p-388,
                0x1.be0eb55770d4ep109,
                0x1.be1d8351d8eeep-467,
                0x1.be20cdd9137ap-56,
                0x1.be249e325bf5ep905,
                0x1.be26a76ef2304p-885,
                0x1.be2dd66200bfp-841,
                0x1.be2dd66200bfp-842,
                0x1.be2dd66200bfp-843,
                0x1.be2dd66200bfp-844,
                0x1.be2dd66200bfp-845,
                0x1.be2dd66200bfp-846,
                0x1.be2dd66200bfp-847,
                0x1.be2dd66200bfp-848,
                0x1.be2dd66200bfp-849,
                0x1.be2dd66200bfp-850,
                0x1.be2dd66200bfp-851,
                0x1.be3546c6998d4p-198,
                0x1.be363baa1eb06p-512,
                0x1.be3671628f488p-178,
                0x1.be36ee58eebeap-680,
                0x1.be372ca2dd7e8p341,
                0x1.be37a26363785p842,
                0x1.be3e7ab6973dap-342,
                0x1.be3e7ab6973dap-343,
                0x1.be4242979ba6ep539,
                0x1.be45a312d08ap-64,
                0x1.be45a312d08ap-65,
                0x1.be45a312d08ap-66,
                0x1.be45a312d08ap-67,
                0x1.be45a312d08ap-68,
                0x1.be45a312d08ap-69,
                0x1.be5374ec8d32ep-529,
                0x1.be5374ec8d32ep-530,
                0x1.be5374ec8d32ep-531,
                0x1.be673571cd944p-1024,
                0x1.be7711f17fb58p-329,
                0x1.be7a647ef6185p-576,
                0x1.be7abd3781ecap+348,
                0x1.be7abd3781ecap349,
                0x1.be7abd3781ecap350,
                0x1.be7abd3781ecap351,
                0x1.be7abd3781ecap+352,
                0x1.be7cp-1060,
                0x1.be872a8b30d7dp-25,
                0x1.be872a8b30d7dp-26,
                0x1.be87c3f4de5a4p682,
                0x1.be87c3f4de5a4p683,
                0x1.be87c3f4de5a4p+684,
                0x1.be87c3f4de5a4p685,
                0x1.be87c3f4de5a4p686,
                0x1.be8826cda9b72p-392,
                0x1.be91df6c55138p-425,
                0x1.be954e087c31cp950,
                0x1.be9655c958fe5p886,
                0x1.be976eac54188p-754,
                0x1.bea341f05f1dp-632,
                0x1.bea341f05f1dp-633,
                0x1.bea341f05f1dp-634,
                0x1.bea341f05f1dp-635,
                0x1.bea7508dd75ffp786,
                0x1.beaaddb70ce9bp470,
                0x1.bead42fe90ef4p-702,
                0x1.bead81e66a128p387,
                0x1.bead81e66a128p+388,
                0x1.bead81e66a128p389,
                0x1.bead81e66a128p390,
                0x1.bead81e66a128p391,
                0x1.beb8f27e6c86cp-169,
                0x1.beb8f27e6c86cp-170,
                0x1.beb8f27e6c86cp-171,
                0x1.beb9361a737afp-867,
                0x1.beba911577c6ep-193,
                0x1.bebf1d56ef40ap1007,
                0x1.becabac98dfc5p295,
                0x1.becba69bf6d8p907,
                0x1.becc95edac96ap834,
                0x1.becc95edac96ap835,
                0x1.bee6ddb41c38cp-916,
                0x1.beeb7bce7fcb7p-858,
                0x1.bef1c9657a686p833,
                0x1.bef1c9657a686p834,
                0x1.bef4b2345db1p-943,
                0x1.bef4d6p-1051,
                0x1.bef9001047ac7p601,
                0x1.bf00baf60b70cp311,
                0x1.bf00baf60b70cp+312,
                0x1.bf0384159b534p+588,
                0x1.bf08d608b9357p-822,
                0x1.bf0fe43bbbfacp+148,
                0x1.bf1f92a29f1b7p-267,
                0x1.bf1f92a29f1b7p-268,
                0x1.bf2bebb27631ep-1019,
                0x1.bf2dec8e53e6fp-502,
                0x1.bf2dec8e53e6fp-503,
                0x1.bf2faf2db24cap926,
                0x1.bf51de62602b9p815,
                0x1.bf51de62602b9p+816,
                0x1.bf51de62602b9p817,
                0x1.bf5b8p-1057,
                0x1.bf7214e6d4e32p-867,
                0x1.bf7b9d9cce00dp-818,
                0x1.bf7b9d9cce00ep-817,
                0x1.bf7b9d9cce00ep-818,
                0x1.bf7d933c14p-1035,
                0x1.bf7d933c14p-1036,
                0x1.bf8769cf64bd8p-485,
                0x1.bf891c92c089p-24,
                0x1.bf891c92c089p-25,
                0x1.bf8d2b747eafep854,
                0x1.bf96113b955p-608,
                0x1.bfa1d38557ccap631,
                0x1.bfa6dda21d5bbp-455,
                0x1.bfa6dda21d5bbp-456,
                0x1.bfb560ca60436p990,
                0x1.bfb596428d3f6p+304,
                0x1.bfb596428d3f6p305,
                0x1.bfc00f95de19ep-549,
                0x1.bfc00f95de19ep-550,
                0x1.bfc0b32deaf9fp-372,
                0x1.bfc903e918e1bp890,
                0x1.bfce2a6fe42a5p+456,
                0x1.bfe3fcb9eebe2p509,
                0x1.bfe3fcb9eebe2p510,
                0x1.bfe596214ecdp-824,
                0x1.bfe596214ecdp-825,
                0x1.bff2b67112204p-590,
                0x1.bff2ee48e053p-333,
                0x1.bff78226c56a8p-572,
                0x1.bff78226c56a8p-573,
                0x1.bff8323dd7c6ap-986,
                0x1.bp-1070,
                0x1.c000b700a869p-137,
                0x1.c0010b55719fep707,
                0x1.c003555e6ba51p485,
                0x1.c004ca9a9ff0cp261,
                0x1.c02bf709fd855p618,
                0x1.c030c7p-1050,
                0x1.c0386744986p289,
                0x1.c0386744986p290,
                0x1.c0400c85ab80ep-762,
                0x1.c042ddcea5618p-545,
                0x1.c042ddcea5618p-546,
                0x1.c042ddcea5618p-547,
                0x1.c05238fbf0ca2p663,
                0x1.c06a5ec5433c6p151,
                0x1.c06a5ec5433c6p+152,
                0x1.c06d366394441p-35,
                0x1.c06d366394441p-36,
                0x1.c06d366394441p-37,
                0x1.c0794d9d40e96p-301,
                0x1.c0794d9d40e96p-302,
                0x1.c0794d9d40e96p-303,
                0x1.c0794d9d40e96p-304,
                0x1.c079d03900cb6p-481,
                0x1.c079d03900cb6p-482,
                0x1.c07e0c9e5bc85p-688,
                0x1.c0804ea8cdc74p866,
                0x1.c0804ea8cdc74p867,
                0x1.c0804ea8cdc74p+868,
                0x1.c088f2148b81bp-296,
                0x1.c08dd32636178p-657,
                0x1.c09ff3eedae8ap-63,
                0x1.c0a3809ab1e8ep-650,
                0x1.c0a3809ab1e8ep-651,
                0x1.c0a3809ab1e8ep-652,
                0x1.c0a91fff70e23p-946,
                0x1.c0ab8b519b556p-384,
                0x1.c0ad2abd90d58p-550,
                0x1.c0ae20b9p-1042,
                0x1.c0b23b82010a2p-17,
                0x1.c0b23b82010a2p-18,
                0x1.c0b81ed5bf19ap+728,
                0x1.c0bffd1b7714p-233,
                0x1.c0c028f0cc719p467,
                0x1.c0cb4bf2e7c0ap-390,
                0x1.c0cb4bf2e7c0ap-391,
                0x1.c0da795cf48d3p+164,
                0x1.c0e1ef1a724eap637,
                0x1.c0ecd3840913p-452,
                0x1.c0ecp-1059,
                0x1.c0f220fb1c70ep-535,
                0x1.c0f220fb1c70ep-536,
                0x1.c0f220fb1c70ep-537,
                0x1.c0f220fb1c70ep-538,
                0x1.c0f220fb1c70ep-539,
                0x1.c0f220fb1c70ep-540,
                0x1.c0f92a6276c9dp-162,
                0x1.c0f92a6276c9dp-163,
                0x1.c0f92a6276c9dp-164,
                0x1.c0fab3d11a7d3p746,
                0x1.c10df90521e62p469,
                0x1.c110e96414eecp835,
                0x1.c1145492dfdbcp-541,
                0x1.c1181b752088ep870,
                0x1.c1249e96b6d8cp-267,
                0x1.c1249e96b6d8cp-268,
                0x1.c1249e96b6d8cp-269,
                0x1.c1249e96b6d8cp-270,
                0x1.c1249e96b6d8cp-271,
                0x1.c1249e96b6d8cp-272,
                0x1.c1249e96b6d8cp-273,
                0x1.c1249e96b6d8cp-274,
                0x1.c12c430708482p+752,
                0x1.c1329253023f4p+148,
                0x1.c133efc58896ep-279,
                0x1.c133efc58896ep-280,
                0x1.c13927584fec8p-25,
                0x1.c149cd41cc8ap-886,
                0x1.c152eb0cc30dp447,
                0x1.c15be23a32603p342,
                0x1.c165d112856a6p862,
                0x1.c16680ab4efe3p-631,
                0x1.c16c5c5253575p-1014,
                0x1.c16c5c5253575p-1015,
                0x1.c16f524443a28p427,
                0x1.c170e0cab282cp323,
                0x1.c17102cb3p-1038,
                0x1.c172045e949c6p-452,
                0x1.c172045e949c6p-453,
                0x1.c172045e949c6p-454,
                0x1.c172045e949c6p-455,
                0x1.c172045e949c6p-456,
                0x1.c172045e949c6p-457,
                0x1.c172045e949c6p-458,
                0x1.c172045e949c6p-459,
                0x1.c172045e949c6p-460,
                0x1.c172045e949c6p-461,
                0x1.c172045e949c6p-462,
                0x1.c172045e949c6p-463,
                0x1.c172045e949c6p-464,
                0x1.c172045e949c6p-465,
                0x1.c18dee2e8b1cap-11,
                0x1.c18e9ece203f6p-214,
                0x1.c18fa5b20cc48p563,
                0x1.c19f76b29760ep-974,
                0x1.c1a5aa81ebbfcp-46,
                0x1.c1b6c9c6910f2p321,
                0x1.c1b9436271e74p-411,
                0x1.c1bf5689b0f24p-536,
                0x1.c1c352fc3c309p899,
                0x1.c1cea0eee8b88p-67,
                0x1.c1cea0eee8b88p-68,
                0x1.c1d2d9135225ep295,
                0x1.c1d2d9135225ep+296,
                0x1.c1e0001eee8f8p+848,
                0x1.c1e0001eee8f8p849,
                0x1.c1e0001eee8f8p850,
                0x1.c1e43171a4a11p-529,
                0x1.c1e8b71e5f23ep+808,
                0x1.c1efd3572b494p+396,
                0x1.c1f2bebc015bp-465,
                0x1.c212b611f6e9ap833,
                0x1.c21df9e17025cp570,
                0x1.c224b6002d55p+676,
                0x1.c22b8e17cb952p-362,
                0x1.c23403e643ec8p+1000,
                0x1.c23efa66403f6p+188,
                0x1.c2456860a856cp-650,
                0x1.c24d9ffded66ap-285,
                0x1.c24d9ffded66ap-286,
                0x1.c250b48ccccdcp-82,
                0x1.c25584881552ap-883,
                0x1.c25584881552ap-884,
                0x1.c25584881552ap-885,
                0x1.c25584881552ap-886,
                0x1.c25584881552ap-887,
                0x1.c25c268497682p-44,
                0x1.c260c6f9481f8p-600,
                0x1.c260c6f9481f8p-601,
                0x1.c2669c2ec709ep-415,
                0x1.c27977585875ap-501,
                0x1.c27977585875ap-502,
                0x1.c283ffc61c87ep+960,
                0x1.c283ffc61c87ep961,
                0x1.c283ffc61c87ep962,
                0x1.c283ffc61c87ep963,
                0x1.c28f0f2cac9fbp-419,
                0x1.c28f9813ead96p-248,
                0x1.c28f9813ead96p-249,
                0x1.c28f9813ead96p-250,
                0x1.c28f9813ead96p-251,
                0x1.c28f9813ead96p-252,
                0x1.c29bc6879dfccp-498,
                0x1.c29bc6879dfccp-499,
                0x1.c2a0f85fe4effp-712,
                0x1.c2a8597192551p-567,
                0x1.c2a8597192551p-568,
                0x1.c2ac959a83feep-379,
                0x1.c2aed74caca4ep-214,
                0x1.c2c24a9af5dfap-379,
                0x1.c2c31a31998aep-355,
                0x1.c2c31a31998aep-356,
                0x1.c2c31a31998aep-357,
                0x1.c2c31a31998aep-358,
                0x1.c2c31a31998aep-359,
                0x1.c2c31a31998aep-360,
                0x1.c2cf507bfc903p-899,
                0x1.c2d43b93b0a8cp441,
                0x1.c2dc1e16b06fep+696,
                0x1.c2dc1e16b06fep697,
                0x1.c2dc1e16b06fep698,
                0x1.c2dd3f870c23fp-57,
                0x1.c2df8da5b6ca8p-151,
                0x1.c2df8da5b6ca8p-152,
                0x1.c2e162cdb252ap+776,
                0x1.c2e566fa2baf3p553,
                0x1.c2f20137d8211p+496,
                0x1.c2f8c73407bc6p-732,
                0x1.c2fa4dc5b0842p-786,
                0x1.c2fa4dc5b0842p-787,
                0x1.c305e0d8849bdp+312,
                0x1.c306327abd37ap-916,
                0x1.c30b853d87fd6p-320,
                0x1.c30b853d87fd6p-321,
                0x1.c30d36622575cp-845,
                0x1.c311a7dbbdbe2p-707,
                0x1.c311a7dbbdbe2p-708,
                0x1.c311a7dbbdbe2p-709,
                0x1.c317b95b74f4p-140,
                0x1.c3190f5dc02e5p+340,
                0x1.c33b33f5f95d4p331,
                0x1.c34413df18ae8p773,
                0x1.c34413df18ae8p774,
                0x1.c34c654cf01a8p-380,
                0x1.c34c70a777a4cp925,
                0x1.c34c70a777a4cp926,
                0x1.c34c70a777a4cp927,
                0x1.c34c70a777a4cp+928,
                0x1.c3553cda7d46ep833,
                0x1.c35aa64b6780bp-366,
                0x1.c35aa64b6780bp-367,
                0x1.c3613bce5e1b3p-885,
                0x1.c367e99a5402p934,
                0x1.c370f75c67012p462,
                0x1.c3725e54b32a8p986,
                0x1.c3725e54b32a8p987,
                0x1.c3725e54b32a8p+988,
                0x1.c3725e54b32a8p989,
                0x1.c3880ea682a29p+676,
                0x1.c395aac524fc2p466,
                0x1.c39984b56777ep661,
                0x1.c39a50f0838fdp225,
                0x1.c39a50f0838fdp226,
                0x1.c3a8e5c89dca6p-468,
                0x1.c3a9a0db45e63p-150,
                0x1.c3aa5e9d52c1cp-581,
                0x1.c3aa5e9d52c1cp-582,
                0x1.c3aa5e9d52c1cp-583,
                0x1.c3aad3b8fb68ep282,
                0x1.c3af8ec4d292cp418,
                0x1.c3af8ec4d292cp419,
                0x1.c3b2cb3d2688p591,
                0x1.c3bc0e6868633p-902,
                0x1.c3c26e087a982p+532,
                0x1.c3ccbc97a5316p171,
                0x1.c3d435770cd12p-166,
                0x1.c3d79c9b8fe2ep-725,
                0x1.c3da7babdf512p791,
                0x1.c3da7babdf512p+792,
                0x1.c3da7babdf512p793,
                0x1.c3da7babdf512p794,
                0x1.c3e4e940fde47p-511,
                0x1.c3ed68a616efcp-838,
                0x1.c3efdbc97a8b2p949,
                0x1.c3f3665eab266p303,
                0x1.c3fe74995d448p919,
                0x1.c3fe8cb0b2e0ep-754,
                0x1.c3fe8cb0b2e0ep-755,
                0x1.c3fe8cb0b2e0ep-756,
                0x1.c3fe8cb0b2e0ep-757,
                0x1.c3fe8cb0b2e0ep-758,
                0x1.c3fe8cb0b2e0ep-759,
                0x1.c402fac8d9242p-374,
                0x1.c40afd65c3e64p-685,
                0x1.c40c762289248p-402,
                0x1.c41c675503295p635,
                0x1.c41c675503295p+636,
                0x1.c42bdf1b95caep-113,
                0x1.c42bf3e0220a6p+308,
                0x1.c4324c6e4a562p743,
                0x1.c4324c6e4a562p+744,
                0x1.c4324c6e4a562p745,
                0x1.c4324c6e4a562p746,
                0x1.c4324c6e4a562p747,
                0x1.c43366691792fp+152,
                0x1.c4339cp-1049,
                0x1.c4339cp-1050,
                0x1.c43845cb17aa6p166,
                0x1.c43845cb17aa6p167,
                0x1.c43cc45f364e6p-587,
                0x1.c43cc45f364e6p-588,
                0x1.c44101861b1aap-899,
                0x1.c44101861b1aap-900,
                0x1.c449d7cf8eab2p-517,
                0x1.c44ad444fcac1p807,
                0x1.c45016d841baap-240,
                0x1.c4677f93p-1039,
                0x1.c467dc709a1a2p-603,
                0x1.c47a593856f98p-1025,
                0x1.c47b1da9a80b6p-254,
                0x1.c47d20a19d1eep-294,
                0x1.c47d20a19d1eep-295,
                0x1.c48p-1056,
                0x1.c4901cdeddb78p-803,
                0x1.c4901cdeddb78p-804,
                0x1.c49437fccfadbp647,
                0x1.c49437fccfadbp+648,
                0x1.c49437fccfadbp649,
                0x1.c4980a4ee94cfp131,
                0x1.c49d91f8ad18cp-241,
                0x1.c4a4ec26f3c5fp-550,
                0x1.c4b200733a076p-908,
                0x1.c4b628a4fd8a6p-93,
                0x1.c4b628a4fd8a6p-94,
                0x1.c4b965f940e7bp+412,
                0x1.c4bac5e0fe3dep-620,
                0x1.c4bac5e0fe3dep-621,
                0x1.c4bac5e0fe3dep-622,
                0x1.c4bac5e0fe3dep-623,
                0x1.c4c8b1349b9b5p245,
                0x1.c4c8b1349b9b6p245,
                0x1.c4c8b1349b9b6p246,
                0x1.c4d6ab2d601c8p-922,
                0x1.c4e0ec80535eap-659,
                0x1.c4e142cef2472p-333,
                0x1.c4e8b226e579ep-778,
                0x1.c4e8b226e579ep-779,
                0x1.c4eb7b44bac35p870,
                0x1.c4ep-1063,
                0x1.c502d6fee1b48p+792,
                0x1.c50388f469b07p479,
                0x1.c504dfe7ee02cp+1016,
                0x1.c5096bcc9ef7ap-296,
                0x1.c50d03a881417p339,
                0x1.c51174dbc41bap-309,
                0x1.c52cc5e2518ccp386,
                0x1.c52d48961bd57p-1005,
                0x1.c5416bb92e3e6p729,
                0x1.c5416bb92e3e6p730,
                0x1.c544p-1060,
                0x1.c547d8c35b5aep-937,
                0x1.c547d8c35b5aep-938,
                0x1.c547d8c35b5aep-939,
                0x1.c561348184df4p835,
                0x1.c56359a56bf06p595,
                0x1.c569e968e0944p+424,
                0x1.c569e968e0944p425,
                0x1.c569e968e0944p426,
                0x1.c569e968e0944p427,
                0x1.c569e968e0944p+428,
                0x1.c569e968e0944p429,
                0x1.c56ccb95584c2p-794,
                0x1.c56ccb95584c2p-795,
                0x1.c57057605a843p-868,
                0x1.c58937ec95327p-649,
                0x1.c58a00bb31864p339,
                0x1.c58a00bb31864p+340,
                0x1.c58a00bb31864p341,
                0x1.c58a00bb31864p342,
                0x1.c58a00bb31864p343,
                0x1.c58ebfc2d9226p-248,
                0x1.c59b27bc2decep657,
                0x1.c5a5523b5f08ap733,
                0x1.c5aec5a0e8f2ap897,
                0x1.c5b2bc141c89fp742,
                0x1.c5b8d6b29f57dp-362,
                0x1.c5cd322b67fffp-921,
                0x1.c5ce2f406b566p-837,
                0x1.c5cfd5c3b348dp-156,
                0x1.c5cfd5c3b348dp-157,
                0x1.c5d1c1e7f26b2p+416,
                0x1.c5e1c16c19d25p958,
                0x1.c5e3f495e23f7p462,
                0x1.c5e40a79c0a71p-48,
                0x1.c5e98cccbp-1038,
                0x1.c5f56d54400e8p-711,
                0x1.c5f56d54400e8p-712,
                0x1.c5f56d54400e8p-713,
                0x1.c5f56d54400e8p-714,
                0x1.c5f56d54400e8p-715,
                0x1.c6033e1c45153p+964,
                0x1.c60e02ba95981p361,
                0x1.c60f15972b1dep-687,
                0x1.c61323f6b4a62p-719,
                0x1.c61323f6b4a62p-720,
                0x1.c6262p-1053,
                0x1.c63345295a1b8p-765,
                0x1.c6463225ab7ecp-436,
                0x1.c6472cff68f9ap130,
                0x1.c64911e0fc084p363,
                0x1.c64911e0fc084p+364,
                0x1.c64a85dea3fep981,
                0x1.c64ac31992476p901,
                0x1.c66f3c419d17dp819,
                0x1.c66f5ea0149ccp418,
                0x1.c672854e82fap-650,
                0x1.c675035c7ed8ep+824,
                0x1.c688f6fadaee6p-878,
                0x1.c690f154fcf11p-211,
                0x1.c690f154fcf11p-212,
                0x1.c69978a8144bcp-730,
                0x1.c69a4c8accd8fp510,
                0x1.c6ac68515804ap285,
                0x1.c6b088c1a727fp-351,
                0x1.c6b821b021228p415,
                0x1.c6b8f1dd0ee8fp486,
                0x1.c6bb5ecc4a5c4p349,
                0x1.c6bb5ecc4a5c4p350,
                0x1.c6bb5ecc4a5c4p351,
                0x1.c6bb5ecc4a5c4p+352,
                0x1.c6bdbed2f26c9p666,
                0x1.c6bf52634p49,
                0x1.c6d1dfb4ce25ap893,
                0x1.c6d4d14db718cp354,
                0x1.c6d4d14db718cp355,
                0x1.c6d4d14db718cp+356,
                0x1.c6d4d14db718cp357,
                0x1.c6e27d4fc7a93p+876,
                0x1.c6e2b2558d0ecp-650,
                0x1.c6e86787c122p-1027,
                0x1.c6f307be4c468p89,
                0x1.c6f307be4c468p90,
                0x1.c6f307be4c468p91,
                0x1.c6f91114f948ap-224,
                0x1.c6f91114f948ap-225,
                0x1.c6f91114f948ap-226,
                0x1.c6f91114f948ap-227,
                0x1.c707c8e3e5c5dp-750,
                0x1.c70c9ae463982p+288,
                0x1.c7124p-1054,
                0x1.c7170219f86bep-182,
                0x1.c72117430fd7cp603,
                0x1.c7293c73a605ep499,
                0x1.c72b1fb28ba04p-653,
                0x1.c73892ecbfbf4p534,
                0x1.c73d221d322dp+648,
                0x1.c73d221d322dp649,
                0x1.c753595abfbf1p826,
                0x1.c7601883dddedp-173,
                0x1.c7601883dddedp-174,
                0x1.c7615c155ac32p811,
                0x1.c76422ef2a332p-990,
                0x1.c76422ef2a332p-991,
                0x1.c77001829ab95p-708,
                0x1.c786444689386p-529,
                0x1.c78870e421a9ap554,
                0x1.c793d4a611912p-145,
                0x1.c794844ff7516p335,
                0x1.c794844ff7516p+336,
                0x1.c794844ff7516p337,
                0x1.c794844ff7516p338,
                0x1.c794844ff7516p339,
                0x1.c79ccac2f461dp+664,
                0x1.c79ef60beacaap-639,
                0x1.c7a268003e9e3p-298,
                0x1.c7a8dc906b10ep993,
                0x1.c7a8dc906b10ep994,
                0x1.c7b1f3cac7433p1019,
                0x1.c7b699a39c67cp983,
                0x1.c7b699a39c67cp+984,
                0x1.c7b699a39c67cp985,
                0x1.c7b699a39c67cp986,
                0x1.c7b699a39c67cp987,
                0x1.c7bfa853ff992p+628,
                0x1.c7c5441d47e9fp-1012,
                0x1.c7c5441d47e9fp-1013,
                0x1.c7c5aea080a4ap+248,
                0x1.c7c5aea080a4ap249,
                0x1.c7c5aea080a4ap250,
                0x1.c7c5aea080a4ap251,
                0x1.c7c5aea080a4ap+252,
                0x1.c7c61447148a7p385,
                0x1.c7c841c72a018p-310,
                0x1.c7c989556f8ebp+220,
                0x1.c7cb7e08796d2p598,
                0x1.c7ccf90c9f8abp155,
                0x1.c7ccf90c9f8abp+156,
                0x1.c7ccf90c9f8abp157,
                0x1.c7d97579eba87p+224,
                0x1.c7e328e7f0162p-981,
                0x1.c7e41c1c3b75p989,
                0x1.c7efa19aecdbbp+208,
                0x1.c7f10f030d0cdp-733,
                0x1.c7f7ac3f5e0acp-30,
                0x1.c7f7ac3f5e0acp-31,
                0x1.c7f7ac3f5e0acp-32,
                0x1.c7fab3baca474p+436,
                0x1.c7fab3baca474p437,
                0x1.c808363f97e95p-107,
                0x1.c80be53f3ac58p+680,
                0x1.c8111764be794p-880,
                0x1.c8140656bf43cp-374,
                0x1.c8140656bf43cp-375,
                0x1.c8193da5ff4b6p650,
                0x1.c82ad1e31228ep606,
                0x1.c82da8cf0cceap689,
                0x1.c836caa446b89p-731,
                0x1.c83e7ad4e6efep-632,
                0x1.c83eb5fb3cf4cp-411,
                0x1.c84c524ab5ebp277,
                0x1.c851bc531e5p903,
                0x1.c8586f0912f1dp734,
                0x1.c8586f0912f1dp735,
                0x1.c8690f7aa2faap-650,
                0x1.c8690f7aa2faap-651,
                0x1.c86d34025463dp474,
                0x1.c8788847adf4dp-1021,
                0x1.c8788847adf4dp-1022,
                0x1.c879848e0f2b5p549,
                0x1.c881d6280782p-677,
                0x1.c88bb27d20e0ap+308,
                0x1.c88bb27d20e0ap309,
                0x1.c8900916ce672p-215,
                0x1.c8900916ce672p-216,
                0x1.c89bd02dca062p-880,
                0x1.c89bd02dca062p-881,
                0x1.c89bd02dca062p-882,
                0x1.c89bd02dca062p-883,
                0x1.c89bd02dca062p-884,
                0x1.c89d077e03e4p-679,
                0x1.c89fad4239084p-557,
                0x1.c89fad4239084p-558,
                0x1.c89fad4239084p-559,
                0x1.c89fad4239084p-560,
                0x1.c89fad4239084p-561,
                0x1.c8a42ae0a8e68p-301,
                0x1.c8a4e5f8e4a42p-777,
                0x1.c8b016999b9f7p-40,
                0x1.c8b1bb121cae4p-844,
                0x1.c8b8218854567p-147,
                0x1.c8b8218854568p-146,
                0x1.c8b8218854568p-147,
                0x1.c8b9c690d825p-691,
                0x1.c8bc99e9f433ap+576,
                0x1.c8c574c0c6be7p+188,
                0x1.c8c7572ae98b4p-600,
                0x1.c8cfc4a49d02dp-946,
                0x1.c8d795f8c09c2p899,
                0x1.c8dbc25253555p-424,
                0x1.c8dbc25253555p-425,
                0x1.c8de659f56243p-125,
                0x1.c8e7edc737c2ap-734,
                0x1.c8ea841a4e2f6p-532,
                0x1.c8f098765c56ep929,
                0x1.c8f769e14e77ep-584,
                0x1.c8ffc6d126775p965,
                0x1.c90094f9f6844p254,
                0x1.c90094f9f6844p255,
                0x1.c90094f9f6844p+256,
                0x1.c900afede5f4cp214,
                0x1.c900afede5f4cp215,
                0x1.c903ed769ce94p+760,
                0x1.c9083e7ebf8c5p129,
                0x1.c9083e7ebf8c5p130,
                0x1.c9099c1ab337ep-386,
                0x1.c90fb315cfd44p-767,
                0x1.c912bb7976094p385,
                0x1.c912bb7976094p386,
                0x1.c91433e7e7b18p-800,
                0x1.c91433e7e7b18p-801,
                0x1.c9218e9410456p-419,
                0x1.c9222d8ae6073p861,
                0x1.c9222d8ae6073p862,
                0x1.c92da18b61796p-608,
                0x1.c92da18b61796p-609,
                0x1.c92da18b61796p-610,
                0x1.c930e29a0a1b9p-984,
                0x1.c931e8ab87173p338,
                0x1.c93555304963p850,
                0x1.c9360f490ce54p-460,
                0x1.c93beef0cb796p+852,
                0x1.c93ec107ap-1036,
                0x1.c93ec107ap-1037,
                0x1.c93ec25684542p841,
                0x1.c9411a611798cp-984,
                0x1.c9411a611798cp-985,
                0x1.c9411a611798cp-986,
                0x1.c9411a611798cp-987,
                0x1.c94fd1b292ec4p245,
                0x1.c9548eab35381p602,
                0x1.c95afb9ec4492p281,
                0x1.c95baa4d490aep-443,
                0x1.c95baa4d490aep-444,
                0x1.c95df2c10d4e2p+712,
                0x1.c962dccdbee15p-381,
                0x1.c969cbb8c121p693,
                0x1.c96ca2e963c2cp399,
                0x1.c96ee81abc514p-658,
                0x1.c96ee81abc514p-659,
                0x1.c98014f9552d8p-690,
                0x1.c98014f9552d8p-691,
                0x1.c980919f2a088p681,
                0x1.c98a7197f170ep+508,
                0x1.c990706227bbp-218,
                0x1.c992d5831453p-127,
                0x1.c992df55abb64p-643,
                0x1.c9a48p-1057,
                0x1.c9a6d2e489e94p535,
                0x1.c9a6d2e489e94p+536,
                0x1.c9a977590d902p777,
                0x1.c9abd0472548p823,
                0x1.c9abd0472548p+824,
                0x1.c9aecaf6228p-953,
                0x1.c9bbe479c6ffdp-449,
                0x1.c9bbe479c6ffdp-450,
                0x1.c9bfd03e4c2d8p-122,
                0x1.c9bfd03e4c2d8p-123,
                0x1.c9bfd03e4c2d8p-124,
                0x1.c9c376184a58ap714,
                0x1.c9c376184a58ap715,
                0x1.c9c4p-1060,
                0x1.c9c645290d4dcp+748,
                0x1.c9c645290d4dcp749,
                0x1.c9cb991e10e0bp-590,
                0x1.c9cd41f7e26aap-199,
                0x1.c9d1954c15414p695,
                0x1.c9d1954c15414p+696,
                0x1.c9d1954c15414p697,
                0x1.c9d5827bb657dp+932,
                0x1.c9d912326843cp702,
                0x1.c9daee960d2bcp401,
                0x1.c9df8c81ff6b6p+448,
                0x1.c9e2941c557bfp-347,
                0x1.c9eb29d7137dap-160,
                0x1.c9f561ff1708fp-842,
                0x1.c9p-1066,
                0x1.ca071de2d602fp-331,
                0x1.ca0b198e96da4p-103,
                0x1.ca0dd76d376ap961,
                0x1.ca2850cc4fda5p-624,
                0x1.ca3314be67933p351,
                0x1.ca38f350b22dep-826,
                0x1.ca38f350b22dep-827,
                0x1.ca38f350b22dep-828,
                0x1.ca38f350b22dep-829,
                0x1.ca3a7a2bcdff2p-184,
                0x1.ca452c4fba87cp-1000,
                0x1.ca452c4fba87cp-1001,
                0x1.ca452c4fba87cp-1002,
                0x1.ca5de5b8cb4c7p725,
                0x1.ca6265e280b91p-339,
                0x1.ca6265e280b91p-340,
                0x1.ca67083374514p-496,
                0x1.ca67083374514p-497,
                0x1.ca67083374514p-498,
                0x1.ca67083374514p-499,
                0x1.ca67574a9172ap703,
                0x1.ca69f1447e1fep-233,
                0x1.ca6be3761436p341,
                0x1.ca7079534ef8ep-535,
                0x1.ca7079534ef8ep-536,
                0x1.ca7eb0ac40baap-357,
                0x1.ca810d81ba018p-643,
                0x1.ca810d81ba018p-644,
                0x1.ca8d288ee3d18p-165,
                0x1.ca9bade45b94ap+260,
                0x1.ca9bade45b94ap261,
                0x1.caa198d551b24p-817,
                0x1.caa20aeedaa5ap+844,
                0x1.caa71d335ccaep-254,
                0x1.caad28f23a101p-574,
                0x1.caad28f23a101p-575,
                0x1.caad28f23a101p-576,
                0x1.cab3210f3bb95p-343,
                0x1.cab3210f3bb96p-342,
                0x1.cab3210f3bb96p-343,
                0x1.cab63f2ffeb64p-365,
                0x1.cab7bd666f388p994,
                0x1.cab9b8de58612p-822,
                0x1.cac3f75f268bep-920,
                0x1.cace13031758p-1029,
                0x1.cadd55723ee5p-559,
                0x1.cadd55723ee5p-560,
                0x1.cadd55723ee5p-561,
                0x1.cadd55723ee5p-562,
                0x1.cadd55723ee5p-563,
                0x1.caddd3f747729p-846,
                0x1.cae3a312a2cd4p-530,
                0x1.cae8d0ff55ba4p219,
                0x1.cae8d0ff55ba4p+220,
                0x1.cae8d0ff55ba4p221,
                0x1.cb07bc7ff39a4p545,
                0x1.cb0d3721454d5p693,
                0x1.cb1145e37949p-705,
                0x1.cb1e0fca5a87p-819,
                0x1.cb26653d6c02ep609,
                0x1.cb26653d6c02ep610,
                0x1.cb26653d6c02ep611,
                0x1.cb26653d6c02ep+612,
                0x1.cb26653d6c02ep613,
                0x1.cb26653d6c02ep614,
                0x1.cb2d6f618c878p142,
                0x1.cb2d6f618c879p142,
                0x1.cb3p-1062,
                0x1.cb49d0de9b901p+680,
                0x1.cb4aa278ecc4fp+1016,
                0x1.cb55eb78a40a3p269,
                0x1.cb59b4ccc1a1fp-706,
                0x1.cb61567bcba11p-233,
                0x1.cb644dc1633cp-72,
                0x1.cb644dc1633cp-73,
                0x1.cb6a6f1f8bc45p-447,
                0x1.cb6ce673b9e0fp-301,
                0x1.cb6ce673b9e0fp-302,
                0x1.cb7791f86d5abp-721,
                0x1.cb7791f86d5abp-722,
                0x1.cb7ae41f3c98ep-486,
                0x1.cb81d18d9c8fcp805,
                0x1.cb82bf3240cbbp935,
                0x1.cb85173136bb6p622,
                0x1.cb857997b78cep-478,
                0x1.cb857997b78cep-479,
                0x1.cb857997b78cep-480,
                0x1.cb88b86878c3ap-185,
                0x1.cb88b86878c3ap-186,
                0x1.cb88b86878c3ap-187,
                0x1.cb9b803439d3ap1021,
                0x1.cb9b803439d3ap1022,
                0x1.cb9b803439d3ap1023,
                0x1.cba03abd64eabp154,
                0x1.cba1f251ecd1ep-340,
                0x1.cba7de5054486p627,
                0x1.cbac09de3907p386,
                0x1.cbae63062497p419,
                0x1.cbb0cee0d5df4p+692,
                0x1.cbb547777a284p+960,
                0x1.cbb547777a284p961,
                0x1.cbb547777a284p962,
                0x1.cbb547777a284p963,
                0x1.cbb547777a284p+964,
                0x1.cbb547777a284p965,
                0x1.cbb547777a284p966,
                0x1.cbb547777a284p967,
                0x1.cbb547777a284p+968,
                0x1.cbb75bffe0926p-58,
                0x1.cbb9aa3a8cc7bp-739,
                0x1.cbc13b345478p+736,
                0x1.cbc2e621f5aecp935,
                0x1.cbdc228c8273cp667,
                0x1.cbe1db3502707p-344,
                0x1.cbe3d11b109fcp-178,
                0x1.cbe95a8363414p911,
                0x1.cbeb762763434p-845,
                0x1.cbf10878c2d69p766,
                0x1.cc01cfa68d3fcp-340,
                0x1.cc08102f0da5cp679,
                0x1.cc0833bd429c6p-342,
                0x1.cc088373391b1p-357,
                0x1.cc14fbeaad55bp-344,
                0x1.cc189a0353c11p218,
                0x1.cc1ea6e980c66p398,
                0x1.cc2281179a089p-148,
                0x1.cc325f3828d6cp-291,
                0x1.cc3541d3b98cep-192,
                0x1.cc359e067a348p-1024,
                0x1.cc359e067a348p-1025,
                0x1.cc40611987936p-420,
                0x1.cc4334015f75p929,
                0x1.cc4334015f75p930,
                0x1.cc4476fc5d0a3p-377,
                0x1.cc47ff7ac70a7p-118,
                0x1.cc481ca0583afp-346,
                0x1.cc504ff123c96p673,
                0x1.cc504ff123c96p674,
                0x1.cc504ff123c96p675,
                0x1.cc504ff123c96p+676,
                0x1.cc520fd7a59dap-485,
                0x1.cc520fd7a59dap-486,
                0x1.cc65d1199c7d3p279,
                0x1.cc6db31805492p313,
                0x1.cc803815b766cp-567,
                0x1.cc93d2d514d35p954,
                0x1.cca2f5b15cc92p-878,
                0x1.cca30ff553127p871,
                0x1.cca7b7ba907e2p998,
                0x1.ccab2341e1a34p-449,
                0x1.ccab2341e1a34p-450,
                0x1.ccab2341e1a34p-451,
                0x1.ccae4e03fdddcp515,
                0x1.ccb0536608d61p-539,
                0x1.ccb0536608d62p-538,
                0x1.ccb0536608d62p-539,
                0x1.ccc1053757284p810,
                0x1.cccd24ba7384ep662,
                0x1.ccce2f3f92a2p+148,
                0x1.ccce2f3f92a2p149,
                0x1.ccd46e4ebfe2ep-227,
                0x1.ccd46e4ebfe2ep-228,
                0x1.ccd46e4ebfe2ep-229,
                0x1.ccdf426637162p-761,
                0x1.cce17ec158eabp-342,
                0x1.ccf3dbbabaa67p+792,
                0x1.ccff3276f868cp-801,
                0x1.ccff3276f868cp-802,
                0x1.ccff3276f868cp-803,
                0x1.ccff3276f868cp-804,
                0x1.ccff3276f868cp-805,
                0x1.cd05d12d12b32p195,
                0x1.cd06167510e62p-777,
                0x1.cd0d70742988ap181,
                0x1.cd0d70742988ap182,
                0x1.cd149f7703cffp-349,
                0x1.cd1c57b669959p-922,
                0x1.cd212b2079feep-783,
                0x1.cd230a7ff47c4p145,
                0x1.cd230a7ff47c4p146,
                0x1.cd2a3456f4c6ap558,
                0x1.cd2a3456f4c6ap559,
                0x1.cd2b297d889bcp-54,
                0x1.cd2c9a6cdd003p-41,
                0x1.cd2e3e9fc7802p-900,
                0x1.cd320f0c9cf1fp-300,
                0x1.cd34d664fb002p-125,
                0x1.cd34d664fb002p-126,
                0x1.cd4fb12d618e5p-403,
                0x1.cd56221494558p-572,
                0x1.cd5bee57763e6p-241,
                0x1.cd5bee57763e6p-242,
                0x1.cd5bee57763e6p-243,
                0x1.cd5bee57763e6p-244,
                0x1.cd634d6eb8a68p-239,
                0x1.cd65116e4dc4bp134,
                0x1.cd7091b33fa74p973,
                0x1.cd7aaaadd6652p+548,
                0x1.cd821069080efp501,
                0x1.cd89d96faeed8p-269,
                0x1.cd8d373b6ef05p-786,
                0x1.cd96a6972a14ap-193,
                0x1.cd96a6972a14ap-194,
                0x1.cd96a6972a14ap-195,
                0x1.cd96a6972a14ap-196,
                0x1.cd96a6972a14ap-197,
                0x1.cd96a6972a14ap-198,
                0x1.cd96a6972a14ap-199,
                0x1.cd96a6972a14ap-200,
                0x1.cd96a6972a14ap-201,
                0x1.cd96a6972a14ap-202,
                0x1.cd9b6550d25acp778,
                0x1.cd9b6550d25acp779,
                0x1.cda62055b2d9ep431,
                0x1.cda62055b2d9ep+432,
                0x1.cda62055b2d9ep433,
                0x1.cda62055b2d9ep434,
                0x1.cda62055b2d9ep435,
                0x1.cda97a4331312p+324,
                0x1.cda97a4331312p325,
                0x1.cda97a4331312p326,
                0x1.cda97a4331312p327,
                0x1.cdaee45288a3cp999,
                0x1.cdb2beca6557ap-647,
                0x1.cdb2beca6557ap-648,
                0x1.cdb2beca6557ap-649,
                0x1.cdb2beca6557ap-650,
                0x1.cdb2beca6557ap-651,
                0x1.cdba557bb2777p262,
                0x1.cdc06b20ef183p-82,
                0x1.cdc06b20ef183p-83,
                0x1.cdc56a1ed2752p-995,
                0x1.cdc56a1ed2752p-996,
                0x1.cdc7894451p-1032,
                0x1.cdc7894451p-1033,
                0x1.cdc7894451p-1034,
                0x1.cdcf64808afecp-713,
                0x1.cdd2a2c0fdbd4p-417,
                0x1.cdd4ef762b4c4p-366,
                0x1.cdda9e971ae78p471,
                0x1.cdda9e971ae78p+472,
                0x1.cddec96267c4cp-487,
                0x1.cde29a4beb8e1p-108,
                0x1.cde29a4beb8e1p-109,
                0x1.cde88353be021p407,
                0x1.cdee98dba97cap-455,
                0x1.cdf5c52ceff2p-796,
                0x1.cdff24bb44723p+200,
                0x1.ce0269d031dfp255,
                0x1.ce08534e32656p-474,
                0x1.ce166d715dfb4p390,
                0x1.ce166d715dfb4p391,
                0x1.ce2137f743382p+916,
                0x1.ce2137f743382p917,
                0x1.ce2137f743382p918,
                0x1.ce3c3c4ea0c98p+824,
                0x1.ce41f9d89cf5bp-42,
                0x1.ce4be2364522p767,
                0x1.ce4be2364522p+768,
                0x1.ce4be2364522p769,
                0x1.ce4be2364522p770,
                0x1.ce4be2364522p771,
                0x1.ce4cadff02fc8p-661,
                0x1.ce4cadff02fc8p-662,
                0x1.ce4d8f7d3333cp285,
                0x1.ce55753872be4p541,
                0x1.ce55753872be4p542,
                0x1.ce55753872be4p543,
                0x1.ce5afa920324fp194,
                0x1.ce6351f6dfe7ep731,
                0x1.ce6351f6dfe7ep+732,
                0x1.ce65e11086197p+700,
                0x1.ce77c2b3328fcp-474,
                0x1.ce77c2b3328fcp-475,
                0x1.ce77c2b3328fcp-476,
                0x1.ce77c2b3328fcp-477,
                0x1.ce77c2b3328fcp-478,
                0x1.ce77c2b3328fcp-479,
                0x1.ce77c2b3328fcp-480,
                0x1.ce77c2b3328fcp-481,
                0x1.ce77c2b3328fcp-482,
                0x1.ce77c2b3328fcp-483,
                0x1.ce77c2b3328fcp-484,
                0x1.ce77c2b3328fcp-485,
                0x1.ce77c2b3328fcp-486,
                0x1.ce77c2b3328fcp-487,
                0x1.ce77c2b3328fcp-488,
                0x1.ce7a6ac3cec35p154,
                0x1.ce7a6ac3cec35p155,
                0x1.ce84b1cb8a9e8p933,
                0x1.ce886fb5ffd6ep-223,
                0x1.ce886fb5ffd6ep-224,
                0x1.ce886fb5ffd6ep-225,
                0x1.ce886fb5ffd6ep-226,
                0x1.ce886fb5ffd6ep-227,
                0x1.ce8b86b625d7p-551,
                0x1.ce8b86b625d7p-552,
                0x1.ce8ceb0ee277ap927,
                0x1.ce90617e4ded9p-706,
                0x1.ce92a2bac81a4p495,
                0x1.ce92a2bac81a4p+496,
                0x1.ce92a2bac81a4p497,
                0x1.ce92a2bac81a4p498,
                0x1.ce970112dfbb8p-563,
                0x1.ce9a4bfe8d4aap-295,
                0x1.ce9d671500ebep+576,
                0x1.ce9d671500ebep577,
                0x1.ce9d671500ebep578,
                0x1.ce9d671500ebep579,
                0x1.ce9d671500ebep+580,
                0x1.ce9f776f5c2aap-961,
                0x1.ceafbafd80e84p-735,
                0x1.ceafbafd80e85p-735,
                0x1.ceb0920211216p923,
                0x1.ceb89f2dafbddp-470,
                0x1.cebb17fcb7b0bp255,
                0x1.cec9bb224f008p135,
                0x1.cecc0c5d3ddd8p+816,
                0x1.cecc0c5d3ddd8p817,
                0x1.cecc540023aebp-197,
                0x1.cedab6ed9b7bcp233,
                0x1.cedbbb3c9dd72p-858,
                0x1.cedbbb3c9dd72p-859,
                0x1.cedbbb3c9dd72p-860,
                0x1.cee15baa2603p-854,
                0x1.cee9888c13966p-157,
                0x1.cf01bbacaab12p+460,
                0x1.cf01bbacaab12p461,
                0x1.cf01bbacaab12p462,
                0x1.cf01bbacaab12p463,
                0x1.cf01bbacaab12p+464,
                0x1.cf041df6dde5p-899,
                0x1.cf045f58ebb9ap-392,
                0x1.cf045f58ebb9ap-393,
                0x1.cf04c4d8d2764p-193,
                0x1.cf04c4d8d2764p-194,
                0x1.cf086bd5b1a24p559,
                0x1.cf0deb2b2dcdp-405,
                0x1.cf1396c333016p446,
                0x1.cf1396c333016p447,
                0x1.cf24d2cb894e9p-1022,
                0x1.cf252b25d2a1cp-539,
                0x1.cf2970e397cb8p635,
                0x1.cf2a1ef3a32bap699,
                0x1.cf2a660b312d7p-836,
                0x1.cf2a660b312d7p-837,
                0x1.cf2a660b312d7p-838,
                0x1.cf2b1970e7258p-249,
                0x1.cf2b1970e7258p-250,
                0x1.cf30c92a5165ep731,
                0x1.cf30c92a5165ep+732,
                0x1.cf30c92a5165ep733,
                0x1.cf30c92a5165ep734,
                0x1.cf30c92a5165ep735,
                0x1.cf389cd46047dp85,
                0x1.cf3a860e5d71ap-702,
                0x1.cf3a860e5d71ap-703,
                0x1.cf3d70f926a7cp+732,
                0x1.cf4311ffp-1042,
                0x1.cf4312p-1049,
                0x1.cf4312p-1050,
                0x1.cf4312p-1051,
                0x1.cf45d333da324p454,
                0x1.cf45d333da324p455,
                0x1.cf45d333da324p+456,
                0x1.cf45d333da324p457,
                0x1.cf47b7b52b204p-535,
                0x1.cf4d84844e7bep+524,
                0x1.cf4d84844e7bep525,
                0x1.cf4d84844e7bep526,
                0x1.cf4d8b4be7739p+640,
                0x1.cf51473be9637p-738,
                0x1.cf5287cbaa392p-661,
                0x1.cf5287cbaa392p-662,
                0x1.cf5308837a0c6p-962,
                0x1.cf53f198a93f2p823,
                0x1.cf5a1f3e5f554p-672,
                0x1.cf5a1f3e5f554p-673,
                0x1.cf7f4dd6c791fp813,
                0x1.cf85317b76385p874,
                0x1.cf86316f2da77p-762,
                0x1.cf8a8a122057ep-941,
                0x1.cf8a8a122057ep-942,
                0x1.cf8a8a122057ep-943,
                0x1.cf8caa7af94b2p-688,
                0x1.cf905112fe36ap489,
                0x1.cf905112fe36ap490,
                0x1.cf94a562b4e1p-307,
                0x1.cf95227c75eefp+444,
                0x1.cf99394cfbc13p-91,
                0x1.cf997e626d288p-520,
                0x1.cf9c585675de4p991,
                0x1.cf9c585675de4p+992,
                0x1.cf9c585675de4p993,
                0x1.cfa698c95390cp235,
                0x1.cfa698c95390cp+236,
                0x1.cfa751357e748p+552,
                0x1.cfaead9244c76p295,
                0x1.cfb1261a6dd7ep+284,
                0x1.cfc476e5148c5p+660,
                0x1.cfc4edef530ep142,
                0x1.cfd151be04b3ep-616,
                0x1.cfd6a2cdb74cap134,
                0x1.cfdb020e75a7dp-140,
                0x1.cfe7ca24cp-1039,
                0x1.cfe9f9074c8fp+476,
                0x1.cff8d77725a24p-629,
                0x1.cff8d77725a24p-630,
                0x1.cff8e0131ae62p287,
                0x1.cff8e0131ae62p+288,
                0x1.cp-1071,
                0x1.cp-1072,
                0x1.d00a31d7a7dc2p-984,
                0x1.d0180392d5db8p-790,
                0x1.d0180392d5db8p-791,
                0x1.d0180392d5db8p-792,
                0x1.d0180392d5db8p-793,
                0x1.d0180392d5db8p-794,
                0x1.d0180392d5db8p-795,
                0x1.d0180392d5db8p-796,
                0x1.d0180392d5db8p-797,
                0x1.d020192cc2618p+936,
                0x1.d022390f8b837p+720,
                0x1.d022390f8b838p+720,
                0x1.d02d86fc90664p-847,
                0x1.d03281e3d0d21p-717,
                0x1.d037646b1e772p-542,
                0x1.d037646b1e772p-543,
                0x1.d048dfcf3128ep-242,
                0x1.d05244fe5066ap-871,
                0x1.d05244fe5066ap-872,
                0x1.d05244fe5066ap-873,
                0x1.d05244fe5066ap-874,
                0x1.d05244fe5066ap-875,
                0x1.d05244fe5066ap-876,
                0x1.d05244fe5066ap-877,
                0x1.d05244fe5066ap-878,
                0x1.d05244fe5066ap-879,
                0x1.d05244fe5066ap-880,
                0x1.d05244fe5066ap-881,
                0x1.d05d13af30f76p-312,
                0x1.d05d13af30f76p-313,
                0x1.d0681489839d5p-237,
                0x1.d0681489839d5p-238,
                0x1.d0681489839d5p-239,
                0x1.d06c1b7c1edc8p322,
                0x1.d06c1b7c1edc8p323,
                0x1.d06c1b7c1edc8p+324,
                0x1.d06c1b7c1edc8p325,
                0x1.d06c1b7c1edc8p326,
                0x1.d07e0b58e840ep+372,
                0x1.d07e0b58e840ep373,
                0x1.d087db5152f5p-470,
                0x1.d088c3bca25e6p435,
                0x1.d08e537b649e4p641,
                0x1.d092bf226f1cap-635,
                0x1.d092bf226f1cap-636,
                0x1.d092bf226f1cap-637,
                0x1.d092bf226f1cap-638,
                0x1.d092bf226f1cap-639,
                0x1.d09f2a3aa9e9p223,
                0x1.d0a0a0c569c4p-1028,
                0x1.d0a3f06ecdf46p197,
                0x1.d0a3f06ecdf46p198,
                0x1.d0b15a491eb84p-930,
                0x1.d0b15a491eb84p-931,
                0x1.d0b1a88a8e929p395,
                0x1.d0be0c8630d54p-776,
                0x1.d0bf3b519b42ep+604,
                0x1.d0ce130ab6fcap-393,
                0x1.d0d1b32e43236p-870,
                0x1.d0d832e865073p-213,
                0x1.d0e2bfe09e3fdp-923,
                0x1.d0e96d1c46aa7p+440,
                0x1.d0f4d2865104p-697,
                0x1.d0fcbf97f8d11p-401,
                0x1.d0fcbf97f8d11p-402,
                0x1.d0fcbf97f8d11p-403,
                0x1.d0fd4a1a93ed4p-778,
                0x1.d107baaf7a208p-680,
                0x1.d107d57134a82p93,
                0x1.d107d57134a82p94,
                0x1.d10dd5cc103eep775,
                0x1.d1139e6718b28p-995,
                0x1.d122ec5151de7p-525,
                0x1.d12d41afca3c2p-446,
                0x1.d12e0c9b0f8ddp751,
                0x1.d14c37de42f8ap-442,
                0x1.d14c37de42f8ap-443,
                0x1.d154f09407d58p-76,
                0x1.d15827601707ap-49,
                0x1.d15b1321990aep611,
                0x1.d15b1321990aep+612,
                0x1.d15b1321990aep613,
                0x1.d15b1321990aep614,
                0x1.d15b1321990aep615,
                0x1.d15c78cb9d8a1p+948,
                0x1.d15c78cb9d8a1p949,
                0x1.d15c78cb9d8a1p950,
                0x1.d16226a2a4eb4p-406,
                0x1.d166988760d62p-863,
                0x1.d1698525d7a9ep+676,
                0x1.d16be8b90551cp991,
                0x1.d16be8b90551cp+992,
                0x1.d16cfc8345ffap795,
                0x1.d16ffa70a58p-1033,
                0x1.d17f3825a17c3p562,
                0x1.d17f3825a17c3p563,
                0x1.d184514a1acf4p953,
                0x1.d1858af579bbp-868,
                0x1.d18fc8b6b68b7p-362,
                0x1.d19267074d12bp-725,
                0x1.d19a188d6f8bep-584,
                0x1.d19a188d6f8bep-585,
                0x1.d19a188d6f8bep-586,
                0x1.d19bffd35ac51p+660,
                0x1.d19fdb4741c87p-635,
                0x1.d1a34949faf63p859,
                0x1.d1a400b33d9cep+500,
                0x1.d1a400b33d9cep501,
                0x1.d1a8715c3da86p470,
                0x1.d1a94a2p39,
                0x1.d1ab2406674cp978,
                0x1.d1ab2406674cp979,
                0x1.d1ab2406674cp+980,
                0x1.d1ab2406674cp981,
                0x1.d1af6e178fdc9p158,
                0x1.d1b2eb15359e2p838,
                0x1.d1c163b587f4dp-760,
                0x1.d1c26db7d0daep649,
                0x1.d1c26db7d0daep650,
                0x1.d1c26db7d0daep651,
                0x1.d1c8b71aee3aep566,
                0x1.d1c8b71aee3aep567,
                0x1.d1c9cc639b8f2p-554,
                0x1.d1c9cc639b8f2p-555,
                0x1.d1c9cc639b8f2p-556,
                0x1.d1c9cc639b8f2p-557,
                0x1.d1c9cc639b8f2p-558,
                0x1.d1dfb07f094dfp-345,
                0x1.d1e0eed746962p-174,
                0x1.d1e0eed746962p-175,
                0x1.d1e0eed746962p-176,
                0x1.d1e0eed746962p-177,
                0x1.d1ebc4c3a7eecp-470,
                0x1.d1f1e30afc14fp-15,
                0x1.d1f570b8c43fcp509,
                0x1.d1f570b8c43fcp510,
                0x1.d1fb61d331422p+180,
                0x1.d20268675ddfep+944,
                0x1.d20e63b7a802p786,
                0x1.d20e63b7a802p787,
                0x1.d20e63b7a802p+788,
                0x1.d20e63b7a802p789,
                0x1.d214fbd9a9548p-606,
                0x1.d214fbd9a9548p-607,
                0x1.d214fbd9a9548p-608,
                0x1.d214fbd9a9548p-609,
                0x1.d214fbd9a9548p-610,
                0x1.d21b1b515686fp-63,
                0x1.d21b1b515686fp-64,
                0x1.d22573a28f19dp+524,
                0x1.d22573a28f19ep+524,
                0x1.d22573a28f19ep525,
                0x1.d2294b85210e6p595,
                0x1.d22c233e6dafcp-964,
                0x1.d22c233e6dafcp-965,
                0x1.d24ffd0f428b1p-658,
                0x1.d24ffd0f428b1p-659,
                0x1.d2525452f6a0ep-1016,
                0x1.d2525452f6a0ep-1017,
                0x1.d252c8c6f198p-683,
                0x1.d25b42bb28962p247,
                0x1.d25b42bb28962p+248,
                0x1.d26add63e2c12p-490,
                0x1.d26add63e2c12p-491,
                0x1.d26add63e2c12p-492,
                0x1.d26add63e2c12p-493,
                0x1.d26add63e2c12p-494,
                0x1.d26c3decf4504p-738,
                0x1.d27524df2f358p827,
                0x1.d27bcec400c5ep+620,
                0x1.d27bcec400c5ep621,
                0x1.d27bcec400c5ep622,
                0x1.d284ff114656ep863,
                0x1.d284ff114656ep+864,
                0x1.d2860881716f3p835,
                0x1.d286fd9af25a2p-64,
                0x1.d28c912cb5bcfp970,
                0x1.d2990c0559abep478,
                0x1.d29f4e3563b6ep218,
                0x1.d2a1be4048f9p1009,
                0x1.d2a1be4048f9p1010,
                0x1.d2a1be4048f9p1011,
                0x1.d2a1be4048f9p+1012,
                0x1.d2a1be4048f9p1013,
                0x1.d2a1be4048f9p1014,
                0x1.d2a1be4048f9p1015,
                0x1.d2a1be4048f9p+1016,
                0x1.d2aeee55e2e01p159,
                0x1.d2b9e4b199259p210,
                0x1.d2bc619dc0aeep590,
                0x1.d2c918ede8c4dp153,
                0x1.d2e489e6db795p-867,
                0x1.d2e5e9cc144edp-608,
                0x1.d2ec75df6ba2ap-232,
                0x1.d2f330b7a8ecp-693,
                0x1.d2f7b208bb6d2p789,
                0x1.d3000353f2295p-297,
                0x1.d30d5a9aa01fep+188,
                0x1.d3127b28fceb4p-947,
                0x1.d3127b28fceb4p-948,
                0x1.d3127b28fceb4p-949,
                0x1.d314dbcddad93p535,
                0x1.d31e80e806956p-526,
                0x1.d31f5cdp-1046,
                0x1.d322635597848p-209,
                0x1.d3260ddefd248p906,
                0x1.d32db533d8574p+696,
                0x1.d331a4b10d3f6p-641,
                0x1.d331a4b10d3f6p-642,
                0x1.d33618a74f93fp-841,
                0x1.d33b75368a4eep-1023,
                0x1.d3409dfbca27p-929,
                0x1.d3409dfbca27p-930,
                0x1.d3409dfbca27p-931,
                0x1.d3409dfbca27p-932,
                0x1.d3409dfbca27p-933,
                0x1.d3409dfbca27p-934,
                0x1.d3409dfbca27p-935,
                0x1.d3409dfbca27p-936,
                0x1.d3409dfbca27p-937,
                0x1.d3409dfbca27p-938,
                0x1.d3409dfbca27p-939,
                0x1.d345768ed6388p-799,
                0x1.d363461619394p714,
                0x1.d363461619394p715,
                0x1.d36365877b259p+372,
                0x1.d374fc5e04b94p-713,
                0x1.d374fc5e04b94p-714,
                0x1.d374fc5e04b94p-715,
                0x1.d379400fed248p-532,
                0x1.d37fea73c8316p302,
                0x1.d39616397b45p993,
                0x1.d39fe3d6b390cp1005,
                0x1.d3a76c459b4ep622,
                0x1.d3ae36d13bbcep-157,
                0x1.d3ae83e4322b3p-38,
                0x1.d3b275025ca5ep-398,
                0x1.d3b3aeae7ecc8p611,
                0x1.d3b7a1d154abap123,
                0x1.d3b7a1d154abap+124,
                0x1.d3b88deca7fb7p-761,
                0x1.d3c31ef77f432p127,
                0x1.d3c66b278a405p-956,
                0x1.d3c66b278a405p-957,
                0x1.d3e34c231ec4ap795,
                0x1.d3e9597e9655p-1001,
                0x1.d3ed8p-1056,
                0x1.d3f30fafd0a98p777,
                0x1.d3f30fafd0a98p778,
                0x1.d3f5775a373c6p-997,
                0x1.d402b39aa402bp-838,
                0x1.d405c87b8133ep339,
                0x1.d405c87b8133ep+340,
                0x1.d405c87b8133ep341,
                0x1.d405c87b8133ep342,
                0x1.d40db1e2063bep-142,
                0x1.d40db1e2063bep-143,
                0x1.d410818p-1049,
                0x1.d415fd4f05f55p527,
                0x1.d42126bb50d94p-165,
                0x1.d421397011886p110,
                0x1.d426624ac47fcp+368,
                0x1.d426624ac47fcp369,
                0x1.d426624ac47fcp370,
                0x1.d42aea2879f2ep+328,
                0x1.d42e658fcf54cp-754,
                0x1.d42ec77e7b442p-705,
                0x1.d434cf1d51e2cp-386,
                0x1.d4446075c4934p130,
                0x1.d44f09adbaa1ap-562,
                0x1.d44f6f92e62bap527,
                0x1.d450ce624a7aap-134,
                0x1.d450ce624a7aap-135,
                0x1.d450ce624a7aap-136,
                0x1.d450ce624a7aap-137,
                0x1.d460f4fca1d37p-145,
                0x1.d460f4fca1d37p-146,
                0x1.d462d84abe7c7p385,
                0x1.d47863b6d8dacp-431,
                0x1.d47d2eda82a4cp-117,
                0x1.d47d2eda82a4cp-118,
                0x1.d47ecb8d39915p-37,
                0x1.d47f6607c271p+96,
                0x1.d47f6607c271p97,
                0x1.d484ada4305fp-290,
                0x1.d484ada4305fp-291,
                0x1.d484ada4305fp-292,
                0x1.d4911f00a3eb2p-948,
                0x1.d49383e3cb932p707,
                0x1.d49383e3cb932p+708,
                0x1.d49383e3cb932p709,
                0x1.d49383e3cb932p710,
                0x1.d49383e3cb932p711,
                0x1.d49383e3cb932p+712,
                0x1.d49383e3cb932p713,
                0x1.d49383e3cb932p714,
                0x1.d49a0ea187205p630,
                0x1.d4a7bebfa31aap813,
                0x1.d4a7bebfa31aap814,
                0x1.d4acaf58c50fp-371,
                0x1.d4b796e84c442p581,
                0x1.d4b9c9288e998p785,
                0x1.d4c3e748491ccp386,
                0x1.d4c491cab8ed4p538,
                0x1.d4c491cab8ed4p539,
                0x1.d4c491cab8ed4p+540,
                0x1.d4c491cab8ed4p541,
                0x1.d4c8f6c905cb4p-987,
                0x1.d4c8f6c905cb4p-988,
                0x1.d4c953b5f018ep-493,
                0x1.d4c953b5f018ep-494,
                0x1.d4c953b5f018ep-495,
                0x1.d4c953b5f018ep-496,
                0x1.d4cd89771bd7ep291,
                0x1.d4d47a895d80ep-80,
                0x1.d4d47a895d80ep-81,
                0x1.d4d7f9ef376b7p+228,
                0x1.d4dcb8d709a54p438,
                0x1.d4e18178f58fcp881,
                0x1.d4e18178f58fcp882,
                0x1.d4e769ba40e1ep-91,
                0x1.d4e769ba40e1ep-92,
                0x1.d4f6c87f6ffbep-534,
                0x1.d4ffa345a1dfp-499,
                0x1.d4ffa345a1dfp-500,
                0x1.d4ffa345a1dfp-501,
                0x1.d508dfeeef07p-888,
                0x1.d509e72cb763dp-671,
                0x1.d50e9e6ef31c9p-632,
                0x1.d50fcc015f1b8p+916,
                0x1.d50fcc015f1b8p917,
                0x1.d5114ab57f13cp911,
                0x1.d5181f1d37b47p353,
                0x1.d52023a0e867p-891,
                0x1.d528fb8cb0dbap+140,
                0x1.d53844ee47dd1p-838,
                0x1.d53844ee47dd2p-835,
                0x1.d53844ee47dd2p-836,
                0x1.d53844ee47dd2p-837,
                0x1.d53844ee47dd2p-838,
                0x1.d53844ee47dd2p-839,
                0x1.d53dd86ae1adcp113,
                0x1.d53dd86ae1adcp114,
                0x1.d547567f72586p-204,
                0x1.d54a8e8f62134p325,
                0x1.d54a8e8f62134p326,
                0x1.d54f9a781a9ap695,
                0x1.d5519e667c438p+656,
                0x1.d5519e667c438p657,
                0x1.d5536bf0e0b9ep-681,
                0x1.d5536bf0e0b9ep-682,
                0x1.d55526cef60fep-809,
                0x1.d55ddf57a828cp-612,
                0x1.d55ddf57a828cp-613,
                0x1.d564f0c36fe2bp+280,
                0x1.d564f0c36fe2bp281,
                0x1.d56659e28e9bfp1013,
                0x1.d56a69832d456p-959,
                0x1.d56a69832d456p-960,
                0x1.d57ac1f725a76p629,
                0x1.d57d36befec31p175,
                0x1.d57d3a01d95ap-847,
                0x1.d58387e1e475p485,
                0x1.d58387e1e475p486,
                0x1.d587349dff3d2p-744,
                0x1.d587349dff3d2p-745,
                0x1.d589ae4d70218p-438,
                0x1.d589ae4d70218p-439,
                0x1.d589ae4d70218p-440,
                0x1.d589ae4d70218p-441,
                0x1.d589ae4d70218p-442,
                0x1.d58a7565691eep+472,
                0x1.d58aed51423e6p-834,
                0x1.d5aae3ca076fep-694,
                0x1.d5ab4c9fc9a7cp257,
                0x1.d5b561574765bp-353,
                0x1.d5b561574765cp-349,
                0x1.d5b561574765cp-350,
                0x1.d5b561574765cp-351,
                0x1.d5b561574765cp-352,
                0x1.d5b561574765cp-353,
                0x1.d5b9745cf3927p-802,
                0x1.d5c1d20b2e1a6p+364,
                0x1.d5d692adbd63ap-738,
                0x1.d5d7aca57df14p-355,
                0x1.d5d9660100a52p507,
                0x1.d5dc3324a720ap-426,
                0x1.d5dc6b54e1ef8p-103,
                0x1.d5dc7f803719fp-211,
                0x1.d5e4b026235ccp1023,
                0x1.d5f2babada406p727,
                0x1.d5f2babada406p+728,
                0x1.d5f2babada406p729,
                0x1.d6037540c2ba2p825,
                0x1.d6037540c2ba2p826,
                0x1.d6187859895eap-794,
                0x1.d62d25882d579p-550,
                0x1.d6329f1c35ca4p+132,
                0x1.d632e1f745624p-25,
                0x1.d632e1f745624p-26,
                0x1.d632e1f745624p-27,
                0x1.d632e1f745624p-28,
                0x1.d63b88ff7a8d3p+928,
                0x1.d63c4bd3d28afp170,
                0x1.d64af4cc52935p-667,
                0x1.d64c29852639bp-371,
                0x1.d64c29852639bp-372,
                0x1.d657059dc79aap+776,
                0x1.d657059dc79aap777,
                0x1.d657059dc79aap778,
                0x1.d657059dc79aap779,
                0x1.d663605229c6ap-676,
                0x1.d66aafddce6b2p934,
                0x1.d671cb56db24fp-960,
                0x1.d67802bca97f4p578,
                0x1.d67a35535c64p251,
                0x1.d67a35535c64p+252,
                0x1.d67a35535c64p253,
                0x1.d67a35535c64p254,
                0x1.d67a35535c64p255,
                0x1.d67a35535c64p+256,
                0x1.d67d51aa642e2p493,
                0x1.d6805e9d3be76p-928,
                0x1.d6851952c95e5p409,
                0x1.d68819cd22088p-577,
                0x1.d68819cd22088p-578,
                0x1.d68c37da9e73cp-223,
                0x1.d68c37da9e73cp-224,
                0x1.d68cc5bec8b72p+528,
                0x1.d68ebe5248fp-1029,
                0x1.d68ebe5248fp-1030,
                0x1.d6a500af205f3p-120,
                0x1.d6affe45f818fp617,
                0x1.d6bd327312c7ap309,
                0x1.d6bdb9d2689bfp+952,
                0x1.d6c6cc5d0653ep-389,
                0x1.d6c6cc5d0653ep-390,
                0x1.d6c6cc5d0653ep-391,
                0x1.d6db69e745bc4p-838,
                0x1.d6dfca3015d16p-808,
                0x1.d6e163b4e2712p166,
                0x1.d6e3e5c736196p251,
                0x1.d6ed5cc17efbbp489,
                0x1.d6ef248b1e3e8p437,
                0x1.d6f30247aa9fp-501,
                0x1.d6faebc7378aep+220,
                0x1.d6faebc7378aep221,
                0x1.d6faebc7378aep222,
                0x1.d7017928p-1045,
                0x1.d71f65ec1e828p269,
                0x1.d72a52153e23ep-598,
                0x1.d72a52153e23ep-599,
                0x1.d72a52153e23ep-600,
                0x1.d72p-1061,
                0x1.d731f52437514p323,
                0x1.d73a88fb534c2p893,
                0x1.d74124e3d1p-1034,
                0x1.d743332dea7e4p-431,
                0x1.d74686c8e9828p815,
                0x1.d74686c8e9828p+816,
                0x1.d74686c8e9828p817,
                0x1.d748087b4379ap-785,
                0x1.d74963f3f8d5fp941,
                0x1.d74ee4ac620cep-699,
                0x1.d763c6b84f0fep-870,
                0x1.d764fa272ae91p-879,
                0x1.d78799b683721p997,
                0x1.d787d61845608p482,
                0x1.d78dc091fd55ap206,
                0x1.d79be9f20cdd8p871,
                0x1.d79be9f20cdd8p+872,
                0x1.d79be9f20cdd8p873,
                0x1.d79be9f20cdd8p874,
                0x1.d79e39736201ap-736,
                0x1.d79f4da9d22ebp-768,
                0x1.d7a71edc2ab86p567,
                0x1.d7becc2f23ac2p-549,
                0x1.d7bfd11062836p746,
                0x1.d7bfd11062836p747,
                0x1.d7c38a1814b9ap786,
                0x1.d7c38a1814b9ap787,
                0x1.d7c38a1814b9ap+788,
                0x1.d7cde915b437ep-276,
                0x1.d7d2d9994681dp-367,
                0x1.d7ddf14a7da6ep-753,
                0x1.d7e442252eaa2p-452,
                0x1.d7e6f3a442968p-525,
                0x1.d7ec0176c677ap-911,
                0x1.d7f2b6c28c07p+776,
                0x1.d7f2b6c28c07p777,
                0x1.d7f43afdd8187p-320,
                0x1.d7f470566561fp-509,
                0x1.d7fd879482fe8p210,
                0x1.d802b63c849eep-245,
                0x1.d803aa0dad9ecp-781,
                0x1.d806173f062e2p923,
                0x1.d806173f062e2p+924,
                0x1.d806173f062e2p925,
                0x1.d812e87fd653ep-380,
                0x1.d81a06997487ap-371,
                0x1.d83c94fb6d2acp-64,
                0x1.d859fa208ad64p-717,
                0x1.d85f96f3fe65ap382,
                0x1.d867f1b9ea4eep183,
                0x1.d869cc2c2304cp+144,
                0x1.d87ed74459a5p-32,
                0x1.d892367318283p259,
                0x1.d892b7b18f0cdp+532,
                0x1.d893b55b521ep113,
                0x1.d89d8aafcdcp-279,
                0x1.d89eb285d9401p281,
                0x1.d8a252b0020dep473,
                0x1.d8a252b0020dep474,
                0x1.d8a4d3b40cfa8p-305,
                0x1.d8a4d3b40cfa8p-306,
                0x1.d8b3940fa51dp530,
                0x1.d8ba7f519c84fp421,
                0x1.d8ba7f519c85p421,
                0x1.d8ba7f519c85p422,
                0x1.d8c01c237e06ap-504,
                0x1.d8c36c85be04fp-793,
                0x1.d8cebb7e8d62dp-477,
                0x1.d8d2f6897ba4cp-378,
                0x1.d8d35b57dfea6p-263,
                0x1.d8dab8d5f43e4p593,
                0x1.d8de68dcc511bp-983,
                0x1.d8e94f9b878e6p-459,
                0x1.d8f64cc95a4f7p-394,
                0x1.d8f7c885578d7p-582,
                0x1.d8p-1068,
                0x1.d91c1eebe76e2p385,
                0x1.d91c1eebe76e2p386,
                0x1.d91e14e502ba4p194,
                0x1.d9351994ba0a2p423,
                0x1.d9388b3aa30a5p906,
                0x1.d9388b3aa30a6p905,
                0x1.d9388b3aa30a6p906,
                0x1.d9388b3aa30a6p907,
                0x1.d9388b3aa30a6p+908,
                0x1.d93cc56f3ce66p-445,
                0x1.d93cc56f3ce66p-446,
                0x1.d93cc56f3ce66p-447,
                0x1.d93cc56f3ce66p-448,
                0x1.d94e734256f4dp-356,
                0x1.d95108f882522p107,
                0x1.d951b0a757249p-755,
                0x1.d951b0a757249p-756,
                0x1.d952117dead3fp-683,
                0x1.d953b08d71958p+772,
                0x1.d953b08d71958p773,
                0x1.d953b08d71958p774,
                0x1.d9571ebbfea06p506,
                0x1.d9571ebbfea06p507,
                0x1.d95ed827dfe46p442,
                0x1.d9602236b1b1dp314,
                0x1.d965e7ae33132p-994,
                0x1.d96b78438293ep-436,
                0x1.d97079f8dcee6p599,
                0x1.d97079f8dcee6p+600,
                0x1.d9782f18db288p359,
                0x1.d981f01abbc36p-553,
                0x1.d9874caa60472p315,
                0x1.d98955603362fp263,
                0x1.d9a133d9abp-1034,
                0x1.d9a284e5d0382p-810,
                0x1.d9af0a05eb352p-591,
                0x1.d9af0a05eb352p-592,
                0x1.d9af0a05eb352p-593,
                0x1.d9af0a05eb352p-594,
                0x1.d9b04p-1055,
                0x1.d9b2f45d800bap750,
                0x1.d9b2f45d800bap751,
                0x1.d9b2f45d800bap+752,
                0x1.d9b2f45d800bap753,
                0x1.d9b2f45d800bap754,
                0x1.d9b92083b2f76p-170,
                0x1.d9b92083b2f76p-171,
                0x1.d9b92083b2f76p-172,
                0x1.d9b92083b2f76p-173,
                0x1.d9bb454ff62aap-658,
                0x1.d9bc0c70020e4p265,
                0x1.d9c55p-1054,
                0x1.d9ca79d89462ap-745,
                0x1.d9cb82ab95eccp947,
                0x1.d9cbe6f8c2c1ap-927,
                0x1.d9cf3d051125p+592,
                0x1.d9cf3d051125p593,
                0x1.d9dfb6f7c84d6p+484,
                0x1.d9fbb3c7b425ap279,
                0x1.da005057f753cp665,
                0x1.da01d733ec353p583,
                0x1.da0dbc841b35cp411,
                0x1.da0f1b20de1dp-705,
                0x1.da0f1b20de1dp-706,
                0x1.da0f1b20de1dp-707,
                0x1.da1b1ceab367ap231,
                0x1.da232347e6032p286,
                0x1.da232347e6032p287,
                0x1.da232347e6032p+288,
                0x1.da232347e6032p289,
                0x1.da232347e6032p290,
                0x1.da232347e6032p291,
                0x1.da232347e6032p+292,
                0x1.da232347e6032p293,
                0x1.da232347e6032p294,
                0x1.da2380603696cp547,
                0x1.da38b0b6e732cp-865,
                0x1.da3f4b253c51cp1001,
                0x1.da4657dfe6d1p-264,
                0x1.da48ce468e7c7p-260,
                0x1.da51893de4a84p-992,
                0x1.da51893de4a84p-993,
                0x1.da52347628a7ap+908,
                0x1.da56fafec28e8p341,
                0x1.da6aee305a74ep+956,
                0x1.da8e600f25b7ep+260,
                0x1.da9675b82dc32p-131,
                0x1.da99cf41be75ap-1009,
                0x1.da9cc95f29c34p-633,
                0x1.daac98af993bcp662,
                0x1.dab2ac51420efp-596,
                0x1.dab4aa585723ep775,
                0x1.dab4aa585723ep+776,
                0x1.dac2958d555cp-752,
                0x1.dac2958d555cp-753,
                0x1.dac74463a989fp225,
                0x1.dac74463a98ap225,
                0x1.dac74463a98ap226,
                0x1.dad713fb8700bp-228,
                0x1.dade9e03f67eap-816,
                0x1.dae9577b4f9f2p+148,
                0x1.daf5e21dd8576p-854,
                0x1.daf6862eadbd6p-14,
                0x1.daf6862eadbd6p-15,
                0x1.daf6862eadbd6p-16,
                0x1.daf6862eadbd6p-17,
                0x1.daff5a33d34c6p-515,
                0x1.daff5a33d34c6p-516,
                0x1.daff5a33d34c6p-517,
                0x1.db02d4a38c70cp-745,
                0x1.db19ff12bp-1038,
                0x1.db311203c8c4ep-660,
                0x1.db311203c8c4ep-661,
                0x1.db311203c8c4ep-662,
                0x1.db311203c8c4ep-663,
                0x1.db38eb71bf39dp-901,
                0x1.db39eb4eef65ep-321,
                0x1.db3beb1874e2p-626,
                0x1.db45dc38e0c82p710,
                0x1.db45dc38e0c82p711,
                0x1.db5b3af91392dp1015,
                0x1.db5b8d54cb3fep-181,
                0x1.db5b8d54cb3fep-182,
                0x1.db5b96c05379ap-740,
                0x1.db5b96c05379ap-741,
                0x1.db5f2aa2ea2c4p-128,
                0x1.db65273dc9eb2p775,
                0x1.db65273dc9eb2p+776,
                0x1.db65273dc9eb2p777,
                0x1.db6f69366f082p689,
                0x1.db7b95d11bdabp335,
                0x1.db81a1c4c2e72p-1002,
                0x1.db8991ff86127p994,
                0x1.db949c0eep-1039,
                0x1.dbb338620cf2ap-554,
                0x1.dbbac6f83a821p-800,
                0x1.dbbac6f83a821p-801,
                0x1.dbbac6f83a821p-802,
                0x1.dbbac6f83a821p-803,
                0x1.dbbac6f83a821p-804,
                0x1.dbc0a4a82d84p535,
                0x1.dbd796660ae8dp519,
                0x1.dbd86cd6238d9p-941,
                0x1.dbd86cd6238dap-939,
                0x1.dbd86cd6238dap-940,
                0x1.dbd86cd6238dap-941,
                0x1.dbd86cd6238dap-942,
                0x1.dbdd354c03e99p+396,
                0x1.dbfe512601fb7p-975,
                0x1.dc03b8fd7016ap-68,
                0x1.dc0f644627f56p-1022,
                0x1.dc0f644627f56p-1023,
                0x1.dc15a4233cb26p774,
                0x1.dc15ee558b51cp-246,
                0x1.dc16121ea6c5ep181,
                0x1.dc1b0cfd3fb22p-334,
                0x1.dc2fbeed89e54p-622,
                0x1.dc3e70b187e87p-531,
                0x1.dc3e70b187e87p-532,
                0x1.dc3ee2213726ap-22,
                0x1.dc4249836552ep+396,
                0x1.dc4249836552ep397,
                0x1.dc4c2acbb9f8ap+304,
                0x1.dc4cefee983bp925,
                0x1.dc4cefee983bp926,
                0x1.dc574d80cf16bp-456,
                0x1.dc579eab3943ep170,
                0x1.dc579eab3943ep171,
                0x1.dc5c01191af31p-256,
                0x1.dc5c01191af31p-257,
                0x1.dc5c173d2bb9fp881,
                0x1.dc73a5139490dp141,
                0x1.dc742c2deb9d9p561,
                0x1.dc7920ab1bafp-875,
                0x1.dc80e3528d388p+888,
                0x1.dc8fbc504d9ep-110,
                0x1.dc9e2d2bba0d4p-262,
                0x1.dcaeee8cfacf4p-393,
                0x1.dcb362ffe4d3ep-141,
                0x1.dcb362ffe4d3ep-142,
                0x1.dcb362ffe4d3ep-143,
                0x1.dcb59404ac4c2p+544,
                0x1.dcc46edbbed5p-863,
                0x1.dccf78aed26b2p+280,
                0x1.dccf78aed26b2p281,
                0x1.dcd018e0a54e4p-535,
                0x1.dcd018e0a54e4p-536,
                0x1.dcd018e0a54e4p-537,
                0x1.dcd018e0a54e4p-538,
                0x1.dcd018e0a54e4p-539,
                0x1.dcd65p29,
                0x1.dcf0947862fe6p-125,
                0x1.dcf0947862fe6p-126,
                0x1.dcf24754726cbp889,
                0x1.dcf7df8f573b8p591,
                0x1.dcf7df8f573b8p+592,
                0x1.dcf7df8f573b8p593,
                0x1.dcf7df8f573b8p594,
                0x1.dcf7df8f573b8p595,
                0x1.dcf7df8f573b8p+596,
                0x1.dcf7df8f573b8p597,
                0x1.dcf7df8f573b8p598,
                0x1.dcf7df8f573b8p599,
                0x1.dcf7df8f573b8p+600,
                0x1.dcfee6690ffc6p625,
                0x1.dcfee6690ffc6p626,
                0x1.dcfee6690ffc6p627,
                0x1.dcfee6690ffc6p+628,
                0x1.dcfee6690ffc6p629,
                0x1.dcfee6690ffc6p630,
                0x1.dcfee6690ffc6p631,
                0x1.dcfee6690ffc6p+632,
                0x1.dd16c1de822a6p-26,
                0x1.dd2515f1159c2p-59,
                0x1.dd33146a0c66dp+512,
                0x1.dd3ap-1059,
                0x1.dd3bfa4339838p243,
                0x1.dd3bfa4339838p+244,
                0x1.dd3bfa4339838p245,
                0x1.dd3bfa4339838p246,
                0x1.dd3bfa4339838p247,
                0x1.dd455061eb3f1p682,
                0x1.dd455061eb3f1p683,
                0x1.dd52aacdddc1cp-597,
                0x1.dd5491731ff45p-731,
                0x1.dd55745cbb7ecp514,
                0x1.dd55745cbb7edp513,
                0x1.dd55745cbb7edp514,
                0x1.dd5732c776657p-441,
                0x1.dd58e73fb2f9ep-545,
                0x1.dd6526e36a326p622,
                0x1.dd69790cb07c9p+404,
                0x1.dd702287b1344p498,
                0x1.dd702287b1344p499,
                0x1.dd72c4c36a9ep-230,
                0x1.dd7516b40af1p+408,
                0x1.dd791a64f1616p671,
                0x1.dd791a64f1616p+672,
                0x1.dd7c99a0a473p531,
                0x1.dd7c99a0a473p+532,
                0x1.dd88a12ac96f9p-143,
                0x1.dd9bf1ab6ac1cp-313,
                0x1.dda592e398dd7p419,
                0x1.dda6b965c962ap-858,
                0x1.dda6b965c962ap-859,
                0x1.dda7b0c87c87cp-920,
                0x1.dda7b0c87c87cp-921,
                0x1.ddc021313f00cp-875,
                0x1.ddc6dd7008593p-667,
                0x1.ddc7e975c5046p-240,
                0x1.ddc7e975c5046p-241,
                0x1.ddc7e975c5046p-242,
                0x1.ddc7e975c5046p-243,
                0x1.ddc7e975c5046p-244,
                0x1.ddc7e975c5046p-245,
                0x1.ddd162b99b73cp-30,
                0x1.ddd4baa009302p+1000,
                0x1.ddd4baa009302p1001,
                0x1.ddd4baa009302p998,
                0x1.ddd4baa009302p999,
                0x1.ddd63c5dd851p-103,
                0x1.ddd965a216a2cp-403,
                0x1.dde480a95e21fp-554,
                0x1.dde480a95e21fp-555,
                0x1.dde4a4ab13e0ap-158,
                0x1.ddea26f012a2cp923,
                0x1.ddea26f012a2cp+924,
                0x1.ddea26f012a2cp925,
                0x1.ddea26f012a2cp926,
                0x1.ddea26f012a2cp927,
                0x1.ddea26f012a2cp+928,
                0x1.ddea26f012a2cp929,
                0x1.ddea8af85535cp-894,
                0x1.ddea8af85535cp-895,
                0x1.ddea8af85535cp-896,
                0x1.dded757246ce6p+200,
                0x1.dded780d7f5dcp169,
                0x1.dded780d7f5dcp170,
                0x1.dded780d7f5dcp171,
                0x1.dded780d7f5dcp+172,
                0x1.de0b1c79ba47p-491,
                0x1.de19c8f6db8e5p965,
                0x1.de1ea791a6e7ep113,
                0x1.de1ea791a6e7ep114,
                0x1.de2696f4a4c72p723,
                0x1.de27e59fb067ap423,
                0x1.de27e59fb067ap+424,
                0x1.de27e59fb067ap425,
                0x1.de27e59fb067ap426,
                0x1.de27e59fb067ap427,
                0x1.de2cd1eb367dcp-756,
                0x1.de2cf2508efecp-762,
                0x1.de2f4249c3fcbp729,
                0x1.de2f4249c3fcbp730,
                0x1.de2f4249c3fcbp731,
                0x1.de30cbd5cad58p677,
                0x1.de379aa26b362p151,
                0x1.de3d37b4174a2p885,
                0x1.de3f2357c7ce2p349,
                0x1.de43ab8bd130fp-671,
                0x1.de43ab8bd130fp-672,
                0x1.de449d4e1295ep-239,
                0x1.de449d4e1295ep-240,
                0x1.de4644481a6cp-192,
                0x1.de4f8fecc0e57p-430,
                0x1.de5be724c122fp253,
                0x1.de5eb1859778p+476,
                0x1.de5eb1859778p477,
                0x1.de6815302e556p-651,
                0x1.de6815302e556p-652,
                0x1.de6a215c321b4p-881,
                0x1.de6a215c321b4p-882,
                0x1.de7308d376497p-971,
                0x1.de83fb2a83efap-349,
                0x1.de887cfbdc94p-455,
                0x1.de887cfbdc94p-456,
                0x1.de887cfbdc94p-457,
                0x1.de8926676168cp830,
                0x1.de91338d46f64p-970,
                0x1.de959671c8e38p-567,
                0x1.dea28acaef0bcp261,
                0x1.dea28acaef0bcp262,
                0x1.dea28acaef0bcp263,
                0x1.dea68b1f397b4p1019,
                0x1.dea68b1f397b4p+1020,
                0x1.dea96a37ad1dep-323,
                0x1.dea96a37ad1dep-324,
                0x1.dea96a37ad1dep-325,
                0x1.deaf8dc0b2decp-266,
                0x1.deb90d143ce2ap+288,
                0x1.debcd30b38727p105,
                0x1.debff00ca2c6cp-357,
                0x1.dec1cd0b7ebd9p-799,
                0x1.dec1f1a91da02p+752,
                0x1.ded797b907cf4p774,
                0x1.ded797b907cf4p775,
                0x1.ded797b907cf4p+776,
                0x1.ded797b907cf4p777,
                0x1.ded797b907cf4p778,
                0x1.ded797b907cf4p779,
                0x1.dedbc994c5e04p-432,
                0x1.dee7a4ad4b81ep-167,
                0x1.dee7a4ad4b81fp-167,
                0x1.dee8bc73c6eap-283,
                0x1.df0ee00a2a6fap-173,
                0x1.df18c70dd708fp-214,
                0x1.df1f96da0f5c2p-328,
                0x1.df2b53922eb8ep-915,
                0x1.df2b53922eb8ep-916,
                0x1.df33b957f5134p194,
                0x1.df33b957f5134p195,
                0x1.df33b957f5134p+196,
                0x1.df362675e3623p-993,
                0x1.df38630c0ccd2p+236,
                0x1.df3b131172342p455,
                0x1.df409e2a2f47cp-488,
                0x1.df4170f0fdeccp123,
                0x1.df4170f0fdeccp+124,
                0x1.df4170f0fdeccp125,
                0x1.df64d656p-1043,
                0x1.df657f43d09eep261,
                0x1.df67562d8b362p318,
                0x1.df67562d8b362p319,
                0x1.df834fe28b792p+988,
                0x1.df834fe28b792p989,
                0x1.df834fe28b792p990,
                0x1.df86751aac6cp+1020,
                0x1.df8d248111b8ep-343,
                0x1.df8d248111b8ep-344,
                0x1.df8d248111b8ep-345,
                0x1.df8d248111b8ep-346,
                0x1.df8d248111b8ep-347,
                0x1.df957b5815d32p-90,
                0x1.df9c3403c6736p471,
                0x1.df9dd942a0914p358,
                0x1.df9dd942a0914p359,
                0x1.dfa5c9e9fbeaap573,
                0x1.dfabfed366a54p-420,
                0x1.dfabfed366a54p-421,
                0x1.dfabfed366a54p-422,
                0x1.dfabfed366a54p-423,
                0x1.dfb39eb87a881p-822,
                0x1.dfb958e19d6c5p+288,
                0x1.dfc06fb0fc6eep-434,
                0x1.dfc11fbf46087p+520,
                0x1.dfc11fbf46087p521,
                0x1.dfc11fbf46087p522,
                0x1.dfcdce4fb4312p554,
                0x1.dfd1d6f839d8cp-379,
                0x1.dfd1d6f839d8cp-380,
                0x1.dfd1d6f839d8cp-381,
                0x1.dfd1d6f839d8cp-382,
                0x1.dfdf8b04e80f9p-922,
                0x1.dfe729b9ff152p802,
                0x1.dfe729b9ff152p803,
                0x1.dfe729b9ff152p+804,
                0x1.dfe729b9ff152p805,
                0x1.dffce19615a2ap955,
                0x1.dp-1069,
                0x1.e010e0dce3db2p+748,
                0x1.e010e0dce3db2p749,
                0x1.e01c85b747cf3p674,
                0x1.e02a667d186b3p-304,
                0x1.e02a667d186b3p-305,
                0x1.e04996f0b0152p-101,
                0x1.e04996f0b0152p-102,
                0x1.e067be2f58d4ap-44,
                0x1.e0698e74cfa0cp285,
                0x1.e06d0e4cbb2d8p-518,
                0x1.e07320602056cp-1000,
                0x1.e07320602056cp-1001,
                0x1.e07320602056cp-993,
                0x1.e07320602056cp-994,
                0x1.e07320602056cp-995,
                0x1.e07320602056cp-996,
                0x1.e07320602056cp-997,
                0x1.e07320602056cp-998,
                0x1.e07320602056cp-999,
                0x1.e0770ca70b164p339,
                0x1.e0770ca70b164p+340,
                0x1.e0770ca70b164p341,
                0x1.e0770ca70b164p342,
                0x1.e07b27dd78b14p-848,
                0x1.e07b27dd78b14p-849,
                0x1.e07bc3988a8b4p-627,
                0x1.e0917906fc398p+188,
                0x1.e0917906fc398p189,
                0x1.e092a05afb928p+296,
                0x1.e0aa5767616cbp+704,
                0x1.e0aa5767616cbp705,
                0x1.e0acd34adba3ap647,
                0x1.e0b8dedbf1ba8p-54,
                0x1.e0b8dedbf1ba8p-55,
                0x1.e0b8dedbf1ba8p-56,
                0x1.e0b8dedbf1ba8p-57,
                0x1.e0b8dedbf1ba8p-58,
                0x1.e0c18a16e81a5p+724,
                0x1.e0c4490a5bf7ep-798,
                0x1.e0d0d512664b8p-37,
                0x1.e0e2069fc4b1ep-1013,
                0x1.e0e2069fc4b1ep-1014,
                0x1.e0e32e09b14bap973,
                0x1.e0f6401f0dc4cp+868,
                0x1.e0fb44f50586ep-363,
                0x1.e0fb44f50586ep-364,
                0x1.e0fbd7d5e6532p622,
                0x1.e10486c01d9e6p913,
                0x1.e1079675b0cccp-922,
                0x1.e115c83a3fe1p-97,
                0x1.e118a6p-1051,
                0x1.e11b33e07e0cp-382,
                0x1.e11b33e07e0cp-383,
                0x1.e11b33e07e0cp-384,
                0x1.e1266df02c0f4p-61,
                0x1.e12e333d09e18p-393,
                0x1.e1306a325fe3ap801,
                0x1.e131a3bff6b42p-736,
                0x1.e131a3bff6b42p-737,
                0x1.e131a3bff6b42p-738,
                0x1.e132a46b9282ep-621,
                0x1.e132a46b9282ep-622,
                0x1.e1330fc6600c4p+896,
                0x1.e1330fc6600c4p897,
                0x1.e13b1a7f3df64p613,
                0x1.e14d3e56a01cep691,
                0x1.e1501ae5a6ebp475,
                0x1.e1534c3e2e98ap577,
                0x1.e1534c3e2e98ap578,
                0x1.e1534c3e2e98ap579,
                0x1.e1534c3e2e98ap+580,
                0x1.e157782119434p-444,
                0x1.e157782119434p-445,
                0x1.e15aac58b78ccp475,
                0x1.e1680a6421187p-521,
                0x1.e16966bcd3d2fp615,
                0x1.e16ee5d60cf47p-784,
                0x1.e178010308be2p-318,
                0x1.e178010308be2p-319,
                0x1.e178010308be2p-320,
                0x1.e178010308be2p-321,
                0x1.e17b84357691bp122,
                0x1.e17b84357691cp122,
                0x1.e17b84357691cp123,
                0x1.e17cf84be3106p-84,
                0x1.e18ab2d23df6p-971,
                0x1.e18d6d1c6d916p-619,
                0x1.e19946a02f407p+660,
                0x1.e1a5f48eb69c8p+468,
                0x1.e1aec8a43689ap278,
                0x1.e1b13d9f88b9ep553,
                0x1.e1b13d9f88b9ep554,
                0x1.e1b256f23c49ep-190,
                0x1.e1b256f23c49ep-191,
                0x1.e1b5ac8da893bp-355,
                0x1.e1b6bc567a2e3p-843,
                0x1.e1b6bc567a2e3p-844,
                0x1.e1b7f612078f4p653,
                0x1.e1cff1p-1050,
                0x1.e1d96786fc948p-324,
                0x1.e1deb9a5492f4p682,
                0x1.e1e30b175a30ep-907,
                0x1.e1f7d2a5f4084p263,
                0x1.e1fbe5a7e7861p607,
                0x1.e1fbe5a7e7862p606,
                0x1.e1fbe5a7e7862p607,
                0x1.e1fbe5a7e7862p+608,
                0x1.e1fbe5a7e7862p609,
                0x1.e1fc7830f2d0ap+820,
                0x1.e20700b1003a1p-128,
                0x1.e220230747c3ap245,
                0x1.e22d50cfdce64p-72,
                0x1.e2335cffc142ep-298,
                0x1.e238e691bf75cp378,
                0x1.e238e691bf75cp379,
                0x1.e238e691bf75cp+380,
                0x1.e248778c3395bp+968,
                0x1.e26368p-1053,
                0x1.e263b82aa728cp910,
                0x1.e2735101012p-1029,
                0x1.e274ce91fc3cp-170,
                0x1.e274ce91fc3cp-171,
                0x1.e274ce91fc3cp-172,
                0x1.e274ce91fc3cp-173,
                0x1.e2785c3a2a20ap227,
                0x1.e27c27fc6331p-744,
                0x1.e2908814p-1044,
                0x1.e290e419a7712p-862,
                0x1.e294c3eec6a8ap+872,
                0x1.e29e9c56687fep-709,
                0x1.e29f04143d4d4p-394,
                0x1.e2a15dfa1cdc4p247,
                0x1.e2a15dfa1cdc4p+248,
                0x1.e2a7c65db17f7p-206,
                0x1.e2bebebafc25ep133,
                0x1.e2c306c81a44cp+264,
                0x1.e2c7410853044p-604,
                0x1.e2d24eeaa481cp-261,
                0x1.e2dcaa4115622p-62,
                0x1.e30493e281a52p775,
                0x1.e30493e281a52p+776,
                0x1.e30493e281a52p777,
                0x1.e3113363787f2p-557,
                0x1.e3113363787f2p-558,
                0x1.e3113363787f2p-559,
                0x1.e322a09b86931p-743,
                0x1.e3354c047fcbap834,
                0x1.e3384c5399ea6p-1023,
                0x1.e33e04da81b1ep-795,
                0x1.e33ed3362ffbdp397,
                0x1.e33f8621642e8p681,
                0x1.e33f8621642e8p682,
                0x1.e35d7f2f41d4cp302,
                0x1.e35d7f2f41d4cp303,
                0x1.e35d7f2f41d4cp+304,
                0x1.e35d7f2f41d4cp305,
                0x1.e35d7f2f41d4cp306,
                0x1.e36cd1be6eebcp987,
                0x1.e36cd1be6eebcp+988,
                0x1.e36cd1be6eebcp989,
                0x1.e36cd1be6eebcp990,
                0x1.e36f3d36909e3p-791,
                0x1.e374080b3b66ap722,
                0x1.e374080b3b66ap723,
                0x1.e374080b3b66ap+724,
                0x1.e374080b3b66ap725,
                0x1.e374080b3b66ap726,
                0x1.e37c8a196135p+692,
                0x1.e38723a5ce17cp-284,
                0x1.e392010175ee6p-72,
                0x1.e392010175ee6p-73,
                0x1.e392010175ee6p-74,
                0x1.e39f22418bc48p990,
                0x1.e3a19671f52bap-254,
                0x1.e3b81c38331d2p-465,
                0x1.e3be2e08826b2p-595,
                0x1.e3c14d6916ce9p971,
                0x1.e3c1cd806cfdcp-152,
                0x1.e3cbc9907fdc8p-289,
                0x1.e3cbc9907fdc8p-290,
                0x1.e3cbc9907fdc8p-291,
                0x1.e3cbc9907fdc8p-292,
                0x1.e3d30a1fe35ffp-381,
                0x1.e3d9c7d27a2d4p-116,
                0x1.e3d9c7d27a2d4p-117,
                0x1.e3dc4d7cff57dp-146,
                0x1.e3e6f5fd95186p494,
                0x1.e3ec0c98d70dep-106,
                0x1.e3f0b69c9fd98p130,
                0x1.e3f68d3bd905cp847,
                0x1.e404d82411058p525,
                0x1.e404d82411058p526,
                0x1.e404d82411058p527,
                0x1.e404d82411058p+528,
                0x1.e404d82411058p529,
                0x1.e412f0f768fadp411,
                0x1.e412f0f768faep410,
                0x1.e412f0f768faep411,
                0x1.e412f0f768faep+412,
                0x1.e412f0f768faep413,
                0x1.e41c185a11638p-486,
                0x1.e41c185a11638p-487,
                0x1.e421107f9b54cp745,
                0x1.e421107f9b54cp746,
                0x1.e42255d800df4p-413,
                0x1.e42255d800df4p-414,
                0x1.e429b9264fa94p-683,
                0x1.e42b9eeff038ap+272,
                0x1.e42b9eeff038ap273,
                0x1.e44706e4066acp-639,
                0x1.e44bde741ba24p+472,
                0x1.e45198badabf8p+716,
                0x1.e45198badabf8p717,
                0x1.e45198badabf8p718,
                0x1.e45198badabf8p719,
                0x1.e45198badabf8p+720,
                0x1.e45198badabf8p721,
                0x1.e45198badabf8p722,
                0x1.e45b7ee49e09p771,
                0x1.e4608ee65e274p+652,
                0x1.e464ff0136f3p782,
                0x1.e469e029f1cf7p-303,
                0x1.e46c6d82802eep+532,
                0x1.e4725cc8ed25cp461,
                0x1.e48226f716b54p-881,
                0x1.e4883a7343d88p-507,
                0x1.e4883a7343d88p-508,
                0x1.e48ab645824aep935,
                0x1.e48ce04ec5856p993,
                0x1.e48f731b61d19p+416,
                0x1.e494034e79e5cp+896,
                0x1.e494034e79e5cp897,
                0x1.e4a8cec0068a6p1011,
                0x1.e4b0bfc208e2ep619,
                0x1.e4b2389f131c5p811,
                0x1.e4b44b3b5049ep-399,
                0x1.e4b44b3b5049ep-400,
                0x1.e4b44b3b5049ep-401,
                0x1.e4b44b3b5049ep-402,
                0x1.e4b44b3b5049ep-403,
                0x1.e4b44b3b5049ep-404,
                0x1.e4b53aeb46debp187,
                0x1.e4bd3e0d4cfbp539,
                0x1.e4bd3e0d4cfbp+540,
                0x1.e4bd3e0d4cfbp541,
                0x1.e4bd3e0d4cfbp542,
                0x1.e4bd3e0d4cfbp543,
                0x1.e4bd3e0d4cfbp+544,
                0x1.e4bd3e0d4cfbp545,
                0x1.e4bd3e0d4cfbp546,
                0x1.e4bf343e6a62ap391,
                0x1.e4bf343e6a62ap+392,
                0x1.e4bf343e6a62ap393,
                0x1.e4bf343e6a62ap394,
                0x1.e4bf343e6a62ap395,
                0x1.e4c3e94e3982cp817,
                0x1.e4c3e94e3982cp818,
                0x1.e4c3e94e3982cp819,
                0x1.e4c3e94e3982cp+820,
                0x1.e4c3e94e3982cp821,
                0x1.e4c61612eabcp-756,
                0x1.e4c61612eabcp-757,
                0x1.e4c61612eabcp-758,
                0x1.e4c61612eabcp-759,
                0x1.e4e021a11a2f1p-590,
                0x1.e4e78d99f9d94p-117,
                0x1.e4e9d3d0a58bbp143,
                0x1.e4f500cff5364p-676,
                0x1.e4f650e061b3p377,
                0x1.e4fd8682766ep951,
                0x1.e4fe5cd201b5ap149,
                0x1.e506653d93b14p-994,
                0x1.e506653d93b14p-995,
                0x1.e506653d93b14p-996,
                0x1.e50f3b8000973p-128,
                0x1.e5160a78f0b96p-749,
                0x1.e519fc495b10ap-804,
                0x1.e51c355f248e6p-194,
                0x1.e528bf5efa60cp-147,
                0x1.e528bf5efa60cp-148,
                0x1.e5297287c2f45p-755,
                0x1.e5297287c2f46p-752,
                0x1.e5297287c2f46p-753,
                0x1.e5297287c2f46p-754,
                0x1.e5297287c2f46p-755,
                0x1.e5297287c2f46p-756,
                0x1.e52ba827827bfp303,
                0x1.e52e52f6da42ap582,
                0x1.e52e52f6da42ap583,
                0x1.e544fa7f7bbb4p438,
                0x1.e5476f2a7fb1p-217,
                0x1.e548cc3492e59p586,
                0x1.e559c67f700dcp-300,
                0x1.e56899011f2bap+744,
                0x1.e56899011f2bap745,
                0x1.e574b8811c0cbp-267,
                0x1.e57c3ade46f62p665,
                0x1.e57c3ade46f62p666,
                0x1.e580659888448p-1025,
                0x1.e58d3023b724p-125,
                0x1.e58d3023b724p-126,
                0x1.e58dcee6f4938p+256,
                0x1.e58ffa48f4fcep-982,
                0x1.e58ffa48f4fcep-983,
                0x1.e58ffa48f4fcep-984,
                0x1.e58ffa48f4fcep-985,
                0x1.e58ffa48f4fcep-986,
                0x1.e596c65a173bep+524,
                0x1.e597224d59ba5p-497,
                0x1.e597224d59ba5p-498,
                0x1.e597c0b94b7aep+452,
                0x1.e597c0b94b7aep453,
                0x1.e597c0b94b7aep454,
                0x1.e59e46f8fcd6p-286,
                0x1.e59e46f8fcd6p-287,
                0x1.e5a8a5d4f4a3dp739,
                0x1.e5aacf2156838p-270,
                0x1.e5b55828953c6p+656,
                0x1.e5b6d92d124c4p+476,
                0x1.e5b6d92d124c4p477,
                0x1.e5b6d92d124c4p478,
                0x1.e5b6d92d124c4p479,
                0x1.e5b83eac07e3cp837,
                0x1.e5bd476cea8a3p-103,
                0x1.e5bf25836515cp-354,
                0x1.e5c56f4e679f6p-208,
                0x1.e5e6da3f8578ap806,
                0x1.e5f0938fdc83ep-61,
                0x1.e5f0938fdc83ep-62,
                0x1.e5f0938fdc83ep-63,
                0x1.e5f0938fdc83ep-64,
                0x1.e5f16268199fcp1001,
                0x1.e5f4527ac5f56p-606,
                0x1.e5f6b1p-1050,
                0x1.e5fb2e0aefa4p-1028,
                0x1.e60980eb2dd36p+920,
                0x1.e60bad57198b1p-105,
                0x1.e61f5ea0439eep-224,
                0x1.e623bf85d0d1fp267,
                0x1.e62499fba699ap-752,
                0x1.e6282c9bfb4eap-805,
                0x1.e62c4e38ff872p214,
                0x1.e62c4e38ff872p215,
                0x1.e6349db6a21d8p-921,
                0x1.e6394a302cbd8p-911,
                0x1.e6394a302cbd8p-912,
                0x1.e6394a302cbd8p-913,
                0x1.e6394a302cbd8p-914,
                0x1.e6394a302cbd8p-915,
                0x1.e6536f12a9abp774,
                0x1.e6563a828daadp-67,
                0x1.e665360539704p-516,
                0x1.e665360539704p-517,
                0x1.e665360539704p-518,
                0x1.e665360539704p-519,
                0x1.e665360539704p-520,
                0x1.e66816f5dff5ap-721,
                0x1.e66c16bad73ddp957,
                0x1.e672208f25ae2p-642,
                0x1.e6734d604d116p845,
                0x1.e6734d604d116p846,
                0x1.e6734d604d116p847,
                0x1.e6734d604d116p+848,
                0x1.e6734d604d116p849,
                0x1.e6737c422e04ep-631,
                0x1.e6797d38da244p-620,
                0x1.e684f9cb5d124p998,
                0x1.e688e6a0dd744p-426,
                0x1.e688e6a0dd744p-427,
                0x1.e688e6a0dd744p-428,
                0x1.e688e6a0dd744p-429,
                0x1.e68c07e6a6602p-464,
                0x1.e68d9a1522efp-527,
                0x1.e68d9a1522efp-528,
                0x1.e69615578ad8ep+472,
                0x1.e69615578ad8ep473,
                0x1.e69615578ad8ep474,
                0x1.e69e13209297ap-191,
                0x1.e69f8567fcfc4p-290,
                0x1.e6aaf37d9ffb8p365,
                0x1.e6adefd7f06aap+700,
                0x1.e6adefd7f06aap701,
                0x1.e6b1a754d5158p-118,
                0x1.e6b1a754d5158p-119,
                0x1.e6b2921fe909p+720,
                0x1.e6b2921fe909p721,
                0x1.e6b7e77207068p697,
                0x1.e6cb3f54a5cbdp-415,
                0x1.e6cb3f54a5cbdp-416,
                0x1.e6d3fc7e6ca18p-917,
                0x1.e6e1041d3a03cp-961,
                0x1.e6e46f4fba946p+976,
                0x1.e6e46f4fba946p977,
                0x1.e6eb75804f40fp+796,
                0x1.e6f8aeae8825dp-850,
                0x1.e6fad111bf346p+892,
                0x1.e702c23682ea6p-832,
                0x1.e71393227bce5p-135,
                0x1.e71423dd1da3ep-538,
                0x1.e71423dd1da3ep-539,
                0x1.e71423dd1da3ep-540,
                0x1.e71423dd1da3ep-541,
                0x1.e71423dd1da3ep-542,
                0x1.e71a64697ae3cp-187,
                0x1.e722e1813916bp+620,
                0x1.e7295a5f7542cp618,
                0x1.e7295a5f7542cp619,
                0x1.e7295a5f7542cp+620,
                0x1.e735b3003e352p455,
                0x1.e735b3003e352p+456,
                0x1.e73bb5a3cf69dp-205,
                0x1.e74123910300ep-97,
                0x1.e7416df9p-1041,
                0x1.e74404f3daadap-949,
                0x1.e74404f3daadap-950,
                0x1.e74404f3daadap-951,
                0x1.e74404f3daadap-952,
                0x1.e746e367600fbp-578,
                0x1.e749d6a738013p-858,
                0x1.e75f9a3816c34p759,
                0x1.e75f9a3816c34p+760,
                0x1.e769eb33d2174p970,
                0x1.e76f222c3ffaep-854,
                0x1.e77551820365ap475,
                0x1.e77551820365ap+476,
                0x1.e7877301c3ebap-581,
                0x1.e7c5f127bd87ep-466,
                0x1.e7c84b7169ca4p-506,
                0x1.e7d5238b5bc7ap529,
                0x1.e7d908eb1caabp766,
                0x1.e7e4ef973fabfp+476,
                0x1.e7e4ef973fabfp477,
                0x1.e7e72f02b7172p391,
                0x1.e7ed0bffc9524p+992,
                0x1.e7f03cp-1050,
                0x1.e7f298222b15ap318,
                0x1.e80933989914ap-713,
                0x1.e80b6d8e19215p210,
                0x1.e80e2510c0d18p-610,
                0x1.e81098732d7c8p-464,
                0x1.e81098732d7c8p-465,
                0x1.e81c169d9158p-304,
                0x1.e81c169d9158p-305,
                0x1.e82a007e40d26p+720,
                0x1.e82fa05f96b8cp198,
                0x1.e83c91008cad4p338,
                0x1.e842ce4792a24p+336,
                0x1.e848p19,
                0x1.e850f94735777p-924,
                0x1.e8548dac7bf24p474,
                0x1.e8563ef4b61aep354,
                0x1.e85c351fa50dcp713,
                0x1.e85c351fa50dcp714,
                0x1.e861f0a96532ep746,
                0x1.e861f0a96532ep747,
                0x1.e861f0a96532ep+748,
                0x1.e861f0a96532ep749,
                0x1.e861f0a96532ep750,
                0x1.e861f0a96532ep751,
                0x1.e86c2c3e84d99p575,
                0x1.e875f2429d674p-366,
                0x1.e87c89eb5a57fp-470,
                0x1.e87c89eb5a57fp-471,
                0x1.e87eaa026d23cp947,
                0x1.e87eaa026d23cp+948,
                0x1.e88765e9d4d79p-663,
                0x1.e893726324655p222,
                0x1.e89b9ea6ff6eep-583,
                0x1.e8aa8823a5db4p735,
                0x1.e8aa8823a5db4p+736,
                0x1.e8c27700ec7b8p-668,
                0x1.e8ca10a2b0387p-722,
                0x1.e8ca3185deb72p+504,
                0x1.e8ca3185deb72p505,
                0x1.e8d8fccf9183ep187,
                0x1.e8e05094b522bp259,
                0x1.e8e84c7452b08p-717,
                0x1.e8e99474f12ccp+208,
                0x1.e8f58ddcb3dcfp-375,
                0x1.e8f8ab74d137ap-794,
                0x1.e8f8ab74d137ap-795,
                0x1.e8f8ab74d137ap-796,
                0x1.e8f8ab74d137ap-797,
                0x1.e8f8ab74d137ap-798,
                0x1.e8fc0f7902514p-686,
                0x1.e8p-1069,
                0x1.e9081ee3b22ccp-436,
                0x1.e9081ee3b22ccp-437,
                0x1.e90bb1e34c3f2p-764,
                0x1.e90bb1e34c3f2p-765,
                0x1.e90bb1e34c3f2p-766,
                0x1.e90bb1e34c3f2p-767,
                0x1.e90be789ba98ep+296,
                0x1.e91315a09edbap-698,
                0x1.e913e48d4cd2ap-141,
                0x1.e913e48d4cd2ap-142,
                0x1.e91b0407ea00bp-556,
                0x1.e921198cb6924p497,
                0x1.e93eacf992f94p-515,
                0x1.e93eacf992f94p-516,
                0x1.e93f50d5cad0ap-824,
                0x1.e940284b5d4ccp741,
                0x1.e940284b5d4ccp742,
                0x1.e94c85c298c4cp989,
                0x1.e94c85c298c4cp990,
                0x1.e94d6fa5cd318p-732,
                0x1.e94d6fa5cd318p-733,
                0x1.e961dd80b8edap278,
                0x1.e961dd80b8edap279,
                0x1.e965d8856739cp190,
                0x1.e9687272c2205p-900,
                0x1.e96a930297eep-903,
                0x1.e96bd62b0f7dep503,
                0x1.e96bd62b0f7dep+504,
                0x1.e978b7780b614p-575,
                0x1.e978b7780b614p-576,
                0x1.e97d8c27e1d47p+968,
                0x1.e9884a659a352p+752,
                0x1.e99266d9a47fbp498,
                0x1.e99266d9a47fbp499,
                0x1.e992e5c477d49p-153,
                0x1.e994c37ca85d8p-895,
                0x1.e994c37ca85d8p-896,
                0x1.e994c37ca85d8p-897,
                0x1.e994c37ca85d8p-898,
                0x1.e9971cc40f256p-337,
                0x1.e997b6b245a98p-219,
                0x1.e998149c1ee86p-480,
                0x1.e9a4f84234925p+1020,
                0x1.e9a89fdb74908p605,
                0x1.e9bb7bd9ea0e1p-1022,
                0x1.e9c3ccbe07468p857,
                0x1.e9cdcee56dd92p-426,
                0x1.e9cdcee56dd92p-427,
                0x1.e9cdcee56dd92p-428,
                0x1.e9cec176c96f8p118,
                0x1.e9dcf149ba322p-892,
                0x1.e9e0ceb2e4f88p-341,
                0x1.e9e0ceb2e4f88p-342,
                0x1.e9e369aa2b597p-662,
                0x1.e9e3c5651b9eep-321,
                0x1.e9ea73d565776p462,
                0x1.e9eaa81e1c14fp637,
                0x1.e9eddbbb259b4p-823,
                0x1.e9eddbbb259b4p-824,
                0x1.e9eddbbb259b4p-825,
                0x1.e9ee2d6c6e69cp173,
                0x1.e9ee2d6c6e69cp174,
                0x1.e9f8d9aae568ap-935,
                0x1.ea02119dcdb51p+816,
                0x1.ea1306016d0bap474,
                0x1.ea1306016d0bap475,
                0x1.ea1428p-1053,
                0x1.ea1aca349cp-1036,
                0x1.ea2039a1fa248p853,
                0x1.ea28424e9efebp-203,
                0x1.ea2ce673662e6p854,
                0x1.ea3398dbe6024p255,
                0x1.ea39e56693c2p235,
                0x1.ea3a123ce8d43p-160,
                0x1.ea42fd42c416ap515,
                0x1.ea438cf5eac46p-86,
                0x1.ea438cf5eac46p-87,
                0x1.ea438cf5eac46p-88,
                0x1.ea438cf5eac46p-89,
                0x1.ea438cf5eac46p-90,
                0x1.ea438cf5eac46p-91,
                0x1.ea510bf41959cp-742,
                0x1.ea537336690dcp279,
                0x1.ea6608e29b24cp-177,
                0x1.ea8642d59a6e1p565,
                0x1.ea8642d59a6e1p566,
                0x1.ea886eab761bcp821,
                0x1.ea8c4ef73811ep-696,
                0x1.ea97b9ebf4af8p-774,
                0x1.ea996a67dccf2p841,
                0x1.eaab707d7621p482,
                0x1.eaab707d7621p483,
                0x1.eab60bac34747p-405,
                0x1.eabadbb1d5062p882,
                0x1.eac0ed4f8c9f2p+216,
                0x1.ead00a57f4a72p+480,
                0x1.ead0d3fe809fep+164,
                0x1.eae8caef261acp+308,
                0x1.eae8caef261acp309,
                0x1.eae9177c67af8p-860,
                0x1.eae9177c67af8p-861,
                0x1.eae9177c67af8p-862,
                0x1.eae9177c67af8p-863,
                0x1.eae9177c67af8p-864,
                0x1.eae9177c67af8p-865,
                0x1.eaea13d8e8c2p262,
                0x1.eaf55d87097edp-251,
                0x1.eb023308281f9p-491,
                0x1.eb0f603bbc9b4p-505,
                0x1.eb1575e9406fcp597,
                0x1.eb1575e9406fcp598,
                0x1.eb1575e9406fcp599,
                0x1.eb1575e9406fcp+600,
                0x1.eb1575e9406fcp601,
                0x1.eb1575e9406fcp602,
                0x1.eb1862b639254p-426,
                0x1.eb1eabf3e1cf4p254,
                0x1.eb2397bfaece1p-912,
                0x1.eb344a229498ep-178,
                0x1.eb436b52eebb9p+748,
                0x1.eb44237004d9cp-140,
                0x1.eb65024edb73cp-60,
                0x1.eb65024edb73cp-61,
                0x1.eb65024edb73cp-62,
                0x1.eb65024edb73cp-63,
                0x1.eb6a26b62282ap806,
                0x1.eb6bafd91596bp+792,
                0x1.eb6bafd91596bp793,
                0x1.eb7845cb2a7fdp234,
                0x1.eb8225d4cedb6p483,
                0x1.eb8225d4cedb6p+484,
                0x1.eb8225d4cedb6p485,
                0x1.eb8e84fa0b278p-1008,
                0x1.eb8e84fa0b278p-1009,
                0x1.eb8e84fa0b278p-1010,
                0x1.eb8ece4e25b22p-925,
                0x1.eb8ece4e25b22p-926,
                0x1.eb923be455b4cp-818,
                0x1.eb95c330d66f4p-785,
                0x1.eb95c330d66f4p-786,
                0x1.eb96dd8497c1ep887,
                0x1.eb96dd8497c1ep+888,
                0x1.eba083fa12adap318,
                0x1.eba083fa12adap319,
                0x1.eba37ff572ed4p+532,
                0x1.eba6688cead2ep577,
                0x1.eba6688cead2ep578,
                0x1.eba6688cead2ep579,
                0x1.eba6688cead2ep+580,
                0x1.eba77c260562p923,
                0x1.eba90e740269ap-396,
                0x1.ebbe59c1a5f1p+476,
                0x1.ebbffa0194079p439,
                0x1.ebca8a16a5cdfp+252,
                0x1.ebce42e77738ap-982,
                0x1.ebce42e77738ap-983,
                0x1.ebddc6593c858p-584,
                0x1.ebddc6593c858p-585,
                0x1.ebe2a2ab3ea6p-165,
                0x1.ebe2a2ab3ea6p-166,
                0x1.ebe2a2ab3ea6p-167,
                0x1.ebe718f2250e2p953,
                0x1.ebe7c0b548988p+660,
                0x1.ebfa4445b9dccp-106,
                0x1.ebfa4445b9dccp-107,
                0x1.ec033b40fea93p-858,
                0x1.ec033b40fea94p-857,
                0x1.ec033b40fea94p-858,
                0x1.ec156a8cec974p-976,
                0x1.ec27efe8a377dp+284,
                0x1.ec352ba995bap538,
                0x1.ec3954361a607p937,
                0x1.ec4e755fabd79p-608,
                0x1.ec4f4ba4da94fp-874,
                0x1.ec4f4ba4da94fp-875,
                0x1.ec53088p-1047,
                0x1.ec55666d8f9ecp151,
                0x1.ec55666d8f9ecp+152,
                0x1.ec56abdbd101p931,
                0x1.ec636b8a1e4abp+548,
                0x1.ec6a9d2889aa6p-310,
                0x1.ec6a9d2889aa6p-311,
                0x1.ec6d65df15264p234,
                0x1.ec746fc6407f8p-881,
                0x1.ec76bb1d467d1p-354,
                0x1.ec77a4e4e4997p841,
                0x1.ec866b79e0cbap-371,
                0x1.ec866b79e0cbap-372,
                0x1.ec866b79e0cbap-373,
                0x1.ec866b79e0cbap-374,
                0x1.ec8b5edb7098ep+964,
                0x1.ec8ca1b1b4e8p1021,
                0x1.ec922478c0422p-320,
                0x1.ec922478c0422p-321,
                0x1.ec922478c0422p-322,
                0x1.ec9b08b15ac0fp138,
                0x1.ec9b08b15ac0fp139,
                0x1.ec9c244d4b01ap-1007,
                0x1.ec9c244d4b01ap-1008,
                0x1.ecb21067c216p-747,
                0x1.ecb8f6ef59d2ep-997,
                0x1.ecbe5afa2764cp-356,
                0x1.ecbf1e4d43b56p-241,
                0x1.ecc537209902p270,
                0x1.ecc8c20d3b82bp717,
                0x1.ecc9e86739fb4p-534,
                0x1.ecd3fca26b84ep+612,
                0x1.ecd3fca26b84ep613,
                0x1.ecd3fca26b84ep614,
                0x1.ecd6c5f186b8fp-778,
                0x1.ecdb4dcddf1eep966,
                0x1.ecdc45013a7dcp-376,
                0x1.ece49354f4eeap678,
                0x1.ece49354f4eeap679,
                0x1.ecf33becd5386p-658,
                0x1.ecf3d71d1c615p262,
                0x1.ecf75fe5a4daap-791,
                0x1.ecf75fe5a4daap-792,
                0x1.ecfb05659383ep-404,
                0x1.ed03d68bc52p-1025,
                0x1.ed03d68bc52p-1026,
                0x1.ed09bead87c03p+112,
                0x1.ed09bead87c04p111,
                0x1.ed09bead87c04p+112,
                0x1.ed09bead87c04p113,
                0x1.ed09bead87c04p114,
                0x1.ed09bead87c04p115,
                0x1.ed11480eb4dep265,
                0x1.ed11480eb4dep266,
                0x1.ed246faaf1fb8p+1012,
                0x1.ed246faaf1fb8p1013,
                0x1.ed246faaf1fb8p1014,
                0x1.ed246faaf1fb8p1015,
                0x1.ed301f44a226cp779,
                0x1.ed33a323575dfp-75,
                0x1.ed33a794a1d0ap125,
                0x1.ed39ea39fe6c7p1014,
                0x1.ed43956537978p+200,
                0x1.ed43f95143984p886,
                0x1.ed43f95143984p887,
                0x1.ed43f95143984p+888,
                0x1.ed446df26ef6ep583,
                0x1.ed57a00d3e6f1p463,
                0x1.ed68bbef2c615p-236,
                0x1.ed798765a92cp815,
                0x1.ed87261b55d85p233,
                0x1.ed87261b55d85p234,
                0x1.ed8c8b1cf9f38p-100,
                0x1.ed8d34e547314p597,
                0x1.ed8e3abb9c4eep165,
                0x1.ed8ff6ab4f3e6p465,
                0x1.ed8ff6ab4f3e6p466,
                0x1.ed8ff6ab4f3e6p467,
                0x1.ed8ff6ab4f3e6p+468,
                0x1.ed8ff6ab4f3e6p469,
                0x1.ed8ff6ab4f3e6p470,
                0x1.ed8ff6ab4f3e6p471,
                0x1.eda167b9c8b32p466,
                0x1.eda33c5fb627ep-723,
                0x1.edaae890bd6bp385,
                0x1.edaae890bd6bp386,
                0x1.edaae890bd6bp387,
                0x1.edaae890bd6bp+388,
                0x1.edb1c9ed7fd18p242,
                0x1.edb1c9ed7fd18p243,
                0x1.edb1c9ed7fd18p+244,
                0x1.edb1c9ed7fd18p245,
                0x1.edbd94cb50054p-566,
                0x1.edbd94cb50054p-567,
                0x1.edbd94cb50054p-568,
                0x1.edbd94cb50054p-569,
                0x1.edcd2fe0bcd3ep734,
                0x1.edceb43cf04dep-666,
                0x1.edd2a253169dp-512,
                0x1.edd2a253169dp-513,
                0x1.edd2a253169dp-514,
                0x1.edd2a253169dp-515,
                0x1.edd2a253169dp-516,
                0x1.edd2a253169dp-517,
                0x1.edd2a253169dp-518,
                0x1.edd2a253169dp-519,
                0x1.edd56d2d0f4a2p331,
                0x1.edeafaebce21p+352,
                0x1.edef8ef827c1cp-395,
                0x1.edf89ca7b101bp-954,
                0x1.ee060e9bd87dbp122,
                0x1.ee0ac87f86223p567,
                0x1.ee127b4231f49p-767,
                0x1.ee1593a550c05p607,
                0x1.ee1593a550c05p+608,
                0x1.ee2568p-1053,
                0x1.ee256p-1054,
                0x1.ee257p-1054,
                0x1.ee270e06ff2c1p-410,
                0x1.ee33d34a25602p-719,
                0x1.ee36f0f6dc832p-585,
                0x1.ee372d0ec3f02p-627,
                0x1.ee384f263f544p+240,
                0x1.ee3f370f63b1p1018,
                0x1.ee4c8bd34f2fcp213,
                0x1.ee4c8bd34f2fcp214,
                0x1.ee4c8bd34f2fcp215,
                0x1.ee54e94752ebap-45,
                0x1.ee56f41812b74p903,
                0x1.ee5781237979ap-729,
                0x1.ee58da41a5ec8p-653,
                0x1.ee58da41a5ec8p-654,
                0x1.ee811f7149b14p861,
                0x1.ee811f7149b14p862,
                0x1.ee8dba44a03d2p957,
                0x1.ee90544935d8p+196,
                0x1.ee90544935d8p197,
                0x1.ee95cf1b4df8ap-577,
                0x1.ee994ed83cfa6p718,
                0x1.ee994ed83cfa6p719,
                0x1.ee994ed83cfa6p+720,
                0x1.ee994ed83cfa6p721,
                0x1.ee994ed83cfa6p722,
                0x1.ee9c319f4f545p810,
                0x1.eea92a61c3118p-569,
                0x1.eead73a46745dp+856,
                0x1.eead73a46745dp857,
                0x1.eeae23423a232p767,
                0x1.eeae23423a232p+768,
                0x1.eeae23423a232p769,
                0x1.eeae23423a232p770,
                0x1.eeae23423a232p771,
                0x1.eeb5421817ee6p402,
                0x1.eeb5421817ee6p403,
                0x1.eeb92bb25db71p229,
                0x1.eebabe0957af3p166,
                0x1.eebabe0957af3p167,
                0x1.eebabe0957af3p+168,
                0x1.eebabe0957af3p169,
                0x1.eebabe0957af3p170,
                0x1.eec6ef8b76488p-143,
                0x1.eec753a6aa8eep-405,
                0x1.eed7b4483c82ep422,
                0x1.eed8376e386fdp-484,
                0x1.eedb8c9787abap-725,
                0x1.eedb8c9787abap-726,
                0x1.eedb8c9787abap-727,
                0x1.eedb8c9787abap-728,
                0x1.eedb8c9787abap-729,
                0x1.eeddc27cfb8cep-122,
                0x1.eee11507aa186p-944,
                0x1.eee2b8676a026p-568,
                0x1.eeebe9ea010f4p569,
                0x1.eeebe9ea010f4p570,
                0x1.eeebe9ea010f4p571,
                0x1.eeebe9ea010f4p+572,
                0x1.eef47ed993c73p+728,
                0x1.eef5e1f90ac34p-182,
                0x1.eef5e1f90ac34p-183,
                0x1.eef5e1f90ac34p-184,
                0x1.eef5e1f90ac34p-185,
                0x1.eef5e1f90ac34p-186,
                0x1.eef5e1f90ac34p-187,
                0x1.eef5e1f90ac34p-188,
                0x1.eef5e1f90ac34p-189,
                0x1.eef5e1f90ac34p-190,
                0x1.eef5e1f90ac34p-191,
                0x1.eef5e1f90ac34p-192,
                0x1.eef5e1f90ac34p-193,
                0x1.eef9fa5d85e5cp662,
                0x1.eef9fa5d85e5cp663,
                0x1.eefa8f45668dcp687,
                0x1.eefac7d04fd27p-725,
                0x1.eeff57768f88cp-345,
                0x1.eeff57768f88cp-346,
                0x1.eeff57768f88cp-347,
                0x1.eeff57768f88cp-348,
                0x1.eeff57768f88cp-349,
                0x1.ef0160ceb47e9p-685,
                0x1.ef0fe34b5ec1ap-142,
                0x1.ef199e4b8ac41p-790,
                0x1.ef1b3cfd5b81cp-180,
                0x1.ef1b3cfd5b81cp-181,
                0x1.ef2d0f5da7dd8p-83,
                0x1.ef2d0f5da7dd8p-84,
                0x1.ef2fe91f1002p-935,
                0x1.ef3052ef0a362p453,
                0x1.ef3052ef0a362p454,
                0x1.ef3052ef0a362p455,
                0x1.ef33f620536e4p203,
                0x1.ef39cae6acda5p973,
                0x1.ef49e95bdf9cap+644,
                0x1.ef49e95bdf9cap645,
                0x1.ef4e65c106b9p-677,
                0x1.ef4e65c106b9p-678,
                0x1.ef62d051abff2p910,
                0x1.ef62d051abff2p911,
                0x1.ef62d051abff2p+912,
                0x1.ef62d051abff2p913,
                0x1.ef63c2de897a7p530,
                0x1.ef63c2de897a7p531,
                0x1.ef654caa0d4d4p-345,
                0x1.ef696965cbf04p-24,
                0x1.ef696965cbf04p-25,
                0x1.ef696965cbf04p-26,
                0x1.ef696965cbf04p-27,
                0x1.ef696965cbf04p-28,
                0x1.ef696965cbf04p-29,
                0x1.ef696965cbf04p-30,
                0x1.ef6a476ea114p+992,
                0x1.ef73d94e572f6p-327,
                0x1.ef79d1b01ff28p286,
                0x1.ef7be2d4907d3p-481,
                0x1.ef7c67569b4bap649,
                0x1.ef7ce1abfa51p469,
                0x1.ef7ce1abfa51p470,
                0x1.ef7d350277665p453,
                0x1.ef890f5de4c86p-378,
                0x1.ef890f5de4c86p-379,
                0x1.ef891f312840bp822,
                0x1.ef8bbc1f06e62p745,
                0x1.ef8bbc1f06e62p746,
                0x1.ef94055c010cp837,
                0x1.ef9beaa7aa584p+692,
                0x1.ef9beaa7aa584p693,
                0x1.ef9beaa7aa584p694,
                0x1.ef9beaa7aa584p695,
                0x1.efa3fdb1c74a8p465,
                0x1.efb1178484134p401,
                0x1.efbaa693d73c8p235,
                0x1.efbe73470b5b1p335,
                0x1.efbe73470b5b1p+336,
                0x1.efe3419c83312p978,
                0x1.efe3419c83312p979,
                0x1.efe3419c83312p+980,
                0x1.efe36e14e5306p-223,
                0x1.efe36e14e5306p-224,
                0x1.efe36e14e5306p-225,
                0x1.efe36e14e5306p-226,
                0x1.ep-1066,
                0x1.ep-1071,
                0x1.f0031f1cadb32p-358,
                0x1.f00dc6be0a44ep-308,
                0x1.f00de54f30deap-911,
                0x1.f00de54f30deap-912,
                0x1.f00de54f30deap-913,
                0x1.f0138dfp-1046,
                0x1.f0168361768ap1013,
                0x1.f020cba0b433bp-230,
                0x1.f020e54cc55e2p574,
                0x1.f022c96a5f8d3p969,
                0x1.f024450ffc3cp506,
                0x1.f0256721db1dep505,
                0x1.f0266a36aeae4p-163,
                0x1.f02d70c989d16p-315,
                0x1.f03542dfb837p886,
                0x1.f038af59605bfp-56,
                0x1.f03b54e5ac5bcp-631,
                0x1.f03b54e5ac5bcp-632,
                0x1.f03b54e5ac5bcp-633,
                0x1.f041b4ae0429p567,
                0x1.f0426b44f6658p-96,
                0x1.f0431669c2fcbp-993,
                0x1.f0484ca3a4b16p-371,
                0x1.f0484ca3a4b16p-372,
                0x1.f0484ca3a4b16p-373,
                0x1.f04dcb7db377bp+496,
                0x1.f04ef12cb04cfp87,
                0x1.f055cb0b16dd6p410,
                0x1.f055cb0b16dd6p411,
                0x1.f0564601e5606p-253,
                0x1.f0564601e5606p-254,
                0x1.f0564fff020f6p-20,
                0x1.f0588bd108691p-1009,
                0x1.f059286a48ebp-679,
                0x1.f05b44d11b823p578,
                0x1.f064652e10a68p-363,
                0x1.f064652e10a68p-364,
                0x1.f064652e10a68p-365,
                0x1.f064652e10a68p-366,
                0x1.f06722c83cdd2p831,
                0x1.f06d9e2dcbaa5p+624,
                0x1.f06d9e2dcbaa5p625,
                0x1.f078c41eb223ap-475,
                0x1.f078c41eb223ap-476,
                0x1.f07cc71919196p974,
                0x1.f07cc71919196p975,
                0x1.f08267534db3dp-222,
                0x1.f0935f1c2e5f2p+188,
                0x1.f098cbc4da56p-889,
                0x1.f0a810cd42d17p-265,
                0x1.f0abfb466fa28p-718,
                0x1.f0b330e20bc6ap-204,
                0x1.f0b501e7bba15p-184,
                0x1.f0b501e7bba15p-185,
                0x1.f0bae8b720504p-433,
                0x1.f0bp-1057,
                0x1.f0bp-1058,
                0x1.f0ce4839198dap-764,
                0x1.f0ce4839198dap-765,
                0x1.f0db18b947652p-809,
                0x1.f0e3d22ae2cfcp119,
                0x1.f0e3d22ae2cfcp+120,
                0x1.f0ef5ced14defp501,
                0x1.f0fcecbf44f08p-817,
                0x1.f0fcecbf44f08p-818,
                0x1.f0fd5f09ce9d4p1013,
                0x1.f10a4a7fa51adp501,
                0x1.f10c8e65fe7bcp-705,
                0x1.f10c8e65fe7bcp-706,
                0x1.f10c8e65fe7bcp-707,
                0x1.f10f8fe21972bp-702,
                0x1.f11105e44929cp503,
                0x1.f11105e44929cp+504,
                0x1.f11105e44929cp505,
                0x1.f11105e44929cp506,
                0x1.f11e25b1c9db2p-310,
                0x1.f11ef3aee61c2p458,
                0x1.f12538123556bp497,
                0x1.f12538123556bp498,
                0x1.f1266b6c13333p+376,
                0x1.f129d2113258ep201,
                0x1.f129d2113258ep202,
                0x1.f12af3d74c196p642,
                0x1.f12af3d74c196p643,
                0x1.f13b5a55e59cbp1006,
                0x1.f1405ff599411p-394,
                0x1.f14727744c63ep-596,
                0x1.f14727744c63ep-597,
                0x1.f151a151d660ep-151,
                0x1.f152bf9f10e9p-280,
                0x1.f15f1723a6808p587,
                0x1.f161199746994p229,
                0x1.f16p-1063,
                0x1.f1717caf0bf46p258,
                0x1.f17600c9e60a5p503,
                0x1.f177a1f9a5f2ep-209,
                0x1.f17813653c45ep877,
                0x1.f17813653c45ep878,
                0x1.f17813653c45ep879,
                0x1.f17813653c45ep+880,
                0x1.f17813653c45ep881,
                0x1.f17813653c45ep882,
                0x1.f1818bb911974p+324,
                0x1.f1869e69f7d74p-750,
                0x1.f1869e69f7d74p-751,
                0x1.f1869e69f7d74p-752,
                0x1.f1869e69f7d74p-753,
                0x1.f1869e69f7d74p-754,
                0x1.f18af750d406ep-655,
                0x1.f18c5783cda94p870,
                0x1.f18f63cce0738p-580,
                0x1.f1968025e9ep-310,
                0x1.f19b420b5463cp+648,
                0x1.f1a026be809b2p-702,
                0x1.f1a026be809b2p-703,
                0x1.f1a5af52c5cb4p-40,
                0x1.f1b1960a3922ep1015,
                0x1.f1b1960a3922ep+1016,
                0x1.f1b4b6a986b0cp531,
                0x1.f1b4b6a986b0cp+532,
                0x1.f1b4b6a986b0cp533,
                0x1.f1bd397c18644p-446,
                0x1.f1c093e6427ebp347,
                0x1.f1cce5ff3a444p+640,
                0x1.f1ce3c887c26p810,
                0x1.f1ce3c887c26p811,
                0x1.f1ce3c887c26p+812,
                0x1.f1d165047c682p-639,
                0x1.f1d2f159b8169p-147,
                0x1.f1d75a5709c1ap205,
                0x1.f1eadb719513ep-468,
                0x1.f1eadb719513ep-469,
                0x1.f1eadb719513ep-470,
                0x1.f1eap-1059,
                0x1.f1fca4a6b735bp498,
                0x1.f1fdc1da587fbp-903,
                0x1.f1fdc1da587fbp-904,
                0x1.f20938e7414bcp702,
                0x1.f20ff62044c48p245,
                0x1.f217f3b7f8a4ep390,
                0x1.f225e285d9316p-889,
                0x1.f225e285d9316p-890,
                0x1.f225e285d9316p-891,
                0x1.f225e285d9316p-892,
                0x1.f225e285d9316p-893,
                0x1.f22bbc168f648p269,
                0x1.f23553ee0fe04p-142,
                0x1.f23b5bed5900ep-517,
                0x1.f23d7565c6bd2p-700,
                0x1.f25c186a6f04cp690,
                0x1.f25dc7a642282p519,
                0x1.f25e5a5c2p-1039,
                0x1.f264d8b4a563p-227,
                0x1.f270c3674e345p345,
                0x1.f275e33972f0ap-106,
                0x1.f275e33972f0ap-107,
                0x1.f275e33972f0ap-108,
                0x1.f275e33972f0ap-109,
                0x1.f27e76ed8472ep555,
                0x1.f27e76ed8472ep+556,
                0x1.f28bebb4052a3p-1005,
                0x1.f28bebb4052a3p-1006,
                0x1.f28fcdcf82198p342,
                0x1.f295cb52bf868p945,
                0x1.f297a5631ff41p475,
                0x1.f29ca0ff893b1p-82,
                0x1.f29ca0ff893b1p-83,
                0x1.f29ca0ff893b1p-84,
                0x1.f2a241ce37078p-1001,
                0x1.f2a5228eb7622p197,
                0x1.f2a8984491b9cp-604,
                0x1.f2a9fc145a5bep-861,
                0x1.f2a9fc145a5bep-862,
                0x1.f2a9fc145a5bep-863,
                0x1.f2abaca11e368p221,
                0x1.f2bf1b484e656p338,
                0x1.f2bf1b484e656p339,
                0x1.f2bf1b484e656p+340,
                0x1.f2bf1b484e656p341,
                0x1.f2bf1b484e656p342,
                0x1.f2c0fef98c0d6p+584,
                0x1.f2c0fef98c0d6p585,
                0x1.f2d38b75588fap-898,
                0x1.f2e679d61852bp-699,
                0x1.f2ebe5f33d7ecp647,
                0x1.f2f5c7a1a488ep-961,
                0x1.f30d41e22b3c8p514,
                0x1.f3121db5fb9f2p-984,
                0x1.f3121db5fb9f2p-985,
                0x1.f319a2d7d7f4ep913,
                0x1.f319a2d7d7f4ep914,
                0x1.f319a2d7d7f4ep915,
                0x1.f322bfa913cf1p-49,
                0x1.f3264f2847a7p-251,
                0x1.f328d58b70878p-515,
                0x1.f32a4509fa1p+520,
                0x1.f32a4509fa1p521,
                0x1.f32bb86d91288p867,
                0x1.f32feb11225b5p1021,
                0x1.f330cca276556p389,
                0x1.f3376a28ba622p-203,
                0x1.f36f8f1cp-1044,
                0x1.f37ad21436d0cp-474,
                0x1.f37ad21436d0cp-475,
                0x1.f37ad21436d0cp-476,
                0x1.f37ad21436d0cp-477,
                0x1.f3868626e3088p+636,
                0x1.f394c23d19dbep-903,
                0x1.f395d4c779d8ep-168,
                0x1.f395d4c779d8ep-169,
                0x1.f395d4c779d8ep-170,
                0x1.f395d4c779d8ep-171,
                0x1.f395d4c779d8ep-172,
                0x1.f395d4c779d8ep-173,
                0x1.f39920dbd6eap179,
                0x1.f39920dbd6eap+180,
                0x1.f39920dbd6eap181,
                0x1.f39bd3861c1dap362,
                0x1.f39bd3861c1dap363,
                0x1.f39bd3861c1dap+364,
                0x1.f3ab7dcfbaf3bp494,
                0x1.f3abced28001fp-926,
                0x1.f3b66b9795bacp963,
                0x1.f3cbdc494bf4bp507,
                0x1.f3e53b3e4623cp478,
                0x1.f3ed99c992dap323,
                0x1.f3f44ac1132p+328,
                0x1.f3fbb3f9c173cp-659,
                0x1.f3fbb3f9c173cp-660,
                0x1.f3fbb3f9c173cp-661,
                0x1.f4031fd512b36p+324,
                0x1.f4031fd512b36p325,
                0x1.f4031fd512b36p326,
                0x1.f4031fd512b36p327,
                0x1.f40be8cf0346p-46,
                0x1.f4173419fbe33p501,
                0x1.f41c7570ab5cap406,
                0x1.f429cb67eb076p127,
                0x1.f429cb67eb076p+128,
                0x1.f42f36fa589a6p-789,
                0x1.f42f36fa589a6p-790,
                0x1.f43f86c430984p-530,
                0x1.f446dce228b66p-834,
                0x1.f44c669e7d3aap-445,
                0x1.f44c669e7d3aap-446,
                0x1.f45338cda4248p-1013,
                0x1.f46740c5a5daap929,
                0x1.f46740c5a5daap930,
                0x1.f46740c5a5daap931,
                0x1.f46740c5a5daap+932,
                0x1.f46740c5a5daap933,
                0x1.f46a1c8bfc138p-789,
                0x1.f46a1c8bfc138p-790,
                0x1.f476f78b5add4p+168,
                0x1.f479fa899fbe6p-147,
                0x1.f47f25835bf94p-202,
                0x1.f47f25835bf94p-203,
                0x1.f47f25835bf94p-204,
                0x1.f47f25835bf94p-205,
                0x1.f47f25835bf94p-206,
                0x1.f47f25835bf94p-207,
                0x1.f482efed9fba4p-815,
                0x1.f485516e7577ep493,
                0x1.f485516e7577ep494,
                0x1.f485516e7577ep495,
                0x1.f485516e7577ep+496,
                0x1.f4866f39796cp846,
                0x1.f4895f9dbf87p-799,
                0x1.f4895f9dbf87p-800,
                0x1.f4907090643fep-541,
                0x1.f49072921c9ccp-1013,
                0x1.f499e026c418dp737,
                0x1.f49cf3420f7c4p+456,
                0x1.f4a17e3d81183p417,
                0x1.f4a1bfc8061cfp942,
                0x1.f4bfc6b97c3d6p-206,
                0x1.f4c07ad6427eep+684,
                0x1.f4c302fc19202p-907,
                0x1.f4f4b7059e066p-962,
                0x1.f4f4b7059e066p-963,
                0x1.f4f73ac65b28bp-269,
                0x1.f4p9,
                0x1.f50387c3bb171p-155,
                0x1.f505065539d78p-769,
                0x1.f505065539d78p-770,
                0x1.f505065539d78p-771,
                0x1.f507e8fc439c5p-332,
                0x1.f50ac6690f1f8p979,
                0x1.f50c5c63e5442p-56,
                0x1.f50d92e5317p849,
                0x1.f5136822a1ee8p446,
                0x1.f519e3e7a9066p642,
                0x1.f52597d95d3cp633,
                0x1.f52597d95d3cp634,
                0x1.f52597d95d3cp635,
                0x1.f53353c7c4decp515,
                0x1.f53feb5f9c45ep715,
                0x1.f53feb5f9c45ep+716,
                0x1.f53feb5f9c45ep717,
                0x1.f543ae43fd132p718,
                0x1.f548f2fdc85bep970,
                0x1.f548f2fdc85bep971,
                0x1.f54c0dde1accep-428,
                0x1.f54d694008c04p+988,
                0x1.f5543b23c1edfp-772,
                0x1.f555c1226c5a3p609,
                0x1.f566d9c68d5bep883,
                0x1.f5685d7145c62p122,
                0x1.f5685d7145c62p123,
                0x1.f568fc38033d9p-199,
                0x1.f5842f54d98a8p-932,
                0x1.f5842f54d98a8p-933,
                0x1.f5842f54d98a8p-934,
                0x1.f587880236c9bp994,
                0x1.f58ac4db68c9p121,
                0x1.f58ac4db68c9p122,
                0x1.f58b225ad5912p873,
                0x1.f58df091490fp-545,
                0x1.f5a18504dfaacp-210,
                0x1.f5a18504dfaacp-211,
                0x1.f5a18504dfaacp-212,
                0x1.f5a18504dfaacp-213,
                0x1.f5a18504dfaacp-214,
                0x1.f5a18504dfaacp-215,
                0x1.f5a18504dfaacp-216,
                0x1.f5a18504dfaacp-217,
                0x1.f5a549627a36cp-672,
                0x1.f5ae07ab5c968p-153,
                0x1.f5b4d3201c80ap-410,
                0x1.f5b81e81909b6p-217,
                0x1.f5b81e81909b6p-218,
                0x1.f5c09fa7c4c11p623,
                0x1.f5c09fa7c4c11p+624,
                0x1.f5c09fa7c4c11p625,
                0x1.f5c4d90ecab0cp701,
                0x1.f5c4d90ecab0cp702,
                0x1.f5cd53504005ep-488,
                0x1.f5cd53504005ep-489,
                0x1.f5d6a08105bc6p+684,
                0x1.f5db8c32b02fcp-954,
                0x1.f5dbc67a22a0bp-735,
                0x1.f5dec98b61fd2p-459,
                0x1.f5e6e6abdb198p+236,
                0x1.f5e6e6abdb198p237,
                0x1.f5e6e6abdb198p238,
                0x1.f5e6e6abdb198p239,
                0x1.f5e7906fa926dp-392,
                0x1.f5e7906fa926dp-393,
                0x1.f5f8bb61e051ep-283,
                0x1.f60b4a930ae18p255,
                0x1.f60b4a930ae18p+256,
                0x1.f60b8c22f5447p-500,
                0x1.f62b0b257c0d2p-186,
                0x1.f62b0b257c0d2p-187,
                0x1.f6387b7d94764p-456,
                0x1.f63b0354519d2p-1011,
                0x1.f63b0354519d2p-1012,
                0x1.f64a7d5606b8bp494,
                0x1.f64af8f3259b4p278,
                0x1.f64af8f3259b4p279,
                0x1.f64af8f3259b4p+280,
                0x1.f64af8f3259b4p281,
                0x1.f64df24988686p310,
                0x1.f653afd8fd734p-651,
                0x1.f655acccb316p1023,
                0x1.f65debb2ec232p191,
                0x1.f65debb2ec232p+192,
                0x1.f663cf0b4a1dcp-562,
                0x1.f66424e2c32bep-146,
                0x1.f66424e2c32bep-147,
                0x1.f669c387c402p515,
                0x1.f6768b6321b82p513,
                0x1.f6768b6321b82p514,
                0x1.f67edfb5882a3p-685,
                0x1.f68a744678eaap-432,
                0x1.f68b53819959fp-807,
                0x1.f693b9143d007p-666,
                0x1.f6b0f092959c7p298,
                0x1.f6b0f092959c8p298,
                0x1.f6b23cfe98806p-267,
                0x1.f6b23cfe98806p-268,
                0x1.f6b23cfe98806p-269,
                0x1.f6b23cfe98806p-270,
                0x1.f6b8676a11bc1p505,
                0x1.f6beb27dad4c5p-812,
                0x1.f6beb27dad4c5p-813,
                0x1.f6c64d3e9d05p-1025,
                0x1.f6c64d3e9d05p-1026,
                0x1.f6cacd184c272p-11,
                0x1.f6caf4c4c4bcp-500,
                0x1.f6caf4c4c4bcp-501,
                0x1.f6dbb330e7d1fp-954,
                0x1.f6debb467567ep-130,
                0x1.f6df04b31a01ap515,
                0x1.f6e67d4de0bcbp138,
                0x1.f6e67d4de0bcbp139,
                0x1.f70237ae04c6cp373,
                0x1.f7148c454706ap-782,
                0x1.f72af2ddd6371p457,
                0x1.f732408775212p586,
                0x1.f732408775212p587,
                0x1.f736f9b3494e8p782,
                0x1.f736f9b3494e8p783,
                0x1.f736f9b3494e8p+784,
                0x1.f736f9b3494e8p785,
                0x1.f736f9b3494e8p786,
                0x1.f73a18468413ep331,
                0x1.f73a18468413ep+332,
                0x1.f73c44dc2835ep-932,
                0x1.f746182e6cd5dp861,
                0x1.f746182e6cd5dp862,
                0x1.f75d9771f1e39p581,
                0x1.f76330a88b04p877,
                0x1.f764dad220566p-615,
                0x1.f764dad220566p-616,
                0x1.f764dad220566p-617,
                0x1.f764dad220566p-618,
                0x1.f76fdbe29623ap407,
                0x1.f76fdbe29623ap+408,
                0x1.f76fdbe29623ap409,
                0x1.f76fdbe29623ap410,
                0x1.f76fdbe29623ap411,
                0x1.f7712457afe73p174,
                0x1.f7731b300b156p+568,
                0x1.f7731b300b156p569,
                0x1.f77dffbf52a62p-513,
                0x1.f77dffbf52a62p-514,
                0x1.f77dffbf52a62p-515,
                0x1.f79bea65da56ap-992,
                0x1.f7b6cb9fe73ep909,
                0x1.f7b6cb9fe73ep910,
                0x1.f7b6cb9fe73ep911,
                0x1.f7b6cb9fe73ep+912,
                0x1.f7b6cb9fe73ep913,
                0x1.f7b6cb9fe73ep914,
                0x1.f7c3991afd58cp+632,
                0x1.f7cf7a72a5348p438,
                0x1.f7d011a904796p1001,
                0x1.f7d011a904796p1002,
                0x1.f7d011a904796p1003,
                0x1.f7d011a904796p+1004,
                0x1.f7d228322baf5p-868,
                0x1.f7d4d2fac58f2p489,
                0x1.f7d6721f7f144p+752,
                0x1.f7d6721f7f144p753,
                0x1.f7d6721f7f144p754,
                0x1.f7e0db3799aa2p-527,
                0x1.f7e0db3799aa2p-528,
                0x1.f7e0db3799aa2p-529,
                0x1.f7e0db3799aa2p-530,
                0x1.f7e0db3799aa2p-531,
                0x1.f7e0db3799aa2p-532,
                0x1.f7e0db3799aa2p-533,
                0x1.f7e0db3799aa2p-534,
                0x1.f7e0db3799aa2p-535,
                0x1.f7e8e8d51067bp-606,
                0x1.f7ea769cf0d4ep-82,
                0x1.f7ea769cf0d4ep-83,
                0x1.f7ed072f49e1dp742,
                0x1.f7ee39e3f4faep577,
                0x1.f7ee39e3f4faep578,
                0x1.f7f617eb3717ep893,
                0x1.f809abdbdc7cep+448,
                0x1.f80b720bf7328p-828,
                0x1.f81332aa392eep190,
                0x1.f816018dbadp-472,
                0x1.f816018dbadp-473,
                0x1.f81d8db83eca5p-973,
                0x1.f820641aef39cp133,
                0x1.f821ac63b89adp-257,
                0x1.f822c980d4bb2p98,
                0x1.f82671b4p-1044,
                0x1.f82c7cbd5ad24p+220,
                0x1.f83a32f69f129p-823,
                0x1.f83a32f69f129p-824,
                0x1.f83a32f69f129p-825,
                0x1.f840445f37368p722,
                0x1.f843198906349p274,
                0x1.f84e0b95f8c26p-967,
                0x1.f855123971214p-889,
                0x1.f855123971214p-890,
                0x1.f855123971214p-891,
                0x1.f855585f3f09cp-749,
                0x1.f856e67509b4cp-338,
                0x1.f85836b735c39p894,
                0x1.f8587e7083e3p-380,
                0x1.f8587e7083e3p-381,
                0x1.f8587e7083e3p-382,
                0x1.f8587e7083e3p-383,
                0x1.f874f6b0a22d2p-89,
                0x1.f87699298d41ap-459,
                0x1.f87699298d41ap-460,
                0x1.f87699298d41ap-461,
                0x1.f8918159fdf3ep-758,
                0x1.f891908a9d0bcp-488,
                0x1.f891908a9d0bcp-489,
                0x1.f891908a9d0bcp-490,
                0x1.f8a89dc374df5p-22,
                0x1.f8a89dc374df5p-23,
                0x1.f8d02a458540ep+308,
                0x1.f8d7eb2f2a644p510,
                0x1.f8d7eb2f2a644p511,
                0x1.f8dbb6bdef738p+492,
                0x1.f8dbb6bdef738p493,
                0x1.f8def8808b024p102,
                0x1.f8e9144666386p777,
                0x1.f8f45c64b4683p-26,
                0x1.f8f919941fe94p-697,
                0x1.f8fbf98fdceecp-507,
                0x1.f8fefbb3249d4p-437,
                0x1.f8fefbb3249d4p-438,
                0x1.f8fefbb3249d4p-439,
                0x1.f8fefbb3249d4p-440,
                0x1.f8fefbb3249d4p-441,
                0x1.f8fefbb3249d4p-442,
                0x1.f8fefbb3249d4p-443,
                0x1.f8fefbb3249d4p-444,
                0x1.f8p-1069,
                0x1.f9024dfb44cd8p241,
                0x1.f904104267e24p745,
                0x1.f914ab8d786c5p-541,
                0x1.f91860eb4dc85p142,
                0x1.f920d5709ecf6p-756,
                0x1.f92c906c177dap+676,
                0x1.f92c906c177dap677,
                0x1.f92c906c177dap678,
                0x1.f92c906c177dap679,
                0x1.f92c906c177dap+680,
                0x1.f92e6ab6521d3p257,
                0x1.f92ef3051bb2ap-71,
                0x1.f933d5cbd0473p-927,
                0x1.f93534842bea6p-225,
                0x1.f93534842bea6p-226,
                0x1.f938253a3240ep-432,
                0x1.f9506b25aa3cep-867,
                0x1.f95077f2e1001p-106,
                0x1.f9508f7b6cfb2p+776,
                0x1.f9508f7b6cfb2p777,
                0x1.f955a027de71ep881,
                0x1.f955a027de71ep882,
                0x1.f955a027de71ep883,
                0x1.f955a027de71ep+884,
                0x1.f955a027de71ep885,
                0x1.f95660c426069p1023,
                0x1.f965966bce056p587,
                0x1.f971ae43def64p-1018,
                0x1.f971ae43def64p-1019,
                0x1.f971ae43def64p-1020,
                0x1.f972c308efe4cp-243,
                0x1.f975c9d57536ep607,
                0x1.f975c9d57536ep+608,
                0x1.f9766b0b20398p134,
                0x1.f9b6f376af588p611,
                0x1.f9c0e8a64ee48p-66,
                0x1.f9c6bbfb2df26p-127,
                0x1.f9d13e6c8c67fp-349,
                0x1.f9d21249db244p-805,
                0x1.f9f3c003caeaap-192,
                0x1.f9fd0bbfa1366p-36,
                0x1.f9fd0bbfa1366p-37,
                0x1.fa08f911195dp-668,
                0x1.fa08f911195dp-669,
                0x1.fa0cdb5dcfb06p-340,
                0x1.fa0cdb5dcfb06p-341,
                0x1.fa0cdb5dcfb06p-342,
                0x1.fa0cdb5dcfb06p-343,
                0x1.fa0cdb5dcfb06p-344,
                0x1.fa0cdb5dcfb06p-345,
                0x1.fa0cdb5dcfb06p-346,
                0x1.fa11eb24abb51p869,
                0x1.fa11eb24abb51p870,
                0x1.fa147ba15b3b8p643,
                0x1.fa147ba15b3b8p+644,
                0x1.fa147ba15b3b8p645,
                0x1.fa147ba15b3b8p646,
                0x1.fa147ba15b3b8p647,
                0x1.fa2bb0f9ffe0ep398,
                0x1.fa2bb0f9ffe0ep399,
                0x1.fa2bb0f9ffe0ep+400,
                0x1.fa2ca0b814a01p221,
                0x1.fa2p-1063,
                0x1.fa370a4c95d76p-812,
                0x1.fa370a4c95d76p-813,
                0x1.fa370a4c95d76p-814,
                0x1.fa370a4c95d76p-815,
                0x1.fa39a20dcbae7p-485,
                0x1.fa3ce1ab6856p671,
                0x1.fa3ce1ab6856p+672,
                0x1.fa3ce1ab6856p673,
                0x1.fa3ce1ab6856p674,
                0x1.fa3ce1ab6856p675,
                0x1.fa3ce1ab6856p+676,
                0x1.fa3ce1ab6856p677,
                0x1.fa3ce1ab6856p678,
                0x1.fa3ce1ab6856p679,
                0x1.fa3ce1ab6856p+680,
                0x1.fa3ce1ab6856p681,
                0x1.fa3ce1ab6856p682,
                0x1.fa3ce1ab6856p683,
                0x1.fa3ce1ab6856p+684,
                0x1.fa410cdfddceep159,
                0x1.fa410cdfddceep+160,
                0x1.fa410cdfddceep161,
                0x1.fa410cdfddceep162,
                0x1.fa4b9340bbdcap717,
                0x1.fa5290b40286bp361,
                0x1.fa66e273c9ebbp+496,
                0x1.fa6b1b92f6fp771,
                0x1.fa6b1b92f6fp+772,
                0x1.fa7208ef04014p-416,
                0x1.fa7ebe304ee3ep-879,
                0x1.fa7ebe304ee3ep-880,
                0x1.fa7ebe304ee3ep-881,
                0x1.fa7ebe304ee3ep-882,
                0x1.fa7ebe304ee3ep-883,
                0x1.fa821e9e12586p-970,
                0x1.fa821e9e12586p-971,
                0x1.fa885c8d117a6p-579,
                0x1.fa8cad23faed8p-350,
                0x1.fa9e6adf0029bp898,
                0x1.faa7ab552a552p-41,
                0x1.faadb29c3265cp555,
                0x1.fab2c0a8e6ff9p-476,
                0x1.fab3ac38ecd24p-78,
                0x1.fab3ac38ecd24p-79,
                0x1.fabbbd9ad7ae2p-844,
                0x1.fabbbd9ad7ae2p-845,
                0x1.fabbc63f3528p817,
                0x1.fabc28ee5e317p+236,
                0x1.fabcc5e7ac02ap-356,
                0x1.fabcc5e7ac02ap-357,
                0x1.fabcc5e7ac02ap-358,
                0x1.fabcc5e7ac02ap-359,
                0x1.fabe6ac9d97a3p-618,
                0x1.fac740a8c9938p-747,
                0x1.fad08668980dbp-295,
                0x1.fad8p-1061,
                0x1.fae03ce03ae06p+164,
                0x1.fae5ce4883ecep-201,
                0x1.fae6ace717a14p+976,
                0x1.fae6ace717a14p977,
                0x1.faeeaa38e474cp841,
                0x1.faeeaa38e474cp842,
                0x1.faf4e50c38d4cp-1002,
                0x1.faf4e50c38d4cp-1003,
                0x1.faf739c3e35bp+544,
                0x1.faffa307ac342p-309,
                0x1.faffa307ac342p-310,
                0x1.fap-1064,
                0x1.fap-1065,
                0x1.fap-1066,
                0x1.fap-1067,
                0x1.fb0f6be506019p-94,
                0x1.fb0f6be50601ap-94,
                0x1.fb29a9fa1000ap-893,
                0x1.fb367fcf1d11cp-817,
                0x1.fb367fcf1d11cp-818,
                0x1.fb367fcf1d11cp-819,
                0x1.fb367fcf1d11cp-820,
                0x1.fb45b30e815c4p-131,
                0x1.fb45b30e815c4p-132,
                0x1.fb49140a1644fp-54,
                0x1.fb63551b069e8p-853,
                0x1.fb670e0f0c5a2p-789,
                0x1.fb670e0f0c5a2p-790,
                0x1.fb670e0f0c5a2p-791,
                0x1.fb7941267fda4p-163,
                0x1.fb7941267fda4p-164,
                0x1.fb7941267fda4p-165,
                0x1.fb8104e45e978p393,
                0x1.fb8104e45e978p394,
                0x1.fb8104e45e978p395,
                0x1.fb8104e45e978p+396,
                0x1.fb83b952d8096p+472,
                0x1.fb83b952d8096p473,
                0x1.fb83b952d8096p474,
                0x1.fb83b952d8096p475,
                0x1.fb83b952d8096p+476,
                0x1.fb83b952d8096p477,
                0x1.fb83b952d8096p478,
                0x1.fb83b952d8096p479,
                0x1.fb9074d564d4ep185,
                0x1.fb969f40042c5p391,
                0x1.fb969f40042c6p391,
                0x1.fb997e41e82c2p-865,
                0x1.fb997e41e82c2p-866,
                0x1.fb9ff80876248p-400,
                0x1.fba1ecc784813p-253,
                0x1.fba56e690097cp726,
                0x1.fba56e690097cp727,
                0x1.fba56e690097cp+728,
                0x1.fba760fb6b685p-189,
                0x1.fbab0bec28b7p595,
                0x1.fbb0e45456d22p-407,
                0x1.fbb0e45456d22p-408,
                0x1.fbb2976f346bp+296,
                0x1.fbbcb0e418ebp-977,
                0x1.fbbcb0e418ebp-978,
                0x1.fbbcb0e418ebp-979,
                0x1.fbbcb0e418ebp-980,
                0x1.fbbcb0e418ebp-981,
                0x1.fbc48ac17067cp545,
                0x1.fbc48ac17067cp546,
                0x1.fbe483bdc5854p-149,
                0x1.fbe483bdc5854p-150,
                0x1.fbead4b2d211ap-227,
                0x1.fc0df330f2239p-961,
                0x1.fc0e6ff853e38p169,
                0x1.fc1562f08f124p+152,
                0x1.fc1562f08f124p153,
                0x1.fc1562f08f124p154,
                0x1.fc1cdf2734f2bp569,
                0x1.fc1cf66c3e54p767,
                0x1.fc1df6a7a61bap+876,
                0x1.fc1df6a7a61bap877,
                0x1.fc287849a9e71p191,
                0x1.fc2fa2e7524cbp-632,
                0x1.fc3af88a74e54p-422,
                0x1.fc3ea477eefeep-897,
                0x1.fc3ea477eefeep-898,
                0x1.fc3eb4f139acap+160,
                0x1.fc438e4c9b2f2p-892,
                0x1.fc438e4c9b2f2p-893,
                0x1.fc47605c54ed6p-997,
                0x1.fc52044df12c5p169,
                0x1.fc575867314eep-330,
                0x1.fc575867314eep-331,
                0x1.fc575867314eep-332,
                0x1.fc575867314eep-333,
                0x1.fc575867314eep-334,
                0x1.fc5b9a1e1c144p981,
                0x1.fc6c26f899dd1p-948,
                0x1.fc6c26f899dd1p-949,
                0x1.fc6c26f899dd1p-950,
                0x1.fc73b333f1e6ap478,
                0x1.fc85eefe4b834p+912,
                0x1.fc85eefe4b834p913,
                0x1.fc8981b92c4c8p-406,
                0x1.fc8981b92c4c8p-407,
                0x1.fc95b658ebb4ep323,
                0x1.fc95b658ebb4ep+324,
                0x1.fc9d68a7a00b4p997,
                0x1.fc9d68a7a00b4p998,
                0x1.fc9e18a79fefp307,
                0x1.fca36c06cf106p+808,
                0x1.fca36c06cf106p809,
                0x1.fca36c06cf106p810,
                0x1.fca36c06cf106p811,
                0x1.fca36c06cf106p+812,
                0x1.fcbaa82a16121p-775,
                0x1.fcbaa82a16122p-774,
                0x1.fcbaa82a16122p-775,
                0x1.fcbfc53e05cf8p562,
                0x1.fcd7142aa1851p466,
                0x1.fcd93e3cdp-1035,
                0x1.fce791fb6c6d1p826,
                0x1.fce791fb6c6d1p827,
                0x1.fcec91d3d859ap-153,
                0x1.fcf24c947aac8p-761,
                0x1.fcf8bed020398p-41,
                0x1.fcfa4e68497ecp-244,
                0x1.fd0dc48573a98p+456,
                0x1.fd11f511fc5aep149,
                0x1.fd11f511fc5aep150,
                0x1.fd22da7d8d659p645,
                0x1.fd30eda805b8p-520,
                0x1.fd30eda805b8p-521,
                0x1.fd30eda805b8p-522,
                0x1.fd30eda805b8p-523,
                0x1.fd30eda805b8p-524,
                0x1.fd30eda805b8p-525,
                0x1.fd332f7e2e3b3p-380,
                0x1.fd332f7e2e3b3p-381,
                0x1.fd3c0cf5c2425p182,
                0x1.fd3c0cf5c2425p183,
                0x1.fd424d6faf031p-290,
                0x1.fd424d6faf03p-290,
                0x1.fd472264125e3p-445,
                0x1.fd562262a9689p90,
                0x1.fd58ed1146958p-801,
                0x1.fd5c6e4f20064p-832,
                0x1.fd5ef2bc6678ep-577,
                0x1.fd81ab46p-1043,
                0x1.fd9709d9aeb19p-34,
                0x1.fd9a057667b38p-879,
                0x1.fda3dea7fe714p287,
                0x1.fdadafcacdf84p-917,
                0x1.fdadafcacdf84p-918,
                0x1.fdbf8cfa4ea6cp-169,
                0x1.fdc243d524a9p497,
                0x1.fdca16e04b86dp195,
                0x1.fdca16e04b86ep195,
                0x1.fddc8cfd6f74ap+892,
                0x1.fddc8cfd6f74ap893,
                0x1.fddf9b738d756p593,
                0x1.fddf9b738d756p594,
                0x1.fddf9b738d756p595,
                0x1.fde2f4cdda833p195,
                0x1.fde4791e0aa76p-604,
                0x1.fdf5d4460d8dp-356,
                0x1.fdf5d4460d8dp-357,
                0x1.fdf6f3244d90ap-478,
                0x1.fe0352e39dbb6p-983,
                0x1.fe03d31dd70f2p485,
                0x1.fe17d29eb0f48p-472,
                0x1.fe1eb956721fcp-548,
                0x1.fe2c63a23383fp-36,
                0x1.fe3544145e9d8p-100,
                0x1.fe3544145e9d8p-101,
                0x1.fe3544145e9d8p-102,
                0x1.fe3544145e9d8p-103,
                0x1.fe3544145e9d8p-95,
                0x1.fe3544145e9d8p-96,
                0x1.fe3544145e9d8p-97,
                0x1.fe3544145e9d8p-98,
                0x1.fe3544145e9d8p-99,
                0x1.fe36afea408ccp241,
                0x1.fe393b8cc0a87p854,
                0x1.fe395220ae5cap-581,
                0x1.fe46e378bf132p+852,
                0x1.fe46e378bf132p853,
                0x1.fe46e378bf132p854,
                0x1.fe46e378bf132p855,
                0x1.fe46e378bf132p+856,
                0x1.fe52048590672p+680,
                0x1.fe52048590673p+680,
                0x1.fe5ea54a4f3c2p114,
                0x1.fe60037ddf584p-910,
                0x1.fe601457dce4ep-616,
                0x1.fe601457dce4ep-617,
                0x1.fe601457dce4ep-618,
                0x1.fe601457dce4ep-619,
                0x1.fe601457dce4ep-620,
                0x1.fe67b2418193bp703,
                0x1.fe6c6dcc3c5acp-118,
                0x1.fe6c6dcc3c5acp-119,
                0x1.fe74acddd7d4ap983,
                0x1.fe7ef4214dc9ep695,
                0x1.fe7ef4214dc9ep+696,
                0x1.fe89cf1bd76acp-900,
                0x1.fe8b3a4121092p+876,
                0x1.fe8e6d510432p-1025,
                0x1.fe8e6d510432p-1026,
                0x1.fe8e6d510432p-1027,
                0x1.fe8fc1ef432d4p-20,
                0x1.fe936d8b5956fp738,
                0x1.fea5f0cb5757p-486,
                0x1.fea5f0cb5757p-487,
                0x1.feb998730dad5p151,
                0x1.feb998730dad5p+152,
                0x1.fec39b55dc488p-949,
                0x1.fecccdfdb7107p-910,
                0x1.feced44cf27e8p+808,
                0x1.fed1f90b278f9p-574,
                0x1.fed2a5a5337c2p-239,
                0x1.fee1bef206b74p-502,
                0x1.fee1bef206b74p-503,
                0x1.feedcaf44cb14p591,
                0x1.feef63f97d79cp-968,
                0x1.feef63f97d79cp-969,
                0x1.feef63f97d79cp-970,
                0x1.feef63f97d79cp-971,
                0x1.feef63f97d79cp-972,
                0x1.ff020d4511a5ep-57,
                0x1.ff031c9468a61p-538,
                0x1.ff070620d6ae6p763,
                0x1.ff110f551845ep+608,
                0x1.ff161d0d00d78p-1007,
                0x1.ff1ce3226396p-1026,
                0x1.ff1ce3226396p-1027,
                0x1.ff2c3494d9147p-500,
                0x1.ff3455d939fb4p-823,
                0x1.ff35f8cd487bbp-125,
                0x1.ff38bf4d8a6d6p-315,
                0x1.ff38bf4d8a6d6p-316,
                0x1.ff3d68595ff1cp+176,
                0x1.ff44249b87eb7p+780,
                0x1.ff5a70d3d2fefp-727,
                0x1.ff5a70d3d2fefp-728,
                0x1.ff5a70d3d2fefp-729,
                0x1.ff6f9c594bfa9p-54,
                0x1.ff75d34443266p170,
                0x1.ff779fd329cb8p-486,
                0x1.ff779fd329cb9p-486,
                0x1.ff7afee2062dap-349,
                0x1.ff7c67e9e2b21p-199,
                0x1.ff868bf4d956ap-151,
                0x1.ff8804053000ap166,
                0x1.ff8804053000ap167,
                0x1.ff8804053000ap+168,
                0x1.ff8804053000ap169,
                0x1.ff923713e3cc6p343,
                0x1.ff9fb03194a7p-93,
                0x1.ff9fb03194a7p-94,
                0x1.ffb3e3d80ac5p-566,
                0x1.ffb3e3d80ac5p-567,
                0x1.ffbb8e7b24979p577,
                0x1.ffbb8e7b24979p578,
                0x1.ffbdd391b149dp-482,
                0x1.ffc81bc29f02ap-324,
                0x1.ffc81bc29f02ap-325,
                0x1.ffc81bc29f02ap-326,
                0x1.ffc81bc29f02ap-327,
                0x1.ffc81bc29f02ap-328,
                0x1.ffc81bc29f02ap-329,
                0x1.ffc81bc29f02ap-330,
                0x1.ffc81bc29f02ap-331,
                0x1.ffc81bc29f02ap-332,
                0x1.ffc81bc29f02ap-333,
                0x1.ffc81bc29f02ap-334,
                0x1.ffcd1097a341cp-249,
                0x1.ffdfe5494a2b5p593,
                0x1.ffe37de66964cp-784,
                0x1.ffeebfc8b81b6p-33,
                0x1.ffeebfc8b81b6p-34,
                0x1.ffeebfc8b81b6p-35,
                0x1.ffeebfc8b81b6p-36,
                0x1.ffeebfc8b81b6p-37,
                0x1.fp-1069,
                0x1p+0,
                0x1p-1069,
                0x1p-1070,
                0x1p-1071,
                0x1p-1072,
                0x1p-10,
                0x1p-11,
                0x1p-12,
                0x1p-13,
                0x1p-14,
                0x1p-1,
                0x1p-15,
                0x1p-16,
                0x1p-17,
                0x1p-18,
                0x1p-19,
                0x1p-20,
                0x1p-21,
                0x1p-22,
                0x1p-2,
                0x1p-23,
                0x1p-24,
                0x1p-3,
                0x1p-4,
                0x1p-5,
                0x1p-6,
                0x1p-7,
                0x1p-8,
                0x1p-9,
        };
    }
}
